/* Copyright (c) 2015, 2025, Oracle and/or its affiliates.*/
/* All rights reserved.*/
 
/* 
   NAME
     jznerr.h - JSON error codes

   DESCRIPTION
     List of all JSON error codes

     The following codes are possible error output generated by
     json parser, path engine, serializer, etc.

     When appending new error codes to the following list, please do not
     forget to update the 'jznErrMessages' array in jznu.c to keep the error
     system in sync.

     Also update xdk/mesg/jznus.msg.

     Also please do not renumber the error numbers.

   RELATED DOCUMENTS
 
   EXPORT FUNCTION(S)

   INTERNAL FUNCTION(S)

   EXAMPLES

   NOTES
     This is a public (to customers) header file and world-viewable.
     The file can include only public headers.

   MODIFIED   (MM/DD/YY)
   apfwkr      04/21/25 - CI #37857619 of dmcmahon_bug-37816616 from main
   apfwkr      04/02/25 - CI #37779246 of dmcmahon_bug-37444223 from main
   apfwkr      01/18/25 - CI #37494099 of dmcmahon_bug-37382482 from main
   apfwkr      01/18/25 - CI #37494098 of dmcmahon_bug-37255952 from main
   apfwkr      01/18/25 - CI #37494094 of dmcmahon_bug-37459932 from main
   apfwkr      11/12/24 - CI# 37274424 of dmcmahon_bug-37159595 from main
   apfwkr      10/30/24 - CI# 37230449 of dmcmahon_bug-36788895 from main
   apfwkr      08/12/24 - CI# 36943393 of dmcmahon_bug-36845202 from main
   apfwkr      08/12/24 - CI# 36943386 of dmcmahon_bug-36913836 from main
   apfwkr      06/06/24 - CI# 36706798 of dmcmahon_bug36357136 from main
   apfwkr      06/03/24 - CI# 36691636 of dmcmahon_bug-36598093 from main
   dmcmahon    04/10/25 - Bug 37816616: new DIFF error messages and parameters
   dmcmahon    03/12/25 - Bug 37681430: JZNERR_NAN_ENCOUNTERED
   zliu        03/05/25 - Bug 37659100/37660960: JZNENG_NO_NST_ARY
   dmcmahon    02/10/25 - Bug 37009487: warning: JZNERR_RUN_OVER_UNDER
   dmcmahon    01/03/25 - Bug 37444223: JZNERR_TOO_MANY_EVENTS
   dmcmahon    12/18/24 - Bug 37409429: BOV errors for write calculations
   dmcmahon    12/11/24 - Bug 37382482: JZNERR_BOV_BAD_WHERE_BIND
   zliu        10/01/24 - Bug 37131969: add JZNERR_RANGE_ARRAY_END
   dmcmahon    09/24/24 - Bug 37097111: JZNERR_OUTPUT_TRUNCATED
   dmcmahon    09/04/24 - Bug 37065330: more array error messages
   dmcmahon    09/13/24 - Bug 37026177: JZNERR_SCHEMA_ONEOF_UNMATCHED
   dmcmahon    08/24/24 - Bug 36985504: JZNERR_ARRAY_DIMENSION_MISMATCH
   dmcmahon    08/19/24 - Bug 36963969: more array errors
   dmcmahon    08/12/24 - Bug 36941964: JZNERR_ARRAY_INVALID_CONV
   dmcmahon    08/08/24 - Bug 36931343: JZNERR_DIGITS_TRUNCATED
   dmcmahon    08/03/24 - Bug 36913836: JZNERR_SORT_TOO_MANY_FIELDS
   dmcmahon    07/08/24 - Bug 36816636: JZNERR_VEC_BAD_NUMBER
   zliu        07/08/24 - Bug 36817752: JZNERR_MODIFIER_xxx errors
   dmcmahon    07/04/24 - Bug 36788895: JZNERR_BOV_LINK_DUP_COL
   dmcmahon    06/17/24 - Bug 36761296: JZNERR_PATH_FUNC_EXPR_TYPE
   dmcmahon    06/06/24 - Bug 36732069: Array operation errors
   dmcmahon    05/31/24 - Bug 36644912: JZNERR_REFUSED
   dmcmahon    05/17/24 - Bug 36598093: JZNERR_UNPRINTABLE
   dmcmahon    03/11/24 - Bug 36390771: JZNERR_ETAG_LEN_MISMATCH
   dmcmahon    02/02/24 - Bug 36255599: JZNERR_RANGE_VECTOR_END
   dmcmahon    01/05/24 - Bug 36170748: JZNERR_VECTOR_TOO_LARGE
   zliu        12/18/23 - bug 36132471: vector type into json xdk
   dmcmahon    11/30/23 - Bug 35886684: errors for setwise ADD/DELETE
   zliu        11/21/23 - bug 36037768,35998857: JZNERR_CHKF_XXX
   dmcmahon    10/18/23 - Bug 35925121: jznErrorGetFieldNameMessage()
   dmcmahon    10/04/23 - Bug 35871361: JZNERR_BOV_NOT_WRITABLE_COL
   dmcmahon    09/22/23 - Bug 35836587: JZNERR_CROSS_PATH_SUBSET
   dmcmahon    09/07/23 - Bug 35787778: JZNERR_BOV_BAD_CHANGE
   dmcmahon    08/29/23 - Bug 35757354: JZNERR_BOV_NO_FIELD_NAME
   dmcmahon    08/16/23 - Bug 35660228: JZNERR_PATCH_NOT_ARRAY/OBJECT
   dmcmahon    07/19/23 - Bug 35597298: JZNDIFF_PATH_TOO_LONG
   dmcmahon    07/19/23 - Bug 35597298: JZNERR_DIFF_PATH_TOO_LONG
   dmcmahon    07/11/23 - Bug 35562545: more diff apply errors
   dmcmahon    06/29/23 - Bug 35549471: JZNERR_CANNOT_APPLY_DIFF
   efjose      06/28/23 - Bug 35547825: Range checks for qjsngGetSqlErr
   dmcmahon    06/27/23 - Bug 34841184: JZNERR_NO_DIFFERENCES
   dmcmahon    06/20/23 - Bug 35519646: JZNERR_CALC_ON_EMPTY
   dmcmahon    05/23/23 - Bug 35420559: JZNERR_BOV_NESTED_FLEX_COL
   dmcmahon    05/22/23 - Bug 35363680: Errors for non-integer subscripts
   dmcmahon    05/17/23 - Bug 35046545: JZNERR_TRANS_TEST_FAILED
   dmcmahon    05/12/23 - Bug 35378595: Flexfield run-time errors
   dmcmahon    05/11/23 - Bug 35378375: JZNERR_BOV_ROW_MISMATCH
   dmcmahon    05/12/12 - Bug 35383407: JZNERR_PATH_PATH_BAD_FUNC
   dmcmahon    05/11/12 - Bug 35383407: JZNERR_PATH_PATH_BAD_FUNC
   dmcmahon    05/10/23 - Bug 35375960: BOV errors for flex fields
   dmcmahon    05/05/23 - Bug 35284528: JZNERR_CROSS errors
   dmcmahon    05/01/23 - Bug 35309415: JZNERR_BOV_COLUMN_MISMATCH
   dmcmahon    04/24/23 - Bug 35315012: JZNERR_BOV_DELETED_ROW
   dmcmahon    04/12/23 - Bug 35277939: string conversion error messages
   dmcmahon    03/30/23 - Bug 35236092: jznErrorRenderString()
   dmcmahon    03/29/23 - Bug 35230657: JZNERR_PATH_BAD_ARITH_ARG
   dmcmahon    03/23/23 - Bug 35210762: JZNERR_PATH_MISSING_EXISTS
   dmcmahon    03/20/23 - Bug 35202337: JZNERR_CODE_LINE
   dmcmahon    02/09/23 - Bug 35069840: JZNERR_SINGLE_QUOTED_STRING
   dmcmahon    02/06/23 - Bug 35008075: JZNERR_PATH_VARIABLE_START
   dmcmahon    02/03/23 - Bug 35048760: JZNERR_<VARIABLE|FUNCITON>_NOT_ALLOWED
   dmcmahon    01/20/23 - Bug 35002198: JZNERR_BAD_SUBSCRIPT_TYPE
   dmcmahon    01/14/23 - Bug 34980720: JZNERR_IGNORED
   dmcmahon    01/11/23 - Bug 34934146: JZNERR_PATH_EXPR_INCOMPLETE
   dmcmahon    01/10/23 - Bug 34960778: JZNERR_CANT_PARSE_OSON
   dmcmahon    01/10/23 - Bug 34960778: JZNERR_CANT_PARSE_BINARY
   dmcmahon    01/06/23 - Bug 34956728: JZNERR_EMPTY_SEQUENCE
   dmcmahon    10/28/22 - Bug 34746340: Errors for JDV operations
   efjose      09/30/22 - Bug 34304247: Add 340,393,93 err codes to improve 43
   dmcmahon    09/16/22 - Bug 34611313: JZNERR_FUNC_ARG_* (parameterized)
   dmcmahon    09/15/22 - Bug 34604566: JZNERR_EXPONENT_OUT_OF_RANGE
   dmcmahon    09/13/22 - Bug 34601260: CALC_NOT_DATE & CALC_NOT_SCALAR
   dmcmahon    08/29/22 - Bug 34552347: JZNERR_CALC_BAD_DATE_OBJ
   dmcmahon    08/27/22 - Bug 34540095: JZNERR_BOV_INVALID_DOMAIN
   dmcmahon    08/19/22 - Bug 34501699: JZNERR_CALC_INVALID_ARG/BAD_PATH
   dmcmahon    08/18/22 - Bug 34511143: JZNERR_VAR_UNDEFINED
   dmcmahon    08/10/22 - Bug 34477101: JZNERR_PATH_BAD_PAT_COMPARE
   dmcmahon    07/29/22 - Bug 34444358: JZNERR_CALC_NOT_DOC_NODE
   dmcmahon    07/22/22 - Bug 34413329: JZNERR_TRANS_RHS_NOT_NODE
   dmcmahon    07/19/22 - Bug 34401283: JZNERR_VAR_UNINITIALIZED
   dmcmahon    07/12/22 - Bug 34373161: JZNERR_INCOMPATIBLE_COMPARE
   dmcmahon    06/10/22 - Bug 34268809: JZNERR_TRANS_BAD_TARGET_BIND
   dmcmahon    06/06/22 - Bug 34256107: JZNERR_TRANS_NO_TARGET_PATH
   dmcmahon    05/10/22 - Bug 34156845: JZNERR_BOV_MISSING_LINK
   dmcmahon    03/15/22 - Bug 33963555: JZNERR_UPD_INV_RHS_PATH
   dmcmahon    03/01/22 - Bug 33901663: jznBovTreeCompare error codes
   dmcmahon    02/25/22 - Bug 33891128: PRED_ID errors
   sriksure    02/14/22 - Bug 33845235: Cast mode of schema validation
   dmcmahon    02/09/22 - Bug 33838672: more path parse errors
   dmcmahon    02/02/22 - Bug 33820835: add JZNERR_PATH_ARITH_NOT_COMP
   dmcmahon    01/31/22 - Bug 33807368: add JZNERR_PATH_NON_COMPARISON
   dmcmahon    01/24/22 - Bug 33781922: new SODA AGG error messages
   dmcmahon    01/16/22 - Bug 33758217: Add JZNERR_TRANS_BAD_DESCENDANT
   dmcmahon    01/16/22 - Bug 33758147: Arrayification errors
   dmcmahon    11/23/21 - Bug 33592881: Add JZNERR_BOV_EMPTY_FIELD_LIST
   dmcmahon    11/13/21 - Bug 33566556: more JZNERR_BOV error codes
   dmcmahon    11/03/21 - Bug 33533237: Add JZNERR_BOV_INVALID_LINK
   dmcmahon    11/01/21 - Bug 33424094: Add JZNERR_BOV_DUPLICATE_ROW
   dmcmahon    09/29/21 - Add sort/pivot related error codes
   dmcmahon    09/28/21 - Bug 33406104: Add BOV error codes
   sriksure    08/10/21 - Bug 33206851: Add JZNERR_SCHEMA_PATTERN_INVALID
   sriksure    08/06/21 - Bug 33200128: Add JZNERR_SCHEMA_THEN_INVALID
   sriksure    07/10/21 - Bug 33099603: Add JZNERR_SCHEMA_DEPNSCHEMAS_INVALID
   sriksure    07/06/21 - Bug 33076646: Add JZNERR_SCHEMA_DEPNREQD_INVALID
   sriksure    07/01/21 - Bug 33014630: Add unevaluated nodes error
   dmcmahon    06/12/21 - Bug 32995128: JZNERR_DUPLICATE_ELEMENTS
   sriksure    06/03/21 - Bug 32956194: Add schema validation errors
   dmcmahon    05/26/21 - Bug 32931540: add JZNERR_QBE_INVALID_ARR_MATCH
   sriksure    05/21/21 - Bug 32910038: Add support for type, enum, const
   bhammers    05/12/21 - Project 83238: JSON Schema validation error coded 
   dmcmahon    03/05/21 - Bug 32593845: JZNERR_QBE_INVALID_COMBO
   dmcmahon    02/11/21 - Bug 32499699: add JZNERR_WRONG_NUM_TYPE_ARGS
   dmcmahon    01/29/21 - Bug 32442285: add JZNERR_CALC codes
   dmcmahon    01/14/21 - Bug 32383277: add JZNERR_LOCATION
   dmcmahon    07/17/20 - Bug 31637902: JZNERR_DOM_BAD_NODE_REF
   dmcmahon    06/05/20 - Add jznerrAssert()
   dmcmahon    05/30/20 - Bug 31423833: OSON encode/decode errors
   dmcmahon    05/22/20 - Add jznErrorGetMessageBuf()
   bhammers    12/02/19 - bug 30601759: JZNERR_TRANS_INVALID_NAME 
   bhammers    11/26/19 - JZNERR_TRANS_NOT_OBJECT, JZNERR_TRANS_APPEND_CREATE
   bhammers    05/17/19 - bug 29794416: Add JZNERR_SCALAR_VALUE
   bhammers    05/21/19 - bug 29765376: add JZNERR_TRANS_CANT_UPDATE_ROOT 
   dmcmahon    03/11/19 - Add JZNERR_QBE_ORDERBY_ARR_STEP
   bhammers    04/04/19 - bug 29405826: add JZNERR_TRANS_NULL_RHS_VAL
   bhammers    03/01/19 - Bug 28441064: add JZNERR_INV_BOOL_TO_NUM_CONV
   bhammers    01/16/19 - adding more json_transform errors
   sriksure    09/27/18 - Bug 28717127: Rationalize includes in public jzn*.h
   bhammers    09/03/18 - add JSON_TRANSFORM errors
   zliu        07/03/18 - Bug 28287398:add JZNOCT3_OSON_PUPD_NOT_FEA
   dmcmahon    05/24/18 - Add JZNERR_PATH_EMPTY_STEP
   sriksure    03/01/18 - Add QBE error 337
   dmcmahon    02/13/18 - QBE errors 335 and 336
   dmcmahon    11/07/17 - CSV error messages
   dmcmahon    09/03/17 - Add JZNERR_PATH_CHAR_NOT_ESCAPED
   dmcmahon    09/02/17 - Bug 26741314: add JZNERR_QBE_ID_NOT_SINGLE
   bhammers    08/16/17 - add JZNERR_NUM_OVERFLOW
   dmcmahon    08/05/17 - Add JZNERR_PATH_ARRAY_OVERLAP
   dmcmahon    03/24/17 - Bug 25773794: wrong subclass error codes
   zliu        03/13/17 - Bug 25722364, 25722384:
                          sql/json path expr typecheck at compile time
                          JZNERR_PATH_WRG_ARG_TYP,JZNERR_PATH_TYPE_MISMATCH,
                          JZNERR_PATH_CONST_CVT_ERR
   dmcmahon    03/12/17 - Add jznErrorGetMessageVA
   dmcmahon    02/13/17 - Bug 25478895: add JZNERR_EMPTY_CACHE
   dmcmahon    09/29/16 - Bug 24758022: add JZNERR_INVALID_REGEX
   dmcmahon    08/20/16 - Add JZNERR_CS_CONV_FAILURE
   dmcmahon    07/17/16 - Add JZNERR_PATH_NESTING_OVERFLOW
   dmcmahon    05/24/16 - More QBE error messages.
   prthiaga    09/09/15 - Bug 21812442: Add JZNERR_PATH_BAD_FUNC_EXPR 
   bhammers    09/15/14 - add JZNERR_EMPTY_INPUT  
   dmcmahon    07/09/15 - Creation

*/

#ifndef JZNERR_ORACLE
# define JZNERR_ORACLE

#ifndef ORATYPES
# include <oratypes.h> 
#endif

#ifndef ORASTRUC
# include <orastruc.h>
#endif

#ifndef XML_ORACLE
# include <xml.h>
#endif

#ifndef JZNTYPES_ORACLE
# include <jzntypes.h>
#endif

/*---------------------------------------------------------------------------
                     PUBLIC TYPES AND CONSTANTS
  ---------------------------------------------------------------------------*/

#define JZNERR_OK                      0
#define JZNERR_END_OF_INPUT            1 /* End of input - not a hard error  */
#define JZNERR_INVALID_ARGUMENT        2 /* Argument or value not allowed    */
#define JZNERR_MISSING_INPUT           3 /* Parse attempted without input    */
#define JZNERR_CANT_OPEN_FILE          4 /* File not found                   */
#define JZNERR_SYNTAX_ERROR            5 /* Invalid JSON syntax              */
#define JZNERR_INVALID_ESCAPE          6 /* Unrecognized \ escape sequence   */
#define JZNERR_DUPLICATE_KEYS          7 /* Duplicate field key in object    */
#define JZNERR_PARSE_IN_PROGRESS       8 /* Parser not reset before input    */
#define JZNERR_UNSUPPORTED_ENCODING    9 /* Unsupported char set conversion  */
#define JZNERR_BAD_CHARACTER_DATA     10 /* Invalid UTF-8 byte sequence      */
#define JZNERR_INPUT_READ_ERROR       11 /* I/O error on stream or file      */
#define JZNERR_PREMATURE_END_OF_DATA  12 /* Required data not available      */
#define JZNERR_STACK_OVERFLOW         13 /* Json nested too many levels      */
#define JZNERR_STACK_UNDERFLOW        14 /* Json parser internal error       */
#define JZNERR_NAME_TOO_LONG          15 /* Json name is too long            */
#define JZNERR_PRINT_STREAM_ERROR     16 /* serializer: stream error         */
#define JZNERR_PRINT_BUF_ERROR        17 /* serializer: buffer error         */
#define JZNERR_PRINT_DATA_TOO_LARGE   18 /* serializer: input size too large */
#define JZNERR_PRINT_MISSING_WRAPPER  19 /* cannot print without wrapper     */
#define JZNERR_PRINT_MISSING_DATA     20 /* nothing to serialize,empty result*/
#define JZNERR_PRINT_NO_SCALAR        21 /* value to be printed not scalar   */
#define JZNERR_EXTRA_INPUT            22 /* Extra bytes after end of input   */
#define JZNERR_DOCLEN_MISMATCH        23 /* Length doesn't match data        */
#define JZNERR_INVALID_MEMBER_CODE    24 /* Unrecognized data type code      */
#define JZNERR_INVALID_BINARY_DATA    25 /* invalid binary data              */
#define JZNERR_TOO_MANY_LEVELS        26 /* Too many nested levels           */
#define JZNERR_INTERNAL               27 /* JSON internal error              */
#define JZNERR_OUT_OF_MEMORY          28 /* Out of memory                    */
#define JZNERR_DATA_VALUE_TOO_LARGE   29 /* Oversized data member            */
#define JZNERR_NULL_POINTER           30 /* Null pointer exception           */
#define JZNERR_INVALID_PATH           31 /* Invalid path expression          */
#define JZNERR_INVALID_ARRAY_INDEX    32 /* Invalid array index in path      */
#define JZNERR_INDEX_WRONG_ORDER      33 /* Array index in wrong order       */
#define JZNERR_FIELD_NAME_TOO_LONG    34 /* field name too long              */
#define JZNERR_TOT_FLD_NAMES_TOO_LONG 35 /* total field names too long       */
#define JZNERR_ARRAY_OUT_OF_BOUNDS    36 /* array out of bounds access       */
#define JZNERR_DOM_OUT_OF_BOUND       36 /* ### Deprecated name              */
#define JZNERR_INVALID_NUM            37 /* invalid number                   */
#define JZNERR_INVALID_DATETIME       38 /* invalid datetime                 */
#define JZNERR_SZ_OVFLW               39 /* size overflow for set encoding   */
#define JZNERR_INVALID_SPEC           40 /* invalid patch/diff/update spec   */
#define JZNERR_PATCH_FAILED           41 /* patch failed against target      */
#define JZNERR_READ_ONLY              42 /* cannot modify, read only target  */
#define JZNERR_UNIMPLEMENTED          43 /* feature not implemented          */
#define JZNERR_CONVERSION_FAILED      44 /* conversion failed (eg geojson)   */
#define JZNERR_BAD_EVENT_SEQUENCE     45 /* invalid event stream             */
#define JZNERR_DOM_KEY_NOT_EXIST      46 /* requested key does not exist     */
#define JZNERR_DOM_CYCLIC_REF         47 /* attempted cyclic reference       */
#define JZNERR_DOM_MULTIPLE_REF       48 /* multiple references to node      */
#define JZNERR_EMPTY_INPUT            49 /* input empty (e.g. empty lob)     */
#define JZNERR_CS_CONV_FAILURE        50 /* charset conversion failed        */
#define JZNERR_INVALID_REGEX          51 /* invalid regular expression       */
#define JZNERR_EMPTY_CACHE            52 /* no cached value available        */
#define JZNERR_DOM_MUST_BE_OBJ        53 /* DOM operation must be on object  */
#define JZNERR_DOM_MUST_BE_ARR        54 /* DOM operation must be on array   */
#define JZNERR_DOM_NOT_SCALAR         55 /* target of DOM op must be scalar  */
#define JZNERR_NUM_OVERFLOW           56 /* number outside range, +inf, -inf */
#define JZNERR_INV_BOOL_TO_NUM_CONV   57 /* invalid Bool to num conversion   */
#define JZNERR_SCALAR_VALUE           58 /* the JSON is a scalar value       */
#define JZNERR_BAD_SCALAR_TYPE        59 /* Unsupported scalar type          */
#define JZNERR_DOM_BAD_NODE_REF       60 /* Use of node after freeing it     */
#define JZNERR_OPERATION_DISALLOWED   61 /* Operation is disabled/disallowed */
#define JZNERR_DUPLICATE_ELEMENTS     62 /* Array has duplicate elements     */
#define JZNERR_SORT_PIVOT_NOT_OBJ     63 /* sort/pivot on non-object element */
#define JZNERR_PIVOT_NAME_NOT_STR     64 /* pivot on non-string name         */
#define JZNERR_PIVOT_VALUE_NOT_FOUND  65 /* pivot on missing value           */
#define JZNERR_INVALID_CAST           66 /* invalid cast requested           */
#define JZNERR_INCOMPATIBLE_COMPARE   67 /* compare of mismatched nodes      */
#define JZNERR_EXPONENT_OUT_OF_RANGE  68 /* exponent too small or too large  */
#define JZNERR_TOO_MANY_EVENTS        69 /* event memory exhausted           */

#define JZNERR_PARSE_EXTRA_COMMA      70 /* unexpected comma                 */
#define JZNERR_PARSE_MISSING_COMMA    71 /* missing comma                    */
#define JZNERR_PARSE_MISSING_VAL      72 /* field name ":" with no value     */
#define JZNERR_PARSE_MISSING_COLON    73 /* field name not followed by ":"   */
#define JZNERR_PARSE_UNQUOTED_FIELD   74 /* field name not in double quotes  */
#define JZNERR_PARSE_UNQUOTED_VAL     75 /* item value not in double quotes  */
#define JZNERR_PARSE_INVALID_NUMBER   76 /* number could not be parsed       */
#define JZNERR_PARSE_INVALID_BYTE     77 /* disallowed ASCII byte            */
#define JZNERR_PARSE_INVALID_LITERAL  78 /* %s must be true/false/null       */
#define JZNERR_PARSE_UNCLOSED_QUOTE   79 /* unclosed quoted string           */
#define JZNERR_OSON_PUPD_NOT_FEA      80 /* OSON partial update not feasible */
#define JZNERR_PARSE_INVALID_HEX      81 /* hex conversion failed            */
#define JZNERR_PARSE_INVALID_SYNTAX   82 /* unexpected syntax character %s   */
#define JZNERR_PARSE_UNCLOSED_ARRAY   83 /* unclosed array                   */
#define JZNERR_PARSE_UNCLOSED_OBJECT  84 /* unclosed object                  */
#define JZNERR_PARSE_UNINTELLIGIBLE   85 /* JSON contains invalid chars      */
#define JZNERR_PARSE_MISSING_ITEM     86 /* missing next field or element    */
#define JZNERR_PARSE_EXCESS_CHARS     87 /* trailing junk chars after JSON   */
/*
** (through 89 reserved for JSON parser failures)
*/
#define JZNERR_OSON_CORRUPT           90 /* Corrupt/invalid binary JSON %s   */ 
#define JZNERR_OSON_PREMATURE_END     91 /* OSON image ends prematurely      */
#define JZNERR_OSON_TOO_LARGE         92 /* JSON too large for OSON encoding */
#define JZNERR_SCALAR_NOT_BINARY      93 /* non-binary source type to BLOB   */
#define JZNERR_CANT_PARSE_BINARY      94 /* OSON parsed as textual JSON      */
#define JZNERR_CHKF_INP_OBJ   95   /* json modifier check fail as inp is obj */
#define JZNERR_CHKF_INP_ARY   96   /* json modifier check fail as inp is ary */
#define JZNERR_CHKF_INP_SCA   97  /* json modifier check fail as inp is scal */
#define JZNERR_UNPRINTABLE            98 /* Scalar value corrupt/invalid     */

/*
** Range from 100-199 reserved for warnings (due to old XML infrastructure)
*/
#define JZNERR_LOCATION               200 /* (line %s, position %s)          */

#define JZNERR_PATH_UNQUOTED_PUNCT    201
#define JZNERR_PATH_UNQUOTED_ESCAPE   202
#define JZNERR_PATH_INVALID_ESCAPE    203
#define JZNERR_PATH_BAD_SURROGATES    204
#define JZNERR_PATH_ARRAY_TOO_LONG    205
#define JZNERR_PATH_ARRAY_BAD_RANGE   206
#define JZNERR_PATH_BAD_VARIABLE      207
#define JZNERR_PATH_DOLLAR_START      208
#define JZNERR_PATH_TRAILING_CHARS    209
#define JZNERR_PATH_ARRAY_EMPTY       210
#define JZNERR_PATH_ARRAY_INVALID     211
#define JZNERR_PATH_ARRAY_WILDCARD    212
#define JZNERR_PATH_OBJECT_WILDCARD   213
#define JZNERR_PATH_UNCLOSED_ARRAY    214
#define JZNERR_PATH_BAD_ARRAY         215
#define JZNERR_PATH_UNCLOSED_QUOTE    216
#define JZNERR_PATH_BAD_KEY           217
#define JZNERR_PATH_NOT_LITERAL       218
#define JZNERR_PATH_BAD_LITERAL       219
#define JZNERR_PATH_INVALID_NUMBER    220
#define JZNERR_PATH_NUMBER_OVERFLOW   221
#define JZNERR_PATH_BAD_REGEX         222
#define JZNERR_PATH_UNSUPPORTED_OP    223
#define JZNERR_PATH_BAD_COMPARISON    224
#define JZNERR_PATH_COMPARE_PATH      225
#define JZNERR_PATH_NOT_PAREN         226
#define JZNERR_PATH_COMPARE_BOOLEAN   227
#define JZNERR_PATH_QUESTION_PAREN    228
#define JZNERR_PATH_CLOSE_PAREN       229
#define JZNERR_PATH_BAD_EXPR          230
#define JZNERR_PATH_UNQUOTED_UNICODE  231
#define JZNERR_PATH_EXPECTED_DOT      232
#define JZNERR_PATH_BAD_STEP_START    233
#define JZNERR_PATH_AT_NOT_IN_PRED    234
#define JZNERR_PATH_DOLLAR_IN_PRED    235
#define JZNERR_PATH_UNKNOWN_FNOP      236
#define JZNERR_PATH_LHS_RHS           237
#define JZNERR_PATH_MULTIPLE_PREDS    238
#define JZNERR_PATH_PRED_NOT_LAST     239
#define JZNERR_PATH_ARRAY_BAD_CHARS   240
#define JZNERR_PATH_BAD_FUNC_EXPR     241
#define JZNERR_PATH_NESTING_OVERFLOW  242
#define JZNERR_PATH_TYPE_MISMATCH     243
#define JZNERR_PATH_WRG_ARG_TYP       244
#define JZNERR_PATH_CONST_CVT_ERR     245
#define JZNERR_PATH_NON_UNIFORM_TYP   246
#define JZNERR_PATH_ARRAY_OVERLAP     247
#define JZNERR_PATH_CHAR_NOT_ESCAPED  248
#define JZNERR_PATH_EMPTY_STEP        249

/*
** JSON patch errors
*/
#define JZNERR_PATCH_ROOT_ARRAY       250
#define JZNERR_PATCH_OP_OBJECT        251
#define JZNERR_PATCH_MISSING_OP_CODE  252
#define JZNERR_PATCH_INVALID_PATH     253
#define JZNERR_PATCH_INVALID_OP_CODE  254
#define JZNERR_PATCH_UNKNOWN_FIELD    255
#define JZNERR_PATCH_PATH_REQUIRED    256
#define JZNERR_PATCH_MUST_BE_STRING   257
#define JZNERR_PATCH_FROM_NOT_ALLOWED 258
#define JZNERR_PATCH_VAL_NOT_ALLOWED  259
#define JZNERR_PATCH_FROM_REQUIRED    260
#define JZNERR_PATCH_VAL_REQUIRED     261
#define JZNERR_PATCH_CYCLIC_MOVE      262
#define JZNERR_PATCH_NOT_OBJECT       263 /* Apply error */
#define JZNERR_PATCH_NOT_ARRAY        264 /* Apply error */
#define JZNERR_PATCH_MISSING_NODE     265 /* Apply error */
#define JZNERR_PATCH_TEST_FAILED      266 /* Apply error */
#define JZNERR_PATCH_BAD_ARR_POS      267 /* Apply error */

/*
** More path errors: 270-299
*/
#define JZNERR_PATH_BAD_VARKEY        270
#define JZNERR_PATH_UNCLOSED_VARKEY   271
#define JZNERR_PATH_EMPTY_VARKEY      272
#define JZNERR_PATH_MULTI_VARKEY      273
#define JZNERR_WRONG_NUM_TYPE_ARGS    274 /* wrong number/type arguments */
#define JZNERR_PATH_FUNC_EXPR_TYPE    275

#define JZNERR_PATH_DOLLAR_PRED_OP    280
#define JZNERR_PATH_ARITH_NOT_COMP    281 /* arithmetic, not comparison */
#define JZNERR_PATH_NON_COMPARISON    282 /* other non-comparison expr */
#define JZNERR_PATH_UNMATCHED_PAREN   283 /* Function with unclosed paren */
#define JZNERR_PATH_NOT_VAR_LIT       284 /* IN sequence not var or lit */
#define JZNERR_PATH_MISSING_PAREN     285 /* IN sequence no (...) */
#define JZNERR_PATH_BAD_INEQUALITY    286
#define JZNERR_PATH_DUPL_PRED_ID      287
#define JZNERR_PATH_UNKNOWN_PRED_ID   288
#define JZNERR_PATH_BAD_PAT_COMPARE   289
#define JZNERR_PATH_EXPR_INCOMPLETE   290
#define JZNERR_BAD_SUBSCRIPT_TYPE     291
#define JZNERR_VARIABLE_NOT_ALLOWED   292
#define JZNERR_FUNCTION_NOT_ALLOWED   293
#define JZNERR_PATH_VARIABLE_START    294 /* path starts with variable */
#define JZNERR_SINGLE_QUOTED_STRING   295
#define JZNERR_BAD_QUOTED_FUNCTION    296
#define JZNERR_PATH_MISSING_EXISTS    297
#define JZNERR_PATH_BAD_ARITH_ARG     298
#define JZNERR_PATH_PATH_BAD_FUNC     299

/*
** Range from 300-399 reserved for SODA QBE
*/
#define JZNERR_QBE_CONFLICTING_RULES  300
#define JZNERR_QBE_PROJECTION_OBJECT  301
#define JZNERR_QBE_BAD_RULE_VALUE     302
#define JZNERR_QBE_INVALID_PATH       303
#define JZNERR_QBE_KEY_SCALARS        304
#define JZNERR_QBE_ARRAY_NO_VALUES    305
#define JZNERR_QBE_ARRAY_SCALARS_ONLY 306
#define JZNERR_QBE_BETWEEN_TWO_VALUES 307
#define JZNERR_QBE_SPATIAL_OUTERMOST  308
#define JZNERR_QBE_GEOMETRY_MISSING   309
#define JZNERR_QBE_DIST_NOT_ALLOWED   310
#define JZNERR_QBE_DISTANCE_NUMERIC   311
#define JZNERR_QBE_UNIT_STRING        312
#define JZNERR_QBE_SPATIAL_UNKNOWN    313
#define JZNERR_QBE_DISTANCE_REQUIRED  314
#define JZNERR_QBE_EMPTY_OBJECT       315
#define JZNERR_QBE_NOT_FOLLOWED_BY    316
#define JZNERR_QBE_SPATIAL_TARGET     317
#define JZNERR_QBE_NOT_OP_DISALLOWED  318
#define JZNERR_QBE_MUST_BE_STRING     319
#define JZNERR_QBE_MUST_BE_SCALAR     320
#define JZNERR_QBE_MUST_BE_COMPARABLE 321
#define JZNERR_QBE_MUST_BE_ARRAY      322
#define JZNERR_QBE_CONJUNCTION_OBJS   323
#define JZNERR_QBE_ID_OUTERMOST       324
#define JZNERR_QBE_MIXED_DOWNSCOPE    325
#define JZNERR_QBE_ORDERBY_INTEGER    326
#define JZNERR_QBE_QUERY_OBJECT       327
#define JZNERR_QBE_ORDERBY_CONTAINER  328
#define JZNERR_QBE_ORDERBY_NOPATH     329
#define JZNERR_QBE_ORDERBY_INVALID    330
#define JZNERR_QBE_SQLJ_OUTERMOST     331
#define JZNERR_QBE_MUST_BE_OBJECT     332
#define JZNERR_QBE_MUST_BE_PATH       333
#define JZNERR_QBE_ID_NOT_SINGLE      334
#define JZNERR_QBE_MUST_BE_BOOLEAN    335 /* parameter %s must be a boolean  */
#define JZNERR_QBE_PARAM_CONFLICT     336 /* cannot use parameter %s with %s */
#define JZNERR_QBE_LOGICAL_ARR_EMPTY  337 /* %s ($and/$or/$nor) is empty     */
#define JZNERR_QBE_ORDERBY_ARR_STEP   338
#define JZNERR_QBE_INVALID_ARR_MATCH  339
#define JZNERR_QBE_UNIMPLEMENTED      340 /* QBE feature unimplemented       */
#define JZNERR_UPD_INVALID_TRANSFORM  350 /* Generic "bad specification" err */
#define JZNERR_UPD_KEEP_REMOVE_MODIFY 351 /* Conflicting/missing operation   */
#define JZNERR_UPD_SCALAR_NOT_ALLOWED 352 /* Block %s must be container      */
#define JZNERR_UPD_EMPTY_NOT_ALLOWED  353 /* Container %s must not be empty  */
#define JZNERR_UPD_MUST_BE_OBJECT     354 /* Block %s must be an object      */
#define JZNERR_UPD_PATH_NOT_STRING    355 /* path for %s must be a string    */
#define JZNERR_UPD_MUST_BE_STRING     356 /* RHS path %s must be a string    */
#define JZNERR_UPD_PATH_LEAD_DOLLAR   357 /* '$' cannot start path string    */
#define JZNERR_UPD_MUST_BE_SCALAR     358 /* Calculation argument not scalar */
#define JZNERR_UPD_MUST_BE_ARRAY      359 /* Calculation argument not array  */
#define JZNERR_UPD_INVALID_PROJECTION 360 /* disallowed oper for projection  */
#define JZNERR_UPD_UNKNOWN_OPERATOR   361 /* Unknown operator %s             */
#define JZNERR_UPD_MUST_BE_FILTER     362 /* keep/remove must be QBE         */
#define JZNERR_UPD_FIELD_REQUIRED     363 /* required field %s is missing    */
#define JZNERR_UPD_NODE_DISALLOWED    364 /* node type %s not allowed for %s */
#define JZNERR_UPD_EXTRA_MEMBERS      365 /* extra members of structure %s   */
#define JZNERR_UPD_TOO_FEW_MEMBERS    366 /* not enough members of %s        */
#define JZNERR_AGG_NO_MATCH           367 /* column %s unmatched on list %s  */
#define JZNERR_AGG_DUPLICATE_COL      368 /* column %s repeated on list %s   */
#define JZNERR_AGG_BAD_GROUP_COL      369 /* column %s not usable for %s     */
#define JZNERR_AGG_MUST_BE_NUMBER     370 /* value of %s must be a number    */
#define JZNERR_UPD_SCALAR_MISMATCH    371 /* scalars for %s not same type    */
#define JZNERR_UPD_LITERAL_DISALLOWED 372 /* literal %s not allowed for %s   */
#define JZNERR_UPD_BAD_TARGET         373 /* operator %s cannot target %s    */
#define JZNERR_AGG_MUST_BE_STRING     374 /* Column alias %s must be string  */
#define JZNERR_QBE_OPER_OUTERMOST     375 /* %s must be at outer level       */
#define JZNERR_QBE_CONTAINS_COMBO     376 /* Cannot combine with %s          */
#define JZNERR_AGG_MISSING_SORT       377 /* Missing sort for first/last     */
#define JZNERR_AGG_UNKNOWN_OP         378 /* Unknown operator %s             */
#define JZNERR_AGG_UNKNOWN_STAGE      379 /* Unknown pipeline stage          */
#define JZNERR_AGG_INVALID_LEVEL      380 /* Operation %s at nested level    */

#define JZNERR_CSV_UNCLOSED_QUOTE     381 /* unclosed quoted string          */
#define JZNERR_CSV_UNESCAPED_QUOTE    382 /* naked quote mark                */
#define JZNERR_CSV_MISSING_DELIMITER  383 /* no delimiter between strings    */
#define JZNERR_CSV_UNQUOTED_SPACES    384 /* space before/after quotes value */

#define JZNERR_UPD_INV_RHS_PATH       390 /* path not allowed in RHS         */
#define JZNERR_UPD_MUST_BE_NUMBER     391 /* function %s arg must be number  */
#define JZNERR_UPD_BAD_VARIABLE_NAME  392 /* variable name is invalid        */
#define JZNERR_UNIMPLEMENTED_SORT     393 /* Array can't be sorted           */
#define JZNERR_QBE_WRONG_ARGS         394 /* QBE wrong number/type arguments */
/*
** Range from 400-440 reserved for JSON_TRANSFORM
*/
#define JZNERR_TRANS_MISSING_DATA     400 /* path expr did not select value  */
#define JZNERR_TRANS_EXISTING_DATA    401     /* path expr selected a value  */
#define JZNERR_TRANS_NOT_ARRAY        402  /* selected value is not an array */
#define JZNERR_TRANS_FIELD_EXISTS     403     /* a field name already exists */
#define JZNERR_TRANS_FIELD_MISSING    404     /* a field name does not exist */
#define JZNERR_TRANS_CANT_REMOVE_ROOT 405  /* removing root node not allowed */
#define JZNERR_TRANS_UNSUPPORTED_OP   406 /* unsupported transform operation */
#define JZNERR_TRANS_BAD_MULTI_SEL    407   /* multi select not allowed: .*  */
#define JZNERR_TRANS_NULL_RHS_VAL     408   /* rhs is null and ERROR ON NULL */
#define JZNERR_TRANS_CANT_UPDATE_ROOT 409 /* update of root node not allowed */
#define JZNERR_TRANS_NOT_OBJECT       410   /* selected val is not an object */
#define JZNERR_TRANS_APPEND_CREATE    411   /* invalid combination of append */
#define JZNERR_TRANS_INVALID_NAME     412      /* invalid key name in RENAME */
#define JZNERR_TRANS_BAD_DESCENDANT   413                /* invalid LHS '..' */
#define JZNERR_TRANS_NO_TARGET_PATH   414                /* LHS path missing */
#define JZNERR_TRANS_BAD_TARGET_BIND  415            /* LHS bind not allowed */
#define JZNERR_TRANS_RHS_NOT_NODE     416      /* RHS expr is not a DOM node */
#define JZNERR_TRANS_LHS_NOT_VAR      417      /* LHS expr is not a variable */
#define JZNERR_TRANS_TEST_FAILED      418                  /* test ~s failed */
#define JZNERR_TRANS_DUP_SET_VALUE    419    /* value for ADD already in set */
#define JZNERR_TRANS_NOT_SET_VALUE    420     /* value for DELETE not in set */

#define JZNERR_CROSS_PARALLEL_ARRAYS  430      /* parallel arrays disallowed */
#define JZNERR_CROSS_KEY_OVERWRITE    431          /* key collision in entry */
#define JZNERR_CROSS_PATH_SUBSET      432  /* path is subset of another path */
#define JZNERR_CROSS_NO_PATHS         435      /* No paths for cross product */
#define JZNERR_CROSS_NON_OBJECT_STEP  436         /* Array steps not allowed */
#define JZNERR_CROSS_INVALID_PATH     437 /* Wildcard/descendant not allowed */
#define JZNERR_CROSS_DUPLICATE_PATH   438                  /* Redundant path */
#define JZNERR_CROSS_TOO_MANY_STEPS   439      /* More than 65535 path steps */

#define JZNERR_OUTPUT_TRUNCATED       440  /* buffer limit forced truncation */
#define JZNERR_RUN_OVER_UNDER         441         /* operation overran count */
#define JZNERR_NOT_MATCHED            442       /* not equal or not matching */
#define JZNERR_DIGITS_TRUNCATED       443  /* numeric conversion lost digits */
#define JZNERR_CAPABILITY_NOT_MET     444        /* capability not available */
#define JZNERR_REFUSED                445         /* operation not performed */
#define JZNERR_EARLY_EXIT             446          /* function returned ASAP */
#define JZNERR_NO_DIFFERENCES         447                    /* no JSON diff */
#define JZNERR_IGNORED                448                 /* request ignored */
#define JZNERR_EMPTY_SEQUENCE         449              /* no values returned */
#define JZNERR_NOT_FOUND              450        /* requested item not found */

#define JZNERR_CALC_FIELD_MISSING     451       /* path did not find a field */
#define JZNERR_CALC_TOO_MANY_MATCHES  452     /* path matched too many nodes */
#define JZNERR_CALC_NOT_NUMBER        453             /* path is not numeric */
#define JZNERR_CALC_NOT_STRING        454            /* path is not a string */
#define JZNERR_CALC_BAD_LITERAL       455        /* literal is of wrong type */
#define JZNERR_CALC_ERROR             456            /* divide by zero, etc. */
#define JZNERR_CALC_CONV_FAILURE      457          /* type conversion failed */
#define JZNERR_ARITH_ON_NON_NUMBER    458        /* arithmetic on non-number */
#define JZNERR_CALC_WRONG_TYPE        459    /* operation on wrong data type */
#define JZNERR_VAR_UNINITIALIZED      460          /* uninitialized variable */
#define JZNERR_CALC_NOT_DOC_NODE      461   /* target is not a document node */
#define JZNERR_VARIABLE_NOT_NODE      462    /* variable for path not a node */
#define JZNERR_BAD_FORMAT_ELEMENT     463   /* format element not recognized */
#define JZNERR_FORMAT_NOT_MATCHED     464    /* string does not match format */
#define JZNERR_VAR_UNDEFINED          465              /* undefined variable */
#define JZNERR_CALC_INVALID_ARG       466            /* negative value, etc. */
#define JZNERR_CALC_BAD_PATH          467             /* not an allowed path */
#define JZNERR_CALC_BAD_DATE_OBJ      468             /* invalid date object */
#define JZNERR_CALC_NOT_DATE          469    /* path is not a date/timestamp */
#define JZNERR_CALC_NOT_SCALAR        470            /* path is not a scalar */
#define JZNERR_FUNC_ARG_INV_TYPE      471  /* wrong argument type to func %s */
#define JZNERR_FUNC_ARG_NOT_STR       472  /* arg %d to func %s not a string */
#define JZNERR_FUNC_ARG_NOT_NUM       473  /* arg %d to func %s not a number */
#define JZNERR_FUNC_ARG_NOT_DATE      474 /* arg %d to func %s not date/time */
#define JZNERR_FUNC_ARG_INV_VAL       475  /* arg %d to func %s out of range */
#define JZNERR_FUNC_ARG_NOT_OBJ       476       /* arg to func %s not object */
#define JZNERR_FUNC_ARG_NOT_ARR       477        /* arg to func %s not array */
#define JZNERR_CALC_FLD_NOT_NUM       478   /* object field was not a number */
#define JZNERR_CALC_FLD_NOT_STR       479   /* object field was not a string */
#define JZNERR_CALC_FLD_INV_VAL       480  /* object field has invalid value */
#define JZNERR_FUNC_ARG_NOT_INT       481   /* arg %d to func %s not integer */
#define JZNERR_CALC_ON_EMPTY          482     /* null/missing value/argument */
#define JZNERR_BAD_INTEGER            483            /* argument not integer */
#define JZNERR_MISSING_ARGUMENT       484   /* target argument not specified */
#define JZNERR_SORT_TOO_MANY_FIELDS   485        /* too many fields for sort */
#define JZNERR_NAN_ENCOUNTERED        486          /* NaN during comparisons */

/*
** Through 489 reserved for JSON_TRANSFORM errors
*/

/*
** 490 - 499 reserved for execution-time errors in path/calc/gather engines
*/
#define JZNERR_AT_FIELD               496                      /* (field %s) */
#define JZNERR_INVALID_KEY_TYPE       497           /* key name invalid type */
#define JZNERR_BAD_SUBSCRIPT_VALUE    498   /* array subscript invalid value */
#define JZNERR_CODE_LINE              499                   /* %s at line %s */

/*
** Range from 500-569 reserved for schema Validation
*/
#define JZNERR_SCHEMA_INVALID             500            /* schema not valid */
#define JZNERR_SCHEMA_VALID_FAILED        501    /* schema validation failed */
#define JZNERR_SCHEMA_BOOLEAN_FALSE       502    /* schema was boolean false */
#define JZNERR_SCHEMA_TYPE_INVALID        503                /* invalid type */
#define JZNERR_SCHEMA_TYPE_INVALID2       504        /* invalid type (array) */
#define JZNERR_SCHEMA_ENUM_INVALID        505                /* invalid enum */
#define JZNERR_SCHEMA_CONST_INVALID       506               /* invalid const */
#define JZNERR_SCHEMA_MIN_INVALID         507             /* invalid minimum */
#define JZNERR_SCHEMA_MAX_INVALID         508             /* invalid maximum */
#define JZNERR_SCHEMA_EXMIN_INVALID       509           /* invalid exminimum */
#define JZNERR_SCHEMA_EXMAX_INVALID       510           /* invalid exmaximum */
#define JZNERR_SCHEMA_MULT_INVALID        511          /* invalid multipleOf */
#define JZNERR_SCHEMA_MINLEN_INVALID      512           /* invalid minLength */
#define JZNERR_SCHEMA_MAXLEN_INVALID      513           /* invalid maxLength */
#define JZNERR_SCHEMA_PROPS_INVALID       514          /* invalid properties */
#define JZNERR_SCHEMA_REQD_INVALID        515            /* invalid required */
#define JZNERR_SCHEMA_MINPROPS_INVALID    516       /* invalid minProperties */
#define JZNERR_SCHEMA_MAXPROPS_INVALID    517       /* invalid maxProperties */
#define JZNERR_SCHEMA_ADDNLPROPS_INVALID  518     /* invalid addnlProperties */
#define JZNERR_SCHEMA_ITEMS_INVALID       519               /* invalid items */
#define JZNERR_SCHEMA_UNIQ_INVALID        520         /* invalid uniqueItems */
#define JZNERR_SCHEMA_CONT_INVALID        521            /* invalid contains */
#define JZNERR_SCHEMA_MINITEMS_INVALID    522            /* invalid minItems */
#define JZNERR_SCHEMA_MAXITEMS_INVALID    523            /* invalid maxItems */
#define JZNERR_SCHEMA_ADDNLITEMS_INVALID  524     /* invalid additionalItems */
#define JZNERR_SCHEMA_NOT_INVALID         525                 /* invalid not */
#define JZNERR_SCHEMA_ANYOF_INVALID       526               /* invalid anyOf */
#define JZNERR_SCHEMA_ALLOF_INVALID       527               /* invalid allOf */
#define JZNERR_SCHEMA_ONEOF_INVALID       528               /* invalid oneOf */
#define JZNERR_SCHEMA_PROPNAMES_INVALID   529       /* invalid propertyNames */
#define JZNERR_SCHEMA_MINCONT_INVALID     530         /* invalid minContains */
#define JZNERR_SCHEMA_MAXCONT_INVALID     531         /* invalid maxContains */
#define JZNERR_SCHEMA_UNEVALPROPS_INVALID 532    /* invalid unevalProperties */
#define JZNERR_SCHEMA_UNEVALITEMS_INVALID 533    /* invalid unevaluatedItems */
#define JZNERR_SCHEMA_DEPNREQD_INVALID    534   /* invalid dependentRequired */
#define JZNERR_SCHEMA_DEPNSCHEMAS_INVALID 535    /* invalid dependentSchemas */
#define JZNERR_SCHEMA_THEN_INVALID        536                /* invalid then */
#define JZNERR_SCHEMA_IF_ELSE_INVALID     537             /* invalid if-else */
#define JZNERR_SCHEMA_PATTERN_INVALID     538             /* invalid pattern */
#define JZNERR_SCHEMA_FORMAT_INVALID      539              /* invalid format */
#define JZNERR_SCHEMA_ONEOF_UNMATCHED     540               /* invalid oneOf */

#define JZNERR_ARRAY_AXES_MISMATCH        579
#define JZNERR_ARRAY_TYPE_MISMATCH        580   /* type %s not valid for dty */
#define JZNERR_ARRAY_TYPE_UNSUPPORTED     581
#define JZNERR_ARRAY_TOO_MANY_DIMENSIONS  582               /* %d dimensions */
#define JZNERR_ARRAY_BAD_DIMENSION        583  /* dimension value %d invalid */
#define JZNERR_ARRAY_INVALID_OP           584    /* operation %s not allowed */
#define JZNERR_ARRAY_MISMATCH             585   /* operation %s inputs wrong */
#define JZNERR_ARRAY_DIM_OUT_OF_BOUNDS    586
#define JZNERR_ARRAY_BAD_POSITION         587
#define JZNERR_ARRAY_BAD_ELEMENT          588
#define JZNERR_ARRAY_PRECISION_LOSS       589
#define JZNERR_ARRAY_NULL_ELEMENT         590
#define JZNERR_ARRAY_INVALID_CONV         591
#define JZNERR_ARRAY_DIM_OP_NOT_ALLOWED   592   /* op %s not allowed on dims */
#define JZNERR_ARRAY_DIM_CONV_FAILED      593
#define JZNERR_ARRAY_NOT_DIMENSIONED      594
#define JZNERR_ARRAY_DIMENSION_MISMATCH   595
#define JZNERR_ARRAY_TOO_MANY_ELEMENTS    596    /* too many elems for op %s */
#define JZNERR_ARRAY_BAD_NUM_DIMENSIONS   597          /* bad dims for op %s */
#define JZNERR_ARRAY_CONCAT_MISMATCH      598
#define JZNERR_ARRAY_START_END            599

/*
** Range from 600-699 reserved for Business Object Views
*/
#define JZNERR_BOV_INVALID_SPEC       600                 /* view invalid ~s */
#define JZNERR_BOV_UNKNOWN_TABLE      601                /* table ~d unknown */
#define JZNERR_BOV_EMPTY_COL_LIST     602               /* ~s has no columns */
#define JZNERR_BOV_TOO_MANY_COLS      603               /* ~d columns for ~s */
#define JZNERR_BOV_MISMATCH_COLS      604                /* ~s != ~d columns */
#define JZNERR_BOV_UNKNOWN_COLUMN     605           /* column ~d vs ~d total */
#define JZNERR_BOV_UNMATCHED_COLUMN   606           /* column ~s vs table ~s */
#define JZNERR_BOV_NESTED_LINK_NAME   607            /* nested must have key */
#define JZNERR_BOV_DUPLICATE_PROP     608                    /* duplicate ~s */
#define JZNERR_BOV_INVALID_STRING     609            /* property ~s value ~s */
#define JZNERR_BOV_INVALID_NUMBER     610            /* property ~s value ~d */
#define JZNERR_BOV_FIELD_NOT_FOUND    611              /* field ~s not found */
#define JZNERR_BOV_INVALID_LINK       612               /* invalid link spec */
#define JZNERR_BOV_FIELD_NOT_ARRAY    613              /* field ~s not array */
#define JZNERR_BOV_FIELD_NOT_OBJECT   614             /* field ~s not object */
#define JZNERR_BOV_FIELD_NOT_SCALAR   615             /* field ~s not scalar */
#define JZNERR_BOV_FIELD_NOT_STRING   616             /* field ~s not string */
#define JZNERR_BOV_FIELD_NOT_NUMBER   617             /* field ~s not number */
#define JZNERR_BOV_FIELD_NOT_BOOLEAN  618            /* field ~s not boolean */
#define JZNERR_BOV_ELEMENT_NOT_OBJECT 620        /* element of ~s not object */
#define JZNERR_BOV_ELEMENT_NOT_SCALAR 621        /* element of ~s not scalar */
#define JZNERR_BOV_ELEMENT_NOT_STRING 622        /* element of ~s not string */
#define JZNERR_BOV_TOO_MANY_LINKS     623   /* target table linked > 1 times */
#define JZNERR_BOV_NO_ROOT_TARGET     624      /* link target was root table */
#define JZNERR_BOV_COLUMN_STRUCTS     625      /* column in multiple structs */
#define JZNERR_BOV_EMPTY_FIELD_LIST   626         /* container has no fields */
#define JZNERR_BOV_TABLE_MISMATCH     627    /* conflicting usage of (~s,~s) */
#define JZNERR_BOV_FIELD_ARRAY        628    /* array column with field name */
#define JZNERR_BOV_INVALID_DOMAIN     629     /* unmatched or invalid domain */
#define JZNERR_BOV_ETAG_MISMATCH      630      /* etag col ~s ~s for (~s,~s) */
#define JZNERR_BOV_PKEY_MISMATCH      631   /* conflicting pkeys for (~s,~s) */
#define JZNERR_BOV_DUPLICATE_KEY      632              /* duplicate field ~s */
#define JZNERR_BOV_DUPLICATE_COL      633     /* duplicate column (~s,~s,~s) */
#define JZNERR_BOV_COLUMN_MISMATCH    634     /* properties conflict (~s,~s) */
#define JZNERR_BOV_COLUMN_IS_MAPPED   635        /* already mapped column ~s */
#define JZNERR_BOV_UNNEST_FLEX_TAB    636   /* flex table ~s linked unnested */
#define JZNERR_BOV_TAB_HAS_FLEX       637    /* table ~s multiple flexfields */
#define JZNERR_BOV_BAD_FLEX_TYPE      638   /* flex col ~s table ~s not JSON */
#define JZNERR_BOV_CANT_MAP_FLEX      639       /* cannot map flex column ~s */

#define JZNERR_BOV_NE_TAB_NAMES       640
#define JZNERR_BOV_NE_TAB_PROPS       641
#define JZNERR_BOV_NE_TAB_COLS        642
#define JZNERR_BOV_NE_TAB_LINKS       643
#define JZNERR_BOV_NE_COL_PROPS       644
#define JZNERR_BOV_NE_COL_LIST        645
#define JZNERR_BOV_NE_KEY_NAMES       646

#define JZNERR_BOV_NO_FIELD_NAME      647
#define JZNERR_BOV_LINK_DUP_COL       648

#define JZNERR_BOV_NODE_NOT_OBJECT    650               /* ~s must be object */
#define JZNERR_BOV_KEY_NO_MATCH       651                   /* ~s not mapped */
#define JZNERR_BOV_KEY_NOT_OBJECT     652            /* ~s not nested object */
#define JZNERR_BOV_KEY_NOT_ARRAY      653             /* ~s not nested array */
#define JZNERR_BOV_KEY_NOT_SCALAR     654                   /* ~s not scalar */
#define JZNERR_BOV_KEY_WRONG_TYPE     655              /* ~s wrong data type */
#define JZNERR_BOV_KEY_MISSING        656                    /* ~s not found */
#define JZNERR_BOV_KEY_TOO_LARGE      657                    /* ~s too large */
#define JZNERR_BOV_DUPLICATE_ROW      658   /* Duplicate primary key (~s,~s) */
#define JZNERR_BOV_COLUMN_NO_MATCH    659   /* Column in dup table unmatched */
#define JZNERR_BOV_CONFLICTING_ROW    660     /* Row change conflict (~s,~s) */
#define JZNERR_BOV_LINK_CONFLICT      661     /* Link source/target mismatch */
#define JZNERR_BOV_EMPTY_OBJECT       662     /* Object contains no root row */
#define JZNERR_BOV_ELEM_NOT_SCALAR    663        /* element of ~s not scalar */
#define JZNERR_BOV_TOO_MANY_ELEMS     664              /* array ~s too large */
#define JZNERR_BOV_MISSING_LINK       665    /* Link from ~s to ~s unmatched */
#define JZNERR_BOV_NOT_WRITABLE_COL   666     /* col ~s (~s,~s) not writable */
#define JZNERR_BOV_BAD_NUM_STRING     671
#define JZNERR_BOV_BAD_DATE_STRING    672
#define JZNERR_BOV_BAD_DUR_STRING     673
#define JZNERR_BOV_BAD_HEX_STRING     674
#define JZNERR_BOV_BAD_BIN_STRING     675
#define JZNERR_BOV_BAD_RAW_STRING     676
#define JZNERR_BOV_DELETED_ROW        677     /* Row delete conflict (~s,~s) */
#define JZNERR_BOV_ROW_MISMATCH       678       /* Column conflicts (~s, ~s) */
#define JZNERR_BOV_NESTED_FLEX_COL    679      /* cannot nest flex column ~s */
#define JZNERR_BOV_UNKNOWN_CHANGE     680      /* change disallowed field ~s */
#define JZNERR_BOV_FLEX_NOT_OBJ       681      /* invalid flex value (~s,~s) */
#define JZNERR_BOV_FLEX_OVERWRITE     682          /* flex name collision ~s */
#define JZNERR_BOV_BAD_CHANGE         683      /* change disallowed field ~s */
#define JZNERR_BOV_BAD_REMOVE         684      /* remove disallowed field ~s */
#define JZNERR_BOV_BAD_INSERT         685      /* insert disallowed field ~s */
#define JZNERR_BOV_BAD_DATATYPE       686         /* wrong type for field ~s */
#define JZNERR_BOV_ETAG_LEN_MISMATCH  687   /* etag length ~d ~d for (~s,~s) */

#define JZNERR_BOV_WHERE_BIND_MISSING 688       /* ~s no bind at position ~d */
#define JZNERR_BOV_WHERE_BIND_NOMATCH 689        /* ~s bind no match (~s,~s) */
#define JZNERR_BOV_BAD_WHERE_BIND     690  /* ~s unresolvable at position ~d */

#define JZNERR_BOV_COL_HAS_DEFAULT    691           /* ~s has another default */
#define JZNERR_BOV_COL_NO_FIELD       692          /* ~s has no field mapping */
/*
** Errors 700-750 for JSON diff/patch
*/
#define JZNERR_DIFF_DUPLICATE_KEY     701
#define JZNERR_DIFF_KEY_MISSING_BAD   702          /* missing step ~s for key */
#define JZNERR_DIFF_NOT_OBJECT        703
#define JZNERR_DIFF_PATH_TOO_LONG     704
#define JZNERR_DIFF_KEY_MISSING       705            /* key step ~s not found */
#define JZNERR_DIFF_KEY_NOT_SCALAR    706        /* key field ~s not a scalar */

#define JZNERR_CANNOT_APPLY_DIFF      720
#define JZNERR_DIFF_KEY_NOMATCH       721
#define JZNERR_DIFF_ARRAY_NOMATCH     722
#define JZNERR_DIFF_ARR_POS_BOUNDS    723
#define JZNERR_DIFF_FIELD_EXISTS      724

#define JZNERR_DIFF_INVALID_SPEC      730
#define JZNERR_DIFF_POS_KEY_MISMATCH  731
#define JZNERR_DIFF_WRONG_OPER        732
#define JZNERR_DIFF_TRUE_FALSE_NULL   733

/*
** Errors 751-800 for VECTOR type
*/
#define JZNERR_VEC_PRERR              751 /* VECTOR processing error         */
#define JZNERR_VEC_NOT_NUMBER         752 /* non-numeric val from JSON array */
#define JZNERR_VEC_DIM_MISMAT         753 /* dimension mismatch to vec err   */
#define JZNERR_VEC_BAD_DIMENSION      754 /* dimension not recognized        */
#define JZNERR_VEC_BAD_NUMBER         755 /* number is Nan or out of range   */
#define JZNERR_VECTOR_TOO_LARGE       756 /* too large for anyType image     */

/*
** Errors 801-820
*/
#define JZNERR_MODIFIER_CHECK_FAILED  801 /* json type modifier check fail   */
#define JZNERR_MODIFIER_SYNTAX_ERROR  802 /* json type modifier spec error   */
#define JZNERR_MODIFIER_PRT_OVERFLOW  803 /* json type modifier print fail   */

/*
** More More path errors: 820-830
*/
#define JZNERR_NESTED_ARY  821 /* nested array encountered for path eval */
/*
** This requests using a generic code from the XMLERR range to support
** legacy error handlers.
*/
#define JZNERR_DEFAULT                -1

/*
** Fatal error handler (for asserts or throws).
*/
typedef void (*jznFatalErrHandler)(xmlctx *, oratext *);
 
/*====================== ERROR TRANSLATION RANGE CHECKS ===================*/
#define JZNERR_RANGE_PATH_EXPR_END   250
#define JZNERR_RANGE_PATCH_SPEC_END  270
#define JZNERR_RANGE_PATH_EXPR_2_END 300
#define JZNERR_RANGE_QBE_SPEC_END    400
#define JZNERR_RANGE_JSON_TRANS_END  490
#define JZNERR_RANGE_PATH_ENG_END    500
#define JZNERR_RANGE_SCHM_VALID_END  570
#define JZNERR_RANGE_ARRAY_END       600
#define JZNERR_RANGE_BUSNS_OBJ_END   700
#define JZNERR_RANGE_PATCH_DIFF_END  750
#define JZNERR_RANGE_VECTOR_END      800

/*---------------------------------------------------------------------------
                           INTERNAL FUNCTIONS
  ---------------------------------------------------------------------------*/

/*
** Format an error message and write the result to a user-supplied buffer.
*/
void jznErrorFormat(xmlctx *xctx,
                    oratext *outbuf, ub4 outlen,
                    sword errcode, ...);

/*
** Format an error message using a standard error buffer from the xmlctx.
** Note that since the xctx has a single buffer, the caller should copy
** or render the message before calling other code that may overwrite it.
** Optionally prepend a newline (addnl).
*/
oratext *jznErrorGetMessageNL(xmlctx *xctx, boolean addnl, sword errcode, ...);
oratext *jznErrorGetMessage(xmlctx *xctx, sword errcode, ...);
oratext *jznErrorGetMessageVA(xmlctx *xctx, boolean addnl, sword errcode,
                              va_list va);

oratext *jznErrorGetMessageBuf_VA(xmlctx *xctx, oratext *msgbuf, ub4 bufsiz,
                                  boolean addnl, sword errcode, va_list va);
oratext *jznErrorGetMessageBuf(xmlctx *xctx, oratext *msgbuf, ub4 bufsiz,
                               boolean addnl, sword errcode, ...);

/*
** This calls a handler that will exit your current thread of control
** either via a longjmp or with an exit to the operating system. For
** the sake of the longjmp handler, <errmsg> must be memory that is
** stable with respect to a stack unwind, typically a constant string
** from an ASSERT, although heap-allocated memory from an object that
** will survive the longjmp is acceptable. Do not use a local variable!
*/
void jznuFireFatalError(xmlctx *xctx, oratext *errmsg);

/*
** Fire the assert handler, if available. Otherwise call the fatal handler.
*/
void jznerrAssert(jznFatalErrHandler errh, xmlctx *xctx, oratext *errmsg);

/*
** Emit a formatted JZN error message via XmlErrOut. If the error
** code is less than 0 (invalid), it will substitute the reserved
** "general error code" from the XMLERR enumeration. Otherwise, it will
** adjust the error code to the JSON reserved range in the XMLERR space.
** This typically returns to the caller. However, some implementations of
** the error handler will throw an exception and there won't be a return.
*/
void JznErrOut(xmlctx *xctx, sword errcode, oratext *msg);

/*
** Emit a JZN error message, with optional parameters, via XmlErrOut.
** This typically returns to the caller, and will return the errcode.
** However, some implementations of the error handler will throw an
** exception and there won't be a return.
*/
sword JznErrFmtOut(xmlctx *xctx, sword errcode, ...);

void  JznErrFmt(xmlctx *xctx, boolean addnl,
                oratext *outbuf, ub4 outlen,
                sword errcode, va_list va);

ub4 jznErrorGetLocationMessage(xmlctx *xctx,
                               oratext *msgbuf, ub4 bufsiz,
                               ub4 linenum, ub4 linepos,
                               boolean spBefore, boolean spAfter);

ub4 jznErrorGetCodeLineMessage(xmlctx *xctx,
                               oratext *msgbuf, ub4 bufsiz,
                               ub4 linenum, oratext *opname,
                               boolean spBefore, boolean spAfter);

ub4 jznErrorGetFieldNameMessage(xmlctx *xctx,
                                oratext *msgbuf, ub4 bufsiz,
                                oratext *fldname,
                                boolean spBefore, boolean spAfter);

oratext *jznErrorRenderString(xmlctx *xctx,
                              oratext *outbuf, ub4 outsz,
                              oratext *instr,  ub4 inlen);

#endif                                              /* JZNERR_ORACLE */
