/* Copyright (c) 2015, 2021, Oracle and/or its affiliates. */
/* All rights reserved.*/
 
/* 
   NAME
     jznerr.h - JSON error codes

   DESCRIPTION
     List of all JSON error codes

   RELATED DOCUMENTS
 
   EXPORT FUNCTION(S)

   INTERNAL FUNCTION(S)

   EXAMPLES

   NOTES

     The following codes are possible error output generated by
     json parser, path engine, serializer, etc.

     When appending new error codes to the following list, please do not
     forget to update the 'jznErrMessages' array in jznu.c to keep the error
     system in sync.

     Also update xdk/mesg/jznus.msg.

     Also please do not renumber the error numbers.

   MODIFIED   (MM/DD/YY)
   apfwkr      04/13/21 - Backport dmcmahon_bug-32499699 from main
   apfwkr      04/09/21 - Backport dmcmahon_bug-32593845 from main
   dmcmahon    03/05/21 - Bug 32593845: JZNERR_QBE_INVALID_COMBO
   dmcmahon    02/11/21 - Bug 32499699: add JZNERR_WRONG_NUM_TYPE_ARGS
   dmcmahon    07/17/20 - Bug 31637902: JZNERR_DOM_BAD_NODE_REF
   dmcmahon    06/05/20 - Add jznerrAssert()
   dmcmahon    05/30/20 - Bug 31423833: OSON encode/decode errors
   dmcmahon    05/22/20 - Add jznErrorGetMessageBuf()
   bhammers    12/02/19 - bug 30601759: JZNERR_TRANS_INVALID_NAME 
   bhammers    11/26/19 - JZNERR_TRANS_NOT_OBJECT, JZNERR_TRANS_APPEND_CREATE
   bhammers    05/17/19 - bug 29794416: Add JZNERR_SCALAR_VALUE
   bhammers    05/21/19 - bug 29765376: add JZNERR_TRANS_CANT_UPDATE_ROOT 
   dmcmahon    03/11/19 - Add JZNERR_QBE_ORDERBY_ARR_STEP
   bhammers    04/04/19 - bug 29405826: add JZNERR_TRANS_NULL_RHS_VAL
   bhammers    03/01/19 - Bug 28441064: add JZNERR_INV_BOOL_TO_NUM_CONV
   bhammers    01/16/19 - adding more json_transform errors
   sriksure    09/27/18 - Bug 28717127: Rationalize includes in public jzn*.h
   bhammers    09/03/18 - add JSON_TRANSFORM errors
   zliu        07/03/18 - Bug 28287398:add JZNOCT3_OSON_PUPD_NOT_FEA
   dmcmahon    05/24/18 - Add JZNERR_PATH_EMPTY_STEP
   sriksure    03/01/18 - Add QBE error 337
   dmcmahon    02/13/18 - QBE errors 335 and 336
   dmcmahon    11/07/17 - CSV error messages
   dmcmahon    09/03/17 - Add JZNERR_PATH_CHAR_NOT_ESCAPED
   dmcmahon    09/02/17 - Bug 26741314: add JZNERR_QBE_ID_NOT_SINGLE
   bhammers    08/16/17 - add JZNERR_NUM_OVERFLOW
   dmcmahon    08/05/17 - Add JZNERR_PATH_ARRAY_OVERLAP
   dmcmahon    03/24/17 - Bug 25773794: wrong subclass error codes
   zliu        03/13/17 - Bug 25722364, 25722384:
                          sql/json path expr typecheck at compile time
                          JZNERR_PATH_WRG_ARG_TYP,JZNERR_PATH_TYPE_MISMATCH,
                          JZNERR_PATH_CONST_CVT_ERR
   dmcmahon    03/12/17 - Add jznErrorGetMessageVA
   dmcmahon    02/13/17 - Bug 25478895: add JZNERR_EMPTY_CACHE
   dmcmahon    09/29/16 - Bug 24758022: add JZNERR_INVALID_REGEX
   dmcmahon    08/20/16 - Add JZNERR_CS_CONV_FAILURE
   dmcmahon    07/17/16 - Add JZNERR_PATH_NESTING_OVERFLOW
   dmcmahon    05/24/16 - More QBE error messages.
   prthiaga    09/09/15 - Bug 21812442: Add JZNERR_PATH_BAD_FUNC_EXPR 
   bhammers    09/15/14 - add JZNERR_EMPTY_INPUT  
   dmcmahon    07/09/15 - Creation

*/

#ifndef JZNERR_ORACLE
# define JZNERR_ORACLE

#ifndef ORATYPES
# include <oratypes.h> 
#endif

#ifndef ORASTRUC
# include <orastruc.h>
#endif

#ifndef XML_ORACLE
# include <xml.h>
#endif

#ifndef JZNTYPES_ORACLE
# include <jzntypes.h>
#endif

/*---------------------------------------------------------------------------
                     PUBLIC TYPES AND CONSTANTS
  ---------------------------------------------------------------------------*/

#define JZNERR_OK                      0
#define JZNERR_END_OF_INPUT            1 /* End of input - not a hard error  */
#define JZNERR_INVALID_ARGUMENT        2 /* Argument or value not allowed    */
#define JZNERR_MISSING_INPUT           3 /* Parse attempted without input    */
#define JZNERR_CANT_OPEN_FILE          4 /* File not found                   */
#define JZNERR_SYNTAX_ERROR            5 /* Invalid JSON syntax              */
#define JZNERR_INVALID_ESCAPE          6 /* Unrecognized \ escape sequence   */
#define JZNERR_DUPLICATE_KEYS          7 /* Duplicate field key in object    */
#define JZNERR_PARSE_IN_PROGRESS       8 /* Parser not reset before input    */
#define JZNERR_UNSUPPORTED_ENCODING    9 /* Unsupported char set conversion  */
#define JZNERR_BAD_CHARACTER_DATA     10 /* Invalid UTF-8 byte sequence      */
#define JZNERR_INPUT_READ_ERROR       11 /* I/O error on stream or file      */
#define JZNERR_PREMATURE_END_OF_DATA  12 /* Required data not available      */
#define JZNERR_STACK_OVERFLOW         13 /* Json nested too many levels      */
#define JZNERR_STACK_UNDERFLOW        14 /* Json parser internal error       */
#define JZNERR_NAME_TOO_LONG          15 /* Json name is too long            */
#define JZNERR_PRINT_STREAM_ERROR     16 /* serializer: stream error         */
#define JZNERR_PRINT_BUF_ERROR        17 /* serializer: buffer error         */
#define JZNERR_PRINT_DATA_TOO_LARGE   18 /* serializer: input size too large */
#define JZNERR_PRINT_MISSING_WRAPPER  19 /* cannot print without wrapper     */
#define JZNERR_PRINT_MISSING_DATA     20 /* nothing to serialize,empty result*/
#define JZNERR_PRINT_NO_SCALAR        21 /* value to be printed not scalar   */
#define JZNERR_EXTRA_INPUT            22 /* Extra bytes after end of input   */
#define JZNERR_DOCLEN_MISMATCH        23 /* Length doesn't match data        */
#define JZNERR_INVALID_MEMBER_CODE    24 /* Unrecognized data type code      */
#define JZNERR_INVALID_BINARY_DATA    25 /* invalid binary data              */
#define JZNERR_TOO_MANY_LEVELS        26 /* Too many nested levels           */
#define JZNERR_INTERNAL               27 /* JSON internal error              */
#define JZNERR_OUT_OF_MEMORY          28 /* Out of memory                    */
#define JZNERR_DATA_VALUE_TOO_LARGE   29 /* Oversized data member            */
#define JZNERR_NULL_POINTER           30 /* Null pointer exception           */
#define JZNERR_INVALID_PATH           31 /* Invalid path expression          */
#define JZNERR_INVALID_ARRAY_INDEX    32 /* Invalid array index in path      */
#define JZNERR_INDEX_WRONG_ORDER      33 /* Array index in wrong order       */
#define JZNERR_FIELD_NAME_TOO_LONG    34 /* field name too long              */
#define JZNERR_TOT_FLD_NAMES_TOO_LONG 35 /* total field names too long       */
#define JZNERR_ARRAY_OUT_OF_BOUNDS    36 /* array out of bounds access       */
#define JZNERR_DOM_OUT_OF_BOUND       36 /* ### Deprecated name              */
#define JZNERR_INVALID_NUM            37 /* invalid number                   */
#define JZNERR_INVALID_DATETIME       38 /* invalid datetime                 */
#define JZNERR_SZ_OVFLW               39 /* size overflow for set encoding   */
#define JZNERR_INVALID_SPEC           40 /* invalid patch/diff/update spec   */
#define JZNERR_PATCH_FAILED           41 /* patch failed against target      */
#define JZNERR_READ_ONLY              42 /* cannot modify, read only target  */
#define JZNERR_UNIMPLEMENTED          43 /* feature not implemented          */
#define JZNERR_CONVERSION_FAILED      44 /* conversion failed (eg geojson)   */
#define JZNERR_BAD_EVENT_SEQUENCE     45 /* invalid event stream             */
#define JZNERR_DOM_KEY_NOT_EXIST      46 /* requested key does not exist     */
#define JZNERR_DOM_CYCLIC_REF         47 /* attempted cyclic reference       */
#define JZNERR_DOM_MULTIPLE_REF       48 /* multiple references to node      */
#define JZNERR_EMPTY_INPUT            49 /* input empty (e.g. empty lob)     */
#define JZNERR_CS_CONV_FAILURE        50 /* charset conversion failed        */
#define JZNERR_INVALID_REGEX          51 /* invalid regular expression       */
#define JZNERR_EMPTY_CACHE            52 /* no cached value available        */
#define JZNERR_DOM_MUST_BE_OBJ        53 /* DOM operation must be on object  */
#define JZNERR_DOM_MUST_BE_ARR        54 /* DOM operation must be on array   */
#define JZNERR_DOM_NOT_SCALAR         55 /* target of DOM op must be scalar  */
#define JZNERR_NUM_OVERFLOW           56 /* number outside range, +inf, -inf */
#define JZNERR_INV_BOOL_TO_NUM_CONV   57 /* invalid Bool to num conversion   */
#define JZNERR_SCALAR_VALUE           58 /* the JSON is a scalar value       */
#define JZNERR_BAD_SCALAR_TYPE        59 /* Unsupported scalar type          */
#define JZNERR_DOM_BAD_NODE_REF       60 /* Use of node after freeing it     */
#define JZNERR_OPERATION_DISALLOWED   61 /* Operation is disabled/disallowed */

#define JZNERR_PARSE_EXTRA_COMMA      70 /* unexpected comma                 */
#define JZNERR_PARSE_MISSING_COMMA    71 /* missing comma                    */
#define JZNERR_PARSE_MISSING_VAL      72 /* field name ":" with no value     */
#define JZNERR_PARSE_MISSING_COLON    73 /* field name not followed by ":"   */
#define JZNERR_PARSE_UNQUOTED_FIELD   74 /* field name not in double quotes  */
#define JZNERR_PARSE_UNQUOTED_VAL     75 /* item value not in double quotes  */
#define JZNERR_PARSE_INVALID_NUMBER   76 /* number could not be parsed       */
#define JZNERR_PARSE_INVALID_BYTE     77 /* disallowed ASCII byte            */
#define JZNERR_PARSE_INVALID_LITERAL  78 /* must be true/false/null          */
#define JZNERR_PARSE_UNCLOSED_QUOTE   79 /* unclosed quoted string           */
#define JZNERR_OSON_PUPD_NOT_FEA      80 /* OSON partial update not feasible */
#define JZNERR_PARSE_INVALID_HEX      81 /* hex conversion failed            */
#define JZNERR_PARSE_INVALID_SYNTAX   82 /* unexpected syntactic character   */
#define JZNERR_PARSE_UNCLOSED_ARRAY   83 /* unclosed array                   */
#define JZNERR_PARSE_UNCLOSED_OBJECT  84 /* unclosed object                  */
#define JZNERR_PARSE_UNINTELLIGIBLE   85 /* JSON contains invalid chars      */
#define JZNERR_PARSE_MISSING_ITEM     86 /* missing next field or element    */
#define JZNERR_PARSE_EXCESS_CHARS     87 /* trailing junk chars after JSON   */
/*
** (through 89 reserved for path parser failures)
*/
#define JZNERR_OSON_CORRUPT           90 /* Corrupt/invalid binary JSON      */ 
#define JZNERR_OSON_PREMATURE_END     91 /* OSON image ends prematurely      */
#define JZNERR_OSON_TOO_LARGE         92 /* JSON too large for OSON encoding */


/*
** Range from 100-199 reserved for warnings (due to old XML infrastructure)
*/

#define JZNERR_PATH_UNQUOTED_PUNCT    201
#define JZNERR_PATH_UNQUOTED_ESCAPE   202
#define JZNERR_PATH_INVALID_ESCAPE    203
#define JZNERR_PATH_BAD_SURROGATES    204
#define JZNERR_PATH_ARRAY_TOO_LONG    205
#define JZNERR_PATH_ARRAY_BAD_RANGE   206
#define JZNERR_PATH_BAD_VARIABLE      207
#define JZNERR_PATH_DOLLAR_START      208
#define JZNERR_PATH_TRAILING_CHARS    209
#define JZNERR_PATH_ARRAY_EMPTY       210
#define JZNERR_PATH_ARRAY_INVALID     211
#define JZNERR_PATH_ARRAY_WILDCARD    212
#define JZNERR_PATH_OBJECT_WILDCARD   213
#define JZNERR_PATH_UNCLOSED_ARRAY    214
#define JZNERR_PATH_BAD_ARRAY         215
#define JZNERR_PATH_UNCLOSED_QUOTE    216
#define JZNERR_PATH_BAD_KEY           217
#define JZNERR_PATH_NOT_LITERAL       218
#define JZNERR_PATH_BAD_LITERAL       219
#define JZNERR_PATH_INVALID_NUMBER    220
#define JZNERR_PATH_NUMBER_OVERFLOW   221
#define JZNERR_PATH_BAD_REGEX         222
#define JZNERR_PATH_UNSUPPORTED_OP    223
#define JZNERR_PATH_BAD_COMPARISON    224
#define JZNERR_PATH_COMPARE_PATH      225
#define JZNERR_PATH_NOT_PAREN         226
#define JZNERR_PATH_COMPARE_BOOLEAN   227
#define JZNERR_PATH_QUESTION_PAREN    228
#define JZNERR_PATH_CLOSE_PAREN       229
#define JZNERR_PATH_BAD_EXPR          230
#define JZNERR_PATH_UNQUOTED_UNICODE  231
#define JZNERR_PATH_EXPECTED_DOT      232
#define JZNERR_PATH_BAD_STEP_START    233
#define JZNERR_PATH_AT_NOT_IN_PRED    234
#define JZNERR_PATH_DOLLAR_IN_PRED    235
#define JZNERR_PATH_UNKNOWN_FNOP      236
#define JZNERR_PATH_LHS_RHS           237
#define JZNERR_PATH_MULTIPLE_PREDS    238
#define JZNERR_PATH_PRED_NOT_LAST     239
#define JZNERR_PATH_ARRAY_BAD_CHARS   240
#define JZNERR_PATH_BAD_FUNC_EXPR     241
#define JZNERR_PATH_NESTING_OVERFLOW  242
#define JZNERR_PATH_TYPE_MISMATCH     243
#define JZNERR_PATH_WRG_ARG_TYP       244
#define JZNERR_PATH_CONST_CVT_ERR     245
#define JZNERR_PATH_IN_NOTIN_UNIFORM_TYP       246
#define JZNERR_PATH_ARRAY_OVERLAP     247
#define JZNERR_PATH_CHAR_NOT_ESCAPED  248
#define JZNERR_PATH_EMPTY_STEP        249

#define JZNERR_PATCH_ROOT_ARRAY       250
#define JZNERR_PATCH_OP_OBJECT        251
#define JZNERR_PATCH_MISSING_OP_CODE  252
#define JZNERR_PATCH_INVALID_PATH     253
#define JZNERR_PATCH_INVALID_OP_CODE  254
#define JZNERR_PATCH_UNKNOWN_FIELD    255
#define JZNERR_PATCH_PATH_REQUIRED    256
#define JZNERR_PATCH_MUST_BE_STRING   257
#define JZNERR_PATCH_FROM_NOT_ALLOWED 258
#define JZNERR_PATCH_VAL_NOT_ALLOWED  259
#define JZNERR_PATCH_FROM_REQUIRED    260
#define JZNERR_PATCH_VAL_REQUIRED     261
#define JZNERR_PATCH_CYCLIC_MOVE      262

#define JZNERR_QBE_CONFLICTING_RULES  300
#define JZNERR_QBE_PROJECTION_OBJECT  301
#define JZNERR_QBE_BAD_RULE_VALUE     302
#define JZNERR_QBE_INVALID_PATH       303
#define JZNERR_QBE_KEY_SCALARS        304
#define JZNERR_QBE_ARRAY_NO_VALUES    305
#define JZNERR_QBE_ARRAY_SCALARS_ONLY 306
#define JZNERR_QBE_BETWEEN_TWO_VALUES 307
#define JZNERR_QBE_SPATIAL_OUTERMOST  308
#define JZNERR_QBE_GEOMETRY_MISSING   309
#define JZNERR_QBE_DIST_NOT_ALLOWED   310
#define JZNERR_QBE_DISTANCE_NUMERIC   311
#define JZNERR_QBE_UNIT_STRING        312
#define JZNERR_QBE_SPATIAL_UNKNOWN    313
#define JZNERR_QBE_DISTANCE_REQUIRED  314
#define JZNERR_QBE_EMPTY_OBJECT       315
#define JZNERR_QBE_NOT_FOLLOWED_BY    316
#define JZNERR_QBE_SPATIAL_TARGET     317
#define JZNERR_QBE_NOT_OP_DISALLOWED  318
#define JZNERR_QBE_MUST_BE_STRING     319
#define JZNERR_QBE_MUST_BE_SCALAR     320
#define JZNERR_QBE_MUST_BE_COMPARABLE 321
#define JZNERR_QBE_MUST_BE_ARRAY      322
#define JZNERR_QBE_CONJUNCTION_OBJS   323
#define JZNERR_QBE_ID_OUTERMOST       324
#define JZNERR_QBE_MIXED_DOWNSCOPE    325
#define JZNERR_QBE_ORDERBY_INTEGER    326
#define JZNERR_QBE_QUERY_OBJECT       327
#define JZNERR_QBE_ORDERBY_CONTAINER  328
#define JZNERR_QBE_ORDERBY_NOPATH     329
#define JZNERR_QBE_ORDERBY_INVALID    330
#define JZNERR_QBE_SQLJ_OUTERMOST     331
#define JZNERR_QBE_MUST_BE_OBJECT     332
#define JZNERR_QBE_MUST_BE_PATH       333
#define JZNERR_QBE_ID_NOT_SINGLE      334
#define JZNERR_QBE_MUST_BE_BOOLEAN    335
#define JZNERR_QBE_PARAM_CONFLICT     336
#define JZNERR_QBE_LOGICAL_ARR_EMPTY  337
#define JZNERR_QBE_ORDERBY_ARR_STEP   338

#define JZNERR_QBE_OPER_OUTERMOST     375 /* %s must be at outer level       */
#define JZNERR_QBE_CONTAINS_COMBO     376 /* Cannot combine with %s          */


#define JZNERR_CSV_UNCLOSED_QUOTE     381 /* unclosed quoted string          */
#define JZNERR_CSV_UNESCAPED_QUOTE    382 /* naked quote mark                */
#define JZNERR_CSV_MISSING_DELIMITER  383 /* no delimiter between strings    */
#define JZNERR_CSV_UNQUOTED_SPACES    384 /* space before/after quotes value */

/*
** Range from 400-449 reserved for JSON_TRANSFORM
*/
#define JZNERR_TRANS_MISSING_DATA     400  /* path expr did not select value  */
#define JZNERR_TRANS_EXISTING_DATA    401      /* path expr selected a value  */
#define JZNERR_TRANS_NOT_ARRAY        402 /* the selected val is not an array */
#define JZNERR_TRANS_FIELD_EXISTS     403      /* a field name already exists */
#define JZNERR_TRANS_FIELD_MISSING    404      /* a field name does not exist */
#define JZNERR_TRANS_CANT_REMOVE_ROOT 405 /* removal of root node not allowed */
#define JZNERR_TRANS_UNSUPPORTED_OP   406  /* unsupported transform operation */
#define JZNERR_TRANS_BAD_MULTI_SEL    407    /* multi select not allowed: .*  */
#define JZNERR_TRANS_NULL_RHS_VAL     408    /* rhs is null and ERROR ON NULL */
#define JZNERR_TRANS_CANT_UPDATE_ROOT 409  /* update of root node not allowed */
#define JZNERR_TRANS_NOT_OBJECT       410    /* selected val is not an object */
#define JZNERR_TRANS_APPEND_CREATE    411    /* invalid combination of append */
#define JZNERR_TRANS_INVALID_NAME     412       /* invalid key name in RENAME */

// next error code outside JSON_TRANSFORM range starts with 450

#define JZNERR_WRONG_NUM_TYPE_ARGS    499     /* wrong number/type arguments */

/*
** This requests using a generic code from the XMLERR range to support
** legacy error handlers.
*/
#define JZNERR_DEFAULT                -1

/*
** Fatal error handler (for asserts or throws).
*/
typedef void (*jznFatalErrHandler)(xmlctx *, oratext *);

/*---------------------------------------------------------------------------
                           INTERNAL FUNCTIONS
  ---------------------------------------------------------------------------*/

/*
** Format an error message and write the result to a user-supplied buffer.
*/
void jznErrorFormat(xmlctx *xctx,
                    oratext *outbuf, ub4 outlen,
                    sword errcode, ...);

/*
** Format an error message using a standard error buffer from the xmlctx.
** Note that since the xctx has a single buffer, the caller should copy
** or render the message before calling other code that may overwrite it.
** Optionally prepend a newline (addnl).
*/
oratext *jznErrorGetMessageNL(xmlctx *xctx, boolean addnl, sword errcode, ...);
oratext *jznErrorGetMessage(xmlctx *xctx, sword errcode, ...);
oratext *jznErrorGetMessageVA(xmlctx *xctx, boolean addnl, sword errcode,
                              va_list va);

oratext *jznErrorGetMessageBuf_VA(xmlctx *xctx, oratext *msgbuf, ub4 bufsiz,
                                  boolean addnl, sword errcode, va_list va);
oratext *jznErrorGetMessageBuf(xmlctx *xctx, oratext *msgbuf, ub4 bufsiz,
                               boolean addnl, sword errcode, ...);

/*
** This calls a handler that will exit your current thread of control
** either via a longjmp or with an exit to the operating system. For
** the sake of the longjmp handler, <errmsg> must be memory that is
** stable with respect to a stack unwind, typically a constant string
** from an ASSERT, although heap-allocated memory from an object that
** will survive the longjmp is acceptable. Do not use a local variable!
*/
void jznuFireFatalError(xmlctx *xctx, oratext *errmsg);

/*
** Fire the assert handler, if available. Otherwise call the fatal handler.
*/
void jznerrAssert(jznFatalErrHandler errh, xmlctx *xctx, oratext *errmsg);

/*
** Emit a formatted JZN error message via XmlErrOut. If the error
** code is less than 0 (invalid), it will substitute the reserved
** "general error code" from the XMLERR enumeration. Otherwise, it will
** adjust the error code to the JSON reserved range in the XMLERR space.
** This typically returns to the caller. However, some implementations of
** the error handler will throw an exception and there won't be a return.
*/
void JznErrOut(xmlctx *xctx, sword errcode, oratext *msg);

/*
** Emit a JZN error message, with optional parameters, via XmlErrOut.
** This typically returns to the caller, and will return the errcode.
** However, some implementations of the error handler will throw an
** exception and there won't be a return.
*/
sword JznErrFmtOut(xmlctx *xctx, sword errcode, ...);

void  JznErrFmt(xmlctx *xctx, boolean addnl,
                oratext *outbuf, ub4 outlen,
                sword errcode, va_list va);

#endif                                              /* JZNERR_ORACLE */
