Rem
Rem $Header: dbgendev/src/langdata/plsql/lang_data.pks /main/44 2025/08/13 01:29:21 jiangnhu Exp $
Rem
Rem lang_data.pls
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      lang_data.pks - LangData Package
Rem
Rem    DESCRIPTION
Rem      This package contains specifications of the external APIs provided 
Rem      as part of LangData.
Rem
Rem    NOTES
Rem      NONE
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/lang_data.pks
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    08/04/25 - Move grant_read_on_user_tables to lang_data pkg
Rem    sathyavc    08/01/25 - DBAI-1087: Add function to populate filter values
Rem                           into URL
Rem    dadoshi     07/31/25 - Add p_sample_queries to generate_sample_queries
Rem    fgurrola    07/30/25 - BAI-1130: remove_expected_report_id and
Rem                           remove_expected_drilldown_id added.
Rem    ruohli      07/24/25 - DBAI-1076: Added update_domain, delete_domain, and
Rem                           get_all_domains
Rem    dadoshi     07/24/25 - JIRA_DBAI1056: Add
Rem                           get_report_match_document_by_id
Rem    dadoshi     07/24/25 - JIRA_DBAI1080: Add
Rem                           get_drilldown_match_document_by_id
Rem    saloshah    07/24/25 - DBAI-1052: Add pipeline functions to 
Rem                           get user search records 
Rem    arevathi    07/23/25 - Added out parameter for add sample queries
Rem    dadoshi     07/22/25 - JIRA_DBAI1080: Added get_drilldown_status_by_id
Rem    dadoshi     07/22/25 - JIRA_DBAI1056, JIRA_DBAI1049: Add pipeline
Rem                           functions to fetch all reports and drilldowns
Rem    dadoshi     07/22/25 - JIRA_DBAI1056: Added get_report_status_by_id
Rem    ruohli      07/21/25 - DBAI-1111: Add Drilldown Ranking Frequency, 
Rem                           Matched Queries of Drilldown at Rank, and
Rem                           Drilldown filter usage as external API
Rem    pryarla     07/21/25 - DBAI-1075: Generate sample queries using LLM
Rem    deveverm    07/18/25 - DBAI-1050: added p_is_new_report parameter to
Rem                           get_report_regression
Rem    ruohli      07/17/25 - DBAI-1089: Add Report Ranking Frequency, 
Rem                           Matched Queries of Reports at Rank, and
Rem                           report filter usage as external API
Rem    jiangnhu    07/16/25 - Add p_genAI_rerank_endpoint_id in 
Rem                           set_oci_credential
Rem    pryarla     07/16/25 - DBAI-882: Added get_top_filters function
Rem    sathyavc    07/10/25 - DBAI-881: Modify feedback count APIs to be
Rem                           filtered based on time.
Rem    sathyavc    07/10/25 - DBAI-881: Add top k most searched reports, least
Rem                           searched reports and most asked questions.
Rem    sathyavc    07/07/25 - DBAI-883: Add functions to track API metrics over 
Rem                           specified time windows.
Rem    arevathi    07/04/25 - Add purge_outdated_search_records procedure
Rem    saloshah    07/03/25 - DBAI-873: Added delete feedback API
Rem    arevathi    07/01/25 - Change onnx_model_dir to mount_dir
Rem    jiangnhu    06/25/25 - Sync parameters of search_from_query to internal
Rem                           API
Rem    jiangnhu    06/09/25 - DBAI-871: Implement APIs to get ID by unique
Rem                           combination of title, version, etc.
Rem    deveverm    06/02/25 - DBAI-794: modified set_oci_credential to take
Rem                           genAI endpoint
Rem    deveverm    05/16/25 - DBAI-761: added updated_sample_query_status,
Rem                           changed get_report_description_regression to
Rem                           get_report_regression, changed
Rem                           get_drilldown_description_regression to
Rem                           get_drilldown_regression
Rem    jiangnhu    04/12/25 - DBAI-739: Add plot_search_vectors
Rem    deveverm    04/10/25 - DBAI-721: added set_oci_credential
Rem    deveverm    04/01/25 - DBAI-523: added
Rem    jiangnhu    03/25/25 - Fix parameter order of search_from_query; expose
Rem                           APIs in config pkg
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/18/25 - DBAI-661: Add external APIs for user to update
Rem                           enumerable set
Rem    deveverm    03/18/25 - DBAI-546: added schema_name for cross_schema
Rem                           support
Rem    jiangnhu    03/06/25 - DBAI-542: Add get_schema_version for backend
Rem    deveverm    02/27/25 - DBAI_598: Fixed to add model_dir
Rem    jiangnhu    02/06/25 - DBAI-542: Add external APIs
Rem    dadoshi     01/22/25 - Add setup_db argument to init()
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE lang_data AUTHID CURRENT_USER AS
    /**
    * PROCEDURE: init
    * 
    * DESCRIPTION:
    * The `init` procedure initializes the LangData package by creating 
    * necessary roles, tables, and granting appropriate privileges. This 
    * procedure should be executed by a user with sufficient privileges to 
    * create roles, tables, and packages. It ensures the required 
    * infrastructure is established for the DataLang package. After executing
    * this procedure, the roles `lang_data_user` and `lang_data_app_expert` 
    * should be granted to the appropriate users.
    * 
    * This procedure performs the following tasks:
    * 1. Create Roles:
    *    - Creates a role for general users (`lang_data_user`).
    *    - Creates a role for application experts (`lang_data_app_expert`).
    * 
    * 2. Create Tables:
    *    - Creates all the Lang-Data related tables (with 'langdata$' prefix),
    *      if p_setup_db is set as True.
    * 
    * 3. Creates Triggers:
    *    - Creates all the Lang-Data related triggers (with 'langdata$' prefix),
    *      if p_setup_db is set as True.
    *
    * 4. Create Global Temp Tables
    *
    * 5. Create Python Fucntions
    * 
    * 6. Create Types
    * 
    * 7. Create User Table Indexes
    * 
    * 8. Create Resource Plan
    * 
    * 9. Populates the annotation table by retrieving the 'DESCRIPTION'
    *    annotations for all user tables and columns. 
    *
    * 10. Updates the langdata$schemaversion table with the version when init()
    *     is executed.
    *
    * PARAMETERS:
    *     - p_setup_db (IN BOOLEAN DEFAULT false):
    *           True, if the database instance is to be setup with Lang-Data
    *           specific tables, triggers, and procedures. If not provided, it
    *           defualts as False.
    *
    * USAGE:
    * To execute the `init` procedure, use the following SQL command:
    * 
    * BEGIN
    *     lang_data_setup_pkg.init(p_setup_db => true);
    * END;
    * Ensure that the user executing this command has the necessary privileges 
    * to create roles, tables, and grant privileges.
    * 
    * EXCEPTIONS:
    * 
    */
	PROCEDURE init(
        mount_dir IN VARCHAR2,
        -- true to enable execution of SQL Scripts that setup tables, 
        -- triggers, and procedures.
        p_setup_db IN BOOLEAN DEFAULT FALSE
    );
    PROCEDURE cleanup_langdata;
    
    PROCEDURE purge_outdated_search_records (
        p_days         IN NUMBER DEFAULT 180,
        p_cutoff_time  IN TIMESTAMP DEFAULT NULL,
        p_action       IN VARCHAR2 DEFAULT 'ARCHIVE',
        p_output_msg   OUT VARCHAR2
    );

    PROCEDURE migrate_schema;
    /*
    PROCEDURE: load_onnx_model

    DESCRIPTION:
        The `load_onnx_model` procedure can load onnx models into oracle 
        database. This is essentially a wrapper over 
        DBMS_VECTOR.load_embedding_model tweaked for langdata.
        On-Prem: In this environment, we run this procedure as part of our
        installation, and function uses server filesystem to load the onnx 
        model from.
        On-Cloud: In this environment, we expect users to run this procedure 
        after installation is completed, and it is a mandatory step as all 
        procedures need an embedding model to function. In this environment this procedure uses cloud bucket to load the onnx model from.
    
    PARAMETERS:
        model_path (IN VARCHAR2): 
            On_prem: Path to directory on server filesystem having onnx model.
            On_Cloud: Location URI of onnx model
        filename (IN VARCHAR2):
            The name of onnx file
        modelname (IN VARCHAR2):
            Name of model, the uploaded onnx model is set to this name.
        cred_name (IN VARCHAR2 DEFAULT NULL):
            Name of credential to use while downloading model from bucket.
    */
    PROCEDURE load_onnx_model(
        model_path IN VARCHAR2,
        filename   IN VARCHAR2,
        modelname  IN VARCHAR2,
        cred_name  IN VARCHAR2 DEFAULT NULL
    );

    /*
    ----------------------------------------------------------------------------
    Package Name: lang_data_config_pkg
    Description: This package contains procedures related to the configuration
    parameters used in various LangData APIs. These procedures are designed to 
	handle CRUD operations on these parameters.
    ----------------------------------------------------------------------------
    Author: Daksh Doshi
    Created: 02/24/2025
    ----------------------------------------------------------------------------
    Public Procedures:
    ----------------------------------------------------------------------------
    Function get_config_variable_names
    ----------------------------------------------------------------------------
    Description:
        This function returns the list of the names of the configuration 
        parameters, concatenated as a string.

    Parameters:
        NONE

    Returns (VARCHAR2):
        List of comma-seperated configuration parameter names as a string.

    Exceptions:
        NONE
    ----------------------------------------------------------------------------
    Function get_config_parameter
    ----------------------------------------------------------------------------
    Description:
        This function returns the value of a configuration parameter based on
        the given name of the configuration parameter.

    Parameters:
        - p_name (IN VARCHAR2):
            Name of the configuration parameter.

    Returns:
        Value of the configuration parameter.

    Exceptions:
        - c_resource_not_found (-20004):
            Raised when no report is found for the given report ID. A custom 
            error with code -20004 is raised, using the lang_data_errors_pkg 
            package, with the message "Resource not found".
    ----------------------------------------------------------------------------
    Procedure update_config_parameter
    ----------------------------------------------------------------------------
    Description:
        This procedure updates the value of the given configuration parameter to
        the one provided.

    Parameters:
        - p_name (IN VARCHAR2):
            Name of the configuration parameter.

        - p_value (IN VARCHAR2):
            New value of the configuration parameter.

    Exceptions:
        - c_unauthorized_code (-20000):
            Raised when the user does not have the required role 
            (LANG_DATA_APP_EXPERT) to access the procedure. This triggers a 
            custom error with code -20000 using the lang_data_errors_pkg
            package.

        - c_invalid_parameters_code (-20002):
            Raised when the report or description parameters are invalid. 
            Logged as an error before raising.

        - c_resource_not_found (-20004):
            Raised when no report is found for the given report ID. A custom 
            error with code -20004 is raised, using the lang_data_errors_pkg 
            package, with the message "Resource not found".
    ----------------------------------------------------------------------------
    */
    FUNCTION get_config_parameter (
        p_name IN VARCHAR2
    ) RETURN VARCHAR2;

    FUNCTION get_config_variable_names RETURN VARCHAR2;

    PROCEDURE update_config_parameter (
        p_name IN VARCHAR2,
        p_value IN VARCHAR2
    );

    FUNCTION set_oci_credential (
        p_cred_name IN VARCHAR2,
        p_compartment_id IN VARCHAR2,
        p_region_code IN VARCHAR2,
        p_genAI_endpoint IN VARCHAR2,
        p_genAI_rerank_endpoint_id IN VARCHAR2
    ) RETURN VARCHAR2;
    
    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_analytics_pkg
    Description: This package contains procedures related to analytics of 
    langdata. Each procedure returns different metrics over the langdata search 
    record tables
    ----------------------------------------------------------------------
    */
    PROCEDURE get_metrics(
        p_timestamp             IN TIMESTAMP DEFAULT NULL,
        p_metrics               OUT JSON_OBJECT_T
    );

    FUNCTION get_top_filters(
        p_top_k         IN NUMBER,
        p_timestamp     IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Public procedure to get top k most searched reports at or after input
    -- timestamp
    FUNCTION get_top_k_most_searched_reports(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Get summary statistics for all external APIs at or after a specified
    -- timestmap
    FUNCTION get_all_api_stats (
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Get daily statistics for specified external API at or after a specified
    -- timestmap
    FUNCTION get_api_daily_stats (
        p_api_name           IN VARCHAR2,
        p_num_error_messages IN NUMBER DEFAULT 5,
        p_timestamp          IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Get failure statistics for specified external API at or after a specified
    -- timestmap
    FUNCTION get_api_failures (
        p_api_name  IN VARCHAR2,
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Public procedure to get top k least searched reports at or after input
    -- timestamp
    FUNCTION get_bottom_k_least_searched_reports(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Public procedure to get top k most asked questions at or after input
    -- timestamp
    FUNCTION get_top_k_most_asked_questions(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Public procedure to get top k questions with the highest number of 
    -- negative feedbacks, asked questions at or after input timestamp
    FUNCTION get_top_k_most_negative_feedback_questions(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB;

    -- Public procedure to get top-1,3,5 count given report id
    -- timestamp
    PROCEDURE get_top_report_metrics(
       p_report_id     IN VARCHAR2,
       p_metrics       OUT JSON_OBJECT_T
    );

    -- Public procedure to get query given report_id, target rank, and k number
    -- of queries
    -- timestamp
    PROCEDURE get_matched_queries_by_report_rank (
        p_report_id IN VARCHAR2,
        p_rank      IN PLS_INTEGER,
        p_k         IN PLS_INTEGER,
        p_queries   OUT SYS.ODCIVARCHAR2LIST
    );

    -- Public procedure to get count of default/non-default filter usage
    -- timestamp
    PROCEDURE get_report_filter_usage(
        p_report_id in VARCHAR2,
        p_default_count OUT NUMBER,
        p_non_default_count OUT NUMBER
    );

    -- Public procedure to get top-1,3,5 count given drilldown id
    -- timestamp
    PROCEDURE get_top_drilldown_metrics(
        p_drilldown_id  IN  VARCHAR2,
        p_metrics       OUT JSON_OBJECT_T
    );

	-- Public procedure to get query given drilldown, target rank, and k number
    -- of queries
    -- timestamp
    PROCEDURE get_matched_queries_by_drilldown_rank (
        p_drilldown_id IN VARCHAR2,
        p_rank      IN PLS_INTEGER,
        p_k         IN PLS_INTEGER,
        p_queries   OUT SYS.ODCIVARCHAR2LIST
    );

    -- Public procedure to get count of default/non-default filter usage from 
    -- Drilldown Report
    -- timestamp
    PROCEDURE get_drilldown_filter_usage(
        p_drilldown_id in VARCHAR2,
        p_default_count OUT NUMBER,
        p_non_default_count OUT NUMBER
    );


    /*
    ----------------------------------------------------------------------
    Package Name: lang_date_annotations_pkg
    Description: This package contains procedures and triggers related to 
                 updating annotations for user_tables. The procedures and 
                 triggers are designed to handle updating annotations and 
                 all the related updating operations.
    ----------------------------------------------------------------------
    */
    PROCEDURE update_annotation_values(
        p_table_name      IN VARCHAR2,
        p_column_name     IN VARCHAR2,
        p_annotation_value IN VARCHAR2,
        p_schema_name     IN VARCHAR2
    );


    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_reports_pkg
    Description: This package contains procedures related to the report
	and report descriptions. These procedures are designed to handle CRUD
	and validation operations on reports/report descriptions.
    ----------------------------------------------------------------------
    */
    -- Public procedure for retriving paginated report descriptions
    PROCEDURE get_report_descriptions_paginated (
        p_report_id     IN VARCHAR2,
        p_limit         IN NUMBER DEFAULT 10,
        p_cursor        IN OUT VARCHAR2,
        p_descriptions  OUT SYS_REFCURSOR
    );

	-- Public procedure declaration for retriving a report
	PROCEDURE get_report_paginated (
		p_id IN VARCHAR2,
		p_title OUT VARCHAR2,
		p_match_document OUT JSON,
		p_status OUT VARCHAR2,
		p_descriptions OUT SYS_REFCURSOR,
        p_description_cur IN OUT VARCHAR2,
        p_descriptions_limit IN NUMBER DEFAULT 10,
		p_sample_queries OUT SYS_REFCURSOR,
        p_sample_query_cur IN OUT VARCHAR2,
        p_sample_query_limit IN NUMBER DEFAULT 10
	);

    -- Public Procedure to create a report
    PROCEDURE create_report (
        p_title                 IN VARCHAR2,
        p_match_document        IN JSON,
        p_description_text      IN VARCHAR2,
        p_description_status    IN VARCHAR2 DEFAULT 'Pending Regression',
        p_report_status         IN VARCHAR2,
        p_sample_queries        IN SYS.ODCIVARCHAR2LIST,
        p_domain                IN VARCHAR2 DEFAULT NULL,
        p_report_id             OUT VARCHAR2,
        p_description_id        OUT VARCHAR2
    );

    -- Public Procedure to fetch regression in a report
    PROCEDURE get_report_regression(
        p_description_id  IN VARCHAR2 DEFAULT NULL,
        p_sample_query_id IN VARCHAR2 DEFAULT NULL,
        p_is_new_report       IN BOOLEAN DEFAULT FALSE,
        p_force               IN BOOLEAN,
        p_json                OUT JSON
    );

    -- Public Procedure to fetch all reports
    PROCEDURE get_all_reports (
        p_status            IN VARCHAR2 DEFAULT NULL,
        p_cursor            IN OUT VARCHAR2,
        p_limit             IN NUMBER DEFAULT 10,
        p_reports           OUT SYS_REFCURSOR
    );

    -- Public Function to retrieve reports row by row in APEX
    FUNCTION get_all_reports_fn (
        p_status IN VARCHAR2 DEFAULT NULL,
        p_limit  IN NUMBER DEFAULT 10
    ) RETURN report_row_table PIPELINED;

    FUNCTION get_report_descriptions_fn (
        p_report_id IN VARCHAR2,
        p_limit     IN NUMBER DEFAULT 10
    ) RETURN data_table PIPELINED;

    -- Public Procedure to delete report
    PROCEDURE delete_report(
        p_report_id         IN VARCHAR2
    );

    -- Public Procedure to add report description
    PROCEDURE add_report_description(
        p_report_id         IN VARCHAR2,
        p_text              IN VARCHAR2,
        p_status            IN VARCHAR2 DEFAULT 'Pending Regression',
        p_description_id    OUT VARCHAR2
    );

    -- Public Procedure to delete report description
    PROCEDURE delete_report_description(
        p_description_id    IN VARCHAR2
    );

    -- Public Procedure to update report description
    PROCEDURE update_report_description(
        p_id                IN VARCHAR2,
        p_text              IN VARCHAR2
    );

    -- Public Procedure to fetch report description
    PROCEDURE get_report_description(
        p_id                IN VARCHAR2,
        p_text              OUT VARCHAR2,
        p_version           OUT NUMBER,
        p_status            OUT VARCHAR2,
        p_report_id         OUT VARCHAR2,
        p_enhanced_text      OUT VARCHAR2
    );

    PROCEDURE execute_report_sql(
        p_report_id IN VARCHAR2,
        p_filter_values IN JSON,
        p_columns OUT SYS.ODCIVARCHAR2LIST,
        p_data OUT CLOB
    );

    PROCEDURE replace_report_filter_enumerable_set (
        p_report_id             IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_new_enumerable_set    IN  JSON_ARRAY_T
    );

    PROCEDURE add_into_report_filter_enumerable_set (
        p_report_id                IN  VARCHAR2,
        p_filter_name              IN  VARCHAR2,
        p_enumerable_set_to_add    IN  JSON_ARRAY_T
    );

    PROCEDURE remove_from_report_filter_enumerable_set (
        p_report_id                   IN  VARCHAR2,
        p_filter_name                 IN  VARCHAR2,
        p_enumerable_set_to_remove    IN  JSON_ARRAY_T
    );

    PROCEDURE update_report_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    );

    PROCEDURE update_report_description_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    );

    FUNCTION get_report_id_by_title(
        p_title IN VARCHAR2
    ) RETURN VARCHAR2;

    FUNCTION get_desc_id_by_report_id_version(
        p_report_id    IN VARCHAR2,
        p_version      IN NUMBER
    ) RETURN VARCHAR2;

    FUNCTION get_report_status_by_id (
        p_report_id     IN VARCHAR2
    ) RETURN VARCHAR2;

    FUNCTION get_report_match_document_by_id(
        p_report_id         VARCHAR2
    ) RETURN CLOB;
    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_drilldowns_pkg
    Description: This package contains procedures related to the drilldown
	and drilldown descriptions. These procedures are designed to handle 
    CRUD and validation operations on drilldown documents and their 
    descriptions.
    ----------------------------------------------------------------------
    */
    -- Public procedure declaration for retriving a drilldown
    PROCEDURE get_drilldown_paginated (
        p_id                  IN VARCHAR2,
        p_report_id           OUT VARCHAR2,
        p_title               OUT VARCHAR2,
		p_match_document      OUT JSON,
		p_status              OUT VARCHAR2,
        p_descriptions        OUT SYS_REFCURSOR,
        p_descriptions_cur    IN OUT VARCHAR2,
        p_descriptions_limit  IN NUMBER DEFAULT 10,
        p_sample_queries      OUT SYS_REFCURSOR,
        p_sample_query_cur    IN OUT VARCHAR2,
        p_sample_query_limit  IN NUMBER DEFAULT 10
    );

    FUNCTION get_all_drilldowns_fn (
        p_report_id IN VARCHAR2,
        p_status IN VARCHAR2 DEFAULT NULL,
        p_limit  IN NUMBER DEFAULT 10
    ) RETURN drilldown_row_table PIPELINED;

    FUNCTION get_drilldown_descriptions_fn (
        p_drilldown_id IN VARCHAR2,
        p_limit        IN NUMBER DEFAULT 10
    ) RETURN drilldown_description_table PIPELINED;

    PROCEDURE delete_drilldown (
        p_drilldown_id IN VARCHAR2
    );

    PROCEDURE delete_drilldown_description (
        p_description_id IN VARCHAR2
    );

    PROCEDURE update_drilldown_description (
        p_drilldown_description_id IN  VARCHAR2,
        p_new_description          IN  VARCHAR2
    );

    PROCEDURE create_drilldown(
        p_title               IN VARCHAR2,
        p_report_id           IN VARCHAR2,
        p_match_document      IN JSON,
        p_description_text    IN VARCHAR2,
        p_description_status  IN VARCHAR2 DEFAULT 'Pending Regression',
        p_drilldown_status    IN VARCHAR2,
        p_sample_queries      IN SYS.ODCIVARCHAR2LIST,
        p_domain              IN VARCHAR2 DEFAULT NULL,
        p_drilldown_id        OUT VARCHAR2,
        p_description_id      OUT VARCHAR2
    );

    PROCEDURE add_drilldown_description(
        p_drilldown_id          IN VARCHAR2,
        p_text                  IN VARCHAR2,
        p_status                IN VARCHAR2 DEFAULT 'Pending Regression',
        p_description_id        OUT VARCHAR2
    );

    PROCEDURE execute_drilldown_sql(
        p_drilldown_id   IN VARCHAR2,
        p_filter_values  IN JSON,
        p_columns        OUT SYS.ODCIVARCHAR2LIST,
        p_data           OUT CLOB
    );

    PROCEDURE get_drilldown_description (
        p_description_id  IN VARCHAR2,
        p_text OUT VARCHAR2,
        p_version OUT NUMBER,
        p_status OUT VARCHAR2,
        p_drilldown_id OUT VARCHAR2,
        p_enhanced_text OUT VARCHAR2
    );

    PROCEDURE get_drilldown_descriptions (
        p_drilldown_id IN VARCHAR2,
        p_descriptions OUT SYS_REFCURSOR
    );

    PROCEDURE get_drilldown_regression(
        p_description_id  IN VARCHAR2 DEFAULT NULL,
        p_sample_query_id IN VARCHAR2 DEFAULT NULL,
        p_force               IN BOOLEAN,
        p_json                OUT JSON
    );

    PROCEDURE replace_drilldown_filter_enumerable_set (
        p_drilldown_id          IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_new_enumerable_set    IN  JSON_ARRAY_T
    );

    PROCEDURE add_into_drilldown_filter_enumerable_set (
        p_drilldown_id             IN  VARCHAR2,
        p_filter_name              IN  VARCHAR2,
        p_enumerable_set_to_add    IN  JSON_ARRAY_T
    );

    PROCEDURE remove_from_drilldown_filter_enumerable_set (
        p_drilldown_id                   IN  VARCHAR2,
        p_filter_name                 IN  VARCHAR2,
        p_enumerable_set_to_remove    IN  JSON_ARRAY_T
    );

    PROCEDURE update_drilldown_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    );

    PROCEDURE update_drilldown_description_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    );

    FUNCTION get_drilldown_id_by_title(
        p_title IN VARCHAR2
    ) RETURN VARCHAR2;

    FUNCTION get_desc_id_by_drilldown_id_version(
        p_drilldown_id    IN VARCHAR2,
        p_version         IN NUMBER
    ) RETURN VARCHAR2;

    FUNCTION get_drilldown_status_by_id (
        p_drilldown_id     IN VARCHAR2
    ) RETURN VARCHAR2;

    FUNCTION get_drilldown_match_document_by_id (
        p_drilldown_id     IN VARCHAR2
    ) RETURN CLOB;

    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_feedback_pkg
    Description: This package contains procedures related to updating
    feedback rating and comments, required feedback action, 
    feedback action priority and expected report and drilldown ids.
    ----------------------------------------------------------------------
    */
    PROCEDURE update_feedback(
        p_search_id         IN VARCHAR2,
        p_feedback_comments IN VARCHAR2 DEFAULT NULL,
        p_feedback_rating   IN BOOLEAN DEFAULT NULL
    );

    PROCEDURE delete_feedback (
        p_search_id         IN VARCHAR2,
        p_delete_rating     IN BOOLEAN DEFAULT FALSE,
        p_delete_comments   IN BOOLEAN DEFAULT FALSE
    );

    PROCEDURE update_required_feedback_action (
        p_id                  IN VARCHAR2,
        p_required_feedback_action IN VARCHAR2,
        p_error_message            IN VARCHAR2 DEFAULT 'Search record not '|| 
                                                'found for the specified ID'
    );

    PROCEDURE update_expected_report_id (
        p_search_id          IN VARCHAR2,
        p_expected_report_id IN VARCHAR2
    );

    PROCEDURE update_expected_drilldown_id (
        p_search_id             IN VARCHAR2,
        p_expected_drilldown_id IN VARCHAR2
    );

    PROCEDURE update_feedback_action_priority (
        p_id                  IN VARCHAR2,
        p_feedback_action_priority IN VARCHAR2,
        p_error_message       IN VARCHAR2 DEFAULT 'Search record not found'|| 
                                                'for the specified ID'
    );

    PROCEDURE remove_expected_report_id (
        p_search_id          IN VARCHAR2
    );

    PROCEDURE remove_expected_drilldown_id (
        p_search_id          IN VARCHAR2
    );

    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_named_entities_pkg 
    Description: This package contains procedures related to adding and
                 retrieving named entities from langdata$named_entities table.
    ----------------------------------------------------------------------
    */
    -- Public procedure declaration for getting all named entities
    PROCEDURE get_all_named_entities (
        p_named_entities OUT SYS_REFCURSOR
    );

    -- Public procedure declaration for adding a new named entity
    PROCEDURE add_named_entity (
        p_id IN VARCHAR2,
        p_name IN VARCHAR2
    );


    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_sample_queries_pkg
    Description: This package contains procedures and functions related 
                 to managing sample queries in the langdata$samplequeries table.
                 It includes retrieving, deleting, and adding deduplicated
                 sample queries.
    ----------------------------------------------------------------------
    */
    PROCEDURE get_report_sample_queries (
        p_report_id IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    );

    PROCEDURE get_drilldown_sample_queries (
        p_drilldown_id IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    );

    -- Public procedure declaration for deleting sample query
    PROCEDURE delete_sample_query (
        p_id IN VARCHAR2
    );

    PROCEDURE add_report_sample_queries(
        p_report_id     IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST,
        p_ids OUT SYS.ODCIVARCHAR2LIST
    );

    PROCEDURE add_drilldown_sample_queries(
        p_drilldown_id  IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST,
        p_ids OUT SYS.ODCIVARCHAR2LIST
    );

    PROCEDURE update_sample_query_status(
        p_id            IN VARCHAR2,
        p_status        IN VARCHAR2,
        p_response      OUT VARCHAR2
    );

    FUNCTION get_sample_query_id_by_report_id_version(
        p_report_id    IN VARCHAR2,
        p_version      IN NUMBER
    ) RETURN VARCHAR2;
    
    FUNCTION get_all_sample_queries_fn (
        p_report_id IN VARCHAR2 DEFAULT NULL,
        p_drilldown_id IN VARCHAR2 DEFAULT NULL,
        p_limit     IN NUMBER DEFAULT 10
    ) RETURN data_table PIPELINED;

    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_search_pkg
    Description: This package contains procedures related to searching and 
                 retrieving user search records from the langdata$searchrecords
                 table. The procedures are designed to handle user search
                 records and related operations.
    ----------------------------------------------------------------------
    */
    -- Public procedure declaration for getting user search record
    PROCEDURE get_user_search_record (
        p_id IN VARCHAR2,
        p_query_text OUT VARCHAR2,
        p_report_matches OUT JSON,
        p_feedback_rating OUT BOOLEAN,
        p_feedback_comments OUT VARCHAR2,
        p_required_feedback_action OUT VARCHAR2,
        p_feedback_action_priority OUT VARCHAR2,
        p_expected_report_id OUT VARCHAR2,
        p_expected_drilldown_id OUT VARCHAR2,
        p_search_type OUT VARCHAR2,
        p_augmented_query_text OUT VARCHAR2
    );

    -- Public procedure declaration for retrieving multiple user search records
    -- with pagination
    PROCEDURE get_all_user_search_records (
        p_feedback_rating IN BOOLEAN DEFAULT NULL,
        p_required_feedback_action IN VARCHAR2 DEFAULT NULL,
        p_feedback_action_priority IN VARCHAR2 DEFAULT NULL,
        p_cursor IN OUT VARCHAR2,
        p_limit IN NUMBER DEFAULT 10,
        p_records OUT SYS_REFCURSOR
    );

    FUNCTION get_all_user_search_records_fn (
        p_feedback_rating            IN VARCHAR2 DEFAULT NULL,
        p_required_feedback_action   IN VARCHAR2 DEFAULT NULL,
        p_feedback_action_priority   IN VARCHAR2 DEFAULT NULL,
        p_limit                      IN NUMBER   DEFAULT 10
    ) RETURN user_search_row_table PIPELINED;
    
    -- Public procedure declaration for deleting user search record
    PROCEDURE delete_user_search_record (
        p_id IN VARCHAR2
    );

    PROCEDURE search_from_query (
        p_query IN VARCHAR2,
        p_domain    IN VARCHAR2 DEFAULT NULL,
        p_search_id OUT VARCHAR2
    );

    PROCEDURE plot_search_vectors(
        p_search_id IN VARCHAR2
    );

    FUNCTION get_schema_version
    RETURN VARCHAR2;

    FUNCTION generate_sample_queries (
        p_title              IN VARCHAR2,
        p_description        IN VARCHAR2,
        p_filter_information IN VARCHAR2,
        p_sample_queries     IN SYS_REFCURSOR DEFAULT NULL,
        p_count              IN NUMBER DEFAULT 10
    ) RETURN CLOB;

    FUNCTION get_text_by_id(
        p_description_id IN VARCHAR2
    ) RETURN VARCHAR2;

    FUNCTION apply_filter_values_for_url (
        p_url_template      VARCHAR2,
        p_report_matches    langdata$searchrecords.report_matches%TYPE
    ) RETURN VARCHAR2;

    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_utils_pkg
    Description: This package provides a wide range of utility functions and 
                 procedures used across the application
    ----------------------------------------------------------------------
    */
    FUNCTION update_domain(
        p_domain_id IN VARCHAR2,
        p_new_name IN VARCHAR2
    ) RETURN BOOLEAN;

    FUNCTION delete_domain(
        p_domain_id IN VARCHAR2
    ) RETURN BOOLEAN;

    FUNCTION get_all_domains RETURN SYS_REFCURSOR;

    FUNCTION get_or_create_domain(
        p_domain_name IN VARCHAR2
    ) RETURN VARCHAR2;

    FUNCTION get_domain_id(
        p_domain_name IN VARCHAR2
    ) RETURN VARCHAR2;

    /*
    Description:
        Grants necessary privileges (READ, SELECT, ALTER, FLASHBACK) on a given
        table to the LANGDATA user. Each privilege is granted independently,
        and failures are logged individually as warnings without halting the 
        process. The overall result is reported via the OUT parameter.

    Parameters:
        - p_schema_name (IN VARCHAR2): The schema that owns the target table.
        - p_table_name  (IN VARCHAR2): The table to which privileges will be 
                                       granted.
        - p_success     (OUT BOOLEAN): Indicates whether all privileges were 
                                       granted successfully (TRUE) or if any 
                                       failed (FALSE).

    Returns:
        - None.

    Exceptions:
        - None. The procedure does not raise errors internally, but leaves it to
          the caller to check p_success and decide whether to handle failures.
    */
    PROCEDURE grant_user_table_privileges (
        p_schema_name   IN  VARCHAR2,
        p_table_name    IN  VARCHAR2,
        p_success       OUT BOOLEAN
    );

    /*
    Description:
        Validates a given match document (in JSON format) by ensuring that it 
        contains all the required fields and that the values conform to the
        expected formats. The procedure processes the document's filters
        array, checks for the presence of required keys, and validates that the
        values of certain fields (such as use_ner, table_name, column_name
        and entity_type) are correctly provided depending on the use_ner
        flag.

        For filters that reference local tables (i.e., use_ner = false and no
        db_link_name), this procedure also grants necessary privileges 
        (READ, SELECT, ALTER, FLASHBACK) on those tables to the LANGDATA user
        using grant_user_table_privileges. This serves both as a permission
        setup and an implicit check that the referenced table exists and is
        accessible.

        This procedure ensures the integrity of the match document by verifying:
        - The existence and type of the filters array.
        - Each filter item contains the necessary keys and values.
        - Attempting to grant privileges on referenced tables when use_ner is 
        false, which will fail (and be logged) if the tables do not exist or are
        inaccessible.
        - Ensuring that if use_ner is true, the entity_type is provided.
        - That referenced database links exist if db_link_name is specified.

    Parameters:
        - match_document (IN JSON): The JSON document containing the filters
                                    array, which is validated by the procedure.

    Returns:
        - None. The procedure raises errors if the validation fails at any point, 
        logging detailed messages for debugging.

    Exceptions:
        - c_invalid_match_document (-20005):
            Raised when the match document is invalid, such as when required
            fields like filter_name, default_value, table_name, or
            column_name are missing, or when invalid values are found in the
            document. The error message specifies the missing or incorrect
            fields.
    */

    PROCEDURE validate_match_document_and_grant_table_privileges(
        match_document IN JSON
    );
    
end lang_data;
/
