Rem
Rem $Header: dbgendev/src/langdata/plsql/lang_data.pkb /main/48 2025/08/13 22:20:34 fgurrola Exp $
Rem
Rem lang_data.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      lang_data.pkb - Package Body of lang_data Package
Rem
Rem    DESCRIPTION
Rem      This package contains implementation of the external APIs provided 
Rem      as part of LangData.
Rem
Rem    NOTES
Rem      NONE
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/lang_data.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    08/04/25 - Move grant_read_on_user_tables to lang_data pkg
Rem    sathyavc    08/01/25 - DBAI-1087: Add function to populate filter values
Rem                           into URL
Rem    dadoshi     07/31/25 - Add p_sample_queries to generate_sample_queries
Rem    jiangnhu    07/30/25 - Call validate_match_document before
Rem                           grant_read_on_user_tables, remove duplicate logic 
Rem                           in grant_read_on_user_tables
Rem    fgurrola    07/30/25 - DBAI-1130: remove_expected_report_id and
Rem                           remove_expected_drilldown_id added.
Rem    ruohli      07/24/25 - DBAI-1076: Added update_domain, delete_domain, and
Rem                           get_all_domains
Rem    dadoshi     07/24/25 - JIRA_DBAI1056: Add
Rem                           get_report_match_document_by_id
Rem    dadoshi     07/24/25 - JIRA_DBAI1080: Add
Rem                           get_drilldown_match_document_by_id
Rem    saloshah    07/24/25 - DBAI-1052: Add pipeline functions to get
Rem                           all user search records
Rem    dadoshi     07/22/25 - JIRA_DBAI1080: Added get_drilldown_status_by_id
Rem    dadoshi     07/22/25 - JIRA_DBAI1056, JIRA_DBAI1049: Add pipeline
Rem                           functions to fetch all reports and drilldowns
Rem    dadoshi     07/22/25 - JIRA_DBAI1056: Added get_report_status_by_id
Rem    ruohli      07/21/25 - DBAI-1111: Add Drilldown Ranking Frequency, 
Rem                           Matched Queries of Drilldown at Rank, and
Rem                           Drilldown filter usage as external API
Rem    pryarla     07/21/25 - DBAI-1075: Generate sample queries using LLM
Rem    deveverm    07/18/25 - DBAI-1050: added p_is_new_report parameter to
Rem                           get_report_regression
Rem    ruohli      07/17/25 - DBAI-1089: Add Report Ranking Frequency, 
Rem                           Matched Queries of Reports at Rank, and
Rem                           report filter usage as external API
Rem    jiangnhu    07/16/25 - Add p_genAI_rerank_endpoint_id in 
Rem                           set_oci_credential
Rem    pryarla     07/16/25 - DBAI-882: Added get_top_filters function
Rem    arevathi    07/14/25 - Move grants on tables to
Rem                           grant_read_on_user_tables
Rem    sathyavc    07/11/25 - DBAI-881: Add top k most searched reports, least
Rem    sathyavc    07/10/25 - DBAI-881: Modify feedback count APIs to be
Rem                           filtered based on time
Rem    sathyavc    07/07/25 - DBAI-883: Add functions to track API metrics over 
Rem                           specified time windows.
Rem    deveverm    07/07/25 - DBAI-926: add Flashback privilege and fix bug in
Rem                           create_report/drilldown
Rem    arevathi    07/04/25 - Add purge_outdated_search_records procedure
Rem    saloshah    07/03/25 - DBAI-873: Added delete feedback API
Rem    arevathi    07/01/25 - Change onnx_model_dir to mount_dir
Rem    jiangnhu    06/25/25 - Sync parameters of search_from_query to internal
Rem                           API
Rem    jiangnhu    06/09/25 - DBAI-871: Implement APIs to get ID by unique
Rem                           combination of title, version, etc.
Rem    deveverm    06/02/25 - DBAI-794: modified set_oci_credential to take
Rem                           genAI endpoint
Rem    saloshah    05/19/25 - DBAI-746: Added the update_api_metrics
Rem    deveverm    05/16/25 - DBAI-761: added updated_sample_query_status,
Rem                           changed get_report_description_regression to
Rem                           get_report_regression, changed
Rem                           get_drilldown_description_regression to
Rem                           get_drilldown_regression
Rem    jiangnhu    05/07/25 - DBAI-768: Commit within all procedures
Rem    jiangnhu    04/12/25 - DBAI-739: Add plot_search_vectors
Rem    deveverm    04/10/25 - DBAI-721: added set_oci_credential
Rem    deveverm    04/01/25 - DBAI-523: added
Rem    jiangnhu    03/25/25 - Fix parameter order of search_from_query
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/18/25 - DBAI-661: Add external APIs for user to update
Rem                           enumerable set
Rem    deveverm    03/18/25 - DBAI-546: added schema_name for cross_schema
Rem                           support
Rem    jiangnhu    03/06/25 - DBAI-542: Add get_schema_version for backend
Rem    deveverm    02/27/25 - DBAI_598: Fixed to add model_dir
Rem    jiangnhu    02/06/25 - DBAI-542: Add external APIs
Rem    dadoshi     01/22/25 - Add setup_db argument to init()
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data AS

    PROCEDURE init(
        mount_dir IN VARCHAR2,
        p_setup_db IN BOOLEAN DEFAULT false
    )
    IS
    BEGIN
        lang_data_auth_pkg.assert_minimum_role_access;
        lang_data_setup_pkg.init(
            mount_dir => mount_dir,
            p_setup_db => p_setup_db
        );
        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            RAISE;
    END init;

    PROCEDURE cleanup_langdata
    IS
    BEGIN
        -- Check if User is Authorized to perform the action
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_cleanup_pkg.cleanup_langdata;
        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            RAISE;
    END cleanup_langdata;

    PROCEDURE purge_outdated_search_records (
        p_days         IN NUMBER DEFAULT 180,
        p_cutoff_time  IN TIMESTAMP DEFAULT NULL,
        p_action       IN VARCHAR2 DEFAULT 'ARCHIVE',
        p_output_msg   OUT VARCHAR2
    ) IS
    BEGIN
        -- Check if User is Authorized to perform the action
        lang_data_auth_pkg.assert_app_expert_role;

        lang_data_cleanup_pkg.purge_outdated_search_records(
            p_days         => p_days,
            p_cutoff_time  => p_cutoff_time,
            p_action       => p_action,
            p_output_msg   => p_output_msg
        );

        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            RAISE;
    END purge_outdated_search_records;

    PROCEDURE migrate_schema
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        -- Check if User is Authorized to perform the action
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_migration_pkg.migrate_schema;
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'migrate_schema',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'migrate_schema',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END migrate_schema;
    
    PROCEDURE load_onnx_model(
        model_path IN VARCHAR2,
        filename   IN VARCHAR2,
        modelname  IN VARCHAR2,
        cred_name  IN VARCHAR2 DEFAULT NULL
    ) IS
    BEGIN
        -- Check if User is Authorized to perform the action
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_setup_pkg.load_onnx_model(
            model_path  => model_path,
            filename    => filename,
            modelname   => modelname,
            cred_name   => cred_name
        );
        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            RAISE;
    END load_onnx_model;

    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_config_pkg
    ----------------------------------------------------------------------
    */
    FUNCTION get_config_parameter (
        p_name IN VARCHAR2
    ) RETURN VARCHAR2
    IS
    BEGIN
        lang_data_auth_pkg.assert_app_expert_role;
        RETURN lang_data_config_pkg.get_config_parameter(
            p_name => p_name
        );
    END get_config_parameter;

    FUNCTION get_config_variable_names RETURN VARCHAR2
    IS
    BEGIN
        lang_data_auth_pkg.assert_app_expert_role;
        RETURN lang_data_config_pkg.get_config_variable_names();
    END get_config_variable_names;

    PROCEDURE update_config_parameter (
        p_name IN VARCHAR2,
        p_value IN VARCHAR2
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_config_pkg.update_config_parameter(
            p_name  =>  p_name,
            p_value =>  p_value
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_config_parameter',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_config_parameter',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_config_parameter;

    FUNCTION set_oci_credential (
        p_cred_name IN VARCHAR2,
        p_compartment_id IN VARCHAR2,
        p_region_code IN VARCHAR2,
        p_genAI_endpoint IN VARCHAR2,
        p_genAI_rerank_endpoint_id IN VARCHAR2
    ) RETURN VARCHAR2
    IS
        v_return_status VARCHAR2(100);
    BEGIN
        lang_data_auth_pkg.assert_app_expert_role;
        v_return_status := lang_data_config_pkg.set_oci_credential(
            p_cred_name => p_cred_name,
            p_compartment_id => p_compartment_id,
            p_region_code => p_region_code,
            p_genAI_endpoint => p_genAI_endpoint,
            p_genAI_rerank_endpoint_id => p_genAI_rerank_endpoint_id
        );
        RETURN v_return_status;
        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            RAISE;
    END set_oci_credential;

    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_analytics_pkg
    Description: This package contains procedures related to analytics of 
    langdata. Each procedure returns different metrics over the langdata search 
    record tables
    ----------------------------------------------------------------------
    */
    PROCEDURE get_metrics(
        p_timestamp  IN TIMESTAMP DEFAULT NULL,
        p_metrics    OUT JSON_OBJECT_T
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_analytics_pkg.get_metrics(
            p_timestamp  =>  p_timestamp,
            p_metrics  =>  p_metrics
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_metrics',v_start_time,v_end_time
        );
    END get_metrics;

    FUNCTION get_top_k_most_searched_reports(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_start_time NUMBER;
        v_end_time   NUMBER;
        v_result     CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_analytics_pkg.get_top_k_most_searched_reports(
            p_k => p_k,
            p_timestamp  =>  p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_top_k_most_searched_reports',v_start_time,v_end_time
        );
        RETURN v_result;
    END get_top_k_most_searched_reports;

    FUNCTION get_all_api_stats (
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_start_time NUMBER;
        v_end_time   NUMBER;
        v_result     CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_analytics_pkg.get_all_api_stats(
            p_timestamp => p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        
        lang_data_analytics_pkg.update_api_logs(
            'get_all_api_stats',v_start_time,v_end_time
        );
        RETURN v_result;
    END get_all_api_stats;

    FUNCTION get_api_daily_stats (
        p_api_name  IN VARCHAR2,
        p_num_error_messages IN NUMBER DEFAULT 5,
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_start_time NUMBER;
        v_end_time   NUMBER;
        v_result     CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_analytics_pkg.get_api_daily_stats(
            p_api_name => p_api_name,
            p_num_error_messages => p_num_error_messages,
            p_timestamp => p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        
        lang_data_analytics_pkg.update_api_logs(
            'get_api_daily_stats',v_start_time,v_end_time
        );
        RETURN v_result;
    END get_api_daily_stats;

    FUNCTION get_api_failures (
        p_api_name  IN VARCHAR2,
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_start_time NUMBER;
        v_end_time   NUMBER;
        v_result     CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_analytics_pkg.get_api_failures(
            p_api_name => p_api_name,
            p_timestamp => p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        
        lang_data_analytics_pkg.update_api_logs(
            'get_api_failures',v_start_time,v_end_time
        );
        RETURN v_result;
    END get_api_failures;

    FUNCTION get_bottom_k_least_searched_reports(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_start_time NUMBER;
        v_end_time   NUMBER;
        v_result     CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_analytics_pkg.get_bottom_k_least_searched_reports(
            p_k => p_k,
            p_timestamp  =>  p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_bottom_k_least_searched_reports',v_start_time,v_end_time
        );
        RETURN v_result;
    END get_bottom_k_least_searched_reports;

    FUNCTION get_top_k_most_asked_questions(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_start_time NUMBER;
        v_end_time   NUMBER;
        v_result     CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_analytics_pkg.get_top_k_most_asked_questions(
            p_k => p_k,
            p_timestamp  =>  p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_top_k_most_asked_questions',v_start_time,v_end_time
        );
        RETURN v_result;
    END get_top_k_most_asked_questions;

    FUNCTION get_top_k_most_negative_feedback_questions(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_start_time NUMBER;
        v_end_time   NUMBER;
        v_result     CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := 
            lang_data_analytics_pkg.get_top_k_most_negative_feedback_questions(
                p_k => p_k,
                p_timestamp  =>  p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_top_k_most_negative_feedback_questions',v_start_time,v_end_time
        );
        RETURN v_result;
    END get_top_k_most_negative_feedback_questions;


    PROCEDURE get_top_report_metrics(
       p_report_id     IN VARCHAR2,
       p_metrics       OUT JSON_OBJECT_T
   ) IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_analytics_pkg.get_top_report_metrics(
            p_report_id        => p_report_id,
            p_metrics       => p_metrics
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_top_report_metrics',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_top_report_metrics',v_start_time,v_end_time,TRUE,
                SQLCODE, SQLERRM
            );
            RAISE;
    END get_top_report_metrics;

    PROCEDURE get_matched_queries_by_report_rank (
        p_report_id IN VARCHAR2,
        p_rank      IN PLS_INTEGER,
        p_k         IN PLS_INTEGER,
        p_queries   OUT SYS.ODCIVARCHAR2LIST
    ) IS 
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_analytics_pkg.get_matched_queries_by_report_rank(
            p_report_id        => p_report_id,
            p_rank             => p_rank,
            p_k                => p_k,
            p_queries          => p_queries
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_matched_queries_by_report_rank',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_matched_queries_by_report_rank',
                v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END get_matched_queries_by_report_rank;

    PROCEDURE get_report_filter_usage(
        p_report_id in VARCHAR2,
        p_default_count OUT NUMBER,
        p_non_default_count OUT NUMBER
    ) IS 
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_analytics_pkg.get_report_filter_usage(
            p_report_id        => p_report_id,
            p_default_count       => p_default_count,
            p_non_default_count       => p_non_default_count
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_filter_usage',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_report_filter_usage',
                v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END get_report_filter_usage;
    
 


    FUNCTION get_top_filters(
        p_top_k     IN NUMBER,
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB IS
        v_start_time   NUMBER;
        v_end_time     NUMBER;
        v_top_filters  CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_minimum_role_access;
        v_top_filters := lang_data_analytics_pkg.get_top_filters(
            p_top_k     => p_top_k,
            p_timestamp => p_timestamp
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_top_filters', v_start_time, v_end_time
        );
        RETURN v_top_filters;
    END get_top_filters;

    PROCEDURE get_top_drilldown_metrics(
       p_drilldown_id     IN VARCHAR2,
       p_metrics       OUT JSON_OBJECT_T
    ) IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_analytics_pkg.get_top_drilldown_metrics(
            p_drilldown_id        => p_drilldown_id,
            p_metrics       => p_metrics
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_top_drilldown_metrics',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_top_drilldown_metrics',v_start_time,v_end_time,TRUE
            );
            RAISE;
    END get_top_drilldown_metrics;

    PROCEDURE get_matched_queries_by_drilldown_rank (
        p_drilldown_id IN VARCHAR2,
        p_rank      IN PLS_INTEGER,
        p_k         IN PLS_INTEGER,
        p_queries   OUT SYS.ODCIVARCHAR2LIST
    ) IS 
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_analytics_pkg.get_matched_queries_by_drilldown_rank(
            p_drilldown_id        => p_drilldown_id,
            p_rank             => p_rank,
            p_k                => p_k,
            p_queries          => p_queries
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_matched_queries_by_drilldown_rank',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_matched_queries_by_drilldown_rank',
                v_start_time,v_end_time,TRUE
            );
            RAISE;
    END get_matched_queries_by_drilldown_rank;

    PROCEDURE get_drilldown_filter_usage(
        p_drilldown_id in VARCHAR2,
        p_default_count OUT NUMBER,
        p_non_default_count OUT NUMBER
    ) IS 
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_analytics_pkg.get_drilldown_filter_usage(
            p_drilldown_id        => p_drilldown_id,
            p_default_count       => p_default_count,
            p_non_default_count       => p_non_default_count
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_filter_usage',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_drilldown_filter_usage',
                v_start_time,v_end_time,TRUE
            );
            RAISE;
    END get_drilldown_filter_usage;



    /*
    ----------------------------------------------------------------------
    Package Name: annotations_pkg
    Description: This package contains procedures and triggers related to 
                 updating annotations for user_tables. The procedures and 
                 triggers are designed to handle updating annotations and 
                 all the related updating operations.
    ----------------------------------------------------------------------
    */
    PROCEDURE update_annotation_values(
        p_table_name      IN VARCHAR2,
        p_column_name     IN VARCHAR2,
        p_annotation_value IN VARCHAR2,
        p_schema_name     IN VARCHAR2
    )
    IS
    BEGIN
        -- Check if User is Authorized to perform the action
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_annotations_pkg.update_annotation_values(
            p_table_name        =>  p_table_name,
            p_column_name       =>  p_column_name,
            p_annotation_value  => p_annotation_value,
            p_schema_name       => p_schema_name
        );
        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            RAISE;
    END update_annotation_values;


    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_reports_pkg
    Description: This package contains procedures related to the report
	and report descriptions. These procedures are designed to handle CRUD
	and validation operations on reports/report descriptions.
    ----------------------------------------------------------------------
    */
    PROCEDURE get_report_descriptions_paginated (
        p_report_id     IN VARCHAR2,
        p_limit         IN NUMBER DEFAULT 10,
        p_cursor        IN OUT VARCHAR2,
        p_descriptions  OUT SYS_REFCURSOR
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        -- Check if User is Authorized to perform the action
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.get_report_descriptions_paginated(
            p_report_id     =>  p_report_id,
            p_limit         =>  p_limit,
            p_cursor        =>  p_cursor,
            p_descriptions  =>  p_descriptions
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_descriptions_paginated',v_start_time,v_end_time
        );
    END get_report_descriptions_paginated;

	PROCEDURE get_report_paginated (
		p_id IN VARCHAR2,
		p_title OUT VARCHAR2,
		p_match_document OUT JSON,
		p_status OUT VARCHAR2,
		p_descriptions OUT SYS_REFCURSOR,
        p_description_cur IN OUT VARCHAR2,
        p_descriptions_limit IN NUMBER DEFAULT 10,
		p_sample_queries OUT SYS_REFCURSOR,
        p_sample_query_cur IN OUT VARCHAR2,
        p_sample_query_limit IN NUMBER DEFAULT 10
	)
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        -- Check if User is Authorized to perform the action
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.get_report_paginated(
            p_id                 => p_id,
            p_title              => p_title,
            p_match_document     => p_match_document,
            p_status             => p_status,
            p_descriptions       => p_descriptions,
            p_description_cur    => p_description_cur,
            p_descriptions_limit => p_descriptions_limit,
            p_sample_queries     => p_sample_queries,
            p_sample_query_cur   => p_sample_query_cur,
            p_sample_query_limit => p_sample_query_limit
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_paginated',v_start_time,v_end_time
        );
    END get_report_paginated;

    PROCEDURE create_report (
        p_title                 IN VARCHAR2,
        p_match_document        IN JSON,
        p_description_text      IN VARCHAR2,
        p_description_status    IN VARCHAR2 DEFAULT 'Pending Regression',
        p_report_status         IN VARCHAR2,
        p_sample_queries        IN SYS.ODCIVARCHAR2LIST,
        p_domain                IN VARCHAR2 DEFAULT NULL,
        p_report_id             OUT VARCHAR2,
        p_description_id        OUT VARCHAR2
    )
    IS
        v_sql VARCHAR2(4000);
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;

        lang_data.validate_match_document_and_grant_table_privileges(
            p_match_document
        );

        lang_data_reports_pkg.create_report(
            p_title              => p_title,
            p_match_document     => p_match_document,
            p_description_text   => p_description_text,
            p_description_status => p_description_status,
            p_report_status      => p_report_status,
            p_sample_queries     => p_sample_queries,
            p_domain             => p_domain,
            p_report_id          => p_report_id,
            p_description_id     => p_description_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'create_report',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'create_report',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END create_report;

    PROCEDURE get_report_regression(
        p_description_id    IN VARCHAR2 DEFAULT NULL,
        p_sample_query_id   IN VARCHAR2 DEFAULT NULL,
        p_is_new_report       IN BOOLEAN DEFAULT FALSE,
        p_force             IN BOOLEAN,
        p_json              OUT JSON
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.get_report_regression(
            p_description_id        => p_description_id,
            p_sample_query_id       => p_sample_query_id,
            p_is_new_report         => p_is_new_report,
            p_force                 => p_force,
            p_json                  => p_json
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_regression',v_start_time,v_end_time
        );
    END get_report_regression;

    PROCEDURE get_all_reports (
        p_status            IN VARCHAR2 DEFAULT NULL,
        p_cursor            IN OUT VARCHAR2,
        p_limit             IN NUMBER DEFAULT 10,
        p_reports           OUT SYS_REFCURSOR
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.get_all_reports(
            p_status => p_status,
            p_cursor => p_cursor,
            p_limit  => p_limit,
            p_reports => p_reports
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_all_reports',v_start_time,v_end_time
        );
    END get_all_reports;

    -- Note: Execution time is not measured to update API Metrics
    --       as part of this function as APEX throws error on 
    --       executing the update_api_metrics API 
    --       (cannot have DML inside a query)
    FUNCTION get_all_reports_fn (
        p_status IN VARCHAR2 DEFAULT NULL,
        p_limit  IN NUMBER DEFAULT 10
    ) RETURN report_row_table PIPELINED
    AS
        v_cursor    SYS_REFCURSOR;
        v_cursor_id VARCHAR2(1000);

        v_id                 VARCHAR2(36);
        v_title              VARCHAR2(255);
        v_match_document     CLOB;
        v_status             VARCHAR2(20);
        v_report_description VARCHAR2(2000);
    BEGIN

        -- Call your main procedure
        lang_data_reports_pkg.get_all_reports(
            p_status  => p_status,
            p_cursor  => v_cursor_id,
            p_limit   => p_limit,
            p_reports => v_cursor
        );

        -- Stream each row as a piped object
        LOOP
            FETCH v_cursor INTO 
                v_id, 
                v_title, 
                v_match_document, 
                v_status, 
                v_report_description;
            EXIT WHEN v_cursor%NOTFOUND;

            PIPE ROW(report_row_obj(
                v_id,
                v_title,
                v_match_document,
                v_status,
                v_report_description
            ));
        END LOOP;

        CLOSE v_cursor;
        RETURN;
    END get_all_reports_fn;

    FUNCTION get_report_descriptions_fn (
        p_report_id IN VARCHAR2,
        p_limit     IN NUMBER DEFAULT 10
    ) RETURN data_table PIPELINED
    AS
        v_cursor      SYS_REFCURSOR;
        v_cursor_id   VARCHAR2(4000);
        
        v_id            VARCHAR2(36);
        v_text          VARCHAR2(2000);
        v_version       NUMBER;
        v_status        VARCHAR2(20);
        v_enhanced_text VARCHAR2(4000);
    BEGIN
        -- Call your existing paginated procedure
        lang_data_reports_pkg.get_report_descriptions_paginated(
            p_report_id   => p_report_id,
            p_limit       => p_limit,
            p_cursor      => v_cursor_id,
            p_descriptions => v_cursor
        );

        -- Pipe each row from the ref cursor
        LOOP
            FETCH v_cursor INTO 
                v_id, 
                v_text, 
                v_version, 
                v_status, 
                v_enhanced_text;
            EXIT WHEN v_cursor%NOTFOUND;

            PIPE ROW(data_row_obj(
                v_id,
                v_text,
                v_version,
                v_status,
                v_enhanced_text
            ));
        END LOOP;

        CLOSE v_cursor;
        RETURN;
    END get_report_descriptions_fn;

    PROCEDURE delete_report (
        p_report_id IN VARCHAR2
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.delete_report(
            p_report_id => p_report_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_report',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_report',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END delete_report;

    PROCEDURE add_report_description (
        p_report_id      IN VARCHAR2,
        p_text           IN VARCHAR2,
        p_status         IN VARCHAR2 DEFAULT 'Pending Regression',
        p_description_id OUT VARCHAR2
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.add_report_description(
            p_report_id      => p_report_id,
            p_text           => p_text,
            p_status         => p_status,
            p_description_id => p_description_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_report_description',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'add_report_description',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END add_report_description;

    PROCEDURE delete_report_description (
        p_description_id IN VARCHAR2
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.delete_report_description(
            p_description_id => p_description_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_report_description',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_report_description',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END delete_report_description;

    PROCEDURE update_report_description (
        p_id   IN VARCHAR2,
        p_text IN VARCHAR2
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.update_report_description(
            p_id   => p_id,
            p_text => p_text
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_report_description',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_report_description',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_report_description;

    PROCEDURE get_report_description (
        p_id            IN VARCHAR2,
        p_text          OUT VARCHAR2,
        p_version       OUT NUMBER,
        p_status        OUT VARCHAR2,
        p_report_id     OUT VARCHAR2,
        p_enhanced_text  OUT VARCHAR2
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.get_report_description(
            p_id           => p_id,
            p_text         => p_text,
            p_version      => p_version,
            p_status       => p_status,
            p_report_id    => p_report_id,
            p_enhanced_text => p_enhanced_text
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_description',v_start_time,v_end_time
        );
    END get_report_description;

    PROCEDURE execute_report_sql (
        p_report_id    IN VARCHAR2,
        p_filter_values IN JSON,
        p_columns      OUT SYS.ODCIVARCHAR2LIST,
        p_data         OUT CLOB
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_minimum_role_access;
        lang_data_reports_pkg.execute_report_sql(
            p_report_id    => p_report_id,
            p_filter_values => p_filter_values,
            p_columns      => p_columns,
            p_data         => p_data
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'execute_report_sql',v_start_time,v_end_time
        );
    END execute_report_sql;

    PROCEDURE replace_report_filter_enumerable_set (
        p_report_id             IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_new_enumerable_set    IN  JSON_ARRAY_T
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.replace_report_filter_enumerable_set(
            p_report_id             => p_report_id,
            p_filter_name           => p_filter_name,
            p_new_enumerable_set    => p_new_enumerable_set
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'replace_report_filter_enumerable_set',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'replace_report_filter_enumerable_set',v_start_time,v_end_time,
                TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END replace_report_filter_enumerable_set;

    PROCEDURE add_into_report_filter_enumerable_set (
        p_report_id             IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_enumerable_set_to_add    IN  JSON_ARRAY_T
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.add_into_report_filter_enumerable_set(
            p_report_id                => p_report_id,
            p_filter_name              => p_filter_name,
            p_enumerable_set_to_add    => p_enumerable_set_to_add
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_into_report_filter_enumerable_set',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'add_into_report_filter_enumerable_set',v_start_time,v_end_time,
                TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END add_into_report_filter_enumerable_set;

    PROCEDURE remove_from_report_filter_enumerable_set (
        p_report_id                   IN  VARCHAR2,
        p_filter_name                 IN  VARCHAR2,
        p_enumerable_set_to_remove    IN  JSON_ARRAY_T
    )
    IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.remove_from_report_filter_enumerable_set(
            p_report_id             => p_report_id,
            p_filter_name           => p_filter_name,
            p_enumerable_set_to_remove    => p_enumerable_set_to_remove
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;        
        lang_data_analytics_pkg.update_api_logs(
            'remove_from_report_filter_enumerable_set',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            lang_data_analytics_pkg.update_api_logs(
                'remove_from_report_filter_enumerable_set',v_start_time,
                v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END remove_from_report_filter_enumerable_set;

    PROCEDURE update_report_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
        v_start_time NUMBER;
        v_end_time  NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.update_report_status(
            p_id            => p_id,
            p_status        => p_status
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_report_status',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_report_status',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM);
            RAISE;
    END update_report_status;

    PROCEDURE update_report_description_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
        v_row_count         VARCHAR2(20);
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_reports_pkg.update_report_description_status(
            p_id            => p_id,
            p_status        => p_status 
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_report_description_status',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_report_description_status',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_report_description_status;

    FUNCTION get_report_id_by_title(
        p_title IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_report_id         VARCHAR2(36);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_report_id := lang_data_reports_pkg.get_report_id_by_title(p_title);

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_id_by_title',v_start_time,v_end_time
        );

        RETURN v_report_id;
    END get_report_id_by_title;

    FUNCTION get_desc_id_by_report_id_version(
        p_report_id    IN VARCHAR2,
        p_version      IN NUMBER
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_desc_id         VARCHAR2(36);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_desc_id := lang_data_reports_pkg.get_desc_id_by_report_id_version(
            p_report_id, p_version
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_desc_id_by_report_id_version',v_start_time,v_end_time
        );

        RETURN v_desc_id;
    END get_desc_id_by_report_id_version;

    FUNCTION get_report_status_by_id (
        p_report_id     IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_status            VARCHAR2(20);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_status := lang_data_reports_pkg.get_report_status_by_id(
            p_report_id
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_status_by_id',v_start_time,v_end_time
        );

        RETURN v_status;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_report_status_by_id',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END get_report_status_by_id;

    FUNCTION get_report_match_document_by_id (
        p_report_id     IN VARCHAR2
    ) RETURN CLOB IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_md            CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_md := lang_data_reports_pkg.get_report_match_document_by_id(
            p_report_id
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_match_document_by_id',v_start_time,v_end_time
        );

        RETURN v_md;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_report_match_document_by_id',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END get_report_match_document_by_id;
    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_drilldowns_pkg
    Description: This package contains procedures related to the drilldown
	and drilldown descriptions. These procedures are designed to handle 
    CRUD and validation operations on drilldown documents and their 
    descriptions.
    ----------------------------------------------------------------------
    */
    PROCEDURE get_drilldown_paginated (
        p_id                 IN VARCHAR2,
        p_report_id          OUT VARCHAR2,
        p_title              OUT VARCHAR2,
        p_match_document     OUT JSON,
        p_status             OUT VARCHAR2,
        p_descriptions       OUT SYS_REFCURSOR,
        p_descriptions_cur   IN OUT VARCHAR2,
        p_descriptions_limit IN NUMBER DEFAULT 10,
        p_sample_queries     OUT SYS_REFCURSOR,
        p_sample_query_cur   IN OUT VARCHAR2,
        p_sample_query_limit IN NUMBER DEFAULT 10
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.get_drilldown_paginated(
            p_id                 => p_id,
            p_report_id          => p_report_id,
            p_title              => p_title,
            p_match_document     => p_match_document,
            p_status             => p_status,
            p_descriptions       => p_descriptions,
            p_descriptions_cur   => p_descriptions_cur,
            p_descriptions_limit => p_descriptions_limit,
            p_sample_queries     => p_sample_queries,
            p_sample_query_cur   => p_sample_query_cur,
            p_sample_query_limit => p_sample_query_limit
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_paginated',v_start_time,v_end_time
        );
    END get_drilldown_paginated;

    FUNCTION get_all_drilldowns_fn (
        p_report_id IN VARCHAR2,
        p_status IN VARCHAR2 DEFAULT NULL,
        p_limit  IN NUMBER DEFAULT 10
    ) RETURN drilldown_row_table PIPELINED
    AS
        v_cursor    SYS_REFCURSOR;
        v_cursor_id VARCHAR2(1000);

        v_id                 VARCHAR2(36);
        v_title              VARCHAR2(255);
        v_match_document     CLOB;
        v_status             VARCHAR2(20);
        v_report_id          VARCHAR2(36);
        v_domain_id          VARCHAR2(36);
        v_drilldown_description VARCHAR2(2000);
    BEGIN
        -- Call your main procedure
        lang_data_drilldowns_pkg.get_all_drilldowns(
            p_report_id,
            p_status,
            v_cursor_id,
            p_limit,
            v_cursor
        );

        -- Stream each row as a piped object
        LOOP
            FETCH v_cursor INTO 
                v_id, 
                v_title, 
                v_match_document, 
                v_status, 
                v_report_id, 
                v_domain_id, 
                v_drilldown_description;
            EXIT WHEN v_cursor%NOTFOUND;

            PIPE ROW(drilldown_row_obj(
                v_id,
                v_title,
                v_match_document,
                v_status,
                v_report_id,
                v_domain_id,
                v_drilldown_description
            ));
        END LOOP;

        CLOSE v_cursor;
        RETURN;
    END get_all_drilldowns_fn;

    FUNCTION get_drilldown_descriptions_fn (
        p_drilldown_id IN VARCHAR2,
        p_limit        IN NUMBER DEFAULT 10
    ) RETURN drilldown_description_table PIPELINED
    AS
        v_cursor      SYS_REFCURSOR;
        v_cursor_id   VARCHAR2(4000);

        v_id              VARCHAR2(36);
        v_text            VARCHAR2(2000);
        v_version         NUMBER;
        v_status          VARCHAR2(20);
        v_enhanced_text   VARCHAR2(4000);
    BEGIN
        -- Call the paginated procedure
        lang_data_drilldowns_pkg.get_drilldown_descriptions_paginated(
            p_id           => p_drilldown_id,
            p_limit        => p_limit,
            p_cursor       => v_cursor_id,
            p_descriptions => v_cursor
        );

        -- Pipe each row from the cursor
        LOOP
            FETCH v_cursor INTO
                v_id,
                v_text,
                v_version,
                v_status,
                v_enhanced_text;
            EXIT WHEN v_cursor%NOTFOUND;

            PIPE ROW(drilldown_description_row_obj(
                v_id,
                v_text,
                v_version,
                v_status,
                v_enhanced_text
            ));
        END LOOP;

        CLOSE v_cursor;
        RETURN;
    END get_drilldown_descriptions_fn;


    PROCEDURE delete_drilldown (
        p_drilldown_id IN VARCHAR2
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.delete_drilldown(
            p_drilldown_id => p_drilldown_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_drilldown',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_drilldown',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END delete_drilldown;

    PROCEDURE delete_drilldown_description (
        p_description_id IN VARCHAR2
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.delete_drilldown_description(
            p_description_id => p_description_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_drilldown_description',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_drilldown_description',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END delete_drilldown_description;

    PROCEDURE update_drilldown_description (
        p_drilldown_description_id IN VARCHAR2,
        p_new_description          IN VARCHAR2
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.update_drilldown_description(
            p_drilldown_description_id => p_drilldown_description_id,
            p_new_description          => p_new_description
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_drilldown_description',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_drilldown_description',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_drilldown_description;

    PROCEDURE create_drilldown(
        p_title               IN VARCHAR2,
        p_report_id           IN VARCHAR2,
        p_match_document      IN JSON,
        p_description_text    IN VARCHAR2,
        p_description_status  IN VARCHAR2 DEFAULT 'Pending Regression',
        p_drilldown_status    IN VARCHAR2,
        p_sample_queries      IN SYS.ODCIVARCHAR2LIST,
        p_domain              IN VARCHAR2 DEFAULT NULL,
        p_drilldown_id        OUT VARCHAR2,
        p_description_id      OUT VARCHAR2
    )
    IS
        v_sql VARCHAR2(4000);
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        
        lang_data.validate_match_document_and_grant_table_privileges(
            p_match_document
        );

        lang_data_drilldowns_pkg.create_drilldown(
            p_title              => p_title,
            p_report_id          => p_report_id,
            p_match_document     => p_match_document,
            p_description_text   => p_description_text,
            p_description_status => p_description_status,
            p_drilldown_status   => p_drilldown_status,
            p_sample_queries     => p_sample_queries,
            p_domain             => p_domain,
            p_drilldown_id       => p_drilldown_id,
            p_description_id     => p_description_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'create_drilldown',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'create_drilldown',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END create_drilldown;

    PROCEDURE add_drilldown_description (
        p_drilldown_id   IN VARCHAR2,
        p_text           IN VARCHAR2,
        p_status         IN VARCHAR2 DEFAULT 'Pending Regression',
        p_description_id OUT VARCHAR2
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.add_drilldown_description(
            p_drilldown_id   => p_drilldown_id,
            p_text           => p_text,
            p_status         => p_status,
            p_description_id => p_description_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_drilldown_description',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'add_drilldown_description',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END add_drilldown_description;

    PROCEDURE execute_drilldown_sql (
        p_drilldown_id  IN VARCHAR2,
        p_filter_values IN JSON,
        p_columns       OUT SYS.ODCIVARCHAR2LIST,
        p_data          OUT CLOB
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_minimum_role_access;
        lang_data_drilldowns_pkg.execute_drilldown_sql(
            p_drilldown_id  => p_drilldown_id,
            p_filter_values => p_filter_values,
            p_columns       => p_columns,
            p_data          => p_data
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'execute_drilldown_sql',v_start_time,v_end_time
        );
    END execute_drilldown_sql;

    PROCEDURE get_drilldown_description (
        p_description_id IN VARCHAR2,
        p_text           OUT VARCHAR2,
        p_version        OUT NUMBER,
        p_status         OUT VARCHAR2,
        p_drilldown_id   OUT VARCHAR2,
        p_enhanced_text   OUT VARCHAR2
    )
    IS
    v_start_time        NUMBER;
    v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.get_drilldown_description(
            p_description_id => p_description_id,
            p_text           => p_text,
            p_version        => p_version,
            p_status         => p_status,
            p_drilldown_id   => p_drilldown_id,
            p_enhanced_text   => p_enhanced_text
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_description',v_start_time,v_end_time
        );
    END get_drilldown_description;

    PROCEDURE get_drilldown_descriptions (
        p_drilldown_id IN VARCHAR2,
        p_descriptions OUT SYS_REFCURSOR
    )
    IS
    v_start_time        NUMBER;
    v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.get_drilldown_descriptions(
            p_drilldown_id => p_drilldown_id,
            p_descriptions => p_descriptions
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_descriptions',v_start_time,v_end_time
        );
    END get_drilldown_descriptions;

    PROCEDURE get_drilldown_regression(
        p_description_id  IN VARCHAR2 DEFAULT NULL,
        p_sample_query_id IN VARCHAR2 DEFAULT NULL,
        p_force               IN BOOLEAN,
        p_json                OUT JSON
    )
    IS
    v_start_time        NUMBER;
    v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.get_drilldown_regression(
            p_description_id        => p_description_id,
            p_sample_query_id       => p_sample_query_id,
            p_force                 => p_force,
            p_json                  => p_json
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_regression',v_start_time,v_end_time
        );
    END get_drilldown_regression;

    PROCEDURE replace_drilldown_filter_enumerable_set (
        p_drilldown_id          IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_new_enumerable_set    IN  JSON_ARRAY_T
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.replace_drilldown_filter_enumerable_set(
            p_drilldown_id          => p_drilldown_id,
            p_filter_name           => p_filter_name,
            p_new_enumerable_set    => p_new_enumerable_set
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'replace_drilldown_filter_enumerable_set',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'replace_drilldown_filter_enumerable_set',v_start_time,
                v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END replace_drilldown_filter_enumerable_set;

    PROCEDURE add_into_drilldown_filter_enumerable_set (
        p_drilldown_id          IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_enumerable_set_to_add    IN  JSON_ARRAY_T
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.add_into_drilldown_filter_enumerable_set(
            p_drilldown_id             => p_drilldown_id,
            p_filter_name              => p_filter_name,
            p_enumerable_set_to_add    => p_enumerable_set_to_add
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_into_drilldown_filter_enumerable_set',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'add_into_drilldown_filter_enumerable_set',
                v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END add_into_drilldown_filter_enumerable_set;

    PROCEDURE remove_from_drilldown_filter_enumerable_set (
        p_drilldown_id                IN  VARCHAR2,
        p_filter_name                 IN  VARCHAR2,
        p_enumerable_set_to_remove    IN  JSON_ARRAY_T
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.remove_from_drilldown_filter_enumerable_set(
            p_drilldown_id          => p_drilldown_id,
            p_filter_name           => p_filter_name,
            p_enumerable_set_to_remove    => p_enumerable_set_to_remove
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'remove_from_drilldown_filter_enumerable_set',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'remove_from_drilldown_filter_enumerable_set',v_start_time,
                v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END remove_from_drilldown_filter_enumerable_set;

    PROCEDURE update_drilldown_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.update_drilldown_status(
            p_id            => p_id,
            p_status        => p_status
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_drilldown_status',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_drilldown_status',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_drilldown_status;

    PROCEDURE update_drilldown_description_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
        v_row_count         VARCHAR2(20);
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_drilldowns_pkg.update_drilldown_description_status(
            p_id            => p_id,
            p_status        => p_status
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_drilldown_description_status',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_drilldown_description_status',v_start_time,v_end_time,
                TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END update_drilldown_description_status;

    FUNCTION get_drilldown_id_by_title(
        p_title IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_drilldown_id         VARCHAR2(36);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_drilldown_id := lang_data_drilldowns_pkg.get_drilldown_id_by_title(
                            p_title
                        );
        
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_id_by_title',v_start_time,v_end_time
        );

        RETURN v_drilldown_id;
    END get_drilldown_id_by_title;

    FUNCTION get_desc_id_by_drilldown_id_version(
        p_drilldown_id    IN VARCHAR2,
        p_version         IN NUMBER
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_desc_id         VARCHAR2(36);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_desc_id := 
            lang_data_drilldowns_pkg.get_desc_id_by_drilldown_id_version(
                p_drilldown_id, p_version
            );
        
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_desc_id_by_drilldown_id_version',v_start_time,v_end_time
        );
        RETURN v_desc_id;
    END get_desc_id_by_drilldown_id_version;

    FUNCTION get_drilldown_status_by_id (
        p_drilldown_id     IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_status            VARCHAR2(20);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_status := lang_data_drilldowns_pkg.get_drilldown_status_by_id(
            p_drilldown_id
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_status_by_id',v_start_time,v_end_time
        );

        RETURN v_status;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_drilldown_status_by_id',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END get_drilldown_status_by_id;

    FUNCTION get_drilldown_match_document_by_id (
        p_drilldown_id     IN VARCHAR2
    ) RETURN CLOB IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_md            CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_md := lang_data_drilldowns_pkg.get_drilldown_match_document_by_id(
            p_drilldown_id
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_match_document_by_id',v_start_time,v_end_time
        );

        RETURN v_md;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_drilldown_match_document_by_id',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END get_drilldown_match_document_by_id;


    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_feedback_pkg
    ----------------------------------------------------------------------
    */
    PROCEDURE update_feedback (
        p_search_id         IN VARCHAR2,
        p_feedback_comments IN VARCHAR2 DEFAULT NULL,
        p_feedback_rating   IN BOOLEAN DEFAULT NULL
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.update_feedback(
            p_search_id         => p_search_id,
            p_feedback_comments => p_feedback_comments,
            p_feedback_rating   => p_feedback_rating
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_feedback',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_feedback',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END update_feedback;

    PROCEDURE delete_feedback (
        p_search_id         IN VARCHAR2,
        p_delete_rating     IN BOOLEAN DEFAULT FALSE,
        p_delete_comments   IN BOOLEAN DEFAULT FALSE
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.delete_feedback(
            p_search_id         => p_search_id,
            p_delete_rating     => p_delete_rating,
            p_delete_comments   => p_delete_comments
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_feedback',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_feedback',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );            
            RAISE;
    END delete_feedback;

    PROCEDURE update_required_feedback_action (
        p_id                     IN VARCHAR2,
        p_required_feedback_action IN VARCHAR2,
        p_error_message            IN VARCHAR2 DEFAULT 'Search record not '|| 
                                                'found for the specified ID'
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.update_required_feedback_action(
            p_id                     => p_id,
            p_required_feedback_action => p_required_feedback_action,
            p_error_message          => p_error_message
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_required_feedback_action',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_required_feedback_action',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_required_feedback_action;

    PROCEDURE update_expected_report_id (
        p_search_id          IN VARCHAR2,
        p_expected_report_id IN VARCHAR2
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.update_expected_report_id(
            p_search_id          => p_search_id,
            p_expected_report_id => p_expected_report_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_expected_report_id',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_expected_report_id',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_expected_report_id;

    PROCEDURE update_expected_drilldown_id (
        p_search_id             IN VARCHAR2,
        p_expected_drilldown_id IN VARCHAR2
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.update_expected_drilldown_id(
            p_search_id             => p_search_id,
            p_expected_drilldown_id => p_expected_drilldown_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_expected_drilldown_id',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_expected_drilldown_id',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_expected_drilldown_id;

    PROCEDURE update_feedback_action_priority (
        p_id                  IN VARCHAR2,
        p_feedback_action_priority IN VARCHAR2,
        p_error_message       IN VARCHAR2 DEFAULT 'Search record not found'|| 
                                                'for the specified ID'
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.update_feedback_action_priority(
            p_id                     => p_id,
            p_feedback_action_priority => p_feedback_action_priority,
            p_error_message          => p_error_message
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_feedback_action_priority',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_feedback_action_priority',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END update_feedback_action_priority;

    PROCEDURE remove_expected_report_id (
        p_search_id         IN VARCHAR2
    ) IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.remove_expected_report_id(
            p_search_id     => p_search_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'remove_expected_report_id', v_start_time, v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'remove_expected_report_id', v_start_time, v_end_time, TRUE,
                SQLCODE, SQLERRM
            );
            RAISE;
    END remove_expected_report_id;

    PROCEDURE remove_expected_drilldown_id (
        p_search_id         IN VARCHAR2
    ) IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_feedback_pkg.remove_expected_drilldown_id(
            p_search_id     => p_search_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'remove_expected_drilldown_id', v_start_time, v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'remove_expected_drilldown_id', v_start_time, v_end_time, TRUE,
                SQLCODE, SQLERRM
            );
            RAISE;
    END remove_expected_drilldown_id;


    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_named_entities_pkg 
    Description: This package contains procedures related to adding and
                 retrieving named entities from langdata$named_entities table.
    ----------------------------------------------------------------------
    */
    PROCEDURE get_all_named_entities (
        p_named_entities OUT SYS_REFCURSOR
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_named_entities_pkg.get_all_named_entities(
            p_named_entities => p_named_entities
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_all_named_entities',v_start_time,v_end_time
        );
    END get_all_named_entities;

    PROCEDURE add_named_entity (
        p_id   IN VARCHAR2,
        p_name IN VARCHAR2
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_named_entities_pkg.add_named_entity(
            p_id   => p_id,
            p_name => p_name
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_named_entity',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'add_named_entity',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END add_named_entity;


    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_sample_queries_pkg
    Description: This package contains procedures and functions related 
                 to managing sample queries in the langdata$samplequeries table.
                 It includes retrieving, deleting, and adding deduplicated
                 sample queries.
    ----------------------------------------------------------------------
    */
    PROCEDURE get_report_sample_queries (
        p_report_id      IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_sample_queries_pkg.get_report_sample_queries(
            p_report_id      => p_report_id,
            p_sample_queries => p_sample_queries
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_report_sample_queries',v_start_time,v_end_time
        );
    END get_report_sample_queries;

    PROCEDURE get_drilldown_sample_queries (
        p_drilldown_id   IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_sample_queries_pkg.get_drilldown_sample_queries(
            p_drilldown_id   => p_drilldown_id,
            p_sample_queries => p_sample_queries
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_drilldown_sample_queries',v_start_time,v_end_time
        );
    END get_drilldown_sample_queries;

    PROCEDURE delete_sample_query (
        p_id IN VARCHAR2
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_sample_queries_pkg.delete_sample_query(
            p_id => p_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_sample_query',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_sample_query',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END delete_sample_query;

    PROCEDURE add_report_sample_queries (
        p_report_id     IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST,
        p_ids OUT SYS.ODCIVARCHAR2LIST
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_sample_queries_pkg.add_report_sample_queries(
            p_report_id     => p_report_id,
            p_sample_queries => p_sample_queries,
            p_ids            => p_ids
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_report_sample_queries',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'add_report_sample_queries',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END add_report_sample_queries;

    PROCEDURE add_drilldown_sample_queries (
        p_drilldown_id  IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST,
        p_ids OUT SYS.ODCIVARCHAR2LIST
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_sample_queries_pkg.add_drilldown_sample_queries(
            p_drilldown_id  => p_drilldown_id,
            p_sample_queries => p_sample_queries,
            p_ids            => p_ids
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_drilldown_sample_queries',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'add_drilldown_sample_queries',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END add_drilldown_sample_queries;

    FUNCTION get_all_sample_queries_fn (
        p_report_id IN VARCHAR2 DEFAULT NULL,
        p_drilldown_id IN VARCHAR2 DEFAULT NULL,
        p_limit     IN NUMBER DEFAULT 10
    ) RETURN data_table PIPELINED
    AS
        v_cursor      SYS_REFCURSOR;
        v_id            VARCHAR2(36);
        v_query_text    VARCHAR2(2000);
        v_version       NUMBER;
        v_report_id     VARCHAR2(36);
        v_drilldown_id  VARCHAR2(36);
        v_enhanced_text VARCHAR2(4000);
        v_status        VARCHAR2(20);
    BEGIN

        -- Call your existing get_drilldown_sample_queries procedure
        IF p_drilldown_id IS NOT NULL THEN

            lang_data_sample_queries_pkg.get_drilldown_sample_queries(
                p_drilldown_id => p_drilldown_id,
                p_sample_queries => v_cursor
            );

        ELSIF p_report_id IS NOT NULL THEN
            
            lang_data_sample_queries_pkg.get_report_sample_queries(
                p_report_id => p_report_id,
                p_sample_queries => v_cursor
            );

        END IF;

        -- Pipe each row from the ref cursor
        LOOP
            FETCH v_cursor INTO 
                v_id, 
                v_query_text, 
                v_version,
                v_report_id,
                v_drilldown_id,
                v_enhanced_text,
                v_status;
            EXIT WHEN v_cursor%NOTFOUND;

            PIPE ROW(data_row_obj(
                v_id,
                v_query_text,
                v_version,
                v_status,
                v_enhanced_text
            ));
        END LOOP;

        CLOSE v_cursor;
        RETURN;
    END get_all_sample_queries_fn;

    PROCEDURE update_sample_query_status(
        p_id            IN VARCHAR2,
        p_status        IN VARCHAR2,
        p_response      OUT VARCHAR2
    )
    IS
        v_status VARCHAR2(20);
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_sample_queries_pkg.update_sample_query_status(
            p_id  => p_id,
            p_status => p_status,
            p_response => v_status
        );
        p_response := v_status;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'add_drilldown_sample_queries',v_start_time,v_end_time
        );
    END update_sample_query_status;

    FUNCTION get_sample_query_id_by_report_id_version(
        p_report_id    IN VARCHAR2,
        p_version      IN NUMBER
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_sample_query_id   VARCHAR2(36);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_sample_query_id := 
            lang_data_sample_queries_pkg.get_sample_query_id_by_report_id_version(
                p_report_id, p_version
            );
        
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_sample_query_id_by_report_id_version',v_start_time,v_end_time
        );

        RETURN v_sample_query_id;
    END get_sample_query_id_by_report_id_version;


    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_search_pkg
    Description: This package contains procedures related to searching and 
                 retrieving user search records from the langdata$searchrecords
                 table. The procedures are designed to handle user search
                 records and related operations.
    ----------------------------------------------------------------------
    */
    PROCEDURE get_user_search_record (
        p_id                        IN VARCHAR2,
        p_query_text                OUT VARCHAR2,
        p_report_matches            OUT JSON,
        p_feedback_rating           OUT BOOLEAN,
        p_feedback_comments         OUT VARCHAR2,
        p_required_feedback_action  OUT VARCHAR2,
        p_feedback_action_priority  OUT VARCHAR2,
        p_expected_report_id        OUT VARCHAR2,
        p_expected_drilldown_id     OUT VARCHAR2,
        p_search_type               OUT VARCHAR2,
        p_augmented_query_text      OUT VARCHAR2
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_minimum_role_access;
        lang_data_search_pkg.get_user_search_record(
            p_id                        => p_id,
            p_query_text                => p_query_text,
            p_report_matches            => p_report_matches,
            p_feedback_rating           => p_feedback_rating,
            p_feedback_comments         => p_feedback_comments,
            p_required_feedback_action  => p_required_feedback_action,
            p_feedback_action_priority  => p_feedback_action_priority,
            p_expected_report_id        => p_expected_report_id,
            p_expected_drilldown_id     => p_expected_drilldown_id,
            p_search_type               => p_search_type,
            p_augmented_query_text      => p_augmented_query_text
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_user_search_record',v_start_time,v_end_time
        );
    END get_user_search_record;

    -- Procedure for retrieving user search records with pagination
    PROCEDURE get_all_user_search_records (
        p_feedback_rating           IN BOOLEAN DEFAULT NULL,
        p_required_feedback_action  IN VARCHAR2 DEFAULT NULL,
        p_feedback_action_priority  IN VARCHAR2 DEFAULT NULL,
        p_cursor                    IN OUT VARCHAR2,
        p_limit                     IN NUMBER DEFAULT 10,
        p_records                   OUT SYS_REFCURSOR
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_minimum_role_access;
        lang_data_search_pkg.get_all_user_search_records(
            p_feedback_rating           => p_feedback_rating,
            p_required_feedback_action  => p_required_feedback_action,
            p_feedback_action_priority  => p_feedback_action_priority,
            p_cursor                    => p_cursor,
            p_limit                     => p_limit,
            p_records                   => p_records
        );
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_all_user_search_records',v_start_time,v_end_time
        );
    END get_all_user_search_records;

    FUNCTION get_all_user_search_records_fn (
        p_feedback_rating            IN VARCHAR2 DEFAULT NULL,
        p_required_feedback_action   IN VARCHAR2 DEFAULT NULL,
        p_feedback_action_priority   IN VARCHAR2 DEFAULT NULL,
        p_limit                      IN NUMBER   DEFAULT 10
    )
    RETURN user_search_row_table PIPELINED
    IS
        v_cursor SYS_REFCURSOR;
        v_cursor_token VARCHAR2(4000) := NULL;

        -- Local variables matching object type
        v_id                       VARCHAR2(36);
        v_query_text               VARCHAR2(2000);
        v_report_matches_clob      CLOB;
        v_feedback_rating          VARCHAR2(20);
        v_feedback_comments        VARCHAR2(2000);
        v_required_action          VARCHAR2(200);
        v_priority                 VARCHAR2(200);
        v_expected_report_id       VARCHAR2(36);
        v_expected_drilldown_id    VARCHAR2(36);
        v_search_type              VARCHAR2(50);
        v_username                 VARCHAR2(200);
        v_augmented_query_text     VARCHAR2(2000);
        v_created_at               TIMESTAMP;
    BEGIN
        -- Call your existing procedure
        lang_data_search_pkg.get_all_user_search_records(
            p_feedback_rating         => NULL,
            p_required_feedback_action => NULL,
            p_feedback_action_priority => NULL,
            p_limit                   => 100,
            p_cursor                  => v_cursor_token,
            p_records                 => v_cursor
        );

        LOOP
            FETCH v_cursor INTO
                v_id,
                v_query_text,
                v_report_matches_clob,
                v_feedback_rating,
                v_feedback_comments,
                v_required_action,
                v_priority,
                v_expected_report_id,
                v_expected_drilldown_id,
                v_search_type,
                v_augmented_query_text,
                v_created_at;
            EXIT WHEN v_cursor%NOTFOUND;

            EXECUTE IMMEDIATE 
            'SELECT username FROM langdata$searchrecords WHERE id = :1'
            INTO v_username
            USING v_id;
            -- v_username := SYS_CONTEXT('USERENV', 'SESSION_USER');
            -- Pipe the row
            PIPE ROW (
                user_search_row_obj(
                    v_id,
                    v_query_text,
                    v_report_matches_clob,
                    v_feedback_rating,
                    v_feedback_comments,
                    v_required_action,
                    v_priority,
                    v_expected_report_id,
                    v_expected_drilldown_id,
                    v_search_type,
                    v_username,
                    v_augmented_query_text,
                    v_created_at
                )
            );
        END LOOP;

        CLOSE v_cursor;

        RETURN;
    END get_all_user_search_records_fn;

    -- Procedure for deleting a user search record
    PROCEDURE delete_user_search_record (
        p_id IN VARCHAR2
    )
    IS
       v_start_time        NUMBER;
       v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_search_pkg.delete_user_search_record(
            p_id => p_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_user_search_record',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_user_search_record',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END delete_user_search_record;

    -- Procedure for performing a search from a query
    PROCEDURE search_from_query (
        p_query          IN VARCHAR2,
        p_domain    IN VARCHAR2 DEFAULT NULL,
        p_search_id      OUT VARCHAR2
    )
    IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_minimum_role_access;
        lang_data_search_pkg.search_from_query(
            p_query        => p_query,
            p_domain       => p_domain,
            p_search_id    => p_search_id
        );
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'search_from_query',v_start_time,v_end_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'search_from_query',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END search_from_query;

    PROCEDURE plot_search_vectors(
        p_search_id IN VARCHAR2
    )
    IS
    BEGIN
        lang_data_auth_pkg.assert_app_expert_role;
        lang_data_search_pkg.plot_search_vectors(p_search_id);
        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            RAISE;
    END plot_search_vectors;

    FUNCTION get_schema_version
        RETURN VARCHAR2 IS
        v_schema_version VARCHAR2(255);
    BEGIN
        v_schema_version := lang_data_utils_pkg.get_schema_version;
        RETURN v_schema_version;
    EXCEPTION
        WHEN OTHERS THEN
            RETURN NULL;
    END get_schema_version;

    FUNCTION generate_sample_queries (
        p_title              IN VARCHAR2,
        p_description        IN VARCHAR2,
        p_filter_information IN VARCHAR2,
        p_sample_queries     IN SYS_REFCURSOR DEFAULT NULL,
        p_count              IN NUMBER DEFAULT 10
    ) RETURN CLOB IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_response          CLOB;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_app_expert_role;
        v_response := lang_data_llm_pkg.generate_sample_queries(
            p_title                 => p_title,
            p_description           => p_description,
            p_filter_information    => p_filter_information,
            p_count                 => p_count
        );
        
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'generate_sample_queries',v_start_time,v_end_time
        );

        RETURN v_response;
    END generate_sample_queries;


    FUNCTION get_text_by_id(
        p_description_id IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_text              VARCHAR2(4000);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;

        lang_data_auth_pkg.assert_minimum_role_access;
        v_text := lang_data_utils_pkg.get_text_by_id(p_description_id);

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_text_by_id',v_start_time,v_end_time
        );
        RETURN v_text;
    END get_text_by_id;

    FUNCTION apply_filter_values_for_url (
        p_url_template      VARCHAR2,
        p_report_matches    langdata$searchrecords.report_matches%TYPE
    ) RETURN VARCHAR2 IS
        v_start_time        NUMBER;
        v_end_time          NUMBER;
        v_final_url         VARCHAR2(4000) := p_url_template;
        CURSOR c_filters IS
            SELECT filter_name, filter_value
            FROM JSON_TABLE(
                p_report_matches,
                '$.filter_values[*]' COLUMNS (
                    filter_name     VARCHAR2(200)  PATH '$.filter_name',
                    filter_value    VARCHAR2(2000) PATH '$.value'
                )
            );

        v_filter_name   VARCHAR2(200);
        v_filter_value  VARCHAR2(2000);
    BEGIN
        OPEN c_filters;
        LOOP
            FETCH c_filters INTO v_filter_name, v_filter_value;
            EXIT WHEN c_filters%NOTFOUND;
            v_final_url := REPLACE(
                v_final_url, ':' || v_filter_name, v_filter_value);
        END LOOP;
        CLOSE c_filters;

        RETURN v_final_url;
    END apply_filter_values_for_url;

    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_utils_pkg
    Description: This package provides a wide range of utility functions and 
                 procedures used across the application
    ----------------------------------------------------------------------
    */

    FUNCTION update_domain(
        p_domain_id IN VARCHAR2,
        p_new_name IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
        v_result BOOLEAN;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_utils_pkg.update_domain(p_domain_id, p_new_name);
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'update_domain',v_start_time,v_end_time
        );
        RETURN v_result;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'update_domain',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END update_domain;

    FUNCTION delete_domain(
        p_domain_id IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
        v_result BOOLEAN;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_utils_pkg.delete_domain(p_domain_id);
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'delete_domain',v_start_time,v_end_time
        );
        RETURN v_result;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'delete_domain',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END delete_domain;

    FUNCTION get_all_domains RETURN SYS_REFCURSOR IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
        v_cursor SYS_REFCURSOR;
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_minimum_role_access;
        v_cursor := lang_data_utils_pkg.get_all_domains;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_all_domains',v_start_time,v_end_time
        );
        RETURN v_cursor;
    EXCEPTION
        WHEN OTHERS THEN
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_all_domains',v_start_time,v_end_time,TRUE,SQLCODE,SQLERRM
            );
            RAISE;
    END get_all_domains;

    FUNCTION get_or_create_domain(
        p_domain_name IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
        v_result VARCHAR2(36);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_utils_pkg.get_or_create_domain(p_domain_name);
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_or_create_domain',v_start_time,v_end_time
        );
        RETURN v_result;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_or_create_domain',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END get_or_create_domain;

    FUNCTION get_domain_id(
        p_domain_name IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_start_time NUMBER;
        v_end_time  NUMBER;
        v_result VARCHAR2(36);
    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_auth_pkg.assert_app_expert_role;
        v_result := lang_data_utils_pkg.get_domain_id(p_domain_name);
        COMMIT;
        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_analytics_pkg.update_api_logs(
            'get_domain_id',v_start_time,v_end_time
        );
        RETURN v_result;
    EXCEPTION
        WHEN OTHERS THEN
            ROLLBACK;
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_analytics_pkg.update_api_logs(
                'get_domain_id',v_start_time,v_end_time,TRUE,
                SQLCODE,SQLERRM
            );
            RAISE;
    END get_domain_id;

    PROCEDURE grant_user_table_privileges (
        p_schema_name   IN  VARCHAR2,
        p_table_name    IN  VARCHAR2,
        p_success       OUT BOOLEAN
    ) IS
        v_sql           VARCHAR2(4000);
        v_schema_name   VARCHAR2(255);
    BEGIN
        p_success := TRUE;
        v_schema_name := lang_data_utils_pkg.normalize_schema_name(
            p_schema_name
        );
        -- GRANT READ
        BEGIN
            v_sql := 'GRANT READ ON ' || v_schema_name || '.' ||
                     UPPER(p_table_name) || ' TO LANGDATA';
            lang_data_logger_pkg.log_info(v_sql);
            EXECUTE IMMEDIATE v_sql;
        EXCEPTION
            WHEN OTHERS THEN
                p_success := FALSE;
                lang_data_logger_pkg.log_warn(
                    'Failed to grant READ: ' || SQLERRM
                );
        END;

        -- GRANT SELECT
        BEGIN
            v_sql := 'GRANT SELECT ON ' || v_schema_name || '.' ||
                     UPPER(p_table_name) || ' TO LANGDATA';
            lang_data_logger_pkg.log_info(v_sql);
            EXECUTE IMMEDIATE v_sql;
        EXCEPTION
            WHEN OTHERS THEN
                p_success := FALSE;
                lang_data_logger_pkg.log_warn(
                    'Failed to grant SELECT: ' || SQLERRM
                );
        END;

        -- GRANT ALTER
        BEGIN
            v_sql := 'GRANT ALTER ON ' || v_schema_name || '.' ||
                     UPPER(p_table_name) || ' TO LANGDATA';
            lang_data_logger_pkg.log_info(v_sql);
            EXECUTE IMMEDIATE v_sql;
        EXCEPTION
            WHEN OTHERS THEN
                p_success := FALSE;
                lang_data_logger_pkg.log_warn(
                    'Failed to grant ALTER: ' || SQLERRM
                );
        END;

        -- GRANT FLASHBACK
        BEGIN
            v_sql := 'GRANT FLASHBACK ON ' || v_schema_name || '.' ||
                     UPPER(p_table_name) || ' TO LANGDATA';
            lang_data_logger_pkg.log_info(v_sql);
            EXECUTE IMMEDIATE v_sql;
        EXCEPTION
            WHEN OTHERS THEN
                p_success := FALSE;
                lang_data_logger_pkg.log_warn(
                    'Failed to grant FLASHBACK: ' || SQLERRM
                );
        END;
    END grant_user_table_privileges;

    PROCEDURE validate_match_document_and_grant_table_privileges(
        match_document IN JSON
    ) IS
        -- Variables to hold parsed JSON data
        v_json            JSON_OBJECT_T;
        v_filters         JSON_ARRAY_T;
        v_filter_item     JSON_OBJECT_T;
        v_default_value   VARCHAR2(4000);
        v_use_ner         BOOLEAN;
        v_table_name      VARCHAR2(128) := NULL;
        v_schema_name     VARCHAR2(128) := NULL;
        v_column_name     VARCHAR2(128) := NULL;
        v_db_link_name    VARCHAR2(128) := NULL;
        v_entity_type     VARCHAR2(255);
        v_data_type       VARCHAR2(255);
        v_unique_count    NUMBER;
        v_is_unique       NUMBER;
        v_entitiy_found   NUMBER := 0;
        v_enumerable_set  JSON_ARRAY_T;
        v_grant_success   BOOLEAN;
    BEGIN
        -- Convert CLOB to JSON
        v_json := JSON_OBJECT_T.parse(json_serialize(match_document));

        -- Get the 'filters' array
        v_filters := v_json.GET_ARRAY('filters');

        -- Check if 'filters' key exists
        IF v_filters IS NULL THEN
            lang_data_logger_pkg.log_debug(
                'Missing ''filters'' key in matchDocument.'
            );
            RETURN;
        END IF;

        -- Check if 'filters' is a JSON array
        IF NOT v_filters.IS_ARRAY THEN
            lang_data_logger_pkg.log_debug('''filters'' should be a list.');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_match_document
            );
        END IF;

        -- Loop through each filter item
        FOR indx IN 0 .. v_filters.get_size - 1
        LOOP
        
            v_filter_item := JSON_OBJECT_T(v_filters.GET(to_number(indx)));

            -- Ensure each filter has required keys
            IF v_filter_item.GET_STRING('filter_name') IS NULL THEN
                lang_data_logger_pkg.log_debug(
                    'Filter item missing ''filter_name''.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_match_document
                );
            END IF;

            IF NOT v_filter_item.HAS('default_value') THEN
                lang_data_logger_pkg.log_debug(
                    'Filter item missing ''default_value''.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_match_document
                );
            END IF;

            -- Validate 'use_ner' is a boolean
            v_use_ner := v_filter_item.GET_BOOLEAN('use_ner');

            IF v_use_ner is NULL THEN
                lang_data_logger_pkg.log_debug(
                    'Filter item missing ''use_ner''.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_match_document
                );
            END IF;

            -- If 'use_ner' is False, ensure 'table_name' and 'column_name' are
            -- present
            IF NOT v_use_ner THEN

                v_enumerable_set := v_filter_item.GET_ARRAY('enumerable_set');
                v_table_name := v_filter_item.GET_STRING('table_name');
                v_column_name := v_filter_item.GET_STRING('column_name');
                v_schema_name := v_filter_item.GET_STRING('schema_name');
                v_db_link_name := v_filter_item.GET_STRING('db_link_name');
                v_schema_name := lang_data_utils_pkg.normalize_schema_name(
                    v_schema_name
                );
                
                IF v_enumerable_set IS NOT NULL THEN
                    IF v_table_name IS NOT NULL OR v_column_name IS NOT NULL THEN
                        lang_data_logger_pkg.log_debug(
                            'Filter item can''t both be associated with ' ||
                            'a table column and ''enumerable_set''.'
                        );
                        lang_data_errors_pkg.raise_error(
                            lang_data_errors_pkg.c_invalid_match_document
                        );
                    END IF;
                    CONTINUE;
                END IF;

                IF v_db_link_name IS NOT NULL AND
                  NOT lang_data_utils_pkg.check_db_link_exists(
                        v_db_link_name
                    ) THEN
                        lang_data_logger_pkg.log_debug(
                            'Database link doesn''t exist.'
                        );
                        lang_data_errors_pkg.raise_error(
                            lang_data_errors_pkg.c_invalid_match_document
                        );
                END IF;

                IF v_table_name IS NULL THEN
                    lang_data_logger_pkg.log_debug(
                        'Filter item missing ''table_name'' ' ||
                        'when ''use_ner'' is False.'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_match_document
                    );
                END IF;

                IF v_column_name IS NULL THEN
                    lang_data_logger_pkg.log_debug(
                        'Filter item missing ''column_name'' ' ||
                        'when ''use_ner'' is False.'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_match_document
                    );
                END IF;

                IF v_schema_name IS NULL THEN
                    -- Use Current schema in case Schema Name is missing
                    lang_data_logger_pkg.log_debug(
                        'Filter item missing ''schema_name'' '
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_match_document
                    );
                END IF;

                -- Cannot grant privileges on remote tables
                -- Privileges on remote tables are the same as remote user
                -- specified by public database link
                IF v_db_link_name IS NULL THEN
                    -- If failed, will throw invalid match document error
                    lang_data.grant_user_table_privileges(
                        v_schema_name, v_table_name, v_grant_success
                    );
                    IF NOT v_grant_success THEN
                        lang_data_errors_pkg.raise_error(
                            lang_data_errors_pkg.c_invalid_match_document
                        );
                    END IF;
                END IF;

                lang_data_utils_pkg.validate_enumerable_column(
                    p_table_name    => v_table_name,
                    p_column_name   => v_column_name,
                    p_schema_name   => v_schema_name,
                    p_db_link_name  => v_db_link_name
                );

            ELSE
                -- If 'use_ner' is True, ensure 'entity_type' is present
                IF  v_filter_item.GET_STRING('entity_type') IS NULL THEN
                    lang_data_logger_pkg.log_debug(
                        'Filter item missing ''entity_type'' ' ||
                        'when ''use_ner'' is True.'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_match_document
                    );

                
                ELSIF lang_data_config_pkg.g_custom_entities = FALSE 
                THEN
                    -- Validate named_entity is a supported entity_type
                    v_entity_type := v_filter_item.GET_STRING('entity_type');
                    
                    SELECT COUNT(*) INTO v_entitiy_found  
                    FROM langdata$named_entities
                    WHERE name = LOWER(v_entity_type);
                    
                    IF v_entitiy_found = 0 THEN
                        lang_data_logger_pkg.log_debug(
                            'Filter item ''entity_type'': '||
                            v_entity_type || ' is not supported.'
                        );
                        lang_data_errors_pkg.raise_error(
                            lang_data_errors_pkg.c_unsupported_entity_type
                        );
                    END IF;

                END IF;

                IF v_filter_item.GET_STRING('table_name') IS NOT NULL AND 
                v_filter_item.GET_STRING('column_name') IS NULL THEN
                    lang_data_logger_pkg.log_debug(
                        'Filter item missing ''column_name'' ' ||
                        'when ''table_name'' is provided.'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_match_document
                    );
                END IF;
            END IF;
        END LOOP;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_match_document THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                -- Log unknown errors as fatal and raise the generic error code
                lang_data_logger_pkg.log_fatal(
                    'An error occurred while validating match document. ' ||
                    'Error: ' || SQLERRM
                );
                RAISE;
            END IF;
    END validate_match_document_and_grant_table_privileges;

END lang_data;
/
