Rem
Rem $Header: dbgendev/src/langdata/plsql/drilldowns/drilldowns_pkg.pkb /main/43 2025/08/06 08:08:53 saloshah Exp $
Rem
Rem drilldowns_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      drilldowns_pkg.pkb - Package Body of drilldowns_pkg
Rem
Rem    DESCRIPTION
Rem      This package contains implementation of functions/procedures for
Rem      managing drilldown documents and their descriptions.
Rem
Rem    NOTES
Rem      NONE
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/drilldowns/drilldowns_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    08/01/25 - Remove duplicate call to validate_match_document,
Rem                           now called by lang_data.create_drilldown
Rem    saloshah    07/30/25 - DBAI-1100: Added exceptions for non existent id
Rem                           for update_drilldown_description
Rem    deveverm    07/30/25 - DBAI-1161: made regression analysis respect 
Rem                           domains
Rem    dadoshi     07/24/25 - JIRA_DBAI1080: Add
Rem                           get_drilldown_match_document_by_id
Rem    dadoshi     07/22/25 - JIRA_DBAI1080: Added get_drilldown_status_by_id
Rem    jiangnhu    07/10/25 - DBAI-1006: Parameterize job class name
Rem    fgurrola    06/27/25 - DBAI-776: Disable adding custom entity_types.
Rem    jiangnhu    06/25/25 - Add validation for p_report_id in create_drilldown
Rem                           add validation for p_drilldown_id in
Rem                           enumerable-set-update APIs
Rem    jiangnhu    06/24/25 - DBAI-909: Integrate text augmentation with domain
Rem    jiangnhu    06/09/25 - DBAI-871: Implement APIs to get ID by unique
Rem                           combination of title, version, etc.
Rem    ruohli      06/06/25 - DBAI-842: Added parallel query for 
Rem                           calculate_drilldown_regression and remove
Rem                           top 2 matching logic
Rem    pryarla     06/06/25 - DBAI-806: Add domains
Rem    jiangnhu    06/02/25 - DBAI-844: Remove call to
Rem                           update_annotation_and_comment_records
Rem    jiangnhu    05/29/25 - Update job name of creating enumerable-set-based
Rem                           value vector partition
Rem    deveverm    05/16/25 - DBAI-761: changed
Rem                           get_drilldown_description_regression to
Rem                           get_drilldown_regression, changed
Rem                           calculate_drilldown_description_regression to
Rem                           calculate_drilldown_regression
Rem    jiangnhu    04/01/25 - DBAI-624: Make embedding model independent
Rem    deveverm    04/01/25 - DBAI-523: added
Rem                           get_drilldown_description_regression and modified
Rem                           regression logic
Rem    jiangnhu    03/26/25 - DBAI-692: make report_id, description_id OUT
Rem                           parameter, make enumeration limit a config
Rem    jiangnhu    03/24/25 - DBAI-551: Use NER to augment description
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/15/25 - DBAI-661: Implement
Rem                           replace_drilldown_filter_enumerable_set
Rem    anisbans    03/11/25 - DBAI-556 : update ref_count for user_ctx_indexes 
Rem                                      upon drilldown deletion
Rem    dadoshi     03/11/25 - JIRA_DBAI574: Add add_named_entities_to_document
Rem                           API usage
Rem    arevathi    03/11/25 - Add Update status API's
Rem    jiangnhu    03/03/25 - Fix create_drilldown INSERT NULL issue
Rem    jiangnhu    02/28/25 - Fix calls to amend_description with amended text
Rem    dadoshi     02/24/25 - Throw errors in create_drilldown() before
Rem                           augmenting, amending, and validating
Rem                           match_document
Rem    dadoshi     11/07/24 - Fix create drilldown in case IDs not provided.
Rem    jiangnhu    02/21/25 - Fix the error handling of
Rem                           delete_drilldown_description
Rem    anisbans    02/19/25 - JIRA_DBAI-557: Apply ref count for value vectors  
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    02/13/25 - DBAI-524: create context index based on match
Rem                           document
Rem    jiangnhu    02/11/25 - Normalize description before calculate md5
Rem    jiangnhu    02/07/25 - Remove update_expected_drilldown_id: duplicate
Rem                           with lang_data_feedback_pkg
Rem    jiangnhu    02/04/25 - Use generate_id
Rem    jiangnhu    01/31/25 - JIRA_DBAI-403: Add augment_text to process 
Rem                           description
Rem    dadoshi     01/22/25 - JIRA_DBAI-507: update create_drilldown to have
Rem                           id, and, description_id as optional args
Rem    jiangnhu    12/09/24 - JIRA_DBAI-458: Use add_drilldown_sample_queries
Rem    dadoshi     11/07/24 - Fix new description regression count based. 
Rem    dadoshi     11/07/24 - Fix get_drilldown_description()
Rem    dadoshi     11/07/24 - Add get_drilldown()
Rem    dadoshi     11/04/24 - Update execute_drilldown_sql to handle large
Rem                           number of rows
Rem    arevathi    10/30/24 - Added drilldown description change regression
Rem                           count
Rem    dadoshi     10/29/24 - Add get_drilldown_descriptions()
Rem    dadoshi     10/29/24 - Format drilldowns_pkg
Rem    dadoshi     10/29/24 - Add documentation
Rem    dadoshi     10/25/24 - Implement drilldowns-specific functionality
Rem    dadoshi     10/25/24 - Created
Rem

create or replace package body lang_data_drilldowns_pkg as
    PROCEDURE get_drilldown_descriptions_paginated (
        p_id            IN VARCHAR2,
        p_limit         IN NUMBER DEFAULT 10,
        p_cursor        IN OUT VARCHAR2,
        p_descriptions  OUT SYS_REFCURSOR
    ) IS
        v_cur                 INTEGER;
        v_rowcount            INTEGER;
        v_sql_text            VARCHAR2(4000);
        v_descriptions_query  VARCHAR2(4000) := 'SELECT id, text, version, ' || 
                 'status, enhanced_text FROM langdata$drilldowndescriptions' || 
                 ' WHERE drilldown_id = :drilldown_id';
        v_conditions          VARCHAR2(4000) := ' AND ' || 
        '(created_at < :cursor_created_at OR ' || 
        '(created_at = :cursor_created_at AND id <= :cursor_id))';
        v_cursor_id           VARCHAR2(4000);
        v_cursor_created_at   TIMESTAMP;
        v_last_row_id         VARCHAR2(4000);
        v_last_row_created_at TIMESTAMP;
    BEGIN
         -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        v_cur := DBMS_SQL.OPEN_CURSOR;

        IF p_cursor IS NOT NULL THEN
            lang_data_utils_pkg.split_cursor(p_cursor => p_cursor, 
                                            p_created_at => v_cursor_created_at, 
                                            p_id => v_cursor_id);

            lang_data_logger_pkg.log_debug(
                'Applied pagination cursor: CreatedAt=' ||
                TO_CHAR(v_cursor_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') || 
                ', ID=' || v_cursor_id
            );
            v_sql_text := v_descriptions_query || v_conditions;
        ELSE
            v_sql_text := v_descriptions_query;
        END IF;

        IF p_limit IS NOT NULL THEN
            v_sql_text := v_sql_text || ' ORDER BY created_at DESC, id DESC' || 
            ' FETCH FIRST :limit ROWS ONLY ';
        ELSE
            v_sql_text := v_sql_text || ' ORDER BY created_at DESC, id DESC ';
        END IF;

        lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);

        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_id);

        IF p_cursor IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':cursor_created_at', v_cursor_created_at
            );
            DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
        END IF;
        
        IF p_limit IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);
        END IF;

        v_rowcount :=  DBMS_SQL.EXECUTE(v_cur);        
        p_descriptions  :=  DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        IF p_limit IS NOT NULL THEN
            lang_data_logger_pkg.log_debug('Checking for next description row');
            
            v_sql_text := 'SELECT id, created_at' || 
            ' FROM langdata$drilldowndescriptions' ||
            ' WHERE drilldown_id = :drilldown_id ';
                            
            IF p_cursor IS NOT NULL THEN
                v_sql_text := v_sql_text || v_conditions;
            END IF;

            v_sql_text := v_sql_text || 
            ' ORDER BY created_at DESC, id DESC OFFSET :limit ROWS ' || 
            'FETCH NEXT 1 ROWS ONLY';

            v_cur :=  DBMS_SQL.OPEN_CURSOR;
            
            lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);
            DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

            IF p_cursor IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(
                    v_cur, ':cursor_created_at', v_cursor_created_at
                );
                DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
            END IF;

            DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_id);
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);


            DBMS_SQL.DEFINE_COLUMN(v_cur, 1, v_last_row_id, 36);
            DBMS_SQL.DEFINE_COLUMN(v_cur, 2, v_last_row_created_at);

            v_rowcount := DBMS_SQL.EXECUTE(v_cur);

            IF DBMS_SQL.FETCH_ROWS(v_cur) > 0 THEN
                DBMS_SQL.COLUMN_VALUE(v_cur, 1, v_last_row_id);
                DBMS_SQL.COLUMN_VALUE(v_cur, 2, v_last_row_created_at);

                -- Set the next cursor for pagination
                p_cursor := TO_CHAR(
                    v_last_row_created_at, 'YYYY-MM-DD HH24:MI:SS.FF'
                ) || '|' || v_last_row_id;
                lang_data_logger_pkg.log_debug(
                    'Next description cursor set to: ' || p_cursor
                );
            ELSE
                p_cursor := NULL;
                lang_data_logger_pkg.log_debug('No further pages.');
            END IF;

            IF DBMS_SQL.IS_OPEN(v_cur) THEN
                -- Close the next cursor
                DBMS_SQL.CLOSE_CURSOR(v_cur);
            END IF;
        ELSE
            p_cursor := NULL;
            lang_data_logger_pkg.log_debug('No further pages.');
        END IF;

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;
    
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_unauthorized_code THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for ID: ' || p_id || 
                    '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;

    END get_drilldown_descriptions_paginated;

    PROCEDURE get_drilldown_paginated (
        p_id                  IN VARCHAR2,
        p_report_id           OUT VARCHAR2,
        p_title               OUT VARCHAR2,
		p_match_document      OUT JSON,
		p_status              OUT VARCHAR2,
        p_descriptions        OUT SYS_REFCURSOR,
        p_descriptions_cur    IN OUT VARCHAR2,
        p_descriptions_limit  IN NUMBER DEFAULT 10,
        p_sample_queries      OUT SYS_REFCURSOR,
        p_sample_query_cur    IN OUT VARCHAR2,
        p_sample_query_limit  IN NUMBER DEFAULT 10
    ) IS 
        v_cur              INTEGER;
        v_dummy            INTEGER;
        v_drilldown_query  VARCHAR2(4000) := 'SELECT report_id, title, ' || 
        'match_document, status from langdata$drilldowndocuments ' || 
        'WHERE id = :drilldown_id';
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure to get_drilldown for ID: ' || p_id
        );
        
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        v_cur := DBMS_SQL.OPEN_CURSOR;
        -- Parse the SQL statement
        DBMS_SQL.PARSE(v_cur, v_drilldown_query, DBMS_SQL.NATIVE);

        -- Bind the drilldown ID to the SQL query
        DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_id);

         -- Define the columns to retrieve the result set into variables
        DBMS_SQL.DEFINE_COLUMN(v_cur, 1, p_report_id, 200);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 2, p_title, 500);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 3, p_match_document);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 4, p_status, 50);

        -- Execute the query
        v_dummy := DBMS_SQL.EXECUTE(v_cur);

        -- Fetch the results
        IF DBMS_SQL.FETCH_ROWS(v_cur) > 0 THEN
            DBMS_SQL.COLUMN_VALUE(v_cur, 1, p_report_id);
            DBMS_SQL.COLUMN_VALUE(v_cur, 2, p_title);
            DBMS_SQL.COLUMN_VALUE(v_cur, 3, p_match_document);
            DBMS_SQL.COLUMN_VALUE(v_cur, 4, p_status);
        ELSE
            -- Raise an error if no data is found
            lang_data_logger_pkg.log_error(
                'No Drilldown found for ID: '|| p_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        -- Load the query to fetch report descriptions
        lang_data_drilldowns_pkg.get_drilldown_descriptions_paginated (
            p_id                => p_id,
            p_limit             => p_descriptions_limit,
            p_cursor            => p_descriptions_cur,
            p_descriptions      => p_descriptions
        );
        -- Load the query to fetch sample queries
        lang_data_utils_pkg.get_sample_queries_paginated(
            p_report_id         =>  NULL,
            p_drilldown_id      =>  p_id,
            p_limit             =>  p_sample_query_limit,
            p_cursor            =>  p_sample_query_cur,
            p_sample_queries    =>  p_sample_queries
        );
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No drilldown found for ID: '|| p_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_resource_not_found,
                lang_data_errors_pkg.c_unauthorized_code
            ) THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for ID: ' || p_id || 
                    '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
            
    END get_drilldown_paginated;

    PROCEDURE get_all_drilldowns (
        p_report_id    IN VARCHAR2 DEFAULT NULL,
        p_status       IN VARCHAR2 DEFAULT NULL,
        p_cursor       IN OUT VARCHAR2,
        p_limit        IN NUMBER DEFAULT 10,
        p_drilldowns   OUT SYS_REFCURSOR
    ) IS
        v_cur               INTEGER;
        v_created_at        TIMESTAMP;
        v_cursor_id         VARCHAR2(36);
        v_last_id           VARCHAR2(36);
        v_last_created_at   TIMESTAMP;
        v_sql_text          VARCHAR2(4000);
        v_conditions        VARCHAR2(4000) := 
            '(d.created_at < :created_at ' || 
            'OR (d.created_at = :created_at AND d.id <= :cursor_id))';
        v_dummy             INTEGER;
    BEGIN
        -- Authorization check
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Validate status
        IF p_status IS NOT NULL AND p_status NOT IN (
            'Pending Review', 'Approved', 'Rejected', 'Published',
            'Inactive', 'Archived', 'Pending Regression'
        ) THEN
            lang_data_logger_pkg.log_error('Drilldown status invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Base SELECT
        v_sql_text := 
            'SELECT d.id, d.title, d.match_document, d.status, ' ||
            'd.report_id, d.domain_id, ' ||
            '(SELECT text FROM (SELECT dd.text FROM ' ||
            'langdata$drilldowndescriptions dd ' ||
            'WHERE dd.drilldown_id = d.id AND dd.status = ''Published'' ' ||
            'ORDER BY dd.created_at DESC) WHERE ROWNUM = 1) ' ||
            'AS drilldown_description_text ' ||
            'FROM langdata$drilldowndocuments d ';

        -- WHERE conditions
        IF p_report_id IS NOT NULL OR p_status IS NOT NULL THEN
            v_sql_text := v_sql_text || 'WHERE ';
            IF p_report_id IS NOT NULL THEN
                v_sql_text := v_sql_text || 'd.report_id = :report_id ';
            END IF;
            IF p_status IS NOT NULL THEN
                IF p_report_id IS NOT NULL THEN
                    v_sql_text := v_sql_text || 'AND ';
                END IF;
                v_sql_text := v_sql_text || 'd.status = :status ';
            END IF;
        END IF;

        -- Cursor-based pagination
        IF p_cursor IS NOT NULL THEN
            lang_data_utils_pkg.split_cursor(
                p_cursor => p_cursor,
                p_created_at => v_created_at,
                p_id => v_cursor_id
            );

            lang_data_logger_pkg.log_debug(
                'Applied pagination cursor: CreatedAt=' || 
                TO_CHAR(v_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') ||
                ', ID=' || v_cursor_id
            );

            IF p_report_id IS NOT NULL OR p_status IS NOT NULL THEN
                v_sql_text := v_sql_text || ' AND ';
            ELSE
                v_sql_text := v_sql_text || 'WHERE ';
            END IF;

            v_sql_text := v_sql_text || v_conditions;
        END IF;

        -- ORDER and LIMIT
        v_sql_text := v_sql_text || 
            ' ORDER BY d.created_at DESC, d.id DESC ';

        IF p_limit IS NOT NULL THEN
            v_sql_text := v_sql_text || 'FETCH FIRST :limit ROWS ONLY';
        END IF;

        -- Log and execute
        lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);
        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

        IF p_report_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);
        END IF;
        IF p_status IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':status', p_status);
        END IF;
        IF p_cursor IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':created_at', v_created_at);
            DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
        END IF;
        IF p_limit IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);
        END IF;

        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_drilldowns := DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Compute next cursor for pagination
        v_sql_text := 
        'SELECT d.id, d.created_at FROM langdata$drilldowndocuments d ';
        IF p_report_id IS NOT NULL OR p_status IS NOT NULL THEN
            v_sql_text := v_sql_text || 'WHERE ';
            IF p_report_id IS NOT NULL THEN
                v_sql_text := v_sql_text || 'd.report_id = :report_id ';
            END IF;
            IF p_status IS NOT NULL THEN
                IF p_report_id IS NOT NULL THEN
                    v_sql_text := v_sql_text || 'AND ';
                END IF;
                v_sql_text := v_sql_text || 'd.status = :status ';
            END IF;
        END IF;

        IF p_cursor IS NOT NULL THEN
            IF p_report_id IS NOT NULL OR p_status IS NOT NULL THEN
                v_sql_text := v_sql_text || 'AND ';
            ELSE
                v_sql_text := v_sql_text || 'WHERE ';
            END IF;
            v_sql_text := v_sql_text || v_conditions;
        END IF;

        v_sql_text := v_sql_text || 
            ' ORDER BY d.created_at DESC, d.id DESC OFFSET ' ||
            ':limit ROWS FETCH NEXT 1 ROWS ONLY';

        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

        IF p_report_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);
        END IF;
        IF p_status IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':status', p_status);
        END IF;
        IF p_cursor IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':created_at', v_created_at);
            DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
        END IF;
        DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);

        DBMS_SQL.DEFINE_COLUMN(v_cur, 1, v_last_id, 36);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 2, v_last_created_at);

        v_dummy := DBMS_SQL.EXECUTE(v_cur);

        IF DBMS_SQL.FETCH_ROWS(v_cur) > 0 THEN
            DBMS_SQL.COLUMN_VALUE(v_cur, 1, v_last_id);
            DBMS_SQL.COLUMN_VALUE(v_cur, 2, v_last_created_at);

            p_cursor := TO_CHAR(v_last_created_at,'YYYY-MM-DD HH24:MI:SS.FF') ||
             '|' || v_last_id;

            lang_data_logger_pkg.log_debug(
                'Next drilldown cursor set to: ' || p_cursor
            );
        ELSE
            p_cursor := NULL;
            lang_data_logger_pkg.log_debug('No further pages.');
        END IF;

        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        lang_data_logger_pkg.log_info('Successfully retrieved all drilldowns');
    EXCEPTION
        WHEN OTHERS THEN
            IF DBMS_SQL.IS_OPEN(v_cur) THEN
                DBMS_SQL.CLOSE_CURSOR(v_cur);
            END IF;
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_error(
                'Unexpected error in get_all_drilldowns: ' || SQLERRM
            );
            RAISE;
    END get_all_drilldowns;

    PROCEDURE delete_drilldown (
        p_drilldown_id IN VARCHAR2
    )
    IS
        v_sql_query VARCHAR2(4000) := 'DELETE FROM ' || 
                        'langdata$drilldowndocuments WHERE id = :drilldown_id';
        v_cur               INTEGER;
        v_dummy             INTEGER;
        v_match_document    JSON;
    
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting delete drilldown procedure: '|| p_drilldown_id
        );
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Verify if the drilldown exists
        IF NOT lang_data_utils_pkg.check_drilldown_exists(
            p_drilldown_id
        ) THEN
            lang_data_logger_pkg.log_error( 'No drilldown exists with the id '
             || p_drilldown_id);

            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );

        END IF;
        
        -- Retrieve match_document to extract associated value vector tables
        SELECT match_document INTO v_match_document
        FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;

        lang_data_utils_pkg.decrement_vector_table_references(v_match_document);

        lang_data_utils_pkg.drop_enumerable_set_value_vector_tables(
            p_drilldown_id, v_match_document
        );

        v_cur := DBMS_SQL.OPEN_CURSOR;
        -- Parse the SQL statement
        DBMS_SQL.PARSE(v_cur, v_sql_query, DBMS_SQL.NATIVE);

        -- Bind the drilldown ID to the SQL query
        DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_drilldown_id);
        
        -- Execute the query
        v_dummy := DBMS_SQL.EXECUTE(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_resource_not_found,
                lang_data_errors_pkg.c_unauthorized_code
            ) THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for ID: ' || p_drilldown_id || 
                    '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
            
    END delete_drilldown;

    PROCEDURE delete_drilldown_description (
        p_description_id IN VARCHAR2
    )
    IS
        v_count NUMBER;
    BEGIN
         -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Check if there is at least one description for the drilldown
        SELECT COUNT(*)
        INTO v_count
        FROM langdata$drilldowndescriptions
        WHERE drilldown_id = (
            SELECT drilldown_id
            FROM langdata$drilldowndescriptions
            WHERE id = p_description_id
        );

        IF v_count = 1 THEN
            lang_data_logger_pkg.log_error(
                'Drilldown must have at least one description.'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_description_invalid_deletion
            );
        END IF;

        -- Delete the specified drilldown description
        DELETE FROM langdata$drilldowndescriptions
        WHERE id = p_description_id;

        -- Check if the deletion was successful
        IF SQL%ROWCOUNT = 0 THEN
            lang_data_logger_pkg.log_error(
                'Drilldown description not found.'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;
        
        COMMIT; -- Commit the changes
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_resource_not_found,
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_description_invalid_deletion
            ) THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for drilldown description with ID: ' 
                    || p_description_id || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
    END delete_drilldown_description;

    -- Note: This mutation is now not used through frontend.
    PROCEDURE update_drilldown_description (
        p_drilldown_description_id IN  VARCHAR2,
        p_new_description          IN  VARCHAR2
    ) IS
        v_ner_augmented_desc        VARCHAR2(4000);
        v_enhanced_description      VARCHAR2(4000);
        v_description_md5           VARCHAR2(32);
        v_drilldown_id              VARCHAR2(36) := NULL;
        v_match_document            JSON;
        v_description_md5_check     NUMBER := 0;
        v_status                    VARCHAR2(20);
        v_normalized_description    VARCHAR2(4000);
        v_augmented_tokens          JSON;
        v_ner_augmentation_enabled  BOOLEAN := FALSE;
        v_entities                  JSON;
        v_domain_id                 VARCHAR2(36);

    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure to update drilldown description of ID: ' || 
            p_drilldown_description_id
        );
        
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Check if new description length is under limit
        IF LENGTH(p_new_description) > 2000 THEN
            lang_data_logger_pkg.log_error(
                'Description length exceeded the limit'
            );  
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        BEGIN 
            SELECT drilldown_id, status INTO v_drilldown_id, v_status 
            FROM langdata$drilldowndescriptions 
            WHERE id = p_drilldown_description_id;

            -- Check if status is in PENDING_REVIEW
            IF v_status != 'Pending Review' THEN
                lang_data_logger_pkg.log_error(
                    'Can only update the description of drilldowns in ' || 
                    ' PENDING_REVIEW. Current status:' || v_status
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_parameters_code
            );
            END IF;
            
        EXCEPTION 
            WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
            'No description found with id ' || p_drilldown_description_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END;

        v_normalized_description := LOWER(TRIM(p_new_description));
        -- Generate MD5 for the new drilldown description
        v_description_md5 := RAWTOHEX(
            DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(v_normalized_description),
                DBMS_CRYPTO.HASH_MD5
            )
        );

        -- Check if MD5 already exists in the table
        SELECT count(1) INTO v_description_md5_check 
        FROM langdata$drilldowndescriptions 
        WHERE ddd_md5 = v_description_md5;

        IF v_description_md5_check > 0 THEN
            lang_data_logger_pkg.log_error(
                'Description with this MD5 already exists'
            );  
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        v_ner_augmented_desc := p_new_description;

        IF v_ner_augmentation_enabled THEN
            v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                            p_new_description
                        );
            IF v_entities IS NOT NULL THEN
                v_ner_augmented_desc := 
                    lang_data_utils_pkg.augment_text_with_ner_entities(
                        p_new_description, v_entities
                    );
            END IF;
        END IF;

        -- Get Drilldown Match Document from drilldown id
        SELECT match_document, domain_id INTO v_match_document, v_domain_id
        FROM langdata$drilldowndocuments 
        WHERE id = v_drilldown_id;
        
        lang_data_utils_pkg.augment_text(
            p_text              =>  v_ner_augmented_desc,
            p_domain_id         =>  v_domain_id,
            p_augmented_text    =>  v_enhanced_description,
            p_augmented_tokens  =>  v_augmented_tokens
        );

        -- Expand Description to include filter descriptions
        v_enhanced_description   := lang_data_utils_pkg.expand_text(
            p_match_document => v_match_document,
            p_original_text  => v_enhanced_description,
            p_text_type      => 'drilldown'
        );

        lang_data_logger_pkg.log_info(
            'v_enhanced_description = ' || v_enhanced_description
        );

        UPDATE langdata$drilldowndescriptions SET text = p_new_description,
            ddd_vector = lang_data_utils_pkg.get_embedding(
                v_enhanced_description
            ),
            ddd_md5 = v_description_md5,
            enhanced_text = v_enhanced_description,
            augmented_tokens = v_augmented_tokens
        WHERE id = p_drilldown_description_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_invalid_parameters_code,
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_resource_not_found
            ) THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for drilldown description with ID: '
                    || p_drilldown_description_id || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
    END update_drilldown_description;

    PROCEDURE create_drilldown(
        p_title               IN VARCHAR2,
        p_report_id           IN VARCHAR2,
        p_match_document      IN JSON,
        p_description_text    IN VARCHAR2,
        p_description_status  IN VARCHAR2,
        p_drilldown_status    IN VARCHAR2,
        p_sample_queries      IN SYS.ODCIVARCHAR2LIST,
        p_domain              IN VARCHAR2 DEFAULT NULL,
        p_drilldown_id        OUT VARCHAR2,
        p_description_id      OUT VARCHAR2
    )
    IS
        v_drilldown_id         VARCHAR2(36);
        v_description_id       VARCHAR2(36);
        v_domain_id            VARCHAR2(36);
        v_ner_augmented_desc   VARCHAR2(4000);
        v_enhanced_description VARCHAR2(4000);
        -- MD5 hash of the normalized query
        v_description_md5      VARCHAR2(32);
        v_duplicate_check      INTEGER;
        v_sql_text             VARCHAR2(2000);
        v_normalized_text      VARCHAR2(4000);
        v_augmented_tokens     JSON;
	    v_dummy		       INTEGER;
        v_ner_augmentation_enabled  BOOLEAN;
        v_entities                  JSON;
        v_sample_query_ids     SYS.ODCIVARCHAR2LIST;
    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        IF NOT lang_data_utils_pkg.check_report_exists(p_report_id) THEN
            lang_data_logger_pkg.log_error('Report id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        EXECUTE IMMEDIATE 
        'SELECT COUNT(1) FROM langdata$drilldowndocuments
        WHERE title = :title and report_id = :report_id' 
        INTO v_dummy USING p_title, p_report_id;
        IF v_dummy > 0 THEN
            lang_data_logger_pkg.log_error(
                'Drilldown with the same title already exists.'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_already_exists
            );
        END IF;

        v_drilldown_id := lang_data_utils_pkg.generate_id();
        v_description_id := lang_data_utils_pkg.generate_id();

        -- Validate title is not NULL
        IF  p_title IS NULL THEN
            lang_data_logger_pkg.log_error(
                'Drilldown title is mandatory'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Validate title length is less than the specified limit
        IF  LENGTH(p_title) > 255 THEN
            lang_data_logger_pkg.log_error(
                'Drilldown title exceeds max title length'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_max_text_length_exceeded
            );
        END IF;

        -- Validate drilldown_status is not null and among required values
        IF p_drilldown_status IS NULL OR p_drilldown_status NOT IN ( 
            'Pending Review',
            'Approved',
            'Rejected',
            'Published',
            'Inactive',
            'Archived' 
        ) THEN
            lang_data_logger_pkg.log_error(
                'Invalid Drilldown Status'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Validate p_description_status is not null and among required values
        IF p_description_status IS NULL OR p_description_status NOT IN (
            'Pending Review',
            'Approved',
            'Rejected',
            'Published',
            'Inactive',
            'Archived',
            'Pending Regression'
        ) THEN
            lang_data_logger_pkg.log_error(
                'Invalid Drilldown description status'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- If p_domain is null inherit the domain value from the parent report
        -- else use p_domain
        IF p_domain IS NULL THEN
            lang_data_logger_pkg.log_debug('Using domain id from report');
            SELECT domain_id INTO v_domain_id
            FROM langdata$reports
            WHERE id = p_report_id;
            lang_data_logger_pkg.log_debug('p_domain_id is ' || v_domain_id);
        ELSE
            v_domain_id := lang_data_utils_pkg.get_or_create_domain(
                p_domain_name => p_domain
            );
            lang_data_logger_pkg.log_debug('p_domain_id is ' || v_domain_id);
        END IF;
        
        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        v_ner_augmented_desc := p_description_text;

        IF v_ner_augmentation_enabled THEN
            v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                            p_description_text
                        );
            IF v_entities IS NOT NULL THEN
                v_ner_augmented_desc := 
                    lang_data_utils_pkg.augment_text_with_ner_entities(
                        p_description_text, v_entities
                    );
            END IF;
        END IF;

        lang_data_utils_pkg.augment_text(
            p_text              =>  v_ner_augmented_desc,
            p_domain_id         =>  v_domain_id,
            p_augmented_text    =>  v_enhanced_description,
            p_augmented_tokens  =>  v_augmented_tokens
        );

        -- Expand Description to include filter descriptions
        v_enhanced_description  :=  lang_data_utils_pkg.expand_text(
            p_match_document => p_match_document,
            p_original_text  => v_enhanced_description,
            p_text_type      => 'drilldown'
        );

        lang_data_logger_pkg.log_info(
            'v_enhanced_description = ' || v_enhanced_description
        );

        v_normalized_text := LOWER(TRIM(p_description_text));
        -- Generate MD5 for the drilldown description
        v_description_md5 := RAWTOHEX(
            DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(v_normalized_text),
                DBMS_CRYPTO.HASH_MD5
            )
        );

        -- Check if description already exists
        SELECT COUNT(1)
        INTO v_duplicate_check
        FROM langdata$drilldowndescriptions
        WHERE ddd_md5 = v_description_md5;

        IF v_duplicate_check > 0 THEN
            lang_data_logger_pkg.log_error(
                'Drilldown description already exists'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_already_exists
            );
        END IF;

        -- Insert Drilldown Document
        INSERT INTO langdata$drilldowndocuments (
            id, 
            title, 
            report_id, 
            match_document, 
            status,
            domain_id
        ) VALUES (
            v_drilldown_id, 
            p_title, 
            p_report_id, 
            p_match_document, 
            p_drilldown_status,
            v_domain_id
        );

        -- Insert Drilldown Description
        EXECUTE IMMEDIATE 
            'INSERT INTO langdata$drilldowndescriptions (
                id, 
                drilldown_id, 
                text, 
                status, 
                ddd_vector, 
                ddd_md5, 
                enhanced_text,
                augmented_tokens
            ) VALUES (
                :description_id, 
                :drilldown_id, 
                :description_text, 
                :description_status, 
                lang_data_utils_pkg.get_embedding(:enhanced_description),
                :description_md5, 
                :enhanced_description,
                :augmented_tokens
            )'
        USING 
            v_description_id, 
            v_drilldown_id, 
            p_description_text, 
            p_description_status, 
            v_enhanced_description, 
            v_description_md5, 
            v_enhanced_description, 
            v_augmented_tokens;

        -- Disable adding custom entity_types.
        IF lang_data_config_pkg.g_custom_entities = TRUE THEN
            lang_data_named_entities_pkg.add_named_entities_to_document(
                p_match_document
            );
        END IF;
        
        lang_data_utils_pkg.create_value_vector_job(
            p_match_document    => p_match_document,
            p_document_id       => v_drilldown_id,
            p_domain_id         => v_domain_id
        );

        IF p_description_status = 'Pending Regression' THEN
            lang_data_utils_pkg.create_or_replace_job(
                p_job_name => 'JOB_LANGDATA_'||
                    substr(v_description_md5,1,5)||'_DRILLDOWN_D_REGRESSION',
                p_job_action => 
                    'BEGIN' ||
        'lang_data_drilldowns_pkg.calculate_drilldown_regression( '
                    || 'p_new_description_id => '''
                    || v_description_id || '''); END;',
                p_job_class       => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                p_comments        => 
                    'Regression calculation for drilldown description',
                p_priority        => 1,
                p_restart_on_fail => FALSE,
                p_restart_on_rec  => TRUE
            );
        END IF;

        IF p_sample_queries IS NOT NULL THEN
            lang_data_sample_queries_pkg.add_drilldown_sample_queries(
                p_drilldown_id   => v_drilldown_id,
                p_sample_queries => p_sample_queries,
                p_ids            => v_sample_query_ids
            );
        END IF;

        p_drilldown_id := v_drilldown_id;
        p_description_id := v_description_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN(  lang_data_errors_pkg.c_invalid_match_document,
                            lang_data_errors_pkg.c_invalid_parameters_code,
                            lang_data_errors_pkg.c_resource_already_exists,
                            lang_data_errors_pkg.c_max_text_length_exceeded,
                            lang_data_errors_pkg.c_unauthorized_code)  THEN
                RAISE;
            END IF;

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || v_drilldown_id || 
                '. Error: ' || SQLERRM
            );
            RAISE;
    END create_drilldown;

    PROCEDURE add_drilldown_description(
        p_drilldown_id          IN VARCHAR2,
        p_text                  IN VARCHAR2,
        p_status                IN VARCHAR2,
        p_description_id        OUT VARCHAR2
    ) IS
        v_ner_augmented_desc        VARCHAR2(4000);
        v_enhanced_description      VARCHAR2(4000);
        v_description_md5           VARCHAR2(32);
        v_drilldown_id              VARCHAR2(36) := NULL;
        v_match_document            JSON;
        v_description_md5_check     NUMBER := 0;
        v_description_id            VARCHAR2(36) := NULL;
        v_normalized_text           VARCHAR2(4000);
        v_augmented_tokens          JSON;
        v_ner_augmentation_enabled  BOOLEAN;
        v_entities                  JSON;
        v_domain_id                 VARCHAR2(36);
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure to add description to drilldown of ID: ' 
            || p_drilldown_id
        );
        
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Check if description length is under limit
        IF LENGTH(p_text) > 2000 THEN
            lang_data_logger_pkg.log_error(
                'Description length exceeded the limit'
            );  
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        v_description_id := lang_data_utils_pkg.generate_id();

        v_normalized_text := LOWER(TRIM(p_text));
        -- Generate MD5 for the new drilldown description
        v_description_md5 := RAWTOHEX(
            DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(v_normalized_text),
                DBMS_CRYPTO.HASH_MD5
            )
        );

        -- Check if MD5 already exists in the table
        SELECT count(1) INTO v_description_md5_check 
        FROM langdata$drilldowndescriptions 
        WHERE ddd_md5 = v_description_md5;

        IF v_description_md5_check > 0 THEN
            lang_data_logger_pkg.log_error(
                'Description with this MD5 already exists'
            );  
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Get Drilldown Match Document from drilldown id
        SELECT id, match_document, domain_id
        INTO v_drilldown_id, v_match_document, v_domain_id
        FROM langdata$drilldowndocuments 
        WHERE id = p_drilldown_id;

        IF v_drilldown_id IS NULL THEN
            lang_data_logger_pkg.log_error(
                'No drilldown exists with the id:' || p_drilldown_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        v_ner_augmented_desc := p_text;

        IF v_ner_augmentation_enabled THEN
            v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                            p_text
                        );
            IF v_entities IS NOT NULL THEN
                v_ner_augmented_desc := 
                    lang_data_utils_pkg.augment_text_with_ner_entities(
                        p_text, v_entities
                    );
            END IF;
        END IF;
        
        lang_data_utils_pkg.augment_text(
            p_text              =>  v_ner_augmented_desc,
            p_domain_id         =>  v_domain_id,
            p_augmented_text    =>  v_enhanced_description,
            p_augmented_tokens  =>  v_augmented_tokens
        );

        -- Expand Description to include filter descriptions
        v_enhanced_description   := lang_data_utils_pkg.expand_text(
            p_match_document        => v_match_document,
            p_original_text         => v_enhanced_description,
            p_text_type             => 'drilldown'
        );

        INSERT INTO langdata$drilldowndescriptions (
            id, 
            drilldown_id, 
            text, 
            status, 
            created_at, 
            updated_at, 
            ddd_vector, 
            ddd_md5, 
            enhanced_text,
            augmented_tokens
        ) 
        VALUES (
            v_description_id, 
            p_drilldown_id, 
            p_text, 
            p_status, 
            SYSTIMESTAMP, 
            SYSTIMESTAMP, 
            lang_data_utils_pkg.get_embedding(v_enhanced_description), 
            v_description_md5, 
            v_enhanced_description,
            v_augmented_tokens
        );

        IF p_status = 'Pending Regression' THEN
            lang_data_utils_pkg.create_or_replace_job(
                p_job_name => 'JOB_LANGDATA_'||
                    substr(v_description_md5,1,5)||'_DRILLDOWN_D_REGRESSION',
                p_job_action => 
                    'BEGIN' ||
        'lang_data_drilldowns_pkg.calculate_drilldown_regression( '
                    || 'p_new_description_id => '''
                    || v_description_id || '''); END;',
                p_job_class       => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                p_comments        => 
                    'Regression calculation for drilldown description',
                p_priority        => 1,
                p_restart_on_fail => FALSE,
                p_restart_on_rec  => TRUE
            );
        END IF;

        p_description_id := v_description_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_invalid_parameters_code,
                lang_data_errors_pkg.c_unauthorized_code
            ) THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for drilldown description ' ||
                    'with ID: ' || p_description_id || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;

    END add_drilldown_description;

    PROCEDURE execute_drilldown_sql(
        p_drilldown_id   IN VARCHAR2,
        p_filter_values  IN JSON,
        p_columns        OUT SYS.ODCIVARCHAR2LIST,
        p_data           OUT CLOB
    )
    IS
        v_drilldown_sql    VARCHAR2(4000);
        v_cur              INTEGER;
        v_dummy            INTEGER;
        v_num_cols         INTEGER;
        v_desc_tab         DBMS_SQL.DESC_TAB;
        v_json             JSON_OBJECT_T;
        v_match_document   JSON;
        v_keys             JSON_KEY_LIST;
        v_key              VARCHAR2(4000);
        v_filter_obj       JSON_OBJECT_T;
        v_value            VARCHAR2(4000);
        v_col_value        VARCHAR2(4000);
        v_row_json         VARCHAR2(32767); -- To hold individual row data
    BEGIN
        SELECT match_document 
        INTO v_match_document
        FROM langdata$drilldowndocuments 
        WHERE id = p_drilldown_id;

        -- Extract the SQL
        v_drilldown_sql := JSON_VALUE(
            v_match_document, '$.sql' RETURNING VARCHAR2(4000)
        );
        lang_data_logger_pkg.log_info(
            'SQL Query to be executed: ' || v_drilldown_sql
        );

        -- Open the cursor for dynamic SQL execution
        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_drilldown_sql, DBMS_SQL.NATIVE);

        -- Dynamically extract all key-value pairs using JSON_TABLE
        v_json := JSON_OBJECT_T.parse(json_serialize(p_filter_values));
        v_keys := v_json.get_keys;
        FOR i IN v_keys.FIRST..v_keys.LAST LOOP
            v_key := v_keys(i);

            v_filter_obj := v_json.get_object(v_key);

            v_value := v_filter_obj.get_string('value');

            lang_data_logger_pkg.log_info(
                'Filter: ' || v_key || ' | Value: ' || v_value
            );
            DBMS_SQL.BIND_VARIABLE(v_cur, ':' || v_key, v_value);
        END LOOP;
        
        -- Describe the columns in the result set
        DBMS_SQL.DESCRIBE_COLUMNS(v_cur, v_num_cols, v_desc_tab);

        -- Add columns to the cursor
        FOR i IN 1 .. v_num_cols LOOP
            DBMS_SQL.DEFINE_COLUMN(v_cur, i, v_col_value, 4000);
        END LOOP;

        DBMS_LOB.CREATETEMPORARY(p_data, TRUE);

        -- Execute the SQL
        v_dummy := DBMS_SQL.EXECUTE(v_cur);

        -- Prepare output for columns
        p_columns := SYS.ODCIVARCHAR2LIST();

        FOR j IN 1 .. v_num_cols LOOP
            p_columns.EXTEND;
            p_columns(j) := v_desc_tab(j).col_name;
            lang_data_logger_pkg.log_info('p_columns(j) = ' || p_columns(j));
        END LOOP;

        lang_data_logger_pkg.log_info('v_num_cols = ' || v_num_cols);

        -- Fetch the rows and build the JSON string
        WHILE DBMS_SQL.FETCH_ROWS(v_cur) > 0 LOOP
            v_row_json := '{'; -- Start the JSON object
            FOR k IN 1 .. v_num_cols LOOP
                DBMS_SQL.COLUMN_VALUE(v_cur, k, v_col_value);
                -- Add each column value to the JSON object
                v_row_json := v_row_json || '"' || p_columns(k) || '": ' || '"' || v_col_value || '"';
                IF k < v_num_cols THEN
                    v_row_json := v_row_json || ', '; -- Add comma if not the last column
                END IF;
            END LOOP;
            v_row_json := v_row_json || '}'; -- End the JSON object
            -- lang_data_logger_pkg.log_info('v_row_json = ' || v_row_json);

            -- Append the row JSON object to the CLOB
            DBMS_LOB.WRITEAPPEND(p_data, LENGTH(v_row_json), v_row_json);
        END LOOP;

        lang_data_logger_pkg.log_info('Current CLOB length: ' || DBMS_LOB.GETLENGTH(p_data));

        -- Close the cursor
        DBMS_SQL.CLOSE_CURSOR(v_cur);
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No drilldown found for ID: '|| p_drilldown_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred. Error: ' || SQLERRM
            );
            RAISE;
    END execute_drilldown_sql;    
    
    -- Note: used in just get_report_matches_from_doc, 
    --       which is not used in any resolver.
    PROCEDURE get_drilldown_description (
        p_description_id  IN VARCHAR2,
        p_text OUT VARCHAR2,
        p_version OUT NUMBER,
        p_status OUT VARCHAR2,
        p_drilldown_id OUT VARCHAR2,
        p_enhanced_text OUT VARCHAR2
    )
    IS
    BEGIN
        -- If the provided ID is NULL, raise an error or handle it accordingly.
        IF p_description_id IS NULL THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT text, version, status, drilldown_id, enhanced_text
        INTO p_text, p_version, p_status, p_drilldown_id, p_enhanced_text
        FROM langdata$drilldowndescriptions
        WHERE id = p_description_id
        AND ROWNUM = 1;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
                lang_data_logger_pkg.log_error(
                    'Drilldown Description with the given description ID ' || 
                    ' not found.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_resource_not_found
                );
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred. Error: ' || SQLERRM
            );
            RAISE;
    END get_drilldown_description;

    PROCEDURE get_drilldown_descriptions (
        p_drilldown_id  IN VARCHAR2,
        p_descriptions  OUT SYS_REFCURSOR
    )
    IS
    BEGIN
        OPEN p_descriptions FOR
            SELECT 
                id, 
                text, 
                version, 
                status, 
                enhanced_text 
            FROM 
                langdata$drilldowndescriptions 
            WHERE 
                drilldown_id = p_drilldown_id 
            ORDER BY 
                created_at DESC;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No drilldown found for ID: '|| p_drilldown_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || p_drilldown_id || 
                '. Error: ' || SQLERRM
            );
            RAISE;
    END get_drilldown_descriptions;

    PROCEDURE calculate_drilldown_regression(
        p_new_description_id  IN VARCHAR2 DEFAULT NULL,
        p_new_sample_query_id IN VARCHAR2 DEFAULT NULL
    ) IS
        -- Local variables
        v_report_id                         VARCHAR2(36);
        v_query_text                        VARCHAR2(4000);
        v_new_distance                      NUMBER;
        v_good_search_records               SYS_REFCURSOR;
        v_search_results                    SYS_REFCURSOR;
        v_new_text                          VARCHAR2(4000);
        v_status                            VARCHAR2(20);
        v_new_drilldown_id                  VARCHAR2(36);
        v_query_vector                      VECTOR(*,*);
        v_regression_count                  NUMBER;
        v_regression_accepted_record_count  NUMBER;
        v_record_id                         VARCHAR2(36);
        v_regression_json_object            JSON_OBJECT_T := JSON_OBJECT_T();
        v_regressing_records                JSON_ARRAY_T := JSON_ARRAY_T();
        v_regression_record_object          JSON_OBJECT_T;
        v_regression_json                   JSON;
        v_is_description                    BOOLEAN := TRUE;
        v_domain_id                         VARCHAR2(36);

        -- top two closest matching drilldowns 
        v_drilldown_id1                     VARCHAR2(36) := '###';
        v_distance1                         NUMBER;
        v_id1                               VARCHAR2(36);
        v_enhanced_text1                    VARCHAR2(4000);
        v_text1                             VARCHAR2(4000);
        v_title1                            VARCHAR2(255);
        v_match_type1                       VARCHAR2(100);
        v_type1                             VARCHAR2(10);
        v_rank1                             NUMBER;
        v_expected_report_id                VARCHAR2(36);
        v_expected_drilldown_id             VARCHAR2(36);
        v_identifier_report   VARCHAR2(36) := '###';
        v_desc_vector           VECTOR(*,*);


        --create TYPE for BULK COLLECT
        TYPE t_match_rec IS RECORD (
            record_id            langdata$searchrecords.id%TYPE,
            query_text           langdata$searchrecords.query_text%TYPE,
            expected_drilldown_id   
                            langdata$searchrecords.expected_report_id%TYPE,
            new_distance         NUMBER,
            --
            drilldown_id1           langdata$reports.id%TYPE,
            description_id1      langdata$reportdescriptions.id%TYPE,
            old_distance1        NUMBER,
            match_type1          VARCHAR2(32)
        );
        TYPE t_match_tab IS TABLE OF t_match_rec INDEX BY PLS_INTEGER;
        l_matches t_match_tab;

        -- batch size for Bulk collect
        c_batch_size CONSTANT PLS_INTEGER := 1000;

        -- cursor for extracting queries candidate for regression
        CURSOR c_matches 
                (p_desc_vec IN langdata$searchrecords.query_vector%TYPE,
                p_report_id IN langdata$searchrecords.id%TYPE,
                p_domain_id IN langdata$searchrecords.domain_id%TYPE) IS
            SELECT /*+ PARALLEL */ 
                sr.id AS record_id,
                sr.query_text AS query_text,
                sr.expected_drilldown_id AS expected_drilldown_id,
                VECTOR_DISTANCE(sr.query_vector, p_desc_vec) AS new_distance,
                t.drilldown_id AS drilldown_id1,
                t.id AS description_id1,
                t.distance AS old_distance1,
                t.match_type AS match_type1
            FROM langdata$searchrecords sr
            LEFT JOIN LATERAL (
                SELECT * FROM (
                    SELECT 
                        drilldown_id, id, distance, match_type,
                        ROW_NUMBER() OVER (
                            PARTITION BY drilldown_id ORDER BY distance ASC
                        ) AS rn
                    FROM (
                        -- Drilldown Description
                        SELECT * FROM (
                            SELECT
                                ddd.drilldown_id,
                                ddd.id,
                                VECTOR_DISTANCE(ddd.ddd_vector, 
                                                sr.query_vector) AS distance,
                                'Drilldown Description' AS match_type
                            FROM langdata$drilldowndescriptions ddd
                            JOIN langdata$drilldowndocuments d2 ON 
                                  ddd.drilldown_id = d2.id
                            WHERE ddd.status = 'Published' AND 
                                  d2.status = 'Published' AND
                                  d2.report_id = p_report_id
                            ORDER BY VECTOR_DISTANCE(
                                ddd.ddd_vector, sr.query_vector)
                            FETCH FIRST 1 ROWS ONLY
                        )

                        UNION ALL
                        -- Sample Query
                        SELECT * FROM (
                            SELECT
                                sq.drilldown_id,
                                sq.id,
                                VECTOR_DISTANCE(sq.query_vector,
                                                sr.query_vector) AS distance,
                                'Report Sample Query' AS match_type
                            FROM langdata$samplequeries sq
                            JOIN langdata$drilldowndocuments d3 ON 
                                  sq.drilldown_id = d3.id
                            WHERE d3.status = 'Published' AND 
                                  sq.status  = 'Published' AND
                                  d3.report_id = p_report_id
                            ORDER BY VECTOR_DISTANCE(
                                sq.query_vector, sr.query_vector)
                            FETCH FIRST 1 ROWS ONLY
                        )
                        
                        UNION ALL
                        -- Search Record
                        SELECT * FROM (
                            SELECT
                                sr2.expected_drilldown_id AS drilldown_id,
                                sr2.id,
                                VECTOR_DISTANCE(sr2.query_vector, 
                                                sr.query_vector) AS distance,
                                'Search Record' AS match_type
                            FROM langdata$searchrecords sr2
                            JOIN langdata$drilldowndocuments d4 
                                    ON sr2.expected_drilldown_id = d4.id
                            WHERE d4.status = 'Published' AND 
                                  sr2.id <> sr.id AND
                                  d4.report_id = p_report_id
                            ORDER BY VECTOR_DISTANCE(
                                sr2.query_vector, sr.query_vector)
                            FETCH FIRST 1 ROWS ONLY
                        )
                    )
                )
                WHERE rn = 1
                ORDER BY distance
                FETCH FIRST 1 ROWS ONLY
            ) t ON 1 = 1
            WHERE p_domain_id IS NULL OR
                p_domain_id = sr.domain_id;


    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        IF p_new_description_id IS NULL 
            AND p_new_sample_query_id IS NULL THEN
            lang_data_logger_pkg.log_error(
                'Exactly one of p_new_description_id and '||
                'p_new_sample_query_id must be non-null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_new_description_id IS NOT NULL 
            AND p_new_sample_query_id IS NOT NULL THEN
            lang_data_logger_pkg.log_error(
                'Exactly one of p_new_description_id and '||
                'p_new_sample_query_id must be non-null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_new_description_id IS NOT NULL THEN
            v_is_description := TRUE;
        ELSE
            v_is_description := FALSE;
        END IF;



        lang_data_logger_pkg.log_debug(
            'Starting Regression calculation for drilldown.');

        IF v_is_description THEN
            SELECT ddd.enhanced_text, ddd.status, 
                   dd.report_id, ddd.drilldown_id
            INTO v_new_text, v_status, 
                v_report_id, v_new_drilldown_id
            FROM langdata$drilldowndescriptions ddd
            JOIN langdata$drilldowndocuments dd
            ON dd.id = ddd.drilldown_id
            WHERE ddd.id = p_new_description_id;
        ELSE
            SELECT enhanced_query_text, status, report_id, drilldown_id
            INTO v_new_text, v_status, v_report_id, v_new_drilldown_id 
            FROM langdata$samplequeries
            WHERE id = p_new_sample_query_id;
        END IF;
        SELECT domain_id INTO v_domain_id FROM langdata$drilldowndocuments
        WHERE id = v_new_drilldown_id;

        IF NOT v_status = 'Pending Regression' THEN
            IF v_is_description THEN
                lang_data_logger_pkg.log_fatal(
                    'Drilldown Description must be in Pending Regression '
                    ||'state to run regression'
                );
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'Sample Query must be in Pending Regression state'||
                    ' to run regression'
                );
            END IF;
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        v_desc_vector := lang_data_utils_pkg.get_embedding(v_new_text);

        v_regression_count := 0;
        v_regression_accepted_record_count := 0;

        OPEN c_matches(v_desc_vector, v_report_id, v_domain_id);

        -- Loop through the cursor
        LOOP

            -- bulk collect from sql query result
            FETCH c_matches BULK COLLECT INTO l_matches LIMIT c_batch_size;
            EXIT WHEN l_matches.COUNT = 0;  -- no more rows

            for i in 1 .. l_matches.COUNT LOOP

                -- extract record and report info
                v_record_id          := l_matches(i).record_id;
                v_query_text         := l_matches(i).query_text;
                v_expected_drilldown_id := l_matches(i).expected_drilldown_id;
                v_new_distance       := l_matches(i).new_distance;

                v_id1        := l_matches(i).description_id1;
                v_drilldown_id1 := l_matches(i).drilldown_id1;
                v_distance1  := l_matches(i).old_distance1;
                v_match_type1:= l_matches(i).match_type1;

                IF (v_drilldown_id1 = v_identifier_report) THEN
                    CONTINUE;
                END IF;


                v_regression_record_object := JSON_OBJECT_T();
                v_regression_record_object.put('record_id', v_record_id);
                v_regression_record_object.put(
                    'previous_best_drilldown_id', v_drilldown_id1);
                v_regression_record_object.put(
                    'previous_best_description_id', v_id1);
                v_regression_record_object.put(
                    'previous_best_description_type', v_match_type1);

                IF  v_new_drilldown_id <> v_drilldown_id1 AND 
                    v_new_distance < v_distance1 THEN
                    
                    v_regression_record_object.put(
                        'new_best_drilldown_id', v_new_drilldown_id);
                    v_regression_record_object.put(
                        'new_best_description_id', p_new_description_id);
                    v_regression_record_object.put(
                        'new_best_description_type', 'description'
                    );
                    
                    IF v_expected_drilldown_id IS NULL THEN
                        v_regression_count := v_regression_count + 1;
                        v_regression_record_object.put(
                            'record_has_accepted_report', 'NONE');
                    ELSE  
                        v_regression_accepted_record_count := 
                            v_regression_accepted_record_count + 1;
                        v_regression_record_object.put(
                         'record_has_accepted_report', v_expected_drilldown_id);
                    END IF;   
                    v_regressing_records.append(v_regression_record_object);
                END IF;
            END LOOP;
        END LOOP;
        
        v_regression_json_object.put('regression_count', v_regression_count);
        v_regression_json_object.put(
            'regression_accepted_record_count', 
            v_regression_accepted_record_count
        );
        v_regression_json_object.put(
            'regressing_records',
            v_regressing_records
        );
        IF v_regression_accepted_record_count <> 0 
        OR v_regression_count <> 0 THEN
            v_regression_json := JSON(v_regression_json_object.to_clob());
            -- Insert regression_json
            IF v_is_description THEN
                UPDATE langdata$drilldowndescriptions 
                SET regression_json = v_regression_json,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_description_id;
            ELSE
                UPDATE langdata$samplequeries
                SET regression_json = v_regression_json,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_sample_query_id;
            END IF;
        ElSE
            lang_data_logger_pkg.log_debug(
                'No regression found setting regression_json to null'
            );
            IF v_is_description THEN
                UPDATE langdata$drilldowndescriptions 
                SET regression_json = NULL,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_description_id;
            ELSE
                UPDATE langdata$samplequeries
                SET regression_json = NULL,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_sample_query_id;
            END IF;
        END IF;

        IF v_is_description THEN
            SELECT status
            INTO v_status
            FROM langdata$drilldowndescriptions
            WHERE id = p_new_description_id;
        ELSE
            SELECT status
            INTO v_status
            FROM langdata$samplequeries
            WHERE id = p_new_sample_query_id;
        END IF;

        IF v_status = 'Pending Regression' THEN
            IF v_is_description THEN
                UPDATE langdata$drilldowndescriptions 
                SET status = 'Pending Review'
                WHERE id = p_new_description_id;
            ELSE
                UPDATE langdata$samplequeries
                SET status = 'Pending Review'
                WHERE id = p_new_sample_query_id;
            END IF;
        END IF;

     EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred while calculating  ' || 
                'drilldown regression. Error: ' || SQLERRM
                );
            RAISE;
 
    END calculate_drilldown_regression;

    PROCEDURE replace_drilldown_filter_enumerable_set (
        p_drilldown_id          IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_new_enumerable_set    IN  JSON_ARRAY_T
    )
    IS
        v_match_document      JSON;
        v_match_document_obj  JSON_OBJECT_T;
        v_filters             JSON_ARRAY_T;
        v_filter_obj          JSON_OBJECT_T;
        v_filter_name         VARCHAR2(255);
        v_filter_description  VARCHAR2(2000);
        v_updated             BOOLEAN := FALSE;
        v_job_name            VARCHAR2(255);
        v_md5                 VARCHAR2(32);
        v_domain_id           VARCHAR2(36);
    BEGIN
        IF NOT lang_data_utils_pkg.check_drilldown_exists(p_drilldown_id) THEN
            lang_data_logger_pkg.log_error('Drilldown id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT match_document, domain_id INTO v_match_document, v_domain_id
        FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;

        v_match_document_obj := JSON_OBJECT_T.PARSE(
            JSON_SERIALIZE(v_match_document)
        );

        v_filters := v_match_document_obj.GET_ARRAY('filters');

        FOR indx IN 0 .. v_filters.GET_SIZE - 1 LOOP
            v_filter_obj := JSON_OBJECT_T(v_filters.GET(TO_NUMBER(indx)));
            v_filter_name := v_filter_obj.GET_STRING('filter_name');
            IF UPPER(v_filter_name) = UPPER(p_filter_name) THEN
                IF NOT v_filter_obj.HAS('enumerable_set') THEN
                    lang_data_logger_pkg.log_error(
                        'Filter ' || p_filter_name ||
                        'does not have enumerable set'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_parameters_code
                    );
                END IF;
                -- Update the 'enumerable_set' field
                v_filter_obj.PUT('enumerable_set', p_new_enumerable_set);
                v_updated := TRUE;
                v_filter_description := v_filter_obj.GET_STRING('description');
                EXIT;
            END IF;

        END LOOP;

        IF v_updated THEN
            v_match_document := JSON(v_match_document_obj.TO_STRING());

            UPDATE langdata$drilldowndocuments 
            SET match_document = v_match_document
            WHERE id = p_drilldown_id;

            v_md5 := RAWTOHEX(
                DBMS_CRYPTO.HASH(
                    UTL_RAW.CAST_TO_RAW(p_drilldown_id),
                    DBMS_CRYPTO.HASH_MD5
                )
            );

            v_job_name := 'JOB_LANGDATA_' || UPPER(p_filter_name) || '_' ||
                          v_md5 || '_VALUE_VECTORS';
            BEGIN
                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name        => v_job_name,
                    p_job_action      =>
                        'DECLARE ' ||
                        '  v_enum_set JSON_ARRAY_T := JSON_ARRAY_T.PARSE(''' || 
                        p_new_enumerable_set.TO_STRING() || '''); ' ||
                        'BEGIN ' ||
                        '  lang_data_utils_pkg.create_value_vector_from_enumerable_set(' ||
                        '    v_enum_set, ' ||
                        '    ''' || p_drilldown_id || ''', ' ||
                        '    ''' || p_filter_name || ''', ' ||
                        '    ''' || 
                        REPLACE(v_filter_description, '''', '''''') || ''', ' ||
                        '    TRUE, ' ||
                        '    ''' || NVL(v_domain_id, '') || '''' ||
                        '  ); ' ||
                        'END;',
                    p_job_class    => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments        => 'Create value vector table from enumerable set',
                    p_priority        => 1,
                    p_restart_on_fail => TRUE,
                    p_restart_on_rec  => TRUE
                );
                lang_data_logger_pkg.log_info(
                    'Submitted job ' || v_job_name ||
                    ' to recreate value vector table for filter ' ||
                    p_filter_name
                );

            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -27477 THEN
                        lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                    ' already exists');
                    ELSE
                        lang_data_logger_pkg.log_fatal(
                            'An unknown error occurred. Error: ' || SQLERRM
                        );
                        RAISE;
                    END IF;
            END;
        ELSE
            lang_data_logger_pkg.log_error(
                'Filter ' || p_filter_name ||
                'does not exist'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
    END replace_drilldown_filter_enumerable_set;

    PROCEDURE add_into_drilldown_filter_enumerable_set (
        p_drilldown_id             IN  VARCHAR2,
        p_filter_name              IN  VARCHAR2,
        p_enumerable_set_to_add    IN  JSON_ARRAY_T
    )
    IS
        v_match_document      JSON;
        v_match_document_obj  JSON_OBJECT_T;
        v_filters             JSON_ARRAY_T;
        v_filter_obj          JSON_OBJECT_T;
        v_filter_name         VARCHAR2(255);
        v_filter_description  VARCHAR2(2000);
        v_old_enumerable_set  JSON_ARRAY_T;
        v_merged_enumerable_set       JSON_ARRAY_T := JSON_ARRAY_T();
        v_new_enumerable_set_json     JSON;
        v_old_enumerable_set_json     JSON;
        v_updated             BOOLEAN := FALSE;
        v_job_name            VARCHAR2(255);
        v_md5                 VARCHAR2(32);
        v_domain_id           VARCHAR2(36);
    BEGIN
        IF NOT lang_data_utils_pkg.check_drilldown_exists(p_drilldown_id) THEN
            lang_data_logger_pkg.log_error('Drilldown id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT match_document, domain_id INTO v_match_document, v_domain_id
        FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;

        v_match_document_obj := JSON_OBJECT_T.PARSE(
            JSON_SERIALIZE(v_match_document)
        );

        v_filters := v_match_document_obj.GET_ARRAY('filters');

        FOR indx IN 0 .. v_filters.GET_SIZE - 1 LOOP
            v_filter_obj := JSON_OBJECT_T(v_filters.GET(TO_NUMBER(indx)));
            v_filter_name := v_filter_obj.GET_STRING('filter_name');
            IF UPPER(v_filter_name) = UPPER(p_filter_name) THEN
                IF NOT v_filter_obj.HAS('enumerable_set') THEN
                    lang_data_logger_pkg.log_error(
                        'Filter ' || p_filter_name ||
                        'does not have enumerable set'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_parameters_code
                    );
                END IF;
                -- Update the 'enumerable_set' field
                v_old_enumerable_set := v_filter_obj.GET_ARRAY('enumerable_set');
                v_new_enumerable_set_json := JSON(p_enumerable_set_to_add.TO_STRING());
                v_old_enumerable_set_json := JSON(v_old_enumerable_set.TO_STRING());

                FOR rec IN (
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_new_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                    UNION
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_old_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                ) LOOP
                    v_merged_enumerable_set.APPEND(rec.value);
                END LOOP;

                v_filter_obj.PUT('enumerable_set', v_merged_enumerable_set);
                v_updated := TRUE;
                v_filter_description := v_filter_obj.GET_STRING('description');
                EXIT;
            END IF;

        END LOOP;

        IF v_updated THEN
            v_match_document := JSON(v_match_document_obj.TO_STRING());

            UPDATE langdata$drilldowndocuments 
            SET match_document = v_match_document
            WHERE id = p_drilldown_id;

            v_md5 := RAWTOHEX(
                DBMS_CRYPTO.HASH(
                    UTL_RAW.CAST_TO_RAW(p_drilldown_id),
                    DBMS_CRYPTO.HASH_MD5
                )
            );

            v_job_name := 'JOB_LANGDATA_' || UPPER(p_filter_name) || '_' ||
                          v_md5 || '_VALUE_VECTORS';
            BEGIN
                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name        => v_job_name,
                    p_job_action      =>
                        'DECLARE ' ||
                        '  v_enum_set JSON_ARRAY_T := JSON_ARRAY_T.PARSE(''' || 
                        v_merged_enumerable_set.TO_STRING() || '''); ' ||
                        'BEGIN ' ||
                        '  lang_data_utils_pkg.create_value_vector_from_enumerable_set(' ||
                        '    v_enum_set, ' ||
                        '    ''' || p_drilldown_id || ''', ' ||
                        '    ''' || p_filter_name || ''', ' ||
                        '    ''' || 
                        REPLACE(v_filter_description, '''', '''''') || ''', ' ||
                        '    TRUE, ' ||
                        '    ''' || NVL(v_domain_id, '') || '''' ||
                        '  ); ' ||
                        'END;',
                    p_job_class    => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments        => 'Create value vector table from enumerable set',
                    p_priority        => 1,
                    p_restart_on_fail => TRUE,
                    p_restart_on_rec  => TRUE
                );

                lang_data_logger_pkg.log_info(
                    'Submitted job ' || v_job_name ||
                    ' to recreate value vector table for filter ' ||
                    p_filter_name
                );

            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -27477 THEN
                        lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                    ' already exists');
                    ELSE
                        lang_data_logger_pkg.log_fatal(
                            'An unknown error occurred. Error: ' || SQLERRM
                        );
                        RAISE;
                    END IF;
            END;
        ELSE
            lang_data_logger_pkg.log_error(
                'Filter ' || p_filter_name ||
                'does not exist'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
    END add_into_drilldown_filter_enumerable_set;

    PROCEDURE remove_from_drilldown_filter_enumerable_set (
        p_drilldown_id                IN  VARCHAR2,
        p_filter_name                 IN  VARCHAR2,
        p_enumerable_set_to_remove    IN  JSON_ARRAY_T
    )
    IS
        v_match_document      JSON;
        v_match_document_obj  JSON_OBJECT_T;
        v_filters             JSON_ARRAY_T;
        v_filter_obj          JSON_OBJECT_T;
        v_filter_name         VARCHAR2(255);
        v_filter_description  VARCHAR2(2000);
        v_old_enumerable_set  JSON_ARRAY_T;
        v_updated_enumerable_set      JSON_ARRAY_T := JSON_ARRAY_T();
        v_new_enumerable_set_json     JSON;
        v_old_enumerable_set_json     JSON;
        v_updated             BOOLEAN := FALSE;
        v_job_name            VARCHAR2(255);
        v_md5                 VARCHAR2(32);
        v_domain_id           VARCHAR2(36);
    BEGIN
        IF NOT lang_data_utils_pkg.check_drilldown_exists(p_drilldown_id) THEN
            lang_data_logger_pkg.log_error('Drilldown id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT match_document, domain_id INTO v_match_document, v_domain_id
        FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;

        v_match_document_obj := JSON_OBJECT_T.PARSE(
            JSON_SERIALIZE(v_match_document)
        );

        v_filters := v_match_document_obj.GET_ARRAY('filters');

        FOR indx IN 0 .. v_filters.GET_SIZE - 1 LOOP
            v_filter_obj := JSON_OBJECT_T(v_filters.GET(TO_NUMBER(indx)));
            v_filter_name := v_filter_obj.GET_STRING('filter_name');
            IF UPPER(v_filter_name) = UPPER(p_filter_name) THEN
                IF NOT v_filter_obj.HAS('enumerable_set') THEN
                    lang_data_logger_pkg.log_error(
                        'Filter ' || p_filter_name ||
                        'does not have enumerable set'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_parameters_code
                    );
                END IF;
                -- Update the 'enumerable_set' field
                v_old_enumerable_set := v_filter_obj.GET_ARRAY('enumerable_set');
                v_new_enumerable_set_json := JSON(p_enumerable_set_to_remove.TO_STRING());
                v_old_enumerable_set_json := JSON(v_old_enumerable_set.TO_STRING());

                FOR rec IN (
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_old_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                    MINUS
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_new_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                ) LOOP
                    v_updated_enumerable_set.APPEND(rec.value);
                END LOOP;

                v_filter_obj.PUT('enumerable_set', v_updated_enumerable_set);
                v_updated := TRUE;
                v_filter_description := v_filter_obj.GET_STRING('description');
                EXIT;
            END IF;

        END LOOP;

        IF v_updated THEN
            v_match_document := JSON(v_match_document_obj.TO_STRING());

            UPDATE langdata$drilldowndocuments 
            SET match_document = v_match_document
            WHERE id = p_drilldown_id;

            v_md5 := RAWTOHEX(
                DBMS_CRYPTO.HASH(
                    UTL_RAW.CAST_TO_RAW(p_drilldown_id),
                    DBMS_CRYPTO.HASH_MD5
                )
            );

            v_job_name := 'JOB_LANGDATA_' || UPPER(p_filter_name) || '_' ||
                          v_md5 || '_VALUE_VECTORS';
            BEGIN
                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name        => v_job_name,
                    p_job_action      =>
                        'DECLARE ' ||
                        '  v_enum_set JSON_ARRAY_T := JSON_ARRAY_T.PARSE(''' || 
                        v_updated_enumerable_set.TO_STRING() || '''); ' ||
                        'BEGIN ' ||
                        '  lang_data_utils_pkg.create_value_vector_from_enumerable_set(' ||
                        '    v_enum_set, ' ||
                        '    ''' || p_drilldown_id || ''', ' ||
                        '    ''' || p_filter_name || ''', ' ||
                        '    ''' || 
                        REPLACE(v_filter_description, '''', '''''') || ''', ' ||
                        '    TRUE, ' ||
                        '    ''' || NVL(v_domain_id, '') || '''' ||
                        '  ); ' ||
                        'END;',
                    p_job_class    => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments        => 'Create value vector table from enumerable set',
                    p_priority        => 1,
                    p_restart_on_fail => TRUE,
                    p_restart_on_rec  => TRUE
                );

                lang_data_logger_pkg.log_info(
                    'Submitted job ' || v_job_name ||
                    ' to recreate value vector table for filter ' ||
                    p_filter_name
                );

            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -27477 THEN
                        lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                    ' already exists');
                    ELSE
                        lang_data_logger_pkg.log_fatal(
                            'An unknown error occurred. Error: ' || SQLERRM
                        );
                        RAISE;
                    END IF;
            END;
        ELSE
            lang_data_logger_pkg.log_error(
                'Filter ' || p_filter_name ||
                'does not exist'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
    END remove_from_drilldown_filter_enumerable_set;

    PROCEDURE update_drilldown_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
    BEGIN
        -- Check if drilldown exists
        IF NOT lang_data_utils_pkg.check_drilldown_exists(p_id) THEN
            lang_data_logger_pkg.log_error('drilldown id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Validate if status is among required values
        IF p_status NOT IN ( 'Pending Review',
                             'Approved',
                             'Rejected',
                             'Published',
                             'Inactive',
                             'Archived' ) THEN
            lang_data_logger_pkg.log_error('Invalid status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        EXECUTE IMMEDIATE 
            'UPDATE langdata$drilldowndocuments
             SET status = :status
             WHERE id = :id' 
        USING p_status, p_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;
            

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' ||
                p_id || '. Error: ' || SQLERRM
            );
            raise;
    END update_drilldown_status;

    PROCEDURE update_drilldown_description_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
        v_row_count         VARCHAR2(20);
    BEGIN
        -- Check description exists
        EXECUTE IMMEDIATE 
           'SELECT COUNT(1) FROM langdata$drilldowndescriptions WHERE id = :id' 
        INTO v_row_count USING p_id;

        IF v_row_count = 0 THEN
            lang_data_logger_pkg.log_error('Invalid description id');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Validate if status is among required values
        IF p_status NOT IN ( 'Pending Review',
                             'Approved',
                             'Rejected',
                             'Published',
                             'Inactive',
                             'Archived',
                             'Pending Regression' ) THEN
            lang_data_logger_pkg.log_error('Invalid status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        EXECUTE IMMEDIATE 
            'UPDATE langdata$drilldowndescriptions
             SET status = :status
             WHERE id = :id' 
        USING p_status, p_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;
            

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' ||
                p_id || '. Error: ' || SQLERRM
            );
            raise;
    END update_drilldown_description_status;

    PROCEDURE get_drilldown_regression(
        p_description_id  IN VARCHAR2 DEFAULT NULL,
        p_sample_query_id IN VARCHAR2 DEFAULT NULL,
        p_force               IN BOOLEAN,
        p_json                OUT JSON
    ) IS
        v_regression_json   JSON;
        v_status            VARCHAR2(20);
        v_text_md5          VARCHAR2(36);
        v_job_name          VARCHAR2(255);
        v_is_description    BOOLEAN := TRUE;
    BEGIN

        IF p_description_id IS NULL 
            AND p_sample_query_id IS NULL THEN
            lang_data_logger_pkg.log_error(
                'Both p_description_id and p_sample_query_id '||
                'cannot be null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_description_id IS NOT NULL 
            AND p_sample_query_id IS NOT NULL THEN
            lang_data_logger_pkg.log_error(
                'Both p_description_id and p_sample_query_id '||
                'cannot be not null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_description_id IS NOT NULL THEN
            v_is_description := TRUE;
        ELSE
            v_is_description := FALSE;
        END IF;

        IF v_is_description THEN
            SELECT status, regression_json, ddd_md5
            INTO v_status, v_regression_json, v_text_md5
            FROM langdata$drilldowndescriptions
            WHERE id = p_description_id;
            v_job_name := 'JOB_LANGDATA_'||
                substr(v_text_md5,1,5)||'_DRILLDOWN_D_REGRESSION';
        ELSE
            SELECT status, regression_json, query_md5
            INTO v_status, v_regression_json, v_text_md5
            FROM langdata$samplequeries
            WHERE id = p_sample_query_id;
            v_job_name := 'JOB_LANGDATA_'||
                substr(v_text_md5,1,5)||'_DRILLDOWN_SQ_REGRESSION';
        END IF;

        IF NOT p_force THEN
            IF v_regression_json IS NULL THEN
                lang_data_logger_pkg.log_info('Regression Info Not available');
                p_json := NULL;
            ELSE
                p_json := v_regression_json;
            END IF;
            RETURN;
        END IF;

        IF v_status = 'Pending Regression' THEN
            lang_data_logger_pkg.log_debug(
                'Regression Job already exists dropping job'
            );

            BEGIN
                DBMS_SCHEDULER.drop_job(
                    job_name => v_job_name,
                    force => TRUE
                );
                lang_data_logger_pkg.log_info(
                    'Job ' || v_job_name || ' dropped successfully.'
                );
            EXCEPTION
                WHEN OTHERS THEN
                    -- Catch the exception if the job does not exist
                    IF SQLCODE != -27475 THEN 
                        -- Error code for "job does not exist"
                        lang_data_logger_pkg.log_info(
                            'Error dropping job: ' || SQLERRM
                        );
                    ELSE
                        lang_data_logger_pkg.log_info(
                            'Job ' || v_job_name ||
                            ' does not exist, proceeding to create.'
                        );
                    END IF;
            END;

        END IF;

        IF v_is_description THEN
            UPDATE langdata$drilldowndescriptions
            SET status = 'Pending Regression'
            WHERE id = p_description_id;
        ELSE
            UPDATE langdata$samplequeries
            SET status = 'Pending Regression'
            WHERE id = p_sample_query_id;
        END IF;

        lang_data_drilldowns_pkg.calculate_drilldown_regression(
            p_new_description_id => p_description_id,
            p_new_sample_query_id => p_sample_query_id
        );

        IF v_is_description THEN
            SELECT regression_json INTO v_regression_json
            FROM langdata$drilldowndescriptions WHERE id = p_description_id;
            p_json := v_regression_json;
        ELSE
            SELECT regression_json INTO v_regression_json
            FROM langdata$samplequeries WHERE id = p_sample_query_id;
            p_json := v_regression_json;
        END IF;

        lang_Data_logger_pkg.log_debug('Finished get_drilldown_regression');
        RETURN;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            IF v_is_description THEN
                lang_data_logger_pkg.log_error(
                    'Description with id: '''|| p_description_id ||
                     ''' not found');
            ELSE
                lang_Data_logger_pkg.log_error(
                    'Sample Query with id: '''|| p_sample_query_id || 
                    '''not found');
            END IF;
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred while fetching  ' || 
                'drilldown description regression. Error: ' || SQLERRM
                );
            RAISE;
    END get_drilldown_regression;

    FUNCTION get_drilldown_id_by_title(
        p_title IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_drilldown_id VARCHAR2(36);
    BEGIN
        SELECT id INTO v_drilldown_id FROM langdata$drilldowndocuments
        WHERE title = p_title;
        RETURN v_drilldown_id;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No drilldown found for title: ' || p_title
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_drilldown_id_by_title;

    FUNCTION get_desc_id_by_drilldown_id_version(
        p_drilldown_id    IN VARCHAR2,
        p_version         IN NUMBER
    ) RETURN VARCHAR2 IS
        v_desc_id VARCHAR2(36);
    BEGIN
        SELECT id INTO v_desc_id FROM langdata$drilldowndescriptions
        WHERE drilldown_id = p_drilldown_id AND version = p_version;
        RETURN v_desc_id;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No drilldown description found for drilldown ID: ' || 
                p_drilldown_id || ', version: ' || p_version
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_desc_id_by_drilldown_id_version;

    FUNCTION get_drilldown_status_by_id (
        p_drilldown_id     IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_status        VARCHAR2(20);
    BEGIN
        SELECT status INTO v_status FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;
        RETURN v_status;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No drilldown found for report ID: ' || p_drilldown_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_drilldown_status_by_id;

    FUNCTION get_drilldown_match_document_by_id(
        p_drilldown_id         VARCHAR2
    ) RETURN CLOB IS
        v_match_document    CLOB;
    BEGIN
        SELECT match_document INTO v_match_document FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;
        RETURN v_match_document;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No drilldown found for report ID: ' || p_drilldown_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_drilldown_match_document_by_id;
end lang_data_drilldowns_pkg;
/

