Rem
Rem $Header: dbgendev/src/langdata/plsql/cleanup/cleanup_pkg.pkb /main/14 2025/07/24 18:45:22 saloshah Exp $
Rem
Rem cleanup_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      cleanup_pkg.pkb - Package body of cleanup_pkg
Rem
Rem    DESCRIPTION
Rem      Defines the procedures/functions responsible for cleanup of the 
Rem	     database.
Rem
Rem    NOTES
Rem      NONE
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/cleanup/cleanup_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    07/21/25 - DBAI-1115: Add dropping filter_rec_type,
Rem                           filter_tab_type to
Rem                           lang_data_cleanup_pkg.drop_types
Rem    jiangnhu    07/10/25 - DBAI-1006: Parameterize hard-coded names in
Rem                           drop_resource_plan
Rem    arevathi    06/03/25 - Add purge_outdated_search_records procedure
Rem    deveverm    04/10/25 - DBAI-722: Added compilation branch for ADB
Rem    jiangnhu    03/07/25 - Update drop_resource_plan
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    12/12/24 - DBAI-463: Drop only langdata jobs and triggers
Rem    jiangnhu    12/05/24 - DBAI-421: Add drop_resource_plan
Rem    jiangnhu    11/27/24 - Add drop_types
Rem    jiangnhu    10/30/24 - DBAI-383: Add drop_user_table_indexes
Rem    dadoshi     10/21/24 - Remove text wrapping
Rem    dadoshi     10/18/24 - JIRA_DBAI399: Update template
Rem    pryarla     10/16/24 - Created
Rem

DECLARE
    v_cloud_service VARCHAR2(10);
BEGIN
    SELECT sys_context('USERENV', 'CLOUD_SERVICE') 
    INTO v_cloud_service FROM dual;

    IF v_cloud_service IS NOT NULL THEN
        EXECUTE IMMEDIATE 'ALTER SESSION SET PLSQL_CCFLAGS = ''Is_Cloud:true''';
    ELSE
        EXECUTE IMMEDIATE 'ALTER SESSION SET PLSQL_CCFLAGS = ''Is_Cloud:false''';
    END IF;
END;
/

CREATE OR REPLACE PACKAGE BODY lang_data_cleanup_pkg AS

    -- Procedure to drop all jobs
    PROCEDURE drop_all_jobs IS
        CURSOR job_cur IS
            SELECT job_name
            FROM user_scheduler_jobs
            WHERE job_name LIKE 'JOB_LANGDATA_%';
    BEGIN
        FOR job_rec IN job_cur LOOP
            BEGIN
                DBMS_SCHEDULER.drop_job(
                    job_name => job_rec.job_name, force => TRUE
                );
                lang_data_logger_pkg.log_info(
                    'Dropped job: ' || job_rec.job_name
                );
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_fatal(
                        'Could not drop job: ' || job_rec.job_name || ' - ' 
                        || SQLERRM
                    );
                    RAISE;
            END;
        END LOOP;
    END drop_all_jobs;

    -- Procedure to drop foreign keys for a specific table
    PROCEDURE drop_foreign_keys(p_table_name IN VARCHAR2) IS
        CURSOR fk_cur IS
            SELECT constraint_name
            FROM user_constraints
            WHERE table_name = UPPER(p_table_name)
              AND constraint_type = 'R';  -- 'R' is for foreign key constraints
    BEGIN
        FOR fk_rec IN fk_cur LOOP
            BEGIN
                EXECUTE IMMEDIATE 'ALTER TABLE ' || p_table_name || 
                        ' DROP CONSTRAINT ' || fk_rec.constraint_name;
                lang_data_logger_pkg.log_info(
                    'Dropped foreign key: ' || fk_rec.constraint_name || 
                    ' from table ' || p_table_name
                );
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_fatal(
                        'Could not drop foreign key ' || 
                        fk_rec.constraint_name || ' - ' || SQLERRM
                    );
                    RAISE;
            END;
        END LOOP;
    END drop_foreign_keys;

    -- Procedure to drop tables and their foreign keys
    PROCEDURE drop_tables IS
        CURSOR table_cur IS
            SELECT table_name
            FROM user_tables
            WHERE table_name LIKE 'LANGDATA$%';
    BEGIN
        FOR table_rec IN table_cur LOOP
            BEGIN
                -- Drop foreign keys first
                drop_foreign_keys(table_rec.table_name);

                -- Drop the table
                EXECUTE IMMEDIATE 'DROP TABLE ' || table_rec.table_name || 
                ' CASCADE CONSTRAINTS';
                lang_data_logger_pkg.log_info(
                    'Dropped table: ' || table_rec.table_name
                );
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_fatal(
                        'Error dropping table ' || table_rec.table_name ||
                         ' - ' || SQLERRM
                    );
                    RAISE;
            END;
        END LOOP;
    END drop_tables;

    -- Procedure to drop all triggers
    PROCEDURE drop_triggers IS
        CURSOR trigger_cur IS
            SELECT trigger_name
            FROM user_triggers
            WHERE trigger_name LIKE 'LANGDATA$%';
    BEGIN
        FOR trigger_rec IN trigger_cur LOOP
            BEGIN
                EXECUTE IMMEDIATE 'DROP TRIGGER ' || trigger_rec.trigger_name;
                lang_data_logger_pkg.log_info(
                    'Dropped trigger: ' || trigger_rec.trigger_name
                );
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_fatal(
                        'Could not drop trigger ' || trigger_rec.trigger_name ||
                         ' - ' || SQLERRM
                    );
                    RAISE;
            END;
        END LOOP;
    END drop_triggers;

    PROCEDURE drop_types IS
    BEGIN
        -- Drop filter_tab_type type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE filter_tab_type';
            lang_data_logger_pkg.log_info(
                'Dropped type filter_tab_type successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type filter_tab_type does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop filter_tab_type: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop filter_rec_type type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE filter_rec_type';
            lang_data_logger_pkg.log_info(
                'Dropped type filter_rec_type successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type filter_rec_type does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop filter_rec_type: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop ranked_doc_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE ranked_doc_table';
            lang_data_logger_pkg.log_info(
                'Dropped type ranked_doc_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type ranked_doc_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop ranked_doc_table: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop ranked_doc_obj type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE ranked_doc_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type ranked_doc_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type ranked_doc_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop ranked_doc_obj: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop ranked_drilldown_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE ranked_drilldown_table';
            lang_data_logger_pkg.log_info(
                'Dropped type ranked_drilldown_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type ranked_drilldown_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop ranked_drilldown_table: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop ranked_drilldown_obj type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE ranked_drilldown_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type ranked_drilldown_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type ranked_drilldown_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop ranked_drilldown_obj: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop ranked_report_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE ranked_report_table';
            lang_data_logger_pkg.log_info(
                'Dropped type ranked_report_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type ranked_report_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop ranked_report_table: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop ranked_report_obj type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE ranked_report_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type ranked_report_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type ranked_report_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop ranked_report_obj: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop report_row_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE report_row_table';
            lang_data_logger_pkg.log_info(
                'Dropped type report_row_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type report_row_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop report_row_table: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop report_row_obj type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE report_row_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type report_row_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type report_row_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop report_row_obj: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop data_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE data_table';
            lang_data_logger_pkg.log_info(
                'Dropped type data_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type data_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop data_table: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop data_row_obj type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE data_row_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type data_row_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type data_row_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop data_row_obj: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop drilldown_row_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE drilldown_row_table';
            lang_data_logger_pkg.log_info(
                'Dropped type drilldown_row_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type drilldown_row_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop drilldown_row_table: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop drilldown_row_obj type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE drilldown_row_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type drilldown_row_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type drilldown_row_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop drilldown_row_obj: ' || SQLERRM
                    );
                    RAISE;
                END IF;
        END;


        -- Drop drilldown_description_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE drilldown_description_table';
            lang_data_logger_pkg.log_info(
                'Dropped type drilldown_description_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type drilldown_description_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop drilldown_description_table: ' || 
                        SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop drilldown_description_row_obj type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE drilldown_description_row_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type drilldown_description_row_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type drilldown_description_row_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop drilldown_description_row_obj: '
                        || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        -- Drop user_search_row_table type
        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE user_search_row_table';
            lang_data_logger_pkg.log_info(
                'Dropped type user_search_row_table successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type user_search_row_table does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop user_search_row_table: '
                        || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

        BEGIN
            EXECUTE IMMEDIATE 'DROP TYPE user_search_row_obj';
            lang_data_logger_pkg.log_info(
                'Dropped type user_search_row_obj successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = -4043 THEN
                    lang_data_logger_pkg.log_info(
                        'Type user_search_row_obj does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Failed to drop user_search_row_obj: '
                        || SQLERRM
                    );
                    RAISE;
                END IF;
        END;

    END drop_types;

    PROCEDURE drop_user_table_indexes IS
    BEGIN
        -- Loop through all indexes with the 'LANGDATA' prefix
        FOR idx IN (
            SELECT index_name, table_name
            FROM user_indexes
            WHERE index_name LIKE 'LANGDATA%'  -- Match all indexes with the LANGDATA prefix
            AND table_name NOT LIKE '%LANGDATA%' 
            AND table_name NOT LIKE '%ALL_MINILM_L12_V2%'
        ) LOOP
            BEGIN
                -- Drop the index
                EXECUTE IMMEDIATE 'DROP INDEX ' || idx.index_name;

                -- Log the dropped index
                lang_data_logger_pkg.log_info(
                    'Dropped index: ' || idx.index_name
                );
            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE != -01418 THEN
                        -- Log the error if index drop fails
                        lang_data_logger_pkg.log_error(
                            'Failed to drop index: ' || idx.index_name || ' - ' || SQLERRM
                        );

                        RAISE;
                    END IF;
            END;
        END LOOP;

        -- Final log message indicating completion
        lang_data_logger_pkg.log_info(
            'Completed dropping LANGDATA-prefixed indexes for user tables.'
        );
    END drop_user_table_indexes;

    PROCEDURE drop_resource_plan AS
        v_job_class_name           VARCHAR2(100);
        v_resource_plan_name       VARCHAR2(100);
        v_consumer_group_name      VARCHAR2(100);
    BEGIN
        v_job_class_name := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_JOB_CLASS_NAME'
        );
        v_resource_plan_name := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_RESOURCE_PLAN_NAME'
        );
        v_consumer_group_name := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_CONSUMER_GROUP_NAME'
        );

        -- Drop job class if it exists
        BEGIN

            DBMS_SCHEDULER.DROP_JOB_CLASS(v_job_class_name);
            lang_data_logger_pkg.log_info(
                'Job class ' || v_job_class_name || ' dropped successfully.'
            );
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                lang_data_logger_pkg.log_info(
                    'Job class ' || v_job_class_name || ' does not exist.'
                );
            WHEN OTHERS THEN
                IF SQLCODE = -27476 THEN
                    lang_data_logger_pkg.log_info(
                        'Job class ' || v_job_class_name || ' does not exist.'
                    );
                ELSE
                    lang_data_logger_pkg.log_error(
                        'Error dropping job class: ' || SQLERRM
                    );
                END IF;
        END;

        $IF $$Is_Cloud = FALSE $THEN

            DBMS_RESOURCE_MANAGER.CREATE_PENDING_AREA;

            -- Deactivate resource plan
            BEGIN
                DBMS_RESOURCE_MANAGER.SWITCH_PLAN(NULL);
                lang_data_logger_pkg.log_info('Resource plan deactivated.');
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_error(
                        'Error deactivating resource plan: ' || SQLERRM
                    );
            END;

            -- Delete plan directives if they exist
            BEGIN
                DBMS_RESOURCE_MANAGER.DELETE_PLAN_DIRECTIVE(
                    plan => v_resource_plan_name,
                    group_or_subplan => v_consumer_group_name
                );
                DBMS_RESOURCE_MANAGER.DELETE_PLAN_DIRECTIVE(
                    plan => v_resource_plan_name,
                    group_or_subplan => 'OTHER_GROUPS'
                );
                lang_data_logger_pkg.log_info(
                    'Plan directives deleted successfully.'
                );
            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -29358 THEN
                        lang_data_logger_pkg.log_info(
                            'Plan directives do not exist.'
                        );
                    ELSE
                        lang_data_logger_pkg.log_error(
                            'Error deleting plan directives: ' || SQLERRM
                        );
                    END IF;
            END;

            -- Delete consumer group if it exists
            BEGIN
                DBMS_RESOURCE_MANAGER.DELETE_CONSUMER_GROUP(
                    v_consumer_group_name
                );
                lang_data_logger_pkg.log_info(
                    'Resource consumer group ' || v_consumer_group_name ||
                    ' has been deleted successfully.'
                );
            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -29368 THEN
                        lang_data_logger_pkg.log_info(
                            'Resource consumer group does not exist.'
                        );
                    ELSE
                        lang_data_logger_pkg.log_error(
                            'Error deleting resource group: ' || SQLERRM
                        );
                    END IF;
            END;

            -- Delete resource plan if it exists
            BEGIN
                DBMS_RESOURCE_MANAGER.DELETE_PLAN(v_resource_plan_name);
                lang_data_logger_pkg.log_info(
                    'Resource plan ' || v_resource_plan_name ||
                    ' has been deleted successfully.'
                );
            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -29358 THEN
                        lang_data_logger_pkg.log_info(
                            'Resource plan does not exist.'
                        );
                    ELSE
                        lang_data_logger_pkg.log_error(
                            'Error deleting resource plan: ' || SQLERRM
                        );
                    END IF;
            END;

            BEGIN
                DBMS_RESOURCE_MANAGER.VALIDATE_PENDING_AREA;
                DBMS_RESOURCE_MANAGER.SUBMIT_PENDING_AREA;
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_error(
                        'Error submitting pending area: ' || SQLERRM
                    );
            END;
        $ElSE
            null;
        $END
        EXCEPTION
            WHEN OTHERS THEN
                lang_data_logger_pkg.log_error(
                    'Error dropping resource plan: ' || SQLERRM
                );
    END drop_resource_plan;

    -- Main cleanup procedure
    PROCEDURE cleanup_langdata IS
    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;
        
        -- Dropping all jobs
        lang_data_cleanup_pkg.drop_all_jobs;

        -- Dropping all triggers
        lang_data_cleanup_pkg.drop_triggers;

        -- Dropping all tables
        lang_data_cleanup_pkg.drop_tables;

        -- Dropping all types
        lang_data_cleanup_pkg.drop_types;

        -- Dropping all indexes on user tables
        lang_data_cleanup_pkg.drop_user_table_indexes;

        -- Dropping resource plan and job class
        lang_data_cleanup_pkg.drop_resource_plan;

        COMMIT;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_unauthorized_code THEN
                RAISE;
            ELSE
                ROLLBACK;
                lang_data_logger_pkg.log_fatal(
                    'Error during cleanup: ' || SQLERRM);
                RAISE;
            END IF;
    END cleanup_langdata;

    PROCEDURE delete_search_records(
        p_cutoff_time  IN TIMESTAMP
    ) IS
    BEGIN
        lang_data_logger_pkg.log_debug(
            'Deleting search records older than ' || p_cutoff_time
        );
        DELETE FROM langdata$searchrecords
        WHERE created_at < p_cutoff_time;

        lang_data_logger_pkg.log_info(
            'Deleted search records created before: ' || p_cutoff_time
        );
    END delete_search_records;

    PROCEDURE archive_search_records(
        p_cutoff_time  IN TIMESTAMP
    ) IS
    BEGIN
        lang_data_logger_pkg.log_debug(
            'Archiving search records older than ' || p_cutoff_time
        );
        INSERT INTO langdata$searchrecords_archive (
            id,
            query_text,
            expected_report_id,
            expected_drilldown_id,
            report_matches,
            feedback_rating,
            feedback_comments,
            search_type,
            username,
            required_feedback_action,
            feedback_action_priority,
            created_at,
            updated_at,
            augmented_query_text
        )
        SELECT 
            id,
            query_text,
            expected_report_id,
            expected_drilldown_id,
            report_matches,
            feedback_rating,
            feedback_comments,
            search_type,
            username,
            required_feedback_action,
            feedback_action_priority,
            created_at,
            updated_at,
            augmented_query_text
        FROM langdata$searchrecords
        WHERE created_at < p_cutoff_time;

        DELETE FROM langdata$searchrecords
        WHERE created_at < p_cutoff_time;

        lang_data_logger_pkg.log_info(
            'Deleted search records created before: ' || p_cutoff_time
        );
    END archive_search_records;

    PROCEDURE purge_outdated_search_records (
        p_days         IN NUMBER DEFAULT 180,
        p_cutoff_time  IN TIMESTAMP DEFAULT NULL,
        p_action       IN VARCHAR2 DEFAULT 'ARCHIVE',
        p_output_msg   OUT VARCHAR2
    ) IS
        v_cutoff_time TIMESTAMP;
    BEGIN
        -- Set default failure message
        p_output_msg := 'Failure';

        lang_data_logger_pkg.log_debug(
            'Starting procedure purge_outdated_search_records with p_days: '
            || p_days || ', p_cutoff_time: '|| p_cutoff_time || ', p_action: ' 
            || p_action
        );

        -- Validate action parameter: only 'ARCHIVE' or 'DELETE' are allowed
        IF UPPER(p_action) NOT IN ('ARCHIVE', 'DELETE') THEN
            lang_data_logger_pkg.log_error(
                'Invalid action parameter: ' || p_action
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Use provided cutoff time or SYSDATE - p_days if cutoff time is NULL
        v_cutoff_time := COALESCE(p_cutoff_time, SYSDATE - p_days);

        lang_data_logger_pkg.log_debug(
            'Dropping search records created before: ' || v_cutoff_time
        );

         -- Perform ARCHIVE operation
        IF UPPER(p_action) = 'ARCHIVE' THEN
            lang_data_cleanup_pkg.archive_search_records(
                p_cutoff_time => v_cutoff_time
            );
        ELSE 
            -- If action is DELETE, remove records older than the cutoff
            lang_data_cleanup_pkg.delete_search_records(
                p_cutoff_time => v_cutoff_time
            );
            
        END IF;

        -- Set output message to indicate success
        p_output_msg := 'Success';
        lang_data_logger_pkg.log_debug(
            'Archived search_records older than ' || 
            v_cutoff_time
        );
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_invalid_parameters_code,
                lang_data_errors_pkg.c_unauthorized_code 
            ) THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                -- Log unknown errors as fatal and raise the generic error code
                lang_data_logger_pkg.log_fatal(
                    'An error occurred while archiving search records, Error: '
                     || SQLERRM
                );
                RAISE;
            END IF;
    END purge_outdated_search_records;

END lang_data_cleanup_pkg;
/

