/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.objectstorage.transfer.internal;

import com.oracle.bmc.http.client.io.DuplicatableInputStream;
import com.oracle.bmc.model.BmcException;
import com.oracle.bmc.objectstorage.model.ChecksumAlgorithm;
import com.oracle.bmc.objectstorage.responses.CommitMultipartUploadResponse;
import com.oracle.bmc.objectstorage.responses.GetObjectResponse;
import com.oracle.bmc.objectstorage.responses.HeadObjectResponse;
import com.oracle.bmc.objectstorage.responses.PutObjectResponse;
import com.oracle.bmc.objectstorage.responses.UploadPartResponse;
import com.oracle.bmc.objectstorage.transfer.UploadManager;
import com.oracle.bmc.objectstorage.transfer.internal.CRC32C;
import com.oracle.bmc.objectstorage.transfer.internal.ChecksumInfo;
import com.oracle.bmc.objectstorage.transfer.internal.StreamHelper;
import com.oracle.bmc.responses.BmcResponse;
import com.oracle.bmc.util.StreamUtils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.security.DigestOutputStream;
import java.util.Arrays;
import java.util.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChecksumUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ChecksumUtils.class);

    private ChecksumUtils() {
    }

    public static MD5Calculation calculateMd5(InputStream stream, Long contentLength) {
        return ChecksumUtils.calculateChecksum(stream, contentLength, StreamHelper::createMd5MessageOutputStream, MD5Calculation::new, "MD5");
    }

    public static SHA384Calculation calculateSha384(InputStream stream, Long contentLength) {
        return ChecksumUtils.calculateChecksum(stream, contentLength, StreamHelper::createSha384MessageOutputStream, SHA384Calculation::new, ChecksumAlgorithm.Sha384.getValue());
    }

    public static SHA256Calculation calculateSha256(InputStream stream, Long contentLength) {
        return ChecksumUtils.calculateChecksum(stream, contentLength, StreamHelper::createSha256MessageOutputStream, SHA256Calculation::new, ChecksumAlgorithm.Sha256.getValue());
    }

    public static CRC32CCalculation calculateCrc32c(InputStream stream, Long contentLength) {
        return ChecksumUtils.calculateChecksum(stream, contentLength, StreamHelper::createCrc32cMessageOutputStream, CRC32CCalculation::new, ChecksumAlgorithm.Crc32C.getValue());
    }

    private static <T> T calculateChecksum(InputStream stream, Long contentLength, DigestOutputStreamFactory outputStreamFactory, ChecksumCalculationFactory<T> calculationFactory, String algorithmName) {
        InputStream streamToReturn;
        String checksum;
        if (stream instanceof DuplicatableInputStream) {
            checksum = ChecksumUtils.performChecksumCalculation(((DuplicatableInputStream)stream).duplicate(), new StreamHelper.NullOutputStream(), contentLength, outputStreamFactory, algorithmName);
            streamToReturn = stream;
        } else {
            LOG.info("About to copy object into memory to calculate {}, may lead to OutOfMemory exceptions", (Object)algorithmName);
            if (contentLength > Integer.MAX_VALUE) {
                throw new BmcException(false, String.format("Cannot compute %s client-side as content length (%d) is larger than max buffer. Disable %s enforcement or provide a DuplicableInputStream to avoid this problem", algorithmName, contentLength, algorithmName), null, null);
            }
            try {
                ByteArrayOutputStream baos = new ByteArrayOutputStream(contentLength.intValue());
                checksum = ChecksumUtils.performChecksumCalculation(stream, baos, contentLength, outputStreamFactory, algorithmName);
                streamToReturn = StreamUtils.createByteArrayInputStream((byte[])baos.toByteArray());
            }
            catch (OutOfMemoryError oom) {
                OutOfMemoryError newOom = new OutOfMemoryError(String.format("Could not compute %s. Disable %s enforcement or provide a DuplicableInputStream to avoid this problem", algorithmName, algorithmName));
                newOom.initCause(oom);
                throw newOom;
            }
        }
        return calculationFactory.create(streamToReturn, checksum);
    }

    private static String performChecksumCalculation(InputStream stream, OutputStream outputStream, long contentLength, DigestOutputStreamFactory outputStreamFactory, String algorithmName) {
        long bytesCopied;
        DigestOutputStream digestOutputStream = outputStreamFactory.create(outputStream);
        try {
            bytesCopied = StreamHelper.copy(stream, digestOutputStream);
        }
        catch (IOException e) {
            throw new BmcException(false, "Unable to calculate " + algorithmName + " checksum", (Throwable)e, null);
        }
        if (bytesCopied != contentLength) {
            throw new BmcException(false, String.format("Failed to read all bytes while calculating %s checksum: %d, %d", algorithmName, bytesCopied, contentLength), null, null);
        }
        return StreamHelper.base64Encode(digestOutputStream.getMessageDigest());
    }

    public static String calculateCrc32cChecksum(byte[] data) {
        CRC32C crc32c = new CRC32C();
        crc32c.update(data, 0, data.length);
        long crcValue = crc32c.getValue();
        byte[] crcBytes = ByteBuffer.allocate(8).putLong(crcValue).array();
        crcBytes = Arrays.copyOfRange(crcBytes, 4, 8);
        return Base64.getEncoder().encodeToString(crcBytes);
    }

    public static ChecksumInfo getExpectedChecksumAndAlgorithm(GetObjectResponse response) {
        ChecksumAlgorithm algorithm;
        String expectedChecksum;
        if (response.getOpcContentCrc32c() != null) {
            expectedChecksum = response.getOpcContentCrc32c();
            algorithm = ChecksumAlgorithm.Crc32C;
        } else if (response.getOpcContentSha256() != null) {
            expectedChecksum = response.getOpcContentSha256();
            algorithm = ChecksumAlgorithm.Sha256;
        } else if (response.getOpcContentSha384() != null) {
            expectedChecksum = response.getOpcContentSha384();
            algorithm = ChecksumAlgorithm.Sha384;
        } else {
            expectedChecksum = response.getContentMd5();
            algorithm = null;
        }
        return new ChecksumInfo(expectedChecksum, algorithm);
    }

    public static String getChecksumFromUploadResponse(UploadManager.UploadResponse uploadResponse, ChecksumAlgorithm checksumAlgorithm, boolean isMultipart) {
        if (checksumAlgorithm == null) {
            return null;
        }
        switch (checksumAlgorithm) {
            case Crc32C: {
                return uploadResponse.getContentCrc32c();
            }
            case Sha256: {
                return isMultipart ? uploadResponse.getMultipartSha256() : uploadResponse.getContentSha256();
            }
            case Sha384: {
                return isMultipart ? uploadResponse.getMultipartSha384() : uploadResponse.getContentSha384();
            }
        }
        throw new IllegalArgumentException("Unsupported checksum algorithm: " + checksumAlgorithm);
    }

    public static String getResponseChecksum(BmcResponse response, ChecksumAlgorithm algorithm, boolean isMultipart) {
        if (algorithm == null) {
            return null;
        }
        if (response instanceof GetObjectResponse) {
            return ChecksumUtils.getChecksumForGetObjectResponse((GetObjectResponse)response, algorithm, isMultipart);
        }
        if (response instanceof HeadObjectResponse) {
            return ChecksumUtils.getChecksumForHeadObjectResponse((HeadObjectResponse)response, algorithm, isMultipart);
        }
        if (response instanceof PutObjectResponse) {
            return ChecksumUtils.getChecksumForPutObjectResponse((PutObjectResponse)response, algorithm);
        }
        if (response instanceof UploadPartResponse) {
            return ChecksumUtils.getChecksumForUploadPartResponse((UploadPartResponse)response, algorithm);
        }
        if (response instanceof CommitMultipartUploadResponse) {
            return ChecksumUtils.getChecksumForCommitMultipartUploadResponse((CommitMultipartUploadResponse)response, algorithm);
        }
        throw new IllegalArgumentException("Unsupported response type");
    }

    private static String getChecksumForGetObjectResponse(GetObjectResponse response, ChecksumAlgorithm algorithm, boolean isMultipart) {
        switch (algorithm) {
            case Crc32C: {
                return response.getOpcContentCrc32c();
            }
            case Sha256: {
                return isMultipart ? response.getOpcMultipartSha256() : response.getOpcContentSha256();
            }
            case Sha384: {
                return isMultipart ? response.getOpcMultipartSha384() : response.getOpcContentSha384();
            }
        }
        throw new IllegalArgumentException("Unsupported checksum algorithm: " + algorithm);
    }

    private static String getChecksumForHeadObjectResponse(HeadObjectResponse response, ChecksumAlgorithm algorithm, boolean isMultipart) {
        switch (algorithm) {
            case Crc32C: {
                return response.getOpcContentCrc32c();
            }
            case Sha256: {
                return isMultipart ? response.getOpcMultipartSha256() : response.getOpcContentSha256();
            }
            case Sha384: {
                return isMultipart ? response.getOpcMultipartSha384() : response.getOpcContentSha384();
            }
        }
        throw new IllegalArgumentException("Unsupported checksum algorithm: " + algorithm);
    }

    private static String getChecksumForPutObjectResponse(PutObjectResponse response, ChecksumAlgorithm algorithm) {
        switch (algorithm) {
            case Crc32C: {
                return response.getOpcContentCrc32c();
            }
            case Sha256: {
                return response.getOpcContentSha256();
            }
            case Sha384: {
                return response.getOpcContentSha384();
            }
        }
        throw new IllegalArgumentException("Unsupported checksum algorithm: " + algorithm);
    }

    private static String getChecksumForUploadPartResponse(UploadPartResponse response, ChecksumAlgorithm algorithm) {
        switch (algorithm) {
            case Crc32C: {
                return response.getOpcContentCrc32c();
            }
            case Sha256: {
                return response.getOpcContentSha256();
            }
            case Sha384: {
                return response.getOpcContentSha384();
            }
        }
        throw new IllegalArgumentException("Unsupported checksum algorithm: " + algorithm);
    }

    private static String getChecksumForCommitMultipartUploadResponse(CommitMultipartUploadResponse response, ChecksumAlgorithm algorithm) {
        switch (algorithm) {
            case Crc32C: {
                return response.getOpcContentCrc32c();
            }
            case Sha256: {
                return response.getOpcMultipartSha256();
            }
            case Sha384: {
                return response.getOpcMultipartSha384();
            }
        }
        throw new IllegalArgumentException("Unsupported checksum algorithm: " + algorithm);
    }

    @FunctionalInterface
    private static interface DigestOutputStreamFactory {
        public DigestOutputStream create(OutputStream var1);
    }

    @FunctionalInterface
    private static interface ChecksumCalculationFactory<T> {
        public T create(InputStream var1, String var2);
    }

    public static class MD5Calculation {
        private final InputStream streamToUse;
        private final String md5;

        public MD5Calculation(InputStream streamToUse, String md5) {
            this.streamToUse = streamToUse;
            this.md5 = md5;
        }

        public InputStream getStreamToUse() {
            return this.streamToUse;
        }

        public String getMd5() {
            return this.md5;
        }
    }

    public static class SHA384Calculation {
        private final InputStream streamToUse;
        private final String sha384;

        public SHA384Calculation(InputStream streamToUse, String sha384) {
            this.streamToUse = streamToUse;
            this.sha384 = sha384;
        }

        public InputStream getStreamToUse() {
            return this.streamToUse;
        }

        public String getSha384() {
            return this.sha384;
        }
    }

    public static class SHA256Calculation {
        private final InputStream streamToUse;
        private final String sha256;

        public SHA256Calculation(InputStream streamToUse, String sha256) {
            this.streamToUse = streamToUse;
            this.sha256 = sha256;
        }

        public InputStream getStreamToUse() {
            return this.streamToUse;
        }

        public String getSha256() {
            return this.sha256;
        }
    }

    public static class CRC32CCalculation {
        private final InputStream streamToUse;
        private final String crc32c;

        public CRC32CCalculation(InputStream streamToUse, String crc32c) {
            this.streamToUse = streamToUse;
            this.crc32c = crc32c;
        }

        public InputStream getStreamToUse() {
            return this.streamToUse;
        }

        public String getCrc32c() {
            return this.crc32c;
        }
    }
}

