/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.objectstorage.transfer.internal;

import com.oracle.bmc.objectstorage.model.ChecksumAlgorithm;
import com.oracle.bmc.objectstorage.transfer.internal.CRC32C;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChecksumInputStream
extends FilterInputStream {
    private static final Logger LOG = LoggerFactory.getLogger(ChecksumInputStream.class);
    private final ChecksumAlgorithm algorithm;
    private final String expectedChecksum;
    private final CRC32C crc32c;
    private final MessageDigest digest;
    private final boolean isMd5;
    private volatile boolean checksumVerified = false;
    private volatile long totalBytesRead = 0L;
    private final long contentLength;

    public ChecksumInputStream(InputStream in, ChecksumAlgorithm algorithm, String expectedChecksum, long contentLength) throws NoSuchAlgorithmException {
        super(in);
        this.isMd5 = algorithm == null;
        this.algorithm = algorithm;
        this.expectedChecksum = expectedChecksum;
        this.crc32c = this.algorithm == ChecksumAlgorithm.Crc32C ? new CRC32C() : null;
        this.digest = ChecksumInputStream.selectDigest(this.algorithm, this.isMd5);
        this.contentLength = contentLength;
    }

    @Override
    public int read() throws IOException {
        int byteRead = super.read();
        if (byteRead != -1) {
            ++this.totalBytesRead;
            this.updateChecksum(byteRead);
            if (this.totalBytesRead == this.contentLength) {
                this.verifyChecksum();
                this.checksumVerified = true;
            }
        }
        return byteRead;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        int bytesRead = super.read(b, off, len);
        if (bytesRead != -1) {
            this.totalBytesRead += (long)bytesRead;
            this.updateChecksum(b, off, bytesRead);
            if (this.totalBytesRead == this.contentLength) {
                this.verifyChecksum();
                this.checksumVerified = true;
            }
        }
        return bytesRead;
    }

    @Override
    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    @Override
    public long skip(long n) throws IOException {
        throw new IOException("skip not supported");
    }

    private void updateChecksum(int byteRead) {
        if (this.algorithm == ChecksumAlgorithm.Crc32C) {
            this.crc32c.update(byteRead);
        } else {
            this.digest.update((byte)byteRead);
        }
    }

    private void updateChecksum(byte[] b, int off, int len) {
        if (this.algorithm == ChecksumAlgorithm.Crc32C) {
            this.crc32c.update(b, off, len);
        } else {
            this.digest.update(b, off, len);
        }
    }

    private void verifyChecksum() throws IOException {
        if (this.expectedChecksum == null) {
            throw new IOException("Expected checksum is null");
        }
        if (this.algorithm == ChecksumAlgorithm.Crc32C) {
            long calculatedChecksum = this.crc32c.getValue();
            byte[] expectedChecksumBytes = Base64.getDecoder().decode(this.expectedChecksum);
            if (expectedChecksumBytes.length != 4) {
                throw new IOException("Expected checksum length is invalid");
            }
            int expectedChecksumValue = ByteBuffer.wrap(expectedChecksumBytes).getInt();
            if (calculatedChecksum != Integer.toUnsignedLong(expectedChecksumValue)) {
                LOG.error("Checksum mismatch: expected {}, but calculated {}", (Object)this.expectedChecksum, (Object)calculatedChecksum);
                throw new IOException("Checksum mismatch: expected " + this.expectedChecksum + " but calculated " + calculatedChecksum);
            }
            LOG.debug("CRC32C checksum verified successfully");
        } else {
            byte[] expectedDigest;
            byte[] calculatedDigest = this.digest.digest();
            if (!MessageDigest.isEqual(calculatedDigest, expectedDigest = Base64.getDecoder().decode(this.expectedChecksum))) {
                LOG.error("Checksum mismatch");
                throw new IOException("Checksum mismatch");
            }
            LOG.debug("{} checksum verified successfully", (Object)(this.isMd5 ? "MD5" : this.algorithm));
        }
    }

    private static MessageDigest selectDigest(ChecksumAlgorithm algorithm, boolean isMd5) throws NoSuchAlgorithmException {
        if (algorithm == ChecksumAlgorithm.Sha256) {
            return MessageDigest.getInstance("SHA-256");
        }
        if (algorithm == ChecksumAlgorithm.Sha384) {
            return MessageDigest.getInstance("SHA-384");
        }
        if (isMd5) {
            return MessageDigest.getInstance("MD5");
        }
        return null;
    }
}

