/* iod-telemetry ActiveSession.sql version 2022.03.23 csierra */
/* Notes:
    1. Subset of v$active_session_history:
       Include only Dimensions with Average Active Sessions (AAS) >= 0.1 (i.e.: 6 or more samples over a period of last one minute)
2022/03/22 - metric_value would be zero instead of null - csierra
2022/03/23 - application_category was returning UN when SQL decoration was longer than 60
2022/05/04 - Fix Grafana Metadata for Blocker Machine (DBPERF-8125)
*/
WITH
FUNCTION application_category (
  p_sql_text     IN VARCHAR2, 
  p_command_name IN VARCHAR2 DEFAULT NULL
)
RETURN VARCHAR2 DETERMINISTIC
IS
  k_appl_handle_prefix CONSTANT VARCHAR2(30) := '/*'||CHR(37);
  k_appl_handle_suffix CONSTANT VARCHAR2(30) := CHR(37)||'*/'||CHR(37);
BEGIN
  IF    p_sql_text LIKE k_appl_handle_prefix||'Transaction Processing'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'addTransactionRow'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'checkEndRowValid'||k_appl_handle_suffix
    OR  p_sql_text LIKE k_appl_handle_prefix||'checkStartRowValid'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'deleteValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'exists'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch commit by idempotency token'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch latest transactions for cache'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find lower commit id for transaction cache warm up'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'findMatchingRow'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getMaxTransactionCommitID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getNewTransactionID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getTransactionProgress'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'lockForCommit'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'lockKievTransactor'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'putBucket'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readTransactionsSince'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'recordTransactionState'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'setValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'SPM:CP'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateIdentityValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateNextKievTransID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateTransactorState'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'upsert_transactor_state'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'writeTransactionKeys'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'QueryTransactorHosts'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'WriteBucketValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'batch commit'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'batch mutation log'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'fetchAllIdentities'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'fetch_epoch'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readFromTxorStateBeginTxn'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readOnlyBeginTxn'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'validateTransactorState'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getDataStoreMaxTransaction'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'legacyGetDataStoreMaxTransaction'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isPartitionDropDisabled'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getWithVersionOffsetSql'||k_appl_handle_suffix 
    OR  LOWER(p_sql_text) LIKE CHR(37)||'lock table kievtransactions'||CHR(37) 
    --
    OR  p_sql_text LIKE k_appl_handle_prefix||'Set ddl lock timeout for session'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.step_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.leases_types'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getForUpdate.dataplane_alias'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.leases_types'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.step_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.dataplane_alias'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Drop partition'||k_appl_handle_suffix 
  THEN RETURN 'TP'; /* Transaction Processing */
  --
  ELSIF p_sql_text LIKE k_appl_handle_prefix||'Read Only'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketIndexSelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketKeySelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketValueSelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countTransactions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch snapshots'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Get system time'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getAutoSequences'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getNextIdentityValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Lock row Bucket_Snapshot'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'longFromDual'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performContinuedScanValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performFirstRowsScanQuery'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performScanQuery'||k_appl_handle_suffix
    OR  p_sql_text LIKE k_appl_handle_prefix||'performSnapshotScanQuery'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performStartScanValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'selectBuckets'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch latest revisions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch max sequence for'||k_appl_handle_suffix -- streaming
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch partition interval for'||k_appl_handle_suffix -- streaming
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find High value for'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find partitions for'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Init lock name for snapshot'||k_appl_handle_suffix -- snapshot
    OR  p_sql_text LIKE k_appl_handle_prefix||'List snapshot tables.'||k_appl_handle_suffix -- snapshot
    OR  p_sql_text LIKE k_appl_handle_prefix||'Tail read bucket'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performSegmentedScanQuery'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'listArchiveStatusByIndexName'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'listAssignmentsByIndexName'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'listHosts'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateHost'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateArchiveStatus'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateOperationLock'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get KIEVWORKFLOWS table indexes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'GetOldestLsn'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'GetStreamRecords'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Check if another workflow is running'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Delete old workflows from'||k_appl_handle_suffix 
    --
    OR  p_sql_text LIKE k_appl_handle_prefix||'getFutureWorkflowDefinition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getPriorWorkflowDefinition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateLeases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateLeaseTypes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getHistoricalAssignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getAllWorkflowDefinitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getVersionHistory'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getInstances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find interval partitions for schema'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getStepInstances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getOldestGcWorkflowInstance'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getNextRecordID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.dataplane_alias'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLeaseNonce'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.leases_types'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getByKey.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getRecordId.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLast.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isPartitionDropDisabled'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Check if there are active rows for partition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getRunningInstancesCount'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLatestWorkflowDefinition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLast.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.step_instances'||k_appl_handle_suffix 
  THEN RETURN 'RO'; /* Read Only */
  --
  ELSIF p_sql_text LIKE k_appl_handle_prefix||'Background'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Bootstrap snapshot table Kiev_S'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketIdentitySelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'checkMissingTables'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countAllBuckets'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countAllRows'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countKievTransactionRows'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countKtkRows'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Delete garbage'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Delete rows from'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'deleteBucketGarbage'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateSequences'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch config'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'fetch_leader_heartbeat'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'gcEventMaxId'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'gcEventTryInsert'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Get txn at time'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get_leader'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getCurEndTime'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getDBSchemaVersion'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getEndTimeOlderThan'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getGCLogEntries'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getMaxTransactionOlderThan'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getSchemaMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getSupportedLibVersions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'hashBucket'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'hashSnapshot'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Populate workspace'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'populateBucketGCWorkspace'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'primeTxCache'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readOnlyRoleExists'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Row count between transactions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'secondsSinceLastGcEvent'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'sync_leadership'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Test if table Kiev_S'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Update snapshot metadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update_heartbeat'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'validateIfWorkspaceEmpty'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'verify_is_leader'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Checking existence of Mutation Log Table'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Checks if KIEVTRANSACTIONKEYS table is empty'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Checks if KIEVTRANSACTIONS table is empty'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch partition interval for KT'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch partition interval for KTK'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Insert dynamic config'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'createProxyUser'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'createSequence'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'deregister_host'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'dropAutoSequenceMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'dropBucketFromMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'dropSequenceMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get KievTransactionKeys table indexes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get KievTransactions table indexes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get session count'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'initializeMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isKtPartitioned'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isPartitioned'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'log'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'register_host'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateSchemaVersionInDB'||k_appl_handle_suffix 
    --
    OR  p_sql_text LIKE k_appl_handle_prefix||'getUnownedLeases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getFutureWorks'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getMinorVersionsAtAndAfter'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLeaseDecorators'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getUnownedLeasesByFiFo'||k_appl_handle_suffix 
  THEN RETURN 'BG'; /* Background */
  --
  ELSIF p_sql_text LIKE k_appl_handle_prefix||'Ignore'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateKievPdbs'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getJDBCSuffix'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'MV_REFRESH'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'null'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'selectColumnsForTable'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'selectDatastoreMd'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'SQL Analyze('||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'validateDataStoreId'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countSequenceInstances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'iod-telemetry'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert snapshot metadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE CHR(37)||k_appl_handle_prefix||'OPT_DYN_SAMP'||k_appl_handle_suffix 
  THEN RETURN 'IG'; /* Ignore */
  --
  ELSIF p_command_name IN ('INSERT', 'UPDATE')
  THEN RETURN 'TP'; /* Transaction Processing */
  --
  ELSIF p_command_name = 'DELETE'
  THEN RETURN 'BG'; /* Background */
  --
  ELSIF p_command_name = 'SELECT'
  THEN RETURN 'RO'; /* Read Only */
  --
  ELSE RETURN 'UN'; /* Unknown */
  END IF;
END application_category;
/****************************************************************************************/
FUNCTION get_sql_text (
  p_sql_id          IN VARCHAR2,
  p_con_id          IN NUMBER DEFAULT NULL,
  p_plan_hash_value IN NUMBER DEFAULT NULL,
  p_child_number    IN NUMBER DEFAULT NULL
)
RETURN VARCHAR2 DETERMINISTIC
IS
  l_return VARCHAR2(4000);
BEGIN
  IF COALESCE(p_sql_id, 'null') = 'null' THEN RETURN 'null'; END IF;
  SELECT COALESCE(regexp_replace(replace(replace(replace(substr(sql_text,1,1000),chr(39)),'\','\\'),'"','\"'),'[^[:print:]]'), 'null') AS sql_text
    INTO l_return
    FROM v$sql
   WHERE sql_id = p_sql_id
     AND con_id = COALESCE(p_con_id, con_id)
     AND plan_hash_value = COALESCE(p_plan_hash_value, plan_hash_value)
     AND child_number = COALESCE(p_child_number, child_number)
  ORDER BY
         CASE is_obsolete WHEN 'N' THEN 1 ELSE 2 END,
         last_active_time DESC
  FETCH FIRST 1 ROW ONLY;
  RETURN l_return;
EXCEPTION
  WHEN NO_DATA_FOUND THEN
    RETURN 'null';
END get_sql_text;
/****************************************************************************************/
FUNCTION get_parsing_schema_name (
  p_sql_id          IN VARCHAR2,
  p_con_id          IN NUMBER DEFAULT NULL,
  p_plan_hash_value IN NUMBER DEFAULT NULL,
  p_child_number    IN NUMBER DEFAULT NULL
)
RETURN VARCHAR2 DETERMINISTIC
IS
  l_return VARCHAR2(4000);
BEGIN
  IF COALESCE(p_sql_id, 'null') = 'null' THEN RETURN 'null'; END IF;
  SELECT COALESCE(parsing_schema_name, 'null') AS parsing_schema_name
    INTO l_return
    FROM v$sql
   WHERE sql_id = p_sql_id
     AND con_id = COALESCE(p_con_id, con_id)
     AND plan_hash_value = COALESCE(p_plan_hash_value, plan_hash_value)
     AND child_number = COALESCE(p_child_number, child_number)
  ORDER BY
         CASE is_obsolete WHEN 'N' THEN 1 ELSE 2 END,
         last_active_time DESC
  FETCH FIRST 1 ROW ONLY;
  RETURN l_return;
EXCEPTION
  WHEN NO_DATA_FOUND THEN
    RETURN 'null';
END get_parsing_schema_name;
/****************************************************************************************/
v$metricname as (
   /* version 3 */
   select /*+ MATERIALIZE NO_MERGE GATHER_PLAN_STATISTICS MONITOR */
      ''              as project
      ,'oracle_db'    as fleet
      ,host_name      as host
      ,'"lifecycle":"'||
            case
               when
                  (regexp_like(instance_name,'(9[0-9])') and host_name not like '%.r1') or
                  regexp_like(host_name,'(dev|sandbox)') or
                  regexp_like(lower(sys_context('USERENV','DB_DOMAIN')),'(dev|sand)')
                  then 'DEV'
               when
                  (regexp_like(instance_name,'(9[0-9])') and host_name like '%.r1%') or
                  regexp_like(host_name,'(test|tst)') or
                  regexp_like(lower(sys_context('USERENV','DB_DOMAIN')),'(test|tst)')
                  then 'TEST'
               else 'PROD'
            end || '"' ||
            ',"hostclass":"'|| nvl(regexp_substr(host_name,'^([^.]*)-',1,1,NULL,1),'N/A') || '"' ||
            ',"instance_name":"'     || instance_name    || '"' ||
            ',"db_name":"'           || nvl(name,'N/A')             || '"' ||
            ',"db_domain":"'         || nvl(sys_context('USERENV','DB_DOMAIN'),'N/A') || '"' ||
            ',"db_unique_name":"'    || db_unique_name   || '"' ||
            ',"database_role":"'     || database_role    || '"' ||
            ',"fsfo_role":"'||
            case
               when database_role = 'PRIMARY' then 'PRIMARY'
               when database_role = 'PHYSICAL STANDBY' and lower(db_unique_name)  = lower(fs_failover_current_target) then 'STANDBY'
               when database_role = 'PHYSICAL STANDBY' and lower(db_unique_name) != lower(fs_failover_current_target) then 'BYSTANDER'
               else 'NONE'
            end || '"' as metric_dimensions
   from v$database, v$instance
),
/****************************************************************************************/
v$metric_history /* active_session raw */ AS (
SELECT  /*+ MATERIALIZE NO_MERGE GATHER_PLAN_STATISTICS MONITOR */
        h.con_id,
        h.session_id,
        h.session_serial#,
        h.blocking_session,
        h.blocking_session_serial#,
        h.sql_child_number,
        COALESCE(c.name, 'null') AS con_name,
        CASE h.session_state WHEN 'ON CPU' THEN h.session_state ELSE h.wait_class END AS wait_class,
        CASE h.session_state WHEN 'ON CPU' THEN h.session_state ELSE h.event END AS event,
        COALESCE(REGEXP_SUBSTR(h.machine, '((?:[^.]*\.){0}[^.]*)'), 'null') AS machine,
        COALESCE(h.sql_opname, 'null') AS sql_opname,
        COALESCE(h.sql_id, 'null') AS sql_id,
        COALESCE(h.top_level_sql_id, 'null') AS top_level_sql_id,
        COALESCE(TRIM(TO_CHAR(h.sql_plan_hash_value)), 'null') AS plan_hash_value,
        COALESCE(h.module, 'null') AS module,
        COALESCE(h.program, 'null') AS program,
        COALESCE(CASE WHEN h.wait_class IN ('Application', 'Concurrency', 'User I/O') AND h.current_obj# IS NOT NULL THEN TRIM(TO_CHAR(h.current_obj#)) END, 'null') AS current_obj#,
        SUBSTR(get_sql_text(h.sql_id, h.con_id, h.sql_plan_hash_value, h.sql_child_number), 1, 500) AS sql_text,
        application_category(get_sql_text(h.sql_id, h.con_id, h.sql_plan_hash_value, h.sql_child_number), 'UNKNOWN') AS sql_type,
        get_parsing_schema_name(h.sql_id, h.con_id, h.sql_plan_hash_value, h.sql_child_number) AS parsing_schema_name,
        COUNT(*) AS samples
  FROM  v$active_session_history h,
        v$containers c
 WHERE h.sample_time > SYSDATE - (1/24/60)
   AND c.con_id = h.con_id
   AND ROWNUM >= 1
GROUP BY
        h.con_id,
        h.session_id,
        h.session_serial#,
        h.blocking_session,
        h.blocking_session_serial#,
        h.sql_id,
        h.sql_plan_hash_value,
        h.sql_child_number,
        COALESCE(c.name, 'null') /*AS con_name*/,
        CASE h.session_state WHEN 'ON CPU' THEN h.session_state ELSE h.wait_class END /*AS wait_class*/,
        CASE h.session_state WHEN 'ON CPU' THEN h.session_state ELSE h.event END /*AS event*/,
        COALESCE(REGEXP_SUBSTR(h.machine, '((?:[^.]*\.){0}[^.]*)'), 'null') /*AS machine*/,
        COALESCE(h.sql_opname, 'null') /*AS sql_opname*/,
        COALESCE(h.sql_id, 'null') /*AS sql_id*/,
        COALESCE(h.top_level_sql_id, 'null') /*AS top_level_sql_id*/,
        COALESCE(TRIM(TO_CHAR(h.sql_plan_hash_value)), 'null') /*AS plan_hash_value*/,
        COALESCE(h.module, 'null') /*AS module*/,
        COALESCE(h.program, 'null') /*AS program*/,
        COALESCE(h.sql_opname, 'null') /*AS sql_opname*/,
        COALESCE(CASE WHEN h.wait_class IN ('Application', 'Concurrency', 'User I/O') AND h.current_obj# IS NOT NULL THEN TRIM(TO_CHAR(h.current_obj#)) END, 'null') /*AS current_obj#*/
),
/****************************************************************************************/
v$metric_group /* active_session */ AS (
SELECT  /*+ MATERIALIZE NO_MERGE GATHER_PLAN_STATISTICS MONITOR */
        h.con_id,
        h.con_name,
        h.wait_class,
        h.event,
        h.machine,
        COALESCE(REGEXP_SUBSTR(f.machine, '((?:[^.]*\.){0}[^.]*)'), 'null') AS blocking_machine,
        h.sql_id,
        h.parsing_schema_name,
        h.sql_text,
        h.sql_type,
        h.sql_opname,
        h.top_level_sql_id,
        h.plan_hash_value,
        h.module,
        h.program,
        h.current_obj#,
        NVL(SUM(h.samples), 0) AS samples,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name), 0) AS samples_con_name,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.wait_class), 0) AS samples_wait_class,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.wait_class, h.event), 0) AS samples_event,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.machine), 0) AS samples_machine,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, COALESCE(REGEXP_SUBSTR(f.machine, '((?:[^.]*\.){0}[^.]*)'), 'null') /*AS blocking_machine*/), 0) AS samples_blocking_machine,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.sql_type, h.sql_opname, h.sql_id), 0) AS samples_sql_id,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.sql_type, h.wait_class, h.sql_opname, h.sql_id), 0) AS samples_wait_class_sql_id,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.parsing_schema_name), 0) AS samples_parsing_schema_name,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.sql_type), 0) AS samples_sql_type,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.sql_opname), 0) AS samples_sql_opname,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.top_level_sql_id), 0) AS samples_top_level_sql_id,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.sql_type, h.sql_opname, h.sql_id, h.plan_hash_value), 0) AS samples_plan_hash_value,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.module), 0) AS samples_module,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.program), 0) AS samples_program,
        NVL(SUM(SUM(h.samples)) OVER (PARTITION BY h.con_name, h.current_obj#), 0) AS samples_current_obj#
  FROM  v$metric_history /* active_session raw */ h,
        v$session b,
        v$session f
--  WHERE s.con_id(+) = h.con_id
--    AND s.sid(+) = h.session_id 
--    AND s.serial#(+) = h.session_serial#
--    AND f.con_id(+) = s.con_id
--    AND f.sid(+) = s.final_blocking_session
--    AND f.final_blocking_session_status(+) = 'VALID'
 WHERE 1 = 1
  --  AND b.con_id(+) = h.con_id -- blocking session could be from cdb$root (e.g.: lgwr)
   AND b.sid(+) = h.blocking_session 
   AND b.serial#(+) = h.blocking_session_serial#
  --  AND f.con_id(+) = b.con_id -- final blocking session could be from cdb$root (e.g.: lgwr)
   AND f.sid(+) = b.final_blocking_session
   AND f.final_blocking_session_status(+) = 'VALID'
   AND ROWNUM >= 1
GROUP BY
        h.con_id,
        h.con_name,
        h.wait_class,
        h.event,
        h.machine,
        COALESCE(REGEXP_SUBSTR(f.machine, '((?:[^.]*\.){0}[^.]*)'), 'null') /*AS blocking_machine*/,
        h.sql_id,
        h.parsing_schema_name,
        h.sql_text,
        h.sql_type,
        h.sql_opname,
        h.top_level_sql_id,
        h.plan_hash_value,
        h.module,
        h.program,
        h.current_obj#
),
/****************************************************************************************/
v$metric AS (
SELECT  'ActiveSession.ConName.AvgSessions'                             AS metric_name,
        ',"con_name":"'||con_name||'"'                                  AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_con_name)                                           AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_con_name >= 6 OR con_name = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.ConName.AvgSessions'                             AS metric_name,
        ',"con_name":"others"'                                          AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_con_name)                                           AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_con_name >= 6 OR con_name = 'null')
UNION
SELECT  'ActiveSession.WaitClass.AvgSessions'                           AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"wait_class":"'||wait_class||'"'                              AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_wait_class)                                         AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_wait_class >= 6 OR wait_class = 'null')
 GROUP BY
        con_name,
        wait_class
UNION
SELECT  'ActiveSession.WaitClass.AvgSessions'                           AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"wait_class":"others"'                                        AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_wait_class)                                         AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_wait_class >= 6 OR wait_class = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.Event.AvgSessions'                               AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"wait_class":"'||wait_class||'"'||
        ',"event":"'||event||'"'                                        AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_event)                                              AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_event >= 6 OR event = 'null')
 GROUP BY
        con_name,
        wait_class,
        event
UNION
SELECT  'ActiveSession.Event.AvgSessions'                               AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"wait_class":"others"'||
        ',"event":"others"'                                             AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_event)                                              AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_event >= 6 OR event = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.Machine.AvgSessions'                             AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"machine":"'||machine||'"'                                    AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_machine)                                            AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_machine >= 6 OR machine = 'null')
 GROUP BY
        con_name,
        machine
UNION
SELECT  'ActiveSession.Machine.AvgSessions'                             AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"machine":"others"'                                           AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_machine)                                            AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_machine >= 6 OR machine = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.BlockingMachine.AvgSessions'                     AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"blocking_machine":"'||blocking_machine||'"'                  AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_blocking_machine)                                   AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_blocking_machine >= 6 OR blocking_machine = 'null')
 GROUP BY
        con_name,
        blocking_machine
UNION
SELECT  'ActiveSession.BlockingMachine.AvgSessions'                     AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"blocking_machine":"others"'                                  AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_blocking_machine)                                   AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_blocking_machine >= 6 OR blocking_machine = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.SqlId.AvgSessions'                               AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"'||sql_type||'"'||
        ',"sql_opname":"'||sql_opname||'"'||
        ',"sql_id":"'||sql_id||'"'||
        ',"sql_text":"'||MAX(sql_text)||'"'                             AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_sql_id)                                             AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_sql_id >= 6 OR sql_id = 'null')
 GROUP BY
        con_name,
        sql_type,
        sql_opname,
        sql_id
UNION
SELECT  'ActiveSession.SqlId.AvgSessions'                               AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"UN"'||
        ',"sql_opname":"others"'||
        ',"sql_id":"others"'||
        ',"sql_text":"others"'                                          AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_sql_id)                                             AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_sql_id >= 6 OR sql_id = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.WaitClassSqlId.AvgSessions'                      AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"'||sql_type||'"'||
        ',"wait_class":"'||wait_class||'"'||
        ',"sql_opname":"'||sql_opname||'"'||
        ',"sql_id":"'||sql_id||'"'||
        ',"sql_text":"'||MAX(sql_text)||'"'                             AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_wait_class_sql_id)                                  AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_wait_class_sql_id >= 6 OR sql_id = 'null')
 GROUP BY
        con_name,
        sql_type,
        wait_class,
        sql_opname,
        sql_id
UNION
SELECT  'ActiveSession.WaitClassSqlId.AvgSessions'                      AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"UN"'||
        ',"wait_class":"others"'||
        ',"sql_opname":"others"'||
        ',"sql_id":"others"'||
        ',"sql_text":"others"'                                          AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_wait_class_sql_id)                                  AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_wait_class_sql_id >= 6 OR sql_id = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.ParsingSchemaName.AvgSessions'                   AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"parsing_schema_name":"'||parsing_schema_name||'"'            AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_parsing_schema_name)                                AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_parsing_schema_name >= 6 OR parsing_schema_name = 'null')
 GROUP BY
        con_name,
        parsing_schema_name
UNION
SELECT  'ActiveSession.ParsingSchemaName.AvgSessions'                   AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"parsing_schema_name":"others"'                               AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_parsing_schema_name)                                AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_parsing_schema_name >= 6 OR parsing_schema_name = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.SqlType.AvgSessions'                             AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"'||sql_type||'"'                                  AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_sql_type)                                           AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_sql_type >= 6 OR sql_type = 'null')
 GROUP BY
        con_name,
        sql_type
UNION
SELECT  'ActiveSession.SqlType.AvgSessions'                             AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"others"'                                          AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_sql_type)                                           AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_sql_type >= 6 OR sql_type = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.SqlOpName.AvgSessions'                           AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_opname":"'||sql_opname||'"'                              AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_sql_opname)                                         AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_sql_opname >= 6 OR sql_opname = 'null')
 GROUP BY
        con_name,
        sql_opname
UNION
SELECT  'ActiveSession.SqlOpName.AvgSessions'                           AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_opname":"others"'                                        AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_sql_opname)                                         AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_sql_opname >= 6 OR sql_opname = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.TopLevelSqlId.AvgSessions'                       AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"top_level_sql_id":"'||top_level_sql_id||'"'                  AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_top_level_sql_id)                                   AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_top_level_sql_id >= 6 OR top_level_sql_id = 'null')
 GROUP BY
        con_name,
        top_level_sql_id
UNION
SELECT  'ActiveSession.TopLevelSqlId.AvgSessions'                       AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"top_level_sql_id":"others"'                                  AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_top_level_sql_id)                                   AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_top_level_sql_id >= 6 OR top_level_sql_id = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.PlanHashValue.AvgSessions'                       AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"'||sql_type||'"'||
        ',"sql_opname":"'||sql_opname||'"'||
        ',"sql_id":"'||sql_id||'"'||
        ',"plan_hash_value":"'||plan_hash_value||'"'||
        ',"sql_text":"'||MAX(sql_text)||'"'                             AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_plan_hash_value)                                    AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_plan_hash_value >= 6 OR plan_hash_value = 'null')
 GROUP BY
        con_name,
        sql_type,
        sql_opname,
        sql_id,
        plan_hash_value
UNION
SELECT  'ActiveSession.PlanHashValue.AvgSessions'                       AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"sql_type":"UN"'||
        ',"sql_opname":"others"'||
        ',"sql_id":"others"'||
        ',"plan_hash_value":"others"'||
        ',"sql_text":"others"'                                          AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_plan_hash_value)                                    AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_plan_hash_value >= 6 OR plan_hash_value = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.Module.AvgSessions'                              AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"module":"'||module||'"'                                      AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_module)                                             AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_module >= 6 OR module = 'null')
 GROUP BY
        con_name,
        module
UNION
SELECT  'ActiveSession.Module.AvgSessions'                              AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"module":"others"'                                            AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_module)                                             AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_module >= 6 OR module = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.Program.AvgSessions'                             AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"program":"'||program||'"'                                    AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_program)                                            AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_program >= 6 OR program = 'null')
 GROUP BY
        con_name,
        program
UNION
SELECT  'ActiveSession.Program.AvgSessions'                             AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"program":"others"'                                           AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_program)                                            AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_program >= 6 OR program = 'null')
 GROUP BY
        con_name
UNION
SELECT  'ActiveSession.CurrentObjNum.AvgSessions'                       AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"current_obj_num":"'||current_obj#||'"'                          AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_current_obj#)                                       AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  (samples_current_obj# >= 6 OR current_obj# = 'null')
 GROUP BY
        con_name,
        current_obj#
UNION
SELECT  'ActiveSession.CurrentObjNum.AvgSessions'                       AS metric_name,
        ',"con_name":"'||con_name||'"'||
        ',"current_obj_num":"others"'                                      AS metric_dimensions,
        ROUND(SUM(samples) / 60, 3)                                     AS metric_value,
        SUM(samples)                                                    AS sum_samples,
        MAX(samples_current_obj#)                                       AS max_samples
  FROM  v$metric_group /* active_session */
 WHERE  NOT (samples_current_obj# >= 6 OR current_obj# = 'null')
 GROUP BY
        con_name
)
/****************************************************************************************/
select
    v$metricname.project                        as project
    ,v$metricname.fleet                         as fleet
    ,v$metricname.host                          as host
    ,v$metric.metric_name                       as metric_name
    ,v$metric.metric_value                      as metric_value
    ,'{'||v$metricname.metric_dimensions||v$metric.metric_dimensions||'}' as metric_dimensions
from v$metricname,v$metric
where v$metric.metric_value >= 0
