Rem $Header: dbgendev/src/langdata/plsql/search/search_pkg.pks /main/26 2025/08/13 01:29:21 jiangnhu Exp $
Rem
Rem $Header: dbgendev/src/langdata/plsql/search/search_pkg.pks /main/26 2025/08/13 01:29:21 jiangnhu Exp $
Rem
Rem search_pkg.pks
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      search_pkg.pks - Package specification for handling user queries and
Rem                       managing search records
Rem
Rem    DESCRIPTION
Rem      Package for handling user queries and managing search records, 
Rem      including retrieval, deletion, and pagination, with role-based access 
Rem      control, error handling, and logging.
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/search/search_pkg.pks
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    dadoshi     07/31/25 - JIRA_DBAI1149: Add optional parameter,
Rem                           p_query_vector to search_from_query
Rem    dadoshi     07/31/25 - JIRA_DBAI1147: Add get_top_report_match
Rem    deveverm    07/28/25 - DBAI-1011: Refactored reranking to rerank_texts
Rem    ruohli      07/01/25 - DBAI-884: Modify 
Rem                           update_non_default_filter_values_stats_from_search
Rem                           into update_filter_values_stats_from_search to 
Rem                           support both default and non-default filter values
Rem    jiangnhu    06/30/25 - DBAI-1003: Pass both original query and augmented
Rem                           query to process_filters
Rem    saloshah    06/18/25 - DBAI-745-757: Added non_default filter values
Rem                           metrics and ner metrics
Rem    anisbans    06/13/25 - Added the API update_report_search_count
Rem    dadoshi     05/15/25 - Remove get_filter_values() procedure
Rem    jiangnhu    04/12/25 - DBAI-739: Add plot_search_vectors
Rem    deveverm    04/10/25 - DBAI-723: Refactored all functions calling NER to
Rem                           named_entities_pkg
Rem    dadoshi     04/07/25 - Add usage of LANG_DATA_VALUE_VECTOR_TABLE_K for 
Rem                           similarity search in process_filters
Rem    dadoshi     04/02/25 - JIRA_DBAI522: Add support for multiple enumerable
Rem                           filters
Rem    jiangnhu    03/26/25 - DBAI-692: make search_id OUT parameter; remove
Rem                           similarity_k
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/07/25 - DBAI-545: Make search_type a config
Rem    dadoshi     03/03/25 - JIRA_DBAI596: Add process_filters() and
Rem                           get_entities_using_ner() to populate filter
Rem                           values of reports/drilldowns
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    01/29/25 - DBAI-505: Implement centroid version of
Rem                           search_from_query procedure
Rem    jiangnhu    01/16/25 - DBAI-508: Add get_report_matching_text,
Rem                           get_drilldown_matching_text, update
Rem                           search_from_query to add content to
Rem                           report_matches
Rem    jiangnhu    12/19/24 - JIRA_DBAI-451: Move match_substring_with_context
Rem                           to utils pkg
Rem    jiangnhu    11/26/24 - Fix error handling of delete_user_search_record
Rem    jiangnhu    11/22/24 - JIRA_DBAI-425: Implement search_from_query and
Rem                         - helper procedures and functions
Rem    jiangnhu    10/22/24 - Format code
Rem    jiangnhu    10/18/24 - Fix text wrapping
Rem    jiangnhu    10/17/24 - Modify header
Rem    pryarla     10/16/24 - Created

CREATE OR REPLACE PACKAGE lang_data_search_pkg IS
    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_search_pkg
    Description: This package contains procedures related to searching and 
                 retrieving user search records from the langdata$searchrecords
                 table. The procedures are designed to handle user search
                 records and related operations.
    ----------------------------------------------------------------------------
    Author: Jiangnan Huang
    Created: 09/23/2024
    ----------------------------------------------------------------------------
    Public Procedures:
    ----------------------------------------------------------------------------
    PROCEDURE get_top_report_match
    ----------------------------------------------------------------------------
    Description:
        This procedures retrieves the top report match given the search ID.
    
    Parameters: 
        - p_search_id (IN VARCHAR2):
            Unique ID of the search record.
    
    Exceptions:
        NONE
    ----------------------------------------------------------------------------
    PROCEDURE get_user_search_record
    ---------------------------------
    Description:
        This procedure retrieves the search record for a specific user based on
        the given search record ID. It returns various details about the user's
        search such as the search query text, report matches, feedback ratings,
        and more.
        
        The procedure implements role-based access control:
        - If the user does not have the `LANG_DATA_APP_EXPERT` role, the search
          is restricted to records owned by the current session user.
        - If the user has the `LANG_DATA_APP_EXPERT` role, no such restriction
          is applied, and the user can retrieve any record by the provided ID.
        
    Parameters:
        - p_id (IN VARCHAR2):
            The ID of the user search record that needs to be fetched.
        - p_query_text (OUT VARCHAR2):
            The search query text provided by the user.
        - p_report_matches (OUT JSON):
            The list of report matches (in JSON format) related to the search.
        - p_feedback_rating (OUT BOOLEAN):
            The feedback rating provided by the user.
        - p_feedback_comments (OUT VARCHAR2):
            The comments or feedback from the user related to the search.
        - p_required_feedback_action (OUT VARCHAR2):
            The required action based on the user’s feedback (if applicable).
        - p_feedback_action_priority (OUT VARCHAR2):
            The priority level for the feedback action, based on user feedback.
        - p_expected_report_id (OUT VARCHAR2):
            The ID of the expected report the user was searching for.
        - p_expected_drilldown_id (OUT VARCHAR2):
            The ID of the expected drilldown associated with the report.
        - p_search_type (OUT VARCHAR2):
            The type of search the user performed ('hierarchical' or 'flat').
        - p_augmented_query_text (OUT VARCHAR2):
            The augmented version of the search query text.
    
    Exceptions:
        - c_resource_not_found (-20004):
            Raised if no search record is found for the given ID. A custom error
            with code -20004 is raised using the `lang_data_errors_pkg` package
            with the message "Resource not found".
    
    PROCEDURE get_all_user_search_records
    -------------------------------------
    Description:
        This procedure retrieves a paginated list of user search records based
        on optional filtering criteria such as feedback rating, required
        feedback action, feedback action priority, and pagination cursor.
        
        The procedure implements role-based access control:
        - If the user does not have the `LANG_DATA_APP_EXPERT` role, the search
          is restricted to records owned by the current session user.
        - If the user has the `LANG_DATA_APP_EXPERT` role, no such restriction
          is applied, and the user can retrieve any records.
        
    Parameters:
        - p_feedback_rating (IN BOOLEAN):
            Optional filter for the feedback rating.
        - p_required_feedback_action (IN VARCHAR2):
            Optional filter for the required feedback action.
        - p_feedback_action_priority (IN VARCHAR2):
            Optional filter for the feedback action priority.
        - p_cursor (IN OUT VARCHAR2):
            Optional serialized pagination cursor containing created_at and id
            in the format created_at|id. For the first page, the cursor is NULL.
            For later pages, the cursor is the created_at|id of the first row
            of current page.
        - p_limit (IN NUMBER):
            The number of records to retrieve per page (for pagination).
        - p_records (OUT SYS_REFCURSOR):
            A cursor that returns the list of user search records.

    Exceptions:
        - None.

    PROCEDURE delete_user_search_record
    -------------------------------------
    Description:
        This procedure deletes a user search record based on the provided search
        record ID. 
        The procedure implements role-based access control:
        - If the user does not have the `LANG_DATA_APP_EXPERT` role, deletion is
          not allowed.
        - If the user has the `LANG_DATA_APP_EXPERT` role, the record is deleted
          without restriction.

    Parameters:
        - p_id (IN VARCHAR2): 
            The ID of the user search record that needs to be deleted.
        
    Exceptions:
        - c_unauthorized_code (-20000):
            Raised when the user does not have the required role 
            (LANG_DATA_APP_EXPERT) to delete the search record. This triggers a 
            custom error with code -20002 using the lang_data_errors_pkg 
            package.
        - c_resource_not_found (-20004): 
            Raised if no search record is found for the given ID. A custom error
            with code -20004 is raised using the `lang_data_errors_pkg` package
            with the message "Resource not found".
    ----------------------------------------------------------------------
    FUNCTION validate_match
    -----------------------
    Description:
        Validates a match between a user query and an identified value using 
        fuzzy matching and a similarity score threshold.

    Parameters:
        - user_query (IN VARCHAR2): The user query text.
        - identified_value (IN VARCHAR2): The value to match against.
        - threshold (IN NUMBER DEFAULT 80): The similarity score threshold.

    Returns:
        - BOOLEAN: `TRUE` if the match meets the threshold, `FALSE` otherwise.

    Exceptions:
        - None.
    ----------------------------------------------------------------------
    PROCEDURE process_filters
    ----------------------------------------------------------------------
    Description:
        Processes the filters of all the match documents, provided as a list,
        and returns the filter values of the respective reports and drilldowns, 
        along with the list of the number of filter values per document used for
        populating the overall search results in search_from_query.
    
    Parameters:
        - p_original_query (IN VARCHAR2): The original user query text.
        - p_augmented_query (IN VARCHAR2): The augmented user query text.
        - p_match_document_arr (IN JSON_ARRAY_T): The list of match documents
                                                  of which the filters' values
                                                  have to be computed.
        - p_report_filter_values (OUT JSON_ARRAY_T): List of filter values
                                                     corresponding to reports.
        - p_drilldown_filter_values (OUT JSON_ARRAY_T): List of filter values
                                                        corresponding to 
                                                        drilldowns.
        - p_num_filters (OUT SYS.ODCIVARCHAR2LIST): List of values representing
                                                    the number of filters for 
                                                    every match document in
                                                    p_match_document_arr. 
    
    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE re_rank_similarity_search_reports
    ----------------------------------------------------------------------------
    Description: 
        Re-ranks the top-k reports fetched as part of the search based
        on the search query, with the help of the embedded python function, 
        re_rank_search_results.

    Parameters:
        - p_query_text (IN VARCHAR2): The query text.
        - p_ranked_results (IN SYS_REFCURSOR): The similarity search results.
        - p_re_ranked_results (OUT SYS_REFCURSOR): Re-ranked search results.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE get_top_k_closest_report_and_description_similarity_search
    ----------------------------------------------------------------------------
    Description:
        Performs a similarity search for the top `k` closest reports and their 
        descriptions based on a query vector.

    Parameters:
        - p_query (IN VARCHAR2): The query text.
        - p_k (IN NUMBER DEFAULT 5): Number of top results to fetch.
        - p_domain_id (IN VARCHAR2): Optional parameter to filter based on 
                                     domain.
        - p_results (OUT SYS_REFCURSOR): Cursor containing the top `k` results.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE re_rank_similarity_search_drilldowns
    ----------------------------------------------------------------------------
    Description: 
        Re-ranks the top-k drilldowns fetched as part of the search based
        on the search query, with the help of the embedded python function, 
        re_rank_search_results.

    Parameters:
        - p_query_text (IN VARCHAR2): The query text.
        - p_ranked_results (IN SYS_REFCURSOR): The similarity search results.
        - p_re_ranked_results (OUT SYS_REFCURSOR): Re-ranked search results.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE get_top_k_closest_drilldown_and_description_similarity_search
    ----------------------------------------------------------------------------
    Description:
        Performs a similarity search for the top `k` closest drilldowns and 
        their descriptions based on a query vector.

    Parameters:
        - p_query (IN VARCHAR2): The query text.
        - p_matched_report_id (IN VARCHAR2): Matched report ID for drilldown
                                             filtering.
        - p_k (IN NUMBER DEFAULT 5): Number of top results to fetch.
        - p_domain_id (IN VARCHAR2): Optional parameter to filter based on 
                                     domain.
        - p_results (OUT SYS_REFCURSOR): Cursor containing the top `k` results.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE re_rank_similarity_search_documents
    ----------------------------------------------------------------------------
    Description: 
        Re-ranks the top-k documents fetched as part of the search based
        on the search query, with the help of the embedded python function, 
        re_rank_search_results.

    Parameters:
        - p_query_text (IN VARCHAR2): The query text.
        - p_ranked_results (IN SYS_REFCURSOR): The similarity search results.
        - p_re_ranked_results (OUT SYS_REFCURSOR): Re-ranked search results.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE get_top_k_closest_document_and_description_similarity_search_flat
    ----------------------------------------------------------------------------
    Description:
        Performs a flat similarity search for the top `k` closest documents 
        (reports or drilldowns) and their descriptions based on a query vector.

    Parameters:
        - p_query (IN VARCHAR2): The query text.
        - p_k (IN NUMBER DEFAULT 5): Number of top results to fetch.
        - p_domain_id (IN VARCHAR2): Optional parameter to filter based on 
                                     domain.
        - p_results (OUT SYS_REFCURSOR): Cursor containing the top `k` results.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE search_from_query
    ----------------------------------------------------------------------------
    Description:
        Combines hierarchical or flat similarity searches for reports and 
        drilldowns, computes filter values, and inserts the search record into 
        the database.

    Parameters:
        - p_query (IN VARCHAR2): The user query text.
        - p_domain (IN VARCHAR2): Optional parameter to filter based on domain.
        - p_query_vector (IN VECTOR DEFAULT NULL): Vector Embedding of query.
        - p_search_id (OUT VARCHAR2): The unique ID for the search record
                                     (optional).

    Exceptions:
        - c_max_text_length_exceeded (-20003):
            Raised if the length of the query exceeds the maximum allowed length
            of 2000 characters. 
        - c_invalid_parameters_code (-20002):
            Raised when p_query is null
    ----------------------------------------------------------------------------
    PROCEDURE get_report_matching_text
    ----------------------------------------------------------------------------
    Description:
        Fetches the `text` and `enhanced_text` for a given report matching the 
        specified ID. It searches across multiple sources in a prioritized
        order: `langdata$reportdescriptions`, `langdata$samplequeries`, and 
        `langdata$searchrecords`. If no match is found, the output parameters 
        are set to `NULL`.

    Parameters:
        - p_match_id (IN VARCHAR2): The unique identifier used to find the 
                                    corresponding report.
        - p_matched_text (OUT VARCHAR2): The matched report `text` corresponding
                                        to the given ID.
        - p_enhanced_text (OUT VARCHAR2): The enhanced report `text` if 
                                          available, corresponding 
                                          to the given ID.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------
    PROCEDURE get_drilldown_matching_text
    ----------------------------------------------------------------------------
    Description:
        Fetches the `text` and `enhanced_text` for a given drilldown matching
        the specified ID. It searches across multiple sources in a 
        prioritized order:`langdata$drilldowndescriptions`, 
        `langdata$samplequeries`, and `langdata$searchrecords`. 
        If no match is found, the output parameters are set to `NULL`.

    Parameters:
        - p_match_id (IN VARCHAR2): The unique identifier used to find the 
                                    corresponding drilldown.
        - p_matched_text (OUT VARCHAR2): The matched drilldown `text`
                                        corresponding to the given ID.
        - p_enhanced_text (OUT VARCHAR2): The enhanced drilldown `text` if
                                        available, corresponding to the given ID

    Exceptions:
        - None.

    ----------------------------------------------------------------------------
    PROCEDURE update_json_analytics
    ----------------------------------------------------------------------------
    Description:
        This procedure is used to update the analytics_data for 
        each report/drilldown based on specific search record 
        in the langdata$reports and langdata$drilldowndocuments. 
        It is internally used by update_rank_stats_from_search
        and update_non_default_filter_values_stats_from_search.

    Parameters:
        - p_table   (IN VARCHAR2) : Table name for which analytics_data 
                                    is to be updated
        - p_id_col  (IN VARCHAR2) : Column name of the primary key
        - p_id_val  (IN VARCHAR2) : ID value to identify the row to be updated
        - p_key     (IN VARCHAR2) : JSON key within `analytics_data` to be 
                                    incremented/decremented
        - p_delta   (IN NUMBER)   : Value to add to the existing key

    Exceptions:
        - None.
    
    ----------------------------------------------------------------------------
    PROCEDURE update_rank_stats_from_search
    ----------------------------------------------------------------------------
    Description:
        To update the ranking statistics (top1_count, top3_count, top5_count) 
        in the langdata$reports and langdata$drilldowndocuments tables based on 
        a specific search record's report_matches JSON field.

    Parameters:
        - p_search_id (IN VARCHAR2): The unique identifier used to find the 
                                    corresponding search record.

    Exceptions:
        - None.

    ----------------------------------------------------------------------------
    PROCEDURE update_ner_stats_from_search
    ----------------------------------------------------------------------------
    Description:
        To update the total no of recognized entities in langdata$ner_metrics
    Parameters:
        - p_search_id (IN VARCHAR2): The unique identifier used to find the 
                                    corresponding search record.
    Exceptions:
        - None.

    ----------------------------------------------------------------------------
    PROCEDURE update_filter_values_stats_from_search (
    ----------------------------------------------------------------------------
    Description:
        To update the non default or default filter values populated count 
        in the langdata$reports and langdata$drilldowndocuments tables based on 
        a specific search record's report_matches JSON field.

    Parameters:
        - p_search_id (IN VARCHAR2): The unique identifier used to find the 
                                    corresponding search record.
        - p_filter_type (IN VARCHAR2): 'default' or 'non_default' that determine
                                        if update default or non default filter
                                        values

    Exceptions:
        - None.

    ----------------------------------------------------------------------------
    PROCEDURE update_stats_from_search
    ----------------------------------------------------------------------------
    Description:
        This procedure updates all the statistics by internally calling
        update_rank_stats_from_search, update_ner_stats_from_search, 
        update_non_default_filter_values_stats_from_search using the 
        a specific search record's report_matches JSON field.

    Parameters:
        - p_search_id (IN VARCHAR2): The unique identifier used to find the 
                                    corresponding search record.

    Exceptions:
        - None.

    ----------------------------------------------------------------------------
    PROCEDURE update_report_search_count
    ----------------------------------------------------------------------------
    Description:
        Updates the 'search_count' and 'last_searched_at' fields in the 
        langdata$reports table for each report matched in the 
        'report_matches' JSON array of a given search record.

        The procedure fetches the report titles from the JSON array associated 
        with the search ID, increments the search count for each corresponding 
        report, and updates the timestamp to the current time.

    Parameters:
        - p_search_id (IN VARCHAR2): The unique identifier used to find the 
                                    corresponding search record.

    Exceptions:
        - None.
    ----------------------------------------------------------------------------       
    PROCEDURE plot_search_vectors
    ----------------------------------------------------------------------------
    Description:
    Generates and stores a visual representation of vector similarities 
    associated with a search record using a registered Python function 
    via `pyqTableEval`.

    This procedure performs the following operations:
    - Fetches the augmented query text, expected report ID, and matched
      report descriptions from `langdata$searchrecords`
    - Prepares textual input for visualization by populating the table 
      `langdata$plot_input`, including:
        - The augmented query text as the actual query
        - Enhanced report descriptions
        - Sample and user queries related to the matched reports
        - Highlight flags for relevant matches
    - Invokes the registered Python script `plot_search_vectors`, which:
        - Embeds the input texts using the SentenceTransformer model
        - Reduces the vector dimensions using PCA
        - Generates a scatter plot to visualize report-query relationships
        - Returns the plot image encoded as a BLOB
    - Updates the `plot_image` column in `langdata$searchrecords` 
      with the generated plot.

    Parameters:
        - p_search_id (IN VARCHAR2): 
            The ID of the search record to visualize.

    Returns:
        - None (The Python function is expected to return a single-row 
          table containing the image blob for internal processing.)

    Exceptions:
        - None.
    */

    FUNCTION get_top_report_match (
        p_search_id                     IN VARCHAR2
    ) RETURN JSON_OBJECT_T;

    -- Public procedure declaration for getting user search record
    PROCEDURE get_user_search_record (
        p_id IN VARCHAR2,
        p_query_text OUT VARCHAR2,
        p_report_matches OUT JSON,
        p_feedback_rating OUT BOOLEAN,
        p_feedback_comments OUT VARCHAR2,
        p_required_feedback_action OUT VARCHAR2,
        p_feedback_action_priority OUT VARCHAR2,
        p_expected_report_id OUT VARCHAR2,
        p_expected_drilldown_id OUT VARCHAR2,
        p_search_type OUT VARCHAR2,
        p_augmented_query_text OUT VARCHAR2
    );

    -- Public procedure declaration for retrieving multiple user search records
    -- with pagination
    PROCEDURE get_all_user_search_records (
        p_feedback_rating IN BOOLEAN DEFAULT NULL,
        p_required_feedback_action IN VARCHAR2 DEFAULT NULL,
        p_feedback_action_priority IN VARCHAR2 DEFAULT NULL,
        p_cursor IN OUT VARCHAR2,
        p_limit IN NUMBER DEFAULT 10,
        p_records OUT SYS_REFCURSOR
    );

    -- Public procedure declaration for deleting user search record
    PROCEDURE delete_user_search_record (
        p_id IN VARCHAR2
    );

    FUNCTION validate_match(
        user_query       IN VARCHAR2,
        identified_value IN VARCHAR2,
        threshold        IN NUMBER DEFAULT 80
    ) RETURN BOOLEAN;

    PROCEDURE print_filters_temptable;

    PROCEDURE filter_candidates_rerank_call(
        p_query              IN VARCHAR2,
        p_filter_candidates  IN JSON_ARRAY_T,
        p_top_candidate      OUT VARCHAR2,
        p_top_score          OUT NUMBER,
        p_top_index          OUT INTEGER
    );

    PROCEDURE process_filters(
        p_original_query            IN VARCHAR2,
        p_augmented_query           IN VARCHAR2,
        p_match_document_arr        IN JSON_ARRAY_T,
        p_report_filter_values      OUT JSON_ARRAY_T,
        p_drilldown_filter_values   OUT JSON_ARRAY_T,
        p_num_filters               OUT SYS.ODCIVARCHAR2LIST
    );

    PROCEDURE re_rank_similarity_search_reports (
        p_query_text            IN VARCHAR2,
        p_ranked_results        IN SYS_REFCURSOR,
        p_re_ranked_results     OUT SYS_REFCURSOR
    );

    PROCEDURE get_top_k_closest_report_and_description_similarity_search (
        p_query             IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_using_centroid    IN  BOOLEAN DEFAULT FALSE,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    );

    PROCEDURE re_rank_similarity_search_drilldowns (
        p_query_text            IN VARCHAR2,
        p_ranked_results        IN SYS_REFCURSOR,
        p_re_ranked_results     OUT SYS_REFCURSOR
    );

    PROCEDURE get_top_k_closest_drilldown_and_description_similarity_search (
        p_query             IN  VARCHAR2,
        p_matched_report_id IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_using_centroid    IN BOOLEAN DEFAULT FALSE,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    );

    PROCEDURE re_rank_similarity_search_documents (
        p_query_text            IN VARCHAR2,
        p_ranked_results        IN SYS_REFCURSOR,
        p_re_ranked_results     OUT SYS_REFCURSOR
    );

    PROCEDURE get_top_k_closest_document_and_description_similarity_search_flat (
        p_query     IN  VARCHAR2,
        p_k         IN  NUMBER DEFAULT 5,
        p_domain_id IN  VARCHAR2 DEFAULT NULL,
        p_results   OUT SYS_REFCURSOR
    );

    PROCEDURE get_report_matching_text (
        p_match_id IN VARCHAR2,
        p_matched_text OUT VARCHAR2,
        p_enhanced_text OUT VARCHAR2
    );

    PROCEDURE get_drilldown_matching_text (
        p_match_id IN VARCHAR2,
        p_matched_text OUT VARCHAR2,
        p_enhanced_text OUT VARCHAR2
    );

    PROCEDURE search_from_query (
        p_query     IN VARCHAR2,
        p_domain    IN VARCHAR2 DEFAULT NULL,
        p_query_vector      IN VECTOR DEFAULT NULL,
        p_search_id OUT VARCHAR2
    );
    
    PROCEDURE update_json_analytics(
        p_table   IN VARCHAR2,
        p_id_col  IN VARCHAR2,
        p_id_val  IN VARCHAR2,
        p_key     IN VARCHAR2,
        p_delta   IN NUMBER
    );

    PROCEDURE update_rank_stats_from_search (
        p_search_id IN VARCHAR2
    );

    PROCEDURE update_ner_stats_from_search (
        p_search_id IN VARCHAR2
    );

    PROCEDURE update_filter_values_stats_from_search (
        p_search_id IN VARCHAR2,
        p_filter_type   IN VARCHAR2
    );

    PROCEDURE update_stats_from_search (
        p_search_id IN VARCHAR2
    );
    
    PROCEDURE plot_search_vectors(
        p_search_id IN VARCHAR2
    );

    PROCEDURE update_report_search_count(
        p_search_id IN VARCHAR2
    );

    FUNCTION rerank_texts(
        p_rerank_obj_list IN rerank_table
    ) RETURN rerank_table;
END lang_data_search_pkg;
/


