Rem
Rem $Header: dbgendev/src/langdata/plsql/samplequeries/samplequeries_pkg.pkb /main/19 2025/07/24 11:33:10 arevathi Exp $
Rem
Rem samplequeries_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      samplequeries_pkg.pkb - Package body for managing and processing
Rem                              sample queries
Rem
Rem    DESCRIPTION
Rem      This package provides procedures to manage sample queries, including 
Rem      retrieving all sample queries, adding new queries with deduplication, 
Rem      and deleting queries. It supports role-based access control and 
Rem      includes error handling and logging mechanisms.
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/samplequeries/samplequeries_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    07/10/25 - DBAI-1006: Parameterize job class name
Rem    jiangnhu    07/07/25 - DBAI-871: Add function
Rem                           get_sample_query_id_by_report_id_version
Rem    jiangnhu    06/24/25 - DBAI-909: Integrate text augmentation with domain
Rem    deveverm    05/16/25 - DBAI-761: added update_sample_query_status and
Rem                           updated
Rem                           add_report_sample_query/add_drilldown_sample_query
Rem                           to create regression job
Rem    jiangnhu    04/01/25 - DBAI-624: Make embedding model independent
Rem    dadoshi     03/24/25 - Use generate_id() for samplequeries ID
Rem    jiangnhu    03/24/25 - DBAI-551: Use NER to augment sample query
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/03/25 - Update deduplicate_sample_queries
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    01/31/25 - DBAI-511: Update function augment_query to
Rem                           procedure augment_text, add augmented_tokens
Rem    jiangnhu    01/29/25 - DBAI-505: Implement centroid version of
Rem                           search_from_query procedure
Rem    jiangnhu    12/09/24 - JIRA_DBAI-458: Split add_sample_queries
Rem    jiangnhu    12/09/24 - JIRA_DBAI-458: Split get_all_sample_queries
Rem    jiangnhu    11/26/24 - Fix error handling of delete_sample_query,
Rem                           add_sample_queries
Rem    jiangnhu    10/29/24 - JIRA_DBAI-403: Add amend_description to process query
Rem    jiangnhu    10/22/24 - Format code
Rem    jiangnhu    10/18/24 - Fix text wrapping
Rem    jiangnhu    10/17/24 - Modify header
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data_sample_queries_pkg AS

    PROCEDURE get_report_sample_queries (
        p_report_id IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    )
    IS
        -- Variables for query construction
        v_sql_text VARCHAR2(4000);
        v_cur INTEGER;
        v_dummy INTEGER;

    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure get_report_sample_queries'
        );

        -- Check if the user has the required role
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Validate report_id
        IF NOT lang_data_utils_pkg.check_report_exists(p_report_id) THEN
            lang_data_logger_pkg.log_error(
                'Report ID not found: ' || p_report_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Build the query
        v_sql_text := 'SELECT id, query_text, version, report_id, ' ||
                    'drilldown_id, enhanced_query_text, status ' ||
                    'FROM langdata$samplequeries '||
                    'WHERE report_id = :report_id ORDER BY created_at DESC';
        lang_data_logger_pkg.log_info('Executing query: ' || v_sql_text);

        -- Open a dynamic cursor
        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

        -- Bind the report_id variable
        DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);

        -- Execute the cursor and convert to REFCURSOR
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_sample_queries := DBMS_SQL.TO_REFCURSOR(v_cur);

        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_info(
                'No sample queries found for report.'
            );
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An error occurred in get_report_sample_queries: ' ||
                    SQLERRM
                );
                RAISE;
            END IF;
    END get_report_sample_queries;

    PROCEDURE get_drilldown_sample_queries (
        p_drilldown_id IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    )
    IS
        -- Variables for query construction
        v_sql_text VARCHAR2(4000);
        v_cur INTEGER;
        v_dummy INTEGER;

    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure get_drilldown_sample_queries'
        );

        -- Check if the user has the required role
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Validate drilldown_id
        IF NOT lang_data_utils_pkg.check_drilldown_exists(p_drilldown_id) THEN
            lang_data_logger_pkg.log_error(
                'Drilldown ID not found: ' || p_drilldown_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Build the query
        v_sql_text := 'SELECT id, query_text, version, report_id, ' ||
                    'drilldown_id, enhanced_query_text, status ' || 
                    'FROM langdata$samplequeries '||
                    'WHERE drilldown_id = :drilldown_id ' ||
                    'ORDER BY created_at DESC';
        lang_data_logger_pkg.log_info('Executing query: ' || v_sql_text);

        -- Open a dynamic cursor
        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

        -- Bind the drilldown_id variable
        DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_drilldown_id);

        -- Execute the cursor and convert to REFCURSOR
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_sample_queries := DBMS_SQL.TO_REFCURSOR(v_cur);

        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_info(
                'No sample queries found for drilldown.'
            );
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An error occurred in get_drilldown_sample_queries: ' ||
                    SQLERRM
                );
                RAISE;
            END IF;
    END get_drilldown_sample_queries;

    PROCEDURE delete_sample_query(
        p_id IN VARCHAR2
    )
    IS
        v_report_id VARCHAR2(36);
        v_drilldown_id VARCHAR2(36);
    BEGIN
        -- Log the start of the procedure
        lang_data_logger_pkg.log_info(
            'Starting procedure delete_sample_query for ID: ' || p_id
        );

        -- Check if the user has the required role
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        BEGIN
            SELECT report_id, drilldown_id
            INTO v_report_id, v_drilldown_id
            FROM langdata$samplequeries
            WHERE id = p_id;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                lang_data_logger_pkg.log_error(
                    'No sample query found for deletion with ID: ' || p_id
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_resource_not_found
                );
                RETURN;
        END;

        -- If the role is enabled, delete the record regardless of the owner
        DELETE FROM langdata$samplequeries
        WHERE id = p_id;

        lang_data_logger_pkg.log_info(
            'Record successfully deleted with ID: ' || p_id
        );

        lang_data_utils_pkg.update_report_query_cluster(v_report_id);
        IF v_drilldown_id IS NOT NULL THEN
            lang_data_utils_pkg.update_drilldown_query_cluster(
                v_drilldown_id
            );
        END IF;
    EXCEPTION
       WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_resource_not_found
            ) THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                -- Log unknown errors as fatal and raise the generic error code
                lang_data_logger_pkg.log_fatal(
                    'An error occurred while deleting sample query with ID: ' || 
                    p_id || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
    END delete_sample_query;

    -- Procedure to add report sample queries
    PROCEDURE add_report_sample_queries(
        p_report_id      IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST, -- List of sample queries
        p_ids OUT SYS.ODCIVARCHAR2LIST
    )
    IS
        -- List for deduplicated queries
        v_deduplicated_queries SYS.ODCIVARCHAR2LIST;
        -- Variables for processing queries
        v_query_text          VARCHAR2(2000);
        v_ner_augmented_query VARCHAR2(4000);
        v_normalized_query    VARCHAR2(2000);
        v_query_md5           VARCHAR2(32);
        v_enhanced_query     VARCHAR2(2000);
        v_query_vector        VECTOR;
        v_query_id            VARCHAR2(36);
        v_duplicate_check     NUMBER;
        v_processed_count     NUMBER := 0;
        v_match_document      JSON;
        v_augmented_tokens    JSON;
        v_ner_augmentation_enabled  BOOLEAN := FALSE;
        v_entities                  JSON;
        v_domain_id           VARCHAR2(36);
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure add_report_sample_queries'
        );

        -- Check for authorization
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        -- Validate the report ID and fetch its match document
        SELECT match_document, domain_id
        INTO v_match_document, v_domain_id
        FROM langdata$reports
        WHERE id = p_report_id;

        -- Initialize out id list
        p_ids := SYS.ODCIVARCHAR2LIST();

        -- Deduplicate queries
        v_deduplicated_queries := deduplicate_sample_queries(p_sample_queries);

        FOR i IN 1 .. v_deduplicated_queries.COUNT LOOP
            v_query_text := v_deduplicated_queries(i);
            v_ner_augmented_query := v_query_text;
            v_normalized_query := LOWER(TRIM(v_query_text));
            v_query_md5 := DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(v_normalized_query), 2
            );

            -- Check for duplicate
            SELECT COUNT(*)
            INTO v_duplicate_check
            FROM langdata$samplequeries
            WHERE query_md5 = v_query_md5;

            IF v_duplicate_check > 0 THEN
                lang_data_logger_pkg.log_info(
                    'Duplicate query found, skipping: ' || v_query_text
                );
                CONTINUE;
            END IF;

            IF v_ner_augmentation_enabled THEN
                v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                                v_query_text
                            );
                IF v_entities IS NOT NULL THEN
                    v_ner_augmented_query :=
                        lang_data_utils_pkg.augment_text_with_ner_entities(
                            v_query_text, v_entities
                        );
                END IF;
            END IF;

            -- Process query
            lang_data_utils_pkg.augment_text(
                p_text              =>  v_ner_augmented_query,
                p_domain_id         =>  v_domain_id,
                p_augmented_text    =>  v_enhanced_query,
                p_augmented_tokens  =>  v_augmented_tokens
            );
            v_enhanced_query := lang_data_utils_pkg.expand_text(
                p_match_document => v_match_document,
                p_original_text  => v_enhanced_query,
                p_text_type      => 'query'
            );

            v_query_vector := lang_data_utils_pkg.get_embedding(
                v_enhanced_query
            );

            v_query_id := lang_data_utils_pkg.generate_id();

            -- Insert the query

            INSERT INTO langdata$samplequeries (
                id, 
                query_text, 
                query_vector, 
                report_id, 
                drilldown_id,
                query_md5, 
                enhanced_query_text, 
                created_at,
                augmented_tokens,
                regression_json,
                status
            ) VALUES (
                v_query_id, 
                v_query_text, 
                v_query_vector, 
                p_report_id,
                NULL, 
                v_query_md5, 
                v_enhanced_query, 
                CURRENT_TIMESTAMP,
                v_augmented_tokens,
                NULL,
                'Published'
            );

            v_processed_count := v_processed_count + 1;
            p_ids.EXTEND;
            p_ids(v_processed_count) := v_query_id;

            BEGIN
                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name => 'JOB_LANGDATA_'||
                        substr(v_query_md5,1,5)||'_REPORT_SQ_REGRESSION',
                    p_job_action => 
                        'BEGIN ' ||
            'lang_data_reports_pkg.calculate_report_regression( ' ||
                    'p_new_sample_query_id => '''|| v_query_id || '''); END;',
                    p_job_class => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments => 
                        'Regression calculation for report Sample query',
                    p_priority => 1,
                    p_restart_on_fail => FALSE,
                    p_restart_on_rec => TRUE
                );
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_error(
                        'Failed to create job for query_id: '||
                            v_query_id
                    );
                    RAISE;
            END;
        END LOOP;

        lang_data_logger_pkg.log_info(
            'Processed and inserted ' || v_processed_count ||
            ' report sample queries after deduplication.'
        );

        IF v_processed_count > 0 THEN
            lang_data_utils_pkg.update_report_query_cluster(p_report_id);
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_unauthorized_code THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An error occurred in add_report_sample_queries: ' ||
                    SQLERRM
                );
                RAISE;
            END IF;
    END add_report_sample_queries;

    -- Procedure to add drilldown sample queries
    PROCEDURE add_drilldown_sample_queries(
        p_drilldown_id   IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST, -- List of sample queries
        p_ids OUT SYS.ODCIVARCHAR2LIST
    )
    IS
        -- List for deduplicated queries
        v_deduplicated_queries SYS.ODCIVARCHAR2LIST;
        -- Variables for processing queries
        v_query_text          VARCHAR2(2000);
        v_ner_augmented_query VARCHAR2(4000);
        v_normalized_query    VARCHAR2(2000);
        v_query_md5           VARCHAR2(32);
        v_enhanced_query     VARCHAR2(4000);
        v_query_vector        VECTOR;
        v_query_id            VARCHAR2(36);
        v_duplicate_check     NUMBER;
        v_processed_count     NUMBER := 0;
        v_match_document      JSON;
        v_parent_report_id    VARCHAR2(36);
        v_augmented_tokens    JSON;
        v_ner_augmentation_enabled  BOOLEAN := FALSE;
        v_entities                  JSON;
        v_domain_id           VARCHAR2(36);
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure add_drilldown_sample_queries'
        );

        -- Check for authorization
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Validate the drilldown ID and fetch its match document and parent
        -- report ID
        SELECT match_document, report_id, domain_id
        INTO v_match_document, v_parent_report_id, v_domain_id
        FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;

        -- Initialize out id list
        p_ids := SYS.ODCIVARCHAR2LIST();

        -- Deduplicate queries
        v_deduplicated_queries := deduplicate_sample_queries(p_sample_queries);

        FOR i IN 1 .. v_deduplicated_queries.COUNT LOOP
            v_query_text := v_deduplicated_queries(i);
            v_ner_augmented_query := v_query_text;
            v_normalized_query := LOWER(TRIM(v_query_text));
            v_query_md5 := DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(v_normalized_query), 2
            );

            -- Check for duplicate
            SELECT COUNT(*)
            INTO v_duplicate_check
            FROM langdata$samplequeries
            WHERE query_md5 = v_query_md5;

            IF v_duplicate_check > 0 THEN
                lang_data_logger_pkg.log_info(
                    'Duplicate query found, skipping: ' || v_query_text
                );
                CONTINUE;
            END IF;

            IF v_ner_augmentation_enabled THEN
                v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                                v_query_text
                            );
                IF v_entities IS NOT NULL THEN
                    v_ner_augmented_query := 
                        lang_data_utils_pkg.augment_text_with_ner_entities(
                            v_query_text, v_entities
                        );
                END IF;
            END IF;

            -- Process query
            lang_data_utils_pkg.augment_text(
                p_text              =>  v_ner_augmented_query,
                p_domain_id         =>  v_domain_id,
                p_augmented_text    =>  v_enhanced_query,
                p_augmented_tokens  =>  v_augmented_tokens
            );
            v_enhanced_query := lang_data_utils_pkg.expand_text(
                p_match_document => v_match_document,
                p_original_text  => v_enhanced_query,
                p_text_type      => 'query'
            );

            v_query_vector := lang_data_utils_pkg.get_embedding(
                v_enhanced_query
            );

            v_query_id := lang_data_utils_pkg.generate_id();

            -- Insert the query with the parent report ID
            EXECUTE IMMEDIATE 
                'INSERT INTO langdata$samplequeries (
                    id, query_text, query_vector, report_id, drilldown_id,
                    query_md5, enhanced_query_text, created_at, status
                ) VALUES (
                    :query_id, :query_text, :query_vector, :parent_report_id,
                    :drilldown_id, :query_md5, :enhanced_query, 
                    CURRENT_TIMESTAMP, :status
                )'
            USING 
                v_query_id, 
                v_query_text, 
                v_query_vector, 
                v_parent_report_id, 
                p_drilldown_id, 
                v_query_md5, 
                v_enhanced_query,
                'Published';

            v_processed_count := v_processed_count + 1;
            p_ids.EXTEND;
            p_ids(v_processed_count) := v_query_id;

        END LOOP;

        lang_data_logger_pkg.log_info(
            'Processed and inserted ' || v_processed_count ||
            ' drilldown sample queries after deduplication.'
        );

        IF v_processed_count > 0 THEN
            lang_data_utils_pkg.update_report_query_cluster(v_parent_report_id);
            lang_data_utils_pkg.update_drilldown_query_cluster(p_drilldown_id);
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_unauthorized_code THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'An error occurred in add_drilldown_sample_queries: ' ||
                    SQLERRM
                );
                RAISE;
            END IF;
    END add_drilldown_sample_queries;

    FUNCTION deduplicate_sample_queries (
        p_sample_queries IN SYS.ODCIVARCHAR2LIST
    ) RETURN SYS.ODCIVARCHAR2LIST
    IS
        v_normalized_query VARCHAR2(4000);
        v_query_lower VARCHAR2(4000);
        v_unique_queries SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        
        -- Using an associative array (Set) for fast deduplication
        TYPE set_type IS TABLE OF BOOLEAN INDEX BY VARCHAR2(4000);
        v_query_set set_type;
    BEGIN
        -- Log the start of the function
        lang_data_logger_pkg.log_info(
            'Starting function deduplicate_sample_queries'
        );

        FOR i IN 1..p_sample_queries.COUNT LOOP
            v_query_lower := LOWER(TRIM(p_sample_queries(i)));

            -- Check if the query already exists in the Set (O(1) complexity)
            IF NOT v_query_set.EXISTS(v_query_lower) THEN
                -- Add the unique query to the result list
                v_unique_queries.EXTEND;
                v_unique_queries(v_unique_queries.COUNT) := p_sample_queries(i);

                -- Add the normalized query to the Set
                v_query_set(v_query_lower) := TRUE;

            ELSE
                -- Log the skipped duplicate query
                lang_data_logger_pkg.log_info(
                    'Skipped duplicate query: ' || p_sample_queries(i)
                );
            END IF;
        END LOOP;

        RETURN v_unique_queries;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'Error during deduplication: ' || SQLERRM
            );
            RAISE;
    END deduplicate_sample_queries;

    PROCEDURE update_sample_query_status(
        p_id IN VARCHAR2,
        p_status IN VARCHAR2,
        p_response OUT VARCHAR2
    )
    IS  
        v_old_status    VARCHAR2(20);
        v_report_id     VARCHAR2(36);
        v_drilldown_id  VARCHAR2(36);
    BEGIN
        -- Check if the user has the required role
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        IF p_status NOT IN ( 'Pending Review',
                             'Approved',
                             'Rejected',
                             'Published',
                             'Inactive',
                             'Archived',
                             'Pending Regression' ) THEN
            lang_data_logger_pkg.log_error('Invalid status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT status, report_id, drilldown_id
        INTO v_old_status, v_report_id, v_drilldown_id
        FROM langdata$samplequeries
        WHERE id = p_id;

        UPDATE langdata$samplequeries
        SET status = p_status
        WHERE id = p_id;

        lang_data_logger_pkg.log_debug(
            'Updated Sample query status, '||
            'Updating Clusters if status changed to/from ''Published''.'
        );
        IF v_old_status = 'Published' OR p_status = 'Published' THEN
            lang_data_utils_pkg.update_report_query_cluster(v_report_id);
            IF v_drilldown_id IS NOT NULL THEN
                lang_data_utils_pkg.update_drilldown_query_cluster(
                    v_drilldown_id
                );
            END IF;
        END IF;

        p_response := p_status;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No Sample query found with id: '|| p_id);
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                -- Log unknown errors as fatal and raise the generic error code
                lang_data_logger_pkg.log_fatal(
                'An error occurred while updating sample query status with ID: '
                    || p_id || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
    END update_sample_query_status;

    FUNCTION get_sample_query_id_by_report_id_version(
        p_report_id    IN VARCHAR2,
        p_version      IN NUMBER
    ) RETURN VARCHAR2 IS
        v_sample_query_id VARCHAR2(36);
    BEGIN
        SELECT id INTO v_sample_query_id FROM langdata$samplequeries
        WHERE report_id = p_report_id AND version = p_version;
        RETURN v_sample_query_id;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No sample query found for report ID: ' || 
                p_report_id || ', version: ' || p_version
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_sample_query_id_by_report_id_version;

END lang_data_sample_queries_pkg;
/

