/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.premigration.dbinspector;

import com.oracle.premigration.api.Premigration;
import com.oracle.premigration.cli.CliOption;
import com.oracle.premigration.cli.CommandLineArgs;
import com.oracle.premigration.commons.EnvironmentContext;
import com.oracle.premigration.commons.enums.ReportFormat;
import com.oracle.premigration.commons.io.OutputFilenamesContainer;
import com.oracle.premigration.commons.io.ZipContributor;
import com.oracle.premigration.commons.io.ZipFileHelper;
import com.oracle.premigration.commons.lang.Language;
import com.oracle.premigration.dbinspector.CheckResult;
import com.oracle.premigration.dbinspector.ExecutionContext;
import com.oracle.premigration.exceptions.AccessDeniedException;
import com.oracle.premigration.exceptions.PremigrationException;
import com.oracle.premigration.helpers.AppVersion;
import com.oracle.premigration.helpers.SqlUtils;
import com.oracle.premigration.helpers.Utilities;
import com.oracle.premigration.logger.PremigrationLogger;
import com.oracle.premigration.reports.CDBReportSummary;
import com.oracle.premigration.reports.PDBAnalysisSummary;
import com.oracle.premigration.reports.html.HtmlPdbReport;
import com.oracle.premigration.reports.json.serialization.JsonSerializer;
import com.oracle.premigration.reports.text.TextReport;
import com.oracle.premigration.reports.text.TextReportContext;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.EnumMap;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

public class CDBSourceInspector {
    private static final PremigrationLogger log = PremigrationLogger.getLogger(CDBSourceInspector.class.getName());
    private static final Language lang = Language.getInstance();

    private CDBSourceInspector() {
    }

    public static CheckResult analyzeSourcePDBs(ExecutionContext exeCtx, String appVersion) throws Exception {
        List<String> analysisPDBs;
        CDBReportSummary cdbReportSummary = new CDBReportSummary();
        CommandLineArgs cla = exeCtx.getParsedArgs();
        EnvironmentContext envCtx = exeCtx.getEnvCtx();
        ArrayList<String> reportAnalysisNotes = new ArrayList<String>();
        if (!exeCtx.isAnalysisOfMultiplePdbs()) {
            throw new PremigrationException("Illegal use of CDBSourceInspector.analyzeSourcePDBs");
        }
        if (!exeCtx.isCdbRootConnection()) {
            String severeMessage = Utilities.getErrorText("ERROR1019", new String[0]);
            Utilities.printAndLogErrorMessage(severeMessage, envCtx, Level.SEVERE);
            return CheckResult.FATAL;
        }
        String propFilePath = cla.getAnalysisPropertiesFile();
        String propFileName = null;
        if (propFilePath == null) {
            reportAnalysisNotes.add(Utilities.getAnalysisNoteText("ERROR1013", new Object[0]));
            String noPropsFileMsg = Utilities.getErrorText("ERROR1013", new String[0]);
            Utilities.printAndLogErrorMessage(noPropsFileMsg, envCtx, Level.INFO);
        } else if (cla.isGenZipFile()) {
            File propertiesFile = new File(propFilePath);
            if (!propertiesFile.exists()) {
                throw new FileNotFoundException(propFilePath + " does not exist");
            }
            propFileName = propertiesFile.getName();
            ZipContributor propFileForZip = ZipFileHelper.createZipContributorForFile(propFileName, propFilePath);
            exeCtx.getZipContents().add(propFileForZip);
        }
        if (cla.getTargetCloud() == null) {
            String severeMessage = Utilities.getErrorText("ERROR1015", new String[0]);
            Utilities.printAndLogErrorMessage(severeMessage, envCtx, Level.SEVERE);
            return CheckResult.FATAL;
        }
        SqlUtils.validateSourceEdition(exeCtx, reportAnalysisNotes);
        cdbReportSummary.setAppVersion(appVersion);
        cdbReportSummary.setAppBuildDate(AppVersion.getAppVersionDate());
        cdbReportSummary.setReportDate(new Date());
        cdbReportSummary.setHostName(Utilities.getHostName());
        cdbReportSummary.setCommandLineOptions(cla.getReconstitutedArgs());
        String cdbOutputDir = exeCtx.getOutputFilenamesContainer().getOutDir();
        cdbReportSummary.setAbsoluteOutputPath(OutputFilenamesContainer.getOutdirValidated(cdbOutputDir));
        int threadPoolSize = CDBSourceInspector.getThreadPoolSize(cla.getMaxSubprocesses(), exeCtx);
        List<String> openPdbs = SqlUtils.getStringList("SELECT NAME FROM V$PDBS WHERE OPEN_MODE IN ('READ ONLY', 'READ WRITE') AND NAME <> 'PDB$SEED' ORDER BY NAME", exeCtx);
        cdbReportSummary.setOpenPDBs(openPdbs);
        cdbReportSummary.setClosedPDBs(SqlUtils.getStringList("SELECT NAME FROM V$PDBS WHERE OPEN_MODE NOT IN ('READ ONLY', 'READ WRITE') ORDER BY NAME", exeCtx));
        if (cla.getPdbNames().isEmpty()) {
            analysisPDBs = openPdbs;
        } else {
            analysisPDBs = SqlUtils.cleanAndValidateOpenPDBNames(cla.getPdbNames(), exeCtx);
            if (analysisPDBs.size() != cla.getPdbNames().size()) {
                ArrayList<String> claPdbs = new ArrayList<String>(cla.getPdbNames());
                claPdbs.removeAll(analysisPDBs);
                String errorMessage = Utilities.getErrorText("ERROR2009", Utilities.join(", ", claPdbs));
                Utilities.printAndLogErrorMessage(errorMessage, envCtx, Level.INFO);
            }
        }
        if (analysisPDBs.isEmpty()) {
            String severeMessage = Utilities.getErrorText("ERROR2010", new String[0]);
            Utilities.printAndLogErrorMessage(severeMessage, envCtx, Level.SEVERE);
            return CheckResult.FATAL;
        }
        if (threadPoolSize < analysisPDBs.size()) {
            String poolSizeStr = String.valueOf(threadPoolSize);
            String pdbListStr = String.valueOf(analysisPDBs.size());
            String messageText = Utilities.getErrorText("ERROR1022", poolSizeStr, pdbListStr, pdbListStr);
            Utilities.printAndLogErrorMessage(messageText, envCtx, Level.INFO);
        }
        envCtx.getPrintStream().println("Analyzing " + analysisPDBs.size() + " PDBs from CDB$ROOT");
        List<PDBAnalysisSummary> pdbSummaries = Collections.synchronizedList(cdbReportSummary.getPdbAnalysisResults());
        CheckResult overallResult = CDBSourceInspector.spawnPdbAnalysis(analysisPDBs, threadPoolSize, cla, envCtx, pdbSummaries);
        cla.nullifyPassword();
        Collections.sort(cdbReportSummary.getPdbAnalysisResults(), PDBAnalysisSummary.getComparator());
        CDBSourceInspector.writeReports(cdbReportSummary, exeCtx);
        if (cla.isGenZipFile()) {
            for (String pdbName : analysisPDBs) {
                List<ZipContributor> pdbFilesForZip = CDBSourceInspector.getZipFileContributors(pdbName, cdbOutputDir, cla.getOutFilePrefix(), propFileName);
                exeCtx.getZipContents().addAll(pdbFilesForZip);
            }
        }
        return overallResult;
    }

    private static void writeReports(CDBReportSummary reportData, ExecutionContext exeCtx) throws Exception {
        CommandLineArgs cla = exeCtx.getParsedArgs();
        EnvironmentContext envCtx = exeCtx.getEnvCtx();
        for (ReportFormat reportFormat : cla.getReportFormats()) {
            String createdFilePath;
            String filePath;
            String fileName;
            OutputFilenamesContainer namesContainer = exeCtx.getOutputFilenamesContainer();
            long startSerialization = System.currentTimeMillis();
            switch (reportFormat) {
                case JSON: {
                    String jsonPath = namesContainer.getValidatedJSONReportPath();
                    fileName = namesContainer.getJsonOutputFilename();
                    filePath = namesContainer.getValidatedJSONReportPath();
                    createdFilePath = JsonSerializer.writeJSON(reportData, jsonPath);
                    break;
                }
                case TEXT: {
                    String textPath = namesContainer.getValidatedTextReportPath();
                    fileName = namesContainer.getTextOutputFilename();
                    filePath = namesContainer.getValidatedTextReportPath();
                    TextReportContext textReportContext = TextReportContext.createTextReportContext(cla, textPath);
                    createdFilePath = TextReport.writeText(reportData, textReportContext);
                    break;
                }
                case HTML: {
                    String htmlPath = namesContainer.getValidatedHtmlReportPath();
                    fileName = namesContainer.getHtmlOutputFilename();
                    filePath = namesContainer.getValidatedHtmlReportPath();
                    createdFilePath = HtmlPdbReport.writeHtml(reportData, htmlPath);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid report format");
                }
            }
            log.info("Elapsed time serializing " + createdFilePath + ": " + Utilities.formatTimeInterval(System.currentTimeMillis() - startSerialization));
            String reportLocationMsg = lang.txt("APP_REPORT_LOCATION_MSG", createdFilePath);
            Utilities.printAndLogMessage(reportLocationMsg, envCtx, Level.INFO);
            if (!exeCtx.getParsedArgs().isGenZipFile()) continue;
            ZipContributor fileForZip = ZipFileHelper.createZipContributorForFile(fileName, filePath);
            exeCtx.getZipContents().add(fileForZip);
        }
    }

    private static CheckResult spawnPdbAnalysis(List<String> openPDBs, int threadPoolSize, CommandLineArgs cla, EnvironmentContext envCtx, List<PDBAnalysisSummary> pdbAnalysisSummaries) throws PremigrationException {
        CheckResult worstResult = CheckResult.PASS;
        List<Object> cdbCommandArgs = cla.getReconstitutedArgArray(true);
        LinkedBlockingQueue<Runnable> threadQueue = new LinkedBlockingQueue<Runnable>();
        PremigrationThreadPoolExecutor executor = new PremigrationThreadPoolExecutor(threadPoolSize, threadPoolSize, 0L, TimeUnit.MILLISECONDS, threadQueue, pdbAnalysisSummaries);
        String premigrationScript = CDBSourceInspector.getPremigrationScriptPath(envCtx);
        for (String pdbName : openPDBs) {
            String[] cpatCommands = CDBSourceInspector.genCPATCommandLineForPDB(premigrationScript, cdbCommandArgs, pdbName);
            PremigrationAppRunner pdbPremigrationThread = new PremigrationAppRunner(cpatCommands, cla.getPassword(), pdbName);
            log.info("Calling executor with thread for the PDB named " + pdbName);
            executor.execute(pdbPremigrationThread);
        }
        executor.shutdown();
        try {
            do {
                Utilities.printAndLogMessage(CDBSourceInspector.getProgressMessage(executor), envCtx, Level.INFO);
            } while (!executor.awaitTermination(10L, TimeUnit.SECONDS));
        }
        catch (InterruptedException e) {
            String errorMessage = Utilities.getErrorText("ERROR2008", new String[0]);
            Utilities.printAndLogErrorMessage(errorMessage, envCtx, Level.SEVERE, e);
            log.warning("There were " + (executor.getTaskCount() - executor.getCompletedTaskCount()) + " tasks that were not completed");
            Thread.currentThread().interrupt();
        }
        Utilities.printAndLogMessage(CDBSourceInspector.getProgressMessage(executor), envCtx, Level.INFO);
        for (PDBAnalysisSummary analysisSummary : pdbAnalysisSummaries) {
            CheckResult analysisResult = analysisSummary.getAnalysisResult();
            if (analysisResult == null || analysisResult.compareSeverityTo(worstResult) <= 0) continue;
            worstResult = analysisResult;
        }
        return worstResult;
    }

    private static String getProgressMessage(ThreadPoolExecutor executor) {
        long taskCount = executor.getTaskCount();
        long activeCount = executor.getActiveCount();
        long completedCount = executor.getCompletedTaskCount();
        String localDateTime = Utilities.getLocalizedDateTimeString(new Date());
        return lang.txt("APP_PROGRESS_SPAWNED_TASKS", completedCount, taskCount, localDateTime, activeCount);
    }

    private static String[] genCPATCommandLineForPDB(String cpatScriptName, List<Object> cdbCommandArgs, String pdbName) {
        ArrayList<String> cpatCommandLineForPDB = new ArrayList<String>();
        boolean foundOutFilePrefix = false;
        boolean addPdbNameToNextArg = false;
        boolean foundOutDir = false;
        boolean addPdbPathToNextArg = false;
        boolean skipNextArg = false;
        cpatCommandLineForPDB.add(cpatScriptName);
        for (Object commandLineArg : cdbCommandArgs) {
            if (skipNextArg) {
                skipNextArg = false;
                continue;
            }
            if (addPdbNameToNextArg) {
                cpatCommandLineForPDB.add(CDBSourceInspector.getPDBOutFilePrefix(commandLineArg.toString(), pdbName));
                addPdbNameToNextArg = false;
            } else if (addPdbPathToNextArg) {
                cpatCommandLineForPDB.add(CDBSourceInspector.getPDBOutputPath(commandLineArg.toString(), pdbName));
                addPdbPathToNextArg = false;
            } else if (CliOption.MAXSUBPROCESSES.getDashedLongOpt().equals(commandLineArg)) {
                skipNextArg = true;
            } else {
                cpatCommandLineForPDB.add(commandLineArg.toString());
            }
            if (CliOption.OUTFILEPREFIX.getDashedLongOpt().equals(commandLineArg)) {
                addPdbNameToNextArg = true;
                foundOutFilePrefix = true;
            }
            if (!CliOption.OUTDIR.getDashedLongOpt().equals(commandLineArg)) continue;
            addPdbPathToNextArg = true;
            foundOutDir = true;
        }
        if (!foundOutFilePrefix) {
            cpatCommandLineForPDB.add(CliOption.OUTFILEPREFIX.getDashedLongOpt());
            cpatCommandLineForPDB.add(pdbName);
        }
        if (!foundOutDir) {
            cpatCommandLineForPDB.add(CliOption.OUTDIR.getDashedLongOpt());
            cpatCommandLineForPDB.add(pdbName);
        }
        cpatCommandLineForPDB.add(CliOption.PDBNAME.getDashedLongOpt());
        cpatCommandLineForPDB.add(pdbName);
        return cpatCommandLineForPDB.toArray(new String[0]);
    }

    private static String getPremigrationScriptPath(EnvironmentContext envCtx) throws PremigrationException {
        try {
            String basePath = CDBSourceInspector.getPremigrationLocation();
            return basePath + File.separator + Utilities.getPremigrationCommand();
        }
        catch (URISyntaxException e) {
            String errorText = Utilities.getErrorText("ERROR4009", new String[0]);
            Utilities.printAndLogErrorMessage(errorText, envCtx, Level.SEVERE, e);
            throw new PremigrationException(errorText, e);
        }
    }

    private static String getPDBOutFilePrefix(String cdbOutFilePrefix, String pdbName) {
        if (cdbOutFilePrefix == null || cdbOutFilePrefix.isEmpty()) {
            return pdbName;
        }
        return pdbName + "_" + cdbOutFilePrefix;
    }

    private static String getPDBOutputPath(String cdbOutDir, String pdbName) {
        return cdbOutDir + File.separator + pdbName;
    }

    private static String getPremigrationLocation() throws URISyntaxException {
        URI jarLocation = Premigration.class.getProtectionDomain().getCodeSource().getLocation().toURI();
        File jarFile = new File(jarLocation);
        return jarFile.getParentFile().getParentFile().getPath();
    }

    private static int getThreadPoolSize(Integer commandLineMaxSubProcesses, ExecutionContext exeCtx) {
        int threadPoolSize;
        if (commandLineMaxSubProcesses != null) {
            threadPoolSize = Math.max(1, commandLineMaxSubProcesses);
        } else {
            try {
                int cpuCount = (int)SqlUtils.getSingleLong("SELECT VALUE FROM V$PARAMETER WHERE LOWER(NAME) LIKE 'cpu_count'", exeCtx);
                log.info("CPU_COUNT: " + cpuCount);
                threadPoolSize = cpuCount > 3 ? cpuCount / 2 : 1;
            }
            catch (Exception e) {
                threadPoolSize = 1;
                log.info("Exception determining default thread pool size.  Defaulting to single threading.", e);
            }
        }
        log.info("Thread Pool Size: " + threadPoolSize);
        return threadPoolSize;
    }

    private static List<ZipContributor> getZipFileContributors(String pdbName, String cdbOutputDir, String cdbOutFilePrefix, String propFileName) throws IOException, AccessDeniedException {
        ArrayList<ZipContributor> pdbZipFileEntries = new ArrayList<ZipContributor>();
        String pdbOutputPath = CDBSourceInspector.getPDBOutputPath(cdbOutputDir, pdbName);
        String pdbOutFilePrefix = CDBSourceInspector.getPDBOutFilePrefix(cdbOutFilePrefix, pdbName);
        OutputFilenamesContainer pdbOutputFilenamesContainer = new OutputFilenamesContainer(pdbOutFilePrefix, pdbOutputPath);
        File pdbZipFile = new File(pdbOutputFilenamesContainer.getValidatedZipFilePath());
        try (ZipInputStream zipIn = new ZipInputStream(Files.newInputStream(pdbZipFile.toPath(), new OpenOption[0]));){
            ZipEntry entry;
            while ((entry = zipIn.getNextEntry()) != null) {
                String entryName = entry.getName();
                if (entryName.equals(propFileName) || entryName.endsWith("/")) {
                    log.trace("Skipping entry " + entryName + " as it should be include elsewhere.");
                    continue;
                }
                String absolutePath = pdbOutputFilenamesContainer.getOutDir() + File.separator + entry.getName();
                String relativePath = pdbName + File.separator + entry.getName();
                pdbZipFileEntries.add(ZipFileHelper.createZipContributorForFile(relativePath, absolutePath));
            }
        }
        return pdbZipFileEntries;
    }

    private static class PremigrationThreadPoolExecutor
    extends ThreadPoolExecutor {
        final List<PDBAnalysisSummary> pdbResults;

        public PremigrationThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, List<PDBAnalysisSummary> pdbResults) {
            super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue);
            this.pdbResults = pdbResults;
        }

        @Override
        protected void afterExecute(Runnable r, Throwable t) {
            PDBAnalysisSummary analysisSummary;
            super.afterExecute(r, t);
            PremigrationAppRunner premigrationAppRunner = (PremigrationAppRunner)r;
            String pdbName = premigrationAppRunner.getPdbName();
            if (t != null) {
                log.severe("Exception analyzing PDB " + pdbName, t);
            }
            if ((analysisSummary = premigrationAppRunner.createPDBAnalysisSummary()) != null) {
                this.pdbResults.add(analysisSummary);
                log.info("PDB " + pdbName + " exit code: " + (Object)((Object)analysisSummary.getAnalysisResult()));
            } else {
                log.info("PDB " + pdbName + " did not set exit code. Assuming FATAL.");
            }
        }
    }

    private static class PremigrationAppRunner
    implements Runnable {
        private final String[] commands;
        private final char[] password;
        private final String pdbName;
        private Integer exitCode = -1;
        Long elapsedMilliSeconds = null;
        boolean done = false;

        protected PremigrationAppRunner(String[] commands, char[] password, String pdbName) {
            this.commands = commands;
            this.password = password;
            this.pdbName = pdbName;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long startTime = System.currentTimeMillis();
            String logPrefix = this.pdbName + " analysis: ";
            try {
                Process cpatProcess = Runtime.getRuntime().exec(this.commands);
                if (this.password != null) {
                    for (char c : this.password) {
                        cpatProcess.getOutputStream().write(c);
                    }
                    cpatProcess.getOutputStream().write(10);
                    cpatProcess.getOutputStream().flush();
                }
                try (BufferedReader stdInput = new BufferedReader(new InputStreamReader(cpatProcess.getInputStream()));
                     BufferedReader stdError = new BufferedReader(new InputStreamReader(cpatProcess.getErrorStream()));){
                    String s;
                    while ((s = stdInput.readLine()) != null) {
                        log.info(logPrefix + s);
                    }
                    while ((s = stdError.readLine()) != null) {
                        log.info(logPrefix + s);
                    }
                }
                this.exitCode = cpatProcess.waitFor();
                log.info(logPrefix + "exitCode = " + this.exitCode);
                this.elapsedMilliSeconds = System.currentTimeMillis() - startTime;
                log.fine(logPrefix + "Elapsed time running CPAT statement: " + Utilities.formatTimeInterval(this.elapsedMilliSeconds));
            }
            catch (IOException | InterruptedException e) {
                log.severe("Exception during analysis of PDB " + this.pdbName, e);
                this.exitCode = CheckResult.FATAL.getExitCode();
                if (e instanceof InterruptedException) {
                    Thread.currentThread().interrupt();
                }
            }
            finally {
                this.done = true;
            }
        }

        protected PDBAnalysisSummary createPDBAnalysisSummary() {
            if (!this.done) {
                return null;
            }
            long analysisDurationSeconds = this.elapsedMilliSeconds != null ? this.elapsedMilliSeconds / 1000L : -1L;
            CheckResult pdbAnalysisResult = CheckResult.fromCommandLineExitCode(this.exitCode);
            PDBAnalysisSummary.PDBAnalysisSummaryBuilder analysisSummaryBuilder = PDBAnalysisSummary.builder().pdbName(this.pdbName).analysisDurationSeconds(analysisDurationSeconds).analysisResult(pdbAnalysisResult).analysisResultDisplayText(pdbAnalysisResult.toString());
            EnumMap<ReportFormat, String> reportMap = new EnumMap<ReportFormat, String>(ReportFormat.class);
            try {
                String textReportPath;
                String htmlReportPath;
                String jsonReportPath;
                String pdbOutputPrefix = this.getOutputPrefix();
                String basePath = this.getOutputBasePath();
                String pdbOutputDir = OutputFilenamesContainer.getOutdirValidated(basePath);
                OutputFilenamesContainer pdbOutputFilenamesContainer = new OutputFilenamesContainer(pdbOutputPrefix, pdbOutputDir);
                String outDir = new File(pdbOutputFilenamesContainer.getOutDir()).getParentFile().getPath();
                analysisSummaryBuilder.analysisOutputBasePath(outDir);
                String logFilePath = pdbOutputFilenamesContainer.getValidatedLogFilePath();
                if (new File(logFilePath).exists()) {
                    analysisSummaryBuilder.analysisLogFile(Utilities.relativizePath(outDir, logFilePath));
                }
                if (new File(jsonReportPath = pdbOutputFilenamesContainer.getValidatedJSONReportPath()).exists()) {
                    reportMap.put(ReportFormat.JSON, Utilities.relativizePath(outDir, jsonReportPath));
                }
                if (new File(htmlReportPath = pdbOutputFilenamesContainer.getValidatedHtmlReportPath()).exists()) {
                    reportMap.put(ReportFormat.HTML, Utilities.relativizePath(outDir, htmlReportPath));
                }
                if (new File(textReportPath = pdbOutputFilenamesContainer.getValidatedTextReportPath()).exists()) {
                    reportMap.put(ReportFormat.TEXT, Utilities.relativizePath(outDir, textReportPath));
                }
            }
            catch (AccessDeniedException | IOException e) {
                String exceptionMessage = "Unexpected error getting PDB Analysis results";
                log.severe("Unexpected error getting PDB Analysis results", e);
                throw new IllegalStateException("Unexpected error getting PDB Analysis results", e);
            }
            analysisSummaryBuilder.reportFiles(reportMap);
            return analysisSummaryBuilder.build();
        }

        private String getOutputBasePath() {
            return this.getCommandArg(CliOption.OUTDIR);
        }

        private String getOutputPrefix() {
            return this.getCommandArg(CliOption.OUTFILEPREFIX);
        }

        private String getCommandArg(CliOption cliOption) {
            for (int i = 0; i < this.commands.length; ++i) {
                String arg = this.commands[i];
                if (!cliOption.getDashedLongOpt().equals(arg)) continue;
                return this.commands[i + 1];
            }
            return null;
        }

        public String getPdbName() {
            return this.pdbName;
        }
    }
}

