Rem
Rem $Header: dbgendev/src/langdata/plsql/sql/create_precheck_procedures.sql /main/4 2025/05/05 15:43:50 deveverm Exp $
Rem
Rem create_precheck_procedures.sql
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      create_precheck_procedures.sql - Creates the precheck procedures for
Rem                                       setup_langdata mutation.
Rem
Rem    DESCRIPTION
Rem      This script creates the following precheck procedures that are executed
Rem      as part of the setup_langdata mutation:
Rem          - check_db_configuration
Rem          - check_user_privileges
Rem
Rem    NOTES
Rem      The delimeter used in this script after every procedure is "`" and not
Rem      "/" as the check_db_configuration procedure uses the "/" symbol for the
Rem      division operation and hence, the python utility method, run_sql_script
Rem      cannot split the script properly.
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/backend/sql/create_precheck_procedures.sql
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    03/11/25 - DBAI-542: Make precheck procedures authid definer
Rem    dadoshi     12/09/24 - Update check_user_privileges
Rem    dadoshi     12/09/24 - Created
Rem

DECLARE
    v_cloud_service VARCHAR2(10);
BEGIN
    SELECT sys_context('USERENV', 'CLOUD_SERVICE') 
    INTO v_cloud_service FROM dual;

    IF v_cloud_service IS NOT NULL THEN
        EXECUTE IMMEDIATE 'ALTER SESSION SET PLSQL_CCFLAGS = ''Is_Cloud:true''';
    ELSE
        EXECUTE IMMEDIATE 'ALTER SESSION SET PLSQL_CCFLAGS = ''Is_Cloud:false''';
    END IF;
END;
/

CREATE OR REPLACE PROCEDURE check_db_configuration IS
    v_banner          VARCHAR2(4000);
    v_version_number  NUMBER;
    v_pdb_name        VARCHAR2(255);
    v_con_id          NUMBER;
    v_service_name    VARCHAR2(255);
    v_network_name    VARCHAR2(255);
    v_cur             SYS_REFCURSOR;
    v_name            VARCHAR(128);
    v_value           VARCHAR(4000);
BEGIN
    -- Check Oracle version
    DBMS_OUTPUT.PUT_LINE('--- Oracle Version ---');
    SELECT banner INTO v_banner FROM v$version WHERE ROWNUM = 1;
    
    DBMS_OUTPUT.PUT_LINE('Version: ' || v_banner);
    
    -- Extract the version number using regex (search for "Oracle Database
    -- <version>")
    v_version_number := TO_NUMBER(
        REGEXP_SUBSTR(v_banner, 'Oracle Database (\d+)', 1, 1, NULL, 1)
    );

    -- Raise error if version is less than 23
    IF v_version_number < 23 THEN
        DBMS_OUTPUT.PUT_LINE(
            'Unsupported Oracle version: ' || v_version_number
        );
        RAISE_APPLICATION_ERROR(
            -20000, 'Precheck failure: Unsupported Oracle version.'
        );
    END IF;

    -- Verify initialization parameters
    DBMS_OUTPUT.PUT_LINE('--- Initialization Parameters ---');
    FOR r IN (
        SELECT PROPERTY_NAME, PROPERTY_VALUE
        FROM DATABASE_PROPERTIES
              WHERE PROPERTY_NAME IN (
                'DEFAULT_PERMANENT_TABLESPACE', 'DEFAULT_TEMP_TABLESPACE'
                )
            ) LOOP
        DBMS_OUTPUT.PUT_LINE(r.PROPERTY_NAME || ' = ' || r.PROPERTY_VALUE);
    END LOOP;

    -- Check user privileges
    DBMS_OUTPUT.PUT_LINE('--- User Roles ---');
    FOR r IN (
        SELECT GRANTED_ROLE
        FROM USER_ROLE_PRIVS
    ) LOOP
        DBMS_OUTPUT.PUT_LINE(r.granted_role);
    END LOOP;

    -- Check disk space
    DBMS_OUTPUT.PUT_LINE('--- Disk Space ---');
    FOR r IN (
        SELECT tablespace_name, SUM(bytes) / (1024 * 1024) AS free_space_mb
        FROM user_free_space
        GROUP BY tablespace_name
    ) LOOP
        DBMS_OUTPUT.PUT_LINE(
            r.tablespace_name || ': ' || r.free_space_mb || ' MB'
        );
    END LOOP;
    
    -- Check Oracle Machine Learning (OML) status
    DBMS_OUTPUT.PUT_LINE('--- Oracle Machine Learning (OML) Status ---');
    BEGIN
        OPEN v_cur FOR 'SELECT name, value FROM sys.pyq_config';
        LOOP
            FETCH v_cur INTO v_name, v_value;
            EXIT WHEN v_cur%NOTFOUND;
            DBMS_OUTPUT.PUT_LINE(
                'Name: ' || v_name || ' Value: ' || v_value
            );
        END LOOP;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = -942 THEN
                DBMS_OUTPUT.PUT_LINE(
                    'Error: OML4Py is not installed.' ||
                    'Please install OML4Py excuting the script as sysdba: '||
                    '''backend/sql/install_oml4py.sql'''
                );
            END IF;
    END;


EXCEPTION
    WHEN OTHERS THEN
        DBMS_OUTPUT.PUT_LINE('An unknown error occurred: ' || SQLERRM);
        RAISE;
END check_db_configuration;
/


CREATE OR REPLACE PROCEDURE check_user_privileges IS
    privilege_name VARCHAR2(255);
    match_count    INTEGER;
BEGIN
    -- List of privileges to check
    FOR privilege_record IN (
        SELECT 'CREATE MINING MODEL' AS privilege FROM DUAL UNION ALL
        SELECT 'CREATE JOB' FROM DUAL UNION ALL
        SELECT 'CREATE ANY CONTEXT' FROM DUAL UNION ALL
        SELECT 'CREATE ROLE' FROM DUAL UNION ALL
        SELECT 'CREATE SESSION' FROM DUAL UNION ALL
        SELECT 'CREATE TABLE' FROM DUAL UNION ALL
        SELECT 'CREATE VIEW' FROM DUAL UNION ALL
        SELECT 'CREATE PROCEDURE' FROM DUAL UNION ALL
        SELECT 'GRANT ANY ROLE' FROM DUAL UNION ALL
        SELECT 'MANAGE SCHEDULER' FROM DUAL
    ) LOOP
        -- Check if the privilege exists
        BEGIN
            SELECT PRIVILEGE
            INTO privilege_name
            FROM USER_SYS_PRIVS
            WHERE PRIVILEGE = privilege_record.privilege;

            DBMS_OUTPUT.PUT_LINE('System Privilege exists: ' || privilege_name);
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                RAISE_APPLICATION_ERROR(
                    -20000,
                    'Missing system privilege: ' || privilege_record.privilege
                );
        END;
    END LOOP;

    -- List of expected object-level privileges and their table names
    FOR tab_priv_check IN (
        SELECT 'EXECUTE' AS privilege, 'DBMS_SQL' AS table_name 
            FROM DUAL UNION ALL
        SELECT 'EXECUTE', 'DBMS_SCHEDULER' FROM DUAL UNION ALL
        SELECT 'EXECUTE', 'DBMS_CRYPTO' FROM DUAL UNION ALL
        SELECT 'EXECUTE', 'CTX_DDL' FROM DUAL
        $IF $$Is_Cloud = FALSE $THEN
            UNION ALL
            SELECT 'EXECUTE', 'PYQSCRIPTCREATE' FROM DUAL
        $END 
    ) LOOP
        BEGIN
            SELECT COUNT(*)
            INTO match_count
            FROM USER_TAB_PRIVS
            WHERE PRIVILEGE = tab_priv_check.privilege
            AND TABLE_NAME = tab_priv_check.table_name;

            IF match_count > 0 THEN
                DBMS_OUTPUT.PUT_LINE(
                    'Object privilege exists: ' || tab_priv_check.privilege ||
                    ' on ' || tab_priv_check.table_name
                );
            ELSE
                RAISE_APPLICATION_ERROR(
                    -20000,
                    'Missing object privilege: ' || tab_priv_check.privilege ||
                    ' on ' || tab_priv_check.table_name
                );
            END IF;
        END;
    END LOOP;
END check_user_privileges;
/
