/*
 * Decompiled with CFR 0.152.
 */
package oracle.dbtools.core.connections.storage;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;
import oracle.dbtools.core.collections.ConfigurationProperties;
import oracle.dbtools.core.connections.ConnectionIdentifiers;
import oracle.dbtools.core.connections.ConnectionsResources;
import oracle.dbtools.core.connections.api.Connections;
import oracle.dbtools.core.connections.storage.ConnectionDefinition;
import oracle.dbtools.core.connections.storage.ConnectionStorage;
import oracle.dbtools.core.io.file.StandardAccessPermissions;
import oracle.dbtools.core.util.MessageLogger;

public class StorageMigrator {
    private final ConnectionStorage origin;
    private final ConnectionStorage destination;
    private final MessageLogger logger;

    static Builder builder() {
        return new Builder();
    }

    public static void migrateConnections(Path oldRoot, MessageLogger logger) {
        ConnectionStorage old = ConnectionStorage.getPriorStorage(oldRoot);
        ConnectionStorage destination = ConnectionStorage.instance();
        if (old != null) {
            StorageMigrator migrator = StorageMigrator.builder().logger(logger).origin(old).destination(destination).build();
            migrator.migrate();
        }
    }

    private StorageMigrator(Builder builder) {
        this.origin = builder.origin;
        this.destination = builder.destination;
        this.logger = builder.logger;
    }

    void migrate() {
        List<String> oldConnections = this.origin.listConnectionDirectories();
        if (oldConnections.size() > 0) {
            HashSet<String> destConnections = new HashSet<String>(this.destination.listConnectionDirectories());
            HashSet connNames = new HashSet();
            destConnections.forEach(id -> {
                String connName = this.getConnectionName(this.destination, (String)id);
                connNames.add(connName);
            });
            oldConnections.forEach(id -> {
                String oldName = this.getConnectionName(this.origin, (String)id);
                if (oldName != null) {
                    this.migrateConnection((String)id, oldName, connNames);
                }
            });
        }
    }

    private String getConnectionName(ConnectionStorage store, String id) {
        String connName = null;
        Path dir = store.store.resolve(id);
        if (dir != null) {
            Path dbTools = dir.resolve("dbtools.properties");
            ConfigurationProperties dbToolsProps = ConnectionStorage.loadProperties(dbTools);
            connName = dbToolsProps.getProperty("name");
        }
        return connName;
    }

    private void migrateConnection(String dir, String connName, Set<String> names) {
        this.logger.logUserMessage(ConnectionsResources.format("MIGRATION_PROGRESS", connName));
        Object migratedName = connName;
        boolean rename = false;
        if (names.contains(connName)) {
            rename = true;
            migratedName = connName + "-migrated";
            if (names.contains(migratedName)) {
                this.logger.logUserMessage(ConnectionsResources.format("MIGRATION_ALREADY_MIGRATED", connName));
                return;
            }
            this.logger.logUserMessage(ConnectionsResources.format("MIGRATION_RENAMING", connName, migratedName));
        }
        Connections.Identifier id = ConnectionIdentifiers.createUniqueIdentifier();
        Path destDir = this.destination.resolvePath(id);
        try {
            this.copyDirectory(this.origin.store.resolve(dir), destDir);
        }
        catch (IOException ex) {
            this.logger.logUserMessage(ConnectionsResources.format("ERROR_MIGRATION_FAILED", migratedName));
            try {
                Files.walkFileTree(destDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        Files.delete(file);
                        return FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
                        if (e == null) {
                            Files.delete(dir);
                            return FileVisitResult.CONTINUE;
                        }
                        throw e;
                    }
                });
            }
            catch (IOException e) {
                this.logger.logError(ex.getLocalizedMessage(), ex);
            }
        }
        if (rename) {
            Path dbTools = destDir.resolve("dbtools.properties");
            ConfigurationProperties dbToolsProps = ConnectionStorage.loadProperties(dbTools);
            dbToolsProps = dbToolsProps.withProperty("name", (String)migratedName);
            try (OutputStream os = Files.newOutputStream(dbTools, new OpenOption[0]);){
                ConnectionDefinition.outputProperties(dbToolsProps, os);
            }
            catch (IOException ex) {
                this.logger.logError(ex.getLocalizedMessage(), ex);
            }
        }
    }

    private void copyDirectory(Path sourceDir, Path destDir) throws IOException {
        StandardAccessPermissions.SECURE_FOLDER.create(destDir);
        try (Stream<Path> stream = Files.list(sourceDir);){
            stream.forEach(p -> {
                try {
                    Path relative = sourceDir.relativize((Path)p);
                    Path child = destDir.resolve(relative);
                    if (Files.isDirectory(p, new LinkOption[0])) {
                        this.copyDirectory((Path)p, child);
                    } else {
                        this.copyFile((Path)p, child);
                    }
                }
                catch (IOException ex) {
                    throw AbortException.of(ex);
                }
            });
        }
        catch (AbortException ex) {
            throw (IOException)ex.getCause();
        }
    }

    private void copyFile(Path sourceFile, Path dest) throws IOException {
        Files.copy(sourceFile, dest, new CopyOption[0]);
    }

    static class Builder {
        private ConnectionStorage origin;
        private ConnectionStorage destination;
        private MessageLogger logger;

        Builder() {
        }

        Builder origin(ConnectionStorage origin) {
            this.origin = origin;
            return this;
        }

        Builder destination(ConnectionStorage destination) {
            this.destination = destination;
            return this;
        }

        Builder logger(MessageLogger logger) {
            this.logger = logger;
            return this;
        }

        public StorageMigrator build() {
            return new StorageMigrator(this);
        }
    }

    private static class AbortException
    extends RuntimeException {
        static AbortException of(IOException t) {
            return new AbortException(t);
        }

        private AbortException(Throwable t) {
            super(t);
        }
    }
}

