Rem
Rem $Header: dbgendev/src/langdata/plsql/utils/utils_pkg.pkb /main/71 2025/08/17 19:34:27 deveverm Exp $
Rem
Rem utils_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      utils_pkg.pkb - Package body for utility functions and procedures
Rem
Rem    DESCRIPTION
Rem      This package provides a wide range of utility functions and procedures 
Rem      used across the application
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/utils/utils_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    deveverm    08/08/25 - Added validate_vector, modified domain functions
Rem                           to remove min/max, gamma, vector_dimensions
Rem    arevathi    08/07/25 - DBAI-930: Acquire lock on table before alter
Rem                           statement
Rem    jiangnhu    08/05/25 - Remove validate_match_document, add
Rem                           validate_match_document_and_grant_table_privileges
Rem                           in lang_data pkg
Rem    arevathi    08/04/25 - DBAI-1138: set enable to false while creation of
Rem                           job
Rem    dadoshi     08/04/25 - JIRA_DBAI1147/1149: Added get_match_confidence
Rem    dadoshi     08/04/25 - JIRA_DBAI1147: Update get_or_create_domain to
Rem                           take in min/max distance, vector dimension for
Rem                           that namespace and gamma required as optional
Rem                           parameters
Rem    dadoshi     08/04/25 - JIRA_DBAI1147: Added get_domain_details
Rem    jiangnhu    08/04/25 - Move grant_read_on_user_tables to lang_data pkg
Rem    jiangnhu    07/30/25 - Call validate_match_document before
Rem                           grant_read_on_user_tables, remove duplicate logic 
Rem                           in grant_read_on_user_tables
Rem    fgurrola    07/29/25 - DBAI-1070: Change UPPER by normalization in the
Rem                           schema name.
Rem    jiangnhu    07/25/25 - DBAI-1128: Add null check for 'filters' in
Rem                           create_value_vector_job
Rem    ruohli      07/24/25 - DBAI-1076: Added update_domain, delete_domain, and
Rem                           get_all_domains
Rem    dadoshi     07/22/25 - JIRA_DBAI1098: Add get_text_by_id for APEX
Rem    arevathi    07/14/25 - Add grant_read_on_user_tables
Rem    fgurrola    07/10/25 - DBAI-1039: Fix bug in validate_match_document for
Rem                           NER.
Rem    jiangnhu    07/10/25 - DBAI-1006: Parameterize job class name
Rem    deveverm    07/09/25 - DBAI-976: replaced c_embedding_model_not_found
Rem                           with c_resource_already_exists
Rem    jiangnhu    07/07/25 - Throw c_job_creation_failed when 
Rem                           create_or_replace_job fail
Rem    fgurrola    06/27/25 - DBAI-776: Update validate_match_document
Rem                           function for predefined name entities.
Rem    deveverm    06/25/25 - DBAI-771: fix bug with get_embedding crashing due
Rem                           to printing vectors
Rem    jiangnhu    06/24/25 - DBAI-909: Integrate text augmentation with domain
Rem    ruohli      06/06/25 - DBAI-842: Added parallel query for
Rem                           alculate_report_description_regression and remove
Rem                           top 2 matching logic
Rem    jiangnhu    06/03/25 - Implement check_annotation_changes_for_user_tables
Rem                           replace DDL trigger with polling
Rem    jiangnhu    06/02/25 - DBAI-844: Merge langdata$value_vector_metadata,
Rem                           langdata$annotations_catalog and
Rem                           langdata$comments_catalog, remove
Rem                           update_annotation_and_comment_records
Rem    deveverm    06/02/25 - DBAI-794: shifted from DBMS_VECTOR to DBMS_CLOUD
Rem    jiangnhu    05/29/25 - DBAI-843: Add handling for Oracle system tables
Rem                           in check_table_exists
Rem    jiangnhu    05/28/25 - DBAI-576, DBAI-828: Implement
Rem                           sync_user_tables_changed_values,
Rem                           manage_langdata_tables_vector_indexes,
Rem                           background_fixup
Rem    jiangnhu    05/21/25 - DBAI-767: Implement parallel execution for
Rem                           augment_text, update create_value_vector_table,
Rem                           create_value_vector_from_enumerable_set,
Rem                           decrement_vector_table_references,
Rem                           drop_enumerable_set_based_value_vector_tables,
Rem                           check_annotation_changes_for_table_column
Rem                           add drop_value_vector_partition
Rem    dadoshi     05/20/25 - JIRA_DBAI804: Add print_clob for debugging
Rem    deveverm    05/16/25 - DBAI-761: added status to sample_queries
Rem    dadoshi     05/15/25 - JIRA_DBAI804: Add json_array_to_clob() function
Rem    jiangnhu    04/30/25 - DBAI-755: Use COMMENT to augment text
Rem    jiangnhu    04/29/25 - Fix return value of
Rem                           fetch_top_k_similarity_search_reports_using_centroid
Rem    jiangnhu    04/24/25 - DBAI-662: Extend the logic of filters: support
Rem                           table column in another PDB or database
Rem    deveverm    04/16/25 - DBAI-735: Added option to use OCI genAI embedding
Rem    dadoshi     04/11/25 - JIRA_DBAI525: Add 
Rem                           fetch_top_n_similairty_search_report_text
Rem    jiangnhu    04/02/25 - DBAI-624: Implement function get_embedding
Rem    jiangnhu    04/09/24 - DBAI-731: Create Context Index on Value Vector
Rem                           Tables instead of User Tables
Rem    dadoshi     26/03/25 - JIRA_DBAI689: Add is_stopword() function.
Rem    jiangnhu    04/04/25 - Move get_entities_from_text from utils pkg to 
Rem                           named entities pkg to avoid mutual dependency
Rem    dadoshi     04/02/25 - JIRA_DBAI704: Fix issue of Augmentation with
Rem                           token having trailing commas
Rem    jiangnhu    04/01/25 - DBAI-661: Implement
Rem                           drop_enumerable_set_value_vector_tables
Rem    deveverm    04/01/25 - DBAI-523: modified
Rem                           fetch_top_k_similarity_search_reports and
Rem                           fetch_top_k_similarity_search_drilldowns to add
Rem                           p_use_records, removed
Rem                           fetch_good_drilldown_description_change_search_records
Rem    dadoshi     03/26/25 - JIRA_DBAI689: Add is_stopword() function.
Rem    jiangnhu    03/26/25 - DBAI-692: Make enumeration limit config
Rem    jiangnhu    03/25/25 - Fix match_substring_with_context to escape 
Rem                           characters like parenthesis
Rem    jiangnhu    03/24/25 - DBAI-551: Implement augment_text_with_ner_entities
Rem    anisbans    03/24/25 - DBAI-518: Update table name
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/14/25 - DBAI-661: Extend the logic of filters: support
Rem                           enumerable distinct value set instead of directly
Rem                           associated with a table column
Rem    anisbans    03/13/25 - DBAI-518: Update table name
Rem    deveverm    03/11/25 - DBAI-546: added schema_name for cross_schema
Rem                           support
Rem    anisbans    03/11/25 - DBAI-556: update create_context_index and add 
Rem                                     decrement_user_table_index_references  
Rem    jiangnhu    03/06/25 - DBAI-542: Add get_schema_version for backend
Rem    jiangnhu    03/06/25 - DBAI-632: Implement update_annotation_records
Rem    jiangnhu    02/28/25 - Update augment_text, match_substring_with_context
Rem                           to handle the case when config variables not set;
Rem                           Deduplicate augmentation text
Rem    jiangnhu    02/24/25 - Update check_table_exists to have parameter
Rem                           is_user_table
Rem    dadoshi     02/24/25 - JIRA_DBAI578: Add config_pkg usage
Rem    anisbans    02/19/25 - JIRA_DBAI-557: Apply ref count for value vectors  
Rem    jiangnhu    02/18/25 - DBAI-542: Change user_tables to all_tables,
Rem                           user_tab_columns to all_tab_columns,
Rem                           user_annotations_usage to all_annotations_usage;
Rem                           Move lang_data_create_value_vectors to utils pkg
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    02/13/25 - DBAI-524, DBAI-555: Implement
Rem                           create_context_index, create_user_table_indexes,
Rem                           recreate_user_index
Rem    saloshah    02/11/25 - Added update_job_status and get_all_jobs procedure
Rem    jiangnhu    02/03/25 - DBAI-511: Update
Rem                           check_annotation_changes_table_column, update
Rem                           amend_description_no_update to
Rem                           generate_amendment_text
Rem    jiangnhu    01/31/25 - DBAI-511: Update function augment_query to
Rem                           procedure augment_text, add augmented_tokens
Rem    jiangnhu    01/29/25 - DBAI-505: Implement centroid version of
Rem                           search_from_query procedure
Rem    dadoshi     12/22/24 - Add check_annotation_changes_for_table_column()
Rem                           from annotations_pkg
Rem    jiangnhu    12/19/24 - JIRA_DBAI-451: Multiple fuzzy match during
Rem                           tokenization
Rem    jiangnhu    12/03/24 - JIRA_DBAI-421: Add create_or_replace_job
Rem    jiangnhu    12/03/24 - JIRA_DBAI-383: Update augment_query to use user
Rem                          table indexes
Rem    saloshah    11/28/24 - DBAI-327: Added get_filter_descriptions and
Rem                           amend_description_no_update
Rem    jiangnhu    11/26/24 - Add check_column_exists
Rem    jiangnhu    11/22/24 - JIRA_DBAI-425: Add get_filter_description,
Rem                           check_table_exists,
Rem                           fetch_top_k_similarity_search_reports
Rem    arevathi    11/12/24 - JIRA_DBAI345: added update_table_column
Rem    saloshah    11/05/24 - DBAI-414 : Fix amend description
Rem    arevathi    10/30/24 - Added
Rem                           fetch_good_drilldown_description_change_search_records
Rem    arevathi    10/29/24 - Added fetch_good_search_records and
Rem                           fetch_top_k_similarity_search_reports
Rem    dadoshi     10/29/24 - Format utils_pkg
Rem    jiangnhu    10/29/24 - JIRA_DBAI-403: Modify amend_description
Rem    dadoshi     10/25/24 - Add fetch_good_drilldown_search_records and
Rem                           fetch_top_k_similarity_search_drilldowns
Rem    dadoshi     10/25/24 - Add update_table_column()
Rem    dadoshi     10/25/24 - Fix get_sample_queries_paginated in utils
Rem    dadoshi     10/25/24 - Update fetch_and_update_filter_description()
Rem    dadoshi     10/25/24 - Add amend_description() to utils
Rem    deveverm    10/24/24 - added get_job
Rem    jiangnhu    10/22/24 - Format code
Rem    jiangnhu    10/18/24 - Fix text wrapping
Rem    jiangnhu    10/17/24 - Modify header
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data_utils_pkg AS

    -- Implementation of split_cursor function
    PROCEDURE split_cursor (
        p_cursor IN VARCHAR2,
        p_created_at OUT TIMESTAMP,
        p_id OUT VARCHAR2
    ) IS
    BEGIN
        IF p_cursor IS NOT NULL THEN
            -- Extract created_at timestamp and id from the cursor
            p_created_at := TO_TIMESTAMP(
                SUBSTR(p_cursor, 1, INSTR(p_cursor, '|') - 1),
                'YYYY-MM-DD HH24:MI:SS.FF'
            );
            p_id := SUBSTR(p_cursor, INSTR(p_cursor, '|') + 1);

            -- Log the split details
            lang_data_logger_pkg.log_info(
                'Cursor split: CreatedAt=' ||
                TO_CHAR(p_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') ||
                ', ID=' || p_id
            );
        ELSE
            -- Set NULL values if cursor is not provided
            p_created_at := NULL;
            p_id := NULL;
        END IF;
    END split_cursor;

    PROCEDURE validate_enumerable_column (
        p_table_name     IN VARCHAR2,
        p_column_name    IN VARCHAR2,
        p_schema_name    IN VARCHAR2,
        p_db_link_name   IN VARCHAR2 DEFAULT NULL
    ) IS
        v_sql           VARCHAR2(4000);
        v_data_type     VARCHAR2(255);
        v_unique_count  NUMBER := 0;
    BEGIN
        -- 1. Construct SQL to get the data type of the column
        v_sql := 'SELECT data_type FROM all_tab_columns';
        IF p_db_link_name IS NOT NULL THEN
            v_sql := v_sql || '@' || p_db_link_name;
        END IF;
        v_sql := v_sql ||
            ' WHERE table_name = :tbl AND column_name = :col AND owner = :own';

        BEGIN
            EXECUTE IMMEDIATE v_sql INTO v_data_type
            USING UPPER(p_table_name), UPPER(p_column_name), p_schema_name;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                lang_data_logger_pkg.log_debug('No matching column found.');
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_match_document
                );
        END;

        -- 2. Construct SQL to check uniqueness of the column
        IF p_db_link_name IS NOT NULL THEN
            v_sql := '
                SELECT COUNT(*)
                FROM all_ind_columns@' || p_db_link_name || ' ic
                JOIN all_indexes@' || p_db_link_name || ' i
                ON ic.index_name = i.index_name
                AND ic.table_owner = i.table_owner
                WHERE i.uniqueness = ''UNIQUE''
                AND ic.table_name = :tbl
                AND ic.column_name = :col
                AND ic.table_owner = :own';
        ELSE
            v_sql := '
                SELECT COUNT(*)
                FROM all_ind_columns ic
                JOIN all_indexes i
                ON ic.index_name = i.index_name
                AND ic.table_owner = i.table_owner
                WHERE i.uniqueness = ''UNIQUE''
                AND ic.table_name = :tbl
                AND ic.column_name = :col
                AND ic.table_owner = :own';
        END IF;

        EXECUTE IMMEDIATE v_sql INTO v_unique_count
        USING UPPER(p_table_name), UPPER(p_column_name), p_schema_name;

        -- 3. Validate whether column is enumerable
        IF v_data_type NOT IN ('VARCHAR2', 'NUMBER', 'CHAR')
          AND v_unique_count = 0 THEN
            lang_data_logger_pkg.log_debug(
                'Column ' || p_column_name || ' in table ' || p_table_name ||
                ' is not enumerable and cannot be used as a filter.'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_match_document
            );
        END IF;
    END validate_enumerable_column;

    FUNCTION get_annotation(
        p_table_name        VARCHAR2,
        p_column_name       VARCHAR2,
        p_schema_name       VARCHAR2,
        p_annotation_name   VARCHAR2 DEFAULT 'DESCRIPTION',
        p_db_link_name      VARCHAR2 DEFAULT NULL
    ) RETURN VARCHAR2 IS
        v_original_description VARCHAR2(4000);
        v_table_name           VARCHAR2(128);
        v_column_name          VARCHAR2(128);
        v_schema_name          VARCHAR2(128);
        v_sql                  VARCHAR2(4000);
    BEGIN
        -- Normalize input names to uppercase to ensure case-insensitive matching
        v_table_name  := UPPER(p_table_name);
        v_column_name := UPPER(p_column_name);
        v_schema_name := lang_data_utils_pkg.normalize_schema_name(
            p_schema_name
        );

        -- Dynamically build SQL to support remote access via database link
        v_sql := '
            SELECT annotation_value
            FROM all_annotations_usage';

        -- Append the DB link if provided
        IF p_db_link_name IS NOT NULL THEN
            v_sql := v_sql || '@' || p_db_link_name;
        END IF;

        -- Add the WHERE clause with bind variables
        v_sql := v_sql || '
            WHERE object_name = :tbl_name
            AND object_type = ''TABLE''
            AND column_name = :col_name
            AND annotation_name = :ann_name
            AND annotation_owner = :schema_name';

        BEGIN
            -- Execute the dynamic query with bind variables
            EXECUTE IMMEDIATE v_sql
            INTO v_original_description
            USING v_table_name, v_column_name, UPPER(p_annotation_name),
                  v_schema_name;

        EXCEPTION
            -- Return NULL on any error (e.g., no data found, DB link unreachable)
            WHEN OTHERS THEN
                v_original_description := NULL;
        END;

        RETURN v_original_description;
    END get_annotation;

    -- Note: a column of any particular table can only have almost one comment
    FUNCTION get_comment(
        p_table_name    VARCHAR2,
        p_column_name   VARCHAR2,
        p_schema_name   VARCHAR2 DEFAULT NULL,
        p_db_link_name  VARCHAR2 DEFAULT NULL
    ) RETURN VARCHAR2 IS
        v_table_name   VARCHAR2(128);
        v_column_name  VARCHAR2(128);
        v_schema_name  VARCHAR2(128);
        v_comment      VARCHAR2(4000);
        v_sql          VARCHAR2(4000);
    BEGIN
        -- Normalize input names to uppercase to ensure case-insensitive matching
        v_table_name  := UPPER(p_table_name);
        v_column_name := UPPER(p_column_name);
        v_schema_name := lang_data_utils_pkg.normalize_schema_name(
            p_schema_name
        );

        -- Dynamically build SQL to support remote access via database link
        v_sql := '
            SELECT comments
            FROM all_col_comments';

        -- Append the DB link if provided
        IF p_db_link_name IS NOT NULL THEN
            v_sql := v_sql || '@' || p_db_link_name;
        END IF;

        -- Add the WHERE clause with bind variables
        v_sql := v_sql || '
            WHERE owner = :schema_name
            AND table_name = :table_name
            AND column_name = :column_name';

        BEGIN
            -- Execute the dynamic query with bind variables
            EXECUTE IMMEDIATE v_sql
            INTO v_comment
            USING v_schema_name, v_table_name, v_column_name;
        EXCEPTION
            -- Return NULL on any error (e.g., no data found, DB link unreachable)
            WHEN OTHERS THEN
                v_comment := NULL;
        END;

        RETURN v_comment;
    END get_comment;

    FUNCTION get_filter_description (
        p_filter_name           IN  VARCHAR2,
        p_table_name            IN  VARCHAR2 DEFAULT NULL,
        p_column_name           IN  VARCHAR2 DEFAULT NULL,
        p_schema_name           IN  VARCHAR2 DEFAULT NULL,
        p_additional_description IN  VARCHAR2 DEFAULT NULL,
        p_db_link_name           IN  VARCHAR2 DEFAULT NULL
    ) RETURN VARCHAR2 AS
        v_original_description   VARCHAR2(4000) := NULL;
        v_comment       VARCHAR2(4000) := NULL;
        v_additional_description VARCHAR2(4000) := p_additional_description;
        v_filter_description    VARCHAR2(4000);
        v_table_name        VARCHAR2(128);
        v_column_name       VARCHAR2(128);
        v_schema_name       VARCHAR2(128);
        v_db_link_name      VARCHAR2(128);
    BEGIN
        v_table_name := UPPER(p_table_name);
        v_column_name := UPPER(p_column_name);
        v_schema_name := lang_data_utils_pkg.normalize_schema_name(
            p_schema_name
        );
        v_db_link_name := UPPER(p_db_link_name);

        -- Check if table_name and column_name are provided
        IF v_table_name IS NOT NULL AND v_column_name IS NOT NULL 
            AND v_schema_name IS NOT NULL THEN
            v_original_description := lang_data_utils_pkg.get_annotation(
                v_table_name,
                v_column_name,
                v_schema_name,
                'DESCRIPTION',
                v_db_link_name
            );
            v_comment := lang_data_utils_pkg.get_comment(
                v_table_name, v_column_name, v_schema_name, v_db_link_name
            );
            IF v_comment IS NOT NULL AND v_original_description IS NOT NULL THEN
                -- If the comment is a substring of original description
                IF INSTR(v_original_description, v_comment) > 0 THEN
                    v_comment := NULL;
                END IF;
                -- If the original description is a substring of comment
                IF INSTR(v_comment, v_original_description) > 0 THEN
                    v_original_description := NULL;
                END IF;
            END IF;
            v_original_description := NVL(v_original_description, '') || ' ' ||
                                      NVL(v_comment, '');
        END IF;

        -- Check if additional description is valid
        IF v_additional_description IS NOT NULL AND 
          INSTR(v_original_description, v_additional_description) > 0 THEN
            v_additional_description := NULL;
        END IF;

        -- Combine descriptions
        IF v_original_description IS NOT NULL
            OR v_additional_description IS NOT NULL THEN
            v_filter_description := TRIM(
                p_filter_name || ': ' || NVL(v_original_description, '') ||
                ' ' || NVL(v_additional_description, '')
            );
        ELSE
            v_filter_description := p_filter_name;
        END IF;

        RETURN v_filter_description;
    END get_filter_description;

    FUNCTION fetch_and_update_filter_descriptions (
        p_match_document JSON
    ) RETURN VARCHAR2 IS
        v_filter_descriptions VARCHAR2(4000);
        v_table_name VARCHAR2(255);
        v_column_name VARCHAR2(255);
        v_schema_name VARCHAR2(255);
        v_db_link_name VARCHAR2(255);
        v_filter_name VARCHAR2(255);
        v_additional_description VARCHAR2(4000);
        v_original_description VARCHAR2(4000);
        v_comment VARCHAR2(4000);
        v_updated_description VARCHAR2(4000);
        v_annotation_value VARCHAR2(4000);
        v_operation VARCHAR2(10);
        v_old_enumerable_annotation VARCHAR2(10);
        v_new_enumerable_annotation VARCHAR2(10);
        v_sql VARCHAR2(4000);
    BEGIN
        v_filter_descriptions := '';

        -- Extract filters using JSON_TABLE
        FOR rec IN (
            SELECT jt.filter_name,
                jt.table_name,
                jt.column_name,
                jt.schema_name,
                jt.description,
                jt.use_ner,
                jt.db_link_name
            FROM JSON_TABLE(
                    p_match_document,
                    '$.filters[*]' COLUMNS (
                        filter_name         VARCHAR2(255) PATH '$.filter_name',
                        table_name          VARCHAR2(255) PATH '$.table_name',
                        column_name         VARCHAR2(255) PATH '$.column_name',
                        schema_name         VARCHAR2(255) PATH '$.schema_name',
                        description         VARCHAR2(4000) PATH '$.description',
                        use_ner             BOOLEAN PATH '$.use_ner',
                        db_link_name        VARCHAR2(255) PATH '$.db_link_name'
                    )
            ) jt
        ) LOOP
            v_table_name := UPPER(rec.table_name);
            v_column_name := UPPER(rec.column_name);
            v_schema_name := lang_data_utils_pkg.normalize_schema_name(
                rec.schema_name
            );
            v_db_link_name := UPPER(rec.db_link_name);

            IF v_schema_name IS NULL 
                OR v_table_name IS NULL 
                OR v_column_name IS NULL THEN
                v_filter_descriptions := v_filter_descriptions ||
                                        rec.filter_name || ': ' ||
                                        rec.description || ', ';
                CONTINUE;
            END IF;
            v_old_enumerable_annotation := lang_data_utils_pkg.get_annotation(
                v_table_name,
                v_column_name,
                v_schema_name,
                'ENUMERABLE',
                v_db_link_name
            );
            IF rec.use_ner THEN
                v_new_enumerable_annotation := 'No';
            ELSE
                v_new_enumerable_annotation := 'Yes';
            END IF;
            IF v_old_enumerable_annotation != v_new_enumerable_annotation THEN
                IF v_old_enumerable_annotation IS NULL THEN
                    v_operation := 'ADD';
                ELSE
                    v_operation := 'REPLACE';
                    lang_data_logger_pkg.log_warn(
                        'Changing enumerable annotation on table column: ' ||
                        v_schema_name|| '.' || v_table_name || '.' ||
                        v_column_name
                    );
                END IF;
                IF v_db_link_name IS NULL THEN
                    EXECUTE IMMEDIATE 'ALTER TABLE ' || v_schema_name|| '.'||
                                v_table_name || ' MODIFY ' || v_column_name ||
                                ' ANNOTATIONS (' || v_operation ||
                                ' Enumerable ''' ||
                                v_new_enumerable_annotation || ''')';
                ELSE
                    lang_data_logger_pkg.log_warn(
                        'Remote column annotations cannot be updated ' ||
                        'via DB Link: ' || v_schema_name || '.' ||
                        v_table_name || '.' || v_column_name
                    );
                END IF;
            END IF;
            
            -- Fetch the annotation for table and column
            v_annotation_value := lang_data_utils_pkg.get_annotation(
                v_table_name,
                v_column_name,
                v_schema_name,
                'DESCRIPTION',
                v_db_link_name
            );
            IF v_annotation_value IS NOT NULL THEN
                v_original_description := v_annotation_value;
                v_operation := 'REPLACE';
            ELSE
                v_original_description := '';
                v_operation := 'ADD';
            END IF;

            -- Append additional description if provided and not included yet
            IF rec.description IS NOT NULL AND
              INSTR(v_original_description, rec.description) = 0 THEN
                v_updated_description := v_original_description || ' ' ||
                                        rec.description;
            ELSE
                v_updated_description := v_original_description;
            END IF;

            IF v_updated_description != v_original_description THEN
                -- Update annotation with ALTER TABLE
                IF v_db_link_name IS NULL THEN
                    EXECUTE IMMEDIATE 'ALTER TABLE ' || 
                                v_schema_name ||'.'|| v_table_name ||
                                ' MODIFY ' || v_column_name ||
                                ' ANNOTATIONS (' || v_operation ||
                                ' Description ''' || v_updated_description ||
                                ''')';

                    -- Call procedure to update vectors after annotation changes
                    BEGIN
                        check_annotation_changes_for_table_column(
                            UPPER(v_table_name),
                            UPPER(v_column_name),
                            v_schema_name
                        );
                    EXCEPTION
                        WHEN OTHERS THEN
                            lang_data_logger_pkg.log_fatal(
                                'Error updating vectors for ' ||
                                v_schema_name ||'.'|| UPPER(v_table_name) ||
                                '.' || v_column_name
                            );
                            RAISE;
                    END;
                ELSE
                    lang_data_logger_pkg.log_warn(
                        'Remote column description cannot be updated ' ||
                        'via DB Link: ' || v_schema_name || '.' ||
                        v_table_name || '.' || v_column_name
                    );
                END IF;
            END IF;

            v_comment := lang_data_utils_pkg.get_comment(
                v_table_name, v_column_name, v_schema_name, v_db_link_name
            );

            IF v_comment IS NOT NULL AND 
              -- If comment is not a substring of updated description
              INSTR(v_updated_description, v_comment) = 0 THEN
                v_updated_description := v_updated_description || ' ' ||
                                         v_comment;
            END IF;

            v_filter_descriptions := v_filter_descriptions || rec.filter_name ||
                                    ': ' || v_updated_description || ', ';
        END LOOP;

        -- Remove trailing comma and space
        v_filter_descriptions := RTRIM(v_filter_descriptions, ', ');

        RETURN v_filter_descriptions;
    END fetch_and_update_filter_descriptions;

    FUNCTION expand_text (
        p_match_document JSON,
        p_original_text VARCHAR2,
        p_text_type VARCHAR2
    ) RETURN VARCHAR2
    IS
        v_expanded_text VARCHAR2(4000);
        v_filter_descriptions VARCHAR2(4000);
    BEGIN
        -- Logging the input parameters
        lang_data_logger_pkg.log_info(
            'Original Text: ' || p_original_text
        );
        lang_data_logger_pkg.log_info(
            'Text Type: ' || p_text_type
        );

        -- If the match document is null, log and return original text
        IF p_match_document IS NULL THEN
            lang_data_logger_pkg.log_info(
                'Match document is null, returning original text.'
            );
            RETURN p_original_text;
        END IF;

        -- Log that the function is fetching filter descriptions
        lang_data_logger_pkg.log_info('Fetching filter descriptions...');

        -- Get filter descriptions
        v_filter_descriptions :=
            lang_data_utils_pkg.fetch_and_update_filter_descriptions(
                p_match_document
            );

        -- Log the retrieved filter descriptions
        lang_data_logger_pkg.log_info(
            'Filter Descriptions: ' ||
            NVL(v_filter_descriptions, 'No filters')
        );

        -- Initialize the expanded text with the original text
        v_expanded_text := p_original_text;

        -- If there are filter descriptions, append them to the original
        -- text
        IF v_filter_descriptions IS NOT NULL AND
           LENGTH(TRIM(v_filter_descriptions)) > 0 THEN
            IF p_text_type = 'query' THEN
                v_expanded_text := v_expanded_text ||
                    ' The query uses the following filters: ' ||
                    v_filter_descriptions || '.';
            ELSE
                v_expanded_text := v_expanded_text ||
                    ' We need the following filters to use this ' ||
                    p_text_type || ': ' || v_filter_descriptions || '.';
            END IF;

            -- Log the expanded text
            lang_data_logger_pkg.log_info(
                'Expanded Text: ' || v_expanded_text
            );
        ELSE
            lang_data_logger_pkg.log_info('No filter descriptions to append.');
        END IF;

        -- Return the expanded text
        RETURN v_expanded_text;
    END expand_text;

    FUNCTION check_table_exists (
        p_table_name    IN VARCHAR2,
        p_schema_name   IN VARCHAR2 DEFAULT NULL,
        p_db_link_name  IN VARCHAR2 DEFAULT NULL
    ) RETURN BOOLEAN IS
        v_count NUMBER;
        v_sql   VARCHAR2(4000);
    BEGIN
        -- Use ALL_OBJECTS instead of ALL_TABLES to cover tables, views, system 
        -- objects, etc.
        v_sql := 'SELECT COUNT(*) FROM all_objects';

        -- If a database link is provided, append it to the query
        IF p_db_link_name IS NOT NULL THEN
            v_sql := v_sql || '@' || p_db_link_name;
        END IF;

        -- Add the WHERE clause for table name
        v_sql := v_sql || ' WHERE object_name = :tbl_name ' ||
                'AND object_type IN (''TABLE'', ''VIEW'')';

        -- Execute the dynamic SQL, using bind variables for safety
        IF p_schema_name IS NOT NULL THEN
            v_sql := v_sql || ' AND owner = :schema_name';
        END IF;

        IF p_schema_name IS NOT NULL THEN
            EXECUTE IMMEDIATE v_sql INTO v_count
            USING UPPER(p_table_name), p_schema_name;
        ELSE
            EXECUTE IMMEDIATE v_sql INTO v_count
            USING UPPER(p_table_name);
        END IF;

        -- Return true if the table was found (count > 0), otherwise false
        RETURN v_count > 0;

    EXCEPTION
        -- On any error (e.g., invalid DB link, permission denied), return false
        WHEN OTHERS THEN
            RETURN FALSE;
    END check_table_exists;

    FUNCTION check_column_exists (
        p_table_name    IN VARCHAR2,
        p_column_name   IN VARCHAR2,
        p_schema_name   IN VARCHAR2 DEFAULT NULL,
        p_db_link_name  IN VARCHAR2 DEFAULT NULL
    ) RETURN BOOLEAN IS
        v_count NUMBER;
        v_sql   VARCHAR2(4000);
    BEGIN
        -- Start building the base SQL query for checking column existence
        v_sql := 'SELECT COUNT(*) FROM all_tab_columns';

        -- Append the database link if provided
        IF p_db_link_name IS NOT NULL THEN
            v_sql := v_sql || '@' || p_db_link_name;
        END IF;

        -- Add the WHERE clause with table name and column name filters
        v_sql := v_sql ||
                 ' WHERE table_name = :tbl_name AND column_name = :col_name';

        -- Add schema owner filter if provided
        IF p_schema_name IS NOT NULL THEN
            v_sql := v_sql || ' AND owner = :schema_name';
        END IF;

        -- Execute the dynamic SQL with bind variables to avoid SQL injection
        IF p_schema_name IS NOT NULL THEN
            EXECUTE IMMEDIATE v_sql INTO v_count
            USING UPPER(p_table_name), UPPER(p_column_name),  p_schema_name;
        ELSE
            EXECUTE IMMEDIATE v_sql INTO v_count
            USING UPPER(p_table_name), UPPER(p_column_name);
        END IF;

        -- Return true if the column exists (count = 1), otherwise false
        RETURN v_count > 0;

    EXCEPTION
        -- On any error (e.g., invalid link, permission issues), return false
        WHEN OTHERS THEN
            RETURN FALSE;
    END check_column_exists;

    -- Function to check if the report ID exists in langdata$reports
    FUNCTION check_report_exists (
        p_report_id IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_count NUMBER;
    BEGIN
        -- Check if the report ID exists
        SELECT COUNT(*) INTO v_count
        FROM langdata$reports
        WHERE id = p_report_id;

        -- Return TRUE if report exists, FALSE otherwise
        RETURN v_count > 0;
    END check_report_exists;

    -- Function to check if the drilldown document ID exists in
    -- langdata$drilldowndocuments
    FUNCTION check_drilldown_exists (
        p_drilldown_id IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_count NUMBER;
    BEGIN
        -- Check if the drilldown document ID exists
        SELECT COUNT(*) INTO v_count
        FROM langdata$drilldowndocuments
        WHERE id = p_drilldown_id;

        -- Return TRUE if drilldown exists, FALSE otherwise
        RETURN v_count > 0;
    END check_drilldown_exists;

    FUNCTION check_db_link_exists (
        p_db_link_name  IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_count NUMBER;
    BEGIN
        SELECT COUNT(*)
        INTO v_count
        FROM all_db_links
        WHERE UPPER(db_link) LIKE UPPER(p_db_link_name) || '%';

        RETURN v_count > 0;
    END check_db_link_exists;

    FUNCTION is_stopword (
        p_word IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_count NUMBER;
    BEGIN
        SELECT COUNT(*) INTO v_count
        FROM langdata$stoplist
        where lower(spw_word) = lower(p_word);

        RETURN v_count>0;
    END;

    FUNCTION match_substring_with_context (
        p_text           IN VARCHAR2,
        p_substring      IN VARCHAR2,
        p_num_words_before IN NUMBER DEFAULT 5,
        p_num_words_after  IN NUMBER DEFAULT 2
    ) RETURN VARCHAR2 AS
        v_context_match VARCHAR2(4000); -- Variable for the matched context
        v_escaped_substring VARCHAR2(4000);
    BEGIN
        v_escaped_substring := REGEXP_REPLACE(
            p_substring,
            '([\\.^$*+?(){}\[\]|])',
            '\\\1'
        );

        -- Use regular expressions to find the context
        SELECT REGEXP_SUBSTR(
                p_text,
                '((\S+\s+){0,' || NVL(p_num_words_before, 5) || '})(' || 
                v_escaped_substring ||
                ')(\s+\S+){0,' || NVL(p_num_words_after, 2) || '}',
                1, -- Start position
                1, -- Occurrence
                'i' -- Case-insensitive matching
            )
        INTO v_context_match
        FROM DUAL;

        -- Handle the case when no match is found
        IF v_context_match IS NULL THEN
            RETURN 'No Match';
        END IF;

        RETURN TRIM(v_context_match);
    END match_substring_with_context;

    PROCEDURE augment_text(
        p_text               IN VARCHAR2,
        p_domain_id          IN VARCHAR2 DEFAULT NULL,
        p_augmented_text     OUT VARCHAR2,
        p_augmented_tokens   OUT JSON
    ) IS
        -- To store the final augmented query
        v_preprocessed_query  VARCHAR2(4000);
        -- Individual token from the query
        v_token               VARCHAR2(255);
        v_best_match_token    VARCHAR2(255);
        v_current_match_token VARCHAR2(255);
        -- Description fetched from the lookup table
        v_description         VARCHAR2(4000);
        v_description_vector  VECTOR(*,*);
        -- Normalized token
        v_normalized_token    VARCHAR2(255);
        v_prev_context_len    NUMBER;
        v_post_context_len    NUMBER;
        v_token_with_context  VARCHAR2(255);
        v_token_with_context_vector  VECTOR(*,*);
        -- Threshold for fuzzy matching
        v_threshold           NUMBER := 80;

        -- DBMS_SQL variables
        cursor_id             INTEGER;
        rows_processed        INTEGER;
        column_value          VARCHAR2(4000);
        v_best_match_dist     NUMBER := 9999999;
        v_best_match_desc     VARCHAR2(4000);
        v_description_exists BOOLEAN := FALSE;
        v_cur_match_dist      NUMBER;
        augmented_tokens_arr  JSON_ARRAY_T := JSON_ARRAY_T();
        TYPE set_type IS TABLE OF BOOLEAN INDEX BY VARCHAR2(4000);
        description_set set_type;
    BEGIN
        -- Initialize the final query
        p_augmented_text := '';

        v_prev_context_len := NVL(
            TO_NUMBER(
                lang_data_config_pkg.get_config_parameter(
                    'LANG_DATA_PREV_CONTEXT_LEN'
                )
            ),
            5
        );

        v_post_context_len := NVL(
            TO_NUMBER(
                lang_data_config_pkg.get_config_parameter(
                    'LANG_DATA_POST_CONTEXT_LEN'
                )
            ),
            2
        );

        v_threshold := NVL(
            TO_NUMBER(
                    lang_data_config_pkg.get_config_parameter(
                        'LANG_DATA_AUGMENT_QUERY_FUZZY_MATCH_THRESHOLD'
                    )
            ),
            70
        );
        
        -- Split the query into tokens using a space delimiter and loop through
        FOR token_rec IN (
            SELECT REGEXP_SUBSTR(p_text, '[^ ]+', 1, LEVEL) AS token
            FROM dual
            CONNECT BY REGEXP_SUBSTR(p_text, '[^ ]+', 1, LEVEL) IS NOT NULL
        ) LOOP
            -- Normalize the token by converting to lowercase
            v_normalized_token := LOWER(token_rec.token);

            IF is_stopword(v_normalized_token) THEN
                p_augmented_text := p_augmented_text ||
                                    token_rec.token || ' ';
                continue;
            END IF;

            -- Skip the token if it is a number
            IF REGEXP_LIKE(v_normalized_token, '^\d+$') THEN
                -- Append the number token as is, without augmentation
                p_augmented_text := p_augmented_text ||
                                    token_rec.token || ' ';
                CONTINUE;
            END IF;

            v_token_with_context := match_substring_with_context(
                p_text,
                token_rec.token,
                v_prev_context_len,
                v_post_context_len
            );
            v_token_with_context_vector := lang_data_utils_pkg.get_embedding(
                v_token_with_context
            );

            -- Initialize the description as NULL
            v_description := NULL;
            v_best_match_dist := 9999999;
            v_best_match_desc := NULL;

            FOR rec IN (
                SELECT /*+ PARALLEL(d 4) */
                    value,
                    d.partition_name,
                    desc_comment_combined,
                    desc_comment_vector
                FROM langdata$drilldownvalues d
                JOIN langdata$value_vector_partition_descriptions v
                ON d.partition_name = v.partition_name
                WHERE (p_domain_id IS NULL OR v.domain_id = p_domain_id)
                  AND CONTAINS(
                        value, 
                        'FUZZY({' || v_normalized_token || '}, ' || v_threshold || ')', 
                        1
                    ) > 0
                ORDER BY VECTOR_DISTANCE(v_token_with_context_vector, desc_comment_vector)
                FETCH FIRST 1 ROW ONLY
            ) LOOP
                v_best_match_desc := rec.desc_comment_combined;
                BEGIN
                    SELECT 
                        schema_name || '_' || table_name || '_' ||
                        column_name || 
                        CASE 
                            WHEN db_link_name IS NOT NULL
                            THEN '_' || db_link_name 
                            ELSE '' 
                        END || 
                        '_' || rec.value
                    INTO v_best_match_token
                    FROM langdata$value_vector_metadata
                    WHERE vvec_table_name = rec.partition_name;

                EXCEPTION
                    WHEN NO_DATA_FOUND THEN
                        v_best_match_token := NULL;
                END;
            END LOOP;

            -- Append the description to the token if it exists
            IF v_best_match_desc IS NOT NULL AND 
                NOT description_set.EXISTS(v_best_match_desc) THEN
                description_set(v_best_match_desc) := TRUE;
                p_augmented_text := p_augmented_text ||
                                    token_rec.token || '(' ||
                                    v_best_match_desc || ') ';
                
                IF v_best_match_token IS NOT NULL THEN
                    augmented_tokens_arr.APPEND(v_best_match_token);
                END IF;
            ELSE
                p_augmented_text := p_augmented_text ||
                                    token_rec.token || ' ';
            END IF;
        END LOOP;

        p_augmented_text := RTRIM(p_augmented_text);
        p_augmented_tokens := JSON(augmented_tokens_arr.TO_STRING);
    END augment_text;

    FUNCTION get_filter_descriptions (
        p_match_document IN JSON
    ) RETURN VARCHAR2 IS
        v_filter_descriptions VARCHAR2(4000);
        v_table_name VARCHAR2(255);
        v_column_name VARCHAR2(255);
        v_schema_name VARCHAR2(255);
        v_filter_name VARCHAR2(255);
        v_db_link_name VARCHAR2(255);
        v_additional_description VARCHAR2(4000);
        v_original_description VARCHAR2(4000);
        v_updated_description VARCHAR2(4000);
        v_annotation_value VARCHAR2(4000);
        v_comment VARCHAR2(4000);
    BEGIN
        v_filter_descriptions := '';

        -- Extract filters using JSON_TABLE
        FOR rec IN (
            SELECT jt.filter_name,
                jt.table_name,
                jt.column_name,
                jt.schema_name,
                jt.db_link_name,
                jt.description
            FROM JSON_TABLE(
                    p_match_document,
                    '$.filters[*]' COLUMNS (
                        filter_name     VARCHAR2(255) PATH '$.filter_name',
                        table_name      VARCHAR2(255) PATH '$.table_name',
                        column_name     VARCHAR2(255) PATH '$.column_name',
                        schema_name     VARCHAR2(255) PATH '$.schema_name',
                        db_link_name    VARCHAR2(255) PATH '$.db_link_name',
                        description     VARCHAR2(4000) PATH '$.description'
                    )
            ) jt
        ) LOOP
            -- Fetch the annotation for table and column
            v_table_name := UPPER(rec.table_name);
            v_schema_name := lang_data_utils_pkg.normalize_schema_name(
                rec.schema_name
            );
            v_column_name := UPPER(rec.column_name);
            v_db_link_name := UPPER(rec.db_link_name);

            v_annotation_value := lang_data_utils_pkg.get_annotation(
                v_table_name,
                v_column_name,
                v_schema_name,
                'DESCRIPTION',
                v_db_link_name
            );
            IF v_annotation_value IS NOT NULL THEN
                v_original_description := v_annotation_value;
            ELSE
                v_original_description := '';
            END IF;

            -- Append additional description if provided and not included
            IF rec.description IS NOT NULL AND 
                INSTR(v_original_description, rec.description) = 0 THEN
                    v_updated_description := v_original_description || ' ' || 
                                            rec.description;
            ELSE
                v_updated_description := v_original_description;
            END IF;

            v_comment := lang_data_utils_pkg.get_comment(
                v_table_name, v_column_name, v_schema_name, v_db_link_name
            );
            IF v_comment IS NOT NULL AND 
                -- If the comment is not a substring of original description
                INSTR(v_original_description, v_comment) = 0 THEN
                    v_updated_description := v_original_description || ' ' || 
                                             v_comment;
            END IF;

            v_filter_descriptions := v_filter_descriptions || rec.filter_name ||
                                    ': ' || v_updated_description || ', ';
        END LOOP;

        -- Remove trailing comma and space
        v_filter_descriptions := RTRIM(v_filter_descriptions, ', ');

        RETURN v_filter_descriptions;
    END get_filter_descriptions;

    FUNCTION generate_expansion_text (
        p_match_document IN JSON,
        p_text_type IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_expansion_text VARCHAR2(4000);
        v_filter_descriptions VARCHAR2(4000);
    BEGIN
        -- Logging the input parameters
        lang_data_logger_pkg.log_info(
            'Text Type: ' || p_text_type
        );

        -- If the match document is null, log and return an empty expansion
        IF p_match_document IS NULL THEN
            lang_data_logger_pkg.log_info(
                'Match document is null, no expansion generated.'
            );
            RETURN NULL;
        END IF;

        -- Log that the function is fetching filter descriptions
        lang_data_logger_pkg.log_info('Fetching filter descriptions...');
        
        -- Get filter descriptions by calling get_filter_descriptions function
        v_filter_descriptions := get_filter_descriptions(p_match_document);

        -- Log the retrieved filter descriptions
        lang_data_logger_pkg.log_info(
            'Filter Descriptions: ' || NVL(v_filter_descriptions, 'No filters')
        );

        -- If there are filter descriptions, generate the expansion text
        IF v_filter_descriptions IS NOT NULL AND 
            LENGTH(TRIM(v_filter_descriptions)) > 0 THEN
            IF p_text_type = 'query' THEN
                v_expansion_text := 'The query uses the following filters: ' || 
                                v_filter_descriptions || '.';
            ELSE
                v_expansion_text := 'We need the following filters to use this '||
                                p_text_type || ': ' ||
                                v_filter_descriptions || '.';
            END IF;

            -- Log the generated expansion text
            lang_data_logger_pkg.log_info(
                'Generated Expansion: ' || v_expansion_text
            );
        ELSE
            lang_data_logger_pkg.log_info(
                'No filter descriptions to generate expansion.'
            );
            v_expansion_text := NULL;
        END IF;

        -- Return the generated expansion text
        RETURN v_expansion_text;
    END generate_expansion_text;

    -- Function to check if the search record exists in langdata$searchrecords
    FUNCTION check_search_record_exists (
        p_search_id IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_count NUMBER;
    BEGIN

        SELECT COUNT(*) INTO v_count
        FROM langdata$searchrecords
        WHERE id = p_search_id;

        -- Return TRUE if report exists, FALSE otherwise
        RETURN v_count > 0;
    END check_search_record_exists;

    PROCEDURE drop_value_vector_partition(p_partition_name IN VARCHAR2) IS
        v_partition_name   VARCHAR2(128);
        v_partition_suffix VARCHAR2(128);
        v_sql              VARCHAR2(4000);
        v_count            NUMBER;
    BEGIN
        v_partition_name := UPPER(p_partition_name);
        -- Delete data from main table
        v_sql := 'DELETE FROM langdata$drilldownvalues ' ||
                'WHERE partition_name = :1';
        EXECUTE IMMEDIATE v_sql USING v_partition_name;

        -- Delete corresponding metadata
        v_sql := 'DELETE FROM langdata$value_vector_partition_descriptions ' ||
                'WHERE partition_name = :1';
        EXECUTE IMMEDIATE v_sql USING v_partition_name;

        -- Compute suffix
        v_partition_suffix := REPLACE(SUBSTR(v_partition_name, 26), '$', '_');
        
        -- Check if partition exists
        SELECT COUNT(*)
        INTO v_count
        FROM USER_TAB_PARTITIONS
        WHERE TABLE_NAME = 'LANGDATA$DRILLDOWNVALUES'
        AND PARTITION_NAME = 'P_' || v_partition_suffix;

        IF v_count > 0 THEN
            -- Acquire an exclusive lock before issuing the DROP PARTITION DDL
            v_sql := 'LOCK TABLE LANGDATA$DRILLDOWNVALUES PARTITION ( P_' || 
                      v_partition_suffix || ' ) IN EXCLUSIVE MODE WAIT 5' ;
            EXECUTE IMMEDIATE v_sql;

            -- Drop the partition
            v_sql := 'ALTER TABLE LANGDATA$DRILLDOWNVALUES DROP PARTITION P_' ||
                    v_partition_suffix || ' UPDATE GLOBAL INDEXES';
            EXECUTE IMMEDIATE v_sql;

            lang_data_logger_pkg.log_info(
                'Dropped partition and metadata for ' || v_partition_name
            );
        ELSE
            lang_data_logger_pkg.log_info(
                'Partition does not exist for ' || v_partition_name ||
                ', skipping drop.'
            );
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_info(
                'Failed to drop partition for ' || v_partition_name || ': ' ||
                SQLERRM
            );
            RAISE;
    END drop_value_vector_partition;

    PROCEDURE create_value_vector_table (
        p_table_name                  IN VARCHAR2,
        p_column_name                 IN VARCHAR2,
        p_schema_name                 IN VARCHAR2,
        p_force_regenerate            IN BOOLEAN DEFAULT FALSE,
        p_override_enumeration_limit  IN BOOLEAN DEFAULT FALSE,
        p_db_link_name                IN VARCHAR2 DEFAULT NULL,
        p_domain_id                   IN VARCHAR2 DEFAULT NULL
    ) IS
        v_view_name         VARCHAR2(100);
        v_vector_table_name VARCHAR2(100);
        v_vector_index_name VARCHAR2(100);
        v_context_index_name  VARCHAR2(100);
        v_annotation_value  VARCHAR2(4000);
        v_comment           VARCHAR2(4000);
        v_expansion_text    VARCHAR2(4000);
        v_sql               VARCHAR2(4000);
        v_count             NUMBER;
        v_ivf_idx_count     NUMBER;
        v_enumeration_limit VARCHAR2(100);
        v_distinct_value_count     NUMBER;
        v_table_name        VARCHAR2(128);
        v_column_name       VARCHAR2(128);
        v_schema_name       VARCHAR2(128);
        v_db_link_name      VARCHAR2(128);
        v_data_type         VARCHAR2(128);
        v_domain_id         VARCHAR2(36);
    BEGIN
        v_table_name := UPPER(p_table_name);
        v_column_name := UPPER(p_column_name);
        v_schema_name := lang_data_utils_pkg.normalize_schema_name(
            p_schema_name
        );
        v_db_link_name := UPPER(p_db_link_name);

        v_vector_table_name := 'LANGDATA$DRILLDOWNVALUES_' || v_schema_name
                                || '_' || v_table_name || '_' || v_column_name;
        
        IF v_db_link_name IS NOT NULL THEN
            v_vector_table_name := v_vector_table_name || '_' || v_db_link_name;
        END IF;

        IF p_domain_id IS NOT NULL AND TRIM(p_domain_id) = '' THEN
            v_domain_id := NULL;
        ELSE
            v_domain_id := p_domain_id;
        END IF;

        SELECT COUNT(*) INTO v_count FROM langdata$drilldownvalues
        WHERE partition_name = v_vector_table_name;
        
        -- If the partition exists and force is FALSE, return from the procedure
        IF v_count > 0 THEN
            IF p_force_regenerate THEN
                lang_data_logger_pkg.log_info(
                    'Partition ' || v_vector_table_name || ' already exists. '
                    || 'Dropping the partition as force is TRUE.'
                                                );
                -- Drop the existing partition
                lang_data_utils_pkg.drop_value_vector_partition(
                    v_vector_table_name
                );
            ELSE
                lang_data_logger_pkg.log_info(
                    'Partition ' || v_vector_table_name || ' already exists. '
                    || 'Exiting procedure.'
                );
                RETURN;
            END IF;
        END IF;

        v_annotation_value := lang_data_utils_pkg.get_annotation(
            v_table_name,
            v_column_name,
            v_schema_name,
            'DESCRIPTION',
            v_db_link_name
        );
        v_comment := lang_data_utils_pkg.get_comment(
            v_table_name, v_column_name, v_schema_name, v_db_link_name
        );
        IF v_comment IS NOT NULL
          -- If the comment is a substring of the annotation
          AND INSTR(v_annotation_value, v_comment) > 0 THEN
            v_comment := NULL;
        END IF;
        IF v_annotation_value IS NULL AND v_comment IS NULL THEN
            v_expansion_text := NULL;
        ELSE
            v_expansion_text := TRIM(
                NVL(v_annotation_value, '') || ' ' || NVL(v_comment, '')
            );
        END IF;

        -- Create a temporary view based on unique values of table and column
        v_view_name := 'LANGDATA_TEMP_VIEW_' || v_schema_name ||'_' ||
                       v_table_name || '_' || v_column_name;
        
        IF v_db_link_name IS NOT NULL THEN
            v_view_name := v_view_name || '_' || v_db_link_name;
        END IF;

        v_enumeration_limit := 
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_VECTOR_TABLE_ENUMERATION_LIMIT'
            );
        
        v_sql := 'SELECT COUNT(DISTINCT ' || v_column_name || ') FROM ' ||
                 v_schema_name || '.' || v_table_name;
        IF v_db_link_name IS NOT NULL THEN
            v_sql := v_sql || '@' || v_db_link_name;
        END IF;
        v_sql := v_sql || ' WHERE ' || v_column_name || ' IS NOT NULL';
        EXECUTE IMMEDIATE v_sql INTO v_distinct_value_count;

        IF v_distinct_value_count = 0 THEN
            lang_data_logger_pkg.log_info(
                'No values found for specified table column'
            );
            RETURN;
        END IF;
                
        IF LOWER(v_enumeration_limit) != 'unlimited' 
          AND NOT p_override_enumeration_limit 
          AND v_distinct_value_count > TO_NUMBER(v_enumeration_limit) THEN
            lang_data_logger_pkg.log_error(
                'Number of distinct values in ' || v_schema_name ||
                '.' || p_table_name || '.' || p_column_name || ' exceeds ' ||
                'enumeration limit, and p_override_enumeration_limit is set ' ||
                'to false.'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_enumeration_limit_exceeded
            );
        END IF;

        v_sql := 'CREATE OR REPLACE VIEW ' || v_view_name 
                || ' AS SELECT DISTINCT ' || v_column_name
                || ' FROM ' || v_schema_name || '.' || v_table_name;
        IF v_db_link_name IS NOT NULL THEN
            v_sql := v_sql || '@' || v_db_link_name;
        END IF;
        v_sql := v_sql || ' WHERE ' || v_column_name || ' IS NOT NULL';
        EXECUTE IMMEDIATE v_sql;
        lang_data_logger_pkg.log_info('View ' || v_view_name || ' created.');

        -- Create partition in the table
        v_sql := 'ALTER TABLE LANGDATA$DRILLDOWNVALUES ADD PARTITION ' ||
                 'P_' || REPLACE(SUBSTR(v_vector_table_name, 26), '$', '_') ||
                 ' VALUES (''' || v_vector_table_name || ''')';
        EXECUTE IMMEDIATE v_sql;
        
        IF v_expansion_text IS NULL THEN
            v_sql := 'INSERT INTO langdata$drilldownvalues ' || 
                    '(value, vvec, partition_name) ' ||
                    'SELECT ' || v_column_name || ' AS VALUE, ' ||
                    'lang_data_utils_pkg.get_embedding(' ||
                    'CONCAT(''' || v_column_name || ''', '':'', ' ||
                    'TO_CHAR(' || v_column_name || ')' || ')' ||') AS vvec, ' ||
                    '''' || v_vector_table_name || ''' AS partition_name ' ||
                    ' FROM ' || v_view_name;
            EXECUTE IMMEDIATE v_sql;
        ELSE
            v_sql := 'INSERT INTO langdata$drilldownvalues ' ||
                    '(value, vvec, partition_name) ' ||
                    'SELECT ' || v_column_name || ' AS value, ' ||
                    'lang_data_utils_pkg.get_embedding(' ||
                    'CONCAT(' ||
                        'CONCAT(''' || v_column_name || ''', '':'')' || ', ' ||
                        'CONCAT(TO_CHAR(' || v_column_name || '), ' ||
                        ''' (' || REPLACE(v_expansion_text, '''', '''''') ||
                        ')''' ||
                        ')' ||
                    ') ' ||
                    ') AS vvec, ' ||
                    '''' || v_vector_table_name || ''' AS partition_name ' ||
                    'FROM ' || v_view_name;
            EXECUTE IMMEDIATE v_sql;
        END IF;

        INSERT INTO langdata$value_vector_partition_descriptions (
            partition_name,
            description,
            comment_text,
            desc_comment_combined,
            desc_comment_vector,
            domain_id
        ) VALUES (
            v_vector_table_name,
            v_annotation_value,
            v_comment,
            v_expansion_text,
            lang_data_utils_pkg.get_embedding(v_expansion_text),
            v_domain_id
        );

        lang_data_logger_pkg.log_info(
            'Partition ' || v_vector_table_name || ' is created and ' ||
            'inserted into LANGDATA$DRILLDOWNVALUES.'
        );

        SELECT COUNT(*)
        INTO v_ivf_idx_count
        FROM user_indexes
        WHERE table_name = 'LANGDATA$DRILLDOWNVALUES'
        AND index_type = 'VECTOR'
        AND index_name IN (
            SELECT index_name
            FROM user_ind_columns
            WHERE table_name = 'LANGDATA$DRILLDOWNVALUES'
            AND column_name = 'VVEC'
        );

        SELECT COUNT(*)
        INTO v_count
        FROM langdata$drilldownvalues;

        -- Create vector index only if number of rows > 10K
        IF v_ivf_idx_count = 0 AND v_count > 10000 THEN
            -- Create vector index
            v_sql := 'CREATE VECTOR INDEX LANGDATA$DRILLDOWNVALUES_IVF_IDX '
                    || 'ON LANGDATA$DRILLDOWNVALUES(vvec) '
                    || 'ORGANIZATION NEIGHBOR PARTITIONS '
                    || 'DISTANCE COSINE '
                    || 'WITH TARGET ACCURACY 95';

            EXECUTE IMMEDIATE v_sql;
            lang_data_logger_pkg.log_info(
                'Vector index LANGDATA$DRILLDOWNVALUES_IVF_IDX created.'
            );
        END IF;

        -- Drop the view once processing is complete
        v_sql := 'DROP VIEW ' || v_view_name;
        EXECUTE IMMEDIATE v_sql;
        lang_data_logger_pkg.log_info(
            'View ' || v_view_name || ' dropped successfully.'
        );
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_enumeration_limit_exceeded THEN
                -- This is an expected error, just raise it
                lang_data_utils_pkg.drop_value_vector_partition(
                    v_vector_table_name
                );
                RAISE;
            ELSE
                lang_data_logger_pkg.log_info('Error: ' || SQLERRM);
                lang_data_utils_pkg.drop_value_vector_partition(
                    v_vector_table_name
                );
                RAISE;
            END IF;
    END create_value_vector_table;

    PROCEDURE create_value_vector_from_enumerable_set (
        p_enumerable_set      IN JSON_ARRAY_T,
        p_document_id         IN VARCHAR2,
        p_filter_name         IN VARCHAR2,
        p_filter_description  IN VARCHAR2,
        p_force               IN BOOLEAN DEFAULT FALSE,
        p_domain_id           IN VARCHAR2 DEFAULT NULL
    )
    IS
        v_vector_table_name VARCHAR2(100);
        v_vector_index_name VARCHAR2(100);
        v_context_index_name  VARCHAR2(100);
        v_sql               VARCHAR2(32767);
        v_enum_set_str      VARCHAR2(32767);
        v_md5               VARCHAR2(32);
        v_count             NUMBER;
        v_ivf_idx_count     NUMBER;
        v_domain_id         VARCHAR2(36);
    BEGIN
        lang_data_logger_pkg.log_info(
            'Start create_value_vector_from_enumerable_set'
        );
        v_md5 := RAWTOHEX(
            DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(p_document_id),
                DBMS_CRYPTO.HASH_MD5
            )
        );
        v_vector_table_name := 'LANGDATA$DRILLDOWNVALUES_' ||
                                UPPER(p_filter_name) || '_' || v_md5;

        v_enum_set_str := p_enumerable_set.TO_STRING();

        IF p_domain_id IS NOT NULL AND TRIM(p_domain_id) = '' THEN
            v_domain_id := NULL;
        ELSE
            v_domain_id := p_domain_id;
        END IF;

        SELECT COUNT(*) INTO v_count FROM langdata$drilldownvalues
        WHERE partition_name = v_vector_table_name;

        IF v_count > 0 THEN
            IF p_force THEN
                lang_data_logger_pkg.log_info(
                    'Partition ' || v_vector_table_name || ' already exists. '
                    || 'Dropping the partition as force is TRUE.'
                );
                -- Drop the existing partition
                lang_data_utils_pkg.drop_value_vector_partition(
                    v_vector_table_name
                );
            ELSE
                lang_data_logger_pkg.log_info(
                    'Partition ' || v_vector_table_name || ' already exists. '
                    || 'Exiting procedure.'
                );
                RETURN;
            END IF;
        END IF;

        -- Create partition in the table
        v_sql := 'ALTER TABLE LANGDATA$DRILLDOWNVALUES ADD PARTITION ' ||
                 'P_' || REPLACE(SUBSTR(v_vector_table_name, 26), '$', '_') ||
                 ' VALUES (''' || v_vector_table_name || ''')';
        EXECUTE IMMEDIATE v_sql;

        IF p_filter_description IS NULL THEN
            v_sql := 'INSERT INTO langdata$drilldownvalues ' ||
                    '(value, vvec, partition_name) ' ||
                    'SELECT value, ' ||
                    'lang_data_utils_pkg.get_embedding(' ||
                    'CONCAT(''' || p_filter_name || ''', '': '', value)' ||
                    ') AS vvec, ' ||
                    '''' || v_vector_table_name || ''' AS partition_name ' ||
                    'FROM JSON_TABLE(''' || v_enum_set_str || ''', ''$[*]'' ' ||
                    'COLUMNS (value VARCHAR2(4000) PATH ''$''))';
            EXECUTE IMMEDIATE v_sql;
        ELSE
            v_sql := 'INSERT INTO langdata$drilldownvalues ' ||
                    '(value, vvec, partition_name) ' ||
                    'SELECT value, ' ||
                    'lang_data_utils_pkg.get_embedding(' ||
                    'CONCAT(CONCAT(''' || p_filter_name || ''', '': '', value), ' ||
                    ''' (' || REPLACE(p_filter_description, '''', '''''') ||
                    ')'')' || ') AS vvec, ' ||
                    '''' || v_vector_table_name || ''' AS partition_name ' ||
                    'FROM JSON_TABLE(''' || v_enum_set_str || ''', ''$[*]'' ' ||
                    'COLUMNS (value VARCHAR2(4000) PATH ''$''))';
            EXECUTE IMMEDIATE v_sql;
            -- Record filter description on value column
            INSERT INTO langdata$value_vector_partition_descriptions (
                partition_name,
                description,
                comment_text,
                desc_comment_combined,
                desc_comment_vector,
                domain_id
            ) VALUES (
                v_vector_table_name,
                p_filter_description,
                NULL,
                p_filter_description,
                lang_data_utils_pkg.get_embedding(p_filter_description),
                v_domain_id
            );
        END IF;

        lang_data_logger_pkg.log_info(
            'Partition ' || v_vector_table_name || ' created.'
        );

        SELECT COUNT(*)
        INTO v_ivf_idx_count
        FROM user_indexes
        WHERE table_name = 'LANGDATA$DRILLDOWNVALUES'
        AND index_type = 'VECTOR'
        AND index_name IN (
            SELECT index_name
            FROM user_ind_columns
            WHERE table_name = 'LANGDATA$DRILLDOWNVALUES'
            AND column_name = 'VVEC'
        );

        SELECT COUNT(*)
        INTO v_count
        FROM langdata$drilldownvalues;

        -- Create vector index only if number of rows > 10K
        IF v_ivf_idx_count = 0 AND v_count > 10000 THEN
            -- Create vector index
            v_sql := 'CREATE VECTOR INDEX LANGDATA$DRILLDOWNVALUES_IVF_IDX '
                    || 'ON LANGDATA$DRILLDOWNVALUES(vvec) '
                    || 'ORGANIZATION NEIGHBOR PARTITIONS '
                    || 'DISTANCE COSINE '
                    || 'WITH TARGET ACCURACY 95';

            EXECUTE IMMEDIATE v_sql;
            lang_data_logger_pkg.log_info(
                'Vector index LANGDATA$DRILLDOWNVALUES_IVF_IDX created.'
            );
        END IF;
    END create_value_vector_from_enumerable_set;

    -- Need to acquire the CREATE JOB or MANAGE SCHEDULER privilege
    PROCEDURE create_value_vector_job(
        p_match_document        IN JSON,
        p_document_id           IN VARCHAR2,
        p_domain_id             IN VARCHAR2 DEFAULT NULL
    )
    IS
        v_filters               JSON_ARRAY_T;
        v_match_document        JSON_OBJECT_T;
        v_filter_item           JSON_OBJECT_T;
        v_use_ner               BOOLEAN;
        v_table_name            VARCHAR2(255);
        v_column_name           VARCHAR2(255);
        v_schema_name           VARCHAR2(255);
        v_filter_name           VARCHAR2(255);
        v_db_link_name          VARCHAR2(255);
        v_filter_description    VARCHAR2(2000);
        v_enumerable_set        JSON_ARRAY_T;
        v_vec_table_name        VARCHAR2(255);
        v_job_name              VARCHAR2(255);
        v_md5                   VARCHAR2(32);
        v_override_enumeration_limit  BOOLEAN;
        v_annotation            VARCHAR2(4000);
        v_comment               VARCHAR2(4000);
        v_table_exists          NUMBER;

    BEGIN

        v_match_document := JSON_OBJECT_T.parse(
            json_serialize(p_match_document)
        );
        v_filters := v_match_document.GET_ARRAY('filters');

        IF v_filters IS NULL THEN
            lang_data_logger_pkg.log_debug('No filters in match document');
            RETURN;
        END IF;
        
        FOR indx IN 0 .. v_filters.get_size - 1
        LOOP
            v_filter_item   := JSON_OBJECT_T(v_filters.GET(to_number(indx)));
            v_use_ner       := v_filter_item.GET_BOOLEAN('use_ner');

            IF v_use_ner THEN
                CONTINUE;
            END IF;

            v_table_name    := UPPER(v_filter_item.GET_STRING('table_name'));
            v_column_name   := UPPER(v_filter_item.GET_STRING('column_name'));
            v_filter_name   := UPPER(v_filter_item.GET_STRING('filter_name'));
            v_db_link_name  := UPPER(v_filter_item.GET_STRING('db_link_name'));
            v_filter_description   := COALESCE(
                v_filter_item.GET_STRING('description'), ''
            );
            v_enumerable_set := v_filter_item.GET_ARRAY('enumerable_set');
            v_schema_name   := lang_data_utils_pkg.normalize_schema_name(
                v_filter_item.GET_STRING('schema_name')
            );

            IF v_enumerable_set IS NULL THEN
                v_vec_table_name := 'LANGDATA$DRILLDOWNVALUES_' || 
                                    v_schema_name ||'_'||
                                    v_table_name || '_' || v_column_name;
                v_override_enumeration_limit := v_filter_item.GET_BOOLEAN(
                    'override_enumeration_limit'
                );
            ELSE
                IF p_document_id IS NULL THEN
                    lang_data_logger_pkg.log_error(
                        'Document ID not provided to generate a unique value' ||
                        ' vector table name for a enumerable-set-based ' ||
                        'value vector table'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_parameters_code
                    );
                END IF;
                v_md5 := RAWTOHEX(
                    DBMS_CRYPTO.HASH(
                        UTL_RAW.CAST_TO_RAW(p_document_id),
                        DBMS_CRYPTO.HASH_MD5
                    )
                );
                v_vec_table_name := 'LANGDATA$DRILLDOWNVALUES_' || 
                                    v_filter_name || '_' || v_md5;
            END IF;

            IF v_enumerable_set IS NULL THEN
                -- Check if value vector partition already exists
                SELECT COUNT(*) INTO v_table_exists
                FROM langdata$value_vector_metadata
                WHERE vvec_table_name = UPPER(v_vec_table_name);

                IF v_table_exists > 0 THEN
                    -- Increment refcount
                    UPDATE langdata$value_vector_metadata
                    SET ref_count = ref_count + 1
                    WHERE vvec_table_name = UPPER(v_vec_table_name);

                    lang_data_logger_pkg.log_debug(
                        v_vec_table_name ||' table already exists.'
                    );
                    CONTINUE;
                ELSE
                    v_annotation := lang_data_utils_pkg.get_annotation(
                        v_table_name,
                        v_column_name,
                        v_schema_name,
                        'DESCRIPTION',
                        v_db_link_name
                    );
                    v_comment := lang_data_utils_pkg.get_comment(
                        v_table_name,
                        v_column_name,
                        v_schema_name,
                        v_db_link_name
                    );
                    INSERT INTO langdata$value_vector_metadata(
                        vvec_table_name,
                        table_name,
                        column_name,
                        schema_name,
                        db_link_name,
                        ref_count,
                        annotation_value,
                        comment_value
                    )
                    VALUES (
                        UPPER(v_vec_table_name),
                        v_table_name,
                        v_column_name,
                        v_schema_name,
                        v_db_link_name,
                        1,
                        v_annotation,
                        v_comment
                    );
                    lang_data_logger_pkg.log_info(
                        'Metadata inserted for table ' || v_vec_table_name
                    );
                END IF;

                v_job_name := 'JOB_LANGDATA_'|| v_schema_name ||'_'|| 
                            v_table_name || '_' || v_column_name;
                IF v_db_link_name IS NOT NULL THEN
                    v_job_name := v_job_name || '_' || v_db_link_name;
                END IF;
                v_job_name := v_job_name || '_VALUE_VECTORS';
                BEGIN
                    lang_data_utils_pkg.create_or_replace_job(
                        p_job_name        => v_job_name,
                        p_job_action      =>
                            'BEGIN ' || 
                            'lang_data_utils_pkg.create_value_vector_table(''' ||
                            v_table_name || ''', ''' || v_column_name ||
                            ''', ''' || v_schema_name || ''', FALSE, ' ||
                            CASE 
                                WHEN v_override_enumeration_limit THEN 'TRUE' 
                                ELSE 'FALSE' 
                            END ||
                            ', ''' || NVL(v_db_link_name, '') || ''', ''' ||
                            NVL(p_domain_id, '') || '''); END;',
                        p_job_class =>lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                        p_comments        => 'Update value vector tables',
                        p_priority        => 1,
                        p_restart_on_fail => TRUE,
                        p_restart_on_rec  => TRUE
                    );

                    -- Insert into catalog table
                    INSERT INTO langdata$jobs_history (
                        object_name,
                        table_name,
                        column_name,
                        schema_name,
                        db_link_name,
                        job_name
                    ) VALUES (
                        v_vec_table_name, 
                        v_table_name, 
                        v_column_name, 
                        v_schema_name, 
                        v_db_link_name,
                        v_job_name
                    );
                
                EXCEPTION
                    WHEN OTHERS THEN
                        IF SQLCODE = -27477 THEN
                            lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                        ' already exists');
                        ELSE
                            lang_data_logger_pkg.log_fatal(
                                'An unknown error occurred. Error: ' || SQLERRM
                            );
                            RAISE;
                        END IF;
                END;
            ELSE
                v_job_name := 'JOB_LANGDATA_' || v_filter_name || '_' ||
                              v_md5 || '_VALUE_VECTORS';
                BEGIN
                    lang_data_utils_pkg.create_or_replace_job(
                        p_job_name        => v_job_name,
                        p_job_action      =>
                            'DECLARE ' ||
                            '  v_enum_set JSON_ARRAY_T := JSON_ARRAY_T.PARSE(''' || 
                            v_enumerable_set.TO_STRING() || '''); ' ||
                            'BEGIN ' ||
                            '  lang_data_utils_pkg.create_value_vector_from_enumerable_set(' ||
                            '    v_enum_set, ' ||
                            '    ''' || p_document_id || ''', ' ||
                            '    ''' || v_filter_name || ''', ' ||
                            '    ''' || REPLACE(v_filter_description, '''', '''''') || ''', ' ||
                            '    FALSE, ' ||
                            '    ''' || NVL(p_domain_id, '') || '''' ||
                            '  ); ' ||
                            'END;',
                        p_job_class =>lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                        p_comments        => 'Create value vector table from enumerable set',
                        p_priority        => 1,
                        p_restart_on_fail => TRUE,
                        p_restart_on_rec  => TRUE
                    );

                EXCEPTION
                    WHEN OTHERS THEN
                        IF SQLCODE = -27477 THEN
                            lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                        ' already exists');
                        ELSE
                            lang_data_logger_pkg.log_fatal(
                                'An unknown error occurred. Error: ' || SQLERRM
                            );
                            RAISE;
                        END IF;
                END;
            END IF;
        END LOOP;
    END create_value_vector_job;
    
    PROCEDURE get_all_jobs (
        p_job_cursor OUT SYS_REFCURSOR
    ) 
    IS
    BEGIN
        OPEN p_job_cursor FOR
            -- Get the latest finished jobs
            WITH finished_jobs AS (
                SELECT job_name, status, actual_start_date, 
                additional_info AS error_message, run_duration
                FROM (
                    SELECT job_name, status, actual_start_date, 
                    additional_info, run_duration,
                    ROW_NUMBER() OVER 
                    (PARTITION BY job_name ORDER BY log_id DESC) AS rn
                    FROM all_scheduler_job_run_details
                    WHERE job_name LIKE 'JOB_LANGDATA_%_VALUE_VECTORS'
                )
                WHERE rn = 1
            ),
            -- Get running jobs
            submitted_jobs AS (
                SELECT job_name, state AS status, start_date, 
                NULL AS error_message, 
                (SYSTIMESTAMP - start_date) AS run_duration
                FROM all_scheduler_jobs
                WHERE job_name LIKE 'JOB_LANGDATA_%_VALUE_VECTORS'
            )
            -- Combine results, prioritizing running jobs
            SELECT job_name, status, start_date, 
                CASE WHEN status = 'RUNNING' THEN 'TRUE' 
                ELSE 'FALSE' END AS is_running, 
                error_message, run_duration
            FROM submitted_jobs
            UNION ALL
            SELECT job_name, status, actual_start_date, 
            'FALSE' AS is_running, 
            error_message, run_duration
            FROM finished_jobs
            WHERE job_name NOT IN (SELECT job_name FROM submitted_jobs);

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred' || '. Error: ' 
                || SQLERRM
            );
            RAISE;
    END get_all_jobs;

    PROCEDURE update_job_status(
        p_action   IN VARCHAR2,
        p_job_name IN VARCHAR2
    ) 
    IS
        v_table_name    VARCHAR2(128);
        v_column_name   VARCHAR2(128);
        v_schema_name   VARCHAR2(128);
        v_db_link_name  VARCHAR2(128);
        v_domain_id     VARCHAR2(36);
        v_partition_name VARCHAR2(255);
    BEGIN
        IF p_action = 'STOP' THEN
            DBMS_SCHEDULER.STOP_JOB(p_job_name);
        
        ELSIF p_action = 'RERUN' THEN
            -- Fetch table and column name
            BEGIN
                SELECT table_name, column_name, schema_name, db_link_name
                INTO v_table_name, v_column_name, v_schema_name, v_db_link_name
                FROM LANGDATA$jobs_history
                WHERE job_name = p_job_name;
            EXCEPTION
                WHEN NO_DATA_FOUND THEN
                    lang_data_logger_pkg.log_error(
                        'Job ' || p_job_name || ' not found'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_resource_not_found
                    );
            END;

            v_partition_name := 'LANGDATA$DRILLDOWNVALUES_' || v_schema_name
                            || '_' || v_table_name || '_' || v_column_name;
    
            IF v_db_link_name IS NOT NULL THEN
                v_partition_name := v_partition_name || '_' || v_db_link_name;
            END IF;

            SELECT domain_id INTO v_domain_id
            FROM langdata$value_vector_partition_descriptions
            WHERE partition_name = v_partition_name;
         
            -- Create a new job
            DBMS_SCHEDULER.create_job (
                job_name        => p_job_name,
                job_type        => 'PLSQL_BLOCK',
                job_action      => 'BEGIN ' ||
                                    'lang_data_utils_pkg.create_value_vector_table(''' || 
                                    v_table_name || ''', ''' || 
                                    v_column_name || ''', ''' || 
                                    v_schema_name || ''', TRUE, FALSE, ''' || 
                                    v_db_link_name || ''', ''' ||
                                    NVL(v_domain_id, '') || '''); END;',
                start_date      => SYSTIMESTAMP,
                enabled         => TRUE
            );
            
            lang_data_logger_pkg.log_info('Created job_name: ' || p_job_name); 
        END IF; 
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred ' || '. Error: ' 
                || SQLERRM
            );
            RAISE;
    END update_job_status;

    PROCEDURE get_sample_queries_paginated (
        p_report_id      IN VARCHAR2 DEFAULT NULL,
        p_drilldown_id   IN VARCHAR2 DEFAULT NULL,
        p_limit          IN NUMBER DEFAULT 10,
        p_cursor         IN OUT VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    ) IS
        v_cur               INTEGER;
        v_cursor_created_at TIMESTAMP;
        v_cursor_id         VARCHAR2(4000);
        v_last_id           VARCHAR2(4000);
        v_last_created_at   TIMESTAMP;
        v_sample_query      VARCHAR2(4000) := 'SELECT id, query_text, version, ' 
                                            || 'enhanced_query_text, status ' 
                                            || 'FROM langdata$samplequeries ' 
                                            || 'WHERE 1=1';
        v_conditions        VARCHAR2(4000) := 
                                    ' AND (created_at < :cursor_created_at ' 
                                    || 'OR (created_at = :cursor_created_at ' 
                                    || 'AND id <= :cursor_id)) ';
        v_sql_text          VARCHAR2(4000);
        v_dummy             INTEGER;
    BEGIN
        v_cur := DBMS_SQL.OPEN_CURSOR;

        IF p_report_id IS NOT NULL THEN
            v_sample_query := v_sample_query 
                                || ' AND report_id = :report_id';
        END IF;

        IF p_drilldown_id IS NOT NULL THEN
            v_sample_query := v_sample_query 
                                || ' AND drilldown_id = :drilldown_id';
        END IF;
    
        IF p_cursor IS NOT NULL THEN
            lang_data_utils_pkg.split_cursor(p_cursor => p_cursor, 
                                            p_created_at => v_cursor_created_at, 
                                            p_id => v_cursor_id);


            lang_data_logger_pkg.log_debug(
                'Applied pagination cursor: CreatedAt=' 
                || TO_CHAR(v_cursor_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') 
                || ', ID=' 
                || v_cursor_id
            );
            v_sql_text := v_sample_query || v_conditions;
        ELSE
            v_sql_text := v_sample_query;
        END IF;

        IF p_limit IS NOT NULL THEN
            v_sql_text := v_sql_text 
                          || ' ORDER BY created_at DESC, id DESC ' 
                          || 'FETCH FIRST :limit ROWS ONLY ';
        ELSE
            v_sql_text := v_sql_text || ' ORDER BY created_at DESC, id DESC ';
        END IF;
        lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);

        -- Load the query to fetch report descriptions
        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

        IF p_report_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);
        END IF;

        IF p_drilldown_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_drilldown_id);
        END IF;

        IF p_cursor IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':cursor_created_at', v_cursor_created_at
            );
            DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
        END IF;
        
        IF p_limit IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);
        END IF;

        -- Return the report descriptions as ref cursor object
        v_dummy :=  DBMS_SQL.EXECUTE(v_cur);
        p_sample_queries  :=  DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        IF p_limit IS NOT NULL THEN
            lang_data_logger_pkg.log_debug(
                'Checking for next sample query row'
            );
            
            v_sql_text := 'SELECT id, created_at ' 
                          || 'FROM langdata$samplequeries WHERE 1=1';

            IF p_report_id IS NOT NULL THEN
                v_sql_text := v_sql_text || ' AND report_id = :report_id';
            END IF;

            IF p_drilldown_id IS NOT NULL THEN
                v_sql_text := v_sql_text || ' AND drilldown_id = :drilldown_id';
            END IF;                

            IF p_cursor IS NOT NULL THEN
                v_sql_text := v_sql_text || v_conditions;
            END IF;

            v_sql_text := v_sql_text 
                          || ' ORDER BY created_at DESC, id DESC ' 
                          || 'OFFSET :limit ROWS FETCH NEXT 1 ROWS ONLY';

            v_cur :=  DBMS_SQL.OPEN_CURSOR;

            DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

            IF p_cursor IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(
                    v_cur, ':cursor_created_at', v_cursor_created_at
                );
                DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
            END IF;

            IF p_report_id IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);
            END IF;

            IF p_drilldown_id IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_drilldown_id);
            END IF;
            
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);


            DBMS_SQL.DEFINE_COLUMN(v_cur, 1, v_last_id, 36);
            DBMS_SQL.DEFINE_COLUMN(v_cur, 2, v_last_created_at);

            v_dummy := DBMS_SQL.EXECUTE(v_cur);

            IF DBMS_SQL.FETCH_ROWS(v_cur) > 0 THEN
                DBMS_SQL.COLUMN_VALUE(v_cur, 1, v_last_id);
                DBMS_SQL.COLUMN_VALUE(v_cur, 2, v_last_created_at);

                -- Set the next cursor for pagination
                p_cursor := TO_CHAR(
                    v_last_created_at, 
                    'YYYY-MM-DD HH24:MI:SS.FF'
                ) || '|' || v_last_id;
                lang_data_logger_pkg.log_debug(
                    'Next sample query cursor set to: ' || p_cursor
                );
            ELSE
                p_cursor := NULL;
                lang_data_logger_pkg.log_debug('No further pages.');
            END IF;

            IF DBMS_SQL.IS_OPEN(v_cur) THEN
                -- Close the next cursor
                DBMS_SQL.CLOSE_CURSOR(v_cur);
            END IF;
        ELSE
            p_cursor := NULL;
            lang_data_logger_pkg.log_debug('No further pages.');
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for report_id: ' || p_report_id 
                || ' and drilldown_id: ' || p_drilldown_id || '. Error: ' 
                || SQLERRM
            );
            RAISE;
    END get_sample_queries_paginated;

    PROCEDURE fetch_top_n_similairty_search_drilldown_text (
        p_query             IN  VARCHAR2,
        p_drilldown_id      IN  VARCHAR2,
        p_n                 IN  NUMBER DEFAULT 3,
        p_use_records       IN  BOOLEAN DEFAULT TRUE,
        p_texts             OUT SYS.ODCIVARCHAR2LIST
    )
    IS
        v_sql_query         VARCHAR2(4000);
        v_status            CONSTANT VARCHAR2(50) := 'Published';
        v_cur               INTEGER;
        v_dummy             INTEGER;
        v_query_vector      VECTOR(*,*);
        v_results           SYS_REFCURSOR;
        v_id                VARCHAR2(36);
        v_drilldown_text    VARCHAR2(4000);
        v_distance          NUMBER;
        v_texts             SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_cnt               NUMBER := 0;
    BEGIN
        v_query_vector := get_embedding(p_query);

        v_sql_query := '
            WITH combined_results AS (
                SELECT
                    id, 
                    text AS drilldown_text,
                    VECTOR_DISTANCE(ddd_vector, :query_vector) AS distance
                FROM langdata$drilldowndescriptions
                WHERE drilldown_id = :drilldown_id
                AND status = :status
                UNION ALL
                SELECT
                    id,
                    query_text AS drilldown_text,
                    VECTOR_DISTANCE(query_vector, :query_vector) AS distance
                FROM langdata$samplequeries
                WHERE drilldown_id = :drilldown_id
                AND status = :status
        ';
        
        IF p_use_records THEN
            v_sql_query := v_sql_query || '
                UNION ALL
                SELECT
                    id,
                    query_text AS drilldown_text,
                    VECTOR_DISTANCE(query_vector, :query_vector) AS distance
                FROM langdata$searchrecords
                WHERE expected_drilldown_id = :drilldown_id
            ';
        END IF;

        v_sql_query := v_sql_query || '
            )
            SELECT
                id,
                drilldown_text,
                distance
            FROM
                combined_results
            ORDER BY
                distance ASC
            FETCH FIRST :n ROWS ONLY
        ';

        v_cur := DBMS_SQL.OPEN_CURSOR;

        DBMS_SQL.PARSE(v_cur, v_sql_query, DBMS_SQL.NATIVE);

        DBMS_SQL.BIND_VARIABLE(v_cur, ':query_vector', v_query_vector);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':drilldown_id', p_drilldown_id);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':status', v_status);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':n', p_n);

        -- Execute the SQL statement
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        v_results := DBMS_SQL.TO_REFCURSOR(v_cur);

        LOOP
            FETCH v_results INTO v_id, v_drilldown_text, v_distance;
            EXIT WHEN v_results%NOTFOUND;

            v_cnt := v_cnt + 1;
            v_texts.EXTEND;
            v_texts(v_cnt) := v_drilldown_text;
        END LOOP;

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        p_texts := v_texts;
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in ' || 
                'fetch_top_n_similairty_search_report_text:  for query' || p_query
                || '. Error: ' || SQLERRM
            );
            RAISE;
    END fetch_top_n_similairty_search_drilldown_text;

    PROCEDURE fetch_top_n_similairty_search_report_text (
        p_query             IN  VARCHAR2,
        p_report_id         IN  VARCHAR2,
        p_n                 IN  NUMBER DEFAULT 3,
        p_use_records       IN  BOOLEAN DEFAULT TRUE,
        p_texts             OUT SYS.ODCIVARCHAR2LIST
    )
    IS
        v_sql_query         VARCHAR2(4000);
        v_status            CONSTANT VARCHAR2(50) := 'Published';
        v_cur               INTEGER;
        v_dummy             INTEGER;
        v_query_vector      VECTOR(*,*);
        v_results           SYS_REFCURSOR;
        v_id                VARCHAR2(36);
        v_report_text       VARCHAR2(4000);
        v_distance          NUMBER;
        v_texts             SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_cnt               NUMBER := 0;
    BEGIN
        v_query_vector := lang_data_utils_pkg.get_embedding(
            p_query
        );

        v_sql_query := '
            WITH combined_results AS (
                SELECT
                    id, 
                    text AS report_text,
                    VECTOR_DISTANCE(description_vector, :query_vector) AS distance
                FROM langdata$reportdescriptions
                WHERE report_id = :report_id
                AND status = :status
                UNION ALL
                SELECT
                    id,
                    query_text AS report_text,
                    VECTOR_DISTANCE(query_vector, :query_vector) AS distance
                FROM langdata$samplequeries
                WHERE report_id = :report_id
                AND status = :status
        ';
        
        IF p_use_records THEN
            v_sql_query := v_sql_query || '
                UNION ALL
                SELECT
                    id,
                    query_text AS report_text,
                    VECTOR_DISTANCE(query_vector, :query_vector) AS distance
                FROM langdata$searchrecords
                WHERE expected_report_id = :report_id
            ';
        END IF;

        v_sql_query := v_sql_query || '
            )
            SELECT
                id,
                report_text,
                distance
            FROM
                combined_results
            ORDER BY
                distance ASC
            FETCH FIRST :n ROWS ONLY
        ';

        v_cur := DBMS_SQL.OPEN_CURSOR;

        DBMS_SQL.PARSE(v_cur, v_sql_query, DBMS_SQL.NATIVE);

        DBMS_SQL.BIND_VARIABLE(v_cur, ':query_vector', v_query_vector);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':status', v_status);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':n', p_n);

        -- Execute the SQL statement
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        v_results := DBMS_SQL.TO_REFCURSOR(v_cur);

        LOOP
            FETCH v_results INTO v_id, v_report_text, v_distance;
            EXIT WHEN v_results%NOTFOUND;

            v_cnt := v_cnt + 1;
            v_texts.EXTEND;
            v_texts(v_cnt) := v_report_text;
        END LOOP;

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        p_texts := v_texts;
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in ' || 
                'fetch_top_n_similairty_search_report_text:  for query' || p_query
                || '. Error: ' || SQLERRM
            );
            RAISE;
    END fetch_top_n_similairty_search_report_text;

    -- Procedure to get top k similarity search reports for a given query
    PROCEDURE fetch_top_k_similarity_search_reports (
        p_query             IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_use_records       IN  BOOLEAN DEFAULT TRUE,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    )  AS
        v_sql_query  VARCHAR2(4000);
        v_status     CONSTANT VARCHAR2(50) := 'Published';
        v_cur      INTEGER;
        v_dummy    INTEGER;
        v_query_vector VECTOR(*,*);
    BEGIN
        v_query_vector := lang_data_utils_pkg.get_embedding(p_query);

        -- Construct base SQL query

        v_sql_query := '
        WITH combined_results AS (
            SELECT 
                rd.report_id, rd.id, 
                rd.enhanced_text AS enhanced_text,
                VECTOR_DISTANCE(description_vector, :query_vector) AS distance,
                rd.text, r.title, ''Report Description'' AS match_type
            FROM langdata$reportdescriptions rd
            JOIN langdata$reports r ON rd.report_id = r.id
            WHERE 
                rd.status = :status AND r.status = :status AND (
                    :domain_id IS NULL OR r.domain_id = :domain_id
                )
            UNION      
            SELECT 
                sq.report_id, sq.id, 
                sq.enhanced_query_text AS enhanced_text,
                VECTOR_DISTANCE(query_vector, :query_vector) AS distance,
                sq.query_text, r.title, ''Report Sample Query'' AS match_type
            FROM langdata$samplequeries sq
            JOIN langdata$reports r ON sq.report_id = r.id
            WHERE 
                sq.status = :status AND r.status = :status AND (
                    :domain_id IS NULL OR r.domain_id = :domain_id
                )
            ';
        IF p_use_records = TRUE THEN
            v_sql_query := v_sql_query || '  
            UNION
            SELECT 
                sr.expected_report_id, sr.id, 
                sr.augmented_query_text AS enhanced_text,
                VECTOR_DISTANCE(query_vector, :query_vector) AS distance,
                sr.query_text, r.title, ''Search Record'' AS match_type
            FROM langdata$searchRecords sr
            JOIN langdata$reports r ON sr.expected_report_id = r.id
            WHERE 
                r.status = :status AND (
                    :domain_id IS NULL OR r.domain_id = :domain_id
                )
            ';
        END IF;

        v_sql_query := v_sql_query || '),
        ranked_reports AS (
            SELECT 
                report_id, id, enhanced_text, distance, text, title, match_type,
                ROW_NUMBER() OVER 
                (PARTITION BY report_id ORDER BY distance ASC) AS rank
            FROM 
                combined_results
        )
        SELECT 
            report_id, id, enhanced_text, distance, text, title, match_type,
            ''report'' AS type,
            RANK() OVER (ORDER BY distance ASC) AS similarity_search_rank
        FROM 
            ranked_reports
        WHERE 
            rank = 1
        ORDER BY 
            distance ASC
        FETCH FIRST :k ROWS ONLY
        ';

        v_cur := DBMS_SQL.OPEN_CURSOR;

        -- Load the query to fetch good drilldown search records
        DBMS_SQL.PARSE(v_cur, v_sql_query, DBMS_SQL.NATIVE);

        -- Bind variables
        DBMS_SQL.BIND_VARIABLE(v_cur, ':query_vector', v_query_vector);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':status', v_status);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':k', p_k);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':domain_id', p_domain_id);

        -- Execute the SQL statement
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_results := DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in ' || 
                'fetch_top_k_similarity_search_reports:  for query' || p_query
                || '. Error: ' || SQLERRM
            );
            RAISE;

    END fetch_top_k_similarity_search_reports;
    
    PROCEDURE fetch_top_k_similarity_search_reports_using_centroid (
        p_query             IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    )  AS
        v_sql_query  VARCHAR2(4000);
        v_status     CONSTANT VARCHAR2(50) := 'Published';
        v_cur      INTEGER;
        v_dummy    INTEGER;
        v_query_vector VECTOR(*,*);
    BEGIN
        v_query_vector := lang_data_utils_pkg.get_embedding(p_query);

        -- Construct base SQL query

        v_sql_query := '
        WITH combined_results AS (
            SELECT 
                rd.report_id, rd.id, 
                rd.enhanced_text AS enhanced_text,
                VECTOR_DISTANCE(description_vector, :query_vector) AS distance,
                rd.text, r.title, ''Report Description'' AS match_type
            FROM langdata$reportdescriptions rd
            JOIN langdata$reports r ON rd.report_id = r.id
            WHERE 
                rd.status = :status AND r.status = :status AND (
                    :domain_id IS NULL OR r.domain_id = :domain_id
                )
            UNION      
            SELECT 
                qc.report_id, null, null,
                VECTOR_DISTANCE(centroid_vector, :query_vector) AS distance,
                null, r.title, ''Report Sample Query'' AS match_type
            FROM langdata$reportquerycluster qc
            JOIN langdata$reports r ON qc.report_id = r.id
            WHERE 
                r.status = :status AND (
                    :domain_id IS NULL OR r.domain_id = :domain_id
                )
        ),
        ranked_reports AS (
            SELECT 
                report_id, id, enhanced_text, distance, text, title, match_type,
                ROW_NUMBER() OVER 
                (PARTITION BY report_id ORDER BY distance ASC) AS rank
            FROM 
                combined_results
        )
        SELECT 
            report_id, id, enhanced_text, distance, text, title, match_type,''report'' AS type,
            RANK() OVER (ORDER BY distance ASC) AS similarity_search_rank
        FROM 
            ranked_reports
        WHERE 
            rank = 1
        ORDER BY 
            distance ASC
        FETCH FIRST :k ROWS ONLY
        ';

        v_cur := DBMS_SQL.OPEN_CURSOR;

        -- Load the query to fetch good drilldown search records
        DBMS_SQL.PARSE(v_cur, v_sql_query, DBMS_SQL.NATIVE);

        -- Bind variables
        DBMS_SQL.BIND_VARIABLE(v_cur, ':query_vector', v_query_vector);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':status', v_status);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':k', p_k);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':domain_id', p_domain_id);

        -- Execute the SQL statement
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_results := DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in ' || 
                'fetch_top_k_similarity_search_reports_using_centroid ' ||
                'for query: ' || p_query || '. Error: ' || SQLERRM
            );
            RAISE;

    END fetch_top_k_similarity_search_reports_using_centroid;

    PROCEDURE get_job(
        p_table_name    IN VARCHAR2,
        p_column_name   IN VARCHAR2,
        p_schema_name   IN VARCHAR2 DEFAULT NULL,
        p_db_link_name  IN VARCHAR2 DEFAULT NULL,
        p_job_name      OUT VARCHAR2,
        p_status        OUT VARCHAR2,
        p_start_date    OUT TIMESTAMP,
        p_is_running    OUT BOOLEAN,
        p_error_msg     OUT VARCHAR2,
        p_run_duration  OUT INTERVAL DAY TO SECOND
    ) IS
        v_sql_text      VARCHAR2(500);
        v_cur           INTEGER;
        v_job_details   SYS_REFCURSOR;
        v_state         VARCHAR2(20);
        v_job_name      VARCHAR2(255);
    BEGIN
        p_is_running := FALSE;
        -- Check if the job has been completed
        v_cur := DBMS_SQL.OPEN_CURSOR;

        v_job_name := 'JOB_LANGDATA_'||
                      lang_data_utils_pkg.normalize_schema_name(p_schema_name)
                      ||'_'|| UPPER(p_table_name) || '_' ||
                      UPPER(p_column_name);
        IF p_db_link_name IS NOT NULL THEN
            v_job_name := v_job_name || '_' || UPPER(p_db_link_name);
        END IF;
        v_job_name := v_job_name || '_VALUE_VECTORS';

        v_sql_text :=
            'SELECT job_name, status, actual_start_date, '||
            'additional_info AS error_message, run_duration '||
            'FROM all_scheduler_job_run_details '||
            'WHERE job_name = :job_name '||
            'ORDER BY actual_start_date DESC FETCH FIRST ROW ONLY';

        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':job_name', v_job_name);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 1, p_job_name, 261);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 2, p_status, 30);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 3, p_start_date);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 4, p_error_msg, 4000);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 5, p_run_duration);

        IF DBMS_SQL.EXECUTE_AND_FETCH(v_cur) > 0 THEN
            DBMS_SQL.COLUMN_VALUE(v_cur, 1, p_job_name);
            DBMS_SQL.COLUMN_VALUE(v_cur, 2, p_status);
            DBMS_SQL.COLUMN_VALUE(v_cur, 3, p_start_date);
            DBMS_SQL.COLUMN_VALUE(v_cur, 4, p_error_msg);
            DBMS_SQL.COLUMN_VALUE(v_cur, 5, p_run_duration);
        END IF;

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        -- Check if job is in running state
        v_cur := DBMS_SQL.OPEN_CURSOR;

        v_sql_text :=
            'SELECT job_name, state, start_date, '||
            '(systimestamp - START_DATE) '||
            'FROM all_scheduler_jobs '||
            'WHERE job_name = :job_name '||
            'ORDER BY start_date DESC FETCH FIRST ROW ONLY';

        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':job_name', v_job_name);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 1, p_job_name, 261);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 2, p_status, 30);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 3, p_start_date);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 4, p_run_duration);

        -- Check if the job is running
        IF DBMS_SQL.EXECUTE_AND_FETCH(v_cur) > 0 THEN
            DBMS_SQL.COLUMN_VALUE(v_cur, 2, v_state);

            -- Only update the job details when the job is in running state
            IF v_state = 'RUNNING' THEN
                    p_is_running := TRUE;
                    DBMS_SQL.COLUMN_VALUE(v_cur, 1, p_job_name);
                    DBMS_SQL.COLUMN_VALUE(v_cur, 3, p_start_date);
                    DBMS_SQL.COLUMN_VALUE(v_cur, 4, p_run_duration);
                    p_error_msg := NULL;
                    p_status := v_state;
            END IF;

        END IF;

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    END get_job;
    
    PROCEDURE update_table_column (
        p_id            IN VARCHAR2,
        p_table_name    IN VARCHAR2,
        p_column_name   IN VARCHAR2,
        p_value         IN VARCHAR2,
        p_error_message IN VARCHAR2 DEFAULT NULL
    )
    IS
        v_update_sql VARCHAR2(4000);
        v_row_count  NUMBER;
    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Construct the dynamic SQL for the update
        v_update_sql := 'UPDATE ' || p_table_name 
                        || ' SET ' || p_column_name || ' = :value, ' 
                        || 'UPDATED_AT = CURRENT_TIMESTAMP' 
                        || ' WHERE id = :id';

        -- Execute the dynamic SQL
        EXECUTE IMMEDIATE v_update_sql
        USING p_value, p_id;

        -- Get the number of rows affected by the update
        v_row_count := SQL%ROWCOUNT;

        -- If no rows were affected, raise an error
        IF v_row_count = 0 THEN
            IF p_error_message IS NOT NULL THEN
                lang_data_logger_pkg.log_error(
                    'No data found for column: ' || p_column_name 
                    || ' of table: ' || p_table_name 
                    || '. Error: ' || p_error_message
                );
            ELSE
                lang_data_logger_pkg.log_error(
                    'No data found for column: ' 
                    || p_column_name || ' of table: ' || p_table_name || '.'
                );
            END IF;
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred: ' || SQLERRM
            );
            RAISE;
    END update_table_column;


    PROCEDURE fetch_top_k_similarity_search_drilldowns (
        p_query             IN  VARCHAR2,
        p_matched_report_id IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_use_records       IN  BOOLEAN DEFAULT TRUE,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    )  AS
        v_sql_query  VARCHAR2(8000);
        v_status     CONSTANT VARCHAR2(50) := 'Published';
        v_cur      INTEGER;
        v_dummy    INTEGER;
        v_query_vector VECTOR(*,*);
    BEGIN
        v_query_vector := lang_data_utils_pkg.get_embedding(p_query);
        -- Construct base SQL query
        v_sql_query := '
        WITH combined_results AS (
            SELECT 
                ddd.drilldown_id, 
                ddd.id, 
                ddd.enhanced_text AS enhanced_text,
                VECTOR_DISTANCE(ddd.ddd_vector, :query_vector) AS distance,
                ddd.text, 
                ddoc.title,
                ''Drilldown Description'' as match_type
            FROM 
                LANGDATA$DRILLDOWNDESCRIPTIONS ddd
            JOIN 
                LANGDATA$DRILLDOWNDOCUMENTS ddoc
            ON 
                ddoc.id = ddd.drilldown_id
            WHERE 
                ddoc.status = :status 
                AND ddd.status = :status';

        -- Filter based on domain
        IF p_domain_id IS NOT NULL THEN
            v_sql_query := v_sql_query || '
                AND  ddoc.domain_id = :domain_id';
        END IF;

        -- Append the condition for matched_report_id only if it is not NULL
        IF p_matched_report_id IS NOT NULL THEN
            v_sql_query := v_sql_query || ' 
                AND ddoc.report_id = :matched_report_id';
        ELSE
            -- Ensure that the drilldowns are associated with at least 
            -- one published report and description
            v_sql_query := v_sql_query || '
                AND EXISTS (
                    SELECT 1 
                    FROM LANGDATA$REPORTS r 
                    JOIN LANGDATA$REPORTDESCRIPTIONS rd 
                    ON r.id = rd.report_id 
                    WHERE r.id = ddoc.report_id 
                    AND r.status = :status 
                    AND rd.status = :status
                )';
        END IF;

        -- Append UNION clauses for other query types
        v_sql_query := v_sql_query || '
                UNION 
                SELECT 
                    sq.drilldown_id, 
                    sq.id, 
                    sq.enhanced_query_text AS enhanced_text,
                    VECTOR_DISTANCE(sq.query_vector, :query_vector) AS distance,
                    sq.query_text, 
                    ddoc.title,
                    ''Drilldown Sample Query'' as match_type
                FROM 
                    LANGDATA$SAMPLEQUERIES sq
                JOIN 
                    LANGDATA$DRILLDOWNDOCUMENTS ddoc
                ON 
                    ddoc.id = sq.drilldown_id
                WHERE 
                    sq.status = :status AND
                    ddoc.status = :status';

        -- Filter based on domain
        IF p_domain_id IS NOT NULL THEN
            v_sql_query := v_sql_query || '
                AND  ddoc.domain_id = :domain_id';
        END IF;
        
        IF p_matched_report_id IS NOT NULL THEN
            v_sql_query := v_sql_query || ' 
                AND ddoc.report_id = :matched_report_id';
        END IF;

        IF p_use_records THEN
            v_sql_query := v_sql_query || '
                    UNION 
                    SELECT 
                        sr.expected_drilldown_id, 
                        sr.id, 
                        sr.augmented_query_text AS enhanced_text,
                        VECTOR_DISTANCE(sr.query_vector, :query_vector) AS distance,
                        sr.query_text, 
                        ddoc.title,
                        ''Search Record'' as match_type
                    FROM 
                        LANGDATA$SEARCHRECORDS sr
                    JOIN 
                        LANGDATA$DRILLDOWNDOCUMENTS ddoc
                    ON 
                        ddoc.id = sr.expected_drilldown_id
                    WHERE 
                        ddoc.status = :status
                        AND VECTOR_DISTANCE(sr.query_vector, :query_vector) != 0
                        AND sr.feedback_rating = TRUE
                        AND sr.expected_report_id != NULL';

            -- Filter based on domain
            IF p_domain_id IS NOT NULL THEN
                v_sql_query := v_sql_query || '
                    AND  ddoc.domain_id = :domain_id';
            END IF;

            IF p_matched_report_id IS NOT NULL THEN
                v_sql_query := v_sql_query || ' 
                    AND ddoc.report_id = :matched_report_id';
            END IF; 
        END IF;

        v_sql_query := v_sql_query || '
            ),
            ranked_drilldowns AS (
                SELECT 
                    drilldown_id, 
                    id, 
                    enhanced_text,
                    distance, 
                    text, 
                    title,
                    match_type,
                    ROW_NUMBER() OVER (
                        PARTITION BY drilldown_id 
                        ORDER BY distance ASC
                    ) AS rank
                FROM 
                    combined_results
            )
            SELECT 
                drilldown_id, id, enhanced_text, distance, text, title, match_type, 
                ''drilldown'' AS type, 
                RANK() OVER (ORDER BY distance ASC) AS similarity_search_rank
            FROM 
                ranked_drilldowns
            WHERE 
                rank = 1
            ORDER BY 
                distance ASC
            FETCH FIRST :k ROWS ONLY';

        v_cur := DBMS_SQL.OPEN_CURSOR;

        -- Load the query to fetch good drilldown search records
        DBMS_SQL.PARSE(v_cur, v_sql_query, DBMS_SQL.NATIVE);

        -- Bind variables
        DBMS_SQL.BIND_VARIABLE(v_cur, ':query_vector', v_query_vector);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':status', v_status);
        IF p_matched_report_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':matched_report_id', p_matched_report_id
            );
        END IF;
        DBMS_SQL.BIND_VARIABLE(v_cur, ':k', p_k);
        IF p_domain_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':domain_id', p_domain_id);
        END IF;

        -- Execute the SQL statement
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_results := DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    END fetch_top_k_similarity_search_drilldowns;

    PROCEDURE fetch_top_k_similarity_search_drilldowns_using_centroid (
        p_query             IN  VARCHAR2,
        p_matched_report_id IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    )  AS
        v_sql_query  VARCHAR2(8000);
        v_status     CONSTANT VARCHAR2(50) := 'Published';
        v_cur      INTEGER;
        v_dummy    INTEGER;
        v_query_vector VECTOR(*,*);
    BEGIN
        v_query_vector := lang_data_utils_pkg.get_embedding(p_query);
        -- Construct base SQL query
        v_sql_query := '
        WITH combined_results AS (
            SELECT 
                ddd.drilldown_id, 
                ddd.id, 
                ddd.enhanced_text AS enhanced_text,
                VECTOR_DISTANCE(ddd.ddd_vector, :query_vector) AS distance,
                ddd.text, 
                ddoc.title,
                ''Drilldown Description'' as match_type
            FROM 
                LANGDATA$DRILLDOWNDESCRIPTIONS ddd
            JOIN 
                LANGDATA$DRILLDOWNDOCUMENTS ddoc
            ON 
                ddoc.id = ddd.drilldown_id
            WHERE 
                ddoc.status = :status 
                AND ddd.status = :status';

        -- Filter based on domain
        IF p_domain_id IS NOT NULL THEN
            v_sql_query := v_sql_query || '
                AND  ddoc.domain_id = :domain_id';
        END IF;

        -- Append the condition for matched_report_id only if it is not NULL
        IF p_matched_report_id IS NOT NULL THEN
            v_sql_query := v_sql_query || ' 
                AND ddoc.report_id = :matched_report_id';
        ELSE
            -- Ensure that the drilldowns are associated with at least 
            -- one published report and description
            v_sql_query := v_sql_query || '
                AND EXISTS (
                    SELECT 1 
                    FROM LANGDATA$REPORTS r 
                    JOIN LANGDATA$REPORTDESCRIPTIONS rd 
                    ON r.id = rd.report_id 
                    WHERE r.id = ddoc.report_id 
                    AND r.status = :status 
                    AND rd.status = :status
                )';
        END IF;

        -- Append UNION clauses for other query types
        v_sql_query := v_sql_query || '
                UNION 
                SELECT 
                    qc.drilldown_id, 
                    null, 
                    null,
                    VECTOR_DISTANCE(centroid_vector, :query_vector) AS distance,
                    null, 
                    ddoc.title,
                    ''Drilldown Sample Query'' as match_type
                FROM 
                    langdata$drilldownquerycluster qc
                JOIN 
                    LANGDATA$DRILLDOWNDOCUMENTS ddoc
                ON 
                    ddoc.id = qc.drilldown_id
                WHERE 
                    ddoc.status = :status';

        -- Filter based on domain
        IF p_domain_id IS NOT NULL THEN
            v_sql_query := v_sql_query || '
                AND  ddoc.domain_id = :domain_id';
        END IF;
        
        IF p_matched_report_id IS NOT NULL THEN
            v_sql_query := v_sql_query || ' 
                AND ddoc.report_id = :matched_report_id';
        END IF;

        v_sql_query := v_sql_query || '
            ),
            ranked_drilldowns AS (
                SELECT 
                    drilldown_id, 
                    id, 
                    enhanced_text,
                    distance, 
                    text, 
                    title,
                    match_type,
                    ROW_NUMBER() OVER (
                        PARTITION BY drilldown_id 
                        ORDER BY distance ASC
                    ) AS rank
                FROM 
                    combined_results
            )
            SELECT 
                drilldown_id, id, enhanced_text, distance, text, title, 
                match_type, ''drilldown'' AS type, 
                RANK() OVER (ORDER BY distance ASC) AS similarity_search_rank
            FROM 
                ranked_drilldowns
            WHERE 
                rank = 1
            ORDER BY 
                distance ASC
            FETCH FIRST :k ROWS ONLY';

        v_cur := DBMS_SQL.OPEN_CURSOR;

        -- Load the query to fetch good drilldown search records
        DBMS_SQL.PARSE(v_cur, v_sql_query, DBMS_SQL.NATIVE);

        -- Bind variables
        DBMS_SQL.BIND_VARIABLE(v_cur, ':query_vector', v_query_vector);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':status', v_status);
        IF p_matched_report_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':matched_report_id', p_matched_report_id
            );
        END IF;
        DBMS_SQL.BIND_VARIABLE(v_cur, ':k', p_k);
        IF p_domain_id IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':domain_id', p_domain_id);
        END IF;

        -- Execute the SQL statement
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_results := DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    END fetch_top_k_similarity_search_drilldowns_using_centroid;

    -- Re-augment but keep the expansion part the same
    PROCEDURE re_augment_affected_texts(
        p_value VARCHAR2 
    ) IS
        v_re_augmented_text VARCHAR2(4000);
        v_augmented_tokens JSON;
        v_vector VECTOR(*,*);
        v_normalized_value VARCHAR2(4000) := LOWER(p_value);
        v_sql VARCHAR2(1000);  -- To hold the dynamic SQL statement
        v_domain_id  VARCHAR2(36);
    BEGIN
        lang_data_logger_pkg.log_info(
            'Re-augmenting affected texts with token: ' || p_value
        );

        -- Update queries in langdata$searchrecords
        FOR rec IN (
            SELECT id, query_text, augmented_tokens, domain_id
            FROM langdata$searchrecords
            WHERE JSON_EXISTS(
                augmented_tokens, 
                '$[*] ? (@ == $p_value)' 
                PASSING p_value AS "p_value"
            )
        ) LOOP
            -- Augment the query again using the new description
            lang_data_logger_pkg.log_info(
                'Processing search record id: ' || rec.id
            );
            lang_data_utils_pkg.augment_text(
                p_text               => rec.query_text,
                p_domain_id          => rec.domain_id,
                p_augmented_text     => v_re_augmented_text,
                p_augmented_tokens   => v_augmented_tokens
            );

            lang_data_logger_pkg.log_info(
                'Newly augmented query '
                || 'for search record id '
                || rec.id 
                || ': ' || v_re_augmented_text
            );

            -- Recompute the query vector using VECTOR_EMBEDDING
            lang_data_logger_pkg.log_info(
                'Executing vector embedding for augmented query: ' 
                || v_re_augmented_text
            );
            v_vector := lang_data_utils_pkg.get_embedding(v_re_augmented_text);

            lang_data_logger_pkg.log_info(
                'Updating search record id: ' || rec.id
            );
            -- Update the record with the new augmented query and vector
            UPDATE langdata$searchrecords
            SET augmented_query_text = v_re_augmented_text,
                query_vector = v_vector,
                augmented_tokens = v_augmented_tokens
            WHERE id = rec.id;
        END LOOP;

        -- Update queries in langdata$samplequeries
        FOR rec IN (
            SELECT id, query_text, enhanced_query_text, augmented_tokens,
                   report_id, drilldown_id
            FROM langdata$samplequeries
            WHERE JSON_EXISTS(
                augmented_tokens, 
                '$[*] ? (@ == $p_value)' 
                PASSING p_value AS "p_value"
            )
        ) LOOP
            -- Augment the query again using the new description
            lang_data_logger_pkg.log_info(
                'Processing sample query id: ' || rec.id
            );
            -- Determine the domain ID associated with the sample query.
            -- If the drilldown_id is present, fetch the domain_id from the 
            -- drilldown document.
            -- Otherwise, fall back to the domain_id of the associated report.
            -- COALESCE ensures we get the first non-NULL domain_id from the
            -- two sources.
            SELECT COALESCE(
                (
                    SELECT domain_id FROM langdata$drilldowndocuments
                    WHERE id = rec.drilldown_id
                ),
                (
                    SELECT domain_id FROM langdata$reports
                    WHERE id = rec.report_id
                )
            )
            INTO v_domain_id
            FROM dual;

            lang_data_utils_pkg.augment_text(
                p_text               => rec.query_text,
                p_domain_id          => v_domain_id,
                p_augmented_text     => v_re_augmented_text,
                p_augmented_tokens   => v_augmented_tokens
            );

            -- Append the old expansion text
            v_re_augmented_text := v_re_augmented_text || ' ' || 
                REGEXP_SUBSTR(
                    rec.enhanced_query_text, 
                    'The query uses the following filters:.*$'
                );

            lang_data_logger_pkg.log_info(
                'Newly augmented query' || 'for sample query id '
                || rec.id || ': ' || v_re_augmented_text
            );

            -- Recompute the query vector using VECTOR_EMBEDDING
            lang_data_logger_pkg.log_info(
                'Executing vector embedding for sample query id: ' 
                || rec.id
            );
            v_vector := lang_data_utils_pkg.get_embedding(v_re_augmented_text);

            lang_data_logger_pkg.log_info(
                'Updating sample query id: ' || rec.id
            );
            -- Update the record with the re-augmented enhanced query and vector
            UPDATE langdata$samplequeries
            SET enhanced_query_text = v_re_augmented_text,
                query_vector = v_vector,
                augmented_tokens = v_augmented_tokens
            WHERE id = rec.id;
        END LOOP;

        -- Update descriptions in langdata$reportdescriptions
        FOR rec IN (
            SELECT rd.id, text, enhanced_text, augmented_tokens, report_id,
                   domain_id
            FROM langdata$reportdescriptions rd
            JOIN langdata$reports r ON rd.report_id = r.id
            WHERE JSON_EXISTS(
                augmented_tokens, 
                '$[*] ? (@ == $p_value)' 
                PASSING p_value AS "p_value"
            )
        ) LOOP
            -- Augment the description again using the new description
            lang_data_logger_pkg.log_info(
                'Processing report description id: ' || rec.id
            );
            lang_data_utils_pkg.augment_text(
                p_text               => rec.text,
                p_domain_id          => rec.domain_id,
                p_augmented_text     => v_re_augmented_text,
                p_augmented_tokens   => v_augmented_tokens
            );

            v_re_augmented_text := v_re_augmented_text || ' ' || 
                REGEXP_SUBSTR(
                    rec.enhanced_text, 
                    'We need the following filters to use this.*$'
                );

            lang_data_logger_pkg.log_info(
                'Newly augmented description '
                || 'for report description id '
                || rec.id 
                || ': ' || v_re_augmented_text
            );

            -- Recompute the description vector using VECTOR_EMBEDDING
            lang_data_logger_pkg.log_info(
                'Executing vector embedding for augmented description: ' 
                || v_re_augmented_text
            );
            v_vector := lang_data_utils_pkg.get_embedding(v_re_augmented_text);

            lang_data_logger_pkg.log_info(
                'Updating report description id: ' || rec.id
            );
            -- Update the record with the re-augmented enhanced_description and
            -- vector
            UPDATE langdata$reportdescriptions
            SET enhanced_text = v_re_augmented_text,
                description_vector = v_vector,
                augmented_tokens = v_augmented_tokens
            WHERE id = rec.id;
        END LOOP;

        -- Update descriptions in langdata$drilldowndescriptions
        FOR rec IN (
            SELECT ddd.id, text, enhanced_text, augmented_tokens, drilldown_id,
                   domain_id
            FROM langdata$drilldowndescriptions ddd
            JOIN langdata$drilldowndocuments ddoc
            ON ddd.drilldown_id = ddoc.id
            WHERE JSON_EXISTS(
                augmented_tokens, 
                '$[*] ? (@ == $p_value)' 
                PASSING p_value AS "p_value"
            )
        ) LOOP
            -- Augment the description again using the new description
            lang_data_logger_pkg.log_info(
                'Processing drilldown description id: ' || rec.id
            );
            lang_data_utils_pkg.augment_text(
                p_text               => rec.text,
                p_domain_id          => rec.domain_id,
                p_augmented_text     => v_re_augmented_text,
                p_augmented_tokens   => v_augmented_tokens
            );

            v_re_augmented_text := v_re_augmented_text || ' ' || 
                REGEXP_SUBSTR(
                    rec.enhanced_text, 
                    'We need the following filters to use this.*$'
                );

            lang_data_logger_pkg.log_info(
                'Newly augmented description '
                || 'for drilldown description id '
                || rec.id 
                || ': ' || v_re_augmented_text
            );

            -- Recompute the description vector using VECTOR_EMBEDDING
            lang_data_logger_pkg.log_info(
                'Executing vector embedding for augmented description: ' 
                || v_re_augmented_text
            );
            v_vector := lang_data_utils_pkg.get_embedding(v_re_augmented_text);

            lang_data_logger_pkg.log_info(
                'Updating drilldown description id: ' || rec.id
            );
            -- Update the record with the new augmented description and vector
            UPDATE langdata$drilldowndescriptions
            SET enhanced_text = v_re_augmented_text,
                ddd_vector = v_vector,
                augmented_tokens = v_augmented_tokens
            WHERE id = rec.id;
        END LOOP;
        -- Commit changes after updating affected queries	
        COMMIT;
        lang_data_logger_pkg.log_info('All queries updated successfully.');
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred ' || '. Error: ' 
                || SQLERRM
            );
            RAISE;
            ROLLBACK;
    END re_augment_affected_texts;

    PROCEDURE update_affected_texts(
        p_table_name    IN VARCHAR2,
        p_column_name   IN VARCHAR2,
        p_enumerable    IN BOOLEAN,
        p_schema_name   IN VARCHAR2,
        p_db_link_name  IN VARCHAR2 DEFAULT NULL
    ) IS
        v_value VARCHAR2(4000);  -- Variable to store distinct values
        v_sql   VARCHAR2(1000);  -- Variable to store dynamic SQL
        v_cursor SYS_REFCURSOR;  -- Cursor for dynamic query
        v_match_document JSON;
        v_old_processed_text  VARCHAR2(4000);
        v_expansion_text      VARCHAR2(4000);
        v_enhanced_text VARCHAR2(4000);
        v_text_type VARCHAR2(255);
        v_report_id VARCHAR2(36);
        v_drilldown_id VARCHAR2(36);
        v_json_table_name VARCHAR2(255);
        v_json_column_name VARCHAR2(255);
        v_schema_name VARCHAR2(128);
    BEGIN
        v_schema_name := lang_data_utils_pkg.normalize_schema_name(
            p_schema_name
        );
        
        lang_data_logger_pkg.log_info(
            'Starting update for table: ' || v_schema_name || '.' ||
            p_table_name || ' and column: ' || p_column_name
        );

        -- Re-augment descriptions and queries
        IF p_enumerable THEN
            v_sql := 'SELECT DISTINCT TO_CHAR(' || p_column_name || ') FROM ' ||
                v_schema_name || '.' || p_table_name;
            
            IF p_db_link_name IS NOT NULL THEN
                v_sql := v_sql || '@' || p_db_link_name;
            END IF;

            -- Open cursor to execute the dynamic SQL
            OPEN v_cursor FOR v_sql;

            -- Loop through the cursor and fetch distinct values
            LOOP
                FETCH v_cursor INTO v_value;
                EXIT WHEN v_cursor%NOTFOUND;

                -- Call re_augment_affected_texts for each distinct value
                lang_data_logger_pkg.log_info(
                    'Re-augment texts for value: ' || 
                    v_schema_name || '_' || 
                    p_table_name || '_' || 
                    UPPER(p_column_name) || '_' || 
                    v_value
                );
                IF p_db_link_name IS NULL THEN
                    re_augment_affected_texts(
                        v_schema_name|| '_' ||
                        UPPER(p_table_name) || '_' || UPPER(p_column_name) ||
                        '_' || v_value
                    );
                ELSE
                    re_augment_affected_texts(
                        v_schema_name|| '_' ||
                        UPPER(p_table_name) || '_' || UPPER(p_column_name) ||
                        '_' || UPPER(p_db_link_name) || '_' || v_value
                    );
                END IF;
            END LOOP;

            -- Close the cursor
            CLOSE v_cursor;
        END IF;

        -- Re-expand descriptions and queries
        -- Update report descriptions where the match_document contains the 
        -- table and column
        FOR rec IN (
            SELECT rd.id AS report_id,
                rd.enhanced_text AS old_processed_text,
                r.match_document AS match_document,
                jt.table_name AS json_table_name,
                jt.column_name AS json_column_name,
                jt.schema_name AS json_schema_name,
                jt.db_link_name AS json_db_link_name
            FROM langdata$reports r
            JOIN langdata$reportdescriptions rd ON r.id = rd.report_id
            JOIN JSON_TABLE(
                    r.match_document,
                    '$.filters[*]' 
                    COLUMNS (
                        table_name PATH '$.table_name',
                        column_name PATH '$.column_name',
                        schema_name PATH '$.schema_name',
                        db_link_name PATH '$.db_link_name'
                    )
                ) jt
            ON LOWER(jt.table_name) = LOWER(p_table_name)
            AND LOWER(jt.column_name) = LOWER(p_column_name)
            AND LOWER(jt.schema_name) = LOWER(p_schema_name)
            AND (
                    LOWER(jt.db_link_name) = LOWER(p_db_link_name)
                    OR (jt.db_link_name IS NULL AND p_db_link_name IS NULL)
                )
        ) LOOP
            v_report_id := rec.report_id;
            v_match_document := rec.match_document;
            v_old_processed_text := rec.old_processed_text;
            v_json_table_name := rec.json_table_name;
            v_json_column_name := rec.json_column_name;
            v_text_type := 'report';
            
            -- Log the extracted JSON table values
            lang_data_logger_pkg.log_info(
                'JSON Schema Name: '|| v_schema_name ||
                ', JSON Table Name: ' || v_json_table_name ||
                ', JSON Column Name: ' || v_json_column_name
            );

            lang_data_logger_pkg.log_info(
                'Processing report ID: ' || v_report_id
            );

            -- Expand the description
            v_expansion_text := 
            lang_data_utils_pkg.generate_expansion_text(
                v_match_document, v_text_type
            );

            v_enhanced_text := REGEXP_REPLACE(
                v_old_processed_text, 
                'We need the following filters to use this.*$', 
                v_expansion_text
            );

            lang_data_logger_pkg.log_info(
                'Enhanced description for report ID: ' || v_report_id
            );

            -- Update the report description with the enhanced text and 
            -- update the vector (if applicable)
            UPDATE langdata$reportdescriptions
            SET enhanced_text = v_enhanced_text,
                description_vector = lang_data_utils_pkg.get_embedding(
                    v_enhanced_text
                )
            WHERE id = v_report_id;

            lang_data_logger_pkg.log_info(
                'Updated report description for report ID: ' || v_report_id
            );
        END LOOP;

        lang_data_logger_pkg.log_info(
            'Finished processing reports for table: ' || v_schema_name ||'.'||p_table_name ||
            ' and column: ' || p_column_name
        );

        -- Update drilldown descriptions where the match_document 
        -- contains the table and column
        FOR rec IN (
            SELECT dd.id AS drilldown_id,
                dd.enhanced_text AS old_processed_text,
                d.match_document AS match_document,
                jt.table_name AS json_table_name,
                jt.column_name AS json_column_name,
                jt.schema_name AS json_schema_name,
                jt.db_link_name AS json_db_link_name
            FROM langdata$drilldowndocuments d
            JOIN langdata$drilldowndescriptions dd ON d.id = dd.drilldown_id
            JOIN JSON_TABLE(
                    d.match_document,
                    '$.filters[*]' 
                    COLUMNS (
                        table_name PATH '$.table_name',
                        column_name PATH '$.column_name',
                        schema_name PATH '$.schema_name',
                        db_link_name PATH '$.db_link_name'
                    )
                ) jt
            ON LOWER(jt.table_name) = LOWER(p_table_name)
            AND LOWER(jt.column_name) = LOWER(p_column_name)
            AND LOWER(jt.schema_name) = LOWER(p_schema_name)
            AND (
                    LOWER(jt.db_link_name) = LOWER(p_db_link_name)
                    OR (jt.db_link_name IS NULL AND p_db_link_name IS NULL)
                )
        ) LOOP
            v_drilldown_id := rec.drilldown_id;
            v_match_document := rec.match_document;
            v_old_processed_text := rec.old_processed_text;
            v_json_table_name := rec.json_table_name;
            v_json_column_name := rec.json_column_name;
            v_text_type := 'drilldown';

            -- Log the extracted JSON table values
            lang_data_logger_pkg.log_info(
                'JSON Schema Name: '|| v_schema_name ||
                ', JSON Table Name: ' || v_json_table_name ||
                ', JSON Column Name: ' || v_json_column_name
            );

            lang_data_logger_pkg.log_info(
                'Processing drilldown ID: ' || v_drilldown_id
            );

            -- Expand the description
            v_expansion_text := 
            lang_data_utils_pkg.generate_expansion_text(
                v_match_document, v_text_type
            );

            v_enhanced_text := REGEXP_REPLACE(
                v_old_processed_text, 
                'We need the following filters to use this.*$', 
                v_expansion_text
            );

            lang_data_logger_pkg.log_info(
                'Enhanced description for drilldown ID: ' || v_drilldown_id
            );

            -- Update the drilldown description with the enhanced text and 
            -- update the vector (if applicable)
            UPDATE langdata$drilldowndescriptions
            SET enhanced_text = v_enhanced_text,
                ddd_vector = lang_data_utils_pkg.get_embedding(v_enhanced_text)
            WHERE id = v_drilldown_id;

            lang_data_logger_pkg.log_info(
                'Updated drilldown description for drilldown ID: ' || 
                 v_drilldown_id
            );
        END LOOP;

        lang_data_logger_pkg.log_info(
            'Finished processing drilldowns for table: ' ||  v_schema_name ||
            '.' || p_table_name || ' and column: ' || p_column_name
        );

        -- Process sample queries where the table and column match
        FOR rec IN (
            SELECT sq.id AS sample_query_id,
                sq.enhanced_query_text AS old_processed_text,
                sq.report_id,
                sq.drilldown_id,
                COALESCE(r.match_document, d.match_document) AS match_document,
                jt.table_name AS json_table_name,
                jt.column_name AS json_column_name,
                jt.schema_name AS json_schema_name,
                jt.db_link_name AS json_db_link_name
            FROM langdata$samplequeries sq
            LEFT JOIN langdata$reports r ON sq.report_id = r.id
            LEFT JOIN langdata$drilldowndocuments d ON sq.drilldown_id = d.id
            JOIN JSON_TABLE(
                    COALESCE(r.match_document, d.match_document),
                    '$.filters[*]' 
                    COLUMNS (
                        table_name PATH '$.table_name',
                        column_name PATH '$.column_name',
                        schema_name PATH '$.schema_name',
                        db_link_name PATH '$.db_link_name'
                    )
            ) jt
            ON LOWER(jt.table_name) = LOWER(p_table_name)
            AND LOWER(jt.column_name) = LOWER(p_column_name)
            AND LOWER(jt.schema_name) = LOWER(p_schema_name)
            AND (
                    LOWER(jt.db_link_name) = LOWER(p_db_link_name)
                    OR (jt.db_link_name IS NULL AND p_db_link_name IS NULL)
                )
        ) LOOP
            v_match_document := rec.match_document;
            v_old_processed_text := rec.old_processed_text;
            v_text_type := 'query';

            v_expansion_text := 
            lang_data_utils_pkg.generate_expansion_text(
                v_match_document, v_text_type
            );

            v_enhanced_text := REGEXP_REPLACE(
                v_old_processed_text, 
                'The query uses the following filters:.*$', 
                v_expansion_text
            );

            -- Log the update process
            lang_data_logger_pkg.log_info(
                'Re-expanded enhanced_query_text for sample query ID: ' ||
                rec.sample_query_id
            );

            -- Update the enhanced query text and vector
            UPDATE langdata$samplequeries
            SET enhanced_query_text = v_enhanced_text,
                query_vector = lang_data_utils_pkg.get_embedding(
                    v_enhanced_text
                )
            WHERE id = rec.sample_query_id;

            lang_data_logger_pkg.log_info(
                'Updated sample query ID: ' || rec.sample_query_id
            );
        END LOOP;
        -- Commit the changes after processing
        COMMIT;
        lang_data_logger_pkg.log_info(
            'Update completed for table: ' ||v_schema_name ||'.'||p_table_name 
                                           ||' and column: ' 
                                           || p_column_name
        );
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred ' || '. Error: ' 
                || SQLERRM
            );
            RAISE;
    END update_affected_texts;

    PROCEDURE check_annotation_changes_for_table_column(
        p_table_name  IN VARCHAR2,  -- Table name to check
        p_column_name IN VARCHAR2,  -- Column name to check
        p_schema_name IN VARCHAR2,
        p_db_link_name IN VARCHAR2 DEFAULT NULL,
        p_immediate   IN BOOLEAN DEFAULT FALSE
    )
    IS
        v_table_name     VARCHAR2(255);
        v_column_name    VARCHAR2(255);
        v_schema_name    VARCHAR2(255);
        v_db_link_name   VARCHAR2(255);
        v_partition_name VARCHAR2(255);
        -- Variable to store distinct values
        v_value         VARCHAR2(4000);
        -- Variable to store dynamic SQL
        v_sql           VARCHAR2(1000);
        -- Cursor for dynamic query
        v_cursor        SYS_REFCURSOR;
        -- Variable to store current annotation value
        v_current_annotation VARCHAR2(4000);
        -- Variable to store tracked annotation value
        v_tracked_annotation VARCHAR2(4000);
        -- Variable to store the job name
        v_job_name      VARCHAR2(255);
        -- Variable to store enumerable annotation value
        v_enumerable    VARCHAR2(10);
        v_domain_id     VARCHAR2(36);
    BEGIN
        v_table_name := UPPER(p_table_name);
        v_column_name := UPPER(p_column_name);
        v_schema_name := lang_data_utils_pkg.normalize_schema_name(
            p_schema_name
        );
        v_db_link_name := UPPER(p_db_link_name);

        v_partition_name := 'LANGDATA$DRILLDOWNVALUES_' || v_schema_name
                            || '_' || v_table_name || '_' || v_column_name;
    
        IF v_db_link_name IS NOT NULL THEN
            v_partition_name := v_partition_name || '_' || v_db_link_name;
        END IF;

        SELECT domain_id INTO v_domain_id
        FROM langdata$value_vector_partition_descriptions
        WHERE partition_name = v_partition_name;

        lang_data_logger_pkg.log_info(
            'Checking for annotation changes for '||v_schema_name||
            '.'|| v_table_name 
            || '.' || v_column_name
        );
        IF v_db_link_name IS NOT NULL THEN
            lang_data_logger_pkg.log_info(
                'In remote database via database link: ' || v_db_link_name
            );
        END IF;

        -- Fetch the current annotation
        v_current_annotation := lang_data_utils_pkg.get_annotation(
            v_table_name,
            v_column_name,
            v_schema_name,
            'DESCRIPTION',
            v_db_link_name
        );

        BEGIN
            -- Fetch the tracked annotation from langdata$annotations
            SELECT t.annotation_value
            INTO v_tracked_annotation
            FROM langdata$value_vector_metadata t
            WHERE t.table_name = v_table_name
            AND t.column_name = v_column_name
            AND t.schema_name = v_schema_name
            AND (
                (db_link_name = v_db_link_name) OR
                (db_link_name IS NULL AND v_db_link_name IS NULL)
            );
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                v_tracked_annotation := NULL;
        
        END;

        -- Check if the current annotation is different from 
        -- the tracked annotation
        lang_data_logger_pkg.log_info(
            'Old annotation: ' || v_tracked_annotation
        );
        lang_data_logger_pkg.log_info(
            'New annotation: ' || v_current_annotation
        );
        IF NVL(
            v_current_annotation, 'NULL_VALUE'
        ) != NVL(v_tracked_annotation, 'NULL_VALUE') THEN
            lang_data_logger_pkg.log_info(
                'Annotation changed for ' || v_schema_name ||'.'|| v_table_name || '.' || 
                v_column_name
            );

            -- Check if the column has the 'ENUMERABLE' annotation
            v_enumerable := lang_data_utils_pkg.get_annotation(
                v_table_name,
                v_column_name,
                v_schema_name,
                'ENUMERABLE',
                v_db_link_name
            );
            -- Treat missing ENUMERABLE as 'No'
            IF v_enumerable IS NULL THEN
                v_enumerable := 'No';
            END IF;

            -- If ENUMERABLE is 'Yes', schedule a job to regenerate
            -- value vectors
            IF v_enumerable = 'Yes' THEN
                IF not p_immediate THEN
                    v_job_name := 'JOB_LANGDATA_'|| v_schema_name ||'_'|| 
                                  v_table_name || '_' || v_column_name;
                    IF v_db_link_name IS NOT NULL THEN
                        v_job_name := v_job_name || '_' || v_db_link_name;
                    END IF;
                    v_job_name := v_job_name || '_VALUE_VECTORS';

                    lang_data_utils_pkg.create_or_replace_job(
                        p_job_name        => v_job_name,
                        p_job_action      =>
                            'BEGIN ' ||
                            'lang_data_utils_pkg.create_value_vector_table(''' || 
                                            v_table_name || ''', ''' || 
                                            v_column_name ||''', ''' || 
                                            v_schema_name ||
                                            ''', TRUE, FALSE, ''' ||
                                            NVL(v_db_link_name, '') ||
                                            ''', ''' || NVL(v_domain_id, '') ||
                                            '''); END;',
                        p_job_class =>lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                        p_comments        => 'Update value vector tables',
                        p_priority        => 1,
                        p_restart_on_fail => TRUE,
                        p_restart_on_rec  => TRUE
                    );
                ELSE
                    lang_data_utils_pkg.create_value_vector_table(
                        v_table_name,
                        v_column_name,
                        v_schema_name,
                        TRUE,
                        FALSE,
                        v_db_link_name,
                        v_domain_id
                    );
                END IF;
            END IF;

            -- Update metadata in langdata$value_vector_partition_descriptions
            BEGIN
                UPDATE langdata$value_vector_partition_descriptions d
                SET 
                    description = v_current_annotation,
                    desc_comment_combined = RTRIM(
                        NVL(v_current_annotation, '') || ' ' ||
                        NVL(d.comment_text, '')
                    ),
                    desc_comment_vector = lang_data_utils_pkg.get_embedding(
                        RTRIM(NVL(v_current_annotation, '') || ' ' ||
                        NVL(d.comment_text, ''))
                    )
                WHERE partition_name = v_partition_name;
                DBMS_SESSION.SLEEP(5);
            EXCEPTION
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_warn(
                        'Failed to update metadata for partition ' ||
                        v_partition_name || ': ' || SQLERRM
                    );
            END;

            IF not p_immediate THEN
                v_job_name := 'JOB_LANGDATA_'|| v_schema_name ||'_'|| 
                              v_table_name || '_' || v_column_name;
                IF v_db_link_name IS NOT NULL THEN
                    v_job_name := v_job_name || '_' || v_db_link_name;
                END IF;
                v_job_name := v_job_name || '_UPDATE_TEXTS';

                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name        => v_job_name,
                    p_job_action      => 
                    'BEGIN lang_data_utils_pkg.update_affected_texts('''||
                        v_table_name || ''', ''' || v_column_name || ''', ' ||
                        CASE 
                            WHEN v_enumerable = 'Yes' THEN 'TRUE'
                            ELSE 'FALSE'
                        END || ', ''' ||
                        v_schema_name || ''', ''' || NVL(v_db_link_name, '') ||
                    '''); END;',
                    p_job_class    => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments        => 'Update affected texts',
                    p_priority        => 2,
                    p_restart_on_fail => TRUE,
                    p_restart_on_rec  => TRUE
                );
            ELSE
                lang_data_utils_pkg.update_affected_texts(
                    v_table_name,
                    v_column_name,
                    CASE 
                        WHEN v_enumerable = 'Yes' THEN TRUE
                        ELSE FALSE
                    END,
                    v_schema_name,
                    v_db_link_name
                );
            END IF;
        ELSE
            lang_data_logger_pkg.log_info(
                'No annotation change detected for ' || v_schema_name ||'.'||
                 v_table_name || '.' || v_column_name
            );
        END IF;

        lang_data_logger_pkg.log_info(
            'Finished checking annotation changes for ' || v_schema_name ||'.'||
            v_table_name || '.' || v_column_name
        );

        -- Log the change
        UPDATE langdata$value_vector_metadata
        SET annotation_value = v_current_annotation,
            annotation_changed_at = CURRENT_TIMESTAMP
        WHERE table_name = v_table_name
        AND column_name = v_column_name
        AND schema_name = v_schema_name
        AND (
                (db_link_name = v_db_link_name) OR
                (db_link_name IS NULL AND v_db_link_name IS NULL)
            );
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred ' || '. Error: ' 
                    || SQLERRM);
    END check_annotation_changes_for_table_column;

    PROCEDURE check_annotation_changes_for_user_tables
    IS
        CURSOR c_annotations IS
            SELECT table_name,
                column_name,
                schema_name,
                db_link_name
            FROM langdata$value_vector_metadata;

    BEGIN
        FOR rec IN c_annotations LOOP
            lang_data_utils_pkg.check_annotation_changes_for_table_column(
                p_table_name   => rec.table_name,
                p_column_name  => rec.column_name,
                p_schema_name  => rec.schema_name,
                p_db_link_name => rec.db_link_name,
                -- Procedure supposed to be a part of backend-fix-up periodical
                -- job, no need to schedule jobs within a job
                p_immediate => TRUE
            );
        END LOOP;
    END check_annotation_changes_for_user_tables;

    PROCEDURE create_or_replace_job (
        p_job_name        IN VARCHAR2,   -- Name of the job
        p_job_action      IN VARCHAR2,   -- PL/SQL block or program to execute
        p_job_class       IN VARCHAR2,   -- Job class to associate
        p_comments        IN VARCHAR2,   -- Comments for the job
        p_priority        IN NUMBER,     -- Job priority (1 = highest)
        p_restart_on_fail IN BOOLEAN,    -- Restart on failure
        p_restart_on_rec  IN BOOLEAN,    -- Restart on recovery
        p_repeat_interval  IN VARCHAR2 DEFAULT NULL  -- Optional repeat interval
    )
    AS
    BEGIN
        -- Drop the job if it already exists
        BEGIN
            DBMS_SCHEDULER.drop_job(job_name => p_job_name, force => TRUE);
            lang_data_logger_pkg.log_info(
                'Job ' || p_job_name || ' dropped successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                -- Catch the exception if the job does not exist
                IF SQLCODE != -27475 THEN -- Error code for "job does not exist"
                    lang_data_logger_pkg.log_info(
                        'Error dropping job: ' || SQLERRM
                    );
                ELSE
                    lang_data_logger_pkg.log_info(
                        'Job ' || p_job_name ||
                        ' does not exist, proceeding to create.'
                    );
                END IF;
        END;

        -- Create a new job
        lang_data_logger_pkg.log_info('Creating job: ' || p_job_name);
        BEGIN
            DBMS_SCHEDULER.create_job(
                job_name   => p_job_name,
                job_type   => 'PLSQL_BLOCK',
                job_action => p_job_action,
                start_date => SYSTIMESTAMP,
                repeat_interval  => p_repeat_interval,
                job_class  => p_job_class,
                comments   => p_comments,
                enabled    => FALSE
            );

            -- Set additional attributes
            DBMS_SCHEDULER.SET_ATTRIBUTE(
                name      => p_job_name,
                attribute => 'job_priority',
                value     => p_priority
            );

            DBMS_SCHEDULER.SET_ATTRIBUTE(
                name      => p_job_name,
                attribute => 'restart_on_failure',
                value     => p_restart_on_fail
            );

            DBMS_SCHEDULER.SET_ATTRIBUTE(
                name      => p_job_name,
                attribute => 'restart_on_recovery',
                value     => p_restart_on_rec
            );

            DBMS_SCHEDULER.enable(p_job_name);

            lang_data_logger_pkg.log_info(
                'Job ' || p_job_name || ' created successfully.'
            );
        EXCEPTION
            WHEN OTHERS THEN
                lang_data_logger_pkg.log_fatal(
                    'Error creating job: ' || SQLERRM
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_job_creation_failed
                );
        END;
    END create_or_replace_job;

    FUNCTION generate_id
    RETURN VARCHAR2 IS
        v_id VARCHAR2(36);
    BEGIN
        v_id := LOWER(REGEXP_REPLACE(
                                        RAWTOHEX(SYS_GUID()),
                                        '^(.{8})(.{4})(.{4})(.{4})(.{12})$',
                                        '\1-\2-\3-\4-\5'
                                    )
                                );
        RETURN v_id;
    END generate_id;

    PROCEDURE update_report_query_cluster(
        p_report_id VARCHAR2
    ) IS
        v_cluster_exists NUMBER;
        v_old_centroid_vector VECTOR;
        v_centroid_vector VECTOR;
        v_cluster_id VARCHAR2(36);

    BEGIN
        -- Check if a cluster entry exists for the given report_id
        BEGIN
            SELECT centroid_vector 
            INTO v_old_centroid_vector
            FROM langdata$reportquerycluster
            WHERE report_id = p_report_id
            FETCH FIRST 1 ROWS ONLY;

            v_cluster_exists := 1;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                v_cluster_exists := 0;
                v_old_centroid_vector := NULL;
        END;

        -- Compute centroid vector using AVG function
        BEGIN
            SELECT TO_VECTOR(AVG(query_vector), *, FLOAT32)
            INTO v_centroid_vector
            FROM langdata$samplequeries
            WHERE report_id = p_report_id AND status = 'Published';

            -- If no data found, set centroid to NULL
            IF v_centroid_vector IS NULL THEN
                IF v_cluster_exists = 1 THEN
                    DELETE FROM langdata$reportquerycluster
                    WHERE report_id = p_report_id;
                END IF;
                RETURN;
            END IF;

        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                -- No queries found, delete existing cluster if it exists
                IF v_cluster_exists = 1 THEN
                    DELETE FROM langdata$reportquerycluster
                    WHERE report_id = p_report_id;
                END IF;
                RETURN;
        END;

        -- Log the distance if the old centroid exists
        IF v_cluster_exists = 1 AND v_old_centroid_vector IS NOT NULL THEN
            DECLARE
                v_distance NUMBER;
            BEGIN
                v_distance := VECTOR_DISTANCE(
                    v_centroid_vector, v_old_centroid_vector
                );
                lang_data_logger_pkg.log_info(
                    'Distance between old and new centroid vectors: ' ||
                    v_distance
                );
            END;
        END IF;

        -- If the cluster already exists, update it
        -- otherwise, insert a new entry
        IF v_cluster_exists = 1 THEN
            UPDATE langdata$reportquerycluster
            SET centroid_vector = v_centroid_vector
            WHERE report_id = p_report_id;
        ELSE
            v_cluster_id := SYS_GUID(); -- Generate a unique identifier
            INSERT INTO langdata$reportquerycluster (
                cluster_id, report_id, centroid_vector
            )
            VALUES (v_cluster_id, p_report_id, v_centroid_vector);
        END IF;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred. Error: ' || SQLERRM
            );
            RAISE;
    END update_report_query_cluster;

    PROCEDURE update_drilldown_query_cluster(
        p_drilldown_id VARCHAR2
    ) IS
        v_cluster_exists NUMBER;
        v_old_centroid_vector VECTOR;
        v_centroid_vector VECTOR;
        v_cluster_id VARCHAR2(36);

    BEGIN
        -- Check if a cluster entry exists for the given drilldown_id
        BEGIN
            SELECT centroid_vector 
            INTO v_old_centroid_vector
            FROM langdata$drilldownquerycluster
            WHERE drilldown_id = p_drilldown_id
            FETCH FIRST 1 ROWS ONLY;

            v_cluster_exists := 1;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                v_cluster_exists := 0;
                v_old_centroid_vector := NULL;
        END;

        -- Compute centroid vector using AVG function
        BEGIN
            SELECT TO_VECTOR(AVG(query_vector), *, FLOAT32)
            INTO v_centroid_vector
            FROM langdata$samplequeries
            WHERE drilldown_id = p_drilldown_id AND status = 'Published';

            -- If no data found, set centroid to NULL
            IF v_centroid_vector IS NULL THEN
                IF v_cluster_exists = 1 THEN
                    DELETE FROM langdata$drilldownquerycluster
                    WHERE drilldown_id = p_drilldown_id;
                END IF;
                RETURN;
            END IF;

        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                -- No queries found, delete existing cluster if it exists
                IF v_cluster_exists = 1 THEN
                    DELETE FROM langdata$drilldownquerycluster
                    WHERE drilldown_id = p_drilldown_id;
                END IF;
                RETURN;
        END;

        -- Log the distance if the old centroid exists
        IF v_cluster_exists = 1 AND v_old_centroid_vector IS NOT NULL THEN
            DECLARE
                v_distance NUMBER;
            BEGIN
                v_distance := VECTOR_DISTANCE(
                    v_centroid_vector, v_old_centroid_vector
                );
                lang_data_logger_pkg.log_info(
                    'Distance between old and new centroid vectors: ' ||
                    v_distance
                );
            END;
        END IF;

        -- If the cluster already exists, update it
        -- otherwise, insert a new entry
        IF v_cluster_exists = 1 THEN
            UPDATE langdata$drilldownquerycluster
            SET centroid_vector = v_centroid_vector
            WHERE drilldown_id = p_drilldown_id;
        ELSE
            v_cluster_id := SYS_GUID(); -- Generate a unique identifier
            INSERT INTO langdata$drilldownquerycluster (
                cluster_id, drilldown_id, centroid_vector
            )
            VALUES (v_cluster_id, p_drilldown_id, v_centroid_vector);
        END IF;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred. Error: ' || SQLERRM
            );
            RAISE;
    END update_drilldown_query_cluster;

    PROCEDURE decrement_vector_table_references (
        p_match_document IN JSON
    ) IS
        v_filters        JSON_ARRAY_T;
        v_filter_item    JSON_OBJECT_T;
        v_table_name     VARCHAR2(128);
        v_column_name    VARCHAR2(128);
        v_schema_name   VARCHAR2(128);
        v_vec_table_name VARCHAR2(256);
        v_table_exists   NUMBER;
        v_ref_count      NUMBER;
        v_use_ner        BOOLEAN;
    BEGIN
        v_filters := JSON_OBJECT_T.parse(
            json_serialize(p_match_document)
        ).GET_ARRAY('filters');

        IF v_filters IS NOT NULL THEN 
            -- Iterate over filters
            FOR indx IN 0 .. v_filters.get_size - 1 LOOP
                v_filter_item := JSON_OBJECT_T(v_filters.GET(to_number(indx)));
                v_table_name := UPPER(v_filter_item.GET_STRING('table_name'));
                v_schema_name := lang_data_utils_pkg.normalize_schema_name(
                    v_filter_item.GET_STRING('schema_name')
                );
                v_column_name := UPPER(v_filter_item.GET_STRING('column_name'));
                v_use_ner := v_filter_item.GET_BOOLEAN('use_ner');
                
                -- Skip iteration if v_use_ner is TRUE or if table_name or 
                -- column_name is NULL
                IF v_use_ner OR v_table_name IS NULL OR v_column_name IS NULL 
                OR v_schema_name IS NULL
                THEN
                    CONTINUE;
                END IF;

                v_vec_table_name := 'LANGDATA$DRILLDOWNVALUES_' || 
                v_schema_name||'_'|| v_table_name || '_' || v_column_name;

                -- Check if the vector table exists
                SELECT COUNT(*) INTO v_table_exists
                FROM langdata$value_vector_metadata
                WHERE vvec_table_name = UPPER(v_vec_table_name);

                IF v_table_exists > 0 THEN  
                    -- Fetch and decrement ref_count
                    SELECT ref_count INTO v_ref_count
                    FROM langdata$value_vector_metadata
                    WHERE vvec_table_name = v_vec_table_name FOR UPDATE;

                    v_ref_count := v_ref_count - 1;

                    -- Update or delete table based on ref_count
                    IF v_ref_count > 0 THEN
                        UPDATE langdata$value_vector_metadata
                        SET ref_count = v_ref_count
                        WHERE vvec_table_name = v_vec_table_name;

                        lang_data_logger_pkg.log_info(
                            'Reference count decremented for ' ||
                            v_vec_table_name
                        );
                    ELSE
                        BEGIN
                            lang_data_utils_pkg.drop_value_vector_partition(
                                v_vec_table_name
                            );
                            DELETE FROM langdata$value_vector_metadata
                            WHERE vvec_table_name = v_vec_table_name;

                            lang_data_logger_pkg.log_info(
                                'Dropped partition ' || v_vec_table_name 
                                || ' as ref_count reached zero.'
                            );
                        EXCEPTION
                            WHEN OTHERS THEN
                                lang_data_logger_pkg.log_error(
                                    'Error dropping partition ' ||
                                    v_vec_table_name || ': ' || SQLERRM
                                );
                                RAISE;
                        END;
                    END IF;
                END IF;
            END LOOP;
        ELSE
           lang_data_logger_pkg.log_debug(
                                    'No filters found in match document'); 
        END IF;
    END decrement_vector_table_references;

    FUNCTION get_embedding(
        p_document IN VARCHAR2
    )
    RETURN VECTOR
    IS
        v_embedding VECTOR;
        v_config_model VARCHAR2(100);
        v_model VARCHAR2(100);
        v_credential_name   VARCHAR2(128);
        v_endpoint          VARCHAR2(2000);

        /*
        -- DBMS_VECTOR implementation
        v_params_json   JSON;
        v_params_json_obj JSON_OBJECT_T;
        v_max_count NUMBER;
        v_max_count_v VARCHAR2(100);
        v_embedding_provider VARCHAR2(100);
        */

        v_embedding_compartment_ocid VARCHAR2(4000);
        v_full_url          VARCHAR2(4000);
        v_serving_mode_json JSON_OBJECT_T;
        v_doc_json_array    JSON_ARRAY_T;
        v_req_body_json     JSON_OBJECT_T;
        v_req_body_hash     VARCHAR2(4000);
        v_req_body_hash_raw VARCHAR2(4000);
        v_req_headers_json  JSON_OBJECT_T;
        v_full_response     DBMS_CLOUD_TYPES.resp;
        v_full_response_json    JSON_OBJECT_T;
        v_embedding_response_array  JSON_ARRAY_T;
    BEGIN
        -- contains model name prefixed with model type (in-database) or
        -- OCI
        v_config_model := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_EMBEDDING_MODEL'
        );
        -- extract model name from config parameter
        v_model := SUBSTR(v_config_model, INSTR(v_config_model, '_') + 1);

        IF v_config_model LIKE 'DB_%' THEN
            BEGIN
                EXECUTE IMMEDIATE
                    'SELECT VECTOR_EMBEDDING( '
                    || v_model 
                    ||' USING :document AS DATA ) FROM dual' 
                    INTO v_embedding USING p_document;
            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -40284 THEN
                        lang_data_errors_pkg.raise_error(
                            lang_data_errors_pkg.c_resource_not_found
                        );
                    ELSE
                        RAISE;
                    END IF;
            END;

        ELSIF v_config_model LIKE 'REST_%' THEN
            
            v_credential_name := lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_OCI_CRED'
            );
            v_endpoint := lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_GENAI_ENDPOINT'
            );

            v_embedding_compartment_ocid := 
                lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_OCI_COMPARTMENT'
            );

            IF v_credential_name IS NULL THEN
                lang_Data_logger_pkg.log_error(
                    'Database OCI credentials not set');
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_OCI_credential_not_set
                );
            END IF;

            IF v_endpoint IS NULL THEN
                lang_Data_logger_pkg.log_error('GenAI endpoint not set');
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_OCI_GenAI_endpoint_not_set
                );
            END IF;

            v_full_url := 
                RTRIM(v_endpoint,'/') || '/20231130/actions/embedText';

            -- Create servingMode Json
            v_serving_mode_json := JSON_OBJECT_T('{}');
            v_serving_mode_json.put('servingType', 'ON_DEMAND');
            v_serving_mode_json.put('modelId', v_model);

            -- Create documents json array
            v_doc_json_array := JSON_ARRAY_T('[]');
            v_doc_json_array.append(p_document);

            -- Create request body
            v_req_body_json := JSON_OBJECT_T('{}');
            v_req_body_json.put('compartmentId', v_embedding_compartment_ocid);
            v_req_body_json.put('servingMode', v_serving_mode_json);
            v_req_body_json.put('inputs', v_doc_json_array);
            
            v_req_body_hash := DBMS_CRYPTO.hash(
                v_req_body_json.to_blob, DBMS_CRYPTO.HASH_SH256);
            v_req_body_hash_raw := UTL_RAW.cast_to_varchar2(
                UTL_ENCODE.base64_encode(v_req_body_hash));
            
            -- Create request headers
            v_req_headers_json := JSON_OBJECT_T('{}');
            v_req_headers_json.put('accept', 'application/json');
            v_req_headers_json.put('content-type', 'application/json');
            v_req_headers_json.put('opc-request-id', UPPER(sys_guid()));
            v_req_headers_json.put('opc-client-info','Oracle-PlsqlSDK/1.9');
            v_req_headers_json.put('x-content-sha256', v_req_body_hash_raw);
            
            lang_data_logger_pkg.log_debug(
                'Embedding Request URL: '|| v_full_url);
            lang_data_logger_pkg.log_debug(
                'Embedding Request Header: '|| v_req_headers_json.to_clob());
            lang_data_logger_pkg.log_debug(
                'Embedding Request Body: '|| v_req_body_json.to_clob());
            
            -- Send REST request
            v_full_response := DBMS_CLOUD.send_request(
                credential_name => v_credential_name,
                uri             => v_full_url,
                method          => 'POST',
                headers         => v_req_headers_json.to_clob(),
                body            => v_req_body_json.to_blob()
            );

            v_full_response_json := 
                JSON_OBJECT_T(DBMS_CLOUD.get_response_text(v_full_response));

            v_embedding_response_array := 
                v_full_response_json.get_array('embeddings');

            -- Take only first vector as we provided a single text document
            v_embedding := TO_VECTOR(
                v_embedding_response_array.get(0).to_clob(), *, *, DENSE);
            
        ELSE
            -- Other supported model endpoints can be added in future
            -- If no model is specified then return null embedding
            v_embedding := NULL;
        END IF;

        RETURN v_embedding;
    END get_embedding;

    FUNCTION get_schema_version
        RETURN VARCHAR2 IS
        v_schema_version VARCHAR2(255);
    BEGIN
        SELECT version_number
        INTO v_schema_version
        FROM langdata$schemaversion
        ORDER BY created_at DESC
        FETCH FIRST 1 ROWS ONLY;

        RETURN v_schema_version;
    EXCEPTION
        WHEN OTHERS THEN
            RETURN NULL;
    END get_schema_version;

    PROCEDURE drop_enumerable_set_value_vector_tables (
        p_document_id     IN VARCHAR2,
        p_match_document  IN JSON
    )
    IS
        v_match_document_obj  JSON_OBJECT_T;
        v_filters             JSON_ARRAY_T;
        v_filter_item         JSON_OBJECT_T;
        v_use_ner             BOOLEAN;
        v_filter_name         VARCHAR2(255);
        v_vec_table_name      VARCHAR2(255);
        v_md5                 VARCHAR2(32);
    BEGIN
        v_match_document_obj := JSON_OBJECT_T.parse(
            JSON_SERIALIZE(p_match_document)
        );

        v_filters := v_match_document_obj.get_array('filters');

        IF v_filters IS NULL THEN
            RETURN;
        END IF;

        FOR indx IN 0 .. v_filters.get_size - 1
        LOOP
            v_filter_item := JSON_OBJECT_T(v_filters.get(to_number(indx)));
            v_use_ner     := v_filter_item.get_boolean('use_ner');

            IF v_use_ner THEN
                CONTINUE;
            END IF;

            v_filter_name := UPPER(v_filter_item.get_string('filter_name'));

            IF v_filter_item.has('enumerable_set') THEN
                v_md5 := RAWTOHEX(
                    DBMS_CRYPTO.HASH(
                        UTL_RAW.CAST_TO_RAW(p_document_id),
                        DBMS_CRYPTO.HASH_MD5
                    )
                );

                v_vec_table_name := 'LANGDATA$DRILLDOWNVALUES_' ||
                                    v_filter_name || '_' || v_md5;

                BEGIN
                    lang_data_utils_pkg.drop_value_vector_partition(
                        v_vec_table_name
                    );
                EXCEPTION
                    WHEN OTHERS THEN
                        NULL;  -- Ignore if table doesn't exist
                END;
            END IF;
        END LOOP;
    END drop_enumerable_set_value_vector_tables;

    FUNCTION augment_text_with_ner_entities (
        p_text IN VARCHAR2,
        p_entities IN JSON
    ) RETURN VARCHAR2
    IS
        v_result       VARCHAR2(32767) := p_text;
        v_augmented    VARCHAR2(32767);
        v_entities_arr JSON_ARRAY_T := JSON_ARRAY_T(p_entities);
        v_entity_obj   JSON_OBJECT_T;
        v_offset       INTEGER := 0;
    BEGIN
        IF p_entities IS NULL THEN
            RETURN '';
        END IF;
        FOR i IN REVERSE 0 .. v_entities_arr.get_size - 1 LOOP
            v_entity_obj := JSON_OBJECT_T(v_entities_arr.get(i));
            DECLARE
                v_start INTEGER := v_entity_obj.get_number('start');
                v_end   INTEGER := v_entity_obj.get_number('end');
                v_text  VARCHAR2(4000) := v_entity_obj.get_string('text');
                v_label VARCHAR2(100)  := v_entity_obj.get_string('label');
                v_aug   VARCHAR2(4000);
            BEGIN
                v_aug := v_text || ' (' || v_label || ')';

                v_result := SUBSTR(v_result, 1, v_start) ||
                            v_aug ||
                            SUBSTR(v_result, v_end + 1);
            END;
        END LOOP;

        RETURN v_result;
    END augment_text_with_ner_entities;

    FUNCTION json_array_to_clob (
        p_json_array JSON_ARRAY_T
    ) RETURN CLOB
    IS
        v_clob          CLOB;
        v_json_object   JSON_OBJECT_T;
    BEGIN
        v_clob := '[';
        FOR idx IN 0 .. p_json_array.get_size - 1 LOOP
            v_json_object := TREAT(
                p_json_array.get(
                    idx
                ) AS JSON_OBJECT_T
            );
            IF idx > 0 THEN
                v_clob := v_clob || ',';
            END IF;
            v_clob := v_clob || v_json_object.to_string;
        END LOOP;
        v_clob := v_clob || ']';
        RETURN v_clob;
    END json_array_to_clob;

    FUNCTION get_or_create_domain(
        p_domain_name IN VARCHAR2
    ) RETURN VARCHAR2 
    IS
        v_domain_id         VARCHAR2(36);
    BEGIN

        v_domain_id := get_domain_id(
            p_domain_name => p_domain_name
        );

        IF v_domain_id IS NULL THEN
            v_domain_id := lang_data_utils_pkg.generate_id();
            INSERT INTO langdata$domains(domain_id, name)
            VALUES (v_domain_id, LOWER(p_domain_name));

            lang_data_logger_pkg.log_info('Created Domain "' || p_domain_name ||
            '" with id: ' || v_domain_id);
        END IF;

        RETURN v_domain_id;
    END get_or_create_domain;

    FUNCTION get_domain_id(
        p_domain_name IN VARCHAR2
    ) RETURN VARCHAR2
    IS
        v_domain_id VARCHAR2(36);
    BEGIN
        SELECT domain_id INTO v_domain_id
        FROM langdata$domains
        WHERE LOWER(name) = LOWER(p_domain_name);

        RETURN v_domain_id;

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            RETURN NULL;
    END get_domain_id;

    FUNCTION get_match_confidence (
        p_distance     IN NUMBER
    ) RETURN NUMBER
    IS
        v_clamped_dist      NUMBER;
        v_scaled            NUMBER;
        v_min_dist          NUMBER;
        v_max_dist          NUMBER;
        v_gamma             NUMBER;
    BEGIN
        v_min_dist := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_MODEL_MIN_DIST'
        );
        v_max_dist := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_MODEL_MAX_DIST'
        );
        v_gamma := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_MODEL_GAMMA'
        );
        -- Clamp distance within bounds
        v_clamped_dist := LEAST(
            GREATEST(p_distance, v_min_dist), 
            v_max_dist
        );

        IF v_max_dist = v_min_dist THEN
            RETURN 0;
        END IF;

        -- Invert distance to compute similarity, then scale
        v_scaled := (v_max_dist - v_clamped_dist) / (v_max_dist - v_min_dist);
        IF v_gamma != 1 THEN
            v_scaled := POWER(v_scaled, v_gamma);
        END IF;

        RETURN ROUND(v_scaled * 100);
    END get_match_confidence;

    PROCEDURE print_clob(p_clob IN CLOB) IS
        v_buffer    VARCHAR2(32767);
        v_pos       INTEGER := 1;
        v_clob_len  INTEGER := DBMS_LOB.GETLENGTH(p_clob);
        v_chunk     INTEGER := 32000;-- safe chunk size for DBMS_OUTPUT.PUT_LINE
    BEGIN
        WHILE v_pos <= v_clob_len LOOP
            v_buffer := DBMS_LOB.SUBSTR(p_clob, v_chunk, v_pos);
            DBMS_OUTPUT.PUT_LINE(v_buffer);
            v_pos := v_pos + v_chunk;
        END LOOP;
    END print_clob;

    PROCEDURE sync_user_tables_changed_values
    IS
        CURSOR c_metadata IS SELECT * FROM langdata$value_vector_metadata;

        v_partition_name   VARCHAR2(512);
        v_sql              CLOB;
        v_exists_in_vvec      NUMBER;
        v_exists_in_usr_tbl   NUMBER;
        v_table_ref        VARCHAR2(512);
        v_column_val       VARCHAR2(4000);

        TYPE t_rowid IS TABLE OF ROWID;
        TYPE t_val IS TABLE OF VARCHAR2(4000);
        t_rowids           t_rowid;
        t_earliest_vals    t_val;
        t_latest_vals      t_val;
        t_vals             t_val;

        i                  PLS_INTEGER;
        v_lookback_minutes VARCHAR(10);

    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting sync_changed_values procedure.'
        );

        v_lookback_minutes := lang_data_config_pkg.get_config_parameter(
            p_name => 'LANG_DATA_FLASHBACK_LOOKBACK_MINUTES'
        );

        FOR rec IN c_metadata LOOP
            lang_data_logger_pkg.log_info(
                'Processing table: ' || rec.schema_name || '.' ||
                rec.table_name || ', column: ' || rec.column_name
            );

            -- Construct partition name
            v_partition_name := 'LANGDATA$DRILLDOWNVALUES_' ||
                                rec.schema_name || '_' || rec.table_name ||
                                '_' || rec.column_name;
            IF rec.db_link_name IS NOT NULL THEN
                v_partition_name := v_partition_name || '_' || rec.db_link_name;
            END IF;

            -- Construct table reference with optional DB link
            v_table_ref := rec.schema_name || '.' || rec.table_name;
            IF rec.db_link_name IS NOT NULL THEN
                v_table_ref := v_table_ref || '@' || rec.db_link_name;
            END IF;

            -- Build a Flashback Versions Query:
            -- This query retrieves all rows whose data has changed within the
            -- past specified period of time, including INSERTs, UPDATEs, and
            -- DELETEs.
            -- For each rowid, it extracts the earliest and latest values of the
            -- target column during that time window.
            v_sql :=
                'SELECT rowid, ' ||
                'MAX(' || rec.column_name || ') KEEP (DENSE_RANK FIRST ' ||
                    'ORDER BY VERSIONS_STARTTIME ASC), ' ||
                'MAX(' || rec.column_name || ') KEEP (DENSE_RANK FIRST ' ||
                    'ORDER BY VERSIONS_STARTTIME DESC) ' ||
                'FROM ' || v_table_ref || ' ' ||
                'VERSIONS BETWEEN TIMESTAMP ' ||
                    'SYSTIMESTAMP - INTERVAL ''' || v_lookback_minutes || 
                    ''' MINUTE AND SYSTIMESTAMP ' ||
                'WHERE VERSIONS_OPERATION IS NOT NULL ' ||
                'GROUP BY rowid';

            lang_data_logger_pkg.log_info(
                'Executing flashback query: ' || v_sql
            );

            EXECUTE IMMEDIATE v_sql
            BULK COLLECT INTO
            t_rowids, t_earliest_vals, t_latest_vals;

            FOR i IN 1 .. t_rowids.COUNT LOOP
                -- Load earliest and latest into temp list
                t_vals := t_val();
                t_vals.EXTEND(2);
                t_vals(1) := t_earliest_vals(i);
                t_vals(2) := t_latest_vals(i);

                FOR j IN 1 .. 2 LOOP
                    v_column_val := t_vals(j);

                    -- Skip NULL value
                    IF v_column_val IS NULL THEN
                        CONTINUE;
                    END IF;

                    -- Check the existence of the value in value vector
                    -- partition and user table
                    -- If it exists in user table only, insert into value vector
                    -- partition
                    -- If it exists in value vector partition only, remove it
                    -- from value vector partition
                    SELECT COUNT(*) INTO v_exists_in_vvec
                    FROM langdata$drilldownvalues
                    WHERE value = v_column_val
                    AND partition_name = v_partition_name;

                    v_sql := 'SELECT COUNT(*) FROM ' || v_table_ref || 
                            ' WHERE ' || rec.column_name || ' = :1';
                    EXECUTE IMMEDIATE v_sql INTO v_exists_in_usr_tbl
                    USING v_column_val;

                    IF v_exists_in_vvec = 0 AND v_exists_in_usr_tbl > 0 THEN
                        lang_data_logger_pkg.log_info(
                            'Inserting new value: ' || v_column_val ||
                            ' into ' || v_partition_name
                        );
                        INSERT INTO langdata$drilldownvalues (
                            value, vvec, partition_name
                        )
                        VALUES (
                            v_column_val,
                            lang_data_utils_pkg.get_embedding(v_column_val),
                            v_partition_name
                        );
                    ELSIF v_exists_in_vvec > 0 AND v_exists_in_usr_tbl = 0 THEN
                        lang_data_logger_pkg.log_info(
                            'Removing old value: ' || v_column_val ||
                            ' from ' || v_partition_name
                        );
                        DELETE FROM langdata$drilldownvalues
                        WHERE value = v_column_val
                        AND partition_name = v_partition_name;
                    END IF;
                END LOOP;
            END LOOP;
        END LOOP;

        lang_data_logger_pkg.log_info(
            'sync_changed_values procedure completed successfully.'
        );

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_info(
                'Error in sync_changed_values: ' || SQLERRM
            );
            RAISE;
    END sync_user_tables_changed_values;

    PROCEDURE manage_langdata_tables_vector_indexes
    IS
        TYPE t_index_info IS RECORD (
            table_name     VARCHAR2(128),
            column_name    VARCHAR2(128),
            index_name     VARCHAR2(128)
        );

        TYPE t_index_list IS TABLE OF t_index_info;

        v_indexes t_index_list := t_index_list(
            t_index_info(
                'LANGDATA$REPORTDESCRIPTIONS',
                'DESCRIPTION_VECTOR',
                'LANGDATA$REPORTDESCRIPTIONS_IVF_IDX'
            ),
            t_index_info(
                'LANGDATA$DRILLDOWNDESCRIPTIONS',
                'DDD_VECTOR',
                'LANGDATA$DRILLDOWNDESCRIPTIONS_IVF_IDX'
            ),
            t_index_info(
                'LANGDATA$SEARCHRECORDS',
                'QUERY_VECTOR',
                'LANGDATA$SEARCHRECORDS_IVF_IDX'
            ),
            t_index_info(
                'LANGDATA$REPORTQUERYCLUSTER',
                'CENTROID_VECTOR',
                'LANGDATA$REPORTQUERYCLUSTER_IVF_IDX'
            ),
            t_index_info(
                'LANGDATA$DRILLDOWNQUERYCLUSTER',
                'CENTROID_VECTOR',
                'LANGDATA$DRILLDOWNQUERYCLUSTER_IVF_IDX'
            )
        );

        v_sql        VARCHAR2(4000);
        v_index_cnt  NUMBER;
        v_row_cnt    NUMBER;
    BEGIN
        FOR i IN 1 .. v_indexes.COUNT LOOP
            -- Check if vector index exists
            SELECT COUNT(*)
            INTO v_index_cnt
            FROM all_indexes
            WHERE index_name = v_indexes(i).index_name
            AND table_name = v_indexes(i).table_name;

            -- Check table row count
            v_sql := 'SELECT COUNT(*) FROM ' || v_indexes(i).table_name;
            EXECUTE IMMEDIATE v_sql INTO v_row_cnt;

            IF v_index_cnt > 0 THEN
                IF v_row_cnt <= 10000 THEN
                    -- Drop index if exists and not enough rows
                    v_sql := 'DROP INDEX ' || v_indexes(i).index_name;
                    EXECUTE IMMEDIATE v_sql;
                    lang_data_logger_pkg.log_info(
                        'Dropped index ' || v_indexes(i).index_name ||
                        ' (row count = ' || v_row_cnt || ')'
                    );
                END IF;
            ELSE
                IF v_row_cnt > 10000 THEN
                    -- Create index if needed and enough rows
                    v_sql := 'CREATE VECTOR INDEX ' ||
                            v_indexes(i).index_name ||
                            ' ON ' || v_indexes(i).table_name ||
                            ' (' || v_indexes(i).column_name || ') ' ||
                            'ORGANIZATION NEIGHBOR PARTITIONS DISTANCE ' ||
                            'COSINE WITH TARGET ACCURACY 95';
                    EXECUTE IMMEDIATE v_sql;
                    lang_data_logger_pkg.log_info(
                        'Created vector index ' || v_indexes(i).index_name ||
                        ' (row count = ' || v_row_cnt || ')'
                    );
                END IF;
            END IF;
        END LOOP;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_info(
                'Error in manage_vector_indexes: ' || SQLERRM
            );
            RAISE;
    END manage_langdata_tables_vector_indexes;

    PROCEDURE background_fixup
    IS
    BEGIN
        lang_data_utils_pkg.sync_user_tables_changed_values;
        lang_data_utils_pkg.manage_langdata_tables_vector_indexes;
        lang_data_utils_pkg.check_annotation_changes_for_user_tables;
    END background_fixup;


    FUNCTION get_text_by_id(
        p_description_id IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_description_text VARCHAR2(4000);
    BEGIN
        BEGIN
            SELECT text 
            INTO v_description_text 
            FROM langdata$reportdescriptions 
            WHERE id = p_description_id;
            RETURN v_description_text;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                NULL;
        END;

        BEGIN
            SELECT text 
            INTO v_description_text 
            FROM langdata$drilldowndescriptions 
            WHERE id = p_description_id;
            RETURN v_description_text;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                NULL;
        END;

        BEGIN
            SELECT query_text 
            INTO v_description_text 
            FROM langdata$samplequeries 
            WHERE id = p_description_id;
            RETURN v_description_text;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                NULL;
        END;

        BEGIN
            SELECT query_text 
            INTO v_description_text 
            FROM langdata$searchrecords 
            WHERE id = p_description_id;
            RETURN v_description_text;
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                NULL;
        END;

        -- If all fail
        lang_data_logger_pkg.log_error(
            'No report description found for id: ' || p_description_id
        );
        lang_data_errors_pkg.raise_error(
            lang_data_errors_pkg.c_resource_not_found
        );

        RETURN NULL; -- Just to satisfy compiler; this line won't be reached
    END get_text_by_id;

    FUNCTION update_domain(
        p_domain_id IN VARCHAR2,
        p_new_name IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_count NUMBER;
    BEGIN
            -- Check if domain exists
            SELECT COUNT(*) INTO v_count
            FROM langdata$domains
            WHERE domain_id = p_domain_id;

            IF v_count = 0 THEN
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_resource_not_found
                );
            END IF;

            -- Check if new name already exists
            SELECT COUNT(*) INTO v_count
            FROM langdata$domains
            WHERE LOWER(name) = LOWER(p_new_name)
            AND domain_id != p_domain_id;

            IF v_count > 0 THEN
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_update
                );
            END IF;

            UPDATE langdata$domains
            SET name = LOWER(p_new_name)
            WHERE domain_id = p_domain_id;

            lang_data_logger_pkg.log_info('Successfully updated domain ' 
                                    || p_domain_id || ' to ' || p_new_name);

            RETURN TRUE;
        END update_domain;

    FUNCTION delete_domain(
        p_domain_id IN VARCHAR2
    ) RETURN BOOLEAN IS
        v_count NUMBER;
    BEGIN
            -- Check if domain exists
            SELECT COUNT(*) INTO v_count
            FROM langdata$domains
            WHERE domain_id = p_domain_id;

            IF v_count = 0 THEN
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_update
                );
            END IF;

            -- Check if referenced in reports
            SELECT COUNT(*) INTO v_count
            FROM langdata$reports
            WHERE domain_id = p_domain_id;

            IF v_count > 0 THEN
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_update
                );
            END IF;

            -- Check if referenced in drilldowns
            SELECT COUNT(*) INTO v_count
            FROM langdata$drilldowndocuments
            WHERE domain_id = p_domain_id;

            IF v_count > 0 THEN
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_update
                );
            END IF;

            DELETE FROM langdata$domains
            WHERE domain_id = p_domain_id;

            lang_data_logger_pkg.log_info('Successfully deleted domain ' 
                                                                || p_domain_id);

            RETURN TRUE;
        END delete_domain;

    FUNCTION get_all_domains RETURN SYS_REFCURSOR IS
        v_cursor SYS_REFCURSOR;
    BEGIN
        

        OPEN v_cursor FOR
            SELECT domain_id, name
            FROM langdata$domains
            ORDER BY name;

        lang_data_logger_pkg.log_info('Fetching all domains');
        
        RETURN v_cursor;
    END get_all_domains;

    FUNCTION normalize_schema_name (
        p_schema_name IN VARCHAR2
    ) RETURN VARCHAR2 IS
    BEGIN
        IF p_schema_name LIKE '"%"' THEN
            RETURN p_schema_name;
        ELSE
            RETURN UPPER(p_schema_name);
        END IF;
    END normalize_schema_name;

    FUNCTION validate_vector(
        p_vector IN VECTOR
    ) RETURN BOOLEAN IS
        v_dims NUMBER;
    BEGIN
        v_dims := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_MODEL_VECTOR_DIMS'
        );
        IF v_dims <> VECTOR_DIMS(p_vector) THEN
            lang_data_logger_pkg.log_debug(
                'Invalid vector dimensions: ' || VECTOR_DIMS(p_vector) ||
                ', expected: ' || v_dims
            );
            RETURN  FALSE;
        END IF;
        RETURN TRUE;
    END validate_vector;
END lang_data_utils_pkg;
/
