Rem
Rem $Header: dbgendev/src/langdata/plsql/samplequeries/samplequeries_pkg.pks /main/9 2025/07/24 11:33:10 arevathi Exp $
Rem
Rem samplequeries_pkg.pks
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      samplequeries_pkg.pks - Package specification for managing and
Rem                              processing sample queries
Rem
Rem    DESCRIPTION
Rem      This package provides procedures to manage sample queries, including 
Rem      retrieving all sample queries, adding new queries with deduplication, 
Rem      and deleting queries. It supports role-based access control and 
Rem      includes error handling and logging mechanisms.
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/samplequeries/samplequeries_pkg.pks
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    07/07/25 - DBAI-871: Add function
Rem                           get_sample_query_id_by_report_id_version
Rem    deveverm    05/16/25 - DBAI-761: added update_sample_query_status and
Rem                           updated
Rem                           add_report_sample_query/add_drilldown_sample_query
Rem                           to create regression job
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    12/09/24 - JIRA_DBAI-458: Split get_all_sample_queries
Rem    jiangnhu    11/26/24 - Fix error handling of delete_sample_query,
Rem                           add_sample_queries
Rem    jiangnhu    10/22/24 - Format code
Rem    jiangnhu    10/18/24 - Fix text wrapping
Rem    jiangnhu    10/17/24 - Modify header
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE lang_data_sample_queries_pkg IS
    /*
    ----------------------------------------------------------------------
    Package Name: lang_data_sample_queries_pkg
    Description: This package contains procedures and functions related 
                 to managing sample queries in the langdata$samplequeries table.
                 It includes retrieving, deleting, and adding deduplicated
                 sample queries.
    ----------------------------------------------------------------------
    Author: Jiangnan Huang
    Created: 09/30/2024
    ----------------------------------------------------------------------
    Public Procedures:
    ----------------------------------------------------------------------
    
    PROCEDURE get_report_sample_queries
    -----------------------------------
    Description:
        This procedure retrieves all sample queries from the 
        langdata$samplequeries table based on the provided report ID.

        The procedure validates the provided report ID to ensure it exists
        in the system. If the report ID is invalid, it raises an error.
        The procedure supports role-based access control, requiring the user 
        to have the appropriate role to execute it.

    Parameters:
        - p_report_id (IN VARCHAR2):
            The report ID used to filter the sample queries.

        - p_sample_queries (OUT SYS_REFCURSOR):
            A cursor that returns the list of sample queries for the given
            report ID.

    Exceptions:
        - c_invalid_parameters_code (-20002):
            Raised if the provided report ID does not exist in the system. A
            custom error with code -20002 is raised using the 
            `lang_data_errors_pkg` package with the message "Invalid parameters"
        
        - c_unauthorized_code (-20003):
            Raised if the user does not have the required role. A custom error
            with code -20003 is raised using the `lang_data_errors_pkg` package
            with the message "Unauthorized access".

    Notes:
        - This procedure assumes that the report ID is valid and exists in the
          database.
        - It orders the results by the `created_at` timestamp in descending
          order.

    PROCEDURE get_drilldown_sample_queries
    ---------------------------------------
    Description:
        This procedure retrieves all sample queries from the 
        langdata$samplequeries table based on the provided drilldown ID.

        The procedure validates the provided drilldown ID to ensure it exists
        in the system. If the drilldown ID is invalid, it raises an error.
        The procedure supports role-based access control, requiring the user 
        to have the appropriate role to execute it.

    Parameters:
        - p_drilldown_id (IN VARCHAR2):
            The drilldown ID used to filter the sample queries.

        - p_sample_queries (OUT SYS_REFCURSOR):
            A cursor that returns the list of sample queries for the given
            drilldown ID.

    Exceptions:
        - c_unauthorized_code (-20002):
            Raised when the user does not have the required role 
            (LANG_DATA_APP_EXPERT) to access the procedure. This triggers a 
            custom error with code -20002 using the lang_data_errors_pkg 
            package.
        - c_invalid_parameters_code (-20002):
            Raised if the provided drilldown ID does not exist in the system. A
            custom error with code -20002 is raised using the 
            `lang_data_errors_pkg` package with the message "Invalid parameters"
        
        - c_unauthorized_code (-20003):
            Raised if the user does not have the required role. A custom error
            with code -20003 is raised using the `lang_data_errors_pkg` package
            with the message "Unauthorized access".

    Notes:
        - This procedure assumes that the drilldown ID is valid and exists in 
          the database.
        - It orders the results by the `created_at` timestamp in descending
          order.
    
    PROCEDURE delete_sample_query
    -------------------------------------
    Description:
        The procedure deletes a sample query based on the given sample query ID. 
        The procedure implements role-based access control:
        - If the user does not have the `LANG_DATA_APP_EXPERT` role, deletion
          is not allowed.
        - If the user has the `LANG_DATA_APP_EXPERT` role, the sample query is
          deleted without restriction.

    Parameters:
        - p_id (IN VARCHAR2): 
            The ID of the sample query that needs to be deleted.
        
    Exceptions:
        - c_unauthorized_code (-20002):
            Raised when the user does not have the required role 
            (LANG_DATA_APP_EXPERT) to access the procedure. This triggers a 
            custom error with code -20002 using the lang_data_errors_pkg 
            package.
        
        - c_resource_not_found (-20004): 
            Raised if no sample query is found for the given ID. A custom error
            with code -20004 is raised using the `lang_data_errors_pkg` package
            with the message "Resource not found".
    
    PROCEDURE add_report_sample_queries
    -------------------------------------
    Description:
        This procedure adds a list of sample queries for a given report. 
        The procedure performs the following steps:
        - Deduplicates the list of sample queries to ensure only unique queries 
          are processed.
        - Augments and modifies the query text based on the report's match 
          document.
        - Generates an embedding vector for each query using augmented query.
        - Inserts the processed queries into the `langdata$samplequeries` table.

        If a duplicate query (based on its MD5 hash) is found in the database, 
        it is skipped during insertion.

    Parameters:
        - p_report_id (IN VARCHAR2):
            The ID of the report associated with the sample queries.
        - p_sample_queries (IN SYS.ODCIVARCHAR2LIST):
            A list of sample queries to be inserted into the database.

    Exceptions:
        - c_unauthorized_code (-20002):
            Raised when the user does not have the required role 
            (LANG_DATA_APP_EXPERT). This triggers a custom error with code 
            -20002 using the `lang_data_errors_pkg` package.

    Notes:
        - The procedure ensures that only unique queries are inserted into the 
          database, skipping duplicates based on the MD5 hash.
        - The embedding vector is generated using the augmented query text.
        - The procedure logs the number of successfully inserted queries.

    PROCEDURE add_drilldown_sample_queries
    ----------------------------------------
    Description:
        This procedure adds a list of sample queries for a given drilldown. 
        The procedure performs the following steps:
        - Deduplicates the list of sample queries to ensure only unique queries 
          are processed.
        - Augments and modifies the query text based on the drilldown's match 
          document.
        - Associates the queries with the parent report of the drilldown.
        - Generates an embedding vector for each query using augmented query.
        - Inserts the processed queries into the `langdata$samplequeries` table.

        If a duplicate query (based on its MD5 hash) is found in the database, 
        it is skipped during insertion.

    Parameters:
        - p_drilldown_id (IN VARCHAR2):
            The ID of the drilldown associated with the sample queries.
        - p_sample_queries (IN SYS.ODCIVARCHAR2LIST):
            A list of sample queries to be inserted into the database.

    Exceptions:
        - c_unauthorized_code (-20002):
            Raised when the user does not have the required role 
            (LANG_DATA_APP_EXPERT). This triggers a custom error with code 
            -20002 using the `lang_data_errors_pkg` package.

    Notes:
        - The procedure ensures that only unique queries are inserted into the 
          database, skipping duplicates based on the MD5 hash.
        - The embedding vector is generated using the augmented query text.
        - The parent report ID is fetched from the drilldown document and used 
          in the insertion process.
        - The procedure logs the number of successfully inserted queries.
    
    FUNCTION deduplicate_sample_queries
    -----------------------------------
    Description:
        This function removes duplicate sample queries from a list of queries, 
        normalizing each query (by converting to lowercase and trimming spaces) 
        and returning a list of unique queries.
    
    Parameters:
        - p_sample_queries (IN SYS.ODCIVARCHAR2LIST):
            The list of sample queries to be deduplicated.
    
    Return:
        - SYS.ODCIVARCHAR2LIST:
            A deduplicated list of queries, with duplicates removed.
    
    Exceptions:
        - None.
    
    FUNCTION update_sample_query_status
    -----------------------------------
    Description:
        This function changes the status of Sample Query. Only those Sample Query which are in 'Published' status will be considered for search.
    
    Parameters:
        - p_id (IN VARCHAR2):
            ID of the Sample Query whose status is being updated.

        - p_status (IN VARCHAR2):
            New Status for the sample query
    
        - p_response (OUT VARCHAR2):
            Updated Status of the sample query.
    
    Exceptions:
        - None.
    
    ----------------------------------------------------------------------
    FUNCTION get_sample_query_id_by_report_id_version
    ----------------------------------------------------------------------
    Description:
        This function retrieves the unique sample query ID associated with a 
        given report ID and version.

    Parameters:
        - p_report_id (IN VARCHAR2):
            The ID of the report to which the sample query belongs.

        - p_version (IN NUMBER):
            The version number of the sample query to retrieve.

    Returns:
        - VARCHAR2:
            The unique ID of the sample query that matches the given 
            report ID and version.

    Exceptions:
        - c_resource_not_found (-20004):
            Raised when no sample query is found for the given report ID and 
            version. A custom error with code -20004 is raised using 
            lang_data_errors_pkg, with the message "Resource not found".
    */

    PROCEDURE get_report_sample_queries (
        p_report_id IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    );

    PROCEDURE get_drilldown_sample_queries (
        p_drilldown_id IN VARCHAR2,
        p_sample_queries OUT SYS_REFCURSOR
    );

    -- Public procedure declaration for deleting sample query
    PROCEDURE delete_sample_query (
        p_id IN VARCHAR2
    );

    PROCEDURE add_report_sample_queries(
        p_report_id     IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST,
        p_ids OUT SYS.ODCIVARCHAR2LIST
    );

    PROCEDURE add_drilldown_sample_queries(
        p_drilldown_id  IN VARCHAR2,
        p_sample_queries IN SYS.ODCIVARCHAR2LIST,
        p_ids OUT SYS.ODCIVARCHAR2LIST
    );

    -- Public function for deduplicating sample queries
    FUNCTION deduplicate_sample_queries (
        p_sample_queries IN SYS.ODCIVARCHAR2LIST
    ) RETURN SYS.ODCIVARCHAR2LIST;

    PROCEDURE update_sample_query_status(
        p_id IN VARCHAR2,
        p_status IN VARCHAR2,
        p_response OUT VARCHAR2
    );

    FUNCTION get_sample_query_id_by_report_id_version(
        p_report_id    IN VARCHAR2,
        p_version      IN NUMBER
    ) RETURN VARCHAR2;

END lang_data_sample_queries_pkg;
/

