Rem $Header: dbgendev/src/langdata/plsql/logging/logging_pkg.pkb /main/4 2025/07/11 20:20:53 ruohli Exp $
Rem
Rem logging_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      logging_pkg.pkb - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      <short description of component this file declares/defines>
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/logging/logging_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    ruohli      07/01/25 - DBAI-957 Fix issue on buffer too small
Rem    lachoud     05/23/25 - Added logging functinality in files
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data_logger_pkg IS

  -- global log level variable, initialized to NULL
  g_log_level NUMBER := NULL;

  -- Retrieves the log level from the configuration table
  FUNCTION get_log_level RETURN PLS_INTEGER IS
  BEGIN
      IF g_log_level IS NULL THEN
          BEGIN
              SELECT variable_value INTO g_log_level
              FROM langdata$config
              WHERE variable_name =  C_LOG_LEVEL;
          EXCEPTION
              WHEN NO_DATA_FOUND THEN
                  g_log_level := 5; -- fallback default
              WHEN OTHERS THEN
                  RAISE_APPLICATION_ERROR(
                      -20002,
                      'Unexpected error in config lookup: LOG_LEVEL ' || SQLERRM
                  );
          END;
      END IF;

      RETURN g_log_level;
  END;

  -- Retrieves the log level from the configuration table
  PROCEDURE refresh_log_level IS
  BEGIN
      g_log_level := NULL;
  END;



  -- Parses the call stack and gets the calling method
  FUNCTION parse_call_stack RETURN VARCHAR2 IS
    v_calling_stack VARCHAR2(4000);
  BEGIN
    v_calling_stack :=
      REPLACE(DBMS_UTILITY.FORMAT_CALL_STACK, CHR(10), ' | ');
    RETURN REGEXP_SUBSTR(
      v_calling_stack,
      '[A-Z0-9_]+_PKG\.[A-Z0-9_]+',
      1,
      5
    );
  END;

  -- Prepares the log string with timestamp, log level, and calling method
  FUNCTION prepare_log_string(
    p_level VARCHAR2,
    p_msg   VARCHAR2
  ) RETURN VARCHAR2 IS
    v_timestamp      VARCHAR2(50) := TO_CHAR(
      SYSTIMESTAMP,
      'YYYY-MM-DD HH24:MI:SS.FF3'
    );
    v_calling_method VARCHAR2(100) := parse_call_stack();
  BEGIN
    RETURN RPAD('[LANGDATA:' || v_timestamp || ']', 35) ||
           RPAD('[' || UPPER(p_level) || ']', 7) ||
           '[' || v_calling_method || '] : ' || p_msg;
  END;

  -- Logs the message to DBMS_USERDIAG
  PROCEDURE log(
    p_level VARCHAR2,
    p_msg   VARCHAR2,
    p_alert NUMBER
  ) IS
    v_full_msg VARCHAR2(32767);
  BEGIN
    v_full_msg := prepare_log_string(p_level, p_msg);
    DBMS_USERDIAG.TRACE(v_full_msg, p_alert);
    -- log errors into trace files as well
    IF p_level = 'ERROR' OR p_level = 'FATAL' THEN
      DBMS_USERDIAG.TRACE(v_full_msg, 0);
    END IF;
    dbms_output.put_line(p_level || ': '|| p_msg);
  END log;

  PROCEDURE log_debug(p_message VARCHAR2) IS
  BEGIN
    IF get_log_level <=  C_LOG_LEVEL_DEBUG THEN
      log('DEBUG', p_message, 0);
    END IF;
  END log_debug;

  PROCEDURE log_trace(p_message VARCHAR2) IS
  BEGIN
    IF get_log_level <=  C_LOG_LEVEL_TRACE THEN
      log('TRACE', p_message, 0);
    END IF;
  END log_trace;

  PROCEDURE log_info(p_message VARCHAR2) IS
  BEGIN
    IF get_log_level <=  C_LOG_LEVEL_INFO THEN
      log('INFO', p_message, 0);
    END IF;
  END log_info;

  PROCEDURE log_warn(p_message VARCHAR2) IS
  BEGIN
    IF get_log_level <=  C_LOG_LEVEL_WARN THEN
      log('WARN', p_message, 0);
    END IF;
  END log_warn;

  PROCEDURE log_fatal(p_message VARCHAR2) IS
  BEGIN
    IF get_log_level <=  C_LOG_LEVEL_FATAL THEN
      log('FATAL', p_message, 1); -- Log to alert file
    END IF;
  END log_fatal;

  PROCEDURE log_error(p_message VARCHAR2) IS
  BEGIN
    IF get_log_level <=  C_LOG_LEVEL_ERROR THEN
      log('ERROR', p_message, 1);
    END IF;
  END log_error;

END lang_data_logger_pkg;

/

CREATE OR REPLACE PACKAGE BODY ld_logger IS
    PROCEDURE debug(p_message VARCHAR2) IS
    BEGIN
        lang_data_logger_pkg.log_debug(p_message);
    END debug;

    PROCEDURE info(p_message VARCHAR2) IS
    BEGIN
        lang_data_logger_pkg.log_info(p_message);
    END info;

    PROCEDURE warn(p_message VARCHAR2) IS
    BEGIN
        lang_data_logger_pkg.log_warn(p_message);
    END warn;

    PROCEDURE trace(p_message VARCHAR2) IS
    BEGIN
        lang_data_logger_pkg.log_trace(p_message);
    END trace;

    PROCEDURE error(p_message VARCHAR2) IS
    BEGIN
        lang_data_logger_pkg.log_error(p_message);
    END error;

    PROCEDURE fatal(p_message VARCHAR2) IS
    BEGIN
        lang_data_logger_pkg.log_fatal(p_message);
    END fatal;
END ld_logger;
/