Rem
Rem $Header: dbgendev/src/langdata/plsql/feedback/feedback_pkg.pkb 
Rem saloshah_jira-dbai-434/1 2024/11/26 13:44:39 saloshah Exp $
Rem
Rem feedback_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      feedback_pkg.pkb - Package body for feedback packages
Rem
Rem    DESCRIPTION
Rem      This package contains implementaion of procedures for managing feedback
Rem      for user searh records.
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/feedback_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    jiangnhu    07/02/25 - DBAI-1005: Update expected report ID to parent 
Rem                           report ID in update_expected_drilldown_id
Rem    saloshah    06/26/25 - DBAI-873: Added delete feedback API
Rem    arevathi    03/20/25 - Add APIs for undo accept report
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    saloshah    11/29/24 - DBAI-434: Fixed the feedback package to handles
Rem                           the edge cases
Rem    arevathi    11/12/24 - Fixed errors, removed unnecessary procedures
Rem    anisbans    11/04/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data_feedback_pkg IS
    
    PROCEDURE update_required_feedback_action (
        p_id                       IN VARCHAR2,
        p_required_feedback_action IN VARCHAR2,
        p_error_message            IN VARCHAR2 DEFAULT 'Search record not '|| 
                                                'found for the specified ID'
    ) IS
        v_user_role           VARCHAR2(100);
    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_id) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        IF p_required_feedback_action NOT IN ('New Report',
                                              'Update Report',
                                              'Pending Review',
                                              'None') THEN

          lang_data_logger_pkg.log_error('p_required_feedback_action should be'
          || ' one of New Report, Update Report, Pending Review, None');

          lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
          );

        END IF;

        lang_data_utils_pkg.update_table_column(
            p_table_name          => 'LANGDATA$SEARCHRECORDS',     
            p_column_name         => 'required_feedback_action',   
            p_id                  => p_id,                         
            p_value               => p_required_feedback_action,  
            p_error_message       => 'Search record not found'             
        );

        -- Check if any rows were updated
        IF SQL%ROWCOUNT = 0 THEN

            lang_data_logger_pkg.log_error(p_error_message);
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );

        END IF;

        lang_data_logger_pkg.log_info(
            'Successfully updated required_feedback_action for ID: ' || p_id
        );

    EXCEPTION
        WHEN OTHERS THEN

            IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;

            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in update_required_feedback_action.'
                || ' Error: ' || SQLERRM
            );
            RAISE;
    END update_required_feedback_action;

    PROCEDURE update_feedback_action_priority (
        p_id                  IN VARCHAR2,
        p_feedback_action_priority IN VARCHAR2,
        p_error_message       IN VARCHAR2 DEFAULT 'Search record not found'|| 
                                                'for the specified ID'
    ) IS
        v_user_role VARCHAR2(100);
    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_id) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        IF p_feedback_action_priority NOT IN ('low',
                                              'medium',
                                              'high') THEN

          lang_data_logger_pkg.log_error('p_feedback_action_priority should be'
          || ' one of low, medium, high');

          lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
          );

        END IF;

        lang_data_utils_pkg.update_table_column(
            p_table_name          => 'LANGDATA$SEARCHRECORDS',     
            p_column_name         => 'feedback_action_priority',   
            p_id                  => p_id,                         
            p_value               => p_feedback_action_priority,  
            p_error_message       => p_error_message             
        );

        -- Check if any rows were updated
        IF SQL%ROWCOUNT = 0 THEN

            lang_data_logger_pkg.log_error(p_error_message);

            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        lang_data_logger_pkg.log_info(
            'Successfully updated feedback_action_priority for ID: ' || p_id
        );
    EXCEPTION
        WHEN OTHERS THEN

           IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;

            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in update_feedback_action_priority.'
                || ' Error: ' || SQLERRM
            );
            RAISE;
    END update_feedback_action_priority;

    PROCEDURE update_expected_report_id (
        p_search_id          IN VARCHAR2,
        p_expected_report_id IN VARCHAR2
    ) IS
        v_current_expected_report_id VARCHAR2(36);
    BEGIN

        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_search_id) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        -- Verify if the report exists
        IF p_expected_report_id IS NOT NULL THEN
            IF NOT lang_data_utils_pkg.check_report_exists(
                p_expected_report_id
            ) THEN
                lang_data_logger_pkg.log_error( 'No report exists with the id'
                || p_expected_report_id);

                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_parameters_code
                );

            END IF;
        END IF;

        -- Fetch the current expected_report_id
        SELECT expected_report_id
        INTO v_current_expected_report_id
        FROM LANGDATA$SEARCHRECORDS
        WHERE id = p_search_id;

        -- If the current expected_report_id equals the 
        -- new expected_report_id, 
        -- throw an error
        IF (v_current_expected_report_id IS NULL AND
            p_expected_report_id IS NULL)
            OR (v_current_expected_report_id = p_expected_report_id) THEN

            lang_data_logger_pkg.log_warn( 'The new expected_report_id is the'
             || ' same as the current expected_report_id');

            RETURN;

        END IF;

        lang_data_utils_pkg.update_table_column(
            p_table_name          => 'LANGDATA$SEARCHRECORDS',     
            p_column_name         => 'expected_report_id',   
            p_id                  => p_search_id,                         
            p_value               => p_expected_report_id,    
            p_error_message       => 'Search record not found' || 
                                     'for the specified ID'     
        );

        lang_data_logger_pkg.log_info(
            'Successfully updated expected_report_id for ID: ' || p_search_id
        );

    EXCEPTION
        WHEN OTHERS THEN
        
            IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;

            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in update_expected_report_id.  ' ||
                'Error: ' || SQLERRM
            );
            RAISE;

    END update_expected_report_id;

    PROCEDURE update_expected_drilldown_id (
        p_search_id         IN VARCHAR2,
        p_expected_drilldown_id IN VARCHAR2
    ) IS
        v_current_expected_drilldown_id VARCHAR2(36);
        v_current_expected_report_id    VARCHAR2(36);
        v_expected_parent_report_id     VARCHAR2(36);
    BEGIN
        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_search_id) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        -- Verify if the drilldown exists
        IF p_expected_drilldown_id IS NOT NULL THEN
            IF NOT lang_data_utils_pkg.check_drilldown_exists(
                p_expected_drilldown_id
            ) THEN
                lang_data_logger_pkg.log_error(
                    'No drilldown exists with the id ' ||
                    p_expected_drilldown_id
                );

                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_parameters_code
                );

            END IF;
        END IF;
        
        -- Fetch parent report ID
        SELECT report_id INTO v_expected_parent_report_id
        FROM langdata$drilldowndocuments
        WHERE id = p_expected_drilldown_id;

        -- Fetch the current expected_drilldown_id
        SELECT expected_report_id, expected_drilldown_id
        INTO v_current_expected_report_id, v_current_expected_drilldown_id
        FROM LANGDATA$SEARCHRECORDS
        WHERE id = p_search_id;

        -- If the current expected_drilldown_id equals the new
        -- expected_drilldown_id, throw an error
        IF (v_current_expected_drilldown_id IS NULL AND
            p_expected_drilldown_id IS NULL)
            OR (v_current_expected_drilldown_id = p_expected_drilldown_id) THEN

            lang_data_logger_pkg.log_warn( 'The new expected_drilldown_id is'
             || ' the same as the current expected_drilldown_id');

            RETURN;
        END IF;

        lang_data_utils_pkg.update_table_column(
            p_table_name          => 'LANGDATA$SEARCHRECORDS',     
            p_column_name         => 'expected_drilldown_id',   
            p_id                  => p_search_id,                         
            p_value               => p_expected_drilldown_id,    
            p_error_message       => 'Search record not found' || 
                                     'for the specified ID'     
        );

        IF v_current_expected_report_id != v_expected_parent_report_id THEN
            lang_data_utils_pkg.update_table_column(
                p_table_name          => 'LANGDATA$SEARCHRECORDS',     
                p_column_name         => 'expected_report_id',   
                p_id                  => p_search_id,                         
                p_value               => v_expected_parent_report_id,    
                p_error_message       => 'Search record not found' || 
                                        'for the specified ID'     
            );
        END IF;

        lang_data_logger_pkg.log_info(
            'Successfully updated expected_drilldown_id for ID: ' || 
            p_search_id
        );

    EXCEPTION
        WHEN OTHERS THEN

            IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;

            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in update_expected_drilldown_id.  ' ||
                'Error: ' || SQLERRM
            );
            RAISE;
    END update_expected_drilldown_id;

    PROCEDURE update_feedback(
        p_search_id         IN VARCHAR2,
        p_feedback_comments IN VARCHAR2 DEFAULT NULL,
        p_feedback_rating   IN BOOLEAN DEFAULT NULL
    ) IS
        v_sql_text VARCHAR2(4000);
        v_search_records_table CONSTANT VARCHAR2(30) := 
        'LANGDATA$SEARCHRECORDS';
        e_invalid_rating EXCEPTION;
        e_comments_too_long EXCEPTION;
        v_first_clause BOOLEAN := TRUE;
    BEGIN
        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_search_id) THEN
            lang_data_logger_pkg.log_error('No search record exists with id '
            || p_search_id);
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        -- Validate feedback rating
        IF p_feedback_rating IS NOT NULL THEN
            IF p_feedback_rating NOT IN (TRUE,FALSE) THEN
                lang_data_logger_pkg.log_error(
                    'Feedback rating must be either "Positive" or "Negative"'
                    );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_parameters_code
                );
            END IF;
        END IF;
    
        -- Validate feedback comments length
        IF p_feedback_comments IS NOT NULL THEN
            IF LENGTH(p_feedback_comments) > 2000 THEN
                lang_data_logger_pkg.log_error('Feedback comments cannot exceed' 
                ||' 2000 characters');
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_max_text_length_exceeded
                );
            END IF;
        END IF;

        -- Prepare dynamic SQL for updating feedback
        v_sql_text := 'UPDATE ' || v_search_records_table || ' SET ';
        
        IF p_feedback_comments IS NOT NULL THEN
            v_sql_text := v_sql_text || 
            'feedback_comments = ''' || p_feedback_comments || '''';
            v_first_clause := FALSE;
        END IF;

        IF p_feedback_rating IS NOT NULL THEN
            IF NOT v_first_clause THEN
                v_sql_text := v_sql_text || ', ';
            END IF;
            v_sql_text := v_sql_text || 'feedback_rating = :feedback_rating';
            v_first_clause := FALSE;
        END IF;

        -- Deleting feedback
        IF v_first_clause THEN
            lang_data_logger_pkg.log_error(  
                'Feedback rating and comments both cannot be NULL');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        ELSE
            v_sql_text := v_sql_text || ' WHERE id = :search_id';
        END IF;
        -- Execute the SQL update
        IF p_feedback_rating IS NOT NULL THEN
            EXECUTE IMMEDIATE v_sql_text
            USING p_feedback_rating, p_search_id;
        ELSE
            EXECUTE IMMEDIATE v_sql_text
            USING p_search_id;
        END IF;

        IF SQL%ROWCOUNT = 0 THEN

            lang_data_logger_pkg.log_error(
                'No search record found with ID ' || p_search_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );

        END IF;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_max_text_length_exceeded,
                        lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || 
                p_search_id || '. Error: ' || SQLERRM
            );
            RAISE;
        
    END update_feedback;

    PROCEDURE delete_feedback (
        p_search_id        IN VARCHAR2,
        p_delete_rating    IN BOOLEAN DEFAULT FALSE,
        p_delete_comments  IN BOOLEAN DEFAULT FALSE
    ) IS
        v_sql_text  VARCHAR2(1000) := 'UPDATE langdata$searchrecords SET ';
        v_first_clause     BOOLEAN := TRUE;
    BEGIN
        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_search_id) THEN
            lang_data_logger_pkg.log_error(
                'No search record exists with id '|| p_search_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        IF NOT p_delete_rating AND NOT p_delete_comments THEN
            lang_data_logger_pkg.log_error(
                'Nothing to delete for search ID: ' || p_search_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_delete_rating THEN
            v_sql_text := v_sql_text || 'feedback_rating = NULL';
            v_first_clause := FALSE;
        END IF;

        IF p_delete_comments THEN
            IF NOT v_first_clause THEN
                v_sql_text := v_sql_text || ', ';
            END IF;
            v_sql_text := v_sql_text || 'feedback_comments = NULL';
        END IF;

        v_sql_text := v_sql_text || ' WHERE id = :search_id';

        EXECUTE IMMEDIATE v_sql_text USING p_search_id;

        lang_data_logger_pkg.log_info(
            'Feedback deleted for search ID: ' || p_search_id
        );

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_invalid_parameters_code) 
            THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || 
                p_search_id || '. Error: ' || SQLERRM
            );
            RAISE;    
    END delete_feedback;

    PROCEDURE remove_expected_report_id (
        p_search_id          IN VARCHAR2
    ) IS
    BEGIN

        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_search_id) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        lang_data_utils_pkg.update_table_column(
            p_table_name          => 'LANGDATA$SEARCHRECORDS',     
            p_column_name         => 'expected_report_id',   
            p_id                  => p_search_id,                         
            p_value               => NULL,    
            p_error_message       => 'Search record not found' || 
                                     'for the specified ID'     
        );

        lang_data_logger_pkg.log_info(
            'Successfully removed expected_report_id for ID: ' || p_search_id
        );

    EXCEPTION
        WHEN OTHERS THEN
        
            IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;

            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in update_expected_report_id.  ' ||
                'Error: ' || SQLERRM
            );
            RAISE;

    END remove_expected_report_id;

    PROCEDURE remove_expected_drilldown_id (
        p_search_id         IN VARCHAR2
    ) IS
    BEGIN
        -- Check if search record exists
        IF NOT lang_data_utils_pkg.check_search_record_exists(p_search_id) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        lang_data_utils_pkg.update_table_column(
            p_table_name          => 'LANGDATA$SEARCHRECORDS',     
            p_column_name         => 'expected_drilldown_id',   
            p_id                  => p_search_id,                         
            p_value               => NULL,    
            p_error_message       => 'Search record not found' || 
                                     'for the specified ID'     
        );

        lang_data_logger_pkg.log_info('Successfully removed ' ||
                     'expected_drilldown_id for ID: ' || p_search_id);

    EXCEPTION
        WHEN OTHERS THEN

            IF SQLCODE IN (
                        lang_data_errors_pkg.c_resource_not_found,
                        lang_data_errors_pkg.c_unauthorized_code,
                        lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;

            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in update_expected_drilldown_id.  ' 
                || 'Error: ' || SQLERRM
            );
            RAISE;
    END remove_expected_drilldown_id;

END lang_data_feedback_pkg;
/
