Rem
Rem $Header: dbgendev/src/langdata/plsql/reports/reports_pkg.pkb /main/49 2025/08/17 19:34:27 deveverm Exp $
Rem
Rem reports_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      reports_pkg.pkb - Lang Data Reports package
Rem
Rem    DESCRIPTION
Rem      This package contains implementaion of procedures for managing reports
Rem      and their descriptions.
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/reports/reports_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    deveverm    08/14/25 - Added purge_reports procedure and updated vector validation logic.
Rem    deveverm    08/12/25 - Added purge_reports procedure
Rem    dadoshi     08/04/25 - JIRA_DBAI1149: Add description vector validation
Rem                           to create_report
Rem    dadoshi     07/31/25 - JIRA_DBAI1149: Add optional parameter,
Rem                           p_description_vector to create_report
Rem    saloshah    07/30/25 - DBAI-1101: Added exceptions for non existent id
Rem                           for update_report_description
Rem    jiangnhu    07/30/25 - Remove duplicate call to validate_match_document,
Rem                           now called by lang_data.create_report
Rem    deveverm    07/30/25 - DBAI-1161: made regression analysis respect 
Rem                           domains
Rem    dadoshi     07/24/25 - JIRA_DBAI1056: Add
Rem                           get_report_match_document_by_id
Rem    dadoshi     07/22/25 - JIRA_DBAI1079: Update report description status
Rem                           validation
Rem    dadoshi     07/22/25 - JIRA_DBAI1056: Added get_report_status_by_id
Rem    deveverm    07/18/25 - DBAI-1050: added new values to regression_json
Rem    ruohli      07/14/25 - DBAI-956: Implement New report creation validation
Rem    jiangnhu    07/10/25 - DBAI-1006: Parameterize job class name
Rem    ruohli      06/30/25 - DBAI-945: Added an option to calculate the
Rem                           regression by report. Change create_report
Rem                           procedure to trigger regression calculation in
Rem                           the same session
Rem    fgurrola    06/27/25 - DBAI-776: Disable adding custom entity_types.
Rem    jiangnhu    06/25/25 - Add validation for p_report_id in
Rem                           enumerable-set-update APIs
Rem    jiangnhu    06/24/25 - DBAI-909: Integrate text augmentation with domain
Rem    jiangnhu    06/09/25 - Implement APIs to get ID by unique combination of 
Rem                           title, version, etc.
Rem    ruohli      06/06/25 - DBAI-842: Added parallel query for 
Rem                           calculate_report_description_regression and remove
Rem                           top 2 matching logic
Rem    jiangnhu    06/02/25 - DBAI-844: Remove call to
Rem                           update_annotation_and_comment_records
Rem    jiangnhu    05/29/25 - Update job name of creating enumerable-set-based
Rem                           value vector partition
Rem    deveverm    05/16/25 - DBAI-761: changed
Rem                           get_report_description_regression to
Rem                           get_report_regression, changed
Rem                           calculate_report_description_regression to
Rem                           calculate_report_regression
Rem    jiangnhu    04/01/25 - DBAI-624: Make embedding model independent
Rem    deveverm    04/01/25 - DBAI-523: added get_report_description_regression
Rem                           and modified regression logic
Rem    jiangnhu    03/26/25 - DBAI-692: make report_id, description_id OUT
Rem                           parameter, make enumeration limit a config
Rem    jiangnhu    03/24/25 - DBAI-551: Use NER to augment description
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/15/25 - DBAI-661: Implement
Rem                           replace_report_filter_enumerable_set
Rem    anisbans    03/11/25 - DBAI-556 : update ref_count for user_ctx_indexes 
Rem                                      upon report deletion
Rem    dadoshi     03/11/25 - JIRA_DBAI574: Add add_named_entities_to_document
Rem                           API usage
Rem    arevathi    03/11/25 - Add Update status API's
Rem    jiangnhu    02/28/25 - Fix calls to amend_description with amended text
Rem    dadoshi     02/24/25 - Throw errors in create_report() before augmenting,
Rem                           amending, and validating match_document
Rem    anisbans    02/19/25 - JIRA_DBAI-557: Apply ref count for value vectors  
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    02/13/25 - DBAI-524: create context index based on match
Rem                           document
Rem    jiangnhu    02/11/25 - Normalize description before calculate md5
Rem    jiangnhu    01/31/25 - DBAI-511: Update function augment_query to	
Rem                           procedure augment_text, add augmented_tokens
Rem    dadoshi     01/29/25 - JIRA_DBAI-506: Update add_report_description and
Rem                           create_report APIs to have IDs as optional
Rem                           arguments
Rem    jiangnhu    12/09/24 - JIRA_DBAI-458: Use add_report_sample_queries
Rem    dadoshi     11/13/24 - Update get_report to get_report_paginated
Rem    dadoshi     11/04/24 - Update execute_report_sql to handle large number
Rem                           of rows
Rem    arevathi    10/29/24 - Added Validation APIs for report
Rem    jiangnhu    10/29/24 - JIRA_DBAI-403: Add augment_query to process 
Rem                           description
Rem    dadoshi     10/25/24 - Add execute_report_sql
Rem    deveverm    10/25/24 - added get_report_description
Rem    deveverm    10/23/24 - added get_all_reports
Rem    deveverm    10/22/24 - added update_report_description
Rem    deveverm    10/22/24 - added delete_report_description
Rem    deveverm    10/22/24 - added add_report_description
Rem    deveverm    10/22/24 - added delete_report
Rem    deveverm    10/21/24 - fix text wrapping
Rem    deveverm    10/18/24 - DBAI-399: Modify Header
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data_reports_pkg IS
    
    PROCEDURE get_report_descriptions_paginated (
        p_report_id            IN VARCHAR2,
        p_limit         IN NUMBER DEFAULT 10,
        p_cursor        IN OUT VARCHAR2,
        p_descriptions  OUT SYS_REFCURSOR
    ) IS
        v_cur               INTEGER;
        v_cursor_created_at TIMESTAMP;
        v_cursor_id         VARCHAR2(4000);
        v_last_id           VARCHAR2(4000);
        v_last_created_at   TIMESTAMP;
        v_description_query VARCHAR2(4000) := 
            'SELECT id, text, version, status, enhanced_text ' ||
            'FROM langdata$reportdescriptions WHERE report_id = :report_id ';
        v_conditions        VARCHAR2(4000) := 
            'AND (created_at < :cursor_created_at OR '||
            '(created_at = :cursor_created_at AND id <= :cursor_id)) ';
        v_sql_text          VARCHAR2(4000);
        v_dummy             INTEGER;
    BEGIN

        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        v_cur := DBMS_SQL.OPEN_CURSOR;
        
        IF p_cursor IS NOT NULL THEN
            lang_data_utils_pkg.split_cursor(
                p_cursor => p_cursor, 
                p_created_at => v_cursor_created_at, 
                p_id => v_cursor_id
            );

            lang_data_logger_pkg.log_debug(
                'Applied pagination cursor: CreatedAt=' || 
                TO_CHAR(v_cursor_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') || 
                ', ID=' || v_cursor_id
            );
            v_sql_text := v_description_query || v_conditions;
        ELSE
            v_sql_text := v_description_query;
        END IF;

        IF p_limit IS NOT NULL THEN
            v_sql_text := v_sql_text || 
            'ORDER BY created_at DESC, id DESC FETCH FIRST :limit ROWS ONLY';
        ELSE
            v_sql_text := v_sql_text || 'ORDER BY created_at DESC, id DESC';
        END IF;
        lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);

        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);
        
        IF p_cursor IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':cursor_created_at', v_cursor_created_at
            );
            DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
        END IF;
        
        IF p_limit IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);
        END IF;

        -- Return the report descriptions as ref cursor object
        v_dummy :=  DBMS_SQL.EXECUTE(v_cur);
        p_descriptions  :=  DBMS_SQL.TO_REFCURSOR(v_cur);

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        IF p_limit IS NOT NULL THEN
            lang_data_logger_pkg.log_debug('Checking for next description row');
            
            v_sql_text := 'SELECT id, created_at '||
            'FROM langdata$reportdescriptions WHERE report_id = :report_id ';
                            
            IF p_cursor IS NOT NULL THEN
                v_sql_text := v_sql_text || v_conditions;
            END IF;

            v_sql_text := v_sql_text || 
            'ORDER BY created_at DESC, id DESC OFFSET :limit ROWS '||
            'FETCH NEXT 1 ROWS ONLY ';

            v_cur :=  DBMS_SQL.OPEN_CURSOR;
            
            lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);
            DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

            IF p_cursor IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(
                    v_cur, ':cursor_created_at', v_cursor_created_at
                );
                DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
            END IF;

            DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_report_id);
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);


            DBMS_SQL.DEFINE_COLUMN(v_cur, 1, v_last_id, 36);
            DBMS_SQL.DEFINE_COLUMN(v_cur, 2, v_last_created_at);

            v_dummy := DBMS_SQL.EXECUTE(v_cur);

            IF DBMS_SQL.FETCH_ROWS(v_cur) > 0 THEN
                DBMS_SQL.COLUMN_VALUE(v_cur, 1, v_last_id);
                DBMS_SQL.COLUMN_VALUE(v_cur, 2, v_last_created_at);

                -- Set the next cursor for pagination
                p_cursor := TO_CHAR(
                    v_last_created_at, 'YYYY-MM-DD HH24:MI:SS.FF'
                    ) || '|' || v_last_id;
                lang_data_logger_pkg.log_debug(
                    'Next description cursor set to: ' || p_cursor
                );
            ELSE
                p_cursor := NULL;
                lang_data_logger_pkg.log_debug('No further pages.');
            END IF;

            IF DBMS_SQL.IS_OPEN(v_cur) THEN
                -- Close the next cursor
                DBMS_SQL.CLOSE_CURSOR(v_cur);
            END IF;
        ELSE
            p_cursor := NULL;
            lang_data_logger_pkg.log_debug('No further pages.');
        END IF;

        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE = lang_data_errors_pkg.c_unauthorized_code THEN
                    RAISE;
                END IF;
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for ID: ' || 
                    p_report_id || '. Error: ' || SQLERRM
                );
                RAISE;
    END get_report_descriptions_paginated;

	PROCEDURE get_report_paginated (
		p_id IN VARCHAR2,
		p_title OUT VARCHAR2,
		p_match_document OUT JSON,
		p_status OUT VARCHAR2,
		p_descriptions OUT SYS_REFCURSOR,
        p_description_cur IN OUT VARCHAR2,
        p_descriptions_limit IN NUMBER DEFAULT 10,
		p_sample_queries OUT SYS_REFCURSOR,
        p_sample_query_cur IN OUT VARCHAR2,
        p_sample_query_limit IN NUMBER DEFAULT 10
	)
    IS
        v_cur               INTEGER;
        v_dummy             INTEGER;
        v_report_query      VARCHAR2(4000)  :=  
        'SELECT title, match_document, status FROM langdata$reports ' ||
                                                'WHERE id = :report_id ';
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure to get_report_paginated for ID: ' || p_id
        );
        
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        --Load query to get report details in langdata$reports table using id
        v_cur := DBMS_SQL.OPEN_CURSOR;

        -- Parse the SQL statement
        DBMS_SQL.PARSE(v_cur, v_report_query, DBMS_SQL.NATIVE);

        -- Bind the report ID to the SQL query
        DBMS_SQL.BIND_VARIABLE(v_cur, ':report_id', p_id);

        -- Define the columns to retrieve the result set into variables
        DBMS_SQL.DEFINE_COLUMN(v_cur, 1, p_title, 200);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 2, p_match_document);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 3, p_status, 50);

        -- Execute the query
        v_dummy := DBMS_SQL.EXECUTE(v_cur);

        -- Fetch the results
        IF DBMS_SQL.FETCH_ROWS(v_cur) > 0 THEN
            DBMS_SQL.COLUMN_VALUE(v_cur, 1, p_title);
            DBMS_SQL.COLUMN_VALUE(v_cur, 2, p_match_document);
            DBMS_SQL.COLUMN_VALUE(v_cur, 3, p_status);
        ELSE
            -- Raise an error if no data is found
            lang_data_logger_pkg.log_error('No report found for ID: '|| p_id);
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        -- Close the cursor
        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        --Load the query to fetch report descriptions
        lang_data_reports_pkg.get_report_descriptions_paginated (
            p_report_id         => p_id,
            p_limit             => p_descriptions_limit,
            p_cursor            => p_description_cur,
            p_descriptions      => p_descriptions
        );
        --Load the query to fetch sample queries
        lang_data_utils_pkg.get_sample_queries_paginated(
            p_report_id         =>  p_id,
            p_drilldown_id      =>  NULL,
            p_limit             =>  p_sample_query_limit,
            p_cursor            =>  p_sample_query_cur,
            p_sample_queries    =>  p_sample_queries
        );

        lang_data_logger_pkg.log_info(
            'Successfully retrieved report for ID: '|| p_id
        );

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error('No report found for ID: '|| p_id);
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        WHEN OTHERS THEN
            IF SQLCODE IN (
                            lang_data_errors_pkg.c_resource_not_found,
                            lang_data_errors_pkg.c_unauthorized_code ) THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || 
                p_id || '. Error: ' || SQLERRM
            );
            RAISE;
    END get_report_paginated;

    PROCEDURE create_report (
        p_title                 IN VARCHAR2,
        p_match_document        IN JSON,
        p_description_text      IN VARCHAR2,
        p_description_status    IN VARCHAR2,
        p_report_status         IN VARCHAR2,
        p_sample_queries        IN SYS.ODCIVARCHAR2LIST,
        p_description_vector    IN VECTOR DEFAULT NULL,
        p_domain                IN VARCHAR2 DEFAULT NULL,
        p_new_report_validation IN BOOLEAN DEFAULT FALSE,
        p_report_id             OUT VARCHAR2,
        p_description_id        OUT VARCHAR2
    )
    IS 
        v_ner_augmented_desc        VARCHAR2(4000);
        v_enhanced_description      VARCHAR2(4000) := '';
        v_description_md5           VARCHAR2(32); -- MD5 hash of normalized query
        v_description_vector        VECTOR;
        v_desc_vector_dim           NUMBER;
        v_domain_vector_dimensions  NUMBER;
        v_cur                       INTEGER;
        v_dummy                     INTEGER;
        v_sql_text                  VARCHAR2(2000);
        v_augmented_tokens          JSON;
        v_report_id                 VARCHAR2(36);
        v_description_id            VARCHAR2(36);
        v_domain_id                 VARCHAR2(36);
        v_normalized_text           VARCHAR2(4000);
        v_ner_augmentation_enabled  BOOLEAN;
        v_entities                  JSON;
        v_sample_query_ids          SYS.ODCIVARCHAR2LIST;
        v_report_validation         JSON;
        v_analytics_data            JSON_OBJECT_T;
        v_report_cursor             SYS_REFCURSOR;
        v_match_document_arr        JSON_ARRAY_T := JSON_ARRAY_T();
        TYPE ranked_report_result_rec IS RECORD(
            report_id              VARCHAR2(36),
            description_id         VARCHAR2(36),
            vector_distance        NUMBER,
            source                 VARCHAR2(100),
            similarity_search_rank NUMBER,
            cross_encoder_rank     NUMBER,
            overall_rank           NUMBER,
            match_type             VARCHAR2(100)
        );
        v_report_rec                ranked_report_result_rec;
    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;
        
        lang_data_logger_pkg.log_info(
            'Starting create_report procedure: '|| p_title
        );

        -- Validate report_status is among required values
        IF p_report_status NOT IN ( 'Pending Review',
                                    'Approved',
                                    'Rejected',
                                    'Published',
                                    'Inactive',
                                    'Archived' ) THEN
            lang_data_logger_pkg.log_error('Invalid p_report_status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        EXECUTE IMMEDIATE 
        'SELECT COUNT(1) FROM langdata$reports 
        WHERE title = :title' 
        INTO v_dummy USING p_title;
        IF v_dummy > 0 THEN
            lang_data_logger_pkg.log_error(
                'Report with the same title already exists.'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_already_exists
            );
        END IF;

        v_report_id := lang_data_utils_pkg.generate_id();
        v_description_id := lang_data_utils_pkg.generate_id();

        IF LENGTH(p_description_text) > 2000 THEN
            lang_data_logger_pkg.log_error(
                'Description text exceeds max text length'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_max_text_length_exceeded
            );
        END IF;

        -- Validate title is not NULL
        IF  p_title IS NULL THEN
            lang_data_logger_pkg.log_error(
                'Report title is mandatory'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Validate title length is less than the specified limit
        IF  LENGTH(p_title) > 255 THEN
            lang_data_logger_pkg.log_error(
                'Report title exceeds max title length'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_max_text_length_exceeded
            );
        END IF;

        -- Validate p_description_status is not null and among required values
        IF p_description_status NOT IN ('Pending Review',
                                        'Approved',
                                        'Rejected',
                                        'Published',
                                        'Inactive',
                                        'Archived',
                                        'Pending Regression') THEN
            lang_data_logger_pkg.log_error('Invalid p_description_status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Create or get existing domain id
        IF p_domain IS NOT NULL THEN
            v_domain_id := lang_data_utils_pkg.get_or_create_domain(
                p_domain_name => p_domain
            );
        END IF;

        IF p_description_vector IS NOT NULL 
            AND NOT 
                lang_data_utils_pkg.validate_vector(p_description_vector) THEN
            lang_data_logger_pkg.log_error(
                'Invalid dimensions of input vector'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        v_ner_augmented_desc := p_description_text;

        IF v_ner_augmentation_enabled THEN
            v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                            p_description_text
                        );
            IF v_entities IS NOT NULL THEN
                v_ner_augmented_desc := 
                    lang_data_utils_pkg.augment_text_with_ner_entities(
                        p_description_text, v_entities
                    );
            END IF;
        END IF;

        IF p_description_vector IS NULL THEN
            -- Initial augmentation of description
            lang_data_utils_pkg.augment_text(
                p_text             =>  v_ner_augmented_desc,
                p_domain_id        =>  v_domain_id,
                p_augmented_text   =>  v_enhanced_description,
                p_augmented_tokens =>  v_augmented_tokens
            );
            
            -- Expand Description to include filter descriptions
            v_enhanced_description   :=  lang_data_utils_pkg.expand_text(
                p_match_document => p_match_document,
                p_original_text  => v_enhanced_description,
                p_text_type      => 'report'
            );
        ELSE
            v_enhanced_description := p_description_text;
        END IF;
        
        v_normalized_text := LOWER(TRIM(p_description_text));
        -- Generate MD5 for the report description
        v_description_md5 := RAWTOHEX(
            DBMS_CRYPTO.HASH(
                    UTL_RAW.CAST_TO_RAW(v_normalized_text), 
                    DBMS_CRYPTO.HASH_MD5
                )
            );

        -- Check if description already exists
        EXECUTE IMMEDIATE 
            'SELECT COUNT(1) FROM langdata$reportdescriptions 
            WHERE description_md5 = :md5' 
        INTO v_dummy USING v_description_md5;

        IF v_dummy > 0 THEN
            lang_data_logger_pkg.log_error('Report description already exists');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_already_exists
            );
        END IF;

        -- Use new report description as query and perform new report validation
        v_analytics_data := JSON_OBJECT_T('{}');
        IF p_new_report_validation THEN
            lang_data_logger_pkg.log_info(
                'New report creation validation: ' || v_report_id
            );

            -- Simple search using new report's description as query
            -- get top 5 result. 
            lang_data_search_pkg.
                    get_top_k_closest_report_and_description_similarity_search(
                        p_query               => p_description_text,
                        p_results             => v_report_cursor
                    );

            -- Get result as report id
            LOOP
                FETCH v_report_cursor INTO v_report_rec;
                EXIT WHEN v_report_cursor%NOTFOUND;
                v_match_document_arr.append(v_report_rec.report_id);
            END LOOP;
            CLOSE v_report_cursor;

            -- Save the result
            v_analytics_data.put('similar_report_ids', 
                                                        v_match_document_arr);

        END IF;

        IF p_description_vector IS NULL THEN
            v_description_vector := lang_data_utils_pkg.get_embedding(
                v_enhanced_description
            );
        ELSE
            v_description_vector := p_description_vector;
        END IF;
        
        -- Insert Report
        EXECUTE IMMEDIATE 
            'INSERT INTO langdata$reports (
                id, 
                title, 
                match_document, 
                status, 
                domain_id,
                analytics_data
            ) VALUES (
                :report_id,
                :title,
                :match_document,
                :report_status,
                :domain_id,
                :analytics_data
            )'
        USING 
            v_report_id, 
            p_title, 
            p_match_document,
            p_report_status,
            v_domain_id,
            v_analytics_data.to_clob();

        -- Insert Report Description
        EXECUTE IMMEDIATE 
            'INSERT INTO langdata$reportdescriptions (
                id, 
                report_id, 
                text, 
                status, 
                description_vector, 
                description_md5, 
                enhanced_text,
                augmented_tokens
            ) VALUES (
                :description_id, 
                :report_id, 
                :description_text, 
                :description_status, 
                lang_data_utils_pkg.get_embedding(:enhanced_description),
                :description_md5, 
                :enhanced_description,
                :augmented_tokens
            )'
        USING 
            v_description_id, 
            v_report_id, 
            p_description_text, 
            p_description_status, 
            v_enhanced_description, 
            v_description_md5, 
            v_enhanced_description,
            v_augmented_tokens;

        -- Disable adding custom entity_types.
        IF lang_data_config_pkg.g_custom_entities = TRUE THEN
            lang_data_named_entities_pkg.add_named_entities_to_document(
                p_match_document
            );
        END IF;

        lang_data_utils_pkg.create_value_vector_job(
            p_match_document    => p_match_document,
            p_document_id       => v_report_id,
            p_domain_id         => v_domain_id
        );

        IF p_description_status = 'Pending Regression' THEN
            calculate_report_regression(
                p_new_description_id  => v_description_id,
                p_is_new_report       => TRUE
            );
        END IF;

        IF p_sample_queries IS NOT NULL THEN
            lang_data_sample_queries_pkg.add_report_sample_queries(
                p_report_id      => v_report_id,
                p_sample_queries => p_sample_queries,
                p_ids            => v_sample_query_ids
            );
        END IF;

        lang_data_logger_pkg.log_info(
            'Successfully created report: ' || v_report_id
        );

        p_report_id := v_report_id;
        p_description_id := v_description_id;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN(  lang_data_errors_pkg.c_invalid_match_document,
                            lang_data_errors_pkg.c_invalid_parameters_code,
                            lang_data_errors_pkg.c_resource_already_exists,
                            lang_data_errors_pkg.c_max_text_length_exceeded,
                            lang_data_errors_pkg.c_unauthorized_code)  THEN
                RAISE;
            END IF;
            

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || 
                v_report_id || '. Error: ' || SQLERRM
            );
            RAISE;
    END create_report;

    PROCEDURE calculate_report_regression (
        p_new_description_id  IN  VARCHAR2 DEFAULT NULL,
        p_new_sample_query_id IN VARCHAR2 DEFAULT NULL,
        p_is_new_report       IN BOOLEAN DEFAULT FALSE 
    ) IS
        -- Local variables
        v_new_text     VARCHAR2(4000);
        v_status  VARCHAR2(20);    
        v_new_report_id       VARCHAR2(36);
        v_query_text          VARCHAR2(2000);
        v_query_vector        VECTOR(*,*);
        v_new_distance        NUMBER;
        v_good_search_records SYS_REFCURSOR;
        v_search_results      SYS_REFCURSOR;
        v_regression_count    NUMBER;
        v_regression_accepted_record_count NUMBER;
        v_record_id                         VARCHAR2(36);
        v_record_created_at                 TIMESTAMP;
        v_regression_json_object            JSON_OBJECT_T := JSON_OBJECT_T();
        v_regressing_records                JSON_ARRAY_T := JSON_ARRAY_T();
        v_regression_record_object          JSON_OBJECT_T;
        v_regression_json                   JSON;
        v_desc_vector           VECTOR(*,*);
        v_is_description    BOOLEAN := TRUE;
        v_new_description     VARCHAR2(4000);
        v_description_status  VARCHAR2(20);  
        v_domain_id           VARCHAR2(36);

        -- top two closest matching reports
        v_report_id1          VARCHAR2(36) := '###';
        v_report_title        VARCHAR2(255);
        v_desc_text            VARCHAR2(4000);
        v_distance1           NUMBER;
        v_id1                 VARCHAR2(36);
        v_enhanced_text1      VARCHAR2(4000);
        v_text1               VARCHAR2(2000);
        v_title1              VARCHAR2(255);
        v_match_type1         VARCHAR2(100);
        v_type1               VARCHAR2(10);
        v_rank1               NUMBER;
        v_identifier_report   VARCHAR2(36) := '###';
        v_expected_report_id                VARCHAR2(36);
        v_expected_drilldown_id             VARCHAR2(36);
        v_new_report_status                 VARCHAR2(20);  

        --create TYPE for BULK COLLECT
        TYPE t_match_rec IS RECORD (
            record_id            langdata$searchrecords.id%TYPE,
            query_text           langdata$searchrecords.query_text%TYPE,
            expected_report_id   langdata$searchrecords.expected_report_id%TYPE,
            domain_id            langdata$searchrecords.domain_id%TYPE,
            created_at           langdata$searchrecords.created_at%TYPE,
            new_distance         NUMBER,
            --
            report_id1           langdata$reports.id%TYPE,
            report_title         langdata$reports.title%TYPE,
            description_id1      langdata$reportdescriptions.id%TYPE,
            desc_text             VARCHAR2(4000),
            old_distance1        NUMBER,
            match_type1          VARCHAR2(32)
        );
        TYPE t_match_tab IS TABLE OF t_match_rec INDEX BY PLS_INTEGER;
        l_matches t_match_tab;

        -- batch size for Bulk collect
        c_batch_size CONSTANT PLS_INTEGER := 1000;

        -- cursor for extracting queries candidate for regression
        CURSOR c_matches (
            p_desc_vec IN langdata$searchrecords.query_vector%TYPE,
            p_domain_id IN langdata$searchrecords.domain_id%TYPE
        ) IS
            SELECT /*+ PARALLEL */ 
                sr.id AS record_id,
                sr.query_text AS query_text,
                sr.expected_report_id AS expected_report_id,
                sr.domain_id as domain_id,
                sr.created_at as created_at,
                VECTOR_DISTANCE(sr.query_vector, p_desc_vec) AS new_distance,
                t.report_id AS report_id1,
                t.title as report_title,
                t.id AS description_id1,
                t.desc_text AS desc_text,
                t.distance AS old_distance1,
                t.match_type AS match_type1
            FROM langdata$searchrecords sr
            LEFT JOIN LATERAL (
                SELECT * FROM (
                    SELECT 
                        report_id, title, id, desc_text, distance, match_type,
                        ROW_NUMBER() OVER (
                            PARTITION BY report_id ORDER BY distance ASC
                        ) AS rn
                    FROM (
                        -- Report Description
                        SELECT * FROM (
                            SELECT
                                rd.report_id,
                                r2.title as title,
                                rd.id,
                                rd.text as desc_text,
                                VECTOR_DISTANCE(rd.description_vector, 
                                                sr.query_vector) AS distance,
                                'Report Description' AS match_type
                            FROM langdata$reportdescriptions rd
                            JOIN langdata$reports r2 ON rd.report_id = r2.id
                            WHERE rd.status = 'Published' AND 
                                  r2.status = 'Published' AND
                                  (sr.domain_id IS NULL OR
                                   sr.domain_id = r2.domain_id)

                            ORDER BY VECTOR_DISTANCE(
                                rd.description_vector, sr.query_vector)
                            FETCH FIRST 1 ROWS ONLY
                        )

                        UNION ALL
                        -- Sample Query
                        SELECT * FROM (
                            SELECT
                                sq.report_id,
                                r3.title as title,
                                sq.id,
                                sq.query_text as desc_text,
                                VECTOR_DISTANCE(sq.query_vector,
                                                sr.query_vector) AS distance,
                                'Report Sample Query' AS match_type
                            FROM langdata$samplequeries sq
                            JOIN langdata$reports r3 ON sq.report_id = r3.id
                            WHERE r3.status = 'Published' AND 
                                  sq.status  = 'Published' AND
                                  (sr.domain_id IS NULL OR
                                   sr.domain_id = r3.domain_id)
                            ORDER BY VECTOR_DISTANCE(
                                sq.query_vector, sr.query_vector)
                            FETCH FIRST 1 ROWS ONLY

                        )
                        
                        UNION ALL
                        -- Search Record
                        SELECT * FROM (
                            SELECT
                                sr2.expected_report_id AS report_id,
                                r4.title as title,
                                sr2.id,
                                sr2.query_text as desc_text,
                                VECTOR_DISTANCE(sr2.query_vector, 
                                                sr.query_vector) AS distance,
                                'Search Record' AS match_type
                            FROM langdata$searchrecords sr2
                            JOIN langdata$reports r4 
                                    ON sr2.expected_report_id = r4.id
                            WHERE r4.status = 'Published' AND 
                                  sr2.id <> sr.id AND
                                  (sr.domain_id IS NULL OR
                                   sr.domain_id = r4.domain_id)
                            ORDER BY VECTOR_DISTANCE(
                                sr2.query_vector, sr.query_vector)
                            FETCH FIRST 1 ROWS ONLY
                        )
                    )
                )
                WHERE rn = 1
                ORDER BY distance
                FETCH FIRST 1 ROWS ONLY
            ) t ON 1 = 1
            WHERE p_domain_id IS NULL OR 
                p_domain_id = sr.domain_id;

    BEGIN
        IF p_new_description_id IS NULL AND p_new_sample_query_id IS NULL THEN
            lang_data_logger_pkg.log_error(
                'Both p_new_description_id and p_new_sample_query_id '||
                'cannot be null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_new_description_id IS NOT NULL 
            AND p_new_sample_query_id IS NOT NULL THEN
            lang_data_logger_pkg.log_error(
                'Both p_new_description_id and p_new_sample_query_id '||
                'cannot be not null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );

        END IF;
        
        IF p_new_description_id IS NOT NULL THEN
            v_is_description := TRUE;
        ELSE
            v_is_description := FALSE;
        END IF;

        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        lang_data_logger_pkg.log_info(
            'Starting Regression calculation for report.');

        IF v_is_description THEN
            SELECT enhanced_text, status, report_id
            INTO v_new_description, v_description_status, v_new_report_id
            FROM langdata$reportdescriptions
            WHERE id = p_new_description_id;
        ELSE
            SELECT enhanced_query_text, status, report_id
            INTO v_new_description, v_description_status, v_new_report_id
            FROM langdata$samplequeries
            WHERE id = p_new_sample_query_id;
        END IF;
        
        SELECT domain_id INTO v_domain_id 
        FROM langdata$reports WHERE id = v_new_report_id;

        IF NOT v_status = 'Pending Regression' THEN
            IF v_is_description THEN
                lang_data_logger_pkg.log_fatal(
                    'Report Description must be in Pending Regression state'||
                    ' to run regression'
                );
            ELSE
                lang_data_logger_pkg.log_fatal(
                    'Sample Query must be in Pending Regression state'||
                    ' to run regression'
                );
            END IF;
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        v_desc_vector := lang_data_utils_pkg.get_embedding(v_new_description);

        -- Force to change the new report into pending regression
        IF p_is_new_report THEN
            -- Store temp report status
            SELECT status                   
            INTO   v_new_report_status      
            FROM   langdata$reports
            WHERE  id = v_new_report_id;  
                    
            -- Update report status
            UPDATE langdata$reports SET status = 'Pending Regression'
            WHERE id = v_new_report_id;

            -- Update description or sample queries status
            IF v_is_description THEN
                UPDATE langdata$reportdescriptions 
                    SET status = 'Pending Regression'
                WHERE id = p_new_description_id;
            ELSE
                UPDATE langdata$samplequeries SET status = 'Pending Regression'
                WHERE id = p_new_sample_query_id;
            END IF;
        END IF;
        
        -- Initialize regression count
        v_regression_count := 0;
        v_regression_accepted_record_count := 0;
        -- Loop through all records, fetch their report matches.
        -- Consider the top two reports matched for regression calculation
        OPEN c_matches(v_desc_vector, v_domain_id);
        
        LOOP

            -- bulk collect from sql query result
            FETCH c_matches BULK COLLECT INTO l_matches LIMIT c_batch_size;
            EXIT WHEN l_matches.COUNT = 0;  -- no more rows

            for i in 1 .. l_matches.COUNT LOOP

                -- extract record and report info
                v_record_id          := l_matches(i).record_id;
                v_query_text         := l_matches(i).query_text;
                v_expected_report_id := l_matches(i).expected_report_id;
                v_new_distance       := l_matches(i).new_distance;
                v_record_created_at := l_matches(i).created_at;

                v_id1        := l_matches(i).description_id1;
                v_report_title := l_matches(i).report_title;
                v_report_id1 := l_matches(i).report_id1;
                v_desc_text := l_matches(i).desc_text;
                v_distance1  := l_matches(i).old_distance1;
                v_match_type1:= l_matches(i).match_type1;

                IF (v_report_id1 = v_identifier_report) THEN
                    CONTINUE;
                END IF;

                v_regression_record_object := JSON_OBJECT_T();
                v_regression_record_object.put('record_id', v_record_id);
                v_regression_record_object.put(
                    'created_at', v_record_created_at);
                v_regression_record_object.put('query', v_query_text);
                v_regression_record_object.put(
                    'prev_report_id', v_report_id1);
                v_regression_record_object.put(
                    'prev_report_title', v_report_title);
                v_regression_record_object.put(
                    'prev_desc_id', v_id1);
                v_regression_record_object.put(
                    'prev_desc_type', v_match_type1);
                v_regression_record_object.put(
                    'prev_dist', v_distance1
                );
                v_regression_record_object.put(
                    'prev_desc', v_desc_text
                );
                -- Description object being added is closer to the query than 
                -- old top descripion object, and report to which it is being added
                -- is not the same as the old top description object's report.
                IF  v_new_report_id <> v_report_id1 AND 
                    v_new_distance < v_distance1 THEN
                    
                    v_regression_record_object.put(
                        'new_report_id', v_new_report_id);
                    v_regression_record_object.put(
                        'new_desc_id', p_new_description_id);
                    v_regression_record_object.put(
                        'new_best_distance', v_new_distance
                    );
                    IF v_is_description THEN
                        v_regression_record_object.put(
                            'new_desc_type', 'Report Description'
                        );
                    ELSE
                        v_regression_record_object.put(
                            'new_desc_type', 'Report Sample Query'
                        );
                    END IF;
                    IF v_expected_report_id IS NULL THEN
                        v_regression_count := v_regression_count + 1;
                        v_regression_record_object.put(
                            'record_has_accepted_report', 'NONE');
                    ELSE  
                        v_regression_accepted_record_count := 
                            v_regression_accepted_record_count + 1;
                        v_regression_record_object.put(
                            'record_has_accepted_report', v_expected_report_id);
                    END IF;   
                    v_regression_record_object.put(
                        'new_dist', v_new_distance
                    );

                    v_regressing_records.append(v_regression_record_object);
                END IF;
            END LOOP;
        END LOOP;
        CLOSE c_matches;

        v_regression_json_object.put('regression_count', v_regression_count);
        v_regression_json_object.put(
            'regression_accepted_record_count', 
            v_regression_accepted_record_count
        );
        v_regression_json_object.put(
            'regressing_records',
            v_regressing_records
        );
        IF v_regression_accepted_record_count <> 0 
        OR v_regression_count <> 0 THEN
            v_regression_json := JSON(v_regression_json_object.to_clob());
            -- Insert regression_json
            IF v_is_description THEN
                UPDATE langdata$reportdescriptions 
                SET regression_json = v_regression_json,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_description_id;
            ELSE
                UPDATE langdata$samplequeries
                SET regression_json = v_regression_json,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_sample_query_id;
            END IF;
        ElSE
            lang_data_logger_pkg.log_debug(
                'No regression found setting regression_json to null'
            );
            IF v_is_description THEN
                UPDATE langdata$reportdescriptions 
                SET regression_json = NULL,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_description_id;
            ELSE
                UPDATE langdata$samplequeries
                SET regression_json = NULL,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = p_new_sample_query_id;
            END IF;
        END IF;

        -- Get current status of text
        IF v_is_description THEN
            SELECT status
            INTO v_status
            FROM langdata$reportdescriptions
            WHERE id = p_new_description_id;
        ELSE
            SELECT status
            INTO v_status
            FROM langdata$samplequeries
            WHERE id = p_new_sample_query_id;
        END IF;
        -- Don't update the status, If status has been changed from initial 
        -- 'Pending Regression to something else
        IF v_status = 'Pending Regression' THEN
            IF v_is_description THEN
                UPDATE langdata$reportdescriptions SET status = 'Pending Review'
                WHERE id = p_new_description_id;
            ELSE
                UPDATE langdata$samplequeries SET status = 'Pending Review'
                WHERE id = p_new_sample_query_id;
            END IF;
        END IF;

        -- Change the report status back to original
        IF p_is_new_report THEN
            UPDATE langdata$reports SET status = v_new_report_status
            WHERE id = v_new_report_id;
        END IF;

        lang_Data_logger_pkg.log_info('Finished regression calculation');

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred while calculating ' || 
                'regression. Error: ' || SQLERRM
                );
            RAISE;

    END calculate_report_regression;

    PROCEDURE get_all_reports (
        p_status            IN VARCHAR2 DEFAULT NULL,
        p_cursor            IN OUT VARCHAR2,
        p_limit             IN NUMBER DEFAULT 10,
        p_reports           OUT SYS_REFCURSOR
    ) AS
        v_cur               INTEGER;
        v_created_at        TIMESTAMP;
        v_cursor_id         VARCHAR2(36);
        v_last_id           VARCHAR2(36);
        v_last_created_at   TIMESTAMP;
        v_sql_text          VARCHAR2(4000);  

        -- Query to fetch all reports and their latest published Description.
        v_reports_query     VARCHAR2(4000) := 
            'SELECT r.id, r.title, r.match_document, r.status, ( ' ||
            'SELECT text FROM ( '||
                'SELECT rd.text FROM langdata$reportdescriptions rd '||
                    'WHERE rd.report_id = r.id AND rd.status = ''Published'' '||
                    'OR (
                        rd.status = ''Approved''
                        AND r.status = ''Approved''
                        AND NOT EXISTS (
                            SELECT 1
                            FROM langdata$reportdescriptions rd2
                            WHERE rd2.report_id = r.id
                            AND rd2.status = ''Published''
                        )
                    )' ||
                    'ORDER BY rd.created_at DESC '||
            ') WHERE ROWNUM = 1 ) AS report_description_text '||
            'FROM langdata$reports r ';

        -- Conditions for breaking in pagination
        v_conditions        VARCHAR2(4000) := 
            '(r.created_at < :created_at OR '||
            '(r.created_at = :created_at AND r.id <= :cursor_id)) ';
        v_dummy             INTEGER;
    BEGIN
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        IF p_status NOT IN ('Pending Review',
                            'Approved',
                            'Rejected',
                            'Published',
                            'Inactive',
                            'Archived' ) THEN
            lang_data_logger_pkg.log_error('Report status invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );        
        END IF;
        
        IF p_status IS NOT NULL THEN
            v_reports_query := v_reports_query || 'WHERE r.status = :status ';
        END IF;

        v_cur := DBMS_SQL.OPEN_CURSOR;

        IF p_cursor IS NOT NULL THEN
            lang_data_utils_pkg.split_cursor(
                p_cursor => p_cursor,
                p_created_at => v_created_at,
                p_id   => v_cursor_id
            );
            lang_data_logger_pkg.log_debug(
                'Applied pagination cursor: CreatedAt=' || 
                TO_CHAR(v_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') || ', ID=' || 
                v_cursor_id
            );
            
            IF p_status IS NOT NULL THEN
                v_reports_query := v_reports_query || 'AND ';
            ELSE
                v_reports_query := v_reports_query || 'WHERE ';
            END IF;
            
            v_sql_text  :=  v_reports_query || v_conditions;
        ELSE
            v_sql_text  := v_reports_query;
        END IF;

        IF p_limit IS NOT NULL THEN
            v_sql_text := v_sql_text || 
                ' ORDER BY r.created_at DESC, r.id DESC '||
                'FETCH FIRST :limit ROWS ONLY ';
        ELSE
            v_sql_text := v_sql_text || 
                ' ORDER BY r.created_at DESC, r.id DESC ';
        END IF;

        lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);

        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);
        
        IF p_status IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':status', p_status);
        END IF;

        IF p_cursor IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':created_at', v_created_at);
            DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
        END IF;

        IF p_limit IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);
        END IF;
        v_dummy     :=  DBMS_SQL.EXECUTE(v_cur);
        p_reports   := DBMS_SQL.TO_REFCURSOR(v_cur);

        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        IF p_limit IS NOT NULL THEN
            lang_data_logger_pkg.log_debug('Checking for next report row');

            v_sql_text  :=  'SELECT id, created_at FROM langdata$reports r ';

            IF p_status IS NOT NULL THEN
                v_sql_text := v_sql_text || 'WHERE status = :status ';
            END IF;

            IF p_cursor IS NOT NULL THEN
                IF p_status IS NOT NULL THEN
                    v_sql_text  := v_sql_text || 'AND ' || v_conditions;
                ELSE
                    v_sql_text  := v_sql_text || 'WHERE' || v_conditions;
                END IF;
            END IF;

            v_sql_text := v_sql_text || 
                'ORDER BY created_at DESC, id DESC OFFSET :limit ROWS '||
                'FETCH NEXT 1 ROWS ONLY ';

            v_cur :=  DBMS_SQL.OPEN_CURSOR;

            lang_data_logger_pkg.log_debug('Executing query: ' || v_sql_text);
            DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

            IF p_cursor IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(v_cur, ':created_at', v_created_at);
                DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
            END IF;

            IF p_status IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(v_cur, ':status', p_status);
            END IF;

            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);

            DBMS_SQL.DEFINE_COLUMN(v_cur, 1, v_last_id, 36);
            DBMS_SQL.DEFINE_COLUMN(v_cur, 2, v_last_created_at);

            v_dummy :=  DBMS_SQL.EXECUTE(v_cur);

            IF DBMS_SQL.FETCH_ROWS(v_cur) > 0 THEN
                DBMS_SQL.COLUMN_VALUE(v_cur, 1, v_last_id);
                DBMS_SQL.COLUMN_VALUE(v_cur, 2, v_last_created_at);

                p_cursor := TO_CHAR(
                    v_last_created_at, 
                    'YYYY-MM-DD HH24:MI:SS.FF'
                ) || '|' || v_last_id;

                lang_data_logger_pkg.log_debug(
                    'Next description cursor set to: ' || p_cursor
                );
            ELSE
                p_cursor := NULL;
                lang_data_logger_pkg.log_debug('No further pages.');
            END IF;

            IF DBMS_SQL.IS_OPEN(v_cur) THEN
                DBMS_SQL.CLOSE_CURSOR(v_cur);
            END IF;
        ELSE
            p_cursor := NULL;
            lang_data_logger_pkg.log_debug('No further pages.');
        END IF;

        lang_data_logger_pkg.log_info('Successfully retrieved all_Reports');
        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE IN (
                    lang_data_errors_pkg.c_unauthorized_code,
                    lang_data_errors_pkg.c_invalid_parameters_code
                ) THEN
                    RAISE;
                END IF;
                
    END get_all_reports;

    PROCEDURE delete_report(
        p_report_id         IN VARCHAR2
    )IS
        v_match_document    JSON;
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting delete report procedure: '|| p_report_id
        );
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- Verify if the report exists
        IF NOT lang_data_utils_pkg.check_report_exists(p_report_id) THEN
            lang_data_logger_pkg.log_error('Report id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
        
        -- Check if report exists and fetch its match_document
        SELECT match_document
        INTO v_match_document
        FROM langdata$reports
        WHERE id = p_report_id;

        lang_data_utils_pkg.decrement_vector_table_references(v_match_document);

        lang_data_utils_pkg.drop_enumerable_set_value_vector_tables(
            p_report_id, v_match_document
        );

        EXECUTE IMMEDIATE 
            'DELETE FROM langdata$reports WHERE id = :report_id' 
        USING p_report_id;
        
        lang_data_logger_pkg.log_info(
            'Successfully deleted report: '|| p_report_id
        );

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN ( lang_data_errors_pkg.c_unauthorized_code,
                            lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || 
                p_report_id || '. Error: ' || SQLERRM
            );
            RAISE;
    END delete_report;
    
    PROCEDURE add_report_description(
        p_report_id         IN VARCHAR2,
        p_text              IN VARCHAR2,
        p_status            IN VARCHAR2,
        p_description_id    OUT VARCHAR2
    ) IS 
        v_match_document        JSON;
        v_ner_augmented_desc    VARCHAR2(4000);
        v_enhanced_description   VARCHAR2(4000);
        v_description_md5       VARCHAR2(32); -- MD5 hash of normalized query
        v_dummy                 NUMBER;
        v_description_id        VARCHAR2(36);
        v_normalized_text       VARCHAR2(4000);
        v_augmented_tokens      JSON;
        v_ner_augmentation_enabled  BOOLEAN;
        v_entities                  JSON;
        v_domain_id             VARCHAR2(36);
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure to add description to report of ID: ' 
            || p_report_id
        );
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;
        
        v_description_id := lang_data_utils_pkg.generate_id();

        IF LENGTH(p_text) > 2000  THEN
            lang_data_logger_pkg.log_error(
                'Description length is more than 2000 characters'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_max_text_length_exceeded
            );
        END IF;

        IF p_status NOT IN ( 'Pending Review',
                                    'Approved',
                                    'Rejected',
                                    'Published',
                                    'Inactive',
                                    'Archived',
                                    'Pending Regression' ) THEN
            lang_data_logger_pkg.log_error('Invalid status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF NOT lang_data_utils_pkg.check_report_exists(p_report_id) THEN
            lang_data_logger_pkg.log_error('Report id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        v_ner_augmented_desc := p_text;

        IF v_ner_augmentation_enabled THEN
            v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                            p_text
                        );
            IF v_entities IS NOT NULL THEN
                v_ner_augmented_desc := 
                    lang_data_utils_pkg.augment_text_with_ner_entities(
                        p_text, v_entities
                    );
            END IF;
        END IF;

        EXECUTE IMMEDIATE 
            'SELECT match_document, domain_id FROM langdata$reports 
            WHERE id = :report_id' 
        INTO v_match_document, v_domain_id USING p_report_id;
        
        lang_data_utils_pkg.augment_text(
            p_text             =>  v_ner_augmented_desc,
            p_domain_id        =>  v_domain_id,
            p_augmented_text   =>  v_enhanced_description,
            p_augmented_tokens =>  v_augmented_tokens
        );

        v_enhanced_description   :=  lang_data_utils_pkg.expand_text(
            p_match_document => v_match_document,
            p_original_text  => v_enhanced_description,
            p_text_type      => 'report'
        );

        v_normalized_text := LOWER(TRIM(p_text));
        -- Generate MD5 for the report description
        v_description_md5 := RAWTOHEX(
            DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(v_normalized_text),
                DBMS_CRYPTO.HASH_MD5
            )
        );

        -- Check if description already exists
        EXECUTE IMMEDIATE 
            'SELECT COUNT(1) FROM langdata$reportdescriptions 
            WHERE description_md5 = :md5' 
        INTO v_dummy USING v_description_md5;

        IF v_dummy > 0 THEN
            lang_data_logger_pkg.log_error('Report description already exists');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_already_exists
            );
        END IF;

        EXECUTE IMMEDIATE 
            'INSERT INTO langdata$reportdescriptions 
                (id, 
                report_id, 
                text, 
                status, 
                description_vector, 
                description_md5, 
                enhanced_text,
                augmented_tokens
            ) 
            VALUES 
                (:description_id, 
                :report_id, 
                :text, 
                :status, 
                lang_data_utils_pkg.get_embedding(:enhanced_description),
                :description_md5, 
                :enhanced_description,
                :augmented_tokens
            )'
        USING 
            v_description_id,
            p_report_id,
            p_text,
            p_status,
            v_enhanced_description,
            v_description_md5,
            v_enhanced_description,
            v_augmented_tokens;

        IF p_status = 'Pending Regression' THEN
            lang_data_utils_pkg.create_or_replace_job(
                p_job_name => 'JOB_LANGDATA_'||
                    substr(v_description_md5,1,5)||'_REPORT_D_REGRESSION',
                p_job_action => 
                    'BEGIN' ||
            'lang_data_reports_pkg.calculate_report_regression( '||
                'p_new_description_id => '''
                || v_description_id ||'''); END;',
                p_job_class       => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                p_comments        => 
                    'Regression calculation for report description',
                p_priority        => 1,
                p_restart_on_fail => FALSE,
                p_restart_on_rec  => TRUE
            );
        END IF;

        lang_data_logger_pkg.log_info(
            'Successfully added report description: '|| v_description_id
        );

        p_description_id := v_description_id;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN(  lang_data_errors_pkg.c_invalid_parameters_code,
                            lang_data_errors_pkg.c_resource_already_exists,
                            lang_data_errors_pkg.c_max_text_length_exceeded,
                            lang_data_errors_pkg.c_unauthorized_code)  THEN
                RAISE;
            END IF;
            

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || 
                v_description_id || '. Error: ' || SQLERRM
            );
            RAISE;

    END add_report_description;
    
    PROCEDURE delete_report_description(
        p_description_id        IN VARCHAR2
    )IS 
        v_row_count             NUMBER;
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting delete report description procedure: '|| p_description_id
        );
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        EXECUTE IMMEDIATE 
            'SELECT COUNT(1) FROM langdata$reportdescriptions WHERE id = :id' 
        INTO v_row_count USING p_description_id;

        IF v_row_count = 0 THEN
            lang_data_logger_pkg.log_error('Invalid description id');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        EXECUTE IMMEDIATE 
            'SELECT COUNT(1) FROM langdata$reportdescriptions WHERE report_id = 
            (SELECT report_id FROM langdata$reportdescriptions WHERE id = :id)' 
            INTO v_row_count USING p_description_id;
        IF v_row_count < 2 THEN
            lang_data_logger_pkg.log_error(
                'At least one description is required for the report.'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_description_invalid_deletion
            );
        END IF;

        EXECUTE IMMEDIATE 
            'DELETE FROM langdata$reportdescriptions 
            WHERE id = :description_id' 
        USING p_description_id;

        lang_data_logger_pkg.log_info(
            'Successfully deleted report description: ' || p_description_id
        );

        EXCEPTION
            WHEN OTHERS THEN
                IF SQLCODE IN ( 
                    lang_data_errors_pkg.c_unauthorized_code,
                    lang_data_errors_pkg.c_invalid_parameters_code,
                    lang_data_errors_pkg.c_description_invalid_deletion
                ) THEN
                    RAISE;
                END IF;

                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred for ID: ' || 
                    p_description_id || '. Error: ' || SQLERRM
                );
                RAISE;
    END delete_report_description;

    PROCEDURE update_report_description(
        p_id                IN VARCHAR2,
        p_text              IN VARCHAR2
    ) IS 
        v_status                VARCHAR2(20);
        v_match_document        JSON;
        v_ner_augmented_desc    VARCHAR2(4000);
        v_enhanced_description  VARCHAR2(4000);
        v_description_md5       VARCHAR2(32); -- MD5 hash of normalized query
        v_dummy                 NUMBER;
        v_normalized_text       VARCHAR2(4000);
        v_augmented_tokens      JSON;
        v_ner_augmentation_enabled  BOOLEAN := FALSE;
        v_entities                  JSON;
        v_domain_id             VARCHAR2(36);
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting update report description procedure: '||p_id
        );
        -- Check if User is Authorized to perform the action
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        IF LENGTH(p_text) > 2000  THEN
            lang_data_logger_pkg.log_error(
                'Description length is more than 2000 characters'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_max_text_length_exceeded
            );
        END IF;

        BEGIN
            SELECT status INTO v_status 
            FROM langdata$reportdescriptions WHERE id = p_id;

            IF NOT v_status = 'Pending Review' THEN
                lang_data_logger_pkg.log_error(
                    'Description can be updated only for Pending Review status'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_parameters_code
                );
            END IF;

        EXCEPTION 
            WHEN NO_DATA_FOUND THEN
                lang_data_logger_pkg.log_error(
                    'No description found with id ' || p_id
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_resource_not_found
                );
        END;

        EXECUTE IMMEDIATE 
            'SELECT match_document, domain_id ' ||
            'FROM langdata$reports WHERE id = ( '||
            'SELECT report_id FROM langdata$reportdescriptions rd '||
            'WHERE rd.id =  :id )' 
        INTO v_match_document, v_domain_id USING p_id;

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        v_ner_augmented_desc := p_text;

        IF v_ner_augmentation_enabled THEN
            v_entities := lang_data_named_entities_pkg.get_entities_from_text(
                            p_text
                        );
            IF v_entities IS NOT NULL THEN
                v_ner_augmented_desc :=
                    lang_data_utils_pkg.augment_text_with_ner_entities(
                        p_text, v_entities
                    );
            END IF;
        END IF;

        lang_data_utils_pkg.augment_text(
            p_text             =>  v_ner_augmented_desc,
            p_domain_id        =>  v_domain_id,
            p_augmented_text   =>  v_enhanced_description,
            p_augmented_tokens =>  v_augmented_tokens
        );
        
        v_enhanced_description   :=  lang_data_utils_pkg.expand_text(
            p_match_document => v_match_document,
            p_original_text  => v_enhanced_description,
            p_text_type      => 'report'
        );

        v_normalized_text := LOWER(TRIM(p_text));
        -- Generate MD5 for the report description
        v_description_md5 := RAWTOHEX(
            DBMS_CRYPTO.HASH(
                UTL_RAW.CAST_TO_RAW(v_normalized_text), 
                DBMS_CRYPTO.HASH_MD5
            )
        );

        -- Check if description already exists
        EXECUTE IMMEDIATE 
            'SELECT COUNT(1) FROM langdata$reportdescriptions '||
            'WHERE description_md5 = :md5' 
        INTO v_dummy USING v_description_md5;

        IF v_dummy > 0 THEN
            lang_data_logger_pkg.log_error(
                'Report description already exists'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_already_exists
            );
        END IF;

        EXECUTE IMMEDIATE 
            'UPDATE langdata$reportdescriptions
            SET text = :text, 
            description_md5 = :md5, 
            enhanced_text = :enhanced_description, 
            description_vector = lang_data_utils_pkg.get_embedding(
                :enhanced_description
            ),
            augmented_tokens = :augmented_tokens
            WHERE id = :id' 
        USING 
            p_text,
            v_description_md5,
            v_enhanced_description,
            v_enhanced_description,
            v_augmented_tokens,
            p_id;

    lang_data_logger_pkg.log_info(
        'Successfully updated report description: '|| p_id
    );
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN(  lang_data_errors_pkg.c_invalid_parameters_code,
                            lang_data_errors_pkg.c_resource_not_found,
                            lang_data_errors_pkg.c_resource_already_exists,
                            lang_data_errors_pkg.c_max_text_length_exceeded,
                            lang_data_errors_pkg.c_unauthorized_code)  THEN
                RAISE;
            END IF;
            
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' ||
                p_id || '. Error: ' || SQLERRM
            );
            RAISE;
    END update_report_description;

    PROCEDURE get_report_description(
        p_id                IN VARCHAR2,
        p_text              OUT VARCHAR2,
        p_version           OUT NUMBER,
        p_status            OUT VARCHAR2,
        p_report_id         OUT VARCHAR2,
        p_enhanced_text      OUT VARCHAR2
    ) IS
        v_cur               INTEGER;
        v_sql_text          VARCHAR2(4000) := 
            'SELECT text, version, status, report_id, enhanced_text '||
            'FROM langdata$reportdescriptions WHERE id = :id';
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting get report description procedure: '||p_id
        );
        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);
        DBMS_SQL.BIND_VARIABLE(v_cur, ':id', p_id);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 1, p_text, 2000);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 2, p_version);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 3, p_status, 20);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 4, p_report_id, 36);
        DBMS_SQL.DEFINE_COLUMN(v_cur, 5, p_enhanced_text, 4000);

        IF DBMS_SQL.EXECUTE_AND_FETCH(v_cur) > 0 THEN
            DBMS_SQL.COLUMN_VALUE(v_cur, 1, p_text);
            DBMS_SQL.COLUMN_VALUE(v_cur, 2, p_version);
            DBMS_SQL.COLUMN_VALUE(v_cur, 3, p_status);
            DBMS_SQL.COLUMN_VALUE(v_cur, 4, p_report_id);
            DBMS_SQL.COLUMN_VALUE(v_cur, 5, p_enhanced_text);
        ELSE
            lang_data_logger_pkg.log_error(
                'Report description not found: '||p_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        END IF;

        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN(lang_data_errors_pkg.c_resource_not_found)  THEN
                RAISE;
            END IF;

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' ||
                p_id || '. Error: ' || SQLERRM
            );
            RAISE;
    END get_report_description;

    PROCEDURE execute_report_sql(
        p_report_id IN VARCHAR2,
        p_filter_values IN JSON,
        p_columns OUT SYS.ODCIVARCHAR2LIST,
        p_data OUT CLOB
    )
    IS
        v_report_sql    VARCHAR2(4000);
        v_cur              INTEGER;
        v_dummy            INTEGER;
        v_num_cols        INTEGER;
        v_desc_tab        DBMS_SQL.DESC_TAB;
        v_json            JSON_OBJECT_T;
        v_match_document JSON;
        v_keys           JSON_KEY_LIST;
        v_key            VARCHAR2(4000);
        v_filter_obj     JSON_OBJECT_T;
        v_value          VARCHAR2(4000);
        v_col_value      VARCHAR2(4000);
        v_row_json         VARCHAR2(32767);
    BEGIN
        SELECT match_document 
        INTO v_match_document
        FROM langdata$reports
        WHERE id = p_report_id;

        -- Extract the SQL
        v_report_sql := JSON_VALUE(
            v_match_document, '$.sql' RETURNING VARCHAR2(4000)
        );
        lang_data_logger_pkg.log_info(
            'SQL Query to be executed: ' || v_report_sql
        );

        -- Open the cursor for dynamic SQL execution
        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_report_sql, DBMS_SQL.NATIVE);

        -- Dynamically extract all key-value pairs using JSON_TABLE
        v_json := JSON_OBJECT_T.parse(json_serialize(p_filter_values));
        v_keys := v_json.get_keys;
        FOR i IN v_keys.FIRST..v_keys.LAST LOOP
            v_key := v_keys(i);

            v_filter_obj := v_json.get_object(v_key);

            v_value := v_filter_obj.get_string('value');

            lang_data_logger_pkg.log_info(
                'Filter: ' || v_key || ' | Value: ' || v_value
            );
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':' || v_key, v_value
            );
        END LOOP;
        
        -- Describe the columns in the result set
        DBMS_SQL.DESCRIBE_COLUMNS(v_cur, v_num_cols, v_desc_tab);

        -- Add columns to the cursor
        FOR i IN 1 .. v_num_cols LOOP
            DBMS_SQL.DEFINE_COLUMN(v_cur, i, v_col_value, 4000);
        END LOOP;

        -- Initialize the LOB p_data
        DBMS_LOB.CREATETEMPORARY(p_data, TRUE);

        -- Execute the SQL
        v_dummy := DBMS_SQL.EXECUTE(v_cur);

        -- Prepare output for columns
        p_columns := SYS.ODCIVARCHAR2LIST();

        FOR j IN 1 .. v_num_cols LOOP
            p_columns.EXTEND;
            p_columns(j) := v_desc_tab(j).col_name;
            lang_data_logger_pkg.log_info('p_columns(j) = ' || p_columns(j));
        END LOOP;

        lang_data_logger_pkg.log_info('v_num_cols = ' || v_num_cols);

        -- Fetch the rows and build the JSON string
        WHILE DBMS_SQL.FETCH_ROWS(v_cur) > 0 LOOP
            v_row_json := '{'; -- Start the JSON object
            FOR k IN 1 .. v_num_cols LOOP
                DBMS_SQL.COLUMN_VALUE(v_cur, k, v_col_value);
                -- Add each column value to the JSON object
                v_row_json := v_row_json || '"' || p_columns(k) || '": ' || '"' || v_col_value || '"';
                IF k < v_num_cols THEN
                    v_row_json := v_row_json || ', '; -- Add comma if not the last column
                END IF;
            END LOOP;
            v_row_json := v_row_json || '}'; -- End the JSON object
            -- lang_data_logger_pkg.log_info('v_row_json = ' || v_row_json);

            -- Append the row JSON object to the CLOB
            DBMS_LOB.WRITEAPPEND(p_data, LENGTH(v_row_json), v_row_json);
        END LOOP;

        lang_data_logger_pkg.log_info('Current CLOB length: ' || DBMS_LOB.GETLENGTH(p_data));

        -- Close the cursor
        DBMS_SQL.CLOSE_CURSOR(v_cur);
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No report found for ID: '|| p_report_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred. Error: ' || SQLERRM
            );
            RAISE;
    END execute_report_sql;

    PROCEDURE replace_report_filter_enumerable_set (
        p_report_id             IN  VARCHAR2,
        p_filter_name           IN  VARCHAR2,
        p_new_enumerable_set    IN  JSON_ARRAY_T
    )
    IS
        v_match_document      JSON;
        v_match_document_obj  JSON_OBJECT_T;
        v_filters             JSON_ARRAY_T;
        v_filter_obj          JSON_OBJECT_T;
        v_filter_name         VARCHAR2(255);
        v_filter_description  VARCHAR2(2000);
        v_updated             BOOLEAN := FALSE;
        v_job_name            VARCHAR2(255);
        v_md5                 VARCHAR2(32);
        v_domain_id           VARCHAR2(36);
    BEGIN
        IF NOT lang_data_utils_pkg.check_report_exists(p_report_id) THEN
            lang_data_logger_pkg.log_error('Report id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT match_document, domain_id INTO v_match_document, v_domain_id
        FROM langdata$reports
        WHERE id = p_report_id;

        v_match_document_obj := JSON_OBJECT_T.PARSE(
            JSON_SERIALIZE(v_match_document)
        );

        v_filters := v_match_document_obj.GET_ARRAY('filters');

        FOR indx IN 0 .. v_filters.GET_SIZE - 1 LOOP
            v_filter_obj := JSON_OBJECT_T(v_filters.GET(TO_NUMBER(indx)));
            v_filter_name := v_filter_obj.GET_STRING('filter_name');
            IF UPPER(v_filter_name) = UPPER(p_filter_name) THEN
                IF NOT v_filter_obj.HAS('enumerable_set') THEN
                    lang_data_logger_pkg.log_error(
                        'Filter ' || p_filter_name ||
                        'does not have enumerable set'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_parameters_code
                    );
                END IF;
                -- Update the 'enumerable_set' field
                v_filter_obj.PUT('enumerable_set', p_new_enumerable_set);
                v_updated := TRUE;
                v_filter_description := v_filter_obj.GET_STRING('description');
                EXIT;
            END IF;

        END LOOP;

        IF v_updated THEN
            v_match_document := JSON(v_match_document_obj.TO_STRING());

            UPDATE langdata$reports 
            SET match_document = v_match_document
            WHERE id = p_report_id;

            v_md5 := RAWTOHEX(
                DBMS_CRYPTO.HASH(
                    UTL_RAW.CAST_TO_RAW(p_report_id),
                    DBMS_CRYPTO.HASH_MD5
                )
            );

            v_job_name := 'JOB_LANGDATA_' || UPPER(p_filter_name) || '_' ||
                          v_md5 || '_VALUE_VECTORS';
            BEGIN
                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name        => v_job_name,
                    p_job_action      =>
                        'DECLARE ' ||
                        '  v_enum_set JSON_ARRAY_T := JSON_ARRAY_T.PARSE(''' || 
                        p_new_enumerable_set.TO_STRING() || '''); ' ||
                        'BEGIN ' ||
                        '  lang_data_utils_pkg.create_value_vector_from_enumerable_set(' ||
                        '    v_enum_set, ' ||
                        '    ''' || p_report_id || ''', ' ||
                        '    ''' || p_filter_name || ''', ' ||
                        '    ''' || 
                        REPLACE(v_filter_description, '''', '''''') || ''', ' ||
                        '    TRUE, ' ||
                        '    ''' || NVL(v_domain_id, '') || '''' ||
                        '  ); ' ||
                        'END;',
                    p_job_class    => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments        => 'Create value vector table from enumerable set',
                    p_priority        => 1,
                    p_restart_on_fail => TRUE,
                    p_restart_on_rec  => TRUE
                );
                lang_data_logger_pkg.log_info(
                    'Submitted job ' || v_job_name ||
                    ' to recreate value vector table for filter ' ||
                    p_filter_name
                );

            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -27477 THEN
                        lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                    ' already exists');
                    ELSE
                        lang_data_logger_pkg.log_fatal(
                            'An unknown error occurred. Error: ' || SQLERRM
                        );
                        RAISE;
                    END IF;
            END;
        ELSE
            lang_data_logger_pkg.log_error(
                'Filter ' || p_filter_name ||
                'does not exist'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
    END replace_report_filter_enumerable_set;

    PROCEDURE add_into_report_filter_enumerable_set (
        p_report_id                IN  VARCHAR2,
        p_filter_name              IN  VARCHAR2,
        p_enumerable_set_to_add    IN  JSON_ARRAY_T
    )
    IS
        v_match_document      JSON;
        v_match_document_obj  JSON_OBJECT_T;
        v_filters             JSON_ARRAY_T;
        v_filter_obj          JSON_OBJECT_T;
        v_filter_name         VARCHAR2(255);
        v_filter_description  VARCHAR2(2000);
        v_old_enumerable_set  JSON_ARRAY_T;
        v_merged_enumerable_set       JSON_ARRAY_T := JSON_ARRAY_T();
        v_new_enumerable_set_json     JSON;
        v_old_enumerable_set_json     JSON;
        v_updated             BOOLEAN := FALSE;
        v_job_name            VARCHAR2(255);
        v_md5                 VARCHAR2(32);
        v_domain_id           VARCHAR2(36);
    BEGIN
        IF NOT lang_data_utils_pkg.check_report_exists(p_report_id) THEN
            lang_data_logger_pkg.log_error('Report id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT match_document, domain_id INTO v_match_document, v_domain_id
        FROM langdata$reports
        WHERE id = p_report_id;

        v_match_document_obj := JSON_OBJECT_T.PARSE(
            JSON_SERIALIZE(v_match_document)
        );

        v_filters := v_match_document_obj.GET_ARRAY('filters');

        FOR indx IN 0 .. v_filters.GET_SIZE - 1 LOOP
            v_filter_obj := JSON_OBJECT_T(v_filters.GET(TO_NUMBER(indx)));
            v_filter_name := v_filter_obj.GET_STRING('filter_name');
            IF UPPER(v_filter_name) = UPPER(p_filter_name) THEN
                IF NOT v_filter_obj.HAS('enumerable_set') THEN
                    lang_data_logger_pkg.log_error(
                        'Filter ' || p_filter_name ||
                        'does not have enumerable set'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_parameters_code
                    );
                END IF;
                -- Update the 'enumerable_set' field
                v_old_enumerable_set := v_filter_obj.GET_ARRAY('enumerable_set');
                v_new_enumerable_set_json := JSON(p_enumerable_set_to_add.TO_STRING());
                v_old_enumerable_set_json := JSON(v_old_enumerable_set.TO_STRING());

                FOR rec IN (
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_new_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                    UNION
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_old_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                ) LOOP
                    v_merged_enumerable_set.APPEND(rec.value);
                END LOOP;

                v_filter_obj.PUT('enumerable_set', v_merged_enumerable_set);
                v_updated := TRUE;
                v_filter_description := v_filter_obj.GET_STRING('description');
                EXIT;
            END IF;

        END LOOP;

        IF v_updated THEN
            v_match_document := JSON(v_match_document_obj.TO_STRING());

            UPDATE langdata$reports 
            SET match_document = v_match_document
            WHERE id = p_report_id;

            v_md5 := RAWTOHEX(
                DBMS_CRYPTO.HASH(
                    UTL_RAW.CAST_TO_RAW(p_report_id),
                    DBMS_CRYPTO.HASH_MD5
                )
            );

            v_job_name := 'JOB_LANGDATA_' || UPPER(p_filter_name) || '_' ||
                          v_md5 || '_VALUE_VECTORS';
            BEGIN
                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name        => v_job_name,
                    p_job_action      =>
                        'DECLARE ' ||
                        '  v_enum_set JSON_ARRAY_T := JSON_ARRAY_T.PARSE(''' || 
                        v_merged_enumerable_set.TO_STRING() || '''); ' ||
                        'BEGIN ' ||
                        '  lang_data_utils_pkg.create_value_vector_from_enumerable_set(' ||
                        '    v_enum_set, ' ||
                        '    ''' || p_report_id || ''', ' ||
                        '    ''' || p_filter_name || ''', ' ||
                        '    ''' || 
                        REPLACE(v_filter_description, '''', '''''') || ''', ' ||
                        '    TRUE, ' ||
                        '    ''' || NVL(v_domain_id, '') || '''' ||
                        '  ); ' ||
                        'END;',
                    p_job_class    => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments        => 'Create value vector table from enumerable set',
                    p_priority        => 1,
                    p_restart_on_fail => TRUE,
                    p_restart_on_rec  => TRUE
                );

                lang_data_logger_pkg.log_info(
                    'Submitted job ' || v_job_name ||
                    ' to recreate value vector table for filter ' ||
                    p_filter_name
                );

            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -27477 THEN
                        lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                    ' already exists');
                    ELSE
                        lang_data_logger_pkg.log_fatal(
                            'An unknown error occurred. Error: ' || SQLERRM
                        );
                        RAISE;
                    END IF;
            END;
        ELSE
            lang_data_logger_pkg.log_error(
                'Filter ' || p_filter_name ||
                'does not exist'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
    END add_into_report_filter_enumerable_set;

    PROCEDURE remove_from_report_filter_enumerable_set (
        p_report_id                   IN  VARCHAR2,
        p_filter_name                 IN  VARCHAR2,
        p_enumerable_set_to_remove    IN  JSON_ARRAY_T
    )
    IS
        v_match_document      JSON;
        v_match_document_obj  JSON_OBJECT_T;
        v_filters             JSON_ARRAY_T;
        v_filter_obj          JSON_OBJECT_T;
        v_filter_name         VARCHAR2(255);
        v_filter_description  VARCHAR2(2000);
        v_old_enumerable_set  JSON_ARRAY_T;
        v_updated_enumerable_set      JSON_ARRAY_T := JSON_ARRAY_T();
        v_new_enumerable_set_json     JSON;
        v_old_enumerable_set_json     JSON;
        v_updated             BOOLEAN := FALSE;
        v_job_name            VARCHAR2(255);
        v_md5                 VARCHAR2(32);
        v_domain_id           VARCHAR2(36);
    BEGIN
        IF NOT lang_data_utils_pkg.check_report_exists(p_report_id) THEN
            lang_data_logger_pkg.log_error('Report id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        SELECT match_document, domain_id INTO v_match_document, v_domain_id
        FROM langdata$reports
        WHERE id = p_report_id;

        v_match_document_obj := JSON_OBJECT_T.PARSE(
            JSON_SERIALIZE(v_match_document)
        );

        v_filters := v_match_document_obj.GET_ARRAY('filters');

        FOR indx IN 0 .. v_filters.GET_SIZE - 1 LOOP
            v_filter_obj := JSON_OBJECT_T(v_filters.GET(TO_NUMBER(indx)));
            v_filter_name := v_filter_obj.GET_STRING('filter_name');
            IF UPPER(v_filter_name) = UPPER(p_filter_name) THEN
                IF NOT v_filter_obj.HAS('enumerable_set') THEN
                    lang_data_logger_pkg.log_error(
                        'Filter ' || p_filter_name ||
                        'does not have enumerable set'
                    );
                    lang_data_errors_pkg.raise_error(
                        lang_data_errors_pkg.c_invalid_parameters_code
                    );
                END IF;
                -- Update the 'enumerable_set' field
                v_old_enumerable_set := v_filter_obj.GET_ARRAY('enumerable_set');
                v_new_enumerable_set_json := JSON(p_enumerable_set_to_remove.TO_STRING());
                v_old_enumerable_set_json := JSON(v_old_enumerable_set.TO_STRING());

                FOR rec IN (
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_old_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                    MINUS
                    SELECT jt.value
                    FROM JSON_TABLE(
                        v_new_enumerable_set_json,
                        '$[*]' COLUMNS (value VARCHAR2(4000) PATH '$')
                    ) jt
                ) LOOP
                    v_updated_enumerable_set.APPEND(rec.value);
                END LOOP;

                v_filter_obj.PUT('enumerable_set', v_updated_enumerable_set);
                v_updated := TRUE;
                v_filter_description := v_filter_obj.GET_STRING('description');
                EXIT;
            END IF;

        END LOOP;

        IF v_updated THEN
            v_match_document := JSON(v_match_document_obj.TO_STRING());

            UPDATE langdata$reports 
            SET match_document = v_match_document
            WHERE id = p_report_id;

            v_md5 := RAWTOHEX(
                DBMS_CRYPTO.HASH(
                    UTL_RAW.CAST_TO_RAW(p_report_id),
                    DBMS_CRYPTO.HASH_MD5
                )
            );

            v_job_name := 'JOB_LANGDATA_' || UPPER(p_filter_name) || '_' ||
                          v_md5 || '_VALUE_VECTORS';
            BEGIN
                lang_data_utils_pkg.create_or_replace_job(
                    p_job_name        => v_job_name,
                    p_job_action      =>
                        'DECLARE ' ||
                        '  v_enum_set JSON_ARRAY_T := JSON_ARRAY_T.PARSE(''' || 
                        v_updated_enumerable_set.TO_STRING() || '''); ' ||
                        'BEGIN ' ||
                        '  lang_data_utils_pkg.create_value_vector_from_enumerable_set(' ||
                        '    v_enum_set, ' ||
                        '    ''' || p_report_id || ''', ' ||
                        '    ''' || p_filter_name || ''', ' ||
                        '    ''' || 
                        REPLACE(v_filter_description, '''', '''''') || ''', ' ||
                        '    TRUE, ' ||
                        '    ''' || NVL(v_domain_id, '') || '''' ||
                        '  ); ' ||
                        'END;',
                    p_job_class    => lang_data_config_pkg.get_config_parameter(
                                        'LANG_DATA_JOB_CLASS_NAME'
                                    ),
                    p_comments        => 'Create value vector table from enumerable set',
                    p_priority        => 1,
                    p_restart_on_fail => TRUE,
                    p_restart_on_rec  => TRUE
                );

                lang_data_logger_pkg.log_info(
                    'Submitted job ' || v_job_name ||
                    ' to recreate value vector table for filter ' ||
                    p_filter_name
                );

            EXCEPTION
                WHEN OTHERS THEN
                    IF SQLCODE = -27477 THEN
                        lang_data_logger_pkg.log_debug('Job ' || v_job_name || 
                                                    ' already exists');
                    ELSE
                        lang_data_logger_pkg.log_fatal(
                            'An unknown error occurred. Error: ' || SQLERRM
                        );
                        RAISE;
                    END IF;
            END;
        ELSE
            lang_data_logger_pkg.log_error(
                'Filter ' || p_filter_name ||
                'does not exist'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
    END remove_from_report_filter_enumerable_set;

    PROCEDURE update_report_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
    BEGIN
        -- Check if report exists
        IF NOT lang_data_utils_pkg.check_report_exists(p_id) THEN
            lang_data_logger_pkg.log_error('Report id invalid');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Validate if status is among required values
        IF p_status NOT IN ( 'Pending Review',
                             'Approved',
                             'Rejected',
                             'Published',
                             'Inactive',
                             'Archived' ) THEN
            lang_data_logger_pkg.log_error('Invalid status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        EXECUTE IMMEDIATE 
            'UPDATE langdata$reports 
             SET status = :status
             WHERE id = :id' 
        USING p_status, p_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;
            
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' ||
                p_id || '. Error: ' || SQLERRM
            );
            RAISE;
    END update_report_status;

    PROCEDURE update_report_description_status(
        p_id                IN VARCHAR2,
        p_status            IN VARCHAR2
    ) IS 
        v_row_count         NUMBER;
        v_report_id         VARCHAR2(36);
        v_prev_desc_status  VARCHAR2(20);
        v_report_status     VARCHAR2(20);
    BEGIN
        -- Check description exists
        EXECUTE IMMEDIATE 
            'SELECT COUNT(1) FROM langdata$reportdescriptions WHERE id = :id' 
        INTO v_row_count USING p_id;

        IF v_row_count = 0 THEN
            lang_data_logger_pkg.log_error('Invalid description id');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Validate if status is among required values
        IF p_status NOT IN ( 'Pending Review',
                             'Approved',
                             'Rejected',
                             'Published',
                             'Inactive',
                             'Archived',
                             'Pending Regression' ) THEN
            lang_data_logger_pkg.log_error('Invalid status');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        -- Check description exists
        EXECUTE IMMEDIATE 
            'SELECT report_id, status FROM 
            langdata$reportdescriptions WHERE id = :id' 
        INTO v_report_id, v_prev_desc_status USING p_id;

        IF v_prev_desc_status=p_status THEN
            RETURN;
        END IF;

        -- Check description exists
        EXECUTE IMMEDIATE 
            'SELECT status FROM langdata$reports WHERE id = :id' 
        INTO v_report_status USING v_report_id;

        -- Validate a Published report has atleast 
        -- one published report description
        IF v_report_status='Published' AND v_prev_desc_status='Published' THEN
            EXECUTE IMMEDIATE 
                'SELECT count(id) FROM langdata$reportdescriptions WHERE 
                status = :status AND report_id = :report_id' 
            INTO v_row_count USING v_prev_desc_status, v_report_id;
            IF v_row_count=1 THEN
                lang_data_logger_pkg.log_error(
                    'A published report must have atleast one ' || 
                    'published description. ' ||
                    'Update report status first to enable update of ' || 
                    ' the description with ID:' ||
                    p_id || '. Error: ' || SQLERRM
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_update
                );
            END IF;
        END IF;

        EXECUTE IMMEDIATE 
            'UPDATE langdata$reportdescriptions
             SET status = :status
             WHERE id = :id' 
        USING p_status, p_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (lang_data_errors_pkg.c_invalid_parameters_code) THEN
                RAISE;
            END IF;
            

            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' ||
                p_id || '. Error: ' || SQLERRM
            );
            raise;
    END update_report_description_status;

    PROCEDURE get_report_regression(
        p_description_id      IN VARCHAR2 DEFAULT NULL,
        p_sample_query_id     IN VARCHAR2 DEFAULT NULL,
        p_is_new_report       IN BOOLEAN DEFAULT FALSE,
        p_force               IN BOOLEAN,
        p_json                OUT JSON
    ) IS
        v_regression_json       JSON;
        v_status                VARCHAR2(20);
        v_text_md5              VARCHAR2(36);
        v_job_name              VARCHAR2(255);
        v_is_description        BOOLEAN := TRUE;
    BEGIN
        IF p_description_id IS NULL AND p_sample_query_id IS NULL THEN
            lang_data_logger_pkg.log_error(
                'Both p_description_id and p_sample_query_id '||
                'cannot be null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_description_id IS NOT NULL 
            AND p_sample_query_id IS NOT NULL THEN
            lang_data_logger_pkg.log_error(
                'Both p_description_id and p_sample_query_id '||
                'cannot be not null'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_description_id IS NOT NULL THEN
            v_is_description := TRUE;
        ELSE
            v_is_description := FALSE;
        END IF;

        IF v_is_description THEN
            SELECT status, regression_json, description_md5
            INTO v_status, v_regression_json, v_text_md5
            FROM langdata$reportdescriptions
            WHERE id = p_description_id;
            v_job_name := 'JOB_LANGDATA_'||
                        substr(v_text_md5,1,5)||'_REPORT_D_REGRESSION';
        ELSE
            SELECT status, regression_json, query_md5
            INTO v_status, v_regression_json, v_text_md5
            FROM langdata$samplequeries
            WHERE id = p_sample_query_id;
            v_job_name := 'JOB_LANGDATA_'||
                        substr(v_text_md5,1,5)||'_REPORT_SQ_REGRESSION';
        END IF;
        
        IF NOT p_force THEN
            IF v_regression_json IS NULL THEN
                lang_data_logger_pkg.log_info('Regression Info Not available');
                p_json := NULL;
            ELSE
                p_json := v_regression_json;
            END IF;
            RETURN;
        END IF;

        IF v_status = 'Pending Regression' THEN
            lang_data_logger_pkg.log_debug(
                'Regression Job already exists dropping job'
            );
            BEGIN
                DBMS_SCHEDULER.drop_job(
                    job_name => v_job_name,
                    force => TRUE
                );
                lang_data_logger_pkg.log_info(
                    'Job ' || v_job_name || ' dropped successfully.'
                );
            EXCEPTION
                WHEN OTHERS THEN
                    -- Catch the exception if the job does not exist
                    IF SQLCODE != -27475 THEN 
                        -- Error code for "job does not exist"
                        lang_data_logger_pkg.log_info(
                            'Error dropping job: ' || SQLERRM
                        );
                    ELSE
                        lang_data_logger_pkg.log_info(
                            'Job ' || v_job_name ||
                            ' does not exist, proceeding to create.'
                        );
                    END IF;
            END;

        END IF;

        IF v_is_description THEN
            UPDATE langdata$reportdescriptions
            SET status = 'Pending Regression'
            WHERE id = p_description_id;
        ELSE
            UPDATE langdata$samplequeries
            SET status = 'Pending Regression'
            WHERE id = p_sample_query_id;
        END IF;


        lang_data_reports_pkg.calculate_report_regression(
            p_new_description_id => p_description_id,
            p_new_sample_query_id => p_sample_query_id,
            p_is_new_report => p_is_new_report
        );

        IF v_is_description THEN
            SELECT regression_json INTO v_regression_json
            FROM langdata$reportdescriptions WHERE id = p_description_id;
            p_json := v_regression_json;
        ELSE
            SELECT regression_json INTO v_regression_json
            FROM langdata$samplequeries WHERE id = p_sample_query_id;
            p_json := v_regression_json;
        END IF;
        lang_Data_logger_pkg.log_debug('Finished get_report_regression');
        RETURN;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            IF v_is_description THEN
                lang_data_logger_pkg.log_error(
                    'Description with id: '''|| p_description_id ||
                     ''' not found');
            ELSE
                lang_Data_logger_pkg.log_error(
                    'Sample Query with id: '''|| p_sample_query_id || 
                    '''not found');
            END IF;
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred while fetching  ' || 
                'report description regression. Error: ' || SQLERRM
                );
            RAISE;
    END get_report_regression;

    FUNCTION get_report_id_by_title(
        p_title IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_report_id VARCHAR2(36);
    BEGIN
        SELECT id INTO v_report_id FROM langdata$reports WHERE title = p_title;
        RETURN v_report_id;
    
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No report found for title: ' || p_title
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_report_id_by_title;

    FUNCTION get_desc_id_by_report_id_version(
        p_report_id    IN VARCHAR2,
        p_version      IN NUMBER
    ) RETURN VARCHAR2 IS
        v_desc_id VARCHAR2(36);
    BEGIN
        SELECT id INTO v_desc_id FROM langdata$reportdescriptions
        WHERE report_id = p_report_id AND version = p_version;
        RETURN v_desc_id;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No report description found for report ID: ' || p_report_id || 
                ', version: ' || p_version
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_desc_id_by_report_id_version;

    FUNCTION get_report_status_by_id (
        p_report_id     IN VARCHAR2
    ) RETURN VARCHAR2 IS
        v_status        VARCHAR2(20);
    BEGIN
        SELECT status INTO v_status FROM langdata$reports
        WHERE id = p_report_id;
        RETURN v_status;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No report found for report ID: ' || p_report_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_report_status_by_id;

    FUNCTION get_report_match_document_by_id(
        p_report_id         VARCHAR2
    ) RETURN CLOB IS
        v_match_document    CLOB;
    BEGIN
        SELECT match_document INTO v_match_document FROM langdata$reports
        WHERE id = p_report_id;
        RETURN v_match_document;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No report found for report ID: ' || p_report_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
    END get_report_match_document_by_id;

    PROCEDURE purge_reports(
        p_domain        IN VARCHAR2,
        p_timestamp     IN TIMESTAMP DEFAULT NULL,
        p_hit_limit     IN NUMBER DEFAULT NULL,
        p_action        IN VARCHAR2 DEFAULT 'COUNT',
        p_num_reports   OUT NUMBER
    ) IS
        -- Cursor to find report IDs matching the purge criteria
        CURSOR c_reports IS
            WITH domain_lookup AS (
                SELECT d.domain_id
                FROM langdata$domains d
                WHERE d.name = p_domain
            ),
            report_hits AS (
                SELECT 
                    JSON_VALUE(s.report_matches, '$[0].report_id') AS report_id,
                    COUNT(*) AS hit_count
                FROM langdata$searchrecords s
                GROUP BY JSON_VALUE(s.report_matches, '$[0].report_id')
            )
            SELECT r.id
            FROM langdata$reports r
            LEFT JOIN domain_lookup dl
                ON p_domain IS NOT NULL AND r.domain_id = dl.domain_id
            LEFT JOIN report_hits h
                ON h.report_id = r.id
            WHERE
                (p_timestamp IS NULL OR r.created_at < p_timestamp)
                AND (p_domain IS NULL OR dl.domain_id IS NOT NULL)
                AND (p_hit_limit IS NULL OR NVL(h.hit_count, 0) < p_hit_limit);


        v_count_deleted INTEGER := 0;
    BEGIN
        -- Loop over matching reports and delete
        FOR r IN c_reports LOOP
            IF p_action = 'DELETE' THEN
                lang_data_reports_pkg.delete_report(r.id);
            ELSIF p_action = 'ARCHIVE' THEN
                lang_data_reports_pkg.update_report_status(
                    p_id => r.id,
                    p_status => 'Archived'
                );
            END IF;
            v_count_deleted := v_count_deleted + 1;
        END LOOP;

        p_num_reports :=  v_count_deleted;
    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No reports found matching the purge criteria.'
            );
            p_num_reports := 0;
    END purge_reports;

END lang_data_reports_pkg;
/
