Rem
Rem $Header: dbgendev/src/langdata/plsql/analytics/analytics_pkg.pkb /main/18 2025/07/25 19:30:58 sathyavc Exp $
Rem
Rem analytics_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      analytics_pkg.pkb - lang data analytics package
Rem
Rem    DESCRIPTION
Rem      This package implements procedures for analytics and fetching metrics
Rem      in lang data.
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/analytics/analytics_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    sathyavc    07/23/25 - DBAI-1048: Make update_api_logs an autonomous
Rem                           transaction. Add RETURNING CLOB clauses to get
Rem                           API stats and failures functions.
Rem    ruohli      07/21/25 - DBAI-1111: get_matched_queries_by_drilldown_rank
Rem                           and get_drilldown_filter_usage
Rem    jiangnhu    07/17/25 - DBAI-1091: Change format of report_filter_values,
Rem                           drilldown_filter_values in report_matches
Rem    sathyavc    07/07/25 - DBAI-883: Add logic to track API metrics over 
Rem                           specified time windows. Remove update_api_metrics
Rem    pryarla     07/03/25 - DBAI-882: Added get_top_filters function
Rem    ruohli      07/01/25 - DBAI-884: Added get_matched_queries_by_report_rank
Rem    sathyavc    06/26/25 - DBAI-881: Add top reports, bottom reports and
Rem                           most asked questions
Rem    sathyavc    06/25/25 - DBAI-881: Modify feedback count APIs to be
Rem                           filtered based on time.
Rem    anisbans    06/13/25 - Added the procedure 
Rem                           get_top_k_most_searched_reports
Rem    saloshah    05/19/25 - DBAI-746: Added the update_api_metrics
Rem    arevathi    04/07/25 - Added Drilldown Metrics
Rem    anisbans    02/21/25 - Return JSON_OBJECT_T in get_metrics
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    saloshah    12/03/24 - DBAI-438: Added error handling for all procedures
Rem    deveverm    10/30/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data_analytics_pkg IS

    PROCEDURE get_success_rate_metrics (
        p_records                       IN JSON_ARRAY_T,
        p_success_rate                  OUT NUMBER
    ) AS
        v_total_searches_with_feedback  NUMBER := 0;
        v_accepted_search_cnt           NUMBER := 0;

        v_record                        JSON_OBJECT_T;
        v_report_match                  JSON_OBJECT_T;
        v_report_matches                JSON_ARRAY_T;
    BEGIN
        IF p_records IS NULL THEN
            lang_data_logger_pkg.log_error('The records cannot be NULL');
            lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
              );
        END IF;
        FOR i in 0 .. p_records.get_size - 1 LOOP
            v_record := TREAT (
                p_records.get(i) AS JSON_OBJECT_T
            );
            IF
                v_record.get_string('expected_report_id') IS NOT NULL
                OR (
                    v_record.has('feedback_rating')
                    AND v_record.get('feedback_rating').is_boolean
                    AND v_record.get_boolean('feedback_rating') IS NOT NULL
                )
                OR v_record.get_string('feedback_comments') IS NOT NULL
            THEN
                v_total_searches_with_feedback :=
                    v_total_searches_with_feedback + 1;
            END IF;

            IF v_record.GET_STRING('expected_report_id') IS NOT NULL THEN
                v_report_matches := JSON_ARRAY_T(
                    v_record.GET('report_matches')
                );
                -- Check for a matching report in report_matches JSON
                FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
                    -- Done as json_array_t.get returns json_element_t
                    -- which needs to be converted to json_object_t
                    v_report_match := TREAT (
                        v_report_matches.get(i) AS JSON_OBJECT_T
                    );

                    IF v_report_match.get_string('report_id') =
                        v_record.GET_STRING('expected_report_id')
                    THEN
                        v_accepted_search_cnt := v_accepted_search_cnt + 1;
                        EXIT;  -- Exit loop once a match is found
                    END IF;
                END LOOP;
            END IF;

        END LOOP;

        IF v_total_searches_with_feedback > 0 THEN
            p_success_rate := ROUND(
                (
                    v_accepted_search_cnt /
                    v_total_searches_with_feedback
                ) * 100, 2);
        ELSE
            p_success_rate := 0;
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in get_success_rate_metrics.'
                || ' Error: ' || SQLERRM
            );
            RAISE;
    END get_success_rate_metrics;

    PROCEDURE get_precision_metrics (
        p_records               IN JSON_ARRAY_T,
        k                       IN INTEGER,
        p_precision             OUT NUMBER
    ) AS
        v_total_searches_with_feedback  NUMBER := 0;
        v_search_count                  NUMBER := 0;
        v_record                        JSON_OBJECT_T;
        v_report_matches                JSON_ARRAY_T;
        v_report_match                  JSON_OBJECT_T;
    BEGIN
        IF p_records IS NULL THEN
            lang_data_logger_pkg.log_error('The records cannot be NULL');
            lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
              );
        END IF;
        FOR i in 0 .. p_records.get_size - 1 LOOP
            v_record := TREAT (
                p_records.get(i) AS JSON_OBJECT_T
            );

            IF
                v_record.get_string('expected_report_id') IS NOT NULL
                OR (
                    v_record.has('feedback_rating')
                    AND v_record.get('feedback_rating').is_boolean
                    AND v_record.get_boolean('feedback_rating') IS NOT NULL
                )
                OR v_record.get_string('feedback_comments') IS NOT NULL
            THEN
                v_total_searches_with_feedback :=
                    v_total_searches_with_feedback + 1;
            END IF;

            IF v_record.GET_STRING('expected_report_id') IS NOT NULL THEN
                IF v_record.GET('report_matches') IS NULL THEN
                    CONTINUE;
                END IF;
                v_report_matches := JSON_ARRAY_T(
                    v_record.GET('report_matches')
                );

                -- Check for a matching report in report_matches JSON
                FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
                    -- Done as json_array_t.get returns json_element_t
                    -- which needs to be converted to json_object_t
                    v_report_match := TREAT (
                        v_report_matches.get(i) AS JSON_OBJECT_T
                    );

                    IF 
                        v_report_match.get_string('report_id') =
                            v_record.GET_STRING('expected_report_id') AND
                        v_report_match.get_NUMBER('overall_rank') <= k 
                    THEN
                        v_search_count := v_search_count + 1;
                        EXIT;  -- Exit loop once a match is found
                    END IF;
                END LOOP;
            END IF;
        END LOOP;

        IF v_total_searches_with_feedback > 0 THEN
            p_precision := ROUND(
                (
                    v_search_count / 
                    v_total_searches_with_feedback
                ) * 100, 2);
        ELSE
            p_precision := 0;
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in get_precision_metrics.'
                || ' Error: ' || SQLERRM
            );
            RAISE;
    END get_precision_metrics;

    PROCEDURE get_mean_reciprocal_rank (
        p_records               IN JSON_ARRAY_T,
        p_mean_reciprocal_rank  OUT NUMBER
    ) IS
        v_total_accepted_count  NUMBER := 0;
        v_record                JSON_OBJECT_T;
        v_overall_count         NUMBER := 0;
        v_report_matches        JSON_ARRAY_T;
        v_report_match          JSON_OBJECT_T;
    BEGIN
        IF p_records IS NULL THEN
            lang_data_logger_pkg.log_error('The records cannot be NULL');
            lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
              );
        END IF;
        p_mean_reciprocal_rank := 0;
        FOR i in 0 .. p_records.get_size - 1 LOOP
            v_record := TREAT (
                p_records.get(i) AS JSON_OBJECT_T
            );

            IF v_record.GET_STRING('expected_report_id') IS NOT NULL THEN
                IF v_record.GET('report_matches') IS NULL THEN
                    CONTINUE;
                END IF;
                v_report_matches := JSON_ARRAY_T(
                    v_record.GET('report_matches')
                );

                -- Check for a matching report in report_matches JSON
                FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
                    -- Done as json_array_t.get() returns json_element_t
                    -- which needs to be converted to json_object_t
                    v_report_match := TREAT (
                        v_report_matches.get(i) AS JSON_OBJECT_T
                    );

                    IF 
                        v_report_match.GET_STRING('report_id') =
                            v_record.GET_STRING('expected_report_id')
                    THEN
                        v_total_accepted_count := v_total_accepted_count + 1;
                        
                        v_overall_count := v_overall_count + 
                            1 / v_report_match.GET_NUMBER('overall_rank');
                        
                        EXIT;  -- Exit loop once a match is found
                    END IF;
                END LOOP;
            END IF;
        END LOOP;
        IF v_total_accepted_count = 0 THEN
            p_mean_reciprocal_rank := 0;
        ELSE
            p_mean_reciprocal_rank := ROUND(
                v_overall_count / v_total_accepted_count,
                2
            );
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in get_mean_reciprocal_rank.'
                || ' Error: ' || SQLERRM
            );
            RAISE;
    END get_mean_reciprocal_rank;

    PROCEDURE get_positive_feedback_percentage(
        p_records                     IN  JSON_ARRAY_T,
        p_positive_feedback_percentage OUT NUMBER
    ) IS
        v_feedback_count         NUMBER := 0;
        v_positive_feedback_count NUMBER := 0;
        v_record                 JSON_OBJECT_T;
    BEGIN
        IF p_records IS NULL THEN
            lang_data_logger_pkg.log_error('The records cannot be NULL');
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        FOR i IN 0 .. p_records.get_size - 1 LOOP
            v_record := TREAT(p_records.get(i) AS JSON_OBJECT_T);

            IF v_record.has('feedback_rating') AND 
            v_record.get_boolean('feedback_rating') IS NOT NULL THEN
                v_feedback_count := v_feedback_count + 1;
                IF v_record.get_boolean('feedback_rating') THEN
                    v_positive_feedback_count := v_positive_feedback_count+1;
                END IF;
            END IF;
        END LOOP;

        IF v_feedback_count = 0 THEN
            p_positive_feedback_percentage := 0;
        ELSE
            p_positive_feedback_percentage := ROUND(
                (v_positive_feedback_count / v_feedback_count) * 100,
                2
            );
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in ' || 
                'get_positive_feedback_percentage.'|| 'Error: ' || SQLERRM
            );
            RAISE;
    END get_positive_feedback_percentage;


    PROCEDURE get_drilldown_precision_metrics (
        p_records               IN JSON_ARRAY_T,
        k                       IN INTEGER,
        p_precision             OUT NUMBER
    ) AS
        v_total_searches_with_feedback  NUMBER := 0;
        v_search_count                  NUMBER := 0;
        v_record                        JSON_OBJECT_T;
        v_report_matches                JSON_ARRAY_T;
        v_report_match                  JSON_OBJECT_T;
    BEGIN
        IF p_records IS NULL THEN
            lang_data_logger_pkg.log_error('The records cannot be NULL');
            lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
              );
        END IF;
        FOR i in 0 .. p_records.get_size - 1 LOOP
            v_record := TREAT (
                p_records.get(i) AS JSON_OBJECT_T
            );

            IF
                v_record.GET_STRING('expected_drilldown_id') IS NOT NULL
            THEN
                v_total_searches_with_feedback :=
                    v_total_searches_with_feedback + 1;
            END IF;

            IF v_record.GET_STRING('expected_drilldown_id') IS NOT NULL THEN
                IF v_record.GET('report_matches') IS NULL THEN
                    CONTINUE;
                END IF;
                v_report_matches := JSON_ARRAY_T(
                    v_record.GET('report_matches')
                );

                -- Check for a matching report in report_matches JSON
                FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
                    -- Done as json_array_t.get returns json_element_t
                    -- which needs to be converted to json_object_t
                    v_report_match := TREAT (
                        v_report_matches.get(i) AS JSON_OBJECT_T
                    );

                    IF 
                        v_report_match.get_string('drilldown_id') =
                            v_record.GET_STRING('expected_drilldown_id') AND
                        v_report_match.get_NUMBER('overall_rank') <= k 
                    THEN
                        v_search_count := v_search_count + 1;
                        EXIT;  -- Exit loop once a match is found
                    END IF;
                END LOOP;
            END IF;
        END LOOP;

        IF v_total_searches_with_feedback > 0 THEN
            p_precision := ROUND(
                (
                    v_search_count / 
                    v_total_searches_with_feedback
                ) * 100, 2);
        ELSE
            p_precision := 0;
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in get_precision_metrics.'
                || ' Error: ' || SQLERRM
            );
            RAISE;
    END get_drilldown_precision_metrics;

    PROCEDURE get_drilldown_within_report_metrics (
        p_records               IN JSON_ARRAY_T,
        k                       IN INTEGER,
        p_precision             OUT NUMBER
    ) AS
        v_total_searches                NUMBER := 0;
        v_search_count                  NUMBER := 0;
        v_record                        JSON_OBJECT_T;
        v_report_matches                JSON_ARRAY_T;
        v_report_match                  JSON_OBJECT_T;
    BEGIN
        IF p_records IS NULL THEN
            lang_data_logger_pkg.log_error('The records cannot be NULL');
            lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
              );
        END IF;
        FOR i in 0 .. p_records.get_size - 1 LOOP
            v_record := TREAT (
                p_records.get(i) AS JSON_OBJECT_T
            );

            IF v_record.GET_STRING('expected_report_id') IS NOT NULL AND 
                v_record.GET_STRING('expected_drilldown_id') IS NOT NULL THEN

                IF v_record.GET('report_matches') IS NULL THEN
                    CONTINUE;
                END IF;
                v_report_matches := JSON_ARRAY_T(
                    v_record.GET('report_matches')
                );

                -- Check for a matching report in report_matches JSON
                FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
                    -- Done as json_array_t.get returns json_element_t
                    -- which needs to be converted to json_object_t
                    v_report_match := TREAT (
                        v_report_matches.get(i) AS JSON_OBJECT_T
                    );

                    IF 
                        v_report_match.get_string('report_id') =
                            v_record.GET_STRING('expected_report_id') AND
                        v_report_match.get_NUMBER('overall_rank') <= k 
                    THEN
                        v_total_searches := v_total_searches + 1;

                        --Check if the drilldown matches the expected drilldown
                        IF v_report_match.get_string('drilldown_id') 
                            = v_record.GET_STRING('expected_drilldown_id') THEN
                            v_search_count := v_search_count + 1;
                        END IF;

                        EXIT;  -- Exit loop once a match is found
                    END IF;
                END LOOP;
            END IF;
        END LOOP;

        IF v_total_searches > 0 THEN
            p_precision := ROUND(
                (
                    v_search_count / 
                    v_total_searches
                ) * 100, 2);
        ELSE
            p_precision := 0;
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            -- Handle all exceptions in a proper block
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in get_precision_metrics.'
                || ' Error: ' || SQLERRM
            );
            RAISE;
    END get_drilldown_within_report_metrics;

    PROCEDURE get_filter_metrics(
        p_records               IN JSON_ARRAY_T,
        p_success_rate          OUT NUMBER,
        p_default_no_data       OUT NUMBER,
        p_default_fuzzy_fail    OUT NUMBER,
        p_default_no_entity     OUT NUMBER
    ) IS
        v_total_filter_count    NUMBER := 0;
        v_default_value_count   NUMBER := 0;
        v_default_no_data       NUMBER := 0;
        v_default_fuzzy_fail    NUMBER := 0;
        v_default_no_entity     NUMBER := 0;
        v_record                JSON_OBJECT_T;
        v_report_matches        JSON_ARRAY_T;
        v_report_match          JSON_OBJECT_T;
        v_filter_values_arr     JSON_ARRAY_T;
        v_filter_value_obj      JSON_OBJECT_T;
        v_filter_name           VARCHAR2(255);
        v_reason                VARCHAR2(4000);
    BEGIN
        p_success_rate := 0;

        IF p_records IS NULL THEN
            
            lang_data_logger_pkg.log_error('The records cannot be NULL');
            lang_data_errors_pkg.raise_error(
              lang_data_errors_pkg.c_invalid_parameters_code
              );
        END IF;

        FOR i in 0 .. p_records.get_size - 1 LOOP
            v_record := TREAT (
                p_records.get(i) AS JSON_OBJECT_T
            );
            IF v_record.GET_STRING('expected_report_id') IS NOT NULL THEN 
                IF v_record.GET('report_matches') IS NULL THEN
                    CONTINUE;
                END IF;
                v_report_matches := JSON_ARRAY_T(
                    v_record.GET('report_matches')
                ); 

                -- Check for a matching report in report_matches JSON
                FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
                    -- Done as json_array_t.get() returns json_element_t
                    -- which needs to be converted to json_object_t
                    v_report_match := TREAT (
                        v_report_matches.get(i) AS JSON_OBJECT_T
                    );

                    IF 
                        v_report_match.GET_STRING('report_id') =
                            v_record.GET_STRING('expected_report_id')
                    THEN
                        
                        IF v_report_match.GET('report_filter_values') IS NULL 
                        THEN
                            EXIT;
                        END IF;

                        v_filter_values_arr := JSON_ARRAY_T(
                            v_report_match.GET('report_filter_values')
                        );

                        FOR j IN 0 .. v_filter_values_arr.get_size - 1 LOOP

                            v_filter_value_obj := TREAT(
                                v_filter_values_arr.get(j) AS JSON_OBJECT_T
                            );
                            
                            v_filter_name := v_filter_value_obj.get_string(
                                                'filter_name'
                                            );
                            v_reason := v_filter_value_obj.get_string('reason');
                                
                            IF v_reason IS NOT NULL THEN
                                IF REGEXP_LIKE(v_reason, 
                                               '^Using default value.*') THEN
                                    v_default_value_count := 
                                            v_default_value_count + 1;
                                END IF;

                                IF REGEXP_LIKE(v_reason, 
                                    '^Using default value as the value vector table.*') OR

                                    REGEXP_LIKE(v_reason, 
                                    '^Using default value as no data found in the value vector table.*')

                                    THEN
                                    v_default_no_data := 
                                            v_default_no_data + 1;
                                END IF;

                                IF v_reason = 'Using default value as' || 
                                ' none of the top 3 results passed fuzzy' ||
                                ' matching post validation.' OR 

                                  v_reason = 'Using default value as top 3 ' ||
                                  'results failed  fuzzy matching post ' ||
                                  'validation.' THEN
                                  
                                    v_default_fuzzy_fail := 
                                            v_default_fuzzy_fail + 1;
                                END IF;

                                IF REGEXP_LIKE(v_reason, 
                                    '^Using default value, as entity type *') 
                                    THEN
                                    v_default_no_entity := 
                                            v_default_no_entity + 1;
                                END IF;

                            END IF;

                            v_total_filter_count := v_total_filter_count +1;
                        END LOOP;
                        
                        EXIT;  -- Exit loop once a match is found
                    END IF;
                END LOOP;
            END IF;
        END LOOP;

        IF v_total_filter_count > 0 THEN
            p_success_rate := ROUND(
                (
                    (v_total_filter_count - v_default_value_count) / 
                    v_total_filter_count
                ) * 100, 2);
        ELSE
            p_success_rate := 0;
        END IF;

        IF v_default_value_count > 0 THEN
            p_default_no_data := ROUND(
                (
                    v_default_no_data / 
                    v_default_value_count
                ) * 100, 2);

            p_default_fuzzy_fail := ROUND(
                (
                    v_default_fuzzy_fail / 
                    v_default_value_count
                ) * 100, 2);

            p_default_no_entity := ROUND(
                (
                    v_default_no_entity / 
                    v_default_value_count
                ) * 100, 2);

        ELSE
            p_default_no_data := 0;
            p_default_fuzzy_fail := 0;
            p_default_no_entity := 0;
        END IF;

    END get_filter_metrics;

    PROCEDURE update_api_logs (
        p_api_name          IN VARCHAR2, 
        p_start_time        IN NUMBER,  
        p_end_time          IN NUMBER,
        p_failed            IN BOOLEAN DEFAULT FALSE,
        p_failure_code      IN NUMBER DEFAULT NULL,
        p_failure_message   IN VARCHAR2 DEFAULT NULL
    ) IS
        PRAGMA AUTONOMOUS_TRANSACTION;
        v_api_time          NUMBER;
        v_log_id            VARCHAR2(36);
    BEGIN
        -- Generate the Log ID for this external API call.
        v_log_id := lang_data_utils_pkg.generate_id();

        -- Calculate the time taken for the procedure execution in centiseconds
        v_api_time := p_end_time - p_start_time;   

        INSERT INTO langdata$api_logs (
            id, api_name, called_at, call_time, failure_status,
            failure_code, failure_message
        )
        VALUES (
            v_log_id, p_api_name, SYSTIMESTAMP, v_api_time, p_failed, 
            p_failure_code, p_failure_message
        );

        COMMIT;
    END update_api_logs;

    FUNCTION get_all_api_stats (
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB 
    IS
        v_result CLOB;
    BEGIN
        SELECT JSON_ARRAYAGG(
            JSON_OBJECT(
                'api_name'           VALUE api_name,
                'call_count'         VALUE COUNT(*),
                'avg_call_time'      VALUE ROUND(AVG(
                                        CASE WHEN failure_status = FALSE 
                                        THEN call_time END), 2),
                'last_call_duration' VALUE MAX(call_time) KEEP
                                        (DENSE_RANK FIRST ORDER BY called_at 
                                            DESC),
                'last_called_at'     VALUE TO_CHAR(
                                        MAX(called_at),
                                        'YYYY-MM-DD"T"HH24:MI:SS'),
                'fail_count'         VALUE COUNT(
                                        CASE WHEN failure_status = TRUE
                                        THEN 1 END)
            ) 
            RETURNING CLOB
        )
        INTO v_result
        FROM langdata$api_logs
        WHERE p_timestamp IS NULL OR called_at >= p_timestamp
        GROUP BY api_name
        ORDER BY api_name;

        RETURN v_result;
    END get_all_api_stats;

    FUNCTION get_api_daily_stats (
        p_api_name           IN VARCHAR2,
        p_num_error_messages IN NUMBER DEFAULT 5,
        p_timestamp          IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS 
        v_result CLOB;
    BEGIN
        SELECT JSON_ARRAYAGG(
            JSON_OBJECT(
                'date' VALUE TO_CHAR(called_date, 'YYYY-MM-DD'),
                'call_count' VALUE call_count,
                'avg_call_time' VALUE ROUND(avg_call_time, 2),
                'success_rate' VALUE success_rate,
                'latest_failure_messages' VALUE (
                    SELECT JSON_ARRAYAGG(
                        JSON_OBJECT(
                            'failure_message' VALUE failure_message,
                            'failure_date' VALUE TO_CHAR(
                                called_at, 'YYYY-MM-DD"T"HH24:MI:SS')
                        )
                        RETURNING CLOB
                    )
                    FROM (
                        SELECT failure_message, al2.called_at
                        FROM langdata$api_logs al2
                        WHERE al2.failure_status = TRUE
                            AND al2.api_name = p_api_name
                            AND (p_timestamp IS NULL
                                    OR al2.called_at >= p_timestamp
                                )      
                            AND TRUNC(al2.called_at) = failure_stats.called_date
                        ORDER BY al2.called_at DESC
                        FETCH FIRST p_num_error_messages ROWS ONLY
                    )
                )
                RETURNING CLOB
            )
            RETURNING CLOB
        )
        INTO v_result
        FROM (
            SELECT
                TRUNC(called_at) AS called_date,
                COUNT(*) AS call_count,
                AVG(CASE WHEN failure_status = FALSE THEN call_time END) 
                    AS avg_call_time,
                (100 * ( COUNT(CASE WHEN failure_status = FALSE THEN 1 END) 
                    / NULLIF(COUNT(*), 0) )
                ) AS success_rate
            FROM langdata$api_logs
            WHERE
                ( p_timestamp IS NULL
                    OR called_at >= p_timestamp
                )
                AND api_name = p_api_name   
            GROUP BY TRUNC(called_at)
            ORDER BY TRUNC(called_at)
        ) failure_stats;

        RETURN v_result;
    END get_api_daily_stats;

    FUNCTION get_api_failures (
        p_api_name  IN VARCHAR2,
        p_timestamp IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB 
    IS
        v_result CLOB;
    BEGIN
        SELECT JSON_ARRAYAGG(
            JSON_OBJECT(
                'failure_message'       VALUE failure_message,
                'call_count'            VALUE COUNT(*),
                'last_called_at'        VALUE TO_CHAR(
                                            MAX(called_at),
                                            'YYYY-MM-DD"T"HH24:MI:SS'),
                'avg_call_time'         VALUE ROUND(AVG(call_time), 2)
            )
            RETURNING CLOB
        )
        INTO v_result
        FROM langdata$api_logs
        WHERE ( p_timestamp IS NULL OR called_at >= p_timestamp )
            AND failure_status = TRUE
            AND api_name = p_api_name
        GROUP BY failure_message
        ORDER BY failure_message;

        RETURN v_result;
    END get_api_failures;
    
    PROCEDURE get_total_positive_feedback(
        p_timestamp IN TIMESTAMP DEFAULT NULL,
        p_total_positive_feedback     OUT NUMBER
    ) IS
    
    BEGIN
        SELECT COUNT(*)
        INTO p_total_positive_feedback
        FROM langdata$searchrecords
        WHERE feedback_rating = TRUE
            AND ( p_timestamp is NULL OR updated_at >= p_timestamp );
    END get_total_positive_feedback;

    PROCEDURE get_total_negative_feedback(
        p_timestamp IN TIMESTAMP DEFAULT NULL,
        p_total_negative_feedback     OUT NUMBER
    ) IS

    BEGIN
        SELECT COUNT(*)
        INTO p_total_negative_feedback
        FROM langdata$searchrecords
        WHERE feedback_rating = FALSE
            AND ( p_timestamp is NULL OR updated_at >= p_timestamp );
    END get_total_negative_feedback;

    PROCEDURE get_total_no_feedback(
        p_timestamp IN TIMESTAMP DEFAULT NULL,
        p_total_no_feedback     OUT NUMBER
    ) IS 
    BEGIN
        SELECT COUNT(*)
        INTO p_total_no_feedback
        FROM langdata$searchrecords
        WHERE feedback_rating IS NULL
            AND ( p_timestamp is NULL OR updated_at >= p_timestamp );
    END get_total_no_feedback;

    PROCEDURE get_top_report_metrics(
        p_report_id     IN VARCHAR2,
        p_metrics       OUT JSON_OBJECT_T
    ) IS
        v_analytics_data  JSON_OBJECT_T;
        v_top1            NUMBER := 0;
        v_top3            NUMBER := 0;
        v_top5            NUMBER := 0;
        v_json_text       CLOB;
    BEGIN
        
        lang_data_logger_pkg.log_info('Top report metrics for Report ID: ' || 
                                        p_report_id);

        SELECT analytics_data
        INTO   v_json_text
        FROM   langdata$reports
        WHERE  id = p_report_id;

        v_analytics_data := JSON_OBJECT_T.parse(v_json_text);

        v_top1 := NVL(v_analytics_data.get_number('top1_count'), 0);
        v_top3 := NVL(v_analytics_data.get_number('top3_count'), 0);
        v_top5 := NVL(v_analytics_data.get_number('top5_count'), 0);

        p_metrics := JSON_OBJECT_T(
            JSON_OBJECT(
                'top1_count' VALUE v_top1,
                'top3_count' VALUE v_top3,
                'top5_count' VALUE v_top5
            )
        );
        lang_data_logger_pkg.log_info(
        'Parsed metrics - top1_count=' || v_top1 || ', top3_count=' || v_top3|| 
        ', top5_count=' || v_top5);

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            p_metrics := JSON_OBJECT_T(
                JSON_OBJECT(
                    'top1_count' VALUE 0,
                    'top3_count' VALUE 0,
                    'top5_count' VALUE 0
                )
            );
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred. Error: ' || SQLERRM
            );
            RAISE;
    END get_top_report_metrics;

    PROCEDURE get_top_drilldown_metrics(
        p_drilldown_id  IN VARCHAR2,
        p_metrics       OUT JSON_OBJECT_T
    ) IS
        v_analytics_data  JSON_OBJECT_T;
        v_top1            NUMBER := 0;
        v_top3            NUMBER := 0;
        v_top5            NUMBER := 0;
        v_json_text       CLOB;
    BEGIN

        lang_data_logger_pkg.log_info('Top drilldown metrics for Report ID: '|| 
                                        p_drilldown_id);

        SELECT analytics_data
        INTO   v_json_text
        FROM   langdata$drilldowndocuments
        WHERE  id = p_drilldown_id;

        v_analytics_data := JSON_OBJECT_T.parse(v_json_text);

        v_top1 := NVL(v_analytics_data.get_number('top1_count'), 0);
        v_top3 := NVL(v_analytics_data.get_number('top3_count'), 0);
        v_top5 := NVL(v_analytics_data.get_number('top5_count'), 0);

        p_metrics := JSON_OBJECT_T(
            JSON_OBJECT(
                'top1_count' VALUE v_top1,
                'top3_count' VALUE v_top3,
                'top5_count' VALUE v_top5
            )
        );
        lang_data_logger_pkg.log_info(
        'Parsed metrics - top1_count=' || v_top1 || ', top3_count=' || v_top3|| 
        ', top5_count=' || v_top5);

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            p_metrics := JSON_OBJECT_T(
                JSON_OBJECT(
                    'top1_count' VALUE 0,
                    'top3_count' VALUE 0,
                    'top5_count' VALUE 0
                )
            );
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred. Error: ' || SQLERRM
            );
            RAISE;
    END get_top_drilldown_metrics;

    FUNCTION get_top_k_most_searched_reports (
        p_k             IN NUMBER,
        p_timestamp     IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_result CLOB;
    BEGIN
        IF p_timestamp is NULL THEN
            SELECT JSON_ARRAYAGG(
                    JSON_OBJECT(
                        'id'           VALUE id,
                        'report_title' VALUE title,
                        'search_count' VALUE 
                            JSON_VALUE(analytics_data,'$.search_count'
                                RETURNING NUMBER)
                    )
                    RETURNING CLOB
                )
            INTO v_result
            FROM (
                SELECT id, title, analytics_data
                FROM langdata$reports
                WHERE JSON_VALUE(analytics_data, '$.search_count'
                    RETURNING NUMBER) IS NOT NULL
                ORDER BY JSON_VALUE(analytics_data,'$.search_count'
                            RETURNING NUMBER) DESC
                FETCH FIRST p_k ROWS ONLY
            );
        ELSE
            SELECT JSON_ARRAYAGG(
                    JSON_OBJECT(
                        'id'           VALUE report_id,
                        'report_title' VALUE report_title,
                        'search_count' VALUE search_count
                    )
                    RETURNING CLOB
                )
            INTO v_result
            FROM (
                SELECT
                    matched_reports.report_id,
                    matched_reports.report_title,
                    COUNT(*) AS search_count
                FROM langdata$searchrecords search_records,
                    JSON_TABLE(
                        search_records.report_matches,
                        '$[*]'
                        COLUMNS (
                            report_id     VARCHAR2(36)    PATH '$.report_id',
                            report_title  VARCHAR2(4000)  PATH '$.report_title'
                        )
                    ) matched_reports
                WHERE search_records.updated_at >= p_timestamp
                GROUP BY matched_reports.report_id, matched_reports.report_title
                ORDER BY search_count DESC
                FETCH FIRST p_k ROWS ONLY
            );
        END IF;
        RETURN v_result;
    END get_top_k_most_searched_reports;

    FUNCTION get_bottom_k_least_searched_reports (
        p_k IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        v_result CLOB;
    BEGIN
        IF p_timestamp is NULL THEN
            SELECT JSON_ARRAYAGG(
                    JSON_OBJECT(
                        'id'           VALUE id,
                        'report_title' VALUE title,
                        'search_count' VALUE 
                            JSON_VALUE(analytics_data,'$.search_count'
                                RETURNING NUMBER)
                    )
                    RETURNING CLOB
                )
            INTO v_result
            FROM (
                SELECT id, title, analytics_data
                FROM langdata$reports
                WHERE JSON_VALUE(analytics_data, '$.search_count'
                    RETURNING NUMBER) IS NOT NULL
                ORDER BY JSON_VALUE(analytics_data,'$.search_count'
                            RETURNING NUMBER) 
                FETCH FIRST p_k ROWS ONLY
            );
        ELSE
            SELECT JSON_ARRAYAGG(
                    JSON_OBJECT(
                        'id'           VALUE report_id,
                        'report_title' VALUE report_title,
                        'search_count' VALUE search_count
                    )
                    RETURNING CLOB
                )
            INTO v_result
            FROM (
                SELECT
                    matched_reports.report_id,
                    matched_reports.report_title,
                    COUNT(*) AS search_count
                FROM langdata$searchrecords search_records,
                    JSON_TABLE(
                        search_records.report_matches,
                        '$[*]'
                        COLUMNS (
                            report_id     VARCHAR2(36)    PATH '$.report_id',
                            report_title  VARCHAR2(4000)  PATH '$.report_title'
                        )
                    ) matched_reports
                WHERE search_records.updated_at >= p_timestamp
                GROUP BY matched_reports.report_id, matched_reports.report_title
                ORDER BY search_count
                FETCH FIRST p_k ROWS ONLY
            );
        END IF;
        RETURN v_result;

    END get_bottom_k_least_searched_reports;

    FUNCTION get_top_k_most_asked_questions(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS 
        v_result CLOB;
    BEGIN
        IF p_timestamp IS NULL THEN
            SELECT JSON_ARRAYAGG(
                JSON_OBJECT(
                    'question_id'   VALUE question_id,
                    'question_text' VALUE question_text,
                    'asked_count'   VALUE asked_count
                )
                RETURNING CLOB
            )
            INTO v_result
            FROM (
                SELECT question_id, question_text, asked_count
                FROM langdata$question_stats
                ORDER BY asked_count DESC
                FETCH FIRST p_k ROWS ONLY
            );
        ELSE
            SELECT JSON_ARRAYAGG(
                JSON_OBJECT(
                    'question_id'   VALUE question_id,
                    'question_text' VALUE question_text,
                    'asked_count'   VALUE ask_count
                )
                RETURNING CLOB
            )
            INTO v_result
            FROM (
                SELECT qs.question_id, qs.question_text, COUNT(*) AS ask_count
                FROM langdata$searchrecords sr
                JOIN langdata$question_stats qs
                    ON sr.question_id = qs.question_id
                WHERE sr.created_at >= p_timestamp
                GROUP BY qs.question_id, qs.question_text
                ORDER BY ask_count DESC
                FETCH FIRST p_k ROWS ONLY
            );
        END IF;

        RETURN v_result;
    END get_top_k_most_asked_questions;

    FUNCTION get_top_k_most_negative_feedback_questions(
        p_k                     IN NUMBER,
        p_timestamp             IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS 
        v_result CLOB;
    BEGIN
        SELECT JSON_ARRAYAGG(
            JSON_OBJECT(
                'question_id'   VALUE question_id,
                'question_text' VALUE question_text,
                'asked_count'   VALUE ask_count
            )
            RETURNING CLOB
        )
        INTO v_result
        FROM (
            SELECT qs.question_id, qs.question_text, COUNT(*) AS ask_count
            FROM langdata$searchrecords sr
            JOIN langdata$question_stats qs
                ON sr.question_id = qs.question_id
            WHERE ( p_timestamp IS NULL  OR sr.created_at >= p_timestamp )
                AND sr.feedback_rating = FALSE
            GROUP BY qs.question_id, qs.question_text
            ORDER BY ask_count DESC
            FETCH FIRST p_k ROWS ONLY
        );

        RETURN v_result;
    END get_top_k_most_negative_feedback_questions;

    PROCEDURE get_metrics(
        p_timestamp IN TIMESTAMP DEFAULT NULL,
        p_metrics               OUT JSON_OBJECT_T
    ) IS
        -- Variables to store results
        v_success_rate              NUMBER := 0;
        v_precision_at_1            NUMBER := 0;
        v_precision_at_3            NUMBER := 0;
        v_precision_drilldown       NUMBER := 0;
        v_precision_within_report   NUMBER := 0;
        v_filter_success_rate       NUMBER := 0;
        v_default_no_data           NUMBER := 0;
        v_default_fuzzy_fail        NUMBER := 0;
        v_default_no_entity         NUMBER := 0;
        v_mean_reciprocal_rank      NUMBER := 0;
        v_positive_feedback_percentage  NUMBER := 0;
        v_cursor                    VARCHAR2(20) := NULL;
        v_records                   SYS_REFCURSOR;
        v_total_negative_feedback   NUMBER;
        v_total_positive_feedback   NUMBER;
        v_total_no_feedback         NUMBER;

        -- Variables to hold fields in a row
        r_feedback_rating               BOOLEAN;
        r_feedback_rating_num           NUMBER(1);  -- 1 = TRUE, 0 = FALSE, 
                                                    -- NULL = NULL
        r_feedback_comments             VARCHAR2(2000);
        r_expected_report_id            VARCHAR2(36);
        r_expected_drilldown_id         VARCHAR2(36);
        r_report_matches                JSON;
        r_cur                           NUMBER;

        v_search_records                JSON_ARRAY_T := JSON_ARRAY_T();
        v_record                        JSON_OBJECT_T;
    BEGIN
        lang_data_search_pkg.get_all_user_search_records(
            p_feedback_rating           => NULL,
            p_required_feedback_action  => NULL,
            p_feedback_action_priority  => NULL,
            p_cursor                    => v_cursor,
            p_limit                     => NULL,
            p_records                   => v_records
        );

        r_cur := DBMS_SQL.TO_CURSOR_NUMBER(v_records);
        DBMS_SQL.DEFINE_COLUMN(r_cur, 3, r_report_matches);
        DBMS_SQL.DEFINE_COLUMN(r_cur, 4, r_feedback_rating_num);
        DBMS_SQL.DEFINE_COLUMN(r_cur, 5, r_feedback_comments, 2000);
        DBMS_SQL.DEFINE_COLUMN(r_cur, 8, r_expected_report_id, 36);
        DBMS_SQL.DEFINE_COLUMN(r_cur, 9, r_expected_drilldown_id, 36);

        WHILE DBMS_SQL.FETCH_ROWS(r_cur) > 0 LOOP

            DBMS_SQL.COLUMN_VALUE(r_cur, 3, r_report_matches);
            DBMS_SQL.COLUMN_VALUE(r_cur, 4, r_feedback_rating_num);
            DBMS_SQL.COLUMN_VALUE(r_cur, 5, r_feedback_comments);
            DBMS_SQL.COLUMN_VALUE(r_cur, 8, r_expected_report_id);
            DBMS_SQL.COLUMN_VALUE(r_cur, 9, r_expected_drilldown_id);

            IF r_feedback_rating_num IS NULL THEN
                r_feedback_rating := NULL;
            ELSIF r_feedback_rating_num = 1 THEN
                r_feedback_rating := TRUE;
            ELSE
                r_feedback_rating := FALSE;
            END IF;

            v_record := JSON_OBJECT_T(
                JSON_OBJECT(
                    'report_matches' VALUE r_report_matches,
                    'feedback_rating' VALUE r_feedback_rating,
                    'feedback_comments' VALUE r_feedback_comments,
                    'expected_report_id' VALUE r_expected_report_id,
                    'expected_drilldown_id' VALUE r_expected_drilldown_id
                )
            );
            v_search_records.append(v_record);

        END LOOP;

        IF DBMS_SQL.IS_OPEN(r_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(r_cur);
        END IF;

        lang_data_analytics_pkg.get_success_rate_metrics(
            p_records       => v_search_records,
            p_success_rate  => v_success_rate
        );

        lang_data_analytics_pkg.get_precision_metrics(
            p_records       => v_search_records,
            K               => 1,
            p_precision     => v_precision_at_1
        );

        lang_data_analytics_pkg.get_precision_metrics(
            p_records       => v_search_records,
            K               => 3,
            p_precision     => v_precision_at_3
        );

        lang_data_analytics_pkg.get_mean_reciprocal_rank(
            p_records               => v_search_records,
            p_mean_reciprocal_rank  => v_mean_reciprocal_rank
        );

        lang_data_analytics_pkg.get_positive_feedback_percentage(
            p_records           => v_search_records,
            p_positive_feedback_percentage  => v_positive_feedback_percentage
        );
        
        lang_data_analytics_pkg.get_drilldown_within_report_metrics (
            p_records       => v_search_records,
            K               => lang_data_config_pkg.get_config_parameter(
                                    'LANG_DATA_DRILLDOWN_SIMILARITY_K'
                                ),
            p_precision     => v_precision_within_report
        );

        lang_data_analytics_pkg.get_drilldown_precision_metrics (
            p_records       => v_search_records,
            K               => lang_data_config_pkg.get_config_parameter(
                                    'LANG_DATA_DRILLDOWN_SIMILARITY_K'
                                ),
            p_precision     => v_precision_drilldown
        );

        lang_data_analytics_pkg.get_filter_metrics (
            p_records       => v_search_records,
            p_success_rate  => v_filter_success_rate,
            p_default_no_data => v_default_no_data,
            p_default_fuzzy_fail => v_default_fuzzy_fail,
            p_default_no_entity => v_default_no_entity
        );
                
        lang_data_analytics_pkg.get_total_positive_feedback(
            p_timestamp => p_timestamp,
            p_total_positive_feedback => v_total_positive_feedback
            
        );

        lang_data_analytics_pkg.get_total_negative_feedback(
            p_timestamp => p_timestamp,
            p_total_negative_feedback => v_total_negative_feedback

        );

        lang_data_analytics_pkg.get_total_no_feedback(
            p_timestamp => p_timestamp,
            p_total_no_feedback => v_total_no_feedback

        );

        p_metrics := JSON_OBJECT_T(
            JSON_OBJECT(
                'success_rate'              VALUE v_success_rate,
                'precision_at_1'            VALUE v_precision_at_1,
                'precision_at_3'            VALUE v_precision_at_3,
                'mean_reciprocal_rank'      VALUE v_mean_reciprocal_rank,
                'positive_feedback_percentage'   
                                           VALUE v_positive_feedback_percentage,
                'precision_drilldown'       VALUE v_precision_drilldown,
                'precision_drilldown_within_report'  
                                            VALUE v_precision_within_report,
                'filter_success_rate'       VALUE v_filter_success_rate,
                'filter_failure_no_data'    VALUE v_default_no_data,
                'filter_failure_no_fuzzy'   VALUE v_default_fuzzy_fail,
                'filter_failure_no_entity'  VALUE v_default_no_entity,
                'total_positive_feedback'   VALUE v_total_positive_feedback,
                'total_negative_feedback'   VALUE v_total_negative_feedback,
                'total_no_feedback'         VALUE v_total_no_feedback
            )
        );

    EXCEPTION
            WHEN OTHERS THEN
                lang_data_logger_pkg.log_fatal(
                    'An unknown error occurred. Error: ' || SQLERRM
                );
                RAISE;
    END get_metrics;

    FUNCTION get_top_filters(
        p_top_k         IN NUMBER,
        p_timestamp     IN TIMESTAMP DEFAULT NULL
    ) RETURN CLOB
    IS
        p_top_filters          JSON_OBJECT_T;

        v_cursor               SYS_REFCURSOR;
        v_report_matches_clob  CLOB;

        -- JSON parsing variables
        v_report_matches       JSON_ARRAY_T;
        v_report_match         JSON_OBJECT_T;
        v_filter_values_obj    JSON_OBJECT_T;
        v_filters_arr          JSON_ARRAY_T;
        v_filter_obj           JSON_OBJECT_T;

        -- Filter counting
        v_filter_name                VARCHAR2(200);
        v_count                      INTEGER;
        v_ner_filter_counts          JSON_OBJECT_T := JSON_OBJECT_T();
        v_enumerated_filter_counts   JSON_OBJECT_T := JSON_OBJECT_T();
        v_keys                       JSON_KEY_LIST;

        -- Sorting
        v_ner_unsorted         filter_tab_type := filter_tab_type();
        v_enum_unsorted        filter_tab_type := filter_tab_type();
        v_ner_sorted           filter_tab_type;
        v_enum_sorted          filter_tab_type;

        -- Result JSON
        v_top_ner_filters           JSON_ARRAY_T := JSON_ARRAY_T();
        v_top_enumerated_filters   JSON_ARRAY_T := JSON_ARRAY_T();
    BEGIN
        -- Step 1: Fetch and parse data
        OPEN v_cursor FOR
            SELECT report_matches
            FROM langdata$searchrecords
            WHERE (p_timestamp IS NULL OR created_at >= p_timestamp);

        LOOP
            FETCH v_cursor INTO v_report_matches_clob;
            EXIT WHEN v_cursor%NOTFOUND;

            BEGIN
                v_report_matches := JSON_ARRAY_T.parse(v_report_matches_clob);
            EXCEPTION
                WHEN OTHERS THEN CONTINUE;
            END;

            -- TODO: Optimze this (DBAI-1078)
            FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
                BEGIN
                    v_report_match := 
                        TREAT(v_report_matches.get(i) AS JSON_OBJECT_T);
                    v_filter_values_obj := JSON_OBJECT_T(
                            v_report_match.get('report_match_document'));

                    IF v_filter_values_obj.has('filters') THEN
                        v_filters_arr := TREAT(
                            v_filter_values_obj.get('filters') AS JSON_ARRAY_T);

                        FOR j IN 0 .. v_filters_arr.get_size - 1 LOOP
                            v_filter_obj := TREAT(
                                v_filters_arr.get(j) AS JSON_OBJECT_T);
                            v_filter_name := 
                                v_filter_obj.get_string('filter_name');

                            IF v_filter_obj.get_boolean('use_ner') THEN
                                IF v_ner_filter_counts.has(v_filter_name) THEN
                                    v_count := v_ner_filter_counts.get_number(
                                        v_filter_name);
                                    v_ner_filter_counts
                                        .put(v_filter_name, v_count + 1);
                                ELSE
                                    v_ner_filter_counts.put(v_filter_name, 1);
                                END IF;
                            ELSE
                                IF v_enumerated_filter_counts.has(v_filter_name) 
                                THEN
                                    v_count := v_enumerated_filter_counts
                                        .get_number(v_filter_name);
                                    v_enumerated_filter_counts
                                        .put(v_filter_name, v_count + 1);
                                ELSE
                                    v_enumerated_filter_counts
                                        .put(v_filter_name, 1);
                                END IF;
                            END IF;
                        END LOOP;
                    END IF;

                EXCEPTION
                    WHEN OTHERS THEN NULL;
                END;
            END LOOP;
        END LOOP;

        CLOSE v_cursor;

        -- Step 2: Convert JSON_OBJECT_Ts to sortable collections

        -- NER filters
        v_keys := v_ner_filter_counts.get_keys;
        FOR i IN 1 .. v_keys.count LOOP
            v_filter_name := v_keys(i);
            v_count := v_ner_filter_counts.get_number(v_filter_name);
            v_ner_unsorted.EXTEND;
            v_ner_unsorted(v_ner_unsorted.COUNT) := 
                filter_rec_type(v_filter_name, v_count);
        END LOOP;

        -- Enumerated (non-NER) filters
        v_keys := v_enumerated_filter_counts.get_keys;
        FOR i IN 1 .. v_keys.count LOOP
            v_filter_name := v_keys(i);
            v_count := v_enumerated_filter_counts.get_number(v_filter_name);
            v_enum_unsorted.EXTEND;
            v_enum_unsorted(v_enum_unsorted.COUNT) := 
                filter_rec_type(v_filter_name, v_count);
        END LOOP;

        -- Step 3: Sort both by count DESC
        SELECT VALUE(t) BULK COLLECT INTO v_ner_sorted
        FROM TABLE(v_ner_unsorted) t
        ORDER BY t.count DESC;

        SELECT VALUE(t) BULK COLLECT INTO v_enum_sorted
        FROM TABLE(v_enum_unsorted) t
        ORDER BY t.count DESC;

        -- Step 4: Build top-K JSON arrays for both
        FOR i IN 1 .. LEAST(p_top_k, v_ner_sorted.COUNT) LOOP
            DECLARE v_obj JSON_OBJECT_T;
            BEGIN
                v_obj := JSON_OBJECT_T();
                v_obj.put('filter_name', v_ner_sorted(i).name);
                v_obj.put('count', v_ner_sorted(i).count);
                v_top_ner_filters.append(v_obj);
            END;
        END LOOP;

        FOR i IN 1 .. LEAST(p_top_k, v_enum_sorted.COUNT) LOOP
            DECLARE v_obj JSON_OBJECT_T;
            BEGIN
                v_obj := JSON_OBJECT_T();
                v_obj.put('filter_name', v_enum_sorted(i).name);
                v_obj.put('count', v_enum_sorted(i).count);
                v_top_enumerated_filters.append(v_obj);
            END;
        END LOOP;

        -- Step 5: Final output
        p_top_filters := JSON_OBJECT_T();
        p_top_filters.put('top_ner_filters', v_top_ner_filters);
        p_top_filters.put('top_enumerated_filters', v_top_enumerated_filters);

        return p_top_filters.to_clob;
    END get_top_filters;


    PROCEDURE get_matched_queries_by_report_rank( 
        p_report_id IN VARCHAR2,
        p_rank      IN PLS_INTEGER,        -- 1-based
        p_k         IN PLS_INTEGER,
        p_queries   OUT SYS.ODCIVARCHAR2LIST) IS
    
        v_cur      SYS_REFCURSOR;
        v_sql      CLOB;
        v_q        VARCHAR2(4000);
        v_rank_ix  PLS_INTEGER := p_rank - 1;   -- 0-based for the JSON path
    BEGIN
        p_queries := SYS.ODCIVARCHAR2LIST();

        v_sql :=
            'SELECT query_text
            FROM langdata$searchrecords
            WHERE JSON_VALUE(report_matches,
                            ''$['||v_rank_ix||'].report_id'') = :id';

        OPEN v_cur FOR v_sql USING p_report_id;

        LOOP
            FETCH v_cur INTO v_q;
            EXIT WHEN v_cur%NOTFOUND OR p_queries.COUNT >= p_k;

            p_queries.EXTEND;
            p_queries(p_queries.COUNT) := v_q;
        END LOOP;

        CLOSE v_cur;
    END get_matched_queries_by_report_rank;


    PROCEDURE get_report_filter_usage(
        p_report_id        IN  VARCHAR2,
        p_default_count    OUT NUMBER,
        p_non_default_count OUT NUMBER
    ) IS
        v_analytics_data  JSON_OBJECT_T;
        v_json_text       CLOB;
        v_report_exists   NUMBER;
    BEGIN
        p_default_count := 0;
        p_non_default_count := 0;

        SELECT COUNT(*) 
        INTO v_report_exists 
        FROM langdata$reports 
        WHERE id = p_report_id;

        IF v_report_exists = 0 THEN
            lang_data_logger_pkg.log_error(
                'report id: ' || p_report_id || ' does not exist '
                );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
                );
        END IF;

        SELECT analytics_data
        INTO   v_json_text
        FROM   langdata$reports
        WHERE  id = p_report_id;

        v_analytics_data := JSON_OBJECT_T.parse(v_json_text);
        p_non_default_count := 
                NVL(v_analytics_data.get_number('non_default_filter_count'), 0);
        p_default_count := 
                NVL(v_analytics_data.get_number('default_filter_count'), 0);

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            p_default_count := 0;
            p_non_default_count := 0;
            IF v_report_exists = 0 THEN
                lang_data_logger_pkg.log_error(
                    'Report ID: ' || p_report_id || ' does not exist.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_parameters_code
                );
                RAISE;
            END IF;

        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in get_precision_metrics.'
                || ' Error: ' || SQLERRM
            );
            RAISE;

    END get_report_filter_usage;   

    PROCEDURE get_matched_queries_by_drilldown_rank( 
        p_drilldown_id IN VARCHAR2,
        p_rank      IN PLS_INTEGER,        -- 1-based
        p_k         IN PLS_INTEGER,
        p_queries   OUT SYS.ODCIVARCHAR2LIST) IS
    
        v_cur      SYS_REFCURSOR;
        v_sql      CLOB;
        v_q        VARCHAR2(4000);
        v_rank_ix  PLS_INTEGER := p_rank - 1;   -- 0-based for the JSON path
    BEGIN
        p_queries := SYS.ODCIVARCHAR2LIST();

        v_sql :=
            'SELECT query_text
            FROM langdata$searchrecords
            WHERE JSON_VALUE(report_matches,
                            ''$['||v_rank_ix||'].drilldown_id'') = :id';

        OPEN v_cur FOR v_sql USING p_drilldown_id;

        LOOP
            FETCH v_cur INTO v_q;
            EXIT WHEN v_cur%NOTFOUND OR p_queries.COUNT >= p_k;

            p_queries.EXTEND;
            p_queries(p_queries.COUNT) := v_q;
        END LOOP;

        CLOSE v_cur;
    END get_matched_queries_by_drilldown_rank;


    PROCEDURE get_drilldown_filter_usage(
        p_drilldown_id        IN  VARCHAR2,
        p_default_count    OUT NUMBER,
        p_non_default_count OUT NUMBER
    ) IS
        v_analytics_data  JSON_OBJECT_T;
        v_json_text       CLOB;
        v_drilldown_exists   NUMBER;
    BEGIN
        p_default_count := 0;
        p_non_default_count := 0;

        SELECT COUNT(*) 
        INTO v_drilldown_exists 
        FROM langdata$drilldowndocuments  
        WHERE id = p_drilldown_id;

        IF v_drilldown_exists = 0 THEN
            lang_data_logger_pkg.log_error(
                'drilldown id: ' || p_drilldown_id || ' does not exist '
                );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
                );
        END IF;

        SELECT analytics_data
        INTO   v_json_text
        FROM   langdata$drilldowndocuments  
        WHERE  id = p_drilldown_id;

        v_analytics_data := JSON_OBJECT_T.parse(v_json_text);
        p_non_default_count := 
                NVL(v_analytics_data.get_number('non_default_filter_count'), 0);
        p_default_count := 
                NVL(v_analytics_data.get_number('default_filter_count'), 0);

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            p_default_count := 0;
            p_non_default_count := 0;
            IF v_drilldown_exists = 0 THEN
                lang_data_logger_pkg.log_error(
                    'Drilldown ID: ' || p_drilldown_id || ' does not exist.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_parameters_code
                );
                RAISE;
            END IF;

        WHEN OTHERS THEN
            IF SQLCODE = lang_data_errors_pkg.c_invalid_parameters_code THEN
                RAISE;
            END IF;
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred in get_precision_metrics.'
                || ' Error: ' || SQLERRM
            );
            RAISE;

    END get_drilldown_filter_usage;

END lang_data_analytics_pkg;
/

