Rem
Rem $Header: dbgendev/src/langdata/plsql/search/search_pkg.pkb /main/48 2025/08/17 19:34:27 deveverm Exp $
Rem
Rem search_pkg.pkb
Rem
Rem Copyright (c) 2024, 2025, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      search_pkg.pkb - Package body for handling search logic and
Rem                       user search records
Rem
Rem    DESCRIPTION
Rem      Package for handling user queries and managing search records, 
Rem      including retrieval, deletion, and pagination, with role-based access 
Rem      control, error handling, and logging.
Rem
Rem    NOTES
Rem      None
Rem
Rem    BEGIN SQL_FILE_METADATA
Rem    SQL_SOURCE_FILE: dbgendev/src/langdata/plsql/search/search_pkg.pkb
Rem    SQL_SHIPPED_FILE:
Rem    SQL_PHASE:
Rem    SQL_STARTUP_MODE: NORMAL
Rem    SQL_IGNORABLE_ERRORS: NONE
Rem    END SQL_FILE_METADATA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    deveverm    08/08/25 - added validation for query_vector in 
Rem                           search_from_query
Rem    deveverm    08/08/25 - Updated data type for v_overall_results_str and 
Rem                           removed debug logging for v_overall_results_str.
Rem    dadoshi     07/31/25 - JIRA_DBAI1149: Add optional parameter,
Rem                           p_query_vector to search_from_query
Rem    dadoshi     07/31/25 - JIRA_DBAI1147: Add get_top_report_match
Rem    fgurrola    07/29/25 - DBAI-1070: Change UPPER by normalization in the
Rem                           schema name.
Rem    deveverm    07/28/25 - DBAI-1011: Refactored reranking to rerank_texts
Rem    jiangnhu    07/17/25 - DBAI-1091: Change format of report_filter_values,
Rem                           drilldown_filter_values in report_matches. e.g.
Rem                           {'country':{'value':'US', 'reason': '...'}} ->
Rem                           [{'filter_name': 'country', 'value': 'US', 
Rem                           'reason': '...'}]
Rem    jiangnhu    07/09/25 - DBAI-314: Replace fuzzy match post validation with
Rem                           reranking
Rem    ruohli      07/01/25 - DBAI-884: Modify 
Rem                           update_non_default_filter_values_stats_from_search
Rem                           into update_filter_values_stats_from_search to 
Rem                           support both default and non-default filter values
Rem    sathyavc    06/30/25 - DBAI-881: Map search record to corresponding
Rem                           langdata$question_stats.question_id
Rem    jiangnhu    06/30/25 - DBAI-1003: Pass both original query and augmented
Rem                           query to process_filters
Rem    fgurrola    06/27/25 - DBAI-776: Measure execution Time for
Rem                           get_entities_using_ner.
Rem    jiangnhu    06/24/25 - DBAI-909: Integrate text augmentation with domain
Rem    anisbans    06/13/25 - Added the API update_report_search_count
Rem    dadoshi     05/15/25 - Remove get_filter_values() procedure
Rem    dadoshi     05/15/25 - JIRA_DBAI804: Update get_entities_using_ner usage
Rem    jiangnhu    04/30/25 - DBAI-662: Extend the logic of filters: support
Rem                           table column in another PDB or database
Rem    jiangnhu    04/12/25 - DBAI-739: Add plot_search_vectors
Rem    deveverm    04/10/25 - DBAI-723: Refactored all functions calling NER to
Rem                           named_entities_pkg
Rem    dadoshi     04/07/25 - Add usage of LANG_DATA_VALUE_VECTOR_TABLE_K for 
Rem                           similarity search in process_filters
Rem    dadoshi     04/02/25 - JIRA_DBAI522: Add support for multiple enumerable
Rem                           filters
Rem    jiangnhu    04/01/25 - DBAI-624: Make embedding model independent
Rem    dadoshi     03/26/25 - JIRA_DBAI691: Handle NULL case before conversion
Rem                           to JSON_ARRAY_T
Rem    jiangnhu    03/25/25 - Fix bug in get_entities_using_ner
Rem    jiangnhu    03/24/25 - DBAI-551: Use result of get_all_entities_using_ner
Rem                           to augment user query
Rem    dadoshi     03/20/25 - JIRA_DBAI525: Update to use 
Rem                           LANG_DATA_RERANK_RESULTS config parameter.
Rem    dadoshi     03/20/25 - JIRA_DBAI525: Add procedures to re-rank the top-k
Rem                           reports, drilldowns, and documents.
Rem    jiangnhu    03/19/25 - DBAI-543: Better naming conventions for
Rem                           augmentation/amending
Rem    jiangnhu    03/14/25 - DBAI-661: Update process_filters to handle
Rem                           non-ner filter with enumerable set
Rem    dadoshi     04/03/25 - JIRA_DBAI620: Add support for reports with no
Rem                           drilldowns to search_from_query.
Rem    deveverm    03/11/25 - DBAI-546: added schema_name for cross_schema
Rem                           support
Rem    jiangnhu    03/07/25 - DBAI-545: Make search_type a config
Rem    dadoshi     03/03/25 - JIRA_DBAI613: Update fetching of the ID of the
Rem                           parent report in case of document being Drilldown
Rem    dadoshi     03/03/25 - JIRA_DBAI596: Update Flat search to minimize
Rem                           NER handling calls
Rem    dadoshi     03/03/25 - JIRA_DBAI596: Add process_filters() and
Rem                           get_entities_using_ner() to populate filter
Rem                           values of reports/drilldowns
Rem    jiangnhu    02/27/25 - DBAI-600: Fix the logic of validate_match
Rem    jiangnhu    02/26/25 - DBAI-587: Handle input queries that has quotes
Rem    jiangnhu    02/25/25 - Update call to check_table_exists
Rem    dadoshi     02/24/25 - JIRA_DBAI578: Add config_pkg usage
Rem    jiangnhu    02/14/25 - DBAI-575: Remove c_unknown_exception_code
Rem    jiangnhu    02/04/25 - use generate_id
Rem    jiangnhu    01/31/25 - DBAI-511: Update function augment_query to
Rem                           procedure augment_text, add augmented_tokens
Rem    jiangnhu    01/29/25 - DBAI-505: Implement centroid version of
Rem                           search_from_query procedure
Rem    jiangnhu    01/22/25 - Make search_id optional
Rem    jiangnhu    01/16/25 - DBAI-508: Add get_report_matching_text,
Rem                           get_drilldown_matching_text, update
Rem                           search_from_query to add content to
Rem                           report_matches
Rem    jiangnhu    12/19/24 - JIRA_DBAI-451: Move match_substring_with_context
Rem                           to utils pkg
Rem    jiangnhu    11/26/24 - Fix error handling of delete_user_search_record
Rem    jiangnhu    11/22/24 - JIRA_DBAI-425: Implement search_from_query and
Rem                         - helper procedures and functions
Rem    jiangnhu    10/22/24 - Format code
Rem    jiangnhu    10/18/24 - Fix text wrapping
Rem    jiangnhu    10/17/24 - Modify header
Rem    pryarla     10/16/24 - Created
Rem

CREATE OR REPLACE PACKAGE BODY lang_data_search_pkg IS

    FUNCTION get_top_report_match (
        p_search_id                     IN VARCHAR2
    ) RETURN JSON_OBJECT_T
        IS
        v_query_text                    VARCHAR2(2000);
        v_feedback_rating               BOOLEAN;
        v_feedback_comments             VARCHAR2(2000);
        v_required_feedback_action      VARCHAR2(20);
        v_feedback_action_priority      VARCHAR2(20);
        v_expected_report_id            VARCHAR2(36);
        v_expected_drilldown_id         VARCHAR2(36);
        v_search_type                   VARCHAR2(32);
        v_augmented_query_text          VARCHAR2(4000);
        v_report_matches_json           JSON;
        v_report_matches_arr            JSON_ARRAY_T;
        v_top_report_match              JSON_OBJECT_T;
    BEGIN
        get_user_search_record(
            p_id => p_search_id,
            p_query_text => v_query_text,
            p_report_matches => v_report_matches_json,
            p_feedback_rating => v_feedback_rating,
            p_feedback_comments => v_feedback_comments,
            p_required_feedback_action => v_required_feedback_action,
            p_feedback_action_priority => v_feedback_action_priority,
            p_expected_report_id => v_expected_report_id,
            p_expected_drilldown_id => v_expected_drilldown_id,
            p_search_type => v_search_type,
            p_augmented_query_text => v_augmented_query_text
        );
        v_report_matches_arr := JSON_ARRAY_T(
            JSON_SERIALIZE(v_report_matches_json)
        );
        v_top_report_match := TREAT(
            v_report_matches_arr.GET(0) AS JSON_OBJECT_T
        );
        return v_top_report_match;
    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_resource_not_found
            ) THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                -- Log unknown errors as fatal and raise the generic error code
                lang_data_logger_pkg.log_fatal(
                    'An error occurred while fetching top report of ' || 
                    'search record with ID: ' || p_search_id || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
    END get_top_report_match;

    -- Implement the get_user_search_record procedure
    PROCEDURE get_user_search_record (
        p_id IN VARCHAR2,
        p_query_text OUT VARCHAR2,
        p_report_matches OUT JSON,
        p_feedback_rating OUT BOOLEAN,
        p_feedback_comments OUT VARCHAR2,
        p_required_feedback_action OUT VARCHAR2,
        p_feedback_action_priority OUT VARCHAR2,
        p_expected_report_id OUT VARCHAR2,
        p_expected_drilldown_id OUT VARCHAR2,
        p_search_type OUT VARCHAR2,
        p_augmented_query_text OUT VARCHAR2
    )
    AS
        -- Variable to hold the current user's name
        v_username VARCHAR2(200);

    BEGIN
        -- Get the current session user
        v_username := SYS_CONTEXT('USERENV', 'SESSION_USER');

        lang_data_logger_pkg.log_info(
            'Starting procedure get_user_search_record for ID: ' || p_id
        );

        -- Check if the role is enabled
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            -- Role is not enabled, restrict by the current session user
            lang_data_logger_pkg.log_info(
                'Role not enabled. Restricting results to current user: ' ||
                v_username
            );

            -- Query to get user search record restricted by username
            SELECT query_text, report_matches, feedback_rating,
                   feedback_comments, required_feedback_action,
                   feedback_action_priority, expected_report_id,
                   expected_drilldown_id, search_type, augmented_query_text
            INTO p_query_text, p_report_matches, p_feedback_rating,
                 p_feedback_comments, p_required_feedback_action, 
                 p_feedback_action_priority, p_expected_report_id,
                 p_expected_drilldown_id, p_search_type, p_augmented_query_text
            FROM langdata$searchrecords
            WHERE id = p_id
              AND username = v_username; -- Restrict to current user

        ELSE
            -- Role is enabled, no need to restrict by username
            lang_data_logger_pkg.log_info('Role enabled. No user restriction.');

            -- Query to get user search record without restriction
            SELECT query_text, report_matches, feedback_rating,
                   feedback_comments, required_feedback_action,
                   feedback_action_priority, expected_report_id,
                   expected_drilldown_id, search_type, augmented_query_text
            INTO p_query_text, p_report_matches, p_feedback_rating,
                 p_feedback_comments, p_required_feedback_action,
                 p_feedback_action_priority, p_expected_report_id,
                 p_expected_drilldown_id, p_search_type, p_augmented_query_text
            FROM langdata$searchrecords
            WHERE id = p_id;
        END IF;

        lang_data_logger_pkg.log_info(
            'Successfully retrieved search record for ID: ' || p_id
        );

    EXCEPTION
        -- Handle the case where no record is found
        WHEN NO_DATA_FOUND THEN
            lang_data_logger_pkg.log_error(
                'No search record found for ID: ' || p_id
            );
            -- Raise the custom error for search record not found
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );

        -- Optionally handle all other exceptions
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred for ID: ' || p_id || '. Error: ' ||
                SQLERRM
            );
            RAISE;
    END get_user_search_record;


    -- Procedure to get all user search records with pagination and filters
    PROCEDURE get_all_user_search_records (
        p_feedback_rating IN BOOLEAN DEFAULT NULL,
        p_required_feedback_action IN VARCHAR2 DEFAULT NULL,
        p_feedback_action_priority IN VARCHAR2 DEFAULT NULL,
        p_cursor IN OUT VARCHAR2,
        p_limit IN NUMBER DEFAULT 10,
        p_records OUT SYS_REFCURSOR
    )
    IS
        -- Variables for query construction
        v_base_query VARCHAR2(4000) := 'SELECT id, ' ||
                                       'query_text, ' ||
                                       'report_matches, ' ||
                                       'feedback_rating, ' ||
                                       'feedback_comments, ' ||
                                       'required_feedback_action, ' ||
                                       'feedback_action_priority, ' ||
                                       'expected_report_id, ' ||
                                       'expected_drilldown_id, ' ||
                                       'search_type, ' ||
                                       'augmented_query_text, ' ||
                                       'created_at ' ||
                                       'FROM langdata$searchrecords ';
        v_conditions VARCHAR2(4000) := NULL;
        v_limit_query VARCHAR2(4000);
        v_username VARCHAR2(200);
        v_last_id VARCHAR2(4000);
        v_last_created_at TIMESTAMP;
        v_cursor_created_at TIMESTAMP;
        v_cursor_id VARCHAR2(4000);

        -- Variables for bind variables and dynamic SQL
        v_cur INTEGER;
        v_dummy INTEGER;
        v_sql_text VARCHAR2(4000);
        v_next_cur INTEGER;

    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting procedure get_all_user_search_records'
        );

        -- Split cursor into created_at and id if it exists
        IF p_cursor IS NOT NULL THEN
            lang_data_utils_pkg.split_cursor(
                p_cursor => p_cursor,
                p_created_at => v_cursor_created_at,
                p_id => v_cursor_id
            );
            lang_data_logger_pkg.log_info(
                'Cursor split: CreatedAt=' ||
                TO_CHAR(v_cursor_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') ||
                ', ID=' || v_cursor_id
            );
        END IF;

        -- Get the current session user
        v_username := SYS_CONTEXT('USERENV', 'SESSION_USER');

        -- Role check for LANG_DATA_APP_EXPERT
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_logger_pkg.log_info(
                'Role not enabled for LANG_DATA_APP_EXPERT. ' ||
                'Restricting search records to current user: ' || v_username
            );
            v_conditions := 'WHERE username = :username ';
        END IF;

        -- Add filters for feedback_rating, required_feedback_action, 
        -- feedback_action_priority
        IF p_feedback_rating IS NOT NULL THEN
            IF v_conditions IS NULL THEN
                v_conditions := 'WHERE feedback_rating = :feedback_rating ';
            ELSE
                v_conditions := v_conditions ||
                'AND feedback_rating = :feedback_rating ';
            END IF;
        END IF;

        IF p_required_feedback_action IS NOT NULL THEN
            IF v_conditions IS NULL THEN
                v_conditions := 
                    'WHERE required_feedback_action = :feedback_action ';
            ELSE
                v_conditions := v_conditions ||
                'AND required_feedback_action = :feedback_action ';
            END IF;
        END IF;

        IF p_feedback_action_priority IS NOT NULL THEN
            IF v_conditions IS NULL THEN
                v_conditions := 'WHERE feedback_action_priority = ' ||
                                ':feedback_action_priority ';
            ELSE
                v_conditions := v_conditions ||
                'AND feedback_action_priority = :feedback_action_priority ';
            END IF;
        END IF;

        -- Handle pagination based on cursor values
        IF p_cursor IS NOT NULL THEN
            IF v_conditions IS NULL THEN
                v_conditions := 'WHERE (created_at < :cursor_created_at OR ' ||
                                '(created_at = :cursor_created_at AND ' ||
                                'id <= :cursor_id)) ';
            ELSE
                v_conditions := v_conditions || 
                                'AND (created_at < :cursor_created_at OR ' ||
                                '(created_at = :cursor_created_at AND ' ||
                                'id <= :cursor_id)) ';
            END IF;
            lang_data_logger_pkg.log_info(
                'Applied pagination cursor: CreatedAt=' || 
                TO_CHAR(v_cursor_created_at, 'YYYY-MM-DD HH24:MI:SS.FF') || 
                ', ID=' || v_cursor_id
            );
        END IF;

        -- Build the final query with optional limit
        IF p_limit IS NOT NULL THEN
            v_sql_text := v_base_query || v_conditions || 
                        'ORDER BY created_at DESC, id DESC ' ||
                        'FETCH FIRST :limit ROWS ONLY';
        ELSE
            v_sql_text := v_base_query || v_conditions || 
                        'ORDER BY created_at DESC, id DESC';
        END IF;

        lang_data_logger_pkg.log_info('Executing query: ' || v_sql_text);

        -- Open a dynamic cursor
        v_cur := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(v_cur, v_sql_text, DBMS_SQL.NATIVE);

        -- Bind variables dynamically
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':username', v_username);
        END IF;

        IF p_feedback_rating IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':feedback_rating', p_feedback_rating
            );
        END IF;

        IF p_required_feedback_action IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':feedback_action', p_required_feedback_action
            );
        END IF;

        IF p_feedback_action_priority IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':feedback_action_priority', p_feedback_action_priority
            );
        END IF;

        IF p_cursor IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(
                v_cur, ':cursor_created_at', v_cursor_created_at
            );
            DBMS_SQL.BIND_VARIABLE(v_cur, ':cursor_id', v_cursor_id);
        END IF;

        -- Bind the limit
        IF p_limit IS NOT NULL THEN
            DBMS_SQL.BIND_VARIABLE(v_cur, ':limit', p_limit);
        END IF;

        -- Execute the cursor and convert to REFCURSOR
        v_dummy := DBMS_SQL.EXECUTE(v_cur);
        p_records := DBMS_SQL.TO_REFCURSOR(v_cur);

        IF DBMS_SQL.IS_OPEN(v_cur) THEN
            DBMS_SQL.CLOSE_CURSOR(v_cur);
        END IF;

        IF p_limit IS NOT NULL THEN
            lang_data_logger_pkg.log_info('Checking for next row');

            v_base_query := 'SELECT id, created_at ' ||
                            'FROM langdata$searchrecords ';
            -- Now check for the next row using DBMS_SQL
            v_sql_text := v_base_query || v_conditions || 
                        'ORDER BY created_at DESC, id DESC ' ||
                        'OFFSET :limit ROWS FETCH NEXT 1 ROWS ONLY';
            lang_data_logger_pkg.log_info('Executing query: ' || v_sql_text);

            -- Open a new dynamic cursor for next row check
            v_next_cur := DBMS_SQL.OPEN_CURSOR;
            DBMS_SQL.PARSE(v_next_cur, v_sql_text, DBMS_SQL.NATIVE);

            -- Bind variables for next row query dynamically
            IF NOT lang_data_auth_pkg.is_role_enabled(
                lang_data_auth_pkg.c_lang_data_app_expert
            ) THEN
                DBMS_SQL.BIND_VARIABLE(v_next_cur, ':username', v_username);
            END IF;

            IF p_feedback_rating IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(
                    v_next_cur, ':feedback_rating', p_feedback_rating
                );
            END IF;

            IF p_required_feedback_action IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(
                    v_next_cur, ':feedback_action', p_required_feedback_action
                );
            END IF;

            IF p_feedback_action_priority IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(
                    v_next_cur,
                    ':feedback_action_priority',
                    p_feedback_action_priority
                );
            END IF;

            IF p_cursor IS NOT NULL THEN
                DBMS_SQL.BIND_VARIABLE(
                    v_next_cur, ':cursor_created_at', v_cursor_created_at
                );
                DBMS_SQL.BIND_VARIABLE(v_next_cur, ':cursor_id', v_cursor_id);
            END IF;

            DBMS_SQL.BIND_VARIABLE(v_next_cur, ':limit', p_limit);

            -- Define output for next row
            DBMS_SQL.DEFINE_COLUMN(v_next_cur, 1, v_last_id, 36);
            DBMS_SQL.DEFINE_COLUMN(v_next_cur, 2, v_last_created_at);

            -- Execute the cursor for next row check
            v_dummy := DBMS_SQL.EXECUTE(v_next_cur);

            -- Fetch the results
            IF DBMS_SQL.FETCH_ROWS(v_next_cur) > 0 THEN
                DBMS_SQL.COLUMN_VALUE(v_next_cur, 1, v_last_id);
                DBMS_SQL.COLUMN_VALUE(v_next_cur, 2, v_last_created_at);

                -- Set the next cursor for pagination
                p_cursor := TO_CHAR(
                                v_last_created_at, 'YYYY-MM-DD HH24:MI:SS.FF'
                            ) || '|' || v_last_id;
                lang_data_logger_pkg.log_info(
                    'Next cursor set to: ' || p_cursor
                );
            ELSE
                p_cursor := NULL;
                lang_data_logger_pkg.log_info('No further pages.');
            END IF;

            IF DBMS_SQL.IS_OPEN(v_next_cur) THEN
                -- Close the next cursor
                DBMS_SQL.CLOSE_CURSOR(v_next_cur);
            END IF;
        ELSE
            p_cursor := NULL;
            lang_data_logger_pkg.log_info('No further pages.');
        END IF;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal('An error occurred: ' || SQLERRM);
            -- Handle unexpected exceptions
            RAISE;
    END get_all_user_search_records;

    PROCEDURE delete_user_search_record (
        p_id IN VARCHAR2
    )
    AS
        v_username VARCHAR2(200);
    BEGIN
        -- Get the current session user
        v_username := SYS_CONTEXT('USERENV', 'SESSION_USER');
        
        -- Log the start of the procedure
        lang_data_logger_pkg.log_info(
            'Starting procedure delete_user_search_record for ID: ' || p_id
        );

        -- Check if the user has the required role
        IF NOT lang_data_auth_pkg.is_role_enabled(
            lang_data_auth_pkg.c_lang_data_app_expert
        ) THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_unauthorized_code
            );
        END IF;

        -- If the role is enabled, delete the record regardless of the owner
        DELETE FROM langdata$searchrecords
        WHERE id = p_id;

        -- Check if any row was deleted
        IF SQL%ROWCOUNT = 0 THEN
            lang_data_logger_pkg.log_error(
                'No record found for deletion with ID: ' || p_id
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_resource_not_found
            );
        ELSE
            lang_data_logger_pkg.log_info(
                'Record successfully deleted with ID: ' || p_id
            );
        END IF;
    EXCEPTION
       WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_unauthorized_code,
                lang_data_errors_pkg.c_resource_not_found
            ) THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                -- Log unknown errors as fatal and raise the generic error code
                lang_data_logger_pkg.log_fatal(
                    'An error occurred while deleting record with ID: ' ||
                    p_id || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;
    END delete_user_search_record;

    -- To optimize, maybe we can just pass the token instead of providing the
    -- augmented query text and tokenizing every time.
    FUNCTION validate_match(
        user_query          IN VARCHAR2,
        identified_value    IN VARCHAR2,
        threshold           IN NUMBER DEFAULT 80
    ) RETURN BOOLEAN IS
        normalized_query VARCHAR2(4000);
        normalized_value VARCHAR2(4000);
        match_score      NUMBER;
        token_score      NUMBER;
        token           VARCHAR2(4000);
        pos             NUMBER := 1;
    BEGIN
        -- Normalize input
        normalized_query := LOWER(TRIM(user_query));
        normalized_value := LOWER(TRIM(identified_value));

        -- Exact substring match check
        IF INSTR(normalized_query, normalized_value) > 0 THEN
            RETURN TRUE;
        END IF;

        -- Initialize best match score
        match_score := 0;

        -- Iterate through each token in user_query
        LOOP
            -- Extract next word using space as a delimiter
            token := REGEXP_SUBSTR(normalized_query, '\S+', 1, pos);
            EXIT WHEN token IS NULL;

            -- Compute similarity with identified_value
            token_score := UTL_MATCH.EDIT_DISTANCE_SIMILARITY(
                token, normalized_value
            );

            -- Update match_score if token has a higher similarity
            IF token_score > match_score THEN
                match_score := token_score;
            END IF;

            -- Move to the next token
            pos := pos + 1;
        END LOOP;

        -- Compare final highest score with threshold
        RETURN match_score >= threshold;
    END validate_match;
    PROCEDURE print_filters_temptable
    IS
        CURSOR temp_table_cur IS
            SELECT filter_name, filter_value, reason
            FROM langdata$filtervaluestemp;
        v_filter_name   VARCHAR2(255);
        v_filter_value  VARCHAR2(4000);
        v_reason        VARCHAR2(4000);
    BEGIN
        -- Print Header
        lang_data_logger_pkg.log_info('FILTER_NAME | FILTER_VALUE | REASON');
        lang_data_logger_pkg.log_info('----------------------------------------');
        
        -- Open Cursor and Loop Through Records
        FOR rec IN temp_table_cur LOOP
            lang_data_logger_pkg.log_info(
                rec.filter_name || ' | ' || 
                rec.filter_value || ' | ' || rec.reason
            );
        END LOOP;
    END print_filters_temptable;

    PROCEDURE filter_candidates_rerank_call(
        p_query              IN VARCHAR2,
        p_filter_candidates  IN JSON_ARRAY_T,
        p_top_candidate      OUT VARCHAR2,
        p_top_score          OUT NUMBER,
        p_top_index          OUT INTEGER
    ) IS
        v_is_adb            VARCHAR2(10);
        v_compartment_id    VARCHAR2(4000);
        v_region_code       VARCHAR2(4000);
        v_cred_name         VARCHAR2(4000);
        v_rerank_endpoint_id  VARCHAR2(4000);
        v_serving_mode      JSON_OBJECT_T;
        v_req_body          JSON_OBJECT_T;
        v_req_headers       JSON_OBJECT_T;
        v_req_body_hash     VARCHAR2(4000);
        v_req_body_hash_raw VARCHAR2(4000);
        v_base_url          VARCHAR2(4000);
        v_second_level_domain  VARCHAR2(4000);
        v_full_resp         DBMS_CLOUD_TYPES.resp;
        v_response_json_obj JSON_OBJECT_T;

        v_filter_candidates_str   VARCHAR2(32767);
        v_pyq_json_object   JSON_OBJECT_T := JSON_OBJECT_T();
        v_pyq_string        VARCHAR2(32767);
        v_model_cache       VARCHAR2(200);
    BEGIN
        v_is_adb := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_IS_AUTONOMOUS_DB'
        );
        v_compartment_id := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_OCI_COMPARTMENT'
        );
        v_region_code := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_OCI_REGION'
        );
        v_cred_name := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_OCI_CRED'
        );
        v_rerank_endpoint_id := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_GENAI_RERANK_ENDPOINT_ID'
        );
        IF LOWER(
                lang_data_config_pkg.get_config_parameter(
                    'LANG_DATA_USE_LLM'
                )
            ) = 'true' THEN
            lang_data_logger_pkg.log_debug('Using LLM for reranking');
            lang_data_llm_pkg.rerank_filter_value_candidates(
                p_query,
                p_filter_candidates,
                p_top_candidate,
                p_top_score,
                p_top_index
            );
        ELSIF LOWER(v_is_adb) = 'true' THEN
            -- Add checks
            IF v_region_code IS NULL THEN
                lang_data_logger_pkg.log_error(
                    '''LANG_DATA_OCI_REGION'' is not set.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_OCI_region_not_set
                );
            END IF;
            IF v_cred_name IS NULL THEN
                lang_data_logger_pkg.log_error(
                    '''LANG_DATA_OCI_CRED'' is not set.'
                );
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_OCI_credential_not_set
                );
            END IF;
            
            v_serving_mode := JSON_OBJECT_T('{}');
            v_serving_mode.put('servingType', 'DEDICATED');
            v_serving_mode.put('endpointId', v_rerank_endpoint_id);

            v_req_body := JSON_OBJECT_T('{}');
            IF v_compartment_id IS NOT NULL THEN
                v_req_body.put('compartmentId', v_compartment_id);
            END IF;
            v_req_body.put('documents', p_filter_candidates);
            v_req_body.put('input', p_query);
            v_req_body.put('servingMode', v_serving_mode);

            v_req_body_hash := DBMS_CRYPTO.hash(
                v_req_body.to_blob, DBMS_CRYPTO.HASH_SH256
            );
            v_req_body_hash_raw := UTL_RAW.cast_to_varchar2(
                UTL_ENCODE.base64_encode(v_req_body_hash)
            );

            -- Create request headers
            v_req_headers := JSON_OBJECT_T('{}');
            v_req_headers.put('accept', 'application/json');
            v_req_headers.put('content-type', 'application/json');
            v_req_headers.put('opc-request-id', UPPER(sys_guid()));
            v_req_headers.put('opc-client-info','Oracle-PlsqlSDK/1.9');
            v_req_headers.put('x-content-sha256', v_req_body_hash_raw);

            v_base_url :=  
                'https://language.aiservice.{region}.oci.{secondLevelDomain}/20231130/actions/rerankText';
            
            -- Using Endpoint reference:
            -- https://docs.oracle.com/en-us/iaas/api/#/en/language/20221001/
            IF v_region_code = 'eu-jovanovac-1' THEN
                v_second_level_domain := 'oraclecloud20.com';
            ELSE
                v_second_level_domain := 'oraclecloud.com';
            END IF;

            v_base_url := REGEXP_REPLACE(
                v_base_url, 
                '{region}',
                v_region_code
            );

            v_base_url := REGEXP_REPLACE(
                v_base_url, 
                '{secondLevelDomain}',
                v_second_level_domain
            );

            lang_data_logger_pkg.log_debug(
                'Rerank REST request header: '|| v_req_headers.to_clob()
            );
            lang_data_logger_pkg.log_debug(
                'Rerank REST request body: '|| v_req_body.to_clob()
            );

            -- Send the request
            v_full_resp := DBMS_CLOUD.send_request(
                credential_name => v_cred_name,
                uri             => v_base_url,
                method          => 'POST',
                headers         => v_req_headers.to_clob(),
                body            => v_req_body.to_blob
            );

            v_response_json_obj := 
                JSON_OBJECT_T(DBMS_CLOUD.GET_RESPONSE_TEXT(v_full_resp));
        ELSE
            FOR i IN 0 .. p_filter_candidates.get_size - 1 LOOP
                IF i > 0 THEN
                    v_filter_candidates_str := v_filter_candidates_str || ', ';
                END IF;
                    v_filter_candidates_str := v_filter_candidates_str || 
                                       p_filter_candidates.get_string(i);
            END LOOP;

            v_pyq_json_object.put('query', REPLACE(p_query, '''', ''''''));
            v_pyq_json_object.put(
                'candidates_str',
                REPLACE(v_filter_candidates_str, '''', '''''')
            );
            v_model_cache := lang_data_config_pkg.get_config_parameter(
                                'MOUNT_DIR'
                            );
            v_model_cache := v_model_cache || '/models';
            v_pyq_json_object.put('cache_dir', v_model_cache);
            v_pyq_string := v_pyq_json_object.to_string();
            SELECT "top_candidate", "top_score", "top_index"
            INTO p_top_candidate, p_top_score, p_top_index
            FROM TABLE(
                pyqEval(
                    v_pyq_string,  
                    '{
                        "top_candidate": "varchar2(100)",
                        "top_score": "number",
                        "top_index": "number"
                    }',
                    'rerank_filter_value_candidates' 
                )
            );
        END IF;
        
    END filter_candidates_rerank_call;
    
    PROCEDURE process_filters(
        p_original_query            IN VARCHAR2,
        p_augmented_query           IN VARCHAR2,
        p_match_document_arr        IN JSON_ARRAY_T,
        p_report_filter_values      OUT JSON_ARRAY_T,
        p_drilldown_filter_values   OUT JSON_ARRAY_T,
        p_num_filters               OUT SYS.ODCIVARCHAR2LIST
    ) IS
        v_json_object               JSON_OBJECT_T;
        v_filter_name               VARCHAR2(255);
        v_use_ner                   BOOLEAN;
        v_default_value             VARCHAR2(4000);
        v_entity_type               VARCHAR2(255);
        v_table_name                VARCHAR2(255);
        v_column_name               VARCHAR2(255);
        v_schema_name               VARCHAR2(255);
        v_db_link_name              VARCHAR2(255);
        v_description               VARCHAR2(4000);
        v_filter_description        VARCHAR2(4000);
        v_query_vector              VECTOR;
        v_entity_value              VARCHAR2(4000);
        v_value_vec_table           VARCHAR2(255);
        v_similarity_result         VARCHAR2(4000);
        v_similarity_distance       NUMBER;
        v_match_threshold           NUMBER;
        v_prev_context_len          NUMBER;
        v_post_context_len          NUMBER;
        v_filter_exists             NUMBER;
        v_document_match_document   JSON_ELEMENT_T;
        v_match_document            JSON;
        v_filter_descriptions       SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_filter_count              NUMBER := 0;
        v_entity_count              NUMBER := 0;
        v_entity_types              SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_ids                       SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_entity_values             SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_object                    JSON_OBJECT_T;
        v_document_type             VARCHAR2(20);
        v_document_id               VARCHAR2(36);
        v_md5                       VARCHAR2(32);
        v_report_filter_values      JSON_ARRAY_T := JSON_ARRAY_T();
        v_drilldown_filter_values   JSON_ARRAY_T := JSON_ARRAY_T();
        v_value                     VARCHAR2(4000);
        v_reason                    VARCHAR2(4000);
        v_num_filters               SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_cnt                       NUMBER := 0;
        v_value_vec_table_k         NUMBER := 3;
        TYPE similarity_result_tab IS TABLE OF VARCHAR2(4000);
        TYPE similarity_distance_tab IS TABLE OF NUMBER;
        v_similarity_results similarity_result_tab;
        v_similarity_distances similarity_distance_tab;
        v_index INTEGER;
        v_filter_candidates         JSON_ARRAY_T := JSON_ARRAY_T();
        v_pyq_json_object           JSON_OBJECT_T := JSON_OBJECT_T();
        v_pyq_string                VARCHAR2(32767);
        v_similarity_score          NUMBER;

        CURSOR filters_cursor(any_match_document JSON) IS
            SELECT filter_name,
                use_ner,
                default_value,
                entity_type,
                schema_name,
                table_name,
                column_name,
                db_link_name,
                description
            FROM JSON_TABLE(
                any_match_document,
                '$.filters[*]'
                COLUMNS (
                    filter_name VARCHAR2(255) PATH '$.filter_name',
                    use_ner BOOLEAN PATH '$.use_ner',
                    default_value VARCHAR2(4000) PATH '$.default_value',
                    entity_type VARCHAR2(255) PATH '$.entity_type',
                    schema_name VARCHAR2(255) PATH '$.schema_name',
                    table_name VARCHAR2(255) PATH '$.table_name',
                    column_name VARCHAR2(255) PATH '$.column_name',
                    db_link_name VARCHAR2(255) PATH '$.db_link_name',
                    description VARCHAR2(4000) PATH '$.description'
                )
            );
        
        TYPE set_type IS TABLE OF BOOLEAN INDEX BY VARCHAR2(4000);
        filter_value_set set_type;
        v_partition_size  NUMBER;
        v_start_time                        NUMBER;
        v_end_time                          NUMBER;
    BEGIN
        -- Get environment variables
        v_match_threshold := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_POST_VALIDATION_FUZZY_MATCH_THRESHOLD'
            )
        );
        v_prev_context_len := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_PREV_CONTEXT_LEN'
            )
        );
        v_post_context_len := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_POST_CONTEXT_LEN'
            )
        );
        v_value_vec_table_k := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_VALUE_VECTOR_TABLE_K'
            )
        );

        -- Generate the query embedding
        v_query_vector := lang_data_utils_pkg.get_embedding(p_augmented_query);

        lang_data_logger_pkg.log_info(
            '(process_filters): Number of Match Documents = ' || 
            p_match_document_arr.get_size
        );

        -- Fetch the filter description, for every filter whose use_ner is TRUE,
        -- for every match document provided in the list.
        FOR indx IN 0 .. p_match_document_arr.get_size - 1
        LOOP

            -- <JSON_ARRAY_T>.GET(NUMBER) returns JSON_ELEMENT_T, 
            -- so need to convert it to JSON.
            v_object := TREAT(p_match_document_arr.GET(indx) AS JSON_OBJECT_T);
            v_document_match_document := v_object.GET('match_document');
            v_match_document := v_document_match_document.to_json();
            v_document_id := v_object.get_string('document_id');
            v_cnt := 0;

            FOR filter_rec IN filters_cursor(v_match_document)
            LOOP
                v_cnt := v_cnt + 1;
                v_filter_name := filter_rec.filter_name;
                v_use_ner := filter_rec.use_ner;
                v_default_value := filter_rec.default_value;
                v_entity_type := filter_rec.entity_type;
                v_schema_name := filter_rec.schema_name;
                v_table_name := filter_rec.table_name;
                v_column_name := filter_rec.column_name;
                v_db_link_name := filter_rec.db_link_name;
                v_description := filter_rec.description;

                -- If every filter in dataset has use_ner=false, flip the 
                -- condition just for testing.
                IF v_use_ner THEN

                    -- Fetch filter description, only if use_ner is TRUE
                    -- Idea: We can cache the filter description while the 
                    --       report/drilldown is created.
                    v_filter_description := 
                        lang_data_utils_pkg.get_filter_description(
                        p_filter_name => v_filter_name,
                        p_table_name => v_table_name,
                        p_column_name => v_column_name,
                        p_schema_name => v_schema_name,
                        p_additional_description => v_description,
                        p_db_link_name => v_db_link_name
                    );

                    v_filter_descriptions.EXTEND;
                    v_entity_types.EXTEND;
                    v_ids.EXTEND;
                    v_filter_count := v_filter_count + 1;
                    v_filter_descriptions(
                        v_filter_count
                    ) := v_filter_description;
                    v_entity_types(v_filter_count) := lower(v_entity_type);
                    v_ids(v_filter_count) := v_document_id;
                END IF;
            END LOOP;
            v_num_filters.EXTEND;
            v_num_filters(indx+1) := v_cnt;
        END LOOP;
        -- Fetch the entities from the Query corresponding to every filter
        -- having use_ner set to TRUE.
        IF v_entity_types.COUNT > 0 THEN
            v_start_time := DBMS_UTILITY.GET_TIME;
            lang_data_named_entities_pkg.get_entities_using_ner(
                p_original_query, 
                v_filter_descriptions, 
                v_entity_types, 
                v_ids, 
                v_entity_values
            );
            v_end_time := DBMS_UTILITY.GET_TIME;

            lang_data_logger_pkg.log_info(
                '-------------------------------------------------------------'
                ||
                ' Execution Time for get_entities_using_ner: ' || 
                    (v_end_time - v_start_time) || ' centiseconds ' ||
                '--------------------------------------------------------------'
            );
        END IF;

        -- Reset to 0
        v_filter_count := 0;

        -- Iterate over all the match documents to populate the filter values.
        FOR j IN 0 .. p_match_document_arr.get_size - 1 LOOP
            -- Same conversion as in the previous loop.
            v_object := TREAT(p_match_document_arr.GET(j) AS JSON_OBJECT_T);
            v_document_match_document := v_object.GET('match_document');
            v_match_document := v_document_match_document.to_json();

            -- This procedure takes in the match documents of both Reports and
            -- and Drilldowns. Hence, we need to store the document type too, so
            -- that we return the filter values in their appropriate list, i.e,
            -- either in report_filter_values or drilldown_filter_values.
            v_document_type := v_object.get_string('document_type');
            v_document_id := v_object.get_string('document_id');
            v_cnt := 0;

            lang_data_logger_pkg.log_info(
                '(process_filters): Processing document of type = ' || 
                v_document_type || ' and match_document = ' || 
                v_document_match_document.to_string
            );

            filter_value_set.DELETE;

            FOR filter_rec IN filters_cursor(v_match_document)
            LOOP
                v_filter_count := v_filter_count + 1;
                v_filter_name := UPPER(filter_rec.filter_name);
                v_use_ner := filter_rec.use_ner;
                v_default_value := filter_rec.default_value;
                v_entity_type := filter_rec.entity_type;
                v_schema_name := lang_data_utils_pkg.normalize_schema_name(
                    filter_rec.schema_name
                );
                v_table_name := UPPER(filter_rec.table_name);
                v_column_name := UPPER(filter_rec.column_name);
                v_db_link_name := UPPER(filter_rec.db_link_name);
                v_description := filter_rec.description;
                v_cnt := v_cnt + 1;
                lang_data_logger_pkg.log_info(
                    '(process_filters): Filter Name = ' || v_filter_name || ' with description = ' || v_description
                );
                -- Check if the filter value already exists in the temporary 
                -- table, langdata$filtervaluestemp
                SELECT COUNT(*)
                INTO v_filter_exists
                FROM langdata$filtervaluestemp
                WHERE filter_name = v_filter_name;

                IF v_filter_exists > 0 THEN
                    lang_data_logger_pkg.log_info(
                        'Filter with name ' || v_filter_name || ' already exists.'
                    );
                    SELECT filter_value, reason
                    INTO v_value, v_reason
                    FROM langdata$filtervaluestemp
                    WHERE filter_name = v_filter_name;

                    v_json_object := JSON_OBJECT_T();
                    v_json_object.PUT('filter_name', v_filter_name);
                    v_json_object.PUT('value', v_value);
                    v_json_object.PUT('reason', v_reason);
                    IF v_document_type = 'report' THEN
                        v_report_filter_values.APPEND(v_json_object);
                    ELSE
                        v_drilldown_filter_values.APPEND(v_json_object);
                    END IF;

                    CONTINUE;
                    
                END IF;

                IF v_use_ner THEN
                    v_entity_count := v_entity_count + 1;
                    IF v_entity_values(v_entity_count) IS NULL THEN
                        INSERT INTO langdata$filtervaluestemp (
                            filter_name, filter_value, reason
                        )
                        VALUES (
                            v_filter_name,
                            v_default_value,
                            'Using default value, as entity type ' ||
                            v_entity_type || ' was not found.'
                        );
                        v_json_object := JSON_OBJECT_T();
                        v_json_object.PUT('filter_name', v_filter_name);
                        v_json_object.PUT('value', v_default_value);
                        v_json_object.PUT(
                            'reason', 
                            'Using default value, as entity type ' || 
                            v_entity_type || ' was not found.'
                        );
                        IF v_document_type = 'report' THEN
                            v_report_filter_values.APPEND(v_json_object);
                        ELSE
                            v_drilldown_filter_values.APPEND(v_json_object);
                        END IF;
                        CONTINUE;
                    ELSE 
                        INSERT INTO langdata$filtervaluestemp (
                            filter_name, filter_value, reason
                        )
                        VALUES (
                            v_filter_name,
                            v_entity_values(v_entity_count),
                            'Found entity type ' || v_entity_type ||
                            ' in user query using NER.'
                        );
                        v_json_object := JSON_OBJECT_T();
                        v_json_object.PUT('filter_name', v_filter_name);
                        v_json_object.PUT('value', v_entity_values(v_entity_count));
                        v_json_object.PUT(
                            'reason', 
                            'Found entity type ' || v_entity_type || 
                            ' in user query using NER.'
                        );
                        IF v_document_type = 'report' THEN
                            v_report_filter_values.APPEND(v_json_object);
                        ELSE
                            v_drilldown_filter_values.APPEND(v_json_object);
                        END IF;
                    END IF;
                ELSE
                    -- Build value vector table name
                    IF v_table_name IS NOT NULL THEN
                        v_value_vec_table := 'LANGDATA$DRILLDOWNVALUES_' ||
                                            v_schema_name ||'_'||v_table_name ||
                                            '_' || v_column_name;
                        IF v_db_link_name IS NOT NULL THEN
                            v_value_vec_table := v_value_vec_table || '_' ||
                                                 v_db_link_name;
                        END IF;
                    ELSE
                        v_md5 := RAWTOHEX(
                            DBMS_CRYPTO.HASH(
                                UTL_RAW.CAST_TO_RAW(v_document_id),
                                DBMS_CRYPTO.HASH_MD5
                            )
                        );
                        v_value_vec_table := 'LANGDATA$DRILLDOWNVALUES_' ||
                                              v_filter_name || '_' ||
                                              v_md5;
                    END IF;

                    SELECT COUNT(*) INTO v_partition_size
                    FROM langdata$drilldownvalues
                    WHERE partition_name = UPPER(v_value_vec_table);

                    -- Check table existence
                    IF v_partition_size = 0 THEN
                        INSERT INTO langdata$filtervaluestemp (
                            filter_name, filter_value, reason
                        )
                        VALUES (
                            v_filter_name,
                            v_default_value,
                            'Using default value as the value vector ' ||
                            'partition ' || v_value_vec_table ||
                            ' has not been created yet.'
                        );
                        v_json_object := JSON_OBJECT_T();
                        v_json_object.PUT('filter_name', v_filter_name);
                        v_json_object.PUT('value', v_default_value);
                        v_json_object.PUT(
                            'reason', 
                            'Using default value as the value vector ' ||
                            'partition ' || v_value_vec_table ||
                            ' has not been created yet.'
                        );
                        IF v_document_type = 'report' THEN
                            v_report_filter_values.APPEND(v_json_object);
                        ELSE
                            v_drilldown_filter_values.APPEND(v_json_object);
                        END IF;
                        CONTINUE;
                    END IF;

                    lang_data_logger_pkg.log_info(
                        '(process_filters): check-partition-exists ' ||
                        'checkpoint passed.'
                    );
                    -- Perform similarity search
                    EXECUTE IMMEDIATE
                        'SELECT VALUE, VECTOR_DISTANCE(VVEC, :query_vector) AS 
                        DISTANCE FROM langdata$drilldownvalues ' ||
                        'WHERE partition_name = ''' ||
                        UPPER(v_value_vec_table) || '''
                        ORDER BY VECTOR_DISTANCE(VVEC, :query_vector) ASC
                        FETCH FIRST ' || v_value_vec_table_k || ' ROWS ONLY'
                    BULK COLLECT INTO v_similarity_results, v_similarity_distances
                    USING v_query_vector, v_query_vector;

                    IF v_similarity_results.COUNT = 0 THEN
                        lang_data_logger_pkg.log_info(
                            '(process_filters): No similarity results found.'
                        );
                        INSERT INTO langdata$filtervaluestemp (
                            filter_name, filter_value, reason
                        )
                        VALUES (
                            v_filter_name,
                            v_default_value,
                            'Using default value as no data found in ' ||
                            'value vector table ' || v_value_vec_table || '.'
                        );
                        v_json_object := JSON_OBJECT_T();
                        v_json_object.PUT('filter_name', v_filter_name);
                        v_json_object.PUT('value', v_default_value);
                        v_json_object.PUT(
                            'reason', 
                            'Using default value as no data found in ' || 
                            'value vector table ' || v_value_vec_table || '.'
                        );
                        IF v_document_type = 'report' THEN
                            v_report_filter_values.APPEND(v_json_object);
                        ELSE
                            v_drilldown_filter_values.APPEND(v_json_object);
                        END IF;
                        CONTINUE;
                    END IF;

                    lang_data_logger_pkg.log_info(
                        '(process_filters): Similarity results > 0 checkpoint passed.'
                    );
                    v_filter_candidates := JSON_ARRAY_T();
                    FOR i IN 1 .. v_similarity_results.COUNT LOOP
                        IF filter_value_set.EXISTS(v_similarity_results(i)) 
                            AND filter_value_set(v_similarity_results(i)) THEN
                            CONTINUE;
                        END IF;
                        v_filter_candidates.APPEND(v_similarity_results(i));
                    END LOOP;

                    filter_candidates_rerank_call(
                        p_query              => p_original_query,
                        p_filter_candidates  => v_filter_candidates,
                        p_top_candidate      => v_similarity_result,
                        p_top_score          => v_similarity_score,
                        p_top_index          => v_index
                    );
                    filter_value_set(v_similarity_result) := TRUE;
                    INSERT INTO langdata$filtervaluestemp (
                        filter_name, filter_value, reason
                    )
                    VALUES (
                        v_filter_name, 
                        v_similarity_result,
                        'Using similarity search result ' ||
                            v_index || ' with distance ' || 
                            v_similarity_distances(v_index) ||
                            ', semantic similarity score ' || v_similarity_score
                    );
                    v_json_object := JSON_OBJECT_T();
                    v_json_object.PUT('filter_name', v_filter_name);
                    v_json_object.PUT('value', v_similarity_result);
                    v_json_object.PUT(
                        'reason', 
                        'Using similarity search result ' ||
                            v_index || ' with distance ' || 
                            v_similarity_distances(v_index) ||
                            ', semantic similarity score ' || v_similarity_score
                    );
                    IF v_document_type = 'report' THEN
                        v_report_filter_values.APPEND(v_json_object);
                    ELSE
                        v_drilldown_filter_values.APPEND(v_json_object);
                    END IF;
                END IF;
            END LOOP;

            lang_data_logger_pkg.log_info(
                '(process_filters): Document Type = ' || v_document_type ||
                ' with ' || v_cnt || ' filters processed.'
            );
        END LOOP;

        p_report_filter_values := v_report_filter_values;
        p_drilldown_filter_values := v_drilldown_filter_values;
        p_num_filters := v_num_filters;

        lang_data_logger_pkg.log_info(
            '(process_filters): Report and Drilldown filter values fetched!'
        );
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_info('(process_filters) Exception');
            RAISE;
    END process_filters;

    PROCEDURE re_rank_similarity_search_reports (
        p_query_text            IN VARCHAR2,
        p_ranked_results        IN SYS_REFCURSOR,
        p_re_ranked_results     OUT SYS_REFCURSOR
    ) 
    IS
        -- Record type to match the structure of
        -- fetch_top_k_similarity_search_reports
        TYPE intermediate_result_rec IS RECORD (
            report_id                       VARCHAR2(36),
            id                              VARCHAR2(36),
            enhanced_text                    VARCHAR2(4000),
            distance                        NUMBER,
            text                            VARCHAR2(2000),
            title                           VARCHAR2(255),
            match_type                      VARCHAR2(100),
            type                            VARCHAR2(36),
            similarity_search_rank          NUMBER
        );
        v_rec                               intermediate_result_rec;
        v_re_ranked_reports        ranked_report_table := ranked_report_table();
        v_rank                              NUMBER := 0;
        v_json_string                       VARCHAR2(32767);
        v_result                            JSON;
        v_re_ranked_results                 JSON;
        v_start_time                        NUMBER;
        v_end_time                          NUMBER;
        v_pyq_json_object                   JSON_OBJECT_T := JSON_OBJECT_T();
        v_pyq_string                        VARCHAR2(32767);
        v_texts                             SYS.ODCIVARCHAR2LIST;
        v_all_texts                         VARCHAR2(6100);
        v_n                                 NUMBER;
        TYPE report_by_description IS TABLE OF VARCHAR2(36) INDEX BY VARCHAR2(36);
        TYPE distance_by_description IS TABLE OF NUMBER INDEX BY VARCHAR2(36);
        TYPE similarity_rank_by_description IS TABLE OF NUMBER INDEX BY VARCHAR2(36);
        TYPE match_type_by_description IS TABLE OF VARCHAR2(36) INDEX BY VARCHAR2(36);
        v_report_by_description             report_by_description;
        v_distance_by_description           distance_by_description;
        v_similarity_rank_by_description    similarity_rank_by_description;
        v_match_type_by_description         match_type_by_description;
        v_model_cache       VARCHAR2(4000);
        v_rerank_obj        RERANK_OBJ;
        v_reranked_texts    RERANK_TABLE := RERANK_TABLE();
    BEGIN
        v_n := NVL(
            TO_NUMBER(
                lang_data_config_pkg.get_config_parameter(
                    'LANG_DATA_REPORT_TEXTS_N'
                )
            ), 
            3
        );
        lang_data_logger_pkg.log_info('Processing intermediate results...');
        LOOP
            FETCH p_ranked_results INTO v_rec;
            EXIT WHEN p_ranked_results%NOTFOUND;
            
            lang_data_utils_pkg.fetch_top_n_similairty_search_report_text(
                p_query => p_query_text,
                p_report_id => v_rec.report_id,
                p_n => v_n,
                p_texts => v_texts
            );
            v_all_texts := '';
            FOR i in 1 .. v_texts.COUNT LOOP
                IF i = 1 THEN
                    v_all_texts := v_all_texts || v_texts(i);
                ELSE
                    v_all_texts := v_all_texts || ', ' || v_texts(i);
                END IF;
            END LOOP;

            v_report_by_description(v_rec.id) := v_rec.report_id;
            v_distance_by_description(v_rec.id) := v_rec.distance;
            v_similarity_rank_by_description(v_rec.id) := 
                                                v_rec.similarity_search_rank;
            v_match_type_by_description(v_rec.id) := v_rec.match_type;

            v_rerank_obj := RERANK_OBJ(
                doc_id => v_rec.id,
                doc_text => v_all_texts,
                query_text => p_query_text,
                doc_rank => NULL,
                score => NULL
            );
            v_reranked_texts.EXTEND;
            v_reranked_texts(v_reranked_texts.COUNT) := v_rerank_obj;
        END LOOP;

        CLOSE p_ranked_results;
        lang_data_logger_pkg.log_info(
            'Intermediate results processed successfully.'
        );

        v_reranked_texts := lang_data_search_pkg.rerank_texts(
            p_rerank_obj_list => v_reranked_texts
        );

        v_rank := 0;
        FOR i IN 1..v_reranked_texts.COUNT LOOP
            v_rank := v_rank + 1;
            lang_data_logger_pkg.log_info(
                'After Re-ranking: Report ID: ' 
                || v_report_by_description(v_reranked_texts(i).doc_id) 
                ||' Similarity Search Rank: ' 
                || v_similarity_rank_by_description(
                    v_reranked_texts(i).doc_id) 
                ||' Cross Encoder Rank: ' 
                || v_rank
            );
            v_re_ranked_reports.EXTEND;
            v_re_ranked_reports(v_rank) := ranked_report_obj(
                report_id               => 
                    v_report_by_description(v_reranked_texts(i).doc_id),
                description_id          => 
                    v_reranked_texts(i).doc_id,
                vector_distance         => 
                    v_distance_by_description(v_reranked_texts(i).doc_id),
                source                  => 
                    'similarity_search',
                similarity_search_rank  => 
                    v_similarity_rank_by_description(
                        v_reranked_texts(i).doc_id),
                cross_encoder_rank      => 
                    v_reranked_texts(i).doc_rank,
                overall_rank            => 
                    v_rank,
                match_type              => 
                    v_match_type_by_description(v_reranked_texts(i).doc_id)
            );

        END LOOP;

        OPEN p_re_ranked_results FOR
            SELECT * FROM TABLE(v_re_ranked_reports);

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            'Time for re-ranking the top-k reports: ' ||
            (v_end_time - v_start_time) || ' centiseconds.'
        );
    END re_rank_similarity_search_reports;

    PROCEDURE get_top_k_closest_report_and_description_similarity_search (
        p_query             IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_using_centroid    IN  BOOLEAN DEFAULT FALSE,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    ) AS
        v_intermediate_results      SYS_REFCURSOR;
        v_re_ranking_enabled        BOOLEAN;
        v_rank                      NUMBER := 0;
        TYPE intermediate_result_rec IS RECORD (
            report_id              VARCHAR2(36),
            id                     VARCHAR2(36),
            enhanced_text           VARCHAR2(4000),
            distance               NUMBER,
            text                   VARCHAR2(2000),
            title                  VARCHAR2(255),
            match_type             VARCHAR2(100),
            type                   VARCHAR2(36),
            similarity_search_rank NUMBER
        );
        v_rec                       intermediate_result_rec;
        v_ranked_reports       ranked_report_table := ranked_report_table();
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting get_top_k_closest_report_and_description_similarity_' ||
            'search procedure.'
        );
        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_RERANK_RESULTS'
            )
        ) = 'true' THEN
            v_re_ranking_enabled := TRUE;
        ELSE
            v_re_ranking_enabled := FALSE;
        END IF;

        -- Step 1: Fetch top-k similarity search reports
        lang_data_logger_pkg.log_info('Fetching intermediate results...');
        IF NOT p_using_centroid THEN
            lang_data_utils_pkg.fetch_top_k_similarity_search_reports(
                p_query     => p_query,       -- Pass query directly
                p_k         => p_k,           -- Top-k parameter
                p_domain_id => p_domain_id,   -- Filter based on domain
                p_results => v_intermediate_results  -- Intermediate cursor
            );
        ELSE
            lang_data_utils_pkg.fetch_top_k_similarity_search_reports_using_centroid(
                p_query     => p_query,       -- Pass query directly
                p_k         => p_k,           -- Top-k parameter
                p_domain_id => p_domain_id,   -- Filter based on domain
                p_results => v_intermediate_results  -- Intermediate cursor
            );
        END IF;
        lang_data_logger_pkg.log_info(
            'Intermediate results fetched successfully.'
        );

        -- Step 2: Iterate through the intermediate cursor and populate
        -- the collection
        IF v_re_ranking_enabled AND p_using_centroid=FALSE THEN
            lang_data_search_pkg.re_rank_similarity_search_reports(
                p_query_text            => p_query,
                p_ranked_results        => v_intermediate_results,
                p_re_ranked_results     => p_results
            );
        ELSE
            LOOP
                FETCH v_intermediate_results INTO v_rec;
                EXIT WHEN v_intermediate_results%NOTFOUND;

                v_rank := v_rank + 1; -- Increment rank for each row
                v_ranked_reports.EXTEND;
                v_ranked_reports(v_rank) := ranked_report_obj(
                    report_id             => v_rec.report_id,
                    description_id        => v_rec.id,
                    vector_distance       => v_rec.distance,
                    source                => 'similarity_search',
                    similarity_search_rank => v_rec.similarity_search_rank,
                    cross_encoder_rank     => v_rank,
                    overall_rank           => v_rank,
                    match_type             => v_rec.match_type
                );
            END LOOP;

            CLOSE v_intermediate_results;
            lang_data_logger_pkg.log_info(
                'Intermediate results processed successfully.'
            );

            OPEN p_results FOR
                SELECT * FROM TABLE(v_ranked_reports);
        END IF;
        
        lang_data_logger_pkg.log_info('Procedure completed successfully.');
    END get_top_k_closest_report_and_description_similarity_search;

    PROCEDURE re_rank_similarity_search_drilldowns (
        p_query_text            IN VARCHAR2,
        p_ranked_results        IN SYS_REFCURSOR,
        p_re_ranked_results     OUT SYS_REFCURSOR
    ) 
    IS
        -- Record type for intermediate results
        TYPE intermediate_result_rec IS RECORD (
            drilldown_id              VARCHAR2(36),
            id                        VARCHAR2(36),
            enhanced_text              VARCHAR2(4000),
            distance                  NUMBER,
            text                      VARCHAR2(2000),
            title                     VARCHAR2(255),
            match_type                VARCHAR2(100),
            type                      VARCHAR2(36),
            similarity_search_rank    NUMBER
        );
        v_re_ranked_results     JSON;
        v_rec                   intermediate_result_rec;
        v_json_string           VARCHAR2(32767);
        v_result                JSON;
        v_re_ranked_drilldowns  ranked_drilldown_table := ranked_drilldown_table();
        v_rank                  NUMBER := 0;
        v_start_time            NUMBER;
        v_end_time              NUMBER;
        v_pyq_json_object       JSON_OBJECT_T := JSON_OBJECT_T();
        v_pyq_string            VARCHAR2(32767);
        v_texts                             SYS.ODCIVARCHAR2LIST;
        v_all_texts                         VARCHAR2(6100);
        v_n                                 NUMBER;
        TYPE drilldown_by_description IS TABLE OF VARCHAR2(36) INDEX BY VARCHAR2(36);
        TYPE distance_by_description IS TABLE OF NUMBER INDEX BY VARCHAR2(36);
        TYPE similarity_rank_by_description IS TABLE OF NUMBER INDEX BY VARCHAR2(36);
        TYPE match_type_by_description IS TABLE OF VARCHAR2(36) INDEX BY VARCHAR2(36);
        v_drilldown_by_description          drilldown_by_description;
        v_distance_by_description           distance_by_description;
        v_similarity_rank_by_description    similarity_rank_by_description;
        v_match_type_by_description         match_type_by_description;
        v_model_cache       VARCHAR2(4000);
        v_rerank_obj        RERANK_OBJ;
        v_reranked_texts    RERANK_TABLE := RERANK_TABLE();
    BEGIN
        v_n := NVL(
            TO_NUMBER(
                lang_data_config_pkg.get_config_parameter(
                    'LANG_DATA_DRILLDOWN_TEXTS_N'
                )
            ), 
            3
        );
        lang_data_logger_pkg.log_info(
            'Starting re-ranking of top-k drilldowns...'
        );
        LOOP
            FETCH p_ranked_results INTO v_rec;
            EXIT WHEN p_ranked_results%NOTFOUND;

            lang_data_utils_pkg.fetch_top_n_similairty_search_drilldown_text(
                p_query => p_query_text,
                p_drilldown_id => v_rec.drilldown_id,
                p_n => v_n,
                p_texts => v_texts
            );
            v_all_texts := '';
            FOR i in 1 .. v_texts.COUNT LOOP
                IF i = 1 THEN
                    v_all_texts := v_all_texts || v_texts(i);
                ELSE
                    v_all_texts := v_all_texts || ', ' || v_texts(i);
                END IF;
            END LOOP;

            v_drilldown_by_description(v_rec.id) := v_rec.drilldown_id;
            v_distance_by_description(v_rec.id) := v_rec.distance;
            v_similarity_rank_by_description(v_rec.id) := 
                                                v_rec.similarity_search_rank;
            v_match_type_by_description(v_rec.id) := v_rec.match_type;

            v_rerank_obj := RERANK_OBJ(
                doc_id => v_rec.id,
                doc_text => v_all_texts,
                query_text => p_query_text,
                doc_rank => NULL,
                score => NULL
            );
            v_reranked_texts.EXTEND;
            v_reranked_texts(v_reranked_texts.COUNT) := v_rerank_obj;
        END LOOP;

        -- Close the intermediate results cursor
        CLOSE p_ranked_results;
        lang_data_logger_pkg.log_info(
            'Intermediate results processed successfully.'
        );

        v_reranked_texts := lang_data_search_pkg.rerank_texts(
            p_rerank_obj_list => v_reranked_texts
        );

        v_rank := 0;
        FOR i IN 1..v_reranked_texts.COUNT LOOP
                    v_rank := v_rank+1;
                
                    lang_data_logger_pkg.log_info(
                        'After Re-ranking: Drilldown ID: ' 
                        ||v_drilldown_by_description(
                            v_reranked_texts(i).doc_id) 
                        ||' Similarity Search Rank: ' 
                        ||v_similarity_rank_by_description(
                            v_reranked_texts(i).doc_id) 
                        ||' Cross Encoder Rank: ' 
                        ||v_rank
                    );
                    v_re_ranked_drilldowns.EXTEND;
                    v_re_ranked_drilldowns(v_rank) := ranked_drilldown_obj(
                        drilldown_id            => 
                            v_drilldown_by_description(
                                v_reranked_texts(i).doc_id),
                        description_id          => v_reranked_texts(i).doc_id,
                        match_type              => 
                            v_match_type_by_description(
                                v_reranked_texts(i).doc_id)
                    );
        END LOOP;

        -- Step 3: Open the results cursor for the collection
        OPEN p_re_ranked_results FOR
            SELECT * FROM TABLE(v_re_ranked_drilldowns);

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            'Time for re-ranking the top-k drilldowns: ' ||
            (v_end_time - v_start_time) || ' centiseconds.'
        );
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_error(DBMS_UTILITY.FORMAT_ERROR_STACK);
    END re_rank_similarity_search_drilldowns;

    PROCEDURE get_top_k_closest_drilldown_and_description_similarity_search (
        p_query             IN  VARCHAR2,
        p_matched_report_id IN  VARCHAR2,
        p_k                 IN  NUMBER DEFAULT 5,
        p_using_centroid    IN  BOOLEAN DEFAULT FALSE,
        p_domain_id         IN  VARCHAR2 DEFAULT NULL,
        p_results           OUT SYS_REFCURSOR
    ) AS
        -- Record type for intermediate results
        TYPE intermediate_result_rec IS RECORD (
            drilldown_id              VARCHAR2(36),
            id                        VARCHAR2(36),
            enhanced_text              VARCHAR2(4000),
            distance                  NUMBER,
            text                      VARCHAR2(2000),
            title                     VARCHAR2(255),
            match_type                VARCHAR2(100),
            type                      VARCHAR2(36),
            similarity_search_rank    NUMBER
        );

        -- Variables
        v_intermediate_results SYS_REFCURSOR;
        -- Temporary record for fetching rows
        v_rec                intermediate_result_rec;
         -- Collection for final results
        v_ranked_drilldowns  ranked_drilldown_table := ranked_drilldown_table();
        v_re_ranked_results     JSON;
        v_rank                  NUMBER := 0;
        v_re_ranking_enabled    BOOLEAN;
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting get_top_k_closest_drilldown_and_description_similarity_'||
            'search procedure.'
        );

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_RERANK_RESULTS'
            )
        ) = 'true' THEN
            v_re_ranking_enabled := TRUE;
        ELSE
            v_re_ranking_enabled := FALSE;
        END IF;

        -- Step 1: Fetch intermediate results from existing procedure
        lang_data_logger_pkg.log_info('Fetching intermediate results...');
        IF NOT p_using_centroid THEN
            lang_data_utils_pkg.fetch_top_k_similarity_search_drilldowns(
                p_query             => p_query,
                p_matched_report_id => p_matched_report_id,
                p_k                 => p_k,
                p_domain_id         => p_domain_id,
                p_results           => v_intermediate_results
            );
        ELSE
            lang_data_utils_pkg.fetch_top_k_similarity_search_drilldowns_using_centroid(
                p_query             => p_query,
                p_matched_report_id => p_matched_report_id,
                p_k                 => p_k,
                p_domain_id         => p_domain_id,
                p_results           => v_intermediate_results
            );
        END IF;
        lang_data_logger_pkg.log_info(
            'Intermediate results fetched successfully.'
        );

        IF v_re_ranking_enabled AND p_using_centroid=FALSE THEN
            lang_data_search_pkg.re_rank_similarity_search_drilldowns(
                p_query_text => p_query,
                p_ranked_results => v_intermediate_results,
                p_re_ranked_results => p_results
            );
        ELSE
            LOOP
                FETCH v_intermediate_results INTO v_rec;
                EXIT WHEN v_intermediate_results%NOTFOUND;

                v_ranked_drilldowns.EXTEND;	
                v_ranked_drilldowns(v_ranked_drilldowns.COUNT) := 	
                    ranked_drilldown_obj(	
                        drilldown_id       => v_rec.drilldown_id,	
                        description_id     => v_rec.id,	
                        match_type         => v_rec.match_type	
                    );
            END LOOP;

            -- Close the intermediate results cursor
            CLOSE v_intermediate_results;
            lang_data_logger_pkg.log_info(
                'Intermediate results processed successfully.'
            );

            OPEN p_results FOR
                SELECT * FROM TABLE(v_ranked_drilldowns);
        END IF;

        -- Debugging message
        lang_data_logger_pkg.log_info(
            'Successfully populated ranked drilldowns.'
        );
    END get_top_k_closest_drilldown_and_description_similarity_search;

    PROCEDURE re_rank_similarity_search_documents (
        p_query_text            IN VARCHAR2,
        p_ranked_results        IN SYS_REFCURSOR,
        p_re_ranked_results     OUT SYS_REFCURSOR
    ) 
    IS
        -- Record type to match the structure of
        -- fetch_top_k_similarity_search_reports
        TYPE intermediate_result_rec IS RECORD (
            doc_id                 VARCHAR2(50),
            description_id         VARCHAR2(50),
            enhanced_text           VARCHAR2(4000),
            is_report              BOOLEAN,
            vector_distance        NUMBER,
            source                 VARCHAR2(50),
            similarity_search_rank NUMBER,
            cross_encoder_rank     NUMBER,
            overall_rank           NUMBER,
            match_type             VARCHAR2(100)
        );
        v_rec                   intermediate_result_rec;
        v_re_ranked_documents   ranked_doc_table := ranked_doc_table();
        v_rank                  NUMBER := 0;
        v_json_string           VARCHAR2(32767);
        v_result                JSON;
        v_re_ranked_results     JSON;
        v_start_time            NUMBER;
        v_end_time              NUMBER;
        v_is_report             BOOLEAN;
        v_pyq_json_object       JSON_OBJECT_T := JSON_OBJECT_T();
        v_pyq_string            VARCHAR2(32767);
        TYPE document_by_description IS TABLE OF VARCHAR2(36) INDEX BY VARCHAR2(36);
        TYPE distance_by_description IS TABLE OF NUMBER INDEX BY VARCHAR2(36);
        TYPE similarity_rank_by_description IS TABLE OF NUMBER INDEX BY VARCHAR2(36);
        TYPE match_type_by_description IS TABLE OF VARCHAR2(36) INDEX BY VARCHAR2(36);
        TYPE enhanced_text_by_description IS TABLE OF VARCHAR2(4000) INDEX BY VARCHAR2(36);
        TYPE is_report_by_description IS TABLE OF BOOLEAN INDEX BY VARCHAR2(36);
        v_document_by_description           document_by_description;
        v_distance_by_description           distance_by_description;
        v_similarity_rank_by_description    similarity_rank_by_description;
        v_match_type_by_description         match_type_by_description;
        v_enhanced_text_by_description      enhanced_text_by_description;
        v_is_report_by_description          is_report_by_description;
        v_model_cache       VARCHAR2(4000);
        v_rerank_obj        RERANK_OBJ;
        v_reranked_texts    RERANK_TABLE := RERANK_TABLE();
    BEGIN
        lang_data_logger_pkg.log_info('Processing intermediate results...');

        LOOP
            FETCH p_ranked_results INTO v_rec;
            EXIT WHEN p_ranked_results%NOTFOUND;

            lang_data_logger_pkg.log_info(
                'Before Re-ranking: Document ID: ' || v_rec.doc_id ||
                ' Vector Distance: ' || v_rec.vector_distance ||
                ' enhanced _text: ' || v_rec.enhanced_text
            );

            v_rerank_obj := RERANK_OBJ(
                doc_id => v_rec.description_id,
                doc_text => v_rec.enhanced_text,
                query_text => p_query_text,
                doc_rank => NULL,
                score => NULL
            );
            v_reranked_texts.EXTEND;
            v_reranked_texts(v_reranked_texts.COUNT) := v_rerank_obj;

            v_document_by_description(v_rec.description_id) := 
                                                                v_rec.doc_id;
            v_distance_by_description(v_rec.description_id) := 
                                                        v_rec.vector_distance;
            v_similarity_rank_by_description(v_rec.description_id) := 
                                                v_rec.similarity_search_rank;
            v_match_type_by_description(v_rec.description_id) := 
                                                            v_rec.match_type;
            v_enhanced_text_by_description(v_rec.description_id) := 
                                                            v_rec.enhanced_text;
            
            IF v_rec.is_report THEN
                v_is_report_by_description(v_rec.description_id) := TRUE;
            ELSE
                v_is_report_by_description(v_rec.description_id) := FALSE;
            END IF;

            -- Append the current record to the JSON array
        END LOOP;

        CLOSE p_ranked_results;
        lang_data_logger_pkg.log_info(
            'Intermediate results processed successfully.'
        );

        v_start_time := DBMS_UTILITY.GET_TIME;
        v_reranked_texts := lang_data_search_pkg.rerank_texts(
            p_rerank_obj_list => v_reranked_texts
        );

        v_rank := 0;
         FOR i IN 1..v_reranked_texts.COUNT LOOP
            v_rank := v_rank+1;
            lang_data_logger_pkg.log_info(
                'After Re-ranking: Document ID: ' 
                ||v_document_by_description(v_reranked_texts(i).doc_id) 
                ||' Cross Encoder Rank: ' 
                || v_rank
            );
            
            v_re_ranked_documents.EXTEND;
            v_re_ranked_documents(v_rank) := ranked_doc_obj(
                doc_id                  => 
                    v_document_by_description(v_reranked_texts(i).doc_id),
                description_id          => v_reranked_texts(i).doc_id,
                enhanced_text           => 
                    v_enhanced_text_by_description(v_reranked_texts(i).doc_id),
                is_report               => 
                    v_is_report_by_description(v_reranked_texts(i).doc_id),
                vector_distance         => 
                    v_distance_by_description(v_reranked_texts(i).doc_id),
                source                  => 'similarity_search',
                similarity_search_rank  => 
                    v_similarity_rank_by_description(
                        v_reranked_texts(i).doc_id),
                cross_encoder_rank      => v_reranked_texts(i).doc_rank,
                overall_rank            => v_rank,
                match_type              => 
                    v_match_type_by_description(v_reranked_texts(i).doc_id)
            );

        END LOOP;

        OPEN p_re_ranked_results FOR
            SELECT * FROM TABLE(v_re_ranked_documents);

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            'Time for re-ranking the top-k reports: ' ||
            (v_end_time - v_start_time) || ' centiseconds.'
        );
    END re_rank_similarity_search_documents;


    PROCEDURE get_top_k_closest_document_and_description_similarity_search_flat (
        p_query     IN  VARCHAR2,
        p_k         IN  NUMBER DEFAULT 5,
        p_domain_id IN  VARCHAR2 DEFAULT NULL,
        p_results   OUT SYS_REFCURSOR
    ) AS
        -- Record type for intermediate results
        TYPE intermediate_result_rec IS RECORD (
            doc_id                  VARCHAR2(36),
            description_id          VARCHAR2(36),
            enhanced_text            VARCHAR(4000),
            vector_distance         NUMBER,
            text                    VARCHAR2(2000),
            title                   VARCHAR2(255),
            match_type              VARCHAR2(100),
            type                    VARCHAR2(36),
            similarity_search_rank  NUMBER
        );

        -- Variables
        v_report_results          SYS_REFCURSOR;
        v_drilldown_results       SYS_REFCURSOR;
        v_ranked_documents        ranked_doc_table := ranked_doc_table();
        v_rec                     intermediate_result_rec;
        v_rank                    NUMBER := 0;
        v_results                 SYS_REFCURSOR;
        v_re_ranking_enabled      BOOLEAN;
    BEGIN
        lang_data_logger_pkg.log_info(
            'Starting ' ||
            'get_top_k_closest_document_and_description_similarity_search_flat.'
        );

        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_RERANK_RESULTS'
            )
        ) = 'true' THEN
            v_re_ranking_enabled := TRUE;
        ELSE
            v_re_ranking_enabled := FALSE;
        END IF;

        -- Step 1: Fetch top-k similarity search reports
        lang_data_logger_pkg.log_info(
            'Fetching top-k similarity search reports...'
        );
        lang_data_utils_pkg.fetch_top_k_similarity_search_reports(
            p_query     => p_query,
            p_k         => p_k,
            p_domain_id => p_domain_id,
            p_results => v_report_results
        );
        lang_data_logger_pkg.log_info('Fetched report results.');

        -- Step 2: Add report results to ranked documents
        lang_data_logger_pkg.log_info('Processing report results...');
        LOOP
            FETCH v_report_results INTO v_rec;
            EXIT WHEN v_report_results%NOTFOUND;

            v_rank := v_rank + 1;
            lang_data_logger_pkg.log_info(
                'Processing report ID: ' || v_rec.doc_id ||  
                ' Vector Distance: ' || v_rec.vector_distance ||
                ', Rank: ' || v_rank
            );

            v_ranked_documents.EXTEND;
            v_ranked_documents(v_rank) := ranked_doc_obj(
                doc_id                 => v_rec.doc_id,
                description_id         => v_rec.description_id,
                enhanced_text           => v_rec.enhanced_text,
                is_report              => TRUE,
                vector_distance        => v_rec.vector_distance,
                source                 => 'similarity_search',
                similarity_search_rank => v_rec.similarity_search_rank,
                cross_encoder_rank     => v_rank,
                overall_rank           => -1,
                match_type             => v_rec.match_type
            );
        END LOOP;

        CLOSE v_report_results;
        lang_data_logger_pkg.log_info('Finished processing report results.');

        -- Step 3: Fetch top-k similarity search drilldowns
        lang_data_logger_pkg.log_info(
            'Fetching top-k similarity search drilldowns...'
        );
        lang_data_utils_pkg.fetch_top_k_similarity_search_drilldowns(
            p_query             => p_query,
            p_matched_report_id => NULL, -- Include all drilldowns
            p_k                 => p_k,
            p_results           => v_drilldown_results
        );
        lang_data_logger_pkg.log_info('Fetched drilldown results.');

        -- Step 4: Add drilldown results to ranked documents
        lang_data_logger_pkg.log_info('Processing drilldown results...');
        LOOP
            FETCH v_drilldown_results INTO v_rec;
            EXIT WHEN v_drilldown_results%NOTFOUND;

            v_rank := v_rank + 1;
            lang_data_logger_pkg.log_info(
                'Processing drilldown ID: ' || v_rec.doc_id ||  
                ' Vector Distance: ' || v_rec.vector_distance ||
                ', Rank: ' || v_rank
            );

            v_ranked_documents.EXTEND;
            v_ranked_documents(v_rank) := ranked_doc_obj(
                doc_id                 => v_rec.doc_id,
                description_id         => v_rec.description_id,
                enhanced_text           => v_rec.enhanced_text,
                is_report              => FALSE,
                vector_distance        => v_rec.vector_distance,
                source                 => 'similarity_search',
                similarity_search_rank => v_rec.similarity_search_rank,
                cross_encoder_rank     => v_rank,
                overall_rank           => -1,
                match_type             => v_rec.match_type
            );
        END LOOP;

        CLOSE v_drilldown_results;
        lang_data_logger_pkg.log_info('Finished processing drilldown results.');

        -- Step 5: Open the results cursor
        lang_data_logger_pkg.log_info('Opening results cursor...');
        OPEN p_results FOR
            SELECT * FROM TABLE(v_ranked_documents)
            ORDER BY vector_distance ASC
            FETCH FIRST p_k ROWS ONLY;
        
        IF v_re_ranking_enabled THEN
            lang_data_search_pkg.re_rank_similarity_search_documents(
                p_query_text => p_query,
                p_ranked_results => p_results,
                p_re_ranked_results => v_results
            );

            p_results := v_results;
        END IF;

        lang_data_logger_pkg.log_info('Procedure completed successfully.');
    END get_top_k_closest_document_and_description_similarity_search_flat;

    PROCEDURE get_report_matching_text (
        p_match_id IN VARCHAR2,
        p_matched_text OUT VARCHAR2,
        p_enhanced_text OUT VARCHAR2
    ) IS
    BEGIN
        SELECT text, enhanced_text
        INTO p_matched_text, p_enhanced_text
        FROM (
            SELECT text, enhanced_text, 1 AS priority
            FROM langdata$reportdescriptions
            WHERE id = p_match_id
            UNION ALL
            SELECT query_text, enhanced_query_text, 2 AS priority
            FROM langdata$samplequeries
            WHERE id = p_match_id
            UNION ALL
            SELECT query_text, augmented_query_text, 3 AS priority
            FROM langdata$searchrecords
            WHERE id = p_match_id
        )
        WHERE ROWNUM = 1
        ORDER BY priority;

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            p_matched_text := NULL;
            p_enhanced_text := NULL;
    END get_report_matching_text;

    PROCEDURE get_drilldown_matching_text (
        p_match_id IN VARCHAR2,
        p_matched_text OUT VARCHAR2,
        p_enhanced_text OUT VARCHAR2
    ) IS
    BEGIN
        SELECT text, enhanced_text
        INTO p_matched_text, p_enhanced_text
        FROM (
            SELECT text, enhanced_text, 1 AS priority
            FROM langdata$drilldowndescriptions
            WHERE id = p_match_id
            UNION ALL
            SELECT query_text, enhanced_query_text, 2 AS priority
            FROM langdata$samplequeries
            WHERE id = p_match_id
            UNION ALL
            SELECT query_text, augmented_query_text, 3 AS priority
            FROM langdata$searchrecords
            WHERE id = p_match_id
        )
        WHERE ROWNUM = 1
        ORDER BY priority;

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            p_matched_text := NULL;
            p_enhanced_text := NULL;
    END get_drilldown_matching_text;

    PROCEDURE search_from_query (
        p_query             IN VARCHAR2,
        p_domain            IN VARCHAR2 DEFAULT NULL,
        p_query_vector      IN VECTOR DEFAULT NULL,
        p_search_id         OUT VARCHAR2
    )
    AS
        -- Record type for intermediate results
        TYPE ranked_report_result_rec IS RECORD(
            report_id              VARCHAR2(36),
            description_id         VARCHAR2(36),
            vector_distance        NUMBER,
            source                 VARCHAR2(100),
            similarity_search_rank NUMBER,
            cross_encoder_rank     NUMBER,
            overall_rank           NUMBER,
            match_type             VARCHAR2(100)
        );
        TYPE ranked_drilldown_result_rec IS RECORD(
            drilldown_id           VARCHAR2(36),
            description_id         VARCHAR2(36),
            match_type             VARCHAR2(100)
        );
        TYPE ranked_doc_result_rec IS RECORD (
            doc_id                 VARCHAR2(50),
            description_id         VARCHAR2(50),
            enhanced_text           VARCHAR2(4000),
            is_report              BOOLEAN,
            vector_distance        NUMBER,
            source                 VARCHAR2(50),
            similarity_search_rank NUMBER,
            cross_encoder_rank     NUMBER,
            overall_rank           NUMBER,
            match_type             VARCHAR2(100)
        );

        v_search_id                 VARCHAR2(36);
        v_domain_id                 VARCHAR2(36) := NULL;
        v_using_centroid            BOOLEAN := FALSE;
        v_ner_augmentation_enabled  BOOLEAN := FALSE;
        v_search_type               VARCHAR2(36);
        v_all_named_entities        SYS_REFCURSOR;
        v_named_entity_id           VARCHAR2(255);
        v_named_entity_name         VARCHAR2(36);
        v_all_labels_str            VARCHAR2(4000) := '';
        v_report_similarity_k       NUMBER;
        v_drilldown_similarity_k    NUMBER;
        v_prev_context_len          NUMBER;
        v_post_context_len          NUMBER;
        v_recognized_entities JSON;
        v_recognized_entities_str   VARCHAR2(4000);
        v_ner_augmented_query       VARCHAR2(4000);
        v_similarity                NUMBER; 
        -- Augmented query text
        v_augmented_query           VARCHAR2(4000);
        -- Generated embedding vector
        v_query_vector              VECTOR;
        -- MD5 hash of the normalized query
        v_query_md5                 VARCHAR2(32);
        -- Normalized query text
        v_normalized_query          VARCHAR2(2000);
        -- Cursor for top similarity reports
        v_top_reports               SYS_REFCURSOR;
        v_report_rec                ranked_report_result_rec;
        v_report_title              VARCHAR2(255);
        v_report_description          VARCHAR2(2000);
        v_enhanced_report_description  VARCHAR2(4000);
        v_report_match_document     JSON;
        v_idx                       NUMBER := 0;
        v_report_filter_values      JSON_ARRAY_T;
        v_report_filter_values_flat      JSON;
        -- Cursor for top drilldowns
        v_top_drilldowns            SYS_REFCURSOR;
        v_drilldown_rec             ranked_drilldown_result_rec;
        v_drilldown_filter_values   JSON_ARRAY_T;
        v_drilldown_filter_values_flat      JSON;
        v_drilldown_title           VARCHAR2(255);
        v_drilldown_description          VARCHAR2(2000);
        v_enhanced_drilldown_description VARCHAR2(4000);
        v_matched_drilldown_id      VARCHAR2(36);
        v_matched_drilldown_desc_id VARCHAR2(36);
        v_drilldown_match_document  JSON;
        v_top_documents             SYS_REFCURSOR;
        v_doc_rec                   ranked_doc_result_rec;
        v_temp_source               VARCHAR2(50);
        v_parent_report_id          VARCHAR2(36);
        v_parent_report_desc_id     VARCHAR2(36);
        v_current_result            JSON_OBJECT_T;
        v_overall_results           JSON_ARRAY_T := JSON_ARRAY_T();
        v_overall_results_str       CLOB;
        v_current_user              VARCHAR2(255);
        v_augmented_tokens          JSON;
        v_drilldown_match_type      VARCHAR2(100);

        -- Time Profiling
        v_start_time                NUMBER;
        v_end_time                  NUMBER;
        v_temp_time                 NUMBER;
        v_temp_str                  VARCHAR2(32767); -- ToDo: Remove this.

        -- NER Updates
        v_match_document_arr JSON_ARRAY_T := JSON_ARRAY_T();
        v_filter_cnt NUMBER := 0;
        v_num_filters SYS.ODCIVARCHAR2LIST := SYS.ODCIVARCHAR2LIST();
        v_num_filters_idx   NUMBER := 0;
        v_report_filter_values_cnt NUMBER := 0;
        v_drilldown_filter_values_cnt NUMBER := 0;

        v_json_object JSON_OBJECT_T;
        v_json_arr    JSON_ARRAY_T;
        v_overall_result_object JSON_OBJECT_T;

        v_question_id     VARCHAR2(36);
        v_existing_vector VECTOR;

        -- Question Stats updates
        v_timestamp            TIMESTAMP;
        v_similarity_threshold NUMBER;

    BEGIN
        v_start_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info('Starting search_from_query procedure.');
        IF p_query IS NULL THEN
            lang_data_logger_pkg.log_error(
                'Invalid input of null query'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        IF p_domain IS NOT NULL THEN
            v_domain_id := lang_data_utils_pkg.get_domain_id(
                p_domain_name => p_domain
            );

            IF v_domain_id IS NULL THEN
                lang_data_errors_pkg.raise_error(
                    lang_data_errors_pkg.c_invalid_domain
                );
            END IF;
        END IF;

        IF p_query_vector IS NOT NULL 
            AND NOT lang_data_utils_pkg.validate_vector(p_query_vector) THEN
            lang_data_logger_pkg.log_error(
                'Invalid dimensions of input vector'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;
            
        IF LENGTH(p_query) > 2000 THEN
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_max_text_length_exceeded
            );
        END IF;

        v_search_type := lang_data_config_pkg.get_config_parameter(
            'LANG_DATA_SEARCH_TYPE'
            );

        IF LOWER(v_search_type) = 'flat' and v_using_centroid THEN
            lang_data_logger_pkg.log_error(
                'Centroid version is not supported for flat search'
            );
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        v_search_id := lang_data_utils_pkg.generate_id();

        v_report_similarity_k := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_REPORT_SIMILARITY_K'
            )
        );
        v_drilldown_similarity_k := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_DRILLDOWN_SIMILARITY_K'
            )
        );
        v_prev_context_len := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_PREV_CONTEXT_LEN'
            )
        );
        v_post_context_len := TO_NUMBER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_POST_CONTEXT_LEN'
            )
        );
        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_USING_CENTROID'
            )
        ) = 'true' THEN
            v_using_centroid := TRUE;
        ELSE
            v_using_centroid := FALSE;
        END IF;


        IF LOWER(
            lang_data_config_pkg.get_config_parameter(
                'LANG_DATA_NER_AUGMENTATION_ENABLED'
            )
        ) = 'true' THEN
            v_ner_augmentation_enabled := TRUE;
        ELSE
            v_ner_augmentation_enabled := FALSE;
        END IF;

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time for pre-checks: ' || (v_end_time - v_start_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );
        v_temp_time := DBMS_UTILITY.GET_TIME;

        -- Step 1: Fetch all entities
        lang_data_named_entities_pkg.get_all_named_entities(
            v_all_named_entities
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to fetch all named entities: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );
        v_temp_time := DBMS_UTILITY.GET_TIME;

        -- Step 2: NER processing
        -- Loop through the cursor
        LOOP
            FETCH v_all_named_entities
            INTO v_named_entity_id, v_named_entity_name;
            EXIT WHEN v_all_named_entities%NOTFOUND;

            -- Concatenate names with commas
            IF v_all_labels_str IS NOT NULL THEN
                v_all_labels_str := v_all_labels_str || ',' ||
                                    v_named_entity_name;
            ELSE
                v_all_labels_str := v_named_entity_name;
            END IF;
        END LOOP;

        -- Close the cursor
        CLOSE v_all_named_entities;

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to store all named entities: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );
        v_temp_time := DBMS_UTILITY.GET_TIME;

        lang_data_named_entities_pkg.get_all_entities_using_ner(
            p_query,
            v_all_labels_str,
            v_prev_context_len,
            v_post_context_len,
            v_recognized_entities
        );
        v_recognized_entities_str := JSON_SERIALIZE(
            v_recognized_entities RETURNING VARCHAR2
        );
        lang_data_logger_pkg.log_debug(
            'Recognized entities: ' || v_recognized_entities_str
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to get_all_entities_using_ner: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );

        v_ner_augmented_query := p_query;

        IF v_ner_augmentation_enabled AND v_recognized_entities IS NOT NULL THEN
            v_temp_time := DBMS_UTILITY.GET_TIME;

            v_ner_augmented_query := lang_data_utils_pkg.augment_text_with_ner_entities(
                p_query, v_recognized_entities
            );
            lang_data_logger_pkg.log_debug(
                'Query augmented with NER: ' || v_ner_augmented_query
            );

            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_logger_pkg.log_info(
                '---------------------------------------------------------------' ||
                ' Execution Time to augment_text_with_ner_entities: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
                '---------------------------------------------------------------'
            );

            v_temp_time := DBMS_UTILITY.GET_TIME;
        END IF;

        -- Step 3: Augment the query using the augment_text function
        lang_data_utils_pkg.augment_text(
            p_text             =>  v_ner_augmented_query,
            p_domain_id        =>  v_domain_id,
            p_augmented_text   =>  v_augmented_query,
            p_augmented_tokens =>  v_augmented_tokens
        );

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to Augment the query text: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );
        v_temp_time := DBMS_UTILITY.GET_TIME;

        -- Step 4: Get query embedding and query md5
        IF p_query_vector IS NULL THEN
            v_query_vector := lang_data_utils_pkg.get_embedding(
                v_augmented_query
            );
        ELSE
            v_query_vector := p_query_vector;
        END IF;

        v_end_time := DBMS_UTILITY.GET_TIME;

        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to generate query embedding: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        ); 

        v_normalized_query := LOWER(TRIM(p_query));
        -- Generate the MD5 hash of the normalized query
        v_query_md5 := DBMS_CRYPTO.HASH(
            UTL_RAW.CAST_TO_RAW(v_normalized_query), 2
        ); 

        lang_data_search_pkg.print_filters_temptable();

        IF LOWER(v_search_type) = 'hierarchical' THEN
            -- Step 5: Similarity search for reports
            v_temp_time := DBMS_UTILITY.GET_TIME;
            get_top_k_closest_report_and_description_similarity_search(
                p_query           => v_augmented_query,
                p_k               => v_report_similarity_k,
                p_using_centroid  => v_using_centroid,
                p_domain_id       => v_domain_id,
                p_results         => v_top_reports
            );

            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_logger_pkg.log_info(
                '---------------------------------------------------------------' ||
                ' Execution Time to get_top_k_closest_reports/descriptions: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
                '---------------------------------------------------------------'
            );
            v_temp_time := DBMS_UTILITY.GET_TIME;

            -- Step 6: Process top reports and drilldowns
            LOOP
                FETCH v_top_reports INTO v_report_rec;
                EXIT WHEN v_top_reports%NOTFOUND;
                v_idx := v_idx + 1;
                v_report_rec.overall_rank := v_idx;
                v_report_title := null;
                v_report_match_document := JSON('{}');
                v_matched_drilldown_id := null;
                v_drilldown_title := null;
                v_drilldown_match_document := JSON('{}');
                v_matched_drilldown_desc_id := null;
                v_drilldown_description := null;

                SELECT match_document
                INTO v_report_match_document
                FROM langdata$reports
                WHERE id = v_report_rec.report_id;

                v_json_object := JSON_OBJECT_T();
                v_json_object.PUT('match_document', v_report_match_document);
                v_json_object.PUT('document_type', 'report');
                v_json_object.PUT('document_id', v_report_rec.report_id);
                v_match_document_arr.APPEND(v_json_object);
                lang_data_logger_pkg.log_info(
                    '(search_from_query): Report with ID = ' || v_report_rec.report_id || ' Appended!'
                );

                get_top_k_closest_drilldown_and_description_similarity_search(
                    p_query             => v_augmented_query,
                    p_matched_report_id => v_report_rec.report_id,
                    p_k                 => v_drilldown_similarity_k,
                    p_using_centroid    => v_using_centroid,
                    p_domain_id         => v_domain_id,
                    p_results           => v_top_drilldowns
                );

                LOOP
                    FETCH v_top_drilldowns INTO v_drilldown_rec;
                    EXIT WHEN v_top_drilldowns%NOTFOUND;
                    v_matched_drilldown_id := v_drilldown_rec.drilldown_id;
                    v_matched_drilldown_desc_id := 
                        v_drilldown_rec.description_id;
                    v_drilldown_match_type := v_drilldown_rec.match_type;

                    SELECT match_document
                    INTO v_drilldown_match_document
                    FROM langdata$drilldowndocuments
                    WHERE id = v_matched_drilldown_id;

                    v_json_object := JSON_OBJECT_T();
                    v_json_object.PUT(
                        'match_document', 
                        v_drilldown_match_document
                    );
                    v_json_object.PUT('document_type', 'drilldown');
                    v_json_object.PUT('document_id', v_matched_drilldown_id);
                    v_match_document_arr.APPEND(v_json_object);
                    lang_data_logger_pkg.log_info(
                        '(search_from_query): Drilldown with ID = ' || v_matched_drilldown_id || ' Appended!'
                    );

                    EXIT; -- Only process the top drilldown
                END LOOP;

                CLOSE v_top_drilldowns;

                v_current_result := JSON_OBJECT_T();
                v_current_result.PUT(
                    'report_id', v_report_rec.report_id
                );
                IF v_report_rec.report_id IS NOT NULL THEN
                    SELECT title
                    INTO v_report_title
                    FROM langdata$reports
                    WHERE id = v_report_rec.report_id;
                END IF;

                v_current_result.PUT(
                    'report_match_type', v_report_rec.match_type
                );
                v_current_result.PUT(
                    'report_title', v_report_title
                );
                v_current_result.PUT(
                    'report_match_document', v_report_match_document
                );
                v_current_result.PUT(
                    'report_description_id', v_report_rec.description_id
                );
                IF v_report_rec.description_id IS NOT NULL THEN
                    lang_data_search_pkg.get_report_matching_text(
                        p_match_id      =>  v_report_rec.description_id,
                        p_matched_text  =>  v_report_description,
                        p_enhanced_text  =>  v_enhanced_report_description
                    );
                END IF;
                v_current_result.PUT(
                    'report_description', v_report_description
                );
                v_current_result.PUT(
                    'enhanced_report_description', v_enhanced_report_description
                );
                v_current_result.PUT('drilldown_id', v_matched_drilldown_id);
                IF v_matched_drilldown_id IS NOT NULL THEN
                    SELECT title
                    INTO v_drilldown_title
                    FROM langdata$drilldowndocuments
                    WHERE id = v_matched_drilldown_id;
                END IF;
                v_current_result.PUT(
                    'drilldown_title', v_drilldown_title
                );
                v_current_result.PUT(
                    'drilldown_match_type', v_drilldown_match_type
                );
                v_current_result.PUT(
                    'drilldown_match_document', v_drilldown_match_document
                );
                v_current_result.PUT(
                    'drilldown_description_id', v_matched_drilldown_desc_id
                );
                IF v_matched_drilldown_desc_id IS NOT NULL THEN
                    lang_data_search_pkg.get_drilldown_matching_text(
                        p_match_id      =>  v_matched_drilldown_desc_id,
                        p_matched_text  =>  v_drilldown_description,
                        p_enhanced_text  =>  v_enhanced_drilldown_description
                    );
                END IF;
                v_current_result.PUT(
                    'drilldown_description', v_drilldown_description
                );
                v_current_result.PUT(
                    'enhanced_drilldown_description',
                    v_enhanced_drilldown_description
                );
                v_current_result.PUT(
                    'source', v_report_rec.source
                );
                v_current_result.PUT(
                    'similarity_search_rank',
                    v_report_rec.similarity_search_rank
                );
                v_current_result.PUT(
                    'cross_encoder_rank',
                    v_report_rec.cross_encoder_rank
                );
                v_current_result.PUT('overall_rank', v_report_rec.overall_rank);
                v_current_result.PUT(
                    'vector_distance', v_report_rec.vector_distance
                );
                v_overall_results.APPEND(v_current_result);
                
            END LOOP;
            
            v_end_time := DBMS_UTILITY.GET_TIME;
            lang_data_logger_pkg.log_info(
                '---------------------------------------------------------------' ||
                ' Execution Time to Process top reports and drilldowns: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
                '---------------------------------------------------------------'
            );
            v_temp_time := DBMS_UTILITY.GET_TIME;
        ELSE
            -- ToDo: add the same procedure calls here.
            get_top_k_closest_document_and_description_similarity_search_flat(
                p_query     => v_augmented_query,
                p_k         => v_report_similarity_k,
                p_domain_id => v_domain_id,
                p_results   => v_top_documents
            );
            LOOP
                FETCH v_top_documents INTO v_doc_rec;
                EXIT WHEN v_top_documents%NOTFOUND;
                
                v_idx := v_idx + 1;
                v_doc_rec.overall_rank := v_idx;
                v_temp_source := 'Flat, ' || v_doc_rec.source;
                v_doc_rec.source := v_temp_source;
                
                IF v_doc_rec.is_report THEN
                    SELECT match_document
                    INTO v_report_match_document
                    FROM langdata$reports
                    WHERE id = v_doc_rec.doc_id;

                    -- Append the match document to the JSON_ARRAY_T, 
                    -- to be passed in process_filters
                    v_json_object := JSON_OBJECT_T();
                    v_json_object.PUT(
                        'match_document', 
                        v_report_match_document
                    );
                    v_json_object.PUT('document_type', 'report');
                    v_json_object.PUT('document_id', v_doc_rec.doc_id);
                    v_match_document_arr.APPEND(v_json_object);

                    -- No drilldown filter values for a report
                    v_json_object := JSON_OBJECT_T();
                    v_json_object.PUT('match_document', JSON_OBJECT_T('{}'));
                    v_json_object.PUT('document_type', 'drilldown');
                    v_match_document_arr.APPEND(v_json_object);
                ELSE
                    -- Fetch the associated report for the drilldown
                    SELECT rt.id, rt.match_document
                    INTO v_parent_report_id, v_report_match_document
                    FROM langdata$reports rt
                    JOIN langdata$drilldowndocuments dd
                    ON rt.id = dd.report_id
                    WHERE dd.id = v_doc_rec.doc_id;

                    -- Fetch the parent report description ID
                    SELECT id INTO v_parent_report_desc_id
                    FROM langdata$reportdescriptions
                    WHERE report_id = v_parent_report_id
                    AND ROWNUM = 1;

                    -- Append the match document to the JSON_ARRAY_T, 
                    -- to be passed in process_filters
                    v_json_object := JSON_OBJECT_T();
                    v_json_object.PUT(
                        'match_document', 
                        v_report_match_document
                    );
                    v_json_object.PUT('document_type', 'report');
                    v_json_object.PUT('document_id', v_parent_report_id);
                    v_match_document_arr.APPEND(v_json_object);
                    
                    SELECT match_document
                    INTO v_drilldown_match_document
                    FROM langdata$drilldowndocuments
                    WHERE id = v_doc_rec.doc_id;

                    -- Append the match document to the JSON_ARRAY_T, 
                    -- to be passed in process_filters
                    v_json_object := JSON_OBJECT_T();
                    v_json_object.PUT(
                        'match_document', 
                        v_drilldown_match_document
                    );
                    v_json_object.PUT('document_type', 'drilldown');
                    v_json_object.PUT('document_id', v_doc_rec.doc_id);
                    v_match_document_arr.APPEND(v_json_object);
                END IF;

                v_current_result := JSON_OBJECT_T();

                -- Populate the JSON object using PUT
                v_current_result.PUT('report_id', 
                    CASE
                        WHEN v_doc_rec.is_report THEN v_doc_rec.doc_id
                        ELSE v_parent_report_id
                    END
                );
                IF v_doc_rec.is_report AND v_doc_rec.doc_id IS NOT NULL THEN
                    SELECT title
                    INTO v_report_title
                    FROM langdata$reports
                    WHERE id = v_doc_rec.doc_id;
                ELSIF v_parent_report_id IS NOT NULL THEN
                    SELECT title
                    INTO v_report_title
                    FROM langdata$reports
                    WHERE id = v_parent_report_id;
                END IF;
                v_current_result.PUT(
                    'report_title', v_report_title
                );
                v_current_result.PUT(
                    'report_match_document', v_report_match_document
                );
                v_current_result.PUT('report_description_id', 
                    CASE
                        WHEN v_doc_rec.is_report THEN v_doc_rec.description_id
                        ELSE v_parent_report_desc_id
                    END
                );
                IF v_doc_rec.is_report
                  AND v_doc_rec.description_id IS NOT NULL THEN
                    lang_data_search_pkg.get_report_matching_text(
                        p_match_id      =>  v_doc_rec.description_id,
                        p_matched_text  =>  v_report_description,
                        p_enhanced_text  =>  v_enhanced_report_description
                    );
                ELSIF v_parent_report_desc_id IS NOT NULL THEN
                    lang_data_search_pkg.get_report_matching_text(
                        p_match_id      =>  v_parent_report_desc_id,
                        p_matched_text  =>  v_report_description,
                        p_enhanced_text  =>  v_enhanced_report_description
                    );
                END IF;
                v_current_result.PUT(
                    'report_description', v_report_description
                );
                v_current_result.PUT(
                    'enhanced_report_description', v_enhanced_report_description
                );
                v_current_result.PUT('drilldown_id', 
                    CASE
                        WHEN NOT v_doc_rec.is_report
                        THEN v_doc_rec.doc_id
                        ELSE NULL
                    END
                );
                IF NOT v_doc_rec.is_report AND v_doc_rec.doc_id IS NOT NULL THEN
                    SELECT title
                    INTO v_drilldown_title
                    FROM langdata$drilldowndocuments
                    WHERE id = v_doc_rec.doc_id;
                END IF;
                v_current_result.PUT(
                    'drilldown_title', v_drilldown_title
                );
                v_current_result.PUT(
                    'drilldown_match_document', v_drilldown_match_document
                );
                v_current_result.PUT('drilldown_description_id', 
                    CASE
                        WHEN NOT v_doc_rec.is_report THEN v_doc_rec.description_id
                        ELSE NULL
                    END
                );
                IF NOT v_doc_rec.is_report
                  AND v_doc_rec.description_id IS NOT NULL THEN
                    lang_data_search_pkg.get_drilldown_matching_text(
                        p_match_id      =>  v_doc_rec.description_id,
                        p_matched_text  =>  v_drilldown_description,
                        p_enhanced_text  =>  v_enhanced_drilldown_description
                    );
                END IF;
                v_current_result.PUT(
                    'drilldown_description', v_drilldown_description
                );
                v_current_result.PUT(
                    'enhanced_drilldown_description',
                    v_enhanced_drilldown_description
                );
                v_current_result.PUT('source', v_doc_rec.source);
                v_current_result.PUT(
                    'similarity_search_rank', v_doc_rec.similarity_search_rank
                );
                v_current_result.PUT(
                    'cross_encoder_rank', v_doc_rec.cross_encoder_rank
                );
                v_current_result.PUT('overall_rank', v_doc_rec.overall_rank);
                v_current_result.PUT(
                    'vector_distance', v_doc_rec.vector_distance
                );

                -- Append the JSON object to the JSON array
                v_overall_results.APPEND(v_current_result);
            END LOOP;
        END IF;

        lang_data_search_pkg.process_filters(
            p_original_query => p_query,
            p_augmented_query => v_augmented_query,
            p_match_document_arr => v_match_document_arr,
            p_report_filter_values => v_report_filter_values,
            p_drilldown_filter_values => v_drilldown_filter_values,
            p_num_filters => v_num_filters
        );

        lang_data_search_pkg.print_filters_temptable();

        lang_data_logger_pkg.log_info(
            'After process_filters(), v_report_filter_values = ' || 
            v_report_filter_values.to_string
        );

        lang_data_logger_pkg.log_info(
            'After process_filters(), v_report_filter_values.size = ' || 
            v_report_filter_values.get_size
        );

        lang_data_logger_pkg.log_info(
            'After process_filters(), v_drilldown_filter_values = ' || 
            v_drilldown_filter_values.to_string
        );

        lang_data_logger_pkg.log_info(
            'After process_filters(), v_drilldown_filter_values.size = ' || 
            v_drilldown_filter_values.get_size
        );

        -- Process the Report/Drilldown Filter Values
        -- Initialize the required counters and indices
        v_filter_cnt := 0;                  -- Stores the filter count
        v_num_filters_idx := 1;             -- Index of v_num_filters
        v_report_filter_values_cnt := 0;    -- Index of v_report_filter_values
        v_drilldown_filter_values_cnt := 0; -- Index of v_drilldown_filter_values

        -- Iterate through the collected Overall Results
        -- NOTE: v_num_filters array will have filter count of the
        --       report on the nth index and that of the drilldown on the
        --       (n+1)th index, i.e, the following index. Hence, we 
        --       increment the index of v_num_filters, v_num_filters_idx, 
        --       2 times in each iteration for v_overall_results, one for 
        --       each report and one for the its best drilldown. 
        FOR i in 0 .. v_overall_results.get_size - 1 LOOP
            -- Get the filter count of the the report
            v_filter_cnt := v_num_filters(v_num_filters_idx);

            -- GET(i) return JSON_ELEMENT_T, 
            -- so need to cast to JSON_OBJECT_T
            v_overall_result_object := TREAT(
                v_overall_results.get(i) AS JSON_OBJECT_T
            );

            v_parent_report_id := 
                                v_overall_result_object.get_string('report_id');
            v_matched_drilldown_id := 
                            v_overall_result_object.get_string('drilldown_id');

            IF v_parent_report_id IS NOT NULL THEN
                -- For report i
                -- Get the sub-list consists of `v_filter_cnt` elements in
                -- v_report_filter_values starting from index
                -- v_report_filter_values_cnt
                v_json_arr := JSON_ARRAY_T();
                FOR j in 1 .. v_filter_cnt LOOP
                    v_json_object := TREAT(v_report_filter_values.get(v_report_filter_values_cnt) AS JSON_OBJECT_T);
                    v_json_arr.APPEND(v_json_object);

                    v_report_filter_values_cnt := 
                                                v_report_filter_values_cnt + 1;
                END LOOP;

                v_overall_result_object.PUT(
                    'report_filter_values', 
                    v_json_arr
                );
                
                v_num_filters_idx := v_num_filters_idx + 1;
            END IF;

            -- Only update overall_result_object with drilldown_filter_values 
            -- when drilldown present, otherwise it messes with the 
            -- v_drilldown_filter_values_cnt variable.
            IF v_matched_drilldown_id IS NOT NULL THEN
                -- Re-initialize for drilldowns
                v_filter_cnt := v_num_filters(v_num_filters_idx);

                v_json_arr := JSON_ARRAY_T();
                FOR j in 1 .. v_filter_cnt LOOP
                    v_json_arr.APPEND(v_drilldown_filter_values.get(
                            v_drilldown_filter_values_cnt
                        ));
                    v_drilldown_filter_values_cnt := 
                                            v_drilldown_filter_values_cnt + 1;
                END LOOP;

                v_overall_result_object.PUT(
                    'drilldown_filter_values', 
                    v_json_arr
                );

                v_num_filters_idx := v_num_filters_idx + 1;
            ELSE 
                lang_data_logger_pkg.log_info(
                    'No drilldown for report with ID = ' || v_parent_report_id
                );
            END IF;
            

            -- Put the updated v_overall_result_object into the 
            -- overall results. TRUE is for override, just to be safe.
            v_overall_results.put(i, v_overall_result_object, TRUE);
        END LOOP;

        v_end_time := DBMS_UTILITY.GET_TIME;
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time for process_filters: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );
        v_temp_time := DBMS_UTILITY.GET_TIME;

        IF v_top_reports%ISOPEN THEN
            CLOSE v_top_reports;
        END IF;

        IF v_top_drilldowns%ISOPEN THEN
            CLOSE v_top_drilldowns;
        END IF;

        IF v_top_documents%ISOPEN THEN
            CLOSE v_top_documents;
        END IF;

        -- Step 7: Update the question statistics.'

        -- Note timestamp for `langdata$question_stats.last_asked_at`.
        v_timestamp := SYSTIMESTAMP;

        BEGIN
            -- Get the most similar question
            SELECT question_id, question_vector
            INTO v_question_id, v_existing_vector
            FROM langdata$question_stats
            ORDER BY VECTOR_DISTANCE(question_vector, v_query_vector)
            FETCH FIRST 1 ROWS ONLY;

            -- Compute similarity
            v_similarity := 1 - VECTOR_DISTANCE(
                v_existing_vector, v_query_vector);
            v_similarity_threshold := TO_NUMBER(
                lang_data_config_pkg.get_config_parameter(
                    'LANG_DATA_QUESTION_SIMILARITY_THRESHOLD'
            ));

            IF v_similarity >= v_similarity_threshold THEN
                UPDATE langdata$question_stats
                SET asked_count = asked_count + 1,
                    last_asked_at = v_timestamp
                WHERE question_id = v_question_id;
            ELSE
                v_question_id := lang_data_utils_pkg.generate_id();
                INSERT INTO langdata$question_stats (
                    question_id, question_text, question_vector, last_asked_at
                )
                VALUES (v_question_id, p_query, v_query_vector, v_timestamp);
            END IF;

        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                -- No similar question found, continue to insert new one
                v_question_id := lang_data_utils_pkg.generate_id();
                INSERT INTO langdata$question_stats (
                    question_id, question_text, question_vector, last_asked_at
                )
                VALUES (v_question_id, p_query, v_query_vector, v_timestamp);
        END;
        
        -- Step 8: Insert search record
        SELECT SYS_CONTEXT('USERENV', 'SESSION_USER')
        INTO v_current_user FROM dual;

        v_overall_results_str := v_overall_results.TO_CLOB;

        EXECUTE IMMEDIATE 
            'INSERT INTO langdata$searchrecords (
                id, query_text, recognized_entities, report_matches, 
                query_vector, query_md5, search_type, username, 
                augmented_query_text, augmented_tokens, domain_id, question_id
            ) VALUES (
                :search_id,
                :query,
                :recognized_entities,
                JSON(:overall_results),
                :query_vector,
                :query_md5,
                :search_type,
                :current_user,
                :augmented_query,
                :augmented_tokens,
                :domain_id,
                :question_id
            )'
        USING 
            v_search_id,
            p_query,
            v_recognized_entities_str,
            v_overall_results_str,
            v_query_vector,
            v_query_md5,
            v_search_type,
            v_current_user,
            v_augmented_query,
            v_augmented_tokens,
            v_domain_id,
            v_question_id;

        lang_data_logger_pkg.log_info('Search record inserted successfully.');
        
        v_end_time := DBMS_UTILITY.GET_TIME;

        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to insert the search record: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );

        v_temp_time := DBMS_UTILITY.GET_TIME;

        -- After inserting new record into langdata$searchrecords
        update_stats_from_search(p_search_id => v_search_id);

        lang_data_logger_pkg.log_info('Statistics for report/drilldown'
        || ' updated successfully.');

        v_end_time := DBMS_UTILITY.GET_TIME;
        
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to update rank statistics for report/drilldown: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );

        v_temp_time := DBMS_UTILITY.GET_TIME;

        -- Refresh search counts from new search records
        update_report_search_count(p_search_id => v_search_id);

        lang_data_logger_pkg.log_info('Search count for reports'
        || ' updated successfully.');

        v_end_time := DBMS_UTILITY.GET_TIME;
        
        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time to update search count for reports: ' || (v_end_time - v_temp_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );

        v_temp_time := DBMS_UTILITY.GET_TIME;

        lang_data_logger_pkg.log_info(
            '---------------------------------------------------------------' ||
            ' Execution Time Overall: ' || (v_temp_time - v_start_time) || ' centiseconds ' ||
            '---------------------------------------------------------------'
        );

        p_search_id := v_search_id;

    EXCEPTION
        WHEN OTHERS THEN
            IF SQLCODE IN (
                lang_data_errors_pkg.c_max_text_length_exceeded,
                lang_data_errors_pkg.c_invalid_parameters_code
            ) THEN
                -- This is an expected error, just raise it
                RAISE;
            ELSE
                -- Log unknown errors as fatal and raise the generic error code
                lang_data_logger_pkg.log_fatal(
                    'An error occurred while searching for query: ' ||
                    p_query || '. Error: ' || SQLERRM
                );
                RAISE;
            END IF;

    END search_from_query;

    PROCEDURE update_json_analytics(
            p_table   IN VARCHAR2,
            p_id_col  IN VARCHAR2,
            p_id_val  IN VARCHAR2,
            p_key     IN VARCHAR2,
            p_delta   IN NUMBER
        ) IS
            v_json_text   CLOB;
            v_obj JSON_OBJECT_T;
            e_lock_timeout EXCEPTION;
            PRAGMA EXCEPTION_INIT(e_lock_timeout, -30006);  -- ORA-00054
        BEGIN
            BEGIN
                EXECUTE IMMEDIATE '
                    SELECT JSON_SERIALIZE(analytics_data RETURNING CLOB)
                    FROM ' || p_table || ' WHERE ' || p_id_col || 
                    ' = :id FOR UPDATE WAIT 5' 
                INTO v_json_text
                USING p_id_val;
            EXCEPTION
                WHEN e_lock_timeout THEN
                    lang_data_logger_pkg.log_warn(
                        'Lock timeout: Skipping analytics update for ' || 
                        p_table || ' ID: ' || p_id_val
                    );
                    RETURN;  -- Skip rest of logic if lock couldn't be acquired
            END;

            IF v_json_text IS NULL OR TRIM(v_json_text) = '{}' THEN
                v_obj := JSON_OBJECT_T();
            ELSE
                v_obj := JSON_OBJECT_T.parse(v_json_text);
            END IF;

            v_obj.put(p_key, NVL(v_obj.get_number(p_key), 0) + p_delta);

            EXECUTE IMMEDIATE '
                UPDATE ' || p_table || '
                SET analytics_data = :data, updated_at = CURRENT_TIMESTAMP
                WHERE ' || p_id_col || ' = :id'
            USING v_obj.to_clob(), p_id_val;

        EXCEPTION
            WHEN NO_DATA_FOUND THEN NULL;
            WHEN OTHERS THEN 
                lang_data_logger_pkg.log_error(p_table || ' JSON error: ' || 
                SQLERRM);
    END update_json_analytics;

    PROCEDURE update_rank_stats_from_search (
        p_search_id IN VARCHAR2
    ) AS
        CURSOR c_matches IS
            SELECT 
                json_value(value, '$.overall_rank')     AS overall_rank,
                json_value(value, '$.report_id')        AS report_id,
                json_value(value, '$.drilldown_id')     AS drilldown_id
            FROM json_table(
                (SELECT report_matches FROM langdata$searchrecords 
                WHERE id = p_search_id),
                '$[*]' COLUMNS (value JSON PATH '$')
            );
        v_json_text   CLOB;
        v_key         VARCHAR2(20);
    BEGIN
        FOR rec IN c_matches LOOP
            IF rec.overall_rank IS NULL THEN
                CONTINUE;
            END IF;

            IF rec.overall_rank = 1 THEN
                v_key := 'top1_count';
            ELSIF rec.overall_rank IN (2, 3) THEN
                v_key := 'top3_count';
            ELSIF rec.overall_rank IN (4, 5) THEN
                v_key := 'top5_count';
            ELSE
                CONTINUE;
            END IF;

            -- Update report analytics
            update_json_analytics(
                p_table  => 'langdata$reports',
                p_id_col => 'id',
                p_id_val => rec.report_id,
                p_key     => v_key,
                p_delta   => 1
            );
            lang_data_logger_pkg.log_info('Report stats updated.');

            -- Update drilldown analytics
            update_json_analytics(
                p_table  => 'langdata$drilldowndocuments',
                p_id_col => 'id',
                p_id_val => rec.drilldown_id,
                p_key     => v_key,
                p_delta   => 1
            );
            lang_data_logger_pkg.log_info('Drilldown stats updated.');

        END LOOP;
    END update_rank_stats_from_search;

    PROCEDURE update_ner_stats_from_search (
        p_search_id IN VARCHAR2
    ) AS
        v_entities_json   CLOB;
        v_entity_count    NUMBER := 0;
        v_current_count   NUMBER;

    BEGIN
        -- Step 1: Get recognized_entities JSON for the given search_id
        SELECT recognized_entities
        INTO v_entities_json
        FROM langdata$searchrecords
        WHERE id = p_search_id;

        -- Step 2: Count entities in the JSON array
        SELECT COUNT(*)
        INTO v_entity_count
        FROM JSON_TABLE(
            v_entities_json,
            '$[*]' COLUMNS (
                label VARCHAR2(50) PATH '$.label'
            )
        );

        -- Step 3: Lock and fetch the current total_count
        SELECT metric_value
        INTO v_current_count
        FROM langdata$ner_metrics
        WHERE metric_name = 'recognized_entity_count'
        FOR UPDATE WAIT 5;

        -- Step 4: Update the counter
        UPDATE langdata$ner_metrics 
        SET metric_value = v_current_count + v_entity_count
        WHERE metric_name = 'recognized_entity_count';

    EXCEPTION
        WHEN NO_DATA_FOUND THEN
            INSERT INTO langdata$ner_metrics (metric_name, metric_value)
            VALUES ('recognized_entity_count', v_entity_count);
    END update_ner_stats_from_search;

    PROCEDURE update_filter_values_stats_from_search(
        p_search_id     IN VARCHAR2,
        p_filter_type   IN VARCHAR2  -- 'default' or 'non_default'
    ) AS
        CURSOR c_matches IS
            SELECT
                json_value(value, '$.report_id') AS report_id,
                json_value(value, '$.drilldown_id') AS drilldown_id,
                json_query(value, '$.report_filter_values' RETURNING CLOB)
                AS report_filter_values,
                json_query(value, '$.drilldown_filter_values' RETURNING CLOB)
                AS drilldown_filter_values
            FROM json_table(
                (SELECT report_matches FROM langdata$searchrecords
                WHERE id = p_search_id),
                '$[*]' COLUMNS (value JSON PATH '$')
            );

        v_count_report_filter_values INTEGER;
        v_count_drilldown_filter_values INTEGER;

        v_reason_clause VARCHAR2(1000);
        v_report_key    VARCHAR2(100);
        v_drilldown_key VARCHAR2(100);
    BEGIN
        -- Determine filtering clause and analytics key
        IF p_filter_type = 'default' THEN
            v_reason_clause := 'LIKE ''Using default value%''';
            v_report_key := 'default_filter_count';
            v_drilldown_key := 'default_drilldown_filter_count';
        ELSIF p_filter_type = 'non_default' THEN
            v_reason_clause := 'NOT LIKE ''Using default value%''';
            v_report_key := 'non_default_filter_count';
            v_drilldown_key := 'non_default_drilldown_filter_count';
        ELSE
            lang_data_errors_pkg.raise_error(
                lang_data_errors_pkg.c_invalid_parameters_code
            );
        END IF;

        FOR match_rec IN c_matches LOOP
            v_count_report_filter_values := 0;
            v_count_drilldown_filter_values := 0;

            -- Dynamic SQL for report filter value count
            EXECUTE IMMEDIATE '
                SELECT COUNT(*)
                FROM json_table(:1, ''$.*'' COLUMNS (
                    reason VARCHAR2(4000) PATH ''$.reason''
                ))
                WHERE reason ' || v_reason_clause
            INTO v_count_report_filter_values
            USING match_rec.report_filter_values;

            update_json_analytics(
                p_table   => 'langdata$reports',
                p_id_col  => 'id',
                p_id_val  => match_rec.report_id,
                p_key     => v_report_key,
                p_delta   => v_count_report_filter_values
            );
            lang_data_logger_pkg.log_info(
              'Report ' || p_filter_type || ' filter values statistics updated.'
            );

            -- Dynamic SQL for drilldown filter value count
            EXECUTE IMMEDIATE '
                SELECT COUNT(*)
                FROM json_table(:1, ''$.*'' COLUMNS (
                    reason VARCHAR2(4000) PATH ''$.reason''
                ))
                WHERE reason ' || v_reason_clause
            INTO v_count_drilldown_filter_values
            USING match_rec.drilldown_filter_values;

            update_json_analytics(
                p_table   => 'langdata$drilldowndocuments',
                p_id_col  => 'id',
                p_id_val  => match_rec.drilldown_id,
                p_key     => v_drilldown_key,
                p_delta   => v_count_drilldown_filter_values
            );
            lang_data_logger_pkg.log_info(
           'Drilldown ' || p_filter_type || ' filter values statistics updated.'
            );
        END LOOP;
    END update_filter_values_stats_from_search;  

    PROCEDURE update_stats_from_search(
        p_search_id IN VARCHAR2
    ) AS
    BEGIN
        lang_data_search_pkg.update_rank_stats_from_search(
            p_search_id => p_search_id
        );
        lang_data_search_pkg.update_ner_stats_from_search(
            p_search_id => p_search_id
        );
        lang_data_search_pkg.
        update_filter_values_stats_from_search(
            p_search_id => p_search_id,
            p_filter_type => 'non_default'
        );
        lang_data_search_pkg.
        update_filter_values_stats_from_search(
            p_search_id => p_search_id,
            p_filter_type => 'default'
        );

    END update_stats_from_search;
    
    PROCEDURE update_report_search_count(
        p_search_id IN VARCHAR2
    ) AS
        v_analytics_data     JSON_OBJECT_T;
        v_analytics_data_str CLOB;
        v_search_count       NUMBER := 0;
        v_report_matches     JSON_ARRAY_T;
        v_match_obj          JSON_OBJECT_T;
        v_report_title       VARCHAR2(4000);
        v_report_id          VARCHAR2(36);
        v_match_text         CLOB;
    BEGIN
        SELECT JSON_SERIALIZE(report_matches RETURNING CLOB)
        INTO v_match_text
        FROM langdata$searchrecords
        WHERE id = p_search_id;

        -- Parse JSON array of report matches
        v_report_matches := JSON_ARRAY_T.parse(v_match_text);

        FOR i IN 0 .. v_report_matches.get_size - 1 LOOP
            v_match_obj := TREAT(v_report_matches.get(i) AS JSON_OBJECT_T);
            v_report_title := v_match_obj.get_string('report_title');
            v_report_id  := v_match_obj.get_string('report_id');

            BEGIN
                -- Lock and fetch analytics_data as CLOB
                SELECT analytics_data
                INTO v_analytics_data_str
                FROM langdata$reports
                WHERE id = v_report_id
                FOR UPDATE;

                IF v_analytics_data_str IS NULL THEN
                    v_analytics_data := JSON_OBJECT_T();
                ELSE
                    v_analytics_data 
                            := JSON_OBJECT_T.parse(v_analytics_data_str);
                END IF;

                -- Get current search_count or initialize to 0
                IF v_analytics_data.has('search_count') THEN
                    v_search_count 
                            := v_analytics_data.get_number('search_count');
                ELSE
                    v_search_count := 0;
                END IF;

                -- Update search count and timestamp
                v_analytics_data.put('search_count', v_search_count + 1);
                v_analytics_data.put('last_searched_at',
                    TO_CHAR(SYSTIMESTAMP, 'YYYY-MM-DD"T"HH24:MI:SS"Z"'));

                -- Serialize JSON to CLOB
                v_analytics_data_str := v_analytics_data.to_clob();

                -- Persist changes
                UPDATE langdata$reports
                SET analytics_data = v_analytics_data_str,
                    updated_at = CURRENT_TIMESTAMP
                WHERE title = v_report_title;

            EXCEPTION
                WHEN NO_DATA_FOUND THEN
                    NULL;
                WHEN OTHERS THEN
                    lang_data_logger_pkg.log_error(
                        'Error updating analytics_data for report: ' 
                        || v_report_title || ' - ' || SQLERRM
                    );
            END;
        END LOOP;

    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_error(
                'Failed to process searchrecord ID ' || p_search_id || 
                ': ' || SQLERRM
            );
    END update_report_search_count;

    PROCEDURE plot_search_vectors(
        p_search_id IN VARCHAR2
    )
    IS
        v_result              VARCHAR2(20);
        v_query_vector        VECTOR(*,*);
        v_report_matches      JSON;
        v_report_matches_arr  JSON_ARRAY_T := JSON_ARRAY_T();
        v_expected_report_id  VARCHAR2(36);
        v_match_obj           JSON_OBJECT_T;
        v_report_id           VARCHAR2(36);
        v_report_title        VARCHAR2(255);
        v_report_desc_id      VARCHAR2(36);
        v_highlight           NUMBER;
        v_img_data            BLOB;
    BEGIN
        SELECT query_vector, report_matches, expected_report_id
        INTO v_query_vector, v_report_matches, v_expected_report_id
        FROM langdata$searchrecords
        WHERE id = p_search_id;

        INSERT INTO langdata$plot_input(report_id, type, vector)
        VALUES(v_expected_report_id, 'actual query', v_query_vector);

        v_report_matches_arr := JSON_ARRAY_T(v_report_matches);
        FOR i IN 0 .. v_report_matches_arr.get_size - 1 LOOP
            v_match_obj := JSON_OBJECT_T(v_report_matches_arr.get(i));
            v_report_id := v_match_obj.get_string('report_id');
            v_report_title := v_match_obj.get_string('report_title');
            v_report_desc_id := v_match_obj.get_string('report_description_id');

            FOR rec IN (
                SELECT id, description_vector
                FROM langdata$reportdescriptions
                WHERE report_id = v_report_id
            ) LOOP
                IF rec.id = v_report_desc_id THEN
                    v_highlight := 1;
                ELSE
                    v_highlight := 0;
                END IF;

                INSERT INTO langdata$plot_input(
                    report_id, report_title, type, vector, highlight
                )
                VALUES(
                    v_report_id,
                    v_report_title,
                    'description',
                    rec.description_vector,
                    v_highlight
                );
            END LOOP;

            FOR rec IN (
                SELECT id, query_vector
                FROM langdata$searchrecords
                WHERE expected_report_id = v_report_id
            ) LOOP
                IF rec.id = v_report_desc_id THEN
                    v_highlight := 1;
                ELSE
                    v_highlight := 0;
                END IF;

                INSERT INTO langdata$plot_input(
                    report_id, report_title, type, vector, highlight
                )
                VALUES(
                    v_report_id,
                    v_report_title,
                    'sample query',
                    rec.query_vector,
                    v_highlight
                );
            END LOOP;

            FOR rec IN (
                SELECT id, query_vector
                FROM langdata$samplequeries
                WHERE report_id = v_report_id
            ) LOOP
                IF rec.id = v_report_desc_id THEN
                    v_highlight := 1;
                ELSE
                    v_highlight := 0;
                END IF;

                INSERT INTO langdata$plot_input(
                    report_id, report_title, type, vector, highlight
                )
                VALUES(
                    v_report_id,
                    v_report_title,
                    'sample query',
                    rec.query_vector,
                    v_highlight
                );
            END LOOP;

        END LOOP;

        COMMIT;

        SELECT "img_data"
        INTO v_img_data
        FROM TABLE(
            pyqEval(
            '{"table_name":"LANGDATA$PLOT_INPUT", "oml_connect":true}', 
            '{"img_data": "blob"}',
            'plot_search_vectors' 
            )
        );

        UPDATE langdata$searchrecords
        SET plot_image = v_img_data
        WHERE id = p_search_id;

        EXECUTE IMMEDIATE 'TRUNCATE TABLE LANGDATA$PLOT_INPUT';

    END plot_search_vectors;

    FUNCTION rerank_texts(
        p_rerank_obj_list IN rerank_table
    ) RETURN rerank_table IS
        -- Needed autonomous transaction as commit is needed for passing table 
        -- to oml4py
        PRAGMA AUTONOMOUS_TRANSACTION;
        TYPE rerank_obj_map IS TABLE OF rerank_obj INDEX BY VARCHAR2(36);
        v_rerank_obj_map rerank_obj_map;
        v_rerank_obj    rerank_obj;
        v_model_cache VARCHAR2(4000);
        v_re_ranked_results JSON;
        v_reranked_tables RERANK_TABLE := RERANK_TABLE();
    BEGIN
        FOR i IN 1..p_rerank_obj_list.COUNT LOOP
            -- Insert rerank obj in table
            INSERT INTO langdata$re_ranking_inputs(
                doc_text_id,
                doc_text, 
                query_text
            ) VALUES (
                p_rerank_obj_list(i).doc_id,
                p_rerank_obj_list(i).doc_text,
                p_rerank_obj_list(i).query_text
            );

            -- Insert rerank obj in a hashmap
            v_rerank_obj_map(p_rerank_obj_list(i).doc_id) 
                := p_rerank_obj_list(i);

        END LOOP;
        COMMIT;

        v_model_cache := lang_data_config_pkg.get_config_parameter(
            'MOUNT_DIR'
        );
        v_model_cache := v_model_cache || '/models';

        FOR rec IN (
            SELECT doc_text_id, rank, rank_score
            FROM TABLE(
                pyqTableEval(
                    'LANGDATA$RE_RANKING_INPUTS',
                    '{"oml_input_type":"pandas.DataFrame", "cache_dir":"'
                        ||v_model_cache||'"}',
                    '{
                        "DOC_TEXT_ID": "varchar2(36)",
                        "RANK": "number",
                        "RANK_SCORE": "number"
                    }',
                    're_rank_search_results'
                )
            )
            ORDER BY rank asc
        ) LOOP
            v_rerank_obj := v_rerank_obj_map(rec.doc_text_id);
            v_rerank_obj.doc_rank := rec.rank;
            v_rerank_obj.score := rec.rank_score;
            v_reranked_tables.EXTEND;
            v_reranked_tables(v_reranked_tables.COUNT) := v_rerank_obj;
        END LOOP;

        EXECUTE IMMEDIATE 'TRUNCATE TABLE LANGDATA$RE_RANKING_INPUTS'; 
        RETURN v_reranked_tables;
    EXCEPTION
        WHEN OTHERS THEN
            lang_data_logger_pkg.log_fatal(
                'An unknown error occurred while reranking. Error: ' ||
                SQLERRM
            );
            RAISE;
    END rerank_texts;


END lang_data_search_pkg;
/
