/*
 * Decompiled with CFR 0.152.
 */
package oracle.dbtools.core.connections.storage;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import oracle.dbtools.core.collections.ConfigurationProperties;
import oracle.dbtools.core.connections.ConnectionIdentifiers;
import oracle.dbtools.core.connections.ConnectionsResources;
import oracle.dbtools.core.connections.api.Connections;
import oracle.dbtools.core.connections.storage.ConnectionDefinition;
import oracle.dbtools.core.connections.storage.StoreException;
import oracle.dbtools.core.connections.util.ConnectionManagementUtil;
import oracle.dbtools.core.io.file.HomeFolder;
import oracle.dbtools.core.io.file.StandardAccessPermissions;

public final class ConnectionStorage {
    static final Logger LOGGER = Logger.getLogger(ConnectionStorage.class.getName());
    private static final String STORE_DIRECTORY_NAME = "connections";
    private static final String CONFIG_DIRECTORY_WINDOWS = "DBTools";
    final Path store;
    private final String errorState;

    private ConnectionStorage(Path store) {
        this.store = store;
        this.errorState = null;
    }

    private ConnectionStorage(Throwable errorState) {
        this.store = null;
        this.errorState = errorState.toString();
    }

    public static ConnectionStorage instance() {
        return Holder.INSTANCE;
    }

    private static Path getConnectionStorageDirectory() {
        return HomeFolder.ofSystem().connectionsPath();
    }

    public static ConnectionStorage getPriorStorage(Path oldRoot) {
        ConnectionStorage storage = null;
        Path store = null;
        if (oldRoot != null) {
            Path config = oldRoot.resolve(CONFIG_DIRECTORY_WINDOWS);
            store = config.resolve(STORE_DIRECTORY_NAME);
        }
        if (store != null && store.compareTo(ConnectionStorage.getConnectionStorageDirectory()) != 0 && Files.isDirectory(store, new LinkOption[0])) {
            storage = ConnectionStorage.createConnectionStorage(store);
        }
        return storage;
    }

    private static void verifyAndCreateDirectory(Path path) throws StoreException {
        if (!Files.exists(path, new LinkOption[0])) {
            try {
                StandardAccessPermissions.SECURE_FOLDER.create(path);
            }
            catch (IOException e) {
                throw new StoreException(ConnectionsResources.getString("ERROR_CREATING_DIRECTORY"), e);
            }
        } else if (!Files.isDirectory(path, new LinkOption[0])) {
            throw new StoreException(ConnectionsResources.format("ERROR_BAD_PATH", path));
        }
    }

    private static ConnectionStorage createConnectionStorage(Path store) {
        ConnectionStorage storage;
        try {
            ConnectionStorage.verifyAndCreateDirectory(store);
            storage = new ConnectionStorage(store);
        }
        catch (StoreException e) {
            Throwable cause = e.getCause();
            storage = new ConnectionStorage(cause != null ? cause : e);
        }
        return storage;
    }

    private static boolean hasPropertiesFile(Path dir) {
        return Files.exists(dir.resolve("dbtools.properties"), new LinkOption[0]);
    }

    private static boolean validatePermissions(Path dir) {
        try {
            return StandardAccessPermissions.SECURE_FOLDER.verify(dir);
        }
        catch (IOException ex) {
            LOGGER.log(Level.SEVERE, ConnectionsResources.format("ERROR_FILE_PERMISSIONS_CHECK", dir));
            return false;
        }
    }

    static ConfigurationProperties loadProperties(Path file) {
        LOGGER.finer(ConnectionsResources.format("STORAGE_LOADFILE_MESSAGE", file));
        if (Files.isRegularFile(file, new LinkOption[0])) {
            ConfigurationProperties configurationProperties;
            block9: {
                InputStream is = Files.newInputStream(file, new OpenOption[0]);
                try {
                    configurationProperties = ConfigurationProperties.read(is);
                    if (is == null) break block9;
                }
                catch (Throwable throwable) {
                    try {
                        if (is != null) {
                            try {
                                is.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException ex) {
                        LOGGER.log(Level.WARNING, ConnectionsResources.format("ERROR_STORAGE_FILEREAD", file), ex);
                    }
                }
                is.close();
            }
            return configurationProperties;
        }
        return ConfigurationProperties.empty();
    }

    public List<String> listConnectionDirectories() {
        if (this.store == null) {
            return Collections.emptyList();
        }
        ArrayList<String> conns = new ArrayList<String>();
        try (Stream<Path> dirs = Files.find(this.store, 1, (p, a) -> a.isDirectory(), new FileVisitOption[0]);){
            dirs.filter(ConnectionStorage::hasPropertiesFile).forEach(dir -> conns.add(dir.getFileName().toString()));
        }
        catch (IOException ex) {
            LOGGER.log(Level.SEVERE, ConnectionsResources.format("ERROR_STORAGE_DIRECTORY_PROCESSING", this.store), ex);
        }
        return conns;
    }

    public void renameConnection(String oldName, String newName) {
        try {
            Optional<String> directory = this.listConnectionDirectories().stream().filter(dir -> {
                ConnectionDefinition def = this.loadConnection((String)dir);
                return oldName.equalsIgnoreCase(def.getConnectionName());
            }).findFirst();
            if (directory.isPresent()) {
                Path dbTools;
                String dirName = directory.get();
                Connections.Identifier connID = ConnectionIdentifiers.createIdentifier(dirName);
                Path folder = this.resolvePath(connID);
                if (Objects.equals(connID, ConnectionIdentifiers.createIdentifier(oldName))) {
                    Connections.Identifier newNameIdentifier = ConnectionIdentifiers.createUniqueIdentifier();
                    Path destDir = this.resolvePath(newNameIdentifier);
                    Files.move(this.resolvePath(connID), destDir, StandardCopyOption.ATOMIC_MOVE);
                    connID = newNameIdentifier;
                    folder = destDir;
                }
                ConfigurationProperties dbToolsProps = Files.exists(dbTools = folder.resolve("dbtools.properties"), new LinkOption[0]) ? ConnectionStorage.loadProperties(dbTools) : ConfigurationProperties.empty();
                dbToolsProps = dbToolsProps.withProperty("name", newName);
                this.storeObject(connID, "dbtools.properties", dbToolsProps, ConnectionDefinition::outputProperties);
            }
        }
        catch (IOException ex) {
            LOGGER.log(Level.SEVERE, ConnectionsResources.format("ERROR_STORAGE_DIRECTORY_PROCESSING", this.store), ex);
        }
    }

    public void deleteConnection(String connName) {
        try (Stream<Path> dirs = Files.find(this.store, 1, (p, a) -> a.isDirectory(), new FileVisitOption[0]);){
            Optional<Path> dirToBeDeleted = dirs.filter(ConnectionStorage::hasPropertiesFile).filter(dir -> {
                ConnectionDefinition def = this.loadConnection(dir.toString());
                return connName.equalsIgnoreCase(def.getConnectionName());
            }).findFirst();
            dirToBeDeleted.ifPresent(dir -> {
                Connections.Identifier identifier = ConnectionIdentifiers.createIdentifier(dir.getFileName().toString());
                Path path = this.resolvePath(identifier);
                try (Stream<Path> paths = Files.walk(path, 1, new FileVisitOption[0]);){
                    paths.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
                }
                catch (IOException e) {
                    LOGGER.log(Level.SEVERE, ConnectionsResources.format("ERROR_STORAGE_DIRECTORY_PROCESSING", this.store), e);
                }
            });
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, ConnectionsResources.format("ERROR_STORAGE_DIRECTORY_PROCESSING", this.store), e);
        }
    }

    public ConnectionDefinition loadConnection(String identifier) {
        ConnectionDefinition def = null;
        if (this.store != null) {
            Path dir = this.store.resolve(identifier);
            if (!ConnectionStorage.validatePermissions(dir)) {
                try {
                    ConnectionStorage.configureDirectories(dir);
                }
                catch (IOException | IllegalArgumentException e) {
                    LOGGER.log(Level.SEVERE, ConnectionsResources.format("ERROR_INCORRECT_FILE_PERMISSIONS", dir), e);
                }
            }
            def = ConnectionDefinition.createConnectionDefinition(dir);
        }
        return def;
    }

    private static void configureDirectories(Path folder) throws IOException {
        StandardAccessPermissions.SECURE_FOLDER.configure(folder);
        try (DirectoryStream<Path> files = Files.newDirectoryStream(folder);){
            files.forEach(path -> {
                try {
                    if (Files.isDirectory(path, new LinkOption[0])) {
                        ConnectionStorage.configureDirectories(path);
                    } else {
                        StandardAccessPermissions.SECURE_FILE.configure((Path)path);
                    }
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
        }
        catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    Path duplicateConnection(Connections.Identifier identifier, Connections.Identifier newIdentifier, CopyErrorHandler errorHandler) throws IOException {
        if (this.store == null) {
            throw UnsupportedStorageOperation.create(this.errorState);
        }
        Path origin = this.resolvePath(identifier);
        Path dest = this.resolvePath(newIdentifier);
        if (Files.exists(dest, new LinkOption[0])) {
            throw new StoreException(ConnectionsResources.format("ERROR_STORAGE_DIRECTORY_EXISTS", dest));
        }
        ConnectionStorage.verifyAndCreateDirectory(dest);
        try (Stream<Path> s = Files.walk(origin, new FileVisitOption[0]);){
            s.forEach(p -> {
                Path rel = origin.relativize((Path)p);
                Path destFile = dest.resolve(rel);
                try {
                    if (Files.isDirectory(p, new LinkOption[0])) {
                        ConnectionStorage.verifyAndCreateDirectory(destFile);
                    } else {
                        Files.copy(p, destFile, StandardCopyOption.COPY_ATTRIBUTES);
                    }
                }
                catch (IOException ex) {
                    errorHandler.handleCopyError(ex);
                }
            });
        }
        return dest;
    }

    public OutputStream resolve(Path relativePath) throws IOException {
        if (this.store == null) {
            throw UnsupportedStorageOperation.create(this.errorState);
        }
        Path resolvedPath = this.store.resolve(relativePath);
        Path connDir = resolvedPath.getParent();
        ConnectionStorage.verifyAndCreateDirectory(connDir);
        return Files.newOutputStream(resolvedPath, new OpenOption[0]);
    }

    public <T> void storeObject(Connections.Identifier identifier, String filename, T data, StoreAction<T> action) throws StoreException {
        if (this.store == null) {
            throw UnsupportedStorageOperation.create(this.errorState);
        }
        Path conndir = this.resolvePath(identifier);
        ConnectionStorage.verifyAndCreateDirectory(conndir);
        try (OutputStream out = ConnectionStorage.outputStream(conndir.resolve(filename));){
            action.store(data, out);
        }
        catch (IOException ex) {
            throw new StoreException(ConnectionsResources.format("ERROR_STORAGE_STORING_PROPERTIES", filename), ex);
        }
    }

    private static OutputStream outputStream(Path path) throws IOException {
        if (Files.exists(path, new LinkOption[0])) {
            return Files.newOutputStream(path, new OpenOption[0]);
        }
        StandardAccessPermissions.SECURE_FILE.create(path);
        return Files.newOutputStream(path, new OpenOption[0]);
    }

    Path resolvePath(Connections.Identifier identifier) {
        String dirName = ConnectionManagementUtil.instance().mapDirectoryName(identifier);
        return this.store.resolve(dirName);
    }

    public void storeDefinition(ConnectionDefinition def) throws StoreException {
        if (this.store == null) {
            throw UnsupportedStorageOperation.create(this.errorState);
        }
        def.write(this);
    }

    public void createSubdirectory(Connections.Identifier identifier, Path dirname) throws IOException {
        if (this.store == null) {
            throw UnsupportedStorageOperation.create(this.errorState);
        }
        Path conndir = this.resolvePath(identifier);
        ConnectionStorage.verifyAndCreateDirectory(conndir);
        Path newDir = conndir.resolve(dirname);
        ConnectionStorage.verifyAndCreateDirectory(newDir);
    }

    public static final class Holder {
        private static final ConnectionStorage INSTANCE;

        static {
            Path store = ConnectionStorage.getConnectionStorageDirectory();
            INSTANCE = ConnectionStorage.createConnectionStorage(store);
        }
    }

    public static interface StoreAction<T> {
        public void store(T var1, OutputStream var2) throws IOException;
    }

    private static class UnsupportedStorageOperation
    extends StoreException {
        private UnsupportedStorageOperation(String msg) {
            super(msg);
        }

        static UnsupportedStorageOperation create(String msg) {
            return new UnsupportedStorageOperation(msg);
        }
    }

    static interface CopyErrorHandler {
        public void handleCopyError(IOException var1);
    }
}

