-- Test data for graph-based SQL injection
-- Everything before the first "+++++" is ignored
--
--  MISSING TESTS:
--  - Named arguments mixed with positional arguments
--  - Overloading by exact type match
--  - Overloading with type conversions
--  - Ambiguous overloads
--  - Q: Which has priority in PL/SQL: Type coversion overloads local, or global exact conversions?
--  - ELLIPSIS argument matching (with and without named args too; can you name an ELLIPSIS arg?)
--  
--  - The various examples from docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/plsql-name-resolution.html
--        (Some have already been added)
--  - The rules enumerated in docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/plsql-name-resolution.html
--  -----------------------------------------------------------------------------------------------------------------
--  
--  
-- Comments start with two hyphens and a space
--
-- Format of each test  (case is ignored):
--
-- +++++ unique, one-line description of what the test does (used to identify results)
-- Keywords (first word on line) are case-insensitive
-- DATAFLOW varA:13 varB:5 varB:2       For each injection inspected, identify the injection chain start to end
-- DATAFLOW varC:13 varD:5 varE:2       ...a second injection chain
-- -- Additional Optional metadata lines before the blank line, in the form "keyword [value]" case insensitive
-- -- You can use any keywords you want. Some already used:
-- INJECTION [true/false]      -- Is an injection expected?
-- ID name                     -- A short, unique identifer for thest test
-- KNOWNBAD    This test is known to give the wrong answer; warn, but don't stop testing
-- THROWS SAF_...              -- List of expected SAF exceptions
-- -- Blank line separates metadata from the PL/SQL code to be analyzed
-- 
-- ... PL/SQL code to be anlayzed...
-- 
-- -- PL/SQL code for test ends at the next ++++ or the end of the file

+++++ Empty file
GREEN
Throws SAF_SYNTAX_ERROR
ID empty
Injection false

+++++ Syntax error
GREEN
Throws SAF_SYNTAX_ERROR
ID Syntax
Injection false

pthhhpt!
/

+++++ Very simple injection
GREEN
INJECTION true
DATAFLOW d1:1 d1:5

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || d1  INTO val; 
 RETURN val;
END;
/

+++++ Numbers safe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (n1 NUMBER, n2 PLS_INTEGER, n3 BINARY_DOUBLE, n4 BINARY_FLOAT, n5 BINARY_INTEGER)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || n1 || n2 || n3 || n4 || n5 INTO val; 
 RETURN val;
END;
/

+++++ Assigning to number makes safe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  num1 NUMBER;
BEGIN
 num1 := d1;  -- Injection with NLS is not totally impossible, but requires substantial other access to exploit ; for now, this is safe
 execute immediate 'SELECT count(1) FROM ' || num1  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.ENQUOTE_LITERAL output is safe (direct)
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.enquote_literal(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.ENQUOTE_LITERAL output is safe (assigned)
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  v1  VARCHAR2(100);
BEGIN
 v1 := dbms_assert.enquote_literal(d1);
 execute immediate 'SELECT count(1) FROM ' || v1 INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.ENQUOTE_NAME output is safe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val1 NUMBER;
  val2 NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.enquote_name(d1)  INTO val1; 
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.enquote_name(d1, TRUE)  INTO val2; 
 RETURN val1 + val2;
END;
/

+++++ DBMS_ASSERT.ENQUOTE_NOOP output is treated as safe (varchar)
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.noop(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.ENQUOTE_NOOP output is treated as safe (CLOB)
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 CLOB)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.noop(d1)  INTO val; 
 RETURN val;
END;
/


+++++ Function calls can return any exception
Knownbad
Exception
Injection true
Dataflow D1:1 D1:-16 D1:18

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  quotient NUMBER;
  numerator NUMBER := 1;
  denominator NUMBER;
  t1 VARCHAR2(100);
BEGIN
  t1 := dbms_assert.noop(d1);  -- Treat as safe; if viewed as throwing, d1 would be dangerous in exception handler
  quotient := numerator / denominator;
  return quotient;
  EXCEPTION
    WHEN VALUE_ERROR OR STORAGE_ERROR THEN
      return 0;
    WHEN ZERO_DIVIDE THEN
      return 0;
    WHEN OTHERS THEN
      -- Since dbms_assert.noop is a function, it could throw any exception
      execute immediate 'SELECT count(1) FROM ' || d1  INTO quotient; 
      RETURN quotient;
END;
/

+++++ DBMS_ASSERT.QUALIFIED_SQL_NAME output is safe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.qualified_sql_name(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.QUALIFIED_SQL_NAME throws if unsafe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  unused VARCHAR2(100);
BEGIN
 unused := dbms_assert.QUALIFIED_SQL_NAME(d1);
 -- d1 safe if we reach here
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.qualified_sql_name(d1)  INTO val; 
 RETURN val;
END;
/

+++++ Handles PRAGMA EXCEPTION_INIT
Injection true
Exception
KnownBad
Dataflow D1:1 D1:-15 D1:16

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  quotient NUMBER;
  numerator NUMBER := 1;
  denominator NUMBER;
  t1 VARCHAR2(100);
  ASSERT_EXCEPTION EXCEPTION;
  PRAGMA EXCEPTION_INIT(ASSERT_EXCEPTION, -44004);
BEGIN
  t1 := dbms_assert.QUALIFIED_SQL_NAME(d1);  -- Unclean if throws
  denominator := d1;		-- Implicit conversion
  quotient := numerator / denominator;
  return quotient;
  EXCEPTION
    WHEN ASSERT_EXCEPTION THEN
      execute immediate 'SELECT count(1) FROM ' || d1  INTO quotient; 
      RETURN quotient;
    WHEN OTHERS THEN
      RETURN 0;
END;
/

+++++ DBMS_ASSERT.QUALIFIED_SQL_NAME may throw exception (dangerous in throw)
Exception
KnownBad
Injection true
Dataflow D1:1 D1:-15 D1:16

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  quotient NUMBER;
  numerator NUMBER := 1;
  denominator NUMBER;
  t1 VARCHAR2(100);
BEGIN
  t1 := dbms_assert.QUALIFIED_SQL_NAME(d1);  -- Unclean if throws
  denominator := d1;		-- Implicit conversion
  quotient := numerator / denominator;
  return quotient;
  EXCEPTION
    WHEN VALUE_ERROR OR STORAGE_ERROR THEN
      RETURN 0; 
    WHEN OTHERS THEN
      execute immediate 'SELECT count(1) FROM ' || d1  INTO quotient; 
      RETURN quotient;
END;
/

+++++ DBMS_ASSERT.SCHEMA_NAME output is safe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.qualified_sql_name(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.SCHEMA_NAME throws if unsafe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  unused VARCHAR2(100);
BEGIN
 unused := dbms_assert.qualified_sql_name(d1);
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.schema_name(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.SIMPLE_SQL_NAME output is safe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.SIMPLE_SQL_NAME(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.SIMPLE_SQL_NAME throws if unsafe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  unused VARCHAR2(100);
BEGIN
 unused := dbms_assert.qualified_sql_name(d1);
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.simple_sql_name(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.SQL_OBJECT_NAME output is safe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
BEGIN
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.sql_object_name(d1)  INTO val; 
 RETURN val;
END;
/

+++++ DBMS_ASSERT.SQL_OBJECT_NAME throws if unsafe
GREEN
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  unused VARCHAR2(100);
BEGIN
 unused := dbms_assert.sql_object_name(d1);
 execute immediate 'SELECT count(1) FROM ' || dbms_assert.sql_object_name(d1)  INTO val; 
 RETURN val;
END;
/


+++++ Assignment clearing var
GREEN
Injection false

CREATE OR REPLACE FUNCTION getCountFromTable ( tabName VARCHAR2)
RETURN NUMBER IS
 val NUMBER;
 t1 VARCHAR2(100);
 t2 VARCHAR2(100);
 t3 VARCHAR2(100);
BEGIN
 t1 := tabName || '_table';
 t2 := upper(tabName);
 IF (tabName != 'PRICE') THEN
   t3 := t2;
 ELSE
   t3 := 'COST_TABLE';
 END IF;
 t3 := 'COST_TABLE'; -- Safe!
 execute immediate 'SELECT count(1) FROM ' || t3 
 INTO val;
 RETURN val;
END;
/

+++++ If - all branches clean
GREEN
IF_THEN
Injection false

CREATE OR REPLACE FUNCTION getCountFromTable ( tabName VARCHAR2)
RETURN NUMBER IS
 val NUMBER;
 t1 VARCHAR2(100);
 t2 VARCHAR2(100);
 t3 VARCHAR2(100);
BEGIN
 t1 := tabName || '_table';
 t2 := upper(tabName);
 t3 := t2;
 IF (tabName != 'PRICE') THEN
   t3 := 'COST_TABLE'; 
 ELSIF (tabName = 'PRICE') THEN
   t3 := 'COST_TABLE';
 ELSE
   t3 := 'COST_TABLE';
 END IF;
 execute immediate 'SELECT count(1) FROM ' || t3 
 INTO val;
 RETURN val;
END;
/

+++++ If - then branch, no else
DEBUG
GREEN
IF_THEN
Injection true
Dataflow TABNAME:1 T2:9 T3:10 T3:-11 T3:14

CREATE OR REPLACE FUNCTION getCountFromTable ( tabName VARCHAR2)
RETURN NUMBER IS
 val NUMBER;
 t1 VARCHAR2(100);
 t2 VARCHAR2(100);
 t3 VARCHAR2(100);
BEGIN
 t1 := tabName || '_table';
 t2 := upper(tabName);
 t3 := t2;
 IF (tabName != 'PRICE') THEN
   t3 := 'COST_TABLE';
 END IF;
 execute immediate 'SELECT count(1) FROM ' || t3 
 INTO val;
 RETURN val;
END;
/

+++++ If - then branch
GREEN
IF_THEN
Injection true
Dataflow tabName:1 T2:9 T3:11 T3:-10 T3:17

CREATE OR REPLACE FUNCTION getCountFromTable ( tabName VARCHAR2)
RETURN NUMBER IS
 val NUMBER;
 t1 VARCHAR2(100);
 t2 VARCHAR2(100);
 t3 VARCHAR2(100);
BEGIN
 t1 := tabName || '_table';
 t2 := upper(tabName);
 IF (tabName != 'PRICE') THEN
   t3 := t2;
 ELSIF (tabName != 'PRICE') THEN
   t3 := 'COST_TABLE';
 ELSE
   t3 := 'COST_TABLE'; -- Does not make t3 safe outside of this block
 END IF;
 execute immediate 'SELECT count(1) FROM ' || t3 
 INTO val;
 RETURN val;
END;
/

+++++ If - elsif branch
GREEN
IF_THEN
Injection true
Dataflow TABNAME:1 T2:9 T3:13 T3:-10 T3:17

CREATE OR REPLACE FUNCTION getCountFromTable ( tabName VARCHAR2)
RETURN NUMBER IS
 val NUMBER;
 t1 VARCHAR2(100);
 t2 VARCHAR2(100);
 t3 VARCHAR2(100);
BEGIN
 t1 := tabName || '_table';
 t2 := upper(tabName);
 IF (tabName != 'PRICE') THEN
   t3 := 'COST_TABLE';
 ELSIF (tabName != 'PRICE') THEN
   t3 := t2;
 ELSE
   t3 := 'COST_TABLE';
 END IF;
 execute immediate 'SELECT count(1) FROM ' || t3 
 INTO val;
 RETURN val;
END;
/

+++++ If - else branch
GREEN
IF_THEN
Injection true
Dataflow tabName:1 t2:9 t3:15 t3:-10 t3:17

CREATE OR REPLACE FUNCTION getCountFromTable ( tabName VARCHAR2)
RETURN NUMBER IS
 val NUMBER;
 t1 VARCHAR2(100);
 t2 VARCHAR2(100);
 t3 VARCHAR2(100);
BEGIN
 t1 := tabName || '_table';
 t2 := upper(tabName);
 IF (tabName != 'PRICE') THEN
   t3 := 'COST_TABLE'; -- Does not make t3 safe outside of this block
 ELSIF (tabName != 'PRICE') THEN
   t3 := 'COST_TABLE'; -- Does not make t3 safe outside of this block
 ELSE
   t3 := t2;
 END IF;
 execute immediate 'SELECT count(1) FROM ' || t3 
 INTO val;
 RETURN val;
END;
/


+++++ LOOP - infinite loop with unreachable statement
LOOP
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  LOOP
 	t3 := d1;
  END LOOP;	-- Infinite loop; select is unrechable and thus safe
  execute immediate 'SELECT count(1) FROM ' || t3 INTO val;
  RETURN val;
END;
/


+++++ LOOP - with EXIT
LOOP
Injection true
Dataflow d1:1 d1:-8 t3:12 t3:-8 t3:-13 t3:14

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  LOOP
	IF (t3 = d1) THEN
      EXIT;
    END IF;  
 	t3 := d1;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3 INTO val; 
 RETURN val;
END;
/

+++++ LOOP - with EXIT WHEN 
LOOP
Injection true
Dataflow d1:1 d1:-8 t3:10 t3:-8 t3:-11 t3:12

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  LOOP
	EXIT WHEN (t3 = d1);
 	t3 := d1;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ LOOP - data flow from EXIT - EXIT is protected by assignment 
LOOP
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    IF (val = 5) THEN
       t3 := 'SAFE';
       EXIT;
    END IF;
 	t3 := d1;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ LOOP - data flow from EXIT - not all exits protected
LOOP
Injection true
Dataflow d1:1 d1:-9 t3:17 t3:-9 t3:-18 t3:19

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    IF (val = 5) THEN
       t3 := 'SAFE';
       EXIT;
    ELSIF (val = 7) THEN
       EXIT;
    END IF ;
 	t3 := d1;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ LOOP - data flow from EXIT - changed var not used above any explicit EXIT
LOOP
Injection true
Dataflow d1:1 d1:-9 t3:17 t3:-9 t3:-19 t3:20

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    IF (val = 5) THEN
       t2 := 'ALSO SAFE';
       EXIT;
    ELSIF (val = 7) THEN
       EXIT;
    END IF ;
 	t3 := d1;
 	t2 := 'SAFE';
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3 || t2  INTO val; 
 RETURN val;
END;
/


+++++ LOOP - data flow from EXIT - local variable hides dangerous real one
LOOP
Injection true
Dataflow d1:1 d1:-9 t3:22 t3:-9 t3:-23 t3:24

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    IF (val = 5) THEN
       DECLARE
         t3 VARCHAR2(100);
       BEGIN
         t3 := 'SAFE';
         EXIT;
       END;
    ELSIF (val = 7) THEN
       t3 := 'SAFE';
       EXIT;
    END IF;
 	t3 := d1;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ LOOP - data flow from EXIT - local variable hides safe real one
LOOP
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    IF (val = 5) THEN
       DECLARE
         t3 VARCHAR2(100);
       BEGIN
         t3 := d1;
         EXIT;
       END;
    ELSIF (val = 7) THEN
       EXIT;
    END IF;
 	t3 := 'SAFE';
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ LOOP - CONTINUE propagates unsafe value
LOOP
Injection true
Dataflow d1:1 d1:-9 t3:14 t3:-9 t3:-19 t3:20

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    IF (val = 5) THEN
         EXIT;
    END IF;
    t3 := d1;
    IF (val = 7) THEN
    		CONTINUE;
    END IF;
 	t3 := 'SAFE';
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ LOOP - CONTINUE WHEN propagates unsafe value
LOOP
Injection true
Dataflow d1:1 d1:-9 t3:14 t3:-9 t3:-17 t3:18

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    IF (val = 5) THEN
         EXIT;
    END IF;
    t3 := d1;
  	CONTINUE WHEN (val = 7);
 	t3 := 'SAFE';
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/




+++++ LOOP - CONTINUE is safe when EXIT wouldn't be
LOOP
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  LOOP
    val := val + 1;
    t3 := d1;
    IF (val = 7) THEN
    		CONTINUE;
    END IF;
 	t3 := 'SAFE';
    IF (val = 5) THEN
         EXIT;
    END IF;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/

+++++ WHILE 
LOOP
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  val := 1;
  WHILE val < 10 LOOP
    val := val + 1;
    t3 := d1;
    IF (val = 7) THEN
    		CONTINUE;
    END IF;
 	t3 := 'SAFE';
    IF (val = 5) THEN
         EXIT;
    END IF;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ FOR
LOOP
knownbad
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  t1 := d1;
  FOR t1 IN 1 .. 10 LOOP    -- Hides t1
    t3 := t1;		-- Promotes integer loop index "t1", not VARCHAR2 "t1"
    IF (t1 = 7) THEN
    		CONTINUE;
    END IF;
    IF (t1 = 5) THEN
         EXIT;
    END IF;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ FOR REVERSE
LOOP
knownbad
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  t1 := d1;
  FOR t1 IN REVERSE 10 .. 1 LOOP    -- Hides t1
    t3 := t1;		-- Promotes integer loop index "t1", not VARCHAR2 "t1"
    IF (t1 = 7) THEN
    		CONTINUE;
    END IF;
    IF (t1 = 5) THEN
         EXIT;
    END IF;
 END LOOP;
 execute immediate 'SELECT count(1) FROM ' || t3  INTO val; 
 RETURN val;
END;
/


+++++ Local scope isolates dangerous var
BEGIN_END
KNOWNBAD
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  t1 := 'SAFE';
  DECLARE
    t1 VARCHAR2(50);
  BEGIN
    t1 := d1;
  END;

 execute immediate 'SELECT count(1) FROM ' || t1  INTO val; 
 RETURN val;
END;
/


+++++ Local scope does not hide dangerous assignment
KNOWNBAD
Injection true
Dataflow d1:1 t1:8 t1:14

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  val NUMBER;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  t1 := d1;
  DECLARE
    t1 VARCHAR2(50);
  BEGIN
    t1 := 'SAFE';
  END;
 execute immediate 'SELECT count(1) FROM ' || t1  INTO val; 
 RETURN val;
END;
/


+++++ Exception possible at any assignment
Injection true
Dataflow d1:1 t1:9 (VALUE_ERROR):10 t1:-12 t1:13

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  othernum NUMBER;
  val NUMBER := 1;
  t1 VARCHAR2(100);
  t2 VARCHAR2(100);
  t3 VARCHAR2(100);
BEGIN
  t1 := d1;		-- Exception possible here (d1 > 100 chars)
  t1 := 'SAFE'; -- Exception possible here (PROGRAM_ERROR, OUT_OF_MEMORY) ... alas.
  return val;
  EXCEPTION WHEN OTHERS THEN
  execute immediate 'SELECT count(1) FROM ' || t1  INTO val; 
  RETURN val;
END;
/


+++++ Exception - ZERO_DIVIDE
Injection true
Dataflow d1:1 t1:9 (ZERO_DIVIDE):10 t1:-14 t1:15

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  quotient NUMBER;
  numerator NUMBER := 1;
  denominator NUMBER;
  t1 VARCHAR2(100);
BEGIN
  denominator := d1;		-- Implicit conversion
  t1 := d1;
  quotient := numerator / denominator;
  t1 := 'SAFE';
  return quotient;
  EXCEPTION
    WHEN ZERO_DIVIDE THEN
      execute immediate 'SELECT count(1) FROM ' || t1  INTO quotient; 
      RETURN quotient;
    WHEN OTHERS THEN
      return 0;
END;
/


+++++ Exception - Data from normal exit gets flowed
Injection true
Dataflow d1:1 t1:9 t1:16

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  quotient NUMBER;
  numerator NUMBER := 1;
  denominator NUMBER;
  t1 VARCHAR2(100);
BEGIN
  BEGIN
    t1 := d1;
	denominator := d1;		-- Implicit conversion
	quotient := numerator / denominator;
	EXCEPTION
	  WHEN ZERO_DIVIDE THEN
	  t1 := 'SAFE';
   END;
   execute immediate 'SELECT count(1) FROM ' || t1  INTO quotient; 
   RETURN quotient;
END;
/


+++++ Exception - Data from exception handler gets flowed
KNOWNBAD
Injection true
Dataflow d1:1 t1:9 (ZERO_DIVIDE):10 t1:-14 t1:15

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  quotient NUMBER;
  numerator NUMBER := 1;
  denominator NUMBER;
  t1 VARCHAR2(100);
BEGIN
  BEGIN
    t1 := 'SAFE';
	denominator := d1;		-- Implicit conversion
	quotient := numerator / denominator;
	return quotient;
	EXCEPTION
	  WHEN ZERO_DIVIDE THEN
	  t1 := d1;
  END;
  execute immediate 'SELECT count(1) FROM ' || t1  INTO quotient; 
  RETURN quotient;
END;
/


+++++ Exception - Nested exception scopes - proves nothing, DELETE THIS TEST
knownbad
Injection false

-- From https://docs.oracle.com/cd/B14117_01/appdev.101/b10807/07_errs.htm
DECLARE
   past_due EXCEPTION;
   acct_num NUMBER;
BEGIN
   DECLARE  ---------- sub-block begins
      past_due EXCEPTION;  -- this declaration prevails
      acct_num NUMBER;
     due_date DATE := SYSDATE - 1;
     todays_date DATE := SYSDATE;
   BEGIN
      IF due_date < todays_date THEN
         RAISE past_due;  -- this is not the same as the exterior past_due; caught by OTHERS
      END IF;
   END;  ------------- sub-block ends
EXCEPTION
   WHEN past_due THEN  -- does not handle RAISEd exception
      dbms_output.put_line('Handling PAST_DUE exception.');
   WHEN OTHERS THEN
     dbms_output.put_line('Could not recognize PAST_DUE_EXCEPTION in this scope.');
END;
/


+++++ Exception - user-declared exception; interior hides exterior
Knownbad
Injection true

-- Modified from https://docs.oracle.com/cd/B14117_01/appdev.101/b10807/07_errs.htm
CREATE OR REPLACE PROCEDURE sampleproc (
  d1 VARCHAR2,
  d2 VARCHAR2
)
IS
   past_due EXCEPTION;
   acct_num NUMBER;
BEGIN
   DECLARE
      past_due EXCEPTION; 
      acct_num NUMBER;
     due_date DATE := SYSDATE - 1;
     todays_date DATE := SYSDATE;
   BEGIN
      IF due_date < todays_date THEN
         RAISE past_due;  -- this is not handled
      END IF;
   END;
EXCEPTION
   WHEN past_due THEN  -- Verify which exception handles the raised exception
      execute immediate 'SELECT count(1) FROM ' || d1  INTO acct_num; 
   WHEN OTHERS THEN
      execute immediate 'SELECT count(1) FROM ' || d2  INTO acct_num; 
END;
/

+++++ Understands function returns clean value
Injection false

create or replace procedure sqli(p_name varchar2) as
l_name varchar2(200);

  function clean_name(x_name varchar2) return varchar2 as
begin
   return 'clean string';
end;


begin
    l_name := clean_name(p_name);
    execute immediate 'update  emp set sal = 1000000000 where ename = '||l_name;
end;
/


+++++ Finds injection in nested function
Injection true
Dataflow p_name:1 x_name:4 x_name:6

create or replace procedure sqli(p_name varchar2) as
l_name varchar2(200);

  function clean_name(x_name varchar2) return varchar2 as
begin
   execute immediate 'update  emp set sal = 1000000000 where ename = '||x_name;
   return 'clean string';
end;


begin
    l_name := clean_name(p_name);
end;
/


+++++ Understands clean parameter to nested function
Injection false

-- TODO Verify that the nested function clean_name is uncallable from outside this compilation unit!

create or replace procedure sqli(p_name varchar2) as
l_name varchar2(200);

  function clean_name(x_name varchar2) return varchar2 as
begin
   execute immediate 'update  emp set sal = 1000000000 where ename = '||x_name;
   return 'clean string';
end;


begin
    l_name := clean_name('clean string');
    l_name := clean_name(dbms_assert.enquote_literal(p_name));
end;
/


+++++ Understands user-function cleans parameter before return
Injection false

create or replace procedure sqli(p_name varchar2) as
l_name varchar2(200);

  function clean_name(x_name varchar2) return varchar2 as
begin
   return dbms_assert.enquote_literal(x_name);
end;


begin
    l_name := clean_name(p_name);

    execute immediate 'update  emp set sal = 1000000000 where ename = '||l_name;
end;
/


+++++ Does parser understand JSON?
Injection false

DECLARE
  je JSON_ELEMENT_T;
  jo JSON_OBJECT_T;
BEGIN
  je := JSON_ELEMENT_T.parse('{"name":"Radio controlled plane"}');
  IF (je.is_Object) THEN
    jo := treat(je AS JSON_OBJECT_T);
    jo.put('price', 149.99);
  END IF;
  DBMS_OUTPUT.put_line(je.to_string);
END;
/

+++++ PRAGMA works
UNSKIP
knownbad
Injection true

CREATE OR REPLACE FUNCTION sample (s1 VARCHAR2, d1 NUMBER)
RETURN NUMBER IS
BEGIN
  dbms_assert.noop('SQLINJ fail');
  RETURN 42;
END;
/


+++++ PRAGMA must be literal
Injection false

CREATE OR REPLACE FUNCTION sample (d1 VARCHAR2)
RETURN NUMBER IS
  injpragma VARCHAR2 :=  'SQLINJ fail';
BEGIN
  dbms_assert.noop(injpragma);
  RETURN 42;
END;
/



+++++ Does parser understand JSON?
Injection false

DECLARE
  je JSON_ELEMENT_T;
  jo JSON_OBJECT_T;
BEGIN
  je := JSON_ELEMENT_T.parse('{"name":"Radio controlled plane"}');
  IF (je.is_Object) THEN
    jo := treat(je AS JSON_OBJECT_T);
    jo.put('price', 149.99);
  END IF;
  DBMS_OUTPUT.put_line(je.to_string);
END;
/


+++++ Parsing of dotted thngs
Injection false

-- Copied from https://docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/plsql-name-resolution.html accessed 2019 October 25
--
-- Cf. https://docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/plsql-language-fundamentals.html accessed 2019 November 06
--
--
-- Cf. https://docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/plsql-language-fundamentals.html accessed 2019 November 06
--
--

CREATE OR REPLACE PACKAGE pkg1 AUTHID DEFINER AS
  m NUMBER;
  TYPE t1 IS RECORD (a NUMBER);
  v1 t1;
  TYPE t2 IS TABLE OF t1 INDEX BY PLS_INTEGER;
  v2 t2; 
  FUNCTION f1 (p1 NUMBER) RETURN t1;
  FUNCTION f2 (q1 NUMBER) RETURN t2;
END pkg1;
/


CREATE OR REPLACE PACKAGE BODY pkg1 AS

  FUNCTION f1 (p1 NUMBER) RETURN t1 IS
    n NUMBER;
  BEGIN
     n := m;             -- Unqualified variable name
     n := pkg1.m;        -- Variable name qualified by package name
     n := pkg1.f1.p1;    -- Parameter name qualified by function name,
                         --  which is qualified by package name
     n := v1.a;          -- Variable name followed by component name
     n := pkg1.v1.a;     -- Variable name qualified by package name
                         --  and followed by component name
     n := v2(10).a;      -- Indexed name followed by component name
     n := f1(10).a;      -- Function invocation followed by component name
     n := f2(10)(10).a;  -- Function invocation followed by indexed name
                         --  and followed by component name
     n := hr.pkg1.f2(10)(10).a;  -- Schema name, package name,
                                 -- function invocation, index, component name
     v1.a := p1;
     RETURN v1;
   END f1;

   FUNCTION f2 (q1 NUMBER) RETURN t2 IS
     v_t1 t1;
     v_t2 t2;
   BEGIN
     v_t1.a := q1;
     v_t2(1) := v_t1;
     RETURN v_t2;
   END f2;
END pkg1;
/

+++++ Example B-2 Variable Name Interpreted as Column Name
Injection falseo


-- From https://docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/plsql-name-resolution.html#GUID-A551A39D-7DF8-4525-BE0F-1F46FE04ED09 accessed 2019 October 25
-- TODO Modify to see if SQLInj does right thing

DROP TABLE employees2;
CREATE TABLE employees2 AS
  SELECT LAST_NAME FROM employees;


DECLARE
  last_name  VARCHAR2(10) := 'King';
BEGIN
  DELETE FROM employees2 WHERE LAST_NAME = last_name;
  DBMS_OUTPUT.PUT_LINE('Deleted ' || SQL%ROWCOUNT || ' rows.');
END;

/

+++++ Fixing Example B-2 with Block Label
Injection falseo


-- From https://docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/plsql-name-resolution.html#GUID-A551A39D-7DF8-4525-BE0F-1F46FE04ED09 accessed 2019 October 25
-- TODO Modify to see if SQLInj does right thing

<<main>>
DECLARE
  last_name  VARCHAR2(10) := 'King';
BEGIN
  DELETE FROM employees2 WHERE last_name = main.last_name;
  DBMS_OUTPUT.PUT_LINE('Deleted ' || SQL%ROWCOUNT || ' rows.');
END;
/


+++++ SQLIJ_TEST package declaration for testing - not an actual test
SETUP
Overloads
Setup
Injection false


CREATE OR REPLACE PACKAGE sqlinj_overload_1 as
  PROCEDURE main(
        n_integer PLS_INTEGER,   unsafe_integer VARCHAR2,
        n_number  NUMBER,        unsafe_number  VARCHAR2,
        n_float   BINARY_FLOAT,  unsafe_float   VARCHAR2,
        n_double  BINARY_DOUBLE, unsafe_double  VARCHAR2,
        vc2       VARCHAR2,      unsafe_vc2     VARCHAR2,
        v_clob    CLOB,          unsafe_clob    VARCHAR2);
  FUNCTION alpha (p1 PLS_INTEGER,      p2 VARCHAR2) RETURN VARCHAR2;
  FUNCTION alpha (p1 NUMBER,           p2 VARCHAR2) RETURN VARCHAR2;
  FUNCTION alpha (p1 BINARY_FLOAT,     p2 VARCHAR2) RETURN VARCHAR2;
  FUNCTION alpha (p1 BINARY_DOUBLE,    p2 VARCHAR2) RETURN VARCHAR2;
  FUNCTION alpha (p1 VARCHAR2,         p2 VARCHAR2) RETURN VARCHAR2;
  FUNCTION alpha (p1 CLOB,             p2 VARCHAR2) RETURN VARCHAR2;
END sqlinj_overload_1;
/


CREATE OR REPLACE PACKAGE BODY sqlinj_overload_1 as
  PROCEDURE main(
    n_integer PLS_INTEGER,   unsafe_integer VARCHAR2,
    n_number  NUMBER,        unsafe_number  VARCHAR2,
    n_float   BINARY_FLOAT,  unsafe_float   VARCHAR2,
    n_double  BINARY_DOUBLE, unsafe_double  VARCHAR2,
    vc2       VARCHAR2,      unsafe_vc2     VARCHAR2,
    v_clob    CLOB,          unsafe_clob    VARCHAR2) IS
  BEGIN
    -- execute immediate alpha(n_integer, unsafe_integer);
    -- execute immediate alpha(n_number,  unsafe_number);
    -- execute immediate alpha(n_float,   unsafe_float);
    -- execute immediate alpha(n_double,  unsafe_double);
    execute immediate alpha(vc2,       unsafe_vc2);
    -- execute immediate alpha(v_clob,    unsafe_clob);
  END;

  FUNCTION alpha (p1 PLS_INTEGER,      p2 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    RETURN p2;
  END;

  FUNCTION alpha (p1 NUMBER,           p2 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    RETURN p2;
  END;

  FUNCTION alpha (p1 BINARY_FLOAT,     p2 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    RETURN p2;
  END;

  FUNCTION alpha (p1 BINARY_DOUBLE,    p2 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    RETURN p2;
  END;

  FUNCTION alpha (p1 VARCHAR2,         p2 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    RETURN p2;
  END;

  FUNCTION alpha (p1 CLOB,             p2 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    RETURN p2;
  END;
END sqlinj_overload_1;
/



+++++ SQLIJ_TEST_2 package declaration for testing - not an actual test
SETUP
FunctionFlow
Setup
Injection true
Dataflow D1:1 D1:-16 D1:18

CREATE OR REPLACE PACKAGE sqlinj_test_2 as
  PROCEDURE main(unsafe VARCHAR2);
  PROCEDURE rotate_two (
       a0 IN OUT VARCHAR2,
       a1 IN OUT VARCHAR2,
       a2 IN OUT VARCHAR2,
       a3 IN OUT VARCHAR2,
       a4 IN OUT VARCHAR2,
       a5 IN OUT VARCHAR2
  );
  PROCEDURE rotate_three (
       b0 IN OUT VARCHAR2,
       b1 IN OUT VARCHAR2,
       b2 IN OUT VARCHAR2,
       b3 IN OUT VARCHAR2,
       b4 IN OUT VARCHAR2,
       b5 IN OUT VARCHAR2
  );
END sqlinj_test_2;
/







+++++ Discrete Math
DEMO
FunctionFlow
Injection true
Dataflow P1:2 F1:11 X:15 ret:16 T1:6 T1:7

CREATE OR REPLACE PACKAGE BODY sqlinj_test_2 as
  PROCEDURE main(unsafe VARCHAR2) IS
       m0 VARCHAR2(10);
       m1 VARCHAR2(10) := 'the';
       m2 VARCHAR2(10) := 'quick';
       m3 VARCHAR2(10) := 'brown';
       m4 VARCHAR2(10) := 'fox';
       m5 VARCHAR2(10) := 'jumped';
       val NUMBER := 0;
  BEGIN
    m0 := unsafe;
    WHILE val < 10 LOOP
      IF (dbms_random.value(1,10) < 5) THEN
        rotate_two(m0, m1, m2, m3, m4, m5);
      ELSE
        rotate_three(m0, m1, m2, m3, m4, m5);
      END IF;
      val := val + 1;
    END LOOP;
    execute immediate 'SELECT count(1) FROM ' || m0 INTO val; 
  END;

  PROCEDURE rotate_two(
       a0 IN OUT VARCHAR2,
       a1 IN OUT VARCHAR2,
       a2 IN OUT VARCHAR2,
       a3 IN OUT VARCHAR2,
       a4 IN OUT VARCHAR2,
       a5 IN OUT VARCHAR2
  ) IS
    temp VARCHAR2(10);
  BEGIN
    temp := a4;
    a4 := a2;
    a2 := a0;
    a0 := temp;
    temp := a5;
    a5 := a3;
    a3 := a1;
    a1 := temp;
  END;

  PROCEDURE rotate_three(
       b0 IN OUT VARCHAR2,
       b1 IN OUT VARCHAR2,
       b2 IN OUT VARCHAR2,
       b3 IN OUT VARCHAR2,
       b4 IN OUT VARCHAR2,
       b5 IN OUT VARCHAR2
  ) IS
    temp VARCHAR2(10);
  BEGIN
    temp := b3;
    b3 := b0;
    b0 := temp;
    temp := b5;
    b5 := b2;
    b2 := temp;
    temp := b4;
    b4 := b1;
    b1 := temp;
  END;

END sqlinj_test_2;
/


+++++ Function calling function
DEMO
FunctionFlow
Injection true
Dataflow P1:2 F1:11 X:15 ret:16 T1:6 T1:7

CREATE OR REPLACE PACKAGE BODY sqlinj_test as
  PROCEDURE main(m1 VARCHAR2) IS
    t1 VARCHAR2(100);
    ret VARCHAR2(100);
  BEGIN
    t1 := alpha(m1);
    execute immediate 'SELECT count(1) FROM ' || t1  INTO ret; 
  END;
  --
  FUNCTION alpha (a1 VARCHAR2) RETURN VARCHAR2 IS
    x   VARCHAR2(100);
    ret VARCHAR2(100);
  BEGIN
    x := a1;
    ret := x;
    RETURN ret;
  END;
  --
  FUNCTION bravo (b1 IN OUT VARCHAR2, b2 IN OUT VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    return 'Safe';
  END;
  FUNCTION charlie (c1 VARCHAR2, c2 VARCHAR2, c3 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    return 'Safe';
  END;
END sqlinj_test;
/



+++++ OUT parameters
DEMO
GREEN
FunctionFlow
Injection true
Dataflow M1:2 T1:7 B1:22 B2:9 T2:9 T2:10

CREATE OR REPLACE PACKAGE BODY sqlinj_test as
  PROCEDURE main(m1 VARCHAR2) IS
    t1 VARCHAR2(100);
    t2 VARCHAR2(100);
    ret VARCHAR2(100);
  BEGIN
    t1 := m1;
    t2 := 'SAFE';
    ret := bravo(t1, t2);
    execute immediate 'SELECT count(1) FROM ' || t2  INTO ret; 
  END;
--
  FUNCTION alpha (a1 VARCHAR2) RETURN VARCHAR2 IS
    x   VARCHAR2(100);
    ret VARCHAR2(100);
  BEGIN
    x := a1;
    ret := x;
    RETURN ret;
  END;
--
  FUNCTION bravo (b1 IN OUT VARCHAR2, b2 IN OUT VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    b2 := b1 || b2;
    return 'Safe';
  END;
--
  FUNCTION charlie (c1 VARCHAR2, c2 VARCHAR2, c3 VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    return 'Safe';
  END;
END sqlinj_test;
/



+++++ Debugging function calling function
FunctionFlow
Injection true
Dataflow M1:2 A1:10 :6 T1:6 T1:7

CREATE OR REPLACE PACKAGE BODY sqlinj_test as
  PROCEDURE main(m1 VARCHAR2) IS
    t1 VARCHAR2(100);
    ret VARCHAR2(100);
  BEGIN
    t1 := alpha(m1);
    execute immediate 'SELECT count(1) FROM ' || t1  INTO ret; 
  END;
  --
  FUNCTION alpha (a1 VARCHAR2) RETURN VARCHAR2 IS
    x   VARCHAR2(100);
    ret VARCHAR2(100);
  BEGIN
    x := a1;
    ret := x;
    RETURN ret;
  END;
END sqlinj_test;
/


