/*
 * Decompiled with CFR 0.152.
 */
package oracle.dbtools.ords.metrics;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.regex.Pattern;
import oracle.dbtools.ords.metrics.AccessLogRecord;
import oracle.dbtools.ords.metrics.MetricReporter;
import oracle.dbtools.ords.metrics.OrdsMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AccessLog {
    private static final Logger log = LoggerFactory.getLogger(AccessLog.class);
    private static final long MSEC_IN_DAY = 86400000L;
    private static final long DEFAULT_WAIT_MSEC = 5000L;
    private static final long FLUSH_AGE_MAX = 15000L;
    private static final long FLUSH_AGE_MIN = 3000L;
    private static final DateTimeFormatter FILE_FORMATTER = DateTimeFormatter.ofPattern("yyyy_MM_dd").withZone(ZoneOffset.UTC);
    static final Pattern fileNamePattern = Pattern.compile("ords[_-]\\d{4}_\\d{2}_\\d{2}(\\.request)?\\.log");
    static long waitTimeMsec = 5000L;
    private final MetricReporter reporter;
    private final File dir;
    private final long startDate;
    private final long endDate;
    private long lastRecordTime = 0L;
    private RecordHash lastScannedRecord = null;
    private long nextFlushTime = 0L;
    private File currentFile;
    private FileNameType fileNameType = FileNameType.UNKNOWN;
    private final boolean verbose;

    AccessLog(MetricReporter reporter, File dir, long startDate, long endDate, boolean verbose) {
        this.reporter = reporter;
        this.dir = dir;
        this.startDate = startDate;
        this.endDate = endDate;
        this.verbose = verbose;
    }

    long scan() {
        File[] files = this.dir.listFiles((dir, name) -> {
            long fileStartDate = AccessLog.startOfDay(this.startDate);
            long fileEndDate = AccessLog.startOfDay(this.endDate);
            if (fileNamePattern.matcher(name).matches()) {
                long fileTimestamp = AccessLog.startOfDay(name);
                return fileTimestamp >= fileStartDate && fileTimestamp <= fileEndDate;
            }
            return false;
        });
        if (files != null) {
            Arrays.stream(files).sorted().forEach(this::scanLog);
            this.reporter.flush();
        }
        return this.lastRecordTime;
    }

    long tail() {
        String[] fileName = new String[]{"ords_%s.log", "ords-%s.request.log"};
        int fileNameIdx = 0;
        while (this.notDone(true)) {
            try {
                if (this.currentFile == null) {
                    long startofToday = AccessLog.startOfDay(System.currentTimeMillis());
                    String fileNameTemplate = this.fileNameType == FileNameType.UNKNOWN ? fileName[fileNameIdx] : (this.fileNameType == FileNameType.ORDS ? fileName[0] : fileName[1]);
                    Path nextFile = this.dir.toPath().resolve(String.format(fileNameTemplate, FILE_FORMATTER.format(Instant.ofEpochMilli(startofToday))));
                    if (Files.exists(nextFile, new LinkOption[0])) {
                        this.currentFile = nextFile.toFile();
                        this.setFileNameType(this.currentFile);
                    } else {
                        if (this.fileNameType == FileNameType.UNKNOWN) {
                            fileNameIdx ^= 1;
                        }
                        log.debug("{} does not exist. Waiting...", (Object)nextFile);
                        AccessLog.nap();
                    }
                }
                if (this.currentFile == null) continue;
                this.tailLog(this.currentFile);
                this.currentFile = null;
            }
            catch (Exception e) {
                log.warn("Unexpected error while tailing log: {}. file: {} last record time: {}", new Object[]{e, this.currentFile, this.lastRecordTime});
                this.currentFile = null;
            }
        }
        this.reporter.flush();
        return this.lastRecordTime;
    }

    private void scanLog(File file) {
        this.currentFile = file;
        this.setFileNameType(this.currentFile);
        try (BufferedReader reader = new BufferedReader(new FileReader(file));){
            String line;
            while (this.notDone(false) && (line = reader.readLine()) != null) {
                long timestamp;
                AccessLogRecord record = AccessLogRecord.getLogRecord(line);
                long l = timestamp = record == null ? AccessLogRecord.getTimestamp(line) : record.timestamp();
                if (timestamp > this.lastRecordTime) {
                    this.lastRecordTime = timestamp;
                }
                if (record != null && timestamp >= this.startDate && timestamp <= this.endDate) {
                    if (this.verbose) {
                        log.info(record.toString());
                    }
                    this.reporter.emit(record);
                }
                this.lastScannedRecord = new RecordHash(file, AccessLog.hash(line));
            }
        }
        catch (IOException e) {
            log.error("Cannot process log file {}: {}", (Object)file.getName(), (Object)e.getMessage());
        }
    }

    void tailLog(File file) {
        log.info("Tailing {}", (Object)file.getName());
        if (this.lastScannedRecord != null && this.lastScannedRecord.file != file) {
            this.lastScannedRecord = null;
        }
        this.nextFlushTime = System.currentTimeMillis() + 15000L;
        long fileStartOfDay = AccessLog.startOfDay(file.getName());
        try (BufferedReader input = new BufferedReader(new FileReader(file));){
            while (this.notDone(false)) {
                String line = input.readLine();
                if (line != null) {
                    long timestamp;
                    if (this.lastScannedRecord != null) {
                        if (this.lastScannedRecord.hash != AccessLog.hash(line)) continue;
                        this.lastScannedRecord = null;
                        continue;
                    }
                    AccessLogRecord record = AccessLogRecord.getLogRecord(line);
                    long l = timestamp = record == null ? AccessLogRecord.getTimestamp(line) : record.timestamp();
                    if (timestamp > this.lastRecordTime) {
                        this.lastRecordTime = timestamp;
                    }
                    if (record != null && timestamp <= this.endDate) {
                        if (this.verbose) {
                            log.info("Processing {}", (Object)record);
                        }
                        this.reporter.emit(record);
                    }
                    this.flushAged();
                    continue;
                }
                if (AccessLog.startOfDay(System.currentTimeMillis()) > fileStartOfDay) break;
                if (System.currentTimeMillis() > this.endDate) {
                    break;
                }
                this.flushAged();
                AccessLog.nap();
            }
        }
        catch (IOException e) {
            log.error("Cannot tail log file {}: {}", (Object)file.getName(), (Object)e.getMessage());
        }
    }

    private void flushAged() {
        long now = System.currentTimeMillis();
        if (now > this.nextFlushTime) {
            this.reporter.flush(now - 3000L);
            this.nextFlushTime = now + 15000L;
        }
    }

    private void setFileNameType(File file) {
        if (this.fileNameType == FileNameType.UNKNOWN) {
            this.fileNameType = file.getName().startsWith("ords_") ? FileNameType.ORDS : FileNameType.ADBS;
        }
    }

    private boolean notDone(boolean checkEndTime) {
        return !OrdsMetrics.shuttingDown.get() && this.lastRecordTime < this.endDate && (!checkEndTime || System.currentTimeMillis() < this.endDate);
    }

    private static int hash(String input) {
        int h = 0;
        for (byte b : input.getBytes()) {
            h *= 16777619;
            h ^= b & 0xFF;
        }
        return h;
    }

    private static long startOfDay(long timestmap) {
        return timestmap - timestmap % 86400000L;
    }

    static long startOfDay(String fileName) {
        String dateString = fileName.substring(5, 15);
        return LocalDate.parse(dateString, FILE_FORMATTER).atStartOfDay(ZoneOffset.UTC).toInstant().toEpochMilli();
    }

    static void setWaitTimeMsec(long msec) {
        waitTimeMsec = msec;
    }

    private static void nap() {
        try {
            Thread.sleep(waitTimeMsec);
        }
        catch (InterruptedException e) {
            log.info("wait interrupted");
        }
    }

    record RecordHash(File file, int hash) {
    }

    private static enum FileNameType {
        UNKNOWN,
        ORDS,
        ADBS;

    }
}

