/**
 * @license
 * Copyright (c) 2014, 2022, Oracle and/or its affiliates.
 * Licensed under The Universal Permissive License (UPL), Version 1.0
 * as shown at https://oss.oracle.com/licenses/upl/
 * @ignore
 */
import 'touchr';
import 'ojs/ojdatasource-common';
import 'ojs/ojdatacollection-utils';
import 'ojs/ojinputnumber';
import 'ojs/ojmenu';
import 'ojs/ojmenuselectmany';
import 'ojs/ojdialog';
import 'ojs/ojbutton';
import oj$1 from 'ojs/ojcore-base';
import { _OJ_CONTAINER_ATTR, subtreeAttached, __GetWidgetConstructor } from 'ojs/ojcomponentcore';
import Context from 'ojs/ojcontext';
import { isMobileTouchDevice, applyMergedInlineStyles, disableAllFocusableElements, enableAllFocusableElements, getFocusableElementsInNode, getFocusableElementsIncludingDisabled, handleActionableTab, handleActionablePrevTab, disableDefaultBrowserStyling } from 'ojs/ojdatacollection-common';
import { setScrollLeft, getCSSTimeUnitAsMillis, isTouchSupported, removeResizeListener, addResizeListener } from 'ojs/ojdomutils';
import { __getTemplateEngine } from 'ojs/ojconfig';
import { CustomElementUtils } from 'ojs/ojcustomelement-utils';
import { getLogicalChildPopup } from 'ojs/ojkeyboardfocus-utils';
import $ from 'jquery';
import { error } from 'ojs/ojlogger';
import { getCachedCSSVarValues } from 'ojs/ojthemeutils';

(function () {
var __oj_data_grid_metadata = 
{
  "properties": {
    "bandingInterval": {
      "type": "object",
      "properties": {
        "column": {
          "type": "number",
          "value": 0
        },
        "row": {
          "type": "number",
          "value": 0
        }
      }
    },
    "cell": {
      "type": "object",
      "properties": {
        "className": {
          "type": "function|string"
        },
        "renderer": {
          "type": "function"
        },
        "style": {
          "type": "function|string"
        }
      }
    },
    "currentCell": {
      "type": "object",
      "writeback": true
    },
    "data": {
      "type": "DataGridProvider",
      "extension": {
        "webelement": {
          "exceptionStatus": [
            {
              "type": "deprecated",
              "since": "11.0.0",
              "description": "Data sets from a DataProvider cannot be sent to WebDriverJS; use ViewModels or page variables instead."
            }
          ]
        }
      }
    },
    "dataTransferOptions": {
      "type": "object",
      "properties": {
        "copy": {
          "type": "string",
          "enumValues": [
            "disable",
            "enable"
          ],
          "value": "disable"
        },
        "cut": {
          "type": "string",
          "enumValues": [
            "disable",
            "enable"
          ],
          "value": "disable"
        },
        "fill": {
          "type": "string",
          "enumValues": [
            "disable",
            "enable"
          ],
          "value": "disable"
        },
        "paste": {
          "type": "string",
          "enumValues": [
            "disable",
            "enable"
          ],
          "value": "disable"
        }
      }
    },
    "dnd": {
      "type": "object",
      "properties": {
        "reorder": {
          "type": "object",
          "properties": {
            "row": {
              "type": "string",
              "enumValues": [
                "disable",
                "enable"
              ],
              "value": "disable"
            }
          }
        }
      }
    },
    "editMode": {
      "type": "string",
      "writeback": true,
      "enumValues": [
        "cellEdit",
        "cellNavigation",
        "none"
      ],
      "value": "none"
    },
    "gridlines": {
      "type": "object",
      "properties": {
        "horizontal": {
          "type": "string",
          "enumValues": [
            "hidden",
            "visible"
          ],
          "value": "visible"
        },
        "vertical": {
          "type": "string",
          "enumValues": [
            "hidden",
            "visible"
          ],
          "value": "visible"
        }
      }
    },
    "header": {
      "type": "object",
      "properties": {
        "column": {
          "type": "object",
          "properties": {
            "className": {
              "type": "function|string"
            },
            "label": {
              "type": "object",
              "properties": {
                "className": {
                  "type": "function|string"
                },
                "renderer": {
                  "type": "function"
                },
                "style": {
                  "type": "function|string"
                }
              }
            },
            "renderer": {
              "type": "function"
            },
            "resizable": {
              "type": "object",
              "properties": {
                "height": {
                  "type": "string",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                },
                "width": {
                  "type": "string|function",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                }
              }
            },
            "sortable": {
              "type": "function|string",
              "enumValues": [
                "auto",
                "disable",
                "enable"
              ],
              "value": "auto"
            },
            "style": {
              "type": "function|string"
            }
          }
        },
        "columnEnd": {
          "type": "object",
          "properties": {
            "className": {
              "type": "function|string"
            },
            "label": {
              "type": "object",
              "properties": {
                "className": {
                  "type": "function|string"
                },
                "renderer": {
                  "type": "function"
                },
                "style": {
                  "type": "function|string"
                }
              }
            },
            "renderer": {
              "type": "function"
            },
            "resizable": {
              "type": "object",
              "properties": {
                "height": {
                  "type": "string",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                },
                "width": {
                  "type": "string|function",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                }
              }
            },
            "style": {
              "type": "function|string"
            }
          }
        },
        "row": {
          "type": "object",
          "properties": {
            "className": {
              "type": "function|string"
            },
            "label": {
              "type": "object",
              "properties": {
                "className": {
                  "type": "function|string"
                },
                "renderer": {
                  "type": "function"
                },
                "style": {
                  "type": "function|string"
                }
              }
            },
            "renderer": {
              "type": "function"
            },
            "resizable": {
              "type": "object",
              "properties": {
                "height": {
                  "type": "string|function",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                },
                "width": {
                  "type": "string",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                }
              }
            },
            "sortable": {
              "type": "function|string",
              "enumValues": [
                "auto",
                "disable",
                "enable"
              ],
              "value": "auto"
            },
            "style": {
              "type": "function|string"
            }
          }
        },
        "rowEnd": {
          "type": "object",
          "properties": {
            "className": {
              "type": "function|string"
            },
            "label": {
              "type": "object",
              "properties": {
                "className": {
                  "type": "function|string"
                },
                "renderer": {
                  "type": "function"
                },
                "style": {
                  "type": "function|string"
                }
              }
            },
            "renderer": {
              "type": "function"
            },
            "resizable": {
              "type": "object",
              "properties": {
                "height": {
                  "type": "string|function",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                },
                "width": {
                  "type": "string",
                  "enumValues": [
                    "disable",
                    "enable"
                  ],
                  "value": "disable"
                }
              }
            },
            "style": {
              "type": "function|string"
            }
          }
        }
      }
    },
    "scrollPolicy": {
      "type": "string",
      "enumValues": [
        "auto",
        "loadMoreOnScroll",
        "scroll"
      ],
      "value": "auto"
    },
    "scrollPolicyOptions": {
      "type": "object",
      "properties": {
        "maxColumnCount": {
          "type": "number",
          "value": 500
        },
        "maxRowCount": {
          "type": "number",
          "value": 500
        }
      }
    },
    "scrollPosition": {
      "type": "object",
      "writeback": true,
      "value": {
        "x": 0,
        "y": 0
      },
      "properties": {
        "columnIndex": {
          "type": "number"
        },
        "columnKey": {
          "type": "any"
        },
        "offsetX": {
          "type": "number"
        },
        "offsetY": {
          "type": "number"
        },
        "rowIndex": {
          "type": "number"
        },
        "rowKey": {
          "type": "any"
        },
        "x": {
          "type": "number"
        },
        "y": {
          "type": "number"
        }
      }
    },
    "scrollToKey": {
      "type": "string",
      "enumValues": [
        "always",
        "auto",
        "capability",
        "never"
      ],
      "value": "auto"
    },
    "selection": {
      "type": "Array<Object>",
      "writeback": true,
      "value": []
    },
    "selectionMode": {
      "type": "object",
      "properties": {
        "cell": {
          "type": "string",
          "enumValues": [
            "multiple",
            "none",
            "single"
          ],
          "value": "none"
        },
        "row": {
          "type": "string",
          "enumValues": [
            "multiple",
            "none",
            "single"
          ],
          "value": "none"
        }
      }
    },
    "translations": {
      "type": "object",
      "value": {},
      "properties": {
        "accessibleActionableMode": {
          "type": "string"
        },
        "accessibleCollapsed": {
          "type": "string"
        },
        "accessibleColumnContext": {
          "type": "string"
        },
        "accessibleColumnEndHeaderContext": {
          "type": "string"
        },
        "accessibleColumnEndHeaderLabelContext": {
          "type": "string"
        },
        "accessibleColumnHeaderContext": {
          "type": "string"
        },
        "accessibleColumnHeaderLabelContext": {
          "type": "string"
        },
        "accessibleColumnSelected": {
          "type": "string"
        },
        "accessibleColumnSpanContext": {
          "type": "string"
        },
        "accessibleContainsControls": {
          "type": "string"
        },
        "accessibleExpanded": {
          "type": "string"
        },
        "accessibleFirstColumn": {
          "type": "string"
        },
        "accessibleFirstRow": {
          "type": "string"
        },
        "accessibleLastColumn": {
          "type": "string"
        },
        "accessibleLastRow": {
          "type": "string"
        },
        "accessibleLevelContext": {
          "type": "string"
        },
        "accessibleMultiCellSelected": {
          "type": "string"
        },
        "accessibleNavigationMode": {
          "type": "string"
        },
        "accessibleRangeSelectModeOff": {
          "type": "string"
        },
        "accessibleRangeSelectModeOn": {
          "type": "string"
        },
        "accessibleRowCollapsed": {
          "type": "string"
        },
        "accessibleRowContext": {
          "type": "string"
        },
        "accessibleRowEndHeaderContext": {
          "type": "string"
        },
        "accessibleRowEndHeaderLabelContext": {
          "type": "string"
        },
        "accessibleRowExpanded": {
          "type": "string"
        },
        "accessibleRowHeaderContext": {
          "type": "string"
        },
        "accessibleRowHeaderLabelContext": {
          "type": "string"
        },
        "accessibleRowSelected": {
          "type": "string"
        },
        "accessibleRowSpanContext": {
          "type": "string"
        },
        "accessibleSelectionAffordanceBottom": {
          "type": "string"
        },
        "accessibleSelectionAffordanceTop": {
          "type": "string"
        },
        "accessibleSortAscending": {
          "type": "string"
        },
        "accessibleSortDescending": {
          "type": "string"
        },
        "accessibleStateSelected": {
          "type": "string"
        },
        "accessibleSummaryEstimate": {
          "type": "string"
        },
        "accessibleSummaryExact": {
          "type": "string"
        },
        "accessibleSummaryExpanded": {
          "type": "string"
        },
        "collapsedText": {
          "type": "string"
        },
        "columnWidth": {
          "type": "string"
        },
        "expandedText": {
          "type": "string"
        },
        "labelCopyCells": {
          "type": "string"
        },
        "labelCut": {
          "type": "string"
        },
        "labelCutCells": {
          "type": "string"
        },
        "labelDisableNonContiguous": {
          "type": "string"
        },
        "labelEnableNonContiguous": {
          "type": "string"
        },
        "labelFillCells": {
          "type": "string"
        },
        "labelPaste": {
          "type": "string"
        },
        "labelPasteCells": {
          "type": "string"
        },
        "labelResize": {
          "type": "string"
        },
        "labelResizeColumn": {
          "type": "string"
        },
        "labelResizeDialogApply": {
          "type": "string"
        },
        "labelResizeDialogCancel": {
          "type": "string"
        },
        "labelResizeDialogSubmit": {
          "type": "string"
        },
        "labelResizeFitToContent": {
          "type": "string"
        },
        "labelResizeHeight": {
          "type": "string"
        },
        "labelResizeRow": {
          "type": "string"
        },
        "labelResizeWidth": {
          "type": "string"
        },
        "labelSelectMultiple": {
          "type": "string"
        },
        "labelSortCol": {
          "type": "string"
        },
        "labelSortColAsc": {
          "type": "string"
        },
        "labelSortColDsc": {
          "type": "string"
        },
        "labelSortRow": {
          "type": "string"
        },
        "labelSortRowAsc": {
          "type": "string"
        },
        "labelSortRowDsc": {
          "type": "string"
        },
        "msgFetchingData": {
          "type": "string"
        },
        "msgNoData": {
          "type": "string"
        },
        "resizeColumnDialog": {
          "type": "string"
        },
        "resizeRowDialog": {
          "type": "string"
        },
        "rowHeight": {
          "type": "string"
        }
      }
    }
  },
  "methods": {
    "getContextByNode": {},
    "getProperty": {},
    "refresh": {},
    "setProperties": {},
    "setProperty": {},
    "getNodeBySubId": {},
    "getSubIdByNode": {}
  },
  "events": {
    "ojBeforeCurrentCell": {},
    "ojBeforeEdit": {},
    "ojBeforeEditEnd": {},
    "ojCollapseRequest": {},
    "ojCopyRequest": {},
    "ojCutRequest": {},
    "ojExpandRequest": {},
    "ojFillRequest": {},
    "ojPasteRequest": {},
    "ojResize": {},
    "ojScroll": {},
    "ojSort": {},
    "ojSortRequest": {}
  },
  "extension": {}
};
  __oj_data_grid_metadata.extension._WIDGET_NAME = 'ojDataGrid';
  oj$1.CustomElementBridge.register('oj-data-grid', { metadata: __oj_data_grid_metadata });
}());

/**
 * @export
 * This class captures all translation resources and style classes used by the DataGrid.
 * This should be populated with information extracted through the framework and set on the DataGrid.
 * Internal.  Developers should never use this class.
 * @constructor
 * @ignore
 */
// eslint-disable-next-line max-len
const DataGridResources = function (rtlMode, translationFunction, defaultOptions, widgetName) {
  this.rtlMode = rtlMode;
  this.translationFunction = translationFunction;
  this.defaultOptions = defaultOptions;
  this.widgetName = widgetName;
  this.styles = {};
  this.styles.datagrid = 'oj-datagrid';
  this.styles.cell = 'oj-datagrid-cell';
  this.styles.banded = 'oj-datagrid-banded';
  this.styles.row = 'oj-datagrid-row';
  this.styles.databody = 'oj-datagrid-databody';
  this.styles.topcorner = 'oj-datagrid-top-corner';
  this.styles.bottomcorner = 'oj-datagrid-bottom-corner';
  this.styles.rowheaderspacer = 'oj-datagrid-row-header-spacer';
  this.styles.colheaderspacer = 'oj-datagrid-column-header-spacer';
  this.styles.status = 'oj-datagrid-status';
  this.styles.loadingicon = 'oj-icon oj-datagrid-loading-icon';
  this.styles.emptytext = 'oj-datagrid-empty-text';
  this.styles.header = 'oj-datagrid-header';
  this.styles.endheader = 'oj-datagrid-end-header';
  this.styles.groupingcontainer = 'oj-datagrid-header-grouping';
  this.styles.headercell = 'oj-datagrid-header-cell';
  this.styles.rowheader = 'oj-datagrid-row-header';
  this.styles.colheader = 'oj-datagrid-column-header';
  this.styles.colheadercell = 'oj-datagrid-column-header-cell';
  this.styles.rowheadercell = 'oj-datagrid-row-header-cell';
  this.styles.endheadercell = 'oj-datagrid-end-header-cell';
  this.styles.rowendheader = 'oj-datagrid-row-end-header';
  this.styles.colendheader = 'oj-datagrid-column-end-header';
  this.styles.colendheadercell = 'oj-datagrid-column-end-header-cell';
  this.styles.rowendheadercell = 'oj-datagrid-row-end-header-cell';
  this.styles.headerlabel = 'oj-datagrid-header-label';
  this.styles.rowheaderlabel = 'oj-datagrid-row-header-label';
  this.styles.columnheaderlabel = 'oj-datagrid-column-header-label';
  this.styles.rowendheaderlabel = 'oj-datagrid-row-end-header-label';
  this.styles.columnendheaderlabel = 'oj-datagrid-column-end-header-label';
  this.styles['scroller-mobile'] = 'oj-datagrid-scroller-touch';
  this.styles.scroller = 'oj-datagrid-scroller';
  this.styles.scrollers = 'oj-datagrid-scrollers';
  this.styles.scrollbarForce = 'oj-scrollbar-force';
  this.styles.focus = 'oj-focus';
  this.styles.hover = 'oj-hover';
  this.styles.active = 'oj-active';
  this.styles.selected = 'oj-selected';
  this.styles.topSelected = 'oj-datagrid-selected-top';
  this.styles.bottomSelected = 'oj-datagrid-selected-bottom';
  this.styles.startSelected = 'oj-datagrid-selected-start';
  this.styles.endSelected = 'oj-datagrid-selected-end';
  this.styles.topEdit = 'oj-datagrid-cell-edit-top';
  this.styles.bottomEdit = 'oj-datagrid-cell-edit-bottom';
  this.styles.startEdit = 'oj-datagrid-cell-edit-start';
  this.styles.endEdit = 'oj-datagrid-cell-edit-end';
  this.styles.topFloodfill = 'oj-datagrid-floodfill-top';
  this.styles.bottomFloodfill = 'oj-datagrid-floodfill-bottom';
  this.styles.startFloodfill = 'oj-datagrid-floodfill-start';
  this.styles.endFloodfill = 'oj-datagrid-floodfill-end';
  this.styles.topResized = 'oj-datagrid-resized-top';
  this.styles.bottomResized = 'oj-datagrid-resized-bottom';
  this.styles.endResized = 'oj-datagrid-resized-end';
  this.styles.startResized = 'oj-datagrid-resized-start';
  this.styles.disabled = 'oj-disabled';
  this.styles.enabled = 'oj-enabled';
  this.styles.default = 'oj-default';
  this.styles.sortIcon = 'oj-datagrid-sort-icon';
  this.styles.sortascending = 'oj-datagrid-sort-ascending-icon';
  this.styles.sortdescending = 'oj-datagrid-sort-descending-icon';
  this.styles.sortdefault = 'oj-datagrid-sort-default-icon';
  this.styles.icon = 'oj-component-icon';
  this.styles.clickableicon = 'oj-clickable-icon-nocontext';
  this.styles.info = 'oj-helper-hidden-accessible';
  this.styles.rowexpander = 'oj-rowexpander';
  this.styles.cut = 'oj-datagrid-cut';
  this.styles.selectaffordance = 'oj-datagrid-touch-selection-affordance';
  this.styles.selectaffordancetopcornerbounded = 'oj-datagrid-touch-selection-affordance-top-corner-bounded';
  this.styles.selectaffordancebottomcornerbounded = 'oj-datagrid-touch-selection-affordance-bottom-corner-bounded';
  this.styles.selectaffordancetoprow = 'oj-datagrid-touch-selection-affordance-top-row';
  this.styles.selectaffordancebottomrow = 'oj-datagrid-touch-selection-affordance-bottom-row';
  this.styles.selectaffordancetopcolumn = 'oj-datagrid-touch-selection-affordance-top-column';
  this.styles.selectaffordancebottomcolumn = 'oj-datagrid-touch-selection-affordance-bottom-column';
  this.styles.floodfillaffordance = 'oj-datagrid-floodfill-affordance';
  this.styles.toucharea = 'oj-datagrid-touch-area';
  this.styles.readOnly = 'oj-read-only';
  this.styles.editable = 'oj-datagrid-editable';
  this.styles.cellEdit = 'oj-datagrid-cell-edit';
  this.styles.draggable = 'oj-draggable';
  this.styles.drag = 'oj-drag';
  this.styles.drop = 'oj-drop';
  this.styles.activedrop = 'oj-active-drop';
  this.styles.validdrop = 'oj-valid-drop';
  this.styles.invaliddrop = 'oj-invalid-drop';
  this.styles.formcontrol = 'oj-form-control-inherit';
  this.styles.borderHorizontalNone = 'oj-datagrid-border-horizontal-none';
  this.styles.borderVerticalNone = 'oj-datagrid-border-vertical-none';
  this.styles.borderHorizontalSmall = 'oj-datagrid-small-content-border-horizontal';
  this.styles.borderVerticalSmall = 'oj-datagrid-small-content-border-vertical';
  this.styles.offsetOutline = 'oj-datagrid-focus-offset';
  this.styles.depth = 'oj-datagrid-depth-';
  this.styles.popupHeader = 'oj-datagrid-popup-header';
  this.styles.popupContent = 'oj-datagrid-popup-content';
  this.styles.popupFooter = 'oj-datagrid-popup-footer';
  this.styles.dialogTitle = 'oj-dialog-title';
  this.styles.resizeDialog = 'oj-datagrid-resize-dialog';
  this.styles.headerAllSelected = 'oj-selected';
  this.styles.headerPartialSelected = 'oj-partial-selected';
  this.styles.disclosureContainer = 'oj-datagrid-disclosure-icon-container';
  this.styles.disclosureIcon = 'oj-datagrid-disclosure-icon';
  this.styles.expanded = 'oj-datagrid-expanded-icon';
  this.styles.collapsed = 'oj-datagrid-collapsed-icon';
  this.styles.spacer = 'oj-datagrid-tree-spacer';
  this.styles.hierarchicalTree = 'oj-datagrid-hierarchical-tree';
  this.styles.hierarchicalGroup = 'oj-datagrid-hierarchical-group';
  this.styles.iconContainer = 'oj-datagrid-icon-container';

  this.commands = {};
  this.commands.sortCol = 'oj-datagrid-sortCol';
  this.commands.sortColAsc = 'oj-datagrid-sortColAsc';
  this.commands.sortColDsc = 'oj-datagrid-sortColDsc';
  this.commands.sortRow = 'oj-datagrid-sortRow';
  this.commands.sortRowAsc = 'oj-datagrid-sortRowAsc';
  this.commands.sortRowDsc = 'oj-datagrid-sortRowDsc';
  this.commands.resize = 'oj-datagrid-resize';
  this.commands.resizeWidth = 'oj-datagrid-resizeWidth';
  this.commands.resizeHeight = 'oj-datagrid-resizeHeight';
  this.commands.resizeFitToContent = 'oj-datagrid-resizeFitToContent';
  this.commands.cut = 'oj-datagrid-cut';
  this.commands.paste = 'oj-datagrid-paste';
  this.commands.cutCells = 'oj-datagrid-cutCells';
  this.commands.copyCells = 'oj-datagrid-copyCells';
  this.commands.pasteCells = 'oj-datagrid-pasteCells';
  this.commands.autoFill = 'oj-datagrid-fillCells';
  this.commands.discontiguousSelection = 'oj-datagrid-discontiguousSelection';

  this.attributes = {};
  this.attributes.busyContext = Context._OJ_CONTEXT_ATTRIBUTE; // 'data-oj-context'
  this.attributes.context = 'data-oj-cellContext';
  this.attributes.resizable = 'data-oj-resizable';
  this.attributes.sortable = 'data-oj-sortable';
  this.attributes.sortDir = 'data-oj-sortdir';
  this.attributes.expander = 'data-oj-expander';
  this.attributes.expanderIndex = 'data-oj-expander-index';
  this.attributes.container = _OJ_CONTAINER_ATTR;
  this.attributes.extent = 'data-oj-extent';
  this.attributes.start = 'data-oj-start';
  this.attributes.depth = 'data-oj-depth';
  this.attributes.level = 'data-oj-level';
  this.attributes.metadata = 'data-oj-metaData';
};

oj$1._registerLegacyNamespaceProp('DataGridResources', DataGridResources);

/**
 * Set whether the reading direction is right to left.
 * @param {boolean} rtl true if reading direction is right to left, false otherwise.
 * @export
 */
DataGridResources.prototype.setRTLMode = function (rtl) {
  this.rtlMode = rtl;
};

/**
 * Whether the reading direction is right to left.
 * @return {boolean} true if reading direction is right to left, false otherwise.
 * @export
 */
DataGridResources.prototype.isRTLMode = function () {
  return (this.rtlMode === 'rtl');
};

/**
 * Gets the translated text
 * @param {string} key the key to the translated text
 * @param {Array=} args optional arguments to format the translated text
 * @return {string|null} the translated text
 * @export
 */
DataGridResources.prototype.getTranslatedText = function (key, args) {
  return this.translationFunction(key, args);
};

/**
 * Gets the default option
 * @param {string} key the key to the default option
 * @return {Object|null} the value of the option
 * @export
 */
DataGridResources.prototype.getDefaultOption = function (key) {
  return this.defaultOptions[key];
};

/**
 * Gets the mapped style class
 * @param {string} key the key to the style class
 * @return {string|null} the style class
 * @export
 */
DataGridResources.prototype.getMappedStyle = function (key) {
  if (key != null) {
    return this.styles[key];
  }
  return null;
};

/**
 * Gets the mapped command class
 * @param {string} key the key to the command class
 * @return {string|null} the command class
 * @export
 */
DataGridResources.prototype.getMappedCommand = function (key) {
  if (key != null) {
    return this.commands[key];
  }
  return null;
};

/**
 * Gets the mapped attribute
 * @param {string} key the key to the attribute
 * @return {string|null} the attribute
 * @export
 */
DataGridResources.prototype.getMappedAttribute = function (key) {
  if (key != null) {
    return this.attributes[key];
  }
  return null;
};

/**
 * @constructor
 * @private
 */
const DataProviderDataGridDataSource = function (dataprovider) {
  this.dataprovider = dataprovider;

  this.pendingHeaderCallback = {};
  if (dataprovider.options && dataprovider.options.implicitSort) {
    this.currentSortCriteria = dataprovider.options.implicitSort;
  }
  this.sortUpdated = false;

  var fetchByOffset = this.dataprovider.getCapability('fetchByOffset');

  if (fetchByOffset != null) {
    this.fetchByOffset = fetchByOffset.implementation === 'randomAccess';
  } else {
    this.fetchByOffset = false;
  }
  this._registerEventListeners();

  DataProviderDataGridDataSource.superclass.constructor.call(this);
};

// TODO: change oj.DataGridDataSource to DataGridDataSource???
oj$1.Object.createSubclass(DataProviderDataGridDataSource, oj$1.DataGridDataSource, 'DataProviderDataGridDataSource');

DataProviderDataGridDataSource.prototype._registerEventListeners = function () {
  this._mutationListener = this._handlDataProviderMutationEvent.bind(this);
  this._refreshListener = this._handlDataProviderRefreshEvent.bind(this);

  this.dataprovider.addEventListener('mutate', this._mutationListener);
  this.dataprovider.addEventListener('refresh', this._refreshListener);
};

DataProviderDataGridDataSource.prototype._handlDataProviderMutationEvent = function (event) {
  var eventDetail = event.detail;
  var adds = eventDetail.add;
  var i;
  if (adds != null) {
    var addEvent = {
      indexes: [],
      keys: []
    };
    var dataArray = [];
    var metadataArray = [];
    var indexesArray = [];
    for (i = 0; i < adds.data.length; i++) {
      addEvent.source = this;
      addEvent.operation = 'insert';
      addEvent.keys.push({ row: adds.metadata[i].key, column: null });
      addEvent.indexes.push({ row: adds.indexes[i], column: -1 });

      indexesArray.push(adds.indexes[i]);
      dataArray.push(adds.data[i]);
      metadataArray.push(adds.metadata[i]);

      if (i === adds.data.length - 1 || adds.indexes[i + 1] !== adds.indexes[i] + 1) {
        addEvent.result = new SingleCellSet(indexesArray, metadataArray,
          dataArray, this.columns);
        this.handleEvent('change', addEvent);
        addEvent = {};
        addEvent.indexes = [];
        addEvent.keys = [];
        dataArray = [];
        metadataArray = [];
        indexesArray = [];
      }
    }
  }

  var removes = event.detail.remove;
  if (removes != null) {
    var removeEvent = {
      source: this,
      operation: 'delete',
      keys: [],
      indexes: []
    };
    for (i = 0; i < removes.data.length; i++) {
      removeEvent.keys.push({ row: removes.metadata[i].key, column: null });
      removeEvent.indexes.push({ row: removes.indexes[i], column: -1 });
    }
    this.handleEvent('change', removeEvent);
  }

  var updates = event.detail.update;
  if (updates != null) {
    for (i = 0; i < updates.data.length; i++) {
      var updateEvent = {
        source: this,
        operation: 'update',
        keys: { row: updates.metadata[i].key, column: null },
        indexes: { row: updates.indexes[i], column: -1 }
      };
      updateEvent.result = new SingleCellSet([updates.indexes[i]], [updates.metadata[i]],
        [updates.data[i]], this.columns);
      this.handleEvent('change', updateEvent);
    }
  }
};

function SingleCellSet(indexes, metadata, data, columns) {
  this.indexes = indexes;
  this.data = data;
  this.metadata = metadata;
  this.columns = columns;
}

SingleCellSet.prototype.getData = function (indexes) {
  var self = this;
  var returnObj = {};
  Object.defineProperty(returnObj, 'data', {
    enumerable: true,
    get: function () {
      return self.data[indexes.row - self.getStart('row')][self.columns[indexes.column]];
    },
    set: function (newValue) {
      self.data[indexes.row - self.getStart('row')][self.columns[indexes.column]] = newValue;
    }
  });
  return returnObj;
};

SingleCellSet.prototype.getMetadata = function (indexes) {
  var self = this;
  var metadata = self.metadata[indexes.row - self.getStart('row')];
  metadata.keys = { row: metadata.key, column: self.columns[indexes.column] };
  return metadata;
};

SingleCellSet.prototype.getStart = function (axis) {
  if (axis === 'row') {
    return this.indexes[0];
  }
  return 0;
};

SingleCellSet.prototype.getCount = function (axis) {
  if (axis === 'row') {
    return this.data.length;
  } else if (axis === 'column') {
    return this.columns.length;
  }
  return 0;
};

SingleCellSet.prototype.getExtent = function () {
  return {
    row: { extent: 1, more: { before: false, after: false } },
    column: { extent: 1, more: { before: false, after: false } }
  };
};

DataProviderDataGridDataSource.prototype._handlDataProviderRefreshEvent = function () {
  this._asyncIterator = null;
  this.handleEvent('change', { operation: 'refresh' });
};

DataProviderDataGridDataSource.prototype.fetchHeaders = function (headerRange,
  callbacks, callbackObjects) {
  if (callbacks != null) {
    var axis = headerRange.axis;
    var callback = {
      headerRange: headerRange,
      callbacks: callbacks,
      callbackObjects: callbackObjects
    };
    this.pendingHeaderCallback[axis] = callback;
  }
};

DataProviderDataGridDataSource.prototype._processPendingHeaderCallbacks = function (axis) {
  // check if there's callback remaining for the axis
  var pendingCallback = this.pendingHeaderCallback[axis];
  if (pendingCallback != null) {
    // todo: check whether pending header range matches result
    var headerRange = pendingCallback.headerRange;
    var callbacks = pendingCallback.callbacks;
    var callbackObjects = pendingCallback.callbackObjects;
    this._handleHeaderFetchSuccess(headerRange, callbacks, callbackObjects);

    // clear any pending callback
    this.pendingHeaderCallback[axis] = null;
  }
};

DataProviderDataGridDataSource.prototype._handleHeaderFetchSuccess = function (
  headerRange, callbacks, callbackObjects) {
  var axis = headerRange.axis;
  var start = headerRange.start;
  var count = headerRange.count;
  var headerSet;

  if (axis === 'column' && this.columns != null) {
    var end = Math.min(this.columns.length, start + count);
    headerSet = new DataProviderHeaderSet(start, end, this.columns, this.currentSortCriteria);
  } else {
    headerSet = null;
  }

  if (callbacks != null && callbacks.success) {
    callbacks.success.call(callbackObjects.success, headerSet, headerRange, null);
  }
};

function DataProviderCellSet(response, cellRanges, columns) {
  this.response = response;
  this.results = response.results;
  this.cellRanges = cellRanges;
  this.columns = columns;

  this._setBounds(response, cellRanges, columns);
}

DataProviderCellSet.prototype._setBounds = function (response, cellRanges, columns) {
  for (var i = 0; i < cellRanges.length; i += 1) {
    var cellRange = cellRanges[i];
    if (cellRange.axis === 'row') {
      this.rowStart = response.fetchParameters.offset ?
        response.fetchParameters.offset : cellRange.start;
      this.rowEnd = (this.rowStart + response.results.length) - 1;
    } else if (cellRange.axis === 'column') {
      this.colStart = cellRange.start;
      this.colEnd = Math.min((this.colStart + cellRange.count) - 1,
                             (this.colStart + columns.length) - 1);
    }
  }
};

DataProviderCellSet.prototype.getData = function (indexes) {
  var self = this;
  var returnObj = {};
  Object.defineProperty(returnObj, 'data', {
    enumerable: true,
    get: function () {
      return self.results[indexes.row - self.rowStart].data[self.columns[indexes.column]];
    },
    set: function (newValue) {
      self.results[indexes.row - self.rowStart].data[self.columns[indexes.column]] = newValue;
    }
  });
  return returnObj;
};

DataProviderCellSet.prototype.getMetadata = function (indexes) {
  var self = this;
  var metadata = self.results[indexes.row - self.rowStart].metadata;
  metadata.keys = { row: this.results[indexes.row - this.rowStart].metadata.key,
    column: this.columns[indexes.column] };
  return metadata;
};


DataProviderCellSet.prototype.getCount = function (axis) {
  if (axis === 'row') {
    return (this.rowEnd - this.rowStart) + 1;
  } else if (axis === 'column') {
    return (this.colEnd - this.colStart) + 1;
  }
  return 0;
};

DataProviderCellSet.prototype.getExtent = function () {
  return {
    row: { extent: 1, more: { before: false, after: false } },
    column: { extent: 1, more: { before: false, after: false } }
  };
};

DataProviderDataGridDataSource.prototype.setUpColumns = function (response) {
  if (this.columns == null || this.columns.length === 0) {
    var columns = [];
    if (response.results.length) {
      var propertyNames = Object.keys(response.results[0].data);
      for (var i = 0; i < propertyNames.length; i++) {
        columns.push(propertyNames[i]);
      }
    }
    this.columns = columns;
  }
};

DataProviderDataGridDataSource.prototype.getRangeInfo = function (cellRanges) {
  var rangeInfo = {};
  for (var i = 0; i < cellRanges.length; i += 1) {
    var cellRange = cellRanges[i];
    if (cellRange.axis === 'row') {
      rangeInfo.rowStart = cellRange.start;
      rangeInfo.rowEnd = (rangeInfo.rowStart + cellRange.count) - 1;
    } else if (cellRange.axis === 'column') {
      rangeInfo.colStart = cellRange.start;
      rangeInfo.colEnd = (rangeInfo.colStart + cellRange.count) - 1;
    }
  }
  return rangeInfo;
};

DataProviderDataGridDataSource.prototype._createResults = function (response) {
  response.results = [];
  for (var i = 0; i < response.value.data.length; i++) {
    var item = {
      data: response.value.data[i],
      metadata: response.value.metadata[i]
    };
    response.results.push(item);
  }
};

DataProviderDataGridDataSource.prototype.handleFetchResult = function (cellRanges,
  callbacks, callbackObjects, response) {
  // normalize fetchByOffset vs fetchFirst

  var fetchResponse = response[0];

  if (!fetchResponse.results) {
    this._createResults(fetchResponse);
    fetchResponse.fetchParameters = fetchResponse.value.fetchParameters;
  }

  this.setUpColumns(fetchResponse);

  this._processPendingHeaderCallbacks('column');
  this._processPendingHeaderCallbacks('row');

  var cellSet = new DataProviderCellSet(fetchResponse, cellRanges, this.columns);

  if (callbacks != null && callbacks.success != null) {
    var success = callbackObjects ? callbackObjects.success : null;
    callbacks.success.call(success, cellSet, cellRanges);
  }
};

DataProviderDataGridDataSource.prototype.fetchCells = function (cellRanges,
  callbacks, callbackObjects) {
  var self = this;

  this.getCount('row');

  var rangeInfo = self.getRangeInfo(cellRanges);
  var size = (rangeInfo.rowEnd - rangeInfo.rowStart) + 1;

  if (self.fetchByOffset) {
    var offset = rangeInfo.rowStart;
    self._fetchPromise = self.dataprovider.fetchByOffset({
      size: size,
      offset: offset,
      sortCriteria: self.currentSortCriteria });
  } else {
    if (self._asyncIterator == null || this.sortUpdated) {
      // Create a clientId symbol that uniquely identify this consumer so that
      // DataProvider which supports it can optimize resources
      self._clientId = self._clientId || Symbol();

      self._asyncIterator = self.dataprovider.fetchFirst({
        clientId: self._clientId,
        size: size,
        sortCriteria: self.currentSortCriteria })[Symbol.asyncIterator]();
    }
    self._fetchPromise = self._asyncIterator.next();
  }

  this.sortUpdated = false;

  Promise.all([self._fetchPromise, self._getCountPromise]).then(
    self.handleFetchResult.bind(self, cellRanges, callbacks, callbackObjects));
};

DataProviderDataGridDataSource.prototype.getCapability = function (feature) {
  if (feature === 'sort') {
    var dpSort = this.dataprovider.getCapability(feature).attributes;
    if (dpSort === 'multiple' || dpSort === 'single') {
      return 'column';
    }
    return dpSort;
  }
  return 'none';
};

DataProviderDataGridDataSource.prototype.getCount = function (axis) {
  var self = this;
  if (axis === 'row') {
    if (self.totalRowCount === undefined) {
      self._getCountPromise = this.dataprovider.getTotalSize();
      self._getCountPromise.then(function (rowCount) {
        self.totalRowCount = rowCount;
      });
    } else {
      return self.totalRowCount;
    }
  } else if (axis === 'column') {
    if (this.columns != null) {
      return this.columns.length;
    }
  }
  return -1;
};

DataProviderDataGridDataSource.prototype.getCountPrecision = function () {
  return 'estimate';
};

DataProviderDataGridDataSource.prototype.indexes = function () {
  return { row: -1, column: -1 };
};

DataProviderDataGridDataSource.prototype.keys = function () {
  return { row: null, column: null };
};

DataProviderDataGridDataSource.prototype.move = function () {
  return false;
};

DataProviderDataGridDataSource.prototype.moveOK = function () {
  return 'invalid';
};

DataProviderDataGridDataSource.prototype.sort = function (criteria, callbacks, callbackObjects) {
  this.sortUpdated = true;
  this.currentSortCriteria = [{ attribute: criteria.key, direction: criteria.direction }];

  if (callbackObjects == null) {
    // eslint-disable-next-line no-param-reassign
    callbackObjects = {};
  }

  if (callbacks != null && callbacks.success != null) {
    callbacks.success.call(callbackObjects.success);
  }
};

function DataProviderHeaderSet(start, end, headers, sortCriteria) {
  this.start = start;
  this.end = end;
  this.headers = headers;
  this.sortCriteria = sortCriteria;
}

DataProviderHeaderSet.prototype.getData = function (index) {
  return this.headers[index];
};

DataProviderHeaderSet.prototype.getMetadata = function (index) {
  var data = this.getData(index);
  var returnObj = { key: data };

  if (this.sortCriteria != null) {
    for (var i = 0; i < this.sortCriteria.length; i++) {
      if (this.sortCriteria[i].attribute === data) {
        returnObj.sortDirection = this.sortCriteria[i].direction;
      }
    }
  }

  return returnObj;
};

DataProviderHeaderSet.prototype.getLevelCount = function () {
  return 1;
};

DataProviderHeaderSet.prototype.getExtent = function () {
  return { extent: 1, more: { before: false, after: false } };
};

DataProviderHeaderSet.prototype.getLabel = function () {
  return null;
};

DataProviderHeaderSet.prototype.getDepth = function () {
  return 1;
};

DataProviderHeaderSet.prototype.getCount = function () {
  return Math.max(0, this.end - this.start);
};

/**
 * This class contains all utility methods used by the Grid.
 * @param {Object} dataGrid the dataGrid using the utils
 * @constructor
 * @private
 */
const DvtDataGridKeyboardHandler = function (dataGrid) {
  this.grid = dataGrid;
};

/**
 * Get the action of a particular keydown event given information about the state of the frid
 * @param {Event} event
 * @param {Object} capabilities
 * @returns {string}
 */
DvtDataGridKeyboardHandler.prototype.getAction = function (event, capabilities) {
  var keyCode = event.keyCode;
  var ctrlKey = this.grid.m_utils.ctrlEquivalent(event);
  var shiftKey = event.shiftKey;
  var altKey = event.altKey;
  var keyCodes = this.grid.keyCodes;

  var cellOrHeader = capabilities.cellOrHeader;
  var readOnly = capabilities.readOnly;
  var currentMode = capabilities.currentMode;
  var activeMove = capabilities.activeMove;
  var rowMove = capabilities.rowMove;
  var columnSort = capabilities.columnSort;
  var selection = capabilities.selection;
  var selectionMode = capabilities.selectionMode;
  var multipleSelection = capabilities.multipleSelection;
  var expandCollapse = capabilities.expandCollapse;
  var copy = capabilities.copyCells;
  var cut = capabilities.cutCells;
  var paste = capabilities.pasteCells;
  var fill = capabilities.fill;

  switch (keyCode) {
    case keyCodes.TAB_KEY:
      if (currentMode === 'actionable') {
        if (shiftKey) {
          return 'TAB_PREV_IN_CELL';
        }
        return 'TAB_NEXT_IN_CELL';
      }
      if (!readOnly) {
        if (shiftKey) {
          return 'FOCUS_LEFT';
        }
        return 'FOCUS_RIGHT';
      }
      break;
    case keyCodes.ENTER_KEY:
      if (cellOrHeader === 'column' && columnSort) {
        return 'SORT';
      }
      if ((!altKey && readOnly && currentMode === 'navigation') ||
          cellOrHeader !== 'cell') {
        // enter actionable mode on headers since they cannot be edited
        return 'ACTIONABLE';
      }
      if (!readOnly && !altKey) {
        if (shiftKey) {
          return 'FOCUS_UP';
        }
        return 'FOCUS_DOWN';
      }
      if (altKey && readOnly && currentMode === 'navigation') {
        return 'EDITABLE';
      }
      if (!readOnly) {
        if (currentMode === 'navigation' || currentMode === 'edit') {
          return 'EDIT';
        }
      }
      break;
    case keyCodes.ESC_KEY:
      if (currentMode === 'actionable') {
        return 'EXIT_ACTIONABLE';
      }
      if (activeMove) {
        return 'CANCEL_REORDER';
      }
      if (!readOnly) {
        if (currentMode === 'navigation') {
          return 'EXIT_EDITABLE';
        }
        if (currentMode === 'edit') {
          return 'CANCEL_EDIT';
        }
      } else if (this.grid.m_discontiguousSelection) {
        return 'SELECT_DISCONTIGUOUS';
      }
      break;
    case keyCodes.SPACE_KEY:
      if (cellOrHeader.indexOf('row') !== -1 && selection &&
          ((selectionMode === 'cell' && multipleSelection) || selectionMode === 'row')) {
        return 'SELECT_ROW';
      }
      if (cellOrHeader.indexOf('column') !== -1 && selection &&
          selectionMode === 'cell' && multipleSelection) {
        return 'SELECT_COLUMN';
      }
      if (cellOrHeader === 'cell') {
        if (readOnly && currentMode === 'navigation') {
          if (ctrlKey && selection && selectionMode === 'cell' && multipleSelection) {
            return 'SELECT_COLUMN';
          }
          if (shiftKey && selection &&
              ((selectionMode === 'cell' && multipleSelection) ||
               selectionMode === 'row')) {
            return 'SELECT_ROW';
          }
        } else if (currentMode === 'navigation') {
          return 'DATA_ENTRY';
        }
      }
      break;
    case keyCodes.PAGEUP_KEY:
      if (currentMode !== 'edit') {
        return 'FOCUS_ROW_FIRST';
      }
      break;
    case keyCodes.PAGEDOWN_KEY:
      if (currentMode !== 'edit') {
        return 'FOCUS_ROW_LAST';
      }
      break;
    case keyCodes.END_KEY:
      if (currentMode !== 'edit') {
        return 'FOCUS_COLUMN_LAST';
      }
      break;
    case keyCodes.HOME_KEY:
      if (currentMode !== 'edit') {
        return 'FOCUS_COLUMN_FIRST';
      }
      break;
    case keyCodes.LEFT_KEY:
      if (ctrlKey && expandCollapse && this.grid._isHeaderExpanded(event.target)) {
        return 'COLLAPSE';
      }
      if (currentMode === 'actionable') {
        return 'NO_OP';
      }
      if (currentMode !== 'edit') {
        if (shiftKey && selection && selectionMode === 'cell' && multipleSelection) {
          return 'SELECT_EXTEND_LEFT';
        }
        if (ctrlKey) {
          return 'FOCUS_ROW_HEADER';
        }
        return 'FOCUS_LEFT';
      }
      break;
    case keyCodes.UP_KEY:
      if (currentMode === 'actionable') {
        return 'NO_OP';
      }
      if (currentMode !== 'edit') {
        if (shiftKey && selection && multipleSelection) {
          return 'SELECT_EXTEND_UP';
        }
        if (ctrlKey) {
          return 'FOCUS_COLUMN_HEADER';
        }
        return 'FOCUS_UP';
      }
      break;
    case keyCodes.RIGHT_KEY:
      if (ctrlKey && expandCollapse && this.grid._isHeaderCollapsed(event.target)) {
        return 'EXPAND';
      }
      if (currentMode === 'actionable') {
        return 'NO_OP';
      }
      if (currentMode !== 'edit') {
        if (shiftKey && selection && selectionMode === 'cell' && multipleSelection) {
          return 'SELECT_EXTEND_RIGHT';
        }
        if (ctrlKey) {
          return 'FOCUS_ROW_END_HEADER';
        }
        return 'FOCUS_RIGHT';
      }
      break;
    case keyCodes.DOWN_KEY:
      if (currentMode === 'actionable') {
        return 'NO_OP';
      }
      if (currentMode !== 'edit') {
        if (shiftKey && selection && multipleSelection) {
          return 'SELECT_EXTEND_DOWN';
        }
        if (ctrlKey) {
          return 'FOCUS_COLUMN_END_HEADER';
        }
        return 'FOCUS_DOWN';
      }
      break;
    case keyCodes.F2_KEY:
      if (cellOrHeader !== 'cell') {
        return 'ACTIONABLE';
      }
      if (readOnly && currentMode === 'navigation') {
        return 'EDITABLE';
      }
      if (!readOnly && currentMode === 'navigation') {
        return 'EDIT';
      }
      break;
    case keyCodes.F8_KEY:
      if (shiftKey && selection && multipleSelection) {
        return 'SELECT_DISCONTIGUOUS';
      }
      break;
    case keyCodes.F10_KEY:
      if (shiftKey) {
        return 'NO_OP';
      }
      break;
    case keyCodes.V_KEY:
      if (currentMode === 'navigation' && ctrlKey) {
        if (rowMove) {
          return 'PASTE';
        }
        if (paste) {
          return 'PASTE_CELLS';
        }
      }
      if (!readOnly && currentMode === 'navigation') {
        return 'DATA_ENTRY';
      }
      break;
    case keyCodes.X_KEY:
      if (currentMode === 'navigation' && ctrlKey) {
        if (rowMove) {
          return 'CUT';
        }
        if (cut) {
          return 'CUT_CELLS';
        }
      }
      if (!readOnly && currentMode === 'navigation') {
        return 'DATA_ENTRY';
      }
      break;
    case keyCodes.C_KEY:
      if (currentMode === 'navigation' && ctrlKey && copy) {
        return 'COPY_CELLS';
      }
      if (!readOnly && currentMode === 'navigation') {
        return 'DATA_ENTRY';
      }
      break;
    case keyCodes.D_KEY:
      if (currentMode === 'navigation' && ctrlKey && fill) {
        return 'FILL';
      }
      break;
    case keyCodes.R_KEY:
      if (currentMode === 'navigation' && ctrlKey && fill) {
        return 'FILL';
      }
    break;
    case keyCodes.SHIFT_KEY:
    case keyCodes.CTRL_KEY:
    case keyCodes.ALT_KEY:
      break;

    case keyCodes.A_KEY:
      if (ctrlKey && selection && multipleSelection) {
        return 'SELECT_ALL';
      }
      // eslint-disable-next-line no-fallthrough
    case keyCodes.NUM5_KEY:
      if (ctrlKey && altKey) {
        return 'READ_CELL';
      }
      // eslint-disable-next-line no-fallthrough
    default:
      if ((keyCode < keyCodes.F1_KEY || keyCode > keyCodes.F15_KEY) &&
          !readOnly && currentMode === 'navigation' && cellOrHeader === 'cell' &&
          !ctrlKey) {
        return 'DATA_ENTRY';
      }
      break;
  }
  return 'NO_OP';
};

/**
 * The DvtDataGridOptions object provides convenient methods to access the options passed to the Grid.
 * @param {Object=} options - options set on the grid.
 * @param {Function=} rendererWrapperFunction - callback function used to fix renderer function for custom element.
 * @constructor
 * @private
 */
const DvtDataGridOptions = function (options, rendererWrapperFunction) {
  this.options = options;
  this.rendererWrapperFunction = rendererWrapperFunction;
};

/**
 * Helper method to extract properties
 * @param {string=} arg1 - key to extract in object
 * @param {string=} arg2 - key to extract in the value of object[arg1]
 * @param {string=} arg3 - key to extract in the value of object[arg1][arg2]
 * @param {string=} arg4 - key to extract in the value of object[arg1][arg2][arg3]
 * @return {string|number|Object|boolean|null}
 */
DvtDataGridOptions.prototype.extract = function (arg1, arg2, arg3, arg4) {
  if (arg1 != null) {
    var val1 = this.options[arg1];
    if (arg2 != null && val1 != null) {
      var val2 = val1[arg2];
      if (arg3 != null && val2 != null) {
        var val3 = val2[arg3];
        if (arg4 != null && val3 != null) {
          return val3[arg4];
        }
        return val3;
      }
      return val2;
    }
    return val1;
  }
  return null;
};

/**
 * Evaluate the a function, if it is a constant return it
 * @param {string|number|Object|boolean|null} value - function or string of the raw property
 * @param {Object} obj - object to pass into value if it is a function
 * @return {string|number|Object|boolean|null} the evaluated value(obj) if value a function, else return value
 */
DvtDataGridOptions.prototype.evaluate = function (value, obj) {
  if (typeof value === 'function') {
    return value.call(this, obj);
  }
  return value;
};

/**
 * Get the property that was set on the option
 * @param {string} prop - the property to get from options
 * @param {string|undefined} axis - subobject to get row/column/cell/rowEnd/columnEnd
 * @return {string|number|Object|boolean|null} the raw value of the property
 */
DvtDataGridOptions.prototype.getRawProperty = function (prop, axis, label) {
  var arg1;
  var arg2;
  var arg3;
  var arg4;

  if (axis === 'row' || axis === 'column' || axis === 'rowEnd' || axis === 'columnEnd') {
    arg1 = 'header';
    arg2 = axis;
    if (label) {
      arg3 = 'label';
      arg4 = prop;
    } else {
      arg3 = prop;
    }
  } else if (axis === 'cell') {
    arg1 = 'cell';
    arg2 = prop;
  }

  return this.extract(arg1, arg2, arg3, arg4);
};

/**
 * Get the evaluated property of the options
 * @param {string} prop - the property to get from options
 * @param {string=} axis - subobject to get row/column/cell
 * @param {Object=} obj - object to pass into property if it is a function
 * @return the evaluated property
 */
DvtDataGridOptions.prototype.getProperty = function (prop, axis, obj, label) {
  if (obj === undefined) {
    return this.extract(prop, axis, label);
  }

  return this.evaluate(this.getRawProperty(prop, axis, label), obj);
};

// //////////////////////// Grid options /////////////////////////////////

/**
 * Get the row banding interval from the grid options
 * @return {string|number|Object|boolean} the row banding interval
 */
DvtDataGridOptions.prototype.getRowBandingInterval = function () {
  var bandingInterval = this.getProperty('bandingInterval', 'row');
  if (bandingInterval != null) {
    return bandingInterval;
  }
  return 0;
};

/**
 * Get the column banding interval from the grid options
 * @return {string|number|Object|boolean} the column banding interval
 */
DvtDataGridOptions.prototype.getColumnBandingInterval = function () {
  var bandingInterval = this.getProperty('bandingInterval', 'column');
  if (bandingInterval != null) {
    return bandingInterval;
  }
  return 0;
};

/**
 * Get the empty text from the grid options
 * @return {string|number|Object|boolean|null} the empty text
 */
DvtDataGridOptions.prototype.getEmptyText = function () {
  return this.getProperty('emptyText');
};

/**
 * Get the horizontal gridlines from the grid options
 * @return {string|number|Object|boolean|null} horizontal gridlines visible/hidden
 */
DvtDataGridOptions.prototype.getHorizontalGridlines = function () {
  var horizontalGridlines = this.extract('gridlines', 'horizontal');
  if (horizontalGridlines != null) {
    return horizontalGridlines;
  }
  return 'visible';
};

/**
 * Get the vertical gridlines from the grid options
 * @return {string|number|Object|boolean|null} vertical gridlines visible/hidden
 */
DvtDataGridOptions.prototype.getVerticalGridlines = function () {
  var verticalGridlines = this.extract('gridlines', 'vertical');
  if (verticalGridlines != null) {
    return verticalGridlines;
  }
  return 'visible';
};

/**
 * Get the scroll to key from the grid options
 * @return {string|null} scroll to key behavior capability/never/auto/always
 */
DvtDataGridOptions.prototype.getScrollToKey = function () {
  var scrollToKey = this.getProperty('scrollToKey');
  if (scrollToKey != null) {
    return scrollToKey;
  }
  return 'auto';
};

/**
 * Get the scroll position from the grid options
 * @return {string|number|Object|boolean|null} scrollPositionObject
 */
DvtDataGridOptions.prototype.getScrollPosition = function () {
  var scrollPosition = this.getProperty('scrollPosition');
  if (scrollPosition != null) {
    return scrollPosition;
  }
  return null;
};

/**
 * Get the selection mode cardinality (none, single multiple)
 * @return {string|number|Object|boolean|null} none/single/multiple
 */
DvtDataGridOptions.prototype.getSelectionCardinality = function () {
  var mode = this.getProperty('selectionMode');
  if (mode == null) {
    return 'none';
  }

  var key = this.getSelectionMode();
  var val = mode[key];
  if (val != null) {
    return val;
  }
  return 'none';
};

/**
 * Get the selection mode (row,cell)
 * @return {string|number|Object|boolean|null} row/cell
 */
DvtDataGridOptions.prototype.getSelectionMode = function () {
  var mode = this.getProperty('selectionMode');
  if (mode == null) {
    return 'cell';
  }

  var cardinality = mode.row;
  if (cardinality != null && cardinality !== 'none') {
    return 'row';
  }
  return 'cell';
};

/**
 * Get the current selection from the grid options
 * @return {Array|null} the current selection from options
 */
DvtDataGridOptions.prototype.getSelection = function () {
  return this.getProperty('selection');
};

/**
 * Get the current cell from the grid options
 * @return {Object|null} the current cell from options
 */
DvtDataGridOptions.prototype.getCurrentCell = function () {
  return this.getProperty('currentCell');
};

/**
 * Get the editMode (none,cell)
 * @return {string|number|Object|boolean|null} default/enter
 */
DvtDataGridOptions.prototype.getEditMode = function () {
  return this.getProperty('editMode');
};

// //////////////////////// Grid header/cell options /////////////////////////////////
/**
 * Is the given header sortable
 * @param {string} axis - axis to check if sort enabled
 * @param {Object} obj - header context
 * @return {string|number|Object|boolean|null} default or null
 */
DvtDataGridOptions.prototype.isSortable = function (axis, obj) {
  return this.getProperty('sortable', axis, obj);
};

/**
 * Is the given header resizable
 * @param {string} axis - axis to check if resizing enabled
 * @param {string} dir - width/height
 * @return {string|number|Object|boolean|null} enable, disable, or null
 */
DvtDataGridOptions.prototype.isResizable = function (axis, dir, obj) {
  var v = this.extract('header', axis, 'resizable', dir);
  if (obj != null) {
    return this.evaluate(v, obj);
  }
  return v;
};

/**
 * Gets the dnd rorderable option
 * @param {string} axis the axis to get the reorder property from
 * @return {string|number|Object|boolean|null} enable, disable, or null
 */
DvtDataGridOptions.prototype.isMoveable = function (axis) {
  return this.extract('dnd', 'reorder', axis);
};

/**
 * Gets the floodfill option
 * @return {string|null} enable, disable, or null
 */
 DvtDataGridOptions.prototype.isFloodFillEnabled = function () {
  let isEnabled = false;
  let floodFill = this.extract('dataTransferOptions', 'fill');
  if (floodFill === 'enable') {
    isEnabled = true;
  }
  return isEnabled;
};

/**
 * Gets the copy option
 * @return {string|null} enable, disable, or null
 */
 DvtDataGridOptions.prototype.isCopyEnabled = function () {
  let isEnabled = false;
  let copy = this.extract('dataTransferOptions', 'copy');
  if (copy === 'enable') {
    isEnabled = true;
  }

  return isEnabled;
};

/**
 * Gets the cut option
 * @return {string|null} enable, disable, or null
 */
 DvtDataGridOptions.prototype.isCutEnabled = function () {
  let isEnabled = false;
  let cut = this.extract('dataTransferOptions', 'cut');
  if (cut === 'enable') {
    isEnabled = true;
  }

  return isEnabled;
};

/**
 * Gets the paste option
 * @return {string|null} enable, disable, or null
 */
 DvtDataGridOptions.prototype.isPasteEnabled = function () {
  let isEnabled = false;
  let paste = this.extract('dataTransferOptions', 'paste');
  if (paste === 'enable') {
    isEnabled = true;
  }

  return isEnabled;
};

/**
 * Get the inline style property on an object
 * @param {string} axis - axis to get style of
 * @param {Object} obj - context
 * @return {string|number|Object|boolean|null} inline style
 */
DvtDataGridOptions.prototype.getInlineStyle = function (axis, obj, label) {
  return this.getProperty('style', axis, obj, label);
};

/**
 * Get the style class name property on an object
 * @param {string} axis - axis to get class name of
 * @param {Object} obj - context
 * @return {string|number|Object|boolean|null} class name
 */
DvtDataGridOptions.prototype.getStyleClass = function (axis, obj, label) {
  return this.getProperty('className', axis, obj, label);
};

/**
 * Get the renderer of an axis
 * @param {string} axis - axis to get style of
 * @param {boolean} label - whether its a label or not.
 * @return {string|number|Object|boolean|null} axis renderer
 */
DvtDataGridOptions.prototype.getRenderer = function (axis, label) {
  // return type expected to be function, so just return without further
  // evaluation
  if (this.rendererWrapperFunction) {
    return this.rendererWrapperFunction(this.getRawProperty('renderer', axis, label));
  }
  return this.getRawProperty('renderer', axis, label);
};

/**
 * Get the scroll mode
 * @return {string} the scroll mode, which can be either "scroll", "loadMoreOnScroll", "auto".
 */
DvtDataGridOptions.prototype.getScrollPolicy = function () {
  var mode = this.getProperty('scrollPolicy');
  if (mode == null) {
    mode = 'auto';
  }

  return mode;
};


/**
 * Get the scroll policy options
 */
DvtDataGridOptions.prototype.getScrollPolicyOptions = function () {
  return this.getProperty('scrollPolicyOptions');
};

/**
 * Class used to keep track of whcih elements have been resized, has an object
 * containing two objects 'row' and 'column', which should have objects of
 * index:{size}. this.sizes = {axis:{index:{size}}}
 * @constructor
 * @private
 */
const DvtDataGridSizingManager = function () {
  this.sizes = { column: new Map(), row: new Map() };
};

/**
 * Set a size in the sizes object in the sizing manager
 * @param {string} axis - column/row
 * @param {any} headerKey - key of the element
 * @param {number} size - the size to put in the object
 */
DvtDataGridSizingManager.prototype.setSize = function (axis, headerKey, size) {
  this.sizes[axis].set(headerKey, size);
};

/**
 * Get a size from the sizing manager for a given axis and index,
 * @param {string} axis - column/row
 * @param {any} headerKey - key of the element
 * @return {number|null} a size if it exists
 */
DvtDataGridSizingManager.prototype.getSize = function (axis, headerKey) {
  // get does reference comparison
  var size = this.sizes[axis].get(headerKey);
  if (size != null) {
    return size;
  }

  // if reference comparison does not work we should check using compare values
  this.sizes[axis].forEach(function (value, key) {
    if (size == null && oj$1.Object.compareValues(key, headerKey)) {
      size = value;
    }
  });

  return size;
};

/**
 * Empty the sizing manager sizes
 */
DvtDataGridSizingManager.prototype.clear = function () {
  this.sizes.column.clear();
  this.sizes.row.clear();
};

/**
 * This class contains all utility methods used by the Grid.
 * @param {Object} dataGrid the dataGrid using the utils
 * @constructor
 * @private
 */
const DvtDataGridUtils = function (dataGrid) {
  this.scrollbarSize = -1;
  this.dataGrid = dataGrid;

  let agentInfo = oj.AgentUtils.getAgentInfo();
  this.os = agentInfo.os;
  this.browser = agentInfo.browser;
  this.engine = agentInfo.engine;
};

/**
 * Get the maximum scrollable browser height
 * @returns {Number}
 */
DvtDataGridUtils.prototype._getMaxDivHeightForScrolling = function () {
  if (this.m_maxDivHeightForScrolling == null) {
    this._setMaxValuesForScrolling();
  }
  return this.m_maxDivHeightForScrolling;
};

/**
 * Get the maximum scrollable browser width
 * @returns {Number}
 */
DvtDataGridUtils.prototype._getMaxDivWidthForScrolling = function () {
  if (this.m_maxDivWidthForScrolling == null) {
    this._setMaxValuesForScrolling();
  }
  return this.m_maxDivWidthForScrolling;
};
  /**
   * Set the maximum scrollable browser height
   */
   DvtDataGridUtils.prototype._setMaxValuesForScrolling = function () {
    this._calculateBrowserDefinedValues();
  };

  DvtDataGridUtils.prototype._calculateBrowserDefinedValues = function () {
    var div1 = document.createElement('div');
    div1.style.width = '1000000000px';
    div1.style.height = '1000000000px';
    div1.style.display = 'none';

    var scrollDiv = document.createElement('div');
    scrollDiv.style.width = '100px';
    scrollDiv.style.height = '100px';
    scrollDiv.style.overflow = 'scroll';
    scrollDiv.style.position = 'absolute';
    scrollDiv.style.top = '-9999px';

    document.body.appendChild(scrollDiv); // @HTMLUpdateOK

    let remove = false;
    // ie lets the value go forever without actual support, so we hard cap it at 1 million pixels
    if (this.browser === oj.AgentUtils.BROWSER.IE || this.browser === oj.AgentUtils.BROWSER.EDGE) {
      this.m_maxDivHeightForScrolling = 1000000;
      this.m_maxDivWidthForScrolling = 1000000;
    } else {
      remove = true;
      document.body.appendChild(div1); // @HTMLUpdateOK
      // for some reason chrome stops rendering absolutely positioned content at half the value on osx
      this.m_maxDivHeightForScrolling =
        parseInt(parseFloat(window.getComputedStyle(div1).height) / 2, 10);
      this.m_maxDivWidthForScrolling =
        parseInt(parseFloat(window.getComputedStyle(div1).width) / 2, 10);
    }

    // Get the scrollbar width/height
    this.scrollbarSize = scrollDiv.offsetWidth - scrollDiv.clientWidth;

    if (remove) {
      document.body.removeChild(div1);
    }
    document.body.removeChild(scrollDiv);
  };

  /**
   * Gets the size of the native scrollbar
   */
  DvtDataGridUtils.prototype.getScrollbarSize = function () {
    if (this.scrollbarSize === -1) {
      this._calculateBrowserDefinedValues();
    }
    return this.scrollbarSize;
  };

/**
 * Gets the size of the native scrollbar
 */
DvtDataGridUtils.prototype.getScrollbarSize = function () {
  if (this.scrollbarSize === -1) {
    this.calculateScrollbarSize();
  }
  return this.scrollbarSize;
};

/**
 * Determine if the current agent is touch device
 */
DvtDataGridUtils.prototype.isTouchDevice = function () {
  if (this.isTouch == null) {
    this.isTouch = isMobileTouchDevice();
  }

  return this.isTouch;
};

/**
 * Adds a CSS className to the dom node if it doesn't already exist in the classNames list,
 * or does nothing if it already exists.
 * @param {Element|Node|null|undefined} domElement DOM Element to add style class name to
 * @param {string|null} className Name of style class to add
 * @return {boolean|null} <code>true</code> if the style class was added
 */

DvtDataGridUtils.prototype.addCSSClassName = function (domElement, className) {
  if (className != null && className !== '' && domElement != null && domElement.classList != null) {
    domElement.classList.add(className);
  }
};

/**
 * Removes a CSS className from the dom node if it exists in the classNames list.
 * @param {Element|Node|null|undefined} domElement DOM Element to remove style class name from
 * @param {string|null} className Name of style class to remove
 * @return {boolean|null} <code>true</code> if the style class was removed
 */

DvtDataGridUtils.prototype.removeCSSClassName = function (domElement, className) {
  if (className != null && className !== '' && domElement != null && domElement.classList != null) {
    domElement.classList.remove(className);
  }
};

/**
 * @param {Element|Node|null|undefined} domElement DOM Element to check for the style <code>className</code>
 * @param {string} className the CSS className to check for
 * @return {boolean} <code>true</code> if the className is in the element's list of classes
 */

DvtDataGridUtils.prototype.containsCSSClassName = function (domElement, className) {
  var exists = false;
  if (className != null && domElement != null && domElement.classList != null) {
    exists = domElement.classList.contains(className);
  }
  return exists;
};

/**
 * Copied from AdfDhtmlCommonUtils used in Faces
 * Returns the index at which <code>className</code> appears within <code>currentClassName</code>
 * with either a space or the beginning or end of <code>currentClassName</code> on either side.
 * This function optimizes the runtime speed by not creating objects in most cases and assuming
 * 1) It is OK to only check for spaces as whitespace characters
 * 2) It is rare for the searched className to be a substring of another className, therefore
 *    if we get a hit on indexOf, we have almost certainly found our className
 * 3) It is even rarer for the searched className to be a substring of more than one className,
 *    therefore, repeating the search from the end of the string should almost always either return
 *    our className or the original search hit from indexOf
 * @param {string} currentClassName Space-separated class name string to search
 * @param {string} className string to search for within currentClassName
 * @return {number} index of className in currentClassName, or -1 if it doesn't exist
 */
DvtDataGridUtils._getCSSClassNameIndex = function (currentClassName, className) {
  // if no current class
  // SVG element className property is not of type string
  if (!currentClassName || !currentClassName.indexOf) {
    return -1;
  }

  // if the strings are equivalent, then the start index is the beginning of the string
  if (className === currentClassName) {
    return 0;
  }

  var classNameLength = className.length;
  var currentClassNameLength = currentClassName.length;

  // check if our substring exists in the string at all
  var nameIndex = currentClassName.indexOf(className);

  // if our substring exists then our class exists if either:
  // 1) It is at the beginning of the classNames string and has a following space
  // 2) It is at the end of the classNames string and has a leading space
  // 3) It has a space on either side
  if (nameIndex >= 0) {
    var hasStart = (nameIndex === 0) || (currentClassName.charAt(nameIndex - 1) === ' ');
    var endIndex = nameIndex + classNameLength;
    var hasEnd = (endIndex === currentClassNameLength) ||
      (currentClassName.charAt(endIndex) === ' ');

    // one of the three condition above has been met so our string is in the parent string
    if (hasStart && hasEnd) {
      return nameIndex;
    }

    // our substring exists in the parent string but didn't meet the above conditions,  Were
    // going to be lazy and retest, searchig for our substring from the back of the classNames
    // string
    var lastNameIndex = currentClassName.lastIndexOf(className);

    // if we got the same index as the search from the beginning then we aren't in here
    if (lastNameIndex !== nameIndex) {
      // recheck the three match cases
      hasStart = currentClassName.charAt(lastNameIndex - 1);
      endIndex = lastNameIndex + classNameLength;
      hasEnd = (endIndex === currentClassNameLength) ||
        (currentClassName.charAt(endIndex) === ' ');

      if (hasStart && hasEnd) {
        return lastNameIndex;
      }

      // this should only happen if the searched for className is a substring of more
      // than one className in the classNames list, so it is OK for this to be slow.  We
      // also know at this point that we definitely didn't get a match at either the very
      // beginning or very end of the classNames string, so we can safely append spaces
      // at either end
      return currentClassName.indexOf(' ' + className + ' ');
    }
  }

  // no match
  return -1;
};

/**
 * Returns either the ctrl key or the command key in Mac OS
 * @param {Event} event
 */
DvtDataGridUtils.prototype.ctrlEquivalent = function (event) {
  var isMac = (this.os === oj.AgentUtils.OS.MAC);
  return (isMac ? event.metaKey : event.ctrlKey);
};

/**
 * Gets the scroll left of an element.  This method abstracts the inconsistency of scrollLeft
 * behavior in RTL mode between browsers.
 * @param {Element} element the dom element to retrieve scroll left
 * @private
 */
DvtDataGridUtils.prototype.getElementScrollLeft = function (element) {
  return Math.abs(element.scrollLeft);
};

/**
 * Sets the scroll left of an element.  This method abstracts the inconsistency of scrollLeft
 * behavior in RTL mode between browsers.
 * @param {Element} element the dom element to set scroll left
 * @param {number} scrollLeft the scroll left of the dom element
 * @private
 */
DvtDataGridUtils.prototype.setElementScrollLeft = function (element, scrollLeft) {
  setScrollLeft(element, scrollLeft);
};


/**
 * Determines the what mousewheel event the browser recognizes
 * All modern browsers support wheel event
 * @return {string} The event which the browser uses to track mosuewheel events
 * @private
 */
DvtDataGridUtils.prototype.getMousewheelEvent = function () {
  return 'wheel';
};

/**
 * The standard wheel event and WheelEvent API now uses deltaMode and just deltaX and deltaY as the
 * properties for determining scroll.
 * @param {Event} event the mousewheel scroll event
 * @return {Object} change in X and Y if applicable through a mousewheel event, properties are deltaX, deltaY
 * @private
 */
DvtDataGridUtils.prototype.getMousewheelScrollDelta = function (event) {
  var scrollConstant = -1;
  var deltaMode = event.deltaMode;

  if (deltaMode === event.DOM_DELTA_PIXEL) {
    scrollConstant = -1;
  } else if (deltaMode === event.DOM_DELTA_LINE || deltaMode === event.DOM_DELTA_PAGE) {
    // only on firefox now, we will scroll 40 times the number of lines they
    // they want to scroll
    scrollConstant = -40;
  }
  var deltaX = event.deltaX * scrollConstant;
  var deltaY = event.deltaY * scrollConstant;

  return { deltaX: deltaX, deltaY: deltaY };
};

/**
 * Empty out (clear all children and attributes) from an element
 * @param {Element} elem the dom element to empty out
 * @private
 */
DvtDataGridUtils.prototype.empty = function (elem) {
  while (elem.firstChild) {
    this.dataGrid._remove(elem.firstChild);
  }
};

/**
 * Does the browser support transitions
 * @private
 */
DvtDataGridUtils.prototype.supportsTransitions = function () {
  var b = document.body || document.documentElement;
  var s = b.style;
  var p = 'transition';
  var ieBefore11 = /MSIE \d/.test(navigator.userAgent) &&
    (document.documentMode == null || document.documentMode < 11);

  if (ieBefore11) {
    return false;
  }

  if (typeof s[p] === 'string') {
    return true;
  }

  // Tests for vendor specific prop
  var v = ['Moz', 'webkit', 'Webkit', 'Khtml', 'O', 'ms'];
  p = p.charAt(0).toUpperCase() + p.substr(1);

  for (var i = 0; i < v.length; i++) {
    if (typeof s[v[i] + p] === 'string') {
      return true;
    }
  }

  return false;
};

/**
 * Return whether the node is editable or clickable
 * @param {Node|Element} node Node
 * @param {Element} databody Databody
 * @return {boolean} true or false
 * @private
 */
DvtDataGridUtils.prototype._isNodeEditableOrClickable = function (node, databody) {
  while (node != null && node !== databody) {
    var nodeName = node.nodeName;

    // If the node is a text node, move up the hierarchy to only operate on elements
    // (on at least the mobile browsers, the node may be a text node)
    if (node.nodeType === 3) { // 3 is Node.TEXT_NODE
      // eslint-disable-next-line no-param-reassign
      node = node.parentNode;
    } else {
      var tabIndex = parseInt(node.getAttribute('tabIndex'), 10);
      // datagrid overrides the tab index, so we should check if the data-oj-tabindex is populated
      var origTabIndex = parseInt(
        node.getAttribute(this.dataGrid.getResources().getMappedAttribute('tabindex')), 10);

      if (tabIndex != null && tabIndex >= 0) {
        if (this.containsCSSClassName(
              node, this.dataGrid.getResources().getMappedStyle('cell')) ||
            this.containsCSSClassName(
              node, this.dataGrid.getResources().getMappedStyle('headerlabel')) ||
            this.containsCSSClassName(
              node, this.dataGrid.getResources().getMappedStyle('headercell')) ||
            this.containsCSSClassName(
              node, this.dataGrid.getResources().getMappedStyle('endheadercell'))) {
          // this is the cell element
          return false;
        }

        return true;
      } else if (nodeName.match(/^INPUT|SELECT|OPTION|BUTTON|^A\b|TEXTAREA/)) {
        // ignore elements with tabIndex == -1
        if (tabIndex !== -1 || origTabIndex !== -1) {
          return true;
        }
      }

      // eslint-disable-next-line no-param-reassign
      node = node.parentNode;
    }
  }
  return false;
};

/**
 * On certain browser the outline is postioned differently and requires offset. Chrome/Safari on Mac.
 * @return {boolean} true if the outline needs to be offset
 * @private
 */
DvtDataGridUtils.prototype.shouldOffsetOutline = function () {
  if (this.os === oj.AgentUtils.OS.MAC && this.engine === oj.AgentUtils.ENGINE.WEBKIT) {
    return true;
  }
  return false;
};

/**
 * Creates a new DataGrid
 * @constructor
 * @private
 */
const DvtDataGrid = function (root) {
  this.m_root = root;

  this.MAX_COLUMN_THRESHOLD = 20;
  this.MAX_ROW_THRESHOLD = 30;

  this.m_utils = new DvtDataGridUtils(this);

  this.m_discontiguousSelection = false;

  this.m_sizingManager = new DvtDataGridSizingManager();

  this.m_keyboardHandler = new DvtDataGridKeyboardHandler(this);

  this.m_rowHeaderWidth = null;
  this.m_rowEndHeaderWidth = null;
  this.m_colHeaderHeight = null;
  this.m_colEndHeaderHeight = null;

  // a mapping of style class+inline style combo to a dimension
  // to reduce the need to do excessive offsetWidth/offsetHeight
  this.m_styleClassDimensionMap = { width: {}, height: {} };

  // cached whether row/column count are unknown
  this.m_isEstimateRowCount = undefined;
  this.m_isEstimateColumnCount = undefined;
  this.m_stopRowFetch = false;
  this.m_stopRowHeaderFetch = false;
  this.m_stopRowEndHeaderFetch = false;
  this.m_stopColumnFetch = false;
  this.m_stopColumnHeaderFetch = false;
  this.m_stopColumnEndHeaderFetch = false;

  // not done initializing until initial headers/cells are fetched
  this.m_initialized = false;
  this.m_shouldFocus = null;
  this.m_renderCount = 0;

  this.callbacks = {};

  this._setupActions();

  this.m_readinessStack = [];
  this.m_modelEvents = [];

  this.m_databodyMap = new Map();
};

// keyCodes for data grid keyboard
DvtDataGrid.prototype.keyCodes =
{
  TAB_KEY: 9,
  ENTER_KEY: 13,
  SHIFT_KEY: 16,
  CTRL_KEY: 17,
  ALT_KEY: 18,
  ESC_KEY: 27,
  SPACE_KEY: 32,
  PAGEUP_KEY: 33,
  PAGEDOWN_KEY: 34,
  END_KEY: 35,
  HOME_KEY: 36,
  LEFT_KEY: 37,
  UP_KEY: 38,
  RIGHT_KEY: 39,
  DOWN_KEY: 40,
  NUM5_KEY: 53,
  V_KEY: 86,
  X_KEY: 88,
  C_KEY: 67,
  D_KEY: 68,
  R_KEY: 82,
  F1_KEY: 112,
  F2_KEY: 113,
  F8_KEY: 119,
  F10_KEY: 121,
  F15_KEY: 126,
  A_KEY: 65
};

// constants for update animation
DvtDataGrid.UPDATE_ANIMATION_FADE_INOUT = 1;
DvtDataGrid.UPDATE_ANIMATION_SLIDE_INOUT = 2;
DvtDataGrid.UPDATE_ANIMATION_DURATION = 250;

// constants for expand/collapse animation
DvtDataGrid.EXPAND_ANIMATION_DURATION = 500;
DvtDataGrid.COLLAPSE_ANIMATION_DURATION = 400;

// swipe gesture constants
DvtDataGrid.MAX_OVERSCROLL_PIXEL = 50;
DvtDataGrid.BOUNCE_ANIMATION_DURATION = 500;
DvtDataGrid.DECELERATION_FACTOR = 0.0006;
DvtDataGrid.TAP_AND_SCROLL_RESET = 300;
// related to timing and x/y position of events
DvtDataGrid.MIN_SWIPE_DURATION = 200;
DvtDataGrid.MAX_SWIPE_DURATION = 400;
DvtDataGrid.MIN_SWIPE_DISTANCE = 10;
// for the actual transition animation
DvtDataGrid.MIN_SWIPE_TRANSITION_DURATION = 100;
DvtDataGrid.MAX_SWIPE_TRANSITION_DURATION = 500;

// constants for touch gestures
DvtDataGrid.CONTEXT_MENU_TAP_HOLD_DURATION = 750;
DvtDataGrid.HEADER_TAP_SHORT_HOLD_DURATION = 300;

// when filling viewport fetch when this close to the edge
DvtDataGrid.FETCH_PIXEL_THRESHOLD = 5;

// visibility constants
DvtDataGrid.VISIBILITY_STATE_HIDDEN = 'hidden';

DvtDataGrid.VISIBILITY_STATE_REFRESH = 'refresh';

DvtDataGrid.VISIBILITY_STATE_RENDER = 'render';

DvtDataGrid.VISIBILITY_STATE_VISIBLE = 'visible';

// Default spacer width
DvtDataGrid.SPACER_DEFAULT_WIDTH = 2;
/**
 * Sets options on DataGrid
 * @param {Object} options - the options to set on the data grid
 * @param {Function} rendererWrapperFunction - callback function used to fix renderer function for custom element
 */
DvtDataGrid.prototype.SetOptions = function (options, rendererWrapperFunction) {
  this.m_options = new DvtDataGridOptions(options, rendererWrapperFunction);
};

/**
 * Sets the original event after a sort event for use in selectionChanged events
 * @param {Event} event - sort event to store
 */
DvtDataGrid.prototype.SetSortOriginalEvent = function (event) {
  if (this.m_sortInfo) {
    this.m_sortInfo.originalEvent = event;
  }
};

/**
 * Update options on DataGrid
 * @param {Object} options - the options to set on the data grid
 * @param {Object=} flags - contains modified subkey
 */
DvtDataGrid.prototype.UpdateOptions = function (options, flags) {
  var candidates = Object.keys(options);
  var candidate;
  var i;

  // We should check each updated option (candidate) from the list of updated options (first loop)
  // against options already presented in the internal DvtDataGridOptions (this.m_options) object in
  // purpose to keep them in sync.
  for (i = 0; i < candidates.length; i++) {
    candidate = candidates[i];
    if (candidate in this.m_options.options) {
      if (this.m_options.options[candidate] !== options[candidate]) {
        this.m_options.options[candidate] = options[candidate];
      }
    }
  }

  // now update it
  for (i = 0; i < candidates.length; i++) {
    candidate = candidates[i];
    if (!this._updateDataGrid(candidate, flags)) {
      // should not get here because refresh is handled by external wrapper
      this.empty();
      this.refresh(this.m_root);
      break;
    }
  }
};

/**
 * Partial update for DataGrid
 * @private
 * @param {string} option - the option to update the data grid based on
 * @param {Object=} flags - contains modified subKey if applicable
 * @return {boolean} true if redraw is not required otherwise return false
 */
DvtDataGrid.prototype._updateDataGrid = function (option, flags) {
  var obj;

  switch (option) {
    // updates the data grid can make without refresh
    case 'bandingInterval':
      this._removeBanding();
      this.updateColumnBanding();
      this.updateRowBanding();
      break;
    case 'currentCell':
      obj = this.m_options.getCurrentCell();
      this._updateActive(obj, true, false);
      break;
    case 'editMode':
      this.m_editMode = this.m_options.getEditMode();
      break;
    case 'gridlines':
      this._updateGridlines();
      break;
    case 'header':
      obj = this.m_options.options.header;
      this._updateHeaderOptions(obj, flags);
      break;
    case 'scrollPosition':
      obj = this.m_options.getScrollPosition();
      this._updateScrollPosition(obj);
      break;
    case 'selection':
      obj = this.m_options.getSelection();
      this._updateSelection(obj);
      break;
    case 'selectionMode':
      this._clearSelection(null);
      break;

      // just refresh
    default:
      return false;
  }
  return true;
};

/**
 * Update selection from option
 * @private
 * @param {Object} selection the selection from options
 */
DvtDataGrid.prototype._updateSelection = function (selection) {
  // selection should not be null
  if (selection == null) {
    return;
  }

  // check whether selection is enabled
  if (this._isSelectionEnabled()) {
    // don't clear the selection so the old one can be passed in
    // sets the new selection
    this.SetSelection(selection);
  }
};

/**
 * Update header resizable and sortable options, all others refresh the grid for now
 * @private
 * @param {Object} headerObject
 * @param {Object=} flags
 */
DvtDataGrid.prototype._updateHeaderOptions = function (headerObject, flags) {
  if (flags == null || flags.subkey == null) {
    // should not get here as handled by outisde wrapper
    return;
  }

  var subKey = flags.subkey;
  var subKeyArray = subKey.split('.');
  var axis = subKeyArray[0];
  var option = subKeyArray[1];
  var headers;

  if (axis === 'column' && this.m_colHeader != null &&
      this.m_colHeader.firstChild != null) {
    headers = this.m_colHeader.firstChild.childNodes;
  } else if (axis === 'row' && this.m_rowHeader != null &&
             this.m_rowHeader.firstChild != null) {
    headers = this.m_rowHeader.firstChild.childNodes;
  } else if (axis === 'columnEnd' && this.m_colEndHeader != null &&
             this.m_colEndHeader.firstChild != null) {
    headers = this.m_colEndHeader.firstChild.childNodes;
  } else if (axis === 'rowEnd' && this.m_rowEndHeader != null &&
             this.m_rowEndHeader.firstChild != null) {
    headers = this.m_rowEndHeader.firstChild.childNodes;
  }

  if (headers != null) {
    for (var i = 0; i < headers.length; i++) {
      var header = headers[i];
      var headerContext = header[this.getResources().getMappedAttribute('context')];
      headerContext.index = this.getHeaderCellIndex(header);

      if (option === 'resizable') {
        if (this._isHeaderResizeEnabled(axis, headerContext)) {
          this._setAttribute(header, option, 'true');
        } else {
          this._setAttribute(header, option, 'false');
        }
      } else if (option === 'sortable') {
        var hasSortContainer =
          this.m_utils.containsCSSClassName(header.lastChild, this.getMappedStyle('sortIcon'));
        if (this._isSortEnabled(axis, headerContext)) {
          if (!hasSortContainer) {
            var sortIcon = this._buildSortIcon(headerContext, header);
            header.appendChild(sortIcon); // @HTMLUpdateOK
          }
          this._setAttribute(header, option, 'true');
        } else {
          if (hasSortContainer) {
            this._remove(header.lastChild);
          }
          this._setAttribute(header, option, 'false');
        }
      }
    }
  }
};

/**
 * Update gridlines
 * @private
 */
 DvtDataGrid.prototype._updateGridlines = function () {
  var horizontalGridlines = this.m_options.getHorizontalGridlines();
  var verticalGridlines = this.m_options.getVerticalGridlines();
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';

  if (this.m_databody && this.m_databody.firstChild) {
    let cells = this.m_databody.firstChild.querySelectorAll('.' + this.getMappedStyle('cell'));
    let lastRow = this._getLastAxis('row');
    let lastColumn = this._getLastAxis('column');
    for (let i = 0; i < cells.length; i++) {
      let cell = cells[i];
      let indexes = this.getCellIndexes(cell);
      if (verticalGridlines === 'hidden' ||
          (indexes.column === lastColumn &&
          ((this.getRowHeaderWidth() + this.getElementDir(cell, dir) +
          this.calculateColumnWidth(cell) >= this.getWidth()) ||
          this.m_endRowEndHeader > -1))) {
        this.m_utils.addCSSClassName(cell, this.getMappedStyle('borderVerticalNone'));
      } else {
        this.m_utils.removeCSSClassName(cell, this.getMappedStyle('borderVerticalNone'));
      }

      if (horizontalGridlines === 'hidden' ||
        (indexes.row === lastRow && ((this.getRowBottom(cell, null) >= this.getHeight()) ||
        this.m_endColEndHeader > -1))) {
        this.m_utils.addCSSClassName(cell, this.getMappedStyle('borderHorizontalNone'));
      } else {
        this.m_utils.removeCSSClassName(cell, this.getMappedStyle('borderHorizontalNone'));
      }
    }
  }
};

/**
 * Updates the borders of cells along the grid edges for focus ring spacing to be appropriated correctly
 * @param {string=} activeValue current border value
 */
DvtDataGrid.prototype._updateEdgeCellBorders = function (activeValue) {
  if (this.m_active != null && this.m_active.type === 'cell') {
    var activeCell = this._getActiveElement();
    if (activeCell != null) {
      if ((this._isCellEditable() && activeValue === '') || activeValue !== '') {
        this._applyBorderClassesAroundRange(activeCell,
            { startIndex: this.m_active.indexes }, activeValue === '', 'Edit');
        }

      if (this._isLastRow(this.m_active.indexes.row)) {
        if (activeValue === 'none') {
          this.m_utils.addCSSClassName(activeCell, this.getMappedStyle('borderHorizontalNone'));
        } else {
          this.m_utils.removeCSSClassName(activeCell, this.getMappedStyle('borderHorizontalNone'));
        }
      }

      if (this._isLastColumn(this.m_active.indexes.column)) {
        if (activeValue === 'none') {
          this.m_utils.addCSSClassName(activeCell, this.getMappedStyle('borderVerticalNone'));
        } else {
          this.m_utils.removeCSSClassName(activeCell, this.getMappedStyle('borderVerticalNone'));
        }
      }
    }
  }
};

/**
 * Sets resources on DataGrid
 * @param {Object} resources - the resources to set on the data grid
 */
DvtDataGrid.prototype.SetResources = function (resources) {
  this.m_resources = resources;
};

/**
 * Gets resources from DataGrid
 * @return {Object} the resources set on the data grid
 */
DvtDataGrid.prototype.getResources = function () {
  return this.m_resources;
};

/**
 * Gets start row header index from DataGrid
 * @return {number} the start row header index
 */
DvtDataGrid.prototype.getStartRowHeader = function () {
  return this.m_startRowHeader;
};

/**
 * Gets start column header index from DataGrid
 * @return {number} the start column header index
 */
DvtDataGrid.prototype.getStartColumnHeader = function () {
  return this.m_startColHeader;
};

/**
 * Gets start row end header index from DataGrid
 * @return {number} the start row end header index
 */
DvtDataGrid.prototype.getStartRowEndHeader = function () {
  return this.m_startRowEndHeader;
};

/**
 * Gets start column end header index from DataGrid
 * @return {number} the start column end header index
 */
DvtDataGrid.prototype.getStartColumnEndHeader = function () {
  return this.m_startColEndHeader;
};

/**
 * Gets mapped style from the resources
 * @private
 * @param {string} key the key to get style on
 * @return {string} the style from the resources
 */
DvtDataGrid.prototype.getMappedStyle = function (key) {
  return this.getResources().getMappedStyle(key);
};

/**
 * Sets the data source on DataGrid
 * @param {Object} dataSource - the data source to set on the data grid
 */
DvtDataGrid.prototype.SetDataSource = function (dataSource) {
  // if we are setting a new data source be sure to clear out any old
  // model events
  this.m_modelEvents = [];

  this.m_dataSource = dataSource;
};

/**
 * Gets the data source from the DataGrid
 * @return {Object} the data source set on the data grid
 */
DvtDataGrid.prototype.getDataSource = function () {
  return this.m_dataSource;
};

/**
 * Set the internal visibility of datagrid
 * @param {string} state a string for the visibility
 */
DvtDataGrid.prototype.setVisibility = function (state) {
  this.m_visibility = state;
};

/**
 * Get the internal visibility of datagrid
 * @return {string} visibility
 */
DvtDataGrid.prototype.getVisibility = function () {
  if (this.m_visibility == null) {
    if (this.m_root.offsetParent != null) {
      this.setVisibility(DvtDataGrid.VISIBILITY_STATE_VISIBLE);
    } else {
      this.setVisibility(DvtDataGrid.VISIBILITY_STATE_HIDDEN);
    }
  }
  return this.m_visibility;
};

/**
 * Set the callback for remove
 * @param {Function} callback a callback for the remove function
 */
DvtDataGrid.prototype.SetOptionCallback = function (callback) {
  this.m_setOptionCallback = callback;
};

/**
 * Set the callback for remove
 * @param {Function} callback a callback for the remove function
 */
DvtDataGrid.prototype.SetContextCallback = function (callback) {
  this.m_contextCallback = callback;
};

/**
 * Set the callback for custom element
 * @param {Function} callback a callback
 */
DvtDataGrid.prototype.SetCustomElementCallback = function (callback) {
  this.m_isCustomElementCallback = callback;
};

/**
 * Set the callback for remove
 * @param {Function} callback a callback for the remove function
 */
DvtDataGrid.prototype.SetRemoveCallback = function (callback) {
  this.m_removeCallback = callback;
};

/**
 * Set the callback for add or remove of id
 * @param {Function} callback a callback for the unique id function
 */
DvtDataGrid.prototype.SetUniqueIdCallback = function (callback) {
  this._uniqueIdCallback = callback;
};

/**
 * Set the callback for compare values
 * @param {Function} callback a callback for the compare values function
 */
DvtDataGrid.prototype.SetCompareValuesCallback = function (callback) {
  this._compareValuesCallback = callback;
};

/**
 * Set the callback for subtreeAttached that should be called when adding content to the dom
 * @param {Function} callback a callback for the subtree attached function
 */
DvtDataGrid.prototype.SetSubtreeAttachedCallback = function (callback) {
  this.m_subtreeAttachedCallback = callback;
};

/**
 * Set the callback for updating scroll position on refresh
 * @param {Function} callback a callback for the update scroll position
 */
DvtDataGrid.prototype.SetUpdateScrollPostionOnRefreshCallback = function (callback) {
  this.m_updateScrollPostionOnRefreshCallback = callback;
};

/**
 * Remove an element from the DOM, if it is not being reattached
 * @param {Element} element the element to remove
 */
DvtDataGrid.prototype._remove = function (element) {
  this._uniqueIdCallback(element, true);

  // callback allows jQuery to clean the node on a remove
  this.m_removeCallback.call(null, element);
};

/**
 * Remove all elements in an array of elements
 * @param {Array} elems an array of the elements that need to be removed
 */
DvtDataGrid.prototype._removeFromArray = function (elems) {
  for (var i = 0; i < elems.length; i++) {
    this._remove(elems[i]);
  }
};

/**
 * Set the callback for signifying not ready
 * @param {Function} callback a callback for the not ready function
 */
DvtDataGrid.prototype.SetNotReadyCallback = function (callback) {
  this.m_notReady = callback;
};

/**
 * Set the callback for signifying ready
 * @param {Function} callback a callback for the make ready function
 */
DvtDataGrid.prototype.SetMakeReadyCallback = function (callback) {
  this.m_makeReady = callback;
};

/**
 * Invoke whenever a task is started. Moves the component out of the ready state if necessary.
 */
DvtDataGrid.prototype._signalTaskStart = function () {
  if (this.m_readinessStack) {
    if (this.m_readinessStack.length === 0) {
      this.m_notReady();
    }
    this.m_readinessStack.push(1);
  }
};

/**
 * Invoke whenever a task finishes. Resolves the readyPromise if component is ready to move into ready state.
 */
DvtDataGrid.prototype._signalTaskEnd = function () {
  if (this.m_readinessStack && this.m_readinessStack.length > 0) {
    this.m_readinessStack.pop();
    if (this.m_readinessStack.length === 0) {
      this.m_makeReady();
    }
  }
};

/**
 * Get the indexes from the data source and call back to a function once they are available.
 * The callback should be a function(keys, indexes)
 * @param {Object} keys the keys to find the index of with properties row, column
 * @param {Function} callback the callback to pass the keys back to
 * @private
 */
DvtDataGrid.prototype._indexes = function (keys, callback) {
  var self = this;
  var indexes = this.getDataSource().indexes(keys);

  if (typeof indexes.then === 'function') {
    // start async indexes call
    self._signalTaskStart();
    indexes.then(function (obj) {
      callback.call(self, obj, keys);
      // end async indexes call
      self._signalTaskEnd();
    }, function () {
      callback.call(self, { row: -1, column: -1 }, keys);
      // end async indexes call
      self._signalTaskEnd();
    });
  } else {
    callback.call(self, indexes, keys);
  }
};

/**
 * Get the keys from the data source and call back to a function once they are available.
 * The callback should be a function(indexes, keys)
 * @param {Object} indexes the indexes to find the keys of with properties row, column
 * @param {Function} callback the callback to pass the indexes back to
 * @private
 */
DvtDataGrid.prototype._keys = function (indexes, callback) {
  var self = this;
  var localKeys = this._getLocalKeys(indexes);
  if (localKeys !== undefined) {
    callback.call(self, localKeys, indexes);
    return;
  }

  // check for individual stuff
  var keys = this.getDataSource().keys(indexes);
  if (typeof keys.then === 'function') {
    // start async call
    self._signalTaskStart();
    keys.then(function (obj) {
      callback.call(self, obj, indexes);
      // end async indexes call
      self._signalTaskEnd();
    }, function () {
      callback.call(self, { row: null, column: null }, indexes);
      // end async indexes call
      self._signalTaskEnd();
    });
  } else {
    callback.call(self, keys, indexes);
  }
};

/**
 * Get keys from the dom based on indexes if possible
 * @param {Object} indexes the indexes to find the keys of with properties row, column
 * @return {Object} keys
 */
DvtDataGrid.prototype._getLocalKeys = function (indexes) {
  var cell = this._getCellByIndex(indexes);
  if (cell) {
    return this.getCellKeys(cell);
  }

  var rowIndex = indexes.row;
  var columnIndex = indexes.column;
  var rowKey;
  var columnKey;
  if (rowIndex !== undefined) {
    if (rowIndex === -1) {
      rowKey = null;
    } else {
      var rowElement = this._getCellOrHeaderByIndex(rowIndex, 'row');
      if (rowElement) {
        rowKey = this._getKey(rowElement, 'row');
      }
    }

    if (rowKey === undefined) {
      return undefined;
    }
  }

  if (columnIndex !== undefined) {
    if (columnIndex === -1) {
      columnKey = null;
    } else {
      var columnElement = this._getCellOrHeaderByIndex(columnIndex, 'column');
      if (columnElement) {
        columnKey = this._getKey(columnElement, 'column');
      }
    }

    if (columnKey === undefined) {
      return undefined;
    }
  }

  return this.createIndex(rowKey, columnKey);
};

/**
 * Register a callback when creating the header context or cell context.
 * @param {function(Object)} callback the callback function to inject addition or modify
 *        properties in the context.
 */
DvtDataGrid.prototype.SetCreateContextCallback = function (callback) {
  this.m_createContextCallback = callback;
};

/**
 * Register the focusable callbacks for handling focus classNames
 * @param {function()} focusInHandler
 * @param {function()} focusOutHandler
 */
DvtDataGrid.prototype.SetFocusableCallback = function (focusInHandler, focusOutHandler) {
  this.m_focusInHandler = focusInHandler;
  this.m_focusOutHandler = focusOutHandler;
};

/**
 * Register a callback when creating the header context or cell context.
 * @param {function(Object)} callback the callback function to inject addition or modify
 *        properties in the context.
 */
DvtDataGrid.prototype.SetFixContextCallback = function (callback) {
  this.m_fixContextCallback = callback;
};

/**
 * Whether high-water mark scrolling is used
 * @return {boolean} true if high-water mark scrolling is used, false otherwise
 * @private
 */
DvtDataGrid.prototype._isHighWatermarkScrolling = function () {
  return (this.m_options.getScrollPolicy() !== 'scroll');
};

/**
 * Destructor method that should be called when the widget is destroyed. Removes event
 * listeners on the document.
 */
DvtDataGrid.prototype.destroy = function () {
  delete this.m_fetching;
  this._removeDataSourceEventListeners();
  this._removeDomEventListeners();
  delete this.m_styleClassDimensionMap;
  this.m_styleClassDimensionMap = { width: {}, height: {} };
};

/**
 * Adds data source event listeners
 * @private
 */
DvtDataGrid.prototype._addDataSourceEventListeners = function () {
  this._removeDataSourceEventListeners();

  if (this.m_dataSource != null) {
    this.m_handleModelEventListener = this.handleModelEvent.bind(this);
    this.m_handleExpandEventListener = this.handleExpandEvent.bind(this);
    this.m_handleCollapseEventListener = this.handleCollapseEvent.bind(this);

    this.m_dataSource.on('change', this.m_handleModelEventListener, this);
    // if it's not flattened datasource, these will be ignored
    this.m_dataSource.on('expand', this.m_handleExpandEventListener, this);
    this.m_dataSource.on('collapse', this.m_handleCollapseEventListener, this);
  }
};

/**
 * Remove data source event listeners
 * @private
 */
DvtDataGrid.prototype._removeDataSourceEventListeners = function () {
  if (this.m_dataSource != null) {
    this.m_dataSource.off('change', this.m_handleModelEventListener);
    this.m_dataSource.off('expand', this.m_handleExpandEventListener);
    this.m_dataSource.off('collapse', this.m_handleCollapseEventListener);
  }
};

/**
 * Adds event listeners registered on the document or the root element
 * @private
 */
DvtDataGrid.prototype._addDomEventListeners = function () {
  if (!this.m_handleDatabodyKeyDown) {
    this.m_handleDatabodyKeyDown = this.handleDatabodyKeyDown.bind(this);
  }
  if (!this.m_handleDatabodyKeyUp) {
    this.m_handleDatabodyKeyUp = this.handleDatabodyKeyUp.bind(this);
  }
  if (!this.m_handleRootFocus) {
    this.m_handleRootFocus = this.handleRootFocus.bind(this);
  }
  if (!this.m_handleRootBlur) {
    this.m_handleRootBlur = this.handleRootBlur.bind(this);
  }

  this.m_root.addEventListener('keydown', this.m_handleDatabodyKeyDown, false);
  this.m_root.addEventListener('keyup', this.m_handleDatabodyKeyUp, false);
  this.m_root.addEventListener('focus', this.m_handleRootFocus, true);
  this.m_root.addEventListener('blur', this.m_handleRootBlur, true);
};

/**
 * Remove dom event listeners
 * @private
 */
DvtDataGrid.prototype._removeDomEventListeners = function () {
  document.removeEventListener('mousemove', this.m_docMouseMoveListener, false);
  document.removeEventListener('mouseup', this.m_docMouseUpListener, false);
  // unregister all listeners

  if (this.m_root != null) {
    if (this.m_handleDatabodyKeyDown) {
      this.m_root.removeEventListener('keydown', this.m_handleDatabodyKeyDown, false);
    }
    if (this.m_handleDatabodyKeyUp) {
      this.m_root.removeEventListener('keyup', this.m_handleDatabodyKeyUp, false);
    }
    if (this.m_handleRootFocus) {
      this.m_root.removeEventListener('focus', this.m_handleRootFocus, true);
    }
    if (this.m_handleRootBlur) {
      this.m_root.removeEventListener('blur', this.m_handleRootBlur, true);
    }
  }
};

/**
 * Get the DataGrid root element
 * @return {Element} the root element
 */
DvtDataGrid.prototype.getRootElement = function () {
  return this.m_root;
};

/**
 * Get the cached width of the root element. If not cached, sets the cached width.
 * @return {number} the cached width of the root element
 * @protected
 */
DvtDataGrid.prototype.getWidth = function () {
  if (this.m_width == null) {
    // clientWidth since we use border box and care about the content of our root div
    this.m_width = this.getRootElement().clientWidth;
  }

  return this.m_width;
};

/**
 * Get the cached height of the root element. If not cached, sets the cached height.
 * @return {number} the cached height of the root element
 * @protected
 */
DvtDataGrid.prototype.getHeight = function () {
  if (this.m_height == null) {
    // clientHeight since we use border box and care about the content of our root div
    this.m_height = this.getRootElement().clientHeight;
  }

  return this.m_height;
};

/**
 * Gets the scrollable width of the grid.
 * @return {number} the scrollable width of the grid
 * @protected
 */
DvtDataGrid.prototype.getScrollableWidth = function () {
  var scrollerContent = this.m_databody.firstChild;
  return this.getElementWidth(scrollerContent);
};

/**
 * Get the viewport width, which is defined as 1.5 the size of the width of Grid
 * @return {number} the viewport width
 */
DvtDataGrid.prototype.getViewportWidth = function () {
  var width = this.getWidth();
  return Math.round(width * 1.5);
};

/**
 * Get the viewport height, which is defined as 1.5 the size of the height of Grid
 * @return {number} the viewport height
 */
DvtDataGrid.prototype.getViewportHeight = function () {
  var height = this.getHeight();
  return Math.round(height * 1.5);
};

/**
 * Get viewport top
 * @return {number} the viewport top
 */
DvtDataGrid.prototype._getViewportTop = function () {
  return this.m_currentScrollTop;
};

/**
 * Get viewport bottom
 * @return {number} the viewport bottom
 */
DvtDataGrid.prototype._getViewportBottom = function () {
  var top = this._getViewportTop();
  var databodyHeight = this.getElementHeight(this.m_databody);
  var scrollbarSize = this.m_utils.getScrollbarSize();

  return this.m_hasHorizontalScroller ?
    (top + databodyHeight) - scrollbarSize : top + databodyHeight;
};

/**
 * Get viewport left
 * @return {number} the viewport left
 */
DvtDataGrid.prototype._getViewportLeft = function () {
  return this.m_currentScrollLeft;
};

/**
 * Get viewport right
 * @return {number} the viewport right
 */
DvtDataGrid.prototype._getViewportRight = function () {
  var left = this._getViewportLeft();
  var databodyWidth = this.getElementWidth(this.m_databody);
  var scrollbarSize = this.m_utils.getScrollbarSize();

  return this.m_hasVerticalScroller ?
    (left + databodyWidth) - scrollbarSize : left + databodyWidth;
};

/**
 * Calculate the fetch size for rows or columns
 * @param {string} axis - the axis 'row'/'column' to get fetch size on
 * @return {number} the fetch size
 */
DvtDataGrid.prototype.getFetchSize = function (axis) {
  // get the cached fetch size, this should be clear when the size changes
  if (axis === 'row') {
    if (this.m_rowFetchSize == null) {
      this.m_rowFetchSize = Math.max(1, Math.round(this.getViewportHeight() /
                                                   this.getDefaultRowHeight()));
    }

    return this.m_rowFetchSize;
  }
  if (axis === 'column') {
    if (this.m_columnFetchSize == null) {
      this.m_columnFetchSize = Math.max(1, Math.round(this.getViewportWidth() /
                                                      this.getDefaultColumnWidth()));
    }
    return this.m_columnFetchSize;
  }

  return 0;
};

/**
 * If the empty text option is 'default' return default empty translated text,
 * otherwise return the emptyText set in the options
 * @return {string} the empty text
 */
DvtDataGrid.prototype.getEmptyText = function () {
  var emptyText = this.m_options.getEmptyText();
  if (emptyText == null) {
    var resources = this.getResources();
    emptyText = resources.getTranslatedText('msgNoData');
  }
  return emptyText;
};

/**
 * Build an empty text div and populate it with empty text
 * @return {Element} the empty text element
 * @private
 */
DvtDataGrid.prototype._buildEmptyText = function () {
  var emptyText = this.getEmptyText();
  var empty = document.createElement('div');
  empty.id = this.createSubId('empty');
  empty.className = this.getMappedStyle('emptytext');
  var height = this.m_endColHeader >= 0 ? this.getColumnHeaderHeight() : 0;
  this.setElementDir(empty, height, 'top');
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';
  var left = this.m_endRowHeader >= 0 ? this.getRowHeaderWidth() : 0;
  this.setElementDir(empty, left, dir);
  empty.textContent = emptyText;
  this.m_empty = empty;
  return empty;
};

/**
 * Determine the size of the buffer that triggers fetching of rows. For example,
 * if the size of the buffer is zero, then the fetch will be triggered when the
 * scroll position reached the end of where the current range ends
 * @return {number} the row threshold
 */
DvtDataGrid.prototype.getRowThreshold = function () {
  return 0;
};

/**
 * Determine the size of the buffer that triggers fetching of columns. For example,
 * if the size of the buffer is zero, then the fetch will be triggered when the
 * scroll position reached the end of where the current range ends.
 * @return {number} the column threshold
 */
DvtDataGrid.prototype.getColumnThreshold = function () {
  return 0;
};


/*
 * Caches the default datagrid dimensions located in the style sheet, creates
 * just one div to reduce createElement calls. This function should get called once on create.
 * Values found in style are:
 *  column width
 *  row height
 */
DvtDataGrid.prototype.setDefaultDimensions = function () {
  var div = document.createElement('div');
  div.style.visibilty = 'hidden';

  var resources = this.getResources();
  // we can avoid a repaint by using both row and headercell here because this isn't where the col height and row width are set
  div.className = resources.getMappedStyle('rowheadercell') + ' ' +
    resources.getMappedStyle('colheadercell') + ' ' +
    resources.getMappedStyle('headercell');
  this.m_root.appendChild(div); // @HTMLUpdateOK
  // Not using offsetWidth/Height due to
  var rect = div.getBoundingClientRect();
  this.m_defaultColumnWidth = Math.round(rect.width);
  this.m_defaultRowHeight = Math.round(rect.height);

  // minimize reflows
  this.getViewportWidth();
  this.getViewportHeight();

  this.m_root.removeChild(div);
};

/**
 * Get the default row height which comes from the style sheet
 * @return {number} the default row height
 */
DvtDataGrid.prototype.getDefaultRowHeight = function () {
  if (this.m_defaultRowHeight == null) {
    this.setDefaultDimensions();
  }
  return this.m_defaultRowHeight;
};

/**
 * Get the default column width which comes from the stylesheet
 * @return {number} the default column width
 */
DvtDataGrid.prototype.getDefaultColumnWidth = function () {
  if (this.m_defaultColumnWidth == null) {
    this.setDefaultDimensions();
  }
  return this.m_defaultColumnWidth;
};

/**
 * Gets the header dimension for an axis, for rows this would be height, for columns, width
 * @param {Element} elem the header element to get dimension of
 * @param {string|null} key the row or column key
 * @param {string} axis row or column
 * @param {string} dimension width ro height
 * @returns {number}
 */
DvtDataGrid.prototype._getHeaderDimension = function (elem, key, axis, dimension) {
  var value = this.m_sizingManager.getSize(axis, key);
  if (value != null) {
    return value;
  }

  // check if inline style set on element
  if (elem.style[dimension] !== '') {
    value = this.getElementDir(elem, dimension);
    // in the event that row height is set via an additional style only on row header store the value
    this.m_sizingManager.setSize(axis, key, value);
    return value;
  }

  // check style class mapping, mapping prevents multiple offsetHeight calls on headers with the same class name
  var className = elem.className;
  value = this.m_styleClassDimensionMap[dimension][className];
  if (value == null) {
    // exhausted all options, use offsetHeight then, remove element in the case of shim element
    value = this.getElementDir(elem, dimension);
  }

  // the value isn't the default the cell will use meaning it's from an external
  // class, so store it in the sizing manager cell can pick it up, header and cell dimension can vary on em
  this.m_sizingManager.setSize(axis, key, value);

  this.m_styleClassDimensionMap[dimension][className] = value;
  return value;
};

/**
 * Helper method to create subid based on the root element's id
 * @param {string} subId - the id to append to the root element id
 * @return {string} the subId to append to the root element id
 */
DvtDataGrid.prototype.createSubId = function (subId) {
  // id empty string if not set, enver null
  var id = this.getRootElement().id;
  return [id, subId].join(':');
};

/**
 * Checks whether header fetching is completed
 * @return {boolean} true if header fetching completed, else false
 */
DvtDataGrid.prototype.isHeaderFetchComplete = function () {
  return (this.m_fetching.row === false && this.m_fetching.column === false);
};

/**
 * Checks whether header AND cell fetching is completed
 * @return {boolean} true if header AND cell fetching completed, else false
 */
DvtDataGrid.prototype.isFetchComplete = function () {
  return (this.m_fetching != null &&
          this.isHeaderFetchComplete() &&
          this.m_fetching.cells === false);
};

/**
 * Checks whether the index is the last row
 * @param {number} index
 * @return {boolean} true if it's the last row, false otherwise
 */
DvtDataGrid.prototype._isLastRow = function (index) {
  if (this._isCountUnknown('row')) {
    // if row count is unknown, then the last row is if the index is before the last row fetched
    // and there's no more rows from datasource
    return (index === this.m_endRow && this.m_stopRowFetch);
  }

  // if column count is known, then just check the index with the total column count
  return (index + 1 === this.getDataSource().getCount('row'));
};

/**
 * Checks whether the index is the last column
 * @param {number} index
 * @return {boolean} true if it's the last column, false otherwise
 */
DvtDataGrid.prototype._isLastColumn = function (index) {
  if (this._isCountUnknown('column')) {
    // if column count is unknown, then the last column is if the index is the last column fetched
    // and there's no more columns from datasource
    return (index === this.m_endCol && this.m_stopColumnFetch);
  }

  // if column count is known, then just check the index with the total column count
  return (index + 1 === this.getDataSource().getCount('column'));
};

DvtDataGrid.prototype._getLastAxis = function (axis) {
  if (this._isCountUnknown(axis)) {
    if (axis === 'row' ? this.m_stopRowFetch : this.m_stopColumnFetch) {
      return axis === 'row' ? this.m_endRow : this.m_endCol;
    }
      return axis === 'row' ? this.m_endRow + 1 : this.m_endCol + 1;
  }
  return this.getDataSource().getCount(axis) - 1;
};

/**
 * Removes all of the datagrid children built by DvtDataGrid, this excludes context menus/popups
 */
DvtDataGrid.prototype.empty = function () {
  // remove everything that will be rebuilt
  if (this.m_empty) {
    this.m_root.removeChild(this.m_empty);
  }
  if (this.m_corner) {
    this._remove(this.m_corner);
  }
  if (this.m_bottomCorner) {
    this._remove(this.m_bottomCorner);
  }
  if (this.m_columnHeaderScrollbarSpacer) {
    this._remove(this.m_columnHeaderScrollbarSpacer);
  }
  if (this.m_rowHeaderScrollbarSpacer) {
    this._remove(this.m_rowHeaderScrollbarSpacer);
  }

  this.m_root.removeChild(this.m_placeHolder);
  this.m_root.removeChild(this.m_status);
  this.m_root.removeChild(this.m_accSummary);
  this.m_root.removeChild(this.m_accInfo);
  this.m_root.removeChild(this.m_stateInfo);
  this.m_root.removeChild(this.m_contextInfo);
  // elements that may contain other components
  this._remove(this.m_colHeader);
  this._remove(this.m_rowHeader);
  this._remove(this.m_colEndHeader);
  this._remove(this.m_rowEndHeader);
  this._remove(this.m_databody);
  this._clearDatabodyMap();
};

/**
 * Re-renders the data grid. Resets all the necessary properties.
 * @param {Element} root - the root dom element for the DataGrid.
 */
DvtDataGrid.prototype.refresh = function (root) {
  this.resetInternal();
  this.render(root);
};

/**
 * Resets internal state of data grid.
 * @private
 */
DvtDataGrid.prototype.resetInternal = function () {
  this.m_initialized = false;
  this.m_readinessStack = [];
  this._signalTaskStart();
  this._signalTaskEnd();

  // databody map
  this._clearDatabodyMap();

  // cursor
  this.m_cursor = null;

  // dom elements
  this.m_corner = null;
  this.m_bottomCorner = null;
  this.m_columnHeaderScrollbarSpacer = null;
  this.m_rowHeaderScrollbarSpacer = null;
  this.m_colHeader = null;
  this.m_colEndHeader = null;
  this.m_rowHeader = null;
  this.m_rowEndHeader = null;
  this.m_databody = null;
  this.m_empty = null;
  this.m_accInfo = null;
  this.m_accSummary = null;
  this.m_contextInfo = null;
  this.m_placeHolder = null;
  this.m_stateInfo = null;
  this.m_status = null;
  this.m_headerLabels = { row: [], column: [], rowEnd: [], columnEnd: [] };

  // fetching
  this.m_isEstimateRowCount = undefined;
  this.m_isEstimateColumnCount = undefined;
  this.m_stopRowFetch = false;
  this.m_stopRowHeaderFetch = false;
  this.m_stopRowEndHeaderFetch = false;
  this.m_stopColumnFetch = false;
  this.m_stopColumnHeaderFetch = false;
  this.m_stopColumnEndHeaderFetch = false;
  this.m_rowFetchSize = null;
  this.m_columnFetchSize = null;
  this.m_fetching = null;
  this.m_processingModelEvent = false;
  this.m_processingEventQueue = false;
  this.m_animating = false;

  // dimensions
  this.m_sizingManager.clear();
  this.m_styleClassDimensionMap = { width: {}, height: {} };
  this.m_height = null;
  this.m_width = null;
  this.m_scrollHeight = null;
  this.m_scrollWidth = null;
  this.m_avgRowHeight = undefined;
  this.m_avgColWidth = undefined;
  this.m_defaultColumnWidth = null;
  this.m_defaultRowHeight = null;
  this.m_colHeaderHeight = null;
  this.m_colEndHeaderHeight = null;
  this.m_rowHeaderWidth = null;
  this.m_rowEndHeaderWidth = null;
  this.m_rowHeaderLevelWidths = [];
  this.m_rowEndHeaderLevelWidths = [];
  this.m_columnHeaderLevelHeights = [];
  this.m_columnEndHeaderLevelHeights = [];
  this.m_collisionResize = false;

  // active
  this.m_active = null;
  this.m_prevActive = null;
  this.m_trueIndex = {};

  // dnd
  this.m_headerDragState = false;
  this.m_databodyDragState = false;
  this.m_databodyMove = false;
  this.m_moveRow = null;
  this.m_moveRowHeader = null;
  this.m_dropTarget = null;
  this.m_dropTargetHeader = null;

  // cut/copy/paste/fill
  this.m_floodFillDragState = false;
  this.m_dataTransferAction = null;

  // selection
  this.m_discontiguousSelection = false;

  // event listeners
  this.m_docMouseMoveListener = null;
  this.m_docMouseUpListener = null;
  this.m_modelEvents = [];

  // scrolling
  this.m_hasHorizontalScroller = null;
  this.m_hasVerticalScroller = null;
  this.m_currentScrollLeft = null;
  this.m_currentScrollTop = null;
  this.m_prevScrollLeft = null;
  this.m_prevScrollTop = null;
  this.m_handleScrollOverflow = null;
  // this.m_scrollOnRefreshEvent = false;
  this._clearScrollPositionTimeout();

  // resizing
  this.m_resizing = false;
  this.m_resizingElement = null;
  this.m_resizingElementMin = null;

  // data states
  this.m_startRow = null;
  this.m_startCol = null;
  this.m_endRow = null;
  this.m_endCol = null;
  this.m_startRowPixel = null;
  this.m_startColPixel = null;
  this.m_endRowPixel = null;
  this.m_endColPixel = null;
  this.m_startRowHeader = null;
  this.m_startColHeader = null;
  this.m_endRowHeader = null;
  this.m_endColHeader = null;
  this.m_startRowHeaderPixel = null;
  this.m_startColHeaderPixel = null;
  this.m_endRowHeaderPixel = null;
  this.m_endColHeaderPixel = null;
  this.m_rowHeaderLevelCount = null;
  this.m_columnHeaderLevelCount = null;
  this.m_startRowEndHeader = null;
  this.m_startColEndHeader = null;
  this.m_endRowEndHeader = null;
  this.m_endColEndHeader = null;
  this.m_startRowEndHeaderPixel = null;
  this.m_startColEndHeaderPixel = null;
  this.m_endRowEndHeaderPixel = null;
  this.m_endColEndHeaderPixel = null;
  this.m_rowEndHeaderLevelCount = null;
  this.m_columnEndHeaderLevelCount = null;
  this.m_sortInfo = null;
  this.m_expandCollapseInfo = null;
  this.m_resizeRequired = null;
  this.m_externalFocus = null;
  this.m_currentMode = null;
  this.m_editMode = null;

  this.m_hasCells = null;
  this.m_hasRowHeader = null;
  this.m_hasRowEndHeader = null;
  this.m_hasColHeader = null;
  this.m_hasColEndHeader = null;
  this.m_isLongScroll = null;

  this.m_addBorderBottom = null;
  this.m_addBorderRight = null;

  this.m_sortContainerWidth = null;
  this.m_sortContainerHeight = null;

  this._destroyEditableClone();
  this._clearFocusoutTimeout();
  this._clearFocusoutBusyState();
};

/**
 * DataGrid should initialize if there's no outstanding fetch, it is unitialized
 * and the databody is attached to the root.
 * @private
 * @returns {boolean} true if we have the properties that signify an end to initialize
 */
DvtDataGrid.prototype._shouldInitialize = function () {
  return (this.isFetchComplete() && !this.m_initialized && this.m_databody.parentNode != null);
};

/**
 * Handle the end of datagrid initialization whether at the end of rendering or fetching
 * @private
 * @param {boolean=} hasData false if there is no data and thus should skip resizing
 */
DvtDataGrid.prototype._handleInitialization = function (hasData) {
  if (hasData === true) {
    this.resizeGrid();
    if (this.m_startRow === 0 && this.m_startCol === 0) {
      this.fillViewport();
    }

    if (this.isFetchComplete()) {
      this._updateActive(this.m_options.getCurrentCell(), !!this.m_focusOnRefresh, true);
      this.m_initialized = true;
      this.fireEvent('ready', {});
      this._runModelEventQueue();
    }
  } else {
    this.m_initialized = true;
    this.fireEvent('ready', {});
    this._runModelEventQueue();
  }
};

/**
 * Run the events in the model event list
 * The queue shifts the first event and runs that.
 * The event is expected to call _runModelEventQueue
 * once it completes
 * If the queue is empty, stop processing
 * Usage: After any animation related event chain
 *        completes, run the event queue.
 *        During the event queue, queued events
 *        should also have a call back to _runModelEventQueue
 *        at its completion. See handleExpandEvent for an example
 * @private
 */
DvtDataGrid.prototype._runModelEventQueue = function () {
  var event;
  // Run the event queue generally after initialization
  // or animations are complete.
  // m_modelEvents acts as the queue and will
  // be initialized normally. In the case that
  // the m_modelEvent queue is accessed before
  // it is initialized, it will act as length 0 queue
  if (this.m_modelEvents != null) {
    this.m_processingEventQueue = true;
    if (this.m_modelEvents.length === 0) {
      this.m_processingEventQueue = false;
      return;
    }

    event = this.m_modelEvents.shift();

    if (event.operation === 'expand') {
      this.handleExpandEvent(event, true);
    } else if (event.operation === 'collapse') {
      this.handleCollapseEvent(event, true);
    } else {
      this.handleModelEvent(event, true);
    }
  } else {
    this.m_processingEventQueue = false;
  }
};

/**
 * Renders the DataGrid, initializes DataGrid properties.
 * @param {Element} root - the root dom element for the DataGrid.
 */
DvtDataGrid.prototype.render = function (root) {
  this.m_renderCount += 1;
  this.m_timingStart = new Date();
  this.m_fetching = {
  };

  // since headers and cells are independently fetched, they could be returned
  // at a different time, therefore we'll need var to keep track the current range
  // for each one of them
  this.m_startRow = 0;
  this.m_startCol = 0;
  this.m_endRow = -1;
  this.m_endCol = -1;
  this.m_startRowPixel = 0;
  this.m_startColPixel = 0;
  this.m_endRowPixel = 0;
  this.m_endColPixel = 0;

  this.m_startRowHeader = 0;
  this.m_startColHeader = 0;
  this.m_endRowHeader = -1;
  this.m_endColHeader = -1;
  this.m_startRowHeaderPixel = 0;
  this.m_startColHeaderPixel = 0;
  this.m_endRowHeaderPixel = 0;
  this.m_endColHeaderPixel = 0;

  this.m_startRowEndHeader = 0;
  this.m_startColEndHeader = 0;
  this.m_endRowEndHeader = -1;
  this.m_endColEndHeader = -1;
  this.m_startRowEndHeaderPixel = 0;
  this.m_startColEndHeaderPixel = 0;
  this.m_endRowEndHeaderPixel = 0;
  this.m_endColEndHeaderPixel = 0;

  this.m_currentScrollLeft = 0;
  this.m_currentScrollTop = 0;
  this.m_prevScrollLeft = 0;
  this.m_prevScrollTop = 0;
  this.m_handleScrollOverflow = false;

  this.m_rowHeaderLevelWidths = [];
  this.m_rowEndHeaderLevelWidths = [];
  this.m_columnHeaderLevelHeights = [];
  this.m_columnEndHeaderLevelHeights = [];

  var enginePromise = this._loadTemplateEngine();

  if (enginePromise) {
    this._signalTaskStart('loading template engine');
    enginePromise.then(() => {
      this._signalTaskEnd();
      this.m_renderCount -= 1;
      if (this.m_renderCount === 0) {
        this.buildGrid(root);
      }
    });
  } else {
    this.m_renderCount -= 1;
    this.buildGrid(root);
  }
};

/**
  * Initiate loading of the template engine.  An error is thrown if the template engine failed to load.
  * @return {Promise} resolves to the template engine, or null if:
  *                   1) there's no need because no templates are specified
  *                   2) this.m_options.data is not an instance of a data grid provider
  * @private
  * @memberof oj.ojDataGrid
  */
 DvtDataGrid.prototype._loadTemplateEngine = function () {
  var slotMap = this._getSlotMap();
  // Greater then 1 because datagridcontextmenu is always returned.
  if (this._isDataGridProvider() && Object.keys(slotMap).length > 1) {
    return new Promise((resolve) => {
      __getTemplateEngine().then((engine) => {
        this.m_engine = engine;
        resolve(engine);
      }, function (reason) {
        throw new Error('Error loading template engine: ' + reason);
      });
    });
  }

  return null;
};

/**
  * Retrieve the template engine, returns null if it has not been loaded yet
  * @private
  * @memberof oj.ojDataGrid
  */
 DvtDataGrid.prototype._getTemplateEngine = function () {
  return this.m_engine;
};

/**
  * Returns true if instance of DataGridProvider
  * @private
  * @memberof oj.ojDataGrid
  */
 DvtDataGrid.prototype._isDataGridProvider = function () {
    return this.m_options.options.data
    && this.m_options.options.data.fetchByOffset
    && !this.m_options.options.data.fetchFirst;
};

/**
 * Returns the slot map object.
 * @return {object} slot Map
 * @private
 * @memberof oj.ojDataGrid
 */
 DvtDataGrid.prototype._getSlotMap = function () {
  return CustomElementUtils.getSlotMap(this.m_root);
};

/**
 * Returns the inline template element inside oj-data-grid by it's slotName
 * @return {Element|null} the inline template element
 * @param {slotName} string The name of slot to be returned.
 * @private
 * @memberof oj.ojDataGrid
 */
 DvtDataGrid.prototype._getItemTemplateBySlotName = function (slotName) {
  var slotMap = this._getSlotMap();
  var slot = slotMap[slotName];
  if (slot && slot.length > 0 && slot[0].tagName.toLowerCase() === 'template') {
    return slot[0];
  }
  return null;
};

/**
 * Builds the DataGrid, adds root children (headers, databody, corners),
 * initializes event listeners, and sets inital scroll position.
 * @param {Element} root - the root dom element for the DataGrid.
 */
DvtDataGrid.prototype.buildGrid = function (root) {
  this.m_root = root;
  // class name set on component create
  this.m_root.setAttribute('role', 'application');
  if (this._isCellEditable()) {
    this.m_utils.addCSSClassName(this.m_root, this.getMappedStyle('editable'));
  } else {
    this.m_utils.addCSSClassName(this.m_root, this.getMappedStyle('readOnly'));
  }
  // this.m_root.setAttribute("aria-describedby", this.createSubId("summary"));

  this.setDefaultDimensions();

  // set a tab index so it can be focus and keyboard navigation to work
  // eslint-disable-next-line no-param-reassign
  root.tabIndex = 0;

  var status = this.buildStatus();
  root.appendChild(status); // @HTMLUpdateOK
  this.m_status = status;

  var accSummary = this.buildAccSummary();
  root.appendChild(accSummary); // @HTMLUpdateOK
  this.m_accSummary = accSummary;

  var accInfo = this.buildAccInfo();
  root.appendChild(accInfo); // @HTMLUpdateOK
  this.m_accInfo = accInfo;

  var stateInfo = this.buildStateInfo();
  root.appendChild(stateInfo); // @HTMLUpdateOK
  this.m_stateInfo = stateInfo;

  var contextInfo = this.buildContextInfo();
  root.appendChild(contextInfo); // @HTMLUpdateOK
  this.m_contextInfo = contextInfo;

  var placeHolder = this.buildPlaceHolder();
  root.appendChild(placeHolder); // @HTMLUpdateOK
  this.m_placeHolder = placeHolder;

  this.m_headerLabels = { row: [], column: [], rowEnd: [], columnEnd: [] };

  if (this.getDataSource() != null) {
    // in the event that the empty text was set when there was no datasource
    this.m_empty = null;

    var rtl = this.getResources().isRTLMode();

    var returnObj = this.buildHeaders('column', this.getMappedStyle('colheader'),
      this.getMappedStyle('colendheader'));
    var colHeader = returnObj.root;
    var colEndHeader = returnObj.endRoot;
    root.insertBefore(colHeader, status); // @HTMLUpdateOK
    root.insertBefore(colEndHeader, status); // @HTMLUpdateOK

    returnObj = this.buildHeaders('row', this.getMappedStyle('rowheader'),
      this.getMappedStyle('rowendheader'));
    var rowHeader = returnObj.root;
    var rowEndHeader = returnObj.endRoot;
    root.insertBefore(rowHeader, status); // @HTMLUpdateOK
    root.insertBefore(rowEndHeader, status); // @HTMLUpdateOK

    var databody = this.buildDatabody();
    root.insertBefore(databody, status); // @HTMLUpdateOK

    if (rtl) {
      colHeader.style.direction = 'rtl';
      databody.style.direction = 'rtl';
    }

    this.m_isResizing = false;
    this.m_resizingElement = null;
    this.m_resizingElementMin = null;
    this.m_databodyDragState = false;

    // store the listeners so we can remove them later (bind creates a new function)
    this.m_docMouseMoveListener = this.handleMouseMove.bind(this);
    this.m_docMouseUpListener = this.handleMouseUp.bind(this);

    // touch event handling
    if (this.m_utils.isTouchDevice()) {
      // databody touch listeners
      databody.addEventListener('touchstart', this.handleTouchStart.bind(this), { passive: true });
      databody.addEventListener('touchmove', this.handleTouchMove.bind(this), { passive: false });
      databody.addEventListener('touchend', this.handleTouchEnd.bind(this), false);
      databody.addEventListener('touchcancel', this.handleTouchCancel.bind(this), false);

      // column header listeners
      colHeader.addEventListener('touchstart', this.handleHeaderTouchStart.bind(this), { passive: true });
      colHeader.addEventListener('touchmove', this.handleHeaderTouchMove.bind(this), { passive: false });
      colHeader.addEventListener('touchend', this.handleHeaderTouchEnd.bind(this), false);
      colHeader.addEventListener('touchcancel', this.handleHeaderTouchCancel.bind(this), false);

      // row header listeners
      rowHeader.addEventListener('touchstart', this.handleHeaderTouchStart.bind(this), { passive: true });
      rowHeader.addEventListener('touchmove', this.handleHeaderTouchMove.bind(this), { passive: false });
      rowHeader.addEventListener('touchend', this.handleHeaderTouchEnd.bind(this), false);
      rowHeader.addEventListener('touchcancel', this.handleHeaderTouchCancel.bind(this), false);

      // column end header listeners
      colEndHeader.addEventListener('touchstart', this.handleHeaderTouchStart.bind(this), { passive: true });
      colEndHeader.addEventListener('touchmove', this.handleHeaderTouchMove.bind(this), { passive: false });
      colEndHeader.addEventListener('touchend', this.handleHeaderTouchEnd.bind(this), false);
      colEndHeader.addEventListener('touchcancel', this.handleHeaderTouchCancel.bind(this), false);

      // row end header listeners
      rowEndHeader.addEventListener('touchstart', this.handleHeaderTouchStart.bind(this), { passive: true });
      rowEndHeader.addEventListener('touchmove', this.handleHeaderTouchMove.bind(this), { passive: false });
      rowEndHeader.addEventListener('touchend', this.handleHeaderTouchEnd.bind(this), false);
      rowEndHeader.addEventListener('touchcancel', this.handleHeaderTouchCancel.bind(this), false);
    } else {
      // non-touch event listening

      var mousewheelEvent = this.m_utils.getMousewheelEvent();
      // databody listeners
      databody.addEventListener(
        mousewheelEvent, this.handleDatabodyMouseWheel.bind(this), { passive: false });
      databody.addEventListener('mousedown', this.handleDatabodyMouseDown.bind(this), false);
      databody.addEventListener('mousemove', this.handleDatabodyMouseMove.bind(this), false);
      databody.addEventListener('mouseup', this.handleDatabodyMouseUp.bind(this), false);
      databody.addEventListener('mouseout', this.handleDatabodyMouseOut.bind(this), false);
      databody.addEventListener('mouseover', this.handleDatabodyMouseOver.bind(this), false);
      databody.addEventListener('dblclick', this.handleDatabodyDoubleClick.bind(this), false);

      // header listeners
      rowHeader.addEventListener(
        mousewheelEvent, this.handleDatabodyMouseWheel.bind(this), { passive: false });
      colHeader.addEventListener(
        mousewheelEvent, this.handleDatabodyMouseWheel.bind(this), { passive: false });
      rowHeader.addEventListener('mousedown', this.handleHeaderMouseDown.bind(this), false);
      colHeader.addEventListener('mousedown', this.handleHeaderMouseDown.bind(this), false);
      rowHeader.addEventListener('mouseover', this.handleHeaderMouseOver.bind(this), false);
      colHeader.addEventListener('mouseover', this.handleHeaderMouseOver.bind(this), false);
      rowHeader.addEventListener('mousemove', this.handleRowHeaderMouseMove.bind(this), false);
      colHeader.addEventListener('mousemove', this.handleColumnHeaderMouseMove.bind(this), false);
      rowHeader.addEventListener('mouseup', this.handleHeaderMouseUp.bind(this), false);
      colHeader.addEventListener('mouseup', this.handleHeaderMouseUp.bind(this), false);
      rowHeader.addEventListener('mouseout', this.handleHeaderMouseOut.bind(this), false);
      colHeader.addEventListener('mouseout', this.handleHeaderMouseOut.bind(this), false);
      rowHeader.addEventListener('click', this.handleHeaderClick.bind(this), false);
      rowHeader.addEventListener('dblclick', this.handleHeaderDoubleClick.bind(this), false);
      colHeader.addEventListener('click', this.handleHeaderClick.bind(this), false);
      colHeader.addEventListener('dblclick', this.handleHeaderDoubleClick.bind(this), false);

      // end header listeners
      rowEndHeader.addEventListener(
        mousewheelEvent, this.handleDatabodyMouseWheel.bind(this), { passive: false });
      colEndHeader.addEventListener(
        mousewheelEvent, this.handleDatabodyMouseWheel.bind(this), { passive: false });
      rowEndHeader.addEventListener('mousedown', this.handleHeaderMouseDown.bind(this), false);
      colEndHeader.addEventListener('mousedown', this.handleHeaderMouseDown.bind(this), false);
      rowEndHeader.addEventListener('mouseover', this.handleHeaderMouseOver.bind(this), false);
      colEndHeader.addEventListener('mouseover', this.handleHeaderMouseOver.bind(this), false);
      rowEndHeader.addEventListener('mousemove', this.handleRowHeaderMouseMove.bind(this), false);
      colEndHeader.addEventListener('mousemove',
        this.handleColumnHeaderMouseMove.bind(this), false);
      rowEndHeader.addEventListener('mouseup', this.handleHeaderMouseUp.bind(this), false);
      colEndHeader.addEventListener('mouseup', this.handleHeaderMouseUp.bind(this), false);
      rowEndHeader.addEventListener('mouseout', this.handleHeaderMouseOut.bind(this), false);
      colEndHeader.addEventListener('mouseout', this.handleHeaderMouseOut.bind(this), false);
      rowEndHeader.addEventListener('click', this.handleHeaderClick.bind(this), false);
      rowEndHeader.addEventListener('dblclick', this.handleHeaderDoubleClick.bind(this), false);
      colEndHeader.addEventListener('click', this.handleHeaderClick.bind(this), false);
      colEndHeader.addEventListener('dblclick', this.handleHeaderDoubleClick.bind(this), false);
    }

    // check if data is fetched and size the grid
    if (this._shouldInitialize()) {
      this._handleInitialization(true);
    }
  } else {
    // if no datasource render empty text
    var empty = this._buildEmptyText();
    this.m_root.appendChild(empty); // @HTMLUpdateOK
    this._handleInitialization(false);
  }
};

/**
 * Handle resize of grid to a new width and height.
 * @param {number} width the new width
 * @param {number} height the new height
 */
DvtDataGrid.prototype.HandleResize = function (width, height) {
  // can either get the client width or subtract the border width.
  // eslint-disable-next-line no-param-reassign
  width = this.getRootElement().clientWidth;
  // eslint-disable-next-line no-param-reassign
  height = this.getRootElement().clientHeight;
  // don't do anything if nothing has changed
  if (width !== this.m_width || height !== this.m_height) {
    // assign new width and height
    this.m_width = width;
    this.m_height = height;

    this.m_rowFetchSize = null;
    this.m_columnFetchSize = null;

    // if it's not initialize (or fetching), then just skip now
    // handleCellsFetchSuccess will attempt to fill the viewport
    if (this.m_initialized) {
      // call resize logic
      this.resizeGrid();
      if (this.isFetchComplete()) {
        this.m_resizeRequired = true;
        // check viewport
        this.fillViewport();
      }
    }
  }
};

/**
 * Size the headers, scroller, databody based on current width and height.
 * @private
 */
DvtDataGrid.prototype.resizeGrid = function () {
  var width = this.getWidth();
  var height = this.getHeight();
  var colHeader = this.m_colHeader;
  var colEndHeader = this.m_colEndHeader;
  var rowHeader = this.m_rowHeader;
  var rowEndHeader = this.m_rowEndHeader;
  var databody = this.m_databody;
  var databodyScroller = databody.firstChild;

  // cache these since they will be used in multiple places and we want to minimize reflow
  var colHeaderHeight = this.getColumnHeaderHeight();
  var colEndHeaderHeight = this.getColumnEndHeaderHeight();
  var rowHeaderWidth = this.getRowHeaderWidth();
  var rowEndHeaderWidth = this.getRowEndHeaderWidth();

  // adjusted to make the databody wrap the databody content, and the scroller to fill the remaing part of the grid
  // this way our scrollbars are always at the edges of our viewport
  var availableHeight = height - colHeaderHeight - colEndHeaderHeight;
  var availableWidth = width - rowHeaderWidth - rowEndHeaderWidth;

  var scrollbarSize = this.m_utils.getScrollbarSize();

  var isEmpty = this._databodyEmpty();
  var empty;
  var emptyHeight;
  var emptyWidth;

  // check if there's no data
  if (isEmpty) {
    // could be getting here in the handle resize of an empty grid
    if (this.m_empty == null) {
      empty = this._buildEmptyText();
      this.m_root.appendChild(empty); // @HTMLUpdateOK
    } else {
      empty = this.m_empty;
    }
    emptyHeight = this.getElementHeight(empty);
    emptyWidth = this.getElementWidth(empty);

    if (emptyHeight > this.getElementHeight(databodyScroller)) {
      this.setElementHeight(databodyScroller, emptyHeight);
    }
    if (emptyWidth > this.getElementWidth(databodyScroller)) {
      this.setElementWidth(databodyScroller, emptyWidth);
    }
  }

  var databodyContentWidth = this.getElementWidth(databody.firstChild);
  var databodyContentHeight = this.getElementHeight(databody.firstChild);
  // determine which scrollbars are required, if needing one forces need of the other, allows rendering within the root div
  var isDatabodyHorizontalScrollbarRequired =
      this.isDatabodyHorizontalScrollbarRequired(availableWidth);
  var isDatabodyVerticalScrollbarRequired;

  if (isDatabodyHorizontalScrollbarRequired) {
    isDatabodyVerticalScrollbarRequired =
      this.isDatabodyVerticalScrollbarRequired(availableHeight - scrollbarSize);
    databody.style.overflow = 'auto';
  } else {
    isDatabodyVerticalScrollbarRequired =
      this.isDatabodyVerticalScrollbarRequired(availableHeight);
    if (isDatabodyVerticalScrollbarRequired) {
      isDatabodyHorizontalScrollbarRequired =
        this.isDatabodyHorizontalScrollbarRequired(availableWidth - scrollbarSize);
      databody.style.overflow = 'auto';
    } else {
      // for an issue where same size child causes scrollbars (similar code used in resizing already)
      // Adding timeout to address firefox async scrolling and pixel perfect scroll bar issues:
      // If the datagrid doesn't need scrolling, ff will skip the async scroll event, so we need to manually handle
      // it with a timeout. Flag added here and handleScroll so that either will execute, but only once.
      this.m_handleScrollOverflow = false;
      var self = this;
      setTimeout(function () {
        // Need to check this in case the state has changed during the timeout period.
        if (!self.m_handleScrollOverflow &&
            !self.m_hasVerticalScroller && !self.m_hasHorizontalScroller) {
          databody.style.overflow = 'hidden';
          self.m_handleScrollOverflow = true;
        }
      }, 10);
    }
  }

  this.m_hasHorizontalScroller = isDatabodyHorizontalScrollbarRequired;
  this.m_hasVerticalScroller = isDatabodyVerticalScrollbarRequired;

  var databodyHeight;
  var rowHeaderHeight;
  var databodyWidth;
  var columnHeaderWidth;

  if (this.m_endColEndHeader !== -1) {
    databodyHeight = Math.min(databodyContentHeight +
                              (isDatabodyHorizontalScrollbarRequired ? scrollbarSize : 0),
    availableHeight);
    rowHeaderHeight = isDatabodyHorizontalScrollbarRequired ?
      databodyHeight - scrollbarSize : databodyHeight;
  } else {
    databodyHeight = availableHeight;
    rowHeaderHeight = Math.min(databodyContentHeight,
      isDatabodyHorizontalScrollbarRequired ?
        databodyHeight - scrollbarSize : databodyHeight);
  }

  if (this.m_endRowEndHeader !== -1) {
    databodyWidth = Math.min(databodyContentWidth +
                             (isDatabodyVerticalScrollbarRequired ? scrollbarSize : 0),
    availableWidth);
    columnHeaderWidth = isDatabodyVerticalScrollbarRequired ?
      databodyWidth - scrollbarSize : databodyWidth;
  } else {
    databodyWidth = availableWidth;
    columnHeaderWidth = Math.min(databodyContentWidth,
      isDatabodyVerticalScrollbarRequired ?
        databodyWidth - scrollbarSize : databodyWidth);
  }

  var rowEndHeaderDir = rowHeaderWidth + columnHeaderWidth +
    (isDatabodyVerticalScrollbarRequired ? scrollbarSize : 0);
  var columnEndHeaderDir = colHeaderHeight + rowHeaderHeight +
    (isDatabodyHorizontalScrollbarRequired ? scrollbarSize : 0);

  var dir = this.getResources().isRTLMode() ? 'right' : 'left';

  this.setElementDir(rowHeader, 0, dir);
  this.setElementDir(rowHeader, colHeaderHeight, 'top');
  this.setElementHeight(rowHeader, rowHeaderHeight);

  this.setElementDir(rowEndHeader, rowEndHeaderDir, dir);
  this.setElementDir(rowEndHeader, colHeaderHeight, 'top');
  this.setElementHeight(rowEndHeader, rowHeaderHeight);

  this.setElementDir(colHeader, rowHeaderWidth, dir);
  this.setElementWidth(colHeader, columnHeaderWidth);

  this.setElementDir(colEndHeader, rowHeaderWidth, dir);
  this.setElementDir(colEndHeader, columnEndHeaderDir, 'top');
  this.setElementWidth(colEndHeader, columnHeaderWidth);

  this.setElementDir(databody, colHeaderHeight, 'top');
  this.setElementDir(databody, rowHeaderWidth, dir);
  this.setElementWidth(databody, databodyWidth);
  this.setElementHeight(databody, databodyHeight);

  // cache the scroll width and height to minimize reflow
  this.m_scrollWidth = databodyContentWidth - columnHeaderWidth;
  this.m_scrollHeight = databodyContentHeight - rowHeaderHeight;

  this.buildCorners();

  // check if we need to remove border on the last column header/add borders to headers and cells
  this._adjustHeaderBorders();
  this._updateGridlines();

  // now we do not need to resize
  this.m_resizeRequired = false;
};

/**
 * Size the databody scroller based on whatever dimensions are available.
 * @private
 */
DvtDataGrid.prototype._sizeDatabodyScroller = function () {
  var databody = this.m_databody;
  var scroller = databody.firstChild;
  var isEmpty = this._databodyEmpty();
  var isHWS = this._isHighWatermarkScrolling();
  var maxHeight = this.m_utils._getMaxDivHeightForScrolling();
  var maxWidth = this.m_utils._getMaxDivWidthForScrolling();
  var rowCount = this.getDataSource().getCount('row');
  var colCount = this.getDataSource().getCount('column');
  var totalHeight = 0;
  var totalWidth = 0;
  var endRowPixel = 0;
  var endColPixel = 0;

  if (isEmpty) {
    // min is 1 so that the scrollbars show up
    endRowPixel = Math.max(Math.max(this.m_endRowHeaderPixel, this.m_endRowEndHeaderPixel), 1);
    endColPixel = Math.max(Math.max(this.m_endColHeaderPixel, this.m_endColEndHeaderPixel), 1);
  } else {
    endRowPixel = this.m_endRowPixel;
    endColPixel = this.m_endColPixel;
  }

  totalHeight = (rowCount !== -1 && !isHWS) ? rowCount * this.m_avgRowHeight : endRowPixel;
  totalWidth = (colCount !== -1 && !isHWS) ? colCount * this.m_avgColWidth : endColPixel;

  this.setElementHeight(scroller, Math.min(maxHeight, totalHeight));
  this.setElementWidth(scroller, Math.min(maxWidth, totalWidth));

  if (this.m_initialized) {
    this.m_scrollWidth = (this.getElementWidth(scroller) - Math.min(this.getElementWidth(scroller),
      this.getElementWidth(databody) -
      (this.m_hasVerticalScroller ? this.m_utils.getScrollbarSize() : 0)));
    this.m_scrollHeight = (this.getElementHeight(scroller) - Math.min(
      this.getElementHeight(scroller), this.getElementHeight(databody) -
      (this.m_hasHorizontalScroller ? this.m_utils.getScrollbarSize() : 0)));
  }
};

/**
 * Adjust the last header on specific axis properties
 * @private
 * @param {number} headerIndex
 * @param {number} headerLevels
 * @param {Element} container
 * @param {number} startIndex
 * @param {string} className
 * @param {boolean} remove
 */
DvtDataGrid.prototype._adjustLastHeadersAlongAxis =
  function (headerIndex, headerLevels, container, startIndex, className, remove) {
    var i = 0;
    while (i < headerLevels) {
      var lastHeader = this._getHeaderByIndex(headerIndex, i, container, headerLevels, startIndex);
      if (remove) {
        this.m_utils.removeCSSClassName(lastHeader, className);
      } else {
        this.m_utils.addCSSClassName(lastHeader, className);
      }
      i += this.getHeaderCellDepth(lastHeader);
    }
  };

/**
 * Adjust the last header and the spacer along a given axis
 *
 * @param {Element} container
 * @param {Function} lastFunction
 * @param {number} endHeaderIndex
 * @param {boolean} dimensionCheck
 * @param {Element} spacer
 * @param {string} className
 * @param {number} headerLevels
 * @param {number} startIndex
 */
DvtDataGrid.prototype._adjustHeaderBordersAlongAxis = function (
  container, lastFunction, endHeaderIndex, dimensionCheck, spacer, className,
  headerLevels, startIndex
) {
  if (container != null && endHeaderIndex >= 0) {
    if (dimensionCheck) {
      this.m_utils.addCSSClassName(spacer, className);
    } else {
      this.m_utils.removeCSSClassName(spacer, className);
    }
    if (lastFunction(endHeaderIndex)) {
      this._adjustLastHeadersAlongAxis(endHeaderIndex, headerLevels, container,
        startIndex, className, dimensionCheck);
    }
  }
};

/**
 * Adjust the border style/width setting on the headers using classNames so that they can be overwritten
 * @private
 */
DvtDataGrid.prototype._adjustHeaderBorders = function () {
  var scrollbarSize = this.m_utils.getScrollbarSize();
  var width = this.getWidth();
  var height = this.getHeight();
  var colHeaderHeight = this.getColumnHeaderHeight();
  var colHeaderWidth = this.getElementWidth(this.m_colHeader);
  var colEndHeaderHeight = this.getColumnEndHeaderHeight();
  var rowHeaderWidth = this.getRowHeaderWidth();
  var rowHeaderHeight = this.getElementHeight(this.m_rowHeader);
  var rowEndHeaderWidth = this.getRowEndHeaderWidth();

  var widthCheck = rowHeaderWidth + colHeaderWidth + rowEndHeaderWidth +
    (this.m_hasVerticalScroller ? scrollbarSize : 0) < width;
  var heightCheck = colHeaderHeight + rowHeaderHeight + colEndHeaderHeight +
    (this.m_hasHorizontalScroller ? scrollbarSize : 0) < height;

  var bw;
  var style;
  var i;
  var tags;
  var lastFunction;

  if (widthCheck && this.m_endRowEndHeader >= 0) {
    bw = true;
    this.m_addBorderRight = true;
  } else if (this.m_addBorderRight === true) {
    bw = false;
  }

  if (bw != null) {
    style = this.getMappedStyle('borderVerticalSmall');
    if (this.m_columnHeaderScrollbarSpacer != null) {
      if (bw) {
        this.m_utils.addCSSClassName(this.m_columnHeaderScrollbarSpacer, style);
      } else {
        this.m_utils.removeCSSClassName(this.m_columnHeaderScrollbarSpacer, style);
      }
    }
    if (this.m_bottomCorner != null) {
      if (bw) {
        this.m_utils.addCSSClassName(this.m_bottomCorner, style);
      } else {
        this.m_utils.removeCSSClassName(this.m_bottomCorner, style);
      }
    }
    tags = this.m_rowEndHeader.firstChild.childNodes;
    for (i = 0; i < tags.length; i++) {
      if (bw) {
        this.m_utils.addCSSClassName(tags[i], style);
      } else {
        this.m_utils.removeCSSClassName(tags[i], style);
      }
    }
  } else {
    style = this.getMappedStyle('borderVerticalNone');
    lastFunction = this._isLastColumn.bind(this);
    this._adjustHeaderBordersAlongAxis(this.m_colHeader, lastFunction, this.m_endColHeader,
      widthCheck, this.m_columnHeaderScrollbarSpacer, style,
      this.m_columnHeaderLevelCount, this.m_startColHeader);
    this._adjustHeaderBordersAlongAxis(this.m_colEndHeader, lastFunction, this.m_endColEndHeader,
      widthCheck, this.m_bottomCorner, style,
      this.m_columnEndHeaderLevelCount, this.m_startColEndHeader);
  }

  bw = null;

  if (heightCheck && this.m_endColEndHeader >= 0) {
    this.m_addBorderBottom = true;
    bw = true;
  } else if (this.m_addBorderBottom === true) {
    bw = false;
  }

  if (bw != null) {
    style = this.getMappedStyle('borderHorizontalSmall');
    if (this.m_rowHeaderScrollbarSpacer != null) {
      if (bw) {
        this.m_utils.addCSSClassName(this.m_rowHeaderScrollbarSpacer, style);
      } else {
        this.m_utils.removeCSSClassName(this.m_rowHeaderScrollbarSpacer, style);
      }
    }
    if (this.m_bottomCorner != null) {
      if (bw) {
        this.m_utils.addCSSClassName(this.m_bottomCorner, style);
      } else {
        this.m_utils.removeCSSClassName(this.m_bottomCorner, style);
      }
    }
    tags = this.m_colEndHeader.firstChild.childNodes;
    for (i = 0; i < tags.length; i++) {
      if (bw) {
        this.m_utils.addCSSClassName(tags[i], style);
      } else {
        this.m_utils.removeCSSClassName(tags[i], style);
      }
    }
  } else {
    style = this.getMappedStyle('borderHorizontalNone');
    lastFunction = this._isLastRow.bind(this);
    this._adjustHeaderBordersAlongAxis(this.m_rowHeader, lastFunction, this.m_endRowHeader,
      heightCheck, this.m_rowHeaderScrollbarSpacer, style,
      this.m_rowHeaderLevelCount, this.m_startRowHeader);
    this._adjustHeaderBordersAlongAxis(this.m_rowEndHeader, lastFunction, this.m_endRowEndHeader,
      heightCheck, this.m_bottomCorner, style,
      this.m_rowEndHeaderLevelCount, this.m_startRowEndHeader);
  }
};

/**
 * @private
 */
DvtDataGrid.prototype._isHeaderLabelCollision = function () {
  return this.m_headerLabels.column[this.m_columnHeaderLevelCount - 1] &&
    this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1];
};

/**
 * Build the corners of the grid. If they exist, removes them and rebuilds them.
 * @private
 */
DvtDataGrid.prototype.buildCorners = function () {
  var scrollbarSize = this.m_utils.getScrollbarSize();
  var width = this.getWidth();
  var height = this.getHeight();
  var colHeaderHeight = this.getColumnHeaderHeight();
  var colHeaderWidth = this.getElementWidth(this.m_colHeader);
  var colEndHeaderHeight = this.getColumnEndHeaderHeight();
  var rowHeaderWidth = this.getRowHeaderWidth();
  var rowEndHeaderWidth = this.getRowEndHeaderWidth();
  var rowHeaderHeight = this.getElementHeight(this.m_rowHeader);
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';
  var corner;
  var bottomCorner;
  var rowHeaderScrollbarSpacer;
  var columnHeaderScrollbarSpacer;
  var label;
  var i;

  // rather than removing and appending the nodes every time just adjust the live ones

  if (this.m_endRowHeader !== -1 && this.m_endColHeader !== -1) {
    // render corner
    if (this.m_corner != null) {
      corner = this.m_corner;
    } else {
      corner = document.createElement('div');
      corner.id = this.createSubId('corner');
      corner.className = this.getMappedStyle('topcorner');
    }

    this.setElementWidth(corner, rowHeaderWidth);
    this.setElementHeight(corner, colHeaderHeight);

    if (this.m_corner == null) {
      if (this.m_utils.isTouchDevice()) {
        corner.addEventListener('touchstart', this.handleCornerMouseDown.bind(this), { passive: true });
      } else {
        corner.addEventListener('mousedown', this.handleCornerMouseDown.bind(this), false);
        corner.addEventListener('mouseover', this.handleCornerMouseOver.bind(this), false);
        corner.addEventListener('mouseout', this.handleCornerMouseOut.bind(this), false);
      }
      corner.addEventListener('click', this.handleCornerClick.bind(this), false);

      this.m_root.appendChild(corner); // @HTMLUpdateOK
      this.m_corner = corner;

      var dimension = this.m_headerLabels.column.length ?
        this.m_columnHeaderLevelHeights[this.m_columnHeaderLevelCount - 1] : this.m_colHeaderHeight;
      for (i = 0; i < this.m_headerLabels.row.length; i++) {
        label = this.m_headerLabels.row[i];
        if (label != null) {
          this.setElementHeight(label, dimension);
          corner.appendChild(label); // @HTMLUpdateOK
        }
      }

      dimension = this.m_headerLabels.row.length ?
        this.m_rowHeaderLevelWidths[this.m_rowHeaderLevelCount - 1] : this.m_rowHeaderWidth;
      for (i = 0; i < this.m_headerLabels.column.length; i++) {
        label = this.m_headerLabels.column[i];
        if (label != null) {
          this.setElementWidth(label, dimension);
          corner.appendChild(label); // @HTMLUpdateOK
        }
      }
      this.m_subtreeAttachedCallback(corner);

      if (this._isHeaderLabelCollision()) {
        var item = this._getHeaderByIndex(this.m_startColHeader,
          this.m_columnHeaderLevelCount - 1,
          this.m_colHeader, this.m_columnHeaderLevelCount,
          this.m_startColHeader);
        var h = this.getElementHeight(item);
        this.m_colHeaderHeight += h;
        this.m_columnHeaderLevelHeights[this.m_columnHeaderLevelCount - 1] += h;

        this.resizeColumnHeightsAndShift(h, this.m_columnHeaderLevelCount - 1, false);
        this.setElementHeight(this.m_colHeader, this.m_colHeaderHeight);

        this.manageResizeScrollbars();
        return;
      }
    } else {
      let isTouchDevice = this.m_utils.isTouchDevice();
      if (isTouchDevice) {
        corner.addEventListener('touchstart', this.handleHeaderLabelMouseDown.bind(this), false);
        corner.addEventListener('touchmove', this.handleHeaderLabelMouseMove.bind(this), false);
      } else {
        corner.addEventListener('mousedown', this.handleHeaderLabelMouseDown.bind(this), false);
        corner.addEventListener('mousemove', this.handleHeaderLabelMouseMove.bind(this), false);
      }
    }
  } else {
    this.m_headerLabels.row = [];
    this.m_headerLabels.column = [];
  }

  if (this.m_corner != null && corner == null) {
    this.m_root.removeChild(this.m_corner);
    this.m_corner = null;
  }

  // no bottom left corner if there are no row headers
  if (this.m_endRowHeader !== -1) {
    if (this.m_hasHorizontalScroller || this.m_endColEndHeader !== -1) {
      if (this.m_rowHeaderScrollbarSpacer != null) {
        rowHeaderScrollbarSpacer = this.m_rowHeaderScrollbarSpacer;
      } else {
        rowHeaderScrollbarSpacer = document.createElement('div');
        rowHeaderScrollbarSpacer.id = this.createSubId('rhSbSpacer');
        rowHeaderScrollbarSpacer.className = this.getMappedStyle('rowheaderspacer');
      }

      this.setElementDir(rowHeaderScrollbarSpacer, (rowHeaderHeight + colHeaderHeight), 'top');
      this.setElementDir(rowHeaderScrollbarSpacer, 0, dir);
      this.setElementWidth(rowHeaderScrollbarSpacer, this.m_rowHeaderWidth);
      if (this.m_endColEndHeader !== -1) {
        this.setElementHeight(rowHeaderScrollbarSpacer,
          colEndHeaderHeight +
                              (this.m_hasHorizontalScroller ? scrollbarSize : 0));
      } else {
        this.setElementHeight(rowHeaderScrollbarSpacer,
          height - rowHeaderHeight - colHeaderHeight);
      }

      if (this.m_rowHeaderScrollbarSpacer == null) {
        if (this.m_utils.isTouchDevice()) {
          rowHeaderScrollbarSpacer.addEventListener('touchstart', this.handleCornerMouseDown.bind(this), { passive: true });
        } else {
          rowHeaderScrollbarSpacer.addEventListener('mousedown',
            this.handleCornerMouseDown.bind(this), false);
          rowHeaderScrollbarSpacer.addEventListener('mouseover',
            this.handleCornerMouseOver.bind(this), false);
          rowHeaderScrollbarSpacer.addEventListener('mouseout',
            this.handleCornerMouseOut.bind(this), false);
          if (this.isResizeEnabled() !== 'disable') {
            rowHeaderScrollbarSpacer.addEventListener('mousedown', this.handleHeaderLabelMouseDown.bind(this), false);
            rowHeaderScrollbarSpacer.addEventListener('mousemove', this.handleHeaderLabelMouseMove.bind(this), false);
          }
        }

        this.m_root.appendChild(rowHeaderScrollbarSpacer); // @HTMLUpdateOK
        this.m_rowHeaderScrollbarSpacer = rowHeaderScrollbarSpacer;

        for (i = 0; i < this.m_headerLabels.columnEnd.length; i++) {
          label = this.m_headerLabels.columnEnd[i];
          if (label != null) {
            rowHeaderScrollbarSpacer.appendChild(label); // @HTMLUpdateOK
          }
        }

        this.m_subtreeAttachedCallback(rowHeaderScrollbarSpacer);
      }
    } else {
      if (this.m_rowHeaderScrollbarSpacer != null) {
        this.m_root.removeChild(this.m_rowHeaderScrollbarSpacer);
      }
      this.m_rowHeaderScrollbarSpacer = null;
      this.m_headerLabels.columnEnd = [];
    }
  }

  // no top right corner if there are no col headers
  if (this.m_endColHeader !== -1) {
    if (this.m_hasVerticalScroller || this.m_endRowEndHeader !== -1) {
      if (this.m_columnHeaderScrollbarSpacer != null) {
        columnHeaderScrollbarSpacer = this.m_columnHeaderScrollbarSpacer;
      } else {
        columnHeaderScrollbarSpacer = document.createElement('div');
        columnHeaderScrollbarSpacer.id = this.createSubId('chSbSpacer');
        columnHeaderScrollbarSpacer.className = this.getMappedStyle('colheaderspacer');
      }

      this.setElementDir(columnHeaderScrollbarSpacer, (rowHeaderWidth + colHeaderWidth), dir);
      this.setElementDir(columnHeaderScrollbarSpacer, 0, 'top');
      if (this.m_endRowEndHeader !== -1) {
        this.setElementWidth(columnHeaderScrollbarSpacer,
          rowEndHeaderWidth + (this.m_hasVerticalScroller ? scrollbarSize : 0));
      } else {
        this.setElementWidth(columnHeaderScrollbarSpacer, width - colHeaderWidth - rowHeaderWidth);
      }
      this.setElementHeight(columnHeaderScrollbarSpacer, this.m_colHeaderHeight);

      if (this.m_columnHeaderScrollbarSpacer == null) {
        if (this.m_utils.isTouchDevice()) {
          columnHeaderScrollbarSpacer.addEventListener('touchstart', this.handleCornerMouseDown.bind(this), { passive: true });
        } else {
          columnHeaderScrollbarSpacer.addEventListener('mousedown',
            this.handleCornerMouseDown.bind(this),
            false);
          columnHeaderScrollbarSpacer.addEventListener('mouseover',
            this.handleCornerMouseOver.bind(this),
            false);
          columnHeaderScrollbarSpacer.addEventListener('mouseout',
            this.handleCornerMouseOut.bind(this),
            false);
          if (this.isResizeEnabled() !== 'disable') {
            columnHeaderScrollbarSpacer.addEventListener('mousedown', this.handleHeaderLabelMouseDown.bind(this), false);
            columnHeaderScrollbarSpacer.addEventListener('mousemove', this.handleHeaderLabelMouseMove.bind(this), false);
          }
        }

        this.m_root.appendChild(columnHeaderScrollbarSpacer); // @HTMLUpdateOK
        this.m_columnHeaderScrollbarSpacer = columnHeaderScrollbarSpacer;

        for (i = 0; i < this.m_headerLabels.rowEnd.length; i++) {
          label = this.m_headerLabels.rowEnd[i];
          if (label != null) {
            columnHeaderScrollbarSpacer.appendChild(label); // @HTMLUpdateOK
          }
        }

        this.m_subtreeAttachedCallback(columnHeaderScrollbarSpacer);
      }
    } else {
      if (this.m_columnHeaderScrollbarSpacer != null) {
        this.m_root.removeChild(this.m_columnHeaderScrollbarSpacer);
      }
      this.m_columnHeaderScrollbarSpacer = null;
      this.m_headerLabels.rowEnd = [];
    }
  }

  if ((this.m_hasHorizontalScroller && this.m_hasVerticalScroller) ||
      (this.m_hasVerticalScroller && this.m_endColEndHeader !== -1) ||
      (this.m_hasHorizontalScroller && this.m_endRowEndHeader !== -1) ||
      (this.m_endRowEndHeader !== -1 && this.m_endColEndHeader !== -1)) {
    // render bottom corner (for both scrollbars) if needed
    if (this.m_bottomCorner != null) {
      bottomCorner = this.m_bottomCorner;
    } else {
      bottomCorner = document.createElement('div');
      bottomCorner.id = this.createSubId('bcorner');
      bottomCorner.className = this.getMappedStyle('bottomcorner');
    }

    this.setElementDir(bottomCorner, (rowHeaderHeight + colHeaderHeight), 'top');
    this.setElementDir(bottomCorner, (rowHeaderWidth + colHeaderWidth), dir);
    if (this.m_endRowEndHeader !== -1) {
      this.setElementWidth(bottomCorner,
        rowEndHeaderWidth + (this.m_hasVerticalScroller ? scrollbarSize : 0));
    } else {
      this.setElementWidth(bottomCorner, width - colHeaderWidth - rowHeaderWidth);
    }

    if (this.m_endColEndHeader !== -1) {
      this.setElementHeight(bottomCorner,
        colEndHeaderHeight +
                            (this.m_hasHorizontalScroller ? scrollbarSize : 0));
    } else {
      this.setElementHeight(bottomCorner, height - rowHeaderHeight - colHeaderHeight);
    }

    if (this.m_bottomCorner == null) {
      this.m_root.appendChild(bottomCorner); // @HTMLUpdateOK
      this.m_bottomCorner = bottomCorner;
    }
  }
  // remove bottom corner on resize if not neccessary
  if (this.m_bottomCorner != null && bottomCorner == null) {
    this.m_root.removeChild(this.m_bottomCorner);
    this.m_bottomCorner = null;
  }
};

/**
 * Move to a desired scoll position object
 * @private
 */
DvtDataGrid.prototype._updateScrollPosition = function (scrollPositionObject) {
  this._scrollToScrollPositionObject(scrollPositionObject);
};

/**
 * Set the scrollPosition attribute on the grid
 * @private
 */
DvtDataGrid.prototype._setScrollPosition = function () {
  this.m_setOptionCallback(
    'scrollPosition',
    this._createScrollPositionObject(this.m_currentScrollLeft, this.m_currentScrollTop), {
      _context: {
        writeback: true,
        internalSet: true
      }
    });
};

/**
 * Set the scrollPosition minus the keys
 * @private
 */
DvtDataGrid.prototype._clearScrollPositionKeys = function () {
  var newPos = this.m_options.getScrollPosition();
  newPos.rowKey = undefined;
  newPos.columnKey = undefined;
  // do not writeback it will be updated once the sync has completed
  this.m_setOptionCallback('scrollPosition', newPos, { _context: { internalSet: true } });
};

/**
 * Create a scroll position object from x,y coordinates
 * @private
 */
DvtDataGrid.prototype._createScrollPositionObject = function (x, y) {
  var scrollPositionObject = { x: x, y: y };
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';

  var cell = this._getCellAtPixel(x, y);
  if (cell != null) {
    scrollPositionObject.rowIndex = this._getIndex(cell, 'row');
    scrollPositionObject.columnIndex = this._getIndex(cell, 'column');
    scrollPositionObject.rowKey = this._getKey(cell, 'row');
    scrollPositionObject.columnKey = this._getKey(cell, 'column');
    scrollPositionObject.offsetX = x - this.getElementDir(cell, dir);
    scrollPositionObject.offsetY = y - this.getElementDir(cell, 'top');
  } else {
    var rowHeader = this._getHeaderAtPixel(y, 'row');
    if (rowHeader != null) {
      scrollPositionObject.rowIndex = this._getIndex(rowHeader);
      scrollPositionObject.rowKey = this._getKey(rowHeader);
      scrollPositionObject.offsetY = y - this.getElementDir(rowHeader, 'top');
    }

    var columnHeader = this._getHeaderAtPixel(x, 'column');
    if (columnHeader != null) {
      scrollPositionObject.columnIndex = this._getIndex(columnHeader);
      scrollPositionObject.columnKey = this._getKey(columnHeader);
      scrollPositionObject.offsetX = x - this.getElementDir(columnHeader, dir);
    }
  }

  return scrollPositionObject;
};

/**
 * Get a cell at x,y coordinates
 * @private
 */
DvtDataGrid.prototype._getCellAtPixel = function (x, y) {
  var cells = this.m_databody.firstChild.childNodes;
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';

  for (var i = 0; i < cells.length; i++) {
    var cell = cells[i];
    var cellLeft = this.getElementDir(cell, dir);
    var cellRight = cellLeft + this.getElementWidth(cell);

    if (cellLeft <= x && x < cellRight) {
      var cellTop = this.getElementDir(cell, 'top');
      var cellBottom = cellTop + this.getElementHeight(cell);
      if (cellTop <= y && y < cellBottom) {
        return cell;
      }
    }
  }
  return null;
};

DvtDataGrid.prototype._getAxisInnerMostHeaders = function (axis) {
  var className = this.getMappedStyle('headercell');
  var root;
  var levelCount;

  switch (axis) {
    case 'row':
      root = this.m_rowHeader;
      levelCount = this.m_rowHeaderLevelCount;
      break;
    case 'column':
      root = this.m_colHeader;
      levelCount = this.m_columnHeaderLevelCount;
      break;
    case 'rowEnd':
      root = this.m_rowEndHeader;
      levelCount = this.m_rowEndHeaderLevelCount;
      break;
    case 'columnEnd':
      root = this.m_colEndHeader;
      levelCount = this.m_columnEndHeaderLevelCount;
      break;
    default:
      break;
  }

  var returnArr = [];
  if (root) {
    var headers = root.getElementsByClassName(className);
    for (var i = 0; i < headers.length; i++) {
      var header = headers[i];
      var context = header[this.getResources().getMappedAttribute('context')];
      if (context.level + context.depth === levelCount) {
        returnArr.push(header);
      }
    }
  }
  return returnArr;
};

/**
 * Get a header at a given coordinate and axis
 * @private
 */
DvtDataGrid.prototype._getHeaderAtPixel = function (pixel, axis) {
  var self = this;
  var headers;
  var endheaders;
  var dir;
  var dimension;

  headers = this._getAxisInnerMostHeaders(axis);
  endheaders = this._getAxisInnerMostHeaders(axis + 'End');

  if (axis === 'row') {
    dir = 'top';
    dimension = 'height';
  } else if (axis === 'column') {
    dir = this.getResources().isRTLMode() ? 'right' : 'left';
    dimension = 'width';
  }

  function loop(elements) {
    for (var i = 0; i < elements.length; i++) {
      var header = elements[i];
      var start = self.getElementDir(header, dir);
      var end = start + self.getElementDir(header, dimension);

      if (start <= pixel && pixel < end) {
        return header;
      }
    }
    return undefined;
  }

  var header = loop(headers);
  if (header == null) {
    header = loop(endheaders);
  }

  return header;
};

/**
 * See if we have reached our desired scroll position
 * @private
 */
DvtDataGrid.prototype._checkScrollPosition = function () {
  if (this.m_desiredScrollPositionObject != null) {
    this._scrollToScrollPositionObject(this.m_desiredScrollPositionObject);
  } else {
    this._setScrollPosition();
  }
};

/**
 * See if the row/column keys are already fetched
 * @private
 */
DvtDataGrid.prototype._areKeysLocallyAvailable = function (rowKey, columnKey) {
  var isKeyAvailable = true;
  if (rowKey) {
    if (this._getCellOrHeaderByKey(rowKey, 'row') == null) {
      isKeyAvailable = false;
    }
  }

  if (columnKey) {
    if (this._getCellOrHeaderByKey(columnKey, 'column') == null) {
      isKeyAvailable = false;
    }
  }

  return isKeyAvailable;
};

/**
 * Call initiate scroll to scroll to a scroll position object
 */
DvtDataGrid.prototype._scrollToScrollPositionObject = function (scrollPositionObject) {
  var x = scrollPositionObject.x;
  var y = scrollPositionObject.y;
  var rowIndex = scrollPositionObject.rowIndex;
  var columnIndex = scrollPositionObject.columnIndex;
  var rowKey = scrollPositionObject.rowKey;
  var columnKey = scrollPositionObject.columnKey;
  var offsetX = scrollPositionObject.offsetX ? scrollPositionObject.offsetX : 0;
  var offsetY = scrollPositionObject.offsetY ? scrollPositionObject.offsetY : 0;
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';

  var scrollToKey = this.m_options.getScrollToKey();
  // ignore the value if key is specified and scrollByKey behavior is 'never'
  if (scrollToKey === 'never' && (rowKey || columnKey)) {
    return;
  }

  // also ignore the value if it is a DataProvider that is not capable of returning results immediately
  // and (one of the) keys are not already fetched yet
  if (scrollToKey !== 'always') {
    // for scrollToKey values 'auto' or 'capability'
    var ds = this.getDataSource();
    if (ds instanceof DataProviderDataGridDataSource) {
      if (!this._areKeysLocallyAvailable(rowKey, columnKey)) {
        var capability = this.m_options.getProperty('data').getCapability('fetchFirst');
        if (capability == null || capability.iterationSpeed !== 'immediate') {
          return;
        }
      }
    }
  }

  var self = this;
  var indexFromKeyPromise = this._getIndexFromKeyPromise(rowKey, columnKey);

  indexFromKeyPromise.then(function (returnObj) {
    // scrollTop and s
    var newScrollX = Math.floor(self._getPositionEstimate('column', dir, columnKey,
      returnObj.columnIndexFromKey,
      columnIndex,
      x, offsetX,
      self.m_currentScrollLeft, self._getMaxRightPixel(),
      self.m_avgColWidth));
    var newScrollY = Math.floor(self._getPositionEstimate('row', 'top', rowKey,
      returnObj.rowIndexFromKey,
      rowIndex,
      y, offsetY,
      self.m_currentScrollTop, self._getMaxBottomPixel(),
      self.m_avgRowHeight));

    // make sure scroll is not the same, and that we aren't trying to scroll
    // out of bounds if we are at the boundry.
    if ((newScrollX !== self.m_currentScrollLeft &&
      (self.m_currentScrollLeft !== self.m_scrollWidth ||
        newScrollX < self.m_currentScrollLeft)) ||
      (newScrollY !== self.m_currentScrollTop &&
      (self.m_currentScrollTop !== self.m_scrollHeight ||
        newScrollY < self.m_currentScrollTop))) {
      if (self.m_desiredScrollPositionObject == null) {
        self._signalTaskStart('begin scrolling to new desired location');
      }
      self.m_desiredScrollPositionObject = scrollPositionObject;
      self._setScrollPositionTimeout();
      self._initiateScroll(newScrollX, newScrollY);
    } else {
      if (self.m_desiredScrollPositionObject != null) {
        self._signalTaskEnd('reached desired location');
      }
      self.m_desiredScrollPositionObject = null;
      self._setScrollPosition();
    }
  });
};

// These methods exist for issues with pixel perfect scrolling when zoomed
// there isn't a great way to guarantee cross browser that the pixel we want
// to scroll to is actually a possible value as scrollTop/Left in the browser when zoomed
// That means when scrollTo is called there's a chance that a scroll event is
// 1) not fired at all becauxse it is not a possible value
// 2) fired but ends up infinite looping trying to get to that impossible value
// Timeout is 300ms which is long, but this should be a rare catch, and I noticed in chrome
// that it can take up to 300ms or longer between initiateScroll and the handler
// if it takes longer end result is scrollPosition will come up short of its goal
// also touch devices don't allow the zoom issues because only zoom in

/**
 * @private
 */
DvtDataGrid.prototype._setScrollPositionTimeout = function () {
  if (!this.m_utils.isTouchDevice()) {
    this.pendingScrollTimeout = setTimeout(function () { // @HTMLUpdateOK
      if (this.m_desiredScrollPositionObject != null) {
        this._signalTaskEnd('reached desired location');
      }
      this.m_desiredScrollPositionObject = null;
      this._setScrollPosition();
    }.bind(this), 300);
  }
};

/**
 * @private
 */
DvtDataGrid.prototype._clearScrollPositionTimeout = function () {
  if (this.pendingScrollTimeout != null) {
    clearTimeout(this.pendingScrollTimeout);
    this.pendingScrollTimeout = null;
  }
};

/**
 * Estimate the location of the scroll based on key/index/x,y following the jsdoc guidelines
 * @private
 */
DvtDataGrid.prototype._getPositionEstimate =
  function (axis, dir, key, indexFromKey, index, pos, offset, current, max, average) {
    var newScrollPos;
    var isHighWatermark = this._isHighWatermarkScrolling();
    var element;

    // indexFromKey === -1 means that the key is definitely not in the data source
    // indexFromKey === null means that the key is possibly not in the data source
    if (key != null && indexFromKey !== -1) {
      element = this._getCellOrHeaderByKey(key, axis);
      if (element != null) {
        newScrollPos = this.getElementDir(element, dir) + offset;
      } else if (isHighWatermark) {
        newScrollPos = max;
      } else if (indexFromKey != null) {
        newScrollPos = (average * indexFromKey) + offset;
      }

      if (newScrollPos != null) {
        return newScrollPos;
      }

      // if key specified and virtual scrolling but no index found, switch to index check
    }

    if (index != null) {
      element = this._getCellOrHeaderByIndex(index, axis);
      if (element != null) {
        newScrollPos = this.getElementDir(element, dir) + offset;
      } else if (isHighWatermark) {
        newScrollPos = max;
      } else {
        newScrollPos = average * index;
      }
    } else if (pos != null) {
      newScrollPos = pos;
    } else {
      newScrollPos = current;
    }

    return newScrollPos;
  };

/**
 * Determine if horizontal scrollbar is needed
 * @param {number} expectedWidth - databody width
 * @return {boolean} true if horizontal scrollbar required
 */
DvtDataGrid.prototype.isDatabodyHorizontalScrollbarRequired = function (expectedWidth) {
  var databody = this.m_databody;
  var scroller = databody.firstChild;
  if (this.getElementWidth(scroller) > expectedWidth) {
    return true;
  }
  return false;
};

/**
 * Determine if vertical scrollbar is needed
 * @param {number} expectedHeight - databody height
 * @return {boolean} true if vertical scrollbar required
 * @private
 */
DvtDataGrid.prototype.isDatabodyVerticalScrollbarRequired = function (expectedHeight) {
  var databody = this.m_databody;
  var scroller = databody.firstChild;
  if (this.getElementHeight(scroller) > expectedHeight) {
    return true;
  }
  return false;
};

/**
 * todo: merge with buildAccInfo, we just need one status role div.
 * Build a status bar div
 * @return {Element} the root of the status bar
 * @private
 */
DvtDataGrid.prototype.buildStatus = function () {
  var icon = document.createElement('div');
  icon.className = this.getMappedStyle('loadingicon');

  var root = document.createElement('div');
  root.id = this.createSubId('status');
  root.className = this.getMappedStyle('status');
  root.setAttribute('role', 'status');
  root.appendChild(icon);

  return root;
};

/**
 * Build the offscreen div used by screenreader for action such as sort
 * @return {Element} the root of the accessibility info div
 */
DvtDataGrid.prototype.buildAccInfo = function () {
  var root = document.createElement('div');
  root.id = this.createSubId('info');
  root.className = this.getMappedStyle('info');
  root.setAttribute('role', 'status');

  return root;
};

/**
 * Build the offscreen div used by screenreader for summary description
 * @return {Element} the root of the accessibility summary div
 */
DvtDataGrid.prototype.buildAccSummary = function () {
  var root = document.createElement('div');
  root.id = this.createSubId('summary');
  root.className = this.getMappedStyle('info');

  return root;
};

/**
 * Build the offscreen div used by screenreader for state information
 * @return {Element} the root of the accessibility state info div
 */
DvtDataGrid.prototype.buildStateInfo = function () {
  var root = document.createElement('div');
  root.id = this.createSubId('state');
  root.className = this.getMappedStyle('info');

  return root;
};

/**
 * Build the offscreen div used by screenreader for cell context information
 * @return {Element} the root of the accessibility context info div
 */
DvtDataGrid.prototype.buildContextInfo = function () {
  var root = document.createElement('div');
  root.id = this.createSubId('context');
  root.className = this.getMappedStyle('info');

  return root;
};

/**
 * Build the offscreen div used by screenreader used for reading current cell context information
 * @return {Element} the root of the accessibility current cell context info div
 */
DvtDataGrid.prototype.buildPlaceHolder = function () {
  var root = document.createElement('div');
  root.id = this.createSubId('placeHolder');
  root.className = this.getMappedStyle('info');

  return root;
};

/**
 * Sets the text on the offscreen div.  The text contains a summary text describing the grid
 * including structure information
 * @private
 */
DvtDataGrid.prototype.populateAccInfo = function () {
  var summary = this.getResources().getTranslatedText('accessibleSummaryExact', {
    rownum: (this.m_endRow + 1),
    colnum: (this.m_endCol + 1)
  });

  // if it's hierarchical, then include specific accessible info about what's expanded
  if (this.getDataSource().getExpandedKeys) {
    var summaryExpanded = this.getResources().getTranslatedText('accessibleSummaryExpanded', {
      num: this.getDataSource().getExpandedKeys().length
    });
    summary = summary + '. ' + summaryExpanded;
  }

  // add instruction text
  summary += '. ';

  // set the summary text on the screen reader div
  this.m_accSummary.textContent = summary;
};

/**
 * Implements Accessible interface.
 * Sets accessible information on the DataGrid.
 * This is currently used by the Row Expander to alert screenreader of such
 * information as depth, expanded state, index etc
 * @param {Object} context an object containing attribute context or state to set m_accessibleContext/state
 */
DvtDataGrid.prototype.SetAccessibleContext = function (context) {
  if (context != null) {
    // got row context info
    if (context.context != null) {
      // save it for updateContextInfo to consume later
      this.m_accessibleContext = context.context;
    }

    // got disclosure state info
    if (context.state != null) {
      this.m_stateInfo.textContent = context.state;
    }

    // got ancestors info
    if (context.ancestors != null && this._isDatabodyCellActive()) {
      var label = '';
      var ancestors = context.ancestors;
      var col = this.m_active.indexes.column;
      if (col != null && col >= 0) {
        // constructs the appropriate parent context info text
        for (var i = 0; i < ancestors.length; i++) {
          if (i > 0) {
            label = label.concat(', ');
          }
          var parent = ancestors[i];
          var rowCells = this._getAxisCellsByKey(parent.key, 'row');
          if (rowCells != null) {
            var cell = rowCells[0];
            // we are just going to extract any text content (or find first aria-label if null?)
            var text = cell.textContent;
            // remove any carriage return, tab etc.
            if (text != null) {
              text = text.replace(/\n|<br\s*\/?>/gi, '').trim();
            } else {
              text = '';
            }
            label = label.concat(parent.label).concat(' ').concat(text);
          }
        }
      }

      // prepend to existing context info
      this.m_accessibleContext = label.concat(', ').concat(this.m_accessibleContext);
    }
  }
};

/**
 * Sets the accessibility state info text
 * @param {Array} items the message key
 * @private
 */
DvtDataGrid.prototype._updateStateInfo = function (items) {
  // allows pause after the data is read out before the state
  var text = '. ';
  for (var i = 0; i < items.length; i++) {
    var state = this.getResources().getTranslatedText(items[i].key, items[i].args);
    if (state != null) {
      // for the original period we will just add text otherwise need a comma
      text = text.length === 2 ? text + state : text + ', ' + state;
    }
  }

  // end state with a period
  // 2 for the original period
  text = text.length === 2 ? text : text + '. ';

  this.m_stateInfo.textContent = text;
};

/**
 * Sets the accessibility context info text
 * @param {Object} context the context info about the cell
 * @param {number=} context.row the row index
 * @param {number=} context.column the column index
 * @param {number=} context.level the level of the header if there is one
 * @param {number=} context.rowHeader the rowHeader index
 * @param {number=} context.columnHeader the rowHeader index
 * @param {string=} skip whether to skip row or column
 * @private
 */
DvtDataGrid.prototype._updateContextInfo = function (context, skip) {
  var row;
  var column;

  if (context.indexes) {
    row = context.indexes.row;
    column = context.indexes.column;
  }

  var level = context.level;
  var rowHeader = context.rowHeader;
  var rowEndHeader = context.rowEndHeader;
  var columnHeader = context.columnHeader;
  var columnEndHeader = context.columnEndHeader;
  var rowHeaderLabel = context.rowHeaderLabel;
  var rowEndHeaderLabel = context.rowEndHeaderLabel;
  var columnHeaderLabel = context.columnHeaderLabel;
  var columnEndHeaderLabel = context.columnEndHeaderLabel;
  var info = '';
  var endContextText = '. ';

  // row context.  Skip if there is an outstanding accessible row context
  if (this.m_accessibleContext == null && !isNaN(row) && skip !== 'row') {
    info = this._updateAccessibleInfoString(info, 'accessibleRowContext', { index: row + 1 });
  }

  // column context
  if (!isNaN(column) && skip !== 'column') {
    info = this._updateAccessibleInfoString(info, 'accessibleColumnContext', { index: column + 1 });
  }

  // rowHeader context
  if (!isNaN(rowHeader)) {
    info = this._updateAccessibleInfoString(info, 'accessibleRowHeaderContext', { index: rowHeader + 1 });
  }

  // columnHeader context
  if (!isNaN(columnHeader)) {
    info = this._updateAccessibleInfoString(info, 'accessibleColumnHeaderContext', { index: columnHeader + 1 });
  }

  // rowEndHeader context
  if (!isNaN(rowEndHeader)) {
    info = this._updateAccessibleInfoString(info, 'accessibleRowEndHeaderContext', { index: rowEndHeader + 1 });
  }

  // columnEndHeader context
  if (!isNaN(columnEndHeader)) {
    info = this._updateAccessibleInfoString(info, 'accessibleColumnEndHeaderContext', { index: columnEndHeader + 1 });
  }

  // rowHeaderLabel context
  if (!isNaN(rowHeaderLabel)) {
    info = this._updateAccessibleInfoString(info, 'accessibleRowHeaderLabelContext', { level: rowHeaderLabel + 1 });
  }

  // columnHeaderLabel context
  if (!isNaN(columnHeaderLabel)) {
    info = this._updateAccessibleInfoString(info, 'accessibleColumnHeaderLabelContext', { level: columnHeaderLabel + 1 });
  }

  // rowEndHeaderLabel context
  if (!isNaN(rowEndHeaderLabel)) {
    info = this._updateAccessibleInfoString(info, 'accessibleRowEndHeaderLabelContext', { level: rowEndHeaderLabel + 1 });
  }

  // columnEndHeaderLabel context
  if (!isNaN(columnEndHeaderLabel)) {
    info = this._updateAccessibleInfoString(info, 'accessibleColumnEndHeaderLabelContext', { level: columnEndHeaderLabel + 1 });
  }

  // level context
  if (!isNaN(level)) {
    info = this._updateAccessibleInfoString(info, 'accessibleLevelContext', { level: level + 1 });
  }

  // Put a period at the end of the context readout
  info = info.length === 0 ? info : info + endContextText;

  // merge with accesssible context (from row expander)
  if (this.m_accessibleContext != null) {
    info += this.m_accessibleContext;
    // reset
    this.m_accessibleContext = null;
  }

  this.m_contextInfo.textContent = info;
};

/**
 * @private
 */
DvtDataGrid.prototype._updateAccessibleInfoString = function (info, translation,
  translationParams) {
  var spacerText = ', ';
  var text = this.getResources().getTranslatedText(translation, translationParams);
  if (text != null) {
    return info.length === 0 ? text : info + spacerText + text;
  }
  return info;
};

/**
 * Determine whether the row/column count is unknown.
 * @param {string} axis the row/column axis
 * @return {boolean|undefined} true if the count for the axis is unknown, false otherwise
 * @private
 */
DvtDataGrid.prototype._isCountUnknown = function (axis) {
  var datasource = this.getDataSource();
  if (axis === 'row' || axis === 'rowEnd') {
    var rowPrecision = datasource.getCountPrecision('row');
    var rowCount = datasource.getCount('row');
    if (rowPrecision === 'estimate' || rowCount < 0) {
      this.m_isEstimateRowCount = true;
    } else {
      this.m_isEstimateRowCount = false;
    }
    return this.m_isEstimateRowCount;
  } else if (axis === 'column' || axis === 'columnEnd') {
    var colPrecision = datasource.getCountPrecision('column');
    var colCount = datasource.getCount('column');
    if (colPrecision === 'estimate' || colCount < 0) {
      this.m_isEstimateColumnCount = true;
    } else {
      this.m_isEstimateColumnCount = false;
    }
    return this.m_isEstimateColumnCount;
  }

  // unrecognize axis, just assume the count is known
  return false;
};

/**
 * Convenient method which returns true if row count is unknown or high-water mark scrolling is used.
 * @param {string} axis the row/column axis
 * @return {boolean} true if count is unknown or high-water mark scrolling is used, false otherwise.
 * @private
 */
DvtDataGrid.prototype._isCountUnknownOrHighwatermark = function (axis) {
  return (this._isCountUnknown(axis) || this._isHighWatermarkScrolling());
};

/**
 * Set display to none
 * @param {Element} root
 * @private
 */
DvtDataGrid.prototype._hideHeader = function (root) {
  // eslint-disable-next-line no-param-reassign
  root.style.display = 'none';
};

/**
 * Set display
 * @param {Element} root
 * @private
 */
DvtDataGrid.prototype._showHeader = function (root) {
  // eslint-disable-next-line no-param-reassign
  root.style.display = '';
};

/**
 * Build a header div
 * @param {string} axis - 'row' or 'column'
 * @param {string} styleClass - class to set on the header
 * @param {string} endStyleClass - class to set on the end header
 * @return {Object} contains the root and endRoot of the header axis
 */
DvtDataGrid.prototype.buildHeaders = function (axis, styleClass, endStyleClass) {
  var scrollerClassName =
    this.getMappedStyle('scroller') +
      (this.m_utils.isTouchDevice() ? ' ' + this.getMappedStyle('scroller-mobile') : '');

  var root = document.createElement('div');
  root.id = this.createSubId(axis + 'Header');
  root.className = styleClass + ' ' + this.getMappedStyle('header');
  var scroller = document.createElement('div');
  scroller.className = scrollerClassName;
  root.appendChild(scroller); // @HTMLUpdateOK

  var endRoot = document.createElement('div');
  endRoot.id = this.createSubId(axis + 'EndHeader');
  endRoot.className = endStyleClass + ' ' + this.getMappedStyle('endheader');
  var endScroller = document.createElement('div');
  endScroller.className = scrollerClassName;
  endRoot.appendChild(endScroller); // @HTMLUpdateOK

  if (axis === 'column') {
    this.m_colHeader = root;
    this.m_colEndHeader = endRoot;
  } else if (axis === 'row') {
    this.m_rowHeader = root;
    this.m_rowEndHeader = endRoot;
  }

  if (!this._isHighWatermarkScrolling()) {
    var self = this;
    var scrollPosition = this.m_options.getScrollPosition();
    this._getIndexesFromScrollPosition(scrollPosition).then(function (fetchIndexes) {
      var index = fetchIndexes[axis];
      if (axis === 'column') {
        self.m_startColHeader = index;
        self.m_startColEndHeader = index;
      } else if (axis === 'row') {
        self.m_startRowHeader = index;
        self.m_startRowEndHeader = index;
      }
      self.m_fetching[axis] = false;
      self.fetchHeaders(axis, index, root, endRoot, null, null);
    });
    this.m_fetching[axis] = true;
  } else {
    var index = 0;
    this.fetchHeaders(axis, index, root, endRoot, null, null);
  }

  return { root: root, endRoot: endRoot };
};

/**
 * Fetch the headers by calling the fetch headers method on the data source. Pass
 * callbacks for success and error to the data source.
 * @param {string} axis - 'row' or 'column'
 * @param {number} start - index to start fetching at
 * @param {Element|DocumentFragment} header - the root element of the axis header
 * @param {Element|DocumentFragment} endHeader - the root element of the axis endHeader
 * @param {number|null=} fetchSize - number of headers to fetch
 * @param {Object=} callbacks - the optional callbacks to invoke when the fetch success or fail
 * @protected
 */
DvtDataGrid.prototype.fetchHeaders =
  function (axis, start, header, endHeader, fetchSize, callbacks) {
    // check if we are already fetching
    if (this.m_fetching[axis]) {
      return;
    }

    // fetch size could be explicitly specified.  If not, use the calculated one.
    if (fetchSize == null) {
      // eslint-disable-next-line no-param-reassign
      fetchSize = this.getFetchCount(axis, start);
    }

    var headerRange = {
      axis: axis, start: start, count: fetchSize, header: header, endHeader: endHeader
    };

    this.m_fetching[axis] = headerRange;

    var successCallback;
    // check if overriding callbacks are specified
    if (callbacks != null && callbacks.success != null) {
      successCallback = callbacks.success;
    } else {
      successCallback = this.handleHeadersFetchSuccess;
    }

    this.showStatusText();
    // start fetch
    this._signalTaskStart();
    this.getDataSource().fetchHeaders(headerRange, {
      success: successCallback,
      error: this.handleHeadersFetchError
    }, {
      success: this,
      error: this
    });
  };

/**
 * Checks whether header fetch result match the request
 * @param {Object} headerRange the header range for the response
 * @protected
 */
DvtDataGrid.prototype.isHeaderFetchResponseValid = function (headerRange) {
  var axis = headerRange.axis;
  if (this.m_fetching == null) {
    return false;
  }

  // do object reference check, imagine fetching 20 2 consecutive times but
  // the data changed in bewteeen and we accidentally accept the first because
  // the counts are the same
  return (headerRange === this.m_fetching[axis]);
};

/**
 * Checks whether the result is within the current viewport
 * @param {Object} headerRange
 * @private
 */
DvtDataGrid.prototype.isHeaderFetchResponseInViewport = function (headerRange) {
  if (!this.m_initialized) {
    // initial scroll these are not defined so just return true, or if not inited or if no databody
    return true;
  }

  // the goal of this method is to make sure we haven't scrolled further since the last fetch
  // so our request is still valid, we run a massive risk of running loops if our logic is wrong otherwise
  // as in we continue to request the same thing but it is never valid.
  var axis = headerRange.axis;
  var start = headerRange.start;
  var returnVal;

  if (axis === 'row') {
    returnVal = this._getLongScrollStart(this.m_currentScrollTop, this.m_prevScrollTop, axis);
  } else {
    returnVal = this._getLongScrollStart(this.m_currentScrollLeft, this.m_prevScrollLeft, axis);
  }

  // return true if the viewport fits inside the fetched range
  return (returnVal.start === start);
};

/**
 * Handle a successful call to the data source fetchHeaders
 * @param {Object} startResults a headerSet object returned from the data source
 * @param {Object} headerRange {"axis":,"start":,"count":,"header":}
 * @param {Object} endResults a headerSet object returned from the data source
 * @param {boolean} rowInsert if this is triggered by a row insert event
 * @protected
 */
DvtDataGrid.prototype.handleHeadersFetchSuccess =
  function (startResults, headerRange, endResults, rowInsert) {
    var scrollOptions = this.m_options.getScrollPolicyOptions();
    var maxRowCount = scrollOptions ? scrollOptions.maxRowCount : null;
    var maxColumnCount = scrollOptions ? scrollOptions.maxColumnCount : null;

    // validate result matches what we currently asks for
    if (!this.isHeaderFetchResponseValid(headerRange)) {
      // end fetch
      this._signalTaskEnd();
      // not valid, so ignore result
      return;
    }

    var axis = headerRange.axis;

    // checks if the response covers the viewport
    if (this.isLongScroll() && !this.isHeaderFetchResponseInViewport(headerRange)) {
      // clear cells fetching flag
      this.m_fetching[axis] = false;
      // store that the header is invalid for the case when there are no cells
      this.m_headerInvalid = true;
      // end fetch
      this._signalTaskEnd();
      return;
    }

    // remove fetching message
    this.m_fetching[axis] = false;

    var root = headerRange.header;
    var endRoot = headerRange.endHeader;
    var start = headerRange.start;
    var count = this.getDataSource().getCount(axis);

    if (axis === 'column') {
      if (startResults != null) {
        this.buildColumnHeaders(root, startResults, start, count, false, false);
        if (startResults.getCount() < headerRange.count ||
            (maxColumnCount && maxColumnCount > 0 &&
             maxColumnCount === start + startResults.getCount())) {
          this.m_stopColumnHeaderFetch = true;
        }
      }
      if (this.m_endColHeader < 0) {
        this._hideHeader(root);
        this.m_stopColumnHeaderFetch = true;
        this.m_startColHeader = 0;
      } else {
        this.m_hasColHeader = true;
        this._buildHeaderLabels(axis, startResults);
      }

      if (endResults != null) {
        this.buildColumnEndHeaders(endRoot, endResults, start, count, false, false);
        if (endResults.getCount() < headerRange.count ||
            (maxColumnCount && maxColumnCount > 0 &&
             maxColumnCount === start + endResults.getCount())) {
          this.m_stopColumnEndHeaderFetch = true;
        }
      }
      if (this.m_endColEndHeader < 0) {
        this._hideHeader(endRoot);
        this.m_stopColumnEndHeaderFetch = true;
        this.m_startColEndHeader = 0;
      } else {
        this.m_hasColEndHeader = true;
        this._buildHeaderLabels('columnEnd', endResults);
      }
    } else if (axis === 'row') {
      if (startResults != null) {
        this.buildRowHeaders(root, startResults, start, count, rowInsert, false);
        if (startResults.getCount() < headerRange.count ||
            (maxRowCount && maxRowCount > 0 &&
             maxRowCount === start + startResults.getCount())) {
          this.m_stopRowHeaderFetch = true;
        }
      }
      if (this.m_endRowHeader < 0) {
        this._hideHeader(root);
        this.m_stopRowHeaderFetch = true;
        this.m_startRowHeader = 0;
      } else {
        this.m_hasRowHeader = true;
        this._buildHeaderLabels(axis, startResults);
      }

      if (endResults != null) {
        this.buildRowEndHeaders(endRoot, endResults, start, count, rowInsert, false);
        if (endResults.getCount() < headerRange.count ||
            (maxRowCount && maxRowCount > 0 &&
             maxRowCount === start + endResults.getCount())) {
          this.m_stopRowEndHeaderFetch = true;
        }
      }
      if (this.m_endRowEndHeader < 0) {
        this._hideHeader(endRoot);
        this.m_stopRowEndHeaderFetch = true;
        this.m_startRowEndHeader = 0;
      } else {
        this.m_hasRowEndHeader = true;
        this._buildHeaderLabels('rowEnd', endResults);
      }
    }

    if (this.isFetchComplete()) {
      this.hideStatusText();
      if (this._shouldInitialize() && !rowInsert) {
        this._handleInitialization(true);
      }
    }

    if (this.m_initialized) {
      // if there are no cells and we are initialized then size the scroller
      this._sizeDatabodyScroller();

      // we cannot syncScroller here. On touch this will trigger a refetch before the fetchCells has been called and will
      // cause an infinite loop. We always call fetchCells after fetchHeaders which calls syncScroller
      // check if we need to sync header scroll position
    }

    // end fetch
    this._signalTaskEnd();
  };

/**
 * Handle an unsuccessful call to the data source fetchHeaders
 * @param {Error} error - the error returned from the data source
 * @param {Object} headerRange - keys of {axis,start,count,header}
 */
DvtDataGrid.prototype.handleHeadersFetchError = function (error, headerRange) {
  // remove fetching message
  var axis = headerRange.axis;
  this.m_fetching[axis] = false;
  // end fetch
  this._signalTaskEnd();
};

/**
 * Build a header context object for a header and return it
 * The header elem and the data can be set to null for cases where there are no headers
 * but varying height and width are desired
 * @param {string} axis - 'row' or 'column'
 * @param {number} index - the index of the header
 * @param {Object|null} data - the data the cell contains
 * @param {Object} metadata - the metadata the cell contains
 * @param {Element|null} elem - the header element
 * @param {number} level - the header level
 * @param {number} extent - the header extent
 * @param {number} depth - the header depth
 * @return {Object} the header context object, keys of {axis,index,data,datagrid}
 */
DvtDataGrid.prototype.createHeaderContext =
  function (axis, index, data, metadata, elem, level, extent, depth) {
    var headerContext = {
      axis: axis,
      index: index,
      data: data
    };
    headerContext.component = this;
    headerContext.datasource = this.m_options.getProperty('data');
    headerContext.level = level;
    headerContext.depth = depth;
    headerContext.extent = extent;

    // set the parent element to the content div
    if (elem != null) {
      headerContext.parentElement = elem;
    }

    // merge properties from metadata into cell context
    // the properties in metadata would have precedence
    var props = Object.keys(metadata);
    for (var i = 0; i < props.length; i++) {
      var prop = props[i];
      headerContext[prop] = metadata[prop];
    }

    // invoke callback to allow ojDataGrid to change datagrid reference
    if (this.m_createContextCallback != null) {
      this.m_createContextCallback.call(this, headerContext);
    }

    return this.m_fixContextCallback.call(this, headerContext);
  };

/**
 * Build a label context object for a header label and return it
 */
DvtDataGrid.prototype._createLabelContext = function (axis, level, data, elem) {
  var labelContext = {
    axis: axis,
    level: level,
    data: data
  };
  labelContext.component = this;
  labelContext.datasource = this.m_options.getProperty('data');

  // set the parent element to the content div
  if (elem != null) {
    labelContext.parentElement = elem;
  }

  // invoke callback to allow ojDataGrid to change datagrid reference
  if (this.m_createContextCallback != null) {
    this.m_createContextCallback.call(this, labelContext);
  }

  return this.m_fixContextCallback.call(this, labelContext);
};

DvtDataGrid.prototype._buildHeaderLabels = function (axis, headerSet) {
  if (this.m_headerLabels[axis].length === 0) {
    if (headerSet && headerSet.getLabel) {
      var count = headerSet.getLevelCount();
      var dir;

      if (axis === 'rowEnd') {
        dir = this.getResources().isRTLMode() ? 'left' : 'right';
      } else if (axis === 'row') {
        dir = this.getResources().isRTLMode() ? 'right' : 'left';
      } else if (axis === 'column') {
        dir = 'top';
      } else if (axis === 'columnEnd') {
        dir = 'bottom';
      }
      var dirValue = 0;

      if (count > 0) {
        for (var i = 0; i < count; i++) {
          var label = document.createElement('div');
          var labelData = headerSet.getLabel(i);
          var dimension;
          if (axis === 'row' || axis === 'rowEnd') {
            dimension = axis === 'row' ?
              this.m_rowHeaderLevelWidths[i] : this.m_rowEndHeaderLevelWidths[i];
          } else {
            dimension = axis === 'column' ?
              this.m_columnHeaderLevelHeights[i] : this.m_columnEndHeaderLevelHeights[i];
          }
          if (labelData != null) {
            var labelContext = this._createLabelContext(axis, i, labelData, label);
            label.setAttribute(this.getResources().getMappedAttribute('container'), // @HTMLUpdateOK
              this.getResources().widgetName);
            label.setAttribute(this.getResources().getMappedAttribute('busyContext'), ''); // @HTMLUpdateOK
            this._createUniqueId(label);
            label[this.getResources().getMappedAttribute('context')] = labelContext;

            var inlineStyle = this.m_options.getInlineStyle(axis, labelContext, true);
            if (inlineStyle != null) {
              applyMergedInlineStyles(label, inlineStyle, '');
            }

            label.className = this.getMappedStyle('headerlabel') + ' ' +
              this.getMappedStyle(axis.toLowerCase() + 'headerlabel');
            var styleClass = this.m_options.getStyleClass(axis, labelContext, true);
            if (styleClass != null) {
              label.className += ' ' + styleClass;
            }

            if (axis === 'row' || axis === 'rowEnd') {
              if (axis === 'rowEnd') {
                label.style.height = '100%';
              }
              this.setElementWidth(label, dimension);
              this.setElementDir(label, dirValue, dir);
              this.setElementDir(label, 0, 'bottom');
            } else {
              if (axis === 'columnEnd') {
                label.style.width = '100%';
              }
              this.setElementHeight(label, dimension);
              this.setElementDir(label, dirValue, dir);
              this.setElementDir(label, 0, this.getResources().isRTLMode() ? 'left' : 'right');
            }
            if (this.m_options.isResizable(axis, 'width') === 'enable' ||
                this.m_options.isResizable(axis, 'height') === 'enable') {
              this._setAttribute(label, 'resizable', 'true');
            }
            var renderer = this.getRendererOrTemplate(axis, true);
            this._renderContent(renderer, labelContext, label, labelData,
              this.buildLabelTemplateContext(labelContext, {}));
            this.m_headerLabels[axis][i] = label;
          }
          dirValue += dimension;
        }
      }
    }
  }
};

DvtDataGrid.prototype.buildLabelTemplateContext = function (context, labelMetaData) {
  return {
    item: {
      data: context.data,
      metadata: labelMetaData.metadata
    },
    datasource: context.datasource
  };
};

DvtDataGrid.prototype.buildHeaderTemplateContext = function (context, headerMetaData) {
  return {
    item: {
      data: context.data,
      depth: context.depth,
      extent: context.extent,
      index: context.index,
      level: context.level,
      metadata: headerMetaData.metadata,
      axis: context.axis
    },
    datasource: context.datasource
  };
};

DvtDataGrid.prototype.buildCellTemplateContext = function (context, cellMetaData) {
  return {
    item: {
      columnExtent: context.extents.column,
      columnIndex: context.indexes.column,
      data: context.data,
      metadata: cellMetaData.metadata,
      rowExtent: context.extents.row,
      rowIndex: context.indexes.row
    },
    datasource: context.datasource,
    mode: context.mode
  };
};

DvtDataGrid.prototype.getRendererOrTemplate = function (axis, label) {
  var renderer = this.m_options.getRenderer(axis, label);
  var templateString;
  if (renderer) {
    return renderer;
  }
  var template;
  if (axis === 'cell') {
    templateString = 'cellTemplate';
  } else {
    templateString = label ? axis + 'HeaderLabelTemplate' : axis + 'HeaderTemplate';
  }
  template = this._getItemTemplateBySlotName(templateString);
  if (template) {
    return template;
  }
  return null;
};
/**
 * Construct the column headers
 * @param {Element} headerRoot
 * @param {Object} headerSet
 * @param {number} start
 * @param {number} totalCount
 * @param {boolean} insert
 * @param {boolean} returnAsFragment
 * @returns {DocumentFragment|Object|undefined}
 */
DvtDataGrid.prototype.buildColumnHeaders =
  function (headerRoot, headerSet, start, totalCount, insert, returnAsFragment) {
    if (this.m_columnHeaderLevelCount == null) {
      this.m_columnHeaderLevelCount = headerSet.getLevelCount();
    }
    if (this.m_columnHeaderLevelCount === 0) {
      return undefined;
    }

    var axis = 'column';
    var count = headerSet.getCount();
    var isAppend = start > this.m_endColHeader;
    // eslint-disable-next-line no-param-reassign
    insert = false;
    var reference = null;
    var atPixel = isAppend ? this.m_endColHeaderPixel : this.m_startColHeaderPixel;
    var currentEnd = this.m_endColHeader;
    var levelCount = this.m_columnHeaderLevelCount;
    var rootClassName =
      this.getMappedStyle('colheader') + ' ' + this.getMappedStyle('header');
    var cellClassName =
      this.getMappedStyle('headercell') + ' ' + this.getMappedStyle('colheadercell');
    // eslint-disable-next-line no-param-reassign
    returnAsFragment = false;

    var returnObj = this.buildAxisHeaders(headerRoot, headerSet, axis, start, count, isAppend,
      insert, reference, atPixel, currentEnd, levelCount,
      rootClassName, cellClassName, returnAsFragment);

    /*
    if (returnAsFragment) {
      return returnObj;
    }
    */

    var totalColumnWidth = returnObj.totalHeaderDimension;
    var totalColumnHeight = returnObj.totalLevelDimension;

    if (totalColumnWidth !== 0 && (this.m_avgColWidth === 0 || this.m_avgColWidth == null)) {
      // the average column width should only be set once, it will only change when the column width varies between columns, but
      // in such case the new average column width would not be any more precise than previous one.
      this.m_avgColWidth = totalColumnWidth / count;
    }

    if (!this.m_colHeaderHeight) {
      this.m_colHeaderHeight = totalColumnHeight;
      this.setElementHeight(headerRoot, this.m_colHeaderHeight);
    }

    // whether this is adding columns to the left or right
    if (!isAppend) {
      // to the left
      this.m_startColHeader -= count;
      this.m_startColHeaderPixel -= totalColumnWidth;
    } else {
      // to the right, in case of long scroll this should alwats be the end header of the set
      this.m_endColHeader = (start + count) - 1;
      this.m_endColHeaderPixel += totalColumnWidth;
    }

    if (totalCount === -1) {
      // eslint-disable-next-line no-param-reassign
      totalCount = this.m_endColHeader;
    }

    // stop subsequent fetching if high-water mark scrolling is used and we have reach the last row, flag it.
    if (!this._isCountUnknown('column') && this._isHighWatermarkScrolling() &&
        this.m_endColHeader + 1 >= totalCount) {
      this.m_stopColumnHeaderFetch = true;
    } else {
      this.m_stopColumnHeaderFetch = returnObj.stopFetch;
    }

    // if virtual scrolling may have to adjust at the beginning
    if (this.m_startColHeader === 0 && this.m_startColHeaderPixel !== 0) {
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, 0,
        this.m_startColHeaderPixel * -1,
        this.getResources().isRTLMode() ? 'right' : 'left',
        this.getMappedStyle('colheadercell'));
      this.m_endColHeaderPixel -= this.m_startColHeaderPixel;
      this.m_startColHeaderPixel = 0;
    }

    if (!this.m_initialized && this.m_startColHeader > 0) {
      var newStartEstimate = Math.round(this.m_avgColWidth * this.m_startColHeader);
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, this.m_startColHeader,
        newStartEstimate - this.m_startColHeaderPixel,
        this.getResources().isRTLMode() ? 'right' : 'left',
        this.getMappedStyle('colheadercell'));
      this.m_endColHeaderPixel = newStartEstimate + totalColumnWidth;
      this.m_startColHeaderPixel = newStartEstimate;
    }

    return undefined;
  };

/**
 * Construct the column end headers
 * @param {Element} headerRoot
 * @param {Object} headerSet
 * @param {number} start
 * @param {number} totalCount
 * @param {boolean} insert
 * @param {boolean} returnAsFragment
 * @returns {DocumentFragment|Object|undefined}
 */
DvtDataGrid.prototype.buildColumnEndHeaders =
  function (headerRoot, headerSet, start, totalCount, insert, returnAsFragment) {
    if (this.m_columnEndHeaderLevelCount == null) {
      this.m_columnEndHeaderLevelCount = headerSet.getLevelCount();
    }
    if (this.m_columnEndHeaderLevelCount === 0) {
      return undefined;
    }

    var axis = 'columnEnd';
    var count = headerSet.getCount();
    var isAppend = start > this.m_endColEndHeader;
    // eslint-disable-next-line no-param-reassign
    insert = false;
    var reference = null;
    var atPixel = isAppend ? this.m_endColEndHeaderPixel : this.m_startColEndHeaderPixel;
    var currentEnd = this.m_endColEndHeader;
    var levelCount = this.m_columnEndHeaderLevelCount;
    var rootClassName =
      this.getMappedStyle('colendheader') + ' ' + this.getMappedStyle('endheader');
    var cellClassName =
      this.getMappedStyle('endheadercell') + ' ' + this.getMappedStyle('colendheadercell');
    // eslint-disable-next-line no-param-reassign
    returnAsFragment = false;

    var returnObj = this.buildAxisHeaders(headerRoot, headerSet, axis, start, count, isAppend,
      insert, reference, atPixel, currentEnd, levelCount,
      rootClassName, cellClassName, returnAsFragment);
/*
    if (returnAsFragment) {
      return returnObj;
    }
*/
    var totalColumnWidth = returnObj.totalHeaderDimension;
    var totalColumnHeight = returnObj.totalLevelDimension;

    if (totalColumnWidth !== 0 && (this.m_avgColWidth === 0 || this.m_avgColWidth == null)) {
      // the average column width should only be set once, it will only change when the column width varies between columns, but
      // in such case the new average column width would not be any more precise than previous one.
      this.m_avgColWidth = totalColumnWidth / count;
    }

    if (!this.m_colEndHeaderHeight) {
      this.m_colEndHeaderHeight = totalColumnHeight;
      this.setElementHeight(headerRoot, this.m_colEndHeaderHeight);
    }

    // whether this is adding columns to the left or right
    if (!isAppend) {
      // to the left
      this.m_startColEndHeader -= count;
      this.m_startColEndHeaderPixel -= totalColumnWidth;
    } else {
      // to the right, in case of long scroll this should alwats be the end header of the set
      this.m_endColEndHeader = start + (count - 1);
      this.m_endColEndHeaderPixel += totalColumnWidth;
    }

    if (totalCount === -1) {
      // eslint-disable-next-line no-param-reassign
      totalCount = this.m_endColEndHeader;
    }

    // stop subsequent fetching if high-water mark scrolling is used and we have reach the last row, flag it.
    if (!this._isCountUnknown('column') && this._isHighWatermarkScrolling() &&
        this.m_endColEndHeader + 1 >= totalCount) {
      this.m_stopColumnEndHeaderFetch = true;
    } else {
      this.m_stopColumnEndHeaderFetch = returnObj.stopFetch;
    }

    // if virtual scrolling may have to adjust at the beginning
    if (this.m_startColEndHeader === 0 && this.m_startColEndHeaderPixel !== 0) {
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, 0,
        this.m_startColEndHeaderPixel * -1,
        this.getResources().isRTLMode() ? 'right' : 'left',
        this.getMappedStyle('colendheadercell'));
      this.m_endColEndHeaderPixel -= this.m_startColEndHeaderPixel;
      this.m_startColEndHeaderPixel = 0;
    }

    if (!this.m_initialized && this.m_startColEndHeader > 0) {
      var newStartEstimate = Math.round(this.m_avgColWidth * this.m_startColEndHeader);
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, this.m_startColEndHeader,
        newStartEstimate - this.m_startColEndHeaderPixel,
        this.getResources().isRTLMode() ? 'right' : 'left',
        this.getMappedStyle('colendheadercell'));
      this.m_endColEndHeaderPixel = newStartEstimate + totalColumnWidth;
      this.m_startColEndHeaderPixel = newStartEstimate;
    }

    return undefined;
  };

/**
 * Construct the row headers
 * @param {Element} headerRoot
 * @param {Object} headerSet
 * @param {number} start
 * @param {number} totalCount
 * @param {boolean} insert
 * @param {boolean} returnAsFragment
 * @returns {DocumentFragment|Object|undefined}
 */
DvtDataGrid.prototype.buildRowHeaders =
  function (headerRoot, headerSet, start, totalCount, insert, returnAsFragment) {
    if (this.m_rowHeaderLevelCount == null) {
      this.m_rowHeaderLevelCount = headerSet.getLevelCount();
    }
    if (this.m_rowHeaderLevelCount === 0) {
      return undefined;
    }

    var axis = 'row';
    var count = headerSet.getCount();
    var isAppend = start > this.m_endRowHeader;
    var atPixel = isAppend ? this.m_endRowHeaderPixel : this.m_startRowHeaderPixel;
    var reference;

    if (insert) {
      reference = headerRoot.firstChild.childNodes[start - this.m_startRowHeader];
      atPixel = this.getElementDir(reference, 'top');
    } else {
      reference = null;
    }

    var currentEnd = this.m_endRowHeader;
    var levelCount = this.m_rowHeaderLevelCount;
    var rootClassName =
      this.getMappedStyle('rowheader') + ' ' + this.getMappedStyle('header');
    var cellClassName =
      this.getMappedStyle('headercell') + ' ' + this.getMappedStyle('rowheadercell');

    var returnObj = this.buildAxisHeaders(headerRoot, headerSet, axis, start, count,
      isAppend, insert, reference, atPixel, currentEnd,
      levelCount, rootClassName, cellClassName,
      returnAsFragment);

    var totalRowHeight = returnObj.totalHeaderDimension;
    var totalRowWidth = returnObj.totalLevelDimension;

    if (returnAsFragment) {
      return returnObj;
    }

    if (totalRowHeight !== 0 && (this.m_avgRowHeight === 0 || this.m_avgRowHeight == null)) {
      // the average row height should only be set once, it will only change when the row height varies between rows, but
      // in such case the new average row height would not be any more precise than previous one.
      this.m_avgRowHeight = totalRowHeight / count;
    }

    if (!this.m_rowHeaderWidth) {
      this.m_rowHeaderWidth = totalRowWidth;
      this.setElementWidth(headerRoot, this.m_rowHeaderWidth);
    }

    if (isAppend) {
      // if appending a row header, make sure the previous fragment has a bottom border if it was the last
      if (this.m_endRowHeader !== -1 && count !== 0) {
        // get the last header in the scroller
        var prev = headerRoot.firstChild.childNodes[this.m_endRowHeader - this.m_startRowHeader];
        if (prev != null) {
          this.m_utils.removeCSSClassName(prev, this.getMappedStyle('borderHorizontalNone'));
        }
      }
      // in case of a long scroll the end should always be the start plus the count - 1 for 0 indexing
      this.m_endRowHeader = start + (count - 1);
      this.m_endRowHeaderPixel += totalRowHeight;
    } else if (insert) {
      if (start < this.m_startRowHeader) {
        // added before the start
        this.m_startRowHeader = start;
        this.m_startRowHeaderPixel = Math.max(0, this.m_startRowHeaderPixel - totalRowHeight);
      }
      // update the endRowHeader and endRowheaderPixel no matter where we insert
      this.m_endRowHeader += count;
      this.m_endRowHeaderPixel = Math.max(0, this.m_endRowHeaderPixel + totalRowHeight);
      this.pushRowHeadersDown(reference, totalRowHeight);
    } else {
      this.m_startRowHeader = Math.max(0, this.m_startRowHeader - count);
      // zero maximum is handled below by realigning when appropriate
      this.m_startRowHeaderPixel -= totalRowHeight;
    }

    if (totalCount === -1) {
      // eslint-disable-next-line no-param-reassign
      totalCount = this.m_endRowHeader;
    }

    // stop subsequent fetching if high-water mark scrolling is used and we have reach the last row, flag it.
    if (!this._isCountUnknown('row') && this._isHighWatermarkScrolling() &&
        this.m_endRowHeader + 1 >= totalCount) {
      this.m_stopRowHeaderFetch = true;
    } else {
      this.m_stopRowHeaderFetch = returnObj.stopFetch;
    }

    // if virtual scrolling may have to adjust at the beginning
    if (this.m_startRowHeader === 0 && this.m_startRowHeaderPixel !== 0) {
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, 0,
        this.m_startRowHeaderPixel * -1, 'top',
        this.getMappedStyle('rowheadercell'));
      this.m_endRowHeaderPixel -= this.m_startRowHeaderPixel;
      this.m_startRowHeaderPixel = 0;
    }

    if (!this.m_initialized && this.m_startRowHeader > 0) {
      var newStartEstimate = Math.round(this.m_avgRowHeight * this.m_startRowHeader);
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, this.m_startRowHeader,
        newStartEstimate - this.m_startRowHeaderPixel, 'top',
        this.getMappedStyle('rowheadercell'));
      this.m_endRowHeaderPixel = newStartEstimate + totalRowHeight;
      this.m_startRowHeaderPixel = newStartEstimate;
    }

    return undefined;
  };

/**
 * Construct the row end headers
 * @param {Element} headerRoot
 * @param {Object} headerSet
 * @param {number} start
 * @param {number} totalCount
 * @param {boolean} insert
 * @param {boolean} returnAsFragment
 * @returns {DocumentFragment|Object|undefined}
 */
DvtDataGrid.prototype.buildRowEndHeaders =
  function (headerRoot, headerSet, start, totalCount, insert, returnAsFragment) {
    if (this.m_rowEndHeaderLevelCount == null) {
      this.m_rowEndHeaderLevelCount = headerSet.getLevelCount();
    }
    if (this.m_rowEndHeaderLevelCount === 0) {
      return undefined;
    }

    var axis = 'rowEnd';
    var count = headerSet.getCount();
    var isAppend = start > this.m_endRowEndHeader;
    var reference;
    var atPixel = isAppend ? this.m_endRowEndHeaderPixel : this.m_startRowEndHeaderPixel;

    if (insert) {
      reference = headerRoot.firstChild.childNodes[start - this.m_startRowEndHeader];
      atPixel = this.getElementDir(reference, 'top');
    } else {
      reference = null;
    }
    var currentEnd = this.m_endRowEndHeader;
    var levelCount = this.m_rowEndHeaderLevelCount;
    var rootClassName =
      this.getMappedStyle('rowendheader') + ' ' + this.getMappedStyle('endheader');
    var cellClassName =
      this.getMappedStyle('endheadercell') + ' ' + this.getMappedStyle('rowendheadercell');

    var returnObj = this.buildAxisHeaders(headerRoot, headerSet, axis, start, count, isAppend,
      insert, reference, atPixel, currentEnd, levelCount,
      rootClassName, cellClassName, returnAsFragment);

    if (returnAsFragment) {
      return returnObj;
    }

    var totalRowHeight = returnObj.totalHeaderDimension;
    var totalRowWidth = returnObj.totalLevelDimension;
    /*
    if (returnAsFragment) {
      return returnObj;
    }
    */

    if (totalRowHeight !== 0 && (this.m_avgRowHeight === 0 || this.m_avgRowHeight == null)) {
      // the average row height should only be set once, it will only change when the row height varies between rows, but
      // in such case the new average row height would not be any more precise than previous one.
      this.m_avgRowHeight = totalRowHeight / count;
    }

    if (!this.m_rowEndHeaderWidth) {
      this.m_rowEndHeaderWidth = totalRowWidth;
      this.setElementWidth(headerRoot, this.m_rowEndHeaderWidth);
    }

    if (isAppend) {
      // if appending a row header, make sure the previous fragment has a bottom border if it was the last
      if (this.m_endRowEndHeader !== -1 && count !== 0) {
        // get the last header in the scroller
        var prev = headerRoot.firstChild.childNodes[this.m_endRowEndHeader -
                                                    this.m_startRowEndHeader];
        if (prev != null) {
          this.m_utils.removeCSSClassName(prev, this.getMappedStyle('borderHorizontalNone'));
        }
      }
      // in case of a long scroll the end should always be the start plus the count - 1 for 0 indexing
      this.m_endRowEndHeader = start + (count - 1);
      this.m_endRowEndHeaderPixel += totalRowHeight;
    } else if (insert) {
      if (start < this.m_startRowEndHeader) {
        // added before the start
        this.m_startRowEndHeader = start;
        this.m_startRowEndHeaderPixel = Math.max(0, this.m_startRowEndHeaderPixel - totalRowHeight);
      }
      // update the endRowEndHeader and endRowEndHeaderPixel no matter where we insert
      this.m_endRowEndHeader += count;
      this.m_endRowEndHeaderPixel = Math.max(0, this.m_endRowEndHeaderPixel + totalRowHeight);
      this.pushRowHeadersDown(reference, totalRowHeight);
    } else {
      this.m_startRowEndHeader = Math.max(0, this.m_startRowEndHeader - count);
      // zero maximum is handled below by realigning
      this.m_startRowEndHeaderPixel -= totalRowHeight;
    }

    if (totalCount === -1) {
      // eslint-disable-next-line no-param-reassign
      totalCount = this.m_endRowEndHeader;
    }

    // stop subsequent fetching if high-water mark scrolling is used and we have reach the last row, flag it.
    if (!this._isCountUnknown('row') && this._isHighWatermarkScrolling() &&
        this.m_endRowEndHeader + 1 >= totalCount) {
      this.m_stopRowEndHeaderFetch = true;
    } else {
      this.m_stopRowEndHeaderFetch = returnObj.stopFetch;
    }


    // if virtual scrolling may have to adjust at the beginning
    if (this.m_startRowEndHeader === 0 && this.m_startRowEndHeaderPixel !== 0) {
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, 0,
        this.m_startRowEndHeaderPixel * -1, 'top',
        this.getMappedStyle('rowendheadercell'));
      this.m_endRowEndHeaderPixel -= this.m_startRowEndHeaderPixel;
      this.m_startRowEndHeaderPixel = 0;
    }

    if (!this.m_initialized && this.m_startRowEndHeader > 0) {
      var newStartEstimate = Math.round(this.m_avgRowHeight * this.m_startRowEndHeader);
      this._shiftHeadersAlongAxisInContainer(headerRoot.firstChild, this.m_startRowEndHeader,
        newStartEstimate - this.m_startRowEndHeaderPixel, 'top',
        this.getMappedStyle('rowendheadercell'));
      this.m_endRowEndHeaderPixel = newStartEstimate + totalRowHeight;
      this.m_startRowEndHeaderPixel = newStartEstimate;
    }

    return undefined;
  };

/**
 * Build headers from the axis info provided
 * @param {Element} headerRoot
 * @param {Object} headerSet
 * @param {string} axis
 * @param {number} start
 * @param {number} count
 * @param {boolean} isAppend
 * @param {boolean} insert
 * @param {Element|null|undefined} reference
 * @param {number} atPixel
 * @param {number} currentEnd
 * @param {number} levelCount
 * @param {string} rootClassName
 * @param {string} cellClassName
 * @param {boolean} returnAsFragment
 * @returns {Object}
 */
DvtDataGrid.prototype.buildAxisHeaders = function (
  headerRoot, headerSet, axis, start, count, isAppend, insert, reference, atPixel,
  currentEnd, levelCount, rootClassName, cellClassName, returnAsFragment
) {
  var columns = axis.indexOf('column') !== -1;
  var styleDir = columns ? 'height' : 'width';
  var stopFetch = false;
  var totalHeaderDimension = 0;
  var totalLevelDimension = 0;
  var left = 0;
  var top = 0;
  var scroller;
  var index;

  if (!returnAsFragment) {
    // if unknown count and there's no more column, mark it so we won't try to fetch again
    if (count === 0 && this._isCountUnknown(axis)) {
      stopFetch = true;
      return {
        totalHeaderDimension: totalHeaderDimension,
        totalLevelDimension: totalLevelDimension,
        stopFetch: stopFetch };
    }
    scroller = headerRoot.firstChild;
    // add class name back if header populated later
    if (currentEnd === -1 && headerRoot.className === '') {
      // eslint-disable-next-line no-param-reassign
      headerRoot.className = rootClassName;
      // eslint-disable-next-line no-param-reassign
      headerRoot.style[styleDir] = '';
      scroller.style[styleDir] = '';
    }
  }

  var renderer = this.getRendererOrTemplate(axis);
  var fragment = document.createDocumentFragment();
  var x = 0;
  while (count - x > 0) {
    if (isAppend || insert) {
      index = start + x;
      left = columns ? atPixel + totalHeaderDimension : 0;
      top = columns ? 0 : atPixel + totalHeaderDimension;
    } else {
      index = start + (count - 1 - x);
      left = columns ? atPixel - totalHeaderDimension : 0;
      top = columns ? 0 : atPixel - totalHeaderDimension;
    }

    var returnVal = this.buildLevelHeaders(fragment, index, 0, left, top, isAppend, insert,
      renderer, headerSet, axis, cellClassName, levelCount);
    // increment the count over the extent of the header
    x += returnVal.count;
    totalHeaderDimension += returnVal.totalHeaderDimension;
    if (returnVal.totalLevelDimension > totalLevelDimension) {
      totalLevelDimension = returnVal.totalLevelDimension;
    }
  }

  if (returnAsFragment) {
    return fragment;
  }

  if (isAppend) {
    scroller.appendChild(fragment); // @HTMLUpdateOK
  } else if (insert) {
    scroller.insertBefore(fragment, reference); // @HTMLUpdateOK
  } else {
    scroller.insertBefore(fragment, scroller.firstChild); // @HTMLUpdateOK
  }

  if (!headerRoot.hasChildNodes() && !insert) {
    headerRoot.appendChild(scroller); // @HTMLUpdateOK
  }

  this.m_subtreeAttachedCallback(scroller);

  return {
    totalHeaderDimension: totalHeaderDimension,
    totalLevelDimension: totalLevelDimension,
    stopFetch: stopFetch
  };
};

/**
 * This method is used to call the renderer
 * @param {Function|undefined|null} renderer
 * @param {Object} context cellContext or headerContext
 * @param {Element} cell cell or header to append content to
 * @param {Object|string} data data for the cell
 * @param {Object|string} templateContext templateContext is template is used
 * @private
 */
 DvtDataGrid.prototype._renderContent = function (renderer, context, cell, data, templateContext) {
  if (renderer != null && typeof renderer === 'function') {
    var content = renderer.call(this, context);
    if (content != null) {
      // allow return of document fragment from jquery create/js document.createDocumentFragment
      if (content.parentNode === null || content.parentNode instanceof DocumentFragment) {
        cell.appendChild(content); // @HTMLUpdateOK
        if (!this.m_isCustomElementCallback()) {
          this.m_subtreeAttachedCallback(content);
        }
      } else if (content.parentNode != null) {
        // parent node exists, do nothing
      } else if (content.toString) {
        cell.appendChild(document.createTextNode(content.toString())); // @HTMLUpdateOK
      }
    }
    this._removeFocusFromChildElements(context, cell);
  } else if (renderer != null && typeof renderer === 'object' && this.m_engine) {
    var nodes = this.m_engine.execute(this.m_root, renderer, templateContext, null);
    for (var i = 0; i < nodes.length; i++) {
      cell.appendChild(nodes[i]); // @HTMLUpdateOK
    }
    this._removeFocusFromChildElements(context, cell);
  } else {
    if (data != null && typeof data === 'object' &&
        Object.prototype.hasOwnProperty.call(data, 'data')) {
      // eslint-disable-next-line no-param-reassign
      data = data.data;
    }
    if (data == null) {
      // eslint-disable-next-line no-param-reassign
      data = '';
    }
    cell.appendChild(document.createTextNode(data.toString())); // @HTMLUpdateOK
  }
};

/**
 * When in edit mode make all focusable elements non-focusable, since we want to manage tab stops
 * @param {Object} context cellContext or headerContext
 * @param {Element} cell cell that we want to make children unfocusable
 * @private
 */
 DvtDataGrid.prototype._removeFocusFromChildElements = function (context, cell) {
  if (context.mode !== 'edit') {
    var self = this;
    this._signalTaskStart();
    var busyContext = this.m_contextCallback(cell).getBusyContext();
    busyContext.whenReady().then(function () {
      disableAllFocusableElements(cell);
      self._signalTaskEnd();
    });
  }
};

/**
 * Build headers along an axis recursively building them within the set
 * @param {DocumentFragment|Element|undefined} fragment the fragment to append the headers to
 * @param {number} index the index to begin rendering at
 * @param {number} level the level of the header to build at
 * @param {number} left the left value of the headers
 * @param {number} top the top value to start at
 * @param {boolean} isAppend is appending after
 * @param {boolean|undefined|null} insert is row or column insert
 * @param {Function|undefined|null} renderer header renderer
 * @param {Object} headerSet object
 * @param {string} axis column or row
 * @param {string} className string of the class names to be applied to the headers
 * @param {number} totalLevels the number of levels on the header
 * @returns {Object}
 */
DvtDataGrid.prototype.buildLevelHeaders = function (
  fragment, index, level, left, top, isAppend, insert, renderer, headerSet, axis,
  className, totalLevels
) {
  var levelDimensionValue = 0;
  var totalLevelDimensionValue = 0;
  var totalHeaderDimensionValue = 0;
  var headerCount = 0;
  var dimensionAxis;
  var groupingRoot;
  var levelDimension;
  var levelDimensionCache;
  var headerDimension;
  var dimensionToAdjust;
  var dimensionToAdjustValue;
  var dimensionSecond;
  var dimensionSecondValue;
  var start;
  var end;
  var groupingContainer;
  var header;
  var i;
  var returnVal;

  if (axis === 'row') {
    dimensionAxis = 'row';
    groupingRoot = this.m_rowHeader;
    levelDimension = 'width';
    levelDimensionCache = this.m_rowHeaderLevelWidths;
    headerDimension = 'height';
    dimensionToAdjust = 'top';
    dimensionToAdjustValue = top;
    dimensionSecond = this.getResources().isRTLMode() ? 'right' : 'left';
    dimensionSecondValue = left;
    start = this.m_startRowHeader;
    end = this.m_endRowHeader;
  } else if (axis === 'rowEnd') {
    dimensionAxis = 'row';
    groupingRoot = this.m_rowEndHeader;
    levelDimension = 'width';
    levelDimensionCache = this.m_rowEndHeaderLevelWidths;
    headerDimension = 'height';
    dimensionToAdjust = 'top';
    dimensionToAdjustValue = top;
    dimensionSecond = this.getResources().isRTLMode() ? 'left' : 'right';
    dimensionSecondValue = left;
    start = this.m_startRowEndHeader;
    end = this.m_endRowEndHeader;
  } else if (axis === 'column') {
    dimensionAxis = 'column';
    groupingRoot = this.m_colHeader;
    levelDimension = 'height';
    levelDimensionCache = this.m_columnHeaderLevelHeights;
    headerDimension = 'width';
    dimensionToAdjust = this.getResources().isRTLMode() ? 'right' : 'left';
    dimensionToAdjustValue = left;
    dimensionSecond = 'top';
    dimensionSecondValue = top;
    start = this.m_startColHeader;
    end = this.m_endColHeader;
  } else {
    dimensionAxis = 'column';
    groupingRoot = this.m_colEndHeader;
    levelDimension = 'height';
    levelDimensionCache = this.m_columnEndHeaderLevelHeights;
    headerDimension = 'width';
    dimensionToAdjust = this.getResources().isRTLMode() ? 'right' : 'left';
    dimensionToAdjustValue = left;
    dimensionSecond = 'bottom';
    dimensionSecondValue = top;
    start = this.m_startColEndHeader;
    end = this.m_endColEndHeader;
  }

  // get the extent info
  var extentInfo = headerSet.getExtent(index, level);
  var headerExtent = extentInfo.extent;
  var patchBefore = extentInfo.more.before;
  var patchAfter = extentInfo.more.after;
  var headerDepth = headerSet.getDepth(index, level);
  let headerContext;

  // if the data source says to patch before this header
  // and the index is 1 more than what is currently in the viewport
  // get the groupingContainer and add to it
  if (patchBefore && index === end + 1) {
    // get the grouping of the container at the previous index
    groupingContainer = this._getHeaderContainer(index - 1, level, 0, null,
      groupingRoot, totalLevels);
    // increment the extent stored in the grouping container
    this._setAttribute(groupingContainer, 'extent',
      this._getAttribute(groupingContainer, 'extent', true) + headerExtent);
    header = groupingContainer.firstChild;
    headerContext = header[this.getResources().getMappedAttribute('context')];
    headerContext.extent += headerExtent;
    levelDimensionValue = this.getElementDir(header, levelDimension);
    // add columns to that grouping container
    for (i = 0; i < headerExtent;) {
      if (axis === 'column' || axis === 'columnEnd') {
        returnVal = this.buildLevelHeaders(groupingContainer, index + i,
          level + headerDepth, dimensionToAdjustValue,
          dimensionSecondValue + levelDimensionValue,
          isAppend, insert, renderer, headerSet, axis,
          className, totalLevels);
      } else {
        returnVal = this.buildLevelHeaders(groupingContainer, index + i,
          level + headerDepth,
          dimensionSecondValue + levelDimensionValue,
          dimensionToAdjustValue, isAppend, insert,
          renderer, headerSet, axis, className, totalLevels);
      }
      // increment the left and total and count and skip ahead to the next header
      dimensionToAdjustValue += returnVal.totalHeaderDimension;
      totalHeaderDimensionValue += returnVal.totalHeaderDimension;
      headerCount += returnVal.count;
      i += returnVal.count;
    }
    // adjust the header size based on the total of the new sizes passed back
    this.setElementDir(header, this.getElementDir(header,
      headerDimension) + totalHeaderDimensionValue,
    headerDimension);
  } else if (patchAfter && index === start - 1) {
    // if the data source says to patch after this header
    // and the index is 1 less than what is currently in the viewport
    // get the groupingContainer and add to it
    // get the grouping of the container at the previous index
    groupingContainer = this._getHeaderContainer(index + 1, level, 0, null,
      groupingRoot, totalLevels);
    // increment the extent stored in the grouping container
    this._setAttribute(groupingContainer, 'extent',
      this._getAttribute(groupingContainer, 'extent', true) + headerExtent);
    // decrement the start stored in the grouping container, since inserting before
    this._setAttribute(groupingContainer, 'start',
      this._getAttribute(groupingContainer, 'start', true) - headerExtent);
    header = groupingContainer.firstChild;
    headerContext = header[this.getResources().getMappedAttribute('context')];
    headerContext.extent += headerExtent;
    headerContext.index -= headerExtent;
    levelDimensionValue = this.getElementDir(header, levelDimension);
    for (i = 0; i < headerExtent;) {
      if (axis === 'column' || axis === 'columnEnd') {
        returnVal = this.buildLevelHeaders(groupingContainer, index - i,
          level + headerDepth, dimensionToAdjustValue,
          dimensionSecondValue + levelDimensionValue,
          isAppend, insert, renderer, headerSet, axis,
          className, totalLevels);
      } else {
        returnVal = this.buildLevelHeaders(groupingContainer, index - i,
          level + headerDepth,
          dimensionSecondValue + levelDimensionValue,
          dimensionToAdjustValue, isAppend, insert,
          renderer, headerSet, axis, className, totalLevels);
      }
      dimensionToAdjustValue -= returnVal.totalHeaderDimension;
      totalHeaderDimensionValue += returnVal.totalHeaderDimension;
      headerCount += returnVal.count;
      i += returnVal.count;
    }
    this.setElementDir(header,
      this.getElementDir(header,
        headerDimension) + totalHeaderDimensionValue,
      headerDimension);
    this.setElementDir(header, dimensionToAdjustValue, dimensionToAdjust);
  } else {
    // get the information from the headers
    var headerData = headerSet.getData(index, level);
    var headerMetadata = headerSet.getMetadata(index, level);

    // create the header element and append the content to it
    header = document.createElement('div');

    // build headerContext to pass to renderer
    headerContext = this.createHeaderContext(axis, index, headerData, headerMetadata,
      header, level, headerExtent, headerDepth);
    header.setAttribute(this.getResources().getMappedAttribute('container'), // @HTMLUpdateOK
      this.getResources().widgetName);
    header.setAttribute(this.getResources().getMappedAttribute('busyContext'), ''); // @HTMLUpdateOK
    this._createUniqueId(header);
    header[this.getResources().getMappedAttribute('context')] = headerContext;
    header[this.getResources().getMappedAttribute('metadata')] = headerMetadata;
    header.extentInfo = extentInfo;
    var inlineStyle = this.m_options.getInlineStyle(axis, headerContext);
    var styleClass = this.m_options.getStyleClass(axis, headerContext);

    // add inline styles to the column header cell
    if (inlineStyle != null) {
      applyMergedInlineStyles(header, inlineStyle, '');
    }

    // add class names to the column header cell
    header.className = className;
    header.className += (' ' + this.getMappedStyle('depth') + headerDepth);
    if (styleClass != null) {
      header.className += ' ' + styleClass;
    }

    // position the element before getting its dimensions because of half pixel rounding issues in chrome/ie
    this.setElementDir(header, dimensionSecondValue, dimensionSecond);
    this.setElementDir(header, dimensionToAdjustValue, dimensionToAdjust);

    var dimensions = this._getHeaderDimensions(header, headerDimension, levelDimension,
      levelDimensionCache, level, dimensionAxis, headerContext.key, headerDepth);

    levelDimensionValue = dimensions[levelDimension];
    this.setElementDir(header, levelDimensionValue, levelDimension);

    // find the size in case it depends on the classes, will be set in the appropriate case
    var headerDimensionValue = dimensions[headerDimension];

    this._setAttribute(header, 'depth', headerDepth);

    // if this is an outer level then add a groupingContainer around it
    if (level !== totalLevels - 1) {
      groupingContainer = document.createElement('div');
      groupingContainer.className = this.getMappedStyle('groupingcontainer');
      groupingContainer.appendChild(header); // @HTMLUpdateOK
      this._setAttribute(groupingContainer, 'start',
        (isAppend || insert) ? index : (index - headerExtent) + 1);
      this._setAttribute(groupingContainer, 'extent', headerExtent);
      this._setAttribute(groupingContainer, 'level', level);
    }

    if (level + headerDepth === totalLevels) {
      // set the px width on the header regardless of unit type currently on it
      this.setElementDir(header, headerDimensionValue, headerDimension);
      totalHeaderDimensionValue += headerDimensionValue;
      headerCount += 1;
      totalLevelDimensionValue = levelDimensionValue;
      if (!(isAppend || insert)) {
        dimensionToAdjustValue -= headerDimensionValue;
        this.setElementDir(header, dimensionToAdjustValue, dimensionToAdjust);
      }
    } else {
      for (i = 0; i < headerExtent; i++) {
        var nextIndex = (isAppend || insert) ? index + i : index - i;
        if (axis === 'column' || axis === 'columnEnd') {
          returnVal = this.buildLevelHeaders(groupingContainer, nextIndex,
            level + headerDepth, dimensionToAdjustValue,
            dimensionSecondValue + levelDimensionValue,
            isAppend, insert, renderer, headerSet, axis,
            className, totalLevels);
        } else {
          returnVal = this.buildLevelHeaders(groupingContainer, nextIndex, level + headerDepth,
            dimensionSecondValue + levelDimensionValue,
            dimensionToAdjustValue, isAppend, insert, renderer,
            headerSet, axis, className, totalLevels);
        }
        headerDimensionValue = returnVal.totalHeaderDimension;
        dimensionToAdjustValue = (isAppend || insert) ?
          dimensionToAdjustValue + headerDimensionValue :
          dimensionToAdjustValue - headerDimensionValue;
        totalHeaderDimensionValue += headerDimensionValue;
        headerCount += returnVal.count;
        i += returnVal.count - 1;
      }
      totalLevelDimensionValue = levelDimensionValue;
      if (returnVal && !isNaN(returnVal.totalLevelDimension)) {
        totalLevelDimensionValue += returnVal.totalLevelDimension;
      }
      if (!(isAppend || insert)) {
        this.setElementDir(header, dimensionToAdjustValue, dimensionToAdjust);
      }
      this.setElementDir(header, totalHeaderDimensionValue, headerDimension);
    }

    if (axis === 'columnEnd' && this.m_addBorderBottom) {
      this.m_utils.addCSSClassName(header, this.getMappedStyle('borderHorizontalSmall'));
    }

    if (axis === 'rowEnd' && this.m_addBorderRight) {
      this.m_utils.addCSSClassName(header, this.getMappedStyle('borderVerticalSmall'));
    }

    // add resizable attribute if resize enabled
    if (this._isHeaderResizeEnabled(axis, headerContext)) {
      this._setAttribute(header, 'resizable', 'true');
    }

    if (!this.m_isCustomElementCallback()) {
      // Temporarily add groupingContainer or header into this.m_root to have them in active DOM before rendering contents
      if (groupingContainer != null) {
        this.m_root.appendChild(groupingContainer); // @HTMLUpdateOK
      } else {
        this.m_root.appendChild(header); // @HTMLUpdateOK
      }
    }

    this._renderContent(renderer, headerContext, header, headerData,
      this.buildHeaderTemplateContext(headerContext, headerMetadata));

    if (axis === 'column' || this._isDataGridProvider()) {
      // check if we need to render sort icons
      if (this._isSortEnabled(axis, headerContext)) {
        if (headerMetadata.sortDirection != null && this.m_sortInfo == null) {
          this.m_sortInfo = {};
          this.m_sortInfo.key = headerMetadata.key;
          this.m_sortInfo.direction = headerMetadata.sortDirection;
          this.m_sortInfo.axis = axis;
        }

        var sortIcon = this._buildSortIcon(headerContext, header, axis);
        header.appendChild(sortIcon); // @HTMLUpdateOK
        this._setAttribute(header, 'sortable', 'true');
      }
    }
    if (this._isParentNode(headerContext)) {
      const disclousreIcon = this._buildDisclosureIcon(headerContext);
      if (this._isHierarchicalGroup(headerContext)) {
        this.m_utils.addCSSClassName(header, this.getMappedStyle('hierarchicalGroup'));
      } else {
        this.m_utils.addCSSClassName(header, this.getMappedStyle('hierarchicalTree'));
      }
      header.prepend(disclousreIcon); // @HTMLUpdateOK
      const spacer = this._buildSpacer(headerContext);
      header.prepend(spacer); // @HTMLUpdateOK @HTMLUpdateOK
    } else if (this._isLeafNode(headerContext)) {
      this.m_utils.addCSSClassName(header, this.getMappedStyle('hierarchical'));
      const spacer = this._buildSpacer(headerContext);
      header.prepend(spacer); // @HTMLUpdateOK
    }

    // Moves groupingContainer/header from this.m_root to fragment
    if (isAppend || insert) {
      // if we are appending to the end, if there is a grouping container append that, if not just append the row header
      if (groupingContainer != null) {
        fragment.appendChild(groupingContainer); // @HTMLUpdateOK
      } else {
        fragment.appendChild(header); // @HTMLUpdateOK
      }
    } else if (groupingContainer != null) {
      // if we are not appending to the end
      // if there is a grouping container append that to the fragment
      // if the fragment already has a firstChild we want to insert before it
      if (fragment.firstChild) {
        // if the firstChild is a groupingContainer just insert before it
        if (this.m_utils.containsCSSClassName(fragment.firstChild,
          this.getMappedStyle('groupingcontainer'))) {
          fragment.insertBefore(groupingContainer, fragment.firstChild); // @HTMLUpdateOK
        } else if (this.m_utils.containsCSSClassName(fragment.firstChild,
          this.getMappedStyle('headercell')) ||
                   this.m_utils.containsCSSClassName(fragment.firstChild,
                     this.getMappedStyle('endheadercell'))) {
          // if the firstChild is a cell need to insert after it
          fragment.insertBefore(groupingContainer, fragment.firstChild.nextSibling); // @HTMLUpdateOK
        }
      } else {
        fragment.appendChild(groupingContainer); // @HTMLUpdateOK
      }
    } else if (this.m_utils.containsCSSClassName(fragment,
      this.getMappedStyle('groupingcontainer'))) {
      // if the fragment itself is a grouping container insert before the other grouping containers
      fragment.insertBefore(header, fragment.firstChild.nextSibling); // @HTMLUpdateOK
    } else {
      // otherwise just insert the header at the beginning of the fragment
      fragment.insertBefore(header, fragment.firstChild); // @HTMLUpdateOK
    }
  }

  // do not put borders on last header cell, treat the index as the index + extent
  // needs to be here and not in loop in case of pactching nested headers
  if (axis === 'column' || axis === 'columnEnd') {
    if (this._isLastColumn(index + (headerExtent - 1))) {
      this.m_utils.addCSSClassName(header, this.getMappedStyle('borderVerticalNone'));
    }
  } else if (this._isLastRow(index + (headerExtent - 1)) && !insert) {
    // do not put bottom border on last row, pass the index + extent to see if it's the last index
    this.m_utils.addCSSClassName(header, this.getMappedStyle('borderHorizontalNone'));
  }

  // return value is the totalHeight of the rendered headers at this level,
  // the total count of headers rendered at that level,
  // and the totalWidth of the levels underneath it
  return {
    totalLevelDimension: totalLevelDimensionValue,
    totalHeaderDimension: totalHeaderDimensionValue,
    count: headerCount
  };
};

DvtDataGrid.prototype._getHeaderDimensions = function (header, dimension, levelDimension,
  levelCache, level, axis, key, depth) {
  var value = {};
  var levelDimensionValue = 0;
  for (var i = 0; i < depth; i++) {
    var cachedLevelDimension = levelCache[level + i];
    if (cachedLevelDimension == null) {
      levelDimensionValue = null;
      break;
    }
    levelDimensionValue += cachedLevelDimension;
  }

  if (levelDimensionValue == null) {
    value = this._computeElementWidthAndHeight(header);
  } else {
    value[levelDimension] = levelDimensionValue;
  }

  if (depth === 1) {
    // eslint-disable-next-line no-param-reassign
    levelCache[level] = value[levelDimension];
  }

  var dimensionValue = this.m_sizingManager.getSize(axis, key);
  if (dimensionValue != null) {
    value[dimension] = dimensionValue;
    return value;
  }

  // check if inline style set on element
  if (header.style[dimension] !== '') {
    if (value[dimension] == null) {
      value[dimension] = this.getElementDir(header, dimension);
    }
    // in the event that row height is set via an additional style only on row header store the value
    this.m_sizingManager.setSize(axis, key, value[dimension]);
    return value;
  }

  // check style class mapping, mapping prevents multiple offsetHeight calls on headers with the same class name
  var className = header.className;
  if (value[dimension] == null) {
    value[dimension] = this.m_styleClassDimensionMap[dimension][className];
    if (value[dimension] == null) {
      // exhausted all options, use offsetHeight then, remove element in the case of shim element
      value[dimension] = this.getElementDir(header, dimension);
    }
  }

  // the value isn't the default the cell will use meaning it's from an external
  // class, so store it in the sizing manager cell can pick it up, header and cell dimension can vary on em
  this.m_sizingManager.setSize(axis, key, value[dimension]);

  this.m_styleClassDimensionMap[dimension][className] = value[dimension];
  return value;
};

/**
 * Get the header dimension at a particular level, which will be cached if set once at that level
 * This permits the user to set the level width on the first row header at that width using renderers.
 * If it is not cached get the width
 * @param {number} level the level to get the dimension of
 * @param {Element} element the row header to get the dimension of if not cached
 * @param {Object} cache the  cache to look in
 * @param {string} dimension width/height
 * @param {number} depth
 * @returns {number} the dimension of that level
 * @private
 */
DvtDataGrid.prototype._getHeaderLevelDimension =
  function (level, element, cache, dimension, depth) {
    var width = 0;

    for (var i = 0; i < depth; i++) {
      var cachedWidth = cache[level + i];
      if (cachedWidth == null) {
        width = null;
        break;
      }
      width += cachedWidth;
    }

    if (width != null) {
      return width;
    }

    width = this.getElementDir(element, dimension);
    if (depth === 1) {
      // eslint-disable-next-line no-param-reassign
      cache[level] = width;
    }
    return width;
  };

/**
 * Get the header container surrounding the headers.
 * The structure of a container is as follows:
 * firstChild: header at that level
 * subsequent children: grouping containers except at the innermost level
 * @param {number|string} index
 * @param {number|string} level
 * @param {number|string} currentLevel
 * @param {Element|Array} headers
 * @param {Element} root
 * @param {number} levelCount
 * @returns {Element|null}
 * @private
 */
DvtDataGrid.prototype._getHeaderContainer =
  function (index, level, currentLevel, headers, root, levelCount) {
    var i;
    if (headers == null) {
      // eslint-disable-next-line no-param-reassign
      headers = root.firstChild.childNodes;
      // if we are on the scroller children there is no first header so start at the first header in the list
      i = 0;
    } else {
      // if we are on a groupingContainer skip the first header which should be a row header at that level
      i = 1;
    }
    // if at the innermost level just return the parent
    if (currentLevel === levelCount - 1) {
      return headers[0].parentNode;
    }

    // loop over all headers skipping firstChild of groups
    for (; i < headers.length; i++) {
      // if the index is between that header start and start+extent dig deeper
      var headerIndex = this._getAttribute(headers[i], 'start', true);
      var headerExtent = this._getAttribute(headers[i], 'extent', true);
      var headerDepth = this._getAttribute(headers[i].firstChild, 'depth', true);
      if (index >= headerIndex && index < headerIndex + headerExtent) {
        if (level < currentLevel + headerDepth) {
          return headers[i];
        }
        return this._getHeaderContainer(index, level, currentLevel + headerDepth,
          headers[i].childNodes, root, levelCount);
      }
    }
    return null;
  };

/**
 * Get the header at a particular index and level for a root
 * @param {number|string} index
 * @param {number|string} level
 * @param {Element} root
 * @param {number} totalLevels
 * @param {number} startIndex for that level
 * @returns {Element|null}
 * @private
 */
DvtDataGrid.prototype._getHeaderByIndex = function (index, level, root, totalLevels, startIndex) {
  var relativeIndex;

  if (level < 0) {
    return null;
  }
  // if there is only one level just get the header by index in the row ehader
  if (totalLevels === 1) {
    var headerContent = root.firstChild.childNodes;
    relativeIndex = index - startIndex;
    return headerContent[relativeIndex];
  }
  // otherwise get the column header container
  var headerContainer = this._getHeaderContainer(index, level, 0, null, root, totalLevels);
  if (headerContainer == null) {
    return null;
  }

  if (level <= ((this._getAttribute(headerContainer, 'level', true) +
                 this._getAttribute(headerContainer.firstChild, 'depth', true)) - 1)) {
    return headerContainer.firstChild;
  }

  // if the innermost level then get the child of the container at the index
  var start = this._getAttribute(headerContainer, 'start', true);
  relativeIndex = (index - start) + 1;
  return headerContainer.childNodes[relativeIndex];
};

/**
 * Get all headers with a specific index from inner to outer
 * @private
 */
DvtDataGrid.prototype._getHeadersByIndex = function (index, root, totalLevels, startIndex) {
  let headers = [];
  let depth = 1;
  for (let level = totalLevels - 1; level >= 0; level -= depth) {
    let header = this._getHeaderByIndex(index, level, root, totalLevels, startIndex);
    if (header) {
      depth = this.getHeaderCellDepth(header);
      headers.push(header);
    }
  }
  return headers;
};

/**
 * Get the attribute value that we have set in our mapping attribute
 * @param {Element} element
 * @param {string} attributeKey
 * @param {boolean} parse
 * @returns {number|string}
 */
DvtDataGrid.prototype._getAttribute = function (element, attributeKey, parse) {
  var value = element.getAttribute(this.getResources().getMappedAttribute(attributeKey));
  if (parse) {
    return parseInt(value, 10);
  }
  return value;
};

/**
 * Set a mapped attribute
 * @param {Element} element
 * @param {string} attributeKey
 * @param {string|number|boolean} value
 */
DvtDataGrid.prototype._setAttribute = function (element, attributeKey, value) {
  element.setAttribute(this.getResources().getMappedAttribute(attributeKey), value); // @HTMLUpdateOK
};

/**
 * Build the databody, fetching cells as well
 * @return {Element} the root of databody
 */
DvtDataGrid.prototype.buildDatabody = function () {
  var root = document.createElement('div');
  root.id = this.createSubId('databody');
  root.className = this.getMappedStyle('databody') + ' ' + this.getMappedStyle('scrollbarForce');
  // workaround for mozilla , where overflow div would make it focusable
  root.tabIndex = '-1';
  this.m_databody = root;
  if (!root.addEventListener) {
    root.attachEvent('onscroll', this.handleScroll.bind(this));
  } else {
    root.addEventListener('scroll', this.handleScroll.bind(this), false);
  }

  var scroller = document.createElement('div');
  scroller.className = this.getMappedStyle('scroller') +
    (this.m_utils.isTouchDevice() ? ' ' + this.getMappedStyle('scroller-mobile') : '');
  root.appendChild(scroller); // @HTMLUpdateOK

  if (!this._isHighWatermarkScrolling()) {
    var self = this;
    var scrollPosition = this.m_options.getScrollPosition();
    this._getIndexesFromScrollPosition(scrollPosition).then(function (fetchIndexes) {
      var rowIndex = fetchIndexes.row;
      var columnIndex = fetchIndexes.column;

      self.m_startRow = rowIndex;
      self.m_startCol = columnIndex;

      self.m_fetching.cells = false;

      self.fetchCells(root, rowIndex, columnIndex);
    });
    this.m_fetching.cells = true;
  } else {
    var rowIndex = 0;
    var columnIndex = 0;
    this.fetchCells(root, rowIndex, columnIndex);
  }

  return root;
};

DvtDataGrid.prototype._getIndexFromKeyPromise = function (rowKey, columnKey) {
  var self = this;
  return new Promise(function (resolve) {
    if (rowKey != null || columnKey != null) {
      self._indexes({ row: rowKey, column: columnKey }, function (indexes) {
        resolve({ rowIndexFromKey: indexes.row, columnIndexFromKey: indexes.column });
      });
    } else {
      resolve({ rowIndexFromKey: null, columnIndexFromKey: null });
    }
  });
};

DvtDataGrid.prototype._getIndexesFromScrollPosition = function (scrollPosition) {
  var self = this;
  var rowKey = scrollPosition.rowKey;
  var columnKey = scrollPosition.columnKey;
  var indexFromKeyPromise = this._getIndexFromKeyPromise(rowKey, columnKey);
  return indexFromKeyPromise.then(function (indexesFromKey) {
    var returnObj = {};
    if (indexesFromKey.rowIndexFromKey != null && indexesFromKey.rowIndexFromKey > 0) {
      returnObj.row = indexesFromKey.rowIndexFromKey;
    } else if (scrollPosition.rowIndex != null) {
      returnObj.row = scrollPosition.rowIndex;
    } else if (scrollPosition.y != null) {
      returnObj.row = Math.round(scrollPosition.y / self.getDefaultRowHeight());
    } else {
      returnObj.row = 0;
    }

    if (indexesFromKey.columnIndexFromKey != null && indexesFromKey.columnIndexFromKey > 0) {
      returnObj.column = indexesFromKey.columnIndexFromKey;
    } else if (scrollPosition.columnIndex != null) {
      returnObj.column = scrollPosition.columnIndex;
    } else if (scrollPosition.x != null) {
      returnObj.column = Math.round(scrollPosition.x / self.getDefaultColumnWidth());
    } else {
      returnObj.column = 0;
    }

    return returnObj;
  });
};

/**
 * Get the fetch count which is limited by the scollPolicyOptions max count along the axis
 * @private
 */
DvtDataGrid.prototype.getFetchCount = function (axis, start) {
  var count = this.getFetchSize(axis);
  if (this._isHighWatermarkScrolling()) {
    var prop = axis === 'row' ? 'maxRowCount' : 'maxColumnCount';
    var maxCount = this.m_options.getScrollPolicyOptions();
    if (maxCount && maxCount[prop] != null && maxCount[prop] > 0) {
      count = Math.max(Math.min(count, maxCount[prop] - start), 0);
    }
  }
  return count;
};

/**
 * Fetch cells to put in the databody. Calls fetch cells on the data source,
 * setting callbacks for success and failure.
 * @param {Element} databody - the root of the databody element
 * @param {number} rowStart - the row to start fetching at
 * @param {number} colStart - the column to start fetching at
 * @param {number|null=} rowCount - the total number of rows in the data source, if undefined then calculated
 * @param {number|null=} colCount - the total number of columns in the data source, if undefined then calculated
 * @param {Object=} callbacks - specifies success and error callbacks.  If undefined then default callbacks are used
 * @protected
 */
DvtDataGrid.prototype.fetchCells =
  function (databody, rowStart, colStart, rowCount, colCount, callbacks) {
    var successCallback;

    // checks if we are already fetching cells
    if (this.m_fetching.cells) {
      return;
    }

    if (rowCount == null) {
      // eslint-disable-next-line no-param-reassign
      rowCount = this.getFetchCount('row', rowStart);
    }

    if (colCount == null) {
      // eslint-disable-next-line no-param-reassign
      colCount = this.getFetchCount('column', colStart);
    }

    var rowRange = {
      axis: 'row', start: rowStart, count: rowCount
    };
    var columnRange = {
      axis: 'column', start: colStart, count: colCount, databody: databody
    };

    this.m_fetching.cells = { rowRange: rowRange, columnRange: columnRange };

    // if there is a override success callback specified, use it, otherwise use default one
    if (callbacks != null && callbacks.success != null) {
      successCallback = callbacks.success;
    } else {
      successCallback = this.handleCellsFetchSuccess;
    }

    this.showStatusText();
    // start fetch
    this._signalTaskStart();
    this.getDataSource().fetchCells([rowRange, columnRange], {
      success: successCallback,
      error: this.handleCellsFetchError
    }, {
      success: this,
      error: this
    });
  };

/**
 * Checks whether the response matches the current request
 * @param {Object} cellRange the cell range of the response
 * @protected
 */
DvtDataGrid.prototype.isCellFetchResponseValid = function (cellRange) {
  if (this.m_fetching == null) {
    return false;
  }

  var responseRowRange = cellRange[0];
  var responseColumnRange = cellRange[1];
  var requestCellRanges = this.m_fetching.cells;

  // do object reference check, imagine fetching 20 2 consecutive times but
  // the data changed in bewteeen and we accidentally accept the first because
  // the counts are the same
  return (responseRowRange === requestCellRanges.rowRange &&
          responseColumnRange === requestCellRanges.columnRange);
};

/**
 * Returns true if this is a long scroll (or initial scroll)
 * @return {boolean} true if it is a long or initial scroll, false otherwise
 */
DvtDataGrid.prototype.isLongScroll = function () {
  return this.m_isLongScroll;
};

/**
 * Checks whether the result is within the current viewport
 * @param {Object} cellSet - a CellSet object which encapsulates the result set of cells
 * @param {Array.<Object>} cellRange - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 * @private
 */
DvtDataGrid.prototype.isCellFetchResponseInViewport = function (cellSet, cellRange) {
  if (isNaN(this.m_avgRowHeight) || isNaN(this.m_avgColWidth) ||
      this.m_empty != null || !this.m_initialized) {
    // initial scroll these are not defined so just return true, or if not inited or if no databody
    return true;
  }

  // the goal of this method is to make sure we haven't scrolled further since the last fetch
  // so our request is still valid, we run a massive risk of running loops if our logic is wrong otherwise
  // as in we continue to request the same thing but it is never valid.

  var rowRange = cellRange[0];
  var rowStart = rowRange.start;

  var columnRange = cellRange[1];
  var columnStart = columnRange.start;

  var rowReturnVal = this._getLongScrollStart(this.m_currentScrollTop,
    this.m_prevScrollTop, 'row');
  var columnReturnVal = this._getLongScrollStart(this.m_currentScrollLeft,
    this.m_prevScrollLeft, 'column');

  // return true if the viewport fits inside the fetched range
  return (rowReturnVal.start === rowStart && columnReturnVal.start === columnStart);
};

/**
 * Handle a successful call to the data source fetchCells. Create new row and
 * cell DOM elements when necessary and then insert them into the databody.
 * @param {Object} cellSet - a CellSet object which encapsulates the result set of cells
 * @param {Array.<Object>} cellRange - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":}]
 * @param {boolean=} rowInsert - if this is triggered by a row insert event
 * @protected
 */
DvtDataGrid.prototype.handleCellsFetchSuccess = function (cellSet, cellRange, rowInsert) {
  var scrollOptions = this.m_options.getScrollPolicyOptions();
  var maxRowCount = scrollOptions ? scrollOptions.maxRowCount : null;
  var maxColumnCount = scrollOptions ? scrollOptions.maxColumnCount : null;
  var totalRowCount = this.getDataSource().getCount('row');
  var totalColumnCount = this.getDataSource().getCount('column');

  // if rowInsert is specified we can skip a couple of checks
  if (rowInsert === undefined) {
    // eslint-disable-next-line no-param-reassign
    rowInsert = false;

    // checks whether result matches what we requested
    if (!this.isCellFetchResponseValid(cellRange)) {
      // end fetch
      this._signalTaskEnd();
      // ignore result if it is not valid
      return;
    }

    // checks if the response covers the viewport or the headers were invalid
    if (this.isLongScroll() &&
        (!this.isCellFetchResponseInViewport(cellSet, cellRange) || this.m_headerInvalid)) {
      // clear cells fetching flag
      this.m_fetching.cells = false;
      this.m_headerInvalid = false;

      // ignore the response and fetch another set for the current viewport
      this.handleLongScroll(this.m_currentScrollLeft, this.m_currentScrollTop);

      // end fetch
      this._signalTaskEnd();
      return;
    }

    this.m_isLongScroll = false;
  }

  var rowRange = cellRange[0];
  var rowStart = rowRange.start;
  var rowCount = cellSet.getCount('row');

  // for short fetch it would be equal for long fetch it would be > (bottom) or < (top)
  var rowRangeNeedsUpdate = rowCount > 0 &&
    (rowStart > this.m_endRow || rowStart + rowCount <= this.m_startRow);

  // if no results returned and count is unknown, flag it so we won't try to fetch again
  // OR if highwater mark scrolling is used and count is known and we have reach the last row, stop fetching
  // OR if result set is less than what's requested, then assumes we have fetched the last row
  if ((rowCount === 0 && this._isCountUnknown('row') && rowRange.count > 0) ||
      (rowRangeNeedsUpdate && this._isHighWatermarkScrolling() &&
       !this._isCountUnknown('row') &&
       (this.m_endRow + rowCount + 1 >= totalRowCount)) ||
      (rowCount < rowRange.count) ||
      (this._isHighWatermarkScrolling() &&
       maxRowCount && maxRowCount > 0 &&
       rowStart + rowCount === maxRowCount)) {
    this.m_stopRowFetch = true;
  }

  var columnRange = cellRange[1];
  var columnStart = columnRange.start;
  var columnCount = cellSet.getCount('column');

  var columnRangeNeedsUpdate = columnCount > 0 &&
    (columnStart > this.m_endCol || columnStart + columnCount === this.m_startCol);

  // if no results returned and count is unknown, flag it so we won't try to fetch again
  // OR if highwater mark scrolling is used and count is known and we have reach the last column, stop fetching
  // OR if result set is less than what's requested, then assumes we have fetched the last column
  if ((columnCount === 0 && this._isCountUnknown('column') && columnRange.count > 0) ||
      (columnRangeNeedsUpdate && this._isHighWatermarkScrolling() &&
       !this._isCountUnknown('column') &&
       (this.m_endCol + columnCount + 1 >= totalColumnCount)) ||
      (columnCount < columnRange.count) ||
      (this._isHighWatermarkScrolling() &&
       maxColumnCount && maxColumnCount > 0 &&
       columnStart + columnCount === maxColumnCount)) {
    this.m_stopColumnFetch = true;
  }

  var databody = this.m_databody;
  if (databody == null) {
    // try to search for it in the param
    databody = columnRange.databody;
  }

  var databodyContent = databody.firstChild;
  var isAppend;
  var top;
  var left;
  var fragment;
  var addResult;
  var totalColumnWidth;
  var totalRowHeight;
  var avgWidth;
  var avgHeight;

  // if these are new rows (append or insert in the middle)
  if (rowRangeNeedsUpdate || rowInsert) {
    // whether this is adding rows to bottom (append) or top (insert)
    isAppend = !!(!rowInsert && rowStart >= this.m_startRow);

    if (isAppend) {
      top = this.m_endRowPixel;
    } else if (rowInsert) {
      var referenceCell = this._getCellByIndex({
        row: rowStart + rowCount,
        column: this.m_startCol
      });
      top = this.getElementDir(referenceCell, 'top');
    } else {
      top = this.m_startRowPixel;
    }

    left = columnStart >= this.m_startCol ? this.m_startColPixel : this.m_endColPixel;

    fragment = document.createDocumentFragment();
    addResult = this._addCellsToFragment(fragment, cellSet, rowStart, top, columnStart, left);
    totalColumnWidth = addResult.totalColumnWidth;
    totalRowHeight = addResult.totalRowHeight;
    avgWidth = addResult.avgWidth;
    avgHeight = totalRowHeight / rowCount;

    this._populateDatabody(databodyContent, fragment);

    if (isAppend) {
      // make sure there is a bottom border if adding a row to the bottom
      if (this.m_endRow !== -1 && rowCount !== 0) {
        // get the previous last row
        this._highlightCellsAlongAxis(this.m_endRow, 'row', 'index', 'remove',
          ['borderHorizontalNone']);
      }
      // update row range info if neccessary
      this.m_endRow = rowStart + (rowCount - 1);
      this.m_endRowPixel += totalRowHeight;
    } else if (rowInsert) {
      // update row range info if neccessary
      if (rowStart < this.m_startRow) {
        // added in the middle
        this.m_startRow = rowStart;
        this.m_startRowPixel = Math.max(0, this.m_startRowPixel - totalRowHeight);
      }
      // update the endRow and endRowPixel no matter where we insert
      this.m_endRow += rowCount;
      this.m_endRowPixel += totalRowHeight;
      this.pushRowsDown(rowStart + rowCount, totalRowHeight);
    } else {
      // update row range info if neccessary
      this.m_startRow -= rowCount;
      // zero maximum is handled by realigning
      this.m_startRowPixel -= totalRowHeight;
    }
  } else if (columnRangeNeedsUpdate) {
    // whether this is adding cols to right (append) or left (insert)
    isAppend = columnStart >= this.m_startCol;

    left = isAppend ? this.m_endColPixel : this.m_startColPixel;
    top = rowStart >= this.m_startRow ? this.m_startRowPixel : this.m_endRowPixel;

    fragment = document.createDocumentFragment();
    addResult = this._addCellsToFragment(fragment, cellSet, rowStart, top, columnStart, left);

    this._populateDatabody(databodyContent, fragment);
    totalColumnWidth = addResult.totalColumnWidth;
  }

  // added to only do this on initialization
  // check to see if the average width and height has change and update the canvas and the scroller accordingly
  if (avgWidth != null && (this.m_avgColWidth === 0 || this.m_avgColWidth == null)) {
    // the average column width should only be set once, it will only change when the column width varies between columns, but
    // in such case the new average column width would not be any more precise than previous one.
    this.m_avgColWidth = avgWidth;
  }

  if (avgHeight != null && (this.m_avgRowHeight === 0 || this.m_avgRowHeight == null)) {
    // the average row height should only be set once, it will only change when the row height varies between rows, but
    // in such case the new average row height would not be any more precise than previous one.
    this.m_avgRowHeight = avgHeight;
  }

  // update column range info if neccessary
  if (columnRangeNeedsUpdate) {
    // add to left or to right
    if (columnStart < this.m_startCol) {
      this.m_startCol -= columnCount;
      this.m_startColPixel -= totalColumnWidth;
    } else {
      // in virtual fetch end should always be set to last
      this.m_endCol = columnStart + (columnCount - 1);
      this.m_endColPixel += totalColumnWidth;
    }
  }

  this._sizeDatabodyScroller();

  if (this.m_endCol >= 0 && this.m_endRow >= 0) {
    this.m_hasCells = true;
  } else {
    this.m_startCol = 0;
    this.m_startRow = 0;
  }

  // if virtual scrolling we may need to adjust when the user hits the beginning
  if (this.m_startCol === 0 && this.m_startColPixel !== 0) {
    this._shiftCellsAlongAxis('column', -this.m_startColPixel, 0, true);
    this.m_endColPixel -= this.m_startColPixel;
    this.m_startColPixel = 0;
  }
  if (this.m_startRow === 0 && this.m_startRowPixel !== 0) {
    this._shiftCellsAlongAxis('row', -this.m_startRowPixel, 0, true);
    this.m_endRowPixel -= this.m_startRowPixel;
    this.m_startRowPixel = 0;
  }

  var newStartEstimate;
  if (!this.m_initialized && this.m_startCol > 0) {
    newStartEstimate = Math.round(this.m_avgColWidth * this.m_startCol);
    this._shiftCellsAlongAxis('column', newStartEstimate - this.m_startColPixel, this.m_startCol, true);
    this.m_endColPixel = newStartEstimate + totalColumnWidth;
    this.m_startColPixel = newStartEstimate;
  }
  if (!this.m_initialized && this.m_startRow > 0) {
    newStartEstimate = Math.round(this.m_avgRowHeight * this.m_startRow);
    this._shiftCellsAlongAxis('row', newStartEstimate - this.m_startRowPixel, this.m_startRow, true);
    this.m_endRowPixel = newStartEstimate + totalRowHeight;
    this.m_startRowPixel = newStartEstimate;
  }

  // fetch is done
  this.m_fetching.cells = false;

  // size the grid if fetch is done
  if (this.isFetchComplete()) {
    this.hideStatusText();

    if (!this.m_initialized) {
      const TRANSLATE3D = 'translate3d(0, 0, 0)';
      // force bitmap (to GPU) to be generated now rather than when doing actual 3d translation to minimize
      // the delay, this should only be done once
      if (this.m_utils.isTouchDevice() &&
        Object.prototype.hasOwnProperty.call(window, 'WebKitCSSMatrix')) {
        databodyContent.style.transform = TRANSLATE3D;
        if (this.m_rowHeader != null) {
          this.m_rowHeader.firstChild.style.transform = TRANSLATE3D;
        }
        if (this.m_colHeader != null) {
          this.m_colHeader.firstChild.style.transform = TRANSLATE3D;
        }
        if (this.m_rowEndHeader != null) {
          this.m_rowEndHeader.firstChild.style.transform = TRANSLATE3D;
        }
        if (this.m_colEndHeader != null) {
          this.m_colEndHeader.firstChild.style.transform = TRANSLATE3D;
        }
      }
      this._updateScrollPosition(this.m_options.getScrollPosition());
    } else if (this._checkScroll) {
      this._checkScrollPosition();
    }

    // highlight focus cell or header if specified
    if (this.m_scrollIndexAfterFetch != null) {
      this.scrollToIndex(this.m_scrollIndexAfterFetch);
      // wait for the scroll event to be fired to avoid using cell.focus() to bring into view, the case where it's in the viewport but hasn't been scrolled to yet
    } else if (this.m_scrollHeaderAfterFetch != null) {
      // if the there is a header that needs to be scrolled to after fetch scroll to the header
      this.scrollToHeader(this.m_scrollHeaderAfterFetch);
    } else if (!this.isActionableMode() &&
               this._getActiveElement() != null &&
               !this.m_utils.containsCSSClassName(this._getActiveElement(),
                 this.getMappedStyle('focus'))) {
      // highliht the active cell if we are virtualized scroll and scrolled away from the active and came back
      // also on a move event insert this will preserve the active cell
      if (this.m_shouldFocus !== true) {
        this.m_shouldFocus = false;
      }
      this._highlightActive();
      this._manageMoveCursor();
    }

    // apply current selection range to newly fetched cells
    // this is more efficient than looping over ranges when rendering cell
    if (this._isSelectionEnabled()) {
      this.applySelection(rowStart, rowStart + rowCount, columnStart, columnStart + columnCount);
    }

    // update accessibility info
    this.populateAccInfo();

    // initialize/resize/fillViewport/trigger ready event
    if (this._shouldInitialize()) {
      this._handleInitialization(true);
    } else if (this.m_initialized) {
      // cases that require resize internally on fetch:
      // 1: the datagrid root node grew / resize required
      // 2: a delete triggered a fillViewport that shrunk and expanded the databody, the check that the end pixel was below the databody height then beyond it
      if (this.m_resizeRequired === true ||
          (this.m_endRowPixel - totalRowHeight < this.getElementHeight(databody))) {
        this.resizeGrid();
      }

      var cleanDirection;
      // clean up rows outside of viewport (for non high-water mark scrolling only)
      if (rowRangeNeedsUpdate) {
        if (isAppend) {
          cleanDirection = 'top';
        } else if (!rowInsert) {
          cleanDirection = 'bottom';
        }
      } else if (columnRangeNeedsUpdate) {
        // add to left or to right
        if (columnStart === this.m_startCol) {
          cleanDirection = 'right';
        } else {
          cleanDirection = 'left';
        }
      }
      this._cleanupViewport(cleanDirection);

      this.fillViewport();
      if (this.isFetchComplete()) {
        this.fireEvent('ready', {});
      }
    }
  }

  // update any row/column selections if necessary.
  if (this._isSelectionEnabled() && this.isMultipleSelection()) {
    this._resetHeaderHighLight();
  }

  // end fetch
  this._signalTaskEnd();
  // this.dumpRanges();
};

/**
 * Set the Z index values before a given row index
 * @param {string} axis
 * @param {number} index
 * @param {boolean} startHeaders
 * @param {boolean} endHeaders
 */
DvtDataGrid.prototype._setZIndexBefore = function (axis, index, startHeaders, endHeaders) {
  if (axis !== 'row') {
    return;
  }

  var start = this.m_startRow;
  var maxPixel = this.m_currentScrollTop;
  var dir = 'top';
  var headerContainer = this.m_rowHeader;
  var levelCount = this.m_rowHeaderLevelCount;
  var headerStart = this.m_startRowHeader;
  var endHeaderContainer = this.m_rowEndHeader;
  var endLevelCount = this.m_rowEndHeaderLevelCount;
  var endHeaderStart = this.m_startRowEndHeader;

  for (var i = index; i >= start; i--) {
    var cells = this._getAxisCellsByIndex(i, axis);
    if (this.getElementDir(cells[0], dir) < maxPixel) {
      break;
    }

    for (var j = 0; j < cells.length; j++) {
      this.changeStyleProperty(cells[j], this.getCssSupport('z-index'), 10);
    }

    var header;
    if (startHeaders) {
      header = this._getHeaderByIndex(i, 0, headerContainer, levelCount, headerStart);
      this.changeStyleProperty(header, this.getCssSupport('z-index'), 10);
    }

    if (endHeaders) {
      header = this._getHeaderByIndex(i, 0, endHeaderContainer, endLevelCount, endHeaderStart);
      this.changeStyleProperty(header, this.getCssSupport('z-index'), 10);
    }
  }
};

DvtDataGrid.prototype._onTransitionEnd = function (target, handler, duration) {
  var endEvents = 'transitionend';
  var transitionTimer;

  function listener() {
    if (transitionTimer) {
      clearTimeout(transitionTimer);
      transitionTimer = undefined;
    }
    // remove handler
    target.removeEventListener(endEvents, listener);

    handler();
  }

  // add transition end listener
  target.addEventListener(endEvents, listener);

  transitionTimer =
    setTimeout(listener, duration + 100);
};

/**
 * Insert rows with animation.
 * @param {DocumentFragment|undefined} cellsFragment
 * @param {DocumentFragment|undefined} rowHeaderFragment
 * @param {number} rowStart the starting row index
 * @private
 */
DvtDataGrid.prototype._insertRowsWithAnimation = function (
  cellsFragment, rowHeaderFragment, rowEndHeaderFragment, rowStart, rowCount,
  totalRowHeight, columnStart, columnCount
) {
  var self = this;
  var i;
  var rowHeaderContent;
  var referenceRowHeader;
  var rowEndHeaderContent;
  var referenceRowEndHeader;
  var rowHeader;
  var newTop;
  var deltaY;
  var rowEndHeader;
  var insertStartPixel = 0;
  var referenceCellTop = 0;
  var referenceCellsIndex = 0;
  var referenceCells;

  // animation start
  self._signalTaskStart();
  var isAppend = rowStart > this.m_endRow;
  var databodyContent = this.m_databody.firstChild;
  var rowHeaderSupport = rowHeaderFragment != null;
  var rowEndHeaderSupport = rowEndHeaderFragment != null;

  // row to be inserted after is the reference row
  if (rowStart > 0) {
    referenceCellsIndex = rowStart - 1;
    referenceCells = this._getAxisCellsByIndex(rowStart - 1, 'row');
    referenceCellTop = this.getElementDir(referenceCells[0], 'top');
    insertStartPixel = referenceCellTop + this.getElementHeight(referenceCells[0]);
  }

  // all inherited animated rows should be hidden under previous rows in view
  this._setZIndexBefore('row', referenceCellsIndex, rowHeaderSupport, rowEndHeaderSupport);


  if (rowHeaderSupport) {
    rowHeaderContent = this.m_rowHeader.firstChild;
    referenceRowHeader = rowHeaderContent.childNodes[rowStart - this.m_startRow - 1];
  }

  if (rowEndHeaderSupport) {
    rowEndHeaderContent = this.m_rowEndHeader.firstChild;
    referenceRowEndHeader = rowEndHeaderContent.childNodes[rowStart - this.m_startRow - 1];
  }

  // loop over the fragment and assign proper top values to the fragment and then hide them
  // with transform behind the reference row
  for (i = 0; i < cellsFragment.childNodes.length; i++) {
    var cell = cellsFragment.childNodes[i];
    newTop = insertStartPixel + this.getElementDir(cell, 'top');
    deltaY = referenceCellTop - newTop - this.getElementHeight(cell);

    // move row to actual new position
    this.setElementDir(cell, newTop, 'top');

    // move row to behind reference row
    this.addTransformMoveStyle(cell, 0, 0, 'linear', 0, deltaY, 0);
  }

  if (rowHeaderSupport) {
    for (i = 0; i < rowHeaderFragment.childNodes.length; i++) {
      rowHeader = rowHeaderFragment.childNodes[i];
      newTop = insertStartPixel + this.getElementDir(rowHeader, 'top');
      deltaY = referenceCellTop - newTop - this.getElementHeight(rowHeader);
      this.setElementDir(rowHeader, newTop, 'top');
      this.addTransformMoveStyle(rowHeader, 0, 0, 'linear', 0, deltaY, 0);
    }
  }

  if (rowEndHeaderSupport) {
    for (i = 0; i < rowEndHeaderFragment.childNodes.length; i++) {
      rowEndHeader = rowEndHeaderFragment.childNodes[i];
      newTop = insertStartPixel + this.getElementDir(rowEndHeader, 'top');
      deltaY = referenceCellTop - newTop - this.getElementHeight(rowEndHeader);
      this.setElementDir(rowEndHeader, newTop, 'top');
      this.addTransformMoveStyle(rowEndHeader, 0, 0, 'linear', 0, deltaY, 0);
    }
  }

  // loop over the row after the insert point, assign new top values, but keep
  // them where they are using transforms
  for (i = rowStart; i <= this.m_endRow; i++) {
    var rowCells = this._getAxisCellsByIndex(i - this.m_startRow, 'row');
    // if (rowCells.length) {
    newTop = totalRowHeight + this.getElementDir(rowCells[0], 'top');
    // }
    deltaY = -totalRowHeight;

    for (var j = 0; j < rowCells.length; j++) {
      // move row to actual new position
      this.setElementDir(rowCells[j], newTop, 'top');

      // move row to original position
      this.addTransformMoveStyle(rowCells[j], 0, 0, 'linear', 0, deltaY, 0);
    }

    if (rowHeaderSupport) {
      rowHeader = rowHeaderContent.childNodes[i];
      this.setElementDir(rowHeader, newTop, 'top');
      this.addTransformMoveStyle(rowHeader, 0, 0, 'linear', 0, deltaY, 0);
    }
    if (rowEndHeaderSupport) {
      rowEndHeader = rowEndHeaderContent.childNodes[i];
      this.setElementDir(rowEndHeader, newTop, 'top');
      this.addTransformMoveStyle(rowEndHeader, 0, 0, 'linear', 0, deltaY, 0);
    }
  }

  this._modifyAxisCellContextIndex('row', rowStart, (this.m_endRow - rowStart) + 1, rowCount);

  // need to resize first in order to ensure visible region is big enough to handle new rows
  this.m_endRow += rowCount;
  this.m_endRowPixel += totalRowHeight;
  if (rowHeaderSupport) {
    this.m_endRowHeader += rowHeaderFragment.childNodes.length;
    this.m_endRowHeaderPixel += totalRowHeight;
  }
  if (rowEndHeaderSupport) {
    this.m_endRowEndHeader += rowEndHeaderFragment.childNodes.length;
    this.m_endRowEndHeaderPixel += totalRowHeight;
  }

  // find the row in which the new rows will be inserted and insert
  databodyContent.appendChild(cellsFragment); // @HTMLUpdateOK
  if (isAppend) {
    if (rowHeaderSupport) {
      rowHeaderContent.appendChild(rowHeaderFragment); // @HTMLUpdateOK
    }
    if (rowEndHeaderSupport) {
      rowEndHeaderContent.appendChild(rowEndHeaderFragment); // @HTMLUpdateOK
    }
  } else {
    if (rowHeaderSupport) {
      rowHeaderContent.insertBefore(rowHeaderFragment, referenceRowHeader.nextSibling); // @HTMLUpdateOK
    }
    if (rowEndHeaderSupport) {
      rowEndHeaderContent.insertBefore(rowEndHeaderFragment, referenceRowEndHeader.nextSibling); // @HTMLUpdateOK
    }
  }
  this.setElementHeight(databodyContent, this.getElementHeight(databodyContent) + totalRowHeight);
  this.resizeGrid();
  this.updateRowBanding();
  this._refreshDatabodyMap();

  if (this._isSelectionEnabled()) {
    this.applySelection(rowStart, rowStart + rowCount, columnStart, columnStart + columnCount);
  }

  var lastAnimatedElement = this._getCellByIndex(this.createIndex(rowStart + (rowCount - 1),
    this.m_endCol));
  function transitionListener() {
    self._handleAnimationEnd();
  }

  this.m_animating = true;

  // must grab duration outside of timeout otherwise processingEventQueue flag would have been reset already
  // note we set the animation duration to 1 instead of 0 because some browsers don't invoke transition end listener if duration is 0
  var duration = self.m_processingEventQueue ? 1 : DvtDataGrid.EXPAND_ANIMATION_DURATION;

  this._onTransitionEnd(lastAnimatedElement, transitionListener, duration);

  setTimeout(function () {
    var _duration = DvtDataGrid.EXPAND_ANIMATION_DURATION;
    var timing = 'ease-out';
    // add animation rules to the inserted rows
    for (var ii = rowStart; ii <= self.m_endRow; ii++) {
      var _rowCells = self._getAxisCellsByIndex(ii, 'row');
      for (var jj = 0; jj < _rowCells.length; jj++) {
        self.addTransformMoveStyle(_rowCells[jj], _duration + 'ms', 0, timing, 0, 0, 0);
      }
      if (rowHeaderSupport) {
        self.addTransformMoveStyle(rowHeaderContent.childNodes[ii], _duration + 'ms',
          0, timing, 0, 0, 0);
      }
      if (rowEndHeaderSupport) {
        self.addTransformMoveStyle(rowEndHeaderContent.childNodes[ii], _duration + 'ms',
          0, timing, 0, 0, 0);
      }
    }
  }, 0);
};

/**
 * Add cells to the fragment passed in
 * @param {DocumentFragment|undefined} fragment
 * @param {Object} cellSet
 * @param {number} rowStart the starting row index
 * @param {number} topStart the starting top value
 * @param {number} columnStart the starting column index
 * @param {number} leftStart the starting left (right) value
 * @private
 */
DvtDataGrid.prototype._addCellsToFragment =
  function (fragment, cellSet, rowStart, topStart, columnStart, leftStart) {
    var dir = this.getResources().isRTLMode() ? 'right' : 'left';

    var renderer = this.getRendererOrTemplate('cell');
    var columnBandingInterval = this.m_options.getColumnBandingInterval();
    var rowBandingInterval = this.m_options.getRowBandingInterval();
    var horizontalGridlines = this.m_options.getHorizontalGridlines();
    var verticalGridlines = this.m_options.getVerticalGridlines();

    var rowCount = cellSet.getCount('row');
    var columnCount = cellSet.getCount('column');

    var rowAppend = rowStart >= this.m_startRow;
    var columnAppend = columnStart >= this.m_startCol;

    var totalWidth = 0;
    var totalHeight = 0;
    var top = topStart;

    var tempArray = [];
    var heights = [];

    for (var i = 0; i < rowCount; i += 1) {
      var left = leftStart;
      var rowIndex = rowAppend ? rowStart + i : rowStart + (rowCount - 1 - i);
      var applyRowBanding = (Math.floor(rowIndex / rowBandingInterval) % 2 === 1);
      var height;
      var extents;
      var cell;
      var cellContext;

      for (var j = 0; j < columnCount; j += extents.column) {
        var width = 0;
        height = 0;
        var columnIndex = columnAppend ? columnStart + j : columnStart + (columnCount - 1 - j);
        var cellExtent;

        if (cellSet.getExtent) {
          cellExtent = cellSet.getExtent(this.createIndex(rowIndex, columnIndex));
        } else {
          cellExtent = {
            row: { extent: 1, more: { before: false, after: false } },
            column: { extent: 1, more: { before: false, after: false } }
          };
        }
        extents = { row: cellExtent.row.extent, column: cellExtent.column.extent };

        if (!columnAppend) {
          columnIndex = (columnIndex - extents.column) + 1;
        }
        // cannot directly modify rowIndex because outside of this loop
        var tempRowIndex = rowAppend ? rowIndex : (rowIndex - extents.row) + 1;

        var indexes = this.createIndex(tempRowIndex, columnIndex);

        if (tempArray[i] && tempArray[i][j]) {
          // update the left value since that cell exists already
          width = tempArray[i][j].width;
          left = columnAppend ? left + width : left - width;
        } else {
          var patched = this._patchExistingCells(cellExtent, columnIndex, tempRowIndex, tempArray);
          if (patched) {
            width = patched.width;
            // cell = patched.cell;
            // cellContext = cell[this.getResources().getMappedAttribute('context')];
          } else {
            var cellData = cellSet.getData(this.createIndex(tempRowIndex, columnIndex));
            var cellMetadata = cellSet.getMetadata(this.createIndex(tempRowIndex, columnIndex));

            cell = document.createElement('div');
            cellContext = this.createCellContext(indexes, cellData, cellMetadata, cell, extents);
            cell.setAttribute(this.getResources().getMappedAttribute('container'), // @HTMLUpdateOK
              this.getResources().widgetName);
            cell.setAttribute(this.getResources().getMappedAttribute('busyContext'), ''); // @HTMLUpdateOK
            this._createUniqueId(cell);
            cell[this.getResources().getMappedAttribute('context')] = cellContext;
            cell[this.getResources().getMappedAttribute('metadata')] = cellMetadata;

            // before setting our own styles, else we will overwrite them
            var inlineStyle = this.m_options.getInlineStyle('cell', cellContext);
            if (inlineStyle != null) {
              applyMergedInlineStyles(cell, inlineStyle, '');
            }

            // don't want developer setting height or width through inline styles on cell
            // should be done through header styles, or through the stylesheet
            if (cell.style.height !== '') {
              cell.style.height = '';
            }
            if (cell.style.width !== '') {
              cell.style.width = '';
            }

            this.m_utils.addCSSClassName(cell, this.getMappedStyle('cell'));
            this.m_utils.addCSSClassName(cell, this.getMappedStyle('formcontrol'));
            // determine if the newly fetched row should be banded
            if (applyRowBanding || (Math.floor(columnIndex / columnBandingInterval) % 2 === 1)) {
              this.m_utils.addCSSClassName(cell, this.getMappedStyle('banded'));
            }

            var inlineStyleClass = this.m_options.getStyleClass('cell', cellContext);
            if (inlineStyleClass != null) {
              cell.className += (' ' + inlineStyleClass);
            }

            // get the row height
            var k;
            for (k = 0; k < extents.row; k++) {
              var rowKey = k === 0 ?
                cellContext.keys.row :
                this._getKey(this._getHeaderByIndex(tempRowIndex + k, 0,
                  this.m_rowHeader, this.m_rowHeaderLevelCount,
                  this.m_startRowHeader), 'row');
              heights[i + k] = this._getCellDimension(cell, tempRowIndex + k, rowKey,
                'row', 'height');
              height += heights[i + k];
            }

            // set the px height on the cell
            this.setElementHeight(cell, height);

            for (k = 0; k < extents.column; k++) {
              var columnKey = k === 0 ?
                cellContext.keys.column :
                this._getKey(this._getHeaderByIndex(columnIndex + k, 0,
                  this.m_colHeader,
                  this.m_columnHeaderLevelCount,
                  this.m_startColHeader), 'column');
              width += this._getCellDimension(cell, columnIndex, columnKey,
                'column', 'width');
            }

            // set the px width on the cell regardless of unit type currently on it
            this.setElementWidth(cell, width);

            // do not put borders on far edge column, edge row, turn off gridlines
            if (verticalGridlines === 'hidden' ||
                (this._isLastColumn(columnIndex + (extents.column - 1)) &&
                 ((this.getRowHeaderWidth() + left + width >= this.getWidth()) ||
                  this.m_endRowEndHeader !== -1))) {
              this.m_utils.addCSSClassName(cell, this.getMappedStyle('borderVerticalNone'));
            }

            if (horizontalGridlines === 'hidden') {
              this.m_utils.addCSSClassName(cell, this.getMappedStyle('borderHorizontalNone'));
            } else if (this._isLastRow(tempRowIndex + (extents.row - 1))) {
              if (this.getRowBottom(cell, top + height) >= this.getHeight() ||
                  this.m_endColEndHeader !== -1) {
                this.m_utils.addCSSClassName(cell, this.getMappedStyle('borderHorizontalNone'));
              }
            }

            if (rowAppend) {
              this.setElementDir(cell, top, 'top');
            } else {
              this.setElementDir(cell, top - height, 'top');
            }

            if (columnAppend) {
              this.setElementDir(cell, left, dir);
            } else {
              this.setElementDir(cell, left - width, dir);
            }

            if (this.m_isCustomElementCallback()) {
              // add cell to live DOM while rendering, row is now in live DOM so do this first
              this.m_root.appendChild(cell); // @HTMLUpdateOK
            }

            this._renderContent(renderer, cellContext, cell, cellData,
              this.buildCellTemplateContext(cellContext, cellMetadata));

            fragment.appendChild(cell);
          }

          // store the extents that were rendered with this
          this._updateTempArray(tempArray, { width: width }, i, j, extents.row, extents.column);

          left = columnAppend ? left + width : left - width;

          if (i === 0) {
            totalWidth += width;
          }
        }
      }

      height = heights[i];

      top = rowAppend ? top + height : top - height;
      totalHeight += height;
    }

    totalHeight = heights.reduce(function (total, num) {
      return total + num;
    }, 0);
    var avgWidth = totalWidth / columnCount;
    return { avgWidth: avgWidth, totalRowHeight: totalHeight, totalColumnWidth: totalWidth };
  };

/**
 * Change the appropriate cell dimension as patching occurs
 * @param {number} axisExtent
 * @param {number} otherAxisExtent
 * @param {number} columnIndex
 * @param {number} rowIndex
 * @param {Array} tempArray
 * @param {string} axis
 * @param {string} dimension
 * @param {boolean} isBefore
 * @private
 */
DvtDataGrid.prototype._updateCellDimension = function (
  axisExtent, otherAxisExtent, columnIndex, rowIndex, tempArray, axis, dimension, isBefore
) {
  var dimensionDelta = 0;
  var otherDimension = dimension === 'width' ? 'height' : 'width';

  var cell = this._getCellByIndex(this.createIndex(rowIndex, columnIndex));
  var cellDimension = this.getElementDir(cell, dimension);
  var cellContext = cell[this.getResources().getMappedAttribute('context')];
  cellContext.extents[axis] += axisExtent;

  var axisIndex;
  var header;
  var headerLevelCount;
  var startHeader;

  if (axis === 'row') {
    axisIndex = rowIndex;
    header = this.m_rowHeader;
    headerLevelCount = this.m_rowHeaderLevelCount;
    startHeader = this.m_startRowHeader;
  } else {
    axisIndex = columnIndex;
    header = this.m_colHeader;
    headerLevelCount = this.m_columnHeaderLevelCount;
    startHeader = this.m_startColHeader;
  }

  for (var k = 1; k <= axisExtent; k++) {
    var key = this._getKey(this._getHeaderByIndex(isBefore ? axisIndex - k : axisIndex + k,
      0, header, headerLevelCount, startHeader),
    axis);
    dimensionDelta += this._getCellDimension(cell, isBefore ? axisIndex - k : axisIndex + k,
      key, axis, dimension);
    for (var j = 0; j < otherAxisExtent; j++) {
      var addRowIndex;
      var addColumnIndex;

      if (axis === 'row') {
        addRowIndex = isBefore ? rowIndex - k : rowIndex + k;
        addColumnIndex = columnIndex + j;
      } else {
        addRowIndex = rowIndex + j;
        addColumnIndex = isBefore ? columnIndex - k : columnIndex + k;
      }
      this._addIndexToDatabodyMap(this.createIndex(addRowIndex, addColumnIndex), cell.id);
    }
  }

  if (isBefore) {
    var dir;
    if (dimension === 'width') {
      if (this.getResources().isRTLMode()) {
        dir = 'right';
      } else {
        dir = 'left';
      }
    } else {
      dir = 'top';
    }

    this.setElementDir(cell, this.getElementDir(cell, dir) - dimensionDelta, dir);
    cellContext.indexes[axis] -= axisExtent;
  }

  this.setElementDir(cell, cellDimension + dimensionDelta, dimension);

  var returnObj = {};
  returnObj[dimension] = dimensionDelta;
  returnObj[otherDimension] = this.getElementDir(cell, otherDimension);
  returnObj.cell = cell;
  return returnObj;
};

/**
 * Patch cells as more data is fetched
 * @param {Object} cellExtent
 * @param {number} columnIndex
 * @param {number} rowIndex
 * @param {Array} tempArray
 * @private
 */
DvtDataGrid.prototype._patchExistingCells =
  function (cellExtent, columnIndex, rowIndex, tempArray) {
    var columnExtent = cellExtent.column.extent;
    var rowExtent = cellExtent.row.extent;

    var patchRowBefore = cellExtent.row.more.before;
    var patchRowAfter = cellExtent.row.more.after;
    var patchColumnBefore = cellExtent.column.more.before;
    var patchColumnAfter = cellExtent.column.more.after;

    if (columnIndex - 1 === this.m_endCol && patchColumnBefore) {
      return this._updateCellDimension(columnExtent, rowExtent, this.m_endCol, rowIndex,
        tempArray, 'column', 'width', false);
    } else if (rowIndex - 1 === this.m_endRow && patchRowBefore) {
      return this._updateCellDimension(rowExtent, columnExtent, columnIndex, this.m_endRow,
        tempArray, 'row', 'height', false);
    } else if (columnIndex + columnExtent === this.m_startCol && patchColumnAfter) {
      return this._updateCellDimension(columnExtent, rowExtent, this.m_startCol, rowIndex,
        tempArray, 'column', 'width', true);
    } else if (rowIndex + rowExtent === this.m_startRow && patchRowAfter) {
      return this._updateCellDimension(rowExtent, columnExtent, columnIndex, this.m_startRow,
        tempArray, 'row', 'height', true);
    }

    return false;
  };

/**
 * Fill spots in the temporary array with the tempObj
 * @param {Array} tempArray
 * @param {any} tempObj
 * @param {number} i
 * @param {number} j
 * @param {number} iExtent
 * @param {number} jExtent
 * @private
 */
DvtDataGrid.prototype._updateTempArray = function (tempArray, tempObj, i, j, iExtent, jExtent) {
  // store the extents that were rendered with this
  for (var k = 0; k < iExtent; k++) {
    if (tempArray[i + k] == null) {
      // eslint-disable-next-line no-param-reassign
      tempArray[i + k] = [];
    }
    for (var l = 0; l < jExtent; l++) {
      // eslint-disable-next-line no-param-reassign
      tempArray[i + k][j + l] = tempObj;
    }
  }
};

/**
 * Get a cells dimension
 * @param {Element} cell
 * @param {number} index
 * @param {string|null} key
 * @param {string} axis
 * @param {string} dimension
 * @private
 */
DvtDataGrid.prototype._getCellDimension = function (cell, index, key, axis, dimension) {
  var headerClassName;
  var endHeader;
  var dimensionOf;

  if (axis === 'row') {
    headerClassName = this.getMappedStyle('rowheadercell') + ' ' +
      this.getMappedStyle('headercell');
    endHeader = this.m_endRowHeader;
  } else if (axis === 'column') {
    headerClassName = this.getMappedStyle('colheadercell') + ' ' +
      this.getMappedStyle('headercell');
    endHeader = this.m_endColHeader;
  }

  // use a shim element so that we don't have to manage class name ordering
  // in the case of no headers this gets called everytime, so added firstPass to make sure it's only the first time
  // initialized doesn't matter because of scroll behavior
  if (endHeader === -1) {
    var shimHeaderContext = this.createHeaderContext(axis, index, null,
      { key: key }, null, 0, 0, 1);
    var inlineStyle = this.m_options.getInlineStyle(axis, shimHeaderContext);
    var styleClass = this.m_options.getStyleClass(axis, shimHeaderContext);
    dimensionOf = document.createElement('div');
    if (inlineStyle != null) {
      applyMergedInlineStyles(dimensionOf, inlineStyle, '');
    }
    dimensionOf.className = headerClassName + ' ' + styleClass;
  } else {
    dimensionOf = cell;
  }

  return this._getHeaderDimension(dimensionOf, key, axis, dimension);
};

/**
 * Adjusts everything in the grid pushing cells and headers based on indexes/dimensions to remove
 * @private
 */
 DvtDataGrid.prototype._modifyAndPushCells =
 function (indexes, dimensions, axis, headerRoot, endHeaderRoot, isAdd) {
  let ltr = this.getResources().isRTLMode() ? 'right' : 'left';
  let dir = axis === 'row' ? 'top' : ltr;
  let contextString = this.getResources().getMappedAttribute('context');
  let modifier = isAdd ? 1 : -1;

  let makeAdjustments = function (items, getIndex, setter) {
    items.forEach((item) => {
      let index = getIndex(item);
      let indexChange = 0;
      if (isAdd) {
        if (index >= indexes[0]) {
          indexChange = indexes.length;
        }
      } else {
        for (indexChange; indexChange < indexes.length; indexChange++) {
          if (indexes[indexChange] >= index) {
            break;
          }
        }
      }
      if (indexChange > 0) {
        let dimensionChange = dimensions.length ? dimensions.slice(0, indexChange).reduce(
          (accumulator, currentValue) => accumulator + currentValue) : 0;
        dimensionChange *= modifier;
        indexChange *= modifier;
        setter(item, dimensionChange, indexChange, index);
      }
    });
  };

  let cells = this.m_databody.firstChild.querySelectorAll('.' + this.getMappedStyle('cell'));
  makeAdjustments(cells,
    (item) => item[contextString].indexes[axis],
    (item, dimensionChange, indexChange) => {
      if (dimensionChange !== 0) {
        let newDimension = this.getElementDir(item, dir) + dimensionChange;
        this.setElementDir(item, newDimension, dir);
      }
      // eslint-disable-next-line no-param-reassign
      item[contextString].indexes[axis] += indexChange;
  });

  let headers = headerRoot.querySelectorAll('.' + this.getMappedStyle('headercell'));
  makeAdjustments(headers,
    (item) => item[contextString].index,
    (item, dimensionChange, indexChange) => {
      if (dimensionChange !== 0) {
        let newDimension = this.getElementDir(item, dir) + dimensionChange;
        this.setElementDir(item, newDimension, dir);
      }
      // eslint-disable-next-line no-param-reassign
      item[contextString].index += indexChange;
  });

  let groupings = headerRoot.querySelectorAll('.' + this.getMappedStyle('groupingcontainer'));
  makeAdjustments(groupings,
    (item) => this._getAttribute(item, 'start', true),
    (item, dimensionChange, indexChange, index) => {
      this._setAttribute(item, 'start', index + indexChange);
  });

  let endheaders = endHeaderRoot.querySelectorAll('.' + this.getMappedStyle('endheadercell'));
  makeAdjustments(endheaders,
    (item) => item[contextString].index,
    (item, dimensionChange, indexChange) => {
      if (dimensionChange !== 0) {
        let newDimension = this.getElementDir(item, dir) + dimensionChange;
        this.setElementDir(item, newDimension, dir);
      }
      // eslint-disable-next-line no-param-reassign
      item[contextString].index += indexChange;
  });

  let endGroupings = endHeaderRoot.querySelectorAll('.' + this.getMappedStyle('groupingcontainer'));
  makeAdjustments(endGroupings,
    (item) => this._getAttribute(item, 'start', true),
    (item, dimensionChange, indexChange, index) => {
      this._setAttribute(item, 'start', index + indexChange);
  });
};

/**
 * Push the row and all of its next siblings down.
 * @param {number} rowIndex the starting row to push down.
 * @param {number} adjustment the amount in pixel to push down.
 * @private
 */
 DvtDataGrid.prototype.pushRowsDown = function (rowIndex, adjustment) {
  while (rowIndex <= this.m_endRow) {
    var cells = this._getAxisCellsByIndex(rowIndex, 'row');
    if (cells.length > 0) {
      for (var i = 0; i < cells.length; i++) {
        var cell = cells[i];
        var top = this.getElementDir(cell, 'top') + adjustment;
        cell.style.top = top + 'px';
      }
    }
    // eslint-disable-next-line no-param-reassign
    rowIndex += 1;
  }
};

/**
 * Push the row header and all of its next siblings up.
 * @param {number} rowIndex the starting row to push up.
 * @param {number} adjustment the amount in pixel to push up.
 * @private
 */
DvtDataGrid.prototype.pushRowsUp = function (rowIndex, adjustment) {
  this.pushRowsDown(rowIndex, -adjustment);
};

/**
 * Push the row header and all of its next siblings down.
 * @param {Element} rowHeader the starting rowHeader to push down.
 * @param {number} adjustment the amount in pixel to push down.
 * @private
 */
DvtDataGrid.prototype.pushRowHeadersDown = function (rowHeader, adjustment) {
  while (rowHeader) {
    var top = this.getElementDir(rowHeader, 'top') + adjustment;
    // eslint-disable-next-line no-param-reassign
    rowHeader.style.top = top + 'px';
    // eslint-disable-next-line no-param-reassign
    rowHeader = rowHeader.nextSibling;
  }
};

/**
 * Push the row and all of its next siblings up.
 * @param {Element} rowHeader the starting rowHeader to push up.
 * @param {number} adjustment the amount in pixel to push up.
 * @private
 */
DvtDataGrid.prototype.pushRowHeadersUp = function (rowHeader, adjustment) {
  this.pushRowHeadersDown(rowHeader, -adjustment);
};

/**
 * Build a cell context object for a cell and return it
 * @param {Object} indexes - the row and column index of the cell
 * @param {Object} data - the data the cell contains
 * @param {Object} metadata - the metadata the cell contains
 * @param {Element} elem - the cell element
 * @return {Object} the cell context object, keys of {indexes,data,keys,datagrid}
 */
DvtDataGrid.prototype.createCellContext = function (indexes, data, metadata, elem, extents) {
  // set the parent to the cell content div
  var cellContext = {
    parentElement: elem,
    indexes: indexes,
    cell: data
  };
  if (this._isDataGridProvider()) {
    cellContext.data = data;
  } else {
    cellContext.data = (data != null && typeof data === 'object' &&
      Object.prototype.hasOwnProperty.call(data, 'data')) ? data.data : data;
  }
  cellContext.component = this;
  cellContext.datasource = this.m_options.getProperty('data');
  cellContext.mode = 'navigation';
  cellContext.extents = extents;

  // merge properties from metadata into cell context
  // the properties in metadata would have precedence
  var props = Object.keys(metadata);
  for (var i = 0; i < props.length; i++) {
    var prop = props[i];
    cellContext[prop] = metadata[prop];
  }

  // invoke callback to allow ojDataGrid to change datagrid reference
  if (this.m_createContextCallback != null) {
    this.m_createContextCallback.call(this, cellContext);
  }

  return this.m_fixContextCallback.call(this, cellContext);
};

/**
 * Creates a unique ID
 * @private
 */
DvtDataGrid.prototype._createUniqueId = function (cell) {
  return this._uniqueIdCallback(cell, false);
};

/**
 * Gets the width of the row header
 * @return {number} the width of the row header in pixel.
 * @protected
 */
DvtDataGrid.prototype.getRowHeaderWidth = function () {
  if (this.m_endRowHeader === -1) {
    // check if there's no row header
    return 0;
  }
  return this.m_rowHeaderWidth;
};

/**
 * Gets the height of the column header
 * @return {number} the height of the column header in pixel.
 * @protected
 */
DvtDataGrid.prototype.getColumnHeaderHeight = function () {
  if (this.m_endColHeader === -1) {
    // check if there's no column header
    return 0;
  }
  return this.m_colHeaderHeight;
};

/**
 * Gets the width of the row end header
 * @return {number} the width of the row end header in pixel.
 * @protected
 */
DvtDataGrid.prototype.getRowEndHeaderWidth = function () {
  if (this.m_endRowEndHeader === -1) {
    // check if there's no row header
    return 0;
  }
  return this.m_rowEndHeaderWidth;
};

/**
 * Gets the height of the column end header
 * @return {number} the height of the column end header in pixel.
 * @protected
 */
DvtDataGrid.prototype.getColumnEndHeaderHeight = function () {
  if (this.m_endColEndHeader === -1) {
    // check if there's no column header
    return 0;
  }
  return this.m_colEndHeaderHeight;
};

/**
 * Gets the bottom value relative to the datagrid in pixel.
 * @param {Element} row the row element
 * @param {number|undefined|null} bottom the bottom value in pixel relative to the databody
 * @return {number} the bottom value relative to the datagrid in pixels.
 * @private
 */
DvtDataGrid.prototype.getRowBottom = function (row, bottom) {
  // gets the height of the column header, if any
  var colHeaderHeight = this.getColumnHeaderHeight();
  // if a bottom value is specified use that
  if (bottom != null) {
    return colHeaderHeight + bottom;
  }

  // otherwise try find it from the row element
  var top = this.getElementDir(row, 'top');
  var height = this.calculateRowHeight(row);
  if (!isNaN(top) && !isNaN(height)) {
    return colHeaderHeight + top + height;
  }


  return colHeaderHeight;
};

/**
 * Handle an unsuccessful call to the data source fetchCells
 * @param {Error} errorStatus - the error returned from the data source
 * @param {Array.<Object>} cellRange - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 */
DvtDataGrid.prototype.handleCellsFetchError = function (errorStatus, cellRange) {
  // remove fetch message
  this.m_fetching.cells = false;

  // hide status message
  this.hideStatusText();

  // update datagrid in responds to failed fetch
  if (this.m_databody.firstChild == null) {
    // if it's initial fetch, then show no data
    if (this._shouldInitialize()) {
      this._handleInitialization(true);
    }
  } else {
    // failed while fetching more data.  stop any future fetching
    var rowRange = cellRange[0];
    var columnRange = cellRange[1];

    if (columnRange.start + (columnRange.count - 1) > this.m_endCol) {
      this.m_stopColumnFetch = true;
      // stop header fetch as well
      this.m_stopColumnHeaderFetch = true;
      this.m_stopColumnEndHeaderFetch = true;
    }

    if (rowRange.start + (rowRange.count - 1) > this.m_endRow) {
      this.m_stopRowFetch = true;
      // stop header fetch as well
      this.m_stopRowHeaderFetch = true;
      this.m_stopRowEndHeaderFetch = true;
    }
  }
};

/**
 * Display the 'fetching' status message
 */
DvtDataGrid.prototype.showStatusText = function () {
  var self = this;

  if (this.m_status.style.display === 'block' || this.m_showStatusTimeout) {
    return;
  }

  this.m_showStatusTimeout = setTimeout(function () {
    var msg = self.getResources().getTranslatedText('msgFetchingData');
    self.m_status.setAttribute('aria-label', msg);
    self.m_status.style.display = 'block';

    var left = (self.getWidth() / 2) - (self.m_status.offsetWidth / 2);
    var top = (self.getHeight() / 2) - (self.m_status.offsetHeight / 2);
    self.m_status.style.left = left + 'px';
    self.m_status.style.top = top + 'px';
    self.m_showStatusTimeout = null;
  }, this.getShowStatusDelay());
};

/**
 * Retrieve the delay before showing status
 * @return {number} the delay in ms
 * @private
 */
DvtDataGrid.prototype.getShowStatusDelay = function () {
  var delay = getCSSTimeUnitAsMillis(this.getResources().getDefaultOption('showIndicatorDelay'));
  return isNaN(delay) ? 0 : delay;
};

/**
 * Hide the 'fetching' status message
 */
DvtDataGrid.prototype.hideStatusText = function () {
  if (this.m_showStatusTimeout) {
    clearTimeout(this.m_showStatusTimeout);
    this.m_showStatusTimeout = null;
  }
  this.m_status.style.display = 'none';
};

/** ******************* focusable/editable element related methods *****************/


DvtDataGrid.prototype._isFocusableElementBeforeCell = function (elem) {
  // if element is null or if we reach the root of DataGrid or if it is the cell
  if (elem == null || elem === this.getRootElement() ||
      this.m_utils.containsCSSClassName(elem, this.getMappedStyle('cell'))) {
    return false;
  }

  var tagName = elem.tagName;
  if (tagName === 'INPUT' ||
      tagName === 'TEXTAREA' ||
      tagName === 'SELECT' ||
      tagName === 'BUTTON' ||
      tagName === 'A' ||
      this.m_utils.containsCSSClassName(elem, this.getMappedStyle('active')) ||
      (elem.getAttribute('tabIndex') != null &&
       parseInt(elem.getAttribute('tabIndex'), 10) >= 0 &&
       this.findCell(elem) !== elem)) {
    return true;
  }
  return this._isFocusableElementBeforeCell(elem.parentNode);
};

/**
 * Enables all focusable elements contained by the element, and sets focus to the first
 * @param {Element} elem
 * @return {boolean} true if focus is set successfully, false otherwise
 */
DvtDataGrid.prototype._setFocusToFirstFocusableElement = function (elem, event) {
  // enable all focusable elements
  enableAllFocusableElements(elem);
  var elems = getFocusableElementsInNode(elem);
  if (elems.length > 0) {
    var firstElement = elems[0];
    firstElement.focus();
    if (event) {
      event.preventDefault();
    }
    if (firstElement.setSelectionRange && firstElement.value) {
      try {
        // ensure focus at the end
        firstElement.setSelectionRange(firstElement.value.length, firstElement.value.length);
      } catch (e) {
        // invalid state error
      }
    }
    if (this._overwriteFlag === true && typeof elems[0].select === 'function') {
      firstElement.select();
    }
    return true;
  }

  return false;
};

/** ************************************ scrolling/virtualization ************************************/

/**
 * Handle a scroll event calling scrollTo
 * @param {Event} event - the scroll event triggering the method
 */
DvtDataGrid.prototype.handleScroll = function (event) {
  this._clearScrollPositionTimeout();

  // Adding to address firefox async scrolling and pixel perfect scroll bar issues:
  // If the datagrid doesn't need scrolling, ff will skip the async scroll event
  // If the handleScroll is called properly, it'll set the databody attribute as usual
  // Flag added here and resizeGrid setTimeout function so that either will execute, but only once.
  if (!this.m_handleScrollOverflow) {
    if (!this.m_hasVerticalScroller && !this.m_hasHorizontalScroller) {
      this.m_databody.style.overflow = 'hidden';
    }
    this.m_handleScrollOverflow = true;
  }

  // prevent scrolling when animating sort
  if (this.m_animating) {
    event.preventDefault();
    return;
  }

  // scroll on touch is handled directly by touch handlers
  if (this.m_utils.isTouchDevice()) {
    return;
  }

  if (this.m_silentScroll === true) {
    this.m_silentScroll = false;
    return;
  }

  if (!event) {
    // eslint-disable-next-line no-param-reassign
    event = window.event;
  }

  var scroller;
  if (!event.target) {
    scroller = event.srcElement;
  } else {
    scroller = event.target;
  }

  var scrollLeft = this.m_utils.getElementScrollLeft(scroller);
  var scrollTop = scroller.scrollTop;

  this.scrollTo(scrollLeft, scrollTop);
};

/**
 * Retrieve the maximum scrollable width.
 * @return {number} the maximum scrollable width.  Returns MAX_VALUE
 *         if canvas size is unknown.
 * @private
 */
DvtDataGrid.prototype._getMaxScrollWidth = function () {
  if (this._isCountUnknownOrHighwatermark('column') && !this.m_stopColumnFetch) {
    return Number.MAX_VALUE;
  }
  return this.m_scrollWidth;
};

/**
 * Retrieve the maximum scrollable height.
 * @return {number} the maximum scrollable width.  Returns MAX_VALUE
 *         if canvas size is unknown.
 * @private
 */
DvtDataGrid.prototype._getMaxScrollHeight = function () {
  if (this._isCountUnknownOrHighwatermark('row') && !this.m_stopRowFetch) {
    return Number.MAX_VALUE;
  }
  return this.m_scrollHeight;
};

/**
 * Handle a programtic scroll
 * @param {Object} options an object containing the scrollTo information
 * @param {Object} options.position scroll to an x,y location which is relative to the origin of the grid
 * @param {Object} options.position.scrollX the x position of the scrollable region, this should always be positive
 * @param {Object} options.position.scrollY the Y position of the scrollable region, this should always be positive
 *
 */
DvtDataGrid.prototype.scroll = function (options) {
  if (options.position != null) {
    var scrollPosObj = {};
    scrollPosObj.x = Math.max(0, Math.min(this.m_scrollWidth, options.position.scrollX));
    scrollPosObj.y = Math.max(0, Math.min(this.m_scrollHeight, options.position.scrollY));
    this._scrollToScrollPositionObject(scrollPosObj);
  }
};

/**
 * Used by mouse wheel and touch scrolling to set the scroll position,
 * since the deltas are obtained instead of new scroll position.
 * @param {number} deltaX - the change in X position
 * @param {number} deltaY - the change in Y position
 */
DvtDataGrid.prototype.scrollDelta = function (deltaX, deltaY) {
  // Make sure to adjust the scroller size in case the scroller is no longer the same size.
  this._adjustScrollerSize();

  var scrollLeft = Math.max(0, Math.min(this._getMaxScrollWidth(),
    this.m_currentScrollLeft - deltaX));
  var scrollTop = Math.max(0, Math.min(this._getMaxScrollHeight(),
    this.m_currentScrollTop - deltaY));
  this._initiateScroll(scrollLeft, scrollTop);
};

/**
 * Used by touch scrolling to adjust the scroll position to prevent diagonal scrolling,
 * since the deltas are obtained instead of new scroll position.
 * @param {number} diffX - the change in X position
 * @param {number} diffY - the change in Y position
 * @returns {Array} adjusted diffX, diffY
 */
DvtDataGrid.prototype.adjustTouchScroll = function (diffX, diffY) {
  // prevent 'diagonal' scrolling
  if (this.m_utils.isTouchDevice()) {
    if (diffX !== 0 && diffY !== 0) {
      // direction depends on which way moves the most
      if (Math.abs(diffX) > Math.abs(diffY)) {
        // eslint-disable-next-line no-param-reassign
        diffY = 0;
        this.m_extraScrollOverY = null;
      } else {
        // eslint-disable-next-line no-param-reassign
        diffX = 0;
        this.m_extraScrollOverX = null;
      }
    }
  }

  return [diffX, diffY];
};

/**
 * Initiate a scroll, this will differentiate between scrolling on touch vs desktop
 * @param {number} scrollLeft
 * @param {number} scrollTop
 */
DvtDataGrid.prototype._initiateScroll = function (scrollLeft, scrollTop) {
  if (!this.m_utils.isTouchDevice()) {
    this.m_utils.setElementScrollLeft(this.m_databody, scrollLeft);
    this.m_databody.scrollTop = scrollTop;
  } else {
    // for touch we'll call scrollTo directly instead of relying on scroll event to fire due to performance
    // or if the scroll position of the databody was already set properly from mousewheel etc, then just sync everything up
    // in scrollTo
    this.scrollTo(scrollLeft, scrollTop);
  }
};

/**
 * Initiate a scroll on attached call with current scroll values
 */
DvtDataGrid.prototype._initiateScrollOnAttached = function () {
  this._initiateScroll(this.m_currentScrollLeft, this.m_currentScrollTop);
};

/**
 * Disable touch scroll animation by setting durations to 0
 * @private
 */
DvtDataGrid.prototype._disableTouchScrollAnimation = function () {
  this.m_databody.firstChild.style.transitionDuration = '0ms';
  this.m_rowHeader.firstChild.style.transitionDuration = '0ms';
  this.m_colHeader.firstChild.style.transitionDuration = '0ms';
  this.m_rowEndHeader.firstChild.style.transitionDuration = '0ms';
  this.m_colEndHeader.firstChild.style.transitionDuration = '0ms';
};

/**
 * Should the datagrid long scroll using appropriate params if no databody but headers.
 * @param {number} scrollLeft - the position the scroller left should be
 * @param {number} scrollTop - the position the scroller top should be
 * @returns {boolean} true if long scroll should init
 */
DvtDataGrid.prototype._shouldLongScroll = function (scrollLeft, scrollTop) {
  // only long scroll if virtual scrolling
  if (this._isHighWatermarkScrolling()) {
    return false;
  }

  return ((scrollLeft + this.getViewportWidth()) < this._getMaxLeftPixel() ||
          (scrollTop + this.getViewportHeight()) < this._getMaxTopPixel() ||
          scrollLeft > this._getMaxRightPixel() ||
          scrollTop > this._getMaxBottomPixel());
};


/**
 * Set the scroller position, using translate3d when permitted
 * @param {number} scrollLeft - the position the scroller left should be
 * @param {number} scrollTop - the position the scroller top should be
 */
DvtDataGrid.prototype.scrollTo = function (scrollLeft, scrollTop) {
  this.m_prevScrollLeft = this.m_currentScrollLeft;
  this.m_currentScrollLeft = scrollLeft;
  this.m_prevScrollTop = this.m_currentScrollTop;
  this.m_currentScrollTop = scrollTop;

  // checkSCroll and isFetchComplete below handle the fact that the fetchCells can return sync or async
  // and we want the last time it happens to actually update the value.
  this._checkScroll = false;

  // check if this is a long scroll
  // don't do this for touch, the check must be done AFTER transition ends otherwise
  // animation will become sluggish, see _syncScroller
  if (!this.m_utils.isTouchDevice()) {
    if (this._shouldLongScroll(scrollLeft, scrollTop)) {
      this.handleLongScroll(scrollLeft, scrollTop);
    } else {
      this.fillViewport();
    }
    this._checkScroll = true;
  }

  // update header and databody scroll position
  this._syncScroller();

  // check if we need to adjust scroller dimension
  this._adjustScrollerSize();

  // check if there's a cell to focus
  if (this.m_cellToFocus != null) {
    var cell = this.m_cellToFocus;
    this.m_cellToFocus = null;
    this._setActive(cell, this._createActiveObject(cell), null, false, false, true);
  }

  // if there's an index we wanted to sctoll to after fetch it has now been scrolled to by scrollToIndex, so highlight it
  if (this.m_scrollIndexAfterFetch != null) {
    if (this._isInViewport(this.m_scrollIndexAfterFetch) === DvtDataGrid.INSIDE) {
      if (this._isDatabodyCellActive() &&
          this.m_scrollIndexAfterFetch.row === this.m_active.indexes.row &&
          this.m_scrollIndexAfterFetch.column === this.m_active.indexes.column) {
        this._highlightActive();
      }
      // should be able to scroll to index without highlighting it
      this.m_scrollIndexAfterFetch = null;
    }
  }

  // do the same for headers
  if (this.m_scrollHeaderAfterFetch != null) {
    if (!this._isDatabodyCellActive() &&
        this.m_scrollHeaderAfterFetch.axis === this.m_active.axis &&
        this.m_scrollHeaderAfterFetch.index === this.m_active.index &&
        this.m_scrollHeaderAfterFetch.level === this.m_active.level) {
      this._highlightActive();
    }
    // should be able to scroll to index without highlighting it
    this.m_scrollHeaderAfterFetch = null;
  }

  if (!this.m_utils.isTouchDevice()) {
    // If detect an actual scroll, fire scroll event
    if (this.m_prevScrollTop !== scrollTop || this.m_prevScrollLeft !== scrollLeft) {
      this.fireEvent('scroll', { event: null, ui: { scrollX: scrollLeft, scrollY: scrollTop } });
    }
  }
  if (!this.m_utils.isTouchDevice() && this.isFetchComplete()) {
    this._checkScrollPosition();
  }
};

/**
 * Callback to run when the final transition ends
 * @private
 */
DvtDataGrid.prototype._scrollTransitionEnd = function () {
  // center touch affordances if row selection multiple
  if (this._isSelectionEnabled()) {
    this._scrollTouchSelectionAffordance();
  }

  // Fire scroll event after physical scrolling finishes
  this.fireEvent('scroll', {
    event: null,
    ui: {
      scrollX: this.m_currentScrollLeft,
      scrollY: this.m_currentScrollTop
    }
  });

  // check how the viewport needs to be filled, through long scroll or HWS fillViewport.
  // This should be replaced once we optimize sort going to the newly sorted location.
  if (this._shouldLongScroll(this.m_currentScrollLeft, this.m_currentScrollTop)) {
    this.handleLongScroll(this.m_currentScrollLeft, this.m_currentScrollTop);
  } else {
    this.fillViewport();
  }

  this._checkScroll = true;

  if (this.isFetchComplete()) {
    this._checkScrollPosition();
  }
};

/**
 * Perform the bounce back animation when a swipe gesture causes over scrolling
 * @private
 */
DvtDataGrid.prototype._bounceBack = function () {
  var scrollLeft = this.m_currentScrollLeft;
  var scrollTop = this.m_currentScrollTop;

  var databody = this.m_databody.firstChild;
  var colHeader = this.m_colHeader.firstChild;
  var rowHeader = this.m_rowHeader.firstChild;
  var colEndHeader = this.m_colEndHeader.firstChild;
  var rowEndHeader = this.m_rowEndHeader.firstChild;

  databody.style.transitionDuration = DvtDataGrid.BOUNCE_ANIMATION_DURATION + 'ms';
  rowHeader.style.transitionDuration = DvtDataGrid.BOUNCE_ANIMATION_DURATION + 'ms';
  rowEndHeader.style.transitionDuration = DvtDataGrid.BOUNCE_ANIMATION_DURATION + 'ms';
  colHeader.style.transitionDuration = DvtDataGrid.BOUNCE_ANIMATION_DURATION + 'ms';
  colEndHeader.style.transitionDuration = DvtDataGrid.BOUNCE_ANIMATION_DURATION + 'ms';

  // process to run after bounce back animation ends
  if (this.m_scrollTransitionEnd == null) {
    this.m_scrollTransitionEnd = this._scrollTransitionEnd.bind(this);
  }
  this._onTransitionEnd(databody, this.m_scrollTransitionEnd,
    DvtDataGrid.BOUNCE_ANIMATION_DURATION);

  // scroll back to actual scrollLeft/scrollTop positions
  if (this.getResources().isRTLMode()) {
    databody.style.transform =
      'translate3d(' + scrollLeft + 'px, ' + (-scrollTop) + 'px, 0)';
    colHeader.style.transform = 'translate3d(' + scrollLeft + 'px, 0, 0)';
    colEndHeader.style.transform = 'translate3d(' + scrollLeft + 'px, 0, 0)';
  } else {
    databody.style.transform =
      'translate3d(' + (-scrollLeft) + 'px, ' + (-scrollTop) + 'px, 0)';
    colHeader.style.transform = 'translate3d(' + (-scrollLeft) + 'px, 0, 0)';
    colEndHeader.style.transform = 'translate3d(' + (-scrollLeft) + 'px, 0, 0)';
  }
  rowHeader.style.transform = 'translate3d(0, ' + (-scrollTop) + 'px, 0)';
  rowEndHeader.style.transform = 'translate3d(0, ' + (-scrollTop) + 'px, 0)';

  // reset
  this.m_extraScrollOverX = null;
  this.m_extraScrollOverY = null;
};

/**
 * Make sure the databody/headers and the scroller are in sync, which could happen when scrolling
 * stopped awaiting fetch to complete.
 * @private
 */
DvtDataGrid.prototype._syncScroller = function () {
  var scrollLeft = this.m_currentScrollLeft;
  var scrollTop = this.m_currentScrollTop;

  var databody = this.m_databody.firstChild;
  var colHeader = this.m_colHeader.firstChild;
  var rowHeader = this.m_rowHeader.firstChild;
  var colEndHeader = this.m_colEndHeader.firstChild;
  var rowEndHeader = this.m_rowEndHeader.firstChild;

  // use translate3d for smoother scrolling
  // this checks determine whether this is webkit and translated3d is supported
  if (this.m_utils.isTouchDevice() &&
      Object.prototype.hasOwnProperty.call(window, 'WebKitCSSMatrix')) {
    this._checkScroll = false;

    // check if the swipe gesture causes over scrolling of scrollable area
    if (this.m_extraScrollOverX != null || this.m_extraScrollOverY != null) {
      // swipe horizontal or vertical
      if (this.m_extraScrollOverX != null) {
        scrollLeft += this.m_extraScrollOverX;
      } else {
        scrollTop += this.m_extraScrollOverY;
      }

      // bounce back animation function
      if (this.m_bounceBack == null) {
        this.m_bounceBack = this._bounceBack.bind(this);
      }

      this._onTransitionEnd(databody, this.m_bounceBack, 500);
    } else if (databody.style.transitionDuration === '0ms') {
      // no transition, just call the handler directly
      this._scrollTransitionEnd();
    } else {
      if (this.m_scrollTransitionEnd == null) {
        this.m_scrollTransitionEnd = this._scrollTransitionEnd.bind(this);
      }
      this._onTransitionEnd(databody, this.m_scrollTransitionEnd,
        databody.style.transitionDuration);
    }

    // actual scrolling of databody and headers
    if (this.getResources().isRTLMode()) {
      databody.style.transform =
        'translate3d(' + scrollLeft + 'px, ' + (-scrollTop) + 'px, 0)';
      colHeader.style.transform = 'translate3d(' + scrollLeft + 'px, 0, 0)';
      colEndHeader.style.transform = 'translate3d(' + scrollLeft + 'px, 0, 0)';
    } else {
      databody.style.transform =
        'translate3d(' + (-scrollLeft) + 'px, ' + (-scrollTop) + 'px, 0)';
      colHeader.style.transform = 'translate3d(' + (-scrollLeft) + 'px, 0, 0)';
      colEndHeader.style.transform = 'translate3d(' + (-scrollLeft) + 'px, 0, 0)';
    }
    rowHeader.style.transform = 'translate3d(0, ' + (-scrollTop) + 'px, 0)';
    rowEndHeader.style.transform = 'translate3d(0, ' + (-scrollTop) + 'px, 0)';
  } else {
    var dir = this.getResources().isRTLMode() ? 'right' : 'left';
    this.setElementDir(colHeader, -scrollLeft, dir);
    this.setElementDir(colEndHeader, -scrollLeft, dir);
    this.setElementDir(rowHeader, -scrollTop, 'top');
    this.setElementDir(rowEndHeader, -scrollTop, 'top');
  }
};

/**
 * Adjust the scroller when we scroll to the ends of the scroller.  The scroller dimension might
 * need adjustment due to 1) variable column width or row height due to custom sizing 2) the row
 * or column count is not exact.
 * @private
 */
DvtDataGrid.prototype._adjustScrollerSize = function () {
  var scrollerContent = this.m_databody.firstChild;
  var scrollerContentHeight = this.getElementHeight(scrollerContent);
  var scrollerContentWidth = this.getElementWidth(scrollerContent);

  // if (1) actual content is higher than scroller (regardless of the current position) OR
  //    (2) we have reached the last row and the actual content is shorter than scroller
  if ((this._getMaxBottomPixel() > scrollerContentHeight) ||
      (this.getDataSource().getCount('row') === (this._getMaxBottom() + 1) &&
       !this._isCountUnknown('row') && this._getMaxBottom() > -1)) {
    this.setElementHeight(scrollerContent, this._getMaxBottomPixel());
  }

  // if (1) actual content is wider than scroller (regardless of the current position) OR
  //    (2) we have reached the last column and the actual content is narrower than scroller
  if ((this._getMaxRightPixel() > scrollerContentWidth) ||
      (this.getDataSource().getCount('column') === (this._getMaxRight() + 1) &&
       !this._isCountUnknown('column') && this._getMaxRight() > -1)) {
    this.setElementWidth(scrollerContent, this._getMaxRightPixel());
  }
};

/**
 * Get the starting position based on scroll
 * @param {number} scrollDir
 * @param {number} prevScrollDir
 * @param {string} axis
 * @returns {Object} contains start and startPixel
 */
DvtDataGrid.prototype._getLongScrollStart = function (scrollDir, prevScrollDir, axis) {
  var scrollerDimension;
  var maxDimension;
  var maxScroll;
  var avgDimension;
  var scrollbarSize;
  var start;
  var startPixel;
  var total;

  // totals must be 0 or higher for long scroll
  if (prevScrollDir !== scrollDir) {
    if (axis === 'row') {
      scrollerDimension = this.getElementHeight(this.m_databody.firstChild);
      maxDimension = this.m_utils._getMaxDivHeightForScrolling();
      maxScroll = this._getMaxScrollHeight();
      avgDimension = this.m_avgRowHeight;
      scrollbarSize = this.m_hasHorizontalScroller ? this.m_utils.getScrollbarSize() : 0;
      total = Math.max(Math.max(this.getDataSource().getCount(axis), this.m_endRow), 0);
    } else if (axis === 'column') {
      scrollerDimension = this.getElementWidth(this.m_databody.firstChild);
      maxDimension = this.m_utils._getMaxDivWidthForScrolling();
      maxScroll = this._getMaxScrollWidth();
      avgDimension = this.m_avgColWidth;
      scrollbarSize = this.m_hasVerticalScroller ? this.m_utils.getScrollbarSize() : 0;
      total = Math.max(Math.max(this.getDataSource().getCount(axis), this.m_endCol), 0);
    }

    var oversizeRatio = Math.max(Math.min(scrollDir / scrollerDimension, 1), 0);
    var fetchSize = this.getFetchSize(axis);
    start = Math.floor(total * oversizeRatio);
    startPixel = maxDimension <= scrollerDimension ?
      Math.min(scrollDir, maxScroll) : start * avgDimension;

    if (oversizeRatio === 1 ||
        (scrollDir + (fetchSize * avgDimension)) > (scrollerDimension - scrollbarSize)) {
      start = Math.max(total - fetchSize, 0);
      startPixel = Math.max(scrollerDimension - (fetchSize * avgDimension), 0);
    }
  } else if (axis === 'row') {
    start = this.m_startRow;
    startPixel = this.m_startRowPixel;
  } else if (axis === 'column') {
    start = this.m_startCol;
    startPixel = this.m_startColPixel;
  }

  return { start: start, startPixel: startPixel };
};

/**
 * Handle scroll to position that is completely outside of the current row/column range
 * For example, in Chrome it is possible to cause a "jump" back to the start position
 * This might also be needed if we decide to use delay scroll (to detect long scroll) to avoid
 * excessive fetching.
 * @param {number} scrollLeft - the position the scroller left should be
 * @param {number} scrollTop - the position the scroller top should be
 */
DvtDataGrid.prototype.handleLongScroll = function (scrollLeft, scrollTop) {
  this.m_isLongScroll = true;

  if (this.isFetchComplete() && this._isScrollBackToEditable(true)) {
    var rowReturnVal = this._getLongScrollStart(scrollTop, this.m_prevScrollTop, 'row');
    var startRow = rowReturnVal.start;
    var startRowPixel = rowReturnVal.startPixel;

    var columnReturnVal = this._getLongScrollStart(scrollLeft, this.m_prevScrollLeft, 'column');
    var startCol = columnReturnVal.start;
    var startColPixel = columnReturnVal.startPixel;

    // reset ranges, just cleaned up to only set if the header is present
    if (this.m_hasCells) {
      this.m_startRow = startRow;
      this.m_endRow = -1;
      this.m_startRowPixel = startRowPixel;
      this.m_endRowPixel = startRowPixel;
      this.m_startCol = startCol;
      this.m_endCol = -1;
      this.m_startColPixel = startColPixel;
      this.m_endColPixel = startColPixel;
    }

    if (this.m_hasRowHeader) {
      this.m_startRowHeader = startRow;
      this.m_endRowHeader = -1;
      this.m_startRowHeaderPixel = startRowPixel;
      this.m_endRowHeaderPixel = startRowPixel;
    }
    if (this.m_hasRowEndHeader) {
      this.m_startRowEndHeader = startRow;
      this.m_endRowEndHeader = -1;
      this.m_startRowEndHeaderPixel = startRowPixel;
      this.m_endRowEndHeaderPixel = startRowPixel;
    }
    if (this.m_hasColHeader) {
      this.m_startColHeader = startCol;
      this.m_endColHeader = -1;
      this.m_startColHeaderPixel = startColPixel;
      this.m_endColHeaderPixel = startColPixel;
    }
    if (this.m_hasColEndHeader) {
      this.m_startColEndHeader = startCol;
      this.m_endColEndHeader = -1;
      this.m_startColEndHeaderPixel = startColPixel;
      this.m_endColEndHeaderPixel = startColPixel;
    }

    this.m_stopRowFetch = false;
    this.m_stopRowHeaderFetch = false;
    this.m_stopRowEndHeaderFetch = false;
    this.m_stopColumnFetch = false;
    this.m_stopColumnHeaderFetch = false;
    this.m_stopColumnEndHeaderFetch = false;

    // custom success callback so that we can reset all ranges and fields
    // initiate fetch of headers and cells
    this.fetchHeaders('row', startRow, this.m_rowHeader, this.m_rowEndHeader, undefined, {
      success: function (headerSet, headerRange, endHeaderSet) {
        this.handleRowHeadersFetchSuccessForLongScroll(headerSet, headerRange, endHeaderSet);
      }
    });
    this.fetchHeaders('column', startCol, this.m_colHeader, this.m_colEndHeader, undefined, {
      success: function (headerSet, headerRange, endHeaderSet) {
        this.handleColumnHeadersFetchSuccessForLongScroll(headerSet, headerRange, endHeaderSet);
      }
    });
    this.fetchCells(this.m_databody, startRow, startCol, null, null, {
      success: function (cellSet, cellRange) {
        this.handleCellsFetchSuccessForLongScroll(cellSet, cellRange, startRow, startCol,
          startRowPixel, startColPixel);
      }
    });
  }
};

/**
 * Handle a successful call to the data source fetchHeaders for long scroll
 * @param {Object} headerSet - the result of the fetch
 * @param {Object} headerRange - {"axis":,"start":,"count":,"header":}
 * @param {Object} endHeaderSet - the result of the fetch
 * @protected
 */
DvtDataGrid.prototype.handleRowHeadersFetchSuccessForLongScroll =
  function (headerSet, headerRange, endHeaderSet) {
    var headerContent = this.m_rowHeader.firstChild;
    var endHeaderContent = this.m_rowEndHeader.firstChild;
    if (headerContent != null) {
      this.m_utils.empty(headerContent);
    }
    if (endHeaderContent != null) {
      this.m_utils.empty(endHeaderContent);
    }
    this.handleHeadersFetchSuccess(headerSet, headerRange, endHeaderSet, false);
  };

/**
 * Handle a successful call to the data source fetchHeaders for long scroll
 * @param {Object} headerSet - the result of the fetch
 * @param {Object} headerRange - {"axis":,"start":,"count":,"header":}
 * @param {Object} endHeaderSet - the result of the fetch
 * @protected
 */
DvtDataGrid.prototype.handleColumnHeadersFetchSuccessForLongScroll =
  function (headerSet, headerRange, endHeaderSet) {
    var headerContent = this.m_colHeader.firstChild;
    var endHeaderContent = this.m_colEndHeader.firstChild;
    if (headerContent != null) {
      this.m_utils.empty(headerContent);
    }
    if (endHeaderContent != null) {
      this.m_utils.empty(endHeaderContent);
    }
    this.handleHeadersFetchSuccess(headerSet, headerRange, endHeaderSet, false);
  };

/**
 * Handle a successful call to the data source fetchCells. Create new row and
 * cell DOM elements when necessary and then insert them into the databody.
 * @param {Object} cellSet - a CellSet object which encapsulates the result set of cells
 * @param {Array.<Object>} cellRange - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 * @param {number} startRow the row to start insert at
 * @param {number} startCol the col to start insert at
 * @param {number} startRowPixel the row pixel to start insert at
 * @param {number} startColPixel the col pixel to start insert at
 * @protected
 */
DvtDataGrid.prototype.handleCellsFetchSuccessForLongScroll =
  // eslint-disable-next-line no-unused-vars
  function (cellSet, cellRange, startRow, startCol, startRowPixel, startColPixel) {
    var databodyContent = this.m_databody.firstChild;
    if (databodyContent != null) {
      this._emptyDatabody(databodyContent);
    }

    // now calls fetch success proc
    this.handleCellsFetchSuccess(cellSet, cellRange);
  };

/**
 * Method to clean up the viewport in one direction, left cleans the first columns, top the first rows etc.
 * This is seperate from fill viewport so that in both the synchronus and asynchronus
 * fetch case the cleanuo happens after we get the data fpor the next area.
 * @param {string|null|undefined} direction left/right/top/bottom
 */
DvtDataGrid.prototype._cleanupViewport = function (direction) {
  if (this._isHighWatermarkScrolling() || !this._isScrollBackToEditable()) {
    return;
  }

  // direction can be null if there are no cells fetched
  if (direction == null) {
    if (this.m_prevScrollLeft > this.m_currentScrollLeft) {
      // eslint-disable-next-line no-param-reassign
      direction = 'right';
    } else if (this.m_prevScrollLeft < this.m_currentScrollLeft) {
      // eslint-disable-next-line no-param-reassign
      direction = 'left';
    } else if (this.m_prevScrollTop > this.m_currentScrollTop) {
      // eslint-disable-next-line no-param-reassign
      direction = 'bottom';
    } else if (this.m_prevScrollTop < this.m_currentScrollTop) {
      // eslint-disable-next-line no-param-reassign
      direction = 'top';
    }
  }

  // the viewport is the scroller, width and height
  var viewportLeft = this._getViewportLeft();
  var viewportRight = this._getViewportRight();
  var viewportTop = this._getViewportTop();
  var viewportBottom = this._getViewportBottom();

  if (direction === 'top' && viewportTop > this._getMaxTopPixel()) {
    this.removeRowsFromTop(this.m_databody);
    this.removeRowHeadersFromTop();
  } else if (direction === 'bottom' && viewportBottom < this._getMaxBottomPixel()) {
    this.removeRowsFromBottom(this.m_databody);
    this.removeRowHeadersFromBottom();
  } else if (direction === 'left' && viewportLeft > this._getMaxLeftPixel()) {
    this.removeColumnsFromLeft(this.m_databody);
    this.removeColumnHeadersFromLeft();
  } else if (direction === 'right' && viewportRight < this._getMaxRightPixel()) {
    this.removeColumnsFromRight(this.m_databody);
    this.removeColumnHeadersFromRight();
  }
};

/**
 * Make sure the viewport is filled of cells, this method has been modified to just fill
 * and so that it will always follow a fetchHeaders call with a fetchCells call to keep them in sync.
 */
DvtDataGrid.prototype.fillViewport = function () {
  var fetchStart;
  var fetchSize;

  if (this.isFetchComplete()) {
    // the viewport is the scroller, width and height
    // fetch slightly before the edge for the zoomed browser case as the pixel mapping isn't perfect
    var viewportLeft = this._getViewportLeft();
    var viewportRight = this._getViewportRight() + DvtDataGrid.FETCH_PIXEL_THRESHOLD;
    var viewportTop = this._getViewportTop();
    var viewportBottom = this._getViewportBottom() + DvtDataGrid.FETCH_PIXEL_THRESHOLD;

    if (this._getMaxBottomPixel() <= viewportBottom) {
      if (!this.m_stopRowHeaderFetch || !this.m_stopRowEndHeaderFetch || !this.m_stopRowFetch) {
        fetchStart = Math.max(0, this._getMaxBottom() + 1);
        fetchSize = Math.max(0, this.getFetchCount('row', fetchStart));
        this.fetchHeaders('row', fetchStart, this.m_rowHeader, this.m_rowEndHeader, fetchSize);
        this.fetchCells(this.m_databody, fetchStart, this.m_startCol, fetchSize,
          (this.m_endCol - this.m_startCol) + 1);
        return;
      }
    }

    if ((this._getMaxTopPixel() > viewportTop || this.m_currentScrollTop === 0) &&
      this._getMaxTop() > 0) {
      fetchStart = Math.max(0, this._getMaxTop() - this.getFetchSize('row'));
      fetchSize = Math.max(0, this._getMaxTop() - fetchStart);
      this.fetchHeaders('row', fetchStart, this.m_rowHeader, this.m_rowEndHeader, fetchSize);
      this.fetchCells(this.m_databody, fetchStart, this.m_startCol, fetchSize,
        (this.m_endCol - this.m_startCol) + 1);
      return;
    }

    if (this._getMaxRightPixel() <= viewportRight) {
      if (!this.m_stopColumnHeaderFetch ||
          !this.m_stopColumnEndHeaderFetch ||
          !this.m_stopColumnFetch) {
        fetchStart = Math.max(0, this._getMaxRight() + 1);
        fetchSize = Math.max(0, this.getFetchCount('column', fetchStart));
        this.fetchHeaders('column', fetchStart, this.m_colHeader, this.m_colEndHeader, fetchSize);
        this.fetchCells(this.m_databody, this.m_startRow, fetchStart,
          (this.m_endRow - this.m_startRow) + 1, fetchSize);
        return;
      }
    }

    if ((this._getMaxLeftPixel() > viewportLeft || this.m_currentScrollLeft === 0)
      && this._getMaxLeft() > 0) {
      fetchStart = Math.max(0, this._getMaxLeft() - this.getFetchSize('column'));
      fetchSize = Math.max(0, this._getMaxLeft() - fetchStart);
      this.fetchHeaders('column', fetchStart, this.m_colHeader, this.m_colEndHeader, fetchSize);
      this.fetchCells(this.m_databody, this.m_startRow, fetchStart,
        (this.m_endRow - this.m_startRow) + 1, fetchSize);
    }
  }
};

/**
 * @returns {number} last column or column start or end header
 */
DvtDataGrid.prototype._getMaxRight = function () {
  return Math.max(Math.max(this.m_endCol, this.m_endColHeader), this.m_endColEndHeader);
};

/**
 * @returns {number} first column or column start or end header
 */
DvtDataGrid.prototype._getMaxLeft = function () {
  return Math.max(Math.max(this.m_startCol, this.m_startColHeader), this.m_startColEndHeader);
};

/**
 * @returns {number} last column or column start or end header pixel
 */
DvtDataGrid.prototype._getMaxRightPixel = function () {
  return Math.max(Math.max(this.m_endColPixel,
    this.m_endColHeaderPixel),
  this.m_endColEndHeaderPixel);
};

/**
 * @returns {number} first column or column start or end header pixel
 */
DvtDataGrid.prototype._getMaxLeftPixel = function () {
  return Math.max(Math.max(this.m_startColPixel,
    this.m_startColHeaderPixel),
  this.m_startColEndHeaderPixel);
};

/**
 * @returns {number} last row or row start or end header
 */
DvtDataGrid.prototype._getMaxBottom = function () {
  return Math.max(Math.max(this.m_endRow, this.m_endRowHeader), this.m_endRowEndHeader);
};

/**
 * @returns {number} first row or row start or end header
 */
DvtDataGrid.prototype._getMaxTop = function () {
  return Math.max(Math.max(this.m_startRow, this.m_startRowHeader), this.m_startRowEndHeader);
};

/**
 * @returns {number} last row or row start or end header pixel
 */
DvtDataGrid.prototype._getMaxBottomPixel = function () {
  return Math.max(Math.max(this.m_endRowPixel,
    this.m_endRowHeaderPixel),
  this.m_endRowEndHeaderPixel);
};

/**
 * @returns {number} first row or row start or end header pixel
 */
DvtDataGrid.prototype._getMaxTopPixel = function () {
  return Math.max(Math.max(this.m_startRowPixel,
    this.m_startRowHeaderPixel),
  this.m_startRowEndHeaderPixel);
};

/**
 * If we are about to remove a cell that is being edited, try to handle it first
 * @private
 */
DvtDataGrid.prototype._isScrollBackToEditable = function (longScroll) {
  var currentMode = this._getCurrentMode();
  var cell = this._getActiveElement();
  if (currentMode === 'edit' && (longScroll || this._isCellGoingToBeRemoved(cell))) {
    return this._handleExitEdit(null, cell);
  }
  return true;
};

/**
 * Check if the cell is supposed to be removed
 * @private
 * @param {Element|null} cell
 */
DvtDataGrid.prototype._isCellGoingToBeRemoved = function (cell) {
  if (!this._isHighWatermarkScrolling()) {
    if ((this.m_endRow - this.m_startRow) > this.MAX_ROW_THRESHOLD) {
      var top = this.getElementDir(cell.parentNode, 'top');
      var height = this.getElementHeight(cell);
      if (top + height < this.m_currentScrollTop ||
          top < this.m_currentScrollTop + this.getViewportHeight()) {
        return true;
      }
    }
    if ((this.m_endCol - this.m_startCol) > this.MAX_COLUMN_THRESHOLD) {
      var left = this.getElementDir(cell, 'left');
      var width = this.getElementWidth(cell);
      if (left + width < this.m_currentScrollLeft ||
          left < this.m_currentScrollLeft + this.getViewportHeight()) {
        return true;
      }
    }
  }
  return undefined;
};

/**
 * Remove cells along a given axis
 * @param {string} axis
 * @param {number} threshold
 * @param {boolean} isFromEnd
 * @private
 */
DvtDataGrid.prototype._removeCellsAlongAxis = function (axis, threshold, isFromEnd) {
  var j;
  var axisStart;
  var axisEnd;
  var axisHeaders;
  var axisLevelCount;
  var axisStartHeader;
  var dimension;
  var axisStartPixel;
  var axisEndPixel;
  var currentScroll;
  var otherAxis;
  var otherAxisStart;
  var otherAxisEnd;
  var dir;
  var totalDimensionChange = 0;
  var totalCountChange = 0;

  if (axis === 'row') {
    axisStart = this.m_startRow;
    axisEnd = this.m_endRow;
    axisHeaders = this.m_rowHeader;
    axisLevelCount = this.m_rowHeaderLevelCount;
    axisStartHeader = this.m_startRowHeader;
    dimension = 'height';
    axisStartPixel = this.m_startRowPixel;
    axisEndPixel = this.m_endRowPixel;
    currentScroll = this.m_currentScrollTop;
    otherAxis = 'column';
    otherAxisStart = this.m_startCol;
    otherAxisEnd = this.m_endCol;
    dir = 'top';
    j = isFromEnd ? axisEnd : axisStart;
  } else {
    axisStart = this.m_startCol;
    axisEnd = this.m_endCol;
    axisHeaders = this.m_colHeader;
    axisLevelCount = this.m_columnHeaderLevelCount;
    axisStartHeader = this.m_startColHeader;
    dimension = 'width';
    axisStartPixel = this.m_startColPixel;
    axisEndPixel = this.m_endColPixel;
    currentScroll = this.m_currentScrollLeft;
    otherAxis = 'row';
    otherAxisStart = this.m_startRow;
    otherAxisEnd = this.m_endRow;
    dir = this.getResources().isRTLMode() ? 'right' : 'left';
    j = isFromEnd ? axisEnd : axisStart;
  }


  while (j <= axisEnd && j >= axisStart) {
    var key = this._getKey(this._getHeaderByIndex(j, axisLevelCount - 1, axisHeaders,
      axisLevelCount, axisStartHeader), axis);
    if (key == null) {
      key = this._getKey(this._getCellByIndex(axis === 'column' ?
        this.createIndex(this.m_startRow, j) :
        this.createIndex(j, this.m_startCol)), axis);
    }
    var dimensionValue = this._getCellDimension(null, j, key, axis, dimension);
    if (isFromEnd ?
      (axisEndPixel - dimensionValue - totalDimensionChange > threshold) :
      (axisStartPixel + dimensionValue + totalDimensionChange < currentScroll - threshold)) {
      var otherExtent;
      for (var i = otherAxisStart; i <= otherAxisEnd; i += otherExtent) {
        var cell = this._getCellByIndex(axis === 'column' ?
          this.createIndex(i, j) : this.createIndex(j, i));
        var cellContext = cell[this.getResources().getMappedAttribute('context')];
        var axisExtent = cellContext.extents[axis];
        otherExtent = cellContext.extents[otherAxis];

        if (axisExtent === 1) {
          this._remove(cell);
        } else {
          cellContext.extents[axis] -= 1;
          this.setElementDir(cell, this.getElementDir(cell, dimension) - dimensionValue,
            dimension);
          if (!isFromEnd) {
            cellContext.indexes[axis] += 1;
            this.setElementDir(cell, this.getElementDir(cell, dir) + dimensionValue, dir);
          }
        }

        for (var k = 0; k < otherExtent; k++) {
          var index = axis === 'column' ? this.createIndex(i + k, j) : this.createIndex(j, i + k);
          this._removeIndexFromDatabodyMap(index);
        }
      }
      totalDimensionChange += dimensionValue;
      totalCountChange += 1;
      j = isFromEnd ? j - 1 : j + 1;
    } else {
      break;
    }
  }

  return { dimensionChange: totalDimensionChange, extentChange: totalCountChange };
};

/**
 * Removes all of the headers in the containing div up until the right value is less than the scroll position minus the threshold.
 * It is recuresively called on inner levels in the multi-level header case.
 * @param {Element} headersContainer
 * @param {Element|null} firstChild
 * @param {number} startPixel
 * @param {number} threshold
 * @param {string} className
 * @param {string} dimension
 * @param {string} dir
 * @param {number} scrollPosition
 * @returns {Object} object with keys extentChange, which denotes how many header
 *      indexes were removed under the parent and dimensionChange which is the
 *      total dimensions of the headers removed
 */
DvtDataGrid.prototype.removeHeadersFromStartOfContainer = function (
  headersContainer, firstChild, startPixel, threshold, className, dimension, dir, scrollPosition
) {
  var removedHeaders = 0;
  var removedDimensionValue = 0;
  var element = firstChild == null ? headersContainer.firstChild : firstChild.nextSibling;

  if (element == null) {
    return { extentChange: 0, dimensionChange: 0 };
  }

  var isHeader = this.m_utils.containsCSSClassName(element, className);
  var header = isHeader ? element : element.firstChild;
  var dimensionValue = this.getElementDir(header, dimension);

  while (startPixel + dimensionValue < scrollPosition - threshold) {
    this._remove(element);
    removedDimensionValue += dimensionValue;
    removedHeaders += isHeader ? 1 : this._getAttribute(element, 'extent', true);
    // eslint-disable-next-line no-param-reassign
    startPixel += dimensionValue;

    element = firstChild == null ? headersContainer.firstChild : firstChild.nextSibling;
    if (element == null) {
      return { extentChange: removedHeaders, dimensionChange: removedDimensionValue };
    }
    isHeader = this.m_utils.containsCSSClassName(element, className);
    header = isHeader ? element : element.firstChild;
    dimensionValue = this.getElementDir(header, dimension);
  }

  if (!isHeader) {
    var returnVal = this.removeHeadersFromStartOfContainer(element, element.firstChild,
      startPixel, threshold, className,
      dimension, dir, scrollPosition);
    this._setAttribute(element, 'start',
      this._getAttribute(element, 'start', true) + returnVal.extentChange);
    this._setAttribute(element, 'extent',
      this._getAttribute(element, 'extent', true) - returnVal.extentChange);
    this.setElementDir(header,
      this.getElementDir(header, dir) + returnVal.dimensionChange,
      dir);
    this.setElementDir(header,
      this.getElementDir(header, dimension) - returnVal.dimensionChange,
      dimension);

    removedHeaders += returnVal.extentChange;
    removedDimensionValue += returnVal.dimensionChange;
  }

  return { extentChange: removedHeaders, dimensionChange: removedDimensionValue };
};

/**
 * Removes all of the headers in the containing div up until the right value is less than the specified threshold.
 * It is recuresively called on inner levels in the multi-level header case.
 * @param {Element} headersContainer
 * @param {number} endPixel
 * @param {number} threshold
 * @param {string} className
 * @param {string} dimension
 * @returns {Object} object with keys extentChange, which denotes how many header
 *      indexes were removed under the parent and dimensionChange which is the
 *      total width of the headers removed
 */
DvtDataGrid.prototype.removeHeadersFromEndOfContainer =
  function (headersContainer, endPixel, threshold, className, dimension) {
    var removedHeaders = 0;
    var removedHeadersDimension = 0;
    var element = headersContainer.lastChild;
    var isHeader = this.m_utils.containsCSSClassName(element, className);
    var header = isHeader ? element : element.firstChild;
    var dimensionValue = this.getElementDir(header, dimension);

    while (endPixel - dimensionValue > threshold) {
      this._remove(element);

      removedHeadersDimension += dimensionValue;
      removedHeaders += isHeader ? 1 : this._getAttribute(element, 'extent', true);

      // eslint-disable-next-line no-param-reassign
      endPixel -= dimensionValue;

      element = headersContainer.lastChild;
      isHeader = this.m_utils.containsCSSClassName(element, className);
      header = isHeader ? element : element.firstChild;
      dimensionValue = this.getElementDir(header, dimension);
    }

    if (!isHeader) {
      var returnVal = this.removeHeadersFromEndOfContainer(element, endPixel, threshold,
        className, dimension);

      this._setAttribute(element, 'extent',
        this._getAttribute(element, 'extent', true) - returnVal.extentChange);
      this.setElementDir(header,
        this.getElementDir(header, dimension) - returnVal.dimensionChange,
        dimension);

      removedHeaders += returnVal.extentChange;
      removedHeadersDimension += returnVal.dimensionChange;
    }

    return { extentChange: removedHeaders, dimensionChange: removedHeadersDimension };
  };

/**
 * Remove column start and end headers to the left of the current viewport
 */
DvtDataGrid.prototype.removeColumnHeadersFromLeft = function () {
  var colThreshold;
  var returnVal;

  // clean up left column headers
  if ((this.m_endColHeader - this.m_startColHeader) > this.MAX_COLUMN_THRESHOLD) {
    var colHeaderContent = this.m_colHeader.firstChild;
    colThreshold = this.getColumnThreshold();
    if (this.m_startColHeaderPixel <= this.m_currentScrollLeft - colThreshold) {
      returnVal = this.removeHeadersFromStartOfContainer(colHeaderContent, null,
        this.m_startColHeaderPixel,
        colThreshold,
        this.getMappedStyle('colheadercell'),
        'width',
        this.getResources().isRTLMode() ?
          'right' : 'left',
        this.m_currentScrollLeft);

      this.m_startColHeaderPixel += returnVal.dimensionChange;
      this.m_startColHeader += returnVal.extentChange;
    }
  }

  if ((this.m_endColEndHeader - this.m_startColEndHeader) > this.MAX_COLUMN_THRESHOLD) {
    var colEndHeaderContent = this.m_colEndHeader.firstChild;
    colThreshold = this.getColumnThreshold();
    if (this.m_startColEndHeaderPixel < this.m_currentScrollLeft - colThreshold) {
      returnVal = this.removeHeadersFromStartOfContainer(colEndHeaderContent, null,
        this.m_startColEndHeaderPixel,
        colThreshold,
        this.getMappedStyle('colendheadercell'),
        'width',
        this.getResources().isRTLMode() ?
          'right' : 'left',
        this.m_currentScrollLeft);

      this.m_startColEndHeaderPixel += returnVal.dimensionChange;
      this.m_startColEndHeader += returnVal.extentChange;
    }
  }
};

/**
 * Remove cells to the left of the current viewport
 * @param {Element} databody - the root of the databody
 */
DvtDataGrid.prototype.removeColumnsFromLeft = function (databody) {
  // clean up right column headers
  if ((this.m_endCol - this.m_startCol) > this.MAX_COLUMN_THRESHOLD) {
    var databodyContent = databody.firstChild;
    var cells = databodyContent.childNodes;
    var colThreshold = this.getColumnThreshold();

    // no rows in databody, nothing to remove
    if (cells.length < 1) {
      return;
    }

    var returnVal = this._removeCellsAlongAxis('column', colThreshold, false);
    this.m_startColPixel += returnVal.dimensionChange;
    this.m_startCol += returnVal.extentChange;
  }
};

/**
 * Remove column start and end headers to the right of the current viewport
 */
DvtDataGrid.prototype.removeColumnHeadersFromRight = function () {
  var colHeaderContent;
  var returnVal;
  var colThreshold = this.m_currentScrollLeft + this.getViewportWidth() + this.getColumnThreshold();

  // clean up right column headers
  if ((this.m_endColHeader - this.m_startColHeader) > this.MAX_COLUMN_THRESHOLD) {
    colHeaderContent = this.m_colHeader.firstChild;
    // don't clean up if end of row header is not below the bottom of viewport
    if (this.m_endColHeaderPixel > colThreshold) {
      if (this.m_stopColumnHeaderFetch) {
        this.m_stopColumnHeaderFetch = false;
      }

      returnVal = this.removeHeadersFromEndOfContainer(colHeaderContent,
        this.m_endColHeaderPixel,
        colThreshold,
        this.getMappedStyle('colheadercell'),
        'width');

      this.m_endColHeaderPixel -= returnVal.dimensionChange;
      this.m_endColHeader -= returnVal.extentChange;
    }
  }

  // clean up right column end headers
  if ((this.m_endColEndHeader - this.m_startColEndHeader) > this.MAX_COLUMN_THRESHOLD) {
    colHeaderContent = this.m_colEndHeader.firstChild;
    // don't clean up if end of row header is not below the bottom of viewport
    if (this.m_endColEndHeaderPixel > colThreshold) {
      if (this.m_stopColumnEndHeaderFetch) {
        this.m_stopColumnEndHeaderFetch = false;
      }

      returnVal = this.removeHeadersFromEndOfContainer(colHeaderContent,
        this.m_endColEndHeaderPixel,
        colThreshold,
        this.getMappedStyle('colendheadercell'),
        'width');

      this.m_endColEndHeaderPixel -= returnVal.dimensionChange;
      this.m_endColEndHeader -= returnVal.extentChange;
    }
  }
};

/**
 * Remove cells to the right of the current viewport
 * @param {Element} databody - the root of the databody
 */
DvtDataGrid.prototype.removeColumnsFromRight = function (databody) {
  // clean up right column headers
  if ((this.m_endCol - this.m_startCol) > this.MAX_COLUMN_THRESHOLD) {
    var databodyContent = databody.firstChild;
    var cells = databodyContent.childNodes;
    var threshold = this.m_currentScrollLeft + this.getViewportWidth() + this.getColumnThreshold();

    // don't clean up if end of row header is not below the bottom of viewport
    // no rows in databody, nothing to remove
    if (this.m_endColPixel <= threshold || cells.length < 1) {
      return;
    }

    if (this.m_stopColumnFetch) {
      this.m_stopColumnFetch = false;
    }

    var returnVal = this._removeCellsAlongAxis('column', threshold, true);
    this.m_endColPixel -= returnVal.dimensionChange;
    this.m_endCol -= returnVal.extentChange;
  }
};

/**
 * Remove row start and end headers above the current viewport
 */
DvtDataGrid.prototype.removeRowHeadersFromTop = function () {
  var returnVal;
  var rowThreshold;

  if ((this.m_endRowHeader - this.m_startRowHeader) > this.MAX_ROW_THRESHOLD) {
    var rowHeaderContent = this.m_rowHeader.firstChild;
    rowThreshold = this.getRowThreshold();
    if (!(this.m_startRowHeaderPixel >= this.m_currentScrollTop - rowThreshold)) {
      returnVal = this.removeHeadersFromStartOfContainer(rowHeaderContent, null,
        this.m_startRowHeaderPixel,
        rowThreshold,
        this.getMappedStyle('rowheadercell'),
        'height', 'top', this.m_currentScrollTop);

      this.m_startRowHeaderPixel += returnVal.dimensionChange;
      this.m_startRowHeader += returnVal.extentChange;
    }
  }

  if ((this.m_endRowEndHeader - this.m_startRowEndHeader) > this.MAX_ROW_THRESHOLD) {
    var rowEndHeaderContent = this.m_rowEndHeader.firstChild;
    rowThreshold = this.getRowThreshold();
    if (!(this.m_startRowEndHeaderPixel >= this.m_currentScrollTop - rowThreshold)) {
      returnVal = this.removeHeadersFromStartOfContainer(rowEndHeaderContent, null,
        this.m_startRowEndHeaderPixel,
        rowThreshold,
        this.getMappedStyle('rowendheadercell'),
        'height', 'top', this.m_currentScrollTop);

      this.m_startRowEndHeaderPixel += returnVal.dimensionChange;
      this.m_startRowEndHeader += returnVal.extentChange;
    }
  }
};

/**
 * Remove rows/cells above the current viewport
 * @param {Element} databody - the root of the databody
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype.removeRowsFromTop = function (databody) {
  if ((this.m_endRow - this.m_startRow) > this.MAX_ROW_THRESHOLD) {
    var rowThreshold = this.getRowThreshold();
    if (this.m_startRowPixel >= this.m_currentScrollTop - rowThreshold) {
      return;
    }

    // remove all rows from top until the threshold is reached
    var returnVal = this._removeCellsAlongAxis('row', rowThreshold, false);
    this.m_startRowPixel += returnVal.dimensionChange;
    this.m_startRow += returnVal.extentChange;
  }
};

/**
 * Remove row start and end headers below the current viewport
 */
DvtDataGrid.prototype.removeRowHeadersFromBottom = function () {
  var returnVal;
  var rowThreshold = this.m_currentScrollTop + this.getViewportHeight() + this.getRowThreshold();

  // clean up bottom row headers
  if ((this.m_endRowHeader - this.m_startRowHeader) > this.MAX_ROW_THRESHOLD) {
    var rowHeaderContent = this.m_rowHeader.firstChild;
    // don't clean up if end of row header is not below the bottom of viewport
    if (!(this.m_endRowHeaderPixel <= rowThreshold)) {
      if (this.m_stopRowHeaderFetch) {
        this.m_stopRowHeaderFetch = false;
      }

      returnVal = this.removeHeadersFromEndOfContainer(rowHeaderContent,
        this.m_endRowHeaderPixel,
        rowThreshold,
        this.getMappedStyle('rowheadercell'),
        'height');

      this.m_endRowHeaderPixel -= returnVal.dimensionChange;
      this.m_endRowHeader -= returnVal.extentChange;
    }
  }

  // clean up bottom row headers
  if ((this.m_endRowEndHeader - this.m_startRowEndHeader) > this.MAX_ROW_THRESHOLD) {
    var rowEndHeaderContent = this.m_rowEndHeader.firstChild;
    // don't clean up if end of row header is not below the bottom of viewport
    if (!(this.m_endRowEndHeaderPixel <= rowThreshold)) {
      if (this.m_stopRowEndHeaderFetch) {
        this.m_stopRowEndHeaderFetch = false;
      }

      returnVal = this.removeHeadersFromEndOfContainer(rowEndHeaderContent,
        this.m_endRowEndHeaderPixel,
        rowThreshold,
        this.getMappedStyle('rowendheadercell'),
        'height');

      this.m_endRowEndHeaderPixel -= returnVal.dimensionChange;
      this.m_endRowEndHeader -= returnVal.extentChange;
    }
  }
};

/**
 * Remove rows/cells below the current viewport
 * @param {Element} databody - the root of the databody
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype.removeRowsFromBottom = function (databody) {
  if ((this.m_endRow - this.m_startRow) > this.MAX_ROW_THRESHOLD) {
    var threshold = this.m_currentScrollTop + this.getViewportHeight() + this.getRowThreshold();

    // don't clean up if end of row header is not below the bottom of viewport
    if (this.m_endRowPixel <= threshold) {
      return;
    }

    if (this.m_stopRowFetch) {
      this.m_stopRowFetch = false;
    }

    var returnVal = this._removeCellsAlongAxis('row', threshold, true);
    this.m_endRowPixel -= returnVal.dimensionChange;
    this.m_endRow -= returnVal.extentChange;
  }
};

/** ********************************* end scrolling/virtualization ************************************/

/** ********************************* start dom event handling ***************************************/
/**
 * Handle the context menu gesture
 * @param {Event} event the event of the context menu gesture
 * @param {string} eventType keyboard/touch/mouse
 * @param {Function} callback where to pass the data back
 */
DvtDataGrid.prototype.handleContextMenuGesture = function (event, eventType, callback) {
  var index;
  var capabilities;
  var launcher;

  // if we are on a touch device and in a cell we need to set the correct active
  // and call focus before triggering the context menu to open. headers take
  // care of this by setting active in the 300ms callback for tap+short hold
  var target = /** @type {Element} */ (event.originalEvent.target);
  var element = this.findCell(target);
  var isHeader = false;
  var axis;
  if (element === null) {
    element = this.findHeader(target);
    if (element) {
      isHeader = true;
      axis = this.getHeaderCellAxis(element);
    }
  }
  if (eventType === 'touch' && element != null) {
    index = isHeader ? this.getHeaderCellIndex(element) : this.getCellIndexes(element);
    let insideSelection = isHeader ?
        this._isHeaderInsideSelection(index, axis) : this._isContainSelection(index);
    // if right click and inside multiple selection or current active do not change anything
    if ((!this.isMultipleSelection() || !insideSelection) ||
        (this._isDatabodyCellActive() && index.row !== this.m_active.indexes.row &&
          index.column !== this.m_active.indexes.column)) {
      if (this._isSelectionEnabled()) {
        this.handleDatabodyClickSelection(event.originalEvent);
      } else {
        // activate on a tap
        this.handleDatabodyClickActive(event.originalEvent);
      }
    }
  }

  // first check if we are invoking on an editable or clickable element, if so bail
  if (this.m_utils._isNodeEditableOrClickable(target, this.m_root)) {
    return;
  }

  // enable and disable context menu items depending on capability of the datasource and options
  // if the action was performed on a cell
  if (element != null && !isHeader) {
    index = this.getCellIndexes(element);
    // if fired from inside a multiple selection
    if (this.isMultipleSelection() && this._isContainSelection(index)) {
      launcher = this._getActiveElement();
      // if there is an active cell we want that to be the launcher of the context menu so
      // that focus can be restored to it. If it fired form the keyboard open with launcher and context
      // of the active cell, if right click or touch open with the context of the clicked cell
      if (this._isDatabodyCellActive()) {
        // handle the case where the active element is no longer rendered (ie scrolled off viewport in virtual scroll)
        // we may have focus loss in this case on escape key but the context menu will open. alternative is to do
        // work that anytime focus scrolls off the screen the root node becomes focusable and will go back to the active node
        // but this is consistant with current offscreen focus behavior.
        if (launcher == null) {
          launcher = element;
        }
        capabilities = eventType === 'keyboard' ?
          this._getCellCapability(launcher) : this._getCellCapability(launcher, element);
        capabilities.resizeFitToContent = 'disable';
      } else {
        // there is the case where header is active and entire row/column selected
        // the launcher will be the active header, and the context of the menu will be relative to the active header
        capabilities = this._getHeaderCapability(launcher, element);
        capabilities.resizeFitToContent = 'enable';
      }
    } else {
      // open on the cell with its context
      launcher = element;
      capabilities = this._getCellCapability(launcher);
      capabilities.resizeFitToContent = 'disable';
    }
    if (this.m_selectionFrontier && this.m_selectionFrontier.axis === 'row') {
      capabilities.resizeWidth = 'disable';
    } else if (this.m_selectionFrontier && this.m_selectionFrontier.axis === 'column') {
      capabilities.resizeHeight = 'disable';
    }
  } else {
    element = this.findHeader(target) || this.findLabel(target);
    if (element == null) {
      // not a header or cell don't do anything
      var disable = 'disable';
      capabilities = {
        resize: disable,
        resizeWidth: disable,
        resizeHeight: disable,
        sortRow: disable,
        sortCol: disable,
        cut: disable,
        paste: disable,
        sortColAsc: disable,
        sortColDsc: disable,
        sortRowAsc: disable,
        sortRowDsc: disable
      };
      launcher = element;
    } else {
      capabilities = this._getHeaderCapability(element);
      if (isHeader) {
        capabilities.resizeFitToContent = 'enable';
      } else {
        capabilities.resizeFitToContent = 'disable';
      }
      if ((axis === 'column' || axis === 'columnEnd') &&
          this.m_selectionFrontier && this.m_selectionFrontier.axis === 'column') {
        capabilities.resizeHeight = 'disable';
      } else if ((axis === 'row' || axis === 'rowEnd') &&
          this.m_selectionFrontier && this.m_selectionFrontier.axis === 'row') {
        capabilities.resizeWidth = 'disable';
      }
      launcher = element;
    }
  }

  callback.call(null, { capabilities: capabilities, launcher: launcher }, event, eventType);
};
/**
 * Get the capabilities for context menu opened on a cell
 * @param {Element} cell the cell whose context we want
 * @param {Element=} actualCell the cell with context menu opened on it
 * @return {Object} capabilities object with props resize, resizeWidth, resizeHeight, sortRow, sortCol, cut, paste
 * @private
 */
DvtDataGrid.prototype._getCellCapability = function (cell, actualCell) {
  var sameColumn = true;
  var sameRow = true;
  var disable = 'disable';
  var enable = 'enable';
  var capabilities = {
    resize: disable,
    resizeWidth: disable,
    resizeHeight: disable,
    sortRow: disable,
    sortCol: disable,
    cut: disable,
    cutCells: disable,
    copyCells: disable,
    paste: disable,
    pasteCells: disable,
    fill: disable,
    sortColAsc: disable,
    sortColDsc: disable,
    sortRowAsc: disable,
    sortRowDsc: disable
  };

  // if there is an actual cell that means we want the context relative to that cell,
  // so if it is the same column, our column operations (resize width, sort column) can
  // be utilized. If it's in the same row the row operations (resize height, sort row, cut, paste)
  // can be utilized
  if (actualCell != null) {
    sameColumn = this._getIndex(cell, 'column') === this._getIndex(actualCell, 'column');
    sameRow = this._getKey(cell, 'row') === this._getKey(actualCell, 'row');
    if (sameRow === false && sameColumn === false) {
      return capabilities;
    }
  }
  if (this.m_options.isCopyEnabled()) {
    capabilities.copyCells = enable;
  }
  if (this.m_options.isCutEnabled()) {
    capabilities.cutCells = enable;
  }
  if (this.m_options.isPasteEnabled()) {
    capabilities.pasteCells = enable;
  }
  if (this.m_options.isFloodFillEnabled()) {
    capabilities.fill = enable;
  }
  var rowHeader = this.getHeaderFromCell(cell, 'row');
  var columnHeader = this.getHeaderFromCell(cell, 'column');
  var resizable = this.getResources().getMappedAttribute('resizable');
  var sortable = this.getResources().getMappedAttribute('sortable');

  if (columnHeader != null && sameColumn) {
    if (columnHeader.getAttribute(resizable) === 'true') {
      capabilities.resize = enable;
      capabilities.resizeWidth = enable;
    }
    if (columnHeader.getAttribute(sortable) === 'true') {
      capabilities.sortCol = enable;
      capabilities.sortColAsc = enable;
      capabilities.sortColDsc = enable;
      var sorted = columnHeader.getAttribute(this.getResources().getMappedAttribute('sortDir'));
      if (sorted === 'ascending') {
        capabilities.sortColAsc = disable;
      } else if (sorted === 'descending') {
        capabilities.sortColDsc = disable;
      }
    }
  }
  if (rowHeader != null && sameRow) {
    if (this._isMoveEnabled('row')) {
      capabilities.cut = enable;
      capabilities.paste = enable;
    }
    if (rowHeader.getAttribute(sortable) === 'true') {
      capabilities.sortRow = enable;
      capabilities.sortRowAsc = enable;
      capabilities.sortRowDsc = enable;
    }
    if (rowHeader != null) {
      if (rowHeader.getAttribute(resizable) === 'true') {
        capabilities.resize = enable;
        capabilities.resizeHeight = enable;
      }
      if (rowHeader.getAttribute(sortable) === 'true') {
        capabilities.sortRow = enable;
      }
    }
  }
  return capabilities;
};

/**
 * Get the capabilities for context menu opened on a header
 * @param {Element} header the header whose context we want
 * @param {Element=} actualCell the cell that we are actually opening on
 * @return {Object} capabilities object with props resizeWidth, resizeHeight, sortRow, sortCol
 * @private
 */
DvtDataGrid.prototype._getHeaderCapability = function (header, actualCell) {
  var sameColumn = true;
  var sameRow = true;
  var disable = 'disable';
  var enable = 'enable';
  var capabilities = {
    resize: disable,
    resizeWidth: disable,
    resizeHeight: disable,
    sortRow: disable,
    sortCol: disable,
    cut: disable,
    paste: disable,
    sortColAsc: disable,
    sortColDsc: disable,
    sortRowAsc: disable,
    sortRowDsc: disable
  };

  // if there is an actual cell that means we want the context relative to that cell,
  // so if it is the same column, our column operations (resize width, sort column) can
  // be utilized. If it's in the same row the row operations (resize height, sort row, cut, paste)
  // can be utilized
  if (actualCell != null) {
    sameColumn = this.getHeaderCellIndex(header) === this._getIndex(actualCell, 'column');
    sameRow = this._getKey(header, 'row') === this._getKey(actualCell, 'row');
    if (sameRow === false && sameColumn === false) {
      return capabilities;
    }
  }

  var axis = this.getHeaderCellAxis(header);
  var resizable = this.getResources().getMappedAttribute('resizable');
  var sortable = this.getResources().getMappedAttribute('sortable');

  if (header !== null) {
    if ((axis === 'column' || axis === 'columnEnd') && sameColumn) {
      if (header.getAttribute(resizable) === 'true') {
        capabilities.resizeWidth = enable;
        capabilities.resize = enable;
      }
      capabilities.resizeHeight = this.m_options.isResizable(axis, 'height');
      if (header.getAttribute(sortable) === 'true') {
        capabilities.sortCol = enable;
        capabilities.sortColAsc = enable;
        capabilities.sortColDsc = enable;
        var sorted = header.getAttribute(this.getResources().getMappedAttribute('sortDir'));
        if (sorted === 'ascending') {
          capabilities.sortColAsc = disable;
        } else if (sorted === 'descending') {
          capabilities.sortColDsc = disable;
        }
      }
    } else if (sameRow) {
      if (this._isMoveEnabled('row')) {
        capabilities.cut = enable;
        capabilities.paste = enable;
      }
      if (header.getAttribute(resizable) === 'true') {
        capabilities.resize = enable;
        capabilities.resizeHeight = enable;
      }
      capabilities.resizeWidth = this.m_options.isResizable(axis, 'width');
      if (header.getAttribute(sortable) === 'true') {
        capabilities.sortRow = enable;
        capabilities.sortRowAsc = enable;
        capabilities.sortRowDsc = enable;
        let isRowSorted = header.getAttribute(this.getResources().getMappedAttribute('sortDir'));
        if (isRowSorted === 'ascending') {
          capabilities.sortRowAsc = disable;
        } else if (isRowSorted === 'descending') {
          capabilities.sortRowDsc = disable;
        }
      }
    }
  }
  capabilities.resize = (capabilities.resizeHeight === enable ||
                         capabilities.resizeWidth === enable) ? enable : disable;

  return capabilities;
};

/**
 * Handle the callback from the widget to resize or sort.
 * @param {Event} event - the original contextmenu event
 * @param {string} id - the id returned from the context menu
 * @param value - the value set in the dialog on resizing
 */
DvtDataGrid.prototype.handleContextMenuReturn = function (event, id, value) {
  var target;
  var direction;

  // the target is the active element at all times
  if (this.m_active != null) {
    target = this._getActiveElement();
  }

  if (id === this.m_resources.getMappedCommand('resizeHeight') ||
      id === this.m_resources.getMappedCommand('resizeWidth')) {
    if ((this.isResizeEnabled())) {
      // target may not be (event.target)
      this.handleContextMenuResize(event, id, value, target);
    }
  } else if (id === this.m_resources.getMappedCommand('resizeFitToContent')) {
    let parent;
    let isCell = this.findCell(event.target);
    if (isCell) {
      parent = this.getHeaderFromCell(isCell, this.m_selectionFrontier.axis);
    }
    if (!parent) {
      parent = this.findHeader(event.target);
    }
    if (!parent) {
      parent = this.findLabel(event.target);
    }
    if (parent) {
      this.m_resizingElement = parent;
    }
    const resizingElementAxis = this.getHeaderCellAxis(this.m_resizingElement);
    const resizingElementLevel = this.getHeaderCellLevel(this.m_resizingElement);
    let allowResizing = false;
    if ((resizingElementAxis === 'row' || resizingElementAxis === 'rowEnd') && resizingElementLevel === this.m_rowHeaderLevelCount - 1) {
      allowResizing = true;
    } else if ((resizingElementAxis === 'column' || resizingElementAxis === 'columnEnd') && resizingElementLevel === this.m_columnHeaderLevelCount - 1) {
      allowResizing = true;
    }
    if (allowResizing && this.isResizeEnabled()) {
      this._getHeadersForResizeFitToContent(event);
    }
  } else if (id === this.m_resources.getMappedCommand('sortColAsc') ||
              id === this.m_resources.getMappedCommand('sortColDsc')) {
    direction = id === this.m_resources.getMappedCommand('sortColAsc') ?
      'ascending' : 'descending';
    if (this.m_utils.containsCSSClassName(target, this.getMappedStyle('cell'))) {
      target = this.getHeaderFromCell(target, 'column');
    }
    if (this._isDOMElementSortable(target)) {
      this._handleCellSort(event, direction, target);
    }
  } else if (id === this.m_resources.getMappedCommand('sortRowAsc') ||
              id === this.m_resources.getMappedCommand('sortRowDsc')) {
    direction = id === this.m_resources.getMappedCommand('sortRowAsc') ?
      'ascending' : 'descending';
    if (this.m_utils.containsCSSClassName(target, this.getMappedStyle('cell'))) {
      target = this.getHeaderFromCell(target, 'row');
    }
    if (this._isDOMElementSortable(target)) {
      this._handleCellSort(event, direction, target);
    }
  } else if (id === this.m_resources.getMappedCommand('cut')) {
    this._handleCut(event, target);
  } else if (id === this.m_resources.getMappedCommand('paste')) {
    this._handlePaste(event, target);
  } else if (id === this.m_resources.getMappedCommand('cutCells')) {
    this._handleCutCells(event, target);
  } else if (id === this.m_resources.getMappedCommand('copyCells')) {
    this._handleCopyCells(event, target);
  } else if (id === this.m_resources.getMappedCommand('pasteCells')) {
    this._handlePasteCells(event, target);
  } else if (id === this.m_resources.getMappedCommand('autoFill')) {
    this._handleAutofill(event, target);
  } else if (id === this.m_resources.getMappedCommand('discontiguousSelection')) {
    // handle discontiguous selection context menu
    this.setDiscontiguousSelectionMode(value);
  }
};

/**
 * Determined if sort is supported for the specified axis.
 * @param {string} axis the axis which we check whether sort is supported.
 * @param {Object} headerContext the header context object
 * @private
 */
DvtDataGrid.prototype._isSortEnabled = function (axis, headerContext) {
  var capability = this.getDataSource().getCapability('sort');
  var sortable = this.m_options.isSortable(axis, headerContext);
  if ((sortable === 'enable' || sortable === 'auto') &&
      (capability === 'full' || capability === axis)) {
    if (this._isDataGridProvider()) {
      if (headerContext.metadata.sortDirection != null) {
        return true;
      }
      return false;
    }

    return true;
  }

  return false;
};

/**
 * Checks if an element is a parentNode of a traditional hierarchy.
 * @param {Object} headerContext the header context object
 * @private
 */
DvtDataGrid.prototype._isParentNode = function (headerContext) {
  if (this._isDataGridProvider()) {
    return (headerContext.metadata.expanded && headerContext.metadata.expanded !== null);
  }
  return false;
};

/**
 * Checks if an element is a parentNode of a hierarichal group.
 * @param {Object} headerContext the header context object
 * @private
 */
DvtDataGrid.prototype._isHierarchicalGroup = function (headerContext) {
  return headerContext.metadata.treeDepth == null;
};

/**
 * Checks if an element is a leafNode of a traditional hierarchy.
 * @param {Object} headerContext the header context object
 * @private
 */
 DvtDataGrid.prototype._isLeafNode = function (headerContext) {
  if (!headerContext.metadata) {
    return false;
  }
  const treeDepth = headerContext.metadata.treeDepth;
  if (this._isDataGridProvider()) {
    return treeDepth != null && treeDepth !== 0;
  }
  return false;
};

/**
 * Determined if sort is supported for the specified element.
 * @param {Element|undefined} element to check if sorting should be on
 * @private
 */
DvtDataGrid.prototype._isDOMElementSortable = function (element) {
  if (element == null) {
    return false;
  }
  var header = this.findHeader(element);
  if (header == null) {
    return false;
  }
  return header.getAttribute(this.getResources().getMappedAttribute('sortable')) === 'true';
};

/**
 * Check if selection enabled by options on the grid
 * @return {boolean} true if selection enabled
 * @private
 */
DvtDataGrid.prototype._isSelectionEnabled = function () {
  return (this.m_options.getSelectionCardinality() !== 'none');
};

/**
 * Check whether multiple row/cell selection is allowed by options on the grid
 * @return {boolean} true if multipl selection enabled
 */
DvtDataGrid.prototype.isMultipleSelection = function () {
  return (this.m_options.getSelectionCardinality() === 'multiple');
};

/**
 * Check if resizing enabled on any header by options on the grid
 * @return {boolean} true if resize enabled
 */
DvtDataGrid.prototype.isResizeEnabled = function () {
  return (this.m_options.isResizable('row', 'width') ||
          this.m_options.isResizable('row', 'height') ||
          this.m_options.isResizable('column', 'width') ||
          this.m_options.isResizable('column', 'height') ||
          this.m_options.isResizable('rowEnd', 'width') ||
          this.m_options.isResizable('rowEnd', 'height') ||
          this.m_options.isResizable('columnEnd', 'width') ||
          this.m_options.isResizable('columnEnd', 'height'));
};

/**
 * Check if resizing enabled on a specific header
 * @param {string} axis the axis which we check whether sort is supported.
 * @param {Object} headerContext the header context object
 * @return {boolean} true if resize enabled
 */
DvtDataGrid.prototype._isHeaderResizeEnabled = function (axis, headerContext) {
  var resizable;
  if (axis === 'column' || axis === 'columnEnd') {
    resizable = this.m_options.isResizable(axis, 'width', headerContext);
    return resizable === 'enable';
  } else if (axis === 'row' || axis === 'rowEnd') {
    resizable = this.m_options.isResizable(axis, 'height', headerContext);
    return resizable === 'enable';
  }
  return false;
};

/**
 * Handle mousemove events on the document
 * @param {Event} event - mousemove event on the document
 */
DvtDataGrid.prototype.handleMouseMove = function (event) {
  if (this.isResizeEnabled() && (this.m_databodyDragState === false)) {
    this.handleResize(event);
  }
};

/**
 * Handle row header mousemove events on the document
 * @param {Event} event - mousemove event on the document
 */
DvtDataGrid.prototype.handleRowHeaderMouseMove = function (event) {
  if (event.buttons === 0) {
    this.handleMouseUp(event);
  }
  if (this.m_databodyMove) {
    this._handleMove(event);
  } else if (this.m_headerDragState &&
    ((this.m_selectionFrontier && this.m_selectionFrontier.axis &&
      this.m_selectionFrontier.axis.indexOf('row') !== -1) ||
    (this.m_deselectInfo && this.m_deselectInfo.axis && this.m_deselectInfo.axis.indexOf('row') !== -1))) {
    this.extendSelectionHeader(event.target, event, true, this.m_deselectInProgress);
  } else if (!this.m_isResizing) {
    // if it is resizing use the mouse move on the document
    this.handleMouseMove(event);
  }
};

/**
 * Handle row header mousemove events on the document
 * @param {Event} event - mousemove event on the document
 */
DvtDataGrid.prototype.handleColumnHeaderMouseMove = function (event) {
  if (event.buttons === 0) {
    this.handleMouseUp(event);
  }
  if (this.m_headerDragState &&
    ((this.m_selectionFrontier && this.m_selectionFrontier.axis &&
    this.m_selectionFrontier.axis.indexOf('column') !== -1) ||
  (this.m_deselectInfo && this.m_deselectInfo.axis && this.m_deselectInfo.axis.indexOf('column') !== -1))) {
    this.extendSelectionHeader(event.target, event, true, this.m_deselectInProgress);
  } else if (!this.m_isResizing) {
    // if it is resizing use the mouse move on the document
    this.handleMouseMove(event);
  }
};

DvtDataGrid.prototype.handleHeaderLabelMouseMove = function (event) {
  if (!this.m_isResizing) {
    // if it is resizing use the mouse move on the document
    this.handleMouseMove(event);
  }
};

/**
 * Handle mousedown events on the headers
 * @param {Event} event - mousedown event on the headers
 */
DvtDataGrid.prototype.handleHeaderMouseDown = function (event) {
  var processed;

  this._exitActionableMode();
  var target = /** @type {Element} */ (event.target);

  if (this._isEditOrEnter()) {
    var cell = this._getActiveElement();
    if (this._leaveEditing(event, cell, false) === false) {
      return;
    }
  }

  if (this._isDisclosureIcon(target)) {
    return;
  }

  // only perform events on left mouse, (right in rtl culture)
  if (event.button === 0) {
    // if mousedown in an icon it the click event will handle mousedown/up
    if ((this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortascending')) ||
         this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortdescending')))
        && this._isDOMElementSortable(target)) {
      event.preventDefault();
      this._handleSortIconMouseDown(target);
      return;
    } else if (this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortIcon')) &&
      this._isDOMElementSortable(target.lastChild)) {
        event.preventDefault();
        this._handleSortIconMouseDown(target.lastChild);
        return;
    }

    // handle resize movements first if we're on the border
    if (this.isResizeEnabled()) {
      processed = this.handleResizeMouseDown(event);
      this._highlightResizeMouseDown();
    }

    // if our move is enabled make sure our row has the active cell in it
    var ctrlKey = this.m_utils.ctrlEquivalent(event);
    if (!this.m_isResizing && !ctrlKey && this._isMoveOnElementEnabled(this.findHeader(target))) {
      this.m_databodyMove = true;
      this.m_currentX = event.pageX;
      this.m_currentY = event.pageY;
      processed = true;
    }
  }
  // activate header on click or right click
  // checking the cursor value to not change selection on resize cursor.
  if (!this.m_isResizing && this.manageHeaderCursor(event, false) === 'default') {
    if (!this.m_root.contains(document.activeElement) ||
        document.activeElement === this.m_root) {
      this.m_externalFocus = true;
    }

    var selectionMode = this.m_options.getSelectionMode();
    var header = this.findHeader(target);
    var cellContext = header[this.getResources().getMappedAttribute('context')];

    // if click or right click we want to adjust the selction
    // no else so that we can select a cell in the same row as long as no drag
    // check if selection is enabled
    if (this._isSelectionEnabled() && this.isMultipleSelection() &&
        !(selectionMode === 'row' && cellContext.axis.indexOf('row') === -1) &&
        !this.m_databodyMove) {
      // only allow drag on left click
      if (event.button === 0) {
        this.m_headerDragState = true;
      }

      this.handleHeaderClickSelection(event);
    } else if (selectionMode === 'row' &&
               cellContext.axis.indexOf('row') !== -1 &&
               this._isSelectionEnabled()) {
      // for single row based selection only
      this.handleHeaderClickSelection(event);
    } else if (selectionMode === 'row' &&
                cellContext.axis.indexOf('column') !== -1
                && this._isSelectionEnabled()) {
      // for row selection and click on column header doesnt clear selection
      let activeOnly = false;
      if (!event.shiftKey) {
        activeOnly = true;
      }
      this.handleHeaderClickActive(event, activeOnly);
    } else {
      // if not selecting, just make active.
      this.handleHeaderClickActive(event);
    }
  }

  if (this.m_options.isFloodFillEnabled()) {
    if (this.m_bottomFloodFillIconContainer && this.m_bottomFloodFillIconContainer.parentNode) {
      this.m_bottomFloodFillIconContainer.parentNode
        .removeChild(this.m_bottomFloodFillIconContainer);
      this.m_bottomFloodFillIconContainer
        .removeEventListener('mouseover', this.handleDatabodyMouseMove);
    }
  }

  if (processed === true) {
    event.preventDefault();
  }
};

DvtDataGrid.prototype.handleHeaderLabelMouseDown = function (event) {
  var processed;
  if (this.isResizeEnabled()) {
    processed = this.handleResizeMouseDown(event);
    this._highlightResizeMouseDown();
  }
  if (processed === true) {
    event.preventDefault();
  }
};

/**
 * Handle mouseup events on the document
 * @param {Event} event - mouseup event on the document
 */
DvtDataGrid.prototype.handleMouseUp = function (event) {
  // toggle off the drag state
  this.m_headerDragState = false;
  this.m_databodyDragState = false;
  this.m_deselectInProgress = false;

  // if we mouseup outside the grid we want to cancel the selection and return the row
  if (this.m_databodyMove) {
    this._handleMoveMouseUp(event, false);
  } else if (this.isResizeEnabled()) {
    this.handleResizeMouseUp(event);
  }

  this.m_databodyMove = false;
};

DvtDataGrid.prototype.shouldHoverHeader = function (header) {
  const headerAxis = header == null ? null : this.getHeaderCellAxis(header);
  const isRow = (headerAxis === 'row' || headerAxis === 'rowEnd');
  const selectionMode = this.m_options.getSelectionMode();
  return this._isSelectionEnabled() && (
    (this.isMultipleSelection() && selectionMode === 'cell') ||
    (selectionMode === 'row' && isRow));
};

DvtDataGrid.prototype.handleHeaderMouseOver = function (event) {
  var target = /** @type {Element} */ (event.target);
  var header = this.findHeader(target);
  if (!this.m_isResizing && this.manageHeaderCursor(event, false) === 'default' && this.shouldHoverHeader(header)) {
    this.m_utils.addCSSClassName(header, this.getMappedStyle('hover'));
  }
};

DvtDataGrid.prototype.handleHeaderMouseOut = function (event) {
  var target = /** @type {Element} */ (event.target);
  this.m_utils.removeCSSClassName(this.findHeader(target), this.getMappedStyle('hover'));
  if (!this.m_isResizing && this.m_resizingElement) {
    this.m_resizingElement.style.cursor = '';
    if (this.m_resizingElementSibling != null) {
      this.m_resizingElementSibling.style.cursor = '';
    }
  }
  if (this._isDOMElementSortable(target)) {
    this._handleSortMouseOut(event);
  }
};

/**
 * Event handler for when header mouse up event
 * @protected
 * @param {Event} event - header mouse up event
 */
DvtDataGrid.prototype.handleHeaderMouseUp = function (event) {
  // handle anchor change while in header drag mode
  this.handleDragAnchorChange(event);

  // toggle off the drag state
  this.m_headerDragState = false;
  this.m_databodyDragState = false;
  this.m_deselectInProgress = false;

  if (this.m_floodFillDragState) {
    this.unhighlightFloodFillRange();
    this.m_selectionRange = null;
    this.m_floodFillRange = null;
    this.m_floodFillDirection = null;
    this.m_databody.style.cursor = 'default';
    this.m_cursor = 'default';
  }
  if (this.m_databodyMove) {
    this._handleMoveMouseUp(event, true);
  }
};

DvtDataGrid.prototype.handleCornerMouseDown = function (event) {
  let target = /** @type {Element} */ (event.target);
  let end = this.m_utils.containsCSSClassName(target, this.getMappedStyle('rowendheaderlabel')) ||
            this.m_utils.containsCSSClassName(target, this.getMappedStyle('columnendheaderlabel'));
  let label = this.findLabel(target);
  let clearSelection;
  if (label != null) {
    clearSelection = end; // endlabels to clear selection.
    this._setActive(label, this._createActiveObject(label), event, clearSelection);
  }
};

DvtDataGrid.prototype.handleCornerMouseOver = function (event) {
  var target = /** @type {Element} */ (event.target);
  if (this._isSelectionEnabled() && this.isMultipleSelection()) {
    this.m_utils.addCSSClassName(this.findLabel(target), this.getMappedStyle('hover'));
  }
};

DvtDataGrid.prototype.handleCornerMouseOut = function (event) {
  var target = /** @type {Element} */ (event.target);
  this.m_utils.removeCSSClassName(this.findLabel(target), this.getMappedStyle('hover'));
};

/**
 * Event handler for when the top left corner is clicked
 * @protected
 * @param {Event} event - click event on the top left corner
 */
DvtDataGrid.prototype.handleCornerClick = function (event) {
  this._handleSelectAll(event);
};

/**
 * Event handler for when row/column header is clicked
 * @protected
 * @param {Event} event - click event on the headers
 */
DvtDataGrid.prototype.handleHeaderClick = function (event) {
  var target = /** @type {Element} */ (event.target);
  if ((this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortascending')) ||
    this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortdescending')) ||
    this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortdefault'))) &&
    this._isDOMElementSortable(target)) {
    this._removeTouchSelectionAffordance();
    this._handleHeaderSort(event);
    event.preventDefault();
  } else if (this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortIcon')) &&
    this._isDOMElementSortable(target.lastChild)) {
    this._removeTouchSelectionAffordance();
    this._handleHeaderSort(event);
    event.preventDefault();
  } else if (this._isDisclosureIcon(event.target)) {
    this._removeTouchSelectionAffordance();
    this._handleExpandCollapseRequest(event);
    event.preventDefault();
  }
};

/**
  * Event handler for when row/column header is double clicked
  * @protected
  * @param {Event} event - click event on the headers
  */
DvtDataGrid.prototype.handleHeaderDoubleClick = function (event) {
  let isHeaderLabel = false;
  // setting this.m_cursor as it would have been reset to default in mouseup.
  this.m_cursor = this.manageHeaderCursor(event, isHeaderLabel);
  let resizeCursor = (this.m_cursor === 'col-resize' || this.m_cursor === 'row-resize');
  if (resizeCursor) {
    if (!this.m_resizingElement) {
      this.m_resizingElement = this.findHeader(event.target);
    }
    const resizingElementAxis = this.getHeaderCellAxis(this.m_resizingElement);
    const resizingElementLevel = this.getHeaderCellLevel(this.m_resizingElement);
    let allowResizing = false;
    // resize to fit currently supports only cell data and not headers.
    if ((resizingElementAxis === 'row' || resizingElementAxis === 'rowEnd') &&
        (resizingElementLevel === this.m_rowHeaderLevelCount - 1) &&
        (this.m_cursor === 'row-resize')) {
      allowResizing = true;
    } else if ((resizingElementAxis === 'column' || resizingElementAxis === 'columnEnd') &&
        (resizingElementLevel === this.m_columnHeaderLevelCount - 1) &&
        (this.m_cursor === 'col-resize')) {
      allowResizing = true;
    }

    if (allowResizing && this.isResizeEnabled()) {
      this._getHeadersForResizeFitToContent(event);
    }
  }
};

DvtDataGrid.prototype._getHeadersForResizeFitToContent = function (event) {
  const resizingElementAxis = this.getHeaderCellAxis(this.m_resizingElement);
  const resizingElementIndex = this.getHeaderCellIndex(this.m_resizingElement);

  let headers = [];
  if (this._isSelectionEnabled() && this.isMultipleSelection() && this.m_selection.length) {
    let selection = this.m_selection[0];
    headers = this._getHeadersWithinSelection(selection, resizingElementIndex,
                                              resizingElementAxis);
  }
  if (!headers.length) {
    headers.push(this.m_resizingElement);
  }

  headers.forEach(header => {
    this.handleResizeFitToContent(event, header, resizingElementAxis);
  });
};

DvtDataGrid.prototype._getHeadersWithinSelection = function (selection, resizingElementIndex,
                                                            resizingElementAxis) {
  let headers = [];
  let headerStart;
  let headerLevel;
  let headerLevelCount;
  let root;
  let genericAxis;
  if (resizingElementAxis === 'column') {
    headerStart = this.m_startColHeader;
    headerLevel = this.m_columnHeaderLevelCount - 1;
    headerLevelCount = this.m_columnHeaderLevelCount;
    root = this.m_colHeader;
    genericAxis = 'column';
  } else if (resizingElementAxis === 'columnEnd') {
    headerStart = this.m_startColEndHeader;
    headerLevel = this.m_columnEndHeaderLevelCount - 1;
    headerLevelCount = this.m_columnEndHeaderLevelCount;
    root = this.m_colEndHeader;
    genericAxis = 'column';
  } else if (resizingElementAxis === 'row') {
    headerStart = this.m_startRowHeader;
    headerLevel = this.m_rowHeaderLevelCount - 1;
    headerLevelCount = this.m_rowHeaderLevelCount;
    root = this.m_rowHeader;
    genericAxis = 'row';
  } else {
    headerStart = this.m_startRowEndHeader;
    headerLevel = this.m_rowEndHeaderLevelCount - 1;
    headerLevelCount = this.m_rowEndHeaderLevelCount;
    root = this.m_rowEndHeader;
    genericAxis = 'row';
  }
  let startIndex;
  let endIndex;
  // corner selection
  if ((selection.startIndex.column === 0 && selection.endIndex.column === -1) &&
      (selection.startIndex.row === 0 && selection.endIndex.row === -1)) {
    let count = genericAxis === 'row' ? this.m_endRowHeader - this.m_startRowHeader :
                                    this.m_endColHeader - this.m_startColHeader;
    startIndex = 0;
    endIndex = count;
  } else if ((selection.startIndex.column === 0 && selection.endIndex.column === -1) ||
              selection.startIndex.column === undefined) {
    // row headers if multiple columns selected
    if (((selection.startIndex.row <= resizingElementIndex) &&
          (resizingElementIndex <= selection.endIndex.row)) &&
          (resizingElementAxis === 'row' || resizingElementAxis === 'rowEnd')) {
      startIndex = selection.startIndex.row;
      endIndex = selection.endIndex.row;
    }
  } else if ((selection.startIndex.row === 0 && selection.endIndex.row === -1) ||
              selection.startIndex.row === undefined) {
    // column headers if multiple rows selected
    if (((selection.startIndex.column <= resizingElementIndex) &&
          (resizingElementIndex <= selection.endIndex.column)) &&
          (resizingElementAxis === 'column' || resizingElementAxis === 'columnEnd')) {
      startIndex = selection.startIndex.column;
      endIndex = selection.endIndex.column;
    }
  }
  for (let i = startIndex; i <= endIndex; i++) {
    let headerCell = this._getHeaderByIndex(i, headerLevel, root, headerLevelCount, headerStart);
    if (headerCell) {
      headers.push(headerCell);
    }
  }
  return headers;
};

/**
 * Event handler for when mouse down anywhere in the databody
 * @protected
 * @param {Event} event - mousedown event on the databody
 */
DvtDataGrid.prototype.handleDatabodyMouseDown = function (event) {
  var target = /** @type {Element} */ (event.target);
  var cell = this.findCell(target);
  if (cell == null) {
    this.m_scrollbarFocus = true;
    return;
  }

  if (this._isEditOrEnter()) {
    var activeCell = this._getActiveElement();
    if (cell !== activeCell) {
      if (this._leaveEditing(event, activeCell, false) === false) {
        return;
      }
    } else {
      return;
    }
  } else {
    // reset actionable mode whenever user clicks in the databody
    this._exitActionableMode();
  }

  var ctrlKey = this.m_utils.ctrlEquivalent(event);
  // only perform events on left mouse, (right in rtl culture)
  if (event.button === 0 && !ctrlKey) {
    if (this._isMoveOnElementEnabled(cell)) {
      this.m_databodyMove = true;
      this.m_currentX = event.pageX;
      this.m_currentY = event.pageY;
    }
  }

  if (!this.m_root.contains(document.activeElement) || document.activeElement === this.m_root) {
    this.m_externalFocus = true;
  }

  if (this._isGridEditable()) {
    this.m_shouldFocus = !this._isFocusableElementBeforeCell(target);
  }

  // if click or right click we want to adjust the selction
  // no else so that we can select a cell in the same row as long as no drag
  // check if selection is enabled
  if (this._isSelectionEnabled()) {
    // only allow drag on left click
    if (this.isMultipleSelection() && event.button === 0) {
      this.m_databodyDragState = true;
    }
    this.handleDatabodyClickSelection(event);
  } else {
    // if selection is disable, we'll still need to highlight the active cell
    this.handleDatabodyClickActive(event);
  }
};

DvtDataGrid.prototype.handleDatabodyMouseOut = function (event) {
  if (!this.m_databodyMove) {
    var target = /** @type {Element} */ (event.target);
    var targetCell = this.findCell(target);
    this._setCellHover(targetCell, 'remove');
  }
};

DvtDataGrid.prototype.handleDatabodyMouseOver = function (event) {
  if (!this.m_databodyMove) {
    var target = /** @type {Element} */ (event.target);
    var targetCell = this.findCell(target);
    this._setCellHover(targetCell, 'add');
  }
};

DvtDataGrid.prototype._setCellHover = function (targetCell, addOrRemove) {
  if (targetCell != null && this._isSelectionEnabled()) {
    var selectionMode = this.m_options.getSelectionMode();
    if (selectionMode === 'cell') {
      if (addOrRemove === 'add') {
        this.m_utils.addCSSClassName(targetCell, this.getMappedStyle('hover'));
      } else {
        this.m_utils.removeCSSClassName(targetCell, this.getMappedStyle('hover'));
      }
    } else if (selectionMode === 'row') {
      var index = this._getIndex(targetCell, 'row');
      var returnObj = this._getSelectionStartAndEnd(this.createIndex(index, this.m_startCol),
        this.createIndex(index, this.m_endCol), 0);
      for (var i = returnObj.min.row; i <= returnObj.max.row; i++) {
        this._highlightCellsAlongAxis(i, 'row', 'index', addOrRemove, ['hover']);
      }
    }
  }
};

DvtDataGrid.prototype.handleDatabodyDoubleClick = function (event) {
  if (this._isGridEditable()) {
    var target = event.target;
    var cell = this.findCell(target);
    var currentMode = this._getCurrentMode();
    if (currentMode === 'edit') {
      var activeCell = this._getActiveElement();
      if (cell === activeCell) {
        // if the active cell is being edited and it is the target do not eat the double click
        return;
      }
      if (!this._handleExitEdit(event, activeCell)) {
        return;
      }
    }
    this._handleEditable(event, cell);
    this._handleEdit(event, cell);
  }
};

/**
 * Event handler for when mouse move anywhere in the databody
 * @protected
 * @param {Event} event - mousemove event on the databody
 */
DvtDataGrid.prototype.handleDatabodyMouseMove = function (event) {
  // handle move first because it should happen first on the second click
  if (event.buttons === 0) {
    this.handleMouseUp(event);
  }
  if (this.m_databodyMove) {
    this._handleMove(event);
  } else if (this.m_databodyDragState) {
    if (!this.m_floodFillDragState) {
      this.handleDatabodySelectionDrag(event);
    } else if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
      if (!this.m_selectionRange) {
        this.m_selectionRange = this.GetSelection();
      }
      this.handleDatabodyFloodFillDrag(event);
    }
  } else if (this.m_headerDragState &&
    ((this.m_selectionFrontier && this.m_selectionFrontier.axis &&
      (this.m_selectionFrontier.axis.indexOf('row') !== -1 || this.m_selectionFrontier.axis.indexOf('column') !== -1)) ||
    (this.m_deselectInfo && this.m_deselectInfo.axis && (this.m_deselectInfo.axis.indexOf('row') !== -1 || this.m_deselectInfo.axis.indexOf('column') !== -1)))) {
    this.extendSelectionHeader(event.target, event, true, this.m_deselectInProgress);
  }
};

/**
 * Event handler for when mouse down anywhere in the databody
 * @protected
 * @param {Event} event - mouseup event on the databody
 */
DvtDataGrid.prototype.handleDatabodyMouseUp = function (event) {
  this.m_databodyDragState = false;
  this.m_headerDragState = false;
  this.m_deselectInProgress = false;
  if (this.m_databodyMove) {
    this._handleMoveMouseUp(event, true);
  }
  if (this.m_options.isFloodFillEnabled() && this.m_floodFillDragState) {
    this._handleFloodFillMouseUp(event);
    this.m_floodFillDragState = false;
  }
};

DvtDataGrid.prototype.handleDatabodyKeyUp = function (event) {
  if (this.m_deselectInProgress) {
    this.m_deselectInProgress = event.shiftKey;
  }
};

/**
 * Event handler for when user press down a key in the databody
 * @protected
 * @param {Event} event - keydown event on the databody
 */
DvtDataGrid.prototype.handleDatabodyKeyDown = function (event) {
  var action;
  // var keyCode = event.keyCode;
  var ctrlKey = this.m_utils.ctrlEquivalent(event);

  // no longer fire keydown, just check if row expander handled the event already
  // also ignore if the component is animating
  if ((event.defaultPrevented && ctrlKey &&
       (this.keyCodes.LEFT_KEY || this.keyCodes.RIGHT_KEY)) ||
      this.m_animating) {
    return;
  }

  // check if header is active
  if (this.m_active != null && this.m_active.type === 'header') {
    action = this._getActionFromKeyDown(event, this.m_active.axis, false);
  } else if (this.m_active != null && this.m_active.type === 'label') {
    action = this._getActionFromKeyDown(event, this.m_active.axis, true);
  } else {
    action = this._getActionFromKeyDown(event, 'cell', false);
  }

  var element = this._getActiveElement();

  if (action != null) {
    if (action.call(this, event, element)) {
      event.preventDefault();
    }
  }
};

/**
 * Find top and left offset of an element relative to the (0,0) point on the page
 * @param {Element} element - the element to find left and top offset of
 * @return {Array.<number>} - [leftOffset, topOffset]
 */
DvtDataGrid.prototype.findPos = function (element) {
  if (element) {
    var parentPos = this.findPos(element.offsetParent);
    var transform = this.getElementTranslationXYZ(element.offsetParent);
    return [
      parseInt(parentPos[0], 10) + parseInt(element.offsetLeft, 10) + transform[0],
      parseInt(parentPos[1], 10) + parseInt(element.offsetTop, 10) + transform[1]
    ];
  }
  return [0, 0];
};

/**
 * Find top and left offset relative to the enclosing header
 * @param {Element} element - the event target
 * @param {Element} header - the enclosing header element
 * @param {Element} headerOffset - initial return value
 * @return {Array.<number>} - [leftOffset, topOffset]
 */
DvtDataGrid.prototype._findHeaderOffset = function (element, header, headerOffset) {
  if (!headerOffset) {
    // eslint-disable-next-line no-param-reassign
    headerOffset = [0, 0];
  }
  if (element !== header) {
    // eslint-disable-next-line no-param-reassign
    headerOffset[0] += parseInt(element.offsetLeft, 10);
    // eslint-disable-next-line no-param-reassign
    headerOffset[1] += parseInt(element.offsetTop, 10);
    return this._findHeaderOffset(element.offsetParent, header, headerOffset);
  }
  return headerOffset;
};
/**
 * Get an elements transform3d X,Y,Z
 * @param {Element} element - the element to find transform3d X,Y,Z of
 * @return {Array.<number>} - [transformX, transformY, transformZ]
 */
DvtDataGrid.prototype.getElementTranslationXYZ = function (element) {
  if (element) {
    var cs = document.defaultView.getComputedStyle(element, null);
    var transform = cs.getPropertyValue('-webkit-transform') ||
      cs.getPropertyValue('-moz-transform') ||
      cs.getPropertyValue('-ms-transform') ||
      cs.getPropertyValue('-o-transform') ||
      cs.getPropertyValue('transform');
    var matrixArray = transform.substr(7, transform.length - 8).split(', ');
    var transformX = isNaN(parseInt(matrixArray[4], 10)) ? 0 : parseInt(matrixArray[4], 10);
    var transformY = isNaN(parseInt(matrixArray[5], 10)) ? 0 : parseInt(matrixArray[5], 10);
    var transformZ = isNaN(parseInt(matrixArray[6], 10)) ? 0 : parseInt(matrixArray[6], 10);
    return [transformX, transformY, transformZ];
  }
  return [0, 0, 0];
};


/**
 * Event handler for when mouse wheel is used on the databody
 * @param {Event} event - mousewheel event on the databody
 */
DvtDataGrid.prototype.handleDatabodyMouseWheel = function (event) {
  var axis;
  var header = this.find(event.target, 'header');
  if (header == null) {
    header = this.find(event.target, 'endheader');
  }
  if (header) {
    axis = (header === this.m_rowHeader || header === this.m_rowEndHeader) ? 'row' : 'column';
  }
  // prevent scrolling of the page, unless there are no more rows, consistant with table/listview
  if (axis === 'row' &&
    (!this.m_stopRowHeaderFetch || !this.m_stopRowEndHeaderFetch || !this.m_stopRowFetch)) {
    event.preventDefault();
  } else if (axis === 'column' && (!this.m_stopColumnHeaderFetch ||
    !this.m_stopColumnEndHeaderFetch || !this.m_stopColumnFetch)) {
    event.preventDefault();
  }

  // prevent scroll if animating sort
  if (this.m_animating) {
    return;
  }

  var delta = this.m_utils.getMousewheelScrollDelta(event);

  var deltaX = delta.deltaX;
  var deltaY = delta.deltaY;

  // prevent horizontal/vertical scroll on row/column headers respectively.
  if (axis === 'row') {
    deltaX = 0;
  } else if (axis === 'column') {
    deltaY = 0;
  }
  var scrollTop = Math.max(0, Math.min(this._getMaxScrollHeight(),
    this.m_currentScrollTop - deltaY));
  // The below check is to ensure double scrolling is avoided when scrolling over row headers.
  if (this._getMaxScrollHeight() !== scrollTop && scrollTop !== 0 &&
      this.find(event.target, 'header')) {
    event.preventDefault();
  }
  this.scrollDelta(deltaX, deltaY);
};
/** ************** touch related methods ********************/

/**
 * Event handler for when touch is started on the databody
 * @param {Event} event - touchstart event on the databody
 */
DvtDataGrid.prototype.handleTouchStart = function (event) {
  var fingerCount = event.touches.length;
  var target = /** @type {Element} */ (event.touches[0].target);

  // move = one finger swipe (or two?)
  if (fingerCount === 1) {
    // get the coordinates of the touch
    this.m_startX = event.touches[0].pageX;
    this.m_startY = event.touches[0].pageY;

    // need these to detect whether touch is hold and move vs. swipe
    this.m_currentX = this.m_startX;
    this.m_currentY = this.m_startY;
    this.m_prevX = this.m_startX;
    this.m_prevY = this.m_startY;
    this.m_startTime = (new Date()).getTime();

    // flag it
    this.m_touchActive = true;

    // if multiple select enabled check to see if the touch start was on a select affordance
    if (this.isMultipleSelection()) {
      // if the target is not the container, but rather the icon itself, choose the container instead
      if (target.classList.contains(this.getMappedStyle('selectaffordance'))) {
        target = target.parentNode;
      }

      // determine which icon was clicked on
      var dir = null;
      if (target === this.m_topSelectIconContainer) {
        dir = 'top';
      } else if (target === this.m_bottomSelectIconContainer) {
        dir = 'bottom';
      }

      if (dir) {
        // keeps track of multiple select mode
        this.m_touchMultipleSelect = true;
        var selection = this.GetSelection();
        if (dir === 'top') {
          // anchor is bottom right of selection for selecting top affordance
          this.m_touchSelectAnchor = selection[selection.length - 1].endIndex;
        } else {
          // anchor is top left of selection for selecting bottom affordance
          this.m_touchSelectAnchor = selection[selection.length - 1].startIndex;
        }
      }
    }

    // if not multiple select, check for row reorder
    if (!this.m_touchMultipleSelect && this._isMoveOnElementEnabled(this.findCell(target))) {
      this.m_databodyMove = true;
    }
  } else {
    // more than one finger touched so cancel
    this.handleTouchCancel(event);
  }
};

/**
 * Event handler for when touch moves on the databody
 * @param {Event} event - touchmove event on the databody
 */
DvtDataGrid.prototype.handleTouchMove = function (event) {
  var target = /** @type {Element} */ (event.target);

  if (this.m_touchActive) {
    if (event.cancelable) {
      event.preventDefault();
    }
    this.m_currentX = event.touches[0].pageX;
    this.m_currentY = event.touches[0].pageY;

    var diffX = this.m_currentX - this.m_prevX;
    var diffY = this.m_currentY - this.m_prevY;
    if (this.getResources().isRTLMode()) {
      diffX *= -1;
    }

    if (this.m_touchMultipleSelect) {
      this.handleDatabodySelectionDrag(event);
    } else if (this.m_databodyMove) {
      this._removeTouchSelectionAffordance();
      this._handleMove(event);
    } else if (this._isEditOrEnter()) {
      var cell = this._getActiveElement();
      if (this.findCell(target) !== cell) {
        this._handleNonSwipeScroll(diffX, diffY);
      }
    } else {
      this._handleNonSwipeScroll(diffX, diffY);
    }

    this.m_prevX = this.m_currentX;
    this.m_prevY = this.m_currentY;
  } else {
    this.handleTouchCancel(event);
  }
};

/**
 * Event handler for when touch ends on the databody
 * @param {Event} event - touchend event on the databody
 */
DvtDataGrid.prototype.handleTouchEnd = function (event) {
  var target = /** @type {Element} */ (event.target);
  var cell;

  if (this._isEditOrEnter()) {
    cell = this._getActiveElement();
    if (this.findCell(target) !== cell) {
      this._leaveEditing(event, cell, false);
    } else {
      this.handleTouchCancel(event);
      return;
    }
  } else {
    // reset actionable mode whenever user clicks in the databody
    this._exitActionableMode();
  }

  if (this.m_lastTapTime != null &&
      (this.m_startTime - this.m_lastTapTime) < 250 &&
      this.m_lastTapTarget === target) {
    this.m_lastTapTime = null;
    this.m_lastTapTarget = null;
    cell = this.findCell(target);
    if (cell != null) {
      this._handleEditable(event, cell);
      this._handleEdit(event, cell);
      if (event.cancelable) {
        event.preventDefault();
      }
    }
  } else {
    this.m_lastTapTarget = event.target;
    this.m_lastTapTime = (new Date()).getTime();
  }

  if (this.m_touchActive && !event.defaultPrevented) {
    if (this.m_touchMultipleSelect) {
      if (event.cancelable) {
        event.preventDefault();
      }
      this.m_touchMultipleSelect = false;
    } else {
      var duration = this.m_lastTapTime - this.m_startTime;
      if (this.m_currentX === this.m_startX && this.m_currentY === this.m_startY) {
        // this means we performed a tap within the row with the active cell
        // and it wasn't actually a move, also only change selection on a tap
        // outside of the current selection, if it was longer than context menu the
        // handleContextMenuGesture will have changed this
        this.m_databodyMove = false;
        if (this._isSelectionEnabled() && duration < DvtDataGrid.CONTEXT_MENU_TAP_HOLD_DURATION) {
          this.handleDatabodyClickSelection(event);
          return;
        }

        // activate on a tap
        this.handleDatabodyClickActive(event);
        return;
      }

      if (this.m_databodyMove) {
        if (event.cancelable) {
          event.preventDefault();
        }
        this.m_databodyMove = false;
        this._handleMoveMouseUp(event, true);
        return;
      }

      this._handleSwipe(event);
    }
  }

  this.handleTouchCancel(event);
};

/**
 * Calculate the momentum based on the distance and duration of the swipe
 * @param {number} current the current touch position
 * @param {number} start the start touch position
 * @param {number} time the duration of the swipe
 * @param {number} currentScroll the current scroll position
 * @param {number} maxScroll the maximum scroll position
 * @param {boolean=} rtl true if right to left, false if left to right, undefined if determining momentum in Y direction
 * @return {Object} an object with three keys:
 *                      destination - the point to scroll to with the momentum
 *                      overScroll - the pixel amount that is scrolled beyond the scrollable region
 *                      duration - the duration of the scroll to that destination
 * @private
 */
DvtDataGrid.prototype._calculateMomentum =
  function (current, start, time, currentScroll, maxScroll, rtl) {
    var distance = current - start;
    var speed = Math.abs(distance) / time;
    var destination = (((speed * speed) / (2 * DvtDataGrid.DECELERATION_FACTOR)) *
                       (distance < 0 ? -1 : 1));
    var duration = speed / DvtDataGrid.DECELERATION_FACTOR;
    var overScroll;

    if (rtl) {
      destination *= -1;
    }

    // if the distance overshoots, then we'll have to adjust and recalculate the duration
    if (currentScroll - destination > maxScroll) {
      // too far bottom/right
      overScroll = Math.max(DvtDataGrid.MAX_OVERSCROLL_PIXEL * -1, destination);
      destination = currentScroll - maxScroll;
      distance = maxScroll - currentScroll;
      duration = distance / speed;
    } else if (currentScroll - destination < 0) {
      // too far top/left
      overScroll = Math.min(DvtDataGrid.MAX_OVERSCROLL_PIXEL, destination);
      destination = currentScroll;
      distance = currentScroll;
      duration = distance / speed;
    }

    return {
      destination: Math.round(destination),
      // durations can be up to 4s currently let's cap them at 500ms
      duration: Math.min(Math.max(DvtDataGrid.MIN_SWIPE_TRANSITION_DURATION,
        duration),
      DvtDataGrid.MAX_SWIPE_TRANSITION_DURATION),
      overScroll: overScroll
    };
  };

/**
 * Event handler for when touch is cancelled on the databody
 * @param {Event} event - touchcancel event on the databody
 */
DvtDataGrid.prototype.handleTouchCancel = function (event) {
  if (this.m_databodyMove) {
    this._handleMoveMouseUp(event, false);
    this.m_databodyMove = false;
  }
  this.m_touchSelectAnchor = null;
  this.m_touchMultipleSelect = false;
  // reset the variables back to default values
  this.m_touchActive = false;
  this.m_startX = 0;
  this.m_startY = 0;
  this.m_prevX = 0;
  this.m_prevY = 0;
  this.m_currentX = 0;
  this.m_currentY = 0;
  this.m_startTime = 0;
};

/**
 * Event handler for when touch is started on the header
 * @param {Event} event - touchstart event on the header
 */
DvtDataGrid.prototype.handleHeaderTouchStart = function (event) {
  // store start time of touch
  this.m_touchStart = (new Date()).getTime();

  var fingerCount = event.touches.length;
  var target = /** @type {Element} */ (event.target);

  // move = one finger swipe (or two?)
  if (fingerCount === 1) {
    // get the coordinates of the touch
    this.m_startX = event.touches[0].pageX;
    this.m_startY = event.touches[0].pageY;

    // need these to detect whether touch is hold and move vs. swipe
    this.m_currentX = this.m_startX;
    this.m_currentY = this.m_startY;
    this.m_prevX = this.m_startX;
    this.m_prevY = this.m_startY;

    // flag it
    this.m_touchActive = true;
    var header = this.findHeader(target);

    if (this.isResizeEnabled()) {
      this.handleResize(event);
      this.handleResizeMouseDown(event);
      this._highlightResizeMouseDown();
    }

    // allow row reorder on headers if our move is enabled make sure our row has the active cell in it
    if (!this.m_isResizing && this._isMoveOnElementEnabled(header)) {
      this.m_databodyMove = true;
    }
  } else {
    // more than one finger touched so cancel
    this.handleHeaderTouchCancel(event);
  }
};

/**
 * Event handler for when touch moves on the header
 * @param {Event} event - touchmove event on the header
 */
DvtDataGrid.prototype.handleHeaderTouchMove = function (event) {
  if (this.m_touchActive) {
    if (event.cancelable) {
      event.preventDefault();
    }

    this.m_currentX = event.touches[0].pageX;
    this.m_currentY = event.touches[0].pageY;

    var diffX = this.m_currentX - this.m_prevX;
    var diffY = this.m_currentY - this.m_prevY;

    if (this.m_isResizing && this.isResizeEnabled()) {
      this.handleResize(event);
    } else if (this.m_databodyMove) {
      this._removeTouchSelectionAffordance();
      this._handleMove(event);
    } else {
      var target = /** @type {Element} */ (event.target);
      // can't swipe column headers in Y and row headers in X
      var header = this.findHeader(target);
      var axis = this.getHeaderCellAxis(header);
      if (axis === 'column' || axis === 'columnEnd') {
        this._handleNonSwipeScroll(diffX, 0);
      } else {
        this._handleNonSwipeScroll(0, diffY);
      }
    }

    this.m_prevX = this.m_currentX;
    this.m_prevY = this.m_currentY;
  } else {
    this.handleTouchCancel(event);
  }
};

/**
 * Event handler for when touch ends on the header
 * @param {Event} event - touchend event on the header
 */
DvtDataGrid.prototype.handleHeaderTouchEnd = function (event) {
  var header;

  if (this.m_touchActive && !event.defaultPrevented) {
    var target = /** @type {Element} */ (event.target);
    // if resizing handle resize first so that we don't conflict and forget to end
    if (this.m_isResizing && this.isResizeEnabled()) {
      this.handleResizeMouseUp(event);
      if (this.m_currentX !== this.m_startX && this.m_currentY !== this.m_startY) {
        if (event.cancelable) {
          event.preventDefault();
        }
      }
    } else if (this.m_currentX === this.m_startX && this.m_currentY === this.m_startY) {
      // if a short tap select
      var selectionMode = this.m_options.getSelectionMode();
      header = this.findHeader(target);
      var cellContext = header[this.getResources().getMappedAttribute('context')];
      var rootId = this.m_root.getAttribute('id');
      var contextMenu = document.querySelector('#' + rootId + 'contextmenu');
      if (contextMenu && contextMenu.style.display === 'none') {
        // if touch in an icon it the click event will handle mousedown/up
        if ((this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortascending')) ||
            this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortdescending')) ||
            this.m_utils.containsCSSClassName(target, this.getMappedStyle('sortdefault')))
            && this._isDOMElementSortable(target)) {
          if (event.cancelable) {
            event.preventDefault();
          }
          this._removeTouchSelectionAffordance();
          this._handleSortIconMouseDown(target);
          this._handleHeaderSort(event);
        } else if (this._isDisclosureIcon(target)) {
          this._removeTouchSelectionAffordance();
          this._handleExpandCollapseRequest(event);
          event.preventDefault();
         } else if (this._isSelectionEnabled() &&
                  this.isMultipleSelection() &&
                  !(selectionMode === 'row' &&
                    cellContext.axis.indexOf('row') === -1) &&
                  !this.m_databodyMove) {
          // if click or right click we want to adjust the selction
          // no else so that we can select a cell in the same row as long as no drag
          // check if selection is enabled
          // only allow drag on left click
          if (event.button === 0) {
            this.m_headerDragState = true;
          }
          this.handleHeaderClickSelection(event);
        } else if (selectionMode === 'row' &&
                  cellContext.axis.indexOf('row') !== -1 &&
                  this._isSelectionEnabled()) {
          // for single row based selection only
          this.handleHeaderClickSelection(event);
        } else {
          // if not selecting, just make active.
          this.handleHeaderClickActive(event);
        }
      }
    } else if (this.m_databodyMove) {
      // if reordering a row
      if (event.cancelable) {
        event.preventDefault();
      }
      this.m_databodyMove = false;
      this._handleMoveMouseUp(event, true);
    } else {
      // handle potential swipe
      header = this.findHeader(target);
      this._handleSwipe(event, this.getHeaderCellAxis(header));
    }
    // tap and long hold shows context menu, through the wrapper layer
  }
  this.handleHeaderTouchCancel(event);
};

/**
 * Event handler for when touch is cancelled on the header
 * @param {Event} event - touchcancel event on the header
 */
DvtDataGrid.prototype.handleHeaderTouchCancel = function (event) {
  if (this.m_databodyMove) {
    this._handleMoveMouseUp(event, false);
    this.m_databodyMove = false;
  }
  // reset the variables back to default values
  this.m_touchActive = false;
  this.m_startX = 0;
  this.m_startY = 0;
  this.m_prevX = 0;
  this.m_prevY = 0;
  this.m_currentX = 0;
  this.m_currentY = 0;
};

/**
 * Handle a touch scroll that is a slow drag
 * @param {number} diffX
 * @param {number} diffY
 */
DvtDataGrid.prototype._handleNonSwipeScroll = function (diffX, diffY) {
  var time = (new Date()).getTime();
  // for non-swipe scroll use 0ms to prevent jiggling
  this._disableTouchScrollAnimation();

  var diff = this.adjustTouchScroll(diffX, diffY);
  // eslint-disable-next-line no-param-reassign
  diffX = diff[0];
  // eslint-disable-next-line no-param-reassign
  diffY = diff[1];

  this.scrollDelta(diffX, diffY);

  // reset start position if this is a tap and scroll, so that we can handle
  // user doing a swipe at the end
  if (time - this.m_startTime > DvtDataGrid.TAP_AND_SCROLL_RESET) {
    this.m_startX = this.m_currentX;
    this.m_startY = this.m_currentY;
    this.m_startTime = (new Date()).getTime();
  }
};

/**
 * Event handler for when touch swipe may have been detected
 * @param {Event} event - touchcancel event on the header
 * @param {string|null=} axis - if a header the header axis so we don't swipe in the direction
 */
DvtDataGrid.prototype._handleSwipe = function (event, axis) {
  var duration = (new Date()).getTime() - this.m_startTime;
  var rtl = this.getResources().isRTLMode();
  var diffX = this.m_currentX - this.m_startX;
  var diffY = this.m_currentY - this.m_startY;

  // if right to left the difference is the opposite on swipe
  if (rtl) {
    diffX *= -1;
  }
  if (Math.abs(diffX) < DvtDataGrid.MIN_SWIPE_DISTANCE &&
      Math.abs(diffY) < DvtDataGrid.MIN_SWIPE_DISTANCE &&
      duration < DvtDataGrid.MIN_SWIPE_DURATION) {
    // detect whether this is a swipe
    if (event.cancelable) {
      event.preventDefault();
    }
    // center touch affordances if row selection multiple
    if (this._isSelectionEnabled()) {
      this._scrollTouchSelectionAffordance();
    }
  } else if (duration < DvtDataGrid.MAX_SWIPE_DURATION) {
    // swipe case
    if (event.cancelable) {
      event.preventDefault();
    }

    var momentumX;
    if (axis !== 'row' && axis !== 'rowEnd') {
      // calculate momentum
      momentumX = this._calculateMomentum(this.m_currentX, this.m_startX, duration,
        this.m_currentScrollLeft, this.m_scrollWidth, rtl);
      if (!isNaN(momentumX.overScroll)) {
        // don't overscroll if there's more rows to fetch
        if (momentumX.overScroll > 0 || this.m_stopColumnFetch) {
          this.m_extraScrollOverX = momentumX.overScroll * -1;
        }
      }
    } else {
      momentumX = { duration: 0, destination: 0 };
      diffX = 0;
    }

    var momentumY;
    if (axis !== 'column' && axis !== 'columnEnd') {
      momentumY = this._calculateMomentum(this.m_currentY, this.m_startY, duration,
        this.m_currentScrollTop, this.m_scrollHeight);
      if (!isNaN(momentumY.overScroll)) {
        // don't overscroll if there's more rows to fetch
        if (momentumY.overScroll > 0 || this.m_stopRowFetch) {
          this.m_extraScrollOverY = momentumY.overScroll * -1;
        }
      }
    } else {
      momentumY = { duration: 0, destination: 0 };
      diffY = 0;
    }

    var transitionDuration = Math.max(momentumX.duration, momentumY.duration);
    this.m_databody.firstChild.style.transitionDuration = transitionDuration + 'ms';
    this.m_rowHeader.firstChild.style.transitionDuration = transitionDuration + 'ms';
    this.m_colHeader.firstChild.style.transitionDuration = transitionDuration + 'ms';
    this.m_rowEndHeader.firstChild.style.transitionDuration = transitionDuration + 'ms';
    this.m_colEndHeader.firstChild.style.transitionDuration = transitionDuration + 'ms';

    diffX += momentumX.destination;
    diffY += momentumY.destination;
    var diff = this.adjustTouchScroll(diffX, diffY);
    diffX = diff[0];
    diffY = diff[1];

    this.scrollDelta(diffX, diffY);
  }
};

/** *********** end touch related methods ********************/

/**
 * Callback on a widget listener
 * @param {string} functionName - the function name to look up in the callbacks
 * @param {Object} details - the object to pass into the callback function
 * @return {boolean|undefined} true if event passes, false if vetoed
 */
DvtDataGrid.prototype.fireEvent = function (functionName, details) {
  if (functionName == null || details == null) {
    return undefined;
  }

  var callback = this.callbacks[functionName];
  if (callback != null) {
    return callback(details);
  }
  return true;
};

/**
 * Add a callback function to the callbacks object
 * @param {string} functionName - the function name to callback on
 * @param {Object.<Function>} handler - the function to callback to
 */
DvtDataGrid.prototype.addListener = function (functionName, handler) {
  this.callbacks[functionName] = handler;
};
/** ********************************* end dom event handling ***************************************/

/**
 * Set the style height on an element in pixels
 * @param {Element} elem - the element to set height on
 * @param {number} height - the pixel height to set the element to
 */
DvtDataGrid.prototype.setElementHeight = function (elem, height) {
  // eslint-disable-next-line no-param-reassign
  elem.style.height = height + 'px';
};

/**
 * Get a number of the style height of an element
 * @param {Element|undefined|null} elem - the element to get height on
 * @return {number} the style height of the element
 */
DvtDataGrid.prototype.getElementHeight = function (elem) {
  return this.getElementDir(elem, 'height');
};

/**
 * Set the style width on an element in pixels
 * @param {Element} elem - the element to set width on
 * @param {number} width - the pixel width to set the element to
 */
DvtDataGrid.prototype.setElementWidth = function (elem, width) {
  // eslint-disable-next-line no-param-reassign
  elem.style.width = width + 'px';
};

/**
 * Get a number of the style pixel width of an element
 * @param {Element|undefined|null} elem - the element to get width on
 * @return {number} the style width of the element
 */
DvtDataGrid.prototype.getElementWidth = function (elem) {
  return this.getElementDir(elem, 'width');
};

/**
 * Set the style left/right/top/bottom on an element in pixels
 * @param {Element|undefined|null} elem - the element to set width on
 * @param {number} pix - the pixel width to set the element to
 * @param {string} dir - 'left','right','top,'bottom'
 * */
DvtDataGrid.prototype.setElementDir = function (elem, pix, dir) {
  // eslint-disable-next-line no-param-reassign
  elem.style[dir] = pix + 'px';
};

/**
 * Get a number of the style left/right/top/bottom of an element
 * @param {Element|undefined|null} elem - the element to get style left/right/top/bottom on
 * @param {string} dir - 'left','right','top,'bottom'
 * @return {number} the style left/right/top/bottom of the element
 */
DvtDataGrid.prototype.getElementDir = function (elem, dir) {
  var value;
  if (elem.style[dir].indexOf('px') > -1 && elem.style[dir].indexOf('e') === -1) {
    // parseFloat does better with big numbers
    return parseFloat(elem.style[dir]);
  }

  if (!document.body.contains(elem)) {
    // eslint-disable-next-line no-param-reassign
    elem.style.visibility = 'hidden';
    this.m_root.appendChild(elem); // @HTMLUpdateOK
    // Started using offset again because of how it handles large numbers and limits on BoundingClient
    // Note that on chrome and IE offsetHeight will round differently if the top value is at a decimal
    // pixel value greater or less than .5
    value = Math.round(elem['offset' + dir.charAt(0).toUpperCase() + dir.slice(1)]);
    this.m_root.removeChild(elem);
    // eslint-disable-next-line no-param-reassign
    elem.style.visibility = '';
  } else {
    value = Math.round(elem['offset' + dir.charAt(0).toUpperCase() + dir.slice(1)]);
  }
  return value;
};

DvtDataGrid.prototype._computeElementWidthAndHeight = function (elem) {
  var value = {};
  if (elem.style.width.indexOf('px') > -1 && elem.style.width.indexOf('e') === -1) {
    // parseFloat does better with big numbers
    value.width = parseFloat(elem.style.width);
  }
  if (elem.style.height.indexOf('px') > -1 && elem.style.height.indexOf('e') === -1) {
    // parseFloat does better with big numbers
    value.height = parseFloat(elem.style.height);
  }
  if (value.width == null || value.height == null) {
    if (!document.body.contains(elem)) {
      // eslint-disable-next-line no-param-reassign
      elem.style.visibility = 'hidden';
      this.m_root.appendChild(elem); // @HTMLUpdateOK
      value.width = Math.round(elem.offsetWidth);
      value.height = Math.round(elem.offsetHeight);
      this.m_root.removeChild(elem);
      // eslint-disable-next-line no-param-reassign
      elem.style.visibility = '';
    } else {
      value.width = Math.round(elem.offsetWidth);
      value.height = Math.round(elem.offsetHeight);
    }
  }
  return value;
};

/** *********************** Model change event *****************************************/
/**
 * @private
 */
DvtDataGrid.BEFORE = 1;

/**
 * @private
 */
DvtDataGrid.AFTER = 2;

/**
 * @private
 */
DvtDataGrid.INSIDE = 3;

/**
 * Checks whether an index (row/column) is within the range of the current viewport.
 * @param {Object} indexes the row and column indexes
 * @return {number} BEFORE if the index is before the current viewport, AFTER if the index is after
 *         the current viewport, INSIDE if the index is within the current viewport
 * @private
 */
DvtDataGrid.prototype._isInViewport = function (indexes) {
  var rowIndex = indexes.row;
  var columnIndex = indexes.column;

  if (rowIndex === -1 && columnIndex === -1) {
    // actually, this is an invalid index... should throw an error?
    return -1;
  }

  // if row index wasn't specified, just verify the column range
  if (rowIndex === -1) {
    return this._isColumnIndexInViewport(columnIndex);
  }

  // if column index wasn't specified, just verify the row range
  if (columnIndex === -1) {
    return this._isRowIndexInViewport(rowIndex);
  }

  // both row and column index are defined, then check both ranges
  if (columnIndex >= this.m_startCol && columnIndex <= this.m_endCol &&
      rowIndex >= this.m_startRow && rowIndex <= this.m_endRow) {
    return DvtDataGrid.INSIDE;
  }

  // undefined
  return -1;
};

/**
 * @private
 */
 DvtDataGrid.prototype._isAxisIndexInViewport = function (index, axis) {
  if (index === -1) {
    return -1;
  }
  if (axis === 'column') {
    return this._isColumnIndexInViewport(index);
  } else if (axis === 'row') {
    return this._isRowIndexInViewport(index);
  }
  return -1;
};

DvtDataGrid.prototype._isColumnIndexInViewport = function (index) {
  if (index < this.m_startCol) {
    return DvtDataGrid.BEFORE;
  }
  if (index > this.m_endCol) {
    return DvtDataGrid.AFTER;
  }
  // if it's not before or after, it must be inside
  return DvtDataGrid.INSIDE;
};

/**
 * @private
 */
DvtDataGrid.prototype._isRowIndexInViewport = function (index) {
  if (index < this.m_startRow) {
    return DvtDataGrid.BEFORE;
  }
  if (index > this.m_endRow) {
    return DvtDataGrid.AFTER;
  }
  // if it's not before or after, it must be inside
  return DvtDataGrid.INSIDE;
};

/**
 * Checks whether any part of the cell is in the viewport
 * @param {number} left
 * @param {number} right
 * @param {number} top
 * @param {number} bottom
 * @return {boolean} true if cell is in the viewport
 * @private
 */
DvtDataGrid.prototype._isCellBoundaryInViewport = function (left, right, top, bottom) {
  var viewportTop = this._getViewportTop();
  var viewportBottom = this._getViewportBottom();
  var viewportLeft = this._getViewportLeft();
  var viewportRight = this._getViewportRight();

  return (((bottom <= viewportBottom && bottom > viewportTop) ||
           (top >= viewportTop && top < viewportBottom)) &&
          ((right <= viewportRight && right > viewportLeft) ||
           (left >= viewportLeft && left < viewportRight)));
};

/**
 * @param {Object} event the model event
 * @return {boolean} true if event is queued, false otherwise
 * @private
 */
DvtDataGrid.prototype.queueModelEvent = function (event) {
  // in case if the model event arrives before the grid is fully rendered or the event arrives during processing
  // of model queue or we are in the middle of processing/animation model event, queue the event and handle it later
  if (!this.m_initialized || this.m_processingEventQueue ||
      this.m_animating || this.m_processingModelEvent) {
    if (this.m_modelEvents == null) {
      this.m_modelEvents = [];
    }
    this.m_modelEvents.push(event);
    return true;
  }

  return false;
};

/**
 * Model event handler
 * @param {Object} event the model change event
 * @param {boolean} fromQueue whether this is invoked from model queue processing, optional
 * @protected
 */
DvtDataGrid.prototype.handleModelEvent = function (event, fromQueue) {
  // in case if the model event arrives before the grid is fully rendered,
  // queue the event and handle it later
  if (fromQueue === undefined && this.queueModelEvent(event)) {
    return;
  }

  var operation = event.operation;
  var keys = event.keys;
  var indexes = event.indexes;
  var cellSet = event.result;
  var headerSet = event.header;
  var endHeaderSet = event.endheader;
  var silent = event.silent;
  var requiresAnimation = false;

  this.m_processingModelEvent = true;

  if (event.detail) {
    // handle new event
    if (operation === 'delete') {
      this._handleDeleteRangeEvent(event.detail);
    }
    if (operation === 'insert') {
      this._handleInsertRangeEvent(event.detail);
    }
    if (operation === 'update') {
      this._handleUpdateRangeEvent(event.detail);
    }
    if (operation === 'refresh') {
      this._handleModelRefreshEvent(event.detail);
    }
  } else if (operation === 'insert') {
      this._adjustActive(operation, indexes);
      this.m_shouldFocus = true;
      this._adjustSelectionOnModelChange(operation, keys, indexes);

      if (cellSet != null) {
        // range insert event with cellset returned
        this._handleModelInsertRangeEvent(cellSet, headerSet, endHeaderSet);
        requiresAnimation = true;
      } else {
        this._handleModelInsertEvent(indexes, keys);
      }
    } else if (operation === 'update') {
      this._handleModelUpdateEvent(indexes, keys, cellSet);
      requiresAnimation = true;
    } else if (operation === 'delete') {
      // adjust selection if neccessary
      // do this before the rows in the databody is mutate
      // (easier this way because of animation delays, plus the selection is immediately updated
      // to reflect the updated state)
      this._adjustActive(operation, indexes);
      this._adjustSelectionOnModelChange(operation, keys, indexes);

      if (this.m_utils.supportsTransitions()) {
        if (!Array.isArray(keys)) {
          // eslint-disable-next-line no-param-reassign
          keys = new Array(keys);
        }
        this._handleModelDeleteEventWithAnimation(keys);
        if (keys.length > 0) {
          requiresAnimation = true;
        }
      } else {
        this._handleModelDeleteEvent(indexes, keys, silent);
      }
    } else if (operation === 'refresh' || operation === 'reset') {
      this._handleModelRefreshEvent();
    } else if (operation === 'sync') {
      this._handleModelSyncEvent(event);
    }

  this.m_processingModelEvent = false;

  // Need to rerun the queued events if
  // coming from the queue. If no animation
  // was involved in the current event,
  // we can directly call _runModelEventQueue
  // from here. Animation events will call
  // _runModelEventQueue at the end of their
  // transition function.
  if (!requiresAnimation && fromQueue) {
    this._runModelEventQueue();
  }
};

/**
 * Adjust selection ranges if neccessary on insert or delete.
 * @param {string} operation the model event operation which triggers selection adjustment.
 * @param {Object} indexes the indexes that identify the rows that got inserted/deleted.
 * @private
 */
DvtDataGrid.prototype._adjustActive = function (operation, indexes) {
  var activeRowIndex;
  var activeHeader;

  if (this.m_active != null) {
    if (this.m_active.type === 'cell') {
      activeHeader = false;
      activeRowIndex = this.m_active.indexes.row;
    } else if (this.m_active.type === 'header' && this.m_active.axis === 'row') {
      activeHeader = true;
      activeRowIndex = this.m_active.index;
    } else {
      return;
    }
  } else {
    return;
  }

  if (!Array.isArray(indexes)) {
    // eslint-disable-next-line no-param-reassign
    indexes = new Array(indexes);
  }

  // if we are getting this from a move event
  if (this.m_moveActive === true) {
    if (operation === 'insert') {
      if (!activeHeader) {
        this.m_active.indexes.row = indexes[0].row;
      } else {
        this.m_active.index = indexes[0].row;
      }
      return;
    } else if (operation === 'delete' && indexes[0].row === activeRowIndex) {
      // do not clear the active since we know the active should be the
      // same once the moved row is returned via insert
      return;
    }
  }

  var adjustment = operation === 'insert' ? 1 : -1;

  for (var i = 0; i < indexes.length; i++) {
    var rowIndex = indexes[i].row;
    if (rowIndex < activeRowIndex) {
      if (!activeHeader) {
        this.m_active.indexes.row += adjustment;
      } else {
        this.m_active.index += adjustment;
      }
    } else if (rowIndex === activeRowIndex && operation === 'delete') {
      this._setActive(null, null);
    }
  }
};

/**
 * Adjust selection ranges if neccessary on insert or delete.
 * @param {string} operation the model event operation which triggers selection adjustment.
 * @param {Object} keys the keys that identify the rows that got inserted/deleted.
 * @param {Object} indexes the indexes that identify the rows that got inserted/deleted.
 * @private
 */
DvtDataGrid.prototype._adjustSelectionOnModelChange = function (operation, keys, indexes) {
  // make it an array if it's a single entry event
  if (!Array.isArray(keys)) {
    // eslint-disable-next-line no-param-reassign
    keys = new Array(keys);
  }

  if (!Array.isArray(indexes)) {
    // eslint-disable-next-line no-param-reassign
    indexes = new Array(indexes);
  }

  var selection = this.GetSelection();

  if (keys == null || indexes == null ||
      keys.length !== indexes.length || selection.length === 0) {
    // on a move reset the selection
    if (this.m_moveActive && operation === 'insert') {
      if (this._isSelectionEnabled() && this._isDatabodyCellActive()) {
        var movedRow;
        if (this.m_options.getSelectionMode() === 'cell') {
          movedRow = this.createRange(this.m_active.indexes, this.m_active.indexes,
            keys[0], keys[0]);
        } else {
          movedRow = this.createRange(indexes[0], indexes[0], keys[0], keys[0]);
        }
        this.m_selectionFrontier = this.m_active.indexes;
        selection.push(movedRow);
      }
      this.m_moveActive = false;
    }
    // we are done
    return;
  }

  var adjustment = operation === 'insert' ? 1 : -1;

  for (var i = 0; i < keys.length; i++) {
    var rowKey = keys[i].row;
    var rowIndex = indexes[i].row;
    var newRowKey;

    // have to do this backwards since we'll be mutating the array at the same time
    for (var j = selection.length - 1; j >= 0; j--) {
      var range = selection[j];
      var startRowKey = range.startKey.row;
      var endRowKey = range.endKey.row;
      var startRowIndex = range.startIndex.row;
      var endRowIndex = range.endIndex.row;

      if (startRowKey === rowKey) {
        if (endRowKey === rowKey) {
          // single row in range, and it has been deleted, so remove from selection
          if (operation === 'delete') {
            selection.splice(j, 1);
            // eslint-disable-next-line no-continue
            continue;
          }
        }

        // adjust start key, index stays the same
        // adjust end index, end key stays the same
        // get the key of the next row, which will become the new start key
        newRowKey = this._getKey(this._getAxisCellsByIndex(range.startIndex.row + 1, 'row')[0], 'row');
        range.startKey.row = newRowKey;
        range.endIndex.row += adjustment;
      } else if (endRowKey === rowKey) {
        // adjust end key and end index
        // get the key of the next row, which will become the new start key
        newRowKey = this._getKey(this._getAxisCellsByIndex(range.startIndex.row - 1, 'row')[0], 'row');
        range.endKey.row = newRowKey;
        range.endIndex.row += adjustment;
      } else if (rowIndex <= startRowIndex) {
        // before start index, so adjust both start and end index
        range.startIndex.row += adjustment;
        range.endIndex.row += adjustment;
      } else if (rowIndex < endRowIndex) {
        // something in between start and end selection, adjust the end index
        range.endIndex.row += adjustment;
      }
    }
  }
};

DvtDataGrid.prototype._simpleAdjustSelectionOnChange = function (operation, indexes, axis) {
  let selection = this.GetSelection();
  let adjustment = operation === 'insert' ? 1 : -1;

  for (let i = 0; i < indexes.length; i++) {
    let index = indexes[i];

    for (let j = selection.length - 1; j >= 0; j--) {
      let range = selection[j];
      let startIndex = range.startIndex[axis];
      let endIndex = range.endIndex[axis];

      if (startIndex === index) {
        if (endIndex === index) {
          if (operation === 'delete') {
            selection.splice(j, 1);
            // eslint-disable-next-line no-continue
            continue;
          }
        }
        if (operation === 'delete') {
          let newKey = this._getKey(
            this._getAxisCellsByIndex(range.startIndex[axis] + 1, axis)[0], axis);
          range.startKey[axis] = newKey;
        } else {
          range.startIndex[axis] += adjustment;
        }
        range.endIndex[axis] += adjustment;
      } else if (endIndex === index) {
        if (operation === 'delete') {
          let newKey = this._getKey(
            this._getAxisCellsByIndex(range.endIndex[axis] - 1, axis)[0], axis);
          range.endKey[axis] = newKey;
        }
        range.endIndex[axis] += adjustment;
      } else if (index < startIndex) {
        range.startIndex[axis] += adjustment;
        range.endIndex[axis] += adjustment;
      } else if (index < endIndex) {
        range.endIndex[axis] += adjustment;
      }
    }
  }
};

/**
 * Handles model insert range event from datagrid provider
 * @private
 */
DvtDataGrid.prototype._handleInsertRangeEvent = function (eventDetail) {
  let axis = eventDetail.axis;
  let ranges = eventDetail.ranges;
  if (ranges.length === 0) {
    this.fillViewport();
    return;
  }

  // sort ranges forwards to ensure we add in correct order as order is relative to final
  ranges.sort(function (a, b) {
    return a.offset - b.offset;
  });

  let range = ranges.shift();

  let start = range.offset;
  let count = range.count;
  let flag = this._isAxisIndexInViewport(start, axis);
  if (flag === DvtDataGrid.INSIDE) {
    let startRow = start;
    let rowCount = count;
    let startCol = this.m_startCol;
    let colCount = (this.m_endCol - this.m_startCol) + 1;
    if (axis === 'column') {
      startRow = this.m_startRow;
      rowCount = (this.m_endRow - this.m_startRow) + 1;
      startCol = start;
      colCount = count;
    }
    let headerFragment = document.createDocumentFragment();
    let endHeaderFragment = document.createDocumentFragment();
    let promiseResolve;
    let promise = new Promise(function (resolve) {
      promiseResolve = resolve;
    });
    let commonProps = {
      axis: axis,
      range: range,
      headerFragment: headerFragment,
      endHeaderFragment: endHeaderFragment,
      totalDimension: 0,
      promiseResolve: promiseResolve
    };


    this.fetchHeaders(axis, start, headerFragment, endHeaderFragment, count, {
      success: this._handleInsertRangeHeaderFetchSuccess.bind(this, commonProps),
      error: this.handleCellsFetchError
    });
    this.fetchCells(this.m_databody, startRow, startCol, rowCount, colCount, {
        success: this._handleInsertRangeCellFetchSuccess.bind(this, commonProps),
        error: this.handleCellsFetchError
      });
    promise.then(this._handleInsertRangeEvent.bind(this, eventDetail));
  } else if (flag === DvtDataGrid.BEFORE) {
    let avg = this.m_avgRowHeight;
    let total = avg * count;
    let headerRoot = this.m_rowHeader;
    let endHeaderRoot = this.m_rowEndHeader;
    if (axis === 'row') {
      if (this.m_endRow >= 0) {
        this.m_startRow += count;
        this.m_endRow += count;
        this.m_startRowPixel += total;
        this.m_endRowPixel += total;
      }
      if (this.m_endRowHeader >= 0) {
        this.m_startRowHeader += count;
        this.m_endRowHeader += count;
        this.m_startRowHeaderPixel += total;
        this.m_endRowHeaderPixel += total;
      }
      if (this.m_endRowEndHeader >= 0) {
        this.m_startRowEndHeader += count;
        this.m_endRowEndHeader += count;
        this.m_startRowEndHeaderPixel += total;
        this.m_endRowEndHeaderPixel += total;
      }
    } else {
      avg = this.m_avgColWidth;
      total = avg * count;
      headerRoot = this.m_colHeader;
      endHeaderRoot = this.m_colEndHeader;
      if (this.m_endCol >= 0) {
        this.m_startCol += count;
        this.m_endCol += count;
        this.m_startColPixel += total;
        this.m_endColPixel += total;
      }
      if (this.m_endColHeader >= 0) {
        this.m_startColHeader += count;
        this.m_endColHeader += count;
        this.m_startColHeaderPixel += total;
        this.m_endColHeaderPixel += total;
      }
      if (this.m_endColEndHeader >= 0) {
        this.m_startColEndHeader += count;
        this.m_endColEndHeader += count;
        this.m_startColEndHeaderPixel += total;
        this.m_endColEndHeaderPixel += total;
      }
    }

    let indexes = new Array(count).fill(start).map((x, y) => x + y);
    let dimensions = new Array(count).fill(avg);
    this._modifyAndPushCells(indexes, dimensions, axis, headerRoot, endHeaderRoot, true);
    this._refreshDatabodyMap();
    this._handleInsertRangeEvent(eventDetail);
  } else if (flag === DvtDataGrid.AFTER) {
    if (axis === 'row') {
      this.m_stopRowFetch = false;
      this.m_stopRowHeaderFetch = false;
      this.m_stopRowEndHeaderFetch = false;
    } else {
      this.m_stopColumnFetch = false;
      this.m_stopColumnHeaderFetch = false;
      this.m_stopColumnEndHeaderFetch = false;
    }
    this._handleInsertRangeEvent(eventDetail);
  }
};

/**
* Handles model insert range event from datagrid provider
* @private
*/
DvtDataGrid.prototype._handleUpdateRangeEvent = function (eventDetail) {
  let ranges = eventDetail.ranges;
  if (ranges.length === 0) {
    this._highlightActive();
    this.applySelection();
    this.fillViewport();
    return;
  }

  // sort ranges forwards to ensure we add in correct order as order is relative to final
  ranges.sort(function (a, b) {
    return a.offset - b.offset;
  });

  let range = ranges.shift();

  let rowStart = range.rowOffset;
  let columnStart = range.columnOffset;
  let rowCount = range.rowCount === -1 ? this._getMaxBottom() : range.rowCount;
  let columnCount = range.columnCount === -1 ? this._getMaxRight() : range.columnCount;
  let rowEnd = rowStart + rowCount - 1;
  let columnEnd = columnStart + columnCount - 1;

  let rowStartFlag = this._isAxisIndexInViewport(rowStart, 'row');
  let rowEndFlag = this._isAxisIndexInViewport(rowEnd, 'row');
  let columnStartFlag = this._isAxisIndexInViewport(columnStart, 'column');
  let columnEndFlag = this._isAxisIndexInViewport(columnEnd, 'column');

  // if a portion of the range is in the viewport
  if (rowStartFlag !== DvtDataGrid.AFTER && rowEndFlag !== DvtDataGrid.BEFORE &&
      columnStartFlag !== DvtDataGrid.AFTER && columnEndFlag !== DvtDataGrid.BEFORE) {
    if (rowStartFlag === DvtDataGrid.BEFORE) {
      rowStart = this._getMaxTop();
    }
    if (rowEndFlag === DvtDataGrid.AFTER) {
      rowEnd = this._getMaxBottom();
    }
    if (columnStartFlag === DvtDataGrid.BEFORE) {
      columnStart = this._getMaxLeft();
    }
    if (columnEndFlag === DvtDataGrid.AFTER) {
      columnEnd = this._getMaxRight();
    }

    rowCount = rowEnd - rowStart + 1;
    columnCount = columnEnd - columnStart + 1;

    let promiseResolve;
    let promise = new Promise(function (resolve) {
      promiseResolve = resolve;
    });
    let commonProps = {
      promiseResolve: promiseResolve
    };

    this.fetchCells(this.m_databody, rowStart, columnStart, rowCount, columnCount, {
      success: this._handleUpdateRangeFetchSuccess.bind(this, commonProps),
      error: this.handleCellsFetchError
    });
    promise.then(this._handleUpdateRangeEvent.bind(this, eventDetail));
  } else {
    this._handleUpdateRangeEvent(eventDetail);
  }
};

DvtDataGrid.prototype._handleInsertRangeHeaderFetchSuccess =
  function (commonProps, headerSet, headerRange, endHeaderSet) {
  const axis = headerRange.axis;

  this._signalTaskEnd();
  this.m_fetching[axis] = false;

  const dir = this.getResources().isRTLMode() ? 'right' : 'left';
  const start = headerRange.start;
  const headerFragment = commonProps.headerFragment;
  const endHeaderFragment = commonProps.endHeaderFragment;
  const insertDimension = axis === 'row' ? 'top' : dir;

  let root;
  let axisStart;
  let insertReference;
  let insertPixel;
  let headerCount;
  let c = 0;
  let index;
  let returnVal;
  let className;
  let renderer;
  let levelCount;
  let leftPixel;
  let topPixel;
  let totalDimension = 0;
  if (headerSet != null) {
    className = this.getMappedStyle('headercell');
    if (axis === 'row') {
      root = this.m_rowHeader;
      axisStart = this.m_startRowHeader;
      levelCount = this.m_rowHeaderLevelCount;
      className += ' ' + this.getMappedStyle('rowheadercell');
    } else {
      root = this.m_colHeader;
      axisStart = this.m_startColHeader;
      levelCount = this.m_columnHeaderLevelCount;
      className += ' ' + this.getMappedStyle('colheadercell');
    }

    insertReference = this._getHeaderByIndex(start, levelCount - 1, root, levelCount, axisStart);
    insertPixel = this.getElementDir(insertReference, insertDimension);

    headerCount = headerSet.getCount();
    renderer = this.getRendererOrTemplate(axis);

    while (headerCount - c > 0) {
      if (axis === 'row') {
        leftPixel = 0;
        topPixel = insertPixel + totalDimension;
      } else {
        leftPixel = insertPixel + totalDimension;
        topPixel = 0;
      }

      index = start + c;
      returnVal = this.buildLevelHeaders(headerFragment, index, 0, leftPixel,
        topPixel, true, false, renderer, headerSet, axis,
        className, levelCount);
      c += returnVal.count;
      totalDimension += returnVal.totalHeaderDimension;
    }
    if (totalDimension > commonProps.totalDimension) {
      // eslint-disable-next-line no-param-reassign
      commonProps.totalDimension = totalDimension;
    }
  }

  totalDimension = 0;
  c = 0;
  if (endHeaderSet != null) {
    className = this.getMappedStyle('endheadercell');

    if (axis === 'row') {
      root = this.m_rowEndHeader;
      axisStart = this.m_startRowEndHeader;
      levelCount = this.m_rowEndHeaderLevelCount;
      className += ' ' + this.getMappedStyle('rowendheadercell');
    } else {
      root = this.m_colEndHeader;
      axisStart = this.m_startColEndHeader;
      levelCount = this.m_columnEndHeaderLevelCount;
      className += ' ' + this.getMappedStyle('colendheadercell');
    }

    insertReference = this._getHeaderByIndex(start, levelCount - 1, root, levelCount, axisStart);
    insertPixel = this.getElementDir(insertReference, insertDimension);

    headerCount = endHeaderSet.getCount();
    renderer = this.getRendererOrTemplate(axis + 'End');
    while (headerCount - c > 0) {
      if (axis === 'row') {
        leftPixel = 0;
        topPixel = insertPixel + totalDimension;
      } else {
        leftPixel = insertPixel + totalDimension;
        topPixel = 0;
      }

      index = start + c;
      returnVal = this.buildLevelHeaders(endHeaderFragment, index, 0, leftPixel,
        topPixel, true, false, renderer, endHeaderSet, axis + 'End',
        className, levelCount);
      c += returnVal.count;
      totalDimension += returnVal.totalHeaderDimension;
    }
    if (totalDimension > commonProps.totalDimension) {
      // eslint-disable-next-line no-param-reassign
      commonProps.totalDimension = totalDimension;
    }
  }
};

DvtDataGrid.prototype._handleInsertRangeCellFetchSuccess =
  function (commonProps, cellSet, cellRanges) {
    const range = commonProps.range;
    const axis = commonProps.axis;
    const newHeaderElements = commonProps.headerFragment;
    const newEndHeaderElements = commonProps.endHeaderFragment;
    let totalDimension = commonProps.totalDimension;
    const dir = this.getResources().isRTLMode() ? 'right' : 'left';
    const newCellElements = document.createDocumentFragment();
    let headerRoot = this.m_rowHeader;
    let endHeaderRoot = this.m_rowEndHeader;
    if (axis === 'column') {
      headerRoot = this.m_colHeader;
      endHeaderRoot = this.m_colEndHeader;
    }

    this._signalTaskEnd();
    this.m_fetching.cells = false;
    this.unhighlightSelection();

    if (cellSet) {
      const rowRange = cellRanges[0];
      const rowStart = rowRange.start;
      const columnRange = cellRanges[1];
      const columnStart = columnRange.start;

      const insertReference = this._getCellByIndex(this.createIndex(rowStart, columnStart));
      const topPixel = axis === 'row' ? this.getElementDir(insertReference, 'top') : this.m_startRowPixel;
      const leftPixel = axis === 'row' ? this.m_startColPixel : this.getElementDir(insertReference, dir);

      const returnVal = this._addCellsToFragment(
        newCellElements, cellSet, rowStart, topPixel, columnStart, leftPixel);
      totalDimension = Math.max(
        totalDimension, axis === 'row' ? returnVal.totalRowHeight : returnVal.totalColumnWidth);
    }

    const offset = range.offset;
    const count = range.count;
    let indexes = new Array(count).fill(offset).map((x, y) => x + y);
    let dimensions = new Array(count).fill(totalDimension / count);

    let hasData = newCellElements.childNodes.length;
    let hasHeaders = newHeaderElements.childNodes.length;
    let hasEndHeaders = newEndHeaderElements.childNodes.length;
    const databodyContent = this.m_databody.firstChild;

    this._modifyAndPushCells(indexes, dimensions, axis, headerRoot, endHeaderRoot, true);
    this._simpleAdjustSelectionOnChange('insert', indexes, axis);
    if (newCellElements.childNodes.length) {
      databodyContent.appendChild(newCellElements); // @HTMLUpdateOK
      this._refreshDatabodyMap();
    }

    this._insertHeaders(axis, offset, newHeaderElements, newEndHeaderElements);

    this.hideStatusText();

    if (axis === 'row') {
      if (hasData) {
        this.m_endRow += count;
        this.m_endRowPixel += totalDimension;
        this.m_stopRowFetch = false;
      }
      if (hasHeaders) {
        this.m_endRowHeader += count;
        this.m_endRowHeaderPixel += totalDimension;
        this.m_stopRowHeaderFetch = false;
      }
      if (hasEndHeaders) {
        this.m_endRowEndHeader += count;
        this.m_endRowEndHeaderPixel += totalDimension;
        this.m_stopRowEndHeaderFetch = false;
      }
      let databodyContentHeight = this.getElementHeight(databodyContent) - totalDimension;
      this.setElementHeight(databodyContent, databodyContentHeight);
      this.updateRowBanding();
    } else if (axis === 'column') {
      if (hasData) {
        this.m_endCol += count;
        this.m_endColPixel += totalDimension;
        this.m_stopColumnFetch = false;
      }
      if (hasHeaders) {
        this.m_endColHeader += count;
        this.m_endColHeaderPixel += totalDimension;
        this.m_stopColumnHeaderFetch = false;
      }
      if (hasEndHeaders) {
        this.m_endColEndHeader += count;
        this.m_endColEndHeaderPixel += totalDimension;
        this.m_stopColumnEndHeaderFetch = false;
      }
      var databodyContentWidth = this.getElementWidth(databodyContent) - totalDimension;
      this.setElementWidth(databodyContent, databodyContentWidth);
      this.updateColumnBanding();
    }

    this.applySelection();
    this._resetHeaderHighLight();

    this.resizeGrid();
    commonProps.promiseResolve();
};

DvtDataGrid.prototype._insertHeaders = function (
  axis, offset, newHeaderElements, newEndHeaderElements) {
  let headerRoot = this.m_rowHeader;
  let endHeaderRoot = this.m_rowEndHeader;
  let startLevelCount = this.m_rowHeaderLevelCount;
  let endLevelCount = this.m_rowEndHeaderLevelCount;
  let insertReference;
  let axisStart = this.m_startRowHeader;
  let endAxisStart = this.m_startRowEndHeader;
  let headerDimension = 'height';
  let dimensionToAdjust = 'top';
  if (axis === 'column') {
    headerRoot = this.m_colHeader;
    endHeaderRoot = this.m_colEndHeader;
    startLevelCount = this.m_columnHeaderLevelCount;
    endLevelCount = this.m_columnEndHeaderLevelCount;
    axisStart = this.m_startColHeader;
    endAxisStart = this.m_startColEndHeader;
    headerDimension = 'width';
    dimensionToAdjust = this.getResources().isRTLMode() ? 'right' : 'left';
  }

  let insertGroupingContainer = (
    container, root, levelCount, start, dimension, adjustDimension) => {
    while (container.childNodes.length) {
      let groupingContainer = container.firstChild;
      let header;
      if (this.m_utils.containsCSSClassName(groupingContainer,
        this.getMappedStyle('groupingcontainer'))) {
        header = groupingContainer.firstChild;
      } else {
        header = groupingContainer;
      }
      let extentInfo = header.extentInfo;
      let patchBefore = extentInfo.more.before;
      let patchAfter = extentInfo.more.after;
      let context = header[this.getResources().getMappedAttribute('context')];
      let index = context.index;
      let extent = context.extent;
      let level = context.level;

      if (patchBefore) {
        let existingGroupingContainer = this._getHeaderContainer(
          index - 1, level, 0, null, root, levelCount);
        let existingHeader = existingGroupingContainer.firstChild;
        let existingHeaderContext = header[this.getResources().getMappedAttribute('context')];
        existingHeaderContext.extent += extent;

        let existingExtent = this._getAttribute(existingGroupingContainer, 'extent', true);
        this._setAttribute(existingGroupingContainer, 'extent', existingExtent + extent);

        let existingDimension = this.getElementDir(existingHeader, dimension);
        let addDimension = this.getElementDir(header, dimension);
        this.setElementDir(existingHeader, existingDimension + addDimension, dimension);
      } else if (patchAfter) {
        let existingGroupingContainer = this._getHeaderContainer(
          index + extent, level, 0, null, root, levelCount);
        let existingHeader = existingGroupingContainer.firstChild;
        let existingHeaderContext = existingHeader[this.getResources().getMappedAttribute('context')];
        existingHeaderContext.extent += extent;
        existingHeaderContext.index -= extent;

        let existingExtent = this._getAttribute(existingGroupingContainer, 'extent', true);
        this._setAttribute(existingGroupingContainer, 'extent', existingExtent + extent);

        let existingStart = this._getAttribute(existingGroupingContainer, 'start', true);
        this._setAttribute(existingGroupingContainer, 'start', existingStart - extent);

        let existingDimension = this.getElementDir(existingHeader, dimension);
        let addDimension = this.getElementDir(header, dimension);
        this.setElementDir(existingHeader, existingDimension + addDimension, dimension);

        let existingDir = this.getElementDir(existingHeader, adjustDimension);
        this.setElementDir(existingHeader, existingDir - addDimension, adjustDimension);
      } else {
        let existingGroupingContainer = this._getHeaderContainer(
          index, level, 0, null, root, levelCount);
        if (existingGroupingContainer) {
          // the container exists, insert the header
          let prevHeader = this._getHeaderByIndex(index - 1, level, root, levelCount, start);
          if (prevHeader === null || prevHeader.parentNode !== existingGroupingContainer) {
            let insertAt = (level === levelCount - 1) ?
              existingGroupingContainer.childNodes[1] : existingGroupingContainer.childNodes[0];
            existingGroupingContainer.insertBefore(header, insertAt); // @HTMLUpdateOK
          } else if (prevHeader.nextSibling &&
            prevHeader.nextSibling.parentNode === existingGroupingContainer) {
            existingGroupingContainer.insertBefore(header, prevHeader.nextSibling); // @HTMLUpdateOK
          } else {
            existingGroupingContainer.appendChild(header); // @HTMLUpdateOK
          }
        } else {
          // the container doesn't exist insert the whole container after the previous one
          let previousGroupingContainer = this._getHeaderContainer(
            index - 1, level, 0, null, root, levelCount);
          if (previousGroupingContainer === null) {
            let scroller = root.firstChild;
            if (scroller.firstChild) {
              scroller.insertBefore(groupingContainer, scroller.firstChild); // @HTMLUpdateOK
            } else {
              scroller.appendChild(groupingContainer); // @HTMLUpdateOK
            }
          } else if (previousGroupingContainer.nextSibling) {
            previousGroupingContainer.parentNode.insertBefore( // @HTMLUpdateOK
              groupingContainer, previousGroupingContainer.nextSibling);
          } else {
            previousGroupingContainer.parentNode.appendChild(groupingContainer); // @HTMLUpdateOK
          }
        }
      }

      if (patchBefore || patchAfter) {
        let innerGroupingContainer = groupingContainer.querySelector(
          '.' + this.getMappedStyle('groupingcontainer'));
        if (innerGroupingContainer) {
          insertGroupingContainer(
            innerGroupingContainer, root, levelCount, start, dimension, adjustDimension);
        }
        container.removeChild(groupingContainer);
      }
    }
  };

  if (newHeaderElements.childNodes.length) {
    if (startLevelCount === 1) {
      insertReference = this._getHeaderByIndex(
        offset, startLevelCount - 1, headerRoot, startLevelCount, axisStart);
      headerRoot.firstChild.insertBefore( // @HTMLUpdateOK
        newHeaderElements, insertReference);
    } else {
      insertGroupingContainer(newHeaderElements, headerRoot, startLevelCount,
        axisStart, headerDimension, dimensionToAdjust);
    }
  }

  if (newEndHeaderElements.childNodes.length) {
    if (endLevelCount === 1) {
      insertReference = this._getHeaderByIndex(
        offset, endLevelCount - 1, endHeaderRoot, endLevelCount, endAxisStart);
      endHeaderRoot.firstChild.insertBefore( // @HTMLUpdateOK
        newEndHeaderElements, insertReference);
    } else {
      insertGroupingContainer(newEndHeaderElements, endHeaderRoot, endLevelCount,
        endAxisStart, headerDimension, dimensionToAdjust);
    }
  }
};

/**
 * Handles model insert event
 * @param {Object} indexes the indexes that identifies the row that got updated.
 * @param {Object} keys the key that identifies the row that got updated.
 * @private
 */
DvtDataGrid.prototype._handleModelInsertEvent = function (indexes, keys) {
  // checks if the new row/column is in the viewport
  var flag = this._isInViewport(indexes);
  // If the model inserted is just the next model fetch it
  if (flag === DvtDataGrid.INSIDE ||
      (flag === DvtDataGrid.AFTER && indexes.row === (this.m_endRow + 1))) {
    // an insert can only be a insert new row or new column.  A cell insert is
    // automatically treated as row insert, keys['row'/'column'] can be the number 0
    if (keys.row != null) {
      // if we have added to an empty grid just refresh, so we can fetch all headers
      if (this._databodyEmpty()) {
        this.empty();
        this.refresh(this.m_root);
      } else {
        // move all rows up an index
        this._modifyAxisCellContextIndex('row', indexes.row, (this.m_endRow - indexes.row) + 1, 1);
        this._refreshDatabodyMap();

        this.fetchHeaders('row', indexes.row, this.m_rowHeader, this.m_rowEndHeader, 1, {
          success: this._handleHeaderInsertsFetchSuccess
        });
        this.fetchCells(this.m_databody, indexes.row, this.m_startCol, 1,
          (this.m_endCol - this.m_startCol) + 1, {
            success: this._handleCellInsertsFetchSuccess
          });
      }
    }
    // else if (keys['column'] != null)
    // {
    // todo: handle column insert
    // }
  } else {
    if (flag === DvtDataGrid.BEFORE) {
      // move all rows up an index
      this._modifyAxisCellContextIndex('row', 0, this.m_endRow + 1, 1);
      this._refreshDatabodyMap();

      this.m_startRow += 1;
      this.m_startRowHeader += 1;
      this.m_endRow += 1;
      this.m_endRowHeader += 1;
      this.m_startRowPixel += this.m_avgRowHeight;
      this.m_startRowHeaderPixel += this.m_avgRowHeight;
      this.m_endRowPixel += this.m_avgRowHeight;
      this.m_endRowHeaderPixel += this.m_avgRowHeight;
      var row = this.m_databody.firstChild.firstChild;
      if (row != null) {
        this.pushRowsDown(row, this.m_avgRowHeight);
      }
      var rowHeader = this.m_rowHeader.firstChild.firstChild;
      if (rowHeader != null) {
        this.pushRowsDown(rowHeader, this.m_avgRowHeight);
      }
      var rowEndHeader = this.m_rowEndHeader.firstChild.firstChild;
      if (rowEndHeader != null) {
        this.pushRowsDown(rowEndHeader, this.m_avgRowHeight);
      }
    }

    this.scrollToIndex(indexes);
  }
};

/**
 * Handle a successful call to the data source fetchCells. Update the row and
 * cell DOM elements when necessary.
 * @param {Object} cellSet - a CellSet object which encapsulates the result set of cells
 * @param {Array.<Object>} cellRanges - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 */
DvtDataGrid.prototype._handleCellInsertsFetchSuccess = function (cellSet, cellRanges) {
  // so that grid will be resize
  // this.m_initialized = false;
  this.m_resizeRequired = true;

  // insert the row
  this.handleCellsFetchSuccess(cellSet, cellRanges, this.m_endRow >= cellRanges[0].start);

  // make sure the new row is in range
  var rowStart = cellRanges[0].start;
  this._scrollRowIntoViewport(rowStart);

  // clean up rows outside of viewport (for non high-water mark scrolling only)
  if (!this._isHighWatermarkScrolling()) {
    this._cleanupViewport('top');
  }
  this.updateRowBanding();
  this.m_stopRowFetch = false;
  if (this.m_endRowHeader !== -1) {
    this.m_stopRowHeaderFetch = false;
  }
  if (this.m_endRowEndHeader !== -1) {
    this.m_stopRowEndHeaderFetch = false;
  }
  // Need to fill viewport in the case of a silent delete of multiple records with an insert following.
  // i.e. a splice of the data which removes 2 models silently and adds 1 back in, need to add the last model to fill view
  this.fillViewport();
};

/**
 * Handle a successful call to the data source fetchHeaderss. Update the row header DOM elements when necessary.
 * @param {Object} headerSet - a HeaderSet object which encapsulates the result set of cells
 * @param {Object} headerRanges - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 */
DvtDataGrid.prototype._handleHeaderInsertsFetchSuccess =
  function (headerSet, headerRanges, endHeaderSet) {
    // so that grid will be resize
    this.m_resizeRequired = true;
    // insert the row
    this.handleHeadersFetchSuccess(headerSet, headerRanges, endHeaderSet,
      this.m_endRowHeader >= headerRanges.start);
  };

/**
 * Scrolls the row with index into the viewport
 * @param {number} index the row index
 * @private
 */
DvtDataGrid.prototype._scrollRowIntoViewport = function (index) {
  var rowCells = this._getAxisCellsByIndex(index, 'row');
  if (rowCells == null) {
    // something is wrong the newly inserted row does not exists
    return;
  }

  var viewportTop = this._getViewportTop();
  var viewportBottom = this._getViewportBottom();

  var rowTop = this.getElementDir(rowCells[0], 'top');
  var diff = viewportTop - rowTop;

  if (diff > 0) {
    // row added to top, scroll up
    this.scrollDelta(0, diff);
  } else {
    diff = viewportBottom - rowTop;
    if (diff < 0) {
      // row added to bottom, scroll down
      this.scrollDelta(0, diff);
    }
  }
};

/**
 * Handles model range insert event
 * @param {Object} cellSet the range of cells inserted.
 * @param {Object=} headerSet the row headers.
 * @param {Object=} endHeaderSet the row end headers.
 * @private
 */
DvtDataGrid.prototype._handleModelInsertRangeEvent = function (cellSet, headerSet, endHeaderSet) {
  var rowHeaderFragment;
  var c;
  var index;
  var totalRowHeight;
  var returnVal;
  var className;
  var renderer;
  var rowEndHeaderFragment;
  var empty = this._databodyEmpty();

  // reconstruct the cell ranges from result
  var rowStart = cellSet.getStart('row');
  var rowCount = cellSet.getCount('row');
  var columnStart = cellSet.getStart('column');
  var columnCount = cellSet.getCount('column');

  // do not insert if not in viewport yet
  if (rowStart > this.m_endRow + 1) {
    return;
  }

  // if empty refresh to get headers
  if (empty) {
    this.empty();
    this.refresh(this.m_root);
  } else if (this.m_utils.supportsTransitions()) {
    // create  a fragment with all of the row headers
    if (headerSet != null) {
      rowHeaderFragment = document.createDocumentFragment();
      var headerCount = headerSet.getCount();
      // add the headers to the row header
      totalRowHeight = 0;
      c = 0;
      className = this.getMappedStyle('headercell') + ' ' + this.getMappedStyle('rowheadercell');
      renderer = this.getRendererOrTemplate('row');
      while (headerCount - c > 0) {
        index = rowStart + c;
        returnVal = this.buildLevelHeaders(rowHeaderFragment, index, 0, 0,
          totalRowHeight, true,
          rowStart !== this.m_endRowHeader + 1,
          renderer, headerSet, 'row', className,
          this.m_rowHeaderLevelCount);
        c += returnVal.count;
        totalRowHeight += returnVal.totalHeaderDimension;
      }
    }

    // create  a fragment with all of the row headers
    if (endHeaderSet != null) {
      rowEndHeaderFragment = document.createDocumentFragment();
      var headerEndCount = endHeaderSet.getCount();
      // add the headers to the row header
      totalRowHeight = 0;
      c = 0;
      className = this.getMappedStyle('endheadercell') + ' ' +
        this.getMappedStyle('rowendheadercell');
      renderer = this.getRendererOrTemplate('rowEnd');
      while (headerEndCount - c > 0) {
        index = rowStart + c;
        returnVal = this.buildLevelHeaders(rowEndHeaderFragment, index, 0, 0,
          totalRowHeight,
          true, rowStart !== this.m_endRowEndHeader + 1,
          renderer, endHeaderSet, 'rowEnd', className,
          this.m_rowEndHeaderLevelCount);
        c += returnVal.count;
        totalRowHeight += returnVal.totalHeaderDimension;
      }
    }

    var newCellElements = document.createDocumentFragment();
    returnVal = this._addCellsToFragment(newCellElements, cellSet, rowStart
      , 0, columnStart, 0);
    if (newCellElements.childNodes.length === 0 &&
        (rowHeaderFragment == null || rowHeaderFragment.childNodes.length === 0) &&
        (rowEndHeaderFragment == null || rowEndHeaderFragment.childNodes.length === 0)) {
      return;
    }
    this._insertRowsWithAnimation(newCellElements, rowHeaderFragment, rowEndHeaderFragment,
      rowStart, cellSet.getCount('row'), returnVal.totalRowHeight, columnStart, columnCount);
  } else {
    var rowRange = { axis: 'row', start: rowStart, count: rowCount };
    var columnRange = { axis: 'column', start: columnStart, count: columnCount };
    if (headerSet != null) {
      var headerRange = {
        axis: 'row',
        header: this.m_rowHeader,
        endHeader: this.m_rowEndHeader,
        start: rowStart,
        count: headerSet.getCount()
      };
      this.m_fetching.row = headerRange;
      this._handleHeaderInsertsFetchSuccess(headerSet, headerRange, endHeaderSet);
    }

    this._modifyAxisCellContextIndex('row', rowStart, (this.m_endRow - rowStart) + 1, rowCount);
    this._refreshDatabodyMap();

    // insert the rows
    this._handleCellInsertsFetchSuccess(cellSet, [rowRange, columnRange]);
  }
};

/**
 * Handles model update event
 * @param {Object} indexes the indexes that identifies the row that got updated.
 * @param {Object} keys the key that identifies the row that got updated.
 * @private
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleModelUpdateEvent = function (indexes, keys, cellSet) {
  // if the new row/column is in the viewport
  var flag = this._isInViewport(indexes);
  if (flag === DvtDataGrid.INSIDE) {
    if (cellSet != null) {
      var renderer = this.getRendererOrTemplate('cell');
      var columnBandingInterval = this.m_options.getColumnBandingInterval();
      this._updateCellsInRow(cellSet, cellSet.getStart('row'), renderer, this.m_startCol, columnBandingInterval);
    } else {
      // if there is a row header update it
      if (this.m_endRowHeader !== -1) {
        // fetch the updated row header and row
        this.fetchHeaders('row', indexes.row, this.m_rowHeader, this.m_rowEndHeader, 1, {
          success: this._handleHeaderUpdatesFetchSuccess,
          error: this.handleHeadersFetchError
        });
      }

      this.fetchCells(this.m_databody, indexes.row, this.m_startCol, 1,
        (this.m_endCol - this.m_startCol) + 1, {
          success: this._handleCellUpdatesFetchSuccess,
          error: this.handleCellsFetchError
        });
    }
  }

  // if it's not in range then do nothing
};

/**
 * Handle a successful call to the data source fetchHeaderss. Update the row header DOM elements when necessary.
 * @param {Object} headerSet - a HeaderSet object which encapsulates the result set of cells
 * @param {Array} headerRange - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 * @param {Object} endHeaderSet - a HeaderSet object which encapsulates the result set of cells
 * @private
 */
DvtDataGrid.prototype._handleHeaderUpdatesFetchSuccess =
  function (headerSet, headerRange, endHeaderSet) {
    var axis = headerRange.axis;
    this.m_fetching[axis] = false;
    var rowStart = headerRange.start;

    this._replaceHeaders(this.buildRowHeaders.bind(this), headerSet, this.m_rowHeader,
      rowStart, this.m_startRowHeader);
    this._replaceHeaders(this.buildRowEndHeaders.bind(this), endHeaderSet, this.m_rowEndHeader,
      rowStart, this.m_startRowEndHeader);

    var row = this.m_rowHeader.firstChild.childNodes[rowStart - this.m_startRowHeader];

    if (this.m_active != null && this.m_active.type === 'header' &&
        (this.m_active.axis === 'row' || this.m_active.axis === 'rowEnd') &&
        this._getKey(row, 'row') === this.m_active.key) {
      this._highlightActive();
    }
    // end fetch
    this._signalTaskEnd();
    // should animate the fragment in the future like updateCells
  };

/**
 * Replace the headers on update
 * @param {Function} buildFunction
 * @param {Object|null|undefined} headerSet
 * @param {Element} root
 * @param {number} index
 * @param {number} start
 * @private
 */
DvtDataGrid.prototype._replaceHeaders = function (buildFunction, headerSet, root, index, start) {
  if (headerSet != null) {
    var fragment = buildFunction(root, headerSet, index, 1, true, true);
    var headerContent = root.firstChild;
    var row = headerContent.childNodes[index - start];
    headerContent.replaceChild(fragment, row);
  }
};

/**
 * Handle a successful call to the data source fetchCells. Update the row and
 * cell DOM elements when necessary.
 * @param {Object} cellSet - a CellSet object which encapsulates the result set of cells
 * @param {Array.<Object>} cellRange - [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 * @private
 */
DvtDataGrid.prototype._handleCellUpdatesFetchSuccess = function (cellSet, cellRange) {
  // fetch complete
  this.m_fetching.cells = false;

  var rowStart = cellRange[0].start;

  var renderer = this.getRendererOrTemplate('cell');
  var columnBandingInterval = this.m_options.getColumnBandingInterval();

  // update the cells in the row
  this._updateCellsInRow(cellSet, rowStart, renderer, this.m_startCol, columnBandingInterval);

  // end fetch
  this._signalTaskEnd();
};

/**
 * Retrieves the update animation duration.
 * @return {number} the animation duration.
 * @private
 */
DvtDataGrid.prototype._getUpdateAnimationDuration = function () {
  return DvtDataGrid.UPDATE_ANIMATION_DURATION;
};

/**
 * Adds cells to a row. Iterate over the cells passed in, create new div elements
 * for them settign appropriate styles, and append or prepend them to the row based on the start column.
 * @param {Object} cellSet - the result set of cell data
 * @param {number} rowIndex - the index of the row element
 * @param {function(Object)} renderer - the cell renderer
 * @param {number} columnStart - the index to start start adding at
 * @param {number} columnBandingInterval - the column banding interval
 * @private
 */
DvtDataGrid.prototype._updateCellsInRow =
  // eslint-disable-next-line no-unused-vars
  function (cellSet, rowIndex, renderer, columnStart, columnBandingInterval) {
    var fragment;
    var animationDuration = this._getUpdateAnimationDuration();

    var cells = this._getAxisCellsByIndex(rowIndex, 'row');
    var top = this.getElementDir(cells[0], 'top');

    // check whether animation should be used
    if (animationDuration === 0 || !this.m_utils.supportsTransitions()) {
      // clear the content of the row first
      this._removeFromArray(cells);

      fragment = document.createDocumentFragment();
      this._addCellsToFragment(fragment, cellSet, rowIndex, top, columnStart, this.m_startColPixel);
      this._populateDatabody(this.m_databody.firstChild, fragment);

      // re-apply selection and active cell since content changed
      if (this._isSelectionEnabled()) {
        this.applySelection();
      }
      this._highlightActive();

      // hide fetching text now that we are done
      this.hideStatusText();
    } else {
      var self = this;
      // animation start
      self._signalTaskStart();

      // clear the content of the row and refill it with new data
      this._removeFromArray(cells);

      fragment = document.createDocumentFragment();
      this._addCellsToFragment(fragment, cellSet, rowIndex, top, columnStart, this.m_startColPixel);
      cells = fragment.childNodes;

      // hide the row
      var width = this.getElementWidth(this.m_databody);
      for (var i = 0; i < cells.length; i++) {
        this.addTransformMoveStyle(cells[i], 0, 0, 'linear', width, 0, 0);
      }

      this._populateDatabody(this.m_databody.firstChild, fragment);

      cells = this._getAxisCellsByIndex(rowIndex, 'row');

      // hide fetching text now that we are done
      this.hideStatusText();

      var listener = function () {
        for (var ii = 0; ii < cells.length; ii++) {
          self.removeTransformMoveStyle(cells[ii]);
        }

        // re-apply selection and active cell since content changed
        if (self._isSelectionEnabled()) {
          self.applySelection();
        }
        self._highlightActive();

        // end animation
        self._signalTaskEnd();

        // runQueue if applicable
        self._runModelEventQueue();
      };

      this._onTransitionEnd(cells[cells.length - 1], listener, animationDuration);

      setTimeout(function () {
        // kick off animation
        for (var ii = 0; ii < cells.length; ii++) {
          self.addTransformMoveStyle(cells[ii], animationDuration + 'ms', 0, 'linear', 0, 0, 0);
        }
      }, 0);
    }
  };

/**
 * @private
 */
DvtDataGrid.prototype._handleUpdateRangeFetchSuccess = function (commonProps, cellSet, cellRange) {
  // fetch complete
  this.m_fetching.cells = false;


  const rowStart = cellRange[0].start;
  const rowCount = cellSet.getCount('row');
  const rowEnd = rowStart + rowCount - 1;


  const columnStart = cellRange[1].start;
  const columnCount = cellSet.getCount('column');
  const columnEnd = columnStart + columnCount - 1;


  var cells = this._getCellsInRange(rowStart, columnStart, rowEnd, columnEnd);
  const top = this.getElementDir(cells[0], 'top');
  const ltr = this.getResources().isRTLMode() ? 'right' : 'left';
  const left = this.getElementDir(cells[0], ltr);


  // clear the content of the row first
  this._removeFromArray(cells);


  let fragment = document.createDocumentFragment();
  this._addCellsToFragment(fragment, cellSet, rowStart, top, columnStart, left);
  this._populateDatabody(this.m_databody.firstChild, fragment);


  // hide fetching text now that we are done
  this.hideStatusText();


  // end fetch
  this._signalTaskEnd();


  commonProps.promiseResolve();
};

  DvtDataGrid.prototype._removeAndModifyCells = function (cells, axis) {
    cells.forEach((cell) => {
      let cellContext = cell[this.getResources().getMappedAttribute('context')];
      let extent = cellContext.extents[axis];
      if (extent === 1) {
        this._remove(cell);
      } else {
        cellContext.extent[axis] -= 1;
      }
    });
  };

  DvtDataGrid.prototype._removeAndModifyHeaders =
  function (headers, dimension, dimensionToSet, dir, index) {
    headers.forEach((header) => {
      let headerContext = header[this.getResources().getMappedAttribute('context')];
      let extent = headerContext.extent;
      let parent = header.parentNode;
      if (extent === 1) {
        this._remove(header);
        if (parent.childNodes.length === 0) {
          this._remove(parent);
        }
      } else {
        let headerDim = this.getElementDir(header, dimensionToSet);
        this.setElementDir(header, headerDim - dimension, dimensionToSet);
        headerContext.extent -= 1;
        let start = headerContext.index;
        if (start === index) {
          headerContext.index += 1;
          let headerDir = this.getElementDir(header, dir);
          this.setElementDir(header, headerDir + dimension, dir);
        }
        if (parent.classList.contains(this.getMappedStyle('groupingcontainer'))) {
          let groupExtent = this._getAttribute(parent, 'extent', true);
          let groupStart = this._getAttribute(parent, 'start', true);
          this._setAttribute(parent, 'extent', groupExtent - 1);
          if (start === index) {
            this._setAttribute(parent, 'start', groupStart + 1);
          }
        }
      }
    });
  };

  DvtDataGrid.prototype._handleDeleteRangeEvent = function (eventDetail) {
    let ranges = eventDetail.ranges;
    // use set to ensure unique indexes
    let indexSet = new Set();
    let axis = eventDetail.axis;
    let ltr = this.getResources().isRTLMode() ? 'right' : 'left';
    let selection = this.m_selection;
    // unhighlight borders around selection range
    for (let i = 0; i < selection.length; i++) {
      this._applyBorderClassesAroundRange(this.getElementsInRange(selection[i]), selection[i], false, 'Selected');
    }
    // extract all indexes that need to be removed
    ranges.forEach(function (range) {
      let start = range.offset;
      let count = range.count;
      for (var i = 0; i < count; i++) {
        indexSet.add(start + i);
      }
    });

    // sort indexes backwards to ensure we remove in correct order
    let indexes = Array.from(indexSet);
    indexes.sort(function (a, b) {
      return b - a;
    });

    // values to track through removal process
    let beforeDeletedDimension = 0;
    let insideDeletedDimension = 0;
    let afterDeletedDimension = 0;
    let beforeDeletedCount = 0;
    let insideDeletedCount = 0;

    // row/column conditional vars
    let headerRoot = this.m_rowHeader;
    let headerStart = this.m_startRowHeader;
    let endHeaderStart = this.m_startRowEndHeader;
    let endHeaderRoot = this.m_rowEndHeader;
    let levelCount = this.m_rowHeaderLevelCount;
    let endLevelCount = this.m_rowEndHeaderLevelCount;
    let avgDimension = this.m_avgRowHeight;
    let hasData = this.m_endRow !== -1;
    let hasHeaders = this.m_endRowHeader !== -1;
    let hasEndHeaders = this.m_endRowEndHeader !== -1;
    let dimensionToRetrieve = 'height';
    let dirToSet = 'top';
    if (axis === 'column') {
      headerStart = this.m_startColHeader;
      endHeaderStart = this.m_startColEndHeader;
      headerRoot = this.m_colHeader;
      endHeaderRoot = this.m_colEndHeader;
      levelCount = this.m_columnHeaderLevelCount;
      endLevelCount = this.m_columnEndHeaderLevelCount;
      avgDimension = this.m_avgColWidth;
      hasData = this.m_endCol !== -1;
      hasHeaders = this.m_endColHeader !== -1;
      hasEndHeaders = this.m_endColEndHeader !== -1;
      dimensionToRetrieve = 'width';
      dirToSet = ltr;
    }

    // track dimensions in array reverse the order of indexes for future modification of dom
    let dimensions = [];

    // we don't want to modify cells multiple times
    // first we remove all cells/headers while the have the correct index
    // we track the dimension change at each index
    // then we will shift all of the cells/headers up based on that
    // this means the only cells acted on multiple times are nested headers
    for (let i = 0; i < indexes.length; i++) {
      let index = indexes[i];
      let dimension = 0;
      let flag = this._isAxisIndexInViewport(index, axis);

      if (flag === DvtDataGrid.BEFORE || flag === DvtDataGrid.INSIDE) {
        if (flag === DvtDataGrid.BEFORE) {
          beforeDeletedCount += 1;
          dimension = avgDimension;
          beforeDeletedDimension += dimension;
        } else if (flag === DvtDataGrid.INSIDE) {
          insideDeletedCount += 1;
          dimension = this.getElementDir(
            this._getCellOrHeaderByIndex(index, axis), dimensionToRetrieve);
          insideDeletedDimension += dimension;

          let cells = this._getAxisCellsByIndex(index, axis);
          if (cells != null) {
            this._removeAndModifyCells(cells, axis);
          }

          let headers = this._getHeadersByIndex(
            index, headerRoot, levelCount, headerStart);
          if (headers.length) {
            this._removeAndModifyHeaders(
              headers, dimension, dimensionToRetrieve, dirToSet, index);
          }

          let endHeaders = this._getHeadersByIndex(
            index, endHeaderRoot, endLevelCount, endHeaderStart);
          if (endHeaders.length) {
            this._removeAndModifyHeaders(
              endHeaders, dimension, dimensionToRetrieve, dirToSet, index);
          }
        }
        dimensions.unshift(dimension);
      } else if (flag === DvtDataGrid.AFTER && this.m_options.getScrollPolicy() === 'scroll') {
        // only concerned with after rows if virtual scroll
        afterDeletedDimension += dimension;
      }
    }

    // we want to walk indexes in order to push things up and modify their context objects
    indexes.reverse();
    this._modifyAndPushCells(indexes, dimensions, axis, headerRoot, endHeaderRoot, false);
    this._simpleAdjustSelectionOnChange('delete', indexes.reverse(), axis);

    // cells in grid are now accurate refresh db map
    this._refreshDatabodyMap();

    var totalDimension = beforeDeletedDimension + insideDeletedDimension + afterDeletedDimension;
    var databodyContent = this.m_databody.firstChild;

    if (axis === 'row') {
      if (hasData) {
        this.m_startRow -= beforeDeletedCount;
        this.m_endRow = this.m_endRow - beforeDeletedCount - insideDeletedCount;
        this.m_startRowPixel -= beforeDeletedDimension;
        this.m_endRowPixel = this.m_endRowPixel - beforeDeletedDimension - insideDeletedDimension;
        this.m_stopRowFetch = false;
      }
      if (hasHeaders) {
        this.m_startRowHeader -= beforeDeletedCount;
        this.m_endRowHeader = this.m_endRowHeader - beforeDeletedCount - insideDeletedCount;
        this.m_startRowHeaderPixel -= beforeDeletedDimension;
        this.m_endRowHeaderPixel = this.m_endRowHeaderPixel -
         beforeDeletedDimension - insideDeletedDimension;
        this.m_stopRowHeaderFetch = false;
      }
      if (hasEndHeaders) {
        this.m_startRowEndHeader -= beforeDeletedCount;
        this.m_endRowEndHeader = this.m_endRowEndHeader - beforeDeletedCount - insideDeletedCount;
        this.m_startRowEndHeaderPixel -= beforeDeletedDimension;
        this.m_endRowEndHeaderPixel = this.m_endRowEndHeaderPixel -
         beforeDeletedDimension - insideDeletedDimension;
        this.m_stopRowEndHeaderFetch = false;
      }
      var databodyContentHeight = this.getElementHeight(databodyContent) - totalDimension;
      this.setElementHeight(databodyContent, databodyContentHeight);
      this.updateRowBanding();
    } else if (axis === 'column') {
      if (hasData) {
        this.m_startCol -= beforeDeletedCount;
        this.m_endCol = this.m_endCol - beforeDeletedCount - insideDeletedCount;
        this.m_startColPixel -= beforeDeletedDimension;
        this.m_endColPixel = this.m_endColPixel - beforeDeletedDimension - insideDeletedDimension;
        this.m_stopColumnFetch = false;
      }
      if (hasHeaders) {
        this.m_startColHeader -= beforeDeletedCount;
        this.m_endColHeader = this.m_endColHeader - beforeDeletedCount - insideDeletedCount;
        this.m_startColHeaderPixel -= beforeDeletedDimension;
        this.m_endColHeaderPixel = this.m_endColHeaderPixel -
          beforeDeletedDimension - insideDeletedDimension;
        this.m_stopColumnHeaderFetch = false;
      }
      if (hasEndHeaders) {
        this.m_startColEndHeader -= beforeDeletedCount;
        this.m_endColEndHeader = this.m_endColEndHeader - beforeDeletedCount - insideDeletedCount;
        this.m_startColEndHeaderPixel -= beforeDeletedDimension;
        this.m_endColEndHeaderPixel = this.m_endColEndHeaderPixel -
          beforeDeletedDimension - insideDeletedDimension;
        this.m_stopColumnEndHeaderFetch = false;
      }
      var databodyContentWidth = this.getElementWidth(databodyContent) - totalDimension;
      this.setElementWidth(databodyContent, databodyContentWidth);
      this.updateColumnBanding();
    }

    this.applySelection();
    if (this.m_utils.isTouchDevice()) {
      if (this.GetSelection().length) {
        this._moveTouchSelectionAffordance();
      } else {
        this._removeTouchSelectionAffordance(true);
      }
    }
    this.resizeGrid();

    this.m_resizeRequired = true;

    this.fillViewport();
};


/**
 * Handles model delete event
 * @param {Array|Object} indexes the indexes that identifies the row that got deleted.
 * @param {Array|Object} keys the key that identifies the row that got deleted.
 * @param {boolean} silent true if the datagrid should not fill the databody
 * @private
 */
DvtDataGrid.prototype._handleModelDeleteEvent = function (indexes, keys, silent) {
  var referenceRow;
  var rowHeader;
  var rowEndHeader;
  var cells;

  // make it an array if it's a single entry event
  if (!Array.isArray(keys)) {
    // eslint-disable-next-line no-param-reassign
    keys = new Array(keys);
    // eslint-disable-next-line no-param-reassign
    indexes = new Array(indexes);
  }

  var beforeRowsHeight = 0;
  var insideRowsHeight = 0;
  var afterRowsHeight = 0;
  var beforeRowsDeleted = 0;
  var insideRowsDeleted = 0;

  for (var i = 0; i < keys.length; i++) {
    var key = keys[i];
    var index = indexes[i];
    if (key.row != null) {
      var height = 0;
      var rowKey = key.row;
      var flag = this._isInViewport(index);
      if (flag === DvtDataGrid.BEFORE) {
        // should only happen in virtual scrolling

        // adjust the cell context index
        this._modifyAxisCellContextIndex('row', this.m_startRow,
          (this.m_endRow - this.m_startRow) + 1, -1);

        beforeRowsDeleted += 1;
        beforeRowsHeight += this.m_avgRowHeight;
        this.m_startRowPixel -= this.m_avgRowHeight;
        this.m_endRowPixel -= this.m_avgRowHeight;
        if (this.m_endRowHeader !== -1) {
          this.m_startRowHeaderPixel -= this.m_avgRowHeight;
          this.m_endRowHeaderPixel -= this.m_avgRowHeight;
        }
        cells = this._getAxisCellsByKey(rowKey, 'row');
        if (cells != null) {
          this.pushRowsUp(this.m_startRow, this.m_avgRowHeight);
        }
        rowHeader = this.m_rowHeader.firstChild.firstChild;
        if (rowHeader != null) {
          this.pushRowHeadersUp(rowHeader, this.m_avgRowHeight);
        }
        rowEndHeader = this.m_rowEndHeader.firstChild.firstChild;
        if (rowEndHeader != null) {
          this.pushRowHeadersUp(rowEndHeader, this.m_avgRowHeight);
        }
      } else if (flag === DvtDataGrid.INSIDE) {
        insideRowsDeleted += 1;

        cells = this._getAxisCellsByKey(rowKey, 'row');
        if (cells != null) {
          height = this.calculateRowHeight(cells[0]);

          index = cells[0][this.getResources().getMappedAttribute('context')].indexes.row;
          this._modifyAxisCellContextIndex('row', index + 1, this.m_endRow - index, -1);

          this._removeFromArray(cells);
          this.pushRowsUp(index, height);
          this.m_endRowPixel -= height;
        }
        rowHeader = this._findHeaderByKey(rowKey, this.m_rowHeader,
          this.getMappedStyle('rowheadercell'));
        if (rowHeader != null) {
          height = this.calculateRowHeaderHeight(rowHeader);
          referenceRow = rowHeader.nextSibling;
          this._remove(rowHeader);
          this.pushRowHeadersUp(referenceRow, height);
          this.m_endRowHeaderPixel -= height;
        }
        rowEndHeader = this._findHeaderByKey(rowKey, this.m_rowEndHeader,
          this.getMappedStyle('rowendheadercell'));
        if (rowEndHeader != null) {
          height = this.calculateRowHeaderHeight(rowEndHeader);
          referenceRow = rowEndHeader.nextSibling;
          this._remove(rowEndHeader);
          this.pushRowHeadersUp(referenceRow, height);
          this.m_endRowEndHeaderPixel -= height;
        }
        insideRowsHeight += height;
      } else if (this.m_options.getScrollPolicy() === 'scroll') {
        // flag === DvtDataGrid.AFTER
        // only include after rows if virtual scroll
        afterRowsHeight += this.m_avgRowHeight;
      }
    }
  }

  this._refreshDatabodyMap();

  this.m_startRow -= beforeRowsDeleted;
  this.m_endRow = this.m_endRow - beforeRowsDeleted - insideRowsDeleted;
  if (this.m_endRowHeader !== -1) {
    this.m_startRowHeader -= beforeRowsDeleted;
    this.m_endRowHeader = this.m_endRowHeader - beforeRowsDeleted - insideRowsDeleted;
  }
  if (this.m_endRowEndHeader !== -1) {
    this.m_startRowEndHeader -= beforeRowsDeleted;
    this.m_endRowEndHeader = this.m_endRowEndHeader - beforeRowsDeleted - insideRowsDeleted;
  }
  var totalHeight = beforeRowsHeight + insideRowsHeight + afterRowsHeight;

  // adjust the databody height
  var databodyContent = this.m_databody.firstChild;
  var databodyContentHeight = this.getElementHeight(databodyContent) - totalHeight;
  this.setElementHeight(databodyContent, databodyContentHeight);
  this.resizeGrid();

  if (!silent && this.m_moveActive !== true) {
    // so that grid will be resize
    this.m_resizeRequired = true;
    // check viewport to see if we need to fetch because of deleted row causing empty spaces
    this.m_stopRowFetch = false;
    if (this.m_endRowHeader !== -1) {
      this.m_stopRowHeaderFetch = false;
    }
    if (this.m_endRowEndHeader !== -1) {
      this.m_stopRowEndHeaderFetch = false;
    }
    this.fillViewport();
  }
  this.updateRowBanding();
};

/**
 * Handles model delete event with animation
 * @param {Array} keys the key that identifies the row that got deleted.
 * @private
 */
DvtDataGrid.prototype._handleModelDeleteEventWithAnimation = function (keys) {
  this._collapseRowsWithAnimation(keys);
};

/**
 * Helper method to process animated rows in responce on the model delete event
 * @param {Object} keys set of keys that identifies rows that got deleted.
 * @private
 */
DvtDataGrid.prototype._collapseRowsWithAnimation = function (keys) {
  var rowCells;
  var i;
  var j;
  var row;
  var rowHeadersToRemove;
  var rowEndHeadersToRemove;
  var rowHeader;
  var rowEndHeader;

  if (keys.length === 0) {
    return;
  }

  var self = this;
  // animation start
  self._signalTaskStart();
  // note we set the duration to 1 instead of 0 because some browsers do not invoke transition end listener if duration is 0
  var duration = this.m_processingEventQueue ? 1 : DvtDataGrid.COLLAPSE_ANIMATION_DURATION;
  var rowsToRemove = [];
  var totalRowHeight = 0;
  var rowHeaderSupport = this.m_endRowHeader !== -1;
  var rowEndHeaderSupport = this.m_endRowEndHeader !== -1;
  var databodyContent = this.m_databody.firstChild;

  var referenceCellsIndex =
    this._getIndex(this._getAxisCellsByKey(keys[0].row, 'row')[0], 'row') - 1;

  // all inherited animated rows should be hidden under previous rows in view
  for (i = referenceCellsIndex; i >= this.m_startRow; i--) {
    rowCells = this._getAxisCellsByIndex(i, 'row');
    if (this.getElementDir(rowCells[0], 'top') + this.getElementHeight(rowCells[0]) <
        this.m_currentScrollTop) {
      break;
    }

    for (j = 0; j < rowCells.length; j++) {
      this.changeStyleProperty(rowCells[j], this.getCssSupport('z-index'), 10);
    }
  }

  if (rowHeaderSupport) {
    rowHeadersToRemove = [];
    var referenceRowHeader =
        this._findHeaderByKey(keys[0].row, this.m_rowHeader,
          this.getMappedStyle('rowheadercell')).previousSibling;
    row = referenceRowHeader;
    while (row) {
      if (this.getElementDir(row, 'top') + this.getElementHeight(row) < this.m_currentScrollTop) {
        break;
      }
      this.changeStyleProperty(row, this.getCssSupport('z-index'), 10);
      row = row.previousSibling;
    }
  }

  if (rowEndHeaderSupport) {
    rowEndHeadersToRemove = [];
    var referenceRowEndHeader =
        this._findHeaderByKey(keys[0].row, this.m_rowEndHeader,
          this.getMappedStyle('rowendheadercell')).previousSibling;
    row = referenceRowEndHeader;
    while (row) {
      if (this.getElementDir(row, 'top') + this.getElementHeight(row) < this.m_currentScrollTop) {
        break;
      }
      this.changeStyleProperty(row, this.getCssSupport('z-index'), 10);
      row = row.previousSibling;
    }
  }

  // get the rows we need to remove and set the new top to align row bottom with
  // the reference row bottom, but keep it where it is for the time being
  for (i = 0; i < keys.length; i++) {
    var rowKey = keys[i].row;
    rowCells = this._getAxisCellsByKey(rowKey, 'row');
    if (rowCells.length) {
      rowsToRemove.push(rowCells);
      totalRowHeight += this.getElementHeight(rowCells[0]);
      for (j = 0; j < rowCells.length; j++) {
        this.setElementDir(rowCells[j], this.getElementDir(rowCells[j], 'top') - totalRowHeight,
          'top');
        this.addTransformMoveStyle(rowCells[j], 0, 0, 'linear', 0, totalRowHeight, 0);
      }
    }
    if (rowHeaderSupport) {
      rowHeader = this._findHeaderByKey(rowKey, this.m_rowHeader,
        this.getMappedStyle('rowheadercell'));
      if (rowHeader != null) {
        rowHeadersToRemove.push(rowHeader);
        this.setElementDir(rowHeader, this.getElementDir(rowHeader, 'top') - totalRowHeight,
          'top');
        this.addTransformMoveStyle(rowHeader, 0, 0, 'linear', 0, totalRowHeight, 0);
      }
    }
    if (rowEndHeaderSupport) {
      rowEndHeader = this._findHeaderByKey(rowKey, this.m_rowEndHeader,
        this.getMappedStyle('rowendheadercell'));
      if (rowEndHeader != null) {
        rowEndHeadersToRemove.push(rowEndHeader);
        this.setElementDir(rowEndHeader,
          this.getElementDir(rowEndHeader, 'top') - totalRowHeight, 'top');
        this.addTransformMoveStyle(rowEndHeader, 0, 0, 'linear', 0, totalRowHeight, 0);
      }
    }
  }

  // for all the rows after the collapse change the top values appropriately
  for (i = referenceCellsIndex + keys.length + 1; i <= this.m_endRow; i++) {
    // change the row top but keep it where it is
    rowCells = this._getAxisCellsByIndex(i, 'row');
    for (j = 0; j < rowCells.length; j++) {
      this.setElementDir(rowCells[j],
        this.getElementDir(rowCells[j], 'top') - totalRowHeight, 'top');
      this.addTransformMoveStyle(rowCells[j], 0, 0, 'linear', 0, totalRowHeight, 0);
    }
    if (rowHeaderSupport) {
      rowHeader = rowHeader.nextSibling;
      this.setElementDir(rowHeader, this.getElementDir(rowHeader, 'top') - totalRowHeight, 'top');
      this.addTransformMoveStyle(rowHeader, 0, 0, 'linear', 0, totalRowHeight, 0);
    }
    if (rowEndHeaderSupport) {
      rowEndHeader = rowEndHeader.nextSibling;
      this.setElementDir(rowEndHeader,
        this.getElementDir(rowEndHeader, 'top') - totalRowHeight, 'top');
      this.addTransformMoveStyle(rowEndHeader, 0, 0, 'linear', 0, totalRowHeight, 0);
    }
  }

  // listen to the last rows transition end
  var lastAnimationElement = this._getCellByIndex(this.createIndex(this.m_endRow, this.m_endCol));
  function transitionListener() {
    if (rowsToRemove.length) {
      self._modifyAxisCellContextIndex('row', referenceCellsIndex + keys.length + 1,
        self.m_endRow - (referenceCellsIndex + keys.length),
        -1 * keys.length);
    }

    if (rowHeaderSupport && rowHeadersToRemove.length) {
      self._modifyAxisHeaderContextIndex('row', referenceCellsIndex + keys.length + 1,
        self.m_endRow - (referenceCellsIndex + keys.length),
        -1 * keys.length);
    }

    if (rowEndHeaderSupport && rowEndHeadersToRemove.length) {
      self._modifyAxisHeaderContextIndex('rowEnd', referenceCellsIndex + keys.length + 1,
        self.m_endRow - (referenceCellsIndex + keys.length),
        -1 * keys.length);
    }

    for (var ii = 0; ii < rowsToRemove.length; ii++) {
      for (var jj = 0; jj < rowsToRemove[ii].length; jj++) {
        self._remove(rowsToRemove[ii][jj]);
      }
      if (rowHeaderSupport) {
        self._remove(rowHeadersToRemove[ii]);
      }
      if (rowEndHeaderSupport) {
        self._remove(rowEndHeadersToRemove[ii]);
      }
    }

    self._refreshDatabodyMap();

    // clean up the variables no longer need because event animation handling
    self.m_endRow -= rowsToRemove.length;
    self.m_endRowPixel -= totalRowHeight;
    self.m_stopRowFetch = false;
    if (rowHeaderSupport) {
      self.m_endRowHeader -= rowHeadersToRemove.length;
      self.m_endRowHeaderPixel -= totalRowHeight;
      self.m_stopRowHeaderFetch = false;
    }
    if (rowEndHeaderSupport) {
      self.m_endRowEndHeader -= rowHeadersToRemove.length;
      self.m_endRowEndHeaderPixel -= totalRowHeight;
      self.m_stopRowEndHeaderFetch = false;
    }

    self.setElementHeight(databodyContent, self.m_endRowPixel - self.m_startRowPixel);
    self.resizeGrid();
    self.updateRowBanding();
    if (self.m_modelEvents != null && self.m_modelEvents.length === 0 &&
       !self.m_moveActive) {
      self.fillViewport();
    }
    self._handleAnimationEnd();
  }
  // if (lastAnimationElement) {
  self._onTransitionEnd(lastAnimationElement, transitionListener, duration);

  // animate all rows
  this.m_animating = true;

  setTimeout(function () {
    for (i = referenceCellsIndex + 1; i <= self.m_endRow; i++) {
      // change the row top but keep it where it is
      rowCells = self._getAxisCellsByIndex(i, 'row');
      for (j = 0; j < rowCells.length; j++) {
        self.addTransformMoveStyle(rowCells[j], duration + 'ms', 0, 'ease-out', 0, 0, 0);
      }
      if (rowHeaderSupport) {
        rowHeader = self._getHeaderByIndex(i, 0, self.m_rowHeader,
          self.m_rowHeaderLevelCount, self.m_startRowHeader);
        self.addTransformMoveStyle(rowHeader, duration + 'ms', 0, 'ease-out', 0, 0, 0);
      }
      if (rowEndHeaderSupport) {
        rowEndHeader = self._getHeaderByIndex(i, 0, self.m_rowEndHeader,
          self.m_rowEndHeaderLevelCount,
          self.m_startRowEndHeader);
        self.addTransformMoveStyle(rowEndHeader, duration + 'ms', 0, 'ease-out', 0, 0, 0);
      }
    }
  }, 0);
  // } else {
  //  transitionListener();
  // }
};

/**
 * Clean up the datagrid animations by resetting transform vars and z-index
 * @private
 */
DvtDataGrid.prototype._handleAnimationEnd = function () {
  // cleanRows
  var i;
  var databodyContent = this.m_databody.firstChild;
  var rowHeaderContent = this.m_rowHeader.firstChild;
  var rowEndHeaderContent = this.m_rowEndHeader.firstChild;
  for (i = 0; i < databodyContent.childNodes.length; i++) {
    this.removeTransformMoveStyle(databodyContent.childNodes[i]);
    this.changeStyleProperty(databodyContent.childNodes[i], this.getCssSupport('z-index'),
      null, 'remove');
  }

  if (this.m_endRowHeader !== -1) {
    for (i = 0; i < rowHeaderContent.childNodes.length; i++) {
      this.removeTransformMoveStyle(rowHeaderContent.childNodes[i]);
      this.changeStyleProperty(rowHeaderContent.childNodes[i], this.getCssSupport('z-index'),
        null, 'remove');
    }
  }

  if (this.m_endRowEndHeader !== -1) {
    for (i = 0; i < rowEndHeaderContent.childNodes.length; i++) {
      this.removeTransformMoveStyle(rowEndHeaderContent.childNodes[i]);
      this.changeStyleProperty(rowEndHeaderContent.childNodes[i], this.getCssSupport('z-index'),
        null, 'remove');
    }
  }
  // end animation
  this.m_animating = false;

  // check event queue for outstanding model events
  this._runModelEventQueue();

  // if we signal ready before emptying the queue there may be outstanding events to immediately follow
  this._signalTaskEnd();
};

/**
 * Get a cell or header from a given key and axis, gets the first cell with that key
 * @private
 */
DvtDataGrid.prototype._getCellOrHeaderByKey = function (key, axis) {
  var element = null;
  var cells = this._getAxisCellsByKey(key, axis, true);
  if (cells != null && cells.length > 0) {
    element = cells[0];
  }

  if (element == null) {
    if (axis === 'row') {
      element = this._findHeaderByKey(key, this.m_rowHeader,
        this.getMappedStyle('rowheadercell'));
      if (element == null) {
        element = this._findHeaderByKey(key, this.m_rowEndHeader,
          this.getMappedStyle('rowendheadercell'));
      }
    } else if (axis === 'column') {
      element = this._findHeaderByKey(key, this.m_colHeader,
        this.getMappedStyle('colheadercell'));
      if (element == null) {
        element = this._findHeaderByKey(key, this.m_colEndHeader,
          this.getMappedStyle('colendheadercell'));
      }
    }
  }
  return element;
};

/**
 * Find the header element by key inside a given root and className
 * @param {string|null} key the key
 * @param {DocumentFragment|Element} root
 * @param {string} className
 * @return {Element|null} the row element
 * @private
 */
DvtDataGrid.prototype._findHeaderByKey = function (key, root, className) {
  if (root == null) {
    return null;
  }

  var headers;

  // getElementsByClassName support is IE9 and up
  if (root.getElementsByClassName) {
    headers = root.getElementsByClassName(className);
  } else {
    headers = root.childNodes;
  }
  for (var i = 0; i < headers.length; i++) {
    var header = headers[i];
    var headerKey = this._getKey(header);
    if (this._shallowThenDeepCompare(headerKey, key)) {
      return header;
    }
  }

  // can't find it, the row is not in viewport
  return null;
};

/**
 * Handles model refresh event
 * @private
 */
DvtDataGrid.prototype._handleModelRefreshEvent = function () {
  var visibility = this.getVisibility();
  this.m_focusOnRefresh = this.m_root.contains(document.activeElement);

  this.m_updateScrollPostionOnRefreshCallback();

  // if we are visible, make sure we are visible, and just refresh the datagrid
  // if we are hidden we want to change the state to refresh so the wrapper know to call refresh when we are shown.
  // if we are already in state refresh we do not need to update.
  // if we are in state render we do not want to update that.
  if (visibility === DvtDataGrid.VISIBILITY_STATE_VISIBLE) {
    this.empty();
    // if the app developer doesn't notify the grid that it has become hidden
    // check to make sure, if it isn't hidden, refresh if it is
    // supported in IE9+
    if (this.m_root.offsetParent != null) {
      this.refresh(this.m_root);
    } else {
      this.setVisibility(DvtDataGrid.VISIBILITY_STATE_REFRESH);
    }
  } else if (visibility === DvtDataGrid.VISIBILITY_STATE_HIDDEN) {
    this.empty();
    this.setVisibility(DvtDataGrid.VISIBILITY_STATE_REFRESH);
  }
};

/**
 * Handles data source fetch end (model sync) event
 * @param {Object} event the model event
 * @private
 */
DvtDataGrid.prototype._handleModelSyncEvent = function (event) {
  // Currently these are set to zero for now, may come from the event later
  var startRow = 0;
  var startRowPixel = 0;
  var startCol = 0;
  var startColPixel = 0;
  var pageSize = event.pageSize;

  // cancel previous fetch calls
  this.m_fetching = {};

  // reset ranges
  this.m_startRow = startRow;
  this.m_endRow = -1;
  this.m_startRowHeader = startRow;
  this.m_endRowHeader = -1;
  this.m_startRowEndHeader = startRow;
  this.m_endRowEndHeader = -1;
  this.m_startRowPixel = startRowPixel;
  this.m_endRowPixel = startRowPixel;
  this.m_startRowHeaderPixel = startRowPixel;
  this.m_endRowHeaderPixel = startRowPixel;
  this.m_startRowEndHeaderPixel = startRowPixel;
  this.m_endRowEndHeaderPixel = startRowPixel;
  this.m_startCol = startCol;
  this.m_endCol = -1;

  this.m_startColHeader = startCol;
  this.m_endColHeader = -1;
  this.m_startColEndHeader = startCol;
  this.m_endColEndHeader = -1;

  this.m_startColPixel = startColPixel;
  this.m_endColPixel = startColPixel;

  this.m_startColHeaderPixel = startColPixel;
  this.m_endColHeaderPixel = startColPixel;

  this.m_startColEndHeaderPixel = startColPixel;
  this.m_endColEndHeaderPixel = startColPixel;

  this.m_rowHeaderLevelCount = undefined;
  this.m_columnHeaderLevelCount = undefined;
  this.m_rowEndHeaderLevelCount = undefined;
  this.m_columnEndHeaderLevelCount = undefined;

  this.m_avgRowHeight = undefined;
  this.m_avgColWidth = undefined;

  this.m_isEstimateRowCount = undefined;
  this.m_isEstimateColumnCount = undefined;
  this.m_stopRowFetch = false;
  this.m_stopRowHeaderFetch = false;
  this.m_stopRowEndHeaderFetch = false;
  this.m_stopColumnFetch = false;
  this.m_stopColumnHeaderFetch = false;
  this.m_stopColumnEndHeaderFetch = false;

  this._clearScrollPositionKeys();

  // clear selections
  this.m_selection = null;
  this.m_selectionRange = null;
  this.m_active = null;
  this.m_prevActive = null;
  this.m_trueIndex = {};

  if (this.m_empty != null) {
    this.m_root.removeChild(this.m_empty);
    this.m_empty = null;
  }

  this._showHeader(this.m_rowHeader);
  this._showHeader(this.m_colHeader);
  this._showHeader(this.m_rowEndHeader);
  this._showHeader(this.m_colEndHeader);

  this._emptyHeaders();
  this._emptyHeaderLabels();
  var databodyContent = this.m_databody.firstChild;
  if (databodyContent != null) {
    this._emptyDatabody(databodyContent);
  }

  this.m_initialized = false;
  this.fetchHeaders('row', startRow, this.m_rowHeader, this.m_rowEndHeader, pageSize, {
    success: function (headerSet, headerRange, endHeaderSet) {
      this.handleHeadersFetchSuccess(headerSet, headerRange, endHeaderSet, false);
    }
  });
  this.fetchHeaders('column', startCol, this.m_colHeader, this.m_colEndHeader, undefined, {
    success: function (headerSet, headerRange, endHeaderSet) {
      this.handleHeadersFetchSuccess(headerSet, headerRange, endHeaderSet, false);
    }
  });

  this.fetchCells(this.m_databody, startRow, startCol, pageSize, null, {
    success: function (cellSet, cellRange) {
      this.handleCellsFetchSuccess(cellSet, cellRange);
    }
  });
};

/**
 * Empties the headers
 * @private
 */
DvtDataGrid.prototype._emptyHeaders = function () {
  var rowHeaderContent = this.m_rowHeader.firstChild;
  var rowEndHeaderContent = this.m_rowEndHeader.firstChild;
  if (rowHeaderContent != null) {
    this.m_utils.empty(rowHeaderContent);
  }
  if (rowEndHeaderContent != null) {
    this.m_utils.empty(rowEndHeaderContent);
  }

  var columnHeaderContent = this.m_colHeader.firstChild;
  var columnEndHeaderContent = this.m_colEndHeader.firstChild;
  if (columnHeaderContent != null) {
    this.m_utils.empty(columnHeaderContent);
  }
  if (columnEndHeaderContent != null) {
    this.m_utils.empty(columnEndHeaderContent);
  }
};

/**
 * Empties the headerLabels
 * @private
 */
DvtDataGrid.prototype._emptyHeaderLabels = function () {
  var rowHeaderLabels = this.m_headerLabels.row;
  var rowEndHeaderLabels = this.m_headerLabels.rowEnd;
  var columnHeaderLabels = this.m_headerLabels.column;
  var columnEndHeaderLabels = this.m_headerLabels.columnEnd;
  if (rowHeaderLabels && rowHeaderLabels.length) {
    rowHeaderLabels.forEach(rowLabel => {
      this.m_utils.empty(rowLabel);
    });
  }
  if (rowEndHeaderLabels && rowEndHeaderLabels.length) {
    rowEndHeaderLabels.forEach(rowEndLabel => {
      this.m_utils.empty(rowEndLabel);
    });
  }
  if (columnHeaderLabels && columnHeaderLabels.length) {
    columnHeaderLabels.forEach(columnLabel => {
      this.m_utils.empty(columnLabel);
    });
  }
  if (columnEndHeaderLabels && columnEndHeaderLabels.length) {
    columnEndHeaderLabels.forEach(columnEndLabel => {
      this.m_utils.empty(columnEndLabel);
    });
  }
};

/** ********************************** active cell navigation ******************************/
/**
 * Sets the active cell by index
 * @param {Object} index row and column index
 * @param {Event=} event the DOM event causing the active cell change
 * @param {boolean=} clearSelection true if we should clear the selection on active change
 * @param {boolean=} silent
 * @param {boolean=} shouldNotScroll
 * @private
 * @return {boolean} true if active was changed, false if not
 */
DvtDataGrid.prototype._setActiveByIndex =
  function (index, event, clearSelection, silent, shouldNotScroll) {
    return this._setActive(this._getCellByIndex(index), { type: 'cell', indexes: index },
      event, clearSelection, silent, shouldNotScroll);
  };

/**
 * Updates the active cell based on external set, do not fire events
 * @param {Object} activeObject set by application could be sparse
 * @param {boolean} shouldFocus
 * @param {boolean=} shouldNotScroll
 * @private
 */
DvtDataGrid.prototype._updateActive = function (activeObject, shouldFocus, shouldNotScroll) {
  // the activeObject is potentially sparse, try to get an element from it
  var newActiveElement;

  if (activeObject == null) {
    this._setActive(null, null, null, true, false, shouldNotScroll);
  } else if (activeObject.keys != null) {
    newActiveElement = this._getCellByKeys(activeObject.keys);
  } else if (activeObject.indexes != null) {
    newActiveElement = this._getCellByIndex(activeObject.indexes);
  } else if (activeObject.axis != null) {
    var level = activeObject.level == null ? 0 : activeObject.level;
    if (activeObject.axis === 'column') {
      if (activeObject.key != null) {
        newActiveElement = this._findHeaderByKey(activeObject.key, this.m_colHeader,
          this.getMappedStyle('colheadercell'));
      } else if (activeObject.index != null) {
        newActiveElement = this._getHeaderByIndex(activeObject.index, level,
          this.m_colHeader, this.m_columnHeaderLevelCount,
          this.m_startColHeader);
      }
    } else if (activeObject.axis === 'row') {
      if (activeObject.key != null) {
        newActiveElement = this._findHeaderByKey(activeObject.key, this.m_rowHeader,
          this.getMappedStyle('rowheadercell'));
      } else if (activeObject.index != null) {
        newActiveElement = this._getHeaderByIndex(activeObject.index, level,
          this.m_rowHeader, this.m_rowHeaderLevelCount,
          this.m_startRowHeader);
      }
    } else if (activeObject.axis === 'columnEnd') {
      if (activeObject.key != null) {
        newActiveElement = this._findHeaderByKey(activeObject.key, this.m_colEndHeader,
          this.getMappedStyle('colendheadercell'));
      } else if (activeObject.index != null) {
        newActiveElement = this._getHeaderByIndex(activeObject.index, level,
          this.m_colEndHeader,
          this.m_columnEndHeaderLevelCount,
          this.m_startColEndHeader);
      }
    } else if (activeObject.axis === 'rowEnd') {
      if (activeObject.key != null) {
        newActiveElement = this._findHeaderByKey(activeObject.key, this.m_rowEndHeader,
          this.getMappedStyle('rowendheadercell'));
      } else if (activeObject.index != null) {
        newActiveElement = this._getHeaderByIndex(activeObject.index, level,
          this.m_rowEndHeader,
          this.m_rowEndHeaderLevelCount,
          this.m_startRowEndHeader);
      }
    }
  }

  if (newActiveElement != null) {
    if (!shouldFocus) {
      this.m_shouldFocus = false;
    }
    this._setActive(newActiveElement, activeObject, null, true, false, shouldNotScroll);
  }
};

/**
 * Sets the active cell or header by element
 * @param {Element|null} element to set active to
 * @param {Event|null=} event the DOM event causing the active cell change
 * @param {boolean|null=} clearSelection true if we should clear the selection on active change
 * @param {boolean|null=} silent true if we should not fire events
 * @param {boolean|null=} shouldNotScroll true if we should not scroll before setting active (in case it came froma  scroll event, prevent loop)
 * @returns {boolean} true if active was changed, false if not
 */
DvtDataGrid.prototype._setActive =
  function (element, cellInfo, event, clearSelection, silent, shouldNotScroll) {
    if (cellInfo != null && !shouldNotScroll) {
      this._scrollToActive(cellInfo);
    }

    var active;

    // element always non-null for labels
    if (element != null) {
      active = this._createActiveObject(element);
      // see if the active cell is actually changing
      if (this._compareActive(active, this.m_active)) {
        // fire vetoable beforeCurrentCell event
        if (silent || this._fireBeforeCurrentCellEvent(active, this.m_active, event)) {
          this.m_prevActive = this.m_active;
          this.m_active = active;

          if (event && event.type === 'mousedown') {
            this.m_trueIndex = null;
          }

          if (clearSelection && this._isSelectionEnabled()) {
            this._clearSelection(event);
          }
          this._unhighlightActiveObject(this.m_prevActive);
          this._highlightActiveObject(this.m_active, this.m_prevActive);
          this._manageMoveCursor();
          if (this._isGridEditable()) {
            if (active.type === 'cell') {
              this._setEditableClone(element);
              this._updateEdgeCellBorders('');
            }
          }
          if (!silent) {
            this._fireCurrentCellEvent(active, event);
          }
          return true;
        }
      } else {
        // still wanted to make sure the cell was highlighted even if focus hasn't actually changed
        this._highlightActive();
      }
    } else if (!this.m_scrollIndexAfterFetch && !this.m_scrollHeaderAfterFetch) {
      if (silent || this._fireBeforeCurrentCellEvent(active, this.m_active, event)) {
        this.m_prevActive = this.m_active;
        this.m_active = null;
        this._unhighlightActiveObject(this.m_prevActive);
        if (!silent) {
          this._fireCurrentCellEvent(active, event);
        }
      }
      return true;
    }
    return false;
  };

/**
 * Create an active object from an element active object contains:
 * For header: type, axis, index, key, level
 * For cell: indexes, keys
 * @param {Element} element - the element to create an active object from
 * @return {Object} an active object
 */
DvtDataGrid.prototype._createActiveObject = function (element) {
  var context = element[this.getResources().getMappedAttribute('context')];
  if (this.m_utils.containsCSSClassName(element, this.getMappedStyle('headercell')) ||
      this.m_utils.containsCSSClassName(element, this.getMappedStyle('endheadercell'))) {
    return {
      type: 'header',
      axis: context.axis,
      index: this.getHeaderCellIndex(element),
      key: context.key,
      level: context.level
    };
  }
  if (this.m_utils.containsCSSClassName(element, this.getMappedStyle('headerlabel'))) {
    return {
      type: 'label',
      axis: context.axis,
      level: context.level
    };
  }

  return {
    type: 'cell',
    indexes: {
      row: context.indexes.row,
      column: context.indexes.column,
    },
    keys: {
      row: context.keys.row,
      column: context.keys.column
    },
    extents: {
      row: context.extents.row,
      column: context.extents.column
    }
  };
};

/**
 * Retrieve the active element.
 * @return {Element|null} the active cell or header cell
 * @private
 */
DvtDataGrid.prototype._getActiveElement = function () {
  return this._getElementFromActiveObject(this.m_active);
};

/**
 * Retrieve the element based on an active object.
 * @param {Object} active the object to get the element of
 * @return {Element|null} the active cell or header cell
 * @private
 */
DvtDataGrid.prototype._getElementFromActiveObject = function (active) {
  if (active != null) {
    if (active.type === 'header') {
      if (active.axis === 'row') {
        return this._findHeaderByKey(active.key, this.m_rowHeader,
          this.getMappedStyle('rowheadercell'));
      } else if (active.axis === 'column') {
        return this._findHeaderByKey(active.key, this.m_colHeader,
          this.getMappedStyle('colheadercell'));
      } else if (active.axis === 'rowEnd') {
        return this._findHeaderByKey(active.key, this.m_rowEndHeader,
          this.getMappedStyle('rowendheadercell'));
      } else if (active.axis === 'columnEnd') {
        return this._findHeaderByKey(active.key, this.m_colEndHeader,
          this.getMappedStyle('colendheadercell'));
      }
    } else if (active.type === 'label') {
      return this._getLabel(active.axis, active.level);
    } else {
      return this._getCellByIndex(active.indexes);
    }
  }
  return null;
};

/**
 * Compare two active objects to see if they are equal
 * @param {Object} active1 an active object
 * @param {Object} active2 a comparison active object
 * @return {boolean} true if not equal
 */
DvtDataGrid.prototype._compareActive = function (active1, active2) {
  if (active1 == null && active2 == null) {
    return false;
  } else if ((active1 == null && active2 != null) || (active1 != null && active2 == null)) {
    return true;
  } else if (active1.type === active2.type) {
    if (active1.type === 'header') {
      if (active1.index !== active2.index ||
          active1.key !== active2.key ||
          active1.axis !== active2.axis ||
          active1.level !== active2.level) {
        return true;
      }
    } else if (active1.type === 'label') {
      if (active1.axis !== active2.axis ||
          active1.level !== active2.level) {
        return true;
      }
    } else if (active1.indexes.row !== active2.indexes.row ||
               active1.indexes.column !== active2.indexes.column ||
               active1.keys.row !== active2.keys.row ||
               active1.keys.column !== active2.keys.column) {
      return true;
    }
  } else {
    return true;
  }
  return false;
};

/**
 * Fires an event before the current cell changes
 * @param {Object|undefined} newActive the new active information
 * @param {Object} oldActive the new active information
 * @param {Event|undefined} event the DOM event
 * @private
 * @return {boolean|undefined} true if event should continue
 */
DvtDataGrid.prototype._fireBeforeCurrentCellEvent = function (newActive, oldActive, event) {
  // the event contains the context info
  var details = {
    event: event,
    ui: {
      currentCell: newActive,
      previousCurrentCell: oldActive
    }
  };

  return this.fireEvent('beforeCurrentCell', details);
};

/**
 * Fires an event to tell the datagrid to update the currentCell option
 * @param {Object|undefined} active the new active information
 * @param {Event|undefined} event the DOM event
 * @private
 */
DvtDataGrid.prototype._fireCurrentCellEvent = function (active, event) {
  // the event contains the context info
  var details = {
    event: event,
    ui: active
  };

  return this.fireEvent('currentCell', details);
};

/**
 * Is the databody cell active
 * @return {boolean} true if active element is a cell
 * @private
 */
DvtDataGrid.prototype._isDatabodyCellActive = function () {
  return (this.m_active != null && this.m_active.type === 'cell');
};

/**
 * Update the context info based on active changess
 * @param {Object} activeObject
 * @param {Object} prevActiveObject
 */
DvtDataGrid.prototype._updateActiveContext = function (activeObject, prevActiveObject) {
  var axis;
  var level;
  var skip;
  var contextObj = {};

  if (activeObject.type === 'header') {
    axis = activeObject.axis;
    var index = activeObject.index;
    level = activeObject.level;

    if (activeObject.axis === 'row') {
      if (this.m_rowHeaderLevelCount > 1) {
        if (prevActiveObject == null ? true : !(level === prevActiveObject.level &&
                                                axis === prevActiveObject.axis)) {
          contextObj.level = level;
        }
      }
      if (prevActiveObject == null ? true : !(index === prevActiveObject.index &&
                                              axis === prevActiveObject.axis)) {
        contextObj.rowHeader = index;
      }
    } else if (axis === 'column') {
      if (this.m_columnHeaderLevelCount > 1) {
        if (prevActiveObject == null ? true : !(level === prevActiveObject.level &&
                                                axis === prevActiveObject.axis)) {
          contextObj.level = level;
        }
      }
      if (prevActiveObject == null ? true : !(index === prevActiveObject.index &&
                                              axis === prevActiveObject.axis)) {
        contextObj.columnHeader = index;
      }
    } else if (activeObject.axis === 'rowEnd') {
      if (this.m_rowEndHeaderLevelCount > 1) {
        if (prevActiveObject == null ? true : !(level === prevActiveObject.level &&
                                                axis === prevActiveObject.axis)) {
          contextObj.level = level;
        }
      }
      if (prevActiveObject == null ? true : !(index === prevActiveObject.index &&
                                              axis === prevActiveObject.axis)) {
        contextObj.rowEndHeader = index;
      }
    } else if (axis === 'columnEnd') {
      if (this.m_columnEndHeaderLevelCount > 1) {
        if (prevActiveObject == null ? true : !(level === prevActiveObject.level &&
                                                axis === prevActiveObject.axis)) {
          contextObj.level = level;
        }
      }
      if (prevActiveObject == null ? true : !(index === prevActiveObject.index &&
                                              axis === prevActiveObject.axis)) {
        contextObj.columnEndHeader = index;
      }
    }
    // update context info
    this._updateContextInfo(contextObj, skip);
  } else if (activeObject.type === 'cell') {
    // check whether the prev and current active cell is in the same row/column so that we can
    // skip row/column header info in aria-labelledby (to make the description more brief)
    if (prevActiveObject != null && prevActiveObject.type === 'cell' && activeObject != null && !this.m_externalFocus) {
      if (activeObject.indexes.row === prevActiveObject.indexes.row) {
        skip = 'row';
      } else if (activeObject.indexes.column === prevActiveObject.indexes.column) {
        skip = 'column';
      }
    }
    // update context info
    this._updateContextInfo(activeObject, skip);
  } else if (activeObject.type === 'label') {
    axis = activeObject.axis;
    level = activeObject.level;

    if (prevActiveObject == null || prevActiveObject.type !== 'label' ||
      (prevActiveObject.type === 'label' && prevActiveObject.axis !== axis) || this.m_externalFocus) {
      if (axis === 'column') {
        contextObj.columnHeaderLabel = level;
      } else if (axis === 'row') {
        contextObj.rowHeaderLabel = level;
      } else if (axis === 'rowEnd') {
        contextObj.rowEndHeaderLabel = level;
      } else if (axis === 'columnEnd') {
        contextObj.columnEndHeaderLabel = level;
      }
    }

    this._updateContextInfo(contextObj, skip);
  }
};

/**
 * Handles click to make a cell active
 * @param {Event} event
 * @private
 */
DvtDataGrid.prototype.handleDatabodyClickActive = function (event) {
  var target = /** @type {Element} */ (event.target);
  var cell = this.findCell(target);
  if (cell != null) {
    this._setActive(cell, this._createActiveObject(cell), event);
  }
};

/**
 * Handles click to select a header
 * @param {Event} event
 */
DvtDataGrid.prototype.handleHeaderClickActive = function (event, activeOnly) {
  var target = /** @type {Element} */ (event.target);
  var header = this.findHeader(target);
  if (header != null) {
    if (!activeOnly) {
      this._clearSelection(event);
    }
    this._setActive(header, this._createActiveObject(header), event);
  }
};

/**
 * Scroll to the active object
 * @param {Object} activeObject
 */
DvtDataGrid.prototype._scrollToActive = function (activeObject) {
  if (activeObject.type === 'header') {
    this.scrollToHeader(activeObject);
  } else if (activeObject.type === 'cell') {
    this.scrollToIndex(activeObject.indexes);
  }
  // do nothing if label
};

/**
 * Retrieve cell by keys, this is rarely used, more common to look up cells by index within the data grid
 * @param {Object} keys
 * @return {Element|null} the active cell
 * @private
 */
DvtDataGrid.prototype._getCellByKeys = function (keys) {
  if (keys == null || keys.row == null || keys.column == null ||
      this.m_databody == null || this.m_databody.firstChild == null) {
    return null;
  }

  var databodyContent = this.m_databody.firstChild;
  var cells = databodyContent.getElementsByClassName(this.getMappedStyle('cell'));
  for (var i = 0; i < cells.length; i++) {
    var cell = cells[i];
    var rowKey = this._getKey(cell, 'row');
    if (this._shallowThenDeepCompare(rowKey, keys.row)) {
      var columnKey = this._getKey(cell, 'column');
      if (this._shallowThenDeepCompare(columnKey, keys.column)) {
        return cell;
      }
    }
  }

  return null;
};

/**
 * @param {any} value1
 * @param {any} value2
 * @return {boolean}
 * @private
 */
DvtDataGrid.prototype._shallowThenDeepCompare = function (value1, value2) {
  if (value1 === value2) {
    return true;
  }
  return this._compareValuesCallback(value1, value2);
};

/**
 * Retrieve the keys of a cell
 * @param {Element} cell
 * @return {Object}
 */
DvtDataGrid.prototype.getCellKeys = function (cell) {
  var cellContext = cell[this.getResources().getMappedAttribute('context')];
  return this.createIndex(cellContext.keys.row, cellContext.keys.column);
};

/**
 * Retrieve the indexes of a cell
 * @param {Element} cell
 * @return {Object}
 */
DvtDataGrid.prototype.getCellIndexes = function (cell) {
  var cellContext = cell[this.getResources().getMappedAttribute('context')];
  return this.createIndex(cellContext.indexes.row, cellContext.indexes.column);
};

/**
 * Retrieve the extents of a cell
 * @param {Element} cell
 * @return {Object}
 */
DvtDataGrid.prototype.getCellExtents = function (cell) {
  var cellContext = cell[this.getResources().getMappedAttribute('context')];
  return this.createIndex(cellContext.extents.row, cellContext.extents.column);
};


/**
 * Retrieve the end indexes of a cell (index + extent - 1)
 * @param {Element} cell
 * @return {Object}
 */
DvtDataGrid.prototype.getCellEndIndexes = function (cell) {
  var cellIndexes = this.getCellIndexes(cell);
  var cellExtents = this.getCellExtents(cell);
  return this.createIndex(cellIndexes.row + (cellExtents.row - 1),
    cellIndexes.column + (cellExtents.column - 1));
};

/**
 * Retrieve the index of a cell/header along a given axis
 * @param {Element|undefined|null} element
 * @param {string=} axis
 * @return {number|null}
 */
DvtDataGrid.prototype._getIndex = function (element, axis) {
  if (element != null) {
    if (axis != null && this.m_utils.containsCSSClassName(element, this.getMappedStyle('cell'))) {
      var cellContext = element[this.getResources().getMappedAttribute('context')];
      if (axis === 'row') {
        return cellContext.indexes.row;
      }
      if (axis === 'column') {
        return cellContext.indexes.column;
      }
    } else {
      return this.getHeaderCellIndex(element);
    }
  }
  return null;
};

/**
 * Retrieve the extent of a cell along a given axis
 * @param {Element|undefined|null} element
 * @param {string} axis
 * @return {number|null}
 */
DvtDataGrid.prototype._getExtent = function (element, axis) {
  if (element != null) {
    if (axis != null && this.m_utils.containsCSSClassName(element, this.getMappedStyle('cell'))) {
      var cellContext = element[this.getResources().getMappedAttribute('context')];
      if (axis === 'row') {
        return cellContext.extents.row;
      }
      if (axis === 'column') {
        return cellContext.extents.column;
      }
    } else {
      return parseInt(this._getAttribute(element, 'extent', true), 10);
    }
  }
  return null;
};

/**
 * Retrieve the index of a header cell
 * @param {Element} header header cell element
 * @return {number}
 */
DvtDataGrid.prototype.getHeaderCellIndex = function (header) {
  var levelCount;
  var start;
  var index;

  var axis = this.getHeaderCellAxis(header);
  switch (axis) {
    case 'column':
      levelCount = this.m_columnHeaderLevelCount;
      start = this.m_startColHeader;
      break;
    case 'row':
      levelCount = this.m_rowHeaderLevelCount;
      start = this.m_startRowHeader;
      break;
    case 'columnEnd':
      levelCount = this.m_columnEndHeaderLevelCount;
      start = this.m_startColEndHeader;
      break;
    case 'rowEnd':
      levelCount = this.m_rowEndHeaderLevelCount;
      start = this.m_startRowEndHeader;
      break;
    default:
      return -1;
  }

  // if there are multiple levels on the row header
  if (levelCount > 1) {
    // get the groupingContainer's start value and set thtat to the index
    index = /** @type {number} */ (this._getAttribute(header.parentNode, 'start', true));
    // if this is the groupingContainer's first child rturn that value
    if (header === header.parentNode.firstChild) {
      return index;
    }
    // decrement the index by one for the first header element at the level above it
    index -= 1;
  } else {
    index = start;
  }

  while (header.previousSibling) {
    index += 1;
    // eslint-disable-next-line no-param-reassign
    header = header.previousSibling;
  }

  return index;
};

/**
 * Retrieve the axis of a header cell
 * @param {Element|undefined|null} header header cell element
 * @return {string|null} row or column
 */
DvtDataGrid.prototype.getHeaderCellAxis = function (header) {
  if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('colheadercell'))) {
    return 'column';
  } else if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('rowheadercell'))) {
    return 'row';
  } else if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('rowendheadercell'))) {
    return 'rowEnd';
  } else if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('colendheadercell'))) {
    return 'columnEnd';
  }
  return null;
};

/**
 * Retrieve the level of a header cell
 * @param {Element} header header cell element
 * @return {number|string} row or column
 */
DvtDataGrid.prototype.getHeaderCellLevel = function (header) {
  if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('colheadercell'))) {
    if (this.m_columnHeaderLevelCount === 1) {
      return 0;
    }
  } else if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('rowheadercell'))) {
    if (this.m_rowHeaderLevelCount === 1) {
      return 0;
    }
  } else if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('colendheadercell'))) {
    if (this.m_columnEndHeaderLevelCount === 1) {
      return 0;
    }
  } else if (this.m_utils.containsCSSClassName(header, this.getMappedStyle('rowendheadercell'))) {
    if (this.m_rowEndHeaderLevelCount === 1) {
      return 0;
    }
  }

  var level = this._getAttribute(header.parentNode, 'level', true);
  if (header === header.parentNode.firstChild) {
    return level;
  }
  // plus one case is if we are on the innermost level the headers do not have their own
  // grouping containers so if it is the first child it is the level of the grouping container
  // but all subsequent children are the next level in
  return level + this.getHeaderCellDepth(header.parentNode.firstChild);
};

/**
 * Retrieve the depth of a header cell
 * @param {Element} header header cell element
 * @return {string|number|null} row or column depth
 */
DvtDataGrid.prototype.getHeaderCellDepth = function (header) {
  return this._getAttribute(header, 'depth', true);
};

DvtDataGrid.prototype.getHeaderLabelLevel = function (headerLabel) {
  let context = headerLabel[this.getResources().getMappedAttribute('context')];
  if (context) {
    return context.level;
  }
  return null;
};

/**
 * Retrieve the axis of a header cell
 * @param {Element|undefined|null} header header cell element
 * @return {string|null} row or column
 */
DvtDataGrid.prototype.getHeaderLabelAxis = function (headerLabel) {
  let context = headerLabel[this.getResources().getMappedAttribute('context')];
  if (context) {
    return context.axis;
  }
  return null;
};

/**
 * Get resize header mode.
 * @param {Element|undefined|null} element header/header label element
 * @return {string|null} row or column
 */
DvtDataGrid.prototype._getResizeHeaderMode = function (element) {
  var resizeHeaderMode = 'row';
  if (this.m_utils.containsCSSClassName(element, this.getMappedStyle('colheadercell')) ||
      this.m_utils.containsCSSClassName(element, this.getMappedStyle('colendheadercell')) ||
      this.m_utils.containsCSSClassName(element, this.getMappedStyle('columnheaderlabel')) ||
      this.m_utils.containsCSSClassName(element, this.getMappedStyle('columnendheaderlabel'))) {
      resizeHeaderMode = 'column';
  }
  return resizeHeaderMode;
};


/**
 * Find the cell or header element (recursively if needed)
 * @private
 * @param {Element} elem
 * @return {Element|undefined|null}
 */
DvtDataGrid.prototype.findCellOrHeader = function (elem) {
  var cell = this.findCell(elem);
  if (cell == null) {
    cell = this.findHeader(elem);
  }
  return cell;
};

/**
 * Find the cell element (recursively if needed)
 * @private
 * @param {Element} elem
 * @return {Element|undefined|null}
 */
DvtDataGrid.prototype.findCell = function (elem) {
  return this.find(elem, 'cell');
};

/**
 * Find the label element (recursively if needed)
 * @private
 * @param {Element} elem
 * @return {Element|undefined|null}
 */
DvtDataGrid.prototype.findLabel = function (elem) {
  return this.find(elem, 'headerlabel');
};

/**
 * Find the cell element (recursively if needed)
 * @param {Element|undefined|null} elem
 * @param {string} key
 * @param {string=} className
 * @return {Element|undefined|null}
 */
DvtDataGrid.prototype.find = function (elem, key, className) {
  // if element is null or if we reach the root of DataGrid
  if (elem == null || elem === this.getRootElement()) {
    return null;
  }

  // recursively walk up the element and find the class name that matches the cell class name
  if (className == null) {
    // eslint-disable-next-line no-param-reassign
    className = this.getMappedStyle(key);
  }

  if (className == null) {
    return null;
  }

  // if the element contains the cell class name, then it's a cell, otherwise go up
  if (this.m_utils.containsCSSClassName(elem, className)) {
    return elem;
  }
  return this.find(elem.parentNode, key, className);
};

/**
 * Highlight the current active element
 * @param {Array=} classNames string of classNames to add to active element
 * @private
 */
DvtDataGrid.prototype._highlightActive = function (classNames) {
  this._highlightActiveObject(this.m_active, this.m_prevActive, classNames);
};


/**
 * Unhighlight the current active element
 * @param {Array=} classNames string of classNames to remove from active element
 * @private
 */
DvtDataGrid.prototype._unhighlightActive = function (classNames) {
  this._unhighlightActiveObject(this.m_active, classNames);
};

/**
 * Highlight the specified object
 * @param {Object} activeObject active to unhighlight
 * @param {Object} prevActiveObject last active to base aria properties on
 * @param {Array=} classNames string of classNames to add to active element
 * @private
 */
DvtDataGrid.prototype._highlightActiveObject =
  function (activeObject, prevActiveObject, classNames) {
    if (classNames == null && this.m_utils.shouldOffsetOutline()) {
      // eslint-disable-next-line no-param-reassign
      classNames = ['offsetOutline'];
    }
    if (activeObject != null) {
      var element = this._getElementFromActiveObject(activeObject);
      // possible in the virtual case
      if (element != null) {
        this.m_focusInHandler(element);
        if (classNames != null) {
          this._highlightElement(element, classNames);
        }
        if (this._isCellEditable() && activeObject.type === 'cell') {
          this._applyBorderClassesAroundRange(element,
            { startIndex: activeObject.indexes }, true, 'Edit');
        }
        this._setAriaProperties(activeObject, prevActiveObject, element);
      }
    }
  };

/**
 * Unhighlight the specified object
 * @param {Object} activeObject to unhighlight
 * @param {Array=} classNames string of classNames to remove from active element
 * @private
 */
DvtDataGrid.prototype._unhighlightActiveObject = function (activeObject, classNames) {
  if (classNames == null && this.m_utils.shouldOffsetOutline()) {
    // eslint-disable-next-line no-param-reassign
    classNames = ['offsetOutline'];
  }
  if (activeObject != null) {
    var element = this._getElementFromActiveObject(activeObject);
    if (element != null) {
      this.m_focusOutHandler(element);
      if (classNames != null) {
        this._unhighlightElement(element, classNames);
      }
      if (this._isGridEditable() && activeObject.type === 'cell') {
        this._applyBorderClassesAroundRange(element,
          { startIndex: activeObject.indexes }, false, 'Edit');
      }
      this._unsetAriaProperties(element);
    }
  }
};

/**
 * Highlight all the cells in a row
 * @param {number|string|null} value key/index
 * @param {string} axis
 * @param {string} indexOrKey
 * @param {string} addOrRemove
 * @param {Array} classNames
 */
DvtDataGrid.prototype._highlightCellsAlongAxis =
  function (value, axis, indexOrKey, addOrRemove, classNames) {
    var cells = indexOrKey === 'key' ?
      this._getAxisCellsByKey(/** @type {string} */ (value), axis) :
      this._getAxisCellsByIndex(/** @type {number} */ (value), axis);

    for (var i = 0; i < cells.length; i++) {
      if (addOrRemove === 'add') {
        this._highlightElement(cells[i], classNames);
      } else {
        this._unhighlightElement(cells[i], classNames);
      }
    }
  };

/**
 * Highlight an element adding classes in the provided array
 * @param {Element} element
 * @param {Array} classNames
 */
DvtDataGrid.prototype._highlightElement = function (element, classNames) {
  for (var i = 0; i < classNames.length; i++) {
    var className = this.getMappedStyle(classNames[i]);
    this.m_utils.addCSSClassName(element, className);
  }
};

/**
 * Unhighlight an element removing classes in the provided array
 * @param {Element} element
 * @param {Array} classNames
 */
DvtDataGrid.prototype._unhighlightElement = function (element, classNames) {
  for (var i = 0; i < classNames.length; i++) {
    var className = this.getMappedStyle(classNames[i]);
    this.m_utils.removeCSSClassName(element, className);
  }
};

/**
 * Unhighlight elements removing classes in the provided array
 * @param {Array} elements
 * @param {Array} classNames
 */
DvtDataGrid.prototype._unhighlightElementsByClassName = function (elements, classNames) {
  if (elements.length && classNames.length) {
    for (var i = 0; i < elements.length; i++) {
      this._unhighlightElement(elements[i], classNames);
    }
  }
};

/**
 * Reset all wai-aria properties on a cell or header.
 * @param {Object} activeObject active to unhighlight
 * @param {Object} prevActiveObject last active to base aria properties on
 * @param {Element} element the element to reset all wai-aria properties
 * @private
 */
DvtDataGrid.prototype._setAriaProperties = function (activeObject, prevActiveObject, element) {
  var label = this.getLabelledBy(activeObject, prevActiveObject, element);
  this._updateActiveContext(activeObject, prevActiveObject);

  element.setAttribute('tabIndex', 0);
  element.setAttribute('aria-labelledby', label);

  // check to see if we should focus on the cell later
  if ((this.m_cellToFocus == null || this.m_cellToFocus !== element) &&
      this.m_shouldFocus !== false) {
    element.focus();
  }
  this.m_shouldFocus = null;
};

/**
 * Reset all wai-aria properties on a cell or header.
 * @param {Element} element the element to reset all wai-aria properties.
 */
DvtDataGrid.prototype._unsetAriaProperties = function (element) {
  if (element != null) {
    // reset focus index
    element.setAttribute('tabIndex', -1);
    // remove aria related attributes
    element.removeAttribute('aria-labelledby');
  }
};

/**
 * Returns the wai-aria labelled by property for a cell
 * @param {Object} activeObject
 * @param {Object} prevActiveObject
 * @param {Element} element
 * @return {string} the wai-aria labelled by property for the cell
 */
DvtDataGrid.prototype.getLabelledBy = function (activeObject, prevActiveObject, element) {
  var previousElement;
  var key;
  var previousRowIndex;
  var previousColumnIndex;
  var label = '';
  var statesArray = [];
  const elementMetadata = element[this.getResources().getMappedAttribute('metadata')];

  if (activeObject.type === 'header') {
    // get the previous active header to compare what the screen reader needs to read for parent Ids,
    // should only need this if multi level header
    if (prevActiveObject != null && prevActiveObject.type === 'header' &&
        !this.m_externalFocus) {
      // remove optimization
      if (prevActiveObject.axis === 'row') {
        previousElement = this._getHeaderByIndex(prevActiveObject.index, prevActiveObject.level,
          this.m_rowHeader, this.m_rowHeaderLevelCount,
          this.m_startRowHeader);
      } else if (prevActiveObject.axis === 'column') {
        previousElement = this._getHeaderByIndex(prevActiveObject.index, prevActiveObject.level,
          this.m_colHeader, this.m_columnHeaderLevelCount,
          this.m_startColHeader);
      } else if (prevActiveObject.axis === 'rowEnd') {
        previousElement = this._getHeaderByIndex(prevActiveObject.index, prevActiveObject.level,
          this.m_rowEndHeader
          , this.m_rowEndHeaderLevelCount,
          this.m_startRowEndHeader);
      } else if (prevActiveObject.axis === 'columnEnd') {
        previousElement = this._getHeaderByIndex(prevActiveObject.index, prevActiveObject.level,
          this.m_colEndHeader,
          this.m_columnEndHeaderLevelCount,
          this.m_startColEndHeader);
      }
    }

    label = [
      this.createSubId('context'),
      this._getHeaderAndParentIds(element, previousElement)
    ].join(' ');
    var direction = element.getAttribute(this.getResources().getMappedAttribute('sortDir'));
    if (direction === 'ascending') {
      key = 'accessibleSortAscending';
      label = label + ' ' + this.createSubId('state');
    } else if (direction === 'descending') {
      key = 'accessibleSortDescending';
      label = label + ' ' + this.createSubId('state');
    }

    if (this.m_externalFocus === true) {
      label = [this.createSubId('summary'), label].join(' ');
      this.m_externalFocus = false;
    }

    if (key != null) {
      statesArray.push({ key: key, args: { id: '' } });
    }

    if (elementMetadata && elementMetadata.metadata) {
      if (elementMetadata.metadata.expanded === 'expanded') {
        statesArray.push({ key: 'accessibleExpanded', args: {} });
      } else if (elementMetadata.metadata.expanded === 'collapsed') {
        statesArray.push({ key: 'accessibleCollapsed', args: {} });
      }
    }

    element.setAttribute('tabIndex', 0);
  } else if (activeObject.type === 'label') {
    label = [this.createSubId('context'), this._getActiveElement().id].join(' ');
  } else {
    if (prevActiveObject != null) {
      if (prevActiveObject.type === 'header') {
        previousRowIndex = prevActiveObject.axis === 'row' ? prevActiveObject.index : null;
        previousColumnIndex = prevActiveObject.axis === 'column' ? prevActiveObject.index : null;
      } else if (prevActiveObject.type === 'cell') {
        previousRowIndex = prevActiveObject.indexes.row;
        previousColumnIndex = prevActiveObject.indexes.column;
      }
    }

    // Add the header labels
    var row = this._getHeaderLabelledBy('row', this.m_rowHeader,
      this.m_rowHeaderLevelCount, this.m_startRowHeader,
      this.m_endRowHeader, activeObject.indexes.row,
      previousRowIndex, element);
    var rowEnd = this._getHeaderLabelledBy('rowEnd', this.m_rowEndHeader,
      this.m_rowEndHeaderLevelCount, this.m_startRowEndHeader,
      this.m_endRowEndHeader, activeObject.indexes.row,
      previousRowIndex, element);
    var column = this._getHeaderLabelledBy('column', this.m_colHeader,
      this.m_columnHeaderLevelCount, this.m_startColHeader,
      this.m_endColHeader, activeObject.indexes.column,
      previousColumnIndex, element);
    var columnEnd = this._getHeaderLabelledBy('columnEnd', this.m_colEndHeader,
      this.m_columnEndHeaderLevelCount,
      this.m_startColEndHeader,
      this.m_endColEndHeader, activeObject.indexes.column,
      previousColumnIndex, element);

    // we want extent information at the end of the readout so put it in the state info
    var rowExtent = activeObject.extents.row;
    var columnExtent = activeObject.extents.column;
    if (rowExtent > 1) {
      statesArray.push({ key: 'accessibleRowSpanContext', args: { extent: rowExtent } });
    }
    if (columnExtent > 1) {
      statesArray.push({ key: 'accessibleColumnSpanContext', args: { extent: columnExtent } });
    }

    label = [this.createSubId('context'), row, rowEnd, column, columnEnd, element.id,
      this.createSubId('state')].join(' ');
    // remove double spaces rather than check everytime
    label = label.replace(/ +(?= )/g, '');

    if (this.m_externalFocus) {
      label = [this.createSubId('summary'), label].join(' ');
      this.m_externalFocus = false;
    }
  }

  let focusableElems = getFocusableElementsIncludingDisabled(element);
  if (focusableElems && focusableElems.length > 0) {
    statesArray.push({ key: 'accessibleContainsControls' });
  }

  this._updateStateInfo(statesArray);

  return label;
};

/**
 * Returns the header that is in line with a cell along an axis.
 * Key Note: in the case of row, we return the row not the headercell
 * @param {Element|undefined|null} cell the element for the cell
 * @param {string} axis the axis along which to find the header, 'row', 'column'
 * @param {boolean=} lastContained true to get the last header contained by the cell along an axis
 * @return {Element} the header Element along the axis
 */
DvtDataGrid.prototype.getHeaderFromCell = function (cell, axis, lastContained) {
  var index;
  var extent;
  var level;
  var container;
  var levelCount;
  var start;

  if (axis === 'row') {
    if (this.m_rowHeader != null) {
      index = this._getIndex(cell, 'row');
      extent = this._getExtent(cell, 'row');
      level = this.m_rowHeaderLevelCount - 1;
      container = this.m_rowHeader;
      levelCount = this.m_rowHeaderLevelCount;
      start = this.m_startRowHeader;
    }
  } else if (axis === 'column') {
    if (this.m_colHeader != null) {
      index = this._getIndex(cell, 'column');
      extent = this._getExtent(cell, 'column');
      level = this.m_columnHeaderLevelCount - 1;
      container = this.m_colHeader;
      levelCount = this.m_columnHeaderLevelCount;
      start = this.m_startColHeader;
    }
  } else if (axis === 'rowEnd') {
    if (this.m_rowEndHeader != null) {
      index = this._getIndex(cell, 'row');
      extent = this._getExtent(cell, 'row');
      level = this.m_rowEndHeaderLevelCount - 1;
      container = this.m_rowEndHeader;
      levelCount = this.m_rowEndHeaderLevelCount;
      start = this.m_startRowEndHeader;
    }
  } else if (axis === 'columnEnd') {
    if (this.m_colEndHeader != null) {
      index = this._getIndex(cell, 'column');
      extent = this._getExtent(cell, 'column');
      level = this.m_columnEndHeaderLevelCount - 1;
      container = this.m_colEndHeader;
      levelCount = this.m_columnEndHeaderLevelCount;
      start = this.m_startColEndHeader;
    }
  }

  if (index != null && level != null && index > -1) {
    if (lastContained) {
      index += extent - 1;
    }
    return this._getHeaderByIndex(index, level, container, levelCount, start);
  }

  return null;
};

/**
 * Creates a range object given the start and end index, will add in keys if they are passed in
 * @param {Object} range - the start index of the range a range object representing the start and end index
 * @return {Object} a range object representing the start and end index
 * @private
 */
DvtDataGrid.prototype._trimRangeForSelectionMode = function (range) {
  if (this.m_options.getSelectionMode() === 'row') {
    // drop the column index
    return this.createRange(this.createIndex(range.startIndex.row),
      this.createIndex(range.endIndex.row));
  }
  return range;
};

/**
 * Creates a range object given the start and end index, will add in keys if they are passed in
 * @param {Object} startIndex - the start index of the range
 * @param {Object=} endIndex - the end index of the range.  Optional, if not specified it represents a single cell/row
 * @param {Object=} startKey - the start key of the range.  Optional, if not specified it represents a single cell/row
 * @param {Object=} endKey - the end key of the range.  Optional, if not specified it represents a single cell/row
 * @return {Object} a range object representing the start and end index, along with the start and end key.
 */
DvtDataGrid.prototype.createRange = function (startIndex, endIndex, startKey, endKey) {
  var startRow;
  var endRow;
  var startColumn;
  var endColumn;
  var startRowKey;
  var endRowKey;
  var startColumnKey;
  var endColumnKey;

  if (endIndex) {
    // -1 means unbound
    if (startIndex.row < endIndex.row || endIndex.row === -1) {
      startRow = startIndex.row;
      endRow = endIndex.row;
      if (startKey) {
        startRowKey = startKey.row;
        endRowKey = endKey.row;
      }
    } else {
      startRow = endIndex.row;
      endRow = startIndex.row;
      if (startKey) {
        startRowKey = endKey.row;
        endRowKey = startKey.row;
      }
    }

    // row based selection does not have column specified for range
    if (!isNaN(startIndex.column) && !isNaN(endIndex.column)) {
      // -1 means unbound
      if (startIndex.column < endIndex.column || endIndex.column === -1) {
        startColumn = startIndex.column;
        endColumn = endIndex.column;
        if (startKey) {
          startColumnKey = startKey.column;
          endColumnKey = endKey.column;
        }
      } else {
        startColumn = endIndex.column;
        endColumn = startIndex.column;
        if (startKey) {
          startColumnKey = endKey.column;
          endColumnKey = startKey.column;
        }
      }

      // eslint-disable-next-line no-param-reassign
      startIndex = {
        row: startRow, column: startColumn
      };
      // eslint-disable-next-line no-param-reassign
      endIndex = {
        row: endRow, column: endColumn
      };
      if (startKey) {
        // eslint-disable-next-line no-param-reassign
        startKey = {
          row: startRowKey, column: startColumnKey
        };
        // eslint-disable-next-line no-param-reassign
        endKey = {
          row: endRowKey, column: endColumnKey
        };
      }
    } else {
      // eslint-disable-next-line no-param-reassign
      startIndex = {
        row: startRow
      };
      // eslint-disable-next-line no-param-reassign
      endIndex = {
        row: endRow
      };
      if (startKey) {
        // eslint-disable-next-line no-param-reassign
        startKey = {
          row: startRowKey, column: startColumnKey
        };
        // eslint-disable-next-line no-param-reassign
        endKey = {
          row: endRowKey, column: endColumnKey
        };
      }
    }
  }

  if (startKey) {
    return { startIndex: startIndex, endIndex: endIndex, startKey: startKey, endKey: endKey };
  }

  return { startIndex: startIndex, endIndex: endIndex };
};


/**
 * Creates a range object given the start and end index
 * @param {Object} startIndex - the start index of the range
 * @param {Object|undefined|null} endIndex - the end index of the range.
 * @param {Function} callback - the callback for the range to call when its fully fetched
 * @private
 */
DvtDataGrid.prototype._createRangeWithKeys = function (startIndex, endIndex, callback) {
  this._keys(startIndex, this._createRangeStartKeyCallback.bind(this, endIndex, callback));
};

/**
 * Creates a range object given the start and end index
 * @param {Object|null|undefined} endIndex - the end index of the range.
 * @param {Function} callback - the callback for the range to call when its fully fetched
 * @param {Object} startKey - the start key of the range
 * @param {Object} startIndex - the start index of the range
 * @private
 */
DvtDataGrid.prototype._createRangeStartKeyCallback =
  function (endIndex, callback, startKey, startIndex) {
    // keys will be the same
    if (endIndex != null && startIndex != null &&
      endIndex.row === startIndex.row && endIndex.column === startIndex.column) {
      this._createRangeEndKeyCallback(startKey, startIndex, callback, startKey, startIndex);
    } else if (endIndex) {
      // new keys needed
      this._keys(endIndex,
        this._createRangeEndKeyCallback.bind(this, startKey, startIndex, callback));
    } else {
      // create range from single key
      callback.call(this, {
        startIndex: startIndex,
        endIndex: startIndex,
        startKey: startKey,
        endKey: startKey
      });
    }
  };

/**
 * Creates a range object given the start and end index
 * @param {Object} startKey - the start key of the range
 * @param {Object} startIndex - the start index of the range
 * @param {Function} callback - the callback for the range to call when its fully fetched
 * @param {Object} endKey - the end key of the range.
 * @param {Object} endIndex - the end index of the range.
 * @private
 */
DvtDataGrid.prototype._createRangeEndKeyCallback =
  function (startKey, startIndex, callback, endKey, endIndex) {
    callback.call(this, this.createRange(startIndex, endIndex, startKey, endKey));
  };

/**
 * Retrieve the end index of the range, return start index if end index is undefined
 * @param {Object} range
 * @return {Object}
 */
DvtDataGrid.prototype.getEndIndex = function (range) {
  return (range.endIndex == null) ? range.startIndex : range.endIndex;
};

/**
 * Grabs all the elements in the databody which are within the specified range.
 * @param {Object} range - the range in which to get the elements
 * @param {number=} startRow
 * @param {number=} endRow
 * @param {number=} startCol
 * @param {number=} endCol
 * @return {Array}
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype.getElementsInRange = function (range, startRow, endRow, startCol, endCol) {
  var rangeStartColumn;
  var rangeEndColumn;
  var i;
  var j;
  var cell;

  if (startRow == null) {
    // eslint-disable-next-line no-param-reassign
    startRow = this.m_startRow;
  }
  if (endRow == null) {
    // eslint-disable-next-line no-param-reassign
    endRow = this.m_endRow + 1;
  }

  var startIndex = range.startIndex;
  var endIndex = this.getEndIndex(range);

  var rangeStartRow = startIndex.row;
  var rangeEndRow = endIndex.row;
  // index = -1 means unbounded index
  if (rangeEndRow === -1) {
    rangeEndRow = Number.MAX_VALUE;
  }

  // check if in the rendered range
  if (endRow < rangeStartRow || rangeEndRow < startRow) {
    return null;
  }

  if (!isNaN(startIndex.column) && !isNaN(endIndex.column)) {
    rangeStartColumn = startIndex.column;
    rangeEndColumn = endIndex.column;
    // index = -1 means unbounded index
    if (rangeEndColumn === -1) {
      rangeEndColumn = Number.MAX_VALUE;
    }

    // check if in the rendered range
    if ((this.m_endCol + 1) < rangeStartColumn || rangeEndColumn < this.m_startCol) {
      return null;
    }
  }

  var nodes = [];
  // now walk the databody to find the nodes in range
  var databodyContent = this.m_databody.firstChild;
  if (databodyContent == null) {
    return null;
  }

  // the range is within the databody, calculate the relative position
  rangeStartRow = Math.max(this.m_startRow, rangeStartRow);
  rangeEndRow = Math.min(this.m_endRow, rangeEndRow);

  // cell case
  if (!isNaN(rangeStartColumn) && !isNaN(rangeEndColumn)) {
    rangeStartColumn = Math.max(this.m_startCol, rangeStartColumn);
    rangeEndColumn = Math.min(this.m_endCol, rangeEndColumn);
    for (i = rangeStartRow; i <= rangeEndRow; i += 1) {
      for (j = rangeStartColumn; j <= rangeEndColumn; j += 1) {
        cell = this._getCellByIndex(this.createIndex(i, j));
        if (cell != null && nodes.indexOf(cell) === -1) {
          nodes.push(cell);
        }
      }
    }
  } else {
    // row case
    rangeStartColumn = Math.max(0, this.m_startCol);
    rangeEndColumn = Math.max(rangeStartColumn, this.m_endCol);
    for (i = rangeStartRow; i <= rangeEndRow; i += 1) {
      for (j = rangeStartColumn; j <= rangeEndColumn; j += 1) {
        cell = this._getCellByIndex(this.createIndex(i, j));
        if (cell != null && nodes.indexOf(cell) === -1) {
          nodes.push(cell);
        }
      }
    }
  }

  return nodes;
};

DvtDataGrid.prototype._getRangeInView = function (range) {
  // removes nulls and negatives from range, and ensures there is an end index to work with
  let startIndex = range.startIndex;
  let endIndex = this.getEndIndex(range);

  let rangeStartRow = startIndex.row;
  let rangeEndRow = endIndex.row;
  let rangeStartColumn = startIndex.column;
  let rangeEndColumn = endIndex.column;

  if (rangeEndRow === -1) {
    rangeEndRow = Number.MAX_VALUE;
  }
  if (rangeEndColumn === -1) {
    rangeEndColumn = Number.MAX_VALUE;
  }

  rangeStartRow = Math.max(this.m_startRow, rangeStartRow);
  rangeEndRow = Math.min(this.m_endRow, rangeEndRow);

  if (isNaN(startIndex.column) || isNaN(endIndex.column)) {
    rangeStartColumn = Math.max(0, this.m_startCol);
    rangeEndColumn = Math.max(rangeStartColumn, this.m_endCol);
  }

  rangeStartColumn = Math.max(this.m_startCol, rangeStartColumn);
  rangeEndColumn = Math.min(this.m_endCol, rangeEndColumn);

  return this.createRange(this.createIndex(rangeStartRow, rangeStartColumn),
    this.createIndex(rangeEndRow, rangeEndColumn));
};

// eslint-disable-next-line consistent-return
DvtDataGrid.prototype._applyBorderClassesAroundRange =
  function (elementsInRange, range, shouldAddClasses, classSuffix) {
  // make sure we have a databody and elements in the range
  let databodyContent = this.m_databody.firstChild;
  if (databodyContent == null || elementsInRange == null || elementsInRange.length === 0) {
    return;
  }

  let normalizedRange = this._getRangeInView(range);
  let rangeStartRow = normalizedRange.startIndex.row;
  let rangeStartColumn = normalizedRange.startIndex.column;
  let rangeEndRow = normalizedRange.endIndex.row;
  let rangeEndColumn = normalizedRange.endIndex.column;

  // we use a different border for these
  let isFirstRow = rangeStartRow === 0;
  let isFirstColumn = rangeStartColumn === 0;
  let startBorderRowIndex = isFirstRow ? 0 : rangeStartRow - 1;
  let endBorderRowIndex = rangeEndRow;
  let startBorderColumnIndex = isFirstColumn ? 0 : rangeStartColumn - 1;
  let endBorderColumnIndex = rangeEndColumn;

  for (let i = rangeStartRow; i <= rangeEndRow; i++) {
    let startCell = this._getCellByIndex(this.createIndex(i, startBorderColumnIndex));
    let endCell = this._getCellByIndex(this.createIndex(i, endBorderColumnIndex));
    let startClassPrefix = isFirstColumn ? 'start' : 'end';
    let endClassPrefix = 'end';
    let startClass = startClassPrefix + classSuffix;
    let endClass = endClassPrefix + classSuffix;
    if (shouldAddClasses) {
      this._highlightElement(startCell, [startClass]);
      this._highlightElement(endCell, [endClass]);
    } else {
      this._unhighlightElement(startCell, [startClass]);
      this._unhighlightElement(endCell, [endClass]);
    }
  }

  for (let i = rangeStartColumn; i <= rangeEndColumn; i++) {
    let startCell = this._getCellByIndex(this.createIndex(startBorderRowIndex, i));
    let endCell = this._getCellByIndex(this.createIndex(endBorderRowIndex, i));
    let startClassPrefix = isFirstRow ? 'top' : 'bottom';
    let endClassPrefix = 'bottom';
    let startClass = startClassPrefix + classSuffix;
    let endClass = endClassPrefix + classSuffix;
    if (shouldAddClasses) {
      this._highlightElement(startCell, [startClass]);
      this._highlightElement(endCell, [endClass]);
    } else {
      this._unhighlightElement(startCell, [startClass]);
      this._unhighlightElement(endCell, [endClass]);
    }
  }
};

/**
 * Read the full content of the active cell (or frontier cell) to the screen reader
 * @protected
 * @returns {boolean} true if there is content to read out
 */
DvtDataGrid.prototype.readCurrentContent = function () {
  var current;
  var currentCell;

  if (this.m_active == null) {
    return false;
  }

  if (this.m_active.type === 'header') {
    current = {};
    if (this.m_active.axis === 'row') {
      if (this.m_rowHeaderLevelCount > 1) {
        current.level = this.m_active.level;
      }
      current.rowHeader = this.m_active.index;
    } else {
      if (this.m_columnHeaderLevelCount > 1) {
        current.level = this.m_active.level;
      }
      current.columnHeader = this.m_active.index;
    }
    currentCell = this._getActiveElement();
  } else if (this.m_active.type === 'cell') {
    current = this.m_active.indexes;
    if (this._isSelectionEnabled() && this.isMultipleSelection()) {
      if (this.m_selectionFrontier != null) {
        current = this.m_selectionFrontier;
      }
    }
    // make sure there is an active cell or frontier cell
    if (current == null) {
      return false;
    }

    // find the cell div
    var range = this.createRange(current);
    var cell = this.getElementsInRange(range);
    if (cell == null || cell.length === 0) {
      return false;
    }

    currentCell = cell[0];
  } else {
    currentCell = this._getActiveElement();
  }

  this._updateActiveContext(this.m_active, this.m_prevActive);

  // Fill in the placeholder div aria-labelledby with the currentCell label so that it
  // will read out the contents of the current cell when focus is shifted to the
  // place holder div. Set tabIndex to -1 so that it can be focused but not tabbed into.
  this.m_placeHolder.tabIndex = -1;
  var labelledBy = currentCell.getAttribute('aria-labelledby');
  this.m_placeHolder.setAttribute('aria-labelledby', labelledBy + ' ' + currentCell.id);

  // Since JAWS screen reader requires changing focus in order to
  // trigger a read command, we are toggling between the placeholder
  // div and the currentCell.
  if (this.m_placeHolder === document.activeElement) {
    currentCell.focus();
  } else {
    this.m_placeHolder.focus();
  }
  return true;
};

/**
 * Enter actionable mode
 * @param {Element|undefined|null} element to set actionable
 * @returns {boolean} false
 */
DvtDataGrid.prototype._enterActionableMode = function (element, event) {
  // focus on first focusable item in the cell
  if (this._setFocusToFirstFocusableElement(element, event)) {
    this.m_focusOutHandler(element);
    this.setActionableMode(true);
  }
  return false;
};

/**
 * Exit actionable mode on the active cell if in actionable mode
 */
DvtDataGrid.prototype._exitActionableMode = function () {
  if (this.isActionableMode()) {
    var elem = this._getActiveElement();
    this.setActionableMode(false);
    disableAllFocusableElements(elem);
    this.m_focusInHandler(elem);
  }
};

/**
 * Re render a cell
 * @param {Element|undefined|null} cell
 * @param {string} mode
 * @param {string} classToToggle class to toggle on or off before rerendering
 * @param {Element|null=} editableClone
 */
DvtDataGrid.prototype._reRenderCell = function (cell, mode, classToToggle, editableClone) {
  var renderer = this.getRendererOrTemplate('cell');
  var cellContext = cell[this.getResources().getMappedAttribute('context')];
  var cellMetaData = cell[this.getResources().getMappedAttribute('metadata')];
  cellContext.mode = mode;

  // empty the cell
  this.m_utils.empty(cell);

  // now that the cell is empty toggle the appropraite edit classes so that alignment never has to shift
  if (this.m_utils.containsCSSClassName(cell, classToToggle)) {
    this.m_utils.removeCSSClassName(cell, classToToggle);
  } else {
    this.m_utils.addCSSClassName(cell, classToToggle);
  }

  if (editableClone) {
    while (editableClone.hasChildNodes()) {
      cell.appendChild(editableClone.firstChild);
    }
    this._destroyEditableClone();
  } else {
    this._renderContent(renderer, cellContext, cell, cellContext.data,
      this.buildCellTemplateContext(cellContext, cellMetaData));
  }
};

/**
 * Set the editable clone property
 * @param {Element|undefined|null} element to clone
 */
DvtDataGrid.prototype._setEditableClone = function (element) {
  this._destroyEditableClone();
  if (element != null) {
    var clone = element.cloneNode(false);
    clone.removeAttribute('id');
    this._createUniqueId(clone);
    clone[this.getResources().getMappedAttribute('context')] =
      element[this.getResources().getMappedAttribute('context')];
    clone[this.getResources().getMappedAttribute('metadata')] =
      element[this.getResources().getMappedAttribute('metadata')];
    clone[this.getResources().getMappedAttribute('context')].parentElement = clone;
    this.m_root.appendChild(clone);
    this._reRenderCell(clone, 'edit', this.getMappedStyle('cellEdit'), null);
    // we can keep the editable clone around, no where should we look for it later
    clone.style.display = 'none';
    clone[this.getResources().getMappedAttribute('context')].parentElement = element;
    this.m_editableClone = clone;
  }
};

/**
 * Remove editable clone and cleanup
 */
DvtDataGrid.prototype._destroyEditableClone = function () {
  if (this.m_editableClone) {
    if (this.m_editableClone.parentNode != null) {
      this.m_root.removeChild(this.m_editableClone);
    }
    delete this.m_editableClone;
  }
};

/**
 *
 * @param {number} keyCode
 * @return {boolean}
 */
DvtDataGrid.prototype.isArrowKey = function (keyCode) {
  return (keyCode === this.keyCodes.UP_KEY ||
          keyCode === this.keyCodes.DOWN_KEY ||
          keyCode === this.keyCodes.LEFT_KEY ||
          keyCode === this.keyCodes.RIGHT_KEY);
};

/**
 * Creates an index object for the cell/row
 * @param {*} row - the start index of the range
 * @param {*} column - the end index of the range.  Optional, if not specified it represents a single cell/row
 * @return {Object} an index object
 */
DvtDataGrid.prototype.createIndex = function (row, column) {
  var returnObj = {};
  if (row !== undefined) {
    returnObj.row = row;
  }
  if (column !== undefined) {
    returnObj.column = column;
  }
  return returnObj;
};

/**
 * Checks if the corners of a selection matches the selection frontier index and parameter index
 * @param {string} axis - the axis of the selection if header
 * @param {number} index - one index of the selection
 * @param {Object} selection - selection being checked
 * @return {boolean} true or false whether or not the corners are matched
 */
DvtDataGrid.prototype.checkCorners = function (axis, index, selection) {
  // ignore nested headers
  if ((this.m_selectionFrontier.axis === 'column' &&
       this.m_columnHeaderLevelCount !== this.m_selectionFrontier.level) ||
      (this.m_selectionFrontier.axis === 'row' &&
       this.m_rowHeaderLevelCount !== this.m_selectionFrontier.level) ||
      (this.m_selectionFrontier.axis === 'rowEnd' &&
       this.m_rowEndHeaderLevelCount !== this.m_selectionFrontier.level) ||
      (this.m_selectionFrontier.axis === 'columnEnd' &&
       this.m_columnEndHeaderLevelCount !== this.m_selectionFrontier.level)) {
    return true;
  }
  if (axis.indexOf('row') !== -1) {
    return ((index === selection.startIndex.row &&
             this.m_selectionFrontier.end === selection.endIndex.row) ||
            (index === selection.endIndex.row &&
             this.m_selectionFrontier.end === selection.startIndex.row));
  } else if (axis.indexOf('column') !== -1) {
    return ((index === selection.startIndex.column &&
             this.m_selectionFrontier.end === selection.endIndex.column) ||
            (index === selection.endIndex.column &&
             this.m_selectionFrontier.end === selection.startIndex.column));
  }

  return false;
};

/**
 * Handles arrow keys navigation on label
 * @param {number} keyCode description
 * @param {Event} event the DOM event that caused the arrow key
 * @param {boolean} isExtend boolean if we are extending a selection
 * @param {boolean} jumpToHeaders jump to opposite labels if possible
 * @return  boolean true if the event was processed
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype.handleLabelFocusChange = function (keyCode, event, isExtend, jumpToHeaders) {
  var newElement;
  var newIndex;
  var root;
  var start;
  var levelCount;

  // ensure that there's no outstanding fetch requests
  if (!this.isFetchComplete()) {
    // act like it's processed until we finish the fetch
    return true;
  }

  if (this.getResources().isRTLMode()) {
    if (keyCode === this.keyCodes.LEFT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.RIGHT_KEY;
    } else if (keyCode === this.keyCodes.RIGHT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.LEFT_KEY;
    }
  }

  var axis = this.m_active.axis;
  var level = this.m_active.level;

  if (axis === 'column') {
    root = this.m_colHeader;
    start = this.m_startColHeader;
    levelCount = this.m_columnHeaderLevelCount;
  } else if (axis === 'row') {
    root = this.m_rowHeader;
    start = this.m_startRowHeader;
    levelCount = this.m_rowHeaderLevelCount;
  }
  if (axis === 'columnEnd') {
    // treat up and down keys opposite of column Headers
    if (keyCode === this.keyCodes.DOWN_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.UP_KEY;
    } else if (keyCode === this.keyCodes.UP_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.DOWN_KEY;
    }
    root = this.m_colEndHeader;
    start = this.m_startColEndHeader;
    levelCount = this.m_columnEndHeaderLevelCount;
  }
  if (axis === 'rowEnd') {
    // treat right and left oppostie of row headers
    if (keyCode === this.keyCodes.LEFT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.RIGHT_KEY;
    } else if (keyCode === this.keyCodes.RIGHT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.LEFT_KEY;
    }
    root = this.m_rowEndHeader;
    start = this.m_startRowEndHeader;
    levelCount = this.m_rowEndHeaderLevelCount;
  }

  switch (keyCode) {
    case this.keyCodes.DOWN_KEY:
      if (axis === 'row' || axis === 'rowEnd') {
        newElement = this._getHeaderByIndex(start, level, root, levelCount, start);
        if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
          // unhighlight and clear selection
          this._clearSelection(event);
          this.m_selectionFrontier = {};
        }
        this._setActive(newElement, {
          type: 'header',
          index: start,
          level: level,
          axis: axis
        }, event);
      }
      if (axis === 'column' || axis === 'columnEnd') {
        newElement = this.m_headerLabels[axis][level + 1];
        if (newElement) {
          this._setActive(newElement, { type: 'label', level: level + 1, axis: axis }, event);
        } else if (axis === 'column' && level === levelCount - 1 &&
                   this.m_headerLabels.row.length) {
          newElement = this.m_headerLabels.row[this.m_headerLabels.row.length - 1];
          this._setActive(newElement, {
            type: 'label',
            level: this.m_headerLabels.row.length - 1,
            axis: 'row'
          }, event);
        } else if (level === levelCount - 1) {
          newIndex = axis === 'column' ? this.m_startRowHeader : this.m_endRowHeader;
          newElement = this._getHeaderByIndex(newIndex, this.m_rowHeaderLevelCount - 1,
            this.m_rowHeader, this.m_rowHeaderLevelCount,
            this.m_startRowHeader);
          if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
            // unhighlight and clear selection
            this._clearSelection(event);
            this.m_selectionFrontier = {};
          }
          if (newElement) {
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: level,
              axis: 'row'
            }, event);
          }
        }
      }
      break;
    case this.keyCodes.UP_KEY:
      if (axis === 'row' && level === levelCount - 1 && this.m_headerLabels.column.length) {
        newElement = this.m_headerLabels.column[this.m_headerLabels.column.length - 1];
        this._setActive(newElement, {
          type: 'label',
          level: this.m_headerLabels.column.length - 1,
          axis: 'column'
        }, event);
      }
      if (axis === 'column' || axis === 'columnEnd') {
        newElement = this.m_headerLabels[axis][level - 1];
        if (newElement) {
          this._setActive(newElement, { type: 'label', level: level - 1, axis: axis }, event);
        }
      }
      break;
    case this.keyCodes.RIGHT_KEY:
      if (axis === 'row' || axis === 'rowEnd') {
        newElement = this.m_headerLabels[axis][level + 1];
        if (newElement) {
          this._setActive(newElement, { type: 'label', level: level + 1, axis: axis }, event);
        } else if (level === levelCount - 1) {
          newIndex = axis === 'row' ? this.m_startColHeader : this.m_endColHeader;
          newElement = this._getHeaderByIndex(newIndex, this.m_columnHeaderLevelCount,
            this.m_colHeader, this.m_columnHeaderLevelCount,
            this.m_startColHeader);

          if (newElement) {
            if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
              // unhighlight and clear selection
              this._clearSelection(event);
              this.m_selectionFrontier = {};
            }
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: level,
              axis: 'column'
            }, event);
          }
        }
      }
      if (axis === 'column' || axis === 'columnEnd') {
        newElement = this._getHeaderByIndex(start, level, root, levelCount, start);
        if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
          // unhighlight and clear selection
          this._clearSelection(event);
          this.m_selectionFrontier = {};
        }
        this._setActive(newElement, {
          type: 'header',
          index: start,
          level: level,
          axis: axis
        }, event);
      }
      break;

    case this.keyCodes.LEFT_KEY:
      if (axis === 'row' || axis === 'rowEnd') {
        newElement = this.m_headerLabels[axis][level - 1];
        if (newElement) {
          this._setActive(newElement, { type: 'label', level: level - 1, axis: axis }, event);
        }
      }
      break;
    default:
      break;
  }
  return true;
};

/**
 * Handles arrow keys navigation on header
 * @param {number} keyCode description
 * @param {Event} event the DOM event that caused the arrow key
 * @param {boolean} isExtend boolean if we are extending a selection
 * @param {boolean} jumpToHeaders jump to opposite headers if possible
 * @return  boolean true if the event was processed
 */
DvtDataGrid.prototype.handleHeaderFocusChange = function (keyCode, event, isExtend, jumpToHeaders) {
  var newCellIndex;
  var newElement;
  var newIndex;
  var newLevel;
  var root;
  var start;
  var end;
  var levelCount;
  var stopFetch;

  // ensure that there's no outstanding fetch requests
  if (!this.isFetchComplete()) {
    // act like it's processed until we finish the fetch
    return true;
  }

  if (this.getResources().isRTLMode()) {
    if (keyCode === this.keyCodes.LEFT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.RIGHT_KEY;
    } else if (keyCode === this.keyCodes.RIGHT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.LEFT_KEY;
    }
  }

  var axis = this.m_active.axis;
  var index = this.m_active.index;
  var level = this.m_active.level;
  var elem = this._getActiveElement();
  var depth = elem != null ? this._getAttribute(elem, 'depth', true) : 1;

  // if cell mode, need to set values properly to check corners
  if (!axis && !index && this.m_active) {
    axis = this.m_selectionFrontier.axis;
    index = this.m_active.indexes[axis];
  }

  // if shiftkey active and we're navigating headers, as long as we don't leave headers, set isExtend to active.
  if (event.shiftKey && !isExtend && this.m_selectionFrontier && this.isMultipleSelection()) {
    // eslint-disable-next-line no-param-reassign
    isExtend = this.checkHeaderToDatabody(axis, keyCode);
  }

  // if extending and anchors are the same, use the selectionFrontier
  if (isExtend && this.isArrowKey(keyCode) &&
      this.isHeaderSelectionType(this.m_selectionFrontier) &&
      this.checkCorners(axis, index, this.m_selection[this.m_selection.length - 1])) {
    axis = this.m_selectionFrontier.axis;
    index = this.m_selectionFrontier.index;
    level = this.m_selectionFrontier.level;

    if (index === -1) {
      index = this.m_active.indexes[axis];
    }

    // don't allow changing levels if anchor is a cell.
    if (this.m_active.type === 'cell') {
      if (((keyCode === this.keyCodes.LEFT_KEY ||
            keyCode === this.keyCodes.RIGHT_KEY) &&
           axis.indexOf('row') !== -1) ||
          ((keyCode === this.keyCodes.UP_KEY ||
            keyCode === this.keyCodes.DOWN_KEY) &&
           axis.indexOf('column') !== -1)) {
        return undefined;
      }
    }
  }

  if (axis === 'column') {
    root = this.m_colHeader;
    start = this.m_startColHeader;
    end = this.m_endColHeader;
    levelCount = this.m_columnHeaderLevelCount;
    stopFetch = this.m_stopColumnHeaderFetch;
  } else if (axis === 'row') {
    root = this.m_rowHeader;
    start = this.m_startRowHeader;
    end = this.m_endRowHeader;
    levelCount = this.m_rowHeaderLevelCount;
    stopFetch = this.m_stopRowHeaderFetch;
  }
  if (axis === 'columnEnd') {
    // treat up and down keys opposite of column Headers
    if (keyCode === this.keyCodes.DOWN_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.UP_KEY;
    } else if (keyCode === this.keyCodes.UP_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.DOWN_KEY;
    }
    root = this.m_colEndHeader;
    start = this.m_startColEndHeader;
    end = this.m_endColEndHeader;
    levelCount = this.m_columnEndHeaderLevelCount;
    stopFetch = this.m_stopColumnEndHeaderFetch;
  }
  if (axis === 'rowEnd') {
    // treat right and left oppostie of row headers
    if (keyCode === this.keyCodes.LEFT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.RIGHT_KEY;
    } else if (keyCode === this.keyCodes.RIGHT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.LEFT_KEY;
    }
    root = this.m_rowEndHeader;
    start = this.m_startRowEndHeader;
    end = this.m_endRowEndHeader;
    levelCount = this.m_rowEndHeaderLevelCount;
    stopFetch = this.m_stopRowEndHeaderFetch;
  }

  // get the selection frontier current header element
  if (isExtend && this.isArrowKey(keyCode) &&
      this.isHeaderSelectionType(this.m_selectionFrontier)) {
    elem = this._getHeaderByIndex(index, level, root, levelCount, start);
    depth = elem != null ? this._getAttribute(elem, 'depth', true) : 1;
  }

  switch (keyCode) {
    case this.keyCodes.LEFT_KEY:
      if ((axis === 'column' || axis === 'columnEnd')) {
        if (index > 0) {
          if (jumpToHeaders && this.m_headerLabels[axis][level]) {
            this._setActive(this.m_headerLabels[axis][level], {
              type: 'label',
              level: level,
              axis: axis
            }, event);
            break;
          }
          if (axis === 'column' && jumpToHeaders &&
              this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1]) {
            this._setActive(this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1], {
              type: 'label',
              level: this.m_rowHeaderLevelCount - 1,
              axis: 'row'
            }, event);
            break;
          }
          if (levelCount === 1) {
            newIndex = index - 1;
            newElement = elem != null ? elem.previousSibling : null;
            newLevel = level;
          } else {
            newElement = this._getHeaderByIndex(index - 1, level, root, levelCount, start);
            newIndex = newElement != null ?
              this._getAttribute(newElement.parentNode, 'start', true) : index - 1;
            newLevel = newElement != null ? this.getHeaderCellLevel(newElement) : level;
            if (newIndex < 0) {
              break;
            }
          }
          if (isExtend) {
            this.extendSelectionHeader(newElement, event, true);
          } else {
            if (this._isSelectionEnabled() && !this.m_discontiguousSelection &&
                this.m_options.getSelectionMode() !== 'row') {
              // unhighlight and clear selection
              this._clearSelection(event);
              this.m_selectionFrontier = {};
            }
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: newLevel,
              axis: axis
            }, event);
            this._highlightActive();
          }
        } else if (this.m_headerLabels[axis][level]) {
          this._setActive(this.m_headerLabels[axis][level], {
            type: 'label',
            level: level,
            axis: axis
          }, event);
        } else if (axis === 'column' &&
                   this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1]) {
          this._setActive(this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1], {
            type: 'label',
            level: this.m_rowHeaderLevelCount - 1,
            axis: 'row'
          }, event);
        }
      } else if ((axis === 'row' || axis === 'rowEnd') && level > 0) {
        // moving down a level in the header
        newElement = this._getHeaderByIndex(index, level - 1, root, levelCount, start);
        newIndex = this._getAttribute(newElement.parentNode, 'start', true);
        newLevel = this.getHeaderCellLevel(newElement);

        if (isExtend) {
          this.extendSelectionHeader(newElement, event, true);
        } else {
          if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
            // unhighlight and clear selection
            this._clearSelection(event);
            this.m_selectionFrontier = {};
          }
          this._setActive(newElement, {
            type: 'header',
            index: newIndex,
            level: newLevel,
            axis: axis
          }, event);
          this._highlightActive();
        }
      }
      break;
    case this.keyCodes.RIGHT_KEY:
      if (axis === 'rowEnd' && jumpToHeaders && this.m_endRowHeader !== -1) {
        newElement = this._getHeaderByIndex(index, this.m_rowHeaderLevelCount,
          this.m_rowHeader, this.m_rowHeaderLevelCount,
          this.m_startRowHeader);
        if (isExtend) {
          this.extendSelectionHeader(newElement, event, true);
        } else {
          if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
            // unhighlight and clear selection
            this._clearSelection(event);
            this.m_selectionFrontier = {};
          }
          this._setActive(newElement, {
            type: 'header',
            index: index,
            level: this.m_rowHeaderLevelCount,
            axis: axis
          }, event);
          this._highlightActive();
        }
      } else if (axis === 'row' && jumpToHeaders && this.m_endRowEndHeader !== -1) {
        newElement = this._getHeaderByIndex(index, this.m_rowEndHeaderLevelCount,
          this.m_rowEndHeader, this.m_rowEndHeaderLevelCount,
          this.m_startRowEndHeader);
        if (isExtend) {
          this.extendSelectionHeader(newElement, event, true);
        } else {
          if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
            // unhighlight and clear selection
            this._clearSelection(event);
            this.m_selectionFrontier = {};
          }
          this._setActive(newElement, {
            type: 'header',
            index: index,
            level: this.m_rowEndHeaderLevelCount,
            axis: axis
          }, event);
          this._highlightActive();
        }
      } else if (axis === 'row' || axis === 'rowEnd') {
        if (level + depth >= levelCount && !isExtend) {
          // row header, move to databody
          // make the first cell of the current row active
          // no need to scroll since it will be in the viewport
          if (axis === 'row') {
            newCellIndex = this.createIndex(index, 0);
          } else if (this._isHighWatermarkScrolling()) {
            newCellIndex = this.createIndex(index, this.m_endCol);
          } else {
            newCellIndex = this.createIndex(index, this.getDataSource().getCount('column') - 1);
          }

          this.m_trueIndex = { row: index };

          if (this._isSelectionEnabled()) {
            this.selectAndFocus(newCellIndex, event);
          } else {
            this._setActiveByIndex(newCellIndex, event);
            this._highlightActive();
          }
        } else {
          // moving down a level in the header
          newElement = this._getHeaderByIndex(index, level + depth, root, levelCount, start);
          newIndex = this._getAttribute(newElement.parentNode, 'start', true);
          newLevel = this.getHeaderCellLevel(newElement);
          if (isExtend) {
            this.extendSelectionHeader(newElement, event, true);
          } else {
            if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
              // unhighlight and clear selection
              this._clearSelection(event);
              this.m_selectionFrontier = {};
            }
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: newLevel,
              axis: axis
            }, event);
            this._highlightActive();
          }
        }
      } else if (axis === 'column' && jumpToHeaders &&
                 this.m_headerLabels.rowEnd[this.m_rowEndHeaderLevelCount - 1]) {
        this._setActive(this.m_headerLabels.rowEnd[this.m_rowEndHeaderLevelCount - 1], {
          type: 'label',
          level: this.m_rowEndHeaderLevelCount - 1,
          axis: 'rowEnd'
        }, event);
      } else {
        if (levelCount === 1) {
          newIndex = index + 1;
          newElement = elem != null ? elem.nextSibling : null;
          newLevel = level;
        } else {
          if (level === levelCount - 1) {
            newIndex = index + 1;
            newElement = this._getHeaderByIndex(newIndex, level, root, levelCount, start);
          } else {
            newIndex = elem != null ?
              (this._getAttribute(elem.parentNode, 'start', true) +
               this._getAttribute(elem.parentNode, 'extent', true)) :
              index + 1;
            newElement = this._getHeaderByIndex(newIndex, level, root, levelCount, start);
          }
          newLevel = newElement != null ? this.getHeaderCellLevel(newElement) : level;
        }

        if (!(newIndex > end && stopFetch) &&
            (this._isCountUnknown('column') ||
             newIndex < this.getDataSource().getCount('column'))) {
          if (isExtend) {
            this.extendSelectionHeader(newElement, event, true);
          } else {
            if (this._isSelectionEnabled() && !this.m_discontiguousSelection &&
                this.m_options.getSelectionMode() !== 'row') {
              // unhighlight and clear selection
              this._clearSelection(event);
              this.m_selectionFrontier = {};
            }
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: newLevel,
              axis: axis
            }, event);
            this._highlightActive();
          }
        } else if (axis === 'column' &&
                   this.m_headerLabels.rowEnd[this.m_rowEndHeaderLevelCount - 1]) {
          this._setActive(this.m_headerLabels.rowEnd[this.m_rowEndHeaderLevelCount - 1], {
            type: 'label',
            level: this.m_rowEndHeaderLevelCount - 1,
            axis: 'rowEnd'
          }, event);
        }
      }
      break;
    case this.keyCodes.UP_KEY:
      if (axis === 'row' || axis === 'rowEnd') {
        if (jumpToHeaders && this.m_headerLabels[axis][level]) {
          this._setActive(this.m_headerLabels[axis][level], {
            type: 'label',
            level: level,
            axis: axis
          }, event);
          break;
        }
        if (axis === 'row' && jumpToHeaders &&
            this.m_headerLabels.column[this.m_columnHeaderLevelCount - 1]) {
          this._setActive(this.m_headerLabels.column[this.m_columnHeaderLevelCount - 1], {
            type: 'label',
            level: this.m_columnHeaderLevelCount - 1,
            axis: 'column'
          }, event);
          break;
        }
        if (index > 0) {
          if (levelCount === 1) {
            newIndex = index - 1;
            newElement = elem != null ? elem.previousSibling : null;
            newLevel = level;
          } else {
            if (level === levelCount - 1) {
              newIndex = index - 1;
              newElement = this._getHeaderByIndex(newIndex, level, root, levelCount, start);
            } else {
              newElement = elem != null ?
                this._getHeaderByIndex(this._getAttribute(elem.parentNode, 'start', true) - 1,
                  level, root, levelCount, start) : null;
              newIndex = newElement != null ?
                this._getAttribute(newElement.parentNode, 'start', true) : index - 1;
            }
            newLevel = newElement != null ? this.getHeaderCellLevel(newElement) : level;
            if (newIndex < 0) {
              break;
            }
          }
          if (isExtend) {
            this.extendSelectionHeader(newElement, event, true);
          } else {
            if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
              // unhighlight and clear selection
              this.m_selectionFrontier = {};
              this._clearSelection(event);
            }
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: newLevel,
              axis: axis
            }, event);
            this._highlightActive();
          }
        } else if (this.m_headerLabels[axis][level]) {
          this._setActive(this.m_headerLabels[axis][level], {
            type: 'label',
            level: level,
            axis: axis
          }, event);
        } else if (axis === 'row' &&
                   this.m_headerLabels.column[this.m_columnHeaderLevelCount - 1]) {
          this._setActive(this.m_headerLabels.column[this.m_columnHeaderLevelCount - 1], {
            type: 'label',
            level: this.m_columnHeaderLevelCount - 1,
            axis: 'column'
          }, event);
        }
      } else if ((axis === 'column' || axis === 'columnEnd') && level > 0) {
        // moving down a level in the header
        newElement = this._getHeaderByIndex(index, level - 1, root, levelCount, start);
        newIndex = this._getAttribute(newElement.parentNode, 'start', true);
        newLevel = this.getHeaderCellLevel(newElement);
        if (isExtend) {
          this.extendSelectionHeader(newElement, event, true);
        } else {
          if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
            // unhighlight and clear selection
            this.m_selectionFrontier = {};
            this._clearSelection(event);
          }
          this._setActive(newElement, {
            type: 'header',
            index: newIndex,
            level: newLevel,
            axis: axis
          }, event);
          this._highlightActive();
        }
      }
      break;
    case this.keyCodes.DOWN_KEY:
      if (axis === 'columnEnd' && jumpToHeaders && this.m_endColHeader !== -1) {
        newElement = this._getHeaderByIndex(index, this.m_columnHeaderLevelCount,
          this.m_colHeader, this.m_columnHeaderLevelCount,
          this.m_startColHeader);
        if (isExtend) {
          this.extendSelectionHeader(newElement, event, true);
        } else {
          if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
            // unhighlight and clear selection
            this.m_selectionFrontier = {};
            this._clearSelection(event);
          }
          this._setActive(newElement, {
            type: 'header',
            index: index,
            level: this.m_columnHeaderLevelCount,
            axis: axis
          }, event);
          this._highlightActive();
        }
      } else if (axis === 'column' && jumpToHeaders && this.m_endColEndHeader !== -1) {
        newElement = this._getHeaderByIndex(index, this.m_columnEndHeaderLevelCount,
          this.m_colEndHeader,
          this.m_columnEndHeaderLevelCount,
          this.m_startColEndHeader);
        if (isExtend) {
          this.extendSelectionHeader(newElement, event, true);
        } else {
          if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
            // unhighlight and clear selection
            this._clearSelection(event);
            this.m_selectionFrontier = {};
          }
          this._setActive(newElement, {
            type: 'header',
            index: index,
            level: this.m_columnEndHeaderLevelCount,
            axis: axis
          }, event);
          this._highlightActive();
        }
      } else if (axis === 'column' || axis === 'columnEnd') {
        if (level + depth >= levelCount && !isExtend) {
          // column header, move to databody
          // make the cell of the first row and current column active
          // no need to scroll since it will be in the viewport
          if (axis === 'column') {
            newCellIndex = this.createIndex(0, index);
          } else if (this._isHighWatermarkScrolling()) {
            newCellIndex = this.createIndex(this.m_endRow, index);
          } else {
            newCellIndex = this.createIndex(this.getDataSource().getCount('row') - 1, index);
          }

          this.m_trueIndex = { column: index };

          if (this._isSelectionEnabled()) {
            this.selectAndFocus(newCellIndex, event);
          } else {
            this._setActiveByIndex(newCellIndex, event);
            this._highlightActive();
          }
        } else {
          // moving down a level in the header
          newElement = this._getHeaderByIndex(index, level + depth, root, levelCount, start);
          newIndex = this._getAttribute(newElement.parentNode, 'start', true);
          newLevel = this.getHeaderCellLevel(newElement);
          if (isExtend) {
            this.extendSelectionHeader(newElement, event, true);
          } else {
            if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
              // unhighlight and clear selection
              this._clearSelection(event);
              this.m_selectionFrontier = {};
            }
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: newLevel,
              axis: axis
            }, event);
            this._highlightActive();
          }
        }
      } else if (axis === 'row' && jumpToHeaders &&
                 this.m_headerLabels.columnEnd[this.m_columnEndHeaderLevelCount - 1]) {
        this._setActive(this.m_headerLabels.columnEnd[this.m_columnEndHeaderLevelCount - 1], {
          type: 'label',
          level: this.m_columnEndHeaderLevelCount - 1,
          axis: 'columnEnd'
        }, event);
      } else {
        if (levelCount === 1) {
          newIndex = index + 1;
          newElement = elem != null ? elem.nextSibling : null;
          newLevel = level;
        } else {
          if (level === levelCount - 1) {
            newIndex = index + 1;
            newElement = this._getHeaderByIndex(newIndex, level, root, levelCount, start);
          } else {
            newIndex = elem != null ?
              (this._getAttribute(elem.parentNode, 'start', true) +
               this._getAttribute(elem.parentNode, 'extent', true)) :
              index + 1;
            newElement = this._getHeaderByIndex(newIndex, level, root, levelCount, start);
          }
          newLevel = newElement != null ? this.getHeaderCellLevel(newElement) : level;
        }

        if (!(newIndex > end && stopFetch) &&
            (this._isCountUnknown('row') || newIndex < this.getDataSource().getCount('row'))) {
          if (isExtend) {
            this.extendSelectionHeader(newElement, event, true);
          } else {
            if (this._isSelectionEnabled() && !this.m_discontiguousSelection) {
              // unhighlight and clear selection
              this._clearSelection(event);
              this.m_selectionFrontier = {};
            }
            this._setActive(newElement, {
              type: 'header',
              index: newIndex,
              level: newLevel,
              axis: axis
            }, event);
            this._highlightActive();
          }
        } else if (axis === 'row' &&
                   this.m_headerLabels.columnEnd[this.m_columnEndHeaderLevelCount - 1]) {
          this._setActive(this.m_headerLabels.columnEnd[this.m_rowEndHeaderLevelCount - 1], {
            type: 'label',
            level: this.m_columnEndHeaderLevelCount - 1,
            axis: 'columnEnd'
          }, event);
        }
      }
      break;
    case this.keyCodes.PAGEUP_KEY:
      if (axis === 'row' || axis === 'rowEnd') {
        // selects the first available row header
        elem = this._getHeaderByIndex(0, level, root, levelCount, start);
        this._setActive(elem, { type: 'header', index: 0, level: level, axis: axis }, event);
      }
      break;
    case this.keyCodes.PAGEDOWN_KEY:
      if (axis === 'row' || axis === 'rowEnd') {
        // selects the last available row header
        if (!this._isCountUnknown('row') && !this._isHighWatermarkScrolling()) {
          index = Math.max(0, this.getDataSource().getCount('row') - 1);
        } else {
          index = Math.max(0, end);
        }
        elem = this._getHeaderByIndex(index, level, root, levelCount, start);
        this._setActive(elem, { type: 'header', index: index, level: level, axis: axis }, event);
      }
      break;
    case this.keyCodes.HOME_KEY:
      if (axis === 'column' || axis === 'columnEnd') {
        // selects the first cell of the current row
        elem = this._getHeaderByIndex(0, level, root, levelCount, start);
        this._setActive(elem, { type: 'header', index: 0, level: level, axis: axis }, event);
      }
      break;
    case this.keyCodes.END_KEY:
      if (axis === 'column' || axis === 'columnEnd') {
        // selects the last cell of the current row
        if (!this._isCountUnknown('column') && !this._isHighWatermarkScrolling()) {
          index = Math.max(0, this.getDataSource().getCount('column') - 1);
        } else {
          index = Math.max(0, end);
        }
        // selects the first cell of the current row
        elem = this._getHeaderByIndex(index, level, root, levelCount, start);
        this._setActive(elem, { type: 'header', index: index, level: level, axis: axis }, event);
      }
      break;
    default:
      break;
  }
  return true;
};

/**
 * Check if the focus is changing from header to databody
 * Get the label of the header
 * @param {string} axis
 * @param {number} keyCode
 * @returns {boolean} True if the header is not leaving the databody
 */
DvtDataGrid.prototype.checkHeaderToDatabody = function (axis, keyCode) {
  if (!((axis === 'row' &&
         this.m_rowHeaderLevelCount === this.m_selectionFrontier.level &&
         keyCode === this.keyCodes.RIGHT_KEY) ||
        (axis === 'rowEnd' &&
         this.m_rowEndHeaderLevelCount === this.m_selectionFrontier.level &&
         keyCode === this.keyCodes.LEFT_KEY) ||
        (axis === 'column' &&
         this.m_columnHeaderLevelCount === this.m_selectionFrontier.level &&
         keyCode === this.keyCodes.DOWN_KEY) ||
        (axis === 'columnEnd' &&
         this.m_columnEndHeaderLevelCount === this.m_selectionFrontier.level &&
         keyCode === this.keyCodes.UP_KEY))) {
    return true;
  }

  return false;
};

/**
 * Get the label of the header
 * @param {string} axis
 * @param {Element} root
 * @param {number} levelCount
 * @param {number} start
 * @param {number} end
 * @param {number} currentIndex
 * @param {number} previousIndex
 * @param {Element} element
 * @returns {string}
 */
DvtDataGrid.prototype._getHeaderLabelledBy =
  function (axis, root, levelCount, start, end, currentIndex, previousIndex, element) {
    var previousElement;
    if (end !== -1 && (currentIndex !== previousIndex || this.m_externalFocus)) {
      var columnEndHeader = this.getHeaderFromCell(element, axis);
      if (previousIndex != null) {
        previousElement = this._getHeaderByIndex(previousIndex, levelCount - 1,
          root, levelCount, start);
      }
      return this._getHeaderAndParentIds(columnEndHeader, previousElement);
    }
    return '';
  };

/**
 * Get the Id's in a string to put in the accessibility labelledby
 * @param {Element=} header
 * @param {Element=} previousHeader
 * @returns {string}
 */
DvtDataGrid.prototype._getHeaderAndParentIds = function (header, previousHeader) {
  var idString = '';
  var previousParents = [];

  if (header == null) {
    // header not rendered
    return '';
  }

  var parents = this._getHeaderAndParents(header);
  if (previousHeader != null) {
    previousParents = this._getHeaderAndParents(previousHeader);
  }
  for (var i = 0; i < parents.length; i++) {
    // always add the header that we are focusing
    if (previousParents[i] !== parents[i] || i === parents.length - 1) {
      idString += (idString === '' ? '' : ' ') + parents[i].id;
    }
  }
  return idString;
};

/**
 * Get the nested headers above the header and including the header.
 * Puts them in an array starting with the outermost.
 * @param {Element} header
 * @returns {Array}
 */
DvtDataGrid.prototype._getHeaderAndParents = function (header) {
  var headers = [header];
  var axis = this.getHeaderCellAxis(header);
  var level = this.getHeaderCellLevel(header);
  var headerLabel = this._getLabel(axis, level);
  var headerLevels;

  if (axis === 'row') {
    headerLevels = this.m_rowHeaderLevelCount;
  } else if (axis === 'column') {
    headerLevels = this.m_columnHeaderLevelCount;
  } else if (axis === 'rowEnd') {
    headerLevels = this.m_rowEndHeaderLevelCount;
  } else if (axis === 'columnEnd') {
    headerLevels = this.m_columnEndHeaderLevelCount;
  }

  if (headerLabel) {
    headers.unshift(headerLabel);
  }

  if (headerLevels === 1) {
    return headers;
  } else if (level === headerLevels - 1) {
    // eslint-disable-next-line no-param-reassign
    header = header.parentNode.firstChild;
    headers.unshift(header);
    level -= 1;
    headerLabel = this._getLabel(axis, level);
    if (headerLabel) {
      headers.unshift(headerLabel);
    }
  }

  while (level > 0) {
    // eslint-disable-next-line no-param-reassign
    header = header.parentNode.parentNode.firstChild;
    headers.unshift(header);
    level -= 1;
    headerLabel = this._getLabel(axis, level);
    if (headerLabel) {
      headers.unshift(headerLabel);
    }
  }
  return headers;
};

/**
 * Checks if the input selection Frontier is type header
 * @param {Object} selectionFrontier
 * @returns {boolean} true if type header, false otherwise
 */
DvtDataGrid.prototype.isHeaderSelectionType = function (selectionFrontier) {
  if (selectionFrontier && selectionFrontier.axis) {
    return true;
  }

  return false;
};

/**
 * Handles arrow keys navigation on cell
 * @param {number} keyCode description
 * @param {boolean} isExtend
 * @param {Event} event the DOM event causing the arrow keys
 * @param {boolean} changeRegions
 * @param {boolean} jumpToHeaders jump to headers if possible
 */
DvtDataGrid.prototype.handleFocusChange = function (
  keyCode, isExtend, event, changeRegions, jumpToHeaders
) {
  var currentCellIndex;
  var newCellIndex;
  var header;
  var rowExtent = 1;
  var columnExtent = 1;
  // ensure that there's no outstanding fetch requests
  if (!this.isFetchComplete()) {
    // act as if processed to prevent page scrolling before fetch done
    return true;
  }

  if (isExtend) {
    currentCellIndex = this.m_selectionFrontier;
    // if extending and selection frontier has an axis component, we are in header realm
    if (this.isHeaderSelectionType(this.m_selectionFrontier)) {
      this.handleHeaderFocusChange(keyCode, event, isExtend, jumpToHeaders);
      return undefined;
    }
  } else {
    currentCellIndex = this.m_active.indexes;
  }

  if (currentCellIndex == null) {
    return undefined;
  }

  if (this.m_trueIndex == null) {
    this.m_trueIndex = {};
  }

  if (this.getResources().isRTLMode()) {
    if (keyCode === this.keyCodes.LEFT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.RIGHT_KEY;
    } else if (keyCode === this.keyCodes.RIGHT_KEY) {
      // eslint-disable-next-line no-param-reassign
      keyCode = this.keyCodes.LEFT_KEY;
    }
  }

  // invoke different function for handling focusing on active cell depending on whether selection is enabled
  var focusFunc = this._isSelectionEnabled() ?
    this.selectAndFocus.bind(this) : this._setActiveByIndex.bind(this);
  var row = currentCellIndex.row;
  var column = currentCellIndex.column;
  var currentCell = this._getCellByIndex(currentCellIndex);
  if (currentCell) {
    var cellContext = currentCell[this.getResources().getMappedAttribute('context')];
    rowExtent = cellContext.extents.row;
    columnExtent = cellContext.extents.column;
  }

  // navigation to cell using arrow keys.  We are using index instead of dom element
  // because the dom element might not be there in all cases
  switch (keyCode) {
    case this.keyCodes.LEFT_KEY:
      if (!this.m_trueIndex.row && !isExtend) {
        this.m_trueIndex = { row: row };
      }
      if (column > 0 && !(jumpToHeaders && this.m_endRowHeader !== -1)) {
        // for left and right key in row selection mode, we'll be only shifting active cell and
        // selection will not be affected
        if (this.m_options.getSelectionMode() === 'row') {
          // ensure active cell index is used for row since it might use frontier if extended
          newCellIndex = this.createIndex(this.m_trueIndex.row, column - 1);
          this._setActiveByIndex(newCellIndex, event);
        } else {
          if (isExtend) {
            newCellIndex = this.createIndex(row, column - 1);
            this.extendSelection(newCellIndex, event, keyCode);
          } else {
            newCellIndex = this.createIndex(this.m_trueIndex.row, column - 1);
            focusFunc(newCellIndex, event);
          }

          // announce to screen reader that we have reached first column
          if (column - 1 === 0) {
            this._setAccInfoText('accessibleFirstColumn');
          }
        }
      } else if (!isExtend && changeRegions) {
        // reached the first column, go to row header if available
        header = this._getHeaderByIndex(this.m_trueIndex.row, this.m_rowHeaderLevelCount - 1,
          this.m_rowHeader, this.m_rowHeaderLevelCount,
          this.m_startRowHeader);
        if (this.m_discontiguousSelection) {
          this.discontiguousHeaderSetActiveFromDatabody(event, 'row', header,
            this.m_rowHeaderLevelCount);
        } else {
          this._setActive(header, {
            type: 'header',
            index: this.m_trueIndex.row,
            level: this.m_rowHeaderLevelCount - 1,
            axis: 'row'
          }, event, true);
        }
      }
      break;
    case this.keyCodes.RIGHT_KEY:
      if (!this.m_trueIndex.row && !isExtend) {
        this.m_trueIndex = { row: row };
      }
      // if condition for unknown count and known count cases on whether we have reached the end
      if (!this._isLastColumn(column + (columnExtent - 1)) &&
          !(jumpToHeaders && this.m_endRowEndHeader !== -1)) {
        // for left and right key in row selection mode, we'll be only shifting active cell and
        // selection will not be affected
        if (this.m_options.getSelectionMode() === 'row') {
          // ensure active cell index is used for row since it might use frontier if extended
          newCellIndex = this.createIndex(this.m_trueIndex.row, column + columnExtent);
          this._setActiveByIndex(newCellIndex, event);
        } else {
          if (isExtend) {
            newCellIndex = this.createIndex(row, column + 1);
            this.extendSelection(newCellIndex, event, keyCode);
          } else {
            newCellIndex = this.createIndex(this.m_trueIndex.row, column + columnExtent);
            focusFunc(newCellIndex, event);
          }

          // announce to screen reader that we have reached last column
          if (this._isLastColumn(column + columnExtent)) {
            this._setAccInfoText('accessibleLastColumn');
          }
        }
      } else if (this.m_endRowEndHeader !== -1 && changeRegions) {
        // reached the last column, go to row end header if available
        header = this._getHeaderByIndex(this.m_trueIndex.row, this.m_rowEndHeaderLevelCount - 1,
          this.m_rowEndHeader, this.m_rowEndHeaderLevelCount,
          this.m_startRowEndHeader);
        if (this.m_discontiguousSelection) {
          this.discontiguousHeaderSetActiveFromDatabody(event, 'rowEnd', header,
            this.m_rowEndHeaderLevelCount);
        } else {
          this._setActive(header, {
            type: 'header',
            index: this.m_trueIndex.row,
            level: this.m_rowEndHeaderLevelCount - 1,
            axis: 'rowEnd'
          }, event, true);
        }
      } else if (!isExtend) {
        // if anchor cell is in the last column, and they arrow right (without Shift), then collapse the range to just the focus cell.  (Matches Excel and intuition.)
        focusFunc(currentCellIndex, event);
      }
      break;
    case this.keyCodes.UP_KEY:
      if (!this.m_trueIndex.column && !isExtend) {
        this.m_trueIndex = { column: column };
      }
      if (row > 0 && !(jumpToHeaders && this.m_endColHeader !== -1)) {
        if (isExtend) {
          newCellIndex = this.createIndex(row - 1, column);
          this.extendSelection(newCellIndex, event, keyCode);
        } else {
          newCellIndex = this.createIndex(row - 1, this.m_trueIndex.column);
          focusFunc(newCellIndex, event);
        }

        // announce to screen reader that we have reached first row
        if (row - 1 === 0) {
          this._setAccInfoText('accessibleFirstRow');
        }
      } else if (!isExtend && changeRegions) {
        // if in multiple selection don't clear the selection
        // reached the first row, go to column header if available
        header = this._getHeaderByIndex(this.m_trueIndex.column,
          this.m_columnHeaderLevelCount - 1,
          this.m_colHeader, this.m_columnHeaderLevelCount,
          this.m_startColHeader);
        if (this.m_discontiguousSelection) {
          this.discontiguousHeaderSetActiveFromDatabody(event, 'column', header,
            this.m_columnHeaderLevelCount);
        } else {
          this._setActive(header, {
            type: 'header',
            index: this.m_trueIndex.column,
            level: this.m_columnHeaderLevelCount - 1,
            axis: 'column'
          }, event, true);
        }
      }
      break;
    case this.keyCodes.DOWN_KEY:
      if (!this.m_trueIndex.column && !isExtend) {
        this.m_trueIndex = { column: column };
      }
      if (!this._isLastRow(row + (rowExtent - 1)) &&
          !(jumpToHeaders && this.m_endColEndHeader !== -1)) {
        if (isExtend) {
          newCellIndex = this.createIndex(row + 1, column);
          this.extendSelection(newCellIndex, event, keyCode);
        } else {
          newCellIndex = this.createIndex(row + rowExtent, this.m_trueIndex.column);
          focusFunc(newCellIndex, event);
        }

        // announce to screen reader that we have reached last row
        if (this._isLastRow(row + rowExtent)) {
          this._setAccInfoText('accessibleLastRow');
        }
      } else if (this.m_endColEndHeader !== -1 && changeRegions) {
        // reached the last column, go to column end header if available
        header = this._getHeaderByIndex(this.m_trueIndex.column,
          this.m_columnEndHeaderLevelCount - 1,
          this.m_colEndHeader, this.m_columnEndHeaderLevelCount,
          this.m_startColEndHeader);
        if (this.m_discontiguousSelection) {
          this.discontiguousHeaderSetActiveFromDatabody(event, 'columnEnd', header,
            this.m_columnEndHeaderLevelCount);
        } else {
          this._setActive(header, {
            type: 'header',
            index: this.m_trueIndex.column,
            level: this.m_columnEndHeaderLevelCount - 1,
            axis: 'columnEnd'
          }, event, true);
        }
      } else if (!isExtend) {
        // if anchor cell is in the last row, and they arrow down (without Shift), then collapse the range to just the focus cell.  (Matches Excel and intuition.)
        focusFunc(currentCellIndex, event);
      }
      break;
    case this.keyCodes.HOME_KEY:
      if (!this.m_trueIndex.row) {
        this.m_trueIndex = { row: row };
      }
      // selects the first cell of the current row
      newCellIndex = this.createIndex(this.m_trueIndex.row, 0);
      focusFunc(newCellIndex, event);
      break;
    case this.keyCodes.END_KEY:
      if (!this.m_trueIndex.row) {
        this.m_trueIndex = { row: row };
      }
      // selects the last cell of the current row
      if (!this._isCountUnknown('column') && !this._isHighWatermarkScrolling()) {
        newCellIndex = this.createIndex(this.m_trueIndex.row,
          Math.max(0, this.getDataSource().getCount('column') - 1));
      } else {
        newCellIndex = this.createIndex(this.m_trueIndex.row, Math.max(0, this.m_endCol));
      }
      focusFunc(newCellIndex, event);
      break;
    case this.keyCodes.PAGEUP_KEY:
      if (!this.m_trueIndex.column) {
        this.m_trueIndex = { column: column };
      }
      // selects the first cell of the current column
      newCellIndex = this.createIndex(0, column);
      focusFunc(newCellIndex, event);
      break;
    case this.keyCodes.PAGEDOWN_KEY:
      if (!this.m_trueIndex.column) {
        this.m_trueIndex = { column: column };
      }
      // selects the last cell of the current column
      if (!this._isCountUnknown('column') && !this._isHighWatermarkScrolling()) {
        newCellIndex = this.createIndex(Math.max(0, this.getDataSource().getCount('row') - 1),
          this.m_trueIndex.column);
      } else {
        newCellIndex = this.createIndex(Math.max(0, this.m_endRow), this.m_trueIndex.column);
      }
      focusFunc(newCellIndex, event);
      break;
    default:
      break;
  }

  return true;
};

/**
 * Scrolls to an  index
 * @param {Object} index - the end index of the selection.
 * @param {boolean|null=} ignoreHighlight - true if we want to ignore highlighting a cell
 * @param {boolean|null=} scrollToOrigin - true if we align the viewport with the origin
 */
DvtDataGrid.prototype.scrollToIndex = function (index, ignoreHighlight, scrollToOrigin) {
  var scrollRows;
  var row = index.row;
  var column = index.column;
  var cell;

  if (ignoreHighlight) {
    this.m_shouldFocus = false;
  }
  if (scrollToOrigin) {
    // eslint-disable-next-line no-param-reassign
    index.scrollToOrigin = true;
  }

  var dir = this.getResources().isRTLMode() ? 'right' : 'left';

  var deltaX = 0;
  var deltaY = 0;
  var viewportTop = this._getViewportTop();
  var viewportBottom = this._getViewportBottom();
  var viewportLeft = this._getViewportLeft();
  var viewportRight = this._getViewportRight();

  // check if index is completely outside of rendered
  if (row < this.m_startRow || row > this.m_endRow) {
    var scrollTop;
    if (row < this.m_startRow) {
      scrollTop = this.m_avgRowHeight * row;
    } else {
      scrollTop = ((this.m_avgRowHeight * (row + 1)) - viewportBottom) + viewportTop;
    }
    deltaY = this.m_currentScrollTop - scrollTop;

    // remember to focus on the row after fetch
    this.m_scrollIndexAfterFetch = index;
    scrollRows = true;
  } else {
    // it's rendered, find location and scroll to it
    cell = this._getCellByIndex(index);
    var rowHeight;
    if (cell === null) {
      // can't guarantee the actual cell is there just one with the same row index
      // we know we can't get the extent of the cell either since it is not there
      // so we know to scroll to the top + height of the key (not height of cell
      // as it can span multiple rows)
      cell = this._getFirstCellWithMatchingStartIndex(row, 'row');
      rowHeight = this.m_sizingManager.getSize('row', this._getKey(cell, 'row'));
    } else {
      rowHeight = this.getElementHeight(cell);
    }
    var rowTop = this.getElementDir(cell, 'top');


    // If we are scrolling to a row position, align it to the top row of the viewport
    // if specified
    if (scrollToOrigin || index.scrollToOrigin) {
      deltaY = viewportTop - rowTop;
    } else if (rowTop + rowHeight > viewportBottom) {
      deltaY = viewportBottom - (rowTop + rowHeight);
    } else if (rowTop < viewportTop) {
      deltaY = viewportTop - rowTop;
    }
  }

  // if column is defined and it's not already a fetch outside of rendered
  // use scrollRows to know it was not pre-defined
  // if initial Scroll, we should adjust the column
  if (!isNaN(column) && scrollRows !== true) {
    // check if index is completely outside of rendered
    // approximate scroll position
    if (column < this.m_startCol || column > this.m_endCol) {
      var scrollLeft;
      if (column < this.m_startCol) {
        scrollLeft = this.m_avgColWidth * column;
      } else {
        scrollLeft = ((this.m_avgColWidth * (column + 1)) - viewportRight) + viewportLeft;
      }
      deltaX = this.m_currentScrollLeft - scrollLeft;

      // remember to focus on the cell after fetch
      this.m_scrollIndexAfterFetch = index;
    } else {
      // it's rendered, find location and scroll to it
      cell = this._getCellByIndex(index);
      var cellWidth;
      if (cell === null) {
        // see comment for row heights above
        cell = this._getFirstCellWithMatchingStartIndex(column, 'column');
        cellWidth = this.m_sizingManager.getSize('column', this._getKey(cell, 'column'));
      } else {
        cellWidth = this.getElementWidth(cell);
      }
      var cellLeft = this.getElementDir(cell, dir);

      if (scrollToOrigin || index.scrollToOrigin) {
        deltaX = viewportLeft - cellLeft;
      } else if (cellLeft < viewportLeft) {
        deltaX = viewportLeft - cellLeft;
      } else if (cellLeft + cellWidth > viewportRight) {
        deltaX = viewportRight - (cellLeft + cellWidth);
      }
    }
  }

  // scroll if either horiz or vert scroll pos has changed
  if (deltaX !== 0 || deltaY !== 0) {
    cell = this._getCellByIndex(index);

    // this.m_shouldFocus for second call after initial scroll.
    if (cell != null && ignoreHighlight !== true && this.m_shouldFocus !== false) {
      // delay focus on cell until databody has scrolled (by the scroll event handler)
      // if we are not highlighting, ignore this
      this.m_cellToFocus = cell;
    }
    this.scrollDelta(deltaX, deltaY);
  } else if (this.m_scrollIndexAfterFetch != null) {
    // if there's an index we wanted to scroll to after fetch it has now been scrolled to by scrollToIndex, so highlight it
    // this.m_shouldFocus for second call after initial scroll.
    if (!ignoreHighlight && this.m_shouldFocus !== false) {
      if (this._setActiveByIndex(this.m_scrollIndexAfterFetch, null, false, false, true)) {
        this.m_scrollIndexAfterFetch = null;
      }
    } else {
      this.m_scrollIndexAfterFetch = null;
    }
  }
};

/**
 * Scrolls to an  index
 * @param {Object} headerInfo
 * @param {string} headerInfo.axis
 * @param {number} headerInfo.index
 * @param {number} headerInfo.level
 */
DvtDataGrid.prototype.scrollToHeader = function (headerInfo) {
  var startIndex;
  var endIndex;
  var averageDiff;
  var currentScroll;
  var newScroll;
  var headerMin;
  var headerDiff;
  var header;
  var viewportMin;
  var viewportMax;
  var axis = headerInfo.axis;
  var index = headerInfo.index;
  var level = headerInfo.level;
  var delta = 0;

  if (axis === 'row') {
    startIndex = this.m_startRowHeader;
    endIndex = this.m_endRowHeader;
    averageDiff = this.m_avgRowHeight;
    currentScroll = this.m_currentScrollTop;
    viewportMin = this._getViewportTop();
    viewportMax = this._getViewportBottom();
  } else if (axis === 'column') {
    startIndex = this.m_startColHeader;
    endIndex = this.m_endColHeader;
    averageDiff = this.m_avgColWidth;
    currentScroll = this.m_currentScrollLeft;
    viewportMin = this._getViewportLeft();
    viewportMax = this._getViewportRight();
  } else if (axis === 'rowEnd') {
    startIndex = this.m_startRowEndHeader;
    endIndex = this.m_endRowEndHeader;
    averageDiff = this.m_avgRowHeight;
    currentScroll = this.m_currentScrollTop;
    viewportMin = this._getViewportTop();
    viewportMax = this._getViewportBottom();
  } else if (axis === 'columnEnd') {
    startIndex = this.m_startColEndHeader;
    endIndex = this.m_endColEndHeader;
    averageDiff = this.m_avgColWidth;
    currentScroll = this.m_currentScrollLeft;
    viewportMin = this._getViewportLeft();
    viewportMax = this._getViewportRight();
  }

  var viewportDiff = viewportMax - viewportMin;

  // check if index is completely outside of rendered
  if (index < startIndex || index > endIndex) {
    if (index < startIndex) {
      newScroll = averageDiff * index;
    } else {
      newScroll = (averageDiff * (index + 1)) - viewportDiff;
    }
    delta = currentScroll - newScroll;

    // remember to focus on the row after fetch
    this.m_scrollHeaderAfterFetch = headerInfo;
  } else {
    if (axis === 'row' || axis === 'rowEnd') {
      if (axis === 'row') {
        header = this._getHeaderByIndex(index, level, this.m_rowHeader,
          this.m_rowHeaderLevelCount, this.m_startRowHeader);
      } else {
        header = this._getHeaderByIndex(index, level, this.m_rowEndHeader,
          this.m_rowEndHeaderLevelCount, this.m_startRowEndHeader);
      }
      headerMin = this.getElementDir(header, 'top');
      headerDiff = this.getElementHeight(header);
    } else if (axis === 'column' || axis === 'columnEnd') {
      if (axis === 'column') {
        header = this._getHeaderByIndex(index, level, this.m_colHeader,
          this.m_columnHeaderLevelCount, this.m_startColHeader);
      } else {
        header = this._getHeaderByIndex(index, level, this.m_colEndHeader,
          this.m_columnEndHeaderLevelCount, this.m_startColEndHeader);
      }
      headerMin = this.getElementDir(header, this.getResources().isRTLMode() ? 'right' : 'left');
      headerDiff = this.getElementWidth(header);
    }

    if (viewportDiff > headerDiff) {
      if (headerMin + headerDiff > viewportMax) {
        delta = viewportMax - (headerMin + headerDiff);
      } else if (headerMin < viewportMin) {
        delta = viewportMin - headerMin;
      }
    } else {
      delta = viewportMin - headerMin;
    }
  }

  // scroll if either horiz or vert scroll pos has changed
  if (delta !== 0) {
    if (header != null && this.m_shouldFocus !== false) {
      // delay focus on cell until databody has scrolled (by the scroll event handler)
      this.m_cellToFocus = header;
    }
    if (axis === 'row' || axis === 'rowEnd') {
      this.scrollDelta(0, delta);
    } else {
      this.scrollDelta(delta, 0);
    }
  } else if (this.m_scrollHeaderAfterFetch != null) {
    // if there's an index we wanted to sctoll to after fetch it has now been scrolled to by scrollToIndex, so highlight it
    this._updateActive(headerInfo, true, true);
    this.m_scrollHeaderAfterFetch = null;
  }
};

/**
 * Locate the header element.  Look up recursively from its parent if neccessary.
 * @param {Element|undefined|null} elem the starting point to locate the header element
 * @param {string=} headerCellClassName the name of the header cell class name
 * @param {string=} endHeaderCellClassName the name of the header cell class name
 * @return {Element|null|undefined} the header element
 * @private
 */
DvtDataGrid.prototype.findHeader = function (elem, headerCellClassName, endHeaderCellClassName) {
  if (headerCellClassName == null) {
    // eslint-disable-next-line no-param-reassign
    headerCellClassName = this.getMappedStyle('headercell');
  }

  if (endHeaderCellClassName == null) {
    // eslint-disable-next-line no-param-reassign
    endHeaderCellClassName = this.getMappedStyle('endheadercell');
  }

  if (headerCellClassName != null) {
    if (this.m_utils.containsCSSClassName(elem, headerCellClassName) ||
        this.m_utils.containsCSSClassName(elem, endHeaderCellClassName)) {
      // found header element
      return elem;
    } else if (elem.parentNode) {
      // recursive call with parent node
      return this.findHeader(elem.parentNode, headerCellClassName, endHeaderCellClassName);
    } else if (elem === this.m_root) {
      // short circuit to terminal when root is reached
      return null;
    }
  }

  // all other case returns null
  return null;
};

/**
 * Ensures row banding is set on the proper rows
 * @private
 */
DvtDataGrid.prototype.updateRowBanding = function () {
  var rowBandingInterval = this.m_options.getRowBandingInterval();
  if (rowBandingInterval > 0) {
    var cells = this.m_databody.firstChild.childNodes;
    var bandingClass = this.getMappedStyle('banded');
    for (var i = 0; i < cells.length; i++) {
      var cell = cells[i];
      var index = this._getIndex(cell, 'row');
      if ((Math.floor(index / rowBandingInterval) % 2 === 1)) {
        if (!this.m_utils.containsCSSClassName(cell, bandingClass)) {
          this.m_utils.addCSSClassName(cell, bandingClass);
        }
      } else if (this.m_utils.containsCSSClassName(cell, bandingClass)) {
        this.m_utils.removeCSSClassName(cell, bandingClass);
      }
    }
  }
};

/**
 * Ensures column banding is set on the proper rows
 * @private
 */
DvtDataGrid.prototype.updateColumnBanding = function () {
  var columnBandingInterval = this.m_options.getColumnBandingInterval();
  if (columnBandingInterval > 0) {
    var cells = this.m_databody.firstChild.childNodes;
    var bandingClass = this.getMappedStyle('banded');
    for (var i = 0; i < cells.length; i += 1) {
      var cell = cells[i];
      var index = this._getIndex(cell, 'column');
      if ((Math.floor(index / columnBandingInterval) % 2 === 1)) {
        if (!this.m_utils.containsCSSClassName(cell, bandingClass)) {
          this.m_utils.addCSSClassName(cell, bandingClass);
        }
      } else if (this.m_utils.containsCSSClassName(cell, bandingClass)) {
        this.m_utils.removeCSSClassName(cell, bandingClass);
      }
    }
  }
};

/**
 * Remove banding (both row and column)
 * @private
 */
DvtDataGrid.prototype._removeBanding = function () {
  var cells = this.m_databody.firstChild.childNodes;
  var bandingClass = this.getMappedStyle('banded');

  for (var i = 0; i < cells.length; i++) {
    if (this.m_utils.containsCSSClassName(cells[i], bandingClass)) {
      this.m_utils.removeCSSClassName(cells[i], bandingClass);
    }
  }
};

/**
 * Sets the accessibility status text
 * @param {string} key the message key
 * @param {Object|Array|null=} args to pass into the translator
 * @private
 */
DvtDataGrid.prototype._setAccInfoText = function (key, args) {
  var text = this.getResources().getTranslatedText(key, args);
  if (text != null) {
    this.m_accInfo.textContent = text;
  }
};

/**
 * Handles expand event from the flattened datasource.
 * @param {Object} event the expand event
 * @param {boolean} fromQueue whether this is invoked from processing the model event queue, optional.
 * @private
 */
DvtDataGrid.prototype.handleExpandEvent = function (event, fromQueue) {
  if (fromQueue === undefined && this.queueModelEvent(event)) {
    // tag the event for discovery later
    // eslint-disable-next-line no-param-reassign
    event.operation = 'expand';
    return;
  }

  // rowKey = event['rowKey'];
  // rowCells = this._getAxisCellsByKey(rowKey, 'row');
  // for (i = 0; i < rowCells.length; i++)
  // {
  //    rowCells[i].setAttribute("aria-expanded", true);
  // }

  // update screen reader alert
  this._setAccInfoText('accessibleRowExpanded');
  this.populateAccInfo();
  if (fromQueue) {
    this._runModelEventQueue();
  }
};

/**
 * Handles collapse event from the flattened datasource.
 * @param {Object} event the collapse event
 * @param {boolean} fromQueue whether this is invoked from processing the model event queue, optional.
 * @private
 */
DvtDataGrid.prototype.handleCollapseEvent = function (event, fromQueue) {
  if (fromQueue === undefined && this.queueModelEvent(event)) {
    // tag the event for discovery later
    // eslint-disable-next-line no-param-reassign
    event.operation = 'collapse';
    return;
  }

  // rowKey = event['rowKey'];
  // rowCells = this._getAxisCellsByKey(rowKey, 'row');
  // for (i = 0; i < rowCells.length; i++)
  // {
  //    rowCells[i].setAttribute("aria-expanded", false);
  // }

  // update screen reader alert
  this._setAccInfoText('accessibleRowCollapsed');
  this.populateAccInfo();
  if (fromQueue) {
    this._runModelEventQueue();
  }
};

/**
 * Retrieve the key from an element.
 * @param {Element|Node|undefined} element the element to retrieve the key from.
 * @param {string=} axis
 * @return {string|null} the key of the element
 * @private
 */
DvtDataGrid.prototype._getKey = function (element, axis) {
  // make sure the element has a context
  if (element != null && element[this.getResources().getMappedAttribute('context')]) {
    if (axis != null && this.m_utils.containsCSSClassName(element, this.getMappedStyle('cell'))) {
      return element[this.getResources().getMappedAttribute('context')].keys[axis];
    }
    return element[this.getResources().getMappedAttribute('context')].key;
  }
  return null;
};

/**
 * Retrieve the active axis key.
 * @param {string} axis
 * @param {boolean=} prev if we want the previous row key instead
 * @return {string|null} the key of the active row
 * @private
 */
DvtDataGrid.prototype._getActiveKey = function (axis, prev) {
  if (prev && this.m_prevActive != null) {
    if (this.m_prevActive.type === 'header' &&
        (this.m_prevActive.axis === axis || this.m_prevActive.axis === axis + 'End')) {
      return this.m_prevActive.key;
    } else if (this.m_prevActive.type === 'cell') {
      return this.m_prevActive.keys[axis];
    }
  } else if (this.m_active != null) {
    if (this.m_active.type === 'header' &&
        (this.m_active.axis === axis || this.m_active.axis === axis + 'End')) {
      return this.m_active.key;
    } else if (this.m_active.type === 'cell') {
      return this.m_active.keys[axis];
    }
  }
  return null;
};

// /////////////////// move methods////////////////////////
/**
 * Handles cut event from the flattened datasource.
 * @param {Event} event the cut event
 * @param {Element=} target the target element
 * @return {boolean} true if the event was processed here
 * @private
 */
DvtDataGrid.prototype._handleCut = function (event, target) {
  if (target == null) {
    // eslint-disable-next-line no-param-reassign
    target = /** @type {Element} */ (event.target);
  }
  var cell = this.findCellOrHeader(target);

  if (this._isMoveOnElementEnabled(cell)) {
    if (this.m_cutCells != null) {
      for (var i = 0; i < this.m_cutCells.length; i++) {
        this.m_utils.removeCSSClassName(this.m_cutCells[i], this.getMappedStyle('cut'));
      }
    }

    var rowKey = this._getKey(cell, 'row');
    // cut row header with row
    this.m_cutCells = this._getAxisCellsByKey(rowKey, 'row');
    this.m_cutRowHeader = this._findHeaderByKey(rowKey, this.m_rowHeader,
      this.getMappedStyle('rowheadercell'));
    this.m_cutRowEndHeader = this._findHeaderByKey(rowKey, this.m_rowEndHeader,
      this.getMappedStyle('rowendheadercell'));

    this._highlightCellsAlongAxis(rowKey, 'row', 'key', 'add', ['cut']);
    if (this.m_cutRowHeader !== null) {
      this.m_utils.addCSSClassName(this.m_cutRowHeader, this.getMappedStyle('cut'));
    }
    if (this.m_cutRowEndHeader !== null) {
      this.m_utils.addCSSClassName(this.m_cutRowEndHeader, this.getMappedStyle('cut'));
    }

    return true;
  }
  return false;
};


/**
 * Handles cut cells event.
 * @param {Event} event the cut event
 * @param {Element=} target the target element
 * @return {boolean} true if the event was processed here
 * @private
 */
DvtDataGrid.prototype._handleCutCells = function (event, target) {
  if (target == null) {
    // eslint-disable-next-line no-param-reassign
    target = /** @type {Element} */ (event.target);
  }

  if (this._isDataGridProvider() && this._isSelectionEnabled() &&
      this.m_options.isCutEnabled()) {
    // if previously cut/copy without pasting, unhighlight that range.
    if (this.m_selectionRange && this.m_selectionRange.length) {
      this.unhighlightFloodFillRange(this.m_selectionRange[0]);
    }
    let selection = this.m_selection[this.m_selection.length - 1];
    this.m_selectionRange = [selection];
    this.m_dataTransferAction = 'cut';

    var details = {
      event: event,
      ui: {
        action: this.m_dataTransferAction,
        sourceRange: this.m_selectionRange[0]
      }
    };

    let cutRequestEvent = this.fireEvent('cutRequest', details);
    if (!cutRequestEvent) {
      return true;
    }
    this.highlightFloodFillRange(selection);
    if (this.m_options.isFloodFillEnabled()) {
      this._removeFloodFillAffordance();
    }
  }
  return true;
};

/**
 * Handles copy cells event.
 * @param {Event} event the copy event
 * @param {Element=} target the target element
 * @return {boolean} true if the event was processed here
 * @private
 */
 DvtDataGrid.prototype._handleCopyCells = function (event, target) {
  if (target == null) {
    // eslint-disable-next-line no-param-reassign
    target = /** @type {Element} */ (event.target);
  }

  if (this._isDataGridProvider() && this._isSelectionEnabled() &&
      this.m_options.isCopyEnabled()) {
    // if previously cut/copy without pasting, unhighlight that range.
    if (this.m_selectionRange && this.m_selectionRange.length) {
      this.unhighlightFloodFillRange(this.m_selectionRange[0]);
    }
    let selection = this.m_selection[this.m_selection.length - 1];
    this.m_selectionRange = [selection];
    this.m_dataTransferAction = 'copy';
    let details = {
      event: event,
      ui: {
        action: this.m_dataTransferAction,
        sourceRange: this.m_selectionRange[0]
      }
    };

    let copyRequestEvent = this.fireEvent('copyRequest', details);
    if (!copyRequestEvent) {
      return true;
    }
    this.highlightFloodFillRange(selection);
    if (this.m_options.isFloodFillEnabled()) {
      this._removeFloodFillAffordance();
    }
    return true;
  }
  return false;
};

/**
 * Handles paste event.
 * @param {Event} event the paste event
 * @param {Element=} target the target element
 *
 * @private
 */
DvtDataGrid.prototype._handlePaste = function (event, target) {
  if (target == null) {
    // eslint-disable-next-line no-param-reassign
    target = /** @type {Element} */ (event.target);
  }
  if (this.m_cutCells != null) {
    for (var i = 0; i < this.m_cutCells.length; i++) {
      this.m_utils.removeCSSClassName(this.m_cutCells[i], this.getMappedStyle('cut'));
    }

    if (this.m_cutRowHeader !== null) {
      // remove css from row header too
      this.m_utils.removeCSSClassName(this.m_cutRowHeader, this.getMappedStyle('cut'));
      this.m_cutRowHeader = null;
    }
    if (this.m_cutRowEndHeader !== null) {
      // remove css from row header too
      this.m_utils.removeCSSClassName(this.m_cutRowEndHeader, this.getMappedStyle('cut'));
      this.m_cutRowEndHeader = null;
    }

    var pasteRowKey = this._getKey(this.findCellOrHeader(target), 'row');
    var cutRowKey = this._getKey(this.m_cutCells[0], 'row');
    if (cutRowKey !== pasteRowKey) {
      if (this._isSelectionEnabled()) {
        // unhighlight and clear selection
        this._clearSelection(event);
      }
      if (this._isDatabodyCellActive()) {
        this._unhighlightActive();
      }
      this.m_moveActive = true;
      this.getDataSource().move(cutRowKey, pasteRowKey);
    }
    this.m_cutCells = null;
  }
  return true;
};

/**
 * Handles paste cells event.
 * @param {Event} event the paste event
 * @param {Element=} target the target element
 *
 * @private
 */
 DvtDataGrid.prototype._handlePasteCells = function (event, target) {
  if (target == null) {
    // eslint-disable-next-line no-param-reassign
    target = /** @type {Element} */ (event.target);
  }
  if (this._isDataGridProvider() && this.m_options.isPasteEnabled()) {
    if (this.m_selectionRange && this._isSelectionEnabled()
      && !this.m_discontiguousSelection &&
      this.m_selection.length === 1) {
      let details = {
        event: event,
        ui: {
          action: this.m_dataTransferAction,
          sourceRange: this.m_selectionRange[0],
          targetRange: this.m_selection[0]
        }
      };

      let pasteRequestEvent = this.fireEvent('pasteRequest', details);
      if (!pasteRequestEvent) {
        return true;
      }
      this.unhighlightFloodFillRange(this.m_selectionRange[0]);
      this.m_selectionRange = null;
      this.m_dataTransferAction = null;
    } else if (!this.m_selectionRange) {
      let details = {
        event: event,
        ui: {
          action: 'unknown',
          sourceRange: {},
          targetRange: this.m_selection[0]
        }
      };
      let pasteRequestEvent = this.fireEvent('pasteRequest', details);
      if (!pasteRequestEvent) {
        return true;
      }
      this.m_selectionRange = null;
      this.m_dataTransferAction = null;
    }
  }
  return true;
};

/**
 * triggers autofill event.
 * @param {Event} event the fill event
 * @param {Element=} target the target element
 *
 * @private
 */
DvtDataGrid.prototype._handleAutofill = function (event, target) {
  if (target == null) {
    // eslint-disable-next-line no-param-reassign
    target = /** @type {Element} */ (event.target);
  }

  if (this._isDataGridProvider() &&
      this._isSelectionEnabled() && !this.m_discontiguousSelection &&
      this.m_options.isFloodFillEnabled() && this.m_selection.length === 1) {
    let fillDirection = 'down';
    if (event.type === 'keydown') {
      fillDirection = event.key === 'd' ? 'down' : 'end';
    }
    let selectionStart = this.m_selection[0].startIndex;
    let selectionEnd = this.m_selection[0].endIndex;
    let sourceRange = this.createRange(selectionStart, selectionEnd);
    let targetRange = this.createRange(selectionStart, selectionEnd);
    let validKeyDown = false;
    if (fillDirection === 'down') {
      sourceRange.endIndex.row = sourceRange.startIndex.row;
      targetRange.startIndex.row = sourceRange.startIndex.row + 1;
      if ((targetRange.startIndex.row >= sourceRange.startIndex.row) &&
          (targetRange.startIndex.row <= targetRange.endIndex.row)) {
            validKeyDown = true;
          }
    } else {
      sourceRange.endIndex.column = sourceRange.startIndex.column;
      targetRange.startIndex.column = sourceRange.startIndex.column + 1;
      if ((targetRange.startIndex.column >= sourceRange.startIndex.column) &&
      (targetRange.startIndex.column <= targetRange.endIndex.column)) {
        validKeyDown = true;
      }
    }

    if (validKeyDown) {
      var details = {
        event: event,
        ui: {
          action: fillDirection,
          sourceRange: sourceRange,
          targetRange: targetRange
        }
      };

      let fillRequestEvent = this.fireEvent('fillRequest', details);
      if (!fillRequestEvent) {
        return true;
      }
    }
    this.unhighlightFloodFillRange(this.m_selection[0]);
    this._removeFloodFillAffordance();
    this.m_selectionRange = null;
    this.m_dataTransferAction = null;
    this.m_floodFillDirection = null;
    return true;
  }
  return false;
};

/**
 * Handles canceling a reorder
 * @param {Object} event the cut event
 * @param {Element=} target the target element
 *
 * @private
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleCancelReorder = function (event, target) {
  if (this.m_cutCells != null) {
    for (var i = 0; i < this.m_cutCells.length; i++) {
      this.m_utils.removeCSSClassName(this.m_cutCells[i], this.getMappedStyle('cut'));
    }
    this.m_cutCells = null;

    if (this.m_cutRowHeader !== null) {
      this.m_utils.removeCSSClassName(this.m_cutRowHeader, this.getMappedStyle('cut'));
      this.m_cutRowHeader = null;
    }
    if (this.m_cutRowEndHeader !== null) {
      this.m_utils.removeCSSClassName(this.m_cutRowEndHeader, this.getMappedStyle('cut'));
      this.m_cutRowEndHeader = null;
    }
    return true;
  } else if (this.m_dataTransferAction !== null) {
    this.unhighlightFloodFillRange(this.m_selectionRange[0]);
    this.m_selectionRange = null;
    this.m_dataTransferAction = null;
  }
  return undefined;
};

/**
 * Handles cut event from the flattened datasource.
 * @param {Object} event the cut event
 * @private
 */
DvtDataGrid.prototype._handleMove = function (event) {
  // initialize the move
  if (this.m_moveCells == null) {
    var target = /** @type {Element} */ (event.target);
    var cell = this.findCellOrHeader(target);

    // get the move row key to set the move row/rowHeader
    var rowKey = this._getKey(cell, 'row');
    this.m_originalMoveIndex = this._getIndex(cell, 'row');
    this.m_moveIndex = /** @type {number} */ (this.m_originalMoveIndex);

    this.m_moveCells = this._getAxisCellsByIndex(this.m_moveIndex, 'row');
    this.m_moveRowHeader = this._findHeaderByKey(rowKey, this.m_rowHeader,
      this.getMappedStyle('rowheadercell'));
    this.m_moveRowEndHeader = this._findHeaderByKey(rowKey, this.m_rowEndHeader,
      this.getMappedStyle('rowendheadercell'));

    // add the move style class to the css
    this._highlightCellsAlongAxis(this.m_moveIndex, 'row', 'index', 'add', ['drag']);

    this.m_originalTop = this.getElementDir(this.m_moveCells[0], 'top');

    this.m_dropTarget = document.createElement('div');
    this.m_utils.addCSSClassName(this.m_dropTarget, this.getMappedStyle('drop'));
    this.setElementHeight(this.m_dropTarget, this.calculateRowHeight(this.m_moveCells[0]));
    this.setElementDir(this.m_dropTarget, this.m_originalTop, 'top');
    this.m_databody.firstChild.appendChild(this.m_dropTarget); // @HTMLUpdateOK

    this._addHeaderDropTarget(this.m_moveRowHeader, this.m_rowHeader, false);
    this._addHeaderDropTarget(this.m_moveRowEndHeader, this.m_rowEndHeader, true);
  }

  // calculate the change in Y direction
  if (!this.m_utils.isTouchDevice()) {
    this.m_prevY = this.m_currentY;
    this.m_currentY = event.pageY;
  }
  var deltaY = this.m_currentY - this.m_prevY;
  var height = this.calculateRowHeight(this.m_moveCells[0]);

  // adjust the top height of the moveRow and moveRowHeader
  for (var i = 0; i < this.m_moveCells.length; i++) {
    this.setElementDir(this.m_moveCells[i],
      (this.getElementDir(this.m_moveCells[i], 'top') + deltaY), 'top');
  }
  if (this.m_moveRowHeader !== null) {
    this.setElementDir(this.m_moveRowHeader,
      (this.getElementDir(this.m_moveRowHeader, 'top') + deltaY), 'top');
  }
  if (this.m_moveRowEndHeader !== null) {
    this.setElementDir(this.m_moveRowEndHeader,
      (this.getElementDir(this.m_moveRowEndHeader, 'top') + deltaY), 'top');
  }

  var nextSiblingIndex = this.m_moveIndex + 1;
  var previousSiblingIndex = this.m_moveIndex - 1;
  var nextSibling = this._getCellByIndex(this.createIndex(nextSiblingIndex, this.m_startCol));
  var previousSibling = this._getCellByIndex(this.createIndex(previousSiblingIndex,
    this.m_startCol));

  // see if the element has crossed the halfway point of the next row
  if (nextSibling != null &&
      (this.getElementDir(nextSibling, 'top') <
       (this.getElementDir(this.m_moveCells[0], 'top') + (height / 2)))) {
    this._moveDropRows('nextSibling', nextSiblingIndex);
  } else if (previousSibling != null &&
             (this.getElementDir(previousSibling, 'top') >
              (this.getElementDir(this.m_moveCells[0], 'top') - (height / 2)))) {
    this._moveDropRows('previousSibling', previousSiblingIndex);
  }
};

/**
 * Add drop header target
 * @param {Element|null|undefined} moveHeader
 * @param {boolean} isEnd
 * @private
 */
DvtDataGrid.prototype._addHeaderDropTarget = function (moveHeader, root, isEnd) {
  var dropTarget;
  if (moveHeader !== null) {
    // need to store the height inline if not already because top values will be changing
    if (moveHeader.style.height == null) {
      this.setElementHeight(moveHeader, this.calculateRowHeight(moveHeader));
    }
    this.m_utils.addCSSClassName(moveHeader, this.getMappedStyle('drag'));
    dropTarget = document.createElement('div');
    this.m_utils.addCSSClassName(dropTarget, this.getMappedStyle('drop'));
    this.setElementHeight(dropTarget, this.calculateRowHeight(moveHeader));
    this.setElementDir(dropTarget, this.m_originalTop, 'top');
    root.firstChild.appendChild(dropTarget); // @HTMLUpdateOK

    if (isEnd) {
      this.m_dropTargetEndHeader = dropTarget;
    } else {
      this.m_dropTargetHeader = dropTarget;
    }
  }
};

/**
 * Determined if move is supported for the specified axis.
 * @param {string} sibling nextSibling/previosusSibling
 * @private
 */
DvtDataGrid.prototype._moveDropRows = function (sibling, index) {
  var newTop;
  var newSiblingTop;
  var headerScroller;
  var endHeaderScroller;
  var siblingCells;

  // move the drop target and the adjacent row
  if (sibling === 'nextSibling') {
    siblingCells = this._getAxisCellsByIndex(index, 'row');
    newTop = this.m_originalTop + this.calculateRowHeight(siblingCells[0]);
    newSiblingTop = this.m_originalTop;
  } else {
    siblingCells = this._getAxisCellsByIndex(index, 'row');
    newTop = this.getElementDir(siblingCells[0], 'top');
    newSiblingTop = newTop + this.calculateRowHeight(siblingCells[0]);
  }

  this.setElementDir(this.m_dropTarget, newTop, 'top');

  for (var i = 0; i < siblingCells.length; i++) {
    this.setElementDir(siblingCells[i], newSiblingTop, 'top');
  }

  if (this.m_moveRowHeader !== null) {
    headerScroller = this.m_moveRowHeader.parentNode;
    this.setElementDir(this.m_dropTargetHeader, newTop, 'top');
    this.setElementDir(this.m_moveRowHeader[sibling], newSiblingTop, 'top');
  }
  if (this.m_moveRowEndHeader !== null) {
    endHeaderScroller = this.m_moveRowEndHeader.parentNode;
    this.setElementDir(this.m_dropTargetEndHeader, newTop, 'top');
    this.setElementDir(this.m_moveRowEndHeader[sibling], newSiblingTop, 'top');
  }

  // store the new top value
  this.m_originalTop = newTop;

  this._highlightCellsAlongAxis(this.m_moveIndex + 1, 'row', 'index', 'remove', ['activedrop']);

  // move the moveRow and rowHeader so we can continue to pull the adjacent header
  if (sibling === 'nextSibling') {
    this._modifyAxisCellContextIndex('row', this.m_moveIndex, 1, 1);
    this._modifyAxisCellContextIndex('row', this.m_moveIndex + 1, 1, -1);
    this.m_moveIndex += 1;

    if (this.m_moveRowHeader !== null && headerScroller) {
      headerScroller.insertBefore(this.m_moveRowHeader, // @HTMLUpdateOK
        this.m_moveRowHeader[sibling][sibling]);
    }
    if (this.m_moveRowEndHeader !== null && endHeaderScroller) {
      endHeaderScroller.insertBefore(this.m_moveRowEndHeader, // @HTMLUpdateOK
        this.m_moveRowEndHeader[sibling][sibling]);
    }
  } else {
    this._modifyAxisCellContextIndex('row', this.m_moveIndex, 1, -1);
    this._modifyAxisCellContextIndex('row', this.m_moveIndex - 1, 1, 1);
    this.m_moveIndex -= 1;

    if (this.m_moveRowHeader !== null && headerScroller) {
      headerScroller.insertBefore(this.m_moveRowHeader, // @HTMLUpdateOK
        this.m_moveRowHeader[sibling]);
    }
    if (this.m_moveRowEndHeader !== null && endHeaderScroller) {
      endHeaderScroller.insertBefore(this.m_moveRowEndHeader, // @HTMLUpdateOK
        this.m_moveRowEndHeader[sibling]);
    }
  }

  this._refreshDatabodyMap();

  this._highlightCellsAlongAxis(this.m_moveIndex + 1, 'row', 'index', 'add', ['activedrop']);
};

/**
 * Determined if move is supported for the specified axis.
 * @param {string} axis the axis which we check whether move is supported.
 * @private
 */
DvtDataGrid.prototype._isMoveEnabled = function (axis) {
  var capability = this.getDataSource().getCapability('move');
  var moveable = this.m_options.isMoveable('row');
  if (moveable === 'enable' && (capability === 'full' || capability === axis)) {
    return true;
  }

  return false;
};

/**
 * Handles a mouse up after move
 * @param {Event} event MouseUp Event
 * @param {boolean} validUp true if in the databody or rowHeader
 * @private
 */
DvtDataGrid.prototype._handleMoveMouseUp = function (event, validUp) {
  if (this.m_moveCells != null) {
    // remove the the drop target div from the databody/rowHeader
    this._remove(this.m_dropTarget);
    if (this.m_moveRowHeader !== null) {
      this._remove(this.m_dropTargetHeader);
    }
    if (this.m_moveRowEndHeader !== null) {
      this._remove(this.m_dropTargetEndHeader);
    }
    if (this.m_active != null && this.m_active.axis !== 'column') {
      this.m_moveActive = true;
    }

    // clear selection
    if (this._isSelectionEnabled()) {
      // unhighlight and clear selection
      this._clearSelection(event);
    }

    var moveCell = this.m_moveCells[0];
    var moveCellKey = this._getKey(moveCell, 'row');

    // if the mousup was in the rowHeader or databody
    if (validUp === true) {
      var insertIndex = this.m_moveIndex + 1;
      var insertKey = this._getKey(this._getCellByIndex(this.createIndex(insertIndex,
        this.m_startCol)), 'row');
      this.getDataSource().move(moveCellKey, insertKey);
    } else {
      this.getDataSource().move(moveCellKey, moveCellKey);
    }
    this.m_moveCells = null;
    this.m_originalMoveIndex = null;
    this.m_moveIndex = null;
  }
  this.m_databodyMove = false;
};

DvtDataGrid.prototype._handleFloodFillMouseUp = function (event) {
  if (this.m_floodFillRange && this.m_floodFillRange.length) {
    var details = {
      event: event,
      ui: {
        action: this.m_floodFillDirection,
        sourceRange: this.m_selectionRange[0],
        targetRange: this.m_floodFillRange[0],
      }
    };

    let fillRequestEvent = this.fireEvent('fillRequest', details);
    if (!fillRequestEvent) {
      return true;
    }
    this.unhighlightFloodFillRange();
  }
  this.m_selectionRange = null;
  this.m_floodFillRange = null;
  this.m_floodFillDirection = null;
  if (this.m_databody) {
    this.m_databody.style.cursor = 'default';
  }
  this.m_cursor = 'default';
  return true;
};

/**
 * Check if a row can be moved, meaning it is the active row and move is enabled
 * @param {Element|null|undefined} cell the row to move
 * @returns {boolean} true if the row can be moved
 */
DvtDataGrid.prototype._isMoveOnElementEnabled = function (cell) {
  if (cell != null && this._isMoveEnabled('row')) {
    if (this._getActiveKey('row') === this._getKey(cell, 'row')) {
      return true;
    }
  }
  return false;
};

/**
 * Applies the draggable class to the new active row and row header, removes it if the active has changed
 */
DvtDataGrid.prototype._manageMoveCursor = function () {
  var activeKey = this._getActiveKey('row');
  var prevActiveKey = this._getActiveKey('row', true);

  var className = this.getMappedStyle('draggable');
  var rowHeaderStyle = this.getMappedStyle('rowheadercell');
  var rowEndHeaderStyle = this.getMappedStyle('rowendheadercell');

  if (prevActiveKey != null) {
    this._highlightCellsAlongAxis(prevActiveKey, 'row', 'key', 'remove', ['draggable']);

    var prevActiveRowHeader = this._findHeaderByKey(prevActiveKey
      , this.m_rowHeader, rowHeaderStyle);
    if (this.m_utils.containsCSSClassName(prevActiveRowHeader, className)) {
      this.m_utils.removeCSSClassName(prevActiveRowHeader, className);
    }

    prevActiveRowHeader = this._findHeaderByKey(prevActiveKey,
      this.m_rowEndHeader, rowEndHeaderStyle);
    if (this.m_utils.containsCSSClassName(prevActiveRowHeader, className)) {
      this.m_utils.removeCSSClassName(prevActiveRowHeader, className);
    }
  }

  if (activeKey != null) {
    var activeCells = this._getAxisCellsByKey(activeKey, 'row');
    // if move enabled and draggable class name
    if (this._isMoveOnElementEnabled(activeCells[0])) {
      this._highlightCellsAlongAxis(activeKey, 'row', 'key', 'add', ['draggable']);

      var activeRowHeader = this._findHeaderByKey(activeKey, this.m_rowHeader, rowHeaderStyle);
      this.m_utils.addCSSClassName(activeRowHeader, className);

      var activeRowEndHeader = this._findHeaderByKey(activeKey,
        this.m_rowEndHeader, rowEndHeaderStyle);
      this.m_utils.addCSSClassName(activeRowEndHeader, className);
    }
  }
};

/**
 * Handles focus on the root and its children by setting focus class on the root
 * @param {Event} event
 */
 DvtDataGrid.prototype.handleRootFocus = function (event, isPopupFocusin) {
  this.m_utils.addCSSClassName(this.m_root, this.getMappedStyle('focus'));
  this._clearFocusoutTimeout();
  this._clearFocusoutBusyState();
  // if nothing is active, and came from the outside of the datagrid, activate first cell
  if (!isPopupFocusin) {
    this._clearOpenPopupListeners();
    if (!this.m_root.contains(document.activeElement) ||
        (document.activeElement === this.m_root &&
        this.m_root.tabIndex === 0) ||
        (document.activeElement === this.m_databody &&
        this.m_scrollbarFocus &&
        this.m_root.tabIndex === 0)) {
      this.m_externalFocus = true;

      if (this._isCellEditable()) {
        this._setAccInfoText('accessibleEditableMode');
      } else if (this._isGridEditable()) {
        this._setAccInfoText('accessibleNavigationMode');
      }

      var shouldNotScroll = false;
      if (this.m_scrollbarFocus === true) {
        this.m_shouldFocus = false;
        this.m_scrollbarFocus = false;
        shouldNotScroll = true;
      }

      if (this.m_active == null && !this._databodyEmpty()) {
        var newCellIndex = this.createIndex(0, 0);

        if (!shouldNotScroll) {
          // make sure it's visible
          this.scrollToIndex(newCellIndex);
        }

        // focus a cell, do not select it unless user actively selects something
        this._setActiveByIndex(newCellIndex, event, null, null, shouldNotScroll);
      } else if (this.m_active != null) {
        this._highlightActive();
      }
    }
    this.m_root.tabIndex = -1;
  }
};

DvtDataGrid.prototype._handlePopupFocusout = function (event) {
  this.handleRootBlur(event, true);
};

DvtDataGrid.prototype._handlePopupFocusin = function (event) {
  this.handleRootFocus(event, true);
};

/**
 * Handles blur on the root and its children by removing focus class on the root
 * @param {Event} event
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype.handleRootBlur = function (event, isPopupFocusout) {
  // There is no cross-browser way to tell if the whole grid is out of focus on blur today.
  // document.activeElement returns null in chrome and firefox on blur events.
  // relatedTarget doesn't return a value in firefox and IE though there a tickets to fix.
  // We could implement a non-timeout solution that exiting and re-entering
  // the grid via tab key would not read the summary text upon re-entry (initial would work)
  this._clearFocusoutTimeout();
  if (!isPopupFocusout) {
    // Components that open popups (such as ojSelect, ojCombobox, ojInputDate, etc.) will trigger
    // focusout, but we don't want to change mode in those cases since the user is still editing.
    this._clearOpenPopupListeners();
    var openPopup = getLogicalChildPopup(this.m_root);
    if (openPopup != null) {
      // setup focus listeners on popup
      this._openPopup = openPopup;
      // eslint-disable-next-line no-param-reassign
      isPopupFocusout = false;
      this._handlePopupFocusinListener = this._handlePopupFocusin.bind(this);
      this._handlePopupFocusoutListener = this._handlePopupFocusout.bind(this);
      openPopup.addEventListener('focusin', this._handlePopupFocusinListener);
      openPopup.addEventListener('focusout', this._handlePopupFocusoutListener);
      return;
    }
  }
  this._setFocusoutBusyState();
  this.m_focusoutTimeout = setTimeout(function () { // @HTMLUpdateOK
    if (!this.m_root.contains(document.activeElement) ||
      isPopupFocusout === true) {
      this.m_root.tabIndex = 0;
      var active = this._getActiveElement();
      if (active != null) {
        this._unsetAriaProperties(active);
        if (this._isEditOrEnter()) {
          this._leaveEditing(event, active, false, false);
        }
      }
    }
    this._clearFocusoutBusyState();
  }.bind(this), 100);

  // don't change the color on move
  if (this.m_moveRow == null) {
    this.m_utils.removeCSSClassName(this.m_root, this.getMappedStyle('focus'));
  }
};

DvtDataGrid.prototype._clearOpenPopupListeners = function () {
  if (this._openPopup != null) {
    this._openPopup.removeEventListener('focusin', this._handlePopupFocusinListener);
    this._openPopup.removeEventListener('focusout', this._handlePopupFocusoutListener);
    this._openPopup = null;
  }
  this._handlePopupFocusinListener = null;
  this._handlePopupFocusoutListener = null;
};

/**
 * @private
 */
DvtDataGrid.prototype._handlePopupFocusout = function (event) {
  this.handleRootBlur(event, true);
};

/**
 * @private
 */
DvtDataGrid.prototype._handlePopupFocusin = function (event) {
  this.handleRootFocus(event, true);
};

DvtDataGrid.prototype._clearFocusoutTimeout = function () {
  if (this.m_focusoutTimeout) {
    clearTimeout(this.m_focusoutTimeout);
    this.m_focusoutTimeout = null;
  }
};

DvtDataGrid.prototype._setFocusoutBusyState = function () {
  if (!this.m_focusoutResolveFunc) {
    var msg = 'is handling focusout.';
    var busyContext = Context.getContext(this.m_root).getBusyContext();
    var options = {
      description: "Datagrid component '" + msg
    };
    this.m_focusoutResolveFunc = busyContext.addBusyState(options);
  }
};

DvtDataGrid.prototype._clearFocusoutBusyState = function () {
  if (this.m_focusoutResolveFunc) {
    this.m_focusoutResolveFunc();
    this.m_focusoutResolveFunc = null;
  }
};

/**
 * Calculate the a row's height using top or endRowPixel
 * @param {Element|undefined|null} row the row to calculate height on
 * @return {number} the row height
 */
DvtDataGrid.prototype.calculateRowHeight = function (row) {
  if (row.style.height !== '') {
    return this.getElementHeight(row);
  }
  if (row.nextSibling != null) {
    return this.getElementDir(row.nextSibling, 'top') - this.getElementDir(row, 'top');
  }
  return this.m_endRowPixel - this.getElementDir(row, 'top');
};

/**
 * Calculate the a row headers's height using top or endRowHeaderPixel
 * @param {Element|undefined|null} rowHeader the rowHeader to calculate height on
 * @return {number} the rowHeader height
 */
DvtDataGrid.prototype.calculateRowHeaderHeight = function (rowHeader) {
  if (rowHeader.style.height !== '') {
    return this.getElementHeight(rowHeader);
  }
  if (rowHeader.nextSibling != null) {
    return this.getElementDir(rowHeader.nextSibling, 'top') - this.getElementDir(rowHeader, 'top');
  }
  return this.m_endRowHeaderPixel - this.getElementDir(rowHeader, 'top');
};

DvtDataGrid.prototype.calculateRowHeaderLabelHeight = function (rowHeaderLabel) {
  return this.getElementHeight(rowHeaderLabel);
};

/**
 * Calculate the a column's width using left/right or endColumnPixel
 * @param {Element|undefined|null} cell the cell to calculate width on
 * @return {number} the cell width
 */
DvtDataGrid.prototype.calculateColumnWidth = function (cell) {
  if (cell.style.width !== '') {
    return this.getElementWidth(cell);
  }
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';
  if (cell.nextSibling != null) {
    return this.getElementDir(cell.nextSibling, dir) - this.getElementDir(cell, dir);
  }
  return this.m_endColPixel - this.getElementDir(cell, dir);
};

/**
 * Calculate the a column headers's width using left/right or endColumnHeaderPixel
 * @param {Element|undefined|null} columnHeader the columnHeader to calculate width on
 * @return {number} the columnHeader width
 */
DvtDataGrid.prototype.calculateColumnHeaderWidth = function (columnHeader) {
  if (columnHeader.style.width !== '') {
    return this.getElementWidth(columnHeader);
  }
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';
  if (columnHeader.nextSibling != null) {
    return this.getElementDir(columnHeader.nextSibling, dir) -
      this.getElementDir(columnHeader, dir);
  }
  return this.m_endColHeaderPixel - this.getElementDir(columnHeader, dir);
};

/**
 * @return {boolean} true if the databody is empty
 */
DvtDataGrid.prototype._databodyEmpty = function () {
  if (this.m_databody.firstChild == null || this.m_databody.firstChild.firstChild == null) {
    return true;
  }
  return false;
};

/**
 * Change or add CSS property of element
 * @param {Element} target the element to which css property will be added
 * @param {string|undefined} prop the style property name
 * @param {string|number|null} value the value of css property
 * @param {string=} action the flag variable if it is require to remove css property
 * @private
 */
DvtDataGrid.prototype.changeStyleProperty = function (target, prop, value, action) {
  if (typeof prop !== 'undefined') {
    // eslint-disable-next-line no-param-reassign
    target.style[prop] = (action === 'remove') ? '' : value;
  }
};

/**
 * Add set of required animation rules to the element
 * @param {Element} target the element to which animation rules will be added
 * @param {number|string} duration the duration of animation
 * @param {number|string} delay the delay of animation
 * @param {string} timing the easing function
 * @param {number|string} x the final position (in pixels) of the current animation
 * @param {number|string} y the final position (in pixels) of the current animation
 * @param {number|string} z the final position (in pixels) of the current animation
 * @private
 */
DvtDataGrid.prototype.addTransformMoveStyle =
  function (target, duration, delay, timing, x, y, z) {
    this.changeStyleProperty(target, this.getCssSupport('transition-delay'), delay);
    this.changeStyleProperty(target, this.getCssSupport('transition-timing-function'), timing);
    this.changeStyleProperty(target, this.getCssSupport('transition-duration'), duration);
    this.changeStyleProperty(target, this.getCssSupport('transform'),
      'translate3d(' + x + 'px,' + y + 'px,' + z + 'px)');
  };

/**
 * Add set of required animation rules to the element
 * @param {Element} target the element to which animation rules will be added
 * @private
 */
DvtDataGrid.prototype.removeTransformMoveStyle = function (target) {
  this.changeStyleProperty(target, this.getCssSupport('transition-delay'),
    null, 'remove');
  this.changeStyleProperty(target, this.getCssSupport('transition-timing-function'),
    null, 'remove');
  this.changeStyleProperty(target, this.getCssSupport('transition-duration'),
    null, 'remove');
  this.changeStyleProperty(target, this.getCssSupport('transform'),
    null, 'remove');
};

/**
 * Check if CSS property is supported by appropriate vendors
 * @param {string} cssprop css property
 * @return {string|undefined} css property with appropiate vendor's prefix
 * @private
 */
DvtDataGrid.prototype.getCssSupport = function (cssprop) {
  var vendors = ['', '-moz-', '-webkit-', '-o-', '-ms-', '-khtml-'];
  var root = document.documentElement;

  function toCamel(str) {
    return str.replace(/-([a-z])/gi, function (match, val) {
      // convert first letter after "-" to uppercase
      return val.toUpperCase();
    });
  }

  for (var i = 0; i < vendors.length; i++) {
    var css3mc = toCamel(vendors[i] + cssprop);
    // if property starts with 'Ms'
    if (css3mc.substr(0, 2) === 'Ms') {
      // Convert 'M' to lowercase
      css3mc = 'm' + css3mc.substr(1);
    }
    if (css3mc in root.style) {
      return css3mc;
    }
  }

  return undefined;
};

/**
 * Clears the databody map and repopulates it based on what's in the databody
 * @private
 */
DvtDataGrid.prototype._refreshDatabodyMap = function () {
  this._clearDatabodyMap();
  this._addNodesToDatabodyMap(this.m_databody.firstChild.childNodes);
};

/**
 * Adds a fragment to the databody content and fills the data body mapKey
 * @param {Element} databodyContent
 * @param {Element|DocumentFragment} fragment
 * @private
 */
DvtDataGrid.prototype._populateDatabody = function (databodyContent, fragment) {
  this._addNodesToDatabodyMap(fragment.childNodes);
  databodyContent.appendChild(fragment); // @HTMLUpdateOK
  this.m_subtreeAttachedCallback(databodyContent);
};

/**
 * Empties the databody and clears the databody map
 * @param {Element} databodyContent
 * @private
 */
DvtDataGrid.prototype._emptyDatabody = function (databodyContent) {
  this._clearDatabodyMap();
  this.m_utils.empty(databodyContent);
};

/**
 * Adds an array of nodes to the databody map
 * @param {NodeList|Array} nodes
 * @private
 */
DvtDataGrid.prototype._addNodesToDatabodyMap = function (nodes) {
  for (var i = 0; i < nodes.length; i++) {
    var node = nodes[i];
    if (this.m_utils.containsCSSClassName(node, this.getMappedStyle('cell'))) {
      var indexes = this.getCellIndexes(node);
      var extents = this.getCellExtents(node);
      var id = node.id;
      this._addToDatabodyMap(indexes, id, extents);
    }
  }
};

/**
 * Adds an index, id pair to the databody map along with its extents
 * @param {Object} indexes
 * @param {string} id
 * @param {Object} extents
 * @private
 */
DvtDataGrid.prototype._addToDatabodyMap = function (indexes, id, extents) {
  var rowExtent = extents.row;
  var columnExtent = extents.column;

  for (var i = 0; i < rowExtent; i++) {
    for (var j = 0; j < columnExtent; j++) {
      this._addIndexToDatabodyMap(this.createIndex(indexes.row + i, indexes.column + j), id);
    }
  }
};

/**
 * Adds an index, id pair to the databody map
 * @param {Object} indexes
 * @param {string} id
 * @private
 */
DvtDataGrid.prototype._addIndexToDatabodyMap = function (indexes, id) {
  var mapKey = 'r' + indexes.row + 'c' + indexes.column;
  this.m_databodyMap.set(mapKey, id); // quoted to make the closure compiler happy
};

/**
 * Removes an index, id pair from the databody map
 * @param {Object} indexes
 * @returns {boolean}
 * @private
 */
DvtDataGrid.prototype._removeIndexFromDatabodyMap = function (indexes) {
  var mapKey = 'r' + indexes.row + 'c' + indexes.column;
  return this.m_databodyMap.delete(mapKey); // quoted to make the closure compiler happy
};

/**
 * Gets an id from the databody based on the index
 * @param {Object} indexes
 * @return the id at the index
 * @private
 */
DvtDataGrid.prototype._getFromDatabodyMap = function (indexes) {
  var mapKey = 'r' + indexes.row + 'c' + indexes.column;
  return this.m_databodyMap.get(mapKey); // quoted to make the closure compiler happy
};

/**
 * Clears the databody map
 * @returns {boolean} the map
 * @private
 */
DvtDataGrid.prototype._clearDatabodyMap = function () {
  return this.m_databodyMap.clear(); // quoted to make the closure compiler happy
};

/**
 * Update the cellContext.indexes of a range of cells
 * @param {string} axis row/column
 * @param {number} atIndex startIndex along the axis
 * @param {number} count number of cells after the start to modify
 * @param {number} value value to increment/decremnt the index value by
 * @private
 */
DvtDataGrid.prototype._modifyAxisCellContextIndex = function (axis, atIndex, count, value) {
  for (var i = atIndex; i < atIndex + count; i++) {
    var axisCells = this._getAxisCellsByIndex(i, axis);
    for (var j = 0; j < axisCells.length; j++) {
      var cell = axisCells[j];
      var cellContext = cell[this.getResources().getMappedAttribute('context')];
      cellContext.indexes[axis] += value;
    }
  }
};

DvtDataGrid.prototype._modifyAxisHeaderContextIndex = function (axis, atIndex, count, value) {
  for (var i = atIndex; i < atIndex + count; i++) {
    var headers;
    if (axis === 'row') {
      headers = this._getHeadersByIndex(i, this.m_rowHeader,
        this.m_rowHeaderLevelCount, this.m_startRowHeader);
    } else {
      headers = this._getHeadersByIndex(i, this.m_rowEndHeader,
        this.m_rowEndHeaderLevelCount, this.m_startRowEndHeader);
    }
    for (var j = 0; j < headers.length; j++) {
      var header = headers[j];
      var headerContext = header[this.getResources().getMappedAttribute('context')];
      headerContext.index += value;
    }
  }
};

/**
 * Get a cell or header by index along a given axis, will return first cell on that axis
 * @private
 */
DvtDataGrid.prototype._getCellOrHeaderByIndex = function (index, axis) {
  var element = null;
  var cells = this._getAxisCellsByIndex(index, axis, true);
  if (cells != null && cells.length > 0) {
    element = cells[0];
  }
  if (element == null) {
    if (axis === 'row') {
      element = this._getHeaderByIndex(index, 0,
        this.m_rowHeader, this.m_rowHeaderLevelCount,
        this.m_startRowHeader);
      if (element == null) {
        element = this._getHeaderByIndex(index, 0,
          this.m_rowEndHeader, this.m_rowEndHeaderLevelCount,
          this.m_startRowEndHeader);
      }
    }
    if (axis === 'column') {
      element = this._getHeaderByIndex(index, 0,
        this.m_colHeader, this.m_columnHeaderLevelCount,
        this.m_startColHeader);
      if (element == null) {
        element = this._getHeaderByIndex(index, 0,
          this.m_colEndHeader, this.m_columnEndHeaderLevelCount,
          this.m_startColEndHeader);
      }
    }
  }
  return element;
};

/**
 * Get a label by axis and level
 * @param {string} axis
 * @param {number} level
 * @returns {Element|null}
 */
DvtDataGrid.prototype._getLabel = function (axis, level) {
  return this.m_headerLabels[axis][level];
};

/**
 * Get a cell by index
 * @param {Object} indexes
 * @returns {Element|null}
 */
DvtDataGrid.prototype._getCellByIndex = function (indexes) {
  var id = this._getFromDatabodyMap(indexes);
  if (id != null) {
    // databody isn't necessarily attached to the document
    return this.m_databody.querySelector('#' + id);
  }
  return null;
};

DvtDataGrid.prototype._getCellsInRange = function (startRow, startColumn, endRow, endColumn) {
  let cells = [];
  for (let i = startRow; i <= endRow; i++) {
    for (let j = startColumn; j <= endColumn; j++) {
      let cell = this._getCellByIndex(this.createIndex(i, j));
      if (cell) {
        cells.push(cell);
      }
    }
  }
  return cells;
};

DvtDataGrid.prototype._getFirstCellWithMatchingStartIndex = function (index, axis) {
  // find the first cell that has the given axis index as its startIndex
  let startAxisIndex = axis === 'row' ? this.m_startCol : this.m_startRow;
  let endAxisIndex = axis === 'row' ? this.m_endCol : this.m_endRow;
  let indexes;
  let cell;
  for (let i = startAxisIndex; i <= endAxisIndex; i++) {
    indexes = this.createIndex(axis === 'row' ? index : i, axis === 'row' ? i : index);
    cell = this._getCellByIndex(indexes);
    // this will be actual start index
    if (this._getIndex(cell, axis) === index) {
      return cell;
    }
  }

  return null;
};

/**
 * Get all the cells along an axis by index
 * @param {number} index
 * @param {string} axis row/column
 * @param {boolean=} breakOnFirstFind
 * @returns {Array|null}
 * @private
 */
DvtDataGrid.prototype._getAxisCellsByIndex = function (index, axis, breakOnFirstFind) {
  var start = axis === 'row' ? this.m_startCol : this.m_startRow;
  var end = axis === 'row' ? this.m_endCol : this.m_endRow;
  var axisExtent;
  var cells = [];

  for (var i = start; i <= end; i += axisExtent) {
    var cell = this._getCellByIndex(this.createIndex(axis === 'row' ? index : i,
      axis === 'row' ? i : index));
    if (cell != null) {
      axisExtent = this.getCellExtents(cell)[axis === 'row' ? 'column' : 'row'];
      cells.push(cell);
      if (breakOnFirstFind) {
        break;
      }
    } else {
      axisExtent = 1;
    }
  }
  return cells;
};

/**
 * Get all the cells along an axis by key
 * @param {string|null} key
 * @param {string} axis row/column
 * @param {boolean=} breakOnFirstFind
 * @returns {Array|null}
 * @private
 */
DvtDataGrid.prototype._getAxisCellsByKey = function (key, axis, breakOnFirstFind) {
  if (key == null || this.m_databody == null || this.m_databody.firstChild == null) {
    return null;
  }

  var matchingCells = [];
  var databodyContent = this.m_databody.firstChild;
  var cells = databodyContent.childNodes;

  for (var i = 0; i < cells.length; i++) {
    var cell = cells[i];
    if (this.m_utils.containsCSSClassName(cell, this.getMappedStyle('cell'))) {
      var axisKey = this._getKey(cell, axis);
      if (axisKey === key) {
        matchingCells.push(cell);
        if (breakOnFirstFind) {
          break;
        }
      }
    }
  }

  // can't find it, the row is not in viewport
  return matchingCells;
};

/**
 * Build the actions object which maps actions to the methods to invoke because of them
 */
DvtDataGrid.prototype._setupActions = function () {
  this.actions = {
    ACTIONABLE: this._handleActionable,
    EXIT_ACTIONABLE: this._handleExitActionable,
    TAB_NEXT_IN_CELL: handleActionableTab,
    TAB_PREV_IN_CELL: handleActionablePrevTab,
    EDITABLE: this._handleEditable, // if editable go into edit, if not go into actionable
    EXIT_EDITABLE: this._handleExitEditable,
    DATA_ENTRY: this._handleDataEntry,
    EXIT_DATA_ENTRY: this._handleExitDataEntry,
    EDIT: this._handleEdit,
    EXIT_EDIT: this._handleExitEdit,
    CANCEL_EDIT: this._handleCancelEdit,
    NO_OP: this._handleNoOp,
    FOCUS_LEFT: this._handleFocusLeft,
    FOCUS_RIGHT: this._handleFocusRight,
    FOCUS_UP: this._handleFocusUp,
    FOCUS_DOWN: this._handleFocusDown,
    FOCUS_ROW_FIRST: this._handleFocusRowFirst,
    FOCUS_ROW_LAST: this._handleFocusRowLast,
    FOCUS_COLUMN_FIRST: this._handleFocusColumnFirst,
    FOCUS_COLUMN_LAST: this._handleFocusColumnLast,
    FOCUS_COLUMN_HEADER: this._handleFocusColumnHeader,
    FOCUS_COLUMN_END_HEADER: this._handleFocusColumnEndHeader,
    FOCUS_ROW_HEADER: this._handleFocusRowHeader,
    FOCUS_ROW_END_HEADER: this._handleFocusRowEndHeader,
    READ_CELL: this.readCurrentContent,
    SORT: this._handleSortKey,
    EXPAND: this._handleExpandKey,
    COLLAPSE: this._handleCollapseKey,
    SELECT_DISCONTIGUOUS: this._handleSelectDiscontiguous,
    SELECT_EXTEND_LEFT: this._handleExtendSelectionLeft,
    SELECT_EXTEND_RIGHT: this._handleExtendSelectionRight,
    SELECT_EXTEND_UP: this._handleExtendSelectionUp,
    SELECT_EXTEND_DOWN: this._handleExtendSelectionDown,
    SELECT_ROW: this._handleSelectRow,
    SELECT_COLUMN: this._handleSelectColumn,
    SELECT_ALL: this._handleSelectAll,
    CUT: this._handleCut,
    CUT_CELLS: this._handleCutCells,
    COPY_CELLS: this._handleCopyCells,
    CANCEL_REORDER: this._handleCancelReorder,
    PASTE: this._handlePaste,
    PASTE_CELLS: this._handlePasteCells,
    FILL: this._handleAutofill
  };
};

/**
 * Get the function for a given keydown event.
 * @param {Event} event
 * @param {string} cellOrHeader 'cell'/'header'
 * @returns {Function|undefined} the function to invoke due to the keydown
 */
DvtDataGrid.prototype._getActionFromKeyDown = function (event, cellOrHeader, label) {
  var capabilities = {
    cellOrHeader: cellOrHeader,
    isLabel: label,
    readOnly: !this._isCellEditable(),
    currentMode: this._getCurrentMode(),
    activeMove: (this.m_cutCells != null || this.m_dataTransferAction != null),
    rowMove: this._isMoveEnabled('row'),
    columnSort: cellOrHeader === 'column' ?
      this._isDOMElementSortable(this._getActiveElement()) : false,
    selection: this._isSelectionEnabled(),
    selectionMode: this.m_options.getSelectionMode(),
    multipleSelection: this.isMultipleSelection(),
    expandCollapse: this._isTargetExpandCollapseEnabled(event.target)
  };
  if (this._isDataGridProvider()) {
    capabilities.cutCells = true;
    capabilities.copyCells = true;
    capabilities.pasteCells = true;
  }
  if (this.m_options.isFloodFillEnabled()) {
    capabilities.fill = true;
  }
  return this.actions[this.m_keyboardHandler.getAction(event, capabilities)];
};

// ////////////////////////////////// ACTIONABLE METHODS/////////////////////////
/**
 * Determine if the data grid is in actionable mode.
 * @return returns true if the data grid is in actionable mode, false otherwise.
 * @protected
 */
DvtDataGrid.prototype.isActionableMode = function () {
  return (this.m_currentMode === 'actionable');
};

/**
 * Sets whether the data grid is in actionable mode or reverts it to navigation mode
 * @param {boolean} flag true to set grid to actionable mode, false otherwise
 * @protected
 */
DvtDataGrid.prototype.setActionableMode = function (flag) {
  if (flag) {
    this.m_currentMode = 'actionable';
  } else {
    this.m_currentMode = 'navigation';
  }

  // update screen reader alert
  this._setAccInfoText(this.isActionableMode() ?
                       'accessibleActionableMode' : 'accessibleNavigationMode');
};

/**
 * Enter actionable mode
 * @param {Event} event the event triggering actionable mode
 * @param {Element|undefined|null} element to set actionable
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleActionable = function (event, element) {
  this._enterActionableMode(element, event);
  return false;
};

/**
 * Exit actionable mode on the active cell if in actionable mode
 * @param {Event} event the event exiting actionable mode
 * @param {Element|undefined|null} element to unset actionable
 * @returns {boolean} false
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleExitActionable = function (event, element) {
  this._exitActionableMode();
  this._highlightActive();
  return false;
};

// ////////////////////////////////// EDITING METHODS/////////////////////////
/**
 * Get the edit mode values can be none/cell
 * @returns {string}  none or cell
 * @private
 */
DvtDataGrid.prototype._getEditMode = function () {
  if (this.m_editMode == null) {
    this.m_editMode = this.m_options.getEditMode();
  }
  return this.m_editMode;
};

/**
 * Get the current mode of the datagrid
 * @returns {string} navigation/actionable/enter/edit
 * @private
 */
DvtDataGrid.prototype._getCurrentMode = function () {
  if (this.m_currentMode == null) {
    this.m_currentMode = 'navigation';
  }
  return this.m_currentMode;
};

/**
 * Is the current mode edit or enter
 * @returns {boolean} true if edit or enter
 * @private
 */
DvtDataGrid.prototype._isEditOrEnter = function () {
  var c = this._getCurrentMode();
  return c === 'edit';
};

/**
 * Can the grid as a whole be editable
 * @returns {boolean} true if the edit mode is cell
 * @private
 */
DvtDataGrid.prototype._isGridEditable = function () {
  var editMode = this._getEditMode();
  if (editMode === 'cellNavigation' || editMode === 'cellEdit') {
    return true;
  }
  return false;
};

/**
 * Is the grid in ediable or readOnly mode
 * @returns {boolean} true if the edit mode is cell and editable enabled
 * @private
 */
DvtDataGrid.prototype._isCellEditable = function () {
  var editMode = this._getEditMode();
  if (editMode === 'cellEdit') {
    return true;
  }
  return false;
};


/**
 * Enter editable mode
 * @param {Event} event the event triggering editable mode
 * @param {Element|undefined|null} element
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleEditable = function (event, element) {
  if (this._isGridEditable()) {
    this.m_editMode = null;
    this.m_setOptionCallback('editMode', 'cellEdit', {
      _context: {
        writeback: true,
        internalSet: true
      }
    });
    this.m_utils.removeCSSClassName(this.m_root, this.getMappedStyle('readOnly'));
    this.m_utils.addCSSClassName(this.m_root, this.getMappedStyle('editable'));
    this._updateEdgeCellBorders('');
    this._setAccInfoText('accessibleEditableMode');
    this._setEditableClone(element);
  } else {
    this._handleActionable(event, element);
  }
  return false;
};

/**
 * Exit editable mode
 * @param {Event} event the event triggering editable mode
 * @param {Element|undefined|null} element
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleExitEditable = function (event, element) {
  this.m_editMode = null;
  this.m_setOptionCallback('editMode', 'cellNavigation', {
    _context: {
      writeback: true,
      internalSet: true
    }
  });
  this.m_utils.addCSSClassName(this.m_root, this.getMappedStyle('readOnly'));
  this.m_utils.removeCSSClassName(this.m_root, this.getMappedStyle('editable'));
  this._updateEdgeCellBorders('none');
  this._setAccInfoText('accessibleNavigationMode');
  this._destroyEditableClone();
};

/**
 * Enter enter mode
 * @param {Event} event the event triggering enter mode
 * @param {Element|undefined|null} element
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleDataEntry = function (event, element) {
  var details = {
    event: event,
    ui: {
      cell: element,
      cellContext: element[this.getResources().getMappedAttribute('context')]
    }
  };
  var rerender = this.fireEvent('beforeEdit', details);
  if (rerender) {
    this._removeFloodFillAffordance();
    this._reRenderCell(element, 'edit', this.getMappedStyle('cellEdit'), this.m_editableClone);
    // focus on first focusable item in the cell
    this._overwriteFlag = true;
    if (this._setFocusToFirstFocusableElement(element)) {
      this.m_currentMode = 'edit';
    } else {
      // if there was nothing to edit remove the edit class
      this.m_utils.removeCSSClassName(element, this.getMappedStyle('cellEdit'));
    }
    this._overwriteFlag = false;
  }
  return false;
};

/**
 * Exit enter mode
 * @param {Event} event the event triggering exit enter mode
 * @param {Element|undefined|null} element
 * @returns {boolean} true if enter is left successfully
 */
DvtDataGrid.prototype._handleExitDataEntry = function (event, element) {
  return this._leaveEditing(event, element, false);
};

/**
 * Enter edit mode
 * @param {Event} event the event triggering edit mode
 * @param {Element|undefined|null} element
 * @returns {boolean} true if edit mode entered
 */
DvtDataGrid.prototype._handleEdit = function (event, element) {
  var details = {
    event: event,
    ui: {
      cell: element,
      cellContext: element[this.getResources().getMappedAttribute('context')]
    }
  };
  var rerender = this.fireEvent('beforeEdit', details);
  if (rerender) {
    this._removeFloodFillAffordance();
    this._reRenderCell(element, 'edit', this.getMappedStyle('cellEdit'), this.m_editableClone);
    this.m_currentMode = 'edit';
    var self = this;
    var busyContext = Context.getContext(element).getBusyContext();
    busyContext.whenReady().then(function () {
      // focus on first focusable item in the cell
      var setFocus = self._setFocusToFirstFocusableElement(element);
      if (!setFocus) {
        // if there was nothing to edit remove the edit class
        self.m_utils.removeCSSClassName(element, self.getMappedStyle('cellEdit'));
        self.m_currentMode = 'navigation';
      }
    });
  } else {
    rerender = false;
    this._enterActionableMode(element);
  }
  return rerender;
};

/**
 * Exit edit mode
 * @param {Event} event
 * @param {Element|undefined|null} element
 * @returns {boolean} true if editing left successully
 */
DvtDataGrid.prototype._handleExitEdit = function (event, element) {
  return this._leaveEditing(event, element, false);
};

/**
 * Cancel an edit
 * @param {Event} event
 * @param {Element|undefined|null} element
 * @returns {boolean} true if editing cancelled successully
 */
DvtDataGrid.prototype._handleCancelEdit = function (event, element) {
  if (this._leaveEditing(event, element, true)) {
    this._setEditableClone(element);
    return true;
  }
  return false;
};

/**
 * Leave editing
 * @param {Event} event the event triggering actionable mode
 * @param {Element|undefined|null} element
 * @param {boolean} cancel
 * @param {boolean} shouldFocus
 * @returns {boolean} false
 */
DvtDataGrid.prototype._leaveEditing = function (event, element, cancel, shouldFocus) {
  var details = {
    event: event,
    ui: {
      cell: element,
      cellContext: element[this.getResources().getMappedAttribute('context')],
      cancelEdit: cancel
    }
  };
  if (!cancel) {
    disableAllFocusableElements(element);
    if (shouldFocus === false) {
      this.m_shouldFocus = shouldFocus;
    }
    this._highlightActive();
  }
  var rerender = this.fireEvent('beforeEditEnd', details);
  if (rerender) {
    this.m_currentMode = 'navigation';
    disableAllFocusableElements(element);
      if (shouldFocus === false) {
        this.m_shouldFocus = shouldFocus;
      }
      this._highlightActive();
      this._reRenderCell(element, 'navigation',
                       this.getMappedStyle('cellEdit'), this.m_editableClone);
  } else {
    rerender = false;
    this._scrollToActive(this.m_active);
    // focus on first focusable item in the cell
    this._setFocusToFirstFocusableElement(element);
  }
  return rerender;
};

// ////////////////////////////////// FOCUS METHODS/////////////////////////
/**
 * Handles all the various focus changes by keystroke
 * @param {Event} event
 * @param {Element} element
 * @param {number} keyCode
 * @param {boolean} isExtend
 * @param {boolean} jumpToHeaders
 * @returns {boolean} true if event processed
 */
DvtDataGrid.prototype._handleFocusKey =
  function (event, element, keyCode, isExtend, jumpToHeaders) {
    var changeFocus = true;
    var changeRegions = true;
    var editing;

    if (this.m_active != null) {
      if (this.m_active.type === 'cell') {
        if (this._isEditOrEnter()) {
          editing = true;
          changeFocus = this._leaveEditing(event, element, false);
          changeRegions = false;
        } else if (this.isActionableMode()) {
          this._exitActionableMode();
        }

        if (changeFocus) {
          if (this.m_options.isFloodFillEnabled()) {
            this._removeFloodFillAffordance();
          }
          var oldActive = this.m_active;
          var returnVal =
            this.handleFocusChange(keyCode, isExtend, event, changeRegions, jumpToHeaders);
          if (this._isGridEditable() && oldActive !== this.m_active &&
              editing && this.m_utils.isTouchDevice()) {
            return this._handleDataEntry(event, this._getActiveElement());
          }
          return returnVal;
        }
        return true;
      } else if (this.m_active.type === 'header') {
        return this.handleHeaderFocusChange(keyCode, event, isExtend, jumpToHeaders);
      } else if (this.m_active.type === 'label') {
        return this.handleLabelFocusChange(keyCode, event, isExtend, jumpToHeaders);
      }
    }
    return false;
  };

/**
 * Handle a focus to the left element
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusLeft = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.LEFT_KEY, false, false);
};

/**
 * Handle a focus to the right element
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusRight = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.RIGHT_KEY, false, false);
};

/**
 * Handle a focus to the up element
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusUp = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.UP_KEY, false, false);
};

/**
 * Handle a focus to the down element
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusDown = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.DOWN_KEY, false, false);
};

/**
 * Handle a focus to the first row same column
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusRowFirst = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.PAGEUP_KEY, false, false);
};

/**
 * Handle a focus to the last row same column
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusRowLast = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.PAGEDOWN_KEY, false, false);
};

/**
 * Handle a focus to the first column same row
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusColumnFirst = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.HOME_KEY, false, false);
};

/**
 * Handle a focus to the last column same row
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusColumnLast = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.END_KEY, false, false);
};

/**
 * Handle a focus to the row header
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusRowHeader = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.LEFT_KEY, false, true);
};

/**
 * Handle a focus to the row end header
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusRowEndHeader = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.RIGHT_KEY, false, true);
};

/**
 * Handle a focus to the column header
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusColumnHeader = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.UP_KEY, false, true);
};

/**
 * Handle a focus to the column end header
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} false
 */
DvtDataGrid.prototype._handleFocusColumnEndHeader = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.DOWN_KEY, false, true);
};

// ///////////////////// SELECTION METHODS ////////////////////////////
/**
 * Handle a selection of the whole row
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
DvtDataGrid.prototype._handleSelectRow = function (event, element) {
  var start;
  var end;
  var level;
  var index;
  var extent = 1;

  if (!this._isSelectionEnabled() ||
      (!this.isMultipleSelection() && this.m_options.getSelectionMode() !== 'row')) {
    return false;
  }

  if (this.m_utils.containsCSSClassName(element, this.getMappedStyle('cell'))) {
    index = this.m_active.indexes.row;
    start = index;
    end = index;
    level = this.m_rowHeaderLevelCount - 1;
  } else {
    if (this.m_active == null || this.m_active.type !== 'header' ||
        this.m_active.axis.indexOf('row') === -1) {
      return false;
    }
    index = this.m_active.index;
    level = this.m_active.level;
    if (this.m_rowHeaderLevelCount - 1 === level) {
      start = index;
      end = index;
    } else {
      var elem = this._getActiveElement();
      start = this._getAttribute(elem.parentNode, 'start', true);
      extent = this._getAttribute(elem.parentNode, 'extent', true);
      end = (start + extent) - 1;
    }
  }

  if (this.m_selectionFrontier) {
    if (this.m_selectionFrontier.axis &&
        this.m_selectionFrontier.axis.indexOf('column') !== -1) {
      this._handleSelectAll(event);
      return true;
    }
  }

  // set the frontier as it would be with header selection
  this.setHeaderSelectionFrontier('row', end, index, level, element, true);

  if (this._shouldDeselectHeader(index, extent, 'column')) {
    var rangeStart = this.createIndex(index, 0);
    var rangeEnd = this.createIndex((index + extent) - 1, -1);
    var returnObj = this._getSelectionStartAndEnd(rangeStart, rangeEnd, 0);
    var range = this.createRange(
      this.createIndex(returnObj.min.row, 0),
      this.createIndex(returnObj.max.row, -1));
    var trimmedRange = this._trimRangeForSelectionMode(range);
    this.m_deselectInfo = { selection: this.GetSelection() };
    this._deselectRange(trimmedRange, event);
    return true;
  }

  // handle the space key in headers
  this._selectEntireRow(start, end, event);


  // announce to screen reader, no need to include context info
  this._setAccInfoText('accessibleRowSelected', { row: index + 1 });
  return true;
};

/**
 * Handle a selection of the whole column
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
DvtDataGrid.prototype._handleSelectColumn = function (event, element) {
  var start;
  var end;
  var level;
  var index;
  var extent = 1;

  if (!this._isSelectionEnabled() || !this.isMultipleSelection() ||
      this.m_options.getSelectionMode() !== 'cell') {
    return false;
  }

  if (this.m_utils.containsCSSClassName(element, this.getMappedStyle('cell'))) {
    index = this.m_active.indexes.column;
    start = index;
    end = index;
    level = this.m_columnHeaderLevelCount - 1;
  } else {
    if (this.m_active == null || this.m_active.type !== 'header' ||
        this.m_active.axis.indexOf('column') === -1) {
      return false;
    }
    index = this.m_active.index;
    level = this.m_active.level;
    if (this.m_columnHeaderLevelCount - 1 === level) {
      start = index;
      end = index;
    } else {
      var elem = this._getActiveElement();
      start = this._getAttribute(elem.parentNode, 'start', true);
      extent = this._getAttribute(elem.parentNode, 'extent', true);
      end = (start + extent) - 1;
    }
  }

  if (this.m_selectionFrontier) {
    if (this.m_selectionFrontier.axis && this.m_selectionFrontier.axis.indexOf('row') !== -1) {
      this._handleSelectAll(event);
      return true;
    }
  }

  // set the frontier as it would be with header selection
  this.setHeaderSelectionFrontier('column', end, index, level, element, true);


  if (this._shouldDeselectHeader(index, extent, 'row')) {
    var rangeStart = this.createIndex(0, index);
    var rangeEnd = this.createIndex(-1, (index + extent) - 1);
    var returnObj = this._getSelectionStartAndEnd(rangeStart, rangeEnd, 0);
    var range = this.createRange(
      this.createIndex(0, returnObj.min.column),
      this.createIndex(-1, returnObj.max.column));
    var trimmedRange = this._trimRangeForSelectionMode(range);
    this.m_deselectInfo = { selection: this.GetSelection() };
    this._deselectRange(trimmedRange, event);
    return true;
  }

  // handle the space key in headers
  this._selectEntireColumn(start, end, event);

  // announce to screen reader, no need to include context info
  this._setAccInfoText('accessibleColumnSelected', { column: index + 1 });
  return true;
};

/**
 * Handle entering discontiguous selection
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleSelectDiscontiguous = function (event, element) {
  this.setDiscontiguousSelectionMode(!this.m_discontiguousSelection);
  return true;
};

/**
 * Handle extend selection left
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleExtendSelectionLeft = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.LEFT_KEY, true, false);
};

/**
 * Handle extend selection right
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleExtendSelectionRight = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.RIGHT_KEY, true, false);
};

/**
 * Handle extend selection up
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleExtendSelectionUp = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.UP_KEY, true, false);
};

/**
 * Handle extend selection down
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleExtendSelectionDown = function (event, element) {
  return this._handleFocusKey(event, element, this.keyCodes.DOWN_KEY, true, false);
};

/**
 * Handle sort key
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
DvtDataGrid.prototype._handleSortKey = function (event, element) {
  // sort, first check if the column is sortable
  if (element.getAttribute(this.getResources().getMappedAttribute('sortable')) === 'true') {
    this._handleKeyboardSort(element, event);
    return true;
  }

  // enter actionable mode but don't prevent default so the action is taken
  return this._handleActionable(event, element);
};

/**
 * Handle expand key
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
 DvtDataGrid.prototype._handleExpandKey = function (event, element) {
  // expand, first check if the column is collapsed
  if (this._isHeaderCollapsed(element)) {
    this._handleExpandCollapseRequest(event);
    return true;
  }
  return false;
};

/**
 * Handle collapse key
 * @param {Event} event the event causing the action
 * @param {Element} element target cell or header of the event
 * @returns {boolean} true if processed
 */
 DvtDataGrid.prototype._handleCollapseKey = function (event, element) {
  // collapse, first check if the column is expanded
  if (this._isHeaderExpanded(element)) {
    this._handleExpandCollapseRequest(event);
    return true;
  }
  return false;
};

/**
 * Enter editable mode
 * @param {Event} event the event triggering actionable mode
 * @param {Element} element to set actionable
 * @returns {boolean} false
 */
// eslint-disable-next-line no-unused-vars
DvtDataGrid.prototype._handleNoOp = function (event, element) {
  return false;
};

DvtDataGrid.RESIZE_OFFSET = 5;
DvtDataGrid.RESIZE_TOUCH_OFFSET = 8;

/**
 * Handles what to do when the mouse moves. Sets the cursor based on .manageHeaderCursor(),
 * If this.m_isResizing is set to true, treats movement as resizing, calling .handleResizeMouseMove()
 * @param {Event} event - a mousemove event
 */
DvtDataGrid.prototype.handleResize = function (event) {
  // if not resizing, monitor the cursor position, otherwise handle resizing
  if (this.m_isResizing === false) {
    var header = this.find(event.target, 'header');
    var headerLabel = this.find(event.target, 'headerlabel');
    if (header == null) {
      header = this.find(event.target, 'endheader');
    }

    // only if we are inside our grid's headers, multiple grid case
    if ((header != null &&
        (header === this.m_rowHeader || header === this.m_colHeader ||
         header === this.m_rowEndHeader || header === this.m_colEndHeader)) ||
         headerLabel != null) {
      let isHeaderLabel = false;
      if (headerLabel) {
        isHeaderLabel = true;
      }
      this.m_cursor = this.manageHeaderCursor(event, isHeaderLabel);
      if (this.m_resizingElement != null) {
        if (this.m_cursor === 'default') {
          this.m_resizingElement.style.cursor = '';
          if (this._isSelectionEnabled()) {
            this.m_utils.addCSSClassName(event.target, this.getMappedStyle('hover'));
          }
          if (this.m_resizingElementSibling != null) {
            this.m_resizingElementSibling.style.cursor = '';
          }
        } else {
          this.m_resizingElement.style.cursor = this.m_cursor;
          this.m_utils.removeCSSClassName(event.target, this.getMappedStyle('hover'));
          if (this.m_resizingElementSibling != null) {
            this.m_resizingElementSibling.style.cursor = this.m_cursor;
          }
        }
      }
    }
  } else {
    this.handleResizeMouseMove(event);
  }
};

/**
 * On mousedown, if the cursor was set to row/col -resize, set the required resize values.
 * @param {Event} event - a mousedown event
 * @return {boolean} true if event processed
 */
DvtDataGrid.prototype.handleResizeMouseDown = function (event) {
  if (this.m_cursor === 'col-resize' || this.m_cursor === 'row-resize') {
    this.m_isResizing = true;
    if (this.m_utils.isTouchDevice()) {
      this.m_lastMouseX = event.touches[0].pageX;
      this.m_lastMouseY = event.touches[0].pageY;
    } else {
      document.addEventListener('mousemove', this.m_docMouseMoveListener, false);
      document.addEventListener('mouseup', this.m_docMouseUpListener, false);
      this.m_lastMouseX = event.pageX;
      this.m_lastMouseY = event.pageY;
    }

    this.m_overResizeLeft = 0;
    this.m_overResizeMinLeft = 0;
    this.m_overResizeTop = 0;
    this.m_overResizeMinTop = 0;
    this.m_overResizeRight = 0;
    this.m_overResizeBottom = 0;
    this.m_orginalResizeDimensions = {
      width: this.getElementWidth(this.m_resizingElement),
      height: this.getElementHeight(this.m_resizingElement)
    };

    return true;
  }
  return false;
};

DvtDataGrid.prototype._resizeSelectedHeaders = function (event, oldWidth, oldHeight,
                                                        newWidth, newHeight, size) {
  let resizingElement = this.m_resizingElement;
  let resizingElementAxis = this.getHeaderCellAxis(this.m_resizingElement);
  let resizingElementIndex = this.getHeaderCellIndex(this.m_resizingElement);
  let resizeHeaderMode = this._getResizeHeaderMode(this.m_resizingElement);
  let resizingElementLevel = this.getHeaderCellLevel(this.m_resizingElement);
  let allowResizeWithinSelection = false;
  if ((resizingElementAxis === 'column' &&
      (resizingElementLevel === this.m_columnHeaderLevelCount - 1)) ||
      (resizingElementAxis === 'row' &&
      (resizingElementLevel === this.m_rowHeaderLevelCount - 1)) ||
      (resizingElementAxis === 'columnEnd' &&
      (resizingElementLevel === this.m_columnEndHeaderLevelCount - 1)) ||
      (resizingElementAxis === 'rowEnd' &&
      (resizingElementLevel === this.m_rowEndHeaderLevelCount - 1))) {
    allowResizeWithinSelection = true;
  }
  let selectedHeaders;
  if (this.m_selection && this.m_selection.length && allowResizeWithinSelection) {
    selectedHeaders = this._getHeadersWithinSelection(this.m_selection[0],
                                  resizingElementIndex, resizingElementAxis);
  }
  if (selectedHeaders && selectedHeaders.length) {
    for (let i = 0; i < selectedHeaders.length; i++) {
      if (resizingElement !== selectedHeaders[i]) {
        this.m_resizingElement = selectedHeaders[i];
        if (resizeHeaderMode === 'column') {
          this.resizeColWidth(this.getElementDir(selectedHeaders[i], 'width'), newWidth);
        } else {
          this.resizeRowHeight(this.getElementDir(selectedHeaders[i], 'height'), newHeight);
        }
        // set the information we want to callback with in the resize event and callback

        this._fireResizeEvent(event, oldWidth, oldHeight, newWidth, newHeight, size);
      }
    }
  } else {
    if (resizeHeaderMode === 'column') {
      this.resizeColWidth(this.getElementDir(this.m_resizingElement, 'width'), newWidth);
    } else {
      this.resizeRowHeight(this.getElementDir(this.m_resizingElement, 'height'), newHeight);
    }
    this._fireResizeEvent(event, oldWidth, oldHeight, newWidth, newHeight, size);
  }
};

DvtDataGrid.prototype._fireResizeEvent = function (event, oldWidth, oldHeight,
                                                  newWidth, newHeight, size) {
  let details = {
    event: event,
    ui: {
      header: this._getKey(this.m_resizingElement),
      oldDimensions: {
        width: oldWidth,
        height: oldHeight
      },
      newDimensions: {
        width: newWidth,
        height: newHeight
      },
      // deprecating this part in 2.1.0
      size: size
    }
  };
  this.fireEvent('resize', details);
};

/**
 * On mouseup, if we were resizing, handle cursor and callback firing.
 * @param {Event} event - a mouseup event
 */
DvtDataGrid.prototype.handleResizeMouseUp = function (event) {
  if (this.m_isResizing === true) {
    var newWidth = this.getElementWidth(this.m_resizingElement);
    var newHeight = this.getElementHeight(this.m_resizingElement);
    let oldWidth = this.m_orginalResizeDimensions.width;
    let oldHeight = this.m_orginalResizeDimensions.height;
    let resizingElement = this.m_resizingElement;
    let size = (this.m_cursor === 'col-resize') ?
        resizingElement.style.width : resizingElement.style.height;
    if (newWidth !== this.m_orginalResizeDimensions.width ||
        newHeight !== this.m_orginalResizeDimensions.height) {
      if (this.m_cursor === 'col-resize' || this.m_cursor === 'row-resize') {
        if (this._isSelectionEnabled() && this.isMultipleSelection() &&
          this.m_selection.length && !this.m_discontiguousSelection) {
          this._resizeSelectedHeaders(event, oldWidth, oldHeight, newWidth, newHeight, size);
        }
      }
    }

    resizingElement.style.cursor = '';
    this._unhighlightResizeBorderColor();
    // no longer resizing
    this.m_isResizing = false;
    this.m_cursor = 'default';
    if (this.m_resizingElementSibling != null) {
      this.m_resizingElementSibling.style.cursor = '';
    }

    this.m_resizingElement = null;
    this.m_resizingElementMin = null;
    this.m_resizingElementSibling = null;
    this.m_orginalResizeDimensions = null;

    document.removeEventListener('mousemove', this.m_docMouseMoveListener, false);
    document.removeEventListener('mouseup', this.m_docMouseUpListener, false);
    // unregister all listeners
  }
};

/**
 * Check if has data-resizable attribute is set to 'true' on a header
 * @param {Element|undefined|null} element - element to check if has data-resizable true
 * @return {boolean} true if data-resizable attribute is 'true'
 */
DvtDataGrid.prototype._isDOMElementResizable = function (element) {
  if (element == null) {
    return false;
  }
  return element.getAttribute(this.getResources().getMappedAttribute('resizable')) === 'true';
};

/**
 * Determine what the document cursor should be for header cells.
 * @param {Event} event - a mousemove event
 * @return {string} the cursor type for a given mouse location
 */
DvtDataGrid.prototype.manageHeaderCursor = function (event, isLabel) {
  var cursorX;
  var cursorY;
  var offsetPixel;
  var widthResizable;
  var heightResizable;
  var siblingResizable;
  var sibling;
  var parent;

  // determine the element/header type that should be used for resizing if applicable
  var elem;
  elem = isLabel ? this.find(event.target, 'headerlabel') : this.find(event.target, 'headercell');
  if (!elem && !isLabel) {
    elem = this.find(event.target, 'endheadercell');
  }

  if (!elem) {
    return 'default';
  }

  var resizeHeaderMode = isLabel ? this.getHeaderLabelAxis(elem) : this.getHeaderCellAxis(elem);
  var index = isLabel ? this.getHeaderLabelLevel(elem) : this.getHeaderCellIndex(elem);
  var level = isLabel ? index : this.getHeaderCellLevel(elem);

  if (resizeHeaderMode === 'column') {
    heightResizable = this.m_options.isResizable(resizeHeaderMode, 'height') === 'enable';
    widthResizable = this._isDOMElementResizable(elem);
    // previous is the previous index same level
    if (!isLabel) {
      sibling = this._getHeaderByIndex(index - 1, level, this.m_colHeader,
                                      this.m_columnHeaderLevelCount, this.m_startColHeader);
      if (!sibling) {
        if (this.m_headerLabels.column.length) {
          sibling = this._getLabel('column', level);
          if (level === this.m_columnHeaderLevelCount - 1 && this._isHeaderLabelCollision()) {
            sibling = this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1];
          }
        } else {
          sibling = this._getLabel('row', level);
        }
      }
      siblingResizable = this._isDOMElementResizable(sibling);
      // parent is the previous level the same index
      parent = this._getHeaderByIndex(index, level - 1, this.m_colHeader,
                                      this.m_columnHeaderLevelCount, this.m_startColHeader);
    } else {
      sibling = null;
      siblingResizable = null;
      parent = this.m_headerLabels.column[index - 1];
    }
  } else if (resizeHeaderMode === 'row') {
    widthResizable = this.m_options.isResizable(resizeHeaderMode, 'width') === 'enable';
    heightResizable = this._isDOMElementResizable(elem);
    // previous is the previous index same level
    if (!isLabel) {
      sibling = this._getHeaderByIndex(index - 1, level, this.m_rowHeader,
                                      this.m_rowHeaderLevelCount, this.m_startRowHeader);
      siblingResizable = this._isDOMElementResizable(sibling);
      // parent is the previous level the same index
      parent = this._getHeaderByIndex(index, level - 1, this.m_rowHeader,
                                    this.m_rowHeaderLevelCount, this.m_startRowHeader);
    } else {
      parent = this.m_headerLabels.row[index - 1];
      // parent is the previous level the same index
      if (this._isHeaderLabelCollision() &&
        elem === this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1]) {
        sibling = this.m_headerLabels.column[this.m_columnHeaderLevelCount - 1];
      }
    }
  } else if (resizeHeaderMode === 'columnEnd') {
    heightResizable = this.m_options.isResizable(resizeHeaderMode).height === 'enable';
    widthResizable = this._isDOMElementResizable(elem);
    // previous is the previous index same level
    if (!isLabel) {
      sibling = this._getHeaderByIndex(index - 1, level, this.m_colEndHeader,
                                      this.m_columnEndHeaderLevelCount, this.m_startColEndHeader);
      if (!sibling) {
        sibling = this._getLabel('columnEnd', level);
      }
      siblingResizable = this._isDOMElementResizable(sibling);
      // parent is the previous level the same index
      parent = this._getHeaderByIndex(index, level - 1, this.m_colEndHeader,
                                      this.m_columnEndHeaderLevelCount, this.m_startColEndHeader);
    } else {
      sibling = null;
      siblingResizable = null;
      parent = this.m_headerLabels.columnEnd[index - 1];
    }
  } else if (resizeHeaderMode === 'rowEnd') {
    widthResizable = this.m_options.isResizable(resizeHeaderMode).width === 'enable';
    heightResizable = this._isDOMElementResizable(elem);
    // previous is the previous index same level
    if (!isLabel) {
      sibling = this._getHeaderByIndex(index - 1, level, this.m_rowEndHeader,
                                      this.m_rowEndHeaderLevelCount, this.m_startRowEndHeader);
      siblingResizable = this._isDOMElementResizable(sibling);
      // parent is the previous level the same index
      parent = this._getHeaderByIndex(index, level - 1, this.m_rowEndHeader,
                                      this.m_rowEndHeaderLevelCount, this.m_startRowEndHeader);
    } else {
      sibling = this.m_headerLabels.rowEnd[index - 1];
      siblingResizable = this._isDOMElementResizable(sibling);
    }
  }

  // touch requires an area 24px for touch gestures
  if (this.m_utils.isTouchDevice()) {
    cursorX = event.touches[0].pageX;
    cursorY = event.touches[0].pageY;
    offsetPixel = DvtDataGrid.RESIZE_TOUCH_OFFSET;
  } else {
    cursorX = event.offsetX;
    cursorY = event.offsetY;
    offsetPixel = DvtDataGrid.RESIZE_OFFSET;
    var headerOffset = this._findHeaderOffset(event.target, elem);
    cursorX += headerOffset[0];
    cursorY += headerOffset[1];
  }

  var edges = this.getHeaderEdgePixels(elem);
  var rtl = this.getResources().isRTLMode();
  var end = (resizeHeaderMode === 'columnEnd' || resizeHeaderMode === 'rowEnd');

  var leftEdgeCheck = cursorX < edges[0] + offsetPixel;
  var topEdgeCheck = cursorY < edges[1] + offsetPixel;
  var rightEdgeCheck = cursorX > edges[2] - offsetPixel;
  var bottomEdgeCheck = cursorY > edges[3] - offsetPixel;

  // check to see if resizable was enabled on the grid and then check the position of the cursor to the element border
  // we always choose the element preceding the border (so for rows the header before the bottom border)
  if (resizeHeaderMode === 'column' || resizeHeaderMode === 'columnEnd') {
    // can we resize the width of this header
    if (widthResizable && (rtl ? leftEdgeCheck : rightEdgeCheck)) {
      this.m_resizingElement = elem;
      return 'col-resize';
    } else if (siblingResizable && (rtl ? rightEdgeCheck : leftEdgeCheck)) {
      // can we resize the width of the previous header
      this.m_resizingElement = sibling;
      this.m_resizingElementSibling = elem;
      if (this.m_resizingElement !== null) {
        return 'col-resize';
      }
    } else if (heightResizable) {
      // can we resize the height of this header
      if ((!end && bottomEdgeCheck) || (end && topEdgeCheck)) {
        this.m_resizingElement = elem;
        return 'row-resize';
      } else if ((!end && topEdgeCheck) || (end && bottomEdgeCheck)) {
        // can we resize the height of the parent header
        this.m_resizingElement = parent;
        this.m_resizingElementSibling = elem;
        return 'row-resize';
      }
    }
  } else if (resizeHeaderMode === 'row' || resizeHeaderMode === 'rowEnd') {
    if (heightResizable && bottomEdgeCheck) {
      this.m_resizingElement = elem;
      return 'row-resize';
    } else if (siblingResizable && topEdgeCheck && !isLabel) {
      this.m_resizingElement = sibling;
      this.m_resizingElementSibling = elem;
      if (this.m_resizingElement !== null) {
        return 'row-resize';
      }
    } else if (parent && topEdgeCheck && isLabel) {
      this.m_resizingElement = parent;
      this.m_resizingElementSibling = elem;
      if (this.m_resizingElement !== null) {
        return 'row-resize';
      }
    }
    if (widthResizable) {
      if ((!end && (rtl ? leftEdgeCheck : rightEdgeCheck)) ||
          (end && (rtl ? rightEdgeCheck : leftEdgeCheck))) {
        this.m_resizingElement = elem;
        return 'col-resize';
      } else if ((!end && (rtl ? rightEdgeCheck : leftEdgeCheck)) ||
                 (end && (rtl ? leftEdgeCheck : rightEdgeCheck))) {
        this.m_resizingElement = parent;
        this.m_resizingElementSibling = elem;
        if (this.m_resizingElement !== null) {
          return 'col-resize';
        }
      }
    }
  }
  return 'default';
};

/**
 * On mousemove see which type of resizing we are doing and call the appropriate resizer after calculating
 * the new elements width based on current and last X and Y page coordinates.
 * @param {Event} event - a mousemove event
 */
DvtDataGrid.prototype.handleResizeMouseMove = function (event) {
  var resizeHeaderMode;
  var oldElementWidth;
  var newElementWidth;
  var oldElementHeight;
  var newElementHeight;

  // update stored mouse position
  this.m_currentMouseX = event.pageX;
  this.m_currentMouseY = event.pageY;

  if (this.m_utils.isTouchDevice()) {
    this.m_currentMouseX = event.touches[0].pageX;
    this.m_currentMouseY = event.touches[0].pageY;
  } else {
    this.m_currentMouseX = event.pageX;
    this.m_currentMouseY = event.pageY;
  }

  // check to see if we are resizing a column or row
  resizeHeaderMode = this._getResizeHeaderMode(this.m_resizingElement);

  var end = this.m_utils.containsCSSClassName(this.m_resizingElement,
                                              this.getMappedStyle('endheadercell')) ||
            this.m_utils.containsCSSClassName(this.m_resizingElement,
                                              this.getMappedStyle('columnendheaderlabel'));
  let isHeaderLabel = this.find(this.m_resizingElement, 'headerlabel');
  // handle width resizing for columns/rows
  if (this.m_cursor === 'col-resize') {
    if (resizeHeaderMode === 'column') {
      end = isHeaderLabel ? false : end;
      oldElementWidth = this.calculateColumnHeaderWidth(this.m_resizingElement);
      newElementWidth = this.getNewElementWidth('column', oldElementWidth, end, null, isHeaderLabel);

      if (isHeaderLabel) {
        this.resizeRowWidth(newElementWidth, newElementWidth - oldElementWidth, end, isHeaderLabel);
      } else {
        this.resizeColWidth(oldElementWidth, newElementWidth);
      }
    } else if (resizeHeaderMode === 'row') {
      if (this.m_utils.containsCSSClassName(this.m_resizingElement,
        this.getMappedStyle('rowendheaderlabel'))) {
          end = true;
      }
      oldElementWidth = this.getElementWidth(this.m_resizingElement);
      newElementWidth = this.getNewElementWidth('row', oldElementWidth, end, null, isHeaderLabel);

      this.resizeRowWidth(newElementWidth, newElementWidth - oldElementWidth, end, isHeaderLabel);
    }
  } else if (this.m_cursor === 'row-resize') {
    // handle height resizing for columns/rows
    if (resizeHeaderMode === 'row') {
      oldElementHeight = this.calculateRowHeaderHeight(this.m_resizingElement);
      newElementHeight = this.getNewElementHeight('row', oldElementHeight, end, null, isHeaderLabel);

      if (isHeaderLabel) {
        this.resizeColHeight(newElementHeight, newElementHeight - oldElementHeight, end);
      } else {
        this.resizeRowHeight(oldElementHeight, newElementHeight);
      }
    } else if (resizeHeaderMode === 'column') {
      oldElementHeight = this.getElementHeight(this.m_resizingElement);
      newElementHeight = this.getNewElementHeight('column', oldElementHeight, end, null, isHeaderLabel);
      this.resizeColHeight(newElementHeight, newElementHeight - oldElementHeight, end);
    }
  }

  // rebuild the corners
  this.buildCorners();

  // re-align touch affordances
  if (this.m_utils.isTouchDevice()) {
    this._moveTouchSelectionAffordance();
  }

  // update the last mouse X/Y
  this.m_lastMouseX = this.m_currentMouseX;
  this.m_lastMouseY = this.m_currentMouseY;
};


/**
 * Resize the width of column headers, and the column cells. Also resize the
 * scroller and databody accordingly. Set the left(or right) style value on all
 * cells/columns following(preceeding) the resizing element. Update the end
 * column pixel as well.
 * @param {number} oldElementWidth - the elements width prior to resizing
 * @param {number} newElementWidth - the elements width after resizing
 */
DvtDataGrid.prototype.resizeColWidth = function (oldElementWidth, newElementWidth) {
  var newScrollerWidth;
  var widthChange = newElementWidth - oldElementWidth;

  if (widthChange !== 0) {
    if (this.m_databody.firstChild != null) {
      var oldScrollerWidth = this.getElementWidth(this.m_databody.firstChild);
      newScrollerWidth = oldScrollerWidth + widthChange;
      this.setElementWidth(this.m_databody.firstChild, newScrollerWidth);
    }

    // helper to update all elements this effects
    this.resizeColumnWidthAndShift(widthChange);

    this.m_endColPixel += widthChange;
    this.m_endColHeaderPixel += widthChange;
    this.m_endColEndHeaderPixel += widthChange;
    if (newScrollerWidth != null) {
      this.m_avgColWidth = newScrollerWidth / this.getDataSource().getCount('column');
    }

    this.manageResizeScrollbars();
  }
};

/**
 * Resize the height of row headers, and the rows cells. Also resize the
 * scroller and databody accordingly. Update the end row pixel as well.
 * @param {number} oldElementHeight - the elements height prior to resizing
 * @param {number} newElementHeight - the elements height after resizing
 */
DvtDataGrid.prototype.resizeRowHeight = function (oldElementHeight, newElementHeight) {
  var newScrollerHeight;
  var heightChange = newElementHeight - oldElementHeight;

  if (heightChange !== 0) {
    if (this.m_databody.firstChild != null) {
      var oldScrollerHeight = this.getElementHeight(this.m_databody.firstChild);
      newScrollerHeight = oldScrollerHeight + heightChange;
      this.setElementHeight(this.m_databody.firstChild, newScrollerHeight);
    }

    // set row height on the appropriate databody row, set the new value in the sizingManager
    this.resizeRowHeightAndShift(heightChange);

    this.m_endRowPixel += heightChange;
    this.m_endRowHeaderPixel += heightChange;
    this.m_endRowEndHeaderPixel += heightChange;
    if (newScrollerHeight != null) {
      this.m_avgRowHeight = newScrollerHeight / this.getDataSource().getCount('row');
    }
    this.manageResizeScrollbars();
  }
};

/**
 * Resize the height of column headers. Also resize the scroller and databody
 * accordingly.
 * @param {number} newElementHeight - the column header height after resizing
 * @param {number} heightChange - the change in height
 * @param {boolean} end
 */
DvtDataGrid.prototype.resizeColHeight = function (newElementHeight, heightChange, end) {
  if (heightChange !== 0) {
    var oldHeight;
    var level;
    var axis;
    // shift header label if there is no collision and if the resizing element
    // is not header label.
    var adjustLabel = true;
    var isHeaderLabel = this.find(this.m_resizingElement, 'headerlabel');
    if (isHeaderLabel) {
      axis = this.getHeaderLabelAxis(this.m_resizingElement);
      if (axis === 'column' || axis === 'columnEnd') {
        level = this.getHeaderLabelLevel(this.m_resizingElement);
      } else if (axis === 'row' || axis === 'rowEnd') {
        level = this.m_columnHeaderLevelCount - 1;
        adjustLabel = false;
      }
    } else {
      level = this.getHeaderCellLevel(this.m_resizingElement) +
      (this.getHeaderCellDepth(this.m_resizingElement) - 1);
      axis = this.getHeaderCellAxis(this.m_resizingElement);
    }
    if (end) {
      this.m_columnEndHeaderLevelHeights[level] += heightChange;
    } else {
      oldHeight = this.m_columnHeaderLevelHeights[level];
      this.m_columnHeaderLevelHeights[level] += heightChange;
    }
    if (!end && level === this.m_columnHeaderLevelCount - 1 &&
      this.m_headerLabels.row.length &&
      this._isHeaderLabelCollision()) {
        adjustLabel = false;
    }
    this.resizeColumnHeightsAndShift(heightChange, level, end, adjustLabel);

    if (!end) {
      this.m_colHeaderHeight += heightChange;
      this.setElementHeight(this.m_colHeader, this.m_colHeaderHeight);
      if (this.m_headerLabels.column.length === 0) {
        this._resizeHeaderLabelDirs(level, heightChange, ['row'], 'height');
      } else if (level === this.m_columnHeaderLevelCount - 1 && this.m_headerLabels.row.length) {
        var rowHeightChange;
        var colHeight;
        if (isHeaderLabel) {
          if (axis === 'column') {
            colHeight = newElementHeight;
            this.m_collisionResize = true;
          } else if (axis === 'row') {
            rowHeightChange = heightChange;
            this.m_collisionResize = true;
          } else if (axis === 'rowEnd') {
            if (this._isHeaderLabelCollision()) {
              let dimension = this._calculateCollisionDimension(this.m_colHeaderHeight,
                oldHeight, isHeaderLabel, axis);
              rowHeightChange = dimension.rowHeightChange;
              colHeight = dimension.colHeight;
            } else {
              rowHeightChange = heightChange;
              colHeight = this.m_colHeaderHeight;
            }
          }
        } else if (this._isHeaderLabelCollision()) {
          let dimension = this._calculateCollisionDimension(newElementHeight, oldHeight,
            true, axis);
          rowHeightChange = dimension.rowHeightChange;
          colHeight = dimension.colHeight;
        } else {
          rowHeightChange = heightChange;
          colHeight = newElementHeight;
        }

        var columnHeaderLabelZero = this._getLabel('column', this.m_columnHeaderLevelCount - 1);
        if (columnHeaderLabelZero) {
          this.setElementHeight(columnHeaderLabelZero, colHeight);
        }
        this._resizeHeaderLabelDirs(this.m_rowHeaderLevelCount - 1,
                                    rowHeightChange,
                                    ['row'], 'height');
      }
    } else {
      this.m_colEndHeaderHeight += heightChange;
      this.setElementHeight(this.m_colEndHeader, this.m_colEndHeaderHeight);
    }
    this.manageResizeScrollbars();
  }
};

/**
 * Calculate dimension of the colliding rows and columns.
 * @param {number} newElementHeight - the element's height after resizing
 * @param {number} oldHeight - the element's height prior to resizing
 */
DvtDataGrid.prototype._calculateCollisionDimension = function (newElementHeight, oldHeight,
  isHeaderLabel, axis) {
  let dimension = {};
  let rowHeightChange;
  let colHeight;
  let minHeightValue = this._getMinValue('height', axis, isHeaderLabel);
  if (this.m_collisionResize) {
    let collisionRowHeight = this.getElementDir(
      this.m_headerLabels.row[this.m_rowHeaderLevelCount - 1], 'height');
    let collisionColumnHeight = this.getElementDir(
      this.m_headerLabels.column[this.m_columnHeaderLevelCount - 1], 'height');
    let totalHeight = collisionRowHeight + collisionColumnHeight;
    let columnPercentage = collisionColumnHeight / totalHeight;
    let rowPercentage = collisionRowHeight / totalHeight;
    if (collisionRowHeight === minHeightValue && collisionColumnHeight === minHeightValue) {
      rowHeightChange = Math.floor(newElementHeight / 2) - Math.floor(oldHeight / 2);
      colHeight = Math.ceil(newElementHeight / 2);
      this.m_collisionResize = false;
    } else if (columnPercentage > rowPercentage) {
      colHeight = Math.floor(newElementHeight * columnPercentage);
      rowHeightChange = (newElementHeight - colHeight) - collisionRowHeight;
      if (collisionRowHeight + rowHeightChange < this._getMinValue('height', axis, isHeaderLabel)) {
        rowHeightChange = 0;
      }
    } else {
      rowHeightChange = Math.floor(newElementHeight * rowPercentage) -
        collisionRowHeight;
      if (collisionRowHeight + rowHeightChange < this._getMinValue('height', axis, isHeaderLabel)) {
        rowHeightChange = 0;
      }
      colHeight = newElementHeight - (collisionRowHeight + rowHeightChange);
      colHeight = Math.max(colHeight, this._getMinValue('height', axis, isHeaderLabel));
    }
  } else {
    rowHeightChange = Math.floor(newElementHeight / 2) - Math.floor(oldHeight / 2);
    colHeight = Math.ceil(newElementHeight / 2);
  }
  dimension.colHeight = colHeight;
  dimension.rowHeightChange = rowHeightChange;
  return dimension;
};

/**
 * Resize the width of row headers. Also resize the scroller and databody
 * accordingly.
 * @param {number} newElementWidth - the row header width after resizing
 * @param {number} widthChange - the change in width
 * @param {boolean} end
 */
DvtDataGrid.prototype.resizeRowWidth = function (newElementWidth, widthChange, end, isHeaderLabel) {
  if (widthChange !== 0) {
    var level;
    if (isHeaderLabel) {
      let axis = this.getHeaderLabelAxis(this.m_resizingElement);
      if (axis === 'column' || axis === 'columnEnd') {
        level = this.m_rowHeaderLevelCount - 1;
      } else if (axis === 'row' || axis === 'rowEnd') {
        level = this.getHeaderLabelLevel(this.m_resizingElement);
      }
    } else {
      level = this.getHeaderCellLevel(this.m_resizingElement) +
      (this.getHeaderCellDepth(this.m_resizingElement) - 1);
    }

    if (end) {
      this.m_rowEndHeaderLevelWidths[level] += widthChange;
    } else {
      this.m_rowHeaderLevelWidths[level] += widthChange;
    }
    this.resizeRowWidthsAndShift(widthChange, level, end);

    if (!end) {
      this.m_rowHeaderWidth += widthChange;
      this.setElementWidth(this.m_rowHeader, this.m_rowHeaderWidth);
      if (level === this.m_rowHeaderLevelCount - 1 ||
        (this.m_headerLabels.row.length === 0 && this.m_headerLabels.column.length)) {
        this._resizeHeaderLabelDirs(level, widthChange, ['column'], 'width');
      }
    } else {
      this.m_rowEndHeaderWidth += widthChange;
      this.setElementWidth(this.m_rowEndHeader, this.m_rowEndHeaderWidth);
    }
    this.manageResizeScrollbars();
  }
};

DvtDataGrid.prototype._resizeHeaderLabelDirs = function (level, dimensionChange, axes, dir) {
  for (var j = 0; j < axes.length; j++) {
    var axis = axes[j];
    for (var i = 0; i < this.m_headerLabels[axis].length; i++) {
      var label = this.m_headerLabels[axis][i];
      if (label != null) {
        var newDir = this.getElementDir(label, dir) + dimensionChange;
        this.setElementDir(label, newDir, dir);
      }
    }
    this._highlightResizeLabelDirs(axis, level);
  }
};

DvtDataGrid.prototype._highlightResizeLabelDirs = function (axis, level) {
  let rowClass = this.getResources().isRTLMode() ? ['startResized'] : ['endResized'];
  let classArray = axis === 'column' ? rowClass : ['bottomResized'];
  if (this.m_corner && ((axis === 'column' && level === this.m_columnHeaderLevelCount - 1) ||
      (axis === 'row' && level === this.m_rowHeaderLevelCount - 1))) {
      this._highlightElement(this.m_corner, classArray);
  }
  let endRowHeaderLabel = this._getLabel('rowEnd', this.m_columnHeaderLevelCount - 1);
  if (endRowHeaderLabel) {
    this._highlightElement(endRowHeaderLabel.parentNode, classArray);
  }
};
/**
 * Unhighlight Resize border color.
 */
DvtDataGrid.prototype._unhighlightResizeBorderColor = function () {
  let elems = this.m_root.querySelectorAll(`.oj-datagrid-resized-start,
    .oj-datagrid-resized-end, .oj-datagrid-resized-top, .oj-datagrid-resized-bottom`);
  this._unhighlightElementsByClassName(elems, ['startResized', 'endResized', 'topResized', 'bottomResized']);
};

/**
 * Determine what the new element width should be based on minimum values.
 * Accounts for the overshoot potential of passing up the boundries set.
 * @param {string} axis - the axis along which we need a new width
 * @param {number} oldElementWidth - the element width prior to resizing
 * @param {boolean} end
 * @param {number=} deltaWidth
 * @return {number} the element width after resizing
 */
DvtDataGrid.prototype.getNewElementWidth = function (axis, oldElementWidth, end, deltaWidth,
  isHeaderLabel) {
  // to account for the 24px resing width
  var minWidth;
  minWidth = this._getMinValue('width', axis, isHeaderLabel);
  if (deltaWidth == null) {
    // eslint-disable-next-line no-param-reassign
    deltaWidth = this.getResources().isRTLMode() ?
      this.m_lastMouseX - this.m_currentMouseX : this.m_currentMouseX - this.m_lastMouseX;
  }

  if (end && axis === 'row') {
    // eslint-disable-next-line no-param-reassign
    deltaWidth *= -1;
  }
  var newElementWidth = oldElementWidth + deltaWidth + this.m_overResizeLeft +
    this.m_overResizeMinLeft + this.m_overResizeRight;

  // check to make sure the element exceeds the minimum width
  if (newElementWidth < minWidth) {
    this.m_overResizeMinLeft += (deltaWidth - minWidth) + oldElementWidth;
    newElementWidth = minWidth;
  } else {
    this.m_overResizeMinLeft = 0;
    this.m_overResizeLeft = 0;
  }
  // check to make sure row header width don't exceed half of the grid width
  if (axis === 'row') {
    // this is the total width of the other headers (nested/end/start)
    var otherHeadersWidth = (this.getRowHeaderWidth() + this.getRowEndHeaderWidth())
      - oldElementWidth;
    // allow headers to grow to entire grid minus scroller and extra header area, minus 1 to make sure some databody is shown
    var maxHeaderWidth = Math.round(this.getWidth() - this.m_utils.getScrollbarSize() - 1)
      - otherHeadersWidth;
    if (newElementWidth > maxHeaderWidth) {
      this.m_overResizeRight += (deltaWidth - maxHeaderWidth) + oldElementWidth;
      newElementWidth = maxHeaderWidth;
    } else {
      this.m_overResizeRight = 0;
    }
  }
  return newElementWidth;
};

/**
 * Determine what the new element height should be based on minimum values.
 * Accounts for the overshoot potential of passing up the boundries set.
 * @param {string} axis - the axis along which we need a new width
 * @param {number} oldElementHeight - the element height prior to resizing
 * @param {boolean} end
 * @param {number=} deltaHeight
 * @return {number} the element height after resizing
 */
DvtDataGrid.prototype.getNewElementHeight = function (axis, oldElementHeight, end, deltaHeight,
  isHeaderLabel) {
  var minHeight;
  let headerLabelMinValue = this._getMinValue('height', axis, true);
  if (isHeaderLabel) {
    minHeight = headerLabelMinValue;
    let level = this.getHeaderLabelLevel(this.m_resizingElement);
    if (this.getHeaderLabelAxis(this.m_resizingElement) === 'rowEnd' &&
      level === this.m_columnHeaderLevelCount - 1 && this._isHeaderLabelCollision()) {
      minHeight += this.getElementHeight(
        this.m_headerLabels.row[this.m_headerLabels.row.length - 1]);
    }
  } else {
    minHeight = this._getMinValue('height', axis, isHeaderLabel);
  }
  if (axis === 'column' && !end &&
      this.getHeaderCellLevel(this.m_resizingElement) +
      this.getHeaderCellDepth(this.m_resizingElement) === this.m_columnHeaderLevelCount &&
      this._isHeaderLabelCollision()) {
    minHeight = 2 * headerLabelMinValue;
  }
  if (deltaHeight == null) {
    // eslint-disable-next-line no-param-reassign
    deltaHeight = this.m_currentMouseY - this.m_lastMouseY;
  }
  if (end && axis === 'column') {
    // eslint-disable-next-line no-param-reassign
    deltaHeight *= -1;
  }
  var newElementHeight = oldElementHeight + deltaHeight + this.m_overResizeTop +
    this.m_overResizeMinTop + this.m_overResizeBottom;

  // Check to make sure the element height exceeds the minimum height
  if (newElementHeight < minHeight) {
    this.m_overResizeMinTop += (deltaHeight - minHeight) + oldElementHeight;
    newElementHeight = minHeight;
  } else {
    this.m_overResizeMinTop = 0;
    this.m_overResizeTop = 0;
  }
  // check to make sure column header width don't exceed half of the grid height
  if (axis === 'column') {
    var otherHeadersHeight = (this.getColumnHeaderHeight() + this.getColumnEndHeaderHeight())
      - oldElementHeight;
    var maxHeaderHeight = Math.round(this.getHeight() - this.m_utils.getScrollbarSize() - 1)
      - otherHeadersHeight;
    if (newElementHeight > maxHeaderHeight) {
      this.m_overResizeBottom += (deltaHeight - maxHeaderHeight) + oldElementHeight;
      newElementHeight = maxHeaderHeight;
    } else {
      this.m_overResizeBottom = 0;
    }
  }
  return newElementHeight;
};

/**
 * Determine what the minimum value for the resizing element is
 * @param {string} dimension - the width or height
 * @param {string} axis - the axis
 * @return {number} the minimum height for the element
 * @private
 */
DvtDataGrid.prototype._getMinValue = function (dimension, axis, isHeaderLabel) {
  var inner;
  var innerDimensionValue;
  var elem = this.m_resizingElement;
  var paddingBorder = this._getCellPaddingBorder(dimension, elem);
  var minCompareValue = paddingBorder;
  if (isHeaderLabel) {
    return Math.max((this.m_utils.isTouchDevice() ?
                              2 * DvtDataGrid.RESIZE_TOUCH_OFFSET : 2 * DvtDataGrid.RESIZE_OFFSET),
                            minCompareValue);
  }
  var level = this.getHeaderCellLevel(elem);
  var depth = this.getHeaderCellDepth(elem);
  var sortable = this.getResources().getMappedAttribute('sortable');
  if (elem.getAttribute(sortable) === 'true') {
    this._setSortContainerSize(this._getSortContainer(elem), paddingBorder);
  }
  if (axis === 'column' && elem.getAttribute(sortable) === 'true') {
      minCompareValue = dimension === 'width' ? this.m_sortContainerWidth : this.m_sortContainerHeight;
  }
  var minValue = Math.max((this.m_utils.isTouchDevice() ?
                            2 * DvtDataGrid.RESIZE_TOUCH_OFFSET : 2 * DvtDataGrid.RESIZE_OFFSET),
                          minCompareValue);
  if ((axis === 'column' &&
       (this.m_columnHeaderLevelCount === 1 ||
        (dimension === 'width' && this.m_columnHeaderLevelCount === level + 1) ||
        (dimension === 'height' && depth === 1))) ||
      (axis === 'row' &&
       (this.m_rowHeaderLevelCount === 1 ||
        (dimension === 'height' && this.m_rowHeaderLevelCount === level + 1) ||
        (dimension === 'width' && depth === 1)))) {
    return minValue;
  }

  var index = this.getHeaderCellIndex(elem);
  var extent = this._getAttribute(this.m_resizingElement.parentNode, 'extent', true);
  var currentDimensionValue = this.getElementDir(elem, dimension);

  if (axis === 'column') {
    if (dimension === 'width') {
      inner = this._getHeaderByIndex(index + (extent - 1), this.m_columnHeaderLevelCount - 1,
                                     this.m_colHeader, this.m_columnHeaderLevelCount,
                                     this.m_startColHeader);
      innerDimensionValue = this.getElementDir(inner, dimension);
    } else {
      innerDimensionValue = this._getHeaderLevelDimension(level + (depth - 1), elem,
                                                          this.m_columnHeaderLevelHeights,
                                                          'height', 1);
    }
  } else if (axis === 'row') {
    if (dimension === 'height') {
      inner = this._getHeaderByIndex(index + (extent - 1), this.m_rowHeaderLevelCount - 1,
                                     this.m_rowHeader, this.m_rowHeaderLevelCount,
                                     this.m_startRowHeader);
      innerDimensionValue = this.getElementDir(inner, dimension);
    } else {
      innerDimensionValue = this._getHeaderLevelDimension(level + (depth - 1), elem,
                                                          this.m_rowHeaderLevelWidths,
                                                          'width', 1);
    }
  }
  return currentDimensionValue - (innerDimensionValue - minValue);
};

DvtDataGrid.prototype._getLabelMinValue = function (dimension) {
  var elem = this.m_resizingElement;
  var paddingBorder = this._getCellPaddingBorder(dimension, elem);
  var minCompareValue = paddingBorder;
  return Math.max((this.m_utils.isTouchDevice() ?
                            2 * DvtDataGrid.RESIZE_TOUCH_OFFSET : 2 * DvtDataGrid.RESIZE_OFFSET),
                          minCompareValue);
};
/**
 * Set the sort container size
 * @param {Element} elem
 * @param {Number} size
 * @returns {Number}
 */
DvtDataGrid.prototype._setSortContainerSize = function (elem, size) {
  this.m_sortContainerWidth = this.getElementWidth(elem)
    + size;
  this.m_sortContainerHeight = this.getElementHeight(elem);
};
/**
 * Get the cell padding + border size along a certain dimenison
 * @param {string} dimension
 * @param {Element} elem
 * @returns {Number}
 */
DvtDataGrid.prototype._getCellPaddingBorder = function (dimension, elem) {
  if (this.m_resizingElementMin == null) {
    var cssExpand = ['top', 'right', 'bottom', 'left'];
    var i = dimension === 'width' ? 1 : 0;
    var val = 0;
    var style = window.getComputedStyle(elem);
    for (; i < 4; i += 2) {
      val += parseFloat(style.getPropertyValue('padding-' + cssExpand[i]));
      val += parseFloat(style.getPropertyValue('border-' + cssExpand[i] + '-width'));
    }
    this.m_resizingElementMin = Math.round(val);
  }
  return this.m_resizingElementMin;
};

/**
 * Manages the databody and scroller sizing when the scrollbars are added and
 * removed scrollbars from the grid. This allows the grid container to maintain
 * size as it renders scrollbars inside rahther than out. Method mimics resizeGrid
 */
DvtDataGrid.prototype.manageResizeScrollbars = function () {
  var width = this.getWidth();
  var height = this.getHeight();
  var colHeader = this.m_colHeader;
  var colEndHeader = this.m_colEndHeader;
  var rowHeader = this.m_rowHeader;
  var rowEndHeader = this.m_rowEndHeader;
  var databody = this.m_databody;
  var databodyScroller = databody.firstChild;

  // cache these since they will be used in multiple places and we want to minimize reflow
  var colHeaderHeight = this.getColumnHeaderHeight();
  var colEndHeaderHeight = this.getColumnEndHeaderHeight();
  var rowHeaderWidth = this.getRowHeaderWidth();
  var rowEndHeaderWidth = this.getRowEndHeaderWidth();

  // adjusted to make the databody wrap the databody content, and the scroller to fill the remaing part of the grid
  // this way our scrollbars are always at the edges of our viewport
  var availableHeight = height - colHeaderHeight - colEndHeaderHeight;
  var availableWidth = width - rowHeaderWidth - rowEndHeaderWidth;

  var scrollbarSize = this.m_utils.getScrollbarSize();
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';
  var isEmpty = this._databodyEmpty();

  // check if there's no data
  if (isEmpty) {
    // could be getting here in the handle resize of an empty grid
    var empty;
    if (this.m_empty == null) {
      empty = this._buildEmptyText();
      this.m_root.appendChild(empty); // @HTMLUpdateOK
    } else {
      empty = this.m_empty;
      this.setElementDir(empty, this.m_endColHeader >= 0 ? colHeaderHeight : 0, 'top');
    }
    var emptyHeight = this.getElementHeight(empty);
    var emptyWidth = this.getElementWidth(empty);

    if (emptyHeight > this.getElementHeight(databodyScroller)) {
      this.setElementHeight(databodyScroller, emptyHeight);
    }
    if (emptyWidth > this.getElementWidth(databodyScroller)) {
      this.setElementWidth(databodyScroller, emptyWidth);
    }
  }

  var databodyContentWidth = this.getElementWidth(databody.firstChild);
  var databodyContentHeight = this.getElementHeight(databody.firstChild);
  // determine which scrollbars are required, if needing one forces need of the other, allows rendering within the root div
  var isDatabodyHorizontalScrollbarRequired =
    this.isDatabodyHorizontalScrollbarRequired(availableWidth);
  var isDatabodyVerticalScrollbarRequired;

  if (isDatabodyHorizontalScrollbarRequired) {
    isDatabodyVerticalScrollbarRequired =
      this.isDatabodyVerticalScrollbarRequired(availableHeight - scrollbarSize);
    databody.style.overflow = 'auto';
  } else {
    isDatabodyVerticalScrollbarRequired =
      this.isDatabodyVerticalScrollbarRequired(availableHeight);
    if (isDatabodyVerticalScrollbarRequired) {
      isDatabodyHorizontalScrollbarRequired =
        this.isDatabodyHorizontalScrollbarRequired(availableWidth - scrollbarSize);
      databody.style.overflow = 'auto';
    } else {
      // for an issue where same size child causes scrollbars (similar code used in resizing already)
      databody.style.overflow = 'hidden';
    }
  }

  this.m_hasHorizontalScroller = isDatabodyHorizontalScrollbarRequired;
  this.m_hasVerticalScroller = isDatabodyVerticalScrollbarRequired;

  var databodyHeight;
  var rowHeaderHeight;
  if (this.m_endColEndHeader !== -1) {
    databodyHeight = Math.min(databodyContentHeight +
                              (isDatabodyHorizontalScrollbarRequired ? scrollbarSize : 0),
                              availableHeight);
    rowHeaderHeight = isDatabodyHorizontalScrollbarRequired ?
      databodyHeight - scrollbarSize : databodyHeight;
  } else {
    databodyHeight = availableHeight;
    rowHeaderHeight = Math.min(databodyContentHeight,
                               isDatabodyHorizontalScrollbarRequired ?
                                 databodyHeight - scrollbarSize : databodyHeight);
  }

  var databodyWidth;
  var columnHeaderWidth;

  if (this.m_endRowEndHeader !== -1) {
    databodyWidth = Math.min(databodyContentWidth +
                             (isDatabodyVerticalScrollbarRequired ? scrollbarSize : 0),
                             availableWidth);
    columnHeaderWidth = isDatabodyVerticalScrollbarRequired ?
      databodyWidth - scrollbarSize : databodyWidth;
  } else {
    databodyWidth = availableWidth;
    columnHeaderWidth = Math.min(databodyContentWidth,
                                 isDatabodyVerticalScrollbarRequired ?
                                   databodyWidth - scrollbarSize : databodyWidth);
  }

  var rowEndHeaderDir = rowHeaderWidth + columnHeaderWidth +
    (isDatabodyVerticalScrollbarRequired ? scrollbarSize : 0);
  var columnEndHeaderDir = colHeaderHeight + rowHeaderHeight +
    (isDatabodyHorizontalScrollbarRequired ? scrollbarSize : 0);

  this.setElementDir(rowHeader, 0, dir);
  this.setElementDir(rowHeader, colHeaderHeight, 'top');
  this.setElementHeight(rowHeader, rowHeaderHeight);

  this.setElementDir(rowEndHeader, rowEndHeaderDir, dir);
  this.setElementDir(rowEndHeader, colHeaderHeight, 'top');
  this.setElementHeight(rowEndHeader, rowHeaderHeight);

  this.setElementDir(colHeader, rowHeaderWidth, dir);
  this.setElementWidth(colHeader, columnHeaderWidth);

  this.setElementDir(colEndHeader, rowHeaderWidth, dir);
  this.setElementDir(colEndHeader, columnEndHeaderDir, 'top');
  this.setElementWidth(colEndHeader, columnHeaderWidth);

  this.setElementDir(databody, colHeaderHeight, 'top');
  this.setElementDir(databody, rowHeaderWidth, dir);
  this.setElementWidth(databody, databodyWidth);
  this.setElementHeight(databody, databodyHeight);

  // cache the scroll width and height to minimize reflow
  this.m_scrollWidth = databodyContentWidth - columnHeaderWidth;
  this.m_scrollHeight = databodyContentHeight - rowHeaderHeight;

  this.buildCorners();

  // check if we need to remove border on the last column header/add borders to headers
  this._adjustHeaderBorders();
  this._updateGridlines();

  // on touch devices the scroller doesn't automatically scroll into view when resizing the last columns or rows to be smaller
  if (this.m_utils.isTouchDevice()) {
    var deltaX = 0;
    var deltaY = 0;

    // if the visible window plus the scrollLeft is bigger than the scrollable region maximum, rescroll the window
    if (this.m_currentScrollLeft > this.m_scrollWidth) {
      deltaX = this.m_scrollWidth - this.m_currentScrollLeft;
    }

    if (this.m_currentScrollTop > this.m_scrollHeight) {
      deltaY = this.m_scrollHeight - this.m_currentScrollTop;
    }

    if (deltaX !== 0 || deltaY !== 0) {
      // eliminate bounce back for touch scroll
      this._disableTouchScrollAnimation();
      var delta = this.adjustTouchScroll(deltaX, deltaY);
      deltaX = delta[0];
      deltaY = delta[1];


      this.scrollDelta(deltaX, deltaY);
    }
  }
};

/**
 * Resizes all cell in the resizing element's column, and updates the left(right)
 * postion on the cells and column headers that follow(preceed) that column.
 * @param {number} widthChange - the change in width of the resizing element
 */
DvtDataGrid.prototype.resizeColumnWidthAndShift = function (widthChange) {
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';
  var colHeaderDisplay = this.m_colHeader.style.display;
  var colEndHeaderDisplay = this.m_colEndHeader.style.display;

  // hide the databody and col header for performance
  this.m_databody.style.display = 'none';
  this.m_colHeader.style.display = 'none';
  this.m_colEndHeader.style.display = 'none';

  // get the index of the header, if it is a nested header make it the last child index
  var index = this.getHeaderCellIndex(this.m_resizingElement);
  if (this.m_columnHeaderLevelCount > 1 &&
      this.m_resizingElement === this.m_resizingElement.parentNode.firstChild &&
      this.m_resizingElement.nextSibling != null) { // has children
    index += this._getAttribute(this.m_resizingElement.parentNode, 'extent', true) - 1;
  }

  var rangeIndex = this.createIndex(-1, index);
  var cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));

  var classArray;

  for (let i = 0; i < cells.length; i++) {
    classArray = this.getResources().isRTLMode() ? ['startResized'] : ['endResized'];
    this._highlightElement(cells[i], classArray);
  }
  // move column headers within the container and adjust the widths appropriately
  this._shiftHeadersAlongAxisInContainer(this.m_colHeader.firstChild, index, widthChange,
                                         dir, this.getMappedStyle('colheadercell'), 'column');

  // move column headers within the container and adjust the widths appropriately
  this._shiftHeadersAlongAxisInContainer(this.m_colEndHeader.firstChild, index, widthChange,
                                         dir, this.getMappedStyle('colendheadercell'), 'column');

  // shift the cells widths and left/right values in the databody
  this._shiftCellsAlongAxis('column', widthChange, index);

  // restore visibility
  this.m_databody.style.display = '';
  this.m_colHeader.style.display = colHeaderDisplay;
  this.m_colEndHeader.style.display = colEndHeaderDisplay;
};

/**
 * Moves cells inside of all rows/columns starting at a certain index, will also resize the given index.
 * @param {string} axis
 * @param {number} dimensionDelta
 * @param {number} startAxisChange
 * @param {boolean|null=} shiftOnly true if only moving cells along an axis by dimensionDelta without resizing
 */
DvtDataGrid.prototype._shiftCellsAlongAxis =
  function (axis, dimensionDelta, startAxisChange, shiftOnly) {
    var tempArray = [];
    var dimension;
    var dir;
    var endAxisChange;
    var startOuterLoop;
    var endOuterLoop;

    if (shiftOnly == null) {
      // eslint-disable-next-line no-param-reassign
      shiftOnly = false;
    }

    if (axis === 'row') {
      dimension = 'height';
      dir = 'top';
      endAxisChange = this.m_endRow;
      startOuterLoop = this.m_startCol;
      endOuterLoop = this.m_endCol;
    } else {
      dimension = 'width';
      dir = this.getResources().isRTLMode() ? 'right' : 'left';
      endAxisChange = this.m_endCol;
      startOuterLoop = this.m_startRow;
      endOuterLoop = this.m_endRow;
    }

    // shift the cells widths and left/right values in the databody
    if (this.m_databody.firstChild != null) {
      for (var i = startOuterLoop; i <= endOuterLoop; i++) {
        // set the new width on the appropriate column
        var index = axis === 'row' ?
          this.createIndex(startAxisChange, i) : this.createIndex(i, startAxisChange);
        var cell = this._getCellByIndex(index);
        var cellContext = cell[this.getResources().getMappedAttribute('context')];
        var cellAxisStartIndex = cellContext.indexes[axis];
        var axisExtent = cellContext.extents[axis];
        var extentWithinCellToIgnore = startAxisChange - cellAxisStartIndex;
        var startAdjustment = (axisExtent - extentWithinCellToIgnore);

        if (!(tempArray[i] && tempArray[i][startAxisChange]) && shiftOnly !== true) {
          this.setElementDir(cell,
                             this.getElementDir(cell, dimension) + dimensionDelta, dimension);
          if (axis === 'row') {
            this._updateTempArray(tempArray, true, i, startAxisChange, cellContext.extents.column,
                                  axisExtent - extentWithinCellToIgnore);
          } else {
            this._updateTempArray(tempArray, true, i, startAxisChange, cellContext.extents.row,
                                  axisExtent - extentWithinCellToIgnore);
          }
        }

        // start value ignore adjustment only if shiftOnly flag is true
        var axisStartValue = startAxisChange;
        if (shiftOnly !== true) {
          axisStartValue = startAxisChange + startAdjustment;
        }

        // move the columns within the data body to account for width change
        for (var j = axisStartValue; j <= endAxisChange; j++) {
          if (!tempArray[i] || !tempArray[i][j]) {
            index = axis === 'row' ? this.createIndex(j, i) : this.createIndex(i, j);
            cell = this._getCellByIndex(index);
            cellContext = cell[this.getResources().getMappedAttribute('context')];
            axisExtent = cellContext.extents[axis];

            var newStart = this.getElementDir(cell, dir) + dimensionDelta;
            this.setElementDir(cell, newStart, dir);
            if (axis === 'row') {
              this._updateTempArray(tempArray, true, i, j, cellContext.extents.column, axisExtent);
            } else {
              this._updateTempArray(tempArray, true, i, j, cellContext.extents.row, axisExtent);
            }
          }
        }
      }
    }
  };

/**
 * Resizes the resizing elements row, and updates the top
 * postion on the rows and row headers that follow that column.
 * @param {number} heightChange - the change in width of the resizing element
 */
DvtDataGrid.prototype.resizeRowHeightAndShift = function (heightChange) {
  var rowHeaderDisplay = this.m_rowHeader.style.display;
  var rowEndHeaderDisplay = this.m_rowEndHeader.style.display;

  // hide the databody and row header for performance
  this.m_databody.style.display = 'none';
  this.m_rowHeader.style.display = 'none';
  this.m_rowEndHeader.style.display = 'none';

  // get the index of the header, if it is a nested header make it the last child index
  var index = this.getHeaderCellIndex(this.m_resizingElement);
  if (this.m_rowHeaderLevelCount > 1
      && this.m_resizingElement === this.m_resizingElement.parentNode.firstChild
      && this.m_resizingElement.nextSibling != null) { // has children)
    index += this._getAttribute(this.m_resizingElement.parentNode, 'extent', true) - 1;
  }

  var rangeIndex = this.createIndex(index, -1);
  var cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));

  var classArray;

  for (let i = 0; i < cells.length; i++) {
    classArray = ['bottomResized'];
    this._highlightElement(cells[i], classArray);
  }

  // move row headers within the container
  this._shiftHeadersAlongAxisInContainer(this.m_rowHeader.firstChild, index, heightChange,
                                         'top', this.getMappedStyle('rowheadercell'), 'row');

  // move row headers within the container
  this._shiftHeadersAlongAxisInContainer(this.m_rowEndHeader.firstChild, index, heightChange,
                                         'top', this.getMappedStyle('rowendheadercell'), 'row');

  // shift the cells hieghts and top values in the databody
  this._shiftCellsAlongAxis('row', heightChange, index);

  this.m_databody.style.display = '';
  this.m_rowHeader.style.display = rowHeaderDisplay;
  this.m_rowEndHeader.style.display = rowEndHeaderDisplay;
};

/**
 * This method recursively shifts a group over until it reaches the index where the group width/height needs to be adjusted
 * @param {Element} headersContainer the header grouping or scroller
 * @param {number|string} index the index that is being adjusted
 * @param {number} dimensionChange the change in width or height
 * @param {string} dir top, left, or right the appropriate value to adjust along the axis
 * @param {string} className the header cell className along that axis
 * @param {string=} axis the axis we are shifting on
 * @private
 */
DvtDataGrid.prototype._shiftHeadersAlongAxisInContainer =
  function (headersContainer, index, dimensionChange, dir, className, axis) {
    var header;
    var groupingContainer;
    var headerStart;
    var newStart;
    var newVal = 0;

    // get the last element in the container
    var element = headersContainer.lastChild;
    if (element == null) {
      return;
    }

    // is the last element a header or a group
    var isHeader = this.m_utils.containsCSSClassName(element, className);
    // what is the index of the container/header
    if (isHeader) {
      groupingContainer = element.parentNode;
      header = element;
      headerStart = this.getHeaderCellIndex(header);
    } else {
      groupingContainer = element;
      header = element.firstChild;
      headerStart = this._getAttribute(groupingContainer, 'start', true);
    }

    // if the group is after the specified index move all the dir values under that group
    while (index < headerStart) {
      if (isHeader) {
        // move this header to the right left up down
        newStart = this.getElementDir(header, dir) + dimensionChange;
        this.setElementDir(header, newStart, dir);

        element = element.previousSibling;
        isHeader = this.m_utils.containsCSSClassName(element, className);
        groupingContainer = element.parentNode;
        header = element;
        headerStart = this.getHeaderCellIndex(header);
      } else {
        // move all children of a group
        var headers = groupingContainer.getElementsByClassName(className);
        for (var i = 0; i < headers.length; i++) {
          newStart = this.getElementDir(headers[i], dir) + dimensionChange;
          this.setElementDir(headers[i], newStart, dir);
        }

        element = element.previousSibling;
        isHeader = this.m_utils.containsCSSClassName(element, className);
        groupingContainer = element;
        header = element.firstChild;
        headerStart = this._getAttribute(groupingContainer, 'start', true);
      }
    }
    var resizingElementLevel;
    if (this.m_resizingElement) {
      resizingElementLevel = this.getHeaderCellLevel(this.m_resizingElement);
    }
    if (axis === 'column') {
      // the last header we moved to should be the one that needs its width updated
      newVal = this.getElementWidth(header) + dimensionChange;
      this.setElementWidth(header, newVal);
    } else if (axis === 'row') {
      newVal = this.getElementHeight(header) + dimensionChange;
      this.setElementHeight(header, newVal);
    } else if (axis == null) {
      newStart = this.getElementDir(header, dir) + dimensionChange;
      this.setElementDir(header, newStart, dir);
    }

    if (axis === 'row' || axis === 'column') {
      let resizeClass = this.getResources().isRTLMode() ? ['startResized'] : ['endResized'];
      let classArray = axis === 'row' ? ['bottomResized'] : resizeClass;
      if (resizingElementLevel !== undefined &&
        this.getHeaderCellLevel(header) >= resizingElementLevel) {
        this._highlightElement(header, classArray);
      }
      if (header === this.m_resizingElement && (header === groupingContainer.lastChild ||
        header === groupingContainer.firstChild)) {
        this._highlightResizeBorder(classArray, axis);
      }
    }
    // if we aren't innermost then repeat for its children
    if (!isHeader && header.nextSibling != null) { // has children
      this._shiftHeadersAlongAxisInContainer(element, index, dimensionChange, dir,
                                             className, axis);
    } else if (axis != null) {
      // store the width/height change in the sizing manager, only care about innermost
      this.m_sizingManager.setSize(axis, this._getKey(header), newVal);
    }
  };

  // eslint-disable-next-line consistent-return
  DvtDataGrid.prototype._highlightResizeBorder = function (classArray, axis) {
    let index = this.getHeaderCellIndex(this.m_resizingElement);
    let levels = axis === 'column' ? this.m_columnHeaderLevelCount : this.m_rowHeaderLevelCount;
    let resizingElementContext = this.m_resizingElement[
      this.getResources().getMappedAttribute('context')];
    let resizingElementEnd = (resizingElementContext.index + resizingElementContext.extent) - 1;
    for (let i = 0; i < levels - 1; i++) {
      let header;
      if (axis === 'row') {
        header = this._getHeaderByIndex(index, i, this.m_rowHeader, levels, this.m_startRowHeader);
      } else {
        header = this._getHeaderByIndex(index, i, this.m_colHeader, levels,
          this.m_startColumnHeader);
      }
      let headerContext = header[this.getResources().getMappedAttribute('context')];
      let headerEnd = (headerContext.index + headerContext.extent) - 1;
      if (headerEnd === resizingElementEnd) {
        this._highlightElement(header, classArray);
      }
    }
  };
/**
 * Resizes all cell in the resizing element's column, and updates the left(right)
 * postion on the cells and column headers that follow(preceed) that column.
 * @param {number} heightChange - the change in width of the resizing element
 * @param {number} level - the level we are resizing
 * @param {boolean} end
 * @param {boolean} adjustLabel
 */
DvtDataGrid.prototype.resizeColumnHeightsAndShift =
  function (heightChange, level, end, adjustLabel) {
    var root;
    var className;
    var axis;
    var dir;

    if (!end) {
      root = this.m_colHeader;
      className = this.getMappedStyle('colheadercell');
      axis = 'column';
      dir = 'top';
    } else {
      root = this.m_colEndHeader;
      className = this.getMappedStyle('colendheadercell');
      axis = 'columnEnd';
      dir = 'bottom';
    }

    root.style.display = 'none';
    this.m_databody.style.display = 'none';
    // move column headers within the container
    if (adjustLabel) {
      this._shiftLabelsDir(this.m_headerLabels[axis], heightChange, level, dir, axis);
    }
    this._shiftHeadersDirInContainer(root.firstChild, heightChange, level, dir, className, axis);
    root.style.display = '';
    this.m_databody.style.display = '';
  };

/**
 * Resizes all cell in the resizing element's column, and updates the left(right)
 * postion on the cells and column headers that follow(preceed) that column.
 * @param {number} widthChange - the change in width of the resizing element
 * @param {number} level - the level we are resizing
 * @param {boolean} end
 */
DvtDataGrid.prototype.resizeRowWidthsAndShift = function (widthChange, level, end) {
  var root;
  var className;
  var axis;
  var dir;

  if (!end) {
    root = this.m_rowHeader;
    className = this.getMappedStyle('rowheadercell');
    axis = 'row';
    dir = this.getResources().isRTLMode() ? 'right' : 'left';
  } else {
    root = this.m_rowEndHeader;
    className = this.getMappedStyle('rowendheadercell');
    axis = 'rowEnd';
    dir = this.getResources().isRTLMode() ? 'left' : 'right';
  }

  root.style.display = 'none';
  this.m_databody.style.display = 'none';
  // move column headers within the container
  this._shiftLabelsDir(this.m_headerLabels[axis], widthChange, level, dir, axis);
  this._shiftHeadersDirInContainer(root.firstChild, widthChange, level, dir, className, axis);
  root.style.display = '';
  this.m_databody.style.display = '';
};

DvtDataGrid.prototype._shiftLabelsDir = function (labels, dimensionChange, level, dir, axis) {
  for (var i = 0; i < labels.length; i++) {
    var label = labels[i];
    var newDir;

    if (i === level) {
      if (axis === 'column' || axis === 'columnEnd') {
        newDir = this.getElementHeight(label) + dimensionChange;
        this.setElementHeight(label, newDir);
        if (axis === 'column' && this._isHeaderLabelCollision() &&
            level === this.m_columnHeaderLevelCount - 1) {
          let classArray = ['bottomResized'];
          this._highlightElement(this.m_corner, classArray);
        } else if (axis === 'columnEnd') {
          let classArray = ['topResized'];
          let columnEndHeaderLabel = this._getLabel('columnEnd',
            this.m_columnEndHeaderLevelCount - 1);
          if (columnEndHeaderLabel) {
            this._highlightElement(columnEndHeaderLabel, classArray);
          }
        } else {
          let classArray = axis === 'column' ? ['bottomResized'] : ['topResized'];
          this._highlightElement(label, classArray);
        }
      } else {
        newDir = this.getElementWidth(label) + dimensionChange;
        this.setElementWidth(label, newDir);
        let rowClass = this.getResources().isRTLMode() ? ['startResized'] : ['endResized'];
        let rowEndClass = this.getResources().isRTLMode() ? ['endResized'] : ['startResized'];
        let classArray = axis === 'row' ? rowClass : rowEndClass;
        this._highlightElement(label, classArray);
        let columnEndHeaderLabel = this._getLabel('columnEnd',
          this.m_columnEndHeaderLevelCount - 1);
        if (columnEndHeaderLabel) {
          this._highlightElement(columnEndHeaderLabel.parentNode, classArray);
        }
      }
    }
    if (i > level) {
      newDir = this.getElementDir(label, dir) + dimensionChange;
      this.setElementDir(label, newDir, dir);
    }
  }
};

/**
 * Shifts the headers after a particular level over and adjusts the dimension of that level across the whole container
 * @param {Element} headersContainer
 * @param {number} dimensionChange
 * @param {number} level
 * @param {string} dir
 * @param {string} className
 * @param {string} axis
 * @private
 */
DvtDataGrid.prototype._shiftHeadersDirInContainer =
  function (headersContainer, dimensionChange, level, dir, className, axis) {
    var groupings = headersContainer.childNodes;
    // for all children in the group
    for (var i = 0; i < groupings.length; i++) {
      var grouping = groupings[i];
      var isHeader = this.m_utils.containsCSSClassName(grouping, className);
      var headerLevel;
      var newDir;

      // if it is a group
      if (!isHeader) {
        headerLevel = this._getAttribute(grouping, 'level', true);
        // if before or on the level we need to go deeper into the grouping
        if (headerLevel <= level) {
          this._shiftHeadersDirInContainer(grouping, dimensionChange, level, dir, className, axis);
        } else {
          // if level is higher then we need to adjust the dir of all the headers under that group
          var headers = grouping.getElementsByClassName(className);
          for (var j = 0; j < headers.length; j++) {
            newDir = this.getElementDir(headers[j], dir) + dimensionChange;
            this.setElementDir(headers[j], newDir, dir);
          }
        }
      } else {
        headerLevel = this.getHeaderCellLevel(grouping);
        var headerDepth = this.getHeaderCellDepth(grouping);

        // if we have a header at that level adjust it's value
        if (headerLevel <= level && level < headerLevel + headerDepth) {
          if (axis === 'column' || axis === 'columnEnd') {
            newDir = this.getElementHeight(grouping) + dimensionChange;
            this.setElementHeight(grouping, newDir);
            if (this.m_resizingElement) {
              var endColHeaderClassName = this.getMappedStyle('colendheadercell');
              if (this.m_utils.containsCSSClassName(grouping, endColHeaderClassName)) {
                let classArray = ['topResized'];
                this._highlightElement(grouping, classArray);
              } else {
                let classArray = ['bottomResized'];
                this._highlightElement(grouping, classArray);
              }
            }
          } else {
            newDir = this.getElementWidth(grouping) + dimensionChange;
            this.setElementWidth(grouping, newDir);
            if (this.m_resizingElement) {
              var endRowHeaderClassName = this.getMappedStyle('rowendheadercell');
              if (this.m_utils.containsCSSClassName(grouping, endRowHeaderClassName)) {
                let classArray = this.getResources().isRTLMode() ? ['endResized'] : ['startResized'];
                this._highlightElement(grouping, classArray);
              } else {
                let classArray = this.getResources().isRTLMode() ? ['startResized'] : ['endResized'];
                this._highlightElement(grouping, classArray);
              }
            }
          }
        } else if (headerLevel > level) {
          // if we have a header inside the group then adjust its dimension
          newDir = this.getElementDir(grouping, dir) + dimensionChange;
          this.setElementDir(grouping, newDir, dir);
        }
      }
    }
  };

/**
 * Takes the original target of the context menu and maps it to the appropriate
 * column/row header to resize and selects the right resize function.
 * @param {Event} event - the event that spawned context menu
 * @param {string} id - 'width' or 'height'
 * @param {string} val - new width or height to resize to
 * @param {Element|undefined} target - the target element
 */
DvtDataGrid.prototype.handleContextMenuResize = function (event, id, val, target) {
  this.m_overResizeLeft = 0;
  this.m_overResizeMinLeft = 0;
  this.m_overResizeTop = 0;
  this.m_overResizeMinTop = 0;
  this.m_overResizeRight = 0;
  this.m_overResizeBottom = 0;

  var value = parseFloat(val, 10);

  var deltaWidth = value - this.getElementWidth(target);
  var deltaHeight = value - this.getElementHeight(target);

  if (this.m_utils.containsCSSClassName(target, this.getMappedStyle('cell'))) {
    if (id === this.m_resources.getMappedCommand('resizeHeight')) {
      // eslint-disable-next-line no-param-reassign
      target = this.getHeaderFromCell(target, 'row', true);
    } else {
      // eslint-disable-next-line no-param-reassign
      target = this.getHeaderFromCell(target, 'column', true);
    }
  }

  this.m_resizingElement = target;
  let isHeaderLabel = this.find(this.m_resizingElement, 'headerlabel');
  var initialWidth = this.getElementWidth(target);
  var initialHeight = this.getElementHeight(target);
  var end = this.m_utils.containsCSSClassName(this.m_resizingElement,
                                              this.getMappedStyle('endheadercell')) ||
            this.m_utils.containsCSSClassName(this.m_resizingElement,
                                              this.getMappedStyle('columnendheaderlabel')) ||
            this.m_utils.containsCSSClassName(this.m_resizingElement,
                                              this.getMappedStyle('rowendheaderlabel'));

  if (id === this.m_resources.getMappedCommand('resizeWidth')) {
    if (initialWidth !== value) {
      if (this._getResizeHeaderMode(this.m_resizingElement) === 'column') {
        if (this._isDOMElementResizable(this.m_resizingElement)) {
          if (isHeaderLabel) {
            this.resizeRowWidth(value, value - initialWidth, end, isHeaderLabel);
            this._fireResizeEvent(event, initialWidth, initialHeight, value, initialHeight, value);
          } else {
            let newWidth = this.getNewElementWidth('column', initialWidth, end,
                                                    deltaWidth, isHeaderLabel);
            this.resizeColWidth(initialWidth, newWidth);
            this._resizeSelectedHeaders(event, initialWidth, initialHeight, value,
                                        initialHeight, value);
          }
        }
      } else {
        this.resizeRowWidth(value, value - initialWidth, end, isHeaderLabel);
        this._fireResizeEvent(event, initialWidth, initialHeight, value, initialHeight, value);
      }
    }
  } else if (id === this.m_resources.getMappedCommand('resizeHeight')) {
    if (initialHeight !== value) {
      if (this._getResizeHeaderMode(this.m_resizingElement) === 'column') {
        this.resizeColHeight(value, value - initialHeight, end);
        this._fireResizeEvent(event, initialWidth, initialHeight, initialWidth, value, value);
      } else if (this._isDOMElementResizable(this.m_resizingElement)) {
        let newElementHeight = this.getNewElementHeight('row', initialHeight, end, deltaHeight, isHeaderLabel);
        if (isHeaderLabel) {
          this.resizeColHeight(newElementHeight, newElementHeight - initialHeight, end);
          this._fireResizeEvent(event, initialWidth, initialHeight, initialWidth, value, value);
        } else {
          this.resizeRowHeight(initialHeight, newElementHeight);
          this._resizeSelectedHeaders(event, initialWidth, initialHeight,
                                      initialWidth, newElementHeight, value);
        }
      }
    }
  }

  var newWidth = this.getElementWidth(target);
  var newHeight = this.getElementHeight(target);
  if (newWidth !== initialWidth || newHeight !== initialHeight) {
    this.buildCorners();
    // re-align touch affordances
    if (this.m_utils.isTouchDevice()) {
      this._moveTouchSelectionAffordance();
    }
  }

  this._unhighlightResizeBorderColor();
  this.m_resizingElement = null;
  this.m_resizingElementMin = null;
};

DvtDataGrid.prototype._getResizeNestedHeaderIndex = function (axis, end) {
  let headerIndex = this.getHeaderCellIndex(this.m_resizingElement);
  let columnHeaderLevelCount = end ? this.m_columnEndHeaderLevelCount :
                                      this.m_columnHeaderLevelCount;
  let rowHeaderLevelCount = end ? this.m_rowEndHeaderLevelCount :
                                  this.m_rowHeaderLevelCount;
  let headerLevelCount = (axis === 'column' || axis === 'columnEnd') ?
                          columnHeaderLevelCount : rowHeaderLevelCount;
  if (headerLevelCount > 1 &&
      this.m_resizingElement === this.m_resizingElement.parentNode.firstChild &&
      this.m_resizingElement.nextSibling != null) { // has children
    headerIndex += this._getAttribute(this.m_resizingElement.parentNode, 'extent', true) - 1;
  }
  return headerIndex;
};

DvtDataGrid.prototype._highlightResizeMouseDown = function () {
  let index = this.getHeaderCellIndex(this.m_resizingElement);
  let resizeHeaderMode = this._getResizeHeaderMode(this.m_resizingElement);

  let isEndHeader = this.m_utils.containsCSSClassName(this.m_resizingElement,
                                              this.getMappedStyle('endheadercell')) ||
            this.m_utils.containsCSSClassName(this.m_resizingElement,
                                              this.getMappedStyle('columnendheaderlabel'));
  let level;
  let axis;
  let dir = this.getResources().isRTLMode() ? 'right' : 'left';
  let isHeaderLabel = this.find(this.m_resizingElement, 'headerlabel');

  if (this.m_cursor === 'col-resize') {
    if (resizeHeaderMode === 'column') {
      isEndHeader = isHeaderLabel ? false : isEndHeader;
      if (isHeaderLabel) {
        axis = this.getHeaderLabelAxis(this.m_resizingElement);
        if (axis === 'column' || axis === 'columnEnd') {
          level = this.m_rowHeaderLevelCount - 1;
        } else if (axis === 'row' || axis === 'rowEnd') {
          level = this.getHeaderLabelLevel(this.m_resizingElement);
        }

        this.resizeRowWidthsAndShift(0, level, isEndHeader);

        if (!isEndHeader) {
          if (level === this.m_rowHeaderLevelCount - 1 ||
            (this.m_headerLabels.row.length === 0 && this.m_headerLabels.column.length)) {
            this._highlightResizeLabelDirs('column', level);
          }
        }
      } else {
        if (this.m_columnHeaderLevelCount > 1 &&
            this.m_resizingElement === this.m_resizingElement.parentNode.firstChild &&
            this.m_resizingElement.nextSibling != null) { // has children
          index += this._getAttribute(this.m_resizingElement.parentNode, 'extent', true) - 1;
        }

        let rangeIndex = this.createIndex(-1, index);
        let cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));

        let classArray;

        for (let i = 0; i < cells.length; i++) {
          classArray = this.getResources().isRTLMode() ? ['startResized'] : ['endResized'];
          this._highlightElement(cells[i], classArray);
        }
        this._shiftHeadersAlongAxisInContainer(this.m_colHeader.firstChild, index, 0,
                                          dir, this.getMappedStyle('colheadercell'), 'column');

        // move column headers within the container and adjust the widths appropriately
        this._shiftHeadersAlongAxisInContainer(this.m_colEndHeader.firstChild, index, 0,
                                          dir, this.getMappedStyle('colendheadercell'), 'column');
      }
    } else if (resizeHeaderMode === 'row') {
      if (this.m_utils.containsCSSClassName(this.m_resizingElement,
        this.getMappedStyle('rowendheaderlabel'))) {
          isEndHeader = true;
      }
      if (isHeaderLabel) {
        axis = this.getHeaderLabelAxis(this.m_resizingElement);
        if (axis === 'column' || axis === 'columnEnd') {
          level = this.m_rowHeaderLevelCount - 1;
        } else if (axis === 'row' || axis === 'rowEnd') {
          level = this.getHeaderLabelLevel(this.m_resizingElement);
        }
      } else {
        level = this.getHeaderCellLevel(this.m_resizingElement) +
        (this.getHeaderCellDepth(this.m_resizingElement) - 1);
      }

      this.resizeRowWidthsAndShift(0, level, isEndHeader);

      if (!isEndHeader) {
        if (level === this.m_rowHeaderLevelCount - 1 ||
          (this.m_headerLabels.row.length === 0 && this.m_headerLabels.column.length)) {
          this._highlightResizeLabelDirs('column', level);
        }
      }
    }
  } else if (this.m_cursor === 'row-resize') {
    if (resizeHeaderMode === 'row') {
      if (isHeaderLabel) {
        let adjustLabel = true;
        // if (isHeaderLabel) {
        axis = this.getHeaderLabelAxis(this.m_resizingElement);
        if (axis === 'column' || axis === 'columnEnd') {
          level = this.getHeaderLabelLevel(this.m_resizingElement);
        } else if (axis === 'row' || axis === 'rowEnd') {
          level = this.m_columnHeaderLevelCount - 1;
          adjustLabel = false;
        }
        if (!isEndHeader && level === this.m_columnHeaderLevelCount - 1 &&
          this.m_headerLabels.row.length &&
          this._isHeaderLabelCollision()) {
            adjustLabel = false;
        }
        this.resizeColumnHeightsAndShift(0, level, isEndHeader, adjustLabel);
        if (!isEndHeader) {
          if (this.m_headerLabels.column.length === 0) {
            this._highlightResizeLabelDirs('row', level);
          } else if (level === this.m_columnHeaderLevelCount - 1 &&
                      this.m_headerLabels.row.length) {
            this._highlightResizeLabelDirs('row', level);
          }
        }
      } else {
        if (this.m_rowHeaderLevelCount > 1
          && this.m_resizingElement === this.m_resizingElement.parentNode.firstChild
          && this.m_resizingElement.nextSibling != null) { // has children)
          index += this._getAttribute(this.m_resizingElement.parentNode, 'extent', true) - 1;
        }

        let rangeIndex = this.createIndex(index, -1);
        let cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));

        let classArray;

        for (let i = 0; i < cells.length; i++) {
          classArray = ['bottomResized'];
          this._highlightElement(cells[i], classArray);
        }
        this._shiftHeadersAlongAxisInContainer(this.m_rowHeader.firstChild, index, 0,
                                          'top', this.getMappedStyle('rowheadercell'), 'row');
        this._shiftHeadersAlongAxisInContainer(this.m_rowEndHeader.firstChild, index, 0,
                                          'top', this.getMappedStyle('rowendheadercell'), 'row');
      }
    } else {
      let adjustLabel = true;
      if (isHeaderLabel) {
        axis = this.getHeaderLabelAxis(this.m_resizingElement);
        if (axis === 'column' || axis === 'columnEnd') {
          level = this.getHeaderLabelLevel(this.m_resizingElement);
        } else if (axis === 'row' || axis === 'rowEnd') {
          level = this.m_columnHeaderLevelCount - 1;
          adjustLabel = false;
        }
      } else {
        level = this.getHeaderCellLevel(this.m_resizingElement) +
        (this.getHeaderCellDepth(this.m_resizingElement) - 1);
        axis = this.getHeaderCellAxis(this.m_resizingElement);
      }

      if (!isEndHeader && level === this.m_columnHeaderLevelCount - 1 &&
        this.m_headerLabels.row.length &&
        this._isHeaderLabelCollision()) {
          adjustLabel = false;
      }
      this.resizeColumnHeightsAndShift(0, level, isEndHeader, adjustLabel);
      if (!isEndHeader) {
        if (this.m_headerLabels.column.length === 0) {
          this._highlightResizeLabelDirs('row', level);
        } else if (level === this.m_columnHeaderLevelCount - 1 && this.m_headerLabels.row.length) {
          this._highlightResizeLabelDirs('row', level);
        }
      }
    }
  }
};

/**
 * Handle height and width resize to fit to content.
 * @param {Event} event - the event that spawned context menu
 * @param {Element|undefined} target - the target element
 * @param {string} resizeAxis - resizing axis
 */

DvtDataGrid.prototype.handleResizeFitToContent = function (event, target, resizeAxis) {
  let headerCell = this.find(target, 'header') || this.find(target, 'endheadercell');
  let endHeaderCell = this.m_utils.containsCSSClassName(target,
    this.getMappedStyle('endheadercell'));
  if (!headerCell) {
    let cell = this.findCell(target);
    if (cell) {
      // eslint-disable-next-line no-param-reassign
      target = this.getHeaderFromCell(cell, resizeAxis);
    } else {
      return;
    }
  }
  this.m_resizingElement = target;
  let header = this.m_resizingElement;
  let axis = this.getHeaderCellAxis(header);
  let cells;
  if (axis === 'column' || axis === 'columnEnd') {
    let index = headerCell ? this._getResizeNestedHeaderIndex(axis, endHeaderCell) :
                              this.getHeaderCellIndex(header);
    if (!endHeaderCell) {
      this.m_resizingElement = this._getHeaderByIndex(index,
        this.m_columnHeaderLevelCount - 1,
        this.m_colHeader,
        this.m_columnHeaderLevelCount,
        this.m_startColHeader);
    } else {
      this.m_resizingElement = this._getHeaderByIndex(index,
        this.m_columnEndHeaderLevelCount - 1,
        this.m_colEndHeader,
        this.m_columnEndHeaderLevelCount,
        this.m_startColEndHeader);
    }
    let oldElementWidth = this.calculateColumnHeaderWidth(this.m_resizingElement);
    let rangeIndex = this.createIndex(-1, index);
    cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));
    cells.push(header);
    let newElementWidth = this._calculateResizeFitToContentValue(cells, 'column');
    this.resizeColWidth(oldElementWidth, newElementWidth);
  } else if (axis === 'row' || axis === 'rowEnd') {
    let index = headerCell ? this._getResizeNestedHeaderIndex(axis, endHeaderCell) :
                              this.getHeaderCellIndex(header);
    if (!endHeaderCell) {
      this.m_resizingElement = this._getHeaderByIndex(index,
        this.m_rowHeaderLevelCount - 1,
        this.m_rowHeader,
        this.m_rowHeaderLevelCount,
        this.m_startRowHeader);
    } else {
      this.m_resizingElement = this._getHeaderByIndex(index,
        this.m_rowEndHeaderLevelCount - 1,
        this.m_rowEndHeader,
        this.m_rowEndHeaderLevelCount,
        this.m_startRowEndHeader);
    }
    let oldElementHeight = this.calculateRowHeaderHeight(this.m_resizingElement);
    let rangeIndex = this.createIndex(index, -1);
    cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));
    cells.push(header);
    let newElementHeight = this._calculateResizeFitToContentValue(cells, 'row');
    this.resizeRowHeight(oldElementHeight, newElementHeight);
  }
  this._unhighlightResizeBorderColor();
};

/**
 * Calculate resize value according to dimension
 * @param {Array} cells - dimension of cells that needs to be adjusted.
 * @param {string} axis - Axis along which resizing has to happen.
 * @return {number} - adjusted value to be set.
 * @private
 */

DvtDataGrid.prototype._calculateResizeFitToContentValue = function (cells, axis) {
  let isHeaderLabel = false;
  let dimension = 'height';
  let resizeValue;
  if (axis === 'column') {
    dimension = 'width';
  }
  let minValue = this._getMinValue(dimension, axis, isHeaderLabel);
  let container = document.createElement('div');
  if (dimension === 'width') {
    container.style.display = 'inline-flex';
    container.style.flexFlow = 'column nowrap';
    container.style.justifyContent = 'flex-start';
    container.style.alignItems = 'stretch';
    for (let i = 0; i < cells.length; i++) {
      let clone = cells[i].cloneNode(true);
      clone.classList.remove(...clone.classList);
      clone.style[dimension] = '';
      clone.style.whiteSpace = 'nowrap !important';
      clone.style.overflow = 'hidden';
      container.appendChild(clone);
    }
  } else if (dimension === 'height') {
    container.style.display = 'flex';
    for (let i = 0; i < cells.length; i++) {
      let clone = cells[i].cloneNode(true);
      clone.style[dimension] = '';
      clone.style.whiteSpace = 'break-spaces';
      container.appendChild(clone);
    }
  }
  container.style.visibility = 'hidden';
  container.style.top = '0px';
  this.m_root.appendChild(container); // @HTMLUpdateOK
  let maxValue = axis === 'column' ? this.m_databody.offsetWidth : this.m_databody.offsetHeight;
  let paddingBorder = axis === 'column' ? this._getCellPaddingBorder(dimension, this.m_resizingElement) : 0;
  let containerElementDimension = axis === 'column' ?
      Math.ceil(container.firstElementChild.getBoundingClientRect().width) :
      Math.ceil(container.firstElementChild.getBoundingClientRect().height);
  resizeValue = containerElementDimension + paddingBorder;
  resizeValue = resizeValue < minValue ? minValue : resizeValue;
  resizeValue = resizeValue > maxValue ? maxValue : resizeValue;
  this.m_root.removeChild(container);
  return resizeValue;
};

/**
 * Get the edges (left,right,top,bottom) pixel locations relative to the element
 * @param {Element} elem - the element to find edges of
 * @return {Array.<number>} An array of numbers [leftEdge, topEdge, rightEdge, bottomEdge]
 */
DvtDataGrid.prototype.getHeaderEdgePixels = function (elem) {
  var leftEdge = 0;
  var topEdge = 0;
  if (this.m_utils.isTouchDevice()) {
    var elementXY = this.findPos(elem);
    leftEdge = elementXY[0];
    topEdge = elementXY[1];
  }
  var targetWidth;
  var targetHeight;

  if (this.m_utils.containsCSSClassName(elem, this.getMappedStyle('colheadercell'))) {
    targetWidth = this.calculateColumnHeaderWidth(elem);
    targetHeight = this.getElementHeight(elem);
  } else {
    targetWidth = this.getElementWidth(elem);
    targetHeight = this.calculateRowHeaderHeight(elem);
  }

  var rightEdge = leftEdge + targetWidth;
  var bottomEdge = topEdge + targetHeight;
  return [leftEdge, topEdge, rightEdge, bottomEdge];
};

DvtDataGrid.prototype.getHeaderLabelEdgePixels = function (elem) {
  var elementXY = this.findPos(elem);
  var leftEdge = elementXY[0];
  var topEdge = elementXY[1];
  var targetWidth;
  var targetHeight;

  if (this.m_utils.containsCSSClassName(elem, this.getMappedStyle('columnheaderlabel'))) {
    targetWidth = this.calculateColumnHeaderWidth(elem);
    targetHeight = this.getElementHeight(elem);
  } else {
    targetWidth = this.getElementWidth(elem);
    targetHeight = this.calculateRowHeaderLabelHeight(elem);
  }

  var rightEdge = leftEdge + targetWidth;
  var bottomEdge = topEdge + targetHeight;
  return [leftEdge, topEdge, rightEdge, bottomEdge];
};

/**
 * Unhighlights the selection.  Does not change selection, focus cell, anchor, or frontier
 */
DvtDataGrid.prototype.unhighlightSelection = function () {
  var ranges = this.GetSelection();
  for (var i = 0; i < ranges.length; i += 1) {
    this.unhighlightRange(ranges[i]);
  }
  if (this.getResources()) {
    this._clearHeaderHighLight();
  }
};

/**
 * Unhighlights the range.
 * @param {Object} range
 */
DvtDataGrid.prototype.unhighlightRange = function (range) {
  var elems = this.getElementsInRange(range);
  this.unhighlightElems(elems);
  this._applyBorderClassesAroundRange(elems, range, false, 'Selected');
};

/**
 * Highlights the range.
 * @param {Object} range
 * @param {boolean=} updateAccInfo
 */
DvtDataGrid.prototype.highlightRange = function (range, updateAccInfo) {
  const rowHeadersInRange = this.getHeadersByRange(range, 'row');
  const colHeadersInRange = this.getHeadersByRange(range, 'column');
  this._highlightHeaders(range, rowHeadersInRange, colHeadersInRange);

  var elems = this.getElementsInRange(range);
  this.highlightElems(elems, range);
  this._applyBorderClassesAroundRange(elems, range, true, 'Selected');

  if (updateAccInfo) {
    var count;

    // if there's islands of cells, then we'll have to count them
    if (this.GetSelection().length === 1) {
      count = elems.length;
    } else {
      count = this._getCurrentSelectionCellCount();
    }
    this._setAccInfoText('accessibleMultiCellSelected', { num: count });
  }
};

/**
 * Returns the headers for given range and axis.
 * @param {Object} range
 * @param {String} axis
 */
DvtDataGrid.prototype.getHeadersByRange = function (range, axis) {
  const headersInRange = new Set();
  let headers;
  let i;
  let j;
  let endIndex;
  if (axis === 'row' && this.m_rowHeaderLevelCount > 0) {
    if (range.endIndex && range.endIndex.row) {
      endIndex = range.endIndex.row;
    }
    if (!endIndex) {
      endIndex = range.startIndex.row;
    } else if (endIndex === -1) {
      endIndex = this.m_endRowHeader;
    }
    let rangeStartRow = Math.max(this.m_startRowHeader, range.startIndex.row);
    for (i = rangeStartRow; i <= endIndex; i++) {
      headers = this._getHeadersByIndex(i, this.m_rowHeader,
        this.m_rowHeaderLevelCount, this.m_startRowHeader);
      for (j = 0; j < headers.length; j++) {
        headersInRange.add(headers[j]);
      }
    }
  } else if (axis === 'column' && this.m_columnHeaderLevelCount > 0) {
    if (range.endIndex && range.endIndex.column) {
      endIndex = range.endIndex.column;
    }
    if (!endIndex) {
      endIndex = range.startIndex.column;
    } else if (endIndex === -1) {
      endIndex = this.m_endColHeader;
    }
    let rangeStartColumn = Math.max(this.m_startColHeader, range.startIndex.column);
    for (i = rangeStartColumn; i <= endIndex; i++) {
      headers = this._getHeadersByIndex(i, this.m_colHeader,
        this.m_columnHeaderLevelCount, this.m_startColHeader);
      for (j = 0; j < headers.length; j++) {
        headersInRange.add(headers[j]);
      }
    }
  }
  if (axis === 'row' && this.m_rowEndHeaderLevelCount > 0) {
    if (range.endIndex && range.endIndex.row) {
      endIndex = range.endIndex.row;
    }
    if (!endIndex) {
      endIndex = range.startIndex.row;
    } else if (endIndex === -1) {
      endIndex = this.m_endRowHeader;
    }
    let rangeStartRow = Math.max(this.m_startRowEndHeader, range.startIndex.row);
    for (i = rangeStartRow; i <= endIndex; i++) {
      headers = this._getHeadersByIndex(i, this.m_rowEndHeader,
        this.m_rowEndHeaderLevelCount, this.m_startRowEndHeader);
      for (j = 0; j < headers.length; j++) {
        headersInRange.add(headers[j]);
      }
    }
  } else if (axis === 'column' && this.m_columnEndHeaderLevelCount > 0) {
    if (range.endIndex && range.endIndex.column) {
      endIndex = range.endIndex.column;
    }
    if (!endIndex) {
      endIndex = range.startIndex.column;
    } else if (endIndex === -1) {
      endIndex = this.m_endColHeader;
    }
    let rangeStartColumn = Math.max(this.m_startColEndHeader, range.startIndex.column);
    for (i = rangeStartColumn; i <= endIndex; i++) {
      headers = this._getHeadersByIndex(i, this.m_colEndHeader,
        this.m_columnEndHeaderLevelCount, this.m_startColEndHeader);
      for (j = 0; j < headers.length; j++) {
        headersInRange.add(headers[j]);
      }
    }
  }
  return headersInRange;
};

/**
 * Highlight headers
 * @param {Array} elem
 */
DvtDataGrid.prototype._highlightHeaders = function (range, rowHeadersInRange, colHeadersInRange) {
  rowHeadersInRange.forEach((element) => {
    const context = element[this.getResources().getMappedAttribute('context')];
    if (!range.endIndex.column ||
      (range.endIndex.row === -1 && range.endIndex.column === -1)
      || ((range.endIndex.column === -1 && range.startIndex.column === 0)
        && this._isHeaderSelected(context, 'row'))) {
      element.classList.remove(this.getMappedStyle('headerPartialSelected'));
      element.classList.add(this.getMappedStyle('headerAllSelected'));
    } else {
      element.classList.remove(this.getMappedStyle('headerAllSelected'));
      element.classList.add(this.getMappedStyle('headerPartialSelected'));
    }
  });
  colHeadersInRange.forEach((element) => {
    const context = element[this.getResources().getMappedAttribute('context')];
    if ((range.endIndex.row === -1 && range.endIndex.column === -1)
      || ((range.endIndex.row === -1 && range.startIndex.row === 0)
        && this._isHeaderSelected(context, 'column'))) {
      element.classList.remove(this.getMappedStyle('headerPartialSelected'));
      element.classList.add(this.getMappedStyle('headerAllSelected'));
    } else {
      element.classList.remove(this.getMappedStyle('headerAllSelected'));
      element.classList.add(this.getMappedStyle('headerPartialSelected'));
    }
  });
};

/**
 * Checks to see if context is contained by a selection block.
 * @param {Object} context context of item to see if is contained by selection.
 */
DvtDataGrid.prototype._isHeaderSelected = function (context, axis) {
  let selected = false;
  this.GetSelection().forEach((block) => {
    if ((context.index >= block.startIndex[axis])
    && (context.index + (context.extent - 1)) <= block.endIndex[axis]) {
      selected = true;
    }
  });
  return selected;
};

/**
 * Clear all header highlight
 */
 DvtDataGrid.prototype._clearHeaderHighLight = function () {
  const someSelectedHeaders = this.m_root.querySelectorAll('.' + this.getMappedStyle('headerPartialSelected'));
  for (let i = 0; i < someSelectedHeaders.length; i++) {
    someSelectedHeaders[i].classList.remove(this.getMappedStyle('headerPartialSelected'));
  }
  const allSelectedHeaders = this.m_root.querySelectorAll('.' + this.getMappedStyle('headerAllSelected'));
  for (let i = 0; i < allSelectedHeaders.length; i++) {
    allSelectedHeaders[i].classList.remove(this.getMappedStyle('headerAllSelected'));
  }
};
/**
 * Calculate the total number of cells within the current selection ranges.
 * @private
 */
DvtDataGrid.prototype._getCurrentSelectionCellCount = function () {
  var total = 0;
  var selection = this.GetSelection();
  for (var i = 0; i < selection.length; i++) {
    // count the number of elements in each selection range
    var elems = this.getElementsInRange(selection[i]);
    if (elems != null) {
      total += elems.length;
    }
  }

  return total;
};

/**
 * Unhighlight elements
 * @param {Array} elems
 */
DvtDataGrid.prototype.unhighlightElems = function (elems) {
  if (elems == null || elems.length === 0) {
    return;
  }

  for (let i = 0; i < elems.length; i += 1) {
    let elem = elems[i];
    let classArray = ['selected',
      'topSelected', 'bottomSelected',
      'startSelected', 'endSelected'
    ];
    this._unhighlightElement(elem, classArray);
  }
};

/**
 * Highlight elements
 * @param {Array} elems
 */
DvtDataGrid.prototype.highlightElems = function (elems) {
  if (elems == null || elems.length === 0) {
    return;
  }
  for (let i = 0; i < elems.length; i += 1) {
    this._highlightElement(elems[i], ['selected']);
  }
};

/**
 * Apply current selection to a range.  This is called when a newly set of cells are
 * rendered and selection needs to be applied on them.
 * @param {number=} startRow
 * @param {number=} endRow
 * @param {number=} startCol
 * @param {number=} endCol
 */
DvtDataGrid.prototype.applySelection = function (startRow, endRow, startCol, endCol) {
  var ranges = this.GetSelection();
  for (var i = 0; i < ranges.length; i += 1) {
    var elems = this.getElementsInRange(ranges[i], startRow, endRow, startCol, endCol);
    this.highlightElems(elems);
    this._applyBorderClassesAroundRange(elems, ranges[i], true, 'Selected');
  }
};

/**
 * Handles click and drag to select multiple cells/rows
 * @param {Event} event
 */
DvtDataGrid.prototype.handleDatabodySelectionDrag = function (event) {
  var cell;
  var target;

  if (this.m_utils.isTouchDevice()) {
    cell = this.findCell(document.elementFromPoint(event.touches[0].clientX,
                                                   event.touches[0].clientY));
  } else {
    target = /** @type {Element} */ (event.target);
    cell = this.findCell(target);
  }

  if (cell != null) {
    var index = this.getCellIndexes(cell);
    if (this.m_deselectInProgress) {
      this.extendDeselection(index, event);
    } else if (this.isHeaderSelectionType(this.m_selectionFrontier)) {
      this.extendSelectionHeader(cell, event);
    } else {
      this.extendSelection(index, event);
    }
  }
};

/**
 * Handles click to select header rows/columns
 * @param {Event} event
 */
DvtDataGrid.prototype.handleHeaderClickSelection = function (event) {
  var target = /** @type {Element} */ (event.target);
  var header = this.findHeader(target);
  var shiftKey = event.shiftKey;
  var multi = this.isMultipleSelection();
  var ctrlKey = this.m_utils.ctrlEquivalent(event);

  if (!(shiftKey && multi) && header) {
    this._setActive(header, this._createActiveObject(header), event);
  }

  // no selection if header was not selected or if cell is active and shiftkey is not selected when selectin gheader
  if (this.m_active == null || (this.m_active.type !== 'header' && !event.shiftKey)) {
    return false;
  }

  var axis;
  var index;
  var level;
  var extent;

  var headerContext = header[this.getResources().getMappedAttribute('context')];
  axis = headerContext.axis;
  index = headerContext.index;
  extent = headerContext.extent;

  if (ctrlKey && this._shouldDeselectHeader(index, extent, axis)) {
    var start;
    var end;
    var range;
    var returnObj;
    if (axis.indexOf('row') !== -1) {
      start = this.createIndex(index, 0);
      end = this.createIndex((index + extent) - 1, -1);
      returnObj = this._getSelectionStartAndEnd(start, end, 0);
      range = this.createRange(
        this.createIndex(returnObj.min.row, 0),
        this.createIndex(returnObj.max.row, -1));
    } else {
      start = this.createIndex(0, index);
      end = this.createIndex(-1, (index + extent) - 1);
      returnObj = this._getSelectionStartAndEnd(start, end, 0);
      range = this.createRange(
        this.createIndex(0, returnObj.min.column),
        this.createIndex(-1, returnObj.max.column));
    }
    var trimmedRange = this._trimRangeForSelectionMode(range);
    this.m_deselectInfo = {
      anchor: trimmedRange.startIndex,
      selection: this.GetSelection(),
      axis: axis.indexOf('row') !== -1 ? 'row' : 'column',
      sourceParent: header };
    this.m_deselectInProgress = this._deselectRange(trimmedRange, event);
    return undefined;
  }

  if (this.m_active.type === 'header') {
    axis = this.m_active.axis;
    index = this.m_active.index;
    level = this.m_active.level;
  }

  if (this.m_utils.isTouchDevice()) {
    // remove the touch affordance on a new tap, unhighlight the active cell, and select the new one
    this._removeTouchSelectionAffordance();
    this._selectHeader(axis, index, level, event);
  } else if (shiftKey && multi) {
    // check if cell anchor, if so inherit row/column axis from target
    if (axis == null) {
      axis = headerContext.axis;
    }
    // make sure we only handle row - row, column - column properly. row-column/column-row is ignored if shiftkey is active.
    if ((axis.indexOf('row') !== -1 && headerContext.axis.indexOf('row') !== -1) ||
        (axis.indexOf('column') !== -1 && headerContext.axis.indexOf('column') !== -1)) {
      this.extendSelectionHeader(header, event);
    }
  } else if (event.button !== 2) {
    this._selectHeader(axis, index, level, event);
  }

  return undefined;
};

/**
 * Handles discontiguous header set active from databody
 * @param {Event} event triggering the operation
 * @param {string} axis header axis
 * @param {Element|null} header to be set active
 * @param {number} axisLevelCount
 */
DvtDataGrid.prototype.discontiguousHeaderSetActiveFromDatabody =
  function (event, axis, header, axisLevelCount) {
    var trueAxis = axis.replace('End', '');
    var currentSelection = this.m_selection[this.m_selection.length - 1];
    this.unhighlightSelection();
    if (currentSelection.startIndex.row === this.m_active.indexes.row &&
        currentSelection.endIndex.row === this.m_active.indexes.row &&
        currentSelection.startIndex.column === this.m_active.indexes.column &&
        currentSelection.endIndex.column === this.m_active.indexes.column) {
      this.m_selection.pop();
    }
    this._setActive(header, {
      type: 'header',
      index: this.m_trueIndex[trueAxis],
      level: axisLevelCount - 1,
      axis: axis
    }, event, false);
    this.rehighlightSelection();
  };

/**
 * Handles header change due to mouse up after dragging between parent/child or child/parent header anchor and target.
 * @param {Event} event triggering the operation
 */
DvtDataGrid.prototype.handleDragAnchorChange = function (event) {
  // If dragging and we land in a parent or child of the anchor, reset the anchor.
  if (this.m_headerDragState) {
    var target = /** @type {Element} */ (event.target);
    var targetElement = this.findHeader(target);
    var targetContext = targetElement[this.getResources().getMappedAttribute('context')];
    var sourceParent = this._getActiveElement();
    var sourceParentContext = sourceParent[this.getResources().getMappedAttribute('context')];
    var sourceParentStart = this._getAttribute(sourceParent.parentNode, 'start', true);
    var sourceParentEnd = sourceParentStart +
      (this._getAttribute(sourceParent.parentNode, 'extent', true) - 1);

    // if no extent, skip
    if (!targetContext.extent) {
      return;
    }

    var targetStart;
    var targetEnd;

    if (targetContext.extent === 1) {
      targetStart = targetContext.index;
      targetEnd = targetContext.index;
    } else {
      targetStart = this._getAttribute(targetElement.parentNode, 'start', true);
      targetEnd = targetStart + (this._getAttribute(targetElement.parentNode, 'extent', true) - 1);
    }

    var extent = sourceParentContext.extent;

    if (extent !== 1) {
      if ((targetStart <= sourceParentStart && targetEnd >= sourceParentEnd) ||
          (targetStart >= sourceParentStart && targetEnd <= sourceParentEnd)) {
        // if moving levels, and we are a parent or child of current cell, just reset active to the parent/child.
        this._setActive(targetElement, this._createActiveObject(targetElement), event);
      }
    } else if (targetStart !== targetEnd &&
               (targetStart <= sourceParentStart && sourceParentStart <= targetEnd)) {
      // if moving levels, and we are a parent or child of current cell, just reset active to the parent/child.
      this._setActive(targetElement, this._createActiveObject(targetElement), event);
    }
  }
};

/**
 * Handles click to select multiple cells/rows
 * @param {Event} event
 */
DvtDataGrid.prototype.handleDatabodyClickSelection = function (event) {
  var index;
  var target = /** @type {Element} */ (event.target);
  var cell = this.findCell(target);

  if (cell != null) {
    index = this.getCellIndexes(cell);
  }

  if (index != null) {
    if (this.isMultipleSelection() && event.button === 2 && this._isContainSelection(index)) {
      // if right click and inside multiple selection do not change anything
      return;
    }

    var ctrlKey = this.m_utils.ctrlEquivalent(event);
    var shiftKey = event.shiftKey;
    var cellSelected = this.m_utils.containsCSSClassName(cell, this.getMappedStyle('selected'));
    if (cellSelected && ctrlKey) {
      this.m_deselectInfo = { anchor: index, selection: this.GetSelection() };
      var endIndex = this.getCellEndIndexes(cell);
      var range = this._trimRangeForSelectionMode(this.createRange(index, endIndex));
      this.m_deselectInProgress = this._deselectRange(range, event);
      return;
    }

    if (this.isMultipleSelection()) {
      // remove the touch affordance on a new tap, unhighlight the active cell, and select the new one
      this._removeTouchSelectionAffordance();
      if (this.m_utils.isTouchDevice()) {
        if (this.m_active != null) {
          this._unhighlightActive();
        }
        this.selectAndFocus(index, event, false);
      } else if (!ctrlKey) {
        if (!shiftKey) {
          this.selectAndFocus(index, event, false);
        } else {
          this.extendSelection(index, event);
        }
      } else {
        this.selectAndFocus(index, event, true);
      }
    } else {
      this.selectAndFocus(index, event, false);
    }
  }
};

/**
 * Check if two ranges intersect
 * @private
 */
DvtDataGrid.prototype._doRangesOverlap = function (range1, range2) {
  var startIndex1 = range1.startIndex;
  var startCol1 = startIndex1.column;
  var startRow1 = startIndex1.row;
  var endIndex1 = range1.endIndex;
  var endCol1 = endIndex1.column;
  var endRow1 = endIndex1.row;

  var startIndex2 = range2.startIndex;
  var startCol2 = startIndex2.column;
  var startRow2 = startIndex2.row;
  var endIndex2 = range2.endIndex;
  var endCol2 = endIndex2.column;
  var endRow2 = endIndex2.row;

  // end info -1 for entire rows
  // end info undefined for row selection mode

  return (startCol1 <= endCol2 || endCol2 === -1 || endCol2 === undefined) &&
    (endCol1 >= startCol2 || endCol1 === -1 || endCol1 === undefined) &&
    (startRow1 <= endRow2 || endRow2 === -1) &&
    (endRow1 >= startRow2 || endRow1 === -1);
};

/**
 * Check if start of one range comes before the other
 * @private
 */
DvtDataGrid.prototype._isRangeValid = function (range) {
  return (range.startIndex.row <= range.endIndex.row ||
    (range.startIndex.row >= 0 && range.endIndex.row === -1)) &&
    (range.startIndex.column <= range.endIndex.column ||
    (range.startIndex.column >= 0 && range.endIndex.column === -1) ||
    (range.startIndex.column === undefined && range.endIndex.column === undefined));
};

/**
 * Deselect range
 * @private
 * @return true if deselect will happen
 */
DvtDataGrid.prototype._deselectRange = function (range, event) {
  var selection = this.m_deselectInfo.selection;

  var removeStartIndex = range.startIndex;
  var removeStartCol = removeStartIndex.column;
  var removeStartRow = removeStartIndex.row;
  var removeEndIndex = range.endIndex;
  var removeEndCol = removeEndIndex.column;
  var removeEndRow = removeEndIndex.row;

  var selectionChanged = false;
  var newSelection = [];
  var insertIndex = 0;
  var promises = [];
  var self = this;

  selection.forEach(function (selectedRange) {
    var selectedStartIndex = selectedRange.startIndex;
    var selectedStartCol = selectedStartIndex.column;
    var selectedStartRow = selectedStartIndex.row;
    var selectedEndIndex = selectedRange.endIndex;
    var selectedEndCol = selectedEndIndex.column;
    var selectedEndRow = selectedEndIndex.row;

    // do they intersect?
    if (this._doRangesOverlap(selectedRange, range)) {
      var startIndex;
      var endIndex;

      var createRangePromise = function (start, end, newSelectionIndex) {
        return new Promise(function (resolve) {
          var createRangeCallback = function (rangeWithKeys) {
            newSelection[newSelectionIndex] = rangeWithKeys;
            resolve();
          };
          self._createRangeWithKeys(start, end, createRangeCallback);
        });
      };

      selectionChanged = true;

      if (selectedStartRow < removeStartRow) {
        startIndex = this.createIndex(selectedStartRow, selectedStartCol);
        endIndex = this.createIndex(removeStartRow - 1, selectedEndCol);
        if (this._isRangeValid({ startIndex: startIndex, endIndex: endIndex })) {
          promises.push(createRangePromise(startIndex, endIndex, insertIndex));
          insertIndex += 1;
        }
      }

      if (removeEndRow !== -1) {
        startIndex = this.createIndex(removeEndRow + 1, selectedStartCol);
        endIndex = this.createIndex(selectedEndRow, selectedEndCol);
        if (this._isRangeValid({ startIndex: startIndex, endIndex: endIndex })) {
          promises.push(createRangePromise(startIndex, endIndex, insertIndex));
          insertIndex += 1;
        }
      }

      if (selectedStartCol < removeStartCol && selectedStartCol !== undefined) {
        startIndex = this.createIndex(Math.max(removeStartRow, selectedStartRow), selectedStartCol);
        endIndex = this.createIndex(Math.min(removeEndRow === -1 ? selectedEndRow : removeEndRow,
          selectedEndRow === -1 ? removeEndRow : selectedEndRow), removeStartCol - 1);
        if (this._isRangeValid({ startIndex: startIndex, endIndex: endIndex })) {
          promises.push(createRangePromise(startIndex, endIndex, insertIndex));
          insertIndex += 1;
        }
      }

      if (removeEndCol !== -1 && selectedStartCol !== undefined) {
        startIndex = this.createIndex(Math.max(removeStartRow, selectedStartRow), removeEndCol + 1);
        endIndex = this.createIndex(Math.min(removeEndRow === -1 ? selectedEndRow : removeEndRow,
          selectedEndRow === -1 ? removeEndRow : selectedEndRow), selectedEndCol);
        if (this._isRangeValid({ startIndex: startIndex, endIndex: endIndex })) {
          promises.push(createRangePromise(startIndex, endIndex, insertIndex));
          insertIndex += 1;
        }
      }
    } else {
      newSelection[insertIndex] = selectedRange;
      insertIndex += 1;
    }
  }, this);

  if (selectionChanged) {
    Promise.all(promises).then(function () {
      var previous = self.m_selection;
      self.SetSelection(newSelection);
      self._compareSelectionAndFire(event, previous);
    });
    return true;
  }
  return false;
};

DvtDataGrid.prototype._shouldDeselectHeader = function (index, extent, axis) {
  var selection = this.GetSelection();
  var primaryAxis;
  var secondaryAxis;
  if (axis === 'row' || axis === 'rowEnd') {
    primaryAxis = 'row';
    secondaryAxis = 'column';
  } else {
    primaryAxis = 'column';
    secondaryAxis = 'row';
  }
  var isSelected = false;
  var selectionMode = this.m_options.getSelectionMode();
  selection.forEach(function (selectedRange) {
    var secondaryStartIndex = selectedRange.startIndex[secondaryAxis];
    var secondaryEndIndex = selectedRange.endIndex[secondaryAxis];
    if ((secondaryStartIndex === 0 && secondaryEndIndex === -1) ||
      (selectionMode === 'row' && primaryAxis === 'row')) {
      var startIndex = selectedRange.startIndex[primaryAxis];
      var endIndex = selectedRange.endIndex[primaryAxis];
      if (startIndex <= index && (endIndex >= ((index + extent) - 1) || endIndex === -1)) {
        isSelected = true;
      }
    }
  });
  return isSelected;
};

/**
 * Handle select all
 * @param {Event} event the event causing the action
 * @returns {boolean} true if processed
 */
DvtDataGrid.prototype._handleSelectAll = function (event) {
  if (this._isSelectionEnabled() && this.isMultipleSelection()) {
    if (this.m_options.getSelectionMode() === 'row') {
      // drop the column index
      this._selectRange(this.createIndex(0), this.createIndex(-1), event);
    } else {
      this._selectRange(this.createIndex(0, 0), this.createIndex(-1, -1), event);
    }

    // if affordances are active, remove them.
    if (this.m_utils.isTouchDevice() &&
        this.m_topSelectIconContainer &&
        this.m_bottomSelectIconContainer) {
      this.m_topSelectIconContainer.parentNode.removeChild(this.m_topSelectIconContainer);
      this.m_bottomSelectIconContainer.parentNode.removeChild(this.m_bottomSelectIconContainer);
    }
    return true;
  }

  return false;
};

/**
 * Determine if the specified cell index is inside the current selection.
 * @param {Object} index the cell index
 * @param {Array=} ranges the selection to see if the index is in, allows us to check old ranges
 * @return {boolean} true is the cell index specified is inside the selection, false otherwise
 * @private
 */
DvtDataGrid.prototype._isContainSelection = function (index, ranges) {
  if (ranges == null) {
    // eslint-disable-next-line no-param-reassign
    ranges = this.GetSelection();
  }

  for (var i = 0; i < ranges.length; i += 1) {
    var range = ranges[i];
    var startIndex = range.startIndex;
    var endIndex = this.getEndIndex(range);

    var rangeStartRow = startIndex.row;
    var rangeEndRow = endIndex.row;

    // checks if row outside of range
    if (index.row >= rangeStartRow &&
        (rangeEndRow === -1 || index.row <= rangeEndRow)) {
      var rangeStartColumn = startIndex.column;
      var rangeEndColumn = endIndex.column;

      if (isNaN(rangeStartColumn) || isNaN(rangeEndColumn)) {
        // no column specified, meaning all columns
        return true;
      }

      // checks if column outside of range
      if (index.column >= rangeStartColumn &&
          (rangeEndColumn === -1 || index.column <= rangeEndColumn)) {
        // within range return immediately
        return true;
      }
    }
  }

  return false;
};

/**
 * Determine if the specified header index is inside the current selection.
 * @param {Object} index the header index
 * @param {Array=} axis header axis
 * @return {boolean} true is the header index specified is inside the selection, false otherwise
 * @private
 */
DvtDataGrid.prototype._isHeaderInsideSelection = function (index, axis) {
  const ranges = this.GetSelection();

  for (let i = 0; i < ranges.length; i += 1) {
    let range = ranges[i];
    let startIndex = range.startIndex;
    let endIndex = this.getEndIndex(range);

    let rangeStartRow = startIndex.row;
    let rangeEndRow = endIndex.row;
    let rangeStartColumn = startIndex.column;
    let rangeEndColumn = endIndex.column;

    if (axis === 'column' || axis === 'columnEnd') {
      if ((index >= rangeStartColumn &&
          (rangeEndColumn === -1 || index <= rangeEndColumn)) &&
          (rangeStartRow === 0 && rangeEndRow === -1)) {
        return true;
      }
    } else if (axis === 'row' || axis === 'rowEnd') {
      if ((index >= rangeStartRow &&
          (rangeEndRow === -1 || index <= rangeEndRow)) &&
          (rangeStartColumn === 0 && rangeEndColumn === -1)) {
        return true;
      }
    }
  }

  return false;
};
/**
 * Determine if the specified cell index is inside the current selection and return applicable class.
 * @param {Object} index the cell index
 * @param {Array=} ranges the selection to see if the index is in, allows us to check old ranges
 * @return {Object} An object with contains and class key.
 * @private
 */

DvtDataGrid.prototype._getContainedSelectionCssClass = function (index, ranges) {
  var classArray = [];
  var returnObj = {
    contains: false,
    class: []
  };
  if (ranges == null) {
    // eslint-disable-next-line no-param-reassign
    ranges = this.GetSelection();
  }

  for (var i = 0; i < ranges.length; i += 1) {
    var range = ranges[i];
    var startIndex = range.startIndex;
    var endIndex = this.getEndIndex(range);

    var rangeStartRow = startIndex.row;
    var rangeEndRow = endIndex.row;

    // checks if row outside of range
    if (index.row >= rangeStartRow &&
        (rangeEndRow === -1 || index.row <= rangeEndRow)) {
      if (index.row === rangeStartRow) {
        classArray.push('topSelected');
      }
      if (index.row === rangeEndRow) {
        classArray.push('bottomSelected');
      }
      var rangeStartColumn = startIndex.column;
      var rangeEndColumn = endIndex.column;

      if (isNaN(rangeStartColumn) || isNaN(rangeEndColumn)) {
        // no column specified, meaning all columns
        returnObj.contains = true;
        returnObj.class = classArray;
      }

      // checks if column outside of range
      if (index.column >= rangeStartColumn &&
          (rangeEndColumn === -1 || index.column <= rangeEndColumn)) {
        // within range return immediately
        returnObj.contains = true;
        returnObj.class = classArray;
      }
    }
  }

  return returnObj;
};
/**
 * Compare the two selection to see if they are identical.
 * @param {Object} selection1 the first selection
 * @param {Object} selection2 the second selection
 * @return {boolean} true if the selections are identical, false otherwise
 * @private
 */
DvtDataGrid.prototype._compareSelections = function (selection1, selection2) {
  // currently assumes all selections will be the same if old and new selection are equal
  // now allows not to fire on every drag event
  // todo: needs to handle discontigous selection case

  if (selection1.length !== selection2.length) {
    return false;
  }

  for (var i = 0; i < selection1.length; i += 1) {
    var foundMatch = false;
    for (var j = 0; j < selection2.length; j += 1) {
      if (this._compareIndividualSelectionObjects(selection1[i], selection2[j])) {
        foundMatch = true;
      }
    }
    if (foundMatch === false) {
      return false;
    }
  }

  return true;
};

/**
 * Compare the two selection to see if they are identical.
 * @param {Object} selection1 the first selection
 * @param {Object} selection2 the second selection
 * @return {boolean} true if the selections are identical, false otherwise
 * @private
 */
DvtDataGrid.prototype._compareIndividualSelectionObjects = function (selection1, selection2) {
  if (selection1.startIndex.row === selection2.startIndex.row &&
            selection1.startIndex.column === selection2.startIndex.column &&
            selection1.endIndex.row === selection2.endIndex.row &&
            selection1.endIndex.column === selection2.endIndex.column) {
    return true;
  }
  return false;
};

/**
 * Unhighlight and clear the current selection. If you are modifying the selection
 * object you should not call this method. It should only be used in the case of a
 * true clear where the selection winds up empty. This fires an event that the selection
 * has changed if it contained values beforehand.
 * @private
 * @param {Event=} event the event triggering the clear
 */
DvtDataGrid.prototype._clearSelection = function (event) {
  // unhighlight previous selection
  this.unhighlightSelection();
  this._removeTouchSelectionAffordance();

  // clear the selection and fire the
  var previous = this.GetSelection();
  this.m_selection = [];

  this._compareSelectionAndFire(event, previous);
};

/** *********************** key handler methods ************************************/
/**
 * Sets whether the data grid is in discontiguous selection mode
 * @param {boolean} flag true to set grid to discontiguous selection mode
 * @private
 */
DvtDataGrid.prototype.setDiscontiguousSelectionMode = function (flag) {
  this.m_discontiguousSelection = flag;

  // announce to screen reader
  this._setAccInfoText(flag ? 'accessibleRangeSelectModeOn' : 'accessibleRangeSelectModeOff');
};

/**
 * Selects the entire row of cells
 * @param {number} rowStart the end row index
 * @param {number} rowEnd the start row index
 * @param {Event} event the dom event that triggers the selection
 * @private
 */
DvtDataGrid.prototype._selectEntireRow = function (rowStart, rowEnd, event) {
  // create the start and end index then selects the range
  var startIndex = this.createIndex(rowStart, 0);
  var endIndex = this.createIndex(rowEnd, -1);
  var returnObj = this._getSelectionStartAndEnd(startIndex, endIndex, 0);

  if (this.m_options.getSelectionMode() === 'row') {
    // drop the column index
    this._selectRange(this.createIndex(returnObj.min.row),
                      this.createIndex(returnObj.max.row), event);
  } else {
    this._selectRange(this.createIndex(returnObj.min.row, 0),
                      this.createIndex(returnObj.max.row, -1), event);
  }
};

/**
 * Selects the entire column of cells
 * @param {number} columnStart the column start index
 * @param {number} columnEnd the column end index
 * @param {Event} event the dom event that triggers the selection
 * @private
 */
DvtDataGrid.prototype._selectEntireColumn = function (columnStart, columnEnd, event) {
  // create the start and end index then selects the range
  var startIndex = this.createIndex(0, columnStart);
  var endIndex = this.createIndex(-1, columnEnd);
  var returnObj = this._getSelectionStartAndEnd(startIndex, endIndex, 0);

  this._selectRange(this.createIndex(0, returnObj.min.column),
                    this.createIndex(-1, returnObj.max.column), event);
};

/**
 * Selects a range of cells.
 * @param {Object} startIndex the start row/column indexes
 * @param {Object} endIndex the end row/column indexes
 * @param {Event} event the dom event that triggers the selection
 * @private
 */
DvtDataGrid.prototype._selectRange = function (startIndex, endIndex, event) {
  // no longer clear selection, if it is cleared here we can't return anything for previous selection
  this.unhighlightSelection();
  this._createRangeWithKeys(startIndex, endIndex, this._selectRangeCallback.bind(this, event));
};

/**
 * Callback for once the new range is constructed
 * @param {Event} event the dom event that triggers the selection
 * @param {Object} newRange the new range to be selected
 * @private
 */
DvtDataGrid.prototype._selectRangeCallback = function (event, newRange) {
  var selection;

  // We need to pass the option change event the previous selection.
  // We also need to overwrite the old selection instance with a new one
  // so clone the old one, update, and then replace so that the object passed
  // as the previous matches the old reference and the new selection is a new
  // reference, create a brand new selection
  var previous = this.GetSelection();
  if (!this.m_discontiguousSelection && event &&
      !(this.m_utils.ctrlEquivalent(event) &&
        this.isMultipleSelection() && event.button === 0)) {
    selection = [];
  } else {
    selection = previous.slice(0);
  }
  selection.push(newRange);
  this.m_selection = selection;

  this.rehighlightSelection();

  if (this._isDatabodyCellActive() &&
      event.target[this.getResources().getMappedAttribute('context')] &&
      !event.target[this.getResources().getMappedAttribute('context')].axis &&
      !this.m_selectionFrontier.axis) {
    // reset frontier to be the same as active
    this.m_selectionFrontier = this.m_active.indexes;

    this._highlightActive();
  }

  // fire selection event if the selection has changed
  this._compareSelectionAndFire(event, previous);
};

/**
 * Highlight all ranges in the selection
 */
DvtDataGrid.prototype.rehighlightSelection = function () {
  for (var i = 0; i < this.m_selection.length; i++) {
    this.highlightRange(this.m_selection[i]);
  }
};

/**
 * Retrieve the current selection
 * @return {Array} an array of ranges
 */
DvtDataGrid.prototype.GetSelection = function () {
  if (this.m_selection == null) {
    this.m_selection = [];
  }
  return this.m_selection;
};

/**
 * Sets a range of selections
 * @param {Object} selection
 */
DvtDataGrid.prototype.SetSelection = function (selection) {
  // it can be null but cannot be undefined
  if (selection !== undefined) {
    if (selection === null) {
      // eslint-disable-next-line no-param-reassign
      selection = [];
    }

    // if we set the selection we should ungihlight the old one
    this.unhighlightSelection();

    this.m_selection = selection;

    // update headers
    this._resetHeaderHighLight();

    // if it's not render yet, don't apply selection
    if (this.m_databody != null) {
      this.applySelection(this.m_startRow, this.m_endRow, this.m_startCol, this.m_endCol);
    }
    // do not fire selection event when set on us externally, it will be taken
    // care of in the wrappers option layer

    if (this.m_bottomFloodFillIconContainer) {
      if (this.m_selection.length === 1) {
        this._moveFloodFillAffordance();
      } else {
        this._removeFloodFillAffordance();
      }
    }
  }
};

/**
 * Fires selection event
 * @param {Event|undefined} event the dom event that triggers the selection
 * @param {Object} previousSelection
 * @protected
 */
DvtDataGrid.prototype.fireSelectionEvent = function (event, previousSelection) {
  var details = {
    event: event,
    ui: {
      selection: this.GetSelection(),
      previousSelection: previousSelection
    }
  };
  this.fireEvent('select', details);
};

/**
 * Shift+click to extend the selection
 * @param {Object} index - the end index of the selection.
 * @param {Event=} event - the DOM event causing the selection to to be extended
 * @param {number=} direction - the keystroke keyCode if applicable
 */
DvtDataGrid.prototype.extendSelection = function (index, event, direction) {
  // prevent cell selection when extending a row/column selection
  if (this.m_active.type === 'header') {
    return;
  }

  var anchor;

  // find the the top left index
  if (this.m_utils.isTouchDevice()) {
    anchor = this.m_touchSelectAnchor;
  } else {
    // do not copy anchor object so we can directly modify it
    var activeRow = this.m_active.indexes.row;
    var activeColumn = this.m_active.indexes.column;
    anchor = { row: activeRow, column: activeColumn };
  }

  if (anchor == null) {
    return;
  }

  // reset focus on previous selection frontier
  this._resetSelectionFrontierFocus();

  // update the selctionFrontier, more complicated dues to merged cells
  var returnObj = this._updateSelectionFrontier(anchor, index, direction);
  var minIndex = returnObj.min;
  var maxIndex = returnObj.max;

  if (this.m_options.getSelectionMode() === 'row') {
    // drop the column index
    minIndex = this.createIndex(minIndex.row);
    maxIndex = this.createIndex(maxIndex.row);
  }

  if (this.m_discontiguousSelection) {
    if (this.m_deselectInProgress ||
      this._isContainSelection(anchor, this.GetSelection().slice(0, -1))) {
      if (!this.m_deselectInProgress) {
        this.m_deselectInfo = { anchor: anchor, selection: this.GetSelection() };
      }
      var range = this._trimRangeForSelectionMode(this.createRange(minIndex, maxIndex));
      this.m_deselectInProgress = this._deselectRange(range, event);
      return;
    }
  }

  this._createRangeWithKeys(minIndex, maxIndex,
    this._extendSelectionCallback.bind(this, event, anchor));
};

/**
 * Extend the deselection for headers
 * @private
 */
DvtDataGrid.prototype.extendDeselection = function (index, event) {
  var anchor = this.m_deselectInfo.anchor;
  var returnObj = this._updateSelectionFrontier(anchor, index);
  var minIndex = returnObj.min;
  var maxIndex = returnObj.max;
  var range = this._trimRangeForSelectionMode(this.createRange(minIndex, maxIndex));
  this._deselectRange(range, event);
};

/**
 * Extend the selection for headers
 * @param {Object} target - {axis, index} target of the selection.
 * @param {Event=} event - the DOM event causing the selection to to be extended
 * @param {boolean=} isExtend - boolean if we are extending a selection
 * @param {boolean=} isDeselect - boolean if we are deselecting
 */
DvtDataGrid.prototype.extendSelectionHeader = function (target, event, isExtend, isDeselect) {
  // no target so just return
  if (target === null) {
    return;
  }

  var targetElement = this.findCellOrHeader(target);
  if (targetElement === null) {
    return;
  }
  var targetContext = targetElement[this.getResources().getMappedAttribute('context')];
  var anchor;
  var axis;
  var targetStart;
  var targetEnd;
  var endpoint;

  if (isExtend == null) {
    // eslint-disable-next-line no-param-reassign
    isExtend = this.m_discontiguousSelection;
  }

  // if no targetContext, break
  if (!targetContext) {
    return;
  } else if (targetContext.cell) {
    if (this.m_selectionFrontier.axis.indexOf('column') !== -1) {
      targetStart = targetContext.indexes.column;
    } else {
      targetStart = targetContext.indexes.row;
    }
    targetEnd = targetStart;
    endpoint = targetStart;
  } else {
    // check if target is the same as the current frontier target
    if (this.m_selectionFrontier && targetContext.index === this.m_selectionFrontier.index &&
        targetContext.level === this.m_selectionFrontier.level) {
      return;
    }

    if (targetContext.extent === 1) {
      targetStart = targetContext.index;
      targetEnd = targetContext.index;
      endpoint = targetContext.index;
    } else {
      targetStart = this._getAttribute(target.parentNode, 'start', true);
      targetEnd = targetStart + (this._getAttribute(target.parentNode, 'extent', true) - 1);
    }
  }

  var sourceParent = isDeselect ? this.m_deselectInfo.sourceParent : this._getActiveElement();
  if (!sourceParent) {
    return;
  }
  var sourceParentContext = sourceParent[this.getResources().getMappedAttribute('context')];
  var sourceParentStart;
  var sourceParentEnd;

  // find the the top left index
  if (isDeselect) {
    axis = this.m_deselectInfo.axis;
    anchor = this.m_deselectInfo.anchor[axis];
    sourceParentStart = this._getAttribute(sourceParent.parentNode, 'start', true);
    sourceParentEnd = sourceParentStart + (this._getAttribute(sourceParent.parentNode, 'extent', true) - 1);
  } else if (this.m_utils.isTouchDevice()) {
    if (this.m_selectionFrontier.axis.indexOf('column') !== -1) {
      anchor = this.m_touchSelectAnchor.column;
    } else {
      anchor = this.m_touchSelectAnchor.row;
    }

    if (anchor === -1) {
      anchor = 0;
    }
    sourceParentStart = anchor;
    sourceParentEnd = anchor;
    axis = this.m_selectionFrontier.axis;
  } else {
    sourceParentStart = this._getAttribute(sourceParent.parentNode, 'start', true);
    sourceParentEnd =
      sourceParentStart + (this._getAttribute(sourceParent.parentNode, 'extent', true) - 1);

    if (this.m_active.type === 'header') {
      axis = this.m_active.axis;
      anchor = this.m_active.index;
    }
  }

  var extent;

  if (this.m_active.type === 'cell') {
    extent = 1;
    axis = targetContext.axis;
    if (axis.indexOf('row') !== -1) {
      anchor = this.m_active.indexes.row;
    } else {
      anchor = this.m_active.indexes.column;
    }
  } else {
    extent = sourceParentContext.extent;
  }

  // pop if anchors match so we don't duplicate a selection
  if (isExtend && !isDeselect) {
    var previousAnchor = this.m_selection[this.m_selection.length - 1];

    // unhighlight selection now
    this.unhighlightSelection();

    // check that the anchor and the selection frontier index match the previous anchor start and end indices
    if (previousAnchor && this.checkCorners(axis, anchor, previousAnchor)) {
      this.m_selection.pop();
    }
  }

  if (extent !== 1) {
    if ((targetStart <= sourceParentStart && targetEnd >= sourceParentEnd) ||
        (targetStart >= sourceParentStart && targetEnd <= sourceParentEnd)) {
      // if moving levels, and we are a parent or child of current cell, just reset active to the parent/child.
      // unless we are click + drag, then we don't change the target
      if (!this.m_headerDragState) {
        this._setActive(targetElement, this._createActiveObject(targetElement), event);
      }

      anchor = targetStart;
      endpoint = targetEnd;
    } else if (targetStart > sourceParentStart && targetEnd > sourceParentEnd) {
      anchor = sourceParentStart;
      endpoint = targetEnd;
    } else if (targetStart < sourceParentStart && targetEnd < sourceParentEnd) {
      anchor = targetStart;
      endpoint = sourceParentEnd;
    }
  } else if (extent === 1) {
    if (targetStart !== targetEnd) {
      if (targetStart <= anchor && anchor <= targetEnd) {
        // if moving levels, and we are a parent or child of current cell, just reset active to the parent/child.
        // unless we are click + drag or have cell type anchor, then we don't change the target
        if (!this.m_headerDragState && this.m_active.type !== 'cell') {
          this._setActive(targetElement, this._createActiveObject(targetElement), event);
        }
        anchor = targetStart;
        endpoint = targetEnd;
      } else if (targetStart <= anchor && targetEnd <= anchor) {
        endpoint = anchor;
        anchor = targetStart;
      } else if (targetStart >= anchor && targetEnd >= anchor) {
        endpoint = targetEnd;
      }
    }
  }

  if (targetContext.cell) {
    this.setHeaderSelectionFrontier(axis, endpoint, endpoint, this.m_selectionFrontier.level,
                                    /** @type {Element} */ (targetElement), true);
  } else {
    this.setHeaderSelectionFrontier(axis, endpoint, targetContext.index, targetContext.level,
                                    /** @type {Element} */ (targetElement), true);
  }

  if (!isDeselect && axis) {
    if (axis.indexOf('column') !== -1) {
      this._selectEntireColumn(anchor, endpoint, event);
    } else if (axis.indexOf('row') !== -1) {
      this._selectEntireRow(anchor, endpoint, event);
    }
  } else {
    var range;
    var start = Math.min(anchor, endpoint);
    var end = Math.max(anchor, endpoint);
    if (axis != null && axis.indexOf('row') !== -1) {
      range = this.createRange(this.createIndex(start, 0), this.createIndex(end, -1));
    } else {
      range = this.createRange(this.createIndex(0, start), this.createIndex(-1, end));
    }
    var trimmedRange = this._trimRangeForSelectionMode(range);
    this._deselectRange(trimmedRange, event);
  }
};

/**
 * @param {Object} anchor - the current selection.
 * @param {Object} index - the anchor of the extend selection.
 * @param {number=} direction - keyCode
 * @returns {Object} with min and max index of the selection
 */
DvtDataGrid.prototype._updateSelectionFrontier = function (anchor, index, direction) {
  var tempAnchor;
  var tempIndex;
  var currentRange;

  if (this.m_options.getSelectionMode() === 'row') {
    // drop the column index
    tempAnchor = this.createIndex(anchor.row, this.m_startCol);
    tempIndex = this.createIndex(index.row, this.m_endCol);
  } else {
    tempAnchor = anchor;
    tempIndex = index;
  }

  // keyboard specific
  if (direction != null) {
    if (this.m_deselectInProgress) {
      currentRange = this.createRange(this.m_deselectInfo.anchor, this.m_selectionFrontier);
    } else {
      var previous = this.GetSelection();
      var lastSelectionIndex = this._getLastAnchoredSelectionIndex(previous, tempAnchor);
      currentRange = previous[lastSelectionIndex];
    }

    var startIndex = currentRange.startIndex;
    var endIndex = currentRange.endIndex;

    // update the tempAnchor if it has extended due to a merge cell changing the corner of the tempAnchor
    if (startIndex.row === this.m_selectionFrontier.row && tempAnchor.row !== startIndex.row) {
      tempAnchor.row = endIndex.row;
    } else if (endIndex.row === this.m_selectionFrontier.row && tempAnchor.row !== endIndex.row) {
      tempAnchor.row = startIndex.row;
    }

    if (startIndex.column === this.m_selectionFrontier.column &&
        tempAnchor.column !== startIndex.column) {
      tempAnchor.column = endIndex.column;
    } else if (endIndex.column === this.m_selectionFrontier.column &&
               tempAnchor.column !== endIndex.column) {
      tempAnchor.column = startIndex.column;
    }
  }

  var returnObj = this._getSelectionStartAndEnd(tempAnchor, tempIndex, 0);
  var useIndex = tempIndex;

  if (direction != null) {
    var minIndex = returnObj.min;
    var maxIndex = returnObj.max;

    if (this.m_options.getSelectionMode() === 'row') {
      // drop the column index
      minIndex = this.createIndex(minIndex.row);
      maxIndex = this.createIndex(maxIndex.row);
    }

    // ensure the range actually grows as a result of the arrow key
    while (this._compareIndividualSelectionObjects(currentRange,
                                                   this.createRange(minIndex, maxIndex))) {
      if (direction === this.keyCodes.LEFT_KEY) {
        if (tempIndex.column === 0) {
          break;
        }
        tempIndex.column -= 1;
      } else if (direction === this.keyCodes.RIGHT_KEY) {
        if (tempIndex.column === this.m_endCol) {
          break;
        }
        tempIndex.column += 1;
      } else if (direction === this.keyCodes.UP_KEY) {
        if (tempIndex.row === 0) {
          break;
        }
        tempIndex.row -= 1;
      } else if (direction === this.keyCodes.DOWN_KEY) {
        if (tempIndex.row === this.m_endRow) {
          break;
        }
        tempIndex.row += 1;
      } else {
        break;
      }

      returnObj = this._getSelectionStartAndEnd(tempAnchor, tempIndex, 0);
      minIndex = returnObj.min;
      maxIndex = returnObj.max;
      if (this.m_options.getSelectionMode() === 'row') {
        // drop the column index
        minIndex = this.createIndex(minIndex.row);
        maxIndex = this.createIndex(maxIndex.row);
      }
    }

    // set the frontier appropriately
    if (direction === this.keyCodes.LEFT_KEY) {
      useIndex.column = (minIndex.column < tempAnchor.column) ? minIndex.column : maxIndex.column;
      useIndex.row = (minIndex.row < tempAnchor.row) ? minIndex.row : maxIndex.row;
    } else if (direction === this.keyCodes.RIGHT_KEY) {
      useIndex.column = (maxIndex.column > tempAnchor.column) ? maxIndex.column : minIndex.column;
      useIndex.row = (minIndex.row < tempAnchor.row) ? minIndex.row : maxIndex.row;
    } else if (direction === this.keyCodes.UP_KEY) {
      useIndex.row = (minIndex.row < tempAnchor.row) ? minIndex.row : maxIndex.row;
      useIndex.column = (minIndex.column < tempAnchor.column) ? minIndex.column : maxIndex.column;
    } else if (direction === this.keyCodes.DOWN_KEY) {
      useIndex.row = (maxIndex.row > tempAnchor.row) ? maxIndex.row : minIndex.row;
      useIndex.column = (minIndex.column < tempAnchor.column) ? minIndex.column : maxIndex.column;
    }
  } else {
    useIndex = this.getCellIndexes(this._getCellByIndex(useIndex));
  }

  if (this.m_options.getSelectionMode() === 'row') {
    useIndex = this.createIndex(useIndex.row, this.m_active.indexes.column);
  }

  this.m_selectionFrontier = useIndex;

  return returnObj;
};

/**
 * Get the min or max between two numbers that could either be NaN
 * @param {number|undefined|null} val1 - val1
 * @param {number|undefined|null} val2 - val2
 * @param {Function} mathFunc - math function to apply to two values
 * @returns {number|undefined|null}
 */
DvtDataGrid.prototype._getMinOrMax = function (val1, val2, mathFunc) {
  if (isNaN(val1)) {
    if (isNaN(val2)) {
      return null;
    }
    return val2;
  } else if (isNaN(val2)) {
    return val1;
  }
  return mathFunc(val1, val2);
};

/**
 * @param {Object} anchor - the current selection.
 * @param {Object} index - the anchor of the extend selection.
 * @param {number} numElems - the number of elements in the range used to break recursion
 * @returns {Object} with min and max index of the selection
 */
DvtDataGrid.prototype._getSelectionStartAndEnd = function (anchor, index, numElems) {
  var cells = this.getElementsInRange(this.createRange(anchor, index));

  // getElementsInRange can return null if not rendered yet
  if (cells == null || cells.length === numElems) {
    return { min: anchor, max: index };
  }

  var minIndex = {
    row: this._getMinOrMax(anchor.row, index.row, Math.min),
    column: this._getMinOrMax(anchor.column, index.column, Math.min)
  };
  var maxIndex = {
    row: this._getMinOrMax(anchor.row, index.row, Math.max),
    column: this._getMinOrMax(anchor.column, index.column, Math.max)
  };

  for (var i = 0; i < cells.length; i++) {
    var startIndex = this.getCellIndexes(cells[i]);
    var endIndex = this.getCellEndIndexes(cells[i]);

    if (startIndex.row < minIndex.row || minIndex.row == null) {
      minIndex.row = startIndex.row;
    }
    if (startIndex.column < minIndex.column || minIndex.column == null) {
      minIndex.column = startIndex.column;
    }
    if (endIndex.row > maxIndex.row || maxIndex.row == null) {
      maxIndex.row = endIndex.row;
    }
    if (endIndex.column > maxIndex.column || maxIndex.column == null) {
      maxIndex.column = endIndex.column;
    }
  }

  return this._getSelectionStartAndEnd(minIndex, maxIndex, cells.length);
};

/**
 * @param {Object} selection - the current selection.
 * @param {Object} anchor - the anchor of the extend selection.
 * @returns {number} the index of the selection that contains the anchor cell as a corner
 * @private
 */
DvtDataGrid.prototype._getLastAnchoredSelectionIndex = function (selection, anchor) {
  var i;
  for (i = selection.length - 1; i > -1; i--) {
    // if row selection just compare rows, also check if its not select all type.
    if (this._isContainSelection(anchor, [selection[i]])) {
      return i;
    }
  }

  // return last index by default to keep old behavior as worst case
  return i - 1;
};

/**
 * Once the range is created from the index continue to extend the selection
 * @param {Event|null|undefined} event - the DOM event causing the selection to to be extended
 * @param {Object} anchor - the anchor cell
 * @param {Object} newRange - the new range of the selection.
 * @private
 */
DvtDataGrid.prototype._extendSelectionCallback = function (event, anchor, newRange) {
  var previous = this.GetSelection();
  var lastSelectionIndex = this._getLastAnchoredSelectionIndex(previous, anchor);
  var currentRange = previous[lastSelectionIndex];

  if (currentRange == null) {
    // Anchor cell removed from selection during marquee selection
    return;
  }

  // checks if selection has changed
  var startIndexesMatch = (currentRange.startIndex.row === newRange.startIndex.row);
  if (currentRange.startIndex.column != null && newRange.startIndex.column != null) {
    startIndexesMatch =
      startIndexesMatch && (currentRange.startIndex.column === newRange.startIndex.column);
  }

  var endIndexesMatch = (currentRange.endIndex.row === newRange.endIndex.row);
  if (currentRange.endIndex.column != null && newRange.endIndex.column != null) {
    endIndexesMatch =
      endIndexesMatch && (currentRange.endIndex.column === newRange.endIndex.column);
  }

  if (startIndexesMatch && endIndexesMatch) {
    return;
  }

  var selection;

  // if ctrl key is active, act as if discontiguous mode is on
  if (this.m_discontiguousSelection ||
      (event && this.m_utils.ctrlEquivalent(event) && event.button === 0)) {
    // We also need to overwrite the old selection instance with a new one
    // so clone the old one, update, and then replace so that the object passed
    // as the previous matches the old reference and the new selection is a new
    // reference
    selection = previous.slice(0);

    // unhighlight the last range
    this.unhighlightRange(currentRange);

    // remove the current range and put it at the end
    selection.splice(lastSelectionIndex, 1);
  } else {
    // not keeping selections so clear selection and unhighlight.
    selection = [];
    this.unhighlightSelection();
  }


  selection.push(newRange);
  this.m_selection = selection;

  // if ctrl key is active, act as if discontiguous mode is on
  if (this.m_discontiguousSelection ||
      (event && this.m_utils.ctrlEquivalent(event) && event.button === 0)) {
    for (var i = 0; i < this.m_selection.length; i++) {
      this.highlightRange(this.m_selection[i]);
    }
  } else {
    this.highlightRange(newRange, true);
  }

  // focus on the frontier cell
  this._makeSelectionFrontierFocus();

  this._compareSelectionAndFire(event, previous);
};

/**
 * Reset focus on selection frontier
 * @private
 */
DvtDataGrid.prototype._resetSelectionFrontierFocus = function () {
  // make sure there is a selection frontier and it's not the same as the active cell
  if (this.m_selectionFrontier == null ||
      (this._isDatabodyCellActive() &&
       this.m_selectionFrontier.row === this.m_active.indexes.row &&
       this.m_selectionFrontier.column === this.m_active.indexes.column)) {
    return;
  }

  var range = this.createRange(this.m_selectionFrontier);
  var cell = this.getElementsInRange(range);

  if (cell != null && cell.length > 0) {
    this._unsetAriaProperties(cell[0]);
  }
};

/**
 * Make the selection frontier focusable.
 * @private
 */
DvtDataGrid.prototype._makeSelectionFrontierFocus = function () {
  // make sure there is a selection frontier and it's not the same as the active cell
  if (this.m_selectionFrontier == null ||
      (this._isDatabodyCellActive() &&
       this.m_selectionFrontier.row === this.m_active.indexes.row &&
       this.m_selectionFrontier.column === this.m_active.indexes.column)) {
    return;
  }

  // unset focus properties on active cell first
  if (this._isDatabodyCellActive()) {
    var activeRange = this.createRange(this.m_active.indexes);
    var activeCell = this.getElementsInRange(activeRange);

    if (activeCell != null && activeCell.length > 0) {
      this._unsetAriaProperties(activeCell[0]);
    }
  }

  var range = this.createRange(this.m_selectionFrontier);
  var rowOrCell = this.getElementsInRange(range);
  if (rowOrCell == null || rowOrCell.length === 0) {
    return;
  }

  // update context info
  this._updateContextInfo(this.m_selectionFrontier);

  // focus on the cell (or first cell in the row)
  var cell = this.m_utils.containsCSSClassName(rowOrCell[0], this.getMappedStyle('row')) ?
    rowOrCell[0].firstChild : rowOrCell[0];
  this._setAriaProperties(this._createActiveObject(cell), null, cell);
};

/**
 * Selects the row or column of the specified header,
 * @param {string} axis - the string axis of the header
 * @param {number} index - the index of the header selected.
 * @param {number} level - the level of the header selected.
 * @param {Event} event - the event causing the header selection
 */
DvtDataGrid.prototype._selectHeader = function (axis, index, level, event) {
  var start;
  var end;

  // get indices and select rows/columns underneath header
  if ((axis.indexOf('row') !== -1 && this.m_rowHeaderLevelCount - 1 === level) ||
      (axis.indexOf('column') !== -1 && this.m_columnHeaderLevelCount - 1 === level)) {
    start = index;
    end = index;
  } else {
    var elem = this._getActiveElement();
    start = /** @type {number} */ (this._getAttribute(elem.parentNode, 'start', true));
    end = start + (this._getAttribute(elem.parentNode, 'extent', true) - 1);
  }

  if (axis.indexOf('row') !== -1) {
    // block selection on row based mode single
    if ((start === end && !this.isMultipleSelection()) || this.isMultipleSelection()) {
      this.setHeaderSelectionFrontier(axis, end, index, level,
                                      /** @type {Element} */ (event.target), true);

      // handle the space key in headers for rows
      this._selectEntireRow(start, end, event);
    }

    // announce to screen reader, no need to include context info
    this._setAccInfoText('accessibleRowSelected', { row: index + 1 });
  } else if (axis.indexOf('column') !== -1) {
    this.setHeaderSelectionFrontier(axis, end, index, level,
                                    /** @type {Element} */ (event.target), true);

    // handle the space key in headers for columns
    this._selectEntireColumn(start, end, event);

    // announce to screen reader, no need to include context info
    this._setAccInfoText('accessibleColumnSelected', { column: index + 1 });
  }

  // In case we are row based single selection mode
  if (this.isMultipleSelection()) {
    this.m_headerDragState = true;
  }
};

/**
 * Sets the selection frontier for headers
 * @param {string} axis - the string axis of the header
 * @param {number} end - the index number of the far row/column from the anchor
 * @param {number} index - the index of the header selected.
 * @param {number} level - the level of the header selected.
 * @param {Element} target target cell or header of the event
 * @param {boolean} reset - whether or not to clear the selection frontier
 */
DvtDataGrid.prototype.setHeaderSelectionFrontier =
  function (axis, end, index, level, target, reset) {
    if (reset) {
      this.m_selectionFrontier = {};
    }
    this.m_selectionFrontier.axis = axis;
    this.m_selectionFrontier.end = end;
    this.m_selectionFrontier.index = index;
    this.m_selectionFrontier.level = level;
    this.m_selectionFrontier.target = target;
  };

/**
 * Resets header highlight from selection state.
 */
DvtDataGrid.prototype._resetHeaderHighLight = function () {
  this.GetSelection().forEach((range) => {
    var rowHeadersInRange = this.getHeadersByRange(range, 'row');
    var colHeadersInRange = this.getHeadersByRange(range, 'column');
    this._highlightHeaders(range, rowHeadersInRange, colHeadersInRange);
  });
};

/**
 * Updates the selection for headers after a fetch
 */
DvtDataGrid.prototype.updateSelectionHeader = function () {
  if (this.m_selectionFrontier && this.m_selectionFrontier.target) {
    if (this.m_utils.isTouchDevice() && this.m_selection.length) {
      this.m_touchSelectAnchor = this.m_selection[this.m_selection.length - 1].startIndex;
    }
    this.extendSelectionHeader(this.m_selectionFrontier.target, null);
  }
  this._resetHeaderHighLight();
};

/**
 * Selects the focus on the specified element, if ctrl+click to add cell/row to the current selection,
 * set the augment flag
 * Select and focus is an asynchronus call
 * @param {Object} index - the end index of the selection.
 * @param {Event=} event - the event causing the selection and setting active
 * @param {boolean=} augment - true if we are augmenting the selecition, default to false
 */
DvtDataGrid.prototype.selectAndFocus = function (index, event, augment) {
  if (augment == null) {
    // eslint-disable-next-line no-param-reassign
    augment = false;
  }

  // reset any focus properties set on frontier cell
  this._resetSelectionFrontierFocus();

  // update active cell
  // if virtual we will still want the new selection to be applied
  this._setActiveByIndex(index, event);

  if (this.m_options.getSelectionMode() === 'row') {
    // eslint-disable-next-line no-param-reassign
    index = this.createIndex(index.row);
  }

  // need the selection frontier maintained until final callback
  var returnObj = this._getSelectionStartAndEnd(index, index, 0);
  var minIndex = returnObj.min;
  var maxIndex = returnObj.max;

  // update selection mode
  if (this.m_options.getSelectionMode() === 'row') {
    // drop the column index
    minIndex = this.createIndex(minIndex.row);
    maxIndex = this.createIndex(maxIndex.row);
  }

  // ensure end index is specified when push to selection
  this._createRangeWithKeys(minIndex, maxIndex,
                            this._selectAndFocusRangeCallback.bind(this, minIndex,
                                                                   event, augment));
};

/**
 * Continue to selectAndFocus and _selectAndFocusActiveCallback
 * @param {Object} index - the end index of the selection.
 * @param {Event|undefined} event - the event causing the selection to to be changed
 * @param {boolean} augment - true if selection being augmented
 * @param {Object} range - the range of the selection.
 * @private
 */
DvtDataGrid.prototype._selectAndFocusRangeCallback = function (index, event, augment, range) {
  var previous = this.GetSelection();
  var selection = previous.slice(0);

  if (!augment) {
    // if we are not augmenting the selection modify the old one appropriately
    if (!this.m_discontiguousSelection && event &&
    !(this.isMultipleSelection() && this.m_utils.ctrlEquivalent(event) && event.button === 0)) {
      this.unhighlightSelection();
      // this should be a new selection
      selection = [];
    } else if (this._isDatabodyCellActive() &&
               this.m_prevActive != null &&
               this.m_prevActive.type === 'cell' &&
               this.m_selectionFrontier.row === this.m_prevActive.indexes.row &&
               this.m_selectionFrontier.column === this.m_prevActive.indexes.column &&
               !this.m_utils.isTouchDevice() &&
               (event.keyCode || !this.isMultipleSelection())) {
      // this is for the Shift + F8 navigate case, we are adding to the selection on every arrow,
      // but if the user is trying to navigate away we are always popping the last selection off because
      // it was just used to navigate away, do not do this on touch because their is no navigation concept

      // remove the last selection
      selection.pop();
      var isContainedObj = this._getContainedSelectionCssClass(this.m_prevActive.indexes,
        selection);
      // unhighlight previous (active and selection)
      // only if it's not in an existing selection
      if (!isContainedObj.contains) {
        this._unhighlightElement(this._getCellByIndex(this.m_prevActive.indexes),
                                ['selected', 'topSelected', 'bottomSelected']);
      } else {
        var selectedClass = isContainedObj.class;
        if (selectedClass.length) {
          var classArray = [];
          if (selectedClass.indexOf('topSelected') === -1) {
            classArray.push('topSelected');
          }
          if (selectedClass.indexOf('bottomSelected') === -1) {
            classArray.push('bottomSelected');
          }
          this._unhighlightElement(this._getCellByIndex(this.m_prevActive.indexes),
                                classArray);
        } else {
          this._unhighlightElement(this._getCellByIndex(this.m_prevActive.indexes),
                                ['topSelected', 'bottomSelected']);
        }
      }
    }
  }

  this.m_selectionFrontier = index;

  // We need to overwrite the old selection instance with a new one
  // so clone the old one, update, and then replace so that the object passed
  // as the previous matches the old reference and the new selection is a new
  // reference
  selection.push(range);
  this.m_selection = selection;

  this.highlightRange(range);

  this._compareSelectionAndFire(event, previous);
};

/** ******************* end key handler methods ************************************/

/** ******************* focusable/editable element related methods *****************/
/**
 * Compare the selection to a clone and fire selection event if it has changed
 * @param {Event|undefined} event the DOM event to pass off in the selection event
 * @param {Object} clone the old selection object
 * @private
 */
DvtDataGrid.prototype._compareSelectionAndFire = function (event, clone) {
  var selection = this.GetSelection();
  // only deal with touch affordances if multiple selection on touch
  if (this.isMultipleSelection() && selection.length > 0) {
    if (this.m_utils.isTouchDevice()) {
      this._addTouchSelectionAffordance(event);
      this._moveTouchSelectionAffordance();
    } else if (this.m_options.isFloodFillEnabled() &&
            this._isSelectionEnabled() && !this.m_discontiguousSelection &&
            this.m_active.type === 'cell') {
      this._addFloodfillAffordance(event);
      this._moveFloodFillAffordance();
    }
  }

  // fire event if selection has changed
  if (!this._compareSelections(selection, clone)) {
    this._resetHeaderHighLight();
    this.fireSelectionEvent(event, clone);
  }
};

/**
 * Add the touch affordance to the grid. It will be added to the row containing the active cell in row/cell selection mode.
 * Sets the position of the affordance to be on the corner of a cell in cell selection or the center of the viewport in row
 * selection.
 * @param {Event|undefined} event the event that drives the need for touch affordance
 * @private
 */
DvtDataGrid.prototype._addTouchSelectionAffordance = function (event) {
  // icon in the corner
  if (this.m_topSelectIconContainer == null && this.m_bottomSelectIconContainer == null) {
    var target = /** @type {Element} */ (event.target);
    var cell = this.findCell(target);
    // if cell not found, try header
    if (!cell) {
      cell = this.findHeader(target);
    }
    // if no cell is present, it is not a header or a cell. Probably corner click to select all.
    // in this case just return since we don't want affordances for select all (immutable selection).
    if (!cell) {
      return;
    }

    // cache the containers so we always know where they are since selection object isn't always current
    // wrap the icon in a container so the touch area is larger than the icon
    this.m_topSelectIconContainer = document.createElement('div');
    this.m_topSelectIconContainer.className = this.getMappedStyle('toucharea');

    var topIcon = document.createElement('div');
    topIcon.className = this.getMappedStyle('selectaffordance');
    topIcon.setAttribute('role', 'button');
    topIcon.setAttribute(
      'aria-label', this.getResources().getTranslatedText('accessibleSelectionAffordanceTop'));
    this.m_topSelectIconContainer.appendChild(topIcon); // @HTMLUpdateOK

    this.m_bottomSelectIconContainer = document.createElement('div');
    this.m_bottomSelectIconContainer.className = this.getMappedStyle('toucharea');

    var bottomIcon = document.createElement('div');
    bottomIcon.className = this.getMappedStyle('selectaffordance');
    bottomIcon.setAttribute('role', 'button');
    bottomIcon.setAttribute(
      'aria-label', this.getResources().getTranslatedText('accessibleSelectionAffordanceBottom'));
    this.m_bottomSelectIconContainer.appendChild(bottomIcon); // @HTMLUpdateOK

    this.m_databody.firstChild.appendChild(this.m_topSelectIconContainer); // @HTMLUpdateOK
    this.m_databody.firstChild.appendChild(this.m_bottomSelectIconContainer); // @HTMLUpdateOK
    this.m_touchSelectionAffordanceHeight =
      this.m_topSelectIconContainer.firstElementChild.offsetHeight;
    this.m_touchSelectionAffordanceWidth =
      this.m_topSelectIconContainer.firstElementChild.offsetWidth;
  }
};

/**
 * Adds selection affordance handles rounded borders that are dependent on selection mode.
 * @private
 */
 DvtDataGrid.prototype._addRoundedAffordanceClasses = function (topIcon, bottomIcon, bounded,
  axis) {
  if (topIcon && bottomIcon) {
    if (!bounded && axis === 'row') {
      topIcon.classList.add(this.getMappedStyle('selectaffordancetoprow'));
      bottomIcon.classList.add(this.getMappedStyle('selectaffordancebottomrow'));
    } else if (!bounded && axis === 'column') {
      topIcon.classList.add(this.getMappedStyle('selectaffordancetopcolumn'));
      bottomIcon.classList.add(this.getMappedStyle('selectaffordancebottomcolumn'));
    } else {
      topIcon.classList.add(this.getMappedStyle('selectaffordancetopcornerbounded'));
      bottomIcon.classList.add(this.getMappedStyle('selectaffordancebottomcornerbounded'));
    }
  } else if (bottomIcon) {
    let styleKey;
    if (!bounded && axis === 'row') {
      styleKey = 'selectaffordancebottomrow';
    } else if (!bounded && axis === 'column') {
      styleKey = 'selectaffordancebottomcolumn';
    } else {
      styleKey = 'selectaffordancebottomcornerbounded';
    }
    bottomIcon.classList.add(this.getMappedStyle(styleKey));
  }
};

/**
 * Removes selection affordance handles rounded borders that are dependent on selection mode.
 * @private
 */
DvtDataGrid.prototype._clearRoundedAffordanceClasses = function (topIcon, bottomIcon) {
  if (topIcon) {
    topIcon.classList.remove(this.getMappedStyle('selectaffordancetopcornerbounded'));
    topIcon.classList.remove(this.getMappedStyle('selectaffordancetopcolumn'));
    topIcon.classList.remove(this.getMappedStyle('selectaffordancetoprow'));
  }
  bottomIcon.classList.remove(this.getMappedStyle('selectaffordancebottomcornerbounded'));
  bottomIcon.classList.remove(this.getMappedStyle('selectaffordancebottomcolumn'));
  bottomIcon.classList.remove(this.getMappedStyle('selectaffordancebottomrow'));
};

/**
 * Finds and removes the touch selection icons from the DOM
 * @private
 */
DvtDataGrid.prototype._removeTouchSelectionAffordance = function (force) {
  if ((this._isDatabodyCellActive() || force) && this.m_topSelectIconContainer &&
      this.m_topSelectIconContainer.parentNode) {
    this.m_topSelectIconContainer.parentNode.removeChild(this.m_topSelectIconContainer);
    this.m_bottomSelectIconContainer.parentNode.removeChild(this.m_bottomSelectIconContainer);
  }
};

/**
 * Finds and moves the touch selection affordances based on the old and new selection
 * @private
 */
DvtDataGrid.prototype._moveTouchSelectionAffordance = function () {
  var topRowCells;
  var bottomRowCells;
  var dir = this.getResources().isRTLMode() ? 'right' : 'left';
  var selection = this.GetSelection();

  if (selection.length > 0) {
    var selectionMode = this.m_options.getSelectionMode();
    var iconSize = this._getTouchSelectionAffordanceSize();
    if (this.m_topSelectIconContainer != null && this.m_bottomSelectIconContainer != null) {
      var topSortIcon = this.m_topSelectIconContainer.firstElementChild;
      var bottomSortIcon = this.m_bottomSelectIconContainer.firstElementChild;
      this._clearRoundedAffordanceClasses(topSortIcon, bottomSortIcon);
      if (selectionMode === 'row' || (selection[selection.length - 1].endIndex.row !== -1 &&
        selection[selection.length - 1].endIndex.column === -1)) {
        // row selection checks if selection mode is row or a row header was clicked and we need to perform row selection
        // add rounded borders
        this._addRoundedAffordanceClasses(topSortIcon, bottomSortIcon, false, 'row');
        const left = ((this.getElementWidth(this.m_databody) / 2) + this.m_currentScrollLeft) -
        (iconSize / 2);
        this.setElementDir(this.m_topSelectIconContainer, left, dir);
        this.setElementDir(this.m_bottomSelectIconContainer, left, dir);
        topRowCells = this._getAxisCellsByKey(selection[selection.length - 1].startKey.row, 'row');
        bottomRowCells = this._getAxisCellsByKey(selection[selection.length - 1].endKey.row, 'row');
        let bottomRowCell;
        if (bottomRowCells && bottomRowCells.length) {
          bottomRowCell = bottomRowCells[0];
        }
        if (!bottomRowCell) {
          let rangeIndex = this.createIndex(-1, 0);
          let cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));
          bottomRowCell = cells[cells.length - 1];
        }
        this.setElementDir(this.m_topSelectIconContainer,
          (this.getElementDir(topRowCells[0], 'top') - this.m_touchSelectionAffordanceHeight - 1), 'top');
        this.setElementDir(this.m_bottomSelectIconContainer,
          (this.getElementDir(bottomRowCell, 'top') + this.getElementHeight(bottomRowCell)
          - iconSize + this.m_touchSelectionAffordanceHeight),
          'top');
      } else if (selection[selection.length - 1].endIndex.column !== -1 &&
                 selection[selection.length - 1].endIndex.row === -1) {
        // col selection
        // add rounded borders
        this._addRoundedAffordanceClasses(topSortIcon, bottomSortIcon, false, 'column');
        topRowCells = this._getAxisCellsByKey(selection[selection.length - 1].startKey.column,
                                              'column');
        bottomRowCells = this._getAxisCellsByKey(selection[selection.length - 1].endKey.column,
                                                 'column');

        var top = ((this.getElementHeight(this.m_databody) / 2) + this.m_currentScrollTop) -
            (iconSize / 2);
        this.setElementDir(this.m_topSelectIconContainer, top, 'top');
        this.setElementDir(this.m_bottomSelectIconContainer, top, 'top');
         // -2 for both borders
         this.setElementDir(this.m_topSelectIconContainer,
                            this.getElementDir(topRowCells[0], 'left') - this.m_touchSelectionAffordanceWidth - 2, 'left');
         // -1 for border
         this.setElementDir(this.m_bottomSelectIconContainer,
                            this.getElementDir(bottomRowCells[0], 'left') + this.getElementWidth(bottomRowCells[0])
                            - (iconSize - this.m_touchSelectionAffordanceWidth - 1),
                            'left');
      } else {
        // Cell selection
        // add rounded borders
        this._addRoundedAffordanceClasses(topSortIcon, bottomSortIcon, true);

        // get the cells for left/right alignment
        var topIconCell = this._getCellByIndex(selection[selection.length - 1].startIndex);
        var bottomIconCell = this._getCellByIndex(selection[selection.length - 1].endIndex);
        if (!bottomIconCell) {
          let rangeIndex = this.createIndex(-1, -1);
          let cells = this.getElementsInRange(this.createRange(rangeIndex, rangeIndex));
          bottomIconCell = cells[cells.length - 1];
        }
        // -1 for border
        this.setElementDir(this.m_topSelectIconContainer,
                           (this.getElementDir(topIconCell, 'top') - this.m_touchSelectionAffordanceHeight - 1), 'top');
        this.setElementDir(this.m_bottomSelectIconContainer,
                           (this.getElementDir(bottomIconCell, 'top') + this.getElementHeight(bottomIconCell)
                           - iconSize + this.m_touchSelectionAffordanceWidth),
                           'top');
        // -2 for both borders
        this.setElementDir(this.m_topSelectIconContainer,
                           this.getElementDir(topIconCell, dir) -
                           this.m_touchSelectionAffordanceWidth - 2, dir);
        this.setElementDir(this.m_bottomSelectIconContainer,
                           this.getElementDir(bottomIconCell, dir)
                           + this.getElementWidth(bottomIconCell) -
                           (iconSize - this.m_touchSelectionAffordanceWidth),
                           dir);
      }

      if (this.m_topSelectIconContainer.parentNode == null) {
        this.m_databody.firstChild.appendChild(this.m_topSelectIconContainer); // @HTMLUpdateOK
        this.m_databody.firstChild.appendChild(this.m_bottomSelectIconContainer); // @HTMLUpdateOK
      }
    }
  }
};
/**
 * Moves the touch selection affordances horizontally in the row to ensure they are in the viewport.
 * Only moved in row selection.
 * @private
 */
DvtDataGrid.prototype._scrollTouchSelectionAffordance = function () {
  var newLeft;
  var dir;
  var selectionMode = this.m_options.getSelectionMode();

  if (selectionMode === 'row') {
    if (this.m_topSelectIconContainer != null) {
      dir = this.getResources().isRTLMode() ? 'right' : 'left';
      newLeft = (this.getElementWidth(this.m_databody) / 2) + this.m_currentScrollLeft;
      this.setElementDir(this.m_topSelectIconContainer, newLeft, dir);
      this.setElementDir(this.m_bottomSelectIconContainer, newLeft, dir);
    }
  } else if (selectionMode === 'cell' && this.isHeaderSelectionType(this.m_selectionFrontier)) {
    if (this.m_topSelectIconContainer != null) {
      if (this.m_selectionFrontier.axis.indexOf('row') !== -1) {
        dir = this.getResources().isRTLMode() ? 'right' : 'left';
        newLeft = (this.getElementWidth(this.m_databody) / 2) + this.m_currentScrollLeft;
        this.setElementDir(this.m_topSelectIconContainer, newLeft, dir);
        this.setElementDir(this.m_bottomSelectIconContainer, newLeft, dir);
      } else {
        var newTop = (this.getElementHeight(this.m_databody) / 2) + this.m_currentScrollTop;
        this.setElementDir(this.m_topSelectIconContainer, newTop, 'top');
        this.setElementDir(this.m_bottomSelectIconContainer, newTop, 'top');
      }
    }
  }
};

/**
 * Get the touch affordance icon size
 * @return {number} the touch affordance icon size
 * @private
 */
DvtDataGrid.prototype._getTouchSelectionAffordanceSize = function () {
  if (this.m_touchSelectionAffordanceSize == null) {
    var div = document.createElement('div');
    div.className = this.getMappedStyle('toucharea');
    div.style.visibilty = 'hidden';
    div.style.top = '0px';
    div.style.visibilty = '0px';
    this.m_root.appendChild(div); // @HTMLUpdateOK
    var divWidth = div.offsetWidth;
    this.m_root.removeChild(div);
    this.m_touchSelectionAffordanceSize = divWidth;
  }
  return this.m_touchSelectionAffordanceSize;
};

DvtDataGrid.SORT_ANIMATION_DURATION = 800;
/**
 * Event handler for handling mouse over event on sort container.
 * @param {Event} event the DOM event
 * @private
 */
DvtDataGrid.prototype._handleSortContainerMouseOver = function (event) {
  var target = /** @type {Element} */ (event.target);
  var header = this.findHeader(target);
  var sortIcon = this._getSortIcon(header);
  // if we are hovering the icon add hover class
  if (this.m_utils.containsCSSClassName(event.currentTarget, this.getMappedStyle('sortIcon'))) {
    this.m_utils.addCSSClassName(event.currentTarget, this.getMappedStyle('hover'));
    this.m_utils.addCSSClassName(sortIcon, this.getMappedStyle('hover'));
    this.m_utils.addCSSClassName(sortIcon, this.getMappedStyle('enabled'));
    this.m_utils.removeCSSClassName(sortIcon, this.getMappedStyle('disabled'));
  }
};

/**
 * Event handler for handling mouse out event on headers.
 * @param {Event} event the DOM event
 * @private
 */
DvtDataGrid.prototype._handleSortMouseOut = function (event) {
  if (!this._databodyEmpty()) {
    var target = /** @type {Element} */ (event.target);
    var relatedTarget = /** @type {Element} */ (event.relatedTarget);
    var header = this.findHeader(target);
    var sortIcon;
    // if there is no header or we didn't just exit the content of the header
    if (header == null ||
        relatedTarget == null ? true : header !== this.findHeader(relatedTarget)) {
      this._displaySortIcon(header);
    }
    sortIcon = this._getSortIcon(header);
    if (sortIcon) {
      this.m_utils.removeCSSClassName(sortIcon, this.getMappedStyle('hover'));
      if (!this.m_utils.containsCSSClassName(sortIcon, this.getMappedStyle('selected'))) {
        this.m_utils.addCSSClassName(sortIcon, this.getMappedStyle('disabled'));
        this.m_utils.removeCSSClassName(sortIcon, this.getMappedStyle('enabled'));
      }
    }
    var sortContainer = this._getSortContainer(header);
    if (sortContainer) {
      this.m_utils.removeCSSClassName(sortContainer, this.getMappedStyle('hover'));
      this.m_utils.removeCSSClassName(sortContainer, this.getMappedStyle('selected'));
    }
  }
};

/**
 * Add the selected color on mousedown
 * @param {Element} icon the icon to set selected on
 * @private
 */
DvtDataGrid.prototype._handleSortIconMouseDown = function (icon) {
  if (!this._databodyEmpty()) {
    this.m_utils.addCSSClassName(icon, this.getMappedStyle('selected'));
    this.m_utils.removeCSSClassName(icon, this.getMappedStyle('disabled'));
  }
};

/**
 * Show or hide the sort indicator icons.
 * @param {Element} header the dom element of the header to switch icon direction in
 * @param {string} direction ascending or descending to switch to
 * @private
 */
DvtDataGrid.prototype._toggleSortIconDirection = function (header, direction) {
  if (header != null) {
    // shows the sort indicator
    var icon = this._getSortIcon(header);
    if (direction === 'descending' &&
        (this.m_utils.containsCSSClassName(icon, this.getMappedStyle('sortascending')) ||
        this.m_utils.containsCSSClassName(icon, this.getMappedStyle('sortdefault')))) {
      this.m_utils.removeCSSClassName(icon, this.getMappedStyle('sortascending'));
      this.m_utils.removeCSSClassName(icon, this.getMappedStyle('sortdefault'));
      this.m_utils.addCSSClassName(icon, this.getMappedStyle('sortdescending'));
    } else if (direction === 'ascending' &&
              (this.m_utils.containsCSSClassName(icon, this.getMappedStyle('sortdescending')) ||
              this.m_utils.containsCSSClassName(icon, this.getMappedStyle('sortdefault')))) {
      this.m_utils.removeCSSClassName(icon, this.getMappedStyle('sortdescending'));
      this.m_utils.removeCSSClassName(icon, this.getMappedStyle('sortdefault'));
      this.m_utils.addCSSClassName(icon, this.getMappedStyle('sortascending'));
    } else if (direction === 'default') {
      this.m_utils.removeCSSClassName(icon, this.getMappedStyle('sortdescending'));
      this.m_utils.removeCSSClassName(icon, this.getMappedStyle('sortascending'));
      this.m_utils.addCSSClassName(icon, this.getMappedStyle('sortdefault'));
    }
  }
};

/**
 * Show the sort indicator icons.
 * @param {Element|undefined|null} header the dom event
 * @private
 */
DvtDataGrid.prototype._displaySortIcon = function (header) {
  var sorted = false;
  if (header != null) {
    var icon = this._getSortIcon(header);
    if (this.m_sortInfo != null) {
      sorted = this.m_sortInfo.key === this._getKey(header);
    }
    if (sorted) {
      this.m_utils.addCSSClassName(icon, this.getMappedStyle('default'));
    }
  }
};

/**
 * Creates the sort indicator icons and the panel around them.
 * @param {Object} headerContext a header context object, contianing key
 * @return {Element} the sort indicator icons panel
 * @private
 */
DvtDataGrid.prototype._buildSortIcon = function (headerContext, header, axis) {
  // sort container is used to create fade effect
  var sortContainer = document.createElement('div');
  this.m_utils.addCSSClassName(sortContainer, this.getMappedStyle('iconContainer'));
  this.m_utils.addCSSClassName(sortContainer, this.getMappedStyle('sortIcon'));

  var sortIcon = document.createElement('div');
  var iconClassString = this.getMappedStyle('icon') + ' ' + this.getMappedStyle('clickableicon');
  var key = (this.m_sortInfo != null && this.m_sortInfo.axis === axis) ? this.m_sortInfo.key : null;
  // handles the case where we scroll the header which was sorted on, off screen and come back to them
  if (headerContext.key === key) {
    var direction = (this.m_sortInfo != null && this.m_sortInfo.axis === axis) ?
                    this.m_sortInfo.direction : null;
    if (direction === 'ascending') {
      sortIcon.className = this.getMappedStyle('sortascending') + ' ' + iconClassString;
      header.setAttribute(this.getResources().getMappedAttribute('sortDir'), direction); // @HTMLUpdateOK
    } else if (direction === 'descending') {
      sortIcon.className = this.getMappedStyle('sortdescending') + ' ' + iconClassString;
      header.setAttribute(this.getResources().getMappedAttribute('sortDir'), direction); // @HTMLUpdateOK
    } else {
      sortIcon.className = this.getMappedStyle('sortdefault') + ' ' + iconClassString;
    }
  } else {
    iconClassString += ' ' + this.getMappedStyle('disabled');
    sortIcon.className = this.getMappedStyle('sortdefault') + ' ' + iconClassString;
  }
  sortContainer.appendChild(sortIcon); // @HTMLUpdateOK
  sortContainer.addEventListener('mouseover', this._handleSortContainerMouseOver.bind(this));
  return sortContainer;
};

/**
 * Handles sorting using keyboard (enter key while focus on header).  See HandleHeaderKeyDown.
 * @param {Element} header header being sorted on
 * @param {Event} event DOM keyboard event triggering sort
 * @private
 */
DvtDataGrid.prototype._handleKeyboardSort = function (header, event) {
  if (!this._databodyEmpty()) {
    var direction = header.getAttribute(this.getResources().getMappedAttribute('sortDir'));
    if (direction == null || direction === 'descending') {
      direction = 'ascending';
    } else {
      direction = 'descending';
    }

    this._doHeaderSort(event, header, direction);
  }
};

/**
 * Handles click on the header, this would perform the sort operation.
 * @param {Event} event the DOM event
 * @param {string=} direction ascending or descending to sort on
 * @private
 */
DvtDataGrid.prototype._handleHeaderSort = function (event, direction) {
  if (!this._databodyEmpty()) {
    var target = /** @type {Element} */ (event.target);

    var header = this.findHeader(target);
    if (header != null) {
      // use the class name to determine if it's asecnding or descending
      if (direction == null) {
        if (this.m_sortInfo != null && this.m_sortInfo.key === this._getKey(header)) {
          if (this.m_sortInfo.direction === 'ascending') {
            // eslint-disable-next-line no-param-reassign
            direction = 'descending';
          } else {
            // eslint-disable-next-line no-param-reassign
            direction = 'ascending';
          }
        } else {
          // we should get here on inital touch sort only
          // eslint-disable-next-line no-param-reassign
          direction = 'ascending';
        }
      }
      this._doHeaderSort(event, header, direction);
    }
  }
};

/**
 * Handles click on the header, this would perform the sort operation.
 * @param {Event} event the DOM event
 * @param {string} direction ascending or descending to switch to
 * @param {Element|undefined} header the header to sort on
 * @private
 */
DvtDataGrid.prototype._handleCellSort = function (event, direction, header) {
  if (header != null && !this._databodyEmpty()) {
    this._removeTouchSelectionAffordance();
    this._doHeaderSort(event, header, direction);
  }
};

/**
 * Handles click on the header, this would perform the sort operation.
 * @param {Event} event the DOM event
 * @param {Element} header the header element
 * @param {string} direction the sort direction
 * @private
 */
DvtDataGrid.prototype._doHeaderSort = function (event, header, direction) {
  if (this.m_isSorting !== true) {
    this.m_delayedSort = null;

    // get the key and axis
    var key = this._getKey(header);
    var axis = this._getAxis(header);

    this._removeSortSelection();

    // needed for toggle and screenreader
    header.setAttribute(this.getResources().getMappedAttribute('sortDir'), direction); // @HTMLUpdateOK
    this.m_sortInfo = { event: event, key: key, axis: axis, direction: direction, header: header };


    // flip the icon direction
    this._toggleSortIconDirection(header, direction);
    this._addSortSelection();

    // update screen reader alert
    this._setAccInfoText(
      direction === 'ascending' ? 'accessibleSortAscending' : 'accessibleSortDescending',
      { id: key });


    // creates the criteria object and invoke sort on the data source
    if (direction != null && key != null && axis != null) {
      if (this._isDataGridProvider()) {
        this._fireSortRequestEvent(axis);
      } else {
        this.m_isSorting = true;
        // show status message
        this.showStatusText();

        // invoke sort
        var criteria = { axis: axis, key: key, direction: direction };
        this.getDataSource().sort(criteria, {
          success: this._handleSortSuccess.bind(this),
          error: this._handleSortError.bind(this)
        });
      }
    }
  } else {
    this.m_delayedSort = { event: event, header: header, direction: direction };
  }
};

/**
 * Callback method invoked when the sort operation failed.
 * @private
 */
DvtDataGrid.prototype._handleSortError = function () {
  this.hideStatusText();
};

/**
 * Remove the selected style class from the previous sorted sort icon, and add disabled back to it
 * @private
 */
DvtDataGrid.prototype._removeSortSelection = function () {
  if (this.m_sortInfo != null) {
    let axis = this.m_sortInfo.axis;
    // get the header that was sorted on and the icon within it based on the values stored in this.m_sortInfo
    var oldSortedHeader;
    if (axis === 'column') {
      oldSortedHeader = this._findHeaderByKey(this.m_sortInfo.key, this.m_colHeader,
        this.getMappedStyle('colheadercell'));
    } else {
      oldSortedHeader = this._findHeaderByKey(this.m_sortInfo.key, this.m_rowHeader,
        this.getMappedStyle('rowheadercell'));
    }
    oldSortedHeader.removeAttribute(this.getResources().getMappedAttribute('sortDir'));
    var oldsortIcon = this._getSortIcon(oldSortedHeader);
    // flip icon back to default
    this._toggleSortIconDirection(oldSortedHeader, 'default');
    if (this.m_sortInfo.direction === 'descending') {
      // switch back to the default ascending icon
      this.m_utils.removeCSSClassName(oldsortIcon, this.getMappedStyle('sortdescending'));
      this.m_utils.addCSSClassName(oldsortIcon, this.getMappedStyle('sortascending'));
    }
    // disable the icon to hide it, remove the selected style
    this.m_utils.addCSSClassName(oldsortIcon, this.getMappedStyle('disabled'));
    this.m_utils.removeCSSClassName(oldsortIcon, this.getMappedStyle('enabled'));
    this.m_utils.removeCSSClassName(oldsortIcon, this.getMappedStyle('default'));
    this.m_utils.removeCSSClassName(oldsortIcon, this.getMappedStyle('selected'));
    this.m_utils.removeCSSClassName(this._getSortContainer(oldSortedHeader),
      this.getMappedStyle('enabled'));
  }
};

/**
 * Add the selected style class to the newly sorted sort icon and remove disabled from it
 * @private
 */
DvtDataGrid.prototype._addSortSelection = function () {
  if (this.m_sortInfo != null) {
    let axis = this.m_sortInfo.axis;
    // get the header that is sorted on and the icon within it based on the values stored in this.m_sortInfo
    var sortedHeader;
    if (axis === 'column') {
      sortedHeader = this._findHeaderByKey(this.m_sortInfo.key, this.m_colHeader,
        this.getMappedStyle('colheadercell'));
    } else {
      sortedHeader = this._findHeaderByKey(this.m_sortInfo.key, this.m_rowHeader,
        this.getMappedStyle('rowheadercell'));
    }
    var sortIcon = this._getSortIcon(sortedHeader);

    // select the icon to show it, remove the disabled style
    this.m_utils.addCSSClassName(sortIcon, this.getMappedStyle('default'));
    this.m_utils.removeCSSClassName(sortIcon, this.getMappedStyle('disabled'));
    this.m_utils.addCSSClassName(sortIcon, this.getMappedStyle('selected'));
  }
};

/**
 * Determine the axis of the header.
 * @param {Element} header the header to determine the axis, returns either "row" or "column".
 * @return {string|null} the axis of the header
 * @private
 */
DvtDataGrid.prototype._getAxis = function (header) {
  var columnHeaderCellClassName = this.getMappedStyle('colheadercell');
  var rowHeaderCellClassName = this.getMappedStyle('rowheadercell');

  if (this.m_utils.containsCSSClassName(header, columnHeaderCellClassName)) {
    return 'column';
  }

  if (this.m_utils.containsCSSClassName(header, rowHeaderCellClassName)) {
    return 'row';
  }

  return null;
};

/**
 * Callback method invoked when the sort operation completed successfully.
 * @private
 */
DvtDataGrid.prototype._handleSortSuccess = function () {
  // hide the message
  this.hideStatusText();

  // sort is completed successfully, now fetch the sorted data
  if (this._isDatabodyCellActive()) {
    // scroll position should go to the new active cell location if virtual
    this._indexes({
      row: this.m_active.keys.row,
      column: this.m_active.keys.column
    }, this._handlePreSortScrolling);
  } else {
    // scroll position should remain unchanged if high-water mark or virtual without an active cell
    this._fetchForSort(this.m_startRow, (this.m_endRow - this.m_startRow) + 1, false);
  }
};

/**
 * Handle scrolling of the datagrid before fetching the data
 * @param {Object} indexes index of the new location of the active cell
 */
DvtDataGrid.prototype._handlePreSortScrolling = function (indexes) {
  var rowIndex = indexes.row === -1 ? 0 : indexes.row;
  var cellTop = rowIndex * this.m_avgRowHeight;
  var cellBottom = cellTop + this.m_avgRowHeight;

  var isHighWatermark = this._isHighWatermarkScrolling();
  var isInVisibleRange = this.m_currentScrollTop <= cellTop &&
    cellBottom <= this.m_currentScrollTop + this.getElementHeight(this.m_databody);

  // cell is in rendered range and visible, or high-water mark regardless of visibilty,
  // do a refetch of the current viewport and no scrolling
  if (isInVisibleRange || (isHighWatermark)) { // && !isInRenderedRange))
    this._fetchForSort(this.m_startRow, (this.m_endRow - this.m_startRow) + 1, false);
  } else {
    // we have decided not to prescroll on high-water mark because the active cell
    // was hard to follow through the animation, if we wanted that behavior in the
    // future simply follow the format commented out below and above
    // cell is in rendered range but not visible on high-water mark,
    // do a scroll to the new position with a refresh of the whole viewport
    // else if (isHighWatermark && isInRenderedRange)
    // {
    //    // set a new scrollTop to the top of that cell or the closest it can be
    //    this.m_currentScrollTop = Math.min(cellTop, this.m_scrollHeight);
    //    this._fetchForSort(this.m_startRow, this.m_endRow - this.m_startRow + 1, true);
    // }
    //
    // in virtual scrolling and not outside of the visible range scroll to the new location and refresh
    // get the scroll top that it will need to be
    this.m_currentScrollTop = Math.min(cellTop, this._getMaxScrollHeight());

    // find the start row we need to fetch at that scroll position
    var startRow = Math.floor(this.m_currentScrollTop / this.m_avgRowHeight);

    var startRowPixel = startRow * this.m_avgRowHeight;
    // reset ranges on rows
    this.m_startRow = startRow;
    this.m_endRow = -1;
    this.m_startRowHeader = startRow;
    this.m_endRowHeader = -1;
    this.m_startRowPixel = startRowPixel;
    this.m_endRowPixel = startRowPixel;
    this.m_startRowHeaderPixel = startRowPixel;
    this.m_endRowHeaderPixel = startRowPixel;

    this._fetchForSort(startRow, null, true);
  }
};

/**
 * A method to fetch data with the correct sort callbacks
 * @param {number} startRow
 * @param {number|null} rowCount
 * @param {boolean} scroll true if we need to pre scroll the datagrid
 */
DvtDataGrid.prototype._fetchForSort = function (startRow, rowCount, scroll) {
  var rowHeaderFragment = document.createDocumentFragment();
  var endRowHeaderFragment = document.createDocumentFragment();
  this.fetchHeaders('row', startRow, rowHeaderFragment, endRowHeaderFragment, rowCount, {
    success: this.handleHeadersFetchSuccessForSort.bind(this),
    error: this.handleCellsFetchError
  });
  this.fetchCells(this.m_databody, startRow, this.m_startCol, rowCount,
    (this.m_endCol - this.m_startCol) + 1, {
      success: this.handleCellsFetchSuccessForSort.bind(this, rowHeaderFragment,
        endRowHeaderFragment,
        scroll),
      error: this.handleCellsFetchError
    });
};

/**
 * Handle a successful call to the data source fetchHeaders for sorting. Used to populate the new headers fragment.
 * @param {Object} headerSet - an array of headers returned from the dataSource
 * @param {Object} headerRange - {"axis":,"start":,"count":,"header":}
 * @param {boolean} rowInsert - if this is triggered by a row insert event
 * @protected
 */
DvtDataGrid.prototype.handleHeadersFetchSuccessForSort =
  function (headerSet, headerRange, endHeaderSet, rowInsert) {
    var headerCount;
    var c;
    var index;
    var totalRowHeight;
    var returnVal;
    var className;
    var renderer;
    var axis = headerRange.axis;
    var start = headerRange.start;
    var headerFragment = headerRange.header;
    var endHeaderFragment = headerRange.endHeader;

    // remove fetching message
    this.m_fetching[axis] = false;

    if (headerSet != null) {
      // add the headers to the row header
      headerCount = headerSet.getCount();
      totalRowHeight = 0;
      c = 0;
      className = this.getMappedStyle('headercell') + ' ' +
        this.getMappedStyle('rowheadercell');
      renderer = this.getRendererOrTemplate('row');
      while (headerCount - c > 0) {
        index = start + c;
        returnVal = this.buildLevelHeaders(headerFragment, index, 0, 0,
          this.m_startRowPixel + totalRowHeight, true,
          rowInsert, renderer, headerSet, 'row',
          className, this.m_rowHeaderLevelCount);
        c += returnVal.count;
        totalRowHeight += returnVal.totalHeaderDimension;
      }
      this.m_endRowHeader = this.m_startRowHeader + (headerCount - 1);
      this.m_endRowHeaderPixel = this.m_startRowHeaderPixel + totalRowHeight;
    }

    if (endHeaderSet != null) {
      headerCount = endHeaderSet.getCount();
      totalRowHeight = 0;
      c = 0;
      className = this.getMappedStyle('endheadercell') + ' ' +
        this.getMappedStyle('rowendheadercell');
      renderer = this.getRendererOrTemplate('rowEnd');
      while (headerCount - c > 0) {
        index = start + c;
        returnVal = this.buildLevelHeaders(endHeaderFragment, index, 0, 0,
          this.m_startRowPixel + totalRowHeight, true,
          rowInsert, renderer, endHeaderSet, 'rowEnd',
          className, this.m_rowEndHeaderLevelCount);
        c += returnVal.count;
        totalRowHeight += returnVal.totalHeaderDimension;
      }
      this.m_endRowEndHeader = this.m_startRowEndHeader + (headerCount - 1);
      this.m_endRowEndHeaderPixel = this.m_startRowEndHeaderPixel + totalRowHeight;
    }

    // end fetch
    this._signalTaskEnd();
  };

/**
 * Handle a successful call to the data source fetchCells after sort.
 * @param {DocumentFragment|Element} newRowHeaderElements a document fragment containing the row headers and the fragment
 * @param {boolean|null} scroll true if we need to pre scroll
 * @param {Object} cellSet a CellSet object which encapsulates the result set of cells
 * @param {Array.<Object>} cellRange [rowRange, columnRange] - [{"axis":,"start":,"count":},{"axis":,"start":,"count":,"databody":,"scroller":}]
 */
DvtDataGrid.prototype.handleCellsFetchSuccessForSort =
  function (newRowHeaderElements, newRowEndHeaderElements, scroll, cellSet, cellRange) {
    var animate;

    this.m_fetching.cells = false;

    var duration = DvtDataGrid.SORT_ANIMATION_DURATION;

    // size the grid if fetch is done
    if (this.isFetchComplete()) {
      this.hideStatusText();
    }

    // obtain params for _addCellsToFragment
    var rowRange = cellRange[0];
    var rowStart = rowRange.start;
    var rowCount = cellSet.getCount('row');

    var columnRange = cellRange[1];
    var columnStart = columnRange.start;

    // the rows AFTER sort should be inside the newCellElements fragment
    var newCellElements = document.createDocumentFragment();

    var returnVal = this._addCellsToFragment(newCellElements, cellSet, rowStart,
      this.m_startRowPixel, columnStart,
      this.m_startColPixel);
    this.m_endRow = this.m_startRow + (rowCount - 1);
    this.m_endRowPixel = this.m_startRowPixel + returnVal.totalRowHeight;

    var oldCellElements = this.m_databody.firstChild;
    var oldRowHeaderElements = this.m_rowHeader.firstChild;
    var oldRowEndHeaderElements = this.m_rowEndHeader.firstChild;

    if (scroll === true) {
      // disable animation on virtual scrolling
      animate = this._isHighWatermarkScrolling();

      // scroll the databody
      if (!this.m_utils.isTouchDevice()) {
        this.m_silentScroll = true;
        this.m_databody.scrollTop = this.m_currentScrollTop;
        this._syncScroller();
      } else {
        // for touch we'll call scrollTo directly instead of relying on scroll event to fire due to performance
        this._disableTouchScrollAnimation();
        this.scrollTo(this.m_currentScrollLeft, this.m_currentScrollTop);
      }
    }

    // if there's only one row we don't need to animate
    // don't animate on multi-level headers
    if (!duration || !this.m_utils.supportsTransitions() || rowCount === 1 ||
        (this.m_rowHeaderLevelCount > 1 && this.m_rowHeaderLevelCount != null) ||
        animate === false) {
      // start task since both animation/non use handle sort end
      this._signalTaskStart();
      this._handleSortEnd(newCellElements, newRowHeaderElements, newRowEndHeaderElements);
    } else if (this.m_isCustomElementCallback()) {
      this._signalTaskStart('processing sort render');

      // need to add the cells to dom to render offscreen before animation
      let cellDiv = document.createElement('div');
      let rDiv = document.createElement('div');
      let reDiv = document.createElement('div');

      cellDiv.setAttribute(this.getResources().getMappedAttribute('busyContext'), ''); // @HTMLUpdateOK
      rDiv.setAttribute(this.getResources().getMappedAttribute('busyContext'), ''); // @HTMLUpdateOK
      reDiv.setAttribute(this.getResources().getMappedAttribute('busyContext'), ''); // @HTMLUpdateOK

      // this is our helper hidden accessible class to hide the add/remove.
      cellDiv.className = this.getMappedStyle('info');
      rDiv.className = this.getMappedStyle('info');
      reDiv.className = this.getMappedStyle('info');

      cellDiv.appendChild(newCellElements); // @HTMLUpdateOK
      rDiv.appendChild(newRowHeaderElements); // @HTMLUpdateOK
      reDiv.appendChild(newRowEndHeaderElements); // @HTMLUpdateOK

      this.m_root.appendChild(cellDiv); // @HTMLUpdateOK
      this.m_root.appendChild(rDiv); // @HTMLUpdateOK
      this.m_root.appendChild(reDiv); // @HTMLUpdateOK

      var cellBusyContext = Context.getContext(cellDiv).getBusyContext().whenReady();
      var rBusyContext = Context.getContext(rDiv).getBusyContext().whenReady();
      var reBusyContext = Context.getContext(reDiv).getBusyContext().whenReady();

      Promise.all([cellBusyContext, rBusyContext, reBusyContext]).then(function () {
        while (cellDiv.childNodes.length > 0) {
          newCellElements.appendChild(cellDiv.childNodes[0]); // @HTMLUpdateOK
        }
        while (rDiv.childNodes.length > 0) {
          newRowHeaderElements.appendChild(rDiv.childNodes[0]); // @HTMLUpdateOK
        }
        while (reDiv.childNodes.length > 0) {
          newRowEndHeaderElements.appendChild(reDiv.childNodes[0]); // @HTMLUpdateOK
        }

        this.m_root.removeChild(cellDiv);
        this.m_root.removeChild(rDiv);
        this.m_root.removeChild(reDiv);

        this._signalTaskEnd();

        this.processSortAnimationToPosition(duration, 0, 'ease-in',
          oldRowHeaderElements, newRowHeaderElements,
          oldCellElements, newCellElements,
          oldRowEndHeaderElements, newRowEndHeaderElements);
      }.bind(this));
    } else {
      this.processSortAnimationToPosition(duration, 0, 'ease-in',
        oldRowHeaderElements, newRowHeaderElements,
        oldCellElements, newCellElements,
        oldRowEndHeaderElements, newRowEndHeaderElements);
    }

    // end fetch
    this._signalTaskEnd();
  };

/**
 * Handles a sort complete by replacing the dom with the new headers and cells,
 * restoring active and firing a sort event
 * @param {DocumentFragment} newCellElements
 * @param {DocumentFragment|Element} newRowHeaderElements
 */
DvtDataGrid.prototype._handleSortEnd =
  function (newCellElements, newRowHeaderElements, newRowEndHeaderElements,
    childNodesList, rowHeaderList, rowEndHeaderList) {
    var headerContent;

    if (newRowHeaderElements.childNodes.length > 1) {
      headerContent = this.m_rowHeader.firstChild;
      if (rowHeaderList) {
        for (let i = 0; i < rowHeaderList.length; i++) {
          let child = rowHeaderList[i];
          newRowHeaderElements.insertBefore(child.element,
            newRowHeaderElements.childNodes[child.index]);
        }
      }
      this.m_utils.empty(headerContent);
      headerContent.appendChild(newRowHeaderElements); // @HTMLUpdateOK
      this.m_subtreeAttachedCallback(headerContent);
    }

    if (newRowEndHeaderElements.childNodes.length > 1) {
      headerContent = this.m_rowEndHeader.firstChild;
      if (rowEndHeaderList) {
        for (let i = 0; i < rowEndHeaderList.length; i++) {
          let child = rowEndHeaderList[i];
          newRowEndHeaderElements.insertBefore(child.element,
            newRowEndHeaderElements.childNodes[child.index]);
        }
      }
      this.m_utils.empty(headerContent);
      headerContent.appendChild(newRowEndHeaderElements); // @HTMLUpdateOK
      this.m_subtreeAttachedCallback(headerContent);
    }

    var databodyContent = this.m_databody.firstChild;
    var rowHeaderContent = this.m_rowHeader.firstChild;
    var rowEndHeaderContent = this.m_rowEndHeader.firstChild;

    if (childNodesList) {
      for (let i = 0; i < childNodesList.length; i++) {
        newCellElements.appendChild(childNodesList[i]);
      }
    }
    this._emptyDatabody(databodyContent);
    this._populateDatabody(databodyContent, newCellElements);

    // restore active cell
    this._fireSortEvent();
    this._restoreActive();
    this.m_isSorting = false;
    this._doDelayedSort();

    // end animation/sort
    this.m_animating = false;
    for (let i = 0; i < databodyContent.childNodes.length; i++) {
      this.removeTransformMoveStyle(databodyContent.childNodes[i]);
    }

    if (this.m_endRowHeader !== -1) {
      for (let i = 0; i < rowHeaderContent.childNodes.length; i++) {
        this.removeTransformMoveStyle(rowHeaderContent.childNodes[i]);
      }
    }

    if (this.m_endRowEndHeader !== -1) {
      for (let i = 0; i < rowEndHeaderContent.childNodes.length; i++) {
        this.removeTransformMoveStyle(rowEndHeaderContent.childNodes[i]);
      }
    }
    this._signalTaskEnd();

    // check event queue for outstanding model events
    this._runModelEventQueue();
  };

/**
 * The main method for animation of the DataGrid rows from before-sort to the after-sort positions
 * @param {number} duration the duration of animation
 * @param {number} delayOffset the initial delay of animation
 * @param {string} timing the easing function
 * @param {Element} oldRowHeaderElements the DOM structure on which the animation will be performed. Initially contains DOM elements in before sorting order
 * @param {DocumentFragment|Element} newRowHeaderElements the element that contains set of sub-elements in "after-sorting" order
 * @param {Object} oldElementSet the DOM structure on which the animation will be performed. Initially contains DOM elements in before sorting order
 * @param {DocumentFragment} newElementSet the element that contains set of sub-elements in "after-sorting" order
 * @private
 */
DvtDataGrid.prototype.processSortAnimationToPosition =
  function (duration, delayOffset, timing, oldRowHeaderElements, newRowHeaderElements,
    oldElementSet, newElementSet, oldRowEndHeaderElements, newRowEndHeaderElements) {
    var rowKey;
    var oldTop;
    var newTop;
    var i;
    var child;
    var oldBottom;
    var newBottom;
    var lastAnimationElement;
    var left;
    var right;

    // initialize variables
    var self = this;
    // animation start
    this._signalTaskStart();
    var rowHeaderSupport = newRowHeaderElements.childNodes.length > 1;
    var rowEndHeaderSupport = newRowEndHeaderElements.childNodes.length > 1;
    var dir = this.getResources().isRTLMode() ? 'right' : 'left';
    var viewportBottom = this._getViewportBottom();
    var viewportLeft = this._getViewportLeft();
    var viewportRight = this._getViewportRight();

    // animation information will be an object of objects as follows {key:{oldTop:val, newTop:val}}
    var animationInformation = {};
    var newAnimationInformation = {};
    var childNodesList = [];

    // loop over the old elements and set their old and new tops which should eb to the bottom of the viewport if visible
    for (i = 0; i < oldElementSet.childNodes.length; i++) {
      child = oldElementSet.childNodes[i];
      rowKey = this._getKey(child, 'row');
      left = this.getElementDir(child, dir);
      right = left + this.getElementWidth(child);
      oldTop = this.getElementDir(child, 'top');
      oldBottom = oldTop + this.getElementHeight(child);

      if (this._isCellBoundaryInViewport(left, right, oldTop, oldBottom)) {
        if (!animationInformation[rowKey]) {
          newTop = viewportBottom;
          animationInformation[rowKey] = { cell: [child], oldTop: oldTop, newTop: newTop };
        } else {
          animationInformation[rowKey].cell.push(child);
        }

        if (newTop - oldTop !== 0) {
          lastAnimationElement = child;
        }
      }
    }

    for (i = 0; i < oldRowHeaderElements.childNodes.length; i++) {
      if (rowHeaderSupport) {
        child = oldRowHeaderElements.childNodes[i];
        rowKey = this._getKey(child, 'row');
        oldTop = this.getElementDir(child, 'top');
        oldBottom = oldTop + this.getElementHeight(child);

        if (this._isCellBoundaryInViewport(viewportLeft, viewportRight, oldTop, oldBottom)) {
          newTop = viewportBottom;
          if (!animationInformation[rowKey]) {
            animationInformation[rowKey] = { oldTop: oldTop, newTop: newTop };
          }
          animationInformation[rowKey].rowHeader = child;
        }
      }
    }

    for (i = 0; i < oldRowEndHeaderElements.childNodes.length; i++) {
      if (rowHeaderSupport) {
        child = oldRowEndHeaderElements.childNodes[i];
        rowKey = this._getKey(child, 'row');
        oldTop = this.getElementDir(child, 'top');
        oldBottom = oldTop + this.getElementHeight(child);

        if (this._isCellBoundaryInViewport(viewportLeft, viewportRight, oldTop, oldBottom)) {
          newTop = viewportBottom;
          if (!animationInformation[rowKey]) {
            animationInformation[rowKey] = { oldTop: oldTop, newTop: newTop };
          }
          animationInformation[rowKey].rowEndHeader = child;
        }
      }
    }

    // loop over the new elements and set their old and new tops
    for (i = 0; i < newElementSet.childNodes.length; i++) {
      child = newElementSet.childNodes[i];
      rowKey = this._getKey(child, 'row');
      newTop = this.getElementDir(child, 'top');

      // if in the old elements, just replace its newTop value
      if (animationInformation[rowKey]) {
        animationInformation[rowKey].newTop = newTop;
      } else {
        // if not in the old elements, create an entry for it with oldTop just outside of view
        oldTop = viewportBottom;
        newBottom = newTop + this.getElementHeight(child);
        left = this.getElementDir(child, dir);
        right = left + this.getElementWidth(child);

        // if new element will be in view at all, we will need to add it to the live DOM
        // if the row is not visible at all, it will not need to move
        if (this._isCellBoundaryInViewport(left, right, newTop, newBottom)) {
          this.changeStyleProperty(child, this.getCssSupport('transform'),
            'translate3d(' + 0 + 'px,' + oldTop + 'px,' + 0 + 'px)');
          childNodesList.push(child);

          if (!newAnimationInformation[rowKey]) {
            newAnimationInformation[rowKey] = {
              cell: [child],
              oldTop: oldTop,
              newTop: newTop,
              add: true
            };
          } else {
            newAnimationInformation[rowKey].cell.push(child);
          }

          if (newTop - oldTop !== 0) {
            lastAnimationElement = child;
          }
        }
      }
    }
    for (let j = 0; j < childNodesList.length; j++) {
      oldElementSet.appendChild(childNodesList[j]);
    }


    // loop over the new elements and set their old and new tops
    var rowHeaderList = [];
    for (i = 0; i < newRowHeaderElements.childNodes.length; i++) {
      if (rowHeaderSupport) {
        var rowHeader = newRowHeaderElements.childNodes[i];
        rowKey = this._getKey(rowHeader, 'row');
        if (!animationInformation[rowKey]) {
          newTop = this.getElementDir(rowHeader, 'top');
          newBottom = newTop + this.getElementHeight(rowHeader);
          oldTop = viewportBottom;

          if (this._isCellBoundaryInViewport(viewportLeft, viewportRight, newTop, newBottom)) {
            this.changeStyleProperty(rowHeader, this.getCssSupport('transform'),
              'translate3d(' + 0 + 'px,' + oldTop + 'px,' + 0 + 'px)');
            rowHeaderList.push({ element: rowHeader, index: i });
            if (!newAnimationInformation[rowKey]) {
              newAnimationInformation[rowKey] = { oldTop: oldTop, newTop: newTop, add: true };
            }
            newAnimationInformation[rowKey].rowHeader = rowHeader;
          }
        }
      }
    }
    for (let j = 0; j < rowHeaderList.length; j++) {
      oldRowHeaderElements.appendChild(rowHeaderList[j].element);
    }

    // loop over the new elements and set their old and new tops
    var rowEndHeaderList = [];
    for (i = 0; i < newRowEndHeaderElements.childNodes.length; i++) {
      if (rowEndHeaderSupport) {
        var rowEndHeader = newRowEndHeaderElements.childNodes[i];
        rowKey = this._getKey(rowEndHeader, 'row');
        if (!animationInformation[rowKey]) {
          newTop = this.getElementDir(rowEndHeader, 'top');
          newBottom = newTop + this.getElementHeight(rowEndHeader);
          oldTop = viewportBottom;

          if (this._isCellBoundaryInViewport(viewportLeft, -1, newTop, newBottom)) {
            this.changeStyleProperty(rowEndHeader, this.getCssSupport('transform'),
              'translate3d(' + 0 + 'px,' + oldTop + 'px,' + 0 + 'px)');
            rowEndHeaderList.push({ element: rowEndHeader, index: i });
            newAnimationInformation[rowKey].rowEndHeader = rowEndHeader;
            if (!newAnimationInformation[rowKey]) {
              newAnimationInformation[rowKey] = { oldTop: oldTop, newTop: newTop, add: true };
            }
          }
        }
      }
    }
    for (let j = 0; j < rowEndHeaderList.length; j++) {
      oldRowEndHeaderElements.appendChild(rowEndHeaderList[j].element);
    }

    // if nothing is animated which could happen, we can just bail and not animate
    if (lastAnimationElement != null) {
      var transitionListener = this._handleSortEnd.bind(this, newElementSet, newRowHeaderElements,
        newRowEndHeaderElements, childNodesList, rowHeaderList, rowEndHeaderList);

      // register transitionend listener on the last row transitioning before applying the transition
      self._onTransitionEnd(lastAnimationElement, transitionListener, duration);

      this.m_animating = true;

      setTimeout(function () {
        var deltaY;
        var delay;
        var row;
        var j = 0;
        var k;
        var rowKeys = Object.keys(animationInformation);

        for (k = 0; k < rowKeys.length; k++) {
          rowKey = rowKeys[k];
          row = animationInformation[rowKeys[k]];
          delay = delayOffset + 'ms';
          deltaY = row.newTop - row.oldTop;
          if (row.cell) {
            for (j = 0; j < row.cell.length; j++) {
              self.addTransformMoveStyle(row.cell[j], (duration / 2) + 'ms',
                delay, timing, 0, deltaY, 0);
            }
          }
          if (rowHeaderSupport && row.rowHeader) {
            self.addTransformMoveStyle(row.rowHeader, (duration / 2) + 'ms',
              delay, timing, 0, deltaY, 0);
          }
          if (rowEndHeaderSupport && row.rowEndHeader) {
            self.addTransformMoveStyle(row.rowEndHeader, (duration / 2) + 'ms',
              delay, timing, 0, deltaY, 0);
          }
        }

        rowKeys = Object.keys(newAnimationInformation);
        for (k = 0; k < rowKeys.length; k++) {
          rowKey = rowKeys[k];
          row = newAnimationInformation[rowKeys[k]];
          delay = (delayOffset * j) + 'ms';
          deltaY = 0;
          if (row.cell) {
            for (var jj = 0; jj < row.cell.length; jj++) {
              self.addTransformMoveStyle(row.cell[jj], (duration / 2) + 'ms',
                delay, timing, 0, deltaY, 0);
            }
          }
          if (rowHeaderSupport && row.rowHeader) {
            self.addTransformMoveStyle(row.rowHeader, (duration / 2) + 'ms',
              delay, timing, 0, deltaY, 0);
          }
          if (rowEndHeaderSupport && row.rowEndHeader) {
            self.addTransformMoveStyle(row.rowEndHeader, (duration / 2) + 'ms',
              delay, timing, 0, deltaY, 0);
          }
        }
      }, 0);
    } else {
      this._handleSortEnd(newElementSet, newRowHeaderElements, newRowEndHeaderElements);
    }
  };

/**
 * Restore the active cell after sort.
 * @private
 */
DvtDataGrid.prototype._restoreActive = function () {
  if (this.m_active != null) {
    var axis = this.m_active.axis;
    var event = this.m_sortInfo.originalEvent; // Use the sort event as the close event's originalEvent
    if (this.m_active.type === 'cell') {
      // see if the cell exists after the sort
      var cell = this._getCellByKeys(this.m_active.keys);
      if (cell != null) {
        var cellIndex = this.getCellIndexes(cell);

        // select it if selection enabled
        if (this._isSelectionEnabled()) {
          // this will clear the selection if there's multiple selection before sort
          // this is the behavior we want since the ranges in the previous selection
          // will in most cases be invalid after sort.  The only one we can maintain and
          // make sense to do is the active cell
          this.selectAndFocus(cellIndex, event);
        } else {
          // make it active
          this._setActiveByIndex(cellIndex, event);
        }
      } else {
        this._setActive(null, null, event, true);
        // clear selection it if selection enabled
        if (this._isSelectionEnabled()) {
          this._clearSelection(event);
        }
      }
    } else if (axis === 'row' || axis === 'rowEnd') {
      var root = axis === 'row' ? this.m_rowHeader : this.m_rowEndHeader;
      var className = axis === 'row' ?
        this.getMappedStyle('rowheadercell') : this.getMappedStyle('rowendheadercell');
      var rowHeader = this._findHeaderByKey(this.m_active.key, root, className);
      if (rowHeader != null) {
        // make it active
        // Need to add context to scroll databody first before making
        // the row header focus to make it focus properly.
        var rowContext = rowHeader[this.getResources().getMappedAttribute('context')];
        this._setActive(rowHeader, {
          type: 'header',
          axis: axis,
          index: rowContext.index,
          key: rowContext.key,
          level: rowContext.level
        }, event);
      } else {
        this._setActive(null, null, event);
      }
    }
  }
};

/**
 * Gets the sort icon from a  header Element
 * @param {Element} header the header to get sort icon for
 * @private
 */
DvtDataGrid.prototype._getSortIcon = function (header) {
  // presently guaranteed to be the first child of the last child of the parent
  return header.lastChild.firstChild;
};

/**
 * Gets the sort container from a  header Element
 * @param {Element} header the header to get sort container for
 * @private
 */
DvtDataGrid.prototype._getSortContainer = function (header) {
  // presently guaranteed to be the last child of the parent
  return header.lastChild;
};

/**
 * Fire Sort event
 * @private
 */
DvtDataGrid.prototype._fireSortEvent = function () {
  var details = {
    event: this.m_sortInfo.event,
    ui: {
      header: this.m_sortInfo.key,
      direction: this.m_sortInfo.direction
    }
  };
  this.fireEvent('sort', details);
};

/**
 * fireSortRequestEvent
 * @private
 */
DvtDataGrid.prototype._fireSortRequestEvent = function () {
  let context = this.m_sortInfo.header[this.getResources().getMappedAttribute('context')];
  let metadata = this.m_sortInfo.header[this.getResources().getMappedAttribute('metadata')];
  let item = this.buildHeaderTemplateContext(context, metadata).item;
  var details = {
    event: this.m_sortInfo.event,
    ui: {
      direction: this.m_sortInfo.direction,
      item: item,
      axis: this.m_sortInfo.axis
    }
  };
  this.fireEvent('sortRequest', details);
  // set scrollOnRefresh flag if required.
  // if (sortRequest) {
  //   this.m_scrollOnRefreshEvent = true;
  // }
};

/**
 * Start a delayed sort
 * @private
 */
DvtDataGrid.prototype._doDelayedSort = function () {
  if (this.m_delayedSort != null) {
    this._doHeaderSort(this.m_delayedSort.event, this.m_delayedSort.header,
      this.m_delayedSort.direction);
  } else {
    // no pending sort so cleanup
    this.fillViewport();
  }
};

/**
 * fireExpandRequestEvent
 * @param {Event} event - click event on the headers
 * @private
 */
DvtDataGrid.prototype._fireExpandRequestEvent = function (event) {
  const header = this.findHeader(event.target);
  const context = header[this.getResources().getMappedAttribute('context')];
  const metadata = header[this.getResources().getMappedAttribute('metadata')];
  const item = this.buildHeaderTemplateContext(context, metadata).item;
  const details = {
    item: item,
    axis: context.axis
  };
  const disclosureIcon = this._getDisclosureIcon(header);
  this.m_utils.removeCSSClassName(disclosureIcon, this.getMappedStyle('collapsed'));
  this.m_utils.addCSSClassName(disclosureIcon, this.getMappedStyle('expanded'));
  this.fireEvent('expandRequest', details);
  // set scrollOnRefresh flag if required.
  // if (expandRequest) {
  //   this.m_scrollOnRefreshEvent = true;
  // }
};

/**
 * fireCollapseRequestEvent
 * @param {Event} event - click event on the headers
 * @private
 */
DvtDataGrid.prototype._fireCollapseRequestEvent = function (event) {
  const header = this.findHeader(event.target);
  const context = header[this.getResources().getMappedAttribute('context')];
  const metadata = header[this.getResources().getMappedAttribute('metadata')];
  const item = this.buildHeaderTemplateContext(context, metadata).item;
  const details = {
    item: item,
    axis: context.axis
  };
  const disclosureIcon = this._getDisclosureIcon(header);
  this.m_utils.removeCSSClassName(disclosureIcon, this.getMappedStyle('expanded'));
  this.m_utils.addCSSClassName(disclosureIcon, this.getMappedStyle('collapsed'));
  this.fireEvent('collapseRequest', details);
  // set scrollOnRefresh flag if required.
  // if (collapseRequest) {
  //   this.m_scrollOnRefreshEvent = true;
  // }
};

/**
 * handles firing expand Collapse Request Event.
 * @param {Event} event - click event on the headers
 * @private
 */
DvtDataGrid.prototype._handleExpandCollapseRequest = function (event) {
  if (this._isHeaderExpanded(event.target)) {
    this._fireCollapseRequestEvent(event);
  } else if (this._isHeaderCollapsed(event.target)) {
    this._fireExpandRequestEvent(event);
  }
};

/**
 * checks if a header is expandable or collapsable.
 * @param {Element} target - target to be checked
 * @private
 */
DvtDataGrid.prototype._isTargetExpandCollapseEnabled = function (target) {
  return this._isHeaderCollapsed(target) || this._isHeaderExpanded(target);
};

/**
 * checks if a header is collapsed.
 * @param {Element} header - header to be checked
 * @private
 */
DvtDataGrid.prototype._isHeaderCollapsed = function (header) {
  const disclosureIcon = this._getDisclosureIcon(header);
  if (!disclosureIcon) {
    return false;
  }
  return this.m_utils.containsCSSClassName(disclosureIcon, this.getMappedStyle('collapsed'));
};

/**
 * checks if a header is expanded.
 * @param {Element} header - header to be checked
 * @private
 */
DvtDataGrid.prototype._isHeaderExpanded = function (header) {
  const disclosureIcon = this._getDisclosureIcon(header);
  if (!disclosureIcon) {
    return false;
  }
  return this.m_utils.containsCSSClassName(disclosureIcon, this.getMappedStyle('expanded'));
};

/**
 * builds disclosure icon.
 * @private
 */
DvtDataGrid.prototype._buildDisclosureIcon = function (headerContext) {
  // disclosure container is used to create fade effect
  const disclosureContainer = document.createElement('div');
  disclosureContainer.classList.add(this.getMappedStyle('iconContainer'));
  disclosureContainer.classList.add(this.getMappedStyle('disclosureIcon'));

  const disclosureIcon = document.createElement('div');
  disclosureIcon.classList.add(this.getMappedStyle('icon'));
  disclosureIcon.classList.add(this.getMappedStyle('clickableicon'));

  let tooltipText;
  if (headerContext.metadata.expanded === 'expanded') {
    disclosureIcon.classList.add(this.getMappedStyle('expanded'));
    tooltipText = this.getResources().getTranslatedText('collapsedText');
  } else if (headerContext.metadata.expanded === 'collapsed') {
    disclosureIcon.classList.add(this.getMappedStyle('collapsed'));
    tooltipText = this.getResources().getTranslatedText('expandedText');
  }

  disclosureContainer.setAttribute('title', tooltipText);
  disclosureContainer.appendChild(disclosureIcon); // @HTMLUpdateOK

  disclosureContainer.addEventListener('mouseover', this._handleExpandCollapseContainerMouseOver.bind(this));
  disclosureContainer.addEventListener('mouseout', this._handleDisclosureMouseOut.bind(this));
  return disclosureContainer;
};

/**
 * Event handler for handling mouse over event on disclosure container.
 * @param {Event} event the DOM event
 * @private
 */
DvtDataGrid.prototype._handleExpandCollapseContainerMouseOver = function (event) {
  const target = /** @type {Element} */ (event.target);
  const header = this.findHeader(target);
  const disclosureIcon = this._getDisclosureIcon(header);
  // if we are hovering the icon add hover class
  if (this.m_utils.containsCSSClassName(event.currentTarget, this.getMappedStyle('disclosureIcon'))) {
    this.m_utils.addCSSClassName(event.currentTarget, this.getMappedStyle('hover'));
    this.m_utils.addCSSClassName(disclosureIcon, this.getMappedStyle('hover'));
    this.m_utils.addCSSClassName(disclosureIcon, this.getMappedStyle('enabled'));
    this.m_utils.removeCSSClassName(disclosureIcon, this.getMappedStyle('disabled'));
  }
};

/**
 * Builds and returns spacer class
 * @private
 */
DvtDataGrid.prototype._buildSpacer = function (headerContext) {
  const indentElement = document.createElement('span');
  this._addDataGridSpacerClass(indentElement);
  this._addIndentation(headerContext, indentElement);
  return indentElement;
};

/**
 * Gets depth from headerContext.
 * @private
 */
DvtDataGrid.prototype._getTreeDepth = function (headerContext) {
  if (headerContext.metadata.treeDepth) {
    return headerContext.metadata.treeDepth;
  }
  return null;
};

/**
 * Adds Spacer class to spacer element.
 * @private
 */
DvtDataGrid.prototype._addDataGridSpacerClass = function (spacer) {
  this.m_utils.addCSSClassName(spacer, this.getMappedStyle('spacer'));
};

/**
 * Returns true false if leaf or not.
 * @private
 */
 DvtDataGrid.prototype._isLeaf = function (headerContext) {
  return !headerContext.metadata.expanded || headerContext.metadata.expanded === null;
};

/**
* Add the indentation on a spacer
* @private
*/
DvtDataGrid.prototype._addIndentation = function (headerContext, spacer) {
  // 0 index the depth for style purposes
  let depth = this._getTreeDepth(headerContext);
  if (depth === null) {
    return;
  }
  this._appendSpacerClass(depth, spacer, headerContext);
};

/**
* Appends proper indentation class to spacer
* @private
*/
DvtDataGrid.prototype._appendSpacerClass = function (depth, disclosureIcon, headerContext) {
  const disclosureStyle = disclosureIcon.style;
  disclosureStyle.width = ((depth * DvtDataGrid.SPACER_DEFAULT_WIDTH) + (this._isLeaf(headerContext) ? DvtDataGrid.SPACER_DEFAULT_WIDTH : 0)) + 'rem';
};

/**
 * Event handler for handling mouse out event on headers.
 * @param {Event} event the DOM event
 * @private
 */
 DvtDataGrid.prototype._handleDisclosureMouseOut = function (event) {
  const target = /** @type {Element} */ (event.target);
  const header = this.findHeader(target);
  const disclosureIcon = this._getDisclosureIcon(header);
  // if we are hovering the icon add hover class
  if (this.m_utils.containsCSSClassName(event.currentTarget, this.getMappedStyle('disclosureIcon'))) {
    this.m_utils.removeCSSClassName(event.currentTarget, this.getMappedStyle('hover'));
    this.m_utils.removeCSSClassName(disclosureIcon, this.getMappedStyle('hover'));
    this.m_utils.removeCSSClassName(disclosureIcon, this.getMappedStyle('enabled'));
  }
};

/**
 * boolean if target is disclosure
 * @param {Element} target the target being tested
 * @private
 */
 DvtDataGrid.prototype._isDisclosureIcon = function (target) {
  return this.m_utils.containsCSSClassName(target, this.getMappedStyle('icon'));
};

/**
 * Gets the disclosure icon from a target Element
 * @param {Element} target the target to get disclosure icon for
 * @private
 */
DvtDataGrid.prototype._getDisclosureIcon = function (target) {
  const isDisclosureIcon = this.m_utils.containsCSSClassName(target, this.getMappedStyle('icon'));
  if (isDisclosureIcon) {
    return target;
  }
  // presently guaranteed to be the only icon child of the first child
  return target.getElementsByClassName(this.getMappedStyle('icon'))[0];
};

/**
 * Gets the disclosure icon container from a  header Element
 * @param {Element} header the header to get disclosure icon for
 * @private
 */
 DvtDataGrid.prototype._getDisclosureIconContainer = function (header) {
  // presently guaranteed to be the only icon child of the first child
  return header.firstChild;
};

/**
 * @constructor
 * @private
 */
const DataGridProviderDataGridDataSource = function (datagridprovider) {
  this.datagridprovider = datagridprovider;

  this.pendingHeaderCallback = {};

  this._registerEventListeners();
  this.totalRowCount = -1;
  this.totalColumnCount = -1;

  this.rowKeyMap = new DataGridProviderDataGridDataSourceKeyMap();
  this.columnKeyMap = new DataGridProviderDataGridDataSourceKeyMap();

  DataGridProviderDataGridDataSource.superclass.constructor.call(this);
};

oj$1.Object.createSubclass(DataGridProviderDataGridDataSource, oj$1.DataGridDataSource, 'DataGridProviderDataGridDataSource');


DataGridProviderDataGridDataSource.prototype.fetchHeaders = function (headerRange,
  callbacks, callbackObjects) {
  if (callbacks != null) {
    var axis = headerRange.axis;
    var callback = {
      headerRange: headerRange,
      callbacks: callbacks,
      callbackObjects: callbackObjects
    };
    this.pendingHeaderCallback[axis] = callback;
  }
};
DataGridProviderDataGridDataSource.prototype.fetchCells =
function (cellRanges, callbacks, callbackObjects) {
  let rowOffset;
  let rowCount;
  let columnOffset;
  let columnCount;

  for (let i = 0; i < cellRanges.length; i += 1) {
    let cellRange = cellRanges[i];
    if (cellRange.axis === 'row') {
      rowOffset = cellRange.start;
      rowCount = cellRange.count;
    } else if (cellRange.axis === 'column') {
      columnOffset = cellRange.start;
      columnCount = cellRange.count;
    }
  }

  let fetchRowCallback = this.pendingHeaderCallback.row;
  let fetchColumnCallback = this.pendingHeaderCallback.column;
  this.pendingHeaderCallback = {};

  let fetchArray;
  if (fetchRowCallback && fetchColumnCallback) {
    fetchArray = ['all'];
  } else if (fetchRowCallback) {
    fetchArray = ['databody', 'rowHeader', 'rowEndHeader', 'rowHeaderLabel', 'rowEndHeaderLabel'];
  } else if (fetchColumnCallback) {
    fetchArray = ['databody', 'columnHeader', 'columnEndHeader', 'columnHeaderLabel', 'columnEndHeaderLabel'];
  } else {
    fetchArray = ['databody'];
  }

  let fetchParameters = {
    rowOffset: rowOffset,
    columnOffset: columnOffset,
    rowCount: rowCount,
    columnCount: columnCount,
    fetchRegions: new Set(fetchArray)
  };

  let fetchByOffsetHandler = (results, newResults) => {
    let next;
    if (newResults != null) {
      // eslint-disable-next-line no-param-reassign
      results.results = Object.assign({}, results.results, newResults.results);
      if (newResults.next != null) {
        next = newResults.next;
      }
    } else if (results.next != null) {
      next = results.next;
    }

    this.totalRowCount = results.totalRowCount;
    this.totalColumnCount = results.totalColumnCount;

    let headerCallbacks;
    let headerCallbackObjects;
    let result = results.results;

    if (fetchColumnCallback && (!next ||
      (result.columnHeader != null && result.columnEndHeader != null &&
        result.columnHeaderLabel != null && result.columnEndHeaderLabel != null))) {
      headerCallbacks = fetchColumnCallback.callbacks;
      if (headerCallbacks != null && headerCallbacks.success != null) {
        headerCallbackObjects = fetchColumnCallback.callbackObjects;
        let startColumnHeaderSet;
        let endColumnHeaderSet;
        if (result.columnHeader != null) {
          startColumnHeaderSet = new DataGridProviderHeaderSet(results, 'column', this.columnKeyMap);
        }
        if (result.columnEndHeader != null) {
          endColumnHeaderSet = new DataGridProviderHeaderSet(results, 'columnEnd', this.columnKeyMap);
        }
        headerCallbacks.success.call(
          headerCallbackObjects.success, startColumnHeaderSet,
          fetchColumnCallback.headerRange, endColumnHeaderSet);
      }
    }

    if (fetchRowCallback && (!next ||
      (result.rowHeader != null && result.rowEndHeader != null &&
        result.rowHeaderLabel != null && result.rowEndHeaderLabel != null))) {
      headerCallbacks = fetchRowCallback.callbacks;
      if (headerCallbacks != null && headerCallbacks.success != null) {
        headerCallbackObjects = fetchRowCallback.callbackObjects;
        let startRowHeaderSet;
        let endRowHeaderSet;
        if (result.rowHeader != null) {
          startRowHeaderSet = new DataGridProviderHeaderSet(results, 'row', this.rowKeyMap);
        }
        if (result.rowEndHeader != null) {
          endRowHeaderSet = new DataGridProviderHeaderSet(results, 'rowEnd', this.rowKeyMap);
        }
        headerCallbacks.success.call(headerCallbackObjects.success,
          startRowHeaderSet, fetchRowCallback.headerRange, endRowHeaderSet);
      }
    }

    if (callbacks != null && callbacks.success != null && !next) {
      let cellSet = new DataGridProviderCellSet(results, this.rowKeyMap, this.columnKeyMap);
      callbacks.success.call(callbackObjects.success, cellSet, cellRanges);
    }

    if (next) {
      next.then(fetchByOffsetHandler.bind(this, results));
    }
  };

  this.datagridprovider.fetchByOffset(fetchParameters).then(fetchByOffsetHandler);
};

DataGridProviderDataGridDataSource.prototype.getCapability = function (feature) {
  if (feature === 'sort') {
    return 'full';
  }
  if (feature === 'sort' || feature === 'move') {
    return 'none';
  }
  return null;
};

DataGridProviderDataGridDataSource.prototype.getCount = function (axis) {
  if (axis === 'row') {
    return this.totalRowCount;
  } else if (axis === 'column') {
    return this.totalColumnCount;
  }
  return -1;
};
DataGridProviderDataGridDataSource.prototype.getCountPrecision = function () {
  return 'exact';
};
DataGridProviderDataGridDataSource.prototype.indexes = function () {
  return { row: -1, column: -1 };
};
DataGridProviderDataGridDataSource.prototype.keys = function () {
  return { row: null, column: null };
};
DataGridProviderDataGridDataSource.prototype.move = function () {};
DataGridProviderDataGridDataSource.prototype.moveOK = function () {
  return 'invalid';
};
DataGridProviderDataGridDataSource.prototype.sort = function () {};

DataGridProviderDataGridDataSource.prototype._registerEventListeners = function () {
  this._addListener = this._handleDataGridProviderAddEvent.bind(this);
  this._removeListener = this._handleDataGridProviderRemoveEvent.bind(this);
  this._updateListener = this._handleDataGridProviderUpdateEvent.bind(this);
  this._refreshListener = this._handleDataGridProviderRefreshEvent.bind(this);

  this.datagridprovider.addEventListener('add', this._addListener);
  this.datagridprovider.addEventListener('remove', this._removeListener);
  this.datagridprovider.addEventListener('update', this._updateListener);
  this.datagridprovider.addEventListener('refresh', this._refreshListener);
};

DataGridProviderDataGridDataSource.prototype._handleDataGridProviderAddEvent = function (event) {
  var newEvent = { source: this, operation: 'insert', detail: event.detail };
  let relevantKeyMap = event.detail.axis === 'row' ? this.rowKeyMap : this.columnKeyMap;
  relevantKeyMap.handleChange(event.detail.ranges, false);
  this.handleEvent('change', newEvent);
};

DataGridProviderDataGridDataSource.prototype._handleDataGridProviderRemoveEvent = function (event) {
  var newEvent = { source: this, operation: 'delete', detail: event.detail };
  let relevantKeyMap = event.detail.axis === 'row' ? this.rowKeyMap : this.columnKeyMap;
  relevantKeyMap.handleChange(event.detail.ranges, true);
  this.handleEvent('change', newEvent);
};

DataGridProviderDataGridDataSource.prototype._handleDataGridProviderUpdateEvent = function (event) {
  var newEvent = { source: this, operation: 'update', detail: event.detail };
  this.handleEvent('change', newEvent);
};

DataGridProviderDataGridDataSource.prototype._handleDataGridProviderRefreshEvent = function () {
  this.rowKeyMap = new DataGridProviderDataGridDataSourceKeyMap();
  this.columnKeyMap = new DataGridProviderDataGridDataSourceKeyMap();
  var newEvent = { source: this, operation: 'refresh' };
  this.handleEvent('change', newEvent);
};

function DataGridProviderCellSet(results, rowKeyMap, columnKeyMap) {
  this.results = results;
  this.rowKeyMap = rowKeyMap;
  this.columnKeyMap = columnKeyMap;
  this.databodyResults = results.results.databody;
  this.rowStart = this.results.rowOffset;
  this.rowCount = this.results.rowCount;
  this.rowEnd = (this.rowStart + this.rowCount) - 1;
  this.columnStart = this.results.columnOffset;
  this.columnCount = this.results.columnCount;
  this.columnEnd = (this.columnStart + this.columnCount) - 1;
  this.lastItem = null;
}
DataGridProviderCellSet.prototype.getData = function (indexes) {
  let item = this._findItemByIndex(indexes.row, indexes.column);
  return item.data;
};
DataGridProviderCellSet.prototype.getMetadata = function (indexes) {
  let item = this._findItemByIndex(indexes.row, indexes.column);
  return {
    metadata: item.metadata,
    keys: {
      row: this.rowKeyMap.get(item.rowIndex),
      column: this.columnKeyMap.get(item.columnIndex)
    }
  };
};
DataGridProviderCellSet.prototype.getCount = function (axis) {
  if (this.databodyResults != null) {
    if (axis === 'row') {
      return this.rowCount;
    }
    if (axis === 'column') {
      return this.columnCount;
    }
  }
  return 0;
};
DataGridProviderCellSet.prototype.getExtent = function (indexes) {
  let rowBefore = false;
  let rowAfter = false;
  let columnBefore = false;
  let columnAfter = false;

  let item = this._findItemByIndex(indexes.row, indexes.column);
  let rowStart = item.rowIndex;
  let rowExtent = item.rowExtent;
  let rowEnd = (rowStart + rowExtent) - 1;
  let columnStart = item.columnIndex;
  let columnExtent = item.columnExtent;
  let columnEnd = (columnStart + columnExtent) - 1;

  if (rowStart < this.rowStart) {
    rowExtent -= (this.rowStart - rowStart);
    rowBefore = true;
  }
  if (rowEnd > this.rowEnd) {
    rowExtent -= (rowEnd - this.rowEnd);
    rowAfter = true;
  }

  if (columnStart < this.columnStart) {
    columnExtent -= (this.columnStart - columnStart);
    columnBefore = true;
  }
  if (columnEnd > this.columnEnd) {
    columnExtent -= (columnEnd - this.columnEnd);
    columnAfter = true;
  }

  return {
    row: { extent: rowExtent, more: { before: rowBefore, after: rowAfter } },
    column: { extent: columnExtent, more: { before: columnBefore, after: columnAfter } }
  };
};
DataGridProviderCellSet.prototype._findItemByIndex = function (rowIndex, columnIndex) {
  let checkfunction = (item) => {
    let itemRowIndex = item.rowIndex;
    let itemRowExtent = item.rowExtent;
    let itemColumnIndex = item.columnIndex;
    let itemColumnExtent = item.columnExtent;
    if (rowIndex >= itemRowIndex && rowIndex < itemRowIndex + itemRowExtent &&
      columnIndex >= itemColumnIndex && columnIndex < itemColumnIndex + itemColumnExtent) {
        return true;
    }
    return false;
  };

  if (this.lastItem != null && checkfunction(this.lastItem)) {
    return this.lastItem;
  }
  this.lastItem = this.databodyResults.find(checkfunction);
  return this.lastItem;
};

function DataGridProviderHeaderSet(results, axis, keyMap) {
  this.results = results;
  this.axis = axis;
  this.axisResults = results.results[this.axis + 'Header'];
  this.axisLabels = results.results[this.axis + 'HeaderLabel'];
  if (this.axis === 'row' || this.axis === 'rowEnd') {
    this.start = this.results.rowOffset;
    this.count = this.results.rowCount;
  } else {
    this.start = this.results.columnOffset;
    this.count = this.results.columnCount;
  }
  this.end = (this.start + this.count) - 1;
  this.lastItem = null;
  this.keyMap = keyMap;
  this.levelCount = null;
}
DataGridProviderHeaderSet.prototype.getData = function (index, level) {
  let item = this._findItemByIndexLevel(index, level);
  return item.data;
};
DataGridProviderHeaderSet.prototype.getMetadata = function (index, level) {
  let item = this._findItemByIndexLevel(index, level);
  let sortDirection = item.metadata.sortDirection;
  if (sortDirection != null) {
    sortDirection = item.metadata.sortDirection === 'unsorted' ? null : sortDirection;
  }
  return {
    metadata: item.metadata,
    key: this.keyMap.get(item.index, item.level, this.getLevelCount(), item.depth),
    sortDirection: sortDirection
  };
};
DataGridProviderHeaderSet.prototype.getLevelCount = function () {
  if (this.levelCount == null) {
    let maxLevel = 0;
    if (this.axisResults != null) {
      this.axisResults.forEach((item) => {
        let itemMaxLevel = item.level + item.depth - 1;
        if (itemMaxLevel > maxLevel) {
          maxLevel = itemMaxLevel;
        }
      });
    }
    this.levelCount = maxLevel + 1;
  }
  return this.levelCount;
};
DataGridProviderHeaderSet.prototype.getExtent = function (index, level) {
  let item = this._findItemByIndexLevel(index, level);
  let extent = item.extent;
  let start = item.index;
  let end = (item.index + extent) - 1;
  let before = false;
  let after = false;
  if (start < this.start) {
    extent -= (this.start - start);
    before = true;
  }
  if (end > this.end) {
    extent -= (end - this.end);
    after = true;
  }
  return { extent: extent, more: { before: before, after: after } };
};
DataGridProviderHeaderSet.prototype.getLabel = function (index) {
  if (this.axisLabels != null) {
    return this.axisLabels[index].data;
  }
  return null;
};
DataGridProviderHeaderSet.prototype.getDepth = function (index, level) {
  let item = this._findItemByIndexLevel(index, level);
  return item.depth;
};
DataGridProviderHeaderSet.prototype.getCount = function () {
  if (this.axisResults != null) {
    return this.count;
  }
  return 0;
};
DataGridProviderHeaderSet.prototype._findItemByIndexLevel = function (index, level) {
  let checkfunction = (item) => {
    let itemIndex = item.index;
    let itemExtent = item.extent;
    let itemLevel = item.level;
    let itemDepth = item.depth;
    if (index >= itemIndex && index < itemIndex + itemExtent &&
      level >= itemLevel && level < itemLevel + itemDepth) {
        return true;
    }
    return false;
  };

  if (this.lastItem != null && checkfunction(this.lastItem)) {
    return this.lastItem;
  }
  this.lastItem = this.axisResults.find(checkfunction);
  return this.lastItem;
};

/**
 * @constructor
 * @private
 */
 function DataGridProviderDataGridDataSourceKeyMap() {
  this.keyMap = new Map();
  this.nextKey = 0;

  this.set = (index, level, totalLevels, depth) => {
    // use string keys to avoid == check issues
    const newKey = '' + this.nextKey;
    if (depth == null) {
    // eslint-disable-next-line no-param-reassign
    depth = 1;
    }
    for (let i = 0; i < depth; i++) {
      let levelCalc = level == null ? level : level + i;
      let indexLevelPair = this.pairIndexLevel(index, levelCalc, totalLevels);
      this.keyMap.set(indexLevelPair, newKey);
    }
    this.nextKey += 1;
    return newKey;
  };

  this.get = (index, level, totalLevels, depth) => {
    const indexLevelPair = this.pairIndexLevel(index, level, totalLevels);
    let key = this.keyMap.get(indexLevelPair);
    if (key == null) {
      key = this.set(index, level, totalLevels, depth);
    }
    return key;
  };

  this.pairIndexLevel = (index, level, totalLevels) => {
    if (level == null || (totalLevels != null && level === totalLevels - 1)) {
       return '' + index;
    }
    return index + ',' + level;
  };

  this.handleChange = (ranges, remove) => {
    let indexSet = new Set();
    ranges.forEach(function (range) {
      let start = range.offset;
      let count = range.count;
      for (var i = 0; i < count; i++) {
        indexSet.add(start + i);
      }
    });
    let indexes = Array.from(indexSet);
    indexes.sort(function (a, b) {
      return a - b;
    });

    let newKeyMap = new Map();
    let oldKeyMap = this.keyMap;

    oldKeyMap.forEach((value, key)=>{
      let parseKey = key.split(',');
      let index = parseInt(parseKey[0], 10);
      let level = parseKey[1];

      if (remove && indexes.some((changedIndex) => { return changedIndex === index; })) {
        return;
      }

      let countBefore = 0;
      if (index >= indexes[0]) {
        let tempCompareIndex = index;
        countBefore = indexes.findIndex((changedIndex) => {
          if (!remove && tempCompareIndex >= changedIndex) {
            tempCompareIndex += 1;
          }
          return changedIndex > tempCompareIndex;
        });
        if (countBefore === -1) {
          countBefore = indexes.length;
        }
      }
      let newIndex = remove ? (index - countBefore) : (index + countBefore);
      let newPair = this.pairIndexLevel(newIndex, level);
      newKeyMap.set(newPair, value);
    });

    this.keyMap = newKeyMap;
  };
}

// import { DataGridProvider } from 'ojs/ojdatagridprovider';

/**
 * @ojcomponent oj.ojDataGrid
 * @augments oj.baseComponent
 * @since 0.6.0
 *
 * @ojrole application
 * @ojrole grid
 * @ojshortdesc A data grid displays data in a cell oriented grid.
 * @ojtsimport {module: "ojdatagridprovider", type: "AMD", imported: ["DataGridProvider","GridBodyItem", "GridHeaderItem", "GridItem"]}
 * @ojtsimport {module: "ojdataprovider", type: "AMD", imported: ["DataProvider"]}
 * @ojsignature [{
 *                target: "Type",
 *                value: "class ojDataGrid<K, D> extends baseComponent<ojDataGridSettableProperties<K,D>>",
 *                genericParameters: [{"name": "K", "description": "Type of key of the dataprovider"}, {"name": "D", "description": "Type of data from the dataprovider"}]
 *               },
 *               {
 *                target: "Type",
 *                value: "ojDataGridSettableProperties<K,D> extends baseComponentSettableProperties",
 *                for: "SettableProperties"
 *               }
 *              ]
 *
 * @ojpropertylayout {propertyGroup: "common", items: ["editMode", "gridlines.horizontal", "gridlines.vertical"]}
 * @ojpropertylayout {propertyGroup: "data", items: ["data", "selection"]}
 * @ojvbdefaultcolumns 12
 * @ojvbmincolumns 2
 *
 * @ojuxspecs ['data-grid']
 *
 * @classdesc
 * <h3 id="datagridOverview-section">
 *   JET DataGrid
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#datagridOverview-section"></a>
 * </h3>
 * <p>Description:</p>
 * <p>A JET DataGrid is a themeable, WAI-ARIA compliant element that displays data in a cell oriented grid.  Data inside the DataGrid can be associated with row and column headers.  Page authors can customize the content rendered inside cells and headers.</p>
 *
 * <pre class="prettyprint"><code>&lt;oj-data-grid
 *   id="datagrid"
 *   style="width:250px;height:250px"
 *   aria-label="My Data Grid"
 *   data='{{dataSource}}'>
 * &lt;/oj-data-grid></code></pre>
 *
 * <h3 id="data-section">
 *   Data
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#data-section"></a>
 * </h3>
 * <p>The JET DataGrid gets its data from a {@link DataGridProvider}.  There are no out of the box DataGridProvider implementations yet.
 * Developers can also create their own DataGridProvider by implementing the DataGridProvider interface. See the cookbook for an example of a custom DataGridProvider.</p>
 *
 * <h3 id="touch-section">
 *   Touch End User Information
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#touch-section"></a>
 * </h3>
 *
 * {@ojinclude "name":"touchDoc"}
 *
 * <h3 id="keyboard-section">
 *   Keyboard End User Information
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#keyboard-section"></a>
 * </h3>
 *
 * {@ojinclude "name":"keyboardDoc"}
 *
 * <h3 id="a11y-section">
 *   Accessibility
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#a11y-section"></a>
 * </h3>
 *
 * <p>Since <code class="prettyprint">role="application"</code> is used in the DataGrid, application should always apply an <code class="prettyprint">aria-label</code> to the DataGrid element so that it can distinguish from other elements with application role.</p>
 *
 * <h3 id="context-section">
 *   Header Context And Cell Context
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#context-section"></a>
 * </h3>
 *
 * <p>For all header and cell attributes, developers can specify a function as the return value.  The function takes a single argument, which is an object that contains contextual information about the particular header or cell.  This gives developers the flexibility to return different value depending on the context.</p>
 *
 * <p>For header attributes, the context parameter contains the following keys:</p>
 * <table class="keyboard-table">
 *   <thead>
 *     <tr>
 *       <th>Key</th>
 *       <th>Description</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td><kbd>axis</kbd></td>
 *       <td>The axis of the header.  Possible values are 'row', 'column', 'rowEnd', and 'columnEnd'.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>componentElement</kbd></td>
 *       <td>A reference to the DataGrid element.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>datasource</kbd></td>
 *       <td>A reference to the data source object.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>index</kbd></td>
 *       <td>The index of the header, where 0 is the index of the first header.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>key</kbd></td>
 *       <td>The key of the header.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>data</kbd></td>
 *       <td>The data object for the header.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>parentElement</kbd></td>
 *       <td>The header cell element.  The renderer can use this to directly append content to the header cell element.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>level</kbd></td>
 *       <td>The level of the header. The outermost header is level zero.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>depth</kbd></td>
 *       <td>The the number of levels the header spans.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>extent</kbd></td>
 *       <td>The number of indexes the header spans.</td>
 *     </tr>
 *   </tbody>
 * </table>
 *
 * <p></p>
 * <p>For cell attributes, the context paramter contains the following keys:</p>
 * <table class="keyboard-table">
 *   <thead>
 *     <tr>
 *       <th>Key</th>
 *       <th>Description</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td><kbd>componentElement</kbd></td>
 *       <td>A reference to the DataGrid element.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>datasource</kbd></td>
 *       <td>A reference to the data source object.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>indexes</kbd></td>
 *       <td>The object that contains both the zero based row index and column index in which the cell is bound to.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>keys</kbd></td>
 *       <td>The object that contains both the row key and column key which identifies the cell.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>cell</kbd></td>
 *       <td>An object containing attribute data which should be used to reference the data in the cell.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>data</kbd></td>
 *       <td>The plain data for the cell.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>parentElement</kbd></td>
 *       <td>The data cell element.  The renderer can use this to directly append content to the data cell element.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>extents</kbd></td>
 *       <td>The object that contains both the row extent and column extent of the cell.</td>
 *     </tr>
 *   </tbody>
 * </table>
 *
 * <p></p>
 * <p>Note that a custom DataGridProvider can return additional header and cell context information in the metadata property.</p>
 *
 * <h3 id="selection-section">
 *   Selection
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#selection-section"></a>
 * </h3>
 *
 * <p>The DataGrid supports both cell based and row based selection mode, which developers can specify using the selectionMode attribute.  For each mode developers can also specify whether single or multiple cells/rows can be selected.</p>
 * <p>Developers can specify or retrieve selection from the DataGrid using the selection attribute.  A selection in DataGrid consists of an array of ranges.  Each range contains the following keys: startIndex, endIndex, startKey, endKey.  Each of the keys contains value for 'row' and 'column'.  If endIndex and endKey are not specified, -1, or null, that means the range is unbounded, i.e. the cells of the entire row/column are selected.</p>
 *
 *
 * <h3 id="geometry-section">
 *   Geometry Management
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#geometry-section"></a>
 * </h3>
 *
 * <p>If the DataGrid is not styled with a fixed size, then it will responds to a change to the size of its container.  Note that unlike Table the content of the cell does not affect the height of the row.  The height of the rows must be pre-determined and specified by the developer or a default size will be used.</p>
 *
 * <p>The DataGrid does not support % width and height values in the header style or style class.</p>
 *
 * <h3 id="rtl-section">
 *   Reading direction
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#rtl-section"></a>
 * </h3>
 *
 * <p>The order of the column headers will be rendered in reverse order in RTL reading direction.  The location of the row header will also be different between RTL and LTR direction.  It is up to the developers to ensure that the content of the header and data cell are rendered correctly according to the reading direction.</p>
 * <p>As with any JET element, in the unusual case that the directionality (LTR or RTL) changes post-init, the DataGrid must be <code class="prettyprint">refresh()</code>ed.
 *
 * <h3 id="templating-section">
 *   Templating Alignment
 *   <a class="bookmarkable-link" title="Templating Alignment" href="#templating-section"></a>
 * </h3>
 * <p>The DataGrid cells are horizontal flex boxes. To change the alignment use the classes documented in the flex layout section of the cookbook along with header and cell className attributes.
 *
 * <h3 id="perf-section">
 *   Performance
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#perf-section"></a>
 * </h3>
 *
 * <h4>Data Set Size</h4>
 * <p>As a rule of thumb, it's recommended that applications limit the amount of data to display.  Displaying large
 * number of items in DataGrid makes it hard for users to find what they are looking for, but affects the
 * scrolling performance as well.  If displaying large number of items is necessary, consider use a paging control with DataGrid
 * to limit the number of items to display at a time.  Also consider setting <code class="prettyprint">scrollPolicy</code> to
 * <code class="prettyprint">'scroll'</code> to enable virtual scrolling to reduce the number of elements in the DOM at any given time .</p>
 *
 * <h4>Cell Content</h4>
 * <p>DataGrid allows developers to specify arbitrary content inside its cells. In order to minimize any negative effect on
 * performance, you should avoid putting large numbers of heavy-weight content inside a cell because as you add more complexity
 * to the structure, the effect will be multiplied because there can be many items in the DataGrid.</p>
 *
 * <h4>Templating</h4>
 * <p>When deciding to use a template or renderer, consider the conditionality of the template. Having templates with several if blocks
 * can significantly hinder performance. If you have this case either specify a function for the template to remove conditions from the template itself or
 * use the renderer attribute.</p>
 *
 * <h3 id="contextmenu-section">
 *   Context Menu
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#contextmenu-section"></a>
 * </h3>
 * <p>The DataGrid has a default context menu for accessibly performing operations such as header resize and sort.
 * When defining a context menu, DataGrid allows the app to use the built-in behavior for operations such as header
 * resize and sort by specifying menu list items as follows.</p>
 *
 * <ul><li> &lt;li data-oj-command="oj-datagrid-['commandname']" /&gt;</li></ul>
 *
 * <p>Note that if no &lt;a&gt; element is specified inside of a list item with a command,
 * the translated text from the default menu will be supplied in an anchor tag.</p>
 *
 * <p>The supported commands:</p>
 * <table class="keyboard-table">
 *   <thead>
 *      <tr>
 *       <th>Default Function</th>
 *       <th>data-oj-command value</th>
 *      </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td><kbd>Resize menu</kbd> (contains width and height resize)</td>
 *       <td>oj-datagrid-resize</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Sort Row menu</kbd> (contains ascending and descending sort)</td>
 *       <td>oj-datagrid-sortRow</td>
 *      </tr>
 *     <tr>
 *        <td><kbd>Sort Column menu</kbd> (contains ascending and descending sort)</td>
 *       <td>oj-datagrid-sortCol</td>
 *      </tr>
 *     <tr>
 *        <td><kbd>Resize Width</kbd></td>
 *       <td>oj-datagrid-resizeWidth</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Resize Height</kbd></td>
 *       <td>oj-datagrid-resizeHeight</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Resize Fit To Content</kbd></td>
 *       <td>oj-datagrid-resizeFitToContent</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Sort Row Ascending</kbd></td>
 *       <td>oj-datagrid-sortRowAsc</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Sort Row Descending</kbd></td>
 *       <td>oj-datagrid-sortRowDsc</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Sort Column Ascending</kbd></td>
 *       <td>oj-datagrid-sortColAsc</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Sort Column Descending</kbd></td>
 *       <td>oj-datagrid-sortColDsc</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Cut (for reordering)</kbd></td>
 *       <td>oj-datagrid-cut</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Paste (for reordering)</kbd></td>
 *       <td>oj-datagrid-paste</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>CutCells (for data transferring)</kbd></td>
 *       <td>oj-datagrid-cutCells</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>CopyCells (for data transferring)</kbd></td>
 *       <td>oj-datagrid-copyCells</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>PasteCells (for data transferring)</kbd></td>
 *       <td>oj-datagrid-pasteCells</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Fill</kbd></td>
 *       <td>oj-datagrid-fillCells</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Select Multiple Cells on Touch Device</kbd></td>
 *       <td>oj-datagrid-discontiguousSelection</td>
 *     </tr>
 * </tbody></table>
 */
//-----------------------------------------------------
//                   Fragments
//-----------------------------------------------------
/**
 * <table class="keyboard-table">
 *   <thead>
 *     <tr>
 *       <th>Target</th>
 *       <th>Gesture</th>
 *       <th>Action</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td rowspan="2">Cell</td>
 *       <td><kbd>Tap</kbd></td>
 *       <td>Focus on the cell.  If <code class="prettyprint">selectionMode</code> for cells is enabled, selects the cell as well.
 *       If multiple selection is enabled the selection handles will appear. Tapping a different cell will deselect the previous selection.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Press & Hold</kbd></td>
 *       <td>Display context menu</td>
 *     </tr>
 *
 *     <tr>
 *       <td rowspan="3">Row</td>
 *       <td><kbd>Tap</kbd></td>
 *       <td>If <code class="prettyprint">selectionMode</code> for rows is enabled, selects the row as well.
 *       If multiple selection is enabled the selection handles will appear. Tapping a different row will deselect the previous selection.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Drag</kbd></td>
 *       <td>If the row that is dragged contains the active cell and <code class="prettyprint">dnd reorder row</code> is enabled the row will be moved within the DataGrid.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Press & Hold</kbd></td>
 *       <td>Display context menu</td>
 *     </tr>
 *
 *     <tr>
 *       <td rowspan="2">Header</td>
 *       <td><kbd>Tap</kbd></td>
 *       <td>If Multiple Selection is enabled or in row <code>selectionMode</code>, the row or column will be selected. Selection handles will appear for multiple selection. Otherwise, header cell will be focused.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Press & Hold</kbd></td>
 *       <td>Display context menu</td>
 *     </tr>
 *     <tr>
 *       <td>Header Gridline</td>
 *       <td><kbd>Drag</kbd></td>
 *       <td>Resizes the header if <code class="prettyprint">resizable</code> enabled along the axis.</td>
 *     </tr>
 *     <tr>
 *       <td>Corner</td>
 *       <td><kbd>Tap</kbd></td>
 *       <td>Tapping on the corner will perform a select all operation on the datagrid if multiple selection is enabled.</td>
 *     </tr>
 *
 *   </tbody>
 * </table>
 *
 * @ojfragment touchDoc - Used in touch section of classdesc, and standalone gesture doc
 * @memberof oj.ojDataGrid
 */

/**
 * <table class="keyboard-table">
 *   <thead>
 *     <tr>
 *       <th>Target</th>
 *       <th>Key</th>
 *       <th>Action</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td rowspan="24">Cell</td>
 *       <td><kbd>Tab</kbd></td>
 *       <td>The first Tab into the DataGrid moves focus to the first cell of the first row.  The second Tab moves focus to the next focusable element outside of the DataGrid.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Tab</kbd></td>
 *       <td>The first Shift + Tab into the DataGrid moves focus to the first cell of the first row.  The second Shift + Tab moves focus to the previous focusable element outside of the DataGrid.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>LeftArrow</kbd></td>
 *       <td>Moves focus to the cell of the previous column within the current row.  There is no wrapping at the beginning or end of the columns.  If a row header is present, then the row header next to the first column of the current row will gain focus.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>RightArrow</kbd></td>
 *       <td>Moves focus to the cell of the next column within the current row.  There is no wrapping at the beginning or end of the columns.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>UpArrow</kbd></td>
 *       <td>Moves focus to the cell of the previous row within the current column.  There is no wrapping at the beginning or end of the rows.  If a column header is present, then the column header above the first row of the current column will gain focus.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>DownArrow</kbd></td>
 *       <td>Moves focus to the cell of the next row within the current column.  There is no wrapping at the beginning or end of the rows.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Home</kbd></td>
 *       <td>Moves focus to the first (available) cell of the current row.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>End</kbd></td>
 *       <td>Moves focus to the last (available) cell of the current row.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>PageUp</kbd></td>
 *       <td>Moves focus to the first (available) cell in the current column.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>PageDown</kbd></td>
 *       <td>Moves focus to the last (available) cell in the current column.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + Space</kbd></td>
 *       <td>Selects all the cells of the current column.  This is only available if multiple cell selection mode is enabled.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Space</kbd></td>
 *       <td>Selects all the cells of the current row.  This is only available if multiple cell selection mode is enabled.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Arrow</kbd></td>
 *       <td>Extends the current selection.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + Arrow</kbd></td>
 *       <td>Move focus to level 0 of the active index of the header in the arrow direction if it exists.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + F8</kbd></td>
 *       <td>Freezes the current selection, therefore allowing user to move focus to another location to add or remove additional cells to the current selection.
 *           To deselect begin the discontiguous selection within an existing selection.
 *           This is used to accomplish non-contiguous selection.  Use the Esc key or press Shift+F8 again to exit this mode.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + F10</kbd></td>
 *       <td>Brings up the context menu.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + C</kbd></td>
 *       <td>It triggers ojCopyRequest with the selected range of cells. Only a single range may be copied.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + X</kbd></td>
 *       <td>Marks the current row to move if dnd is enabled and the datasource supports move operation. If datasource is datagridProvider,
 *            It triggers ojCutRequest with the selected range of cells. Only a single range may be cut.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + V</kbd></td>
 *       <td>Move the row that is marked to directly under the current row.  If the row with the focused cell is the last row, then it will be move to the row above the current row.
 *            If datasource is datagridProvider, it triggers ojPasteRequest with the selected range of cells.
 *            Investigate the source selection/action to determine how to handle the paste operation.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + A</kbd></td>
 *       <td>If multiple selection is enabled, performs a select all on the datagrid.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + Alt + 5</kbd></td>
 *       <td>Read the context and content of the current cell to the screen reader.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>F2</kbd></td>
 *       <td>Makes the content of the cell actionable, such as a link.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Enter</kbd></td>
 *       <td>Makes the content of the cell actionable and acts on the content, such as going to a link.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Alt + Enter</kbd></td>
 *       <td>Makes the content of the cell actionable, such as a link.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Esc</kbd></td>
 *       <td>If the cell is actionable it exits actionable mode.</td>
 *     </tr>
 *     <tr>
 *       <td rowspan="15">Column Header Cell</td>
 *       <td><kbd>LeftArrow</kbd></td>
 *       <td>Moves focus to the previous column header.  There is no wrapping at the beginning or end of the column headers.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>RightArrow</kbd></td>
 *       <td>Moves focus to the next column header.  There is no wrapping at the beginning or end of the column headers.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>DownArrow</kbd></td>
 *       <td>Moves focus to the cell of the first row directly below the column header. If using nested headers will move focus up a level.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>UpArrow</kbd></td>
 *       <td>If using nested headers will move focus down a level.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + UpArrow</kbd></td>
 *       <td>If in the column end header, move focus to level 0 of the active index in the column header if it exists.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + LeftArrow</kbd></td>
 *       <td>If outside header and hierarchical header will expand, if expanded or not hierarchical, move focus left.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + RightArrow</kbd></td>
 *       <td>If outside header and hierarchical header will expand, if expanded or not hierarchical, move focus right.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + DownArrow</kbd></td>
 *       <td>If in the column header, move focus to level 0 of the active index in the column end header if it exists.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Enter</kbd></td>
 *       <td>Toggle the sort order of the column if the column is sortable.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + F10</kbd></td>
 *       <td>Brings up the context menu.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Space</kbd></td>
 *       <td>If multiple selection is enabled and not in <code>selectionMode</code> row, the column(s) underneath the header will be selected.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Right Arrow</kbd></td>
 *       <td>If multiple selection is enabled and not in <code>selectionMode</code> row, the column selection will extend to the right by the number of columns covered by the header to the right of the current selection frontier header.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Left Arrow</kbd></td>
 *       <td>If multiple selection is enabled and not in <code>selectionMode</code> row, the column selection will extend to the right left the number of columns covered by the header to the left of the current selection frontier header.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Up Arrow</kbd></td>
 *       <td>If multiple selection is enabled and not in <code>selectionMode</code> row and the current selection frontier header has a parent nested header, the column selection will extend to cover the columns beneath the parent header.
 *           <br/> Extending the selection with arrow keys will use the parent level. If the parent header is directly above the anchor header, the anchor will shift to the parent header and future selections will be based on the parent header.
 *           <br/> If we are already at the highest level, nothing will happen.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Down Arrow</kbd></td>
 *       <td>If multiple selection is enabled and not in <code>selectionMode</code> row and the current selection frontier header has a child nested header, the column selection will extend to cover the columns beneath the child header.
 *           <br/> Extending the selection with arrow keys will use the child level. If the child header is directly below the anchor header, the anchor will shift to the child header and future selections will be based on the child header.
 *           <br/> If we are already at the lowest level, it will simply move into the databody and select the first cell underneath the header.</td>
 *     </tr>
 *     <tr>
 *       <td rowspan="12">Row Header Cell</td>
 *       <td><kbd>UpArrow</kbd></td>
 *       <td>Moves focus to the previous row header.  There is no wrapping at the beginning or end of the row headers.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>DownArrow</kbd></td>
 *       <td>Moves focus to the next row header.  There is no wrapping at the beginning or end of the row headers.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>RightArrow</kbd></td>
 *       <td>Moves focus to the cell of the first column directly next to the row header. If using nested headers will move focus up a level.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>LeftArrow</kbd></td>
 *       <td>Moves focus to the cell of the first column directly next to the row header in RTL direction. If using nested headers will move focus down a level.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + LeftArrow</kbd></td>
 *       <td>If inside header and hierarchical header will expand, if expanded or not hierarchical, move focus left. If in the row end header, move focus to level 0 of the active index in the row header if it exists.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Ctrl + RightArrow</kbd></td>
 *       <td>If inside header and hierarchical header will expand, if expanded or not hierarchical, move focus right. If in the row header, move focus to level 0 of the active index in the row end header if it exists.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + F10</kbd></td>
 *       <td>Brings up the context menu.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Space</kbd></td>
 *       <td>If multiple selection is enabled, the row(s) underneath the header will be selected.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Up Arrow</kbd></td>
 *       <td>If multiple selection is enabled, the row selection will extend up by the number of rows covered by the header above the current selection frontier header.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Down Arrow</kbd></td>
 *       <td>If multiple selection is enabled, the row selection will extend down by the number of rows covered by the header below the current selection frontier header.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Left Arrow</kbd></td>
 *       <td>If multiple selection is enabled and the current selection frontier header has a parent nested header, the row selection will extend to cover the rows beneath the parent header.
 *           <br/> Extending the selection with arrow keys will use the parent level. If the parent header is directly above the anchor header, the anchor will shift to the parent header and future selections will be based on the parent header.
 *           <br/> If we are already at the highest level, nothing will happen.</td>
 *     </tr>
 *     <tr>
 *       <td><kbd>Shift + Right Arrow</kbd></td>
 *       <td>If multiple selection is enabled  and the current selection frontier header has a child nested header, the row selection will extend to cover the rows beneath the child header.
 *           <br/> Extending the selection with arrow keys will use the child level. If the child header is directly below the anchor header, the anchor will shift to the child header and future selections will be based on the child header.
 *           <br/> If we are already at the lowest level, it will simply move into the databody and select the first cell underneath the header.</td>
 *     </tr>


 *   </tbody>
 * </table>
 *
 * @ojfragment keyboardDoc - Used in keyboard section of classdesc, and standalone gesture doc
 * @memberof oj.ojDataGrid
 */

//----------------------------------------------
//             SUB-IDS
//----------------------------------------------

/**
 * <p>Sub-ID for the DataGrid element's cells.</p>
 *
 * To lookup a cell the locator object should have the following:
 * <ul>
 * <li><b>subId</b>: 'oj-datagrid-cell'</li>
 * <li><b>rowIndex</b>: the zero based absolute row index</li>
 * <li><b>columnIndex</b>: the zero based absolute column index</li>
 * </ul>
 *
 * @ojsubid oj-datagrid-cell
 * @memberof oj.ojDataGrid
 * @example <caption>Get the cell at the specified location:</caption>
 * var node = myDataGrid.getNodeBySubId({subId: 'oj-datagrid-cell', rowIndex: rowIndexValue, columnIndex: columnIndexValue});
 */

/**
 * <p>Sub-ID for the DataGrid element's headers.</p>
 *
 * To lookup a header the locator object should have the following:
 * <ul>
 * <li><b>subId</b>: 'oj-datagrid-header'</li>
 * <li><b>axis</b>: 'column'/'row'/'columnEnd'/'rowEnd'</li>
 * <li><b>index</b>: the zero based absolute row/column index.</li>
 * <li><b>level</b>: the zero based header level, 0 is the outer edge, if not specified will default to 0</li>
 * </ul>
 *
 * @ojsubid oj-datagrid-header
 * @memberof oj.ojDataGrid
 *
 * @example <caption>Get the header at the specified location:</caption>
 * var node = myDataGrid.getNodeBySubId({subId: 'oj-datagrid-header', axis: 'axisValue', index: indexValue, level: levelValue});
 */

/**
 * <p>Sub-ID for the DataGrid element's sort ascending icon in column headers.</p>
 *
 * To lookup a sort icon the locator object should have the following:
 * <ul>
 * <li><b>subId</b>: 'oj-datagrid-sort-ascending'</li>
 * <li><b>axis</b>: 'column'</li>
 * <li><b>index</b>: the zero based absolute column index</li>
 * <li><b>level</b>: the zero based header level, 0 is the outer edge, if not specified will default to 0</li>
 * </ul>
 *
 * @ojsubid oj-datagrid-sort-ascending
 * @memberof oj.ojDataGrid
 *
 * @example <caption>Get the sort icon from the header at the specified location:</caption>
 * var node = myDataGrid.getNodeBySubId({subId: 'oj-datagrid-sort-ascending', axis: 'axisValue', index: indexValue, level: levelValue});
 */

/**
 * <p>Sub-ID for the DataGrid element's sort descending icon in column headers.</p>
 *
 * To lookup a sort icon the locator object should have the following:
 * <ul>
 * <li><b>subId</b>: 'oj-datagrid-sort-descending'</li>
 * <li><b>axis</b>: 'column'</li>
 * <li><b>index</b>: the zero based absolute column index</li>
 * <li><b>level</b>: the zero based header level, 0 is the outer edge, if not specified will default to 0</li>
 * </ul>
 *
 * @ojsubid oj-datagrid-sort-descending
 * @memberof oj.ojDataGrid
 *
 * @example <caption>Get the descending sort icon from the header at the specified location:</caption>
 * var node = myDataGrid.getNodeBySubId({subId: 'oj-datagrid-sort-descending', axis: 'axisValue', index: indexValue, level: levelValue});
 */
//-----------------------------------------------------
//                   Contexts
//-----------------------------------------------------

/**
 * <p>Context for the ojDataGrid element's cells.</p>
 *
 * @property {function} component a reference to the DataGrid widgetConstructor
 * @property {Object} cell the container data object for the header
 * @property {Object} data the data object for the header
 * @property {Object} datasource a reference to the data source object
 * @property {Object} indexes the object that contains both the zero based row index and column index in which the cell is bound to
 * @property {number} indexes.row the zero based absolute row index
 * @property {number} indexes.column the zero based absolute column index
 * @property {Object} keys the object that contains both the row key and column key which identifies the cell
 * @property {number|string} keys.row the row key
 * @property {number|string} keys.column the column key
 * @property {Object} extents the object that contains both the row extent and column extent of the cell
 * @property {number} extents.row the row extent
 * @property {number} extents.column the column extent
 * @property {string} mode the mode the cell is rendered in
 * @property {string} subId the subId of the cell
 *
 * @ojnodecontext oj-datagrid-cell
 * @memberof oj.ojDataGrid
 */
/**
 * <p>Context for the ojDataGrid element's headers.</p>
 *
 * @property {number} axis the axis of the header, possible values are 'row'/'column'/'columnEnd'/'rowEnd'
 * @property {function} component a reference to the DataGrid widgetConstructor
 * @property {Object} data the data object for the header
 * @property {Object} datasource a reference to the data source object
 * @property {number} depth the the number of levels the header spans
 * @property {number} extent the number of indexes the header spans
 * @property {number} index the index of the header, where 0 is the index of the first header
 * @property {number|string} key the key of the header
 * @property {number} level the level of the header. The outermost header is level zero
 * @property {string} subId the subId of the header
 *
 * @ojnodecontext oj-datagrid-header
 * @memberof oj.ojDataGrid
 */

/**
 * <p>Context for the ojDataGrid element's header labels.</p>
 *
 * @property {number} axis the axis of the header label, possible values are 'row'/'column'/'columnEnd'/'rowEnd'
 * @property {function} component a reference to the DataGrid widgetConstructor
 * @property {Object} data the data object for the header label
 * @property {Object} datasource a reference to the data source object
 * @property {number} level the level of the header label. The outermost header label is level zero
 * @property {string} subId the subId of the header label
 *
 * @ojnodecontext oj-datagrid-header-label
 * @memberof oj.ojDataGrid
 */
/**
 * @typedef {Object} oj.ojDataGrid.CellTemplateContext
 * @property {GridBodyItem<D>} item True if the current item is a leaf node.
 * @property {DataGridProvider<D>} datasource A reference to the grid data provider object.
 * @property {string} mode The mode of the row containing the cell. It can be "edit" or "navigation".
 * @ojsignature {target:"Type", value:"<D>", for:"genericTypeParameters"}
 */
/**
 * @typedef {Object} oj.ojDataGrid.HeaderTemplateContext
 * @property {GridHeaderItem<D>} item True if the current item is a leaf node.
 * @property {DataGridProvider<D>} datasource A reference to the grid data provider object.
 * @ojsignature {target:"Type", value:"<D>", for:"genericTypeParameters"}
 */
/**
 * @typedef {Object} oj.ojDataGrid.LabelTemplateContext
 * @property {GridItem<D>} item True if the current item is a leaf node.
 * @property {DataGridProvider<D>} datasource A reference to the grid data provider object.
 * @ojsignature {target:"Type", value:"<D>", for:"genericTypeParameters"}
 */
//-----------------------------------------------------
//                   Slots
//-----------------------------------------------------
/**
 * <b>Note: Inline Template Slots are only available when using a DataGridProvider.</b>
 * <p>The <code class="prettyprint">rowHeaderTemplate</code> slot is used to specify the template for the content of the row header.
 * <p>When the template is executed for each item, it will have access to the binding context containing the following properties:</p>
 * <ul>
 *   <li>$current - an object that contains information for the current item. (See [oj.ojDataGrid.HeaderTemplateContext]{@link oj.ojDataGrid.HeaderTemplateContext} </li>
 *   <li>alias - if as attribute was specified, the value will be used to provide an application-named alias for $current.</li>
 * </ul>
 *
 * @ojslot rowHeaderTemplate
 * @ojshortdesc The rowHeaderTemplate slot is used to specify the template for rendering the content of the row header. See the Help documentation for more information.
 * @ojmaxitems 1
 * @memberof oj.ojDataGrid
 * @ojtemplateslotprops oj.ojDataGrid.HeaderTemplateContext
 * @example <caption>Initialize the DataGrid with an inline row header template specified:</caption>
 * &lt;oj-data-grid>
 *   &lt;template slot='rowHeaderTemplate' data-oj-as='cell'>
 *     &lt;span>&lt;oj-bind-text value='[[cell.data.name]]'>&lt;/span>
 *   &lt;template>
 * &lt;/oj-data-grid>
 */
/**
 * <b>Note: Inline Template Slots are only available when using a DataGridProvider.</b>
 * <p>The <code class="prettyprint">rowEndHeaderTemplate</code> slot is used to specify the template for the content of the row end header.
 * <p>When the template is executed for each item, it will have access to the binding context containing the following properties:</p>
 * <ul>
 *   <li>$current - an object that contains information for the current item. (See [oj.ojDataGrid.HeaderTemplateContext]{@link oj.ojDataGrid.HeaderTemplateContext} </li>
 *   <li>alias - if as attribute was specified, the value will be used to provide an application-named alias for $current.</li>
 * </ul>
 *
 * @ojslot rowEndHeaderTemplate
 * @ojshortdesc The rowEndHeaderTemplate slot is used to specify the template for rendering the content of the row end header. See the Help documentation for more information.
 * @ojmaxitems 1
 * @memberof oj.ojDataGrid
 * @ojtemplateslotprops oj.ojDataGrid.HeaderTemplateContext
 * @example <caption>Initialize the DataGrid with an inline row end header template specified:</caption>
 * &lt;oj-data-grid>
 *   &lt;template slot='rowEndHeaderTemplate' data-oj-as='cell'>
 *     &lt;span>&lt;oj-bind-text value='[[cell.data.name]]'>&lt;/span>
 *   &lt;template>
 * &lt;/oj-data-grid>
 */
/**
 * <b>Note: Inline Template Slots are only available when using a DataGridProvider.</b>
 * <p>The <code class="prettyprint">columnHeaderTemplate</code> slot is used to specify the template for the content of the column header.
 * <p>When the template is executed for each item, it will have access to the binding context containing the following properties:</p>
 * <ul>
 *   <li>$current - an object that contains information for the current item. (See [oj.ojDataGrid.HeaderTemplateContext]{@link oj.ojDataGrid.HeaderTemplateContext} </li>
 *   <li>alias - if as attribute was specified, the value will be used to provide an application-named alias for $current.</li>
 * </ul>
 *
 * @ojslot columnHeaderTemplate
 * @ojshortdesc The columnHeaderTemplate slot is used to specify the template for rendering the content of the column header. See the Help documentation for more information.
 * @ojmaxitems 1
 * @memberof oj.ojDataGrid
 * @ojtemplateslotprops oj.ojDataGrid.HeaderTemplateContext
 * @example <caption>Initialize the DataGrid with an inline column header template specified:</caption>
 * &lt;oj-data-grid>
 *   &lt;template slot='columnHeaderTemplate' data-oj-as='cell'>
 *     &lt;span>&lt;oj-bind-text value='[[cell.data.name]]'>&lt;/span>
 *   &lt;template>
 * &lt;/oj-data-grid>
 */
/**
 * <b>Note: Inline Template Slots are only available when using a DataGridProvider.</b>
 * <p>The <code class="prettyprint">columnEndHeaderTemplate</code> slot is used to specify the template for the content of the column end header.
 * <p>When the template is executed for each item, it will have access to the binding context containing the following properties:</p>
 * <ul>
 *   <li>$current - an object that contains information for the current item. (See [oj.ojDataGrid.HeaderTemplateContext]{@link oj.ojDataGrid.HeaderTemplateContext} </li>
 *   <li>alias - if as attribute was specified, the value will be used to provide an application-named alias for $current.</li>
 * </ul>
 *
 * @ojslot columnEndHeaderTemplate
 * @ojshortdesc The columnEndHeaderTemplate slot is used to specify the template for rendering the content of the column end header. See the Help documentation for more information.
 * @ojmaxitems 1
 * @memberof oj.ojDataGrid
 * @ojtemplateslotprops oj.ojDataGrid.HeaderTemplateContext
 * @example <caption>Initialize the DataGrid with an inline column end header template specified:</caption>
 * &lt;oj-data-grid>
 *   &lt;template slot='columnEndHeaderTemplate' data-oj-as='cell'>
 *     &lt;span>&lt;oj-bind-text value='[[cell.data.name]]'>&lt;/span>
 *   &lt;template>
 * &lt;/oj-data-grid>
 */
/**
 * <b>Note: Inline Template Slots are only available when using a DataGridProvider.</b>
 * <p>The <code class="prettyprint">cellTemplate</code> slot is used to specify the template for the content of the cell.
 * <p>When the template is executed for each item, it will have access to the binding context containing the following properties:</p>
 * <ul>
 *   <li>$current - an object that contains information for the current item. (See [oj.ojDataGrid.CellTemplateContext]{@link oj.ojDataGrid.CellTemplateContext} </li>
 *   <li>alias - if as attribute was specified, the value will be used to provide an application-named alias for $current.</li>
 * </ul>
 *
 * @ojslot cellTemplate
 * @ojshortdesc The cellTemplate slot is used to specify the template for rendering the content of the cell. See the Help documentation for more information.
 * @ojmaxitems 1
 * @memberof oj.ojDataGrid
 * @ojtemplateslotprops oj.ojDataGrid.CellTemplateContext
 * @example <caption>Initialize the DataGrid with an inline cell template specified:</caption>
 * &lt;oj-data-grid>
 *   &lt;template slot='cellTemplate' data-oj-as='cell'>
 *     &lt;span>&lt;oj-bind-text value='[[cell.data.name]]'>&lt;/span>
 *   &lt;template>
 * &lt;/oj-data-grid>
 */
/**
 * <b>Note: Inline Template Slots are only available when using a DataGridProvider.</b>
 * <p>The <code class="prettyprint">rowHeaderLabelTemplate</code> slot is used to specify the template for the content of the row header label.
 * <p>When the template is executed for each item, it will have access to the binding context containing the following properties:</p>
 * <ul>
 *   <li>$current - an object that contains information for the current item. (See [oj.ojDataGrid.LabelTemplateContext]{@link oj.ojDataGrid.LabelTemplateContext} </li>
 *   <li>alias - if as attribute was specified, the value will be used to provide an application-named alias for $current.</li>
 * </ul>
 *
 * @ojslot rowHeaderLabelTemplate
 * @ojshortdesc The rowHeaderLabelTemplate slot is used to specify the template for rendering the content of the row header label. See the Help documentation for more information.
 * @ojmaxitems 1
 * @memberof oj.ojDataGrid
 * @ojtemplateslotprops oj.ojDataGrid.LabelTemplateContext
 * @example <caption>Initialize the DataGrid with