/** Copyright (c) 2019, 2022, Oracle and/or its affiliates. */

declare module "mle-js-oracledb" {

    /*
    Order of loading dependencies of mle-js-oracledb
    - soda-api.d.ts
    - api.d.ts
    - types.d.ts
    - connection.d.ts
    - soda.d.ts
    - mle-js-oracledb-common.d.ts
    */

    // ******************************************************
    // mle-js-oracledb-esm/mle-js-oracledb-common/soda-api.d.ts
    // ******************************************************

    /**
     * SODA API for MLE. This is compatible with node-oracledb 5.0.0.
     */
    /**
     * SodaDatabase.createCollection() options. The metadata must conform to the
     * JSON object layout specified in the Oracle Database
     * "SODA Collection Metadata Components (Reference)" documentation.
     */
    interface ICreateCollectionOptions {
        metaData?: Record<string, any>;
        mode?: number;
    }
    /**
     * SodaDatabase.createDocument() options.
     */
    interface ICreateDocumentOptions {
        key?: string;
        mediaType?: string;
    }
    /**
     * SodaDatabase.getCollectionNames() options.
     */
    interface IGetCollectionNameOptions {
        limit?: number;
        startsWith?: string;
    }
    /**
     * SodaCollection.dropIndex() options.
     */
    interface IDropIndexOptions {
        force?: boolean;
    }
    /**
     * SodaCollection.drop() return value.
     */
    interface IDropResult {
        dropped: boolean;
    }
    /**
     * SodaCollection.dropIndex() result.
     */
    interface IDropIndexResult {
        dropped: boolean;
    }
    /**
     * SodaOperation.count() result.
     */
    interface ICountResult {
        count: number;
    }
    /**
     * SodaOperation.remove() result.
     */
    interface IRemoveResult {
        count: number;
    }
    /**
     * SodaOperation.replace() result.
     */
    interface IReplaceResult {
        replaced: boolean;
    }
    /**
     * SODA database access class. SodaDatabase is the top level object for
     * SODA operations. A 'SODA database' is an abstraction, allowing access to SODA
     * collections in that 'SODA database', which then allow access to documents in
     * those collections.
     */
    interface ISodaDatabase {
        /**
         * Creates a SODA collection of the given name.
         * If a collection with the same name already exists,
         * then that existing collection is opened without error.
         * @param collectionName name of the collection to be created.
         * @param options the options that specify the collection, see
         *        CreateCollectionOptions.
         * @return a new SodaCollection object.
         */
        createCollection(collectionName: string, options?: ICreateCollectionOptions): ISodaCollection;
        /**
         * Constructs a proto SodaDocument object usable for SODA insert and replace
         * methods. SodaDocument attributes like createdOn will not be defined, and
         * neither will attributes valid in options but not specified. The document
         * will not be stored in the database until an insert or replace method is
         * called.
         * @param content the document content.
         * @param options the options that specify the document, see
         *                CreateDocumentOptions.
         * @return a new SodaDocument object.
         */
        createDocument(content: string | Uint8Array | Record<string, any>, options?: ICreateDocumentOptions): ISodaDocument;
        /**
         * Gets an array of collection names in alphabetical order.
         * Returns names that start with the given string, and all subsequent names,
         * in alphabetic order.
         * @param options see GetCollectionNameOptions.
         * @return an array of matching collection names.
         */
        getCollectionNames(options?: IGetCollectionNameOptions): Array<string>;
        /**
         * Opens an existing SodaCollection of the given name.
         * The collection can then be used to access documents.
         * @param collectionName the name of the collection to open.
         * @return a new SodaCollection object if the collection exists.
         *         If the requested collection does not exist undefined will be
         *         returned.
         */
        openCollection(collectionName: string): ISodaCollection;
    }
    /**
     * SODA collection class. A SODA collection is a set of SodaDocuments.
     */
    interface ISodaCollection {
        /**
         * Name of the collection.
         */
        readonly name: string;
        /**
         * Metadata for the collection. The metadata will conform to the JSON object
         * layout specified in the Oracle Database
         * "SODA Collection Metadata Components (Reference)" documentation.
         */
        readonly metaData: Record<string, any>;
        /**
         * Creates an index on a SODA collection, to improve the performance of SODA
         * query-by-examples (QBE) or enable text searches. Different index types can
         * be created, the indexSpec parameter must conform to the JSON object layout
         * specified in the Oracle Database "SODA Index Specifications (Reference)"
         * documentation.
         * @param indexSpec index specification,
         *        see "SODA Index Specifications (Reference)"
         * @throws an exception if the index creation fails.
         */
        createIndex(indexSpec: Record<string, any>): void;
        /**
         * Drops the current collection.
         * @return a DropResult containing a dropped value of true if the drop
         * operation succeeded or false if the collection does not exist.
         * @throws an exception if the collection drop fails.
         */
        drop(): IDropResult;
        /**
         * Drops the specified index.
         * @param indexName the name of the index to drop
         * @param options an optional force flag may be specified
         * @return a DropIndexResult containing a dropped value of true if the
         * drop index operation succeeded or false if the index doesn't exist.
         * @throws an exception if the index drop fails.
         */
        dropIndex(indexName: string, options?: IDropIndexOptions): IDropIndexResult;
        /**
         * Locate and order a set of SODA documents for retrieval, replacement,
         * or removal.
         * with non-terminal and terminal methods, see SodaOperation for details.
         * @return a SodaOperation object which is used via method chaining
         */
        find(): ISodaOperation;
        /**
         * Infers the schema of a collection of JSON documents.
         * The data guide is represented as JSON content in a SodaDocument.
         * @return a new SodaDocument containing the inferred schema.
         * @throws an exception if the schema inference fails.
         */
        getDataGuide(): ISodaDocument;
        /**
         * Inserts an array of Objects or SodaDocuments into the collection.
         * @param docs an array of Objects or SodaDocuments to be inserted into the
         *             collection.
         * @throws an exception if a document insertion fails. The offset attribute on
         * the Error object will contain the number of documents that were
         * successfully inserted. Subsequent documents in the input array will not be
         * inserted.
         */
        insertMany(docs: Array<Record<string, any> | ISodaDocument>): void;
        /**
         * Inserts an array of Objects or SodaDocuuments into the collection and
         * returns the documents which contain all SodaDocument components except for
         * content, for performance reasons.
         * @param docs an array of Objects or SodaDocuments to be inserted into the
         *             collection.
         * @return an array of inserted SodaDocuments.
         * @throws an exception if a document insertion fails. The offset attribute on
         * the Error object will contain the number of documents that were
         * successfully inserted. Subsequent documents in the input array will not be
         * inserted.
         */
        insertManyAndGet(docs: Array<Record<string, any> | ISodaDocument>): Array<ISodaDocument>;
        /**
         * Inserts a given document to the collection. The input document can be
         * either a JavaScript object representing the data content, or it can be an
         * existing SodaDocument.,
         * @param doc an Object or SodaDocument to insert into the collection.
         * @throws an exception if insertion fails.
         */
        insertOne(doc: Record<string, any> | ISodaDocument): void;
        /**
         * Inserts a document in a collection and returns the result document which
         * contains all SodaDocument components except for content, for performance
         * reasons.
         * @param doc the Object or SodaDoc to insert into the collection.
         * @return the inserted SodaDocument.
         * @throws an exception if insertion fails.
         */
        insertOneAndGet(doc: Record<string, any> | ISodaDocument): ISodaDocument;
        /**
         * This method behaves like sodaCollection.insertOne() with the exception that
         * if a document with the same key already exists, then it is updated instead.
         * The collection must use client-assigned keys keys, which is why save()
         * accepts only a SodaDocument, unlike insertOne(). If the collection is not
         * configured with client-assigned keys, then the behaviour is exactly the
         * same as sodaCollection.insertOne().
         * @param doc the document to save.
         */
        save(doc: ISodaDocument): void;
        /**
         * This method behaves like sodaCollection.insertOneAndGet() with the
         * exception that if a document with the same key already exists, then it is
         * updated instead. The collection must use client-assigned keys keys, which
         * is why saveAndGet() accepts only a SodaDocument, unlike insertOneAndGet().
         * If the collection is not configured with client-assigned keys, then the
         * behaviour is exactly the same as sodaCollection.insertOneAndGet().
         * @param doc the document to save.
         * @return the saved document.
         */
        saveAndGet(doc: ISodaDocument): ISodaDocument;
        /**
         * This method truncates a collection, removing all documents. The collection
         * will not be deleted.
         * @throws an exception if truncation fails.
         */
        truncate(): any;
    }
    /**
     * SODA find operation class. This class is used to search and retrieve SODA
     * documents from a SodaCollection. It provides non-terminal search condition
     * operations and terminal SodaDocument retrieval operations.
     */
    interface ISodaOperation {
        /**
         * Non-terminals.
         */
        /**
         * Sets the size of an internal buffer used for fetching documents from a
         * collection with the terminal SodaOperation methods getCursor() and
         * getDocuments().
         * @param size the buffer size to use.
         * @return the SodaOperation object.
         */
        fetchArraySize(size: number): ISodaOperation;
        /**
         * Sets a filter specification for the operation, allowing for complex
         * document queries and ordering of JSON documents.
         * @param filter the filter specification to use.
         * @return the SodaOperation object.
         */
        filter(filter: Record<string, any>): ISodaOperation;
        /**
         * Sets the key value to be used to match a document for the operation.
         * Any previous calls made to this method or keys() will be ignored.
         * @param key the search key to use.
         * @return the SodaOperation object.
         */
        key(key: string): ISodaOperation;
        /**
         * Sets the keys to be used to match multiple documents for the operation.
         * Any previous calls made to this method or key() will be ignored.
         * @param keys the search keys to use.
         * @return the SodaOperation object.
         */
        keys(keys: Array<string>): ISodaOperation;
        /**
         * Sets the maximum number of documents that a terminal method will apply to.
         * @param n the maximum number of documents to return. Must be greater than 0.
         * @return the SodaOperation object.
         */
        limit(n: number): ISodaOperation;
        /**
         * Sets the number of documents that will be skipped before the terminal
         * method is applied. n must be greater than or equal to 0.
         * @param n the number of documents to skip.
         * @return the SodaOperation object.
         */
        skip(n: number): ISodaOperation;
        /**
         * Sets the document version that retrieved documents must have.
         * @param value the version of retrieved documents.
         * @return the SodaOperation object.
         */
        version(value: string): ISodaOperation;
        /**
         * Terminals.
         */
        /**
         * Returns the number of documents matching the given SodaOperation query
         * criteria. If skip() or limit() are set, then an exception will be thrown.
         * @return a result object with a count field containing the number of
         * matching documents.
         * @throws an exception id skip() or limit() are set.
         */
        count(): ICountResult;
        /**
         * Returns a SodaDocumentCursor for documents that match the SodaOperation
         * query criteria.
         * @return a cursor that can be used to iterate over the matched documents.
         */
        getCursor(): ISodaDocumentCursor;
        /**
         * Gets an array of SodaDocuments matching the SodaOperation query criteria.
         * @return an array of documents, empty if none match.
         */
        getDocuments(): Array<ISodaDocument>;
        /**
         * Obtains one document matching the SodaOperation query criteria.
         * If more than one document is matched, then only the first is returned.
         * @return the first matching document, or undefined if none match.
         */
        getOne(): ISodaDocument;
        /**
         * Removes a set of documents matching the SodaOperation query criteria.
         * If skip() or limit(0 are set they are ignored.
         * @return a result object with a count field containing the number of
         * removed documents.
         */
        remove(): IRemoveResult;
        /**
         * Replaces a document in a collection. The input document can be either a
         * JavaScript object representing the data content, or it can be an existing
         * SodaDocument. The key() non-terminal must be used when using replaceOne().
         * @param newDoc the new content or SodaDocument.
         * @return a result object with a boolean replaced field which will be
         * true if the document was replaced successfully and false otherwise.
         */
        replaceOne(newDoc: Record<string, any> | ISodaDocument): IReplaceResult;
        /**
         * Replaces a document in a collection and return the result document which
         * contains all SodaDocument components except for the content.
         * @param newDoc the new content or SodaDocument.
         * @return The updated SodaDocument if replacement was successful, otherwise
         *         undefined.
         */
        replaceOneAndGet(newDoc: Record<string, any> | ISodaDocument): ISodaDocument;
    }
    /**
     * SODA document cursor class.
     * A SodaDocumentCursor is used to walk through a set of SODA documents returned
     * from a find() getCursor() method.
     */
    interface ISodaDocumentCursor {
        /**
         * Close the cursor.
         */
        close(): void;
        /**
         * Returns the next SodaDocument in the cursor returned by a find().
         * @return the next document, or undefined when there are no further
         * documents.
         */
        getNext(): ISodaDocument;
    }
    /**
     * SODA document class. SodaDocuments represents the document for SODA read and
     * write operations.
     */
    interface ISodaDocument {
        /**
         * SODA document properties.
         */
        /**
         * The creation time of the document as a string in the UTC time zone using an
         * ISO8601 format. By default, SODA sets this automatically.
         */
        readonly createdOn: string;
        /**
         * A unique key value for this document. By default, SODA automatically
         * generates the key.
         */
        readonly key: string;
        /**
         * Last modified time of the document as a string in the UTC time zone using
         * an ISO8601 format. By default, SODA sets this automatically.
         */
        readonly lastModified: string;
        /**
         * An arbitrary string value designating the content media type. The
         * recommendation when creating documents is to use a MIME type for the media
         * type. By default this property will be 'application/json'.
         */
        readonly mediaType: string;
        /**
         * Version of the document. By default, SODA automatically updates the version
         * each time the document is changed.
         */
        readonly version: string;
        /**
         * Return the document content as an object.
         * @return the document content as an object.
         * @throws an exception if the document content is not JSON and cannot be
         * converted to an object.
         */
        getContent(): Record<string, any>;
        /**
         * Return the document content as a Uint8Array. If the collection storage is
         * BLOB (default) and the mediaType is 'application/json', then the returned
         * Uint8Array is identical to the one that was stored. If the storage is not
         * BLOB, it is UTF-8 encoded.
         * @return the document content as a Uint8Array.
         */
        getContentAsUint8Array(): Uint8Array;
        /**
         * Return JSON document content as a string. If the document encoding is
         * unknown, UTF-8 will be used.
         * @return the document content as a string.
         */
        getContentAsString(): string;
    }

    // ******************************************************
    // mle-js-oracledb-esm/mle-js-oracledb-common/api.d.ts
    // ******************************************************

    /**
     * Interface for errors thrown by {@link execute}() or {@link executeMany}().
     */
    interface IError {
        /**
         * The Oracle error number. This value is undefined for non-Oracle errors.
         */
        errorNum?: number;
        /**
         * The text of the error message.
         */
        message: string;
        /**
         * The character offset into the SQL text that resulted in the Oracle
         * error. The value may be 0 in non-SQL contexts. This value is undefined
         * for non-Oracle errors.
         */
        offset?: number;
    }
    /**
     * Interface for representing entries in {@link IFetchInfo}.
    */
    interface IFetchInfoColumnSpec {
        /**
         * The JavaScript data type to be fetched. One of the mle-js-oracledb JS
         * Type Constants.
         */
        type: JsType;
    }
    /**
     * Interface for representing {@link fetchInfo} in {@link execute}().
     */
    interface IFetchInfo {
        [columnName: string]: IFetchInfoColumnSpec;
    }
    /**
     * Interface for options used in {@link execute}().
     */
    interface IExecuteOptions {
        /**
         * Overrides parameters.{@link extendedMetaData}.
         */
        extendedMetaData?: boolean;
        /**
         * Overrides parameters.{@link fetchArraySize}.
         */
        fetchArraySize?: number;
        /**
         * Object defining how query column data should be represented in
         * JavaScript. It can be used in conjunction with, or instead of, the global
         * settings parameters.{@link fetchAsString}, parameters.{@link
         * fetchAsPlsqlWrapper}, and parameters.{@link fetchAsUint8Array}.
         */
        fetchInfo?: IFetchInfo;
        /**
         * Overrides parameters.{@link maxRows}.
         */
        maxRows?: number;
        /**
         * Overrides parameters.{@link outFormat}.
         */
        outFormat?: OutFormatType;
        /**
         * Determines whether query results should be returned as a {@link
         * IResultSet} object or directly. The default is false.
         */
        resultSet?: boolean;
    }
    /**
     * Interface for representing an entry in {@link IObjectBindDefs} or {@link ArrayBindDefs}.
    */
    interface IBindDef {
        /**
         * The direction of the bind. One of the Execute Bind Direction Constants
         * {@link BIND_IN}, {@link BIND_INOUT}, or {@link BIND_OUT}. The default is
         * {@link BIND_IN}.
         */
        dir: number;
        /**
         * Required for Strings and Uint8Arrays. Ignored for other types. Specifies
         * the maximum number of bytes allocated when processing each value of this
         * bind variable. When data is being passed into the database, maxSize
         * must be at least the size of the longest value. When data is being
         * returned from the database, maxSize must be the size of the longest
         * value. If maxSize is too small, {@link executeMany} will throw an error.
         */
        maxSize?: number;
        /**
         * The JavaScript data type to be bound. One of the mle-js-oracledb JS Type
         * Constants.
         */
        type: JsType;
    }
    /**
     * Interface for Object BindDefs.
     */
    interface IObjectBindDefs {
        [bindName: string]: IBindDef;
    }
    /**
     * Interface for Array BindDefs.
     */
    type ArrayBindDefs = IBindDef[];
    /**
     * Interface for BindDefs which are either Array- or Object BindDefs.
     */
    type ExecuteManyBindDefs = IObjectBindDefs | ArrayBindDefs;
    /**
     * Interface for the options used in {@link executeMany}().
     */
    interface IExecuteManyOptions {
        /**
         * The bindDefs object defines the bind variable types, sizes and
         * directions.  This object is optional, but makes execution more efficient
         * if set explicitly.
         *
         * It must be an array (see {@link ArrayBindDefs}) or an object (see
         * {@link IObjectBindDefs}), depending on the structure of the binds
         * parameter.
         *
         * Each value in the bindDefs array or object must be an object containing
         * the keys dir, maxSize, and type for each bind variable, similar to how
         * {@link execute} bind parameters are defined (see {@link IBindDef}).
         */
        bindDefs?: ExecuteManyBindDefs;
        dmlRowCounts?: boolean;
    }
    /**
     * Interface representing meta data as used in {@link IResultSet}s and statement info.
     */
    interface IMetaData {
        /**
         * The column name follows Oracle’s standard name-casing rules. It will commonly be uppercase,
         * since most applications create tables using unquoted, case-insensitive names.
         */
        name: string;
        /**
         * One of the mle-js-oracledb JS Type Constants.
         */
        fetchType?: JsType;
        /**
         * One of the mle-js-oracledb Database Type Constants, see {@link DbType}.
         */
        dbType?: number;
        /**
         * Database byte size. This is only set for {@link DB_TYPE_VARCHAR}, {@link
         * DB_TYPE_CHAR} and {@link DB_TYPE_RAW} column types.
         */
        byteSize?: number;
        /**
         * Set only for {@link DB_TYPE_NUMBER}, {@link DB_TYPE_TIMESTAMP}, {@link
         * DB_TYPE_TIMESTAMP_TZ} and {@link DB_TYPE_TIMESTAMP_LTZ} columns.
         */
        precision?: number;
        /**
         * Set only for {@link DB_TYPE_NUMBER} columns.
         */
        scale?: number;
        /**
         * Indicates whether NULL values are permitted for this column.
         */
        nullable?: boolean;
        /**
         * Name of the database type, such as “NUMBER” or “VARCHAR2”.
         */
        dbTypeName?: string;
    }
    /**
     * Interface for representing result sets as returned by {@link execute}().
     */
    interface IResultSet {
        /**
         * Contains an array of objects with metadata about the query.
         *
         * Each column's name is always given. If {@link extendedMetaData} is true,
         * additional information is included.
         */
        readonly metaData: IMetaData[];
        /**
         * Closes a result set.
         *
         * Applications must always call this at the end of fetch or when no more rows are needed.
         *
         * It must also be called if no rows are ever going to be fetched from the result set.
         */
        close(): any;
        /**
         * This call fetches one row of the result set as an object or an array of
         * column values, depending on the value of outFormat.
         *
         * At the end of fetching, the result set must be freed by calling {@link close}().
         *
         * Performance of getRow() can be tuned by adjusting the value of
         * {@link fetchArraySize}.
         */
        getRow(): any;
        /**
         * This call fetches numRows rows of the result set as an object or an array of
         * column values, depending on the value of outFormat.
         *
         * @param numRows specifies the number of rows to be returned.
         * the default value of numRows is 0 and it returns all rows.
         *
         * At the end of fetching, the result set must be freed by calling {@link close}().
         *
         * Performance of getRows() can be tuned by adjusting the value of
         * {@link fetchArraySize}.
         */
        getRows(numRows: number): any[];
        /**
         * Convenience function for getting an iterator of this IResultSet.
         *
         * This is equivalent to calling rs[Symbol.iterator]().
         *
         * @returns an iterator over the rows of this IResultSet.
         *
         * @throws {@link IError} if the result set has already been closed
         * @throws {@link IError} if the result set is already being iterated over
         */
        iterator(): IterableIterator<any>;
        /**
         * This function defines the default iterator for a result set which can be
         * used to iterate over its rows. Using the default iterator, a result set
         * can be iterated over using the for..of construct.
         *
         * @throws {@link IError} if the result set has already been closed
         * @throws {@link IError} if the result set is already being iterated over
         *
         */
        [Symbol.iterator](): IterableIterator<any>;
    }
    /**
     * Interface for the result of {@link execute}().
     */
    interface IExecuteReturn {
        /**
         * For SELECT statements, this contains an array of objects describing
         * details of columns for the select list. For non queries, this property is
         * undefined.
         */
        metaData?: IMetaData[];
        /**
         * This contains the output values of OUT and IN OUT binds. If bindParams is
         * passed as an array, then outBinds is returned as an array. If bindParams
         * is passed as an object, then outBinds is returned as an object. If there
         * are no OUT or IN OUT binds, the value is undefined.
         */
        outBinds?: Record<string, any> | any[];
        /**
         * For SELECT statements when the {@link resultSet} option is true, use the
         * resultSet object to fetch rows. See {@link IResultSet}.
         *
         * When using this option, resultSet.{@link close}() must be called when the
         * result set is no longer needed. This is true whether or not rows have
         * been fetched.
         */
        resultSet?: IResultSet;
        /**
         * For SELECT statements using direct fetches, rows contains an array of
         * fetched rows. It will be NULL if there is an error or the SQL statement
         * was not a SELECT statement. By default, the rows are in an array of
         * column value arrays, but this can be changed to arrays of objects by
         * setting outFormat to oracledb.{@link OBJECT}. If a single row is fetched,
         * then rows is an array that contains one single row.
         *
         * The number of rows returned is limited by parameters.{@link maxRows} or
         * the {@link maxRows} option in an {@link execute}() call. If maxRows is 0,
         * then the number of rows is limited by memory constraints.
         */
        rows?: any[];
        /**
         * For DML statements (including SELECT FOR UPDATE) this contains the number
         * of rows affected, for example the number of rows inserted. For non-DML
         * statements such as queries and PL/SQL statements, rowsAffected is
         * undefined.
         */
        rowsAffected?: number;
    }
    /**
     * Interface for the result of {@link executeMany}().
     */
    interface IExecuteManyReturn {
        /**
         * This contains the value of any returned IN OUT or OUT binds. It is an
         * array of arrays, or an array of objects, depending on the {@link binds}
         * parameters structure. The length of the array will correspond to the
         * length of the array passed as the binds parameter. It will be present
         * only if there is at least one OUT bind variable identified.
         */
        outBinds?: (Record<string, any> | any[])[];
        /**
         * This is an integer identifying the total number of database rows affected
         * by the processing of all records of the binds parameter. It is only
         * present if a DML statement was executed.
         */
        rowsAffected?: number;
    }
    /**
     * Interface for object binds in {@link execute}().
     */
    interface IBindObjectValue {
        /**
         * The direction of the bind. One of the constants {@link BIND_IN},
         * {@link BIND_INOUT}, or {@link BIND_OUT}. The default is
         * {@link BIND_IN}.
         */
        dir?: number;
        /**
         * The maximum number of bytes that an OUT or INOUT bind variable of type
         * {@link STRING} or {@link UINT8ARRAY} can use to get data. The default
         * value is 200. The maximum limit depends on the database type. When
         * binding INOUT, then maxSize refers to the size of the returned value.
         * The input value can be smaller or bigger. For IN binds, maxSize is
         * ignored.
         */
        maxSize?: number;
        /**
         * The JavaScript data type to be bound. One of the mle-js-oracledb JS
         * Constants.  With IN or INOUT binds the type can be explicitly set with
         * type or it will default to the type of the input data value. With OUT
         * binds, the type defaults to {@link STRING} whenever type is not
         * specified.
         */
        type?: number;
        /**
         * The input value or variable to be used for an IN or INOUT bind variable.
         */
        val: any;
    }
    /**
     * Interface for a single bind parameter as used in {@link execute}(). Can
     * either be an bind object or a scalar value.
     */
    type BindValue = IBindObjectValue | any;
    /**
     * Interface for object binds (also called "named binds").
     */
    interface INamedBinds {
        [bindName: string]: BindValue;
    }
    /**
     * Interface for array binds.
     */
    type PosBinds = BindValue[];
    /**
     * Interface for the collection of bind parameters in {@link execute}(). Can
     * either be an object ({@link INamedBinds}) or an array ({@link PosBinds}) of
     * values.
     */
    type BindParameters = INamedBinds | PosBinds;
    /**
     * Interface for the result of {@link getStatementInfo}().
     */
    interface IStatementInfo {
        /**
         * Array of strings corresponding to the unique names of the bind variables
         * used in the SQL statement.
         */
        bindNames: string[];
        /**
         * Array containing properties equivalent to those in {@link
         * extendedMetaData} on {@link execute}(). This property exists only for
         * queries.
         */
        metaData?: IMetaData[];
        /**
         * Integer corresponding to one of the mle-js-oracledb SQL Statement Type
         * Constants, e.g.  {@link STMT_TYPE_SELECT}.
         */
        statementType: number;
    }
    /**
     * Interface for the connection object obtained by {@link defaultConnection}.
     */
    interface IConnection {
        /**
         * This read-only property gives a numeric representation of the Oracle
         * database version which is useful in comparisons. For version a.b.c.d.e,
         * this property gives the number:
         * (100000000 * a) + (1000000 * b) + (10000 * c) + (100 * d) + e
         */
        readonly oracleServerVersion: number;
        /**
         * This read-only property gives a string representation of the Oracle
         * database version which is useful for display.
         */
        readonly oracleServerVersionString: string;
        /**
         * This read-only property always returns 0 and exists for consistency with
         * node-oracledb.
         */
        readonly stmtCacheSize: number;
        /**
         * This call commits the current transaction in progress.
         */
        commit(): void;
        /**
         * This call executes a single SQL or PL/SQL statement.
         *
         * The statement to be executed may contain IN binds, OUT or IN OUT bind
         * values or variables, which are bound using either an object or an array.
         *
         * The function returns a result object, containing any fetched rows, the
         * values of any OUT and IN OUT bind variables, and the number of rows
         * affected by the execution of DML statements, see {@link IExecuteReturn}.
         *
         * @param sql SQL statement that is executed. The statement may contain bind
         * parameters.
         * @param bindParams needed if there are bind parameters in the SQL
         * statement, see {@link BindParameters}.
         * @param options an optional parameter to execute() that may be used to
         * control statement execution.
         */
        execute(sql: string): IExecuteReturn;
        execute(sql: string, bindParams: BindParameters, options?: IExecuteOptions): IExecuteReturn;
        /**
         * This method allows sets of data values to be bound to one DML or PL/SQL
         * statement for execution. It is like calling {@link execute}() multiple
         * times but requires fewer context switches. This is an efficient way to
         * handle batch changes, for example when inserting or updating multiple
         * rows. The method cannot be used for queries.
         *
         * The executeMany() method supports IN, IN OUT and OUT binds for most data
         * types.
         *
         * The version of this function which accepts a number of iterations must
         * be used when no bind parameters are required or when all bind parameters
         * are OUT binds.
         *
         * @param sql SQL or PL/SQL statement that executeMany() executes.
         * @param binds contains the values or variables to be bound to the executed
         * statement. It must be an array of arrays ({@link ArrayBindDefs}) or an
         * array of objects whose keys match the bind variable names in the SQL
         * statement ({@link IObjectBindDefs}). Each sub-array or sub-object must
         * contain values for the bind variables used in the SQL statement. At least
         * one such record must be specified.
         *
         * If a record contains fewer values than expected, NULL values will be
         * used. For bind by position, empty values can be specified using syntax
         * like [a,,c,d].
         *
         * By default, the direction of binds is {@link BIND_IN}. The first data
         * record determines the number of bind variables, each bind variable's data
         * type, and its name (when binding by name). If a variable in the first
         * record contains a null, this value is ignored and a subsequent record is
         * used to determine that variable's characteristics. If all values in all
         * records for a particular bind variable are null, the type of that bind is
         * {@link STRING} with a maximum size of 1.
         *
         * The maximum sizes of strings and Uint8Arrays are determined by scanning
         * all records in the bind data.
         *
         * If a {@link bindDefs} property is used in options, no data scanning
         * occurs. This property explicitly specifies the characteristics of each
         * bind variable.
         * @param numIterations number of times the SQL statement should be
         * executed. Can be used instead of the binds parameter.
         * @param options The options parameter is optional. It can contain the
         * properties specified in {@link IExecuteManyOptions}.
         */
        executeMany(sql: string, binds: BindParameters[], options?: IExecuteManyOptions): IExecuteManyReturn;
        executeMany(sql: string, numIterations: number, options?: IExecuteManyOptions): IExecuteManyReturn;
        /**
         * Parses a SQL statement and returns information about it. This is most
         * useful for finding column names of queries, and for finding the names of
         * bind variables used.
         *
         * This method performs a call to the SQL layer of the database, so
         * unnecessary calls should be avoided.
         *
         * The information is provided by lower level APIs that have some
         * limitations. Some uncommon statements will return the statement type as
         * {@link STMT_TYPE_UNKNOWN}. DDL statements are not parsed, so syntax
         * errors in them will not be reported. The direction and types of bind
         * variables cannot be determined.
         *
         * @param sql SQL statement to parse.
         */
        getStatementInfo(sql: string): IStatementInfo;
        /**
         * This call rolls back the current transaction in progress.
         */
        rollback(): void;
        /**
         * Returns a parent SodaDatabase object.
         * @return a new SodaDatabase object.
         */
        getSodaDatabase(): ISodaDatabase;
    }
    /**
     * Type for mle-js-oracledb Query OutFormat Constants.
     */
    type OutFormatType = number;
    /**
     * Fetch each row as array of column values
     * This constant is deprecated. Use OUT_FORMAT_ARRAY instead.
     */
    const ARRAY: OutFormatType;
    /**
     * Fetch each row as array of column values.
     */
    const OUT_FORMAT_ARRAY: OutFormatType;
    /**
     * Fetch each row as an object
     * This constant is deprecated. Use OUT_FORMAT_OBJECT instead.
     */
    const OBJECT: OutFormatType;
    /**
     * Fetch each row as an object of column values.
     */
    const OUT_FORMAT_OBJECT: OutFormatType;
    type JsType = number;
    /**
     * Used with fetchInfo to reset the fetch type to the database type
     */
    const DEFAULT: JsType;
    /**
     * Bind as JavaScript String type. Can be used for most database types.
     */
    const STRING: JsType;
    /**
     * Bind as JavaScript number type. Can also be used for fetchAsString and
     * fetchInfo.
     */
    const NUMBER: JsType;
    /**
     * Bind as JavaScript date type. Can also be used for fetchAsString and
     * fetchInfo.
     */
    const DATE: JsType;
    /**
     * Bind a JavaScript boolean to a PL/SQL BOOLEAN.
     */
    const BOOLEAN: JsType;
    /**
     * Bind a NUMBER to a OracleNumber object.
     */
    const ORACLE_NUMBER: JsType;
    /**
     * Bind a DATE to a OracleDate object.
     */
    const ORACLE_DATE: JsType;
    /**
     * Bind a BLOB to a OracleBLOB object.
     */
    const ORACLE_BLOB: JsType;
    /**
     * Bind a CLOB to a OracleCLOB object.
     */
    const ORACLE_CLOB: JsType;
    /**
     * Bind a INTERVAL DAY TO SECOND to a OracleIntervalDayToSecond object.
     */
    const ORACLE_INTERVAL_DS = 9012;
    /**
     * Bind a INTERVAL YEAR TO MONTH to a OracleIntervalYearToMonth object.
     */
    const ORACLE_INTERVAL_YM = 9013;
    /**
     * Bind a NCLOB to a OracleNCLOB object.
     */
    const ORACLE_NCLOB: JsType;
    /**
     * Bind a RAW, LONG RAW or BLOB to a Uint8Array typed array.
     */
    const UINT8ARRAY = 9015;
    /**
     * Bind a TIMESTAMP to a OracleTimestamp object.
     */
    const ORACLE_TIMESTAMP = 9016;
    /**
     * Bind a TIMESTAMP WITH TIME ZONE or TIMESTAMP WITH LOCAL TIME ZONE to a OracleTimestampTZ object.
     */
    const ORACLE_TIMESTAMP_TZ = 9017;
    /**
     * Bind a DTYDJSON value.
     */
    const ORACLE_JSON = 9018;
    /**
     * Type for mle-js-oracledb Database Type Constants.
     */
    type DbType = number;
    /**
     * Returns the literal type name of a mle-js-oracledb Database Type Constant.
     *
     * @param typeConst Database Type Constant
     * @returns name of the Database Type Constant
     */
    function dbTypeNameByValue(typeConst: DbType): string;
    /**
     * VARCHAR2
     */
    const DB_TYPE_VARCHAR: DbType;
    /**
     * NUMBER or FLOAT
     */
    const DB_TYPE_NUMBER: DbType;
    /**
     * LONG
     */
    const DB_TYPE_LONG: DbType;
    /**
     * DATE
     */
    const DB_TYPE_DATE: DbType;
    /**
     * RAW
     */
    const DB_TYPE_RAW: DbType;
    /**
     * LONG RAW
     */
    const DB_TYPE_LONG_RAW: DbType;
    /**
     * CHAR
     */
    const DB_TYPE_CHAR: DbType;
    /**
     * BINARY_FLOAT
     */
    const DB_TYPE_BINARY_FLOAT: DbType;
    /**
     * BINARY_DOUBLE
     */
    const DB_TYPE_BINARY_DOUBLE: DbType;
    /**
     * BINARY_INTEGER
     */
    const DB_TYPE_BINARY_INTEGER: DbType;
    /**
     * ROWID
     */
    const DB_TYPE_ROWID: DbType;
    /**
     * CLOB
     */
    const DB_TYPE_CLOB: DbType;
    /**
     * BLOB
     */
    const DB_TYPE_BLOB: DbType;
    /**
     * TIMESTAMP
     */
    const DB_TYPE_TIMESTAMP: DbType;
    /**
     * TIMESTAMP WITH TIME ZONE
     */
    const DB_TYPE_TIMESTAMP_TZ: DbType;
    /**
     * INTERVAL YEAR TO MONTH
     */
    const DB_TYPE_INTERVAL_YM: DbType;
    /**
     * INTERVAL DAY TO SECOND
     */
    const DB_TYPE_INTERVAL_DS: DbType;
    /**
     * UROWID
     */
    const DB_TYPE_UROWID: DbType;
    /**
     * BOOLEAN
     */
    const DB_TYPE_BOOLEAN: DbType;
    /**
     * TIMESTAMP WITH LOCAL TIME ZONE
     */
    const DB_TYPE_TIMESTAMP_LTZ: DbType;
    /**
     * NVARCHAR
     */
    const DB_TYPE_NVARCHAR: DbType;
    /**
     * NCHAR
     */
    const DB_TYPE_NCHAR: DbType;
    /**
     * NCLOB
     */
    const DB_TYPE_NCLOB: DbType;
    /**
     * JSON
     */
    const DB_TYPE_JSON: DbType;
    /**
     * Direction for IN binds
     */
    const BIND_IN = 3001;
    /**
     * Direction for INOUT binds
     */
    const BIND_INOUT = 3002;
    /**
     * Direction for OUT binds
     */
    const BIND_OUT = 3003;
    /**
     * Unknown statement type
     */
    const STMT_TYPE_UNKNOWN = 0;
    /**
     * SELECT
     */
    const STMT_TYPE_SELECT = 1;
    /**
     * UPDATE
     */
    const STMT_TYPE_UPDATE = 2;
    /**
     * DELETE
     */
    const STMT_TYPE_DELETE = 3;
    /**
     * INSERT
     */
    const STMT_TYPE_INSERT = 4;
    /**
     * CREATE
     */
    const STMT_TYPE_CREATE = 5;
    /**
     * DROP
     */
    const STMT_TYPE_DROP = 6;
    /**
     * ALTER
     */
    const STMT_TYPE_ALTER = 7;
    /**
     * BEGIN
     */
    const STMT_TYPE_BEGIN = 8;
    /**
     * DECLARE
     */
    const STMT_TYPE_DECLARE = 9;
    /**
     * CALL
     */
    const STMT_TYPE_CALL = 10;
    /**
     * EXPLAIN PLAN
     */
    const STMT_TYPE_EXPLAIN_PLAN = 15;
    /**
     * MERGE
     */
    const STMT_TYPE_MERGE = 16;
    /**
     * ROLLBACK
     */
    const STMT_TYPE_ROLLBACK = 17;
    /**
     * COMMIT
     */
    const STMT_TYPE_COMMIT = 21;
    /**
     * SODA_COLL_MAP_MODE
     */
    const SODA_COLL_MAP_MODE = 5001;
    /**
     * Class for representing global mle-js-oracledb properties.
     */
    class Parameters {
        /**
         * The maximum number of rows that are fetched by a query with
         * connection.{@link execute}() when not using an {@link IResultSet}. Rows
         * beyond this limit are not fetched from the database. A value of 0 means
         * there is no limit.
         *
         * The default value is 0, meaning unlimited.
         *
         * This property may be overridden in an {@link execute}() call.
         *
         * To improve database efficiency, SQL queries should use a row limiting clause
         * like OFFSET / FETCH or equivalent. The maxRows property can be used to stop
         * badly coded queries from returning unexpectedly large numbers of rows.
         *
         * When the number of query rows is relatively big, or can not be predicted, it
         * is recommended to use an {@link IResultSet}. This allows applications to
         * process rows in smaller chunks or individually, preventing the PGA limits
         * being exceeded or query results being unexpectedly truncated by a maxRows
         * limit.
         */
        private _maxRows;
        get maxRows(): number;
        set maxRows(value: number);
        /**
         * The format of query rows fetched when using connection.{@link execute}().
         * It affects both IResultSet and non-IResultSet queries. This can be either
         * of the constants {@link OUT_FORMAT_ARRAY} or {@link OUT_FORMAT_OBJECT}. The default value is
         * {@link OUT_FORMAT_ARRAY}.
         *
         * If specified as {@link OUT_FORMAT_ARRAY}, each row is fetched as an array of column
         * values.
         *
         * If specified as {@link OUT_FORMAT_OBJECT}, each row is fetched as a JavaScript object.
         * The object has a property for each column name, with the property value set
         * to the respective column value. The property name follows Oracle's standard
         * name-casing rules. It will commonly be uppercase, since most applications
         * create tables using unquoted, case-insensitive names.
         *
         * This property may be overridden in an {@link execute}() call.
         */
        private _outFormat;
        get outFormat(): OutFormatType;
        set outFormat(value: OutFormatType);
        /**
         * This property sets the size of an internal buffer used for fetching query
         * rows from Oracle Database. Changing it may affect query performance but
         * does not affect how many rows are returned to the application.
         *
         * The default value is 10. The maximum allowed value is 1000, any value
         * greater than that will silently be limited to 1000.
         *
         * The property is used during the default direct fetches and during
         * {@link IResultSet}.{@link getRow}() and {@link getRows}() calls.
         *
         * Increasing this value reduces the number of context switches, but increases
         * memory usage for each data fetch. For queries that return a large number of
         * rows, higher values of fetchArraySize may give better performance. For
         * queries that only return a few rows, reduce the value of fetchArraySize to
         * minimize the amount of memory management during data fetches. Note that as
         * mle-js-oracledb is co-located with the database, large values are unlikely
         * to yield significant benefit.
         *
         * For direct fetches (those using {@link resultSet}: false), the
         * internal buffer size will be based on the lesser of maxRows and
         * fetchArraySize.
         */
        private _fetchArraySize;
        get fetchArraySize(): number;
        set fetchArraySize(value: number);
        /**
         * Determines whether additional metadata is available for queries.
         *
         * The default value is false. With this value, the result.{@link metaData}
         * and result.{@link resultSet}.{@link metaData} objects only include column
         * names.
         *
         * This property may be overridden in an execute() call.
         */
        private _extendedMetaData;
        get extendedMetaData(): boolean;
        set extendedMetaData(value: boolean);
        /**
         * An array of mle-js-oracledb JS Type values. The valid types are {@link
         * DATE}, {@link NUMBER}, {@link UINT8ARRAY}, and {@link ORACLE_CLOB}. When
         * any column having one of the specified types is queried with {@link
         * execute}(), the column data is returned as a string instead of the default
         * representation.
         *
         * By default in mle-js-oracledb, all columns are returned as native types or
         * or as OracleClob/OracleBlob wrapper types, in the case of CLOB and BLOB
         * types.
         *
         * This property helps avoid situations where using JavaScript types can lead
         * to numeric precision loss, or where date conversion is unwanted.
         *
         * For raw data returned as a string, Oracle returns the data as a
         * hex-encoded string. For dates and numbers returned as a string, the
         * maximum length of a string created by this mapping is 200 bytes. Strings
         * created for CLOB columns will generally be limited by memory restrictions.
         *
         * Individual query columns in {@link execute}() calls can override the
         * fetchAsString global property by using {@link fetchInfo}.
         */
        private _fetchAsString;
        get fetchAsString(): JsType[];
        set fetchAsString(value: JsType[]);
        /**
         * An array of mle-js-oracledb JS Type values. Currently the only valid type
         * is {@link ORACLE_BLOB}. When a BLOB column is queried with {@link
         * execute}(), the column data is returned as a Uint8Array instead of the
         * default representation.
         *
         * By default in mle-js-oracledb, all columns are returned as native types or
         * or as OracleClob/OracleBlob wrapper types, in the case of CLOB and BLOB
         * types.
         *
         * Individual query columns in {@link execute}() calls can override the
         * fetchAsUint8Array global property by using {@link fetchInfo}.
         */
        private _fetchAsUint8Array;
        get fetchAsUint8Array(): JsType[];
        set fetchAsUint8Array(value: JsType[]);
        /**
         * An array of mle-js-oracledb JS Type values. The valid types are {@link
         * DATE}, {@link NUMBER}, and {@link STRING}. When any column having one of
         * the specified types is queried with {@link execute}(), the column data is
         * returned as a PL/SQL wrapper type instead of the default representation.
         *
         * By default in mle-js-oracledb, all columns are returned as native types or
         * or as OracleClob/OracleBlob wrapper types, in the case of CLOB and BLOB
         * types.
         *
         * For types that be set in both properties ({@link fetchAsString} and
         * fetchAsPlsqlWrapper), i.e. {@link DATE} and {@link NUMBER}, the {@link
         * fetchAsString} property has precedence over the fetchAsPlsqlWrapper
         * property.
         *
         * Individual query columns in {@link execute}() calls can override the
         * fetchAsPlsqlWrapper global property by using {@link fetchInfo}.
         */
        private _fetchAsPlsqlWrapper;
        get fetchAsPlsqlWrapper(): JsType[];
        set fetchAsPlsqlWrapper(value: JsType[]);
    }
    /**
     * Object that holds the global mle-js-oracledb properties.
     */
    const parameters: Parameters;

    // ******************************************************
    // mle-js-oracledb-esm/mle-js-oracledb-common/types.d.ts
    // ******************************************************

    /**
     * Interop types tell MSD which type to generate when converting from RDBMS values.
     */
    enum InteropType {
        NUMBER,
        BOOLEAN,
        STRING,
        INTEGRAL,
        FLOATING_POINT,
        ORACLE_NUMBER,
        ORACLE_DATE,
        ORACLE_CLOB,
        ORACLE_BLOB,
        BYTE_ARRAY,
        ORACLE_TIMESTAMP_TZ,
        ORACLE_TIMESTAMP,
        ORACLE_INTERVAL_YM,
        ORACLE_INTERVAL_DS,
        ORACLE_JSON
    }
    enum SQLKind {
        SELECT,
        DELETE,
        INSERT,
        UPDATE,
        PLSQL,
        CALL,
        CREATETABLE,
        CREATEPROCEDURE,
        CREATEFUNCTION,
        CREATEPACKAGE,
        DROPPROCEDURE,
        DROPFUNCTION,
        DROPPACKAGE,
        TRANSACTION
    }
    /**
     * Mapping: SQL kind -> statement type
     */
    const sqlKindToStmtType: {
        [sqlKind: number]: number;
    };
    /**
     * Mapping: RDBMS type -> JS default type for result columns
     */
    function getDefaultJsTypeForColumnFetch(dbType: number, columnId: number): number;
    /**
     * Mapping: database type number to database type name.
     */
    function getDbTypeName(dbType: number): string;
    /**
     * Mapping: JS type -> RDBMS type for setting IN/INOUT bind values.
     * From node-oracledb 4.0 JS Types Values changed and became alias of common Oracle Database Type Constants
     */
    function getDbTypeForBindSet(jsType: number, parameter: number | string): number;
    /**
     * Determine the database type that should be used to fetch a column. The fetch type depends on the original DB type of the column and the target JavaScript type.
     *
     * @param rdbmsType Original DB type of a column
     * @param jsType Target JavaScript type
     * @param parameter Optional parameters for error reporting
     */
    function getFetchType(rdbmsType: DbType, jsType: JsType, parameter?: string | number): DbType;
    /**
     * Determine the interop type to request from MSD when converting values of a given RDBMS type into values of a given JavaScript type.
     * @param rdbmsType RDBMS type code (origin type)
     * @param jsType mle-js-oracledb JavaScript type code (target type)
     * @param parameter column id or bind identifier (position or name). Used for rendering of error messages.
     */
    function getBridgeInteropType(rdbmsType: number, jsType: number, parameter?: string | number): InteropType;
    /**
     *  Maps a given database type to a wrapper type if that wrapper type exists.
     *  Otherwise returns the default JavaScript type instead.
     * @param rdbmsType RDBMS type code (origin type)
     */
    function getPlsqlWrapperType(rdbmsType: number, defaultJsType: number): number;
    /**
     * Convert a value of an interop type into a proper JavaScript value.
     * @param dbType RDBMS type code (origin type)
     * @param jsType mle-js-oracledb JavaScript type code (target type)
     * @param col value to convert. Has to be a value of the appropriate interop type that can be processed by JavaScript
     */
    function toJsValue(dbType: number, jsType: number, col: any): any;
    function isOracleNumber(value: any): boolean;
    function isOracleBlob(value: any): boolean;
    function isOracleDate(value: any): boolean;
    function isOracleClob(value: any): boolean;
    function isOracleNClob(value: any): boolean;
    function isOracleIntervalDayToSecond(value: any): boolean;
    function isOracleIntervalYearToMonth(value: any): boolean;
    function isOracleTimestamp(value: any): boolean;
    function isOracleTimestampTZ(value: any): boolean;

    // ******************************************************
    // mle-js-oracledb-esm/mle-js-oracledb-common/connection.d.ts
    // ******************************************************

    interface ColumnConversionTarget {
        interopType: InteropType;
        jsType: JsType;
    }
    class Connection implements IConnection {
        readonly oracleServerVersion: number;
        readonly oracleServerVersionString: string;
        readonly stmtCacheSize: number;
        constructor();
        /**
         * Throw a mle-js-oracledb error if condition is false.
         * @param condition condition to check
         * @param errorCode NJS-XXX error code
         * @param messageArg1 error string argument
         */
        private static assert;
        private static isObject;
        private static isObjectOrArray;
        private static createBaton;
        private static expectedExecuteOptions;
        private static validateFetchInfo;
        /**
         * Validates execute() options and throws an exception if one or several options are invalid.
         *
         * @param options executeMany() options
         */
        private static validateExecuteOptions;
        /**
         * Configure fetch size in MSD and throw an exception if an option is invalid.
         *
         * @param stmt MSD statement
         * @param options execute() options
         */
        private static setFetchSize;
        private static executeQuery;
        private static executeNonQuery;
        private getOracleServerVersion;
        private getOracleServerVersionString;
        /**
         * Processes the arguments provided by the caller and place them on the baton.
         */
        private static _executeProcessArgs;
        /**
         * Worker function for Connection.execute.
         */
        private static _execute;
        private static getColumnJsType;
        /**
         * Processes the arguments provided by the caller and places them on the
         * baton.
         */
        private static _executeManyProcessArgs;
        /**
         * Prepare statement and bind data to the statement.
         */
        private static _prepareAndBind;
        /**
         * Worker function for Connection.executeMany.
         */
        private static _executeMany;
        /**
         * Executes a SQL statement and returns the results.
         */
        execute(sqlText: string, bindParams?: BindParameters, options?: IExecuteOptions, ...restArgs: any[]): IExecuteReturn;
        commit(...args: any[]): void;
        /**
         * Executes a SQL statement multiple times and returns the results.
         */
        executeMany(sql: string, bindsOrNumIters: BindParameters[] | number, a3?: IExecuteManyOptions, ...restArgs: any[]): IExecuteManyReturn;
        getStatementInfo(sqlText: string, ...args: any[]): IStatementInfo;
        rollback(...args: any[]): void;
        getSodaDatabase(...args: any[]): ISodaDatabase;
    }

    // ******************************************************
    // mle-js-oracledb-esm/mle-js-oracledb-common/soda.d.ts
    // ******************************************************

    class SodaDatabase implements ISodaDatabase {
        private readonly sdb;
        static _toSodaDoc(sdb: any, value: any): any;
        constructor(msd: any);
        private static readonly createCollectionOpts;
        createCollection(collectionName: string, options?: ICreateCollectionOptions, ...args: any[]): ISodaCollection;
        createDocument(content: string | Uint8Array | Record<string, any>, options?: ICreateDocumentOptions, ...args: any[]): ISodaDocument;
        private static readonly getCollectionNameOpts;
        getCollectionNames(options?: IGetCollectionNameOptions, ...args: any[]): Array<string>;
        openCollection(collectionName: string, ...args: any[]): ISodaCollection;
    }
    class SodaCollection implements ISodaCollection {
        private readonly sdb;
        private readonly sc;
        constructor(sdb: any, sc: any);
        get name(): string;
        get metaData(): Record<string, any>;
        createIndex(indexSpec: Record<string, any>, ...args: any[]): void;
        drop(...args: any[]): IDropResult;
        private static readonly dropIndexOpts;
        dropIndex(indexName: string, options?: IDropIndexOptions, ...args: any[]): IDropIndexResult;
        find(...args: any[]): ISodaOperation;
        getDataGuide(...args: any[]): ISodaDocument;
        insertMany(docs: Array<Record<string, any> | ISodaDocument>, ...args: any[]): void;
        insertManyAndGet(docs: Array<Record<string, any> | ISodaDocument>, ...args: any[]): Array<ISodaDocument>;
        insertOne(doc: Record<string, any> | ISodaDocument, ...args: any[]): void;
        insertOneAndGet(doc: Record<string, any> | ISodaDocument, ...args: any[]): ISodaDocument;
        save(doc: ISodaDocument, ...args: any[]): void;
        saveAndGet(doc: ISodaDocument, ...args: any[]): ISodaDocument;
        truncate(...args: any[]): void;
    }
    class SodaOperation implements ISodaOperation {
        private readonly sdb;
        private readonly sc;
        private readonly soo;
        constructor(sdb: any, sc: any);
        fetchArraySize(size: number, ...args: any[]): ISodaOperation;
        filter(filter: Record<string, any>, ...args: any[]): ISodaOperation;
        key(key: string, ...args: any[]): ISodaOperation;
        keys(keys: Array<string>, ...args: any[]): ISodaOperation;
        limit(n: number, ...args: any[]): ISodaOperation;
        skip(n: number, ...args: any[]): ISodaOperation;
        version(v: string, ...args: any[]): ISodaOperation;
        count(...args: any[]): ICountResult;
        getCursor(...args: any[]): ISodaDocumentCursor;
        getDocuments(...args: any[]): Array<ISodaDocument>;
        getOne(...args: any[]): ISodaDocument;
        remove(...args: any[]): IRemoveResult;
        replaceOne(newDoc: Record<string, any> | ISodaDocument): IReplaceResult;
        replaceOneAndGet(newDoc: Record<string, any> | ISodaDocument): ISodaDocument;
    }
    class SodaDocumentCursor implements ISodaDocumentCursor {
        private readonly sdc;
        constructor(sdc: any);
        close(...args: any[]): void;
        getNext(...args: any[]): ISodaDocument;
    }
    class SodaDocument implements ISodaDocument {
        private readonly sd;
        _getSd(): any;
        static _fromContent(sdb: any, content: string | Uint8Array | Record<string, any>, options?: ICreateDocumentOptions): SodaDocument;
        private static readonly constructorOpts;
        constructor(sd: any);
        get createdOn(): string;
        get key(): string;
        get lastModified(): string;
        get mediaType(): string;
        get version(): string;
        getContent(...args: any[]): Record<string, any>;
        getContentAsUint8Array(...args: any[]): Uint8Array;
        getContentAsString(...args: any[]): string;
    }


    // ******************************************************
    // mle-js-oracledb-esm/mle-js-oracledb-common/mle-js-oracledb-common.d.ts
    // ******************************************************

    class OracleDb {
        OUT_FORMAT_ARRAY: number;
        ARRAY: number;
        OUT_FORMAT_OBJECT: number;
        OBJECT: number;
        DEFAULT: number;
        STRING: number;
        NUMBER: number;
        DATE: number;
        ORACLE_NUMBER: number;
        ORACLE_DATE: number;
        ORACLE_BLOB: number;
        ORACLE_CLOB: number;
        ORACLE_INTERVAL_DS: number;
        ORACLE_INTERVAL_YM: number;
        ORACLE_TIMESTAMP: number;
        ORACLE_TIMESTAMP_TZ: number;
        ORACLE_JSON: number;
        UINT8ARRAY: number;
        BOOLEAN: number;
        DB_TYPE_VARCHAR: number;
        DB_TYPE_NUMBER: number;
        DB_TYPE_LONG: number;
        DB_TYPE_DATE: number;
        DB_TYPE_RAW: number;
        DB_TYPE_LONG_RAW: number;
        DB_TYPE_CHAR: number;
        DB_TYPE_BINARY_FLOAT: number;
        DB_TYPE_BINARY_DOUBLE: number;
        DB_TYPE_BINARY_INTEGER: number;
        DB_TYPE_ROWID: number;
        DB_TYPE_UROWID: number;
        DB_TYPE_BOOLEAN: number;
        DB_TYPE_CLOB: number;
        DB_TYPE_BLOB: number;
        DB_TYPE_TIMESTAMP: number;
        DB_TYPE_TIMESTAMP_TZ: number;
        DB_TYPE_TIMESTAMP_LTZ: number;
        DB_TYPE_NVARCHAR: number;
        DB_TYPE_NCHAR: number;
        DB_TYPE_NCLOB: number;
        DB_TYPE_JSON: number;
        DB_TYPE_INTERVAL_YM: number;
        DB_TYPE_INTERVAL_DS: number;
        BIND_IN: number;
        BIND_INOUT: number;
        BIND_OUT: number;
        STMT_TYPE_UNKNOWN: number;
        STMT_TYPE_SELECT: number;
        STMT_TYPE_UPDATE: number;
        STMT_TYPE_DELETE: number;
        STMT_TYPE_INSERT: number;
        STMT_TYPE_CREATE: number;
        STMT_TYPE_DROP: number;
        STMT_TYPE_ALTER: number;
        STMT_TYPE_BEGIN: number;
        STMT_TYPE_DECLARE: number;
        STMT_TYPE_CALL: number;
        STMT_TYPE_EXPLAIN_PLAN: number;
        STMT_TYPE_MERGE: number;
        STMT_TYPE_ROLLBACK: number;
        STMT_TYPE_COMMIT: number;
        SODA_COLL_MAP_MODE: number;
        parameters: Parameters;
        SodaCollection: typeof SodaCollection;
        SodaDatabase: typeof SodaDatabase;
        SodaDocument: typeof SodaDocument;
        SodaDocumentCursor: typeof SodaDocumentCursor;
        SodaOperation: typeof SodaOperation;
        OracleDb: typeof OracleDb;
        get outFormat(): OutFormatType;
        set outFormat(value: OutFormatType);
        get extendedMetaData(): boolean;
        set extendedMetaData(value: boolean);
        get fetchArraySize(): number;
        set fetchArraySize(value: number);
        get fetchAsPlsqlWrapper(): JsType[];
        set fetchAsPlsqlWrapper(value: JsType[]);
        get fetchAsString(): JsType[];
        set fetchAsString(value: JsType[]);
        get maxRows(): number;
        set maxRows(value: number);
        /**
         * Returns the default connection object for executing SQL queries in the Oracle
         * Database using mle-js-oracledb. Note that since in MLE, JavaScript is
         * executed directly in the database, there is no need to establish a specific
         * connection which is why the default connection object should be used.
         *
         * @returns default connection object for executing SQL queries with mle-js-oracledb.
         */
        defaultConnection(): IConnection;
    }


    // ******************************************************
    // mle-js-oracledb-esm/mle-js-oracledb-esm/index.d.ts
    // ******************************************************

    /**
     * Returns the default connection object for executing SQL queries in the Oracle
     * Database using mle-js-oracledb. Note that since in MLE, JavaScript is
     * executed directly in the database, there is no need to establish a specific
     * connection which is why the default connection object should be used.
     *
     * @returns default connection object for executing SQL queries with mle-js-oracledb.
     */
    export function defaultConnection(): IConnection;
    const _default: OracleDb;
    export default _default;
}

declare module "mle-js-bindings" {
    export enum JSTypes {
        /** Type String */
        STRING = 0,
        /** Type Number */
        NUMBER = 1,
        /** Type Date */
        DATE = 2,
        /** Type Boolean */
        BOOLEAN = 3,
        /** Type Object */
        OBJECT = 4,
        /** UINT8ARRAY */
        UINT8ARRAY = 5,
        /** Type OracleNumber */
        ORACLE_NUMBER = 6,
        /** Type OracleDate */
        ORACLE_DATE = 7,
        /** Type OracleTimeStamp */
        ORACLE_TIMESTAMP = 8,
        /** Type OracleTimeStampTZ */
        ORACLE_TIMESTAMP_TZ = 9,
        /** Type OracleIntervalYearToDay */
        ORACLE_INTERVAL_YM = 10,
        /** Type OracleIntervalDayToSecond */
        ORACLE_INTERVAL_DS = 11,
        /** Type OracleCLOB */
        ORACLE_CLOB = 12,
        /** Type OracleBLOB */
        ORACLE_BLOB = 13,
        /** Type OracleJSON */
        ORACLE_JSON = 14
    }
    /**
     * Import a value exported from PL/SQL into the current context
     *
     * Import the value, identified by the given name, that was previously exported
     * from PL/SQL.
     * The desired result type can be specified explicitly with the jstype parameter.
     * If no result type is specified, the default PL/SQL-to-JavaScript mapping is
     * used to determine the result type implicitly.
     *
     * @param name Name of the property to be retrieved. Cannot be null, undefined or empty.
     * @param jstype JavaScript type of the result.
     * @return a JavaScript value of the desired type. Returns undefined if the
     * property does not exist.
     * @throws an exception (Invalid property name) if name is null, undefined or empty.
     */
    export function importValue(name: string, jstype?: JSTypes): any;
    /**
     * Export a value to PL/SQL
     *
     * Export the given value to PL/SQL.
     * The exported value can be imported into PL/SQL using the given name.
     *
     * @param name Name of the property to be exported. Cannot be null, undefined or empty.
     * @param value Value of the property to be exported.
     * @throws an exception (Invalid property name) if name is null, undefined or empty.
     */
    export function exportValue(name: string, value: any): void;
}

declare module "mle-jsplsqltypes" {
    /**
     * JavaScript API for Oracle type TIMESTAMP.
     */
    export const OracleTimestamp: IOracleTimestamp;
    /**
     * JavaScript API for Oracle type TIMESTAMP WITH TIMEZONE.
     */
    export const OracleTimestampTZ: IOracleTimestampTZ;
    /**
     * JavaScript API for Oracle type INTERVAL DAY TO SECOND.
     */
    export const OracleIntervalDayToSecond: IOracleIntervalDayToSecond;
    /**
     * JavaScript API for Oracle type INTERVAL YEAR TO MONTH.
     */
    export const OracleIntervalYearToMonth: IOracleIntervalYearToMonth;
    /**
     * JavaScript API for Oracle type CLOB.
     */
    export const OracleClob: IOracleClob;
    /**
     * JavaScript API for Oracle type DATE.
     */
    export const OracleDate: IOracleDate;
    /**
     * JavaScript API for Oracle type NUMBER.
     */
    export class OracleNumber {
        private impl;
        /**
         * Construct an OracleNumber from a JavaScript number or a string.
         *
         * @param n
         */
        constructor(n: number | string);
        /**
         * Construct an OracleNumber from a JavaScript number.
         * @param from - number used to create the OracleNumber
         * @returns a new OracleNumber object
         */
        static fromNumber(n: number): OracleNumber;
        /**
         * Parse a string into a OracleNumber, with an optional number format
         * and NLS parameters. If no format string and NLS parameters are provided,
         * the default values for the session are used.
         * @param s the number string
         * @param format optional number format
         * @param nlsParam optional NLS parameters
         * @returns OracleNumber value from parsing the input string
         */
        static fromString(s: string, format?: string, nlsParam?: string): OracleNumber;
        /**
         * The pi constant.
         */
        static pi: OracleNumber;
        /**
         * The zero constant.
         */
        static zero: OracleNumber;
        /**
         * The e constant.
         */
        static e: OracleNumber;
        /**
         * The ln10 constant.
         */
        static ln10: OracleNumber;
        /**
         * Computes the absolute value of the Oracle number
         * @returns the absolute value of the Oracle number
         */
        abs(): OracleNumber;
        /**
         * Adds a Oracle number to another Oracle number
         * @param other - the other Oracle number
         * @returns returns the sum of the two Oracle NUMBERs
         */
        add(other: OracleNumber): OracleNumber;
        /**
         * Subtracts an Oracle number from the Oracle number and returns the resulting Oracle number
         * @param other - the other number to be subtracted from the OracleNumber
         * @returns the result of the subtraction as OracleNumber
         */
        sub(other: OracleNumber): OracleNumber;
        /**
         * Multiplies the Oracle number with another Oracle number
         * @param other - the other Oracle number
         * @returns the result of the multiplication as Oracle number
         */
        mul(other: OracleNumber): OracleNumber;
        /**
         * Divides two Oracle numbers
         * @param other - divisor
         * @returns the result of the division as Oracle number
         */
        div(other: OracleNumber): OracleNumber;
        /**
         * Computes the modulus of two Oracle numbers.
         * @param other - the other Oracle number
         * @returns this number modulo the other number
         */
        mod(other: OracleNumber): OracleNumber;
        /**
         * Computes the sine of the Oracle number
         * @returns the sine of the Oracle number
         */
        sin(): OracleNumber;
        /**
         * Computes the cosine of the Oracle number
         * @returns the cosine
         */
        cos(): OracleNumber;
        /**
         * Computes the tangent of the Oracle number
         * @returns the tangent of the Oracle number
         */
        tan(): OracleNumber;
        /**
         * Computes the arc sine in radians of the Oracle number
         * @returns the arc sine in radians of the Oracle number
         */
        arcSin(): OracleNumber;
        /**
         * Computes the arc cosine in radians of the Oracle number
         * @returns the arc cosine in radians of the Oracle number
         */
        arcCos(): OracleNumber;
        /**
         * Computes the arc tangent in radians of the Oracle number
         * @returns the arc tangent in radians of the Oracle number
         */
        arcTan(): OracleNumber;
        /**
         * Computes the arc tangent of two Oracle numbers
         * @param other Oracle number used for the calculation
         * @returns the arc tangent of two Oracle numbers
         */
        arcTan2(other: OracleNumber): OracleNumber;
        /**
         * Computes the hyperbolic sine of the Oracle number
         * @returns the hyperbolic sine of the Oracle number
         */
        hypSin(): OracleNumber;
        /**
         * Computes the hyperbolic cosine of the Oracle number
         * @returns the hyperbolic cosine of the Oracle number
         */
        hypCos(): OracleNumber;
        /**
         * Computes the hyperbolic tangent of the Oracle number
         * @returns the hyperbolic tangent of the Oracle number
         */
        hypTan(): OracleNumber;
        /**
         * Compares two Oracle numbers
         * @returns -1 if this < other
         *           0 if this = other
         *           1 if this > other
         */
        compare(other: OracleNumber): number;
        /**
         * Checks if the Oracle number is equal to another Oracle number
         * @param other - the other Oracle number
         * @returns true if both Oracle numbers are equal otherwise false
         */
        equals(other: OracleNumber): boolean;
        /**
         * Raises e to the power of this Oracle number
         * @returns the result of the exponentiation as Oracle number
         */
        exp(): OracleNumber;
        /**
         * Raises this Oracle number to the given exponent
         * @returns the result of the exponentiation
         */
        power(exp: OracleNumber | number): OracleNumber;
        /**
         * Computes the natural logarithm of the Oracle number
         * @returns the natural logarithm of the Oracle number
         */
        ln(): OracleNumber;
        /**
         * Computes the logarithm to an arbitrary base
         * @param base the base of the logarithm
         * @returns the logarithm to an arbitrary base
         */
        log(base: OracleNumber | number): OracleNumber;
        /**
         * Computes the square root of the Oracle number
         * @returns the square root of the Oracle number
         */
        sqrt(): OracleNumber;
        /**
         * Tests if the number is an integer
         * @returns true if the Oracle number is an integer otherwise false
         */
        isInt(): boolean;
        /**
         * Tests if the number is zero
         * @returns true if the Oracle number is zero otherwise false
         */
        isZero(): boolean;
        /**
         * Returns -1 if the sign of the number is negative, 0 if the number is 0, and > 0 if the number is positive.
         * @returns a number indicating the sign of the Oracle number
         */
        sign(): number;
        /**
         * Negates the number
         * @returns the negated Oracle number
         */
        neg(): OracleNumber;
        /**
         * Computes the ceiling of the Oracle number
         * @returns the ceiling of the Oracle number
         */
        ceil(): OracleNumber;
        /**
         * Computes the floor of the number
         * @returns the floor value of the Oracle number
         */
        floor(): OracleNumber;
        /**
         * Rounds the number to a specified number of digits
         * @param ndigs - the number of digits
         * @returns the rounded Oracle number
         */
        roundDigits(ndigs: OracleNumber | number): OracleNumber;
        /**
         * Rounds the Oracle number to a specified number of decimal places
         * @param decplace - the decimal place
         * @returns the rounded Oracle number
         */
        roundPlaces(decplace: OracleNumber | number): OracleNumber;
        /**
         * Shift the number by the specified number of decimal places
         * @param digits - number of decimal places to shift. Can be negative. Positive values shift the decimal place to the right and negative values to the left. For example, if NUMBER corresponds to 1234.5 and digits == -1, the new NUMBER object will correspond to 123.45.
         * @returns an Oracle number containing the shifted result
         */
        shift(digits: OracleNumber | number): OracleNumber;
        /**
         * Truncate the number to the specified number of decimal places.
         * @param decplace - Number of decimal digits to the right of the decimal point at which to truncate. Negative values are allowed.
         * @returns a truncated Oracle number
         */
        trunc(decplace: OracleNumber | number): OracleNumber;
        /**
         * Scale the digits to the left and right of the decimal point.
         * @param left maximum number of decimal digits to the left of the decimal point. it will not effect the number, but throw an exception if this value is exceeded.
         * @param right maximum number of decimal digits to the right of the decimal point. The number is rounded at this point. Negative values are allowed.
         * @returns a new scaled Oracle number scaled according to the arguments
         * @throws throws an exception if the number of left-hand-side digits is exceeded
         */
        scale(left: number, right: number): OracleNumber;
        /**
         * Converts an Oracle NUMBER to type number
         * @returns the converted number
         */
        toNumber(): number;
        /**
         * Converts an Oracle NUMBER to a string, with an optional number format
         * and NLS parameters. If no format string and NLS parameters are provided,
         * the default values for the session are used.
         * @param format optional number format
         * @param nlsParam optional NLS parameters
         * @returns the converted string
         */
        toString(format?: string, nlsParam?: string): string;
        isOracleNumber(): boolean;
        asPackedDecimalNumber(): any;
    }
    /**
     * JavaScript API for Oracle type INTERVAL DAY TO SECOND.
     */
    class IOracleIntervalDayToSecond {
        /**
         * Constructor for IOracleIntervalDayToSecond using day, hour, minute, and
         * second. All arguments must be integral numbers.
         *
         * @param dy - number of days
         * @param hr - number of hours
         * @param mm - number of minutes
         * @param ss - number of seconds
         * @param fs - fractional seconds
         * @returns new IOracleIntervalDayToSecond
         */
        constructor(dy: number, hr: number, mm: number, ss: number, fs: number);
        /**
         * Parse an interval string into IOracleIntervalDayToSecond. fromString
         * accepts the same input formats as the Oracle SQL function TO_DSINTERVAL:
    
         * - SQL interval format compatible with the SQL standard (ISO/IEC 9075)
         * - ISO duration format compatible with the ISO 8601:2004 standard
         *
         * @param interval string to parse.
         * @returns new IOracleIntervalDayToSecond
         */
        static fromString(interval: string): IOracleIntervalDayToSecond;
        /**
         * Returns an IOracleIntervalDayToSecond for a given timezone offset or
         * timezone name. The input string must be of the form [+/-]TZH:TZM or
         * 'TZR'.
         *
         * @param tz - timezone offset or timezone name as string
         * @returns new IOracleIntervalDayToSecond that contains the current absolute offset
         */
        static fromTimeZone(tz: string): IOracleIntervalDayToSecond;
        /**
         * Returns an IOracleIntervalDayToSecond for the given number of days. If the
         * number of days contains a fractional part, the number of hours, minutes
         * and seconds will be set accordingly.
         *
         * @param days - number of days
         * @returns new IOracleIntervalDayToSecond for the given number of days.
         */
        static fromNumberOfDays(days: number | OracleNumber): IOracleIntervalDayToSecond;
        /**
         * Compares two intervals
         *
         * @param i1 - first interval to use for the comparison
         * @param i2 - second interval to use for the comparison
         * @returns -1 if i1 < i2
         *           0 if i1 = i2
         *           1 if i1 > i2
         */
        static compare(i1: IOracleIntervalDayToSecond, i2: IOracleIntervalDayToSecond): any;
        /**
         * Returns the number of days from the interval
         *
         * @returns the number of days from the interval
         */
        getDays(): number;
        /**
         * Returns the number of hours from the interval
         *
         * @returns the number of hours from the interval
         */
        getHours(): number;
        /**
         * Returns the number of minutes from the interval
         *
         * @returns the number of minutes from the interval
         */
        getMinutes(): number;
        /**
         * Returns the number of seconds including fractional seconds from the
         * interval
         *
         * @returns the number of seconds including fractional seconds from the
         *          interval
         */
        getSeconds(): number;
        /**
         * Adds the interval to another interval and returns the resulting interval
         *
         * @param summand - the other interval
         * @returns the resulting interval as IOracleIntervalDayToSecond
         */
        add(summand: IOracleIntervalDayToSecond): IOracleIntervalDayToSecond;
        /**
         * Subtracts another interval from the interval and returns the resulting
         * interval
         *
         * @param subtrahend - the other interval
         * @returns the resulting interval as IOracleIntervalDayToSecond
         */
        subtract(subtrahend: IOracleIntervalDayToSecond): IOracleIntervalDayToSecond;
        /**
         * Divides the interval by an Oracle number or JavaScript number and returns
         * the resulting interval
         *
         * @param divisor - the divisor used in the division
         * @returns the resulting interval
         */
        divide(divisor: OracleNumber | number): IOracleIntervalDayToSecond;
        /**
         * Multiplies the interval with an Oracle number or a JavaScript number and
         * returns the resulting interval
         *
         * @param factor - the factor used in the multiplication
         * @returns the resulting interval
         */
        multiply(factor: OracleNumber | number): IOracleIntervalDayToSecond;
        /**
         * Produces a string representing the interval. The string is formatted with
         * 2 digits for the number of days and 6 digits for the fractional seconds
         * of the interval.
         *
         * @returns a string representing the interval
         */
        toString(): string;
        /**
         * Produces a string representing the interval. The string is formatted
         * according to the specified precision for days and fractional seconds.
         *
         * @param daysPrecision - The number of digits used to represent days in the interval string.
         * @param fractionalSecondsPrecision - Fractional second precision of the interval string (the number of digits used to represent the fractional seconds).
         * @returns a string representing the interval
         */
        toStringWithPrecision(daysPrecision: number, fractionalSecondsPrecision: number): string;
    }
    /**
     * JavaScript API for Oracle type INTERVAL YEAR TO MONTH.
     */
    class IOracleIntervalYearToMonth {
        /**
         * Constructor for IOracleIntervalYearToMonth using year and month. All
         * arguments must be integral numbers.
         *
         * @param yr year value
         * @param mnth month value
         * @returns new IOracleIntervalYearToMonth
         */
        constructor(yr: number, mnth: number);
        /**
         * Parse an interval string into IOracleIntervalYearToMonth. fromString
         * accepts the same input formats as the Oracle SQL function TO_YMINTERVAL:
         *
         * - SQL interval format compatible with the SQL standard (ISO/IEC 9075)
         * - ISO duration format compatible with the ISO 8601:2004 standard
         *
         * @param interval string to parse
         * @returns new IOracleIntervalYearToMonth
         */
        static fromString(interval: string): IOracleIntervalYearToMonth;
        /**
         * Returns an IOracleIntervalYearToMonth for the given number of years. If
         * the number of years contains a fractional part, the number of months will
         * be set accordingly.
         *
         * @param days - number of months
         * @returns new IOracleIntervalYearToMonth for the given number of years.
         */
        static fromNumberOfYears(years: number | OracleNumber): IOracleIntervalYearToMonth;
        /**
         * Compares two intervals
         *
         * @param i1 - first interval to use for the comparison
         * @param i2 - second interval to use for the comparison
         * @returns -1 if i1 < i2
         *           0 if i1 = i2
         *           1 if i1 > i2
         */
        static compare(i1: IOracleIntervalYearToMonth, i2: IOracleIntervalYearToMonth): number;
        /**
         * Returns the number of years from the interval
         *
         * @returns the number of years from the interval
         */
        getYears(): number;
        /**
         * Returns the total number of months from the interval
         *
         * @returns the total number of months from the interval
         */
        getMonths(): number;
        /**
         * Adds the interval to another interval and returns the resulting interval
         *
         * @param other - the other interval
         * @returns the resulting interval as IOracleIntervalYearToMonth
         */
        add(other: IOracleIntervalYearToMonth): IOracleIntervalYearToMonth;
        /**
         * Subtracts another interval from the interval and returns the resulting
         * interval
         *
         * @param other - the other interval
         * @returns the resulting interval as IOracleIntervalYearToMonth
         */
        subtract(other: IOracleIntervalYearToMonth): IOracleIntervalYearToMonth;
        /**
         * Divides the interval by an OracleNumber or JavaScript number and returns
         * the resulting interval
         *
         * @param dividend - the dividend used in the division
         * @returns the resulting interval
         */
        divide(dividend: OracleNumber | number): IOracleIntervalYearToMonth;
        /**
         * Multiplies the interval with an OracleNumber or a JavaScript number and
         * returns the resulting interval
         *
         * @param nfactor - the factor used in the multiplication
         * @returns the resulting interval
         */
        multiply(nfactor: OracleNumber | number): IOracleIntervalYearToMonth;
        /**
         * Given an interval, produces a string representing the interval. The
         * string is formatted with at least 2 digits for the number of years.
         *
         * @returns a string representing the year-to-month interval
         */
        toString(): string;
        /**
         * Produces a string representing the interval. The string is formatted
         * according to the specified precision for days and fractional seconds.
         *
         * @param yearsPrecision - The number of digits used to represent years in the interval string.
         * @returns a string representing the interval
         */
        toStringWithPrecision(yearsPrecision: number): string;
    }
    /**
     * Optional arguments for the compare method of OracleBlob and OracleClob.
     */
    interface CompareOptionalArguments {
        /**
         * Number of bytes/characters to compare.
         */
        amount?: number;
        /**
          * Offset (bytes/characters) into the first LOB. Defaults to 1.
          */
        offset1?: number;
        /**
         * Offset (bytes/characters) into the second LOB. Defaults to 1.
         */
        offset2?: number;
    }
    /**
     * JavaScript API for Oracle type Binary Large Object (BLOB).
     */
    export class OracleBlob {
        private delegate;
        /**
         * Constant for read-only mode.
         */
        static LOB_READONLY: number;
        /**
         * Constant for read/write mode.
         */
        static LOB_READWRITE: number;
        private static checkArgPresent;
        private static checkNoArgs;
        /**
         * This constructor creates a temporary BLOB and its corresponding index in
         * your default temporary tablespace. The temporary BLOB is created with
         * SESSION duration.
         *
         * @param cache Specified if LOB should be read into buffer cache or not.
         */
        static createTemporary(cache: boolean): OracleBlob;
        constructor(delegate: any);
        /**
         * This method compares two entire LOBs or parts of two LOBs.
         *
         * @param lob1 First target for comparison.
         * @param lob2 Second target for comparison.
         * @param optArgs Optional arguments that specify amount and offsets.
         * @returns 0 if the compared portions are equal, non-zero if not
         * @throws Error if offset1 or offset2 is not a valid LOB offset value.
         */
        static compare(lob1: OracleBlob, lob2: OracleBlob, optArgs?: CompareOptionalArguments): number;
        /**
         * When creating the table, you can specify the chunking factor, a multiple
         * of tablespace blocks in bytes. This corresponds to the chunk size used
         * by the LOB data layer when accessing or modifying the LOB value. Part of
         * the chunk is used to store system-related information, and the rest
         * stores the LOB value. This method returns the amount of space used in
         * the LOB chunk to store the LOB value.
         */
        getChunkSize(...args: any[]): number;
        /**
         * This method gets the length of the specified LOB. The length in bytes
         * is returned.
         */
        length(...args: any[]): number;
        /**
         * This method checks to see if the LOB was already opened using the
         * input locator.
         */
        isOpen(...args: any[]): boolean;
        /**
         * This method determines whether a LOB instance is temporary.
         */
        isTemporary(...args: any[]): boolean;
        /**
         * This method opens a BLOB in the indicated mode. Valid modes include
         * read-only, and read/write.
         *
         * @param mode
         */
        open(mode: number): void;
        /**
         * This method closes a previously opened BLOB.
         */
        close(...args: any[]): void;
        /**
         * This method frees the temporary BLOB in the default temporary
         * tablespace.
         */
        freeTemporary(...args: any[]): void;
        /**
         * This method reads a piece of a LOB, and returns the specified amount
         * into the buffer parameter, starting from an absolute offset from the
         * beginning of the LOB. If the input offset points past the End of LOB, a
         * NO_DATA_FOUND exception is raised.
         *
         * @param amount Number of bytes to read.
         * @param offset Offset in bytes from the start of the LOB (origin: 1).
         * @returns a Uint8Array that contains the bytes actually read.
         */
        read(amount: number, offset: number): Uint8Array;
        /**
         * This method writes data into an internal LOB, starting from an absolute
         * offset from the beginning of the LOB. The data is written from the buffer
         * parameter. WRITE replaces (overwrites) any data that already exists in
         * the LOB at the offset.
         *
         * @param offset Offset in bytes from the start of the LOB (origin: 1) for the write operation.
         * @param buffer Data to write.
         */
        write(offset: number, buffer: Uint8Array): void;
        getDelegate(): any;
        isOracleBlob(): boolean;
    }
    /**
     * JavaScript API for Oracle type Character Large Object (CLOB).
     */
    class IOracleClob {
        /**
         * Constant for read-only mode.
         */
        static LOB_READONLY: number;
        /**
         * Constant for read/write mode.
         */
        static LOB_READWRITE: number;
        /**
         * createTemporary creates a temporary CLOB and its corresponding index in
         * your default temporary tablespace. The temporary BLOB is created with
         * SESSION duration.
         *
         * @param cache Specifies if LOB should be read into buffer cache or not.
         */
        static createTemporary(cache: boolean, isNClob?: boolean): any;
        /**
         * This method compares two entire LOBs or parts of two LOBs.
         *
         * @param lob1 First target for comparison.
         * @param lob2 Second target for comparison.
         * @param optArgs Optional arguments that specify amount and offsets.
         * @returns 0 if the compared portions are equal, non-zero if not
         * @throws Error if offset1 or offset2 is not a valid LOB offset value.
         */
        static compare(lob1: IOracleClob, lob2: IOracleClob, optArgs?: CompareOptionalArguments): number;
        /**
         * When creating the table, you can specify the chunking factor, a multiple
         * of tablespace blocks in characters. This corresponds to the chunk size used
         * by the LOB data layer when accessing or modifying the LOB value. Part of
         * the chunk is used to store system-related information, and the rest
         * stores the LOB value. This method returns the amount of space used in
         * the LOB chunk to store the LOB value.
         */
        getChunkSize(): number;
        /**
         * This method gets the length of the specified LOB. The length in
         * characters is returned.
         */
        length(): number;
        /**
         * This method checks to see if the LOB was already opened using the
         * input locator.
         */
        isOpen(): boolean;
        /**
         * This method determines whether a LOB instance is temporary.
         */
        isTemporary(): boolean;
        /**
         * This method opens a BLOB in the indicated mode. Valid modes include
         * read-only, and read/write.
         *
         * @param mode
         */
        open(mode: number): any;
        /**
         * This method closes a previously opened BLOB.
         */
        close(): any;
        /**
         * This method frees the temporary BLOB in the default temporary
         * tablespace.
         */
        freeTemporary(): any;
        /**
         * This method reads a piece of a LOB, and returns the specified amount
         * into the buffer parameter, starting from an absolute offset from the
         * beginning of the LOB. If the input offset points past the End of LOB, a
         * NO_DATA_FOUND exception is raised.
         *
         * @param amount Number of characters to read.
         * @param offset Offset in characters from the start of the LOB (origin: 1).
         * @returns a string that contains the characters actually read.
         */
        read(amount: number, offset: number): string;
        /**
         * This method writes data into an internal LOB, starting from an absolute
         * offset from the beginning of the LOB. The data is written from the buffer
         * parameter. WRITE replaces (overwrites) any data that already exists in
         * the LOB at the offset.
         *
         * @param offset Offset in characters from the start of the LOB (origin: 1) for the write operation.
         * @param buffer Data to write.
         */
        write(offset: number, buffer: string): any;
    }
    /**
     * JavaScript API for Oracle type TIMESTAMP WITH TIME ZONE.
     */
    class IOracleTimestampTZ {
        /**
         * Parse a string into an IOracleTimestampTZ. An optional datetime format
         * model and an optional NLS parameter string can be provided. If no format
         * model is specified, the string must be in the default format for the
         * Oracle TIMESTAMP WITH TIME ZONE data type, which is determined by the
         * NLS_TIMESTAMP_TZ_FORMAT parameter.
         *
         * If a NLS parameter string is specified, it has the effect of temporarily
         * altering the session's NLS settings for the duration of the call to this
         * method. The NLS parameter string may set the NLS_DATE_LANGUAGE
         * parameter.
         *
         * If a datetime format model is specified, it overrides the datetime format
         * determined by NLS settings (i.e. the setting of the NLS_TIMESTAMP_TZ
         * parameter).
    
         * The function returns an error if the specified date lies in a DST
         * transition period and if the Oracle Database parameter
         * ERROR_ON_OVERLAP_TIME is true for the current session.
         *
         * @param date date represented as a string
         * @param format optional datetime format model
         * @param nlsParam optional NLS parameter string. Must have the same format
         * as the NLS argument to the TO_TIMESTAMP_TZ SQL function.
         * @returns a IOracleTimestampTZ object
         */
        static fromString(date: string, format?: string, nlsParam?: string): any;
        /**
         * Convert to a OracleTimestamp that contains the local date/time of the
         * IOracleTimestampTZ.
         */
        asLocalDateTime(): IOracleTimestamp;
        /**
         * Convert this IOracleTimestampTZ to a IOracleTimestampTZ in the specified
         * time zone. The time zone string has to contain either a time zone offset
         * of the form '(+|-)HH:MM' or a time zone region name.
         *
         * The function returns an error if the date lies in a DST transition period
         * in the specified time zone and if the Oracle Database parameter
         * ERROR_ON_OVERLAP_TIME is true for the current session.
         *
         * @param timezone The time zone string.
         */
        atTimeZone(timezone: string): IOracleTimestampTZ;
        /**
         * Compares two datetime values.
         *
         * @param other - second timestamp to be compared
         * @returns -1 if date1 < date2
         *           0 if date1 = date2
         *           1 if date1 > date2
         */
        static compare(date1: IOracleTimestampTZ, date2: IOracleTimestampTZ): number;
        /**
         * Gets the year component in the Gregorian calendar.
         *
         * @returns year component as number
         */
        getYear(): number;
        /**
         * Gets the month component in the Gregorian calendar.
         *
         * @returns month component as number
         */
        getMonth(): number;
        /**
         * Gets the day component in the Gregorian calendar.
         *
         * @returns day component as number
         */
        getDay(): number;
        /**
         * Gets the hour component.
         *
         * @returns hour component as number
         */
        getHour(): number;
        /**
         * Gets the minute component.
         *
         * @returns minute component as number
         */
        getMinute(): number;
        /**
         * Gets the second component.
         *
         * @returns second component as number
         */
        getSecond(): number;
        /**
         * Gets the fractional second component.
         *
         * @returns fractional second component as number
         */
        getFractionalSecond(): number;
        /**
         * Gets the time zone of a IOracleTimestampTZ.
         *
         * @returns the time zone name as string
         */
        getTimeZone(): string;
        /**
         * Adds an interval to the timestamp to obtain a new timestamp
         * @param interval - the interval used to obtain the new timestamp
         * @returns resulting timestamp
         */
        addInterval(interval: IOracleIntervalDayToSecond | IOracleIntervalYearToMonth): IOracleTimestampTZ;
        /**
         * Subtracts an interval from the timestamp to obtain a new timestamp
         * @param interval - the interval used to obtain the new timestamp
         * @returns resulting timestamp
         */
        subtractInterval(interval: IOracleIntervalDayToSecond | IOracleIntervalYearToMonth): IOracleTimestampTZ;
        /**
         * Subtracts another timestamp from the timestamp to obtain a year-to-month interval
         * @param other - the other timestamp used as subtrahend
         * @returns year-to-month interval defined by the two timestamps
         */
        subtractIntoYearToMonth(other: IOracleTimestampTZ): IOracleIntervalYearToMonth;
        /**
         * Subtracts another timestamp from the timestamp to obtain a day-to-second
         * interval.
         *
         * @param other - the other timestamp used as subtrahend
         * @returns year-to-month interval defined by the two timestamps
         */
        subtractIntoDayToSecond(other: IOracleTimestampTZ): IOracleIntervalDayToSecond;
        /**
         * Converts the given datetime to a string in a given format. An optional
         * datetime format model and an optional NLS parameter string can be
         * provided. If no format model and/or no NLS parameter string is provided,
         * the respective default values for the session are used.
         *
         * If a NLS parameter string is specified, it has the effect of temporarily
         * altering the session's NLS settings for the duration of the call to this
         * method. The NLS parameter string may set the NLS_DATE_LANGUAGE
         * parameter.
         *
         * If a datetime format model is specified, it overrides the datetime format
         * determined by NLS settings.
         *
         * @param format datetime format model
         * @param nlsParam optional NLS parameter string
         * @returns a string of the timestamp
         */
        toString(format?: string, nlsParam?: string): string;
    }
    /**
     * JavaScript API for Oracle type TIMESTAMP.
     */
    class IOracleTimestamp {
        /**
         * Parse a string into an IOracleTimestamp. An optional datetime format model
         * and an optional NLS parameter string can be provided. If no format model
         * is specified, the string must be in the default format for the Oracle
         * TIMESTAMP data type, which is determined by the NLS_TIMESTAMP_FORMAT
         * parameter.
         *
         * If a NLS parameter string is specified, it has the effect of temporarily
         * altering the session's NLS settings for the duration of the call to this
         * method. The NLS parameter string may set the NLS_DATE_LANGUAGE
         * parameter.
         *
         * If a datetime format model is specified, it overrides the datetime format
         * determined by NLS settings (i.e. the setting of the NLS_TIMESTAMP FORMAT
         * parameter).
         *
         * @param date date represented as a string
         * @param format optional datetime format model
         * @param nlsParam optional NLS parameter string. Must have the same format
         * as the NLS argument to the TO_TIMESTAMP SQL function.
         * @returns a IOracleTimestamp object
         */
        static fromString(date: string, format?: string, nlsParam?: string): any;
        /**
         * Convert to a IOracleTimestampTZ using the session time zone.
         */
        atSessionTimeZone(): IOracleTimestampTZ;
        /**
         * Convert to a IOracleTimestampTZ in the specified time zone. The time zone
         * string has to contain either a time zone offset of the form '(+|-)HH:MM'
         * or a time zone region name.
         *
         * @param timezone The time zone string.
         */
        atTimeZone(timezone: string): IOracleTimestampTZ;
        /**
         * Compares two IOracleTimestamp values.
         *
         * @param other - second timestamp to be compared
         * @returns -1 if date1 < date2
         *           0 if date1 = date2
         *           1 if date1 > date2
         */
        static compare(date1: IOracleTimestamp, date2: IOracleTimestamp): number;
        /**
         * Gets the year component in the Gregorian calendar.
         *
         * @returns year component as number
         */
        getYear(): number;
        /**
         * Gets the month component in the Gregorian calendar.
         *
         * @returns month component as number
         */
        getMonth(): number;
        /**
         * Gets the day component in the Gregorian calendar.
         *
         * @returns day component as number
         */
        getDay(): number;
        /**
         * Gets the hour component.
         *
         * @returns hour component as number
         */
        getHour(): number;
        /**
         * Gets the minute component.
         *
         * @returns minute component as number
         */
        getMinute(): number;
        /**
         * Gets the second component, including the fractional part.
         *
         * @returns second component as number
         */
        getSecond(): number;
        /**
         * Adds an interval to the timestamp to obtain a new timestamp
         * @param interval - the interval used to obtain the new timestamp
         * @returns resulting timestamp
         */
        addInterval(interval: IOracleIntervalDayToSecond | IOracleIntervalYearToMonth): IOracleTimestamp;
        /**
         * Subtracts an interval from the timestamp to obtain a new timestamp
         * @param interval - the interval used to obtain the new timestamp
         * @returns resulting timestamp
         */
        subtractInterval(interval: IOracleIntervalDayToSecond | IOracleIntervalYearToMonth): IOracleTimestamp;
        /**
         * Subtracts another timestamp from the timestamp to obtain a year-to-month interval
         * @param other - the other timestamp used as subtrahend
         * @returns year-to-month interval defined by the two timestamps
         */
        subtractIntoYearToMonth(other: IOracleTimestamp): IOracleIntervalYearToMonth;
        /**
         * Subtracts another timestamp from the timestamp to obtain a day-to-second
         * interval.
         *
         * @param other - the other timestamp used as subtrahend
         * @returns year-to-month interval defined by the two timestamps
         */
        subtractIntoDayToSecond(other: IOracleTimestamp): IOracleIntervalDayToSecond;
        /**
         * Converts the given datetime to a string in a certain format. An optional
         * datetime format model and an optional NLS parameter string can be
         * provided. If no format model and/or no NLS parameter string is provided,
         * the respective default values for the session are used.
         *
         * If a NLS parameter string is specified, it has the effect of temporarily
         * altering the session's NLS settings for the duration of the call to this
         * method. The NLS parameter string may set the NLS_DATE_LANGUAGE
         * parameter.
         *
         * If a datetime format model is specified, it overrides the datetime format
         * determined by NLS settings.
         *
         * @param format datetime format model
         * @param nlsParam optional NLS parameter string
         * @returns a string of the timestamp
         */
        toString(format?: string, nlsParam?: string): string;
    }
    /**
     * JavaScript API for Oracle type DATE.
     */
    class IOracleDate {
        /**
         * Construct a new IOracleDate object from date and time components
         * (Gregorian calendar).
         *
         * @param year year component (-4712 <= y <= 9999)
         * @param month month component (1 <= m <= 12)
         * @param day day component (1 <= d <= 31)
         * @param hour hour component (0 <= h <= 23)
         * @param minute minute component (0 <= m <= 59)
         * @param second second component (0 <= s <= 59)
         */
        constructor(year: number, month: number, day: number, hour: number, minute: number, second: number);
        /**
         * Construct a new IOracleDate object from date components (Gregorian
         * calendar).
         *
         * @param year year component (-4712 <= y <= 9999)
         * @param month month component (1 <= m <= 12)
         * @param day day component (1 <= d <= 31)
         */
        constructor(year: number, month: number, day: number);
        /**
         * Parse a date string into an IOracleDate. An optional datetime format
         * model and an optional NLS parameter string can be provided. If no format
         * model and/or no NLS parameter string is provided, the respective default
         * values for the session are used.
         *
         * If a NLS parameter string is specified, it has the effect of temporarily
         * altering the session's NLS settings for the duration of the call to this
         * method. The NLS parameter string may only set the NLS_DATE_LANGUAGE
         * parameter.
         *
         * If a datetime format model is specified, it overrides the date format
         * determined by NLS settings.
         *
         * @param date date represented as a string
         * @param fmt optional datetime format model
         * @param nlsParam optional NLS parameter string. Must have the same format
         * as the NLS argument to the TO_DATE SQL function.
         * @returns the new date as IOracleDate object
         */
        static fromString(date: string, fmt?: string, nlsParam?: string): IOracleDate;
        /**
         * Gets the current system date and time as an IOracleDate.
         * @returns the system date and time as IOracleDate
         */
        static sysDate(): IOracleDate;
        /**
         * Converts an IOracleDate to a string and returns it. An optional datetime
         * format model and an optional NLS parameter string can be provided. If no
         * format model and/or no NLS parameter string is provided, the respective
         * default values for the session are used.
         *
         * If a NLS parameter string is specified, it has the effect of temporarily
         * altering the session's NLS settings for the duration of the call to this
         * method. The NLS parameter string may only set the NLS_DATE_LANGUAGE
         * parameter.
         *
         * If a datetime format model is specified, it overrides the date format
         * determined by NLS settings.
         *
         * @param fmt optional datetime format model
         * @param nlsParam optional NLS parameter string. Must have the same format
         * as the NLS argument to the TO_DATE SQL function.
         * @returns the date as string
         */
        toString(fmt?: string, nlsParam?: string): string;
        /**
         * Construct a new IOracleDate object by adding number of months to this
         * date. The result of this function is sensitive to the setting for the
         * NLS_CALENDAR parameter in the current session.
         *
         * @param numberOfMonths number of months to add
         * @returns the new date as IOracleDate object
         */
        addMonths(numberOfMonths: number): IOracleDate;
        /**
         * Construct a new IOracleDate object by adding number of days to this date.
         * The result of this function is sensitive to the setting for the
         * NLS_CALENDAR parameter in the current session.
         *
         * @param numberOfDays - number of days to add
         * @returns the new date as IOracleDate object
         */
        addDays(numberOfDays: number): IOracleDate;
        /**
         * Compares two dates.
         *
         * @param d1 - first date value
         * @param d2 - second date value
         * @returns -1 if d1 < d2
         *           0 if d1 = d2
         *           1 if d1 > d2
         */
        static compare(d1: IOracleDate, d2: IOracleDate): number;
        /**
         * Gets the number of days between two dates. The result of this function
         * is sensitive to the setting for the NLS_CALENDAR parameter in the
         * current session.
         *
         * @param d1 - first date value
         * @param d2 - date value to compare with
         * @returns number of days between d1 and d2
         */
        static daysBetween(d1: IOracleDate, d2: IOracleDate): number;
        /**
         * Gets the date at the last day of the month specified by this date. The
         * result of this function is sensitive to the setting for the NLS_CALENDAR
         * parameter of the current session.
         *
         * @returns a new date set set to the last day of the month
         */
        lastDayOfMonth(): IOracleDate;
        /**
         * Gets the date of the first weekday named by dayOfTheWeek that is later
         * than this date. The result of this method is sensitive to the
         * NLS_DATE_LANGUAGE parameter of the current session.
         *
         * @returns a new date set to the first weekday name by dayOfTheWeek that
         * is later than this date.
         */
        nextDay(dayOfTheWeek: string): IOracleDate;
        /**
         * Gets the year component according to the Gregorian calendar.
         *
         * @returns year component (-4712 <= y <= 9999)
         */
        getYear(): number;
        /**
         * Gets the month component according to the Gregorian calendar.
         * @returns month component (1 <= m <= 12)
         */
        getMonth(): number;
        /**
         * Gets the day component according to the Gregorian calendar.
         * @returns day component (1 <= d <= 31)
         */
        getDay(): number;
        /**
         * Gets the hour component.
         * @returns our component (0 <= h <= 23)
         */
        getHour(): number;
        /**
         * Gets the minute component.
         * @returns minute component (0 <= m <= 59)
         */
        getMinute(): number;
        /**
         * Gets the second component.
         * @returns second component (0 <= s <= 59)
         */
        getSecond(): number;
        /**
         * Checks if this has a valid date.
         * @returns true iff the date is valid
         */
        isValid(): boolean;
    }
    export enum JsonScalarTargetType {
        BINARY,
        DATE,
        ORA_NUMBER,
        ORA_DATE,
        ORA_TIMESTAMP,
        ORA_TIMESTAMPTZ
    }
    export enum JsonNodeType {
        SCALAR,
        OBJECT,
        ARRAY
    }
    export { };
}