/*!
 Copyright (c) 2015, 2022, Oracle and/or its affiliates.
*/
/*
 * There are two widgets in this file:
 *  tableModelViewBase - the base widget
 *  tableModelView - simple view for list or table rendering based on a template
 */
/**
 * @uiwidget tableModelViewBase
 * @abstract
 * @since 5.1
 *
 * @classdesc
 * <p>This is a base widget that supports pagination over a {@link model} as well as base support for model editing.
 * It is not intended to be used directly. The examples may use a specific derived widget such as grid or
 * a generic "derived-view". See the {@link grid} and {@link tableModelView} widgets.</p>
 *
 * <p>Any widget that uses column items to edit a model can benefit from the editing support in this base widget.
 * Even if this base widget isn't used similar logic should be implemented for initializing column items,
 * setting model values from the column items, setting column item values from the model, rendering read only
 * view of model field values, and triggering the {@link apex.event:apexbeginrecordedit} and
 * {@link apex.event:apexendrecordedit} events.</p>
 */
/* Todo:
 * - Control break rows don't count in the auto rows per page calculation and no way to know how many there will be so
 * end up with scroll bars. Consider if there is anyway to solve this. Meaning avoid nested scroll bars.
 * - Split base and tmv into different files
 *
 * Depends:
 *    jquery.ui.core.js
 *    jquery.ui.widget.js
 *    apex/util.js
 *    apex/lang.js
 *    apex/debug.js
 *    apex/model.js
 *    apex/item.js
 *    apex/widget.stickyWidget.js (only if stickyFooter or stickyTop is not false)
 */
/* About pagination:
 * Kinds of pagination:
 * 1) Page at a time (aka traditional)
 * 1.a) fetch markup (or data) from server for each page. This is not supported by this widget,
 *   which always uses a model.
 *   In theory if the model has paginationType set to "one" and the model and view have the same page size
 *   it would in effect be an example of 1.a.
 * 1.b) local client side model to fetch data as needed and keep data in the model. View shows just one page at a time
 *   Set pagination.scroll: false
 * With page at a time there are different options for controls to move among the pages and control how many
 * items are on a page. See options pagination.* and rowsPerPage.
 * The model option hasTotalRecords only controls if the total number of records/items is shown.
 * There are 2 kinds of scroll pagination.
 * 2) Add More Scroll (aka high-water-mark)
 * 2.a) Auto add more. As you scroll near to the end of the report more content is added automatically.
 *   Set pagination.scroll: true, model option hasTotalRecords: false
 *       pagination.loadMore: false
 * 2.b) Manual add more. At the bottom of the report is a "load more" button. User clicks the button to add
 *   more content.
 *   Set pagination.scroll: true, model option hasTotalRecords: false
 *       pagination.loadMore: true
 * TODO currently data is only added to the DOM never removed
 * Note it is possible to do add more scroll pagination without a model (see listview widget for an example)
 * but this base widget always uses a model.
 * 3) Virtual scrolling (aka true virtual scrolling)
 * Viewport always looks like it holds all the data even though not all records are rendered to the DOM.
 * 3.a) Add to DOM only
 * In this case empty filler content is added so that it looks like all the data is in the report
 * (you can see this by looking at the proportional scroll bar thumb size). As you scroll the fillers are replaced with
 * report data.
 *   Set pagination.scroll: true, model option hasTotalRecords: true
 * Note: Which kind of scroll pagination is used (2. add more or 3. virtual) depends on the model option hasTotalRecords.
 * Todo consider if a distinct pagination option is needed for this. It would allow the model to know the total records but
 * still do add more scrolling.
 * 3.b) Add and Remove
 * Like 3.a but as records are scrolled out of the viewport they are removed from the DOM (but not the model)
 * Todo implement this option
 * For this to work with selection the selection state must be kept in the model.
 * Note: twitter also does virtual scrolling (3.b) but uses a different technique. Rather than a filler item it puts
 * an ever growing min-height on item container and uses cssTransform on the items.
 * Todo consider that 2 and 3 are the same except that the total
 * Todo work this information into the tmvbase widget overview doc
 */
(function ( util, model, debug, lang, item, $ ) {
    "use strict";

    // todo consider if class prefix should change for this base widget; change GV to ???, GRID to ???
    const C_GRID_NO_DATA = "a-GV-noDataMsg",
        C_GRID_MORE_DATA = "a-GV-moreDataMsg",
        C_GRID_ALT_MSG = "a-GV-altMessage",
        SEL_GRID_ALT_MSG = "." + C_GRID_ALT_MSG,
        C_GRID_ALT_MSG_TEXT = C_GRID_ALT_MSG + "-text",
        SEL_GRID_ALT_MSG_TEXT = "." + C_GRID_ALT_MSG_TEXT,
        C_GRID_ALT_MSG_ICON = C_GRID_ALT_MSG + "-icon",
        C_GRID_FOOTER = "a-GV-footer",
        C_GRID_HIDE_DELETED = "a-GV--hideDeleted",
        C_GRID_SCROLL_FILLER = "a-GV-scrollFiller",
        SEL_GRID_SCROLL_FILLER = "." + C_GRID_SCROLL_FILLER,
        SEL_LOAD_MORE = ".a-GV-loadMoreButton",
        C_DELETED = "is-deleted",
        SEL_DELETED = "." + C_DELETED,
        A_SELECTED = "aria-selected",
        C_ACTIVE = "is-active",
        C_SELECTED = "is-selected",
        SEL_SELECTED = "." + C_SELECTED,
        SEL_AGGREGATE = ".is-aggregate",
        A_LBL_BY = "aria-labelledby",
        C_JS_TABBABLE = "js-tabbable",
        SEL_JS_TABBABLE = "." + C_JS_TABBABLE,
        SEL_TABBABLE = ":tabbable",
        SEL_VISIBLE = ":visible",
        DATA_START = "data-start",
        DATA_END = "data-end",
        DATA_ID = "data-id",
        DATA_ROWNUM = "data-rownum",
        DATA_STATE = "data-state",
        DATA_PAGE = "data-page",
        A_DISABLED = "disabled",
        SEL_PAGE_CONTROLS = ".js-pg-prev,.js-pg-next,.js-pg-first,.js-pg-last";

    const SCROLL_PAGE_CHECK = 300, // ms between check if paging needed
        INSERT_BEFORE = "before",
        INSERT_AFTER = "after",
        MIN_SCROLL_PAGE_SIZE = 40,
        VAR_HEIGHT_AUTO_PAGE_SIZE = 20,
        SAFE_DEFAULT_ROW_HEIGHT = 43; // nothing special about this number other than it is a reasonable size and not 0.

    /**
     * <p>This event is triggered when a {@link model} row/record is about to be edited (when a new row/record is
     * selected or enters edit mode).</p>
     *
     * @event apexbeginrecordedit
     * @memberof apex
     * @property {Event} event <code class="prettyprint">jQuery</code> event object.
     * @property {object} data Additional event data.
     * @property {model} data.model The {@link model} that is being edited.
     * @property {model.Record} data.record The record that is beginning to be edited.
     * @property {string} data.recordId The record id that is beginning to be edited.
     */
    /**
     * <p>This event is triggered when a {@link model} row/record is done being edited (when a new row/record is
     * selected or exits edit mode).</p>
     *
     * @event apexendrecordedit
     * @memberof apex
     * @property {Event} event <code class="prettyprint">jQuery</code> event object.
     * @property {object} data Additional event data.
     * @property {model} data.model The {@link model} that is being edited.
     * @property {model.Record} data.record The record that is done being edited.
     * @property {string} data.recordId The record id that is done being edited.
     */

    const EVENT_BEGIN_RECORD_EDIT = "apexbeginrecordedit",
        EVENT_END_RECORD_EDIT = "apexendrecordedit",
        EVENT_PAGE_CHANGE = "pageChange";

    const hasOwnProperty = util.hasOwnProperty,
        applyTemplate = util.applyTemplate,
        isArray = Array.isArray;

    const toInteger = ( numStr )  => {
        return parseInt( numStr, 10 );
    };

    // Because multiple widgets can share the same column items we use a cache to avoid having
    // to recreate the items multiple times. This assumes that item elements are not destroyed and recreated with the
    // same id. If they were the widgets using them would also need to be recreated.
    let gColumnItemCache = {};

    function getMessage( key ) {
        return lang.getMessage( "APEX.GV." + key );
    }

    function formatMessage( key, ...args ) {
        return lang.formatMessage( "APEX.GV." + key, ...args );
    }

    function pageBoundary( offset, pageSize ) {
        return Math.floor( offset / pageSize ) * pageSize;
    }

    function setFillerHeight( f$, start, end, recPerRow, rowHeight ) {
        let h = Math.ceil( ( end + 1 - start ) / recPerRow ) * rowHeight;
        f$.css( "height", h );
    }

    function updateFillerHeights( dataContainer$, recPerRow, avgRowHeight ) {
        dataContainer$.find( SEL_GRID_SCROLL_FILLER ).each( function () {
            let f$ = $( this ),
                start = toInteger( f$.attr( DATA_START ) ),
                end = toInteger( f$.attr( DATA_END ) );

            setFillerHeight( f$, start, end, recPerRow, avgRowHeight );
        } );
    }

    // the filler rows need to be measurable so can't use show/hide i.e. display:none; Need to get their offset.
    function fillerVisible( filler$ ) {
        return filler$.css( "visibility" ) !== "collapse";
    }

    function toggleFillerVisible( filler$, show ) {
        filler$.css( "visibility", show ? "" : "collapse" ).toggleClass( "is-hidden", !show );
    }

    function modelHasTotalRecords( model ) {
        return model.getOption( "hasTotalRecords" );
    }

    $.widget( "apex.tableModelViewBase",
        /**
         * @lends tableModelViewBase.prototype
         */
        {
        version: "21.1",
        widgetEventPrefix: "tableModelViewBase",
        options: {
            /**
             * <p>Identifier of model that this view widget will display data from. Can include an instance as well.
             * The model must already exist. This option is required. See {@link apex.model.create}
             * <code class="prettyprint">modelId</code> argument.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {model.ModelId}
             * @example [ "myModel", "1011" ]
             * @example "myModel"
             */
            modelName: null,
            /**
             * <p>Options object to pass to {@link apex.util.showSpinner}. The default depends on the
             * <code class="prettyprint">hasSize</code> option.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {object}
             * @default { fixed: !options.hasSize }
             * @example null
             * @example null
             */
            progressOptions: null,
            /**
             * <p>Text to display when there is no data.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {string}
             * @default ""
             * @example "No employees found."
             * @example "No records found."
             */
            noDataMessage: "",
            /**
             * <p>Icon to display when there is no data. The icon is displayed above the
             * <code class="prettyprint">noDataMessage</code> text.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {string}
             * @default "icon-irr-no-results"
             * @example "fa fa-warning"
             * @example "fa fa-warning"
             */
            noDataIcon: "icon-irr-no-results",
            /**
             * <p>Max records exceeded message to display if the server has more data than the configured maximum</p>
             * todo determine if this is needed and if so finish implementation
             * @ignore
             * @memberof tableModelViewBase
             * @instance
             * @type {string}
             * @default ""
             */
            moreDataMessage: "",
            /**
             * <p>Determine if the view allows editing. If true the {@link model} must also allow editing but
             * if false the model could still allow editing.
             * If true the view data can be edited according to what the model allows. Only applies if the
             * view supports editing.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            editable: false,
            /**
             * <p>Specifies if a new record should be automatically added when the model doesn't contain any data.
             * If supported by the derived view a new record may be added when moving beyond the last record in the view
             * while editing.
             * Must only be true if the model and view are editable and the model allows adding records.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            autoAddRecord: false,
            /**
             * <p>Determine if deleted rows (records) are removed from the view right away or shown with a visual effect
             * to indicate they are going to be deleted. If true (and the view is editable) deleted records will not be visible,
             * otherwise they are visible but have a visual indication that they are deleted. The actual records are not
             * deleted on the server until the model is saved. The visual effect is determined by CSS rules and is
             * typically strike through. See also {@link apex.model.create} <code class="prettyprint">onlyMarkForDelete</code> option.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            hideDeletedRows: false,
            /**
             * <p>This affects scrolling and how any header (if the view has a header) or footer position is handled.</p>
             *
             * <p>Set to true if the view is in a container that has a specific height defined. When hasSize is true
             * the record content will scroll within the bounds of the region.</p>
             * <p>Set to false if the view does not have a defined height. The view height will be as large as needed to
             * contain the view records as determined by pagination settings. The view may scroll within the browser
             * window. Other options may control if the header (if the view has a header) or footer sticks to the
             * window.
             * </p>
             * <p>The container width must always be defined.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             */
            // todo consider if/why the width must always be defined was this specific to grid?
            // todo consider that this option should not be changed or if it is need to do a re-render refresh
            // Used in resize logic and in figuring out the scroll parent. Also helps with busy spinner placement.
            // Grid needs it for frozen cols logic.
            // Dependent options: stickyTop, stickyFooter, scrollParentOverride
            hasSize: false,
            /**
             * <p>Determine if the view will include a footer to show status and pagination controls and information.
             * If true a footer is shown at the bottom of the view. If false no footer is shown.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default true
             * @example false
             * @example false
             */
            footer: true,
            /**
             * <p>Specify if all the rows will have the same height or variable heights.
             * </p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default true
             * @example false
             * @example false
             */
            fixedRowHeight: true,
            /**
             * <p>Text to display when a field/column value is null or empty string.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {string}
             * @default "-"
             * @example "- null -"
             * @example "- null -"
             */
            showNullAs: "-",
            /**
             * <p>Options to pass to the {@link apex.util.applyTemplate} function when processing any templates.
             * See {@link apex.util.applyTemplate} for details on the option properties.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {object}
             * @default {}
             * @example {
             *         // This example would enable you to use the placeholder #TODAY# in any templates.
             *         placeholders: { TODAY: (new Date()).toISOString() }
             *     }
             * @example {
             *     // This example would enable you to use the placeholder #TODAY# in any templates.
             *     placeholders: { TODAY: (new Date()).toISOString() }
             * }
             */
            applyTemplateOptions: {},
            /**
             * <p>Determine if the header will stick to the top of the page as it scrolls.</p>
             * <p>Only applies if {@link tableModelViewBase#hasSize} is false.
             * If false the header will not stick to the page.
             * If true or a function the header will stick to the top of the page using the
             * undocumented <code class="prettyprint">stickyWidget</code> widget.
             * If the value is a function then it is passed to the
             * <code class="prettyprint">stickyWidget</code> as the top option.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {(boolean|function)}
             * @default false
             * @example true
             * @example true
             */
            stickyTop: false,
            /**
             * <p>Determine if the footer will stick to the bottom of the page. Only applies if
             * <code class="prettyprint">hasSize</code> is false and
             * <code class="prettyprint">footer</code> is true.
             * If false the footer will not stick to the bottom of the page.
             * If true the footer will stick to the bottom of the page.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            stickyFooter: false,
            /**
             * <p>Hide the footer if there is no data. This only applies if
             * <code class="prettyprint">footer</code> is true.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            hideEmptyFooter: false,
            /**
             * <p>Pagination settings.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {object}
             * @property {boolean} scroll If true the scroll bar is used to page through the results a.k.a infinite
             *   scrolling or virtual paging. If false then next and previous buttons are shown. This is
             *   'page at a time' or traditional pagination. Default is false.
             * @property {boolean} virtual Only applies if <code class="prettyprint">scroll</code> is true.
             *   If false new records are rendered and added to the DOM as the user scrolls to the bottom of
             *   the view. Records are never removed from the DOM. This is 'add more' (aka high-water-mark)
             *   scroll pagination.
             *   If true records can be removed from the DOM as the user scrolls and the records are no longer visible.
             *   If true and in addition <code class="prettyprint">loadMore</code> is false and the model knows the
             *   total number of records (model option <code class="prettyprint">hasTotalRecords</code> is true) then
             *   the view looks as if it contains all the records but only the records that are currently visible
             *   are rendered. This allows virtual scroll paging in both directions. This is 'virtual' scroll
             *   pagination (aka true virtual scrolling).
             *   In this case, if the view supports selection the {@link persistSelection} option should be true so that
             *   selection state isn't lost when records are removed from the DOM.
             *   Default is false.
             * @property {boolean} loadMore If true show a load more button rather than auto paging.
             *   Only applies if <code class="prettyprint">scroll</code> is true. Default is false.
             * @property {boolean} showPageLinks If true show page links between buttons. Only applies if
             *   <code class="prettyprint">scroll</code> is false
             *   The model must know the total number of rows for this to be true. Default is false.
             * @property {number} maxLinks The maximum number of links to show when
             *   <code class="prettyprint">showPageLinks</code> is true. Default is 5.
             * @property {boolean} showPageSelector If true show a drop down page selector between the buttons.
             *   Only applies if <code class="prettyprint">scroll</code> is false.
             *   The model must know the total number of rows for this to be true. Default is false.
             * @property {boolean} showRange If true the range of rows/records is shown. It is shown between the
             *   buttons unless <code class="prettyprint">showPageLinks</code> or
             *   <code class="prettyprint">showPageSelector</code> is true. The range is shown as "X to Y" if the
             *   model doesn't know the total rows and "X to Y of Z" if the model does know the total number of rows.
             *   Default is true.
             * @property {boolean} firstAndLastButtons Only applies if <code class="prettyprint">scroll</code> is false.
             *   If true first and last buttons are included. For this to be true the model must know the total number of rows.
             *   Default is false.
             * @property {boolean} hideSinglePage Only applies if <code class="prettyprint">scroll</code> is false. When
             *   true and there is just one page of results the pagination controls are hidden. When false the pagination
             *   controls are disabled when there is just one page. The default is false.
             * @example
             *     {
             *         showRange: true,
             *         showPageLinks: true,
             *         maxLinks: 6,
             *         firstAndLastButtons: true
             *     }
             *  @example {...}
             */
            pagination: {
                scroll: false,
                virtual: false,
                loadMore: false,
                showPageLinks: false,
                maxLinks: 5,
                showPageSelector: false,
                showRange: true,
                firstAndLastButtons: false,
                hideSinglePage: false
            },
            /**
             * <p>If true and the view supports selection, the selection state for each row or item will be saved as
             * record metadata in the model.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example jsinit(Interactive Grid [defaultGridViewOptions])
             *     {
             *         persistSelection: true
             *     }
             * @example true
             */
            persistSelection: false,
            /**
             * <p>Defines highlight color information for the view. Only applies to views that support highlighting.
             * Style rules are injected into the document based on the highlight object.</p>
             * <p>The object is a mapping of highlight id to color definition.</p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {object}
             * @property {object} * A highlight ID. A unique ID for the highlight rule. The object can contain
             *   any number of highlight rules. The {@link model} record or field <code class="prettyprint">highlight</code>
             *   metadata (see {@link model.RecordMetadata}) is used to associate the model data with the
             *   highlight rule. One of <code class="prettyprint">color</code> or
             *   <code class="prettyprint">background</code> must be given.
             * @property {number} *.seq A number that defines the order of the CSS rule that is added.
             * @property {boolean} *.row If true the highlight applies to the record/row.
             * @property {string} [*.color] The foreground color. If given, must be a valid CSS color value.
             * @property {string} [*.background] The background color. If given, must be a valid CSS color value.
             * @property {string} [*.cssClass] The class name for the rule. This is the class used in the rule and
             *   given to the appropriate element in the view so that the desired highlight colors are applied.
             *   The cssClass defaults to the highlight id prefixed with "hlr_" if row is true and  "hlc_" otherwise.
             * @example
             *     {
             *         "hlid0001": {
             *             seq: 1,
             *             row: true,
             *             color: "#FF7755"
             *         },
             *         ...
             *     }
             * @example {...}
             */
            highlights: {},
            /**
             * A function to call to post process cell/field or record output just prior to rendering to the DOM.
             * Only applies to cells/fields that have a template or a value. Derived views may apply this function
             * to a whole record rather than cell by cell.
             * function( context, value, [col])
             * currently internal use only
             * todo doc?
             * @ignore
             * @memberof tableModelViewBase
             * @instance
             * @type {?function}
             * @default null
             */
            highlighter: null,
            /**
             * A value to pass to the highlighter function. Typically an object.
             * currently internal use only
             * todo doc? todo consider make this an array of 1 so it doesn't get deep copied
             * @ignore
             * @memberof tableModelViewBase
             * @instance
             * @type {*}
             * @default null
             */
            highlighterContext: null,
            /**
             * <p>Determine how many records to show in one page.
             * Only applies if <code class="prettyprint">pagination.scroll</code> is false,
             * otherwise this value is ignored. If null this value is determined by the viewport height.
             * </p>
             * <p>Note the name of this option is a little confusing because some views put more than one
             * record on the same visual row. This value is the number of records (or items) shown on a page.
             * For example if <code class="prettyprint">rowsPerPage</code> is 10 and the view shows
             * two records per row then there will be a total of 5 rows showing 10 records.
             * </p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {?number}
             * @default null
             * @example 50
             * @example 50
             */
            // todo update description for rowsPerPage: applies when loadMore paging
            rowsPerPage: null,
            /**
             * <p>The text message key to use for showing the number of selected rows/records in the footer.
             * The message key must have exactly one parameter %0 which is replaced with the number of rows/records
             * selected.
             * </p>
             *
             * @memberof tableModelViewBase
             * @instance
             * @type {string}
             * @default "APEX.GV.SELECTION_COUNT"
             * @example "MY_SELECTION_MESSAGE"
             * @example "MY_SELECTION_MESSAGE"
             */
            selectionStatusMessageKey: "APEX.GV.SELECTION_COUNT",
            // todo consider a similar option to override APEX.GV.DELETED_COUNT
            /**
             * Experimental; to be used for UT maximize region
             * Only applies if hasSize is false.
             * Optional setting for scrollParent$ rather than window.
             * @ignore
             * @memberof tableModelViewBase
             * @instance
             * @type {Element}
             * @default null
             */
            scrollParentOverride: null
        },

        //
        // Public pagination methods
        //

        /**
         * <p>Display the first page of records. If option <code class="prettyprint">pagination.scroll</code>
         * is true simply scrolls to the top of the viewport
         * and a new page of records is added if needed. If <code class="prettyprint">pagination.scroll</code>
         * is false and not already on the first page the view is refreshed and shows the first page.</p>
         *
         * @return {boolean} true for success, false if a page is currently being rendered.
         * @example <caption>This example goes to the first page.</caption>
         * $( ".selector" ).grid( "firstPage" );
         */
        firstPage: function() {
            let sp$;

            if ( this.renderInProgress ) {
                return false;
            } // else
            if ( this.options.pagination.scroll ) {
                sp$ = this.scrollParent$;
                sp$.scrollTop( this.scrollDelta );
            } else {
                if ( this.pageOffset > 0 ) {
                    this.pageOffset = 0;
                    this.refresh( false );
                }
            }
            return true;
        },

        /**
         * <p>Display the previous page of records. If <code class="prettyprint">pagination.scroll</code>
         * is true the viewport scrolls up one page and
         * records are added if needed. If <code class="prettyprint">pagination.scroll</code>
         * is false and not on the first page refresh the view to show the previous page.</p>
         *
         * @return {boolean} true for success, false if a page is currently being rendered.
         * @example <caption>This example goes to the previous page.</caption>
         * $( ".selector" ).grid( "previousPage" );
         */
        previousPage: function() {
            let sp$, st;

            if ( this.renderInProgress ) {
                return false;
            } // else
            if ( this.options.pagination.scroll ) {
                sp$ = this.scrollParent$;
                st = sp$.scrollTop() - this.scrollDelta - this.viewportHeight;
                if ( st < 0 ) {
                    st = 0;
                }
                sp$.scrollTop( st + this.scrollDelta );
            } else {
                if ( this.pageOffset > 0 ) {
                    this.pageOffset -= this.pageSize;
                    if ( this.pageOffset < 0 ) {
                        this.pageOffset = 0;
                    }
                    this.refresh( false );
                }
            }
            return true;
        },

        /**
         * <p>Display the next page of records. If <code class="prettyprint">pagination.scroll</code>
         * is true the viewport scrolls down one page and
         * records are added if needed. If <code class="prettyprint">pagination.scroll</code>
         * is false and not on the last page refresh the view to show the next page.</p>
         *
         * @return {boolean} true for success, false if a page is currently being rendered.
         * @example <caption>This example goes to the next page.</caption>
         * $( ".selector" ).grid( "nextPage" );
         */
        nextPage: function() {
            let sp$, max, st, total;

            if ( this.renderInProgress ) {
                return false;
            } // else
            if ( this.options.pagination.scroll ) {
                sp$ = this.scrollParent$;
                max = this._getDataContainer().first().height() - this.viewportHeight;
                st = sp$.scrollTop() - this.scrollDelta + this.viewportHeight;
                if ( st > max ) {
                    st = max;
                }
                sp$.scrollTop( st + this.scrollDelta );
            } else {
                total = this.model.getTotalRecords();
                if ( total < 0 || this.pageOffset + this.pageCount < total ) { // page size can change so use pageCount to check bounds
                    this.pageOffset += this.pageSize; // ok to use pageSize here; page boundary is forced in _addPageOfRecords
                    this.refresh( false );
                }
            }
            return true;
        },

        /**
         * <p>Display the last page of records. If <code class="prettyprint">pagination.scroll</code>
         * is true simply scrolls to the bottom of the viewport
         * and a new page of records is added if needed. If <code class="prettyprint">pagination.scroll</code>
         * is false and not already on the last page the view is refreshed and shows the last page.
         * This method only works correctly if the model knows the total number of rows.</p>
         *
         * @return {boolean} true for success, false if a page is currently being rendered.
         * @example <caption>This example goes to the last page.</caption>
         * $( ".selector" ).grid( "lastPage" );
         */
        lastPage: function() {
            let totalPages, sp$, max, total;

            if ( this.renderInProgress ) {
                return false;
            } // else
            if ( this.options.pagination.scroll ) {
                sp$ = this.scrollParent$;
                max = this._getDataContainer().first().height() - this.viewportHeight;
                sp$.scrollTop( max + this.scrollDelta );
            } else {
                total = this.model.getTotalRecords();
                if ( total > 0 && this.pageOffset + this.pageCount < total ) {
                    totalPages = Math.ceil( total / this.pageSize );
                    this.pageOffset = ( totalPages - 1 ) * this.pageSize;
                    this.refresh( false );
                }
            }
            return true;
        },

        /**
         * <p>Load more records into the view. If option <code class="prettyprint">pagination.scroll</code>
         * is true this adds a new page of records to the end.
         * If <code class="prettyprint">pagination.scroll</code> is false this is the same as <code class="prettyprint">nextPage</code>.
         * This is intended to be used when <code class="prettyprint">pagination.loadMore</code> is true.</p>
         *
         * @return {boolean} true for success, false if a page is currently being rendered.
         */
        loadMore: function() {
            if ( this.options.pagination.scroll && !this._scrollHandler ) {
                return this._addNextPage();
            } else {
                return this.nextPage();
            }
        },

        /**
         * <p>An object with properties that describe the current pagination state.</p>
         *
         * @typedef tableModelViewBase.pageInfo
         * @type {object}
         *
         * @property {number} rowHeight The height of a view row. Some views show multiple records per view row.
         *   If records have variable heights
         *   (option {@link tableModelViewBase#fixedRowHeight} is false) then this is a running approximate
         *   average based on the rows currently rendered to the DOM.
         * @property {number} recordsPerRow The number of records displayed in a view row.
         * @property {number} firstOffset The 1 based offset of the first record in the page for 'page at a time' pagination.
         * @property {number} lastOffset The 1 based offset of the last record in the page for 'page at a time' pagination.
         * @property {number} pageSize The number of records requested from the model and rendered at a time.
         *   For 'page at a time' pagination this is the number of records shown in a full report page and depends on
         *   option {@link tableModelViewBase#rowsPerPage}.
         * @property {number} pageOffset The 0 based offset of the page most recently retrieved from the model.
         *   For 'page at a time' pagination this is the current page.
         * @property {number} [total] The number of records in the report. Only present if the model
         *   knows the total number of records (model option <code class="prettyprint">hasTotalRecords</code> is true).
         * @property {number} [scrollOffset] This is the viewport scroll offset in pixels.
         *   Only present for scroll pagination.
         * @property {number} [viewOffset] The 0 based offset of the first record in the vieport.
         *   Only present for scroll pagination.
         * @property {number} [currentPage] The 0 based current page number. Only present for 'page at a time' pagination.
         * @property {number} [totalPages] The total number of pages. Only present for 'page at a time' pagination and
         *   the when model knows the total number of records (model option <code class="prettyprint">hasTotalRecords</code>
         *   is true).
         */

        /**
         * <p>Return information about the current pagination state of the view.
         * Returns null if there is no data in the report.</p>
         *
         * @return {?tableModelViewBase.pageInfo}
         */
        getPageInfo: function() {
            let result = null,
                hasTotalRecords = modelHasTotalRecords( this.model ),
                pagination = this.options.pagination,
                total = this.model.getTotalRecords(),
                serverTotal = this.model.getServerTotalRecords();

            if ( !this.noData ) {
                result = {
                    rowHeight: this.avgRowHeight, // with variable height items/records this will be a rough average
                    recordsPerRow: this._getRecordsPerRow(), // assumes all (but possibly last) rows have same number of records
                    firstOffset: this.pageFirstOffset || null,
                    lastOffset: this.pageLastOffset || null,
                    pageSize: this.pageSize, // how many records get rendered at a time
                    pageOffset: this.pageOffset // offset of last page of records fetched
                };
                if ( hasTotalRecords ) {
                    result.total = serverTotal;
                }
                if ( pagination.scroll ) {
                    let index,
                        spIsWindow = this.scrollParent$[0] === window,
                        isTable = this._getDataContainer()[0].nodeName === "TBODY",
                        items = this._getDataContainer().children().not( SEL_AGGREGATE ); // todo make it clear that this is a requirement for aggregate items

                    result.scrollOffset = this.scrollParent$.scrollTop();
                    index = util.binarySearch( items, result.scrollOffset, ( a, b ) => {
                        let el$ = $(b),
                            curOffset = spIsWindow ? el$.offset().top : el$.position().top;

                        if ( !( spIsWindow || isTable ||
                            el$.offsetParent().closest( this.scrollParent$ ).length > 0 ) ) {
                            curOffset += result.scrollOffset;
                        }
                        return a - curOffset;
                    } );
                    result.viewOffset = toInteger( items.eq( index ).attr( DATA_ROWNUM ) ) - 1;
                } else {
                    result.currentPage = Math.floor( this.pageOffset / this.pageSize );
                    if ( total >= 0 ) {
                        result.totalPages = Math.ceil( total / this.pageSize );
                    }
                }
            }
            return result;
        },

        /**
         * <p>Go to the specified page number. This should only be used when
         * <code class="prettyprint">pagination.scroll</code> is false and the model knows the total number of records.</p>
         *
         * @param pPageNumber zero based page number
         * @return {boolean} true for success, false if a page is currently being rendered.
         */
        gotoPage: function( pPageNumber ) {
            let totalPages,
                total = this.model.getTotalRecords();

            if ( this.renderInProgress ) {
                return false;
            } // else
            if ( total > 0 ) {
                totalPages = Math.ceil( total / this.pageSize );
                if ( pPageNumber >= 0 && pPageNumber < totalPages ) {
                    this.pageOffset = pPageNumber * this.pageSize;
                    this.refresh( false );
                }
            }
            return true;
        },

        //
        // Public methods
        //

        /**
         * <p>Returns the identity of the active record or null if there is no active record.
         * The active record is the one currently being edited.</p>
         *
         * @return {string} Active record id.
         */
        getActiveRecordId: function() {
            return this.activeRecordId;
        },

        /**
         * <p>Returns the active record or null if there is no active record.
         * The active record is the one currently being edited.</p>
         *
         * @return {model.Record} Active record.
         */
        getActiveRecord: function() {
            return this.activeRecord;
        },

        /**
         * <p>Return the model currently being used by this view.
         * The model can change over time so the returned model should not be saved and used later.
         * If you need to store a reference to the model use {@link apex.model.get} and release it with
         * {@link apex.model.release}.
         * </p>
         * @returns {model} The current {@link model}.
         */
        getModel: function() {
            return this.model;
        },

        /**
         * <p>Use after a column item value is set without triggering a change event to update the model and grid view.
         * Has no effect if there is no active record.</p>
         * <p>When a dynamic action or other event handler on a change event updates the value of the same item that
         * triggered the change event, the change event from setting the value should be suppressed to avoid
         * an infinite loop. However the model is only updated from a change event. This method offers a solution
         * to the model not being updated if the value is set asynchronously.
         * Call this method anytime a column item is updated and the change event is suppressed.</p>
         * @param {string} pColumn The name of the column.
         * @example <caption>This example updates the "SALARY" column, which has static id "C_SALARY", in
         * interactive grid with static id "MyGrid", to add 10 to whatever the user enters.
         * <code class="prettyprint">setTimeout</code> is used to simulate an async value update.
         * The active row must be locked around the async update.</caption>
         * var salary = apex.item( "C_SALARY" );
         * $( salary.node ).change( function( event ) {
         *     // assume the current view is grid and not single row view.
         *     var grid$ = apex.region( "MyGrid" ).call( "getCurrentView" ).view$;
         *     grid$.grid("lockActive");
         *     setTimeout( function() {
         *         // suppress this change otherwise this handler will be triggered again
         *         salary.setValue( parseFloat( salary.getValue() ) + 10, null, true );
         *         // suppressing the value means the grid model is not updated so call setActiveRecordValue
         *         grid$.grid( "setActiveRecordValue", "SALARY" )
         *             .grid( "unlockActive" );
         *     }, 10 ):
         * } );
         */
        setActiveRecordValue: function( pColumn ) {
            var ci = this.columnItems[pColumn];

            if ( this.activeRecord && ci ) {
                this.ignoreItemChange = true;
                this._setModelValue( null, ci.item, this.activeRecord, pColumn, true );
                this.ignoreItemChange = false;
            }
        },

        /**
         * <p>Call to lock the active row while async processing is in progress.</p>
         *
         * <p>The view edits one row/record at a time. This is known as the active row. In edit mode as the user changes
         * the focused cell with the mouse, tab or enter keys if the new cell is on a different row the previous row
         * is deactivated and the new row is activated. Any dynamic actions or other code that manipulates Column items
         * are acting on the active row. If any actions are asynchronous such as using Ajax to set a column item value
         * then the row must not be deactivated while the async action is in progress otherwise the result would be applied
         * to the wrong row!</p>
         *
         * <p>So this method must be called before starting an async operation. It can be called multiple times if there are
         * multiple async operations. For each call to <code class="prettyprint">lockActive</code>
         * there must be exactly one call to <code class="prettyprint">unlockActive</code>. See also
         * See {@link tableModelViewBase#unlockActive}</p>
         *
         * <p>If the view is part of an APEX region plugin, that region should implement the
         * <code class="prettyprint">beforeAsync</code> and <code class="prettyprint">afterAsync</code>
         * functions on the object returned from {@link region#getSessionState} by calling
         * <code class="prettyprint">lockActive</code> and <code class="prettyprint">unlockActive</code>
         * respectively. Then if an appropriate target option is passed to {@Link apex.server.plugin} then the locking will
         * be done automatically. Dynamic Actions that act on column items pass the correct target option.
         * The bottom line is that for Dynamic Actions on columns of an Interactive Grid these lock/unlock methods are
         * called automatically.</p>
         * @example <caption>See {@link grid#setActiveRecordValue} for an example.</caption>
         */
        lockActive: function() {
            if ( this.activeRecord ) {
                this.activeLockCount += 1;
            }
        },

        /**
         * <p>Call to unlock the active row after async processing is complete.</p>
         * <p>Call after the async operation completes. See {@link tableModelViewBase#lockActive} for more information.
         * @example <caption>See {@link grid#setActiveRecordValue} for an example.</caption>
         */
        unlockActive: function() {
            if ( this.activeRecord ) {
                this.activeLockCount -= 1;
                // just in case don't let it go negative
                if ( this.activeLockCount < 0 ) {
                    this.activeLockCount = 0;
                }
                if ( this.activeLockCount === 0 ) {
                    if ( this.activeUnlockCallback ) {
                        this.activeUnlockCallback();
                        this.activeUnlockCallback = null;
                    }
                    if ( this.finishEditingCallback ) {
                        this.finishEditingCallback();
                        this.finishEditingCallback = null;
                    }
                }
            }
        },

        /**
         * <p>This method makes sure that the model is up to date with all current edits.
         * While the active row is being edited it may at times be out of sync with the model.</p>
         * <p>Any code that wants to interact with the model should call this method
         * to make sure the view and model are in sync and then interact with the model
         * when the returned promise is resolved. You must still check for changes in the model.
         * Just because the promise is resolved doesn't mean there where or are any changes.</p>
         * <p>Note: This does not affect any edit mode.</p>
         * @returns {Promise} A promise that is resolved when the model has been synchronized with the view.
         * @example <caption>The following function saves the grid view model for the Interactive Grid region
         * given by static id <code>igRegion</code>. This shows how <code>finishEditing</code> is used but it is
         * generally much better to use the built-in Interactive Grid "save" action.</caption>
         * function doSave( igRegion ) {
         *     var p, finished,
         *         grid = apex.region( igRegion ).call( "getViews" ).grid;
         *
         *     finished = grid.view$.grid( "finishEditing" );
         *     finished.done( function() {
         *         // now the model has all the current changes from the view
         *         p = apex.model.save( null, null, grid.modelName, true );
         *         p.done( function( data ) {
         *             // do something after save completes
         *         } );
         *     } );
         * }
         */
        finishEditing: function() {
            var self = this,
                deferred = $.Deferred();

            function finish() {
                if ( self._finishEditing ) {
                    self._finishEditing( deferred );
                } else {
                    deferred.resolve();
                }
            }

            if ( this.activeLockCount > 0 ) {
                this.finishEditingCallback = finish;
            } else {
                finish();
            }
            return deferred.promise();
        },

        addStateIcon: function( pStateName, pIconClass, pMessage ) {
            var out = util.htmlBuilder();

            this.stateIconMessages[pStateName] = pMessage;
            if ( this._findStateIcon( pStateName ).length === 0 ) {
                out.markup( "<span tabindex='0'" )
                    .attr( "class", pIconClass )
                    .attr( DATA_STATE, pStateName )
                    .markup( "></span>");
                this.stateIcons$.append( out.toString() );
            }
        },

        removeStateIcon: function( pStateName ) {
            this._findStateIcon( pStateName ).remove();
            delete this.stateIconMessages[pStateName];
        },

        //
        // Methods for derived views to call
        //

        _create: function() {
            var o = this.options;

            if ( o.hideDeletedRows ) {
                this.element.addClass( C_GRID_HIDE_DELETED );
            }

            // ** Model and data related
            // this.modelName The name of the model used by this view instance. Given to _initModel.
            // this.model The model used by this view instance set in _initModel.
            // this.modelViewId The model viewId returned by model.subscribe, set in _initModel.
            this.noData = true; // true when the view has no data because the model has no data, false otherwise
            // this.noData$ the element containing the no data message. See options noDataMessage, noDataIcon
            this.pendingRefresh = null; // used for lazy rendering; don't refresh if invisible
            this.renderInProgress = false;

            // ** Pagination related
            this.pageOffset = 0; // The model offset of the first record in the current page for traditional pagination.
                // The offset of the next page of records to get from the model.
                // init to 0 when model name changes or model reset notification, and also for scroll pagination in _initPagination
                // pageOffset and pageCount generally set to 0 at the same time
                // used by all the pagination APIs such as nextPage and by _scrollPage
                // used to update pagination footer
            this.highWaterMark = 0; // the offset of the last most rendered record
            this.pageCount = 0; // The number of records displayed in the current page for traditional pagination.
                // cleared along with pageOffset. Set after addPageOfRecords finishes.
                // used to update pagination footer
            this.totalRenderedRecords = 0; // Tracks the number of records currently rendered. Doesn't include control breaks.
            this.pageSize = 10; // How many records in a page for traditional paging
                // and also how any records to request from the model. Used by nextPage, previousPage, gotoPage etc.
                // and determines what the current page is and total pages.
                // this can change over time either by setting the rowsPerPage option (if not scroll paging) or by resize
                // set in _create from rowsPerPage (only if rowsPerPage is set and it is cleared if pagination.scroll)
                // in _initPageSize if not set set to: o.pagination.scroll ? Math.max( 40, 2 * pageSize ) : pageSize;
                // used in _addPageOfRecords for how much to get from model
                // used to calculate a page boundary
                // used to update the pagination footer data (_updateTotalRecords)
            this.onLastPage = false; // true when on the last page. Can be used by derived views.
            // this.scrollParent$ Either value passed into _updateScrollHandler, o.scrollParentOverride, or window.
                // Set in _updateScrollHandler.
                // Used in pagination methods like nextPage etc. when scroll paging,
                // in _initPageSize to get the viewport height
                // in _updateScrollHandler
                // in _scrollPage for scroll pagination
                // in _updateTotalRecords to restore scroll offsets
            // this.scrollDelta Used when setting scrollTop to account for full page scrolling case. Set in _updateScrollHandler
            this.lastScrollLeft = null; // used to preserve scroll offsets
            this.lastScrollTop = null; // used to preserve scroll offsets
            // this.pageFirstOffset Server offset of first record in page. Used to display the page range in footer.
            // this.pageLastOffset Server offset of last record in page. Used to display the page range in footer.
            // this.avgRowHeight The running average row height. If option fixedRowHeight is true then this
                // is the fixed row height.
            // this.viewportHeight Viewport height adjusted to be a multiple of avg row height.
                // set in _initPageSize and used by pagination APIs such as nextPage when scroll paging
            this.scrollPageTimer = null; // Used to delay/debounce scroll pagination processing. See _scrollPage
            this.hasScrollFillers = false; // true when fillers are used in virtual scrolling
            // this.controlBreakCollapsed Coordinate collapsed control break behavior between addPageOfRecords and
                // derived view.
            this._scrollHandler = null; // Handler function stored here so it can be turned off easily.
                // Set in _updateScrollHandler, turned off and cleared in _scrollPage.

            // ** footer related
            this.stateIconMessages = {};
            // this.footer$ Footer element. Set in _initPagination
            // this.pageRange$ Page range element within the footer. Set in _initPagination, used in _updateTotalRecords.
            // this.pageSelector$ Page selector element within the footer. Set in _initPagination, used in _updateTotalRecords.
            // this.status$ Status element within footer. Set in _initPagination, used in _updateStatus.
            // stateIcons$ State icon element within footer. Set in _initPagination, see add/removeStateIcon
            // pageKey Used by pagination.showPageLinks, showPageSelector to know when to update the page selector UI.

            // ** Editing related
            this.ignoreFieldChange = null; // name of field/prop being set by the UI while it is being set, null otherwise
            this.reinitCallbacks = null;
            this.activeRecord = null; // tracks the record currently being edited
            this.activeRecordId = null; // tracks the record currently being edited
            this.activeLockCount = 0; // managed by [un]lockActive; if there are async operations pending this is > 0
            this.activeUnlockCallback = null; // function to call once lock count goes back to zero. Just one thing not a queue
            this.columnItems = null; // map from field property to column item (element and item) that edits the field
            this.columnItemsContainer$ = null; // container of all the column items for this record view

            if ( !o.progressOptions ) {
                o.progressOptions = { fixed: !o.hasSize };
            }

            if ( o.pagination.scroll && !o.pagination.loadMore ) {
                o.rowsPerPage = null;
            }
            if ( o.rowsPerPage ) {
                this.pageSize = o.rowsPerPage;
            }
            this._setOption( "applyTemplateOptions", o.applyTemplateOptions );
        },

        /**
         * Call from _destroy
         * @private
         */
        _tableModelViewDestroy: function() {
            if ( this.footer$ ) {
                this.footer$.remove();
                this.footer$ = null;
            }
            this.pageRange$ = null;
            this.pageSelector$ = null;
            this.status$ = null;
            this.stateIcons$ = null;
            this.element.removeClass( C_GRID_HIDE_DELETED );
            this._removeScrollHandler();
        },

        /**
         * Call from refresh method of derived view to support lazy rendering. If not visible
         * set pendingRefresh true return false so that the view will not be initialized/refreshed.
         * If visible set pendingRefresh false and return true to proceed as usual.
         * @returns {boolean} true if visible and false if not visible
         * @private
         */
        _refreshCheckIfVisible: function() {
            // This offsetParent method of checking visibility is OK because view widget should never be position fixed
            if ( this.element[0].offsetParent === null ) {
                // View is invisible so don't bother rendering anything expect a resize or refresh later
                this.pendingRefresh = true;
                return false;
            }
            this.pendingRefresh = false;
            return true;
        },

        /**
         * Call to initialize the model. Typically done when the widget is created and anytime the modelName option
         * is changed. Also call with null for model name when the widget is destroyed or any other time the model is
         * no longer needed.
         *
         * @param modelName
         * @param altHandler
         * @private
         */
        _initModel: function( modelName, altHandler ) {
            const o = this.options,
                self = this;
            let markDeletes;

            function modelChangeHandler(type, change) {
                var i, rows, row, rec, id, oldId, meta, row$, inserted, records, copy;

                if ( self.pendingRefresh ) {
                    // Ignore any changes that happen before the view is initialized. This can happen if view is initially invisible
                    return;
                }
                if ( type === "refresh" ) {
                    self.pageOffset = 0;
                    self.highWaterMark = 0;
                    self.pageCount = 0;
                    self.refresh();
                } else if ( type === "refreshRecords" || type === "revert" ) {
                    rows = self._elementsFromRecordIds( change.recordIds );
                    for ( i = 0; i < rows.length; i++ ) {
                        row = rows[i];
                        if (!row) {
                            continue;
                        }
                        rec = change.records[i];
                        // for update after insert this is the previous (temp) id and newIds holds the new value
                        // for revert after identity field was changed this is the changed value and newIds holds the new original/reverted value
                        oldId = change.recordIds[i];
                        id = change.newIds[oldId] || oldId;
                        meta = self.model.getRecordMetadata( id );
                        self._replaceRecord( row, rec, oldId, id, meta );
                    }
                } else if ( type === "move" ) {
                    // TODO consider if/when move should be handled as a refresh?
                    records = change.records;
                    if ( change.insertAfterId ) {
                        rows = self._elementsFromRecordIds( [change.insertAfterId] );
                        row = rows.length > 0 ? rows[0] : null;
                    }
                    // remove each moved record then insert in new place
                    for ( i = records.length - 1; i >= 0; i-- ) {
                        rec = records[i];
                        id = self.model.getRecordId( rec );
                        meta = self.model.getRecordMetadata( id );

                        rows = self._elementsFromRecordIds( [id] );
                        if ( rows.length > 0 ) {
                            if ( rows[0] ) {
                                self._removeRecord( rows[0] );
                            }
                        }
                        row$ = self._insertRecord( row, rec, id, meta, row ? INSERT_AFTER : INSERT_BEFORE );
                    }
                } else if ( type === "insert" || type === "copy" ) {
                    if ( type === "insert" ) {
                        copy = false;
                        records = [change.record];
                    } else { // else copy
                        copy = true;
                        records = change.records;
                    }
                    if ( change.insertAfterId ) {
                        rows = self._elementsFromRecordIds( [change.insertAfterId] );
                        row = rows.length > 0 ? rows[0] : null;
                    }
                    if ( self.noData ) {
                        // This insert happens during _addPageOfRecords completion processing due to model refresh event
                        // so make sure that has completed before refreshing again to pick up the newly inserted record.
                        setTimeout( function() {
                            self.refresh();
                            rows = self._elementsFromRecordIds( [self.model.getRecordId(records[0])] );
                            // make sure the row is found
                            if ( rows.length > 0 && rows[0] ) {
                                self._afterInsert( rows, copy );
                            }
                        }, 1 );
                    } else {
                        // insert each added record
                        inserted = [];
                        for ( i = records.length - 1; i >= 0; i-- ) {
                            rec = records[i];
                            id = self.model.getRecordId( rec );
                            meta = self.model.getRecordMetadata( id );

                            row$ = self._insertRecord( row, rec, id, meta, row ? INSERT_AFTER : INSERT_BEFORE );
                            inserted.push( row$ );
                        }
                        self._afterInsert( inserted, copy );
                    }
                } else if ( type === "clearChanges") {
                    if ( markDeletes ) {
                        rows = self._elementsFromRecordIds( change.deletedIds );
                        for ( i = 0; i < rows.length; i++ ) {
                            if ( rows[i] ) {
                                self._removeRecord( rows[i] );
                            }
                        }
                        // todo any other adjustments needed like in the case below?
                    }
                    rows = self._elementsFromRecordIds( change.changedIds );
                    for ( i = 0; i < rows.length; i++ ) {
                        row = rows[i];
                        if ( row ) {
                            id = change.changedIds[i];
                            meta = self.model.getRecordMetadata( id );
                            rec = self.model.getRecord( id );
                            self._updateRecordState( row, id, rec, meta );
                        }
                    }
                } else if ( change.recordIds ) {
                    rows = self._elementsFromRecordIds( change.recordIds );
                    for ( i = 0; i < rows.length; i++ ) {
                        row = rows[i];
                        if ( !row ) {
                            continue;
                        }
                        id = change.recordIds[i];
                        meta = self.model.getRecordMetadata( id );
                        rec = change.records[i];

                        if ( type === "delete" ) {
                            // if record will be removed or if hideDeletedRows and the record is focused then need to
                            // select the next (or prev if at end) row only the derived view knows if the record is focused
                            if ( o.hideDeletedRows || !markDeletes || meta === null ) {
                                self._removeFocus( row );
                            }
                            if ( !markDeletes || meta === null ) {
                                // when meta is null it means the record was really deleted from the model such as in the case of deleting an inserted record
                                self._removeRecord( row );
                            } else {
                                self._updateRecordState( row, id, rec,  meta );
                            }
                        } else {
                            self._updateRecordState( row, id, rec, meta );
                        }
                    }
                    if ( type === "delete" ) {
                        if ( self.model.getTotalRecords() === 0 ) {
                            self.refresh();
                        } else if ( o.hideDeletedRows ) {
                            // make sure none of the deleted rows are selected
                            self.element.find( SEL_DELETED ).removeClass( C_SELECTED ).removeAttr( A_SELECTED );
                            self._updateStatus();
                        }
                    }
                } else if ( type === "set" || type === "metaChange" ) {
                    // Ignore if change came from user editing cell in this widget
                    if ( self.ignoreFieldChange === null || self.ignoreFieldChange !== change.field ) {
                        let property = change.property; // only exists for metaChange notification

                        rec = change.record;
                        id = self.model.getRecordId( rec );
                        if ( change.oldIdentity ) {
                            self._identityChanged( change.oldIdentity, id );
                        }
                        meta = self.model.getRecordMetadata( id );
                        rows = self._elementsFromRecordIds( [id] );
                        if ( rows.length > 0 && rows[0] ) {
                            if ( type === "set" ) {
                                self._updateRecordField( rows[0], rec, change.field, meta );
                            }
                            self._updateRecordState( rows[0], id, rec, meta, property );
                        }
                    }
                } else if ( type === "instanceRename" ) {
                    // update our name for the model so it can be released
                    if ( self.modelName[1] === change.oldInstance ) {
                        self.modelName[1] = change.newInstance;
                    }
                }
                self._updateTotalRecords();
            }

            let modelChanged = true;
//            console.log("xxx init model ", this.modelName, modelName, this.renderInProgress)

            if ( this.model ) {
                // if there was a model unbind our listener
                this.model.unSubscribe( this.modelViewId );
                if ( isArray( this.modelName ) && isArray( modelName ) ) {
                    modelChanged = !util.arrayEqual(this.modelName, modelName);
                } else {
                    modelChanged = this.modelName !== modelName;
                }
                if ( modelChanged ) {
                    // and release the model if it is different from what is being set
                    model.release( this.modelName );
                    this.model = null;
                    this.modelName = null;
                }
            }
            // modelChanged property used to coordinate between _initModel and view refresh.
            // there are many reasons for a view to be refreshed and it may need to know if the refresh was due to
            // the model changing but changing the model can happen distinct from refresh so let it be known here
            // and in the view refresh this flag should be set back to false
            this.modelChanged = modelChanged;
            if ( modelName ) {
                if ( modelChanged ) {
                    this.modelName = modelName;
                    this.model = model.get( modelName );
                    if ( !this.model ) {
                        throw new Error( "Model not found: " + modelName );
                    }
                }
                this.modelViewId = this.model.subscribe( {
                    onChange: altHandler || modelChangeHandler,
                    progressView: this.element,
                    progressOptions: o.progressOptions
                } );
                markDeletes = this.model.getOption( "onlyMarkForDelete" );
                // When the model changes need to go back to the beginning
                if ( this.options.pagination.scroll && this.scrollParent$ ) {
                    // only adjust the scroll top if scrolled past the start of the view already. When scroll parent
                    // is window, this is unlikely to be the case because typically the master comes before the detail
                    // and you have to see the master records to select them. But for hasSize true (heading fixed to region)
                    // it is important to reset the scroll top.
                    if ( this.scrollParent$.scrollTop() > this.scrollDelta ) {
                        this.scrollParent$.scrollTop( this.scrollDelta );
                    }
                }
                this.pageOffset = 0;
                this.highWaterMark = 0;
                this.pageCount = 0;
            }
        },

        /**
         * Call to render a button.
         * @param out
         * @param cls
         * @param icon
         * @param label
         * @param [expanded]
         * @private
         */
        _renderButton: function( out, cls, icon, label, expanded ) {
            out.markup( `<button class='${cls}' type='button'` )
                .attr( "aria-label", label )
                .attr( "title", label )
                .optionalAttr( "aria-expanded", expanded == null ? null : expanded ? "true" : "false" )
                .markup( "><span aria-hidden='true' class='a-Icon " )
                .attr( icon )
                .markup( "'></span></button>" );
        },

        /**
         * Call to render the no data message area
         * @param out
         * @param baseId
         * @private
         */
        _renderAltDataMessages: function( out, baseId ) {
            const o = this.options;

            function msg( cls, suffix, iconCls, text ) {
                out.markup( `<div class='${cls} ${C_GRID_ALT_MSG}' style='display:none;'><div class='${C_GRID_ALT_MSG_ICON}'>\
<span aria-hidden='true' class='a-Icon ${iconCls}'></span></div><span class='${C_GRID_ALT_MSG_TEXT}'` )
                    .optionalAttr( "id", baseId ? baseId + suffix + "_msg" : null ) // an id is added to the message so it can be included in the widget's accessible label
                    .markup( `>${text}</span></div>` ); // the message may contain markup just like IR
            }
            msg( C_GRID_NO_DATA, "_no", o.noDataIcon , o.noDataMessage );
            msg( C_GRID_MORE_DATA, "_more", "icon-warning", o.moreDataMessage );
        },

        /**
         * Call during rendering of the widget after all the data but inside the scroll view port
         * @param out
         * @private
         */
        _renderLoadMore: function( out ) {
            var pagination = this.options.pagination;

            if ( pagination.scroll && pagination.loadMore ) {
                out.markup( "<div class='a-GV-loadMore'><button type='button' class='js-load a-GV-loadMoreButton'>")
                    .content( getMessage( "LOAD_MORE" ) )
                    .markup( "</button></div>" );
            }
        },

        /**
         * Call during rendering of the widget at the very end
         * @param out
         * @param baseId
         * @private
         */
        _renderFooter: function( out, baseId ) {
            var rangeAdded, modelHasTotal,
                buttonClass = "a-GV-pageButton a-GV-pageButton--nav ",
                o = this.options;

            function pageRange() {
                var label = getMessage( "PAGE_RANGE" );
                out.markup( "<span class='js-rangeDisplay'" )
                    .optionalAttr( "id", baseId ? baseId + "_pageRange" : null )
                    .attr( "title", label )
                    .markup( "><span class='u-vh'>" )
                    .content( label )
                    .markup( "</span> <span class='a-GV-pageRange'></span></span>" );
            }

            if ( o.footer ) {
                out.markup( `<div class='${C_GRID_FOOTER}'><div class='a-GV-stateIcons'></div><div` )
                    .optionalAttr( "id", baseId ? baseId + "_status" : null )
                    .markup( " class='a-GV-status'></div><div class='a-GV-pagination'>" );

                if ( o.pagination ) {
                    rangeAdded = false;
                    modelHasTotal = modelHasTotalRecords( this.model );
                    if ( !o.pagination.scroll ) {
                        if ( modelHasTotal && o.pagination.firstAndLastButtons ) {
                            this._renderButton( out, buttonClass + "js-pg-first", "icon-first",
                                getMessage( "FIRST_PAGE" ) );
                        }
                        // todo aria-controls="myreport" for all these buttons. get from this.element[0].id
                        this._renderButton( out, buttonClass + "js-pg-prev", "icon-prev",
                            getMessage( "PREV_PAGE" ) );
                    }

                    if ( modelHasTotal && ( o.pagination.showPageLinks || o.pagination.showPageSelector ) ) {
                        out.markup( "<span class='a-GV-pageSelector'></span>" );
                    } else if ( o.pagination.showRange ) {
                        pageRange();
                        rangeAdded = true;
                    }

                    if ( !o.pagination.scroll ) {
                        this._renderButton( out, buttonClass + "js-pg-next", "icon-next",
                            getMessage( "NEXT_PAGE" ) );
                        if ( modelHasTotal && o.pagination.firstAndLastButtons ) {
                            this._renderButton( out, buttonClass + "js-pg-last", "icon-last",
                                getMessage( "LAST_PAGE" ) );
                        }
                    }

                    if ( !rangeAdded && o.pagination.showRange ) {
                        pageRange();
                    }

                }
                out.markup( "</div></div>" );
            }

        },

        /**
         * Call when the view is being refreshed before it is rendered.
         * @param scrollParent$
         * @private
         */
        _refreshPagination: function( scrollParent$ ) {
            let o = this.options,
                pagination = o.pagination;

            this.hasScrollFillers = pagination.scroll && pagination.virtual && !pagination.loadMore;

            // preserve scroll offsets
            this.lastScrollTop = null;
            this.lastScrollLeft = null;
            if ( scrollParent$.length ) {
                if ( !o.hasSize ) {
                    scrollParent$ = o.scrollParentOverride ? $( o.scrollParentOverride ) : $( window );
                }
                this.lastScrollLeft = scrollParent$.scrollLeft();
                this.lastScrollTop = scrollParent$.scrollTop();
            }
        },

        /**
         * Return a opaque indication of what in the pagination area has focus or false if nothing has focus.
         * Derived widgets use this method and _restorePaginationFocus to maintain focus in pagination area after view
         * or pagination area has been re-rendered.
         * @private
         */
        _paginationHasFocus: function() {
            var m,
                activeElement$ = $( document.activeElement );

            if ( activeElement$.closest( "." + C_GRID_FOOTER ).length > 0 ) {
                if ( activeElement$.parent().hasClass( "a-GV-pageSelector-item" ) ) {
                    return activeElement$.parent().attr( DATA_PAGE );
                } else {
                    m = /js-pg-\S+/.exec( activeElement$.attr( "class" ) );
                    if ( m ) {
                        return "." + m[0];
                    }
                }
            }
            return false;
        },

        /**
         * Call after the view has been refreshed to restore the focus in the pagination area if it was there.
         *
         * @param prevFocus the return value from _paginationHasFocus.
         * @private
         */
        _restorePaginationFocus: function( prevFocus ) {
            if ( prevFocus ) {
                let btn$,
                    sel = prevFocus;

                if ( sel >= 0 ) {
                    sel = `[${DATA_PAGE}='${prevFocus}'] button`;
                }
                btn$ = this.footer$.find( sel );
                if ( btn$[0] && btn$[0].disabled ) {
                    btn$ = this.footer$.find( ":focusable" ).eq( 0 );
                }
                btn$.focus();
            }
        },

        /**
         * Call after the footer (and the rest of the view) has been rendered and inserted into the DOM.
         * @private
         */
        _initPagination: function( header$, scrollParent$ ) {
            let footer$,
                o = this.options,
                ctrl$ = this.element,
                tooltipEdge = ctrl$.hasClass("u-RTL") ? "right" : "left";

            this.pageKey = null; // cause the pagination if any to get updated

            footer$ = this.footer$ = ctrl$.find( "." + C_GRID_FOOTER );
            this.status$ = footer$.find( ".a-GV-status" );
            this.stateIcons$ = footer$.find( ".a-GV-stateIcons" );
            this.pageRange$ = footer$.find( ".a-GV-pageRange" );
            this.pageSelector$ = footer$.find( ".a-GV-pageSelector" );
            this.noData$ = ctrl$.find( "." + C_GRID_NO_DATA );
            this.moreData$ = ctrl$.find( "." + C_GRID_MORE_DATA );

            if ( $.ui.tooltip ) {
                let messages = this.stateIconMessages;
                this.stateIcons$.tooltip( {
                    content: function() { // can't use arrow function here because this is the item element
                        let name = $( this ).attr( DATA_STATE );
                        return messages[name];
                    },
                    items: `[${DATA_STATE}]`,
                    show: apex.tooltipManager.defaultShowOption(),
                    tooltipClass: "a-GV-tooltip",
                    position: {
                        my: tooltipEdge + " bottom",
                        at: tooltipEdge + " top-10",
                        collision: "flipfit"
                    }
                });
            }

            this._updateScrollHandler( scrollParent$ );

            if ( o.pagination.scroll ) {
                this.pageOffset = 0;
                this.highWaterMark = 0;
                this.pageCount = 0;
                this.totalRenderedRecords = 0;
            }

            // if the region has no defined height and the stickyWidget is available then stick the
            // header to the top of the page and/or the footer to the bottom of the page
            if ( !o.hasSize && $.apex.stickyWidget ) {
                if ( o.stickyTop ) {
                    let swOpt = {
                        stickToEnd: true,
                        toggleWidth: true,
                        bottom: () => {
                            return ctrl$.offset().top + ctrl$.outerHeight() - this._footerHeight();
                        }
                    };
                    if ( typeof o.stickyTop === "function" ) {
                        swOpt.top = o.stickyTop;
                    }
                    // overflow hidden so that header controls don't cause horizontal scrollbar. (case 2 of bug 26171679)
                    header$.css( "overflow", "hidden" ).stickyWidget( swOpt );
                }
                if ( o.footer && o.stickyFooter ) {
                    footer$.stickyWidget( {
                        isFooter: true,
                        toggleWidth: true,
                        stickToEnd: true,
                        top: () => {
                            return ctrl$.offset().top + this._getHeaderHeight();
                        }
                    } );
                }
            }

            if ( o.pagination.showPageLinks ) {
                this.pageSelector$.click( event => {
                    let item$ = $( event.target ).parent();
                    if ( item$.length ) {
                        this.gotoPage( toInteger( item$.attr( DATA_PAGE ) ) );
                    }
                    event.preventDefault();
                } );
            } else if ( o.pagination.showPageSelector ) {
                this.pageSelector$.change( event => {
                    this.gotoPage( toInteger( $( event.target ).val() ) );
                } );
            }

            footer$.find( ".js-pg-first" ).click( () => {
                this.firstPage();
            } );
            footer$.find( ".js-pg-prev" ).click( () => {
                this.previousPage();
            } );
            footer$.find( ".js-pg-next" ).click( () => {
                this.nextPage();
            } );
            footer$.find( ".js-pg-last" ).click( () => {
                this.lastPage();
            } );
            ctrl$.find( ".js-load" ).click( () => {
                this.loadMore();
            } );
        },

        /**
         * Call during resizing or anytime you need to make adjustments for the footer height
         * @return {number}
         * @private
         */
        _footerHeight: function() {
            return this.footer$.outerHeight() || 0;
        },

        /**
         * Call to add the next page
         * Only for scroll pagination!
         * @private
         */
        _addNextPage: function() {
            if ( this.renderInProgress ) {
                return false;
            } // else
            let total = this.model.getTotalRecords();

            if ( total < 0 || this.pageOffset + this.pageCount < total ) {
                this.pageOffset += this.pageCount; // pageSize can change so use pageCount see bug 28834484.
                this._addPageOfRecords();
            }
            return true;
        },

        /**
         * Call to add records from the model to the view
         * @private
         */
        _addPageOfRecords: function( callback ) {
            let count, pageSize, pageOffset, dataContainer$, data$, insertControlBreak,
                // startTime, // uncomment for performance timing
                hasControlBreaks = this._hasControlBreaks(),
                hasTotalRecords = modelHasTotalRecords( this.model ),
                self = this,
                o = this.options,
                out = this._getDataRenderContext();

            function updateFiller( start, end, f$, recPerRow, rowHeight) {
                if ( start >= end ) {
                    // updateFiller is done before inserting new data in an attempt to not disturb the scroll offset too much
                    // but the filler can't be removed before inserting so just indicated to remove empty filler
                    return true;
                } else {
                    // adjust filler size
                    f$.attr( DATA_START, start );
                    f$.attr( DATA_END, end );
                    setFillerHeight( f$, start, end, recPerRow, rowHeight );
                    return false;
                }
            }

            function getItemsToRemove( items$, begin, end, skip ) {
                let a = skip || 0,
                    b = end - begin;

                if ( hasControlBreaks ) {
                    // when there are control breaks have to filter all the items
                    // because the break columns don't count as report rows but they do take up space in the DOM
                    a = begin + a;
                    b = end;
                    // expand the slice range to filter
                    begin = 0;
                    end += items$.filter( ".a-GV-controlBreak" ).length; // xxx todo need a way for derived widget to specify control break class currently hard coded to use GV class (applies to all uses of a-GV-controlBreak)
                }

                return items$.slice( begin, end ).filter( ( i, el ) => {
                    let el$ = $(el);

                    if ( el$.hasClass( "a-GV-controlBreak" ) ) {
                        // for each break item the index range shifts
                        if ( i < a) {
                            a += 1;
                        }
                        b += 1;
                    }
                    if ( el$.hasClass( C_GRID_SCROLL_FILLER ) ) {
                        b = i;
                    }
                    return i >= a && i < b;
                } );
            }

            function nextVisible( filler$, removeOffset, next ) {
                let resultOffset = removeOffset,
                    row$ = dataContainer$.find( `[${DATA_ROWNUM}="${removeOffset + 1}"]` );

                if ( ( filler$ && !fillerVisible( filler$ ) ) || !row$.is( SEL_VISIBLE ) ) {
                    // todo This is grid view specific because of a-GV-row!!! must be general but currently grid view is the only one with collapsible breaks
                    resultOffset = toInteger( row$[next ? "nextAll" : "prevAll"]( `.a-GV-row[${DATA_ROWNUM}]:visible` ).first().attr( DATA_ROWNUM ) ) || 0;
                }
                return resultOffset;
            }

            function finish( error ) {
                let fillers$, outFiller, totalEnd, recPerRow, prevEnd, refItem$, prevTop,
                    rowHeight = self.avgRowHeight;

                // check if empty
                self.pageCount = count;
                self.totalRenderedRecords += count;
                if ( self.pageOffset + count > self.highWaterMark ) {
                    self.highWaterMark = self.pageOffset + count;
                }
                if ( self.pageOffset === 0 && count === 0 ) {
                    self.noData = true;
                    data$.hide();
                    self.moreData$.hide();
                    self.noData$.show();
                    self._trigger( EVENT_PAGE_CHANGE, null, {
                        offset: 0,
                        count: 0
                    } );
                    if ( o.autoAddRecord && self.model.allowAdd() && !error ) {
                        self._autoAddRecord();
                    }
                } else {
                    if ( count === 0 ) {
                        self.renderInProgress = false;
                        if ( !o.pagination.scroll ) {
                            // page offset is not zero and yet count is zero so somehow went off the end of the model xxx or there was an error
                            self.pageOffset -= self.pageSize; // go back one page
                            if ( self.pageOffset < 0 ) {
                                self.pageOffset = 0;
                            }
                            self._addPageOfRecords(); // and try again. todo consider loss of callback
                        }
                        return;
                    }
                    self.noData = false;
                    self.noData$.hide();
                    if ( self.model.getDataOverflow() && o.moreDataMessage ) {
                        self.moreData$.show();
                        // todo consider if want to show the data under the more data warning message
                        data$.hide();
                    } else {
                        self.moreData$.hide();
                        data$.show();
                    }
                    if ( self.hasScrollFillers ) {
                        // The table consists of actual rows and filler rows. Filer rows have a height that represents
                        // the height of the not yet rendered rows.
                        // Find all the filler rows
                        fillers$ = dataContainer$.find( SEL_GRID_SCROLL_FILLER );
                        totalEnd = self.model.getTotalRecords( true ) - 1;
                        prevEnd = totalEnd;
                        if ( fillers$.length === 0 && hasTotalRecords ) {
                            // if there are none it must mean that the view is empty (and doing full virtual) so add
                            // initial filler record that represents all the data
                            outFiller = self._getDataRenderContext();
                            self._renderFillerRecord( outFiller, C_GRID_SCROLL_FILLER );
                            fillers$ = self._insertFiller( outFiller, null );
                            fillers$.attr(DATA_START, 0);
                            fillers$.attr(DATA_END, totalEnd);
                            fillers$ = fillers$.last();
                        } else if ( fillers$.last().next().length === 0 ) {
                            // if there is a filler at the end then it has the previous totalEnd records
                            prevEnd = toInteger( fillers$.last().attr(DATA_END) );
                        }
                        if ( totalEnd !== prevEnd ) {
                            fillers$.last().attr( DATA_END, totalEnd );
                        }
                        recPerRow = self._getRecordsPerRow();
                        // figure out where to insert the rendered data
                        let fillerIndex = -1,
                            found = false;

                        fillers$.each( function( i ) {
                            let newFiller$,
                                remove = false,
                                f$ = $( this ),
                                start = toInteger( f$.last().attr( DATA_START ) ),
                                end = toInteger( f$.last().attr( DATA_END ) );

                            // because of logic in _scrollPage the pageOffset should be start or somewhere between start and end.
                            if ( pageOffset === start ) {
                                // insert just before the filler
                                remove = updateFiller( pageOffset + count, end, f$, recPerRow, rowHeight );
                                self._insertData( out, pageOffset, count, f$, INSERT_BEFORE );
                                fillerIndex = i;
                                found = true;
                            } else if ( pageOffset + count < end ) {
                                // the new rows go in the middle of the filler so split it by adding a new filler
                                // before this one and then insert the rows just before this filler
                                // first adjust the size of the current filter Note: start should not equal end
                                updateFiller( pageOffset + count, end, f$, recPerRow, rowHeight );
                                outFiller = self._getDataRenderContext();
                                self._renderFillerRecord( outFiller, C_GRID_SCROLL_FILLER );
                                newFiller$ = self._insertFiller( outFiller, f$ );
                                updateFiller( start, pageOffset - 1, newFiller$, recPerRow, rowHeight );
                                self._insertData( out, pageOffset, count, f$, INSERT_BEFORE );
                                fillerIndex = i + 1;
                                found = true;
                            } else if ( pageOffset <= end && pageOffset + count > end )  {
                                // insert just after the filler
                                remove = updateFiller( start, pageOffset - 1, f$, recPerRow, rowHeight );
                                self._insertData( out, pageOffset, count, f$, INSERT_AFTER );
                                fillerIndex = i + 1;
                                found = true;
                            }
                            if ( found ) {
                                if ( remove ) {
                                    if ( fillerIndex === i ) {
                                        fillerIndex = i + 1;
                                    }
                                    f$.remove();
                                }
                                return false; // exit each early
                            }
                        } );
                        if ( !found ) {
                            // if the data doesn't replace a filler then it goes at the end
                            self._insertData( out, pageOffset, count );
                            fillerIndex = fillers$.length;
                        }
                        // check if records should be removed and replaced with a filler
                        // Want to leave some extra records rendered in pagesize chunks. The current viewport plus
                        // 2 pages before and 2 after = 5 + 2 at the beginning and 1 at the end totals 8 pages
                        if ( self.totalRenderedRecords > 8 * self.pageSize ) { // todo consider other criteria such as > 999 rows
                            let start, end, show, toRemove$, removeOffset, mergeFiller$, lastFiller,
                                distance = 0,
                                newFiller$ = null,
                                removeCount = 0,
                                firstFiller = 0,
                                addFiller = false;

                            if ( !o.fixedRowHeight ) {
                                refItem$ = dataContainer$.find( `[${DATA_ROWNUM}="${self.pageOffset + 1}"]` );
                                if ( refItem$[0] ) {
                                    prevTop = refItem$.offset().top;
                                }
                            }
                            // the set of fillers may have changed
                            fillers$ = dataContainer$.find( SEL_GRID_SCROLL_FILLER );
                            lastFiller = fillers$.length - 1;

                            // first try to remove from the beginning
                            if ( firstFiller >= fillerIndex ) {
                                firstFiller = -1;
                                removeOffset = 2 * self.pageSize;
                            } else {
                                newFiller$ = fillers$.eq( firstFiller );
                                // if there are multiple adjacent fillers (due to different visibility) want the last one
                                while ( newFiller$.next().is( SEL_GRID_SCROLL_FILLER ) ) {
                                    newFiller$ = newFiller$.next();
                                }
                                removeOffset = toInteger( newFiller$.attr( DATA_END ) );
                            }
                            if ( removeOffset > 0 ) {
                                // check distance between place to remove from and current page
                                // note because of collapsed breaks can't just look at the removeOffset but must consider
                                // the next closest visible row.
                                distance = Math.abs( self.pageOffset - nextVisible( newFiller$, removeOffset + 1, true ) );
                            }
                            if ( distance >= 3 * self.pageSize ) {
                                /*
                                 * possible cases
                                 * - the rows to remove leave no rows between 2 fillers so merge them
                                 * - the rows to remove are next to a filler so just extend the filler
                                 *    - if the rows to remove and the adjacent filler have different visibility then add a new filler
                                 * - the rows are not next to a filler so add a new filler
                                 */
                                if ( firstFiller >= 0 ) {
                                    // remove the next page size records but don't run into a new filler
                                    toRemove$ = getItemsToRemove( newFiller$.nextAll(), 0, self.pageSize );
                                    removeCount = toRemove$.not( ".a-GV-controlBreak" ).length;
                                    // consider if the rows to remove are hidden because of collapsed control break
                                    // by checking only the start and end for visibility the fillers only approximately
                                    // align with collapsed control breaks
                                    show = toRemove$.first().is( SEL_VISIBLE ) && toRemove$.last().is( SEL_VISIBLE );
                                    //console.log("xxx to remove from start ", distance, removeCount, show)
                                    mergeFiller$ = toRemove$.last().next().filter( SEL_GRID_SCROLL_FILLER );
                                    if ( mergeFiller$.length && show === fillerVisible( mergeFiller$ ) ) {
                                        // merge adjacent fillers
                                        start = toInteger( newFiller$.attr( DATA_START ) );
                                        end = toInteger( mergeFiller$.attr( DATA_END ) );
                                        updateFiller( start, end, mergeFiller$, recPerRow, rowHeight );
                                        newFiller$.remove();
                                    } else if ( show === fillerVisible( newFiller$ ) ) {
                                        // extend filler
                                        start = toInteger( newFiller$.attr( DATA_START ) );
                                        end = toInteger( newFiller$.attr( DATA_END ) ) + removeCount;
                                        updateFiller( start, end, newFiller$, recPerRow, rowHeight );
                                    } else {
                                        start = toInteger( newFiller$.attr( DATA_END ) ) + 1;
                                        addFiller = true;
                                    }
                                } else {
                                    start = removeOffset;
                                    end = start + self.pageSize;
                                    toRemove$ = getItemsToRemove( dataContainer$.children(), start, end );
                                    removeCount = toRemove$.not( ".a-GV-controlBreak" ).length;
                                    show = toRemove$.first().is( SEL_VISIBLE ) && toRemove$.last().is( SEL_VISIBLE );
                                    //console.log("xxx to remove from start add new ", distance, removeCount, show)
                                    addFiller = true;
                                }
                                if ( addFiller ) {
                                    outFiller = self._getDataRenderContext();
                                    self._renderFillerRecord( outFiller, C_GRID_SCROLL_FILLER );
                                    newFiller$ = self._insertFiller( outFiller, toRemove$.eq( 0 ) );
                                    updateFiller( start, start + removeCount - 1, newFiller$, recPerRow, rowHeight );
                                    toggleFillerVisible( newFiller$, show );
                                }
                            } else {
                                let last$;

                                distance = 0;
                                // next try to remove from the end
                                if ( lastFiller < fillerIndex ) {
                                    lastFiller = -1;
                                    removeOffset = -1;
                                    // may not have rendered all the way to the end so really want the offset of the last rendered record
                                    last$ = dataContainer$.children().not( SEL_AGGREGATE ).last(); // don't include aggregate records because they don't have a rownum
                                    if ( last$.attr( DATA_ID ) ) {
                                        removeOffset = self.model.indexOf( self.model.getRecord( last$.attr( DATA_ID ) ) );
                                        // leave just one page rendered at the end so that more rows are available to render in the middle
                                        removeOffset = pageBoundary( removeOffset, self.pageSize ) - self.pageSize;
                                    }
                                } else {
                                    newFiller$ = fillers$.eq( lastFiller );
                                    // if there are multiple adjacent fillers (due to different visibility) want the first one
                                    while ( newFiller$.prev().is( SEL_GRID_SCROLL_FILLER ) ) {
                                        newFiller$ = newFiller$.prev();
                                    }
                                    removeOffset = toInteger( newFiller$.attr( DATA_START ) );
                                }
                                if ( removeOffset > 0 ) {
                                    // check distance between place to remove from and current page
                                    distance = Math.abs( self.pageOffset + self.pageSize - nextVisible( newFiller$, removeOffset - 1, false ) );
                                }
                                if ( distance >= 3 * self.pageSize ) {
                                    // same possible cases as removing from the start
                                    if ( lastFiller >= 0 ) {
                                        // remove the next page size records but don't run into a new filler
                                        toRemove$ = getItemsToRemove( newFiller$.prevAll(), 0, self.pageSize );
                                        removeCount = toRemove$.not( ".a-GV-controlBreak" ).length;
                                        show = toRemove$.first().is( SEL_VISIBLE ) && toRemove$.last().is( SEL_VISIBLE );
                                        //console.log("xxx to remove from end ", distance, removeCount, show)
                                        mergeFiller$ = toRemove$.prev( SEL_GRID_SCROLL_FILLER );
                                        if ( mergeFiller$.length && show === fillerVisible( mergeFiller$ ) ) {
                                            // merge adjacent fillers
                                            start = toInteger( mergeFiller$.attr( DATA_START ) );
                                            end = toInteger( newFiller$.attr( DATA_END ) );
                                            updateFiller( start, end, mergeFiller$, recPerRow, rowHeight );
                                            newFiller$.remove();
                                        } else if ( show === fillerVisible( newFiller$ ) ) {
                                            // extend filler
                                            start = toInteger( newFiller$.attr( DATA_START ) ) - removeCount;
                                            end = toInteger( newFiller$.attr( DATA_END ) );
                                            updateFiller( start, end, newFiller$, recPerRow, rowHeight );
                                        } else {
                                            start = toInteger( newFiller$.attr( DATA_START ) ) - removeCount;
                                            end = start + removeCount - 1;
                                            addFiller = true;
                                        }
                                    } else {
                                        start = totalEnd - removeOffset;
                                        end = start + self.pageSize;
                                        toRemove$ = getItemsToRemove( last$.prevAll(), 0, end, start );
                                        removeCount = toRemove$.not( ".a-GV-controlBreak" ).length;
                                        if ( removeCount > 0 ) {
                                            show = toRemove$.first().is( SEL_VISIBLE ) && toRemove$.last().is( SEL_VISIBLE );
                                            //console.log( "xxx to remove from end add new ", distance, removeCount, show, last$ )
                                            addFiller = true;
                                            end = removeOffset - 1;
                                            start = removeOffset - removeCount;
                                        }
                                    }
                                    if ( addFiller ) {
                                        outFiller = self._getDataRenderContext();
                                        self._renderFillerRecord( outFiller, C_GRID_SCROLL_FILLER );
                                        newFiller$ = self._insertFiller( outFiller, toRemove$.eq( 0 ) );
                                        updateFiller( start, end, newFiller$, recPerRow, rowHeight );
                                        toggleFillerVisible( newFiller$, show );
                                    }
                                }
                            }
                            if ( removeCount > 0 ) {
                                toRemove$.remove(); // todo give derived view a chance to remove
                                self.totalRenderedRecords -= removeCount;
                            }
                        }
                        // With variable height rows, after rendering more data the average height may have changed
                        // so in the case of virtual scrolling adjust the heights of fillers to be more accurate
                        if ( !o.fixedRowHeight ) {
                            let delta = 0;

                            // todo think this gets done a second time when heading fixed to page because the region resizes which ends up calling _initPageSize where updateFillerHeights is done again
                            self._updateAvgRowHeight(); // after this rowHeight is out of date
                            updateFillerHeights( self._getDataContainer(), recPerRow, self.avgRowHeight );
                            if ( refItem$ && refItem$[0] ) {
                                delta = refItem$.offset().top - prevTop;
                            }
                            if ( delta !== 0 ) {
                                self.scrollParent$.scrollTop( self.scrollParent$.scrollTop() + delta );
                            }
                        }
                    } else {
                        // for non-scroll paging or (non-virtual scroll paging) just append the new records.
                        // In the first case the container is already empty. In the second case "load more" adds
                        // to the existing records.
                        self._insertData( out, pageOffset, count );
                    }
                }
                self._updateTotalRecords();
                self.renderInProgress = false;

                // begin uncomment for performance timing
                //debug.timeEnd("TMVforEachInPage");
                //debug.info("Render page of records duration:", performance.now() - startTime);
                // end uncomment for performance timing

                // There are 2 cases where after getting a page of records we may need to get another page both
                // have to do with scroll paging and collapsible control breaks
                // 1) If data is added to a control break that is collapsed want to keep going until the next
                //    control break is found because collapsed data is not seen.
                // 2) If the report height is less than the scroll viewport height (because of a control break
                //    getting collapsed) need to get more because scrolling is not possible.
                // there must be data (bug 33336777)
                if ( !self.noData && o.pagination.scroll && ( self.controlBreakCollapsed ||
                        !hasTotalRecords && dataContainer$.height() < self.scrollParent$.height() ) ) {
                    let total = self.model.getTotalRecords(),
                        nextOffset = self.hasScrollFillers ? self.pageOffset + self.pageCount : self.highWaterMark;

                    if ( total < 0 || nextOffset < total ) {
                        let needData = true;

                        self.pageOffset = nextOffset;
                        if ( self.hasScrollFillers ) {
                            needData = false;
                            // make sure it is not data we already have; find a filler where it belongs
                            fillers$ = dataContainer$.find( SEL_GRID_SCROLL_FILLER );
                            fillers$.each( function () {
                                if ( self.pageOffset === toInteger( $( this ).attr( DATA_START ) ) ) {
                                    needData = true;
                                    return false; // no need to keep looking
                                }
                            } );
                        }
                        if ( needData ) {
                            self._addPageOfRecords( callback );
                            return;
                        }
                    }
                }

                if ( callback ) {
                    callback();
                }
            }

            if ( this.renderInProgress ) {
                return;
            }

            data$ = self.noData$.parent().children().not( SEL_GRID_ALT_MSG ); // this is all but the alternative message elements
            if ( this.noData || this.moreData$.is( SEL_VISIBLE ) ) {
                // assume there will be data and show the data areas so the content can be sized correctly
                this.noData$.hide();
                this.moreData$.hide();
                data$.show();
                this.resize();
            }
            // need to know where the data is going to go to be able to search for filler and control break elements
            dataContainer$ = self._getDataContainer();

            if ( this.pageOffset === 0 || !o.pagination.scroll ) {
                this.pageFirstOffset = "";
                this.pageLastOffset = "";
            }
            if ( !o.pagination.scroll ) {
                this.totalRenderedRecords = 0;
            }

            // if have control break start out unknown else false for NA.
            this.controlBreakCollapsed = hasControlBreaks ? null : false;

            if ( hasControlBreaks ) {
                if ( this.pageOffset === 0 || !o.pagination.scroll ) {
                    // if at the beginning or for page pagination always start with a control break
                    insertControlBreak = true;
                } else {
                    let metaKey = this.model.getOption( "metaField" );

                    if ( metaKey ) {
                        // look at record just before the ones about to render
                        let index = this.pageOffset - 1;

                        while ( index > 0 ) {
                            let meta,
                                rowItem = this.model.recordAt( index );

                            if ( rowItem ) {
                                meta = this.model.getValue( rowItem, metaKey );
                                // skip aggregate records when looking for endControlBreak. See bug 33384701.
                                if ( !meta.agg ) {
                                    insertControlBreak = meta.endControlBreak === true;
                                    break;
                                }
                            }
                            index -= 1;
                        }
                    }
                }
            }

            count = 0;
            pageSize = this.pageSize;
            if ( !o.pagination.scroll ) {
                // the page size could have changed need to get back on the new page size boundary
                this.pageOffset = pageBoundary( this.pageOffset, pageSize );
            } else if ( self.hasScrollFillers ) {
                // if pagination is scroll make sure count doesn't go off end of current filler
                let fillers$ = dataContainer$.find( SEL_GRID_SCROLL_FILLER );

                fillers$.each( function() {
                    let f$ = $( this ),
                        start = toInteger( f$.attr( DATA_START ) ),
                        end = toInteger( f$.attr( DATA_END ) );

                    if ( self.pageOffset >= start && self.pageOffset < end ) {
                        // trim page size to keep it within filler otherwise end up with duplicate records rendered
                        pageSize = Math.min( pageSize, end + 1 - self.pageOffset );
                        return false; // no need to keep looking
                    }
                });
            }
            pageOffset = this.pageOffset;

            // begin uncomment for performance timing
            //debug.timeBegin("TMVforEachInPage");
            //startTime = performance.now();
            // end uncomment for performance timing

            this.renderInProgress = true;
            this.model.forEachInPage( this.pageOffset, pageSize, function( rowItem, index, id, error ) {
                if ( rowItem ) {
                    let expandControl, currentBreakData,
                        meta = id ? this.model.getRecordMetadata( id ) : {},
                        serverOffset = meta.serverOffset;

                    if ( hasControlBreaks && !meta.agg ) {
                        expandControl = true; // todo not currently used, waiting for option to control initial state
                                              // this will impact virtual scrolling. setting expandControl to false causes all rows to be rendered as hidden all at once; not desired
                        if ( insertControlBreak ) {
                            insertControlBreak = false;
                            currentBreakData = this._controlBreakLabel( rowItem );
                            this._renderBreakRecord( out, expandControl, currentBreakData, serverOffset );
                        }
                        if ( meta.endControlBreak ) {
                            insertControlBreak = true;
                        }
                    }
                    this._renderRecord( out, rowItem, index, id, meta );
                    if ( serverOffset !== undefined ) {
                        serverOffset += 1;
                        if ( this.pageFirstOffset === "" ) {
                            this.pageFirstOffset = serverOffset;
                        }
                        if ( serverOffset > this.pageLastOffset ) {
                            this.pageLastOffset = serverOffset;
                        }
                    }
                    count += 1;
                }
                if ( count === pageSize || !rowItem || error ) {
                    finish( error );
                }
            }, this );
        },

        /**
         * Call when widget is created and anytime the size changes
         *
         * @private
         */
        _initPageSize: function() {
            let rowHeight, viewportHeight, pageSize, top,
                o = this.options,
                recPerRow = this._getRecordsPerRow();

            if ( this.hasScrollFillers ) {
                // number of records per row may have changed so first update fillers based
                // on new rows per page but with current row height
                if ( this.recPerRow !== recPerRow ) {
                    // adjust the height of the filler items
                    updateFillerHeights( this._getDataContainer(), recPerRow, this.avgRowHeight );
                }
                // just in case the scroll offset changed since last initialized page size
                this._scrollPage();
            }
            this.recPerRow = recPerRow;

            this._updateAvgRowHeight();
            rowHeight = this.avgRowHeight;
            if ( !o.hasSize ) {
                viewportHeight = this.scrollParent$.height();
                // if there is a sticky header or footer (even if not currently stuck) subtract the height of each
                if ( this.footer$.hasClass( "js-stickyWidget-toggle" ) ) {
                    viewportHeight -= this._footerHeight();
                }
                top = this._getStickyTop();
                if ( top > 0 ) {
                    viewportHeight -= top + this._getHeaderHeight();
                }
            } else {
                viewportHeight = this.scrollParent$[0].offsetHeight; // doesn't include the scroll bar
            }
            // always leave room for possible horizontal scroll bar
            viewportHeight -= util.getScrollbarSize().height;
            pageSize = Math.floor( viewportHeight / rowHeight );
            if ( pageSize < 1 ) {
                pageSize = 1; // don't ever let page size be 0
            }
            this.viewportHeight = pageSize * rowHeight; // adjust to be a multiple of row height

            // if the user has specified how many rows make a page use it otherwise ...
            if ( !o.rowsPerPage ) {
                if ( !o.fixedRowHeight && !o.pagination.scroll ) {
                    // for traditional paging the "auto" setting for rows per page doesn't make sense for variable
                    // height rows (can't have page size keep shifting) so auto is VAR_HEIGHT_AUTO_PAGE_SIZE.
                    this.pageSize = VAR_HEIGHT_AUTO_PAGE_SIZE;
                } else {
                    // Note for a grid (of items) view, rows per page really means records (or items) per page because
                    // multiple records fit on one row
                    pageSize = pageSize * recPerRow;

                    // if scroll paging want the page size to be a little bigger than what is visible (keep in mind that when
                    // scroll paging rowsPerPage is forced to null) otherwise the "auto" page size is just what fits
                    // in the view
                    if ( o.pagination.scroll ) {
                        this.pageSize = Math.max( MIN_SCROLL_PAGE_SIZE, 2 * pageSize );
                        // make a multiple of recPerRow
                        this.pageSize = Math.ceil( this.pageSize / recPerRow ) * recPerRow;
                    } else {
                        this.pageSize = pageSize;
                    }
                }
            }
        },

        /**
         * Call any time the selection state changes or the deleted state of any records may have changed or
         * the hideDeletedRows option changes, or any other case that could affect the status area of the footer.
         * @private
         */
        _updateStatus: function() {
            var deleteCount,
                selCount = this._selectedCount(),
                text = "";

            if ( this.options.hideDeletedRows ) {
                deleteCount = this._deletedCount();
            }
            if ( selCount > 0 ) {
                text += lang.format( this._selectedStatusMessage(), selCount );
            }
            if ( deleteCount > 0 ) {
                if ( text ) {
                    text += ", ";
                }
                text += formatMessage( "DELETED_COUNT", deleteCount ) + " ";
            }
            this.status$.text( text );
        },

        /**
         * Call to update the model with a new value from a column item
         * @param [element$] the active element (an ancestor of the column item element)
         * @param columnItem the column item
         * @param record the model record
         * @param property the column/field/property of the model record
         * @param {boolean} [notify]
         * @private
         */
        _setModelValue: function( element$, columnItem, record, property, notify ) {
            var result, value, prevValue, validity, id, prevId;

            if ( element$ ) {
                element$.removeClass( C_ACTIVE );
            }
            value = columnItem.getValue();
            if ( !notify ) {
                this.ignoreFieldChange = property; // ignore the update that this setValue *may* cause
            }
            prevId = this.model.getRecordId( record );
            prevValue = this.model.getValue( record, property );
            if ( prevValue !== null && typeof prevValue === "object" && hasOwnProperty( prevValue,"v" ) ) {
                value = {
                    v: value,
                    d: columnItem.displayValueFor( value )
                };
            }
            result = this.model.setValue( record, property, value );
            if ( result === "DUP" ) {
                this._setColumnItemValue( columnItem, record, property );
                apex.message.alert( getMessage( "DUP_REC_ID" ) );
            } else {
                validity = columnItem.getValidity();
                id = this.model.getRecordId( record );
                if ( !validity.valid ) {
                    this.model.setValidity( "error", id, property, columnItem.getValidationMessage() );
                } else {
                    this.model.setValidity( "valid", id, property );
                }
                if ( id !== prevId ) {
                    this._identityChanged( prevId, id );
                }
            }
            if ( !notify ) {
                this.ignoreFieldChange = null;
            }
        },

        _initColumnItems: function( fields ) {
            let allFromCache = true;

            this.columnItems = {};
            this.columnItemsContainer$ = null;
            this.asyncColumnItems = [];

            for ( let i = 0; i < fields.length; i++ ) {
                let curItem,
                    column = fields[i],
                    eid = column.elementId,
                    ci = null;

                if ( eid ) {
                    // Because multiple widgets can share the same column items we use a cache to avoid having
                    // to recreate the items multiple times.
                    // first check if in cache
                    if ( gColumnItemCache[eid] ) {
                        ci = this.columnItems[column.property] = gColumnItemCache[eid];
                        curItem = ci.item;
                    } else {
                        // if not create a column item and save in cache
                        curItem = item( eid );

                        if ( curItem.node ) { // make sure the item really exists
                            allFromCache = false;
                            gColumnItemCache[eid] = ci = this.columnItems[column.property] = {
                                element$: curItem.element.closest( ".a-GV-columnItem" ),
                                item: curItem
                            };
                            // keep track of any items that may take a little longer to load
                            if ( curItem.whenReady ) {
                                this.asyncColumnItems.push( curItem.whenReady() );
                            }
                        }
                    }
                    if ( ci && !this.columnItemsContainer$ ) {
                        this.columnItemsContainer$ = ci.element$.parent();
                    }
                }
            }
            if ( !this.columnItemsContainer$ ) {
                if ( this.editable ) {
                    debug.error( `An editable ${this.widgetName} must have at least one column with a column item.` );
                }
            } else {
                if ( !allFromCache ) {
                    // take the hidden off-screen column items out of the tab order
                    this.columnItemsContainer$.find( SEL_TABBABLE ).addClass( C_JS_TABBABLE ).attr( "tabindex", -1);
                }
            }
        },

        _waitForColumnItems: function() {
            return $.when( ...this.asyncColumnItems );
        },

        // must only be used on an editable widget
        _activateColumnItem: function( columnItem, labelId ) {
            var el$ = columnItem.element$;
            // deactivate marks the item's tabbable elements, this makes them tabbable again
            el$.find( SEL_JS_TABBABLE ).attr( "tabindex", 0 ).removeClass( C_JS_TABBABLE );
            if ( labelId ) {
                // todo should be a better way to find the element that is to be labeled
                el$.find( SEL_TABBABLE ).first()
                    .attr( A_LBL_BY, labelId );
            }
        },

        // must only be used on an editable widget
        _deactivateColumnItem: function( columnItem ) {
            var el$ = columnItem.element$,
                tabs$ = el$.find( SEL_TABBABLE );
            tabs$.first().removeAttr( A_LBL_BY );
            this.columnItemsContainer$.append( el$ ); // return it to the off screen container
            // take it out of the tab order: make the item's tabbable elements focusable (if it was visible) and remember them
            tabs$.addClass( C_JS_TABBABLE ).attr( "tabindex", -1);
        },

        /*
        * This supports the derived widget having these column fields
        * linkTargetColumn
        * linkText
        * linkAttributes
        * cellTemplate
        * escape
        * readonly (only for inserted records identity fields)
        * And the widget options supported are:
        * showNullAs, highlighter, highlighterContext
        */
        // todo make a public static version of this in model, uitl, or here
        _renderFieldDataValue: function( out, col, rowItem, meta, cellMeta ) {
            var value, substOptions, columnItem,
                targetUrl = null,
                o = this.options;

            // check if the cell has a target url. It can come from cell metadata or linkTargetColumn; aggregate rows cannot have links
            if ( ( ( cellMeta && cellMeta.url ) || col.linkTargetColumn ) && !meta.agg ) {
                if ( col.linkTargetColumn ) {
                    targetUrl = this.model.getValue( rowItem, col.linkTargetColumn ) || null;
                } else {
                    targetUrl = cellMeta.url;
                }
            }

            value = this.model.getValue( rowItem, col.property );

            // don't show the internally generated primary key unless it is an editable inserted row
            if ( ( meta.agg || ( meta.inserted && col.readonly ) ) && this.model.isIdentityField( col.property ) ) {
                value = "";
            }

            if ( col.linkText || col.linkAttributes || col.cellTemplate ) {
                substOptions = this.atOptions;
                substOptions.model = this.model;
                substOptions.record = rowItem;
            }

            // the anchor wraps the whole cell value
            if ( targetUrl ) {
                out.markup( "<a")
                    .attr( "href", targetUrl );
                if ( col.linkAttributes ) {
                    out.markup( applyTemplate( col.linkAttributes, substOptions ) );
                }
                out.markup( " tabindex='-1'>" );
            }

            if ( targetUrl && col.linkText ) {
                out.markup( applyTemplate( col.linkText, substOptions ) );
            } else if ( col.cellTemplate ) {
                if ( meta.agg ) {
                    // If value is undefined (row actions column) or null just use empty string, otherwise use the value
                    value = value == null ? "" : value;
                    out.content( "" + value );
                } else {
                    value = applyTemplate( col.cellTemplate, substOptions );
                    if ( o.highlighter ) {
                        value = o.highlighter( o.highlighterContext, value, col );
                    }
                    out.markup( value );
                }
            } else {
                if ( ( value === null || value === "" ) && ( o.showNullAs || meta.agg || targetUrl ) ) {
                    if ( meta.agg || meta.inserted || ( cellMeta && ( cellMeta.changed || cellMeta.error || cellMeta.warning ) ) ) {
                        value = "";
                    } else if ( targetUrl ) {
                        // strange to have a link with no text but also don't want to use showNulAs so default to the url
                        value = targetUrl;
                    } else {
                        value = o.showNullAs;
                    }
                    out.content( value );
                } else {
                    if ( value === null ) {
                        value = "";
                    }
                    // check to see if the model has a display value
                    if ( typeof value === "object" && hasOwnProperty( value, "d" ) ) {
                        value = value.d;
                    } else {
                        // otherwise if there is a column item it may have a display value
                        columnItem = this.columnItems[ col.property ];
                        if ( columnItem ) {
                            value = columnItem.item.displayValueFor( value, {
                                readonly: ( cellMeta && !!cellMeta.ck ) || col.readonly || meta.deleted || false,
                                disabled: ( cellMeta && cellMeta.disabled ) || false
                            });
                        }
                    }

                    // if escape is false then value can contain markup otherwise it is escaped as element content
                    // note that the !== false test is so that anything else defaults to true.
                    if ( col.escape !== false ) {
                        value = util.escapeHTML( "" + value );
                    }
                    if ( o.highlighter ) {
                        value = o.highlighter( o.highlighterContext, value, col );
                    }
                    out.markup( value );
                }
            }
            if ( targetUrl ) {
                out.markup( "</a>" );
            }
        },

        _beginSetColumnItems: function() {
            this.reinitCallbacks = [];
        },

        _setColumnItemValue: function( columnItem, record, field, meta ) {
            var cb,
                display = null,
                value = this.model.getValue( record, field );

            if ( value !== null && typeof value === "object" && hasOwnProperty( value, "d" ) ) {
                display = value.d;
                value = value.v;
            }
            if ( this.reinitCallbacks ) {
                // reinit does not cause a change event and if needed returns a function
                // to call after all items have been set
                cb = columnItem.reinit( value, display );
                if ( cb ) {
                    this.reinitCallbacks.push( cb );
                }
            } else {
                // tree this as a normal set, which includes the change event
                columnItem.setValue( value, display );
            }
            // update item disabled state from model metadata if available
            if ( meta ) {
                if ( meta.fields && meta.fields[field] && meta.fields[field].disabled ) {
                    columnItem.disable();
                } else {
                    columnItem.enable();
                }
            }
        },

        _endSetColumnItems: function() {
            var i,
                cbList = this.reinitCallbacks;

            if ( cbList ) {
                for ( i = 0; i < cbList.length; i++ ) {
                    cbList[i]();
                }
            }
            this.reinitCallbacks = null;
        },

        _autoAddRecord: function( after ) {
            var newPK = this.model.insertNewRecord( null, after ); // this may cause a refresh
            this.model.getRecordMetadata( newPK ).autoInserted = true;
        },

        _triggerBeginEditing: function( record, recordId ) {
            this.element.trigger( EVENT_BEGIN_RECORD_EDIT, [{
                model: this.model,
                record: record,
                recordId: recordId
            }]);
        },

        _triggerEndEditing: function( record, recordId ) {
            this.element.trigger( EVENT_END_RECORD_EDIT, [{
                model: this.model,
                record: record,
                recordId: recordId
            }]);
        },

        _updateHighlights: function() {
            let o = this.options,
                sortedHighlights = [],
                styles = "";

            for ( const [i, h] of util.objectEntries( o.highlights ) ) {
                h.id = i;
                sortedHighlights.push(h);
            }
            // highlight style rules must be added in reverse order
            sortedHighlights.sort( ( a, b ) => {
                return b.seq - a.seq;
            } );

            for (let i = 0; i < sortedHighlights.length; i++) {
                let h = sortedHighlights[i];
                if ( !h.cssClass ) {
                    h.cssClass = ( h.row ? "hlr_" : "hlc_" ) + h.id;
                }
                if ( h.color || h.background ) {
                    if ( h.row ) {
                        styles += "." + h.cssClass + " td"; // todo think td doesn't seem right
                    } else {
                        styles += "td." + h.cssClass;
                    }
                    styles += " { ";
                    if ( h.color ) {
                        styles += "color: " + h.color + " !important; ";
                    }
                    if ( h.background ) {
                        styles += "background-color: " + h.background + " !important; ";
                    }
                    styles += "}\n";
                }
            }
            if ( !this.gridStyles$ && styles ) {
                this.gridStyles$ = $( "<style></style>" );
                $( "head" ).append( this.gridStyles$ );
            }
            if ( this.gridStyles$ ) {
                this.gridStyles$ = this.gridStyles$.text( styles );
            }
        },

        /**
         * Internal
         * @private
         */

        _findStateIcon: function ( pStateName ) {
            return this.stateIcons$.find( `[${DATA_STATE}='${pStateName}']` );
        },

        _setOption: function ( key, value ) {
            var o = this.options;

            this._super( key, value );

            if ( key === "noDataMessage" ) {
                this.element.find( SEL_GRID_ALT_MSG + " > " + SEL_GRID_ALT_MSG_TEXT ).text( value );
            } else if ( key === "noDataIcon" ) {
                    this.element.find( SEL_GRID_ALT_MSG + " > ." + C_GRID_ALT_MSG_ICON + " span" ).attr( "class", "a-Icon " + value );
            } else if ( key === "rowsPerPage" ) {
                if ( o.pagination.scroll && !o.pagination.loadMore ) {
                    debug.warn("Ignoring rowsPerPage when scroll pagination.");
                    o.rowsPerPage = null;
                }
                if ( o.rowsPerPage ) {
                    this.pageSize = o.rowsPerPage;
                }
            } else if ( key === "hideDeletedRows" ) {
                this.element.toggleClass( C_GRID_HIDE_DELETED, value );
                if ( value === true ) {
                    // make sure none of the deleted rows are selected
                    this.element.find( SEL_DELETED ).removeClass( C_SELECTED ).removeAttr( A_SELECTED );
                    // todo may need to move last focused too
                }
                this._updateStatus();
            } else if ( key === "scrollParentOverride" ) {
                if ( o.hasSize ) {
                    debug.warn("Ignoring scrollParentOverride when hasSize is true.");
                } else {
                    this._updateScrollHandler();
                }
            } else if ( key === "applyTemplateOptions" ) {
                this.atOptions = $.extend( true, {}, o.applyTemplateOptions || {}, {
                    extraSubstitutions: { }
                } );
            }
        },

        /*
         * Call as part of view refresh. Either via _initPagination or directly.
         * scrollParent$ is only used when hasSize option is true
         */
        _updateScrollHandler: function( scrollParent$ ) {
            let o = this.options,
                prevScrollParent$ = this.scrollParent$,
                scrollParentOverride = o.scrollParentOverride;

            if ( o.hasSize ) {
                this.scrollParent$ = scrollParent$;
                this.scrollDelta = 0;
            } else {
                // if the widget has no fixed height then the scrollParent will never scroll and what is really
                // needed is to use the window as the scrollParent - scroll page when the window scrolls
                this.scrollParent$ = scrollParentOverride ? $( scrollParentOverride ) : $( window );
                this.scrollDelta = this.element.offset().top - this._getStickyTop();
            }

            this.totalRenderedRecords = 0;

            if ( o.pagination.scroll && !o.pagination.loadMore ) {
                if ( prevScrollParent$ && this._scrollHandler ) {
                    prevScrollParent$.off( "scroll", this._scrollHandler );
                } else {
                    this._scrollHandler = () => {
                        // only handle scroll paging if visible
                        if ( this.element[0].clientWidth > 0 ) {
                            this._scrollPage();
                        }
                    };
                }
                this.scrollParent$.on( "scroll", this._scrollHandler );
            }
        },

        _removeScrollHandler: function() {
            if ( this._scrollHandler ) {
                this.scrollParent$.off( "scroll", this._scrollHandler );
                this._scrollHandler = null;
            }
        },

        /*
         * This method is called by the _scrollHandler set up by _updateScrollHandler.
         * It is used for auto load more (no fillers) and virtual scrolling (fillers).
         * Fillers are processed in:
         * - addPageOfRecords: finish: where to put the data and update fillers and checking bounds of collapsible control break
         * - _scrollPage handler: what offset to fetch new data at
         * - _initPageSize (due to resize) need to update filler heights.
         * - focus/selection navigation needs to be aware of fillers, collapse/expand also
         */
        _scrollPage: function() {
            var self = this,
                st = this.scrollParent$.scrollTop();

            // When user scrolls need to determine what range of data to render if any (pageOffset).
            // This is done by comparing the offset of the filler rows to the scroll view port
            // to see if any of the filler rows are in in or near the view port.

            // throttle how often we check the scroll offset for paging
            if ( st !== this.lastScrollTop ) {
                this.lastScrollTop = st;

                if ( !this.scrollPageTimer ) {
                    this.scrollPageTimer = setTimeout( function() {
                        let dataCont$, st, prevOffset, vpHeight, vpTop, vpBottom, total, isTable,
                            fillers$ = null,
                            pageAdded = false,
                            hasTotalRecords = modelHasTotalRecords( self.model ),
                            spIsWindow = self.scrollParent$[0] === window,
                            rowHeight = self.avgRowHeight;

                        self.scrollPageTimer = null;

                        // while rendering the start, end bounds of fillers are in flux so just wait for next scroll event
                        if ( self.renderInProgress ) {
                            // avoid dup scroll page checks but try again just in case another scroll event doesn't
                            // happen soon so that we are not left with blank area showing.
                            self.lastScrollTop = null;
                            self._scrollPage();
                            return;
                        }
                        vpHeight = self.scrollParent$.height();
                        vpTop = 0 - ( vpHeight / 2 ); // widen the view port bounds to render more rows early to avoid blank spaces
                        vpHeight *= 2; // because of widening the view port
                        vpBottom = vpTop + vpHeight;

                        dataCont$ = self._getDataContainer();
                        st = self.scrollParent$.scrollTop();
                        if ( self.hasScrollFillers ) {
                            // if the data container has more than one element they should each have the same filter offsets so only need the first one
                            fillers$ = dataCont$.find( SEL_GRID_SCROLL_FILLER );
                            if ( !fillers$.length && hasTotalRecords ) {
                                // if there are no more filler then must have all the data and can remove this scroll paging handler
                                self._removeScrollHandler();
                            }
                            prevOffset = self.pageOffset;
                            isTable = dataCont$[0].nodeName === "TBODY";
                            fillers$.each(function() {
                                var bottom, top,
                                    f$ = $( this ),
                                    fHeight = toInteger( this.style.height ), // want the actual property value. don't measure with f$.height() because of collapsed fillers
                                    start = toInteger( f$.attr( DATA_START ) ),
                                    end = toInteger( f$.attr( DATA_END ) );

                                // empty filler rows should be removed but just in case
                                if ( start === end ) {
                                    return;
                                }
                                // want the top of the filler adjusted by the scroll offset to determine where
                                // it is relative to the viewport.
                                top = spIsWindow ? f$.offset().top : f$.position().top;
                                // in some cases the scroll position is already included otherwise subtract the scroll offset
                                if ( spIsWindow || isTable ||
                                    f$.offsetParent().closest( self.scrollParent$ ).length > 0 ) {
                                    top -= st;
                                }
                                bottom = top + fHeight;
                                if ( top <= vpBottom && bottom >= vpTop ) {
                                    // Figure out if adding to start, middle or end of filler based on where the filler is
                                    // within the view port.
                                    if ( fHeight < ( vpHeight * 2 ) || top > vpTop  ) {
                                        self.pageOffset = start;
                                    } else if ( bottom < vpBottom ) {
                                        self.pageOffset = pageBoundary( end, self.pageSize );
                                    } else {
                                        self.pageOffset = pageBoundary( start + Math.ceil( ( vpTop + vpHeight/2 - top ) * self._getRecordsPerRow() / rowHeight ), self.pageSize );
                                    }

                                    if ( self.pageOffset !== prevOffset ) {
                                        // just in case don't add pages for offset already at
                                        self._addPageOfRecords();
                                    }
                                    pageAdded = true;
                                    if ( !fillerVisible( f$ ) ) {
                                        // if the filler is invisible that means the rows were in a collapsed control break
                                        // need to make the filler visible again
                                        toggleFillerVisible( f$, true );
                                    }
                                    return false; // no need to check any more fillers
                                }
                            } );
                        }
                        if ( !pageAdded && ( !hasTotalRecords || !self.hasScrollFillers ) ) { // when have total records there will be a filler at the end if needed
                            let bottom = dataCont$.height();

                            if ( spIsWindow || dataCont$.offsetParent().closest( self.scrollParent$ ).length > 0 ) {
                                bottom += spIsWindow ? dataCont$.offset().top : dataCont$.position().top;
                            }
                            bottom = bottom - st;
                            if ( bottom > 0 && bottom <= ( vpHeight * 1.5 ) ) {
                                total = self.model.getTotalRecords();
                                // Check highWaterMark to see if more data should be added a the very end. Can't use pageOffset here because
                                // with virtual paging it can be at random places not just at the end.
                                if ( total < 0 || self.highWaterMark < total ) {
                                    self.pageOffset = self.highWaterMark;
                                    self._addPageOfRecords();
                                } else if ( ( fillers$ && fillers$.length === 0 ) || ( !fillers$ && total === self.totalRenderedRecords ) ) {
                                    self._removeScrollHandler();
                                }
                            }
                        }

                    }, SCROLL_PAGE_CHECK );
                }
            }

        },

        /**
         * Internal
         * @private
         */
        _updateTotalRecords: function() {
            var i, start, end, range, pages, pageKey, hasTotalRecords, ellip,
                totalPages = null,
                currentPage = null,
                o = this.options,
                footer$ = this.footer$,
                pageSelector$ = this.pageSelector$,
                pagination = o.pagination,
                total = this.model.getTotalRecords(),
                serverTotal = this.model.getServerTotalRecords();

            if ( total >= 0 ) {
                totalPages = Math.ceil( total / this.pageSize );
                currentPage = Math.floor( this.pageOffset / this.pageSize );
            }

            // if data is added after _refreshPagination (as when being refreshed) preserve scroll offsets
            if ( this.lastScrollLeft !== null )  {
                this.scrollParent$.scrollLeft( this.lastScrollLeft );
                this.scrollParent$.scrollTop( this.lastScrollTop );
                this.lastScrollLeft = null;
                this.lastScrollTop = null;
            }

            this.onLastPage = false;
            if ( this.pageCount === 0 && this.pageOffset === 0 ) {
                range = "";
                this.onLastPage = true;
                footer$.find( SEL_PAGE_CONTROLS ).attr( A_DISABLED, true );
            } else {
                footer$.find( SEL_PAGE_CONTROLS ).attr( A_DISABLED, false );
                this.element.find( ".js-load" ).attr( A_DISABLED, false );
                if ( this.pageOffset === 0 ) {
                    // at the beginning disable prev and first
                    footer$.find( ".js-pg-prev,.js-pg-first" ).attr( A_DISABLED, true );
                }
                if ( total > 0 && currentPage === totalPages - 1 ) {
                    this.onLastPage = true;
                    // at the end disable next and last and loadMore
                    footer$.find( ".js-pg-next,.js-pg-last" ).attr( A_DISABLED, true );
                    this.element.find( ".js-load" ).attr( A_DISABLED, true );
                }
                hasTotalRecords = modelHasTotalRecords( this.model );
                if ( pagination.scroll && pagination.virtual && !pagination.loadMore && hasTotalRecords ) {
                    range = formatMessage( "TOTAL_PAGES", serverTotal );
                } else {
                    if ( hasTotalRecords ) {
                        if ( pagination.hideSinglePage && totalPages === 1 ) {
                            range = formatMessage( "TOTAL_PAGES", serverTotal );
                        } else {
                            range = formatMessage( "PAGE_RANGE_XYZ", this.pageFirstOffset, this.pageLastOffset, serverTotal );
                        }
                    } else {
                        range = formatMessage( "PAGE_RANGE_XY", this.pageFirstOffset, this.pageLastOffset );
                    }
                }
            }

            // The active element is sometimes null on IE11 and according to the spec null is in general possible
            if ( document.activeElement && document.activeElement.disabled ) {
                // When at the end of load more paging don't lose focus
                if ( $( document.activeElement ).hasClass( "js-load" ) ) {
                    this.focus();
                }
                // todo consider if focus should be adjusted if any of the page buttons are focused but disabled
            }

            if ( this.pageRange$ ) {
                this.pageRange$.text( range );
            }

            if ( totalPages > 0 ) {
                if ( pagination.showPageLinks ) {
                    // only update the page links if the total pages, page size changes, or the current page changes because the links need to slide
                    pageKey = `${total}_${this.pageSize}_${currentPage}`;
                    if ( this.pageKey !== pageKey ) {
                        ellip = "<li class='a-GV-pageSelector-item'>&hellip;</li>";
                        this.pageKey = pageKey;

                        pages = "<ul class='a-GV-pageSelector-list'>";
                        // at most maxLinks links
                        start = currentPage - Math.floor( pagination.maxLinks / 2 );
                        if ( start < 0 ) {
                            start = 0;
                        }
                        end = start + pagination.maxLinks;
                        if ( end >= totalPages ) {
                            end = totalPages;
                            start = end - pagination.maxLinks;
                            if ( start < 0 ) {
                                start = 0;
                            }
                        }

                        if ( start > 0 ) {
                            pages += ellip;
                        }
                        for ( i = start; i < end; i++ ) {
                            pages += `<li class='a-GV-pageSelector-item' ${DATA_PAGE}='${i}'><button class='a-GV-pageButton' type='button'>${i + 1}</button></li>`;
                        }
                        if ( end < totalPages ) {
                            pages += ellip;
                        }
                        pages += "</ul>";
                        pageSelector$.html( pages );
                    }
                    pageSelector$.find( SEL_SELECTED ).removeClass( C_SELECTED );
                    pageSelector$.find( `[${DATA_PAGE}='${currentPage}']` ).addClass( C_SELECTED );
                } else if ( pagination.showPageSelector ) {
                    // only update the page selector if the total pages or page size changes
                    pageKey = `${total}_${this.pageSize}`;
                    if ( this.pageKey !== pageKey ) {
                        this.pageKey = pageKey;

                        pages = `<select class='a-GV-pageSelectlist' size='1' aria-label='${getMessage( "PAGE_SELECTION" )}'>`;
                        for ( i = 0; i < totalPages; i++ ) {
                            pages += `<option value='${i}'>${formatMessage( "SELECT_PAGE_N", i + 1 )}</option>`;
                        }
                        pages += "</select>";
                        pageSelector$.html( pages );
                    }
                    pageSelector$.children( "select" ).val( currentPage );
                }
            } else {
                // no pages so need to clear out page selector if any.
                pageSelector$.empty();
                this.pageKey = null;
            }

            if ( pagination.hideSinglePage && !pagination.scroll ) {
                footer$
                    .find( SEL_PAGE_CONTROLS )
                    .add( pageSelector$ )[totalPages === 1 ? "hide" : "show"]();

                footer$.find('.js-rangeDisplay')[hasTotalRecords ? "show" : "hide"]();
            }

            footer$[this.noData && o.hideEmptyFooter ? "hide" : "show"]();
        },

        /**
         * Internal
         * @private
         */
        _updateAvgRowHeight: function() {
            let newAvgHeight,
                o = this.options;

            if ( o.fixedRowHeight ) {
                newAvgHeight = this._getFixedRecordHeight();
            } else {
                let totalHeight,
                    recPerRow = this._getRecordsPerRow(),
                    renderedRows = Math.ceil( this.totalRenderedRecords / recPerRow );

                if ( renderedRows <= 0 ) {
                    newAvgHeight = this.avgRowHeight || this._getFixedRecordHeight();
                } else {
                    totalHeight = this._getDataContainer().height();
                    // todo maybe the total filler height can be tracked rather than measured each time
                    this._getDataContainer().find( SEL_GRID_SCROLL_FILLER ).each( function () {
                        totalHeight -= $( this ).height();
                    } );
                    newAvgHeight = totalHeight / renderedRows;
                }
            }
            // just in case never let row height be non positive
            if ( newAvgHeight <= 0 ) {
                // could be because view not visible? must never let row height be 0 to avoid divide by 0
                newAvgHeight = SAFE_DEFAULT_ROW_HEIGHT;
            }
            this.avgRowHeight = newAvgHeight;
        },

        /**
         * Return the height of the stickyTop of the header.
         * Return 0 if there is no sticky header.
         * @return {Number}
         * @private
         */
        _getStickyTop: function() {
            let stickyTop = this.options.stickyTop;

            return typeof stickyTop === "function" ? stickyTop() : 0;
        },

        //
        // Methods to implement/override
        //

        // refresh

        /**
         * Return the height of a widget header that should be considered when the footer is stuck.
         * Return 0 if there is no header.
         * @return {Number}
         * @private
         */
        _getHeaderHeight: function() {
            return 0;
        },

        /**
         * Return the number of records displayed on a row. Typically this is 1.
         * @return {Number}
         * @private
         */
        _getRecordsPerRow: function() {
            return 1;
        },

        /**
         * Return the height of a record in the view.
         * @return {Number}
         * @private
         */
//        _getFixedRecordHeight: function() {
//            return 40;
//        },

        /**
         * Return a html builder rendering context to be passed to all the rendering functions.
         * Typically this is just a html builder but it can be an object including more context.
         * @return {*}
         * @private
         */
        _getDataRenderContext: function () {
            return util.htmlBuilder();
        },

        /**
         * Return a jQuery object for the element
         * that will contain all the records. It is the element that scrolls not the scroll view port.
         * @return {jQuery}
         * @private
         */
//        _getDataContainer: function() {
//            return $();
//        },

        /**
         * Return the number of selected records.
         * @return {Number}
         * @private
         */
        _selectedCount: function() {
            return 0;
        },

        /**
         * Return a localized message string used to display the number things selected.
         * The string must contain the %0 parameter that will be replaced with the actual count.
         * @returns {string}
         * @private
         */
        _selectedStatusMessage: function() {
            return lang.getMessage( this.options.selectionStatusMessageKey ); // don't use the local getMessage function here
        },

        /**
         * Return the number of deleted records.
         * @return {Number}
         * @private
         */
        _deletedCount: function() {
            return 0;
        },

        _hasControlBreaks: function() {
            return false;
        },

        _elementsFromRecordIds: function( /* ids */ ) {
            return [];
        },

        _renderFillerRecord: function ( /* out, cssClass */ ) {
        },

        _insertFiller: function( /* out, curFiller$ */ ) {
            return $();
        },

        _insertData: function( out, offset, count /*, filler$, how */ ) {
            this._trigger( EVENT_PAGE_CHANGE, null, {
                offset: offset,
                count: count
            } );
        },

        // returns break data { label: "" ...}
        _controlBreakLabel: function( /* record */ ) {
            return {};
        },

        _renderBreakRecord: function ( out, expandControl, breakData /*, serverOffset */ ) {
            out.markup( "<h3>" )
                .content( breakData.label )
                .markup( "</h3>" );
        },

        _renderRecord: function( /* out, record, index, id, meta */ ) {
        },

        _removeRecord: function( /* element */ ) {
        },

        _insertRecord: function( /* element, record, id, meta, where */ ) {
            return $();
        },

        _afterInsert: function( /* insertedElements, copy */ ) {
        },

        _identityChanged: function( /* prevId, id */ ) {
        },

        _replaceRecord: function( /* element, record, oldId, id, meta */ ) {
        },

        _updateRecordField: function( /* element, record, field, meta */ ) {
        },

        _removeFocus: function( /* element */ ) {
        },

        _updateRecordState: function( /* element, id, record, meta, property */ ) {
        }
    });


    //
    // tableModelView
    //
    const C_TMV = "a-TMV",
        C_TMV_VARHEIGHT = C_TMV + "--variableHeight",
        C_TMV_BODY = "a-TMV-body",
        C_TMV_WRAP_SCROLL = "a-TMV-w-scroll",
        SEL_TMV_WRAP_SCROLL = "." + C_TMV_WRAP_SCROLL,
        C_TMV_HEADER = "a-TMV-hdr",
        SEL_TMV_HEADER = "." + C_TMV_HEADER,
        C_DISABLED = "is-disabled";

    const SEL_ACTION_SET = "set",
        SEL_ACTION_UNSET= "unset",
        SEL_ACTION_TOGGLE = "toggle",
        SEL_ACTION_RANGE = "range",
        SEL_ACTION_ADD = "add",
        SEl_ACTION_NONE = "none",
        SEL_ACTION_ALL = "all";

    const EVENT_SELECTION_CHANGE = "selectionChange";

    const C_SELECTOR = "u-selector",
        SEL_SELECTOR = "." + C_SELECTOR,
        C_FOCUSED = "is-focused",
        ARIA_SELECTED = "aria-selected",
        ARIA_MULTI = "aria-multiselectable",
        A_TABINDEX = "tabindex",
        C_RTL = "u-RTL";

    const keys = $.ui.keyCode,
        startsWithTagRE = /^\s*<[\w]+[^>]*>/,
        firstTagRE = /<([^ <>]*)/;

    const NAV_NONE = "none",
        NAV_SELECTION = "selection",
        NAV_FOCUS = "focus";

    function clearSubstitutions(opt) {
        opt.APEX$ROW_STATE_CLASSES = opt.APEX$AGG = opt.APEX$AGG_TOTAL = opt.APEX$ROW_URL = opt.APEX$ROW_ID
            = opt.APEX$ROW_INDEX = opt.APEX$VALIDATION_MESSAGE = null;
    }

    function validateItemNavigation( value ) {
        if ( ![NAV_NONE, NAV_FOCUS, NAV_SELECTION].includes( value ) ) {
            debug.warn( "Invalid value for itemNavigation ignored" );
            value = "none";
        }
        return value;
    }

    /**
     * @uiwidget tableModelView
     * @since 5.1
     * @extends {tableModelViewBase}
     *
     * @classdesc
     * <p>Template driven view for a table {@link model} that supports pagination.
     * Derived from {@link tableModelViewBase}. Does not directly support editing but does respond to model changes.
     * Supports selection when using an {@link iconList} widget to handle the records.</p>
     *
     * <p>Note: Not all of the options and methods from the base widget apply to this widget. For example
     * options and methods related to editing do not apply.</p>
     *
     * <p>The expected markup is an empty element; typically a <code class="prettyprint">&lt;div></code>.</p>
     *
     * <p>There are two ways to define the markup for the view.
     * Configure with options {@link tableModelView#beforeTemplate}, {@link tableModelView#recordTemplate}, and
     * {@link tableModelView#afterTemplate} for complete control over the
     * markup. Or configure with options {@link tableModelView#iconClassColumn}, {@link tableModelView#imageURLColumn},
     * {@link tableModelView#imageAttributes}, {@link tableModelView#labelColumn}, {@link tableModelView#linkTarget},
     * {@link tableModelView#linkTargetColumn}, and {@link tableModelView#linkAttributes} for default list markup.</p>
     *
     * @desc Creates a tableModelView widget.
     *
     * @param {Object} options A map of option-value pairs to set on the widget.
     *
     * @example <caption>Create a tableModelView for name value paris displayed in a simple table.</caption>
     *   var fields = {
     *           PARTNO: {
     *               index: 0
     *           },
     *           PART_DESC: {
     *               index: 1
     *           }
     *       },
     *       data = [
     *           ["B-10091", "Spark plug"],
     *           ["A-12872", "Radiator hose"],
     *           ...
     *       ];
     *   apex.model.create("parts", {
     *           shape: "table",
     *           recordIsArray: true,
     *           fields: fields,
     *           paginationType: "progressive"
     *       }, data, data.length );
     *   $("#partsView").tableModelView( {
     *       modelName: "parts",
     *       beforeTemplate: '<table class="u-Report"><thead><tr><th>Part No</th><th>Description</th></tr></thead><tbody>',
     *       afterTemplate: '</tbody></table>',
     *       recordTemplate: '<tr><td>&PARTNO.</td><td>&PART_DESC.</td></tr>',
     *       pagination: {
     *           scroll: true
     *       }
     *   } );
     */
    // todo specify expected markup for templates and example should show use of expected data attributes
    // todo document selection support including template requirements u-selector [u-selector--single] and keyboard info
    // todo consider tooltip support and context menu mixin
    $.widget( "apex.tableModelView", $.apex.tableModelViewBase,
        /**
         * @lends tableModelView.prototype
         */
        {
        version: "21.1",
        widgetEventPrefix: "tableModelView",
        options: {
            /**
             * <p>Optional markup for a header to render before the report. The header does not scroll
             * with the report and depending on <code class="prettyprint">stickyTop</code> option may stick to the
             * top of the page. See also option {@link tableModelView#syncHeaderHScroll}.</p>
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default ""
             * @example "<h4>My Report</h4>"
             * @example "<h4>My Report</h4>"
             */
            // todo consider documenting APEX$COLUMNS
            headerTemplate: "",
            /**
             * <p>Markup to render before the record data.
             * The markup must include an opening element that will contain all the records.
             * For example <code class="prettyprint">&lt;ul></code>.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default "&lt;ul>"
             * @example "<ol>"
             * @example "<ol>"
             */
            beforeTemplate: "<ul>",
            /**
             * <p>Markup to render for each record. Can use substitution tokens from the
             * model using column names. In addition you can use the following special substitution symbols:</p>
             * <ul>
             *     <li>APEX$ROW_ID - The record id</li>
             *     <li>APEX$ROW_INDEX - The record index</li>
             *     <li>APEX$ROW_URL - The record url if any</li>
             *     <li>APEX$ROW_STATE_CLASSES - Various record states such as "is-inserted" or "is-deleted"</li>
             *     <li>APEX$VALIDATION_MESSAGE - If the record is in a validation error or warning state this is the associated message</li>
             * </ul>
             * <p>At a minimum one of {@link tableModelView#labelColumn} or {@link tableModelView#recordTemplate} is required.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default Markup depends on several other options.
             * @example "<li>&NAME. - &SALARY.</li>"
             * @example "<li>&NAME. - &SALARY.</li>"
             */
            recordTemplate: "",
            /**
             * <p>Markup to render for a control break. This should only be set if the model data
             * has control breaks. In addition you can use the following special substitution symbol:</p>
             * <ul>
             *     <li>APEX$ROW_INDEX - The record index</li>
             * </ul>
             *
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default ""
             * @example "<li><h4>&CATEGORY%heading. &CATEGORY.</h4></li>"
             * @example "<li><h4>&CATEGORY%heading. &CATEGORY.</h4></li>"
             */
            controlBreakTemplate: "",
            /**
             * <p>Markup to render for an aggregate record. This should only be set if the model data
             * contains aggregate records. In addition you use the following special substitution symbols:</p>
             * <ul>
             *     <li>APEX$ROW_ID - The record id from the model as if from {@link model#recordId}.</li>
             *     <li>APEX$ROW_INDEX - The record index</li>
             *     <li>APEX$AGG - The name of the aggregate function.
             *     One of: "COUNT", "COUNT_DISTINCT", "SUM", "AVG", "MIN" ,"MAX", "MEDIAN"</li>
             *     <li>APEX$AGG_TOTAL - This is true when the aggregate record is a grand total and false otherwise.</li>
             * </ul>
             *
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default ""
             * @example "<li>{case APEX$AGG/}{when SUM/}Total:</b> &SALARY.{endcase/}</li>"
             * @example "<li>{case APEX$AGG/}{when SUM/}Total:</b> &SALARY.{endcase/}</li>"
             */
            aggregateTemplate: "",
            /**
             * <p>Markup to render after the record data.
             * The markup must include an element that closes the opening element from the
             * <code class="prettyprint">beforeTemplate</code> option.
             * For example <code class="prettyprint">&lt;/ul></code>.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default "&lt;/ul>"
             * @example "</ol>"
             * @example "</ol>"
             */
            afterTemplate: "</ul>",
            /**
             * <p>If there is a {@link tableModelView#headerTemplate} and this is true the horizontal scroll offset
             * will be synchronized between the header and the view body. This is useful in cases such as a table
             * where the header columns need to align with the body columns.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            syncHeaderHScroll: false,
            /**
             * <p>Extra CSS classes to add to the element that is the parent of the collection of records.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default "a-TMV-defaultIconView" if {@link tableModelView#recordTemplate} is null and null otherwise.
             * @example "EmployeeList"
             * @example "EmployeeList"
             */
            collectionClasses: null,
            /**
             * <p>The CSS class column to use for the icon. At most one of <code class="prettyprint">iconClassColumn</code>
             * and <code class="prettyprint">imageURLColumn</code> should be given.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default null
             * @example "PERSON_AVATAR"
             * @example "PERSON_AVATAR"
             */
            iconClassColumn: null,
            /**
             * <p>The URL column of image to use for the icon. At most one of <code class="prettyprint">iconClassColumn</code>
             * and <code class="prettyprint">imageURLColumn</code> should be given.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default null
             * @example "PROD_IMAGE_URL"
             * @example "PROD_IMAGE_URL"
             */
            imageURLColumn: null,
            /**
             * <p>Attributes for the <code class="prettyprint">&lt;img></code> element.
             * Only used if {@link tableModelView#imageURLColumn} is specified.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default null
             */
            imageAttributes: null,
            /**
             * <p>Name of the column that contains the label text.</p>
             * <p>At a minimum one of {@link tableModelView#labelColumn} or {@link tableModelView#recordTemplate} is required.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default null
             * @example "EMP_NAME"
             * @example "EMP_NAME"
             */
            labelColumn: null,
            /**
             * <p>If true the record metadata should contain a <code class="prettyprint">url</code> property that contains the link target.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            linkTarget: false,
            /**
             * <p>The name of the column that contains the anchor <code class="prettyprint">href</code>.
             * If not given the <code class="prettyprint">href</code> comes from the record field metadata
             * <code class="prettyprint">url</code> property. If there is no <code class="prettyprint">url</code>
             * property then this item has no link.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default null
             * @example "PROD_TARGET"
             * @example "PROD_TARGET"
             */
            linkTargetColumn: null,
            /**
             * <p>Additional anchor attributes. Ignored unless a link is present.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default null
             * @example "target='_blank'"
             * @example "target='_blank'"
             */
            linkAttributes: null,
            /**
             * <p>A jQuery selector that identifies item content that can be a tab stop when an item has focus.
             * </p>
             * todo is this needed and if so is noNavKeyContent needed?
             * @ignore
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default null
             * @example "a,button"
             * @example "a,button"
             */
            tabbableContent: null,
            /**
             * <p>If true use the {@link iconList} widget to display the records.</p>
             *
             * @deprecated
             * @memberof tableModelView
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            useIconList: false,
            /**
             * <p>Additional options to pass to the {@link iconList} widget. See {@link iconList} for information about the
             * options it supports. Only applies if {@link tableModelView#useIconList} option is true.</p>
             *
             * @deprecated
             * @memberof tableModelView
             * @instance
             * @type {object}
             * @default {}
             */
            iconListOptions: {},
            /**
             * <p>Controls how the focus and selection state is handled for items in the view. It can be one of these
             * values:</p>
             * <ul>
             * <li><strong>none</strong> - The view does not support focus or selection.</li>
             * <li><strong>focus</strong> - The view supports focus state. Focus can be moved among
             *   the items of the view using keyboard our mouse.
             *   The app or theme must include styles for the is-focused state.</li>
             * <li><strong>selection</strong> - The view supports focus and selection state.
             * The template is responsible for including markup for checkbox/radio selection control and
             * The app or theme must include styles for the is-selected state.</li>
             * </ul>
             * <p>When set to "focus" or "selection" the {@link tableModelView#itemSelector} option is required.
             * When useIconList is true this is forced to "none" because the icon list handles selection.
             * </p>
             * todo doc once ready
             * @ignore
             * @memberof tableModelView
             * @instance
             * @type {string}
             * @default "none"
             */
            itemNavigation: NAV_NONE,
            /**
             * <p>A CSS selector that selects the outermost item element in the view collection.
             * This is required if {@link tableModelView#itemNavigation} is set to "focus" or "selection".</p>
             * todo doc once ready
             * @ignore
             * @memberof tableModelView
             * @instance
             * @type {?string}
             * @default "a-TMV-item" if {@link tableModelView#recordTemplate} is null and null otherwise.
             * @default null
             */
            itemSelector: null,
            /**
             * <p>If true multiple items can be selected otherwise only a single item can be selected.
             * This option only applies when {@link tableModelView#itemNavigation} is "selection".</p>
             *
             * todo doc once ready
             * @ignore
             * @memberof tableModelView
             * @instance
             * @type {boolean}
             * @default false
             * @example true
             * @example true
             */
            multiple: false,
            /**
             * <p>Normally keydown handling will call preventDefault so that arrow key navigation has no effect outside
             * this control. This prevents text selection and keeps parents from scrolling. By setting this to false
             * it would allow a nested container to respond to arrow navigation keys. Note this only applies when
             * useIconList is true since it is the only case where arrow keys are used for navigation of the list.</p>
             *
             * @memberof tableModelView
             * @instance
             * @type {boolean}
             * @default true
             */
            constrainNavigation: true,

            /**
             * todo not fully implemented tbd does this belong in this widget or external?
             * probably also must have itemNavigation = "selection"
             * @ignore
             * @memberof tableModelView
             * @instance
             * @type {boolean}
             * @default false
             */
            sortable: false,

            //
            // events:
            //

            /**
             * Triggered when the selection state changes. It has no additional data. Only tableModelViews with
             * {@link tableModelView#useIconList} true support selection.
             *
             * @event selectionchange
             * @memberof tableModelView
             * @instance
             * @property {Event} event <code class="prettyprint">jQuery</code> event object.
             *
             * @example <caption>Initialize the tableModelView with the <code class="prettyprint">selectionChange</code> callback specified:</caption>
             * $( ".selector" ).tableModelView({
             *     selectionChange: function( event ) {}
             * });
             *
             * @example <caption>Bind an event listener to the <code class="prettyprint">tablemodelviewselectionchange</code> event:</caption>
             * $( ".selector" ).on( "tablemodelviewselectionchange", function( event ) {} );
             */
            selectionChange: null,
            /**
             * Triggered when there is a pagination event that results in new records being displayed.
             *
             * @event pagechange
             * @memberof tableModelView
             * @instance
             * @property {Event} event <code class="prettyprint">jQuery</code> event object.
             * @property {Object} data Additional data for the event.
             * When there are no records to display <code class="prettyprint">offset</code> and <code class="prettyprint">count</code> are 0.
             * @property {number} data.offset the offset of the first record in the page.
             * @property {number} data.count the number of records in the page that were added to the view.
             *
             * @example <caption>Initialize the tableModelView with the <code class="prettyprint">pageChange</code> callback specified:</caption>
             * $( ".selector" ).tableModelView({
             *     pageChange: function( event, data ) {}
             * });
             *
             * @example <caption>Bind an event listener to the <code class="prettyprint">tablemodelviewpagechange</code> event:</caption>
             * $( ".selector" ).on( "tablemodelviewpagechange", function( event, data ) {} );
             */
            pageChange: null,
            /**
             * todo remove this?
             * @ignore
             */
            rowOrderChanged: null
        },

        _create: function () {
            const o = this.options,
                ctrl$ = this.element;

            debug.info( `TableModelView '${ctrl$[0].id}' created. Model: ${o.modelName}` );

            ctrl$.addClass( C_TMV );
            if ( !o.fixedRowHeight ) {
                ctrl$.addClass( C_TMV_VARHEIGHT );
            }
            o.itemNavigation = validateItemNavigation( o.itemNavigation );
            // todo when supporting focus or selection the view should probably have an aria role such as grid
            //   is this the responsibility of the caller or this widget? grid may not be the right role but what is?

            if ( o.itemNavigation === NAV_SELECTION && o.multiple ) {
                ctrl$.attr( ARIA_MULTI, "true" );
            }

            this.forwardKey = keys.RIGHT;
            this.backwardKey = keys.LEFT;
            if ( ctrl$.css( "direction" ) === "rtl" ) {
                ctrl$.addClass( C_RTL );
                this.forwardKey = keys.LEFT;
                this.backwardKey = keys.RIGHT;
            }

            this._enforceOptionConstraints();

            this._super(); // init table model view base

            this.recordHeight = null;
            this.recPerRow = null;
            this.lastFocused = null;    // last cell that had focus or has focus could be button/link inside a cell
            this.hasSelection = false;  // is there currently a selection or not, only used when persistSelection

            // get the model
            this._initModel( o.modelName );

            this._on( this._eventHandlers );

            this._initRecordTemplate();

            // use debounce (timer) to make sure the focus happens first and also throttle rapid changes from keyboard navigation.
            this._selNotifyDelay = util.debounce( this._selChangeNotify, 1 );
            this._selNotifyLongDelay = util.debounce( this._selChangeNotify, 350 );

            this.refresh();
            if ( o.disabled ) {
                this._setOption( "disabled", o.disabled );
            }
        },

        _eventHandlers: {
            keydown: function( event ) {
                const o = this.options;
                let iconListOpts,
                    handled = false,
                    target$ = $( event.target ),
                    next$ = null,
                    kc = event.which;

                if ( event.isDefaultPrevented() ) {
                    return;
                }

                if ( o.useIconList ) {
                    iconListOpts = this.getIconList().options;
                    if ( !iconListOpts.navigation && iconListOpts.noNavKeyContent && $( event.target ).closest( iconListOpts.noNavKeyContent ).length ) {
                        // don't do key processing if focus is in an element that needs all the keys
                        return;
                    }

                    // let down arrow or forward arrow move to load more button if there is one
                    if ( ( kc === keys.DOWN || kc === this.getIconList().forwardKey ) && o.pagination.loadMore ) {
                        if ( this.element.find( ".js-load" ).not( ":disabled" ).focus()[0] ) {
                            event.preventDefault();
                        } // else should navigate through the list items
                    } else if ( ( kc === keys.UP || kc === this.getIconList().backwardKey ) &&
                                o.pagination.loadMore && $( event.target ).hasClass( "js-load" ) ) {
                        this.getIconList().focus();
                        event.preventDefault();
                    }
                    if ( o.constrainNavigation && ( kc === keys.DOWN || kc === keys.UP || kc === keys.LEFT || kc === keys.RIGHT ||
                            kc === keys.PAGE_UP || kc === keys.PAGE_DOWN ) ) {
                        event.preventDefault();
                    }
                } else if ( o.itemNavigation !== NAV_NONE ) {
                    if ( kc === keys.HOME ) {
                        // todo consider start of report vs start of row
                        this._navToReportStart( event, true );
                        handled = true;
                    } else if ( kc === keys.END ) {
                        // todo consider end of report vs end of row
                        this._navToReportEnd( event, true );
                        handled = true;
                    } else if ( kc === keys.DOWN || kc === this.forwardKey ) {
                        if ( kc === this.forwardKey && event.altKey ) {
                            return; // let the browser have this key
                        }
                        // Current behavior is linear even if grid layout so down and forward are the same
                        // todo handle 2D grid nav
                        next$ = this._getItem( $( this.lastFocused ) ).nextAll( SEL_VISIBLE ).first();
                        if ( (!next$ || !next$[0] ) && o.pagination.loadMore ) {
                            if ( this.element.find( ".js-load" ).focus()[0] ) {
                                handled = true;
                            }
                        }
                        if ( kc === keys.DOWN && o.constrainNavigation ) {
                            // don't let selection happen or scrolling
                            handled = true;
                        }
                    } else if ( kc === keys.UP || kc === this.backwardKey ) {
                        if ( kc === this.forwardKey && ( event.altKey || event.metaKey ) ) {
                            return; // let the browser have this key
                        }
                        if ( target$.hasClass( "js-load" ) ) {
                            this.focus();
                            handled = true;
                        } else {
                            // Current behavior is linear even if grid layout so up and backward are the same
                            // todo handle 2D grid nav
                            next$ = this._getItem( $( this.lastFocused ) ).prevAll( SEL_VISIBLE ).first();
                        }
                        if ( kc === keys.UP ) {
                            event.stopPropagation(); // Don't let a containing tab or accordion act on Ctrl+Up
                        }
                    } else if ( kc === keys.SPACE ) {
                        // ignore if on a button
                        if ( event.target.nodeName === "BUTTON" ) {
                            return;
                        }
                        // next is really current item
                        next$ = this._getItem( $( this.lastFocused ) );
                    }
                    // todo page up/down
                    // todo Ctrl+A for select all

                    if ( next$ && next$[0] ) {
                        if ( o.itemNavigation === NAV_SELECTION ) {
                            this._select( next$, event, true, true );
                        } else {
                            $( this._getItemFocusable( next$ ) ).focus();
                        }
                        handled = true;
                    }
                    if ( handled ) {
                        event.preventDefault();
                    }

                    /* TODO need to handle keyboard support for sortable
                        // needs to involve the model
                        if ( kc === keys.DOWN || kc === this.forwardKey ) {
                            target$.insertAfter( target$.next() );
                        } else if ( kc === keys.UP || kc === this.backwardKey ) {
                            target$.insertBefore( target$.prev() );
                        } else if ( kc === keys.HOME ) {    // Page Up
                            ctrl$.prepend(target$);
                        } else if ( kc === keys.END ) {   // Page Down
                            ctrl$.append(target$);
                        }
                        target$.focus();
                     */
                }
            },
            keyup: function( event ) {
                let sortable = this.options.sortable,
                    kc = event.which;

                /**
                 * checks that it's the OS or Ctrl key that is releases
                 * todo needs to check if it's from an active move of an item
                 */
                if ( sortable && ( kc === 91 || kc === 17 ) ) { // sortable && ( mac os || ctrl )
                    this.element.trigger( "sortUpdate" );
                }

            },
            mousedown: function ( event ) {
                const o = this.options;

                // when sortable we need to make sure the item is selected before it gets dragged
                // and for multiple selection don't want text selection.
                if ( o.itemNavigation === NAV_SELECTION && ( o.sortable || o.multiple) ) {
                    let item$ = this._getItem( $( event.target ) );

                    if ( item$ && item$.length > 0 ) {
                        event.preventDefault(); // this prevents text selection
                        if ( o.sortable ) {
                            // todo WIP multiple selection doesn't work with sortable
                            let fakeEvent = {
                                type: "click",
                                shiftKey: event.shiftKey,
                                ctrlKey: event.ctrlKey,
                                metaKey: event.metaKey
                            };
                            this._select( item$, fakeEvent, true, false, false );
                        }
                    }
                }
            },
            resize: function( event ) {
                if (event.target !== this.element[0]) {
                    return;
                }
                this.resize();
                event.stopPropagation();
            },
            click: function ( event ) {
                const o = this.options;

                if ( o.itemNavigation === NAV_SELECTION ) {
                    let target$ = $( event.target ),
                        item$ = this._getItem( target$ );

                    if ( item$ && item$.length > 0 ) {
                        if ( target$.is( SEL_SELECTOR ) ) {
                            // when click the selector checkbox/radio always behave like toggle
                            event.ctrlKey = true;
                            event.shiftKey = false;
                        }
                        this._select( item$, event, true, false, false );
                        // prevent default unless intend to be clicking on some tabbable content
                        if ( !o.tabbableContent || !target$.closest( o.tabbableContent ).length ) {
                            event.preventDefault();
                        } else {
                            // when click on tabbable content focus it after selection.
                            target$.focus();
                        }
                    }
                }
            },
            sortUpdate: function( event ) {
                let target$ = $( event.target );
                if ( this.options.rowOrderChanged ) {
                    this._trigger( "rowOrderChanged" );
                }
                target$.focus();
            },
            focusin: function ( event ) {
                if ( this.options.itemNavigation !== NAV_NONE ) {
                    let target = event.target,
                        item$ = this._getItem( $( target ) );

                    if ( item$ && item$.length > 0 ) {
                        item$.addClass( C_FOCUSED );
                        this._setFocusable( target );
                    }
                }
            },
            focusout: function ( event ) {
                if ( this.options.itemNavigation !== NAV_NONE ) {
                    let item$ = this._getItem( $( event.target ) );

                    if ( item$ && item$.length > 0 ) {
                        item$.removeClass( C_FOCUSED );
                    }
                }
            }
        },

        _destroy: function() {
            const ctrl$ = this.element;

            this._tableModelViewDestroy();
            ctrl$.removeClass( C_TMV + " " + C_DISABLED + " " + C_RTL )
                .removeAttr( ARIA_MULTI )
                .empty(); // this will destroy the iconList if any

            debug.info( `TableModelView '${ctrl$[0].id}' destroyed. Model: ${this.options.modelName}` );

            this.data$ = null;

            // disconnect from the model
            this._initModel( null ); // this will cleanup change listener
        },

        _setOption: function ( key, value ) {
            const ctrl$ = this.element,
                o = this.options;

            debug.info( `TabelModelView '${this.element[0].id}' set option '${key}' to: ${value}` );

            if ( key === "iconListOptions" ) {
                throw new Error( `TabelModelView '${key}' cannot be set. Set options directly on the iconList widget` );
            }

            if ( key === "itemNavigation" ) {
                value = validateItemNavigation( value );
            }

            this._super( key, value );

            this._enforceOptionConstraints();

            if ( key === "disabled" ) {
                this.element.toggleClass( C_DISABLED, value );
                if ( o.useIconList ) {
                    return this.getIconList().option( key, value );
                } else {
                    if ( this.lastFocused ) {
                        // todo remove old tab stops
                        if ( value ) {
                            this.lastFocused.tabIndex = -1;
                        } else {
                            this._setFocusable( this.lastFocused );
                        }
                    }
                }
                if ( value ) {
                    // when enabling do this just in case it was resized while disabled
                    this.resize();
                }
            } else if ( key === "modelName" ) {
                this._initModel( value );
                this.refresh( false );
            } else if ( key === "highlights" ) {
                this._updateHighlights();
            } else if ( key === "hideEmptyFooter" ) {
                this._updateTotalRecords();
            } else if ( key === "multiple" || key === "itemNavigation" ) {
                if ( o.itemNavigation === NAV_SELECTION && o.multiple ) {
                    ctrl$.attr( ARIA_MULTI, "true" );
                } else {
                    ctrl$.removeAttr( ARIA_MULTI );
                }
                // todo make sure focus/selection states get cleared
                this._initRecordTemplate();
                this.refresh();
            } else if ( key === "fixedRowHeight" ) {
                this.element.toggleClass( C_TMV_VARHEIGHT, !value );
                this.refresh();
            } else if ( ["beforeTemplate", "recordTemplate", "afterTemplate", "collectionClasses",
                        "iconClassColumn", "imageURLColumn", "imageAttributes", "itemSelector",
                        "labelColumn", "linkTarget", "linkTargetColumn", "linkAttributes"].includes( key ) ) {
                this._initRecordTemplate();
                this.refresh();// todo allow multiple set option but refresh just once
            } else if ( ["footer", "useIconList", "pagination", "rowsPerPage", "sortable"].includes( key ) ) {
                this.refresh();
            }
        },

        _enforceOptionConstraints: function() {
            const o = this.options;

            if ( o.useIconList && o.itemNavigation !== NAV_NONE ) {
                debug.warn( "Forced itemNavigation to none when useIconList is true" );
                o.itemNavigation = NAV_NONE;
            }
            if ( o.itemNavigation !== NAV_SELECTION && o.persistSelection ) {
                // can't persist selection if not doing selection
                debug.warn( "Forced persistSelection to false when itemNavigation is not selection" );
                o.persistSelection = false;
            }
        },

        /**
         * <p>Refresh the view. Typically no need to call this method because it is called automatically when
         * the model data is refreshed.</p>
         *
         * @param {boolean} [pFocus] if true put focus in the view, if false don't. If undefined/omitted maintain
         * focus if the view already has focus.
         */
        refresh: function( pFocus ) {
            const o = this.options,
                self = this,
                ctrl$ = this.element;
            let selection, hadSelection,
                paginationFocus = false;

            if ( pFocus !== true ) {
                paginationFocus = this._paginationHasFocus();
            }
            if ( pFocus === undefined ) {
                pFocus = $( document.activeElement ).closest( ctrl$ ).length > 0;
            }

            if ( !this._refreshCheckIfVisible() ) {
                return; // be lazy and don't refresh if invisible
            }

            // has the model changed since the last refresh?
            let modelChanged = this.modelChanged;
            this.modelChanged = false;

            // preserve current selection
            if ( this.data$ && !o.persistSelection ) {
                // xxx is getSelectedRecords reliable? see grid
                selection = this.getSelectedRecords();
            } else if ( o.persistSelection ) {
                // The model may have been cleared in which case the selection is lost. If the model wasn't cleared
                // no need to save the selection here because it will be restored as the rows are rendered.
                // Because the model may have lost its selection the view remembers if it once had anything selected
                // just so that a selection change notification can be given if needed
                hadSelection = this.hasSelection;
                this.hasSelection = false;
            }

            if ( !o.hasSize && this.data$ ) {
                // when the widget doesn't have a height it will get very small during a refresh
                // don't want to mess up scroll offsets
                ctrl$.css( "min-height", ctrl$.height() + "px" ); // removed after data is rendered
            }
            this._refreshPagination( ctrl$.find( SEL_TMV_WRAP_SCROLL ) );

            // todo don't redraw everything unless needed xxx this wipes out the footer which clears state icons which is not desired also want to keep the header
            this._initView();

            if ( o.sortable ) {
                // todo consider may want to set some other sortable options
                this.data$.sortable( {
                    placeholder: "sortable-placeholder",
                    cursor: "move",
                    cursorAt: { left: 2 },
                    stop: function( /* event, ui */ ) {
                        ctrl$.trigger( "sortUpdate" ); // xxx can this made up event be avoided?
                    }
                } );
            }

            this._updateStatus();
            this.resize();
            this._addPageOfRecords( function() {
                let lastFocused,
                    selectionMade = false;

                // favor returning focus to pagination area if it was there unless pFocus is true to explicitly set focus to the data.
                if ( paginationFocus ) {
                    self._restorePaginationFocus( paginationFocus );
                    pFocus = false;
                }

                if ( !o.hasSize ) {
                    ctrl$.css( "min-height", "" );
                }

                if ( o.itemNavigation !== NAV_NONE && !self.lastFocused ) {
                    // if the last focused element was lost (or never established)
                    self.lastFocused = self._getItemFocusable( self.data$.children( o.itemSelector ) );
                    if ( self.lastFocused && o.itemNavigation === NAV_SELECTION ) {
                        self.selectAnchor = $( self.lastFocused ).closest( o.itemSelector )[0];
                    }
                    if ( self.lastFocused && !o.disabled ) {
                        lastFocused = self.lastFocused; // restoring the selection can change this so set it again below
                        self._setFocusable( self.lastFocused );
                    }
                }
                // restore selection if any
                if ( selection && selection.length > 0 ) {
                    selectionMade = self.setSelectedRecords( selection, false, true ) > 0;

                    // there was a selection but not able to make a selection after refresh so the selection has changed
                    if ( !selectionMade || modelChanged ) {
                        self._selNotifyDelay( true, null );
                    }
                } else if ( o.persistSelection ) {
                    // selection state is kept in the model it could be that there was no selection before refresh or
                    // it could be that the model was cleared and there was a selection that was lost.
                    // hadSelection is used to distinguish those cases If true there was a selection and isn't now so notify
                    if ( hadSelection || modelChanged ) {
                        self._selNotifyDelay( true, null );
                    }
                }
                if ( lastFocused && selectionMade ) {
                    self._setFocusable( lastFocused );
                }
                if ( pFocus ) {
                    self.focus();
                }
                if ( o.sortable ) {
                    self.data$.sortable( "refresh" );
                }
            } );
        },

        /**
         * <p>This method must be called if the size of the container changes so that pagination state, footer position,
         * and nested {@link iconList} if any can be updated to reflect the new size.</p>
         */
        resize: function() {
            var w, h,
                ctrl$ = this.element,
                o = this.options,
                ctrlH = ctrl$.height(),
                body$ = ctrl$.children().first(),
                wrapper$ = ctrl$.find( SEL_TMV_WRAP_SCROLL );

            if ( ctrl$[0].offsetParent === null ) {
                // View is invisible so nothing to resize. Expect a resize or refresh later when made visible
                return;
            }
            if ( !body$.length || this.pendingRefresh ) {
                // view was never initialized probably because it was initially invisible
                // or was refreshed while invisible. So do that now
                this.refresh();
                return; // because refresh calls resize
            }

            w = ctrl$.width();
            wrapper$.width( w );
            this.recordHeight = null;
            this.recPerRow = null;

            if ( o.hasSize ) {
                h = ctrlH - this._footerHeight();
                body$.height( h );
                wrapper$.height( h - (ctrl$.find( SEL_TMV_HEADER ).height() || 0) );
            }
            if ( o.useIconList ) {
                this.data$.iconList( "resize" );
            }
            this._initPageSize();
        },

        /**
         * <p>Give focus to the tableModelView. Only applies if the view supports selection or focus. This is the case
         * when {@link tableModelView#useIconList} is true. Focus is given to the last item that had focus.</p>
         * @example <caption>This example focuses the view.</caption>
         * $( ".selector" ).tableModelView( "focus" );
         */
        focus: function() {
            const iconList = this.getIconList();

            if ( iconList ) {
                return iconList.focus();
            } else if ( this.options.itemNavigation !== NAV_NONE && this.lastFocused ) {
                this.lastFocused.focus();
            } else {
                this.data$.find( ":focusable" ).first().focus();
            }
        },

        /**
         * <p>Return the currently selected elements.</p>
         *
         * <p>This is only applicable if the {@link tableModelView#useIconList} option is true.</p>
         *
         * @return {jQuery} The selected record elements. Returns null if not using an iconList.
         */
        getSelection: function() {
            const o = this.options,
                iconList = this.getIconList();

            if ( o.useIconList ) {
                // it is possible (because of preserve selection) to try to get the selection
                // before the icon list has been initialized
                return iconList ? iconList.getSelection() : $();
            } else if ( this.data$ && o.itemNavigation === NAV_SELECTION ) {
                return this.data$.find( SEL_SELECTED );
            }
            return null;
        },

        /**
         * <p>Set the selected record elements.</p>
         *
         * <p>This is only applicable if the {@link tableModelView#useIconList} option is true.</p>
         *
         * @param {jQuery} pElements$ A jQuery object with record elements such as the return value of getSelection.
         * @param {boolean} [pFocus] If true the first element of the selection is given focus.
         * @param {boolean} [pNoNotify] If true the selection change event will be suppressed.
         */
        setSelection: function( pElements$, pFocus, pNoNotify ) {
            const iconList = this.getIconList();

            if ( iconList ) {
                iconList.setSelection( pElements$, pFocus, pNoNotify );
            } else if ( this.data$ && this.options.itemNavigation === NAV_SELECTION ) {
                this._select( pElements$, null, pFocus, false, pNoNotify );
            }
        },

        // todo add selectAll: function( pFocus, pNoNotify )

        /**
         * <p>Given a jQuery object with one or more record elements return the corresponding model records.
         * For this to work the elements must have a data-id attribute with the value of the record id.</p>
         *
         * @param {jQuery} pElements$ A jQuery object of record elements such as returned
         *   by {@link tableModelView#getSelection}.
         * @return {model.Record[]} Array of records from the model corresponding to the record elements.
         */
        getRecords: function( pElements$ ) {
            let thisModel = this.model,
                records = [];

            pElements$.each( function() {
                let value,
                    id = $( this ).attr( DATA_ID );

                if ( id ) {
                    value = thisModel.getRecord( id );
                    if ( value ) {
                        records.push( value );
                    }
                }
            } );

            return records;
        },

        /**
         * <p>Return the underlying data model records corresponding to the current selection.</p>
         * <p>This is only applicable if the {@link tableModelView#useIconList} option is true. If it is false then null is returned.</p>
         *
         * @return {model.Record[]} Array of records from the model corresponding to the selected record elements.
         */
        getSelectedRecords: function() {
            const o = this.options;

            if ( this.data$ && ( o.useIconList || o.itemNavigation === NAV_SELECTION ) ) {
                return this.getRecords( this.getSelection() );
            } // else
            return null;
        },

        /**
         * <p>Select the elements that correspond to the given data model records.</p>
         *
         * <p>This is only applicable if the {@link tableModelView#useIconList} option is true.</p>
         *
         * @param {model.Record[]} pRecords Array of data model records to select.
         * @param {boolean} [pFocus] If true the first record of the selection is given focus.
         * @param {boolean} [pNoNotify] If true the selection change event will be suppressed.
         */
        // todo update doc once selection options ready
        // todo doc return value once ready count selected or -1 if not supported
        setSelectedRecords: function( pRecords, pFocus, pNoNotify ) {
            const o = this.options;
            let items = [],
                count = 0,
                len = pRecords.length,
                keys = new Set(),
                data$ = this.data$;

            if ( !data$ || this.renderInProgress || ( o.itemNavigation !== NAV_SELECTION && !o.useIconList ) ) {
                return -1;
            } // else

            if ( !o.multiple && len > 1 ) {
                len = 1;
            }
            for ( let i = 0; i < len; i++ ) {
                keys.add( this.model.getRecordId( pRecords[i] ) );
            }
            this.data$.children().each(function() {
                let id = $( this ).attr( DATA_ID );

                if ( keys.has( id ) ) {
                    keys.delete( id );
                    items.push( this );
                    count += 1;
                }
            });
            this.setSelection( $( items ), pFocus, pNoNotify );

            // if model is keeping the selection and using virtual paging then it is possible to select records that
            // are not in the DOM at the moment
            if ( o.persistSelection ) {
                for ( let id of keys ) {
                    let rec = this.model.getRecord( id );
                    if ( rec ) {
                        this.model.setSelectionState( id, true );
                        count += 1;
                    }
                }
                // if there were no items it means there were no rendered records to be selected
                // which means the selection change event would not have been triggered
                if ( items.length === 0 && !pNoNotify ) {
                    // todo don't just assume something changed
                    this._selNotifyDelay( true, null );
                }
            }
            return count;
        },

        /**
         * <p>Return the iconList instance if option {@link tableModelView#useIconList} is true, and null otherwise.</p>
         * <p>Note: This returns the instance and not the jQuery object.</p>
         *
         * @return {object} iconList The {@link iconList} widget instance.
         * @example <caption>This example gets the iconList and calls the getColumns method.</caption>
         * $(".selector").tableModelView("getIconList").getColumns();
         */
        getIconList: function() {
            if ( this.options.useIconList && this.data$ ) {
                return this.data$.data("apex-iconList");
            }
            return null;
        },

        //
        // Internal methods
        //
        _getItem: function( el$ ) {
            let itemSelector = this.options.itemSelector;

            // this method should only be called if there is an itemSelector but just to be safe
            if ( itemSelector ) {
                return el$.closest( itemSelector );
            } // else
            return null;
        },

        _initRecordTemplate: function() {
            const o = this.options;

            if ( !o.recordTemplate ) {
                let template, closeTag, href;

                if ( !o.collectionClasses ) {
                    o.collectionClasses = "a-TMV-defaultIconView";
                }
                if ( !o.itemSelector ) {
                    o.itemSelector = ".a-TMV-item";
                }
                if ( !o.labelColumn ) {
                    throw new Error( "Option recordTemplate or labelColumn is required" );
                }

                o.beforeTemplate = "<ul>";
                o.afterTemplate = "</ul>";

                template = `<li ${DATA_ID}='&APEX$ROW_ID.' ${DATA_ROWNUM}='&APEX$ROW_INDEX.' class='a-TMV-item &APEX$ROW_STATE_CLASSES.'>`;
                if ( o.linkTarget || o.linkTargetColumn ) {
                    if ( o.linkTargetColumn ) {
                        href = o.linkTargetColumn;
                    } else {
                        href = "APEX$ROW_URL";
                    }
                    template += `<a href='&${href}.' class='a-IconList-content' ${o.linkAttributes ? o.linkAttributes : ""}>`;
                    closeTag = "</a>";
                } else {
                    template += "<span class='a-IconList-content'>";
                    closeTag = "</span>";
                }
                if ( o.iconClassColumn || o.imageURLColumn ) {
                    template += "<span class='a-IconList-icon'>";
                    if ( o.iconClassColumn ) {
                        template += `<span class='&${o.iconClassColumn}.'></span>`;
                    } else {
                        // todo acc should have alt attr
                        template += `<img src='&${o.imageURLColumn}.' ${o.imageAttributes ? o.imageAttributes : ""}/>`;
                    }
                    template += "</span>";
                }
                template += `<span class='a-IconList-label'>&${o.labelColumn}.</span>${closeTag}</li>`;
                o.recordTemplate = template;
            }

            if ( o.itemNavigation !== NAV_NONE && !o.itemSelector ) {
                debug.warn( "Forced itemNavigation to none because itemSelector not defined" );
                o.itemNavigation = NAV_NONE;
            }
            // just processing the template to test the markup used so no need to supply any data
            let atOptions = this.atOptions;
            atOptions.model = this.model;
            atOptions.record = null;
            clearSubstitutions(atOptions.extraSubstitutions);

            let m = startsWithTagRE.exec( applyTemplate( o.recordTemplate, atOptions ) );
            if ( !m ) {
                if ( applyTemplate( o.beforeTemplate, atOptions ).toLowerCase().match( /<ul|<ol/ ) ) {
                    o.recordTemplate = `<li>${o.recordTemplate}</li>`;
                } else {
                    o.recordTemplate = `<div>${o.recordTemplate}</div>`;
                }
            }
        },

        _initView: function() {
            var m, loadMore$, hdr$, main$,
                self = this,
                ctrl$ = this.element,
                o = this.options,
                curScrollLeft = 0,
                lastScrollLeft = 0,
                timerID = null,
                scrollEvent = "scroll.tmvss",
                out = this._getDataRenderContext(),
                atOptions = this.atOptions,
                substs = atOptions.extraSubstitutions;

            function syncScroll() {
                if ( curScrollLeft !== lastScrollLeft ) {
                    hdr$[0].scrollLeft = curScrollLeft = lastScrollLeft;
                    main$[0].scrollLeft = curScrollLeft;
                    loadMore$.css( "left", curScrollLeft );
                }
                timerID = null;
            }

            atOptions.model = this.model;
            atOptions.record = null; // no record to render here
            clearSubstitutions( substs );

            // rendering the recordTemplate to get the root element would cause any image sources to be fetched
            // right away so use an RE to extract the recordElement
            this.recordElement = "div";
            m = firstTagRE.exec( applyTemplate( o.recordTemplate, atOptions ) );
            if ( m && m.length > 1 ) {
                this.recordElement = m[1].toLowerCase();
            }

            out.markup( "<div" )
                .attr( "class", C_TMV_BODY )
                .markup( ">" );
            this._renderAltDataMessages( out );

            let columns = [];
            for ( const [, col] of util.objectEntries( this.model.getOption( "fields" ) ) ) {
                columns.push(col);
            }
            columns.sort( (a, b) => {
                return a.seq - b.seq;
            } );
            substs.APEX$COLUMNS = columns; // this extra substitution is intentionally not cleared

            if ( o.headerTemplate ) {
                out.markup( `<div class='${C_TMV_HEADER}'>${applyTemplate( o.headerTemplate, atOptions )}</div>` );
            }
            out.markup( "<div" )
                .attr( "class", C_TMV_WRAP_SCROLL )
                .markup( " tabindex='-1'>" )
                    .markup( applyTemplate( o.beforeTemplate, atOptions ) )
                        .markup( `<${this.recordElement} class='js-data-placeholder' style='display:none;'></${this.recordElement}>` )
                    .markup( applyTemplate( o.afterTemplate, atOptions ) );
                this._renderLoadMore( out );
                out.markup( "</div></div>" );
            this._renderFooter( out );

            ctrl$.html( out.toString() );

            this.data$ = ctrl$.find( ".js-data-placeholder" ).parent();
            this.data$.empty().addClass( o.collectionClasses || "" );
            this._clearItemCache();

            if ( o.useIconList ) {
                this.data$.iconList( o.iconListOptions );
                this.data$.on("iconlistselectionchange", function( event ) {
                    self._updateStatus();
                    self._trigger( EVENT_SELECTION_CHANGE, event );
                } );
            }

            main$ = ctrl$.find( SEL_TMV_WRAP_SCROLL );
            hdr$ = ctrl$.find( SEL_TMV_HEADER );
            this._initPagination( hdr$, ctrl$.find( SEL_TMV_WRAP_SCROLL ) );

            if ( o.headerTemplate && o.syncHeaderHScroll ) {
                // coordinate the scrolling of the various areas
                loadMore$ = this.data$.find( SEL_LOAD_MORE );
                main$.on( scrollEvent, function () {
                    lastScrollLeft = this.scrollLeft;
                    if ( !timerID ) {
                        timerID = util.invokeAfterPaint( syncScroll );
                    }
                } );
                hdr$.on( scrollEvent, function () {
                    lastScrollLeft = this.scrollLeft;
                    if ( !timerID ) {
                        timerID = util.invokeAfterPaint( syncScroll );
                    }
                } );
            }
        },

        _setFocusable: function ( el ) {
            const tabbableContent = this.options.tabbableContent;
            let focusableContentLen, item$;

            if ( this.lastFocused && this.lastFocused !== el ) {
                // remove old tab stops
                if ( tabbableContent ) {
                    item$ = this._getItem( $( this.lastFocused ) );
                    item$.add( item$.find( tabbableContent ).not( ":disabled" ) ).prop( "tabIndex", -1 );
                } else {
                    this.lastFocused.tabIndex = -1;
                }
            }

            if ( tabbableContent ) {
                item$ = this._getItem( $( el ) );
                focusableContentLen = item$.find( tabbableContent ).not( "disabled" ).prop( "tabIndex", 0 ).length;
                if ( !focusableContentLen ) {
                    // otherwise focus the item
                    item$.prop( "tabIndex", 0 );
                }
            } else {
                el.tabIndex = 0;
            }
            this.lastFocused = el;
        },

        _getItemFocusable: function( item$ ) {
            const tabbableContent = this.options.tabbableContent;

            if ( tabbableContent ) {
                let a$ = item$.find( tabbableContent );

                if ( a$.length ) {
                    return a$[0];
                } // else
            }
            return item$[0];
        },

        _initRecordMetrics: function() {
            let r$, tempRecords$,
                recordHeight = 0,
                o = this.options,
                substs = this.atOptions.extraSubstitutions;

            clearSubstitutions( substs );
            this.atOptions.record = null; // no data to render here
            this.recPerRow = 1;
            r$ = this.data$.children().filter( SEL_VISIBLE ).first();
            if ( !r$.length ) {
                let i,
                    markup = "";
                // assume won't be more than 20 records in a row
                for ( i = 0; i < 20; i++ ) {
                    markup += applyTemplate( o.recordTemplate, this.atOptions );
                }
                tempRecords$ = $( markup );
                r$ = this.data$.append( tempRecords$ ).children().filter( SEL_VISIBLE ).first();
                if ( o.useIconList ) {
                    // can't just resize because need fixup to list items that refresh does
                    this.getIconList().refresh();
                }
            }

            if ( o.useIconList ) {
                this.recPerRow = this.getIconList().getColumns();
            } else {
                let cur$ = r$,
                    initOffset = cur$.length > 0 ? Math.trunc( cur$.offset().top ) : 0,
                    lastOffset = initOffset,
                    count = 0;

                while ( lastOffset === initOffset && cur$[0] ) {
                    count += 1;
                    cur$ = cur$.next();
                    if ( cur$[0] ) {
                        lastOffset = Math.trunc( cur$.offset().top );
                    }
                }
                this.recPerRow = count;
                recordHeight = lastOffset - initOffset;
            }
            if ( this.recPerRow < 1 ) {
                this.recPerRow = 1; // just in case, don't let this be zero otherwise pageSize could be NaN due to divide by 0
            }
            if ( recordHeight === 0 ) {
                recordHeight = r$.outerHeight( true ); // include margins
                if ( this.data$.css( "display" ) === "grid" ) {
                    // include grid layout grid gap as part of row height
                    recordHeight += toInteger( this.data$.css( "grid-row-gap" ) );
                }
            }
            this.recordHeight = recordHeight || SAFE_DEFAULT_ROW_HEIGHT;

            // if dummy records were rendered just for the purpose of measuring remove them now
            if ( tempRecords$ ) {
                tempRecords$.remove();
            }
        },

        _clearItemCache: function() {
            // clear the cache of children items
            this.itemsCache$ = null;
        },

        _itemIndex: function( item$ ) {
            let tableChildren$;

            // cache the children collection because it is used often
            // do both so the cache is available as a side effect
            if ( !this.itemsCache$ ) {
                this.itemsCache$ = this.data$.children();
            }
            tableChildren$ = this.itemsCache$;
            return tableChildren$.index( item$ );
        },

        // event object is modified
        _navToReportStart( event, delayNotify ) {
            const o = this.options,
                select = o.itemNavigation === NAV_SELECTION;

            // when go to select don't consider the control key
            event.ctrlKey = false;
            if ( o.pagination.scroll ) {
                // don't ever expect there to be a filler row at the beginning
                this.firstPage();
            }

            let next$ = this.data$.children( o.itemSelector ).filter( SEL_VISIBLE ).first();

            if ( next$[0] ) {
                if ( select ) {
                    this._select( next$, event, true, delayNotify );
                } else {
                    $( this._getItemFocusable( next$ ) ).focus();
                }
            }
        },

        // event object is modified
        _navToReportEnd: function( event, delayNotify ) {
            const o = this.options,
                select = o.itemNavigation === NAV_SELECTION;

            const finish = () => {
                if ( select ) {
                    this._select( next$, {
                        type:"keydown",
                        shiftKey: event.shiftKey
                    }, true, delayNotify );
                } else {
                    $( this._getItemFocusable( next$ ) ).focus();
                }
            };

            // when go to select don't consider the control key
            event.ctrlKey = false;
            if ( o.pagination.scroll ) {
                this.lastPage();
            }
            let next$ = this.data$.children( SEL_VISIBLE ).last();
            // in the case of scroll paging there may not be any rows at the end yet
            if ( next$.hasClass( C_GRID_SCROLL_FILLER ) ) {
                next$ = null;
                // so wait for the page to load
                this.element.one( "tablemodelviewpagechange", () => {
                    next$ = this.data$.children( o.itemSelector ).filter( SEL_VISIBLE ).last();
                    finish();
                } );
            } else {
                next$ = next$.filter( o.itemSelector ).last();
                finish();
            }
        },

        _select: function ( items$, event, focus, delayTrigger, noNotify ) {
            const o = this.options;
            let prevSelected, toFocus, selectableItems$,
                selectable, doNotify, modelRangeAnchor, prevSel$,
                actionSelectAll = false, // todo
                checked = null,
                action = SEL_ACTION_SET;

            const updateModelSelectionState = ( rows$, selectState ) => {
                const updateForRows = rows$ => {
                    rows$.each( ( i, el ) => {
                        let id = $( el ).attr( DATA_ID );

                        if ( id ) {
                            this.model.setSelectionState( id, selectState, action );
                        }
                    } );
                };

                if ( o.persistSelection && !actionSelectAll ) {
                    updateForRows( rows$ );
                    if ( action !== SEL_ACTION_TOGGLE && rows$.length > 1 ) {
                        updateForRows( selectableItems$.first() ); // to set the anchor
                    }
                }
            };

            // un-selecting is allowed for invisible items so don't filter them out
            if ( event !== SEL_ACTION_UNSET ) { // xxx unset is not yet used
                // can't select something that isn't visible
                items$ = items$.filter( SEL_VISIBLE );
            }

            // calculate selection action
            if ( event && o.multiple ) {
                if ( typeof event === "string" ) {
                    action = event; // override normal event processing
                } else if ( event.type === "click" ) {
                    // control+click for Windows and command+click for Mac
                    if ( event.ctrlKey || event.metaKey ) {
                        action = SEL_ACTION_TOGGLE;
                    } else if ( event.shiftKey ) {
                        action = SEL_ACTION_RANGE;
                    }
                } else if ( event.type === "keydown" ) {
                    // Mac has no concept of toggle with the keyboard
                    if ( event.which === keys.SPACE ) {
                        if ( event.ctrlKey ) {
                            action = SEL_ACTION_TOGGLE;
                        } else if ( event.shiftKey ) {
                            action = SEL_ACTION_RANGE;
                        } else {
                            action = SEL_ACTION_ADD;
                        }
                    } else if ( event.ctrlKey ) {
                        action = SEl_ACTION_NONE;
                    } else if ( event.shiftKey ) {
                        action = SEL_ACTION_RANGE;
                    }
                }
                // if there is no target it is a fake event so get rid of it so not used in notification
                if ( !event.target ) {
                    event = null;
                }
            }

            if ( o.persistSelection ) {
                modelRangeAnchor = this.model.getSelectionState().rangeAnchor;
            }
            if ( action === SEL_ACTION_ALL ) { // todo selectAll
                actionSelectAll = true;
                action = SEL_ACTION_SET;
            } else if ( action === SEL_ACTION_RANGE && ( !this.selectAnchor && !modelRangeAnchor ) ) {
                action = SEL_ACTION_SET; // when there is no anchor turn range selection into set
            }

            // clear out previous selection if needed
            if ( action === SEL_ACTION_SET || action === SEL_ACTION_RANGE ) {
                prevSel$ = this.data$.find( SEL_SELECTED );
                prevSel$.removeClass( C_SELECTED ).removeAttr( ARIA_SELECTED )
                    .find( SEL_SELECTOR ).removeClass( C_SELECTED );

                if ( o.persistSelection ) {
                    this.model.clearSelection();
                }
            }

            // perform selection action
            if ( o.persistSelection && actionSelectAll ) {
                this.model.setSelectionState( null, true, SEL_ACTION_ALL );
            }

            prevSelected = items$.hasClass( C_SELECTED );
            selectableItems$ = items$; // xxx currently nothing to filter out? perhaps only itemSelector?
            if ( action === SEL_ACTION_SET || action === SEL_ACTION_ADD || ( action === SEL_ACTION_TOGGLE && !prevSelected ) ) {
                selectableItems$.addClass( C_SELECTED ).attr( ARIA_SELECTED, true );
                updateModelSelectionState( selectableItems$, true );
                this.selectAnchor = selectableItems$[0];
                checked = true;
            } else if ( action === SEL_ACTION_RANGE ) {
                let start, end, temp, modelAnchorRow,
                    anchor$ = $( this.selectAnchor ),
                    end$ = items$.last();

                start = this._itemIndex( anchor$ );
                if ( o.persistSelection ) {
                    // the model has its own anchor which should take priority over the view anchor if any
                    let meta = this.model.getRecordMetadata( modelRangeAnchor );

                    if ( meta && meta.serverOffset ) {
                        modelAnchorRow = meta.serverOffset + 1;
                        if ( modelAnchorRow !== toInteger( anchor$.attr( DATA_ROWNUM ) ) ) {
                            start = -1;
                        }
                    }
                }
                end = this._itemIndex( end$ );
                if ( start < 0 ) {
                    let anchorRow = modelAnchorRow || toInteger( anchor$.attr( DATA_ROWNUM ) ),
                        endRow = toInteger( end$.attr( DATA_ROWNUM ) );

                    if ( o.pagination.virtual ) {
                        // with virtual scroll paging the anchor may not be in the DOM
                        let fillers = this.data$.find( SEL_GRID_SCROLL_FILLER ).toArray();

                        // move anchor to next closest row that is in the DOM
                        if ( endRow > anchorRow ) {
                            for ( let i = 0; i < fillers.length; i++ ) {
                                anchor$ = $( fillers[i] ).next();
                                if ( toInteger( anchor$.attr( DATA_ROWNUM ) ) > anchorRow ) {
                                    break;
                                }
                            }
                        } else {
                            for ( let i = fillers.length - 1; i >= 0; i-- ) {
                                anchor$ = $( fillers[i] ).prev();
                                if ( toInteger( anchor$.attr( DATA_ROWNUM ) ) < anchorRow ) {
                                    break;
                                }
                            }
                        }
                        start = this._itemIndex( anchor$ );
                    } else if ( !o.pagination.scroll && o.persistSelection ) {
                        // with traditional paging and selection kept in the model the anchor could be on a different page
                        anchor$ = this.itemsCache$[endRow > anchorRow ? "first" : "last"]();
                        start = this._itemIndex( anchor$ );
                    }
                }

                if ( o.persistSelection ) {
                    /* xxx consider control break template
                    if ( end$.hasClass( C_CONTROL_BREAK ) ) {
                        // control breaks are not actual rows in the model so use the adjacent one
                        end$ = end$[start > end ? "next" : "prev"]();
                    } */
                    updateModelSelectionState( end$, true ); // for range only need to update the last record
                }

                if ( start > end ) {
                    temp = end;
                    end = start;
                    start = temp;
                }
                end += 1;

                selectable = [];
                // because _itemIndex called can be sure itemsCache$ exist
                this.itemsCache$.slice( start, end ).each( function ( index, el ) {
                    let el$ = $( el );

                    if ( el$.is( SEL_VISIBLE ) ) {
                        selectable.push( el );
                    }
                } );
                selectableItems$ = $( selectable );
                selectableItems$.addClass( C_SELECTED ).attr( ARIA_SELECTED, true );
                checked = true;
            } else if ( ( action === SEL_ACTION_TOGGLE && prevSelected ) || action === SEL_ACTION_UNSET ) {
                selectableItems$.removeClass( C_SELECTED ).removeAttr( ARIA_SELECTED );
                updateModelSelectionState( selectableItems$, false );
                this.selectAnchor = selectableItems$[0];
                checked = false;
            }

            if ( checked !== null ) {
                selectableItems$.find( SEL_SELECTOR ).toggleClass( C_SELECTED, checked );
            }
            if ( o.multiple && o.selectAll ) { // todo selectAll not yet implemented
                this._updateSelectAll();
            }

            // focus if needed
            if ( items$.length > 0 && focus !== null ) {
                toFocus = this._getItemFocusable( items$.first() );
                if ( focus ) {
                    $( toFocus ).focus();
                } else {
                    this._setFocusable( toFocus );
                }
            }

            if ( o.persistSelection ) {
                this.hasSelection = this.model.getSelectedCount() > 0; // todo if only care about selection or not and not how many there should be a more efficient way
            }

            // notify if needed
            if ( action === SEL_ACTION_TOGGLE ||
                action === SEL_ACTION_UNSET || // assuming at least some of the things to unset were previously selected
                (action === SEL_ACTION_RANGE && !prevSelected ) ||
                action === SEL_ACTION_ADD || // assuming at least some of the things to add were previously not selected
                (action === SEL_ACTION_SET && !util.arrayEqual( prevSel$, selectableItems$ ) ) ) { // todo xxx what if selection differs in model only

                doNotify = !(noNotify || ( o.navigation && event && event.type === "click" )); // xxx navigation is not an option. Is this for links?
                delayTrigger ? this._selNotifyLongDelay( doNotify, event ) : this._selNotifyDelay( doNotify, event );
            }
        },

        // don't call directly go through _selNotifyDelay, _selNotifyLongDelay
        _selChangeNotify: function( notify, event ) {
            if ( this.element.hasClass( C_TMV ) ) { // make sure the tableModelView widget has not been destroyed
                this._updateStatus();
                if ( notify ) {
                    this._trigger( EVENT_SELECTION_CHANGE, event );
                }
            }
        },

        //
        // Methods to override
        //

        _getHeaderHeight: function() {
            return this.element.find( SEL_TMV_HEADER ).outerHeight();
        },

        // This is used when all records are the same height, which means all rows will be the same height.
        // It is also used as a starting point to calculate the average row height.
        _getFixedRecordHeight: function() {
            if ( !this.recordHeight ) {
                this._initRecordMetrics();
            }
            return this.recordHeight;
        },

        _getRecordsPerRow: function() {
            if ( !this.recPerRow ) {
                this._initRecordMetrics();
            }
            return this.recPerRow;
        },

        _getDataContainer: function() {
            return this.data$;
        },

        _selectedCount: function() {
            let o = this.options;

            if ( o.useIconList ) {
                return this.getIconList().getSelection().length;
            } else if ( o.itemNavigation === NAV_SELECTION ) {
                return this.getSelectedRecords().length;
            } // else
            return 0;
        },

        _hasControlBreaks: function() {
            // todo think it is one thing to have a break template and another or the data to actually have breaks in it.
            return this.options.controlBreakTemplate != null;
        },

        _elementsFromRecordIds: function( ids ) {
            let elements = [];

            for ( let i = 0; i < ids.length; i++ ) {
                let item = this.data$[0].querySelector( `[${DATA_ID}="${util.escapeCSS(ids[i])}"]` ); // using querySelector for performance

                if ( item ) {
                    elements.push( $( item ) );
                }
            }
            return elements;
        },

        _renderFillerRecord: function ( out, cssClass ) {
            out.markup( `<${this.recordElement} class='${cssClass}'></${this.recordElement}>` );
        },

        _insertFiller: function( out, curFiller$ ) {
            let filler$ = $( out.toString() );

            if ( curFiller$ ) {
                curFiller$.before( filler$ );
            } else {
                this.data$.html( filler$ );
            }
            this._clearItemCache();
            return filler$;
        },

        _initItems: function( items$ ) {
            const o = this.options;

            if ( o.itemNavigation !== NAV_NONE ) {
                // the items should match the itemSelector class
                items$.attr( A_TABINDEX, -1 )
                    .find( o.tabbableContent ).attr( A_TABINDEX, -1 ); // todo think use js-tabbable class or not?
            }
        },

        _insertData: function( out, offset, count, filler$, how ) {
            const o = this.options;
            let items$ = $( out.toString() );

            if ( !filler$ ) {
                this.data$.append( items$ );
            } else {
                // else must have filler$ and how must be before or after
                filler$[how]( items$ );
            }
            this._clearItemCache();

            // todo when support selectAll may need to select items here
            this._initItems( items$ );

            if ( o.useIconList ) {
                let vp$ = this.element.find( SEL_TMV_WRAP_SCROLL ),
                    top = vp$.scrollTop(),
                    iconList = this.getIconList();

                iconList.refresh();
                vp$.scrollTop( top ); // restore the scroll offset that was changed by refresh
            }
            // after all records rendered set to clear cache for next time
            this.atOptions.clearItemCache = true;

            this._super( out, offset, count, filler$, how );
        },

        _controlBreakLabel: function( record ) {
            return {
                record: record
            };
        },

        _renderBreakRecord: function ( out, expandControl, breakData, serverOffset ) {
            var itemMarkup,
                o = this.options,
                atOptions = this.atOptions,
                substs = atOptions.extraSubstitutions;

            clearSubstitutions( substs );
            // todo what if anything to do with expandControl or serverOffset?
            atOptions.model = this.model;
            atOptions.record = breakData.record;
            substs.APEX$ROW_INDEX = serverOffset != null ? "" + ( serverOffset + 1 ) : "";

            itemMarkup = applyTemplate( this.options.controlBreakTemplate, atOptions );
            // todo does it make sense to run the highlighter here?
            if ( o.highlighter ) {
                itemMarkup = o.highlighter( o.highlighterContext, itemMarkup );
            }
            // todo think is interactive content allowed in break record?
            out.markup( itemMarkup );
            // after rendering the first record don't clear the item cache again until all records are rendered.
            this.atOptions.clearItemCache = false;
        },

        _renderRecord: function( out, record, index, id, meta ) {
            const o = this.options;
            let highlight, itemMarkup,
                template = o.recordTemplate,
                cls = "",
                atOptions = this.atOptions,
                substs = atOptions.extraSubstitutions;

            clearSubstitutions( substs );

            if ( meta.agg ) {
                if ( o.aggregateTemplate ) {
                    template = o.aggregateTemplate;
                    // add special symbols
                    substs.APEX$AGG = meta.agg;
                    substs.APEX$AGG_TOTAL = meta.grandTotal ? "Y" : "";
                } else {
                    // aggregate records are not supported when there is no template
                    return;
                }
            } else {
                // add special symbols
                if ( meta.url ) {
                    substs.APEX$ROW_URL = meta.url;
                }
                if ( meta.hidden ) {
                    cls += " u-hidden";
                }
                if ( meta.sel ) {
                    cls += " " + C_SELECTED;
                }
                if ( meta.deleted ) {
                    cls += " " + C_DELETED;
                } else {
                    if ( meta.inserted ) {
                        cls += " is-inserted";
                    } else if ( meta.updated ) {
                        cls += " is-updated";
                    }
                    if ( meta.error ) {
                        cls += " is-error";
                    } else if ( meta.warning ) {
                        cls += " is-warning";
                    }
                    if ( meta.highlight ) {
                        highlight = o.highlights[meta.highlight];
                        if ( highlight && highlight.cssClass ) {
                            cls += " " + highlight.cssClass;
                        } else {
                            cls += " " + meta.highlight;
                        }
                    }
                    if ( meta.message ) {
                        substs.APEX$VALIDATION_MESSAGE = meta.message;
                    }
                }
                if ( cls ) {
                    substs.APEX$ROW_STATE_CLASSES = cls;
                }
            }

            // add special symbols
            substs.APEX$ROW_ID = "" + id;
            substs.APEX$ROW_INDEX = "" + ( index + 1 );

            atOptions.model = this.model;
            atOptions.record = record;
            itemMarkup = applyTemplate( template, atOptions );
            if ( o.highlighter && !meta.agg ) {
                itemMarkup = o.highlighter( o.highlighterContext, itemMarkup );
            }
            out.markup( itemMarkup );
            // after rendering the first record don't clear the item cache again until all records are rendered.
            this.atOptions.clearItemCache = false;
        },

        _removeRecord: function( element ) {
            element.remove();
            this._clearItemCache();
        },

        _insertRecord: function( element, record, id, meta, where ) {
            const out = util.htmlBuilder();

            this._renderRecord( out, record, null, id, meta );
            this.atOptions.clearItemCache = true;
            let newItem$ = $( out.toString() );

            if ( where === INSERT_AFTER && element ) {
                element.after( newItem$ );
            } else {
                this.data$.prepend( newItem$ );
            }
            this._clearItemCache();
            this._initItems( newItem$ );
            if ( this.options.useIconList ) {
                this.getIconList().refresh(); // xxx this messes up the scroll offset
                // xxx any way to delay until visible?
            }
            return newItem$;
        },

        _afterInsert: function( /* insertedElements, copy */ ) {
        },

        _identityChanged: function( prevId, id ) {
            var rec, meta,
                element$ = this._elementsFromRecordIds( [prevId] )[0];

            if ( element$ ) {
                rec = this.model.getRecord( id );
                meta = this.model.getRecordMetadata( id );
                // Don't know details of presentation so can't update identity. Just render record and replace.
                this._replaceRecord( element$, rec, prevId, id, meta );
            }
        },

        _replaceRecord: function( element, record, oldId, id, meta ) {
            const out = util.htmlBuilder();
            let item$,
                index = element.attr( DATA_ROWNUM ) || null;

            if ( index !== null ) {
                index = toInteger( index ) - 1;
            }
            this._renderRecord( out, record, index, id, meta );
            this.atOptions.clearItemCache = true;
            item$ = $( out.toString() );
            element.replaceWith( out.toString() );
            this._clearItemCache();
            this._initItems( item$ );
            if ( this.options.useIconList ) {
                this.getIconList().refresh(); // xxx this messes up the scroll offset
                this.getIconList().setSelection($('[data-id='+ meta.record.id +']')); // set record in list to the matching ID after refresh
                this.focus();
                // xxx any way to delay until visible?
            } else {
                // xxx BLASI not sure if I'll need to do anything here yet
            }
        },

        // _updateRecordField: function( element, record, field, meta )
        // this view doesn't know about fields or cells so can't update them but the whole item is updated due to _updateRecordState

        _removeFocus: function( /* element */ ) {
            // todo this will only affect icon view
        },

        _checkAllHidden: util.debounce( function() {
            const pagination = this.options.pagination;

            if ( !this.data$.children().is( ":visible" ) ) {
                // no items are visible so switch to showing no data
                this.noData$.show();
                this.data$.hide();
                this.noData = true;
            } else {
                // at least one item is visible
                // but some may be hidden so check if need to render more items
                if ( pagination.scroll && !pagination.loadMore && !pagination.virtual) {
                    if ( this.data$.height() < this.scrollParent$.height() ) {
                        // If add more scrolling it is possible for visible items to be smaller than the viewport but still
                        // not have all the data. In this case it is not possible to add more data because can't scroll.
                        this._addNextPage();
                    }
                }
                // Note client side filtering works best when all the records are rendered. This means it makes
                // no sense for pagination.page unless the rowsPerPage includes all records. It does not work
                // for virtual scroll pagination. It works somewhat for non-virtual scroll pagination. For
                // load more it works but it can get tedious clicking on load more button.
                // Pagination footer info may be confusing while filtering
                // Doesn't work with highlighter because record not re-rendered
                // TODO doc these limitations
                // TODO need to keep adding pages in _addPageOfRecords if the viewport is not full
            }
        }, 10 ),

        _updateRecordState: function( element, id, record, meta, property ) {
            if ( property === "hidden" ) {
                // just showing or hiding.
                if ( this.noData && !meta.hidden ) {
                    // was showing no data and now something is visible so switch to showing data
                    this.noData$.hide();
                    this.data$.show();
                    this.noData = false;
                }
                element.toggleClass( "u-hidden", meta.hidden );
                if ( !this.noData ) {
                    // check if all items are hidden after all the calls to update the hidden state
                    this._checkAllHidden();
                }
                return;
            } // else
            // Don't know details of presentation so can't update state. Just render record and replace.
            this._replaceRecord( element, record, id, id, meta );
        }
    });

})( apex.util, apex.model, apex.debug, apex.lang, apex.item, apex.jQuery );
