/*!
 * Copyright (c) 2021, 2022, Oracle and/or its affiliates.
 */
/**
 * @fileOverview
 * miniMap is a basic implementaion of the underlying maplibregl library to display maps pointing to given coordinates
 *
 * The markup expected by this widget is simply a empty div element. Optionally all provided options could also
 * be set by using data-* attributes.
 *
 * Supported options:
 * - background:        String which supplies name of the background map,
 *                      "default" uses "osm-bright" and for dark mode "osm-dark-matter"
 *                      Valid values: "default", "osm-bright", "osm-positron", "bi-world-map",
 *                                    "osm-dark-matter", "world-map"
 * - center:            An array containing the coordinates with latitude & longitude
 *                      [ longitude, latitude ], [ -122.26516, 37.52938 ]
 * - zoom:              [Defaults 16] The zoom level of the map, range between 0 and 18
 *                      additional values will add to the existing selection.
 * - marker:            [Defaults true] Whether to show a marker or not
 * - markerColor:       Color of the marker, if empty default color of maplibre is used
 * - controls:          [Defaults false] Whether to show controls or not
 * - interactive:       [Defaults true] Whether to have an interactive map or not
 * - tooltip:           If supplied, shows a tooltip popup above a marker, which is visible when the marker was clicked
 *                      The tooltip also supports HTML markup and APEX template directives
 *
 * File URL to reference: #APEX_FILES#libraries/apex/#MIN_DIRECTORY#widget.miniMap#MIN#.js
 *
 * Open Issues:
 *
 * Future:
 *
 *
 * Depends:
 *      jquery-3.5.1.js
 *      jquery.ui.core.js
 *      jquery.ui.widget.js
 *      core.js
 *      debug.js
 *      maplibre-gl.js
 *
 * @example <caption>Create miniMap using options</caption>
 *
 * <div id="myMap" style="width:100%;height:300px"></div>
 *
 * $( "#myMap" ).miniMap( {
 *     center: [ -122.26516, 37.52938 ],
 *     background: "osm-bright",
 *     zoom: 14,
 *     marker: true,
 *     markerColor: "red",
 *     controls: true,
 *     interactive: true,
 *     tooltip: "Oracle Redwood Shores"
 * } );
 *
 * @example <caption>Create miniMap using data attributes</caption>
 *
 * <div id="myMap"
 *     data-center="[-122.26516, 37.52938]"
 *     data-background="osm-bright"
 *     data-zoom="14"
 *     data-marker="true"
 *     data-marker-color="red"
 *     data-controls="true"
 *     data-interactive="true"
 *     data-tooltip="Oracle Redwood Shores"
 *     style="width:100%;height:300px">
 * </div>
 *
 * $( "#myMap" ).miniMap();
 */

/* global maplibregl */

( function ( $, widget, util, env, lang, server, debug, libVersions ) {
    "use strict";

    // Constants
    var MINIMAP = "miniMap",
        MINIMAP_CLASS = "apex-minimap",
        MAPLIBRE_BASE_PATH = env.APEX_FILES + `libraries/maplibre-gl-js/${libVersions.maplibre}/`,
        DEFAULT_MARKER_COLOR = "var(--a-palette-primary, #006BD8)",
        DEFAULT_ELOCATION = "elocation-osm-bright",
        ELOCATIONS = {
            version: 8,
            id: "elocation-osm-raster-tiles",
            name: "OSM Bright (Oracle)",
            metadata: {
                "sgtech:version": "19.1.0",
                "sgtech:sources:type": "raster"
            },
            sources: {
                "elocation-osm-bright": {
                    type: "raster",
                    attribution: "",
                    tiles: ["https://elocation.oracle.com/appidGC000110/mapviewer/mcserver/ELOCATION_MERCATOR/osm_bright/{z}/{y}/{x}.png"],
                    tileSize: 256
                },
                "elocation-osm-positron": {
                    type: "raster",
                    attribution: "",
                    tiles: ["https://elocation.oracle.com/appidGC000110/mapviewer/mcserver/ELOCATION_MERCATOR/osm_positron/{z}/{y}/{x}.png"],
                    tileSize: 256
                },
                "elocation-bi-world-map": {
                    type: "raster",
                    attribution: "",
                    tiles: ["https://elocation.oracle.com/appidGC000110/mapviewer/mcserver/ELOCATION_MERCATOR/bi_world_map_mb/{z}/{y}/{x}.png"],
                    tileSize: 256
                },
                "elocation-osm-dark-matter": {
                    type: "raster",
                    attribution: "",
                    tiles: ["https://elocation.oracle.com/appidGC000110/mapviewer/mcserver/ELOCATION_MERCATOR/osm_darkmatter/{z}/{y}/{x}.png"],
                    tileSize: 256
                },
                "elocation-world-map": {
                    type: "raster",
                    attribution: "",
                    tiles: ["https://elocation.oracle.com/appidGC000110/mapviewer/mcserver/ELOCATION_MERCATOR/world_map_mb/{z}/{y}/{x}.png"],
                    tileSize: 256
                }
            },
            layers: [
                {
                    id: "elocation-osm-bright-layer",
                    type: "raster",
                    source: DEFAULT_ELOCATION,
                    minzoom: 0,
                    maxzoom: 18.01
                }
            ]
        };

    $.widget( "apex." + MINIMAP, {
        version: "21.2",
        widgetEventPrefix: MINIMAP,
        options: {
            background: "default",
            center: [0, 0],
            zoom: 16,
            marker: true,
            markerColor: "",
            controls: false,
            interactive: true,
            tooltip: ""
        },
        _create: function () {
            var el$ = this.element;

            if ( el$.hasClass( MINIMAP_CLASS ) ) {
                debug.warn( "Map is already initialized." );
                return;
            }

            this._setAutoHeight();

            this._createMap();
            el$.addClass( MINIMAP_CLASS );
        },
        _getCreateOptions: function () {
            var options = {},
                el$ = this.element;

            if ( el$[0].hasAttribute( "data-background" ) ) {
                options.background = el$.data( "background" );
            }
            if ( el$[0].hasAttribute( "data-center" ) ) {
                options.center = el$.data( "center" );
            }
            if ( el$[0].hasAttribute( "data-zoom" ) ) {
                options.zoom = el$.data( "zoom" );
            }
            if ( el$[0].hasAttribute( "data-marker" ) ) {
                options.marker = el$.data( "marker" );
            }
            if ( el$[0].hasAttribute( "data-marker-color" ) ) {
                options.markerColor = el$.data( "marker-color" );
            }
            if ( el$[0].hasAttribute( "data-controls" ) ) {
                options.controls = el$.data( "controls" );
            }
            if ( el$[0].hasAttribute( "data-interactive" ) ) {
                options.interactive = el$.data( "interactive" );
            }
            if ( el$[0].hasAttribute( "data-tooltip" ) ) {
                options.tooltip = el$.data( "tooltip" );
            }
            return options;
        },
        _setOption: function ( key, value ) {
            var el$ = this.element;

            this._super( key, value );

            if ( !this.map ) {
                debug.warn( "miniMap map object is not initialized yet." );
                return;
            }

            switch ( key ) {
            case "background":
                this.map.setStyle( this._getElocation() );
                if ( el$[0].hasAttribute( "data-background" ) ) {
                    el$.data( "background", value );
                }
                break;
            case "center":
                this.map.setCenter( value );
                if ( this.marker ) {
                    this.marker.setLngLat( value ).addTo( this.map );
                }
                if ( el$[0].hasAttribute( "data-center" ) ) {
                    el$.data( "center", value );
                }
                break;
            case "zoom":
                this.map.setZoom( value );
                if ( el$[0].hasAttribute( "data-zoom" ) ) {
                    el$.data( "zoom", value );
                }
                break;
            case "marker":
                if ( this.marker ) {
                    if ( !value ) {
                        this.marker.remove();
                        this.marker = null;
                        this.popup = null;
                    }
                } else {
                    if ( value ) {
                        this._createMapMarker();
                    }
                }
                if ( el$[0].hasAttribute( "data-marker" ) ) {
                    el$.data( "marker", value );
                }
                break;
            case "markerColor":
                if ( this.marker ) {
                    this.marker.remove();
                    this.marker = null;
                    this.popup = null;
                    this._createMapMarker();
                }
                if ( el$[0].hasAttribute( "data-marker-color" ) ) {
                    el$.data( "marker-color", value );
                }
                break;
            case "tooltip":
                if ( this.popup ) {
                    if ( value ) {
                        this.popup.setHTML( util.applyTemplate( value ) );
                    } else {
                        this.popup.setText( "" );
                        this.popup.remove();
                        this.popup = null;
                    }
                } else {
                    if ( value ) {
                        if ( this.marker ) {
                            this._createMapPopup();
                        }
                    }
                }
                if ( el$[0].hasAttribute( "data-tooltip" ) ) {
                    el$.data( "tooltip", value );
                }
                break;
            case "controls":
                if ( this.controls ) {
                    if ( !value ) {
                        this.map.removeControl( this.controls );
                        this.controls = null;
                    }
                } else {
                    if ( value ) {
                        this._createMapControls();
                    }
                }
                if ( el$[0].hasAttribute( "data-controls" ) ) {
                    el$.data( "controls", value );
                }
                break;
            case "interactive":
                this.map.remove();
                this.map = null;
                this.marker = null;
                this.popup = null;
                this.controls = null;
                this.attribution = null;
                this._createMap();
                if ( el$[0].hasAttribute( "data-interactive" ) ) {
                    el$.data( "interactive", value );
                }
                break;
            }
        },
        _destroy: function () {
            this.map.remove();
            this.element.empty();
            this.element.removeClass( MINIMAP_CLASS );
            this.map = null;
            this.marker = null;
            this.popup = null;
            this.controls = null;
            this.attribution = null;
        },
        _setAutoHeight: function() {
            var el$ = this.element;

            // set automatic height (3:2 aspect ratio) if not provided as inline style
            // set timeout because theme adjustments to displayed width during page load
            if ( !el$[0].style.height || el$.height() === 0 ) {
                setTimeout( function () {
                    el$.css( "height", el$.width() * ( 2 / 3 ) );
                }, 20 );
            }
        },
        _loadMaplibreFiles: function () {
            var deferred = $.Deferred();

            function _loadScript( callback ) {
                if ( typeof require === "undefined" ) {
                    server.loadScript(
                        {
                            path: MAPLIBRE_BASE_PATH + "maplibre-gl.js"
                        },
                        function () {
                            callback();
                        }
                    );
                } else {
                    require.config( {
                        paths: {
                            maplibre: MAPLIBRE_BASE_PATH + "maplibre-gl"
                        },
                        shim: {
                            maplibre: {
                                exports: ["maplibregl"]
                            }
                        }
                    } );

                    require( ["maplibre"], function ( maplibregl ) {
                        window.maplibregl = maplibregl;
                        callback();
                    } );
                }
            }

            if ( typeof maplibregl === "undefined" ) {
                // load css
                if ( $( "link[href='" + MAPLIBRE_BASE_PATH + "maplibre-gl.css']" ).length === 0 ) {
                    $( "<link/>", {
                        rel: "stylesheet",
                        type: "text/css",
                        href: MAPLIBRE_BASE_PATH + "maplibre-gl.css"
                    } ).appendTo( "head" );
                }

                // load js
                _loadScript( function () {
                    deferred.resolve();
                } );
            } else {
                deferred.resolve();
            }
            return deferred.promise();
        },
        _getElocation: function () {
            var options = this.options,
                elocationName = "",
                elocations = $.extend( true, {}, ELOCATIONS );

            function _isDarkMode() {
                return $( "body" ).hasClass( "apex-theme-vita-dark" );
            }

            if ( !options.background || options.background === "default" ) {
                elocationName = _isDarkMode() ? "elocation-osm-dark-matter" : DEFAULT_ELOCATION;
            } else {
                elocationName = "elocation-" + options.background;
            }

            elocations.layers[0].id = elocationName + "-layer";
            elocations.layers[0].source = elocationName;

            return elocations;
        },
        _addCopyrightNotice: function () {
            var options = this.options,
                map = this.map,
                showOSMCopyright = !options.background || options.background === "default" || options.background.includes( "osm" ),
                copyrightMessage = showOSMCopyright ? lang.getMessage( "APEX.MAPS.OSM_MAP_COPYRIGHT" ) : lang.getMessage( "APEX.MAPS.ORACLE_MAP_COPYRIGHT" ),
                toggleMessage = apex.lang.getMessage( "APEX.MAPS.TOGGLE_COPYRIGHT" ),
                attrBtn$;

            this.attribution = new maplibregl.AttributionControl( {
                customAttribution: copyrightMessage
            } );

            map.addControl( this.attribution );

            // When map is small a button will be displayed to toggle copyright text
            attrBtn$ = $( this.attribution._container ).find( "button" );
            attrBtn$.attr( {
                "type": "button",
                "title": toggleMessage,
                "aria-label": toggleMessage
            } );
        },
        _createMapMarker: function () {
            var options = this.options,
                map = this.map;

            if ( options.marker ) {
                this.marker = new maplibregl.Marker( { color: options.markerColor || DEFAULT_MARKER_COLOR } ).setLngLat( options.center ).addTo( map );
                this._createMapPopup();
            }
        },
        _createMapPopup: function () {
            var self = this,
                options = this.options;

            if ( options.marker ) {
                if ( options.tooltip ) {
                    this.popup = new maplibregl.Popup( { offset: 25 } ).setHTML( util.applyTemplate( options.tooltip ) );
                    this.marker.setPopup( this.popup );
                    this.popup.on( "open", function () {
                        if ( self.popup && self.popup.getElement() ) {
                            $( self.popup.getElement() ).css( "color", "#000" );
                        }
                    } );
                }
            }
        },
        _createMapControls: function () {
            var options = this.options,
                map = this.map;

            if ( options.controls ) {
                this.controls = new maplibregl.NavigationControl( {
                    showCompass: options.interactive
                } );
                map.addControl( this.controls );
            }
        },
        _createMap: function () {
            var self = this,
                options = this.options;

            self._loadMaplibreFiles().done( function () {
                self.map = new maplibregl.Map( {
                    container: self.element[0],
                    style: self._getElocation(),
                    center: options.center,
                    zoom: options.zoom,
                    interactive: options.interactive
                } );

                self._createMapMarker();
                self._createMapControls();
                self._addCopyrightNotice();
            } );

            // resize the map when container was resized
            widget.util.onElementResize( self.element[0], function () {
                if ( self.map ) {
                    self.map.resize();
                }
            } );

            // resize the map when an inline dialog is resized
            // widget.util.onElementResize does not work properly in inline dialogs
            // we should consider fixing this more generic or on theme level in future
            self.element.closest( ".js-regionDialog" ).on( "dialogresize", function () {
                if ( self.map ) {
                    self.map.resize();
                }
            } );
            

            // set automatic height when map becomes visible
            widget.util.onVisibilityChange( self.element[0], function () {
                self._setAutoHeight();
                if ( self.map ) {
                    self.map.resize();
                }
            } );
        }
    } );
} )( apex.jQuery, apex.widget, apex.util, apex.env, apex.lang, apex.server, apex.debug, apex.libVersions );
