/*!
 Copyright (c) 2012, 2022, Oracle and/or its affiliates.
*/
/* eslint-env amd */
/*
 * The color picker item of Oracle APEX.
 * There are a few main parts to this:
 * - The APEX item interface and the attach handler that sets up the main item behaviors: colorpickerItemPrototype, attachColorpicker,
 * - A "compound component" consisting of a JET color spectrum and related controls used in a popup or inline: colorSpectrumPickerPrototype, makeColorSpectrumPicker
 * - Function to configure and open the popup: openPopupColorPicker
 * - Color related utilities that rely on JET: initJETColorUtilities
 * - Some contrast utilities exported via: apex.widget.util.colorPicker
 */
/*
 * todo consider including as part of interface description the expected markup including configuration attributes
 *
 * Expected markup
 * Native
 *  <input id="{item-name}" name="{item-name}" type="color" data-display-as="NATIVE"
 *      class="color_picker apex-item-color-picker apex-item-color-picker-native"
 *      value="{initial-color-value}"/>
 *
 * Popup
 *  <input id="{item-name}" name="{item-name}" type="text" data-display-as="POPUP"
 *      class="color_picker apex-item-text apex-item-color-picker"
 *      size="{size}" maxlength="{maxlength}"
 *      value="{initial-color-value}"/>
 *
 * Inline
 *  <input id="{item-name}" name="{item-name}" type="hidden" data-display-as="INLINE"
 *      class="color_picker apex-item-color-picker"
 *      value="{initial-color-value}"/>
 *
 * Color Only
 *  <input id="{item-name}" name="{item-name}" type="hidden" data-display-as="COLOR_ONLY"
 *      class="color_picker apex-item-color-picker"
 *      value="{initial-color-value}"/>
 *
 * In addition all but the native one can have these additional attributes on the input element.
 * data-contrast-item="{other-item-name}" check the contrast of this item color value against the other item
 * data-contrast-color="{color-value}" check the contrast of this item color value against the color value
 * data-return-value-as="{HEX|RGB|RGBA|HSL|HSLA|CSS}" Default is HEX
 * data-display-mode="{SIMPLE|FULL}" Default is SIMPLE
 * data-colors={THEME|color-list} A color list is a ; separated list of color values
 * Advanced:
 * data-popup-class="{classes}"
 * data-colors-inline="true" Only applies for display as POPUP and if have colors
 * data-max-colors="{n}" default is 5. Only applies if have colors
 *
 * The above markup should be wrapped in <div class="apex-item-group apex-item-group--color-picker"></div>
 */

/**
 * @interface colorPickerItem
 * @since 21.2
 * @extends {item}
 * @classdesc
 * <p>The colorPickerItem interface is used to access the properties and methods of the Color Picker item.
 * You get access to the colorPickerItem interface with the {@link apex.item} function when passed
 * the item id (name) of a Color Picker item.
 * </p>
 * <p>The colorPickerItem implementation loads asynchronously (delayLoading is true) because it uses the
 * Oracle JET library for the color spectrum to pick a color. Some functionality such
 * as {@link colorPickerItem#contrastWith} is not available until JET is loaded. When the 'Display As' item configuration
 * is Native Color Picker, JET is not loaded. See {@link colorPickerItem#whenReady}.
 * </p>
 * <p>When the user selects or enters a value it will be formatted according to the 'Return Value As' property.
 * If the page or item is validated on the client, which is done by default before the page is submitted,
 * this item will report validation errors. It validates if the field is required, and that it is a valid color.</p>
 */
(function( item, $, debug, util, lang ) {
    "use strict";

    const keys = $.ui.keyCode,
        isMac = navigator.appVersion.includes("Mac");

    const DISPLAY_AS_INLINE = "INLINE",
        DISPLAY_AS_POPUP = "POPUP",
        DISPLAY_AS_COLOR_ONLY = "COLOR_ONLY",
        DISPLAY_AS_NATIVE = "NATIVE";

    const RETURN_AS_HEX = "HEX", // default
        RETURN_AS_RGB = "RGB",
        RETURN_AS_RGBA = "RGBA",
        RETURN_AS_HSL = "HSL",
        RETURN_AS_HSLA = "HSLA",
        RETURN_AS_CSS = "CSS";

    const colorExamples = {
        HEX: "#3caf85",
        RGB: "rgb(60, 175, 133)",
        RGBA: "rgba(60, 175, 133, 0.5)",
        HSL: "hsl(158, 49%, 46%)",
        HSLA: "hsla(158, 49%, 46%, 0.5)",
        CSS: "#3caf85",
    };

    const P_DISABLED      = "disabled",
        C_HIDDEN          = "u-hidden",
        A_ARIA_EXPANDED   = "aria-expanded";

    const defaultColor    = '000000', // match default for native input type=color which is #000000
        dialogClass       = "ui-dialog-color-picker",
        colorPickerPrefixClass      = "a-ColorPicker",
    // preview classes
        colorPreviewClass           = colorPickerPrefixClass + "-preview",
        colorPreviewCurrentClass    = colorPreviewClass + "--current",
        colorPreviewInitClass       = colorPreviewClass + "--initial",
        colorPreviewValueClass      = colorPreviewClass + "-value",
        colorContrastClass          = colorPickerPrefixClass + "-contrast",
        colorContrastIconClass      = colorContrastClass + "Icon",
        colorContrastResultClass    = colorContrastClass + "Result",
        colorContrastRatingClass    = colorContrastClass + "Rating",
        colorContrastColor1Class    = colorContrastClass + "Color1",
        colorContrastColor2Class    = colorContrastClass + "Color2",
        itemColorPreviewClass       = "apex-item-color-picker-preview",
        itemColorNoPreviewClass     = itemColorPreviewClass + "--noPreview",
    // color detail classes
        colorDetailClass      = colorPickerPrefixClass + "-detail",
        colorDetailRGBClass   = colorDetailClass + "--rgb",
        colorDetailHSLClass   = colorDetailClass + "--hsl",
        colorDetailHexClass   = colorDetailClass + "--hex",

        colorDetailItemClass     = colorDetailClass + "Item",
        colorDetailItemRClass    = colorDetailItemClass + "--r",
        colorDetailItemGClass    = colorDetailItemClass + "--g",
        colorDetailItemBClass    = colorDetailItemClass + "--b",
        colorDetailItemHClass    = colorDetailItemClass + "--h",
        colorDetailItemSClass    = colorDetailItemClass + "--s",
        colorDetailItemLClass    = colorDetailItemClass + "--l",
        colorDetailItemAClass    = colorDetailItemClass + "--a",
        colorDetailItemHexClass  = colorDetailItemClass + "--hex",
        colorDetailLabelClass     = colorDetailClass + "Label",
        colorDetailInputClass    = colorDetailClass + "Input",

        colorDetailToggleClass = colorPickerPrefixClass + "-detailsToggle";
    
    // preset classes
    const colorPresetClass           = colorPickerPrefixClass + "-preset",
          colorPresetMenuButtonClass = colorPresetClass + "MenuButton",
          colorPresetMenuClass       = colorPresetClass + "Menu";
    /*
     * APP UI styles define 15 color preset variables.
     * --a-color-picker-preset-1 to --a-color-picker-preset-15
     * A theme can override the color values.
     */
    const colorPresetMaxCount = 5,
        colorPresetThemeStyleTotal = 15, // Keep in sync with theme presets
        colorPresetCssVariableName  = '--a-color-picker-preset-';

    const BG_COLOR_PROP = "background-color",
        COLOR_PROP = "color",
    // regexp
        HSLA_FN_RE = /hsla?\(\s*(\d+),\s*(\d+)%,\s*(\d+)%(?:,\s*(\d+.\d+))?\s*\)/, // matches hsl(h, s%, l%) or hsla(h, s%, l%, a)
        HEX_RE =     /^(#)?([a-f0-9]{3}|[a-f0-9]{6})$/i,
        RGB_RE =     /^([01]?[0-9]{1,2}|2[0-4][0-9]|25[0-5])$/i,   // between 0 and 255
        ALPHA_RE =   /^([0-9]*\.)?[0-9]{1,2}$/i,
        HUE_RE =     /^([012]?[0-9]{1,2}|3[0-5][0-9])$/i,          // between 0 and 360
        SAT_LUM_RE = /^([0]?[0-9]{1,2}|100)$/i;                   // between 0 and 100

    // todo consider moving this to item as a general facility
    function whenRemoved( element, callback ) {
        let observer = new MutationObserver(function(mutations) {
            if ( mutations.filter( mutation => mutation.type === "childList" && mutation.removedNodes.length > 0 ).length > 0 ) {
                // something was removed
                // it is tricky to go through the removed node to figure out if the item or an ancestor of it was removed
                // so just see if it has been detached from the DOM
                if ( !element.closest( "html" ) ) {
                    observer.disconnect();
                    callback();
                }
            }
        });
        observer.observe( document.body, {
            subtree: true,
            childList: true
        } );
    }

    function getMessage( key ) {
        return lang.getMessage( "APEX.COLOR_PICKER." + key );
    }

    function getCssVariableColor( cssVariableName ) {
        return getComputedStyle( document.documentElement ).getPropertyValue( cssVariableName.trim() );
    }

    /* Sets the background color for the preview element */
    function setColorPreview ( pElem, pColor ) {
        pElem.css( BG_COLOR_PROP, pColor.toString() );
    }

    // id is optional
    function renderColorContrastReport( id ) {
        return `<div ${id ? 'id="' + id + '"' : ""} class="${colorContrastClass}">\
<span class="a-ColorPicker-contrastText">${getMessage( "CONTRAST" )}</span>\
<div class="a-ColorPicker-contrastColorCheck">\
<span class="${colorContrastColor1Class}"></span>\
<span class="${colorContrastColor2Class}"></span>\
</div>\
<span class="${colorContrastResultClass}"></span>\
<span class="${colorContrastRatingClass}"></span>\
<span aria-hidden="true" class="${colorContrastIconClass} a-Icon"></span>\
</div>`;
    }

    function updateColorContrastReport( report$, contrastInfo ) {
        let resultIcon, ratioText, ratingText = "",
            warn = false;

        report$.children().toggle( !!contrastInfo );
        if ( contrastInfo ) {
            if ( contrastInfo.aaa_small ) { // if AAA passed
                resultIcon = 'icon-check u-success-text';
                ratingText = 'AAA';
            } else if ( contrastInfo.aa_small ) { // if AA passed
                resultIcon = 'icon-check u-success-text';
                ratingText = 'AA';
            } else { //else nothing passed
                resultIcon = 'icon-warning u-warning-text';
                warn = true;
            }
            ratioText = Math.round( (contrastInfo.ratio + Number.EPSILON) * 100 ) / 100; // to ensure things like 1.005 round correctly

            report$.toggleClass( "is-warning", warn ).find( "." + colorContrastIconClass )
                .removeClass( "icon-check icon-warning u-success-text u-warning-text" )
                .addClass( resultIcon );
            report$.find( "." + colorContrastResultClass ).text( ratioText );
            report$.find( "." + colorContrastRatingClass ).text( ratingText );
            setColorPreview( report$.find( "." + colorContrastColor1Class ), contrastInfo.color_1 );
            setColorPreview( report$.find( "." + colorContrastColor2Class ), contrastInfo.color_2 );
        } else {
            report$.removeClass ( "is-warning" );
        }
    }

    // make these available for internal use
    apex.widget.util.colorPicker = {
        renderColorContrastReport: renderColorContrastReport,
        updateColorContrastReport: updateColorContrastReport
    };

    function renderColorPresets( out, idPrefix, presetColors, maxCount ) {
        let colorValues = [],
            menuId = null,
            colorPresetsMenuItems = [];

        out.markup( `<div id="${idPrefix}presets" class="a-ColorPicker-presets">` );

        if ( presetColors === "THEME" ) {
            for ( let i = 0; i < colorPresetThemeStyleTotal; i++ ) {
                let color = getCssVariableColor( colorPresetCssVariableName + (i + 1) );
                if ( color ) {
                    colorValues.push( color );
                }
            }
        } else {
            colorValues = presetColors.split( ';' );
        }

        let colorPresetsVisible  = colorValues.slice( 0, maxCount ),
            colorPresetsDropdown = colorValues.slice( maxCount );

        // render the visible color presets
        for (let i = 0; i < colorPresetsVisible.length; i++) {
            let preset = colorPresetsVisible[i].trim(),
                ojColor = colorUtilities.getValidColor( preset );

            if ( ojColor ) {
                let colorPreview = BG_COLOR_PROP + ':' + ojColor.toString() + ';';

                out.markup( `<button type="button" class="a-Button ${colorPresetClass}" aria-label="${preset}"` )
                    .optionalAttr( "style", colorPreview )
                    .attr( "data-color-value", preset )
                    .markup( '></button>' );
            }
        }

        // render the dropdown color presets
        if ( colorPresetsDropdown.length ) {
            menuId = idPrefix + 'presetMenu';
            out.markup( `<button type="button" data-menu="${menuId}"\
class="a-Button ${colorPresetClass} ${colorPresetMenuButtonClass}">\
<span aria-hidden="true" class="a-Icon icon-down-chevron"></span></button>
<div id="${menuId}" class="${colorPresetMenuClass}"></div>` );

            for ( let i = 0; i < colorPresetsDropdown.length; i++ ) {
                let preset = colorPresetsDropdown[i];

                colorPresetsMenuItems.push( {
                    type:"action",
                    iconType: 'a-Icon',
                    icon: 'icon-color-preview',
                    iconStyle: COLOR_PROP + ':' + colorUtilities.getValidColor( preset ) + ';',
                    value: preset,
                    label: preset
                    // action to be set later
                } );
            }
        }

        out.markup('</div>');
        return colorPresetsMenuItems;
    }

    function initColorPresets( idPrefix, colorPresetsMenuItems, setColor ) {
        let escId = util.escapeCSS( idPrefix ),
            presets$ = $( "#" + escId + "presets" ),
            colorPresetsButton$ = presets$.find( "." + colorPresetClass ).not( "." + colorPresetMenuButtonClass ),
            colorPresetsMenuButton$ = presets$.find( "." + colorPresetMenuButtonClass ),
            colorPresetsMenu$ = $( "#" + escId + "presetMenu" );

        // handle the color presets
        colorPresetsButton$.click( function() {
            let colorValue = $( this ).attr( 'data-color-value' );

            setColor( colorValue );
        } );

        // color preset dropdown menu
        if ( colorPresetsMenuButton$ ) {
            colorPresetsMenuItems.forEach( item => {
                item.action = function() {
                    setColor( item.value );
                };
            } );
            colorPresetsMenu$.menu( {
                items: colorPresetsMenuItems
            } );
        }
    }

    const colorSpectrumPickerPrototype = {
        // two step initialization, first render then initialize, is so that callers and initialize can wait until the jet color spectrum is ready
        render: function() {
            this.colorSpectrumContent$ = $( this._render() ); // rendered but not yet inserted into the document

            let colorSpectrum$ = this.colorSpectrumContent$.find( ".color-spectrum" );
            // oj color spectrum wants to have some value before it is inserted into the document but can't specify
            // a value in the markup (won't take a string or a color object literal) so set its value now
            colorSpectrum$[0].value = new colorUtilities.getValidColor( "#000000" );
            // insert into the document so JET turns it into a web component.
            // Want it offscreen or hidden so doesn't affect page until moved into the popup but it seems OK as is
            $( document.body ).append( this.colorSpectrumContent$ );

            // return a promise for when the color spectrum is ready
            return colorUtilities.whenReady( colorSpectrum$[0] );
        },
        initialize: function( element$ ) {
            let colorSpectrumContent$, colorDetailItems$, colorDetailToggleButton$;

            // handles the color format toggle
            const colorFormatToggle = () => {
                let details$ = colorSpectrumContent$.find( "." + colorDetailClass ),
                    currentDetail$ = details$.not( "." + C_HIDDEN ),
                    nextDetail$ = currentDetail$.next( "." + colorDetailClass );

                if ( !currentDetail$.length ) {
                    // initially they are all hidden. Pick the one to show initially based on the return as type returnAs
                    let suffix = this.returnAs.substr(0,3).toLowerCase();
                    if ( !["hex","rgb","hsl"].includes( suffix ) ) {
                        suffix = "hex";
                    }
                    nextDetail$ = details$.filter( "." + colorDetailClass + "--" + suffix );
                }

                currentDetail$.addClass( C_HIDDEN );

                if ( nextDetail$.length ) {
                    // show the next color type
                    nextDetail$.removeClass( C_HIDDEN );
                } else {
                    // circle back and show the first one
                    details$.eq(0).removeClass( C_HIDDEN );
                }
            };
            const setColorSpectrum = ( colorValue ) => {
                let ojColor = colorUtilities.getValidColor( colorValue );

                if ( ojColor !== null ) {
                    this.colorSpectrum$[0].value = ojColor;
                }
            };
            // Handle color changes
            const colorSelectionChanged = ( event ) => {
                let ojColor,
                    target$ = $( event.target );

                // if changed from the color spectrum
                if ( target$[0] === this.colorSpectrum$[0] ) {
                    ojColor = event.detail.value;

                    // update rgb values
                    this._setColorDetailRGB( ojColor );

                    // update hsl values
                    this._setColorDetailHSL( ojColor );

                    // update hex value
                    this._setColorDetailHex( ojColor );
                } else if ( target$.hasClass( colorDetailInputClass ) ) { // if changed from one of the details input
                    let detail$ = target$.closest("." + colorDetailClass);

                    if ( detail$.hasClass( colorDetailHexClass ) ) { // HEX
                        ojColor = this._getHexDetailColor();

                        // check if value is valid
                        if ( ojColor === null ) {
                            this._setColorDetailHex( this.ojColorCurrent );
                            return;
                        }
                    } else if ( detail$.hasClass( colorDetailHSLClass ) ) { // HSL
                        ojColor = this._getHSLDetailColor();

                        // check if value is valid
                        if ( ojColor === null ) {
                            this._setColorDetailHSL( this.ojColorCurrent );
                            return;
                        }
                    } else { // RGB
                        ojColor = this._getRGBDetailColor();

                        // check if value is valid
                        if (ojColor === null){
                            this._setColorDetailRGB( this.ojColorCurrent );
                            return;
                        }
                   }

                    // updating the color spectrum causes a transientValueChanged event which will update the other details
                    this.colorSpectrum$[0].value = ojColor;
                }

                // update the current color preview
                setColorPreview( this.colorPreviewCurrent$, ojColor.toString() );

                this.ojColorCurrent = ojColor;
                this.valueChanged( ojColor );

                // set contrast result; must be done after the valueChanged callback
                if ( this.contrastCheck ) {
                    this.updateColorContrast();
                }
            };

            /*
            * Create the color spectrum picker content and add to given container
            */
            colorSpectrumContent$ = this.colorSpectrumContent$;
            element$.empty().append( colorSpectrumContent$ );

            // References for later
            this.colorSpectrum$ = colorSpectrumContent$.find( ".color-spectrum" );

            this.colorPreviewCurrent$ = colorSpectrumContent$.find("." + colorPreviewCurrentClass + " ." + colorPreviewValueClass);
            this.colorPreviewInitial$ = colorSpectrumContent$.find("." + colorPreviewInitClass    + " ." + colorPreviewValueClass);

            this.colorSpectrumContrast$ = colorSpectrumContent$.find("." + colorContrastClass);

            colorDetailItems$        = colorSpectrumContent$.find("." + colorDetailInputClass);
            colorDetailToggleButton$ = colorSpectrumContent$.find("." + colorDetailToggleClass);

            // don't show the alpha slider if the setting is disabled
            // Warning! there is no native option in the JET component so digging into the internal markup
            if ( !this.showAlpha ) {
                this.colorSpectrum$.parent().find( '.oj-colorspectrum-alpha' ).closest( '.oj-slider' ).hide(); // hide seems sufficient; no need to remove
            }

            // handle the color spectrum change event
            // use the transient event instead of the change so that the color updates as it changes (e.g. when dragging)
            this.colorSpectrum$.on( 'transientValueChanged', colorSelectionChanged );

            // handle the color details change event
            colorDetailItems$.on ( 'change', colorSelectionChanged );

            // handle the color type toggle
            colorDetailToggleButton$.on( 'click', colorFormatToggle );
            colorFormatToggle();

            // color presets
            if ( this.presetColors ) {
                initColorPresets( this.idPrefix, this.colorPresetsMenuItems, setColorSpectrum );
            }

            colorSpectrumContent$.closest( '.ui-dialog,.a-ColorPicker-inlineWrap' ).on( 'keydown', event => {
                let kc = event.which;

                if ( ( kc === 90 && ( ( !isMac && event.ctrlKey ) || ( isMac && event.metaKey ) ) ) ||
                    kc === keys.BACKSPACE ) { // Ctrl-Z or Command-Z on Mac or backspace
                    if ( event.target.nodeName !== "INPUT" ) {
                        event.preventDefault();
                        this.revert();
                    }
                }
            } );

            colorSpectrumContent$.find( ".a-ColorPicker-preview--initial .a-ColorPicker-preview-value" ).click( () => {
                this.revert();
            } );

        },
        setColor: function( colorValue, isInitial ) {
            let ojColor = colorUtilities.getValidColor( colorValue );

            // if not a valid color use the default
            if ( !ojColor ) {
                ojColor = colorUtilities.makeColor( defaultColor );
            }
            this.ojColorCurrent = ojColor;
            if ( isInitial ) {
                this.ojColorInitial = ojColor;
            }

            if ( !this.showAlpha ) {
                // remove alpha if it is present
                if ( this.ojColorInitial.getAlpha() !== 1) {
                    // change alpha back to 1
                    this.ojColorInitial._a = 1;
                }
            }

            this.colorSpectrum$[0].value = ojColor;

            setColorPreview( this.colorPreviewCurrent$, ojColor);
            if ( isInitial ) {
                setColorPreview( this.colorPreviewInitial$, ojColor );
            }
            this._setColorDetailRGB( ojColor );
            this._setColorDetailHSL( ojColor );
            this._setColorDetailHex( ojColor );

            this.updateColorContrast();
        },
        revert: function() {
            this.colorSpectrum$[0].value = this.ojColorInitial;
        },
        getColor: function() {
            return this.ojColorCurrent;
        },
        getCurrentFormat: function() {
            let details$ = this.colorSpectrumContent$.find( "." + colorDetailClass ),
                currentDetail$ = details$.not( "." + C_HIDDEN );

            return currentDetail$.attr( "data-format" );
        },
        updateColorContrast: function() {
            let contrast = this.getContrast(); // callback to get contrast

            updateColorContrastReport( this.colorSpectrumContrast$, contrast );
        },
        _setColorDetailRGB: function( ojColor ) {
            let colorDetailItems$ = this.colorSpectrumContent$.find( "." + colorDetailRGBClass + " input" );

            // order is RGBA
            colorDetailItems$.eq( 0 ).val( ojColor.getRed() );
            colorDetailItems$.eq( 1 ).val( ojColor.getGreen() );
            colorDetailItems$.eq( 2 ).val( ojColor.getBlue() );

            if ( this.showAlpha ) {
                colorDetailItems$.eq( 3 ).val( ojColor.getAlpha() );
            }
        },
        _setColorDetailHSL: function( ojColor ) {
            let colorDetailItems$ = this.colorSpectrumContent$.find( "." + colorDetailHSLClass + " input" ),
                hsl = colorUtilities.ojColorToHSL(ojColor);

            // order is HSLA
            colorDetailItems$.eq( 0 ).val( hsl.h );
            colorDetailItems$.eq( 1 ).val( hsl.s );
            colorDetailItems$.eq( 2 ).val( hsl.l );

            if ( this.showAlpha ){
                colorDetailItems$.eq( 3 ).val( hsl.a );
            }
        },
        _setColorDetailHex: function( ojColor ) {
            let colorDetailItems$ = this.colorSpectrumContent$.find( "." + colorDetailHexClass + " input" ),
                hexColor = colorUtilities.colorConverterHEX.format( ojColor );

            colorDetailItems$.val( hexColor.replace( '#', '' ) );
        },
        /* Get the color from the Hex input field */
        _getHexDetailColor: function() {
            let detailItem$ = this.colorSpectrumContent$.find( "." + colorDetailHexClass + " input" ),
                hexValue    = detailItem$.val();

            // if not a color color pattern
            if ( !HEX_RE.test( hexValue ) ) {
                return null;
            } // else
            return colorUtilities.makeColor( hexValue );
        },
        /* Get the color from the R, G and B input field */
        _getHSLDetailColor: function() {
            let aNum,
                detailItem$ = this.colorSpectrumContent$.find( "." + colorDetailHSLClass + " input" ), // order is HSLA
                hValue      = detailItem$.eq( 0 ).val(),
                sValue      = detailItem$.eq( 1 ).val(),
                lValue      = detailItem$.eq( 2 ).val(),
                aValue      = ( this.showAlpha ) ? detailItem$.eq( 3 ).val() : '1';

            // validate hue:         must be between 0 and 360
            // validate saturation:  must be between 0 and 100
            // validate lightness:   must be between 0 and 100
            if ( !HUE_RE.test( hValue ) || !SAT_LUM_RE.test( sValue ) || !SAT_LUM_RE.test( lValue ) ){
                return null;
            } // else

            // validate alpha:       must be between 0 and 1 with up to two digits
            aNum = parseFloat( aValue );
            if ( this.showAlpha && ( !ALPHA_RE.test( aValue ) || aNum < 0 || aNum > 1 ) ) {
                return null;
            } // else
            return colorUtilities.makeColor( {
                h: hValue,
                s: sValue,
                l: lValue,
                a: aValue
            } );
        },
        /* Get the color from the R, G and B input field */
        _getRGBDetailColor: function() {
            let aNum,
                detailItem$ = this.colorSpectrumContent$.find("." + colorDetailRGBClass + " input"), // order is RGBA
                rValue      = detailItem$.eq( 0 ).val(),
                gValue      = detailItem$.eq( 1 ).val(),
                bValue      = detailItem$.eq( 2 ).val(),
                aValue      = ( this.showAlpha ) ? detailItem$.eq( 3 ).val() : '1';

            // validate colors: must be between 0 and 255
            if ( !RGB_RE.test( rValue ) || !RGB_RE.test( gValue ) || !RGB_RE.test( bValue ) ){
                return null;
            } // else

            // validate alpha:       must be between 0 and 1 with up to two digits
            aNum = parseFloat( aValue );
            if ( this.showAlpha && (!ALPHA_RE.test(aValue) || aNum < 0 || aNum > 1 ) ) {
                return null;
            } // else

            return colorUtilities.makeColor( {
                r: rValue,
                g: gValue,
                b: bValue,
                a: aValue
            } );
        },
        _render: function() {
            /* HTML Structure:
            *
            * .color-picker-inline-wrapper (wrapper for inline only)
            * .a-ColorPicker
            *     .a-ColorPicker-spectrum
            *         .color-spectrum
            *
            *     .a-ColorPicker-previews
            *         .a-ColorPicker-preview.a-ColorPicker-preview--current
            *               .a-ColorPicker-preview-label
            *               .a-ColorPicker-preview-value
            *         .a-ColorPicker-preview.a-ColorPicker-preview--initial
            *               .a-ColorPicker-preview-label
            *               .a-ColorPicker-preview-value
            *
            *     .a-ColorPicker-colorContrast
            *
            *     .a-ColorPicker-details
            *         .a-ColorPicker-detail.a-ColorPicker-detail--rgb
            *             .a-ColorPicker-detailItem.a-ColorPicker-detailItem--r
            *               label.a-ColorPicker-detailLabel
            *               input.apex-item-text.a-ColorPicker-detailInput
            *             .a-ColorPicker-detailItem.a-ColorPicker-detailItem--g
            *               label.a-ColorPicker-detailLabel
            *               input.apex-item-text.a-ColorPicker-detailInput
            *             .a-ColorPicker-detailItem.a-ColorPicker-detailItem--b
            *               label.a-ColorPicker-detailLabel
            *               input.apex-item-text.a-ColorPicker-detailInput
            *
            *         .a-ColorPicker-detail.a-ColorPicker-detail--hsl
            *             .a-ColorPicker-detailItem.a-ColorPicker-detailItem--h
            *               label.a-ColorPicker-detailLabel
            *               input.apex-item-text.a-ColorPicker-detailInput
            *             .a-ColorPicker-detailItem.a-ColorPicker-detailItem--s
            *               label.a-ColorPicker-detailLabel
            *               input.apex-item-text.a-ColorPicker-detailInput
            *             .a-ColorPicker-detailItem.a-ColorPicker-detailItem--l
            *               label.a-ColorPicker-detailLabel
            *               input.apex-item-text.a-ColorPicker-detailInput
            *
            *         .a-ColorPicker-detail.a-ColorPicker-detail--hex
            *             .a-ColorPicker-detailItem.a-ColorPicker-detailItem--hex
            *               label.a-ColorPicker-detailLabel
            *               input.apex-item-text.a-ColorPicker-detailInput
            *
            *         .a-ColorPicker-detailsToggle
            *             .a-ColorPicker-detailsToggle-button
            *
            *     .a-ColorPicker-presets
            *         .a-ColorPicker-preset
            *         .a-ColorPicker-preset
            *         .a-ColorPicker-preset
            *         .a-ColorPicker-preset
            *         .a-ColorPicker-preset
            *         .a-ColorPicker-presetMenuButton (if more than 5 presets)
            *         .a-ColorPicker-presetMenus      (if more than 5 presets)
            */
            let cls, title,
                idPrefix = this.idPrefix,
                displayModeClass = colorPickerPrefixClass + '--simple',
                out = util.htmlBuilder();

            // Utility function for the color preview element
            const renderColorPreview = ( pLabel, pElemClass ) => {
                out.markup( `<div class="${colorPreviewClass} ${pElemClass}"><div class="${colorPreviewValueClass}"` )
                    .attr( "title", pLabel )
                    .attr( "aria-label", pLabel )
                    .markup( "></div></div>" );
            };

            // Utility function for the detail input field
            // min, max, step only apply if isNumber is true
            const renderDetailField = ( pIdSuffix, pLabel, pMaxLength, isNumber, pElemClass, min, max, step = 1 ) => {
                let detailItemId = idPrefix + 'colorPickerDetail_' + pIdSuffix;

                out.markup( `<div class="${colorDetailItemClass} ${pElemClass}">\
<label for="${detailItemId}" class="${colorDetailLabelClass}">${pLabel}</label>\
<input id="${detailItemId}" type="${( isNumber ? 'number' : 'text' )}"`)
                    .optionalAttr( "min", isNumber ? min : null )
                    .optionalAttr( "max", isNumber ? max : null )
                    .optionalAttr( "step", isNumber ? step : null )
                    .markup( `maxlength="${pMaxLength}" size="5" class="apex-item-text ${colorDetailInputClass}"></div>` );

            };

            const renderNumDetailField = ( pIdSuffix, pLabel, pElemClass, min, max, step=1 ) => {
                renderDetailField( pIdSuffix, pLabel, 4, true, pElemClass, min, max, step );
            };

            if ( this.displayMode === 'FULL' ) {
                displayModeClass = colorPickerPrefixClass + '--full';
            }

            // generate the color picker markup
            // todo binding-provider not needed for inline and ideally not at all
            // Because the value is a class can't set it in markup. See render above.
            out.markup( `<div class="${colorPickerPrefixClass} ${displayModeClass}" data-oj-binding-provider="none">\
<div class="${colorPickerPrefixClass}-spectrum"><oj-color-spectrum id="${idPrefix}jetColorSpectrum" class="color-spectrum">\
</oj-color-spectrum></div><div class="${colorPickerPrefixClass}-previews">` ); // open preview wrapper

            renderColorPreview( getMessage( "CURRENT" ), colorPreviewCurrentClass );
            renderColorPreview( getMessage( "INITIAL" ), colorPreviewInitClass );

            cls = colorDetailClass + ' ' + C_HIDDEN;
            // close preview wrapper then color details (RBG/HSL/Hex)
            out.markup( `</div><div class="${colorPickerPrefixClass}-details">\
<div class="${cls} ${colorDetailRGBClass}" data-format="${this.showAlpha ? RETURN_AS_RGBA : RETURN_AS_RGB}">` );

            // It seems that RGB[A] is not translated
            renderNumDetailField('RGB_R', 'R', colorDetailItemRClass, "0", "255" );
            renderNumDetailField('RGB_G', 'G', colorDetailItemGClass, "0", "255" );
            renderNumDetailField('RGB_B', 'B', colorDetailItemBClass, "0", "255" );

            if ( this.showAlpha ) {
                renderNumDetailField('RGB_A', 'A', colorDetailItemAClass, "0", "1", "0.01" );
            }

            out.markup( `</div><div class="${cls} ${colorDetailHSLClass}" data-format="${this.showAlpha ? RETURN_AS_HSLA : RETURN_AS_HSL}">` );

            // It seems that HSL[A] is not translated
            renderNumDetailField('HSL_H', 'H', colorDetailItemHClass, "0", "360" );
            renderNumDetailField('HSL_S', 'S', colorDetailItemSClass, "0", "100" );
            renderNumDetailField('HSL_L', 'L', colorDetailItemLClass, "0", "100" );

            if ( this.showAlpha ) {
                renderNumDetailField('HSL_A', 'A', colorDetailItemAClass, "0", "1", "0.01" );
            }

            out.markup( `</div><div class="${cls} ${colorDetailHexClass}" data-format="${RETURN_AS_HEX}">` );

            renderDetailField( 'HEX', 'Hex', 7, false, colorDetailItemHexClass ); // 7 to allow room for optional leading #

            title = getMessage( "TOGGLE_TITLE" );
            out.markup( `</div><button type="button" title="${title}" aria-label="${title}"\
class="a-Button ${colorDetailToggleClass}"><span aria-hidden="true" class="a-Icon icon-colorpicker-select">\
</span></button></div>` );

            // color contrast
            if ( this.contrastCheck ) {
                out.markup( renderColorContrastReport() );
            }

            // color presets
            if ( this.presetColors ) {
                this.colorPresetsMenuItems = renderColorPresets( out, this.idPrefix, this.presetColors, this.maxPresetColors );
            }

            out.markup( '</div>' );

            return out.toString();
        },
    };

    /**
     * Make a widget like object for a color spectrum picker. This is the content of an inline color picker or
     * a color picker popup. Cannot be called until jet is loaded.
     *
     * @param options:
     *   idPrefix
     *   presetColors
     *   maxPresetColors extras are put into an overflow menu
     *   popupClass
     *   returnAs
     *   showAlpha
     *   displayMode
     *   contrastCheck
     * callbacks:
     *   getContrast()
     *   valueChanged( newColor )
     *
     * @ignore
     * @returns a color spectrum picker
     */
    function makeColorSpectrumPicker( options ) {
        let instance = Object.create( colorSpectrumPickerPrototype );

        $.extend( instance, options );
        instance.ojColorInitial = null;
        instance.ojColorCurrent = null;
        return instance;
    }

    function openPopupColorPicker( cpItem, options, callback ) {
        let csp,
            item$ = cpItem.element,
            initialColorValue,
            ojInitialColor;

        // dialog properties
        const dialogId = options.idPrefix + "ColorPickerDlg";

        function getPreferredReturnFormat( ojColor ) {
            let returnFormat = options.returnAs;

            if ( returnFormat === RETURN_AS_CSS ) {
                returnFormat = csp.getCurrentFormat() || RETURN_AS_HEX;
            }
            return colorUtilities.getReturnColorString( returnFormat, ojColor );
        }

        let popup$ = $( "#" + util.escapeCSS( dialogId ) );

        item$.attr( A_ARIA_EXPANDED, "true" );

        if ( !popup$[0] ) {
            let popupOptions = {
                closeText: lang.getMessage( "APEX.DIALOG.CLOSE" ),
                title: getMessage( "TITLE" ),
                noOverlay: true,
                width: 'auto',
                height: 'auto',
                dialogClass: dialogClass + ( options.popupClass ? " " + options.popupClass : "" ),
                parentElement: item$.parent(),
                close: function() {
                    let newColorValue,
                        ojNewColor = csp.getColor();

                    // the popup has closed see if there is any change in the value
                    if ( options.returnAs === RETURN_AS_CSS && ojNewColor.isEqual( ojInitialColor ) ) {
                        // if the color didn't change there is a chance that the initial color was a name or css var
                        // or just in a different format so keep the original value
                        cpItem.setValue( initialColorValue, null, true );
                    } else {
                        newColorValue = getPreferredReturnFormat( ojNewColor );
                        if ( newColorValue !== initialColorValue ) {
                            cpItem.setValue( newColorValue ); // set the value such that a change event is fired
                        }
                    }
                    item$.attr( A_ARIA_EXPANDED, "false" );
                    callback( newColorValue );
                },
                create: function() {
                    csp.initialize( popup$ );

                    popup$.closest('.ui-dialog').on( 'keydown', event => {
                        let kc = event.which;

                        if ( kc === keys.ENTER ) {
                            // if not in a button treat enter as close
                            if ( event.target.nodeName !== "BUTTON" ) {
                                event.preventDefault();
                                popup$.popup( "close" );
                            }
                        }
                    } );
                },
                open: function( /* event */ ) {
                    // set the values of the color spectrum, the color preview and the color details
                    initialColorValue = cpItem.getValue();
                    csp.setColor( initialColorValue, true );
                    ojInitialColor = csp.getColor();

                    // the color spectrum is rather small, assume it will fit on even a small screen device.
                    // todo think if big relative to the screen size want to just center
                }
            };

            let cspOptions = $.extend( {}, options, {
                getContrast: function() {
                    return cpItem.contrastWith();
                },
                valueChanged: function( ojColor ) {
                    cpItem.setValue( getPreferredReturnFormat( ojColor), null, true );
                    cpItem.element.trigger( "transientchange" );
                }
            } );

            popup$ = $( `<div id="${dialogId}" class='a-ColorPicker-dialog'></div>` );
            $("body").append( popup$ );

            csp = makeColorSpectrumPicker( cspOptions );
            csp.render().then( () => {
                popup$.popup( popupOptions );
            } );
        } else {
            popup$.popup( "open" );
        }

        return dialogId;
    }

    // interface to jet color picker functions. Null until jet is loaded (if at all)
    let colorUtilities = null;

    // preloading JET if available (it is optional if all color pickers are native)
    function initJETColorUtilities( oj ) {
        // color converters
        let colorConverterHSL = new oj.ColorConverter( {format: "hsl"} ),
            colorConverterHEX = new oj.ColorConverter( {format: "hex"} ),
            colorConverterRGB = new oj.ColorConverter( {format: "rgb"} );

        /* Converts, checks if the color is valid and returns the ojColor given a color value string */
        function getValidColor( colorValue ) {
            let ojColor,
                colorUpper = colorValue.toUpperCase(),
                cssVarColor = getCssVariableColor( colorValue );

            // check if it's a named color
            if ( oj.Color[colorUpper] ) {
                ojColor = oj.Color[colorUpper];
            } else {
                if ( cssVarColor.length ) { // check if it's a CSS variable
                    colorValue = cssVarColor;
                }
                // check if the value is a valid color
                try {
                    ojColor = new oj.Color( colorValue );
                } catch ( e ) {
                    ojColor = null;
                }
            }

            return ojColor;
        }

        // gets the luminance value for a single color
        function getSingleColorLuminance( singleColorValue ) {
            let value = singleColorValue / 255;

            return value <= 0.03928 ? value / 12.92 : Math.pow( (value + 0.055) / 1.055, 2.4 );
        }

        // gets the relative luminance value for all colors
        function getRelativeLuminance( ojColor ) {
            let r = ojColor.getRed(),
                g = ojColor.getGreen(),
                b = ojColor.getBlue();
            // a = ojColor.getAlpha(); // todo determine if alpha can/should be included in the calculation

            return getSingleColorLuminance( r ) * 0.2126 + getSingleColorLuminance( g ) * 0.7152 + getSingleColorLuminance( b ) * 0.0722;
        }

        colorUtilities = {
            makeColor: function ( colorValue ) {
                return new oj.Color( colorValue );
            },
            colorConverterHEX: colorConverterHEX,
            colorConverterHSL: colorConverterHSL,
            colorConverterRGB: colorConverterRGB,
            getValidColor: getValidColor,
            getColorContrast: function ( colorValue1, colorValue2 ) {
                // if either color is missing
                if ( !colorValue1 || !colorValue2 ) {
                    return null;
                }

                let ojColor1 = getValidColor( colorValue1 ),
                    ojColor2 = getValidColor( colorValue2 );

                // if either color is invalid
                if ( !ojColor1 || !ojColor2 ) {
                    return null;
                }
                // calculate the relative luminance
                let color1luminance = getRelativeLuminance( ojColor1 ),
                    color2luminance = getRelativeLuminance( ojColor2 );

                // calculate the color contrast ratio
                let ratio = color1luminance > color2luminance ? ((color1luminance + 0.05) / (color2luminance + 0.05)) : ((color2luminance + 0.05) / (color1luminance + 0.05));

                return {
                    color_1: ojColor1.toString(),
                    color_2: ojColor2.toString(),
                    ratio: ratio,
                    aa_large: ratio > 3,
                    aa_small: ratio > 4.5,
                    aaa_large: ratio > 4.5,
                    aaa_small: ratio > 7
                };
            },
            getReturnColorString: function ( returnAs, ojColor ) {
                let colorString;

                if ( [RETURN_AS_HEX, RETURN_AS_RGB, RETURN_AS_HSL].includes( returnAs ) ) {
                    // remove alphs
                    ojColor._a = 1;
                }
                switch ( returnAs ) {
                    case RETURN_AS_HEX:
                        colorString = colorConverterHEX.format( ojColor );
                        break;
                    case RETURN_AS_RGB:
                    case RETURN_AS_RGBA:
                        colorString = colorConverterRGB.format( ojColor );
                        break;
                    case RETURN_AS_HSL:
                    case RETURN_AS_HSLA:
                        colorString = colorConverterHSL.format( ojColor );
                        break;
                    case RETURN_AS_CSS:
                        colorString = ojColor.toString(); // just take the default format
                }

                return colorString;
            },
            ojColorToHSL: function ( ojColor ) {
                let m,
                    colorString = colorConverterHSL.format( ojColor );

                // the color formatter returns a string like: hsl(h, s%, l%) or hsla(h, s%, l%, a)
                m = HSLA_FN_RE.exec( colorString ) || [];
                return {
                    h: m[1],
                    s: m[2],
                    l: m[3],
                    a: m[4] || '1'
                };
            },
            whenReady: function ( element ) {
                return oj.Context.getContext( element ).getBusyContext().whenReady();
            }
        };
    }

    $( () => {
        // this is for the case where there are only native color pickers but JET is on the page so may as
        // well initialize this so that item getNativeValue and contrastWith can be used.
        if ( typeof require !== "undefined" ) {
            // if the list of require files is changed please change also Grunt.js list for colorpicker
            require( ["ojs/ojcore", "jquery", "ojs/ojconverter-color",
                "ojs/ojcolor", "ojs/ojcolorspectrum", "ojs/ojvalidation-base"], function ( oj ) {

                if ( !colorUtilities ) {
                    initJETColorUtilities( oj );
                }
            } );
        }
    } );

    function colorPickerWrapper( el$ ) {
        el$.closest( '.apex-item-group.apex-item-group--color-picker' );
    }

    let setCheckFormat = ( item, value ) => {
        let ojColor, returnAsValue,
            input$ = item.element;

        value = value.trim();
        returnAsValue = value;

        // JET may not be loaded
        if ( colorUtilities ) {
            item._color = ojColor= colorUtilities.getValidColor( value );
        }

        if ( !colorUtilities || item._displayAs === DISPLAY_AS_NATIVE ) {
            // native doesn't need validating (bad values are changed to the default) and does its own preview
            input$.val( value );
            return;
        }

        if ( item._returnAs !== RETURN_AS_CSS && ojColor ) {
            returnAsValue = colorUtilities.getReturnColorString( item._returnAs, ojColor );
        }

        // set color preview
        let itemColorPreview$ = item.element.parent().find( '.' + itemColorPreviewClass );
        // if it's a valid color
        if ( ojColor ) {
            itemColorPreview$.removeClass( itemColorNoPreviewClass )
                .css( BG_COLOR_PROP, ojColor.toString() );
        } else {
            itemColorPreview$.addClass( itemColorNoPreviewClass )
                .css( BG_COLOR_PROP, '' );
        }

        input$[0].setCustomValidity( "" ); // clear any existing error and assume no errors
        if ( value === "" ) {
            input$.val( value ); // base item handles required validation message
        } else {
            let message,
                valid = item._returnAs === RETURN_AS_CSS || ojColor !== null; // don't validate when return as css

            input$.val( returnAsValue );
            if ( !valid ) {
                message = lang.formatMessage( "APEX.COLOR_PICKER.INVALID_COLOR", colorExamples[item._returnAs] );
                input$[0].setCustomValidity( message );
            }
        }
    };

    // don't doc methods that don't apply to colorPickerItem
    /**
     * @ignore
     * @method addValue
     * @memberOf colorPickerItem.prototype
     */
    /**
     * @ignore
     * @method removeValue
     * @memberOf colorPickerItem.prototype
     */
    /**
     * @ignore
     * @method refresh
     * @memberOf colorPickerItem.prototype
     */

    /**
     * @lends colorPickerItem.prototype
     */
    let colorpickerItemPrototype = {
        /**
         * <p>The color picker item type is "COLOR_PICKER".</p>
         * @type {string}
         */
        item_type: "COLOR_PICKER",
        delayLoading: true,
        enable: function() {
            this.element.parent().find( "input, button, oj-color-spectrum" ).prop( P_DISABLED, false );
            this.element.filter( ".apex-item-text" ).removeClass( "apex_disabled" );
        },
        disable: function() {
            this.element.parent().find( "input, button, oj-color-spectrum" ).prop( P_DISABLED, true );
            this.element.filter( ".apex-item-text" ).addClass( "apex_disabled" );
        },
        isDisabled: function() {
            return this.element.prop( P_DISABLED );
        },
        show: function() {
            colorPickerWrapper( this.element ).show();
        },
        hide: function() {
            colorPickerWrapper( this.element ).hide();
        },
        setValue: function( value, display, suppressChangeEvent ) {
            setCheckFormat( this, value );

            if ( !suppressChangeEvent ) {
                // used to prevent the attached change event in order to prevent a second call to setCheckFormat
                this._preventChangeHandler = true;
            }
        },
        isChanged: function() {
            return this.node.value !== this._originalValue;
        },
        getValue: function() {
            return this.element.val();
        },
        // requires that the item is ready
        displayValueFor: function ( colorValue /*, state*/ ) {
            let bgColor,
                colorClass = '',
                colorStyle = '',
                ojColor = this.isReady() ? colorUtilities.getValidColor( colorValue ) : "";

            if ( ojColor === "" ) {
                // if not ready just hope it is a valid color
                // todo think the effect is that if it is a bad color the no preview class isn't added so doesn't look like a bad color.
                //    could also be some other css value such as initial but no harm in that
                bgColor = colorValue;
            } else if ( ojColor ) {
                // if it's a valid color
                bgColor = ojColor.toString();
            }

            if ( bgColor ) {
                colorStyle = BG_COLOR_PROP + ':' + bgColor + ";";
            } else {
                colorClass = ' ' + itemColorNoPreviewClass;
            }

            return `<div class="apex-item-group apex-item-group--color-picker">
<span class="${itemColorPreviewClass}${colorClass}" style="${colorStyle}"></span>
<span class="apex-item-color-picker-value">${util.escapeHTML( colorValue )}</span></div>`;
        },
        setFocusTo: function() {
            let focus$ = this.element;
            if ( this._displayAs === DISPLAY_AS_INLINE ) {
                focus$ = focus$.parent().find( "oj-color-spectrum" );
            } else if ( this._displayAs === DISPLAY_AS_COLOR_ONLY ) {
                focus$ = this.element.next();
            }
            return focus$;
        },
        getPopupSelector: function() {
            return ".ui-dialog-color-picker";
        },
        // requires that the item is ready
        /**
         * <p>Returns a color object representing the current item value. The object contains
         * properties: _r for red, _g for green, _b for blue, and _a for alpha. If the item value is empty
         * or invalid this returns null.
         * Oracle JET must be loaded and the item must be ready for this method to be successful.
         * See {@link colorPickerItem#whenReady}.
         * </p>
         *
         * @returns {object} color object.
         * Returns null if JET is not loaded or before the item is ready or if the color is invalid.
         * @example <caption>This example gets the value of an item as a color object.</caption>
         * var color1 = apex.item( "P1_COLOR1" ).getNativeValue();
         * // do something with the red, green, blue, and alpha channels of the color
         */
        getNativeValue: function() {
            return this._color;
        },
        /**
         * @typedef colorPickerItem.contrastInfo
         * @desc
         * Object describing the contrast between two colors according to the Web Content Accessibility Guidelines (WCAG).
         *
         * @property {boolean} aaa_large This is true if the contrast passes the WCAG AAA guidelines for large text.
         * @property {boolean} aa_large This is true if the contrast passes the WCAG AA guidelines for large text.
         * @property {boolean} aaa_small This is true if the contrast passes the WCAG AAA guidelines for small text.
         * @property {boolean} aa_small This is true if the contrast passes the WCAG AA guidelines for small text.
         * @property {number} ratio The contrast ratio.
         * @property {string} color_1 The first color.
         * @property {string} color_2 The second color.
         */

        /**
         * <p>Returns information about the contrast between the color of this item and another color.
         * Oracle JET must be loaded and the item must be ready for this method to be successful.
         * See {@link colorPickerItem#whenReady}.
         * </p>
         * @param {string} [pColorValue] Optional color to compare with. If this is not provide the
         *   configured item settings are used.
         * @returns {colorPickerItem.contrastInfo} A color contrast object.
         * Returns null if JET is not loaded or before the item is ready.
         * @example <caption>This example gets the contrast between item P1_COLOR1 and a static color and displays a
         * message if it does not pass for AA Small text.</caption>
         * var contrastInfo = apex.item( "P1_COLOR1" ).contrastWith( "#90ee90" );
         * if ( contrastInfo && !contrastInfo.aa_small ) {
         *     apex.message.alert( "Contrast check failed." );
         * }
         */
        contrastWith: function( pColorValue ) {
            if ( !colorUtilities ) {
                return null;
            }
            if ( !pColorValue ) {
                if ( this._contrastItem ) {
                    pColorValue = item( this._contrastItem ).getValue();
                } else if ( this._contrastColor ) {
                    pColorValue = this._contrastColor;
                }
            }
            return colorUtilities.getColorContrast( this.getValue(), pColorValue );
        }
    };

    function attachColorpicker( context$ ) {
        // initialize all color picker items
        $( ".apex-item-color-picker", context$ ).each( ( _, el ) => {
            let thisItem, inputType, button$, inline$, progress$,
                itemId = el.id,
                idPrefix = itemId + "_",
                item$ = $( el ),
                // data attributes
                displayAs = ( item$.attr( 'data-display-as' ) || DISPLAY_AS_POPUP ).toUpperCase(), // INLINE / POPUP / COLOR_ONLY / NATIVE
                returnAs = ( item$.attr( 'data-return-value-as' ) || RETURN_AS_HEX ).toUpperCase(), // HEX / RGB / RGBA / HSL / HSLA
                contrastItem = item$.attr( 'data-contrast-item' ) || null,
                contrastColor = item$.attr( 'data-contrast-color' ) || null,
                contrastCheck = contrastItem || contrastColor,
                presetColors = item$.attr( 'data-colors' ) || "",
                colorsInline = ( item$.attr( 'data-colors-inline' ) || "" ).toLowerCase() === "true",
                maxColors = item$.attr( 'data-max-colors' ),
                popupClass = item$.attr( 'data-popup-class' ) || "",
                displayMode = ( item$.attr( 'data-display-mode' ) || 'SIMPLE' ).toUpperCase(),
                showAlpha = [RETURN_AS_RGBA, RETURN_AS_HSLA].includes( returnAs );

            let dialogId,
                popupOpen         = false,
                popupRecentlyOpen = false;

            // Opens the popup/dialog don't call until item is ready
            function open() {
                popupOpen = true;
                dialogId = openPopupColorPicker( thisItem, {
                    idPrefix: idPrefix,
                    presetColors: colorsInline ? "" : presetColors,
                    maxPresetColors: maxColors,
                    popupClass: popupClass,
                    returnAs: returnAs,
                    showAlpha: showAlpha,
                    displayMode: displayMode,
                    contrastCheck: contrastCheck
                }, () => {
                    popupOpen = false;
                });
            }

            function waitOpen() {
                let progress$;

                if ( thisItem.isReady() ) {
                    open();
                } else {
                    // jet may not be ready yet, need to wait until it is
                    // use a delay/linger spinner while waiting
                    util.delayLinger.start( idPrefix, () => {
                        progress$ = util.showSpinner( inline$ );
                    });
                    thisItem.whenReady().then( () => {
                        open();
                        util.delayLinger.finish( idPrefix, () => {
                            progress$.remove();
                        } );
                    } );
                }
            }

            function initButtonHandlers( button$ ) {
                button$.click(function() {
                    if ( !popupRecentlyOpen ) {
                        waitOpen();
                    }
                    popupRecentlyOpen = false;
                } ).mousedown( function() {
                    if ( popupOpen ) {
                        // avoid re-opening the inline popup if the same mousedown that turns into a click closed it
                        popupRecentlyOpen = true;
                    }
                });
                // not a button handler but common for popup and color only
                whenRemoved( item$[0], () => {
                    // when item is removed make sure popup dialog is also removed
                    if ( dialogId ) {
                        $( "#" + util.escapeCSS( dialogId ) ).popup("destroy").closest( ".ui-dialog--popup" ).remove();
                    }
                });
            }

            /*
             * Some config validation
             */
            if ( ![RETURN_AS_HEX, RETURN_AS_RGBA, RETURN_AS_HSLA, RETURN_AS_RGB, RETURN_AS_HSL, RETURN_AS_CSS].includes( returnAs ) ) {
                debug.warn( 'Color Picker invalid return value as: ', returnAs );
                returnAs = RETURN_AS_HEX;
            }
            if ( !['FULL', 'SIMPLE'].includes( displayMode ) ) {
                debug.warn( 'Color Picker invalid display mode: ',  displayMode);
                displayMode === 'SIMPLE';
            }
            if ( maxColors ) {
                maxColors = parseInt( maxColors, 10 );
                if ( isNaN( maxColors ) ) {
                    maxColors = colorPresetMaxCount;
                }
            } else {
                maxColors = colorPresetMaxCount;
            }

            /*
             * Upgrade item markup
             */
            switch ( displayAs ) {
                case DISPLAY_AS_POPUP:
                    inputType = "text";
                    item$.before( `<span class="${itemColorPreviewClass}"></span>` )
                        .after( `<button aria-hidden="true" type="button" class="a-Button a-Button--colorPicker" tabindex="-1">\
<span class="a-Icon icon-color-picker"></span></button>` );
                    break;

                case DISPLAY_AS_COLOR_ONLY:
                    inputType = "hidden";
                    item$.after( `<button type="button" class="a-Button a-Button--colorPicker a-Button--colorPickerOnly">\
<span class="${itemColorPreviewClass}"></span></button>` );
                    break;

                case DISPLAY_AS_INLINE:
                    inputType = "hidden";
                    inline$ = $( `<div class="a-ColorPicker-inlineWrap"></div>` );
                    item$.after( inline$ );
                    inline$ = item$.next( '.a-ColorPicker-inlineWrap' );
                    break;

                case DISPLAY_AS_NATIVE:
                    // nothing to add just force type to be color just in case
                    inputType = "color";
                    break;
                default:
                    debug.error( "Color Picker invalid display-as ", displayAs );
            }
            item$.attr( "type", inputType );

            /*
             * Create item interface
             */
            let deferred = item.create( itemId, colorpickerItemPrototype );

            const waitForJet = ( d, thisItem ) => {
                // if the list of require files is changed please change also Grunt.js list for colorpicker
                require([ "ojs/ojcore", "jquery", "ojs/ojconverter-color", "ojs/ojcolor", "ojs/ojcolorspectrum", "ojs/ojvalidation-base" ], function( oj ) {
                    if ( !colorUtilities ) {
                        initJETColorUtilities( oj );
                    }
                    setCheckFormat( thisItem, item$.val() );
                    d.resolve();
                } );
            };

            thisItem = item( itemId );
            thisItem._originalValue = item$.val(); // can't use input's defaultValue because for hidden items it changes on set
            thisItem._color = null;
            thisItem._contrastItem = contrastItem;
            thisItem._contrastColor = contrastColor;
            thisItem._returnAs = returnAs;
            thisItem._displayAs = displayAs;

            if ( displayAs === DISPLAY_AS_NATIVE ) {
                // jet is not needed for this so it is already ready
                deferred.resolve();
            } else {
                waitForJet( deferred, thisItem );
            }

            /*
             * Add item behavior
             */
            switch ( displayAs ) {
                case DISPLAY_AS_POPUP:
                    button$ = item$.next( '.a-Button--colorPicker' );

                    // initialize controls
                    // support keyboard to open
                    item$.keydown( function( event ) {
                        let kc = event.which;

                        if ( kc === keys.DOWN || kc === keys.UP ) {
                            waitOpen();
                            event.preventDefault();
                        } else if ( kc === keys.ENTER ) {
                            // prevent the browser default to submit the page when this is the only text item on the page
                            // todo think enter should still cause a change event if changed
                            event.preventDefault();
                        } // Otherwise the typing keys could be used to enter a value
                    } );

                    // change handler to keep it in sync with the value
                    item$.change( () => {
                        //if the change event is triggered by the setValue, we don't need to do any transformation
                        if ( thisItem._preventChangeHandler ) {
                            thisItem._preventChangeHandler = false;
                            return;
                        }

                        setCheckFormat( thisItem, item$.val() );
                    } );

                    initButtonHandlers( button$ );
                    button$.focus( function() {
                        // the button is not a separate entity to focus
                        item$.focus();
                    });

                    if ( colorsInline && presetColors ) {
                        thisItem.whenReady().then( () => {
                            let out = util.htmlBuilder(),
                                colorPresetsMenuItems = renderColorPresets( out, idPrefix, presetColors, maxColors );

                            button$.after( out.toString() );
                            item$.parent().addClass( "a-ColorPicker--hasPresets" );
                            initColorPresets( idPrefix, colorPresetsMenuItems, function( color )  {
                                thisItem.setValue( color );
                            } );
                        } );
                    }
                    break;

                case DISPLAY_AS_COLOR_ONLY:
                    button$ = item$.next( '.a-Button--colorPicker' );
                    initButtonHandlers( button$ );
                    break;

                case DISPLAY_AS_INLINE:
                    // jet may not be ready yet, need to wait until it is
                    // use a delay/linger spinner while waiting
                    util.delayLinger.start( idPrefix, () => {
                        progress$ = util.showSpinner( inline$ );
                    });
                    thisItem.whenReady().then( () => {
                        let csp = makeColorSpectrumPicker( {
                            idPrefix: idPrefix,
                            presetColors: presetColors,
                            maxPresetColors: maxColors,
                            returnAs: returnAs,
                            showAlpha: showAlpha,
                            displayMode: displayMode,
                            contrastCheck: contrastCheck,
                            getContrast: function( ) {
                                return thisItem.contrastWith();
                            },
                            valueChanged: function( ojColor ) {
                                thisItem.setValue( colorUtilities.getReturnColorString( returnAs, ojColor), null, true );
                                // like a select list that is closed every user interaction triggers a change event
                                thisItem.element.trigger( "change" );
                            }
                        } );
                        csp.render().then( () => {
                            csp.initialize( inline$ );
                            csp.setColor( thisItem.getValue(), true );
                            util.delayLinger.finish( idPrefix, () => {
                                progress$.remove();
                            });
                        } );
                        let baseSetValue = thisItem.setValue;
                        thisItem.setValue = function( value, display, suppress ) {
                            baseSetValue.call( this, value, display, suppress );
                            csp.setColor( thisItem.getValue() );
                        };
                    } );
                    break;
            }

        } );
    }

    // register attachColorpicker to run when needed
    item.addAttachHandler( attachColorpicker );

})( apex.item, apex.jQuery, apex.debug, apex.util, apex.lang );