/*!
 * Copyright (c) 2020, 2022, Oracle and/or its affiliates.
 */

/**
 * The {@link apex.widget.ckeditor5} is used for the Rich Text Editor widget of Oracle APEX.
 * Internally, CKEditor5 http://www.ckeditor.com is used.
 * 
 * See the CKEditor5 documentation for available options.
 * 
 * Library changes:
 *  - v32.0.0 - APEX 22.1
 * 		- Major change: Now exposing CKEditor5 & all its packages as opposed to ClassicEditor & selective packages
 *  - v30.0.0
 * 		- no big changes
 * 	- v29.2.0
 * 		- no big changes
 *  - v29.1.0
 *  	- no big changes
 *  - v29.0.0
 *  	- Major changes in the image package. The way options are passed to the image plug-in in the js init code has changed!
 *          See: https://ckeditor.com/docs/ckeditor5/latest/builds/guides/migration/migration-to-29.html#migration-to-ckeditor-5-v2900
 *  - v28.0.0
 * 		- Added tableCaption package
 *  - v27.1.0 - APEX 21.1
 *  - v21.0.0 - APEX 20.2
 */

/* global CKEditor5,ClassicEditor,marked,DOMPurify */

( function( item, lang, util, widget, debug, $ ) {
    "use strict";

    const
        MODE_HTML = "html",
        MODE_MARKDOWN = "markdown",
        TOOLBAR_BASIC = "basic",
        TOOLBAR_INTERMEDIATE = "intermediate",
        TOOLBAR_FULL = "full",
        DISPLAY_VALUE_RICH_TEXT = "rich-text",
        DISPLAY_VALUE_ESCAPED = "escaped",
        DISPLAY_VALUE_PLAIN_TEXT = "plain-text";

    /**
     * The global object ClassicEditor is deprecated as of APEX 22.1 and will be removed in a future version
     * DO NOT USE !
     * Use the global object CKEditor5 instead.
     */
    window.ClassicEditor = CKEditor5.editorClassic.ClassicEditor;
    // exposing some useful Classes for building custom buttons
    // the creation of simple buttons and dropdowns requires: Plugin, ButtonView, [...]Dropdown, Model, Collection
    // the namespace "libraryClasses" was chosen arbitrarily but should be kept for consistency
    ClassicEditor.libraryClasses = {
        "Plugin": CKEditor5.core.Plugin,
        "ButtonView": CKEditor5.ui.ButtonView,
        "Model": CKEditor5.ui.Model,
        "Collection": CKEditor5.utils.Collection,
        "dropdownUtils": CKEditor5.ui,
        "clipboard": CKEditor5.clipboard,
        "language": CKEditor5.language
    };
    // using namespace extraPlugins to save references to optional plugins
    // these can be referenced and applies when instantiating a new editor
    // by using the extraPlugins initialization option
    ClassicEditor.extraPlugins = {
        "Markdown": CKEditor5.markdown.Markdown,
        "SimpleUploadAdapter": CKEditor5.upload.SimpleUploadAdapter,
        "ImageUpload": CKEditor5.image.ImageUpload,
        "CKFinder": CKEditor5.ckfinder.CKFinder,
        "CKFinderUploadAdapter": CKEditor5.adapterCkfinder.UploadAdapter,
        "RestrictedEditingMode": CKEditor5.restrictedEditing.RestrictedEditingMode,
        "Title": CKEditor5.heading.Title,
        "MediaEmbed": CKEditor5.mediaEmbed.MediaEmbed,
        "MediaEmbedToolbar": CKEditor5.mediaEmbed.MediaEmbedToolbar,
        "PendingActions": CKEditor5.core.PendingActions,
        "ListStyle": CKEditor5.list.ListStyle
    };

    const sanitizeHtml = html => {
        return DOMPurify.sanitize( html, {
            SAFE_FOR_JQUERY: true
        } );
    };

    const markdown2html = markdown => {
        marked.use( {
            tokenizer: {
                // Disable the autolink rule in the lexer.
                autolink: () => null,
                url: () => null
            },
            renderer: {
                checkbox( ...args ) {
                    // Remove bogus space after <input type="checkbox"> because it would be preserved
                    // by DomConverter as it's next to an inline object.
                    return Object.getPrototypeOf( this ).checkbox.call( this, ...args ).trimRight();
                },
                code( ...args ) {
                    // Since marked v1.2.8, every <code> gets a trailing "\n" whether it originally
                    // ended with one or not (see https://github.com/markedjs/marked/issues/1884 to learn why).
                    // This results in a redundant soft break in the model when loaded into the editor, which
                    // is best prevented at this stage. See https://github.com/ckeditor/ckeditor5/issues/11124.
                    return Object.getPrototypeOf( this ).code.call( this, ...args ).replace( '\n</code>', '</code>' );
                }
            }
        } );

        return marked.parse( markdown, {
            gfm: true,
            breaks: true,
            tables: true,
            xhtml: true,
            headerIds: false
        } );
    };

    /**
     * @param {String}   pSelector                  jQuery selector to identify APEX page item(s) for this widget.
     * @param {Object}   [pOptions]                 Optional object holding overriding options.
     * @param {Function} [pPluginInitJavascript]    Optional function which allows overriding or extending of the widget options.
     *
     * @function ckeditor5
     * @memberOf apex.widget
     * */
    widget.ckeditor5 = function( pSelector, pOptions, pPluginInitJavascript ) {

        // Based on our custom settings, add additional properties to the rich text editor options
        let options = $.extend( true, {
            mode: MODE_HTML,
            label: null,
            toolbar: TOOLBAR_INTERMEDIATE,
            minHeight: 180,
            maxHeight: null,
            // function to be executed right after editor initialization
            // function(editor){
            //     // perform any actions on the editor before any page load actions
            // }
            executeOnInitialization: null,
            // wraps the content in <div class="ck-content"></div>
            // if it contains any classes or content needing styling
            automaticStyleWrap: true,
            // what to return as Display Value, e.g in IG grid view
            displayValueMode: DISPLAY_VALUE_PLAIN_TEXT,
            // options to be passed to the CKEditor5 instance
            editorOptions: {
                language: "en",
                plugins: [
                    // alignment
                    CKEditor5.alignment.Alignment,
                    // image
                    CKEditor5.image.AutoImage,
                    CKEditor5.image.Image,
                    CKEditor5.image.ImageCaption,
                    CKEditor5.image.ImageInsert,
                    ... ( pOptions.mode === MODE_HTML ? [ CKEditor5.image.ImageResize ] : [] ),
                    CKEditor5.image.ImageStyle,
                    CKEditor5.image.ImageTextAlternative,
                    CKEditor5.image.ImageToolbar,
                    // autoformat
                    CKEditor5.autoformat.Autoformat,
                    // link
                    CKEditor5.link.AutoLink,
                    // autosave
                    CKEditor5.autosave.Autosave,
                    // blockquote
                    CKEditor5.blockQuote.BlockQuote,
                    // basicStyles
                    CKEditor5.basicStyles.Bold,
                    CKEditor5.basicStyles.Code,
                    CKEditor5.basicStyles.Italic,
                    CKEditor5.basicStyles.Strikethrough,
                    CKEditor5.basicStyles.Subscript,
                    CKEditor5.basicStyles.Superscript,
                    CKEditor5.basicStyles.Underline,
                    // codeBlock
                    CKEditor5.codeBlock.CodeBlock,
                    // essentials
                    CKEditor5.essentials.Essentials,
                    // font
                    CKEditor5.font.Font,
                    // heading
                    CKEditor5.heading.Heading,
                    // highlight
                    CKEditor5.highlight.Highlight,
                    // horizontalLine
                    CKEditor5.horizontalLine.HorizontalLine,
                    // htmlEmbed
                    ... ( pOptions.mode === MODE_HTML ? [ CKEditor5.htmlEmbed.HtmlEmbed] : [] ),
                    // indent
                    CKEditor5.indent.Indent,
                    CKEditor5.indent.IndentBlock,
                    // link
                    CKEditor5.link.Link,
                    CKEditor5.link.LinkImage,
                    // list
                    CKEditor5.list.List,
                    ... ( pOptions.mode === MODE_HTML && pOptions.toolbar === TOOLBAR_FULL ? [ CKEditor5.list.ListProperties ] : [] ),
                    CKEditor5.list.TodoList,
                    // markdown
                    ... ( pOptions.mode === MODE_MARKDOWN ? [ CKEditor5.markdown.Markdown ] : [] ),
                    // mention
                    CKEditor5.mention.Mention,
                    // pageBreak
                    CKEditor5.pageBreak.PageBreak,
                    // paragraph
                    CKEditor5.paragraph.Paragraph,
                    // pasteFromOffice
                    CKEditor5.pasteFromOffice.PasteFromOffice,
                    // removeFormat
                    CKEditor5.removeFormat.RemoveFormat,
                    // specialCharacters
                    CKEditor5.specialCharacters.SpecialCharacters,
                    CKEditor5.specialCharacters.SpecialCharactersArrows,
                    CKEditor5.specialCharacters.SpecialCharactersCurrency,
                    CKEditor5.specialCharacters.SpecialCharactersEssentials,
                    CKEditor5.specialCharacters.SpecialCharactersLatin,
                    CKEditor5.specialCharacters.SpecialCharactersMathematical,
                    CKEditor5.specialCharacters.SpecialCharactersText,
                    // table
                    CKEditor5.table.Table,
                    CKEditor5.table.TableCellProperties,
                    CKEditor5.table.TableProperties,
                    CKEditor5.table.TableToolbar,
                    CKEditor5.table.TableCaption,
                    // typing
                    CKEditor5.typing.TextTransformation,
                    // wordCount
                    CKEditor5.wordCount.WordCount
                ],
                extraPlugins: [],
                removePlugins: [],
                toolbar: (() => {
                    if( pOptions.mode === MODE_HTML ) {
                        if( pOptions.toolbar === TOOLBAR_BASIC ) {
                            return [
                                "bold", "italic", "|",
                                "bulletedList", "numberedList", "|",
                                "undo", "redo"
                            ];
                        } else if( pOptions.toolbar === TOOLBAR_INTERMEDIATE ) {
                            return [
                                "heading", "|",
                                "bold", "italic", "underline", "strikethrough", "|",
                                "link", "bulletedList", "numberedList", "|",
                                "blockQuote", "insertTable", "|",
                                "undo", "redo"
                            ];
                        } else if( pOptions.toolbar === TOOLBAR_FULL ) {
                            return [
                                "heading", "|",
                                "fontSize", "fontFamily", "fontColor", "fontBackgroundColor", "highlight", "|",
                                "bold", "italic", "underline", "strikethrough", "subScript", "superScript", "code", "removeFormat", "|",
                                "alignment", "indent", "outdent", "|",
                                "bulletedList", "numberedList", "todoList", "|",
                                "specialCharacters", "link", "blockQuote", "pageBreak", "horizontalLine", "insertTable", "codeBlock", "|",
                                "undo", "redo", "|", "htmlEmbed"
                            ];
                        }
                    } else if( pOptions.mode === MODE_MARKDOWN ) {
                        return [
                            "heading", "|",
                            "bold", "italic", "code", "|",
                            "link", "bulletedList", "numberedList", "|",
                            "blockQuote", "codeBlock", "|",
                            "undo", "redo"
                        ];
                    }
                })(),
                image: pOptions.mode === MODE_HTML ? {
                    styles: [
                        "full", "side", "alignLeft", "alignCenter", "alignRight", "alignBlockLeft", "alignBlockRight"
                    ],
                    toolbar: [
                        "imageStyle:inline",
                        // A dropdown containing `alignLeft` and `alignRight` options
                        "imageStyle:wrapText",
                        // A dropdown containing `alignBlockLeft`, `block` (default) and  `alignBlockRight` options.
                        "imageStyle:breakText",
                        // separator
                        "|",
                        // allows adding of a caption
                        "toggleImageCaption",
                        "imageTextAlternative"
                    ]
                } : {
                    styles: [],
                    toolbar: [ "imageTextAlternative" ]
                },
                codeBlock: {
                    languages: [
                        {language: "plaintext", label: "Plaintext"},
                        {language: "html", label: "HTML"},
                        {language: "css", label: "CSS"},
                        {language: "js", label: "JavaScript"},
                        {language: "sql", label: "SQL"}
                    ]
                },
                table: {
                    contentToolbar: [
                        "tableColumn",
                        "tableRow",
                        "mergeTableCells",
                        "tableCellProperties",
                        "tableProperties"
                    ]
                },
                fontSize: {
                    options: [
                        "tiny",
                        "small",
                        "default",
                        "big",
                        "huge"
                    ],
                    supportAllValues: false
                },
                markdown: {
                    markdown2html: markdown => {
                        return sanitizeHtml( markdown2html( markdown ) );
                    }
                },
                htmlEmbed: {
                    showPreviews: true,
                    sanitizeHtml: ( inputHtml ) => {
                        const outputHtml = sanitizeHtml( inputHtml );
                        return {
                            html: outputHtml,
                            hasChanged: inputHtml !== outputHtml
                        };
                    }
                }
            }
        }, pOptions );

        // JavaScript Initialization Code
        if( $.isFunction( pPluginInitJavascript ) ) {
            var changeOptions = pPluginInitJavascript( options );
            if( changeOptions ) {
                options = changeOptions;
            }
        }

        // Instantiate the CKEditor
        $( pSelector ).each( function() {
            var textArea$ = $( this ),
                widgetId = this.id + "_WIDGET",
                itemName = this.id,
                initialValue = textArea$.val(),
                itemImpl,
                editor,
                setFocus,
                deferredObject;

            textArea$
                .hide()
                .parent().append( `<div id="${util.escapeHTMLAttr( widgetId )}"></div>` );

            setFocus = function() {
                editor.focus();
            };

            itemImpl = {
                item_type: 'RICH_TEXT_EDITOR',
                delayLoading: true,
                enable: function() {
                    editor.isReadOnly = false;
                },
                disable: function() {
                    editor.isReadOnly = true;
                },
                setValue: function( pValue ) {
                    editor.setData( ( pValue === undefined || pValue === null ) ? "" : "" + pValue );
                },
                getValue: function() {
                    let data = editor.getData(),
                        stylePrefix = `<div class="ck-content">`,
                        stylePostfix = "</div>";

                    /*
                     * For mode HTML
                     * if the content contains any classes (we assume all classes are used for styling)
                     * and only for elements blockquote, code, hr and pre (as of v25.0.0)
                     * We wrap the content in a div.ck-content
                     * so the content will keep its formatting outside of the editor as well.
                     * note that ck-content styles are part of apex_ui
                     */ 
                    if( options.mode === MODE_HTML &&
                        options.automaticStyleWrap &&
                        /( class="|<blockquote|<code|<hr|<pre)/.test( data ) &&
                        !data.startsWith( stylePrefix )
                    ) {
                        data = stylePrefix + data + stylePostfix;
                    }

                    return data;
                },
                displayValueFor: function( value ) {
                    if ( options.mode === MODE_HTML ) {
                        if ( options.displayValueMode === DISPLAY_VALUE_RICH_TEXT ) {
                            return sanitizeHtml( value );
                        } else if ( options.displayValueMode === DISPLAY_VALUE_ESCAPED ) {
                            return util.escapeHTML( value );
                        } else if ( options.displayValueMode === DISPLAY_VALUE_PLAIN_TEXT ) {
                            // replace new line elements with a space, strip html tags, collapse consecutive whitespaces
                            return util.stripHTML( value.replace( /<p>|<p\/>|<br>|<br\/>|<br \/>|<BR>|\n|&nbsp;/g, " " ) ).replace(/\s\s+/g, ' ');
                        } else {
                            debug.error( "Illegal value for displayValueMode: " + options.displayValueMode );
                        }
                    } else if ( options.mode === MODE_MARKDOWN ) {
                        if ( options.displayValueMode === DISPLAY_VALUE_RICH_TEXT ) {
                            // using .ck-content as opposed to .is-markdownified on purpose to completely match the editor's styling
                            return `<div class="ck-content">${ sanitizeHtml( markdown2html( value ) ) }</div>`;
                        } else if ( options.displayValueMode === DISPLAY_VALUE_ESCAPED ) {
                            return util.escapeHTML( value );
                        } else if ( options.displayValueMode === DISPLAY_VALUE_PLAIN_TEXT ) {
                            // replace new line elements with a space, strip html tags, collapse consecutive whitespaces
                            return util.stripHTML( sanitizeHtml( markdown2html( value ) ).replace( /<p>|<p\/>|<br>|<br\/>|<br \/>|<BR>|\n|&nbsp;/g, " " ) ).replace(/\s\s+/g, ' ');
                        } else {
                            debug.error( "Illegal value for displayValueMode: " + options.displayValueMode );
                        }
                    }
                },
                setFocusTo: function() {
                    return {focus: setFocus};
                },
                isChanged: function() {
                    // use getValue as opposed to editor.getData
                    // othwerwise there might be discrepancies with the potential wrapping of div.ck-content
                    return initialValue !== this.getValue();
                },
                getEditor: function() {
                    return editor;
                },
                getValidity: function() {
                    // must synchronize with original textarea
                    // to reuse the native validation functionality
                    textArea$.val( this.getValue() );
                    return textArea$[0].validity || {valid: true};
                },
                getPopupSelector: function() {
                    // when editing in the Interactive Grid
                    // the editor would be hidden when various editor popup-s would receive focus
                    // e.g. the Link popup
                    // providing a general selector that all popups share fixes this
                    return ".ck.ck-balloon-panel";
                }
            };

            deferredObject = item.create( itemName, itemImpl );

            options.editorOptions.initialData = initialValue;

            CKEditor5.editorClassic.ClassicEditor
                .create( document.getElementById( widgetId ), options.editorOptions )
                .then( function( newEditor ) {
                    editor = newEditor;

                    editor.editing.view.change( writer => {
                        writer.setStyle( "min-height", ( options.minHeight || 150 ) + "px", editor.editing.view.document.getRoot() );
                        if( options.maxHeight ) {
                            writer.setStyle( "max-height", options.maxHeight + "px", editor.editing.view.document.getRoot() );
                        }
                    } );

                    var snapshot;
                    editor.editing.view.document.on( "change:isFocused", function( evt, data, isFocused ) {
                        if( isFocused ) {
                            snapshot = editor.getData();
                        } else if( editor.getData() !== snapshot ) {
                            textArea$.trigger( "change" );
                        }
                    } );

                    // the aria label seems to not be configurable, so we simply override it
                    var newLabel = lang.formatMessage( "APEX.RICH_TEXT_EDITOR.ACCESSIBLE_LABEL", options.label || "" );
                    $( "label.ck.ck-label.ck-voice-label", editor.ui.view.element ).text( newLabel );

                    if( options.executeOnInitialization ) {
                        options.executeOnInitialization( editor );
                    }

                    // normalizes the initial value. e.g: test => <p>test</p>
                    // this ensures that in such cases, no unnecessary "Value has changed" warnings are raised
                    initialValue = itemImpl.getValue();
                    deferredObject.resolve();
                } )
                .catch( function( error ) {
                    debug.error( error );
                } );

            // register focus handling, so when the non-displayed textarea of the CKEditor
            // receives focus, focus is moved to the editor.
            textArea$.focus( setFocus );
        } );
    }; // ckeditor5

    /*
     * Allow various editor popups to work in jQuery UI dialogs
     */
    if( $.ui.dialog ) {
        $.widget( "ui.dialog", $.ui.dialog, {
            _allowInteraction: function( event ) {
                return $( event.target ).closest( ".ck" ).length > 0 || this._super( event );
            }
        } );
    }

} )( apex.item, apex.lang, apex.util, apex.widget, apex.debug, apex.jQuery );