/*!
 Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 */

/* global gInstalledApps */
( function ( util, apexItem, apexRegion, lang, locale, navigation, page, message, env, debug, $ ) {
    "use strict";

    const ORACLE_REPO_BASE_URL = "https://raw.githubusercontent.com/oracle/apex/",
          ORACLE_AUTHOR_NAME = "Oracle",
          INSTALL_AJAX_CALLBACK = "INIT_INSTALL_APP",
          REMOVE_APP_ID_ITEM = "P50_REMOVE_FLOW_ID",
          APP_GROUP_ITEM = "P50_APP_GROUP",
          DEFAULT_APP_ICON = env.APEX_FILES + "apex_ui/img/icons/pkg-apps/app-placeholder.svg",
          APEX_BASE_VERSION = env.APEX_BASE_VERSION,
          MSG_REMOVE_APP = lang.getMessage( "GALLERY.REMOVE_APP" ),
          MSG_FILE_DOWNLOAD_ERROR = lang.getMessage( "GALLERY.FILE_DOWNLOAD_ERROR" ),
          MSG_MISSING_VALUES_ERROR = lang.getMessage( "GALLERY.MISSING_VALUES_ERROR" );

    let spinner$;

    /**
     * Download a file from a given URL and return the response as blob
     * @function loadData
     * @param {string} pFileUrl
     * @return {Promise<object>} file blob
     */
    async function loadData( pFileUrl = "" ) {
        let errorResponse;

        function _throwNetworkError( pResponse ) {
            debug.error( `Failed network request for ${pResponse.url}`, pResponse );
            showError( MSG_FILE_DOWNLOAD_ERROR );
            throw new Error( `Failed network request: ${pResponse.status} ${pResponse.statusText}` );
        }

        async function _fetchFile() {
            let data;

            showSpinner();

            if ( pFileUrl ) {
                data = await fetch( pFileUrl, {
                    method: "GET",
                    mode: "cors",
                    credentials: "omit",
                    cache: "no-store",
                    redirect: "follow",
                    referrerPolicy: "no-referrer"
                } )
                    .then( ( response ) => {
                        if ( response.ok ) {
                            return response.blob();
                        } else {
                            errorResponse = response;
                        }
                    } )
                    .catch( () => {
                        errorResponse = { url: pFileUrl, status: "failed", statusText: "Connection refused" };
                    } );
            }

            if ( errorResponse ) {
                _throwNetworkError( errorResponse );
            }

            removeSpinner();

            return data;
        }

        return await _fetchFile();
    }

    /**
     * Shows a waiting spinner
     * @function showSpinner
     */
    function showSpinner() {
        spinner$ = util.showSpinner( $( "body" ) );
    }

    /**
     * Remove the waiting spinner
     * @function removeSpinner
     */
    function removeSpinner() {
        if ( spinner$ && spinner$.length > 0 ) {
            spinner$.remove();
        }
    }

    /**
     * Show a better looking confirm dialog with icon and a more modern UI
     * @function showConfirm
     * @param {string} pMessage
     * @param {string} pOkLabel
     * @param {string} pStyle
     * @param {function} callback
     */
    function showConfirm( pMessage, pOkLabel, pStyle, callback ) {
        removeSpinner();

        message.showDialog( util.applyTemplate( pMessage ), {
            confirm: true,
            callback: function ( ok ) {
                if ( !ok ) {
                    callback( false );
                } else {
                    callback( true );
                }
            },
            okLabel: pOkLabel,
            dialogClass: "ui-dialog--notification",
            unsafe: false,
            modern: true,
            style: pStyle || "information"
        } );
    }

    /**
     * Shows a error message
     * @function showError
     * @param {string} pMessage
     */
    function showError( pMessage ) {
        removeSpinner();

        message.clearErrors();
        message.showErrors( [
            {
                type: "error",
                location: "page",
                message: pMessage,
                unsafe: false
            }
        ] );
    }

    /**
     * Saves the fetched file with additional information to DB using an multipart/form-data request to not run in any size limitations
     * (AJAX callback stores that information on the server side)
     * @function saveInstallFile
     * @param {string} pAppName
     * @param {string} pAppDesc
     * @param {string} pAppAuthor
     * @param {string} pAppIntName
     * @param {string} pAppIcon
     * @param {blob} pAppFile
     */
    async function saveInstallFile( pAppName, pAppDesc, pAppAuthor, pAppIntName, pAppIcon, pAppFile ) {
        const errorMsg = lang.formatMessage( "GALLERY.INSTALL_APP_ERROR", pAppName ),
              appFileName = "packaged_app_" + new Date().getTime() + ".zip",
              appFile = new File( [pAppFile], appFileName, { type: pAppFile.type, lastModified: new Date().getTime() } );
        let result;

        // FormData request using fetch
        async function _makeFormDataRequest( pFormData ) {
            let errorResponse;

            const response = await fetch( "wwv_flow.ajax", {
                method: "POST",
                mode: "same-origin",
                cache: "no-store",
                body: pFormData
            } )
                .then( ( response ) => {
                    if ( response.ok ) {
                        return response.json();
                    } else {
                        errorResponse = { success: false, error: response.status + " " + response.statusText };
                    }
                } )
                .catch( () => {
                    errorResponse = { success: false, error: "Connection refused" };
                } );

            return errorResponse || response;
        }

        // build formData request and upload file to server
        async function _uploadFile( pFile ) {
            let formData = new FormData(),
                response;

            formData.append( "p_request", "APPLICATION_PROCESS=" + INSTALL_AJAX_CALLBACK );
            formData.append( "p_flow_id", env.APP_ID );
            formData.append( "p_flow_step_id", env.APP_PAGE_ID );
            formData.append( "p_instance", env.APP_SESSION );
            formData.append( "p_debug", $v( "pdebug" ) );
            formData.append( "F01", pFile, pFile.name );
            formData.append( "X01", pAppName );
            formData.append( "X02", pAppDesc );
            formData.append( "X03", pAppAuthor );
            formData.append( "X04", pAppIntName );
            formData.append( "X05", pAppIcon );

            response = await _makeFormDataRequest( formData );

            return response;
        }

        showSpinner();

        result = await _uploadFile( appFile );

        if ( result.success ) {
            removeSpinner();
            if ( result.url ) {
                navigation.redirect( result.url );
            }
        } else {
            debug.error( "upload error", result.error );
            showError( errorMsg );
        }
    }

    /**
     * Removes a installed app (submits page which executes delete logic)
     * @function removeInstalledApp
     * @param {object} pRemoveButton$
     */
    function removeInstalledApp( pRemoveButton$ ) {
        const appName = pRemoveButton$.data( "name" ),
              appId = pRemoveButton$.data( "app-id" ),
              confirmMsg = lang.formatMessage( "GALLERY.REMOVE_APP_CONFIRM", appName, appId );

        showConfirm( confirmMsg, MSG_REMOVE_APP, "danger", function ( okPressed ) {
            if ( okPressed ) {
                page.submit( {
                    request: "REMOVE_APP",
                    set: { [REMOVE_APP_ID_ITEM]: appId },
                    showWait: true
                } );
            }
        } );
    }

    /**
     * Initializes the installation process by downloading zip file and storing it with additional info to DB
     * @function processInstallFile
     * @param {object} pInstallButton$
     */
    async function processInstallFile( pInstallButton$ ) {
        const appName = pInstallButton$.data( "name" ),
              appDesc = pInstallButton$.data( "desc" ),
              appAuthor = pInstallButton$.data( "author" ),
              appIntName = pInstallButton$.data( "internal-name" ),
              appIcon = pInstallButton$.data( "icon" ),
              zipUrl = pInstallButton$.data( "zip-url" );

        if ( !zipUrl ) {
            showError( MSG_FILE_DOWNLOAD_ERROR );
            return;
        }

        if ( !appAuthor || !appIntName ) {
            debug.error( "Missing required values: author, internalName" );
            showError( MSG_MISSING_VALUES_ERROR );
            return;
        }

        const fileData = await loadData( zipUrl );

        if ( fileData ) {
            await saveInstallFile( appName, appDesc, appAuthor, appIntName, appIcon, fileData );
        }
    }

    /**
     * Logic for switching between tabs (sample apps, starter apps, custom apps)
     * @function switchAppGroupTab
     */
    function switchAppGroupTab() {
        let regionName = util.escapeCSS( apexItem( APP_GROUP_ITEM ).getValue() ),
            region$ = $( "#" + regionName ),
            regionContent$ = $( "#" + regionName + "_CONTENT" );

        $( ".apps_region, .apps_region_container" ).hide();
        region$.show();

        if ( region$.hasClass( "apps_region" ) ) {
            $( "#" + regionName + "_ClientSideTemplator" ).css( "width", "100%" );
            apexRegion( regionName ).call( "resize" );
        }

        if ( region$.hasClass( "apps_region_container" ) ) {
            if ( regionContent$.length > 0 ) {
                regionContent$.show();
                $( "#" + regionName + "_CONTENT_ClientSideTemplator" ).css( "width", "100%" );
                apexRegion( regionName + "_CONTENT" ).call( "resize" );
            }
        }
    }

    /**
     * JS init code of "Client Side Templator" regions used on the page
     * @function clientSideTemplatorJSInitCode
     * @param {object} pOptions
     * @return {object} modified options
     */
    function clientSideTemplatorJSInitCode( pOptions ) {
        const defaultLang = "en",
              lang = locale.getLanguage().toLowerCase() || defaultLang,
              installedApps = gInstalledApps || [];
        
        function _installedAppMatch( pInstalledApp, pInternalName, pAuthor ) {
            return pInstalledApp.name === pInternalName.toLowerCase() && pInstalledApp.author === pAuthor.toLowerCase();
        }

        // don't show built in errors, we have our own error handling on the page using events provided by the plugin
        pOptions.showErrors = false;

        // prepare the row data which is used for our html record template
        pOptions.prepareRowData = function ( row ) {
            let newRow = {},
                hasLangObject = row.i18n && row.i18n[lang],
                hasLangName = hasLangObject && row.i18n[lang].name,
                hasLangDesc = hasLangObject && row.i18n[lang].description;

            newRow.name = row.name ? row.name : hasLangName ? row.i18n[lang].name : row.i18n[defaultLang].name;
            newRow.description = row.description ? row.description : hasLangDesc ? row.i18n[lang].description : row.i18n[defaultLang].description;
            newRow.lang = hasLangName ? lang : row.i18n && row.i18n[defaultLang] && row.i18n[defaultLang].name ? defaultLang : "";
            newRow.internalName = row.internalName ? row.internalName.toLowerCase() : "";
            newRow.author = row.author ? row.author : "";
            newRow.version = row.version ? row.version : "";
            newRow.apexVersion = row.apexVersion ? row.apexVersion : APEX_BASE_VERSION;
            newRow.icon = row.icon ? row.icon : DEFAULT_APP_ICON;
            newRow.url = row.url ? row.url : "";
            newRow.zip = row.zip ? row.zip : "";

            newRow.appId = ( installedApps.find( ( installedApp ) => _installedAppMatch( installedApp, newRow.internalName, installedApp.author ) ) || {} ).id || "";
            newRow.appLink = ( installedApps.find( ( installedApp ) => _installedAppMatch( installedApp, newRow.internalName, installedApp.author ) ) || {} ).link || "";
            newRow.isInstalled = installedApps.some( ( installedApp ) => _installedAppMatch( installedApp, newRow.internalName, installedApp.author ) ) ? "Y" : "N";
            newRow.installedVersion = ( installedApps.find( ( installedApp ) => _installedAppMatch( installedApp, newRow.internalName, installedApp.author ) ) || {} ).version || "";
            newRow.updateAvailable = installedApps.some( ( installedApp ) => _installedAppMatch( installedApp, newRow.internalName, installedApp.author ) && installedApp.version !== newRow.version ) ? "Y" : "N";
            newRow.isOracle = newRow.author.toLowerCase() === ORACLE_AUTHOR_NAME.toLowerCase() && newRow.zip.startsWith( ORACLE_REPO_BASE_URL ) ? "Y" : "N";
            newRow.isCompatible = parseFloat( newRow.apexVersion ) <= parseFloat( APEX_BASE_VERSION ) ? "Y" : "N";

            return newRow;
        };

        // filter data to only display correct entries
        pOptions.filterData = function( data ) {
            data = data.filter( function ( item ) {
                return item.isCompatible === "Y";
            } );
            // sort by apexVersion descending & remove duplicate entries --> if same app is supplied with different versions only keep newest
            data = data.sort( ( a, b ) => parseFloat( b.apexVersion ) - parseFloat( a.apexVersion ) );
            data = data.filter( ( item, index, array ) => array.findIndex( elem => ( elem.internalName === item.internalName && elem.author === item.author ) ) === index );

            return data;
        };

        // sort our results by application name alphabetically
        pOptions.sortData = function ( a, b ) {
            let nameA = a.name.toLowerCase(),
                nameB = b.name.toLowerCase();

            return nameA < nameB ? -1 : nameA > nameB ? 1 : 0;
        };

        // search logic when something is entered into the search item
        // search in name, description & author fields
        pOptions.searchItemFilter = function ( data, fields, searchValue ) {
            const fieldKeyName = fields.NAME ? fields.NAME.index : 0,
                  fieldKeyDesc = fields.DESCRIPTION ? fields.DESCRIPTION.index : 0,
                  fieldKeyAuthor = fields.AUTHOR ? fields.AUTHOR.index : 0;

            data = data.filter( function ( item ) {
                return (
                    item[fieldKeyName].toLowerCase().includes( searchValue.toLowerCase() ) ||
                    item[fieldKeyDesc].toLowerCase().includes( searchValue.toLowerCase() ) ||
                    item[fieldKeyAuthor].toLowerCase().includes( searchValue.toLowerCase() )
                );
            } );
            return data;
        };

        return pOptions;
    }

    // make some functions public for integration into page
    window.processInstallFile = processInstallFile;
    window.removeInstalledApp = removeInstalledApp;
    window.switchAppGroupTab = switchAppGroupTab;
    window.clientSideTemplatorJSInitCode = clientSideTemplatorJSInitCode;
} )( apex.util, apex.item, apex.region, apex.lang, apex.locale, apex.navigation, apex.page, apex.message, apex.env, apex.debug, apex.jQuery );
