set define '^' verify off
prompt ...wwv_flow_upgrade.sql
create or replace package wwv_flow_upgrade as
--------------------------------------------------------------------------------
--
--  Copyright (c) 2001, 2022, Oracle and/or its affiliates.
--
--    NAME
--      wwv_flow_upgrade
--
--    DESCRIPTION
--      This package facilitates flows upgrades.
--      Follow the following steps to upgrade:
--      1. Install a new version of flows into a "new flows" schema.
--      2. Login to sqlplus as the "new flows" user.
--      3. exec wwv_flow_upgrade.copy_flow_meta_data('FLOWS1','FLOWS2')
--         assume flows1 is the old schema and flows2 is the new schema.
--      4. Review the upgrade log (see example query)
--      5. exec wwv_flow_upgrade.recreate_public_synonyms('FLOWS2')
--
--    NOTES
--      Example Log Query:
--         column upgrade_sequence format 9999
--         column upgrade_action format a44 wrapped
--         column upgrade_error format a50 wrapped
--         column upgrade_command format a50 wrapped
--         set linesize 160
--         set pagesize 1000
--         select upgrade_sequence, upgrade_action, upgrade_error, upgrade_command
--         from   wwv_flow_upgrade_progress
--         order by upgrade_id desc, upgrade_sequence desc
--
--      Example Log Query 2:
--         column upgrade_sequence format 9999
--         column upgrade_action format a44 wrapped
--         column upgrade_error format a50 wrapped
--         column upgrade_command format a50 wrapped
--         set linesize 160
--         set pagesize 1000
--         select upgrade_sequence, upgrade_action, upgrade_error, upgrade_command
--         from   wwv_flow_upgrade_progress
--         order by upgrade_id desc, upgrade_sequence
--
--    RUNTIME DEPLOYMENT: YES
--
--    MODIFIED   (MM/DD/YYYY)
--      mhichwa    07/03/2001 - Created
--
--------------------------------------------------------------------------------

type d_arr   is table of date            index by binary_integer;

g_f                    number;
c                      number;
p                      number;
i                      number;
g_row_cnt              number := 0;
g_session_seq1         number := 0;
g_session_seq2         number := 0;
g_pref_name            wwv_flow.flow_vc_arr;
g_pref_value           wwv_flow.flow_vc_arr;
g_pref_desc            wwv_flow.flow_vc_arr;
g_pref_created_on      d_arr;
g_pref_last_updated_on d_arr;
g_tabs                 wwv_flow.flow_vc_arr;

subtype t_install_phase is varchar2(4);
c_phase_pre  constant t_install_phase := 'PRE';
c_phase_hot  constant t_install_phase := 'HOT';
c_phase_post constant t_install_phase := 'POST';

c_cdn        constant varchar2(255) := 'https://static.oracle.com/cdn/apex/';

--==============================================================================
procedure increment_session(
    p_old_schema        in varchar2,
    p_new_schema        in varchar2 );

--==============================================================================
procedure init (
    p_owner in varchar2,
    p_phase in t_install_phase default null );

--==============================================================================
procedure disable_cons_and_trg (
   p_owner_name        in varchar2,
   p_object_name       in varchar2 );

--==============================================================================
procedure enable_cons_and_trg (
   p_owner_name        in varchar2,
   p_object_name       in varchar2 );

--==============================================================================
-- drop public synonyms to flow objects
--
-- sqlplus example:
--    exec wwv_flow_upgrade.drop_public_synonyms;
--==============================================================================
procedure drop_public_synonyms (
    p_drop_all in boolean default false );

--==============================================================================
-- create public synonyms only without dropping or granting
-- sqlplus example:
--    exec wwv_flow_upgrade.create_public_synonyms('FLOWS2')
--==============================================================================
procedure create_public_synonyms (
   p_owner_to          in varchar2 );

--==============================================================================
-- issue grants to public synonyms only without dropping or creating
-- sqlplus example:
--    exec wwv_flow_upgrade.grant_public_synonyms('FLOWS2')
--==============================================================================
procedure grant_public_synonyms (
   p_owner_to          in varchar2 );

--==============================================================================
-- create all needed public synonyms for a flows environment.
--
-- sqlplus example:
--    exec wwv_flow_upgrade.recreate_public_synonyms('FLOWS2')
--==============================================================================
procedure recreate_public_synonyms (
   p_owner_to          in varchar2 );

--==============================================================================
-- Copy flows meta data from once schema to another.
--
-- sqlplus example:
--    exec wwv_flow_upgrade.copy_flow_meta_data('FLOWS1','FLOWS2')
--==============================================================================
procedure copy_flow_meta_data (
   p_owner_from        in varchar2,
   p_owner_to          in varchar2,
   p_cdb_install       in varchar2 default 'NO' );

--==============================================================================
procedure drop_upgrade_triggers (
    p_from in varchar2 );

--==============================================================================
-- revert phase2, so old schema is usable again
--==============================================================================
procedure rollback_phase2 (
    p_from in varchar2 );

--==============================================================================
-- Remove dev-only jobs for a schema
--==============================================================================
procedure remove_dev_jobs (
    p_owner     in varchar2 );

--==============================================================================
-- Remove all of the standard dbms_jobs for a schema
--==============================================================================
procedure remove_jobs;

--==============================================================================
-- Create dev-only jobs for a schema
--==============================================================================
procedure create_dev_jobs (
    p_owner     in varchar2 );

--==============================================================================
-- Create all of the standard dbms_jobs for a given schema
--==============================================================================
procedure create_jobs (
    p_owner     in varchar2 );

--==============================================================================
-- In new APEX instances, jobs in the APEX_nnnnnn schema are created disabled by
-- default. Creating the first app workspace automatically enables the jobs via
-- this procedure.
--==============================================================================
procedure enable_internal_jobs;

--==============================================================================
-- Create the proper grants and synonyms for the flows_files schema
-- used for file upload / download
--==============================================================================
procedure flows_files_objects_create (
    p_flow_owner    in varchar2,
    p_owner         in varchar2 default 'FLOWS_FILES' );

--==============================================================================
-- Remove the proper grants and synonyms for the flows_files schema
-- used for file upload / download
--==============================================================================
procedure flows_files_objects_remove (
    p_flow_owner    in varchar2,
    p_owner         in varchar2 default 'FLOWS_FILES' );

--==============================================================================
procedure meta_cleanup (
    p_from      in varchar2,
    P_to        in varchar2 );

--==============================================================================
procedure report_column_cleanup;

--==============================================================================
procedure template_name_cleanup(
    p_owner     in varchar2);

--==============================================================================
-- This procedure should be run after an additional copy of flows is installed,
-- as it is the final step in the upgrade process.
-- Once this procedure is run, you will be running in the new version of flows.
-- This will:
--     1.  Remove all old dbms_jobs
--     2.  Create all new dbms_jobs
--     3.  Drop All public synonyms
--     4.  Create all new public synonyms
--     5.  Perform all necessary grants on public synoyms
--==============================================================================
procedure switch_schemas (
    p_from      in varchar2,
    P_to        in varchar2 );

--==============================================================================
-- Copy and migrate metadata in POST phase. The switch_schemas procedure above
-- creates a DB job to run this procedure.
--==============================================================================
procedure copy_post_metadata (
    p_from      in varchar2,
    p_to        in varchar2 );

--==============================================================================
-- Copys old preferences from schema1.WWV_FLOW_PLATFORM_PREFS
-- to schema2.WWV_FLOW_PLATFORM_PREFS
--==============================================================================
procedure copy_prefs (
    p_owner_from        in varchar2,
    p_owner_to          in varchar2 );

--==============================================================================
-- Native Plug-ins can have attributes of scope APPLICATION. This procedure will
-- make sure that new attributes or new plug-ins are added for existing applications.
--==============================================================================
procedure update_plugin_settings (
    p_to in varchar2 );

function to_template_id(
  p_in varchar2)
  return number;

procedure upgrade_to_latest (
    p_owner             in varchar2,
    p_from              in varchar2)
    ;

procedure upgrade_wwv_flow_flash_chart (
    p_flash_chart in wwv_flow_flash_charts%rowtype );

procedure upgrade_flash_to_jet (
    p_flash_chart_id in number );

procedure upgrade_flash_series_to_jet (
    p_series in wwv_flow_flash_chart5_series%rowtype );

procedure upgrade_html_chart (
    p_plug in wwv_flow_page_plugs%rowtype );

procedure upgrade_svg_chart (
    p_plug       in wwv_flow_page_plugs%rowtype);

procedure upgrade_svg_chart_series (
    p_plug        in wwv_flow_page_plugs%rowtype,
    p_series_id   in number,
    p_a001        in varchar2 default null,
    p_a002        in varchar2 default null,
    p_a003        in varchar2 default null,
    p_a004        in varchar2 default null,
    p_a005        in varchar2 default null );

procedure sw_cleanup (
    p_from    in varchar2,
    p_to      in varchar2)
    ;

procedure enable_ws_constraints (
    p_to      in varchar2 )
    ;

--==============================================================================
-- HTML-escapes a string without affecting item references like &P1_ITEM.
--==============================================================================
function escape_but_allow_items (
    p_input   in varchar2 )
    return varchar2;

--==============================================================================
-- Set the patch status to APPLIED
--
-- PARAMETERS
--   * p_from: The schema you upgraded from
--   * p_to: The schema you upgraded to
--==============================================================================
procedure set_patch_applied (
    p_from in varchar2 );

--==============================================================================
-- Sets the table comment and column comments in bulk to reduce network roundtrips
--
-- PARAMETERS
--   * p_table_name: The table name
--   * p_table_comment: The table comment
--   * p_column_comments: A plist of column names and column comments
--==============================================================================
procedure add_table_comments (
    p_table_name        in varchar2,
    p_table_comment     in varchar2,
    p_column_comments   in wwv_flow_t_varchar2 );

end wwv_flow_upgrade;
/
show errors
