set define off verify off
prompt ...wwv_flow_template_directive
create or replace package wwv_flow_template_directive authid definer as
--------------------------------------------------------------------------------
--
--  Copyright (c) 2021, 2022, Oracle and/or its affiliates.
--
--    NAME
--      wwv_flow_template_directive.sql
--
--    DESCRIPTION
--      This package is responsible for handling templates directives in the runtime engine.
--
--      Use template directives to control how attributes that support substitution strings are processed.
-- 
--      These directives are processed as part of server side substitutions. 
--      See apex.util.applyTemplate in Oracle Application Express JavaScript API Reference for more details on directive syntax.
--
--      Currently support for:
--          - If Condition Directives (if elseif else endif)
--          - Case Condition Directives (case when otherwise endcase)
--          - Loop Directives (loop endloop)
--          - Comment Directives (!)
--
--    MODIFIED   (MM/DD/YYYY)
--      mhoogend  12/02/2021 - Created
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Statement Type Constants
--------------------------------------------------------------------------------
subtype t_token_type            is varchar2( 20 );

--------------------------------------------------------------------------------
-- If Type Constants
--------------------------------------------------------------------------------
subtype t_if_type               is pls_integer range 0 .. 5;

--------------------------------------------------------------------------------
-- If Condition Definition
--------------------------------------------------------------------------------
type t_if_condition is record (
    if_type                     t_if_type,
    substitution                wwv_flow_session_state.t_substitution );

--------------------------------------------------------------------------------
-- Loop Type Constants
--------------------------------------------------------------------------------
subtype t_loop_type             is pls_integer range 0 .. 1;

c_loop_delimitered              constant t_loop_type := 0;
c_loop_context                  constant t_loop_type := 1;

--------------------------------------------------------------------------------
-- Loop Definition
--------------------------------------------------------------------------------
type t_loop is record (
    loop_type                   t_loop_type,
    separator                   varchar2( 32767 ),
    substitution                wwv_flow_session_state.t_substitution );

--------------------------------------------------------------------------------
-- Token Definition
--------------------------------------------------------------------------------
type t_token is record (
    token_type                  t_token_type,
    -- extra attributes per statement type (if needed)
    tk_text                     clob,
    tk_static_substitution      wwv_flow_session_state.t_substitution,
    tk_data_substitution        wwv_flow_session_state.t_substitution,
    tk_if                       t_if_condition,
    tk_elseif                   t_if_condition,
    tk_case                     wwv_flow_session_state.t_substitution,
    tk_when                     varchar2( 32767 ),
    tk_loop                     t_loop );

--------------------------------------------------------------------------------
-- Array of Token Definitions.
--------------------------------------------------------------------------------
type t_tokens                   is table of t_token index by pls_integer;

--------------------------------------------------------------------------------
-- Statement Type Constants
-----------------------------------------------------------------------
subtype t_statement_type        is pls_integer range 0 .. 1;

--------------------------------------------------------------------------------
-- Statement Definition
--------------------------------------------------------------------------------
type t_statement is record (
    statement_type              t_statement_type,
    token                       t_token,
    goto_block                  pls_integer );

--------------------------------------------------------------------------------
-- Array of Statement Definitions.
--------------------------------------------------------------------------------
type t_statements               is table of t_statement index by pls_integer;

--------------------------------------------------------------------------------
-- Statement Block Type Constants
--------------------------------------------------------------------------------
subtype t_statement_block_type  is varchar2( 6 );

c_block_simple                  constant t_statement_block_type := 'SIMPLE';
c_block_if                      constant t_statement_block_type := 'IF';
c_block_case                    constant t_statement_block_type := 'CASE';
c_block_loop                    constant t_statement_block_type := 'LOOP';

--------------------------------------------------------------------------------
-- Statement Block Definition
--------------------------------------------------------------------------------
type t_statement_block is record (
    parent_id                   pls_integer,
    block_type                  t_statement_block_type,
    statements                  t_statements,
    -- Extra attributes used during compilation
    valid                       boolean,
    missing_token               t_token_type );

--------------------------------------------------------------------------------
-- Array of Token Block Definitions.
--------------------------------------------------------------------------------
type t_statement_blocks         is table of t_statement_block index by pls_integer;

--------------------------------------------------------------------------------
-- Placeholder Type Constants
--------------------------------------------------------------------------------
subtype t_placeholder_type      is pls_integer range 1..2;

c_placeholder_varchar2          constant t_placeholder_type := 1;
c_placeholder_varchar2s         constant t_placeholder_type := 2;

--------------------------------------------------------------------------------
-- Placeholder definition. placeholders can be passed when a template is 
-- applied. 
--
-- Attributes:
-- * placeholder_type:  Data type of the placeholder.
-- * value_varchar2:    VARCHAR2 value of the placeholder.
-- * values_varchar2:   VARCHAR2 values of the placeholder.
--------------------------------------------------------------------------------
type t_placeholder is record (
    placeholder_type            t_placeholder_type,
    value_varchar2              varchar2( 32767 ),
    values_varchar2             wwv_flow_t_varchar2 );

--------------------------------------------------------------------------------
-- Array of input placeholders.
-- index by placeholder_name:   This is the name of a data value. It is upper case alphanumeric 
--                              plus underscore and dollar sign. 
--------------------------------------------------------------------------------
type t_placeholders             is table of t_placeholder index by varchar2( 32767 );

--==============================================================================
-- Analyzes a template on placeholders, substitutions and directives
-- Compiles the results to statement blocks which are used to apply the template.
--
-- ARGUMENTS:
-- * p_template:                    The template to compile
-- * p_placeholders_only:           Whether to only substitute placeholders and ignore 
--                                  others like session state substitution 
-- * p_supports_context:            Whether looping over a query context is supported
--                         
--
-- Example:
--
--   declare
--      l_blocks    wwv_flow_template_directive.t_statement_blocks;
--      l_with      wwv_flow_template_directive.t_placeholders;
--      l_output    clob;
--   begin
--      l_blocks := wwv_flow_template_directive.compile(
--          p_template          => 'Hello {if NAME/}#NAME#{else/}World{endif/}!' );
--
--      wwv_flow_template_directive.add_placeholder(
--          p_placeholders      => l_with,
--          p_name              => 'NAME',
--          p_value             => 'Scott' );
--
--      l_output := wwv_flow_template_directive.apply( 
--          p_statement_blocks  => l_blocks,
--          p_with              => l_with,
--          p_escape_mode       => wwv_flow_session_state.c_escape_mode_raw );
--
--      dbms_output.put_line( l_output );
--   end;
--
--   Output:
--
--   Hello Scott!
--
-- Returns:
-- The statement blocks.
--==============================================================================
function compile(
    p_template          in clob,
    p_placeholders_only in boolean  default false,
    p_supports_context  in boolean  default false )
    return t_statement_blocks;

--==============================================================================
-- Transforms a compiled template to a JSON object
-- The JSON can be stored as BLOB and turned into statement blocks
-- using the to_statement_blocks function
--
-- ARGUMENTS:
-- * p_statement_blocks:    The compiled statement blocks from a template
--
--
-- Example:
--
--   declare
--      l_blocks    wwv_flow_template_directive.t_statement_blocks;
--      l_json      blob;
--      l_with      wwv_flow_template_directive.t_placeholders;
--      l_output    clob;
--   begin
--      l_blocks := wwv_flow_template_directive.compile(
--          p_template          => 'Hello {if NAME/}#NAME#{else/}World{endif/}!' );
--
--      l_json := wwv_flow_template_directive.to_json(
--          p_statement_blocks  => l_blocks );
--
--      wwv_flow_template_directive.add_placeholder(
--          p_placeholders      => l_with,
--          p_name              => 'NAME',
--          p_value             => 'Scott' );
--
--      l_output := wwv_flow_template_directive.apply( 
--          p_statement_blocks  => wwv_flow_template_directive.to_statement_blocks( l_json ),
--          p_with              => l_with,
--          p_escape_mode       => wwv_flow_session_state.c_escape_mode_raw );
--
--      dbms_output.put_line( l_output );
--   end;
--
--   Output:
--
--   Hello Scott!
--
-- Returns:
--  The statement blocks in JSON format.
--==============================================================================
function to_json(
    p_statement_blocks in t_statement_blocks )
    return blob;

--==============================================================================
-- Analyzes a template on placeholders, substitutions and directives
-- Compiles the results to statement blocks which are used to apply the template.
--
-- ARGUMENTS:
-- * p_template:                    The template to compile
-- * p_placeholders_only:           Whether to only substitute placeholders and ignore 
--                                  others like session state substitution 
-- * p_supports_context:            Whether looping over a query context is supported
--                         
--
-- Example:
--
--   declare
--      l_json      blob;
--      l_with      wwv_flow_template_directive.t_placeholders;
--      l_output    clob;
--   begin
--
--      l_json := wwv_flow_template_directive.compile_to_json(
--          p_template          => 'Hello {if NAME/}#NAME#{else/}World{endif/}!' );
--
--      wwv_flow_template_directive.add_placeholder(
--          p_placeholders      => l_with,
--          p_name              => 'NAME',
--          p_value             => 'Scott' );
--
--      l_output := wwv_flow_template_directive.apply( 
--          p_statement_blocks  => wwv_flow_template_directive.to_statement_blocks( l_json ),
--          p_with              => l_with,
--          p_escape_mode       => wwv_flow_session_state.c_escape_mode_raw );
--
--      dbms_output.put_line( l_output );
--   end;
--
--   Output:
--
--   Hello Scott!
--
-- Returns:
-- JSON object of compiled statement blocks
--==============================================================================
function compile_to_json(
    p_template          in clob,
    p_placeholders_only in boolean  default false,
    p_supports_context  in boolean  default false )
    return blob;

--==============================================================================
-- Compiles the template and returns an error message if the template is
-- invalid.
--
-- ARGUMENTS:
-- * p_template:                    The template to compile
-- * p_placeholders_only:           Whether to only substitute placeholders and ignore 
--                                  others like session state substitution 
-- * p_supports_context:            Whether looping over a query context is supported
--                         
--
-- Returns:
-- The error message
--==============================================================================
function validate_template(
    p_template          in clob,
    p_placeholders_only in boolean  default false,
    p_supports_context  in boolean  default false )
    return varchar2;

--==============================================================================
-- Validates the a template placeholder name
-- Allowed characters are upper case alphanumeric plus underscore and dollar sign. 
-- When invalid, an error is raised.
--
-- ARGUMENTS:
-- * p_name:                        The name of the placeholder
--==============================================================================
procedure validate_placeholder_name (
    p_name              in varchar2 );

--==============================================================================
-- Validates the a template placeholder name
-- Allowed characters are upper case alphanumeric plus underscore and dollar sign. 
-- When invalid, an error is raised.
--
-- ARGUMENTS:
-- * p_name:                        The name of the placeholder
--
-- Returns:
-- The error message
--==============================================================================
function validate_placeholder_name (
    p_name              in varchar2 )
    return varchar2;

--==============================================================================
-- Adds a placeholder to the placeholder collection. Placeholder collections 
-- can be passed to the APPLY calls. Placeholders can be used in the following ways:
--  - by directives
--  - As placeholder substitutions
--
-- Parameters:
-- * p_placeholders:    Placeholder collection.
-- * p_name:            This is the name of a data value. It is upper case alphanumeric 
--                      plus underscore and dollar sign. 
-- * p_value:           Value of the placeholder. This can be any value. The value
--                      may contain template directives which will be processed.
--
--
-- Example:
--
--   declare
--      l_blocks    wwv_flow_template_directive.t_statement_blocks;
--      l_with      wwv_flow_template_directive.t_placeholders;
--      l_output    clob;
--   begin
--      l_blocks := wwv_flow_template_directive.compile(
--          p_template          => 'Hello {if NAME/}#NAME#{else/}World{endif/}!' );
--
--      wwv_flow_template_directive.add_placeholder(
--          p_placeholders      => l_with,
--          p_name              => 'NAME',
--          p_value             => 'Scott' );
--
--      l_output := wwv_flow_template_directive.apply( 
--          p_statement_blocks  => l_blocks,
--          p_with              => l_with,
--          p_escape_mode       => wwv_flow_session_state.c_escape_mode_raw );
--
--      dbms_output.put_line( l_output );
--   end;
--
--   Output:
--
--   Hello Scott!
--==============================================================================
procedure add_placeholder(
    p_placeholders      in out nocopy   t_placeholders,
    p_name              in              varchar2,
    p_value             in              varchar2 );

--==============================================================================
-- Adds a placeholder to the placeholder collection. Placeholder collections 
-- can be passed to the APPLY calls. Placeholders can be used in the following ways:
--  - by directives
--  - As placeholder substitutions
--
-- Parameters:
-- * p_placeholders:    Placeholder collection.
-- * p_name:            This is the name of a data value. It is upper case alphanumeric 
--                      plus underscore and dollar sign. 
-- * p_values:          Collection of placeholder values
--
--
-- Example:
--
--   declare
--      l_blocks    wwv_flow_template_directive.t_statement_blocks;
--      l_with      wwv_flow_template_directive.t_placeholders;
--      l_output    clob;
--   begin
--      l_blocks := wwv_flow_template_directive.compile(
--          p_template          => '<ul>{loop FRUITS/}<li>&APEX$ITEM.</li>{endloop/}</ul>' );
--
--      wwv_flow_template_directive.add_placeholder(
--          p_placeholders      => l_with,
--          p_name              => 'FRUITS',
--          p_value             => wwv_flow_t_varchar2( 'Apple', 'Banana', 'Orange' ) );
--
--      l_output := wwv_flow_template_directive.apply( 
--          p_statement_blocks  => l_blocks,
--          p_with              => l_with,
--          p_escape_mode       => wwv_flow_session_state.c_escape_mode_html );
--
--      dbms_output.put_line( l_output );
--   end;
--
--   Output:
--
--   <ul><li>Apple</li><li>Banana</li><li>Orange</li></ul>
--==============================================================================
procedure add_placeholder(
    p_placeholders      in out nocopy   t_placeholders,
    p_name              in              varchar2,
    p_values            in              wwv_flow_t_varchar2 );

--==============================================================================
-- Applies input arguments and substitutions to a compiled template.
--
-- ARGUMENTS:
-- * p_statement_blocks:    A compiled template
-- * p_with:                Input arguments
-- * p_context:             Optional query context which can be used in a loop
--                          directive.
--
-- Example:
--
--   declare
--      l_blocks    wwv_flow_template_directive.t_statement_blocks;
--      l_with      wwv_flow_template_directive.t_placeholders;
--      l_output    clob;
--   begin
--      l_blocks := wwv_flow_template_directive.compile(
--          p_template          => 'Hello {if NAME/}#NAME#{else/}World{endif/}!' );
--
--      wwv_flow_template_directive.add_placeholder(
--          p_placeholders      => l_with,
--          p_name              => 'NAME',
--          p_value             => 'Scott' );
--
--      l_output := wwv_flow_template_directive.apply( 
--          p_statement_blocks  => l_blocks,
--          p_with              => l_with,
--          p_escape_mode       => wwv_flow_session_state.c_escape_mode_raw );
--
--      dbms_output.put_line( l_output );
--   end;
--
--   Output:
--
--   Hello Scott!
--
-- Returns:
-- The output of the template.
--==============================================================================
function apply( 
    p_statement_blocks  in              t_statement_blocks,
    p_with              in              t_placeholders,
    p_context           in out nocopy   wwv_flow_exec.t_context,
    p_escape_mode       in              wwv_flow_session_state.t_escape_mode )
    return clob;

function apply( 
    p_statement_blocks  in  t_statement_blocks,
    p_with              in  t_placeholders,
    p_escape_mode       in  wwv_flow_session_state.t_escape_mode )
    return clob;

--==============================================================================
-- Analyzes a template on placeholders, substitutions and directives
-- Compiles the results to statement blocks which are used to apply the template.
--
-- ARGUMENTS:
-- * p_template:                    The template to compile
-- * p_placeholders_only:           Whether to only substitute placeholders and ignore 
--                                  others like session state substitution 
-- * p_supports_context:            Whether looping over a query context is supported
--                         
--
-- Example:
--
--   declare
--      l_json      blob;
--      l_with      wwv_flow_template_directive.t_placeholders;
--      l_blocks    wwv_flow_template_directive.t_statement_blocks;
--      l_output    clob;
--   begin
--
--      l_json := wwv_flow_template_directive.compile_to_json(
--          p_template          => 'Hello {if NAME/}#NAME#{else/}World{endif/}!' );
--
--      wwv_flow_template_directive.add_placeholder(
--          p_placeholders      => l_with,
--          p_name              => 'NAME',
--          p_value             => 'Scott' );
--
--      l_output := wwv_flow_template_directive.apply_compiled_json( 
--          p_statement_blocks  => l_blocks,
--          p_json              => l_json,
--          p_with              => l_with,
--          p_escape_mode       => wwv_flow_session_state.c_escape_mode_raw );
--
--      dbms_output.put_line( l_output );
--   end;
--
--   Output:
--
--   Hello Scott!
--
-- Returns:
-- JSON object of compiled statement blocks
--==============================================================================
function apply_compiled_json( 
    p_statement_blocks  in out nocopy   t_statement_blocks,
    p_json              in              blob,
    p_with              in              t_placeholders,
    p_escape_mode       in              wwv_flow_session_state.t_escape_mode )
    return clob;

--==============================================================================
-- Transforms JSON to statement blocks
-- The JSON must be created using the to_json function.
--
-- ARGUMENTS:
-- * p_json:    Compiled statement blocks in JSON format
--==============================================================================
function to_statement_blocks(
    p_json in blob )
    return t_statement_blocks;

end wwv_flow_template_directive;
/
show errors

set define '^'
