set define '^' verify off
prompt ...wwv_flow_data_loading_api.sql
create or replace package wwv_flow_data_loading_api as
--------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 1999 - 2020. All Rights Reserved.
--
--    NAME
--      wwv_flow_data_loading.sql
--
--    DESCRIPTION
--      Implementation for the data loading process type
--
--
--    MODIFIED   (MM/DD/YYYY)
--    cczarski    12/07/2020 - created
-- 
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Public type definitions
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Public constant definitions
--------------------------------------------------------------------------------

type t_data_load_result is record(
    processed_rows    pls_integer,
    error_rows        pls_integer );

--==============================================================================
-- executes data loading
--
-- PARAMETERS:
-- * p_application_id        ID of the application which contains the automation
-- * p_static_id             Static ID of the data loading definition to execute
-- * p_data_to_load          file to be loaded
-- * p_xlsx_sheet_name       for XLSX files, the worksheet to extract
--
-- RETURNS:
-- loading status information, containing processed rows and error rows
--
-- EXAMPLE:
-- fetch a file (uploaded with the PX_FILEBROWSE_ITEM) from the APEX_APPLICATION_TEMP_FILES
-- table and execute the "my-load-definition" data loading definition.
--
-- declare
--     l_file blob;
-- begin
--     apex_session.create_session( 100, 1, 'ADMIN' );
--
--     select blob_content
--       into l_file
--       from apex_application_temp_files
--      where name = :PX_FILEBROWSE_ITEM;
--
--     apex_data_loading.load_data (
--         p_static_id    => 'my-load-definition',
--         p_data_to_load => l_file );
-- end;
--==============================================================================
function load_data (
    p_application_id   in number      default wwv_flow.g_flow_id,
    p_static_id        in varchar2,
    p_data_to_load     in blob,
    p_xlsx_sheet_name  in varchar2    default null )
    return t_data_load_result;

--==============================================================================
function load_data (
    p_application_id   in number      default wwv_flow.g_flow_id,
    p_static_id        in varchar2,
    p_data_to_load     in clob,
    p_xlsx_sheet_name  in varchar2    default null )
    return t_data_load_result;

--==============================================================================
-- This function returns the file profile ( determinded by the data loading definition )
-- in JSON format.
--
-- PARAMETERS:
-- * p_application_id        ID of the application which contains the automation
-- * p_static_id             Static ID of the data loading definition to execute
--
-- RETURNS:
-- Returns file profile computed from the data load definition in JSON format.
--
-- EXAMPLE:
-- Parse a XLSX file using the file profile computed from the data load defintion.
--
-- # Parse the file using the file profile computed from the data load defintion. The result can be shown as a preview to end users.
--
--   select line_number, col001, col002, col003, col004, col005, col006, col007, col008  
--     from table( 
--              apex_data_parser.parse(
--                  p_content         => {BLOB containing XLSX file},
--                  p_file_name       => 'test.xlsx',
--                  p_xlsx_sheet_name => 'sheet1.xml',
--                  p_file_profile    => apex_data_loading.get_file_profile(
--                                           p_static_id => 'my-load-definition' ) )
--
--   Output:
--
--   LINE_NUMBER COL001   COL002       COL003       COL004   COL005          COL006   COL007       COL008 
--   ----------- -------- ------------ ------------ -------- --------------- -------- ------------ ------------- 
--             1 0        First Name   Last Name    Gender   Country         Age      Date         Id       
--             2 1        Dulce        Abril        Female   United States   32       15/10/2017   1562     
--             3 2        Mara         Hashimoto    Female   Great Britain   25       16/08/2016   1582     
--             4 3        Philip       Gent         Male     France          36       21/05/2015   2587     
--             5 4        Kathleen     Hanner       Female   United States   25       15/10/2017   3549     
--             6 5        Nereida      Magwood      Female   United States   58       16/08/2016   2468     
--             7 6        Gaston       Brumm        Male     United States   24       21/05/2015   2554     
--             8 7        Etta         Hurn         Female   Great Britain   56       15/10/2017   3598     
--             9 8        Earlean      Melgar       Female   United States   27       16/08/2016   2456     
--            10 9        Vincenza     Weiland      Female   United States   40       21/05/2015   6548     
--             : :        :            :            :        :               :        :            :
--
-- # Get the the file profile computed from the data load defintion in JSON format.
--
--   select apex_data_loading.get_file_profile( p_static_id => 'my-load-definition' ) from sys.dual
--
--   Output:
--
--   {
--       "file-type" : 1,
--       "csv-delimiter" : "",
--       "xslx-worksheet" : "sheet1.xml",
--       "headings-in-first-row" : true,
--       "file-encoding" : "AL32UTF8",
--       "single-row" : false,
--       "parsed-rows" : 2378,
--       "columns" : [
--          {
--              "format-mask" : "",
--              "name" : "C0",
--              "data-type" : 2,
--              "selector" : ""
--          },
--          {
--              "name" : "FIRST_NAME",
--              "data-type" : 1,
--              "selector" : "",
--              "format-mask" : ""
--          },
--          :
--          {
--              "format-mask" : "DD\"/\"MM\"/\"YYYY",
--              "selector" : "",
--              "data-type" : 3,
--              "name" : "DATE_"
--          },
--          {
--              "name" : "ID",
--              "data-type" : 2,
--              "selector" : "",
--              "format-mask" : ""
--          }
--       ],
--       "row-selector" : ""
--   }
--
--==============================================================================
function get_file_profile (
    p_application_id   in number      default wwv_flow.g_flow_id,
    p_static_id        in varchar2 )
    return clob;
end wwv_flow_data_loading_api;
/
show err

set define '^'

