set define off verify off
prompt ...wwv_flow_credential_api.sql
create or replace package wwv_flow_credential_api authid current_user
------------------------------------------------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 1999 - 2022. All Rights Reserved.
--
--  You can use the APEX_CREDENTIAL package to change stored credentials either persistently
--  or for the current Oracle APEX session only.
--
------------------------------------------------------------------------------------------------------------------------
is

subtype t_token_type is pls_integer range 1..3;

c_token_access  constant t_token_type := 1;
c_token_refresh constant t_token_type := 2;
c_token_id      constant t_token_type := 3;

subtype t_credential_type is wwv_credentials.authentication_type%type;

c_type_basic             constant t_credential_type := 'BASIC';
c_type_oauth_client_cred constant t_credential_type := 'OAUTH2_CLIENT_CREDENTIALS';
c_type_jwt               constant t_credential_type := 'JWT';
c_type_oci               constant t_credential_type := 'OCI';
c_type_http_header       constant t_credential_type := 'HTTP_HEADER';
c_type_http_query_string constant t_credential_type := 'HTTP_QUERY_STRING';

type t_credential is record (
    id             wwv_credentials.id%type,
    auth_type      t_credential_type,
    scope          wwv_credentials.scope%type,
    client_id      wwv_credentials.client_id%type,
    client_secret  wwv_credentials.client_secret%type,
    namespace      wwv_credentials.namespace%type,
    fingerprint    wwv_credentials.fingerprint%type,
    valid_for_urls wwv_credentials.valid_for_urls%type);

type t_token is record (
    value_type  wwv_credential_instances.value_type%type,
    value       varchar2(32767),
    expiry_date date );

type t_tokens is table of t_token index by binary_integer;

--==============================================================================
-- This procedure sets username and password for a given credential for the current session.
-- Typically used for `BASIC` authentication when the credentials to be used are to be
-- provided by the end user.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_username:             The credential username.
-- * p_password:             The credential password.
--
-- Example:
--
-- The following example sets credential `Login`.
--
--   begin
--       apex_credential.set_session_credentials (
--           p_credential_static_id => 'Login',
--           p_username             => 'scott',
--           p_password             => 'tiger );
--   end;
--==============================================================================
procedure set_session_credentials(
    p_credential_static_id  in varchar2,
    p_username              in varchar2,
    p_password              in varchar2 );

--==============================================================================
-- This procedure sets `Client ID` and `Client Secret` for a given credential
-- for the current session. Typically used for the `OAuth2` Client Credentials flow.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_client_id:            The OAuth2 Client ID.
-- * p_client_secret:        The OAuth2 Client Secret.
-- * p_namespace:            Optional namespace (used for OCI)
-- * p_fingerprint:          Optional fingerprint (used for OCI)
--
-- Example:
--
-- The following example sets credential `OAuth Login`.
--
--   begin
--       apex_credential.set_session_credentials (
--           p_credential_static_id => 'OAuth Login',
--           p_client_id            => 'dnkjq237o8832ndj98098-..',
--           p_client_secret        => '1278672tjksaGSDA789312..' );
--   end;
--==============================================================================
procedure set_session_credentials(
    p_credential_static_id  in varchar2,
    p_client_id             in varchar2,
    p_client_secret         in varchar2,
    p_namespace             in varchar2 default null,
    p_fingerprint           in varchar2 default null );

--==============================================================================
-- This procedure is a generic overload to set session credentials.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_key:                  Credential key (name of the HTTP Header or Query String Parameter).
-- * p_value:                Credential secret value.
--
-- Example:
--
-- The following example sets the credential `API Key`.
--
--   begin
--       apex_credential.set_session_credentials (
--           p_credential_static_id => 'my_API_key',
--           p_key                  => 'api_key',
--           p_value                => 'lsjkgjw4908902ru9fj879q367891hdaw' );
--   end;
--==============================================================================
procedure set_session_credentials(
    p_credential_static_id  in varchar2,
    p_key                   in varchar2,
    p_value                 in varchar2 );

--==============================================================================
-- This procedure clears all acquired tokens for a given credential.
-- Applies only to `OAuth2` based flows, where the `Client ID` and `Client Secret`
-- are used to obtain an Access Token with a certain expiry time. This call clears
-- the obtained tokens.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
--
-- Example:
--
-- The following example clears all obtained tokens for the credential `OAuth Login`.
--
--   begin
--       apex_credential.clear_tokens(
--           p_credential_static_id => 'OAuth Login' );
--   end;
--==============================================================================
procedure clear_tokens( p_credential_static_id in varchar2);

--==============================================================================
-- This procedure uses an autonomous transaction in order to store the token in
-- the database table.
--
-- Stores a token into a credential store which is obtained with manual or custom
-- PL/SQL code. The credential store saves this token in encrypted form for subsequent
-- use by Oracle APEX components. The token is stored for the lifetime of the
-- Oracle APEX session. Other sessions cannot use this token. When tokens
-- are obtained with custom PL/SQL code, Client ID, and Client Secret are not stored
-- in that credential store: it contains the tokens set by this procedure only.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_token_type:           The token type: APEX_CREDENTIAL.C_TOKEN_ACCESS,
--                           APEX_CREDENTIAL.C_TOKEN_REFRESH or APEX_CREDENTIAL.C_TOKEN_ID.
-- * p_token_value:          The value of the token.
-- * p_token_expires:        The expiry date of the token
--
-- Example:
--
-- The following example stores `OAuth2` access token with value
-- `sdakjjkhw7632178jh12hs876e38..` and expiry date of `2017-10-31`
-- into credential `OAuth Login`.
--
--   begin
--       apex_credential.set_session_token (
--           p_credential_static_id => 'OAuth Login',
--           p_token_type           => apex_credential.C_TOKEN_ACCESS,
--           p_token_value          => 'sdakjjkhw7632178jh12hs876e38..',
--           p_token_expires        => to_date('2017-10-31', 'YYYY-MM-DD') );
--   end;
--==============================================================================
procedure set_session_token(
    p_credential_static_id  in varchar2,
    p_token_type            in t_token_type,
    p_token_value           in varchar2,
    p_token_expires         in date );

--==============================================================================
-- This procedure uses an autonomous transaction in order to store the token in
-- the database table.
--
-- SET_PERSISTENT_TOKEN stores a token into a credential store which is obtained
-- with manual or custom PL/SQL code. The credential store saves this token in
-- encrypted form for subsequent use by Oracle APEX components.
-- The token is stored for the lifetime of the APEX session.
-- Other sessions cannot use this token. When tokens are obtained with custom
-- PL/SQL code, Client ID, and Client Secret are not stored in that
-- credential store: it contains the tokens set by this procedure only.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_token_type:           The token type: APEX_CREDENTIAL.C_TOKEN_ACCESS,
--                           APEX_CREDENTIAL.C_TOKEN_REFRESH or APEX_CREDENTIAL.C_TOKEN_ID.
-- * p_token_value:          The value of the token.
-- * p_token_expiry:         The expiry date of the token
--
-- Example:
--
-- The following example stores OAuth2 access token with value
-- `sdakjjkhw7632178jh12hs876e38..` and expiry date of `2017-10-31` into
-- credential `OAuth Login`.
--
--   begin
--       apex_credential.set_persistent_token (
--           p_credential_static_id => 'OAuth Login',
--           p_token_type           => apex_credential.C_TOKEN_ACCESS,
--           p_token_value          => 'sdakjjkhw7632178jh12hs876e38..',
--           p_token_expiry         => to_date('2017-10-31', 'YYYY-MM-DD') );
--   end;
--==============================================================================
procedure set_persistent_token(
    p_credential_static_id in varchar2,
    p_token_type           in t_token_type,
    p_token_value          in varchar2,
    p_token_expires        in date );

--==============================================================================
-- This procedure sets `Client ID` and `Client Secret` for a given credential.
-- Typically used for the `OAuth2` Client Credentials flow. The new credentials
-- are stored persistently and are valid for all current and future sessions.
-- Stored access, refresh or ID tokens for that credential, will be deleted.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_client_id:            The OAuth2 Client ID.
-- * p_client_secret:        The OAuth2 Client Secret
-- * p_namespace:            Optional namespace (for OCI)
-- * p_fingerprint:          Optional fingerprint (for OCI)
--
-- Example:
--
-- The following example sets credential `OAuth Login`.
--
--   begin
--       apex_credential.set_persistent_credentials (
--           p_credential_static_id  => 'OAuth Login',
--           p_client_id             => 'dnkjq237o8832ndj98098-..',
--           p_client_secret         => '1278672tjksaGSDA789312..' );
--   end;
--==============================================================================
procedure set_persistent_credentials(
    p_credential_static_id  in varchar2,
    p_client_id             in varchar2,
    p_client_secret         in varchar2,
    p_namespace             in varchar2 default null,
    p_fingerprint           in varchar2 default null );

--==============================================================================
-- This procedure sets a list of URLs that can be used for this credential.
--
-- # Usage Notes
--
-- If an HTTP request target URL for these credentials matches one of these URLs,
-- credential usage is allowed. If not, an error is raised.
--
-- URLs are matched on a starts-with basis. For example, if P_ALLOWED_URLS is passed in as
-- `apex_t_varchar2('https://www.oracle.com','https://apex.oracle.com/ords/')`,
-- then the credential can be used for HTTP requests to:
--
-- - https://www.oracle.com/
-- - https://www.oracle.com/myrest/service
-- - https://apex.oracle.com/ords/secret/workspace
--
-- However, the credential is not allowed for requests to:
--
-- - https://web.oracle.com
-- - https://apex.oracle.com/apex/workspace
-- - http://www.oracle.com/
--
-- The Client Secret needs to be provided again if the allowed URLs change.
-- If the client secret is provided as NULL, it will be cleared.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_allowed_urls:         List of URLs (as APEX_T_VARCHAR2) that these credentials can access.
-- * p_client_secret:        Client Secret. If allowed URLs are changed,
--                           this must be provided again.
--
-- Example:
--
-- This example sets allowed URLs for the credential `OAuth Login`.
--
--   begin
--       apex_credential.set_allowed_urls (
--           p_credential_static_id => 'OAuth Login',
--           p_allowed_urls         => apex_t_varchar2(
--                                        'https://tokenserver.mycompany.com/oauth2/token',
--                                        'https://www.oracle.com' ),
--           p_client_secret        => '1278672tjksaGSDA789312..' );
--   end;
--==============================================================================
procedure set_allowed_urls(
    p_credential_static_id  in varchar2,
    p_allowed_urls          in wwv_flow_t_varchar2,
    p_client_secret         in varchar2 );

--==============================================================================
-- This procedure sets username and password for a given credential.
-- This is typically be used by a security person after application import,
-- and allows to separate responsibilities between a person importing the application
-- and another person storing the credentials.
--
-- Parameters:
-- * p_credential_static_id: The credential static ID.
-- * p_username:             The credential username.
-- * p_password:             The credential password.
--
-- Example:
--
-- The following example sets credential `Login`.
--
--   begin
--       apex_credential.set_persistent_credentials (
--           p_credential_static_id => 'Login',
--           p_username             => 'scott',
--           p_password             => 'tiger );
--   end;
--==============================================================================
procedure set_persistent_credentials(
    p_credential_static_id  in varchar2,
    p_username              in varchar2,
    p_password              in varchar2 );

--==============================================================================
-- Internal:
-- Generic overload to set persistent credentials
--
-- Parameter:
-- * p_credential_static_id: The credential static ID
-- * p_key:                  credential key (e.g. HTTP Header or Cookie name)
-- * p_value:                credential value
--
-- Example:
--
-- Set credential `API Key`.
--
--   begin
--       apex_credential.set_persistent_credentials (
--           p_credential_static_id => 'my_API_key',
--           p_key                  => 'api_key',
--           p_value                => 'lsjkgjw4908902ru9fj879q367891hdaw' );
--   end;
--==============================================================================
procedure set_persistent_credentials(
    p_credential_static_id  in varchar2,
    p_key                   in varchar2,
    p_value                 in varchar2 );

--==============================================================================
-- Create a credential definition
--
-- PARAMETERS
-- * p_credential_name:         The credential name.
-- * p_credential_static_id:    The credential static ID.
-- * p_authentication_type:     The authentication type. Supported types: apex_credential.C_TYPE_BASIC, apex_credential.C_TYPE_OAUTH_CLIENT_CRED,
--                                                                        apex_credential.C_TYPE_JWT, apex_credential.C_TYPE_OCI,
--                                                                        apex_credential.C_TYPE_HTTP_HEADER, apex_credential.C_TYPE_HTTP_QUERY_STRING
-- * p_scope:                   Optional OAuth 2.0 scope.
-- * p_allowed_urls:            Optional List of URLs (as APEX_T_VARCHAR2) that these credentials can access.
-- * p_prompt_on_install:       Optional Choose whether prompts for this credential should be displayed when the application is being imported on another Oracle APEX instance.
-- * p_credential_comment:      Optional credential comment.
--
-- EXAMPLE
--   Creates a credential definition "OAuth Login"
--
--   begin
--       -- first set the workspace
--       apex_util.set_workspace(p_workspace => 'MY_WORKSPACE');
--
--       apex_credential.create_credential (
--           p_credential_name       => 'OAuth Login',
--           p_credential_static_id  => 'OAUTH_LOGIN',
--           p_authentication_type   => apex_credential.C_TYPE_OAUTH_CLIENT_CRED,
--           p_scope                 => 'email',
--           p_allowed_urls          => apex_t_varchar2( 'https://tokenserver.mycompany.com/oauth2/token', 'https://www.oracle.com' ),
--           p_prompt_on_install     => false,
--           p_credential_comment    => 'Credential for OAuth Login' );
--
--      -- should be followed by set_persistent_credentials
--      apex_credential.set_persistent_credentials (
--          p_credential_static_id  => 'OAUTH_LOGIN',
--          p_client_id             => 'dnkjq237o8832ndj98098-..',
--          p_client_secret         => '1278672tjksaGSDA789312..' );
--   end;
--==============================================================================
procedure create_credential (
    p_credential_name         in varchar2,
    p_credential_static_id    in varchar2,
    p_authentication_type     in varchar2,
    p_scope                   in varchar2 default null,
    p_allowed_urls            in wwv_flow_t_varchar2 default null,
    p_prompt_on_install       in boolean  default false,
    p_credential_comment      in varchar2 default null );

--==============================================================================
-- Drop a credential definition
--
-- PARAMETERS
-- * p_credential_static_id:    The credential static ID.
--
-- EXAMPLE
--   Drops a credential definition "OAuth Login"
--
--   begin
--       -- first set the workspace
--       apex_util.set_workspace(p_workspace => 'MY_WORKSPACE');
--
--       apex_credential.drop_credential (
--           p_credential_static_id => 'OAUTH_LOGIN );
--   end;
--==============================================================================
procedure drop_credential (
    p_credential_static_id    in varchar2 );

end wwv_flow_credential_api;
/
sho err

set define '^'
