set define '^' verify off
prompt ...wwv_flow_approval_api
create or replace package wwv_flow_approval_api authid current_user as
--------------------------------------------------------------------------------
--
-- Copyright (c) Oracle Corporation 1999 - 2022. All Rights Reserved.
--
--
-- The APEX_APPROVAL package provides API's for the management of approvals and 
-- Human Tasks. This package includes functionality to create new Human Tasks 
-- for a user to approve as well as operations dealing with the lifecycle
-- management and state handling of Human Tasks. This package is part of the
-- Oracle APEX Workflow functionality.
--------------------------------------------------------------------------------


--==============================================================================
-- Global types 
--==============================================================================
subtype t_task_participant_type is varchar2(15);

subtype t_task_identity_type    is varchar2(32);

subtype t_task_outcome          is varchar2(32);

subtype t_task_state            is varchar2(15);

subtype t_task_operation        is varchar2(30);

subtype t_task_list_context     is varchar2(15);

--==============================================================================
-- Data Types.
--==============================================================================

--------------------------------------------------------------------------------
-- Task Parameter (Value)
--
-- Attributes:
-- * static_id :    The static id of the parameter. This ID must match the 
--                  the static id of the corresponding parameter in the task
--                  definition.
-- * string_value:  The value of the parameter as a string.
--------------------------------------------------------------------------------
type t_task_parameter is record (
    static_id                varchar2(255),
    string_value             varchar2(32767)
);

--------------------------------------------------------------------------------
-- Collection of Task Parameter Values
--------------------------------------------------------------------------------
type t_task_parameters is table of t_task_parameter index by pls_integer;

--------------------------------------------------------------------------------
-- Collection of Task Participant Types
--------------------------------------------------------------------------------
type t_task_participant_types is table of t_task_participant_type
    index by pls_integer;

--==============================================================================
-- Constants.
--==============================================================================

--------------------------------------------------------------------------------
-- Task List Context Types
--------------------------------------------------------------------------------
c_context_my_tasks           constant t_task_list_context := 'MY_TASKS';
c_context_admin_tasks        constant t_task_list_context := 'ADMIN_TASKS';
c_context_initiated_by_me    constant t_task_list_context := 'INITIATED_BY_ME';
c_context_single_task        constant t_task_list_context := 'SINGLE_TASK';

--------------------------------------------------------------------------------
-- Task Definition Participant Types
--------------------------------------------------------------------------------
c_task_potential_owner       constant t_task_participant_type := 'POTENTIAL_OWNER';
c_task_business_admin        constant t_task_participant_type := 'BUSINESS_ADMIN';

--------------------------------------------------------------------------------
-- Task Definition Participant Identity Types
--------------------------------------------------------------------------------
c_task_identity_type_user    constant t_task_identity_type := 'USER';

--------------------------------------------------------------------------------
-- Task (Instance) Priority constants
--------------------------------------------------------------------------------
c_task_priority_lowest       constant integer := 5;
c_task_priority_low          constant integer := 4;
c_task_priority_medium       constant integer := 3;
c_task_priority_high         constant integer := 2;
c_task_priority_urgent       constant integer := 1; 

--------------------------------------------------------------------------------
-- Task (Instance) states
--------------------------------------------------------------------------------
c_task_state_unassigned      constant t_task_state := 'UNASSIGNED';
c_task_state_assigned        constant t_task_state := 'ASSIGNED';
c_task_state_completed       constant t_task_state := 'COMPLETED';
c_task_state_cancelled       constant t_task_state := 'CANCELLED';
c_task_state_failed          constant t_task_state := 'FAILED';
c_task_state_errored         constant t_task_state := 'ERRORED';

--------------------------------------------------------------------------------
-- Task (Instance) outcomes
--------------------------------------------------------------------------------
c_task_outcome_approved      constant t_task_outcome := 'APPROVED';
c_task_outcome_rejected      constant t_task_outcome := 'REJECTED';

--------------------------------------------------------------------------------
-- Task (Instance) operations
--------------------------------------------------------------------------------
c_task_op_approve            constant t_task_operation := 'APPROVE_TASK';
c_task_op_reject             constant t_task_operation := 'REJECT_TASK';
c_task_op_complete           constant t_task_operation := 'COMPLETE_TASK';
c_task_op_claim              constant t_task_operation := 'CLAIM_TASK';
c_task_op_delegate           constant t_task_operation := 'DELEGATE_TASK';
c_task_op_release            constant t_task_operation := 'RELEASE_TASK';
c_task_op_cancel             constant t_task_operation := 'CANCEL_TASK';
c_task_op_set_priority       constant t_task_operation := 'SET_TASK_PRIORITY';
c_task_op_add_comment        constant t_task_operation := 'ADD_TASK_COMMENT';
c_task_op_add_owner          constant t_task_operation := 'ADD_TASK_POTENTIAL_OWNER';

--------------------------------------------------------------------------------
-- Task (Instance) date formats
--------------------------------------------------------------------------------
c_canonical_date_format      constant varchar2(16)     := 'YYYYMMDDHH24MISS';
--------------------------------------------------------------------------------
-- Task parameters default
--------------------------------------------------------------------------------
c_empty_task_parameters t_task_parameters;

--==============================================================================
-- Create a new task. A new Task (Instance) is created. Depending on the task
-- definition participant setting, the Task is set to state Unassigned or Assigned.
--
-- Since: 22.1
--
-- Parameters:
-- * p_application_id        The application ID that creates the Task
-- * p_task_def_static_id    The Task Definition static ID.
-- * p_subject               The subject (expression of the Task
-- * p_parameters            The task parameters
-- * p_priority              A (optional) task priority, default is 5
-- * p_initiator:            A (optional) initiator information for the task
-- * p_detail_pk:            A (optional) primary key value for the task details
--
-- Return:
-- Returns the ID of the newly created task.
--
-- Example:
-- Below example creates a requisition item in the system of record in the database and then
-- creates a new Human Task to get the requisition item approved by a user.
-- ~~~
-- declare
--    l_req_id number;
--    l_req_item varchar2(100) := 'Some requisition item requiring approval';
--    l_req_amount number := 2499.42;
--    l_task_id number;
-- begin
--     insert into requisitions(created_by, creator_emailid, item, item_amount, item_category)
--     values (:emp_uid, :emp_email, l_req_item, l_req_amount, 'Equipment')
--     returning id into l_req_id;
--     commit;
--
--     l_task_id := apex_approval.create_task(
--         p_application_id => 110,
--         p_task_def_static_id => 'REQAPPROVALS',
--         p_subject => 'Requisition ' || l_req_id || ': ' || l_req_item || ' for ' || l_req_amount, 
--         p_initiator => :emp_uid,
--         p_parameters => apex_approval.t_task_parameters(
--             1 => apex_approval.t_task_parameter(static_id => 'REQ_DATE', string_value => sysdate),
--             3 => apex_approval.t_task_parameter(static_id => 'REQ_AMOUNT', string_value => l_req_amount),
--             4 => apex_approval.t_task_parameter(static_id => 'REQ_ITEM', string_value => l_req_item),
--             5 => apex_approval.t_task_parameter(static_id => 'REQ_ID', string_value => l_req_id)
--         ),
--         p_detail_pk => l_req_id
--     );
-- end;
-- ~~~
--==============================================================================
function create_task(
    p_application_id         in number            default wwv_flow.g_flow_id,
    p_task_def_static_id     in varchar2,
    p_subject                in varchar2          default null,
    p_parameters             in t_task_parameters default c_empty_task_parameters,
    p_priority               in integer           default null,
    p_initiator              in varchar2          default null,
    p_detail_pk              in varchar2          default null)
return number;
   
--==============================================================================
-- Add a comment to a task. Any Potential Owner or Business Administrator of
-- a Task can add comments to a Task. Comments are useful as additional 
-- information regarding a Task. For example, a manager might add his notes
-- to a Task he's working on before delegating the Task to his manager.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id: The Task Id.
-- * p_text:    The comment text
--
-- Example:
-- ~~~
-- begin
--     add_task_comment(
--         p_task_id => 1234,
--         p_text    => 'Please review and approve'
--     );
-- end;
-- ~~~
--
--==============================================================================
procedure add_task_comment(
    p_task_id                in number,
    p_text                   in varchar2);

--==============================================================================
-- Add a potential owner to a task.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:         The Task Id.
-- * p_potential_owner: The potential owner
-- * p_identity_type:   The identity type of the potential owner, default is
--                      'USER'
--
-- Example:
-- Add user STIGER as potential owner for Task Id 1234
-- ~~~
-- begin
--     apex_approval.add_task_potential_owner(
--         p_task_id         => 1234,
--         p_potential_owner => 'STIGER'
--     );
-- end;
-- ~~~
--==============================================================================
procedure add_task_potential_owner(
    p_task_id                in number,
    p_potential_owner        in varchar2,
    p_identity_type          in t_task_identity_type default c_task_identity_type_user);
 
--==============================================================================
-- Claim responsibility for a task. A task can be claimed by potential owners
-- of the Task. A Task must be in Unassigned state to claim it. Once the task is
-- claimed by a user, the Task transitions to Assigned state and the actual
-- owner of the task is set to the user who claimed the task.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id: The Task Id.
--
-- # State Handling
-- Pre-State: UNASSIGNED
-- Post-State: ASSIGNED
--
-- Example:
-- ~~~
-- begin
--     apex_approval.claim_task(
--         p_task_id => 1234
--     );
-- end;
-- ~~~
--==============================================================================
procedure claim_task(
    p_task_id                in number);

--==============================================================================
-- Complete a task with an outcome. Tasks in Assigned state might be completed
-- with an outcome. This operation transitions the Task from Assigned state to
-- Completed state and sets the outcome of the task. Once a Task is in Completed
-- state, it is subject for purging and archival.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:      The Task Id.
-- * p_outcome:      The outcome of the Task.
-- * p_autoclaim:    If Task is in state UNASSIGNED then claim the task implicitly.
--
-- # State Handling
-- Pre-State:  ASSIGNED|UNASSIGNED (p_autoclaim=true)
-- Post-State: COMPLETED
--
-- Example:
-- ~~~
-- begin
--     apex_approval.complete_task(
--         p_task_id => 1234,
--         p_outcome => apex_approval.c_task_outcome_approved
--     );
-- end;
-- ~~~
--==============================================================================
procedure complete_task(
    p_task_id                in number,
    p_outcome                in t_task_outcome,
    p_autoclaim              in boolean default false );
 
--==============================================================================
-- Approve a Task.
-- Move the state of the Task to Completed and set the outcome of the Task
-- to Approved. This is a convenience procedure and equivalent to calling 
-- complete_task with outcome apex_approval.c_task_outcome_approved
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id: The Task Id.
-- * p_autoclaim:    If Task is in state UNASSIGNED then claim the task implicitly.
--
-- # State Handling
-- Pre-State:  ASSIGNED|UNASSIGNED (p_autoclaim=true)
-- Post-State: COMPLETED
--
-- Example:
-- ~~~
-- begin
--     apex_approval.approve_task(
--         p_task_id => 1234
--     );
-- end;
-- ~~~
--==============================================================================
procedure approve_task(
    p_task_id                in number,
    p_autoclaim              in boolean default false );
 
--==============================================================================
-- Reject a Task.
-- Move the state of the Task to Completed and set the outcome of the Task
-- to Rejected. This is a convenience procedure and equivalent to calling 
-- complete_task with outcome apex_approval.c_task_outcome_rejected 
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id: The Task Id.
-- * p_autoclaim:    If Task is in state UNASSIGNED then claim the task implicitly.
--
-- # State Handling
-- Pre-State:  ASSIGNED|UNASSIGNED (p_autoclaim=true)
-- Post-State: COMPLETED
--
-- Example:
-- ~~~
-- begin
--     apex_approval.reject_task(
--         p_task_id => 1234
--     );
-- end;
-- ~~~
--==============================================================================
procedure reject_task(
    p_task_id                in number,
    p_autoclaim              in boolean default false );

--==============================================================================
-- Assign the task to one Potential Owner and set the task state to Assigned. 
-- This operation can be performed by Potential Owners of a Task in normal 
-- operation. Business Admins are able to delegate tasks for tasks in 
-- Failed state.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:            The Task Id.
-- * p_to_user:    A (user) participant
--
-- # State Handling
-- Pre-State: UNASSIGNED, ASSIGNED
-- Post-State: ASSIGNED
--
-- Example:
-- ~~~
-- begin
--     apex_approval.delegate_task(
--         p_task_id            => 1234,
--         p_to_user            => 'STIGER'
--     );
-- end;
-- ~~~
--==============================================================================
procedure delegate_task(
    p_task_id                in number,
    p_to_user                in varchar2);
       
--==============================================================================
-- Get the value of a Task parameter. This function can be used in SQL or PL/SQL
-- to get the value of a Task parameter for a given task.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:          The Task Id.
-- * p_param_static_id:  The static id of the parameter
-- * p_ignore_not_found: If set to false (default) and no data is found, a no_data_found
--                       exception will be raised. If set to true and no data is found, 
--                       null will be returned.
--
-- Returns:
-- The task parameter value for the given static id or null
-- 
-- Exception:
-- no_data_found        In the case where p_ignore_not_found is set to false and no data
--                      is being found (parameter of given name doesn't exist for example)
--
-- Example:
-- ~~~
-- declare
--     l_req_item varchar2(100);
-- begin
--     l_req_item := apex_approval.get_task_parameter_value(
--         p_task_id         => 1234,
--         p_param_static_id => 'REQ_ITEM'
--     );
--     dbms_output.put_line('Parameter REQ_ITEM of task 1234 has value ' || l_req_item);
-- end;
-- ~~~
--==============================================================================
function get_task_parameter_value(
    p_task_id                in number,
    p_param_static_id        in varchar2,
    p_ignore_not_found       in boolean default false)
return varchar2;
  
--==============================================================================
-- Cancel the task, i.e. set the task to state 'CANCELLED'. Cancelling a task is
-- useful, when an approval is no longer required. For example, consider a travel
-- approval for a business trip. So now imagine the person requesting the approval
-- is not feeling well and can't make the trip. In such cases, a Task might be
-- canceled.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id: The Task Id.
--
-- # State Handling
-- Pre-State: <Any>
-- Post-State: CANCELLED
--
-- Example:
-- ~~~
-- begin
--     apex_approval.cancel_task(
--         p_task_id => 1234
--     );
-- end;
-- ~~~
--=============================================================================
procedure cancel_task(
    p_task_id                in number);
 
--==============================================================================
-- Release the task, i.e. set the task back to state Unassigned.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id: The Task Id.
--
-- # State Handling
-- Pre-State: ASSIGNED
-- Post-State: UNASSIGNED
--
-- Example:
-- ~~~
-- begin
--     apex_approval.release_task(
--         p_task_id            => 1234
--     );
-- end;
-- ~~~
--=============================================================================
procedure release_task(
    p_task_id                in number);
 
--==============================================================================
-- Set the priority of a task.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:  The Task Id.
-- * p_priority: The task priority (between 0..10)
--
-- Example:
-- ~~~
-- begin
--     apex_approval.set_task_priority(
--         p_task_id  => 1234,
--         p_priority => apex_approval.c_task_priority_highest
--     );
-- end;
-- ~~~
--==============================================================================
procedure set_task_priority(
    p_task_id                in number,
    p_priority               in integer);
  

--==============================================================================
-- Checks whether the given user is of a certain participant type for a Task.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:          The Task Id.
-- * p_participant_type: The participant type. Default is POTENTIAL_OWNER
-- * p_user:             The user to check for. Default is logged in user
--
-- Returns:
-- true, if the user given by p_user is a participant of given participant
-- type for a given task, false otherwise
--
-- Example:
-- ~~~
-- declare
--     l_is_potential_owner boolean;
-- begin
--     l_is_potential_owner := apex_approval.is_of_aprticipant_type(
--         p_task_id          => 1234,
--         p_participant_type => apex_approval.c_task_potential_owner,
--         p_user             => 'STIGER'
--     );
--     if l_is_potential_owner then
--         dbms_output.put_line('STIGER is a potential owner for task 1234');
--     end if;
-- end;
-- ~~~
--==============================================================================
function is_of_participant_type(
    p_task_id                in number,
    p_participant_type       in t_task_participant_type
                                default c_task_potential_owner,
    p_user                   in varchar2
                                default wwv_flow_security.g_user)
return boolean;

--==============================================================================
-- Checks whether the given user is allowed to perform a certain operation on
-- a Task.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:           The Task Id.
-- * p_operation:         The operation to check (see constants c_task_op_<XXX>)
-- * p_user:              The user to check for. Default is logged in user
-- * p_new_participant:   (Optional) The new assignee in case of Delegate operation
--
-- Returns:
-- true, if the user given by p_user is allowed to perform the operation 
-- given by p_operation, false otherwise
--
-- Example:
-- ~~~
-- declare
--     l_is_allowed boolean;
-- begin
--     l_is_allowed := apex_approval.is_allowed(
--         p_task_id         => 1234,
--         p_operation       => apex_approval.c_task_op_delegate
--         p_user            => 'STIGER',
--         p_new_participant => 'SMOON'
--     );
--     if l_is_allowed then
--         dbms_output.put_line('STIGER is a allowed to delegate the task to SMOON for task 1234');
--     end if;
-- end;
-- ~~~
--==============================================================================
function is_allowed(
    p_task_id                in number,
    p_operation              in wwv_flow_approval_api.t_task_operation,
    p_user                   in varchar2 default wwv_flow_security.g_user,
    p_new_participant        in varchar2 default null)
return boolean;

--==============================================================================
-- Checks whether the given user is a business administrator for at least one 
-- task definition.
--
-- Since: 22.1
--
-- Parameters:
-- * p_user:              The user to check for. Default is logged in user
-- * p_application_id:    The application to check for. Default behavior 
--                        checks against all applications in the workspace.
--
-- Return:
-- true, if the user given by p_user is at least in on task definition configured
-- as participant type BUSINESS_ADMIN, false otherwise
--
-- Example:
-- ~~~
-- declare
--     l_is_business_admin boolean;
-- begin
--     l_is_business_admin := apex_approval.is_business_admin(
--         p_user => 'STIGER'
--     );
--     if l_is_business_admin then
--         dbms_output.put_line('STIGER is a Business Administrator');
--     end if;
-- end;
-- ~~~
--==============================================================================
function is_business_admin(
    p_user           in varchar2 default wwv_flow_security.g_user,
    p_application_id in number   default null)
return boolean;

--==============================================================================
-- Get the tasks of a user depending on the given context. Context can be
-- MY_TASKS, ADMIN_TASKS, INITIATED_BY_ME or SINGLE_TASK.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_context:              The list context. Default is MY_TASKS.
-- * p_user:                 The user to check for. Default is logged in user. 
--                           Needs p_context set to MY_TASKS, ADMIN_TASKS or INITIATED_BY_ME.
-- * p_task_id:              Filter for a task ID instead of a user. Default is null.
--                           Needs p_context set to SINGLE_TASK.
-- * p_application_id:       Filter for an application. Default is null (all applications)
--
-- Return:
-- A table of tasks (type apex_t_approval_tasks)
--
-- Example:
-- ~~~
-- select *
--   from table ( apex_approval.get_tasks ( p_context => 'MY_TASKS' ) )
-- ~~~
--==============================================================================
function get_tasks (
    p_context        in varchar2 default wwv_flow_approval_api.c_context_my_tasks,
    p_user           in varchar2 default wwv_flow_security.g_user,
    p_task_id        in number   default null,
    p_application_id in number   default null )
return wwv_flow_t_approval_tasks pipelined;

--==============================================================================
-- Get the approval log for a task.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:   The task ID.
--
-- Return:
-- A table of approval log entries (type apex_t_approval_log_table)
--
-- Example:
-- ~~~
-- select *
--   from table ( apex_approval.get_task_history ( p_task_id => 1234 ) )
-- ~~~
--==============================================================================
function get_task_history (
    p_task_id        in number )
return wwv_flow_t_approval_log_table pipelined;

--==============================================================================
-- Get the potential new owners of a task. 
-- The actual owner is excluded from the list.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:  The task ID.
--
-- Return:
-- A table of apex_t_temp_lov_data
--
-- Example:
-- ~~~
-- select disp,
--        val
--   from table ( apex_approval.get_task_delegates ( p_task_id => 1234 ) )
-- ~~~
--==============================================================================
function get_task_delegates (
    p_task_id in number )
return wwv_flow_t_temp_lov_data pipelined;

--==============================================================================
-- Get the potential new priorities of a task. 
-- The actual priority is excluded from the list.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:  The task ID.
--
-- Return:
-- A table of apex_t_temp_lov_data
--
-- Example:
-- ~~~
-- select disp,
--        val
--   from table ( apex_approval.get_task_priorities ( p_task_id => 1234 ) )
-- ~~~
--==============================================================================
function get_task_priorities (
    p_task_id in number )
return wwv_flow_t_temp_lov_data pipelined;

--==============================================================================
-- Get the list of value data for the task attribute priority.
--
-- Since: 22.1
--
-- Return:
-- A table of apex_t_temp_lov_data
--
-- Example:
-- ~~~
-- select disp,
--        val
--   from table ( apex_approval.get_lov_priority )
-- ~~~
--==============================================================================
function get_lov_priority
return wwv_flow_t_temp_lov_data pipelined;

--==============================================================================
-- Get the list of value data for the task attribute state.
--
-- Since: 22.1
--
-- Return:
-- A table of apex_t_temp_lov_data
--
-- Example:
-- ~~~
-- select disp,
--        val
--   from table ( apex_approval.get_lov_state )
-- ~~~
--==============================================================================
function get_lov_state
return wwv_flow_t_temp_lov_data pipelined;

end wwv_flow_approval_api;
/
show err

set define '^'
