set define '^' verify off
prompt ...wwv_flow_approval
create or replace package wwv_flow_approval as
--------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 1999 - 2021. All Rights Reserved.
--
--    NAME
--      wwv_flow_approval.sql
--
--    DESCRIPTION
--      Oracle APEX Approvals Component.
--
--    SECURITY
--
--    NOTES:
--
--    RUNTIME DEPLOYMENT: YES
--
--    MODIFIED   (MM/DD/YYYY)
--      ralmuell   08/30/2021 - Created
--
--------------------------------------------------------------------------------

--==============================================================================
-- Data Types.
--==============================================================================

subtype t_task_state         is wwv_flow_tasks$.state%type;

--------------------------------------------------------------------------------
-- Task Participant Definition
--------------------------------------------------------------------------------
type t_task_def_participant is record (
    type                     wwv_flow_task_def_participants.participant_type%type,
    identity                 wwv_flow_task_def_participants.identity_type%type,
    value_type               wwv_flow_task_def_participants.value_type%type,
    value_language           wwv_flow_task_def_participants.value_language%type,
    value                    wwv_flow_task_def_participants.value%type);

--------------------------------------------------------------------------------
-- Collection of Task Participant Definitions 
--------------------------------------------------------------------------------
type t_task_def_participants is table of t_task_def_participant 
    index by pls_integer;

--------------------------------------------------------------------------------
-- Task Parameter Definition
--------------------------------------------------------------------------------
type t_task_def_parameter is record (
    static_id                wwv_flow_task_def_params.static_id%type,
    label                    wwv_flow_task_def_params.label%type,
    data_type                wwv_flow_task_def_params.data_type%type,
    is_required              wwv_flow_task_def_params.is_required%type,
    param_comment            wwv_flow_task_def_params.param_comment%type);

--------------------------------------------------------------------------------
-- Collection of Task Parameter Definitions
--------------------------------------------------------------------------------
type t_task_def_parameters is table of t_task_def_parameter 
    index by pls_integer;

--------------------------------------------------------------------------------
-- Task Action Definition
--------------------------------------------------------------------------------
type t_task_def_action is record (
    on_event                 wwv_flow_task_def_actions.on_event%type,
    outcome                  wwv_flow_task_def_actions.outcome%type,
    action_type              wwv_flow_task_def_actions.action_type%type,
    action_language          wwv_flow_task_def_actions.action_clob_language%type,
    action                   wwv_flow_task_def_actions.action_clob%type);

--------------------------------------------------------------------------------
-- Collection of Task Action Definitions
--------------------------------------------------------------------------------
type t_task_def_actions is table of t_task_def_action index by pls_integer;

--------------------------------------------------------------------------------
-- Task Definition
--------------------------------------------------------------------------------
type t_task_def is record (
    subject                  wwv_flow_task_defs.subject%type,
    priority                 wwv_flow_task_defs.priority%type,
    due_on                   wwv_flow_task_defs.due_on_interval%type,
    participants             t_task_def_participants,
    actions                  t_task_def_actions);

--------------------------------------------------------------------------------
-- Task Participant
--------------------------------------------------------------------------------
type t_task_participant is record (
    type                     wwv_flow_task_def_participants.participant_type%type,
    identity                 wwv_flow_task_def_participants.identity_type%type,
    value                    varchar2(255));

--------------------------------------------------------------------------------
-- Collection of Task Participants
--------------------------------------------------------------------------------
type t_task_participants is table of t_task_participant index by pls_integer;

--==============================================================================
-- Constants
--==============================================================================

--------------------------------------------------------------------------------
-- Task Participants related constants.
--------------------------------------------------------------------------------
c_task_participants_default  t_task_participants;


--------------------------------------------------------------------------------
-- Task (Instance) States
--------------------------------------------------------------------------------
c_task_state_assigned        constant t_task_state   := 'ASSIGNED';
c_task_state_unassigned      constant t_task_state   := 'UNASSIGNED';
c_task_state_completed       constant t_task_state   := 'COMPLETED';
c_task_state_failed          constant t_task_state   := 'FAILED';
c_task_state_errored         constant t_task_state   := 'ERRORED';
c_task_state_cancelled       constant t_task_state   := 'CANCELLED';



--==============================================================================
-- find a task definition ID by its static ID
--
-- Parameters:
-- * p_application_id        ID of the application which contains the 
--                           task definition
-- * p_static_id             ID of the task definition (varchar2)
--
-- Returns:
-- * the ID of the task definition
--==============================================================================
function get_task_def_id(
    p_application_id      in number,
    p_static_id           in varchar2) 
return number;

--==============================================================================
-- compute component values for task definition parameters and return the name-value pairs as
-- wwv_flow_approvals_api.t_parameter_values array
--
-- PARAMETERS:
-- * p_page_process_id        page process ID to fetch parameters for
-- * p_task_def_id            task definition ID to fetch parameters for
-- RETURNS:
--   fetched task parameters for the given component
--==============================================================================
function get_task_comp_parameters (
    p_page_process_id        in number,
    p_task_def_id            in number )
return wwv_flow_approval_api.t_task_parameters;

--==============================================================================
-- Create a new task. A new Task (Instance) is created. Depending on the task
-- definition participant setting, the Task is set to state Unassigned or Assigned.
--
-- Parameters:
-- * p_application_id        The application ID that creates the Task
-- * p_task_def_id           The Task Definition ID.
-- * p_subject               The subject (expression of the Task
-- * p_parameters            The task parameters
-- * p_priority              A (optional) task priority, default is 3
-- * p_initiator:            A (optional) initiator information for the task
-- * p_detail_pk:            Primary key value of the system of records for the task
--
-- Returns:
-- Returns the ID of the newly created task.
--
-- Since:
-- 22.1
--==============================================================================
function create_task(
    p_application_id         in number           default wwv_flow.g_flow_id,
    p_task_def_id            in number,
    p_subject                in varchar2         default null,
    p_parameters             in wwv_flow_approval_api.t_task_parameters,
    p_priority               in integer          default null,
    p_initiator              in varchar2         default null,
    p_detail_pk              in varchar2         default null)
return number;

--==============================================================================
-- Add a comment to a task.
--
-- Parameters:
-- * p_task_id: The Task Id.
-- * p_text:    The comment text
--
-- Since:
-- 22.1
--==============================================================================
procedure add_task_comment(
    p_task_id                in number,
    p_text                   in varchar2);

--==============================================================================
-- Add a potential owner to a task.
--
-- Parameters:
-- * p_task_id:         The Task Id.
-- * p_potential_owner: The potential owner
-- * p_identity_type:   The identity type of the potential owner, default is
--                      'USER'
--
-- Since:
-- 22.1
--==============================================================================
procedure add_task_potential_owner(
    p_task_id                in number,
    p_potential_owner        in varchar2,
    p_identity_type          in wwv_flow_approval_api.t_task_identity_type 
                                default wwv_flow_approval_api.c_task_identity_type_user);

--==============================================================================
-- Complete a task.
--
-- Parameters:
-- * p_task_id               The Task Id.
-- * p_outcome               The outcome of the Task
--
-- # State Handling
-- Pre-State: ASIGNED
-- Post-State: COMPLETED
--
-- Since:
-- 22.1
--==============================================================================
procedure complete_task(
    p_task_id                in number,
    p_outcome                in wwv_flow_approval_api.t_task_outcome,
    p_autoclaim              in boolean default false);

--==============================================================================
-- Assign the task to one Potential Owner and set the task state to Assigned.
--
-- Parameters:
-- * p_task_id:     The Task Id.
-- * p_to_potential_owner: A new potential owner
--
-- # State Handling
-- Pre-State: UNASIGNED, ASIGNED
-- Post-State: ASIGNED
--
-- Since:
-- 22.1
--==============================================================================
procedure delegate_task(
    p_task_id                in number,
    p_to_user                in varchar2);

--==============================================================================
-- Cancel the task, i.e. set the task to state 'CANCELLED'.
--
-- Parameters:
-- * p_task_id: The Task Id.
--
-- # State Handling
-- Pre-State: <Any>
-- Post-State: CANCELLED
--
-- Since:
-- 22.1
--=============================================================================
procedure cancel_task(
    p_task_id                in number);
 
--==============================================================================
-- Release the task, i.e. set the task back to state Unassigned.
--
-- Parameters:
-- * p_task_id: The Task Id.
--
-- # State Handling
-- Pre-State: ASIGNED
-- Post-State: UNASIGNED
--
-- Since:
-- 22.1
--=============================================================================
procedure release_task(
    p_task_id                in number);

--==============================================================================
-- Claim responsibility for a task.
--
-- Parameters:
-- * p_task_id: The Task Id.
--
-- # State Handling
-- Pre-State: UNASIGNED
-- Post-State: ASIGNED
--
-- Since:
-- 22.1
--==============================================================================
procedure claim_task(
    p_task_id                in number);

--==============================================================================
-- Set the priority of a task.
--
-- Parameters:
-- * p_task_id:  The Task Id.
-- * p_priority: The task priority (between 0..10)
--
-- Since:
-- 22.1
--==============================================================================
procedure set_task_priority(
    p_task_id                in number,
    p_priority               in integer);
  
--==============================================================================
-- Get the value of a Task parameter.
--
-- Parameters:
-- * p_task_id:         The Task Id.
-- * p_param_static_id  The static id of the parameter
--
-- Returns:
-- The task parameter value for the given static id.
--
-- Since:
-- 22.1
--==============================================================================
function get_task_parameter_value(
    p_task_id                in number,
    p_param_static_id        in varchar2,
    p_ignore_not_found       in boolean default false)
return varchar2;

--==============================================================================
-- Checks whether the given user is of a certain participant type for a Task.
--
-- Parameters:
-- * p_task_id:          The Task Id.
-- * p_participant_type: The participant type. Default is POTENTIAL_OWNER
-- * p_user:             The user to check for. Default is the current user.
--
-- Returns:
-- true, if the user given by p_user is a participant of given participant
-- type for a given task, false otherwise
--
-- Since:
-- 22.1
--==============================================================================
function is_of_participant_type(
    p_task_id                in number,
    p_participant_type       in wwv_flow_approval_api.t_task_participant_type
                                default wwv_flow_approval_api.c_task_potential_owner,
    p_user                   in varchar2 
                                default wwv_flow_security.g_user)
return boolean;

--==============================================================================
-- Checks whether the given user is allowed to perform a certain operation on
-- a Task.
--
-- Parameters:
-- * p_task_id:           The Task Id.
-- * p_operation:         The operation to check (see constants c_task_op_<XXX>)
-- * p_user:              The user to check for. 
--
-- Returns:
-- true, if the user given by p_user is allowed to perform the operation 
-- given by p_operation, false otherwise
--
-- Since:
-- 22.1
--==============================================================================
function is_allowed(
    p_task_id                in number,
    p_operation              in wwv_flow_approval_api.t_task_operation,
    p_user                   in varchar2
                                default wwv_flow_security.g_user,
    p_new_participant        in varchar2 default null)
return boolean;

--==============================================================================
-- Checks whether the given user is a business administrator for at least one 
-- task definition.
--
-- Parameters:
-- * p_user:              The user to check for. Default is logged in user
-- * p_application_id:    The application to check for. Default behavior 
--                        checks against all applications in the workspace.
--
-- Returns:
-- true, if the user given by p_user is at least in on task definition configured
-- as participant type BUSINESS_ADMIN, false otherwise
--
-- Since:
-- 22.1
--==============================================================================
function is_business_admin(
    p_user in varchar2 default wwv_flow_security.g_user,
    p_application_id in number   default null)
return boolean;

--==============================================================================
-- Get the tasks of a user depending on the given context. Context can be
-- MY_TASKS, ADMIN_TASKS, INITIATED_BY_ME or SINGLE_TASK.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_context:              The list context. Default is MY_TASKS.
-- * p_user:                 The user to check for. Default is logged in user. 
--                           Needs p_context set to MY_TASKS, ADMIN_TASKS or INITIATED_BY_ME.
-- * p_task_id:              Filter for a task ID instead of a user. Default is null.
--                           Needs p_context set to SINGLE_TASK.
-- * p_application_id:       Filter for an application. Default is null (all applications)
--
-- Return:
-- sys_refcursor
--==============================================================================
function get_tasks (
    p_context        in varchar2 default wwv_flow_approval_api.c_context_my_tasks,
    p_user           in varchar2 default wwv_flow_security.g_user,
    p_task_id        in number   default null,
    p_application_id in number   default null )
return sys_refcursor;

--==============================================================================
-- Get the approval log for a task.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:   The task ID.
--
-- Return:
-- sys_refcursor
--
-- Example:
-- ~~~
-- select *
--   from table ( apex_approval.get_task_history ( p_task_id => 1234 ) )
-- ~~~
--==============================================================================
function get_task_history (
    p_task_id        in number )
return sys_refcursor;

--==============================================================================
-- Get the potential new owners of a task. 
-- The actual owner is excluded from the list.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:  The task ID.
--
-- Return:
-- sys_refcursor
--==============================================================================
function get_task_delegates (
    p_task_id in number )
return sys_refcursor;

--==============================================================================
-- Get the potential new priorities of a task. 
-- The actual priority is excluded from the list.
--
-- Note: The function will only return data in the context of a valid APEX 
-- session. It will return no data in SQL Workshop.
--
-- Since: 22.1
--
-- Parameters:
-- * p_task_id:  The task ID.
--
-- Return:
-- wwv_flow_t_number
--==============================================================================
function get_task_priorities (
    p_task_id in number )
return sys_refcursor;

end wwv_flow_approval;
/
show err

set define '^'
 