set define off verify off
prompt ...create flow objects

Rem  Copyright (c) 1999, 2022, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      tab.sql
Rem
Rem    DESCRIPTION
Rem      Flow object creation script.  Create tables, views,
Rem      sequences, Indexes
Rem
Rem    NOTES
Rem      Tables that describe a given flow.  Every attribute relates
Rem      back to the ID column of the WWV_FLOWS table.
Rem
Rem    RUNTIME DEPLOYMENT: YES
Rem
Rem    MODIFIED   (MM/DD/YYYY)
Rem       mhichwa  08/04/1999 - Created

--##############################################################################
--#
--# GENERIC OBJECTS
--#
--##############################################################################
prompt GENERIC OBJECTS

--==============================================================================
-- SEQUENCES:
--   - wwv_flow_session_seq: Previously used for sessions, now only for debug page view IDs.
--   - wwv_seq:              Component IDs (see package wwv_id).
--
create sequence wwv_flow_session_seq cache 1000
/
create sequence wwv_seq cache 1000
/


--==============================================================================
create table wwv_flow_charsets (
    id                         number
                               constraint wwv_flow_charsets_pk
                               primary key,
    display_name               varchar2(4000) not null,
    iana_charset               varchar2(255)  not null,
    db_charset                 varchar2(255)  not null
    )
/


--==============================================================================
create table wwv_flow_standard_icons(
    id            number        not null,
    icon_name     varchar2(255) not null,
    icon_size     number,
    icon_library  varchar2(20)  not null
                  constraint wwv_flow_standard_icon_lib_ck
                  check ( icon_library in ( 'APEX','FONTAPEX','FONTAPEX_LATEST')),
    icon_category varchar2(200),
    icon_filters  varchar2(4000),
    constraint wwv_flow_standard_icons_pk primary key (id))
/

create unique index wwv_flow_std_icons_idx1 on wwv_flow_standard_icons (icon_name,icon_library)
/

--==============================================================================
-- Standard Item Types for page items
create table wwv_flow_standard_item_types (
    id           number
                 constraint wwv_flow_std_item_type_pk
                 primary key,
    name         varchar2(45)  not null
                               constraint wwv_flow_std_item_type_name_ck
                               check (upper(name)=name),
    display_name varchar2(100) not null
  );

alter table wwv_flow_standard_item_types
  add constraint wwv_flow_std_item_type_uk unique (name);


--==============================================================================
create table wwv_flow_customized_tasks (
    id                          number primary key,
    task_name                   varchar2(4000),
    task_link                   varchar2(4000) null,
    display_sequence            number,
    display_location            varchar2(30)
                                constraint wwv_flow_ctasks_location
                                check (display_location in ('WORKSPACE_HOME','WORKSPACE_LOGIN')),
    displayed                   varchar2(1)
                                constraint wwv_flow_ctasks_displayed
                                check (displayed in ('Y','N')),
    security_group_id number    null)
/

--==============================================================================
create table wwv_flow_restricted_schemas(
    id                number
                      constraint wwv_flow_restrict_schema_pk primary key,
    schema            varchar2(128)      not null,
    created_by        varchar2(255)      not null,
    created_on        date               not null,
    last_updated_by   varchar2(255),
    last_updated_on   date)
/

create unique index wwv_flow_restricted_schemas_uk on wwv_flow_restricted_schemas (schema)
/

--==============================================================================
create table wwv_flow_rschema_exceptions (
    id                number
                      constraint wwv_flow_rschema_exceptions_pk primary key,
    schema_id         number             not null
                      constraint wwv_flow_rschema_exceptions_fk
                      references wwv_flow_restricted_schemas(id)
                      on delete cascade,
    workspace_name    varchar2(255)      not null,
    created_by        varchar2(255)      not null,
    created_on        date               not null,
    last_updated_by   varchar2(255),
    last_updated_on   date)
/

create index wwv_flow_rs_exp_fk1 on wwv_flow_rschema_exceptions (schema_id)
/

--==============================================================================
-- Entry points that can be called via URLs. See
-- * wwv_flow_epg_include_modules: Verification.
-- * apex_install_data.sql: INTERN/LEGACY entry points.
-- * wwv_flow_upgrade: Copying CUSTOM entry points.
--
create table wwv_flow_web_entry_points (
    id                number
                      constraint wwv_flow_ords_entry_points_pk
                      primary key,
    name              varchar2(60)  not null
                      constraint wwv_flow_web_entry_points_ck1
                      check (name = upper(name))
                      constraint wwv_flow_web_entry_points_uk
                      unique,
    request_methods   varchar2(52)   not null
                      constraint wwv_flow_web_entry_points_ck2
                      check (regexp_like (
                                 request_methods,
                                 '^((^|,)(GET|PUT|POST|DELETE|HEAD|OPTIONS|PATCH|CONNECT|TRACE))+$' )),
    origin            varchar2(6)   not null
                      constraint wwv_flow_ords_entry_points_ck3
                      check (origin in ('INTERN',
                                        'LEGACY',
                                        'CUSTOM')),
    --
    created_by        varchar2(255) not null,
    created_on        date          not null )
/

--##############################################################################
--#
--# WORKSPACE
--#
--##############################################################################
prompt WORKSPACE

--==============================================================================
-- id                      = internal ID number
-- provisioning_company_id = security_group_id (workspace id)
-- short_name              = the company name / workspace name
-- schema_name             = name used to login into the service
--
create table wwv_flow_companies (
    id                            number
                                  constraint wwv_flow_companies_pk
                                  primary key,
    provisioning_company_id       number        not null,
    short_name                    varchar2(255) not null,
    display_name                  varchar2(4000),
    source_identifier             varchar2(8),
    workspace_service_id          number,
    first_schema_provisioned      varchar2(128) not null,
    cookie_name                   varchar2(40)  not null,
    allow_plsql_editing           varchar2(1),
    --
    allow_app_building_yn         varchar2(1),
    allow_packaged_app_ins_yn     varchar2(1)
                                  constraint allow_packaged_app_ins_ck
                                  check (allow_packaged_app_ins_yn in ('Y','N')),
    allow_sql_workshop_yn         varchar2(1),
    allow_team_development_yn     varchar2(1),
    allow_team_dev_files_yn       varchar2(1)
                                  constraint wwv_flow_companies_files_ck
                                  check (allow_team_dev_files_yn in ('Y','N')),
    team_dev_fs_limit             number,
    allow_issue_files_yn          varchar2(1)
                                  constraint wwv_flow_comp_issue_files_ck
                                  check (allow_issue_files_yn in ('Y','N')),
    issue_fs_limit                number,
    allow_to_be_purged_yn         varchar2(1)
                                  constraint wwv_flow_companies_purge_ck
                                  check (allow_to_be_purged_yn in ('Y','N')),
    allow_restful_services_yn     varchar2(1) default 'Y' not null
                                  constraint wwv_flow_companies_restful_ck
                                  check (allow_restful_services_yn in ('Y','N')),
    auto_archive_yn               varchar2(1)
                                  constraint wwv_flow_companies_a_arch_ck
                                  check (auto_archive_yn in ('Y','N')),
    webservice_logging_yn         varchar2(1)
                                  constraint wwv_flow_companies_ws_log_ck
                                  check (webservice_logging_yn in ('Y','N')),
    --
    last_login                    date,
    expire_fnd_user_accounts      varchar2(1)
                                  constraint wwv_flow_companies_expire
                                  check (expire_fnd_user_accounts in ('Y','N')),
    account_lifetime_days         number,
    fnd_user_max_login_failures   number,
    account_status                varchar2(10)
                                  constraint wwv_flow_companies_ck2
                                  check (account_status in ('AVAILABLE','ASSIGNED','SUSPENDED','TERMINATE','ASSIGN_LCK')),
    builder_notification_message  varchar2(4000),
    --
    rm_consumer_group             varchar2(255),
    --
    created_on                    date,
    assigned_on                   date,
    --
    files_version                 number default 1 not null,
    --
    host_prefix                   varchar2(255)
                                  constraint wwv_flow_companies_host_chk
                                  check (host_prefix=upper(host_prefix)),
    path_prefix                   varchar2(255)
                                  constraint wwv_flow_companies_path_chk
                                  check (path_prefix=upper(path_prefix)),
    cloud_group_name              varchar2(255)
                                  constraint wwv_flow_companies_group_chk
                                  check (cloud_group_name=upper(cloud_group_name)),
    cloud_service_name            varchar2(255)
                                  constraint wwv_flow_companies_service_chk
                                  check (cloud_service_name=upper(cloud_service_name)),
    qos_max_workspace_requests    number,
    qos_max_session_requests      number,
    qos_max_session_kill_timeout  number,
    max_session_length_sec        number,
    max_session_idle_sec          number,
    session_timeout_warning_sec   number,
    max_file_bytes                number,
    max_email_messages            number,
    max_webservice_requests       number,
    --
    content_cache_size_target     number,
    content_cache_max_file_size   number,
    env_banner_yn                 varchar2(1) default 'N' not null
                                  constraint wwv_flow_companies_envban_ck
                                  check (env_banner_yn in ('Y','N')),
    env_banner_label              varchar2(255),
    env_banner_color              varchar2(128),
    env_banner_pos                varchar2(5) default 'LEFT'
                                  constraint wwv_flow_companies_envbanp_ck
                                  check (env_banner_pos in ('LEFT','TOP')),
    workspace_description         varchar2(4000),
    --
    -- constraints
    --
    constraint wwv_flow_companies_name_uk
        unique (short_name)
        using index (
            create index wwv_flow_companies_name_uk
            on wwv_flow_companies(short_name, provisioning_company_id) ),
    constraint wwv_flow_companies_prov_uk
        unique (provisioning_company_id),
    constraint wwv_flow_companies_path_uk
        unique (path_prefix),
    constraint wwv_flow_companies_cloud_uk
        unique (cloud_group_name,cloud_service_name)
        using index compress 1
    )
/

--==============================================================================
create table wwv_flow_company_schemas (
    id                      number
                            constraint wwv_flow_company_schemas_pk
                            primary key,
    security_group_id       number not null
                            constraint wwv_flow_comp_schemas_fk
                            references wwv_flow_companies(provisioning_company_id)
                            on delete cascade,
    schema                  varchar(128),
    is_apex$_schema         varchar2(1)
                            constraint wwv_flow_comp_schemas_apex$_ck
                            check (is_apex$_schema in ('Y','N')),
    company_schema_comments varchar2(4000)
    )
/

create unique index wwv_flow_company_schemas_idx on wwv_flow_company_schemas (security_group_id, schema);

create unique index wwv_flow_company_schemas_idx2 on wwv_flow_company_schemas
    (case when nvl(is_apex$_schema,'N') = 'Y' then security_group_id end,
     case when nvl(is_apex$_schema,'N') = 'Y' then is_apex$_schema end);

--==============================================================================
create table wwv_flow_authorized_urls(
    id                 number primary key,
    security_group_id  number         not null,
    url                varchar2(4000) not null,
    url_md5            varchar2(255)  not null,
    description        varchar2(4000),
    created_on         date,
    created_by         varchar2(255),
    last_updated_on    date,
    last_updated_by    varchar2(255) )
/

create unique index wwv_flow_authorized_urls_idx1 on wwv_flow_authorized_urls ( security_group_id, url_md5 )
/

--==============================================================================
create table wwv_flow_company_static_files (
    id                        number         not null
                                             constraint wwv_flow_comp_stat_file_pk
                                             primary key,
    security_group_id         number         not null
                                             constraint wwv_flow_comp_stat_file_sg_fk
                                             references wwv_flow_companies(provisioning_company_id) on delete cascade,
    file_name                 varchar2(255)  not null,
    mime_type                 varchar2(255)  not null,
    file_charset              varchar2(128),
    file_content              blob,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  );

alter table wwv_flow_company_static_files
  add constraint wwv_flow_comp_stat_file_uk unique (security_group_id, file_name);

--==============================================================================
create table wwv_flow_app_build_pref (
  id                                  number not null,
  security_group_id                   number not null
                                      constraint wwv_flow_app_bldprf_fk
                                      references wwv_flow_companies(provisioning_company_id)
                                      on delete cascade,
  default_parsing_schema              varchar2(255),
  default_authentication_scheme       varchar2(255),
  default_application_theme           number,
  default_tabs                        varchar2(255),
  default_proxy_server                varchar2(500),
  default_language                    varchar2(255),
  default_language_derived            varchar2(255),
  date_format                         varchar2(255),
  date_time_format                    varchar2(255),
  timestamp_format                    varchar2(255),
  timestamp_tz_format                 varchar2(255),
  --
  created_on                          date,
  created_by                          varchar2(255),
  updated_on                          date,
  updated_by                          varchar2(255),
  constraint wwv_flow_app_bldprf_pk   primary key (id)
  );

create unique index wwv_flow_app_build_pref_idx1 on
    wwv_flow_app_build_pref (security_group_id);

--==============================================================================
create table wwv_flow_platform_prefs (
    id                number
                      constraint wwv_flow_platform_prefs_pk
                      primary key,
    name              varchar2(255) not null,
    value             varchar2(4000) not null,
    created_on        date,
    last_updated_on   date,
    pref_desc         varchar2(4000),
    security_group_id number not null
                      constraint wwv_flow_platform_prefs_fk
                      references wwv_flow_companies(provisioning_company_id)
                      on delete cascade)
/

create index wwv_flow_platform_prefs_idx1 on wwv_flow_platform_prefs (name)
/

create index wwv_flow_platform_prefs_fkidx on wwv_flow_platform_prefs (security_group_id,name)
/

--==============================================================================
create table wwv_flow_ui_types (
    id                             number         not null
                                                  constraint wwv_flow_ui_type_pk
                                                  primary key,
    name                           varchar2(30)   not null /* eg. DESKTOP, SMARTPHONE, TABLET, PHONEGAP_SMARTPHONE, PHONEGAP_TABLET, EXTJS */
                                                  constraint wwv_flow_ui_type_name_ck
                                                       check ( name = upper(name) and instr(name, ':') = 0 ),
    based_on_ui_type_id            number,         /* Foreign key to inherit the components of that base ui type. Eg. PHONEGAP_SMARTPHONE inherits everything from SMARTPHONE */
    autodetect_js_file_urls        varchar2(4000),
    autodetect_js_function_body    varchar2(4000), /* used to automatically determine the UI in an automatically generated page. Similar to our automatic timezone detection */
    autodetect_plsql_function_body varchar2(4000), /* used to automatically determine the UI on the server side. Eg. by sniffing the USER_AGENT (for iPhone, iPad, ...) */
    security_group_id              number,         /* nullable! All our standard UI types (see above) are configured with security group id NULL, all
                                                               custom ui types defined by customers or plug-in developers will use the current workspace */
    created_by                     varchar2(255),
    created_on                     date,
    last_updated_by                varchar2(255),
    last_updated_on                date
  );

alter table wwv_flow_ui_types add constraint wwv_flow_ui_type_name_uk unique (name, security_group_id)
/

alter table wwv_flow_ui_types add constraint wwv_flow_ui_type_based_on_fk
                                             foreign key (based_on_ui_type_id)
                                             references wwv_flow_ui_types(id)
/

create index wwv_flow_ui_type_idx1 on wwv_flow_ui_types (based_on_ui_type_id)
/

--==============================================================================
create table wwv_flow_builder_features (
    id                             number         not null
                                                  constraint wwv_flow_bld_feature_pk
                                                  primary key,
    name                           varchar2(30)   not null
                                                  constraint wwv_flow_bld_feature_name_ck
                                                       check ( name = upper(name) )
  );

alter table wwv_flow_builder_features add constraint wwv_flow_bld_feature_name_uk unique (name)
/

--==============================================================================
create table wwv_flow_ui_type_features (
    id                             number         not null
                                                  constraint wwv_flow_ui_type_feature_pk
                                                  primary key,
    ui_type_id                     number         not null
                                                  constraint wwv_flow_ui_type_feature_fk1
                                                  references wwv_flow_ui_types,
    builder_feature_id             number         not null
                                                  constraint wwv_flow_ui_type_feature_fk2
                                                  references wwv_flow_builder_features,
    security_group_id              number,         /* nullable! All our standard UI types (see above) are configured with security group id NULL, all
                                                               custom ui types defined by customers or plug-in developers will use the current workspace */
    created_by                     varchar2(255),
    created_on                     date,
    last_updated_by                varchar2(255),
    last_updated_on                date
  );

alter table wwv_flow_ui_type_features add constraint wwv_flow_ui_type_feature_uk unique (ui_type_id, builder_feature_id)
/

create index wwv_flow_builder_features_idx2 on wwv_flow_ui_type_features (builder_feature_id) compress
/

--##############################################################################
--#
--# SESSION STATE MANAGEMENT
--#
--##############################################################################
prompt SESSION STATE MANAGEMENT

--==============================================================================
-- master session table
--
-- IF YOU CHANGE THIS TABLE OR IT'S INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--
create table wwv_flow_sessions$ (
    id                        number
                              constraint wwv_flow_sessions_pk
                              primary key,
    cookie_value              varchar2(32)  not null,
    security_group_id         number        not null,
    crypto_salt               raw(32)       not null,
    --
    created_on                date          not null,
    max_idle_sec              number(5,0)   not null,
    idle_timeout_on           date          not null,
    life_timeout_on           date          not null,
    --
    remote_addr               varchar2(255),
    on_new_instance_fired_for varchar2(4000),
    --
    username                  varchar2(255),
    authentication_result     number,
    --
    screen_reader_mode_yn     varchar2(1),
    high_contrast_mode_yn     varchar2(1),
    --
    session_time_zone         varchar2(255),
    session_lang              varchar2(5)
                              constraint wwv_flow_sessions_ck_lang
                              check(regexp_instr(trim(lower(session_lang)), '^[abcdefghijklmnopqrstuvwxyz]{2}(-[abcdefghijklmnopqrstuvwxyz]{2})?$') = 1),
    session_territory         varchar2(255),
    ras_sessionid             raw(16),
    workspace_user_id         number,
    authentication_id         number,
    tenant_id                 varchar2(32),
    --
    debug_level               number(1)
                              constraint wwv_flow_sessions_ck_debug
                              check(debug_level between 1 and 9),
    trace_mode                number(1)
                              constraint wwv_flow_sessions_ck_trace
                              check(trace_mode between 1 and 2)
    )
    storage (initial 100K next 1M freelists 20)
/

alter table wwv_flow_sessions$ add (
    timeout_computation as ( to_number(to_char (
                                 least(idle_timeout_on, life_timeout_on),
                                 'YYDDDHH24' )) )
)
/

alter table wwv_flow_sessions$ add (
    is_subscription_session as ( 
        case when     life_timeout_on = date'2099-12-31'
                  and max_idle_sec    = 99999 then
            'Y'
        else
            'N'
        end ) )
/

create index wwv_flow_sessions$_idx_cookie on
    wwv_flow_sessions$(cookie_value)
/
create index wwv_flow_sessions$_idx_timeout on
    wwv_flow_sessions$(timeout_computation)
    compress
/
create index wwv_flow_sessions$_idx_sgid on
    wwv_flow_sessions$(security_group_id)
    compress
/

--==============================================================================
--
-- passed/failed authorizations for a session
--
-- the fk from authorization_id to wwv_flow_security_schemes is intentionally
-- missing, to save space
--
-- IF YOU CHANGE THIS TABLE OR IT'S INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--
create table wwv_flow_session_authz$ (
    session_id           number        not null
                                       constraint wwv_flow_session_authz$_fk1
                                       references wwv_flow_sessions$
                                       on delete cascade,
    component_type       varchar2(30)  not null,
    component_id         number        not null,
    authorization_id     number        not null,
    is_successful        varchar2(1)   not null
                                       constraint wwv_flow_session_authz$_ck1
                                       check (is_successful in ('Y','N')),
    constraint wwv_flow_session_authz$_pk
        primary key(session_id, component_type, component_id, authorization_id)
) organization index compress 3
/

--==============================================================================
--
-- session assigned groups
--
-- IF YOU CHANGE THIS TABLE OR IT'S INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--
create table wwv_flow_session_groups$ (
    session_id           number        not null
                                       constraint wwv_flow_session_groups$_fk1
                                       references wwv_flow_sessions$
                                       on delete cascade,
    group_name           varchar2(255) not null,
    constraint wwv_flow_session_groups$_pk
        primary key(session_id, group_name)
) organization index compress 1
/

--==============================================================================
--
-- session state
--
-- IF YOU CHANGE THIS TABLE OR IT'S INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--
create table wwv_flow_data (
    flow_instance        number        constraint wwv_flow_data_session_fk
                                       references wwv_flow_sessions$
                                       on delete cascade,
    item_id              number,
    item_filter          varchar2(1)   constraint valid_item_filter
                                       check (item_filter in ('Y','N')),
    session_state_status varchar2(1)   constraint valid_session_state_status
                                       check (session_state_status in ('I','U','R')),
    data_type            number        default 1 not null
                                       constraint wwv_flow_data_ck_type
                                       check (data_type in (1,11)),
    flow_id              number,
    item_name            varchar2(255),
    is_encrypted         varchar2(1),
    item_value_vc2       varchar2(4000),
    item_value_clob      clob,
    constraint wwv_flow_data_pk primary key (flow_instance, item_id)
                         using index (
                             create index wwv_flow_data_pk on wwv_flow_data (
                                 flow_instance, item_id, flow_id )
                             compress 1) )
    storage (initial 1M next 1M freelists 20)
    initrans 8
    lob (item_value_clob) store as (cache reads disable storage in row)
/

create index WWV_FLOW_DATA_IDX1 on wwv_flow_data (flow_id) compress 1
/

--==============================================================================
-- C O L L E C T I O N S
--
-- IF YOU CHANGE THIS TABLE OR IT'S INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--
create table wwv_flow_collections$ (
    id                 number        not null,
    session_id         number        not null,
    user_id            varchar2(255) not null,
    flow_id            number        not null,
    collection_name    varchar2(255) not null,
    collection_changed varchar2(10)  not null,
    created_on         date          not null,
    security_group_id  number        not null,
    constraint wwv_flow_collections_pk primary key(id),
    constraint wwv_flow_collections_fk foreign key(session_id)
        references wwv_flow_sessions$(id)
        on delete cascade,
    constraint wwv_flow_collections_uk unique (
        session_id, user_id, flow_id, collection_name, security_group_id )
)
/
--==============================================================================
--
-- IF YOU CHANGE THIS TABLE OR IT'S INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--

create table wwv_flow_collection_members$ (
    collection_id     number        not null,
    seq_id            number        not null,
    security_group_id number        not null,
    c001              varchar2(4000),
    c002              varchar2(4000),
    c003              varchar2(4000),
    c004              varchar2(4000),
    c005              varchar2(4000),
    c006              varchar2(4000),
    c007              varchar2(4000),
    c008              varchar2(4000),
    c009              varchar2(4000),
    c010              varchar2(4000),
    c011              varchar2(4000),
    c012              varchar2(4000),
    c013              varchar2(4000),
    c014              varchar2(4000),
    c015              varchar2(4000),
    c016              varchar2(4000),
    c017              varchar2(4000),
    c018              varchar2(4000),
    c019              varchar2(4000),
    c020              varchar2(4000),
    c021              varchar2(4000),
    c022              varchar2(4000),
    c023              varchar2(4000),
    c024              varchar2(4000),
    c025              varchar2(4000),
    c026              varchar2(4000),
    c027              varchar2(4000),
    c028              varchar2(4000),
    c029              varchar2(4000),
    c030              varchar2(4000),
    c031              varchar2(4000),
    c032              varchar2(4000),
    c033              varchar2(4000),
    c034              varchar2(4000),
    c035              varchar2(4000),
    c036              varchar2(4000),
    c037              varchar2(4000),
    c038              varchar2(4000),
    c039              varchar2(4000),
    c040              varchar2(4000),
    c041              varchar2(4000),
    c042              varchar2(4000),
    c043              varchar2(4000),
    c044              varchar2(4000),
    c045              varchar2(4000),
    c046              varchar2(4000),
    c047              varchar2(4000),
    c048              varchar2(4000),
    c049              varchar2(4000),
    c050              varchar2(4000),
    n001              number,
    n002              number,
    n003              number,
    n004              number,
    n005              number,
    d001              date,
    d002              date,
    d003              date,
    d004              date,
    d005              date,
    clob001           clob default empty_clob(),
    blob001           blob default empty_blob(),
    md5_original      varchar2(4000),
    xmltype001        sys.xmltype,
    constraint wwv_flow_collection_members_pk primary key (
        collection_id, seq_id )
        using index compress 1 initrans 8,
    constraint wwv_flow_collection_members_fk foreign key (collection_id)
        references wwv_flow_collections$(id)
        on delete cascade
) initrans 8
    lob (clob001) store as (cache reads enable storage in row),
    lob (blob001) store as (cache reads enable storage in row)
/

--##############################################################################
--#
--# PROPERTY EDITOR
--#
--##############################################################################
prompt PROPERTY EDITOR

--==============================================================================
create table wwv_pe_component_types (
    id                              number          not null
                                                    constraint wwv_pe_component_type_pk
                                                    primary key,
    name                            varchar2(45)    not null
                                                    constraint wwv_pe_component_type_name_ck
                                                    check (upper(name)=name),
    plural_title                    varchar2(80)    not null,
    singular_title                  varchar2(80)    not null,
    table_name                      varchar2(30)    not null,
    view_name                       varchar2(30),
    display_sequence                number          not null,
    pk_column                       varchar2(30)    not null
                                                    constraint wwv_pe_comp_type_pk_col_ck
                                                    check (upper(pk_column)=pk_column),
    parent_component_type_id        number          constraint wwv_pe_comp_type_parent_fk
                                                    references wwv_pe_component_types on delete cascade,
    parent_fk_column                varchar2(30)    constraint wwv_pe_comp_type_par_fk_col_ck
                                                    check (upper(parent_fk_column)=parent_fk_column),
    page_id_column                  varchar2(30)    constraint wwv_pe_comp_type_pageid_col_ck
                                                    check (upper(page_id_column)=page_id_column),
    display_column                  varchar2(30)    constraint wwv_pe_comp_type_displ_col_ck
                                                    check (upper(display_column)=display_column),
    where_clause                    varchar2(4000),
    validation_plsql_ret_error      varchar2(4000),
    plugin_type                     varchar2(30)    constraint wwv_pe_comp_plugin_type_ck
                                                    check (plugin_type in ( 'ALL',
                                                                            'ITEM TYPE',
                                                                            'DYNAMIC ACTION',
                                                                            'REGION TYPE',
                                                                            'PROCESS TYPE',
                                                                            'AUTHORIZATION TYPE',
                                                                            'AUTHENTICATION TYPE',
                                                                            'WEB SOURCE TYPE' )),
    plugin_column                   varchar2(30)    constraint wwv_pe_comp_type_plugin_col_ck
                                                    check (upper(plugin_column)=plugin_column),
    plugin_attributes_seq_offset    number,
    create_url                      varchar2(255),
    edit_url                        varchar2(255),
    copy_url                        varchar2(255),
    file_path                       varchar2(255),
    is_page_designer_shared         varchar2(1)     not null
                                                    constraint wwv_pe_comp_type_page_des_ck
                                                    check (is_page_designer_shared in ('Y','N')),
    is_query_only                   varchar2(1)     not null
                                                    constraint wwv_pe_comp_type_query_only_ck
                                                    check (is_query_only in ('Y','N'))
    );
alter table wwv_pe_component_types add constraint wwv_pe_component_type_uk unique (name);
create index wwv_pe_comp_type_parent_fx on wwv_pe_component_types (parent_component_type_id);

--==============================================================================
create table wwv_pe_display_groups (
    id                              number          not null
                                                    constraint wwv_pe_display_group_pk
                                                    primary key,
    name                            varchar2(45)    not null
                                                    constraint wwv_pe_display_group_name_ck
                                                    check (upper(name)=name),
    title                           varchar2(80)    not null
    );
alter table wwv_pe_display_groups add constraint wwv_pe_display_group_uk unique (name);

--==============================================================================
create table wwv_pe_properties (
    id                              number          not null
                                                    constraint wwv_pe_property_pk
                                                    primary key,
    name                            varchar2(45)    not null
                                                    constraint wwv_pe_property_name_ck
                                                    check (upper(name)=name),
    prompt                          varchar2(80)    not null,
    property_type                   varchar2(40)    not null
                                                    constraint wwv_pe_property_prop_type_ck
                                                    check (property_type in (
                                                        'COMBOBOX',
                                                        'CHECKBOXES',
                                                        'COLOR',
                                                        'COLUMN',
                                                        'COMPONENT',
                                                        'CSS',
                                                        'JAVASCRIPT',
                                                        'HIDDEN',
                                                        'HTML',
                                                        'ICON',
                                                        'INTEGER',
                                                        'LINK',
                                                        'NUMBER',
                                                        'OWNER',
                                                        'ORDER BY CLAUSE',
                                                        'ORDER BY ITEM',
                                                        'ITEM',
                                                        'FACET',
                                                        'PAGE',
                                                        'PASSWORD',
                                                        'PLSQL',
                                                        'PLSQL EXPRESSION VARCHAR2',
                                                        'PLSQL EXPRESSION BOOLEAN',
                                                        'PLSQL FUNCTION BODY VARCHAR2',
                                                        'PLSQL FUNCTION BODY BOOLEAN',
                                                        'PLSQL FUNCTION BODY SQL',
                                                        'CODE LANGUAGE',
                                                        'MLE JAVASCRIPT',
                                                        'MLE JAVASCRIPT EXPRESSION VARCHAR2',
                                                        'MLE JAVASCRIPT EXPRESSION BOOLEAN',
                                                        'MLE JAVASCRIPT FUNCTION BODY VARCHAR2',
                                                        'MLE JAVASCRIPT FUNCTION BODY BOOLEAN',
                                                        'MLE JAVASCRIPT FUNCTION BODY SQL',
                                                        'MLE JAVA',
                                                        'MLE JAVA EXPRESSION VARCHAR2',
                                                        'MLE JAVA EXPRESSION BOOLEAN',
                                                        'MLE JAVA FUNCTION BODY VARCHAR2',
                                                        'MLE JAVA FUNCTION BODY BOOLEAN',
                                                        'MLE JAVA FUNCTION BODY SQL',
                                                        'MLE PYTHON',
                                                        'MLE PYTHON EXPRESSION VARCHAR2',
                                                        'MLE PYTHON EXPRESSION BOOLEAN',
                                                        'MLE PYTHON FUNCTION BODY VARCHAR2',
                                                        'MLE PYTHON FUNCTION BODY BOOLEAN',
                                                        'MLE PYTHON FUNCTION BODY SQL',
                                                        'REGION COLUMN',
                                                        'POPUP LOV',
                                                        'RADIOS',
                                                        'SELECT LIST',
                                                        'SQL',
                                                        'SQL EXPRESSION',
                                                        'SQL EXPRESSION BOOLEAN',
                                                        'SUBSCRIPTION',
                                                        'TABLE',
                                                        'TEXT',
                                                        'TEXTAREA',
                                                        'TEXT EDITOR',
                                                        'FILE URLS JAVASCRIPT',
                                                        'FILE URLS CSS',
                                                        'TEMPLATE OPTIONS',
                                                        'SUPPORTED UI',
                                                        'STATIC LOV',
                                                        'WHERE CLAUSE',
                                                        'XML',
                                                        'HTML',
                                                        'YES NO',
                                                        'PILL BUTTONS' )),
    prop_type_yesno_override_y      varchar2(30),
    prop_type_yesno_override_n      varchar2(30),
    multi_value_delimiter           varchar2(1)     constraint wwv_pe_property_multi_del_ck
                                                    check (multi_value_delimiter in ( ',', ':', ' ' )),
    db_column                       varchar2(30)    not null,
    is_query_only                   varchar2(1)     not null
                                                    constraint wwv_pe_property_query_only_ck
                                                    check (is_query_only in ('Y','N')),
    display_length                  number          constraint wwv_pe_property_disp_l_ck
                                                    check (display_length between 1 and 4000),
    max_length                      number          constraint wwv_pe_property_max_l_ck
                                                    check (max_length between 1 and 32767),
    min_value                       number,
    max_value                       number,
    unit                            varchar2(80),
    validation_plsql_ret_error      varchar2(4000),
    text_case                       varchar2(10)    constraint wwv_pe_property_value_case_ck
                                                    check ( text_case in (
                                                        'LOWER',
                                                        'UPPER' )),
    restricted_characters           varchar2(30)    constraint wwv_pe_property_value_restr_ck
                                                    check ( restricted_characters in ('US_ONLY', 'WEB_SAFE', 'NO_SPECIAL_CHAR', 'NO_SPECIAL_CHAR_NL', 'SAFE_COMPONENT_NAME', 'SAFE_COMPONENT_NAME_NO_NUMBER', 'US_ONLY_NO_SPACE', 'US_ONLY_NO_SPACE', 'US_ONLY_UNDERSCORE_NO_SPACE', 'DOM_STATIC_ID' )),
    sql_min_column_count            number(3)       constraint wwv_pe_property_mincol_ck
                                                    check (sql_min_column_count > 0),
    sql_max_column_count            number(3)       constraint wwv_pe_property_maxcol_ck
                                                    check (sql_max_column_count > 0),
    column_data_types               varchar2(255),
    examples                        varchar2(80),
    supported_ui_types              varchar2(255)   not null,
    display_group_id                number          not null
                                                    constraint wwv_pe_property_disp_grp_fk
                                                    references wwv_pe_display_groups,
    lov_type                        varchar2(30)    constraint wwv_pe_property_lov_type_ck
                                                    check (lov_type in (
                                                        'ALL_FORMAT_MASKS',
                                                        'COMPONENT',
                                                        'DATE_FORMAT_MASKS',
                                                        'DISPLAY_POINTS',
                                                        'DISTINCT',
                                                        'EVENTS',
                                                        'GRID_COLUMNS',
                                                        'NUMBER_FORMAT_MASKS',
                                                        'PLUGINS',
                                                        'STATIC',
                                                        'TEMPLATE_OPTIONS' ) ),
    lov_component_type_id           number          constraint wwv_pe_property_lov_comp_t_fk
                                                    references wwv_pe_component_types,
    lov_component_type_scope        varchar2(15)    constraint wwv_pe_property_lov_scope_ck
                                                    check (lov_component_type_scope in ( 'COLUMN', 'PARENT', 'REGION', 'PAGE', 'PAGE_AND_GLOBAL', 'THEME', 'APPLICATION', 'WORKSPACE' )),
    is_translatable                 varchar2(1)     not null
                                                    constraint wwv_pe_property_is_trans_ck
                                                    check (is_translatable in ('Y', 'N')),
    deprecated_values               varchar2(30),
    is_searchable                   varchar2(1)     not null
                                                    constraint wwv_pe_property_is_search_ck
                                                    check (is_searchable in ('Y', 'N')),
    is_internal                     varchar2(1)     not null
                                                    constraint wwv_pe_property_is_int_ck
                                                    check (is_internal in ('Y', 'N')),
    help_text                       varchar2(80),
    comment_text                    varchar2(4000)
    );
alter table wwv_pe_properties add constraint wwv_pe_property_uk unique (name);
create index wwv_pe_property_disp_group_fx on wwv_pe_properties (display_group_id);
create index wwv_pe_property_lov_comp_t_fx on wwv_pe_properties (lov_component_type_id);

--==============================================================================
create table wwv_pe_property_lov_values (
    id                              number          not null
                                                    constraint wwv_pe_prop_lov_val_pk
                                                    primary key,
    property_id                     number          not null
                                                    constraint wwv_pe_prop_lov_val_prop_fk
                                                    references wwv_pe_properties on delete cascade,
    display_sequence                number          not null,
    display_value                   varchar2(255)   not null,
    return_value                    varchar2(255)   not null,
    icon                            varchar2(255),
    is_quick_pick                   varchar2(1)     not null
                                                    constraint wwv_pe_prop_lov_val_is_qp_ck
                                                    check (is_quick_pick in ('Y', 'N')),
    is_internal                     varchar2(1)     not null
                                                    constraint wwv_pe_prop_lov_val_is_int_ck
                                                    check (is_internal in ('Y', 'N')),
    is_deprecated                   varchar2(1)     not null
                                                    constraint wwv_pe_prop_lov_val_is_dep_ck
                                                    check (is_deprecated in ('Y', 'N')),
    help_text                       varchar2(255),
    comment_text                    varchar2(4000)
    );
alter table wwv_pe_property_lov_values add constraint wwv_pe_prop_lov_val_uk unique (property_id, return_value);

--==============================================================================
create table wwv_pe_component_props (
    id                              number          not null
                                                    constraint wwv_pe_comp_prop_pk
                                                    primary key,
    component_type_id               number          not null
                                                    constraint wwv_pe_comp_prop_comp_fk
                                                    references wwv_pe_component_types on delete cascade,
    property_id                     number          not null
                                                    constraint wwv_pe_comp_prop_prop_fk
                                                    references wwv_pe_properties on delete cascade,
    db_column                       varchar2(30),
    view_column                     varchar2(30),
    display_sequence                number          not null,
    is_required                     varchar2(1)     constraint wwv_pe_comp_prop_req_ck /* if NULL the value is derived from plug-in config */
                                                    check (is_required in ('Y', 'N')),
    check_uniqueness                varchar2(15)    constraint wwv_pe_comp_prop_unique_ck
                                                    check (check_uniqueness in ( 'PARENT', 'PAGE', 'APPLICATION', 'WORKSPACE', 'INSTANCE' )),
    is_common                       varchar2(1)     not null
                                                    constraint wwv_pe_comp_prop_common_ck
                                                    check (is_common in ('Y', 'N')),
    is_read_only                    varchar2(1)     not null
                                                    constraint wwv_pe_comp_prop_readonly_ck
                                                    check (is_read_only in ('Y', 'N')),
    reference_on_delete             varchar2(12)    constraint wwv_pe_comp_prop_ref_delete_ck
                                                    check (reference_on_delete in (
                                                        'WEAK_CASCADE',
                                                        'WEAK_NULL',
                                                        'DB_CASCADE',
                                                        'DB_NULL' )),
    reference_scope                 varchar2(10)    constraint wwv_pe_comp_prop_ref_ck
                                                    check (reference_scope in ('COMPONENT', 'ROW', 'DEPENDS')),
    supports_substitution           varchar2(1)     not null
                                                    constraint wwv_pe_comp_prop_subst_ck
                                                    check (supports_substitution in ('Y', 'N')),
    supports_template_directive     varchar2(1)     not null
                                                    constraint wwv_pe_comp_prop_templ_ck
                                                    check (supports_template_directive in ('Y', 'N')),
    null_text                       varchar2(255),
    null_help_text                  varchar2(255),
    default_value                   varchar2(4000),
    plugin_std_attributes           varchar2(255),
    depending_on_comp_prop_id       number          constraint wwv_pe_comp_prop_depend_fk
                                                    references wwv_pe_component_props on delete set null deferrable initially deferred,
    depending_on_has_to_exist       varchar2(1)     constraint wwv_pe_comp_prop_depend_ex_ck
                                                    check (depending_on_has_to_exist in ('Y', 'N')),
    depending_on_condition_type     varchar2(30)    constraint wwv_pe_comp_prop_cond_ck
                                                    check (depending_on_condition_type in (
                                                        'EQUALS',
                                                        'NOT_EQUALS',
                                                        'LIKE',
                                                        'NOT_LIKE',
                                                        'NULL',
                                                        'NOT_NULL',
                                                        'IN_LIST',
                                                        'NOT_IN_LIST',
                                                        'IN_LIST_LIKE',
                                                        'NOT_IN_LIST_LIKE' )),
    depending_on_expression         varchar2(4000),
    depending_on_comp_prop_id2      number          constraint wwv_pe_comp_prop_depend2_fk
                                                    references wwv_pe_component_props on delete set null deferrable initially deferred,
    depending_on_has_to_exist2      varchar2(1)     constraint wwv_pe_comp_prop_depend_ex2_ck
                                                    check (depending_on_has_to_exist2 in ('Y', 'N')),
    depending_on_condition_type2    varchar2(30)    constraint wwv_pe_comp_prop_cond2_ck
                                                    check (depending_on_condition_type2 in (
                                                        'EQUALS',
                                                        'NOT_EQUALS',
                                                        'LIKE',
                                                        'NOT_LIKE',
                                                        'NULL',
                                                        'NOT_NULL',
                                                        'IN_LIST',
                                                        'NOT_IN_LIST',
                                                        'IN_LIST_LIKE',
                                                        'NOT_IN_LIST_LIKE' )),
    depending_on_expression2        varchar2(4000),
    parent_comp_prop_id             number          constraint wwv_pe_comp_prop_parent_fk
                                                    references wwv_pe_component_props on delete set null deferrable initially deferred,
    examples                        varchar2(255),
    help_text                       varchar2(255),
    comment_text                    varchar2(4000)
    );
alter table wwv_pe_component_props add constraint wwv_pe_component_prop_uk unique (component_type_id, property_id);

create index wwv_pe_comp_prop_prop_type_fx on wwv_pe_component_props (property_id);
create index wwv_pe_comp_prop_depend_fx  on wwv_pe_component_props (depending_on_comp_prop_id);
create index wwv_pe_comp_prop_depend2_fx on wwv_pe_component_props (depending_on_comp_prop_id2);
create index wwv_pe_comp_prop_parent_fx  on wwv_pe_component_props (parent_comp_prop_id);

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > CREDENTIALS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > CREDENTIALS

--==============================================================================
create table wwv_credentials (
    id                  number
                        constraint wwv_credential_pk
                        primary key,
    security_group_id   number not null
                        constraint wwv_credential_wsid_fk
                        references wwv_flow_companies( provisioning_company_id )
                        on delete cascade,
    --
    name                varchar2(255) not null,
    static_id           varchar2(255) not null,
    scope               varchar2(255),
    authentication_type varchar2(25) not null
                        constraint wwv_credential_auth_type_ck check (
                            authentication_type in ( 'BASIC',
                                                     'OAUTH2_CLIENT_CREDENTIALS',
                                                     'JWT',
                                                     'OCI',
                                                     'HTTP_HEADER',
                                                     'HTTP_QUERY_STRING',
                                                     'CERTIFICATE_PAIR' )),
    client_id           varchar2(4000),
    client_secret       varchar2(4000),
    namespace           varchar2(4000),
    fingerprint         varchar2(4000),
    --
    valid_for_urls      varchar2(4000),
    --
    prompt_on_install   varchar2(1) not null
                        constraint wwv_credential_prompt_inst_ck
                        check ( prompt_on_install in ('Y', 'N' )),
    --
    credential_comment  varchar2(4000),
    --
    created_by          varchar2(255),
    created_on          date,
    last_updated_by     varchar2(255),
    last_updated_on     date )
/

alter table wwv_credentials add constraint wwv_credential_ws_name_uk unique ( security_group_id, name );
alter table wwv_credentials add constraint wwv_credential_ws_staticid_uk unique ( security_group_id, static_id );

create index wwv_credential_sgid_fx on wwv_credentials ( security_group_id ) compress;

--==============================================================================

create table wwv_credential_instances (
    id                number
                      constraint wwv_cred_instance_pk
                      primary key,
    security_group_id number not null
                      constraint wwv_cred_instance_sgid_fk
                      references wwv_flow_companies( provisioning_company_id )
                      on delete cascade,
    credential_id     number not null
                      constraint wwv_cred_instance_cred_fk
                      references wwv_credentials
                      on delete cascade,
    session_id        number
                      constraint wwv_cred_instance_sess_fk
                      references wwv_flow_sessions$
                      on delete cascade,
    value_type        varchar2(13) not null
                      constraint wwv_cred_instance_type_ck check (
                          value_type in ( 'CLIENT_ID',
                                          'CLIENT_SECRET',
                                          'ACCESS_TOKEN',
                                          'REFRESH_TOKEN',
                                          'ID_TOKEN',
                                          'NAMESPACE',
                                          'FINGERPRINT' )),
    value             clob not null,
    expiry_date       date
)
/

create index wwv_cred_instance_val_ix  on wwv_credential_instances( credential_id, value_type, session_id );
create index wwv_cred_instance_sgid_fx on wwv_credential_instances( security_group_id ) compress;
create index wwv_cred_instance_cred_fx on wwv_credential_instances( credential_id );
create index wwv_cred_instance_sess_fx on wwv_credential_instances( session_id );

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > REMOTE SERVERS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > REMOTE SERVERS

--==============================================================================
create table wwv_remote_servers (
    id                        number
                              constraint wwv_remote_srvr_pk
                              primary key,
    security_group_id         number not null
                              constraint wwv_remote_srvr_sgid_fk
                              references wwv_flow_companies( provisioning_company_id )
                              on delete cascade,
    --
    name                      varchar2(255) not null,
    static_id                 varchar2(255) not null,
    base_url                  varchar2(500) not null,
    server_type               varchar2(18) not null
                              constraint wwv_remote_srvr_type_ck
                              check ( server_type in ( 'WEB_SERVICE', 'PROXY', 'REMOTE_SQL', 'AUTHENTICATION', 'CLOUD_OBJECT_STORE', 'PRINT_SERVER' )),
    credential_id             number
                              constraint wwv_remote_srvr_cred_fk
                              references wwv_credentials ( id )
                              on delete set null,
    -- name of a credential created with the DBMS_CLOUD package
    cloud_credential_name     varchar2(128),
    --
    https_host                varchar2(500),
    --
    prompt_on_install         varchar2(1) not null
                              constraint wwv_remote_srvr_prmpt_inst_ck
                              check ( prompt_on_install in ('Y', 'N' )),
    --
    ords_version              varchar2(255),
    ords_timezone             varchar2(255),
    ords_init_code            varchar2(4000),
    ords_cleanup_code         varchar2(4000),
    --
    remote_sql_database_type  varchar2(6)
                              constraint wwv_remote_srvr_dbtype_ck
                              check( remote_sql_database_type in ( 'ORACLE', 'MYSQL' ) ),
    remote_sql_database_info  varchar2(4000),
    remote_sql_default_schema varchar2(255),
    --
    mysql_sql_modes           varchar2(4000),
    --
    print_server_type         varchar2(3)
                              constraint wwv_remote_srvr_printtype_ck
                              check( print_server_type in ( 'AOP', 'BIP', 'FOP' )),
    server_timeout            number,
    --
    server_comment            varchar2(4000),
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date )
/

alter table wwv_remote_servers add constraint wwv_remote_srvr_ws_name_uk unique ( security_group_id, name );
alter table wwv_remote_servers add constraint wwv_remote_srvr_ws_stid_uk unique ( security_group_id, static_id );

create index wwv_remote_srvr_sgid_fx       on wwv_remote_servers ( security_group_id ) compress;
create index wwv_remote_srvr_credential_fx on wwv_remote_servers ( credential_id )     compress;

--##############################################################################
--#
--# APPLICATION
--#
--##############################################################################
prompt APPLICATION
--==============================================================================
-- id                = flow number
-- security_group_id = identifies company
-- owner             = identifies who the flow is parsed as
-- name              = identifies the logical name of the flow
create table wwv_flows (
    id                            number
                                  constraint wwv_flow_flow_PK
                                  primary key,
    security_group_id             number not null
                                  constraint wwv_flows_fk
                                  references wwv_flow_companies(provisioning_company_id)
                                  on delete cascade,
    alias                         varchar2(255) not null,
    owner                         varchar2(128) not null,
    name                          varchar2(255) not null,
    default_page_template         number,
    allow_feedback_yn             varchar2(1) constraint wwv_flow_allow_feedback_yn_ck
                                  check (allow_feedback_yn in (
                                  'Y',
                                  'N')),
    webdb_logging                 varchar2(30)
                                  constraint wwv_valid_flow_loggingyesno
                                  check (webdb_logging in (
                                  'YES',
                                  'NO')),
    printer_friendly_template     number,
    default_region_template       number,
    default_label_template        number,
    default_report_template       number,
    default_list_template         number,
    default_menu_template         number,
    default_button_template       number,
    default_calendar_template     number,
    error_template                number,
    flow_language                 varchar2(255),
    flow_language_derived_from    varchar2(30)
                                  constraint wwv_valid_flow_lang_from
                                  check (flow_language_derived_from in (
                                  'FLOW_PRIMARY_LANGUAGE',
                                  'BROWSER',
                                  'ITEM_PREFERENCE',
                                  'FLOW_PREFERENCE',
                                  'SESSION',
                                  '0')),
    charset                       varchar2(255),
    media_type                    varchar2(255),
    date_format                   varchar2(255),
    date_time_format              varchar2(255),
    timestamp_format              varchar2(255),
    timestamp_tz_format           varchar2(255),
    nls_sort                      varchar2(255),
    nls_comp                      varchar2(255)
                                  constraint wwv_flow_nls_comp_ck
                                  check (nls_comp in (
                                  'BINARY',
                                  'LINGUISTIC'
                                  )),
    direction_right_to_left       varchar2(1) default 'N' not null
                                  constraint wwv_flow_direction_ck
                                  check (direction_right_to_left in ('Y','N')),
    flow_image_prefix             varchar2(255),
    --
    max_session_length_sec        number,
    on_max_session_timeout_url    varchar2(4000),
    max_session_idle_sec          number,
    on_max_idle_timeout_url       varchar2(4000),
    session_timeout_warning_sec   number,
    --
    logo_type                     varchar2(2)
                                  constraint wwv_flows_logo_types_ck
                                  check (logo_type in ('T','I','IT', 'C')),
    logo                          varchar2(4000),
    logo_text                     varchar2(4000),
    --
    favicons                      varchar2(4000), -- deprecated
    app_builder_icon_name         varchar2(255)
                                  constraint wwv_flow_valid_appbuildericon
                                  check (not regexp_like(app_builder_icon_name, '\.\.|\/|\\')),
    public_user                   varchar2(255),
    proxy_server                  varchar2(255),
    no_proxy_domains              varchar2(500),
    authentication_id             number,         -- fk to wwv_flow_authentications follows after creation of wwv_flow_authentications
    populate_roles                varchar2(1) default 'R' not null
                                  constraint wwv_flows_populate_roles_ck
                                  check (populate_roles in ('R','A','C')),
    flow_version                  varchar2(255),
    flow_status                   varchar2(30)
                                  constraint wwv_flow_valid_status
                                  check (flow_status in (
                                  'AVAILABLE',
                                  'AVAILABLE_W_EDIT_LINK',
                                  'AVAILABLE_W_INLINE_EDIT',
                                  'DEVELOPERS_ONLY',
                                  'UNAVAILABLE_URL',
                                  'UNAVAILABLE_PLSQL',
                                  'RESTRICTED_ACCESS',
                                  'UNAVAILABLE')),
    flow_unavailable_text         varchar2(4000),
    restrict_to_user_list         varchar2(4000),
    application_tab_set           number,
    exact_substitutions_only      varchar2(1)
                                  constraint wwv_flow_exact_subs_flag
                                  check (exact_substitutions_only in ('Y','N')),
    csv_encoding                  varchar2(1) constraint wwv_flow_csv_encoding
                                  check (csv_encoding in (
                                  'Y',
                                  'N')),
    auto_time_zone                varchar2(1) constraint wwv_flow_auto_time_zone
                                  check (auto_time_zone in (
                                  'Y',
                                  'N')),
    javascript_file_urls          varchar2(4000), /* obsolete, has been moved to wwv_flow_user_interfaces */
    include_legacy_javascript     varchar2(1),    /* obsolete, has been moved to wwv_flow_user_interfaces */
    ui_detection_css_urls         varchar2(4000),
    --
    build_status                  varchar2(30)
                                  constraint wwv_flow_valid_build_status
                                  check (build_status in (
                                  'RUN_AND_BUILD',
                                  'RUN_ONLY',
                                  'RUN_AND_HIDDEN')),
    publish_yn                    varchar2(1),
    friendly_url                  varchar2(1) default 'N'
                                  not null
                                  constraint wwv_flows_friendly_url_ck
                                  check (friendly_url in ('Y', 'N')),
    --
    vpd                           varchar2(4000),
    vpd_teardown_code             varchar2(4000),
    runtime_api_usage             varchar2(5)
                                  constraint wwv_flow_runtime_api_usage_ck
                                       check ( regexp_like(runtime_api_usage, '^[TOW:]+$')),
    security_scheme               varchar2(255),
    authorize_public_pages_yn     varchar2(1) default 'Y' not null
                                  constraint wwv_flow_authz_public_pages_ck
                                  check (authorize_public_pages_yn in ('Y', 'N')),
    authorize_batch_job           varchar2(1) default 'Y' not null
                                  constraint wwv_flow_authz_batch_job_ck
                                  check (authorize_batch_job in ('Y', 'N')),
    rejoin_existing_sessions      varchar2(1) default 'N' not null
                                  constraint wwv_flow_rejoin_es_ck
                                  check (rejoin_existing_sessions in ('Y', 'P', 'N')),
    --
    checksum_salt                 raw(255),
    checksum_salt_last_reset      date,
    bookmark_checksum_function    varchar2(5) not null
                                  constraint wwv_flow_bookmark_cs_fnc_ck
                                  check (bookmark_checksum_function in ('MD5','SH1','SH256','SH384','SH512')),
    accept_old_checksums          varchar2(1)
                                  constraint wwv_flow_accept_old_cksums_ck
                                  check (accept_old_checksums in ( 'Y', 'N' )),
    --
    substitution_string_01        varchar2(255),
    substitution_value_01         varchar2(4000),
    substitution_string_02        varchar2(255),
    substitution_value_02         varchar2(4000),
    substitution_string_03        varchar2(255),
    substitution_value_03         varchar2(4000),
    substitution_string_04        varchar2(255),
    substitution_value_04         varchar2(4000),
    substitution_string_05        varchar2(255),
    substitution_value_05         varchar2(4000),
    substitution_string_06        varchar2(255),
    substitution_value_06         varchar2(4000),
    substitution_string_07        varchar2(255),
    substitution_value_07         varchar2(4000),
    substitution_string_08        varchar2(255),
    substitution_value_08         varchar2(4000),
    substitution_string_09        varchar2(255),
    substitution_value_09         varchar2(4000),
    substitution_string_10        varchar2(255),
    substitution_value_10         varchar2(4000),
    substitution_string_11        varchar2(255),
    substitution_value_11         varchar2(4000),
    substitution_string_12        varchar2(255),
    substitution_value_12         varchar2(4000),
    substitution_string_13        varchar2(255),
    substitution_value_13         varchar2(4000),
    substitution_string_14        varchar2(255),
    substitution_value_14         varchar2(4000),
    substitution_string_15        varchar2(255),
    substitution_value_15         varchar2(4000),
    substitution_string_16        varchar2(255),
    substitution_value_16         varchar2(4000),
    substitution_string_17        varchar2(255),
    substitution_value_17         varchar2(4000),
    substitution_string_18        varchar2(255),
    substitution_value_18         varchar2(4000),
    substitution_string_19        varchar2(255),
    substitution_value_19         varchar2(4000),
    substitution_string_20        varchar2(255),
    substitution_value_20         varchar2(4000),
    --
    last_updated_by               varchar2(255),
    last_updated_on               date,
    -- 4.0
    created_by                    varchar2(255),
    created_on                    date,
    --
    default_chart_template        number,
    default_reportr_template      number,
    default_form_template         number,
    default_wizard_template       number,
    default_tabform_template      number,
    default_menur_template        number,
    default_listr_template        number,
    default_irr_template          number,
    theme_id                      number,
    group_id                      number,
    error_handling_function       varchar2(255),
    default_error_display_location varchar2(40)
                                   constraint wwv_flow_default_edl
                                   check ( default_error_display_location in (
                                               'INLINE_WITH_FIELD',
                                               'INLINE_WITH_FIELD_AND_NOTIFICATION',
                                               'INLINE_IN_NOTIFICATION') ),
    global_notification           varchar2(4000),
    page_protection_enabled_y_n   varchar2(1)
                                    constraint wwv_flow_page_prot_enable_flag
                                    check (page_protection_enabled_y_n in ('Y','N')),
    application_type              varchar2(30) default 'STANDARD'
                                  constraint wwv_flow_app_type
                                  check (application_type in ('STANDARD','TEMPLATE','SAMPLE','TEMPLATE_HIDDEN','THEME')),
    browser_cache                 varchar2(1) default 'Y'
                                  not null
                                  constraint wwv_flows_ck_browser_cache
                                  check (browser_cache in ('Y','N')),
    browser_frame                 varchar2(1) default 'A'
                                  not null
                                  constraint wwv_flows_ck_browser_frame
                                  check (browser_frame in ('D','A','S')),
    referrer_policy               varchar2(31) default 'strict-origin'
                                  not null
                                  constraint wwv_flows_ck_referrer_policy
                                  check ( referrer_policy in (
                                              'no-referrer',
                                              'no-referrer-when-downgrade',
                                              'origin',
                                              'origin-when-cross-origin',
                                              'same-origin',
                                              'strict-origin',
                                              'strict-origin-when-cross-origin',
                                              'unsafe-url' )),
    deep_linking                  varchar2(1) default 'Y'
                                  not null
                                  constraint wwv_flows_ck_deep_linking
                                  check (deep_linking in ('Y','N')),
    compatibility_mode            varchar2(4) default '4.0'
                                  not null
                                  constraint wwv_flows_ck_comp_mode
                                  check (compatibility_mode in ('4.0', '4.1', '4.2', '5.0', '5.1', '19.1', '19.2', '21.2')),
    html_escaping_mode            varchar2(1) default 'E'
                                  not null
                                  constraint wwv_flow_ck_esc_mode
                                  check (html_escaping_mode in ('B', 'E')),
    http_response_headers         varchar2(4000),
    email_from                    varchar2(255),
    oracle_text_function          varchar2(500),
    tokenize_row_search           varchar2(1) default 'N' not null
                                  constraint wwv_flows_ck_tokenizerowsearch
                                  check ( tokenize_row_search in ( 'Y', 'N' ) ),
    import_orig_app_id            number,
    import_orig_security_group_id number,
    import_orig_owner             varchar2(128),
    import_id_offset              number,
    import_orig_release           varchar2(30),
    file_prefix                   varchar2(255),
    files_version                 number default 1 not null,
    --
    shared_components_scn         number default 1 not null,
    --
    print_server_type             varchar2(8) default 'NATIVE'
                                  not null
                                  constraint wwv_flows_ck_print_srv_type
                                  check (print_server_type in ('INSTANCE','NATIVE','REMOTE')),
    print_remote_server_id        number
                                  constraint wwv_flows_prn_srv_fk
                                  references wwv_remote_servers(id)
                                  deferrable initially deferred,
    print_credential_id           number
                                  constraint wwv_flows_prn_cred_fk
                                  references wwv_credentials(id)
                                  deferrable initially deferred,
    --
    is_pwa                        varchar2(1) default 'N' not null
                                  constraint wwv_flows_is_pwa_ck
                                  check (is_pwa in ('Y', 'N')),
    pwa_is_installable            varchar2(1)
                                  constraint wwv_flows_pwa_installable_ck
                                  check (pwa_is_installable in ('Y', 'N')),
    pwa_manifest_description      varchar2(4000),
    pwa_manifest_display          varchar2(10)
                                  constraint wwv_flows_pwa_display_ck
                                  check (pwa_manifest_display in (
                                            'fullscreen',
                                            'standalone',
                                            'minimal-ui',
                                            'browser' )),
    pwa_manifest_orientation      varchar2(19)
                                  constraint wwv_flows_pwa_orientation_ck
                                  check (pwa_manifest_orientation in (
                                            'any',
                                            'natural',
                                            'landscape',
                                            'landscape-primary',
                                            'landscape-secondary',
                                            'portrait',
                                            'portrait-primary',
                                            'portrait-secondary' )),
    pwa_manifest_theme_color      varchar2(128),
    pwa_manifest_background_color varchar2(128),
    pwa_manifest_icon_url         varchar2(255),
    pwa_manifest_custom           varchar2(4000),
    pwa_service_worker_hooks      clob,
    pwa_apple_status_bar_style    varchar2(17)
                                  constraint wwv_flows_pwa_apple_bar_ck
                                  check (pwa_apple_status_bar_style in (
                                            'default',
                                            'black',
                                            'black-translucent' )),
    copyright_banner              varchar2(4000),
    --
    mle_environment               varchar2(255),
    --
    -- obsolete, but needed for wwv_flow_upgrade
    --
    authentication                varchar2(255),
    custom_authentication_process varchar2(4000),
    home_link                     varchar2(2000),
    login_url                     varchar2(2000),
    --
    -- constraints
    --
    constraint wwv_flows_pk_idx    unique(security_group_id, id),
    constraint wwv_flows_alias_idx unique(alias, security_group_id)
    )
/

alter table wwv_flows add constraint wwv_flows_print_server_ck check (
    ( print_server_type = 'REMOTE'
        and print_remote_server_id is not null )
    or ( print_server_type != 'REMOTE'
        and print_remote_server_id is null
        and print_credential_id is null ) ) deferrable initially deferred
/

alter index wwv_flows_pk_idx rebuild compress
/

alter index wwv_flows_alias_idx rebuild compress
/
create index wwv_flows_idx_authentication on wwv_flows (authentication_id)
/
create index wwv_flows_idx_print_server on wwv_flows (print_remote_server_id)
/
create index wwv_flows_idx_print_credential on wwv_flows (print_credential_id)
/

--==============================================================================
prompt ...wwv_flow_application_groups
create table wwv_flow_application_groups (
   id                         number
                              constraint wwv_flow_app_group_pk
                              primary key,
   group_name                 varchar2(255) not null,
   group_comment              varchar2(4000),
   created_on                 date,
   created_by                 varchar2(255),
   updated_on                 date,
   updated_by                 varchar2(255),
   security_group_id          number not null
                              constraint wwv_flow_app_groups_fk
                              references wwv_flow_companies(provisioning_company_id)
                              on delete cascade)
/

create index wwv_flow_app_grp_idx1 on wwv_flow_application_groups (security_group_id);

--==============================================================================
-- This table allows flows developers to reserve an ID and then choose
-- to create a flow or not to create a flow.  This prevents two developers
-- from being able to create the same flow id using the create flow wizard.
--
prompt ...wwv_flows_reserved
create table wwv_flows_reserved (
   id                          number
                               constraint wwv_flow_reserved_pk
                               primary key,
   created_on                  date,
   created_by                  varchar2(255),
   permanent_security_group_id number
   )
/

create index wwv_flows_reserved_idx on wwv_flows_reserved (created_on)
/

--==============================================================================
-- This table allows flows developers to reserve an ID and then choose
-- to create a page or not to create a page.  This prevents two developers
-- from being able to create the same flow page using the create page wizard.
--
prompt ...wwv_flow_pages_reserved

create table wwv_flow_pages_reserved (
   id                number,
   flow_id           number
                     constraint wwv_flow_pages_reserved_fk
                     references wwv_flows
                     on delete cascade,
   flow_session      number,
   created_on        date,
   created_by        varchar2(255)
   )
/

alter table wwv_flow_pages_reserved add constraint wwv_flow_pages_reserved_pk primary key (id, flow_id)
/

create index wwv_flow_pages_reserved_idx on wwv_flow_pages_reserved (created_on)
/

create index wwv_flow_pages_reserved_idx2 on wwv_flow_pages_reserved (flow_id)
/

--==============================================================================
-- Comments
--
create table  wwv_flow_app_comments (
   id                         number constraint wwv_flow_app_comments_pk
                              primary key,
   flow_id                    number not null
                              constraint wwv_flow_app_comments_fk
                              references wwv_flows
                              on delete cascade,
   pages                      varchar2(4000),
   app_comment                varchar2(4000) not null,
   created_on                 date,
   created_by                 varchar2(255),
   updated_on                 date,
   updated_by                 varchar2(255),
   comment_owner              varchar2(255),
   comment_flag               varchar2(4000),
   app_version                varchar2(255),
   security_group_id          number not null)
/

create index wwv_flow_commments_idx1 on wwv_flow_app_comments (flow_id);


--==============================================================================
-- Application Static Files
--
create table wwv_flow_static_files (
    id                        number         not null
                                             constraint wwv_flow_static_files_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_static_files_flow_fk
                                             references wwv_flows on delete cascade,
    file_name                 varchar2(255)  not null,
    mime_type                 varchar2(255)  not null,
    file_charset              varchar2(128),
    file_content              blob,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  );

alter table wwv_flow_static_files
  add constraint wwv_flow_static_files_uk unique (flow_id, file_name);

create index wwv_flow_static_files_idx1 on wwv_flow_static_files (security_group_id);

--==============================================================================
create table wwv_flow_banner (
    id                           number
                                 constraint wwv_flow_banner_fk
                                 references wwv_flows
                                 on delete cascade,
    banner                       clob,
    security_group_id            number not null)
    lob (banner) store as (cache reads enable storage in row)
/
create unique index wwv_flow_banner_u_idx on wwv_flow_banner (id)
/

--##############################################################################
--#
--# WORKSPACE USERS AND GROUPS
--#
--##############################################################################
prompt WORKSPACE USERS AND GROUPS
--==============================================================================
prompt ...wwv_flow_fnd_user
create table wwv_flow_fnd_user (
    user_id                                  number
                                             constraint wwv_flow_fnd_user_pk
                                             primary key,
    security_group_id                        number not null
                                             constraint wwv_flow_fnd_user_fk
                                             references wwv_flow_companies(provisioning_company_id)
                                             on delete cascade,
    user_name                                varchar2(100) not null,
    first_name                               varchar2(255),
    last_name                                varchar2(255),
    --
    creation_date                            date not null,
    created_by                               varchar2(255),
    last_update_date                         date not null,
    last_updated_by                          varchar2(255),
    --
    start_date                               date,
    end_date                                 date,
    person_type                              varchar2(1),
    email_address                            varchar2(240),
    --
    web_password2                            raw(255),
    web_password_version                     varchar2(20) not null
                                             constraint wwv_flow_fnd_user_ck_pwd_ver
                                             check (regexp_like (
                                                        web_password_version,
                                                        '^[135];[12345];' )),
    last_login                               date,
    builder_login_count                      number,
    last_agent                               varchar2(4000),
    last_ip                                  varchar2(4000),
    account_locked                           varchar2(1)
                                             constraint fnd_user_account_locked
                                             check (account_locked in ('Y','N')),
    account_expiry                           date,
    failed_access_attempts                   number,
    last_failed_login                        date,
    first_password_use_occurred              varchar2(1)
                                             constraint fnd_user_first_pw_use
                                             check (first_password_use_occurred in ('Y','N')),
    change_password_on_first_use             varchar2(1)
                                             constraint fnd_user_change_pw_on_first
                                             check (change_password_on_first_use in ('Y','N')),
    --
    allow_app_building_yn                    varchar2(1),
    allow_sql_workshop_yn                    varchar2(1),
    allow_team_development_yn                varchar2(1),
    --
    default_schema                           varchar2(128),
    allow_access_to_schemas                  varchar2(4000),
    --
    -- mostly empty and legacy columns go to the end
    --
    description                              varchar2(240),
    -- old password fields (still needed for on the fly migration on password change)
    web_password                             varchar2(255),
    web_password_raw                         raw(255),
    -- unused password information
    password_date                            date,
    password_accesses_left                   number(15),
    password_lifespan_accesses               number(15),
    password_lifespan_days                   number(15),
    --
    default_date_format                      varchar2(255),
    known_as                                 varchar2(255),
    employee_id                              number(15),
    person_id                                number,
    --
    profile_image                            blob,
    profile_image_name                       varchar2(100),
    profile_mimetype                         varchar2(255),
    profile_filename                         varchar2(255),
    profile_last_update                      date,
    profile_charset                          varchar2(128),
    --
    attribute_01                             varchar2(4000),
    attribute_02                             varchar2(4000),
    attribute_03                             varchar2(4000),
    attribute_04                             varchar2(4000),
    attribute_05                             varchar2(4000),
    attribute_06                             varchar2(4000),
    attribute_07                             varchar2(4000),
    attribute_08                             varchar2(4000),
    attribute_09                             varchar2(4000),
    attribute_10                             varchar2(4000),
    --
    constraint wwv_flow_fnd_user_uq1         unique (security_group_id, user_name)
                                             using index compress 1
)
/

--==============================================================================
-- FLOW DEVELOPERS
--
-- EDIT        edit one or more flows and page builder
-- ADMIN       root priv on flows
-- BROWSE      priv to run the flow that allows database browsing
-- USER_MANAGERpriv to run the cookie based user manager
-- CREATE      create a new flow
-- MONITOR     monitor flow application information
-- DB_MONITOR  monitor database parameters and state
-- SQL         sql workshop
--
prompt ...wwv_flow_developers
create table wwv_flow_developers (
    id                 number
                       constraint wwv_flow_developers_pk
                       primary key,
    user_id            number
                       constraint wwv_flow_dev_fk2
                       references wwv_flow_fnd_user(user_id)
                       on delete cascade,
    userid             varchar2(255) not null,
    flow_id            number,
    email              varchar2(255),
    developer_role     varchar2(4000) not null,
    developer_comments varchar2(4000),
    security_group_id  number not null
                       constraint wwv_flow_dev_fk
                       references wwv_flow_companies(provisioning_company_id)
                       on delete cascade
    )
/
create unique index wwv_flow_developers_pk2 on wwv_flow_developers (userid,flow_id, security_group_id);
create index wwv_flow_developers_id on wwv_flow_developers (user_id);
create index wwv_flow_developers_fkidx on wwv_flow_developers (security_group_id,userid);

--==============================================================================
prompt ...wwv_flow_password_history
create table wwv_flow_password_history (
    id                                       number primary key,
    user_id                                  number
                                             constraint wwv_flow_pwhist_uid_fk
                                             references wwv_flow_fnd_user(user_id)
                                             on delete cascade
                                             initially deferred,
    password                                 raw(255),
    password_version                         varchar2(20),
    created                                  date,
    security_group_id                        number not null
                                             constraint wwv_flow_pwhist_sgid_fk
                                             references wwv_flow_companies(provisioning_company_id)
                                             on delete cascade
    )
/

create index wwv_flow_passwd_history_fkidx1 on wwv_flow_password_history (security_group_id) compress
/
create index wwv_flow_passwd_history_fkidx2 on wwv_flow_password_history (user_id) compress
/

--==============================================================================
prompt ...wwv_flow_fnd_user_groups
create table wwv_flow_fnd_user_groups (
    id                  number
                        constraint fnd_flow_user_group_pk
                        primary key,
    group_name          varchar2(255) not null,
    static_id           varchar2(255) not null,
    flow_id             number
                        constraint wwv_flow_fnd_user_grps_fk_flow
                        references wwv_flows(id)
                        on delete cascade,
    flow_group_name     varchar2(300) generated always as (
                                          case
                                          when flow_id is null then group_name
                                          else 'f'||flow_id||'-'||group_name
                                          end ) virtual,
    flow_static_id      varchar2(300) generated always as (
                                          case
                                          when flow_id is null then static_id
                                          else 'f'||flow_id||'-'||static_id
                                          end ) virtual,
    security_group_id   number not null
                        constraint wwv_flow_fnd_user_grps_fk
                        references wwv_flow_companies(provisioning_company_id)
                        on delete cascade,
    group_desc          varchar2(4000),
    constraint wwv_flow_fnd_user_groups_uk1
        unique (security_group_id, group_name, flow_id)
        using index compress 1,
    constraint wwv_flow_fnd_user_groups_uk2
        unique (security_group_id, static_id, flow_id)
        using index compress 1
)
/

create index wwv_flow_fnd_user_group_ix1 on wwv_flow_fnd_user_groups(flow_id) compress;

--==============================================================================
prompt ...wwv_flow_fnd_group_users
create table wwv_flow_fnd_group_users (
    group_id            number not null
                        constraint wwv_flow_fnd_gu_int_g_fk
                        references wwv_flow_fnd_user_groups (id)
                        on delete cascade,
    user_id             number not null
                        constraint wwv_flow_fnd_gu_int_u_fk
                        references wwv_flow_fnd_user (user_id)
                        on delete cascade,
    security_group_id   number not null,
    constraint wwv_flow_fnd_group_users_pk primary key(group_id, user_id)
) organization index compress 1
/

create index wwv_flow_fnd_group_users_ix1 on wwv_flow_fnd_group_users(user_id) compress;
create index wwv_flow_fnd_group_users_ix2 on wwv_flow_fnd_group_users(security_group_id) compress;

--==============================================================================
prompt ...wwv_flow_fnd_group_groups
create table wwv_flow_fnd_group_groups (
    grantee_group_id    number not null
                        constraint wwv_flow_fnd_group_groups_fk1
                        references wwv_flow_fnd_user_groups(id)
                        on delete cascade,
    granted_group_id    number not null
                        constraint wwv_flow_fnd_group_groups_fk2
                        references wwv_flow_fnd_user_groups(id)
                        on delete cascade,
    security_group_id   number not null,
    constraint wwv_flow_fnd_group_groups_pk primary key(grantee_group_id, granted_group_id)
) organization index compress 1
/
create index wwv_flow_fnd_group_groups_ix1 on wwv_flow_fnd_group_groups(granted_group_id) compress;
create index wwv_flow_fnd_group_groups_ix2 on wwv_flow_fnd_group_groups(security_group_id) compress;

--==============================================================================
prompt ...wwv_flow_acl_group_users
create table wwv_flow_acl_group_users (
    id                             number not null
                                   constraint wwv_flow_acl_group_users_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_flow_acl_group_users_fk1
                                   references wwv_flow_companies(provisioning_company_id)
                                   on delete cascade,
    user_name                      varchar2(255) not null,
    group_id                       number not null
                                   constraint wwv_flow_acl_group_users_fk2
                                   references wwv_flow_fnd_user_groups(id)
                                   on delete cascade,
    --
    created_by          varchar2(255),
    created_on          date,
    last_updated_by     varchar2(255),
    last_updated_on     date,
    --
    constraint wwv_flow_acl_group_users_uk
        unique (security_group_id, user_name, group_id)
)
/
create index wwv_flow_acl_group_users_ix1 on wwv_flow_acl_group_users(group_id) compress;

--==============================================================================
prompt ...wwv_flow_preferences$
create table wwv_flow_preferences$ (
    id                     number
                           constraint wwv_flow_prefs_pk
                           primary key,
    user_id                varchar2(255) not null,
    preference_name        varchar2(255) not null,
    attribute_value        varchar2(4000),
    tenant_id              varchar2(32),
    security_group_id      number not null
                           constraint wwv_flow_prefs_fk
                           references wwv_flow_companies(provisioning_company_id)
                           on delete cascade
    )
/

create index wwv_flow_pref_user_pref_idx on wwv_flow_preferences$ (user_id, preference_name) compress 1;
create index wwv_flow_preferences$_fkidx on wwv_flow_preferences$(security_group_id,user_id,preference_name) compress 2;

--==============================================================================
prompt ...wwv_flow_persistent_auth$

create table wwv_flow_persistent_auth$ (
    id                          number          not null
                                constraint wwv_persistent_auth_pk
                                primary key,
    security_group_id           number          not null,
    user_name                   varchar2(255)   not null,
    client_fingerprint_hash     varchar2(100)   not null,
    fingerprint_version         number          not null,
    cookie_name                 varchar2(4000)  not null,
    cookie_value                varchar2(32)    not null,
    expires_on                  date            not null,
    created_on                  date            not null,
    last_updated_on             date            not null,
    --
    constraint wwv_persistent_auth_un unique (security_group_id, cookie_value, user_name)
        using index
	)
/
create index wwv_flow_persistent_auth$_idx on
    wwv_flow_persistent_auth$(expires_on)
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > DATA PROFILES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > DATA PROFILES

--==============================================================================
create table wwv_flow_data_profiles (
    id                             number
                                   constraint wwv_flow_data_profiles_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_data_profiles_sgid_fk
                                   references wwv_flow_companies( provisioning_company_id ),
    flow_id                        number not null
                                   constraint wwv_data_profiles_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    --
    name                           varchar2(255) not null,
    format                         varchar2(4)   not null
                                   constraint wwv_data_profile_format_ck
                                   check ( format in ( 'XML', 'JSON', 'CSV', 'XLSX' )),
    encoding                       varchar2(255),
    --
    csv_separator                  varchar2(1 CHAR),    -- support multi-byte chars
    csv_enclosed                   varchar2(1 CHAR),    -- support multi-byte chars
    default_xlsx_sheet_name        varchar2(255),
    skip_rows                      number,
    has_header_row                 varchar2(1)
                                   constraint wwv_data_profile_has_header
                                   check (has_header_row in ('Y', 'N')),
    --
    decimal_characters             varchar2(2),
    --
    row_selector                   varchar2(255),
    is_single_row                  varchar2(1)
                                   constraint wwv_data_profile_single_row
                                   check (is_single_row in ('Y', 'N')),
    single_row_selector            varchar2(255),
    xml_namespaces                 varchar2(4000),
    --
    profile_comment                varchar2(4000),
    last_updated_by                varchar2(255),
    last_updated_on                date,
    created_by                     varchar2(255),
    created_on                     date )
/

alter table wwv_flow_data_profiles add constraint wwv_data_profiles_name_uk unique (flow_id, name)
/

create index wwv_data_profiles_flow_fx on wwv_flow_data_profiles (flow_id)           compress
/
create index wwv_data_profiles_sgid_fx on wwv_flow_data_profiles (security_group_id) compress
/

--==============================================================================
create table wwv_flow_data_profile_cols (
    id                             number
                                   constraint wwv_flow_data_prof_cols_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_dataprof_cols_sgid_fk
                                   references wwv_flow_companies(provisioning_company_id),
    flow_id                        number not null
                                   constraint wwv_dataprof_cols_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    --
    data_profile_id                number not null
                                   constraint wwv_dataprof_cols_prof_fk
                                   references wwv_flow_data_profiles
                                   on delete cascade,
    --
    name                           varchar2(128) not null,
    --
    sequence                       number not null,
    is_primary_key                 varchar2(1) not null
                                   constraint wwv_dataprof_col_is_pk_ck
                                   check (is_primary_key in ('Y', 'N')),
    --
    column_type                    varchar2(14)
                                   constraint wwv_dataprof_col_type_ck
                                   check (column_type in ('DATA', 'LOOKUP', 'SQL_EXPRESSION', 'SQL_QUERY')),
    data_type                      varchar2(30) not null
                                   constraint wwv_dataprof_col_datatype_ck
                                   check ( data_type in ( 'NUMBER',
                                                          'VARCHAR2',
                                                          'CLOB',
                                                          -- 'BLOB',
                                                          'DATE',
                                                          'SDO_GEOMETRY',
                                                          'TIMESTAMP',
                                                          'TIMESTAMP WITH TIME ZONE',
                                                          'TIMESTAMP WITH LOCAL TIME ZONE',
                                                          'DOCUMENT_FRAGMENT') ),
    max_length                     number(5),
                                   constraint wwv_dataprof_col_maxlen_ck
                                   check ( max_length > 0 ),
    decimal_char                   varchar2(1 CHAR),
    group_char                     varchar2(1 CHAR),
    format_mask                    varchar2(255),
    has_time_zone                  varchar2(1)
                                   constraint wwv_dataprof_col_has_tz_ck
                                   check (has_time_zone in ('Y', 'N')),
    --
    is_hidden                      varchar2(1) not null
                                   constraint wwv_dataprof_col_hidden_ck
                                   check (is_hidden in ('Y', 'N')),
    is_filterable                  varchar2(1) default 'Y' not null
                                   constraint wwv_dataprof_col_filterable_ck
                                   check (is_filterable in ('Y', 'N')),
    --
    selector_type                  varchar2(6)
                                   constraint wwv_dataprof_col_seltype_ck
                                   check (selector_type in ( 'NAME', 'SEQ', 'REGEXP' ) ),
    selector                       varchar2(255),
    remote_attribute_name          varchar2(255),
    remote_data_type               varchar2(255),
    transform_type                 varchar2(14)
                                   constraint wwv_dataprof_col_trans_type_ck
                                   check (transform_type in ('SQL_EXPRESSION', 'SQL_QUERY', 'LTRIM', 'RTRIM', 'TRIM', 'REPLACE', 'LOWER', 'UPPER', 'NULLIF', 'REGEXP_REPLACE', 'REGEXP_NULLIF' ) ),
    expression1                    varchar2(4000),
    expression2                    varchar2(4000),
    --
    lookup_owner                   varchar2(128),
    lookup_table                   varchar2(128),
    lookup_column1                 varchar2(128),
    lookup_value1                  varchar2(128),
    lookup_column2                 varchar2(128),
    lookup_value2                  varchar2(128),
    lookup_column3                 varchar2(128),
    lookup_value3                  varchar2(128),
    lookup_key_column              varchar2(128),
    lookup_where_clause            varchar2(4000),
    --
    column_comment                 varchar2(4000),
    last_updated_by                varchar2(255),
    last_updated_on                date,
    created_by                     varchar2(255),
    created_on                     date )
/

alter table wwv_flow_data_profile_cols add constraint wwv_dataprof_cols_name_uk unique (data_profile_id, name)
/

alter table wwv_flow_data_profile_cols add constraint wwv_dataprof_col_integrity_ck check (
    ( column_type = 'LOOKUP' and selector is     null and expression1 is null and expression2 is null ) or
    ( column_type = 'DATA'   and selector is not null )                                                 or
    ( column_type in ( 'SQL_QUERY', 'SQL_EXPRESSION' ) and expression1 is not null )
)
/

create index wwv_dataprof_cols_sgid_fx      on wwv_flow_data_profile_cols (security_group_id)         compress
/
create index wwv_dataprof_cols_flow_prof_ix on wwv_flow_data_profile_cols (flow_id, data_profile_id)  compress 1
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > WEB SOURCES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > WEB SOURCES

--==============================================================================
create table wwv_flow_web_src_modules (
    id                             number
                                   constraint wwv_flow_web_src_modules_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_websrc_module_sgid_fk
                                   references wwv_flow_companies(provisioning_company_id),
    flow_id                        number not null
                                   constraint wwv_websrc_module_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    name                           varchar2(255) not null,
    static_id                      varchar2(255) not null,
    --
    web_source_type                varchar2(255) not null
                                   constraint wwv_websrc_module_type_ck
                                   check (substr(web_source_type,1,7) in ('NATIVE_','PLUGIN_')),
    data_profile_id                number not null
                                   constraint wwv_websrc_module_dataprof_fk
                                   references wwv_flow_data_profiles
                                   on delete cascade,
    --
    remote_server_id               number not null
                                   constraint wwv_websrc_module_rserver_fk
                                   references wwv_remote_servers,
    url_path_prefix                varchar2(500) not null,
    --
    auth_remote_server_id          number
                                   constraint wwv_websrc_mod_auth_rserver_fk
                                   references wwv_remote_servers,
    auth_url_path_prefix           varchar2(500),
    credential_id                  number
                                   constraint wwv_websrc_mod_credential_fk
                                   references wwv_credentials (id)
                                   on delete set null,
    --
    timeout                        number,
    pass_ecid                      varchar2(1) not null
                                   constraint wwv_websrc_module_passecid_ck
                                   check( pass_ecid in ( 'Y', 'N' ) ),
    max_requests_timeframe_sec     number,
    max_requests                   number,
    max_requests_exceed_action     varchar2(5),
                                   constraint wwv_websrc_mod_maxreqexcact_ck
                                   check( max_requests_exceed_action in ( 'ERROR', 'WAIT' ) ),
    --
    sync_is_active                 varchar2(1) default 'N' not null
                                   constraint wwv_websrc_mod_syncactive_ck
                                   check( sync_is_active in ( 'Y', 'N' ) ),
    sync_table_owner               varchar2(128),
    sync_table_name                varchar2(128),
    sync_type                      varchar2(8)
                                   constraint wwv_websrc_module_synctype_ck
                                   check (sync_type in ( 'APPEND', 'MERGE', 'REPLACE', 'TRUNCATE' ) ),
    sync_interval                  varchar2(255),
    sync_commit_interval           number,
    sync_max_http_requests         number,
    --
    sync_last_run_timestamp        timestamp with time zone,
    sync_next_run_timestamp        timestamp with time zone,
    --
    reference_id                   number,
    --
    module_comment                 varchar2(4000),
    --
    attribute_01                   varchar2(4000),
    attribute_02                   varchar2(4000),
    attribute_03                   varchar2(4000),
    attribute_04                   varchar2(4000),
    attribute_05                   varchar2(4000),
    attribute_06                   varchar2(4000),
    attribute_07                   varchar2(4000),
    attribute_08                   varchar2(4000),
    attribute_09                   varchar2(4000),
    attribute_10                   varchar2(4000),
    attribute_11                   varchar2(4000),
    attribute_12                   varchar2(4000),
    attribute_13                   varchar2(4000),
    attribute_14                   varchar2(4000),
    attribute_15                   varchar2(4000),
    --
    -- reference to the REST Source catalog. Since these are workspace objects, and are not part of
    -- the application, we cannot reference by ID with a foreign key here.
    catalog_internal_name          varchar2(255),
    catalog_service_name           varchar2(255),
    --
    last_refreshed                 timestamp with time zone,
    catalog_service_version        number,
    --
    last_updated_by                varchar2(255),
    last_updated_on                date,
    created_by                     varchar2(255),
    created_on                     date )
/

alter table wwv_flow_web_src_modules add constraint wwv_websrc_module_name_uk unique (flow_id, name) using index compress 1
/
alter table wwv_flow_web_src_modules add constraint wwv_websrc_module_staticid_uk unique (flow_id, static_id) using index compress 1
/

create index wwv_websrc_module_sgid_fx      on wwv_flow_web_src_modules (security_group_id)     compress
/
create index wwv_websrc_module_dataprof_fx  on wwv_flow_web_src_modules (data_profile_id)       compress
/
create index wwv_websrc_module_rserver_fx   on wwv_flow_web_src_modules (remote_server_id)      compress
/
create index wwv_websrc_module_ref_ix       on wwv_flow_web_src_modules (reference_id)          compress
/
create index wwv_websrc_mod_credential_fx   on wwv_flow_web_src_modules (credential_id)         compress
/
create index wwv_websrc_mod_auth_rserver_fx on wwv_flow_web_src_modules (auth_remote_server_id) compress
/
create index wwv_websrc_mod_nextsync_ts_ix  on wwv_flow_web_src_modules (sync_next_run_timestamp) compress
/
create index wwv_websrc_mod_catalog_ix      on wwv_flow_web_src_modules (catalog_internal_name,catalog_service_name) compress
/

--==============================================================================
create table wwv_flow_web_src_operations (
    id                             number
                                   constraint wwv_flow_web_src_opers_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_websrc_opers_sgid_fk
                                   references wwv_flow_companies(provisioning_company_id),
    flow_id                        number not null
                                   constraint wwv_websrc_opers_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    --
    web_src_module_id              number not null
                                   constraint wwv_websrc_opers_module_fk
                                   references wwv_flow_web_src_modules
                                   on delete cascade,
    data_profile_id                number
                                   constraint wwv_websrc_opers_dataprof_fk
                                   references wwv_flow_data_profiles
                                   on delete cascade,
    --
    operation                      varchar2(255) not null,
    database_operation             varchar2(16)
                                   constraint wwv_websrc_opers_dboper_ck
                                   check (database_operation in ('INSERT', 'UPDATE', 'DELETE', 'FETCH_COLLECTION', 'FETCH_SINGLE_ROW')),
    --
    url_pattern                    varchar2(500) not null,
    -- legacy SOAP Envelope to be stored as Request Body Template
    -- can also be used for custom REST services where we cannot map items to JSON - will use a template / replace mechanism
    -- like in previous versions with SOAP
    request_body_template          clob,
    --
    legacy_ords_fixed_page_size    number,
    force_error_for_http_404       varchar2(1)
                                   constraint wwv_websrc_opers_404_err_ck
                                   check (force_error_for_http_404 in ('Y', 'N')),
    allow_fetch_all_rows           varchar2(1),
                                   constraint wwv_websrc_opers_allrows_ck
                                   check (allow_fetch_all_rows in ('Y', 'N')),
    fetch_all_rows_timeout         number,
    --
    caching                        varchar2(9)
                                   constraint wwv_websrc_opers_cache_ck
                                   check (caching in ( 'ALL_USERS', 'USER', 'SESSION')),
    invalidate_when                varchar2(4000),
    --
    security_scheme                varchar2(255),
    --
    operation_comment              varchar2(4000),
    --
    last_updated_by                varchar2(255),
    last_updated_on                date,
    created_by                     varchar2(255),
    created_on                     date )
/

alter table wwv_flow_web_src_operations add constraint wwv_websrc_opers_oper_uk   unique (flow_id, web_src_module_id, url_pattern, operation)
/

-- for the unique index, map database operations of NULL to the operation ID. That will allow to have non-NULL database
-- operations only once, but multiple NULL database operations.
create unique index wwv_websrc_opers_dboper_uk on wwv_flow_web_src_operations( flow_id, web_src_module_id, coalesce( database_operation, to_char( id ) ) )
/

create index wwv_websrc_opers_sgid_fx     on wwv_flow_web_src_operations (security_group_id) compress
/
create index wwv_websrc_opers_flow_fx     on wwv_flow_web_src_operations (flow_id)           compress
/
create index wwv_websrc_opers_module_fx   on wwv_flow_web_src_operations (web_src_module_id) compress
/
create index wwv_websrc_opers_dataprof_fx on wwv_flow_web_src_operations (data_profile_id)   compress
/

--==============================================================================
create table wwv_flow_web_src_sync_steps (
    id                             number
                                   constraint wwv_flow_web_src_sync_step_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_websrc_syncstep_sgid_fk
                                   references wwv_flow_companies(provisioning_company_id),
    flow_id                        number not null
                                   constraint wwv_websrc_syncstep_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    web_src_module_id              number not null
                                   constraint wwv_websrc_syncstep_module_fk
                                   references wwv_flow_web_src_modules
                                   on delete cascade,
    static_id                      varchar2(255) not null,
    execution_sequence             number not null,
    is_active                      varchar2(1) not null
                                   constraint wwv_websrc_syncstep_active_ck
                                   check( is_active in ( 'Y', 'N' ) ),
    external_filter_expr           varchar2(4000),
    --
    sync_step_comment              varchar2(4000),
    --
    last_updated_by                varchar2(255),
    last_updated_on                date,
    created_by                     varchar2(255),
    created_on                     date )
/

create unique index wwv_websrc_syncstep_statid_uk on wwv_flow_web_src_sync_steps( flow_id, web_src_module_id, static_id ) compress
/
create index wwv_websrc_syncstep_sgid_fx   on wwv_flow_web_src_sync_steps (security_group_id) compress
/
create index wwv_websrc_syncstep_module_fx on wwv_flow_web_src_sync_steps (web_src_module_id) compress
/

--==============================================================================
create table wwv_flow_web_src_params (
    id                             number
                                   constraint wwv_flow_web_src_params_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_websrc_param_sgid_fk
                                   references wwv_flow_companies(provisioning_company_id),
    flow_id                        number not null
                                   constraint wwv_websrc_param_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    --
    web_src_module_id              number
                                   constraint wwv_websrc_param_module_fk
                                   references wwv_flow_web_src_modules
                                   on delete cascade,
    web_src_operation_id           number
                                   constraint wwv_websrc_param_opers_fk
                                   references wwv_flow_web_src_operations
                                   on delete cascade,
    --
    name                           varchar2(255) not null,
    param_type                     varchar2(12)  not null,
                                   constraint wwv_websrc_param_type_ck
                                   check (param_type in ('HEADER', 'QUERY_STRING', 'URL_PATTERN', 'BODY', 'COOKIE' )),
    is_required                    varchar2(1)   not null,
                                   constraint wwv_websrc_param_isreq_ck
                                   check (is_required in ('Y', 'N')),
    direction                      varchar2(6)   not null,
                                   constraint wwv_websrc_param_direction_ck
                                   check (direction in ('IN', 'OUT', 'IN_OUT')),
    --
    value                          varchar2(4000),
    is_static                      varchar2(1) not null,
                                   constraint wwv_websrc_param_isstatic_ck
                                   check (is_static in ('Y', 'N')),
    is_array                       varchar2(1) not null,
                                   constraint wwv_websrc_param_isarray_ck
                                   check (is_array in ('Y', 'N')),
    array_separator_char           varchar2(4),
    omit_when_null                 varchar2(1) default 'N' not null
                                   constraint wwv_websrc_param_omitnull_ck
                                   check (omit_when_null in ('Y', 'N')),
    is_query_param                 varchar2(1) default 'N' not null
                                   constraint wwv_websrc_param_isquery_ck
                                   check (is_query_param in ('Y', 'N')),
    --
    param_comment                  varchar2(4000),
    --
    last_updated_by                varchar2(255),
    last_updated_on                date,
    created_by                     varchar2(255),
    created_on                     date )
/

alter table wwv_flow_web_src_params add constraint wwv_websrc_parameter_uk unique (flow_id, web_src_module_id, web_src_operation_id, direction, name)
/

create index wwv_websrc_param_sgid_fx     on wwv_flow_web_src_params (security_group_id)    compress
/
create index wwv_websrc_param_flow_fx     on wwv_flow_web_src_params (flow_id)              compress
/
create index wwv_websrc_param_module_fx   on wwv_flow_web_src_params (web_src_module_id)    compress
/
create index wwv_websrc_param_opers_fx    on wwv_flow_web_src_params (web_src_operation_id) compress
/

--==============================================================================
create table wwv_flow_content_cache1$(
    id                   number not null
                         constraint wwv_content_cache1_pk
                         primary key,
    security_group_id    number not null,
    flow_id              number not null
                         constraint wwv_content_cache1_flow_fk
                         references wwv_flows
                         on delete cascade,
    session_id           number
                         constraint wwv_content_cache1_session_fk
                         references wwv_flow_sessions$
                         on delete cascade,
    user_name            varchar2(255),
    --
    component_type       number not null,
    component_id         number not null,
    --
    priority             number,
    --
    key                  varchar2(255) not null,
    --
    content_clob         clob,
    content_length       number,
    --
    cached_on            date,
    valid_until          date )
    --
    pctfree 0
    lob (content_clob) store as (
        pctversion 0
        cache
        disable storage in row )
;

alter table wwv_flow_content_cache1$ add constraint wwv_content_cache_record_un unique(
    security_group_id, flow_id, component_id, user_name, session_id, key, valid_until );

create index wwv_contentcache1_flowkey_ix  on wwv_flow_content_cache1$ ( security_group_id, flow_id, key ) compress;

create index wwv_contentcache1_flowcomp_ix on wwv_flow_content_cache1$ ( security_group_id, flow_id, component_id ) compress;

create index wwv_contentcache1_sess_ix     on wwv_flow_content_cache1$ ( session_id );

create index wwv_contentcache1_flow_ix     on wwv_flow_content_cache1$ ( flow_id );


--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > AUTHORIZATION
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > AUTHORIZATION

--==============================================================================
prompt ...create wwv_flow_security_schemes
create table wwv_flow_security_schemes (
   id                         number
                              constraint wwv_flow_sec_scheme_pk
                              primary key,
   security_group_id          number not null,
   flow_id                    number not null
                              constraint wwv_flow_sec_schemes_fk
                              references wwv_flows
                              on delete cascade,
   name                       varchar2(255) not null,
   scheme_type                varchar2(255) not null
                              constraint wwv_flow_sec_schemes_type_ck
                              check (
                                  scheme_type in (
                                    'EXISTS',
                                    'NOT_EXISTS',
                                    'ITEM_IS_NULL',
                                    'ITEM_IS_NOT_NULL',
                                    'FUNCTION_RETURNING_BOOLEAN',
                                    'PREFERENCE_EQ_COND_TEXT',
                                    'PREFERENCE_NOT_EQ_COND_TEXT',
                                    'VAL_OF_ITEM_IN_COND_EQ_COND_TEXT',
                                    'VAL_OF_ITEM_IN_COND_NOT_EQ_COND_TEXT')
                                  or scheme_type like 'NATIVE\_%' escape '\'
                                  or scheme_type like 'PLUGIN\_%' escape '\'),
   --
   attribute_01               varchar2(4000),
   attribute_02               varchar2(4000),
   attribute_03               varchar2(4000),
   attribute_04               varchar2(4000),
   attribute_05               varchar2(4000),
   attribute_06               varchar2(4000),
   attribute_07               varchar2(4000),
   attribute_08               varchar2(4000),
   attribute_09               varchar2(4000),
   attribute_10               varchar2(4000),
   attribute_11               varchar2(4000),
   attribute_12               varchar2(4000),
   attribute_13               varchar2(4000),
   attribute_14               varchar2(4000),
   attribute_15               varchar2(4000),
   --
   -- $$$ scheme+scheme_text are obsolete (replaced by plugin attributes but required for migration)
   scheme                     varchar2(4000),
   scheme_text                varchar2(4000),
   --
   error_message              varchar2(4000),
   caching                    varchar2(20)   not null
                              constraint wwv_flow_sec_scheme_cache check (caching in (
                              'BY_USER_BY_SESSION',
                              'BY_USER_BY_PAGE_VIEW',
                              'BY_COMPONENT',
                              'NOCACHE' )),
   reference_id               number,
   last_updated_by            varchar2(255),
   last_updated_on            date,
   created_by                 varchar2(255),
   created_on                 date,
   comments                   varchar2(4000)
   )
/

create index wwv_flow_security_schem_fkidx on wwv_flow_security_schemes (flow_id,name)
compress 1
/
create index wwv_flow_security_schem_idx1 on wwv_flow_security_schemes (reference_id)
compress
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > BUILD OPTIONS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > BUILD OPTIONS

--==============================================================================
create table wwv_flow_patches (
    id                           number
                                 constraint wwv_flow_patches_pk
                                 primary  key,
    flow_id                      number  not null
                                 constraint wwv_flow_patches_fk
                                 references wwv_flows
                                 on delete cascade,
    patch_name                   varchar2(255) not null,
    patch_status                 varchar2(30)  not null
                                 constraint wwv_flow_patch_valid_status check (patch_status in ('INCLUDE','EXCLUDE')),
    security_group_id            number not null,
    --
    default_on_export            varchar2(30),
    on_upgrade_keep_status       varchar2(1) default 'N' not null
                                 constraint wwv_flow_patch_keep_status_ck check (on_upgrade_keep_status in ('Y', 'N')),
    --
    feature_identifier           varchar2(4000),
    attribute1                   varchar2(255),
    attribute2                   varchar2(255),
    attribute3                   varchar2(255),
    attribute4                   varchar2(255),
    --
    last_updated_by              varchar2(255),
    last_updated_on              date,
     -- 4.0
    created_by                   varchar2(255),
    created_on                   date,
    patch_comment                varchar2(4000)
    )
/

create unique index wwv_flow_patches_fkidx on wwv_flow_patches(flow_id,id)
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > THEMES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > THEMES

--==============================================================================
prompt themes
create table wwv_flow_themes (
     id                          number
                                 constraint wwv_flow_themes_PK
                                 primary key,
     flow_id                     number not null
                                 constraint wwv_flow_themes_2f_fk
                                 references wwv_flows
                                 on delete cascade,
     theme_id                    number not null,
     ui_type_id                  number default 1 not null /* defaults to DESKTOP */
                                 constraint wwv_flow_theme_ui_type_fk
                                 references wwv_flow_ui_types,
     navigation_type             varchar2(1) default 'T' not null
                                 constraint wwv_flow_theme_nav_type
                                      check ( navigation_type in ('T', 'L' ) ),
     nav_bar_type                varchar2(20) default 'NAVBAR' not null
                                 constraint wwv_flow_theme_navbar_type
                                      check ( nav_bar_type in ('NAVBAR', 'LIST' ) ),
     security_group_id           number not null,
     theme_name                  varchar2(255) not null,
     theme_internal_name         varchar2(255)
                                 constraint wwv_flow_theme_int_name_ck
                                      check (upper(theme_internal_name)=theme_internal_name),
     reference_id                number,
     is_locked                   varchar2(1) default 'N' not null
                                 constraint wwv_flow_theme_is_locked_ck
                                      check ( is_locked in ('Y', 'N' ) ),
     default_page_template       number,
     default_dialog_template     number,
     error_template              number,
     printer_friendly_template   number,
     login_template              number,
     default_button_template     number,
     default_region_template     number,
     default_chart_template      number,
     default_form_template       number,
     default_reportr_template    number,
     default_tabform_template    number,
     default_wizard_template     number,
     default_menur_template      number,
     default_listr_template      number,
     default_irr_template        number,
     default_report_template     number,
     default_label_template      number,
     default_menu_template       number,
     default_calendar_template   number,
     default_list_template       number,
     default_nav_list_position   varchar2(20) constraint wwv_flow_themes_nav_pos_ck
                                     check ( default_nav_list_position in ('TOP','SIDE') ),
     default_nav_list_template   number,
     default_top_nav_list_template  number,
     default_side_nav_list_template number,
     default_option_label        number,
     default_required_label      number,
     default_header_template     number,
     default_footer_template     number,
     default_dialogr_template    number,
     default_dialogbtnr_template number,
     breadcrumb_display_point    varchar2(2000),
     sidebar_display_point       varchar2(2000),
     default_page_transition     varchar2(20) default 'NONE' not null
                                 constraint wwv_flow_theme_page_trans_ck
                                      check ( default_page_transition in ('NONE', 'FADE', 'POP', 'FLIP', 'TURN', 'FLOW', 'SLIDEFADE', 'SLIDE', 'SLIDEUP', 'SLIDEDOWN') ),
     default_popup_transition    varchar2(20) default 'NONE' not null
                                 constraint wwv_flow_theme_popup_trans_ck
                                      check ( default_popup_transition in ('NONE', 'FADE', 'POP', 'FLIP', 'TURN', 'FLOW', 'SLIDEFADE', 'SLIDE', 'SLIDEUP', 'SLIDEDOWN') ),
     default_nav_bar_list_template number,
     --
     /* $$$ mobile_* to be removed */
     mobile_login_template       number,
     mobile_page_template        number,
     mobile_region_template      number,
     mobile_list_template        number,
     mobile_report_template      number,
     mobile_calendar_template    number,
     mobile_button_template      number,
     mobile_required_label       number,
     mobile_optional_label       number,
     --
     theme_description           varchar2(4000),
     theme_image                 blob,
     theme_image_mime_type       varchar2(255),
     custom_image                varchar2(1),
     theme_security_group_id     number,
     file_prefix                 varchar2(255),
     files_version               number default 1 not null,
     --
     calendar_icon               varchar2(255),
     calendar_icon_attr          varchar2(4000),
     --
     custom_icon_classes         varchar2(4000),
     custom_library_file_urls    varchar2(4000),
     custom_icon_prefix_class    varchar2(30),
     icon_library                varchar2(255)
                                 constraint wwv_flow_theme_icon_lib_ck
                                 check ( icon_library in ( 'FONTAPEX', 'FONTAPEX_LATEST' ) ),
     javascript_file_urls        varchar2(4000),
     css_file_urls               varchar2(4000),
     --
     last_updated_on             date,
     last_updated_by             varchar2(255),
     -- 4.0
     created_by                  varchar2(255),
     created_on                  date
     )
/

create unique index wwv_flow_themes_idx1 on wwv_flow_themes (flow_id, theme_id)
/

create index wwv_flow_themes_idx2 on wwv_flow_themes (security_group_id, reference_id)
/

create index wwv_flow_themes_idx3 on wwv_flow_themes (ui_type_id)
/

--==============================================================================
create table wwv_flow_theme_styles (
    id                             number         not null
                                                  constraint wwv_flow_theme_style_pk
                                                  primary key,
    flow_id                        number         not null
                                                  constraint wwv_flow_theme_style_flow_fk
                                                  references wwv_flows on delete cascade,
    theme_id                       number         not null, /* no FK, because that's the theme no and not the internal id */
    name                           varchar2(30)   not null,
    css_file_urls                  varchar2(4000),
    css_classes                    varchar2(255),
    is_current                     varchar2(1)    not null
                                                  constraint wwv_flow_theme_style_curr_ck
                                                       check ( is_current in ( 'Y', 'N') ),
    --
    theme_roller_input_file_urls   varchar2(4000),
    theme_roller_config            clob,
    theme_roller_output_file_url   varchar2(4000),
    theme_roller_read_only         varchar2(1)    constraint wwv_flow_theme_style_ro_ck
                                                       check ( theme_roller_read_only in ( 'Y', 'N') ),
    --
    is_public                      varchar2(1)    default 'N' not null
                                                  constraint wwv_flow_theme_style_pu_ck
                                                       check ( is_public in ( 'Y', 'N') ),
    is_accessible                  varchar2(1)    default 'N' not null
                                                  constraint wwv_flow_theme_style_acc_ck
                                                       check ( is_accessible in ( 'Y', 'N') ),
    --
    reference_id                   number,
    component_comment              varchar2(4000),
    security_group_id              number         not null,
    created_by                     varchar2(255),
    created_on                     date,
    last_updated_by                varchar2(255),
    last_updated_on                date
  )
/

alter table wwv_flow_theme_styles add constraint wwv_flow_theme_style_uk unique (flow_id, theme_id, name)
/

create index wwv_flow_theme_style_idx1 on wwv_flow_theme_styles (security_group_id, reference_id)
/

--==============================================================================
create table wwv_flow_theme_files (
    id                        number         not null
                                             constraint wwv_flow_theme_file_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_theme_file_flow_fk
                                             references wwv_flows on delete cascade,
    theme_id                  number         not null, /* no FK, because that's the theme no and not the internal id */
    file_name                 varchar2(255)  not null,
    mime_type                 varchar2(255)  not null,
    file_charset              varchar2(128),
    file_content              blob,
    reference_id              number,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  );

alter table wwv_flow_theme_files
  add constraint wwv_flow_theme_files_uk unique (flow_id, theme_id, file_name);

create index wwv_flow_theme_files_idx1 on wwv_flow_theme_files (security_group_id, reference_id);



--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > TEMPLATES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > TEMPLATES

--==============================================================================
-- P A G E   T E M P L A T E S
--
create table wwv_flow_templates (
    id                            number
                                  constraint wwv_flow_templates_pk
                                  primary key,
    flow_id                       number not null
                                  constraint wwv_flow_templates_fk
                                  references wwv_flows
                                  on delete cascade,
    name                          varchar2(255) not null,
    internal_name                 varchar2(255)
                                              constraint wwv_flow_page_tmpl_int_name_ck
                                                   check (upper(internal_name)=internal_name),
    look                          number,
    is_popup                      varchar2(1) default 'N' not null
                                              constraint wwv_flow_templates_is_popup_ck
                                                   check ( is_popup in ('Y', 'N' ) ),
    javascript_file_urls          varchar2(4000),
    javascript_code               clob,
    javascript_code_onload        clob,
    dialog_js_init_code           varchar2(255),
    dialog_js_close_code          varchar2(255),
    dialog_js_cancel_code         varchar2(255),
    css_file_urls                 varchar2(4000),
    inline_css                    clob,
    header_template               clob,
    footer_template               clob,
    success_message               varchar2(4000),
    --
    current_tab                   varchar2(4000),
    current_tab_font_attr         varchar2(255),
    non_current_tab               varchar2(4000),
    non_current_tab_font_attr     varchar2(255),
    --
    current_image_tab             varchar2(4000),
    non_current_image_tab         varchar2(4000),
    --
    top_current_tab               varchar2(4000),
    top_current_tab_font_attr     varchar2(255),
    top_non_current_tab           varchar2(4000),
    top_non_current_tab_font_attr varchar2(255),
    --
    box                           clob,
    navigation_bar                varchar2(4000),
    navbar_entry                  varchar2(4000),
    body_title                    varchar2(4000),
    message                       varchar2(4000),
    attribute1                    varchar2(4000),
    attribute2                    varchar2(4000),
    attribute3                    varchar2(4000),
    attribute4                    varchar2(4000),
    attribute5                    varchar2(4000),
    attribute6                    varchar2(4000),
    default_button_position       varchar2(30),
    --
    table_bgcolor                 varchar2(255),
    heading_bgcolor               varchar2(255),
    table_cattributes             varchar2(255),
    font_size                     varchar2(255),
    font_face                     varchar2(255),
    --
    region_table_cattributes      varchar2(255),
    --
    error_page_template           varchar2(4000),
    reference_id                  number,
    --
    breadcrumb_def_reg_pos        varchar2(255),
    sidebar_def_reg_pos           varchar2(255),
    --
    grid_type                     varchar2(8)    default 'TABLE' not null
                                                 constraint wwv_tmpl_grid_type_ck
                                                 check (grid_type in ('TABLE', 'FIXED', 'VARIABLE')),
    grid_max_columns              number(2)
                                                 constraint wwv_tmpl_grid_max_col_ck
                                                 check (grid_max_columns between 1 and 99),
    grid_always_use_max_columns   varchar2(1)
                                                 constraint wwv_tmpl_grid_always_max_ck
                                                 check ( grid_always_use_max_columns in ( 'Y', 'N') ),
    grid_has_column_span          varchar2(1)
                                                 constraint wwv_tmpl_grid_colspan_ck
                                                 check ( grid_has_column_span in ( 'Y', 'N') ),
    grid_always_emit              varchar2(1)
                                                 constraint wwv_tmpl_grid_always_emit_ck
                                                 check ( grid_always_emit in ( 'Y', 'N') ),
    grid_emit_empty_leading_cols  varchar2(1)
                                                 constraint wwv_tmpl_grid_leading_cols_ck
                                                 check ( grid_emit_empty_leading_cols in ( 'Y', 'N') ),
    grid_emit_empty_trailing_cols varchar2(1)
                                                 constraint wwv_tmpl_grid_trail_cols_ck
                                                 check ( grid_emit_empty_trailing_cols in ( 'Y', 'N') ),
    grid_default_label_col_span   number
                                                 constraint wwv_tmpl_grid_label_colspan_ck
                                                 check (grid_default_label_col_span between 1 and 99),
    grid_template                 varchar2(4000),
    grid_row_template             varchar2(4000),
    grid_column_template          varchar2(4000),
    grid_first_column_attributes  varchar2(255),
    grid_last_column_attributes   varchar2(255),
    grid_javascript_debug_code    varchar2(4000),
    --
    dialog_width                  varchar2(20),
    dialog_max_width              varchar2(20),
    dialog_height                 varchar2(20),
    dialog_css_classes            varchar2(255),
    dialog_browser_frame          varchar2(20)
                                                 constraint wwv_tmpl_dialog_frame_ck
                                                 check ( regexp_like( dialog_browser_frame, '^((^|:)(MODAL|NON_MODAL))*$' )),
    --
    required_patch                number,
    security_group_id             number not null,
    --
    theme_id                      number,
    theme_class_id                number,
    default_template_options      varchar2(255),
    preset_template_options       varchar2(255),
    --
    last_updated_by               varchar2(255),
    last_updated_on               date,
    -- 4.0
    created_by                    varchar2(255),
    created_on                    date,
    translate_this_template       varchar2(1)
                                    constraint wwv_flow_trans_temp
                                    check (translate_this_template in ('Y','N')),
    template_comment              varchar2(4000),
    -- $$$ remove
    mobile_page_template          varchar2(1)
                                    constraint wwv_flow_mobile_temp
                                    check (mobile_page_template in ('Y','N')),
    --
    app_tab_before_tabs           varchar2(4000), -- obsolete
    app_tab_current_tab           varchar2(4000), -- obsolete
    app_tab_non_current_tab       varchar2(4000), -- obsolete
    app_tab_after_tabs            varchar2(4000), -- obsolete
    --
    has_edit_links                varchar2(1) /* Deprecated */
)   lob (header_template, footer_template, box) store as (cache reads enable storage in row)
/

create index wwv_flow_template_idx1 on wwv_flow_templates (security_group_id, reference_id) compress
/
create unique index wwv_flow_template_idx2 on wwv_flow_templates (flow_id, theme_id, name) compress
/


--==============================================================================
create table wwv_flow_page_tmpl_disp_points (
    id                        number         not null
                                             constraint wwv_page_tmpl_dp_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_page_tmpl_dp_fk
                                             references wwv_flows on delete cascade,
    page_template_id          number         not null
                                             constraint wwv_page_tmpl_dp_parent_fk
                                             references wwv_flow_templates on delete cascade,
    name                      varchar2(30)   not null,
    placeholder               varchar2(30)   not null
                                             constraint wwv_page_tmpl_dp_upper_ck
                                                  check ( placeholder = upper(placeholder) ),
    has_grid_support          varchar2(1)    not null
                                             constraint wwv_page_tmpl_dp_grid_ck
                                                  check ( has_grid_support in ( 'Y', 'N') ),
    has_region_support        varchar2(1)    default 'Y' not null
                                             constraint wwv_page_tmpl_dp_region_ck
                                                  check ( has_region_support in ( 'Y', 'N') ),
    has_item_support          varchar2(1)    default 'N' not null
                                             constraint wwv_page_tmpl_dp_item_ck
                                                  check ( has_item_support in ( 'Y', 'N') ),
    has_button_support        varchar2(1)    default 'N' not null
                                             constraint wwv_page_tmpl_dp_button_ck
                                                  check ( has_button_support in ( 'Y', 'N') ),
    glv_new_row               varchar2(1)    default 'Y' not null
                                             constraint wwv_page_tmpl_dp_glv_nr_ck
                                                  check ( glv_new_row in ( 'Y', 'N' )),
    max_fixed_grid_columns    number,
    component_comment         varchar2(4000),
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  )
/

alter table wwv_flow_page_tmpl_disp_points
  add constraint wwv_page_tmpl_dp_uk1 unique (page_template_id, placeholder);

alter table wwv_flow_page_tmpl_disp_points
  add constraint wwv_page_tmpl_dp_placeholdr_ck
  check ( placeholder not in (
    'AFTER_HEADER',
    'BEFORE_FOOTER',
    'BODY_1',
    'BODY_2',
    'BODY_3',
    'LEGACY_ORPHAN_COMPONENTS',
    'APEX_CSS',
    'APEX_JAVASCRIPT',
    'APP_VERSION',
    'APPLICATION_CSS',
    'APPLICATION_JAVASCRIPT',
    'APPLICATION_TABS',
    'BODY_TITLE',
    'BUILT_WITH_LOVE_USING_APEX',
    'CLOSE_NOTIFICATION',
    'CUSTOMIZE',
    'DEVELOPER_TOOLBAR',
    'DIALOG_CSS_CLASSES',
    'ENVIRONMENT_BANNER',
    'ERROR_MESSAGE_HEADING',
    'EXPAND_COLLAPSE_NAV_LABEL',
    'EXPAND_COLLAPSE_SIDE_COL_LABEL',
    'FAVICONS',
    'FLOW_NAME',
    'FLOW_VERSION',
    'FORM_CLOSE',
    'FORM_OPEN',
    'GENERATED_CSS',
    'GENERATED_JAVASCRIPT',
    'GLOBAL_NOTIFICATION',
    'HIGH_CONTRAST_TOGGLE',
    'HOME_LINK',
    'LANGUAGE_SELECTOR',
    'LOGIN_URL',
    'LOGO',
    'LOGOUT_URL',
    'MESSAGE',
    'NAVIGATION_BAR',
    'NOTIFICATION_MESSAGE',
    'ONLOAD',
    'OUTDATED_BROWSER',
    'PAGE_CSS',
    'PAGE_CSS_CLASSES',
    'PAGE_JAVASCRIPT',
    'PAGE_STATIC_ID',
    'PAGE_TRANSITION',
    'PARENT_TAB_CELLS',
    'POPUP_TRANSITION',
    'REQUESTED_URL',
    'RTL_CLASS',
    'SCREEN_READER_TOGGLE',
    'SIDE_GLOBAL_NAVIGATION_LIST',
    'SUCCESS_MESSAGE',
    'SUCCESS_MESSAGE_HEADING',
    'TAB_CELLS',
    'TEMPLATE_CSS',
    'TEMPLATE_JAVASCRIPT',
    'TEXT_DIRECTION',
    'THEME_CSS',
    'THEME_JAVASCRIPT',
    'THEME_STYLE_CSS',
    'TITLE',
    'TOP_GLOBAL_NAVIGATION_LIST',
    'WELCOME_USER' ) );

create index wwv_page_tmpl_dp_flow_fx on wwv_flow_page_tmpl_disp_points (flow_id);

--==============================================================================
-- F I E L D   T E M P L A T E S
create table wwv_flow_field_templates (
    id                            number
                                    constraint wwv_flow_field_template_pk
                                        primary key,
    flow_id                       number not null
                                    constraint wwv_flow_field_temp_f_fk
                                        references wwv_flows on delete cascade,
    security_group_id             number not null,
    --
    theme_id                      number,
    theme_class_id                number,
    default_template_options      varchar2(255),
    preset_template_options       varchar2(255),
    --
    template_name                 varchar2(255) not null,
    internal_name                 varchar2(255)
                                        constraint wwv_flow_fld_tmpl_int_name_ck
                                            check (upper(internal_name)=internal_name),
    template_body1                varchar2(4000),
    template_body2                varchar2(4000),
    before_item                   varchar2(4000),
    after_item                    varchar2(4000),
    item_pre_text                 varchar2(4000),
    item_post_text                varchar2(4000),
    before_element                varchar2(4000),
    after_element                 varchar2(4000),
    help_link                     varchar2(4000),
    inline_help_text              varchar2(4000),
    reference_id                  number,
    on_error_before_label         varchar2(4000),
    on_error_after_label          varchar2(4000),
    error_template                varchar2(4000),
    last_updated_on               date,
    last_updated_by               varchar2(255),
    -- 4.0
    created_by                    varchar2(255),
    created_on                    date,
    translate_this_template       varchar2(1)
                                        constraint wwv_flow_field_trans_temp
                                            check (translate_this_template in ('Y','N')),
    template_comment              varchar2(4000)
    )
/

create index wwv_flow_field_temp_idx1 on wwv_flow_field_templates (security_group_id, reference_id)
/
create unique index wwv_flow_field_temp_idx2 on wwv_flow_field_templates (flow_id, theme_id, template_name)
/

--==============================================================================
-- P A G E   B U T T O N    T E M P L A T E S
create table wwv_flow_button_templates (
    id                        number
                                constraint wwv_flow_button_temp_pk
                                    primary key,
    flow_id                   number not null
                                constraint wwv_flow_buttont_fk
                                    references wwv_flows
                                    on delete cascade,
    template_name             varchar2(255) not null,
    internal_name             varchar2(255)
                                constraint wwv_flow_bt_tmpl_int_name_ck
                                    check (upper(internal_name)=internal_name),
    template                  clob not null,
    hot_template              clob,
    security_group_id         number not null,
    reference_id              number,
    last_updated_by           varchar2(255),
    last_updated_on           date,
    -- 4.0
    created_by                varchar2(255),
    created_on                date,
    --
    theme_id                  number,
    theme_class_id            number,
    default_template_options  varchar2(255),
    preset_template_options   varchar2(255),
    --
    translate_this_template   varchar2(1)
                                constraint wwv_flow_button_trans_temp
                                    check (translate_this_template in ('Y','N')),
    template_comment          varchar2(4000)
    ) lob (template) store as (cache reads enable storage in row)
/

create index wwv_flow_button_temp_idx1 on wwv_flow_button_templates (security_group_id, reference_id)
/
create unique index wwv_flow_button_temp_idx2 on wwv_flow_button_templates (flow_id, theme_id, template_name)
/

--==============================================================================
-- P A G E   R E G I O N    T E M P L A T E S   (PLUG TEMPLATES)
create table wwv_flow_page_plug_templates (
    id                             number
                                        constraint wwv_flow_plug_temp_pk
                                            primary key,
    flow_id                        number not null
                                        constraint wwv_flow_plug_temp_fk
                                            references wwv_flows
                                            on delete cascade,
    page_plug_template_name        varchar2(255) not null,
    internal_name                  varchar2(255)
                                        constraint wwv_flow_reg_tmpl_int_name_ck
                                            check (upper(internal_name)=internal_name),
    javascript_file_urls           varchar2(4000),
    javascript_code_onload         varchar2(4000),
    css_file_urls                  varchar2(4000),
    layout                         varchar2(10) default 'TABLE' not null
                                        constraint wwv_flow_plug_temp_layout_ck
                                            check ( layout in ('TABLE', 'FIELD_TEMPLATE', 'FLEXIBLE' ) ),
/* $$$ render_form_items_in_table should be removed */
    render_form_items_in_table     varchar2(1) default 'Y'
                                     constraint page_plug_templ_form_as_table
                                     check (render_form_items_in_table in ('Y','N')),
    template                       clob not null,
    template2                      clob,
    template3                      clob,
    sub_plug_header_template       varchar2(4000),
    sub_plug_header_entry_template varchar2(4000),
    sub_plug_template              varchar2(4000),
    plug_table_bgcolor             varchar2(255),
    plug_heading_bgcolor           varchar2(255),
    plug_font_size                 varchar2(255),
    image_template                 varchar2(4000),
    default_field_template_id      number
                                        constraint wwv_flow_plug_temp_field_fk
                                            references wwv_flow_field_templates (id),
    default_req_field_template_id  number
                                        constraint wwv_flow_plug_temp_req_fld_fk
                                            references wwv_flow_field_templates (id),
    default_label_alignment        varchar2(30) default 'RIGHT'
                                        constraint wwv_flow_plug_temp_label_a_ck
                                                check (default_label_alignment in ( 'LEFT',
                                                                                    'RIGHT',
                                                                                    'CENTER',
                                                                                    'ABOVE',
                                                                                    'BELOW',
                                                                                    'LEFT-TOP',
                                                                                    'RIGHT-TOP',
                                                                                    'CENTER-TOP',
                                                                                    'LEFT-BOTTOM',
                                                                                    'RIGHT-BOTTOM',
                                                                                    'CENTER-BOTTOM',
                                                                                    'LEFT-CENTER',
                                                                                    'RIGHT-CENTER',
                                                                                    'CENTER-CENTER'
                                                                            )),
    default_field_alignment        varchar2(30) default 'LEFT'
                                        constraint wwv_flow_plug_temp_field_a_ck
                                            check (default_field_alignment in ( 'LEFT',
                                                                                'RIGHT',
                                                                                'CENTER',
                                                                                'LEFT-TOP',
                                                                                'RIGHT-TOP',
                                                                                'CENTER-TOP',
                                                                                'LEFT-BOTTOM',
                                                                                'RIGHT-BOTTOM',
                                                                                'CENTER-BOTTOM',
                                                                                'LEFT-CENTER',
                                                                                'RIGHT-CENTER',
                                                                                'CENTER-CENTER'
                                                                            )),
    default_button_template_id     number
                                        constraint wwv_flow_plug_temp_button_fk
                                            references wwv_flow_button_templates (id),
    default_button_position        varchar2(30),
    security_group_id              number not null,
    reference_id                   number,
    last_updated_by                varchar2(255),
    last_updated_on                date,
    -- 4.0
    created_by                     varchar2(255),
    created_on                     date,
    --
    theme_id                       number,
    theme_class_id                 number,
    default_template_options       varchar2(255),
    preset_template_options        varchar2(255),
    --
    translate_this_template        varchar2(1)
                                        constraint wwv_flow_region_trans_temp
                                            check (translate_this_template in ('Y','N')),
    template_comment               varchar2(4000)
    ) lob (template,template2,template3) store as (cache reads enable storage in row)
/

create index wwv_flow_page_plug_idx1 on wwv_flow_page_plug_templates (security_group_id, reference_id)
/
create unique index wwv_flow_page_plug_idx2 on wwv_flow_page_plug_templates (flow_id, theme_id, page_plug_template_name)
/
create unique index wwv_flow_page_plug_idx3 on wwv_flow_page_plug_templates (default_field_template_id)
/
create unique index wwv_flow_page_plug_idx4 on wwv_flow_page_plug_templates (default_req_field_template_id)
/
create unique index wwv_flow_page_plug_idx5 on wwv_flow_page_plug_templates (default_button_template_id)
/

--==============================================================================
create table wwv_flow_plug_tmpl_disp_points (
    id                        number         not null
                                             constraint wwv_plug_tmpl_dp_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_plug_tmpl_dp_fk
                                             references wwv_flows on delete cascade,
    plug_template_id          number         not null
                                             constraint wwv_plug_tmpl_dp_parent_fk
                                             references wwv_flow_page_plug_templates on delete cascade,
    name                      varchar2(30)   not null,
    placeholder               varchar2(30)   not null
                                             constraint wwv_plug_tmpl_dp_upper_ck
                                                  check ( placeholder = upper(placeholder) ),
    has_grid_support          varchar2(1)    not null
                                             constraint wwv_plug_tmpl_dp_grid_ck
                                                  check ( has_grid_support in ( 'Y', 'N') ),
    has_region_support        varchar2(1)    default 'N' not null
                                             constraint wwv_plug_tmpl_dp_region_ck
                                                  check ( has_region_support in ( 'Y', 'N') ),
    has_item_support          varchar2(1)    default 'Y' not null
                                             constraint wwv_plug_tmpl_dp_item_ck
                                                  check ( has_item_support in ( 'Y', 'N') ),
    has_button_support        varchar2(1)    default 'Y' not null
                                             constraint wwv_plug_tmpl_dp_button_ck
                                                  check ( has_button_support in ( 'Y', 'N') ),
    glv_new_row               varchar2(1)    default 'Y' not null
                                             constraint wwv_plug_tmpl_dp_glv_nr_ck
                                                  check ( glv_new_row in ( 'Y', 'N' )),
    max_fixed_grid_columns    number,
    component_comment         varchar2(4000),
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  )
/

alter table wwv_flow_plug_tmpl_disp_points
  add constraint wwv_plug_tmpl_dp_uk1 unique (plug_template_id, placeholder);

alter table wwv_flow_plug_tmpl_disp_points
  add constraint wwv_plug_tmpl_dp_placeholdr_ck
  check ( placeholder not in (
    'TOP',
    'TOP_AND_BOTTOM',
    'ABOVE_BOX',
    'TEMPLATE_DEFAULT',
    'BELOW_BOX',
    'BOTTOM',
    'RIGHT_OF_IR_SEARCH_BAR',
    'LEFT_OF_TITLE',
    'RIGHT_OF_TITLE',
    'ENTRIES',
    'FIRST_ROW_FETCHED',
    'FORM_CLOSE',
    'FORM_OPEN',
    'ICON_CSS_CLASSES',
    'INTERNET_PAGINATION',
    'LAST_ROW_FETCHED',
    'PAGE_TITLE',
    'REGION_ATTRIBUTES',
    'REGION_CSS_CLASSES',
    'REGION_FOOTER',
    'REGION_HEADER',
    'REGION_ID',
    'REGION_SEQUENCE_ID',
    'REGION_STATIC_ID',
    'ROW_RANGES',
    'ROW_RANGES_IN_SELECT_LIST',
    'ROWS_FETCHED',
    'SUB_REGION_HEADERS',
    'SUB_REGION_ICON_CSS_CLASSES',
    'SUB_REGION_ID',
    'SUB_REGION_TITLE',
    'TIMING',
    'TITLE',
    'TOTAL_ROWS' ) );

create index wwv_plug_tmpl_dp_flow_fx on wwv_flow_plug_tmpl_disp_points (flow_id);

--==============================================================================
-- R E P O R T   T E M P L A T E S
create table wwv_flow_row_templates (
    id                            number
                                    constraint wwv_flow_row_template_pk
                                        primary key,
    flow_id                       number not null
                                    constraint wwv_flow_row_template_fk
                                        references wwv_flows
                                        on delete cascade,
    row_template_name             varchar2(255) not null,
    internal_name                 varchar2(255)
                                    constraint wwv_flow_row_tmpl_int_name_ck
                                        check (upper(internal_name)=internal_name),
    row_template_type             varchar2(30)
                                    constraint wwv_flow_row_template_ty
                                        check (row_template_type in (
                                        'GENERIC_COLUMNS',
                                        'NAMED_COLUMNS')),
    javascript_file_urls          varchar2(4000),
    javascript_code_onload        varchar2(4000),
    css_file_urls                 varchar2(4000),
    before_column_heading         varchar2(4000),
    column_heading_template       varchar2(4000),
    column_heading_sort_asc_temp  varchar2(4000),
    column_heading_sort_desc_temp varchar2(4000),
    column_heading_sort_temp      varchar2(4000),
    after_column_heading          varchar2(4000),
    row_template1                 clob,
    row_template_condition1       varchar2(4000),
    row_template_display_cond1    varchar2(30),
    row_template2                 clob,
    row_template_condition2       varchar2(4000),
    row_template_display_cond2    varchar2(30),
    row_template3                 clob,
    row_template_condition3       varchar2(4000),
    row_template_display_cond3    varchar2(30),
    row_template4                 clob,
    row_template_condition4       varchar2(4000),
    row_template_display_cond4    varchar2(30),
    row_template_before_rows      varchar2(4000),
    row_template_after_rows       varchar2(4000),
    ROW_TEMPLATE_BEFORE_FIRST     VARCHAR2 (4000),
    ROW_TEMPLATE_AFTER_LAST       VARCHAR2 (4000),
    row_template_table_attributes varchar2(4000),
    --
    pagination_template           varchar2(4000),
    next_page_template            varchar2(4000),
    previous_page_template        varchar2(4000),
    next_set_template             varchar2(4000),
    previous_set_template         varchar2(4000),
    --
    row_style_mouse_over          varchar2(255),
    row_style_mouse_out           varchar2(255),
    row_style_checked             varchar2(255),
    row_style_unchecked           varchar2(255),
    --
    reference_id                  number,
    security_group_id             number not null,
    last_updated_by               varchar2(255),
    last_updated_on               date,
    -- 4.0
    created_by                   varchar2(255),
    created_on                   date,
    --
    theme_id                      number,
    theme_class_id                number,
    default_template_options      varchar2(255),
    preset_template_options       varchar2(255),
    --
    translate_this_template       varchar2(1)
                                        constraint wwv_flow_report_trans_temp
                                            check (translate_this_template in ('Y','N')),
    row_template_comment          varchar2(4000)
    ) lob (row_template1,row_template2,row_template3,row_template4) store as (cache reads enable storage in row)
/



create index wwv_flow_row_templates_idx1 on wwv_flow_row_templates (security_group_id, reference_id)
/
create unique index wwv_flow_row_templates_idx2 on wwv_flow_row_templates (flow_id, theme_id, row_template_name)
/

--==============================================================================
prompt ...wwv_flow_list_templates
create table wwv_flow_list_templates (
    id                            number
                                    constraint wwv_flow_list_template_pk
                                        primary key,
    flow_id                       number not null
                                    constraint wwv_flow_list_template_fk
                                        references wwv_flows
                                        on delete cascade,
    list_template_name            varchar2(255) not null,
    internal_name                 varchar2(255)
                                    constraint wwv_flow_lst_tmpl_int_name_ck
                                        check (upper(internal_name)=internal_name),
    --
    javascript_file_urls          varchar2(4000),
    javascript_code_onload        varchar2(4000),
    css_file_urls                 varchar2(4000),
    inline_css                    varchar2(4000),
    --
    list_template_current         clob,
    list_template_noncurrent      clob,
    first_list_template_noncurrent      varchar2(4000), -- apex 4.0
    first_list_template_current         varchar2(4000), -- apex 4.0
    --
    list_template_before_rows     varchar2(4000),
    list_template_after_rows      varchar2(4000),
    --
    between_items                 varchar2(4000),
    before_sub_list               varchar2(4000),
    after_sub_list                varchar2(4000),
    between_sub_list_items        varchar2(4000),
    sub_list_item_current         clob,
    sub_list_item_noncurrent      clob,
    --
    item_template_curr_w_child     clob,
    item_template_noncurr_w_child  clob,
    f_item_template_curr_w_child   varchar2(4000), -- apex 4.0
    fitem_template_noncurr_w_child varchar2(4000), -- apex 4.0
    --
    sub_template_curr_w_child     clob,
    sub_template_noncurr_w_child  clob,
    --
    a01_label                     varchar2(4000),
    a02_label                     varchar2(4000),
    a03_label                     varchar2(4000),
    a04_label                     varchar2(4000),
    a05_label                     varchar2(4000),
    a06_label                     varchar2(4000),
    a07_label                     varchar2(4000),
    a08_label                     varchar2(4000),
    a09_label                     varchar2(4000),
    a10_label                     varchar2(4000),
    a11_label                     varchar2(4000),
    a12_label                     varchar2(4000),
    a13_label                     varchar2(4000),
    a14_label                     varchar2(4000),
    a15_label                     varchar2(4000),
    a16_label                     varchar2(4000),
    a17_label                     varchar2(4000),
    a18_label                     varchar2(4000),
    a19_label                     varchar2(4000),
    a20_label                     varchar2(4000),
    --
    reference_id                  number,
    security_group_id             number not null,
    last_updated_by               varchar2(255),
    last_updated_on               date,
    -- 4.0
    created_by                    varchar2(255),
    created_on                    date,
    --
    theme_id                      number,
    theme_class_id                number,
    default_template_options      varchar2(255),
    preset_template_options       varchar2(255),
    --
    translate_this_template       varchar2(1)
                                    constraint wwv_flow_list_trans_temp
                                        check (translate_this_template in ('Y','N')),
    list_template_comment         varchar2(4000)
    ) lob (list_template_current,
           list_template_noncurrent,
           sub_list_item_current,
           sub_list_item_noncurrent,
           item_template_curr_w_child,
           item_template_noncurr_w_child,
           sub_template_curr_w_child,
           sub_template_noncurr_w_child) store as (cache reads enable storage in row)
/

create index wwv_flow_list_templates_idx1 on wwv_flow_list_templates (security_group_id, reference_id)
/
create unique index wwv_flow_list_templates_idx2 on wwv_flow_list_templates (flow_id ,list_template_name, theme_id)
/

--==============================================================================
prompt calendar templates
create table wwv_flow_cal_templates (
   id                      number
                            constraint wwv_flow_cal_templates_pk
                                primary key,
   name                    varchar2(512) not null,
   internal_name           varchar2(255)
                            constraint wwv_flow_cal_tmpl_int_name_ck
                                check (upper(internal_name)=internal_name),
   flow_id                 number
                            constraint wwv_flow_cal_templ_to_flow_fk
                                references wwv_flows
                                on delete cascade,
   security_group_id       number not null,
   last_updated_by         varchar2(255),
   last_updated_on         date,
   -- 4.0
   created_by              varchar2(255),
   created_on              date,
   --
   theme_id                number,
   theme_class_id          number,
   --
   translate_this_template varchar2(1),
   reference_id            number,
   --
   month_title_format      varchar2(4000),
   day_of_week_format      varchar2(4000),
   month_open_format       varchar2(4000),
   month_close_format      varchar2(4000),
   day_title_format        varchar2(4000),
   day_open_format         varchar2(4000),
   day_close_format        varchar2(4000),
   today_open_format       varchar2(4000),
   weekend_title_format    varchar2(4000),
   weekend_open_format     varchar2(4000),
   weekend_close_format    varchar2(4000),
   nonday_title_format     varchar2(4000),
   nonday_open_format      varchar2(4000),
   nonday_close_format     varchar2(4000),
   week_title_format       varchar2(4000), -- used for week headers
   week_open_format        varchar2(4000), -- used for week headers
   week_close_format       varchar2(4000), -- used for week headers
   daily_title_format      varchar2(4000), -- used for daily view header
   daily_open_format       varchar2(4000), -- used for daily view header
   daily_close_format      varchar2(4000), -- used for daily view header
   weekly_title_format     varchar2(4000),
   weekly_day_of_week_format   varchar2(4000),
   weekly_month_open_format    varchar2(4000),
   weekly_month_close_format   varchar2(4000),
   weekly_day_title_format     varchar2(4000),
   weekly_day_open_format      varchar2(4000),
   weekly_day_close_format     varchar2(4000),
   weekly_today_open_format    varchar2(4000),
   weekly_weekend_title_format varchar2(4000),
   weekly_weekend_open_format  varchar2(4000),
   weekly_weekend_close_format varchar2(4000),
   weekly_time_open_format     varchar2(4000),
   weekly_time_close_format    varchar2(4000),
   weekly_time_title_format    varchar2(4000),
   weekly_hour_open_format     varchar2(4000),
   weekly_hour_close_format    varchar2(4000),
   daily_day_of_week_format    varchar2(4000),
   daily_month_title_format    varchar2(4000),
   daily_month_open_format     varchar2(4000),
   daily_month_close_format    varchar2(4000),
   daily_day_title_format      varchar2(4000),
   daily_day_open_format       varchar2(4000),
   daily_day_close_format      varchar2(4000),
   daily_today_open_format     varchar2(4000),
   daily_time_open_format      varchar2(4000),
   daily_time_close_format     varchar2(4000),
   daily_time_title_format     varchar2(4000),
   daily_hour_open_format      varchar2(4000),
   daily_hour_close_format     varchar2(4000),
   custom_month_title_format   varchar2(4000),
   custom_day_of_week_format   varchar2(4000),
   custom_month_open_format    varchar2(4000),
   custom_month_close_format   varchar2(4000),
   custom_week_title_format    varchar2(4000),
   custom_week_open_format     varchar2(4000),
   custom_week_close_format    varchar2(4000),
   custom_day_title_format     varchar2(4000),
   custom_day_open_format      varchar2(4000),
   custom_day_close_format     varchar2(4000),
   custom_today_open_format    varchar2(4000),
   custom_daily_title_format   varchar2(4000),
   custom_daily_open_format    varchar2(4000),
   custom_daily_close_format   varchar2(4000),
   custom_nonday_title_format  varchar2(4000),
   custom_nonday_open_format   varchar2(4000),
   custom_nonday_close_format  varchar2(4000),
   custom_weekend_title_format varchar2(4000),
   custom_weekend_open_format  varchar2(4000),
   custom_weekend_close_format varchar2(4000),
   custom_wk_month_title_format varchar2(4000),
   custom_wk_day_of_week_format varchar2(4000),
   custom_wk_month_open_format  varchar2(4000),
   custom_wk_month_close_format varchar2(4000),
   custom_wk_week_title_format  varchar2(4000),
   custom_wk_week_open_format   varchar2(4000),
   custom_wk_week_close_format  varchar2(4000),
   custom_wk_day_title_format   varchar2(4000),
   custom_wk_day_open_format    varchar2(4000),
   custom_wk_day_close_format   varchar2(4000),
   custom_wk_today_open_format    varchar2(4000),
   custom_wk_weekend_title_format varchar2(4000),
   custom_wk_weekend_open_format  varchar2(4000),
   custom_wk_weekend_close_format varchar2(4000),
   custom_hour_open_format     varchar2(4000),
   custom_hour_close_format    varchar2(4000),
   custom_time_title_format    varchar2(4000),
   custom_time_open_format     varchar2(4000),
   custom_time_close_format    varchar2(4000),
   custom_month_day_height_pix varchar2(20),
   custom_month_day_height_per varchar2(20),
   custom_week_day_width_pix   varchar2(20),
   custom_week_day_width_per   varchar2(20),
   agenda_format               varchar2(4000),
   agenda_past_day_format      varchar2(4000),
   agenda_today_day_format     varchar2(4000),
   agenda_future_day_format    varchar2(4000),
   agenda_past_entry_format    varchar2(4000),
   agenda_today_entry_format   varchar2(4000),
   agenda_future_entry_format  varchar2(4000),
   month_data_format           varchar2(4000),
   month_data_entry_format     varchar2(4000),
   template_comments           varchar2(4000)
   )
/

create index wwv_flow_cal_temp_idx1 on wwv_flow_cal_templates (security_group_id, reference_id)
/
create unique index wwv_flow_cal_temp_idx2 on wwv_flow_cal_templates (flow_id, theme_id, name)
/

--==============================================================================
create table wwv_flow_popup_lov_template (
    id                 number
                       constraint wwv_flow_pk_poplov_temp
                       primary key,
    security_group_id  number not null,
    flow_id            number
                       constraint wwv_flow_fk_poplov_temp
                       references wwv_flows
                       on delete cascade,
    --
    popup_icon         varchar2(4000) default '#APEX_FILES#list.gif',
    popup_icon_attr    varchar2(4000) default 'width=13 height=13',
    popup_icon2        varchar2(4000) default null,
    popup_icon_attr2   varchar2(4000) default null,
    --
    page_name          varchar2(255)  default 'winlov',
    page_title         varchar2(255)  default 'Search Dialog',
    page_html_head     varchar2(4000) default '<link rel=stylesheet href=#APEX_FILES#platform2.css type=text/css>',
    page_body_attr     varchar2(255)  default 'bgcolor=white OnLoad=first_field()',
    before_field_text  varchar2(4000) default ' ',
    after_field_text   varchar2(4000) default '</div><br />',
    page_heading_text  varchar2(4000) default '<link rel=stylesheet href=#APEX_FILES#platform2.css type=text/css><style>a:link { color:#336699; text-decoration:none; padding:2px;} a:visited { color:#336699; text-decoration:none;} a:hover { color:red; text-decoration:underline;} body { font-family:arial; background-color:#ffffff;} </style>',
    page_footer_text   varchar2(4000) default '</center></td></tr></table>',
    --
    before_result_set  varchar2(4000) default '',
    after_result_set   varchar2(4000) default '',
    --
    filter_width       varchar2(255)  default '20',
    filter_max_width   varchar2(255)  default '100',
    filter_text_attr   varchar2(4000) default null,
    --
    find_button_text   varchar2(255)  default 'Search',
    find_button_image  varchar2(255)  default null,
    find_button_attr   varchar2(4000) default '',
    --
    close_button_text  varchar2(255)  default 'Close',
    close_button_image varchar2(255)  default null,
    close_button_attr  varchar2(4000) default '',
    --
    next_button_text   varchar2(255)  default 'Next',
    next_button_image  varchar2(255)  default null,
    next_button_attr   varchar2(4000) default '',
    --
    prev_button_text   varchar2(255)  default 'Previous',
    prev_button_image  varchar2(255)  default null,
    prev_button_attr   varchar2(4000) default '',
    --
    --
    scrollbars         varchar2(255)  default '1',
    resizable          varchar2(255)  default '1',
    width              varchar2(255)  default '400',
    height             varchar2(255)  default '450',
    --
    result_row_x_of_y  varchar2(255)  default 'Row(s) #FIRST_ROW# - #LAST_ROW#',
    result_rows_per_pg number         default 10,
    --
    when_no_data_found_message   varchar2(4000),
    before_first_fetch_message   varchar2(4000),
    minimum_characters_required  number,
    --
    theme_id                      number,
    theme_class_id                number,
    reference_id                  number,
    --
    last_updated_on               date,
    last_updated_by               varchar2(255),
    -- 4.0
    created_by                    varchar2(255),
    created_on                    date,
    translate_this_template       varchar2(1)
                                    constraint wwv_flow_popup_trans_temp
                                    check (translate_this_template in ('Y','N')),
    template_comment   varchar2(4000)
    )
/

create index wwv_flow_poplov_temp_idx1 on wwv_flow_popup_lov_template (security_group_id, reference_id)
/
create unique index wwv_flow_lovt_u_flow_id on wwv_flow_popup_lov_template (flow_id,theme_id)
/

--==============================================================================
create table wwv_flow_menu_templates (
     id                       number
                                constraint wwv_flow_menu_templs_pk
                                    primary key,
     flow_id                  number not null
                                constraint wwv_flow_menus_t_flow_fk
                                    references wwv_flows
                                    on delete cascade,
     name                     varchar2(255) not null,
     internal_name            varchar2(255)
                                constraint wwv_flow_menu_tmpl_int_name_ck
                                        check (upper(internal_name)=internal_name),
     --
     before_first             varchar2(4000),
     current_page_option      varchar2(4000),
     non_current_page_option  varchar2(4000),
     menu_link_attributes     varchar2(4000),
     between_levels           varchar2(4000),
     after_last               varchar2(4000),
     max_levels               number,
     start_with_node          varchar2(255),
     reference_id             number,
     security_group_id        number not null,
     last_updated_by          varchar2(255),
     last_updated_on          date,
     -- 4.0
     created_by                 varchar2(255),
     created_on                 date,
     --
     theme_id                 number,
     theme_class_id           number,
     default_template_options varchar2(255),
     preset_template_options  varchar2(255),
     --
     translate_this_template  varchar2(1)
                                constraint wwv_flow_menu_trans_temp
                                    check (translate_this_template in ('Y','N')),
     template_comments        varchar2(4000)
     )
/

create index wwv_flow_menu_temp_idx1 on wwv_flow_menu_templates (security_group_id, reference_id)
/
create unique index wwv_flow_menu_temp_idx2 on wwv_flow_menu_templates (flow_id, name, theme_id)
/


create table wwv_flow_template_opt_groups (
    id                        number         not null
                                             constraint wwv_flow_temp_opt_grp_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_temp_opt_grp_fk
                                             references wwv_flows on delete cascade,
    theme_id                  number         not null, /* no FK, because that's the theme no and not the internal id */
    name                      varchar2(30)   not null
                                             constraint wwv_flow_temp_opt_grp_name_ck
                                             check (name = upper(name)),
    display_name              varchar2(30)   not null,
    display_sequence          number         not null,
    template_types            varchar2(255)  not null,
    help_text                 varchar2(4000) ,
    null_text                 varchar2(30)   ,
    is_advanced               varchar2(1)    not null
                                             constraint wwv_flow_temp_opt_grp_adv_ck
                                                  check ( is_advanced in ( 'Y', 'N') ),
    reference_id              number,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
)
/

alter table wwv_flow_template_opt_groups add constraint wwv_flow_template_opt_grp_uk unique (flow_id, theme_id, template_types, name)
/

create table wwv_flow_template_options (
    id                        number         not null
                                             constraint wwv_flow_temp_opt_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_temp_opt_fk
                                             references wwv_flows on delete cascade,
    theme_id                  number         not null, /* no FK, because that's the theme no and not the internal id */
    name                      varchar2(30)   not null
                                             constraint wwv_flow_temp_opt_name_ck
                                             check (name = upper(name) and name <> 'DEFAULT' and regexp_like(name, '^[[:alnum:]_-]+$')),
    display_name              varchar2(30)   not null,
    display_sequence          number         not null,
    page_template_id          number         constraint wwv_flow_temp_opt_page_fk
                                             references wwv_flow_templates
                                             on delete cascade,
    region_template_id        number         constraint wwv_flow_temp_opt_region_fk
                                             references wwv_flow_page_plug_templates
                                             on delete cascade,
    report_template_id        number         constraint wwv_flow_temp_opt_report_fk
                                             references wwv_flow_row_templates
                                             on delete cascade,
    breadcrumb_template_id    number         constraint wwv_flow_temp_opt_bc_fk
                                             references wwv_flow_menu_templates
                                             on delete cascade,
    list_template_id          number         constraint wwv_flow_temp_opt_list_fk
                                             references wwv_flow_list_templates
                                             on delete cascade,
    field_template_id         number         constraint wwv_flow_temp_opt_label_fk
                                             references wwv_flow_field_templates
                                             on delete cascade,
    button_template_id        number         constraint wwv_flow_temp_opt_but_fk
                                             references wwv_flow_button_templates
                                             on delete cascade,
    virtual_template_type     varchar2(255)  generated always as (
                                                 case
                                                   when page_template_id       is not null then 'PAGE'
                                                   when region_template_id     is not null then 'REGION'
                                                   when report_template_id     is not null then 'REPORT'
                                                   when breadcrumb_template_id is not null then 'BREADCRUMB'
                                                   when list_template_id       is not null then 'LIST'
                                                   when field_template_id      is not null then 'FIELD'
                                                   when button_template_id     is not null then 'BUTTON'
                                                   else template_types
                                                 end ) virtual,
    virtual_template_id       number         generated always as (
                                                 coalesce(
                                                     page_template_id, region_template_id, report_template_id,
                                                     breadcrumb_template_id, list_template_id,
                                                     field_template_id, button_template_id )) virtual,
    css_classes               varchar2(255)  not null,
    group_id                  number         constraint wwv_flow_temp_opt_opt_grp_fk
                                             references wwv_flow_template_opt_groups on delete set null,
    template_types            varchar2(255)  ,
    help_text                 varchar2(4000) ,
    is_advanced               varchar2(1)    constraint wwv_flow_temp_opt_adv_ck
                                                  check ( is_advanced in ( 'Y', 'N') ),
    reference_id              number,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  )
/

alter table wwv_flow_template_options add constraint wwv_flow_template_opt_uk unique (flow_id, theme_id, name, virtual_template_type, virtual_template_id)
/

alter table wwv_flow_template_options add constraint wwv_flow_template_opt_uk2 unique (flow_id, theme_id, css_classes, virtual_template_type, virtual_template_id)
/

create index wwv_flow_template_options_idx1 on wwv_flow_template_options (page_template_id)
/

create index wwv_flow_template_options_idx2 on wwv_flow_template_options (region_template_id)
/

create index wwv_flow_template_options_idx3 on wwv_flow_template_options (report_template_id)
/

create index wwv_flow_template_options_idx4 on wwv_flow_template_options (breadcrumb_template_id)
/

create index wwv_flow_template_options_idx5 on wwv_flow_template_options (list_template_id)
/

create index wwv_flow_template_options_idx6 on wwv_flow_template_options (field_template_id)
/

create index wwv_flow_template_options_idx7 on wwv_flow_template_options (button_template_id)
/

create index wwv_flow_template_options_idx8 on wwv_flow_template_options (group_id)
/

create index wwv_flow_template_options_idx9 on wwv_flow_template_options(security_group_id, virtual_template_id)
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > SHORTCUTS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > SHORTCUTS
--==============================================================================
create table wwv_flow_shortcuts (
    id                       number
                             constraint wwv_flow_shortcuts_pk
                             primary key,
    flow_id                  number not null
                             constraint wwv_flow_shortcuts_to_flow_fk
                             references wwv_flows
                             on delete cascade,
    security_group_id        number not null,
    shortcut_name            varchar2(255) not null,
    shortcut_consideration_seq number,
    shortcut_type            varchar2(30)
                             constraint wwv_flow_valid_shortcut_type
                             check (shortcut_type in (
                             'HTML_TEXT',
                             'HTML_TEXT_ESCAPE_SC',
                             'FUNCTION_BODY',
                             'IMAGE',
                             'MESSAGE',
                             'MESSAGE_ESCAPE_JS',
                             'TEXT_ESCAPE_JS')),
    shortcut_language        varchar2(10)
                             constraint wwv_flow_valid_shortcut_lang
                             check (shortcut_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    shortcut                 clob,
    shortcut_condition_type1 varchar2(255), -- obsolete
    shortcut_condition1      varchar2(4000), -- obsolete
    shortcut_condition_type2 varchar2(255), -- obsolete
    shortcut_condition2      varchar2(4000), -- obsolete
    build_option             number,
    error_text               varchar2(4000),
    reference_id             number,
    last_updated_by          varchar2(255),
    last_updated_on          date,
    -- 4.0
    created_by               varchar2(255),
    created_on               date,
    comments                 varchar2(4000),
    --
    constraint wwv_flow_shortcut_idx1 unique (
        flow_id, shortcut_name, shortcut_consideration_seq )
        using index compress 1
    ) lob (shortcut) store as (cache reads enable storage in row)
/

create index wwv_flow_shortcut_idx2 on wwv_flow_shortcuts(security_group_id)
    compress
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > PLUGINS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > PLUGINS

--==============================================================================
create table wwv_flow_plugins (
    id                           number
                                 constraint wwv_flow_plugin_pk
                                 primary key,
    flow_id                      number not null
                                 constraint wwv_flow_plugin_flow_fk
                                 references wwv_flows on delete cascade,
    plugin_type                  varchar2(30)  not null
                                               constraint wwv_flow_plugin_type_ck
                                               check (plugin_type in ( 'ITEM TYPE',
                                                                       'DYNAMIC ACTION',
                                                                       'REGION TYPE',
                                                                       'PROCESS TYPE',
                                                                       'AUTHORIZATION TYPE',
                                                                       'AUTHENTICATION TYPE',
                                                                       'WEB SOURCE TYPE' )),
    name                         varchar2(45)  not null
                                               constraint wwv_flow_plugin_name_ck
                                               check (upper(name)=name),
    display_name                 varchar2(100) not null,
    category                     varchar2(30),
    supported_ui_types           varchar2(255) default 'DESKTOP' not null,
    supported_component_types    varchar2(255) default 'APEX_APPLICATION_PAGE_ITEMS',
    supported_data_types         varchar2(255),
    image_prefix                 varchar2(255),
    javascript_file_urls         varchar2(4000),
    css_file_urls                varchar2(4000),
    --
    plsql_code                   clob,
    api_version                  number        default 1 not null,
    render_function              varchar2(255),
    meta_data_function           varchar2(255),
    emit_value_function          varchar2(255),
    ajax_function                varchar2(255),
    validation_function          varchar2(255),
    execution_function           varchar2(255),
    session_sentry_function      varchar2(255),
    invalid_session_function     varchar2(255),
    authentication_function      varchar2(255),
    post_logout_function         varchar2(255),
    builder_validation_function  varchar2(255),
    migration_function           varchar2(255),
    wsm_capabilities_function    varchar2(255),
    wsm_fetch_function           varchar2(255),
    wsm_dml_function             varchar2(255),
    wsm_execute_function         varchar2(255),
    wsm_discover_function        varchar2(255),
    standard_attributes          varchar2(255),
    substitute_attributes        varchar2(1) default 'Y' not null,
                                 constraint wwv_flow_plugin_subst_attr_ck
                                 check (substitute_attributes in ('Y', 'N')),
    reference_id                 number,
    subscribe_plugin_settings    varchar2(1) default 'Y' not null,
                                 constraint wwv_flow_plugin_settings_ck
                                 check (subscribe_plugin_settings in ('Y', 'N')),
    is_quick_pick                varchar2(1) default 'N' not null
                                             constraint wwv_flow_plugin_is_quickp_ck
                                             check (is_quick_pick in ('Y', 'N')),
    is_deprecated                varchar2(1) default 'N' not null
                                             constraint wwv_flow_plugin_is_deprecat_ck
                                             check (is_deprecated in ('Y', 'N')),
    is_legacy                    varchar2(1) default 'N' not null
                                             constraint wwv_flow_plugin_is_legacy_ck
                                             check (is_legacy in ('Y', 'N')),
    help_text                    varchar2(4000),
    version_identifier           varchar2(30),
    about_url                    varchar2(512),
    plugin_comment               varchar2(4000),
    files_version                number default 1 not null,
    security_group_id            number not null,
    created_by                   varchar2(255),
    created_on                   date,
    last_updated_by              varchar2(255),
    last_updated_on              date,
    --
    -- obsolete columns:
    -- * attribute_XX now stored in wwv_flow_plugin_settings
    --
    attribute_01                 varchar2(4000),
    attribute_02                 varchar2(4000),
    attribute_03                 varchar2(4000),
    attribute_04                 varchar2(4000),
    attribute_05                 varchar2(4000),
    attribute_06                 varchar2(4000),
    attribute_07                 varchar2(4000),
    attribute_08                 varchar2(4000),
    attribute_09                 varchar2(4000),
    attribute_10                 varchar2(4000),
    attribute_11                 varchar2(4000),
    attribute_12                 varchar2(4000),
    attribute_13                 varchar2(4000),
    attribute_14                 varchar2(4000),
    attribute_15                 varchar2(4000),
    -- * these columns are now stored in wwv_flow_plugin_std_attributes
    sql_min_column_count         number(3)      constraint wwv_flow_plugin_mincol_ck
                                                check (sql_min_column_count > 0),
    sql_max_column_count         number(3)      constraint wwv_flow_plugin_maxcol_ck
                                                check (sql_max_column_count > 0),
    sql_examples                 varchar2(4000)
  ) lob (plsql_code) store as (cache reads enable storage in row);

alter  table wwv_flow_plugins add constraint wwv_flow_plugin_uk unique (flow_id, plugin_type, name);
alter  index wwv_flow_plugin_uk rebuild compress 2;
create index wwv_flow_plugins_idx1 on wwv_flow_plugins (reference_id) compress;
comment on table wwv_flow_plugins is
    'Stores the meta data for the plug-ins of an application.';
-- Add comments to the columns
comment on column wwv_flow_plugins.plugin_type is
    'Type of the plugin.';
comment on column wwv_flow_plugins.name is
    'Internal name of the plug-in which is used to reference it for example in wwv_flow_step_items.';
comment on column wwv_flow_plugins.display_name is
    'Contains the name of the plug-in which is displayed on the UI.';
comment on column wwv_flow_plugins.category is
    'Category under which the plug-in should displayed on the UI.';
comment on column wwv_flow_plugins.image_prefix is
    'File prefix which is used by the plug-in to load additional files like CSS, Javascript and images.';
comment on column wwv_flow_plugins.plsql_code is
    'PL/SQL code which contains the logic of the plugin.';
comment on column wwv_flow_plugins.render_function is
    'During rendering of the page this function is called to render the plug-in.';
comment on column wwv_flow_plugins.meta_data_function is
    'Function which is called for plug-in to get dynamic meta data about plug-in.';
comment on column wwv_flow_plugins.ajax_function is
    'Function which is called for plug-in when there is an incoming Ajax call from the browser.';
comment on column wwv_flow_plugins.validation_function is
    'Function which is called to validate the plug-ins data before the defined validations are fired.';
comment on column wwv_flow_plugins.execution_function is
    'Function which is called for plug-ins that are just used on the server side.';
comment on column wwv_flow_plugins.session_sentry_function is
    'Function which is called by an authorization plug-in each time a request is made to Oracle APEX.';
comment on column wwv_flow_plugins.authentication_function is
    'Function which is called by an authentication plug-in to verify login credentials.';
comment on column wwv_flow_plugins.post_logout_function is
    'Function which is called by an authorization plug-in after a session has been marked as terminated.';
comment on column wwv_flow_plugins.builder_validation_function is
    'Function which is called to validate the entered attribute values in the builder.';
comment on column wwv_flow_plugins.migration_function is
    'Function which is called when a new version of a plug-in is installed to migrate existing attribute values.';
comment on column wwv_flow_plugins.standard_attributes is
    'Contains the APEX provided standard attributes which should be displayed for the plug-in.';
comment on column wwv_flow_plugins.sql_min_column_count is
    'Minimum number of columns the SQL query of the LOV or region source has to have.';
comment on column wwv_flow_plugins.sql_max_column_count is
    'Maximum number of columns the SQL query of the LOV or region source can have.';
comment on column wwv_flow_plugins.sql_examples is
    'LOV or region source SQL examples which are displayed for the plug-in in the Builder.';
comment on column wwv_flow_plugins.substitute_attributes is
    'Are the attribute values automatically substituted when the plug-in is called?';
comment on column wwv_flow_plugins.attribute_01 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_02 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_03 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_04 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_05 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_06 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_07 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_08 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_09 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_10 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_11 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_12 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_13 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_14 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.attribute_15 is
    'Dynamic attribute to store additional data';
comment on column wwv_flow_plugins.reference_id is
    'Identifies the master component from which this component is subscribed';
comment on column wwv_flow_plugins.help_text is
    'Help text which is displayed for the plug-in in the Builder.';
comment on column wwv_flow_plugins.version_identifier is
    'Version identifier of the plug-in.';
comment on column wwv_flow_plugins.about_url is
    'URL to get additional information about the plug-in.';

--==============================================================================
create table wwv_flow_plugin_attributes (
    id                          number         not null
                                               constraint wwv_flow_plugin_attr_pk
                                               primary key,
    flow_id                     number         not null
                                               constraint wwv_flow_plugin_attr_flow_fk
                                               references wwv_flows on delete cascade,
    plugin_id                   number         not null
                                               constraint wwv_flow_plugin_attr_plugin_fk
                                               references wwv_flow_plugins on delete cascade,
    attribute_scope             varchar2(15)   not null
                                               constraint wwv_flow_plugin_attr_scope_ck
                                               check (attribute_scope in ('APPLICATION', 'COMPONENT', 'COLUMN')),
    attribute_sequence          number(2)      not null
                                               constraint wwv_flow_plugin_attr_seq_ck
                                               check (attribute_sequence between 1 and 25),
    display_sequence            number         not null,
    prompt                      varchar2(255)  not null,
    attribute_type              varchar2(40)   not null
                                               constraint wwv_flow_plugin_attr_type_ck
                                               check (attribute_type in (
                                                        'CHECKBOX',
                                                        'CHECKBOXES',
                                                        'COLOR',
                                                        'ICON',
                                                        'COMPONENT',
                                                        'LINK',
                                                        'SELECT LIST',
                                                        'TEXT',
                                                        'TEXTAREA',
                                                        'NUMBER',
                                                        'INTEGER',
                                                        'PAGE ITEM',
                                                        'PAGE ITEMS',
                                                        'PAGE NUMBER',
                                                        'PAGE NUMBERS',
                                                        'AUTHORIZATION GROUPS',
                                                        'CREDENTIAL ID',
                                                        'SQL',
                                                        'PLSQL',
                                                        'PLSQL EXPRESSION',
                                                        'PLSQL EXPRESSION BOOLEAN',
                                                        'PLSQL FUNCTION BODY',
                                                        'PLSQL FUNCTION BODY BOOLEAN',
                                                        'CODE LANGUAGE',
                                                        'MLE JAVASCRIPT',
                                                        'MLE JAVASCRIPT EXPRESSION VARCHAR2',
                                                        'MLE JAVASCRIPT EXPRESSION BOOLEAN',
                                                        'MLE JAVASCRIPT FUNCTION BODY VARCHAR2',
                                                        'MLE JAVASCRIPT FUNCTION BODY BOOLEAN',
                                                        'REGION SOURCE COLUMN',
                                                        'DATA LOAD TABLE ID',
                                                        'DATA LOAD DEFINITION ID',
                                                        'EMAIL TEMPLATE ID',
                                                        'OWNER',
                                                        'TABLE',
                                                        'COLUMN',
                                                        'HTML',
                                                        'JAVASCRIPT',
                                                        'PLACEHOLDER JSON',
                                                        'XML',
                                                        'TASK DEFINITION ID')),
    is_required                  varchar2(1)    not null
                                                constraint wwv_flow_plugin_attr_req_ck
                                                check (is_required in ('Y', 'N')),
    is_common                    varchar2(1)    default 'Y'
                                                not null
                                                constraint wwv_flow_plugin_attr_common_ck
                                                check (is_common in ('Y', 'N')),
    show_in_wizard               varchar2(1)    default 'Y'
                                                not null
                                                constraint wwv_flow_plugin_attr_wizard_ck
                                                check (show_in_wizard in ('Y', 'N')),
    default_value                varchar2(4000),
    display_length               number         constraint wwv_flow_plugin_attr_disp_l_ck
                                                check (display_length between 1 and 4000),
    max_length                   number         constraint wwv_flow_plugin_attr_max_l_ck
                                                check (max_length between 1 and 4000),
    min_value                    number,
    max_value                    number,
    unit                         varchar2(30),
    sql_min_column_count         number(3)      constraint wwv_flow_plugin_attr_mincol_ck
                                                check (sql_min_column_count > 0),
    sql_max_column_count         number(3)      constraint wwv_flow_plugin_attr_maxcol_ck
                                                check (sql_max_column_count > 0),
    column_data_types            varchar2(255),
    parent_attribute_id          number         constraint wwv_flow_plugin_attr_parent_fk
                                                references wwv_flow_plugin_attributes on delete set null
                                                deferrable initially deferred,
    lov_type                     varchar2(10)   constraint wwv_flow_plugin_attr_ltype_ck
                                                check (lov_type in (
                                                    'COMPONENT',
                                                    'DISTINCT',
                                                    'STATIC' )),
    lov_component_type_id        number         constraint wwv_flow_plugin_attr_lcompt_fk
                                                /* needed for apex_install_pe_data.sql which deletes all entries */
                                                references wwv_pe_component_types deferrable initially deferred,
    lov_component_type_scope     varchar2(15)   constraint wwv_flow_plugin_attr_lscope_ck
                                                check (lov_component_type_scope in ( 'PARENT', 'PAGE', 'PAGE_AND_GLOBAL', 'THEME' )),
    lov_component_type_on_delete varchar2(15)   constraint wwv_flow_plugin_attr_ldel_ck
                                                check (lov_component_type_on_delete in (
                                                    'WEAK_CASCADE',
                                                    'WEAK_NULL' )),
    text_case                    varchar2(5)    constraint wwv_flow_plugin_attr_case_ck
                                                check ( text_case in (
                                                    'LOWER',
                                                    'UPPER' )),
    null_text                    varchar2(255),
    deprecated_values            varchar2(30),
    supported_ui_types           varchar2(255),
    supported_component_types    varchar2(255),
    is_translatable              varchar2(1)    not null
                                                constraint wwv_flow_plugin_attr_transl_ck
                                                check (is_translatable in ('Y', 'N')),
    depending_on_attribute_id    number         constraint wwv_flow_plugin_attr_depend_fk
                                                references wwv_flow_plugin_attributes on delete set null
                                                deferrable initially deferred,
    depending_on_has_to_exist    varchar2(1)    constraint wwv_flow_plugin_attr_dep_ex_ck
                                                check (depending_on_has_to_exist in ('Y', 'N')),
    depending_on_comp_prop_id    number         constraint wwv_flow_plugin_attr_depen2_fk
                                                references wwv_pe_component_props on delete set null deferrable initially deferred,
    depending_on_condition_type  varchar2(30)   constraint wwv_flow_plugin_attr_cond_ck
                                                check (depending_on_condition_type in (
                                                       'EQUALS',
                                                       'NOT_EQUALS',
                                                       'NULL',
                                                       'NOT_NULL',
                                                       'IN_LIST',
                                                       'NOT_IN_LIST' )),
    depending_on_expression      varchar2(4000),
    reference_scope              varchar2(10)   default 'COMPONENT'
                                                not null
                                                constraint wwv_flow_plugin_attr_refscp_ck
                                                check (reference_scope in ('COMPONENT', 'ROW')),
    examples                     varchar2(4000),
    help_text                    varchar2(4000),
    attribute_comment            varchar2(4000),
    security_group_id            number not null,
    created_by                   varchar2(255),
    created_on                   date,
    last_updated_by              varchar2(255),
    last_updated_on              date,
    attribute_sequence_uk        number generated always as (
                                     /* conditional attributes of native plugs-ins should be able to re-use plug-in attributes */
                                     case
                                       when flow_id between 4411 and 4444 then id
                                       else attribute_sequence
                                     end ) virtual
  );

alter table wwv_flow_plugin_attributes
  add constraint wwv_flow_plugin_attr_uk unique (plugin_id, attribute_scope, attribute_sequence_uk );

create index wwv_flow_plugin_attr_flow_fx on wwv_flow_plugin_attributes (flow_id) compress;
create index wwv_flow_plugin_attr_depend_fx on wwv_flow_plugin_attributes (depending_on_attribute_id);
create index wwv_flow_plugin_attr_depen2_fx on wwv_flow_plugin_attributes (depending_on_comp_prop_id);
create index wwv_flow_plugin_attr_parent_fx on wwv_flow_plugin_attributes (parent_attribute_id);
create index wwv_flow_plugin_attr_compon_fx on wwv_flow_plugin_attributes (lov_component_Type_id) compress;


comment on table wwv_flow_plugin_attributes is
    'Stores the meta data for the dynamic attributes of a plug-in.';
-- Add comments to the columns
comment on column wwv_flow_plugin_attributes.attribute_scope is
    'Is the attribute set for an application or a component.';
comment on column wwv_flow_plugin_attributes.attribute_sequence is
    'Sequence which correlates with the attribute_xx columns for example in wwv_flow_step_items.';
comment on column wwv_flow_plugin_attributes.display_sequence is
    'Sequence in which the attributes are displayed in the Builder.';
comment on column wwv_flow_plugin_attributes.prompt is
    'Prompt which is displayed for that attribute in the Builder.';
comment on column wwv_flow_plugin_attributes.help_text is
    'Help text which is displayed for that attribute in the Builder.';
comment on column wwv_flow_plugin_attributes.attribute_type is
    'Type of control which is used to render the attribute field in the Builder.';
comment on column wwv_flow_plugin_attributes.is_required is
    'Should the Builder enforce this attribute to be filled out?';
comment on column wwv_flow_plugin_attributes.default_value is
    'Default value which is used in the Builder when the item type is selected.';
comment on column wwv_flow_plugin_attributes.display_length is
    'Display length of the attribute used when displayed in the Builder.';
comment on column wwv_flow_plugin_attributes.max_length is
    'Maximum length of the attribute used when displayed in the Builder.';
comment on column wwv_flow_plugin_attributes.sql_min_column_count is
    'Minimum number of columns the SQL query of the attribute has to have.';
comment on column wwv_flow_plugin_attributes.sql_max_column_count is
    'Maximum number of columns the SQL query of the attribute can have.';
comment on column wwv_flow_plugin_attributes.column_data_types is
    'Data types which are supported by the attribute of type "Region SQL Statement Column".';
comment on column wwv_flow_plugin_attributes.is_translatable is
    'Is the attribute value exported into the XLIFF file for translation.';
comment on column wwv_flow_plugin_attributes.depending_on_attribute_id is
    'Attribute which defines if the current attribute is displayed or not.';
comment on column wwv_flow_plugin_attributes.depending_on_condition_type is
    'Identifies the type of condition used to define when the attribute gets displayed.';
comment on column wwv_flow_plugin_attributes.depending_on_expression is
    'Identifies the condition expression used to define when the attribute gets displayed.';
comment on column wwv_flow_plugin_attributes.examples is
    'Examples displayed in APEX Builder for this attribute.';


--==============================================================================
create table wwv_flow_plugin_attr_values (
    id                        number         not null
                                             constraint wwv_flow_plugin_attrv_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_plugin_attrv_flow_fk
                                             references wwv_flows on delete cascade,
    plugin_attribute_id       number         not null
                                             constraint wwv_flow_plugin_attrv_attr_fk
                                             references wwv_flow_plugin_attributes on delete cascade,
    display_sequence          number         not null,
    display_value             varchar2(255)  not null,
    return_value              varchar2(255)  not null,
    is_quick_pick             varchar2(1)    default 'N' not null
                                             constraint wwv_flow_plugin_attrv_is_qp_ck
                                             check (is_quick_pick in ('Y', 'N')),
    is_deprecated             varchar2(1)    default 'N' not null
                                             constraint wwv_flow_plugin_attrv_isdep_ck
                                             check (is_deprecated in ('Y', 'N')),
    help_text                 varchar2(4000),
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
    );

create index wwv_flow_plugin_attrv_flow_fx on wwv_flow_plugin_attr_values (flow_id);

create index wwv_flow_plugin_attrv_attr_fx on wwv_flow_plugin_attr_values (plugin_attribute_id);

comment on table wwv_flow_plugin_attr_values is
    'Stores the possible values of a plug-in attribute if it''s of type select list.';
-- Add comments to the columns
comment on column wwv_flow_plugin_attr_values.display_sequence is
    'Order sequence in which the values are displayed.';
comment on column wwv_flow_plugin_attr_values.display_value is
    'Value displayed to end users';
comment on column wwv_flow_plugin_attr_values.return_value is
    'Value stored in attribute_xx column.';
comment on column wwv_flow_plugin_attr_values.help_text is
    'Help text which is displayed for the value in the Builder.';

--==============================================================================
create table wwv_flow_plugin_std_attributes (
    id                           number         not null
                                                constraint plugin_std_attr_pk
                                                primary key,
    flow_id                      number         not null
                                                constraint plugin_std_attr_flow_fk
                                                references wwv_flows on delete cascade,
    plugin_id                    number         not null
                                                constraint plugin_std_attr_plugin_fk
                                                references wwv_flow_plugins on delete cascade,
    name                         varchar2(30)   not null
                                                constraint plugin_std_attr_name_ck
                                                check (name in ('SOURCE_LOCATION',
                                                                'SOURCE_SQL',
                                                                'SOURCE_PLSQL',
                                                                'SOURCE_PLSQL_FUNCTION',
                                                                'SOURCE_HTML',
                                                                'SOURCE_PLAIN',
                                                                'ENDPOINT_HINT',
                                                                'LOV',
                                                                'INIT_JAVASCRIPT_CODE')),
    is_required                  varchar2(1)    not null
                                                constraint plugin_std_attr_required_ck
                                                check (is_required in ('Y', 'N')),
    default_value                varchar2(4000),
    sql_min_column_count         number(3)      constraint plugin_std_attr_mincol_ck
                                                check (sql_min_column_count > 0),
    sql_max_column_count         number(3)      constraint plugin_std_attr_maxcol_ck
                                                check (sql_max_column_count > 0),
    supported_ui_types           varchar2(255),
    supported_component_types    varchar2(255),
    depending_on_attribute_id    number         constraint plugin_std_attr_customattr_fk
                                                references wwv_flow_plugin_attributes on delete set null
                                                deferrable initially deferred,
    depending_on_has_to_exist    varchar2(1)    constraint plugin_std_attr_hastoexist_ck
                                                check (depending_on_has_to_exist in ('Y', 'N')),
    depending_on_condition_type  varchar2(30)   constraint plugin_std_attr_condition_ck
                                                check (depending_on_condition_type in (
                                                       'EQUALS',
                                                       'NOT_EQUALS',
                                                       'NULL',
                                                       'NOT_NULL',
                                                       'IN_LIST',
                                                       'NOT_IN_LIST' )),
    depending_on_expression      varchar2(4000),
    examples                     varchar2(4000),
    help_text                    varchar2(4000),
    security_group_id            number not null,
    attribute_comment            varchar2(4000),
    created_by                   varchar2(255),
    created_on                   date,
    last_updated_by              varchar2(255),
    last_updated_on              date
  );

alter table wwv_flow_plugin_std_attributes
  add constraint plugin_std_attribute_uk unique (plugin_id, name);

create index plugin_std_attribute_flow_fx   on wwv_flow_plugin_std_attributes (flow_id) compress;
create index plugin_std_attribute_depend_fx on wwv_flow_plugin_std_attributes (depending_on_attribute_id);


comment on table wwv_flow_plugin_std_attributes is
    'Stores additional meta data for standard attributes of a plug-in.';

comment on column wwv_flow_plugin_std_attributes.name  is
    'Name of the plug-in standard attribute';
comment on column wwv_flow_plugin_std_attributes.is_required is
    'Should the Builder enforce this attribute to be filled out?';
comment on column wwv_flow_plugin_std_attributes.default_value  is
    'Default value which is used in the Builder when the item type is selected.';
comment on column wwv_flow_plugin_std_attributes.sql_min_column_count  is
    'Minimum number of columns the SQL query of the attribute has to have.';
comment on column wwv_flow_plugin_std_attributes.sql_max_column_count  is
    'Maximum number of columns the SQL query of the attribute has to have.';
comment on column wwv_flow_plugin_std_attributes.supported_ui_types  is
    'User Interface types that this standard attribute supports';
comment on column wwv_flow_plugin_std_attributes.supported_component_types  is
    'Component types for which the attribute should be displayed on the UI';
comment on column wwv_flow_plugin_std_attributes.depending_on_attribute_id  is
    'Attribute which defines if the current attribute is displayed or not';
comment on column wwv_flow_plugin_std_attributes.depending_on_condition_type  is
    'Identifies the type of condition used to define when the attribute gets displayed.';
comment on column wwv_flow_plugin_std_attributes.depending_on_expression  is
    'Identifies the condition expression used to define when the attribute gets displayed.';
comment on column wwv_flow_plugin_std_attributes.examples  is
    'Examples displayed in APEX Builder for this attribute';
comment on column wwv_flow_plugin_std_attributes.help_text  is
    'Help text to be displayed in APEX Builder for this attribute';
comment on column wwv_flow_plugin_std_attributes.attribute_comment  is
    'Developer comment for this attribute';


--==============================================================================
create table wwv_flow_plugin_files (
    id                        number         not null
                                             constraint wwv_flow_plugin_file_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_plugin_file_flow_fk
                                             references wwv_flows on delete cascade,
    plugin_id                 number         not null
                                             constraint wwv_flow_plugin_file_parent_fk
                                             references wwv_flow_plugins on delete cascade,
    file_name                 varchar2(255)  not null,
    mime_type                 varchar2(255)  not null,
    file_charset              varchar2(128),
    file_content              blob,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  );

alter table wwv_flow_plugin_files
  add constraint wwv_flow_plugin_file_uk unique (plugin_id, file_name);

create index wwv_flow_plugin_file_flow_fx on wwv_flow_plugin_files (flow_id) compress;

create index wwv_flow_plugin_file_sgid_fx on wwv_flow_plugin_files(security_group_id) compress;

comment on table wwv_flow_plugin_files is
    'Stores the files like CSS, images, javascript files, ... of a plug-in.';
-- Add comments to the columns
comment on column wwv_flow_plugin_files.file_name is
    'Name of the file.';
comment on column wwv_flow_plugin_files.mime_type is
    'Mime type of the file.';
comment on column wwv_flow_plugin_files.file_charset is
    'IANA charset used for text files.';
comment on column wwv_flow_plugin_files.file_content is
    'Blob content of the file.';

--==============================================================================
create table wwv_flow_plugin_events (
    id                        number         not null
                                             constraint wwv_flow_plugin_evnt_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_plugin_evnt_flow_fk
                                             references wwv_flows on delete cascade,
    plugin_id                 number         not null
                                             constraint wwv_flow_plugin_evnt_parent_fk
                                             references wwv_flow_plugins on delete cascade,
    name                      varchar2(45)   not null
                                             constraint wwv_flow_plugin_evnt_name_ck
                                             check (lower(name)=name),
    display_name              varchar2(100)  not null,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  );

alter table wwv_flow_plugin_events
  add constraint wwv_flow_plugin_evnt_uk unique (plugin_id, name);

create index wwv_flow_plugin_evnt_flow_fx on wwv_flow_plugin_events (flow_id);

comment on table wwv_flow_plugin_events is
    'Stores which events can be triggered by this plug-in. This events are used for binding dynamic actions.';
-- Add comments to the columns
comment on column wwv_flow_plugins.name is
    'Internal name of the event which is used to reference it from wwv_flow_page_da_events and to bind the Javascript event.';
comment on column wwv_flow_plugins.display_name is
    'Contains the name of the event which is displayed on the UI.';

--==============================================================================
create table wwv_flow_plugin_item_filters (
    id                        number         not null
                                             constraint wwv_flow_plugin_iflt_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_plugin_iflt_flow_fk
                                             references wwv_flows on delete cascade,
    plugin_id                 number         not null
                                             constraint wwv_flow_plugin_iflt_parent_fk
                                             references wwv_flow_plugins on delete cascade,
    name                      varchar2(40)   not null
                                             constraint wwv_flow_plugin_iflt_name_ck
                                             check (upper(name)=name),
    display_name              varchar2(40)   not null,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  );

alter table wwv_flow_plugin_item_filters
  add constraint wwv_flow_plugin_iflt_uk unique (plugin_id, name);

create index wwv_flow_plugin_iflt_flow_fx on wwv_flow_plugin_item_filters (flow_id);

comment on table wwv_flow_plugin_item_filters is
    'Stores which filter operators can be used by this plug-in.';
-- Add comments to the columns
comment on column wwv_flow_plugin_item_filters.name is
    'Internal name of the filter which is used to reference it in the plug-in and in Page Designer.';
comment on column wwv_flow_plugin_item_filters.display_name is
    'Contains the name of the filter operator which is displayed in Page Designer.';

--==============================================================================
create table wwv_flow_plugin_settings (
    id                           number
                                 constraint wwv_flow_plugin_set_pk
                                 primary key,
    flow_id                      number not null
                                 constraint wwv_flow_plugin_set_flow_fk
                                 references wwv_flows on delete cascade,
    plugin_type                  varchar2(30)  not null
                                               constraint wwv_flow_plugin_set_type_ck
                                               check (plugin_type in ( 'ITEM TYPE',
                                                                       'WEB SOURCE TYPE',
                                                                       'DYNAMIC ACTION',
                                                                       'REGION TYPE',
                                                                       'VALIDATION TYPE',
                                                                       'PROCESS TYPE',
                                                                       'REPORT COLUMN TYPE',
                                                                       'AUTHORIZATION TYPE',
                                                                       'AUTHENTICATION TYPE' )),
    plugin                       varchar2(255) not null
                                               constraint wwv_flow_plugin_set_plugin_ck
                                               check (  plugin like 'NATIVE\_%' escape '\'
                                                     or plugin like 'PLUGIN\_%' escape '\' ),
    attribute_01                 varchar2(4000),
    attribute_02                 varchar2(4000),
    attribute_03                 varchar2(4000),
    attribute_04                 varchar2(4000),
    attribute_05                 varchar2(4000),
    attribute_06                 varchar2(4000),
    attribute_07                 varchar2(4000),
    attribute_08                 varchar2(4000),
    attribute_09                 varchar2(4000),
    attribute_10                 varchar2(4000),
    attribute_11                 varchar2(4000),
    attribute_12                 varchar2(4000),
    attribute_13                 varchar2(4000),
    attribute_14                 varchar2(4000),
    attribute_15                 varchar2(4000),
    security_group_id            number not null,
    created_by                   varchar2(255),
    created_on                   date,
    last_updated_by              varchar2(255),
    last_updated_on              date
  );

alter table wwv_flow_plugin_settings add constraint wwv_flow_plugin_set_uk unique (flow_id, plugin_type, plugin);
alter index wwv_flow_plugin_set_uk rebuild compress 2;

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > REPORTS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > REPORTS

--==============================================================================
create table wwv_flow_report_layouts (
    id                             number
                                       constraint wwv_flow_report_layouts_pk
                                       primary key,
    flow_id                        number not null
                                       constraint wwv_flow_report_layoutse_fk
                                       references wwv_flows
                                       on delete cascade,
    report_layout_name             varchar2(4000) not null,
    report_layout_type             varchar2(255) constraint wwv_flow_report_layout_type
                                     check (report_layout_type in (
                                          'RTF_FILE',
                                          'XSL_FILE',
                                          'XSL_GENERIC'
                                      )),
    --
    page_template                  clob not null,
    xslfo_column_heading_template  varchar2(4000),
    xslfo_column_template          varchar2(4000),
    xslfo_column_template_width    varchar2(4000),
    --
    reference_id                   number,
    security_group_id              number not null,
    --
    created_on                     date,
    created_by                     varchar2(255),
    last_updated_by                varchar2(255),
    last_updated_on                date,
    --
    report_layout_comment          varchar2(4000)
    ) lob (page_template) store as (cache reads enable storage in row)
/

create index wwv_flow_report_layouts_idx1 on wwv_flow_report_layouts (flow_id,security_group_id)
/
create unique index wwv_flow_report_layouts_idx2 on wwv_flow_report_layouts (flow_id,substr(report_layout_name,1,500))
/

--==============================================================================
create table  wwv_flow_shared_queries (
    id                    number
                             constraint wwv_flow_shared_qry_pk
                             primary key,
    flow_id               number not null
                             constraint wwv_flow_shdqry_flow_fk
                             references wwv_flows
                             on delete cascade,
    name                  varchar2(255) not null enable,
    security_group_id     number not null,
     --
    query_text            clob   not null,
    xml_structure         varchar2(255) constraint wwv_flow_shared_qry_xml_strc
                             check (xml_structure in (
                                 'STANDARD',
                                 'APEX'
                             )),
    report_layout_id      number,
    format                varchar2(255) constraint wwv_flow_shared_qry_format
                             check (format in (
                                 'PDF',
                                 'RTF',
                                 'XLS',
                                 'HTM',
                                 'XML'
                             )),
    format_item           varchar2(4000),
    output_file_name      varchar2(4000),
    content_disposition   varchar2(255) constraint shared_qry_content_disp
                              check (content_disposition in (
                                  'INLINE',
                                  'ATTACHMENT'
                              )),
    document_header       varchar2(255) constraint shared_qry_header
                             check (document_header in (
                                 'SERVER',
                                 'APEX'
                             )),
    xml_items             varchar2(4000),
    --
    created_on            date,
    created_by            varchar2(255),
    last_updated_on       date,
    last_updated_by       varchar2(255)
    ) lob (query_text) store as (cache reads enable storage in row)
/

create unique index wwv_flow_stored_queries_uk1 on wwv_flow_shared_queries (name, flow_id)
/

create index wwv_flow_stored_qry_fid_fk on wwv_flow_shared_queries (flow_id)
/

--==============================================================================
-- shared query SQL Statements
create table  wwv_flow_shared_qry_sql_stmts (
    id                    number
                             constraint wwv_flow_sqry_sql_pk
                             primary key,
    flow_id               number not null
                             constraint wwv_flow_sqry_sql_flow_fk
                             references wwv_flows
                             on delete cascade,
    shared_query_id       number not null
                             constraint wwv_flow_sqry_sql_sqry_fk
                             references wwv_flow_shared_queries
                             on delete cascade,
    security_group_id     number not null,
     --
    sql_statement         clob   not null,
    --
    created_on            date,
    created_by            varchar2(255),
    last_updated_on       date,
    last_updated_by       varchar2(255)
) lob (sql_statement) store as (cache reads enable storage in row)
/

create index wwv_flow_sqry_sql_fk1 on wwv_flow_shared_qry_sql_stmts (flow_id)
/

create index wwv_flow_sqry_sql_fk2 on wwv_flow_shared_qry_sql_stmts (shared_query_id)
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > GLOBALIZATION
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > GLOBALIZATION

--==============================================================================
create table wwv_flow_languages (
    id                   number
                         constraint wwv_flow_languages_pk
                         primary key,
    lang_id              varchar2(255) not null,
    lang_id_upper        varchar2(255) not null,
    lang_name            varchar2(255) not null,
    nls_language         varchar2(255) not null,
    nls_territory        varchar2(255),
    nls_sort             varchar2(255),
    nls_windows_charset  varchar2(255),
    translated_lang_name nvarchar2(255),
    dl_format            nvarchar2(255),
    ds_format            nvarchar2(255),
    ms_lcid              number not null,
    mysql_lang_id        varchar2(255)
    )
/

create unique index wwv_flow_language_idx1 on wwv_flow_languages (lang_id)
/

create index wwv_flow_lang_comp_idx1 on wwv_flow_languages (lang_id_upper, nls_language, nls_territory)
/

create index wwv_flow_lang_comp_idx2 on wwv_flow_languages (nls_language, nls_territory)
/


--==============================================================================
create table wwv_flow_language_map (
     id                             number
                                    constraint wwv_flow_lang_map_pk
                                    primary key,
     primary_language_flow_id       number
                                    constraint   wwv_flow_lang_flow_id_fk
                                    references   wwv_flows on delete cascade,
     translation_flow_id            number       not null,
     translation_flow_language_code varchar2(30) not null,
     trans_flow_lang_code_root      varchar2(30),
     translation_image_directory    varchar2(255),
     translation_comments           varchar2(4000),
     direction_right_to_left        varchar2(1) default 'N' not null
                                    constraint wwv_flow_language_map_dir_ck
                                    check (direction_right_to_left in ('Y','N')),
     security_group_id              number not null,
     last_updated_by                varchar2(255),
     last_updated_on                date,
     -- 4.0
     created_by                     varchar2(255),
     created_on                     date,
     map_comments                   varchar2(4000)
     )
/

create unique index wwv_flow_language_map_idx1 on wwv_flow_language_map
    (primary_language_flow_id, translation_flow_language_code)
/
create unique index wwv_flow_language_map_idx2 on wwv_flow_language_map
    (translation_flow_id)
/

--==============================================================================
create table wwv_flow_dbcharset_lang_map (
     id                             number
                                    constraint wwv_flow_dbcharset_lang_map_pk
                                    primary key,
     nls_db_charset                 varchar2(30) not null,
     language_code                  varchar2(30) not null
     )
/

create unique index wwv_flow_dbchar_lang_comp_uk on wwv_flow_dbcharset_lang_map (nls_db_charset, language_code)
/

--==============================================================================
create table wwv_flow_translatable_cols$ (
    id                         number
                               constraint wwv_flow_trans_cols_pk
                               primary key,
    table_name                 varchar2(30),
    column_name                varchar2(30),
    table_pk                   varchar2(30),
    flow_id_column             varchar2(30),
    page_id_column             varchar2(30),
    datatype                   varchar2(30),
    template_translatable      varchar2(1)
                               constraint wwv_flow_trans_cols_chk
                               check (template_translatable in ('Y','N')),
    column_description         varchar2(4000)
    )
/
create index wwv_flow_trans_cols$_idx1 on wwv_flow_translatable_cols$
    (table_name,column_name)
    compress 1
/

--==============================================================================
-- text_substr_1_255 column may be obsolete
create table wwv_flow_translatable_text$ (
    id                            number
                                  constraint wwv_flow_trans_text_pk
                                  primary key,
    translated_flow_id            number,
    flow_id                       number  not null
                                  constraint wwv_flow_trans_text_fk
                                  references wwv_flows
                                  on delete cascade,
    page_id                       number,
    translate_from_id             number,
    translate_to_id               number,
    translate_from_flow_table     varchar2(61),
    translate_from_flow_column    varchar2(30),
    translate_to_lang_code        varchar2(30),
    translation_specific_to_item  varchar2(3)
                                  constraint wwv_flow_trans_specific_check
                                  check (translation_specific_to_item in ('YES','NO')),
    translate_to_text             nclob,
    translate_from_text           nclob,
    template_translatable         varchar2(1)
                                  constraint wwv_flow_trans_temp_trans_chk
                                  check (template_translatable in ('Y','N')),
    security_group_id             number not null,
    last_updated_by               varchar2(255),
    last_updated_on               date,
    -- 4.0
    created_by                    varchar2(255),
    created_on                    date,
    text_substr_1_255             nvarchar2(255)
    )
/


create unique index wwv_flow_translatable_text_pk on wwv_flow_translatable_text$ (
     flow_id, page_id, translate_from_id, translate_from_flow_table, translate_from_flow_column,
     translate_to_lang_code) compress;


create index wwv_flow_translatable_substr  on wwv_flow_translatable_text$ (text_substr_1_255) ;
create index wwv_flow_translatable_from_id on wwv_flow_translatable_text$ (translate_from_id, translated_flow_id, translate_from_flow_table, translate_from_flow_column);
create index wwv_flow_translatable_key2    on wwv_flow_translatable_text$ (flow_id, translate_from_flow_table, translate_from_flow_column);

--==============================================================================
create table wwv_flow_dynamic_translations$ (
    id                         number
                               constraint wwv_flow_dyn_trans_pk
                               primary key,
    flow_id                    number  not null
                               constraint wwv_flow_dynamic_trans_fk1
                               references wwv_flows
                               on delete cascade,
    translate_to_lang_code     varchar2(30),
    translate_from_text        varchar2(4000),
    translate_to_text          varchar2(4000),
    created_by                 varchar2(255),
    created_on                 date,
    last_updated_by            varchar2(255),
    last_updated_on            date,
    security_group_id          number not null
                               constraint wwv_flow_dynamic_trans_fk2
                               references wwv_flow_companies(provisioning_company_id)
                               on delete cascade)
/

create index wwv_flow_dynamic_trans_idx1 on wwv_flow_dynamic_translations$ ( flow_id )
/

create index WWV_FLOW_DYNAMIC_TRANSL_FKIDX ON WWV_FLOW_DYNAMIC_TRANSLATIONS$
    (SECURITY_GROUP_ID,flow_id,TRANSLATE_TO_LANG_CODE)
/

--==============================================================================
-- MESSAGES
create table wwv_flow_messages$ (
    id                           number
                                 constraint wwv_flow_messages_pk
                                 primary  key,
    flow_id                      number not null
                                 constraint wwv_flow_messages_fk
                                 references wwv_flows
                                 on delete cascade,
    name                         varchar2(255) not null,
    message_language             varchar2(50)  not null,
    message_text                 varchar2(4000) not null,
    is_js_message                varchar2(1) default 'N' not null
                                             constraint wwv_flow_messages_is_js_msg_ck
                                             check (is_js_message in ('Y', 'N')),
    security_group_id            number not null,
    last_updated_by              varchar2(255),
    last_updated_on              date,
    -- 4.0
    created_by                   varchar2(255),
    created_on                   date,
    message_comment              varchar2(4000),
    constraint wwv_flow_messages$_uq unique (name, flow_id, message_language) using index compress 1
    )
/
create index wwv_flow_messages$_ix1 on wwv_flow_messages$ (flow_id, message_language, is_js_message)
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > EMAIL TEMPLATES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > EMAIL TEMPLATES

--==============================================================================
create table wwv_flow_email_templates (
    id                             number not null
                                   constraint wwv_flow_email_templates_pk primary key,
    security_group_id              number not null
                                   constraint wwv_flow_email_temp_sgid_fk
                                   references wwv_flow_companies( provisioning_company_id )
                                   on delete cascade,
    flow_id                        number not null
                                   constraint wwv_flow_email_templates_fk
                                   references wwv_flows
                                   on delete cascade,
    name                           varchar2(255) not null,
    static_id                      varchar2(255) not null,
    version_number                 number default 1 not null,
    comments                       varchar2(4000),
    --
    subject                        varchar2(255) not null,
    subject_compiled               blob,
    --
    text_template                  clob,
    text_template_compiled         blob,
    html_template                  clob,
    html_body                      clob not null,
    html_body_compiled             blob,
    html_header                    clob,
    html_header_compiled           blob,
    html_footer                    clob,
    html_footer_compiled           blob,
    --
    created_on                     date,
    created_by                     varchar2(255),
    last_updated_on                date,
    last_updated_by                varchar2(255)
    ) lob ( 
        subject_compiled, 
        text_template_compiled, 
        html_body_compiled, 
        html_header_compiled, 
        html_footer_compiled ) store as (cache reads enable storage in row)
/

alter table wwv_flow_email_templates add constraint wwv_flow_app_eml_temp_name_uk unique ( security_group_id, flow_id, static_id );

alter table wwv_flow_email_templates add constraint wwv_flow_app_eml_temp_is_comp
    check ( version_number = 1
            or ( version_number > 1 
                and ( subject           is null or ( subject        is not null and subject_compiled        is not null ) )
                and ( text_template     is null or ( text_template  is not null and text_template_compiled  is not null ) )
                and ( html_body         is null or ( html_body      is not null and subject_compiled        is not null ) )
                and ( html_header       is null or ( html_header    is not null and html_header_compiled    is not null ) )
                and ( html_footer       is null or ( html_footer    is not null and html_footer_compiled    is not null ) ) ) )
/

create index wwv_flow_app_eml_temp_sgid_fx on wwv_flow_email_templates( security_group_id ) compress;
create index wwv_flow_app_eml_temp_app_fx  on wwv_flow_email_templates( flow_id );

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > APPLICATION PROCESSES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > APPLICATION PROCESSES

--==============================================================================
create table wwv_flow_processing (
    id                number
                      constraint wwv_flow_processing_pk
                      primary key,
    flow_id           number not null
                      constraint wwv_flow_processing_fk
                      references wwv_flows
                      on delete cascade,
    --
    process_sequence  number not null,
    process_point     varchar2(255) not null
                      constraint valid_fproc_procpoint
                      check (process_point in (
                      'ON_DEMAND',
                      'ON_SUBMIT_BEFORE_COMPUTATION',
                      'ON_NEW_INSTANCE',      -- after a new session has been created
                      'AFTER_LOGIN',          -- after a successful authentication
                      'BEFORE_HEADER',
                      'AFTER_HEADER',
                      'AFTER_ERROR_HEADER',
                      'BEFORE_ERROR_FOOTER',
                      'BEFORE_BOX_BODY',
                      'BEFORE_SHOW_ITEMS', -- obsolete
                      'AFTER_SHOW_ITEMS', -- obsolete
                      'AFTER_BOX_BODY',
                      'BEFORE_FOOTER',
                      'AFTER_FOOTER',
                      'AFTER_SUBMIT_BEFORE_VALIDATION',
                      'AFTER_SUBMIT')),
    --
    process_type           varchar2(255) not null
                           constraint wwv_valid_proc_type
                           check (process_type in (
                           'SET_ITEM_USING_STATIC_ASSIGNMENT',
                           'SET_ITEM_USING_PLSQL_FUNCTION_BODY',
                           'SET_ITEM_USING_PLSQL_EXPRESSION',
                           'SET_ITEM_USING_SQL_EXPRESSION',
                           'SET_ITEM_USING_SQL_QUERY',
                           'INITIALIZE_ALL_PAGE_ITEMS'
                           ) or
                           process_type like 'NATIVE\_%' ESCAPE '\' or
                           process_type like 'PLUGIN\_%' ESCAPE '\'),
    --
    process_name           varchar2(255)  not null,
    process_sql_clob       clob,
    process_clob_language   varchar2(10)
                           constraint wwv_valid_process_clob_lang
                           check (process_clob_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    location               varchar2(10)
                           constraint wwv_app_proc_location_ck
                           check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE' )),
    remote_server_id       number
                           constraint wwv_app_proc_remote_server_fk
                           references wwv_remote_servers,
    web_src_operation_id   number
                           constraint wwv_app_proc_websrc_opers_fk
                           references wwv_flow_web_src_operations,
    email_template_id      number generated always as (
                                                        case when process_type = 'NATIVE_SEND_EMAIL' and attribute_11 is not null
                                                            then to_number( attribute_11 )
                                                        end ) virtual,
    task_def_id            number generated always as (
                                                        case when process_type = 'NATIVE_CREATE_TASK' and attribute_01 is not null
                                                            then to_number( attribute_01 )
                                                        end ) virtual,                                                    
    --
    attribute_01           varchar2(4000),
    attribute_02           varchar2(4000),
    attribute_03           varchar2(4000),
    attribute_04           varchar2(4000),
    attribute_05           varchar2(4000),
    attribute_06           varchar2(4000),
    attribute_07           varchar2(4000),
    attribute_08           varchar2(4000),
    attribute_09           varchar2(4000),
    attribute_10           varchar2(4000),
    attribute_11           varchar2(4000),
    attribute_12           varchar2(4000),
    attribute_13           varchar2(4000),
    attribute_14           varchar2(4000),
    attribute_15           varchar2(4000),
    --
    process_error_message  varchar2(4000),
    error_display_location varchar2(40) default 'ON_ERROR_PAGE'
                           constraint wwv_flow_proc_edl
                           check (error_display_location in ('INLINE_IN_NOTIFICATION', 'ON_ERROR_PAGE')),
    process_when           varchar2(4000),
    process_when_type      varchar2(255),
    process_when2          varchar2(4000),
    process_when_type2     varchar2(255), -- obsolete
    item_name              varchar2(255), -- obsolete
    required_patch         number,
    security_scheme        varchar2(255),
    runtime_where_clause   varchar2(4000),
    --
    security_group_id      number not null,
    last_updated_by        varchar2(255),
    last_updated_on        date,
    -- 4.0
    created_by             varchar2(255),
    created_on             date,
    process_comment        varchar2(4000),
    --
    constraint wwv_app_proc_mail_templ_fk foreign key( email_template_id )
                                          references wwv_flow_email_templates
                                          deferrable initially deferred
    ) lob (process_sql_clob) store as (cache reads enable storage in row)
/

create unique index wwv_flow_processing_fkidx on wwv_flow_processing(flow_id,id);

create index wwv_app_proc_remote_server_fx   on wwv_flow_processing( remote_server_id )      compress;
create index wwv_app_proc_websrc_opers_fx    on wwv_flow_processing( web_src_operation_id )  compress;
create index wwv_app_proc_email_template_fx  on wwv_flow_processing( email_template_id )     compress;
create index wwv_app_proc_task_def_fx        on wwv_flow_processing( task_def_id )           compress;


--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > APPLICATION ITEMS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > APPLICATION ITEMS

--==============================================================================
create table wwv_flow_items (
    id                         number
                               constraint wwv_flow_items_pk
                               primary key,
    flow_id                    number not null
                               constraint wwv_flow_items_fk
                               references wwv_flows
                               on delete cascade,
    name                       varchar2(255) not null,
    name_length                number,
    scope                      varchar2(6)
                               default 'APP'
                               not null
                               constraint wwv_flow_items_ck_scope
                               check(scope in ('APP','GLOBAL')),
    data_type                  varchar2(30)
                               constraint valid_fitems_datatype
                               check (data_type in (
                               'VARCHAR',
                               'NUMBER',
                               'DATE',
                               'CLOB' )),
    is_persistent              varchar2(1)
                               constraint valid_fitems_ispers
                               check (is_Persistent in ('Y','N','U','A')),
    protection_level           varchar2(1)
                               constraint valid_fitems_protection_level
                               check (protection_level in ('N','B','P','S','I')),
    escape_on_http_output      varchar2(1) default 'N' not null
                               constraint wwv_fitems_esc_output
                               check (escape_on_http_output in ('Y','N')),
    required_patch             number,
    security_group_id          number not null,
    last_updated_by            varchar2(255),
    last_updated_on            date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    item_comment               varchar2(4000)
    )
/

create index wwv_flow_items_u_name on wwv_flow_items (flow_id,name);
create unique index wwv_flow_items_idx3 on wwv_flow_items (flow_id,security_group_id,name);
create index wwv_flow_items_idx4 on wwv_flow_items (flow_id, protection_level);

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > APPLICATION COMPUTATIONS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > APPLICATION COMPUTATIONS

--==============================================================================
create table wwv_flow_computations (
    id                    number
                          constraint wwv_flow_computations_pk
                          primary key,
    flow_id               number not null
                          constraint wwv_flow_computations_fk
                          references wwv_flows
                          on delete cascade,
    --
    computation_sequence  number        not null,
    computation_item      varchar2(255) not null,
    computation_point     varchar2(255) not null,
                          constraint valid_fcomp_point
                          check (computation_point in (
                          'ON_NEW_INSTANCE',
                          'AFTER_LOGIN',
                          'BEFORE_HEADER',
                          'AFTER_ERROR_HEADER',
                          'BEFORE_ERROR_FOOTER',
                          'AFTER_HEADER',
                          'BEFORE_BOX_BODY',
                          'AFTER_BOX_BODY',
                          'BEFORE_FOOTER',
                          'AFTER_FOOTER',
                          'AFTER_SUBMIT')),
    -- computation_item_type to be obsoleted (SSpadafore)
    computation_item_type varchar2(30)
                          constraint valid_fcomp_itemtype
                          check (computation_item_type in (
                          'VARCHAR',
                          'VC_ARR',
                          'NUMBER')),
    computation_type      varchar2(30) not null
                          constraint valid_fcomp_comptype
                          check (computation_type in (
                          'SET_ITEM_EQUAL_THIS_PREFERENCE',
                          'STATIC_ASSIGNMENT',
                          'FUNCTION_BODY',
                          'QUERY',
                          'QUERY_COLON',
                          'EXPRESSION',
                          'ITEM_VALUE')),
    computation_language  varchar2(10)
                          constraint valid_fcomp_comp_lang
                          check (computation_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    computation_processed varchar2(30)
                          constraint valid_fcomp_compproc
                          check (computation_processed in (
                          'ON_NEW_INSTANCE',
                          'REPLACE_EXISTING',
                          'ADD_TO_EXISTING',
                          'RAISE_ERROR_ON_REPLACE',
                          'REPLACE_NULL_VALUES')),
    computation                varchar2(4000),
    required_patch             number,
    compute_when               varchar2(4000),
    compute_when_type          varchar2(255),
    compute_when_text          varchar2(4000),
    computation_error_message  varchar2(4000),
    security_scheme            varchar2(255),
    security_group_id          number not null,
    last_updated_by            varchar2(255),
    last_updated_on            date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    computation_comment        varchar2(4000)
    )
/
create index wwv_flow_computations_fkidx on wwv_flow_computations(flow_id,id);

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > AUTHENTICATIONS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > AUTHENTICATIONS

--==============================================================================
-- legacy authentication table (pre 4.1), still needed for upgrades
--
create table wwv_flow_custom_auth_setups(
    id                                                 number
        constraint wwv_flow_auth_setups_pk
        primary key,
    flow_id                                            number not null
        constraint wwv_flow_auth_setups_fk
        references wwv_flows on delete cascade,
    reference_id                                       number,
    name                                               varchar2(255) not null,
    description                                        varchar2(4000),
    page_sentry_function                               clob,
    sess_verify_function                               clob,
    invalid_session_page                               number,
    invalid_session_url                                varchar2(4000),
    pre_auth_process                                   clob,
    auth_function                                      clob,
    post_auth_process                                  clob,
    cookie_name                                        varchar2(4000),
    cookie_path                                        varchar2(4000),
    cookie_domain                                      varchar2(4000),
    use_secure_cookie_yn                               varchar2(1)
                                                       constraint wwv_flow_custom_auth_chk1
                                                       check (use_secure_cookie_yn in ('Y','N')),
    ldap_host                                          varchar(255),
    ldap_port                                          varchar(255),
    ldap_string                                        varchar(255),
    attribute_01                                       varchar(4000),
    attribute_02                                       varchar(4000),
    attribute_03                                       varchar(4000),
    attribute_04                                       varchar(4000),
    attribute_05                                       varchar(4000),
    attribute_06                                       varchar(4000),
    attribute_07                                       varchar(4000),
    attribute_08                                       varchar(4000),
    required_patch                                     number,
    security_group_id                                  number not null,
    last_updated_by                                    varchar2(255),
    last_updated_on                                    date,
    created_by                                         varchar2(255),
    created_on                                         date)
    lob (page_sentry_function, sess_verify_function, pre_auth_process, auth_function, post_auth_process) store as (cache reads enable storage in row)
/
create index wwv_flow_auth_setups_idx1 on wwv_flow_custom_auth_setups (flow_id);

--==============================================================================
create table wwv_flow_authentications (
    id                                                  number,
    security_group_id                                   number         not null,
    flow_id                                             number         not null,
    reference_id                                        number,
    name                                                varchar2(255)  not null,
    scheme_type                                         varchar2(255)  not null,
    --
    invalid_session_type                                varchar2(13),
    invalid_session_url                                 varchar2(4000),
    --
    switch_in_session_yn                                varchar2(1)    default 'N' not null,
    --
    cookie_type                                         varchar2(1) generated always as (
                                                                        case
                                                                        when use_secure_cookie_yn <> 'N'
                                                                             or cookie_domain     is not null
                                                                             or cookie_path       is not null
                                                                        then 'C'
                                                                        when cookie_name is null
                                                                        then 'A'
                                                                        when cookie_name='&WORKSPACE_COOKIE.'
                                                                        then 'W'
                                                                        else 'C'
                                                                        end ) virtual,
    cookie_name                                         varchar2(4000),
    cookie_path                                         varchar2(4000),
    cookie_domain                                       varchar2(4000),
    use_secure_cookie_yn                                varchar2(1)    default 'N' not null,
    logout_url                                          varchar2(4000),
    logout_type                                         varchar2(4) generated always as (
                                                                        case
                                                                        when logout_url is null then 'HOME'
                                                                        else 'URL'
                                                                        end ) virtual,
    --
    plsql_code                                          clob,
    verification_function                               varchar2(255),
    pre_auth_process                                    varchar2(255),
    post_auth_process                                   varchar2(255),
    --
    ras_mode                                            number(1,0)    default 0 not null,
    ras_dynamic_roles                                   varchar2(4000),
    ras_namespaces                                      varchar2(4000),
    --
    attribute_01                                        varchar2(4000),
    attribute_02                                        varchar2(4000),
    attribute_03                                        varchar2(4000),
    attribute_04                                        varchar2(4000),
    attribute_05                                        varchar2(4000),
    attribute_06                                        varchar2(4000),
    attribute_07                                        varchar2(4000),
    attribute_08                                        varchar2(4000),
    attribute_09                                        varchar2(4000),
    attribute_10                                        varchar2(4000),
    attribute_11                                        varchar2(4000),
    attribute_12                                        varchar2(4000),
    attribute_13                                        varchar2(4000),
    attribute_14                                        varchar2(4000),
    attribute_15                                        varchar2(4000),
    --
    help_text                                           varchar2(4000),
    comments                                            varchar2(4000),
    --
    last_updated_by                                     varchar2(255),
    last_updated_on                                     date,
    created_by                                          varchar2(255),
    created_on                                          date,
    --
    credential_id                                       number generated always as (
                                                                   case
                                                                   when scheme_type in ( 'NATIVE_SOCIAL',
                                                                                         'NATIVE_SAML' )
                                                                   then to_number(attribute_01)
                                                                   end ) virtual,
    --
    constraint wwv_flow_authentications_pk              primary key(id),
    constraint wwv_flow_authentications_fk              foreign key(flow_id)
                                                        references wwv_flows
                                                        on delete cascade,
    constraint wwv_flow_authn_fk_cred                   foreign key(credential_id)
                                                        references wwv_credentials(id),
    constraint wwv_flow_authentications_uk              unique(flow_id,name),
    constraint wwv_flow_authentications_ck1             check (use_secure_cookie_yn in ('Y','N')),
    constraint wwv_flow_authentications_ck2             check (
                                                             scheme_type like 'NATIVE\_%' escape '\'
                                                          or scheme_type like 'PLUGIN\_%' escape '\' ),
    constraint wwv_flow_authentications_ck3             check (invalid_session_type in ('LOGIN','BUILTIN_LOGIN','URL')),
    constraint wwv_flow_authentications_ck4             check (ras_mode in (0,1,2)),
    constraint wwv_flow_authentications_ck5             check (switch_in_session_yn in ('Y','N'))
) lob (plsql_code) store as (cache reads enable storage in row)
/

alter table wwv_flows add (
    constraint wwv_flows_fk_authentication              foreign key (authentication_id)
                                                        references wwv_flow_authentications(id)
                                                        initially deferred
)
/

alter index wwv_flow_authentications_uk rebuild compress 1
/
create index wwv_flow_authentications_idx1 on wwv_flow_authentications (reference_id) compress
/
create index wwv_flow_authentications_idx2 on wwv_flow_authentications (credential_id)
/

comment on column wwv_flow_authentications.ras_mode is
'12c Real Application Security: 0 - Disabled, 1 - External Users, 2 - Internal users'
/
comment on column wwv_flow_authentications.ras_dynamic_roles is
'12c Real Application Security: Colon-separated list of dynamic roles that are enabled on assign_user'
/
comment on column wwv_flow_authentications.ras_namespaces is
'12c Real Application Security: Colon-separated list of namespaces that are enabled on create_session'
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > WEB SERVICE REFERENCES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > WEB SERVICE REFERENCES

--==============================================================================
create table wwv_flow_shared_web_services (
   id                            number
                                 constraint wwv_flow_web_services_pk
                                 primary key,
   --
   -- columns that identifies who owns the service description
   --
   security_group_id             number not null,
   flow_id                       number
                                 constraint wwv_flow_ws_fk
                                 references wwv_flows
                                 on delete cascade,
   name                          varchar2(255) not null,
   --
   -- columns that describe the service
   --
   url                           varchar2(4000),
   action                        varchar2(4000),
   proxy_override                varchar2(4000),
   soap_version                  varchar2(30),
   --
   -- can be generated, but can also be edited after generation
   --
   soap_envelope                 clob,
   --
   -- body / envelope substitution strings
   --
   flow_items_comma_delimited    clob,
   static_parm_01                varchar2(4000),
   static_parm_02                varchar2(4000),
   static_parm_03                varchar2(4000),
   static_parm_04                varchar2(4000),
   static_parm_05                varchar2(4000),
   static_parm_06                varchar2(4000),
   static_parm_07                varchar2(4000),
   static_parm_08                varchar2(4000),
   static_parm_09                varchar2(4000),
   static_parm_10                varchar2(4000),
   --
   -- columns that deal with soap responce
   --
   stylesheet                    clob,
   --
   -- administrative columns
   --
   reference_id                  number,
   last_updated_by               varchar2(255),
   last_updated_on               date,
   -- 4.0
   created_by                 varchar2(255),
   created_on                 date
   ) lob (soap_envelope, stylesheet) store as (cache reads enable storage in row)
/

comment on column wwv_flow_shared_web_services.url is 'SOAP Service URL';
comment on column wwv_flow_shared_web_services.action is 'SOAP Action';
comment on column wwv_flow_shared_web_services.proxy_override is 'HTTP proxy for SOAP request';
comment on column wwv_flow_shared_web_services.soap_envelope is 'May contain #ITEM_NAME# substitutions';
comment on column wwv_flow_shared_web_services.flow_items_comma_delimited is 'A comma delmited list of flow items contained in the body for example: ITEM1,ITEM2,ITEM3';

create index wwv_flow_shared_web_serv_idx1 on wwv_flow_shared_web_services (flow_id)
/

--==============================================================================
create table wwv_flow_ws_operations (
    id                      number
                            constraint wwv_flow_ws_opers_pk
                            primary key,
    ws_id                   number
                            constraint wwv_flow_ws_opers_fk
                            references wwv_flow_shared_web_services
                            on delete cascade,
    name                    varchar2(255),
    input_message_name      varchar2(255),
    input_message_ns        varchar2(4000),
    input_message_style     varchar2(255),
    output_message_name     varchar2(255),
    output_message_ns       varchar2(4000),
    output_message_style    varchar2(255),
    header_message_name     varchar2(4000),
    header_message_style    varchar2(255),
    soap_action             varchar2(4000),
    --
    security_group_id       number not null,
    last_updated_by         varchar2(255),
    last_updated_on         date,
    -- 4.0
    created_by              varchar2(255),
    created_on              date
    )
/

create index wwv_flow_ws_opers_idx1 on wwv_flow_ws_operations (ws_id)
/

--==============================================================================
create table wwv_flow_ws_parameters (
    id                      number
                            constraint wwv_flow_ws_parms_pk
                            primary key,
    ws_opers_id             number
                            constraint wwv_flow_ws_parms_fk
                            references wwv_flow_ws_operations
                            on delete cascade,
    name                    varchar2(255),
    input_or_output         varchar2(1)
                            constraint wwv_flow_ws_parms_chk1
                            check (input_or_output in ('I','O','H','A')),
    parm_type               varchar2(255),
    path                    varchar2(4000),
    type_is_xsd             varchar2(1)
                            constraint wwv_flow_ws_parms_chk2
                            check (type_is_xsd in ('Y','N')),
    form_qualified          varchar2(1)
                            constraint wwv_flow_ws_parms_chk3
                            check (form_qualified in ('Y','N')),
    parent_id               number,
    --
    security_group_id       number not null,
    last_updated_by         varchar2(255),
    last_updated_on         date
    )
/

create index wwv_flow_ws_parms_idx1 on wwv_flow_ws_parameters (ws_opers_id)
/


--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > TABS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > TABS

--==============================================================================
create table wwv_flow_tabs (
    id                         number
                               constraint wwv_flow_tabs_pk
                               primary key,
    flow_id                    number not null
                               constraint wwv_flow_tabs_fk
                               references wwv_flows
                               on delete cascade,
    tab_set                    varchar2(255)  not null,
    tab_sequence               number         not null,
    tab_name                   varchar2(255)  not null,
    tab_image                  varchar2(255),
    tab_non_current_image      varchar2(255),
    tab_image_attributes       varchar2(255),
    tab_text                   varchar2(2000) not null,
    tab_step                   number not null,
    tab_also_current_for_pages varchar2(4000),
    tab_parent_tabset          varchar2(255),
    tab_plsql_condition        varchar2(4000),
    display_condition_type     varchar2(255),
    tab_disp_cond_text         varchar2(4000),
    required_patch             number,
    security_scheme            varchar2(255),
    security_group_id          number not null,
    last_updated_by            varchar2(255),
    last_updated_on            date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    tab_comment                varchar2(4000)
    )
/

create unique index wwv_flow_tabs_unique on wwv_flow_tabs (flow_id, tab_set, tab_name)
/

create unique index wwv_flow_tabs_unique2 on wwv_flow_tabs (flow_id, security_group_id, tab_set, tab_name)
/

--==============================================================================
create table wwv_flow_toplevel_tabs (
    id                         number
                               constraint wwv_flow_toplev_tab_pk
                               primary key,
    flow_id                    number not null
                               constraint wwv_flow_toplev_tab_fk
                               references wwv_flows
                               on delete cascade,
    tab_set                    varchar2(255)  not null,
    tab_sequence               number         not null,
    tab_name                   varchar2(255)  not null,
    tab_image                  varchar2(255),
    tab_non_current_image      varchar2(255),
    tab_image_attributes       varchar2(255),
    tab_text                   varchar2(2000) not null,
    tab_target                 varchar2(4000) not null,
    display_condition          varchar2(4000),
    display_condition2         varchar2(4000),
    display_condition_type     varchar2(255),
    current_on_tabset          varchar2(255),
    required_patch             number,
    security_scheme            varchar2(255),
    security_group_id          number not null,
    last_updated_by            varchar2(255),
    last_updated_on            date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    tab_comment                varchar2(4000)
    )
/

create unique index wwv_flow_tl_tabs_unique on wwv_flow_toplevel_tabs (flow_id, tab_set, tab_name)
/
create unique index wwv_flow_tl_tabs_unique2 on wwv_flow_toplevel_tabs (flow_id, security_group_id, tab_set, tab_name)
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > LISTS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > LISTS

--==============================================================================
create table wwv_flow_lists (
     id                       number
                              constraint wwv_flow_lists_pk
                              primary key,
     flow_id                  number not null
                              constraint wwv_flow_lists_flow_fk
                              references wwv_flows
                              on delete cascade,
     name                     varchar2(255) not null,
     list_type                varchar2(30) default 'STATIC' not null
                              constraint wwv_flow_val_list_type
                              check (list_type in ('STATIC','SQL_QUERY','FUNCTION_RETURNING_SQL_QUERY')),
     list_language            varchar2(10)
                              constraint wwv_flow_val_list_lang
                              check (list_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
     list_query               varchar2(4000),
     list_status              varchar2(255),
                              constraint wwv_flows_val_list_status
                              check (list_status in ('PERSONAL','PUBLIC')),
     list_displayed           varchar2(255)
                              constraint wwv_flow_val_listdisplayed1
                              check (list_displayed in ('ON_DEMAND','ALWAYS','NEVER','BY_DEFAULT')),
     display_row_template_id  number, /* $$$ obsolete */
     required_patch           number,
     reference_id             number,
     security_group_id        number not null,
     last_updated_by          varchar2(255),
     last_updated_on          date,
     -- 4.0
     created_by               varchar2(255),
     created_on               date,
     list_comment             varchar2(4000)
     )
/

create unique index wwv_flow_lists_fkidx on wwv_flow_lists(flow_id,id)
/

create index wwv_flow_lists_idx1 on wwv_flow_lists (reference_id)
compress
/
--==============================================================================
create table wwv_flow_list_items (
     id                           number
                                  constraint wwv_flow_list_items_pk
                                  primary key,
     list_id                      number
                                  constraint wwv_flow_list_items_fk
                                  references wwv_flow_lists
                                  on delete cascade,
     flow_id                      number,
     --
     sub_item_count               number,
     parent_list_item_id          number
                                  constraint parent_list_item_fk
                                  references wwv_flow_list_items
                                  on delete cascade,
     --
     list_item_type               varchar2(255)
                                  constraint wwv_flows_val_listitemtype
                                  check (list_item_type in ('LINK','FILE','TABLE')),
     list_item_status             varchar2(255),
                                  constraint wwv_flows_val_listitemstatus
                                  check (list_item_status in ('PERSONAL','PUBLIC')),
     item_displayed               varchar2(255)
                                  constraint wwv_flow_val_listdisplayed
                                  check (item_displayed in ('ON_DEMAND','ALWAYS','NEVER','BY_DEFAULT')),
     list_item_display_sequence   number,
     list_item_link_text          varchar2(4000),
     list_item_link_target        varchar2(4000),
     list_item_icon               varchar2(4000),
     list_item_icon_attributes    varchar2(4000),
     list_item_icon_alt_attribute varchar2(4000),
     --
     list_item_icon_exp           varchar2(4000),
     list_item_icon_exp_attr      varchar2(4000),
     --
     list_item_owner              varchar2(255),
     list_item_current_for_pages  varchar2(4000),
     list_item_current_type       varchar2(30)
                                  constraint wwv_flow_lst_curr_type_val
                                  check (list_item_current_type in (
                                  'COLON_DELIMITED_PAGE_LIST',
                                  'EXPRESSION',
                                  'ALWAYS',
                                  'NEVER',
                                  'EXISTS',
                                  'NOT_EXISTS',
                                  'TARGET_PAGE')),
     list_item_current_language   varchar2(10)
                                  constraint wwv_flow_list_curr_lang
                                  check (list_item_current_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
     list_item_disp_cond_type     varchar2(255),
     list_item_disp_condition     varchar2(4000),
     list_item_disp_cond_type2    varchar2(255), -- obsolete
     list_item_disp_condition2    varchar2(4000),
     list_countclicks_y_n         varchar2(1)
                                  constraint wwv_flow_list_count_chk
                                  check (list_countclicks_y_n in ('Y','N')),
     list_countclicks_cat         varchar2(255),
     list_text_01                 varchar2(4000),
     list_text_02                 varchar2(4000),
     list_text_03                 varchar2(4000),
     list_text_04                 varchar2(4000),
     list_text_05                 varchar2(4000),
     list_text_06                 varchar2(4000),
     list_text_07                 varchar2(4000),
     list_text_08                 varchar2(4000),
     list_text_09                 varchar2(4000),
     list_text_10                 varchar2(4000),
     list_text_11                 varchar2(4000),
     list_text_12                 varchar2(4000),
     list_text_13                 varchar2(4000),
     list_text_14                 varchar2(4000),
     list_text_15                 varchar2(4000),
     list_text_16                 varchar2(4000),
     list_text_17                 varchar2(4000),
     list_text_18                 varchar2(4000),
     list_text_19                 varchar2(4000),
     list_text_20                 varchar2(4000),
     translate_list_text_y_n     varchar2(1)
                                  constraint wwv_flow_list_items_ck1
                                  check (translate_list_text_y_n in ('Y','N')),
     required_patch               number,
     security_scheme              varchar2(255),
     security_group_id            number not null,
     last_updated_by              varchar2(255),
     last_updated_on              date,
     -- 4.0
     created_by                   varchar2(255),
     created_on                   date,
     list_item_comment            varchar2(4000)
     )
/


create index parent_list_item_id_idx on wwv_flow_list_items ( parent_list_item_id )
/

create index wwv_flow_list_items_idx1 on wwv_flow_list_items ( security_group_id, flow_id, list_id )
/

create unique index wwv_flow_list_items_idx2 on wwv_flow_list_items (list_id,id,item_displayed,required_patch)
/

create index wwv_flow_list_items_fkidx on wwv_flow_list_items(list_id,flow_id,list_item_display_sequence)
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > BREADCRUMBS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > BREADCRUMBS

--==============================================================================
create table wwv_flow_menus (
     id                       number
                              constraint wwv_flow_menus_pk
                              primary key,
     flow_id                  number not null
                              constraint wwv_flow_menus_flow_fk
                              references wwv_flows
                              on delete cascade,
     name                     varchar2(255) not null,
     security_group_id        number not null,
     last_updated_by          varchar2(255),
     last_updated_on          date,
     -- 4.0
     created_by                 varchar2(255),
     created_on                 date,
     menu_comment             varchar2(4000)
     )
/
create index wwv_flow_menus_idx1 on wwv_flow_menus (flow_id)
/

--==============================================================================
create table wwv_flow_menu_options (
     id                       number
                              constraint wwv_flow_menu_opts_pk
                              primary key,
     parent_id                number,
     menu_id                  number not null
                              constraint wwv_flow_opt_menus_fk
                              references wwv_flow_menus
                              on delete cascade,
     flow_id                  number,
     option_sequence          number,
     short_name               varchar2(255) not null,
     long_name                varchar2(4000),
     link                     varchar2(4000),
     page_id                  number,
     also_current_for_pages   varchar2(4000),
     display_when_cond_type   varchar2(255),
     display_when_condition   varchar2(4000),
     display_when_condition2  varchar2(4000),
     security_scheme          varchar2(255),
     required_patch           number,
     security_group_id        number not null,
     last_updated_by          varchar2(255),
     last_updated_on          date,
     -- 4.0
     created_by                 varchar2(255),
     created_on                 date,
     menu_option_comment      varchar2(4000)
     )
/

create index wwv_flow_menu_opt_idx1 on wwv_flow_menu_options (flow_id, id);

create index wwv_flow_menu_opt_idx2 on wwv_flow_menu_options (page_id,flow_id);

create index wwv_flow_menu_opt_idx3 on wwv_flow_menu_options (menu_id, page_id);


--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > TREES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > TREES

--==============================================================================
create table wwv_flow_trees  (
    id                        number
                              constraint wwv_flow_tree_pk
                              primary key,
    flow_id                   number not null
                              constraint wwv_flow_tree_fk
                              references wwv_flows
                              on delete cascade,
    tree_name                 varchar2(255)  not null,
    tree_type                 varchar2(30)
                              constraint wwv_flow_tree_type
                              check (tree_type in ('DYNAMIC','STATIC')),
    tree_query                varchar2(4000),
    flow_item                 varchar2(255),
    security_group_id         number not null,
    max_levels                number not null
                              constraint wwv_flow_valid_level
                              check (max_levels between 1 and 100),
    unexpanded_parent         varchar2(4000),
    unexpanded_parent_last    varchar2(4000),
    expanded_parent           varchar2(4000),
    expanded_parent_last      varchar2(4000),
    leaf_node                 varchar2(4000),
    leaf_node_last            varchar2(4000),
    drill_up                  varchar2(4000),
    name_link_anchor_tag      varchar2(4000),
    name_link_not_anchor_tag  varchar2(4000),
    indent_vertical_line      varchar2(4000),
    indent_vertical_line_last varchar2(4000),
    --
    before_tree               varchar2(4000),
    after_tree                varchar2(4000),
    --
    level_1_template          varchar2(4000),
    level_2_template          varchar2(4000),
    level_3_template          varchar2(4000),
    level_4_template          varchar2(4000),
    level_5_template          varchar2(4000),
    level_6_template          varchar2(4000),
    level_7_template          varchar2(4000),
    level_8_template          varchar2(4000),
    last_updated_by           varchar2(255),
    last_updated_on           date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    tree_comment              varchar2(4000)
    )
/
create unique index wwv_flow_trees_fkidx on wwv_flow_trees(flow_id,id);

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > NAVIGATION BAR ENTRIES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > NAVIGATION BAR ENTRIES

--==============================================================================
create table wwv_flow_icon_bar (
    id                          number
                                constraint wwv_flow_icon_bar_pk
                                primary key,
    flow_id                     number not null
                                constraint wwv_flow_icon_bar_fk
                                references wwv_flows
                                on delete cascade,
    icon_sequence               number not null,
    --
    icon_image                  varchar2(255),
    icon_image2                 varchar2(255),
    icon_image3                 varchar2(255),
    icon_subtext                varchar2(255),
    icon_subtext2               varchar2(255),
    icon_subtext3               varchar2(255),
    icon_target                 varchar2(4000),
    icon_image_alt              varchar2(4000),
    icon_height                 number,
    icon_width                  number,
    icon_height2                number,
    icon_width2                 number,
    icon_height3                number,
    icon_width3                 number,
    --
    icon_bar_disp_cond          varchar2(4000),
    icon_bar_disp_cond_type     varchar2(255),
    icon_bar_flow_cond_instr    varchar2(4000),
    begins_on_new_line          varchar2(30)
                                constraint valid_ficon_beginonnewline
                                check (begins_on_new_line in ('NO','YES')),
    cell_colspan                number,
    onclick                     varchar2(4000),
    required_patch              number,
    security_scheme             varchar2(255),
    reference_id                number,
    nav_entry_is_feedback_yn    varchar2(1) default 'N'
                                constraint valid_ficon_navisfeedback
                                check (nav_entry_is_feedback_yn in ('Y','N')),
    security_group_id           number not null,
    last_updated_by             varchar2(255),
    last_updated_on             date,
    -- 4.0
    created_by                  varchar2(255),
    created_on                  date,
    icon_bar_comment            varchar2(4000)
    )
/

create index wwv_flow_icon_bar_idx2 on wwv_flow_icon_bar (flow_id, security_group_id)
/


--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > LISTS OF VALUES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > LISTS OF VALUES

--==============================================================================
create table wwv_flow_lists_of_values$ (
    id                          number
                                constraint wwv_flow_lov_pk
                                primary key,
    flow_id                     number not null
                                constraint wwv_flow_lov_fk
                                references wwv_flows
                                on delete cascade,
    lov_name                    varchar2(255) not null,
    lov_query                   varchar2(4000),
    source_type                 varchar2(23)
                                constraint wwv_flow_lov_source_type_ck
                                check ( source_type in ( 'TABLE', 'SQL','FUNC_BODY_RETURNING_SQL', 'LEGACY_SQL' )),
    function_body_language      varchar2(10)
                                constraint wwv_flow_lov_func_body_lang_ck
                                check (function_body_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    security_group_id           number         not null,
    reference_id                number,
    --
    location                    varchar2(13)
                                constraint wwv_flow_lov_location_ck
                                check ( location in ( 'STATIC', 'LOCAL', 'REMOTE', 'WEB_SOURCE' )),
    remote_server_id            number
                                constraint wwv_flow_lov_remote_server_fk
                                references wwv_remote_servers,
    web_src_module_id           number
                                constraint wwv_flow_lov_web_src_module_fk
                                references wwv_flow_web_src_modules,
    use_local_sync_table        varchar2(1)
                                constraint wwv_flow_lov_wsm_usesynctab_ck
                                check (use_local_sync_table in ('Y','N')),
    query_owner                 varchar2(128),
    query_table                 varchar2(128),
    query_where                 varchar2(4000),
    source_post_processing      varchar2(23),
                                constraint wwv_flow_lov_post_proc_ck
                                check ( source_post_processing in ( 'WHERE_ORDER_BY_CLAUSE', 'SQL', 'FUNC_BODY_RETURNING_SQL' )),
    optimizer_hint              varchar2(255),
    --
    remote_sql_caching          varchar2(9)
                                constraint wwv_flow_lov_rem_sql_cache_ck
                                check (remote_sql_caching in ( 'ALL_USERS', 'USER', 'SESSION' )),
    remote_sql_invalidate_when  varchar2(255),
    --
    external_filter_expr        varchar2(4000),
    external_order_by_expr      varchar2(255),
    --
    return_column_name          varchar2(128),
    display_column_name         varchar2(128),
    icon_column_name            varchar2(128),
    group_column_name           varchar2(128),
    group_sort_direction        varchar2(16)
                                constraint wwv_flow_lov_group_sort_dir_ck
                                check (group_sort_direction in ( 'ASC', 'ASC_NULLS_FIRST', 'DESC', 'DESC_NULLS_FIRST' )),
    default_sort_column_name    varchar2(128),
    default_sort_direction      varchar2(16)
                                constraint wwv_flow_lov_def_sort_dir_ck
                                check (default_sort_direction in ( 'ASC', 'ASC_NULLS_FIRST', 'DESC', 'DESC_NULLS_FIRST' ) ),
    oracle_text_column_name     varchar2(128),
    --
    last_updated_by             varchar2(255),
    last_updated_on             date,
    -- 4.0
    created_by                  varchar2(255),
    created_on                  date,
    lov_comment                 varchar2(4000)
    )
/

create index wwv_flow_lov_idx1 on wwv_flow_lists_of_values$ (flow_id, lov_name)
/
create unique index wwv_flow_lov_unique2 on wwv_flow_lists_of_values$ (flow_id, security_group_id, lov_name)
/
create index wwv_flow_lov_idx2 on wwv_flow_lists_of_values$ ( security_group_id, reference_id, id )
/
create index wwv_flow_lov_idx3 on wwv_flow_lists_of_values$ ( remote_server_id )
/
create index wwv_flow_lov_idx4 on wwv_flow_lists_of_values$ ( web_src_module_id )
/

--==============================================================================
create table wwv_flow_list_of_values_cols (
    id                            number
                                  constraint wwv_flow_lov_cols_pk
                                  primary key,
    lov_id                        number not null
                                  constraint wwv_flow_lov_cols_lov_fk
                                  references wwv_flow_lists_of_values$ on delete cascade,
    flow_id                       number not null,
    security_group_id             number not null,
    --
    query_column_name             varchar2(255) not null,
    heading                       varchar2(255),
    display_sequence              number not null,
    format_mask                   varchar2(255),
    data_type                     varchar2(30) not null
                                  constraint wwv_flow_lov_cols_data_type_ck
                                  check (data_type in ( 'VARCHAR2', 'NUMBER', 'DATE',
                                                        'TIMESTAMP', 'TIMESTAMP_TZ', 'TIMESTAMP_LTZ',
                                                        'INTERVAL_D2S', 'INTERVAL_Y2M',
                                                        'CLOB', 'ROWID' )),
    is_visible                    varchar2(1) not null
                                  constraint wwv_flow_lov_cols_is_vis_ck
                                  check (is_visible in ('Y', 'N')),
    is_searchable                 varchar2(1) not null
                                  constraint wwv_flow_lov_cols_search_ck
                                  check (is_searchable in ('Y', 'N')),
    --
    column_comment                varchar2(4000),
    last_updated_by               varchar2(255),
    last_updated_on               date,
    created_by                    varchar2(255),
    created_on                    date
    )
/

create index wwv_flow_lovc_idx1 on wwv_flow_list_of_values_cols ( lov_id )
/

alter table wwv_flow_list_of_values_cols add constraint wwv_flow_lovc_uk1 unique (lov_id, query_column_name)
/


--==============================================================================
create table wwv_flow_list_of_values_data (
    id                        number
                              constraint wwv_flow_lov_data_pk
                              primary key,
    lov_id                    number
                              constraint wwv_flow_lov_data_fk
                              references wwv_flow_lists_of_values$
                              on delete cascade,
    flow_id                   number,
    lov_disp_sequence         number not null,
    lov_disp_value            varchar2(4000) not null,
    lov_return_value          varchar2(4000) not null,
    lov_template              varchar2(4000),
    lov_disp_cond_type        varchar2(255),
    lov_disp_cond             varchar2(4000),
    lov_disp_cond2            varchar2(4000),
    required_patch            number,
    security_group_id         number not null,
    last_updated_by           varchar2(255),
    last_updated_on           date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    lov_data_comment          varchar2(4000)
    )
/

create unique index wwv_flow_lov_data_idx1 on wwv_flow_list_of_values_data (
    lov_id, security_group_id, lov_disp_sequence, id)
/

create index wwv_flow_lov_data_idx2 on wwv_flow_list_of_values_data (
    flow_id, id)
/

create index wwv_flow_lov_data_idx3 on wwv_flow_list_of_values_data (
    lov_id, substr(lov_return_value, 1, 500))
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > DATA LOADING
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > DATA LOADING

--==============================================================================
create table wwv_flow_load_tables (
    id                              number
                                    constraint wwv_flow_load_tab_pk
                                    primary key,
    flow_id                         number not null
                                    constraint wwv_flow_load_tab_fk1
                                    references wwv_flows
                                    on delete cascade,
    name                            varchar2(255) not null,
    static_id                       varchar2(255),
    --
    target_type                     varchar2(10)
                                    constraint wwv_load_tab_target_ck
                                    check (target_type in ('TABLE', 'COLLECTION' )),
    owner                           varchar2(255),
    table_name                      varchar2(255),
    collection_name                 varchar2(128),
    data_profile_id                 number
                                    constraint wwv_load_tab_profile_fk
                                    references wwv_flow_data_profiles
                                    on delete cascade,
    loading_method                  varchar2(7)
                                    constraint wwv_load_tab_method_ck
                                    check( loading_method in ( 'MERGE', 'APPEND', 'REPLACE' ) ),
    commit_interval                 number,
    error_handling                  varchar2(9)
                                    constraint wwv_load_tab_error_ck
                                    check( error_handling in ( 'ABORT', 'IGNORE', 'LOG_ROWS', 'ERROR_LOG' ) ),
    error_collection_name           varchar2(128),
    --
    unique_column_1                 varchar2(128),
    is_uk1_case_sensitive           varchar2(1)
                                    constraint wwv_flow_load_tab_ck1
                                    check (is_uk1_case_sensitive in ('Y','N')),
    unique_column_2                 varchar2(128),
    is_uk2_case_sensitive           varchar2(1)
                                    constraint wwv_flow_load_tab_ck2
                                    check (is_uk2_case_sensitive in ('Y','N')),
    unique_column_3                 varchar2(128),
    is_uk3_case_sensitive           varchar2(1)
                                    constraint wwv_flow_load_tab_ck3
                                    check (is_uk3_case_sensitive in ('Y','N')),
    wizard_page_ids                 varchar2(4000),
    version_column_name             varchar2(128),
    skip_validation                 varchar2(1) default 'N' not null
                                    constraint wwv_flow_load_tab_ck4
                                    check (skip_validation in ('Y','N')),
    column_names_lov_id             number
                                    constraint wwv_flow_load_tab_fk2
                                    references wwv_flow_lists_of_values$
                                    on delete set null,
    reference_id                    number,
    comments                        varchar2(4000),
    --
    security_group_id               number  not null,
    created_on                      date,
    created_by                      varchar2(255),
    last_updated_on                 date,
    last_updated_by                 varchar2(255) )
/

alter table wwv_flow_load_tables
    add constraint wwv_flow_load_tab_uk1 unique (flow_id, name)
/

-- for the unique index, map static IDs of NULL to the load ID. That will allow to have non-NULL static IDs
-- only once, but multiple NULL static IDs
create unique index wwv_flow_load_tab_uk2 on wwv_flow_load_tables (flow_id, nvl( static_id, to_char( id ) ) )
/

create index wwv_flow_load_tab_idx1 on wwv_flow_load_tables (column_names_lov_id)
/

create index wwv_flow_load_tab_dataprof_fx on wwv_flow_load_tables (data_profile_id) compress
/

create index wwv_flow_load_tab_ref_idx on wwv_flow_load_tables (reference_id) compress
/


--==============================================================================
create table wwv_flow_load_table_lookups (
    id                              number
                                        constraint wwv_flow_load_tab_lk_pk
                                        primary key,
    flow_id                         number not null
                                    constraint wwv_flow_load_tab_lk_fk1
                                    references wwv_flows
                                    on delete cascade,
    load_table_id                   number  not null
                                    constraint wwv_flow_load_tab_lk_fk2
                                    references wwv_flow_load_tables
                                    on delete cascade,
    load_column_name                varchar2 (255) not null,
    lookup_owner                    varchar2 (255) not null,
    lookup_table_name               varchar2 (255) not null,
    key_column                      varchar2 (255) not null,
    display_column                  varchar2 (255) not null,
    alternative_key_column1         varchar2 (255) ,
    alternative_display_column1     varchar2 (255) ,
    alternative_key_column2         varchar2 (255) ,
    alternative_display_column2     varchar2 (255) ,
    where_clause                    varchar2 (4000),
    insert_new_value                varchar2 (1)
                                        constraint wwv_flow_load_tab_lk_ck1
                                        check (insert_new_value in ('Y','N')) not null,
    error_message                   varchar2(4000) ,
    security_group_id               number  not null ,
    created_on                      date ,
    created_by                      varchar2 (255) ,
    last_updated_on                 date ,
    last_updated_by                 varchar2 (255) )
/
alter table wwv_flow_load_table_lookups
    add constraint wwv_flow_load_tab_lk_uk1 unique ( load_table_id , load_column_name )
/
create index wwv_flow_load_tab_lk_idx2 on wwv_flow_load_table_lookups (flow_id) compress
/

--==============================================================================
create table wwv_flow_load_table_rules (
    id                              number    not null
                                        constraint wwv_flow_load_tab_rule_pk
                                        primary key,
    flow_id                         number    not null
                                        constraint wwv_flow_load_tab_rule_fk1
                                        references wwv_flows on delete cascade,
    load_table_id                   number    not null
                                        constraint wwv_flow_load_tab_rule_fk2
                                        references wwv_flow_load_tables on delete cascade,
    load_column_name                varchar2(255)   not null,
    rule_name                       varchar2(255)   not null,
    rule_type                       varchar2(255)
                                        constraint wwv_flow_load_tab_rule_ck1
                                        check (rule_type in (
                                                'TO_UPPER_CASE',
                                                'TO_LOWER_CASE',
                                                'REPLACE',
                                                'TRIM','LTRIM','RTRIM',
                                                'SINGLE_WHITESPACES',
                                                'PLSQL_EXPRESSION',
                                                'REGULAR_EXPRESSION',
                                                'PLSQL_FUNCTION_BODY',
                                                'SQL_QUERY_SINGLE_VALUE',
                                                'SQL_QUERY_SEMI_COLON' )) not null,
    rule_sequence                   number   not null,
    rule_expression1                varchar2(4000),
    rule_expression2                varchar2(4000),
    error_message                   varchar2(4000),
    security_group_id               number          not null,
    created_by                      varchar2(255),
    created_on                      date,
    last_updated_by                 varchar2(255),
    last_updated_on                 date )
/

alter table wwv_flow_load_table_rules
    add constraint wwv_flow_load_tab_rule_uk1 unique (load_table_id, load_column_name, rule_name)
/
create index wwv_flow_load_tab_rule_idx2 on wwv_flow_load_table_rules (flow_id) compress
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > USER INTERFACE
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > USER INTERFACE

--==============================================================================
create table wwv_flow_user_interfaces (
    id                           number         not null
                                                constraint wwv_flow_ui_pk
                                                primary key,
    flow_id                      number         not null
                                                constraint wwv_flow_ui_flow_fk
                                                references wwv_flows on delete cascade,
    ui_type_id                   number         not null
                                                constraint wwv_flow_ui_type_fk
                                                references wwv_flow_ui_types,
    display_name                 varchar2(30)   not null, /* used for our switch placeholder -> #APEX_SWITCH_UI_[SMARTPHONE]# and on the auto detection page */
    display_seq                  number         not null,
    use_auto_detect              varchar2(1)    not null
                                                constraint wwv_flow_ui_use_auto_detect_ck
                                                   check ( use_auto_detect in ('Y', 'N' ) ),
    is_default                   varchar2(1)    not null  /* Default user interface if not other UI is detected by the auto detect */
                                                constraint wwv_flow_ui_is_default_ck
                                                   check ( is_default in ('Y', 'N' ) ),
    theme_id                     number         not null, /* No FK to be consistent with other tables which are referencing a theme */
    home_url                     varchar2(2000) not null,
    login_url                    varchar2(2000),
    global_page_id               number,        /* FK is created after wwv_flow_steps has been created */
    navigation_list_id           number         constraint wwv_flow_ui_nav_list_fk
                                                references wwv_flow_lists
                                                deferrable initially deferred,
    navigation_list_position     varchar2(20)   constraint wwv_flow_ui_list_position_ck
                                                    check ( navigation_list_position in ('TOP','SIDE') ),
    navigation_list_template_id  number         constraint wwv_flow_ui_list_template_fk
                                                    references wwv_flow_list_templates
                                                    deferrable initially deferred,
    nav_list_template_options    varchar2(255)  default '#DEFAULT#',
    javascript_file_urls         varchar2(4000),
    css_file_urls                varchar2(4000),
    include_legacy_javascript    varchar2(255),
    include_jquery_migrate       varchar2(1)    default 'Y' not null -- Set default to Y so that during migration of schema existing apps include jQuery migrate
                                                constraint wwv_flow_ui_inc_jq_migr_ck
                                                   check (include_jquery_migrate in ( 'Y', 'N' )),
    required_patch               number,
    --
    nav_bar_type                 varchar2(20)   default 'NAVBAR' not null
                                                constraint wwv_flow_ui_nav_position_ck
                                                    check (nav_bar_type in ('NAVBAR','LIST')),
    nav_bar_list_id              number         constraint wwv_flow_ui_navbar_list_fk
                                                    references wwv_flow_lists
                                                    deferrable initially deferred,
    nav_bar_list_template_id     number         constraint wwv_flow_ui_navbar_temp_fk
                                                    references wwv_flow_list_templates
                                                    deferrable initially deferred,
    nav_bar_template_options     varchar2(255)  default '#DEFAULT#',
    --
    theme_style_by_user_pref     varchar2(1)    default 'N' not null
                                                constraint wwv_flow_ui_thesty_usrpref_ck
                                                    check ( theme_style_by_user_pref in ('Y', 'N') ),
    built_with_love              varchar2(1)    default 'Y' not null
                                                constraint wwv_flow_ui_built_with_love_ck
                                                    check ( built_with_love in ('Y', 'N') ),
    --
    security_group_id            number         not null,
    created_by                   varchar2(255),
    created_on                   date,
    last_updated_by              varchar2(255),
    last_updated_on              date
    );

alter table wwv_flow_user_interfaces add constraint wwv_flow_user_int_type_uk unique (flow_id, ui_type_id)
/

create unique index wwv_flow_user_int_idx1 on wwv_flow_user_interfaces (flow_id, display_name)
/

create index wwv_flow_user_int_idx2 on wwv_flow_user_interfaces (ui_type_id)
/

create index wwv_flow_user_int_idx3 on wwv_flow_user_interfaces (required_patch)
/

create index wwv_flow_user_int_idx4 on wwv_flow_user_interfaces (flow_id, global_page_id)
/

create index wwv_flow_user_int_idx5 on wwv_flow_user_interfaces (navigation_list_id)
/

create index wwv_flow_user_int_idx6 on wwv_flow_user_interfaces( navigation_list_template_id )
/

create index wwv_flow_user_int_idx7 on wwv_flow_user_interfaces (nav_bar_list_id)
/

create index wwv_flow_user_int_idx8 on wwv_flow_user_interfaces( nav_bar_list_template_id )
/

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > USER INTERFACE > COMBINED FILES
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > USER INTERFACE > COMBINED FILES

--==============================================================================
create table wwv_flow_combined_files (
    id                        number         not null
                                             constraint wwv_flow_combined_files_pk
                                             primary key,
    flow_id                   number         not null
                                             constraint wwv_flow_comb_files_flow_fk
                                             references wwv_flows on delete cascade,
    user_interface_id         number         not null
                                             constraint wwv_flow_comb_files_ui_fk
                                             references wwv_flow_user_interfaces on delete cascade,
    page_id                   number,        /* FK is created after wwv_flow_steps has been created */
    combined_file_url         varchar2(255)  not null,
    single_file_urls          varchar2(4000) not null,
    required_patch            number,
    security_group_id         number         not null,
    created_by                varchar2(255),
    created_on                date,
    last_updated_by           varchar2(255),
    last_updated_on           date
  );

alter table wwv_flow_combined_files
  add constraint wwv_flow_combined_files_uk unique (user_interface_id, combined_file_url, page_id);

create index wwv_flow_combined_files_idx1 on wwv_flow_combined_files (flow_id, user_interface_id, page_id);
create index wwv_flow_combined_files_idx2 on wwv_flow_combined_files (security_group_id);
create index wwv_flow_combined_files_idx3 on wwv_flow_combined_files (required_patch);
create index wwv_flow_combined_files_idx4 on wwv_flow_combined_files (flow_id, page_id);


--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > APPLICATION SETTINGS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > APPLICATION SETTINGS

--==============================================================================
create table wwv_flow_app_settings (
    id                             number not null
                                   constraint wwv_flow_appset_pk primary key,
    security_group_id              number not null
                                   constraint wwv_flow_appset_sgid_fk
                                   references wwv_flow_companies( provisioning_company_id )
                                   on delete cascade,
    flow_id                        number not null
                                   constraint wwv_flow_appset_app_fk
                                   references wwv_flows
                                   on delete cascade,
    required_patch                 number,
    name                           varchar2(255) not null,
    value                          varchar2(4000),
    is_required                    varchar2(1) default 'N' not null
                                   constraint wwv_flow_appset_req_ck
                                   check (is_required in ('Y', 'N')),
    valid_values                   varchar2(4000),
    on_upgrade_keep_value          varchar2(1) default 'N' not null
                                   constraint wwv_flow_appset_keep_value_ck check (on_upgrade_keep_value in ('Y', 'N')),
    comments                       varchar2(4000),
    --
    created_on                     date,
    created_by                     varchar2(255),
    last_updated_on                date,
    last_updated_by                varchar2(255)
    )
/

alter table wwv_flow_app_settings add constraint wwv_flow_appset_name_uk unique ( security_group_id, flow_id, name );

create index wwv_flow_app_settings_sgid_fx on wwv_flow_app_settings( security_group_id ) compress;
create index wwv_flow_app_settings_app_fx  on wwv_flow_app_settings( flow_id );

--##############################################################################
--#
--# APPLICATION > SHARED COMPONENTS > AUTOMATIONS
--#
--##############################################################################
prompt APPLICATION > SHARED COMPONENTS > AUTOMATIONS

--==============================================================================
create table wwv_flow_automations (
    id                          number          not null
                                                constraint wwv_am_pk
                                                primary key,
    flow_id                     number          not null
                                                constraint wwv_am_flow_fk
                                                references wwv_flows on delete cascade,
    security_group_id           number          not null,
    --
    name                        varchar2(255)   not null,
    static_id                   varchar2(255)   not null,
    --
    trigger_type                varchar2(7)     not null
                                constraint wwv_am_trigger_type_ck
                                check (trigger_type in ( 'POLLING', 'API' ) ),
    --
    polling_interval            varchar2(255),
    polling_last_run_timestamp  timestamp with time zone,
    polling_next_run_timestamp  timestamp with time zone,
    polling_status              varchar2(8)
                                constraint wwv_am_poll_status_ck
                                check ( polling_status in ( 'ACTIVE', 'DISABLED', 'ERROR' )),
    --
    result_type                 varchar2(27)
                                constraint wwv_am_result_type_ck
                                check (result_type in ('ROWS', 'NO_ROWS', 'FUNC_BODY_RETURNING_BOOLEAN', 'ALWAYS')),
    location                    varchar2(10)
                                constraint wwv_am_location_ck
                                check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE' )),
    remote_server_id            number
                                constraint wwv_am_remote_srv_fk
                                references wwv_remote_servers,
    web_src_module_id           number
                                constraint wwv_am_websrc_mod_fk
                                references wwv_flow_web_src_modules,
    use_local_sync_table        varchar2(1)
                                constraint wwv_am_wsm_usesync_ck
                                check (use_local_sync_table in ('Y','N')),
    query_type                  varchar2(23)
                                constraint wwv_am_query_type_ck
                                check (query_type in ('TABLE', 'SQL', 'FUNC_BODY_RETURNING_SQL' )),
    query_source                clob,
    function_body_language      varchar2(10)
                                constraint wwv_am_func_body_lang_ck
                                check (function_body_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    query_owner                 varchar2(128),
    query_table                 varchar2(128),
    query_where                 varchar2(4000),
    query_order_by              varchar2(255),
    source_post_processing      varchar2(23)
                                constraint wwv_am_trg_src_post_proc_ck
                                check ( source_post_processing in ( 'WHERE_ORDER_BY_CLAUSE', 'SQL', 'FUNC_BODY_RETURNING_SQL' )),
    --
    include_rowid_column        varchar2(1)
                                constraint wwv_am_incl_rowid_ck
                                check (include_rowid_column in ('Y','N')),
    optimizer_hint              varchar2(255),
    --
    external_filter_expr        varchar2(4000),
    external_order_by_expr      varchar2(255),
    --
    pk_column_name              varchar2(128),
    max_rows_to_process         number,
    commit_each_row             varchar2(1)     not null
                                constraint wwv_am_commit_row_ck
                                check ( commit_each_row in ('Y','N')),
    error_handling_type         varchar2(7)
                                constraint wwv_am_error_ck
                                check ( error_handling_type in ( 'IGNORE', 'ABORT', 'DISABLE' )),
    --
    code_snippet                clob,
    init_proc_name              varchar2(255),
    before_processing_proc_name varchar2(255),
    teardown_proc_name          varchar2(255),
    --
    condition_type              varchar2(255),
    condition_expr1             varchar2(4000),
    condition_expr2             varchar2(4000),
    --
    build_option_id             number,
    automation_comment          varchar2(4000),
    --
    created_by                  varchar2(255),
    created_on                  date,
    last_updated_by             varchar2(255),
    last_updated_on             date
    )
    lob (query_source, code_snippet) store as (cache reads enable storage in row)
/
alter table wwv_flow_automations add constraint wwv_am_name_uk unique (flow_id, name) using index compress 1
/
alter table wwv_flow_automations add constraint wwv_am_static_id_uk unique (flow_id, static_id) using index compress 1
/
create index wwv_am_websrc_modul_fx   on wwv_flow_automations (web_src_module_id) compress
/
create index wwv_am_remote_server_fx  on wwv_flow_automations (remote_server_id) compress
/
create index wwv_am_nextpolling_ts_ix on wwv_flow_automations (polling_next_run_timestamp) compress
/

--==============================================================================
create table wwv_flow_automation_actions (
    id                      number
                            constraint wwv_am_action_pk
                            primary key,
    flow_id                 number not null
                            constraint wwv_am_action_flow_fk
                            references wwv_flows
                            on delete cascade,
    security_group_id       number          not null,
    automation_id           number          not null
                            constraint wwv_am_action_parent_fk
                            references wwv_flow_automations
                            on delete cascade,
    --
    name                    varchar2(255)   not null,
    execution_sequence      number          not null,
    action_type             varchar2(255)   not null
                            constraint wwv_am_action_type_ck
                            check (
                                action_type like 'NATIVE\_%' escape '\' or
                                action_type like 'PLUGIN\_%' escape '\'),
    --
    action_clob             clob,
    action_clob_language    varchar2(10)
                            constraint wwv_am_action_clob_lang_ck
                            check (action_clob_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    attribute_01            varchar2(4000),
    attribute_02            varchar2(4000),
    attribute_03            varchar2(4000),
    attribute_04            varchar2(4000),
    attribute_05            varchar2(4000),
    attribute_06            varchar2(4000),
    attribute_07            varchar2(4000),
    attribute_08            varchar2(4000),
    attribute_09            varchar2(4000),
    attribute_10            varchar2(4000),
    attribute_11            varchar2(4000),
    attribute_12            varchar2(4000),
    attribute_13            varchar2(4000),
    attribute_14            varchar2(4000),
    attribute_15            varchar2(4000),
    --
    location                varchar2(10)
                            constraint wwv_am_action_location_ck
                            check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE' )),
    remote_server_id        number
                            constraint wwv_am_action_remote_server_fk
                            references wwv_remote_servers,
    web_src_module_id       number
                            constraint wwv_am_action_websrc_module_fk
                            references wwv_flow_web_src_modules,
    web_src_operation_id    number
                            constraint wwv_am_action_websrc_opers_fk
                            references wwv_flow_web_src_operations,
    email_template_id       number generated always as (
                                                         case when action_type = 'NATIVE_SEND_EMAIL' and attribute_11 is not null
                                                             then to_number( attribute_11 )
                                                         end ) virtual,
    --
    error_message           varchar2(4000),
    stop_execution_on_error varchar2(1) not null
                            constraint wwv_am_action_stop_on_error_ck
                            check (stop_execution_on_error in ( 'Y', 'N' )),
    --
    condition_type          varchar2(255),
    condition_expr1         varchar2(4000),
    condition_expr2         varchar2(4000),
    exec_cond_for_each_row  varchar2(1)
                            constraint wwv_am_action_cond_each_row_ck
                            check (exec_cond_for_each_row in ('Y', 'N')),
    --
    build_option_id         number,
    action_comment          varchar2(4000),
    --
    created_by              varchar2(255),
    created_on              date,
    last_updated_by         varchar2(255),
    last_updated_on         date,
    --
    constraint wwv_am_action_mail_templ_fk foreign key( email_template_id )
                                           references wwv_flow_email_templates
                                           deferrable initially deferred
    )
    lob (action_clob) store as (cache reads enable storage in row)
/
alter table wwv_flow_automation_actions add constraint wwv_am_action_name_uk unique (flow_id, automation_id, name) using index compress 2
/
alter table wwv_flow_automation_actions add constraint wwv_am_act_websrc_mod_ooper_ck
    check ( web_src_module_id is null or web_src_operation_id is null )
/

create index wwv_am_action_parent_fx     on wwv_flow_automation_actions (automation_id)
/
create index wwv_am_act_websrc_mod_fx     on wwv_flow_automation_actions (web_src_module_id)       compress
/
create index wwv_am_act_websrc_opers_fx   on wwv_flow_automation_actions (web_src_operation_id)    compress
/
create index wwv_am_act_remote_server_fx  on wwv_flow_automation_actions (remote_server_id)        compress
/
create index wwv_am_act_email_templ_fx    on wwv_flow_automation_actions (email_template_id)       compress
/

--##############################################################################
--#
--# APPLICATION > PAGE
--#
--##############################################################################
prompt APPLICATION > PAGE

--==============================================================================
create table wwv_flow_steps (
    id                  number,
    flow_id             number not null
                        constraint wwv_flow_step_flow_fk
                        references wwv_flows on delete cascade,
    user_interface_id   number /* $$$ should be not null */
                        constraint wwv_flow_step_ui_fk
                        references wwv_flow_user_interfaces on delete cascade,
    tab_set             varchar2(255),
    name                varchar2(255) not null,
    alias               varchar2(255),
    page_component_map  varchar2(255),
    page_mode           varchar2(20) default 'NORMAL' not null
                        constraint wwv_flow_steps_page_mode
                        check (page_mode in ('NORMAL','MODAL','NON_MODAL')),
    --
    step_title          varchar2(255),
    first_item                varchar2(255)
                              constraint wwv_fsteps_first_item_ck
                              check (first_item in ('NO_FIRST_ITEM', 'AUTO_FIRST_ITEM')),
    include_apex_css_js_yn    varchar2(1),
    welcome_text              varchar2(4000),
    box_welcome_text          varchar2(4000),
    footer_text               varchar2(4000),
    help_text                 clob,
    step_template             number,
    page_css_classes          varchar2(255),
    page_template_options     varchar2(255) default '#DEFAULT#',
    required_role             varchar2(255),
    required_patch            number,
    html_page_header          clob,
    html_page_onload          varchar2(4000),
    javascript_file_urls      varchar2(4000),
    javascript_code           clob,
    javascript_code_onload    clob,
    css_file_urls             varchar2(4000),
    inline_css                clob,
    allow_duplicate_submissions varchar2(1) default 'Y' not null
                              constraint valid_all_dup_page_sub
                              check (allow_duplicate_submissions in (
                              'Y',
                              'N')),
    on_dup_submission_goto_url  varchar2(4000),
    reload_on_submit          varchar2(1) default 'A'
                              constraint wwv_flow_step_reload_submit_ck
                                  check (reload_on_submit in ('A', 'S')), /* A=Always, S=Only for Success, D=On Demand (Future) */
    warn_on_unsaved_changes   varchar2(1) default 'N'
                              constraint wwv_flow_step_warn_unsaved_ck
                              check (warn_on_unsaved_changes in ('Y','N')),
    page_is_public_y_n        varchar2(1) default 'N' not null
                              constraint wwv_flow_pages_pgpub_flag
                              check (page_is_public_y_n in ('Y','N')),
    protection_level          varchar2(1) default 'N' not null
                              constraint valid_fsteps_protection_level
                              check (protection_level in ('N','C','U','D')),
    browser_cache             varchar2(1)
                              constraint wwv_fsteps_ck_browser_cache
                              check (browser_cache in ('Y','N')),
    deep_linking              varchar2(1)
                              constraint wwv_fsteps_ck_deep_linking
                              check (deep_linking in ('Y','N')),
    rejoin_existing_sessions  varchar2(1)
                              constraint wwv_flow_step_rejoin_es_ck
                              check (rejoin_existing_sessions in ('Y', 'P', 'N')),
    security_group_id         number not null,
    error_handling_function   varchar2(255),
    error_notification_text   varchar2(4000),
    autocomplete_on_off       varchar2(3) default 'ON' not null
                              constraint wwv_flow_step_autocomplete_ck
                                  check (autocomplete_on_off in ('ON', 'OFF')),
    page_transition           varchar2(20)
                              constraint wwv_flow_step_page_trans_ck
                                   check ( page_transition in ('NONE', 'FADE', 'POP', 'FLIP', 'TURN', 'FLOW', 'SLIDEFADE', 'SLIDE', 'SLIDEUP', 'SLIDEDOWN') ),
    popup_transition          varchar2(20)
                              constraint wwv_flow_step_popup_trans_ck
                                   check ( popup_transition in ('NONE', 'FADE', 'POP', 'FLIP', 'TURN', 'FLOW', 'SLIDEFADE', 'SLIDE', 'SLIDEUP', 'SLIDEDOWN') ),
    read_only_when_type       varchar2(255),
    read_only_when            varchar2(4000),
    read_only_when2           varchar2(4000),
    --
    cache_mode                varchar2(7) default 'NOCACHE' not null
                              constraint wwv_fsteps_ck_cache_mode
                              check (cache_mode in ('NOCACHE',
                                                    'CACHE',
                                                    'USER',
                                                    'SESSION')),
    cache_timeout_seconds     number,
    cache_when_condition_type varchar2(255),
    cache_when_condition_e1   varchar2(4000),
    cache_when_condition_e2   varchar2(4000),
    --
    dialog_width              varchar2(20),
    dialog_max_width          varchar2(20),
    dialog_height             varchar2(20),
    dialog_title              varchar2(255),
    dialog_attributes         varchar2(255),
    dialog_css_classes        varchar2(255),
    dialog_chained            varchar2(1) default 'Y'
                              constraint wwv_flow_step_chained
                              check (dialog_chained in ('Y','N')),
    --
    overwrite_navigation_list   varchar2(1) default 'N' not null
                                    constraint wwv_flow_step_lst_overwrite
                                    check (overwrite_navigation_list in ('Y','N')),
    navigation_list_id          number
                                     constraint wwv_flow_step_nav_list_fk
                                     references wwv_flow_lists deferrable initially deferred,
    navigation_list_template_id number
                                     constraint wwv_flow_step_nav_list_tmpl_fk
                                     references wwv_flow_list_templates deferrable initially deferred,
    navigation_list_position    varchar2(20)
                                    constraint wwv_flow_steps_nav_pos_ck
                                    check ( navigation_list_position in ('TOP','SIDE') ),
    nav_list_template_options   varchar2(255 )default '#DEFAULT#',
    --
    created_on                date,
    created_by                varchar2(255),
    --
    last_updated_by           varchar2(255),
    last_updated_on           date,
    --
    page_components_scn       number default 1 not null,
    --
    group_id                  number,
    media_type                varchar2(255),
    page_comment              varchar2(4000),
    --
    -- obsolete, but needed for upgrade
    --
    cache_page_yn             varchar2(1),
    cache_by_user_yn          varchar2(1),
    -- constraints
    constraint wwv_flow_steps_pk primary key (flow_id, id)
    )    lob (html_page_header, javascript_code, javascript_code_onload) store as (cache reads enable storage in row)
/

prompt ...create indexes on wwv_flow_steps table
create unique index wwv_flow_page_id_idx2  on wwv_flow_steps (id, flow_id, security_group_id) compress
/
create unique index wwv_flow_page_alias_idx1  on wwv_flow_steps (flow_id, nvl( alias, 'null$' || to_char(flow_id) || '$' || to_char(id))) compress
/
create index wwv_flow_page_public_idx3  on wwv_flow_steps (flow_id, page_is_public_y_n, id) compress
/
create index wwv_flow_page_protection_idx4  on wwv_flow_steps (flow_id, protection_level, id) compress
/
create index wwv_flow_page_ui_idx5  on wwv_flow_steps (user_interface_id) compress
/
create index wwv_flow_page_nv_lst_idx6  on wwv_flow_steps (navigation_list_id) compress
/
create index wwv_flow_page_nv_lst_tmpl_idx7  on wwv_flow_steps (navigation_list_template_id) compress
/

--==============================================================================
alter table wwv_flow_user_interfaces
  add constraint wwv_flow_user_int_page_fk
      foreign key ( flow_id, global_page_id )
      references wwv_flow_steps (flow_id, id)
      on delete set null
      -- deferrable is necessary for importing an app because user interface rows will be created before pages
      deferrable initially deferred;

alter table wwv_flow_combined_files
  add constraint wwv_flow_comb_file_page_fk
      foreign key ( flow_id, page_id )
      references wwv_flow_steps (flow_id, id)
      -- deferrable is necessary for importing an app because combined files will be created before pages
      on delete cascade deferrable initially deferred;

--==============================================================================
create table wwv_flow_page_groups (
    id                number
                      constraint wwv_flow_page_grp_pk
                      primary key,
    flow_id           number
                      references wwv_flows
                      on delete cascade,
    group_name        varchar2(255) not null,
    group_desc        varchar2(4000),
    security_group_id number not null)
/

create index wwv_flow_page_groups_idx on wwv_flow_page_groups (flow_id)
/

--##############################################################################
--#
--# APPLICATION > PAGE > REGION (GENERIC)
--#
--##############################################################################
prompt APPLICATION > PAGE > REGION (GENERIC)

--==============================================================================
create table wwv_flow_page_plugs (
    id                           number
                                 constraint wwv_flow_page_plugs_pk
                                 primary key,
    flow_id                      number not null
                                 constraint wwv_flow_plug_to_flow_fk
                                 references wwv_flows
                                 on delete cascade,
    page_id                      number not null,
    plug_name                    varchar2(255) not null,
    region_name                  varchar2(255),
    parent_plug_id               number
                                 constraint wwv_flow_plug_parent_fk
                                 references wwv_flow_page_plugs
                                 on delete cascade deferrable initially deferred,
                                 -- deferrable is necessary for page copy, because the order of the regions is undefined
    filtered_region_id           number
                                 constraint wwv_flow_filter_plug_fk
                                 references wwv_flow_page_plugs
                                 on delete cascade deferrable initially deferred,
    --
    plug_template                number,
    plug_display_sequence        number not null,
    plug_new_grid                varchar2(1) default 'N',
                                 constraint wwv_plug_new_grid
                                 check (plug_new_grid in ('Y', 'N')),
    plug_new_grid_row            varchar2(1) default 'N',
                                 constraint wwv_plug_new_grid_row
                                 check (plug_new_grid_row in ('Y', 'N')),
    plug_grid_row_css_classes    varchar2(255),
    plug_new_grid_column         varchar2(1) default 'N',
                                 constraint wwv_plug_new_grid_column
                                 check (plug_new_grid_column in ('Y', 'N')),
    plug_display_column          number
                                 constraint wwv_valid_plug_disp_column
                                 check (plug_display_column between 1 and 99),
    plug_grid_column_span        number
                                 constraint wwv_plug_grid_col_span
                                 check (plug_grid_column_span between 1 and 99),
    plug_grid_column_css_classes varchar2(255),
    plug_display_point           varchar2(255) default 'BODY' not null,
    plug_item_display_point      varchar2(5) default 'ABOVE' not null
                                 constraint wwv_plug_item_disp_point
                                 check (plug_item_display_point in ('ABOVE', 'BELOW')),
    --
    plug_source_type             varchar2(255) not null
                                 constraint wwv_flow_page_plugs_ck_srctype
                                 check (substr(plug_source_type,1,7) in ('NATIVE_','PLUGIN_')),
    --
    location                     varchar2(10)
                                 constraint wwv_plug_location_ck
                                 check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE' )),
    remote_server_id             number
                                 constraint wwv_plug_remote_server_fk
                                 references wwv_remote_servers,
    web_src_module_id            number
                                 constraint wwv_plug_web_src_module_fk
                                 references wwv_flow_web_src_modules,
    use_local_sync_table         varchar2(1)
                                 constraint wwv_plug_wsm_usesynctab_ck
                                 check (use_local_sync_table in ('Y','N')),
    query_type                   varchar2(23)
                                 constraint wwv_plug_query_type_ck
                                 check (query_type in ('TABLE', 'SQL', 'FUNC_BODY_RETURNING_SQL' )),
    function_body_language       varchar2(10)
                                 constraint wwv_plug_func_body_lang_ck
                                 check (function_body_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    plug_source                  clob,
    query_owner                  varchar2(128),
    query_table                  varchar2(128),
    query_where                  varchar2(4000),
    query_order_by_type          varchar2(6)
                                 constraint wwv_plug_orderby_type_ck
                                 check ( query_order_by_type in ( 'ITEM', 'STATIC' ) ),
    query_order_by               varchar2(4000),
    source_post_processing       varchar2(23)
                                 constraint wwv_plug_src_post_proc_ck
                                 check ( source_post_processing in ( 'WHERE_ORDER_BY_CLAUSE', 'SQL', 'FUNC_BODY_RETURNING_SQL' )),
    --
    include_rowid_column         varchar2(1)
                                 constraint wwv_plug_incl_rowid_ck
                                 check (include_rowid_column in ('Y','N')),
    optimizer_hint               varchar2(255),
    --
    remote_sql_caching           varchar2(9)
                                 constraint wwv_plug_remote_sql_caching_ck
                                 check (remote_sql_caching in ('ALL_USERS', 'USER', 'SESSION')),
    remote_sql_invalidate_when   varchar2(255),
    --
    external_filter_expr         varchar2(4000),
    external_order_by_type       varchar2(6)
                                 constraint wwv_plug_extorderby_type_ck
                                 check ( external_order_by_type in ( 'ITEM', 'STATIC' ) ),
    external_order_by_expr       varchar2(4000),
    -- Standard Attributes for editable regions
    is_editable                  varchar2(1),
                                 constraint wwv_plug_is_editable_ck
                                 check ( is_editable in ( 'Y', 'N' )),
    edit_operations              varchar2(5)
                                 constraint wwv_plug_edit_operations_ck
                                 check ( regexp_instr( edit_operations, '[^iud:]+') = 0 ), /* only allow i, u, d, : */
    edit_row_operations_column   varchar2(128),
    add_authorization_scheme     varchar2(40),
    update_authorization_scheme  varchar2(40),
    delete_authorization_scheme  varchar2(40),
    lost_update_check_type       varchar2(6)
                                 constraint wwv_plug_lost_update_type_ck
                                 check ( lost_update_check_type in ( 'VALUES', 'COLUMN' )),
    row_version_column           varchar2(128),
    add_row_if_empty             varchar2(1)
                                 constraint wwv_plug_add_row_if_empty_ck
                                 check ( add_row_if_empty in ( 'Y', 'N' )),
    lazy_loading                 varchar2(1),
                                 constraint wwv_plug_lazy_loading_ck
                                 check ( lazy_loading in ( 'Y', 'N' )),
    --
    menu_id                      number
                                 constraint wwv_flow_plug_menu_fk
                                 references wwv_flow_menus(id)
                                 deferrable initially deferred,
    list_id                      number
                                 constraint wwv_flow_plug_list_fk
                                 references wwv_flow_lists(id)
                                 deferrable initially deferred,
    menu_template_id             number,
    list_template_id             number,
    -- Master - Detail Relationship
    master_region_id             number
                                 constraint wwv_flow_page_plugs_master_fk
                                 references wwv_flow_page_plugs
                                 deferrable initially deferred,
    --
    include_in_reg_disp_sel_yn   varchar2(1) not null
                                 constraint wwv_plug_incl_reg_disp_sel_ck
                                 check (include_in_reg_disp_sel_yn in ('Y','N')),
    --
    plug_required_role           varchar2(255),
    plug_display_condition_type  varchar2(255),
    plug_display_when_condition  varchar2(4000),
    plug_display_when_cond2      varchar2(4000),
    --
    plug_read_only_when_type     varchar2(255),
    plug_read_only_when          varchar2(4000),
    plug_read_only_when2         varchar2(4000),
    --
    plug_header                  varchar2(4000),
    plug_footer                  varchar2(4000),
    region_image                 varchar2(255),
    region_image_attr            varchar2(255),
    region_image_alt_text        varchar2(255),
    escape_on_http_output        varchar2(1) default 'Y'
                                 constraint wwv_plug_escape_on_http_outp
                                 check (escape_on_http_output in ('Y','N')),
    --
    plug_query_parse_override    varchar2(255),
    plug_query_row_template      number,
    plug_query_headings          varchar2(4000),
    plug_query_headings_type     varchar2(255)
                                 constraint wwv_plug_head_type_check
                                 check (plug_query_headings_type in (
                                 'NO_HEADINGS',
                                 'SORT_ENABLED_COLON_DELMITED_LIST',
                                 'SORT_ENABLED_QUERY_COLUMNS',
                                 'SORT_ENABLED_QUERY_COLUMNS_INITCAP',
                                 'COLON_DELMITED_LIST',
                                 'COLON_DELIMITED_LIST',
                                 'PIPE_DELMITED_LIST',
                                 'BACKSLASH_DELMITED_LIST',
                                 'QUERY_COLUMNS',
                                 'QUERY_COLUMNS_INITCAP',
                                 'FUNCTION_BODY_RETURNING_COLON_DELIMITED_LIST',
                                 'FUNCTION_BODY_RETURNING_PIPE_DELIMITED_LIST')),
    plug_query_num_rows          number,
    plug_query_num_rows_type     varchar2(255)
                                 constraint wwv_plug_query_num_rows_type
                                 check (plug_query_num_rows_type in (
                                 'NEXT_PREVIOUS_LINKS',
                                 'COMPUTED_BUT_NOT_DISPLAYED',
                                 'ROW_RANGES',
                                 'ROW_RANGES_RIGHT',
                                 'ROW_RANGES_WITH_LINKS_RIGHT',
                                 'ROW_RANGES_WITH_LINKS',
                                 'ROW_RANGES_IN_SELECT_LIST',
                                 'ROW_RANGES_IN_SELECT_LIST_RIGHT',
                                 'ROWS_PER_PAGE_FROM_ITEM',
                                 'ROWS_X_TO_Y_OF_Z',
                                 'ROWS_X_TO_Y',
                                 'SEARCH_ENGINE',
                                 'SEARCH_ENGINE_RIGHT',
                                 'SET',
                                 'SCROLL',
                                 '0'
                                 )),
    show_total_row_count         varchar2(1)
                                 constraint wwv_plug_show_total_row_cnt_ck
                                 check ( show_total_row_count in ( 'Y', 'N' )),
    pagination_display_position  varchar2(255),
    ajax_enabled                 varchar2(1),
    ajax_items_to_submit         varchar2(4000),
    rest_enabled                 varchar2(1)
                                 constraint wwv_plug_rest_chk
                                 check(rest_enabled in ('Y','N')),
    plug_query_num_rows_item     varchar2(255),
    plug_query_no_data_found     varchar2(4000),
    no_data_found_icon_classes   varchar2(255),
    plug_query_more_data         varchar2(4000),
    plug_query_row_count_max     number,
    --
    plug_query_format_out        varchar2(255),
    plug_query_show_nulls_as     varchar2(255),
    plug_query_col_allignments   varchar2(255),
    plug_query_break_cols        varchar2(255),
    plug_query_sum_cols          varchar2(255),
    plug_query_number_formats    varchar2(4000),
    plug_query_table_border      varchar2(255),
    plug_query_hit_highlighting  varchar2(4000),
    plug_query_asc_image         varchar2(4000),
    plug_query_asc_image_attr    varchar2(4000),
    plug_query_desc_image        varchar2(4000),
    plug_query_desc_image_attr   varchar2(4000),
    --
    plug_query_exp_filename      varchar2(255),
    plug_query_exp_separator     varchar2(1),
    plug_query_exp_enclosed_by   varchar2(1),
    plug_query_strip_html        varchar2(1)
                                 constraint wwv_plug_strip_html
                                 check(plug_query_strip_html in ('Y','N')),
    --
    plug_query_options           varchar2(4000),
    plug_query_max_columns       number,
    plug_column_width            varchar2(255),
    plug_customized              varchar2(255) not null
                                 constraint wwv_plug_plug_customized_ck
                                 check(plug_customized in ('0', '1', '2')),
    plug_customized_name         varchar2(4000),
    plug_override_reg_pos        varchar2(255),
    --
    required_patch               number,
    --
    plug_caching                 varchar2(30) not null
                                 constraint wwv_page_plugs_ck_plug_caching
                                 check (plug_caching in (
                                 'NOCACHE',
                                 'CACHE',
                                 'USER',
                                 'SESSION')),
    plug_cache_when              varchar2(255),
    plug_cache_expression1       varchar2(4000),
    plug_cache_expression2       varchar2(4000),
    plug_caching_max_age_in_sec  number,
    plug_cache_depends_on_items  varchar2(4000),
    plug_ignore_pagination       number,
    --
    plug_chart_font_size         varchar2(255),
    plug_chart_max_rows          number,
    plug_chart_num_mask          varchar2(255),
    plug_chart_scale             varchar2(255),
    plug_chart_axis              varchar2(255),
    plug_chart_show_summary      varchar2(255),
    --
    report_total_text_format    varchar2(4000),
    break_column_text_format    varchar2(4000),
    break_before_row            varchar2(4000),
    break_generic_column        varchar2(4000),
    break_after_row             varchar2(4000),
    break_type_flag             varchar2(255)
                                constraint wwv_break_flag_cc
                                check(break_type_flag in (
                                'REPEAT_HEADINGS_ON_BREAK_1',
                                'DEFAULT_BREAK_FORMATTING')),
    break_repeat_heading_format varchar2(4000),
    csv_output                   varchar2(1)
                                 constraint wwv_csv_output_flag_ck
                                 check(csv_output in ('Y','N')),
    csv_output_link_text         varchar2(4000),
    supplemental_text            varchar2(4000),
    print_url                    varchar2(4000),
    print_url_label              varchar2(4000),
    --
    prn_output                   varchar2(1) constraint plug_prn_output
                                     check (prn_output in ('Y','N')),
    prn_template_id              number,
    prn_format                   varchar2(255) constraint plug_prn_format
                                     check (prn_format in (
                                          'PDF',
                                          'RTF',
                                          'XLS',
                                          'HTM',
                                          'XML'
                                      )),
    prn_format_item              varchar2(4000),
    prn_output_show_link         varchar2(1)
                                     constraint plug_prn_show_link
                                         check(prn_output_show_link in ('Y','N')),
    prn_output_link_text         varchar2(4000),
    prn_output_file_name         varchar2(4000),
    prn_content_disposition      varchar2(255) constraint plug_prn_content_disp
                                     check (prn_content_disposition in (
                                         'INLINE',
                                         'ATTACHMENT'
                                     )),
    prn_units                    varchar2(4000) constraint plug_prn_units
                                     check (prn_units in (
                                         'POINTS',
                                         'INCHES',
                                         'MILLIMETERS',
                                         'CENTIMETERS'
                                     )),
    prn_paper_size               varchar2(4000) constraint plug_prn_paper_size
                                     check (prn_paper_size in (
                                        'CUSTOM',
                                        'LETTER',
                                        'LEGAL',
                                        'TABLOID',
                                        'A4',
                                        'A3'
                                     )),
    prn_width_units              varchar2(255) constraint plug_prn_width_units
                                     check (prn_width_units in (
                                         'PERCENTAGE',
                                         'POINTS'
                                     )),
    prn_width                    number,
    prn_height                   number,
    prn_orientation              varchar2(4000) constraint plug_prn_orientation
                                     check (prn_orientation in (
                                         'HORIZONTAL',
                                         'VERTICAL'
                                     )),
    prn_page_header              varchar2(4000),
    prn_page_header_alignment    varchar2(255) constraint plug_prn_header_align
                                     check (prn_page_header_alignment in ('LEFT','CENTER','RIGHT')),
    prn_page_header_font_color   varchar2(4000),
    prn_page_header_font_family  varchar2(4000),
    prn_page_header_font_weight  varchar2(4000),
    prn_page_header_font_size    varchar2(4000),
    prn_page_footer              varchar2(4000),
    prn_page_footer_alignment    varchar2(255) constraint plug_prn_footer_align
                                     check (prn_page_footer_alignment in ('LEFT','CENTER','RIGHT')),
    prn_page_footer_font_color   varchar2(4000),
    prn_page_footer_font_family  varchar2(4000),
    prn_page_footer_font_weight  varchar2(4000),
    prn_page_footer_font_size    varchar2(4000),
    prn_header_bg_color          varchar2(4000),
    prn_header_font_color        varchar2(4000),
    prn_header_font_family       varchar2(4000),
    prn_header_font_weight       varchar2(4000),
    prn_header_font_size         varchar2(4000),
    prn_body_bg_color            varchar2(4000),
    prn_body_font_color          varchar2(4000),
    prn_body_font_family         varchar2(4000),
    prn_body_font_weight         varchar2(4000),
    prn_body_font_size           varchar2(4000),
    prn_border_color             varchar2(4000),
    prn_border_width             number,
    --
    shared_query_id              number,
    --
    translate_title              varchar2(1) not null
                                 constraint wwv_translate_title_ck
                                 check(translate_title in ('Y','N')),
    --
    security_group_id            number not null,
    last_updated_by              varchar2(255),
    last_updated_on              date,
    --
    created_by                   varchar2(255),
    created_on                   date,
    plug_comment                 varchar2(4000),
    --
    use_custom_item_layout       varchar2(1),
    custom_item_layout           clob,
    sort_null                    varchar2(1)
                                 constraint wwv_pp_sort_null_ck
                                 check (sort_null in (null,'F','L')),
    region_css_classes           varchar2(255),
    icon_css_classes             varchar2(255),
    region_sub_css_classes       varchar2(255),
    region_template_options      varchar2(255) default '#DEFAULT#',
    component_template_options   varchar2(255) default '#DEFAULT#',
    region_attributes_substitution varchar2(4000),
    report_attributes_substitution varchar2(4000),
    std_rpt_max_query_cost       number,
    fixed_header                 varchar2(6)
                                 constraint wwv_flow_pp_fixed_header_ck
                                 check (fixed_header in (
                                 'NONE',
                                 'PAGE',
                                 'REGION')),
    fixed_header_max_height      number,
    --
    plugin_init_javascript_code  varchar2(4000),
    --
    attribute_01                 varchar2(4000),
    attribute_02                 varchar2(4000),
    attribute_03                 varchar2(4000),
    attribute_04                 varchar2(4000),
    attribute_05                 varchar2(4000),
    attribute_06                 varchar2(4000),
    attribute_07                 varchar2(4000),
    attribute_08                 varchar2(4000),
    attribute_09                 varchar2(4000),
    attribute_10                 varchar2(4000),
    attribute_11                 varchar2(4000),
    attribute_12                 varchar2(4000),
    attribute_13                 varchar2(4000),
    attribute_14                 varchar2(4000),
    attribute_15                 varchar2(4000),
    attribute_16                 varchar2(4000),
    attribute_17                 varchar2(4000),
    attribute_18                 varchar2(4000),
    attribute_19                 varchar2(4000),
    attribute_20                 varchar2(4000),
    attribute_21                 varchar2(4000),
    attribute_22                 varchar2(4000),
    attribute_23                 varchar2(4000),
    attribute_24                 varchar2(4000),
    attribute_25                 varchar2(4000),
    -- obsolete, but needed for upgrade
    plug_url_text_begin          varchar2(4000),
    plug_url_text_end            varchar2(4000),
    -- constraints
    constraint                   wwv_flow_plug_to_page_fk
    foreign key                  (flow_id, page_id) references wwv_flow_steps (flow_id, id) on delete cascade )
    -- storage
    lob (plug_source,custom_item_layout) store as (cache reads enable storage in row)
/

create index wwv_flow_page_plugs_fk_page   on wwv_flow_page_plugs (flow_id, page_id) compress 1;
create index wwv_flow_page_plugs_fk_parent on wwv_flow_page_plugs (parent_plug_id);
create index wwv_flow_page_plugs_fk_remote on wwv_flow_page_plugs (remote_server_id);
create index wwv_flow_page_plugs_fk_menu   on wwv_flow_page_plugs (menu_id);
create index wwv_flow_page_plugs_fk_list   on wwv_flow_page_plugs (list_id);
create index wwv_flow_page_plugs_fk_master on wwv_flow_page_plugs (master_region_id);
create index wwv_flow_page_plugs_fk_sgid   on wwv_flow_page_plugs (security_group_id) compress;

create index wwv_plug_web_src_module_fx    on wwv_flow_page_plugs (web_src_module_id);
create index wwv_flow_page_plugs_fk_filter on wwv_flow_page_plugs (filtered_region_id);

--==============================================================================
create table wwv_flow_page_generic_attr (
    id                           number
                                 constraint wwv_flow_page_generic_pk
                                 primary key,
    region_id                    number not null
                                 constraint wwv_flow_genattr_to_region_fk
                                 references wwv_flow_page_plugs
                                 on delete cascade,
    attribute_id                 number not null,
    security_group_id            number not null,
    attribute_value              clob)
    lob (attribute_value) store as (cache reads enable storage in row)
/

create unique index wwv_flow_page_genattr_idx on wwv_flow_page_generic_attr (region_id,attribute_id)
/
--##############################################################################
--#
--# APPLICATION > PAGE > BUTTON
--#
--##############################################################################
prompt APPLICATION > PAGE > BUTTON

--==============================================================================
create table wwv_flow_step_buttons (
    id                        number
                              constraint wwv_flow_step_buttons_pk
                              primary key,
    flow_id                   number  not null
                              constraint wwv_flow_step_buttons_fk1
                              references wwv_flows
                              on delete cascade,
    flow_step_id              number not null,
    button_sequence           number not null,
    button_plug_id            number
                              constraint wwv_flow_step_buttons_plug_fk
                              references wwv_flow_page_plugs
                              on delete set null,
    button_name               varchar2(2000) not null,
    button_static_id          varchar2(2000),
    button_template_id        number
                              constraint wwv_flow_step_buttons_templ_fk
                              references wwv_flow_button_templates (id)
                              deferrable initially deferred,
    button_template_options   varchar2(255) default '#DEFAULT#',
    button_is_hot             varchar2(1) default 'N'
                              not null
                              constraint wwv_flow_valid_fbutton_hot
                              check (button_is_hot in ('Y', 'N')),
    button_image_alt          varchar2(2000),
    button_position           varchar2(30) default 'BODY' not null,
    button_alignment          varchar2(30)
                              constraint wwv_flow_step_buttons_ck_ba
                              check (button_alignment in (
                              'LEFT',
                              'RIGHT',
                              'CENTER',
                              'LEFT-TOP',
                              'RIGHT-TOP',
                              'CENTER-TOP',
                              'LEFT-BOTTOM',
                              'RIGHT-BOTTOM',
                              'CENTER-BOTTOM',
                              'LEFT-CENTER',
                              'RIGHT-CENTER',
                              'CENTER-CENTER' )),
    button_redirect_url       varchar2(4000),
    button_action             varchar2(13) not null
                              constraint wwv_flow_valid_fbutton_action
                              check (button_action in (
                              'SUBMIT',
                              'REDIRECT_URL',
                              'REDIRECT_PAGE',
                              'REDIRECT_APP',
                              'DEFINED_BY_DA',
                              'RESET',
                              'NEXT_PAGE',
                              'PREVIOUS_PAGE' )),
    button_execute_validations varchar2(1)
                              constraint wwv_flow_valid_fbutton_execval
                              check (button_execute_validations in ('Y', 'N')),
    warn_on_unsaved_changes   varchar2(1) default 'I'
                              constraint step_buttons_warn_unsaved_ck
                              check (warn_on_unsaved_changes = 'I'),
    confirm_enabled           varchar2(1) generated always as ( nvl2(confirm_message, 'Y', 'N') ) virtual,
    confirm_message           varchar2(4000),
    confirm_style             varchar2(11),
                              constraint wwv_flow_step_buttons_ck_cnfs
                              check (confirm_style in (
                              'warning',
                              'danger',
                              'information',
                              'success' ) ),
    button_condition          varchar2(4000),
    button_condition2         varchar2(4000),
    button_condition_type     varchar2(255),
    button_css_classes        varchar2(255),
    icon_css_classes          varchar2(255),
    button_cattributes        varchar2(2000),
    database_action           varchar2(30),
    required_patch            number,
    security_scheme           varchar2(255),
    security_group_id         number not null,
    last_updated_by           varchar2(255),
    last_updated_on           date,
    created_by                varchar2(255),
    created_on                date,
    button_comment            varchar2(4000),
    request_source            varchar2(4000),
    request_source_type       varchar2(30)
                              constraint wwv_step_buttons_ck_rst
                              check (request_source_type in (
                              'PREFERENCE',
                              'ALWAYS_NULL',
                              'STATIC',
                              'QUERY',
                              'QUERY_COLON',
                              'ITEM',
                              'FUNCTION',
                              'FUNCTION_BODY',
                              'DB_COLUMN',
                              'PLSQL_ANONYMOUS_BLOCK')),
    pre_element_text          varchar2(4000),
    post_element_text         varchar2(4000),
    grid_new_grid             varchar2(1) default 'N',
                              constraint wwv_flow_step_buttons_ck_ng
                              check (grid_new_grid in ('Y', 'N')),
    grid_new_row              varchar2(1)
                              constraint wwv_flow_step_buttons_ck_bonl
                              check (grid_new_row in (
                              'Y',
                              'N')),
    grid_row_css_classes      varchar2(255),
    grid_new_column           varchar2(1)
                              constraint wwv_flow_step_buttons_ck_bonf
                              check (grid_new_column in (
                              'Y',
                              'N')),
    grid_column_span          number,
    grid_row_span             number,
    grid_column               number
                              constraint wwv_step_buttons_grid_ck_gc
                              check (grid_column between 1 and 99),
    grid_column_css_classes   varchar2(255),
    grid_column_attributes    varchar2(4000),
    -- obsolete columns
    button_image              varchar2(2000),
    button_image_attributes   varchar2(500),
    -- constraints
    constraint wwv_flow_step_buttons_fk2
    foreign key (flow_id, flow_step_id) references wwv_flow_steps (flow_id, id) on delete cascade)
/

create index wwv_flow_step_buttons_idx1 on wwv_flow_step_buttons (flow_id, flow_step_id) compress;
create index wwv_flow_step_buttons_idx2 on wwv_flow_step_buttons (button_plug_id) compress;
create index wwv_flow_step_buttons_idx3 on wwv_flow_step_buttons (button_template_id) compress;
create index wwv_flow_step_buttons_idx4 on wwv_flow_step_buttons (flow_step_id, flow_id, security_group_id) compress;

--##############################################################################
--#
--# APPLICATION > PAGE > ITEM
--#
--##############################################################################
prompt APPLICATION > PAGE > ITEM

--==============================================================================
create table wwv_flow_step_item_groups (
    id                          number
                                constraint wwv_flow_itemgroup_pk
                                primary key,
    flow_id                     number  not null
                                constraint wwv_flow_itemgroup_flow_fk
                                references wwv_flows
                                on delete cascade,
    flow_step_id                number not null,
    page_plug_id                number
                                constraint wwv_flow_itemgroup_plug_fk
                                references wwv_flow_page_plugs
                                on delete cascade,
    --
    label                       varchar2(255) not null,
    display_sequence            number not null,
    --
    icon_css_classes            varchar2(255),
    css_classes                 varchar2(255),
    --
    suggestions_type            varchar2(25)
                                constraint wwv_valid_fcgrp_sugges_type_ck
                                check (suggestions_type in ( 'DYNAMIC', 'STATIC', 'SQL_QUERY', 'FUNC_BODY_RETURNING_ARRAY' )),
    suggestions_language        varchar2(10)
                                constraint wwv_valid_fcgrp_sugges_lang_ck
                                check (suggestions_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    suggestions_source          varchar2(4000),
    max_suggestions             number,
    --
    fc_filter_combination       varchar2(3) not null,
                                constraint wwv_valid_fcgrp_filtercomb
                                check( fc_filter_combination in ( 'AND', 'OR' ) ),
    fc_max_height               number,
    fc_show_label               varchar2(1)
                                constraint wwv_valid_fcgrp_show_label_ck
                                check (fc_show_label in ( 'Y', 'N' )),
    --
    -- the collapsible settings need to be configured at a group level.
    --
    fc_collapsible              varchar2(1)
                                constraint wwv_valid_fcgrp_collapse
                                check ( fc_collapsible in ('Y', 'N') ),
    fc_initial_collapsed        varchar2(1)
                                constraint wwv_valid_fcgrp_init_collapse
                                check ( fc_initial_collapsed in ('Y', 'N') ),
    --
    -- of course, there is no "Compute Counts" or "Show Counts" for a facet group. This setting is an override
    -- for all simple checkbox facets belonging to this group.
    -- - If disabled, then counts will be computed for *no* facet within this group
    -- - If enabled, then counts will be computed based on the individual facet settings
    --
    fc_compute_counts           varchar2(1)
                                constraint wwv_valid_fcgrp_compcounts
                                check ( fc_compute_counts in ('Y', 'N') ),
    fc_show_counts              varchar2(1)
                                constraint wwv_valid_fcgrp_showcounts
                                check ( fc_show_counts in ('Y', 'N') ),
    --
    -- Handling of Zero Count Entries and whether to show selected values first does not make any sense
    -- on a facet level any more once facets belong to a group.
    -- So for grouped facets, this must be configued at the group level.
    --
    fc_zero_count_entries       varchar2(1)
                                constraint wwv_valid_fcgrp_zerocount_entr
                                check ( fc_zero_count_entries in ('H', 'D', 'L') ),
    fc_show_selected_first      varchar2(1)
                                constraint wwv_valid_fcgrp_selectfirst
                                check ( fc_show_selected_first in ('Y', 'N') ),
    fc_sort_by_top_counts       varchar2(1)
                                constraint wwv_valid_fcgrp_sortbytopcnts
                                check ( fc_sort_by_top_counts in ('Y', 'N') ),
    fc_show_more_count          number,
    fc_show_chart               varchar2(1)
                                constraint wwv_valid_fcgrp_showchart
                                check ( fc_show_chart in ('Y', 'N') ),
    fc_initial_chart            varchar2(1)
                                constraint wwv_valid_fcgrp_initchart
                                check ( fc_initial_chart in ('Y', 'N') ),
    --
    fc_toggleable               varchar2(1) default 'N'
                                constraint wwv_valid_fcgrp_toggleable
                                check ( fc_toggleable in ('Y', 'N') ),
    fc_initial_toggled          varchar2(1)
                                constraint wwv_valid_fcgrp_init_toggled
                                check ( fc_initial_toggled in ('Y', 'N') ),
    --
    condition_type              varchar2(255),
    condition_expr1             varchar2(4000),
    condition_expr2             varchar2(4000),
    --
    build_option_id             number,
    authorization_scheme        varchar2(255),
    --
    security_group_id           number not null,
    last_updated_by             varchar2(255),
    last_updated_on             date,
    created_by                  varchar2(255),
    created_on                  date,
    --
    group_comment               varchar2(4000) )
/

create index wwv_step_item_grp_ix      on wwv_flow_step_item_groups (flow_id, flow_step_id, display_sequence) compress 2;

create index wwv_step_item_grp_plug_fx on wwv_flow_step_item_groups (page_plug_id) compress;

--==============================================================================
create table wwv_flow_step_items (
    id                          number
                                constraint wwv_flow_step_items_pk
                                primary key,
    flow_id                     number  not null
                                constraint wwv_flow_step_items_fk
                                references wwv_flows on
                                delete cascade,
    flow_step_id                number not null,
    item_group_id               number
                                constraint wwv_flow_step_items_group_fk
                                references wwv_flow_step_item_groups( id )
                                on delete set null,
    name                        varchar2(255) not null,
    name_length                 number,
    data_type                   varchar2(30)
                                constraint valid_fsitem_datatype
                                check (data_type in ( 'VARCHAR', 'NUMBER', 'DATE', 'CLOB')),
    source_data_type            varchar2(30)
                                constraint valid_fsitem_source_datatype
                                check (source_data_type in ( 'VARCHAR2', 'NUMBER', 'DATE',
                                                             'TIMESTAMP', 'TIMESTAMP_TZ', 'TIMESTAMP_LTZ',
                                                             'INTERVAL_D2S', 'INTERVAL_Y2M',
                                                             'CLOB', 'BLOB', 'BFILE', 'ROWID', 'SDO_GEOMETRY' )),
    is_required                 varchar2(1)
                                constraint wwv_valid_fsitem_is_required
                                check (is_required in ('Y', 'N')),
    is_query_only               varchar2(1) default 'N' not null
                                constraint wwv_valid_fsitem_queryonly
                                check (is_query_only in ('Y', 'N')),
    is_primary_key              varchar2(1) default 'N' not null
                                constraint wwv_valid_fsitem_primarykey
                                check (is_primary_key in ('Y', 'N')),
    standard_validations        varchar2(1)
                                constraint wwv_valid_fsitem_std_val
                                check (standard_validations in ('BROWSER', 'SERVER', 'BROWSER_AND_SERVER', 'NONE')),
    accept_processing           varchar2(30)
                                constraint wwv_valid_fsitem_accpproc
                                check (accept_processing in (
                                'REPLACE_EXISTING',
                                'REPLACE_NULL_VALUES',
                                'RAISE_ERROR_ON_REPLACE',
                                'ADD_TO_EXISTING')),
    item_sequence               number not null,
    item_plug_id                number
                                constraint wwv_flow_step_items_plug_fk
                                references wwv_flow_page_plugs
                                on delete set null,
    item_display_point          varchar2(30) default 'BODY' not null,
    item_source_plug_id         number
                                constraint wwv_flow_step_items_srcplug_fk
                                references wwv_flow_page_plugs
                                on delete set null,
    use_cache_before_default    varchar2(30)
                                constraint wwv_valid_fsitem_usecache
                                check (use_cache_before_default in (
                                'YES',
                                'NO')),
    item_default                varchar2(4000),
    item_default_type           varchar2(255)
                                constraint wwv_valid_fsitem_idt
                                check (item_default_type in (
                                'STATIC_TEXT_WITH_SUBSTITUTIONS',
                                'ITEM',
                                'FUNCTION_BODY',
                                'EXPRESSION',
                                'SQL_QUERY',
                                'SQL_QUERY_COLON',
                                'SEQUENCE')),
    item_default_language       varchar2(10)
                                constraint wwv_valid_fsitem_default_lang
                                check (item_default_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    prompt                      varchar2(4000),
    placeholder                 varchar2(255),
    pre_element_text            varchar2(4000),
    post_element_text           varchar2(4000),
    format_mask                 varchar2(255),
    item_field_template         number,
    item_css_classes            varchar2(255),
    item_template_options       varchar2(255) default '#DEFAULT#',
    item_icon_css_classes       varchar2(255),
    source                      varchar2(4000),
    source_type                 varchar2(30)
                                constraint wwv_valid_fsitem_sourcetype
                                check (source_type in (
                                    'PREFERENCE',
                                    'ALWAYS_NULL',
                                    'STATIC',
                                    'QUERY',
                                    'QUERY_COLON',
                                    'ITEM',
                                    'EXPRESSION',
                                    'FUNCTION_BODY',
                                    'DB_COLUMN',
                                    'REGION_SOURCE_COLUMN',
                                    'PLSQL_ANONYMOUS_BLOCK',
                                    'FACET_COLUMN')),
    source_language             varchar2(10)
                                constraint wwv_valid_fsitem_source_lang
                                check (source_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    source_post_computation     varchar2(4000),
    display_as                  varchar2(255) not null
                                constraint wwv_valid_fsitem_displayas
                                check (display_as in (
                                    'DISPLAY_ONLY_PLSQL',
                                    'FILE',
                                    'BUTTON',
                                    'HIDDEN',
                                    'HIDDEN_PROTECTED',
                                    'DISPLAY_ONLY_HTML',
                                    'STOP_AND_START_HTML_TABLE',
                                    'DISPLAY_ONLY_ESCAPE_SC',
                                    'IMAGE',
                                    'DISPLAY_AND_SAVE',
                                    'DISPLAY_AND_SAVE_LOV',
                                    'CHECKBOX',
                                    'POPUP',
                                    'POPUP_FILTER',
                                    'POPUP2',
                                    'POPUP3',
                                    'POPUP4',
                                    'POPUP5',
                                    'POPUP6',
                                    'POPUP_KEY_LOV',
                                    'POPUP_KEY_LOV_NOFETCH',
                                    'POPUP_COLOR',
                                    'PICK_DATE_USING_FLOW_FORMAT_MASK',
                                    'PICK_DATE_USING_APP_FORMAT_MASK',
                                    'PICK_DATE_USING_APP_DATE_FORMAT',
                                    'PICK_DATE_USING_FORMAT_MASK',
                                    'PICK_DATE_DD_MON_RR',
                                    'PICK_DATE_DD_MON_RR_HH_MI',
                                    'PICK_DATE_DD_MON_RR_HH24_MI',
                                    'PICK_DATE_DD_MON_YYYY',
                                    'PICK_DATE_DD_MON_YYYY_HH_MI',
                                    'PICK_DATE_DD_MON_YYYY_HH24_MI',
                                    'PICK_DATE_DD_MM_YYYY',
                                    'PICK_DATE_DD_MM_YYYY_HH_MI',
                                    'PICK_DATE_DD_MM_YYYY_HH24_MI',
                                    'PICK_DATE_MM_DD_YYYY',
                                    'PICK_DATE_MM_DD_YYYY_HH_MI',
                                    'PICK_DATE_MM_DD_YYYY_HH24_MI',
                                    'PICK_DATE_DD_MON_YY',
                                    'PICK_DATE_DD_MON_YY_HH_MI',
                                    'PICK_DATE_DD_MON_YY_HH24_MI',
                                    'PICK_DATE_YYYY_MM_DD',
                                    'PICK_DATE_YYYY_MM_DD_HH_MI',
                                    'PICK_DATE_YYYY_MM_DD_HH24_MI',
                                    'PICK_DATE_RR_MON_DD',
                                    'PICK_DATE_RR_MON_DD_HH_MI',
                                    'PICK_DATE_RR_MON_DD_HH24_MI',
                                    'PICK_DATE_DD_MM_YYYY_DASH',
                                    'PICK_DATE_DD_MM_YYYY_HH_MI_DASH',
                                    'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH',
                                    'PICK_DATE_DD_MM_YYYY_DOT',
                                    'PICK_DATE_DD_MM_YYYY_HH_MI_DOT',
                                    'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT',
                                    'PICK_DATE_YYYY_DD_MM_DOT',
                                    'PICK_DATE_YYYY_DD_MM_HH_MI_DOT',
                                    'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT',
                                    'PICK_DATE_YYYY_MM_DD_SLASH',
                                    'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH',
                                    'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH',
                                    'DISPLAY_ONLY_LOV',
                                    'COMBOBOX',
                                    'SHOW_AS_SL_WITH_POPUP',
                                    'COMBOBOX_WITH_URL_REDIRECT',
                                    'COMBOBOX_WITH_BRANCH_TO_PAGE',
                                    'COMBOBOX_WITH_SUBMIT',
                                    'COMBOBOX_WITH_JS_POST',
                                    'COMBOBOX_WITH_JS_REDIRECT',
                                    'MULTIPLESELECT',
                                    'SHUTTLE',
                                    'RADIOGROUP',
                                    'RADIOGROUP2',
                                    'RADIOGROUP_WITH_REDIRECT',
                                    'RADIOGROUP_WITH_SUBMIT',
                                    'RADIOGROUP_WITH_JS_SUBMIT',
                                    'TEXTAREA',
                                    'TEXTAREA_WITH_SPELL_CHECK',
                                    'TEXTAREA-AUTO-HEIGHT',
                                    'TEXTAREA_WITH_CONTROLS',
                                    'TEXTAREA_WITH_HTML_EDITOR_BASIC',
                                    'TEXTAREA_WITH_HTML_EDITOR_STANDARD',
                                    'TEXTAREA_CHAR_COUNT',
                                    'TEXTAREA_CHAR_COUNT_SPELL',
                                    'TEXT',
                                    'TEXT_WITH_ENTER_SUBMIT',
                                    'TEXT_DISABLED',
                                    'TEXT_DISABLED_AND_SAVE',
                                    'TEXT_WITH_CALCULATOR',
                                    'PASSWORD',
                                    'PASSWORD_WITH_ENTER_SUBMIT',
                                    'PASSWORD_DNSS',
                                    'PASSWORD_WITH_SUBMIT_DNSS',
                                    'LIST_MGR',
                                    'LIST_MGR2',
                                    'LIST_MGR3',
                                    'LIST_MGR_VIEW',
                                    'LIST_MGR_VIEW2',
                                    -- 4.0
                                    'SELECT_LIST_WITH_POPUP',
                                    'TEXT_WITH_SELECT_LIST',
                                    'TEXTAREA_WITH_POPUP',
                                    'TEXTAREA_RESIZABLE'
                                    ) or
                                    display_as like 'NATIVE\_%' ESCAPE '\' or
                                    display_as like 'PLUGIN\_%' ESCAPE '\'),
    read_only_when_type         varchar2(255),
    read_only_when              varchar2(4000),
    read_only_when2             varchar2(4000),
    read_only_disp_attr         varchar2(4000),
    lov_language                varchar2(10)
                                constraint wwv_valid_fsitem_lov_lang_ck
                                check (lov_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    named_lov                   varchar2(255),
    lov                         varchar2(4000),
    lov_columns                 number, /* deprecated */
    lov_display_extra           varchar2(30)
                                constraint valid_fsitem_lovextra
                                check (lov_display_extra in (
                                    'YES',
                                    'NO')),
    lov_display_null            varchar2(30)
                                constraint valid_fsitem_lovdispnull
                                check (lov_display_null in (
                                    'YES',
                                    'NO')),
    lov_null_text               varchar2(255),
    lov_null_value              varchar2(255),
    lov_translated              varchar2(1)
                                constraint wwv_valid_lov_trans
                                check (lov_translated in ('Y','N',null)),
    lov_cascade_parent_column   varchar2(128),
    lov_cascade_parent_items    varchar2(255),
    ajax_items_to_submit        varchar2(4000),
    ajax_optimize_refresh       varchar2(1)
                                constraint valid_fsitem_ajax_opt_refresh
                                check (ajax_optimize_refresh in ('Y','N')),
    cSize                       number,
    cMaxlength                  number,
    cHeight                     number,
    cAttributes                 varchar2(2000),
    cattributes_element         varchar2(4000),
    tag_css_classes             varchar2(255),
    icon_css_classes            varchar2(255),
    tag_attributes              varchar2(2000),
    tag_attributes2             varchar2(4000),
    --
    button_image                varchar2(255),
    button_image_attr           varchar2(255),
    --
    new_grid                    varchar2(1) default 'N',
                                constraint wwv_step_items_new_grid_ck
                                check (new_grid in ('Y', 'N')),
    begin_on_new_line           varchar2(3) -- keep length of 3 because of migration
                                constraint valid_fsitem_beginonnewline
                                check (begin_on_new_line in (
                                'Y',
                                'N')),
    grid_row_css_classes        varchar2(255),
    begin_on_new_field          varchar2(3) -- keep length of 3 because of migration
                                constraint valid_fsitem_beginonnewfield
                                check (begin_on_new_field in (
                                'Y',
                                'N')),
    colspan                     number,
    rowspan                     number,
    grid_column                 number
                                constraint wwv_step_items_grid_col_ck
                                check (grid_column between 1 and 99),
    grid_label_column_span      number
                                constraint wwv_fsitem_label_colspan_ck
                                check (grid_label_column_span between 0 and 99),
    grid_column_css_classes     varchar2(255),
    label_alignment             varchar2(30)
                                constraint valid_fsitem_labelalign
                                check (label_alignment in (
                                    'LEFT',
                                    'RIGHT',
                                    'CENTER',
                                    'ABOVE',
                                    'BELOW',
                                    'LEFT-TOP',
                                    'RIGHT-TOP',
                                    'CENTER-TOP',
                                    'LEFT-BOTTOM',
                                    'RIGHT-BOTTOM',
                                    'CENTER-BOTTOM',
                                    'LEFT-CENTER',
                                    'RIGHT-CENTER',
                                    'CENTER-CENTER'
                                    )),
    field_alignment             varchar2(30)
                                constraint valid_fsitem_fieldalign
                                check (field_alignment in (
                                    'LEFT',
                                    'RIGHT',
                                    'CENTER',
                                    'LEFT-TOP',
                                    'RIGHT-TOP',
                                    'CENTER-TOP',
                                    'LEFT-BOTTOM',
                                    'RIGHT-BOTTOM',
                                    'CENTER-BOTTOM',
                                    'LEFT-CENTER',
                                    'RIGHT-CENTER',
                                    'CENTER-CENTER'
                                    )),
    field_template              number,
    label_cell_attr             varchar2(4000),
    field_cell_attr             varchar2(4000),
    display_when                varchar2(4000),
    display_when2               varchar2(4000),
    display_when_type           varchar2(255),
    --
    depending_on_id             number
                                constraint flow_step_items_depend_on_fk
                                references wwv_flow_step_items
                                on delete set null
                                deferrable initially deferred,
    depending_on_condition_type varchar2(11)
                                constraint flow_step_items_depend_cond_ck
                                check (depending_on_condition_type in (
                                           'EQUALS',
                                           'NOT_EQUALS',
                                           'NULL',
                                           'NOT_NULL',
                                           'IN_LIST',
                                           'NOT_IN_LIST' )),
    depending_on_expression     varchar2(4000),
    --
    warn_on_unsaved_changes     varchar2(1)
                                constraint step_items_warn_unsaved_ck
                                check (warn_on_unsaved_changes = 'I'),
    --
    is_Persistent               varchar2(1)
                                constraint valid_fsitem_ispersistent
                                check (is_Persistent in ('Y','N','U')),
    protection_level            varchar2(1) not null
                                constraint valid_fsitem_protection_level
                                check (protection_level in ('N','B','P','S','I')),
    escape_on_http_input        varchar2(1),
                                constraint valfsitem_escape_on_http_input
                                check (escape_on_http_input in ('Y','N')),
    escape_on_http_output       varchar2(1),
                                constraint valfsitem_escape_on_http_outp
                                check (escape_on_http_output in ('Y','N')),
    restricted_characters       varchar2(20)
                                constraint valfsitem_restricted_chars
                                check (restricted_characters in ('US_ONLY', 'WEB_SAFE', 'NO_SPECIAL_CHAR', 'NO_SPECIAL_CHAR_NL', 'WORKSPACE_SCHEMA')),
    security_scheme             varchar2(255),
    required_patch              number,
    encrypt_session_state_yn    varchar2(1),
    inline_help_text            varchar2(4000),
    show_quick_picks            varchar2(30),
    quick_pick_label_01         varchar2(255),
    quick_pick_value_01         varchar2(255),
    quick_pick_label_02         varchar2(255),
    quick_pick_value_02         varchar2(255),
    quick_pick_label_03         varchar2(255),
    quick_pick_value_03         varchar2(255),
    quick_pick_label_04         varchar2(255),
    quick_pick_value_04         varchar2(255),
    quick_pick_label_05         varchar2(255),
    quick_pick_value_05         varchar2(255),
    quick_pick_label_06         varchar2(255),
    quick_pick_value_06         varchar2(255),
    quick_pick_label_07         varchar2(255),
    quick_pick_value_07         varchar2(255),
    quick_pick_label_08         varchar2(255),
    quick_pick_value_08         varchar2(255),
    quick_pick_label_09         varchar2(255),
    quick_pick_value_09         varchar2(255),
    quick_pick_label_10         varchar2(255),
    quick_pick_value_10         varchar2(255),
    quick_pick_link_attr        varchar2(255),
    --
    suggestions_type            varchar2(25)
                                constraint wwv_step_items_sugges_type_ck
                                check (suggestions_type in ( 'DYNAMIC', 'STATIC', 'SQL_QUERY', 'COLUMN', 'FUNC_BODY_RETURNING_ARRAY' )),
    suggestions_language        varchar2(10)
                                constraint wwv_step_items_sugges_lang_ck
                                check (suggestions_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    suggestions_source          varchar2(4000),
    max_suggestions             number,
    --
    plugin_init_javascript_code varchar2(4000),
    --
    attribute_01                varchar2(4000),
    attribute_02                varchar2(4000),
    attribute_03                varchar2(4000),
    attribute_04                varchar2(4000),
    attribute_05                varchar2(4000),
    attribute_06                varchar2(4000),
    attribute_07                varchar2(4000),
    attribute_08                varchar2(4000),
    attribute_09                varchar2(4000),
    attribute_10                varchar2(4000),
    attribute_11                varchar2(4000),
    attribute_12                varchar2(4000),
    attribute_13                varchar2(4000),
    attribute_14                varchar2(4000),
    attribute_15                varchar2(4000),
    -- filter control columns
    fc_max_height               number,
    fc_show_label               varchar2(1)
                                constraint wwv_valid_fc_show_label_ck
                                check (fc_show_label in ( 'Y', 'N' )),
    fc_collapsible              varchar2(1)
                                constraint wwv_valid_fc_collapse
                                check ( fc_collapsible in ('Y', 'N') ),
    fc_initial_collapsed        varchar2(1)
                                constraint wwv_valid_fc_init_collapse
                                check ( fc_initial_collapsed in ('Y', 'N') ),
    fc_compute_counts           varchar2(1)
                                constraint wwv_valid_fc_comp_counts
                                check ( fc_compute_counts in ('Y', 'N') ),
    fc_zero_count_entries       varchar2(1)
                                constraint wwv_valid_fc_zero_count_entr
                                check ( fc_zero_count_entries in ('H', 'D', 'L') ),
    fc_show_selected_first      varchar2(1)
                                constraint wwv_valid_fc_select_first
                                check ( fc_show_selected_first in ('Y', 'N') ),
    fc_show_counts              varchar2(1)
                                constraint wwv_valid_fc_show_counts
                                check ( fc_show_counts in ('Y', 'N') ),
    fc_sort_by_top_counts       varchar2(1)
                                constraint wwv_valid_fc_sort_by_top_cnts
                                check ( fc_sort_by_top_counts in ('Y', 'N') ),
    fc_show_more_count          number,
    fc_filter_values            varchar2(1)
                                constraint wwv_valid_fc_filter_vals
                                check ( fc_filter_values in ('Y', 'N') ),
    fc_show_chart               varchar2(1)
                                constraint wwv_valid_fc_showchart
                                check ( fc_show_chart in ('Y', 'N') ),
    fc_initial_chart            varchar2(1)
                                constraint wwv_valid_fc_initchart
                                check ( fc_initial_chart in ('Y', 'N') ),
    fc_actions_filter           varchar2(1)
                                constraint wwv_valid_fc_act_filter
                                check ( fc_actions_filter in ('Y', 'N') ),
    fc_toggleable               varchar2(1)
                                constraint wwv_valid_fc_toggleable
                                check ( fc_toggleable in ('Y', 'N') ),
    fc_initial_toggled          varchar2(1)
                                constraint wwv_valid_fc_init_toggled
                                check ( fc_initial_toggled in ('Y', 'N') ),
    --
    multi_value_type            varchar2(10)
                                constraint  wwv_step_item_multival_type_ck
                                check ( multi_value_type in ( 'SEPARATED','JSON_ARRAY' ) ),
    multi_value_separator       varchar2(5),  -- one char or an "\xxxx" code
    multi_value_trim_space      varchar2(1)
                                constraint wwv_step_item_multival_trim_ck
                                check (multi_value_trim_space in ('Y', 'N')),
    fc_filter_combination       varchar2(3),
                                constraint wwv_item_fc_filter_combi_ck
                                check( fc_filter_combination in ( 'AND', 'OR' ) ),
    --
    -- button columns are obsolete
    --
    button_execute_validations  varchar2(1)
                                constraint wwv_valid_fsitem_b_exec_val
                                check (button_execute_validations in ('Y', 'N')),
    button_redirect_url         varchar2(4000),
    button_action               varchar2(30)
                                constraint wwv_valid_fsitem_b_action
                                check (button_action in (
                                    'SUBMIT',
                                    'REDIRECT_URL',
                                    'REDIRECT_PAGE',
                                    'DEFINED_BY_DA')),
    button_is_hot               varchar2(1)
                                constraint wwv_valid_fsitem_b_hot
                                check (button_is_hot in ('Y', 'N')),
    --
    security_group_id           number not null,
    created_by                  varchar2(255),
    created_on                  date,
    last_updated_by             varchar2(255),
    last_updated_on             date,
    item_comment                varchar2(4000),
    constraint                  wwv_flow_step_items_fk2
    foreign key                 (flow_id, flow_step_id) references wwv_flow_steps (flow_id, id) on delete cascade,
    constraint                  wwv_flow_step_items_u_name
                                unique (name, flow_id) deferrable initially deferred using index compress 1
)
/

create index wwv_flow_step_item_idx1 on wwv_flow_step_items (flow_id, flow_step_id, item_sequence) compress 2;
create index wwv_flow_step_item_idx2 on wwv_flow_step_items (item_plug_id) compress;
create unique index wwv_flow_step_item_idx3 on wwv_flow_step_items (flow_id, protection_level, id) compress 2;
create index wwv_flow_step_item_idx4 on wwv_flow_step_items (item_source_plug_id) compress;
create index wwv_flow_step_item_idx5 on wwv_flow_step_items (depending_on_id) compress;
create index wwv_flow_step_item_idx6 on wwv_flow_step_items (item_group_id) compress;

--==============================================================================
create table wwv_flow_step_item_help (
    id                        number
                              constraint wwv_flow_page_helptext_pk
                              primary key,
    flow_id                   number  not null
                              constraint wwv_flow_page_helptext_fk
                              references wwv_flows
                              on delete cascade,
    flow_item_id              number
                              constraint wwv_flow_item_helptext_fk
                              references wwv_flow_step_items
                              on delete cascade,
    help_text                 varchar2(4000),
    reference_id              number,
    security_group_id         number not null,
    last_updated_by           varchar2(255),
    last_updated_on           date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    item_help_text_comment    varchar2(4000)
    )
/

create unique index wwv_flow_step_item_help_idx on wwv_flow_step_item_help (flow_id, flow_item_id, security_group_id);
create index WWV_FLOW_STEP_ITEM_HELP_FKIDX ON WWV_FLOW_STEP_ITEM_HELP(FLOW_ITEM_ID);

--##############################################################################
--#
--# APPLICATION > PAGE > DYNAMIC ACTIONS
--#
--##############################################################################
prompt APPLICATION > PAGE > DYNAMIC ACTIONS

--==============================================================================
-- table wwv_flow_standard_events to store standard browser events
create table wwv_flow_standard_events (
    id                          number          not null
                                                constraint wwv_flow_std_evnt_pk
                                                primary key,
    name                        varchar2(45)    not null
                                                constraint wwv_flow_std_evnt_name_ck
                                                check (lower(name)=name),
    display_name                varchar2(100)   not null,
    event_type                  varchar2(30)    not null
                                                constraint wwv_flow_std_evnt_type_ck
                                                check (event_type in (
                                                    'BROWSER',
                                                    'APEX',
                                                    'CUSTOM' )),
    supported_ui_types           varchar2(255)  default 'DESKTOP' not null
    );

--unique constraint on name and event type
alter table wwv_flow_standard_events
    add constraint wwv_flow_std_evnt_uk unique (name,event_type);


--==============================================================================
-- table wwv_flow_page_da_events to store triggering events defined on pages in applications.
create table wwv_flow_page_da_events (
    id                          number          not null
                                                constraint wwv_flow_page_da_e_pk
                                                primary key,
    flow_id                     number          not null
                                                constraint wwv_flow_page_da_e_flow_fk
                                                references wwv_flows on delete cascade,
    page_id                     number          not null,
    security_group_id           number          not null,
    name                        varchar2(255)   not null,
    event_sequence              number          not null,
    triggering_element          varchar2(4000),
    triggering_element_type     varchar2(30)    constraint wwv_flow_page_da_e_te_type_ck
                                                check (triggering_element_type in (
                                                    'ITEM',
                                                    'REGION',
                                                    'COLUMN',
                                                    'DOM_OBJECT', -- deprecated
                                                    'JQUERY_SELECTOR',
                                                    'JAVASCRIPT_EXPRESSION',
                                                    'BUTTON' )),
    triggering_region_id        number          constraint wwv_flow_page_da_e_tr_fk
                                                references wwv_flow_page_plugs on delete set null,
    triggering_button_id        number,
    condition_element_type 		varchar2(6)	    constraint wwv_flow_page_da_e_cond_t_ck
    											check (condition_element_type in (
    												'ITEM',
    												'COLUMN')),
    condition_element         	varchar2(4000),
    triggering_condition_type   varchar2(30)    constraint wwv_flow_page_da_e_cond_ck
                                                check (triggering_condition_type in (
                                                    'EQUALS',
                                                    'NOT_EQUALS',
                                                    'GREATER_THAN',
                                                    'GREATER_THAN_OR_EQUAL',
                                                    'LESS_THAN',
                                                    'LESS_THAN_OR_EQUAL',
                                                    'NULL',
                                                    'NOT_NULL',
                                                    'IN_LIST',
                                                    'NOT_IN_LIST',
                                                    'JAVASCRIPT_EXPRESSION')),
    triggering_expression       varchar2(4000),
    bind_type                   varchar2(4)     not null
                                                constraint wwv_flow_page_da_e_type_ck
                                                check (bind_type in (
                                                    'bind',
                                                    'live',
                                                    'one')),
    bind_delegate_to_selector   varchar2(255),
    bind_event_type             varchar2(130)   not null
                                                constraint wwv_flow_page_da_e_btype_ck
                                                check (bind_event_type in (
                                                    'focusout',
                                                    'focusin',
                                                    'load',
                                                    'resize',
                                                    'scroll',
                                                    'unload',
                                                    'click',
                                                    'dblclick',
                                                    'mousedown',
                                                    'mouseup',
                                                    'mousemove',
                                                    'mouseenter',
                                                    'mouseleave',
                                                    'change',
                                                    'select',
                                                    'keydown',
                                                    'keypress',
                                                    'keyup',
                                                    'ready',
                                                    'apexbeforerefresh',
                                                    'apexafterrefresh',
                                                    'apexbeforepagesubmit',
                                                    'apexafterclosedialog',
                                                    'apexafterclosecanceldialog',
                                                    'custom',
                                                    'tap',
                                                    'taphold',
                                                    'swipe',
                                                    'swipeleft',
                                                    'swiperight',
                                                    'vmousedown',
                                                    'vmousemove',
                                                    'vmouseup',
                                                    'vclick',
                                                    'vmousecancel',
                                                    'orientationchange',
                                                    'scrollstart',
                                                    'scrollstop',
                                                    'apextap',
                                                    'apexdoubletap',
                                                    'apexpress',
                                                    'apexpan',
                                                    'apexswipe') or
                                                    bind_event_type like 'NATIVE\_%' escape '\' or
                                                    bind_event_type like 'PLUGIN\_%' escape '\'),
    bind_event_type_custom      varchar2(130),
    bind_event_data             varchar2(4000),
    display_when_type           varchar2(255),
    display_when_cond           varchar2(4000),
    display_when_cond2          varchar2(4000),
    required_patch              number,
    security_scheme             varchar2(255),
    da_event_comment            varchar2(4000),
    created_on                  date,
    created_by                  varchar2(255),
    last_updated_on             date,
    last_updated_by             varchar2(255) )
/
-- foreign key constraint for wwv_flow_page_da_events on page and app id to wwv_flow_steps
alter table wwv_flow_page_da_events
      add constraint wwv_flow_page_da_e_page_fk
      foreign key
      (flow_id, page_id) references wwv_flow_steps(flow_id, id)
      on delete cascade
      enable;

create index wwv_flow_page_da_e_region_idx1 on wwv_flow_page_da_events (triggering_region_id);

create index wwv_flow_page_da_e_region_idx2 on wwv_flow_page_da_events (flow_id,page_id);

create index wwv_flow_page_da_e_button_idx1 on wwv_flow_page_da_events (triggering_button_id);


--==============================================================================
-- table wwv_flow_page_da_actions to store actions defined in applications on pages.
create table wwv_flow_page_da_actions (
    id                            number          not null
                                                  constraint wwv_flow_page_da_a_pk
                                                  primary key,
    event_id                      number          not null
                                                  constraint wwv_flow_page_da_a_evnt_fk
                                                  references wwv_flow_page_da_events on delete cascade,
    flow_id                       number          not null,
    page_id                       number          not null,
    security_group_id             number          not null,
    event_result                  varchar2(5)     not null
                                                  constraint wwv_flow_page_da_a_er_ck
                                                  check (event_result in (
                                                      'TRUE',
                                                      'FALSE')),
    action_sequence               number          not null,
    execute_on_page_init          varchar2(1)     not null
                                                  constraint wwv_flow_page_da_a_on_init_ck
                                                  check (execute_on_page_init in (
                                                      'Y',
                                                      'N' )),
    wait_for_result               varchar2(1)     constraint wwv_flow_page_da_a_wfr_ck
                                                  check (wait_for_result in (
                                                      'Y',
                                                      'N' )),
    name                          varchar2(255),
    action                        varchar2(255)   not null
                                                  constraint wwv_flow_page_da_a_action_ck
                                                  check (
                                                      action like 'NATIVE\_%' escape '\' or
                                                      action like 'PLUGIN\_%' escape '\'),
    affected_elements_type        varchar2(30)   constraint wwv_flow_page_da_a_ae_type_ck
                                                 check (affected_elements_type in (
                                                     'ITEM',
                                                     'BUTTON',
                                                     'REGION',
                                                     'COLUMN',
                                                     'DOM_OBJECT', -- deprecated
                                                     'JQUERY_SELECTOR',
                                                     'JAVASCRIPT_EXPRESSION',
                                                     'TRIGGERING_ELEMENT',
                                                     'EVENT_SOURCE' )),
    affected_region_id            number         constraint wwv_flow_page_da_a_ar_fk
                                                 references wwv_flow_page_plugs on delete set null,
    affected_button_id            number,
    affected_elements             varchar2(4000),
    --
    plugin_init_javascript_code varchar2(4000),
    --additional optional attributes used to store up to 10 additional parameters
    attribute_01                  varchar2(4000),
    attribute_02                  varchar2(4000),
    attribute_03                  varchar2(4000),
    attribute_04                  varchar2(4000),
    attribute_05                  varchar2(4000),
    attribute_06                  varchar2(4000),
    attribute_07                  varchar2(4000),
    attribute_08                  varchar2(4000),
    attribute_09                  varchar2(4000),
    attribute_10                  varchar2(4000),
    attribute_11                  varchar2(4000),
    attribute_12                  varchar2(4000),
    attribute_13                  varchar2(4000),
    attribute_14                  varchar2(4000),
    attribute_15                  varchar2(4000),
    --
    stop_execution_on_error       varchar2(1)     constraint wwv_flow_page_da_a_on_err_ck
                                                      check (stop_execution_on_error in ( 'Y', 'N' )),
    --
    client_condition_type         varchar2(30)    constraint wwv_flow_page_da_a_ccond_ck
                                                      check (client_condition_type in (
                                                          'EQUALS',
                                                          'NOT_EQUALS',
                                                          'GREATER_THAN',
                                                          'GREATER_THAN_OR_EQUAL',
                                                          'LESS_THAN',
                                                          'LESS_THAN_OR_EQUAL',
                                                          'NULL',
                                                          'NOT_NULL',
                                                          'IN_LIST',
                                                          'NOT_IN_LIST',
                                                          'JAVASCRIPT_EXPRESSION')),
    client_condition_element_type varchar2(6)      constraint wwv_flow_page_da_a_ccond_et_ck
                                                      check (client_condition_element_type in (
                                                          'ITEM',
                                                          'COLUMN')),
    client_condition_element      varchar2(4000),
    client_condition_expression   varchar2(4000),
    --
    server_condition_type         varchar2(255)    constraint wwv_flow_page_da_a_scond_ck
                                                      check (server_condition_type in (
                                                          'EXISTS',
                                                          'NOT_EXISTS',
                                                          'EXPRESSION',
                                                          'FUNCTION_BODY',
                                                          'REQUEST_EQUALS_CONDITION',
                                                          'REQUEST_NOT_EQUAL_CONDITION',
                                                          'REQUEST_IN_CONDITION',
                                                          'REQUEST_NOT_IN_CONDITION',
                                                          'VAL_OF_ITEM_IN_COND_EQ_COND2',
                                                          'VAL_OF_ITEM_IN_COND_NOT_EQ_COND2',
                                                          'ITEM_IS_NULL',
                                                          'ITEM_IS_NOT_NULL',
                                                          'ITEM_IS_ZERO',
                                                          'ITEM_IS_NOT_ZERO',
                                                          'ITEM_IS_NULL_OR_ZERO',
                                                          'ITEM_NOT_NULL_OR_ZERO',
                                                          'ITEM_CONTAINS_NO_SPACES',
                                                          'ITEM_IS_NUMERIC',
                                                          'ITEM_IS_NOT_NUMERIC',
                                                          'ITEM_IS_ALPHANUMERIC',
                                                          'VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST',
                                                          'VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST',
                                                          'USER_PREF_IN_COND_EQ_COND2',
                                                          'USER_PREF_IN_COND_NOT_EQ_COND2',
                                                          'CURRENT_PAGE_EQUALS_CONDITION',
                                                          'CURRENT_PAGE_NOT_EQUAL_CONDITION',
                                                          'CURRENT_PAGE_IN_CONDITION',
                                                          'CURRENT_PAGE_NOT_IN_CONDITION',
                                                          'PAGE_IS_IN_PRINTER_FRIENDLY_MODE',
                                                          'PAGE_IS_NOT_IN_PRINTER_FRIENDLY_MODE',
                                                          'IS_READ_ONLY',
                                                          'IS_NOT_READ_ONLY',
                                                          'USER_IS_NOT_PUBLIC_USER',
                                                          'USER_IS_PUBLIC_USER',
                                                          'DISPLAYING_INLINE_VALIDATION_ERRORS',
                                                          'NOT_DISPLAYING_INLINE_VALIDATION_ERRORS',
                                                          'CURRENT_LANG_EQ_COND1',
                                                          'CURRENT_LANG_NOT_EQ_COND1',
                                                          'CURRENT_LANG_IN_COND1',
                                                          'CURRENT_LANG_NOT_IN_COND1',
                                                          'NEVER' )),
    server_condition_expr1        varchar2(4000),
    server_condition_expr2        varchar2(4000),
    --
    build_option_id               number,
    security_scheme               varchar2(255),
    --
    da_action_comment             varchar2(4000),
    --audit
    created_by                    varchar2(255),
    created_on                    date,
    last_updated_by               varchar2(255),
    last_updated_on               date)
/

-- foreign key constraint for wwv_flow_page_da_actions on page and app id to wwv_flow_steps
alter table wwv_flow_page_da_actions
      add constraint wwv_flow_page_da_a_page_fk
      foreign key
      (flow_id, page_id) references wwv_flow_steps(flow_id, id)
      on delete cascade
      enable;

create index wwv_flow_page_da_a_event_idx1  on wwv_flow_page_da_actions (event_id);
create index wwv_flow_page_da_a_region_idx2 on wwv_flow_page_da_actions (affected_region_id);
create index wwv_flow_page_da_a_page_idx3   on wwv_flow_page_da_actions (flow_id,page_id);
create index wwv_flow_page_da_a_button_idx1 on wwv_flow_page_da_actions (affected_button_id);

--##############################################################################
--#
--# APPLICATION > PAGE > COMPUTATION
--#
--##############################################################################

--==============================================================================
create table wwv_flow_step_computations (
    id                    number
                          constraint wwv_flow_step_comp_pk
                          primary key,
    flow_id               number not null
                          constraint wwv_flow_step_comp_fk
                          references wwv_flows
                          on delete cascade,
    flow_step_id          number not null,
    --
    computation_sequence  number        not null,
    computation_item      varchar2(255) not null,
    computation_point     varchar2(255) not null
                          constraint valid_fstepcomp_point
                          check (computation_point in (
                          'ON_NEW_INSTANCE',
                          'BEFORE_HEADER',
                          'AFTER_HEADER',
                          'BEFORE_BOX_BODY',
                          'AFTER_BOX_BODY',
                          'BEFORE_FOOTER',
                          'AFTER_FOOTER',
                          'AFTER_SUBMIT')),
    -- computation_item_type to be obsoleted (SSpadafore)
    computation_item_type varchar2(30)
                          constraint valid_fstepcomp_itemtype
                          check (computation_item_type in (
                          'VARCHAR',
                          'VC_ARR',
                          'NUMBER')),
    computation_type      varchar2(30) not null
                          constraint valid_fstepcomp_comptype
                          check (computation_type in (
                          'SET_ITEM_EQUAL_THIS_PREFERENCE',
                          'STATIC_ASSIGNMENT',
                          'FUNCTION_BODY',
                          'QUERY',
                          'QUERY_COLON',
                          'EXPRESSION',
                          'ITEM_VALUE')),
    computation_language  varchar2(10)
                          constraint valid_fstepcomp_comp_lang
                          check (computation_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    computation_processed varchar2(30)
                          constraint valid_fstepcomp_compproc
                          check (computation_processed in (
                          'ON_NEW_INSTANCE',
                          'REPLACE_EXISTING',
                          'ADD_TO_EXISTING',
                          'RAISE_ERROR_ON_REPLACE',
                          'REPLACE_NULL_VALUES')),
    computation               varchar2(4000),
    compute_when              varchar2(4000),
    compute_when_type         varchar2(255),
    compute_when_text         varchar2(4000),
    computation_error_message varchar2(4000),
    required_patch            number,
    security_scheme           varchar2(255),
    security_group_id         number not null,
    last_updated_by           varchar2(255),
    last_updated_on           date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    computation_comment       varchar2(4000),
    constraint                wwv_flow_step_comp_fk2
    foreign key               (flow_id, flow_step_id) references wwv_flow_steps (flow_id, id) on delete cascade)
/

create unique index WWV_FLOW_STEP_COMPUTATI_FKIDX ON WWV_FLOW_STEP_COMPUTATIONS(FLOW_ID,FLOW_STEP_ID,ID) compress;

--##############################################################################
--#
--# APPLICATION > PAGE > VALIDATION
--#
--##############################################################################

--==============================================================================
-- Validation Types:
--
-- SQL_EXPRESION    &FIELD1 = &FIELD2
--                  &FIELD1 in ('A','B','C')
--                  success if select 1 from sys.dual where SQL_EXPRESION returns a row
-- PLSQL_EXPRESION  1 = my_func(&FIELD1)
--                  since some operations may use impure calls
-- PLSQL_ERROR      declare x number; begin x := &FIELD1; end;
--                  success if plsql does not raise an error
-- EXISTS           select 1 from sys.dual where &FIELD > 1
--                  success if query returns at least one value
--
create table wwv_flow_step_validations (
    id                        number
                              constraint wwv_flow_step_val_pk
                              primary key,
    flow_id                   number not null
                              constraint wwv_flow_step_val_fk
                              references wwv_flows
                              on delete cascade,
    flow_step_id              number not null,
    reference_id              number,
    tabular_form_region_id    number
                              constraint wwv_flow_step_val_to_reg_fk
                              references wwv_flow_page_plugs
                              on delete cascade,
    validation_name           varchar2(255) not null,
    validation_sequence       number not null,
    validation                varchar2(4000),
    validation2               varchar2(4000),
    validation_type           varchar2(255) not null,
                              constraint wwv_valid_fsval_valtype
                              check (validation_type in (
                              'ITEM_NOT_NULL',
                              'ITEM_NOT_NULL_OR_ZERO',
                              'ITEM_NOT_ZERO',
                              'ITEM_IS_ALPHANUMERIC',
                              'ITEM_IS_NUMERIC',
                              'ITEM_IS_NOT_NUMERIC',
                              'ITEM_IS_DATE',
                              'ITEM_IS_TIMESTAMP',
                              'ITEM_CONTAINS_NO_SPACES',
                              'EXPRESSION',
                              'REGULAR_EXPRESSION',
                              'FUNC_BODY_RETURNING_ERR_TEXT',
                              'FUNC_BODY_RETURNING_BOOLEAN',
                              'PLSQL_ERROR',
                              'NOT_EXISTS',
                              'EXISTS',
                              'ITEM_IN_VALIDATION_IN_STRING2',
                              'ITEM_IN_VALIDATION_NOT_IN_STRING2',
                              'ITEM_IN_VALIDATION_EQ_STRING2',
                              'ITEM_IN_VALIDATION_NOT_EQ_STRING2',
                              'ITEM_IN_VALIDATION_CONTAINS_AT_LEAST_ONE_CHAR_IN_STRING2',
                              'ITEM_IN_VALIDATION_CONTAINS_ONLY_CHAR_IN_STRING2',
                              'ITEM_IN_VALIDATION_CONTAINS_NO_CHAR_IN_STRING2',
                              'COL_NOT_NULL',
                              'COL_NOT_NULL_OR_ZERO',
                              'COL_NOT_ZERO',
                              'COL_IS_ALPHANUMERIC',
                              'COL_IS_NUMERIC',
                              'COL_IS_NOT_NUMERIC',
                              'COL_IS_DATE',
                              'COL_IS_TIMESTAMP',
                              'COL_CONTAINS_NO_SPACES',
                              'COL_IN_VALIDATION_IN_STRING2',
                              'COL_IN_VALIDATION_NOT_IN_STRING2',
                              'COL_IN_VALIDATION_EQ_STRING2',
                              'COL_IN_VALIDATION_NOT_EQ_STRING2',
                              'COL_IN_VALIDATION_CONTAINS_AT_LEAST_ONE_CHAR_IN_STRING2',
                              'COL_IN_VALIDATION_CONTAINS_ONLY_CHAR_IN_STRING2',
                              'COL_IN_VALIDATION_CONTAINS_NO_CHAR_IN_STRING2'
                              )),
    always_execute            varchar2(1) default 'N' not null
                              constraint wwv_valid_val_exec_val
                              check (always_execute in ('Y', 'N')),
    validation_condition      varchar2(4000),
    validation_condition2     varchar2(4000),
    validation_condition_type varchar2(255),
    exec_cond_for_each_row    varchar2(1)
                              constraint wwv_valid_val_cond_each_row
                              check (exec_cond_for_each_row in ('Y', 'N')),
    when_button_pressed       number,
    only_for_changed_rows     varchar2(1)
                              constraint wwv_valid_val_changed_rows
                              check (only_for_changed_rows in ('Y', 'N')),
    error_message             varchar2(4000),
    associated_item           number,
    associated_column         varchar2(255),
    error_display_location    varchar2(255) not null
                              constraint wwv_valid_val_edl
                              check (error_display_location in (
                              'INLINE_WITH_FIELD',
                              'INLINE_WITH_FIELD_AND_NOTIFICATION',
                              'INLINE_IN_NOTIFICATION',
                              'ON_ERROR_PAGE')),
    required_patch            number,
    security_scheme           varchar2(255),
    security_group_id         number not null,
    last_updated_by           varchar2(255),
    last_updated_on           date,
    -- 4.0
    created_by                varchar2(255),
    created_on                date,
    validation_comment        varchar2(4000),
    bind_01                   varchar2(255),
    bind_02                   varchar2(255),
    bind_03                   varchar2(255),
    bind_04                   varchar2(255),
    bind_05                   varchar2(255),
    bind_06                   varchar2(255),
    constraint                wwv_flow_step_val_fk2
    foreign key               (flow_id, flow_step_id) references wwv_flow_steps (flow_id, id) on delete cascade)
/

create index wwv_flow_step_validations_idx3 on wwv_flow_step_validations (tabular_form_region_id);

create unique index wwv_flow_step_validatio_fkidx on wwv_flow_step_validations(flow_id,flow_step_id,id) compress
/

--##############################################################################
--#
--# APPLICATION > PAGE > PROCESS
--#
--##############################################################################

--==============================================================================
create table wwv_flow_step_processing (
    id                      number
                            constraint wwv_flow_step_proc_pk
                            primary key,
    flow_id                 number not null
                            constraint wwv_flow_step_proc_fk
                            references wwv_flows
                            on delete cascade,
    flow_step_id            number not null,
    --
    process_sequence        number not null,
    process_point           varchar2(255) not null
                            constraint valid_fsproc_point
                            check (process_point in (
                            'ON_NEW_INSTANCE',
                            'ON_SUBMIT_BEFORE_COMPUTATION',
                            'BEFORE_HEADER',
                            'AFTER_HEADER',
                            'BEFORE_BOX_BODY',
                            'BEFORE_SHOW_ITEMS', -- obsolete
                            'AFTER_SHOW_ITEMS', -- obsolete
                            'AFTER_BOX_BODY',
                            'BEFORE_FOOTER',
                            'AFTER_FOOTER',
                            'AFTER_SUBMIT',
                            'AFTER_SUBMIT_BEFORE_VALIDATION',
                            'ON_DEMAND',
                            'AFTER_ERROR_HEADER',
                            'BEFORE_ERROR_FOOTER')),
    region_id               number
                            constraint wwv_flow_step_proc_reg_fk
                            references wwv_flow_page_plugs
                            on delete cascade,
    --
    process_type            varchar2(255) not null
                            constraint wwv_valid_fsproc_type
                            check (   process_type like 'NATIVE\_%' ESCAPE '\'
                                   or process_type like 'PLUGIN\_%' ESCAPE '\'),
    --
    process_name            varchar2(255)  not null,
    process_sql_clob        clob,
    process_clob_language   varchar2(10)
                            constraint wwv_valid_fsproc_clob_lang
                            check (process_clob_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    --
    location                varchar2(13)
                            constraint wwv_process_location_ck
                            check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE', 'REGION_SOURCE' )),
    remote_server_id        number
                            constraint wwv_process_remote_server_fk
                            references wwv_remote_servers,
    web_src_module_id       number
                            constraint wwv_process_websrc_module_fk
                            references wwv_flow_web_src_modules,
    web_src_operation_id    number
                            constraint wwv_process_websrc_opers_fk
                            references wwv_flow_web_src_operations,
    email_template_id       number generated always as (
                                                        case when process_type = 'NATIVE_SEND_EMAIL' and attribute_11 is not null
                                                            then to_number( attribute_11 )
                                                        end ) virtual,
    task_def_id             number generated always as (
                                                        case when process_type = 'NATIVE_CREATE_TASK' and attribute_01 is not null
                                                            then to_number( attribute_01 )
                                                        end ) virtual,                                                        
    --
    attribute_01            varchar2(4000),
    attribute_02            varchar2(4000),
    attribute_03            varchar2(4000),
    attribute_04            varchar2(4000),
    attribute_05            varchar2(4000),
    attribute_06            varchar2(4000),
    attribute_07            varchar2(4000),
    attribute_08            varchar2(4000),
    attribute_09            varchar2(4000),
    attribute_10            varchar2(4000),
    attribute_11            varchar2(4000),
    attribute_12            varchar2(4000),
    attribute_13            varchar2(4000),
    attribute_14            varchar2(4000),
    attribute_15            varchar2(4000),
    --
    process_error_message   varchar2(4000),
    error_display_location  varchar2(40) default 'ON_ERROR_PAGE'
                            constraint wwv_flow_step_proc_edl
                            check (error_display_location in ('INLINE_IN_NOTIFICATION', 'ON_ERROR_PAGE')),
    process_when_button_id  number,
    process_when            varchar2(4000),
    process_when_type       varchar2(255),
    process_when2           varchar2(4000),
    exec_cond_for_each_row  varchar2(1)
                            constraint wwv_valid_fsproc_cond_each_row
                            check (exec_cond_for_each_row in ('Y', 'N')),
    only_for_changed_rows   varchar2(1)
                            constraint wwv_valid_fsproc_changed_rows
                            check (only_for_changed_rows in ('Y', 'N')),
    process_when_type2      varchar2(255)                                  -- obsolete
                            constraint valid_fsproc_whentype2
                            check (process_when_type2 in (
                            'CURRENT_PAGE_NOT_IN_CONDITION',
                            'CURRENT_PAGE_IN_CONDITION',
                            'NEVER',
                            'CONDITION_IN_REQUEST',
                            'CONDITION_NOT_IN_REQUEST',
                            'REQUEST_EQUALS_CONDITION',
                            'REQUEST_IN_CONDITION',
                            'REQUEST_NOT_IN_CONDITION',
                            'ITEM_IS_NULL',
                            'ITEM_IS_NOT_NULL',
                            'ITEM_IS_NOT_NULL_OR_ZERO')),
    process_is_stateful_y_n VARCHAR2(1) not null
                            constraint valid_fsproc_stateful
                            check(process_is_stateful_y_n in ('Y','N')),
    --
    process_success_message varchar2(4000),
    required_patch          number,
    security_scheme         varchar2(255),
    security_group_id       number not null,
    last_updated_by         varchar2(255),
    last_updated_on         date,
    -- 4.0
    created_by              varchar2(255),
    created_on              date,
    process_comment         varchar2(4000),
    --
    -- obsolete columns:
    -- * runtime_where_clause, return_key_into_itemX, item_name were used by
    --   builtin process types before 5.0
    --
    runtime_where_clause    varchar2(4000),
    return_key_into_item1   varchar2(255),
    return_key_into_item2   varchar2(255),
    item_name               varchar2(255),
    --
    -- constraints
    --
    constraint wwv_flow_step_proc_fk2      foreign key (flow_id, flow_step_id)
                                           references wwv_flow_steps (flow_id, id)
                                           on delete cascade,
    constraint wwv_step_proc_mail_templ_fk foreign key( email_template_id )
                                           references wwv_flow_email_templates
                                           deferrable initially deferred
    )
    lob (process_sql_clob) store as (cache reads enable storage in row)
/

alter table         wwv_flow_step_processing add constraint wwv_proc_websrc_mod_or_oper_ck
    check ( web_src_module_id is null or web_src_operation_id is null )
/


create index        wwv_flow_step_processing_idx3  on wwv_flow_step_processing (region_id)
/
create unique index wwv_flow_step_processin_fkidx  on wwv_flow_step_processing (flow_id,flow_step_id,id) compress
/
create index        wwv_step_proc_websrc_module_fx on wwv_flow_step_processing (web_src_module_id)       compress
/
create index        wwv_step_proc_websrc_opers_fx  on wwv_flow_step_processing (web_src_operation_id)    compress
/
create index        wwv_step_proc_remote_server_fx on wwv_flow_step_processing (remote_server_id)        compress
/
create index        wwv_step_proc_email_templ_fx   on wwv_flow_step_processing (email_template_id)       compress
/
create index        wwv_step_proc_task_def_fx      on wwv_flow_step_processing (task_def_id)             compress
/
--==============================================================================
-- web service parameters
create table wwv_flow_ws_process_parms_map (
    id                      number
                            constraint wwv_flow_ws_map_pk
                            primary key,
    parameter_id            number
                            constraint wwv_flows_ws_map_fk1
                            references wwv_flow_ws_parameters
                            on delete cascade,
    process_id              number
                            constraint wwv_flow_ws_map_fk2
                            references wwv_flow_step_processing
                            on delete cascade,
    map_type                varchar2(10)
                            constraint wwv_flow_ws_map_chk
                            check (map_type in ('ITEM','STATIC','FUNCTION','COLLECTION')),
    parm_value              varchar2(4000),
    --
    security_group_id       number not null,
    last_updated_by         varchar2(255),
    last_updated_on         date
    )
/

create index wwv_flow_ws_map_idx1 on wwv_flow_ws_process_parms_map (parameter_id)
/
create index wwv_flow_ws_map_idx2 on wwv_flow_ws_process_parms_map (process_id)
/

--##############################################################################
--#
--# APPLICATION > PAGE > BRANCH
--#
--##############################################################################

--==============================================================================
create table wwv_flow_step_branches (
    id                number
                      constraint wwv_flow_step_branches_pk
                      primary key,
    flow_id           number  not null
                      constraint wwv_flow_step_branches_fk
                      references wwv_flows on delete cascade,
    flow_step_id      number not null,
    branch_name       varchar2(255),
    branch_action     varchar2(4000),
    branch_point      varchar2(30) not null
                      constraint valid_fsbranch_point
                      check (branch_point in (
                      'BEFORE_HEADER',
                      'BEFORE_COMPUTATION',
                      'BEFORE_VALIDATION',
                      'BEFORE_PROCESSING',
                      'AFTER_PROCESSING')),
    branch_when_button_id number,
    branch_type       varchar2(255) not null
                      constraint valid_fsbranch_branchtype
                      check (branch_type in (
                      'BRANCH_TO_FUNCTION_RETURNING_PAGE',
                      'BRANCH_TO_FUNCTION_RETURNING_URL',
                      'BRANCH_TO_PAGE_IDENT_BY_ITEM',
                      'BRANCH_TO_URL_IDENT_BY_ITEM',
                      'BRANCH_TO_STEP',
                      'BRANCH_TO_PAGE_ACCEPT',
                      'PLSQL',
                      'REDIRECT_URL',
                      'BRANCH_TO_ENTRY_POINT',
                      'BRANCH_TO_ENTRY_POINT_IDENT_BY_ITEM')),
    branch_language           varchar2(10)
                              constraint valid_fsbranch_branch_lang
                              check (branch_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    branch_sequence           number not null,
    clear_page_cache          varchar2(4000),
    save_state_before_branch_yn varchar2(1)
                                constraint wwv_flow_step_branches_svstate
                                check (save_state_before_branch_yn in ('Y','N')),
    branch_condition_type     varchar2(255),
    branch_condition          varchar2(4000),
    branch_condition_text     varchar2(4000),
    required_patch            number,
    security_scheme           varchar2(255),
    security_group_id         number not null,
    last_updated_by           varchar2(255),
    last_updated_on           date,
    -- 4.0
    created_by                 varchar2(255),
    created_on                 date,
    branch_comment            varchar2(4000),
    constraint wwv_flow_step_branches_fk2
    foreign key (flow_id, flow_step_id) references wwv_flow_steps (flow_id, id) on delete cascade)
/

create unique index wwv_flow_step_branches_fkidx on wwv_flow_step_branches(flow_id,flow_step_id,id) compress;

--##############################################################################
--#
--# APPLICATION > PAGE > REGION (TREE)
--#
--##############################################################################
prompt APPLICATION > PAGE > REGION (TREE)

--==============================================================================
create table wwv_flow_tree_regions  (
    id                           number
                                  constraint wwv_flow_treeregion_pk
                                  primary key,
    flow_id                      number not null
                                  constraint wwv_flow_treeregion_fk
                                   references wwv_flows
                                   on delete cascade,
    page_id                      number,
    region_id                    number not null
                                  constraint wwv_flow_treeregion_fk2
                                  references wwv_flow_page_plugs
                                  on delete cascade,
    tree_template                varchar2(255) constraint wwv_tree_templates_con
                                  check (tree_template in (
                                        'classic',
                                        'apple',
                                        'default',
                                        'themeroller'
                                 )),
    tree_source                  varchar2(255),
    tree_query                   varchar2(4000),
    tree_node_title              varchar2(255),
    tree_node_value              varchar2(4000),
    tree_node_icon               varchar2(4000),
    tree_node_link               varchar2(4000),
    tree_node_hints              varchar2(4000),
    tree_start_item              varchar2(255),
    tree_start_value             varchar2(1000),
    tree_button_option           varchar2(1000),
    tree_comment                 varchar2(4000),
    --
    show_hints                   varchar2(20),
    tree_hint_text               varchar2(4000),
    tree_has_focus               varchar2(1),
    tree_click_action            varchar2(1) constraint wwv_tree_click_act
                                  check (tree_click_action in ('S','D')),
    show_node_link               varchar2(1),
    node_link_checksum_type      varchar2(255) constraint wwv_node_link_checksum_type
                                  check (node_link_checksum_type in (
                                    '1', -- workspace
                                    '2', -- user
                                    '3'  -- session
                                 )),
    tree_selected_node           varchar2(255),
    --
    security_group_id            number not null,
    last_updated_by              varchar2(255),
    last_updated_on              date,
    created_by                   varchar2(255),
    created_on                   date
    )
/

create index wwv_flow_treereg_fk1 on wwv_flow_tree_regions (region_id)
/

create index wwv_flow_treereg_fk2 on wwv_flow_tree_regions (flow_id)
/

--==============================================================================
create table wwv_flow_tree_state(
  id         number not null,
  tree_id    number not null,
  expand     clob default empty_clob(),
  contract   clob default empty_clob(),
  constraint wwv_flow_tree_state$pk primary key(id,tree_id),
  constraint wwv_flow_tree_state$fk foreign key(id) references wwv_flow_sessions$(id) on delete cascade)
  lob (expand,contract) store as (cache reads enable storage in row)
/

--##############################################################################
--#
--# APPLICATION > PAGE > REGION (CALENDAR)
--#
--##############################################################################
prompt APPLICATION > PAGE > REGION (CALENDAR)

--==============================================================================
create table wwv_flow_cals (
  id                          number
                              constraint      wwv_flow_cals_pk
                              primary key,
  flow_id                     number          not null
                              constraint      wwv_flow_cal_to_flow_fk
                              references      wwv_flows
                              on delete       cascade,
  plug_id                     number          not null
                              constraint      wwv_flow_plug_calendar_fk
                              references      wwv_flow_page_plugs
                              on delete       cascade,
  start_date                  varchar2(255),
  end_date                    varchar2(255),
  begin_at_start_of_interval  varchar2(1)
                              constraint      wwv_flow_val_calendar_si
                              check           (begin_at_start_of_interval in ('Y','N')),
  date_item                   varchar2(255)   default null,
  end_date_item               varchar2(255)   default null,
  display_as                  varchar2(100),
  display_item                varchar2(255),
  display_type                varchar2(10)
                               constraint      wwv_flow_dis_calendar_typ
                               check           (display_type in ('NONE','COL','CUS')),
  item_format                 varchar2(4000),
  easy_sql_owner              varchar2(255),
  easy_sql_table              varchar2(255),
  date_column                 varchar2(255),
  end_date_column             varchar2(255),
  display_column              varchar2(255),
  start_of_week               number,
  template_id                 number,
  item_link                   varchar2(4000),
  day_link                    varchar2(4000),
  start_time                  number,
  end_time                    number,
  time_format                 varchar2(6)
                              constraint wwv_flow_dis_time_format
                              check(time_format in ('12HOUR','24HOUR') ),
  week_start_day              number,
  week_end_day                number,
  date_type_column            varchar2(255),
  security_group_id           number not null,
  last_updated_by             varchar2(255),
  last_updated_on             date,
  calendar_type               varchar2(1),
  include_custom_cal          varchar2(1),
  custom_cal_days             number default 3,
  primary_key_column          varchar2(255),
  drag_drop_required_role     varchar2(255),
  drag_drop_process_id        number
                              constraint wwv_flow_step_process_fk
                              references wwv_flow_step_processing
                              on delete cascade deferrable initially deferred,
  item_link_open_in           varchar2(1)   default 'W'
                              constraint wwv_flow_item_link_in_ck
                              check (  item_link_open_in in ('W','P')) ,
  data_background_color       varchar2(255),
  data_text_color             varchar2(255),
  item_link_primary_key_item  varchar2(255)    default null,
  item_link_date_item         varchar2(255)    default null,
  include_time_with_date      varchar2(1)      default 'N',
  agenda_cal_days_type        varchar2(10)     default 'MONTH'
                              constraint wwv_flow_agenda_type_in_ck
                              check (  agenda_cal_days_type in ('MONTH','YEAR','CUSTOM')) ,
  agenda_cal_days             number default 30,
  calendar_comments           varchar2(4000)
  )
/

create index wwv_flow_cal_fk_idx1 on wwv_flow_cals (flow_id, security_group_id)
/
create index wwv_flow_cal_fk_idx2 on wwv_flow_cals (plug_id, security_group_id)
/
create index wwv_flow_cal_fk_idx3 on wwv_flow_cals (drag_drop_process_id)
/

create table wwv_flow_region_report_column (
  id                            number          not null
                                constraint report_column_pk
                                primary key,
  region_id                     number          not null
                                constraint report_column_to_region_fk
                                references wwv_flow_page_plugs
                                on delete cascade,
  flow_id                       number,
  query_column_id               number,
  form_element_id               number,
  column_alias                  varchar2(4000)   not null,
  static_id                     varchar2(255),
  column_display_sequence       number           not null,
  column_heading                varchar2(4000),
  use_as_row_header             varchar2(1) default 'N'
                                constraint wwv_flow_regrepcol_row_hdr
                                check (use_as_row_header in ('Y', 'N')),
  column_format                 varchar2(4000),
  column_html_expression        varchar2(4000),
  column_css_class              varchar2(4000),
  column_css_style              varchar2(4000),
  column_hit_highlight          varchar2(4000),
  column_link                   varchar2(4000),
  column_linktext               varchar2(4000),
  column_link_attr              varchar2(4000),
  column_link_checksum_type     varchar2(255) constraint wwv_regrepcol_checksum_type
                                check (column_link_checksum_type in (
                                '1', -- workspace
                                '2', -- user
                                '3'  -- session
                                )),
  column_alignment              varchar2(4000),
  heading_alignment             varchar2(4000),
  default_sort_column_sequence  number,
  default_sort_dir              varchar2(255),
  disable_sort_column           varchar2(1),
  sum_column                    varchar2(1),
  hidden_column                 varchar2(1),
  display_when_cond_type        varchar2(255),
  display_when_condition        varchar2(4000),
  display_when_condition2       varchar2(4000),
  report_column_required_role   varchar2(255),
  --
  display_as                   varchar2(255)
                               constraint wwv_regrepcol_display_as
                               check ( display_as in (
                               'READONLY',
                               'DISPLAY_AND_SAVE',
                               'WITHOUT_MODIFICATION',
                               'ESCAPE_SC',
                               'TEXT',
                               'TEXT_FROM_LOV',
                               'TEXT_FROM_LOV_ESC',
                               'TEXTAREA',
                               'SELECT_LIST',
                               'SELECT_LIST_FROM_LOV',
                               'SELECT_LIST_FROM_QUERY',
                               'CHECKBOX',
                               'SIMPLE_CHECKBOX',
                               'CHECKBOX_FROM_LOV',
                               'SIMPLE_CHECKBOX',
                               'RADIOGROUP',
                               'RADIOGROUP_FROM_LOV',
                               'RADIOGROUP_FROM_QUERY',
                               'HIDDEN',
                               'HIDDEN_PROTECTED',
                               'RICH_TEXT',
                               'POPUP',
                               'POPUP_QUERY',
                               'POPUPKEY',
                               'POPUPKEY_QUERY',
                               'DATE_POPUP',
                               'DATE_PICKER',
                               'PICK_DATE_USING_APP_FORMAT_MASK',
                               'PICK_DATE_USING_APP_DATE_FORMAT',
                               'PICK_DATE_USING_FORMAT_MASK',
                               'PICK_DATE_DD_MON_RR',
                               'PICK_DATE_DD_MON_RR_HH_MI',
                               'PICK_DATE_DD_MON_RR_HH24_MI',
                               'PICK_DATE_DD_MON_YYYY',
                               'PICK_DATE_DD_MON_YYYY_HH_MI',
                               'PICK_DATE_DD_MON_YYYY_HH24_MI',
                               'PICK_DATE_DD_MM_YYYY',
                               'PICK_DATE_DD_MM_YYYY_HH_MI',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI',
                               'PICK_DATE_MM_DD_YYYY',
                               'PICK_DATE_MM_DD_YYYY_HH_MI',
                               'PICK_DATE_MM_DD_YYYY_HH24_MI',
                               'PICK_DATE_DD_MON_YY',
                               'PICK_DATE_DD_MON_YY_HH_MI',
                               'PICK_DATE_DD_MON_YY_HH24_MI',
                               'PICK_DATE_YYYY_MM_DD',
                               'PICK_DATE_YYYY_MM_DD_HH_MI',
                               'PICK_DATE_YYYY_MM_DD_HH24_MI',
                               'PICK_DATE_RR_MON_DD',
                               'PICK_DATE_RR_MON_DD_HH_MI',
                               'PICK_DATE_RR_MON_DD_HH24_MI',
                               'PICK_DATE_DD_MM_YYYY_DASH',
                               'PICK_DATE_DD_MM_YYYY_HH_MI_DASH',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH',
                               'PICK_DATE_DD_MM_YYYY_DOT',
                               'PICK_DATE_DD_MM_YYYY_HH_MI_DOT',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT',
                               'PICK_DATE_YYYY_DD_MM_DOT',
                               'PICK_DATE_YYYY_DD_MM_HH_MI_DOT',
                               'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT',
                               'PICK_DATE_YYYY_MM_DD_SLASH',
                               'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH',
                               'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH'
                               ) or
                               display_as like 'NATIVE\_%' escape '\' or
                               display_as like 'PLUGIN\_%' Escape '\'),
  named_lov                    number,
  inline_lov                   varchar2(4000),
  lov_language                 varchar2(10)
                               constraint wwv_flow_regrepcol_lov_lang_ck
                               check (lov_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
  lov_show_nulls               varchar2(3)
                               constraint wwv_flow_regrepcol_lovn
                               check (lov_show_nulls in (
                               'YES',
                               'NO')),
  lov_null_text                varchar2(255),
  lov_null_value               varchar2(255),
  column_width                 number,
  report_column_width          number,
  column_height                number,
  css_classes                  varchar2(255),
  cattributes                  varchar2(4000),
  cattributes_element          varchar2(4000),
  column_field_template        number,
  --
  attribute_01                 varchar2(4000),
  attribute_02                 varchar2(4000),
  attribute_03                 varchar2(4000),
  attribute_04                 varchar2(4000),
  attribute_05                 varchar2(4000),
  attribute_06                 varchar2(4000),
  attribute_07                 varchar2(4000),
  attribute_08                 varchar2(4000),
  attribute_09                 varchar2(4000),
  attribute_10                 varchar2(4000),
  attribute_11                 varchar2(4000),
  attribute_12                 varchar2(4000),
  attribute_13                 varchar2(4000),
  attribute_14                 varchar2(4000),
  attribute_15                 varchar2(4000),
  --
  is_required                  varchar2(1)
                               constraint wwv_flow_regrepcol_is_required
                               check (is_required in ('Y', 'N')),
  standard_validations         varchar2(1)
                               constraint wwv_flow_regrepcol_std_val
                               check (standard_validations in ('BROWSER', 'SERVER', 'BROWSER_AND_SERVER', 'NONE')),
  --
  pk_col_source_type           varchar2(1)
                                   constraint wwv_flow_regrepcol_src_type
                                   check (pk_col_source_type in (
                                   'S',
                                   'F',
                                   'T')),
  pk_col_source                clob,
  derived_column               varchar2(1)
                                   constraint wwv_flow_regrepcol_derv_col
                                   check (derived_column in (
                                   'Y',
                                   'N')),
  --
  column_default               varchar2(4000),
  column_default_type          varchar2(255),
  lov_display_extra            varchar2(30),
  --
  include_in_export            varchar2(1)
                                   constraint wwv_flow_regrepcol_incl_exp
                                   check (include_in_export in (
                                   'Y',
                                   'N')),
  print_col_width              varchar2(4000),
  print_col_align              varchar2(4000),
  --
  ref_schema                   varchar2(255),
  ref_table_name               varchar2(255),
  ref_column_name              varchar2(255),
  --
  column_comment               varchar2(4000),
  --
  required_patch               number,
  --
  security_group_id            number           not null,
  last_updated_by              varchar2(255),
  last_updated_on              date
) lob (pk_col_source) store as (cache reads enable storage in row)
/

create index wwv_flow_region_report_col_idx on wwv_flow_region_report_column (region_id)
/

create index wwv_flow_region_rpt_col_idx2 on wwv_flow_region_report_column (security_group_id,flow_id)
/

create index wwv_flow_region_rpt_col_idx3 on wwv_flow_region_report_column (named_lov,flow_id)
/

--==============================================================================
-- UPDATEABLE report attributes
create table wwv_flow_region_upd_rpt_cols (
    id                           number
                                 constraint wwv_flow_upd_rpt_col
                                 primary key,
    flow_id                      number not null
                                 constraint wwv_flow_urc_to_flow_fk
                                 references wwv_flows
                                 on delete cascade,
    security_group_id            number not null,
    plug_id                      number not null,
    column_sequence              number not null,
    query_column_name            varchar2(255),
    display_as                   varchar2(255)
                                 constraint wwv_flow_upc_display_as
                                 check ( display_as in (
                                 'DISPLAY_AND_SAVE',
                                 'WITHOUT_MODIFICATION',
                                 'ESCAPE_SC',
                                 'TEXT',
                                 'TEXTAREA',
                                 'SELECT_LIST',
                                 'SELECT_LIST_FROM_LOV',
                                 'SELECT_LIST_FROM_QUERY',
                                 'HIDDEN',
                                 'POPUP',
                                 'DATE_POPUP'
                                 )),
    named_lov                    number,
    inline_lov                   varchar2(4000),
    lov_show_nulls               varchar2(3)
                                 constraint wwv_flow_upd_rpt_cols_lovn
                                 check (lov_show_nulls in (
                                 'YES',
                                 'NO')),
    lov_null_text                varchar2(255),
    lov_null_value               varchar2(255),
    column_width                 number,
    column_height                number,
    cattributes                  varchar2(4000),
    cattributes_element          varchar2(4000),
    column_comment               varchar2(4000),
    constraint                   wwv_flow_region_upd_rpt_u_name
                                 unique (query_column_name, plug_id) deferrable initially deferred using index compress 1
    )
/

create index wwv_flow_region_upd_rpt_ci2 on wwv_flow_region_upd_rpt_cols (plug_id,column_sequence);
create index wwv_flow_region_upd_rpt_ci3 on wwv_flow_region_upd_rpt_cols (flow_id)
/

alter table wwv_flow_region_upd_rpt_cols
    add constraint wwv_flow_urc_to_plug_fk
        foreign key (plug_id)
        references wwv_flow_page_plugs (id)
        on delete cascade
/

--##############################################################################
--#
--# APPLICATION > PAGE > REGION > COLUMNS
--#
--##############################################################################
prompt APPLICATION > PAGE > REGION > COLUMN GROUPS
--==============================================================================
create table wwv_flow_region_column_groups (
  id                     number not null
                             constraint region_col_group_pk primary key,
  flow_id                number not null,
  page_id                number not null,
  region_id              number not null
                             constraint region_col_group_region_fk
                             references wwv_flow_page_plugs on delete cascade,
  security_group_id      number not null,
  --
  heading                varchar2(255) not null,
  label                  varchar2(255),
  --
  last_updated_by        varchar2(255),
  last_updated_on        date
  )
/
alter table wwv_flow_region_column_groups
    add constraint region_col_group_page_fk foreign key ( flow_id, page_id ) references wwv_flow_steps( flow_id, id ) on delete cascade
/
create index region_col_group_page_fx on wwv_flow_region_column_groups ( flow_id, page_id ) compress
/
create index region_col_group_region_fx on wwv_flow_region_column_groups ( region_id ) compress
/


prompt APPLICATION > PAGE > REGION > COLUMNS
--==============================================================================
create table wwv_flow_region_columns (
  id                     number not null
                             constraint region_col_pk primary key,
  flow_id                number not null,
  page_id                number not null,
  region_id              number not null
                             constraint region_col_region_fk
                             references wwv_flow_page_plugs on delete cascade,
  security_group_id      number not null,
  -- Source
  name                   varchar2(128) not null,
  source_type            varchar2(14)
                             constraint region_col_source_type_ck
                             check ( source_type in (
                                         'DB_COLUMN',
                                         'SQL_EXPRESSION',
                                         'NONE' )),
  source_expression      varchar2(4000),
  data_type              varchar2(128),
  is_query_only          varchar2(1)
                             constraint region_col_is_query_only_ck
                             check ( is_query_only in ( 'Y', 'N' )),
  --
  item_type              varchar2(255)
                             constraint region_col_item_type_fk
                             check ( item_type like 'NATIVE\_%' ESCAPE '\' or
                                     item_type like 'PLUGIN\_%' ESCAPE '\' ),
  --
  is_visible             varchar2(1)
                             constraint region_col_is_visible_ck
                             check ( is_visible in ( 'Y', 'N' )),
  -- Heading
  heading                varchar2(255),
  label                  varchar2(255),
  heading_alignment      varchar2(6)
                             constraint region_col_heading_align_ck
                             check ( heading_alignment in (
                                         'LEFT',
                                         'RIGHT',
                                         'CENTER' )),
  -- Layout
  display_sequence       number not null,
  value_alignment        varchar2(6)
                             constraint region_col_value_align_ck
                             check ( value_alignment in (
                                         'LEFT',
                                         'RIGHT',
                                         'CENTER' )),
  value_css_classes      varchar2(255),
  value_attributes       varchar2(255),
  --
  group_id               number
                             constraint region_col_group_fk
                             references wwv_flow_region_column_groups on delete set null,
  use_group_for          varchar2(7)
                             constraint region_col_use_group_for_ck
                             check ( use_group_for in ( 'BOTH', 'SRV', 'HEADING' )),
  stretch                varchar2(1)
                             constraint region_col_stretch_ck
                             check ( stretch in ( 'N', 'A' )),
  -- Plug-in Settings
  plugin_init_javascript_code varchar2(4000),
  --
  attribute_01           varchar2(4000),
  attribute_02           varchar2(4000),
  attribute_03           varchar2(4000),
  attribute_04           varchar2(4000),
  attribute_05           varchar2(4000),
  attribute_06           varchar2(4000),
  attribute_07           varchar2(4000),
  attribute_08           varchar2(4000),
  attribute_09           varchar2(4000),
  attribute_10           varchar2(4000),
  attribute_11           varchar2(4000),
  attribute_12           varchar2(4000),
  attribute_13           varchar2(4000),
  attribute_14           varchar2(4000),
  attribute_15           varchar2(4000),
  attribute_16           varchar2(4000),
  attribute_17           varchar2(4000),
  attribute_18           varchar2(4000),
  attribute_19           varchar2(4000),
  attribute_20           varchar2(4000),
  attribute_21           varchar2(4000),
  attribute_22           varchar2(4000),
  attribute_23           varchar2(4000),
  attribute_24           varchar2(4000),
  attribute_25           varchar2(4000),
  -- Appearance
  format_mask            varchar2(255),
  item_width             number,
  item_height            number,
  item_placeholder       varchar2(255),
  item_css_classes       varchar2(255),
  item_icon_css_classes  varchar2(255),
  item_attributes        varchar2(255),
  -- Validation
  is_required            varchar2(1)
                             constraint region_col_required_ck
                             check ( is_required in ( 'Y', 'N' )),
  max_length             number,
  -- List of Values
  lov_type                 varchar2(19)
                               constraint region_col_lov_type_ck
                               check ( lov_type in ( 'SHARED', 'SQL_QUERY', 'STATIC', 'FUNCTION_BODY' )),
  lov_id                   number
                               constraint region_col_lov_fk
                               references wwv_flow_lists_of_values$
                               deferrable initially deferred,
  lov_language             varchar2(10)
                               constraint region_col_lov_lang_ck
                               check ( lov_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
  lov_source               varchar2(4000),
  lov_display_extra        varchar2(1)
                               constraint region_col_lov_displ_extra_ck
                               check ( lov_display_extra in ( 'Y', 'N' )),
  lov_display_null         varchar2(1)
                               constraint region_col_lov_displ_null_ck
                               check ( lov_display_null in ( 'Y', 'N' )),
  lov_null_text            varchar2(255),
  lov_null_value           varchar2(255),
  lov_cascade_parent_items varchar2(255),
  ajax_items_to_submit     varchar2(4000),
  ajax_optimize_refresh    varchar2(1)
                               constraint region_col_lov_ajax_refresh_ck
                               check ( ajax_optimize_refresh in ( 'Y', 'N' )),
  -- Link
  link_target              varchar2(4000),
  link_text                varchar2(255),
  link_attributes          varchar2(255),
  -- Filter
  enable_filter            varchar2(1)
                               constraint region_col_enable_filter_ck
                               check ( enable_filter in ( 'Y', 'N' )),
  filter_operators         varchar2(255),
  filter_is_required       varchar2(1)
                               constraint region_col_filter_is_req_ck
                               check ( filter_is_required in ( 'Y', 'N' )),
  filter_text_case         varchar2(5)
                               constraint region_col_filter_text_case_ck
                               check ( filter_text_case in ( 'UPPER', 'LOWER', 'MIXED' )),
  filter_exact_match       varchar2(1)
                               constraint region_col_filt_exact_match_ck
                               check ( filter_exact_match in ( 'Y', 'N' )),
  filter_date_ranges       varchar2(6)
                               constraint region_col_filt_date_ranges_ck
                               check ( filter_date_ranges in ( 'ALL', 'PAST', 'FUTURE' )),
  filter_lov_type          varchar2(19)
                               constraint region_col_filter_lov_type_ck
                               check ( filter_lov_type in ( 'NONE', 'DISTINCT', 'SHARED', 'SQL_QUERY', 'STATIC', 'FUNCTION_BODY', 'LOV' )),
  filter_lov_id            number
                               constraint region_col_filter_lov_fk
                               references wwv_flow_lists_of_values$
                               deferrable initially deferred,
  filter_lov_language      varchar2(10)
                               constraint region_col_filter_lov_lang_ck
                               check ( filter_lov_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
  filter_lov_query         varchar2(4000),
  -- Advanced
  static_id                varchar2(255),
  use_as_row_header        varchar2(1) default 'N'
                               constraint region_col_use_as_row_head_ck
                               check ( use_as_row_header in ( 'Y', 'N' )),
  javascript_code          varchar2(4000),
  -- Enable Users To
  enable_sort_group        varchar2(1)
                               constraint region_col_enable_sort_grp_ck
                               check ( enable_sort_group in ( 'Y', 'N' )),
  enable_control_break     varchar2(1)
                               constraint region_col_enable_ctrl_brk_ck
                               check ( enable_control_break in ( 'Y', 'N' )),
  enable_hide              varchar2(1)
                               constraint region_col_enable_hide_ck
                               check ( enable_hide in ( 'Y', 'N' )),
  enable_pivot             varchar2(1)
                               constraint region_col_enable_pivot_ck
                               check ( enable_pivot in ( 'Y', 'N' )),
  -- Primary Key
  is_primary_key           varchar2(1)
                               constraint region_col_is_primary_key_ck
                               check ( is_primary_key in ( 'Y', 'N' )),
  -- Parent - Child
  parent_column_id         number
                               constraint region_col_parent_column_fk
                               references wwv_flow_region_columns
                               deferrable initially deferred, -- deferrable is necessary for import
  -- Default
  default_type             varchar2(20)
                               constraint region_col_default_type_ck
                               check ( default_type in ( 'STATIC',
                                                         'ITEM',
                                                         'SQL_QUERY',
                                                         'SQL_QUERY_COLON',
                                                         'EXPRESSION',
                                                         'FUNCTION_BODY',
                                                         'SEQUENCE' )),
  default_language         varchar2(10)
                               constraint region_col_default_lang_ck
                               check ( default_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
  default_expression       varchar2(4000),
  duplicate_value          varchar2(1)
                               constraint region_col_duplicate_value_ck
                               check ( duplicate_value in ( 'Y', 'N' )),
  -- Export / Printing
  include_in_export        varchar2(1)
                               constraint region_col_incl_in_export_ck
                               check ( include_in_export in ( 'Y', 'N' )),
  -- Condition
  display_condition_type   varchar2(255),
  display_condition        varchar2(4000),
  display_condition2       varchar2(4000),
  -- Read Only Condition
  readonly_condition_type  varchar2(255),
  readonly_condition       varchar2(4000),
  readonly_condition2      varchar2(4000),
  readonly_for_each_row    varchar2(1)
                               constraint region_col_readonly_for_row_ck
                               check ( readonly_for_each_row in ( 'Y', 'N' )),
  -- Security
  escape_on_http_output    varchar2(1)
                               constraint region_col_escape_ck
                               check ( escape_on_http_output in ( 'Y', 'N' )),
  security_scheme          varchar2(255),
  restricted_characters    varchar2(20)
                               constraint region_col_restricted_char_ck
                               check ( restricted_characters in ( 'US_ONLY', 'WEB_SAFE', 'NO_SPECIAL_CHAR', 'NO_SPECIAL_CHAR_NL', 'WORKSPACE_SCHEMA' )),
  -- Help Text
  help_text                varchar2(4000),
  --
  required_patch           number,
  -- Comment
  column_comment           varchar2(4000),
  -- Audit
  last_updated_by          varchar2(255),
  last_updated_on          date
)
/

-- foreign key constraint for wwv_flow_region_columns on page and app id to wwv_flow_steps
alter table wwv_flow_region_columns
    add constraint region_col_page_fk foreign key ( flow_id, page_id ) references wwv_flow_steps( flow_id, id ) on delete cascade
/
create index region_col_page_fx on wwv_flow_region_columns ( flow_id, page_id ) compress
/
alter table wwv_flow_region_columns
    add constraint region_col_name_uk unique ( region_id, name ) deferrable initially deferred using index compress 1
/
create index region_col_group_fx on wwv_flow_region_columns ( group_id ) compress
/
create index region_col_lov_fx on wwv_flow_region_columns ( lov_id ) compress
/
create index region_col_filter_lov_fx on wwv_flow_region_columns ( filter_lov_id ) compress
/
create index region_col_parent_column_fx on wwv_flow_region_columns ( parent_column_id ) compress
/

--##############################################################################
--#
--# APPLICATION > PAGE > REGION (INTERACTIVE REPORTS - GENERIC)
--#
--##############################################################################
prompt APPLICATION > PAGE > REGION (INTERACTIVE REPORTS - GENERIC)
--==============================================================================
create table wwv_flow_worksheets (
    id                        number not null,
    flow_id                   number not null
                              constraint wwv_flow_worksheets_flow_fk
                              references wwv_flows
                              on delete cascade,
    page_id                   number,
    region_id                 number
                              constraint wwv_flow_worksheets_reg_fk
                              references wwv_flow_page_plugs
                              on delete cascade,
    name                      varchar2(255),
    folder_id                 number,
    alias                     varchar2(255),
    -- view settings
    report_id_item            varchar2(255),
    max_row_count             varchar2(255),
    max_row_count_message     varchar2(4000),
    no_data_found_message     varchar2(4000),
    max_rows_per_page         varchar2(4000),
    search_button_label       varchar2(4000),
    page_items_to_submit      varchar2(4000), /* obsolete, is stored in wwv_flow_page_plugs.ajax_items_to_submit now */
    sort_asc_image            varchar2(4000),
    sort_asc_image_attr       varchar2(4000),
    sort_desc_image           varchar2(4000),
    sort_desc_image_attr      varchar2(4000),
    -- base table
    sql_query                 clob,
    base_table_or_view        varchar2(4000),
    base_pk1                  varchar2(128),
    base_pk2                  varchar2(128),
    base_pk3                  varchar2(128),
    uniquely_identify_rows_by varchar2(6) generated always as (
                                            case
                                              when base_pk1 is not null then 'COLUMN'
                                              else 'ROWID'
                                            end ) virtual,
    sql_hint                  varchar2(4000),
    oracle_text_index_column  varchar2(128),
    -- status
    status                    varchar2(30)
                              constraint wwv_flow_ws_status_ck
                              check (status in (
                              'AVAILABLE_FOR_OWNER',
                              'NOT_AVAILABLE',
                              'AVAILABLE',
                              'ACL')),
    --
    allow_report_saving       varchar2(1)
                              constraint wwv_flow_ws_allow_save_ck
                              check (allow_report_saving in (
                              'Y',
                              'N')),
    allow_save_rpt_public     varchar2(1)
                              constraint wwv_flow_ws_allow_save_pub_ck
                              check (allow_save_rpt_public in (
                              'Y',
                              'N')),
    save_rpt_public_auth_scheme    varchar2(255),
    allow_report_categories   varchar2(1)
                              constraint wwv_flow_ws_allow_cat_ck
                              check (allow_report_categories in (
                              'Y',
                              'N')),
    --
    show_nulls_as             varchar2(255),
    pagination_type           varchar2(255)
                              constraint wwv_flow_ws_pag_type
                              check (pagination_type in (
                              'ROWS_X_TO_Y_OF_Z',
                              'ROWS_X_TO_Y'
                              )),
    pagination_display_position    varchar2(255),
    button_template           number,   -- obsolete
    --
    show_finder_drop_down     varchar2(1) not null
                              constraint wwv_flow_show_fnd_drop_ck
                              check (show_finder_drop_down in (
                              'Y',
                              'N')),
    show_display_row_count    varchar2(1) not null
                              constraint wwv_flow_show_row_cnt_ck
                              check (show_display_row_count in (
                              'Y',
                              'N')),
    show_search_bar           varchar2(1) not null
                              constraint wwv_flow_show_search_bar_ck
                              check (show_search_bar in (
                              'Y',
                              'N')),
    show_search_textbox       varchar2(1) not null
                              constraint wwv_flow_show_search_box_ck
                              check (show_search_textbox in (
                              'Y',
                              'N')),
    show_actions_menu         varchar2(1) not null
                              constraint wwv_flow_show_actions_ck
                              check (show_actions_menu in (
                              'Y',
                              'N')),
    actions_menu_icon         varchar2(4000),      -- obsolete
    finder_icon               varchar2(4000),      -- obsolete
    report_list_mode          varchar2(255)
                              constraint wwv_flow_list_reps_ck
                              check (report_list_mode in (
                              'TABS',
                              'NONE')),
    show_select_columns       varchar2(1) not null
                              constraint wwv_flow_ws_show_col_ck
                              check (show_select_columns in (
                              'Y',
                              'N')),
    show_rows_per_page        varchar2(1) not null
                              constraint wwv_flow_ws_show_rows_ck
                              check (show_rows_per_page in (
                              'Y',
                              'N')),
    show_filter               varchar2(1) not null
                              constraint wwv_flow_ws_show_filter_ck
                              check (show_filter in (
                              'Y',
                              'N')),
    show_sort                 varchar2(1) not null
                              constraint wwv_flow_ws_show_sort_ck
                              check (show_sort in (
                              'Y',
                              'N')),
    show_control_break        varchar2(1) not null
                              constraint wwv_flow_ws_show_break_ck
                              check (show_control_break in (
                              'Y',
                              'N')),
    show_highlight            varchar2(1) not null
                              constraint wwv_flow_ws_show_hl_ck
                              check (show_highlight in (
                              'Y',
                              'N')),
    show_computation          varchar2(1) not null
                              constraint wwv_flow_ws_show_comp_ck
                              check (show_computation in (
                              'Y',
                              'N')),
    show_aggregate            varchar2(1) not null
                              constraint wwv_flow_ws_show_agg_ck
                              check (show_aggregate in (
                              'Y',
                              'N')),
    show_chart                varchar2(1) not null
                              constraint wwv_flow_ws_show_chart_ck
                              check (show_chart in (
                              'Y',
                              'N')),
    show_notify               varchar2(1) not null
                              constraint wwv_flow_ws_show_notify_ck
                              check (show_notify in (
                              'Y',
                              'N')),
    show_group_by             varchar2(1) not null
                              constraint wwv_flow_ws_show_group_by_ck
                              check (show_group_by in (
                              'Y',
                              'N')),
    show_pivot                varchar2(1) default 'N' not null
                              constraint wwv_flow_ws_show_pivot_ck
                              check (show_pivot in (
                              'Y',
                              'N')),
    show_calendar             varchar2(1) not null
                              constraint wwv_flow_ws_show_cal_ck
                              check (show_calendar in (
                              'Y',
                              'N')),
    show_flashback            varchar2(1) not null
                              constraint wwv_flow_ws_show_fb_ck
                              check (show_flashback in (
                              'Y',
                              'N')),
    show_reset                varchar2(1) not null
                              constraint wwv_flow_ws_show_reset_ck
                              check (show_reset in (
                              'Y',
                              'N')),
    show_download             varchar2(1) not null
                              constraint wwv_flow_ws_show_dl_ck
                              check (show_download in (
                              'Y',
                              'N')),
    show_help                 varchar2(1) not null
                              constraint wwv_flow_ws_show_help_ck
                              check (show_help in (
                              'Y',
                              'N')),
    show_detail_link          varchar2(1) not null
                              constraint wwv_flow_ws_detail_link_ck
                              check (show_detail_link in (
                              'Y',
                              'N',
                              'C')),
    download_auth_scheme      varchar2(40),
    download_formats          varchar2(30),
    download_filename         varchar2(255),
    enable_mail_download      varchar2(1) default 'N'
                                constraint wwv_flow_ws_mail_download_ck
                                check ( enable_mail_download in ( 'Y', 'N' )),
    csv_output_separator      varchar2(1),
    csv_output_enclosed_by    varchar2(1),
    supplemental_text         varchar2(4000),
    --
    email_from                varchar2(255),
    --
    detail_link               varchar2(4000),
    detail_link_text          varchar2(4000),
    detail_link_attr          varchar2(4000),
    detail_link_checksum_type varchar2(255) constraint wwv_ir_link_checksum_type
                                check (detail_link_checksum_type in (
                                '1', -- workspace
                                '2', -- user
                                '3'  -- session
                                )),
    detail_link_condition_type varchar2(255),
    detail_link_cond          varchar2(4000),
    detail_link_cond2         varchar2(4000),
    detail_link_auth_scheme   varchar2(4000),
    allow_exclude_null_values varchar2(1) not null
                              constraint wwv_flow_ws_exclude_null_ck
                              check (allow_exclude_null_values in (
                              'Y',
                              'N')),
    allow_hide_extra_columns  varchar2(1) not null
                              constraint wwv_flow_ws_hide_extra_ck
                              check (allow_hide_extra_columns in (
                              'Y',
                              'N')),
    --
    max_query_cost            varchar2(4000),
    max_flashback             varchar2(4000),
    --
    worksheet_flags           varchar2(4000),
    --
    fixed_header              varchar2(6) default 'NONE' not null
                              constraint wwv_flow_ws_fixed_header_ck
                              check (fixed_header in (
                              'NONE',
                              'PAGE',
                              'REGION')),
    fixed_header_max_height   number,
    --
    lazy_loading              varchar2(1) default 'N' not null
                              constraint wwv_wksh_lazy_loading_ck
                              check ( lazy_loading in ( 'Y', 'N' )),
    --
    description               varchar2(4000),
    owner                     varchar2(255),
    --
    icon_view_enabled_yn      varchar2(1) default 'N' not null
                              constraint wwv_flow_ws_iv_enab_ck
                              check (icon_view_enabled_yn in (
                              'Y','N')),
    icon_view_columns_per_row number,
    icon_view_use_custom      varchar2(1) default 'N'
                              constraint wwv_flow_ws_iv_cus_ck
                              check (icon_view_use_custom in (
                              'Y','N')),
    icon_view_custom_link     varchar2(4000),
    icon_view_link_column     varchar2(255),
    icon_view_img_src_column  varchar2(255),
    icon_view_label_column    varchar2(255),
    icon_view_img_attr_text   varchar2(512),
    icon_view_alt_text        varchar2(512),
    icon_view_title_text      varchar2(512),
    --
    detail_view_enabled_yn    varchar2(1) default 'N' not null
                              constraint wwv_flow_ws_enab_dv_ck
                              check (detail_view_enabled_yn in (
                              'Y','N')),
    detail_view_before_rows   varchar2(4000),
    detail_view_for_each_row  varchar2(4000),
    detail_view_after_rows    varchar2(4000),
    --
    internal_uid              number,
    --
    created_on                date,
    created_by                varchar2(255),
    updated_on                date,
    updated_by                varchar2(255),
    security_group_id         number        not null
    )
/

alter table wwv_flow_worksheets
    add constraint wwv_flow_worksheets_pk
    primary key(id)
/

create index wwv_flow_ws_idx1 on wwv_flow_worksheets (flow_id, page_id);
create index wwv_flow_ws_idx2 on wwv_flow_worksheets (region_id);

create unique index wwv_flow_ws_unq_alias_idx on wwv_flow_worksheets
    (case when alias is not null then flow_id end,
     alias)
/


--==============================================================================
--##############################################################################
--#
--# APPLICATION > PAGE > REGION (CHART)
--#
--##############################################################################

--==============================================================================
-- JET Charts
--==============================================================================
create table wwv_flow_jet_charts (
      id                          number
                                  constraint jet_charts_pk
                                  primary key,
      flow_id                     number not null
                                  constraint jet_charts_flow_fk
                                  references wwv_flows
                                  on delete cascade,
      page_id                     number,
      region_id                   number not null
                                  constraint jet_charts_region_fk
                                  references wwv_flow_page_plugs
                                  on delete cascade,
      security_group_id           number not null,
      chart_type                  varchar2(15) not null
                                  constraint jet_charts_type
                                  check (chart_type in ('area', 'bar', 'boxPlot', 'bubble', 'combo','funnel', 'gantt', 'line', 'lineWithArea','pie' ,'scatter', 'stock', 'dial', 'donut', 'range', 'polar', 'radar', 'pyramid')),
      title                       varchar2(255),
      width                       varchar2(10),
      height                      varchar2(10),
      animation_on_display        varchar2(10) not null
                                  constraint jet_charts_animation_disp
                                  check (animation_on_display in ('alphaFade', 'auto', 'none', 'zoom' )),
      animation_on_data_change    varchar2(15) not null
                                  constraint jet_charts_animation_data
                                  check (animation_on_data_change in ('auto', 'slideToLeft', 'slideToRight', 'none')),
      orientation                 varchar2(10)
                                  constraint jet_charts_orientation
                                  check (orientation in ('horizontal','vertical')),
      data_cursor                 varchar2(5)
                                  constraint jet_charts_datacursor
                                  check (data_cursor in ('auto','on','off')),
      data_cursor_behavior        varchar2(10)
                                  constraint jet_charts_data_cb
                                  check (data_cursor_behavior in ('auto','smooth','snap')),
      hide_and_show_behavior      varchar2(15)
                                  constraint jet_charts_hideshow
                                  check (hide_and_show_behavior in ('none','withRescale','withoutRescale')),
      hover_behavior              varchar2(5)
                                  constraint jet_charts_hover
                                  check (hover_behavior in ('dim','none')),
      stack                       varchar2(3)
                                  constraint jet_charts_stack
                                  check (stack in ('on','off')),
      stack_label                 varchar2(3)
                                  constraint jet_charts_stack_label
                                  check (stack_label in ('on','off')),
      stack_font_family           varchar2(20)
                                  constraint jet_stack_font_family
                                  check (stack_font_family in ('Arial','Arial Black','Bookman','Comic Sans MS','Courier','Courier New','Garamond','Georgia','Helvetica','Impact','Palatino','Times','Times New Roman','Trebuchet MS','Verdana')),
      stack_font_style            varchar2(10)
                                  constraint jet_stack_font_style
                                  check (stack_font_style in ('normal','italic','oblique')),
      stack_font_size             varchar2(2)
                                  constraint jet_stack_font_size
                                  check (stack_font_size in ('8','10','12','14','16','18','20','22')),
      stack_font_color            varchar2(128),
      spark_chart                 varchar2(1)
                                  constraint jet_charts_spark
                                  check (spark_chart in ('Y','N')),
      connect_nulls               varchar2(1)
                                  constraint jet_series_connect_nulls
                                  check (connect_nulls in ('Y','N')),
      stock_render_as             varchar2(15)
                                  constraint jet_charts_stock
                                  check (stock_render_as in ('area','bar','candlestick','line','lineWithArea')),
      dial_indicator              varchar2(15)
                                  constraint jet_charts_dial_indicator
                                  check (dial_indicator in ('needleAlta', 'needleLight','needleDark','needleAntique')),
      dial_background             varchar2(30)
                                  constraint jet_charts_dial_background
                                  check (dial_background in ('circleAlta','domeAlta','rectangleAlta','circleLight','domeLight','rectangleLight',
                                                             'circleDark','domeDark','rectangleDark','circleAntique','domeAntique','rectangleAntique')),
      value_min                   number,
      value_text_type             varchar2(10)
                                  constraint jet_charts_text_type
                                  check (value_text_type in ('number','percent')),
      value_position              varchar2(20)
                                  constraint jet_charts_value_position
                                  check (value_position in ('auto','center','insideIndicatorEdge','outsideIndicatorEdge','outsidePlotArea','withLabel')),
      value_format_type           varchar2(20)
                                  constraint jet_charts_format_type
                                  check (value_format_type in ('date-short','date-medium','date-long','date-full','time-short','time-medium', 'time-long','time-full',
                                                               'datetime-short','datetime-medium','datetime-long','datetime-full','decimal','currency','percent')),
      value_decimal_places        number,
      value_currency              varchar2(128),
      value_numeric_pattern       varchar2(30),
      value_format_scaling        varchar2(12)
                                  constraint jet_charts_scaling
                                  check (value_format_scaling in ('auto','none','thousand','million','billion','trillion','quadrillion')),
      sorting                     varchar2(10)
                                  constraint jet_chart_sorting
                                  check (sorting in ('value-asc','value-desc','label-desc','label-asc')),
      fill_multi_series_gaps      varchar2(1)
                                  constraint jet_charts_fill_gaps
                                  check (fill_multi_series_gaps in ('Y','N')),
      zoom_and_scroll             varchar2(20)
                                  constraint jet_charts_zoom
                                  check (zoom_and_scroll in ('delayed','delayedScrollOnly','live','liveScrollOnly','off')),
      zoom_direction              varchar2(4)
                                  constraint jet_charts_zoom_direction
                                  check (zoom_direction in ('auto','x','y')),
      initial_zooming             varchar2(5)
                                  constraint jet_charts_init_zoom
                                  check (initial_zooming in ('none','first','last')),
      tooltip_rendered            varchar2(1)
                                  constraint jet_charts_tooltip
                                  check (tooltip_rendered in ('Y','N','C')),
      show_series_name            varchar2(1)
                                  constraint jet_charts_series
                                  check (show_series_name in ('Y','N')),
      show_group_name             varchar2(1)
                                  constraint jet_charts_group
                                  check (show_group_name in ('Y','N')),
      show_value                  varchar2(1)
                                  constraint jet_charts_value
                                  check (show_value in ('Y','N')),
      show_label                  varchar2(1)
                                  constraint jet_charts_label
                                  check (show_label in ('Y','N')),
      -- Gantt Tooltip columns
      show_row                    varchar2(1)
                                  constraint jet_charts_gantt_row
                                  check (show_row in ('Y','N')),
      show_start                  varchar2(1)
                                  constraint jet_charts_gantt_start
                                  check (show_start in ('Y','N')),
      show_end                    varchar2(1)
                                  constraint jet_charts_gantt_end
                                  check (show_end in ('Y','N')),
      show_progress               varchar2(1)
                                  constraint jet_charts_gantt_progress
                                  check (show_progress in ('Y','N')),
      show_baseline               varchar2(1)
                                  constraint jet_charts_gantt_baseline
                                  check (show_baseline in ('Y','N')),
      --
      custom_tooltip              varchar2(255),
      legend_rendered             varchar2(3)
                                  constraint jet_charts_legend
                                  check (legend_rendered in ('on','off')),
      legend_title                varchar2(255),
      legend_position             varchar2(10)
                                  constraint jet_charts_legend_pos
                                  check (legend_position in ('auto','bottom','end','start','top')),
      legend_font_family          varchar2(20)
                                  constraint jet_legend_font_family
                                  check (legend_font_family in ('Arial','Arial Black','Bookman','Comic Sans MS','Courier','Courier New','Garamond','Georgia','Helvetica','Impact','Palatino','Times','Times New Roman','Trebuchet MS','Verdana')),
      legend_font_style           varchar2(10)
                                  constraint jet_legend_font_style
                                  check (legend_font_style in ('normal','italic','oblique')),
      legend_font_size            varchar2(2)
                                  constraint jet_legend_font_size
                                  check (legend_font_size in ('8','10','12','14','16','18','20','22')),
      legend_font_color           varchar2(128),
      overview_rendered           varchar2(3)
                                  constraint jet_charts_overview
                                  check (overview_rendered in ('on','off')),
      overview_height             varchar2(5),
      pie_other_threshold         number
                                  constraint jet_charts_pie_other
                                  check (pie_other_threshold between 0 and 1),
      pie_selection_effect        varchar2(20)
                                  constraint jet_charts_pie_effect
                                  check (pie_selection_effect in ('explode','highlightAndExplode','highlight')),
      time_axis_type              varchar2(15)
                                  constraint jet_charts_axis
                                  check (time_axis_type in ('enabled','mixedFrequency','disabled','auto', 'skipGaps')),
      -- Gantt Chart columns
      horizontal_grid             varchar2(10)
                                  constraint jet_charts_horizontal_grid
                                  check (horizontal_grid in ('auto','hidden','visible')),
      vertical_grid               varchar2(10)
                                  constraint jet_charts_vertical_grid
                                  check (vertical_grid in ('auto','hidden','visible')),
      row_axis_rendered           varchar2(3)
                                  constraint jet_row_axis
                                  check (row_axis_rendered in ('on','off')),
      gantt_axis_position         varchar2(6)
                                  constraint jet_charts_axis_position
                                  check (gantt_axis_position in ('top','bottom')),
      -- Status Meter Gauge columns
      gauge_orientation           varchar2(10)
                                  constraint jet_charts_gauge_orient
                                  check (gauge_orientation in ('horizontal','vertical','circular')),
      gauge_indicator_size        number,
      gauge_inner_radius          number,
      gauge_plot_area             varchar2(3)
                                  constraint jet_charts_gauge_plot
                                  check (gauge_plot_area in ('on','off')),
      gauge_start_angle           number,
      gauge_angle_extent          number,
      show_gauge_value            varchar2(1)
                                  constraint jet_charts_gauge_value
                                  check (show_gauge_value in ('Y','N')),
      no_data_found_message       varchar2(4000),
      javascript_code             varchar2(4000),
      automatic_refresh_interval  number,
      created_on                  date,
      created_by                  varchar2(255),
      updated_on                  date,
      updated_by                  varchar2(255)
  )
/

create index wwv_flow_jet_ch_fid_fk on wwv_flow_jet_charts (flow_id,page_id)
/

create index wwv_flow_jet_ch_fid_fk2 on wwv_flow_jet_charts (region_id)
/

--==============================================================================
create table wwv_flow_jet_chart_series (
       id                           number
                                    constraint jet_series_pk
                                    primary key,
       chart_id                     number not null
                                    constraint jet_series_fk
                                    references wwv_flow_jet_charts
                                    on delete cascade,
       static_id                    varchar2(128),
       flow_id                      number not null
                                    constraint jet_series_flow_fk references wwv_flows
                                    on delete cascade,
       page_id                      number,
       security_group_id            number not null,
       seq                          number not null,
       name                         varchar2(255) not null,
       data_source_type             varchar2(30)
                                    constraint jet_series_data_src_type
                                    check (data_source_type in ('SQL','FUNC_BODY_RETURNING_SQL','TABLE')),
       function_body_language       varchar2(10)
                                    constraint jet_series_data_func_lang_ck
                                       check (function_body_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
       data_source                  clob,
       max_row_count                number,
       ajax_items_to_submit         varchar2(4000),
       --
       location                     varchar2(13) not null
                                    constraint jet_series_location_ck
                                    check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE', 'REGION_SOURCE' )),
       remote_server_id             number
                                    constraint jet_series_remote_server_fk
                                    references wwv_remote_servers,
       web_src_module_id            number
                                    constraint jet_series_web_src_module_fk
                                    references wwv_flow_web_src_modules,
       use_local_sync_table         varchar2(1)
                                    constraint jet_series_wsm_usesynctab_ck
                                    check (use_local_sync_table in ('Y','N')),
       query_owner                  varchar2(128),
       query_table                  varchar2(128),
       query_where                  varchar2(4000),
       query_order_by_type          varchar2(6)
                                    constraint jet_series_orderby_type_ck
                                    check ( query_order_by_type in ( 'ITEM', 'STATIC' ) ),
       query_order_by               varchar2(4000),
       source_post_processing       varchar2(23)
                                    constraint jet_series_post_proc_ck
                                    check ( source_post_processing in ( 'WHERE_ORDER_BY_CLAUSE', 'SQL', 'FUNC_BODY_RETURNING_SQL' )),
       include_rowid_column         varchar2(1)
                                    constraint jet_include_rowid_columnn_ck
                                    check (include_rowid_column in ('Y', 'N')),
       optimizer_hint               varchar2(255),
       --
       remote_sql_caching           varchar2(9)
                                    constraint jet_remote_sql_caching_ck
                                    check (remote_sql_caching in ('ALL_USERS', 'USER', 'SESSION')),
       remote_sql_invalidate_when   varchar2(255),
       --
       external_filter_expr         varchar2(4000),
       external_order_by_type       varchar2(6)
                                    constraint jet_series_extorderby_type_ck
                                    check ( external_order_by_type in ( 'ITEM', 'STATIC' ) ),
       external_order_by_expr       varchar2(4000),
       --
       series_type                  varchar2(15)
                                    constraint jet_series_type
                                    check (series_type in ('area','areaRange','bar','barRange','boxPlot','bubble','stock','dial','donut','funnel','line','lineWithArea','pie','scatter','pyramid','gantt')),
       series_name_column_name      varchar2(128),
       items_value_column_name      varchar2(128),
       items_low_column_name        varchar2(128),
       items_high_column_name       varchar2(128),
       items_open_column_name       varchar2(128),
       items_close_column_name      varchar2(128),
       items_volume_column_name     varchar2(128),
       items_x_column_name          varchar2(128),
       items_y_column_name          varchar2(128),
       items_z_column_name          varchar2(128),
       items_target_value           varchar2(128),
       items_min_value              varchar2(128),
       items_max_value              varchar2(128),
       --
       aggregate_function           varchar2(21)
                                    constraint jet_aggregate_func
                                    check (aggregate_function in ( 'SUM',
                                                                   'AVG', 'MEDIAN',
                                                                   'COUNT', 'COUNT_DISTINCT', 'APPROX_COUNT_DISTINCT',
                                                                   'MIN', 'MAX' )),
       -- Gantt Chart Attributes
       gantt_start_date_source      varchar2(30)
                                    constraint jet_gantt_start_src
                                    check (gantt_start_date_source in ('ITEM','DB_COLUMN')),
       gantt_start_date_column      varchar2(128),
       gantt_start_date_item        varchar2(128),
       gantt_end_date_source        varchar2(30)
                                    constraint jet_gantt_end_src
                                    check (gantt_end_date_source in ('ITEM','DB_COLUMN')),
       gantt_end_date_column        varchar2(128),
       gantt_end_date_item          varchar2(128),
       -- Row
       gantt_row_id                 varchar2(128),
       gantt_row_name               varchar2(128),
       -- Tasks
       gantt_task_id                varchar2(128),
       gantt_task_name              varchar2(128),
       gantt_task_start_date        varchar2(128),
       gantt_task_end_date          varchar2(128),
       gantt_task_css_style         varchar2(128),
       gantt_task_css_class         varchar2(128),
       -- Dependencies
       gantt_predecessor_task_id    varchar2(128),
       gantt_successor_task_id      varchar2(128),
       --  Baseline Settings
       gantt_baseline_start_column  varchar2(128),
       gantt_baseline_end_column    varchar2(128),
       gantt_baseline_css_class     varchar2(128),
       -- Progress
       gantt_progress_column        varchar2(128),
       gantt_progress_css_class     varchar2(128),
       -- Viewport
       gantt_viewport_start_source  varchar2(30)
                                    constraint jet_gantt_view_start_src
                                    check (gantt_viewport_start_source in ('ITEM','DB_COLUMN')),
       gantt_viewport_start_column  varchar2(128),
       gantt_viewport_start_item    varchar2(128),
       gantt_viewport_end_source    varchar2(30)
                                    constraint jet_gantt_view_end_src
                                    check (gantt_viewport_end_source in ('ITEM','DB_COLUMN')),
       gantt_viewport_end_column    varchar2(128),
       gantt_viewport_end_item      varchar2(128),
       task_label_position          varchar2(15)
                                    constraint jet_task_label_position
                                    check (task_label_position in ('end','innerCenter','innerEnd','innerStart','none','start')),
       --
       group_name_column_name       varchar2(128),
       group_short_desc_column_name varchar2(128),
       items_label_column_name      varchar2(128),
       items_short_desc_column_name varchar2(128),
       custom_column_name           varchar2(128),
       color                        varchar2(128),
       q2_color                     varchar2(128),
       q3_color                     varchar2(128),
       line_style                   varchar2(6)
                                    constraint jet_series_line_style
                                    check (line_style in ('dashed','dotted','solid')),
       line_width                   number
                                    constraint jet_series_line_width
                                    check (line_width between 0 and 20),
       line_type                    varchar2(20)
                                    constraint jet_series_line_type
                                    check (line_type in ('centeredStepped','centeredSegmented','curved','segmented','stepped','straight','auto','none')),
       marker_rendered              varchar2(5)
                                    constraint jet_series_show_marker
                                    check (marker_rendered in ('auto','on','off')),
       marker_shape                 varchar2(15)
                                    constraint jet_series_marker
                                    check (marker_shape in ('auto','circle','diamond','plus','triangleDown','triangleUp','human','square','star')),
       assigned_to_y2               varchar2(3)
                                    constraint jet_series_assign_to_y2
                                    check (assigned_to_y2 in ('on','off')),
       stack_category               varchar2(100),
       items_label_rendered         varchar2(1) not null
                                    constraint jet_series_show_label
                                    check (items_label_rendered in ('Y','N')),
       items_label_position         varchar2(15)
                                    constraint jet_series_label_position
                                    check (items_label_position in ('center','outsideSlice','aboveMarker','belowMarker','beforeMarker','afterMarker','insideBarEdge','outsideBarEdge','auto','none','start')),
       items_label_display_as       varchar2(7)
                                    constraint jet_series_label_display_as
                                    check (items_label_display_as in ('LABEL','VALUE','PERCENT','ALL','COMBO','LBL_VAL')),
       items_label_css_classes      varchar2(255),
       items_label_font_family      varchar2(20)
                                    constraint jet_label_font_family
                                    check (items_label_font_family in ('Arial','Arial Black','Bookman','Comic Sans MS','Courier','Courier New','Garamond','Georgia','Helvetica','Impact','Palatino','Times','Times New Roman','Trebuchet MS','Verdana')),
       items_label_font_style       varchar2(10)
                                    constraint jet_label_font_style
                                    check (items_label_font_style in ('normal','italic','oblique')),
       items_label_font_size        varchar2(2)
                                    constraint jet_label_font_size
                                    check (items_label_font_size in ('8','10','12','14','16','18','20','22')),
       items_label_font_color       varchar2(128),
       --
       gauge_plot_area_color        varchar2(128),
       threshold_values             varchar2(4000),
       threshold_colors             varchar2(4000),
       threshold_display            varchar2(11)
                                    constraint jet_series_threshold_display
                                    check (threshold_display in ('all','currentOnly','onIndicator')),
       reference_line_values        varchar2(4000),
       reference_line_colors        varchar2(4000),
       --
       link_target                  varchar2(4000),
       link_target_type             varchar2(22)
                                    constraint jet_series_link_type
                                    check (link_target_type in ('REDIRECT_URL','REDIRECT_PAGE','REDIRECT_APP','DEFINED_BY_DA', 'DEPRECATED_LINK_COLUMN' )),
       security_scheme              varchar2(255),
       required_patch               number,
       series_comment               varchar2(4000),
       display_when_cond_type       varchar2(255),
       display_when_condition       varchar2(4000),
       display_when_condition2      varchar2(4000),
       created_on                   date,
       created_by                   varchar2(255),
       updated_on                   date,
       updated_by                   varchar2(255)
   ) lob (data_source) store as (cache enable storage in row)
/

create index jet_series_idx2 on wwv_flow_jet_chart_series (chart_id)
/

create index jet_series_idx3 on wwv_flow_jet_chart_series (flow_id,page_id) compress
/

create index jet_series_idx4 on wwv_flow_jet_chart_series ( remote_server_id )
/

create index jet_series_idx5  on wwv_flow_jet_chart_series ( web_src_module_id )
/

--==============================================================================
create table wwv_flow_jet_chart_axes (
      id                          number
                                  constraint jet_axes_pk
                                  primary key,
      chart_id                    number not null
                                  constraint jet_axes_fk
                                  references wwv_flow_jet_charts
                                  on delete cascade,
      static_id                   varchar2(128),
      flow_id                     number not null
                                  constraint jet_axes_flow_fk references wwv_flows
                                  on delete cascade,
      page_id                     number,
      security_group_id           number not null,
      axis                        varchar2(5)
                                  constraint jet_axes_axis
                                  check (axis in ('x','y','y2','major','minor')),
      is_rendered                 varchar2(5)
                                  constraint jet_axes_rendered
                                  check (is_rendered in ('on','off')),
      title                       varchar2(255),
      title_font_family           varchar2(20)
                                  constraint jet_title_font_family
                                  check (title_font_family in ('Arial','Arial Black','Bookman','Comic Sans MS','Courier','Courier New','Garamond','Georgia','Helvetica','Impact','Palatino','Times','Times New Roman','Trebuchet MS','Verdana')),
      title_font_style            varchar2(10)
                                  constraint jet_title_font_style
                                  check (title_font_style in ('normal','italic','oblique')),
      title_font_size             varchar2(2)
                                  constraint jet_title_font_size
                                  check (title_font_size in ('8','10','12','14','16','18','20','22')),
      title_font_color            varchar2(128),
      min                         number,
      max                         number,
      format_type                 varchar2(20)
                                  constraint jet_axes_format_type
                                  check (format_type in ('date-short','date-medium','date-long','date-full',
                                                         'time-short','time-medium', 'time-long','time-full',
                                                         'datetime-short','datetime-medium','datetime-long','datetime-full',
                                                         'decimal','currency','percent')),
      decimal_places              number,
      currency                    varchar2(128),
      numeric_pattern             varchar2(30),
      format_scaling              varchar2(12)
                                  constraint jet_axes_format_scaling
                                  check (format_scaling in ('auto','none','thousand','million','billion','trillion','quadrillion')),
      scaling                     varchar2(10)
                                  constraint jet_axes_scaling
                                  check (scaling in ('linear','log')),
      baseline_scaling            varchar2(5)
                                  constraint jet_axes_baseline_scale
                                  check (baseline_scaling in ('min','zero')),
      step                        number
                                  constraint jet_axes_step
                                  check (step >= 0),
      position                    varchar2(10)
                                  constraint jet_axes_position
                                  check (position in ('start','end','top','bottom','auto')),
      major_tick_rendered         varchar2(5)
                                  constraint jet_axes_major_tick
                                  check (major_tick_rendered in ('auto','on','off')),
      min_step                    number
                                  constraint jet_axes_min_step
                                  check (min_step >= 0),
      minor_tick_rendered         varchar2(5)
                                  constraint jet_axes_minor_tick
                                  check (minor_tick_rendered in ('auto','on','off')),
      minor_step                  number
                                  constraint jet_axes_minor_step
                                  check (minor_step >= 0),
      tick_label_rendered         varchar2(5)
                                  constraint jet_axes_tick_label
                                  check (tick_label_rendered in ('on','off')),
      tick_label_rotation         varchar2(5)
                                  constraint jet_axes_label_rotation
                                  check (tick_label_rotation in ('auto','none')),
      tick_label_position         varchar2(10)
                                  constraint jet_axes_label_position
                                  check (tick_label_position in ('inside','outside')),
      tick_label_font_family      varchar2(20)
                                  constraint jet_tick_font_family
                                  check (tick_label_font_family in ('Arial','Arial Black','Bookman','Comic Sans MS','Courier','Courier New','Garamond','Georgia','Helvetica','Impact','Palatino','Times','Times New Roman','Trebuchet MS','Verdana')),
      tick_label_font_style       varchar2(10)
                                  constraint jet_tick_font_style
                                  check (tick_label_font_style in ('normal','italic','oblique')),
      tick_label_font_size        varchar2(2)
                                  constraint jet_tick_font_size
                                  check (tick_label_font_size in ('8','10','12','14','16','18','20','22')),
      tick_label_font_color       varchar2(128),
      split_dual_y                varchar2(5)
                                  constraint jet_axes_split_dual_y
                                  check (split_dual_y in ('auto','on','off')),
      splitter_position           number,
      -- Gantt Chart Attributes
      axis_scale                  varchar2(10)
                                  constraint jet_axes_scale
                                  check (axis_scale in ('seconds','minutes','hours','days','weeks','months','quarters','years')),
      zoom_order_seconds          varchar2(1)
                                  constraint jet_zoom_order_secs
                                  check (zoom_order_seconds in ('Y','N')),
      zoom_order_minutes          varchar2(1)
                                  constraint jet_zoom_order_mins
                                  check (zoom_order_minutes in ('Y','N')),
      zoom_order_hours            varchar2(1)
                                  constraint jet_zoom_order_hrs
                                  check (zoom_order_hours in ('Y','N')),
      zoom_order_days             varchar2(1)
                                  constraint jet_zoom_order_days
                                  check (zoom_order_days in ('Y','N')),
      zoom_order_weeks            varchar2(1)
                                  constraint jet_zoom_order_wks
                                  check (zoom_order_weeks in ('Y','N')),
      zoom_order_months           varchar2(1)
                                  constraint jet_zoom_order_mths
                                  check (zoom_order_months in ('Y','N')),
      zoom_order_quarters         varchar2(1)
                                  constraint jet_zoom_order_qtrs
                                  check (zoom_order_quarters in ('Y','N')),
      zoom_order_years            varchar2(1)
                                  constraint jet_zoom_order_yrs
                                  check (zoom_order_years in ('Y','N')),
      created_on                  date,
      created_by                  varchar2(255),
      updated_on                  date,
      updated_by                  varchar2(255)
  )
/

create index wwv_flow_jet_ax_fid_fk on wwv_flow_jet_chart_axes (flow_id, page_id)
/

create index wwv_flow_jet_ax_fid_fk2 on wwv_flow_jet_chart_axes (chart_id)
/

--==============================================================================
--##############################################################################
--#
--# APPLICATION > PAGE > REGION (MAP)
--#
--##############################################################################
prompt APPLICATION > PAGE > REGION (MAP)

--==============================================================================
create table wwv_flow_map_regions (
    id                              number
                                    constraint mapregion_pk
                                    primary key,
    flow_id                         number not null
                                    constraint mapregion_flow_fk
                                    references wwv_flows
                                    on delete cascade,
    page_id                         number not null,
    region_id                       number not null
                                    constraint mapregion_region_fk
                                    references wwv_flow_page_plugs
                                    on delete cascade,
    security_group_id               number not null,
    --
    height                          number not null,
    tilelayer_type                  varchar2(7) not null
                                    constraint mapregion_tiletype_ck
                                    check ( tilelayer_type in ('DEFAULT','CUSTOM' ) ),
    tilelayer_name_default          varchar2(255),
    tilelayer_name_dark             varchar2(255),
    --
    navigation_bar_type             varchar2(5) not null
                                    constraint mapregion_navbar_type_ck
                                    check ( navigation_bar_type in ( 'NONE', 'SMALL', 'FULL' ) ),
    navigation_bar_position         varchar2(7)
                                    constraint mapregion_navbar_pos_ck
                                    check ( navigation_bar_position in ( 'START', 'END' ) ),
    --
    init_position_zoom_type         varchar2(13) not null
                                    constraint mapregion_initposzoom_type_ck
                                    check ( init_position_zoom_type in ( 'STATIC', 'SQL', 'QUERY_RESULTS' ) ),
    init_position_from_browser      varchar2(1),
                                    constraint mapregion_initpos_browser_ck
                                    check ( init_position_from_browser in ( 'Y', 'N' ) ),
    init_position_zoom_source       varchar2(4000),
    --
    init_position_geometry_type     varchar2(14)
                                    constraint mapregion_initgeomtype_ck
                                    check ( init_position_geometry_type in ( 'SDO_GEOMETRY', 'GEOJSON', 'LONLAT_COLUMNS' ) ),
    init_position_geometry_column   varchar2(128),
    init_position_lon_column        varchar2(128),
    init_position_lat_column        varchar2(128),
    init_zoomlevel_column           varchar2(255), -- can store item substitutions as well
    init_position_lon_static        varchar2(255),
    init_position_lat_static        varchar2(255),
    init_zoomlevel_static           varchar2(255), -- can store item substitutions as well
    --
    layer_messages_position         varchar2(8),
                                    constraint mapregion_msg_pos
                                    check ( layer_messages_position in ( 'ABOVE', 'BELOW', 'SELECTOR' ) ),
    layer_messages_selector         varchar2(255),
    --
    show_legend                     varchar2(1) not null
                                    constraint mapregion_showlegend_ck
                                    check (show_legend in ( 'Y', 'N' )),
    legend_title                    varchar2(4000),
    legend_position                 varchar2(8),
                                    constraint mapregion_legend_pos_ck
                                    check ( legend_position in ( 'START', 'END', 'SELECTOR' ) ),
    legend_position_selector        varchar2(255),
    legend_css_classes              varchar2(255),
    --
    map_status_item                 varchar2(255),
    --
    features                        varchar2(4000)
                                    constraint mapregion_features_ck
                                    check ( regexp_instr( features,
                                                          '^(:|MOUSEWHEEL_ZOOM|SCALE_BAR|RECTANGLE_ZOOM|INFINITE_MAP|OVERVIEW_MAP|CIRCLE_TOOL|DISTANCE_TOOL|BROWSER_LOCATION)+$') > 0 ),
    copyright_notice                varchar2(4000),
    custom_styles                   varchar2(4000),
    unit_system                     varchar2(8) not null
                                    constraint mapregion_unitsystem_ck
                                    check ( unit_system in ( 'METRIC', 'IMPERIAL', 'ITEM' ) ),
    unit_system_page_item_name      varchar2(255),
    -- Audit
    last_updated_by                 varchar2(255),
    last_updated_on                 date,
    --
    constraint map_region_page_uk
        unique (flow_id, page_id, region_id) deferrable initially deferred
        using index compress
    )
/

create unique index map_region_region_fx on wwv_flow_map_regions (region_id)
/
create index map_region_region_ix1 on wwv_flow_map_regions (security_group_id, flow_id, region_id) compress
/

--==============================================================================
create table wwv_flow_map_region_layers (
    id                             number
                                   constraint maplayer_pk
                                   primary key,
    map_region_id                  number not null
                                   constraint maplayer_map_fk
                                   references wwv_flow_map_regions
                                   on delete cascade,
    flow_id                        number not null
                                   constraint maplayer_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    page_id                        number not null,
    security_group_id              number not null,
    --
    -- Generic attributes
    --
    name                           varchar2(255) not null,
    label                          varchar2(4000),
    layer_type                     varchar2(10) not null
                                   constraint maplayer_type_ck
                                   check( layer_type in ( 'POINT', 'LINE', 'POLYGON', 'HEATMAP', 'POLYGON_3D' ) ),
    display_sequence               number not null,
    visible_min_zoom_level         number,
    visible_max_zoom_level         number,
    --
    -- data source attributes
    --
    location                       varchar2(13) not null
                                   constraint maplayer_location_ck
                                   check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE', 'REGION_SOURCE' )),
    --
    query_type                     varchar2(23)
                                   constraint maplayer_querytype_ck
                                   check (query_type in ('SQL','FUNC_BODY_RETURNING_SQL','TABLE')),
    -- query type = TABLE
    table_owner                    varchar2(128),
    table_name                     varchar2(128),
    where_clause                   varchar2(4000),
    order_by_clause                varchar2(255),
    include_rowid_column           varchar2(1)
                                   constraint maplayer_include_rowid_ck
                                   check (include_rowid_column in ('Y', 'N')),
    -- query type = SQL Query or query type = PLSQL returning SQL
    layer_source                   clob,
    function_body_language         varchar2(10)
                                   constraint maplayer_functype_ck
                                   check (function_body_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    optimizer_hint                 varchar2(255),
    -- REST Data Source
    web_src_module_id              number
                                   constraint maplayer_web_src_module_fk
                                   references wwv_flow_web_src_modules,
    use_local_sync_table           varchar2(1)
                                   constraint maplayer_wsm_usesynctab_ck
                                   check (use_local_sync_table in ('Y','N')),
    source_post_processing         varchar2(23)
                                   constraint maplayer_post_proc_ck
                                   check ( source_post_processing in ( 'WHERE_ORDER_BY_CLAUSE', 'SQL', 'FUNC_BODY_RETURNING_SQL' )),
    external_filter_expr           varchar2(4000),
    external_order_by_expr         varchar2(255),
    -- REST Enabled SQL
    remote_server_id               number
                                   constraint maplayer_remote_server_fk
                                   references wwv_remote_servers,
    remote_sql_caching             varchar2(9)
                                   constraint maplayer_remote_sql_caching_ck
                                   check (remote_sql_caching in ('ALL_USERS', 'USER', 'SESSION')),
    remote_sql_invalidate_when     varchar2(255),
    --
    -- generic data source attributes
    --
    items_to_submit                varchar2(4000),
    max_row_count                  number,
    no_data_found_message          varchar2(4000),
    no_data_found_icon_classes     varchar2(255),
    more_data_found_message        varchar2(4000),
    --
    has_spatial_index              varchar2(1)
                                   constraint maplayer_geometry_hasindex_ck
                                   check( has_spatial_index in ( 'Y', 'N' ) ),
    --
    row_assignment_column          varchar2(128),
    row_assignment_value           varchar2(4000),
    --
    -- Column Mapping
    --
    --
    pk_column                      varchar2(128),
    geometry_column_data_type      varchar2(14) not null
                                   constraint maplayer_geometry_type_ck
                                   check ( geometry_column_data_type in ( 'SDO_GEOMETRY', 'GEOJSON', 'LONLAT_COLUMNS' ) ),
    geometry_column                varchar2(128),
    longitude_column               varchar2(128),
    latitude_column                varchar2(128),
    --
    stroke_color                   varchar2(128),
    stroke_width                   number,
    stroke_style                   varchar2(128)
                                   constraint maplayer_strokestyle_ck
                                   check ( stroke_style in ( 'SOLID', 'DOTTED', 'DASHED', 'DOT_DASHED' ) ),
    fill_color                     varchar2(4000),
    fill_color_is_spectrum         varchar2(1),
                                   constraint maplayer_fillcolorspectrum_ck
                                   check ( fill_color_is_spectrum in ( 'Y', 'N' ) ),
    fill_color_spectrum_type       varchar2(255)
                                   constraint color_spectrum_type_ck
                                   check ( fill_color_spectrum_type in ( 'SEQUENTIAL', 'DIVERGING', 'QUALITATIVE' ) ),
    fill_color_spectrum_name       varchar2(255),
    fill_value_column              varchar2(128),
    fill_opacity                   number,
    extrude_value_column           varchar2(128),
    extrude_unit                   varchar2(4)
                                   constraint maplayer_extrude_unit_ck
                                   check ( extrude_unit in ( 'M', 'KM', 'MILE', 'FOOT' ) ),
    --
    point_display_type             varchar2(4)
                                   constraint maplayer_style_type_ck
                                   check ( point_display_type in ( 'ICON', 'SVG' ) ),
    point_icon_source_type         varchar2(14)
                                   constraint maplayer_pointicon_src_type_ck
                                   check (point_icon_source_type in ('STATIC_CLASS', 'DYNAMIC_CLASS', 'URL')),
    point_icon_class_column        varchar2(128),
    point_icon_image_url           varchar2(4000),
    point_icon_css_classes         varchar2(255),
    point_svg_shape                varchar2(255),  -- column name
    point_svg_shape_scale          varchar2(255),  -- column name
    --
    feature_clustering             varchar2(1)
                                   constraint maplayer_featurecluster_ck
                                   check ( feature_clustering in ( 'Y', 'N' ) ),
    cluster_threshold_pixels       number,
    cluster_point_image_path       varchar2(4000), -- substitutions
    cluster_point_css_classes      varchar2(255),  -- substitutions
    cluster_point_svg_shape        varchar2(255),  -- substitutions
    cluster_point_svg_shape_scale  varchar2(255),  -- substitutions
    cluster_point_svg_fill_color   varchar2(255),  -- substitutions
    cluster_tooltip                varchar2(4000), -- substitutions
    --
    tooltip_adv_formatting         varchar2(1),
                                   constraint maplayer_tooltip_adv_format_ck
                                   check ( tooltip_adv_formatting in ('Y', 'N' ) ),
    tooltip_column                 varchar2(128),
    tooltip_html_expr              varchar2(4000), -- substitution / templates
    tooltip_css_classes            varchar2(255), -- substitution / templates
    --
    info_window_adv_formatting     varchar2(1),
                                   constraint maplayer_infowin_adv_format_ck
                                   check ( info_window_adv_formatting in ('Y', 'N' ) ),
    info_window_title_column       varchar2(128),
    info_window_body_column        varchar2(128),
    info_window_html_expr          varchar2(4000), -- substitution / templates
    info_window_css_classes        varchar2(255), -- substitution / templates
    --
    link_target_type               varchar2(22)
                                   constraint maplayer_link_type
                                   check (link_target_type in ('REDIRECT_URL','REDIRECT_PAGE','REDIRECT_APP','DEFINED_BY_DA' )),
    link_target                    varchar2(4000), -- substitution
    --
    display_in_legend              varchar2(1) not null
                                   constraint maplayer_displayinlegend_ck
                                   check( display_in_legend in ( 'Y', 'N' ) ),
    legend_icon_css_classes        varchar2(255),
    allow_hide                     varchar2(1)
                                   constraint maplayer_allowhide_ck
                                   check( allow_hide in ( 'Y', 'N' ) ),
    -- serverside condition
    condition_type                 varchar2(255),
    condition_expr1                varchar2(4000),
    condition_expr2                varchar2(4000),
    exec_cond_for_each_row         varchar2(1)
                                   constraint maplayer_cond_each_row_ck
                                   check (exec_cond_for_each_row in ('Y', 'N')),
    -- security
    authorization_scheme           varchar2(255),
    -- configuration
    build_option_id                number,
    layer_comments                 varchar2(4000),
    -- Audit
    last_updated_by                varchar2(255),
    last_updated_on                date,
    --
    -- Constraints
    --
    constraint maplayer_has_geom_col
    check ( geometry_column is not null or ( longitude_column is not null and latitude_column is not null ) ) )
/

create index map_layers_map_fx on wwv_flow_map_region_layers (map_region_id)
/
create index map_layers_page_fx on wwv_flow_map_region_layers (flow_id, page_id) compress
/
create index map_layers_web_src_module_fx on wwv_flow_map_region_layers (web_src_module_id) compress
/
create index map_layers_remote_server_fx on wwv_flow_map_region_layers (remote_server_id) compress
/
create index map_layers_ix1 on wwv_flow_map_region_layers (build_option_id) compress
/
create index map_layers_ix2 on wwv_flow_map_region_layers (security_group_id) compress
/
--==============================================================================
--##############################################################################
--#
--# APPLICATION > PAGE > REGION (CARD)
--#
--##############################################################################
prompt APPLICATION > PAGE > REGION (CARD)

--==============================================================================
create table wwv_flow_cards (
    id                              number
                                    constraint cards_pk
                                    primary key,
    flow_id                         number not null
                                    constraint cards_flow_fk
                                    references wwv_flows
                                    on delete cascade,
    page_id                         number not null,
    region_id                       number not null
                                    constraint cards_region_fk
                                    references wwv_flow_page_plugs
                                    on delete cascade,
    security_group_id               number not null,
    -- appearance
    layout_type                     varchar2(20) not null
                                    constraint cards_layout_type_ck
                                    check (layout_type in ('GRID', 'FLOAT', 'ROW')),
    grid_column_count               number,
    component_css_classes           varchar2(255),
    -- card
    card_appearance                 varchar2(8)
                                    constraint cards_card_appearance_ck
                                    check (card_appearance in ('BLOCK', 'FEATURED', 'COMPACT')),
    card_css_classes                varchar2(255),
    -- title
    title_adv_formatting            varchar2(1) not null
                                    constraint cards_title_adv_format_ck
                                    check (title_adv_formatting in ('Y', 'N')),
    title_column_name               varchar2(128),
    title_html_expr                 varchar2(4000),
    title_css_classes               varchar2(255),
    sub_title_adv_formatting        varchar2(1) not null
                                    constraint cards_sub_title_adv_format_ck
                                    check (sub_title_adv_formatting in ('Y', 'N')),
    sub_title_column_name           varchar2(128),
    sub_title_html_expr             varchar2(4000),
    sub_title_css_classes           varchar2(255),
    -- Body
    body_adv_formatting             varchar2(1) not null
                                    constraint cards_body_adv_format_ck
                                    check (body_adv_formatting in ('Y', 'N')),
    body_column_name                varchar2(128),
    body_html_expr                  varchar2(4000),
    body_css_classes                varchar2(255),
    second_body_adv_formatting      varchar2(1) not null
                                    constraint cards_sec_body_adv_format_ck
                                    check (second_body_adv_formatting in ('Y', 'N')),
    second_body_column_name         varchar2(128),
    second_body_html_expr           varchar2(4000),
    second_body_css_classes         varchar2(255),
    -- icon and badge
    icon_source_type                varchar2(14)
                                    constraint cards_icon_src_type_ck
                                    check (icon_source_type in ('INITIALS', 'STATIC_CLASS', 'DYNAMIC_CLASS', 'BLOB', 'URL')),
    icon_class_column_name          varchar2(128),
    icon_blob_column_name           varchar2(128),
    icon_image_url                  varchar2(4000),
    icon_css_classes                varchar2(255),
    icon_position                   varchar2(5)
                                    constraint cards_icon_pos_ck
                                    check(icon_position in ('TOP', 'START', 'END')),
    icon_description                varchar2(4000),
    badge_column_name               varchar2(128),
    badge_label                     varchar2(255),
    badge_css_classes               varchar2(255),
    -- Media
    media_adv_formatting            varchar2(1) not null
                                    constraint cards_media_adv_format_ck
                                    check (media_adv_formatting in ('Y', 'N')),
    media_html_expr                 varchar2(4000),
    media_source_type               varchar2(11)
                                    constraint cards_media_src_type_ck
                                    check (media_source_type in ('STATIC_URL','DYNAMIC_URL', 'BLOB')),
    media_url                       varchar(4000),
    media_url_column_name           varchar2(128),
    media_blob_column_name          varchar2(128),
    media_display_position          varchar2(10)
                                    constraint cards_media_display_pos_ck
                                    check (media_display_position in ('FIRST', 'BODY', 'BACKGROUND')),
    media_appearance                varchar2(10)
                                    constraint cards_media_appearance_ck
                                    check (media_appearance in ('SQUARE', 'WIDESCREEN')),
    media_sizing                    varchar2(5)
                                    constraint cards_media_sizing_ck
                                    check( media_sizing in ('FIT', 'COVER')),
    media_css_classes               varchar2(255),
    media_description               varchar2(4000),
    -- pk columns for BLOB support
    uses_blob_column                varchar2(1) generated always as (
                                        case when icon_source_type = 'BLOB' or media_source_type = 'BLOB' then 'Y' else 'N' end ) virtual,
    pk1_column_name                 varchar2(128),
    pk2_column_name                 varchar2(128),
    mime_type_column_name           varchar2(128),
    last_updated_column_name        varchar2(128),
    -- Audit
    last_updated_by                 varchar2(255),
    last_updated_on                 date,
    --
    constraint card_page_uk
        unique (flow_id, page_id, region_id) deferrable initially deferred
        using index compress
    )
/

create index card_region_idx on wwv_flow_cards (region_id) compress
/

--==============================================================================
create table wwv_flow_card_actions (
    id                          number
                                constraint card_actions_pk
                                primary key,
    card_id                     number not null
                                constraint card_actions_card_fk
                                references wwv_flow_cards
                                on delete cascade,
    flow_id                     number not null
                                constraint card_actions_flow_fk
                                references wwv_flows
                                on delete cascade,
    page_id                     number not null,
    security_group_id           number not null,
    --
    action_type                 varchar2(9) not null
                                constraint card_actions_type_ck
                                check ( action_type in (
                                    'BUTTON', 'FULL_CARD', 'TITLE', 'SUB_TITLE', 'MEDIA')),
    position                    varchar2(10)
                                constraint card_actions_pos_ck
                                check ( position in ('PRIMARY', 'SECONDARY')),
    display_sequence            number,
    label                       varchar2(255),
    link_target_type            varchar2(13)
                                constraint card_actions_link_type_ck
                                check (link_target_type in ('REDIRECT_URL','REDIRECT_PAGE','REDIRECT_APP','DEFINED_BY_DA' )),
    link_target                 varchar2(4000),
    link_attributes             varchar2(255),
    button_display_type         varchar2(15)
                                constraint card_actions_btn_disp_type_ck
                                check (button_display_type in ('TEXT','ICON','TEXT_WITH_ICON')),
    icon_css_classes            varchar2(255),
    action_css_classes          varchar2(255),
    is_hot                      varchar2(1)
                                constraint card_actions_is_hot
                                check (is_hot in ('Y', 'N')),
    -- serverside condition
    condition_type              varchar2(255),
    condition_expr1             varchar2(4000),
    condition_expr2             varchar2(4000),
    exec_cond_for_each_row      varchar2(1)
                                constraint card_actions_cond_each_row_ck
                                check (exec_cond_for_each_row in ('Y', 'N')),
    -- security
    authorization_scheme        varchar2(255),
    -- configuration
    build_option_id             number,
    action_comments             varchar2(4000),
    -- Audit
    last_updated_by             varchar2(255),
    last_updated_on             date  )
/

create index card_actions_card_fx on wwv_flow_card_actions (card_id)
/
create index card_actions_page_fx on wwv_flow_card_actions (flow_id, page_id) compress
/

--##############################################################################
--#
--# APPLICATION > PAGE > CACHING
--#
--##############################################################################
prompt APPLICATION > PAGE > CACHING

--==============================================================================
create table wwv_flow_page_cache (
    id                   number
                         constraint wwv_flow_page_cache_pk
                         primary key,
    flow_id              number not null, -- app
    page_id              number not null, -- page
    region_id            number           -- support caching regions as well
                         constraint wwv_flow_page_cache_fk_region
                         references wwv_flow_page_plugs
                         on delete cascade,
    security_group_id    number not null, -- workspace
    language             varchar2(255),   -- all cached pages are by language
    user_name            varchar2(255),   -- the user at the time of caching
    owning_session_id    number           -- reference to the owning session if caching is session based
                         constraint wwv_flow_page_cache_fk_session
                         references wwv_flow_sessions$
                         on delete cascade,
    session_id           number,          -- the session at the time of caching
    builder_session_id   number,          -- the builder session at the time of caching
    item_hash            varchar2(255),   -- hash for item values that the cached component depends on
    page_text            clob,            -- HTML cache of page
    valid_until          date   not null, -- cached value is valid until this time
    cached_on            date   not null, -- create date
    constraint wwv_flow_page_cache_fk_page
        foreign key(flow_id, page_id)
        references wwv_flow_steps(flow_id, id)
        on delete cascade
    )
    pctfree 0
    lob (page_text) store as (
        pctversion 0
        cache
        disable storage in row )
/
create index wwv_flow_page_cache_idx1 on wwv_flow_page_cache (flow_id, page_id, user_name) compress
/
create index wwv_flow_page_cache_idx2 on wwv_flow_page_cache (owning_session_id, flow_id, page_id) compress
/
create index wwv_flow_page_cache_idx3 on wwv_flow_page_cache (region_id)
/

create table wwv_flow_page_code_cache (
    id            number
                  constraint wwv_flow_page_code_cache_pk
                  primary key,
    page_cache_id number
                  constraint wwv_flow_page_code_cache_fk
                  references wwv_flow_page_cache
                  on delete cascade,
    code_type     varchar2(10),
    type          varchar2(20),
    key           varchar2(4000),
    code          clob
    )
    pctfree 0
    lob (code) store as (
        pctversion 0
        cache
        enable storage in row)
/

create index wwv_flow_page_code_cache_fkidx on wwv_flow_page_code_cache (page_cache_id)
/

--##############################################################################
--#
--# APPLICATION > DEVELOPER LOCKING
--#
--##############################################################################
prompt APPLICATION > DEVELOPER LOCKING

--==============================================================================
create table wwv_flow_lock_page (
  id                  number
                      constraint wwv_flow_lock_page_pk
                      primary key,
  flow_id             number
                      references wwv_flows (id)
                      on delete cascade,
  object_id           number,
  locked_by           varchar2(255),
  locked_on           date,
  lock_comment        varchar2(4000),
  security_group_id   number not null
  )
/

create index wwv_flow_lock_page_fk on wwv_flow_lock_page (flow_id)
/

create index wwv_flow_lock_page_by on wwv_flow_lock_page (locked_by)
/

create unique index wwv_flow_lock_page_pg on wwv_flow_lock_page (object_id,flow_id)
/

--==============================================================================
create table wwv_flow_lock_page_log (
    id             number
                   constraint wwv_flow_lock_pg_log_pk
                   primary key,
    lock_id        number not null,
    lock_flow      number,
    lock_page      number,
    action         varchar2(30) not null
                   constraint dev_file_lock_log_action
                   check ( action in ('LOCK','UNLOCK','UPDATE')),
    action_date    date,
    developer      varchar2(255),
    lock_comment   varchar2(4000)
    )
/


create index wwv_flow_lock_page_log_idx1 on wwv_flow_lock_page_log (lock_flow)
/

create index wwv_flow_lock_page_log_idx2 on wwv_flow_lock_page_log (lock_id)
/

--##############################################################################
--#
--# APPLICATION > SUPPORTING OBJECTS
--#
--##############################################################################
prompt APPLICATION > SUPPORTING OBJECTS

--==============================================================================
create table wwv_flow_install (
     id                        number
                               constraint wwv_flow_install_pk
                               primary key,
     flow_id                   number  not null
                               constraint wwv_flow_install_fk
                               references wwv_flows
                               on delete cascade,
     security_group_id         number not null
                               constraint wwv_flow_install_fk2
                               references wwv_flow_companies(provisioning_company_id),
     --
     include_in_export_yn      varchar2(1)
                               constraint wwv_install_in_export
                               check (include_in_export_yn in ('Y','N','I')),
     --
     welcome_message           varchar2(4000),
     license_message           varchar2(4000),
     configuration_message     varchar2(4000),
     build_options_message     varchar2(4000),
     validation_message        varchar2(4000),
     install_message           varchar2(4000),
     install_success_message   varchar2(4000),
     install_failure_message   varchar2(4000),
     --
     get_version_sql_query     varchar2(4000),
     upgrade_message           varchar2(4000),
     upgrade_confirm_message   varchar2(4000),
     upgrade_success_message   varchar2(4000),
     upgrade_failure_message   varchar2(4000),
     --
     deinstall_message         varchar2(4000),
     deinstall_success_message varchar2(4000),
     deinstall_failure_message varchar2(4000),
     deinstall_script          clob,
     --
     prompt_substitution_01_yn varchar2(1)
                               constraint wwv_install_sub_01
                               check (prompt_substitution_01_yn in ('Y','N')),
     prompt_substitution_02_yn varchar2(1)
                               constraint wwv_install_sub_02
                               check (prompt_substitution_02_yn in ('Y','N')),
     prompt_substitution_03_yn varchar2(1)
                               constraint wwv_install_sub_03
                               check (prompt_substitution_03_yn in ('Y','N')),
     prompt_substitution_04_yn varchar2(1)
                               constraint wwv_install_sub_04
                               check (prompt_substitution_04_yn in ('Y','N')),
     prompt_substitution_05_yn varchar2(1)
                               constraint wwv_install_sub_05
                               check (prompt_substitution_05_yn in ('Y','N')),
     prompt_substitution_06_yn varchar2(1)
                               constraint wwv_install_sub_06
                               check (prompt_substitution_06_yn in ('Y','N')),
     prompt_substitution_07_yn varchar2(1)
                               constraint wwv_install_sub_07
                               check (prompt_substitution_07_yn in ('Y','N')),
     prompt_substitution_08_yn varchar2(1)
                               constraint wwv_install_sub_08
                               check (prompt_substitution_08_yn in ('Y','N')),
     prompt_substitution_09_yn varchar2(1)
                               constraint wwv_install_sub_09
                               check (prompt_substitution_09_yn in ('Y','N')),
     prompt_substitution_10_yn varchar2(1)
                               constraint wwv_install_sub_10
                               check (prompt_substitution_10_yn in ('Y','N')),
     prompt_substitution_11_yn varchar2(1)
                               constraint wwv_install_sub_11
                               check (prompt_substitution_11_yn in ('Y','N')),
     prompt_substitution_12_yn varchar2(1)
                               constraint wwv_install_sub_12
                               check (prompt_substitution_12_yn in ('Y','N')),
     prompt_substitution_13_yn varchar2(1)
                               constraint wwv_install_sub_13
                               check (prompt_substitution_13_yn in ('Y','N')),
     prompt_substitution_14_yn varchar2(1)
                               constraint wwv_install_sub_14
                               check (prompt_substitution_14_yn in ('Y','N')),
     prompt_substitution_15_yn varchar2(1)
                               constraint wwv_install_sub_15
                               check (prompt_substitution_15_yn in ('Y','N')),
     prompt_substitution_16_yn varchar2(1)
                               constraint wwv_install_sub_16
                               check (prompt_substitution_16_yn in ('Y','N')),
     prompt_substitution_17_yn varchar2(1)
                               constraint wwv_install_sub_17
                               check (prompt_substitution_17_yn in ('Y','N')),
     prompt_substitution_18_yn varchar2(1)
                               constraint wwv_install_sub_18
                               check (prompt_substitution_18_yn in ('Y','N')),
     prompt_substitution_19_yn varchar2(1)
                               constraint wwv_install_sub_19
                               check (prompt_substitution_19_yn in ('Y','N')),
     prompt_substitution_20_yn varchar2(1)
                               constraint wwv_install_sub_20
                               check (prompt_substitution_20_yn in ('Y','N')),
     install_prompt_01         varchar2(255),
     install_prompt_02         varchar2(255),
     install_prompt_03         varchar2(255),
     install_prompt_04         varchar2(255),
     install_prompt_05         varchar2(255),
     install_prompt_06         varchar2(255),
     install_prompt_07         varchar2(255),
     install_prompt_08         varchar2(255),
     install_prompt_09         varchar2(255),
     install_prompt_10         varchar2(255),
     install_prompt_11         varchar2(255),
     install_prompt_12         varchar2(255),
     install_prompt_13         varchar2(255),
     install_prompt_14         varchar2(255),
     install_prompt_15         varchar2(255),
     install_prompt_16         varchar2(255),
     install_prompt_17         varchar2(255),
     install_prompt_18         varchar2(255),
     install_prompt_19         varchar2(255),
     install_prompt_20         varchar2(255),
     --
     prompt_if_mult_auth_yn    varchar2(1)
                               constraint wwv_auth_prompt
                               check (prompt_if_mult_auth_yn in ('Y','N')),
     --
     trigger_install_when_cond varchar2(255),
     trigger_install_when_exp1 varchar2(4000),
     trigger_install_when_exp2 varchar2(4000),
     trigger_failure_message   varchar2(4000),
     required_free_kb          number,
     required_sys_privs        varchar2(4000),
     required_names_available  varchar2(4000),
     --
     last_updated_by           varchar2(255),
     last_updated_on           date,
     created_by                varchar2(255),
     created_on                date
)
/

create unique index wwv_flow_install_idx1 on wwv_flow_install (flow_id)
/
create index wwv_flow_install_idx2 on wwv_flow_install (security_group_id)
/

--==============================================================================
create table wwv_flow_install_scripts (
    id                    number
                          constraint wwv_flow_install_scripts_pk
                          primary key,
    flow_id               number not null
                          constraint wwv_flow_install_scripts_fk
                          references wwv_flows,
    security_group_id     number not null
                          constraint wwv_flow_install_scripts_fk2
                          references wwv_flow_companies(provisioning_company_id),
    install_id            number  not null
                          constraint wwv_flow_install_scripts_fk3
                          references wwv_flow_install
                          on delete cascade,
    name                  varchar2(255) not null,
    sequence              number not null,
    script                clob not null,
    script_type           varchar2(30)
                          constraint wwv_flow_install_st_ck
                          check (script_type in ('INSTALL','UPGRADE','DEINSTALL')),
    script_option         varchar2(30)
                          constraint wwv_flow_install_so_ck
                          check (script_option in ('PACKAGE_SPEC','PACKAGE_BODY','TABLE')),
    ---
    condition_type        varchar2(255),
    condition             varchar2(4000),
    condition2            varchar2(4000),
    ---
    last_updated_by       varchar2(255),
    last_updated_on       date,
    created_by            varchar2(255),
    created_on            date
)
/
create unique index wwv_flow_install_scripts_idx1 on wwv_flow_install_scripts (flow_id, name, script_type)
/
create index wwv_flow_install_scripts_idx2 on wwv_flow_install_scripts (security_group_id)
/
create index wwv_flow_install_scripts_idx3 on wwv_flow_install_scripts (install_id)
/

--==============================================================================
create table wwv_flow_install_objects (
    id                    number
                          constraint wwv_flow_install_objects_pk
                          primary key,
    flow_id               number not null
                          constraint wwv_flow_install_objects_fk
                          references wwv_flows,
    security_group_id     number not null
                          constraint wwv_flow_install_objects_fk2
                          references wwv_flow_companies(provisioning_company_id),
    script_id             number  not null
                          constraint wwv_flow_install_objects_fk3
                          references wwv_flow_install_scripts
                          on delete cascade,
    object_owner          varchar2(128),
    object_type           varchar2(30),
    object_name           varchar2(128),
    ---
    last_updated_by       varchar2(255),
    last_updated_on       date,
    created_by            varchar2(255),
    created_on            date
)
/

create index wwv_flow_install_objects_idx1 on wwv_flow_install_objects (flow_id)
/
create index wwv_flow_install_objects_idx2 on wwv_flow_install_objects (security_group_id)
/
create index wwv_flow_install_objects_idx3 on wwv_flow_install_objects (script_id)
/

--==============================================================================
create table wwv_flow_install_checks (
    id                    number
                          constraint wwv_flow_install_checks_pk
                          primary key,
    flow_id               number not null
                          constraint wwv_flow_install_checks_fk
                          references wwv_flows,
    security_group_id     number not null
                          constraint wwv_flow_install_checks_fk2
                          references wwv_flow_companies(provisioning_company_id),
    install_id            number  not null
                          constraint wwv_flow_install_checks_fk3
                          references wwv_flow_install
                          on delete cascade,
    name                  varchar2(255) not null,
    sequence              number not null,
    check_type            varchar2(255),
    check_condition       varchar2(4000),
    check_condition2      varchar2(4000),
    failure_message       varchar2(4000),
    ---
    condition_type        varchar2(255),
    condition             varchar2(4000),
    condition2            varchar2(4000),
    ---
    last_updated_by       varchar2(255),
    last_updated_on       date,
    created_by            varchar2(255),
    created_on            date
)
/

create unique index wwv_flow_install_checks_idx1 on wwv_flow_install_checks (flow_id, name)
/
create index wwv_flow_install_checks_idx2 on wwv_flow_install_checks (security_group_id)
/
create index wwv_flow_install_checks_idx3 on wwv_flow_install_checks (install_id)
/

--==============================================================================
create table wwv_flow_install_build_opt (
    id                    number
                          constraint wwv_flow_install_build_opt_pk
                          primary key,
    flow_id               number not null
                          constraint wwv_flow_install_build_opt_fk
                          references wwv_flows,
    security_group_id     number not null
                          constraint wwv_flow_install_build_opt_fk2
                          references wwv_flow_companies(provisioning_company_id),
    install_id            number  not null
                          constraint wwv_flow_install_build_opt_fk3
                          references wwv_flow_install
                          on delete cascade,
    build_opt_id          number  not null
                          constraint wwv_flow_install_build_opt_fk4
                          references wwv_flow_patches
                          on delete cascade
)
/

create index wwv_flow_install_build_idx1 on wwv_flow_install_build_opt (flow_id)
/
create index wwv_flow_install_build_idx2 on wwv_flow_install_build_opt (security_group_id)
/
create index wwv_flow_install_build_idx3 on wwv_flow_install_build_opt (install_id)
/
create index wwv_flow_install_build_idx4 on wwv_flow_install_build_opt (build_opt_id)
/

--==============================================================================
-- 5.0 feature #1449: supporting objects script lock
--==============================================================================
create table wwv_flow_lock_script (
  id                  number
                      constraint wwv_flow_lock_script_pk
                      primary key,
  flow_id             number not null
                      constraint wwv_flow_lock_script_fk1
                      references wwv_flows,
  security_group_id   number not null
                      constraint wwv_flow_lock_script_fk2
                      references wwv_flow_companies(provisioning_company_id),
  script_id           number  not null
                      constraint wwv_flow_lock_script_fk3
                      references wwv_flow_install_scripts
                      on delete cascade,
  locked_by           varchar2(255),
  locked_on           date,
  lock_comment        varchar2(4000)
  )
/

create index wwv_flow_lock_script_idx1 on wwv_flow_lock_script (flow_id) compress
/
create index wwv_flow_lock_script_idx2 on wwv_flow_lock_script (security_group_id) compress
/
create unique index wwv_flow_lock_script_idx4 on wwv_flow_lock_script (script_id, flow_id) compress
/
create index wwv_flow_lock_script_idx5 on wwv_flow_lock_script (locked_by) compress
/


create table wwv_flow_lock_script_log (
    id                number
                      constraint wwv_flow_lock_script_log_pk
                      primary key,
    flow_id           number not null,
    security_group_id number not null,
    lock_id           number not null,
    script_id         number not null,
    action            varchar2(30) not null
                      constraint wwv_flow_lcksr_log_action_ck
                      check ( action in ('LOCK','UNLOCK','UPDATE','STEAL_LOCK')),
    action_date       date,
    locked_by         varchar2(255),
    lock_comment      varchar2(4000)
    )
/

create index wwv_flow_lock_script_log_idx1 on wwv_flow_lock_script_log (flow_id)
/
create index wwv_flow_lock_script_log_idx2 on wwv_flow_lock_script_log (security_group_id)
/
create index wwv_flow_lock_script_log_idx3 on wwv_flow_lock_script_log (lock_id)
/
create index wwv_flow_lock_script_log_idx4 on wwv_flow_lock_script_log (script_id)
/


--##############################################################################
--#
--# APPLICATION > APPLICATION MODELS
--#
--##############################################################################
prompt APPLICATION > APPLICATION MODELS

--==============================================================================
create table  wwv_flow_models  (
   id                number         not null
                        constraint wwv_flow_models_pk primary key,
   security_group_id number not null
                        constraint wwv_flow_models_fk
                            references wwv_flow_companies(provisioning_company_id)
                            on delete cascade,
   session_id        number not null,
   flow_id           number,
   owner             varchar2(4000),
   name              varchar2(4000),
   model_complete    varchar2(1)
                         constraint wwv_flow_model_complete
                              check (model_complete in (
                                 'Y',
                                 'N')),
   created_by        varchar2(255)  not null,
   created_on        date           not null,
   last_updated_by   varchar2(255),
   last_updated_on   date
)
/

create index wwv_flow_models_sec_group_idx on wwv_flow_models (security_group_id)
/

--==============================================================================
create table  wwv_flow_model_pages (
   id                       number        not null
                                constraint wwv_flow_model_pages_pk primary key,
   model_id                 number        not null
                                constraint wwv_flow_model_pages_fk
                                references wwv_flow_models (id)
                                    on delete cascade,
   page_id                  number,
   parent_page_id           number
                                constraint wwv_flow_model_pages_fk2
                                references wwv_flow_model_pages (id)
                                    on delete cascade,
   display_sequence         number,
   page_type                varchar2(255)
                                constraint wwv_flow_mp_page_type
                                check (page_type in (
                                    'BLANK',
                                    'REPORT',
                                    'LISTVIEW',
                                    'FORM',
                                    'TABULARFORM',
                                    'REPORTANDFORM',
                                    'LISTVIEWANDFORM',
                                    'COLUMNTOGGLE',
                                    'REFLOWTABLE',
                                    'CHART',
                                    'MASTERDETAIL')),
   page_source              varchar2(255)
                                constraint wwv_flow_mp_page_source
                                check (page_source in (
                                    'TABLE',
                                    'QUERY',
                                    'SPREADSHEET')),
   page_name                varchar2(255),
   page_mode                varchar2(20),
   source                   clob,
   block_id                 number,
   report_id                number,
   mig_comments             varchar2(4000),
   security_group_id        number         not null,
   created_by               varchar2(255)  not null,
   created_on               date           not null,
   last_updated_by          varchar2(255),
   last_updated_on          date
)
/

create index wwv_flow_model_pages_model_idx on wwv_flow_model_pages (model_id)
/

create index wwv_flow_mp_parent_page_idx on wwv_flow_model_pages (parent_page_id)
/

create index wwv_flow_mp_block_id_idx on wwv_flow_model_pages (block_id)
/

create index wwv_flow_mp_rep_id_idx on wwv_flow_model_pages (report_id)
/

--==============================================================================
create table wwv_flow_model_page_regions (
   id                       number not null
                                constraint wwv_flow_mpr_pk primary key,
   model_page_id            number not null
                                constraint wwv_flow_mpr_fk
                                references wwv_flow_model_pages (id)
                                on delete cascade,
   region_id                number,
   region_name              varchar2(4000),
   region_source            varchar2(255),
   source                   clob,
   display_sequence         number,
   report_implementation    varchar(30)
                                constraint wwv_flow_mpr_rpt_impl_type
                                check (report_implementation in ( 'CLASSIC','INTERACTIVE','IG')),
   search_enabled           varchar2(1)
                                constraint wwv_flow_mpr_search_enables
                                check (search_enabled in ('Y','N')),
   link_column              varchar2(255),
   link_text                varchar2(255),
   parent_link_from_column1 varchar2(255),
   parent_link_from_column2 varchar2(255),
   link_to_column1          varchar2(255),
   link_to_column2          varchar2(255),
   display_column           varchar2(255),
   supp_info_column         varchar2(255),
   report_filter            varchar2(255),
   column_heading_sorting   varchar2(255),
   download_link            varchar2(255),
   chart_title              varchar2(255),
   chart_type               varchar2(30),
   x_axis_title             varchar2(255),
   y_axis_title             varchar2(255),
   chart_orientation        varchar2(10),
   chart_value_column       varchar2(128),
   chart_label_column       varchar2(128),
   chart_low_column         varchar2(128),
   chart_high_column        varchar2(128),
   chart_open_column        varchar2(128),
   chart_close_column       varchar2(128),
   chart_volume_column      varchar2(128),
   chart_x_column           varchar2(128),
   chart_y_column           varchar2(128),
   chart_z_column           varchar2(128),
   chart_target_column      varchar2(128),
   chart_min_column         varchar2(128),
   chart_max_column         varchar2(128),
   security_group_id        number         not null,
   created_by               varchar2(255)  not null,
   created_on               date           not null,
   last_updated_by          varchar2(255),
   last_updated_on          date
)
/

create index wwv_flow_mpr_model_idx on wwv_flow_model_page_regions (model_page_id)
/

--==============================================================================
create table  wwv_flow_model_page_cols  (
   id                   number         not null
                           constraint wwv_flow_model_page_cols_pk primary key,
   model_region_id        number         not null
                           constraint wwv_flow_model_page_cols_fk
                           references wwv_flow_model_page_regions (id)
                             on delete cascade,
   column_name          varchar2(4000),
   column_display_name  varchar2(4000),
   column_sequence      number,
   help_text            varchar2(4000),
   display_as_form      varchar2(50)
                            constraint wwv_mpc_display_as_form
                            check (display_as_form in (
                                'DISPLAY_ONLY_PLSQL',
                                'FILE',
                                'BUTTON',
                                'HIDDEN',
                                'HIDDEN_PROTECTED',
                                'DISPLAY_ONLY_HTML',
                                'STOP_AND_START_HTML_TABLE',
                                'DISPLAY_ONLY_ESCAPE_SC',
                                'IMAGE',
                                'DISPLAY_AND_SAVE',
                                'DISPLAY_AND_SAVE_LOV',
                                'CHECKBOX',
                                'POPUP',
                                'POPUP_FILTER',
                                'POPUP2',
                                'POPUP3',
                                'POPUP4',
                                'POPUP5',
                                'POPUP6',
                                'POPUP_KEY_LOV',
                                'POPUP_KEY_LOV_NOFETCH',
                                'PICK_DATE_USING_FLOW_FORMAT_MASK',
                                'PICK_DATE_USING_APP_FORMAT_MASK',
                                'PICK_DATE_USING_APP_DATE_FORMAT',
                                'PICK_DATE_USING_FORMAT_MASK',
                                'PICK_DATE_DD_MON_RR',
                                'PICK_DATE_DD_MON_RR_HH_MI',
                                'PICK_DATE_DD_MON_RR_HH24_MI',
                                'PICK_DATE_DD_MON_YY',
                                'PICK_DATE_DD_MON_YY_HH_MI',
                                'PICK_DATE_DD_MON_YY_HH24_MI',
                                'PICK_DATE_DD_MON_YYYY',
                                'PICK_DATE_DD_MON_YYYY_HH_MI',
                                'PICK_DATE_DD_MON_YYYY_HH24_MI',
                                'PICK_DATE_DD_MM_YYYY',
                                'PICK_DATE_DD_MM_YYYY_HH_MI',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI',
                                'PICK_DATE_MM_DD_YYYY',
                                'PICK_DATE_MM_DD_YYYY_HH_MI',
                                'PICK_DATE_MM_DD_YYYY_HH24_MI',
                                'PICK_DATE_YYYY_MM_DD',
                                'PICK_DATE_YYYY_MM_DD_HH_MI',
                                'PICK_DATE_YYYY_MM_DD_HH24_MI',
                                'PICK_DATE_RR_MON_DD',
                                'PICK_DATE_RR_MON_DD_HH_MI',
                                'PICK_DATE_RR_MON_DD_HH24_MI',
                                'PICK_DATE_YYYY_MM_DD_SLASH',
                                'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH',
                                'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH',
                                'PICK_DATE_YYYY_DD_MM_DOT',
                                'PICK_DATE_YYYY_DD_MM_HH_MI_DOT',
                                'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT',
                                'PICK_DATE_DD_MM_YYYY_DASH',
                                'PICK_DATE_DD_MM_YYYY_HH_MI_DASH',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH',
                                'PICK_DATE_DD_MM_YYYY_DOT',
                                'PICK_DATE_DD_MM_YYYY_HH_MI_DOT',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT',
                                'DISPLAY_ONLY_LOV',
                                'COMBOBOX',
                                'SHOW_AS_SL_WITH_POPUP',
                                'COMBOBOX_WITH_URL_REDIRECT',
                                'COMBOBOX_WITH_BRANCH_TO_PAGE',
                                'COMBOBOX_WITH_SUBMIT',
                                'COMBOBOX_WITH_JS_POST',
                                'COMBOBOX_WITH_JS_REDIRECT',
                                'MULTIPLESELECT',
                                'RADIOGROUP',
                                'RADIOGROUP2',
                                'RADIOGROUP_WITH_REDIRECT',
                                'RADIOGROUP_WITH_SUBMIT',
                                'RADIOGROUP_WITH_JS_SUBMIT',
                                'TEXTAREA',
                                'TEXTAREA_WITH_SPELL_CHECK',
                                'TEXTAREA-AUTO-HEIGHT',
                                'TEXTAREA_WITH_CONTROLS',
                                'TEXTAREA_WITH_HTML_EDITOR_BASIC',
                                'TEXTAREA_WITH_HTML_EDITOR_STANDARD',
                                'TEXTAREA_CHAR_COUNT',
                                'TEXTAREA_CHAR_COUNT_SPELL',
                                'TEXT',
                                'TEXT_WITH_ENTER_SUBMIT',
                                'TEXT_DISABLED',
                                'TEXT_DISABLED_AND_SAVE',
                                'TEXT_WITH_CALCULATOR',
                                'PASSWORD',
                                'PASSWORD_WITH_ENTER_SUBMIT',
                                'PASSWORD_DNSS',
                                'PASSWORD_WITH_SUBMIT_DNSS',
                                'LIST_MGR',
                                'LIST_MGR2',
                                'LIST_MGR3',
                                'LIST_MGR_VIEW',
                                'LIST_MGR_VIEW2',
                                -- 4.0
                                'SELECT_LIST_WITH_POPUP',
                                'TEXT_WITH_SELECT_LIST',
                                'TEXTAREA_WITH_POPUP'
                                ) or
                                display_as_form like 'NATIVE\_%' escape '\'),
   --
   form_attribute_01   varchar2(4000),
   form_attribute_02   varchar2(4000),
   form_attribute_03   varchar2(4000),
   form_attribute_04   varchar2(4000),
   form_attribute_05   varchar2(4000),
   form_attribute_06   varchar2(4000),
   form_attribute_07   varchar2(4000),
   form_attribute_08   varchar2(4000),
   form_attribute_09   varchar2(4000),
   form_attribute_10   varchar2(4000),
   form_attribute_11   varchar2(4000),
   form_attribute_12   varchar2(4000),
   form_attribute_13   varchar2(4000),
   form_attribute_14   varchar2(4000),
   form_attribute_15   varchar2(4000),
   --
   display_as_tab_form varchar2(255)
                            constraint wwv_mpc_display_as_tab_form
                            check (display_as_tab_form in (
                                'READONLY',
                                'DISPLAY_AND_SAVE',
                                'WITHOUT_MODIFICATION',
                                'ESCAPE_SC',
                                'TEXT',
                                'TEXTAREA',
                                'TEXT_FROM_LOV',
                                'SELECT_LIST',
                                'SELECT_LIST_FROM_LOV',
                                'SELECT_LIST_FROM_QUERY',
                                'HIDDEN',
                                'HIDDEN_PROTECTED',
                                'POPUP',
                                'POPUP_QUERY',
                                'DATE_POPUP',
                                'DATE_PICKER',
                                'PICK_DATE_USING_APP_FORMAT_MASK',
                                'PICK_DATE_USING_APP_DATE_FORMAT',
                                'PICK_DATE_USING_FORMAT_MASK',
                                'PICK_DATE_DD_MON_RR',
                                'PICK_DATE_DD_MON_RR_HH_MI',
                                'PICK_DATE_DD_MON_RR_HH24_MI',
                                'PICK_DATE_DD_MON_YY',
                                'PICK_DATE_DD_MON_YY_HH_MI',
                                'PICK_DATE_DD_MON_YY_HH24_MI',
                                'PICK_DATE_DD_MON_YYYY',
                                'PICK_DATE_DD_MON_YYYY_HH_MI',
                                'PICK_DATE_DD_MON_YYYY_HH24_MI',
                                'PICK_DATE_DD_MM_YYYY',
                                'PICK_DATE_DD_MM_YYYY_HH_MI',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI',
                                'PICK_DATE_MM_DD_YYYY',
                                'PICK_DATE_MM_DD_YYYY_HH_MI',
                                'PICK_DATE_MM_DD_YYYY_HH24_MI',
                                'PICK_DATE_YYYY_MM_DD',
                                'PICK_DATE_YYYY_MM_DD_HH_MI',
                                'PICK_DATE_YYYY_MM_DD_HH24_MI',
                                'PICK_DATE_RR_MON_DD',
                                'PICK_DATE_RR_MON_DD_HH_MI',
                                'PICK_DATE_RR_MON_DD_HH24_MI',
                                'PICK_DATE_YYYY_MM_DD_SLASH',
                                'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH',
                                'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH',
                                'PICK_DATE_YYYY_DD_MM_DOT',
                                'PICK_DATE_YYYY_DD_MM_HH_MI_DOT',
                                'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT',
                                'PICK_DATE_DD_MM_YYYY_DASH',
                                'PICK_DATE_DD_MM_YYYY_HH_MI_DASH',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH',
                                'PICK_DATE_DD_MM_YYYY_DOT',
                                'PICK_DATE_DD_MM_YYYY_HH_MI_DOT',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT',
                                'CHECKBOX',
                                'SIMPLE_CHECKBOX',
                                'RADIOGROUP')),
   datatype             varchar2(4000)
                            constraint datatype
                            check (datatype in ('DATE','NUMBER','TIMESTAMP','TIMESTAMP_LTZ','VARCHAR2','BLOB','CLOB')),
   alignment            varchar2(4000)
                            constraint wwv_flow_mpc_alignment
                            check (alignment in ('LEFT','CENTER','RIGHT')),
   display_width        number,
   max_width            number,
   height               number,
   format_mask          varchar2(4000),
   hidden_column        varchar2(1),
   sort_sequence        number,
   sort_dir             number,        -- new 03/28, values ASC or DESC
   security_group_id    number         not null,
   created_by           varchar2(255)  not null,
   created_on           date           not null,
   last_updated_by      varchar2(255),
   last_updated_on      date
)
/

create index wwv_flow_mpc_mp_idx on wwv_flow_model_page_cols (model_region_id)
/

--##############################################################################
--#
--# WORKSHEETS (INTERACTIVE REPORTS)
--#
--##############################################################################
prompt WORKSHEETS (INTERACTIVE REPORTS)
--==============================================================================
create table wwv_flow_worksheet_categories (
    id                     number
                           constraint wwv_flow_worksheet_cat_pk
                           primary key,
    worksheet_id           number not null,
    flow_id                number not null,
    application_user       varchar2(255),
    name                   varchar2(255),
    display_sequence       number,
    --
    created_on             date,
    created_by             varchar2(255),
    updated_on             date,
    updated_by             varchar2(255),
    security_group_id      number        not null
    )
/

alter table wwv_flow_worksheet_categories
    add constraint wwv_flow_worksheet_cat_fk
    foreign key( security_group_id ) references wwv_flow_companies(provisioning_company_id)
    on delete cascade
/

alter table wwv_flow_worksheet_categories
   add constraint wwv_flow_worksheet_cat_fk2
    foreign key( flow_id ) references wwv_flows(id)
    on delete cascade
/

create index wwv_flow_worksheet_cat_idx1 on wwv_flow_worksheet_categories (security_group_id)
/

create index wwv_flow_worksheet_cat_idx2 on wwv_flow_worksheet_categories (flow_id)
/

--==============================================================================
create table wwv_flow_worksheet_col_groups (
    id                  number        not null,
    flow_id             number        not null
                        constraint wwv_flow_worksheet_col_grp_fk
                        references wwv_flows
                        on delete cascade,
    worksheet_id        number        not null,
    name                varchar2(255) not null,
    description         varchar2(4000),
    display_sequence    number not null,
    --
    created_on          date,
    created_by          varchar2(255),
    updated_on          date,
    updated_by          varchar2(255),
    security_group_id   number        not null
    )
/

alter table wwv_flow_worksheet_col_groups
    add constraint wwv_flow_worksheet_col_grps_pk
    primary key(id)
/

alter table wwv_flow_worksheet_col_groups
    add constraint wwv_flow_worksheet_col_grws_fk
    foreign key( worksheet_id ) references wwv_flow_worksheets(id)
    on delete cascade
/

alter table wwv_flow_worksheet_col_groups
    add constraint wwv_flow_worksheet_col_grps_uk
    unique(worksheet_id, name) deferrable initially deferred
/

create index wwv_flow_ws_col_grps_idx1 on wwv_flow_worksheet_col_groups (flow_id)
/

--==============================================================================
create table wwv_flow_worksheet_columns (
    id                  number    not null,
    flow_id             number    not null
                        constraint wwv_flow_worksheet_col_fk
                        references wwv_flows
                        on delete cascade,
    page_id             number,
    worksheet_id        number    not null,
    db_column_name      varchar2(128) not null,
    display_order       number    not null,
    data_grid_form_seq  number,
    group_id            number,
    column_identifier   varchar2(10) not null,
    column_expr         varchar2(4000),
    -- label
    column_label        varchar2(4000),
    report_label        varchar2(4000) not null,
    sync_form_label     varchar2(1) not null
                        constraint wwv_flow_val_wscol_sync_lab
                        check (sync_form_label in ('Y','N')),
    -- display
    display_in_default_rpt varchar2(1)
                           constraint wwv_flow_ws_display_col
                           check (display_in_default_rpt in ('Y','N')),
    column_html_expression varchar2(4000),
    column_link            varchar2(4000),
    column_linktext        varchar2(4000),
    column_link_attr       varchar2(4000),
    column_link_checksum_type varchar2(255)
                              constraint wwv_flow_ws_checksum_type
                              check (column_link_checksum_type in (
                              '1', -- workspace
                              '2', -- user
                              '3'  -- session
                              )),
    is_sortable         varchar2(1)
                        constraint wwv_flow_ws_col_sort_ck
                        check (is_sortable in ('Y','N')),
    allow_sorting       varchar2(1) not null
                        constraint wwv_flow_ws_col_sorting_ck
                        check (allow_sorting in ('Y','N')),
    allow_filtering     varchar2(1) not null
                        constraint wwv_flow_ws_col_filter_ck
                        check (allow_filtering in ('Y','N')),
    allow_highlighting  varchar2(1) not null
                        constraint wwv_flow_ws_col_highlight_ck
                        check (allow_highlighting in ('Y','N')),
    allow_ctrl_breaks   varchar2(1) not null
                        constraint wwv_flow_ws_col_ctrl_ck
                        check (allow_ctrl_breaks in ('Y','N')),
    allow_aggregations  varchar2(1) not null
                        constraint wwv_flow_ws_col_agg_ck
                        check (allow_aggregations in ('Y','N')),
    allow_computations  varchar2(1) not null
                        constraint wwv_flow_ws_col_comp_ck
                        check (allow_computations in ('Y','N')),
    allow_charting      varchar2(1) not null
                        constraint wwv_flow_ws_col_chart_ck
                        check (allow_charting in ('Y','N')),
    allow_group_by      varchar2(1) not null
                        constraint wwv_flow_ws_col_group_by_ck
                        check (allow_group_by in ('Y','N')),
    allow_pivot         varchar2(1) default 'N' not null
                        constraint wwv_flow_ws_col_pivot_ck
                        check (allow_pivot in ('Y','N')),
    allow_hide          varchar2(1) not null
                        constraint wwv_flow_ws_col_hide_ck
                        check (allow_hide in ('Y','N')),
    -- column filters
    use_custom          varchar2(1),
    custom_filter       varchar2(4000),
    base_column         varchar2(128),
    allow_filters       varchar2(1),
    -- security
    others_may_edit     varchar2(1),
    others_may_view     varchar2(1),
    -- display controls
    column_type         varchar2(255) not null
                        constraint wwv_flow_valid_ws_col_type
                        check (column_type in (
                        'STRING',
                        'DATE',
                        'NUMBER',
                        'CLOB',
                        'FILE',
                        'COMPUTED',
                        'DETAIL_LINK',
                        'OTHER')),
    display_as          varchar2(255)
                        constraint wwv_flow_ws_col_disp_as_ck
                        check (display_as in (
                        'READONLY',
                        'TEXT',
                        'TEXTAREA',
                        'SELECTLIST',
                        'CHECKBOX',
                        'RADIOGROUP',
                        'POPUP_LOV',
                        'DATE_PICKER')),
    display_text_as     varchar2(255) not null
                        constraint wwv_flow_ws_col_disp_txt_as_ck
                        check (display_text_as in (
                        'WITHOUT_MODIFICATION',
                        'ESCAPE_SC',
                        'STRIP_HTML_ESCAPE_SC',
                        'LOV_ESCAPE_SC',
                        'HIDDEN',
                        'HIDDEN_ESCAPE_SC',
                        'RICH_TEXT')),
    heading_alignment   varchar2(255) not null
                        constraint wwv_flow_ws_col_head_al_ck
                        check (heading_alignment in (
                        'LEFT',
                        'RIGHT',
                        'CENTER')),
    column_alignment    varchar2(255) not null
                        constraint wwv_flow_ws_col_col_al_ck
                        check (column_alignment in (
                        'LEFT',
                        'RIGHT',
                        'CENTER')),
    max_length          number,
    display_width       integer,
    display_height      integer,
    allow_null          varchar2(1)
                        constraint wwv_flow_wscol_allow_null
                        check (allow_null in ('Y','N')),
    format_mask         varchar2(255),
    tz_dependent        varchar2(1)
                        constraint wwv_flow_ws_col_tz_dep_ck
                        check (tz_dependent in ('Y','N')),
    static_id           varchar2(255),
    css_classes         varchar2(255),
    use_as_row_header   varchar2(1) default 'N'
                        constraint wwv_flow_wscol_row_hdr_ck
                        check (use_as_row_header in ('Y', 'N')),
    -- settings
    rich_text_format    varchar2(8)
                        constraint wwv_flow_wscol_rich_txt_format
                        check (rich_text_format in ('HTML', 'MARKDOWN')),
    -- warehouse
    fact_table_key        varchar2(255),
    dimension_table       varchar2(255),
    dimension_table_id    varchar2(255),
    dimension_table_value varchar2(255),
    -- report list of values
    rpt_distinct_lov    varchar2(1)
                        constraint wwv_flow_wscol_lov
                        check (rpt_distinct_lov in ('Y','N')),
    rpt_lov             varchar2(4000),
    rpt_named_lov       number,
    rpt_show_filter_lov varchar2(1)
                        constraint wwv_flow_wscol_showfil_lov_ck
                        check (rpt_show_filter_lov in ('D','S','N','C','1','2')),
    rpt_filter_date_ranges varchar2(255)
                        constraint wwv_flow_wscol_fil_dtrng_ck
                        check (rpt_filter_date_ranges in ('ALL','PAST','FUTURE')),
    -- form list of values
    static_lov          varchar2(4000),
    lov_null_text       varchar2(255),
    lov_allow_new_values varchar2(1)
                        constraint wwv_flow_ws_lov_allow_new
                        check (lov_allow_new_values in ('Y','N')),
    lov_is_distinct_values varchar2(1)
                        constraint wwv_flow_ws_lov_distinct_vals
                        check (lov_is_distinct_values in ('Y','N')),
    lov_num_columns     number,
    lov_id              number,
    --
    computation_type    varchar2(255),
    computation_expr_1  varchar2(4000),
    computation_expr_2  varchar2(4000),
    --
    validation_type     varchar2(255),
    validation_expr_1   varchar2(4000),
    validation_expr_2   varchar2(4000),
    --
    display_condition_type    varchar2(255),
    display_condition         varchar2(4000),
    display_condition2        varchar2(4000),
    -- default values
    default_value       varchar2(4000),
    default_when        varchar2(30),
    --
    help_text           varchar2(4000),
    --
    security_scheme     varchar2(255),
    --
    column_flags        varchar2(4000),
    column_comment      varchar2(4000),
    --
    required_patch      number,--
    --
    created_on          date,
    created_by          varchar2(255),
    updated_on          date,
    updated_by          varchar2(255),
    security_group_id   number        not null
    )
/

alter table wwv_flow_worksheet_columns
    add constraint wwv_flow_worksheet_columns_pk
    primary key(id)
/

alter table wwv_flow_worksheet_columns
    add constraint wwv_flow_worksheet_columns_fk
    foreign key( worksheet_id ) references wwv_flow_worksheets(id)
    on delete cascade
/

alter table wwv_flow_worksheet_columns
    add constraint wwv_flow_worksheet_col_grps_fk
    foreign key( group_id ) references wwv_flow_worksheet_col_groups(id)
    on delete set null
/

alter table wwv_flow_worksheet_columns
    add constraint wwv_flow_worksheet_columns_uk2
    unique(worksheet_id, db_column_name) deferrable initially deferred
/

create index wwv_flow_worksheet_cols_idx1 on wwv_flow_worksheet_columns (worksheet_id, column_identifier);
create index wwv_flow_worksheet_cols_idx2 on wwv_flow_worksheet_columns (flow_id);
create index wwv_flow_worksheet_cols_idx3 on wwv_flow_worksheet_columns (group_id);
create index wwv_flow_worksheet_cols_idx5 on wwv_flow_worksheet_columns (rpt_named_lov,flow_id);

--==============================================================================
-- Reports
create table wwv_flow_worksheet_rpts (
    id                     number not null,
    worksheet_id           number not null,
    flow_id                number not null,
    page_id                number not null,
    session_id             number,
    tenant_id              varchar2(32),
    base_report_id         number,
    application_user       varchar2(255),
    name                   varchar2(255),
    description            varchar2(4000),
    report_seq             number not null,
    report_type            varchar2(8)
                           constraint wwv_flow_ws_rpt_type_ck
                           check (report_type in (
                           'REPORT',
                           'CHART',
                           'GROUP_BY',
                           'PIVOT')),
    report_alias           varchar2(255),
    status                 varchar2(7)
                           constraint wwv_flow_ws_rpt_status_ck
                           check (status in (
                           'PRIVATE',
                           'PUBLIC')),
    view_mode              varchar2(30)
                           constraint wwv_flow_ws_rpt_vm_ck
                           check (view_mode in ('REPORT','ICON','DETAIL')),
    category_id            number,
    autosave               varchar2(1),
    is_default             varchar2(1) not null
                           constraint wwv_flow_ws_rpt_def_ck
                           check (is_default in (
                           'Y',
                           'N')),
    -- column display
    display_rows           number,
    pagination_min_row     number,
    report_columns         varchar2(4000),
    -- sorting
    sort_column_1          varchar2(255),
    sort_direction_1       varchar2(255),
    sort_column_2          varchar2(255),
    sort_direction_2       varchar2(255),
    sort_column_3          varchar2(255),
    sort_direction_3       varchar2(255),
    sort_column_4          varchar2(255),
    sort_direction_4       varchar2(255),
    sort_column_5          varchar2(255),
    sort_direction_5       varchar2(255),
    sort_column_6          varchar2(255),
    sort_direction_6       varchar2(255),
    -- control break
    break_on               varchar2(4000),
    break_enabled_on       varchar2(4000),
    control_break_options  varchar2(255),
    -- aggregation
    sum_columns_on_break   varchar2(4000),
    avg_columns_on_break   varchar2(4000),
    max_columns_on_break   varchar2(4000),
    min_columns_on_break   varchar2(4000),
    median_columns_on_break varchar2(4000),
    mode_columns_on_break  varchar2(4000),
    count_columns_on_break varchar2(4000),
    count_distnt_col_on_break varchar2(4000),
    -- flashback
    flashback_mins_ago     varchar2(255),
    flashback_enabled      varchar2(1) not null
                           constraint wwv_flow_ws_rpt_fb_ck
                           check (flashback_enabled in (
                           'Y',
                           'N')),
    -- chart
    chart_type             varchar2(255),
    chart_3d               varchar2(1),
    chart_label_column     varchar2(255),
    chart_label_title      varchar2(255),
    chart_value_column     varchar2(255),
    chart_aggregate        varchar2(255)
                           constraint wwv_flow_ws_rpt_chart_agg_ck
                           check (chart_aggregate in (
                           'SUM',
                           'AVG',
                           'MIN',
                           'MAX',
                           'COUNT')),
    chart_value_title      varchar2(255),
    chart_sorting          varchar2(30)
                           constraint wwv_flow_ws_rpt_chrt_ck
                           check (chart_sorting in (
                           'DEFAULT',
                           'VALUE_DESC',
                           'VALUE_ASC',
                           'LABEL_DESC',
                           'LABEL_ASC')),
    chart_orientation      varchar2(10)
                           constraint wwv_flow_ws_rpt_chrt_or_ck
                           check (chart_orientation in (
                           'vertical',
                           'horizontal')),
    -- calendar
    calendar_date_column    varchar2(255),
    calendar_display_column varchar2(255),
    -- audit
    created_on             date,
    created_by             varchar2(255),
    updated_on             date,
    updated_by             varchar2(255),
    security_group_id      number        not null
    )
/


alter table wwv_flow_worksheet_rpts
    add constraint wwv_flow_worksheet_rpts_pk
    primary key(id)
/

alter table wwv_flow_worksheet_rpts
    add constraint wwv_flow_worksheet_rpts_fk
    foreign key ( category_id ) references wwv_flow_worksheet_categories(id)
/

alter table wwv_flow_worksheet_rpts
    add constraint wwv_flow_worksheet_rpts_fk2
    foreign key( security_group_id ) references wwv_flow_companies(provisioning_company_id)
    on delete cascade
/

create index wwv_flow_ws_rpts_idx on wwv_flow_worksheet_rpts (worksheet_id, is_default);
create index wwv_flow_ws_rpts_idx4 on wwv_flow_worksheet_rpts ( session_id, worksheet_id, application_user );
create index wwv_flow_ws_base_rpt_idx on wwv_flow_worksheet_rpts (base_report_id, application_user);
create index wwv_flow_ws_cat_rpt_idx on wwv_flow_worksheet_rpts (category_id);
create index wwv_flow_ws_rpts_sgid_idx on wwv_flow_worksheet_rpts (security_group_id);

create unique index  wwv_flow_worksheet_rpts_uk on wwv_flow_worksheet_rpts
    (case when session_id is null then security_group_id end,
     case when session_id is null then worksheet_id end,
     case when session_id is null then application_user end,
     case when session_id is null then category_id end,
     case when session_id is null then name end)
/

--==============================================================================
create table wwv_flow_worksheet_conditions (
     id                  number not null,
     flow_id             number not null,
     page_id             number,
     worksheet_id        number not null,
     report_id           number not null,
     name                varchar2(255),
     condition_type      varchar2(255)
                         constraint wwv_flow_ws_c_type_ck
                         check (condition_type in ('FILTER','HIGHLIGHT','SEARCH')),
     allow_delete        varchar2(1) not null
                         constraint wwv_flow_ws_c_del_ck
                         check (allow_delete in ('Y','N','R')), -- R = Filter can be replaced by another on the same column
     -- filter expression
     column_name         varchar2(4000),
     operator            varchar2(30),
     expr_type           varchar2(255),
     expr                varchar2(4000),
     expr2               varchar2(4000),
     time_zone           varchar2(100),
     condition_sql       varchar2(4000),
     condition_display   varchar2(4000),
     -- enabled?
     enabled             varchar2(1) not null
                         constraint wwv_flow_ws_c_enabled_ck
                         check (enabled in ('Y','N')),
     -- highlighting settings
     highlight_sequence  number,
     row_bg_color        varchar2(255),
     row_font_color      varchar2(255),
     row_format          varchar2(3),
     column_bg_color     varchar2(255),
     column_font_color   varchar2(255),
     column_format       varchar2(3),
     -- audit
     created_on          date,
     created_by          varchar2(255),
     updated_on          date,
     updated_by          varchar2(255),
     security_group_id   number not null
     )
/

alter table wwv_flow_worksheet_conditions
    add constraint wwv_flow_worksheet_cond_pk
    primary key(id)
/

alter table wwv_flow_worksheet_conditions
    add constraint wwv_flow_worksheet_cond_fk
    foreign key ( report_id ) references wwv_flow_worksheet_rpts(id)
    on delete cascade
/

create unique index wwv_flow_worksheet_cond_uk2 on wwv_flow_worksheet_conditions
    (report_id,
     substr(condition_type,1,30),
     substr(column_name,1,30),
     operator,
     substr(expr,1,250),
     substr(expr2,1,100))
/

create index wwv_flow_ws_conditions_idx on wwv_flow_worksheet_conditions (worksheet_id, report_id, security_group_id, condition_type )
/

--==============================================================================
create table wwv_flow_worksheet_computation (
     id                  number not null,
     flow_id             number not null,
     page_id             number,
     worksheet_id        number not null,
     report_id           number not null,
     --
     db_column_name      varchar2(128) not null,
     column_identifier   varchar2(10) not null,
     computation_expr    varchar2(4000),
     format_mask         varchar2(255),
     column_type         varchar2(255)
                         constraint wwv_flow_valid_ws_comp_type
                         check (column_type in (
                         'STRING',
                         'DATE',
                         'NUMBER',
                         'CLOB',
                         'OTHER')),
     -- label
     column_label        varchar2(4000) not null,
     report_label        varchar2(4000) not null,
     -- audit
     created_on          date,
     created_by          varchar2(255),
     updated_on          date,
     updated_by          varchar2(255),
     security_group_id   number not null
     )
/

alter table wwv_flow_worksheet_computation
    add constraint wwv_flow_ws_comp_cols_pk
    primary key(id)
/

alter table wwv_flow_worksheet_computation
    add constraint wwv_flow_ws_comp_cols_fk
    foreign key ( report_id ) references wwv_flow_worksheet_rpts(id)
    on delete cascade
/

alter table wwv_flow_worksheet_computation
    add constraint wwv_flow_ws_comp_cols_uk
    unique(report_id, db_column_name)
/

alter table wwv_flow_worksheet_computation
    add constraint wwv_flow_ws_comp_cols_uk2
    unique(report_id, column_identifier)
/

create index wwv_flow_ws_comp_cols_idx on wwv_flow_worksheet_computation (worksheet_id)
/

--==============================================================================
create table wwv_flow_worksheet_group_by (
     id                         number not null
                                constraint wwv_flow_ws_groupby_pk
                                primary key,
     flow_id                    number not null,
     page_id                    number,
     worksheet_id               number not null,
     report_id                  number not null
                                constraint wwv_flow_ws_groupby_fk2
                                references wwv_flow_worksheet_rpts(id)
                                on delete cascade,
     --
     group_by_columns           varchar2(4000),
     -- audit
     created_on                 date,
     created_by                 varchar2(255),
     updated_on                 date,
     updated_by                 varchar2(255),
     security_group_id          number not null,
     --
     function_01                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn1_ck
                                check (function_01 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_01         varchar2(255),
     function_db_column_name_01 varchar2(128),
     function_label_01          varchar2(4000),
     function_format_mask_01    varchar2(255),
     function_sum_01            varchar2(1),
     --
     function_02                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn2_ck
                                check (function_02 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_02         varchar2(255),
     function_db_column_name_02 varchar2(128),
     function_label_02          varchar2(4000),
     function_format_mask_02    varchar2(255),
     function_sum_02            varchar2(1),
     --
     function_03                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn3_ck
                                check (function_03 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_03         varchar2(255),
     function_db_column_name_03 varchar2(128),
     function_label_03          varchar2(4000),
     function_format_mask_03    varchar2(255),
     function_sum_03            varchar2(1),
     --
     function_04                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn4_ck
                                check (function_04 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_04         varchar2(255),
     function_db_column_name_04 varchar2(128),
     function_label_04          varchar2(4000),
     function_format_mask_04    varchar2(255),
     function_sum_04            varchar2(1),
     --
     function_05                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn5_ck
                                check (function_05 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_05         varchar2(255),
     function_db_column_name_05 varchar2(128),
     function_label_05          varchar2(4000),
     function_format_mask_05    varchar2(255),
     function_sum_05            varchar2(1),
     --
     function_06                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn6_ck
                                check (function_06 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_06         varchar2(255),
     function_db_column_name_06 varchar2(128),
     function_label_06          varchar2(4000),
     function_format_mask_06    varchar2(255),
     function_sum_06            varchar2(1),
     --
     function_07                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn7_ck
                                check (function_07 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_07         varchar2(255),
     function_db_column_name_07 varchar2(128),
     function_label_07          varchar2(4000),
     function_format_mask_07    varchar2(255),
     function_sum_07            varchar2(1),
     --
     function_08                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn8_ck
                                check (function_08 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_08         varchar2(255),
     function_db_column_name_08 varchar2(128),
     function_label_08          varchar2(4000),
     function_format_mask_08    varchar2(255),
     function_sum_08            varchar2(1),
     --
     function_09                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn9_ck
                                check (function_09 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_09         varchar2(255),
     function_db_column_name_09 varchar2(128),
     function_label_09          varchar2(4000),
     function_format_mask_09    varchar2(255),
     function_sum_09            varchar2(1),
     --
     function_10                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn10_ck
                                check (function_10 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_10         varchar2(255),
     function_db_column_name_10 varchar2(128),
     function_label_10          varchar2(4000),
     function_format_mask_10    varchar2(255),
     function_sum_10            varchar2(1),
     --
     function_11                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn11_ck
                                check (function_11 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_11         varchar2(255),
     function_db_column_name_11 varchar2(128),
     function_label_11          varchar2(4000),
     function_format_mask_11    varchar2(255),
     function_sum_11            varchar2(1),
     --
     function_12                varchar2(255)
                                constraint wwv_flow_ws_grpby_funtn12_ck
                                check (function_12 in (
                                'SUM',
                                'AVG',
                                'MAX',
                                'MIN',
                                'MEDIAN',
                                'COUNT',
                                'COUNT_DISTINCT',
                                'RATIO_TO_REPORT_SUM',
                                'RATIO_TO_REPORT_COUNT')),
     function_column_12         varchar2(255),
     function_db_column_name_12 varchar2(128),
     function_label_12          varchar2(4000),
     function_format_mask_12    varchar2(255),
     function_sum_12            varchar2(1),
     --
     sort_column_01             varchar2(255),
     sort_direction_01          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir1_ck
                                check (sort_direction_01 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_02             varchar2(255),
     sort_direction_02          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir2_ck
                                check (sort_direction_02 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_03             varchar2(255),
     sort_direction_03          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir3_ck
                                check (sort_direction_03 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_04             varchar2(255),
     sort_direction_04          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir4_ck
                                check (sort_direction_04 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_05             varchar2(255),
     sort_direction_05          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir5_ck
                                check (sort_direction_05 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_06             varchar2(255),
     sort_direction_06          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir6_ck
                                check (sort_direction_06 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_07             varchar2(255),
     sort_direction_07          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir7_ck
                                check (sort_direction_07 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_08             varchar2(255),
     sort_direction_08          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir8_ck
                                check (sort_direction_08 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_09             varchar2(255),
     sort_direction_09          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir9_ck
                                check (sort_direction_09 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_10             varchar2(255),
     sort_direction_10          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir10_ck
                                check (sort_direction_10 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_11             varchar2(255),
     sort_direction_11          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir11_ck
                                check (sort_direction_11 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST')),
     --
     sort_column_12             varchar2(255),
     sort_direction_12          varchar2(255)
                                constraint wwv_flow_ws_grpby_sortdir12_ck
                                check (sort_direction_12 in (
                                'ASC',
                                'ASC NULLS LAST',
                                'ASC NULLS FIRST',
                                'DESC',
                                'DESC NULLS LAST',
                                'DESC NULLS FIRST'))
     )
/

create index wwv_flow_ws_groupby_idx on wwv_flow_worksheet_group_by (worksheet_id)
/

create index wwv_flow_ws_groupby_idx3 on wwv_flow_worksheet_group_by (report_id)
/

--==============================================================================
create table wwv_flow_worksheet_notify (
    id                          number
                                constraint wwv_flow_worksheet_notify_pk
                                primary key,
    worksheet_id                number,
    report_id                   number
                                constraint wwv_flow_worksheet_notify_fk2
                                references wwv_flow_worksheet_rpts(id)
                                on delete cascade,
    session_id                  number /*no FK to avoid mutating table issues */,
    owner                       varchar2(255) not null,
    language                    varchar2(255),
    download_format             varchar2(255) default 'HTML' not null
                                constraint wwv_flow_ws_format_ck
                                     check (download_format in (
                                          'PDF',
                                          'XLSX',
                                          'HTML',
                                          'CSV'
                                      )),
    strip_rich_text             varchar2(1) constraint wwv_flow_ws_strip_rt_ck
                                 check(strip_rich_text in ('Y','N')),
    data_only                   varchar2(1) constraint wwv_flow_ws_data_only_ck
                                 check(data_only in ('Y','N')),
    pdf_page_size               varchar2(7) constraint wwv_flow_ws_pdf_page_size
                                     check (pdf_page_size in (
                                        'CUSTOM',
                                        'LETTER',
                                        'LEGAL',
                                        'TABLOID',
                                        'A4',
                                        'A3'
                                     )),
    pdf_page_orientation        varchar2(10) constraint wwv_flow_ws_pdf_orientation
                                     check (pdf_page_orientation in (
                                         'HORIZONTAL',
                                         'VERTICAL'
                                     )),
    pdf_accessible              varchar2(1) constraint wwv_flow_ws_pdf_accessible_ck
                                 check(pdf_accessible in ('Y','N')),
    skip_no_data_found          varchar2(1) default 'N' not null
                                constraint wwv_flow_ws_skip_ndf_ck
                                 check(skip_no_data_found in ('Y','N')),
    email_from                  varchar2(255),
    email_address               varchar2(255) not null,
    email_subject               varchar2(4000),
    start_date                  timestamp with time zone not null,
    end_date                    timestamp with time zone,
    end_day                     number,
    end_day_unit                varchar2(255),
    offset_date                 timestamp with time zone,
    notify_interval             varchar2(1)
                                constraint wwv_flow_worksheet_notify_ck check
                                (notify_interval in ('H','D','BW','W','M')),
    status                      varchar2(255)
                                constraint wwv_flow_worksheet_nstatus_ck check
                                (status in ('SUBMITTED','SENT','ERROR','SKIPPED')),
    notify_error                varchar2(4000),
    security_group_id           number not null
                                constraint wwv_flow_worksheet_notify_fk5
                                references wwv_flow_companies(provisioning_company_id)
                                on delete cascade,
    created_by                  varchar2(255),
    created_on                  date,
    updated_by                  varchar2(255),
    updated_on                  date
    )
/

create index wwv_flow_worksheet_ntfy_idx1 on wwv_flow_worksheet_notify (worksheet_id)
/
create index wwv_flow_worksheet_ntfy_idx3 on wwv_flow_worksheet_notify (report_id)
/
create index wwv_flow_worksheet_ntfy_idx4 on wwv_flow_worksheet_notify (security_group_id) compress
/

--==============================================================================
create table wwv_flow_worksheet_pivot (
    id                         number not null
                               constraint wwv_flow_ws_pivot_pk
                               primary key,
    flow_id                    number not null,
    page_id                    number,
    worksheet_id               number not null,
    report_id                  number not null
                               constraint wwv_flow_ws_pivot_fk2
                               references wwv_flow_worksheet_rpts(id)
                               on delete cascade,
    --
    pivot_columns              varchar2(4000) not null, -- colone separated columns to pivot on, vertical columns
    row_columns                varchar2(4000) not null, -- colone separated columns to display in pivot view, horizontal columns
    -- audit
    created_on                 date,
    created_by                 varchar2(255),
    updated_on                 date,
    updated_by                 varchar2(255),
    security_group_id          number not null
    )
/

create index wwv_flow_ws_pivot_idx2 on wwv_flow_worksheet_pivot (report_id)
/

-- pivot can occur once per IR saved report
create unique index wwv_flow_ws_pivot_idx3 on wwv_flow_worksheet_pivot (worksheet_id, report_id) compress 1
/

--
-- pivot detail table to store aggregate information
--
create table wwv_flow_worksheet_pivot_agg (
    id                         number not null
                               constraint wwv_flow_ws_pivot_agg_pk
                               primary key,
    pivot_id                   number not null
                               constraint wwv_flow_ws_pivot_agg_fk1
                               references wwv_flow_worksheet_pivot(id)
                               on delete cascade,
    display_seq                number not null,
    function_name              varchar2(255) not null
                               constraint wwv_flow_ws_pivot_agg_funtn_ck
                               check (function_name in (
                               'SUM',
                               'AVG',
                               'MAX',
                               'MIN',
                               'MEDIAN',
                               'COUNT',
                               'COUNT_DISTINCT',
                               'RATIO_TO_REPORT_SUM',
                               'RATIO_TO_REPORT_COUNT')),
    column_name                varchar2(255) not null,
    db_column_name             varchar2(128) not null,
    column_label               varchar2(4000),
    format_mask                varchar2(255),
    display_sum                varchar2(1) default 'N'
                               constraint wwv_flow_ws_pivot_agg_sum_ck
                               check (display_sum in (
                               'Y',
                               'N')),
    -- audit
    created_on                 date,
    created_by                 varchar2(255),
    updated_on                 date,
    updated_by                 varchar2(255),
    security_group_id          number not null
    )
/

create unique index wwv_flow_ws_pivot_agg_idx2 on wwv_flow_worksheet_pivot_agg (pivot_id, function_name, column_name, db_column_name) compress
/

--
-- pivot detail table to store sorting
--
create table wwv_flow_worksheet_pivot_sort (
    id                         number not null
                               constraint wwv_flow_ws_pivot_sort_pk
                               primary key,
    pivot_id                   number not null
                               constraint wwv_flow_ws_pivot_sort_fk1
                               references wwv_flow_worksheet_pivot(id)
                               on delete cascade,
    sort_seq                   number not null,
    sort_column_name           varchar2(255) not null,
    sort_direction             varchar2(255) not null
                               constraint wwv_flow_ws_pivot_sortdir_ck
                               check (sort_direction in (
                               'ASC',
                               'ASC NULLS LAST',
                               'ASC NULLS FIRST',
                               'DESC',
                               'DESC NULLS LAST',
                               'DESC NULLS FIRST')),
    -- audit
    created_on                 date,
    created_by                 varchar2(255),
    updated_on                 date,
    updated_by                 varchar2(255),
    security_group_id          number not null
    )
/

create unique index wwv_flow_ws_pivot_sor_idx2 on wwv_flow_worksheet_pivot_sort (pivot_id, sort_column_name) compress
/

--==============================================================================
create table wwv_flow_worksheet_geocache (
    address varchar2(4000),
    geocode varchar2(512)
    )
/

--##############################################################################
--#
--# REGION > INTERACTIVE GRID
--#
--##############################################################################
prompt REGION > INTERACTIVE GRID

--==============================================================================
create table wwv_flow_interactive_grids (
    id                          number
                                constraint ig_pk
                                primary key,
    flow_id                     number not null
                                constraint ig_flow_fk
                                references wwv_flows
                                on delete cascade,
    page_id                     number not null,
    region_id                   number not null
                                constraint ig_region_fk
                                references wwv_flow_page_plugs
                                on delete cascade,
    security_group_id           number not null,
    --
    internal_uid                number not null,
    -- Edit
    is_editable                 varchar2(1) not null
                                constraint ig_is_editable_ck
                                check ( is_editable in ( 'Y', 'N' )),
    edit_operations             varchar2(5)
                                constraint ig_edit_operations_ck
                                check ( regexp_instr( edit_operations, '[^iud:]+') = 0 ), /* only allow i, u, d, : */
    edit_row_operations_column  varchar2(128),
    add_authorization_scheme    varchar2(40),
    update_authorization_scheme varchar2(40),
    delete_authorization_scheme varchar2(40),
    lost_update_check_type      varchar2(6)
                                constraint ig_lost_update_check_type_ck
                                check ( lost_update_check_type in ( 'VALUES', 'COLUMN' )),
    row_version_column          varchar2(128),
    add_row_if_empty            varchar2(1)
                                constraint ig_add_row_if_empty_ck
                                check ( add_row_if_empty in ( 'Y', 'N' )),
    submit_checked_rows         varchar2(1)
                                constraint ig_submit_checked_rows_ck
                                check ( submit_checked_rows in ( 'Y', 'N' )),
    -- Performance
    lazy_loading                varchar2(1) not null
                                constraint ig_lazy_loading_ck
                                check ( lazy_loading in ( 'Y', 'N' )),
    requires_filter             varchar2(1) not null
                                constraint ig_requires_filter_ck
                                check ( requires_filter in ( 'Y', 'N' )),
    max_row_count               number,
    -- Appearance
    show_nulls_as               varchar2(30),
    fixed_row_height            varchar2(1) default 'Y' not null
                                constraint ig_fixed_row_height
                                check ( fixed_row_height in ( 'Y', 'N' )),
    select_first_row            varchar2(1) default 'Y' not null
                                constraint ig_select_first_row_ck
                                check ( select_first_row in ( 'Y', 'N' )),
    -- Pagination
    pagination_type             varchar2(6) not null
                                constraint ig_pagination_type_ck
                                check ( pagination_type in ( 'NONE', 'SET', 'SCROLL' )),
    show_total_row_count        varchar2(1)
                                constraint ig_show_total_row_count_ck
                                check ( show_total_row_count in ( 'Y', 'N' )),
    -- Messages
    no_data_found_message       varchar2(4000),
    max_row_count_message       varchar2(4000),
    -- Toolbar
    show_toolbar                varchar2(1) not null
                                constraint ig_show_toolbar_ck
                                check ( show_toolbar in ( 'Y', 'N' )),
    toolbar_buttons             varchar2(255),
    add_button_label            varchar2(255),
    -- Enable End Users to
    enable_save_public_report   varchar2(1) not null
                                constraint ig_enable_save_public_rpt_ck
                                check ( enable_save_public_report in ( 'Y', 'N' )),
    public_report_auth_scheme   varchar2(40),
    enable_subscriptions        varchar2(1) not null
                                constraint ig_enable_subscriptions_ck
                                check ( enable_subscriptions in ( 'Y', 'N' )),
    enable_flashback            varchar2(1) default 'Y' not null
                                constraint ig_enable_flashback_ck
                                check ( enable_flashback in ( 'Y', 'N' )),
    define_chart_view           varchar2(1) default 'Y' not null
                                constraint ig_def_chart_view_ck
                                check ( define_chart_view in ( 'Y', 'N' )),
    enable_download             varchar2(1) not null
                                constraint ig_enable_download
                                check ( enable_download in ( 'Y', 'N' )),
    download_auth_scheme        varchar2(40),
    download_formats            varchar2(20)
                                constraint ig_download_formats_ck
                                check ( regexp_instr( download_formats, '[^(CSV)|(HTML)|(XLSX)|(PDF)|:]+') = 0 ), /* only allow CSV, HTML, ... and : */
    download_filename           varchar2(255),
    enable_mail_download        varchar2(1)
                                constraint ig_enable_mail_download_ck
                                check ( enable_mail_download in ( 'Y', 'N' )),
    csv_separator               varchar2(1),
    csv_enclosed_by             varchar2(1),
    supplemental_text           varchar2(4000),
    -- Heading
    fixed_header                varchar2(6) not null
                                constraint ig_fixed_header_ck
                                check ( fixed_header in ( 'PAGE', 'REGION', 'NONE' )),
    fixed_header_max_height     number,
    -- Icon View
    show_icon_view              varchar2(1) not null
                                constraint ig_show_icon_view_ck
                                check ( show_icon_view in ( 'Y', 'N' )),
    icon_view_use_custom        varchar2(1)
                                constraint ig_icon_view_use_custom_ck
                                check ( icon_view_use_custom in ( 'Y', 'N' )),
    icon_view_custom            varchar2(4000),
    icon_view_icon_type         varchar2(11)
                                constraint ig_icon_view_icon_type_ck
                                check ( icon_view_icon_type in ( 'BLOB', 'URL', 'CSS_CLASSES' )),
    icon_view_icon_column       varchar2(128),
    icon_view_icon_attributes   varchar2(255),
    icon_view_link_target       varchar2(4000),
    icon_view_link_attributes   varchar2(255),
    icon_view_label_column      varchar2(128),
    -- Detail View
    show_detail_view            varchar2(1) not null
                                constraint ig_show_detail_view_ck
                                check ( show_detail_view in ( 'Y', 'N' )),
    detail_view_before_rows     varchar2(4000),
    detail_view_for_each_row    varchar2(4000),
    detail_view_after_rows      varchar2(4000),
    -- Advanced
    oracle_text_index_column    varchar2(128),
    email_from                  varchar2(255),
    javascript_code             varchar2(4000),
    -- Help Text
    help_text                   varchar2(4000),
    -- Audit
    last_updated_by             varchar2(255),
    last_updated_on             date
  )
/
alter table wwv_flow_interactive_grids add constraint ig_region_uk unique ( region_id )
/
create index ig_page_ix on wwv_flow_interactive_grids ( flow_id, page_id )
/


--##############################################################################
--#
--# REGION > INTERACTIVE GRID > SAVED REPORTS
--#
--##############################################################################
prompt REGION > INTERACTIVE GRID > REPORTS

--==============================================================================
create table wwv_flow_ig_reports (
    id                          number
                                constraint ig_rpt_pk
                                primary key,
    flow_id                     number not null
                                constraint ig_rpt_flow_fk
                                references wwv_flows
                                on delete cascade,
    page_id                     number not null,
    interactive_grid_id         number not null
                                constraint ig_rpt_grid_fk
                                references wwv_flow_interactive_grids
                                on delete cascade,
    tenant_id                   varchar2(32),
    security_group_id           number not null,
    --
    name                        varchar2(255),
    static_id                   varchar2(255),
    description                 varchar2(4000),
    type                        varchar2(11) not null
                                constraint ig_rpt_type_ck
                                check ( type in ( 'PRIMARY', 'ALTERNATIVE', 'PRIVATE', 'PUBLIC' )),
    default_view                varchar2(7) not null
                                constraint ig_rpt_default_view_ck
                                check ( default_view in ( 'ICON', 'GRID', 'DETAIL', 'GROUPBY', 'PIVOT', 'CHART' )),
    -- Information for alternative or private reports
    authorization_scheme        varchar2(40),
    application_user            varchar2(255),
    -- Information for temporary reports
    session_id                  number
                                constraint ig_session_fk
                                references wwv_flow_sessions$ on delete cascade,
    base_report_id              number
                                constraint ig_base_report_fk
                                references wwv_flow_ig_reports
                                on delete set null,
    -- global settings
    rows_per_page               number,
    show_row_number             varchar2(1) not null
                                constraint ig_rpt_show_row_number_ck
                                check ( show_row_number in ( 'Y', 'N' )),
    settings_area_expanded      varchar2(1) not null
                                constraint ig_rpt_settings_area_exp_ck
                                check ( settings_area_expanded in ( 'Y', 'N' )),
    --
    flashback_mins_ago          number,
    flashback_is_enabled        varchar2(1)
                                constraint ig_rpt_flashback_is_enabled_ck
                                check ( flashback_is_enabled in ( 'Y', 'N' )),
    -- Audit
    version_number              number default 1 not null,
    last_updated_by             varchar2(255),
    last_updated_on             date
)
/
create index ig_rpt_page_ix    on wwv_flow_ig_reports ( flow_id, page_id )
/
create index ig_rpt_access_ix  on wwv_flow_ig_reports ( interactive_grid_id, type, application_user, session_id )
/
create index ig_rpt_sessopm_fx on wwv_flow_ig_reports ( session_id )
/
create index ig_rpt_base_report_fx on wwv_flow_ig_reports ( base_report_id )
/
create unique index ig_rpt_name_uk on wwv_flow_ig_reports ( interactive_grid_id, type, name, application_user, tenant_id, session_id, base_report_id )
/

--==============================================================================
create table wwv_flow_ig_report_views (
    id                          number
                                constraint ig_rpt_view_pk
                                primary key,
    report_id                   number not null
                                constraint ig_rpt_view_report_fk
                                references wwv_flow_ig_reports
                                on delete cascade,
    security_group_id           number not null,
    --
    view_type                   varchar2(7) not null
                                constraint ig_rpt_view_type_ck
                                check ( view_type in ( 'ICON', 'GRID', 'DETAIL', 'GROUPBY', 'PIVOT', 'CHART' )),
    -- Control Break
    breaks_expanded             varchar2(1)
                                constraint ig_rpt_view_breaks_exp_ck
                                check ( breaks_expanded in ( 'Y', 'N' )),
    stretch_columns             varchar2(1)
                                constraint ig_rpt_view_stretch_cols_ck
                                check ( stretch_columns in ( 'Y', 'N' )),
    -- Single Row View
    srv_exclude_null_values     varchar2(1)
                                constraint ig_rpt_view_excl_null_ck
                                check ( srv_exclude_null_values in ( 'Y', 'N' )),
    srv_only_display_columns    varchar2(1)
                                constraint ig_rpt_view_displ_cols_ck
                                check ( srv_only_display_columns in ( 'Y', 'N' )),
    -- Grid View specific attributes
    edit_mode                   varchar2(1)
                                constraint ig_rpt_view_edit_mode_ck
                                check ( edit_mode in ( 'Y', 'N' )),
    -- Chart View specific attributes
    chart_type                  varchar2(12)
                                constraint ig_rpt_view_chart_type_ck
                                check ( chart_type in ( 'area', 'bar', 'bubble', 'donut',
                                                        'funnel', 'line', 'lineWithArea',
                                                        'pie', 'polar', 'radar', 'range',
                                                        'scatter', 'stock' )),
    chart_orientation           varchar2(10)
                                constraint ig_rpt_view_chart_orient_ck
                                check ( chart_orientation in ( 'horizontal', 'vertical' )),
    chart_stack                 varchar2(3)
                                constraint ig_rpt_view_chart_stack_ck
                                check ( chart_stack in ( 'on', 'off' )),
    axis_label                  varchar2(255),
    axis_value                  varchar2(255),
    axis_value_decimal_places   number
)
/
alter table wwv_flow_ig_report_views
  add constraint ig_rpt_view_uk unique ( report_id, view_type )
/


--==============================================================================
create table wwv_flow_ig_report_computes (
    id                          number
                                constraint ig_rpt_comp_pk
                                primary key,
    report_id                   number not null
                                constraint ig_rpt_comp_report_fk
                                references wwv_flow_ig_reports
                                on delete cascade,
    view_id                     number
                                constraint ig_rpt_comp_view_fk
                                references wwv_flow_ig_report_views
                                on delete cascade,
    security_group_id           number not null,
    -- Computation
    function                    varchar2(21) not null
                                constraint ig_rpt_comp_function_ck
                                check ( function in ( 'SUM', 'AVG', 'COUNT', 'COUNT_DISTINCT', 'APPROX_COUNT_DISTINCT', 'MIN', 'MAX', 'MEDIAN', 'LISTAGG', 'SQL_EXPRESSION' )),
    column_id                   number
                                constraint ig_rpt_comp_column_fk
                                references wwv_flow_region_columns
                                on delete cascade,
    /* ??? when would this be used? For example in a Group By/Pivot when doing this on a computed column */
    comp_column_id              number
                                constraint ig_rpt_comp_comp_column_fk
                                references wwv_flow_ig_report_computes
                                on delete cascade deferrable initially deferred,
                                -- deferrable is necessary for ig report import, because the order of compute is undefined
    partition_by_clause         varchar2(4000),
    order_by_clause             varchar2(4000),
    sql_expression              varchar2(4000),
    data_type                   varchar2(128),
    -- Display
    heading                     varchar2(255),
    label                       varchar2(255),
    heading_alignment           varchar2(6)
                                constraint ig_rpt_comp_heading_align_ck
                                check ( heading_alignment in (
                                            'LEFT',
                                            'RIGHT',
                                            'CENTER' )),
    column_alignment            varchar2(6)
                                constraint ig_rpt_comp_column_align_ck
                                check ( column_alignment in (
                                            'LEFT',
                                            'RIGHT',
                                            'CENTER' )),
    group_id                    number
                                constraint ig_rpt_comp_group_fk
                                references wwv_flow_region_column_groups on delete set null,
    use_group_for               varchar2(7)
                                constraint ig_rpt_comp_use_group_for_ck
                                check ( use_group_for in ( 'BOTH', 'SRV', 'HEADING' )),
    format_mask                 varchar2(50),
    --
    is_enabled                  varchar2(1) not null
                                constraint ig_rpt_comp_is_enabled_ck
                                check ( is_enabled in ( 'Y', 'N' ))
)
/
create index  ig_rpt_comp_report_fx on wwv_flow_ig_report_computes ( report_id )
/
create index  ig_rpt_comp_view_fx on wwv_flow_ig_report_computes ( view_id )
/
create index  ig_rpt_comp_column_fx on wwv_flow_ig_report_computes ( column_id )
/
create index  ig_rpt_comp_comp_column_fx on wwv_flow_ig_report_computes ( comp_column_id )
/
create index  ig_rpt_comp_group_fx on wwv_flow_ig_report_computes ( group_id )
/


--==============================================================================
create table wwv_flow_ig_report_filters (
    id                          number
                                constraint ig_rpt_filter_pk
                                primary key,
    report_id                   number not null
                                constraint ig_rpt_filter_report_fk
                                references wwv_flow_ig_reports
                                on delete cascade,
    view_id                     number
                                constraint ig_rpt_filter_view_fk
                                references wwv_flow_ig_report_views
                                on delete cascade,
    security_group_id           number not null,
    --
    type                        varchar2(7) not null
                                constraint ig_rpt_filter_type_ck
                                check ( type in ( 'ROW', 'COLUMN', 'COMPLEX' )),
    --
    name                        varchar2(40),
    --
    column_id                   number
                                constraint ig_rpt_filter_column_fk
                                references wwv_flow_region_columns
                                on delete cascade,
    comp_column_id              number
                                constraint ig_rpt_filter_comp_column_fk
                                references wwv_flow_ig_report_computes
                                on delete cascade,
    --
    operator                    varchar2(40),
    is_case_sensitive           varchar2(1)
                                constraint ig_rpt_filter_case_ck
                                check ( is_case_sensitive in ( 'Y', 'N' )),
    expression                  varchar2(4000),
    --
    is_enabled                  varchar2(1) not null
                                constraint ig_rpt_filter_is_enabled_ck
                                check ( is_enabled in ( 'Y', 'N' ))
    --
)
/
create index  ig_rpt_filter_report_fx on wwv_flow_ig_report_filters ( report_id )
/
create index  ig_rpt_filter_view_fx on wwv_flow_ig_report_filters ( view_id )
/
create index  ig_rpt_filter_column_fx on wwv_flow_ig_report_filters ( column_id )
/
create index  ig_rpt_filter_comp_column_fx on wwv_flow_ig_report_filters ( comp_column_id )
/


--==============================================================================
create table wwv_flow_ig_report_columns (
    id                          number
                                constraint ig_rpt_col_pk
                                primary key,
    view_id                     number not null
                                constraint ig_rpt_col_view_fk
                                references wwv_flow_ig_report_views
                                on delete cascade,
    security_group_id           number not null,
    --
    display_seq                 number not null,
    --
    column_id                   number
                                constraint ig_rpt_col_column_fk
                                references wwv_flow_region_columns
                                on delete cascade,
    comp_column_id              number
                                constraint ig_rpt_col_comp_column_fk
                                references wwv_flow_ig_report_computes
                                on delete cascade,
    --
    is_visible                  varchar2(1) not null
                                constraint ig_rpt_col_is_visible_ck
                                check ( is_visible in ( 'Y', 'N' )),
    is_frozen                   varchar2(1) not null
                                constraint ig_rpt_col_is_frozen_ck
                                check ( is_frozen in ( 'Y', 'N' )),
    width                       number,
    priority                    number,
    --
    break_order                 number,
    break_is_enabled            varchar2(1)
                                constraint ig_rpt_col_break_is_enabled_ck
                                check ( break_is_enabled in ( 'Y', 'N' )),
    break_sort_direction        varchar2(4)
                                constraint ig_rpt_col_break_sort_dir_ck
                                check ( break_sort_direction in ( 'ASC', 'DESC' )),
    break_sort_nulls            varchar2(5)
                                constraint ig_rpt_col_break_sort_nulls_ck
                                check ( break_sort_nulls in ( 'FIRST', 'LAST' )),
    --
    sort_order                  number,
    sort_direction              varchar2(4)
                                constraint ig_rpt_col_sort_direction_ck
                                check ( sort_direction in ( 'ASC', 'DESC' )),
    sort_nulls                  varchar2(5)
                                constraint ig_rpt_col_sort_nulls_ck
                                check ( sort_nulls in ( 'FIRST', 'LAST' ))
)
/
create index  ig_rpt_col_view_fx on wwv_flow_ig_report_columns ( view_id )
/
create index  ig_rpt_col_column_fx on wwv_flow_ig_report_columns ( column_id )
/
create index  ig_rpt_col_comp_column_fx on wwv_flow_ig_report_columns ( comp_column_id )
/


--==============================================================================
create table wwv_flow_ig_report_highlights (
    id                          number
                                constraint ig_rpt_highlight_pk
                                primary key,
    view_id                     number not null
                                constraint ig_rpt_highlight_view_fk
                                references wwv_flow_ig_report_views
                                on delete cascade,
    security_group_id           number not null,
    --
    execution_seq               number not null,
    name                        varchar2(40) not null,
    -- Highlight row/column
    column_id                   number
                                constraint ig_rpt_high_column_fk
                                references wwv_flow_region_columns
                                on delete cascade,
    comp_column_id              number
                                constraint ig_rpt_high_comp_column_fk
                                references wwv_flow_ig_report_computes
                                on delete cascade,
    --
    background_color            varchar2(15),
    text_color                  varchar2(15),
    -- Highlight condition
    condition_type              varchar2(7) not null
                                constraint ig_rpt_high_cond_type_ck
                                check ( condition_type in ( 'ROW', 'COLUMN', 'COMPLEX' )),
    condition_column_id         number
                                constraint ig_rpt_high_cond_column_fk
                                references wwv_flow_region_columns
                                on delete cascade,
    condition_comp_column_id    number
                                constraint ig_rpt_high_cond_comp_col_fk
                                references wwv_flow_ig_report_computes
                                on delete cascade,
    --
    condition_operator          varchar2(40),
    condition_is_case_sensitive varchar2(1)
                                constraint ig_rpt_highlight_case_ck
                                check ( condition_is_case_sensitive in ( 'Y', 'N' )),
    condition_expression        varchar2(4000),
    --
    is_enabled                  varchar2(1) not null
                                constraint ig_rpt_highlight_is_enabled_ck
                                check ( is_enabled in ( 'Y', 'N' ))
    --
)
/
create index  ig_rpt_highlight_view_fx on wwv_flow_ig_report_highlights ( view_id )
/
create index  ig_rpt_high_column_fx on wwv_flow_ig_report_highlights ( column_id )
/
create index  ig_rpt_high_comp_column_fx on wwv_flow_ig_report_highlights ( comp_column_id )
/
create index  ig_rpt_high_cond_column_fx on wwv_flow_ig_report_highlights ( condition_column_id )
/
create index  ig_rpt_high_cond_comp_col_fx on wwv_flow_ig_report_highlights ( condition_comp_column_id )
/


--==============================================================================
create table wwv_flow_ig_report_aggregates (
    id                          number
                                constraint ig_rpt_agg_pk
                                primary key,
    view_id                     number not null
                                constraint ig_rpt_agg_view_fk
                                references wwv_flow_ig_report_views
                                on delete cascade,
    security_group_id           number not null,
    --
    tooltip                     varchar2(255),
    function                    varchar2(21) not null
                                constraint ig_rpt_agg_function_ck
                                check ( function in ( 'SUM', 'AVG', 'COUNT', 'COUNT_DISTINCT', 'MIN', 'MAX', 'MEDIAN' )),
    --
    column_id                   number
                                constraint ig_rpt_agg_column_fk
                                references wwv_flow_region_columns
                                on delete cascade,
    comp_column_id              number
                                constraint ig_rpt_agg_comp_column_fk
                                references wwv_flow_ig_report_computes
                                on delete cascade,
    --
    show_grand_total            varchar2(1) not null
                                constraint ig_rpt_agg_grand_total_ck
                                check ( show_grand_total in ( 'Y', 'N' )),
    --
    is_enabled                  varchar2(1) not null
                                constraint ig_rpt_agg_is_enabled_ck
                                check ( is_enabled in ( 'Y', 'N' ))
)
/
create index  ig_rpt_agg_view_fx on wwv_flow_ig_report_aggregates ( view_id )
/
create index  ig_rpt_agg_column_fx on wwv_flow_ig_report_aggregates ( column_id )
/
create index  ig_rpt_agg_comp_column_fx on wwv_flow_ig_report_aggregates ( comp_column_id )
/


--==============================================================================
create table wwv_flow_ig_report_chart_cols (
    id                          number
                                constraint ig_rpt_chart_col_pk
                                primary key,
    view_id                     number not null
                                constraint ig_rpt_chart_col_view_fk
                                references wwv_flow_ig_report_views
                                on delete cascade,
    security_group_id           number not null,
    --
    column_type                 varchar2(11) not null
                                constraint ig_rpt_chart_col_type_ck
                                check ( column_type in ( 'SERIES_NAME', 'LABEL', 'VALUE', 'OPEN', 'LOW', 'HIGH', 'CLOSE', 'VOLUME', 'X', 'Y', 'Z', 'TARGET' )),
    column_id                   number
                                constraint ig_rpt_chart_col_column_fk
                                references wwv_flow_region_columns
                                on delete cascade,
    comp_column_id              number
                                constraint ig_rpt_chart_col_comp_col_fk
                                references wwv_flow_ig_report_computes
                                on delete cascade,
    function                    varchar2(21)
                                constraint ig_rpt_chart_col_function_ck
                                check ( function in ( 'SUM', 'AVG', 'COUNT', 'COUNT_DISTINCT', 'APPROX_COUNT_DISTINCT', 'MIN', 'MAX', 'MEDIAN' )),
    --
    sort_order                  number,
    sort_direction              varchar2(4)
                                constraint ig_rpt_chart_col_sort_dir_ck
                                check ( sort_direction in ( 'ASC', 'DESC' )),
    sort_nulls                  varchar2(5)
                                constraint ig_rpt_chart_col_sort_nulls_ck
                                check ( sort_nulls in ( 'FIRST', 'LAST' ))
)
/
alter table wwv_flow_ig_report_chart_cols
  add constraint ig_rpt_chart_col_uk unique ( view_id, column_type )
/
create index  ig_rpt_chart_col_column_fx on wwv_flow_ig_report_chart_cols ( column_id )
/
create index  ig_rpt_chart_col_comp_col_fx on wwv_flow_ig_report_chart_cols ( comp_column_id )
/



--##############################################################################
--#
--# ADVISOR
--#
--##############################################################################
prompt ADVISOR

--==============================================================================
create table wwv_flow_dictionary_views (
    id                        number
                              constraint wwv_flow_dict_views_pk
                              primary key,
    view_name                 varchar2(30)   not null
                                             constraint wwv_flow_dict_views_name_ck
                                             check (upper(view_name)=view_name),
    parent_view_id            number,
    component_type_id         number         constraint wwv_flow_dict_views_compt_fk
                                             /* needed for apex_install_pe_data.sql which deletes all entries */
                                             references wwv_pe_component_types deferrable initially deferred,
    pk_column_name            varchar2(30)   not null
                                             constraint wwv_flow_dict_views_pk_col_ck
                                             check (upper(pk_column_name)=pk_column_name),
    display_expression        varchar2(255)  not null,
    order_expression          varchar2(255)  not null,
    order_seq                 number(2)      not null,
    lookup_or_lov_columns     varchar2(4000) constraint wwv_flow_dict_views_lookup_ck
                                             check (upper(lookup_or_lov_columns)=lookup_or_lov_columns),
    link_url                  varchar2(255)
  );

alter table wwv_flow_dictionary_views
  add constraint wwv_flow_dict_view_name_uk unique (view_name);

alter table wwv_flow_dictionary_views
  add constraint wwv_flow_dict_view_parent_fk foreign key (parent_view_id)
  references wwv_flow_dictionary_views (id);

create index wwv_flow_dict_view_parent_fx on wwv_flow_dictionary_views (parent_view_id);

create index wwv_flow_dict_view_compt_fx on wwv_flow_dictionary_views (component_type_id);

comment on table wwv_flow_dictionary_views is
    'Adding some additional data to the meta data of the APEX Dictionary views.';
-- Add comments to the columns
comment on column wwv_flow_dictionary_views.view_name is
    'Name of the APEX dictionary view.';
comment on column wwv_flow_dictionary_views.parent_view_id is
    'Reference to the parent dictionary view.';
comment on column wwv_flow_dictionary_views.component_type_id is
    'Reference to Page Designer component type.';
comment on column wwv_flow_dictionary_views.pk_column_name is
    'Primary key column in the APEX dictionary view.';
comment on column wwv_flow_dictionary_views.display_expression is
    'Column/Expression which is used to display an entry of that dictionary view.';
comment on column wwv_flow_dictionary_views.order_expression is
    'Column/Expression which is used to order the entries on the same level.';
comment on column wwv_flow_dictionary_views.order_seq is
    'Order sequence within the same hierarchy level.';
comment on column wwv_flow_dictionary_views.link_url is
    'APEX Builder URL to maintain this object. Valid placeholders are %application_id%, %page_id% and %pk_value%';
comment on column wwv_flow_dictionary_views.link_url is
    'Comma delimited list of columns which are ignored by application search.';

--==============================================================================
create table wwv_flow_advisor_categories (
    id                number
                      constraint wwv_flow_adv_cat_pk
                      primary key,
    code              varchar2(30)  not null
                      constraint wwv_flow_adv_cat_code_ck
                      check (upper(code)=code),
    order_seq         number(2)     not null
    );

alter table wwv_flow_advisor_categories
  add constraint wwv_flow_adv_cat_code_uk unique (code);

comment on table wwv_flow_advisor_categories is
    'Contains the different categories of an Advisor Check (e.g. Security, Error, ...).';
comment on column wwv_flow_advisor_categories.code is
    'Code used in program code to reference the record.';
comment on column wwv_flow_advisor_categories.order_seq is
    'Order sequence to display the categories based on there priority.';

--==============================================================================
create table wwv_flow_advisor_checks (
    id                number
                      constraint wwv_flow_adv_chk_pk
                      primary key,
    code              varchar2(30)   not null
                      constraint wwv_flow_adv_chk_code_ck
                      check (upper(code)=code),
    description       varchar2(80),
    help_text         varchar2(2000),
    category_id       number
                      constraint wwv_flow_adv_chk_cat_fk
                      references wwv_flow_advisor_categories(id)
                      on delete cascade,
    is_default        varchar2(1)    not null
                      constraint wwv_flow_adv_chk_is_default_ck
                      check (is_default in ('Y', 'N')),
    is_custom         varchar2(1)    not null
                      constraint wwv_flow_adv_chk_is_custom_ck
                      check (is_custom in ('Y', 'N')),
    order_seq         number(4)      not null,
    check_statement   clob,
    flow_id           number /* no foreign key by intension -> would be dropped if application gets reinstalled */
    );

alter table wwv_flow_advisor_checks
  add constraint wwv_flow_adv_chk_code_uk unique (code);

create index wwv_flow_adv_chk_cat_idx1 on wwv_flow_advisor_checks (category_id);

comment on table wwv_flow_advisor_checks is
    'Contains the meta data of the checks performed by the APEX Advisory module.';
comment on column wwv_flow_advisor_checks.code is
    'Code used in program code to reference the record.';
comment on column wwv_flow_advisor_checks.description is
    'A description for the check.';
comment on column wwv_flow_advisor_checks.help_text is
    'More information what the check does.';
comment on column wwv_flow_advisor_checks.category_id is
    'References a category for the check.';
comment on column wwv_flow_advisor_checks.is_default is
    'If set to true (Y=True, N=False) the check will be checked by default.';
comment on column wwv_flow_advisor_checks.is_custom is
    'If set to true (Y=True, N=False) it indicates that it''s a custom check added to the advisor.';
comment on column wwv_flow_advisor_checks.order_seq is
    'Order sequence to display the categories based on there priority.';
comment on column wwv_flow_advisor_checks.check_statement is
    'SQL Statement to check something. Has to return the following columns
      VIEW_NAME, PK_VALUE, COLUMN_NAME, MESSAGE_CODE, PARAMETER1, PARAMETER2
      The PK_VALUE is used for looking up the record in VIEW_NAME and COLUMN_NAME
      is the column which shows up as having an error. MESSAGE_CODE is an entry in
      WWV_FLOW_ADVISOR_CHECK_MSGS. PARAMETER1 and 2 are placeholders (%1, %2) in
      the defined message.
      The Statement also has to have two bind variables called :p_application_id and
      :p_page_id where the last one is used with NVL(:p_page_id, PAGE_ID)';
comment on column wwv_flow_advisor_checks.flow_id IS
     'References an application id to make the check only available for a specific application. Eg. for QA checks.';

--==============================================================================
create table wwv_flow_advisor_check_msgs (
    id           number
                 constraint wwv_flow_adv_chk_msg_pk
                 primary key,
    check_id     number         not null
                 constraint wwv_flow_adv_chk_msg_check_fk
                 references wwv_flow_advisor_checks(id)
                 on delete cascade,
    code         varchar2(30)   not null
                 constraint wwv_flow_adv_chk_msg_code_ck
                 check (upper(code)=code),
    message_text varchar2(2000),
    help_text    varchar2(2000)
    );

alter table wwv_flow_advisor_check_msgs
  add constraint wwv_flow_adv_chk_msg_code_uk unique (check_id, code);

comment on table wwv_flow_advisor_check_msgs is
    'Contains the possible messages for an advisor check.';
comment on column wwv_flow_advisor_check_msgs.code is
    'Code used in program code to reference the record.';
comment on column wwv_flow_advisor_check_msgs.message_text is
    'Message which is displayed when the check fails. Can contain the placeholders %1 and %2.';
comment on column wwv_flow_advisor_check_msgs.help_text is
    'More information why the check is performed and what should be fixed.';

--##############################################################################
--#
--# USER INTERFACE DEFAULTS
--#
--##############################################################################
prompt USER INTERFACE DEFAULTS

--==============================================================================
create table wwv_flow_hnt_table_info (
    table_id             number        not null
                             constraint wwv_flow_hnt_table_info_pk
                             primary key,
    schema               varchar2(128)  not null,
    table_name           varchar2(128)  not null,
    --
    form_region_title    varchar2(255),
    report_region_title  varchar2(255),
    --
    created_by           varchar2(255)  not null,
    created_on           date          not null,
    last_updated_by      varchar2(255),
    last_updated_on      date,
    --
    title                varchar2(255)
 )
/

alter table wwv_flow_hnt_table_info
    add constraint wwv_flow_hnt_table_info_uk
        unique (schema, table_name)
/

--==============================================================================
create table wwv_flow_hnt_groups (
    group_id            number        not null
                            constraint wwv_flow_hnt_groups_pk
                            primary key,
    table_id            number        not null
                            constraint wwv_flow_hnt_groups_tab_fk
                            references wwv_flow_hnt_table_info
                            on delete cascade,
    group_name          varchar2(255) not null,
    description         varchar2(4000),
    display_sequence    number,
    --
    created_by           varchar2(255)  not null,
    created_on           date          not null,
    last_updated_by      varchar2(255),
    last_updated_on      date
    )
/

alter table wwv_flow_hnt_groups
    add constraint wwv_flow_hnt_groups_uk
        unique (table_id, group_name)
/

--==============================================================================
create table wwv_flow_hnt_column_info (
    column_id           number        not null
                            constraint wwv_flow_hnt_column_info_pk
                            primary key,
    table_id            number        not null
                            constraint wwv_flow_hnt_col_info_tab_fk
                            references wwv_flow_hnt_table_info
                            on delete cascade,
    column_name         varchar2(128)  not null,
    --
    group_id            number
                            constraint wwv_flow_hnt_col_info_grp_fk
                            references wwv_flow_hnt_groups
                            on delete set null,
    label               varchar2(255),
    help_text           varchar2(4000),
    mask_form           varchar2(255),
    display_seq_form    number,
    display_in_form     varchar2(1)
                            constraint wwv_fhc_display_in_form
                            check (display_in_form in ('Y','N')),
    display_as_form     varchar2(50)
                            constraint wwv_fhc_display_as_form
                            check (display_as_form in (
                                'DISPLAY_ONLY_PLSQL',
                                'FILE',
                                'BUTTON',
                                'HIDDEN',
                                'HIDDEN_PROTECTED',
                                'DISPLAY_ONLY_HTML',
                                'STOP_AND_START_HTML_TABLE',
                                'DISPLAY_ONLY_ESCAPE_SC',
                                'IMAGE',
                                'DISPLAY_AND_SAVE',
                                'DISPLAY_AND_SAVE_LOV',
                                'CHECKBOX',
                                'POPUP',
                                'POPUP_FILTER',
                                'POPUP2',
                                'POPUP3',
                                'POPUP4',
                                'POPUP5',
                                'POPUP6',
                                'POPUP_KEY_LOV',
                                'POPUP_KEY_LOV_NOFETCH',
                                'POPUP_COLOR',
                                'PICK_DATE_USING_FLOW_FORMAT_MASK',
                                'PICK_DATE_USING_APP_FORMAT_MASK',
                                'PICK_DATE_USING_APP_DATE_FORMAT',
                                'PICK_DATE_USING_FORMAT_MASK',
                                'PICK_DATE_DD_MON_RR',
                                'PICK_DATE_DD_MON_RR_HH_MI',
                                'PICK_DATE_DD_MON_RR_HH24_MI',
                                'PICK_DATE_DD_MON_YYYY',
                                'PICK_DATE_DD_MON_YYYY_HH_MI',
                                'PICK_DATE_DD_MON_YYYY_HH24_MI',
                                'PICK_DATE_DD_MM_YYYY',
                                'PICK_DATE_DD_MM_YYYY_HH_MI',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI',
                                'PICK_DATE_MM_DD_YYYY',
                                'PICK_DATE_MM_DD_YYYY_HH_MI',
                                'PICK_DATE_MM_DD_YYYY_HH24_MI',
                                'PICK_DATE_DD_MON_YY',
                                'PICK_DATE_DD_MON_YY_HH_MI',
                                'PICK_DATE_DD_MON_YY_HH24_MI',
                                'PICK_DATE_YYYY_MM_DD',
                                'PICK_DATE_YYYY_MM_DD_HH_MI',
                                'PICK_DATE_YYYY_MM_DD_HH24_MI',
                                'PICK_DATE_RR_MON_DD',
                                'PICK_DATE_RR_MON_DD_HH_MI',
                                'PICK_DATE_RR_MON_DD_HH24_MI',
                                'PICK_DATE_DD_MM_YYYY_DASH',
                                'PICK_DATE_DD_MM_YYYY_HH_MI_DASH',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH',
                                'PICK_DATE_DD_MM_YYYY_DOT',
                                'PICK_DATE_DD_MM_YYYY_HH_MI_DOT',
                                'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT',
                                'PICK_DATE_YYYY_DD_MM_DOT',
                                'PICK_DATE_YYYY_DD_MM_HH_MI_DOT',
                                'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT',
                                'PICK_DATE_YYYY_MM_DD_SLASH',
                                'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH',
                                'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH',
                                'DISPLAY_ONLY_LOV',
                                'COMBOBOX',
                                'SHOW_AS_SL_WITH_POPUP',
                                'COMBOBOX_WITH_URL_REDIRECT',
                                'COMBOBOX_WITH_BRANCH_TO_PAGE',
                                'COMBOBOX_WITH_SUBMIT',
                                'COMBOBOX_WITH_JS_POST',
                                'COMBOBOX_WITH_JS_REDIRECT',
                                'MULTIPLESELECT',
                                'SHUTTLE',
                                'RADIOGROUP',
                                'RADIOGROUP2',
                                'RADIOGROUP_WITH_REDIRECT',
                                'RADIOGROUP_WITH_SUBMIT',
                                'RADIOGROUP_WITH_JS_SUBMIT',
                                'TEXTAREA',
                                'TEXTAREA_WITH_SPELL_CHECK',
                                'TEXTAREA-AUTO-HEIGHT',
                                'TEXTAREA_WITH_CONTROLS',
                                'TEXTAREA_WITH_HTML_EDITOR_BASIC',
                                'TEXTAREA_WITH_HTML_EDITOR_STANDARD',
                                'TEXTAREA_CHAR_COUNT',
                                'TEXTAREA_CHAR_COUNT_SPELL',
                                'TEXT',
                                'TEXT_WITH_ENTER_SUBMIT',
                                'TEXT_DISABLED',
                                'TEXT_DISABLED_AND_SAVE',
                                'TEXT_WITH_CALCULATOR',
                                'PASSWORD',
                                'PASSWORD_WITH_ENTER_SUBMIT',
                                'PASSWORD_DNSS',
                                'PASSWORD_WITH_SUBMIT_DNSS',
                                'LIST_MGR',
                                'LIST_MGR2',
                                'LIST_MGR3',
                                'LIST_MGR_VIEW',
                                'LIST_MGR_VIEW2',
                                -- 4.0
                                'SELECT_LIST_WITH_POPUP',
                                'TEXT_WITH_SELECT_LIST',
                                'TEXTAREA_WITH_POPUP'
                                ) or
                                display_as_form like 'NATIVE\_%' escape '\'),
    --
    form_attribute_01   varchar2(4000),
    form_attribute_02   varchar2(4000),
    form_attribute_03   varchar2(4000),
    form_attribute_04   varchar2(4000),
    form_attribute_05   varchar2(4000),
    form_attribute_06   varchar2(4000),
    form_attribute_07   varchar2(4000),
    form_attribute_08   varchar2(4000),
    form_attribute_09   varchar2(4000),
    form_attribute_10   varchar2(4000),
    form_attribute_11   varchar2(4000),
    form_attribute_12   varchar2(4000),
    form_attribute_13   varchar2(4000),
    form_attribute_14   varchar2(4000),
    form_attribute_15   varchar2(4000),
    --
    display_as_tab_form varchar2(255)
                            constraint wwv_fhc_display_as_tab_form
                            check (display_as_tab_form in (
                               'READONLY',
                               'DISPLAY_AND_SAVE',
                               'WITHOUT_MODIFICATION',
                               'ESCAPE_SC',
                               'TEXT',
                               'TEXT_FROM_LOV',
                               'TEXTAREA',
                               'SELECT_LIST',
                               'SELECT_LIST_FROM_LOV',
                               'SELECT_LIST_FROM_QUERY',
                               'CHECKBOX',
                               'SIMPLE_CHECKBOX',
                               'RADIOGROUP',
                               'HIDDEN',
                               'HIDDEN_PROTECTED',
                               'POPUP',
                               'POPUP_QUERY',
                               'DATE_POPUP',
                               'DATE_PICKER',
                               'PICK_DATE_USING_APP_FORMAT_MASK',
                               'PICK_DATE_USING_APP_DATE_FORMAT',
                               'PICK_DATE_USING_FORMAT_MASK',
                               'PICK_DATE_DD_MON_RR',
                               'PICK_DATE_DD_MON_RR_HH_MI',
                               'PICK_DATE_DD_MON_RR_HH24_MI',
                               'PICK_DATE_DD_MON_YYYY',
                               'PICK_DATE_DD_MON_YYYY_HH_MI',
                               'PICK_DATE_DD_MON_YYYY_HH24_MI',
                               'PICK_DATE_DD_MM_YYYY',
                               'PICK_DATE_DD_MM_YYYY_HH_MI',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI',
                               'PICK_DATE_MM_DD_YYYY',
                               'PICK_DATE_MM_DD_YYYY_HH_MI',
                               'PICK_DATE_MM_DD_YYYY_HH24_MI',
                               'PICK_DATE_DD_MON_YY',
                               'PICK_DATE_DD_MON_YY_HH_MI',
                               'PICK_DATE_DD_MON_YY_HH24_MI',
                               'PICK_DATE_YYYY_MM_DD',
                               'PICK_DATE_YYYY_MM_DD_HH_MI',
                               'PICK_DATE_YYYY_MM_DD_HH24_MI',
                               'PICK_DATE_RR_MON_DD',
                               'PICK_DATE_RR_MON_DD_HH_MI',
                               'PICK_DATE_RR_MON_DD_HH24_MI',
                               'PICK_DATE_DD_MM_YYYY_DASH',
                               'PICK_DATE_DD_MM_YYYY_HH_MI_DASH',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH',
                               'PICK_DATE_DD_MM_YYYY_DOT',
                               'PICK_DATE_DD_MM_YYYY_HH_MI_DOT',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT',
                               'PICK_DATE_YYYY_DD_MM_DOT',
                               'PICK_DATE_YYYY_DD_MM_HH_MI_DOT',
                               'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT',
                               'PICK_DATE_YYYY_MM_DD_SLASH',
                               'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH',
                               'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH')),
    display_seq_report  number,
    display_in_report   varchar2(1)
                            constraint wwv_fhc_display_in_report
                            check (display_in_report in ('Y','N')),
    display_as_report   varchar2(255)
                            constraint wwv_fhc_display_as_report
                            check (display_as_report in (
                               'READONLY',
                               'DISPLAY_AND_SAVE',
                               'WITHOUT_MODIFICATION',
                               'ESCAPE_SC',
                               'TEXT',
                               'TEXT_FROM_LOV',
                               'TEXTAREA',
                               'SELECT_LIST',
                               'SELECT_LIST_FROM_LOV',
                               'SELECT_LIST_FROM_QUERY',
                               'CHECKBOX',
                               'SIMPLE_CHECKBOX',
                               'RADIOGROUP',
                               'HIDDEN',
                               'HIDDEN_PROTECTED',
                               'POPUP',
                               'POPUP_QUERY',
                               'DATE_POPUP',
                               'DATE_PICKER',
                               'PICK_DATE_USING_APP_FORMAT_MASK',
                               'PICK_DATE_USING_APP_DATE_FORMAT',
                               'PICK_DATE_USING_FORMAT_MASK',
                               'PICK_DATE_DD_MON_RR',
                               'PICK_DATE_DD_MON_RR_HH_MI',
                               'PICK_DATE_DD_MON_RR_HH24_MI',
                               'PICK_DATE_DD_MON_YYYY',
                               'PICK_DATE_DD_MON_YYYY_HH_MI',
                               'PICK_DATE_DD_MON_YYYY_HH24_MI',
                               'PICK_DATE_DD_MM_YYYY',
                               'PICK_DATE_DD_MM_YYYY_HH_MI',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI',
                               'PICK_DATE_MM_DD_YYYY',
                               'PICK_DATE_MM_DD_YYYY_HH_MI',
                               'PICK_DATE_MM_DD_YYYY_HH24_MI',
                               'PICK_DATE_DD_MON_YY',
                               'PICK_DATE_DD_MON_YY_HH_MI',
                               'PICK_DATE_DD_MON_YY_HH24_MI',
                               'PICK_DATE_YYYY_MM_DD',
                               'PICK_DATE_YYYY_MM_DD_HH_MI',
                               'PICK_DATE_YYYY_MM_DD_HH24_MI',
                               'PICK_DATE_RR_MON_DD',
                               'PICK_DATE_RR_MON_DD_HH_MI',
                               'PICK_DATE_RR_MON_DD_HH24_MI',
                               'PICK_DATE_DD_MM_YYYY_DASH',
                               'PICK_DATE_DD_MM_YYYY_HH_MI_DASH',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH',
                               'PICK_DATE_DD_MM_YYYY_DOT',
                               'PICK_DATE_DD_MM_YYYY_HH_MI_DOT',
                               'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT',
                               'PICK_DATE_YYYY_DD_MM_DOT',
                               'PICK_DATE_YYYY_DD_MM_HH_MI_DOT',
                               'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT',
                               'PICK_DATE_YYYY_MM_DD_SLASH',
                               'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH',
                               'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH')),
    mask_report         varchar2(255),
    aggregate_by        varchar2(1)
                            constraint wwv_fhc_aggregate_by
                            check (aggregate_by in ('Y','N')),
    lov_query           varchar2(4000),
    default_value       varchar2(255),
    required            varchar2(1)
                            constraint wwv_fhc_required
                            check (required in ('Y','N')),
    alignment           varchar2(1)
                            constraint wwv_fhc_alignment
                            check (alignment in ('L','C','R')),
    display_width       number,
    max_width           number,
    height              number,
    group_by            varchar2(1)
                            constraint wwv_fhc_group_by
                            check (group_by in ('Y','N')),
    order_by_seq        number,
    order_by_asc_desc   varchar2(1)
                            constraint wwv_fhc_order_by
                            check (order_by_asc_desc in ('A','D')),
    searchable          varchar2(1)
                            constraint wwv_fhc_searchable
                            check (searchable in ('Y','N')),
    created_by          varchar2(255)  not null,
    created_on          date          not null,
    last_updated_by     varchar2(255),
    last_updated_on     date)
/

alter table wwv_flow_hnt_column_info
    add constraint wwv_flow_hnt_column_info_uk
        unique (table_id, column_name)
/

create index wwv_flow_hnt_column_info_idx on wwv_flow_hnt_column_info (group_id)
/

--==============================================================================
create table wwv_flow_hnt_lov_data (
    id                 number         not null
                           constraint wwv_flow_hnt_lov_data_pk
                           primary key,
    --
    column_id          number         not null
                           constraint wwv_flow_hnt_lov_data_col_fk
                           references wwv_flow_hnt_column_info
                           on delete cascade,
    lov_disp_sequence  number         not null,
    lov_disp_value     varchar2(4000) not null,
    lov_return_value   varchar2(4000) not null,
    last_updated_by    varchar2(255)   not null,
    last_updated_on    date           not null,
    -- 4.0
    created_by         varchar2(255),
    created_on         date)
/

create index wwv_flow_hnt_lov_data_idx on wwv_flow_hnt_lov_data (column_id)
/

--==============================================================================
create table wwv_flow_hnt_column_dict (
    column_id           number        not null
                            constraint wwv_fhcd_pk
                            primary key,
    security_group_id   number not null
                            constraint wwv_fhcd_sgid_fk
                            references wwv_flow_companies(provisioning_company_id)
                            on delete cascade,
    column_name         varchar2(128)  not null,
    --
    label               varchar2(4000),
    help_text           varchar2(4000),
    format_mask         varchar2(255),
    default_value       varchar2(255),
    --
    form_format_mask    varchar2(255),
    form_display_width  number,
    form_display_height number,
    form_data_type      varchar2(30)
                            constraint wwv_fhcd_form_data_type
                            check (form_data_type in (
                            'VARCHAR',
                            'NUMBER',
                            'DATE')),
    --
    report_format_mask    varchar2(4000),
    report_col_alignment  varchar2(10)
                            constraint wwv_fhcd_col_alignment
                            check (report_col_alignment in ('LEFT','CENTER','RIGHT')),
    --
    created_by          varchar2(255)  not null,
    created_on          date           not null,
    last_updated_by     varchar2(255),
    last_updated_on     date)
/
alter table wwv_flow_hnt_column_dict
    add constraint wwv_fhcd_uk
        unique (security_group_id, column_name)
/

--==============================================================================
create table wwv_flow_hnt_col_dict_syn (
    syn_id              number        not null
                            constraint wwv_flow_hnt_col_dict_syn_pk
                            primary key,
    column_id           number        not null
                            constraint wwv_flow_hnt_col_dict_syn_fk
                            references wwv_flow_hnt_column_dict
                            on delete cascade,
    security_group_id   number        not null,
    syn_name            varchar2(128)  not null,
    --
    created_by          varchar2(255) not null,
    created_on          date          not null,
    last_updated_by     varchar2(255),
    last_updated_on     date)
/
alter table wwv_flow_hnt_col_dict_syn
    add constraint wwv_flow_hnt_col_dict_syn_uk
        unique (security_group_id, syn_name)
/

create index wwv_flow_hnt_col_dict_syn_idx1 on wwv_flow_hnt_col_dict_syn (column_id)
/

--==============================================================================
create table wwv_flow_region_chart_ser_attr (
    id                           number
                                 constraint wwv_flow_chart_series_pk
                                 primary key,
    region_id                    number not null
                                 constraint wwv_flow_seattr_to_region_fk
                                 references wwv_flow_page_plugs
                                 on delete cascade,
    series_id                    number not null,
    security_group_id            number not null,
    a001                         clob,
    a002                         varchar2(4000),
    a003                         varchar2(4000),
    a004                         varchar2(4000),
    a005                         varchar2(4000),
    a006                         varchar2(4000),
    a007                         varchar2(4000),
    a008                         varchar2(4000),
    a009                         varchar2(4000),
    a010                         varchar2(4000),
    a011                         varchar2(4000),
    a012                         varchar2(4000),
    a013                         varchar2(4000),
    a014                         varchar2(4000),
    a015                         varchar2(4000),
    a016                         varchar2(4000),
    a017                         varchar2(4000),
    a018                         varchar2(4000),
    a019                         varchar2(4000),
    a020                         varchar2(4000),
    a021                         varchar2(4000),
    a022                         varchar2(4000),
    a023                         varchar2(4000),
    a024                         varchar2(4000),
    a025                         varchar2(4000),
    a026                         varchar2(4000),
    a027                         varchar2(4000),
    a028                         varchar2(4000),
    a029                         varchar2(4000),
    a030                         varchar2(4000),
    a031                         varchar2(4000),
    a032                         varchar2(4000),
    a033                         varchar2(4000),
    a034                         varchar2(4000),
    a035                         varchar2(4000),
    a036                         varchar2(4000),
    a037                         varchar2(4000),
    a038                         varchar2(4000),
    a039                         varchar2(4000),
    a040                         varchar2(4000),
    a041                         varchar2(4000),
    a042                         varchar2(4000),
    a043                         varchar2(4000),
    a044                         varchar2(4000),
    a045                         varchar2(4000),
    a046                         varchar2(4000),
    a047                         varchar2(4000),
    a048                         varchar2(4000),
    a049                         varchar2(4000),
    a050                         varchar2(4000))
/

create unique index wwv_flow_chart_seattr_idx on wwv_flow_region_chart_ser_attr (region_id,series_id)
/

--##############################################################################
--#
--# FEEDBACK
--#
--##############################################################################
prompt FEEDBACK

--==============================================================================
create table  wwv_flow_feedback (
    id                                number,
    feedback_id                       number,
    feedback_comment                  varchar2(4000),
    feedback_type                     number,
    feedback_rating                   number,
    --
    feedback_status                   number,
    developer_comment                 varchar2(4000),
    public_response                   varchar2(4000),
    --
    application_id                    number,
    application_name                  varchar2(255),
    page_id                           number,
    page_name                         varchar2(255),
    page_last_updated_by              varchar2(255),
    page_last_updated_on              date,
    session_id                        varchar2(255),
    apex_user                         varchar2(255),
    user_email                        varchar2(255),
    application_version               varchar2(255),
    session_info                      varchar2(4000),
    session_state                     varchar2(4000),
    parsing_schema                    varchar2(255),
    --
    screen_width                      varchar2(30),
    screen_height                     varchar2(30),
    http_user_agent                   varchar2(4000),
    remote_addr                       varchar2(4000),
    remote_user                       varchar2(4000),
    http_host                         varchar2(4000),
    server_name                       varchar2(4000),
    server_port                       varchar2(4000),
    tags                              varchar2(4000),
    --
    label_01                          varchar2(4000),
    label_02                          varchar2(4000),
    label_03                          varchar2(4000),
    label_04                          varchar2(4000),
    label_05                          varchar2(4000),
    label_06                          varchar2(4000),
    label_07                          varchar2(4000),
    label_08                          varchar2(4000),
        --
    attribute_01                      varchar2(4000),
    attribute_02                      varchar2(4000),
    attribute_03                      varchar2(4000),
    attribute_04                      varchar2(4000),
    attribute_05                      varchar2(4000),
    attribute_06                      varchar2(4000),
    attribute_07                      varchar2(4000),
    attribute_08                      varchar2(4000),
    --
    bug_id                            number,
    feature_id                        number,
    task_id                           number,
    issue_id  					      number,
    --
    logging_email                     varchar2(255),
    logging_security_group_id         number,
    logged_by_workspace_name          varchar2(255),
    deployment_system                 varchar2(8),
    security_group_id                 number not null
                                      constraint wwv_flow_feedback_fk
                                      references wwv_flow_companies(provisioning_company_id)
                                      on delete cascade,
    created_by                        varchar2(255) not null enable,
    created_on                        timestamp with time zone not null,
    updated_by                        varchar2(255),
    updated_on                        timestamp with time zone,
    constraint wwv_flow_feedback_pk   primary key (id) enable
)
/

create index wwv_flow_feedback_idx1 on wwv_flow_feedback (security_group_id);
create index wwv_flow_feedback_idx2 on wwv_flow_feedback (application_id, page_id);
create index wwv_flow_feedback_idx3 on wwv_flow_feedback (bug_id);
create index wwv_flow_feedback_idx4 on wwv_flow_feedback (feature_id);
create index wwv_flow_feedback_idx5 on wwv_flow_feedback (task_id);
create index wwv_flow_feedback_idx6 on wwv_flow_feedback (feedback_id);
create index wwv_flow_feedback_idx7 on wwv_flow_feedback (issue_id);

--==============================================================================
create table wwv_flow_feedback_followup (
    id                               number,
    feedback_id                      constraint wwv_flow_feedback_fup_fk
                                     references wwv_flow_feedback(id)
                                     on delete cascade,
    follow_up                        varchar2(4000) not null,
    email                            varchar2(255),
    security_group_id                number not null
                                     constraint wwv_flow_feedback_fup_fk2
                                     references wwv_flow_companies(provisioning_company_id)
                                     on delete cascade,
    created_by                       varchar2(255) not null enable,
    created_on                       timestamp with time zone not null,
    updated_by                       varchar2(255),
    updated_on                       timestamp with time zone,
    constraint wwv_flow_feedback_fup_pk primary key (id) enable
)
/

create index wwv_flow_feedback_fup_idx1 on wwv_flow_feedback_followup (feedback_id);
create index wwv_flow_feedback_fup_idx2 on wwv_flow_feedback_followup (security_group_id);

--##############################################################################
--#
--# MAIL QUEUE
--#
--##############################################################################
prompt MAIL QUEUE

--==============================================================================
create table wwv_flow_mail_queue
(
  id number constraint wwv_flow_mail_queue_pk
                       primary key,
  mail_to              varchar2(4000),
  mail_from            varchar2(4000),
  mail_replyto         varchar2(4000),
  mail_subj            varchar2(4000),
  mail_cc              varchar2(4000),
  mail_bcc             varchar2(4000),
  mail_body            clob,
  mail_body_html       clob,
  mail_send_count      number(2),
  mail_send_error      varchar2(4000),
  mail_message_created date,
  includes_html        number(1) default 0,
  last_updated_by      varchar2(255),
  last_updated_on      date,
  flow_id              number,
  security_group_id    number not null
)
/

create index wwv_flow_mail_queue_idx1 on wwv_flow_mail_queue ( security_group_id )
/

--==============================================================================
create table wwv_flow_mail_attachments
(
  id number constraint wwv_flow_mail_attachments_pk
                       primary key,
  mail_id              number not null
                           constraint wwv_flow_mail_attachments_fk1
                           references wwv_flow_mail_queue(id)
                           on delete cascade,
  filename             varchar2(4000) not null,
  mime_type            varchar2(255)   not null,
  inline               varchar2(1)    not null
                           constraint wwv_flow_mail_attachments_ck1
                           check (inline in ('Y','N')),
  content_id           varchar2(255),
  attachment           blob,
  last_updated_by      varchar2(255),
  last_updated_on      date,
  security_group_id    number not null
)
/

create index wwv_flow_mail_attachments_idx1 on wwv_flow_mail_attachments ( mail_id, security_group_id )
/

--==============================================================================
create table wwv_flow_mail_log(
    mail_id                 number,
    mail_to                 varchar2(4000),
    mail_from               varchar2(4000),
    mail_replyto            varchar2(4000),
    mail_subj               varchar2(4000),
    mail_cc                 varchar2(4000),
    mail_bcc                varchar2(4000),
    mail_send_error         varchar2(4000),
    mail_message_id         varchar2(4000),
    mail_message_created    date,
    mail_message_send_begin date,
    mail_message_send_end   date,
    mail_body_size          number,
    mail_body_html_size     number,
    mail_attachment_count   number,
    mail_attachment_size    number,
    last_updated_by         varchar2(255),
    last_updated_on         date,
    flow_id                 number,
    security_group_id       number not null
    )
    pctfree 0
/


create index wwv_flow_mail_log_idx1 on wwv_flow_mail_log ( substr(mail_to,1,500) )
/

create index wwv_flow_mail_log_idx2 on wwv_flow_mail_log ( last_updated_by )
/

create index wwv_flow_mail_log_idx3 on wwv_flow_mail_log ( security_group_id, last_updated_on )
/


--##############################################################################
--#
--# LOGS
--#
--##############################################################################
prompt LOGS

--==============================================================================
create table wwv_flow_log_numbers (
    current_log_number     number not null
                           constraint wwv_flow_log_number_ck1
                           check (current_log_number in (1,2)),
    log_switched_date      date   not null,
    log_switch_after_days  number not null
                           constraint wwv_flow_log_numbers_ck2
                           check (log_switch_after_days between 1 and 180),
    log_name               varchar2(255) not null
                           constraint wwv_flow_log_numbers_ck3
                           check (log_name in ( 'ACCESS',
                                                'ACTIVITY',
                                                'CLICKTHRU',
                                                'DEBUG',
                                                'WEBSERVICE',
                                                'WEBSOURCESYNC',
                                                'AUTOMATION',
                                                'KILLED_SESSIONS' )),
    constraint wwv_flow_log_numbers_pk primary key( log_name )
    )
    organization index
/

alter table wwv_flow_log_numbers add constraint wwv_flow_log_numbers_ck4
    check (trunc(abs(log_switch_after_days)) = log_switch_after_days)
/

--==============================================================================
create table wwv_flow_auto_file_delete_log(
    id                     number
                           constraint wwv_flow_auto_file_del_log_pk
                           primary key,
    security_group_id      number             not null,
    workspace_name         varchar2(4000)     not null,
    workspace_display_name varchar2(4000),
    file_name              varchar2(4000)     not null,
    file_id                number             not null,
    file_type              varchar2(4000),
    file_size              number,
    deleted_on             date )
/

--==============================================================================
-- Duplicate Page Submision Prevension
create table wwv_flow_page_submissions (
    submit_id          number
                       constraint wwv_flow_page_submissions_pk
                       primary key,
    security_group_id  number not null,
    flow_id            number not null
                       constraint wwv_flow_page_sub_fk
                       references wwv_flows on delete cascade,
    page_id            number,
    session_id         number
                       constraint wwv_flow_page_sub_sess_fk
                       references wwv_flow_sessions$ on delete cascade
    )
    pctfree 0
/

create index wwv_flow_page_sub_idx1 on wwv_flow_page_submissions (flow_id)
/
create index wwv_flow_page_sub_idx2 on wwv_flow_page_submissions (session_id)
/

--==============================================================================
create table wwv_flow_clickthru_log$ (
    clickdate           date,
    category            varchar2(255),
    id                  number,
    flow_user           varchar2(255),
    ip                  varchar2(30),
    security_group_id   number not null
    )
    pctfree 0
/

--==============================================================================
create table wwv_flow_clickthru_log2$ (
    clickdate           date,
    category            varchar2(255),
    id                  number,
    flow_user           varchar2(255),
    ip                  varchar2(30),
    security_group_id   number not null
    )
    pctfree 0
/

--==============================================================================
create table wwv_flow_builder_audit_trail (
   id                        number
                             constraint wwv_flow_bld_audit_pk
                             primary key,
   audit_date                date,
   audit_action              varchar2(1)
                             constraint wwv_flow_bld_audit_ck_a
                             check (audit_action in ('I','U','D')),
   flow_table                varchar2(30),
   flow_table_pk             number,
   flow_user                 varchar2(255),
   flow_id                   number,
   page_id                   number,
   security_group_id         number not null,
   scn                       number,
   audit_comment             varchar2(255),
   object_name               varchar2(4000))
   pctfree 0
/

create index wwv_flow_build_audit_trail_i3 on wwv_flow_builder_audit_trail (flow_id, security_group_id) compress
/

create index wwv_flow_build_audit_trail_i5 on wwv_flow_builder_audit_trail (security_group_id,flow_user,flow_id,audit_date) compress 3
/

create index wwv_flow_build_audit_trail_i6 on wwv_flow_builder_audit_trail (audit_date)
/


--==============================================================================
-- F L O W   A C T I V I T Y   L O G G I N G
--
-- logging can be disabled
-- you may not wish to maintain indexes on log tables for active applications
--

--==============================================================================
-- IF YOU CHANGE THIS TABLE OR ITS INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--
-- security group id for logs is not set via a trigger for performance reasons
--==============================================================================
create table wwv_flow_activity_log1$ (
    time_stamp_tz          timestamp with time zone not null,
    time_stamp             date generated always as (
                             cast(time_stamp_tz as date)
                           ) virtual not null,
    elap                   number,
    num_rows               number,
    userid                 varchar2(255),
    flow_id                number,
    step_id                number,
    session_id             number,
    security_group_id      number not null,
    ecid                   varchar2(64),
    tenant_id              varchar2(32),
    page_mode              varchar2(1),
    page_view_type         number
                           constraint wwv_flow_activity_log1$ck1 check(page_view_type between 0 and 5),
    cached_regions         number,
    content_length         number,
    request_value          varchar2(4000),
    sqlerrm                varchar2(4000),
    sqlerrm_component_type varchar2(255),
    sqlerrm_component_name varchar2(255),
    debug_page_view_id     number,
    --
    ip_address             varchar2(4000),
    user_agent             varchar2(4000),
    worksheet_id           number,         -- IR logging set for interactive report regions
    ir_search              varchar2(4000), -- IR logging set for interactive report regions
    ir_report_id           number          -- IR logging set for interactive report regions
    )
    initrans 8
    pctfree  0
/

create index wwv_flow_activity_log1$_idx1 on wwv_flow_activity_log1$ (
    time_stamp )
    initrans 8
    compress
/

create index wwv_flow_activity_log1$_idx2 on wwv_flow_activity_log1$ (
    security_group_id,
    flow_id )
    initrans 8
    compress 2
/

--==============================================================================
-- IF YOU CHANGE THIS TABLE OR IT'S INDEXES
-- do not forget to synchronize ../utilities/apxpart.sql
--==============================================================================
create table wwv_flow_activity_log2$ (
    time_stamp_tz          timestamp with time zone not null,
    time_stamp             date generated always as (
                             cast(time_stamp_tz as date)
                           ) virtual not null,
    elap                   number,
    num_rows               number,
    userid                 varchar2(255),
    flow_id                number,
    step_id                number,
    session_id             number,
    security_group_id      number not null,
    ecid                   varchar2(64),
    tenant_id              varchar2(32),
    page_mode              varchar2(1),
    page_view_type         number
                           constraint wwv_flow_activity_log2$ck1 check(page_view_type between 0 and 5),
    cached_regions         number,
    content_length         number,
    request_value          varchar2(4000),
    sqlerrm                varchar2(4000),
    sqlerrm_component_type varchar2(255),
    sqlerrm_component_name varchar2(255),
    debug_page_view_id     number,
    --
    ip_address             varchar2(4000),
    user_agent             varchar2(4000),
    worksheet_id           number,         -- IR logging set for interactive report regions
    ir_search              varchar2(4000), -- IR logging set for interactive report regions
    ir_report_id           number          -- IR logging set for interactive report regions
    )
    initrans 8
    pctfree 0
/

create index wwv_flow_activity_log2$_idx1 on wwv_flow_activity_log2$ (
    time_stamp )
    initrans 8
    compress
/

create index wwv_flow_activity_log2$_idx2 on wwv_flow_activity_log2$ (
    security_group_id,
    flow_id )
    initrans 8
    compress 2
/

--==============================================================================
-- Access Log
--
create table wwv_flow_user_access_log1$ (
    login_name             varchar2(255) not null,
    authentication_method  varchar2(255),
    application            number,
    owner                  varchar2(255),
    access_date            date,
    ip_address             varchar2(255),
    remote_user            varchar2(255),
    authentication_result  number, -- 0:okay,1:invalid user,2:account locked,3:account expired,4:password mismatch,5:pw first use,6:attempts exceeded
    custom_status_text     varchar2(4000),
    security_group_id      number not null
    )
    pctfree 0
/
create index wwv_flow_user_access_log1$_ix1 on  wwv_flow_user_access_log1$ (
    security_group_id,
    application,
    access_date )
    compress 2
/

--==============================================================================
create table wwv_flow_user_access_log2$ (
    login_name             varchar2(255) not null,
    authentication_method  varchar2(255),
    application            number,
    owner                  varchar2(255),
    access_date            date,
    ip_address             varchar2(255),
    remote_user            varchar2(255),
    authentication_result  number, -- 0:okay,1:invalid user,2:account locked,3:account expired,4:password mismatch,5:pw first use,6:attempts exceeded
    custom_status_text     varchar2(4000),
    security_group_id      number not null
    )
    pctfree 0
/
create index wwv_flow_user_access_log2$_ix1 on  wwv_flow_user_access_log2$ (
    security_group_id,
    application,
    access_date )
    compress 2
/

--==============================================================================
create table wwv_flow_debug_messages (
    id                            number,
    page_view_id                  number,
    message_timestamp             timestamp with time zone not null,
    message                       varchar2(4000),
    flow_id                       number,
    page_id                       number,
    session_id                    number not null,
    apex_user                     varchar2(255),
    elap                          number,
    message_level                 number,
    security_group_id             number not null,
    call_stack                    varchar2(4000),
    constraint wwv_flow_debug_messages_pk primary key (
        id )
    )
    initrans 8
    pctfree  0
/
alter  index wwv_flow_debug_messages_pk rebuild initrans 8;
create index wwv_flow_debug_messages_idx1 on wwv_flow_debug_messages (session_id)        compress initrans 8;
create index wwv_flow_debug_messages_idx2 on wwv_flow_debug_messages (security_group_id) compress initrans 8;
create index wwv_flow_debug_messages_idx3 on wwv_flow_debug_messages (page_view_id)      compress initrans 8;

create table wwv_flow_debug_messages2 (
    id                            number,
    page_view_id                  number,
    message_timestamp             timestamp with time zone not null,
    message                       varchar2(4000),
    flow_id                       number,
    page_id                       number,
    session_id                    number not null,
    apex_user                     varchar2(255),
    elap                          number,
    message_level                 number,
    security_group_id             number not null,
    call_stack                    varchar2(4000),
    constraint wwv_flow_debug_messages2_pk primary key (
        id )
    )
    initrans 8
    pctfree  0
/

alter  index wwv_flow_debug_messages2_pk rebuild initrans 8;
create index wwv_flow_debug_messages2_idx1 on wwv_flow_debug_messages2 (session_id)        compress initrans 8;
create index wwv_flow_debug_messages2_idx2 on wwv_flow_debug_messages2 (security_group_id) compress initrans 8;
create index wwv_flow_debug_messages2_idx3 on wwv_flow_debug_messages2 (page_view_id)      compress initrans 8;

--==============================================================================
create table wwv_flow_webservice_log1$ (
    url                 varchar2(4000),
    http_method         varchar2(30),
    web_src_module_id   number,
    web_src_sync_id     number,
    content_length      number,
    status_code         number,
    resp_content_length number,
    elapsed_sec         number,
    request_date        date,
    flow_user           varchar2(255),
    security_group_id   number not null
)
/

create index wwv_flow_webservice_log1$_idx1 on wwv_flow_webservice_log1$ (request_date)
    initrans 8
    compress
/

create index wwv_flow_webservice_log1$_idx2 on wwv_flow_webservice_log1$ (security_group_id, web_src_module_id, request_date)
    initrans 8
    compress
/

create table wwv_flow_webservice_log2$ (
    url                 varchar2(4000),
    http_method         varchar2(30),
    web_src_module_id   number,
    web_src_sync_id     number,
    content_length      number,
    status_code         number,
    resp_content_length number,
    elapsed_sec         number,
    request_date        date,
    flow_user           varchar2(255),
    security_group_id   number not null
)
/

create index wwv_flow_webservice_log2$_idx1 on wwv_flow_webservice_log2$ (request_date)
    initrans 8
    compress
/

create index wwv_flow_webservice_log2$_idx2 on wwv_flow_webservice_log2$ (security_group_id, web_src_module_id, request_date)
    initrans 8
    compress
/

--==============================================================================
create table wwv_flow_websource_sync_log1$ (
    synchronization_id      number,
    synchronization_run_id  number,
    flow_id                 number,
    web_src_module_id       number,
    web_src_sync_static_id  varchar2(255),
    sync_to_table_owner     varchar2(128),
    sync_to_table_name      varchar2(128),
    is_job                  varchar2(1) not null
                            constraint wwv_websrc_synclog1_isjob_ck
                            check( is_job in ( 'Y', 'N' ) ),
    status                  varchar2(19)
                            constraint wwv_websrc_synclog1_status_ck
                            check ( status in ( 'SUCCESS', 'INCOMPLETE', 'FAILURE', 'ABORTED', 'JOB_ALREADY_RUNNING' ) ),
    start_timestamp         timestamp with time zone,
    end_timestamp           timestamp with time zone,
    processed_row_count     number,
    messages                varchar2(4000),
    security_group_id       number not null
)
initrans 8
pctfree  0
/

create index wwv_websrc_sync_log1$_strtts on wwv_flow_websource_sync_log1$ ( start_timestamp )
    initrans 8
    compress
/

create index wwv_websrc_sync_log1$_lastsync on wwv_flow_websource_sync_log1$ ( web_src_module_id, status, start_timestamp )
    initrans 8
    compress
/

create index wwv_websrc_sync_log1$_sgflid on wwv_flow_websource_sync_log1$ (security_group_id, flow_id)
    initrans 8
    compress
/

create table wwv_flow_websource_sync_log2$ (
    synchronization_id      number,
    synchronization_run_id  number,
    flow_id                 number,
    web_src_module_id       number,
    web_src_sync_static_id  varchar2(255),
    sync_to_table_owner     varchar2(128),
    sync_to_table_name      varchar2(128),
    is_job                  varchar2(1) not null
                            constraint wwv_websrc_synclog2_isjob_ck
                            check( is_job in ( 'Y', 'N' ) ),
    status                  varchar2(19)
                            constraint wwv_websrc_synclog2_status_ck
                            check ( status in ( 'SUCCESS', 'INCOMPLETE', 'FAILURE', 'ABORTED', 'JOB_ALREADY_RUNNING' ) ),
    start_timestamp         timestamp with time zone,
    end_timestamp           timestamp with time zone,
    processed_row_count     number,
    messages                varchar2(4000),
    security_group_id       number not null
)
initrans 8
pctfree  0
/

create index wwv_websrc_sync_log2$_strtts on wwv_flow_websource_sync_log2$ ( start_timestamp )
    initrans 8
    compress
/

create index wwv_websrc_sync_log2$_lastsync on wwv_flow_websource_sync_log2$ ( web_src_module_id, status, start_timestamp )
    initrans 8
    compress
/

create index wwv_websrc_sync_log2$_sgflid on wwv_flow_websource_sync_log2$ (security_group_id, flow_id)
    initrans 8
    compress
/


--==============================================================================
create table wwv_flow_automation_log1$ (
    id                   number not null,
    flow_id              number not null,
    automation_id        number not null,
    is_job               varchar2(1) not null
                         constraint wwv_am_log1_is_job_ck
                         check( is_job in ( 'Y', 'N' ) ),
    status               varchar2(19) not null
                         constraint wwv_am_log1_status_ck
                         check ( status in ( 'RUNNING', 'SUCCESS', 'NO_ACTION', 'INCOMPLETE', 'FAILURE', 'ABORTED', 'JOB_ALREADY_RUNNING' ) ),
    start_timestamp      timestamp with time zone not null,
    end_timestamp        timestamp with time zone,
    successful_row_count number,
    error_row_count      number,
    security_group_id    number not null
)
initrans 8
pctfree  0
/

create index wwv_am_log1$_strtts on wwv_flow_automation_log1$ ( start_timestamp )
    initrans 8
    compress
/

create index wwv_am_log1$_lastsync on wwv_flow_automation_log1$ ( automation_id, status, start_timestamp )
    initrans 8
    compress
/

create index wwv_am_log1$_sgid_flow on wwv_flow_automation_log1$ (security_group_id, flow_id)
    initrans 8
    compress
/

create table wwv_flow_automation_log2$ (
    id                   number not null,
    flow_id              number not null,
    automation_id        number not null,
    is_job               varchar2(1) not null
                         constraint wwv_am_log2_is_job_ck
                         check( is_job in ( 'Y', 'N' ) ),
    status               varchar2(19) not null
                         constraint wwv_am_log2_status_ck
                         check ( status in ( 'RUNNING', 'SUCCESS', 'NO_ACTION', 'INCOMPLETE', 'FAILURE', 'ABORTED', 'JOB_ALREADY_RUNNING' ) ),
    start_timestamp      timestamp with time zone not null,
    end_timestamp        timestamp with time zone,
    successful_row_count number,
    error_row_count      number,
    security_group_id    number not null
)
initrans 8
pctfree  0
/

create index wwv_am_log2$_strtts on wwv_flow_automation_log2$ ( start_timestamp )
    initrans 8
    compress
/

create index wwv_am_log2$_lastsync on wwv_flow_automation_log2$ ( automation_id, status, start_timestamp )
    initrans 8
    compress
/

create index wwv_am_log2$_sgid_flow on wwv_flow_automation_log2$ (security_group_id, flow_id)
    initrans 8
    compress
/

--==============================================================================
create table wwv_flow_automation_msg_log1$ (
    log_id            number not null,
    automation_id     number not null,
    action_id         number,
    message_type      varchar2(1) not null
                      constraint wwv_am_msg_log1_type_ck
                      check ( message_type in ( 'I', 'W', 'E' ) ),
    message           varchar2(4000) not null,
    pk_value          varchar2(4000),
    message_timestamp timestamp with time zone not null,
    security_group_id number not null
)
initrans 8
pctfree  0
/

create index wwv_am_msg_log1$_parent on wwv_flow_automation_msg_log1$ ( log_id, message_timestamp )
    initrans 8
    compress
/

create index wwv_am_msg_log1$_sgid on wwv_flow_automation_msg_log1$ (security_group_id)
    initrans 8
    compress
/

create table wwv_flow_automation_msg_log2$ (
    log_id            number not null,
    automation_id     number not null,
    action_id         number,
    message_type      varchar2(1) not null
                      constraint wwv_am_msg_log2_type_ck
                      check ( message_type in ( 'I', 'W', 'E' ) ),
    message           varchar2(4000) not null,
    pk_value          varchar2(4000),
    message_timestamp timestamp with time zone not null,
    security_group_id number not null
)
initrans 8
pctfree  0
/

create index wwv_am_msg_log2$_parent on wwv_flow_automation_msg_log2$ ( log_id, message_timestamp )
    initrans 8
    compress
/

create index wwv_am_msg_log2$_sgid on wwv_flow_automation_msg_log2$ (security_group_id)
    initrans 8
    compress
/

--==============================================================================
create table wwv_flow_log_history (
    log_day               date,            -- identifies day
    workspace_id          number,          -- identifies workspace
    workspace_name        varchar2(255),   -- workspace name
    application_id        number,          -- identifies application
    application_name      varchar2(255),   -- application name
    --
    page_events           number,          -- total page events of all type
    page_views            number,          -- count of page gets
    page_accepts          number,          -- count of page posts
    partial_page_views    number,          -- count of partial page requests
    websheet_views        number,          -- count of views of websheet applications
    rows_fetched          number,          -- sum of rows fetched
    ir_searches           number,          -- count of events that are interactive report filtered executions
    --
    distinct_pages        number,          -- count of distinct pages viewed
    distinct_users        number,          -- count of distinct usernames
    distinct_sessions     number,          -- count of distinct sessions
    --
    average_render_time   number,          -- averge elapsed page generation time
    median_render_time    number,          -- median  elapsed page generation time
    maximum_render_time   number,          -- maximum elapsed page generation time
    total_render_time     number,          -- total elapsed page generation time
    content_length        number,          -- total content size in bytes
    error_count           number,          -- number of page events that raised errors
    --
    public_page_events     number,         -- total page events for public pages
    workspace_login_events number,         -- total page events for APEX Workspace Login (4550:1)
    additional_attributes  varchar2(4000)  -- additional attributes in JSON format
    )
    pctfree 0
/

create unique index wwv_flow_log_history_idx1 on wwv_flow_log_history (workspace_id, application_id, log_day) compress;

create index wwv_flow_log_history_idx2 on wwv_flow_log_history (log_day, workspace_id) compress;

--==============================================================================
declare
    procedure ddl (
        p_num in number,
        p_sql in varchar2 )
    is
        l_sql varchar2(32767) := replace (
                                     p_sql,
                                     '#NUM#',
                                     p_num );
    begin
        execute immediate l_sql;
    exception when others then
        sys.dbms_output.put_line(l_sql);
        sys.dbms_output.put_line(sqlerrm);
    end ddl;
begin
    for i in 1 .. 2 loop
        ddl (
            p_num => i,
            p_sql => 'create table wwv_flow_killed_db_sessions#NUM#$ (
    time_stamp           date not null,
    --
    inst_id              number not null,
    sid                  number not null,
    serial#              number not null,
    --
    module               varchar2(64),
    action               varchar2(64),
    client_info          varchar2(64),
    client_identifier    varchar2(64),
    sql_id               varchar2(13),
    sql_child_number     number,
    plsql_object_id      number,
    plsql_subprogram_id  number,
    --
    event                varchar2(64),
    p1                   number,
    p2                   number,
    p3                   number,
    --
    blocking_instance    number,
    blocking_session     number,
    --
    last_call_et         number,
    wait_time_micro      number,
    time_remaining_micro number
)');
        ddl (
            p_num => i,
            p_sql => 'create index wwv_flow_killed_db_sess#NUM#$ix1
on wwv_flow_killed_db_sessions#NUM#$(time_stamp)');
    end loop;
end;
/

--##############################################################################
--#
--# PURGE
--#
--##############################################################################
prompt PURGE

--==============================================================================
create table wwv_purge_workspaces(
    id                 number not null,
    security_group_id  number not null,
    workspace_name     varchar2(255) not null,
    admin_email        varchar2(255),
    status             varchar2(50)
                       constraint wwv_purge_workspaces_ck1
                       check (status in ('NEW', 'STAGE1', 'STAGE2', 'RETAIN', 'PURGING', 'PURGED')),
    created_ts         timestamp,
    updated_ts         timestamp,
    date_to_purge      date,
    purged_ts          timestamp )
/

alter table wwv_purge_workspaces add constraint wwv_purge_workspaces_pk
    primary key( id, security_group_id )
/


create unique index wwv_purge_workspaces_idx1 on wwv_purge_workspaces ( security_group_id )
/
create index wwv_purge_workspaces_idx2 on wwv_purge_workspaces ( status )
/
create index wwv_purge_workspaces_idx3 on wwv_purge_workspaces ( date_to_purge )
/
create index wwv_purge_workspaces_idx4 on wwv_purge_workspaces ( created_ts )
/

--==============================================================================
create table wwv_purge_log(
    id                 number primary key,
    workspace_name     varchar2(255) not null,
    purge_workspace_id number not null,
    security_group_id  number not null,
    created_ts         timestamp,
    log_code           varchar2(100)
                       constraint wwv_purge_log_ck1 check(log_code in ('NEW','DELETE','EMAIL','RESPONSE_RETAIN',
                           'RESPONSE_PURGE','MOVED_TO_NEW','EMAIL_COMPLETE','REMINDER_EMAIL_COMPLETE','BEGIN_PURGE','PURGE_SCHEMA_EVAL',
                           'PURGE_TABLESPACE_EVAL','PURGE_COMPLETE', 'ERROR_DROP_USER', 'ERROR_DROP_TABLESPACE', 'BEGIN_DROP_USER',
                           'END_DROP_USER', 'BEGIN_DROP_TABLESPACE', 'END_DROP_TABLESPACE', 'BEGIN_DELETE_WORKSPACE',
                           'END_DELETE_WORKSPACE', 'ERROR_DELETE_WORKSPACE')),
    log_action         varchar2(4000),
    log_statement      varchar2(4000),
    log_error          varchar2(4000) )
/

create index wwv_purge_log_idx1 on wwv_purge_log ( security_group_id, log_code, purge_workspace_id )
/

create index wwv_purge_log_idx2 on wwv_purge_log ( log_code, created_ts )
/

--==============================================================================
create table wwv_purge_emails(
    id                 number primary key,
    purge_workspace_id number not null,
    security_group_id  number not null,
    email              varchar2(255) not null,
    created_ts         timestamp )
/

alter table wwv_purge_emails
    add constraint wwv_purge_emails_fk1
    foreign key( purge_workspace_id, security_group_id ) references wwv_purge_workspaces( id, security_group_id )
    on delete cascade
/

create index wwv_purge_emails_idx1 on wwv_purge_emails ( purge_workspace_id, security_group_id, created_ts )
/

--==============================================================================
create table wwv_purge_workspace_responses(
    id                 number primary key,
    purge_email_id     number not null,
    response_code      varchar2(20) not null,  -- PURGE, SAVE
    created_ts         timestamp )
/

alter table wwv_purge_workspace_responses
    add constraint wwv_purge_workspace_resp_fk1
    foreign key( purge_email_id ) references wwv_purge_emails( id )
    on delete cascade
/


create index wwv_purge_workspace_resp_idx1 on wwv_purge_workspace_responses ( purge_email_id, created_ts )
/

--==============================================================================
create table wwv_purge_datafiles(
    id                 number primary key,
    purge_workspace_id number        not null,
    security_group_id  number        not null,
    tablespace_name    varchar2(30)  not null,
    file_name          varchar2(513) not null,
    bytes              number        not null,
    status             varchar2(30)  not null
                       constraint wwv_purge_datafiles_ck1 check(status in('NOT_EVAL','QUOTA_BY_OTHER','OTHER_WORKSPACE','NOT_FLOW_TABLESPACE','PURGE')),
    created_ts         timestamp )
/

alter table wwv_purge_datafiles
    add constraint wwv_purge_datafiles_fk1
    foreign key( purge_workspace_id, security_group_id ) references wwv_purge_workspaces( id, security_group_id )
    on delete cascade
/

create index wwv_purge_datafiles_idx1 on wwv_purge_datafiles ( purge_workspace_id, security_group_id  )
/

--==============================================================================
create table wwv_purge_schemas(
    id                 number primary key,
    purge_workspace_id number        not null,
    security_group_id  number        not null,
    schema             varchar2(128) not null,
    status             varchar2(30)  not null
                       constraint wwv_purge_schemas_ck1 check(status in('NOT_EVAL','INTERNAL','OTHER_WORKSPACE','RESTRICTED','NOT_FLOW_DEFAULT_TABLESPACE','PURGE')),
    created_ts         timestamp )
/

alter table wwv_purge_schemas
    add constraint wwv_purge_schemas_fk1
    foreign key( purge_workspace_id, security_group_id ) references wwv_purge_workspaces( id, security_group_id )
    on delete cascade
/

create index wwv_purge_schemas_idx1 on wwv_purge_schemas ( purge_workspace_id, security_group_id  )
/

--##############################################################################
--#
--# TEAM DEVELOPMENT
--#
--##############################################################################
prompt TEAM DEVELOPMENT
--==============================================================================
create table wwv_flow_team_tags (
   id                                number,
   tag                               varchar2(255) not null enable,
   tag_type                          varchar2(30)
                                     constraint wwv_flow_team_tags_tp
                                     check (tag_type in ('BUG','FEATURE','FEEDBACK','MILESTONE','DOCUMENT', 'LINK',  'TODO')),
   component_id                      number,
   --
   security_group_id                 number not null
                                     constraint wwv_flow_team_tags_fk
                                      references wwv_flow_companies(provisioning_company_id)
                                      on delete cascade,
   constraint wwv_flow_team_tags_pk  primary key (id) enable
)
/

create index wwv_flow_team_tags_idx1 on wwv_flow_team_tags (security_group_id,tag)
/
create unique index wwv_flow_team_tags_idx2 on wwv_flow_team_tags (component_id,tag)
/
create index wwv_flow_team_tags_idx3 on wwv_flow_team_tags (tag)
/

--##############################################################################
--#
--# SQL WORKSHOP > QUERY BUILDER
--#
--##############################################################################
prompt SQL WORKSHOP > QUERY BUILDER

--==============================================================================
create table wwv_flow_qb_saved_query(
    id                number
                      constraint wwv_flow_qb_saved_query_pk primary key,
    query_owner       varchar2(255)      not null,
    title             varchar2(255)      not null,
    qb_sql            clob,
    description       varchar2(4000),
    query_type        varchar2(1)        not null
                      check (query_type in ('P','R','W')),
    created_by        varchar2(255)      not null,
    created_on        date               not null,
    last_updated_by   varchar2(255),
    last_updated_on   date,
    security_group_id number             not null
                      constraint wwv_flow_qb_saved_query_fk
                      references wwv_flow_companies(provisioning_company_id)
                      on delete cascade)
/

create index wwv_flow_qb_saved_query_fkidx on wwv_flow_qb_saved_query (security_group_id)
/
create unique index wwv_flow_qb_saved_query_uk on wwv_flow_qb_saved_query (title, created_by, security_group_id )
/

--==============================================================================
create table wwv_flow_qb_saved_cond(
  id     number                          not null,
  col    varchar2(255)                   not null,
  alias  varchar2(255)                   not null,
  fv     varchar2(255),
  fp     varchar2(255),
  out    varchar2(255)    constraint wwv_flow_qb_saved_cond_out
                          check (out in ('true','false')),
  st     varchar2(255)    constraint wwv_flow_qb_saved_cond_st
                          check (st in ('ASC','DESC')),
  so     varchar2(255),
  grp    varchar2(255)    constraint wwv_flow_qb_saved_cond_grp
                          check (grp in ('true','false')),
  con    varchar2(255),
  ord    number                          not null,
  security_group_id number               not null,
  constraint pk_wwv_flow_qb_saved_cond primary key (id, col),
  foreign key (id)
    references wwv_flow_qb_saved_query (id)
    on delete cascade)
/

--==============================================================================
create table wwv_flow_qb_saved_join(
  id      number                         not null,
  field1  varchar2(255)                  not null,
  field2  varchar2(255)                  not null,
  cond    varchar2(1)                    not null,
  security_group_id number               not null,
  constraint pk_wwv_flow_qb_saved_join primary key (id, field1, field2),
  foreign key (id)
    references wwv_flow_qb_saved_query (id)
    on delete cascade)
/

create table wwv_flow_qb_saved_tabs(
  id     number                             not null,
  oid    number                             not null,
  cnt    number                             not null,
  top    varchar2(255),
  left   varchar2(255),
  tname  varchar2(255)                      not null,
  security_group_id number                  not null,
  constraint pk_wwv_flow_qb_saved_tabs primary key (id, oid, cnt),
  foreign key (id)
    references wwv_flow_qb_saved_query (id)
    on delete cascade)
/

--##############################################################################
--#
--# SQL WORKSHOP > TRANSACTIONS
--#
--##############################################################################

--==============================================================================
create table wwv_flow_sc_trans (
    session_id          number not null
                                  constraint wwv_flow_sc_trans_fk2
                                  references wwv_flow_sessions$(id)
                                  on delete cascade,
    transaction_id      number not null,
    security_group_id   number not null
                                  constraint wwv_flow_sc_trans_fk1
                                  references wwv_flow_companies(provisioning_company_id)
                                  on delete cascade,
    transaction_user    varchar2(2000) not null,
    transaction_schema  varchar2(2000) not null,
    transaction_type    varchar2(10) not null
                                 constraint wwv_flow_sc_trans_ck1
                                 check (transaction_type in (
                                 'STOP',
                                 'SQL',
                                 'PLSQL',
                                 'JAVASCRIPT',
                                 'JAVA',
                                 'PYTHON')),
    transaction_status  varchar2(1) not null
                                constraint wwv_flow_sc_trans_ck2
                                 check (transaction_status in (
                                 'N',
                                 'R')),
    transaction_rows    number not null,
    transaction_binds   varchar2(4000),
    transaction_sql     clob not null,
    transaction_result  clob,
    transaction_output  clob,
    created_on          date not null,
    updated_on          date
    );

alter table wwv_flow_sc_trans add constraint wwv_flow_sc_trans_pk primary key (SESSION_ID, transaction_id)
/

create index wwv_flow_sc_trans_fk1 on wwv_flow_sc_trans (security_group_id)
/

--==============================================================================
create table wwv_flow_sw_results (
    id                    number
                          constraint wwv_flow_sw_result_pk
                          primary key,
    file_id               number not null
                          constraint wwv_flow_sw_result_fk
                          references flows_files.wwv_flow_file_objects$(id)
                          on delete cascade,
    job_id                number,
    run_by                varchar2(255) not null,
    run_as                varchar2(255) not null,
    started               date not null,
    start_time            number not null,
    ended                 date,
    end_time              number,
    status                varchar2(255)
                          constraint wwv_flow_result_status
                          check (status in (
                          'SUBMITTED',
                          'EXECUTING',
                          'COMPLETE',
                          'CANCELED',
                          'CANCELING')),
    run_comments          varchar2(4000),
    security_group_id     number not null
                          constraint wwv_flow_sw_result_fk2
                          references wwv_flow_companies(provisioning_company_id)
                          on delete cascade
    )
/

create index wwv_flow_sw_result_idx1 on wwv_flow_sw_results (file_id)
/
create unique index wwv_flow_sw_result_idx3 on wwv_flow_sw_results (security_group_id,id,status) compress 1
/

--==============================================================================
create table wwv_flow_sw_detail_results (
    id                    number
                          constraint wwv_flow_sw_d_result_pk
                          primary key,
    result_id             number not null
                          constraint wwv_flow_sw_d_result_fk
                          references wwv_flow_sw_results(id)
                          on delete cascade,
    file_id               number not null,
    seq_id                number not null,
    stmt_num              number not null,
    stmt_text             clob default empty_clob(),
    result                clob default empty_clob(),
    result_size           number,
    result_rows           number,
    msg                   varchar2(4000),
    success               varchar2(1) default 'N'
                          constraint wwv_run_success_flg
                          check (success in (
                          'Y',
                          'N')),
    failure               varchar2(1) default 'N'
                          constraint wwv_run_failure_flag
                          check (failure in (
                          'Y',
                          'N')),
    started               date not null,
    start_time            number not null,
    ended                 date,
    end_time              number,
    run_complete          varchar2(1) default 'N'
                          constraint wwv_stmt_run_complete_flag
                          check (run_complete in (
                          'Y',
                          'N')),
    last_updated          date,
    security_group_id     number not null
                          constraint wwv_flow_sw_d_result_fk2
                          references wwv_flow_companies(provisioning_company_id)
                          on delete cascade
    ) lob (stmt_text,result) store as (
        pctversion 0
        cache reads
        enable storage in row )
/

create index wwv_flow_sw_d_result_idx1 on wwv_flow_sw_detail_results (result_id)
/
create index wwv_flow_sw_d_result_idx3 on wwv_flow_sw_detail_results (security_group_id,result_id,file_id) compress 2
/

--==============================================================================
create table wwv_flow_sw_sqlplus_cmd (
    command                varchar2(30) primary key
    )
/

--==============================================================================
create table wwv_flow_sw_sql_cmds (
    id                              number
                                    constraint wwv_flow_sw_sql_cmds_pk
                                    primary key,
    command                         clob,
    parsed_schema                   varchar2(128),
    created_by                      varchar2(255),
    created_on                      date,
    security_group_id               number not null
                                    constraint WWV_FLOW_SW_SQL_CMDS_fk
                                    references wwv_flow_companies(provisioning_company_id)
                                    on delete cascade
    )
/

create unique index WWV_FLOW_SW_SQL_CMDS_FKIDX ON WWV_FLOW_SW_SQL_CMDS(SECURITY_GROUP_ID,PARSED_SCHEMA,id)
/

--==============================================================================
-- Parser output table.  Will store each line of the statement.
prompt ...create wwv_flow_sw_stmts
create table wwv_flow_sw_stmts (
   id                  number
                       constraint wwv_flow_sw_stmts_pk
                       primary key,
   file_id             number not null
                       constraint wwv_flow_sw_stmts_fk
                       references flows_files.wwv_flow_file_objects$(id)
                       on delete cascade,
   stmt_number         number,
   src_line_number     number,
   offset              number,
   length              number,
   stmt_class          number,
   stmt_id             number,
   isrunnable          varchar2(1),
   stmt_vc2            varchar2(4000),
   stmt                clob,
   security_group_id   number not null
                       constraint wwv_flow_sw_stmts_fk2
                       references wwv_flow_companies(provisioning_company_id)
                       on delete cascade
) lob (stmt) store as (
    pctversion 0
    cache reads
    enable storage in row )
/

create index wwv_flow_sw_stmts_idx on wwv_flow_sw_stmts (file_id);
create index wwv_flow_sw_stmts_idx2 on wwv_flow_sw_stmts (security_group_id,file_id,isrunnable,stmt_id,stmt_number) compress 2;

--==============================================================================
-- Holds the list of non special-character first keywords
create table wwv_flow_sw_main_keywords (
    id         number
               constraint wwv_flow_sw_mkeywords_pk
               primary key,
    keyword    VARCHAR2(20),
    stmt_class INTEGER,
    stmt_id    INTEGER,
    isrunnable VARCHAR2(1)
  );

create unique index wwv_flow_sw_mkeywords_idx on wwv_flow_sw_main_keywords (keyword);

--==============================================================================
-- Holds Quick SQL saved models
create table wwv_qs_saved_models (
    id                             number not null
                                   constraint wwv_qs_saved_models_pk primary key,
    security_group_id              number not null
                                   constraint wwv_qs_saved_models_fk
                                   references wwv_flow_companies(provisioning_company_id)
                                   on delete cascade,
    row_version                    integer not null,
    name                           varchar2(255) not null,
    description                    varchar2(4000),
    model                          clob not null,
    created                        date not null,
    created_by                     varchar2(255) not null,
    updated                        date,
    updated_by                     varchar2(255) );

-- model name must be unique within a workspace
create unique index wwv_qs_saved_models_unq_idx on wwv_qs_saved_models (security_group_id, name);

--===============================================================================
-- Holds installed Patch Set Exception bundle information - 20.1 feature #2588
create table wwv_flow_pses (
    id                 number not null
                       constraint wwv_flow_pses_pk
                       primary key,
    patch_number       number not null,
    patch_version      varchar2(30) not null,
    images_version     varchar2(30) not null,
    is_bundle_patch    varchar2(1) default 'Y',
    installed_on       date
                       default sysdate
                       not null
);

create index wwv_flow_pses_idx on wwv_flow_pses ( installed_on );

--##############################################################################
--#
--# PDF Tables
--#
--##############################################################################

--==============================================================================
-- table wwv_flow_pdf_fonts to store PDF fonts
create table wwv_flow_pdf_fonts (
    id                          number          not null
                                                constraint pdf_fnt_pk
                                                primary key,
    is_standard                 varchar2(1)     not null
                                                constraint pdf_fnt_is_standard_ck
                                                check (is_standard in (
                                                    'Y',
                                                    'N' )),
    font_name                   varchar2(50)    not null,
    family_name                 varchar2(50)    not null,
    font_style                  varchar2(7)     not null,
                                                constraint pdf_fnt_stl_ck
                                                check (font_style in (
                                                    'normal',
                                                    'italic' )),
    font_weight                 varchar2(7)     not null
                                                constraint pdf_fnt_wgt_ck
                                                check (font_weight in (
                                                    'normal',
                                                    'bold' )),
    line_height                 number          ,
    fallback_priority           number          ,
    font_file                   blob            ,
    constraint pdf_fnt_name_uk unique(font_name),
    constraint pdf_fnt_standard_ck check (
        (   is_standard = 'Y'
            and font_name in (
                'Courier-Bold',
                'Courier-BoldOblique',
                'Courier-Oblique',
                'Courier',
                'Helvetica-Bold',
                'Helvetica-BoldOblique',
                'Helvetica-Oblique',
                'Helvetica',
                'Symbol',
                'Times-Bold',
                'Times-BoldItalic',
                'Times-Italic',
                'Times-Roman',
                'ZapfDingbats'
            )
            and family_name in (
                'Courier',
                'Helvetica',
                'Symbol',
                'Times',
                'ZapfDingbats'
            )
            and fallback_priority is null
            and font_file is null
        ) or
        (   is_standard = 'N'
            and fallback_priority is not null
            and font_file is not null
        )
    )
);

--==============================================================================
-- table wwv_flow_pdf_std_fnt_glyphs to store the 14 standard PDF fonts glyph details
create table wwv_flow_pdf_std_fnt_glyphs (
    id                          number          not null
                                                constraint pdf_std_fnt_gly_pk
                                                primary key,
    font_id                     number          not null
                                                constraint pdf_std_fnt_gly_fk
                                                references wwv_flow_pdf_fonts(id)
                                                on delete cascade,
    char_code                   number          not null,
    width                       number(4)       not null,
    constraint pdf_std_fnt_gly_uk unique (font_id,char_code)
);

--==============================================================================
-- table wwv_flow_pdf_ext_fnt_ranges to store the used codepoint ranges of the font
create table wwv_flow_pdf_ext_fnt_ranges (
    id                          number          not null
                                                constraint pdf_ext_fnt_rng_pk
                                                primary key,
    font_id                     number          not null
                                                constraint pdf_ext_fnt_rng_fk
                                                references wwv_flow_pdf_fonts(id)
                                                on delete cascade,
    start_codepoint             number          not null,
    end_codepoint               number          not null,
    constraint pdf_ext_fnt_rng_uk unique (font_id,start_codepoint,end_codepoint)
);

--==============================================================================
-- table wwv_flow_pdf_unicode_data to store unicode data for each character
create table wwv_flow_pdf_unicode_data (
    codepoint_start             number          not null,
    codepoint_end               number          not null,
    bidi_class                  varchar2(3)     not null,
    bidi_mirrored               varchar2(1)     not null
                                                constraint pdf_uni_mir_ck
                                                check (bidi_mirrored in (
                                                    'Y',
                                                    'N' )),
    script                      varchar2(50),
    unicode_block               varchar2(50),
    constraint pdf_uni_pk primary key (codepoint_start,codepoint_end)
) organization index;

--==============================================================================
-- table wwv_flow_pdf_unicode_arab to store Arabic shaping information for those characters
create table wwv_flow_pdf_unicode_arab (
    codepoint                   number          constraint pdf_uni_ara_pk
                                                primary key,
    joining_type                varchar2(1)     not null
                                                constraint pdf_uni_ara_ck
                                                check (joining_type in (
                                                    'R',
                                                    'L',
                                                    'D',
                                                    'C',
                                                    'U',
                                                    'T' ))
);

--==============================================================================
-- table wwv_flow_pdf_unicode_mirroring to store mirrored codepoints
create table wwv_flow_pdf_unicode_mirroring (
    codepoint                   number          not null,
    codepoint_mirrored          number          not null,
    constraint pdf_uni_mir_pk primary key (codepoint,codepoint_mirrored)
);

--##############################################################################
--#
--# Application Gallery
--#
--##############################################################################

--==============================================================================
-- table wwv_flow_gallery_installs to store installed packaged apps
create table wwv_flow_gallery_installs (
    id                        number         not null
                                             constraint wwv_flow_gallery_inst_pk
                                             primary key,
    security_group_id         number         not null
                                             constraint wwv_flow_gallery_inst_sg_fk
                                             references wwv_flow_companies(provisioning_company_id) on delete cascade,
    flow_id                   number         not null
                                             constraint wwv_flow_gallery_inst_flow_fk
                                             references wwv_flows on delete cascade,
    author                    varchar2(255)  not null,
    internal_name             varchar2(255)  not null,
    created_by                varchar2(255)  not null,
    created_on                date           not null
  );

alter table wwv_flow_gallery_installs add constraint wwv_flow_gallery_inst_uk unique (security_group_id, author, internal_name);

create index wwv_flow_gallery_inst_flow_fx on wwv_flow_gallery_installs ( flow_id ) compress;

--==============================================================================
-- Desupported AnyChart Tables - retained for upgrade logic
--==============================================================================
create table wwv_flow_flash_charts ( -- obsolete
      id                            number
                                    constraint wwv_flow_flash_charts_pk
                                    primary key,
      flow_id                       number not null
                                    constraint wwv_flow_flash_charts_fk
                                    references wwv_flows
                                    on delete cascade,
      page_id                       number,
      region_id                     number not null
                                    constraint wwv_flow_flash_charts_fk2
                                    references wwv_flow_page_plugs
                                    on delete cascade,
      default_chart_type            varchar2(255) not null,
      chart_title                   varchar2(255),
      chart_width                   number,
      chart_height                  number,
      chart_animation               varchar2(255),
      display_attr                  varchar2(4000),
      dial_tick_attr                varchar2(4000),
      margins                       varchar2(4000),
      omit_label_interval           number,
      --
      color_scheme                  varchar2(255),
      custom_colors                 varchar2(4000),
      bgtype                        varchar2(255),
      bgcolor1                      varchar2(30),
      bgcolor2                      varchar2(30),
      gradient_rotation             number,
      --
      x_axis_title                  varchar2(4000),
      x_axis_min                    number,
      x_axis_max                    number,
      x_axis_grid_spacing           number,
      x_axis_prefix                 varchar2(4000),
      x_axis_postfix                varchar2(4000),
      x_axis_group_sep              varchar2(1)
                                    constraint wwv_flow_flash_charts_xgroup
                                      check (x_axis_group_sep in ('Y','N')),
      x_axis_decimal_place          number,
      y_axis_title                  varchar2(4000),
      y_axis_min                    number,
      y_axis_max                    number,
      y_axis_grid_spacing           number,
      y_axis_prefix                 varchar2(4000),
      y_axis_postfix                varchar2(4000),
      y_axis_group_sep              varchar2(1)
                                    constraint wwv_flow_flash_charts_ygroup
                                      check (y_axis_group_sep in ('Y','N')),
      y_axis_decimal_place          number,
      async_update                  varchar2(1)
                                    constraint wwv_flow_flash_charts_update
                                      check (async_update in ('Y','N')),
      async_time                    number,
      --
      names_font                    varchar2(4000),
      names_rotation                number,
      values_font                   varchar2(4000),
      values_rotation               number,
      hints_font                    varchar2(4000),
      legend_font                   varchar2(4000),
      grid_labels_font              varchar2(4000),
      chart_title_font              varchar2(4000),
      x_axis_title_font             varchar2(4000),
      y_axis_title_font             varchar2(4000),
      --
      use_chart_xml                 varchar2(1)
                                    constraint wwv_flow_flash_charts_xml
                                      check (use_chart_xml in ('Y','N')),
      chart_xml                     clob,
      attribute_01                  varchar2(4000),
      attribute_02                  varchar2(4000),
      attribute_03                  varchar2(4000),
      attribute_04                  varchar2(4000),
      attribute_05                  varchar2(4000),
      created_on                    date,
      created_by                    varchar2(255),
      updated_on                    date,
      updated_by                    varchar2(255),
      security_group_id             number not null
  ) lob (chart_xml) store as (cache enable storage in row)
/

create index wwv_flow_fl_ch_fid_fk on wwv_flow_flash_charts (flow_id)
/

create index wwv_flow_fl_ch_idx on wwv_flow_flash_charts (region_id, flow_id, security_group_id)
/

--==============================================================================
create table wwv_flow_flash_chart_series ( -- obsolete
       id                           number
                                    constraint wwv_flow_flash_chart_series_pk
                                    primary key,
       chart_id                     number not null
                                    constraint wwv_flow_flash_chart_series_fk
                                    references wwv_flow_flash_charts
                                    on delete cascade,
       flow_id                      number,
       series_seq                   number not null,
       series_name                  varchar2(255) not null,
       series_query                 clob not null,
       series_query_type            varchar2(255) not null
                                    constraint wwv_flow_flash_query_type
                                       check (series_query_type in ('SQL_QUERY','FUNCTION_RETURNING_SQL_QUERY')),
       series_query_parse_opt       varchar2(255),
       series_query_no_data_found   varchar2(4000),
       series_query_row_count_max   number,
       created_on                   date,
       created_by                   varchar2(255),
       updated_on                   date,
       updated_by                   varchar2(255),
       security_group_id            number not null
   )
/

create index wwv_flow_fl_ch_sr_idx on wwv_flow_flash_chart_series (chart_id, flow_id, security_group_id)
/

--==============================================================================
create table wwv_flow_flash_charts_5 (
      id                            number
                                    constraint wwv_flow_flash_charts_5_pk
                                    primary key,
      flow_id                       number not null
                                    constraint wwv_flow_flash_charts_5_fk
                                    references wwv_flows
                                    on delete cascade,
      page_id                       number,
      region_id                     number not null
                                    constraint wwv_flow_flash_charts_5_fk2
                                    references wwv_flow_page_plugs
                                    on delete cascade,
      chart_type                    varchar2(255) not null,
      chart_rendering               varchar2(20) default 'FLASH_PREFERRED' not null
                                    constraint wwv_flow_flash_5_rendering
                                     check (chart_rendering in ('FLASH_PREFERRED', 'SVG_ONLY')),
      chart_title                   varchar2(4000),
      chart_name                    varchar2(255),
      chart_width                   number,
      chart_height                  number,
      chart_animation               varchar2(255),
      --
      display_attr                  varchar2(4000),
      dial_tick_attr                varchar2(4000),
      margins                       varchar2(4000),
      bar_shape_type                varchar2(10),
      pie_attr                      varchar2(4000),
      map_source                    varchar2(4000),
      gantt_attr                    varchar2(4000),
      map_attr                      varchar2(4000),
      gantt_date_format             varchar2(255),
      gantt_start_date              date,
      gantt_end_date                date,
      --
      chart_bgtype                  varchar2(255),
      chart_bgimage_mode            varchar2(10),
      chart_bgimage_url             varchar2(4000),
      chart_bgcolor                 varchar2(4000),
      chart_bgcolor2                varchar2(4000),
      chart_bgcorners               varchar2(15),
      chart_gradient_angle          number,
      grid_bgtype                   varchar2(255),
      grid_bgcolor                  varchar2(4000),
      grid_bgcolor2                 varchar2(4000),
      grid_gradient_angle           number,
      data_plot_bgtype              varchar2(10),
      data_plot_bgimage_mode        varchar2(10),
      data_plot_bgimage_url         varchar2(4000),
      data_plot_bgcorners           varchar2(15),
      color_scheme                  varchar2(255),
      custom_colors                 varchar2(4000),
      map_undef_color_scheme        varchar2(255),
      map_undef_custom_colors       varchar2(4000),
      --
      x_axis_title                  varchar2(4000),
      x_axis_title_font             varchar2(4000),
      x_axis_title_rotation         number,
      x_axis_min                    number,
      x_axis_max                    number,
      x_axis_decimal_place          number,
      x_axis_scroll_start           varchar2(4000),
      x_axis_scroll_end             varchar2(4000),
      x_axis_scroll_range           varchar2(4000),
      x_axis_scroll_range_unit      varchar2(255),
      x_axis_label_rotation         number,
      x_axis_label_font             varchar2(4000),
      x_axis_prefix                 varchar2(4000),
      x_axis_postfix                varchar2(4000),
      x_axis_major_interval         number,
      x_axis_minor_interval         number,
      --
      y_axis_title                  varchar2(4000),
      y_axis_title_font             varchar2(4000),
      y_axis_title_rotation         number,
      y_axis_min                    number,
      y_axis_max                    number,
      extra_y_axis_min              number,
      extra_y_axis_max              number,
      y_axis_scroll_start           varchar2(4000),
      y_axis_scroll_end             varchar2(4000),
      y_axis_scroll_range           varchar2(4000),
      y_axis_scroll_range_unit      varchar2(255),
      y_axis_decimal_place          number,
      y_axis_label_rotation         number,
      y_axis_label_font             varchar2(4000),
      y_axis_prefix                 varchar2(4000),
      y_axis_postfix                varchar2(4000),
      y_axis_major_interval         number,
      y_axis_minor_interval         number,
      values_prefix                 varchar2(4000),
      values_postfix                varchar2(4000),
      --
      chart_title_font              varchar2(4000),
      values_font                   varchar2(4000),
      values_rotation               number,
      tooltip_font                  varchar2(4000),
      legend_font                   varchar2(4000),
      legend_title                  varchar2(4000),
      legend_title_font             varchar2(4000),
      grid_labels_font              varchar2(4000),
      gauge_labels_font             varchar2(4000),
      --
      async_update                  varchar2(1)
                                    constraint wwv_flow_flash_5_update
                                      check (async_update in ('Y','N')),
      async_time                    number,
      use_chart_xml                 varchar2(1)
                                    constraint wwv_flow_flash_5_xml
                                      check (use_chart_xml in ('Y','N')),
      chart_xml                     clob,
      created_on                    date,
      created_by                    varchar2(255),
      updated_on                    date,
      updated_by                    varchar2(255),
      security_group_id             number not null
  ) lob (chart_xml) store as (cache enable storage in row)
/

create index wwv_flow_flch5_fid_fk on wwv_flow_flash_charts_5 (flow_id)
/

create index wwv_flow_flch5_idx on wwv_flow_flash_charts_5 (region_id, flow_id, security_group_id)
/

--==============================================================================
create table wwv_flow_flash_chart5_series (
       id                           number
                                    constraint wwv_flow_flash_5_series_pk
                                    primary key,
       chart_id                     number not null
                                    constraint wwv_flow_flash_5_series_fk
                                    references wwv_flow_flash_charts_5
                                    on delete cascade,
       flow_id                      number,
       series_seq                   number not null,
       series_name                  varchar2(255) not null,
       series_type                  varchar2(255),
       series_query_type            varchar2(255) not null
                                    constraint wwv_flow_flash5_qry_type
                                       check (series_query_type in ('SQL_QUERY','FUNCTION_RETURNING_SQL_QUERY')),
       series_query                 clob not null,
       series_ajax_items_to_submit  varchar2(4000),
       series_query_parse_opt       varchar2(255),
       series_query_no_data_found   varchar2(4000),
       series_query_row_count_max   number,
       action_link                  varchar2(4000),
       show_action_link             varchar2(1),
       action_link_checksum_type    varchar2(255) constraint wwv_chart_link_checksum_type
                                    check (action_link_checksum_type in (
                                    '1', -- workspace
                                    '2', -- user
                                    '3'  -- session
                                )),
       series_required_role         varchar2(255),
       required_patch               number,
       display_when_cond_type       varchar2(255),
       display_when_condition       varchar2(4000),
       display_when_condition2      varchar2(4000),
       created_on                   date,
       created_by                   varchar2(255),
       updated_on                   date,
       updated_by                   varchar2(255),
       security_group_id            number not null
   )
/

create index wwv_flow_flch5sr_idx on wwv_flow_flash_chart5_series (chart_id, flow_id, security_group_id)
/

--==============================================================================
-- table wwv_flow_task_defs to store the task definitions for the approval component
create table wwv_flow_task_defs (
    id                          number not null
                                constraint wwv_task_def_pk
                                primary key,
    flow_id                     number not null
                                constraint wwv_task_def_flow_fk
                                references wwv_flows on delete cascade,
    security_group_id           number not null   
                                constraint wwv_task_def_sgid_fk
                                references wwv_flow_companies(provisioning_company_id)
                                on delete cascade,                                            
    --
    name                        varchar2(255)  not null, 
    static_id                   varchar2(255)  not null, 
    --
    subject                     varchar2(1000) not null,  
    priority                    number(1)      not null
                                constraint wwv_task_def_prio_ck
                                check (priority between 1 and 5),
    due_on_interval             varchar2(255),  
    details_link_target         varchar2(4000),
    --
    actions_sql_query           varchar2(4000), 
    actions_table_owner         varchar2(128),  
    actions_table_name          varchar2(128),  
    actions_pk_column_name      varchar2(128),  
    --     
    task_def_comment            varchar2(4000),
    --
    created_by                  varchar2(255),
    created_on                  date,
    last_updated_by             varchar2(255),
    last_updated_on             date
)
/
alter table wwv_flow_task_defs add constraint wwv_task_def_name_uk unique (flow_id, name) using index compress 1
/
alter table wwv_flow_task_defs add constraint wwv_task_def_static_id_uk unique (flow_id, static_id) using index compress 1
/

create index wwv_task_def_sgid_fx on wwv_flow_task_defs(security_group_id) compress
/

--==============================================================================
-- table wwv_flow_task_def_params to store the task definition payload
create table wwv_flow_task_def_params (
    id                      number
                            constraint wwv_task_def_param_pk
                            primary key,
    flow_id                 number not null
                            constraint wwv_task_def_param_flow_fk
                            references wwv_flows
                            on delete cascade,
    security_group_id       number  not null
                            constraint wwv_task_def_param_sgid_fk
                            references wwv_flow_companies(provisioning_company_id)
                            on delete cascade,              
    task_def_id             number not null
                            constraint wwv_task_def_param_tdid_fk
                            references wwv_flow_task_defs(id)
                            on delete cascade,
    --
    static_id               varchar2(255) not null,
    label                   varchar2(255) not null,
    data_type               varchar2(8)   not null,                 
                            constraint wwv_task_def_param_type_ck
                            check (data_type in ('VARCHAR2', 'NUMBER', 'DATE' )),
    is_required             varchar2(1)   not null,
                            constraint wwv_task_def_param_reqd_ck
                            check (is_required in ('Y', 'N')),
    is_visible              varchar2(1)   not null
                            constraint wwv_task_def_param_vis_ck
                            check (is_visible in ('Y', 'N')),                        
    --
    param_comment           varchar2(4000),
    --
    created_by              varchar2(255),
    created_on              date,
    last_updated_by         varchar2(255),
    last_updated_on         date 
)
/ 

alter table wwv_flow_task_def_params add constraint wwv_task_def_param_uk unique (task_def_id, static_id)
/
create index wwv_task_def_params_flowid_fx  on wwv_flow_task_def_params (flow_id) compress
/
create index wwv_task_def_params_sgid_fx  on wwv_flow_task_def_params (security_group_id) compress
/
create index wwv_task_def_params_td_fx   on wwv_flow_task_def_params (task_def_id) compress
/

--==============================================================================
-- table  wwv_flow_task_def_actions to store the task actions 
--==============================================================================
create table wwv_flow_task_def_actions (
    id                      number
                            constraint wwv_task_def_act_pk
                            primary key,
    flow_id                 number not null
                            constraint wwv_task_def_act_flow_fk
                            references wwv_flows
                            on delete cascade,
    security_group_id       number not null   
                            constraint wwv_task_def_act_sgid_fk
                            references wwv_flow_companies(provisioning_company_id)
                            on delete cascade,                                                
    task_def_id             number          not null
                            constraint wwv_task_def_act_parent_fk
                            references wwv_flow_task_defs
                            on delete cascade,
    --
    name                    varchar2(255)   not null,
    outcome                 varchar2(8)    constraint wwv_task_def_act_outcome_ck
                                            check (outcome in ('APPROVED', 'REJECTED')),
    on_event                varchar2(15)     not null,
                            constraint wwv_task_def_act_event_ck  
                            check (on_event in ('CREATE', 'CLAIM', 'COMPLETE', 'DELEGATE', 'FAIL', 'FORWARD', 
                            'UPDATE_COMMENT', 'UPDATE_PRIORITY','UPDATE_DUE_ON', 'RELEASE', 'CANCEL')),
    execution_sequence      number          not null,
    action_type             varchar2(255)   not null
                            constraint wwv_task_def_act_type_ck
                            check (
                                action_type like 'NATIVE\_%' escape '\' or
                                action_type like 'PLUGIN\_%' escape '\'),
    --
    action_clob             clob,
    action_clob_language    varchar2(10)
                            constraint wwv_task_def_act_clang_ck
                            check (action_clob_language in ( 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    attribute_01            varchar2(4000),
    attribute_02            varchar2(4000),
    attribute_03            varchar2(4000),
    attribute_04            varchar2(4000),
    attribute_05            varchar2(4000),
    attribute_06            varchar2(4000),
    attribute_07            varchar2(4000),
    attribute_08            varchar2(4000),
    attribute_09            varchar2(4000),
    attribute_10            varchar2(4000),
    attribute_11            varchar2(4000),
    attribute_12            varchar2(4000),
    attribute_13            varchar2(4000),
    attribute_14            varchar2(4000),
    attribute_15            varchar2(4000),
    --
    location                varchar2(10)
                            constraint wwv_task_def_act_loc_ck
                            check (location in ('LOCAL', 'REMOTE', 'WEB_SOURCE' )),
    remote_server_id        number
                            constraint wwv_task_def_act_rs_fk
                            references wwv_remote_servers,
    web_src_module_id       number
                            constraint wwv_task_def_act_ws_mod_fk
                            references wwv_flow_web_src_modules,
    web_src_operation_id    number
                            constraint wwv_task_def_act_op_fk
                            references wwv_flow_web_src_operations,
    email_template_id       number generated always as (
                                                         case when action_type = 'NATIVE_SEND_EMAIL' and attribute_11 is not null 
                                                             then to_number( attribute_11 )
                                                         end ) virtual,
    --
    success_message         varchar2(4000), /* could this be useful???? */
    error_message           varchar2(4000),
    stop_execution_on_error varchar2(1) not null
                            constraint wwv_task_def_act_stop_err_ck
                            check (stop_execution_on_error in ( 'Y', 'N' )),
    --
    condition_type          varchar2(255), /* Time to add a constraint? */
    condition_expr1         varchar2(4000),
    condition_expr2         varchar2(4000),
    --
    build_option_id         number,
    /* do we need authorization? */
    action_comment          varchar2(4000),
    --
    created_by              varchar2(255),
    created_on              date,
    last_updated_by         varchar2(255),
    last_updated_on         date,
    --
    constraint wwv_task_def_act_email_fk foreign key( email_template_id )
                                                 references wwv_flow_email_templates
                                                 deferrable initially deferred
    )
lob (action_clob) store as (cache reads enable storage in row)
/
alter table wwv_flow_task_def_actions add constraint wwv_task_def_act_uk unique (task_def_id, name) using index compress 1
/

alter table wwv_flow_task_def_actions add constraint wwv_task_def_act_ws_mod_ck
    check ( web_src_module_id is null or web_src_operation_id is null )
/

create index wwv_task_def_act_flowid_fx       on wwv_flow_task_def_actions (flow_id)
/
create index wwv_task_def_act_parent_fx     on wwv_flow_task_def_actions (task_def_id)
/
create index wwv_task_def_act_ws_mod_fx     on wwv_flow_task_def_actions (web_src_module_id)       compress
/
create index wwv_task_def_act_ws_ops_fx     on wwv_flow_task_def_actions (web_src_operation_id)    compress
/
create index wwv_task_def_act_rs_fx         on wwv_flow_task_def_actions (remote_server_id)        compress
/
create index wwv_task_def_act_eti_fx     on wwv_flow_task_def_actions (email_template_id)       compress
/
create index wwv_task_def_act_sgid_fx  on wwv_flow_task_def_actions (security_group_id) compress
/


--==============================================================================
-- table wwv_flow_task_def_comp_params table for Page Designer metadata
create table wwv_flow_task_def_comp_params (
    id                      number
                            constraint wwv_task_def_cparam_pk
                            primary key,
    security_group_id       number not null
                            constraint wwv_task_def_cparam_sgid_fk
                            references wwv_flow_companies(provisioning_company_id),
    flow_id                 number not null
                            constraint wwv_task_def_cparam_flow_fk
                            references wwv_flows
                            on delete cascade,
    page_id                 number,
    task_def_param_id       number not null
                            constraint wwv_task_def_cparam_tdid_fk
                            references wwv_flow_task_def_params
                            on delete cascade,
    --
    page_process_id         number
                            constraint wwv_task_def_cparam_procid_fk
                            references wwv_flow_step_processing
                            on delete cascade,
    app_process_id          number
                            constraint wwv_task_def_cparam_apid_fk
                            references wwv_flow_processing
                            on delete cascade,
    automation_action_id    number
                            constraint wwv_task_def_cparam_amid_fk
                            references wwv_flow_automation_actions( id )
                            on delete cascade,
    task_def_action_id      number
                            constraint wwv_task_def_cparam_actid_fk
                            references wwv_flow_task_def_actions( id )
                            on delete cascade,
    --
    value_type              varchar2(13) not null
                            constraint wwv_task_def_cparam_valtype_ck
                            check (value_type in (
                                'NULL',
                                'STATIC',
                                'SQL_QUERY',
                                'ITEM',
                                'EXPRESSION',
                                'FUNCTION_BODY')),
    value_language          varchar2(10)
                            constraint wwv_task_def_cparam_vlang_ck
                            check (value_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    value                   varchar2(4000),
    --
    created_by              varchar2(255),
    created_on              date,
    last_updated_by         varchar2(255),
    last_updated_on         date 
)
/
create index wwv_task_def_cparam_fp_ix on wwv_flow_task_def_comp_params (flow_id, page_id)     compress 1
/
create index wwv_task_def_cparam_tpar_fx on wwv_flow_task_def_comp_params (task_def_param_id)   compress
/
-- create index wwv_wf_task_def_cparam_fx3 on wwv_wf_task_def_comp_params (page_id)          compress
-- /
create index wwv_task_def_cparam_pid_fx on wwv_flow_task_def_comp_params (page_process_id)        compress
/
create index wwv_task_def_cparam_aid_fx on wwv_flow_task_def_comp_params (app_process_id)         compress
/
create index wwv_task_def_cparam_amid_fx on wwv_flow_task_def_comp_params (automation_action_id)  compress
/
create index wwv_task_def_cparam_actid_fx on wwv_flow_task_def_comp_params (task_def_action_id)  compress
/
create index wwv_task_def_cparam_sgid_fx on wwv_flow_task_def_comp_params (security_group_id)  compress
/
--==============================================================================
-- table  wwv_flow_task_def_participants to store the task instance payload
create table  wwv_flow_task_def_participants (
     id                         number          not null
                                                constraint wwv_task_def_part_pk 
                                                primary key,
    security_group_id           number          not null
                                                constraint wwv_task_def_part_sg_fk
                                                references wwv_flow_companies(provisioning_company_id)
                                                on delete cascade,
    flow_id                     number          not null
                                                constraint wwv_task_def_part_flow_fk
                                                references wwv_flows
                                                on delete cascade,                                            
    task_def_id                 number          not null
                                                constraint wwv_task_def_part_parent_fk
                                                references wwv_flow_task_defs(id)
                                                on delete cascade,
    participant_type            varchar2(15)    not null
                                                constraint wwv_task_def_part_ptype_ck
                                                check (participant_type in 
                                                ('POTENTIAL_OWNER','BUSINESS_ADMIN')),
    identity_type               varchar2(5)    constraint wwv_task_def_part_itype_ck
                                                check (identity_type in ('USER')),
    value_type                 varchar2(13)    not null   
                                                constraint wwv_task_def_part_vtype_ck
                                                check (value_type in ( 'STATIC', 'SQL_QUERY', 'EXPRESSION', 'FUNCTION_BODY' )),
    value_language           varchar2(10)
                                                constraint wwv_task_def_part_vlang_ck
                                                check (value_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    value                    varchar2(4000),
    --
    created_by              varchar2(255),
    created_on              date,
    last_updated_by         varchar2(255),
    last_updated_on         date 
)
/

create index wwv_task_def_part_tdid_fx on wwv_flow_task_def_participants (task_def_id)  compress
/
create index wwv_task_def_part_flowid_fx on wwv_flow_task_def_participants (flow_id)  compress
/
create index wwv_task_def_part_sgid_fx on wwv_flow_task_def_participants (security_group_id)  compress
/

--==============================================================================
-- table wwv_flow_tasks runtime tasks for the approval component
create table wwv_flow_tasks$ (
    id                          number          not null
                                                constraint wwv_task_pk
                                                primary key,
    flow_id                     number          not null
                                                constraint wwv_task_flow_fk
                                                references wwv_flows
                                                on delete cascade,                                             
    security_group_id           number          not null
                                                constraint wwv_task_sgid_fk
                                                references wwv_flow_companies(provisioning_company_id)
                                                on delete cascade,                                            
    task_def_id                 number          constraint wwv_task_taskdef_fk
                                                references wwv_flow_task_defs(id)
                                                on delete set null
                                                deferrable initially deferred,
    subject                     varchar2(1000),
    payload                     clob,
    detail_pk                   varchar2(400),
    priority                    number(1)       not null
                                                constraint wwv_task_prio_ck
                                                check (priority between 1 and 5),
    initiator                   varchar2(255),
    actual_owner                varchar2(255),
    due_on                      timestamp(6) with time zone,
    outcome                     varchar2(32)    constraint wwv_task_outcome_ck
                                                check (outcome in ('APPROVED', 'REJECTED')),
    state                       varchar2(32)    not null   
                                                constraint wwv_task_state_ck
                                                check (state in 
                                                ('UNASSIGNED', 'ASSIGNED', 'COMPLETED', 
                                                 'FAILED', 'ERRORED', 'CANCELLED')),
    created_by                  varchar2(255),
    created_on                  timestamp(6) with time zone not null,
    last_updated_by             varchar2(255),
    last_updated_on             timestamp(6) with time zone,
    -- Participants are stored as JSON in clob attribute
    participants                clob,
    -- Comments are stored as JSON in clob attribute
    comments                    clob,
    -- Task History is stored as JSON in clob attribute
    history                     clob
)
lob (payload, comments, history) store as (
    pctversion 0
    cache reads
    enable storage in row
)
/

create index wwv_task_flowid on wwv_flow_tasks$ (flow_id)
    initrans 8
    compress
/
create index wwv_task_tdid on wwv_flow_tasks$ (task_def_id)
    initrans 8
    compress
/
create index wwv_task_dtpktdid on wwv_flow_tasks$ (detail_pk, task_def_id)
    initrans 8
    compress
/
create index wwv_task_actual_owner on wwv_flow_tasks$ (security_group_id, state, actual_owner)
   initrans 8
   compress
/   




--==============================================================================
--##############################################################################
--#
--# REGION > WEB SOURCE > COMPONENT PARAMETERS
--#
--##############################################################################
prompt REGION > WEB SOURCE > COMPONENT PARAMETERS

--==============================================================================
create table wwv_flow_web_src_comp_params (
    id                             number
                                   constraint wwv_flow_web_src_comp_par_pk
                                   primary key,
    security_group_id              number not null
                                   constraint wwv_websrc_cparam_sgid_fk
                                   references wwv_flow_companies(provisioning_company_id),
    flow_id                        number not null
                                   constraint wwv_websrc_cparam_flow_fk
                                   references wwv_flows
                                   on delete cascade,
    page_id                        number,
    --
    web_src_param_id               number not null
                                   constraint wwv_websrc_cparam_param_fk
                                   references wwv_flow_web_src_params
                                   on delete cascade,
    page_plug_id                   number
                                   constraint wwv_websrc_cparam_plug_fk
                                   references wwv_flow_page_plugs
                                   on delete cascade,
    page_process_id                number
                                   constraint wwv_websrc_cparam_pageproc_fk
                                   references wwv_flow_step_processing
                                   on delete cascade,
    app_process_id                 number
                                   constraint wwv_websrc_cparam_appproc_fk
                                   references wwv_flow_processing
                                   on delete cascade,
    jet_chart_series_id            number
                                   constraint wwv_websrc_cparam_jetseries_fk
                                   references wwv_flow_jet_chart_series
                                   on delete cascade,
    map_region_layer_id            number
                                   constraint wwv_websrc_cparam_maplayer_fk
                                   references wwv_flow_map_region_layers
                                   on delete cascade,
    shared_lov_id                  number
                                   constraint wwv_websrc_cparam_sharedlov_fk
                                   references wwv_flow_lists_of_values$
                                   on delete cascade,
    module_sync_step_id            number
                                   constraint wwv_websrc_cparam_syncstep_fk
                                   references wwv_flow_web_src_sync_steps
                                   on delete cascade,
    automation_id                  number
                                   constraint wwv_websrc_cparam_am_fk
                                   references wwv_flow_automations( id )
                                   on delete cascade,
    automation_action_id           number
                                   constraint wwv_websrc_cparam_am_action_fk
                                   references wwv_flow_automation_actions( id )
                                   on delete cascade,
    task_def_action_id             number
                                   constraint wwv_websrc_cparam_taskd_act_fk
                                   references wwv_flow_task_def_actions( id )
                                   on delete cascade,                               
    --
    value_type                     varchar2(30) not null
                                   constraint wwv_websrc_cparam_valtype_ck
                                   check (value_type in (
                                       'PREFERENCE',
                                       'NULL',
                                       'STATIC',
                                       'SQL_QUERY',
                                       'COLLECTION',
                                       'ITEM',
                                       'DEFAULT',
                                       'EXPRESSION',
                                       'FUNCTION_BODY')),
    value_language                 varchar2(10)
                                   constraint wwv_websrc_cparam_vallang_ck
                                   check (value_language in ( 'SQL', 'PLSQL', 'JAVASCRIPT', 'JAVA', 'PYTHON' )),
    value                          varchar2(4000),
    --
    last_updated_by                varchar2(255),
    last_updated_on                date,
    created_by                     varchar2(255),
    created_on                     date )
/

create index wwv_websrc_cparam_flowpage_ix      on wwv_flow_web_src_comp_params (flow_id, page_id)      compress 1
/
create index wwv_websrc_cparam_sgid_fx          on wwv_flow_web_src_comp_params (security_group_id)     compress
/
create index wwv_websrc_cparam_param_fx         on wwv_flow_web_src_comp_params (web_src_param_id)      compress
/
create index wwv_websrc_cparam_plug_fx          on wwv_flow_web_src_comp_params (page_plug_id)          compress
/
create index wwv_websrc_cparam_page_proc_fx     on wwv_flow_web_src_comp_params (page_process_id)       compress
/
create index wwv_websrc_cparam_app_proc_fx      on wwv_flow_web_src_comp_params (app_process_id)        compress
/
create index wwv_websrc_cparam_chart_fx         on wwv_flow_web_src_comp_params (jet_chart_series_id)   compress
/
create index wwv_websrc_cparam_shd_lov_fx       on wwv_flow_web_src_comp_params (shared_lov_id)         compress
/
create index wwv_websrc_cparam_sync_fx          on wwv_flow_web_src_comp_params (module_sync_step_id)   compress
/
create index wwv_websrc_cparam_am_fx            on wwv_flow_web_src_comp_params (automation_id)         compress
/
create index wwv_websrc_cparam_am_act_fx        on wwv_flow_web_src_comp_params (automation_action_id)  compress
/
create index wwv_websrc_cparam_ml_fx            on wwv_flow_web_src_comp_params (map_region_layer_id)   compress
/  
create index wwv_websrc_cparam_taskd_act_fx    on wwv_flow_web_src_comp_params (task_def_action_id)    compress
/
prompt ...done create tables
set define '^'
