set define off verify off
prompt ...wwv_flow
create or replace package wwv_flow as
--------------------------------------------------------------------------------
--
--  Copyright (c) 1999, 2022, Oracle and/or its affiliates.
--
--    NAME
--      flow.sql
--
--    DESCRIPTION
--      Oracle APEX rendering engine package specification.
--
--      This program shows and accepts Oracle APEX pages.
--      The Oracle APEX engine is also known as the flows engine.
--      Frequently called from the procedure f.
--
--    RUNTIME DEPLOYMENT: YES
--
--    MODIFIED   (MM/DD/YYYY)
--      mhichwa   08/04/1999 - Created
--
--------------------------------------------------------------------------------

----------------------------------------------------------------------------
-- F L O W   G L O B A L    V A R I A B L E S
--

    -------------------------
    -- flow public data types
    --
    type flow_vc_arr is table of varchar2(32767) index by binary_integer;
    empty_flow_vc_arr flow_vc_arr;
    --
    empty_vc_arr wwv_flow_global.vc_arr2;
    type vc_long_arr is table of varchar2(32767) index by binary_integer;

    empty_vc_long_arr vc_long_arr;

    type clob_arr      is table of clob index by binary_integer;
    type vc_assoc_arr  is table of varchar2(32767) index by varchar2(255);
    type num_assoc_arr is table of number index by varchar2(255);

    type t_component is record (
        type varchar2(30), /* APEX dictionary view name of the component where an error occurred. For example APEX_APPLICATION_PAGE_PROC */
        id   number,       /* Internal component id which triggered the error. The id is always the id of the primary application  */
        name varchar2(32767) /* Name of the component which triggered the error like the process name */
        );

    -------------------------------------
    -- Globals for generic and public use
    --
    g_image_prefix                 varchar2(32767) := wwv_flow_global.g_image_prefix;
    g_files_version                varchar2(32767);
    g_company_images               varchar2(32767);
    g_flow_images                  varchar2(32767);
    g_theme_file_prefix            varchar2(32767);
    g_id                           number;
    g_notification                 varchar2(32767);
    g_global_notification          varchar2(32767);
    g_value                        varchar2(32767);
    g_sysdate                      date;
    g_systimestamp_gmt             timestamp;
    g_boolean                      boolean := false;
    g_excel_format                 boolean := false;
    g_error_message_override       varchar2(32767);         -- if set programmatically overrides process error messages
    g_unique_page_id               number;
    g_form_painted                 boolean := false;
    g_print_success_message        varchar2(32767);
    g_return_key_1                 varchar2(32767);         -- used to return keys from dml operations
    g_return_key_2                 varchar2(32767);         -- used to return keys from dml operations
    g_base_href                    varchar2(32767);         -- BASE HREF for APEX references
    g_edition                      wwv_flow_global.t_dbms_id;

    CR                             constant varchar2(1) := unistr('\000d');
    LF                             constant varchar2(1) := unistr('\000a');
    CRLF                           constant varchar2(2) := unistr('\000d\000a');
    ---------------------
    -- translated strings
    g_nls_edit                     varchar2(32767) := 'Edit';

    ----------------------------------------------------
    -- Optimization and performance feedback information
    --
    g_form_count                   pls_integer := 0;
    g_package_instantiated         number;
    g_ok_to_cache_sql              boolean := false;
    g_page_text_generated          boolean := false;
    g_import_in_progress           boolean := false;
    g_workspace_delete_in_progress boolean := false;
    g_cached_region_count          pls_integer := 0;

    -----------------------------------
    -- Debug and error page information
    --
    g_debug                        boolean := false;        -- identifies if flow is running in debug mode
    g_unrecoverable_error          boolean := false;        -- NOTE: This global variable is deprecated,
                                                            -- use apex_application.stop_apex_engine instead!
    g_last_query_text              varchar2(32767);         -- if set is the last user sql query executed

    ----------------------------------------
    -- Authentication and login, logout info
    --
    g_user                         varchar2(32767);         -- corresponds to username used to login
    g_user_known_as                varchar2(32767);
    g_flow_schema_owner            constant varchar2(11)  := 'APEX_220100'; -- the owner of Oracle APEX
    g_login_url                    varchar2(32767);
    g_logout_url                   varchar2(32767);
    g_logo_type                    varchar2(2);
    g_logo                         varchar2(32767);
    g_logo_text                    varchar2(32767);
    g_app_builder_icon_name        varchar2(32767);
    g_use_zero_sid                 boolean := false;
    g_public_auth_scheme           boolean := false;

    -----------------------------
    -- Optimistic Locking Globals
    --
    g_md5_checksum                 varchar2(32767) := '0';

    ------------------
    -- spatial globals
    --
    g_map1                         wwv_flow_global.vc_arr2;
    g_map2                         wwv_flow_global.vc_arr2;
    g_map3                         wwv_flow_global.vc_arr2;

    ---------------------
    -- Pagination Globals
    --
    g_rownum                       pls_integer := 0;
    g_flow_current_min_row         pls_integer := 1;
    g_flow_current_max_rows        pls_integer := 10;
    g_flow_current_rows_fetched    pls_integer := 0;
    g_flow_total_row_count         pls_integer := 0;
    g_pagination_buttons_painted   boolean := false;

    ---------------------
    -- Tabular Form Globals
    --
    g_item_idx_offset              number := 0;
    g_rownum_offset                number := 0;
    g_fmap                         num_assoc_arr;
    g_fhdr                         wwv_flow_global.vc_arr2;

    ---------------------
    -- Sorting Globals
    --
    g_fsp_region_id                number := 0;

    ----------------------------
    -- Translation (NLS) Globals
    --
    g_flow_language                varchar2(32767);        -- language flow written in (primary language)
    g_flow_language_derived_from   varchar2(30);           -- how the language preference of the user is determined
    g_browser_language             varchar2(32767);        -- users language preference (set using lang_derived_from method)
    g_browser_version              varchar2(32767);        -- browser version
    g_translated_flow_id           number;                 -- flow ID for translated flow
    g_translated_page_id           number;                 -- page ID for translated flow, e.g. page_id.trans_flow_id
    g_translated_global_page_id    number;                 -- global page id for translated flow. e.g. page_id.trans_flow_id
    g_nls_date_format              varchar2(32767);        -- current database date format from nls_session_parameters table.
    g_nls_timestamp_format         varchar2(32767);        -- current database timestamp with time zone format from nls_session_parameters table.
    g_nls_timestamp_tz_format      varchar2(32767);        -- current database timestamp with time zone format from nls_session_parameters table.
    g_nls_decimal_separator        varchar2(10);           -- current database session decimal separator (derived from NLS_NUMERIC_CHARACTERS)
    g_nls_group_separator          varchar2(10);           -- current database session numeric group separator (derived from NLS_NUMERIC_CHARACTERS)
    g_nls_sort                     varchar2(32767);        -- NLS_SORT value for the current application
    g_nls_comp                     varchar2(32767);        -- NLS_COMP value for the current application
    g_direction_right_to_left      boolean := false;       -- Indicates whether page is rendered right-to-left

    ----------------------
    -- Session Information
    --
    g_new_instance                 boolean := false;

    -------------------
    -- Websheet Information
    --
    g_ws_app_id                   number;                   -- wwv_flow_ws_application pk

    -------------------
    -- Flow Information
    --
    g_flow_id                      number;                  -- flow pk
    g_flow_theme_id                number;                  -- current theme for flow
    g_flow_alias                   varchar2(32767);         -- flow alphanumeric alias
    g_flow_step_id                 number;                  -- page pk
    g_page_alias                   varchar2(32767);         -- page alias
    g_instance                     number;                  -- flow session
    g_edit_cookie_session_id       number;                  -- flow builder session
    g_page_submitted               number;                  -- set when page posted
    g_exact_substitutions_only     varchar2(1);             -- Y or N
    g_arg_names                    wwv_flow_global.vc_arr2; -- array of item names
    g_arg_values                   wwv_flow_global.vc_arr2; -- array of item values
    g_flow_name                    varchar2(32767);         -- name of flow
    g_flow_charset                 varchar2(32767);         -- used in html header
    g_date_format                  varchar2(32767);         -- Application default date format
    g_date_time_format             varchar2(32767);         -- Application date time format
    g_timestamp_format             varchar2(32767);         -- Application default timestamp format
    g_timestamp_tz_format          varchar2(32767);         -- Application default timestamp with time zone format
    g_flow_owner                   wwv_flows.owner%type;    -- for secure use wwv_flow_security.g_parse_as_schema
    g_home_link                    varchar2(32767);         -- home page for this flow
    --g_box_width                    varchar2(30);          -- obsolete ?
    g_default_page_template        varchar2(32767);         --
    g_default_dialog_page_template varchar2(32767);         -- Application Theme default dialog page template
    g_printer_friendly_template    varchar2(32767);         --
    g_error_template               varchar2(32767);         --
    g_webdb_logging                varchar2(30);            -- YES (insert entries into a log table), NO (do not do inserts)

    g_public_user                  varchar2(32767);         -- identifies public user name
    g_proxy_server                 varchar2(32767);         -- used for some regions of type url and web services
    g_no_proxy_domains             varchar2(32767);         -- used for some regions of type url and web services
    g_media_type                   varchar2(32767);         -- Media Type used in Content-Type HTTP header
    g_flow_version                 varchar2(32767);         --
    g_flow_status                  varchar2(30);            -- controls availability of flow
    g_build_status                 varchar2(30);            --
    g_friendly_url                 boolean := false;        -- should a friendly url be generated for the application
    g_rejoin_existing_sessions     varchar2(1);             -- wwv_flows.rejoin_existing_sessions
    g_request                      varchar2(32767);         -- method of submitting page
    g_sqlerrm                      varchar2(32767);         -- unexpected sql error message to be logged into log tables
    g_err_comp_type                varchar2(32767);         -- sqlerrm_component_type identifies what type of component caused the error
    g_err_comp_name                varchar2(32767);         -- sqlerrm_component_name identifies the name of the component that raised the error
    g_cache_mode                   varchar2(1) := 'D';      -- R = rendered from cache, C = Cache Created, D = Dynamic
    g_auto_time_zone               varchar2(1);             -- automatic time zone (Y or N)
    g_default_error_display_loc    varchar2(40);            -- default error display location for validations
    g_javascript_file_urls         varchar2(32767);         -- application javascript file URLs #APPLICATION_JAVASCRIPT#
    g_include_legacy_javascript    wwv_flow_user_interfaces.include_legacy_javascript%type; -- should legacy javascript functions be included?
    g_include_jquery_migrate       boolean := false;        -- should the jQuery Migrate plug-in be included?
    g_css_file_urls                varchar2(32767);         -- application css file URLs #APPLICATION_CSS#
    g_built_with_love_using_apex   boolean := false;        -- show Built with Love using Oracle APEX?
    g_email_from                   varchar2(32767);
    g_oracle_text_function         varchar2(500);
    g_favicons                     varchar2(32767);         -- (Deprecated)
    g_global_page_id               number;
    g_shared_components_scn        number;

    g_json_response                boolean default false;   -- should the output of the accept procedure be in JSON

    g_allow_feedback               boolean := false;

    -------------------
    -- Page Information
    --
    g_popup_filter                 varchar2(32767);         --
    g_printer_friendly             boolean := false;        -- if true use printer friendly page template
    g_first_field_displayable      boolean := false;        --
    g_step_name                    varchar2(32767);         -- page name
    g_step_user_interface_id       number;                  -- page user interface id
    g_step_tab_set                 varchar2(32767);         -- page current tab set
    g_step_title                   varchar2(32767);         -- page title typically becomes html page title
    g_step_media_type              varchar2(32767);         -- Media Type used in Content-Type HTTP header
    g_step_first_item              varchar2(32767);         -- name of item to put cursor in
    g_step_welcome_text            varchar2(32767);         -- wwv_flow_steps.welcome_text displayed after page template header
    g_step_box_welcome_text        varchar2(32767);         -- wwv_flow_steps.box_welcome_text displayed before #BOX_BODY# in page template body
    g_step_footer_text             varchar2(32767);         -- wwv_flow_steps.footer_text displayed before showing page template footer
    g_step_template                varchar2(32767);         -- page template
    g_page_css_classes             varchar2(32767);         -- page css classes
    g_page_template_options        varchar2(32767);         -- page template options
    g_step_required_role           varchar2(32767);         -- priv required to view page
    g_allow_duplicate_submissions  varchar2(3);             -- Y or N
    g_reload_on_submit             varchar2(1);             -- A=Always), S=Only for Success
    g_warn_on_unsaved_changes      boolean := true;
    g_head                         varchar2(32767);         -- page header for CSS, etc. #HEAD#
    g_page_onload                  varchar2(32767);         -- allows control over #ONLOAD# in page template
    g_step_javascript_file_urls    varchar2(32767);         -- page specific javascript file URLs #PAGE_JAVASCRIPT#
    g_javascript_code              varchar2(32767);         -- page specific javascript #PAGE_JAVASCRIPT#
    g_javascript_code_onload       varchar2(32767);         -- javascript code execute onload
    g_step_css_file_urls           varchar2(32767);         -- page specific CSS file URLs #PAGE_CSS#
    g_step_inline_css              varchar2(32767);         -- page specific inline CSS #PAGE_CSS#
    g_autocomplete_on_off          varchar2(3);             -- should autocomplete="off" be included in form tag
    g_include_apex_css_js_yn       varchar2(1);             -- Y is default, N does not include standard apex css and js files for mobile devices
    g_browser_cache                boolean;                 -- If false (the default) this sends the cache-control: no-store http response in wwv_flow.show
    g_browser_frame                varchar2(1);             -- If D, send X-Frame-Options:DENY header, if S, send X-Frame-Options:SAMEORIGIN header
    g_compatibility_mode           number;                  -- Compatibility Mode which contains a version number is used to determine how the runtime engine should behave when the application is executed
    g_page_transition              varchar2(20);            -- transition which is used for navigating to a new page or submitting the page
    g_popup_transition             varchar2(20);            -- transition which is used to open a popup
    g_page_mode                    varchar2(20);
    g_dialog_height                varchar2(20);
    g_dialog_width                 varchar2(20);
    g_dialog_max_width             varchar2(20);
    g_dialog_title                 varchar2(32767);
    g_dialog_css_classes           varchar2(32767);
    g_dialog_attributes            varchar2(32767);

    g_navigation_list_id           number;
    g_navigation_list_position     wwv_flow_user_interfaces.navigation_list_position%type;
    g_navigation_list_template_id  number;
    g_nav_list_template_options    wwv_flow_user_interfaces.nav_list_template_options%type;

    g_nav_bar_type                 wwv_flow_user_interfaces.nav_bar_type%type;              -- Navigation Bar Type: Render either as a List or a Navigation Bar
    g_nav_bar_list_id              number;                                                  -- Navigation Bar List ID: where type is set to List
    g_nav_bar_list_template_id     number;                                                  -- Navigation Bar List Template ID: where type is set to List
    g_nav_bar_template_options     wwv_flow_user_interfaces.nav_bar_template_options%type;  -- Navigation Bar List Template Options: where type is set to List

    -- Do only use this constant for the parameter p_compatibility_mode in wwv_flow_imp.create_flow!
    -- It's a string and would result in NLS errors if you compare it against g_compatibility_mode
    c_current_compatibility_mode   constant varchar2(4) := '21.2';

    --
    -- computation result
    --
    g_computation_result_vc       varchar2(32767);
    g_computation_result_vc_arr   wwv_flow_global.vc_arr2;
    g_computation_result_num      number;
    --
    -- validations
    --
    g_column_ids_in_error         wwv_flow_global.vc_arr2;
    -- if g_execute_validations is programmatically set before the validations
    -- are evaluated then this flag will determin if built-in validations and the
    -- validations defined for the page are evaluated. If set to NULL, the
    -- "Execute Validations" flag for buttons and select lists/radio groups will
    -- be evaluated
    g_execute_validations         boolean default null;

    --------------------------------------------
    -- Global input values for updatable reports
    --
    g_f01             wwv_flow_global.vc_arr2;
    g_f02             wwv_flow_global.vc_arr2;
    g_f03             wwv_flow_global.vc_arr2;
    g_f04             wwv_flow_global.vc_arr2;
    g_f05             wwv_flow_global.vc_arr2;
    g_f06             wwv_flow_global.vc_arr2;
    g_f07             wwv_flow_global.vc_arr2;
    g_f08             wwv_flow_global.vc_arr2;
    g_f09             wwv_flow_global.vc_arr2;
    g_f10             wwv_flow_global.vc_arr2;
    g_f11             wwv_flow_global.vc_arr2;
    g_f12             wwv_flow_global.vc_arr2;
    g_f13             wwv_flow_global.vc_arr2;
    g_f14             wwv_flow_global.vc_arr2;
    g_f15             wwv_flow_global.vc_arr2;
    g_f16             wwv_flow_global.vc_arr2;
    g_f17             wwv_flow_global.vc_arr2;
    g_f18             wwv_flow_global.vc_arr2;
    g_f19             wwv_flow_global.vc_arr2;
    g_f20             wwv_flow_global.vc_arr2;
    g_f21             wwv_flow_global.vc_arr2;
    g_f22             wwv_flow_global.vc_arr2;
    g_f23             wwv_flow_global.vc_arr2;
    g_f24             wwv_flow_global.vc_arr2;
    g_f25             wwv_flow_global.vc_arr2;
    g_f26             wwv_flow_global.vc_arr2;
    g_f27             wwv_flow_global.vc_arr2;
    g_f28             wwv_flow_global.vc_arr2;
    g_f29             wwv_flow_global.vc_arr2;
    g_f30             wwv_flow_global.vc_arr2;
    g_f31             wwv_flow_global.vc_arr2;
    g_f32             wwv_flow_global.vc_arr2;
    g_f33             wwv_flow_global.vc_arr2;
    g_f34             wwv_flow_global.vc_arr2;
    g_f35             wwv_flow_global.vc_arr2;
    g_f36             wwv_flow_global.vc_arr2;
    g_f37             wwv_flow_global.vc_arr2;
    g_f38             wwv_flow_global.vc_arr2;
    g_f39             wwv_flow_global.vc_arr2;
    g_f40             wwv_flow_global.vc_arr2;
    g_f41             wwv_flow_global.vc_arr2;
    g_f42             wwv_flow_global.vc_arr2;
    g_f43             wwv_flow_global.vc_arr2;
    g_f44             wwv_flow_global.vc_arr2;
    g_f45             wwv_flow_global.vc_arr2;
    g_f46             wwv_flow_global.vc_arr2;
    g_f47             wwv_flow_global.vc_arr2;
    g_f48             wwv_flow_global.vc_arr2;
    g_f49             wwv_flow_global.vc_arr2;
    g_f50             wwv_flow_global.vc_arr2;
    g_fcs             wwv_flow_global.vc_arr2;
    g_fcud            wwv_flow_global.vc_arr2;
    g_frowid          wwv_flow_global.vc_arr2;
    g_survey_map      varchar2(32767);

    --------------------------------------------
    -- Global input variables for AJAX utilities
    --
    g_widget_name       varchar2(32767);
    g_widget_mod        varchar2(32767);
    g_widget_action     varchar2(32767);
    g_widget_action_mod	varchar2(32767);
    g_widget_num_return	varchar2(32767);
    g_widget_view_mode  varchar2(32767);

    g_x01             varchar2(32767);
    g_x02             varchar2(32767);
    g_x03             varchar2(32767);
    g_x04             varchar2(32767);
    g_x05             varchar2(32767);
    g_x06             varchar2(32767);
    g_x07             varchar2(32767);
    g_x08             varchar2(32767);
    g_x09             varchar2(32767);
    g_x10             varchar2(32767);
    g_x11             varchar2(32767);
    g_x12             varchar2(32767);
    g_x13             varchar2(32767);
    g_x14             varchar2(32767);
    g_x15             varchar2(32767);
    g_x16             varchar2(32767);
    g_x17             varchar2(32767);
    g_x18             varchar2(32767);
    g_x19             varchar2(32767);
    g_x20             varchar2(32767);
    g_clob_01         clob;

    ------------------------------------
    -- Session Timeout
    --

    g_max_session_length_sec        number;
    g_max_session_idle_sec          number;
    g_session_timeout_warning_sec   number;

    ----------------------------------------------------------------------------
    -- Global array of form item values fetched by automated row fetch processes
    -- and other globals previously in wwv_flow_dml
    --
    g_column_values             wwv_flow.flow_vc_arr;
    g_clob_column_values        wwv_flow.clob_arr;
    g_dml_blob                  blob;
    g_dml_mimetype              varchar2(32767);
    g_dml_filename              varchar2(32767);
    g_dml_charset               varchar2(32767);
    g_dml_last_updated_date     date;
    g_dml_clob_text             clob;
    g_dml_varchar32767_text     varchar2(32767);
    g_dml_rowid                 varchar2(32767);
    g_dml_blob_length           number := 0;

    -----------------------------
    -- Stop APEX Engine
    -----------------------------
    c_stop_apex_engine_no constant pls_integer := -20876;
    e_stop_apex_engine    exception;
    pragma exception_init(e_stop_apex_engine, -20876);

    ---------------------------
    -- Maximum DB VARCHAR2 size
    --
$if sys.dbms_db_version.ver_le_10_2 $then
    c_max_size_vc2 constant number := 4000;
$elsif sys.dbms_db_version.ver_le_11_1 $then
    c_max_size_vc2 constant number := 4000;
$elsif sys.dbms_db_version.ver_le_11_2 $then
    c_max_size_vc2 constant number := 4000;
$else
    c_max_size_vc2 constant number := 32767;
$end

    ------------------------------------------------
    -- For Internal use only, will be removed in 5.2
    ------------------------------------------------
    g_json wwv_flow_global.vc_arr2;

--==============================================================================
-- Return global flags that were previously exposed as variables
--==============================================================================
function g_in_accept
    return boolean;
function g_in_process
    return boolean;

----------------------------------------------------------------------------
-- S H O W
--
-- This procedure is the entry point for the display of Oracle APEX pages.
--
--
-- p_request              -- Request which can be the tab pressed, the button pressed, an
--                           arbitrary value etc.
-- p_instance             -- Flow Session ID, must be numeric.
-- p_flow_id              -- Flow ID, must be numeric.
-- p_flow_step_id         -- Page ID, must be numeric (for example: 1)
-- p_debug                -- If "YES" then flows will display debug messages.
-- p_arg_names            -- Comma seperated list of flow item names.  Item names should
--                           have corresponding (p_arg_values) values.  For example:
--                           p_arg_names => 'A,B,C'
--                           p_arg_values=> '1,2,3'
--                           This assumes that a flow or page item called A, B, and C
--                           exists.  When called a value of 1 will be assigned to A,
--                           2 to B, etc.
-- p_arg_values           -- Comma seperated list of values that corresponds to a
--                           comma seperated list of names (p_arg_names).  The session
--                           state for the flow items identified will be set to these
--                           values.
-- p_arg_name             -- use when passing one item name, itentifies a single item name
-- p_arg_value            -- use when passing one item value, identifies a single item value
-- p_clear_cache          -- Comma seperated list of pages (e.g. 2,3,4).
--                           Sets to null the values of any flow item declared for a
--                           list of pages.
-- p_box_border           -- obsolete
-- p_printer_friendly     -- If "YES" then show page using printer friendly template.
--                           Do not generate tabs or nav bar icons.
-- p_trace                -- If "YES", generate trace file for debugging and performance tuning
-- p_company              -- ID of company (security group id) must be numeric
-- p_md5_checksum         -- checksum to prevent lost updates
-- p_last_button_pressed  -- facilitates reference to :flow_last_button_pressed
-- p_dialog_cs            -- check sum for modal dialog page to allow use of iframe when Embed in Frame security setting is DENY

procedure show (
    p_request             in varchar2   default null,
    p_instance            in varchar2   default null,
    p_flow_id             in varchar2   default null,
    p_flow_step_id        in varchar2   default null,
    p_debug               in varchar2   default 'NO',
    p_arg_names           in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_arg_values          in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_clear_cache         in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_box_border          in varchar2   default '0',
    p_printer_friendly    in varchar2   default 'NO',
    p_trace               in varchar2   default 'NO',
    p_company             in number     default null,
    p_md5_checksum        in varchar2   default '0',
    p_last_button_pressed in varchar2   default null,
    p_arg_name            in varchar2   default null,
    p_arg_value           in varchar2   default null,
    f01                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f02                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f03                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f04                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f05                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f06                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f07                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f08                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f09                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f10                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f11                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f12                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f13                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f14                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f15                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f16                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f17                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f18                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f19                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f20                   in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_widget_name         in varchar2   default null,
    p_widget_mod          in varchar2   default null,
    p_widget_action       in varchar2   default null,
    p_widget_action_mod   in varchar2   default null,
    p_widget_num_return   in varchar2   default null,
    p_widget_view_mode    in varchar2   default null,
    p_fsp_region_id       in number     default null,
    p_pg_min_row          in number     default null,
    p_pg_max_rows         in number     default null,
    p_pg_rows_fetched     in number     default null,
    p_time_zone           in varchar2   default null,
    x01                   in varchar2   default null,
    x02                   in varchar2   default null,
    x03                   in varchar2   default null,
    x04                   in varchar2   default null,
    x05                   in varchar2   default null,
    x06                   in varchar2   default null,
    x07                   in varchar2   default null,
    x08                   in varchar2   default null,
    x09                   in varchar2   default null,
    x10                   in varchar2   default null,
    x11                   in varchar2   default null,
    x12                   in varchar2   default null,
    x13                   in varchar2   default null,
    x14                   in varchar2   default null,
    x15                   in varchar2   default null,
    x16                   in varchar2   default null,
    x17                   in varchar2   default null,
    x18                   in varchar2   default null,
    x19                   in varchar2   default null,
    x20                   in varchar2   default null,
    p_json                in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_clob_01             in clob       default null,
    p_lang                in varchar2   default null,
    p_territory           in varchar2   default null,
    p_cs                  in varchar2   default null,
    j                     in varchar2   default null,
    XMLCallDate           in varchar2   default null,
    p_dialog_cs           in varchar2   default null,
    p_context             in varchar2   default null );


----------------------------------------------------------------------------
-- A C C E P T
--
-- This procedure accepts virtually every flow page.
-- Reference show procedure for input argument descriptions.
--
--
--
--

procedure accept (
    p_request          in varchar2 default null,
    p_instance         in varchar2 default null,
    p_flow_id          in varchar2 default null,
    p_company          in number   default null,
    p_flow_step_id     in varchar2 default null,
    p_reload_on_submit in varchar2 default 'A',
    p_arg_names        in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_arg_values       in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_json             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_files            in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f01             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f02             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f03             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f04             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f05             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f06             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f07             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f08             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f09             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f10             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f11             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f12             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f13             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f14             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f15             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f16             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f17             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f18             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f19             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f20             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f21             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f22             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f23             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f24             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f25             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f26             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f27             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f28             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f29             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f30             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f31             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f32             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f33             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f34             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f35             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f36             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f37             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f38             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f39             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f40             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f41             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f42             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f43             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f44             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f45             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f46             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f47             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f48             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f49             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f50             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    fcs             in wwv_flow_global.vc_arr2 default empty_vc_arr,
    fmap            in wwv_flow_global.vc_arr2 default empty_vc_arr,
    fhdr            in wwv_flow_global.vc_arr2 default empty_vc_arr,
    fcud            in wwv_flow_global.vc_arr2 default empty_vc_arr,
    frowid          in wwv_flow_global.vc_arr2 default empty_vc_arr,
    x01             in varchar2   default null,
    x02             in varchar2   default null,
    x03             in varchar2   default null,
    x04             in varchar2   default null,
    x05             in varchar2   default null,
    x06             in varchar2   default null,
    x07             in varchar2   default null,
    x08             in varchar2   default null,
    x09             in varchar2   default null,
    x10             in varchar2   default null,
    x11             in varchar2   default null,
    x12             in varchar2   default null,
    x13             in varchar2   default null,
    x14             in varchar2   default null,
    x15             in varchar2   default null,
    x16             in varchar2   default null,
    x17             in varchar2   default null,
    x18             in varchar2   default null,
    x19             in varchar2   default null,
    x20             in varchar2   default null,
    p_listener      in wwv_flow_global.vc_arr2 default empty_vc_arr, -- used to communicate with apex listner
    p_map1          in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_map2          in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_map3          in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_debug              in varchar2 default 'NO',
    p_trace              in varchar2 default 'NO',
    p_page_submission_id in varchar2 default null,
    p_time_zone          in varchar2 default null,
    p_ignore_01     in varchar2 default null,
    p_ignore_02     in varchar2 default null,
    p_ignore_03     in varchar2 default null,
    p_ignore_04     in varchar2 default null,
    p_ignore_05     in varchar2 default null,
    p_ignore_06     in varchar2 default null,
    p_ignore_07     in varchar2 default null,
    p_ignore_08     in varchar2 default null,
    p_ignore_09     in varchar2 default null,
    p_ignore_10     in varchar2 default null,
    p_lang          in varchar2 default null,
    p_territory     in varchar2 default null,
    p_dialog_cs     in varchar2 default null,
    p_context       in varchar2 default null );

--==============================================================================
-- This procedure is used for Ajax requests where p_json contains a JSON document
-- with all the necessary information to dispatch the Ajax request(s) to the
-- different components.
--==============================================================================
procedure ajax (
    p_flow_id           in varchar2 default null,
    p_flow_step_id      in varchar2 default null,
    p_instance          in varchar2 default null,
    p_debug             in varchar2 default 'NO',
    p_trace             in varchar2 default 'NO',
    p_request           in varchar2 default null,
    p_arg_names         in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_arg_values        in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_json              in wwv_flow_global.vc_arr2 default empty_vc_arr,
    x01                 in varchar2 default null,
    x02                 in varchar2 default null,
    x03                 in varchar2 default null,
    x04                 in varchar2 default null,
    x05                 in varchar2 default null,
    x06                 in varchar2 default null,
    x07                 in varchar2 default null,
    x08                 in varchar2 default null,
    x09                 in varchar2 default null,
    x10                 in varchar2 default null,
    x11                 in varchar2 default null,
    x12                 in varchar2 default null,
    x13                 in varchar2 default null,
    x14                 in varchar2 default null,
    x15                 in varchar2 default null,
    x16                 in varchar2 default null,
    x17                 in varchar2 default null,
    x18                 in varchar2 default null,
    x19                 in varchar2 default null,
    x20                 in varchar2 default null,
    f01                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f02                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f03                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f04                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f05                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f06                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f07                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f08                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f09                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f10                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f11                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f12                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f13                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f14                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f15                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f16                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f17                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f18                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f19                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    f20                 in wwv_flow_global.vc_arr2 default empty_vc_arr,
    -- For backward compatibility, don't use for new apps
    p_clob_01           in clob     default null,
    p_widget_name       in varchar2 default null,
    p_widget_mod        in varchar2 default null,
    p_widget_action     in varchar2 default null,
    p_widget_action_mod in varchar2 default null,
    p_widget_num_return in varchar2 default null,
    p_widget_view_mode  in varchar2 default null,
    p_fsp_region_id     in number   default null,
    p_clear_cache       in wwv_flow_global.vc_arr2 default empty_vc_arr,
    p_pg_min_row        in number   default null,
    p_pg_max_rows       in number   default null,
    p_pg_rows_fetched   in number   default null,
    p_context           in varchar2 default null );

--==============================================================================
-- Returns all text messages of the APEX Runtime Engine and the specified
-- application which are marked as JavaScript messages.
-- If p_names is specified, only those text messages will be returned. An entry
-- in p_names can use wildcards of the LIKE operator.
-- Those messages don't have to be marked as JavaScript messages.
--==============================================================================
procedure js_messages (
    p_app_id    in varchar2 default null,
    p_lang      in varchar2 default null,
    p_version   in varchar2 default null,
    p_builder   in varchar2 default 'N',
    p_names     in wwv_flow_global.vc_arr2 default wwv_flow_global.c_empty_vc_arr2 );

----------------------------------------------------------------------------
-- H E L P   S Y S T E M
--

procedure help (
    --
    -- Returns Page and Column level help from flow builder meta data
    -- repository.
    --
    -- Arguments:
    --    p_flow_id        = flow ID
    --    p_flow_step_id   = page ID
    --    p_show_item_help = YES (include item level help), NO do not show item level help
    --
    p_request        in varchar2 default null,
    p_flow_id        in varchar2 default null,
    p_flow_step_id   in varchar2 default null,
    p_show_item_help in varchar2 default 'YES',
    p_show_regions   in varchar2 default 'YES',
    --
    p_before_page_html     in varchar2 default '<p>',
    p_after_page_html      in varchar2 default null,
    p_before_region_html   in varchar2 default null,
    p_after_region_html    in varchar2 default '</td></tr></table></p>',
    p_before_prompt_html   in varchar2 default '<p><strong>',
    p_after_prompt_html    in varchar2 default '</strong></p>:&nbsp;',
    p_before_item_html     in varchar2 default null,
    p_after_item_html      in varchar2 default null)
    ;







----------------------------------------------------------------------------
-- U T I L I T I E S
--

function do_substitutions (
    --
    -- Perform substitutions of ampersand prefixed flow items with
    -- current flow session state for current user and current session.
    --
    p_string                       in varchar2 default null,
    p_sub_type                     in varchar2 default 'SQL',
    p_perform_oracle_substitutions in boolean default false )
    return varchar2
    ;

function trim_sql (
    --
    -- Given a SQL statement , trim trailing and leading
    -- white spaces.  Optionally perform session state substitutions
    -- as well as ensuring the statement ends in a semicolon.
    --
    p_sql               in varchar2 default null,
    p_ends_in_semicolon in boolean default false,
    p_do_substitutions  in boolean default true)
    return varchar2
    ;

function trim_sql (
    --
    -- This function gets a SQL statement ready for execution
    -- Function is overloaded; p_owner may be provided in order to cause package global
    -- to be temporarily replaced with the value of p_owner in package state during
    -- the execution of this function and until its return to the caller.
    --
    p_sql               in varchar2 default null,
    p_ends_in_semicolon in boolean default false,
    p_do_substitutions  in boolean default true,
    p_owner             in varchar2)
    return varchar2
    ;

function trim_nl_sp_tab (
    -- This function removes all whitespace, tab, and new line characters at the begin and end of the input string
    p_string varchar2 )
    return varchar2;

procedure reset_g_nls
    ;

procedure set_g_nls_date_format
    ;

procedure reset_g_nls_date_format
    ;


procedure set_g_nls_decimal_separator
    ;

procedure reset_g_nls_decimal_separator
    ;

function get_nls_decimal_separator return varchar2
    ;

function get_nls_group_separator return varchar2
    ;

function get_translated_app_id return number
    ;

procedure set_g_base_href
    ;

procedure reset_g_base_href
    ;

function get_g_base_href return varchar2
    ;

function get_page_alias return varchar2
    ;

function is_custom_auth_page return boolean
    ;

function get_application_id return number
    ;

function get_page_id return number
    ;

function get_session_id return number
    ;

function get_security_group_id return number
    ;

function is_pressed_button (
    p_button_id in number )
    return boolean;

----------------------------------------------------------------------------
-- E R R O R   H A N D L I N G
--
--==============================================================================
-- Returns the number of inline validation errors.
-- Note: That's a compatibility wrapper,
--       use wwv_flow_error.get_validation_error_count instead.
--==============================================================================
function g_inline_validation_error_cnt return pls_integer;

--==============================================================================
-- Signals the APEX engine to stop further processing and immediately exit so
-- that no additional HTML code is added to the HTTP buffer.
--
-- Note: This procedure will internally raise the exception e_stop_apex_engine.
--       You have to reraise that exception if you use a WHEN OTHERS exception
--       handler!
--
-- Example:
--
-- owa_util.redirect_url('http://apex.oracle.com');
-- apex_application.stop_apex_engine;
--
-- Complex example with a when others exception handler:
--
-- begin
--     ... code which can raise an exception ...
--     owa_util.redirect_url('http://apex.oracle.com');
--     apex_application.stop_apex_engine;
-- exception
--     when apex_application.e_stop_apex_engine then
--         raise; -- reraise the stop APEX engine exception
--     when others then
--         ...; -- code to handle the exception
-- end;
--
--==============================================================================
procedure stop_apex_engine;

----------------------------------------
-- I N T E R N A L     U T I L I T I E S
--
-- Internal utilities used by the flow engine runtime
-- that are not intened and are not useful to the
-- flows programmer.
--

procedure set_component (
    p_type in varchar2,
    p_id   in number   default null,
    p_name in varchar2 default null );

function get_component return t_component;

procedure clear_component;

function paint_formOpen
    -- Return the HTML form open tag given current flow state.
    return varchar2
    ;


function get_form_close_tag return varchar2;

procedure s (
    p in varchar2 default null)
    ;

----------------------------------------------------------------------------
-- D E B U G G I N G
--
procedure debug (
    -- Given a string this will result in the generation of a debug entry
    p_string         in varchar2 default null)
    ;


function get_elapsed_time
    -- Get the elapsed time from package instantiation
    return number;




----------------------------------------------------------------------------
-- S E S S I O N   S T A T E   M A N A G E M E N T
--
-- The following routines can be called to read and write session state.
--
--
--
--

function get_next_session_id
    -- Get integer ID values, session ID is a sequence, unique ID is a sequence
    -- with a random number which produces a virtual global unique ID.
    return number
    ;

function get_unique_id
    -- Return a number which is virually globally unique.
    return number
    ;

procedure clear_page_cache (
    -- Reset all cached items for a given page to null
    p_flow_page_id in number default null)
    ;

procedure clear_page_caches (
    -- Reset all cached items for pages in array to null
    p_flow_page_id in wwv_flow_global.vc_arr2 default empty_vc_arr)
    ;

procedure clear_flow_cache (
    -- For the current session remove session state for the given flow.
    -- Requires g_instance to be set to the current flows session.
    p_flow_id in varchar2 default null)
    ;

procedure clear_app_cache (
    -- For the current session remove session state for the given flow.
    -- Requires g_instance to be set to the current flows session.
    p_app_id in varchar2 default null)
    ;

procedure clear_user_cache
    -- For the current users session remove session state and flow system preferences.
    -- Run this procedure if you reuse session IDs and want to run flows without
    -- the benifit of existing session state.  Requires g_instance to be set to the
    -- current flows session.
    ;

function find_item_id (
    -- Given a flow page or flow level items name return its numeric identifier.
    p_name in varchar2 default null)
    return number
    ;

function find_item_name (
    -- Given a flow page or flow level items numeric identifier return the items name.
    p_id in number default null)
    return varchar2
    ;

procedure update_cache_with_write (
    -- For the current session update an items cached value.  This update is persistent
    -- for the life of the flow session, unless the session state is cleared or updated.
    p_name    in varchar2 default null,
    p_value   in varchar2 default null)
    ;

procedure reset_security_check
    -- Security checks are cached to increase performance, this procedure allows you to
    -- undo the caching and thus require all security checks to be revalidated for the
    -- current user.  Use this routine if you allow a user to change "responcibilities"
    -- within an application, thus changing their authentication scheme.
    ;

function public_role_check (
    p_role      in varchar2 default null,
    p_component in varchar2 default null)
    return boolean
    ;

function public_security_check (
    -- Given the name of a flow security scheme determine if the current user
    -- passes the security check.
    p_security_scheme in varchar2)
    return boolean
    ;

function fetch_flow_item(
    -- Given a flow-level item name, locate item in current or specified
    -- flow and current or specified session and return item value.
    p_item         in varchar2,
    p_flow         in number default null,
    p_instance     in number default null)
    return varchar2
    ;

function fetch_app_item(
    -- Given a flow-level item name, locate item in current or specified
    -- flow and current or specified session and return item value.
    p_item         in varchar2,
    p_app          in number default null,
    p_instance     in number default null)
    return varchar2
    ;

------------------------------------------------------------------
-- V I R T U A L   P R I V A T E   D A T A B A S E   S U P P O R T
--
-- flows are owned by companies which are identified by a security
-- group ID.  The flow meta data repository is "sliced up" by
-- the security group id (sgid).
--

function get_sgid return number
    -- Given the current users context return the security group ID.
    ;
function get_browser_version return varchar2;
     -- return browser versiob

function get_company_name (
    p_escape in boolean default true
) return varchar2
    -- Given the current users context return the company name.
    ;

------------------------------------------------------------------
-- Stateful processes
--

function process_state(
    p_process_id in number)
    return varchar2
    ;

procedure reset_page_process(
    p_process_id in number)
    ;

procedure reset_page_processess(
    p_page_id in number)
    ;


------------------------------------------------------------------
-- A U T H E N T I C A T I O N
--

function get_custom_auth_login_url return varchar2
    -- for use with custom authentication
    ;

function replace_cgi_env(
    p_in varchar2)
    return varchar2
    ;

function get_home_link_url return varchar2
    -- internal use only
    ;

--==============================================================================
-- Public procedure used as "Path Alias Procedure" (see http://download.oracle.com/docs/cd/A97335_02/apps.102/a90099/feature.htm#1007126
-- to translate a friendly URL of APEX into a related wwv_flow.show or
-- wwv_flow_file_mgr.get_file call.
--==============================================================================
procedure resolve_friendly_url (
    p_path in varchar2 );

------------------------------------------------------------------
-- Deprecated, obsolete procedures and functions.
-- Don't use them anymore!
--

procedure null_page_cache (
    p_flow_page_id in number default null)
    ;

procedure null_step_cache (
    p_flow_step_id in number default null)
    ;

procedure null_page_caches (
    p_flow_page_id in wwv_flow_global.vc_arr2 default empty_vc_arr)
    ;

procedure null_step_caches (
    p_flow_step_id in wwv_flow_global.vc_arr2 default empty_vc_arr)
    ;

procedure show_error_message (
    p_message  in varchar2 default 'Error',
    p_footer   in varchar2 default null,
    p_query    in varchar2 default null)
    ;

function get_application_id (
    p_application_id_or_alias in varchar2,
    p_security_group_id       in number   default null )
    return number;

procedure set_ppr_query_session_state (p_region_id in number);

end wwv_flow;
/
show errors

set define '^'
