set define '^'
prompt Creating APEX dictionary views

Rem  Copyright (c) 2006, 2022, Oracle and/or its affiliates.
Rem
Rem    NAME
Rem      apex_views.sql
Rem
Rem    DESCRIPTION
Rem      data dictionary views of Oracle APEX meta data
Rem
Rem    MODIFIED   (MM/DD/YYYY)
Rem    mhichwa     06/22/2006 - Created

col APPUN noprint new_val APPUN
select sys_context('USERENV','CURRENT_SCHEMA') APPUN from sys.dual;

--------------------------------------------
-- B A S E   V I E W S
--

--==============================================================================
prompt ...wwv_flow_authorized
create or replace view wwv_flow_authorized
as
select w.short_name              as workspace,
       w.display_name            as workspace_display_name,
       w.provisioning_company_id as workspace_id,
       f.id                      as application_id,
       f.name                    as application_name,
       f.rejoin_existing_sessions,
       f.authentication_id,
       f.flow_language
  from ( select /*+ no_merge cardinality(t 1) */
                nvl(wwv_flow.get_sgid,0)              sgid,
                sys_context('USERENV','CURRENT_USER') cu,
                sys_context('USERENV','NLS_SORT')     nls_sort
           from sys.dual t ) d,
       wwv_flow_companies w,
       wwv_flows f
where (   f.security_group_id = d.sgid
       or (    f.owner  = d.cu
           and (   d.nls_sort = 'BINARY'
                or nlssort(f.owner,'NLS_SORT=BINARY')=nlssort(d.cu,'NLS_SORT=BINARY')))
       or (select wwv_flow_security.has_apex_admin_read_role_yn(d.cu) from sys.dual)='Y' )
  and exists ( select null
                 from wwv_flow_company_schemas s
                where f.security_group_id = s.security_group_id
                  and f.owner             = s.schema
                  and (   d.nls_sort = 'BINARY'
                       or nlssort(f.owner,'NLS_SORT=BINARY')=nlssort(s.schema,'NLS_SORT=BINARY')))
  and w.provisioning_company_id != 0
  and f.security_group_id       =  w.provisioning_company_id
  and (  d.sgid != 0
      or nvl(f.build_status, 'x') != 'RUN_AND_HIDDEN')
/

--==============================================================================
-- Alternative version that allows DML on the main view. Note that predicate
-- pushdown does not automatically work on this view. The outer query can use
-- the push_pred hint, though.
--==============================================================================
prompt ...wwv_flow_authorized_for_dml
create or replace view wwv_flow_authorized_for_dml
as
select w.short_name              as workspace,
       w.display_name            as workspace_display_name,
       w.provisioning_company_id as workspace_id,
       f.id                      as application_id,
       f.name                    as application_name,
       f.rejoin_existing_sessions,
       f.authentication_id,
       f.flow_language
  from ( select /*+ no_merge cardinality(t 1) */
                nvl(wwv_flow.get_sgid,0)              sgid,
                sys_context('USERENV','CURRENT_USER') cu,
                sys_context('USERENV','NLS_SORT')     nls_sort
           from table(^APPUN..wwv_flow_t_varchar2('X')) t ) d,
       wwv_flow_companies w,
       wwv_flows f
where (   f.security_group_id = d.sgid
       or (    f.owner  = d.cu
           and (   d.nls_sort = 'BINARY'
                or nlssort(f.owner,'NLS_SORT=BINARY')=nlssort(d.cu,'NLS_SORT=BINARY')))
       or (select wwv_flow_security.has_apex_admin_read_role_yn(d.cu) from sys.dual)='Y' )
  and exists ( select null
                 from wwv_flow_company_schemas s
                where f.security_group_id = s.security_group_id
                  and f.owner             = s.schema
                  and (   d.nls_sort = 'BINARY'
                       or nlssort(f.owner,'NLS_SORT=BINARY')=nlssort(s.schema,'NLS_SORT=BINARY')))
  and w.provisioning_company_id != 0
  and f.security_group_id       =  w.provisioning_company_id
  and (  d.sgid != 0
      or nvl(f.build_status, 'x') != 'RUN_AND_HIDDEN')
/

--==============================================================================
prompt ...wwv_flow_authorized_and_4411
create or replace view wwv_flow_authorized_and_4411
as
select w.short_name              as workspace,
       w.display_name            as workspace_display_name,
       w.provisioning_company_id as workspace_id,
       f.id                      as application_id,
       f.name                    as application_name
  from ( select /*+ no_merge cardinality(t 1) */
                nvl(wwv_flow.get_sgid,0)              sgid,
                sys_context('USERENV','CURRENT_USER') cu,
                sys_context('USERENV','NLS_SORT')     nls_sort
           from sys.dual t ) d,
       wwv_flow_companies w,
       wwv_flows f
where (   f.security_group_id = d.sgid
      or ( f.security_group_id = 10 and f.id = 4411 )
      or (    f.owner  = d.cu
          and (   d.nls_sort = 'BINARY'
               or nlssort(f.owner,'NLS_SORT=BINARY')=nlssort(d.cu,'NLS_SORT=BINARY')))
      or (select wwv_flow_security.has_apex_admin_read_role_yn(d.cu) from sys.dual)='Y' )
  and exists ( select null
                 from wwv_flow_company_schemas s
                where f.security_group_id = s.security_group_id
                  and f.owner             = s.schema
                  and (   d.nls_sort = 'BINARY'
                       or nlssort(f.owner,'NLS_SORT=BINARY')=nlssort(s.schema,'NLS_SORT=BINARY')))
  and w.provisioning_company_id != 0
  and f.security_group_id       =  w.provisioning_company_id
  and (  d.sgid != 0
      or nvl(f.build_status, 'x') not in ('RUN_ONLY', 'RUN_AND_HIDDEN'))
/

--==============================================================================
prompt ...apex_instance_parameters
create or replace view apex_instance_parameters (
    name,
    value,
    created_on,
    last_updated_on )
as
select name,
       case
       when name in ('WALLET_PWD','SMTP_PASSWORD') and value is not null then '***'
       else value
       end,
       created_on,
       last_updated_on
  from wwv_flow_platform_prefs,
       wwv_flow_current_sgid sgid
 where sgid.has_apex_admin_read_role_yn = 'Y'
/

--------------------------------------------
-- W O R K S P A C E S
--
prompt ...apex_workspace views
prompt ...apex_workspaces
create or replace view apex_workspaces as
select
    w.short_name                                        workspace,
    w.display_name                                      workspace_display_name,
    w.workspace_description                             workspace_description,
    w.workspace_service_id                              workspace_service_id,
    w.source_identifier                                 source_identifier,
    w.path_prefix                                       path_prefix,
    w.team_dev_fs_limit                                 team_dev_fs_limit,
    w.issue_fs_limit                                    issue_fs_limit,
    --w.FIRST_SCHEMA_PROVISIONED                          first_schema_provisioned,
    --
    count(distinct s.schema)                            schemas,
    max(w.LAST_LOGIN)                                   last_purged_session,
    w.builder_notification_message                      workspace_announcement,
    nvl(w.env_banner_yn,'N')                            workspace_banner_yn,
    w.env_banner_label                                  workspace_banner_label,
    w.env_banner_color                                  workspace_banner_color,
    nvl(w.env_banner_pos,'LEFT')                        workspace_banner_pos,
    w.rm_consumer_group                                 rm_consumer_group,
    --
    nvl(w.allow_app_building_yn,'Y')                    allow_app_building_yn,
    null                                                allow_packaged_app_ins_yn,  -- obsolete, used by ORDS DB API
    nvl(w.allow_sql_workshop_yn,'Y')                    allow_sql_workshop_yn,
    null                                                allow_websheet_dev_yn,
    nvl(w.allow_team_development_yn,'Y')                allow_team_development_yn,
    nvl(w.allow_team_dev_files_yn,'Y')                  allow_team_dev_files_yn,
    nvl(w.allow_issue_files_yn,'Y')                     allow_issue_files_yn,
    nvl(w.allow_to_be_purged_yn,'Y')                    allow_to_be_purged_yn,
    nvl(w.allow_restful_services_yn,'Y')                allow_restful_services_yn,
    --
    (select count(*)
     from wwv_flow_sessions$
     where security_group_id =
           w.PROVISIONING_COMPANY_ID)                   sessions,
    --
    (select count(*)
     from   wwv_flows
     where  security_group_id =
            w.PROVISIONING_COMPANY_ID)                  applications,
    --
    (select count(*)
     from   wwv_flow_steps
     where  security_group_id =
            w.PROVISIONING_COMPANY_ID)                  application_pages,
    --
    (select count(*)
     from   wwv_flow_fnd_user
     where  security_group_id =
            w.PROVISIONING_COMPANY_ID)                  apex_users,
    --
    (select count(*)
     from wwv_flow_developers d, wwv_flow_fnd_user u
     where (instr(nvl(d.developer_role,'x'),'EDIT') > 0 or
            instr(nvl(d.developer_role,'x'),'ADMIN') > 0 ) and
           d.security_group_id = w.PROVISIONING_COMPANY_ID and
           u.security_group_id = w.PROVISIONING_COMPANY_ID and
           d.userid = u.user_name)
                                                        apex_developers,
    --
    (select count(*)
     from   wwv_flow_developers d, wwv_flow_fnd_user u
     where  instr(nvl(d.developer_role,'x'),'ADMIN') > 0  and
           d.security_group_id = w.PROVISIONING_COMPANY_ID and
           u.security_group_id = w.PROVISIONING_COMPANY_ID and
           d.userid = u.user_name)
                                                        apex_workspace_administrators,
    --
    (select count(*)
     from   wwv_flow_company_filestats
     where  security_group_id =
            w.PROVISIONING_COMPANY_ID)                  files,
    --
    (select count(*)
     from    WWV_FLOW_FILE_OBJECTS$
     where   security_group_id =
            w.PROVISIONING_COMPANY_ID and
            FILE_TYPE = 'SCRIPT' and
            to_char(deleted_as_of,
      'MM.DD.YYYY') = '01.01.0001')               sql_scripts,
    --
    (select count(*)
     from   WWV_FLOW_MESSAGES$
     where  security_group_id =
            w.PROVISIONING_COMPANY_ID)                  translation_messages,
    --
    (select sum(file_bytes)
     from   wwv_flow_company_filestats
     where  security_group_id =
            w.PROVISIONING_COMPANY_ID)                  file_storage,
     w.max_file_bytes                                   file_storage_max,
    --
    nvl(
        (select max(time_stamp) d
        from wwv_flow_activity_log l
        where l.security_group_id = w.PROVISIONING_COMPANY_ID and l.time_stamp > sysdate - 1),
        (select max(log_day) d
        from wwv_flow_log_history h
        where h.workspace_id = w.PROVISIONING_COMPANY_ID)
        ) last_logged_page_view,
    --
     (select count(*)
     from   wwv_flow_activity_log l
     where  security_group_id =
            w.PROVISIONING_COMPANY_ID and
            l.TIME_STAMP > sysdate - 14)                page_views,
     --
     w.files_version,
     w.max_email_messages                               email_maximum,
     w.max_webservice_requests                          webservice_request_maximum,
     --
     max_session_length_sec                             maximum_session_life_seconds,
     max_session_idle_sec                               maximum_session_idle_seconds,
     session_timeout_warning_sec                        session_timeout_warn_seconds,
     qos_max_workspace_requests                         max_workspace_requests,
     qos_max_session_requests                           max_session_requests,
     qos_max_session_kill_timeout                       max_session_kill_timeout,
     --
     w.PROVISIONING_COMPANY_ID                          workspace_id,
     w.created_on                                       created_on,
     nvl(w.assigned_on,w.created_on)                    assigned_on
from
     wwv_flow_companies       w,
     wwv_flow_company_schemas s,
     wwv_flow_current_sgid    sgid
where (   sgid.has_apex_admin_read_role_yn = 'Y'
       or sgid.security_group_id           = w.provisioning_company_id
       or (    s.schema = sgid.cu
           and (   sgid.nls_sort='BINARY'
                or nlssort(s.schema,'NLS_SORT=BINARY')=nlssort(sgid.cu,'NLS_SORT=BINARY'))))
  and s.security_group_id = w.provisioning_company_id
  and w.provisioning_company_id != 0
group by  w.provisioning_company_id,
          w.short_name,
          w.source_identifier,
          w.path_prefix,
          w.team_dev_fs_limit,
          w.issue_fs_limit,
          w.first_schema_provisioned,
          w.allow_app_building_yn,
          w.allow_sql_workshop_yn,
          w.allow_team_development_yn,
          w.allow_team_dev_files_yn,
          w.allow_issue_files_yn,
          w.allow_to_be_purged_yn,
          w.allow_restful_services_yn,
          w.builder_notification_message,
          w.env_banner_yn,
          w.env_banner_label,
          w.env_banner_color,
          w.env_banner_pos,
          w.rm_consumer_group,
          w.display_name,
          w.workspace_service_id,
          w.workspace_description,
          w.files_version,
          w.max_email_messages,
          w.max_webservice_requests,
          w.max_file_bytes,
          w.max_session_length_sec,
          w.max_session_idle_sec,
          w.session_timeout_warning_sec,
          w.qos_max_workspace_requests,
          w.qos_max_session_requests,
          w.qos_max_session_kill_timeout,
          w.created_on,
          w.assigned_on
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspaces',
    p_table_comment     => 'Available Oracle APEX workspaces',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'                   , 'Primary key that identifies the workspace',
      'workspace'                      , 'A work area mapped to one or more database schemas',
      'workspace_display_name'         , 'Display name for the workspace',
      'workspace_description'          , 'Optional workspace administrator managed description of the workspace',
      'workspace_service_id'           , 'Common identifier which associates workspaces belonging to the same service',
      'rm_consumer_group'              , 'Resource Manager consumer group',
      'team_dev_fs_limit'              , 'Specifies the maximum size for any file upload to the team development file repository',
      'issue_fs_limit'                 , 'Specifies the maximum size for any file upload to the issues file repository',
      'allow_app_building_yn'          , 'Controls availability of identified component',
      'allow_sql_workshop_yn'          , 'Controls availability of identified component',
      'allow_packaged_app_ins_yn'      , 'Obsolete',
      'allow_websheet_dev_yn'          , 'Obsolete',
      'allow_team_development_yn'      , 'Controls availability of identified component',
      'allow_team_dev_files_yn'        , 'Controls availability of identified component',
      'allow_issue_files_yn'           , 'Controls availability of identified component',
      'allow_to_be_purged_yn'          , 'Controls availability of identified component',
      'allow_restful_services_yn'      , 'Controls availability of identified component',
      'source_identifier'              , 'Identifies the workspace when data is synchronized to other systems.',
      'path_prefix'                    , 'Identifies the workspace path_prefix, used in RESTful Services URL to identify the source of a RESTful Service request.',
      'schemas'                        , 'Number of database schemas currently mapped to this workspace',
      'last_purged_session'            , 'Creation date of the most recently purged session.  Used to track last login for workspaces who no longer have entries in the activity log.',
      'workspace_announcement'         , 'Workspace announcement message set from within Workspace Administration',
      'workspace_banner_yn'            , 'Enable workspace banner.  When enabled the banner displays in development environment of the workspace.',
      'workspace_banner_label'         , 'Workspace banner label',
      'workspace_banner_color'         , 'Workspace banner color',
      'workspace_banner_pos'           , 'Workspace banner position',
      'sessions'                       , 'Number of non-purged APEX sessions',
      'applications'                   , 'Number of applications created in this workspace',
      'application_pages'              , 'Number of application pages created in this workspace',
      'apex_users'                     , 'Number of APEX users created in this workspace',
      'apex_developers'                , 'Number of APEX users with developer privilege',
      'apex_workspace_administrators'  , 'Number of APEX users with workspace administrator privilege',
      'files'                          , 'Number of APEX files associated with the workspace',
      'sql_scripts'                    , 'Number of APEX SQL Scripts associated with the workspace',
      'translation_messages'           , 'Number of translatable and translated messages within the workspace',
      'file_storage'                   , 'Size in bytes of all files associated with the workspace',
      'file_storage_max'               , 'Maximum size in bytes of all files associated with the workspace',
      'last_logged_page_view'          , 'Date of most recent page view from the APEX activity log.  The archived activity log is used when no activity is recorded in the standard activity log for the last day.',
      'page_views'                     , 'Count of page views recorded for this workspace in the APEX activity log for the last 2 weeks',
      'files_version'                  , 'Version counter of the attached workspace files.',
      'maximum_session_life_seconds'   , 'Maximum lifetime of session in seconds',
      'maximum_session_idle_seconds'   , 'Maximum session idle time in seconds',
      'session_timeout_warn_seconds'   , 'The time in seconds before a session times out, to display a warning',
      'max_workspace_requests'         , 'Maximum number of concurrent requests for a workspace',
      'max_session_requests'           , 'Maximum number of concurrent requests for a session',
      'max_session_kill_timeout'       , 'Number of seconds a database session has to be active, before it can be killed if MAX_SESSION_REQUESTS is exceeded',
      'email_maximum'                  , 'Maximum number of email messages over any 24 hour period',
      'webservice_request_maximum'     , 'Maximum number of webservice requests over any 24 hour period',
      'created_on'                     , 'The date the workspace was created',
      'assigned_on'                    , 'The date the workspace was assigned to the customer'
    )
  );
end;
/

prompt ...apex_workspace_files
create or replace view apex_workspace_files as
select
    w.workspace_id,
    w.workspace                                         workspace_name,
    w.workspace_display_name,
    files.id                                            file_id,
    files.flow_id                                       application_id,
    f.name                                              application_name,
    nvl(files.title,files.filename)                     file_name,
    files.mime_type                                     mime_type,
    files.doc_size                                      file_size,
    files.created_by                                    created_by,
    (select max(email_address)
    from wwv_flow_fnd_user
    where security_group_id = w.workspace_id
    and user_name=files.created_by)                     email,
    files.created_on                                    created_on,
    decode(files.file_type,
      'FLOW_EXPORT','Application Export',
      'IMAGE_EXPORT','Text Image Export',
      'PAGE_EXPORT','Application Page Export',
      'SCRIPT','SQL Script',
      'THEME','User Interface Theme Export',
      'XLIFF','XLIFF Application Translation Export',
      files.file_type)                                  file_type,
    files.blob_content                                  document,
    files.session_id                                    temp_file_session_id
from wwv_flow_file_objects$      files,
     wwv_flows                   f,
     wwv_flow_current_workspaces w
where files.security_group_id             = w.workspace_id
  and files.security_group_id             = f.security_group_id(+)
  and files.flow_id                       = f.id(+)
  and to_char(deleted_as_of,'MM.DD.YYYY') = '01.01.0001'
  and nvl(files.file_type, 'x') not       in ( 'CSS_FILE', 'STATIC_FILE', 'IMAGE_FILE' )
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_files',
    p_table_comment     => 'Identifies uploaded files belonging to the workspace in the modplsql or EPG documents table',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'           , 'Primary key that identifies the workspace',
      'workspace_name'         , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'file_id'                , 'Primary key that identifies the file',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'created_by'             , 'Name of the APEX user who created the file',
      'email'                  , 'Email address that corresponds to the APEX user name',
      'created_on'             , 'Date the file was created',
      'file_name'              , 'Name of the file stored in a database BLOB',
      'mime_type'              , 'Mime type of the file used when fetching the file over the web',
      'DOCUMENT'               , 'The file',
      'FILE_SIZE'              , 'Size of the file',
      'CREATED_BY'             , 'Identifies the APEX User Name who created the file',
      'CREATED_ON'             , 'Identifies the date the file was loaded into the database BLOB',
      'FILE_TYPE'              , 'Identifies the APEX file type if available',
      'temp_file_session_id'   , 'Null for persistent files. For temporary files, the session id where the file was uploaded.'
    )
  );
end;
/

prompt ...apex_workspace_static_files
create or replace view apex_workspace_static_files as
select f.id                      as workspace_file_id,
       w.workspace,
       w.workspace_display_name,
       w.workspace_id,
       f.file_name,
       f.mime_type,
       f.file_charset,
       f.file_content,
       f.created_by,
       f.created_on,
       f.last_updated_by,
       f.last_updated_on
  from wwv_flow_current_workspaces   w,
       wwv_flow_company_static_files f
 where f.security_group_id = w.workspace_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_static_files',
    p_table_comment     => 'Stores the files like CSS, images, javascript files, ... of a workspace.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_file_id'      , 'Identifies the primary key of this file',
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_id'           , 'Primary key that identifies the workspace',
      'workspace_display_name' , 'Display name for the workspace',
      'file_name'              , 'Name of the file.',
      'mime_type'              , 'Mime type of the file.',
      'file_charset'           , 'IANA charset used for text files.',
      'file_content'           , 'Blob content of the file.',
      'created_by'             , 'APEX developer who created the file',
      'created_on'             , 'Date of creation',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update'
    )
  );
end;
/

prompt ...apex_workspace_sql_scripts
create or replace view apex_workspace_sql_scripts
as
select
    w.workspace_id,
    w.workspace_name,
    w.workspace_display_name,
    f.filename                  sql_script_name,
    (select max(email_address)
    from wwv_flow_fnd_user
    where security_group_id = w.workspace_id
    and user_name           = f.created_by) email,
    --
    --f.title                     file_name,
    f.mime_type                 mime_type,
    f.doc_size                  file_size,
    f.created_by                owner,
    f.blob_content              sql_script
from wwv_flow_current_workspaces w,
     wwv_flow_file_objects$      f
where f.security_group_id                   = w.workspace_id
  and to_char(f.deleted_as_of,'MM.DD.YYYY') = '01.01.0001'
  and f.file_type                           = 'SCRIPT'
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_sql_scripts',
    p_table_comment     => 'Identifies SQL Scripts used to execute SQL and PL/SQL commands',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'        , 'Primary key that identifies the workspace',
      'workspace_name'      , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'sql_script_name'     , 'Identifies name of the SQL Script',
      'EMAIL'               , 'Identifies email address that corresponds to the APEX User Name who owns the SQL Script',
      'MIME_TYPE'           , 'Mime type associated with the file',
      'FILE_SIZE'           , 'Size of the file in the BLOB',
      'OWNER'               , 'Identifies the APEX User Name who crated and owns the SQL Script',
      'SQL_SCRIPT'          , 'The SQL Script file'
    )
  );
end;
/

prompt ...apex_workspace_sessions
create or replace view apex_workspace_sessions as
select
    w.workspace_id,
    w.workspace_name,
    w.workspace_display_name,
    s.id                                                apex_session_id,
    s.username                                          user_name,
    s.remote_addr                                       remote_addr,
    s.created_on                                        session_created,
    s.idle_timeout_on                                   session_idle_timeout_on,
    s.life_timeout_on                                   session_life_timeout_on,
    s.max_idle_sec                                      session_max_idle_sec,
    s.session_time_zone                                 session_time_zone,
    s.session_lang                                      session_lang,
    s.session_territory                                 session_territory,
    s.debug_level                                       session_debug_level,
    case s.trace_mode
    when 1 then 'SQL'
    when 2 then 'HPROF'
    end                                                 session_trace_mode,
    s.authentication_id                                 session_authentication_id,
    s.tenant_id                                         session_tenant_id,
    a.name                                              session_authentication_name
from wwv_flow_sessions$          s,
     wwv_flow_current_workspaces w,
     wwv_flow_authentications    a
where s.security_group_id = w.workspace_id
  and s.security_group_id = a.security_group_id (+)
  and s.authentication_id = a.id (+)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_sessions',
    p_table_comment     => 'Oracle APEX sessions by workspace and APEX user',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'                , 'Primary key that identifies the workspace',
      'workspace_name'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name'      , 'Display name for the workspace',
      'apex_session_id'             , 'Primary key of the apex session',
      'user_name'                   , 'Name of the authenticated or public user',
      'remote_addr'                 , 'Remote client IP address',
      'session_created'             , 'Date the APEX session was created',
      'session_idle_timeout_on'     , 'Date when the APEX session times out because of inactivity',
      'session_life_timeout_on'     , 'Date when the APEX session times out because of the maximum session length has been reached',
      'session_max_idle_sec'        , 'Maximum idle time in seconds of the APEX session',
      'session_time_zone'           , 'Time zone offset of the APEX session',
      'session_lang'                , 'Language preference of the APEX session',
      'session_territory'           , 'NLS territory preference of the APEX session',
      'session_debug_level'         , 'Debug level of the APEX session',
      'session_trace_mode'          , 'Trace mode of the APEX session',
      'session_authentication_id'   , 'ID of the authentication scheme, if switched at runtime',
      'session_authentication_name' , 'Name of the authentication scheme, if switched at runtime',
      'Session_tenant_id'           , 'Identifier of the tenant for the session'
    )
  );
end;
/

prompt ...apex_workspace_session_groups
create or replace view apex_workspace_session_groups as
select
    w.workspace_id,
    w.workspace_name,
    w.workspace_display_name,
    s.id                                                apex_session_id,
    s.username                                          user_name,
    sg.group_name                                       group_name
from wwv_flow_sessions$          s,
     wwv_flow_session_groups$    sg,
     wwv_flow_current_workspaces w
where s.id                 = sg.session_id
  and s.security_group_id  = w.workspace_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_session_groups',
    p_table_comment     => 'Oracle APEX groups that are dynamically assigned to sessions',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'       , 'Primary key that identifies the workspace',
      'workspace_name'     , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'apex_session_id'    , 'Primary key of the apex session',
      'user_name'          , 'Name of the authenticated or public user',
      'group_name'         , 'Name of the dynamically assigned group'
    )
  );
end;
/

prompt ...apex_workspace_apex_users
create or replace view apex_workspace_apex_users as
select
    workspace_id,
    workspace_name,
    workspace_display_name,
    first_schema_provisioned,
    user_name,
    first_name,
    last_name,
    email,
    date_created,
    date_last_updated,
    available_schemas,
    substr(decode(instr(nvl(role_privs,'x'),'ADMIN'),0,'No','Yes'),1,3)  is_admin,
    substr(decode(instr(nvl(role_privs,'x'),'EDIT'),0,'No','Yes'),1,3)   is_application_developer,
    decode(account_locked,'Y','Yes','No')                                account_locked,
    description,
    password_version,
    account_expiry,
    failed_access_attempts,
    profile_image_name,
    profile_mimetype,
    profile_filename,
    profile_charset
from (
select
    w.workspace_id,
    w.workspace                                                       workspace_name,
    w.workspace_display_name,
    w.first_schema_provisioned,
    u.user_name                                                       user_name,
    u.first_name                                                      first_name,
    u.last_name                                                       last_name,
    u.EMAIL_address                                                   email,
    u.CREATION_DATE                                                   date_created,
    u.LAST_UPDATE_DATE                                                date_last_updated,
    ( select count(*)
        from wwv_flow_company_schemas s
       where s.security_group_id = w.workspace_id )                   available_schemas,
    (select developer_role
       from wwv_flow_developers
      where userid            = u.user_name
        and security_group_id = w.workspace_id)                       role_privs,
    u.account_locked                                                  account_locked,
    u.description                                                     description,
    u.web_password_version                                            password_version,
    u.account_expiry                                                  account_expiry,
    u.failed_access_attempts                                          failed_access_attempts,
    u.profile_image_name                                              profile_image_name,
    u.profile_mimetype                                                profile_mimetype,
    u.profile_filename                                                profile_filename,
    u.profile_charset                                                 profile_charset
from
     wwv_flow_fnd_user           u,
     wwv_flow_current_workspaces w
where u.security_group_id = w.workspace_id
group by  w.workspace_id,
          w.workspace,
          w.workspace_display_name,
          w.first_schema_provisioned,
          u.user_name,
          u.first_name,
          u.last_name,
          u.EMAIL_address,
          u.CREATION_DATE,
          u.LAST_UPDATE_DATE,
          u.account_locked,
          u.description,
          u.web_password_version,
          u.account_expiry,
          u.failed_access_attempts,
          u.profile_image_name,
          u.profile_mimetype,
          u.profile_filename,
          u.profile_charset
) ilv
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_apex_users',
    p_table_comment     => 'Oracle APEX workspace users',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'             , 'Primary key that identifies the workspace',
      'workspace_name'           , 'A work area mapped to one or more database schemas',
      'workspace_display_name'   , 'Display name for the workspace',
      'first_schema_provisioned' , 'The associated database schema identified when this workspace was created',
      'user_name'                , 'The APEX user name used to authenticate to an APEX web page or APEX application',
      'first_name'               , 'The APEX user''s first name',
      'last_name'                , 'The APEX user''s last name',
      'email'                    , 'The email associated with this APEX user',
      'date_created'             , 'The date the APEX user was created',
      'date_last_updated'        , 'The date the APEX user definition was last updated',
      'available_schemas'        , 'The number of database schemas available to the workspace developer',
      'is_admin'                 , 'Identifies if the APEX user has APEX workspace administration privilege',
      'is_application_developer' , 'Identifies if the APEX user has APEX application development privilege',
      'account_locked'           , 'Identifies if the APEX user account is locked',
      'description'              , 'Description of the APEX user',
      'password_version'         , 'Version of the algorithm that is used for this user''s password',
      'account_expiry'           , 'The date of the account expiry',
      'failed_access_attempts'   , 'The number of failed attempts to access the account',
      'profile_image_name'       , 'The name of the account profile image',
      'profile_mimetype'         , 'The mimetype of the account profile image',
      'profile_filename'         , 'The file name of the account profile image',
      'profile_charset'          , 'The character set of the account profile image'
    )
  );
end;
/

prompt ...apex_workspace_developers
create or replace view apex_workspace_developers as
select *
from apex_workspace_apex_users
where is_application_developer = 'Yes'
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_developers',
    p_table_comment     => 'Oracle APEX workspace developers, APEX users with privilege to develop applications',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'             , 'Primary key that identifies the workspace',
      'workspace_name'           , 'A work area mapped to one or more database schemas',
      'first_schema_provisioned' , 'The associated database schema identified when this workspace was created',
      'user_name'                , 'The APEX user name used to authenticate to an APEX web page or APEX application',
      'email'                    , 'The email associated with this APEX developer',
      'date_created'             , 'The date the APEX developer was created',
      'date_last_updated'        , 'The date the APEX developer definition was last updated',
      'available_schemas'        , 'The number of database schemas available to the workspace developer',
      'is_admin'                 , 'Identifies if the APEX developer has APEX workspace administration privilege',
      'is_application_developer' , 'Identifies if the APEX user has APEX application development privilege',
      'WORKSPACE_DISPLAY_NAME'   , 'Display name for the workspace',
      'FIRST_NAME'               , 'The APEX developer''s first name',
      'LAST_NAME'                , 'The APEX developer''s last name',
      'ACCOUNT_LOCKED'           , 'Identifies if the APEX developer account is locked',
      'DESCRIPTION'              , 'Description of the APEX developer',
      'PASSWORD_VERSION'         , 'Version of the algorithm that is used for this developer''s password',
      'ACCOUNT_EXPIRY'           , 'The date of the account expiry',
      'FAILED_ACCESS_ATTEMPTS'   , 'The number of failed attempts to access the account',
      'PROFILE_IMAGE_NAME'       , 'The name of the account profile image',
      'PROFILE_MIMETYPE'         , 'The mimetype of the account profile image',
      'PROFILE_FILENAME'         , 'The file name of the account profile image',
      'PROFILE_CHARSET'          , 'The character set of the account profile image'
    )
  );
end;
/

prompt ...apex_workspace_group_users
create or replace view apex_workspace_group_users (
    workspace_id,
    workspace_name,
    workspace_display_name,
    user_name,
    group_name ) as
select
    w.workspace_id,
    w.workspace_name,
    w.workspace_display_name,
    u.user_name,
    g.group_name
from
     wwv_flow_fnd_user           u,
     wwv_flow_fnd_group_users    ug,
     wwv_flow_fnd_user_groups    g,
     wwv_flow_current_workspaces w
where u.security_group_id         = w.workspace_id
  and u.user_id                   = ug.user_id
  and u.security_group_id         = ug.security_group_id
  and ug.group_id                 = g.id
  and ug.security_group_id        = g.security_group_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_group_users',
    p_table_comment     => 'Oracle APEX workspace users and groups relationship',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'   , 'Primary key that identifies the workspace',
      'workspace_name' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'user_name'      , 'The APEX user name used to authenticate to an APEX web page or APEX application',
      'group_name'     , 'A workspace group the user_name is associated with'
    )
  );
end;
/

prompt ...apex_workspace_group_groups
create or replace view apex_workspace_group_groups (
    workspace_id,
    workspace_name,
    workspace_display_name,
    grantee_name,
    group_name,
    grantee_application_id,
    group_application_id ) as
select
    w.workspace_id,
    w.workspace_name,
    w.workspace_display_name,
    ge.group_name,
    gt.group_name,
    ge.flow_id,
    gt.flow_id
from
     wwv_flow_fnd_user_groups  ge,
     wwv_flow_fnd_group_groups gg,
     wwv_flow_fnd_user_groups  gt,
     wwv_flow_current_workspaces w
where gg.security_group_id        = w.workspace_id
  and ge.id                       = gg.grantee_group_id
  and ge.security_group_id        = gg.security_group_id
  and gg.granted_group_id         = gt.id
  and gg.security_group_id        = gt.security_group_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_group_groups',
    p_table_comment     => 'Oracle APEX workspace group/application role to group/role relationship',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'   , 'Primary key that identifies the workspace',
      'workspace_name' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'grantee_name'   , 'The workspace group/application ACL role that another group/role has been granted to',
      'group_name'     , 'A workspace group/application ACL role the grantee_name is associated with',
      'grantee_application_id' , 'The grantee''s application ID if it is an application ACL role',
      'group_application_id' , 'The group''s application ID if it is an application ACL role'
    )
  );
end;
/

prompt ...apex_workspace_groups
create or replace view apex_workspace_groups (
    workspace_id,
    workspace_name,
    workspace_display_name,
    group_name,
    group_desc,
    application_id,
    application_name ) as
select
    w.workspace_id,
    w.workspace_name,
    w.workspace_display_name,
    g.group_name,
    g.group_desc,
    g.flow_id,
    f.name
from
     wwv_flow_fnd_user_groups    g,
     wwv_flows                   f,
     wwv_flow_current_workspaces w
where g.security_group_id = w.workspace_id
  and g.flow_id           = f.id (+)
  and g.security_group_id = f.security_group_id (+)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_groups',
    p_table_comment     => 'Oracle APEX workspace groups/application roles',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'   , 'Primary key that identifies the workspace',
      'workspace_name' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'group_name'     , 'A workspace group',
      'group_desc'     , 'The workspace group''s description',
      'application_id' , 'The application ID if the group is an application ACL role',
      'application_name' , 'Identifies the application'
    )
  );
end;
/

prompt ...apex_appl_acl_roles
create or replace view apex_appl_acl_roles
as
select
    f.workspace_id,
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    g.id                             role_id,
    g.group_name                     role_name,
    g.static_id                      role_static_id,
    g.group_desc                     role_desc,
    --
    g.group_name||'.'||g.static_id    component_signature
from wwv_flow_fnd_user_groups g,
     wwv_flow_authorized      f
where f.workspace_id      = g.security_group_id
  and f.application_id    = g.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_acl_roles',
    p_table_comment     => 'Identifies Application Roles, which are workspace groups that are tied to a specific application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'           , 'Primary key that identifies the workspace',
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'role_id'                , 'Primary key that identifies the role',
      'role_name'              , 'Role name',
      'role_static_id'         , 'Static identifier for a role',
      'role_desc'              , 'The role''s description',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

prompt ...apex_appl_acl_user_roles
create or replace view apex_appl_acl_user_roles
as
select
    u.id                             user_role_id,
    --
    f.workspace_id,
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    u.user_name                      user_name,
    g.id                             role_id,
    g.group_name                     role_name,
    g.static_id                      role_static_id,
    g.group_desc                     role_desc,
    --
    u.created_on                     created_on,
    u.created_by                     created_by,
    u.last_updated_on                last_updated_on,
    u.last_updated_by                last_updated_by,
    --
    u.user_name||
    '.'||g.static_id||
    '.'||g.flow_id                   component_signature
from
     wwv_flow_fnd_user_groups    g,
     wwv_flow_authorized         f,
     wwv_flow_acl_group_users    u
where f.workspace_id      = g.security_group_id
  and f.application_id    = g.flow_id
  and u.security_group_id = f.workspace_id
  and u.group_id          = g.id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_acl_user_roles',
    p_table_comment     => 'Identifies Application Users used to map application users to application roles',
    p_column_comments   => wwv_flow_t_varchar2 (
      'user_role_id'           , 'Primary key that identifies the user role assignment',
      'workspace_id'           , 'Primary key that identifies the workspace',
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'user_name'              , 'Name of the application user',
      'role_id'                , 'Primary key that identifies the role',
      'role_name'              , 'Name of the assigned application role',
      'role_static_id'         , 'Static identifier of the assigned role',
      'role_desc'              , 'Description of the assigned application role',
      'created_on'             , 'Date of creation',
      'created_by'             , 'APEX developer who created the assignment',
      'last_updated_on'        , 'Date of last_update',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'LAST_UPDATED_BY'        , 'APEX developer who last updated the assignment'
    )
  );
end;
/

prompt ...apex_appl_acl_users
create or replace view apex_appl_acl_users
as
select
    min( u.id ) id,
    --
    f.workspace_id,
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    u.user_name,
    lower(u.user_name) user_name_lc,
    listagg( u.group_id, ':' )    within group ( order by u.group_id )   as role_ids,
    listagg( g.group_name, ', ' ) within group ( order by g.group_name ) as role_names
from wwv_flow_authorized_for_dml  f,
     wwv_flow_fnd_user_groups     g,
     wwv_flow_acl_group_users     u
where g.flow_id           = f.application_id
  and g.security_group_id = f.workspace_id
  and u.group_id          = g.id
  and u.security_group_id = g.security_group_id
group by f.workspace_id,
         f.workspace,
         f.workspace_display_name,
         f.application_id,
         f.application_name,
         u.user_name
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_acl_users',
    p_table_comment     => 'Identifies Application Users used to map application users to application roles',
    p_column_comments   => wwv_flow_t_varchar2 (
      'id'                     , 'Virtual Primary key that identifies the user',
      'workspace_id'           , 'Primary key that identifies the workspace',
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'user_name'              , 'Name of the application user',
      'user_name_lc'           , 'Name of the application user, in lower case',
      'role_ids'               , 'Colon delimited list of application role ids which are assigned to the user',
      'role_names'             , 'Comma delimited list of application role names which are assigned to the user'
    )
  );
end;
/
create or replace trigger apex_appl_acl_users_iot
instead of insert or update or delete
on apex_appl_acl_users
for each row
begin
    if inserting or updating then
        wwv_flow_acl_api.replace_user_roles (
            p_application_id => coalesce( :old.application_id, :new.application_id ), -- don't support updates
            p_user_name      => coalesce( :old.user_name, :new.user_name ),           -- don't support updates
            p_role_ids       => wwv_flow_string.split_numbers( :new.role_ids, ':' ) );
    elsif deleting then
        wwv_flow_acl_api.remove_all_user_roles (
            p_application_id => :old.application_id, -- don't support updates
            p_user_name      => :old.user_name );    -- don't support updates
    end if;
end;
/

prompt ...apex_workspace_schemas
create or replace view apex_workspace_schemas as
select
    w.PROVISIONING_COMPANY_ID                           workspace_id,
    w.short_name                                        workspace_name,
    w.display_name                                      workspace_display_name,
    w.FIRST_SCHEMA_PROVISIONED                          first_schema_provisioned,
    s.schema                                            schema,
    (select created
     from   sys.all_users
     where  username = s.schema)                        schema_created,
    (select count(*)
     from wwv_flows
     where security_group_id=w.PROVISIONING_COMPANY_ID
      and s.schema = owner)                             applications
from wwv_flow_companies       w,
     wwv_flow_company_schemas s,
     wwv_flow_current_sgid    sgid
where s.security_group_id = w.provisioning_company_id
  and (   sgid.security_group_id          = w.provisioning_company_id
       or sgid.has_apex_admin_read_role_yn = 'Y'
       or (    s.schema = sgid.cu
           and (   sgid.nls_sort='BINARY'
                or nlssort(s.schema,'NLS_SORT=BINARY')=nlssort(sgid.cu,'NLS_SORT=BINARY'))))
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_schemas',
    p_table_comment     => 'Database Schemas mapped to APEX workspaces',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'             , 'Primary key that identifies the workspace',
      'workspace_name'           , 'A work area mapped to one or more database schemas',
      'workspace_display_name'   , 'Display name for the workspace',
      'first_schema_provisioned' , 'The associated database schema identified when this workspace was created',
      'schema'                   , 'Database schema name mapped to workspace',
      'schema_created'           , 'Identifies the date that the database schema was created',
      'applications'             , 'Number of applications within the current workspace and schema'
    )
  );
end;
/

prompt ...apex_workspace_ui_types
create or replace view apex_workspace_ui_types
as
select u.id                      as ui_type_id,
       w.workspace,
       w.workspace_display_name,
       w.workspace_id,
       u.name,
       u.based_on_ui_type_id,
       u.autodetect_js_file_urls,
       u.autodetect_js_function_body,
       u.autodetect_plsql_function_body,
       u.created_by,
       u.created_on,
       u.last_updated_by,
       u.last_updated_on
  from wwv_flow_ui_types           u,
       wwv_flow_current_workspaces w
 where w.workspace_id = u.security_group_id
union all
select u.id                      as ui_type_id,
       to_char(null),
       to_char(null),
       to_number(null),
       u.name,
       u.based_on_ui_type_id,
       u.autodetect_js_file_urls,
       u.autodetect_js_function_body,
       u.autodetect_plsql_function_body,
       u.created_by,
       u.created_on,
       u.last_updated_by,
       u.last_updated_on
  from wwv_flow_ui_types u
 where u.security_group_id is null
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_ui_types',
    p_table_comment     => 'The UI Types identify the available user interface types of a workspace',
    p_column_comments   => wwv_flow_t_varchar2 (
      'ui_type_id'                     , 'Identifies the primary key of this component',
      'workspace'                      , 'A work area mapped to one or more database schemas',
      'workspace_id'                   , 'Primary key that identifies the workspace',
      'workspace_display_name'         , 'Display name for the workspace',
      'name'                           , 'Contains the name of the UI type.',
      'based_on_ui_type_id'            , 'Foreign key to inherit the components of the specified base UI type.',
      'autodetect_js_file_urls'        , 'List of JavaScript URLs which should be loaded for the auto detection of the UI type.',
      'autodetect_js_function_body'    , 'JavaScript function which returns true if the current device supports the UI type.',
      'autodetect_plsql_function_body' , 'PL/SQL function which returns true if the current device supports the UI type.',
      'created_by'                     , 'APEX developer who created the theme style',
      'created_on'                     , 'Date of creation',
      'last_updated_by'                , 'APEX developer who made last update',
      'last_updated_on'                , 'Date of last update'
    )
  );
end;
/
--------------------------------------------
-- A P P L I C A T I O N S
-- 4001

prompt ...apex_applications
create or replace view apex_applications as
select
    w.workspace,
    w.workspace_display_name,
    f.ID                             application_id,
    f.NAME                           application_name,
    f.ALIAS                          alias,
    f.OWNER                          owner,
    (select gr.group_name from wwv_flow_application_groups gr where gr.security_group_id = f.security_group_id and gr.id = f.group_id) application_group,
    group_id                         application_group_id,
    decode(upper(f.WEBDB_LOGGING),
        'YES','Yes',
        'NO','No',f.WEBDB_LOGGING)   logging,
    f.FLOW_LANGUAGE                  application_primary_language,
    f.FLOW_LANGUAGE_DERIVED_FROM     language_derived_from,
    decode(upper(f.csv_encoding),
        'Y','Yes',
        'N','No', f.csv_encoding)    csv_encoding,
    f.date_format                    date_format,
    f.date_time_format               date_time_format,
    f.timestamp_format               timestamp_format,
    f.timestamp_tz_format            timestamp_tz_format,
    f.nls_sort                       nls_sort,
    f.nls_comp                       nls_comp,
    f.direction_right_to_left        direction_right_to_left,
    f.auto_time_zone                 auto_time_zone,
    f.ui_detection_css_urls          ui_detection_css_urls,
    f.error_handling_function        error_handling_function,
    f.default_error_display_location default_error_display_location,
    f.FLOW_IMAGE_PREFIX              image_prefix,
    case substr(authentication.scheme_type,1,7)
      when 'NATIVE_' then
        ( select pl.display_name
            from wwv_flow_plugins pl
           where pl.security_group_id = 10
             and pl.flow_id     = 4411
             and pl.plugin_type = 'AUTHENTICATION TYPE'
             and pl.name        = substr(authentication.scheme_type, 8) )
      when 'PLUGIN_' then
        ( select plf.display_name
            from wwv_flow_plugins plf
           where plf.security_group_id = f.security_group_id
             and plf.flow_id     = f.id
             and plf.plugin_type = 'AUTHENTICATION TYPE'
             and plf.name        = substr(authentication.scheme_type, 8) )
      else
        decode(f.AUTHENTICATION,
      'CUSTOM2','Authentication Scheme',
      'COOKIE', 'APEX Authentication',
            f.AUTHENTICATION)
    end                              authentication_scheme_type,
    authentication.name              authentication_scheme,
    f.authentication_id              authentication_scheme_id,
    authentication.LOGOUT_URL        logout_url,
    case f.logo_type
      when 'T'  then 'Text Logo'
      when 'I'  then 'Image Logo'
      when 'IT' then 'Image and Text Logo'
      when 'C'  then 'Custom Logo'
    end                              logo_type,
    f.LOGO                           logo,
    f.LOGO_TEXT                      logo_text,
    f.app_builder_icon_name          app_builder_icon_name,
    f.favicons                       favicons,
    --f.PUBLIC_URL_PREFIX              public_url_prefix,
    f.PUBLIC_USER                    public_user,
    --f.DBAUTH_URL_PREFIX              db_auth_url_prefix,
    f.PROXY_SERVER                   proxy_server,
    f.no_proxy_domains               no_proxy_domains,
    f.media_type                     media_type,
    --
    --f.CUSTOM_AUTHENTICATION_PAGE     custom_auth_page,
    --f.CUSTOM_AUTH_LOGIN_URL          custom_auth_login_url,
    f.FLOW_VERSION                     version,
    decode(f.FLOW_STATUS,
        'AVAILABLE','Available',
        'AVAILABLE_W_EDIT_LINK','Available with Edit Links',
        'DEVELOPERS_ONLY','Available to Developers Only',
        'RESTRICTED_ACCESS','Restricted Access',
        'UNAVAILABLE','Unavailable',
        'UNAVAILABLE_PLSQL','Unavailable (Status Shown with PL/SQL)',
        'UNAVAILABLE_URL','Unavailable (Redirect to URL)',
        f.flow_status)               availability_status,
    f.FLOW_UNAVAILABLE_TEXT          unavailable_text,
    f.RESTRICT_TO_USER_LIST          restrict_to_user,
    decode(f.APPLICATION_TAB_SET,
        1,'Allowed',0,'Not Allowed',
        'Allowed')                   debugging,
    decode(f.EXACT_SUBSTITUTIONS_ONLY,
       'N','No','Y','Yes',
       f.EXACT_SUBSTITUTIONS_ONLY)   exact_substitutions,
    decode(f.BUILD_STATUS,
       'RUN_ONLY','Run Only',
       'RUN_AND_BUILD','Run and Develop',
       'RUN_AND_HIDDEN','Run and Hidden',
       f.BUILD_STATUS)               build_status,
    decode(f.publish_yn,'Y','Yes','N','No','No') publish,
    decode(f.friendly_url,'Y', 'Yes', 'N', 'No') friendly_url,
    f.VPD                            vpd, /* for compatibility reasons */
    f.VPD                            db_session_init_code,
    f.vpd_teardown_code              db_session_cleanup_code,
    --
    decode(substr(f.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
     from   wwv_flow_security_schemes ss
     where  f.security_group_id = ss.security_group_id
     and    to_char(ss.id) = ltrim(f.SECURITY_SCHEME,'!')
     and    ss.flow_id = f.id),
     f.SECURITY_SCHEME)              authorization_scheme,
    f.security_scheme                authorization_scheme_id,
    decode(f.authorize_public_pages_yn,'Y','Yes','No') authorize_public_pages,
    decode(f.authorize_batch_job,'Y','Yes','No') authorize_batch_job,
    f.CREATED_BY                     created_by,
    f.CREATED_ON                     created_on,
    f.LAST_UPDATED_BY                last_updated_by,
    f.LAST_UPDATED_ON                last_updated_on,
    f.GLOBAL_NOTIFICATION            global_notification,
    decode(f.PAGE_PROTECTION_ENABLED_Y_N,
      'Y','Enabled',
      'N','Disabled',
      'Disabled')                    Session_State_Protection,
    f.MAX_SESSION_LENGTH_SEC         Maximum_Session_Life_Seconds,
    ON_MAX_SESSION_TIMEOUT_URL       Session_Lifetime_Exceeded_URL,
    f.MAX_SESSION_IDLE_SEC           Maximum_Session_Idle_Seconds,
    ON_MAX_IDLE_TIMEOUT_URL          Session_Idle_Time_Exceeded_URL,
    f.session_timeout_warning_sec    Session_Timeout_Warn_Seconds,
    case f.browser_cache
      when 'Y' then 'Enabled'
      else          'Disabled'
    end                              browser_cache,
    case f.browser_frame
      when 'D' then 'Deny'
      when 'S' then 'Allow from same origin'
      else          'Allow'
    end                              browser_frame,
    f.referrer_policy,
    case f.deep_linking
      when 'Y' then 'Enabled'
      else          'Disabled'
    end                              deep_linking,
    compatibility_mode,
    case html_escaping_mode
      when 'B' then 'Basic'
      when 'E' then 'Extended'
    end html_escaping_mode,
    case allow_feedback_yn
      when 'Y' then 'Enabled'
      else          'Disabled'
    end                              feedback,
    email_from,
    f.file_prefix,
    f.files_version,
    f.runtime_api_usage,
    case f.rejoin_existing_sessions
    when 'Y' then 'All Sessions'
    when 'P' then 'Public Sessions'
    when 'N' then 'Disabled'
    end                              rejoin_existing_sessions,
    f.http_response_headers          http_response_headers,
    f.bookmark_checksum_function     bookmark_checksum_function,
    decode(f.accept_old_checksums, 'Y', 'Yes', 'N', 'No', 'No')                accept_old_checksums,
    decode(f.tokenize_row_search, 'Y', 'Yes', 'N', 'No', 'No')                 tokenize_row_search,
    f.oracle_text_function,
    -- sub components
    (select count(*) from wwv_flow_steps st where st.security_group_id = f.security_group_id and st.flow_id = f.id)               pages,
    (select count(*) from wwv_flow_tabs ta where ta.security_group_id = f.security_group_id and ta.flow_id = f.id)                tabs,
    (select count(*) from wwv_flow_toplevel_tabs tt where tt.security_group_id = f.security_group_id and tt.flow_id = f.id)       parent_tabs,
    (select count(*) from wwv_flow_items it where it.security_group_id = f.security_group_id and it.flow_id = f.id)               application_items,
    (select count(*) from wwv_flow_processing fp where fp.security_group_id = f.security_group_id and fp.flow_id = f.id)          application_processes,
    (select count(*) from wwv_flow_computations fc where fc.security_group_id = f.security_group_id and fc.flow_id = f.id)        application_computations,
    (select count(*) from wwv_flow_app_settings fas where fas.security_group_id = f.security_group_id and fas.flow_id = f.id)     application_settings,
    (select count(*) from wwv_flow_shortcuts sc where sc.security_group_id = f.security_group_id and sc.flow_id = f.id)           shortcuts,
    (select count(*) from wwv_flow_shared_web_services ws where ws.security_group_id = f.security_group_id and ws.flow_id = f.id) web_services,
    (select count(*) from wwv_flow_trees tr where tr.security_group_id = f.security_group_id and tr.flow_id = f.id)               trees,
    (select count(*) from wwv_flow_patches pa where pa.security_group_id = f.security_group_id and pa.flow_id = f.id)             build_options,
    (select count(*) from wwv_flow_menus me where me.security_group_id = f.security_group_id and me.flow_id = f.id)               breadcrumbs,
    (select count(*) from wwv_flow_icon_bar ib where ib.security_group_id = f.security_group_id and ib.flow_id = f.id)            nav_bar_entries,
    (select count(*) from wwv_flow_lists li where li.security_group_id = f.security_group_id and li.flow_id = f.id)               lists,
    (select count(*) from wwv_flow_lists_of_values$ lo where lo.security_group_id = f.security_group_id and lo.flow_id = f.id)    lists_of_values,
    (select count(*) from wwv_flow_themes th where th.security_group_id = f.security_group_id and th.flow_id = f.id)              themes,
    (select count(*) from wwv_flow_authentications at where at.security_group_id = f.security_group_id and at.flow_id = f.id)     authentication_schemes,
    (select count(*) from WWV_FLOW_SECURITY_SCHEMES ss where ss.security_group_id = f.security_group_id and ss.flow_id = f.id)    authorization_schemes,
    (select count(*) from wwv_flow_fnd_user_groups ug where ug.security_group_id = f.security_group_id and ug.flow_id = f.id)     application_acl_roles,
    (select count(*) from WWV_FLOW_MESSAGES$ ms where ms.security_group_id = f.security_group_id and ms.flow_id = f.id)           translation_messages,
    (select count(*) from wwv_flow_install_scripts sc where sc.security_group_id = f.security_group_id and sc.flow_id = f.id)     installation_scripts,
    (select count(*) from WWV_FLOW_STEPS st where st.security_group_id = f.security_group_id and st.flow_id = f.id and CACHE_MODE <> 'NOCACHE') cached_pages,
    (select count(*) from WWV_FLOW_PAGE_PLUGS pp where pp.security_group_id = f.security_group_id and pp.flow_id = f.id and PLUG_CACHING <> 'NOCACHE') cached_regions,
    --
    'a='||f.ALIAS
    ||' o='||f.OWNER
    ||' l='||decode(upper(f.WEBDB_LOGGING),'YES','Yes','NO','No',f.WEBDB_LOGGING)
    ||' l='||f.FLOW_LANGUAGE||' '||f.FLOW_LANGUAGE_DERIVED_FROM
    ||' i='||substr(f.FLOW_IMAGE_PREFIX,1,20)||length(f.FLOW_IMAGE_PREFIX)
    ||' a='||substr(f.AUTHENTICATION,1,20)||length(f.AUTHENTICATION)
    ||' l='||f.logo_type
    ||','||substr(f.LOGO,1,20)||length(f.LOGO)
    ||','||substr(f.LOGO_TEXT,1,20)||length(f.LOGO_TEXT)
    ||' p='||f.PUBLIC_USER
    ||' p='||substr(f.PROXY_SERVER,1,20)||length(f.PROXY_SERVER)
    ||' v='||f.FLOW_VERSION
    ||' s='||decode(f.FLOW_STATUS,
        'AVAILABLE','Available',
        'AVAILABLE_W_EDIT_LINK','AvailwEL',
        'DEVELOPERS_ONLY','DevOnly',
        'RESTRICTED_ACCESS','Rests',
        'UNAVAILABLE','Unavail',
        'UNAVAILABLE_PLSQL','UnavailPL/SQL)',
        'UNAVAILABLE_URL','UnavailableRedir',
        f.flow_status)
    ||' u='||substr(f.FLOW_UNAVAILABLE_TEXT,1,20)||length(f.FLOW_UNAVAILABLE_TEXT)
    ||' r='||substr(f.RESTRICT_TO_USER_LIST,1,20)||length(f.RESTRICT_TO_USER_LIST)
    ||' d='||decode(f.APPLICATION_TAB_SET,
        1,'Allowed',0,'!Allowed',
        'Allowed')
    ||' s='||decode(f.EXACT_SUBSTITUTIONS_ONLY,
       'N','No','Y','Yes',
       f.EXACT_SUBSTITUTIONS_ONLY)
    ||' s='||decode(f.BUILD_STATUS,
       'RUN_ONLY','Run Only',
       'RUN_AND_BUILD','Run+Dev',
       f.BUILD_STATUS)
    ||' v='||substr(f.VPD,1,20)||length(f.vpd)
    ||' v2='||substr(f.vpd_teardown_code,1,20)||length(f.vpd_teardown_code)
    ||' a='||decode(substr(f.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
     from   wwv_flow_security_schemes ss
     where  to_char(ss.id) = ltrim(f.SECURITY_SCHEME,'!')
     and    ss.security_group_id = f.security_group_id
     and    ss.flow_id = f.id),
     'Not found')
    ||' gn='||substr(f.GLOBAL_NOTIFICATION,1,20)||length(f.GLOBAL_NOTIFICATION)
    ||' pp='||decode(f.PAGE_PROTECTION_ENABLED_Y_N,'Y','Enabled','N','Disabled','Disabled')
    ||' timeout='||f.MAX_SESSION_LENGTH_SEC||'.'||ON_MAX_SESSION_TIMEOUT_URL||'.'||f.MAX_SESSION_IDLE_SEC||'.'||ON_MAX_IDLE_TIMEOUT_URL||'.'||f.SESSION_TIMEOUT_WARNING_SEC
    component_signature,
    f.security_group_id workspace_id,
    --
    -- For compatibility with SQL Developer and existing applications (bug# 14545162)
    -- Get the values from the user interfaces table, but only if the app has only one user interface
    ( select case when count(*) = 1 then min(theme_id) end
        from wwv_flow_user_interfaces ui
       where ui.security_group_id = f.security_group_id
         and ui.flow_id           = f.id ) as theme_number,
    ( select case when count(*) = 1 then min(home_url) end
        from wwv_flow_user_interfaces ui
       where ui.security_group_id = f.security_group_id
         and ui.flow_id           = f.id ) as home_link,
    ( select case when count(*) = 1 then min(login_url) end
        from wwv_flow_user_interfaces ui
       where ui.security_group_id = f.security_group_id
         and ui.flow_id           = f.id ) as login_url,
    ( select case when count(*) = 1 then min(pt.name) end
        from wwv_flow_user_interfaces ui,
             wwv_flow_themes t,
             wwv_flow_templates pt
       where ui.security_group_id = f.security_group_id
         and ui.flow_id           = f.id
         and t.security_group_id  = ui.security_group_id
         and t.flow_id            = ui.flow_id
         and t.theme_id           = ui.theme_id
         and pt.security_group_id = t.security_group_id
         and pt.id                = t.default_page_template ) as page_template,
    ( select case when count(*) = 1 then min(pt.name) end
        from wwv_flow_user_interfaces ui,
             wwv_flow_themes t,
             wwv_flow_templates pt
       where ui.security_group_id = f.security_group_id
         and ui.flow_id           = f.id
         and t.security_group_id  = ui.security_group_id
         and t.flow_id            = ui.flow_id
         and t.theme_id           = ui.theme_id
         and pt.security_group_id = t.security_group_id
         and pt.id                = t.error_template ) as error_page_template,
    null as content_delivery_network,
    ( select case when count(*) = 1 then min(javascript_file_urls) end
        from wwv_flow_user_interfaces ui
       where ui.security_group_id = f.security_group_id
         and ui.flow_id           = f.id ) as javascript_file_urls,
    ( select case when count(*) = 1 then min(include_legacy_javascript) end
        from wwv_flow_user_interfaces ui
       where ui.security_group_id = f.security_group_id
         and ui.flow_id           = f.id ) as include_legacy_javascript,
    case f.print_server_type
        when 'INSTANCE' then 'Use Instance Settings'
        when 'NATIVE'   then 'Native Printing'
        when 'REMOTE'   then 'Remote Print Server'
        end                              print_server_type,
    print_server_type                    print_server_type_code,
    f.print_remote_server_id,
    (select static_id from wwv_remote_servers where security_group_id = f.security_group_id and id = f.print_remote_server_id) as print_remote_server_static_id,
    (select name from wwv_remote_servers where security_group_id = f.security_group_id and id = f.print_remote_server_id) as print_remote_server_name,
    f.print_credential_id,
    (select static_id from wwv_credentials where security_group_id = f.security_group_id and id = f.print_credential_id) as print_credential_static_id,
    (select name from wwv_credentials where security_group_id = f.security_group_id and id = f.print_credential_id) as print_credential_name,
    case is_pwa
      when 'Y' then 'Yes'
      else 'No'
    end                             is_pwa,
    case pwa_is_installable
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                               pwa_is_installable,
    f.pwa_manifest_description        pwa_manifest_description,
    f.pwa_manifest_display            pwa_manifest_display,
    f.pwa_manifest_orientation        pwa_manifest_orientation,
    f.pwa_manifest_theme_color        pwa_manifest_theme_color,
    f.pwa_manifest_background_color   pwa_manifest_background_color,
    f.pwa_manifest_icon_url           pwa_manifest_icon_url,
    f.pwa_manifest_custom             pwa_manifest_custom,
    case 
      when f.pwa_service_worker_hooks                                 is null     then 'Default'
      when json_value(f.pwa_service_worker_hooks, '$.FILE_REFERENCE') is not null then 'File URL'
      else                                                                             'Configure Hooks'
    end                               pwa_service_worker_type,
    coalesce(
      to_clob(json_value(f.pwa_service_worker_hooks, '$.FILE_REFERENCE')), 
      f.pwa_service_worker_hooks )    pwa_service_worker_hooks,
    f.pwa_apple_status_bar_style      pwa_apple_statusbar_style_code,
    case f.pwa_apple_status_bar_style
      when 'default'           then 'Black Text and White Background'
      when 'black'             then 'Black Text and Black Background'
      when 'black-translucent' then 'Translucent'
    end                               pwa_apple_statusbar_style,
    ( select fb.banner from wwv_flow_banner fb where f.security_group_id = fb.security_group_id and f.id = fb.id ) application_comment,
    f.copyright_banner              copyright_banner,
    f.mle_environment               mle_environment
from wwv_flows                   f,
     wwv_flow_authentications    authentication,
     wwv_flow_current_workspaces w
where f.security_group_id       =  w.workspace_id
  and f.security_group_id       =  authentication.security_group_id (+)
  and f.authentication_id       =  authentication.id (+)
  /* keep this not exists */
  and not exists ( select 1
                     from wwv_flow_language_map m
                    where m.translation_flow_id = f.id
                      and m.security_group_id   = f.security_group_id
                      and f.authentication_id   <> trunc(f.authentication_id) )
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_applications',
    p_table_comment     => 'Applications defined in the current workspace or database user.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                    , 'A work area mapped to one or more database schemas',
      'workspace_display_name'       , 'Display name for the workspace',
      'application_id'               , 'Application Primary Key, Unique over all workspaces',
      'application_name'             , 'Identifies the application',
      'application_group'            , 'Identifies the name of the application group this application is associated with',
      'application_group_id'         , 'Identifies the ID of the application group this application is associated with',
      'alias'                        , 'Assigns an alternate alphanumeric application identifier',
      'owner'                        , 'Identifies the database schema that this application will parse SQL and PL/SQL statements as',
      'logging'                      , 'Determines whether or not user activity is recorded in the activity log',
      'application_primary_language' , 'Identifies the language in which an application is developed',
      'language_derived_from'        , 'For use when translating an application; specifies how APEX determines or derives the application language',
      'csv_encoding'                 , 'Automatic CSV Encoding.  If Automatic CSV Encoding is enabled, CSV report output will be properly converted to a character set compatible with localized desktop applications. The character set for CSV Encoding will be determined by the the specific application language setting.',
      'date_format'                  , 'Application default date format.  Will set NLS_DATE_FORMAT prior to showing or posting a page',
      'date_time_format'             , 'Application date time format.  Can be referenced via APP_DATE_TIME_FORMAT',
      'timestamp_format'             , 'Application default timestamp format.  Will set NLS_TIMESTAMP_FORMAT prior to showing or posting a page',
      'timestamp_tz_format'          , 'Application default timestamp time zone format.  Will set NLS_TIMESTAMP_TZ_FORMAT prior to showing or posting a page',
      'nls_sort'                     , 'Collating sequence for character value comparison in SQL operations and clauses.  Will set NLS_SORT prior to rendering individual components',
      'nls_comp'                     , 'Collation behavior of SQL operations.  Will set NLS_COMP prior to rendering individual components',
      'direction_right_to_left'      , 'Indicates whether page is rendered left to right or right to left ',
      'auto_time_zone'               , 'Automatic time zone.  Will derive the client time zone from the browser and set the database session imezone automatically at the beginning of each APEX session',
      'ui_detection_css_urls'        , 'Specifies CSS file URLs that will be loaded on the User Interface detection page',
      'error_handling_function'      , 'Function which is called if an error occurs. It can be used to modify or log the error.',
      'default_error_display_location' , 'The default error display location identifies where validation error message will display. Messages can be displayed on an error page, or inline with the existing page. Inline validations are displayed in the "notification" area (defined as part of the page template), and/or within the item label.',
      'image_prefix'                 , 'Determines the virtual path the Web server uses to point to the images directory distributed with APEX',
      'authentication_scheme'        , 'Identifies the current authentication method used by this application. The purpose of authentication is to determine the application users identity',
      'authentication_scheme_id'     , 'Identifies the current authentication method foreign key',
      'authentication_scheme_type'   , 'Identifies the type of authentication scheme',
      'publish'                      , 'Display this application in a list of available applications',
      'friendly_url'                 , 'Determines whether URLs should be friendlier by using application and page alias.',
      'logout_url'                   , 'Used as the LOGOUT_URL substitution string and identifies a URL to redirect to when the user wishes to logout of the application',
      'logo_type'                    , 'Identifies the logo as text, an image or custom',
      'logo'                         , 'Used as the LOGO substitution string, declarative image name or custom markup which can be referenced from page templates',
      'logo_text'                    , 'Used as the LOGO substitution string, declarative name of the application which can be referenced from page templates',
      'app_builder_icon_name'        , 'Legacy. URL to an image to be used as the application item in App Builder.',
      'favicons'                     , 'Deprecated. Used as the FAVICONS substitution string, contains HTML code for to display the application "favicons"',
      'public_user'                  , 'If the APP_USER equals this database schema name then the user is considered a public or unauthenticated user.  Based on this declarative IS_PUBLIC_USER conditions are determined.',
      'proxy_server'                 , 'Use this proxy server when a proxy server is needed',
      'no_proxy_domains'             , 'Domains not to use the proxy server for',
      'media_type'                   , 'Application-level Internet media type, used in the Content-Type HTTP header',
      'VERSION'                      , 'Includes the application''s version number on a page',
      'AVAILABILITY_STATUS'          , 'Specifies whether or not the application is available or unavailable for use',
      'UNAVAILABLE_TEXT'             , 'This attribute in conjunction with the Availability Status.  Identifies the text to display when the application is not available',
      'RESTRICT_TO_USER'             , 'Use this attribute in conjunction with the Availability Status "Restricted Access". Only the users listed in this attribute can run the application.',
      'DEBUGGING'                    , 'Determines whether pages may be display in debug mode.  Debugging is typically on for development and turned off for production.',
      'EXACT_SUBSTITUTIONS'          , 'Determines if text substitutions require a trailing dot.  This values should always be Yes.  Provided for compatibility with preproduction releases.',
      'BUILD_STATUS'                 , 'Determines if the application is available for development.  Production applications can be set to "Run Only" which will cause the application to appear in the APEX App Builder.',
      'VPD'                          , 'Deprecated. Use column DB_SESSION_INIT_CODE',
      'DB_SESSION_INIT_CODE'         , 'Identifies PL/SQL that is dynamically executed immediately after the user is authenticated and before any application logic is processed. This attribute can assign security policies to restrict access to database tables and views.',
      'DB_SESSION_CLEANUP_CODE'      , 'Identifies PL/SQL that is dynamically executed after any application logic is processed. This attribute can free resources like database links or application contexts.',
      'runtime_api_usage'            , 'A colon separated list that identifies what metadata the application is allowed to change: (T)his application, (O)ther applications, (W)orkspace repository',
      'rejoin_existing_sessions'     , 'Determines whether sessions for this application can be joined without session id in the URL',
      'http_response_headers'        , 'This attribute contains http response headers that APEX should send each time for this application',
      'bookmark_checksum_function'   , 'The algorithm which is used to produce URL checksums',
      'accept_old_checksums'         , 'For User- and Application-level checksums: whether checksums generated by APEX 20.1 and earlier are still accepted.',
      'tokenize_row_search'          , 'Whether to split search terms into words when a Row Search is performed in an APEX component.',
      'oracle_text_function'         , 'When Oracle Text is used: Name of the PL/SQL function to convert user input to Oracle Text query syntax.',
      'AUTHORIZATION_SCHEME'         , 'Identifies an Authorization Scheme that will be applied to all pages of the application',
      'AUTHORIZATION_SCHEME_ID'      , 'Identifies an Authorization Scheme foreign key',
      'AUTHORIZE_PUBLIC_PAGES'       , 'Determines if the application-level authorization scheme should be checked on public pages',
      'authorize_batch_job'          , 'Determines if the authorization schemes should be checked when running backround job (i.e subscription )',
      'GLOBAL_NOTIFICATION'          , 'This text is displayed on every page of the application that uses a page template with the #GLOBAL_NOTIFICATION# substitution string',
      'Session_State_Protection'     , 'Enabling Session State Protection can prevent hackers from tampering with URLs within your application.  When enabled pages and items can be set to require checksums to validate inputs.',
      'created_by'                   , 'APEX developer who created the application',
      'created_on'                   , 'Date of creation',
      'last_updated_by'              , 'APEX developer who made last update',
      'last_updated_on'              , 'Date of last update',
      'pages'                        , 'Count of subordinate objects',
      'tabs'                         , 'Count of subordinate objects',
      'parent_tabs'                  , 'Count of subordinate objects',
      'application_items'            , 'Count of subordinate objects',
      'application_processes'        , 'Count of subordinate objects',
      'application_computations'     , 'Count of subordinate objects',
      'application_settings'         , 'Count of subordinate objects',
      'shortcuts'                    , 'Count of subordinate objects',
      'web_services'                 , 'Count of subordinate objects',
      'trees'                        , 'Count of subordinate objects',
      'build_options'                , 'Count of subordinate objects',
      'breadcrumbs'                  , 'Count of subordinate objects',
      'nav_bar_entries'              , 'Count of subordinate objects',
      'lists'                        , 'Count of subordinate objects',
      'lists_of_values'              , 'Count of subordinate objects',
      'themes'                       , 'Count of subordinate objects',
      'authentication_schemes'       , 'Count of subordinate objects',
      'authorization_schemes'        , 'Count of subordinate objects',
      'application_acl_roles'        , 'Count of subordinate objects',
      'translation_messages'         , 'Count of subordinate objects',
      'installation_scripts'         , 'Count of subordinate objects',
      'cached_pages'                 , 'Count of pages in this application that are defined as cachable',
      'cached_regions'               , 'Count of page regions in this application that are defined as cachable',
      'component_signature'          , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'workspace_id'                 , 'Primary Key of the Workspace',
      'Maximum_Session_Life_Seconds' , 'Maximum lifetime of session in seconds',
      'Session_Lifetime_Exceeded_URL' , 'Go to URL when session lifetime exceeded',
      'Maximum_Session_Idle_Seconds' , 'Maximum session idle time in seconds',
      'Session_Idle_Time_Exceeded_URL' , 'Go to URL when session idle time exceeded',
      'Session_Timeout_Warn_Seconds' , 'The time in seconds before a session times out, to display a warning',
      'browser_cache'                , 'Determines if the browser should cache pages',
      'browser_frame'                , 'Determines if the browser should allow to display pages in a frame',
      'referrer_policy'              , 'Determines the referrer-policy HTTP response header',
      'deep_linking'                 , 'Determines if deep linking into the application is allowed',
      'compatibility_mode'           , 'Compatibility Mode is used to determine how the runtime engine should behave when the application is executed',
      'html_escaping_mode'           , 'HTML Escaping Mode is used to determine if the Basic or an Extended algorithm for html escaping should be used',
      'feedback'                     , 'Determines if feedback can be accepted for the application.',
      'email_from'                   , 'Identifies email address to use as from address when sending email from Interactive report email download or subscription',
      'file_prefix'                  , 'File prefix which is used by the application to load additional files like CSS, Javascript and images.',
      'files_version'                , 'Version counter of the attached application files.',
      'THEME_NUMBER'                 , 'Identifies the applications current user interface theme (obsolete)',
      'home_link'                    , 'URL used to navigate to the home page of the application (obsolete)',
      'login_url'                    , 'When running this application redirect to this URL (obsolete)',
      'page_template'                , 'The default page template for displaying pages, may be overridden at the page level (obsolete)',
      'error_page_template'          , 'Display unexpected errors on an error page using this page template. (obsolete)',
      'content_delivery_network'     , 'Obsolete. Will be removed in a future version.',
      'javascript_file_urls'         , 'Allows definition of JavaScript file URLs that will be loaded on every page in the application. (obsolete)',
      'include_legacy_javascript'    , 'Specified whether the legacy JavaScript functions are included in the application. (obsolete)',
      'print_server_type'            , 'Application-level Report Printing settings type.',
      'print_server_type_code'       , 'Identifies the internal code of print_server_type',
      'print_remote_server_id'       , 'Remote Print Server internal ID',
      'print_remote_server_static_id' , 'Static ID to identify the Remote Print Server',
      'print_remote_server_name'     , 'Name to identify the Remote Print Server',
      'print_credential_id'          , 'Internal ID of the credential store assigned to this Remote Print Server',
      'print_credential_static_id'   , 'Static ID of the credential assigned to this Remote Print Server',
      'print_credential_name'        , 'Name of the credential assigned to this Remote Print Server',
      'is_pwa'                       , 'Determines if application is a Progressive Web App',
      'pwa_is_installable'           , 'Determines if application is installable using Progressive Web App methods.',
      'pwa_manifest_description'     , 'The PWA app description explains what the application does',
      'pwa_manifest_display'         , 'The PWA display determines the preferred display mode for the app.',
      'pwa_manifest_orientation'     , 'The PWA orientation determines the device orientation expectation',
      'pwa_manifest_theme_color'     , 'The PWA theme color determines the default color when the application is launching',
      'pwa_manifest_background_color' , 'The PWA background color determines the background color when the application is launching',
      'pwa_manifest_icon_url'        , 'Legacy. The PWA icon to be used on a device home screen',
      'pwa_manifest_custom'          , 'The PWA custom manifest is used to store additional JSON properties for the declarative manifest',
      'pwa_service_worker_hooks'     , 'The PWA service worker hooks used to customize the service worker through JavaScript functions',
      'pwa_apple_statusbar_style'    , 'Meta tag value to set the style of the status bar for a web application on Apple devices',
      'pwa_apple_statusbar_style_code' , 'Identifies the internal code of pwa_apple_statusbar_style',
      'application_comment'          , 'APEX Application comments',
      'copyright_banner'             , 'APEX Application copyright text. This will be included in the APEX application export file.',
      'mle_environment'              , 'The MLE Environment in which to execute all MLE snippets within the application.'
    )
  );
end;
/
create or replace view apex_application_trans_map
as
select f.workspace                      workspace,
       f.workspace_display_name,
       m.ID                             map_id,
       m.primary_language_flow_id       primary_application_id,
       f.application_name               primary_application_name,
       m.translation_flow_id            translated_application_id,
       m.translation_flow_language_code translated_app_language,
       m.translation_image_directory    translated_appl_img_dir,
       m.translation_comments           translation_comments,
       m.direction_right_to_left        direction_right_to_left,
       m.map_comments                   translation_map_comments,
       m.last_updated_by,
       m.last_updated_on,
       m.created_by,
       m.created_on
  from wwv_flow_authorized f,
       wwv_flow_language_map m
 where m.primary_language_flow_id = f.application_id
   and f.workspace_id             = m.security_group_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_trans_map',
    p_table_comment     => 'Application Groups defined per workspace.  Applications can be associated with an application group.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                 , 'A work area mapped to one or more database schemas',
      'workspace_display_name'    , 'Display name for the workspace',
      'map_id'                    , 'Unique ID that identifies this translation mapping',
      'primary_application_id'    , 'Unique ID of the application that is the target of the translation',
      'primary_application_name'  , 'Name of the application that is the target of the translation',
      'translated_application_id' , 'Unique ID of the translated application',
      'translated_app_language'   , 'Language code, for example "fr" or "pt-br"',
      'translated_appl_img_dir'   , 'Optional directory of translated images',
      'translation_comments'      , 'Comments associated with this translation',
      'direction_right_to_left'   , 'Indicates whether page is rendered left to right or right to left ',
      'translation_map_comments'  , 'Comments associated with this mapping',
      'LAST_UPDATED_BY'           , 'Last user to update this translation mapping',
      'LAST_UPDATED_ON'           , 'Date of last update to this translation mapping',
      'CREATED_BY'                , 'User that created this translation mapping',
      'CREATED_ON'                , 'Date this translation mapping was created'
    )
  );
end;
/

create or replace view apex_application_groups as
select
    w.workspace,
    w.workspace_display_name,
    g.ID                         application_group_id,
    g.group_name                 group_name,
    g.group_comment              group_comment,
    w.workspace_id
from
     wwv_flow_application_groups g,
     wwv_flow_current_workspaces w
where w.workspace_id = g.security_group_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_groups',
    p_table_comment     => 'Application Groups defined per workspace.  Applications can be associated with an application group.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'group_name'             , 'Identifies the application group',
      'group_comment'          , 'Identifies comments for a given application group',
      'application_group_id'   , 'Identifies the ID of the application group this application is associated with',
      'workspace_id'           , 'Primary key of the corresponding workspace'
    )
  );
end;
/
--==============================================================================
prompt ...apex_application_substitutions
create or replace view apex_application_substitutions as
select workspace,
       workspace_display_name,
       application_id,
       application_name,
       substitution_string,
       substitution_value
  from ( select auth.workspace,
                auth.workspace_display_name,
                f.id                     as application_id,
                f.name                   as application_name,
                f.substitution_string_01, f.substitution_value_01,
                f.substitution_string_02, f.substitution_value_02,
                f.substitution_string_03, f.substitution_value_03,
                f.substitution_string_04, f.substitution_value_04,
                f.substitution_string_05, f.substitution_value_05,
                f.substitution_string_06, f.substitution_value_06,
                f.substitution_string_07, f.substitution_value_07,
                f.substitution_string_08, f.substitution_value_08,
                f.substitution_string_09, f.substitution_value_09,
                f.substitution_string_10, f.substitution_value_10,
                f.substitution_string_11, f.substitution_value_11,
                f.substitution_string_12, f.substitution_value_12,
                f.substitution_string_13, f.substitution_value_13,
                f.substitution_string_14, f.substitution_value_14,
                f.substitution_string_15, f.substitution_value_15,
                f.substitution_string_16, f.substitution_value_16,
                f.substitution_string_17, f.substitution_value_17,
                f.substitution_string_18, f.substitution_value_18,
                f.substitution_string_19, f.substitution_value_19,
                f.substitution_string_20, f.substitution_value_20
            from wwv_flow_authorized auth,
                wwv_flows f
            where f.security_group_id = auth.workspace_id
              and f.id                = auth.application_id )
 unpivot (
    (substitution_string, substitution_value)
    for rec in (
        (substitution_string_01, substitution_value_01) as '01',
        (substitution_string_02, substitution_value_02) as '02',
        (substitution_string_03, substitution_value_03) as '03',
        (substitution_string_04, substitution_value_04) as '04',
        (substitution_string_05, substitution_value_05) as '05',
        (substitution_string_06, substitution_value_06) as '06',
        (substitution_string_07, substitution_value_07) as '07',
        (substitution_string_08, substitution_value_08) as '08',
        (substitution_string_09, substitution_value_09) as '09',
        (substitution_string_10, substitution_value_10) as '10',
        (substitution_string_11, substitution_value_11) as '11',
        (substitution_string_12, substitution_value_12) as '12',
        (substitution_string_13, substitution_value_13) as '13',
        (substitution_string_14, substitution_value_14) as '14',
        (substitution_string_15, substitution_value_15) as '15',
        (substitution_string_16, substitution_value_16) as '16',
        (substitution_string_17, substitution_value_17) as '17',
        (substitution_string_18, substitution_value_18) as '18',
        (substitution_string_19, substitution_value_19) as '19',
        (substitution_string_20, substitution_value_20) as '20' ))
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_substitutions',
    p_table_comment     => 'Application level definitions of substitution strings.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'           , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'      , 'Application Primary Key, Unique over all workspaces',
      'application_name'    , 'Identifies the application',
      'substitution_string' , 'Name of substitution string',
      'substitution_value'  , 'Value of substitution string'
    )
  );
end;
/

--==============================================================================
prompt ...apex_application_caching
create or replace view apex_application_caching as
select
    a.workspace,
    a.workspace_display_name,
    a.application_id,
    a.application_name,
    c.page_id                                           page_id,
    (select name
     from wwv_flow_steps s
     where s.security_group_id = c.security_group_id
       and s.flow_id           = c.flow_id
       and s.id                = c.page_id)             page_name,
    case
    when c.region_id is null then 'Page Cache'
    when p.plug_source_type like 'NATIVE%CHART%' then 'Chart Region Cache'
    else 'Region Cache'
    end                                                 cache_type,
    c.language                                          language,
    c.user_name                                         caching_user,
    sys.dbms_lob.getlength(c.page_text)                 cache_size,
    p.plug_name                                         region_name,
    c.cached_on                                         cached_on,
    case
    when c.region_id is null then ( select cache_timeout_seconds
                                      from wwv_flow_steps s
                                     where s.security_group_id = c.security_group_id
                                       and s.flow_id           = c.flow_id
                                       and s.id                = c.page_id )
    else p.plug_caching_max_age_in_sec
    end                                                 cached_for_seconds,
    --
    round((sysdate - c.cached_on) * 3600 * 24,0)        age_in_seconds,
    --
    case
    when c.region_id is null then ( select cache_timeout_seconds
                                      from wwv_flow_steps s
                                     where s.security_group_id = c.security_group_id
                                       and s.flow_id           = c.flow_id
                                       and s.id                = c.page_id )
    else p.plug_caching_max_age_in_sec
    end - round((sysdate - c.cached_on) * 3600 * 24, 0) timeout_in_seconds,
    c.REGION_ID                                         region_id,
    c.SECURITY_GROUP_ID                                 workspace_id
from
     wwv_flow_page_cache c,
     wwv_flow_page_plugs p,
     wwv_flow_authorized a
where c.flow_id           = a.application_id
  and c.security_group_id = a.workspace_id
  and c.flow_id           = p.flow_id(+)
  and c.region_id         = p.id(+)
  and c.security_group_id = p.security_group_id(+)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_caching',
    p_table_comment     => 'Applications defined in the current workspace or database user.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'           , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'      , 'Application Primary Key, Unique over all workspaces',
      'application_name'    , 'Identifies the application',
      'page_name'           , 'Identifies page name',
      'page_id'             , 'Identifies page number',
      'cache_type'          , 'Cache type, Page, Region, or Chart Region',
      'language'            , 'Language of Cache',
      'caching_user'        , 'User who caused the page or region to be cached',
      'cache_size'          , 'Size of the cache, sum this column to see full cache size',
      'region_name'         , 'Identifies region name, null for page caches',
      'cached_on'           , 'Date cache was created',
      'cached_for_seconds'  , 'Iimeout in seconds identified in page or region caching meta data',
      'age_in_seconds'      , 'Seconds elapsed since cache was created',
      'timeout_in_seconds'  , 'Seconds until cache will expire.  Negitive values indicate an expired cache',
      'region_id'           , 'Corresponding primary key of region',
      'workspace_id'        , 'Corresponding primary key of workspace'
    )
  );
end;
/

create or replace view apex_standard_conditions
as
          select 'EXISTS' d,'Exists (SQL query returns at least one row)'r from sys.dual
union all select 'NOT_EXISTS' d,'NOT Exists (SQL query returns no rows)'r from sys.dual
union all select 'EXPRESSION' d,'Expression'r from sys.dual
union all select 'FUNCTION_BODY' d,'Function Body Returning a Boolean'r from sys.dual
union all select 'REQUEST_EQUALS_CONDITION' d,'Request = Expression 1'r from sys.dual
union all select 'REQUEST_NOT_EQUAL_CONDITION' d,'Request != Expression 1'r from sys.dual
union all select 'REQUEST_IN_CONDITION' d,'Request Is Contained within Expression 1'r from sys.dual
union all select 'REQUEST_NOT_IN_CONDITION' d,'Request Is NOT Contained within Expression 1'r from sys.dual
union all select 'VAL_OF_ITEM_IN_COND_EQ_COND2' d,'Value of Item in Expression 1 = Expression 2'r from sys.dual
union all select 'VAL_OF_ITEM_IN_COND_NOT_EQ_COND2' d,'Value of Item in Expression 1 != Expression 2'r from sys.dual
union all select 'ITEM_IS_NULL' d,'Value of Item / Column in Expression 1 Is NULL'r from sys.dual
union all select 'ITEM_IS_NOT_NULL' d,'Value of Item / Column in Expression 1 Is NOT NULL'r from sys.dual
union all select 'ITEM_IS_ZERO' d,'Value of Item / Column in Expression 1 = Zero'r from sys.dual
union all select 'ITEM_IS_NOT_ZERO' d,'Value of Item / Column in Expression 1 != Zero'r from sys.dual
union all select 'ITEM_IS_NULL_OR_ZERO' d,'Value of Item / Column in Expression 1 Is NULL or Zero'r from sys.dual
union all select 'ITEM_NOT_NULL_OR_ZERO' d,'Value of Item / Column in Expression 1 Is NOT null and the Item / Column Is NOT Zero'r from sys.dual
union all select 'ITEM_CONTAINS_NO_SPACES' d,'Value of Item / Column in Expression 1 Contains No Spaces'r from sys.dual
union all select 'ITEM_IS_NUMERIC' d,'Value of Item / Column in Expression 1 Is Numeric' r from sys.dual
union all select 'ITEM_IS_NOT_NUMERIC' d,'Value of Item / Column in Expression 1 Is Not Numeric' r from sys.dual
union all select 'ITEM_IS_ALPHANUMERIC' d,'Value of Item / Column in Expression 1 Is Alphanumeric'r from sys.dual
union all select 'VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST' d,'Value of Item / Column in Expression 1 Is Contained within Colon Delimited List in Expression 2'r from sys.dual
union all select 'VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST' d,'Value of Item / Column in Expression 1 Is NOT Contained within Colon Delimited List in Expression 2'r from sys.dual
union all select 'USER_PREF_IN_COND_EQ_COND2' d,'Value of User Preference in Expression 1 = Expression 2'r from sys.dual
union all select 'USER_PREF_IN_COND_NOT_EQ_COND2' d,'Value of User Preference in Expression 1 != Expression 2'r from sys.dual
union all select 'CURRENT_PAGE_EQUALS_CONDITION' d,'Current page = Expression 1'r from sys.dual
union all select 'CURRENT_PAGE_NOT_EQUAL_CONDITION' d,'Current page != Expression 1'r from sys.dual
union all select 'CURRENT_PAGE_IN_CONDITION' d,'Current Page Is Contained Within Expression 1 (comma delimited list of pages)'r from sys.dual
union all select 'CURRENT_PAGE_NOT_IN_CONDITION' d,'Current Page Is NOT in Expression 1 (comma delimited list of pages)'r from sys.dual
union all select 'WHEN_THIS_PAGE_SUBMITTED' d,'Current Page = Page Submitted (this page was posted)'r from sys.dual
union all select 'WHEN_THIS_PAGE_NOT_SUBMITTED' d,'Current Page != Page Submitted (this page was not the page posted)'r from sys.dual
union all select 'PAGE_IS_IN_PRINTER_FRIENDLY_MODE' d,'Current Page Is in Printer Friendly Mode'r from sys.dual
union all select 'PAGE_IS_NOT_IN_PRINTER_FRIENDLY_MODE' d,'Current page is NOT in Printer Friendly Mode'r from sys.dual
union all select 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2' d,'Text in Expression 1 Is Contained in Value of Item in Expression 2'r from sys.dual
union all select 'DISPLAY_COND_IN_COND_TEXT' d,'Text in Expression 1 Is Contained within the Text in Expression 2'r from sys.dual
union all select 'DISPLAY_COND_NOT_IN_COND_TEXT' d,'Text in Expression 1 Is NOT Contained within the Text in Expression 2'r from sys.dual
union all select 'DISPLAY_COND_EQUAL_COND_TEXT' d,'Text in Expression 1 = Expression 2 (includes &ITEM. substitutions)'r from sys.dual
union all select 'DISP_COND_NOT_EQUAL_COND_TEXT' d,'Text in Expression 1 != Expression 2 (includes &ITEM. substitutions)'r from sys.dual
union all select 'IS_READ_ONLY' d,'Page/Region is Read Only' r from sys.dual
union all select 'IS_NOT_READ_ONLY' d,'Page/Region is NOT Read Only' r from sys.dual
union all select 'USER_IS_NOT_PUBLIC_USER' d,'User is Authenticated  (not public)'r from sys.dual
union all select 'USER_IS_PUBLIC_USER' d,'User is the Public User (user has not authenticated)'r from sys.dual
union all select 'DISPLAYING_INLINE_VALIDATION_ERRORS' d,'Inline Validation Errors Displayed'r from sys.dual
union all select 'NOT_DISPLAYING_INLINE_VALIDATION_ERRORS' d,'No Inline Validation Errors Displayed'r from sys.dual
union all select 'MAX_ROWS_LT_ROWS_FETCHED' d,'SQL Reports (OK to show the forward button)'r from sys.dual
union all select 'MIN_ROW_GT_THAN_ONE' d,'SQL Reports (OK to show the back button)'r from sys.dual
union all select 'BROWSER_IS_NSCP' d,'Client Browser: Mozilla, Netscape 6.x/7x or higher'r from sys.dual
union all select 'BROWSER_IS_MSIE' d,'Client Browser: Microsoft Internet Explorer 5.5, 6.0 or higher'r from sys.dual
union all select 'BROWSER_IS_MSIE_OR_NSCP' d,'Client Browser: XHTML / CSS capable browser (NS 6.x,7x, Mozilla, IE 5.5, 6.0 or higher)'r from sys.dual
union all select 'BROWSER_IS_OTHER' d,'Client Browser: Other browsers (or older version)'r from sys.dual
union all select 'CURRENT_LANG_IN_COND1' d,'Current Language Is Contained within Expression 1'r from sys.dual
union all select 'CURRENT_LANG_NOT_IN_COND1' d,'Current Language Is NOT Contained within Expression 1'r from sys.dual
union all select 'CURRENT_LANG_NOT_EQ_COND1' d,'Current Language != Expression 1'r from sys.dual
union all select 'CURRENT_LANG_EQ_COND1' d,'Current Language = Expression 1'r from sys.dual
union all select 'DAD_NAME_EQ_CONDITION' d,'When CGI_ENV DAD_NAME = Expression 1'r from sys.dual
union all select 'DAD_NAME_NOT_EQ_CONDITION' d,'When CGI_ENV DAD_NAME != Expression 1'r from sys.dual
union all select 'SERVER_NAME_EQ_CONDITION' d,'When CGI_ENV SERVER_NAME = Expression 1'r from sys.dual
union all select 'SERVER_NAME_NOT_EQ_CONDITION' d,'When CGI_ENV SERVER_NAME != Expression 1'r from sys.dual
union all select 'HTTP_HOST_EQ_CONDITION' d,'When CGI_ENV HTTP_HOST = Expression 1'r from sys.dual
union all select 'HTTP_HOST_NOT_EQ_CONDITION' d,'When CGI_ENV HTTP_HOST != Expression 1'r from sys.dual
union all select 'NEVER' d,'Never'r from sys.dual
union all select 'ALWAYS' d,'Always'r from sys.dual ;

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_standard_conditions',
    p_table_comment     => 'Internal condition names and their descriptions',
    p_column_comments   => null );
end;
/

prompt ...apex_appl_user_interfaces

create or replace view apex_appl_user_interfaces
as
select ui.id                     as user_interface_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       ui.ui_type_id,
       uit.name                  as ui_type_name,
       ui.display_name,
       ui.display_seq,
       case ui.use_auto_detect
         when 'Y' then 'Yes'
         else 'No'
       end as use_auto_detect,
       case ui.is_default
         when 'Y' then 'Yes'
         else 'No'
       end as is_default,
       case ui.theme_style_by_user_pref
         when 'Y' then 'Yes'
         else 'No'
       end                       as theme_style_by_user_pref,
       case ui.built_with_love
         when 'Y' then 'Yes'
         else 'No'
       end                       as built_with_love,
       ui.theme_id               as theme_number,
       ui.home_url,
       ui.login_url,
       ui.global_page_id,
       (select fi.name from wwv_flow_lists fi where f.workspace_id = fi.security_group_id and fi.id = ui.navigation_list_id) as navigation_list,
       ui.navigation_list_id,
       ui.navigation_list_position,
       (select flt.list_template_name from wwv_flow_list_templates flt where f.workspace_id = flt.security_group_id and flt.id = ui.navigation_list_template_id) navigation_list_template,
       ui.navigation_list_template_id,
       ui.nav_list_template_options,
       --
       ui.nav_bar_type,
       (select fi.name from wwv_flow_lists fi where f.workspace_id = fi.security_group_id and fi.id = ui.nav_bar_list_id) as nav_bar_list,
       ui.nav_bar_list_id,
       (select flt.list_template_name from wwv_flow_list_templates flt where f.workspace_id = flt.security_group_id and flt.id = ui.nav_bar_list_template_id) nav_bar_list_template,
       ui.nav_bar_list_template_id,
       ui.nav_bar_template_options,
       --
       null as content_delivery_network,
       ui.javascript_file_urls,
       ui.css_file_urls,
       case
         when ui.include_legacy_javascript is not null then 'Yes'
         else 'No'
       end                       as include_legacy_javascript,
       case ui.include_jquery_migrate
         when 'Y' then 'Yes'
         else 'No'
       end                       as include_jquery_migrate,
       ui.required_patch         as build_option_id,
       ( select case
                  when ui.required_patch > 0 then pa.patch_name
                  else '{Not ' || pa.patch_name || '}'
                end
           from wwv_flow_patches pa
          where f.workspace_id = pa.security_group_id
          and pa.id = abs(ui.required_patch)
       )                         as build_option,
       ui.created_by,
       ui.created_on,
       ui.last_updated_by,
       ui.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_user_interfaces ui,
       wwv_flow_ui_types uit
 where ui.flow_id           = f.application_id
   and ui.security_group_id = f.workspace_id
   and uit.id               = ui.ui_type_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_user_interfaces',
    p_table_comment     => 'User interfaces of an application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'content_delivery_network'    , 'Obsolete. Will be removed in a future version.',
      'javascript_file_urls'        , 'Allows definition of JavaScript file URLs that will be loaded on every page in the application.',
      'include_legacy_javascript'   , 'Specified whether the legacy JavaScript functions are included in the application.',
      'include_jquery_migrate'      , 'Specified whether the jQuery Migrate plug-in is included in the application.',
      'navigation_list_position'    , 'Default position used for navigation menu lists',
      'navigation_list_template'    , 'Template used for navigation mune lists',
      'navigation_list_template_id' , 'Template used for navigation menu lists',
      'nav_list_template_options'   , 'Template options used for navigation menu lists',
      'nav_bar_type'                , 'Default display type used for navigation bar entries',
      'nav_bar_list'                , 'Identifies navigation bar list',
      'nav_bar_list_id'             , 'List used for navigation bar entries rendered as a traditional list',
      'nav_bar_list_template'       , 'Identifies navigation bar list template',
      'nav_bar_list_template_id'    , 'List template used for navigation bar entries rendered as a traditional list',
      'nav_bar_template_options'    , 'Template options used for navigation menu lists',
      'css_file_urls'               , 'Allows definition of CSS file URLs that will be loaded on every page in the application.',
      'theme_style_by_user_pref'    , 'Identifies whether end users can change session theme styles with a customization dialog',
      'built_with_love'             , 'Identifies if Build with Love using Oracle APEX gets displayed in the footer.',
      'USER_INTERFACE_ID'           , 'Primary key of the user interface',
      'WORKSPACE'                   , 'A work area mapped to one or more database schemas',
      'WORKSPACE_DISPLAY_NAME'      , 'Display name for the workspace',
      'APPLICATION_ID'              , 'Application Primary Key, Unique over all workspaces',
      'APPLICATION_NAME'            , 'Identifies the application',
      'UI_TYPE_ID'                  , 'Identifies the primary key of the user interface type',
      'UI_TYPE_NAME'                , 'Name of the user interface type',
      'DISPLAY_NAME'                , 'Name of the user interface',
      'DISPLAY_SEQ'                 , 'Sequence in which to display this user interface',
      'USE_AUTO_DETECT'             , 'Automatically detect the appropriate user interface',
      'IS_DEFAULT'                  , 'Denotes the default user interface for an application',
      'THEME_NUMBER'                , 'Numeric identifier of the theme to use for this user interface',
      'HOME_URL'                    , 'Home page of the application for the user interface.',
      'LOGIN_URL'                   , 'Login page of the application for this user interface.',
      'GLOBAL_PAGE_ID'              , 'Numeric identifier of the global page for this user interface.',
      'NAVIGATION_LIST'             , 'Name of the list utilized for the navigation menu for the application.',
      'NAVIGATION_LIST_ID'          , 'Numeric identifier of the navigation menu list.',
      'BUILD_OPTION_ID'             , 'Identifies the primary key of build option.',
      'BUILD_OPTION'                , 'User interface will be available if the Build Option is enabled',
      'CREATED_BY'                  , 'APEX developer who created the user interface',
      'CREATED_ON'                  , 'Date of creation',
      'LAST_UPDATED_BY'             , 'APEX developer who made last update',
      'LAST_UPDATED_ON'             , 'Date of last update'
    )
  );
end;
/

prompt ...apex_appl_concatenated_files

create or replace view apex_appl_concatenated_files
as
select cf.id                     as concatenated_file_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       cf.user_interface_id,
       cf.page_id,
       cf.combined_file_url,
       cf.single_file_urls,
       cf.required_patch         as build_option_id,
       ( select case
                  when cf.required_patch > 0 then pa.patch_name
                  else '{Not ' || pa.patch_name || '}'
                end
           from wwv_flow_patches pa
          where f.workspace_id = pa.security_group_id
          and pa.id = abs(cf.required_patch)
       )                         as build_option,
       cf.created_by,
       cf.created_on,
       cf.last_updated_by,
       cf.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_combined_files cf
 where cf.security_group_id = f.workspace_id
   and cf.flow_id           = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_concatenated_files',
    p_table_comment     => 'Concatenated files of a user interface',
    p_column_comments   => wwv_flow_t_varchar2 (
      'concatenated_file_id'   , 'Identifies the primary key of this concatenated file',
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'user_interface_id'      , 'Identifies the user interface',
      'page_id'                , 'Concatenated file is only used for specified page id',
      'combined_file_url'      , 'File URL of the concatenated file to be loaded, instead of multiple single JavaScript or CSS files.',
      'single_file_urls'       , 'Multiple file URLs for which a concatenated file should be loaded instead.',
      'build_option'           , 'Concatenated file will be used if the build option is enabled',
      'created_by'             , 'APEX developer who created the concatenated file',
      'created_on'             , 'Date of creation',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'build_option_id'        , 'Identifies the primary key of the build option'
    )
  );
end;
/


create or replace view apex_application_static_files
as
select a.id                     as application_file_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       a.file_name,
       a.mime_type,
       a.file_charset,
       a.file_content,
       a.created_by,
       a.created_on,
       a.last_updated_by,
       a.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_static_files a
 where a.security_group_id = f.workspace_id
   and a.flow_id           = f.application_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_static_files',
    p_table_comment     => 'Stores the files like CSS, images, javascript files, ... of an application.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'application_file_id' , 'Identifies the primary key of this file',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'file_name' , 'Name of the file.',
      'mime_type' , 'Mime type of the file.',
      'file_charset' , 'IANA charset used for text files.',
      'file_content' , 'Blob content of the file.',
      'created_by' , 'APEX developer who created the file',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/

prompt ...apex_appl_developer_comments

create or replace view apex_appl_developer_comments
as
select c.id                     as comment_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       c.pages,
       c.app_comment as developer_comment,
       c.created_by,
       c.created_on,
       c.updated_by as last_updated_by,
       c.updated_on as last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_app_comments c
 where c.security_group_id = f.workspace_id
   and c.flow_id           = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_developer_comments',
    p_table_comment     => 'Developer comments of an application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'comment_id'             , 'Identifies the primary key of this developer comment',
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'pages'                  , 'Relevant Page(s) for this developer comment',
      'developer_comment'      , 'Developer comment',
      'created_by'             , 'APEX developer who created the developer comment',
      'created_on'             , 'Date of creation',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update'
    )
  );
end;
/



prompt ...apex_application_locked_pages

create or replace view apex_application_locked_pages
as
select l.id                     as lock_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       p.id   as page_id,
       p.name as page_name,
       l.lock_comment,
       l.locked_by,
       l.locked_on
  from wwv_flow_authorized f,
       wwv_flow_lock_page l,
       wwv_flow_steps p
 where l.flow_id           = f.application_id
   and l.security_group_id = f.workspace_id
   and p.id                = l.object_id
   and p.flow_id           = l.flow_id
   and p.security_group_id = l.security_group_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_locked_pages',
    p_table_comment     => 'Locked pages of an application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'lock_id'                , 'Identifies the primary key of this page lock',
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'page_id'                , 'Page id which is locked',
      'page_name'              , 'Identifies the page',
      'lock_comment'           , 'Lock comment',
      'locked_by'              , 'APEX developer who locked the page',
      'locked_on'              , 'Date of lock'
    )
  );
end;
/

--------------------------------------------
-- A P P L I C A T I O N   P A G E S
-- 4301

prompt ...apex_application_pages

create or replace view apex_application_pages
as
select
    f.workspace,
    f.workspace_display_name,
    p.flow_id                             application_id,
    f.application_name,
    --
    p.id                                  page_id,
    p.name                                page_name,
    p.STEP_TITLE                          page_title,
    --
    decode(nvl(p.page_mode,'NORMAL'),
       'NORMAL','Normal',
       'MODAL','Modal Dialog',
       'NON_MODAL','Non-Modal Dialog')            page_mode,
    --
    p.user_interface_id,
    p.media_type                          media_type,
    p.TAB_SET                             tab_set,
    p.ALIAS                               page_alias,
    decode(substr(p.PAGE_COMPONENT_MAP,1,2),
         '01','Tabular Form',
         '02','Form',
         '03','Report',
         '04','Chart',
         '05','Web Service',
         '06','Navigation Page',
         '07','Tree',
         '08','Calendar',
         '09','URL',
         '10','Dynamic HTML',
         '11','Static HTML',
         '12','Login',
         '13','Home',
         '14','Global Page',
         '15','Empty Page',
         '16','Dynamic Form',
         '17','Wizard Form',
         '18','Interactive Report',
         '19','Map',
         '20','Tree',
         '21','Interactive Grid',
         '22','Faceted Search',
         '23','Card',
         '24','Smart Filters',
         'Unknown')                       page_function,
    decode(
      p.ALLOW_DUPLICATE_SUBMISSIONS,
      'N','No','Y','Yes',null,'Yes',
      p.ALLOW_DUPLICATE_SUBMISSIONS)      allow_duplicate_submissions,
    --
    decode(INCLUDE_APEX_CSS_JS_YN,null,'Yes','Y','Yes','N','No','Unknown') INCLUDE_APEX_CSS_JS_YN,
    --
    decode(p.FIRST_ITEM,
      'AUTO_FIRST_ITEM','First Item on Page',
      'NO_FIRST_ITEM','Do not focus cursor',
      p.FIRST_ITEM)                       focus_cursor,
    p.WELCOME_TEXT                        Header_Text,
    p.BOX_WELCOME_TEXT                    Body_Header,
    --p.BOX_FOOTER_TEXT                     Body_footer,
    p.FOOTER_TEXT                         Footer_Text,
    p.HELP_TEXT                           help_text,
    (select name
     from wwv_flow_templates
     where id = coalesce( p.step_template, ( select case
                                                      when p.page_mode <> 'NORMAL'
                                                      then t.default_dialog_template
                                                      else t.default_page_template
                                                    end
                                               from wwv_flow_user_interfaces ui,
                                                    wwv_flow_themes t
                                              where ui.security_group_id = p.security_group_id
                                                and ui.flow_id           = p.flow_id
                                                and ui.ui_type_id        = 1 -- desktop
                                                and t.security_group_id  = ui.security_group_id
                                                and t.flow_id            = ui.flow_id
                                                and t.theme_id           = ui.theme_id ) ))          page_template,
    p.page_css_classes                    page_css_classes,
    p.page_template_options               page_template_options,
     --
    decode(substr(p.REQUIRED_ROLE,1,1),'!','Not ')||
    nvl((select ss.name
     from   wwv_flow_security_schemes ss
     where  to_char(ss.id)       = ltrim(p.REQUIRED_ROLE,'!')
       and  ss.security_group_id = f.workspace_id
       and  ss.flow_id           = f.application_id),
     p.REQUIRED_ROLE)                     authorization_scheme,
    p.required_role                       authorization_scheme_id,
     --
    (select case when p.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
     from   wwv_flow_patches pa
     where  pa.security_group_id = p.security_group_id
       and  pa.id =abs(p.REQUIRED_PATCH))         build_option,
    p.HTML_PAGE_HEADER                    page_html_header,
    p.HTML_PAGE_ONLOAD                    page_html_onload,
    p.javascript_file_urls,
    p.javascript_code,
    p.javascript_code_onload,
    p.css_file_urls,
    p.inline_css,
    --sec
    decode(p.PAGE_IS_PUBLIC_Y_N,
      'N','Yes','Y','No','Yes')            page_requires_Authentication,
    decode(nvl(p.PROTECTION_LEVEL,'N'),
       'N','Unrestricted',
       'C','Arguments Must Have Checksum',
       'U','No Arguments Allowed',
       'D','No URL Access',
       p.PROTECTION_LEVEL )               page_access_protection,
    --
    p.dialog_title                        dialog_title,
    p.dialog_height                       dialog_height,
    p.dialog_width                        dialog_width,
    p.dialog_max_width                    dialog_max_width,
    p.dialog_attributes                   dialog_attributes,
    p.dialog_css_classes                  dialog_css_classes,
    p.dialog_chained                      dialog_chained,
    --
    p.overwrite_navigation_list           overwrite_navigation_list,
    (select li.name from wwv_flow_lists li where li.security_group_id = p.security_group_id and li.id = p.navigation_list_id)                                           navigation_list,
    p.navigation_list_id,
    (select te.list_template_name from wwv_flow_list_templates te where te.security_group_id = p.security_group_id and te.id = p.navigation_list_template_id)           navigation_list_template,
    p.navigation_list_template_id,
    p.nav_list_template_options,
    p.navigation_list_position,
    --
    (select count(*) from wwv_flow_page_plugs pp where pp.security_group_id = p.security_group_id and pp.flow_id = f.application_id and pp.page_id = p.id)              regions,
    (select count(*) from wwv_flow_step_items ii where ii.security_group_id = p.security_group_id and ii.flow_id = f.application_id and ii.flow_step_id = p.id)         items,
    (select count(*) from wwv_flow_step_buttons sb where sb.security_group_id = p.security_group_id and sb.flow_id = f.application_id and sb.flow_step_id = p.id)       buttons,
    (select count(*) from wwv_flow_step_computations sc where sc.security_group_id = p.security_group_id and sc.flow_id = f.application_id and sc.flow_step_id = p.id)  computations,
    (select count(*) from wwv_flow_step_validations sv where sv.security_group_id = p.security_group_id and sv.flow_id = f.application_id and sv.flow_step_id = p.id)   validations,
    (select count(*) from wwv_flow_step_processing sp where sp.security_group_id = p.security_group_id and sp.flow_id = f.application_id and sp.flow_step_id = p.id)    processes,
    (select count(*) from wwv_flow_step_branches sb where sb.security_group_id = p.security_group_id and sb.flow_id = f.application_id and sb.flow_step_id = p.id)      branches,
    (select count(*) from wwv_flow_region_report_column rc where rc.security_group_id = p.security_group_id and rc.flow_id = f.application_id and rc.region_id in
        (select pp.id from wwv_flow_page_plugs pp where pp.security_group_id = p.security_group_id and pp.flow_id = f.application_id and pp.page_id = p.id))            report_columns,
    --
    (select pg.GROUP_NAME from wwv_flow_page_groups pg where pg.security_group_id = p.security_group_id and pg.id = p.group_id and pg.flow_id = p.flow_id)              page_group,
    p.group_id                                                                                                                                                          page_group_id,
    --
    ON_DUP_SUBMISSION_GOTO_URL            ,
    p.error_handling_function,
    ERROR_NOTIFICATION_TEXT               ,
    decode(autocomplete_on_off,null,'On','ON','On','OFF','Off') form_autocomplete,
    p.page_transition,
    p.popup_transition,
    -- read only condition
    coalesce (
        ( select r
            from apex_standard_conditions
           where p.read_only_when_type is not null
             and d = p.read_only_when_type
        ),
        p.read_only_when_type )           read_only_condition_type,
    p.read_only_when_type                 read_only_condition_type_code,
    p.read_only_when                      read_only_condition_exp1,
    p.read_only_when2                     read_only_condition_exp2,
    --
    decode(cache_mode,
        'CACHE'   , 'Cached'            ,
        'USER'    , 'Cached by User'    ,
        'SESSION' , 'Cached by Session' ,
        'NOCACHE' , 'Not Cached'        ,
        cache_mode)                       cache_mode,
    decode(CACHE_MODE,
        'NOCACHE', 'No',
        'Yes')                            cached,
    CACHE_TIMEOUT_SECONDS                 cache_timeout_seconds,
    decode(CACHE_MODE,
        'USER', 'Yes',
        'No')                             cached_by_user,
    CACHE_WHEN_CONDITION_TYPE             cache_condition_type,
    CACHE_WHEN_CONDITION_E1               cache_condition_exp_1,
    CACHE_WHEN_CONDITION_E2               cache_condition_exp_2,
    page_comment                          page_comment,
    (select count(*) from wwv_flow_page_plugs pp where pp.security_group_id = p.security_group_id and pp.flow_id =  f.application_id and pp.page_id =  p.id and pp.PLUG_CACHING <> 'NOCACHE' ) cached_regions,
    --
    p.created_by,
    p.created_on,
    p.LAST_UPDATED_BY                     last_updated_by,
    p.LAST_UPDATED_ON                     last_updated_on,
    --
    lpad(p.id,5,'00000')
    ||p.step_title||
    ',tabset='||p.tab_set
    ||' help='||sys.dbms_lob.getlength(p.HELP_TEXT)||
    decode(p.BOX_WELCOME_TEXT,null,null,'bodyhead='||length(p.BOX_WELCOME_TEXT))||
    decode(p.ALIAS,null,null,' alias='||p.ALIAS)||
    p.step_template||
    nvl((select ss.name from wwv_flow_security_schemes ss where ss.security_group_id = p.security_group_id and to_char(ss.id) = ltrim(p.REQUIRED_ROLE,'!') and ss.flow_id = f.application_id),
    decode(substr(p.REQUIRED_ROLE,1,1),'!','Not ')||
    nvl((select name
     from   wwv_flow_security_schemes
     where  to_char(id) = ltrim(p.REQUIRED_ROLE,'!')
     and    flow_id = f.application_id),
     'Not found')
    )||
    --
    (select pa.PATCH_NAME from wwv_flow_patches pa where pa.security_group_id = p.security_group_id and pa.id =abs(p.REQUIRED_PATCH))||
    decode(p.FIRST_ITEM,'AUTO_FIRST_ITEM','First Item on Page','NO_FIRST_ITEM','Do not focus cursor',p.FIRST_ITEM)||
    decode(p.WELCOME_TEXT,null,null,'head='||length(p.WELCOME_TEXT))||
    decode(p.FOOTER_TEXT,null,null,',foot='||length(FOOTER_TEXT))||
    ',PgAuth='||decode(p.PAGE_IS_PUBLIC_Y_N,'N','Yes','Y','No','Yes')||
    ','||decode(nvl(p.PROTECTION_LEVEL,'N'),
       'N','Unrestricted',
       'C','Arguments Must Have Checksum',
       'U','No Arguments Allowed',
       'D','No URL Access',
       p.PROTECTION_LEVEL )||
     ',onload='||length(p.HTML_PAGE_ONLOAD)||
     ',hh='||nvl(length(p.HTML_PAGE_HEADER),0)
     component_signature,
    case p.browser_cache
      when 'Y' then 'Enabled'
      when 'N' then 'Disabled'
      else 'Application Default'
    end browser_cache,
    case p.deep_linking
      when 'Y' then 'Enabled'
      when 'N' then 'Disabled'
      else 'Application Default'
    end deep_linking,
    case nvl(p.rejoin_existing_sessions, f.rejoin_existing_sessions)
    when 'Y' then 'All Sessions'
    when 'P' then 'Public Sessions'
    when 'N' then 'Disabled'
    end rejoin_existing_sessions,
    case p.reload_on_submit
      when 'A' then 'Always'
      when 'S' then 'Only for Success'
    end                as reload_on_submit,
    p.reload_on_submit as reload_on_submit_code,
    case p.warn_on_unsaved_changes
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end as warn_on_unsaved_changes
from wwv_flow_steps      p,
     wwv_flow_authorized f
where f.workspace_id   = p.security_group_id
  and f.application_id = p.flow_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_pages',
    p_table_comment     => 'A Page definition is the basic building block of page. Page components including regions, items, buttons, computations, branches, validations, and processes further define the definition of a page.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                    , 'A work area mapped to one or more database schemas',
      'workspace_display_name'       , 'Display name for the workspace',
      'application_id'               , 'Application Primary Key, Unique over all workspaces',
      'application_name'             , 'Identifies the application',
      'page_id'                      , 'Identifies the page.  The primary key of a page is the combination of the Application ID and the Page ID.',
      'page_name'                    , 'Identifies a page within an application',
      'page_title'                   , 'Used by page templates to display the HTML title tag',
      'page_mode'                    , 'Identifies whether the page type has been defined as Normal, Modal Dialog, or Non-Modal Dialog.',
      'media_type'                   , 'Page-level Internet media type, used in the Content-Type HTTP header',
      'TAB_SET'                      , 'Identifies the standard Tab Set to be used for this page. A Tab Set is a collection of tabs, the current tab is a property of the tab and is identified by page ID.',

      'INCLUDE_APEX_CSS_JS_YN'       , 'Determines if Oracle APEX will automatically include APEX standard CSS and JS in the page hearder.  Typical value is Yes',

      'PAGE_ALIAS'                   , 'An alphanumeric identifier which can be used as a more readable identifier then the page ID.',
      'ALLOW_DUPLICATE_SUBMISSIONS'  , 'Determines if a rendered page may be posted more then once, for example by pressing reload.  Defaults to Yes.',
      'focus_cursor'                 , 'Specifies if APEX should automatically focus the cursor in the first field on page load',
      'HEADER_TEXT'                  , 'Text or HTML that will appear immediately following the Page Template Header and before the Page Template Body content.',
      'BODY_HEADER'                  , 'Text or HTML that will appear before the Page Template #BOX_BODY# substitution string',

      'FOOTER_TEXT'                  , 'Text or HTML that will appear before the Page Template Footer and after the Page Template Body content.',
      'HELP_TEXT'                    , 'Page help text',
      'PAGE_TEMPLATE'                , 'Identifies the page template, if no page template is specified the application default page template will be used',
      'PAGE_CSS_CLASSES'             , 'Identifies CSS classes to be applied to the page',
      'PAGE_TEMPLATE_OPTIONS'        , 'Identifies template options to be applied to the page template',
      'AUTHORIZATION_SCHEME'         , 'An authorization scheme must evaluate to TRUE in order for this page to be displayed',
      'AUTHORIZATION_SCHEME_ID'      , 'Foreign Key',
      'BUILD_OPTION'                 , 'Page will only be displayed if no Build Option is specified or if the Build Options is enabled',
      'PAGE_HTML_HEADER'             , 'Identifies text that replaces the #HEAD# substitution string in the Page Template',
      'PAGE_HTML_ONLOAD'             , 'Identifies JavaScript or text to be substituted in for the Page Template #ONLOAD# substitution string',
      'javascript_code'              , 'Identifies Javascript code that replaces the #PAGE_JAVASCRIPT# substitution string in the Page Template',
      'javascript_code_onload'       , 'Identifies JavaScript code which is executed after page load',
      'PAGE_REQUIRES_AUTHENTICATION' , 'Specifies whether this page has been defined as "Public" or requires authentication',
      'PAGE_ACCESS_PROTECTION'       , 'Identifies the Page Access Protection level which is used to restrict the setting of session state or to require checksums to pass session state',

      'DIALOG_TITLE'                 , 'Identifies the dialog title',
      'DIALOG_HEIGHT'                , 'Identifies the height of the dialog',
      'DIALOG_WIDTH'                 , 'Identifies the width of the dialog',
      'DIALOG_MAX_WIDTH'             , 'Identifies the maximum width of the dialog',
      'DIALOG_ATTRIBUTES'            , 'Identifies custom attributes to be applied to the dialog',
      'DIALOG_CSS_CLASSES'           , 'Identifies CSS classes to be applied to the dialog',
      'DIALOG_CHAINED'               , 'Identifies whether the dialog page is opened by another dialog page, and should reuse the same window, or whether it should be opened in a new window',

      'OVERWRITE_NAVIGATION_LIST'    , 'Defines whether to overwrite application level navigation menu list settings',
      'NAVIGATION_LIST'              , 'Identifies navigation menu list',
      'NAVIGATION_LIST_ID'           , 'Identifies navigation menu list ID',
      'NAVIGATION_LIST_TEMPLATE'     , 'Identifies navigation menu list template',
      'NAVIGATION_LIST_TEMPLATE_ID'  , 'Identifies navigation menu list template ID',
      'NAVIGATION_list_position'     , 'Identifies the position of the navigation menu list (if overwritten)',
      'NAV_LIST_TEMPLATE_OPTIONS'    , 'Identifies template options to be applied to the navigation menu list template',

      'REGIONS'                      , 'Number of regions defined for this page',
      'ITEMS'                        , 'Number of items defined for this page',
      'BUTTONS'                      , 'Number of buttons defined for this page',
      'COMPUTATIONS'                 , 'Number of computations defined for this page',
      'VALIDATIONS'                  , 'Number of validations defined for this page',
      'PROCESSES'                    , 'Number of processes defined for this page',
      'BRANCHES'                     , 'Number of branches defined for this page',
      'report_columns'               , 'Number of report columns defined for this page',
      'page_group'                   , 'Identifies assigned page group',
      'page_group_id'                , 'Foreign Key',
      'created_by'                   , 'APEX developer who created the page.',
      'created_on'                   , 'Date the page was created.',
      'last_updated_by'              , 'APEX developer who made last update',
      'last_updated_on'              , 'Date of last update',

      'component_signature'          , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'ON_DUP_SUBMISSION_GOTO_URL'   , 'Oracle APEX displays an error message and a link to this URL to continue with the application',
      'error_handling_function'      , 'Function which is called if an error occurs. It can be used to modify or log the error.',
      'ERROR_NOTIFICATION_TEXT'      , 'Enter the error message displayed when an error occurs on this page. The error message will be substituted for #NOTIFICATION_MESSAGE# in the page template.',
      'form_autocomplete'            , 'Whether or not to enable autocomplete for forms',
      'read_only_condition_type'     , 'Identifies the condition type when all items of the page will be rendered read only',
      'read_only_condition_type_code' , 'Identifies the internal code of read_only_condition_type',
      'read_only_condition_exp1'     , 'Identifies the condition that determines if all items of the page will be rendered read only',
      'read_only_condition_exp2'     , 'Identifies the condition that determines if all items of the page will be rendered read only',
      'cache_mode'                   , 'Identifies the caching method',
      'cached'                       , 'Whether or not to enable page caching',
      'cache_timeout_seconds'        , 'Amount of time for which a cached page is valid.',
      'cached_by_user'               , 'Cached by user yes or no',
      'cache_condition_type'         , 'If the condition returns true, the page is displayed from cache.',
      'cache_condition_exp_1'        , 'Cache condition expression 1',
      'cache_condition_exp_2'        , 'Cache condition expression 2',
      'page_comment'                 , 'Page comment',
      'page_function'                , 'Primary function of the page based using a best guess method.',
      'browser_cache'                , 'Determines if the browser should cache pages',
      'deep_linking'                 , 'Determines if deep linking is allowed to this page',
      'rejoin_existing_sessions'     , 'Determines whether sessions for this application can be joined without session id in the URL',
      'reload_on_submit'             , 'Determines when the page should be reloaded on submit.',
      'reload_on_submit_code'        , 'Internal code of RELOAD_ON_SUBMIT',
      'warn_on_unsaved_changes'      , 'Determines if a warning should be displayed in case of unsaved changes.',

      'USER_INTERFACE_ID'            , 'Identifies the user interface',
      'JAVASCRIPT_FILE_URLS'         , 'Allows definition of JavaScript file URLs that will be loaded into the page',
      'CSS_FILE_URLS'                , 'Allows definition of CSS file URLs that will be loaded into the page.',
      'INLINE_CSS'                   , 'Inline CSS to be included on the page.',
      'PAGE_TRANSITION'              , 'Transition which should be used when navigating to the page or when the page submitted.',
      'POPUP_TRANSITION'             , 'Transition which should be used when the page is opened as a popup.',
      'CACHED_REGIONS'               , 'Count of regions in the page that are defined as cachable.'
    )
  );
end;
/
create or replace view apex_application_page_groups
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    g.group_name                          page_group_name,
    g.group_desc                          page_group_description,
    g.id                                  group_id
from wwv_flow_page_groups g,
     wwv_flow_authorized f
where f.workspace_id   = g.security_group_id
  and f.application_id = g.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_groups',
    p_table_comment     => 'Identifies page groups',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'               , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'          , 'Application Primary Key, Unique over all workspaces',
      'application_name'        , 'Identifies the application',
      'page_group_name'         , 'page group name',
      'page_group_description'  , 'Page group description',
      'group_id'                , 'Unique numeric identifier'
    )
  );
end;
/

-----------------------------------------------
-- A P P L I C A T I O N   P A G E   I T E M S
-- 4311
prompt ...apex_appl_page_filter_groups

create or replace view apex_appl_page_filter_groups
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                            page_id,
    p.name                          page_name,
    (select pp.plug_name
     from  wwv_flow_page_plugs pp
     where pp.security_group_id = g.security_group_id
       and id = g.page_plug_id)     region,
    g.page_plug_id                  region_id,
    --
    g.label                         label,
    g.display_sequence              display_sequence,
    g.icon_css_classes              icon_css_classes,
    g.css_classes                   css_classes,
    --
    case g.suggestions_type
      when 'DYNAMIC'                   then 'Dynamic'
      when 'STATIC'                    then 'Static Values'
      when 'SQL_QUERY'                 then 'SQL Query'
      when 'FUNC_BODY_RETURNING_ARRAY' then 'Function Body Returning Array'
      else g.suggestions_type
    end                             suggestions_type,
    g.suggestions_type              suggestions_type_code,
    g.suggestions_language,
    g.suggestions_source,
    g.max_suggestions,
    --
    g.fc_max_height                 max_height,
    case g.fc_show_label
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                             show_label,
    --
    decode(g.fc_filter_combination,
       'AND','AND (Intersect)',
       'OR', 'OR (Union)',
       g.fc_filter_combination)     group_filter_combination,
    decode(g.fc_compute_counts,
       'Y','Yes',
       'N','No',
       g.fc_compute_counts)         compute_counts,
    decode(g.fc_show_counts,
       'Y','Yes',
       'N','No',
       g.fc_show_counts)            show_counts,
    case g.fc_zero_count_entries
      when 'H' then 'Hide'
      when 'D' then 'Disable'
      when 'L' then 'Show Last'
    end                             zero_count_entries,
    decode(g.fc_sort_by_top_counts,
       'Y','Yes',
       'N','No',
       g.fc_sort_by_top_counts)     sort_by_top_counts,
    decode(g.fc_show_selected_first,
       'Y','Yes',
       'N','No',
       g.fc_show_selected_first)    show_selected_first,
    fc_show_more_count              max_displayed_entries,
    --
    decode(g.fc_collapsible,
       'Y','Yes',
       'N','No',
       g.fc_collapsible)            is_collapsible,
    decode(g.fc_initial_collapsed,
       'Y','Yes',
       'N','No',
       g.fc_initial_collapsed)      is_initial_collapsed,
    decode(g.fc_show_chart,
       'Y','Yes',
       'N','No',
       g.fc_show_chart)             show_chart,
    decode(g.fc_initial_chart,
       'Y','Yes',
       'N','No',
       g.fc_initial_chart)          initial_chart,
    decode(g.fc_toggleable,
       'Y','Yes',
       'N','No',
       g.fc_toggleable)             is_toggleable,
    decode(g.fc_initial_toggled,
       'Y','Yes',
       'N','No',
       g.fc_initial_toggled)        is_initial_toggled,
    --
    --
    nvl((select r from apex_standard_conditions where d = g.condition_type),g.condition_type)
                                    condition_type,
    g.condition_type                condition_type_code,
    g.condition_expr1               condition_expression1,
    g.condition_expr2               condition_expression2,
    --
    decode(substr(g.authorization_scheme,1,1),'!','Not ')||
    nvl((select name
     from   wwv_flow_security_schemes
     where  to_char(id) = ltrim(g.authorization_scheme,'!')
     and    flow_id = f.application_id),
     g.authorization_scheme)        authorization_scheme,
    g.authorization_scheme          authorization_scheme_id,
    (select case when g.build_option_id > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
     from   wwv_flow_patches pa
     where  pa.security_group_id = g.security_group_id
       and  pa.id =abs(g.build_option_id))   build_option,
    g.build_option_id                build_option_id,
    --
    g.LAST_UPDATED_BY               last_updated_by,
    g.LAST_UPDATED_ON               last_updated_on,
    --
    g.id                            item_group_id,
    g.group_comment                 component_comment
from wwv_flow_step_item_groups g,
     wwv_flow_steps            p,
     wwv_flow_authorized       f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and f.workspace_id      = g.security_group_id
  and f.application_id    = g.flow_id
  and p.security_group_id = g.security_group_id
  and p.flow_id           = g.flow_id
  and p.id                = g.flow_step_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_page_filter_groups',
    p_table_comment     => 'Identifies filter groups on a page.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_id'                    , 'Identifies the application',
      'page_name'                  , 'Identifies a page within an application',
      'region'                     , 'Identifies the region in which the item group will be displayed',
      'region_id'                  , 'Identifies the foreign key to the apex_application_page_regions view',
--
      'label'                      , 'Identifies the group label',
      'display_sequence'           , 'Identifies the sequence the group will be displayed within a region',
      'icon_css_classes'           , 'Identifies CSS classes to display an icon for the filter group',
      'css_classes'                , 'Identifies additional CSS classes to be applied to the group',
      'max_height'                 , 'Identifies the height of the facet',
      'show_label'                 , 'Identifies whether the filter should show the label',
--
      'suggestions_type'           , 'Identifies how suggestions are derived.',
      'suggestions_type_code'      , 'Identifies the internal code of SUGGESTIONS_TYPE.',
      'suggestions_language'       , 'Identifies the programming language of SUGGESTION_SOURCE.',
      'suggestions_source'         , 'Identifies static suggestions values or a SQL query.',
      'max_suggestions'            , 'Show only up to this amount of suggestions',
--
      'group_filter_combination'   , 'Identifies how the selected checkboxes within this group are combined.',
      'compute_counts'             , 'Identifies whether the filter within the group will compute counts.',
      'show_counts'                , 'Identifies whether the filter within the group will show counts.',
      'zero_count_entries'         , 'Identifies how facet values within the group with a count of zero should get displayed.',
      'sort_by_top_counts'         , 'Identifies whether the values with the highest count are displayed first.',
      'show_selected_first'        , 'Identifies whether the selected values are displayed first in the list of available values.',
      'max_displayed_entries'      , 'Show only up to this amount of values.',
--
      'is_collapsible'             , 'Identifies whether the filter is collapsible',
      'is_initial_collapsed'       , 'Identifies whether the filter is initially collapsed',
      'show_chart'                 , 'Identifies whether the filter values can be shown as a chart.',
      'initial_chart'              , 'Identifies whether the chart displaying filter values is initially shown.',
      'is_toggleable'              , 'Identifies whether end-users can toggle the visibility of the filter group',
      'is_initial_toggled'         , 'Identifies whether the filter group is displayed by default',

      'condition_type'             , 'Identifies a condition that must be met in order for this item to be displayed',
      'condition_type_code'        , 'Identifies the internal code CONDITION_TYPE',
      'condition_expression1'      , 'Identifies the item display condition',
      'condition_expression2'      , 'Identifies the item display condition',
      'authorization_scheme'       , 'Identifies the authorization scheme which must evaluate to TRUE in order for this component to be rendered',
--
      'authorization_scheme_id'    , 'Foreign Key',
--
      'build_option'               , 'Item will be displayed if the Build Option is enabled',
      'build_option_id'            , 'Build Option internal ID',
      'component_comment'          , 'Identifies a developer comment for this item',
--
      'item_group_id'              , 'Primary key of this component',

      'last_updated_by'            , 'APEX developer who made last update',
      'last_updated_on'            , 'Date of last update'
    )
  );
end;
/

prompt ...apex_application_page_items


create or replace view apex_application_page_items
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                            page_id,
    p.name                          page_name,
    --
    i.name                          item_name,
    case substr(i.display_as, 1, 7)
      when 'NATIVE_' then
          ( select np.display_name from wwv_flow_plugins np where np.security_group_id = 10 and np.flow_id = 4411 and np.plugin_type = 'ITEM TYPE' and np.name = substr(i.display_as, 8) )
      when 'PLUGIN_' then
          ( select fp.display_name from wwv_flow_plugins fp where fp.security_group_id = p.security_group_id and fp.flow_id = p.flow_id and fp.plugin_type = 'ITEM TYPE' and fp.name = substr(i.display_as, 8) )
      else
          nvl((select display_name from wwv_flow_standard_item_types where name = i.display_as), i.display_as)
    end                             display_as,
    i.display_as                    display_as_code,
    decode(i.DATA_TYPE,
       'NUMBER','Numbers Only',
       'VARCHAR','Varchar',
       i.DATA_TYPE)                 item_data_type,
    case
      when i.is_required = 'Y' then 'Yes'
      else 'No'
    end                             is_required,
    case
      when i.is_query_only = 'Y' then 'Yes'
      else 'No'
    end                             is_query_only,
    case
      when i.is_primary_key = 'Y' then 'Yes'
      else 'No'
    end                             is_primary_key,
    nvl(i.standard_validations, 'Application Level Setting') standard_validations,
    --
    i.ITEM_SEQUENCE                 display_sequence,
    (select pp.plug_name
     from wwv_flow_page_plugs pp
     where pp.security_group_id = i.security_group_id
       and pp.id                = i.ITEM_PLUG_ID)  region,
    i.item_plug_id                  region_id,
    --
    case
      when i.item_plug_id is not null then
          coalesce( (
              select dp.name
                from wwv_flow_plug_tmpl_disp_points dp
               where dp.plug_template_id  = r.plug_template
                 and dp.security_group_id = i.security_group_id
                 and dp.flow_id           = i.flow_id
                 and dp.placeholder       = i.item_display_point
        ), i.item_display_point )
      else
          coalesce( (
              select dp.name
                from wwv_flow_page_tmpl_disp_points dp
               where dp.page_template_id  = coalesce( p.step_template, ( select t.default_page_template
                                                                           from wwv_flow_user_interfaces ui,
                                                                                wwv_flow_themes t
                                                                          where ui.security_group_id = i.security_group_id
                                                                            and ui.flow_id           = i.flow_id
                                                                            and ui.ui_type_id        = 1 -- desktop
                                                                            and t.security_group_id  = ui.security_group_id
                                                                            and t.flow_id            = ui.flow_id
                                                                            and t.theme_id           = ui.theme_id ) )
                 and dp.flow_id           = i.flow_id
                 and dp.security_group_id = i.security_group_id
                 and dp.placeholder       = i.item_display_point
        ), decode( i.item_display_point,
                      'LEGACY_ORPHAN_COMPONENTS', 'Legacy Orphan Components',
                      i.item_display_point ) )
    end                                 display_position,
    i.item_display_point                display_position_code,
    --
    (select pp.plug_name
     from wwv_flow_page_plugs pp
     where pp.security_group_id = i.security_group_id
       and pp.id                = i.ITEM_SOURCE_PLUG_ID)     data_source_region,
    i.item_source_plug_id                  data_source_region_id,
    --
    case when i.item_source_plug_id is null then
        decode(
          i.USE_CACHE_BEFORE_DEFAULT,
      'NO','Always, replacing any existing value in session state',
      'YES','Only when current value in session state is null',
          i.USE_CACHE_BEFORE_DEFAULT)
         else null end   source_used,
    i.ITEM_DEFAULT                  item_default,
    i.ITEM_DEFAULT_TYPE             item_default_type,
    i.item_default_language         item_default_language,
    --
    i.PROMPT                        label,
    i.placeholder                   placeholder,
    i.PRE_ELEMENT_TEXT              pre_element_text,
    i.POST_ELEMENT_TEXT             post_element_text,
    i.FORMAT_MASK                   format_mask,
    (select ft.template_name
     from   wwv_flow_field_templates ft
     where  ft.security_group_id = i.security_group_id
     and    ft.id                = to_char(i.ITEM_FIELD_TEMPLATE)
     and    ft.flow_id           = f.application_id)         item_label_template,
    i.ITEM_FIELD_TEMPLATE           item_label_template_id,
    i.item_css_classes              item_css_classes,
    i.item_icon_css_classes         item_icon_css_classes,
    i.item_template_options         item_template_options,
    i.SOURCE                        item_source,
    case when i.ITEM_SOURCE_PLUG_ID is not null
        then 'Region Source Column'
        else
            decode(i.SOURCE_TYPE,
              'ALWAYS_NULL','Always Null',
              'STATIC','Static Assignment (value equals source attribute)',
              'QUERY','SQL Query (return single value)',
              'QUERY_COLON','SQL Query (return colon separated value)',
              'ITEM','Item (application or page item name)',
              'EXPRESSION','Expression',
              'FUNCTION_BODY','Function Body',
              'DB_COLUMN','Database Column',
              'PREFERENCE','Preference',
              'FACET_COLUMN','Facet Column',
              i.SOURCE_TYPE)
        end                         item_source_type,
    i.source_data_type              item_source_data_type,
    i.source_language               item_source_language,
    decode(i.encrypt_session_state_yn,'Y','Yes','No') encrypt_session_state,
    i.SOURCE_POST_COMPUTATION       source_post_computation,
    -- read only conditionality
    nvl((select r from apex_standard_conditions where d = i.READ_ONLY_WHEN_TYPE),i.READ_ONLY_WHEN_TYPE)
                                    read_only_condition_type,
    i.read_only_when_type           read_only_condition_type_code,
    i.READ_ONLY_WHEN                read_only_condition_exp1,
    i.READ_ONLY_WHEN2               read_only_condition_exp2,
    i.READ_ONLY_DISP_ATTR           read_only_display_attr,
    --
    i.NAMED_LOV                     lov_named_lov,
    i.lov_language                  lov_language,
    i.LOV                           lov_definition,
    decode(i.LOV_DISPLAY_EXTRA,
      'NO','No',
      'YES','Yes',
      i.LOV_DISPLAY_EXTRA)          lov_display_extra,
    decode(i.LOV_DISPLAY_NULL,
      'NO','No',
      'YES','Yes',
      i.LOV_DISPLAY_NULL)           lov_display_null,
    i.LOV_NULL_TEXT                 lov_null_text,
    i.LOV_NULL_VALUE                lov_null_value,
    decode(i.LOV_TRANSLATED,
      'N','No','Y','Yes',
      i.LOV_TRANSLATED)             lov_query_result_translated,
    i.lov_cascade_parent_items,
    i.ajax_items_to_submit,
    case nvl(i.ajax_optimize_refresh, 'Y')
      when 'Y' then 'Yes'
      when 'N' then 'No'
      else i.ajax_optimize_refresh
    end                             ajax_optimize_refresh,
    --
    i.csize                         item_element_width,
    i.cmaxlength                    item_element_max_length,
    i.cheight                       item_element_height,
    --
    i.cattributes                   html_table_cell_attr_label,
    i.cattributes_element           html_table_cell_attr_element, /* for backwards compatibility, replaced by grid_column_attributes */
    i.tag_css_classes               html_form_element_css_classes,
    i.tag_attributes                html_form_element_attributes,
    i.tag_attributes2               form_element_option_attributes,
    --
    i.button_image                  item_button_image,
    i.button_image_attr             item_button_image_attributes,
    --
    case i.new_grid
      when 'Y' then 'Yes'
      else 'No'
    end                             new_grid,
    case i.begin_on_new_line
      when 'Y' then 'Yes'
      else 'No'
    end                             new_grid_row,
    i.grid_row_css_classes,
    case i.begin_on_new_field
      when 'Y' then 'Yes'
      else 'No'
    end                             new_grid_column,
    i.grid_column                   grid_column,
    i.colspan                       grid_column_span,
    i.grid_label_column_span,
    i.grid_column_css_classes,
    i.cattributes_element           grid_column_attributes,
    --
    decode(i.BEGIN_ON_NEW_LINE,
      'N','No','Y','Yes',
      i.BEGIN_ON_NEW_LINE)          begins_on_new_row,  /* for backward compatibility */
    decode(i.BEGIN_ON_NEW_FIELD,
      'N','No','Y','Yes',
      i.BEGIN_ON_NEW_FIELD)         begins_on_new_cell, /* for backward compatibility */
    i.COLSPAN                       column_span,        /* for backward compatibility */
    --
    i.ROWSPAN                       row_span,
    --
    decode(i.LABEL_ALIGNMENT,
      'ABOVE',         'Above',
      'BELOW',         'Below',
      'CENTER',        'Center',
      'CENTER-BOTTOM', 'Bottom center',
      'CENTER-CENTER', 'Center center',
      'CENTER-TOP',    'Top center',
      'LEFT',          'Left',
      'LEFT-BOTTOM',   'Bottom left',
      'LEFT-CENTER',   'Center left',
      'LEFT-TOP',      'Top left',
      'RIGHT',         'Right',
      'RIGHT-BOTTOM',  'Bottom right',
      'RIGHT-CENTER',  'Center right',
      'RIGHT-TOP',     'Top right',
      i.LABEL_ALIGNMENT)            label_alignment,
    decode(i.FIELD_ALIGNMENT,
      'CENTER',       'Center',
      'CENTER-BOTTOM','Center bottom',
      'CENTER-CENTER','Center center',
      'CENTER-TOP',   'Center top',
      'LEFT',         'Left',
      'LEFT-BOTTOM',  'Left bottom',
      'LEFT-CENTER',  'Left center',
      'LEFT-TOP',     'Left top',
      'RIGHT',        'Right',
      'RIGHT-BOTTOM', 'Right bottom',
      'RIGHT-CENTER', 'Right center',
      'RIGHT-TOP',    'Right top',
      i.FIELD_ALIGNMENT)            item_alignment,
    inline_help_text,
    --
    --i.FIELD_TEMPLATE,
    --
    decode(i.show_quick_picks,
      'NO','No','YES','Yes',
      i.show_quick_picks)           show_quick_picks,
    i.quick_pick_link_attr,
    i.quick_pick_label_01,
    i.quick_pick_value_01,
    i.quick_pick_label_02,
    i.quick_pick_value_02,
    i.quick_pick_label_03,
    i.quick_pick_value_03,
    i.quick_pick_label_04,
    i.quick_pick_value_04,
    i.quick_pick_label_05,
    i.quick_pick_value_05,
    i.quick_pick_label_06,
    i.quick_pick_value_06,
    i.quick_pick_label_07,
    i.quick_pick_value_07,
    i.quick_pick_label_08,
    i.quick_pick_value_08,
    i.quick_pick_label_09,
    i.quick_pick_value_09,
    i.quick_pick_label_10,
    i.quick_pick_value_10,
    --
    i.plugin_init_javascript_code   init_javascript_code,
    --
    i.attribute_01,
    i.attribute_02,
    i.attribute_03,
    i.attribute_04,
    i.attribute_05,
    i.attribute_06,
    i.attribute_07,
    i.attribute_08,
    i.attribute_09,
    i.attribute_10,
    i.attribute_11,
    i.attribute_12,
    i.attribute_13,
    i.attribute_14,
    i.attribute_15,
    --
    nvl((select r from apex_standard_conditions where d = i.DISPLAY_WHEN_TYPE),i.DISPLAY_WHEN_TYPE)
                                    condition_type,
    i.display_when_type             condition_type_code,
    i.DISPLAY_WHEN                  condition_expression1,
    i.DISPLAY_WHEN2                 condition_expression2,
    --
    case
      when i.warn_on_unsaved_changes = 'I' then 'Ignore'
      else 'Page Default'
    end                             warn_on_unsaved_changes,
    i.warn_on_unsaved_changes       warn_on_unsaved_changes_code,
    --
    case i.restricted_characters
        when 'US_ONLY' then 'Allowlist for a-Z, 0-9 and space'
        when 'WEB_SAFE' then 'Blocklist HTML command characters (<>")'
        when 'NO_SPECIAL_CHAR' then 'Blocklist &<>"/;,*|=% and --'
        when 'NO_SPECIAL_CHAR_NL' then 'Blocklist &<>"/;,*|=% or -- and new line'
        when 'WORKSPACE_SCHEMA' then 'Workspace Schema'
    end as restricted_characters,
    i.restricted_characters restricted_characters_code,
    --
    decode(i.IS_PERSISTENT,
       'Y','Per Session',
       'U','Per User',
       'N','Do Not Save Session State',
       i.IS_PERSISTENT)             maintain_session_state,
    decode(nvl(i.PROTECTION_LEVEL,'N'),
       'N','Unrestricted',
       'C','Arguments Must Have Checksum',
       'U','No Arguments Allowed',
       'D','No URL Access',
       'S','Checksum Required - Session Level',
       'I','Restricted - May not be set from browser',
       'B','Checksum Required - Application Level',
       'P','Checksum Required - User Level',
       i.PROTECTION_LEVEL)          item_protection_level,
    case i.escape_on_http_output
      when 'Y' then 'Yes'
      when 'N' then 'No'
      else i.escape_on_http_output
    end                             escape_on_http_output,
    --i.ESCAPE_ON_HTTP_INPUT
    --
    decode(substr(i.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
     from   wwv_flow_security_schemes ss
     where  ss.security_group_id = i.security_group_id
       and  to_char(ss.id)       = ltrim(i.SECURITY_SCHEME,'!')
       and  ss.flow_id           = f.application_id),
     i.SECURITY_SCHEME)             authorization_scheme,
    i.security_scheme               authorization_scheme_id,
    (select case when i.required_patch > 0 then fp.PATCH_NAME else '{Not '||fp.PATCH_NAME||'}' end PATCH_NAME
     from   wwv_flow_patches fp
     where fp.security_group_id = i.security_group_id
       and fp.id                = abs(i.REQUIRED_PATCH))   build_option,
    i.REQUIRED_PATCH                build_option_id,
    --
    (select ih.help_text
    from  wwv_flow_step_item_help ih
    where ih.security_group_id = i.security_group_id
      and ih.flow_id           = i.flow_id
      and ih.FLOW_ITEM_ID      = i.id) item_help_text,
    --
    i.LAST_UPDATED_BY               last_updated_by,
    i.LAST_UPDATED_ON               last_updated_on,
    i.ITEM_COMMENT                  component_comment,
    i.id                            item_id,
    --
    i.name
    ||' da='||i.display_as
    ||decode(i.DATA_TYPE,'NUMBER','NOnly','VARCHAR','V',i.DATA_TYPE)
    ||',seq='||lpad(i.ITEM_SEQUENCE,5,'00000')
    ||',r='||(select plug_name from wwv_flow_page_plugs where id=i.ITEM_PLUG_ID)
    ||',c='||decode(i.USE_CACHE_BEFORE_DEFAULT,'NO','Always','YES','Only null',i.USE_CACHE_BEFORE_DEFAULT)
    ||',d='||substr(i.ITEM_DEFAULT,1,20)||length(i.ITEM_DEFAULT)||i.ITEM_DEFAULT_TYPE
    ||',l='||substr(i.PROMPT,1,20)||length(i.prompt)
    ||substr(i.PRE_ELEMENT_TEXT,1,10)||length(i.PRE_ELEMENT_TEXT)
    ||substr(i.POST_ELEMENT_TEXT,1,10)||length(i.POST_ELEMENT_TEXT)
    ||',m='||i.FORMAT_MASK
    ||'t='||(select template_name from   wwv_flow_field_templates where  id = to_char(i.ITEM_FIELD_TEMPLATE) and flow_id = f.application_id)
    ||'s='||substr(i.SOURCE,1,30)||length(i.source)
    ||decode(i.SOURCE_TYPE,
      'ALWAYS_NULL','Always Null',
      'STATIC','StaticAs',
      'QUERY','SQLQuery',
      'QUERY_COLON','SQLQueryColon',
      'ITEM','Item ',
      'EXPRESSION','Expression',
      'FUNCTION_BODY','FunctionBody',
      'DB_COLUMN','DBColumn',
      'PREFERENCE','Pref',
      i.SOURCE_TYPE)
    ||length(i.SOURCE_POST_COMPUTATION)
    ||' ro='||nvl((select r from apex_standard_conditions where d = i.READ_ONLY_WHEN_TYPE),i.READ_ONLY_WHEN_TYPE)
    ||length(i.READ_ONLY_WHEN)
    ||length(i.READ_ONLY_WHEN2)
    ||length(i.READ_ONLY_DISP_ATTR)
    ||',lov='||i.NAMED_LOV
    ||decode(i.named_lov,null,null,length(i.LOV))
    ||decode(i.LOV_DISPLAY_EXTRA,
      'NO','No',
      'YES','Yes',
      i.LOV_DISPLAY_EXTRA)
    ||decode(i.LOV_DISPLAY_NULL,
      'NO','No',
      'YES','Yes',
      i.LOV_DISPLAY_NULL)
    ||substr(LOV_NULL_TEXT,1,20)||length(LOV_NULL_TEXT)
    ||i.LOV_NULL_VALUE
    ||',t='||decode(i.LOV_TRANSLATED,'N','No','Y','Yes',i.LOV_TRANSLATED)
    ||',s='||i.csize||i.cmaxlength||i.cheight
    ||',a='||substr(i.cattributes,1,10)||length(i.cattributes)
    ||length(i.cattributes_element)
    ||length(i.tag_attributes)
    ||length(i.tag_attributes2)
    ||',b='||length(i.button_image)||length(i.button_image_attr)
    ||',disp='||decode(i.BEGIN_ON_NEW_LINE,
      'N','No','Y','Yes',
      i.BEGIN_ON_NEW_LINE)||decode(i.BEGIN_ON_NEW_FIELD,
      'N','No','Y','Yes',
      i.BEGIN_ON_NEW_FIELD)||i.COLSPAN||i.ROWSPAN
    ||' l='||decode(i.LABEL_ALIGNMENT,
      'ABOVE',         'Above',
      'BELOW',         'Below',
      'CENTER',        'Center',
      'CENTER-BOTTOM', 'Bottom center',
      'CENTER-CENTER', 'Center center',
      'CENTER-TOP',    'Top center',
      'LEFT',          'Left',
      'LEFT-BOTTOM',   'Bottom left',
      'LEFT-CENTER',   'Center left',
      'LEFT-TOP',      'Top left',
      'RIGHT',         'Right',
      'RIGHT-BOTTOM',  'Bottom right',
      'RIGHT-CENTER',  'Center right',
      'RIGHT-TOP',     'Top right',
      i.LABEL_ALIGNMENT)||decode(i.FIELD_ALIGNMENT,
      'CENTER',       'Center',
      'CENTER-BOTTOM','Center bottom',
      'CENTER-CENTER','Center center',
      'CENTER-TOP',   'Center top',
      'LEFT',         'Left',
      'LEFT-BOTTOM',  'Left bottom',
      'LEFT-CENTER',  'Left center',
      'LEFT-TOP',     'Left top',
      'RIGHT',        'Right',
      'RIGHT-BOTTOM', 'Right bottom',
      'RIGHT-CENTER', 'Right center',
      'RIGHT-TOP',    'Right top',
      i.FIELD_ALIGNMENT)
    ||',c='||i.DISPLAY_WHEN_TYPE||substr(i.DISPLAY_WHEN,1,20)||length(i.DISPLAY_WHEN)||length(i.DISPLAY_WHEN2)
    ||decode(i.IS_PERSISTENT,
       'Y','PerSes',
       'U','PerU',
       'N','Do Not Save SesSt',
       i.IS_PERSISTENT)||
    decode(nvl(i.PROTECTION_LEVEL,'N'),
       'N','Unrest',
       'C','Arg Must Have Cs',
       'U','No Arguments Allowed',
       'D','No URL Access',
       'S','Cs Required - Session Level',
       'I','Restricted - May not be set from browser',
       'B','Cs Req-AL',
       'P','Cs Req-UL',
       i.PROTECTION_LEVEL)
    ||decode(substr(i.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select name
     from   wwv_flow_security_schemes
     where  to_char(id) = ltrim(i.SECURITY_SCHEME,'!')
     and    flow_id = f.application_id),
     'Not found')
     ||(select PATCH_NAME
     from   wwv_flow_patches
     where  id =abs(i.REQUIRED_PATCH))
    component_signature
from wwv_flow_step_items i,
     wwv_flow_steps      p,
     wwv_flow_page_plugs r,
     wwv_flow_authorized f
where f.workspace_id          = p.security_group_id
  and f.application_id        = p.flow_id
  and f.workspace_id          = i.security_group_id
  and f.application_id        = i.flow_id
  and p.security_group_id     = i.security_group_id
  and p.flow_id               = i.flow_id
  and p.id                    = i.flow_step_id
  and r.security_group_id (+) = i.security_group_id
  and r.flow_id (+)           = i.flow_id
  and r.id (+)                = i.item_plug_id
  and ( i.source_type    <> 'FACET_COLUMN' or i.source_type is null )
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_items',
    p_table_comment     => 'Identifies Page Items which are used to render HTML form content.  Items automatically maintain session state which can be accessed using bind variables or substitution stings.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'page_id'                       , 'Identifies the application',
      'page_name'                     , 'Identifies a page within an application',
      'item_name'                     , 'Identifies a page item and is used to maintain session state.  Value may be referenced as a SQL bind variable or using APEX substitution string syntax.',
      'display_as'                    , 'Identifies how the item will be displayed in the HTML page',
      'display_as_code'               , 'Internal code of DISPLAY_AS',
      'item_data_type'                , 'Typically VARCHAR but may be set to NUMBER to restrict values to only numbers',
      'is_required'                   , 'If set to Yes, APEX will automatically perform a not null validation during page submit.',
      'is_primary_key'                , 'Determines whether this item is used as the Primary Key in DML operations.',
      'is_query_only'                 , 'Determines whether this is a "Query Only" item, i.e. changes are not written for DML operations.',
      'standard_validations'          , 'Identifies where APEX will perform the built-in standard validations for the item type.',
      'DISPLAY_SEQUENCE'              , 'Identifies the sequence the page item will be displayed within a region',
      'REGION'                        , 'Identifies the region in which the item will be displayed',
      'REGION_ID'                     , 'Identifies the foreign key to the apex_application_page_regions view',
      'display_position'              , 'Identifies the position where the item will be displayed',
      'display_position_code'         , 'Identifies the internal code of DISPLAY_POSITION',
      'DATA_SOURCE_REGION'            , 'Identifies the data source region this item is assigned to.',
      'DATA_SOURCE_REGION_ID'         , 'Identifies the foreign key to the apex_application_page_regions view',
      'SOURCE_USED'                   , 'Identifies how the source of the item value is determined.  The item can be set by the source value on each view only when the session state for the item is null.',
      'ITEM_DEFAULT'                  , 'When the item has no source or session state, use this default value',
      'ITEM_DEFAULT_TYPE'             , 'Identifies how the Item Default is set, based on a dynamic computation or a static assignment',
      'item_default_language'         , 'Identifies the programming language of ITEM_DEFAULT.',
      'LABEL'                         , 'Identifies the item label',
      'placeholder'                   , 'Identifies the placeholder used for the element of the item',
      'PRE_ELEMENT_TEXT'              , 'Identifies text placed before the item.',
      'POST_ELEMENT_TEXT'             , 'Identifies text placed after the item.',
      'FORMAT_MASK'                   , 'Identifies a format mask which can be used to format numeric or date values',
      'ITEM_LABEL_TEMPLATE'           , 'Identifies the template used to display the item label',
      'ITEM_LABEL_TEMPLATE_ID'        , 'Identifies the template ID foreign key',
      'item_css_classes'              , 'Identifies additional CSS classes to be applied to the item',
      'item_icon_css_classes'         , 'Identifies CSS classes to display an icon for the item',
      'item_template_options'         , 'Identifies template options to be applied to the item template',
      'ITEM_SOURCE'                   , 'Identifies the items value.  Reference the Source Used and Item Source Type attributes.',
      'ITEM_SOURCE_TYPE'              , 'Identifies how the item source is determined, for example from a Database Column, Static assignment, or a Query or PL/SQL expression.',
      'item_source_data_type'         , 'Identifies the column data type if the item source type is "Region Source Column"',
      'item_source_language'          , 'Identifies the programming language of ITEM_SOURCE.',
      'SOURCE_POST_COMPUTATION'       , 'After determining the Item Source value apply this computation',
      'READ_ONLY_CONDITION_TYPE'      , 'Identifies the condition type when the item will be rendered read only',
      'READ_ONLY_CONDITION_TYPE_CODE' , 'Identifies the internal code of READ_ONLY_CONDITION_TYPE',
      'READ_ONLY_CONDITION_EXP1'      , 'Identifies the condition that determines if the item will be rendered read only',
      'READ_ONLY_CONDITION_EXP2'      , 'Identifies the condition that determines if the item will be rendered read only',
      'READ_ONLY_DISPLAY_ATTR'        , 'Identifies the HTML SPAN tag attributes when rendering the item read only',
      'LOV_NAMED_LOV'                 , 'Identifies the List of Values to be used to render this domain of values for this item',
      'lov_language'                  , 'Identifies the programming language of LOV_DEFINITION.',
      'LOV_DEFINITION'                , 'Identifies the List of Values used as the domain of values for this item',
      'LOV_DISPLAY_EXTRA'             , 'When an items values is not included in the domain of the List of Values, display the value of the item',
      'LOV_DISPLAY_NULL'              , 'Yes or No, identifies if a null value option is to be provided',
      'LOV_NULL_TEXT'                 , 'Identifies the text to be displayed to the end user for a null value',
      'LOV_NULL_VALUE'                , 'Identifies the value to be used to identify a null value for the item',
      'LOV_QUERY_RESULT_TRANSLATED'   , 'Yes or No identifies if List of Values elements will be checked for dynamic translation',
      'lov_cascade_parent_items'      , 'Identifies the cascading LOV parent item(s) for the item.',
      'ajax_items_to_submit'          , 'Identifies the page items which should be submitted when an Ajax call is performed for the page item.',
      'ajax_optimize_refresh'         , 'Yes or No, identifies if the Ajax call should be optimized.',

      'ITEM_ELEMENT_WIDTH'      , 'Identifies the width of the item element',
      'ITEM_ELEMENT_MAX_LENGTH' , 'Identifies the maximum length of the item element',
      'ITEM_ELEMENT_HEIGHT'     , 'Identifies the height of the item element for item display types that use a height',

      'HTML_TABLE_CELL_ATTR_LABEL'     , 'Identifies HTML table cell label attributes',
      'HTML_TABLE_CELL_ATTR_ELEMENT'   , 'Identifies HTML table cell element attributes',
      'HTML_FORM_ELEMENT_ATTRIBUTES'   , 'Identifies HTML table form element attributes',
      'FORM_ELEMENT_OPTION_ATTRIBUTES' , 'Identifies HTML table form element option attributes',

      'ITEM_BUTTON_IMAGE'              , 'For buttons displayed among items, the name of a button image',
      'ITEM_BUTTON_IMAGE_ATTRIBUTES'   , 'For buttons displayed among items, the name of a button HTML IMG tag attributes',

      'new_grid'                      , 'Identifies if the rendering of the item begins on a new grid',
      'new_grid_row'                  , 'Identifies if the rendering of the item begins on a new grid row',
      'grid_row_css_classes'          , 'Additional CSS classes which are substituted if the grid row template contains the substitution string #CSS_CLASSES#',
      'new_grid_column'               , 'Identifies if the rendering of the item begins on a new grid column',
      'grid_column'                   , 'Identifies the grid column used to display the item',
      'grid_column_span'              , 'Identifies over how many grid columns the item spans',
      'ROW_SPAN'                      , 'Identifies the HTML table TD tag ROWSPAN value',
      'grid_label_column_span'        , 'Identifies over how many grid columns the label spans',
      'grid_column_css_classes'       , 'Additional CSS classes which are substituted if the grid column template contains the substitution string #CSS_CLASSES#',
      'grid_column_attributes'        , 'Additional attributes which are substituted if the grid column template contains the substitution string #ATTRIBUTES#',

      'LABEL_ALIGNMENT' , 'Identifies the alignment of the Item Label',
      'ITEM_ALIGNMENT'  , 'Identifies the alignment of the Item Form Element',

      'inline_help_text'      , 'Help text to be displayed inline with the page item.',
      'show_quick_picks'      , 'If set to Yes, the entered quick picks will be displayed below the page item.',
      'quick_pick_link_attr'  , 'Link attribute which is added to each quick pick link.',
      'quick_pick_label_01'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_01'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_02'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_02'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_03'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_03'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_04'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_04'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_05'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_05'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_06'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_06'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_07'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_07'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_08'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_08'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_09'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_09'   , 'The value defines the value which is used when the quick pick is clicked.',
      'quick_pick_label_10'   , 'The label is the the text the end user of the application will see for the quick pick.',
      'quick_pick_value_10'   , 'The value defines the value which is used when the quick pick is clicked.',

      'init_javascript_code'         , 'JavaScript code for item initialization',

      'attribute_01'                 , 'Dynamic attribute to store additional data',
      'attribute_02'                 , 'Dynamic attribute to store additional data',
      'attribute_03'                 , 'Dynamic attribute to store additional data',
      'attribute_04'                 , 'Dynamic attribute to store additional data',
      'attribute_05'                 , 'Dynamic attribute to store additional data',
      'attribute_06'                 , 'Dynamic attribute to store additional data',
      'attribute_07'                 , 'Dynamic attribute to store additional data',
      'attribute_08'                 , 'Dynamic attribute to store additional data',
      'attribute_09'                 , 'Dynamic attribute to store additional data',
      'attribute_10'                 , 'Dynamic attribute to store additional data',
      'attribute_11'                 , 'Dynamic attribute to store additional data',
      'attribute_12'                 , 'Dynamic attribute to store additional data',
      'attribute_13'                 , 'Dynamic attribute to store additional data',
      'attribute_14'                 , 'Dynamic attribute to store additional data',
      'attribute_15'                 , 'Dynamic attribute to store additional data',

      'CONDITION_TYPE'               , 'Identifies a condition that must be met in order for this item to be displayed',
      'CONDITION_TYPE_CODE'          , 'Identifies the internal code CONDITION_TYPE',
      'CONDITION_EXPRESSION1'        , 'Identifies the item display condition',
      'CONDITION_EXPRESSION2'        , 'Identifies the item display condition',

      'warn_on_unsaved_changes'      , 'Determines if a warning should be displayed in case of unsaved changes.',
      'warn_on_unsaved_changes_code' , 'Identifies the internal code WARN_ON_UNSAVED_CHANGES',
      'restricted_characters'        , 'Identifies how to restrict the characters that can be saved in session state',
      'restricted_characters_code'   , 'Identifies the internal code of RESTRICTED_CHARACTERS',
      'MAINTAIN_SESSION_STATE'       , 'Identifies the method used to maintain session state, per session, per user, or not maintained',
      'ITEM_PROTECTION_LEVEL'        , 'If Session State Protection is enabled, identifies if a checksum will be required to change an items session state via the URL or POSTDATA.',
      'AUTHORIZATION_SCHEME'         , 'Identifies the authorization scheme which must evaluate to TRUE in order for this component to be rendered',
      'AUTHORIZATION_SCHEME_ID'      , 'Foreign Key',
      'encrypt_session_state'        , 'If Yes values of session state written to APEX session state tables is encrypted.  Decryption is automatic.',
      'escape_on_http_output'        , 'If Yes values will be escaped when they are written to the HTTP output.',
      'BUILD_OPTION'                 , 'Item will be displayed if the Build Option is enabled',
      'COMPONENT_COMMENT'            , 'Identifies a developer comment for this item',
      'ITEM_ID'                      , 'Primary key of this component',
      'last_updated_by'              , 'APEX developer who made last update',
      'last_updated_on'              , 'Date of last update',
      'component_signature'          , 'Identifies attributes defined at a given component level to facilitate application comparisons',

      'HTML_FORM_ELEMENT_CSS_CLASSES', 'CSS classes to apply to the HTML element of this page item',
      'BEGINS_ON_NEW_ROW'            , 'For backward compatibility. Use new_grid_row instead.',
      'BEGINS_ON_NEW_CELL'           , 'For backward compatibility. Use new_grid_column instead.',
      'COLUMN_SPAN'                  , 'For backward compatibility. Use grid_column_span instead.',
      'BUILD_OPTION_ID'              , 'Build Option internal ID',
      'ITEM_HELP_TEXT'               , 'Help text for the page item.'
    )
  );
end;
/

create or replace view apex_application_page_db_items
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                            page_id,
    p.name                          page_name,
    (select pp.plug_name
     from wwv_flow_page_plugs pp
     where pp.id                = i.ITEM_PLUG_ID
       and pp.security_group_id = i.security_group_id) region,
    --
    i.name                          item_name,
    i.display_as                    display_as,
    i.ITEM_SEQUENCE                 display_sequence,
    --
    i.PROMPT                        item_label,
    i.SOURCE                        db_column_name,
    ( select attribute_02
        from wwv_flow_step_processing proc
       where proc.security_group_id = p.security_group_id
         and proc.flow_id           = p.flow_id
         and proc.flow_step_id      = p.id
         and proc.process_type      = 'NATIVE_FORM_FETCH' )
                                    db_table_name,
    h.help_text                     help_text,
    i.inline_help_text              inline_help_text,
    i.id                            item_id
from wwv_flow_step_items     i,
     wwv_flow_step_item_help h,
     wwv_flow_steps          p,
     wwv_flow_authorized     f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and f.workspace_id      = i.security_group_id
  and f.application_id    = i.flow_id
  and p.flow_id           = i.flow_id
  and p.id                = i.flow_step_id
  and i.id                = h.flow_item_id (+)
  and i.security_group_id = h.security_group_id (+)
  and i.source_type       = 'DB_COLUMN'
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_db_items',
    p_table_comment     => 'Identifies Page Items which are associated with Database Table Columns.  This view represents a subset of the items in the APEX_APPLICATION_PAGE_ITEMS view.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'         , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'    , 'Application Primary Key, Unique over all workspaces',
      'application_name'  , 'Identifies the application',
      'page_id'           , 'Identifies the application',
      'page_name'         , 'Identifies a page within an application',
      'item_name'         , 'APEX page item name',
      'DISPLAY_AS'        , 'Identifies how the item will be displayed in the HTML page',
      'DISPLAY_SEQUENCE'  , 'Identifies the sequence the page item will be displayed within a region',
      'REGION'            , 'Identifies the region in which the item will be displayed',
      'ITEM_LABEL'        , 'Identifies the item label',
      'DB_COLUMN_NAME'    , 'Identifies the item source database column name',
      'DB_TABLE_NAME'     , 'Identifies the item source database table name',
      'INLINE_HELP_TEXT'  , 'Identifies the help text to be displayed inline with the page item',
      'HELP_TEXT'         , 'Identifies the help text for the page item',
      'ITEM_ID'           , 'Primary key of this component'
    )
  );
end;
/
create or replace view apex_appl_page_filters
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                            page_id,
    p.name                          page_name,
    --
    (select pp.plug_name
     from wwv_flow_page_plugs pp
     where pp.id                = i.ITEM_PLUG_ID
       and pp.security_group_id = i.security_group_id) region,
    i.item_plug_id                  region_id,
    --
    (select ig.label
     from wwv_flow_step_item_groups ig
     where ig.id                = i.item_group_id
       and ig.security_group_id = i.security_group_id) filter_group,
    i.item_group_id                 filter_group_id,
    --
    i.name                          item_name,
    case substr(i.display_as, 1, 7)
      when 'NATIVE_' then
          ( select np.display_name from wwv_flow_plugins np where np.security_group_id = 10 and np.flow_id = 4411 and np.plugin_type = 'ITEM TYPE' and np.name = substr(i.display_as, 8) )
      when 'PLUGIN_' then
          ( select fp.display_name from wwv_flow_plugins fp where fp.security_group_id = i.security_group_id and fp.flow_id = p.flow_id and fp.plugin_type = 'ITEM TYPE' and fp.name = substr(i.display_as, 8) )
    end                             item_type,
    i.display_as                    item_type_code,
    --
    i.PROMPT                        label,
    --
    i.attribute_01,
    i.attribute_02,
    i.attribute_03,
    i.attribute_04,
    i.attribute_05,
    i.attribute_06,
    i.attribute_07,
    i.attribute_08,
    i.attribute_09,
    i.attribute_10,
    i.attribute_11,
    i.attribute_12,
    i.attribute_13,
    i.attribute_14,
    i.attribute_15,
    --
    i.ITEM_SEQUENCE                 display_sequence,
    i.fc_max_height                 max_height,
    case i.fc_show_label
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                             show_label,
    --
    i.NAMED_LOV                     lov_named_lov,
    i.LOV                           lov_definition,
    decode(i.LOV_DISPLAY_NULL,
      'NO','No',
      'YES','Yes',
      i.LOV_DISPLAY_NULL)           lov_display_null,
    i.LOV_NULL_TEXT                 lov_null_text,
    lov_cascade_parent_items        lov_cascade_parent_filter,
    lov_cascade_parent_column,
    case i.ajax_optimize_refresh
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                             lov_cascade_parent_required,
    --
    case i.suggestions_type
      when 'DYNAMIC'                   then 'Dynamic'
      when 'STATIC'                    then 'Static Values'
      when 'SQL_QUERY'                 then 'SQL Query'
      when 'COLUMN'                    then 'Column'
      when 'FUNC_BODY_RETURNING_ARRAY' then 'Function Body Returning Array'
      else i.suggestions_type
    end                             suggestions_type,
    i.suggestions_type              suggestions_type_code,
    i.suggestions_language,
    i.suggestions_source,
    i.max_suggestions,
    --
    decode(i.fc_compute_counts,
       'Y','Yes',
       'N','No',
       i.fc_compute_counts)         compute_counts,
    decode(i.fc_show_counts,
       'Y','Yes',
       'N','No',
       i.fc_show_counts)            show_counts,
    case i.fc_zero_count_entries
      when 'H' then 'Hide'
      when 'D' then 'Disable'
      when 'L' then 'Show Last'
    end                             zero_count_entries,
    decode(i.fc_sort_by_top_counts,
       'Y','Yes',
       'N','No',
       i.fc_sort_by_top_counts)     sort_by_top_counts,
    decode(i.fc_show_selected_first,
       'Y','Yes',
       'N','No',
       i.fc_show_selected_first)    show_selected_first,
    fc_show_more_count              max_displayed_entries,
    decode(i.fc_filter_values,
       'Y','Yes',
       'N','No',
       i.fc_filter_values)          has_client_side_filtering,
    decode(i.fc_show_chart,
       'Y','Yes',
       'N','No',
       i.fc_show_chart)             show_chart,
    decode(i.fc_initial_chart,
       'Y','Yes',
       'N','No',
       i.fc_initial_chart)          initial_chart,
    decode(i.fc_actions_filter,
       'Y','Yes',
       'N','No',
       i.fc_actions_filter)         actions_filter,
    decode(i.fc_toggleable,
       'Y','Yes',
       'N','No',
       i.fc_toggleable)             is_toggleable,
    decode(i.fc_initial_toggled,
       'Y','Yes',
       'N','No',
       i.fc_initial_toggled)        is_initial_toggled,
    --
    i.placeholder                   placeholder,
    i.PRE_ELEMENT_TEXT              pre_element_text,
    i.POST_ELEMENT_TEXT             post_element_text,
    --
    decode(i.fc_collapsible,
       'Y','Yes',
       'N','No',
       i.fc_collapsible)            is_collapsible,
    decode(i.fc_initial_collapsed,
       'Y','Yes',
       'N','No',
       i.fc_initial_collapsed)      is_initial_collapsed,
    i.item_icon_css_classes         icon_css_classes,
    i.item_css_classes              item_css_classes,
    --
    i.SOURCE                        item_source,
    i.source_data_type              item_source_data_type,
    --
    case i.multi_value_type
        when 'SEPARATED'  then 'Separated'
        when 'JSON_ARRAY' then 'JSON Array'
    end                             multi_value_type,
    i.multi_value_type              multi_value_type_code,
    i.multi_value_separator         multi_value_separator,
    case i.multi_value_trim_space
        when 'Y' then 'Yes'
        when 'N' then 'No'
        else i.multi_value_trim_space
    end                             multi_value_trim_space,
    --
    case i.fc_filter_combination
        when 'AND' then 'AND (Intersect)'
        when 'OR'  then 'OR (Union)'
        else i.fc_filter_combination
    end                             multi_value_filter_combination,
    --
    i.ITEM_DEFAULT_TYPE             item_default_type,
    i.ITEM_DEFAULT                  item_default,
    --
    i.depending_on_id,
    i.depending_on_condition_type,
    i.depending_on_expression,
    --
    nvl((select r from apex_standard_conditions where d = i.DISPLAY_WHEN_TYPE),i.DISPLAY_WHEN_TYPE)
                                    condition_type,
    i.display_when_type             condition_type_code,
    i.DISPLAY_WHEN                  condition_expression1,
    i.DISPLAY_WHEN2                 condition_expression2,
    --
    decode(substr(i.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
     from   wwv_flow_security_schemes ss
     where  to_char(ss.id) = ltrim(i.SECURITY_SCHEME,'!')
     and    ss.flow_id = f.application_id
     and    ss.security_group_id = f.workspace_id),
     i.SECURITY_SCHEME)             authorization_scheme,
    i.security_scheme               authorization_scheme_id,
    decode(i.encrypt_session_state_yn,'Y','Yes','No') encrypt_session_state,
    case i.escape_on_http_output
      when 'Y' then 'Yes'
      when 'N' then 'No'
      else i.escape_on_http_output
    end                             escape_on_http_output,
    --
    (select case when i.required_patch > 0 then fp.PATCH_NAME else '{Not '||fp.PATCH_NAME||'}' end PATCH_NAME
     from   wwv_flow_patches fp
     where  fp.id                = abs(i.REQUIRED_PATCH)
     and    fp.security_group_id = i.security_group_id) build_option,
    i.REQUIRED_PATCH                build_option_id,
    --
    i.LAST_UPDATED_BY               last_updated_by,
    i.LAST_UPDATED_ON               last_updated_on,
    i.ITEM_COMMENT                  component_comment,
    i.id                            item_id,
    --
    to_char(null)                   component_signature
from wwv_flow_step_items i,
     wwv_flow_steps      p,
     wwv_flow_authorized f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and f.workspace_id      = i.security_group_id
  and f.application_id    = i.flow_id
  and p.security_group_id = i.security_group_id
  and p.flow_id           = i.flow_id
  and p.id                = i.flow_step_id
  and i.source_type       = 'FACET_COLUMN'
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_page_filters',
    p_table_comment     => 'Identifies Filters which are used within a Faceted Search or Smart Filters region.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'         , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'    , 'Application Primary Key, Unique over all workspaces',
      'application_name'  , 'Identifies the application',
      'page_id'           , 'Identifies the application',
      'page_name'         , 'Identifies a page within an application',
      'REGION'            , 'Identifies the region in which the item will be displayed',
      'REGION_ID'         , 'Identifies the foreign key to the apex_application_page_regions view',
      'filter_group'      , 'Identifies the group this filter belongs to',
      'filter_group_id'   , 'ID of the group this filter belongs to',
      'item_name'         , 'Identifies a filter and is used to maintain session state.  Value may be referenced as a SQL bind variable or using APEX substitution string syntax.',
      'item_type'         , 'Identifies how the filter will be displayed in the HTML page',
      'item_type_code'    , 'Internal code of ITEM_TYPE',
--
      'LABEL'             , 'Identifies the filter label',
--
      'attribute_01'          , 'Dynamic attribute to store additional data',
      'attribute_02'          , 'Dynamic attribute to store additional data',
      'attribute_03'          , 'Dynamic attribute to store additional data',
      'attribute_04'          , 'Dynamic attribute to store additional data',
      'attribute_05'          , 'Dynamic attribute to store additional data',
      'attribute_06'          , 'Dynamic attribute to store additional data',
      'attribute_07'          , 'Dynamic attribute to store additional data',
      'attribute_08'          , 'Dynamic attribute to store additional data',
      'attribute_09'          , 'Dynamic attribute to store additional data',
      'attribute_10'          , 'Dynamic attribute to store additional data',
      'attribute_11'          , 'Dynamic attribute to store additional data',
      'attribute_12'          , 'Dynamic attribute to store additional data',
      'attribute_13'          , 'Dynamic attribute to store additional data',
      'attribute_14'          , 'Dynamic attribute to store additional data',
      'attribute_15'          , 'Dynamic attribute to store additional data',
--
      'DISPLAY_SEQUENCE'      , 'Identifies the sequence the filter will be displayed within a region',
      'max_height'            , 'Identifies the height of the filter',
      'show_label'            , 'Identifies whether the filter should show the label',
--
      'LOV_NAMED_LOV' , 'Identifies the List of Values to be used to render this domain of values for this filter',
      'LOV_DEFINITION' , 'Identifies the List of Values used as the domain of values for this filter',
      'LOV_DISPLAY_NULL' , 'Yes or No, identifies if a null value option is to be provided',
      'LOV_NULL_TEXT' , 'Identifies the text to be displayed to the end user for a null value',
      'lov_cascade_parent_filter'   , 'Identifies the cascading LOV parent filter.',
      'lov_cascade_parent_column'   , 'Identifies the cascading LOV parent column.',
      'lov_cascade_parent_required' , 'Yes or No, identifies if the parent filter has to contain a value to display the LOV.',
--
      'suggestions_type'           , 'Identifies how suggestions are derived.',
      'suggestions_type_code'      , 'Identifies the internal code of SUGGESTIONS_TYPE.',
      'suggestions_language'       , 'Identifies the programming language of SUGGESTION_SOURCE.',
      'suggestions_source'         , 'Identifies static suggestions values or a SQL query.',
      'max_suggestions'            , 'Show only up to this amount of suggestions',
--
      'compute_counts'             , 'Identifies whether the filter will compute counts.',
      'show_counts'                , 'Identifies whether the fitler will show counts.',
      'zero_count_entries'         , 'Identifies how filter values with a count of zero should get displayed.',
      'sort_by_top_counts'         , 'Identifies whether the filter values with the highest count are displayed first.',
      'show_selected_first'        , 'Identifies whether the selected filter values are displayed first in the list of available values.',
      'max_displayed_entries'      , 'Show only up to this amount of values for the filter.',
      'has_client_side_filtering'  , 'Identifies whether the filter values can be filtered.',
      'show_chart'                 , 'Identifies whether the filter values can be shown as a chart.',
      'initial_chart'              , 'Identifies whether the chart displaying filter values is initially shown.',
      'actions_filter'             , 'Identifies whether facet provides UI to toggle the client-side filtering on/off',
      'is_toggleable'              , 'Identifies whether end-users can toggle the visibility of the filter',
      'is_initial_toggled'         , 'Identifies whether the filter is displayed by default',
--
      'placeholder'       , 'Identifies the placeholder used for the filter',
      'PRE_ELEMENT_TEXT' , 'Identifies text placed before the filter.',
      'POST_ELEMENT_TEXT' , 'Identifies text placed after the filter.',
--
      'is_collapsible'             , 'Identifies whether the filter is collapsible',
      'is_initial_collapsed'       , 'Identifies whether the filter is initially collapsed',
      'icon_css_classes'           , 'Identifies CSS classes to display an icon for the filter',
      'item_css_classes'           , 'Identifies additional CSS classes to be applied to the filter',
--
      'item_source'                , 'Identifies the filter column.',
      'item_source_data_type'      , 'Identifies the column data type if the item source type is "Region Source Column"',
      'multi_value_type'           , 'Identifies whether the column contains multiple values and how these are stored.',
      'multi_value_type_code'      , 'Internal Code for MULTI_VALUE_TYPE.',
      'multi_value_separator'      , 'Separator character for multi-value columns of the "Separated" type.',
      'multi_value_trim_space'     , 'Identifies whether to trim leading and trailing whitespace from values after split.',
      'multi_value_filter_combination' , 'Identifies how the selected checkboxes within this group are combined.',
--
      'ITEM_DEFAULT_TYPE' , 'Identifies how the Item Default is set, based on a dynamic computation or a static assignment',
      'ITEM_DEFAULT'      , 'When the filter has no session state, use this default value',
--
      'depending_on_id'             , 'Identifies a filter that is used to check if the current filter gets displayed',
      'depending_on_condition_type' , 'Identifies a condition that must be met in order for this filter to be displayed',
      'depending_on_expression'     , 'Identifies the filter depending on condition',
--
      'CONDITION_TYPE'        , 'Identifies a condition that must be met in order for this filter to be displayed',
      'CONDITION_TYPE_CODE'   , 'Identifies the internal code CONDITION_TYPE',
      'CONDITION_EXPRESSION1' , 'Identifies the filter display condition',
      'CONDITION_EXPRESSION2' , 'Identifies the filter display condition',
--
      'AUTHORIZATION_SCHEME'  , 'Identifies the authorization scheme which must evaluate to TRUE in order for this component to be rendered',
      'AUTHORIZATION_SCHEME_ID'      , 'Foreign Key',
      'encrypt_session_state' , 'If Yes values of session state written to APEX session state tables is encrypted.  Decryption is automatic.',
      'escape_on_http_output' , 'If Yes values will be escaped when they are written to the HTTP output.',
--
      'BUILD_OPTION'          , 'Item will be displayed if the Build Option is enabled',
      'COMPONENT_COMMENT'     , 'Identifies a developer comment for this item',
--
      'ITEM_ID'               , 'Primary key of this component',
      'last_updated_by'       , 'APEX developer who made last update',
      'last_updated_on'       , 'Date of last update',
--
      'BUILD_OPTION_ID'       , 'Build Option internal ID',
      'COMPONENT_SIGNATURE'   , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------
-- A P P L I C A T I O N   P A G E   R E G I O N S
-- 4651

prompt ...apex_application_page_regions

create or replace view apex_application_page_regions
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                                 page_id,
    p.name                               page_name,
    --
    r.plug_name                          region_name,
    r.parent_plug_id                     parent_region_id,
    case
      when r.parent_plug_id is not null then
          ( select pp.plug_name
              from wwv_flow_page_plugs pp
             where pp.id                = r.parent_plug_id
               and pp.security_group_id = r.security_group_id)
    end                                  parent_region_name,
    r.filtered_region_id                 filtered_region_id,
    nvl(decode(nvl(r.PLUG_TEMPLATE,0),0,
     'No Template',
     (select pt.PAGE_PLUG_TEMPLATE_NAME
     from wwv_flow_page_plug_templates pt
     where pt.id                = r.PLUG_TEMPLATE
       and pt.security_group_id = r.security_group_id )),'No Template') template,
    r.PLUG_TEMPLATE                      template_id,
    r.region_name                        static_id,
    decode(nvl(r.rest_enabled,'N'),'Y','Yes','No') rest_enabled,
    r.PLUG_DISPLAY_SEQUENCE              display_sequence,
    case r.plug_new_grid
      when 'Y' then 'Yes'
      else 'No'
    end                                  new_grid,
    case r.plug_new_grid_row
      when 'Y' then 'Yes'
      else 'No'
    end                                  new_grid_row,
    r.plug_grid_row_css_classes          grid_row_css_classes,
    case r.plug_new_grid_column
      when 'Y' then 'Yes'
      else 'No'
    end                                  new_grid_column,
    r.PLUG_DISPLAY_COLUMN                grid_column,
    r.plug_grid_column_span              grid_column_span,
    r.plug_grid_column_css_classes       grid_column_css_classes,
    r.plug_column_width                  grid_column_attributes,
    decode(nvl(r.INCLUDE_IN_REG_DISP_SEL_YN,'N'),
        'Y','Yes','N','No',r.INCLUDE_IN_REG_DISP_SEL_YN)
                                         display_region_selector,
    r.region_css_classes,
    r.icon_css_classes,
    r.region_sub_css_classes,
    r.region_template_options            region_template_options,
    r.component_template_options         component_template_options,
    r.REGION_ATTRIBUTES_SUBSTITUTION     REGION_ATTRIBUTES_SUBSTITUTION,
    case r.escape_on_http_output
      when 'Y' then 'Yes'
      when 'N' then 'No'
      else r.escape_on_http_output
    end                                  escape_on_http_output,
    r.PLUG_DISPLAY_COLUMN                display_column, /* for backward compatibility, replaced by grid_column */
    case
      when r.parent_plug_id is not null then
          coalesce( (
              select dp.name
                from wwv_flow_plug_tmpl_disp_points dp
               where dp.plug_template_id  = ( select pr.plug_template
                                                from wwv_flow_page_plugs pr
                                               where pr.id                = r.parent_plug_id
                                                 and pr.security_group_id = r.security_group_id)
                 and dp.flow_id           = r.flow_id
                 and dp.security_group_id = r.security_group_id
                 and dp.placeholder       = r.plug_display_point
          ), r.plug_display_point )
      else
          coalesce( (
              select dp.name
                from wwv_flow_page_tmpl_disp_points dp
               where dp.page_template_id  = coalesce( p.step_template, ( select t.default_page_template
                                                                           from wwv_flow_user_interfaces ui,
                                                                                wwv_flow_themes t
                                                                          where ui.security_group_id = r.security_group_id
                                                                            and ui.flow_id           = r.flow_id
                                                                            and ui.ui_type_id        = 1 -- desktop
                                                                            and t.security_group_id  = ui.security_group_id
                                                                            and t.flow_id            = ui.flow_id
                                                                            and t.theme_id           = ui.theme_id ) )
                 and dp.security_group_id = r.security_group_id
                 and dp.flow_id           = r.flow_id
                 and dp.placeholder       = r.plug_display_point
          ), decode( r.plug_display_point,
                      'AFTER_HEADER',     'After Header',
                      'BEFORE_FOOTER',    'Before Footer',
                      'BODY_1',           'Page Template Body (1)',
                      'BODY_2',           'Page Template Body (2)',
                      'BODY_3',           'Page Template Body (3)',
                      r.plug_display_point ) )
    end                                  display_position,
    r.plug_display_point                 display_position_code,
    case plug_item_display_point
    when 'ABOVE' then 'Above Content'
    when 'BELOW' then 'Below Content'
    end                                  display_item_position,
    r.plug_item_display_point            display_item_position_code,
    --
    r.location                           location_code,
    decode( r.location,
       'LOCAL',       'Local Database',
       'REMOTE',      'Remote Database',
       'WEB_SOURCE',  'Web Source' )     location,
    r.query_type                         query_type_code,
    decode( r.query_type,
       'TABLE',       'Table',
       'SQL',         'SQL Query',
       'FUNC_BODY_RETURNING_SQL', 'Function Body returning SQL Query' ) query_type,
    r.function_body_language               function_body_language,
    r.query_owner                          table_owner,
    r.query_table                          table_name,
    r.query_where                          where_clause,
    r.query_order_by_type                  order_by_type_code,
    case r.query_order_by_type
      when 'STATIC' then 'Static Value'
      when 'ITEM'   then 'Item'
    end                                    order_by_type,
    json_value(
        r.query_order_by,
        '$.itemName' )                     order_by_item,
    r.query_order_by                       order_by_clause,
    decode( r.source_post_processing,
       'SQL', 'SQL Query',
       'WHERE_ORDER_BY_CLAUSE', 'Where/Order By Clause',
       'FUNC_BODY_RETURNING_SQL', 'Function Body returning SQL Query' ) source_post_processing,
    case r.include_rowid_column
      when 'Y' then 'Yes'
      else 'No'
    end                                    include_rowid_column,
    r.plug_source                          region_source,
    r.optimizer_hint,
    --
    (select rs.name
       from wwv_remote_servers rs
      where rs.id                = r.remote_server_id
        and rs.security_group_id = r.security_group_id)       remote_database_name,
    r.remote_server_id                     remote_database_id,
    (select wm.name
       from wwv_flow_web_src_modules wm
      where wm.id                = r.web_src_module_id
        and wm.security_group_id = r.security_group_id)      web_source_module_name,
    r.web_src_module_id                    web_source_module_id,
    case r.use_local_sync_table
      when 'Y' then 'Yes'
      when 'N' then 'No' end               use_local_sync_table,
    --
    external_filter_expr,
    r.external_order_by_type               external_order_by_type_code,
    case r.external_order_by_type
      when 'STATIC' then 'Static Value'
      when 'ITEM'   then 'Item'
    end                                    external_order_by_type,
    json_value(
        r.external_order_by_expr,
        '$.itemName' )                     external_order_by_item,
    external_order_by_expr,
    --
    case r.is_editable
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end as is_editable,
    r.edit_operations,
    r.edit_row_operations_column,
    --
    case when substr( r.add_authorization_scheme, 1, 1) = '!' then 'Not ' end ||
    coalesce (
        ( select ss.name
            from wwv_flow_security_schemes ss
           where to_char( ss.id )        = ltrim( r.add_authorization_scheme, '!' )
             and ss.flow_id              = r.flow_id
             and ss.security_group_id = r.security_group_id
        ),
        r.add_authorization_scheme ) as add_authorization_scheme,
    r.add_authorization_scheme       as add_authorization_scheme_id,
    --
    case when substr( r.update_authorization_scheme, 1, 1) = '!' then 'Not ' end ||
    coalesce (
        ( select ss.name
            from wwv_flow_security_schemes ss
           where to_char( ss.id )     = ltrim( r.update_authorization_scheme, '!' )
             and ss.flow_id           = r.flow_id
             and ss.security_group_id = r.security_group_id
        ),
        r.update_authorization_scheme ) as update_authorization_scheme,
    r.update_authorization_scheme       as update_authorization_scheme_id,
    --
    case when substr( r.delete_authorization_scheme, 1, 1) = '!' then 'Not ' end ||
    coalesce (
        ( select ss.name
            from wwv_flow_security_schemes ss
           where to_char( ss.id ) = ltrim( r.delete_authorization_scheme, '!' )
             and ss.flow_id       = r.flow_id
             and ss.security_group_id = r.security_group_id
        ),
        r.delete_authorization_scheme ) as delete_authorization_scheme,
    r.delete_authorization_scheme       as delete_authorization_scheme_id,
    --
    case r.lost_update_check_type
      when 'VALUES' then 'Row Values'
      when 'COLUMN' then 'Row Version Column'
    end as lost_update_check_type,
    r.lost_update_check_type lost_update_check_type_code,
    r.row_version_column,
    case r.add_row_if_empty
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end as add_row_if_empty,
    case r.lazy_loading
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end as lazy_loading,
    --
    (select mt.name
    from  wwv_flow_menu_templates mt
    where mt.id                = to_char(r.MENU_TEMPLATE_ID)
      and mt.flow_id           = f.application_id
      and mt.security_group_id = r.security_group_id)                 breadcrumb_template,
    r.MENU_TEMPLATE_ID                     breadcrumb_template_id,
    --
    (select lt.list_template_name
    from wwv_flow_list_templates lt
    where lt.id                = r.LIST_TEMPLATE_ID
      and lt.security_group_id = r.security_group_id)         list_template_override,
    r.LIST_TEMPLATE_ID                     list_template_override_id,
    --
    case r.plug_source_type
      when 'NATIVE_PLSQL'                 then 'PL/SQL'
      when 'NATIVE_SIMPLE_CHART'          then 'HTML Chart'
      when 'NATIVE_SVG_CHART'             then 'SVG Chart'
      when 'NATIVE_JET_CHART'             then 'Chart'
      when 'NATIVE_SQL_REPORT'            then 'Report'
      when 'NATIVE_IR'                    then 'Interactive Report'
      when 'NATIVE_STATIC'                then 'HTML/Text'||
                                               case when attribute_02='TEXT' then ' (escape special characters)' end||
                                               case when attribute_01='Y' then ' (with shortcuts)' end
      when 'NATIVE_TREE'                  then 'Tree'
      when 'NATIVE_TABFORM'               then 'Tabular Form'
      when 'NATIVE_URL'                   then 'URL'
      when 'NATIVE_DISPLAY_SELECTOR'      then 'Region Display Selector'
      when 'NATIVE_CALENDAR'              then 'Calendar'
      when 'NATIVE_JSTREE'                then 'JavaScript Tree'
      when 'NATIVE_HELP_TEXT'             then 'Help Text'
      when 'NATIVE_FACETED_SEARCH'        then 'Faceted Search'
      when 'NATIVE_SMART_FILTERS'         then 'Smart Filters'
      else
          case
            when r.plug_source_type like 'NATIVE\_%' escape '\' then
                ( select np.display_name from wwv_flow_plugins np where np.security_group_id = 10 and np.flow_id = 4411 and np.plugin_type = 'REGION TYPE' and np.name = substr(r.plug_source_type, 8) )
            when r.plug_source_type like 'PLUGIN\_%' escape '\' then
                ( select fp.display_name from wwv_flow_plugins fp where fp.security_group_id = r.security_group_id and fp.flow_id = r.flow_id and fp.plugin_type = 'REGION TYPE' and fp.name = substr(r.plug_source_type, 8) )
          end
      end                                source_type,
    case -- compute mapping to pre 5.0 source type
      when r.plug_source_type in ('NATIVE_HELP_TEXT', 'NATIVE_URL', 'NATIVE_TREE', 'NATIVE_JSTREE',
                                  'NATIVE_SVG_CHART', 'NATIVE_SIMPLE_CHART')
      then substr(r.plug_source_type, 8)
      when r.plug_source_type = 'NATIVE_PLSQL'                            then 'PLSQL_PROCEDURE'
      when r.plug_source_type = 'NATIVE_IR'                               then 'DYNAMIC_QUERY'
      when r.plug_source_type = 'NATIVE_SQL_REPORT' and r.query_type = 'FUNC_BODY_RETURNING_SQL' then 'FUNCTION_RETURNING_SQL_QUERY'
      when r.plug_source_type = 'NATIVE_SQL_REPORT'                       then 'SQL_QUERY'
      when r.plug_source_type = 'NATIVE_TABFORM'                          then 'UPDATABLE_SQL_QUERY'
      when r.plug_source_type = 'NATIVE_STATIC' and r.attribute_02='TEXT' then 'STATIC_TEXT_ESCAPE_SC'
      when r.plug_source_type = 'NATIVE_STATIC' and r.attribute_01='Y'    then 'STATIC_TEXT_WITH_SHORTCUTS'
      when r.plug_source_type = 'NATIVE_STATIC'                           then 'STATIC_TEXT'
      when r.plug_source_type = 'NATIVE_DISPLAY_SELECTOR'                 then 'REGION_DISPLAY_SELECTOR'
      when r.plug_source_type = 'NATIVE_CALENDAR'                         then 'CALENDAR'
      when r.plug_source_type = 'NATIVE_BREADCRUMB'                       then 'M'||trunc(r.menu_id)
      when r.plug_source_type = 'NATIVE_LIST'                             then to_char(trunc(r.list_id))
      else r.plug_source_type
      end                                source_type_code,
    r.plug_source_type                   source_type_plugin_name,
    --
    --
    decode(substr(r.PLUG_REQUIRED_ROLE,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(r.PLUG_REQUIRED_ROLE,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = r.security_group_id),
     r.PLUG_REQUIRED_ROLE)               authorization_scheme,
    r.PLUG_REQUIRED_ROLE                 authorization_scheme_id,
    nvl((select r from apex_standard_conditions where d = r.PLUG_DISPLAY_CONDITION_TYPE),r.PLUG_DISPLAY_CONDITION_TYPE)
                                         condition_type,
    r.plug_display_condition_type        condition_type_code,
    r.PLUG_DISPLAY_WHEN_CONDITION        condition_expression1,
    r.PLUG_DISPLAY_WHEN_COND2            condition_expression2,
    -- read only condition
    coalesce (
        ( select r
            from apex_standard_conditions
           where r.plug_read_only_when_type is not null
             and d = r.plug_read_only_when_type
        ),
        r.plug_read_only_when_type )     read_only_condition_type,
    r.plug_read_only_when_type           read_only_condition_type_code,
    r.plug_read_only_when                read_only_condition_exp1,
    r.plug_read_only_when2               read_only_condition_exp2,
    --
    r.PLUG_HEADER                        region_header_text,
    r.PLUG_FOOTER                        region_footer_text,
    r.REGION_IMAGE                       image_file_url,
    r.REGION_IMAGE_ATTR                  image_custom_attributes,
    r.REGION_IMAGE_ALT_TEXT              image_accessible_description,
    (select rt.row_template_name from wwv_flow_row_templates rt where rt.security_group_id = r.security_group_id and rt.id = r.PLUG_QUERY_ROW_TEMPLATE)
                                         report_template,
    r.PLUG_QUERY_ROW_TEMPLATE            report_template_id,
    r.PLUG_QUERY_HEADINGS                report_column_headings,
    r.PLUG_QUERY_HEADINGS_TYPE           headings_type,
    r.PLUG_QUERY_NUM_ROWS                maximum_rows_to_query,
    decode(r.PLUG_QUERY_NUM_ROWS_TYPE,
       'ROWS_X_TO_Y_OF_Z','Row Ranges X to Y of Z (no pagination)',
       'ROWS_X_TO_Y','Row Ranges X to Y (no pagination)',
       'SEARCH_ENGINE','Search Engine 1,2,3,4 (set based pagination)',
       'COMPUTED_BUT_NOT_DISPLAYED','Use Externally Created Pagination Buttons',
       'ROW_RANGES','Row Ranges 1-15 16-30 (with set pagination)',
       'ROW_RANGES_IN_SELECT_LIST','Row Ranges 1-15 16-30 in select list (with pagination)',
       'ROW_RANGES_WITH_LINKS','Row Ranges X to Y of Z (with pagination)',
       'NEXT_PREVIOUS_LINKS','Row Ranges X to Y (with next and previous links)',
       r.PLUG_QUERY_NUM_ROWS_TYPE)       pagination_scheme,
    decode(r.PAGINATION_DISPLAY_POSITION,
      'BOTTOM_LEFT','Bottom - Left',
      'BOTTOM_RIGHT','Bottom - Right',
      'TOP_LEFT','Top - Left',
      'TOP_RIGHT','Top - Right',
      'TOP_AND_BOTTOM_LEFT','Top and Bottom - Left',
      'TOP_AND_BOTTOM_RIGHT','Top and Bottom - Right',
      r.PAGINATION_DISPLAY_POSITION)     pagination_display_position,
    decode(r.ajax_enabled,'Y','Yes','N','No',r.ajax_enabled) ajax_enabled,
    r.ajax_items_to_submit,
    r.PLUG_QUERY_NUM_ROWS_ITEM           number_of_rows_item,
    r.PLUG_QUERY_NO_DATA_FOUND           no_data_found_message,
    r.PLUG_QUERY_MORE_DATA               more_data_found_message,
    r.PLUG_QUERY_ROW_COUNT_MAX           maximum_row_count,
    --r.PLUG_QUERY_FORMAT_OUT              query_format_out,
    r.PLUG_QUERY_SHOW_NULLS_AS           report_null_values_as,
    --r.PLUG_QUERY_COL_ALLIGNMENTS         ,
    r.PLUG_QUERY_BREAK_COLS              breaks,
    --r.PLUG_QUERY_SUM_COLS                ,
    --r.PLUG_QUERY_NUMBER_FORMATS          ,
    --r.PLUG_QUERY_TABLE_BORDER            ,
    --r.PLUG_QUERY_HIT_HIGHLIGHTING        ,
    r.PLUG_QUERY_ASC_IMAGE               ascending_image,
    r.PLUG_QUERY_ASC_IMAGE_ATTR          ascending_image_attributes,
    r.PLUG_QUERY_DESC_IMAGE              descending_image,
    r.PLUG_QUERY_DESC_IMAGE_ATTR         descending_image_attributes,
    r.PLUG_QUERY_EXP_FILENAME            filename,
    r.PLUG_QUERY_EXP_SEPARATOR           separator,
    r.PLUG_QUERY_EXP_ENCLOSED_BY         enclosed_by,
    decode(r.PLUG_QUERY_STRIP_HTML,
      'Y','Yes',
      'N','No',
      r.PLUG_QUERY_STRIP_HTML)           strip_html,
    r.PLUG_QUERY_OPTIONS                 report_column_source_type,
    r.PLUG_QUERY_MAX_COLUMNS             max_dynamic_report_cols,
    r.PLUG_COLUMN_WIDTH                  HTML_table_cell_attributes, /* for backward compatibility, replaced by grid_column_attributes */
    decode(r.fixed_header,
        'NONE',     'None',
        'PAGE',     'Page',
        'REGION',   'Region',
        r.fixed_header) fixed_header,
    r.fixed_header_max_height,
    decode(r.PLUG_CUSTOMIZED,
       null,'None',
       '2','Customizable and Not Shown By Default',
       '1','Customizable and Shown By Default',
       '0','Not Customizable By End Users',
       r.PLUG_CUSTOMIZED)                customization,
    r.PLUG_CUSTOMIZED_NAME               customization_name,
    --r.PLUG_OVERRIDE_REG_POS              ,
    (select case when r.required_patch > 0 then fp.PATCH_NAME else '{Not '||fp.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches fp
     where fp.id                = abs(r.REQUIRED_PATCH)
       and fp.security_group_id = r.security_group_id)    build_option,
    r.required_patch                     build_option_id,
    --
    --r.PLUG_URL_TEXT_BEGIN                ,
    --r.PLUG_URL_TEXT_END                  ,
    --
    -- region caching
    --
    decode(r.PLUG_CACHING,
        'CACHE'   , 'Cached'            ,
        'USER'    , 'Cached by User'    ,
        'SESSION' , 'Cached by Session' ,
        'NOCACHE' , 'Not Cached'        ,
        r.plug_caching)                  region_caching,
    --r.plug_caching_session_state         ,
    r.plug_caching_max_age_in_sec        timeout_cache_after,
    r.plug_cache_when                    cache_when,
    r.plug_cache_expression1             cache_when_expression_1,
    r.plug_cache_expression2             cache_when_expression_2,
    r.plug_cache_depends_on_items        cache_depends_on_items,
    --r.PLUG_IGNORE_PAGINATION             ,
    --r.PLUG_CHART_FONT_SIZE               ,
    --r.PLUG_CHART_MAX_ROWS                ,
    --r.PLUG_CHART_NUM_MASK                ,
    --r.PLUG_CHART_SCALE                   ,
    --r.PLUG_CHART_AXIS                    ,
    --r.PLUG_CHART_SHOW_SUMMARY            ,
    r.REPORT_TOTAL_TEXT_FORMAT           sum_display_text,
    r.BREAK_COLUMN_TEXT_FORMAT           break_display_text,
    r.BREAK_BEFORE_ROW                   before_break_display_text,
    r.BREAK_GENERIC_COLUMN               break_column_display_text,
    r.BREAK_AFTER_ROW                    after_break_display_text,
    decode(
       r.BREAK_TYPE_FLAG,
       'REPEAT_HEADINGS_ON_BREAK_1','Repeat Headings on Break',
       'DEFAULT_BREAK_FORMATTING','Default Break Formatting',
       r.BREAK_TYPE_FLAG)                break_display_flag,
    r.BREAK_REPEAT_HEADING_FORMAT        repeat_heading_break_format,
    decode(r.CSV_OUTPUT,
    'Y','Yes','N','No',r.CSV_OUTPUT)     enable_csv_output,
    r.CSV_OUTPUT_LINK_TEXT               csv_link_label,
    r.PRINT_URL                          URL,
    r.PRINT_URL_LABEL                    link_label,
    decode(r.TRANSLATE_TITLE,
    'Y','Yes','N','No','Yes')            translate_region_title,
    --
    r.plugin_init_javascript_code        init_javascript_code,
    --
    r.attribute_01,
    r.attribute_02,
    r.attribute_03,
    r.attribute_04,
    r.attribute_05,
    r.attribute_06,
    r.attribute_07,
    r.attribute_08,
    r.attribute_09,
    r.attribute_10,
    r.attribute_11,
    r.attribute_12,
    r.attribute_13,
    r.attribute_14,
    r.attribute_15,
    r.attribute_16,
    r.attribute_17,
    r.attribute_18,
    r.attribute_19,
    r.attribute_20,
    r.attribute_21,
    r.attribute_22,
    r.attribute_23,
    r.attribute_24,
    r.attribute_25,
    --
    r.LAST_UPDATED_BY                    last_updated_by,
    r.LAST_UPDATED_ON                    last_updated_on,
    r.PLUG_COMMENT                       component_comment,
    r.id                                 region_id,
    r.master_region_id,
    --
    (select count(*) from wwv_flow_step_items fi where fi.security_group_id = r.security_group_id and r.id = fi.ITEM_PLUG_ID and r.flow_id = fi.flow_id) items,
    (select count(*) from wwv_flow_step_buttons fb where fb.security_group_id = r.security_group_id and r.id = fb.button_plug_id and r.flow_id = fb.flow_id) buttons,
     --
     region_name||'.'||
     lpad(r.plug_display_sequence,5,'00000')
     ||',c='||r.PLUG_DISPLAY_column
     ||',temp='||nvl(decode(nvl(r.PLUG_TEMPLATE,0),0,'No Template',(select pt.PAGE_PLUG_TEMPLATE_NAME from wwv_flow_page_plug_templates pt where pt.id = r.PLUG_TEMPLATE and pt.security_group_id = r.security_group_id)),'No Template')
     ||',type='||r.plug_source_type
     ||',src='||sys.dbms_lob.substr(r.PLUG_SOURCE,30,1)||'.'||sys.dbms_lob.getlength(r.PLUG_SOURCE)
     ||(select ',bo='||pa.PATCH_NAME b from wwv_flow_patches pa where pa.id = abs(r.REQUIRED_PATCH) and pa.security_group_id = r.security_group_id)
     ||nvl((select ',auth='||ss.name n from wwv_flow_security_schemes ss where to_char(ss.id) = ltrim(r.PLUG_REQUIRED_ROLE,'!') and ss.flow_id = f.application_id and ss.security_group_id = r.security_group_id),'Not found')
     ||decode(r.PAGINATION_DISPLAY_POSITION,
      'BOTTOM_LEFT','Bottom-L',
      'BOTTOM_RIGHT','Bottom-R',
      'TOP_LEFT','Top-L',
      'TOP_RIGHT','Top-R',
      'TOP_AND_BOTTOM_LEFT','Top+Bottom-L',
      'TOP_AND_BOTTOM_RIGHT','Top+Bottom-R',
      r.PAGINATION_DISPLAY_POSITION)
      ||r.PLUG_QUERY_NUM_ROWS_TYPE
      ||r.PLUG_DISPLAY_CONDITION_TYPE
      ||substr(r.PLUG_DISPLAY_WHEN_CONDITION,1,20)||'.'||length(r.PLUG_DISPLAY_WHEN_CONDITION)
      ||substr(r.PLUG_DISPLAY_WHEN_COND2,1,20)||'.'||length(r.PLUG_DISPLAY_WHEN_COND2)
      ||decode(
       r.BREAK_TYPE_FLAG,
       'REPEAT_HEADINGS_ON_BREAK_1','RepHead on Br',
       'DEFAULT_BREAK_FORMATTING','DefBreakFor',
       r.BREAK_TYPE_FLAG)
      ||decode(r.CSV_OUTPUT,'Y','Yes','N','No',r.CSV_OUTPUT)
      ||(select ',rtmp='||rt.row_template_name t from wwv_flow_row_templates rt where rt.id = r.PLUG_QUERY_ROW_TEMPLATE and rt.security_group_id = r.security_group_id)
      ||decode(r.PLUG_QUERY_NUM_ROWS,null,null,'mr='||r.PLUG_QUERY_NUM_ROWS)
      ||r.BREAK_REPEAT_HEADING_FORMAT
      ||r.CSV_OUTPUT_LINK_TEXT
      ||substr(r.PRINT_URL,1,10)||length(r.PRINT_URL)
      ||length(PRINT_URL_LABEL)
      ||decode(r.TRANSLATE_TITLE,'Y','Yes','N','No','Yes')||
      length(PLUG_HEADER)||length(PLUG_FOOTER)
      ||r.PLUG_CUSTOMIZED
      ||length(r.PLUG_COLUMN_WIDTH)
      ||substr(r.region_name,1,15)
      ||length(REGION_ATTRIBUTES_SUBSTITUTION)
      ||r.REGION_IMAGE
      ||r.REGION_IMAGE_ATTR
      ||r.REGION_IMAGE_ALT_TEXT
     component_signature,
     list_id,
     menu_id breadcrumb_id
from wwv_flow_page_plugs r,
     wwv_flow_steps      p,
     wwv_flow_authorized f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and f.workspace_id      = r.security_group_id
  and f.application_id    = r.flow_id
  and p.security_group_id = r.security_group_id
  and p.flow_id           = r.flow_id
  and p.id                = r.page_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_regions',
    p_table_comment     => 'Identifies a content container associated with a Page and displayed within a position defined by the Page Template',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'page_id'                       , 'Identifies the application',
      'page_name'                     , 'Identifies a page within an application',
      'REGION_NAME'                   , 'Identifies the Region Name.  The display of the region name is controlled by the Region Template substitution string TITLE.',
      'parent_region_id'              , 'Identifies the region id of the parent region',
      'parent_region_name'            , 'Identifies the parent region',
      'filtered_region_id'            , 'Identifies the id of the region filtered by this faceted search or smart filters region',
      'TEMPLATE'                      , 'Identifies the template used to display the region',
      'TEMPLATE_ID'                   , 'Foreign Key',
      'DISPLAY_SEQUENCE'              , 'Identifies the Display Sequence of the Region within each Display Position',
      'region_css_classes'            , 'Additional CSS classes which are substituted if the region template contains the substitution string #REGION_CSS_CLASSES#',
      'region_css_classes'            , 'Additional CSS classes which are used to display a region icon',
      'region_sub_css_classes'        , 'Additional CSS classes which are substituted if the report, list or meny template contains the substitution string #[REPORT/LIST/MENU]_CSS_CLASSES#',
      'region_template_options'       , 'Identifies template options to be applied to the region template',
      'component_template_options'    , 'Identifies template options to be applied to the component template',
      'REGION_ATTRIBUTES_SUBSTITUTION' , 'Identifies text to be substituted by the region template #REGION_ATTRIBUTES# substituion string',
      'ESCAPE_ON_HTTP_OUTPUT'         , 'If Yes values will be escaped when they are written to the HTTP output.',
      'new_grid'                      , 'Identifies if the rendering of region begins on a new grid',
      'new_grid_row'                  , 'Identifies if the rendering of region begins on a new grid row',
      'grid_row_css_classes'          , 'Additional CSS classes which are substituted if the grid row template contains the substitution string #CSS_CLASSES#',
      'new_grid_column'               , 'Identifies if the rendering of region begins on a new grid column',
      'grid_column'                   , 'Identifies the grid column used to display the region',
      'grid_column_span'              , 'Identifies over how many grid columns the item spans',
      'grid_column_css_classes'       , 'Additional CSS classes which are substituted if the grid column template contains the substitution string #CSS_CLASSES#',
      'grid_column_attributes'        , 'Additional attributes which are substituted if the grid column template contains the substitution string #ATTRIBUTES#',
      'display_position'              , 'Identifies the position where the region will be displayed',
      'display_position_code'         , 'Identifies the internal code of DISPLAY_POSITION',
      'DISPLAY_ITEM_POSITION'         , 'Identifies if page items should be rendered before or after the region content',
      'DISPLAY_ITEM_POSITION_CODE'    , 'Identifies the internal code of DISPLAY_ITEM_POSITION',
      'REGION_SOURCE'                 , 'Identifies the source of the region, reference Region Source Type',
      'location_code'                 , 'Identifies the internal code of LOCATION',
      'location'                      , 'Identifies whether the data source of the region is the local database, a REST Enabled SQL service or a REST data source',
      'query_type_code'               , 'Identifies the internal code of QUERY_TYPE',
      'query_type'                    , 'Identifies whether the region data comes from a table, a SQL query or a PL/SQL function returning a SQL query',
      'function_body_language'        , 'Identifies the programming language of REGION_SOURCE.',
      'table_owner'                   , 'Denotes the table owner for TABLE query types',
      'table_name'                    , 'Denotes the table name for TABLE query types',
      'where_clause'                  , 'Identifies the WHERE clause provided by the developer and to be applied to TABLE or REST data sources.',
      'order_by_type_code'            , 'Identifies the internal code of ORDER_BY_TYPE',
      'order_by_type'                 , 'Identifies whether the region data is sorted by a static order by clause or derived from a page item.',
      'order_by_item'                 , 'Identifies the page item which is used to derive the current order by clause.',
      'order_by_clause'               , 'Identifies the ORDER BY clause provided by the developer and to be applied to TABLE or REST data sources',
      'source_post_processing'        , 'Identifies type of SQL post processing for data from external data sources',
      'include_rowid_column'          , 'Identifies if the pseudo column ROWID should be selected',
      'optimizer_hint'                , 'Optimizer hint which is added to the region source SQL statement when it is executed',
      'remote_database_id'            , 'Identifies the primary key of the REST Enabled SQL definition',
      'remote_database_name'          , 'Identifies the name of the REST Enabled SQL service',
      'web_source_module_id'          , 'Identifies the primary key of the Rest Data Source',
      'web_source_module_name'        , 'Identifies the Rest Data Source name',
      'use_local_sync_table'          , 'Whether to use the local synchronization table instead of the REST service',
      'external_filter_expr'          , 'Filter expression to send to the external data source',
      'external_order_by_type_code'    , 'Identifies the internal code of EXTERNAL_ORDER_BY_TYPE',
      'external_order_by_type'         , 'Identifies whether the external data source is sorted by a static order by clause or derived from a page item.',
      'external_order_by_item'         , 'Identifies the page item which is used to derive the current external order by clause.',
      'external_order_by_expr'         , 'Order By expression to send to the external data source',
      'is_editable'                    ,  'Identifies if end users can edit this region',
      'edit_operations'                ,  'Identifies if data can be updated, inserted, or deleted',
      'edit_row_operations_column'     ,  'A column where the column value can contain U (update) and or D (delete) to indicate if the row can be updated or deleted',
      'add_authorization_scheme'       ,  'Authorization scheme to restrict who can add rows',
      'add_authorization_scheme_id'    ,  'Identifies the internal code of ADD_AUTHORIZATION_SCHEME',
      'update_authorization_scheme'    ,  'Authorization scheme to restrict who can update rows',
      'update_authorization_scheme_id' ,  'Identifies the internal code of UPDATE_AUTHORIZATION_SCHEME',
      'delete_authorization_scheme'    ,  'Authorization scheme to restrict who can delete rows',
      'delete_authorization_scheme_id' ,  'Identifies the internal code of DELETE_AUTHORIZATION_SCHEME',
      'lost_update_check_type'         ,  'Identifies how to protect data against lost updates',
      'lost_update_check_type_code'    ,  'Identifies the internal code of LOST_UPDATE_CHECK_TYPE',
      'row_version_column'             ,  'A column to use as lost update check',
      'add_row_if_empty'               ,  'Identifies whether to add a new row if the region is empty',
      'lazy_loading'                   ,  'Identifies whether to defer getting initial set of data from the server until needed',
      'LIST_TEMPLATE_OVERRIDE'        , 'Identifies the List Template to be used to display regions of type List.  By default the List Template is defined in the List definition, if a value is specified in this attribute, this template will be used to render the List.',
      'SOURCE_TYPE'                   , 'Identifies how APEX will interpret the Region Source',
      'SOURCE_TYPE_CODE'              , 'Deprecated internal code of SOURCE_TYPE. Use SOURCE_TYPE_PLUGIN_NAME instead.',
      'source_type_plugin_name'       , 'Name of the plugin that implements the region. Names of built-in plugins start with "NATIVE_", application specific plugin names start with "PLUGIN_"',
      'AUTHORIZATION_SCHEME'          , 'An authorization scheme must evaluate to TRUE in order for this component to be displayed',
      'AUTHORIZATION_SCHEME_ID'       , 'Foreign Key',
      'CONDITION_TYPE'                , 'Identifies the condition type used to conditionally display the region',
      'CONDITION_TYPE_CODE'           , 'Identifies the internal code of CONDITION_TYPE',
      'CONDITION_EXPRESSION1'         , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'         , 'Specifies an expression based on the specific condition type selected.',
      'read_only_condition_type'      , 'Identifies the condition type when all items of the region will be rendered read only',
      'read_only_condition_type_code' , 'Identifies the internal code of read_only_condition_type',
      'read_only_condition_exp1'      , 'Identifies the condition that determines if all items of the region will be rendered read only',
      'read_only_condition_exp2'      , 'Identifies the condition that determines if all items of the region will be rendered read only',
      'REPORT_TEMPLATE'               , 'Report templates provide control over the results of a row from your SQL query',
      'MAXIMUM_ROWS_TO_QUERY'         , 'Specifies the maximum rows that can be retuned by a given query.  Avoids attempting to send millions of rows to a web browser.',
      'PAGINATION_SCHEME'             , 'Pagination provides the user with information about the number of rows and the current position within the result set. Pagination also defines the style of links or buttons that are used to navigate to the next or previous page.',
      'PAGINATION_DISPLAY_POSITION'   , 'Pagination can be displayed on the left side, right side, at the bottom, or above the report.',
      'AJAX_ENABLED'                  , 'Specifies if the region is ajax enabled.',
      'AJAX_ITEMS_TO_SUBMIT'          , 'Identifies the page items which should be submitted when an Ajax call is performed for the region.',
      'NUMBER_OF_ROWS_ITEM'           , 'Defines the maximum number of rows to display per page.',
      'NO_DATA_FOUND_MESSAGE'         , 'Defines the text message that displays when the query does not return any rows.',
      'MORE_DATA_FOUND_MESSAGE'       , 'Defines the text message that displays when more rows exist.',
      'MAXIMUM_ROW_COUNT'             , 'Defines the maximum number of rows to query, relevant for pagination schemes which display "Rows X - Y of Z".  Counting fewer rows can improve performance and counting thousands of rows can degrade performance.',
      'REPORT_NULL_VALUES_AS'         , 'Identifies the text to display for null columns. The default value is "(null)".',
      'BREAKS'                        , 'Identifies how breaks should be formatted.',
      'ASCENDING_IMAGE'               , 'Defines the image shown in report headings to sort column values in ascending order.',
      'ASCENDING_IMAGE_ATTRIBUTES'    , 'Image attributes for sort images used to define attributes such width and height of image.',
      'DESCENDING_IMAGE'              , 'Defines the image shown in report headings to sort column values in descending order.',
      'DESCENDING_IMAGE_ATTRIBUTES'   , 'Image attributes for sort images used to define attributes such width and height of image.',
      'FILENAME'                      , 'Specify a name for the export file. If no name is specified, the region name is used followed by the extension .csv.',
      'SEPARATOR'                     , 'Identifies a column separator. If no value is entered, a comma or semicolon is used depending on your current NLS settings.',
      'ENCLOSED_BY'                   , 'Identifies a delimiter character. This character is used to delineate the starting and ending boundary of a data value. Default delimiter is double quotes.',

      'static_id'                     , 'Reference this value using the substitution string #REGION_STATIC_ID#',
      'rest_enabled'                  , 'Identifies whether the region can be accessed via a RESTful call',
      'headings_type'                 , 'Identifies the how report column headings are computed',
      'report_column_headings'        , 'Report column heading override, can be used to define dynamic report column headings',
      'customization'                 , 'Identifies level of customization support for this page region',
      'customization_name'            , 'Name of region to show in popup customization window',
      'list_template_override_id'     , 'Foreign key of list template override',
      'HTML_table_cell_attributes'    , 'When generating HTML to display items use these HTML table cell attributes',

      'max_dynamic_report_cols'       , 'Maximum number of dynamic report columns',
      'report_column_source_type'     , 'Determines how report columns will be computed, values include DERIVED_REPORT_COLUMNS and GENERIC_REPORT_COLUMNS',
      'break_display_text'            , 'Text displayed on control breaks',
      'STRIP_HTML'                    , 'Specifies whether or not to remove HTML tags from the original column values for HTML expressions, column links and report data exported as CSV files.',
      'BUILD_OPTION'                  , 'Region will be displayed if the Build Option is enabled',
      'BUILD_OPTION_ID'               , 'Foreign Key',
      'SUM_DISPLAY_TEXT'              , 'Display this text when printing report sums',
      'BEFORE_BREAK_DISPLAY_TEXT'     , 'Defines the text that displays before break columns when displaying a break row.',
      'BREAK_COLUMN_DISPLAY_TEXT'     , 'Defines the column template to use when displaying a column break. Use #COLUMN_VALUE# substitutions.',
      'AFTER_BREAK_DISPLAY_TEXT'      , 'Defines the text that displays after break columns when displaying a break row.',
      'REPEAT_HEADING_BREAK_FORMAT'   , 'Defines the heading template for repeating headings on column 1. Use #COLUMN_VALUE# substitutions.',
      'ENABLE_CSV_OUTPUT'             , 'Enables the query results to be spooled to a CSV file. To enable this option, you must use a report template with a #CSV_LINK# substitution string and set this option to YES.',
      'CSV_LINK_LABEL'                , 'Specifies the text for the link which will invoke the CSV download.',
      'URL'                           , 'Specifies the URL to a server for post processing of a report. See documentation for instructions.',
      'LINK_LABEL'                    , 'Specifies the text for the link which will invoke the external processing engine.',
      'TRANSLATE_REGION_TITLE'        , 'Identifies if this region title should be a candidate for translation or not (Yes or No)',
      'break_display_flag'            , 'Identify how you would like your breaks to be displayed',
      'COMPONENT_COMMENT'             , 'Developer comment',
      'REGION_ID'                     , 'Primary Key of this Region',
      'master_region_id'              , 'Associated master region to identify master detail relationship',
      'ITEMS'                         , 'Count of Items corresponding to this region',
      'BUTTONS'                       , 'Count of Buttons corresponding to this region',
      'last_updated_by'               , 'APEX developer who made last update',
      'last_updated_on'               , 'Date of last update',
      'region_caching'                , 'Identifies caching method',
      'timeout_cache_after'           , 'Identify how long a cached region will remain valid in seconds',
      'cache_when'                    , 'Identifies a condition must be true for the region to be cached or to render from cache',
      'cache_when_expression_1'       , 'Identifies expression corresponding to Cache When condition',
      'cache_when_expression_2'       , 'Identifies expression corresponding to Cache When condition',
      'cache_depends_on_items'        , 'Identifies items that the cached content depends on',
      'component_signature'           , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'breadcrumb_template'           , 'Identifies breadcrumb template',
      'breadcrumb_template_id'        , 'Identifies breadcrumb template ID',
      'report_template_id'            , 'Identifies report template ID',

      'init_javascript_code'          , 'JavaScript code for region initialization',

      'attribute_01'                  , 'Dynamic attribute to store additional data',
      'attribute_02'                  , 'Dynamic attribute to store additional data',
      'attribute_03'                  , 'Dynamic attribute to store additional data',
      'attribute_04'                  , 'Dynamic attribute to store additional data',
      'attribute_05'                  , 'Dynamic attribute to store additional data',
      'attribute_06'                  , 'Dynamic attribute to store additional data',
      'attribute_07'                  , 'Dynamic attribute to store additional data',
      'attribute_08'                  , 'Dynamic attribute to store additional data',
      'attribute_09'                  , 'Dynamic attribute to store additional data',
      'attribute_10'                  , 'Dynamic attribute to store additional data',
      'attribute_11'                  , 'Dynamic attribute to store additional data',
      'attribute_12'                  , 'Dynamic attribute to store additional data',
      'attribute_13'                  , 'Dynamic attribute to store additional data',
      'attribute_14'                  , 'Dynamic attribute to store additional data',
      'attribute_15'                  , 'Dynamic attribute to store additional data',
      'attribute_16'                  , 'Dynamic attribute to store additional data',
      'attribute_17'                  , 'Dynamic attribute to store additional data',
      'attribute_18'                  , 'Dynamic attribute to store additional data',
      'attribute_19'                  , 'Dynamic attribute to store additional data',
      'attribute_20'                  , 'Dynamic attribute to store additional data',
      'attribute_21'                  , 'Dynamic attribute to store additional data',
      'attribute_22'                  , 'Dynamic attribute to store additional data',
      'attribute_23'                  , 'Dynamic attribute to store additional data',
      'attribute_24'                  , 'Dynamic attribute to store additional data',
      'attribute_25'                  , 'Dynamic attribute to store additional data',
      'list_id'                       , 'Associated list, if region is of type list.  Can be joined to APEX_APPLICATION_LISTS.',
      'breadcrumb_id'                 , 'Associated breadcrumb, if region is of type breadcrumb.  Can be joined to APEX_APPLICATION_BREADCRUMBS.',
      'fixed_header'                  , 'Identifies if the report has a fixed header, and if so where it is fixed to (Page or Region)',
      'fixed_header_max_height'       , 'For reports with a header fixed to Region, this identifies the maximum height',

      'DISPLAY_REGION_SELECTOR'       , 'Is this region included in the Region Display Selector.',
      'ICON_CSS_CLASSES'              , 'The CSS classes that will be added to the region icon.',
      'DISPLAY_COLUMN'                , 'For backward compatibility, replaced by grid_column ',
      'REGION_HEADER_TEXT'            , 'HTML text displayed at the top of the region.',
      'REGION_FOOTER_TEXT'            , 'HTML text that displays directly after the body content.',
      'image_file_url'                , 'File URL for an image displayed in the region.',                   
      'image_custom_attributes'       , 'Custom attributes to style the region image.',
      'image_accessible_description'  , 'Accessible description placed in the alt attribute of the region image HTML.'
    )
  );
end;
/
create or replace view apex_application_page_rpt
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                                 page_id,
    p.name                               page_name,
    --
    r.plug_name                          region_name,
    decode(r.PLUG_SOURCE_TYPE,
        'NATIVE_SQL_REPORT','Report',
        'NATIVE_TABFORM','Tabular Form',
        'NATIVE_IR','Interactive Report',
        'NATIVE_IG','Interactive Grid') source_type,
    r.supplemental_text                       supplemental_text,
    PRN_BODY_BG_COLOR                         body_background_color,
    PRN_BODY_FONT_COLOR                       body_font_color,
    PRN_BODY_FONT_FAMILY                      body_font_family,
    PRN_BODY_FONT_SIZE                        body_font_size,
    PRN_BODY_FONT_WEIGHT                      body_font_weight,
    PRN_BORDER_COLOR                          border_color,
    PRN_BORDER_WIDTH                          border_width,
    PRN_FORMAT                                format,
    PRN_FORMAT_ITEM                           format_item,
    PRN_PAGE_HEADER                           page_header,
    PRN_PAGE_HEADER_ALIGNMENT                 page_header_alignment,
    PRN_PAGE_HEADER_FONT_COLOR                page_header_font_color,
    PRN_PAGE_HEADER_FONT_FAMILY               page_header_font_family,
    PRN_PAGE_HEADER_FONT_SIZE                 page_header_font_size,
    PRN_PAGE_HEADER_FONT_WEIGHT               page_header_font_weight,
    PRN_HEADER_BG_COLOR                       header_background_color,
    PRN_HEADER_FONT_COLOR                     header_font_color,
    PRN_HEADER_FONT_FAMILY                    header_font_family,
    PRN_HEADER_FONT_SIZE                      header_font_size,
    PRN_HEADER_FONT_WEIGHT                    header_font_weight,
    PRN_HEIGHT                                height,
    PRN_ORIENTATION                           orientation,
    PRN_OUTPUT                                output,
    cast(null as varchar2(4000))              print_server_override,        -- obsolete
    PRN_OUTPUT_FILE_NAME                      output_file_name,
    prn_content_disposition                   content_disposition,
    cast(null as varchar2(255))               document_header,              -- obsolete
    PRN_OUTPUT_LINK_TEXT                      output_link_text,
    PRN_OUTPUT_SHOW_LINK                      show_output_link,
    PRN_PAGE_FOOTER                           footer,
    PRN_PAGE_FOOTER_ALIGNMENT                 footer_alignment,
    PRN_PAGE_FOOTER_FONT_COLOR                footer_font_color,
    PRN_PAGE_FOOTER_FONT_FAMILY               footer_font_family,
    PRN_PAGE_FOOTER_FONT_SIZE                 footer_font_size,
    PRN_PAGE_FOOTER_FONT_WEIGHT               footer_font_weight,
    --
    PRN_PAPER_SIZE                            paper_size,
    PRN_TEMPLATE_ID                           template_id,
    component_template_options                component_template_options,
    PRN_UNITS                                 paper_size_units,
    prn_width_units                           paper_size_width_units,
    PRN_WIDTH                                 paper_size_width,
    --
    r.LAST_UPDATED_BY                    last_updated_by,
    r.LAST_UPDATED_ON                    last_updated_on,
    r.PLUG_COMMENT                       component_comment,
    r.id                                 region_id,
    --
     region_name||'.'
     ||PRN_BODY_BG_COLOR
     ||PRN_BODY_FONT_COLOR
     ||PRN_BODY_FONT_FAMILY
     ||PRN_BODY_FONT_SIZE
     ||PRN_BODY_FONT_WEIGHT
     ||PRN_BORDER_COLOR
     ||PRN_BORDER_WIDTH
     ||length(PRN_FORMAT)
     ||PRN_FORMAT_ITEM
     ||length(PRN_PAGE_HEADER)
     ||PRN_PAGE_HEADER_ALIGNMENT
     ||PRN_PAGE_HEADER_FONT_COLOR
     ||PRN_PAGE_HEADER_FONT_FAMILY
     ||PRN_PAGE_HEADER_FONT_SIZE
     ||PRN_PAGE_HEADER_FONT_WEIGHT
     ||PRN_HEADER_BG_COLOR
     ||PRN_HEADER_FONT_COLOR
     ||PRN_HEADER_FONT_FAMILY
     ||PRN_HEADER_FONT_SIZE
     ||PRN_HEADER_FONT_WEIGHT
     ||PRN_HEIGHT
     ||PRN_ORIENTATION
     ||PRN_OUTPUT
     ||PRN_OUTPUT_FILE_NAME
     ||PRN_CONTENT_DISPOSITION
     ||PRN_UNITS
     ||PRN_OUTPUT_LINK_TEXT
     ||PRN_OUTPUT_SHOW_LINK
     ||length(PRN_PAGE_FOOTER)
     ||PRN_PAGE_FOOTER_ALIGNMENT
     ||PRN_PAGE_FOOTER_FONT_COLOR
     ||PRN_PAGE_FOOTER_FONT_FAMILY
     ||PRN_PAGE_FOOTER_FONT_SIZE
     ||PRN_PAGE_FOOTER_FONT_WEIGHT
     ||PRN_PAPER_SIZE
     ||PRN_TEMPLATE_ID
     ||PRN_UNITS
     ||PRN_WIDTH
     component_signature
from wwv_flow_page_plugs r,
     wwv_flow_steps      p,
     wwv_flow_authorized f
where f.workspace_id      =  p.security_group_id
  and f.application_id    =  p.flow_id
  and p.security_group_id =  r.security_group_id
  and p.flow_id           =  r.flow_id
  and p.id                =  r.page_id
  and r.PLUG_SOURCE_TYPE  in ('NATIVE_SQL_REPORT', 'NATIVE_TABFORM', 'NATIVE_IR', 'NATIVE_IG')
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_rpt',
    p_table_comment     => 'Printing attributes for regions that are reports',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_id'                    , 'Identifies the application',
      'page_name'                  , 'Identifies a page within an application',
      'REGION_NAME'                , 'Identifies the Region Name.  The display of the region name is controlled by the Region Template substitution string TITLE.',
      'supplemental_text'          , 'Identifies text at the top of all download formats.',
      'component_template_options' , 'Identifies template options to be applied to the component template',
      'component_signature'        , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'SOURCE_TYPE'                , 'Type of report, form, or grid from which the region is obtained.',
      'BODY_BACKGROUND_COLOR'      , 'Background color of the region body.',
      'BODY_FONT_COLOR'            , 'Font color of the region body.',
      'BODY_FONT_FAMILY'           , 'Font family used in the region body.',
      'BODY_FONT_SIZE'             , 'Size of the font used in the region body.',
      'BODY_FONT_WEIGHT'           , 'Weight of the font used in the region body.',
      'BORDER_COLOR'               , 'Color of the region border.',
      'BORDER_WIDTH'               , 'Width of the region border.',
      'FORMAT'                     , 'Identifies the output format of the report',
      'FORMAT_ITEM'                , 'Identifies the page item that can be used to define the output format string',
      'PAGE_HEADER'                , 'Text to appear in the header section of the report.',
      'PAGE_HEADER_ALIGNMENT'      , 'Alignment of the page header.',
      'PAGE_HEADER_FONT_COLOR'     , 'Color used for the font in the header.',
      'PAGE_HEADER_FONT_FAMILY'    , 'Font typeface used in the report header.',
      'PAGE_HEADER_FONT_SIZE'      , 'Size of the font used in the page header.',
      'PAGE_HEADER_FONT_WEIGHT'    , 'Weight to use for the font in the page header.',
      'HEADER_BACKGROUND_COLOR'    , 'Background color for column headings.',
      'HEADER_FONT_COLOR'          , 'Font color used in column headings.',
      'HEADER_FONT_FAMILY'         , 'Font typeface for use in column heaings.',
      'HEADER_FONT_SIZE'           , 'Size of font used in column headings.',
      'HEADER_FONT_WEIGHT'         , 'Weight of the font in column headings',
      'HEIGHT'                     , 'Height of the page in the defined uints.',
      'ORIENTATION'                , 'Orientation of the page. Either landscape or portrait.',
      'OUTPUT'                     , 'Type of output to create.',
      'PRINT_SERVER_OVERRIDE'      , 'Obsolete.',
      'OUTPUT_FILE_NAME'           , 'File name to save the output document to.',
      'CONTENT_DISPOSITION'        , 'Whether the output should be displayed inline or downloaded.',
      'DOCUMENT_HEADER'            , 'Obsolete.',
      'OUTPUT_LINK_TEXT'           , 'Text to display for the output or download link.',
      'SHOW_OUTPUT_LINK'           , 'Whether or not to display the output link.',
      'FOOTER'                     , 'Text to display in the document footer.',
      'FOOTER_ALIGNMENT'           , 'Alignment of the text in the document footer.',
      'FOOTER_FONT_COLOR'          , 'Color of text in the page footer.',
      'FOOTER_FONT_FAMILY'         , 'Fonr typeface to use in the page footer.',
      'FOOTER_FONT_SIZE'           , 'Size of the font in the footer.',
      'FOOTER_FONT_WEIGHT'         , 'Font weight for text in the document footer.',
      'PAPER_SIZE'                 , 'Output document page size or Custom for non-standard sizes.',
      'TEMPLATE_ID'                , 'Identifier of the template to use for this report.',
      'PAPER_SIZE_UNITS'           , 'Units to use for specifying the page size.',
      'PAPER_SIZE_WIDTH_UNITS'     , 'Units to use for specifying the page width.',
      'PAPER_SIZE_WIDTH'           , 'Width of the page in type defined units.',
      'LAST_UPDATED_BY'            , 'APEX developer who made last update.',
      'LAST_UPDATED_ON'            , 'Date of last update.',
      'COMPONENT_COMMENT'          , 'Developer comment.',
      'REGION_ID'                  , 'Identifies the Primary Key of the report Region.'
    )
  );
end;
/

create or replace view apex_application_page_rpt_cols
as
select
     f.workspace,
     f.workspace_display_name,
     f.application_id,
     f.application_name,
     p.id                                  page_id,
     p.name                                page_name,
     region.plug_Name                      region_name,
     --
     --r.QUERY_COLUMN_ID                     ,
     --r.FORM_ELEMENT_ID                     ,
     r.COLUMN_ALIAS                        column_alias,
     r.static_id                           static_id,
     case substr(r.display_as, 1, 7)
       when 'NATIVE_' then
           ( select np.display_name from wwv_flow_plugins np where np.security_group_id = 10 and np.flow_id = 4411 and np.plugin_type = 'REPORT COLUMN TYPE' and np.name = substr(r.display_as, 8) )
       when 'PLUGIN_' then
           ( select fp.display_name from wwv_flow_plugins fp where fp.security_group_id = r.security_group_id and fp.flow_id = p.flow_id and fp.plugin_type = 'REPORT COLUMN TYPE' and fp.name = substr(r.display_as, 8) )
       else
           case r.display_as
             when 'WITHOUT_MODIFICATION'   then 'Standard Report Column'
             when 'TEXT_FROM_LOV'          then 'Display as Text (based on LOV, does not save state)'
             when 'DISPLAY_AND_SAVE'       then 'Display as Text (saves state)'
             when 'ESCAPE_SC'              then 'Display as Text (escape special characters, does not save state)'
             when 'DATE_POPUP'             then 'Date Picker'
             when 'TEXT'                   then 'Text Field'
             when 'TEXTAREA'               then 'Text Area'
             when 'SELECT_LIST'            then 'Select List (static LOV)'
             when 'SELECT_LIST_FROM_LOV'   then 'Select List (named LOV)'
             when 'SELECT_LIST_FROM_QUERY' then 'Select List (query based LOV)'
             when 'HIDDEN'                 then 'Hidden'
             when 'POPUP'                  then 'Popup LOV (named LOV)'
             when 'POPUP_QUERY'            then 'Popup LOV (query based LOV)'
             when 'RICH_TEXT'              then 'Rich Text'
             else r.display_as
         end
     end                                   display_as,
     r.display_as                          display_as_code,
     r.COLUMN_DISPLAY_SEQUENCE             display_sequence,
     r.COLUMN_HEADING                      heading,
     r.USE_AS_ROW_HEADER                   use_as_row_header,
     r.COLUMN_FORMAT                       format_mask,
     r.COLUMN_HTML_EXPRESSION              html_expression,
     r.COLUMN_CSS_CLASS                    css_class,
     r.COLUMN_CSS_STYLE                    css_style,
     r.COLUMN_HIT_HIGHLIGHT                highlight_words,
     --
     r.COLUMN_LINK                         column_link_url,
     r.COLUMN_LINKTEXT                     column_link_text,
     r.COLUMN_LINK_ATTR                    column_link_attributes,
     --
     r.COLUMN_LINK_CHECKSUM_TYPE           page_checksum,
     --
     r.COLUMN_ALIGNMENT                    column_alignment,
     r.HEADING_ALIGNMENT                   heading_alignment,
     --
     r.DEFAULT_SORT_COLUMN_SEQUENCE        default_sort_sequence,
     r.DEFAULT_SORT_DIR                    default_sort_direction,
     decode(r.DISABLE_SORT_COLUMN,
       'Y','Yes','N','No',
       r.DISABLE_SORT_COLUMN)              sortable_column,
     decode(r.SUM_COLUMN,
       'Y','Yes','N','No',
       r.SUM_COLUMN)                       sum_column,
     decode(r.HIDDEN_COLUMN ,
       'Y','Yes','N','No',
       r.HIDDEN_COLUMN )                   column_is_hidden,
     --
     r.DISPLAY_WHEN_COND_TYPE              condition_type,
     r.display_when_cond_type              condition_type_code,
     r.DISPLAY_WHEN_CONDITION              condition_expression1,
     r.DISPLAY_WHEN_CONDITION2             condition_expression2,
     --
     decode((select lv.lov_name
       from wwv_flow_lists_of_values$ lv
      where lv.id                = r.NAMED_LOV
        and lv.security_group_id = r.security_group_id),null,null,
      (select lv.lov_name
       from wwv_flow_lists_of_values$ lv
      where lv.id                = r.NAMED_LOV
        and lv.security_group_id = r.security_group_id))             named_list_of_values,
     r.INLINE_LOV                          inline_list_of_values,
     r.lov_language                        lov_language,
     decode(r.LOV_SHOW_NULLS,
       'YES','Yes','NO','No',
       r.LOV_SHOW_NULLS)                   LOV_SHOW_NULLS,
     decode(r.LOV_DISPLAY_EXTRA,
       'YES','Yes','NO','No',
       r.LOV_DISPLAY_EXTRA)                LOV_DISPLAY_EXTRA_VALUES,
     r.LOV_NULL_TEXT                       lov_null_text,
     r.LOV_NULL_VALUE                      lov_null_value,
     --
     r.COLUMN_WIDTH                        form_element_width,
     r.COLUMN_HEIGHT                       form_element_height,
     r.css_classes                         form_element_css_classes,
     r.CATTRIBUTES                         form_Element_Attributes,
     r.CATTRIBUTES_ELEMENT                 form_Element_Option_Attributes,
     (select ft.template_name
       from  wwv_flow_field_templates ft
      where  ft.id                = to_char(r.COLUMN_FIELD_TEMPLATE)
        and  ft.flow_id           = f.application_id
        and  ft.security_group_id = r.security_group_id)               column_field_template,
     r.COLUMN_FIELD_TEMPLATE               column_field_template_id,
     case
       when r.is_required = 'Y' then 'Yes'
       else 'No'
     end                                   is_required,
     nvl(r.standard_validations, 'Application Level Setting') standard_validations,
     --
     r.attribute_01,
     r.attribute_02,
     r.attribute_03,
     r.attribute_04,
     r.attribute_05,
     r.attribute_06,
     r.attribute_07,
     r.attribute_08,
     r.attribute_09,
     r.attribute_10,
     r.attribute_11,
     r.attribute_12,
     r.attribute_13,
     r.attribute_14,
     r.attribute_15,
     --
     --r.COLUMN_COMMENT                      ,
     r.PK_COL_SOURCE_TYPE                  primary_key_column_source_type,
     r.PK_COL_SOURCE                       primary_key_column_source,
     decode(r.DERIVED_COLUMN,
       'Y','Yes',
        r.DERIVED_COLUMN)                  derived_column,
     --
     r.COLUMN_DEFAULT                      column_default,
     r.COLUMN_DEFAULT_TYPE                 column_default_type,
     --
     r.REF_SCHEMA                          reference_schema,
     r.REF_TABLE_NAME                      reference_table_name,
     r.REF_COLUMN_NAME                     reference_column_name,
     --
     decode(substr(r.REPORT_COLUMN_REQUIRED_ROLE,1,1),'!','Not ')||
     nvl((select ss.name
      from  wwv_flow_security_schemes ss
     where  to_char(ss.id)       = ltrim(r.REPORT_COLUMN_REQUIRED_ROLE,'!')
       and  ss.flow_id           = f.application_id
       and  ss.security_group_id = r.security_group_id),
            r.REPORT_COLUMN_REQUIRED_ROLE)
                                           authorization_scheme,
     r.REPORT_COLUMN_REQUIRED_ROLE         authorization_scheme_id,
     --
     decode(include_in_export,'Y','Yes','N','No') include_in_export,
     print_col_width                       print_column_width,
     print_col_align                       print_column_alignment,
     report_column_width                   report_column_width,
     --
     (select case when r.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
       from wwv_flow_patches pa
      where pa.id                = abs(r.REQUIRED_PATCH)
        and pa.security_group_id = r.security_group_id)     build_option,
     r.required_patch                      build_option_id,
     COLUMN_COMMENT                        column_comment,
     --
     r.LAST_UPDATED_BY                     last_updated_by,
     r.LAST_UPDATED_ON                     last_updated_on,
     --
     region.id                             region_id,
     r.id                                  region_report_column_id,
     --
     substr(r.COLUMN_ALIAS,1,30)
     ||' s='||r.COLUMN_DISPLAY_SEQUENCE
     ||' h='||substr(r.COLUMN_HEADING,1,15)||length(r.COLUMN_HEADING)
     ||substr(r.COLUMN_FORMAT,1,15)||length(r.COLUMN_FORMAT)
     ||' e='||substr(r.COLUMN_HTML_EXPRESSION,1,15)||length(r.COLUMN_HTML_EXPRESSION)
     ||substr(r.COLUMN_CSS_CLASS,1,20)
     ||substr(r.COLUMN_CSS_STYLE,1,20)
     ||substr(r.COLUMN_HIT_HIGHLIGHT,1,15)||length(r.COLUMN_HIT_HIGHLIGHT)
     ||' l='||substr(r.COLUMN_LINK,1,20)||length(r.COLUMN_LINK)
     ||substr(r.COLUMN_LINKTEXT,1,15)||length(r.COLUMN_LINKTEXT)
     ||substr(r.COLUMN_LINK_ATTR,1,15)||length(r.COLUMN_LINK_ATTR)
     ||substr(r.COLUMN_LINK_CHECKSUM_TYPE,1,15)
     ||' a='||substr(r.COLUMN_ALIGNMENT,1,6)||substr(r.HEADING_ALIGNMENT,1,6)
     ||' s='||DEFAULT_SORT_COLUMN_SEQUENCE
     ||substr(DEFAULT_SORT_DIR,1,6)
     ||substr(r.DISABLE_SORT_COLUMN,1,10)
     ||substr(r.SUM_COLUMN,1,10)
     ||substr(r.HIDDEN_COLUMN,1,15)
     ||' c='||substr(r.DISPLAY_WHEN_COND_TYPE,1,20)
      ||substr(r.DISPLAY_WHEN_CONDITION,1,15)||length(r.DISPLAY_WHEN_CONDITION)
     ||substr(r.DISPLAY_WHEN_CONDITION2,1,15)||length(r.DISPLAY_WHEN_CONDITION2)
     ||' a='||substr(decode(substr(r.REPORT_COLUMN_REQUIRED_ROLE,1,1),'!','Not ')||
     nvl((select ss.name
      from  wwv_flow_security_schemes ss
     where  to_char(ss.id)       = ltrim(r.REPORT_COLUMN_REQUIRED_ROLE,'!')
       and  ss.flow_id           = f.application_id
       and  ss.security_group_id = r.security_group_id),
      'Not found'),1,30)
     ||' d='||substr(r.DISPLAY_AS,1,20)
     ||' l='||substr(decode((select lv.lov_name
       from wwv_flow_lists_of_values$ lv
      where lv.id                = r.NAMED_LOV
        and lv.security_group_id = r.security_group_id),null,null,
      (select lv.lov_name
       from wwv_flow_lists_of_values$ lv
      where lv.id = r.NAMED_LOV
        and lv.security_group_id = r.security_group_id)),1,30)
     ||substr(r.INLINE_LOV,1,30)||substr(r.LOV_SHOW_NULLS,1,20)||substr(r.LOV_DISPLAY_EXTRA,1,6)||substr(r.LOV_NULL_VALUE,1,20)||r.COLUMN_WIDTH||r.COLUMN_HEIGHT
     ||' c='||substr(r.CATTRIBUTES,1,20)||length(r.CATTRIBUTES)
     ||' c='||substr(r.CATTRIBUTES_ELEMENT,1,20)||length(r.CATTRIBUTES_ELEMENT)
     ||' pk='||r.PK_COL_SOURCE_TYPE||sys.dbms_lob.substr(r.PK_COL_SOURCE,20,1)||sys.dbms_lob.getlength(r.PK_COL_SOURCE)
     ||' d='||substr(r.DERIVED_COLUMN,1,6)
     ||' d='||substr(r.COLUMN_DEFAULT,1,20)||length(r.COLUMN_DEFAULT)
     ||' t='||r.COLUMN_DEFAULT_TYPE||include_in_export||print_col_width||print_col_align
     ||' bo='||(select pa.PATCH_NAME from wwv_flow_patches pa where pa.id = abs(r.REQUIRED_PATCH) and pa.security_group_id = r.security_group_id)
     component_signature
from wwv_flow_region_report_column r,
     wwv_flow_page_plugs           region,
     wwv_flow_steps                p,
     wwv_flow_authorized           f
where f.workspace_id           =  p.security_group_id
  and f.application_id         =  p.flow_id
  and p.security_group_id      =  region.security_group_id
  and p.flow_id                =  region.flow_id
  and p.id                     =  region.page_id
  and region.security_group_id =  r.security_group_id
  and region.id                =  r.region_id
  and region.plug_source_type  != 'NATIVE_IG'
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_rpt_cols',
    p_table_comment     => 'Report column definitions used for Classic Reports, Tabular Forms and Interactive Reports',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                        , 'A work area mapped to one or more database schemas',
      'workspace_display_name'           , 'Display name for the workspace',
      'application_id'                   , 'Application Primary Key, Unique over all workspaces',
      'application_name'                 , 'Identifies the application',
      'page_id'                          , 'Identifies the application',
      'page_name'                        , 'Identifies a page within an application',
      'REGION_NAME'                      , 'Report region name',
      'COLUMN_ALIAS'                     , 'SQL query column alias',
      'STATIC_ID'                        , 'Identifies the Static ID of the column',
      'DISPLAY_SEQUENCE'                 , 'Identifies the sequence in the report that this column is to be displayed',
      'HEADING'                          , 'Report column heading',
      'USE_AS_ROW_HEADER'                , 'Value Identifies Row: Identifies if the current column value should be used as the row header',
      'FORMAT_MASK'                      , 'Number or Date format mask',
      'HTML_EXPRESSION'                  , 'HTML column template used to display this column',
      'CSS_CLASS'                        , 'Use this CSS class in the HTML TD tag when displaying this report column',
      'CSS_STYLE'                        , 'Use this CSS style in the HTML TD tag when displaying this report column',
      'HIGHLIGHT_WORDS'                  , 'Identify keywords to highlight, for example "&P1_SEARCH."',
      'COLUMN_LINK_URL'                  , 'URL target of report column',
      'COLUMN_LINK_TEXT'                 , 'Text displayed for linked columns',
      'COLUMN_LINK_ATTRIBUTES'           , 'HTML "A" tag attributes',
      'PAGE_CHECKSUM'                    , 'An appropriate checksum when linking to protected pages',
      'COLUMN_ALIGNMENT'                 , 'Report column alignment',
      'HEADING_ALIGNMENT'                , 'Report heading alignment',
      'DEFAULT_SORT_SEQUENCE'            , 'For reports with column heading sorting, identifies the default sort order',
      'DEFAULT_SORT_DIRECTION'           , 'Default sort direction, ascending or descending',
      'SORTABLE_COLUMN'                  , 'Identifies if the column is column heading sortable',
      'SUM_COLUMN'                       , 'Identifies if this column is to be summed',
      'COLUMN_IS_HIDDEN'                 , 'Identifies the column as hidden, the values will be returned to the browser but they will not be displayed.',
      'CONDITION_TYPE'                   , 'Identifies the condition type used to conditionally display this Report Column',
      'CONDITION_TYPE_CODE'              , 'Identifies the internal code of CONDITION_TYPE',
      'CONDITION_EXPRESSION1'            , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'            , 'Specifies an expression based on the specific condition type selected.',
      'AUTHORIZATION_SCHEME'             , 'An authorization scheme must evaluate to TRUE in order for this component to be displayed',
      'AUTHORIZATION_SCHEME_ID'          , 'Foreign Key',
      'LAST_UPDATED_BY'                  , 'APEX User Name that last updated this report column',
      'LAST_UPDATED_ON'                  , 'Date of last update',
      'DISPLAY_AS'                       , 'Identifies how the report column is to be displayed',
      'display_as_code'                  , 'Internal code of DISPLAY_AS',
      'report_column_width'              , 'Width of a report column in pixels (px).  Requires #COLUMN_WIDTH# substitution in report template.',
      'NAMEd_LIST_OF_VALUES'             , 'Identifies the Shared List of Values to be used to display this report column',
      'INLINE_LIST_OF_VALUES'            , 'Identifies an inline List of Values to display this column value',
      'lov_language'                     , 'Identifies the programming language of INLINE_LIST_OF_VALUES.',
      'LOV_SHOW_NULLS'                   , 'For column "Display As" ',
      'LOV_DISPLAY_EXTRA_VALUES'         , 'Identifies if the column value is to be displayed if the List of Values domain does not include the column value.',
      'LOV_NULL_TEXT'                    , 'Identifies the text to be displayed for a null value',
      'LOV_NULL_VALUE'                   , 'Identifies the text to be returned for a null value',
      'FORM_ELEMENT_WIDTH'               , 'For form elements, identifies the element width',
      'FORM_ELEMENT_HEIGHT'              , 'For form elements, identifies the element height',
      'FORM_ELEMENT_ATTRIBUTES'          , 'Identifies HTML attributes for the HTML form element',
      'FORM_ELEMENT_OPTION_ATTRIBUTES'   , 'Identifies HTML attributes for the HTML form element options',
      'COLUMN_FIELD_TEMPLATE'            , 'Identifies the template used to display the column label',
      'COLUMN_FIELD_TEMPLATE_ID'         , 'Identifies the template ID foreign key',
      'is_required'                      , 'If set to Yes, APEX will automatically perform a not null validation during page submit.',
      'standard_validations'             , 'Identifies where APEX will perform the built-in standard validations for the report column type.',
      'ATTRIBUTE_01'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_02'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_03'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_04'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_05'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_06'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_07'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_08'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_09'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_10'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_11'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_12'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_13'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_14'                     , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_15'                     , 'Dynamic attribute to store additional data',
      'PRIMARY_KEY_COLUMN_SOURCE_TYPE'   , 'Identifies the datatype of an updatable reports primary key.',
      'PRIMARY_KEY_COLUMN_SOURCE'        , 'Identifies the source value for an updatable reports primary key.',
      'DERIVED_COLUMN'                   , 'Column is generated by the reporting engine and is not derived from the SQL query',
      'COLUMN_DEFAULT'                   , 'Identifies the default value source for this updatable report column',
      'COLUMN_DEFAULT_TYPE'              , 'Identifies the default value source type for an updatable report column',
      'REGION_ID'                        , 'Identifies the Primary Key of the report Region',
      'REGION_REPORT_COLUMN_ID'          , 'Identifies the Primary Key of the Report Column Entry',
      'include_in_export'                , 'Include column in download',
      'print_column_width'               , 'Print column width for exact control',
      'print_column_alignment'           , 'Print column alignment',
      'COLUMN_COMMENT'                   , 'Comment on Report Column',
      'reference_schema'                 , 'Referenced column',
      'reference_table_name'             , 'Referenced column',
      'reference_column_name'            , 'Referenced column',
      'build_option'                     , 'Column will be displayed if the Build Option is enabled',
      'build_option_id'                  , 'Foreign Key',
      'component_signature'              , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'FORM_ELEMENT_CSS_CLASSES'         , 'CSS classes to apply to this report column.'
    )
  );
end;
/

create or replace view apex_application_page_reg_cols
as
select c.id                        as region_column_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       c.page_id,
       p.name                      as page_name,
       c.region_id,
       r.plug_name                 as region_name,
       c.name,
       c.data_type,
       c.item_type,
       case c.is_visible
         when 'Y' then 'Yes'
         else          'No'
       end as is_visible,
       c.display_sequence,
       c.heading,
       c.heading_alignment,
       c.value_alignment,
       c.value_css_classes,
       c.value_attributes,
       c.format_mask,
       c.attribute_01,
       c.attribute_02,
       c.attribute_03,
       c.attribute_04,
       c.attribute_05,
       c.attribute_06,
       c.attribute_07,
       c.attribute_08,
       c.attribute_09,
       c.attribute_10,
       c.attribute_11,
       c.attribute_12,
       c.attribute_13,
       c.attribute_14,
       c.attribute_15,
       c.attribute_16,
       c.attribute_17,
       c.attribute_18,
       c.attribute_19,
       c.attribute_20,
       c.attribute_21,
       c.attribute_22,
       c.attribute_23,
       c.attribute_24,
       c.attribute_25,
       case c.escape_on_http_output
         when 'Y' then 'Yes'
         when 'N' then 'No'
         else c.escape_on_http_output
       end as escape_on_http_output,
       coalesce (
           ( select r
               from apex_standard_conditions
              where c.display_condition_type is not null
                and d = c.display_condition_type
           ),
           c.display_condition_type ) as condition_type,
       c.display_condition_type       as condition_type_code,
       c.display_condition            as condition_expression1,
       c.display_condition2           as condition_expression2,
       case when substr( c.security_scheme, 1, 1) = '!' then 'Not ' end ||
       coalesce (
           ( select ss.name
               from wwv_flow_security_schemes ss
              where to_char( ss.id )     = ltrim( c.security_scheme, '!' )
                and ss.flow_id           = c.flow_id
                and ss.security_group_id = r.security_group_id
           ),
           c.security_scheme ) as authorization_scheme,
       c.security_scheme       as authorization_scheme_id,
       c.column_comment        as component_comment,
       c.last_updated_by,
       c.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_region_columns c,
       wwv_flow_page_plugs r,
       wwv_flow_steps p
 where c.flow_id           = f.application_id
   and c.security_group_id = f.workspace_id
   and r.id                = c.region_id
   and r.security_group_id = c.security_group_id
   and p.flow_id           = c.flow_id
   and p.id                = c.page_id
   and p.security_group_id = c.security_group_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_reg_cols',
    p_table_comment     => 'Region column definitions used for regions',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'               , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'          , 'Application Primary Key, Unique over all workspaces',
      'application_name'        , 'Identifies the application',
      'page_id'                 , 'Identifies the application',
      'page_name'               , 'Identifies a page within an application',
      'region_name'             , 'Region name',
      'name'                    , 'SQL query column name',
      'data_type'               , 'Data type of the SQL query column',
      'item_type'               , 'Item type of the SQL query column',
      'is_visible'              , 'Identifies the column as visible',
      'display_sequence'        , 'Identifies the sequence in the region that this column is to be displayed',
      'heading'                 , 'Region column heading',
      'heading_alignment'       , 'Region heading alignment',
      'value_alignment'         , 'Region column alignment',
      'value_css_classes'       , 'CSS classes that are applied for the region column',
      'value_attributes'        , 'HTML attributes that are applied for the region column',
      'format_mask'             , 'Number or Date format mask',
      'attribute_01'            , 'Dynamic attribute to store additional data',
      'attribute_02'            , 'Dynamic attribute to store additional data',
      'attribute_03'            , 'Dynamic attribute to store additional data',
      'attribute_04'            , 'Dynamic attribute to store additional data',
      'attribute_05'            , 'Dynamic attribute to store additional data',
      'attribute_06'            , 'Dynamic attribute to store additional data',
      'attribute_07'            , 'Dynamic attribute to store additional data',
      'attribute_08'            , 'Dynamic attribute to store additional data',
      'attribute_09'            , 'Dynamic attribute to store additional data',
      'attribute_10'            , 'Dynamic attribute to store additional data',
      'attribute_11'            , 'Dynamic attribute to store additional data',
      'attribute_12'            , 'Dynamic attribute to store additional data',
      'attribute_13'            , 'Dynamic attribute to store additional data',
      'attribute_14'            , 'Dynamic attribute to store additional data',
      'attribute_15'            , 'Dynamic attribute to store additional data',
      'attribute_16'            , 'Dynamic attribute to store additional data',
      'attribute_17'            , 'Dynamic attribute to store additional data',
      'attribute_18'            , 'Dynamic attribute to store additional data',
      'attribute_19'            , 'Dynamic attribute to store additional data',
      'attribute_20'            , 'Dynamic attribute to store additional data',
      'attribute_21'            , 'Dynamic attribute to store additional data',
      'attribute_22'            , 'Dynamic attribute to store additional data',
      'attribute_23'            , 'Dynamic attribute to store additional data',
      'attribute_24'            , 'Dynamic attribute to store additional data',
      'attribute_25'            , 'Dynamic attribute to store additional data',
      'escape_on_http_output'   , 'If Yes values will be escaped when they are written to the HTTP output.',
      'condition_type'          , 'Identifies the condition type used to conditionally display this Region Column',
      'condition_type_code'     , 'Identifies the internal code of CONDITION_TYPE',
      'condition_expression1'   , 'Specifies an expression based on the specific condition type selected.',
      'condition_expression2'   , 'Specifies an expression based on the specific condition type selected.',
      'authorization_scheme'    , 'An authorization scheme must evaluate to TRUE in order for this Region Column to be displayed',
      'authorization_scheme_id' , 'Foreign Key',
      'last_updated_by'         , 'APEX developer who made last update',
      'last_updated_on'         , 'Date of last update',
      'component_comment'       , 'Developer comment',
      'region_column_id'        , 'Identifies the Primary Key of the region column',
      'region_id'               , 'Identifies the Primary Key of the region'
    )
  );
end;
/
-------------------------------------------------------
-- A P P L I C A T I O N   P A G E   P R O C E S S E S
-- 4312

prompt ...apex_application_page_proc

create or replace view apex_application_page_proc
as
select
     f.workspace,
     f.workspace_display_name,
     f.application_id,
     f.application_name,
     p.id                           page_id,
     p.name                         page_name,
     --
     pr.process_name                process_name,
     pr.process_sequence            execution_sequence,
     --
     decode(pr.process_point,
       'RETURN_VALUE','DISPLAY_VALUE',
       'ON_NEW_INSTANCE','On New Instance After Authentication',
       'BEFORE_HEADER','On Load - Before Header',
       'AFTER_HEADER','On Load - After Header',
       'BEFORE_BOX_BODY','On Load - Before Regions',
       'AFTER_BOX_BODY','On Load - After Regions',
       'BEFORE_FOOTER','On Load - Before Footer',
       'AFTER_FOOTER','On Load - After Footer',
       'ON_SUBMIT_BEFORE_COMPUTATION','On Submit - Before Computations and Validations ',
       'AFTER_SUBMIT','On Submit - After Computations and Validations',
       'ON_DEMAND','Ajax Callback - Run this process when requested by Ajax',
       'AFTER_ERROR_HEADER','On Error - After Header',
       'BEFORE_ERROR_FOOTER','On Error - Before Footer',
       pr.process_point)            process_point,
     case pr.process_point
       when 'ON_NEW_INSTANCE' then 'AFTER_AUTHENTICATION' /*backwards compatibility*/
       else pr.process_point
     end process_point_code,
     pr.region_id,
     ( select pp.plug_name
         from wwv_flow_page_plugs pp
        where pp.id                = pr.region_id
          and pp.security_group_id = pr.security_group_id)   region_name,
     --
     decode( pr.location,
        'LOCAL',         'Local Database',
        'REMOTE',        'Remote Database',
        'REGION_SOURCE', 'Region Source',
        'WEB_SOURCE',    'Web Source' )     location,
     --
     (select rs.name
        from wwv_remote_servers rs
       where rs.id                = pr.remote_server_id
         and rs.security_group_id = pr.security_group_id)    remote_database_name,
     pr.remote_server_id                  remote_database_id,
     (select wm.name
        from wwv_flow_web_src_modules wm
       where wm.id                = pr.web_src_module_id
         and wm.security_group_id = pr.security_group_id)   web_source_module_name,
     pr.web_src_module_id                 web_source_module_id,
     pr.web_src_operation_id              web_source_operation_id,
     --
     case pr.process_type
     when 'NATIVE_PLSQL'               then 'PL/SQL anonymous block'
     when 'NATIVE_CLOSE_WINDOW'        then 'Close popup window'
     when 'NATIVE_SESSION_STATE'       then case pr.attribute_01
                                            when 'CLEAR_CACHE_FOR_FLOWS' then 'Clear Cache For Applications (removes all session state for listed applications)'
                                            when 'CLEAR_CACHE_CURRENT_FLOW' then 'Clear Cache For Current Application (removes all session state for current application)'
                                            when 'RESET_SESSION_STATE'      then 'Clear Cache For Current Session (removes all state for current session)'
                                            when 'CLEAR_CACHE_FOR_ITEMS'    then 'Clear Cache for Items (ITEM,ITEM,ITEM)'
                                            when 'CLEAR_CACHE_FOR_PAGES'    then 'Clear Cache for all Items on Pages (PageID,PageID,PageID)'
                                            when 'CLEAR_CACHE_CURRENT_PAGE' then 'Clear Cache for all Items on Pages (PageID,PageID,PageID)'
                                            else pr.process_type||'-'||pr.attribute_01 /*fallback*/
                                            end
     when 'NATIVE_USER_PREFERENCES'    then case pr.attribute_01
                                            when 'RESET_USER_PREFERENCES' then 'Reset Preferences (remove all preferences for current user)'
                                            when 'SET_PREFERENCE_TO_ITEM_VALUE' then 'Set Preference to value of Item'
                                            when 'SET_PREFERENCE_TO_ITEM_VALUE_IF_ITEM_NOT_NULL' then 'Set Preference to value of Item if item is not null (PreferenceName:ITEM)'
                                            else pr.process_type||'-'||pr.attribute_01 /*fallback*/
                                            end
     when 'NATIVE_RESET_PAGINATION'    then case pr.attribute_01
                                            when 'ALL_PAGES' then 'Reset Pagination For All Pages'
                                            when 'PAGES'     then 'Reset Pagination For Page(s) (PageID,PageID,PageID)'
                                            when 'THIS_PAGE' then 'Reset Pagination'
                                            else pr.process_type||'-'||pr.attribute_01 /*fallback*/
                                            end
     when 'NATIVE_TABFORM_ADD_ROWS'    then 'Add rows to tabular form'
     when 'NATIVE_TABFORM_UPDATE'      then 'Multi Row Update'
     when 'NATIVE_TABFORM_DELETE'      then 'Multi Row Delete'
     when 'NATIVE_FORM_FETCH'          then 'Automated Row Fetch'
     when 'NATIVE_FORM_PROCESS'        then 'Automatic Row Processing (DML)'
     when 'NATIVE_FORM_PAGINATION'     then 'Get Next or Previous Primary Key Value'
     when 'NATIVE_WEB_SERVICE'         then 'Web Service'
     when 'NATIVE_WEB_SERVICE_LEGACY'  then 'Web Service'
     else case substr(pr.process_type, 1, 7)
          when 'NATIVE_' then
          ( select np.display_name from wwv_flow_plugins np where np.security_group_id = 10 and np.flow_id = 4411 and np.plugin_type = 'PROCESS TYPE' and np.name = substr(pr.process_type, 8) )
          when 'PLUGIN_' then
          ( select fp.display_name from wwv_flow_plugins fp where fp.security_group_id = pr.security_group_id and fp.flow_id = p.flow_id and fp.plugin_type = 'PROCESS TYPE' and fp.name = substr(pr.process_type, 8) )
          end
     end                            process_type,
     case pr.process_type
     when 'NATIVE_PLSQL'               then 'PLSQL'
     when 'NATIVE_CLOSE_WINDOW'        then 'CLOSE_WINDOW'
     when 'NATIVE_SESSION_STATE'       then case pr.attribute_01
                                            when 'CLEAR_CACHE_CURRENT_PAGE' then 'CLEAR_CACHE_FOR_PAGES'
                                            else pr.attribute_01
                                            end
     when 'NATIVE_USER_PREFERENCES'    then pr.attribute_01
     when 'NATIVE_RESET_PAGINATION'    then case pr.attribute_01
                                            when 'ALL_PAGES' then 'INITIALIZE_PAGINATION_FOR_ALL_PAGES'
                                            when 'PAGES'     then 'INITIALIZE_PAGE_PAGINATION'
                                            when 'THIS_PAGE' then 'RESET_PAGINATION'
                                            else pr.process_type||'-'||pr.attribute_01 /*fallback*/
                                            end
     when 'NATIVE_TABFORM_ADD_ROWS'    then 'ADD_ROWS_TO_TABULAR_FORM'
     when 'NATIVE_TABFORM_UPDATE'      then 'MULTI_ROW_UPDATE'
     when 'NATIVE_TABFORM_DELETE'      then 'MULTI_ROW_DELETE'
     when 'NATIVE_FORM_PROCESS_LEGACY' then 'DML_PROCESS_'||pr.attribute_01||'_ROW'
     when 'NATIVE_FORM_FETCH'          then 'DML_FETCH_ROW'
     when 'NATIVE_FORM_PROCESS'        then 'DML_PROCESS_ROW'
     when 'NATIVE_FORM_PAGINATION'     then 'GET_NEXT_OR_PREV_PK'
     when 'NATIVE_WEB_SERVICE'         then 'WEB_SERVICE'
     when 'NATIVE_WEB_SERVICE_LEGACY'  then 'WEB_SERVICE'
     else pr.process_type
     end                            process_type_code,
     pr.process_type                process_type_plugin_name,
     --pr.ITEM_NAME,
     case
       when pr.process_type in ('NATIVE_TABFORM_UPDATE', 'NATIVE_TABFORM_DELETE') then attribute_05
       when pr.process_type in ('NATIVE_FORM_FETCH', 'NATIVE_FORM_PROCESS')       then attribute_08
       when pr.process_type =   'NATIVE_FORM_PROCESS_LEGACY'                      then attribute_06
       when pr.process_type =   'NATIVE_FORM_PAGINATION'                          then attribute_14
     end                            runtime_where_clause,
     case
     when pr.process_type = 'NATIVE_CLOSE_WINDOW' then to_clob('CLOSE_WINDOW')
     when pr.process_type = 'NATIVE_SESSION_STATE' and pr.attribute_01 = 'CLEAR_CACHE_FOR_FLOWS' then to_clob(pr.attribute_02)
     when pr.process_type = 'NATIVE_SESSION_STATE' and pr.attribute_01 in ('CLEAR_CACHE_CURRENT_FLOW','RESET_SESSION_STATE') then to_clob(pr.attribute_01)
     when pr.process_type = 'NATIVE_SESSION_STATE' and pr.attribute_01 = 'CLEAR_CACHE_FOR_ITEMS' then to_clob(pr.attribute_03)
     when pr.process_type = 'NATIVE_SESSION_STATE' and pr.attribute_01 = 'CLEAR_CACHE_FOR_PAGES' then to_clob(pr.attribute_04)
     when pr.process_type = 'NATIVE_SESSION_STATE' and pr.attribute_01 = 'CLEAR_CACHE_CURRENT_PAGE' then to_clob(pr.flow_step_id)
     when pr.process_type = 'NATIVE_USER_PREFERENCES' and pr.attribute_01 = 'RESET_USER_PREFERENCES' then to_clob(pr.attribute_01)
     when pr.process_type = 'NATIVE_USER_PREFERENCES' then to_clob(pr.attribute_02||':'||pr.attribute_03)
     when pr.process_type = 'NATIVE_RESET_PAGINATION' and pr.attribute_01 = 'PAGES' then to_clob(pr.attribute_02)
     when pr.process_type = 'NATIVE_RESET_PAGINATION' and pr.attribute_01 = 'THIS_PAGE' then to_clob('reset_pagination')
     when pr.process_type = 'NATIVE_TABFORM_ADD_ROWS' then to_clob(pr.attribute_01)
     when pr.process_type like 'NATIVE_TABFORM%' then
          to_clob(rtrim(nvl(pr.attribute_01,'#OWNER#')||':'||pr.attribute_02||':'||pr.attribute_03||':'||pr.attribute_04,':'))
     when pr.process_type = 'NATIVE_FORM_FETCH' then
          to_clob(case when pr.attribute_15='D' then null else 'F|' end||
                  rtrim(nvl(pr.attribute_01,'#OWNER#')||':'||pr.attribute_02||':'||pr.attribute_03||':'||pr.attribute_04||
                        ':'||pr.attribute_05||':'||pr.attribute_06||':'||pr.attribute_07,':'))
     when pr.process_type = 'NATIVE_FORM_PROCESS' then
          to_clob(rtrim(nvl(pr.attribute_01,'#OWNER#')||':'||pr.attribute_02||':'||pr.attribute_03||':'||pr.attribute_04||
                        ':'||pr.attribute_05||':'||pr.attribute_06||':'||pr.attribute_07,':')||
                  '|'||replace(pr.attribute_11,':',null))
     when pr.process_type = 'NATIVE_FORM_PAGINATION' then
          to_clob(nvl(pr.attribute_01,'#OWNER#')||':'||pr.attribute_02||':'||pr.attribute_04||':'||pr.attribute_06||
                  ':'||pr.attribute_07||':'||pr.attribute_08||':'||pr.attribute_03||
                  ':'||pr.attribute_09||':'||pr.attribute_10||':'||pr.attribute_05||
                  ':'||pr.attribute_11||':'||pr.attribute_12||':'||pr.attribute_13||
                  ':'||replace(pr.attribute_14, ':', '<cbchoesc>'))
     when pr.process_type = 'NATIVE_WEB_SERVICE' then to_clob(nvl((select ws_id||':'||id
                                                                     from wwv_flow_ws_operations o
                                                                    where o.security_group_id = pr.security_group_id
                                                                      and to_char(o.id)       = pr.attribute_01),
                                                                  pr.attribute_01))
     when pr.process_type = 'NATIVE_WEB_SERVICE_LEGACY' then to_clob(pr.attribute_01)
     else pr.process_sql_clob
     end                            process_source,
     pr.process_clob_language       process_source_language,
     pr.attribute_01,
     pr.attribute_02,
     pr.attribute_03,
     pr.attribute_04,
     pr.attribute_05,
     pr.attribute_06,
     pr.attribute_07,
     pr.attribute_08,
     pr.attribute_09,
     pr.attribute_10,
     pr.attribute_11,
     pr.attribute_12,
     pr.attribute_13,
     pr.attribute_14,
     pr.attribute_15,
     pr.process_error_message       process_error_message,
     pr.error_display_location      error_display_location,
     --
     coalesce(
         (select sb.button_name
           from wwv_flow_step_buttons sb
          where sb.id                = pr.process_when_button_id
            and sb.security_group_id = pr.security_group_id),
         (select si.name
           from wwv_flow_step_items si
          where si.id                = pr.process_when_button_id
            and si.security_group_id = pr.security_group_id) )
                                    when_button_pressed,
     pr.process_when_button_id      when_button_pressed_id,
     case nvl(pr.only_for_changed_rows, 'Y')
       when 'Y' then 'Yes'
       when 'N' then 'No'
     end                            only_for_changed_rows,
     --
     nvl((select r from apex_standard_conditions where d = pr.process_when_type),pr.process_when_type)
                                    condition_type,
     pr.process_when_type           condition_type_code,
     pr.process_when                condition_expression1,
     pr.process_when2               condition_expression2,
     case nvl(pr.exec_cond_for_each_row, 'N')
       when 'Y' then 'Yes'
       when 'N' then 'No'
     end                            exec_cond_for_each_row,
     --pr.PROCESS_WHEN_TYPE2,
     --
     decode(pr.process_is_stateful_y_n,
       'Y','Once Per Session or When Reset',
       'N','Once Per Page Visit (default)',
       pr.PROCESS_IS_STATEFUL_Y_N)  run_process,
     --
     case pr.process_type
       when 'NATIVE_FORM_PROCESS' then attribute_09
     end                            return_key_into_item_1,
     case pr.process_type
       when 'NATIVE_FORM_PROCESS' then attribute_10
     end                            return_key_into_item_2,
     pr.process_success_message     success_message,
     --
     (select case when pr.required_patch > 0 then pa.patch_name else '{Not '||pa.patch_name||'}' end patch_name
      from wwv_flow_patches pa
     where pa.id                = abs(pr.required_patch)
       and pa.security_group_id = pr.security_group_id)    build_option,
     --
     decode(substr(pr.security_scheme,1,1),'!','Not ')||
     nvl((select ss.name
       from  wwv_flow_security_schemes ss
      where  to_char(ss.id)       = ltrim(pr.security_scheme,'!')
        and  ss.flow_id           = f.application_id
        and  ss.security_group_id = pr.security_group_id),
      pr.security_scheme)           authorization_scheme,
     pr.security_scheme             authorization_scheme_id,
     --
     pr.last_updated_by             last_updated_by,
     pr.last_updated_on             last_updated_on,
     pr.process_comment             component_comment,
     pr.id                          process_id,
     --
     lpad(pr.PROCESS_SEQUENCE,5,'00000')
     ||',point='||decode(pr.PROCESS_POINT,
       'RETURN_VALUE','DISPLAY_VALUE',
       'ON_NEW_INSTANCE','On New Instance After Authentication',
       'BEFORE_HEADER','On Load - Before Header',
       'AFTER_HEADER','On Load - After Header',
       'BEFORE_BOX_BODY','On Load - Before Regions',
       'AFTER_BOX_BODY','On Load - After Regions',
       'BEFORE_FOOTER','On Load - Before Footer',
       'AFTER_FOOTER','On Load - After Footer',
       'ON_SUBMIT_BEFORE_COMPUTATION','On Submit - Before Computations and Validations ',
       'AFTER_SUBMIT','On Submit - After Computations and Validations',
       'ON_DEMAND','Ajax Callback - Run this process when requested by Ajax',
       'AFTER_ERROR_HEADER','On Error - After Header',
       'BEFORE_ERROR_FOOTER','On Error - Before Footer',
       pr.PROCESS_POINT)
       ||',type='||pr.PROCESS_TYPE
       ||',src='||sys.dbms_lob.substr(pr.PROCESS_SQL_CLOB,30,1)||'.'||
       sys.dbms_lob.getlength(pr.PROCESS_SQL_CLOB)
       ||coalesce(
              (select ',wbp='||sb.button_name n from wwv_flow_step_buttons sb where sb.id = pr.PROCESS_WHEN_BUTTON_ID and sb.security_group_id = pr.security_group_id),
              (select ',wbp='||si.name n from wwv_flow_step_items si where si.id = pr.PROCESS_WHEN_BUTTON_ID and si.security_group_id = pr.security_group_id))
       ||decode(pr.PROCESS_ERROR_MESSAGE,null,null,',errm='||
       substr(pr.PROCESS_ERROR_MESSAGE,1,20)||length(pr.PROCESS_ERROR_MESSAGE))
       ||decode(pr.PROCESS_SUCCESS_MESSAGE,null,null,',succm='||
       substr(pr.PROCESS_SUCCESS_MESSAGE,1,20)||length(pr.PROCESS_SUCCESS_MESSAGE))
       ||nvl((select ss.name from wwv_flow_security_schemes ss where to_char(ss.id) = ltrim(pr.SECURITY_SCHEME,'!') and ss.flow_id = f.application_id and ss.security_group_id = pr.security_group_id),'Not found')
       ||',cond='||pr.PROCESS_WHEN_TYPE
       ||substr(pr.PROCESS_WHEN,1,20)||length(pr.PROCESS_WHEN)||'.'
       ||substr(pr.PROCESS_WHEN2,1,20)||length(pr.PROCESS_WHEN2)
       ||(select pa.PATCH_NAME from wwv_flow_patches pa where pa.id = abs(pr.REQUIRED_PATCH) and pa.security_group_id = pr.security_group_id)
       ||decode(pr.PROCESS_IS_STATEFUL_Y_N,'Y','Once Per Session or When Reset','N','Once Per Page Visit (default)',pr.PROCESS_IS_STATEFUL_Y_N)
     ||',key='||pr.RETURN_KEY_INTO_ITEM1||pr.RETURN_KEY_INTO_ITEM1
     component_signature
from wwv_flow_step_processing pr,
     wwv_flow_steps           p,
     wwv_flow_authorized      f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = pr.security_group_id
  and p.flow_id           = pr.flow_id
  and p.id                = pr.flow_step_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_proc',
    p_table_comment     => 'Identifies SQL or PL/SQL processing associated with a page',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                , 'A work area mapped to one or more database schemas',
      'workspace_display_name'   , 'Display name for the workspace',
      'application_id'           , 'Application Primary Key, Unique over all workspaces',
      'application_name'         , 'Identifies the application',
      'page_id'                  , 'Identifies the application',
      'page_name'                , 'Identifies a page within an application',
      'process_name'             , 'Identifies Page Process Name',
      'execution_sequence'       , 'Identifies order of execution within each Process Point',
      'process_point'            , 'Specifies the point at which the process is executed',
      'process_point_code'       , 'Internal code of PROCESS_POINT',
      'region_id'                , 'Identifies the region ID of the tabular form region for process',
      'region_name'              , 'Identifies the region name of the tabular form region for process',
      'location'                 , 'Identifies whether the data source of the region is the local database, a REST Enabled SQL service or a REST Data Source',
      'remote_database_id'       , 'Identifies the primary key of the REST Enabled SQL definition',
      'remote_database_name'     , 'Identifies the name of the REST Enabled SQL service',
      'web_source_module_id'     , 'Identifies the primary key of the REST Data Source',
      'web_source_operation_id'  , 'Identifies the primary key of the REST source operation',
      'web_source_module_name'   , 'Identifies the REST Data Source name',
      'process_type'             , 'Identifies process type',
      'process_type_code'        , 'Deprecated internal code of PROCESS_TYPE. Use PROCESS_TYPE_PLUGIN_NAME instead.',
      'process_type_plugin_name' , 'Name of the plugin that implements the process. Names of built-in plugins start with "NATIVE_", application specific plugin names start with "PLUGIN_"',
      'runtime_where_clause'     , 'Appended to Oracle APEX generated SELECT, UPDATE, and DELETE statements',
      'process_source'           , 'Identifies the corresponding process text for the process type',
      'process_source_language'  , 'Identifies the programming language of PROCESS_SOURCE.',
      'attribute_01'             , 'Dynamic attribute to store additional data',
      'attribute_02'             , 'Dynamic attribute to store additional data',
      'attribute_03'             , 'Dynamic attribute to store additional data',
      'attribute_04'             , 'Dynamic attribute to store additional data',
      'attribute_05'             , 'Dynamic attribute to store additional data',
      'attribute_06'             , 'Dynamic attribute to store additional data',
      'attribute_07'             , 'Dynamic attribute to store additional data',
      'attribute_08'             , 'Dynamic attribute to store additional data',
      'attribute_09'             , 'Dynamic attribute to store additional data',
      'attribute_10'             , 'Dynamic attribute to store additional data',
      'attribute_11'             , 'Dynamic attribute to store additional data',
      'attribute_12'             , 'Dynamic attribute to store additional data',
      'attribute_13'             , 'Dynamic attribute to store additional data',
      'attribute_14'             , 'Dynamic attribute to store additional data',
      'attribute_15'             , 'Dynamic attribute to store additional data',
      'process_error_message'    , 'Error message to be displayed when this process raises an exception',
      'error_display_location'   , 'The Error display location identifies where the process error message will display. Messages can be displayed on an error page, or inline with the existing page. Inline errors are displayed in the "notification" area (defined as part of the page template).',
      'success_message'          , 'Identifies the message to be displayed upon successful execution of this Page Process',
      'when_button_pressed'      , 'This process will only be executed if a user clicks the button identified',
      'when_button_pressed_id'   , 'Foreign key to button',
      'only_for_changed_rows'    , 'If set to Yes this process will only be evaluated if the tabular form row has been created or modified.',
      'condition_type'           , 'Identifies the condition type used to conditionally execute the Page Process',
      'condition_type_code'      , 'Internal code of CONDITION_TYPE',
      'condition_expression1'    , 'Specifies an expression based on the specific condition type selected.',
      'condition_expression2'    , 'Specifies an expression based on the specific condition type selected.',
      'exec_cond_for_each_row'   , 'If set to Yes the condition of this process will be executed for each row because it references tabular form columns.',
      'run_process'              , 'Specifies when the process should run; Once Per Session or When Reset, or Once Per Page Visit',
      'return_key_into_item_1'   , 'After performing a SQL INSERT statement, take the first (or only) column of the primary key and return it into this item. Use this feature to get back ID fields populated from sequences within database triggers.',
      'return_key_into_item_2'   , 'After performing a SQL INSERT statement, take the second column of the primary key and return it into this item. Use this feature to get back ID fields populated from sequences within database triggers.',
      'build_option'             , 'Page Process will be considered for execution if the Build Option is enabled',
      'authorization_scheme'     , 'An authorization scheme must evaluate to TRUE in order for this Page Process to be considered for execution',
      'authorization_scheme_id'  , 'Foreign Key',
      'component_comment'        , 'Developer comment',
      'process_id'               , 'Primary Key of this Page Process',
      'last_updated_by'          , 'APEX developer who made last update',
      'last_updated_on'          , 'Date of last update',
      'component_signature'      , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
------------------------------------------------------------
-- A P P L I C A T I O N   P A G E   C O M P U T A T I O N S
-- 4315

prompt ...apex_application_page_comp


create or replace view apex_application_page_comp
as
select
     f.workspace,
     f.workspace_display_name,
     f.application_id,
     f.application_name,
     p.id                           page_id,
     p.name                         page_name,
     --
     c.COMPUTATION_ITEM             item_name,
     c.COMPUTATION_SEQUENCE         execution_sequence,
     --
     decode(c.COMPUTATION_POINT,
       'ON_NEW_INSTANCE','On New Instance (e.g. On Login)',
       'BEFORE_HEADER','Before Header',
       'AFTER_HEADER','After Header',
       'BEFORE_BOX_BODY','Before Region(s)',
       'AFTER_BOX_BODY','After Region(s)',
       'BEFORE_FOOTER','Before Footer',
       'AFTER_FOOTER','After Footer',
       'AFTER_SUBMIT','After Submit',
       c.COMPUTATION_POINT)         computation_point,
     --
     decode(c.COMPUTATION_TYPE,
        'STATIC_ASSIGNMENT','Static Assignment',
        'FUNCTION_BODY','Function Body',
        'QUERY','SQL Query (return single value)',
        'QUERY_COLON','SQL Query (return colon separated value)',
        'EXPRESSION','Expression',
        'ITEM_VALUE','Item Value',
        c.COMPUTATION_TYPE)         computation_type,
     c.computation_language         computation_language,
     --c.COMPUTATION_PROCESSED        computation_processed,
     c.COMPUTATION                  computation,
     --
     decode(c.COMPUTE_WHEN_TYPE,'%'||'null%',null,
     nvl((select r from apex_standard_conditions where d = c.COMPUTE_WHEN_TYPE),c.COMPUTE_WHEN_TYPE))
                                    condition_type,
     c.compute_when_type            condition_type_code,
     c.COMPUTE_WHEN                 condition_expression1,
     c.COMPUTE_WHEN_TEXT            condition_expression2,
     c.COMPUTATION_ERROR_MESSAGE    error_message,
     (select case when c.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(c.REQUIRED_PATCH)
       and pa.security_group_id = c.security_group_id
)    build_option,
     --
     decode(substr(c.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(c.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = c.security_group_id),
     c.SECURITY_SCHEME)             authorization_scheme,
     c.SECURITY_SCHEME              authorization_scheme_id,
     c.LAST_UPDATED_BY              last_updated_by,
     c.LAST_UPDATED_ON              last_updated_on,
     c.COMPUTATION_COMMENT          component_comment,
     --
     c.COMPUTATION_ITEM
     ||c.COMPUTATION_SEQUENCE
     ||c.COMPUTATION_POINT
     ||c.COMPUTATION_TYPE
     ||length(c.COMPUTATION)
     ||c.COMPUTE_WHEN_TYPE
     ||length(c.COMPUTE_WHEN)||length(c.COMPUTE_WHEN_TEXT)
     ||length(c.COMPUTATION_ERROR_MESSAGE)
     ||(select case when c.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(c.REQUIRED_PATCH)
       and pa.security_group_id = c.security_group_id)
     ||decode(substr(c.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(c.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = c.security_group_id),
     'Not found')             component_signature,
     --
     c.id                           computation_id
from wwv_flow_step_computations c,
     wwv_flow_steps             p,
     wwv_flow_authorized        f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = c.security_group_id
  and p.flow_id           = c.flow_id
  and p.id                = c.flow_step_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_comp',
    p_table_comment     => 'Identifies the computation of Item Session State',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'             , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'        , 'Application Primary Key, Unique over all workspaces',
      'application_name'      , 'Identifies the application',
      'page_id'               , 'Identifies the application',
      'page_name'             , 'Identifies a page within an application',
      'item_name'             , 'Item name to compute the value of',
      'EXECUTION_SEQUENCE'    , 'Identifies the execution order of the computations within each computation point',
      'COMPUTATION_POINT'     , 'Identifies the computation point that the computation will be considered for execution',
      'COMPUTATION_TYPE'      , 'Identifies the type of computation, reference the Computation attribute.',
      'computation_language'  , 'Identifies the programming language of COMPUTATION.',
      'COMPUTATION'           , 'Identifies the Computation which corresponds to the Computation Type',
      'CONDITION_TYPE'        , 'Identifies the condition type used to conditionally execute the Page Computation',
      'CONDITION_TYPE_CODE'   , 'Internal code of CONDITION_TYPE',
      'CONDITION_EXPRESSION1' , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2' , 'Specifies an expression based on the specific condition type selected.',
      'ERROR_MESSAGE'         , 'Identifies the error message to display if the Page Computation raises an error',
      'BUILD_OPTION'          , 'Page Computation will be considered for execution if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'  , 'An authorization scheme must evaluate to TRUE in order for this computation to be considered for execution',
      'AUTHORIZATION_SCHEME_ID'  , 'Foriegn Key',
      'COMPONENT_COMMENT'     , 'Developer comment',
      'COMPUTATION_ID'        , 'Primary key of this page computation',
      'last_updated_by'       , 'APEX developer who made last update',
      'last_updated_on'       , 'Date of last update',
      'component_signature'   , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

------------------------------------------------------------
-- A P P L I C A T I O N   P A G E   B R A N C H E S
-- 4315

prompt ...apex_application_page_branches


create or replace view apex_application_page_branches
as
select
     f.workspace,
     f.workspace_display_name,
     f.application_id,
     f.application_name,
     p.id                           page_id,
     p.name                         page_name,
     --
     b.branch_name                  branch_name,
     b.BRANCH_ACTION                branch_action,
     b.BRANCH_POINT                 branch_point,
     decode(b.BRANCH_WHEN_BUTTON_ID,
     null,null,coalesce(
         (select sb.button_name from wwv_flow_step_buttons sb where sb.id = b.BRANCH_WHEN_BUTTON_ID and sb.security_group_id = b.security_group_id),
         (select si.name from wwv_flow_step_items si where si.id = b.BRANCH_WHEN_BUTTON_ID and si.security_group_id = b.security_group_id)))
                                    when_button_pressed,
     decode(b.BRANCH_TYPE,
       'BRANCH_TO_FUNCTION_RETURNING_PAGE','Branch to Function Returning a Page',
       'BRANCH_TO_FUNCTION_RETURNING_URL', 'Branch to Function Returning a URL',
       'BRANCH_TO_PAGE_IDENT_BY_ITEM',     'Branch to Page Identified by Item (Use Item Name)',
       'BRANCH_TO_PAGE_ACCEPT',            'Branch to Page Accept Processing (not common)',
       'BRANCH_TO_STEP',                   'Branch to Page',
       'BRANCH_TO_URL_IDENT_BY_ITEM',      'Branch to URL Identified by Item (Use Item Name)',
       'PLSQL',                            'Branch to PL/SQL Procedure',
       'REDIRECT_URL',                     'Branch to Page or URL',
       b.BRANCH_TYPE)               branch_type,
     b.branch_language              branch_language,
     b.BRANCH_SEQUENCE              process_sequence,
     --b.CLEAR_PAGE_CACHE,
     nvl((select r from apex_standard_conditions where d = b.BRANCH_CONDITION_TYPE),b.BRANCH_CONDITION_TYPE)
                                    condition_type,
     b.branch_condition_type        condition_type_code,
     b.BRANCH_CONDITION             condition_expression1,
     b.BRANCH_CONDITION_TEXT        condition_expression2,
     decode(nvl(b.save_state_before_branch_yn,'N'),'N','No','Yes')  save_state_before_branch,
     (select case when b.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(b.REQUIRED_PATCH)
       and pa.security_group_id = b.security_group_id)   build_option,
     --
     decode(substr(b.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(b.SECURITY_SCHEME,'!')
      and  ss.flow_id           = f.application_id
      and  ss.security_group_id = b.security_group_id),
     b.SECURITY_SCHEME)             authorization_scheme,
     b.SECURITY_SCHEME              authorization_scheme_id,
     b.LAST_UPDATED_BY              last_updated_by,
     b.LAST_UPDATED_ON              last_updated_on,
     b.BRANCH_COMMENT               component_comment,
     b.id                           branch_id,
     b.BRANCH_WHEN_BUTTON_ID        branch_when_button_id,
     --
     'Point='||BRANCH_POINT
     ||',seq='||lpad(b.BRANCH_SEQUENCE,5,'00000')
     ||',act='||substr(b.BRANCH_ACTION,1,30)||'.'||length(b.BRANCH_ACTION)
     ||',type='||decode(b.BRANCH_TYPE,
       'BRANCH_TO_FUNCTION_RETURNING_PAGE','Branch to Function Returning a Page',
       'BRANCH_TO_FUNCTION_RETURNING_URL', 'Branch to Function Returning a URL',
       'BRANCH_TO_PAGE_IDENT_BY_ITEM',     'Branch to Page Identified by Item (Use Item Name)',
       'BRANCH_TO_PAGE_ACCEPT',            'Branch to Page Accept Processing (not common)',
       'BRANCH_TO_STEP',                   'Branch to Page',
       'BRANCH_TO_URL_IDENT_BY_ITEM',      'Branch to URL Identified by Item (Use Item Name)',
       'PLSQL',                            'Branch to PL/SQL Procedure',
       'REDIRECT_URL',                     'Branch to Page or URL',
       b.BRANCH_TYPE)||'.'
     ||decode(b.BRANCH_WHEN_BUTTON_ID,null,null,coalesce((select sb.button_name from wwv_flow_step_buttons sb where sb.id = b.BRANCH_WHEN_BUTTON_ID and sb.security_group_id = b.security_group_id), (select si.name from wwv_flow_step_items si where si.id = b.BRANCH_WHEN_BUTTON_ID and si.security_group_id = b.security_group_id)))||
     decode(substr(b.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name from wwv_flow_security_schemes ss where to_char(ss.id) = ltrim(b.SECURITY_SCHEME,'!') and ss.flow_id = f.application_id and ss.security_group_id = b.security_group_id),'Not found')||
     'cond='||nvl((select r from apex_standard_conditions where d = b.BRANCH_CONDITION_TYPE),b.BRANCH_CONDITION_TYPE)||
        '.'||substr(b.BRANCH_CONDITION,1,20)||'.'||length(b.BRANCH_CONDITION)||'.'||length(b.BRANCH_CONDITION_TEXT)||
     'save='||decode(nvl(b.save_state_before_branch_yn,'N'),'N','No','Yes')||
     'build='||(select pa.PATCH_NAME from wwv_flow_patches pa where pa.id = abs(b.REQUIRED_PATCH) and pa.security_group_id = b.security_group_id)
     component_signature
from wwv_flow_step_branches b,
     wwv_flow_steps         p,
     wwv_flow_authorized    f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = b.security_group_id
  and p.flow_id           = b.flow_id
  and p.id                = b.flow_step_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_branches',
    p_table_comment     => 'Identifies branch processing associated with a page.  A branch is a directive to navigate to a page or URL which is run at the conclusion of page accept processing.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'             , 'A branch fires on page submit and direct a posted page to the next page to view',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'        , 'Application Primary Key, Unique over all workspaces',
      'application_name'      , 'Identifies the application',
      'page_id'               , 'Identifies the application',
      'page_name'             , 'Identifies a page within an application',
      'branch_name'           , 'Identifies branch name',
      'BRANCH_ACTION'         , 'Identifies the branch target, typically a page',
      'BRANCH_POINT'          , 'Identifies the point at which the branch will be evaluated for execution',
      'WHEN_BUTTON_PRESSED'   , 'Identifies a button that must be pressed to consider the branch for execution',
      'BRANCH_TYPE'           , 'Identifies how the branch action will be processed, typically identifies a page ID or URL',
      'branch_language'       , 'Identifies the programming language of BRANCH_ACTION.',
      'PROCESS_SEQUENCE'      , 'Identifies the order in which the branch will be evaluated for execution for each branch point',
      'CONDITION_TYPE'        , 'Identifies a condition that must be met in order for this branch to be processed',
      'CONDITION_TYPE_CODE'   , 'Internal code of CONDITION_TYPE',
      'CONDITION_EXPRESSION1' , 'Identifies a condition which is specific to the selected condition type',
      'CONDITION_EXPRESSION2' , 'Identifies a condition which is specific to the selected condition type',
      'BUILD_OPTION'          , 'Build options enable or disable components',
      'AUTHORIZATION_SCHEME'  , 'An authorization scheme must evaluate to TRUE in order for this component to be processed',
      'AUTHORIZATION_SCHEME_ID'  , 'Foreign Key',
      'COMPONENT_COMMENT'     , 'Developer comment',
      'BRANCH_ID'             , 'Primary key for this component',
      'branch_when_button_id' , 'Foreign Key to Branch When Button ID',
      'save_state_before_branch'   , 'Yes - Save session state and clear cache before branch. No - Redirect to URL with clear-cache and session state settings',
      'last_updated_by'       , 'APEX developer who made last update',
      'last_updated_on'       , 'Date of last update',
      'component_signature'   , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
----------------------------------------------------------
-- A P P L I C A T I O N   P A G E   V A L I D A T I O N S
-- 4316

prompt ...apex_application_page_val

create or replace view apex_application_page_val
as
select
     f.workspace,
     f.workspace_display_name,
     f.application_id,
     f.application_name,
     p.id                          page_id,
     p.name                        page_name,
     --
     v.VALIDATION_NAME             validation_name,
     v.VALIDATION_SEQUENCE         validation_sequence,
     v.TABULAR_FORM_REGION_ID      region_id,
     ( select pp.plug_name
         from wwv_flow_page_plugs pp
        where pp.id                = v.tabular_form_region_id
          and pp.security_group_id = v.security_group_id) region_name,
     --
     decode(v.VALIDATION_TYPE,
        'EXISTS','Exists',
        'FUNC_BODY_RETURNING_BOOLEAN','Function Returning Boolean',
        'FUNC_BODY_RETURNING_ERR_TEXT','Function Returning Error Text',
        'ITEM_NOT_NULL','Item\Column specified is NOT NULL',
        'ITEM_IN_VALIDATION_CONTAINS_AT_LEAST_ONE_CHAR_IN_STRING2','Item\Column in Expression 1 contains at least one of the characters in Expression 2',
        'ITEM_IN_VALIDATION_CONTAINS_ONLY_CHAR_IN_STRING2','Item\Column in Expression 1 contains only characters in Expression 2',
        'ITEM_IN_VALIDATION_NOT_EQ_STRING2','Item\Column in Expression 1 does NOT equal string literal in Expression2',
        'ITEM_IN_VALIDATION_CONTAINS_NO_CHAR_IN_STRING2','Item\Column in Expression 1 does not contain any of the characters in Expression 2',
        'ITEM_IN_VALIDATION_EQ_STRING2','Item\Column in Expression 1 equals string literal in Expression 2',
        'ITEM_IN_VALIDATION_NOT_IN_STRING2','Item\Column in Expression 1 is NOT contained in Expression 2',
        'ITEM_IN_VALIDATION_IN_STRING2','Item\Column in Expression 1 is contained in Expression 2',
        'ITEM_NOT_ZERO','Item\Column specified is NOT zero',
        'ITEM_CONTAINS_NO_SPACES','Item\Column specified contains no spaces',
        'ITEM_NOT_NULL_OR_ZERO','Item\Column specified is NOT NULL or zero',
        'ITEM_IS_ALPHANUMERIC','Item\Column specified is alphanumeric',
        'ITEM_IS_NUMERIC','Item\Column specified is numeric',
        'ITEM_IS_DATE','Item\Column specified is a valid date',
        'ITEM_IS_TIMESTAMP','Item\Column specified is a valid timestamp',
        'REGULAR_EXPRESSION','Item\Column in Expression 1 matches Regular Expression in Expression 2',
        'NOT_EXISTS','NOT Exists',
        'PLSQL_ERROR','PL/SQL Error',
        'EXPRESSION','Expression',
         v.VALIDATION_TYPE)        validation_type,
     v.validation_type             validation_type_code,
     v.VALIDATION                  validation_expression1,
     v.VALIDATION2                 validation_expression2,
     --
     case nvl(v.always_execute, 'N')
       when 'Y' then 'Yes'
       when 'N' then 'No'
     end                           always_execute,
     nvl((select r from apex_standard_conditions where d = v.VALIDATION_CONDITION_TYPE),v.VALIDATION_CONDITION_TYPE)
                                   condition_type,
     v.validation_condition_type   condition_type_code,
     v.VALIDATION_CONDITION        condition_expression1,
     v.VALIDATION_CONDITION2       condition_expression2,
     case nvl(v.exec_cond_for_each_row, 'N')
       when 'Y' then 'Yes'
       when 'N' then 'No'
     end                           exec_cond_for_each_row,
     --
     coalesce(
        (select sb.button_name from wwv_flow_step_buttons sb where sb.id = v.WHEN_BUTTON_PRESSED and sb.security_group_id = v.security_group_id),
        (select si.name from wwv_flow_step_items si where si.id = v.WHEN_BUTTON_PRESSED and si.security_group_id = v.security_group_id) )
                                   when_button_pressed,
     v.WHEN_BUTTON_PRESSED         when_button_pressed_id,
     case nvl(v.only_for_changed_rows, 'Y')
       when 'Y' then 'Yes'
       when 'N' then 'No'
     end                           only_for_changed_rows,
     --
     v.ERROR_MESSAGE               validation_failure_text,
     (select name from wwv_flow_step_items
      where id = v.ASSOCIATED_ITEM and
            flow_id = f.application_id)        associated_item,
     v.associated_column           associated_column,
     v.ERROR_DISPLAY_LOCATION      error_display_location,
     (select case when v.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
     from wwv_flow_patches pa
     where pa.id                = abs(v.REQUIRED_PATCH)
       and pa.security_group_id = v.security_group_id)   build_option,
     --
     decode(substr(v.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id) = ltrim(v.SECURITY_SCHEME,'!')
       and ss.flow_id = f.application_id
       and ss.security_group_id = v.security_group_id),
     v.SECURITY_SCHEME)            authorization_scheme,
     v.SECURITY_SCHEME             authorization_scheme_id,
     --
     v.LAST_UPDATED_BY             last_updated_by,
     v.LAST_UPDATED_ON             last_updated_on,
     v.VALIDATION_COMMENT          component_comment,
     v.id                          validation_id,
     'seq='||v.VALIDATION_SEQUENCE
     ||',item='||(select si.name from wwv_flow_step_items si where si.id = v.ASSOCIATED_ITEM and si.security_group_id = v.security_group_id)
     ||',type='||v.VALIDATION_TYPE
     ||',val='||substr(v.VALIDATION,1,40)||'.'||length(v.VALIDATION)||'.'
      ||substr(v.VALIDATION2,1,40)||'.'||length(v.VALIDATION2)||'.'
     ||decode(v.VALIDATION_CONDITION_TYPE,null,null,',cond='||v.VALIDATION_CONDITION_TYPE||'.'||
         substr(v.VALIDATION_CONDITION,1,30)||length(v.VALIDATION_CONDITION)||'.'||
         substr(v.VALIDATION_CONDITION2,1,30)||length(v.VALIDATION_CONDITION2))
     component_signature
from wwv_flow_step_validations v,
     wwv_flow_steps            p,
     wwv_flow_authorized       f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = v.security_group_id
  and p.flow_id           = v.flow_id
  and p.id                = v.flow_step_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_val',
    p_table_comment     => 'Identifies Validations associated with an Application Page',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'               , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'          , 'Application Primary Key, Unique over all workspaces',
      'application_name'        , 'Identifies the application',
      'page_id'                 , 'Identifies the application',
      'page_name'               , 'Identifies a page within an application',
      'VALIDATION_NAME'         , 'Identifies the name of the validation',
      'VALIDATION_SEQUENCE'     , 'Identifies the sequence in which this validation will be considered for execution',
      'region_id'               , 'Identifies the region ID of the tabular form region for column validations',
      'region_name'             , 'Identifies the region name of the tabular form region for column validations',
      'VALIDATION_TYPE'         , 'Specifies predefined validation type with the corresponding appropriate values in the Expression 1 and Expression 2 fields.',
      'validation_type_code'    , 'Internal code of VALIDATION_TYPE.',
      'VALIDATION_EXPRESSION1'  , 'Identifies the validation which corresponds to the specified Validation Type',
      'VALIDATION_EXPRESSION2'  , 'Identifies the validation which corresponds to the specified Validation Type',
      'always_execute'          , 'If set to Yes this flag will overwrites the "Execute Validations" flag for buttons and always execute the validation.',
      'CONDITION_TYPE'          , 'Identifies the condition type used to conditionally execute the Page Validation',
      'condition_type_code'     , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'   , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'   , 'Specifies an expression based on the specific condition type selected.',
      'exec_cond_for_each_row'  , 'If set to Yes the condition of this validation will be executed for each row because it references tabular form columns.',
      'WHEN_BUTTON_PRESSED'     , 'This validation will only be evaluated if the identified button is pressed',
      'WHEN_BUTTON_PRESSED_ID'  , 'Foreign key to button',
      'only_for_changed_rows'   , 'If set to Yes this validation will only be evaluated if the tabular form row has been created or modified.',
      'VALIDATION_FAILURE_TEXT' , 'Specifies the text that will be displayed in the event the validation raises an error',
      'ASSOCIATED_ITEM'         , 'If applicable, select the item associated with this validation error message.',
      'ASSOCIATED_COLUMN'       , 'If applicable, select the column associated with this validation error message.',
      'ERROR_DISPLAY_LOCATION'  , 'The Error display location identifies where the validation error message will display. Messages can be displayed on an error page, or inline with the existing page. Inline validations are displayed in the "notification" area (defined as part of the page template), and/or within the item label.',
      'BUILD_OPTION'            , 'Page Computation will be considered for execution if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'    , 'An authorization scheme must evaluate to TRUE in order for this validation to be considered for execution',
      'AUTHORIZATION_SCHEME_id' , 'Foreign Key',
      'COMPONENT_COMMENT'       , 'Developer comment',
      'VALIDATION_ID'           , 'Primary key of this page validation',
      'last_updated_by'         , 'APEX developer who made last update',
      'last_updated_on'         , 'Date of last update',
      'component_signature'     , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/


--------------------------------------------------
-- A P P L I C A T I O N   P A G E   B U T T O N S
--
prompt ...apex_application_page_buttons

create or replace view apex_application_page_buttons
as
select
     f.workspace,
     f.workspace_display_name,
     f.application_id,
     f.application_name,
     p.id                          page_id,
     p.name                        page_name,
     --
     b.BUTTON_SEQUENCE             button_sequence,
     (select pp.plug_name
       from wwv_flow_page_plugs pp
      where pp.id                = b.BUTTON_PLUG_ID
        and pp.security_group_id = b.security_group_id)   region,
     b.button_plug_id              region_id,
     b.BUTTON_NAME                 button_name,
     b.button_static_id            button_static_id,
     (select bt.template_name bt
       from wwv_flow_button_templates bt
      where bt.id                = b.button_template_id
        and bt.flow_id           = f.application_id
        and bt.security_group_id = b.security_group_id)          button_template,
     to_char(b.button_template_id) button_template_id,
     b.button_template_options     button_template_options,
     case b.button_is_hot
       when 'Y' then 'Yes'
       when 'N' then 'No'
     end                           button_is_hot,
     cast(null as varchar2(10))    image_name,
     b.BUTTON_IMAGE_ALT            label,
     --
     case
       when b.button_plug_id is not null then
           coalesce( (
               select dp.name
                 from wwv_flow_plug_tmpl_disp_points dp
                where dp.plug_template_id  = r.plug_template
                  and dp.flow_id           = b.flow_id
                  and dp.security_group_id = b.security_group_id
                  and dp.placeholder       = b.button_position
        ), b.button_position )
      else
          coalesce( (
              select dp.name
                from wwv_flow_page_tmpl_disp_points dp
               where dp.page_template_id  = coalesce( p.step_template, ( select t.default_page_template
                                                                           from wwv_flow_user_interfaces ui,
                                                                                wwv_flow_themes t
                                                                          where ui.security_group_id = b.security_group_id
                                                                            and ui.flow_id           = b.flow_id
                                                                            and ui.ui_type_id        = 1 -- desktop
                                                                            and t.security_group_id  = ui.security_group_id
                                                                            and t.flow_id            = ui.flow_id
                                                                            and t.theme_id           = ui.theme_id ) )
                 and dp.flow_id           = b.flow_id
                 and dp.security_group_id = b.security_group_id
                 and dp.placeholder       = b.button_position
          ), decode( b.button_position,
                      'LEGACY_ORPHAN_COMPONENTS', 'Legacy Orphan Components',
                      b.button_position ) )
     end                             display_position,
     b.button_position               display_position_code,
     --
     case b.grid_new_grid
       when 'Y' then 'Yes'
       else 'No'
     end                            new_grid,
     case b.grid_new_row
       when 'Y' then 'Yes'
       else 'No'
     end                            new_grid_row,
     b.grid_row_css_classes,
     case b.grid_new_column
       when 'Y' then 'Yes'
       else 'No'
     end                            new_grid_column,
     grid_column,
     grid_column_span,
     grid_column_css_classes,
     grid_column_attributes,
     b.BUTTON_ALIGNMENT            alignment,
     --
     decode(b.button_action,
                'SUBMIT',           'Submit Page',
                'REDIRECT_PAGE',    'Redirect to Page in this Application',
                'REDIRECT_URL',     'Redirect to URL',
                'DEFINED_BY_DA',    'Defined by Dynamic Action',
                                    b.button_action)        button_action,
     b.button_action               button_action_code,
     b.BUTTON_REDIRECT_URL         redirect_url,
     case nvl(b.button_execute_validations, 'Y')
       when 'Y' then 'Yes'
       when 'N' then 'No'
     end                           execute_validations,
     --
     case
       when b.warn_on_unsaved_changes = 'I' then 'Do Not Check'
       else 'Page Default'
     end                           warn_on_unsaved_changes,
     b.warn_on_unsaved_changes     warn_on_unsaved_changes_code,
     --
     case b.confirm_enabled
       when 'Y' then 'Yes'
       else 'No'
     end                           confirm_enabled,
     b.confirm_message,
     b.confirm_style,
     --
     nvl((select r from apex_standard_conditions where d = b.BUTTON_CONDITION_TYPE),b.BUTTON_CONDITION_TYPE)
                                   condition_type,
     b.button_condition_type       condition_type_code,
     b.BUTTON_CONDITION            condition_expression1,
     b.BUTTON_CONDITION2           condition_expression2,
     cast(null as varchar2(10))    image_attributes,
     b.button_css_classes          button_css_classes,
     b.icon_css_classes            icon_css_classes,
     b.BUTTON_CATTRIBUTES          button_attributes,
     decode(
        b.DATABASE_ACTION,
        'DELETE','SQL DELETE action',
        'INSERT','SQL INSERT action',
        'UPDATE','SQL UPDATE action',
        b.DATABASE_ACTION)         database_action,
     --
     (select case when b.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(b.REQUIRED_PATCH)
       and pa.security_group_id = b.security_group_id)   build_option,
     --
     decode(substr(b.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)= ltrim(b.SECURITY_SCHEME,'!')
       and ss.flow_id = f.application_id
       and ss.security_group_id = b.security_group_id),
     b.SECURITY_SCHEME)            authorization_scheme,
     b.SECURITY_SCHEME             authorization_scheme_id,
     --
     b.LAST_UPDATED_BY             last_updated_by,
     b.LAST_UPDATED_ON             last_updated_on,
     b.BUTTON_COMMENT              component_comment,
     case button_position
     when 'BODY' then 'Among region items'
     else 'Region Position'
     end                           button_position,
     b.id                          button_id,
     --
     b.BUTTON_NAME
     ||',s='||rpad(b.BUTTON_SEQUENCE,5,'00000')
     ||',r='||(select pp.plug_name from wwv_flow_page_plugs pp where pp.id = b.BUTTON_PLUG_ID and pp.security_group_id = b.security_group_id)
     ||',bs='||b.button_static_id
     ||' t='||(select t.template_name from wwv_flow_button_templates t where b.button_template_id = t.id and t.flow_id = f.application_id and t.security_group_id = b.security_group_id)
     ||' label='||b.BUTTON_IMAGE_ALT
     ||' a='||b.BUTTON_ALIGNMENT
     ||' ba='||b.button_action
     ||' u='||substr(b.BUTTON_REDIRECT_URL,1,30)||length(b.BUTTON_REDIRECT_URL)
     ||' c='||b.BUTTON_CONDITION_TYPE
     ||substr(b.BUTTON_CONDITION,1,20)||length(b.BUTTON_CONDITION)||'.'
     ||substr(b.BUTTON_CONDITION2,1,20)||length(b.BUTTON_CONDITION2)
     ||' ca='||substr(b.BUTTON_CATTRIBUTES,1,20)||length(b.BUTTON_CATTRIBUTES)
     ||' b='||decode(
        b.DATABASE_ACTION,
        'DELETE','SQLDELETE',
        'INSERT','SQLINSERT',
        'UPDATE','SQLUPDATE',
        b.DATABASE_ACTION)
     ||' b='||(select PATCH_NAME
     from wwv_flow_patches
     where id= abs(b.REQUIRED_PATCH))
     ||' s='||decode(substr(b.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(b.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = b.security_group_id),
     'Not found')
     component_signature,
    b.PRE_ELEMENT_TEXT              pre_element_text,
    b.POST_ELEMENT_TEXT             post_element_text,
    b.request_source_type           request_source_type
from wwv_flow_step_buttons b,
     wwv_flow_steps        p,
     wwv_flow_page_plugs   r,
     wwv_flow_authorized   f
where f.workspace_id          = p.security_group_id
  and f.application_id        = p.flow_id
  and p.security_group_id     = b.security_group_id
  and p.flow_id               = b.flow_id
  and p.id                    = b.flow_step_id
  and r.security_group_id (+) = b.security_group_id
  and r.flow_id (+)           = b.flow_id
  and r.id (+)                = b.button_plug_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_buttons',
    p_table_comment     => 'Identifies buttons associated with a Page and Region',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'             , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'        , 'Application Primary Key, Unique over all workspaces',
      'application_name'      , 'Identifies the application',
      'page_id'               , 'Identifies the application',
      'page_name'             , 'Identifies a page within an application',
      'BUTTON_SEQUENCE'       , 'Identifies the display sequence of the button within the region and display position',
      'REGION'                , 'Identifies the Page Region in which this Button is displayed',
      'REGION_ID'             , 'Identifies the Page Region foreign key to the apex_application_page_regions view',
      'BUTTON_NAME'           , 'Identifies the name of the button, which when submitted becomes the REQUEST value',
      'BUTTON_STATIC_ID'      , 'Identifies the Static ID of the button',
      'BUTTON_TEMPLATE'       , 'Identifies the template used to display this button',
      'BUTTON_TEMPLATE_ID'    , 'Identifies the ID of the template used to display the button',
      'BUTTON_TEMPLATE_OPTIONS' , 'Identifies template options to be applied to the component template',
      'button_is_hot'         , 'Identifies if it''s a hot button and the hot button template should be used.',
      'LABEL'                 , 'Identifies the Button Label',
      'image_name'            , 'obsolete - always null',
      'display_position'      , 'Identifies the position where the button will be displayed',
      'display_position_code' , 'Identifies the internal code of DISPLAY_POSITION',
      'new_grid'              , 'Identifies if the rendering of the button begins on a new grid',
      'new_grid_row'          , 'Identifies if the rendering of the button begins on a new grid row',
      'grid_row_css_classes'  , 'Additional CSS classes which are substituted if the grid row template contains the substitution string #CSS_CLASSES#',
      'new_grid_column'       , 'Identifies if the rendering of the button begins on a new grid column',
      'grid_column'           , 'Identifies the grid column used to display the button',
      'grid_column_span'      , 'Identifies over how many grid columns the button spans',
      'grid_column_css_classes' , 'Additional CSS classes which are substituted if the grid column template contains the substitution string #CSS_CLASSES#',
      'grid_column_attributes'  , 'Additional attributes which are substituted if the grid column template contains the substitution string #ATTRIBUTES#',
      'ALIGNMENT'             , 'Identifies the button alignment used for selected display positions',
      'BUTTON_ACTION'         , 'Identifies the Button Action',
      'BUTTON_ACTION_CODE'    , 'Internal code of Button Action',
      'REDIRECT_URL'          , 'Identifies an optional Page or URL to redirect to when this button is pressed.',
      'execute_validations'   , 'Identifies if built-in validations and page validations should be executed when the page is submitted.',
      'warn_on_unsaved_changes'      , 'Determines if a warning should be displayed in case of unsaved changes.',
      'warn_on_unsaved_changes_code' , 'Identifies the internal code WARN_ON_UNSAVED_CHANGES',
      'confirm_enabled'       , 'Identifies whether the button action is protected by a confirmation dialog',
      'confirm_message'       , 'Identifies the message of the confirmation dialog',
      'confirm_style'         , 'Identifies the style of the confirmation dialog',
      'CONDITION_TYPE'        , 'Identifies the condition type used to conditionally display the Page Button',
      'CONDITION_TYPE_CODE'   , 'Internal code of CONDITION_TYPE',
      'CONDITION_EXPRESSION1' , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2' , 'Specifies an expression based on the specific condition type selected.',
      'IMAGE_ATTRIBUTES'      , 'obsolete - always null',
      'button_css_classes'    , 'CSS classes that are used to substitute #BUTTON_CSS_CLASSES# in button template',
      'icon_css_classes'      , 'CSS classes that are used to substitute #ICON_CSS_CLASSES# in button template',
      'BUTTON_ATTRIBUTES'     , 'Identifies the HTML INPUT tag attributes',
      'DATABASE_ACTION'       , 'A button can be used to trigger table row processing (insert, update, and delete). This attribute determines the type of DML action to be performed.',
      'BUILD_OPTION'          , 'Button will be displayed if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'  , 'An authorization scheme must evaluate to TRUE in order for this button to be displayed',
      'AUTHORIZATION_SCHEME_ID'  , 'Foreign Key',
      'COMPONENT_COMMENT'     , 'Developer comment',
      'BUTTON_POSITION'       , 'Identifies where the button displays',
      'BUTTON_ID'             , 'Primary key of this page button',
      'last_updated_by'       , 'APEX developer who made last update',
      'last_updated_on'       , 'Date of last update',
      'component_signature'   , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'pre_element_text' , 'Identifies text placed before the item.',
      'post_element_text' , 'Identifies text placed after the item.',
      'request_source_type'  , 'Identifies how the button request source is determined, for example from a Database Column, Static assignment, or a Query or PL/SQL expression.'
    )
  );
end;
/

---------------------------------------------------
-- A P P L I C A T I O N   P A G E  D Y N A M I C   A C T I O N S
--
prompt ...apex_application_page_da

create or replace view apex_application_page_da
as
select  f.workspace,
        f.workspace_display_name,
        f.application_id,
        f.application_name,
        e.page_id                                               page_id,
        p.name                                                  page_name,
        --
        e.name                                                  dynamic_action_name,
        e.event_sequence                                        dynamic_action_sequence,
        e.triggering_element                                    when_element,
        decode(e.triggering_element_type,
      'ITEM',                  'Item',
      'BUTTON',                'Button',
      'REGION',                'Region',
      'COLUMN',                'Column',
      'DOM_OBJECT',            'DOM Object',
      'JQUERY_SELECTOR',       'jQuery Selector',
      'JAVASCRIPT_EXPRESSION', 'JavaScript Expression',
                                     e.triggering_element_type) when_selection_type,
        e.triggering_element_type                               when_selection_type_code,
        nvl((select plug_name
               from wwv_flow_page_plugs
              where id = e.triggering_region_id),
            e.triggering_region_id)                             when_region,
        e.triggering_region_id                                  when_region_id,
        --
        decode(e.triggering_button_id,
            null,   null,   coalesce(
                    (select button_name from wwv_flow_step_buttons where id = e.triggering_button_id),
                    (select name from wwv_flow_step_items where id = e.triggering_button_id)))
                                                                when_button,
        e.triggering_button_id                                  when_button_id,
        decode(e.condition_element_type,
      'ITEM',                  'Item',
      'COLUMN',                'Column',
                                     e.condition_element_type) 	when_condition_element_type,
        e.condition_element 									when_condition_element,
        --
        decode(e.triggering_condition_type,
      'EQUALS',               'equal to',
      'NOT_EQUALS',           'not equal to',
      'GREATER_THAN',         'greater than',
      'GREATER_THAN_OR_EQUAL','greater than or equal to',
      'LESS_THAN',            'less than',
      'LESS_THAN_OR_EQUAL',   'less than or equal to',
      'NULL',                 'is null',
      'NOT_NULL',             'is not null',
      'IN_LIST',              'in list',
      'NOT_IN_LIST',          'not in list',
      'JAVASCRIPT_EXPRESSION','JavaScript expression',
                                    e.triggering_condition_type)when_condition,
        e.triggering_expression                                 when_expression,
        --
        e.bind_type                                             when_event_scope,
        e.bind_delegate_to_selector                             when_event_static_container,
        e.bind_event_type                                       when_event_internal_name,
        (select d
          from wwv_flow_javascript_events
         where r = e.bind_event_type)                           when_event_name,
        e.bind_event_type_custom                                when_event_custom_name,
        --
        nvl((select r
               from apex_standard_conditions
              where d = e.display_when_type),
            e.display_when_type)                                condition_type,
        e.display_when_type                                     condition_type_code,
        e.display_when_cond                                     condition_expression1,
        e.display_when_cond2                                    condition_expression2,
        --
        (select case when e.required_patch > 0 then
                    patch_name
                else
                    '{Not '||patch_name||'}'
                end patch_name
           from wwv_flow_patches
          where id = abs(e.required_patch))                     build_option,
        e.required_patch                                        build_option_id,
        --
        decode(substr(e.security_scheme,1,1),'!','Not ')||
        nvl((select name
               from wwv_flow_security_schemes
              where to_char(id) = ltrim(e.security_scheme,'!')
                and flow_id = e.flow_id),
            e.security_scheme)                                  authorization_scheme,
        e.security_scheme                                       authorization_scheme_id,
        --
        e.last_updated_on                                       last_updated_on,
        e.last_updated_by                                       last_updated_by,
        e.da_event_comment                                      component_comment,
        e.id                                                    dynamic_action_id,
        --
        (select count(*)
           from wwv_flow_page_da_actions
          where event_id = e.id)                                number_of_actions,
        --
        'name='     || e.name ||
        ',seq='     || lpad(e.event_sequence,5,'00000') ||
        ',tee='     || substr(e.triggering_element,1,30) ||
        ',teet='    || e.triggering_element_type ||
        ',ter='     || nvl( (select plug_name
                              from wwv_flow_page_plugs
                             where id=e.triggering_region_id),
                            e.triggering_region_id) ||
        ',teb='     || decode(e.triggering_button_id,
                                null,   null,   coalesce(
                                                    (select button_name from wwv_flow_step_buttons where id = e.triggering_button_id),
                                                    (select name from wwv_flow_step_items where id = e.triggering_button_id))) ||
        ',tec='     || e.triggering_condition_type ||
        ',teexp='   || substr(e.triggering_expression,1,30) ||
        ',tebt='    || e.bind_type ||
        ',tebdts='  || e.bind_delegate_to_selector ||
        ',tebe='    || e.bind_event_type ||
        ',cond='    || e.display_when_type ||
                       substr(e.display_when_cond,1,20) ||
                       length(e.display_when_cond) ||
                       length(e.display_when_cond2) ||
        ',build='   || nvl( (select patch_name
                               from wwv_flow_patches
                              where id = abs(e.required_patch)),
                            e.required_patch) ||
        ' auth='    || decode(substr(e.security_scheme,1,1),'!','Not ')||
                       nvl( (select name
                              from wwv_flow_security_schemes
                             where to_char(id)= ltrim(e.security_scheme,'!')
                               and flow_id = e.flow_id),
                            e.security_scheme)                  component_signature
  from  wwv_flow_page_da_events e,
        wwv_flow_steps          p,
        wwv_flow_authorized     f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = e.security_group_id
  and p.flow_id           = e.flow_id
  and p.id                = e.page_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_da',
    p_table_comment     => 'Identifies Dynamic Actions associated with a Page',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_id'                    , 'Identifies page number',
      'page_name'                  , 'Identifies a page within an application',
      'dynamic_action_name'        , 'Identifies the name of the Dynamic Action',
      'dynamic_action_sequence'    , 'Identifies the sequence the Dynamic Action is executed',
      'when_element'               , 'Identifies the element(s) that will be the trigger for the Dynamic Action',
      'when_selection_type'        , 'Identifies the type of selector used for the When Element',
      'when_selection_type_code'   , 'Internal code of WHEN_SELECTION_TYPE',
      'when_region'                , 'Identifies the name of the region containing the When Element(s)',
      'when_region_id'             , 'Identifies the ID of the region containing the When Element(s)',
      'when_button'                , 'Identifies the name of the button that will be the trigger for the Dynamic Action',
      'when_button_id'             , 'Identifies the ID of the button that will be the trigger for the Dynamic Action',
      'when_condition_element_type', 'Conditions can be based on a different element to the When Element(s). Identifies the type of element used for the When Condition',
      'when_condition_element'     , 'Conditions can be based on a different element to the When Element(s). Identifies the element(s) that will be used for the When Condition',
      'when_condition'             , 'Identifies the optional condition that can be used to control when the Action''s fire',
      'when_expression'            , 'Identifies the condition expression used to control when the Action''s fire',
      'when_event_scope'           , 'Identifies the scope of the event, can be either ''bind'', ''live'' or ''once''',
      'when_event_static_container', 'Identifies a jQuery selector to select the Static Container element, that can be used to delegate the event handling to',
      'when_event_internal_name'   , 'Identifies the internal name of the event that is used to define when the dynamic action fires',
      'when_event_name'            , 'Identifies the display name of the event that is used to define when the dynamic action fires',
      'condition_type'             , 'Identifies a condition that must be met in order for this Dynamic Action to be processed',
      'condition_type_code'        , 'Internal code of CONDITION_TYPE',
      'condition_expression1'      , 'Specifies an expression based on the specific condition type selected.',
      'condition_expression2'      , 'Specifies an expression based on the specific condition type selected.',
      'build_option'               , 'Dynamic Action will be processed if the Build Option is enabled',
      'build_option_id'            , 'Foreign Key',
      'authorization_scheme'       , 'An authorization scheme must evaluate to TRUE in order for this Dynamic Action to be processed',
      'authorization_scheme_id'    , 'Foreign Key',
      'last_updated_on'            , 'Date of last update',
      'last_updated_by'            , 'APEX developer who made last update',
      'component_comment'          , 'Developer comment',
      'dynamic_action_id'          , 'Primary Key of this Dynamic Action',
      'number_of_actions'          , 'Number of Actions defined for this Dynamic Action',
      'component_signature'        , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'when_event_custom_name'     , 'Name of the custom event which defines when the dynamic action fires.'
    )
  );
end;
/

prompt ...apex_application_page_da_acts

create or replace view apex_application_page_da_acts
as
select  fa.workspace                                        workspace,
        fa.workspace_display_name                           workspace_display_name,
        fa.application_id                                   application_id,
        fa.application_name                                 application_name,
        a.page_id                                           page_id,
        p.name                                              page_name,
        a.event_id                                          dynamic_action_id,
        e.name                                              dynamic_action_name,
        case substr(a.action, 1, 7)
          when 'NATIVE_' then
              ( select pl.display_name from wwv_flow_plugins pl where pl.security_group_id = 10 and pl.flow_id = 4411 and pl.plugin_type = 'DYNAMIC ACTION' and pl.name = substr(a.action, 8) )
          when 'PLUGIN_' then
              ( select pl.display_name from wwv_flow_plugins pl where pl.security_group_id = a.security_group_id and pl.flow_id = a.flow_id and pl.plugin_type = 'DYNAMIC ACTION' and pl.name = substr(a.action, 8) )
          else a.action
        end                                                 action_name,
        a.action                                            action_code,
        a.name                                              action_pd_name,
        a.action_sequence                                   action_sequence,
        decode(a.event_result,  'TRUE',     'True',
                                'FALSE',    'False',
                                            a.event_result )                dynamic_action_event_result,
        decode(a.execute_on_page_init,  'Y',    'Yes',
                                        'N',    'No',
                                                a.execute_on_page_init )    execute_on_page_init,
        a.affected_elements                                 affected_elements,
        decode(a.affected_elements_type,
      'ITEM',                  'Item',
      'REGION',                'Region',
      'COLUMN',                'Column',
      'DOM_OBJECT',            'DOM Object',
      'JQUERY_SELECTOR',       'jQuery Selector',
      'JAVASCRIPT_EXPRESSION', 'JavaScript Expression',
      'TRIGGERING_ELEMENT',    'Triggering Element',
      'EVENT_SOURCE',          'Event Source',
                                     a.affected_elements_type) affected_elements_type,
        a.affected_elements_type                            affected_elements_type_code,
        nvl((select plug_name
               from wwv_flow_page_plugs
              where id = a.affected_region_id),
            a.affected_region_id)                           affected_region,
        a.affected_region_id                                affected_region_id,
        decode(a.affected_button_id,
            null,   null,   coalesce(
                    (select b.button_name from wwv_flow_step_buttons b where b.security_group_id = a.security_group_id and b.id = a.affected_button_id),
                    (select i.name from wwv_flow_step_items i where i.security_group_id = a.security_group_id and i.id = a.affected_button_id)))
                                                            affected_button,
        a.affected_button_id                                affected_button_id,
        --
        a.plugin_init_javascript_code                       init_javascript_code,
        --
        a.attribute_01                                      attribute_01,
        a.attribute_02                                      attribute_02,
        a.attribute_03                                      attribute_03,
        a.attribute_04                                      attribute_04,
        a.attribute_05                                      attribute_05,
        a.attribute_06                                      attribute_06,
        a.attribute_07                                      attribute_07,
        a.attribute_08                                      attribute_08,
        a.attribute_09                                      attribute_09,
        a.attribute_10                                      attribute_10,
        a.attribute_11                                      attribute_11,
        a.attribute_12                                      attribute_12,
        a.attribute_13                                      attribute_13,
        a.attribute_14                                      attribute_14,
        a.attribute_15                                      attribute_15,
        case when a.stop_execution_on_error = 'Y' then 'Yes' else 'No' end as
                                                            stop_execution_on_error,
        case a.wait_for_result
            when 'Y' then 'Yes'
            when 'N' then 'No'
        end                                                 wait_for_result,
        --
        case a.client_condition_type
          when 'EQUALS'                then 'equal to'
          when 'NOT_EQUALS'            then 'not equal to'
          when 'GREATER_THAN'          then 'greater than'
          when 'GREATER_THAN_OR_EQUAL' then 'greater than or equal to'
          when 'LESS_THAN'             then 'less than'
          when 'LESS_THAN_OR_EQUAL'    then 'less than or equal to'
          when 'NULL'                  then 'is null'
          when 'NOT_NULL'              then 'is not null'
          when 'IN_LIST'               then 'in list'
          when 'NOT_IN_LIST'           then 'not in list'
          when 'JAVASCRIPT_EXPRESSION' then 'JavaScript expression'
          else a.client_condition_type
        end                                                 client_condition_type,
        a.client_condition_type                             client_condition_type_code,
        case a.client_condition_element_type
          when 'ITEM'   then 'Item'
          when 'COLUMN' then 'Column'
          else a.client_condition_element_type
        end                                                 client_condition_element_type,
        a.client_condition_element                          client_condition_element,
        a.client_condition_expression                       client_condition_expression,
        --
        nvl((select r
               from apex_standard_conditions
              where d = a.server_condition_type),
            a.server_condition_type)                        server_condition_type,
        a.server_condition_type                             server_condition_type_code,
        a.server_condition_expr1                            server_condition_expression1,
        a.server_condition_expr2                            server_condition_expression2,
        --
        (select case when a.build_option_id > 0 then
                    pa.patch_name
                else
                    '{Not '||pa.patch_name||'}'
                end patch_name
           from wwv_flow_patches pa
          where a.security_group_id = pa.security_group_id
          and pa.id = abs(a.build_option_id))               build_option,
        a.build_option_id                                   build_option_id,
        --
        decode(substr(a.security_scheme,1,1),'!','Not ')||
        nvl((select s.name
               from wwv_flow_security_schemes s
              where s.security_group_id = a.security_group_id
                and to_char(s.id) = ltrim(a.security_scheme,'!')
                and s.flow_id = a.flow_id),
            a.security_scheme)                              authorization_scheme,
        a.security_scheme                                   authorization_scheme_id,
        --
        a.last_updated_by                                   last_updated_by,
        a.last_updated_on                                   last_updated_on,
        a.da_action_comment                                 component_comment,
        a.id                                                action_id,
        --
        'action='   || a.action ||
        ',seq='     || lpad(a.action_sequence,5,'00000') ||
        ',er='      || a.event_result ||
        ',eopi='    || a.execute_on_page_init ||
        ',ae='      || substr(a.affected_elements,1,30) ||
        ',aet='     || a.affected_elements_type ||
        ',aer='     || nvl( (select p.plug_name
                               from wwv_flow_page_plugs p
                              where p.security_group_id = a.security_group_id
                              and p.id=a.affected_region_id),
                            a.affected_region_id) ||
        ',aeb='     || decode(a.affected_button_id,
                                null,   null,   coalesce(
                                                   (select b.button_name from wwv_flow_step_buttons b where b.security_group_id = a.security_group_id and b.id = a.affected_button_id),
                                                   (select i.name from wwv_flow_step_items i where i.security_group_id = a.security_group_id and i.id = a.affected_button_id))) ||
        ',seoe='    || a.stop_execution_on_error ||
        ',wfr='     || a.wait_for_result
        component_signature
  from  wwv_flow_page_da_actions a,
        wwv_flow_page_da_events e,
        wwv_flow_authorized fa,
        wwv_flow_steps p
 where  fa.workspace_id = a.security_group_id
   and  a.flow_id = fa.application_id
   and  e.security_group_id = a.security_group_id
   and  e.id      = a.event_id
   and  p.security_group_id = a.security_group_id
   and  p.flow_id = a.flow_id
   and  p.id      = a.page_id;

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_da_acts',
    p_table_comment     => 'Identifies the Actions of a Dynamic Action associated with a Page',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'page_id' , 'Identifies page number',
      'page_name' , 'Identifies a page within an application',
      'dynamic_action_id' , 'Primary Key of the associated Dynamic Action',
      'dynamic_action_name' , 'Identifies the name of the Dynamic Action that these Actions are associated with',
      'action_name' , 'Identifies the name of the Action (eg Show, Hide, Execute JavaScript Code etc.)',
      'action_code' , 'Internal code of ACTION_NAME',
      'action_pd_name' , 'Identified the name defined in Page Designer for the Action',
      'action_sequence' , 'Identifies the sequence the Action is executed',
      'dynamic_action_event_result' , 'Defines when the Action will execute as a result of whether the Dynamic Action When Condition has passed or failed',
      'execute_on_page_init' , 'Defines whether the Action will also execute when the page initially loads',
      'affected_elements' , 'Identifies the element(s) that will be affected by the Action',
      'affected_elements_type' , 'Identifies the type of selector used for the Affected Element(s)',
      'affected_elements_type_code' , 'Internal code of AFFECTED_ELEMENTS_TYPE',
      'affected_region' , 'Identifies the name of the region containing the Affected Element(s)',
      'affected_region_id' , 'Identifies the ID of the region containing the Affected Element(s)',
      'affected_button' , 'Identifies the button that will be affected by the Action',
      'affected_button_id' , 'Identifies the ID of the button that will be affected by the Action',
      'init_javascript_code' , 'JavaScript code for dynamic action initialization',
      'attribute_01' , 'Identifies attribute 01 for Plug-In Actions',
      'attribute_02' , 'Identifies attribute 02 for Plug-In Actions',
      'attribute_03' , 'Identifies attribute 03 for Plug-In Actions',
      'attribute_04' , 'Identifies attribute 04 for Plug-In Actions',
      'attribute_05' , 'Identifies attribute 05 for Plug-In Actions',
      'attribute_06' , 'Identifies attribute 06 for Plug-In Actions',
      'attribute_07' , 'Identifies attribute 07 for Plug-In Actions',
      'attribute_08' , 'Identifies attribute 08 for Plug-In Actions',
      'attribute_09' , 'Identifies attribute 09 for Plug-In Actions',
      'attribute_10' , 'Identifies attribute 10 for Plug-In Actions',
      'attribute_11' , 'Identifies attribute 11 for Plug-In Actions',
      'attribute_12' , 'Identifies attribute 12 for Plug-In Actions',
      'attribute_13' , 'Identifies attribute 13 for Plug-In Actions',
      'attribute_14' , 'Identifies attribute 14 for Plug-In Actions',
      'attribute_15' , 'Identifies attribute 15 for Plug-In Actions',
      'stop_execution_on_error' , 'Defines whether the following Actions are executed in case of an error',
      'wait_for_result' , 'Defines whether the following Actions wait for the result of the current action, before executing',
      'client_condition_type' , 'Identifies the client-side condition that can be used to control when the Action fires',
      'client_condition_type_code' , 'Internal code of CLIENT_CONDITION_TYPE',
      'client_condition_element_type' , 'Identifies the type of element used for the client-side Condition',
      'client_condition_element' , 'Identifies the element that will be used for the client-side Condition',
      'client_condition_expression' , 'Identifies the client-side condition expression used to control when the Action fires',
      'server_condition_type' , 'Identifies a server-side condition that must be met in order for this Action of a Dynamic Action to be processed',
      'server_condition_type_code' , 'Internal code of SERVER_CONDITION_TYPE',
      'server_condition_expression1' , 'Specifies an expression based on the specific server condition type selected.',
      'server_condition_expression2' , 'Specifies an expression based on the specific server condition type selected.',
      'build_option' , 'Dynamic Action will be processed if the Build Option is enabled',
      'build_option_id' , 'Foreign Key',
      'authorization_scheme' , 'An authorization scheme must evaluate to TRUE in order for this Action of a Dynamic Action to be processed',
      'authorization_scheme_id' , 'Foreign Key',
      'last_updated_by' , 'APEX Developer who made the last update',
      'last_updated_on' , 'Date of last update',
      'component_comment' , 'Developer comment',
      'action_id' , 'Primary Key of the Action',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------
-- A P P L I C A T I O N   B U I L D  O P T I O N S
--
prompt ...apex_application_build_options

create or replace view apex_application_build_options
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    b.PATCH_NAME                     build_option_name,
    decode(b.PATCH_STATUS,
        'INCLUDE','Include',
        'EXCLUDE','Exclude',
        b.patch_status)              build_option_status,
    decode(nvl(b.DEFAULT_ON_EXPORT,
        b.patch_status),
        'INCLUDE','Include',
        'EXCLUDE','Exclude',
        b.DEFAULT_ON_EXPORT)         status_on_export,
    case b.on_upgrade_keep_status
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                              on_upgrade_keep_status,
    b.feature_identifier,
    --b.ATTRIBUTE1                   attribute1,
    --b.ATTRIBUTE2                   attribute2,
    --b.ATTRIBUTE3                   attribute3,
    --b.ATTRIBUTE4                   attribute4,
    b.LAST_UPDATED_BY                last_updated_by,
    b.LAST_UPDATED_ON                last_updated_on,
    b.PATCH_COMMENT                  component_comment,
    b.id                             build_option_id,
    --
    b.PATCH_NAME
    ||' s='||decode(b.PATCH_STATUS,
        'INCLUDE','Include',
        'EXCLUDE','Exclude',
        b.patch_status)
    ||' e='||decode(nvl(b.DEFAULT_ON_EXPORT,
        b.patch_status),
        'INCLUDE','Include',
        'EXCLUDE','Exclude',
        b.DEFAULT_ON_EXPORT)
    component_signature
from wwv_flow_patches    b,
     wwv_flow_authorized f
where f.workspace_id      = b.security_group_id
  and f.application_id    = b.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_build_options',
    p_table_comment     => 'Identifies Build Options available to an application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'            , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'       , 'Application Primary Key, Unique over all workspaces',
      'application_name'     , 'Identifies the application',
      'BUILD_OPTION_NAME'    , 'Identifies the build option',
      'BUILD_OPTION_STATUS'  , 'Identifies the current status of the Build option; Include or Exclude.',
      'STATUS_ON_EXPORT'     , 'Identifies the status (Include or Exclude) of this Build Option when the build option is exported.',
      'on_upgrade_keep_status' , 'Should the current status be kept or set to the specified status when the application is upgraded?',
      'feature_identifier'   , 'Feature identifier',
      'COMPONENT_COMMENT'    , 'Developer Comment',
      'BUILD_OPTION_ID'      , 'Identifies the primary key of this component',
      'last_updated_by'      , 'APEX developer who made last update',
      'last_updated_on'      , 'Date of last update',
      'component_signature'  , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------
-- A P P L I C A T I O N   T A B S
--
prompt ...apex_application_tabs

create or replace view apex_application_tabs
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.TAB_SET                        tab_set,
    t.TAB_SEQUENCE                   display_sequence,
    t.TAB_NAME                       tab_name,
    t.TAB_IMAGE                      when_current_tab_image,
    t.TAB_NON_CURRENT_IMAGE          when_not_current_tab_image,
    t.TAB_IMAGE_ATTRIBUTES           tab_image_attributes,
    t.TAB_TEXT                       tab_label,
    t.TAB_STEP                       tab_page,
    t.TAB_ALSO_CURRENT_FOR_PAGES     tab_also_current_for_pages,
    t.TAB_PARENT_TABSET              parent_tabset,
    --
    nvl((select r from apex_standard_conditions where d = t.DISPLAY_CONDITION_TYPE),t.DISPLAY_CONDITION_TYPE)
                                     condition_type,
    t.display_condition_type         condition_type_code,
    t.TAB_PLSQL_CONDITION            condition_expression1,
    t.TAB_DISP_COND_TEXT             condition_expression2,
    --
    (select case when t.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
     from   wwv_flow_patches pa
     where  pa.id                = abs(t.REQUIRED_PATCH)
       and  pa.security_group_id = t.security_group_id)    build_option,
     --
    decode(substr(t.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(t.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = t.security_group_id),
     t.SECURITY_SCHEME)              authorization_scheme,
    t.SECURITY_SCHEME                authorization_scheme_id,
    --
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    t.TAB_COMMENT                    component_comment,
    t.id                             tab_id,
    --
    t.TAB_SET
    ||' '||lpad(t.TAB_SEQUENCE,5,'00000')
    ||' '||t.TAB_NAME
    ||substr(t.TAB_IMAGE,1,30)||length(t.TAB_IMAGE)
    ||substr(t.TAB_NON_CURRENT_IMAGE,1,30)||length(t.TAB_NON_CURRENT_IMAGE)
    ||substr(t.TAB_IMAGE_ATTRIBUTES,1,30)||length(t.TAB_IMAGE_ATTRIBUTES)
    ||' text='||substr(t.TAB_TEXT,1,30)||length(t.TAB_TEXT)
    ||' p='||t.TAB_STEP
    ||' a='||substr(t.TAB_ALSO_CURRENT_FOR_PAGES,1,30)||length(t.TAB_ALSO_CURRENT_FOR_PAGES)
    ||substr(t.TAB_PARENT_TABSET,1,30)||length(t.TAB_PARENT_TABSET)
    ||' c='||t.DISPLAY_CONDITION_TYPE
    ||substr(t.TAB_PLSQL_CONDITION,1,30)||length(t.TAB_PLSQL_CONDITION)
    ||substr(t.TAB_DISP_COND_TEXT,1,30)||length(t.TAB_DISP_COND_TEXT)
    ||(select pa.PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(t.REQUIRED_PATCH)
       and pa.security_group_id = t.security_group_id)
    ||decode(substr(t.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)= ltrim(t.SECURITY_SCHEME,'!')
       and ss.flow_id = f.application_id
       and ss.security_group_id = t.security_group_id),
     'Not found')
    component_signature
from wwv_flow_tabs       t,
     wwv_flow_authorized f
where f.workspace_id      = t.security_group_id
  and f.application_id    = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_tabs',
    p_table_comment     => 'Identifies a set of tabs collected into tab sets which are associated with a Standard Tab Entry',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'TAB_SET'                    , 'Identifies a collection of tabs that will be displayed together.  If an application uses tabs each page identifies the Tab Set to be displayed with the page.',
      'DISPLAY_SEQUENCE'           , 'Identifies the display sequence of the Tab within the Tab Set',
      'TAB_NAME'                   , 'Identifies the name of the tab; which will be the value of the REQUEST when the tab is pressed',
      'WHEN_CURRENT_TAB_IMAGE'     , 'For tabs displayed using images and not a Tab Label, identifies the current tab image',
      'WHEN_NOT_CURRENT_TAB_IMAGE' , 'For tabs displayed using images and not a Tab Label, identifies the non current tab image',
      'TAB_IMAGE_ATTRIBUTES'       , 'Identifies the HTML IMG tag image attributes',
      'TAB_LABEL'                  , 'Identifies the display Tab Label for tabs that are not based on image',
      'TAB_PAGE'                   , 'Identifies the page which is current and associated with this tab.',
      'TAB_ALSO_CURRENT_FOR_PAGES' , 'Identifies one or more other page ID''s which are also current for this tab',
      'PARENT_TABSET'              , 'Identifies the Parent Tab Tab-Set to be displayed when this tab is current.  Used only when using two levels of tabs.',
      'CONDITION_TYPE'             , 'Identifies the condition type used to conditionally display the tab.',
      'CONDITION_TYPE_CODE'        , 'Internal code of CONDITION_TYPE',
      'CONDITION_EXPRESSION1'      , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'      , 'Specifies an expression based on the specific condition type selected.',
      'BUILD_OPTION'               , 'Tab will be displayed if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'       , 'An authorization scheme must evaluate to TRUE in order for this tab to be displayed',
      'AUTHORIZATION_SCHEME_ID'    , 'Foreign Key',
      'COMPONENT_COMMENT'          , 'Developer comment',
      'TAB_ID'                     , 'Primary key of this tab',
      'last_updated_by'            , 'APEX developer who made last update',
      'last_updated_on'            , 'Date of last update',
      'component_signature'        , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
---------------------------------------------------
-- A P P L I C A T I O N   P A R E N T   T A B S
--
prompt ...apex_application_parent_tabs

create or replace view apex_application_parent_tabs
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.TAB_SET                        tab_set,
    t.TAB_SEQUENCE                   display_sequence,
    t.TAB_NAME                       tab_name,
    t.TAB_IMAGE                      when_current_image,
    t.TAB_NON_CURRENT_IMAGE          when_non_current_image,
    t.TAB_IMAGE_ATTRIBUTES           image_attributes,
    t.TAB_TEXT                       tab_label,
    t.TAB_TARGET                     tab_target,
    --
    nvl((select r from apex_standard_conditions where d = t.DISPLAY_CONDITION_TYPE),t.DISPLAY_CONDITION_TYPE)
                                     condition_type,
    t.display_condition_type         condition_type_code,
    t.DISPLAY_CONDITION              condition_expression1,
    t.DISPLAY_CONDITION2             condition_expression2,
    --
    t.CURRENT_ON_TABSET              current_for_tabset,
    --
    (select case when t.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(t.REQUIRED_PATCH)
       and pa.security_group_id = t.security_group_id)    build_option,
    --
    decode(substr(t.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)= ltrim(t.SECURITY_SCHEME,'!')
       and ss.flow_id = f.application_id
       and ss.security_group_id = t.security_group_id),
     t.SECURITY_SCHEME)              authorization_scheme,
    t.SECURITY_SCHEME                authorization_scheme_id,
    --
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    t.TAB_COMMENT                    component_comment,
    t.id                             parent_tab_id,
    --
    t.TAB_SET
    ||' '||lpad(t.TAB_SEQUENCE,5,'00000')
    ||' '||substr(t.TAB_NAME              ,1,30)||length(t.TAB_NAME             )
    ||' '||substr(t.TAB_IMAGE             ,1,30)||length(t.TAB_IMAGE            )
    ||' '||substr(t.TAB_NON_CURRENT_IMAGE ,1,30)||length(t.TAB_NON_CURRENT_IMAGE)
    ||' '||substr(t.TAB_IMAGE_ATTRIBUTES  ,1,30)||length(t.TAB_IMAGE_ATTRIBUTES )
    ||' '||substr(t.TAB_TEXT              ,1,30)||length(t.TAB_TEXT             )
    ||' '||substr(t.TAB_TARGET            ,1,30)||length(t.TAB_TARGET           )
    ||' cond='||t.DISPLAY_CONDITION_TYPE
    ||substr(t.DISPLAY_CONDITION,1,30)||length(t.DISPLAY_CONDITION)
    ||substr(t.DISPLAY_CONDITION2,1,30)||length(t.DISPLAY_CONDITION2)
    ||' ct='||t.CURRENT_ON_TABSET
    ||' bo='||(select PATCH_NAME
     from   wwv_flow_patches
     where  id =abs(t.REQUIRED_PATCH))
    ||' sec='||decode(substr(t.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)= ltrim(t.SECURITY_SCHEME,'!')
       and ss.flow_id = f.application_id
       and ss.security_group_id = t.security_group_id),
     'Not found')
    component_signature
from wwv_flow_toplevel_tabs t,
     wwv_flow_authorized    f
where f.workspace_id      = t.security_group_id
  and f.application_id    = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_parent_tabs',
    p_table_comment     => 'Identifies a collection of tabs called a Tab Set.  Each tab is part of a tab set and can be current for one or more pages.  Each tab can also have a corresponding Parent Tab if two levels of Tabs are defined.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'TAB_SET'                , 'Identifies a collection of Parent Tabs that are displayed together.',
      'DISPLAY_SEQUENCE'       , 'Identifies the display sequence within the Tab Set',
      'TAB_NAME'               , 'Identifies the Tab Name',
      'WHEN_CURRENT_IMAGE'     , 'For tabs displayed using images and not a Tab Label, identifies the current tab image',
      'WHEN_NON_CURRENT_IMAGE' , 'For tabs displayed using images and not a Tab Label, identifies the non current tab image',
      'IMAGE_ATTRIBUTES'       , 'Identifies the HTML IMG tag image attributes',
      'TAB_LABEL'              , 'Identifies the display Tab Label for tabs that are not based on image',
      'TAB_TARGET'             , 'Identifies the Page or URL to branch to when this Parent Tab is pressed',
      'CONDITION_TYPE'         , 'Identifies the condition type used to conditionally display the tab.',
      'CONDITION_TYPE_CODE'    , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'  , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'  , 'Specifies an expression based on the specific condition type selected.',
      'CURRENT_FOR_TABSET'     , 'Identifies the child (corresponding standard tab) of this parent tab.',
      'BUILD_OPTION'           , 'Tab will be displayed if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'   , 'An authorization scheme must evaluate to TRUE in order for this tab to be displayed',
      'AUTHORIZATION_SCHEME_id' , 'Foreign Key',
      'COMPONENT_COMMENT'      , 'Developer comment',
      'PARENT_TAB_ID'          , 'Primary key of the Parent Tab component',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- S U P P OR T I N G    O B J E C T S
--
prompt ...apex_application_supp_objects

create or replace view apex_application_supp_objects
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    (select count(*)
     from wwv_flow_install_build_opt bo
    where bo.install_id        = i.id
      and bo.security_group_id = i.security_group_id)         build_options,
    --
    decode(i.INCLUDE_IN_EXPORT_YN,
      'Y','Yes',
      'N','No',
      i.INCLUDE_IN_EXPORT_YN)        include_in_application_export,
    i.WELCOME_MESSAGE                ,
    i.LICENSE_MESSAGE                ,
    i.VALIDATION_MESSAGE             ,
    --
    i.INSTALL_MESSAGE                installation_message,
    i.INSTALL_SUCCESS_MESSAGE        installation_success_message,
    i.INSTALL_FAILURE_MESSAGE        installation_failure_message,
    (select count(*)
     from wwv_flow_install_scripts sc
     where i.id = sc.install_id
       and nvl(sc.script_type,'INSTALL') = 'INSTALL'
       and sc.security_group_id = i.security_group_id)
                                     installation_scripts,
    --
    i.get_version_sql_query          upgrade_when_query,
    i.upgrade_message                upgrade_message,
    i.upgrade_confirm_message        upgrade_confirm_message,
    i.upgrade_success_message        upgrade_success_message,
    i.upgrade_failure_message        upgrade_failure_message,
    (select count(*)
     from wwv_flow_install_scripts sc
     where i.id = sc.install_id
       and nvl(sc.script_type,'INSTALL') = 'UPGRADE'
       and sc.security_group_id = i.security_group_id)
                                     upgrade_scripts,
    --
    --
    i.DEINSTALL_MESSAGE              ,
    i.DEINSTALL_SUCCESS_MESSAGE      ,
    --i.DEINSTALL_FAILURE_MESSAGE      ,
    i.DEINSTALL_SCRIPT               deinstallation_script,
    --
    --i.PROMPT_SUBSTITUTION_01_YN      ,
    --i.PROMPT_SUBSTITUTION_02_YN      ,
    --i.PROMPT_SUBSTITUTION_03_YN      ,
    --i.PROMPT_SUBSTITUTION_04_YN      ,
    --i.PROMPT_SUBSTITUTION_05_YN      ,
    --i.PROMPT_SUBSTITUTION_06_YN      ,
    --i.PROMPT_SUBSTITUTION_07_YN      ,
    --i.PROMPT_SUBSTITUTION_08_YN      ,
    --i.PROMPT_SUBSTITUTION_09_YN      ,
    --i.PROMPT_SUBSTITUTION_10_YN      ,
    --i.PROMPT_SUBSTITUTION_11_YN      ,
    --i.PROMPT_SUBSTITUTION_12_YN      ,
    --i.PROMPT_SUBSTITUTION_13_YN      ,
    --i.PROMPT_SUBSTITUTION_14_YN      ,
    --i.PROMPT_SUBSTITUTION_15_YN      ,
    --i.PROMPT_SUBSTITUTION_16_YN      ,
    --i.PROMPT_SUBSTITUTION_17_YN      ,
    --i.PROMPT_SUBSTITUTION_18_YN      ,
    --i.PROMPT_SUBSTITUTION_19_YN      ,
    --i.PROMPT_SUBSTITUTION_20_YN      ,
    ----
    --i.PROMPT_IF_MULT_AUTH_YN         ,
    --i.PROMPT_BUILD_OPTIONS           ,
    --
    i.LAST_UPDATED_BY                ,
    i.LAST_UPDATED_ON                ,
    i.CREATED_BY                     ,
    i.CREATED_ON                     ,
    --
    --i.INSTALL_PROMPT_01              ,
    --i.INSTALL_PROMPT_02              ,
    --i.INSTALL_PROMPT_03              ,
    --i.INSTALL_PROMPT_04              ,
    --i.INSTALL_PROMPT_05              ,
    --i.INSTALL_PROMPT_06              ,
    --i.INSTALL_PROMPT_07              ,
    --i.INSTALL_PROMPT_08              ,
    --i.INSTALL_PROMPT_09              ,
    --i.INSTALL_PROMPT_10              ,
    --i.INSTALL_PROMPT_11              ,
    --i.INSTALL_PROMPT_12              ,
    --i.INSTALL_PROMPT_13              ,
    --i.INSTALL_PROMPT_14              ,
    --i.INSTALL_PROMPT_15              ,
    --i.INSTALL_PROMPT_16              ,
    --i.INSTALL_PROMPT_17              ,
    --i.INSTALL_PROMPT_18              ,
    --i.INSTALL_PROMPT_19              ,
    --i.INSTALL_PROMPT_20              ,
    i.CONFIGURATION_MESSAGE          ,
    -- obsolete i.TRIGGER_INSTALL_WHEN_COND      ,
    -- obsolete i.TRIGGER_INSTALL_WHEN_EXP1      ,
    -- obsolete i.TRIGGER_INSTALL_WHEN_EXP2      ,
    -- obsolete i.TRIGGER_FAILURE_MESSAGE        ,
    i.BUILD_OPTIONS_MESSAGE          ,
    i.REQUIRED_FREE_KB               ,
    i.REQUIRED_SYS_PRIVS             required_system_privileges,
    i.REQUIRED_NAMES_AVAILABLE       ,
    --
    i.id                             supporting_object_id
from wwv_flow_install    i,
     wwv_flow_authorized f
where f.workspace_id   = i.security_group_id
  and f.application_id = i.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_supp_objects',
    p_table_comment     => 'Identifies the Supporting Object installation messages',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'BUILD_OPTIONS'                 , 'Count of Build Options referenced in this applications supporting object installation wizard',
      'INCLUDE_IN_APPLICATION_EXPORT' , 'Flag that specifies if the application export will include the supporting object definitions and scripts',
      'WELCOME_MESSAGE'               , 'After successfully importing and installing an application definition, the installation wizard prompts the user to install supporting objects for the application.',
      'LICENSE_MESSAGE'               , 'If the use of this application requires the user to accept a license, enter it here. The user will be prompted to accept the message before installing database objects.',
      'VALIDATION_MESSAGE'            , 'Message to display above the list of installation validations when this application installs.',
      'INSTALLATION_MESSAGE'          , 'Message to display during installation confirmation before the installation scripts are run and configuration options are applied.',
      'INSTALLATION_SUCCESS_MESSAGE'  , 'Enter a message to display after all installation scripts have run without errors.',
      'INSTALLATION_FAILURE_MESSAGE'  , 'Enter a message to display after installation scripts have been run with errors.',
      'INSTALLATION_SCRIPTS'          , 'Count of installation SQL scripts',
      'upgrade_when_query'            , 'SQL query that returns at least one row if supporting objects exist, which means an upgrade should be performed',
      'upgrade_message'               , 'Message to display before this application''s supporting objects are upgraded',
      'upgrade_confirm_message'       , 'Message to display during upgrade confirmation before the upgrade scripts are run and configuration options are applied',
      'upgrade_success_message'       , 'Message to display after all upgrade scripts have run without errors',
      'upgrade_failure_message'       , 'Message to display after upgrade scripts have been run with errors',
      'upgrade_scripts'               , 'Count of upgrade SQL scripts',
      'DEINSTALL_MESSAGE'             , 'Enter a message to display during deinstallation confirmation when this application is deinstalled.',
      'DEINSTALL_SUCCESS_MESSAGE'     , 'Enter a message to display after all deinstallation scripts have run.',
      'DEINSTALLATION_SCRIPT'         , 'The deinstallation SQL script, uses SQL*plus like script syntax',
      'CREATED_BY'                    , 'APEX developer who created the component',
      'CREATED_ON'                    , 'Date created',
      'CONFIGURATION_MESSAGE'         , 'Message to display above the application substitution prompts when this application installs.',
      'BUILD_OPTIONS_MESSAGE'         , 'Message to display above the build option prompts when this application installs.',
      'REQUIRED_FREE_KB'              , 'Freespace required to install this application.',
      'required_system_privileges'    , 'Identifies the system privileges required to install this application.',
      'REQUIRED_NAMES_AVAILABLE'      , 'Installation is prevented if these objects already exist in the parsing schema.',
      'SUPPORTING_OBJECT_ID'          , 'Primary Key of this component',
      'last_updated_by'               , 'APEX developer who made last update',
      'last_updated_on'               , 'Date of last update'
    )
  );
end;
/

create or replace view apex_application_supp_obj_bopt
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    (select pa.patch_name
      from   wwv_flow_patches pa
     where pa.id                = i.BUILD_OPT_ID
       and pa.security_group_id = i.security_group_id)       build_option,
    --
    decode(
    (select pa.patch_status
      from   wwv_flow_patches pa
     where pa.id                = i.BUILD_OPT_ID
       and pa.security_group_id = i.security_group_id),
     'INCLUDE','Include',
     'EXCLUDE','Exclude',
     i.BUILD_OPT_ID)                 default_status,
    --
    (select pa.patch_comment
      from wwv_flow_patches pa
     where pa.id                = i.BUILD_OPT_ID
       and pa.security_group_id = i.security_group_id)       developer_comment,
     i.id                            install_build_option_id
from wwv_flow_install_build_opt i,
     wwv_flow_authorized        f
where f.workspace_id   = i.security_group_id
  and f.application_id = i.flow_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_supp_obj_bopt',
    p_table_comment     => 'Identifies the Application Build Options that will be exposed to the Supporting Object installation',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'               , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'          , 'Application Primary Key, Unique over all workspaces',
      'application_name'        , 'Identifies the application',
      'build_option'            , 'Identifies the Application Build Option that will be prompted for during the installation of the Supporting Objects',
      'default_status'          , 'Identifies the default Status (Excluded, Included) of the Build Option',
      'developer_comment'       , 'Identifies the developer comment applied to the Build Option definition',
      'install_build_option_id' , 'Primary Key of this Supporting Object Build Option'
    )
  );
end;
/

create or replace view apex_application_supp_obj_scr
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    i.NAME                           script_name,
    i.SEQUENCE                       execution_sequence,
    decode(i.SCRIPT_TYPE,
           'UPGRADE','Upgrade',
           'INSTALL','Install',
           i.SCRIPT_TYPE)            script_type,
    --
    i.SCRIPT                         sql_script,
    --
    nvl((select r from apex_standard_conditions where d = i.condition_type), i.condition_type) condition_type,
    i.CONDITION_TYPE                 condition_type_code,
    i.CONDITION                      condition_expression1,
    i.CONDITION2                     condition_expression2,
    --
    i.LAST_UPDATED_BY                last_updated_by,
    i.LAST_UPDATED_ON                last_updated_on,
    i.CREATED_BY                     created_by,
    i.CREATED_ON                     created_on,
    --
    i.ID                             supporting_object_script_id
from wwv_flow_install_scripts i,
     wwv_flow_authorized      f
where f.workspace_id   = i.security_group_id
  and f.application_id = i.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_supp_obj_scr',
    p_table_comment     => 'Identifies the Supporting Object installation SQL Scripts',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                   , 'A work area mapped to one or more database schemas',
      'workspace_display_name'      , 'Display name for the workspace',
      'application_id'              , 'Application Primary Key, Unique over all workspaces',
      'application_name'            , 'Identifies the application',
      'SCRIPT_NAME'                 , 'Identifies the name of the installation SQL Script',
      'EXECUTION_SEQUENCE'          , 'Identifies the execution of the installation SQL Script',
      'SCRIPT_TYPE'                 , 'Identifies whether this is an install or upgrade SQL Script',
      'SQL_SCRIPT'                  , 'Identifies the SQL Script.  Most basic SQL*plus syntax can be used to create database objects and load sample data.',
      'CONDITION_TYPE'              , 'Identifies the condition type used to conditionally execute the Installation SQL Script',
      'CONDITION_TYPE_CODE'         , 'Internal code of CODE_TYPE.',
      'CONDITION_EXPRESSION1'       , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'       , 'Specifies an expression based on the specific condition type selected.',
      'CREATED_BY'                  , 'APEX User Name of the developer who created this SQL Script',
      'CREATED_ON'                  , 'Date that this SQL Script was created',
      'SUPPORTING_OBJECT_SCRIPT_ID' , 'Primary Key of this SQL Script component',
      'last_updated_by'             , 'APEX developer who made last update',
      'last_updated_on'             , 'Date of last update'
    )
  );
end;
/

create or replace view apex_application_supp_obj_chck
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    c.ID                             check_id,
    c.NAME                           check_name,
    c.SEQUENCE                       check_sequence,
    --
    c.CHECK_TYPE                     check_type,
    c.CHECK_CONDITION                check_expression1,
    c.CHECK_CONDITION2               check_expression2,
    --
    --c.SUCCESS_MESSAGE                ,
    c.FAILURE_MESSAGE                error_message,
    --
    nvl((select r from apex_standard_conditions where d = c.condition_type), c.condition_type) condition_type,
    c.CONDITION_TYPE                 condition_type_code,
    c.CONDITION                      condition_expression1,
    c.CONDITION2                     condition_expression2,
    --
    c.LAST_UPDATED_BY                last_updated_by,
    c.LAST_UPDATED_ON                last_updated_on,
    c.CREATED_BY                     created_by,
    c.CREATED_ON                     created_on
from wwv_flow_install_checks c,
     wwv_flow_authorized     f
where f.workspace_id   = c.security_group_id
  and f.application_id = c.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_supp_obj_chck',
    p_table_comment     => 'Identifies the Supporting Object pre-installation checks to ensure the database is compatible with the objects to be installed',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'             , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'        , 'Application Primary Key, Unique over all workspaces',
      'application_name'      , 'Identifies the application',
      'CHECK_ID'              , 'Specifies the ID for this validation.',
      'CHECK_NAME'            , 'Specifies the name for this validation.',
      'CHECK_SEQUENCE'        , 'Specifies the sequence for this validation. The sequence number determines the order of evaluation.',
      'CHECK_TYPE'            , 'Specifies the condition type that must be met during installation before installation scripts are run.',
      'CHECK_EXPRESSION1'     , 'Use this attribute to conditionally control whether installation can continue. Values correspond to the specific condition type selected.',
      'CHECK_EXPRESSION2'     , 'Use this attribute to conditionally control whether installation can continue. Values correspond to the specific condition type selected.',
      'error_MESSAGE'         , 'Enter a message to be displayed when the conditions of this validation are not met.',
      'CONDITION_TYPE'        , 'Specifies a condition type from the list that conditionally controls whether this validation is performed.',
      'CONDITION_TYPE_CODE'   , 'Internal code of CODE_TYPE.',
      'CONDITION_EXPRESSION1' , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2' , 'Specifies an expression based on the specific condition type selected.',
      'CREATED_BY'            , 'Identifies the User Name of the APEX developer who created this check condition',
      'CREATED_ON'            , 'Identifies the date that this component was created',
      'last_updated_by'       , 'APEX developer who made last update',
      'last_updated_on'       , 'Date of last update'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   T R A N S L A T I O N S
--
prompt ...apex_application_translations

create or replace view apex_application_translations
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.NAME                           translatable_message,
    t.MESSAGE_LANGUAGE               language_code,
    --
    t.MESSAGE_TEXT                   message_text,
    t.is_js_message,
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    t.MESSAGE_COMMENT                developer_comment,
    t.id                             translation_entry_id,
    t.name || ' l=' || t.message_language || ' m=' || substr(t.message_text, 1, 30) || length(t.message_text)
        as component_signature
from wwv_flow_messages$  t,
     wwv_flow_authorized f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
  /* keep this not exists */
  and not exists ( select 1
                     from wwv_flow_language_map m
                    where m.translation_flow_id = f.application_id
                      and m.security_group_id   = f.workspace_id
                      and f.authentication_id   <> trunc(f.authentication_id) )
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_translations',
    p_table_comment     => 'Identifies message primary language text and translated text',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'             , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'        , 'Application Primary Key, Unique over all workspaces',
      'application_name'      , 'Identifies the application',
      'TRANSLATABLE_MESSAGE'  , 'Identifies the Message Name',
      'LANGUAGE_CODE'         , 'Identifies the Language Code',
      'MESSAGE_TEXT'          , 'Identifies the message text in the Language Code language',
      'is_js_message'         , 'Identifies if the message is used in JavaScript code',
      'DEVELOPER_COMMENT'     , 'Developer Comment',
      'last_updated_by'       , 'APEX developer who made last update',
      'last_updated_on'       , 'Date of last update',
      'translation_entry_id'  , 'Primary Key of this Translation Entry',
      'component_signature'   , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   I T E M S
--
prompt ...apex_application_items

create or replace view apex_application_items
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    i.name                           item_name,
    i.scope                          scope,
    --i.name_length
    i.data_type                      data_type,
    --i.is_persistent                  ,
    i.protection_level               session_state_protection,
    case i.escape_on_http_output
    when 'Y' then 'Yes'
    when 'N' then 'No'
    else i.escape_on_http_output
    end                              escape_on_http_output,
    (select case when i.required_patch > 0 then pa.patch_name else '{Not '||pa.patch_name||'}' end patch_name
      from wwv_flow_patches pa
     where pa.id                = abs(i.required_patch)
       and pa.security_group_id = i.security_group_id)    build_option,
    i.last_updated_by                ,
    i.last_updated_on                ,
    i.item_comment                   component_comment,
    i.id                             application_item_id,
    --
    i.name
    ||' dt='||i.data_type
    ||' prot='||i.protection_level
    ||' bo='||(select pa.patch_name
      from   wwv_flow_patches pa
     where pa.id                = i.required_patch
       and pa.security_group_id = i.security_group_id)
    component_signature
from wwv_flow_items      i,
     wwv_flow_authorized f
where f.workspace_id   = i.security_group_id
  and f.application_id = i.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_items',
    p_table_comment     => 'Identifies Application Items used to maintain session state that are not associated with a page',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                 , 'A work area mapped to one or more database schemas',
      'workspace_display_name'    , 'Display name for the workspace',
      'application_id'            , 'Application Primary Key, Unique over all workspaces',
      'application_name'          , 'Identifies the application',
      'item_name'                 , 'Name of the Application Item used to maintain session state',
      'scope'                     , 'Scope of Application: GLOBAL (across all sessions) or APP (the item''s application only)',
      'data_type'                 , 'Datatype (Varchar or Number) of the Application Item.  Use a numeric data type to limit exposure to URL hackers for management of numeric session state',
      'Session_State_Protection'  , 'Identifies the Session State Protection for this item.  If the item is "Unrestricted" the item may be set by passing the item name/value in a URL or in a form.  Other protection levels provide enhanced protection of session state.',
      'escape_on_http_output'     , 'If Yes values will be escaped when they are used in substitutions.',
      'build_option'              , 'Application Item will be available if the Build Option is enabled',
      'component_comment'         , 'Developer comment',
      'application_item_id'       , 'Primary key of this Application Item',
      'last_updated_by'           , 'APEX developer who made last update',
      'last_updated_on'           , 'Date of last update',
      'component_signature'       , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   P R O C E S S E S
--
prompt ...apex_application_processes

create or replace view apex_application_processes
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    p.PROCESS_SEQUENCE               process_sequence,
    --
    decode(p.PROCESS_POINT,
      'BEFORE_HEADER','On Load: Before Header (page template header)',
      'AFTER_HEADER','On Load: After Header (page template header)',
      'BEFORE_BOX_BODY','On Load: Before Body Region(s)',
      'AFTER_BOX_BODY','On Load: After Body Region(s)',
      'BEFORE_FOOTER','On Load: Before Footer (page template footer)',
      'AFTER_FOOTER','On Load: After Footer (page template footer)',
      'ON_SUBMIT_BEFORE_COMPUTATION','On Submit: After Page Submission - Before Computations and Validations',
      'AFTER_SUBMIT','On Submit: After Page Submission - After Computations and Validations',
      'ON_NEW_INSTANCE','On New Session: After Authentication',
      'AFTER_LOGIN','After Login',
      'ON_DEMAND','Ajax Callback: Run this application process when requested by a page process',
      p.process_point)               process_point,
    --
    decode(p.PROCESS_TYPE,
      'NATIVE_PLSQL','Anonymous Block',
      p.PROCESS_TYPE)                process_type,
    decode(p.PROCESS_TYPE,
      'NATIVE_PLSQL', 'PLSQL',
      p.PROCESS_TYPE)                process_type_code,
    p.process_type                   process_type_plugin_name,
    --
    p.PROCESS_NAME                   process_name,
    p.PROCESS_SQL_CLOB               process,
    p.process_clob_language          process_language,
    p.attribute_01,
    p.attribute_02,
    p.attribute_03,
    p.attribute_04,
    p.attribute_05,
    p.attribute_06,
    p.attribute_07,
    p.attribute_08,
    p.attribute_09,
    p.attribute_10,
    p.attribute_11,
    p.attribute_12,
    p.attribute_13,
    p.attribute_14,
    p.attribute_15,
    p.PROCESS_ERROR_MESSAGE          error_message,
    p.error_display_location         error_display_location,
    --
    nvl((select r
    from apex_standard_conditions
    where d = p.PROCESS_WHEN_TYPE ),
    p.PROCESS_WHEN_TYPE)
                                     condition_type,
    p.process_when_type              condition_type_code,
    p.PROCESS_WHEN                   condition_expression1,
    p.PROCESS_WHEN2                  condition_expression2,
    --p.PROCESS_WHEN_TYPE2             ,
    --p.ITEM_NAME                      ,
    nvl((select case when p.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where                pa.id = abs(p.REQUIRED_PATCH)
       and pa.security_group_id = p.security_group_id),
     p.REQUIRED_PATCH)               build_option,
     --
     decode(substr(p.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(p.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = p.security_group_id),
     p.SECURITY_SCHEME)              authorization_scheme,
    p.SECURITY_SCHEME                authorization_scheme_id,
    p.LAST_UPDATED_BY                last_updated_by,
    p.LAST_UPDATED_ON                last_updated_on,
    p.PROCESS_COMMENT                component_comment,
    p.id                             application_process_id,
    --
    p.PROCESS_NAME
    ||' seq='||lpad(p.PROCESS_SEQUENCE,5,'00000')
    ||' '||p.PROCESS_POINT
    ||' '||decode(p.PROCESS_TYPE,'PLSQL','Anonymous Block',p.PROCESS_TYPE)
    ||' txt='||sys.dbms_lob.substr(p.PROCESS_SQL_CLOB,50,1)||sys.dbms_lob.getlength(p.PROCESS_SQL_CLOB)
    ||' m='||substr(p.PROCESS_ERROR_MESSAGE,1,30)||length(p.PROCESS_ERROR_MESSAGE)
    ||' cond='||p.PROCESS_WHEN_TYPE
    ||substr(p.PROCESS_WHEN,1,20)||length(p.PROCESS_WHEN)
    ||substr(p.PROCESS_WHEN2,1,20)||length(p.PROCESS_WHEN2)
    ||' b='||nvl((select ss.name from    wwv_flow_security_schemes ss where   to_char(ss.id)= abs(p.REQUIRED_PATCH) and ss.flow_id = f.application_id and ss.security_group_id = p.security_group_id),p.REQUIRED_PATCH)
    ||' s='||decode(substr(p.SECURITY_SCHEME,1,1),'!','Not ')||
     nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(p.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = p.security_group_id),
     'Not found')
    component_signature
from wwv_flow_processing p,
     wwv_flow_authorized f
where f.workspace_id   = p.security_group_id
  and f.application_id = p.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_processes',
    p_table_comment     => 'Identifies Application Processes which can run for every page, on login or upon demand',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'PROCESS_SEQUENCE'       , 'Identifies the sequence that processes will be considered for execution within each process point',
      'PROCESS_POINT'          , 'Identifies the point of execution of this process',
      'PROCESS_TYPE'           , 'Identifies the type of process this is',
      'process_type_code'      , 'Deprecated internal code of PROCESS_TYPE. Use PROCESS_TYPE_PLUGIN_NAME instead.',
      'process_type_plugin_name' , 'Name of the plugin that implements the process. Names of built-in plugins start with "NATIVE_", application specific plugin names start with "PLUGIN_"',
      'PROCESS_NAME'           , 'Identifies this process',
      'PROCESS'                , 'Text of the Application Process',
      'process_language'       , 'Identifies the programming language of PROCESS.',
      'ATTRIBUTE_01'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_02'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_03'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_04'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_05'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_06'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_07'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_08'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_09'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_10'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_11'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_12'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_13'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_14'           , 'Dynamic attribute to store additional data',
      'ATTRIBUTE_15'           , 'Dynamic attribute to store additional data',
      'ERROR_MESSAGE'          , 'Identifies the error message to be displayed if this processes raises an exception',
      'error_display_location' , 'The Error display location identifies where the process error message will display. Messages can be displayed on an error page, or inline with the existing page. Inline errors are displayed in the "notification" area (defined as part of the page template).',
      'CONDITION_type'         , 'Specifies a condition type from the Application Process that conditionally controls whether this Application Process is performed.',
      'CONDITION_TYPE_CODE'    , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'  , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'  , 'Specifies an expression based on the specific condition type selected.',
      'BUILD_OPTION'           , 'Application Process will be available for execution if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'   , 'An authorization scheme must evaluate to TRUE in order for this process to be executed',
      'AUTHORIZATION_SCHEME_ID' , 'Foreign Key',
      'COMPONENT_COMMENT'      , 'Developer comment',
      'APPLICATION_PROCESS_ID' , 'Primary key of this Application Process',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   C O M P U T A T I O N S
--
prompt ...apex_application_computations

create or replace view apex_application_computations
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    c.COMPUTATION_SEQUENCE           computation_sequence,
    c.COMPUTATION_ITEM               computation_item,
    c.COMPUTATION_POINT              computation_point,
    --c.COMPUTATION_ITEM_TYPE
    c.COMPUTATION_TYPE               computation_type,
    c.computation_language           computation_language,
    --c.COMPUTATION_PROCESSED          computation_processed,
    c.COMPUTATION                    computation,
    --
    decode(substr(c.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(c.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = c.security_group_id),
     c.SECURITY_SCHEME)              authorization_scheme,
    c.SECURITY_SCHEME                authorization_scheme_id,
    --
    nvl((select case when c.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(c.REQUIRED_PATCH)
       and pa.security_group_id = c.security_group_id),
     c.REQUIRED_PATCH)               build_option,
    --
    nvl((select r from apex_standard_conditions where d = c.COMPUTE_WHEN_TYPE),c.COMPUTE_WHEN_TYPE)
                                     condition_type,
    c.compute_when_type              condition_type_code,
    c.COMPUTE_WHEN                   condition_expression1,
    c.COMPUTE_WHEN_TEXT              condition_expression2,
    --
    c.COMPUTATION_ERROR_MESSAGE      error_message,
    c.LAST_UPDATED_BY                last_updated_by,
    c.LAST_UPDATED_ON                last_updated_on,
    c.COMPUTATION_COMMENT            component_comment,
    c.id                             application_computation_id,
    --
    c.COMPUTATION_ITEM
    ||' seq='||lpad(c.COMPUTATION_SEQUENCE,5,'00000')
    ||' p='||c.COMPUTATION_POINT
    ||' t='||c.COMPUTATION_TYPE
    ||' c='||substr(c.COMPUTATION,1,50)||length(c.COMPUTATION)
    ||' s='||decode(substr(c.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(c.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = c.security_group_id),
     'Not found')
    ||' c='||c.COMPUTE_WHEN_TYPE
    ||substr(c.COMPUTE_WHEN,1,30)||length(c.COMPUTE_WHEN)
    ||substr(c.COMPUTE_WHEN_TEXT,1,30)||length(c.COMPUTE_WHEN_TEXT)
    ||' m='||substr(c.COMPUTATION_ERROR_MESSAGE,1,30)||length(c.COMPUTATION_ERROR_MESSAGE)
    component_signature
from wwv_flow_computations c,
     wwv_flow_authorized   f
where f.workspace_id   = c.security_group_id
  and f.application_id = c.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_computations',
    p_table_comment     => 'Identifies Application Computations which can run for every page or on login',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'COMPUTATION_SEQUENCE'   , 'Identifies the sequence within the Display Point that this computation will be considered for execution',
      'COMPUTATION_ITEM'       , 'Compute this item value; the purpose of this computation is to set this item''s value.',
      'COMPUTATION_POINT'      , 'Identifies a pre-defined point at which this computation will be performed',
      'COMPUTATION_TYPE'       , 'Identifies the manner in which this computation will be performed',
      'computation_language'   , 'Identifies the programming language of the computation.',
      'COMPUTATION'            , 'Identifies the computation logic that corresponds to the computation type',
      'AUTHORIZATION_SCHEME'   , 'An authorization scheme must evaluate to TRUE in order for this computation to be executed',
      'AUTHORIZATION_SCHEME_ID'  , 'Foreign Key',
      'BUILD_OPTION'           , 'Computation will be performend if the Build Option is enabled.',
      'CONDITION_TYPE'         , 'Specifies a condition type from the application computation that conditionally controls whether this computation is performed.',
      'CONDITION_TYPE_CODE'    , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'  , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'  , 'Specifies an expression based on the specific condition type selected.',
      'ERROR_MESSAGE'          , 'Identifies the error text that is displayed if this computation raises an error',
      'COMPONENT_COMMENT'      , 'Developer comment',
      'APPLICATION_COMPUTATION_ID' , 'Identifies the primary key of this Application Computation',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

-------------------------------------------------------
-- A P P L I C A T I O N   S E T T I N G S
--
prompt ...apex_application_settings

create or replace view apex_application_settings
as
select
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       s.id application_setting_id,
       s.name,
       s.is_required,
       s.valid_values,
       s.value,
       case s.on_upgrade_keep_value
       when 'Y' then 'Yes'
       when 'N' then 'No'
       end on_upgrade_keep_value,
       s.comments,
       (select case when s.required_patch > 0 then pa.patch_name else '{Not '||pa.patch_name||'}' end patch_name
        from wwv_flow_patches pa
       where pa.id                = abs(s.required_patch)
         and pa.security_group_id = s.security_group_id) build_option,
       s.required_patch build_option_id,
       s.created_on,
       s.created_by,
       s.last_updated_by,
       s.last_updated_on,
       s.name
       ||'.'||s.is_required
       ||'.'||s.valid_values
       ||'.'||s.value
       component_signature
from wwv_flow_app_settings s,
     wwv_flow_authorized f
where f.workspace_id   = s.security_group_id
  and f.application_id = s.flow_id;

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_settings',
    p_table_comment     => 'Identifies application settings typically used by applications to manage configuration parameter names and values',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application primary key, unique over all workspaces',
      'application_name' , 'Identifies the application',
      'application_setting_id' , 'Identifies the primary key of this Application Setting',
      'name' , 'Identifies the name of the setting used to get and set values',
      'is_required' , 'Identifies if the value is required',
      'valid_values' , 'Identifies comma separated valid values',
      'value' , 'Current value of the setting',
      'on_upgrade_keep_value' , 'Should the current value be kept or set to the specified value when the application is upgraded?',
      'comments' , 'Developer comment',
      'build_option' , 'Application Setting value will be used if the Build Option is enabled',
      'build_option_id' , 'Identifies the primary key of this component',
      'created_by' , 'APEX developer who created the application setting',
      'created_on' , 'Date of application setting was created',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

-------------------------------------------------------
-- A P P L I C A T I O N   S H O R T C U T S
--
prompt ...apex_application_shortcuts

create or replace view apex_application_shortcuts
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    s.SHORTCUT_NAME                  shortcut_name,
    --s.SHORTCUT_CONSIDERATION_SEQ     consideration_sequence,
    s.SHORTCUT_TYPE                  shortcut_type,
    s.shortcut_language              shortcut_language,
    nvl((select r from apex_standard_conditions where d = s.SHORTCUT_CONDITION_TYPE1),s.SHORTCUT_CONDITION_TYPE1)
                                     condition_type,
    s.shortcut_condition_type1       condition_type_code,
    s.SHORTCUT_CONDITION1            condition_expression1,
    --s.SHORTCUT_CONDITION_TYPE2       ,
    s.SHORTCUT_CONDITION2            condition_expression2,
    s.ERROR_TEXT                     error_text,
    (select case when s.build_option > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id =abs(s.BUILD_OPTION)
       and pa.security_group_id = s.security_group_id)      build_option,
    s.SHORTCUT                       shortcut,
    --
    decode(s.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select sh.flow_id||'. '||sh.shortcut_name n
      from wwv_flow_shortcuts sh
     where sh.id                = s.reference_id
       and sh.security_group_id = s.security_group_id)      subscribed_from,
    --
    s.LAST_UPDATED_BY                last_updated_by,
    s.LAST_UPDATED_ON                last_updated_on,
    s.COMMENTS                       component_comments,
    s.id                             shortcut_id,
    --
    s.SHORTCUT_NAME
    ||' '||s.SHORTCUT_TYPE
    ||' cond='||s.SHORTCUT_CONDITION_TYPE1
    ||substr(s.SHORTCUT_CONDITION1,1,30)||length(s.shortcut_condition1)||'.'
    ||substr(s.SHORTCUT_CONDITION2,1,30)||length(s.shortcut_condition2)
    ||' e='||substr(s.ERROR_TEXT,1,30)||length(s.ERROR_TEXT)
    ||' bo='||(select pa.PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(s.BUILD_OPTION)
       and pa.security_group_id = s.security_group_id)
    ||' s='||sys.dbms_lob.substr(s.SHORTCUT,50,1)||sys.dbms_lob.getlength(s.SHORTCUT)
    ||' r='||decode(s.REFERENCE_ID,null,'No','Yes')
    component_signature
from wwv_flow_shortcuts  s,
     wwv_flow_authorized f
where f.workspace_id   = s.security_group_id
  and f.application_id = s.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_shortcuts',
    p_table_comment     => 'Identifies Application Shortcuts which can be referenced "MY_SHORTCUT" syntax',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'SHORTCUT_NAME'          , 'Identifies the Shortcut Name, which can be referenced using "SHORTCUT_NAME" syntax',
      'SHORTCUT_TYPE'          , 'The shortcut type identifies how the Shortcut text is to be interpreted; for example it could by PL/SQL or HTML.',
      'shortcut_language'      , 'Identifies the programming language of the Shortcut text.',
      'CONDITION_TYPE'         , 'Identifies the condition type used to conditionally execute the shortcut',
      'CONDITION_TYPE_CODE'    , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'  , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'  , 'Specifies an expression based on the specific condition type selected.',
      'ERROR_TEXT'             , 'Text to be displayed if this shortcut raises an exception',
      'BUILD_OPTION'           , 'Shortcut will be available if the Build Option is enabled',
      'SHORTCUT'               , 'Text and definition of this Shortcut',
      'IS_SUBSCRIBED'          , 'Identifies if this Shortcut is subscribed from another Shortcut',
      'SUBSCRIBED_FROM'        , 'Identifies the master component from which this component is subscribed',
      'COMPONENT_COMMENTS'     , 'Developer comments',
      'SHORTCUT_ID'            , 'Primary Key of this Shortcut',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   T R E E S
--
prompt ...apex_application_trees

create or replace view apex_application_trees
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.TREE_NAME                      tree_name,
    t.TREE_TYPE                      tree_type,
    t.TREE_QUERY                     tree_query,
    t.FLOW_ITEM                      ,
    t.MAX_LEVELS                     maximum_levels,
    t.UNEXPANDED_PARENT              ,
    t.UNEXPANDED_PARENT_LAST         ,
    t.EXPANDED_PARENT                ,
    t.EXPANDED_PARENT_LAST           ,
    t.LEAF_NODE                      ,
    t.LEAF_NODE_LAST                 ,
    t.DRILL_UP                       ,
    t.NAME_LINK_ANCHOR_TAG           ,
    t.NAME_LINK_NOT_ANCHOR_TAG       ,
    t.INDENT_VERTICAL_LINE           ,
    t.INDENT_VERTICAL_LINE_LAST      ,
    t.BEFORE_TREE                    ,
    t.AFTER_TREE                     ,
    t.LEVEL_1_TEMPLATE               ,
    t.LEVEL_2_TEMPLATE               ,
    t.LEVEL_3_TEMPLATE               ,
    t.LEVEL_4_TEMPLATE               ,
    t.LEVEL_5_TEMPLATE               ,
    t.LEVEL_6_TEMPLATE               ,
    t.LEVEL_7_TEMPLATE               ,
    t.LEVEL_8_TEMPLATE               ,
    --
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    --
    t.TREE_COMMENT                   component_comment,
    t.id                             application_tree_id,
    --
    t.TREE_NAME
    ||' t='||t.TREE_TYPE
    ||substr(t.TREE_QUERY,1,50)||length(t.tree_query)
    ||' i='||t.FLOW_ITEM
    ||' l='||t.MAX_LEVELS
    ||'.'||length(t.UNEXPANDED_PARENT        )
    ||'.'||length(t.UNEXPANDED_PARENT_LAST   )
    ||'.'||length(t.EXPANDED_PARENT          )
    ||'.'||length(t.EXPANDED_PARENT_LAST     )
    ||'.'||length(t.LEAF_NODE                )
    ||'.'||length(t.LEAF_NODE_LAST           )
    ||'.'||length(t.DRILL_UP                 )
    ||'.'||length(t.NAME_LINK_ANCHOR_TAG     )
    ||'.'||length(t.NAME_LINK_NOT_ANCHOR_TAG )
    ||'.'||length(t.INDENT_VERTICAL_LINE     )
    ||'.'||length(t.INDENT_VERTICAL_LINE_LAST)
    ||'.'||length(t.BEFORE_TREE              )
    ||'.'||length(t.AFTER_TREE               )
    ||'.'||length(t.LEVEL_1_TEMPLATE         )
    ||'.'||length(t.LEVEL_2_TEMPLATE         )
    ||'.'||length(t.LEVEL_3_TEMPLATE         )
    ||'.'||length(t.LEVEL_4_TEMPLATE         )
    ||'.'||length(t.LEVEL_5_TEMPLATE         )
    ||'.'||length(t.LEVEL_6_TEMPLATE         )
    ||'.'||length(t.LEVEL_7_TEMPLATE         )
    ||'.'||length(t.LEVEL_8_TEMPLATE         )
    component_signature
from wwv_flow_trees      t,
     wwv_flow_authorized f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_trees',
    p_table_comment     => 'Identifies a tree control which can be referenced and displayed by creating a region with a source of this tree',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'tree_name'                  , 'Component name',
      'tree_type'                  , 'Tree component type',
      'tree_query'                 , 'Query which will be used to generate this tree',
      'flow_item'                  , 'Identifies an application or page item which specifies the starting point of the tree',
      'maximum_levels'             , 'This attribute specifies how many levels will appear when a tree first displays',
      'unexpanded_parent'          , 'HTML template for unexpanded parent nodes',
      'unexpanded_parent_last'     , 'HTML template for the last unexpanded parent node',
      'expanded_parent'            , 'HTML template for the expanded parent node',
      'expanded_parent_last'       , 'HTML template for the last expanded parent node',
      'leaf_node'                  , 'Controls the text that is painted before showing the text of the leaf node',
      'leaf_node_last'             , 'Controls the text that is painted before showing the text of the last leaf node.',
      'drill_up'                   , 'Identifies the link text shown when drilling up is possible in the tree',
      'name_link_anchor_tag'       , 'Identifies the manner in which a Name will render if the name has a link',
      'name_link_not_anchor_tag'   , 'Tag for node names which are not links',
      'indent_vertical_line'       , 'Controls vertical line or spacing between peers',
      'indent_vertical_line_last'  , 'Indent Vertical Line Last, controls a blank space',
      'before_tree'                , 'Text before displaying any nodes of the tree',
      'after_tree'                 , 'Text after displaying nodes of the tree',
      'level_1_template'           , 'Parent Node Template',
      'level_2_template'           , 'Node Text Template',
      'level_3_template'           , 'Text Template for level 3 nodes',
      'level_4_template'           , 'Text Template for level 4 nodes',
      'level_5_template'           , 'Text Template for level 5 nodes',
      'level_6_template'           , 'Text Template for level 6 nodes',
      'level_7_template'           , 'Text Template for level 7 nodes',
      'level_8_template'           , 'Text Template for level 8 nodes',
      'COMPONENT_COMMENT'          , 'Developer comment',
      'APPLICATION_TREE_ID'        , 'Primary Key of this Application Tree shared component',
      'last_updated_by'            , 'APEX developer who made last update',
      'last_updated_on'            , 'Date of last update',
      'component_signature'        , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

prompt ...apex_application_page_trees
set define off

create or replace view apex_application_page_trees
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                             page_id,
    p.name                           page_name,
    t.id                             region_id,
    t.plug_name                      region_name,
    --
    case
    when regexp_like(t.attribute_06, '^tree\d+$') then to_number(regexp_substr(t.attribute_06, '\d+'))
    else t.id
    end                              tree_id,
    t.plug_source                    tree_query,
    regexp_substr (
        t.plug_source,
        'from[:space:]*"?([^"[:space:]]+)',
        1,
        1,
        'i',
        1 )                          tree_source,
    t.attribute_01                   tree_template,
    t.attribute_02                   tree_click_action,
    cast(null as varchar2(10))       tree_start_value,
    cast(null as varchar2(10))       tree_start_item,
    t.attribute_03                   tree_selected_node,
    t.attribute_04                   show_hints,
    t.attribute_05                   tree_hint_text,
    cast(null as varchar2(10))       tree_has_focus,
    cast(null as varchar2(10))       tree_button_option,
    --
    t.last_updated_by                last_updated_by,
    t.last_updated_on                last_updated_on,
    --
    t.plug_comment                   component_comment,
    --
    p.name
    ||' t='||sys.dbms_lob.substr(t.plug_source,50)||sys.dbms_lob.getlength(t.plug_source)
    ||' template='||t.attribute_01
    ||' links='||t.attribute_02
    ||' sel='||t.attribute_03
    ||' hints='||t.attribute_04||':'||t.attribute_05
    component_signature
from wwv_flow_page_plugs t,
     wwv_flow_steps      p,
     wwv_flow_authorized f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = t.security_group_id
  and p.flow_id           = t.flow_id
  and p.id                = t.page_id
  and t.plug_source_type  = 'NATIVE_JSTREE'
/

set define '^'

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_trees',
    p_table_comment     => 'Identifies a tree control which can be referenced and displayed by creating a region with a source of this tree',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_name'                  , 'Page name',
      'page_id'                    , 'Primary Key of this Application page',
      'region_id'                  , 'Identifies the Page Region foreign key to the apex_application_page_regions view',
      'region_name'                , 'Identifies the region name in which this Tree is displayed',
      'tree_id'                    , 'Primary Key of this Tree region',
      'tree_query'                 , 'Query which will be used to generate this tree',
      'tree_source'                , 'Identifies the database table name used as the tree source',
      'tree_template'              , 'The template used to display the tree',
      'tree_click_action'          , 'Identifies whether node links should be activated on a single or double click action',
      'tree_start_value'           , 'obsolete',
      'tree_start_item'            , 'obsolete',
      'tree_selected_node'         , 'To save tree state, identifies the last selected node of the tree',
      'show_hints'                 , 'Identifes whether to display no tooltip or a tooltip based on a database column or static text',
      'tree_hint_text'             , 'The static text to be used as the tooltip for each node of the tree',
      'tree_has_focus'             , 'obsolete',
      'tree_button_option'         , 'obsolete',
      'component_comment'          , 'Developer comment',
      'last_updated_by'            , 'APEX developer who made last update',
      'last_updated_on'            , 'Date of last update',
      'component_signature'        , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
---------------------------------------------------------
-- A P P L I C A T I O N   N A V I G A T I O N   B A R
--
prompt ...apex_application_nav_bar

create or replace view apex_application_nav_bar
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    n.ICON_SEQUENCE                  display_sequence,
    decode(n.BEGINS_ON_NEW_LINE,
       'YES','Yes',
       'NO','No',
       n.BEGINS_ON_NEW_LINE)         begins_on_new_line,
    n.CELL_COLSPAN                   cell_column_span,
    --
    n.ICON_IMAGE                     ,
    --n.ICON_IMAGE2                    ,
    --n.ICON_IMAGE3                    ,
    n.ICON_SUBTEXT                   ,
    --n.ICON_SUBTEXT2                  ,
    --n.ICON_SUBTEXT3                  ,
    n.ICON_TARGET                    ,
    n.ICON_IMAGE_ALT                 ,
    n.ICON_HEIGHT                    ,
    n.ICON_WIDTH                     ,
    --n.ICON_HEIGHT2                   ,
    --n.ICON_WIDTH2                    ,
    --n.ICON_HEIGHT3                   ,
    --n.ICON_WIDTH3                    ,
    --
    nvl((select r from apex_standard_conditions where d = n.ICON_BAR_DISP_COND_TYPE),n.ICON_BAR_DISP_COND_TYPE)
                                     condition_type,
    n.icon_bar_disp_cond_type        condition_type_code,
    n.ICON_BAR_DISP_COND             condition_expression1,
    n.ICON_BAR_FLOW_COND_INSTR       condition_expression2,
    --
    n.ONCLICK                        onclick_javascript,
    --
    (select case when n.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(n.REQUIRED_PATCH)
       and pa.security_group_id = n.security_group_id)    build_option,
     --
    decode(substr(n.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(n.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = n.security_group_id),
     n.SECURITY_SCHEME)              authorization_scheme,
    n.SECURITY_SCHEME                authorization_scheme_id,
    --
    decode(n.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select ib.flow_id||'. '||ib.icon_subtext n
      from wwv_flow_icon_bar ib
     where ib.id                = n.reference_id
       and ib.security_group_id = n.security_group_id)      subscribed_from,
    --
    n.LAST_UPDATED_BY                last_updated_by,
    n.LAST_UPDATED_ON                last_updated_on,
    n.ICON_BAR_COMMENT               component_comment,
    n.id                             nav_bar_id,
    decode(n.nav_entry_is_feedback_yn,'Y','Yes','No') is_feedback_link,
    n.REFERENCE_ID                   referenced_nav_bar_id,
    --
    lpad(n.ICON_SEQUENCE,5,'00000')
    ||decode(n.BEGINS_ON_NEW_LINE,
       'YES','Yes',
       'NO','No',
       n.BEGINS_ON_NEW_LINE)
    ||n.CELL_COLSPAN
    ||' img='||n.ICON_IMAGE
    ||' '||substr(n.ICON_SUBTEXT,1,30)||length(n.ICON_SUBTEXT)
    ||' t='||substr(n.ICON_TARGET,1,30)||length(n.ICON_TARGET)
    ||' a='||substr(n.ICON_IMAGE_ALT,1,20)||length(n.ICON_IMAGE_ALT)
    ||n.ICON_HEIGHT
    ||n.ICON_WIDTH
    ||' c='||n.ICON_BAR_DISP_COND_TYPE
    ||substr(n.ICON_BAR_DISP_COND,1,30)||length(n.ICON_BAR_DISP_COND)
    ||substr(n.ICON_BAR_FLOW_COND_INSTR,1,30)||length(n.ICON_BAR_FLOW_COND_INSTR)
    ||' ck='||substr(n.ONCLICK,1,20)||length(n.onclick)
    ||' bo='||
    (select pa.PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(n.REQUIRED_PATCH)
       and pa.security_group_id = n.security_group_id)
    ||' a='||decode(substr(n.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(n.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = n.security_group_id),
     'Not found')
    ||' s='||decode(n.REFERENCE_ID,
    null,'No','Yes')
    component_signature
from wwv_flow_icon_bar   n,
     wwv_flow_authorized f
where f.workspace_id   = n.security_group_id
  and f.application_id = n.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_nav_bar',
    p_table_comment     => 'Identifies navigation bar entries displayed on pages that use a Page Template that include a #NAVIGATION_BAR# substitution string',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'DISPLAY_SEQUENCE'       , 'Identifies the order in which each Navigation Bar Entry will be displayed',
      'ICON_IMAGE'             , 'Identifies image to be displayed in the Navigation Bar Entry',
      'ICON_SUBTEXT'           , 'Identifies text to be displayed in the Navigation Bar Entry',
      'ICON_TARGET'            , 'Identifies the URL that this Navigation Bar Entry will link to',
      'ICON_IMAGE_ALT'         , 'Identifies the HTML IMG tag ALT value',
      'ICON_HEIGHT'            , 'Identifies the HTML IMG tag HEIGHT value',
      'ICON_WIDTH'             , 'Identifies the HTML IMG tag WIDTH value',
      'CONDITION_TYPE'         , 'Identifies the condition type used to conditionally display the Navigation Bar Entry',
      'CONDITION_TYPE_CODE'    , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'  , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'  , 'Specifies an expression based on the specific condition type selected.',
      'BEGINS_ON_NEW_LINE'     , 'Navigation icons are displayed in table cells, if you set begins-on-new-line to YES then a new table row will be created.',
      'CELL_COLumn_SPAN'       , 'By default each navigation bar entry spans one cell (i.e. the HTML table data colspan=1).  If your HTML requires a different COLSPAN you can set it using this control.',
      'ONCLICK_javascript'     , 'The link generated for this icon include the following onClick javascript.',
      'BUILD_OPTION'           , 'Navigation Bar Entry will be displayed if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'   , 'An authorization scheme must evaluate to TRUE in order for this Navigation Bar Entry to be displayed',
      'AUTHORIZATION_SCHEME_ID' , 'Foreign Key',
      'IS_SUBSCRIBED'          , 'Identifies if this Navigation Bar Entry is subscribed from another Navigation Bar Entry',
      'SUBSCRIBED_FROM'        , 'Identifies the master component from which this component is subscribed',
      'COMPONENT_COMMENT'      , 'Developer comment',
      'NAV_BAR_ID'             , 'Primary Key of this Navigation Bar component',
      'referenced_nav_bar_id'  , 'Primary Key of this Navigation Bar component',
      'is_feedback_link'       , 'Specifies whether navigation bar entry is feedback link',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

----------------------------------------------------------
-- A P P L I C A T I O N   L I S T S    O F    V A L U E S
--

prompt ...apex_application_lovs

create or replace view apex_application_lovs
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    l.LOV_NAME                              list_of_values_name,
    --
    decode(substr(l.lov_query,1,1),
    '.','Static','Dynamic')                 lov_type,
    decode(substr(l.lov_query,1,1),
    '.',null,l.LOV_QUERY)                   list_of_values_query,
    l.source_type                           source_type_code,
    decode( l.source_type,
       'TABLE',                     'Table',
       'SQL',                       'SQL Query',
       'FUNC_BODY_RETURNING_SQL',   'Function Body returning SQL Query',
       'LEGACY_SQL',                'Legacy SQL Query' )
                                            source_type,
    function_body_language                  function_body_language,
    (select count(*)
     from wwv_flow_list_of_values_data vd
    where vd.flow_id           = f.application_id
      and vd.lov_id            = l.id
      and vd.security_group_id = l.security_group_id)                    lov_entries,
    --
    decode(l.REFERENCE_ID,null,'No','Yes')
                                            is_subscribed,
    (select lv.flow_id||'. '||lv.lov_name n
      from wwv_flow_lists_of_values$ lv
     where lv.id                = l.reference_id
       and lv.security_group_id = l.security_group_id)             subscribed_from,
    --
    l.location                              location_code,
    decode( l.location,
       'LOCAL',       'Local Database',
       'REMOTE',      'Remote Database',
       'WEB_SOURCE',  'Web Source',
       'STATIC',      'Static' )            location,
    (select rs.name
       from wwv_remote_servers rs
      where rs.id                = l.remote_server_id
        and rs.security_group_id = l.security_group_id)        remote_database_name,
    l.remote_server_id                      remote_database_id,
    (select wm.name
       from wwv_flow_web_src_modules wm
      where wm.id = l.web_src_module_id
        and wm.security_group_id = l.security_group_id)       web_source_module_name,
    l.web_src_module_id                     web_source_module_id,
    case l.use_local_sync_table
        when 'Y' then 'Yes'
        when 'N' then 'No' end              use_local_sync_table,
    l.query_owner                           table_owner,
    l.query_table                           table_name,
    l.query_where                           where_clause,
    decode( l.source_post_processing,
       'SQL',                       'SQL Query',
       'WHERE_ORDER_BY_CLAUSE',     'Where Clause',
       'FUNC_BODY_RETURNING_SQL',   'Function Body returning SQL Query' )
                                            source_post_processing,
    l.optimizer_hint,
    decode( l.remote_sql_caching,
        'ALL_USERS' , 'Cached'            ,
        'USER'      , 'Cached by User'    ,
        'SESSION'   , 'Cached by Session' ) cache_mode,
    l.remote_sql_invalidate_when            cache_invalidation_expr,
    l.external_filter_expr,
    l.external_order_by_expr,
    --
    l.return_column_name,
    l.display_column_name,
    l.icon_column_name,
    l.group_column_name,
    l.group_sort_direction                  group_sort_direction_code,
    decode( l.group_sort_direction,
        'ASC',              'Ascending (Nulls Last)',
        'ASC_NULLS_FIRST',  'Ascending (Nulls First)',
        'DESC',             'Descending (Nulls Last)',
        'DESC_NULLS_FIRST', 'Descending (Nulls First)' )
                                            group_sort_direction,
    l.default_sort_column_name,
    l.default_sort_direction                default_sort_direction_code,
    decode( l.default_sort_direction,
        'ASC',              'Ascending (Nulls Last)',
        'ASC_NULLS_FIRST',  'Ascending (Nulls First)',
        'DESC',             'Descending (Nulls Last)',
        'DESC_NULLS_FIRST', 'Descending (Nulls First)' )
                                            default_sort_direction,
    l.oracle_text_column_name,
    --
    l.LAST_UPDATED_BY                       last_updated_by,
    l.LAST_UPDATED_ON                       last_updated_on,
    l.LOV_COMMENT                           component_comment,
    --
    l.ID                                    lov_id,
    l.REFERENCE_ID                          referenced_lov_id,
    --
    l.LOV_NAME
    ||' t='||decode(substr(l.lov_query,1,1),'.','Static','Dynamic')
    ||' q='||decode(substr(l.lov_query,1,1),'.',null,substr(l.LOV_QUERY,1,30)||length(l.LOV_QUERY))
    ||' ref='||decode(l.REFERENCE_ID,null,'No','Yes')
    component_signature
from wwv_flow_lists_of_values$ l,
     wwv_flow_authorized       f
where f.workspace_id   = l.security_group_id
  and f.application_id = l.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_lovs',
    p_table_comment     => 'Identifies a shared list of values that can be referenced by a Page Item or Report Column',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                       , 'A work area mapped to one or more database schemas',
      'workspace_display_name'          , 'Display name for the workspace',
      'application_id'                  , 'Application Primary Key, Unique over all workspaces',
      'application_name'                , 'Identifies the application',
      'LIST_OF_VALUES_NAME'             , 'Identifies the name of the List of Values',
      'LIST_OF_VALUES_QUERY'            , 'SQL SELECT statement used to identify dynamic Lists of Values.  Static Lists of Values are defined in a child List of Values Entries table.',
      'LOV_TYPE'                        , 'List of Values type, Static or Dynamic.',
      'source_type_code'                , 'Identifies the internal code of the source type',
      'source_type'                     , 'Identifies the source type the List of Values is based on, can be Table, SQL Query, PL/SQL Function Body returning SQL Query or Legacy SQL Query',
      'function_body_language'          , 'Identifies the programming language of LIST_OF_VALUES_QUERY.',
      'LOV_ENTRIES'                     , 'Count of List of Values static entries',
      'IS_SUBSCRIBED'                   , 'Identifies if this List of Values is subscribed from another List of Values',
      'SUBSCRIBED_FROM'                 , 'Identifies the master component from which this component is subscribed',
      'location_code'                   , 'Identifies the internal code of LOCATION',
      'location'                        , 'Identifies whether the List of Values location is static, the local database, a REST Enabled SQL service or a REST Data Source',
      'remote_database_name'            , 'Identifies the name of the REST Enabled SQL reference',
      'remote_database_id'              , 'Identifies the primary key of the REST Enabled SQL definition',
      'web_source_module_name'          , 'Identifies the REST Data Source name',
      'web_source_module_id'            , 'Identifies the primary key of the REST Data Source',
      'use_local_sync_table'            , 'Whether to use the local synchronization table instead of the REST service',
      'table_owner'                     , 'Denotes the table owner for TABLE query types',
      'table_name'                      , 'Denotes the table name for TABLE query types',
      'where_clause'                    , 'Identifies the WHERE clause provided by the developer and to be applied to Table or REST Data Source',
      'source_post_processing'          , 'Identifies type of SQL post processing for data from external data sources',
      'optimizer_hint'                  , 'Optimizer hint which is added to the region source SQL statement when it is executed',
      'cache_mode'                      , 'Identifies the caching method',
      'cache_invalidation_expr'         , 'DBMS_SCHEDULER calendaring expression to determine cache invalidation',
      'external_filter_expr'            , 'Filter expression to be sent to the external data source',
      'external_order_by_expr'          , 'Order By expression to be sent to the external data source',
      'return_column_name'              , 'Identify the column used as the return value',
      'display_column_name'             , 'Identify the column used as the display value',
      'icon_column_name'                , 'Identify the column used for an icon',
      'group_column_name'               , 'Identify the column used for grouping',
      'group_sort_direction_code'       , 'Identify the internal code that defines the group sort direction',
      'group_sort_direction'            , 'Identify the group sort direction',
      'default_sort_column_name'        , 'Identify the column used as the default sort',
      'default_sort_direction_code'     , 'Identify the internal code that defines the default sort direction',
      'default_sort_direction'          , 'Identify the default sort direction',
      'oracle_text_column_name'         , 'A column which is indexed by an Oracle Text index',
      'component_comment'               , 'Developer comment',
      'lov_id'                          , 'Primary Key of the Shared List of Values',
      'referenced_lov_id'               , 'Foreign Key of referenced Shared List of Values',
      'last_updated_by'                 , 'APEX developer who made last update',
      'last_updated_on'                 , 'Date of last update',
      'component_signature'             , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

prompt ...apex_application_lov_cols

create or replace view apex_application_lov_cols
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    (select lv.lov_name
       from wwv_flow_lists_of_values$ lv
      where lv.id                = c.lov_id
        and lv.security_group_id = c.security_group_id)                  list_of_values_name,
    --
    c.query_column_name                     query_column_name,
    c.heading                               heading,
    c.display_sequence                      display_sequence,
    c.format_mask                           format_mask,
    c.data_type                             data_type,
    c.is_searchable                         is_searchable_code,
    decode( c.is_searchable,
        'Y',    'Yes',
        'N',    'No')                       is_searchable,
    c.is_visible                            is_visible_code,
    decode( c.is_visible,
        'Y',    'Yes',
        'N',    'No')                       is_visible,
    --
    c.last_updated_by                       last_updated_by,
    c.last_updated_on                       last_updated_on,
    c.column_comment                        component_comment,
    --
    c.id                                    lov_column_id,
    c.lov_id                                lov_id,
    --
    (select lv.lov_name from wwv_flow_lists_of_values$ lv where lv.id = c.lov_id and lv.security_group_id = c.security_group_id)
    ||' col='||c.query_column_name
    ||' seq='||lpad(c.display_sequence,5,'00000')
    component_signature
  from wwv_flow_list_of_values_cols c,
       wwv_flow_authorized f
 where f.workspace_id   = c.security_group_id
   and f.application_id = c.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_lov_cols',
    p_table_comment     => 'Identifies column metadata for a shared list of values.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                   , 'A work area mapped to one or more database schemas',
      'workspace_display_name'      , 'Display name for the workspace',
      'application_id'              , 'Application Primary Key, Unique over all workspaces',
      'application_name'            , 'Identifies the application',
      'list_of_values_name'         , 'Identifies the name of the List of Values',
      'query_column_name'           , 'The name of the column from the shared list of values source definition',
      'heading'                     , 'The label used for the column in the dialog / popup',
      'display_sequence'            , 'Identifies the display sequence used by the column',
      'format_mask'                 , 'Format mask for this column',
      'data_type'                   , 'Used as the data type for the column',
      'is_searchable_code'          , 'Identifies the internal code for the searchable flag',
      'is_searchable'               , 'Identifies if the column is searchable',
      'is_visible_code'             , 'Identifies the internal code for whether the column is visible',
      'is_visible'                  , 'Identifies if the column is visible',
      'last_updated_by'             , 'APEX developer who made last update',
      'last_updated_on'             , 'Date of last update',
      'component_comment'           , 'Developer comment',
      'lov_column_id'               , 'Primary Key of the List of Values Column',
      'lov_id'                      , 'Foreign Key of the List of Values',
      'component_signature'         , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

prompt ...apex_application_lov_entires

create or replace view apex_application_lov_entries
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    (select lv.lov_name from wwv_flow_lists_of_values$ lv where lv.id = l.lov_id and lv.security_group_id = l.security_group_id) list_of_values_name,
    l.LOV_DISP_SEQUENCE              display_sequence,
    l.LOV_DISP_VALUE                 display_value,
    l.LOV_RETURN_VALUE               return_value,
    --
    l.LOV_TEMPLATE                   lov_entry_template,
    -- conditionality
    nvl((select r from apex_standard_conditions where d = l.LOV_DISP_COND_TYPE),l.LOV_DISP_COND_TYPE)
                                     condition_type,
    l.lov_disp_cond_type             condition_type_code,
    l.LOV_DISP_COND                  condition_expression1,
    l.LOV_DISP_COND2                 condition_expression2,
    (select case when l.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(l.REQUIRED_PATCH)
       and pa.security_group_id = l.security_group_id)    build_option,
    --
    -- translation columns
    nvl((select tm.PRIMARY_APPLICATION_ID
      from APEX_APPLICATION_TRANS_MAP tm
     where f.application_id = tm.TRANSLATED_APPLICATION_ID),f.application_id) primary_application_id,
    lower(substr(nvl((select TRANSLATED_APP_LANGUAGE
     from   APEX_APPLICATION_TRANS_MAP
     where  f.application_id = TRANSLATED_APPLICATION_ID),f.FLOW_LANGUAGE),1,2)) TRANSLATED_APP_LANGUAGE,
    --
    l.LAST_UPDATED_BY                last_updated_by,
    l.LAST_UPDATED_ON                last_updated_on,
    l.LOV_DATA_COMMENT               component_comment,
    --
    l.LOV_ID                         lov_id,
    l.id                             lov_entry_id,
    --
    (select lv.lov_name from wwv_flow_lists_of_values$ lv where lv.id = l.lov_id and lv.security_group_id = l.security_group_id)
    ||' seq='||lpad(l.LOV_DISP_SEQUENCE,5,'00000')
    ||' v='||substr(l.LOV_DISP_VALUE,1,30)||length(l.LOV_DISP_VALUE)
    ||' r='||substr(l.LOV_RETURN_VALUE,1,30)||length(l.LOV_RETURN_VALUE)
    ||' t='||l.LOV_TEMPLATE
    ||' cond='||l.LOV_DISP_COND_TYPE
    ||substr(l.LOV_DISP_COND,1,30)||length(l.LOV_DISP_COND)
    ||substr(l.LOV_DISP_COND2,1,30)||length(l.LOV_DISP_COND2)
    ||' bo='||(select pa.PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(l.REQUIRED_PATCH)
       and pa.security_group_id = l.security_group_id)
    component_signature
from wwv_flow_list_of_values_data l,
     wwv_flow_authorized          f
where f.workspace_id   = l.security_group_id
  and f.application_id = l.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_lov_entries',
    p_table_comment     => 'Identifies the List of Values Entries which comprise a shared List of Values',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'LIST_OF_VALUES_NAME'    , 'Identifies the name of the List of Values',
      'DISPLAY_SEQUENCE'       , 'List of Values display sequence',
      'DISPLAY_VALUE'          , 'Value displayed to end users',
      'RETURN_VALUE'           , 'Value returned and stored in session state',
      'LOV_ENTRY_TEMPLATE'     , 'Identifies a template used to display a List of Values Entry',
      'CONDITION_TYPE'         , 'Identifies the condition type used to conditionally execute the List of Values Entry',
      'CONDITION_TYPE_CODE'    , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'  , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'  , 'Specifies an expression based on the specific condition type selected.',
      'BUILD_OPTION'           , 'List of Values Entry will be displayed if the Build Option is enabled',
      'COMPONENT_COMMENT'      , 'Developer Comment',
      'LOV_ID'                 , 'Foreign Key of the List of Values',
      'LOV_ENTRY_ID'           , 'Primary Key of the List of Values Entry',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'primary_application_id' , 'Unique ID of the application that is the target of the translation',
      'translated_app_language', 'Language code, for example "fr" or "pt-br"'
    )
  );
end;
/

----------------------------------------------------------------------
-- A P P L I C A T I O N   A U T H E N T I C A T I O N   S C H E M E S
--

prompt ...apex_application_auth

create or replace view apex_application_auth
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    case
      when f.authentication_id = a.id then 'Y'
      else 'N'
    end                              is_current_authentication,
    decode(a.reference_id,
      null,'No','Yes')               is_subscribed,
    case substr(a.scheme_type, 1, 7)
      when 'NATIVE_' then
          ( select np.display_name
              from wwv_flow_plugins np
             where np.security_group_id = 10
               and np.flow_id           = 4411
               and np.plugin_type       = 'AUTHENTICATION TYPE'
               and np.name              = substr(a.scheme_type, 8) )
      when 'PLUGIN_' then
          ( select fp.display_name
              from wwv_flow_plugins fp
             where fp.security_group_id = a.security_group_id
               and fp.flow_id           = a.flow_id
               and fp.plugin_type       = 'AUTHENTICATION TYPE'
               and fp.name              = substr(a.scheme_type, 8) )
      else a.scheme_type
    end                              scheme_type,
    a.scheme_type                    scheme_type_code,
    a.name                           authentication_scheme_name,
    --
    a.invalid_session_type           invalid_session_type,
    a.invalid_session_url            invalid_session_url,
    --
    a.plsql_code                     plsql_code,
    a.verification_function          verification_function,
    a.pre_auth_process               pre_authentication_process,
    a.post_auth_process              post_authentication_process,
    --
    decode(a.switch_in_session_yn,'Y','Enabled','Disabled') switch_in_session,
    --
    a.cookie_name                    cookie_name,
    a.cookie_path                    cookie_path,
    a.cookie_domain                  cookie_domain,
    decode(a.use_secure_cookie_yn,'Y','Yes','N','No','No') cookie_secure,
    --
    a.logout_url                     logout_url,
    --
    a.attribute_01                   attribute_01,
    a.attribute_02                   attribute_02,
    a.attribute_03                   attribute_03,
    a.attribute_04                   attribute_04,
    a.attribute_05                   attribute_05,
    a.attribute_06                   attribute_06,
    a.attribute_07                   attribute_07,
    a.attribute_08                   attribute_08,
    a.attribute_09                   attribute_09,
    a.attribute_10                   attribute_10,
    a.attribute_11                   attribute_11,
    a.attribute_12                   attribute_12,
    a.attribute_13                   attribute_13,
    a.attribute_14                   attribute_14,
    a.attribute_15                   attribute_15,
    --
    case a.ras_mode
      when 0 then 'Disabled'
      when 1 then 'External Users'
      when 2 then 'Internal Users'
    end                              ras_mode,
    a.ras_dynamic_roles              ras_dynamic_roles,
    a.ras_namespaces                 ras_namespaces,
    --
    a.help_text                      help_text,
    a.comments                       comments,
    --
    a.id                             authentication_scheme_id,
    a.reference_id                   referenced_schema_id,
    --
    a.last_updated_by                last_updated_by,
    a.last_updated_on                last_updated_on,
    a.created_by                     created_by,
    a.created_on                     created_on,
    --
    a.name
    || ' '                                       || a.scheme_type
    || ' '                                       || decode(a.reference_id,null,'No','Yes')
    || substr(a.invalid_session_type ,1,20)      || length(a.invalid_session_type )         || '.'
    || substr(a.invalid_session_url ,1,20)       || length(a.invalid_session_url )          || '.'
    || sys.dbms_lob.substr(a.pre_auth_process,20,1)  || sys.dbms_lob.getlength(a.pre_auth_process)  || '.'
    || sys.dbms_lob.substr(a.post_auth_process,20,1) || sys.dbms_lob.getlength(a.post_auth_process) || '.'
    || substr(a.cookie_name         ,1,20)       || length(a.cookie_name         )          || '.'
    || substr(a.cookie_path         ,1,20)       || length(a.cookie_path         )          || '.'
    || substr(a.cookie_domain       ,1,20)       || length(a.cookie_domain       )          || '.'
    || substr(a.use_secure_cookie_yn,1,1)        || length(a.use_secure_cookie_yn)          || '.'
    || substr(a.attribute_01        ,1,20)       || length(a.attribute_01        )          || '.'
    || substr(a.attribute_02        ,1,20)       || length(a.attribute_02        )          || '.'
    || substr(a.attribute_03        ,1,20)       || length(a.attribute_03        )          || '.'
    || substr(a.attribute_04        ,1,20)       || length(a.attribute_04        )          || '.'
    || substr(a.attribute_05        ,1,20)       || length(a.attribute_05        )          || '.'
    || substr(a.attribute_06        ,1,20)       || length(a.attribute_06        )          || '.'
    || substr(a.attribute_07        ,1,20)       || length(a.attribute_07        )          || '.'
    || substr(a.attribute_08        ,1,20)       || length(a.attribute_08        )          || '.'
    || substr(a.attribute_09        ,1,20)       || length(a.attribute_09        )          || '.'
    || substr(a.attribute_10        ,1,20)       || length(a.attribute_10        )          || '.'
    || substr(a.attribute_11        ,1,20)       || length(a.attribute_11        )          || '.'
    || substr(a.attribute_12        ,1,20)       || length(a.attribute_12        )          || '.'
    || substr(a.attribute_13        ,1,20)       || length(a.attribute_13        )          || '.'
    || substr(a.attribute_14        ,1,20)       || length(a.attribute_14        )          || '.'
    || substr(a.attribute_15        ,1,20)       || length(a.attribute_15        )
    component_signature
from wwv_flow_authentications a,
     wwv_flow_authorized      f
where f.workspace_id   = a.security_group_id
  and f.application_id = a.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_auth',
    p_table_comment     => 'Identifies the available Authentication Schemes defined for an Application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                   , 'A work area mapped to one or more database schemas',
      'workspace_display_name'      , 'Display name for the workspace',
      'application_id'              , 'Application Primary Key, Unique over all workspaces',
      'application_name'            , 'Identifies the application',
      'is_current_authentication'   , '"Y" if this is the application''s current authentication scheme, "N" otherwise',
      'is_subscribed'               , 'Identifies if this Authentication Scheme is subscribed from another Authentication Scheme.',
      'scheme_type'                 , 'Type of Authentication Scheme which defines how the Authentication Scheme Source will be interpreted',
      'SCHEME_TYPE_CODE'            , 'Internal code of SCHEME_TYPE',
      'authentication_scheme_name'  , 'Identifies this Authentication Scheme',
      'invalid_session_type'        , 'Specifies a URL Oracle APEX should redirect to if the authentication function returns false: LOGIN - the login page, BUILTIN_LOGIN - the built-in login page, URL - the invalid_session_url',
      'invalid_session_url'         , 'Specifies a URL Oracle APEX should redirect to if the authentication function returns false. This would normally be a URL to a login page or single signon portal.',
      'plsql_code'                  , 'Contains custom authentication code.',
      'verification_function'       , 'The name of the function (within plsql_code, stand-alone or in a package) that will be executed if the session sentry function succeeds. It should return true or false.',
      'pre_authentication_process'  , 'The name of the procedure (within plsql_code, stand-alone or in a package) that will be executed after the login page is submitted and just before credentials verification is performed.',
      'post_authentication_process' , 'The name of the procedure (within plsql_code, stand-alone or in a package) that will be executed after the authentication step (login credentials verification).',
      'switch_in_session'           , 'Specifies whether the authentication scheme can be enabled at runtime',
      'cookie_name'                 , 'Name for the session cookie to be set by the post-login procedure and checked by the Verify Session function.',
      'cookie_path'                 , 'Path attribute for the session cookie to be set by the post-login procedure and checked by the Verify Session function.',
      'cookie_domain'               , 'Domain attribute for the session cookie to be set by the post-login procedure and checked by the Verify Session function. Defaults to null.',
      'cookie_secure'               , 'Specifies whether to set the secure attribute of the session management cookie when sending cookie to browser. Secure cookies can be used only with HTTPS. Defaults to No.',
      'logout_url'                  , 'Specify a URL to become the LOGOUT_URL substitution string.',
      'attribute_01'                , 'Dynamic attribute to store additional data',
      'attribute_02'                , 'Dynamic attribute to store additional data',
      'attribute_03'                , 'Dynamic attribute to store additional data',
      'attribute_04'                , 'Dynamic attribute to store additional data',
      'attribute_05'                , 'Dynamic attribute to store additional data',
      'attribute_06'                , 'Dynamic attribute to store additional data',
      'attribute_07'                , 'Dynamic attribute to store additional data',
      'attribute_08'                , 'Dynamic attribute to store additional data',
      'attribute_09'                , 'Dynamic attribute to store additional data',
      'attribute_10'                , 'Dynamic attribute to store additional data',
      'attribute_11'                , 'Dynamic attribute to store additional data',
      'attribute_12'                , 'Dynamic attribute to store additional data',
      'attribute_13'                , 'Dynamic attribute to store additional data',
      'attribute_14'                , 'Dynamic attribute to store additional data',
      'attribute_15'                , 'Dynamic attribute to store additional data',
      'ras_mode'                    , 'Real Application Security: Disabled, External Users, Internal users',
      'ras_dynamic_roles'           , 'Real Application Security: Colon-separated list of dynamic roles that are enabled on assign_user',
      'ras_namespaces'              , 'Real Application Security: Colon-separated list of namespaces that are enabled on create_session',
      'help_text'                   , 'Text to be displayed in login help popup window off login page. This should offer guidance and links to resources to assist users of the Oracle APEX built-in login page, specific to the type of authentication your application is using (Open Door, Oracle APEX account, or LDAP).',
      'comments'                    , 'Developer comments',
      'authentication_scheme_id'    , 'Primary Key of this component',
      'referenced_schema_id'        , 'Foreign Key to referenced Authentication Scheme',
      'last_updated_by'             , 'APEX developer who made last update',
      'last_updated_on'             , 'Date of last update',
      'created_by'                  , 'APEX developer who created the authentication scheme',
      'created_on'                  , 'Date when the authentication scheme was created',
      'component_signature'         , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

----------------------------------------------------------------------
-- A P P L I C A T I O N   A U T H O R I Z A T I O N   S C H E M E S
-- 4008

prompt ...apex_application_authorization

create or replace view apex_application_authorization
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    a.NAME                           authorization_scheme_name,
    case substr(a.scheme_type, 1, 7)
      when 'NATIVE_' then
          ( select np.display_name from wwv_flow_plugins np where np.security_group_id = 10 and np.flow_id = 4411 and np.plugin_type = 'AUTHORIZATION TYPE' and np.name = substr(a.scheme_type, 8) )
      when 'PLUGIN_' then
          ( select fp.display_name from wwv_flow_plugins fp where fp.security_group_id = a.security_group_id and fp.flow_id = a.flow_id and fp.plugin_type = 'AUTHORIZATION TYPE' and fp.name = substr(a.scheme_type, 8) )
      else a.scheme_type
    end                              scheme_type,
    a.SCHEME_TYPE                    scheme_type_code,
    a.attribute_01,
    a.attribute_02,
    a.attribute_03,
    a.attribute_04,
    a.attribute_05,
    a.attribute_06,
    a.attribute_07,
    a.attribute_08,
    a.attribute_09,
    a.attribute_10,
    a.attribute_11,
    a.attribute_12,
    a.attribute_13,
    a.attribute_14,
    a.attribute_15,
    --
    a.ERROR_MESSAGE                  error_message,
    --
    decode(a.CACHING,
      'BY_USER_BY_PAGE_VIEW','Once per page view',
      'BY_USER_BY_SESSION','Once per session',
      a.CACHING)                     caching,
    decode(a.REFERENCE_ID,null,'No','Yes') is_subscribed,
    a.LAST_UPDATED_BY                last_updated_by,
    a.LAST_UPDATED_ON                last_updated_on,
    a.COMMENTS                       component_comment,
    --
    a.id                             authorization_scheme_id,
    a.REFERENCE_ID                   referenced_scheme_id,
    --
    substr(a.NAME,1,30)||length(a.NAME         )
    ||' t='||substr(a.SCHEME_TYPE  ,1,30)||length(a.SCHEME_TYPE  )
    ||' e='||substr(a.ERROR_MESSAGE,1,30)||length(a.ERROR_MESSAGE)
    ||' s='||decode(a.CACHING,'BY_USER_BY_PAGE_VIEW','Once per page view','BY_USER_BY_SESSION','Once per session',substr(a.CACHING,1,20))
    ||' r='||decode(a.REFERENCE_ID,null,'No','Yes')
    component_signature
from wwv_flow_security_schemes a,
     wwv_flow_authorized       f
where f.workspace_id   = a.security_group_id
  and f.application_id = a.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_authorization',
    p_table_comment     => 'Identifies Authorization Schemes which can be applied at the application, page or component level',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                 , 'A work area mapped to one or more database schemas',
      'workspace_display_name'    , 'Display name for the workspace',
      'application_id'            , 'Application Primary Key, Unique over all workspaces',
      'application_name'          , 'Identifies the application',
      'AUTHORIZATION_SCHEME_NAME' , 'Name of the Authorization Scheme',
      'SCHEME_TYPE'               , 'Type of Authorization Scheme which defines how the Authorization Scheme Source will be interpreted',
      'SCHEME_TYPE_CODE'          , 'Internal code of SCHEME_TYPE',
      'attribute_01'              , 'Dynamic attribute to store additional data',
      'attribute_02'              , 'Dynamic attribute to store additional data',
      'attribute_03'              , 'Dynamic attribute to store additional data',
      'attribute_04'              , 'Dynamic attribute to store additional data',
      'attribute_05'              , 'Dynamic attribute to store additional data',
      'attribute_06'              , 'Dynamic attribute to store additional data',
      'attribute_07'              , 'Dynamic attribute to store additional data',
      'attribute_08'              , 'Dynamic attribute to store additional data',
      'attribute_09'              , 'Dynamic attribute to store additional data',
      'attribute_10'              , 'Dynamic attribute to store additional data',
      'attribute_11'              , 'Dynamic attribute to store additional data',
      'attribute_12'              , 'Dynamic attribute to store additional data',
      'attribute_13'              , 'Dynamic attribute to store additional data',
      'attribute_14'              , 'Dynamic attribute to store additional data',
      'attribute_15'              , 'Dynamic attribute to store additional data',
      'ERROR_MESSAGE'             , 'Identifies the Error Message end users will see when this Authentication Scheme fails',
      'CACHING'                   , 'Identifies the level of Caching used for this Authentication Scheme; typically by Session or Page View.',
      'IS_SUBSCRIBED'             , 'Identifies if this Authorization Scheme is subscribed from another Authorization Scheme.',
      'LAST_UPDATED_BY'           , 'APEX Developer User Name who last updated this component',
      'LAST_UPDATED_ON'           , 'Date this component was last updated',
      'COMPONENT_COMMENT'         , 'Developer comment',
      'AUTHORIZATION_SCHEME_ID'   , 'Primary Key of this Authorization Scheme',
      'referenced_scheme_id'      , 'Foreign Key of referenced Authorization Scheme',
      'last_updated_by'           , 'APEX developer who made last update',
      'last_updated_on'           , 'Date of last update',
      'component_signature'         , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

----------------------------------------------------------
-- A P P L I C A T I O N   W E B  S E R V I C E S
--

prompt ...apex_application_web_services

create or replace view apex_application_web_services
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    w.NAME                                 web_service_name,
    w.URL                                  ,
    w.ACTION                               ,
    w.PROXY_OVERRIDE                       ,
    w.soap_version                         ,
    w.SOAP_ENVELOPE                        ,
    w.FLOW_ITEMS_COMMA_DELIMITED           ,
    w.STATIC_PARM_01                       ,
    w.STATIC_PARM_02                       ,
    w.STATIC_PARM_03                       ,
    w.STATIC_PARM_04                       ,
    w.STATIC_PARM_05                       ,
    w.STATIC_PARM_06                       ,
    w.STATIC_PARM_07                       ,
    w.STATIC_PARM_08                       ,
    w.STATIC_PARM_09                       ,
    w.STATIC_PARM_10                       ,
    w.STYLESHEET                           ,
    --
    decode(w.REFERENCE_ID,
        null,'No','Yes')                   is_subscribed,
    (select ws.flow_id||'. '||ws.name n
       from wwv_flow_shared_web_services ws
      where ws.id = w.id
        and ws.security_group_id = w.security_group_id) subscribed_from,
    --
    w.LAST_UPDATED_BY                      last_updated_by,
    w.LAST_UPDATED_ON                      last_updated_on,
    --
    w.id                                   web_service_id,
    --
    substr(w.name, 1, 30) || length(w.name) ||
    ' u=' || substr(w.url, 1, 30) || length(w.url) ||
    ' a=' || substr(w.action, 1, 30) || ' sv=' || w.soap_version ||
    ' p1=' || substr(w.static_parm_01, 1, 30) || length(w.static_parm_01) ||
    ' p2=' || substr(w.static_parm_02, 1, 30) || length(w.static_parm_02) ||
    ' p3=' || substr(w.static_parm_03, 1, 30) || length(w.static_parm_03)
    as component_signature
from wwv_flow_shared_web_services w,
     wwv_flow_authorized          f
where f.workspace_id   = w.security_group_id
  and f.application_id = w.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_web_services',
    p_table_comment     => 'Web Services referenceable from this Application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                   , 'A work area mapped to one or more database schemas',
      'workspace_display_name'      , 'Display name for the workspace',
      'application_id'              , 'Application Primary Key, Unique over all workspaces',
      'application_name'            , 'Identifies the application',
      'WEB_SERVICE_NAME'            , 'Identifies the name of the Web Service',
      'URL'                         , 'Specifies the URL used to post the SOAP request over HTTP',
      'ACTION'                      , 'Indicates the intent of the SOAP HTTP request',
      'PROXY_OVERRIDE'              , 'Overrides the system defined HTTP proxy for request',
      'soap_version'                , 'Identifies the SOAP version of the service, 1.1 or 1.2',
      'SOAP_ENVELOPE'               , 'Specifies the SOAP envelope to be used for the SOAP request to the Web service',
      'FLOW_ITEMS_COMMA_DELIMITED'  , 'Comma delimited list of application items',
      'STATIC_PARM_01'              , 'Identifies static parameters',
      'STATIC_PARM_02'              , 'Identifies static parameters',
      'STATIC_PARM_03'              , 'Identifies static parameters',
      'STATIC_PARM_04'              , 'Identifies static parameters',
      'STATIC_PARM_05'              , 'Identifies static parameters',
      'STATIC_PARM_06'              , 'Identifies static parameters',
      'STATIC_PARM_07'              , 'Identifies static parameters',
      'STATIC_PARM_08'              , 'Identifies static parameters',
      'STATIC_PARM_09'              , 'Identifies static parameters',
      'STATIC_PARM_10'              , 'Identifies static parameters',
      'STYLESHEET'                  , 'Stylesheet will be used to apply an XML transformation against the result of the SOAP Request',
      'IS_SUBSCRIBED'               , 'Identifies if this Web Service is subscribed from another Web Service',
      'SUBSCRIBED_FROM'             , 'Identifies the master component from which this component is subscribed',
      'web_service_id'              , 'Identifies the primary key of the Web Service',
      'last_updated_by'             , 'APEX developer who made last update',
      'last_updated_on'             , 'Date of last update',
      'component_signature'         , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N    B R E A D C R U M B S
--

prompt ...apex_application_breadcrumbs

create or replace view apex_application_breadcrumbs
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    m.NAME                           breadcrumb_name,
    --
    (select count(*) from wwv_flow_menu_options mo where mo.menu_id = m.id and mo.security_group_id = m.security_group_id) breadcrumb_entries,
    --
    m.LAST_UPDATED_BY                last_updated_by,
    m.LAST_UPDATED_ON                last_updated_on,
    m.MENU_COMMENT                   component_comment,
    m.id                             breadcrumb_id,
    --
    m.NAME
    component_signature
from wwv_flow_menus      m,
     wwv_flow_authorized f
where f.workspace_id   = m.security_group_id
  and f.application_id = m.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_breadcrumbs',
    p_table_comment     => 'Identifies the definition of a collection of Breadcrumb Entries which are used to identify a page Hierarchy',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'          , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'     , 'Application Primary Key, Unique over all workspaces',
      'application_name'   , 'Identifies the application',
      'BREADCRUMB_name'    , 'Identifies the Breadcrumb Name, a breadcrumb is a collection of Breadcrumb Entries used to show page context.',
      'BREADCRUMB_ENTRIES' , 'Count of Entries defined for this Breadcrumb',
      'COMPONENT_COMMENT'  , 'Developer Comment',
      'BREADCRUMB_ID'      , 'Primary key of this Breadcrumb',
      'last_updated_by'    , 'APEX developer who made last update',
      'last_updated_on'    , 'Date of last update',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

prompt ...apex_application_bc_entries

create or replace view apex_application_bc_entries
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    m.PARENT_ID                      parent_breadcrumb_id,
    m.MENU_ID                        breadcrumb_id,
    --m.OPTION_SEQUENCE                display_sequence,
    m.SHORT_NAME                     entry_label,
    m.LONG_NAME                      entry_long_label,
    m.LINK                           url,
    m.PAGE_ID                        defined_for_page,
    m.ALSO_CURRENT_FOR_PAGES         current_for_pages,
    --
    nvl((select r from apex_standard_conditions where d = m.DISPLAY_WHEN_COND_TYPE),m.DISPLAY_WHEN_COND_TYPE)
                                     condition_type,
    m.display_when_cond_type         condition_type_code,
    m.DISPLAY_WHEN_CONDITION         condition_expression1,
    m.DISPLAY_WHEN_CONDITION2        condition_expression2,
    --
    decode(substr(m.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(m.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = m.security_group_id),
     m.SECURITY_SCHEME)              authorization_scheme,
    m.SECURITY_SCHEME                authorization_scheme_id,
     --
    (select case when m.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(m.REQUIRED_PATCH)
       and pa.security_group_id = m.security_group_id)   build_option,
    m.LAST_UPDATED_BY                last_updated_by,
    m.LAST_UPDATED_ON                last_updated_on,
    m.MENU_OPTION_COMMENT            component_comment,
    m.id                             breadcrumb_entry_id,
    --
    substr(m.SHORT_NAME,1,30)||length(m.SHORT_NAME)
    ||substr(m.LONG_NAME,1,30)||length(m.LONG_NAME)
    ||' l='||substr(m.LINK,1,30)||length(m.LINK)
    ||' p='||PAGE_ID
    ||' c='||substr(m.ALSO_CURRENT_FOR_PAGES,1,30)||length(m.ALSO_CURRENT_FOR_PAGES)
    ||' cond='||m.DISPLAY_WHEN_COND_TYPE
    ||substr(m.DISPLAY_WHEN_CONDITION,1,30)||length(m.DISPLAY_WHEN_CONDITION)
    ||substr(m.DISPLAY_WHEN_CONDITION2,1,30)||length(m.DISPLAY_WHEN_CONDITION2)
    ||' sec='||decode(substr(m.SECURITY_SCHEME,1,1),'!','Not ')||
      nvl((select ss.name from wwv_flow_security_schemes ss where   to_char(ss.id)= ltrim(m.SECURITY_SCHEME,'!') and ss.flow_id = f.application_id and ss.security_group_id = m.security_group_id), 'Not found')
    ||' bo='||(select PATCH_NAME
     from   wwv_flow_patches
     where  id =abs(m.REQUIRED_PATCH) )
    component_signature
from wwv_flow_menu_options m,
     wwv_flow_authorized   f
where f.workspace_id   = m.security_group_id
  and f.application_id = m.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_bc_entries',
    p_table_comment     => 'Identifies Breadcrumb Entries which map to a Page and identify a pages parent',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'             , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'        , 'Application Primary Key, Unique over all workspaces',
      'application_name'      , 'Identifies the application',
      'PARENT_BREADCRUMB_ID'  , 'Identifies the Parent Breadcrumb ID.',
      'BREADCRUMB_ID'         , 'Identifies the Primary Key of this Breadcrumb Entry',
      'ENTRY_LABEL'           , 'Identifies the label which will be displayed for this Breadcrumb Entry',
      'ENTRY_LONG_LABEL'      , 'Identifies an optional longer version of the Breadcrumb Entry',
      'URL'                   , 'The URL corresponding to a breadcrumb entry.  When a user clicks the breadcrumb they will navigate to this URL.',
      'DEFINED_FOR_PAGE'      , 'Identifies the pages for which this Breadcrumb Entry is defined',
      'CURRENT_FOR_PAGES'     , 'Identifies a page or pages for which this breadcrumb is current.  Current and non current breadcrumbs have different templates and may be rendered differently.',
      'CONDITION_TYPE'        , 'For breadcrumbs that are displayed conditionally identifies the condition type',
      'CONDITION_TYPE_CODE'   , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1' , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2' , 'Specifies an expression based on the specific condition type selected.',
      'AUTHORIZATION_SCHEME'  , 'An authorization scheme must evaluate to TRUE in order for this page to be displayed',
      'AUTHORIZATION_SCHEME_ID' , 'Foreign Key',
      'BUILD_OPTION'          , 'Breadcrumb Entry will be displayed if the Build Option is enabled',
      'COMPONENT_COMMENT'     , 'Developer comment',
      'BREADCRUMB_ENTRY_ID'   , 'Foreign key identifying the Breadcrumb definition',
      'last_updated_by'       , 'APEX developer who made last update',
      'last_updated_on'       , 'Date of last update',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
---------------------------------------------------------
-- A P P L I C A T I O N   L I S T S
--

prompt ...apex_application_lists

create or replace view apex_application_lists
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    l.NAME                           list_name,
    decode(l.list_type,null,'Static',
                      'STATIC','Static',
                      'SQL_QUERY','SQL Query',
                      'FUNCTION_RETURNING_SQL_QUERY','Function Returning SQL Query') list_type,
    decode(l.list_type,null,'STATIC',l.list_type) list_type_code,
    l.list_language                  list_language,
    l.list_query                     list_query,
    --l.LIST_STATUS                    list_status,
    --l.LIST_DISPLAYED,
    (select case when l.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
       from wwv_flow_patches pa
      where pa.id                = abs(l.REQUIRED_PATCH)
        and pa.security_group_id = l.security_group_id)    build_option,
    (select count(*) from wwv_flow_list_items li where li.list_id = l.id and li.security_group_id = l.security_group_id) list_entries,
    --
    decode(l.reference_id, null,'No','Yes') is_subscribed,
    (select flow_id || '. ' || name n
      from wwv_flow_lists
     where id = l.reference_id) subscribed_from,
    l.reference_id subscribed_from_id,
    --
    l.LAST_UPDATED_BY                last_updated_by,
    l.LAST_UPDATED_ON                last_updated_on,
    l.LIST_COMMENT                   component_comment,
    l.ID                             list_id,
    --
    l.NAME
    ||' s='||l.LIST_STATUS
    ||' bo='||(select pa.PATCH_NAME
      from wwv_flow_patches pa
     where pa.id                = abs(l.REQUIRED_PATCH)
       and pa.security_group_id = l.security_group_id)
    component_signature,
    --
    -- For compatibility with SQL Developer
    cast(null as varchar2(255)) template
from wwv_flow_lists      l,
     wwv_flow_authorized f
where f.workspace_id   = l.security_group_id
  and f.application_id = l.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_lists',
    p_table_comment     => 'Identifies a named collection of Application List Entries which can be included on any page using a region of type List.  Display attributes are controlled using a List Template.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'         , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'    , 'Application Primary Key, Unique over all workspaces',
      'application_name'  , 'Identifies the application',
      'LIST_NAME'         , 'Identifies the name of the Application List',
      'LIST_TYPE'         , 'Identifies the type of the Application List',
      'LIST_TYPE_CODE'    , 'Internal code of LIST_TYPE',
      'list_language'     , 'Identifies the programming language of LIST_QUERY.',
      'LIST_QUERY'        , 'Identifies the query of a Dynamic Application List',
      'BUILD_OPTION'      , 'List will be displayed if the Build Option is enabled',
      'COMPONENT_COMMENT' , 'Developer Comment',
      'LIST_ID'           , 'Identifies the primary key of this component',
      'last_updated_by'   , 'APEX developer who made last update',
      'last_updated_on'   , 'Date of last update',
      'list_entries'      , 'Number of list entries defined for list',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'TEMPLATE'          , 'Identifies the List Template to use by default when rendering this List within a Region.  Regions may override this default template. (obsolete)',
      'is_subscribed'     , 'Identifies if this list is subscribed from another list',
      'subscribed_from'   , 'Identifies the master list from which this list is subscribed',
      'subscribed_from_id', 'Id of the master list from which this list is subscribed'
    )
  );
end;
/
create or replace view apex_application_list_entries
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    l.NAME                           list_name,
    --
    --e.SUB_ITEM_COUNT                 ,
    --e.PARENT_LIST_ITEM_ID            ,
    (select li.list_item_link_text
       from wwv_flow_list_items li
      where li.id                = e.PARENT_LIST_ITEM_ID
        and li.flow_id           = e.flow_id
        and li.security_group_id = e.security_group_id)     parent_entry_text,
    --e.LIST_ITEM_TYPE                 ,
    --e.LIST_ITEM_STATUS               ,
    --e.ITEM_DISPLAYED                 ,
    e.LIST_ITEM_DISPLAY_SEQUENCE     display_sequence,
    e.LIST_ITEM_LINK_TEXT            entry_text,
    e.LIST_ITEM_LINK_TARGET          entry_target,
    --
    e.LIST_ITEM_ICON                 entry_image,
    e.LIST_ITEM_ICON_ATTRIBUTES      entry_image_attributes,
    e.LIST_ITEM_ICON_ALT_ATTRIBUTE   entry_image_alt_attribute,
    --e.LIST_ITEM_ICON_EXP             ,
    --e.LIST_ITEM_ICON_EXP_ATTR        ,
    --e.LIST_ITEM_OWNER                ,
    --
    decode(e.LIST_ITEM_CURRENT_TYPE,
      'ALWAYS',                      'Always',
      'COLON_DELIMITED_PAGE_LIST',   'Colon Delimited Page List',
      'EXISTS',                      'Exists SQL Query',
      'NEVER',                       'Never',
      'NOT_EXISTS',                  'Not Exists SQL Query',
      'EXPRESSION',                  'Expression',
      'PAGE_RANGE',                  'Page Range (X-Y)',
      e.LIST_ITEM_CURRENT_TYPE)      current_for_pages_type,
    e.list_item_current_type         current_for_pages_type_code,
    e.list_item_current_language     current_for_pages_language,
    e.LIST_ITEM_CURRENT_FOR_PAGES    current_for_pages_expression,
    --
    nvl((select r from apex_standard_conditions where d = e.LIST_ITEM_DISP_COND_TYPE),e.LIST_ITEM_DISP_COND_TYPE)
                                     condition_type,
    e.list_item_disp_cond_type       condition_type_code,
    e.LIST_ITEM_DISP_CONDITION       condition_expression1,
    --e.LIST_ITEM_DISP_COND_TYPE2      ,
    e.LIST_ITEM_DISP_CONDITION2      condition_expression2,
    --
    decode(e.LIST_COUNTCLICKS_Y_N,
      'Y','Yes','N','No',
      e.LIST_COUNTCLICKS_Y_N)        count_clicks,
    e.LIST_COUNTCLICKS_CAT           click_count_category,
    --
    e.LIST_TEXT_01                   entry_attribute_01,
    e.LIST_TEXT_02                   entry_attribute_02,
    e.LIST_TEXT_03                   entry_attribute_03,
    e.LIST_TEXT_04                   entry_attribute_04,
    e.LIST_TEXT_05                   entry_attribute_05,
    e.LIST_TEXT_06                   entry_attribute_06,
    e.LIST_TEXT_07                   entry_attribute_07,
    e.LIST_TEXT_08                   entry_attribute_08,
    e.LIST_TEXT_09                   entry_attribute_09,
    e.LIST_TEXT_10                   entry_attribute_10,
    e.translate_list_text_y_n        translate_attributes,
    --e.LIST_TEXT_11                   entry_attribute_11,
    --e.LIST_TEXT_12                   entry_attribute_12,
    --e.LIST_TEXT_13                   entry_attribute_13,
    --e.LIST_TEXT_14                   entry_attribute_14,
    --e.LIST_TEXT_15                   entry_attribute_15,
    --e.LIST_TEXT_16                   entry_attribute_16,
    --e.LIST_TEXT_17                   entry_attribute_17,
    --e.LIST_TEXT_18                   entry_attribute_18,
    --e.LIST_TEXT_19                   entry_attribute_19,
    --e.LIST_TEXT_20                   entry_attribute_20,
    --
    (select case when e.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
       from wwv_flow_patches pa
      where pa.id                = abs(e.REQUIRED_PATCH)
        and pa.security_group_id = e.security_group_id)    build_option,
    --
    decode(substr(e.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(e.SECURITY_SCHEME,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = e.security_group_id),
     e.SECURITY_SCHEME)              authorization_scheme,
    e.SECURITY_SCHEME                authorization_scheme_id,
    --
    -- translation columns
    nvl((select PRIMARY_APPLICATION_ID
     from   APEX_APPLICATION_TRANS_MAP
     where  f.application_id = TRANSLATED_APPLICATION_ID),f.application_id) primary_application_id,
    lower(substr(nvl((select TRANSLATED_APP_LANGUAGE
     from   APEX_APPLICATION_TRANS_MAP
     where  f.application_id = TRANSLATED_APPLICATION_ID),f.FLOW_LANGUAGE),1,2)) TRANSLATED_APP_LANGUAGE,
    --
    e.LAST_UPDATED_BY                last_updated_by,
    e.LAST_UPDATED_ON                last_updated_on,
    e.LIST_ITEM_COMMENT              component_comment,
    e.list_id                        list_id,
    e.PARENT_LIST_ITEM_ID            list_entry_parent_id,
    e.id                             list_entry_id,
    --
    l.NAME
    ||' p='||(select li.list_item_link_text from wwv_flow_list_items li where  li.id = e.PARENT_LIST_ITEM_ID and li.flow_id = e.flow_id and li.security_group_id = l.security_group_id)
    ||' '||lpad(e.LIST_ITEM_DISPLAY_SEQUENCE,5,'00000')
    ||' '||substr(e.LIST_ITEM_LINK_TEXT      ,1,30)
    ||' '||substr(e.LIST_ITEM_LINK_TARGET    ,1,30)||length(e.LIST_ITEM_LINK_TARGET)
    ||' '||substr(e.LIST_ITEM_ICON           ,1,30)
    ||' '||substr(e.LIST_ITEM_ICON_ATTRIBUTES,1,30)||length(e.LIST_ITEM_ICON_ATTRIBUTES)
    ||' '||decode(e.LIST_ITEM_CURRENT_TYPE,
      'ALWAYS',                      'Always',
      'COLON_DELIMITED_PAGE_LIST',   'Colon Delimited Page List',
      'EXISTS',                      'Exists SQL Query',
      'NEVER',                       'Never',
      'NOT_EXISTS',                  'Not Exists SQL Query',
      'EXPRESSION',                  'Expression',
      e.LIST_ITEM_CURRENT_TYPE)
    ||' '||substr(e.LIST_ITEM_CURRENT_FOR_PAGES,1,30)||length(e.LIST_ITEM_CURRENT_FOR_PAGES)
    ||' c='||e.LIST_ITEM_DISP_COND_TYPE
    ||substr(e.LIST_ITEM_DISP_CONDITION,1,30)||length(e.LIST_ITEM_DISP_CONDITION)
    ||substr(e.LIST_ITEM_DISP_CONDITION2,1,30)||length(e.LIST_ITEM_DISP_CONDITION2)
    ||' c='||decode(e.LIST_COUNTCLICKS_Y_N,
      'Y','Yes','N','No',
      e.LIST_COUNTCLICKS_Y_N)
    ||' c='||e.LIST_COUNTCLICKS_CAT
    ||' t='
    ||length(e.LIST_TEXT_01)
    ||length(e.LIST_TEXT_02)
    ||length(e.LIST_TEXT_03)
    ||length(e.LIST_TEXT_04)
    ||length(e.LIST_TEXT_05)
    ||length(e.LIST_TEXT_06)
    ||length(e.LIST_TEXT_07)
    ||length(e.LIST_TEXT_08)
    ||length(e.LIST_TEXT_09)
    ||length(e.LIST_TEXT_10)
    ||translate_list_text_y_n
    ||' b='||(select pa.PATCH_NAME
      from wwv_flow_patches pa
     where pa.id =abs(e.REQUIRED_PATCH)
       and pa.security_group_id = e.security_group_id)
    ||' s='||decode(substr(e.SECURITY_SCHEME,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id) = ltrim(e.SECURITY_SCHEME,'!')
       and ss.flow_id = f.application_id
       and ss.security_group_id = e.security_group_id),
     'Not found')
    component_signature
from wwv_flow_list_items e,
     wwv_flow_lists      l,
     wwv_flow_authorized f
where f.workspace_id      = l.security_group_id
  and f.application_id    = l.flow_id
  and l.security_group_id = e.security_group_id
  and l.id                = e.list_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_list_entries',
    p_table_comment     => 'Identifies the List Entries which define a List.  List Entries can be hierarchical or flat.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'         , 'A work area mapped to one or more database schemas',
      'workspace_display_name'       , 'Display name for the workspace',
      'application_id'    , 'Application Primary Key, Unique over all workspaces',
      'application_name'  , 'Identifies the application',
      'LIST_NAME'                    , 'Name of the Application List this List Entry is part of',
      'PARENT_entry_TEXT'            , 'Identifies the Parent of this List Entry',
      'DISPLAY_SEQUENCE'             , 'Identifies the display sequence',
      'entry_TEXT'                   , 'Identifies the Link Text which will be displayed to the end user of the application',
      'entry_TARGET'                 , 'Identifies the URL target of this List Entry',
      'entry_IMAGE'                  , 'Identifies an optional image associated with this List Entry',
      'entry_IMAGE_ATTRIBUTES'       , 'Identifies image attributes that will be rendered within the HTML IMG tag for list entries that include Link Images',
      'entry_IMAGE_ALT_ATTRIBUTE'    , 'Identifies an optional image alt attribute associated with this List Entry image',
      'CURRENT_FOR_PAGES_TYPE'       , 'Identifies a Condition Type used to determine if this List Entry is current; reference Current For Pages Expression attribute',
      'CURRENT_FOR_PAGES_TYPE_CODE'  , 'Internal code of CURRENT_FOR_PAGES_TYPE',
      'current_for_pages_language'   , 'Identifies the programming language of CURRENT_FOR_PAGES_EXPRESSION.',
      'CURRENT_FOR_PAGES_EXPRESSION' , 'Identifies a page or pages (or other condition) used to determine if this List Entry is to be rendered as a Current List Entry',
      'CONDITION_TYPE'               , 'For conditionally displayed list entries; identifies the condition type.  The condition is specified in the Expression 1 and Expression 2 attributes.',
      'CONDITION_TYPE_CODE'          , 'Internal code of CONDITION_TYPE.',
      'CONDITION_EXPRESSION1'        , 'Specifies an expression based on the specific condition type selected.',
      'CONDITION_EXPRESSION2'        , 'Specifies an expression based on the specific condition type selected.',
      'COUNT_CLICKS'                 , 'Determines if the APEX click counter log should record a click entry when this list entry is clicked',
      'CLICK_COUNT_CATEGORY'         , 'Determines the APEX click counter category used to provide greater context to the click count log',
      'ENTRY_ATTRIBUTE_01'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_02'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_03'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_04'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_05'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_06'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_07'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_08'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_09'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'ENTRY_ATTRIBUTE_10'           , 'Identifies attributes which may be referenced in the corresponding List Template',
      'translate_attributes'         , 'Include attributes 1 through 10 in a generated XLIFF translation file',
      'BUILD_OPTION'                 , 'List Entry will be displayed if the Build Option is enabled',
      'AUTHORIZATION_SCHEME'         , 'An authorization scheme must evaluate to TRUE in order for this page to be displayed',
      'AUTHORIZATION_SCHEME_ID'      , 'Foreign Key',
      'LAST_UPDATED_BY'              , 'APEX User who last updated this component',
      'LAST_UPDATED_ON'              , 'Date on which this component was last updated last',
      'COMPONENT_COMMENT'            , 'Developer Comment',
      'last_updated_by'              , 'APEX developer who made last update',
      'last_updated_on'              , 'Date of last update',
      'list_id'                      , 'Foreign key of the List',
      'list_entry_parent_id'         , 'Foreign key of the Parent List Entry',
      'list_entry_id'                , 'Primary Key of this List Entry',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'primary_application_id'       , 'Unique ID of the application that is the target of the translation',
      'translated_app_language'      , 'Language code, for example "fr" or "pt-br"'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   R E P O R T  Q U E R I E S
--

prompt ...apex_application_rpt_queries

create or replace view apex_application_rpt_queries
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    q.name                           report_query_name,
    q.query_text                     query_text,
    q.xml_structure                  xml_structure,
    (select rl.report_layout_name from wwv_flow_report_layouts rl where rl.id = q.report_layout_id and rl.security_group_id = q.security_group_id) report_layout_name,
    q.format                         output_format,
    q.format_item                    output_format_item,
    q.output_file_name               output_file_name,
    q.content_disposition            content_disposition,
    q.document_header                document_header,
    q.xml_items                      xml_items,
    --
    q.last_updated_by                last_updated_by,
    q.last_updated_on                last_updated_on,
    --
    q.id                             report_query_id,
    --
    q.name
    ||' xs='||q.xml_structure
    ||' l=' ||report_layout_id
    ||' f=' ||q.format
    ||' fi='||q.format_item
    ||' o=' ||q.output_file_name
    ||' c=' ||q.content_disposition
    ||' d=' ||q.document_header
    ||' xi='||substr(q.xml_items,1,30)||length(q.xml_items)
    component_signature
from wwv_flow_shared_queries q,
     wwv_flow_authorized f
where f.workspace_id   = q.security_group_id
  and f.application_id = q.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_rpt_queries',
    p_table_comment     => 'Identifies report queries, which are printable documents that can be integrated with an application using buttons, list items, branches',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'report_query_name'      , 'Identifies the name of the report query',
      'query_text'             , 'Identifies SQL query of the report query',
      'xml_structure'          , 'Identifies XML structure of report data: standard or apex',
      'report_layout_name'     , 'Identifies the name of the repory layout',
      'output_format'          , 'Identifies the output format of the report',
      'output_format_item'     , 'Identifies the page item that can be used to define the output format string',
      'output_file_name'       , 'Identifies the output file name of the report',
      'content_disposition'    , 'Identifies the content disposition of the download file',
      'document_header'        , 'Identifies the document header of the download file',
      'xml_items'              , 'Identifies the item session state to be included in the XML data',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'report_query_id'        , 'Identified the prinmary key of this component',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   R E P O R T  Q U E R Y  S Q L  S T A T E M E N T S
--

prompt ...apex_application_rpt_qry_stmts

create or replace view apex_application_rpt_qry_stmts
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    q.shared_query_id                report_query_id,
    (select sq.name from wwv_flow_shared_queries sq where sq.id = q.shared_query_id and sq.security_group_id = q.security_group_id) report_query_name,
    sql_statement                    sql_statement,
    --
    q.last_updated_by                last_updated_by,
    q.last_updated_on                last_updated_on,
    --
    q.id                             report_query_stmt_id,
    --
    (select sq.name from wwv_flow_shared_queries sq where sq.id = q.shared_query_id and sq.security_group_id = q.security_group_id)
    ||' q=' ||substr(q.sql_statement,1,30)||length(q.sql_statement)
    component_signature
from wwv_flow_shared_qry_sql_stmts q,
     wwv_flow_authorized f
where f.workspace_id   = q.security_group_id
  and f.application_id = q.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_rpt_qry_stmts',
    p_table_comment     => 'Identifies inidividual SQL statements, which are part of a report quert',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'              , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'         , 'Application Primary Key, Unique over all workspaces',
      'application_name'       , 'Identifies the application',
      'report_query_id'        , 'ID of the report query',
      'report_query_name'      , 'Identifies the name of the report query',
      'sql_statement'          , 'Inidividual SQL statements',
      'last_updated_by'        , 'APEX developer who made last update',
      'last_updated_on'        , 'Date of last update',
      'report_query_stmt_id'   , 'Identified the prinmary key of this component',
      'component_signature'    , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   R E P O R T  L A Y O U T S
--

prompt ...apex_application_rpt_layouts

create or replace view apex_application_rpt_layouts
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    l.report_layout_name                report_layout_name,
    l.report_layout_type                report_layout_type,
    l.page_template                     page_template,
    l.xslfo_column_heading_template     xslfo_column_heading_template,
    l.xslfo_column_template             xslfo_column_template,
    l.xslfo_column_template_width       xslfo_column_template_width,
    l.report_layout_comment             report_layout_comment,
    --
    l.last_updated_by                   last_updated_by,
    l.last_updated_on                   last_updated_on,
    --
    l.id                                report_layout_id,
    --
    l.report_layout_name
    ||' t=' ||l.report_layout_type
    ||' ht='||substr(l.xslfo_column_heading_template,1,30)||length(l.xslfo_column_heading_template)
    ||' ct='||substr(l.xslfo_column_template,1,30)||length(l.xslfo_column_template)
    ||' w=' ||substr(l.xslfo_column_template_width,1,30)||length(l.xslfo_column_template_width)
    ||' c=' ||substr(l.report_layout_comment,1,30)||length(l.report_layout_comment)
    component_signature
from wwv_flow_report_layouts l,
     wwv_flow_authorized f
where f.workspace_id   = l.security_group_id
  and f.application_id = l.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_rpt_layouts',
    p_table_comment     => 'Identifies report layout which can be referenced by report queries and classic reports',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                      , 'A work area mapped to one or more database schemas',
      'workspace_display_name'         , 'Display name for the workspace',
      'application_id'                 , 'Application Primary Key, Unique over all workspaces',
      'application_name'               , 'Identifies the application',
      'report_layout_name'             , 'Identifies the name of the report layout',
      'report_layout_type'             , 'Identifies the type of the report layout',
      'page_template'                  , 'Identifies the page template',
      'xslfo_column_heading_template'  , 'Identifies the column heading template',
      'xslfo_column_template'          , 'Identifies the column template',
      'xslfo_column_template_width'    , 'Identifies the width of the column template',
      'report_layout_comment'          , 'Identifies a commnet for the report layout',
      'last_updated_by'                , 'APEX developer who made last update',
      'last_updated_on'                , 'Date of last update',
      'report_layout_id'               , 'Identified the prinmary key of this component',
      'component_signature'            , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
---------------------------------------------------------
-- A P P L I C A T I O N   T H E M E S
--
prompt ...apex_application_themes

create or replace view apex_application_themes
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.theme_id                                                                                               theme_number,
    t.theme_name                                                                                             theme_name,
    t.theme_internal_name                                                                                    theme_internal_name,
    t.ui_type_id,
    uit.name                                                                                                 ui_type_name,
    t.reference_id,
    case t.navigation_type
      when 'T'    then 'Tabs'
      when 'L'    then 'Navigation Menu List'
    end as navigation_type,
    t.navigation_type as navigation_type_code,
    t.default_nav_list_position,
    case t.nav_bar_type
      when 'NAVBAR'    then 'Navigation Bar'
      when 'LIST'      then 'List'
    end as nav_bar_type,
    t.nav_bar_type as nav_bar_type_code,
    t.is_locked,
    case
      when exists ( select 1
                      from wwv_flow_user_interfaces ui
                     where ui.flow_id           = t.flow_id
                       and ui.theme_id          = t.theme_id
                       and ui.security_group_id = t.security_group_id) then 'Yes'
      else 'No'
    end as is_current,
    (select tl.name from wwv_flow_templates tl where tl.id = t.default_page_template and tl.security_group_id = t.security_group_id) default_page_template,
    --t.error_template                   ,
    --t.printer_friendly_template        ,
    --t.breadcrumb_display_point         ,
    --t.sidebar_display_point,
    --t.login_template                   ,
    (select bt.template_name from wwv_flow_button_templates bt where bt.id = t.default_button_template and bt.security_group_id = t.security_group_id)                  default_button_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_region_template and pt.security_group_id = t.security_group_id)     default_region_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_chart_template and pt.security_group_id = t.security_group_id)      default_chart_rg_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_form_template and pt.security_group_id = t.security_group_id)       default_form_rg_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_reportr_template and pt.security_group_id = t.security_group_id)    default_report_region_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_tabform_template and pt.security_group_id = t.security_group_id)    default_tabular_form_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_wizard_template and pt.security_group_id = t.security_group_id)     default_wizard_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_menur_template and pt.security_group_id = t.security_group_id)      default_breadcrumb_rg_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_listr_template and pt.security_group_id = t.security_group_id)      default_list_region_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_irr_template and pt.security_group_id = t.security_group_id)        default_irr_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_header_template and pt.security_group_id = t.security_group_id)     default_header_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_footer_template and pt.security_group_id = t.security_group_id)     default_footer_template,
    (select rt.row_template_name from wwv_flow_row_templates rt where rt.id = t.default_report_template and rt.security_group_id = t.security_group_id)                 default_report_row_template,
    (select ft.template_Name from wwv_flow_field_templates ft where ft.id = t.default_label_template and ft.security_group_id = t.security_group_id)                    default_item_label_template,
    (select mt.name from wwv_flow_menu_templates mt where mt.id = t.default_menu_template and mt.security_group_id = t.security_group_id)                               default_breadcrumb_template,
    (select ct.name from wwv_flow_cal_templates ct where ct.id = t.default_calendar_template and ct.security_group_id = t.security_group_id)                            default_calendar_template,
    (select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_list_template and lt.security_group_id = t.security_group_id)                 default_list_template,
    (select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_nav_list_template and lt.security_group_id = t.security_group_id)             default_nav_list_template,
    (select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_top_nav_list_template and lt.security_group_id = t.security_group_id)         default_top_nav_list_template,
    (select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_side_nav_list_template and lt.security_group_id = t.security_group_id)        default_side_nav_list_template,
    (select ft.template_name from wwv_flow_field_templates ft where ft.id = t.default_option_label and ft.security_group_id = t.security_group_id)                      default_option_label,
    (select ft.template_name from wwv_flow_field_templates ft where ft.id = t.default_required_label and ft.security_group_id = t.security_group_id)                    default_required_label,
    (select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_nav_bar_list_template and lt.security_group_id = t.security_group_id)         default_nav_bar_list_template,
    --
    (select ft.name from wwv_flow_templates ft where ft.id = t.default_dialog_template and ft.security_group_id = t.security_group_id)                                  default_dialog_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_dialogr_template and pt.security_group_id = t.security_group_id)    default_dialogr_template,
    (select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_dialogbtnr_template and pt.security_group_id = t.security_group_id) default_dialogbtnr_template,
    --
    t.default_page_transition,
    t.default_popup_transition,
    --
    t.last_updated_on                                                                                        last_updated_on,
    t.last_updated_by                                                                                        last_updated_by,
    t.calendar_icon                                                                                          calendar_icon_image_name,
    t.calendar_icon_attr                                                                                     calendar_icon_attributes,
    t.custom_icon_classes,
    t.custom_icon_prefix_class,
    t.custom_library_file_urls,
    t.icon_library,
    t.javascript_file_urls,
    t.css_file_urls,
    t.id                                                                                                     theme_id,
    t.theme_description                                                                                      theme_description,
    t.file_prefix,
    t.files_version,
    --
    t.theme_id
    ||' n='||substr(t.theme_name                                                                                             ,1,30)
    ||' p='||substr((select ft.name from wwv_flow_templates ft where ft.id = t.default_page_template and ft.security_group_id = t.security_group_id)                                 ,1,30)
    ||' e='||substr((select ft.name from wwv_flow_templates ft where ft.id = t.error_template and ft.security_group_id = t.security_group_id),1,30)
    ||' b='||substr((select bt.template_name from wwv_flow_button_templates bt where bt.id = t.default_button_template and bt.security_group_id = t.security_group_id)              ,1,30)
    ||' r='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_region_template and pt.security_group_id = t.security_group_id)  ,1,30)
    ||' c='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_chart_template and pt.security_group_id = t.security_group_id)   ,1,30)
    ||' f='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_form_template and pt.security_group_id = t.security_group_id)    ,1,30)
    ||' r='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_reportr_template and pt.security_group_id = t.security_group_id) ,1,30)
    ||' t='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_tabform_template and pt.security_group_id = t.security_group_id) ,1,30)
    ||' w='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_wizard_template and pt.security_group_id = t.security_group_id)  ,1,30)
    ||' b='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_menur_template and pt.security_group_id = t.security_group_id)   ,1,30)
    ||' l='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_listr_template and pt.security_group_id = t.security_group_id)   ,1,30)
    ||' i='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_irr_template and pt.security_group_id = t.security_group_id)     ,1,30)
    ||' h='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_header_template and pt.security_group_id = t.security_group_id)  ,1,30)
    ||' f='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_footer_template and pt.security_group_id = t.security_group_id)  ,1,30)
    ||' r='||substr((select rt.row_template_name from wwv_flow_row_templates rt where rt.id = t.default_report_template and rt.security_group_id = t.security_group_id)              ,1,30)
    ||' i='||substr((select ft.template_Name from wwv_flow_field_templates ft where ft.id = t.default_label_template and ft.security_group_id = t.security_group_id)                 ,1,30)
    ||' b='||substr((select mt.name from wwv_flow_menu_templates mt where mt.id = t.default_menu_template and mt.security_group_id = t.security_group_id)                            ,1,30)
    ||' c='||substr((select ct.name from wwv_flow_cal_templates ct where ct.id = t.default_calendar_template and ct.security_group_id = t.security_group_id)                         ,1,30)
    ||' l='||substr((select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_list_template and lt.security_group_id = t.security_group_id)              ,1,30)
    ||' l='||substr((select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_nav_list_template and lt.security_group_id = t.security_group_id)          ,1,30)
    ||' l='||substr((select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_top_nav_list_template and lt.security_group_id = t.security_group_id)      ,1,30)
    ||' l='||substr((select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_side_nav_list_template and lt.security_group_id = t.security_group_id)     ,1,30)
    ||' l='||substr((select ft.template_Name from wwv_flow_field_templates ft where ft.id = t.default_option_label and ft.security_group_id = t.security_group_id)                   ,1,30)
    ||' l='||substr((select ft.template_Name from wwv_flow_field_templates ft where ft.id = t.default_required_label and ft.security_group_id = t.security_group_id)                 ,1,30)
    ||' l='||substr((select lt.list_template_name from wwv_flow_list_templates lt where lt.id = t.default_nav_bar_list_template and lt.security_group_id = t.security_group_id)      ,1,30)
    ||' c='||substr(t.calendar_icon,1,30)
    ||' c='||substr(t.calendar_icon_attr,1,30)
    ||' dbr='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_dialogbtnr_template and pt.security_group_id = t.security_group_id)   ,1,30)
    ||' dr='||substr((select pt.page_plug_template_name from wwv_flow_page_plug_templates pt where pt.id = t.default_dialogr_template and pt.security_group_id = t.security_group_id)   ,1,30)
    component_signature
from wwv_flow_themes     t,
     wwv_flow_ui_types   uit,
     wwv_flow_authorized f
where f.workspace_id      = t.security_group_id
  and f.application_id    = t.flow_id
  and t.ui_type_id        = uit.id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_themes',
    p_table_comment     => 'Identifies a named collection of Templates',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                        , 'A work area mapped to one or more database schemas',
      'workspace_display_name'           , 'Display name for the workspace',
      'application_id'                   , 'Application Primary Key, Unique over all workspaces',
      'application_name'                 , 'Identifies the application',
      'THEME_NUMBER'                     , 'Identifies the theme number associated with all templates within the theme',
      'THEME_NAME'                       , 'Identifies the name of the theme',
      'THEME_INTERNAL_NAME'              , 'Identifies the internal name of the theme',
      'NAVIGATION_TYPE'                  , 'Identifies whether theme uses tabs or navigation menu lists',
      'NAVIGATION_TYPE_CODE'             , 'Identifies whether theme uses tabs or navigation menu lists',
      'default_nav_list_position'        , 'Identifies whether theme uses top or side navigation menu lists',
      'NAV_BAR_TYPE'                     , 'Identifies whether theme uses navigation bar or lists',
      'NAV_BAR_TYPE_CODE'                , 'Identifies whether theme uses navigation bar or lists',
      'is_current'                       , 'Identifies whether the theme is the current one for the user interface in the application',
      'DEFAULT_PAGE_TEMPLATE'            , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_DIALOG_TEMPLATE'          , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_BUTTON_TEMPLATE'          , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_REGION_TEMPLATE'          , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_CHART_RG_TEMPLATE'        , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_FORM_RG_TEMPLATE'         , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_REPORT_REGION_TEMPLATE'   , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_TABULAR_FORM_TEMPLATE'    , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_WIZARD_TEMPLATE'          , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_BREADCRUMB_RG_TEMPLATE'   , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_LIST_REGION_TEMPLATE'     , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_IRR_TEMPLATE'             , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_HEADER_TEMPLATE'          , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_FOOTER_TEMPLATE'          , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_REPORT_ROW_TEMPLATE'      , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_ITEM_LABEL_TEMPLATE'      , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_BREADCRUMB_TEMPLATE'      , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_CALENDAR_TEMPLATE'        , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_LIST_TEMPLATE'            , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_NAV_LIST_TEMPLATE'        , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_TOP_NAV_LIST_TEMPLATE'    , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_SIDE_NAV_LIST_TEMPLATE'   , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_OPTION_LABEL'             , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_REQUIRED_LABEL'           , 'Identifies the default template when creating new components of a corresponding type',
      'DEFAULT_DIALOGBTNR_TEMPLATE'      , 'Identifies the default template when creating new button region on a dialog',
      'DEFAULT_DIALOGR_TEMPLATE'         , 'Identifies the default template when creating new component region on a dialog',
      'DEFAULT_NAV_BAR_LIST_TEMPLATE'    , 'Identifies the default template when creating new components of a corresponding type',
      'CALENDAR_ICON_IMAGE_NAME'         , 'Identifies the default template when creating new components of a corresponding type',
      'CALENDAR_ICON_ATTRIBUTES'         , 'Identifies the default template when creating new components of a corresponding type',
      'custom_icon_classes'              , 'List of custom icon CSS classes available with this theme',
      'custom_icon_prefix_class'         , 'CSS class which is automatically added during runtime if a custom icon CSS class is used',
      'custom_library_file_urls'         , 'List of URLs which should be loaded to support the custom icon css classes. ',
      'icon_library'                     , 'Current icon library used with theme',
      'javascript_file_urls'             , 'Allows definition of JavaScript file URLs that will be loaded on every page in the application.',
      'css_file_urls'                    , 'Allows definition of CSS file URLs that will be loaded on every page in the application.',
      'THEME_ID'                         , 'Primary Key of the Theme',
      'theme_description'                , 'Comment field',
      'file_prefix'                      , 'File prefix which is used by the theme to load additional files like CSS, Javascript and images.',
      'files_version'                    , 'Version counter of the attached theme files.',
      'last_updated_by'                  , 'APEX developer who made last update',
      'last_updated_on'                  , 'Date of last update',
      'component_signature'              , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'UI_TYPE_ID'                       , 'Identifies the primary key of the user interface type.',
      'UI_TYPE_NAME'                     , 'Name of the user interface type.',
      'REFERENCE_ID'                     , 'Foreign key of the referenced theme.',
      'IS_LOCKED'                        , 'Is the referenced theme locked to prevent changes',
      'DEFAULT_PAGE_TRANSITION'          , 'Default transition which for pages using the referenced theme.',
      'DEFAULT_POPUP_TRANSITION'         , 'Default transition which for popups using the referenced theme.'
    )
  );
end;
/

/

prompt ...apex_application_theme_styles

create or replace view apex_application_theme_styles
as
select s.id                              as theme_style_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       s.theme_id                        as theme_number,
       s.name,
       s.css_file_urls,
       s.css_classes,
       case s.is_current
         when 'Y' then 'Yes'
         else 'No'
       end as is_current,
       case s.is_public
           when 'Y' then 'Yes'
           else 'No'
       end as is_public,
       case s.is_accessible
           when 'Y' then 'Yes'
           else 'No'
       end as is_accessible,
       s.theme_roller_input_file_urls,
       s.theme_roller_config,
       s.theme_roller_output_file_url,
       s.theme_roller_read_only,
       nvl2(s.reference_id, 'Yes', 'No') as is_subscribed,
       ( select s2.flow_id||'. '||s2.name
           from wwv_flow_theme_styles s2
          where s2.security_group_id in (f.workspace_id, 12)
          and s2.id = s.reference_id )   as subscribed_from,
       s.reference_id                    as subscribed_from_id,
       s.component_comment,
       s.created_by,
       s.created_on,
       s.last_updated_by,
       s.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_theme_styles s
 where s.security_group_id = f.workspace_id
 and s.flow_id = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_theme_styles',
    p_table_comment     => 'The Theme Style identifies the CSS file URLs which should be used for a theme',
    p_column_comments   => wwv_flow_t_varchar2 (
      'theme_style_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'theme_number' , 'Identifies the theme number associated with this theme style',
      'name' , 'Contains the name of the theme style.',
      'css_file_urls' , 'List of URLs which should be loaded for the theme style.',
      'css_classes' , 'Identifies CSS classes to be applied to all pages',
      'is_current' , 'Identifies if this theme style is the current one for the theme.',
      'is_public' , 'Identifies if this theme style can be chosen by end users in the customization dialog.',
      'is_accessible' , 'Identifies if this theme style is suited for accessible user interfaces.',
      'theme_roller_input_file_urls' , 'List of URLs that are loaded as input files for the Thene Roller',
      'theme_roller_config' , 'Theme Roller configuartion key value pairs in JSON format',
      'theme_roller_output_file_url' , 'Theme Roller generated CSS output file',
      'theme_roller_read_only' , 'Identified whether Theme Roller allows saving back / overwriting of theme styles',
      'is_subscribed' , 'Identifies if this theme style is subscribed from another theme style',
      'subscribed_from' , 'Identifies the master component from which this component is subscribed',
      'subscribed_from_id' , 'Id the master component from which this component is subscribed',
      'component_comment' , 'Developer Comment',
      'created_by' , 'APEX developer who created the theme style',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/

prompt ...apex_appl_template_opt_groups

create or replace view apex_appl_template_opt_groups
as
select o.id                              as template_opt_group_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       o.theme_id                        as theme_number,
       o.name                            as name,
       o.display_name                    as display_name,
       o.display_sequence                as display_sequence,
       o.template_types                  as template_types,
       o.help_text                       as help_text,
       o.null_text                       as null_text,
       o.is_advanced                     as is_advanced,
       nvl2(o.reference_id, 'Yes', 'No') as is_subscribed,
       ( select o2.flow_id||'. '||o2.name
           from wwv_flow_template_options o2
          where o2.security_group_id in (f.workspace_id, 12)
          and o2.id = o.reference_id )   as subscribed_from,
       o.reference_id                    as subscribed_from_id,
       o.created_by,
       o.created_on,
       o.last_updated_by,
       o.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_template_opt_groups o
 where o.security_group_id = f.workspace_id
 and o.flow_id = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_template_opt_groups',
    p_table_comment     => 'The Template Option Groups identify groups associated with tempalte options',
    p_column_comments   => wwv_flow_t_varchar2 (
      'template_opt_group_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'theme_number' , 'Identifies the theme number associated with this template option group',
      'name' , 'Identifies the interal name of with this template option group',
      'display_name' , 'Identifies the display name of with this template option group',
      'display_sequence' , 'Identifies the display sequence of with this template option group',
      'template_types' , 'Identifies the template types associated with this template option group',
      'help_text' , 'Identifies the help text associated with this template option group',
      'null_text' , 'Identifies the NULL text associated with this template option group',
      'is_advanced' , 'Identifies whether this is an advanced template option group',
      'is_subscribed' , 'Identifies if this template option groupis subscribed from another template option group',
      'subscribed_from' , 'Identifies the master component from which this component is subscribed',
      'subscribed_from_id' , 'Id the master component from which this component is subscribed',
      'created_by' , 'APEX developer who created the template option group',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/

prompt ...apex_appl_template_options

create or replace view apex_appl_template_options
as
select o.id                              as template_option_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       o.theme_id                        as theme_number,
       o.name                            as name,
       o.display_name                    as display_name,
       o.display_sequence                    as display_sequence,
       (select flt.name
          from wwv_flow_templates flt
         where flt.security_group_id = f.workspace_id
           and flt.id = o.page_template_id
           and flt.flow_id = o.flow_id)     as page_template,
       o.page_template_id               as page_template_id,
       (select pt.page_plug_template_name
          from wwv_flow_page_plug_templates pt
         where pt.security_group_id = f.workspace_id
           and pt.id = o.region_template_id
           and pt.flow_id = o.flow_id)      as region_template,
       o.region_template_id              as region_template_id,
       (select rt.row_template_name
          from wwv_flow_row_templates rt
         where rt.security_group_id = f.workspace_id
           and rt.id = o.report_template_id
           and rt.flow_id = o.flow_id)      as report_template,
       o.report_template_id              as report_template_id,
       (select mt.name
          from wwv_flow_menu_templates mt
         where mt.security_group_id = f.workspace_id
           and mt.id = o.breadcrumb_template_id
           and mt.flow_id = o.flow_id)      as breadcrumb_template,
       o.breadcrumb_template_id          as breadcrumb_template_id,
       (select lt.list_template_name
          from wwv_flow_list_templates lt
         where lt.security_group_id = f.workspace_id
           and lt.id = o.list_template_id
           and lt.flow_id = o.flow_id)      as list_template,
       o.list_template_id                as list_template_id,
       (select ft.template_name
          from wwv_flow_field_templates ft
         where ft.security_group_id = f.workspace_id
           and ft.id = o.field_template_id
           and ft.flow_id = o.flow_id)   as field_template,
       o.field_template_id               as field_template_id,
       (select bt.template_name
          from wwv_flow_button_templates bt
         where bt.security_group_id = f.workspace_id
           and bt.id = o.button_template_id
           and bt.flow_id = o.flow_id)   as button_template,
       o.button_template_id              as button_template_id,
       o.virtual_template_type,
       o.virtual_template_id,
       o.css_classes                     as css_classes,
       o.group_id                        as group_id,
       o.help_text                       as help_text,
       o.is_advanced                     as is_advanced,
       o.template_types                  as template_types,
       nvl2(o.reference_id, 'Yes', 'No') as is_subscribed,
       ( select o2.flow_id||'. '||o2.name
           from wwv_flow_template_options o2
          where o2.security_group_id in (f.workspace_id, 12)
          and o2.id = o.reference_id )   as subscribed_from,
       o.reference_id                    as subscribed_from_id,
       o.created_by,
       o.created_on,
       o.last_updated_by,
       o.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_template_options o
 where o.security_group_id = f.workspace_id
 and o.flow_id = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_template_options',
    p_table_comment     => 'The Template Options identify available CSS classes for pages, regions, report, breadcrumbs, lists, items and buttons',
    p_column_comments   => wwv_flow_t_varchar2 (
      'template_option_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'theme_number' , 'Identifies the theme number associated with this template option',
      'name' , 'Identifies the interal name of with this template option',
      'display_name' , 'Identifies the display name of with this template option',
      'display_sequence' , 'Identifies the display sequece of with this template option',
      'page_template' , 'Identifies the page template associated with this template option',
      'page_template_id' , 'Identifies the page template ID associated with this template option',
      'region_template' , 'Identifies the region template associated with this template option',
      'region_template_id' , 'Identifies the region template ID associated with this template option',
      'report_template' , 'Identifies the report template associated with this template option',
      'report_template_id' , 'Identifies the report template ID associated with this template option',
      'breadcrumb_template' , 'Identifies the breadcrumb template associated with this template option',
      'breadcrumb_template_id' , 'Identifies the breadcrumb template ID associated with this template option',
      'list_template' , 'Identifies the list template associated with this template option',
      'list_template_id' , 'Identifies the list template ID associated with this template option',
      'field_template' , 'Identifies the field template associated with this template option',
      'field_template_id' , 'Identifies the field template ID associated with this template option',
      'button_template' , 'Identifies the button template associated with this template option',
      'button_template_id' , 'Identifies the button template ID associated with this template option',
      'virtual_template_type' , 'Identifies the template type associated with this template option',
      'virtual_template_id' , 'Identifies the template ID associated with this template option',
      'css_classes' , 'Identifies the CSS classes associated with this template option',
      'group_id' , 'Identifies the template option group associated with this template option',
      'template_types' , 'Identifies the template types associated with this template option',
      'help_text' , 'Identifies the help text associated with this template option',
      'is_advanced' , 'Identifies whether this is an advanced template option',
      'is_subscribed' , 'Identifies if this template option is subscribed from another template option',
      'subscribed_from' , 'Identifies the master component from which this component is subscribed',
      'subscribed_from_id' , 'Id the master component from which this component is subscribed',
      'created_by' , 'APEX developer who created the template option',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/
create or replace view apex_application_theme_files
as
select a.id                              as theme_file_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       a.theme_id                        as theme_number,
       a.file_name,
       a.mime_type,
       a.file_charset,
       a.file_content,
       nvl2(a.reference_id, 'Yes', 'No') as is_subscribed,
       ( select a2.flow_id||'. '||a2.file_name
           from wwv_flow_theme_files a2
          where a2.security_group_id in (f.workspace_id, 12)
          and a2.id = a.reference_id ) as subscribed_from,
       a.reference_id                    as subscribed_from_id,
       a.created_by,
       a.created_on,
       a.last_updated_by,
       a.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_theme_files a
 where a.security_group_id = f.workspace_id
 and a.flow_id = f.application_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_theme_files',
    p_table_comment     => 'Stores the files like CSS, images, javascript files, ... of a theme.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'theme_file_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'theme_number' , 'Identifies the theme number associated with this theme style',
      'file_name' , 'Name of the file.',
      'mime_type' , 'Mime type of the file.',
      'file_charset' , 'IANA charset used for text files.',
      'file_content' , 'Blob content of the file.',
      'is_subscribed' , 'Identifies if this theme style is subscribed from another theme style',
      'subscribed_from' , 'Identifies the master component from which this component is subscribed',
      'subscribed_from_id' , 'Id the master component from which this component is subscribed',
      'created_by' , 'APEX developer who created the file',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/

---------------------------------------------------------
-- A P P L I C A T I O N   T E M P L A T E S
--
prompt ...apex_application_templates

create or replace view apex_application_temp_page
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.NAME                           template_name,
    t.INTERNAL_NAME                  internal_name,
    --t.LOOK
    --
    t.is_popup,
    t.javascript_file_urls,
    t.javascript_code,
    t.javascript_code_onload,
    t.dialog_js_init_code,
    t.dialog_js_close_code,
    t.dialog_js_cancel_code,
    t.css_file_urls,
    t.inline_css,
    t.HEADER_TEMPLATE                ,
    t.BOX                            page_body,
    t.FOOTER_TEMPLATE                ,
    --
    t.SUCCESS_MESSAGE                ,
    t.CURRENT_TAB                    ,
    t.CURRENT_TAB_FONT_ATTR          ,
    t.NON_CURRENT_TAB                ,
    t.NON_CURRENT_TAB_FONT_ATTR      ,
    t.CURRENT_IMAGE_TAB              ,
    t.NON_CURRENT_IMAGE_TAB          ,
    t.TOP_CURRENT_TAB                Current_Parent_Tab,
    t.TOP_CURRENT_TAB_FONT_ATTR      Current_Parent_Tab_attr,
    t.TOP_NON_CURRENT_TAB            NonCurrent_Parent_Tab,
    t.TOP_NON_CURRENT_TAB_FONT_ATTR  NonCurrent_Parent_Tab_attr,
    --
    t.NAVIGATION_BAR                 ,
    t.NAVBAR_ENTRY                   ,
    --t.BODY_TITLE                     ,
    t.MESSAGE                        ,
    --t.ATTRIBUTE1                     ,
    --t.ATTRIBUTE2                     ,
    --t.ATTRIBUTE3                     ,
    --t.ATTRIBUTE4                     ,
    --t.ATTRIBUTE5                     ,
    --t.ATTRIBUTE6                     ,
    --t.DEFAULT_BUTTON_POSITION        ,
    --t.TABLE_BGCOLOR                  ,
    --t.HEADING_BGCOLOR                ,
    --t.TABLE_CATTRIBUTES              ,
    --t.FONT_SIZE                      ,
    --t.FONT_FACE                      ,
    case t.grid_type
      when 'TABLE'    then 'Table'
      when 'FIXED'    then 'Fixed'
      when 'VARIABLE' then 'Variable'
    end                               as grid_type,
    t.grid_type                       as grid_type_code,
    t.grid_max_columns,
    case t.grid_always_use_max_columns
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                               as grid_always_use_max_columns,
    case t.grid_has_column_span
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                               as grid_has_column_span,
    case t.grid_always_emit
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                               as grid_always_emit,
    case t.grid_emit_empty_leading_cols
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                               as grid_emit_empty_leading_cols,
    case t.grid_emit_empty_trailing_cols
      when 'Y' then 'Yes'
      when 'N' then 'No'
    end                               as grid_emit_empty_trailing_cols,
    t.grid_default_label_col_span,
    t.grid_template,
    t.grid_row_template,
    t.grid_column_template,
    t.grid_first_column_attributes,
    t.grid_last_column_attributes,
    t.grid_javascript_debug_code,
    t.dialog_height,
    t.dialog_width,
    t.dialog_max_width,
    t.dialog_css_classes,
    t.dialog_browser_frame,
    t.REGION_TABLE_CATTRIBUTES        as grid_table_attributes,
    t.REGION_TABLE_CATTRIBUTES        as MultiColumn_Region_Table_Attr, /* for backward compatibility */
    --
    --t.APP_TAB_BEFORE_TABS            ,
    --t.APP_TAB_CURRENT_TAB            ,
    --t.APP_TAB_NON_CURRENT_TAB        ,
    --t.APP_TAB_AFTER_TABS             ,
    --
    'Y'                              as has_edit_links,
    --
    t.ERROR_PAGE_TEMPLATE            ,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select ft.flow_id||'. '||ft.name
      from wwv_flow_templates ft
     where ft.id                = t.REFERENCE_ID
       and ft.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    --t.BREADCRUMB_DEF_REG_POS         ,
    --t.SIDEBAR_DEF_REG_POS            ,
    --t.REQUIRED_PATCH                 ,
    t.THEME_ID                       theme_number,
    decode(t.THEME_CLASS_ID,
      '1', 'One Level Tabs',
      '2', 'Two Level Tabs',
      '3', 'No Tabs',
      '4', 'Popup',
      '5', 'Printer Friendly',
      '6', 'Login',
      '7', 'Unknown',
      '8',  'Custom 1',
      '9',  'Custom 2',
      '10', 'Custom 3',
      '11', 'Custom 4',
      '12', 'Custom 5',
      '13', 'Custom 6',
      '14', 'Custom 7',
      '15', 'Custom 8',
      '16', 'One Level Tabs with Sidebar',
      '17', 'No Tabs with Sidebar',
      '18', 'Two Level Tabs with Sidebar',
      t.THEME_CLASS_ID)              theme_class,
    t.default_template_options       default_template_options,
    t.preset_template_options        preset_template_options,
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    decode(
       t.TRANSLATE_THIS_TEMPLATE,
       'N','No','Y','Yes','Yes')     translatable,
    t.TEMPLATE_COMMENT               component_comment,
    t.id                             template_id,
    t.id                             page_template_id,
    --
    substr(t.NAME,1,40)||'.'||length(t.NAME)
    ||' h='||sys.dbms_lob.substr(t.HEADER_TEMPLATE                  ,40,1)||'.'||sys.dbms_lob.getlength(t.HEADER_TEMPLATE                )
    ||' b='||sys.dbms_lob.substr(t.BOX                              ,40,1)||'.'||sys.dbms_lob.getlength(t.BOX                            )
    ||' f='||sys.dbms_lob.substr(t.FOOTER_TEMPLATE                  ,40,1)||'.'||sys.dbms_lob.getlength(t.FOOTER_TEMPLATE                )
    ||' s='||substr(t.SUCCESS_MESSAGE                  ,1,40)||'.'||length(t.SUCCESS_MESSAGE                )
    ||' t='||substr(t.CURRENT_TAB                      ,1,40)||'.'||length(t.CURRENT_TAB                    )
    ||' t='||substr(t.CURRENT_TAB_FONT_ATTR            ,1,40)||'.'||length(t.CURRENT_TAB_FONT_ATTR          )
    ||' n='||substr(t.NON_CURRENT_TAB                  ,1,40)||'.'||length(t.NON_CURRENT_TAB                )
    ||' n='||substr(t.NON_CURRENT_TAB_FONT_ATTR        ,1,40)||'.'||length(t.NON_CURRENT_TAB_FONT_ATTR      )
    ||' i='||substr(t.CURRENT_IMAGE_TAB                ,1,40)||'.'||length(t.CURRENT_IMAGE_TAB              )
    ||' i='||substr(t.NON_CURRENT_IMAGE_TAB            ,1,40)||'.'||length(t.NON_CURRENT_IMAGE_TAB          )
    ||' t='||substr(t.TOP_CURRENT_TAB                  ,1,40)||'.'||length(t.TOP_CURRENT_TAB                )
    ||' t='||substr(t.TOP_CURRENT_TAB_FONT_ATTR        ,1,40)||'.'||length(t.TOP_CURRENT_TAB_FONT_ATTR      )
    ||' t='||substr(t.TOP_NON_CURRENT_TAB              ,1,40)||'.'||length(t.TOP_NON_CURRENT_TAB            )
    ||' t='||substr(t.TOP_NON_CURRENT_TAB_FONT_ATTR    ,1,40)||'.'||length(t.TOP_NON_CURRENT_TAB_FONT_ATTR  )
    ||' n='||substr(t.NAVIGATION_BAR                   ,1,40)||'.'||length(t.NAVIGATION_BAR                 )
    ||' n='||substr(t.NAVBAR_ENTRY                     ,1,40)||'.'||length(t.NAVBAR_ENTRY                   )
    ||' m='||substr(t.MESSAGE                          ,1,40)||'.'||length(t.MESSAGE                        )
    ||' e='||substr(t.ERROR_PAGE_TEMPLATE              ,1,40)||'.'||length(t.ERROR_PAGE_TEMPLATE            )
    --
    ||' s='||decode(t.REFERENCE_ID,null,'No','Yes')
    ||' t='||t.THEME_ID
    ||' c='||decode(t.THEME_CLASS_ID,
      '1', 'One Level Tabs',
      '2', 'Two Level Tabs',
      '3', 'No Tabs',
      '4', 'Popup',
      '5', 'Printer Friendly',
      '6', 'Login',
      '7', 'Unknown',
      '8',  'Custom 1',
      '9',  'Custom 2',
      '10', 'Custom 3',
      '11', 'Custom 4',
      '12', 'Custom 5',
      '13', 'Custom 6',
      '14', 'Custom 7',
      '15', 'Custom 8',
      '16', 'One Level Tabs with Sidebar',
      '17', 'No Tabs with Sidebar',
      '18', 'Two Level Tabs with Sidebar',
      t.THEME_CLASS_ID)
    ||' t='||t.TRANSLATE_THIS_TEMPLATE
    component_signature
from wwv_flow_templates  t,
     wwv_flow_authorized f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_page',
    p_table_comment     => 'The Page Template which identifies the HTML used to organized and render a page content',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                      , 'A work area mapped to one or more database schemas',
      'workspace_display_name'         , 'Display name for the workspace',
      'application_id'                 , 'Application Primary Key, Unique over all workspaces',
      'application_name'               , 'Identifies the application',
      'TEMPLATE_NAME'                  , 'Name of the Page Template',
      'INTERNAL_NAME'                  , 'Internal name of the Page Template',
      'javascript_code'                , 'Identifies JavaScript code that replaces the #TEMPLATE_JAVASCRIPT# substitution string in the Page Template',
      'javascript_code_onload'         , 'Identifies JavaScript code which is executed after page load',
      'HEADER_TEMPLATE'                , 'In the Header is the first of 3 parts of the page template.  Enter the HTML that makes up the HEAD section of the HTML document.  That is, all the required HTML tags before the BODY of the HTML document.',
      'page_body'                      , 'The Body is the second of 3 parts of the page template.  Enter HTML that makes up the BODY of the HTML document.',
      'FOOTER_TEMPLATE'                , 'The Footer is the third template and displays after the body.',
      'SUCCESS_MESSAGE'                , 'Enter HTML that will substitute the string #SUCCESS_MESSAGE# in the template body',
      'CURRENT_TAB'                    , 'Enter HTML or text that will be substituted for the currently selected standard tab, use #TAB_LINK#" and #TAB_LABEL# substitutions.',
      'CURRENT_TAB_FONT_ATTR'          , 'This attribute is part of the Standard Tab subtemplate.  This value replaces the #TAB_FONT_ATTRIBUTES# substitution string.',
      'NON_CURRENT_TAB'                , 'HTML or text that will be substituted for the unselected standard tabs, use #TAB_LINK#" and #TAB_LABEL# substitutions',
      'NON_CURRENT_TAB_FONT_ATTR'      , 'This attribute is part of the Parent Tab subtemplate and expands the #PARENT_TAB_CELLS# substitution string.',
      'CURRENT_IMAGE_TAB'              , 'HTML to be used to indicate that an image-based tab is currently selected',
      'NON_CURRENT_IMAGE_TAB'          , 'HTML to be used to indicate that an image-based tab is not currently selected',
      'Current_Parent_Tab'             , 'HTML or text that will be substituted for the selected parent tabs',
      'Current_Parent_Tab_ATTR'        , 'This value replaces the #TAB_FONT_ATTRIBUTES# substitution string',
      'NonCurrent_Parent_Tab'          , 'HTML or text that will be substituted for the unselected standard tabs, use #TAB_LINK# and #TAB_LABEL# substitutions',
      'NonCurrent_Parent_Tab_ATTR'     , 'This value replaces the #TAB_FONT_ATTRIBUTES# substitution string',
      'NAVIGATION_BAR'                 , 'HTML or text that will be substituted when the string #NAVIGATION_BAR# is referenced in the template header, body or footer',
      'NAVBAR_ENTRY'                   , 'HTML or text that will be substituted into the navigation bar #BAR_BODY# for each navigation bar entry',
      'MESSAGE'                        , 'HTML or text that will be substituted when the string #NOTIFICATION_MESSAGE# is referenced in the template header, body or footer',
      'ERROR_PAGE_TEMPLATE'            , 'Used only when this page template will be designated as an error template.  Use #MESSAGE# to place the error message and #BACK_LINK# to display a link back to the previous page.',
      'grid_type'                      , 'Identifies the type of grid template',
      'grid_type_code'                 , 'Internal code of TYPE.',
      'grid_max_columns'               , 'Identifies the maximum number of columns the grid can have.',
      'grid_always_use_max_columns'    , 'Identifies if the maximum number of columns should always be used from the template definition.',
      'grid_has_column_span'           , 'Identifies if the grid supports specifying a column span.',
      'grid_always_emit'               , 'Identifies if the grid should always be rendered, even if only the first column of the grid is used.',
      'grid_emit_empty_leading_cols'   , 'Identifies if the grid should emit empty leading grid column before a fixed positioned grid column.',
      'grid_emit_empty_trailing_cols'  , 'Identifies if the grid should emit empty trailing grid column to fill up the maximum available grid columns.',
      'grid_default_label_col_span'    , 'Identifies over how many grid columns the label spans if not specified on item level.',
      'grid_template'                  , 'HTML that will be used for the grid.',
      'grid_row_template'              , 'HTML that will be used for a grid row.',
      'grid_column_template'           , 'HTML that will be used for a grid column.',
      'grid_first_column_attributes'   , 'Identifies the attributes which should be substituted for #FIRST_LAST_COLUMN_ATTRIBUTES# if it''s the first grid column.',
      'grid_last_column_attributes'    , 'Identifies the attributes which should be substituted for #FIRST_LAST_COLUMN_ATTRIBUTES# if it''s the last grid column.',
      'grid_javascript_debug_code'     , 'JavaScript code that can be used to toggle on/off a debug mode for grid layouts.',
      'grid_table_attributes'          , 'This attribute controls the attributes of the HTML table tag used to display regions in multiple columns.',
      'dialog_js_init_code'            , 'Identifies JavaScript code which is executed to initialize a dialog page',
      'dialog_js_close_code'           , 'Identifies JavaScript code which is executed to close a dialog page',
      'dialog_js_cancel_code'          , 'Identifies JavaScript code which is executed to cancel a dialog page',
      'dialog_height'                  , 'The template level default height setting for a dialog',
      'dialog_width'                   , 'The template level default width setting for a dialog',
      'dialog_max_width'               , 'The template level default maximum width setting for a dialog',
      'dialog_css_classes'             , 'The template level default CSS classes setting for a dialog',
      'dialog_browser_frame'           , 'The template level setting to control if the browser is allowed to display the application Modal and Non-Modal dialog with a frame',
      'has_edit_links'                 , 'Deprecated',
      'THEME_NUMBER'                   , 'Identifies the theme number associated with all templates within the theme',
      'THEME_CLASS'                    , 'Identifies a specific usage for this template',
      'default_template_options'       , 'Identifies the default template options selected for this template',
      'preset_template_options'        , 'Identifies the preset template options selected for this template',
      'TRANSLATABLE'                   , 'Identifies if this component is to be identified as translatable (yes or no)',
      'COMPONENT_COMMENT'              , 'Developer comment',
      'IS_SUBSCRIBED'                  , 'Identifies if this template is subscribed from another template',
      'SUBSCRIBED_FROM'                , 'Identifies the Application ID and Template Name this template is subscribed from',
      'TEMPLATE_ID'                    , 'Primary Key of this Page Template (deprecated column)',
      'PAGE_TEMPLATE_ID'               , 'Primary Key of this Page Template',
      'last_updated_by'                , 'APEX developer who made last update',
      'last_updated_on'                , 'Date of last update',
      'component_signature'            , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'MultiColumn_Region_Table_Attr'  , 'This attribute controls the attributes of the HTML table tag used to display regions in multiple columns.',
      'IS_POPUP'                       , 'This template includes HTML for popup pages.',
      'JAVASCRIPT_FILE_URLS'           , 'URLs of JavaScript files to be included in pages utilising this template.',
      'CSS_FILE_URLS'                  , 'URLs of CSS files to be included in pages utilising this template.',
      'INLINE_CSS'                     , 'CSS styles to be directly included in pages based on this template.'
    )
  );
end;
/

prompt ...apex_appl_temp_page_dp

create or replace view apex_appl_temp_page_dp
as
select d.id                      as page_tmpl_display_point_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       d.page_template_id        as page_template_id,
       d.name,
       d.placeholder,
       case d.has_grid_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_grid_support,
       case d.has_region_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_region_support,
       case d.has_item_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_item_support,
       case d.has_button_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_button_support,
       case d.glv_new_row
         when 'Y' then 'Yes'
         else 'No'
       end as glv_new_row,
       d.max_fixed_grid_columns,
       d.component_comment,
       d.created_by,
       d.created_on,
       d.last_updated_by,
       d.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_page_tmpl_disp_points d
 where d.security_group_id = f.workspace_id
 and d.flow_id = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_temp_page_dp',
    p_table_comment     => 'The Page Template Display Points identify the available display points in a page template',
    p_column_comments   => wwv_flow_t_varchar2 (
      'page_tmpl_display_point_id' , 'Identifies the primary key of this component',
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'name'                       , 'Contains the name of the display point.',
      'placeholder'                , 'Contains the placeholder of the display point in the templates.',
      'has_grid_support'           , 'Identifies if grids are supported for this display point.',
      'has_region_support'         , 'Identifies if regions are supported for this display point.',
      'has_item_support'           , 'Identifies if items are supported for this display point.',
      'has_button_support'         , 'Identifies if buttons are supported for this display point.',
      'glv_new_row'                , 'Identifies the position for this display point.',
      'max_fixed_grid_columns'     , 'For a fixed grid it identifies the maximum number of columns which are supported for this display point.',
      'component_comment'          , 'Developer Comment',
      'created_by'                 , 'APEX developer who created the display point',
      'created_on'                 , 'Date of creation',
      'last_updated_by'            , 'APEX developer who made last update',
      'last_updated_on'            , 'Date of last update',
      'page_template_id'           , 'Foreign key of the page template'
     )
  );
end;
/

create or replace view apex_application_temp_region
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.PAGE_PLUG_TEMPLATE_NAME        template_name,
    t.INTERNAL_NAME                  internal_name,
    --
    t.JAVASCRIPT_FILE_URLS           javascript_file_urls,
    t.JAVASCRIPT_CODE_ONLOAD         javascript_code_onload,
    t.CSS_FILE_URLS                  css_file_urls,
    --
    t.THEME_ID                       theme_number,
    --
    decode(t.THEME_CLASS_ID,
        'RETURN_VALUE','DISPLAY_VALUE',
        '1','Hide and Show Region',
        '2','Sidebar Region',
        '3','Sidebar Region, Alternative 1',
        '4','Button Region with Title',
        '5','Navigation Region',
        '6','Breadcrumb Region',
        '7','Borderless Region',
        '8','Form Region',
        '9','Reports Region',
        '10','Reports Region, Alternative 1',
        '11','Region without Title',
        '12','Wizard Region',
        '13','Reports Region 100% Width',
        '16','Navigation Region, Alternative 1',
        '17','Button Region without Title',
        '18','Bracketed Region',
        '19','Region without Buttons and Title',
        '20','Wizard Region with Icon',
        '21','Custom 1',
        '22','Custom 2',
        '23','Custom 3',
        '24','Custom 4',
        '25','Custom 5',
        '26','Custom 6',
        '27','Custom 7',
        '28','Custom 8',
        '29','List Region with Icon',
        '30','Chart Region',
        t.theme_class_id)            theme_class,
    t.default_template_options       default_template_options,
    t.preset_template_options        preset_template_options,
    --
    t.layout,
    t.template,
    t.template2,
    t.template3,
    t.sub_plug_header_template       as sub_region_header_template,
    t.sub_plug_header_entry_template as sub_region_header_entry_templ,
    t.sub_plug_template              as sub_region_template,
    t.plug_table_bgcolor,
    t.plug_heading_bgcolor,
    t.plug_font_size,
    t.image_template,
    t.default_field_template_id,
    t.default_req_field_template_id,
    t.default_label_alignment,
    t.default_field_alignment,
    t.default_button_template_id,
    t.default_button_position,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select pt.flow_id||'. '||pt.internal_name
      from WWV_FLOW_PAGE_PLUG_TEMPLATES pt
     where pt.id                = t.REFERENCE_ID
       and pt.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    decode(t.TRANSLATE_THIS_TEMPLATE,
      'N','No','Y','Yes','Yes')      translatable,
    t.TEMPLATE_COMMENT               component_comment,
    t.id                             region_template_id,
    --
    t.PAGE_PLUG_TEMPLATE_NAME
    ||' t='||t.THEME_ID
    ||' c='||THEME_CLASS_ID
    ||' 1='||sys.dbms_lob.substr(t.TEMPLATE,40,1)||'.'||sys.dbms_lob.getlength(t.TEMPLATE)
    ||' 2='||sys.dbms_lob.substr(t.TEMPLATE2,40,1)||'.'||sys.dbms_lob.getlength(t.TEMPLATE2)
    ||' 3='||sys.dbms_lob.substr(t.TEMPLATE3,40,1)||'.'||sys.dbms_lob.getlength(t.TEMPLATE3)
    ||' b='||substr(t.PLUG_TABLE_BGCOLOR,1,20)
    ||' b='||substr(t.PLUG_HEADING_BGCOLOR,1,20)
    ||' f='||substr(t.PLUG_FONT_SIZE,1,20)
    ||' t='||TRANSLATE_THIS_TEMPLATE
    ||' r='||decode(t.REFERENCE_ID,null,'N','Y')
    component_signature
from WWV_FLOW_PAGE_PLUG_TEMPLATES t,
     wwv_flow_authorized          f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_region',
    p_table_comment     => 'Identifies a regions HTML template display attributes',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'TEMPLATE_NAME'                 , 'Identifies the region template',
      'INTERNAL_NAME'                 , 'Internal name of the region template',
      'javascript_file_urls'          , 'Allows definition of JavaScript file URLs that will be loaded on every page in the application.',
      'javascript_code_onload'        , 'Identifies JavaScript code which is executed after page load',
      'css_file_urls'                 , 'List of URLs which should be loaded for the list style.',
      'THEME_NUMBER'                  , 'Identifies the theme number associated with all templates within the theme',
      'THEME_CLASS'                   , 'Identifies a specific usage for this template',
      'default_template_options'      , 'Identifies the default template option selected for this template',
      'preset_template_options'       , 'Identifies the preset template option selected for this template',
      'TEMPLATE'                      , 'HTML that defines the appearance for a page region',
      'sub_region_header_template'    , 'HTML that defines the appearance of a sub region header',
      'sub_region_header_entry_templ' , 'HTML that defines the appearance of a single sub region header entry',
      'sub_region_template'           , 'HTML that defines the appearance of a sub region',
      'TEMPLATE2'                     , 'HTML that defines the appearance for a page region',
      'TEMPLATE3'                     , 'HTML that defines the appearance for a page region',
      'PLUG_TABLE_BGCOLOR'            , 'Background color for plugin table regions',
      'PLUG_HEADING_BGCOLOR'          , 'Background color for plugin heading regions',
      'PLUG_FONT_SIZE'                , 'Font size for plugin regions',
      'image_template'                , 'HTML that defines the appearance of a region image',
      'is_subscribed'                 , 'Identifies if this Region Template is subscribed from another Region Template',
      'subscribed_from'               , 'Identifies the master component from which this component is subscribed',
      'LAST_UPDATED_BY'               , 'APEX Developer User Name who last updated this component',
      'LAST_UPDATED_ON'               , 'Date this component was last updated',
      'TRANSLATABLE'                  , 'Identifies if this component is to be included in the list of translatable components',
      'COMPONENT_COMMENT'             , 'Developer comment',
      'last_updated_by'               , 'APEX developer who made last update',
      'last_updated_on'               , 'Date of last update',
      'region_template_id'            , 'Primary Key of this Region Template',
      'component_signature'           , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'LAYOUT'                        , 'Type of layout which the template uses',
      'DEFAULT_FIELD_TEMPLATE_ID'     , 'Foreign key of the template to use for regular fields',
      'DEFAULT_REQ_FIELD_TEMPLATE_ID' , 'Foreign key of the template to use for required fields',
      'DEFAULT_LABEL_ALIGNMENT'       , 'Default alignment of field labels',
      'DEFAULT_FIELD_ALIGNMENT'       , 'Default alignment of fields',
      'DEFAULT_BUTTON_TEMPLATE_ID'    , 'Foreign key of the template to use, by default, for buttons',
      'DEFAULT_BUTTON_POSITION'       , 'Default position of buttons'
    )
  );
end;
/

prompt ...apex_appl_temp_region_dp

create or replace view apex_appl_temp_region_dp
as
select d.id                      as region_tmpl_display_point_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       d.plug_template_id        as region_template_id,
       d.name,
       d.placeholder,
       case d.has_grid_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_grid_support,
       case d.has_region_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_region_support,
       case d.has_item_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_item_support,
       case d.has_button_support
         when 'Y' then 'Yes'
         else 'No'
       end as has_button_support,
       case d.glv_new_row
         when 'Y' then 'Yes'
         else 'No'
       end as glv_new_row,
       d.max_fixed_grid_columns,
       d.component_comment,
       d.created_by,
       d.created_on,
       d.last_updated_by,
       d.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_plug_tmpl_disp_points d
 where d.security_group_id = f.workspace_id
 and d.flow_id = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_temp_region_dp',
    p_table_comment     => 'The Region Template Display Points identify the available display points in a region template',
    p_column_comments   => wwv_flow_t_varchar2 (
      'region_tmpl_display_point_id' , 'Identifies the primary key of this component',
      'workspace'                    , 'A work area mapped to one or more database schemas',
      'workspace_display_name'       , 'Display name for the workspace',
      'application_id'               , 'Application Primary Key, Unique over all workspaces',
      'application_name'             , 'Identifies the application',
      'name'                         , 'Contains the name of the display point.',
      'placeholder'                  , 'Contains the placeholder of the display point in the templates.',
      'has_grid_support'             , 'Identifies if grids are supported for this display point.',
      'has_region_support'           , 'Identifies if regions are supported for this display point.',
      'has_item_support'             , 'Identifies if items are supported for this display point.',
      'has_button_support'           , 'Identifies if buttons are supported for this display point.',
      'glv_new_row'                  , 'Identifies the position for this display point.',
      'max_fixed_grid_columns'       , 'For a fixed grid it identifies the maximum number of columns which are supported for this display point.',
      'component_comment'            , 'Developer Comment',
      'created_by'                   , 'APEX developer who created the display point',
      'created_on'                   , 'Date of creation',
      'last_updated_by'              , 'APEX developer who made last update',
      'last_updated_on'              , 'Date of last update',
      'REGION_TEMPLATE_ID'           , 'Foreign key of the region template to which the display point relates'
    )
  );
end;
/

create or replace view apex_application_temp_label
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.THEME_ID                       theme_number,
    decode(t.THEME_CLASS_ID,
      '1','Optional Label with Help',
      '2','Required Label with Help',
      '3','Optional Label',
      '4','Required Label',
      '5','Custom 1',
      '6','Custom 2',
      '7','Custom 3',
      '8','Custom 4',
      '9','Custom 5',
      '10','Custom 6',
      '11','Custom 7',
      '12','Custom 8',
      '13','No Label',
      t.THEME_CLASS_ID)              theme_class,
    t.default_template_options       default_template_options,
    t.preset_template_options        preset_template_options,
    --
    t.TEMPLATE_NAME                  template_name,
    t.INTERNAL_NAME                  internal_name,
    t.TEMPLATE_BODY1                 before_label,
    t.TEMPLATE_BODY2                 after_label,
    --
    t.BEFORE_ITEM                    before_item,
    t.AFTER_ITEM                     after_item,
    --
    t.item_pre_text                  item_pre_text,
    t.item_post_text                 item_post_text,
    --
    t.BEFORE_ELEMENT                 before_element,
    t.AFTER_ELEMENT                  after_element,
    t.HELP_LINK                      help_link,
    t.INLINE_HELP_TEXT               inline_help_text,
    t.ERROR_TEMPLATE                 error_template,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select ft.flow_id||'. '||ft.internal_name
      from WWV_FLOW_FIELD_TEMPLATES ft
     where ft.id                = t.REFERENCE_ID
       and ft.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    t.ON_ERROR_BEFORE_LABEL,
    t.ON_ERROR_AFTER_LABEL,
    t.LAST_UPDATED_ON                last_updated_on,
    t.LAST_UPDATED_BY                last_updated_by,
    t.TRANSLATE_THIS_TEMPLATE        ,
    t.TEMPLATE_COMMENT               component_comment,
    t.id                             label_template_id,
    --
    t.TEMPLATE_NAME
    ||' t='||t.THEME_ID
    ||' c='||t.THEME_CLASS_ID
    ||' 1='||substr(t.TEMPLATE_BODY1,1,40)||length(t.TEMPLATE_BODY1)
    ||' 2='||substr(t.TEMPLATE_BODY2,1,40)||length(t.TEMPLATE_BODY2)
    ||' r='||decode(t.REFERENCE_ID,null,'N','Y')
    ||' e='||substr(t.ON_ERROR_BEFORE_LABEL,1,40)||length(t.ON_ERROR_BEFORE_LABEL)
    ||' e='||substr(t.ON_ERROR_AFTER_LABEL,1,40)||length(t.ON_ERROR_AFTER_LABEL)
    ||' t='||t.TRANSLATE_THIS_TEMPLATE
    component_signature
from WWV_FLOW_FIELD_TEMPLATES t,
     wwv_flow_authorized      f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_label',
    p_table_comment     => 'Identifies a Page Item Label HTML template display attributes',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'               , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'          , 'Application Primary Key, Unique over all workspaces',
      'application_name'        , 'Identifies the application',
      'THEME_NUMBER'            , 'Identifies the theme number associated with all templates within the theme',
      'THEME_CLASS'             , 'Identifies a specific usage for this template',
      'default_template_options' , 'Identifies the default template option selected for this template',
      'preset_template_options'  , 'Identifies the preset template option selected for this template',
      'TEMPLATE_NAME'           , 'Identifies the name of this item label template',
      'INTERNAL_NAME'           , 'Internal name of the item label template',
      'before_label'            , 'HTML to be displayed before an item label',
      'after_label'             , 'HTML to be displayed after an item label',
      'before_item'             , 'HTML field container to be displayed before label and item ',
      'after_item'              , 'HTML field container to be displayed after label and item ',
      'item_pre_text'           , 'HTML subtemplate to format "Item Pre Text" display ',
      'item_post_text'          , 'HTML subtemplate to format "Item Post Text" display ',
      'before_element'          , 'HTML to be displayed before item ',
      'after_element'           , 'HTML to be displayed after item ',
      'help_link'               , 'HTML to be displayed where #HELP_LINK# substitution string is defined ',
      'inline_help_text'        , 'HTML to be displayed where #INLINE_HELP_TEXT# substitution string is defined ',
      'error_template'          , 'HTML to be displayed where #ERROR_MESSAGE# substitution string is defined and error has occured',
      'is_subscribed'           , 'Identifies if this Item Label Template is subscribed from another Item Label Template',
      'subscribed_from'         , 'Identifies the master component from which this component is subscribed',
      'ON_ERROR_BEFORE_LABEL'   , 'HTML to precede the item label when a application displays an inline validation error message for the item',
      'ON_ERROR_AFTER_LABEL'    , 'HTML to be appended to the item label when a application  displays an inline validation error message for the item',
      'TRANSLATE_THIS_TEMPLATE' , 'Identifies if this template should be translated',
      'COMPONENT_COMMENT'       , 'Developer comment',
      'label_template_id'       , 'Primary Key of this Item Label Template',
      'last_updated_by'         , 'APEX developer who made last update',
      'last_updated_on'         , 'Date of last update',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
create or replace view apex_application_temp_list
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.LIST_TEMPLATE_NAME             template_name,
    t.INTERNAL_NAME                  internal_name,
    --
    t.JAVASCRIPT_FILE_URLS           javascript_file_urls,
    t.JAVASCRIPT_CODE_ONLOAD         javascript_code_onload,
    t.CSS_FILE_URLS                  css_file_urls,
    t.INLINE_CSS                     inline_css,
    --
    t.LIST_TEMPLATE_CURRENT,
    t.LIST_TEMPLATE_NONCURRENT,
    t.LIST_TEMPLATE_BEFORE_ROWS,
    t.LIST_TEMPLATE_AFTER_ROWS,
    t.BETWEEN_ITEMS,
    t.BEFORE_SUB_LIST,
    t.AFTER_SUB_LIST,
    t.BETWEEN_SUB_LIST_ITEMS,
    t.SUB_LIST_ITEM_CURRENT,
    t.SUB_LIST_ITEM_NONCURRENT,
    t.ITEM_TEMPLATE_CURR_W_CHILD,
    t.ITEM_TEMPLATE_NONCURR_W_CHILD,
    t.SUB_TEMPLATE_CURR_W_CHILD,
    t.SUB_TEMPLATE_NONCURR_W_CHILD,
    --
    t.FIRST_LIST_TEMPLATE_NONCURRENT,
    t.FIRST_LIST_TEMPLATE_CURRENT,
    t.F_ITEM_TEMPLATE_CURR_W_CHILD,
    t.FITEM_TEMPLATE_NONCURR_W_CHILD,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select lt.flow_id||'. '||lt.internal_name
     from WWV_FLOW_LIST_TEMPLATES lt
     where lt.id                = t.REFERENCE_ID
       and lt.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    t.LAST_UPDATED_BY                ,
    t.LAST_UPDATED_ON                ,
    t.THEME_ID                       theme_number,
    decode(t.THEME_CLASS_ID,
       '1','Vertical Unordered List with Bullets',
       '2','Vertical Ordered List',
       '3','Horizontal Links List',
       '4','Horizontal Images with Label List',
       '5','Vertical Images List',
       '6','Button List',
       '7','Tabbed Navigation List',
       '9','Custom 1',
       '10','Custom 2',
       '11','Custom 3',
       '12','Custom 4',
       '13','Custom 5',
       '14','Custom 6',
       '15','Custom 7',
       '16','Custom 8',
       '17','Wizard Progress List',
       '18','Vertical Unordered List without Bullets',
       '19','Vertical Sidebar List',
       '20','Pull Down Menu',
       '21','Pull Down Menu with Image',
       '22','Hierarchical Expanding',
       '23','Hierarchical Expanded',
       t.THEME_CLASS_ID)             theme_class,
    t.default_template_options       default_template_options,
    t.preset_template_options        preset_template_options,
    t.TRANSLATE_THIS_TEMPLATE,
    t.LIST_TEMPLATE_COMMENT          component_comment,
    t.id                             list_template_id,
    --
    t.a01_label,
    t.a02_label,
    t.a03_label,
    t.a04_label,
    t.a05_label,
    t.a06_label,
    t.a07_label,
    t.a08_label,
    t.a09_label,
    t.a10_label,
    t.a11_label,
    t.a12_label,
    t.a13_label,
    t.a14_label,
    t.a15_label,
    t.a16_label,
    t.a17_label,
    t.a18_label,
    t.a19_label,
    t.a20_label,
    --
    t.LIST_TEMPLATE_NAME
    ||' t='||t.THEME_ID
    ||' c='||t.THEME_CLASS_ID
    ||' 1='||sys.dbms_lob.substr(t.LIST_TEMPLATE_CURRENT,40,1)||'.'||sys.dbms_lob.getlength(t.LIST_TEMPLATE_CURRENT)
    ||' 2='||sys.dbms_lob.substr(t.LIST_TEMPLATE_NONCURRENT,40,1)||'.'||sys.dbms_lob.getlength(t.LIST_TEMPLATE_NONCURRENT)
    ||' 3='||sys.dbms_lob.substr(t.SUB_LIST_ITEM_CURRENT,40,1)||'.'||sys.dbms_lob.getlength(t.SUB_LIST_ITEM_CURRENT)
    ||' 4='||sys.dbms_lob.substr(t.SUB_LIST_ITEM_NONCURRENT,40,1)||'.'||sys.dbms_lob.getlength(t.SUB_LIST_ITEM_NONCURRENT)
    ||' 5='||sys.dbms_lob.substr(t.ITEM_TEMPLATE_CURR_W_CHILD,40,1)||'.'||sys.dbms_lob.getlength(t.ITEM_TEMPLATE_CURR_W_CHILD)
    ||' 6='||sys.dbms_lob.substr(t.ITEM_TEMPLATE_NONCURR_W_CHILD,40,1)||'.'||sys.dbms_lob.getlength(t.ITEM_TEMPLATE_NONCURR_W_CHILD)
    ||' 7='||sys.dbms_lob.substr(t.SUB_TEMPLATE_CURR_W_CHILD,40,1)||'.'||sys.dbms_lob.getlength(t.SUB_TEMPLATE_CURR_W_CHILD)
    ||' 8='||sys.dbms_lob.substr(t.SUB_TEMPLATE_NONCURR_W_CHILD,40,1)||'.'||sys.dbms_lob.getlength(t.SUB_TEMPLATE_NONCURR_W_CHILD)
    ||' t='||t.TRANSLATE_THIS_TEMPLATE
    ||' r='||decode(t.REFERENCE_ID,null,'N','Y')
    ||' b='||substr(t.LIST_TEMPLATE_BEFORE_ROWS,1,20)||length(t.LIST_TEMPLATE_BEFORE_ROWS)
    ||' a='||substr(t.LIST_TEMPLATE_AFTER_ROWS,1,20)||length(t.LIST_TEMPLATE_AFTER_ROWS)
    ||' b='||substr(t.BETWEEN_ITEMS,1,20)||length(t.BETWEEN_ITEMS)
    ||' b='||substr(t.BEFORE_SUB_LIST,1,20)||length(t.BEFORE_SUB_LIST)
    ||' a='||substr(t.AFTER_SUB_LIST,1,20)||length(t.AFTER_SUB_LIST)
    ||' b='||substr(t.BETWEEN_SUB_LIST_ITEMS,1,20)||length(t.BETWEEN_SUB_LIST_ITEMS)
    component_signature
from WWV_FLOW_LIST_TEMPLATES t,
     wwv_flow_authorized     f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_list',
    p_table_comment     => 'Identifies HTML template markup used to render a List with List Elements',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'TEMPLATE_NAME'                 , 'Identifies the list template name',
      'INTERNAL_NAME'                 , 'Internal name of the list template',
      'javascript_file_urls'          , 'Allows definition of JavaScript file URLs that will be loaded on every page in the application.',
      'javascript_code_onload'        , 'Identifies JavaScript code which is executed after page load',
      'css_file_urls'                 , 'List of URLs which should be loaded for the list style.',
      'inline_css'                    , 'Inline CSS to be included on the page for styling of list.',
      'LIST_TEMPLATE_CURRENT'         , 'HTML or text to be substituted for the selected (or current) list entry',
      'LIST_TEMPLATE_NONCURRENT'      , 'HTML or text to be substituted for the non selected (or non-current) list entry',
      'LIST_TEMPLATE_BEFORE_ROWS'     , 'HTML that displays before any list elements. You can use this attribute to open an HTML table or HTML table row',
      'LIST_TEMPLATE_AFTER_ROWS'      , 'HTML that displays after list elements. You can use this attribute to close an HTML table or HTML table row',
      'BETWEEN_ITEMS'                 , 'HTML that displays between list elements',
      'BEFORE_SUB_LIST'               , 'HTML that displays before any sub list elements. ',
      'AFTER_SUB_LIST'                , 'HTML that displays after any sub list elements.',
      'BETWEEN_SUB_LIST_ITEMS'        , 'HTML that displays between sub list elements',
      'SUB_LIST_ITEM_CURRENT'         , 'HTML or text to be substituted for the selected (or current) sub list entry',
      'SUB_LIST_ITEM_NONCURRENT'      , 'HTML or text to be substituted for the unselected (or noncurrent) sub list entry',
      'ITEM_TEMPLATE_CURR_W_CHILD'    , 'HTML or text to be substituted for the selected (or current) sub list template used when an item has sub list entries',
      'ITEM_TEMPLATE_NONCURR_W_CHILD' , 'HTML or text to be substituted for the unselected (or noncurrent) list template used when item has sub list items',
      'SUB_TEMPLATE_CURR_W_CHILD'     , 'HTML or text to be substituted for the selected (or current) sub list template used when an item has sub list entries',
      'SUB_TEMPLATE_NONCURR_W_CHILD'  , 'HTML or text to be substituted for the unselected (or noncurrent) list template used when item has sub list items',
      'is_subscribed'                 , 'Identifies if this List Template is subscribed from another List Template',
      'subscribed_from'               , 'Identifies the master component from which this component is subscribed',
      'THEME_number'                  , 'Identifies the numeric identifier of this theme to which this template is associated',
      'THEME_CLASS'                   , 'Identifies a specific usage for this template',
      'default_template_options'      , 'Identifies the default template options selected for this template',
      'preset_template_options'       , 'Identifies the preset template options selected for this template',
      'TRANSLATE_THIS_TEMPLATE'       , 'Identifies if this template should be translated',
      'COMPONENT_COMMENT'             , 'Developer comment',
      'list_template_id'              , 'Primary Key of this template',
      'last_updated_by'               , 'APEX developer who made last update',
      'last_updated_on'               , 'Date of last update',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons',

      'FIRST_LIST_TEMPLATE_NONCURRENT' , 'First list template for non current entry template.  Defaults to list template non current.',
      'FIRST_LIST_TEMPLATE_CURRENT' , 'First list template for current entry template.  Defaults to list template current.',
      'F_ITEM_TEMPLATE_CURR_W_CHILD' , 'First item template for current entry with child',
      'FITEM_TEMPLATE_NONCURR_W_CHILD' , 'First item template for non current entry with child',
      'a01_label'                     , 'Identifies the definition of the substition string A01 for this template',
      'a02_label'                     , 'Identifies the definition of the substition string A02 for this template',
      'a03_label'                     , 'Identifies the definition of the substition string A03 for this template',
      'a04_label'                     , 'Identifies the definition of the substition string A04 for this template',
      'a05_label'                     , 'Identifies the definition of the substition string A05 for this template',
      'a06_label'                     , 'Identifies the definition of the substition string A06 for this template',
      'a07_label'                     , 'Identifies the definition of the substition string A07 for this template',
      'a08_label'                     , 'Identifies the definition of the substition string A08 for this template',
      'a09_label'                     , 'Identifies the definition of the substition string A09 for this template',
      'a10_label'                     , 'Identifies the definition of the substition string A10 for this template',
      'a11_label'                     , 'Identifies the definition of the substition string A11 for this template',
      'a12_label'                     , 'Identifies the definition of the substition string A12 for this template',
      'a13_label'                     , 'Identifies the definition of the substition string A13 for this template',
      'a14_label'                     , 'Identifies the definition of the substition string A14 for this template',
      'a15_label'                     , 'Identifies the definition of the substition string A15 for this template',
      'a16_label'                     , 'Identifies the definition of the substition string A16 for this template',
      'a17_label'                     , 'Identifies the definition of the substition string A17 for this template',
      'a18_label'                     , 'Identifies the definition of the substition string A18 for this template',
      'a19_label'                     , 'Identifies the definition of the substition string A19 for this template',
      'a20_label'                     , 'Identifies the definition of the substition string A20 for this template'
    )
  );
end;
/

create or replace view apex_application_temp_bc
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.THEME_ID                       theme_number,
    --
    decode(t.THEME_CLASS_ID,
      '1','Breadcrumb',
      '2','Hierarchical',
      '3','Custom 1',
      '4','Custom 2',
      '5','Custom 3',
      '6','Custom 4',
      '7','Custom 5',
      '8','Custom 6',
      '9','Custom 7',
      '10','Custom 10',
      t.THEME_CLASS_ID)              theme_class,
    t.default_template_options       default_template_options,
    t.preset_template_options        preset_template_options,
    --
    t.NAME                           template_name,
    t.INTERNAL_NAME                  internal_name,
    t.BEFORE_FIRST                   ,
    t.CURRENT_PAGE_OPTION            ,
    t.NON_CURRENT_PAGE_OPTION        ,
    t.MENU_LINK_ATTRIBUTES           breadcrumb_link_attributes,
    t.BETWEEN_LEVELS                 ,
    t.AFTER_LAST                     ,
    t.MAX_LEVELS                     ,
    decode(t.START_WITH_NODE,
       'CHILD_MENU','Child Breadcrumb Entries',
       'CURRENT_MENU','Current Breadcrumb',
       'PARENT_MENU','Parent Breadcrumb Entries',
       'PARENT_TO_LEAF','Parent to Leaf (breadcrumb style)',
       t.START_WITH_NODE)            start_with,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select mt.flow_id||'. '||mt.internal_name
      from WWV_FLOW_MENU_TEMPLATES mt
     where mt.id                = t.REFERENCE_ID
       and mt.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    decode(t.TRANSLATE_THIS_TEMPLATE,
      'Y','Yes','N','No','Yes')      translatable,
    t.TEMPLATE_COMMENTS              component_comments,
    t.id                             breadcrumb_template_id,
    --
    t.NAME
    ||' t='||t.THEME_ID
    ||' c='||t.THEME_CLASS_ID
    ||' 1='||substr(t.BEFORE_FIRST           ,1,30)||length(t.BEFORE_FIRST           )
    ||' 2='||substr(t.CURRENT_PAGE_OPTION    ,1,30)||length(t.CURRENT_PAGE_OPTION    )
    ||' 3='||substr(t.NON_CURRENT_PAGE_OPTION,1,30)||length(t.NON_CURRENT_PAGE_OPTION)
    ||' 4='||substr(t.MENU_LINK_ATTRIBUTES   ,1,30)||length(t.MENU_LINK_ATTRIBUTES   )
    ||' 5='||substr(t.BETWEEN_LEVELS         ,1,30)||length(t.BETWEEN_LEVELS         )
    ||' 6='||substr(t.AFTER_LAST             ,1,30)||length(t.AFTER_LAST             )
    ||' l='||t.MAX_LEVELS
    ||' r='||decode(t.REFERENCE_ID,null,'N','Y')
    ||' n='||t.START_WITH_NODE
    component_signature
from WWV_FLOW_MENU_TEMPLATES t,
     wwv_flow_authorized     f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_bc',
    p_table_comment     => 'Identifies the HTML template markup used to render a Breadcrumb',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'THEME_NUMBER'               , 'Identifies the theme number associated with all templates within the theme',
      'THEME_CLASS'                , 'Identifies a specific usage for this template',
      'default_template_options'   , 'Identifies the default template options selected for this template',
      'preset_template_options'    , 'Identifies the preset template options selected for this template',
      'TEMPLATE_NAME'              , 'Identifies the name of this template',
      'INTERNAL_NAME'              , 'Internal name of this menu template',
      'BEFORE_FIRST'               , 'Defines text that displays before the first breadcrumb entry.',
      'CURRENT_PAGE_OPTION'        , 'Defines current entry, use #LINK#, #NAME#, #NAME_ESC_SC#, #LONG_NAME# substitution strings',
      'NON_CURRENT_PAGE_OPTION'    , 'Defines non current entry, use #LINK#, #NAME#, #NAME_ESC_SC#, #LONG_NAME# substitution strings',
      'breadcrumb_LINK_ATTRIBUTES' , 'Displayed within the HTML "A" tag',
      'BETWEEN_LEVELS'             , 'Defines text that displays between levels of breadcrumb entries',
      'AFTER_LAST'                 , 'Defines text that displays after the last breadcrumb entry.',
      'MAX_LEVELS'                 , 'Specifies the number of levels that appear when displaying breadcrumbs in a breadcrumb style.',
      'START_WITH'                 , 'Defines the breadcrumb display style',
      'is_subscribed'              , 'Identifies if this Breadcrumb Template is subscribed from another Breadcrumb Template',
      'subscribed_from'            , 'Identifies the master component from which this component is subscribed',
      'TRANSLATABLE'               , 'Identifies if this component is to be identified as translatable (yes or no)',
      'COMPONENT_COMMENTS'         , 'Developer Comments',
      'breadcrumb_template_id'     , 'Primary Key of this Breadcrumb Template',
      'last_updated_by'            , 'APEX developer who made last update',
      'last_updated_on'            , 'Date of last update',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

create or replace view apex_application_temp_report
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.ROW_TEMPLATE_NAME              template_name,
    t.INTERNAL_NAME                  internal_name,
    --
    t.JAVASCRIPT_FILE_URLS           javascript_file_urls,
    t.JAVASCRIPT_CODE_ONLOAD         javascript_code_onload,
    t.CSS_FILE_URLS                  css_file_urls,
    --
    decode(t.ROW_TEMPLATE_TYPE,
      'GENERIC_COLUMNS','Generic Columns (column template)',
      'NAMED_COLUMNS','Named Column (row template)',
      t.ROW_TEMPLATE_TYPE)           template_type,
      --
    t.before_column_heading          ,
    t.COLUMN_HEADING_TEMPLATE        ,
    t.column_heading_sort_asc_temp,
    t.column_heading_sort_desc_temp,
    t.column_heading_sort_temp,
    t.after_column_heading           ,
    t.ROW_TEMPLATE1                  col_TEMPLATE1                ,
    t.ROW_TEMPLATE_CONDITION1        col_TEMPLATE_CONDITION1      ,
    t.ROW_TEMPLATE_DISPLAY_COND1     col_TEMPLATE_DISPLAY_COND1   ,
    t.ROW_TEMPLATE2                  col_TEMPLATE2                ,
    t.ROW_TEMPLATE_CONDITION2        col_TEMPLATE_CONDITION2      ,
    t.ROW_TEMPLATE_DISPLAY_COND2     col_TEMPLATE_DISPLAY_COND2   ,
    t.ROW_TEMPLATE3                  col_TEMPLATE3                ,
    t.ROW_TEMPLATE_CONDITION3        col_TEMPLATE_CONDITION3      ,
    t.ROW_TEMPLATE_DISPLAY_COND3     col_TEMPLATE_DISPLAY_COND3   ,
    t.ROW_TEMPLATE4                  col_TEMPLATE4                ,
    t.ROW_TEMPLATE_CONDITION4        col_TEMPLATE_CONDITION4      ,
    t.ROW_TEMPLATE_DISPLAY_COND4     col_TEMPLATE_DISPLAY_COND4   ,
    t.ROW_TEMPLATE_BEFORE_ROWS       col_TEMPLATE_BEFORE_ROWS     ,
    t.ROW_TEMPLATE_AFTER_ROWS        col_TEMPLATE_AFTER_ROWS      ,
    t.ROW_TEMPLATE_BEFORE_FIRST      col_TEMPLATE_BEFORE_FIRST    ,
    t.ROW_TEMPLATE_AFTER_LAST        col_TEMPLATE_AFTER_LAST      ,
    --t.ROW_TEMPLATE_TABLE_ATTRIBUTES  col_TEMPLATE_TABLE_ATTRIBUTES,
    t.PAGINATION_TEMPLATE            ,
    t.NEXT_PAGE_TEMPLATE             ,
    t.PREVIOUS_PAGE_TEMPLATE         ,
    t.NEXT_SET_TEMPLATE              ,
    t.PREVIOUS_SET_TEMPLATE          ,
    t.ROW_STYLE_MOUSE_OVER           ,
    --t.ROW_STYLE_MOUSE_OUT            ,
    t.ROW_STYLE_CHECKED              ,
    --t.ROW_STYLE_UNCHECKED            ,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select rt.flow_id||'. '||rt.internal_name
      from WWV_FLOW_ROW_TEMPLATES rt
     where rt.id                = t.REFERENCE_ID
       and rt.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    t.LAST_UPDATED_BY                ,
    t.LAST_UPDATED_ON                ,
    t.THEME_ID                       theme_number,
    --
    decode(t.THEME_CLASS_ID,
      '1','Borderless',
      '2','Horizontal Border',
      '3','One Column Unordered List',
      '4','Standard',
      '5','Standard, Alternating Row Colors',
      '6','Value Attribute Pairs',
      '7','Custom 1',
      '8','Custom 2',
      '9','Custom 3',
      '10','Custom 4',
      '11','Custom 5',
      '12','Custom 6',
      '13','Custom 7',
      '14','Custom 8',
      t.THEME_CLASS_ID)              theme_class,
    t.default_template_options       default_template_options,
    t.preset_template_options        preset_template_options,
    decode(t.TRANSLATE_THIS_TEMPLATE,
       'Y','Yes','N','No','Yes')     translate_this_template,
    t.ROW_TEMPLATE_COMMENT           component_comment,
    --
    t.ROW_TEMPLATE_NAME
    ||' t='||t.THEME_ID
    ||' c='||t.THEME_CLASS_ID
    ||' t='||t.ROW_TEMPLATE_TYPE
    ||' 1='||sys.dbms_lob.substr(t.ROW_TEMPLATE1,1,40)||sys.dbms_lob.getlength(t.ROW_TEMPLATE1)
    ||' c='||substr(t.ROW_TEMPLATE_CONDITION1   ,1,20)||length(t.ROW_TEMPLATE_CONDITION1)
    ||' c='||substr(t.ROW_TEMPLATE_DISPLAY_COND1,1,20)||length(t.ROW_TEMPLATE_DISPLAY_COND1)
    ||' 2='||sys.dbms_lob.substr(t.ROW_TEMPLATE2,1,40)||sys.dbms_lob.getlength(t.ROW_TEMPLATE2)
    ||' c='||substr(t.ROW_TEMPLATE_CONDITION2   ,1,20)||length(t.ROW_TEMPLATE_CONDITION2)
    ||' c='||substr(t.ROW_TEMPLATE_DISPLAY_COND2,1,20)||length(t.ROW_TEMPLATE_DISPLAY_COND2)
    ||' 3='||sys.dbms_lob.substr(t.ROW_TEMPLATE3,1,40)||sys.dbms_lob.getlength(t.ROW_TEMPLATE3)
    ||' c='||substr(t.ROW_TEMPLATE_CONDITION3   ,1,20)||length(t.ROW_TEMPLATE_CONDITION3)
    ||' c='||substr(t.ROW_TEMPLATE_DISPLAY_COND3,1,20)||length(t.ROW_TEMPLATE_DISPLAY_COND3)
    ||' 4='||sys.dbms_lob.substr(t.ROW_TEMPLATE4,1,40)||sys.dbms_lob.getlength(t.ROW_TEMPLATE4)
    ||decode(t.REFERENCE_ID,null,'N','Y')
    component_signature,
    t.id template_id
from WWV_FLOW_ROW_TEMPLATES t,
     wwv_flow_authorized    f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_report',
    p_table_comment     => 'Identifies the HTML template markup used to render a Report Headings and Rows',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'TEMPLATE_NAME'                 , 'Identifies a name for this template',
      'INTERNAL_NAME'                 , 'Internal name for this report template',
      'javascript_file_urls'          , 'Allows definition of JavaScript file URLs that will be loaded on every page in the application.',
      'javascript_code_onload'        , 'Identifies JavaScript code which is executed after page load',
      'css_file_urls'                 , 'List of URLs which should be loaded for the list style.',
      'TEMPLATE_TYPE'                 , 'Displays the type of the template - either Named columns or Generic Columns',
      'before_column_heading'         , 'Emit this before the column header cell.',
      'COLUMN_HEADING_TEMPLATE'       , 'This attribute is only applicable to generic column templates. Use this template to colorize each column header cell.',
      'after_column_heading'          , 'Emit this after the column header cell.',
      'col_TEMPLATE1'                 , 'Defines the column template, use #COLUMN_VALUE#, #ALIGNMENT#, #COLNUM#, #COLUMN_HEADER#, #COLCOUNT#, #ROW_NUM# substitutions',
      'col_TEMPLATE_CONDITION1'       , 'Optionally select a condition type that must be met in order to apply this column template',
      'col_TEMPLATE_DISPLAY_COND1'    , 'A condition that must be met in order to apply this column template',
      'col_TEMPLATE2'                 , 'Defines the column template, use #COLUMN_VALUE#, #ALIGNMENT#, #COLNUM#, #COLUMN_HEADER#, #COLCOUNT#, #ROW_NUM# substitutions',
      'col_TEMPLATE_CONDITION2'       , 'Optionally select a condition type that must be met in order to apply this column template',
      'col_TEMPLATE_DISPLAY_COND2'    , 'A condition that must be met in order to apply this column template',
      'col_TEMPLATE3'                 , 'Defines the column template, use #COLUMN_VALUE#, #ALIGNMENT#, #COLNUM#, #COLUMN_HEADER#, #COLCOUNT#, #ROW_NUM# substitutions',
      'col_TEMPLATE_CONDITION3'       , 'Optionally select a condition type that must be met in order to apply this column template',
      'col_TEMPLATE_DISPLAY_COND3'    , 'A condition that must be met in order to apply this column template',
      'col_TEMPLATE4'                 , 'Defines the column template, use #COLUMN_VALUE#, #ALIGNMENT#, #COLNUM#, #COLUMN_HEADER#, #COLCOUNT#, #ROW_NUM# substitutions',
      'col_TEMPLATE_CONDITION4'       , 'Optionally select a condition type that must be met in order to apply this column template',
      'col_TEMPLATE_DISPLAY_COND4'    , 'A condition that must be met in order to apply this column template',
      'col_TEMPLATE_BEFORE_ROWS'      , 'HTML which will be displayed one time at the beginning of a report template',
      'col_TEMPLATE_AFTER_ROWS'       , 'HTML which will be displayed one time at the beginning of a report template',
      'col_TEMPLATE_BEFORE_FIRST'     , 'Display this text before displaying all columns for the report. Use this attribute to open a new HTML row',
      'col_TEMPLATE_AFTER_LAST'       , 'Display this HTML after all columns for the report display. Typically used to close an HTML table row',
--          'col_TEMPLATE_TABLE_ATTRIBUTES' , '',
      'PAGINATION_TEMPLATE'           , 'This attribute will be applied to the entire pagination subtemplate.',
      'NEXT_PAGE_TEMPLATE'            , 'HTML that will modify how the "Next Page" portion of the pagination subtemplate will appear',
      'PREVIOUS_PAGE_TEMPLATE'        , 'HTML that will modify how the "Previous Page" portion of the pagination subtemplate will appear',
      'NEXT_SET_TEMPLATE'             , 'HTML that will modify how the "Next Set" portion of the pagination subtemplate will appear',
      'PREVIOUS_SET_TEMPLATE'         , 'HTML that will modify how the "Previous Set" portion of the pagination subtemplate will appear',
      'ROW_STYLE_MOUSE_OVER'          , 'This attribute controls the background color of a report row when the user moves the mouse over the row',
--          'ROW_STYLE_MOUSE_OUT'           , '',
      'ROW_STYLE_CHECKED'             , 'This attribute controls the background color of a report row when the row selector is checked',
--          'ROW_STYLE_UNCHECKED'           , '',
      'is_subscribed'                 , 'Identifies if this Report Template is subscribed from another Report Template',
      'subscribed_from'               , 'Identifies the master component from which this component is subscribed',
      'THEME_number'                  , 'Identifies the numeric identifier of this theme to which this template is associated',
      'THEME_CLASS'                   , 'Identifies a specific usage for this template',
      'default_template_options'      , 'Identifies the default template options selected for this template',
      'preset_template_options'       , 'Identifies the preset template options selected for this template',
      'TRANSLATE_THIS_TEMPLATE'       , 'Identifies if this template should be translated',
      'COMPONENT_COMMENT'             , 'Developer Comment',
      'last_updated_by'               , 'APEX developer who made last update',
      'last_updated_on'               , 'Date of last update',
      'component_signature'           , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'template_id'                   , 'Unique ID of report template',
      'COLUMN_HEADING_SORT_ASC_TEMP'  , 'HTML to use when heading is used to sort in ascending order',
      'COLUMN_HEADING_SORT_DESC_TEMP' , 'HTML to use when heading is used to sort in descending order',
      'COLUMN_HEADING_SORT_TEMP'      , 'HTML to use for sortable headings'
    )
  );
end;
/

create or replace view apex_appl_page_calendars
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    cal.id                          calendar_id,
    p.page_id ,
    p.plug_name                     region_name,
    cal.plug_id                     region_id,
    cal.template_id                 calendar_template_id,
    cal.easy_sql_owner              calendar_source_owner,
    cal.easy_sql_table              calendar_source,
    cal.date_column               ,
    case when nvl(cal.include_time_with_date,'Y') = 'Y' then 'Date and Time' else'Date Only' end date_format,
    cal.date_item                 ,
    cal.end_date_item             ,
    cal.date_type_column            calendar_type_column,
    case cal.display_type when 'COL' then 'Column'
                          when 'CUS' then 'Custom'
                          else 'No Display Value' end display_type,
    cal.display_column            ,
    cal.primary_key_column        ,
    cal.item_format                 column_format,
    case when nvl(cal.begin_at_start_of_interval,'N') = 'Y' then 'Yes' else 'No' end  begin_at_start_of_interval,
    cal.start_of_week               start_day_of_monthly,
    cal.week_start_day              start_day_of_weekly,
    cal.week_end_day                end_day_of_weekly,
    case when cal.time_format = '12HOUR' then '12 Hour' else '24 Hour' end time_format,
    cal.start_time                ,
    cal.end_time                  ,
    cal.data_background_color     ,
    cal.data_text_color           ,
    case cal.agenda_cal_days_type when 'MONTH' then 'Current Month'
                                  when 'YEAR' then 'Current Year'
                                  else 'Custom' end list_view_days_display,
    cal.agenda_cal_days             list_view_no_of_days,
    cal.item_link                 ,
    cal.item_link_primary_key_item,
    cal.item_link_date_item       ,
    case when cal.item_link_open_in = 'S' then 'Same Window' else 'Popup Window' end open_link_in,
    cal.day_link                  ,
    cal.drag_drop_process_id      ,
    decode(substr(cal.drag_drop_required_role,1,1),'!','Not ')||
    nvl((select ss.name
      from wwv_flow_security_schemes ss
     where to_char(ss.id)       = ltrim(cal.drag_drop_required_role,'!')
       and ss.flow_id           = f.application_id
       and ss.security_group_id = cal.security_group_id),
    cal.drag_drop_required_role)    drag_drop_authorization_scheme,
    cal.drag_drop_required_role     drag_drop_authorization_id,
    cal.last_updated_by           ,
    cal.last_updated_on
from wwv_flow_cals       cal,
     wwv_flow_page_plugs p,
     wwv_flow_authorized f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = cal.security_group_id
  and p.flow_id           = cal.flow_id
  and p.id                = cal.plug_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_page_calendars',
    p_table_comment     => 'Identifies Application Calendars',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'calendar_id'                , 'Unique ID for the Calendar',
      'page_id'                    , 'Identifies page number',
      'region_id'                  , 'The Region ID of the Calendar',
      'region_name'                , 'Identifies the region name in which this Calendar is displayed',
      'calendar_template_id'       , 'The Template used to display the Calendar Region.',
      'calendar_source_owner'      , 'Name of the User who have access to Calendar Source',
      'calendar_source'            , 'Calendar Source Table or Select statement.',
      'date_column'                , 'The Date column of the Calendar Source which will be used to display Calendar data.',
      'date_format'                , 'Identifies whether to include Time part with the Date, will be used in Week/Day and List View.',
      'date_item'                  , 'The Page Item used to store the Calendar Date',
      'end_date_item'              , 'The Page Item used to store the end date of the Calendar when in Custom View',
      'calendar_type_column'       , 'The Page Item used to store the Calendar Type',
      'display_type'               , 'Identifies how the data will be displayed',
      'display_column'             , 'The Column which will be used to diaplay in the Calendar based on Calendar date.',
      'primary_key_column'         , 'The Column of the Calendar Source which should be used as Primary key, used for Drag & Drop and Data manipulation.',
      'column_format'              , 'The Display Columns and HTML formatted Text',
      'begin_at_start_of_interval' , 'Identifies whether the Calendar should be rendered at the beginining of the Interval',
      'start_day_of_monthly'       , 'The Start day of the Week, which should be used to display for the Month view of Calendar.',
      'start_day_of_weekly'        , 'The Start Day of Week View.',
      'end_day_of_weekly'          , 'The End Day of the Week View.',
      'time_format'                , 'The format like 12 or 24 Hour which should be used for Displaying the Calendar Time.',
      'start_time'                 , 'The Start Time of Week/Day View of Calendar',
      'end_time'                   , 'The End Time of the Week/Day view of Calendar.',
      'data_background_color'      , 'The Color which should be used for Calendar Data Background.',
      'data_text_color'            , 'The Color which should be used to display Calendar Data.',
      'list_view_days_display'     , 'Identifies whether to display Current Month, Year or Custom Days in List View.',
      'list_view_no_of_days'       , 'Number of Days to display in List View.',
      'open_link_in'               , 'Identifies where the URL should be opened either in Same window or Popup.',
      'item_link'                  , 'The Page/URL link which should be used to navigate when clicking on Calendar data.',
      'day_link'                   , 'The Page/URL link which should be used to navigate when clicking on the Date.',
      'drag_drop_process_id'       , 'The Page process which should be used for Drag & Drop.',
      'drag_drop_authorization_scheme'   , 'The Authorization scheme name should be used for Drag & Drop.',
      'drag_drop_authorization_id' , 'The Authorization id should be used for Drag & Drop.',
      'item_link_primary_key_item' , 'The Page Item which should be used for passing primary key value when editing Data.',
      'item_link_date_item'        , 'The Page Item which should be used for passing date value when adding Calendar Data.',
      'last_updated_by'            , 'The Developer who last updated the Calendar region.',
      'last_updated_on'            , 'The last updated date and time of Calendar region.'
    )
  );
end;
/

create or replace view apex_application_temp_calendar
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.NAME                           template_name,
    t.INTERNAL_NAME                  internal_name,
    t.LAST_UPDATED_BY                ,
    t.LAST_UPDATED_ON                ,
    t.THEME_ID                       theme_number,
    decode(t.THEME_CLASS_ID,
       '1','Calendar',
       '2','Calendar, Alternative 1',
       '3','Small Calendar',
       '4','Custom 1',
       '5','Custom 2',
       '6','Custom 3',
       '7','Custom 4',
       '8','Custom 5',
       '9','Custom 6',
       '10','Custom 7',
       '11','Custom 8',
       t.THEME_CLASS_ID)             theme_class,
    --
    decode(t.TRANSLATE_THIS_TEMPLATE,
      'Y','Yes','N','No','Yes')      translatable,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select flow_id||'. '||name
     from WWV_FLOW_CAL_TEMPLATES
     where id = t.REFERENCE_ID)      subscribed_from,
    --
    t.MONTH_TITLE_FORMAT             ,
    t.DAY_OF_WEEK_FORMAT             ,
    t.MONTH_OPEN_FORMAT              ,
    t.MONTH_CLOSE_FORMAT             ,
    t.DAY_TITLE_FORMAT               ,
    t.DAY_OPEN_FORMAT                ,
    t.DAY_CLOSE_FORMAT               ,
    t.TODAY_OPEN_FORMAT              ,
    t.WEEKEND_TITLE_FORMAT           ,
    t.WEEKEND_OPEN_FORMAT            ,
    t.WEEKEND_CLOSE_FORMAT           ,
    t.NONDAY_TITLE_FORMAT            ,
    t.NONDAY_OPEN_FORMAT             ,
    t.NONDAY_CLOSE_FORMAT            ,
    t.WEEK_TITLE_FORMAT              ,
    t.WEEK_OPEN_FORMAT               ,
    t.WEEK_CLOSE_FORMAT              ,
    t.DAILY_TITLE_FORMAT             ,
    t.DAILY_OPEN_FORMAT              ,
    t.DAILY_CLOSE_FORMAT             ,
    t.weekly_title_format            ,
    t.weekly_day_of_week_format      ,
    t.weekly_month_open_format       ,
    t.weekly_month_close_format      ,
    t.weekly_day_title_format        ,
    t.weekly_day_open_format         ,
    t.weekly_day_close_format        ,
    t.weekly_today_open_format       ,
    t.weekly_weekend_title_format    ,
    t.weekly_weekend_open_format     ,
    t.weekly_weekend_close_format    ,
    t.weekly_time_open_format        ,
    t.weekly_time_close_format       ,
    t.weekly_time_title_format       ,
    t.weekly_hour_open_format        ,
    t.weekly_hour_close_format       ,
    t.daily_day_of_week_format       ,
    t.daily_month_title_format       ,
    t.daily_month_open_format        ,
    t.daily_month_close_format       ,
    t.daily_day_title_format         ,
    t.daily_day_open_format          ,
    t.daily_day_close_format         ,
    t.daily_today_open_format        ,
    t.daily_time_open_format         ,
    t.daily_time_close_format        ,
    t.daily_time_title_format        ,
    t.daily_hour_open_format         ,
    t.daily_hour_close_format        ,
    t.agenda_format                  ,
    t.agenda_past_day_format         ,
    t.agenda_today_day_format        ,
    t.agenda_future_day_format       ,
    t.agenda_past_entry_format       ,
    t.agenda_today_entry_format      ,
    t.agenda_future_entry_format     ,
    t.month_data_format              ,
    t.month_data_entry_format        ,
    t.TEMPLATE_COMMENTS              component_comment,
    t.id                             calendar_template_id,
    --
    t.NAME
    ||' t='||t.THEME_ID
    ||' c='||t.THEME_CLASS_ID
    component_signature
from WWV_FLOW_CAL_TEMPLATES t,
     wwv_flow_authorized    f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_calendar',
    p_table_comment     => 'Identifies the HTML template markup used to display a Calendar',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'               , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'          , 'Application Primary Key, Unique over all workspaces',
      'application_name'        , 'Identifies the application',
      'TEMPLATE_NAME'           , 'Identifies the template name',
      'INTERNAL_NAME'           , 'Internal name of the calendar template ',
      'THEME_number'            , 'Identifies the numeric identifier of this theme to which this template is associated',
      'THEME_CLASS'             , 'Identifies a specific usage for this template',
      'translatable'            , 'Identifies if this component is to be identified as translatable (yes or no)',
      'IS_SUBSCRIBED'           , 'Identifies if this Calendar Template is subscribed from another Calendar Template',
      'SUBSCRIBED_FROM'         , 'Identifies the master component from which this component is subscribed',
      'MONTH_TITLE_FORMAT'      , 'Format for the monthly title that appears at the top of each month',
      'DAY_OF_WEEK_FORMAT'      , 'Format for the week day names which display as the column header for that day of the week',
      'MONTH_OPEN_FORMAT'       , 'Printed immediately after the "Month Title Format". Usually this would be an HTML tag which is a container such as a table. Include substitution strings to include dynamic content.',
      'MONTH_CLOSE_FORMAT'      , 'HTML to be used to close a month',
      'DAY_TITLE_FORMAT'        , 'HTML to be used for the day''s title which the first thing printed after the "Day Open Format',
      'DAY_OPEN_FORMAT'         , 'HTML to be used to opening a day. This is printed for each day. Usually this would be an HTML tag which is a container such as a TD.',
      'DAY_CLOSE_FORMAT'        , 'HTML to be used to close a day',
      'TODAY_OPEN_FORMAT'       , 'HTML to be used to open today. Usually this would be an HTML tag which is a container such as a td and would be different somehow from the "Day Open".',
      'WEEKEND_TITLE_FORMAT'    , 'HTML be used to for a day occurring on a weekend',
      'WEEKEND_OPEN_FORMAT'     , 'HTML used to open a day which is on a weekend',
      'WEEKEND_CLOSE_FORMAT'    , 'HTML used to close a day which is on a weekend',
      'NONDAY_TITLE_FORMAT'     , 'HTML used to format a non-day title. For example, suppose the first of a month is a Monday, but the week starts on a Sunday. Since Sunday is not be part of the current month, Sunday would be a non-day.',
      'NONDAY_OPEN_FORMAT'      , 'HTML to open a non-day',
      'NONDAY_CLOSE_FORMAT'     , 'HTML which will close a non-day',
      'WEEK_TITLE_FORMAT'       , 'HTML be used to for a day occurring on a weekend',
      'WEEK_OPEN_FORMAT'        , 'HTML to be used to open a week. This is printed for each week. Usually this would be an HTML tag which is a container such as a TR.',
      'WEEK_CLOSE_FORMAT'       , 'HTML to be used to close a week.',
      'weekly_title_format'            , 'Format for the weekly title that appears at the top of each week',
      'weekly_day_of_week_format'      , 'Format for the week day names which display as the column header for that day of the Week',
      'weekly_month_open_format'       , 'Printed immediately after the "Week Title Format". Usually this would be an HTML tag which is a container such as a table. Include substitution strings to include dynamic content.',
      'weekly_month_close_format'      , 'HTML to be used to close a Week',
      'weekly_day_title_format'        , 'HTML to be used for the day''s title which the first thing printed after the "Day Open Format',
      'weekly_day_open_format'         , 'HTML to be used to opening a day. This is printed for each day. Usually this would be an HTML tag which is a container such as a TD.',
      'weekly_day_close_format'        , 'HTML to be used to close a day',
      'weekly_today_open_format'       , 'HTML to be used to open today. Usually this would be an HTML tag which is a container such as a td and would be different somehow from the "Day Open".',
      'weekly_weekend_title_format'    , 'HTML be used to for a day occurring on a weekend',
      'weekly_weekend_open_format'     , 'HTML used to open a day which is on a weekend',
      'weekly_weekend_close_format'    , 'HTML used to close a day which is on a weekend',
      'weekly_time_open_format'        , 'HTML to be used to display the Time. This is printed for each Hour for a week. Usually this would be an HTML tag which is a container such as a TD.',
      'weekly_time_close_format'       , 'HTML to be used to close the Time display column',
      'weekly_time_title_format'       , 'HTML be used for displaying the Time',
      'weekly_hour_open_format'        , 'HTML to be used to opening the Hour for the week. This is printed for each Hour. Usually this would be an HTML tag which is a container such as a TR.',
      'weekly_hour_close_format'       , 'HTML to be used to close the Hour',
      'daily_day_of_week_format'       , 'Format for the week day names which display as the column header for that day of the Week',
      'daily_month_title_format'       , 'Title format for the Daily Calendar.',
      'daily_month_open_format'        , 'Printed immediately after the "Day Title Format". Usually this would be an HTML tag which is a container such as a table. Include substitution strings to include dynamic content.',
      'daily_month_close_format'       , 'HTML to be used to close a Day',
      'daily_day_title_format'         , 'HTML to be used for the day''s title which the first thing printed after the "Day Open Format',
      'daily_day_open_format'          , 'HTML to be used to opening a day. This is printed for each day. Usually this would be an HTML tag which is a container such as a TD.',
      'daily_day_close_format'         , 'HTML to be used to close a day',
      'daily_today_open_format'        , 'HTML to be used to open today. Usually this would be an HTML tag which is a container such as a td and would be different somehow from the "Day Open".',
      'daily_time_open_format'         , 'HTML to be used to display the Time. This is printed for each Hour. Usually this would be an HTML tag which is a container such as a TD.',
      'daily_time_close_format'        , 'HTML to be used to close the Time display column',
      'daily_time_title_format'        , 'HTML be used for displaying the Time',
      'daily_hour_open_format'         , 'HTML to be used to opening the Hour for the Day. This is printed for each Hour. Usually this would be an HTML tag which is a container such as a TR.',
      'daily_hour_close_format'        , 'HTML to be used to close the Hour',
      'agenda_format'                  , 'HTML to be used for Open the Agenda View',
      'agenda_past_day_format'         , 'HTML to be used to display agenda entry title where Calendar Date are lesser than today''s date.',
      'agenda_today_day_format'        , 'HTML to be used to display agenda entry title where Calendar Date are equual to today''s date.',
      'agenda_future_day_format'       , 'HTML to be used to display agenda entry title where Calendar Date are greater than today''s date.',
      'agenda_past_entry_format'       , 'HTML to be used to display agenda Data where Calendar Date are lesser than today''s date.',
      'agenda_today_entry_format'      , 'HTML to be used to display agenda Data where Calendar Date are equual to today''s date.',
      'agenda_future_entry_format'     , 'HTML to be used to display agenda Data where Calendar Date are greater than today''s date.',
      'month_data_format'              , 'HTML to be used for Open and close the Month data view independent of Calendar',
      'month_data_entry_format'        , 'HTML to be used to display Month Data independent of Calendar',
      'COMPONENT_COMMENT'              , 'Developer comment',
      'last_updated_by'                , 'APEX developer who made last update',
      'last_updated_on'                , 'Date of last update',
      'calendar_template_id'           , 'Identifies the Primary Key of this Calendar Template',
      'component_signature'            , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'DAILY_TITLE_FORMAT'             , 'HTML to be used for the title of a daily calendar.',
      'DAILY_OPEN_FORMAT'              , 'HTML to be used to open a daily calendar.',
      'DAILY_CLOSE_FORMAT'             , 'HTML to be used to close a daily calendar.'
    )
  );
end;
/
create or replace view apex_application_temp_button
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.THEME_ID                       theme_number,
    decode(t.THEME_CLASS_ID,
       '1','Button',
       '2','Button, Alternative 3',
       '4','Button, Alternative 1',
       '5','Button, Alternative 2',
       '6','Custom 1',
       '7','Custom 2',
       '8','Custom 3',
       '9','Custom 4',
       '10','Custom 5',
       '11','Custom 6',
       '12','Custom 7',
       '13','Custom 8',
       t.THEME_CLASS_ID)             theme_class,
    t.default_template_options       default_template_options,
    t.preset_template_options        preset_template_options,
    --
    t.TEMPLATE_NAME                  template_name,
    t.INTERNAL_NAME                  internal_name,
    t.TEMPLATE,
    t.hot_template,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select bt.flow_id||'. '||bt.internal_name
      from WWV_FLOW_button_TEMPLATES bt
     where bt.id                = t.REFERENCE_ID
       and bt.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    t.LAST_UPDATED_BY                last_updated_by,
    t.LAST_UPDATED_ON                last_updated_on,
    --
    decode(t.TRANSLATE_THIS_TEMPLATE,
      'Y','Yes','N','No','Yes')      translatable,
    t.TEMPLATE_COMMENT               component_comment,
    t.id                             button_template_id,
    --
    t.template_name
    ||' t='||t.THEME_ID
    ||' c='||t.THEME_CLASS_ID
    component_signature
from wwv_flow_button_templates t,
     wwv_flow_authorized       f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_button',
    p_table_comment     => 'Identifies the HTML template markup used to display a Button',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                 , 'A work area mapped to one or more database schemas',
      'workspace_display_name'    , 'Display name for the workspace',
      'application_id'            , 'Application Primary Key, Unique over all workspaces',
      'application_name'          , 'Identifies the application',
      'THEME_NUMBER'              , 'Identifies the templates corresponding theme',
      'THEME_CLASS'               , 'Identifies a specific usage for this template',
      'default_template_options'  , 'Identifies the default template option selected for this template',
      'preset_template_options'   , 'Identifies the preset template option selected for this template',
      'TEMPLATE_NAME'             , 'Identifies the button template',
      'INTERNAL_NAME'             , 'Internal name of the button template',
      'TEMPLATE'                  , 'HTML Template of a normal button',
      'hot_template'              , 'HTML Template of a hot button',
      'is_subscribed'             , 'Identifies if this Button Template is subscribed from another Button Template',
      'SUBSCRIBED_FROM'           , 'Identifies the master component from which this component is subscribed',
      'TRANSLATable'              , 'Identifies if this component is to be identified as translatable (yes or no)',
      'COMPONENT_COMMENT'         , 'Developer comment',
      'last_updated_by'           , 'APEX developer who made last update',
      'last_updated_on'           , 'Date of last update',
      'button_template_id'        , 'Primary Key of this Button Template',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
create or replace view apex_application_temp_popuplov
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    --
    t.POPUP_ICON                     ,
    t.POPUP_ICON_ATTR                ,
    t.POPUP_ICON2                    ,
    t.POPUP_ICON_ATTR2               ,
    --t.PAGE_NAME                      ,
    t.PAGE_TITLE                     ,
    t.PAGE_HTML_HEAD                 ,
    t.PAGE_BODY_ATTR                 ,
    t.BEFORE_FIELD_TEXT              ,
    t.AFTER_FIELD_TEXT               ,
    t.PAGE_HEADING_TEXT              ,
    t.PAGE_FOOTER_TEXT               ,
    t.BEFORE_RESULT_SET              ,
    t.AFTER_RESULT_SET               ,
    t.FILTER_WIDTH                   ,
    t.FILTER_MAX_WIDTH               ,
    t.FILTER_TEXT_ATTR               ,
    t.FIND_BUTTON_TEXT               ,
    t.FIND_BUTTON_IMAGE              ,
    t.FIND_BUTTON_ATTR               ,
    t.CLOSE_BUTTON_TEXT              ,
    t.CLOSE_BUTTON_IMAGE             ,
    t.CLOSE_BUTTON_ATTR              ,
    t.NEXT_BUTTON_TEXT               ,
    t.NEXT_BUTTON_IMAGE              ,
    t.NEXT_BUTTON_ATTR               ,
    t.PREV_BUTTON_TEXT               ,
    t.PREV_BUTTON_IMAGE              ,
    t.PREV_BUTTON_ATTR               ,
    t.SCROLLBARS                     ,
    t.RESIZABLE                      ,
    t.WIDTH                          ,
    t.HEIGHT                         ,
    t.RESULT_ROW_X_OF_Y              ,
    t.RESULT_ROWS_PER_PG             ,
    t.WHEN_NO_DATA_FOUND_MESSAGE     ,
    t.BEFORE_FIRST_FETCH_MESSAGE     ,
    t.MINIMUM_CHARACTERS_REQUIRED    ,
    t.THEME_ID                       theme_number,
    --
    decode(t.THEME_CLASS_ID,
     '1','Standard',
     t.THEME_CLASS_ID)               theme_class,
    --
    decode(t.REFERENCE_ID,
    null,'No','Yes')                 is_subscribed,
    (select lt.flow_id||'. '||lt.page_title
      from WWV_FLOW_POPUP_LOV_TEMPLATE lt
     where lt.id                = t.REFERENCE_ID
       and lt.security_group_id in (t.security_group_id, 12))      subscribed_from,
    --
    t.LAST_UPDATED_ON                last_updated_on,
    t.LAST_UPDATED_BY                last_updated_by,
    decode(t.TRANSLATE_THIS_TEMPLATE,
      'N','No','Y','Yes','Yes')      translatable,
    t.TEMPLATE_COMMENT               component_comment,
    t.id                             template_id,
    --
    'pop'
    ||' ='||substr(POPUP_ICON                 ,1,20)||length(POPUP_ICON                 )
    ||' ='||substr(POPUP_ICON_ATTR            ,1,20)||length(POPUP_ICON_ATTR            )
    ||' ='||substr(POPUP_ICON2                ,1,20)||length(POPUP_ICON2                )
    ||' ='||substr(POPUP_ICON_ATTR2           ,1,20)||length(POPUP_ICON_ATTR2           )
    ||' ='||substr(PAGE_TITLE                 ,1,20)||length(PAGE_TITLE                 )
    ||' ='||substr(PAGE_HTML_HEAD             ,1,20)||length(PAGE_HTML_HEAD             )
    ||' ='||substr(PAGE_BODY_ATTR             ,1,20)||length(PAGE_BODY_ATTR             )
    ||' ='||substr(BEFORE_FIELD_TEXT          ,1,20)||length(BEFORE_FIELD_TEXT          )
    ||' ='||substr(AFTER_FIELD_TEXT           ,1,20)||length(AFTER_FIELD_TEXT           )
    ||' ='||substr(PAGE_HEADING_TEXT          ,1,20)||length(PAGE_HEADING_TEXT          )
    ||' ='||substr(PAGE_FOOTER_TEXT           ,1,20)||length(PAGE_FOOTER_TEXT           )
    ||' ='||substr(BEFORE_RESULT_SET          ,1,20)||length(BEFORE_RESULT_SET          )
    ||' ='||substr(AFTER_RESULT_SET           ,1,20)||length(AFTER_RESULT_SET           )
    ||' ='||substr(FILTER_WIDTH               ,1,20)||length(FILTER_WIDTH               )
    ||' ='||substr(FILTER_MAX_WIDTH           ,1,20)||length(FILTER_MAX_WIDTH           )
    ||' ='||substr(FILTER_TEXT_ATTR           ,1,20)||length(FILTER_TEXT_ATTR           )
    ||' ='||substr(FIND_BUTTON_TEXT           ,1,20)||length(FIND_BUTTON_TEXT           )
    ||' ='||substr(FIND_BUTTON_IMAGE          ,1,20)||length(FIND_BUTTON_IMAGE          )
    ||' ='||substr(FIND_BUTTON_ATTR           ,1,20)||length(FIND_BUTTON_ATTR           )
    ||' ='||substr(CLOSE_BUTTON_TEXT          ,1,20)||length(CLOSE_BUTTON_TEXT          )
    ||' ='||substr(CLOSE_BUTTON_IMAGE         ,1,20)||length(CLOSE_BUTTON_IMAGE         )
    ||' ='||substr(CLOSE_BUTTON_ATTR          ,1,20)||length(CLOSE_BUTTON_ATTR          )
    ||' ='||substr(NEXT_BUTTON_TEXT           ,1,20)||length(NEXT_BUTTON_TEXT           )
    ||' ='||substr(NEXT_BUTTON_IMAGE          ,1,20)||length(NEXT_BUTTON_IMAGE          )
    ||' ='||substr(NEXT_BUTTON_ATTR           ,1,20)||length(NEXT_BUTTON_ATTR           )
    ||' ='||substr(PREV_BUTTON_TEXT           ,1,20)||length(PREV_BUTTON_TEXT           )
    ||' ='||substr(PREV_BUTTON_IMAGE          ,1,20)||length(PREV_BUTTON_IMAGE          )
    ||' ='||substr(PREV_BUTTON_ATTR           ,1,20)||length(PREV_BUTTON_ATTR           )
    ||' ='||substr(SCROLLBARS                 ,1,20)||length(SCROLLBARS                 )
    ||' ='||substr(RESIZABLE                  ,1,20)||length(RESIZABLE                  )
    ||' ='||substr(WIDTH                      ,1,20)||length(WIDTH                      )
    ||' ='||substr(HEIGHT                     ,1,20)||length(HEIGHT                     )
    ||' ='||substr(RESULT_ROW_X_OF_Y          ,1,20)||length(RESULT_ROW_X_OF_Y          )
    ||' ='||RESULT_ROWS_PER_PG
    ||' ='||substr(WHEN_NO_DATA_FOUND_MESSAGE ,1,20)||length(WHEN_NO_DATA_FOUND_MESSAGE )
    ||' ='||substr(BEFORE_FIRST_FETCH_MESSAGE ,1,20)||length(BEFORE_FIRST_FETCH_MESSAGE )
    ||' ='||substr(MINIMUM_CHARACTERS_REQUIRED,1,20)||length(MINIMUM_CHARACTERS_REQUIRED)
    ||' r='||decode(t.REFERENCE_ID,null,'N','Y')
    component_signature
from WWV_FLOW_POPUP_LOV_TEMPLATE t,
     wwv_flow_authorized         f
where f.workspace_id   = t.security_group_id
  and f.application_id = t.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_temp_popuplov',
    p_table_comment     => 'Identifies the HTML template markup and some functionality of all Popup List of Values controls for this application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'POPUP_ICON'                    , 'Name of the image used for all Popup List of Values Icons within this application',
      'POPUP_ICON_ATTR'               , 'HTML IMG attributes used to render the Popup List of Values Image',
      'POPUP_ICON2'                   , 'Name of the image used for Color Picker Popup List of Values Icon within this application',
      'POPUP_ICON_ATTR2'              , 'HTML IMG attributes used to render the Color Picker Popup List of Values Image',
      'PAGE_TITLE'                    , 'Page title of the popup list of values page',
      'PAGE_HTML_HEAD'                , 'HTML Page Heading',
      'PAGE_BODY_ATTR'                , 'Defines text that is added into the popup list of values HTML BODY',
      'BEFORE_FIELD_TEXT'             , 'Defines text to display before the popup list of values search field displays',
      'AFTER_FIELD_TEXT'              , 'Defines text to display after displaying the search field, the search button, and the close button',
      'PAGE_HEADING_TEXT'             , 'Defines text to display after opening the form, but before displaying any text',
      'PAGE_FOOTER_TEXT'              , 'Defines text to display after displaying the popup page including the search list results',
      'BEFORE_RESULT_SET'             , 'Display this text before displaying the result set. The result set is the result of the list of values query',
      'AFTER_RESULT_SET'              , 'Display this text after displaying the result set. The result set is the result of the list of values query',
      'FILTER_WIDTH'                  , 'Display the HTML INPUT TYPE = TEXT widget using this width',
      'FILTER_MAX_WIDTH'              , 'Display the HTML INPUT TYPE = TEXT widget using this maximum width',
      'FILTER_TEXT_ATTR'              , 'Display the HTML INPUT TYPE = TEXT widget using these attributes',
      'FIND_BUTTON_TEXT'              , 'Defines text that displays on the name of the button used to search on the popup page',
      'FIND_BUTTON_IMAGE'             , 'Defines button image of the button used to search on the popup page',
      'FIND_BUTTON_ATTR'              , 'Defines additional attributes for the Find button',
      'CLOSE_BUTTON_TEXT'             , 'Defines the name of the close button. The close button dismisses the popup list of values page',
      'CLOSE_BUTTON_IMAGE'            , 'Defines close button image. The close button dismisses the popup list of values page',
      'CLOSE_BUTTON_ATTR'             , 'Defines additional attributes of the Close button',
      'NEXT_BUTTON_TEXT'              , 'Defines the name of the Next button used when paginating result sets',
      'NEXT_BUTTON_IMAGE'             , 'Defines the button image of the Next button used when paginating result sets',
      'NEXT_BUTTON_ATTR'              , 'Defines additional attributes for the Next button',
      'PREV_BUTTON_TEXT'              , 'Defines the name of Previous page button. This button is used to paginate result sets',
      'PREV_BUTTON_IMAGE'             , 'Defines the button image of Previous page button. This button is used to paginate result sets.',
      'PREV_BUTTON_ATTR'              , 'Defines additional attributes of the Previous button',
      'SCROLLBARS'                    , 'Popup lists of values are executed using JavaScript. Use this attribute to control the values of scrollbars',
      'RESIZABLE'                     , 'Popup lists of values are executed using JavaScript. Use this attribute to control the value of resizable',
      'WIDTH'                         , 'Popup lists of values are executed using JavaScript. Use this attribute to control the values of width',
      'HEIGHT'                        , 'Popup lists of values are executed using JavaScript. Use this attribute to control the values of height',
      'RESULT_ROW_X_OF_Y'             , 'Defines how row count results display',
      'RESULT_ROWS_PER_PG'            , 'Defines the number of rows per page',
      'WHEN_NO_DATA_FOUND_MESSAGE'    , 'Message to display when no data is found for a search',
      'BEFORE_FIRST_FETCH_MESSAGE'    , 'Message to display before the first search is performed',
      'MINIMUM_CHARACTERS_REQUIRED'   , 'Minimum number of charaters which must be entered before a search is performed',
      'THEME_NUMBER'                  , 'Identifies the templates corresponding theme',
      'THEME_CLASS'                   , 'Identifies a specific usage for this template',
      'is_subscribed'                 , 'Identifies if this Popup List of Values Template is subscribed from another Popup List of Values Template',
      'subscribed_from'               , 'Identifies the master component from which this component is subscribed',
      'TRANSLATable'                  , 'Identifies if this component is to be identified as translatable (yes or no)',
      'COMPONENT_COMMENT'             , 'Developer comment',
      'template_id'                   , 'Primary Key of this Popup List of Values Template',
      'last_updated_by'               , 'APEX developer who made last update',
      'last_updated_on'               , 'Date of last update',
      'component_signature'           , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

create or replace view apex_application_templates
as
select
  ff.workspace,
  ff.workspace_display_name,
  ff.application_id,
  ff.application_name,
  x.theme_id                            theme_number,
  decode(x.template_type,
  'PAGE','Page',
  'REGION','Region',
  'LABEL','Item Label',
  'LIST','List',
  'POPUP_LOV','Popup List of Values',
  'CALENDAR','Calendar',
  'MENU','Breadcrumb',
  'BUTTON','Button',
  'REPORT','Report',
  'GRID', 'Grid',
  x.template_type)                      template_type,
  x.template_name                       template_name,
  x.internal_name                       internal_name,
  x.reference_count                     reference_count,
  x.last_updated_on                     last_updated_on,
  x.last_updated_by                     last_updated_by,
  decode(x.reference_id,null,
      'No',0,'No','Yes')                is_subscribed,
  decode(x.is_default,0,'No','Yes')     is_default,
  tid                                   template_id
from wwv_flow_authorized ff,
(
select
    flow_id,
    security_group_id,
    'POPUP_LOV' template_type,
    'Popup LOV' template_name,
    'POPUP_LOV' internal_name,
    1 reference_count,
    last_updated_on,
    last_updated_by,
    null reference_id,
    1 is_default,
    0 preview,
    theme_id,
    id tid
from wwv_flow_popup_lov_template plt
union all
select
    flow_id,
    security_group_id,
    'CALENDAR' template_type,
    NAME template_name,
    internal_name,
    (select count(*) from WWV_FLOW_CALS ca where ca.template_id = c.id and ca.flow_id = c.flow_id and ca.security_group_id = c.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = c.flow_id and th.theme_id = c.theme_id and th.default_calendar_template = c.id and th.security_group_id = c.security_group_id) reference_count,
    last_updated_on,
    last_updated_by,
    reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = c.flow_id and th.theme_id = c.theme_id and th.default_calendar_template = c.id and th.security_group_id = c.security_group_id) is_default,
    0 preview,
    theme_id,
    id tid
from wwv_flow_cal_templates c
union all
select
    flow_id,
    security_group_id,
    'MENU'   template_type,
    NAME     template_name,
    internal_name internal_name,
    (select count(*) from wwv_flow_page_plugs r where r.flow_id = m.flow_id and r.menu_template_id = m.id and r.security_group_id = m.security_group_id)+
    (select count(*) from wwv_flow_themes th where th.flow_id = m.flow_id and th.theme_id = m.theme_id and th.default_menu_template = m.id and th.security_group_id = m.security_group_id)
    reference_count,
    last_updated_on,
    last_updated_by,
    reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = m.flow_id and th.theme_id = m.theme_id and th.default_menu_template = m.id and th.security_group_id = m.security_group_id) is_default,
    0 preview,
    theme_id,
    id tid
from wwv_flow_menu_templates m
union all
select
    flow_id,
    security_group_id,
    'BUTTON'             template_type,
    TEMPLATE_NAME        template_name,
    internal_name        internal_name,
    (select count(*)
       from wwv_flow_step_buttons sb
      where sb.flow_id            = b.flow_id
        and sb.button_template_id = b.id
        and sb.security_group_id  = b.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = b.flow_id and th.theme_id = b.theme_id and th.default_button_template = b.id and th.security_group_id = b.security_group_id) reference_count,
    last_updated_on,
    last_updated_by,
    reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = b.flow_id and th.theme_id = b.theme_id and th.default_button_template = b.id and th.security_group_id = b.security_group_id) is_default,
    0 preview,
    theme_id,
    id tid
from wwv_flow_button_templates b
union all
select
    flow_id,
    security_group_id,
    'LABEL' template_type,
    TEMPLATE_NAME template_name,
    internal_name internal_name,
    (select count(*) from wwv_flow_step_items i where i.flow_id = f.flow_id and i.ITEM_FIELD_TEMPLATE = f.id and i.security_group_id = f.security_group_id)+
    (select count(*) from wwv_flow_themes th where th.flow_id = f.flow_id and th.theme_id = f.theme_id and th.default_label_template = f.id and th.security_group_id = f.security_group_id) reference_count,
    last_updated_on, last_updated_by, reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = f.flow_id and th.theme_id = f.theme_id and th.default_label_template = f.id and th.security_group_id = f.security_group_id) is_default,
    0 preview,
    theme_id,
    id tid
from wwv_flow_field_templates f
union all
select
    flow_id,
    security_group_id,
    'LIST' template_type,
    l.LIST_TEMPLATE_NAME template_name,
    internal_name        internal_name,
    (select count(*) from wwv_flow_page_plugs p where p.flow_id = l.flow_id and p.list_template_id = l.id and p.security_group_id = l.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = l.flow_id and th.theme_id = l.theme_id and th.default_list_template = l.id and th.security_group_id = l.security_group_id) reference_count,
    last_updated_on, last_updated_by, reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = l.flow_id and th.theme_id = l.theme_id and th.default_list_template = l.id and th.security_group_id = l.security_group_id) is_default,
    0 preview,
    theme_id,
    id tid
from wwv_flow_list_templates l
union all
select
    flow_id,
    security_group_id,
    'REPORT'                 template_type,
    ROW_TEMPLATE_NAME        template_name,
    internal_name            internal_name,
    (select count(*) from wwv_flow_page_plugs p where p.flow_id = ro.flow_id and p.PLUG_QUERY_ROW_TEMPLATE = ro.id and p.security_group_id = ro.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = ro.flow_id and th.theme_id = ro.theme_id and th.default_report_template = ro.id and th.security_group_id = ro.security_group_id) reference_count,
    last_updated_on, last_updated_by, reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = ro.flow_id and th.theme_id = ro.theme_id and th.default_report_template = ro.id and th.security_group_id = ro.security_group_id) is_default,
    ro.id preview,
    theme_id,
    id tid
from wwv_flow_row_templates ro
union all
select
    flow_id,
    security_group_id,
    'PAGE'        template_type,
    name          template_name,
    internal_name internal_name,
    (select count(*) from wwv_flow_steps s where s.flow_id = p.flow_id and s.step_template = p.id and s.security_group_id = p.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = p.flow_id and th.theme_id = p.theme_id and th.DEFAULT_PAGE_TEMPLATE = p.id and th.security_group_id = p.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = p.flow_id and th.theme_id = p.theme_id and th.error_template=p.id and th.security_group_id = p.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = p.flow_id and th.theme_id = p.theme_id and th.printer_friendly_template=p.id and th.security_group_id = p.security_group_id) reference_count,
    last_updated_on, last_updated_by, reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = p.flow_id and th.theme_id = p.theme_id and th.DEFAULT_PAGE_TEMPLATE=p.id and th.security_group_id = p.security_group_id) is_default,
    p.id preview,
    theme_id,
    id tid
from wwv_flow_templates p
union all
select
    flow_id,
    security_group_id,
    'REGION' template_type,
    PAGE_PLUG_TEMPLATE_NAME template_name,
    internal_name           internal_name,
    (select count(*) from wwv_flow_page_plugs p where p.flow_id = r.flow_id and p.PLUG_TEMPLATE = r.id and p.security_group_id = r.security_group_id) +
    (select count(*) from wwv_flow_themes th where th.flow_id = r.flow_id and th.theme_id = r.theme_id and th.default_region_template = r.id and th.security_group_id = r.security_group_id) reference_count,
    last_updated_on,
    last_updated_by,
    reference_id,
    (select count(*) from wwv_flow_themes th where th.flow_id = r.flow_id and th.theme_id = r.theme_id and th.default_region_template = r.id and th.security_group_id = r.security_group_id)  is_default,
    r.id preview,
    theme_id,
    id tid
from wwv_flow_page_plug_templates r
) x
where ff.workspace_id   = x.security_group_id
  and ff.application_id = x.flow_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_templates',
    p_table_comment     => 'Identifies reference counts for templates of all types',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'         , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'    , 'Application Primary Key, Unique over all workspaces',
      'application_name'  , 'Identifies the application',
      'THEME_number'      , 'Identifies the numeric identifier of this theme to which this template is associated',
      'TEMPLATE_TYPE'     , 'Identifies the Template Type',
      'TEMPLATE_NAME'     , 'Name of the template',
      'INTERNAL_NAME'     , 'Internal name of the template',
      'REFERENCE_COUNT'   , 'Number of references to this template',
      'LAST_UPDATED_ON'   , 'Date of last update',
      'LAST_UPDATED_BY'   , 'APEX User Name who made the last update',
      'IS_SUBSCRIBED'     , 'Identifies if this template is subscribed from another template',
      'IS_DEFAULT'        , 'Identifies this template as the default template for new components created with a corresponding type',
      'TEMPLATE_ID'       , 'Primary key of this component'
    )
  );
end;
/



--------------------------------------------
-- S I T E   M A P
--
prompt ...apex_application_page_map

create or replace view apex_application_page_map as
select a.workspace,
       a.workspace_display_name,
       a.application_id,
       a.application_name,
       p.id                         page_id,
       p.name                       page_name,
       p.step_title                 page_title,
       --
       o1.short_name                breadcrumb,
       ltrim (
           o8.short_name||' > '||
           o7.short_name||' > '||
           o6.short_name||' > '||
           o5.short_name||' > '||
           o4.short_name||' > '||
           o3.short_name||' > '||
           o2.short_name||' > '||
           o1.short_name,
           ' >' )                   full_breadcrumb,
       help_text
from wwv_flow_authorized a,
     wwv_flow_steps p,
     ( select row_number() over (partition by p.security_group_id,
                                              p.flow_id,
                                              p.page_id
                                 order by p.id) rn,
              p.security_group_id,
              p.flow_id,
              p.page_id,
              o.short_name,
              o.menu_id,
              o.parent_id
         from wwv_flow_page_plugs p,
              wwv_flow_menu_options o
        where p.menu_id           = o.menu_id
          and p.page_id           = o.page_id
          and p.security_group_id = o.security_group_id) o1,
     wwv_flow_menu_options o2,
     wwv_flow_menu_options o3,
     wwv_flow_menu_options o4,
     wwv_flow_menu_options o5,
     wwv_flow_menu_options o6,
     wwv_flow_menu_options o7,
     wwv_flow_menu_options o8
where a.workspace_id          = p.security_group_id
  and a.application_id        = p.flow_id
  and p.security_group_id     = o1.security_group_id(+)
  and p.flow_id               = o1.flow_id(+)
  and p.id                    = o1.page_id(+)
  and (o1.rn is null or o1.rn = 1)
  and o1.parent_id            = o2.id(+)
  and o1.flow_id              = o2.flow_id(+)
  and o1.security_group_id    = o2.security_group_id(+)
  and o2.parent_id            = o3.id(+)
  and o2.flow_id              = o3.flow_id(+)
  and o2.security_group_id    = o3.security_group_id(+)
  and o3.parent_id            = o4.id(+)
  and o3.flow_id              = o4.flow_id(+)
  and o3.security_group_id    = o4.security_group_id(+)
  and o4.parent_id            = o5.id(+)
  and o4.flow_id              = o5.flow_id(+)
  and o4.security_group_id    = o5.security_group_id(+)
  and o5.parent_id            = o6.id(+)
  and o5.flow_id              = o6.flow_id(+)
  and o5.security_group_id    = o6.security_group_id(+)
  and o6.parent_id            = o7.id(+)
  and o6.flow_id              = o7.flow_id(+)
  and o6.security_group_id    = o7.security_group_id(+)
  and o7.parent_id            = o8.id(+)
  and o7.flow_id              = o8.flow_id(+)
  and o7.security_group_id    = o8.security_group_id(+)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_map',
    p_table_comment     => 'Identifies the full breadcrumb path for each page with a breadcrumb entry',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'         , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'    , 'Application Primary Key, Unique over all workspaces',
      'application_name'  , 'Identifies the application',
      'page_id'           , 'Identifies the application',
      'page_name'         , 'Identifies a page within an application',

      'PAGE_TITLE'        , 'Identifies the Page Title',
      'BREADCRUMB'        , 'Identifies the corresponding Page Breadcrumb Entry Text',
      'FULL_BREADCRUMB'   , 'Identifies the full breadcrumb hierarchy',
      'HELP_TEXT'         , 'Identifies the help text associated with the page'
    )
  );
end;
/

---------------------------------------------------
-- A P E X    D I C T I O N A R Y   C O M M E N T S
-- includes column and view names, view names have column id of zero
-- parent view name is used to display views in hierarchy

create or replace view apex_dictionary as
select apex_view_name,
       column_id,
       column_name,
       comments,
       comment_type,
       parent_view
from
(
select t.table_name   apex_view_name,
       t.column_id    column_id,
       t.column_name  column_name,
       c.comments     comments,
       'Column'       comment_type,
       case
         when        t.table_name      ='APEX_REST_CATALOG_GROUPS'       then 'APEX_WORKSPACES'
         when        t.table_name      ='APEX_REST_CATALOGS'             then 'APEX_REST_CATALOG_GROUPS'
         when        t.table_name      ='APEX_REST_CATALOG_PLUGINS'      then 'APEX_REST_CATALOGS'
         when        t.table_name      ='APEX_REST_CATALOG_SERVICES'     then 'APEX_REST_CATALOGS'
         when        t.table_name      ='APEX_APPL_WEB_SRC_OPERATIONS'   then 'APEX_APPL_WEB_SRC_MODULES'
         when        t.table_name      ='APEX_APPL_WEB_SRC_SYNC_STEPS'   then 'APEX_APPL_WEB_SRC_MODULES'
         when        t.table_name      ='APEX_APPL_WEB_SRC_PARAMETERS'   then 'APEX_APPL_WEB_SRC_OPERATIONS'
         when        t.table_name      ='APEX_APPL_WEB_SRC_COMP_PARAMS'  then 'APEX_APPL_WEB_SRC_PARAMETERS'
         when        t.table_name      ='APEX_APPLICATIONS'              then 'APEX_WORKSPACES'
         when        t.table_name      ='APEX_APPLICATION_LOV_ENTRIES'   then 'APEX_APPLICATION_LOVS'
         when        t.table_name      ='APEX_APPLICATION_LOV_COLS'      then 'APEX_APPLICATION_LOVS'
         when        t.table_name      ='APEX_APPLICATION_PAGE_ITEMS'    then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      ='APEX_APPL_PAGE_FILTER_GROUPS'   then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      ='APEX_APPLICATION_PAGE_BUTTONS'  then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      ='APEX_APPL_PAGE_FILTERS'         then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      ='APEX_APPL_PLUGIN_ATTRIBUTES'    then 'APEX_APPL_PLUGINS'
         when        t.table_name      ='APEX_APPL_PLUGIN_STD_ATTRS'     then 'APEX_APPL_PLUGINS'
         when        t.table_name      ='APEX_APPL_PLUGIN_ATTR_VALUES'   then 'APEX_APPL_PLUGIN_ATTRIBUTES'
         when        t.table_name      ='APEX_APPL_PLUGIN_FILES'         then 'APEX_APPL_PLUGINS'
         when        t.table_name      ='APEX_APPL_PLUGIN_EVENTS'        then 'APEX_APPL_PLUGINS'
         when        t.table_name      ='APEX_APPL_PLUGIN_ITEM_FILTERS'  then 'APEX_APPL_PLUGINS'
         when        t.table_name      ='APEX_APPL_ACL_USER_ROLES'       then 'APEX_APPL_ACL_USERS'
         when        t.table_name      ='APEX_APPLICATION_LIST_ENTRIES'  then 'APEX_APPLICATION_LISTS'
         when        t.table_name      ='APEX_APPLICATION_RPT_QRY_STMTS' then 'APEX_APPLICATION_RPT_QUERIES'
         when        t.table_name      ='APEX_APPLICATION_TEMPLATES'     then 'APEX_APPLICATION_THEMES'
         when        t.table_name      ='APEX_APPLICATION_THEME_STYLES'  then 'APEX_APPLICATION_THEMES'
         when        t.table_name      ='APEX_APPLICATION_THEME_FILES'   then 'APEX_APPLICATION_THEMES'
         when        t.table_name      ='APEX_APPL_TEMPLATE_OPTIONS'     then 'APEX_APPLICATION_THEMES'
         when        t.table_name      ='APEX_APPL_TEMPLATE_OPT_GROUPS'  then 'APEX_APPLICATION_THEMES'
         when        t.table_name      ='APEX_APPL_TEMP_PAGE_DP'         then 'APEX_APPLICATION_TEMP_PAGE'
         when        t.table_name      ='APEX_APPL_TEMP_REGION_DP'       then 'APEX_APPLICATION_TEMP_REGION'
         when substr(t.table_name,1,26)='APEX_APPLICATION_SUPP_OBJ_'     then 'APEX_APPLICATION_SUPP_OBJECTS'
         when substr(t.table_name,1,18)='APEX_WORKSPACE_LOG'             then 'APEX_WORKSPACE_ACTIVITY_LOG'
         when substr(t.table_name,1,21)='APEX_APPLICATION_TEMP'          then 'APEX_APPLICATION_THEMES'
         when substr(t.table_name,1,30)='APEX_APPLICATION_PAGE_RPT_COLS' then 'APEX_APPLICATION_PAGE_RPT'
         when substr(t.table_name,1,25)='APEX_APPLICATION_PAGE_RPT'      then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      ='APEX_APPLICATION_PAGE_REG_COLS' then 'APEX_APPLICATION_PAGE_REGIONS'
         when substr(t.table_name,1,30)='APEX_APPLICATION_PAGE_CHART_S'  then 'APEX_APPLICATION_PAGE_CHARTS'
         when substr(t.table_name,1,30)='APEX_APPLICATION_PAGE_CHART_A'  then 'APEX_APPLICATION_PAGE_CHARTS'
         when substr(t.table_name,1,27)='APEX_APPLICATION_PAGE_TREES'    then 'APEX_APPLICATION_PAGE_TREES'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_COMP'  then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_COND'  then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_SUB'   then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_GRPBY' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_PIVOT' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_PVAGG' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_PVSRT' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when substr(t.table_name,1,25)='APEX_APPLICATION_PAGE_IR_'      then 'APEX_APPLICATION_PAGE_IR'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR'       then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_COLUMNS'      then 'APEX_APPL_PAGE_IGS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_COL_GROUPS'   then 'APEX_APPL_PAGE_IGS'
         when        t.table_name      = 'APEX_APPL_PAGE_IGS'             then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPTS'         then 'APEX_APPL_PAGE_IGS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_VIEWS'    then 'APEX_APPL_PAGE_IG_RPTS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_COMPUTES' then 'APEX_APPL_PAGE_IG_RPTS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_FILTERS'  then 'APEX_APPL_PAGE_IG_RPTS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_COLUMNS'  then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_HIGHLTS'  then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_AGGS'     then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_CHT_COLS' then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_CARDS'           then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      = 'APEX_APPL_PAGE_CARD_ACTIONS'    then 'APEX_APPL_PAGE_CARDS'
         when        t.table_name      = 'APEX_APPL_PAGE_MAPS'            then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      = 'APEX_APPL_PAGE_MAP_LAYERS'      then 'APEX_APPL_PAGE_MAPS'
         when        t.table_name      ='APEX_APPLICATION_PAGE_CHARTS'   then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.table_name      ='APEX_APPLICATION_PAGE_DA_ACTS'  then 'APEX_APPLICATION_PAGE_DA'
         when substr(t.table_name,1,22)='APEX_APPLICATION_PAGE_'         then 'APEX_APPLICATION_PAGES'
         when substr(t.table_name,1,15)='APEX_WORKSPACE_'                then 'APEX_WORKSPACES'
         when substr(t.table_name,1,15)='APEX_APPL_PAGE_'                then 'APEX_APPLICATION_PAGES' --#14833918
         when substr(t.table_name,1,17)='APEX_APPLICATION_' or
              substr(t.table_name,1,10)='APEX_APPL_'                     then 'APEX_APPLICATIONS'
         when        t.table_name      ='APEX_TEAM_FEEDBACK_FOLLOWUP'    then 'APEX_TEAM_FEEDBACK'
         when substr(t.table_name,1,10)='APEX_TEAM_'                     then 'APEX_WORKSPACES'
         when substr(t.table_name,1,10)='APEX_REST_'                     then 'APEX_WORKSPACES'
         when        t.table_name      ='APEX_WS_APPLICATIONS'           then 'APEX_WORKSPACES'
         when        t.table_name      ='APEX_WS_APP_PAGES'              then 'APEX_WS_APPLICATIONS'
         when        t.table_name      ='APEX_WS_DATA_GRID'              then 'APEX_WS_APPLICATIONS'
         when        t.table_name      ='APEX_WS_DATA_GRID_COL'          then 'APEX_WS_DATA_GRID'
         when        t.table_name      ='APEX_APPL_LOAD_TABLE_LOOKUPS'   then 'APEX_APPL_LOAD_TABLES'
         when        t.table_name      ='APEX_APPL_LOAD_TABLE_RULES'     then 'APEX_APPL_LOAD_TABLES'
         when        t.table_name      ='APEX_APPL_CONCATENATED_FILES'   then 'APEX_APPL_USER_INTERFACES'
         when        t.table_name      ='APEX_APPL_AUTOMATION_ACTIONS'   then 'APEX_APPL_AUTOMATIONS'
         when        t.table_name      ='APEX_AUTOMATION_LOG'            then 'APEX_APPL_AUTOMATIONS'
         when        t.table_name      ='APEX_AUTOMATION_MSG_LOG'        then 'APEX_AUTOMATION_LOG'
         when        t.table_name      ='APEX_APPL_TASKDEF_ACTIONS'      then 'APEX_APPL_TASKDEFS'
         when        t.table_name      ='APEX_APPL_TASKDEF_PARTICIPANTS' then 'APEX_APPL_TASKDEF_PARTICIPANTS'
         when        t.table_name      ='APEX_APPL_TASKDEF_PARAMS'       then 'APEX_APPL_TASKDEFS'
         when        t.table_name      ='APEX_APPL_TASKDEF_COMP_PARAMS'  then 'APEX_APPL_TASKDEF_PARAMS'
         when        t.table_name      ='APEX_TASKS'                     then 'APEX_TASKS'
         when        t.table_name      ='APEX_TASK_COMMENTS'             then 'APEX_TASKS'
         when        t.table_name      ='APEX_TASK_HISTORY'              then 'APEX_TASKS'
         when        t.table_name      ='APEX_TASK_PARTICIPANTS'         then 'APEX_TASKS'
         when        t.table_name      ='APEX_TASK_PARAMETERS'           then 'APEX_TASKS'
       end         parent_view,
       case
         when        t.table_name      ='APEX_WORKSPACES'                then null
         when        t.table_name      ='APEX_APPLICATIONS'              then 'APEX_WORKSPACES/'
         when        t.table_name      ='APEX_REST_CATALOG_GROUPS'       then 'APEX_WORKSPACES/'
         when        t.table_name      ='APEX_REST_CATALOGS'             then 'APEX_WORKSPACES/APEX_REST_CATALOG_GROUPS/'
         when        t.table_name      ='APEX_REST_CATALOG_PLUGINS'      then 'APEX_WORKSPACES/APEX_REST_CATALOGS/'
         when        t.table_name      ='APEX_REST_CATALOG_SERVICES'     then 'APEX_WORKSPACES/APEX_REST_CATALOGS/'
         when        t.table_name      ='APEX_APPL_WEB_SRC_OPERATIONS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/'
         when        t.table_name      ='APEX_APPL_WEB_SRC_SYNC_STEPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/'
         when        t.table_name      ='APEX_APPL_WEB_SRC_PARAMETERS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/APEX_APPL_WEB_SRC_OPERATIONS'
         when        t.table_name      ='APEX_APPL_WEB_SRC_COMP_PARAMS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/APEX_APPL_WEB_SRC_OPERATIONS/APEX_APPL_WEB_SRC_PARAMETERS'
         when        t.table_name      ='APEX_APPLICATION_LOV_ENTRIES'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_LOVS/'
         when        t.table_name      ='APEX_APPLICATION_LOV_COLS'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_LOVS/'
         when        t.table_name      ='APEX_APPLICATION_PAGE_ITEMS'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      ='APEX_APPL_PAGE_FILTER_GROUPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      ='APEX_APPLICATION_PAGE_BUTTONS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      ='APEX_APPL_PAGE_FILTERS'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      ='APEX_APPL_PLUGIN_ATTRIBUTES'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.table_name      ='APEX_APPL_PLUGIN_STD_ATTRS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.table_name      ='APEX_APPL_PLUGIN_ATTR_VALUES'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/APEX_APPL_PLUGIN_ATTRIBUTES/'
         when        t.table_name      ='APEX_APPL_PLUGIN_FILES'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.table_name      ='APEX_APPL_PLUGIN_EVENTS'        then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.table_name      ='APEX_APPL_PLUGIN_ITEM_FILTERS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.table_name      ='APEX_APPL_ACL_USER_ROLES'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_ACL_USERS/'
         when        t.table_name      ='APEX_APPLICATION_LIST_ENTRIES'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_LISTS/'
         when        t.table_name      ='APEX_APPLICATION_RPT_QRY_STMTS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_RPT_QUERIES'
         when        t.table_name      ='APEX_APPLICATION_TEMPLATES'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.table_name      ='APEX_APPLICATION_THEME_STYLES'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.table_name      ='APEX_APPLICATION_THEME_FILES'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.table_name      ='APEX_APPL_TEMPLATE_OPTIONS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.table_name      ='APEX_APPL_TEMPLATE_OPT_GROUPS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.table_name      ='APEX_APPL_TEMP_PAGE_DP'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/APEX_APPLICATION_TEMP_PAGE/'
         when        t.table_name      ='APEX_APPL_TEMP_REGION_DP'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/APEX_APPLICATION_TEMP_REGION/'
         when substr(t.table_name,1,18)='APEX_WORKSPACE_LOG'             then 'APEX_WORKSPACES/APEX_WORKSPACE_ACTIVITY_LOG/'
         when substr(t.table_name,1,26)='APEX_APPLICATION_SUPP_OBJ_'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_SUPP_OBJECTS/'
         when substr(t.table_name,1,21)='APEX_APPLICATION_TEMP'          then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when substr(t.table_name,1,30)='APEX_APPLICATION_PAGE_RPT_COLS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_RPT/'
         when substr(t.table_name,1,25)='APEX_APPLICATION_PAGE_RPT'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      ='APEX_APPLICATION_PAGE_REG_COLS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when substr(t.table_name,1,30)='APEX_APPLICATION_PAGE_CHART_S'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_CHARTS/'
         when substr(t.table_name,1,30)='APEX_APPLICATION_PAGE_CHART_A'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_CHARTS/'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_COMP'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_COND'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_SUB'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_GRPBY' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_PIVOT' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_PVAGG' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR_PVSRT' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when substr(t.table_name,1,25)='APEX_APPLICATION_PAGE_IR_'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/'
         when        t.table_name      ='APEX_APPLICATION_PAGE_IR'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_COLUMNS'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_COL_GROUPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS'
         when        t.table_name      = 'APEX_APPL_PAGE_IGS'             then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPTS'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_VIEWS'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_COMPUTES' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_FILTERS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_COLUMNS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_HIGHLTS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_AGGS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_IG_RPT_CHT_COLS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.table_name      = 'APEX_APPL_PAGE_CARDS'           then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      = 'APEX_APPL_PAGE_CARD_ACTIONS'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_CARDS'
         when        t.table_name      = 'APEX_APPL_PAGE_MAPS'            then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      = 'APEX_APPL_PAGE_MAP_LAYERS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_MAPS'
         when        t.table_name      ='APEX_APPLICATION_PAGE_CHARTS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when substr(t.table_name,1,27)='APEX_APPLICATION_PAGE_TREES'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.table_name      ='APEX_APPLICATION_PAGE_DA_ACTS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_DA/'
         when substr(t.table_name,1,22)='APEX_APPLICATION_PAGE_'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/'
         when substr(t.table_name,1,15)='APEX_APPL_PAGE_'                then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/'    --#14833918
         when substr(t.table_name,1,15)='APEX_WORKSPACE_'                then 'APEX_WORKSPACES/'
         when substr(t.table_name,1,17)='APEX_APPLICATION_'              then 'APEX_WORKSPACES/APEX_APPLICATIONS/'
         when        t.table_name      ='APEX_TEAM_FEEDBACK_FOLLOWUP'    then 'APEX_WORKSPACES/APEX_TEAM_FEEDBACK/'
         when substr(t.table_name,1,10)='APEX_TEAM_'                     then 'APEX_WORKSPACES/'
         when substr(t.table_name,1,10)='APEX_REST_'                     then 'APEX_WORKSPACES/'
         when        t.table_name      ='APEX_WS_APPLICATIONS'           then 'APEX_WORKSPACES/'
         when        t.table_name      ='APEX_WS_APP_PAGES'              then 'APEX_WORKSPACES/APEX_WS_APPLICATIONS/'
         when        t.table_name      ='APEX_WS_DATA_GRID'              then 'APEX_WORKSPACES/APEX_WS_APPLICATIONS/'
         when        t.table_name      ='APEX_WS_DATA_GRID_COL'          then 'APEX_WORKSPACES/APEX_WS_APPLICATIONS/APEX_WS_DATA_GRID/'
         when        t.table_name      ='APEX_APPL_LOAD_TABLE_LOOKUPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_LOAD_TABLES/'
         when        t.table_name      ='APEX_APPL_LOAD_TABLE_RULES'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_LOAD_TABLES/'
         when        t.table_name      ='APEX_APPL_CONCATENATED_FILES'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_USER_INTERFACES/'
         when        t.table_name      ='APEX_APPL_AUTOMATION_ACTIONS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_AUTOMATIONS/'
         when        t.table_name      ='APEX_AUTOMATION_LOG'            then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_AUTOMATIONS/'
         when        t.table_name      ='APEX_AUTOMATION_MSG_LOG'        then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_AUTOMATIONS/APEX_AUTOMATION_LOG/'
         when        t.table_name      ='APEX_APPL_TASKDEF_ACTIONS'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/'
         when        t.table_name      ='APEX_APPL_TASKDEF_PARAMS'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/'
         when        t.table_name      ='APEX_APPL_TASKDEF_PARTICIPANTS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/'
         when        t.table_name      ='APEX_APPL_TASKDEF_COMP_PARAMS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_APPL_TASKDEF_PARAMS/'
         when substr(t.table_name,1,10)='APEX_APPL_'                     then 'APEX_WORKSPACES/APEX_APPLICATIONS/'
         when        t.table_name      ='APEX_TASKS'                     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/'
         when        t.table_name      ='APEX_TASK_COMMENTS'             then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         when        t.table_name      ='APEX_TASK_HISTORY'              then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         when        t.table_name      ='APEX_TASK_PARTICIPANTS'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         when        t.table_name      ='APEX_TASK_PARAMETERS'           then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         end ||t.table_name        path
from  sys.all_tab_columns t, sys.all_col_comments c
where t.owner = '^APPUN.' and
      t.owner = c.owner and
      t.table_name = c.table_name and
      t.column_Name = c.column_name and
      (t.table_name like 'APEX\_APPL\_%' escape '\'
      or t.table_name like 'APEX_APPLICATION%'
      or t.table_name like 'APEX_TEAM%'
      or t.table_name like 'APEX_REST%'
      or t.table_name like 'APEX_WORKSPACE%'
      or t.table_name like 'APEX_UI_DEFAULTS%'
      or t.table_name like 'APEX_WS%'
      or t.table_name like 'APEX_ISSUE%'
      or t.table_name like 'APEX_REST_CATALOG%'
      or t.table_name in ('APEX_THEMES', 'APEX_WEBSERVICE_LOG', 'APEX_REST_SOURCE_SYNC_LOG', 'APEX_AUTOMATION_LOG', 'APEX_AUTOMATION_MSG_LOG', 'APEX_DEBUG_MESSAGES',
      'APEX_TASKS','APEX_TASK_COMMENTS','APEX_TASK_HISTORY','APEX_TASK_PARTICIPANTS','APEX_TASK_PARAMETERS') ) and
      t.table_name != 'apex_dictionary'
union all
select t.view_name   apex_view_name,
       0             column_id,
       null          column_name,
       c.comments    apex_view_comments,
       'View'        comment_type,
       case
         when        t.view_name      ='APEX_APPLICATIONS'              then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_REST_CATALOG_GROUPS'       then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_REST_CATALOGS'             then 'APEX_REST_CATALOG_GROUPS'
         when        t.view_name      ='APEX_REST_CATALOG_PLUGINS'      then 'APEX_REST_CATALOGS'
         when        t.view_name      ='APEX_REST_CATALOG_SERVICES'     then 'APEX_REST_CATALOGS'
         when        t.view_name      ='APEX_APPL_WEB_SRC_OPERATIONS'   then 'APEX_APPL_WEB_SRC_MODULES'
         when        t.view_name      ='APEX_APPL_WEB_SRC_SYNC_STEPS'   then 'APEX_APPL_WEB_SRC_MODULES'
         when        t.view_name      ='APEX_APPL_WEB_SRC_PARAMETERS'   then 'APEX_APPL_WEB_SRC_OPERATIONS'
         when        t.view_name      ='APEX_APPL_WEB_SRC_COMP_PARAMS'  then 'APEX_APPL_WEB_SRC_PARAMETERS'
         when        t.view_name      ='APEX_APPLICATION_LOV_ENTRIES'   then 'APEX_APPLICATION_LOVS'
         when        t.view_name      ='APEX_APPLICATION_LOV_COLS'      then 'APEX_APPLICATION_LOVS'
         when        t.view_name      ='APEX_APPLICATION_PAGE_ITEMS'    then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      ='APEX_APPL_PAGE_FILTER_GROUPS'   then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      ='APEX_APPLICATION_PAGE_BUTTONS'  then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      ='APEX_APPL_PAGE_FILTERS'         then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      ='APEX_APPL_PLUGIN_ATTRIBUTES'    then 'APEX_APPL_PLUGINS'
         when        t.view_name      ='APEX_APPL_PLUGIN_STD_ATTRS'     then 'APEX_APPL_PLUGINS'
         when        t.view_name      ='APEX_APPL_PLUGIN_ATTR_VALUES'   then 'APEX_APPL_PLUGIN_ATTRIBUTES'
         when        t.view_name      ='APEX_APPL_PLUGIN_FILES'         then 'APEX_APPL_PLUGINS'
         when        t.view_name      ='APEX_APPL_PLUGIN_EVENTS'        then 'APEX_APPL_PLUGINS'
         when        t.view_name      ='APEX_APPL_PLUGIN_ITEM_FILTERS'  then 'APEX_APPL_PLUGINS'
         when        t.view_name      ='APEX_APPL_ACL_USER_ROLES'       then 'APEX_APPL_ACL_USERS'
         when        t.view_name      ='APEX_APPLICATION_LIST_ENTRIES'  then 'APEX_APPLICATION_LISTS'
         when        t.view_name      ='APEX_APPLICATION_RPT_QRY_STMTS' then 'APEX_APPLICATION_RPT_QUERIES'
         when        t.view_name      ='APEX_APPLICATION_TEMPLATES'     then 'APEX_APPLICATION_THEMES'
         when        t.view_name      ='APEX_APPLICATION_THEME_STYLES'  then 'APEX_APPLICATION_THEMES'
         when        t.view_name      ='APEX_APPLICATION_THEME_FILES'   then 'APEX_APPLICATION_THEMES'
         when        t.view_name      ='APEX_APPL_TEMPLATE_OPTIONS'     then 'APEX_APPLICATION_THEMES'
         when        t.view_name      ='APEX_APPL_TEMPLATE_OPT_GROUPS'  then 'APEX_APPLICATION_THEMES'
         when        t.view_name      ='APEX_APPL_TEMP_PAGE_DP'         then 'APEX_APPLICATION_TEMP_PAGE'
         when        t.view_name      ='APEX_APPL_TEMP_REGION_DP'       then 'APEX_APPLICATION_TEMP_REGION'
         when substr(t.view_name,1,26)='APEX_APPLICATION_SUPP_OBJ_'     then 'APEX_APPLICATION_SUPP_OBJECTS'
         when substr(t.view_name,1,18)='APEX_WORKSPACE_LOG'             then 'APEX_WORKSPACE_ACTIVITY_LOG'
         when substr(t.view_name,1,21)='APEX_APPLICATION_TEMP'          then 'APEX_APPLICATION_THEMES'
         when substr(t.view_name,1,30)='APEX_APPLICATION_PAGE_RPT_COLS' then 'APEX_APPLICATION_PAGE_RPT'
         when substr(t.view_name,1,25)='APEX_APPLICATION_PAGE_RPT'      then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      ='APEX_APPLICATION_PAGE_REG_COLS' then 'APEX_APPLICATION_PAGE_REGIONS'
         when substr(t.view_name,1,30)='APEX_APPLICATION_PAGE_CHART_S'  then 'APEX_APPLICATION_PAGE_CHARTS'
         when substr(t.view_name,1,30)='APEX_APPLICATION_PAGE_CHART_A'  then 'APEX_APPLICATION_PAGE_CHARTS'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_COMP'  then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_COND'  then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_SUB'   then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_GRPBY' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_PIVOT' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_PVAGG' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_PVSRT' then 'APEX_APPLICATION_PAGE_IR_RPT'
         when substr(t.view_name,1,25)='APEX_APPLICATION_PAGE_IR_'      then 'APEX_APPLICATION_PAGE_IR'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR'       then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_COLUMNS'      then 'APEX_APPL_PAGE_IGS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_COL_GROUPS'   then 'APEX_APPL_PAGE_IGS'
         when        t.view_name      = 'APEX_APPL_PAGE_IGS'             then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPTS'         then 'APEX_APPL_PAGE_IGS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_VIEWS'    then 'APEX_APPL_PAGE_IG_RPTS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_COMPUTES' then 'APEX_APPL_PAGE_IG_RPTS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_FILTERS'  then 'APEX_APPL_PAGE_IG_RPTS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_COLUMNS'  then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_HIGHLTS'  then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_AGGS'     then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_CHT_COLS' then 'APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_CARDS'           then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      = 'APEX_APPL_PAGE_CARD_ACTIONS'    then 'APEX_APPL_PAGE_CARDS'
         when        t.view_name      = 'APEX_APPL_PAGE_MAPS'            then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      = 'APEX_APPL_PAGE_MAP_LAYERS'     then 'APEX_APPL_PAGE_MAPS'
         when        t.view_name      ='APEX_APPLICATION_PAGE_CHARTS'   then 'APEX_APPLICATION_PAGE_REGIONS'
         when substr(t.view_name,1,27)='APEX_APPLICATION_PAGE_TREES'    then 'APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      ='APEX_APPLICATION_PAGE_DA_ACTS'  then 'APEX_APPLICATION_PAGE_DA'
         when substr(t.view_name,1,22)='APEX_APPLICATION_PAGE_'         then 'APEX_APPLICATION_PAGES'
         when substr(t.view_name,1,15)='APEX_WORKSPACE_'                then 'APEX_WORKSPACES'
         when substr(t.view_name,1,15)='APEX_APPL_PAGE_'                then 'APEX_APPLICATION_PAGES'       --#14833918
         when substr(t.view_name,1,17)='APEX_APPLICATION_' or
              substr(t.view_name,1,10)='APEX_APPL_'                     then 'APEX_APPLICATIONS'
         when        t.view_name      ='APEX_TEAM_FEEDBACK_FOLLOWUP'    then 'APEX_TEAM_FEEDBACK'
         when substr(t.view_name,1,10)='APEX_TEAM_'                     then 'APEX_WORKSPACES'
         when substr(t.view_name,1,10)='APEX_REST_'                     then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_WS_APPLICATIONS'           then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_WS_APP_PAGES'              then 'APEX_WS_APPLICATIONS'
         when        t.view_name      ='APEX_WS_DATA_GRID'              then 'APEX_WS_APPLICATIONS'
         when        t.view_name      ='APEX_WS_DATA_GRID_COL'          then 'APEX_WS_DATA_GRID'
         when        t.view_name      ='APEX_APPL_LOAD_TABLE_LOOKUPS'   then 'APEX_APPL_LOAD_TABLES'
         when        t.view_name      ='APEX_APPL_LOAD_TABLE_RULES'     then 'APEX_APPL_LOAD_TABLES'
         when        t.view_name      ='APEX_APPL_CONCATENATED_FILES'   then 'APEX_APPL_USER_INTERFACES'
         when        t.view_name      ='APEX_APPL_AUTOMATION_ACTIONS'   then 'APEX_APPL_AUTOMATIONS'
         when        t.view_name      ='APEX_AUTOMATION_LOG'            then 'APEX_APPL_AUTOMATIONS'
         when        t.view_name      ='APEX_AUTOMATION_MSG_LOG'        then 'APEX_AUTOMATION_LOG'
         when        t.view_name      ='APEX_ISSUES'                    then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_ISSUE_EVENTS'              then 'APEX_ISSUES'
         when        t.view_name      ='APEX_ISSUE_COMMENTS'            then 'APEX_ISSUES'
         when        t.view_name      ='APEX_ISSUE_ATTACHMENTS'         then 'APEX_ISSUES'
         when        t.view_name      ='APEX_ISSUE_STAKEHOLDERS'        then 'APEX_ISSUES'
         when        t.view_name      ='APEX_ISSUE_MILESTONES'          then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_ISSUE_LABEL_GROUPS'        then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_ISSUE_LABELS'              then 'APEX_ISSUE_LABEL_GROUPS'
         when        t.view_name      ='APEX_ISSUE_LABEL_RELATION'      then 'APEX_ISSUES'
         when        t.view_name      ='APEX_ISSUE_MILESTONE_RELATION'  then 'APEX_ISSUES'
         when        t.view_name      ='APEX_ISSUE_EMAIL_PREFS'         then 'APEX_WORKSPACES'
         when        t.view_name      ='APEX_ISSUE_NOTIFICATIONS'       then 'APEX_ISSUES'
         when        t.view_name      ='APEX_APPL_TASKDEF_ACTIONS'      then 'APEX_APPL_TASKDEFS'
         when        t.view_name      ='APEX_APPL_TASKDEF_PARTICIPANTS' then 'APEX_APPL_TASKDEFS'
         when        t.view_name      ='APEX_APPL_TASKDEF_PARAMS'      then 'APEX_APPL_TASKDEFS'
         when        t.view_name      ='APEX_APPL_TASKDEF_COMP_PARAMS' then 'APEX_APPL_TASKDEF_PARAMS'
         when        t.view_name      ='APEX_TASKS'                     then 'APEX_APPL_TASKDEFS'
         when        t.view_name      ='APEX_TASK_COMMENTS'             then 'APEX_TASKS'
         when        t.view_name      ='APEX_TASK_HISTORY'              then 'APEX_TASKS'
         when        t.view_name      ='APEX_TASK_PARTICIPANTS'         then 'APEX_TASKS'
         when        t.view_name      ='APEX_TASK_PARAMETERS'           then 'APEX_TASKS'
         end         parent_view,
       case
         when        t.view_name      ='APEX_WORKSPACES'                then null
         when        t.view_name      ='APEX_APPLICATIONS'              then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_REST_CATALOG_GROUPS'       then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_REST_CATALOGS'             then 'APEX_WORKSPACES/APEX_REST_CATALOG_GROUPS/'
         when        t.view_name      ='APEX_REST_CATALOG_PLUGINS'      then 'APEX_WORKSPACES/APEX_REST_CATALOGS/'
         when        t.view_name      ='APEX_REST_CATALOG_SERVICES'     then 'APEX_WORKSPACES/APEX_REST_CATALOGS/'
         when        t.view_name      ='APEX_APPL_WEB_SRC_OPERATIONS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/'
         when        t.view_name      ='APEX_APPL_WEB_SRC_SYNC_STEPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/'
         when        t.view_name      ='APEX_APPL_WEB_SRC_PARAMETERS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/APEX_APPL_WEB_SRC_OPERATIONS'
         when        t.view_name      ='APEX_APPL_WEB_SRC_COMP_PARAMS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_WEB_SRC_MODULES/APEX_APPL_WEB_SRC_OPERATIONS/APEX_APPL_WEB_SRC_PARAMETERS'
         when        t.view_name      ='APEX_APPLICATION_LOV_ENTRIES'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_LOVS/'
         when        t.view_name      ='APEX_APPLICATION_LOV_COLS'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_LOVS/'
         when        t.view_name      ='APEX_APPLICATION_PAGE_ITEMS'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      ='APEX_APPL_PAGE_FILTER_GROUPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      ='APEX_APPLICATION_PAGE_BUTTONS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      ='APEX_APPL_PAGE_FILTERS'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      ='APEX_APPL_PLUGIN_ATTRIBUTES'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.view_name      ='APEX_APPL_PLUGIN_STD_ATTRS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.view_name      ='APEX_APPL_PLUGIN_ATTR_VALUES'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/APEX_APPL_PLUGIN_ATTRIBUTES/'
         when        t.view_name      ='APEX_APPL_PLUGIN_FILES'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.view_name      ='APEX_APPL_PLUGIN_EVENTS'        then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.view_name      ='APEX_APPL_PLUGIN_ITEM_FILTERS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_PLUGINS/'
         when        t.view_name      ='APEX_APPL_ACL_USER_ROLES'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_ACL_USERS/'
         when        t.view_name      ='APEX_APPLICATION_LIST_ENTRIES'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_LISTS/'
         when        t.view_name      ='APEX_APPLICATION_RPT_QRY_STMTS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_RPT_QUERIES/'
         when        t.view_name      ='APEX_APPLICATION_TEMPLATES'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.view_name      ='APEX_APPLICATION_THEME_STYLES'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.view_name      ='APEX_APPLICATION_THEME_FILESS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.view_name      ='APEX_APPL_TEMPLATE_OPTIONS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.view_name      ='APEX_APPL_TEMPLATE_OPT_GROUPS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when        t.view_name      ='APEX_APPL_TEMP_PAGE_DP'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/APEX_APPLICATION_TEMP_PAGE/'
         when        t.view_name      ='APEX_APPL_TEMP_REGION_DP'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/APEX_APPLICATION_TEMP_REGION/'
         when substr(t.view_name,1,18)='APEX_WORKSPACE_LOG'             then 'APEX_WORKSPACES/APEX_WORKSPACE_ACTIVITY_LOG/'
         when substr(t.view_name,1,26)='APEX_APPLICATION_SUPP_OBJ_'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_SUPP_OBJECTS/'
         when substr(t.view_name,1,21)='APEX_APPLICATION_TEMP'          then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_THEMES/'
         when substr(t.view_name,1,30)='APEX_APPLICATION_PAGE_RPT_COLS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_RPT/'
         when substr(t.view_name,1,25)='APEX_APPLICATION_PAGE_RPT'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      ='APEX_APPLICATION_PAGE_REG_COLS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when substr(t.view_name,1,30)='APEX_APPLICATION_PAGE_CHART_S'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_CHARTS/'
         when substr(t.view_name,1,30)='APEX_APPLICATION_PAGE_CHART_A'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_CHARTS/'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_COMP'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_COND'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_SUB'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_GRPBY' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_PIVOT' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_PVAGG' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR_PVSRT' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/APEX_APPLICATION_PAGE_IR_RPT'
         when substr(t.view_name,1,25)='APEX_APPLICATION_PAGE_IR_'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPLICATION_PAGE_IR/'
         when        t.view_name      ='APEX_APPLICATION_PAGE_IR'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_COLUMNS'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_COL_GROUPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS'
         when        t.view_name      = 'APEX_APPL_PAGE_IGS'             then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPTS'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_VIEWS'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_COMPUTES' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_FILTERS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_COLUMNS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_HIGHLTS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_AGGS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_IG_RPT_CHT_COLS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_IGS/APEX_APPL_PAGE_IG_RPTS/APEX_APPL_PAGE_IG_RPT_VIEWS'
         when        t.view_name      = 'APEX_APPL_PAGE_CARDS'           then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      = 'APEX_APPL_PAGE_CARD_ACTIONS'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_CARDS'
         when        t.view_name      = 'APEX_APPL_PAGE_MAPS'            then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS'
         when        t.view_name      = 'APEX_APPL_PAGE_MAP_LAYERS'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/APEX_APPL_PAGE_MAPS'
         when        t.view_name      ='APEX_APPLICATION_PAGE_CHARTS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when substr(t.view_name,1,27)='APEX_APPLICATION_PAGE_TREES'    then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_REGIONS/'
         when        t.view_name      ='APEX_APPLICATION_PAGE_DA_ACTS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/APEX_APPLICATION_PAGE_DA/'
         when substr(t.view_name,1,22)='APEX_APPLICATION_PAGE_'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/'
         when substr(t.view_name,1,15)='APEX_WORKSPACE_'                then 'APEX_WORKSPACES/'
         when substr(t.view_name,1,17)='APEX_APPLICATION_'              then 'APEX_WORKSPACES/APEX_APPLICATIONS/'
         when substr(t.view_name,1,15)='APEX_APPL_PAGE_'                then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPLICATION_PAGES/'     --#14833918
         when substr(t.view_name,1,10)='APEX_APPL_'                     then 'APEX_WORKSPACES/APEX_APPLICATIONS/'
         when        t.view_name      ='APEX_TEAM_FEEDBACK_FOLLOWUP'    then 'APEX_WORKSPACES/APEX_TEAM_FEEDBACK/'
         when substr(t.view_name,1,10)='APEX_TEAM_'                     then 'APEX_WORKSPACES/'
         when substr(t.view_name,1,10)='APEX_REST_'                     then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_WS_APPLICATIONS'           then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_WS_APP_PAGES'              then 'APEX_WORKSPACES/APEX_WS_APPLICATIONS/'
         when        t.view_name      ='APEX_WS_DATA_GRID'              then 'APEX_WORKSPACES/APEX_WS_APPLICATIONS/'
         when        t.view_name      ='APEX_WS_DATA_GRID_COL'          then 'APEX_WORKSPACES/APEX_WS_APPLICATIONS/APEX_WS_DATA_GRID/'
         when        t.view_name      ='APEX_APPL_LOAD_TABLE_LOOKUPS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_LOAD_TABLES/'
         when        t.view_name      ='APEX_APPL_LOAD_TABLE_RULES'     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_LOAD_TABLES/'
         when        t.view_name      ='APEX_APPL_CONCATENATED_FILES'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_USER_INTERFACES/'
         when        t.view_name      ='APEX_APPL_AUTOMATION_ACTIONS'   then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_AUTOMATIONS/'
         when        t.view_name      ='APEX_AUTOMATION_LOG'            then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_AUTOMATIONS/'
         when        t.view_name      ='APEX_AUTOMATION_MSG_LOG'        then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_AUTOMATIONS/APEX_AUTOMATION_LOG/'
         when substr(t.view_name,1,10)='APEX_APPL_'                     then 'APEX_WORKSPACES/APEX_APPLICATIONS/'
         when        t.view_name      ='APEX_ISSUES'                    then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_ISSUE_EVENTS'              then 'APEX_WORKSPACES/APEX_ISSUES/'
         when        t.view_name      ='APEX_ISSUE_COMMENTS'            then 'APEX_WORKSPACES/APEX_ISSUES/'
         when        t.view_name      ='APEX_ISSUE_ATTACHMENTS'         then 'APEX_WORKSPACES/APEX_ISSUES/'
         when        t.view_name      ='APEX_ISSUE_STAKEHOLDERS'        then 'APEX_WORKSPACES/APEX_ISSUES/'
         when        t.view_name      ='APEX_ISSUE_MILESTONES'          then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_ISSUE_LABEL_GROUPS'        then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_ISSUE_LABELS'              then 'APEX_WORKSPACES/APEX_ISSUE_LABEL_GROUPS/'
         when        t.view_name      ='APEX_ISSUE_LABEL_RELATION'      then 'APEX_WORKSPACES/APEX_ISSUES/'
         when        t.view_name      ='APEX_ISSUE_MILESTONE_RELATION'  then 'APEX_WORKSPACES/APEX_ISSUES/'
         when        t.view_name      ='APEX_ISSUE_EMAIL_PREFS'         then 'APEX_WORKSPACES/'
         when        t.view_name      ='APEX_ISSUE_NOTIFICATIONS'       then 'APEX_WORKSPACES/APEX_ISSUES/'
         when        t.view_name      ='APEX_APPL_TASKDEF_ACTIONS'      then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS'
         when        t.view_name      ='APEX_APPL_TASKDEF_PARAMS'       then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS'
         when        t.view_name      ='APEX_APPL_TASKDEF_PARTICIPANTS' then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS'
         when        t.view_name      ='APEX_APPL_TASKDEF_COMP_PARAMS'  then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_APPL_TASKDEF_PARAMS/'
         when        t.view_name      ='APEX_TASKS'                     then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/'
         when        t.view_name      ='APEX_TASK_COMMENTS'             then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         when        t.view_name      ='APEX_TASK_HISTORY'              then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         when        t.view_name      ='APEX_TASK_PARTICIPANTS'         then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         when        t.view_name      ='APEX_TASK_PARAMETERS'           then 'APEX_WORKSPACES/APEX_APPLICATIONS/APEX_APPL_TASKDEFS/APEX_TASKS'
         end ||t.view_name        path
from  sys.all_views t, sys.all_tab_comments c
where t.owner = '^APPUN.' and
      t.owner = c.owner and
      t.view_name = c.table_name and
      (t.view_name like 'APEX\_APPL\_%' escape '\'
      or t.view_name like 'APEX_APPLICATION%'
      or t.view_name like 'APEX_WORKSPACE%'
      or t.view_name like 'APEX_TEAM%'
      or t.view_name like 'APEX_REST%'
      or t.view_name like 'APEX_UI_DEFAULTS%'
      or t.view_name like 'APEX_WS%'
      or t.view_name like 'APEX_ISSUE%'
      or t.view_name like 'APEX_REST_CATALOG%'
      or t.view_name in ('APEX_THEMES', 'APEX_WEBSERVICE_LOG', 'APEX_REST_SOURCE_SYNC_LOG', 'APEX_AUTOMATION_LOG', 'APEX_AUTOMATION_MSG_LOG', 'APEX_DEBUG_MESSAGES', 
      'APEX_TASKS', 'APEX_TASK_COMMENTS','APEX_TASK_HISTORY', 'APEX_TASK_PARTICIPANTS', 'APEX_TASK_PARAMETERS') )
order by 7,2
) 
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_dictionary',
    p_table_comment     => 'Identifies the comment associated with each APEX view and view column',
    p_column_comments   => wwv_flow_t_varchar2 (
      'APEX_VIEW_NAME' , 'Name of the APEX dictionary view',
      'COLUMN_ID'      , 'Display order of the APEX dictionary column within the dictionary view',
      'COLUMN_NAME'    , 'Identifies the column name for column dictionary comments, null for view dictionary comments',
      'COMMENTS'       , 'Description of the APEX view name or column name',
      'COMMENT_TYPE'   , 'Type of dictionary comment, View or Column',
      'parent_view'    , 'Identifies the component parent'
    )
  );
end;
/

----------------------------------------------
-- A C T I V I T Y    V I E W S
--
prompt ...activity views

create or replace view apex_workspace_access_log as
select
    w.workspace,
    w.workspace_display_name,
    l.APPLICATION            application_id,
    f.name                   application_name,
    upper(l.LOGIN_NAME)      user_name,
    l.AUTHENTICATION_METHOD  authentication_method,
    l.OWNER                  application_schema_owner,
    l.ACCESS_DATE,
    86400 * (sysdate - l.access_date) seconds_ago,
    l.IP_ADDRESS,
    --l.REMOTE_USER,
    decode(nvl(l.authentication_result,0),
        0,'AUTH_SUCCESS',
        1,'AUTH_UNKNOWN_USER',
        2,'AUTH_ACCOUNT_LOCKED',
        3,'AUTH_ACCOUNT_EXPIRED',
        4,'AUTH_PASSWORD_INCORRECT',
        5,'AUTH_PASSWORD_FIRST_USE',
        6,'AUTH_ATTEMPTS_EXCEEDED',
        7,'AUTH_INTERNAL_ERROR',
        authentication_result) authentication_result,
    l.CUSTOM_STATUS_TEXT       custom_status_text,
    l.SECURITY_GROUP_ID        workspace_id
from wwv_flow_user_access_log_v  l,
     wwv_flows                   f,
     wwv_flow_current_workspaces w
where l.security_group_id = w.workspace_id
  and l.application       = f.id(+)
  and l.security_group_id = f.security_group_id(+)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_access_log',
    p_table_comment     => 'One row is logged for each login attempt.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                , 'A work area mapped to one or more database schemas',
      'workspace_display_name'   , 'Display name for the workspace',
      'application_id'           , 'Application Primary Key, Unique over all workspaces',
      'application_name'         , 'Name of Application',
      'user_name'                , 'Identifies the Oracle APEX username',
      'authentication_method'    , 'Identifies the authentication method',
      'application_schema_owner' , 'Identifies the parse as schema owner for the application',
      'custom_status_text'       , 'Identifies the status of the authentication attempt',
      'access_date'              , 'Date of login attempt',
      'ip_address'               , 'IP Address of user',
      'AUTHENTICATION_RESULT'    , 'Status of authentication attempt',
      'workspace_id'             , 'Workspace numeric identifier',
      'seconds_ago'              , 'Elapsed seconds since event occured'
    )
  );
end;
/

create or replace view apex_workspace_activity_log as
    select
           w.workspace,
           w.workspace_display_name,
           l.userid                  apex_user,
           l.flow_id                 application_id,
           f.name                    application_name,
           f.owner                   application_schema_owner,
           l.step_id                 page_id,
           (select st.name
             from wwv_flow_steps st
            where st.id                = l.step_id and
                  st.flow_id           = f.id
              and st.security_group_id = l.security_group_id)    page_name,
           l.time_stamp              view_date,
           to_number(null)           think_time,
           round(86400 * (sysdate - l.time_stamp))     seconds_ago,
           --l.component_type          component_type,
           --l.component_name          component_name,
           (select upper(st.name)
            from wwv_flow_steps st
            where st.id                = l.step_id and
                  st.flow_id           = f.id
              and st.security_group_id = l.security_group_id)    log_context,
           --l.information             page_view_information,
           l.elap                    elapsed_time,
           l.num_rows                rows_queried,
           l.ip_address              ip_address,
           l.USER_AGENT              agent,
           l.session_id              apex_session_id,
           l.tenant_id               application_tenant_id,
           l.sqlerrm                 error_message,
           l.sqlerrm_component_type  error_on_component_type,
           l.sqlerrm_component_name  error_on_component_name,
           decode(l.page_mode,
             'D','Dynamic',
             'C','Cache Created',
             'R','Cached',
             'P','Partial Page',
             'A','Page Processing',
             l.page_mode)            page_view_mode,
           null application_info,
           -- ir log
           l.worksheet_id            interactive_report_id,
           l.ir_report_id            ir_saved_report_id,
           l.ir_search               ir_search,
           -- Obsolete websheet log
           null                      ws_application_id,
           null                      ws_page_id,
           null                      ws_datagrid_id,
           --
           l.content_length,
           --
           nvl(l.cached_regions,0)   regions_from_cache,
           l.security_group_id       workspace_id,
           --
           case l.page_view_type
           when 0 then 'Other'
           when 1 then 'Processing'
           when 2 then 'Rendering'
           when 3 then 'Ajax'
           when 4 then 'Logout'
           when 5 then 'Authentication Callback'
           end                       page_view_type,
           l.request_value,
           l.debug_page_view_id,
           l.time_stamp_tz           view_timestamp,
           l.ecid
from wwv_flow_activity_log       l,
     wwv_flow_current_workspaces w,
     wwv_flows                   f
where l.security_group_id = w.workspace_id
  and l.security_group_id = f.security_group_id(+)
  and l.flow_id           = f.id(+)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_activity_log',
    p_table_comment     => 'Page view activity log detail.  One row is logged for each page view for application with logging enabled.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                , 'A work area mapped to one or more database schemas',
      'workspace_display_name'   , 'Display name for the workspace',
      'application_id'           , 'Application Primary Key, Unique over all workspaces',
      'application_name'         , 'Identifies the application',
      'page_id'                  , 'ID of the application page',
      'page_name'                , 'Name of the application page',
      'view_date'                , 'Date of page view with precision to the second (indexed)',
      'log_context'              , 'Context of Page View',
      'ELAPSED_TIME'             , 'Elapsed time to generate page source',
      'ROWS_QUERIED'             , 'Number of rows fetched by the APEX reporting engine',
      'IP_ADDRESS'               , 'IP Address for this page view',
      'AGENT'                    , 'HTTP User Agent for this page view',
      'APEX_SESSION_ID'          , 'APEX Session ID for this page view',
      'application_tenant_id'    , 'Application Tenant ID for this page view',
      'ERROR_MESSAGE'            , 'Error message raised for this page view',
      'ERROR_ON_COMPONENT_TYPE'  , 'The component type that caused an error to be raised',
      'ERROR_ON_COMPONENT_NAME'  , 'The component name which caused the error to be raised',
      'PAGE_VIEW_MODE'           , 'The page view mode, typically Static or Dynamic',
      'apex_user'                , 'Name of the end user of the application',
      'application_schema_owner' , 'Parsing Schema of the Application',
      'regions_from_cache'       , 'Count of regions on this page that are rendered from cache',
      'think_time'               , 'Always NULL - see APEX 18.1 Release Notes, Changed Behavior',
      'application_info'         , 'Information provided by the application to provide additional application context',
      'interactive_report_id'    , 'Identifies the Interactive Report ID foreign key to the APEX_APPLICATION_PAGE_IR view',
      'ir_saved_report_id'       , 'Identifies the Saved Interactive Report ID foreign key to the APEX_APPLICATION_PAGE_IR_RPT view',
      'ir_search'                , 'The search text entered from Interactive Report search bar',
      'ws_application_id'        , 'Obsolete',
      'ws_page_id'               , 'Obsolete',
      'ws_datagrid_id'           , 'Obsolete',
      'workspace_id'             , 'Primary Key of the Workspace',
      'content_length'           , 'The size of the web page served.',
      'page_view_type'           , 'The type of page view (Rendering, Processing, Ajax, Logout, Authentication callback, Other)',
      'request_value'            , 'The value of the request parameter for this page view',
      'debug_page_view_id'       , 'The ID for debug messages stored for this page view',
      'view_timestamp'           , 'The timestamp with time zone of the page view (not indexed)',
      'ecid'                     , 'The web server''s Execution Context ID of the page view',
      'seconds_ago'              , 'Number of seconds since this activity was recorded based upon the current system date'
    )
  );
end;
/

create or replace view apex_workspace_log_summary as
    select
           w.workspace,
           w.workspace_display_name,
           l.flow_id                              application_id,
           f.name                                 application_name,
           count(*)                               page_views,
           count(distinct l.step_id)              distinct_pages,
           sum(l.elap)                            total_elapsed_time,
           avg(l.elap)                            average_elapsed_time,
           min(l.elap)                            minimum_elapsed_time,
           max(l.elap)                            maximum_elapsed_time,
           sum(l.num_rows)                        total_rows_queried,
           count(distinct l.ip_address)           ip_addresses,
           count(distinct l.USER_AGENT)           agents,
           count(distinct l.session_id)           apex_sessions,
           sum(decode(l.sqlerrm,null,0,1))        page_views_with_errors,
           sum(decode(l.page_mode,'D',1,'C',1,0)) dynamic_page_views,
           sum(decode(l.page_mode,'R',1,0))       cached_page_views,
           min(l.time_stamp)                      first_view,
           max(l.time_stamp)                      last_view,
           max(l.time_stamp)-min(l.time_stamp)    period_in_days
from wwv_flow_activity_log       l,
     wwv_flows                   f,
     wwv_flow_current_workspaces w
where l.security_group_id = w.workspace_id
  and l.security_group_id = f.security_group_id(+)
  and l.flow_id           = f.id(+)
  and l.time_stamp        > sysdate - 14
group by w.workspace_name, w.workspace_display_name, l.flow_id, f.name
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_log_summary',
    p_table_comment     => 'Page view activity log summarized by application for the last 14 days',
    p_column_comments   => wwv_flow_t_varchar2 (
      'WORKSPACE'                , 'Workspace for which this page view log was recorded',
      'workspace_display_name'   , 'Display name for the workspace',
      'APPLICATION_ID'           , 'Application ID for which this page view log was recorded',
      'APPLICATION_NAME'         , 'Application Name for which this page view log was recorded',
      'PAGE_VIEWS'               , 'Page views aggregated by Workspace and Application',
      'DISTINCT_PAGES'           , 'Distinct page views aggregated by Workspace and Application',
      'TOTAL_ELAPSED_TIME'       , 'Total elapsed time generating page source aggregated by Workspace and Application',
      'AVERAGE_ELAPSED_TIME'     , 'Average elapsed time generating page source aggregated by Workspace and Application',
      'MINIMUM_ELAPSED_TIME'     , 'Minimum elapsed time generating page source aggregated by Workspace and Application',
      'MAXIMUM_ELAPSED_TIME'     , 'Maximum elapsed time generating page source aggregated by Workspace and Application',
      'TOTAL_ROWS_QUERIED'       , 'Total rows queried by the APEX reporting engine aggregated by Workspace and Application',
      'IP_ADDRESSES'             , 'Distinct IP addresses aggregated by Workspace and Application',
      'AGENTS'                   , 'Distinct User Agents aggregated by Workspace and Application',
      'APEX_SESSIONS'            , 'Count of sessions aggregated by Workspace and Application',
      'PAGE_VIEWS_WITH_ERRORS'   , 'Count of page views generating error text aggregated by Workspace and Application',
      'DYNAMIC_PAGE_VIEWS'       , 'Count of dynamic page views aggregated by Workspace and Application',
      'CACHED_PAGE_VIEWS'        , 'Count of static page views aggregated by Workspace and Application',
      'FIRST_VIEW'               , 'First logged page by Workspace and Application',
      'LAST_VIEW'                , 'Most recently logged page by Workspace and Application',
      'PERIOD_IN_DAYS'           , 'Period in days between first logged page view and most recent page view'
    )
  );
end;
/

create or replace view apex_workspace_log_summary_usr as
    select
           w.workspace,
           w.workspace_display_name,
           nvl(l.userid,'PUBLIC')                 apex_user,
           count(distinct l.flow_id)              applications,
           count(*)                               page_views,
           count(distinct l.flow_id||'.'||l.step_id) distinct_pages,
           sum(l.elap)                            total_elapsed_time,
           avg(l.elap)                            average_elapsed_time,
           min(l.elap)                            minimum_elapsed_time,
           max(l.elap)                            maximum_elapsed_time,
           sum(l.num_rows)                        total_rows_queried,
           count(distinct l.ip_address)           distinct_ip_addresses,
           count(distinct l.USER_AGENT)           distinct_agents,
           count(distinct l.session_id)           distinct_apex_sessions,
           sum(decode(l.sqlerrm,null,0,1))        page_views_with_errors,
           sum(decode(l.page_mode,'D',1,'C',1,0)) dynamic_page_views,
           sum(decode(l.page_mode,'R',1,0))       cached_page_views,
           min(l.time_stamp)                      first_view,
           max(l.time_stamp)                      last_view,
           max(l.time_stamp)-min(l.time_stamp)    period_in_days,
           --
           sum(decode(greatest(s.the_date - (1/1440),l.time_stamp),l.time_stamp,1,0) )  last_1_minute,
           sum(decode(greatest(s.the_date - (5/1440),l.time_stamp),l.time_stamp,1,0) )  last_5_minutes,
           sum(decode(greatest(s.the_date - (10/1440),l.time_stamp),l.time_stamp,1,0) ) last_10_minutes,
           sum(decode(greatest(s.the_date - (15/1440),l.time_stamp),l.time_stamp,1,0) ) last_15_minutes,
           sum(decode(greatest(s.the_date - (30/1440),l.time_stamp),l.time_stamp,1,0) ) last_30_minutes,
           sum(decode(greatest(s.the_date - (1/24),l.time_stamp),l.time_stamp,1,0) )    last_1_hour,
           sum(decode(greatest(s.the_date - (2/24),l.time_stamp),l.time_stamp,1,0) )    last_2_hours,
           sum(decode(greatest(s.the_date - (6/24),l.time_stamp),l.time_stamp,1,0) )    last_6_hours,
           sum(decode(greatest(s.the_date - (2/24),l.time_stamp),l.time_stamp,1,0) )    last_12_hours,
           sum(decode(greatest(s.the_date - 1,l.time_stamp),l.time_stamp,1,0)      )    last_24_hours,
           sum(decode(greatest(s.the_date - 2,l.time_stamp),l.time_stamp,1,0)      )    last_48_hours,
           sum(decode(greatest(s.the_date - 7,l.time_stamp),l.time_stamp,1,0)      )    last_7_days,
           sum(decode(greatest(s.the_date - 14,l.time_stamp),l.time_stamp,1,0)     )    last_14_days,
           --
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'01',1,0),0))  today_HH01,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'02',1,0),0))  today_HH02,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'03',1,0),0))  today_HH03,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'04',1,0),0))  today_HH04,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'05',1,0),0))  today_HH05,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'06',1,0),0))  today_HH06,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'07',1,0),0))  today_HH07,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'08',1,0),0))  today_HH08,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'09',1,0),0))  today_HH09,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'10',1,0),0))  today_HH10,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'11',1,0),0))  today_HH11,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'12',1,0),0))  today_HH12,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'13',1,0),0))  today_HH13,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'14',1,0),0))  today_HH14,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'15',1,0),0))  today_HH15,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'16',1,0),0))  today_HH16,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'17',1,0),0))  today_HH17,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'18',1,0),0))  today_HH18,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'19',1,0),0))  today_HH19,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'20',1,0),0))  today_HH20,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'21',1,0),0))  today_HH21,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'22',1,0),0))  today_HH22,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'23',1,0),0))  today_HH23,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,decode(to_char(l.time_stamp,'HH24'),'24',1,0),0))  today_HH24,
           sum(decode(greatest(trunc(s.the_date),l.time_stamp),l.time_stamp,1,0))                                              today
from wwv_flow_activity_log       l,
     wwv_flow_current_workspaces w,
     wwv_flows                   f,
     (select sysdate the_date from sys.dual) s
where l.security_group_id = w.workspace_id
  and l.security_group_id = f.security_group_id (+)
  and l.flow_id           = f.id (+)
  and l.time_stamp        > sysdate - 14
group by w.workspace_name, w.workspace_display_name, nvl(l.userid,'PUBLIC')
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_log_summary_usr',
    p_table_comment     => 'Page view activity log summarized by user for the last two weeks',
    p_column_comments   => wwv_flow_t_varchar2 (
      'WORKSPACE'                      , 'Name of Workspace that generated the page view log entry',
      'workspace_display_name'         , 'Display name for the workspace',
      'APEX_USER'                      , 'Name of the APEX User name associated with the page view log entry',
      'APPLICATIONS'                   , 'Number of applications contained in the aggregation by APEX User Name and Workspace',
      'PAGE_VIEWS'                     , 'Page views aggregated by APEX User Name and Workspace',
      'DISTINCT_PAGES'                 , 'Distinct page views aggregated by APEX User Name and Workspace',
      'TOTAL_ELAPSED_TIME'             , 'Total elapsed time logged aggregated by APEX User Name and Workspace',
      'AVERAGE_ELAPSED_TIME'           , 'Average elapsed time logged aggregated by APEX User Name and Workspace',
      'MINIMUM_ELAPSED_TIME'           , 'Minimum elapsed time logged aggregated by APEX User Name and Workspace',
      'MAXIMUM_ELAPSED_TIME'           , 'Maximum elapsed time aggregated by APEX User Name and Workspace',
      'TOTAL_ROWS_QUERIED'             , 'Total rows queried by the APEX reporting engine aggregated by APEX User Name and Workspace',
      'DISTINCT_IP_ADDRESSES'          , 'Distinct IP addresses aggregated by APEX User Name and Workspace',
      'DISTINCT_AGENTS'                , 'Distinct User Agents aggregated by APEX User Name and Workspace',
      'DISTINCT_APEX_SESSIONS'         , 'Distinct APEX Sessions aggregated by APEX User Name and Workspace',
      'PAGE_VIEWS_WITH_ERRORS'         , 'Count of page views with recorded errors aggregated by APEX User Name and Workspace',
      'DYNAMIC_PAGE_VIEWS'             , 'Count of dynamic page views aggregated by APEX User Name and Workspace',
      'CACHED_PAGE_VIEWS'              , 'Count of cached page views aggregated by APEX User Name and Workspace',
      'FIRST_VIEW'                     , 'Date of first page view by APEX User Name and Workspace',
      'LAST_VIEW'                      , 'Date of most recent page view by APEX User Name and Workspace',
      'PERIOD_IN_DAYS'                 , 'Period in days between the first recorded page view an the most recent page view',
      'LAST_1_MINUTE'                  , 'Page views recorded in the last 1 minute aggregated by APEX User Name and Workspace',
      'LAST_5_MINUTES'                 , 'Page views recorded in the last 5 minutes aggregated by APEX User Name and Workspace',
      'LAST_10_MINUTES'                , 'Page views recorded in the last 10 minutes aggregated by APEX User Name and Workspace',
      'LAST_15_MINUTES'                , 'Page views recorded in the last 15 minutes aggregated by APEX User Name and Workspace',
      'LAST_30_MINUTES'                , 'Page views recorded in the last 30 minutes aggregated by APEX User Name and Workspace',
      'LAST_1_HOUR'                    , 'Page views recorded in the last 1 hour aggregated by APEX User Name and Workspace',
      'LAST_2_HOURS'                   , 'Page views recorded in the last 2 hours aggregated by APEX User Name and Workspace',
      'LAST_6_HOURS'                   , 'Page views recorded in the last 6 hours aggregated by APEX User Name and Workspace',
      'LAST_12_HOURS'                  , 'Page views recorded in the last 12 hours aggregated by APEX User Name and Workspace',
      'LAST_24_HOURS'                  , 'Page views recorded in the last 24 hours aggregated by APEX User Name and Workspace',
      'LAST_48_HOURS'                  , 'Page views recorded in the last 48 hours aggregated by APEX User Name and Workspace',
      'LAST_7_DAYS'                    , 'Page views recorded in the last 7 days aggregated by APEX User Name and Workspace',
      'LAST_14_DAYS'                   , 'Page views recorded in the last 14 days aggregated by APEX User Name and Workspace',
      'TODAY_HH01'                     , 'Page views recorded today for the hour 00-01',
      'TODAY_HH02'                     , 'Page views recorded today for the hour 01-02',
      'TODAY_HH03'                     , 'Page views recorded today for the hour 02-03',
      'TODAY_HH04'                     , 'Page views recorded today for the hour 03-04',
      'TODAY_HH05'                     , 'Page views recorded today for the hour 04-05',
      'TODAY_HH06'                     , 'Page views recorded today for the hour 05-06',
      'TODAY_HH07'                     , 'Page views recorded today for the hour 06-07',
      'TODAY_HH08'                     , 'Page views recorded today for the hour 07-08',
      'TODAY_HH09'                     , 'Page views recorded today for the hour 08-09',
      'TODAY_HH10'                     , 'Page views recorded today for the hour 09-10',
      'TODAY_HH11'                     , 'Page views recorded today for the hour 10-11',
      'TODAY_HH12'                     , 'Page views recorded today for the hour 11-12',
      'TODAY_HH13'                     , 'Page views recorded today for the hour 12-13',
      'TODAY_HH14'                     , 'Page views recorded today for the hour 13-14',
      'TODAY_HH15'                     , 'Page views recorded today for the hour 14-15',
      'TODAY_HH16'                     , 'Page views recorded today for the hour 15-16',
      'TODAY_HH17'                     , 'Page views recorded today for the hour 16-17',
      'TODAY_HH18'                     , 'Page views recorded today for the hour 17-18',
      'TODAY_HH19'                     , 'Page views recorded today for the hour 18-19',
      'TODAY_HH20'                     , 'Page views recorded today for the hour 19-20',
      'TODAY_HH21'                     , 'Page views recorded today for the hour 20-21',
      'TODAY_HH22'                     , 'Page views recorded today for the hour 21-22',
      'TODAY_HH23'                     , 'Page views recorded today for the hour 22-23',
      'TODAY_HH24'                     , 'Page views recorded today for the hour 23-24',
      'TODAY'                          , 'Todays date on the server to provide greater context for people in different timezones'
    )
  );
end;
/

create or replace view apex_workspace_log_archive as
select
    l.workspace_name                       workspace,
    w.display_name                         workspace_display_name,
    l.log_day,
    l.workspace_id,
    l.application_id,
    l.application_name,
    l.page_events,
    l.page_views,
    l.page_accepts,
    l.partial_page_views,
    null websheet_views,
    l.rows_fetched,
    l.ir_searches,
    l.distinct_pages,
    l.distinct_users,
    l.distinct_sessions,
    l.average_render_time,
    l.median_render_time,
    l.maximum_render_time,
    l.total_render_time,
    l.error_count,
    l.content_length,
    l.public_page_events,
    l.workspace_login_events,
    l.additional_attributes
from wwv_flow_log_history  l,
     wwv_flow_companies    w,
     wwv_flow_current_sgid sgid
where (   l.workspace_id in (select security_group_id
                               from wwv_flow_company_schemas s
                              where s.schema = sgid.cu
                                and (   sgid.nls_sort='BINARY'
                                     or nlssort(s.schema,'NLS_SORT=BINARY')=nlssort(sgid.cu,'NLS_SORT=BINARY')))
       or sgid.has_apex_admin_read_role_yn = 'Y'
       or sgid.security_group_id           = l.workspace_id )
and l.workspace_id = w.provisioning_company_id (+)
and (w.provisioning_company_id != 0 or w.provisioning_company_id is null)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_log_archive',
    p_table_comment     => 'Page view activity is a daily summary of workspace acitivity that is retained until physically purged',
    p_column_comments   => wwv_flow_t_varchar2 (
      'WORKSPACE'             , 'Name of Workspace that generated the page view log entry',
      'workspace_display_name' , 'Display name for the workspace',
      'LOG_DAY'               , 'Day that all workspace statistics are summarized for',
      'WORKSPACE_ID'          , 'Identifies workspace',
      'APPLICATION_ID'        , 'Identifies the application',
      'APPLICATION_NAME'      , 'Name of the application',
      'PAGE_EVENTS'           , 'Total number of page events logged for a given day, application and workspace',
      'PAGE_VIEWS'            , 'Total page views for a given day and workspace',
      'PAGE_ACCEPTS'          , 'Total page accepts for a given day, application and workspace',

      'PARTIAL_PAGE_VIEWS'    , 'Total partial page views for a given day, application and workspace',
      'WEBSHEET_VIEWS'        , 'Obsolete',
      'ROWS_FETCHED'          , 'Total rows fetched by apex reporting engines for a given day, application and workspace',
      'IR_SEARCHES'           , 'Total not null interactive report search values logged to the activity log',
      'DISTINCT_PAGES'        , 'Summarized information by day, application and workspace',
      'DISTINCT_USERS'        , 'Summarized information by day, application and workspace',
      'DISTINCT_SESSIONS'     , 'Summarized information by day, application and workspace',
      'AVERAGE_RENDER_TIME'   , 'Summarized information by day, application  and workspace',
      'MEDIAN_RENDER_TIME'    , 'Summarized information by day, application and workspace',
      'MAXIMUM_RENDER_TIME'   , 'Summarized information by day, application and workspace',
      'TOTAL_RENDER_TIME'     , 'Summarized information by day, application and workspace',
      'ERROR_COUNT'           , 'Summarized information by day, application and workspace',
      'content_length'        , 'Summarized information by day, applicatio  and workspace',
      'public_page_events'    , 'Total public page events for a given day, application and workspace',
      'workspace_login_events' , 'Total workspace login events by day, application and workspace',
      'additional_attributes' , 'Additional information about the application in JSON format'
    )
  );
end;
/

---------------------------------------------------------
-- C L I C K  T H R U S
--
create or replace view apex_workspace_clicks as
select w.workspace,
       w.workspace_display_name,
       a.category                     category,
       a.flow_user                    apex_user,
       a.clickdate                    clickdate,
       a.id                           click_id,
       a.ip                           clicker_ip,
       w.workspace_id
  from (select clickdate, category, id, flow_user, ip, security_group_id
          from wwv_flow_clickthru_log$ l
        union all
        select clickdate, category, id, flow_user, ip, security_group_id
          from wwv_flow_clickthru_log2$ l
        ) a,
        wwv_flow_current_workspaces w
 where a.security_group_id = w.workspace_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_workspace_clicks',
    p_table_comment     => 'Clicks in Oracle APEX that are tracked by using APEX_UTIL.COUNT_CLICKS',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                , 'A work area mapped to one or more database schemas',
      'workspace_display_name'   , 'Display name for the workspace',
      'category'                 , 'Optional category used to track the click',
      'apex_user'                , 'Name of Oracle APEX user that clicked',
      'clickdate'                , 'Date of the recorded click',
      'click_id'                 , 'Optional secondary ID to further track the click',
      'clicker_ip'               , 'The IP address of the clicker',
      'workspace_id'             , 'Primary key that identifies the workspace'
    )
  );
end;
/

--------------------------------------------------------------------------------
-- Plugins
--------------------------------------------------------------------------------

create or replace view apex_appl_plugins
as
select p.id                        as plugin_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       case p.plugin_type
         when 'ITEM TYPE'           then 'Item Type'
         when 'DYNAMIC ACTION'      then 'Dynamic Action'
         when 'REGION TYPE'         then 'Region Type'
         when 'REPORT COLUMN TYPE'  then 'Report Column Type'
         when 'VALIDATION TYPE'     then 'Validation Type'
         when 'PROCESS TYPE'        then 'Process Type'
         when 'AUTHENTICATION TYPE' then 'Authentication Type'
         when 'AUTHORIZATION TYPE'  then 'Authorization Type'
         when 'WEB SOURCE TYPE'     then 'REST Data Source Type'
         else                       p.plugin_type
       end                         as plugin_type,
       p.name,
       p.display_name,
       p.category,
       p.supported_ui_types,
       p.supported_component_types,
       p.supported_data_types,
       p.image_prefix              as file_prefix,
       p.javascript_file_urls,
       p.css_file_urls,
       p.plsql_code,
       p.api_version,
       p.render_function,
       p.meta_data_function,
       -- p.emit_value_function,
       p.ajax_function,
       p.validation_function,
       p.execution_function,
       p.session_sentry_function,
       p.invalid_session_function,
       p.authentication_function,
       p.post_logout_function,
       p.builder_validation_function,
       p.migration_function,
       --
       p.wsm_fetch_function,
       p.wsm_dml_function,
       p.wsm_execute_function,
       p.wsm_discover_function,
       p.wsm_capabilities_function,
       --
       p.standard_attributes,
       p.substitute_attributes,
       nvl2(p.reference_id, 'Yes', 'No') as is_subscribed,
       nvl(( select s.flow_id||'. '||s.display_name
               from wwv_flow_plugins s
              where s.id                = p.reference_id
                and s.security_group_id = p.security_group_id),
            p.reference_id)              as subscribed_from,
       p.reference_id                    as subscribed_from_id,
       case p.subscribe_plugin_settings
         when 'Y' then 'Yes'
         else          'No'
       end                               as subscribe_plugin_settings,
       p.help_text,
       p.version_identifier,
       p.about_url,
       p.plugin_comment                  as component_comment,
       p.files_version,
       p.created_by,
       p.created_on,
       p.last_updated_by,
       p.last_updated_on,
       p.name || ' ' || substr(p.display_name, 1, 30) || length(p.display_name) ||
       ' t=' || p.plugin_type || ' c=' || p.category || ' v=' || p.version_identifier ||
       ' u=' || substr(p.about_url, 1, 30) || length(p.about_url)
       as component_signature,
       -- For compatibility with existing applications (bug# 14545162)
       -- Obsolete columns because they are accessible with view apex_appl_plugin_settings
       s.attribute_01,
       s.attribute_02,
       s.attribute_03,
       s.attribute_04,
       s.attribute_05,
       s.attribute_06,
       s.attribute_07,
       s.attribute_08,
       s.attribute_09,
       s.attribute_10,
       s.attribute_11,
       s.attribute_12,
       s.attribute_13,
       s.attribute_14,
       s.attribute_15,
       p.sql_min_column_count,
       p.sql_max_column_count,
       p.sql_examples
  from wwv_flow_authorized_and_4411 f,
       wwv_flow_plugins p,
       wwv_flow_plugin_settings s
 where p.security_group_id    = f.workspace_id
   and p.flow_id              = f.application_id
   and s.security_group_id(+) = p.security_group_id
   and s.flow_id (+)          = p.flow_id
   and s.plugin_type (+)      = p.plugin_type
   and s.plugin (+)           = 'PLUGIN_' || p.name
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugins',
    p_table_comment     => 'Stores the meta data for the plug-ins of an application.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'plugin_type' , 'Type of the plug-in.',
      'name' , 'Internal name of the plug-in which is used to reference it for example in apex_application_page_items.',
      'display_name' , 'Contains the name of the plug-in which is displayed on the UI.',
      'category' , 'Category under which the plug-in should displayed on the UI.',
      'supported_ui_types' , 'User Interface types for which the plug-in should be displayed on the UI.',
      'supported_component_types' , 'Component types for which the plug-in should be displayed on the UI.',
      'supported_data_types' , 'Data types for which the plug-in should be displayed on the UI.',
      'file_prefix' , 'File prefix which is used by the plug-in to load additional files like CSS, Javascript and images.',
      'javascript_file_urls' , 'List of JavaScript file URLs which should be loaded for the plug-in.',
      'css_file_urls' , 'List of URLs which should be loaded for the plug-in style.',
      'plsql_code' , 'PL/SQL code which contains the logic of the plug-in.',
      'api_version' , 'Callback API version used by the plug-in.',
      'render_function' , 'Procedure/Function which is called for a plug-in during rendering of the page.',
      'meta_data_function' , 'Procedure/Function which is called for a plug-in to get dynamic meta data about the plug-in.',

--          'emit_value_function' , 'Procedure/Function which is called for a plug-in to emit a value.',

      'ajax_function' , 'Procedure/Function which is called for a plug-in when there is an incoming Ajax call from the browser.',
      'validation_function' , 'Procedure/Function which is called to validate the plug-ins data before the defined validations are fired.',
      'execution_function' , 'Function which is called for plug-ins that are just used on the server side.',
      'session_sentry_function' , 'Function which is called by an authentication plug-in each time a request is made to Oracle APEX.',
      'invalid_session_function' , 'Function which is called by an authentication plug-in if the session sentry function or the authentication session verification return false.',
      'authentication_function' , 'Function which is called by an authentication plug-in to verify login credentials.',
      'post_logout_function' , 'Function which is called by an authentication plug-in after a session has been marked as terminated.',
      'builder_validation_function' , 'Function which is called to validate the entered attribute values in the builder.',
      'migration_function' , 'Function which is called when a new version of a plug-in is installed to migrate existing attribute values.',
      'wsm_fetch_function' , 'Function which is called when an APEX component uses the REST Data Source to fetch rows',
      'wsm_dml_function' , 'Function which is called when an APEX component uses the REST Data Source to manipulate rows (insert, update, delete)',
      'wsm_execute_function' , 'Function which is called when an APEX component uses the REST Data Source to execute a page process',
      'wsm_discover_function' , 'Function which is called when a new REST Data Source is created in Application Builder',
      'wsm_capabilities_function' , 'Function to declare the REST Data Source''s capabilities to the APEX Engine.',
      'standard_attributes' , 'Contains the APEX provided standard attributes which should be displayed for the plug-in.',
      'substitute_attributes' , 'Are the attribute values automatically substituted when the plug-in is called?',
      'is_subscribed' , 'Identifies if this plug-in is subscribed from another plug-in',
      'subscribed_from' , 'Identifies the master component from which this component is subscribed',
      'subscribed_from_id' , 'Id the master component from which this component is subscribed',
      'subscribe_plugin_settings' , 'Are the defaults of the subscribed component copied?',
      'help_text' , 'Help text which is displayed for the plug-in in the Builder.',
      'version_identifier' , 'Version identifier of the plug-in.',
      'about_url' , 'URL to get additional information about the plug-in.',
      'component_comment' , 'Developer Comment',
      'files_version' , 'Version counter of the attached plug-in files.',
      'created_by' , 'APEX developer who created the plug-in',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update',
      'component_signature' , 'Identifies attributes defined at a given component level to facilitate application comparisons',
      'attribute_01' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_02' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_03' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_04' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_05' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_06' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_07' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_08' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_09' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_10' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_11' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_12' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_13' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_14' , 'Dynamic attribute to store additional data (obsolete)',
      'attribute_15' , 'Dynamic attribute to store additional data (obsolete)',
      'sql_min_column_count' , 'Minimum number of columns the SQL query of the LOV or region source has to have (obsolete)',
      'sql_max_column_count' , 'Maximum number of columns the SQL query of the LOV or region source can have (obsolete)',
      'sql_examples' , 'LOV or region source SQL examples which are displayed for the plug-in in the Builder (obsolete)'
    )
  );
end;
/


create or replace view apex_appl_plugin_std_attrs
as
select a.id                 as plugin_std_attribute_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       a.plugin_id,
       p.name               as plugin_name,
       a.name               as name,
       case a.is_required
         when 'Y' then 'Yes'
         else          'No'
       end                  as is_required,
       a.default_value,
       a.sql_min_column_count,
       a.sql_max_column_count,
       a.supported_ui_types,
       a.supported_component_types,
       a.depending_on_attribute_id,
       case a.depending_on_has_to_exist
         when 'Y' then 'Yes'
         when 'N' then 'No'
       end as depending_on_always_eval,
       case a.depending_on_condition_type
         when 'EQUALS'       then 'equal to'
         when 'NOT_EQUALS'   then 'not equal to'
         when 'NULL'         then 'is null'
         when 'NOT_NULL'     then 'is not null'
         when 'IN_LIST'      then 'in list'
         when 'NOT_IN_LIST'  then 'not in list'
         else a.depending_on_condition_type
       end depending_on_condition_type,
       a.depending_on_expression,
       a.examples,
       a.help_text,
       a.attribute_comment as component_comment,
       a.created_by,
       a.created_on,
       a.last_updated_by,
       a.last_updated_on
  from wwv_flow_authorized_and_4411 f,
       wwv_flow_plugins p,
       wwv_flow_plugin_std_attributes a
 where p.security_group_id = f.workspace_id
   and p.flow_id           = f.application_id
   and a.security_group_id = p.security_group_id
   and a.plugin_id         = p.id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugin_std_attrs',
    p_table_comment     => 'Stores additional meta data for plug-in standard attributes.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_std_attribute_id'     , 'Identifies the primary key of this component',
      'workspace'                   , 'A work area mapped to one or more database schemas',
      'workspace_display_name'      , 'Display name for the workspace',
      'application_id'              , 'Application Primary Key, Unique over all workspaces',
      'application_name'            , 'Identifies the application',
      'plugin_id'                   , 'Id of the plug-in this plugin attribute is part of',
      'plugin_name'                 , 'Name of the plug-in this plugin attribute is part of',
      'name'                        , 'Name of the plug-in standard attribute',
      'is_required'                 , 'Should the Builder enforce this attribute to be filled out?',
      'sql_min_column_count'        , 'Minimum number of columns the SQL query of the attribute has to have.',
      'sql_max_column_count'        , 'Maximum number of columns the SQL query of the attribute can have.',
      'supported_ui_types'          , 'User Interface types for which the attribute should be displayed on the UI.',
      'supported_component_types'   , 'Component types for which the attribute should be displayed on the UI.',
      'depending_on_attribute_id'   , 'Attribute which defines if the current attribute is displayed or not.',
      'depending_on_always_eval'    , 'Is the dependency check only evaluated if the depending attribute is displayed?',
      'depending_on_condition_type' , 'Identifies the type of condition used to define when the attribute gets displayed.',
      'depending_on_expression'     , 'Identifies the condition expression used to define when the attribute gets displayed.',
      'examples'                    , 'Examples displayed in APEX Builder for this attribute.',
      'help_text'                   , 'Help text displayed in APEX Builder for this attribute.',
      'component_comment'           , 'Developer comment',
      'created_by'                  , 'APEX developer who created the plugin attribute',
      'created_on'                  , 'Date of creation',
      'last_updated_by'             , 'APEX developer who made last update',
      'last_updated_on'             , 'Date of last update',
      'DEFAULT_VALUE'               , 'Default value of the attribute'
    )
  );
end;
/
create or replace view apex_appl_plugin_attributes
as
select a.id                 as plugin_attribute_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       a.plugin_id,
       p.name               as plugin_name,
       case a.attribute_scope
         when 'APPLICATION' then 'Application'
         when 'COMPONENT'   then 'Component'
         else a.attribute_scope
       end                  as attribute_scope,
       a.attribute_sequence,
       a.display_sequence,
       a.prompt,
       case a.attribute_type
         when 'CHECKBOX'         then 'Checkbox'
         when 'SELECT LIST'      then 'Select List'
         when 'TEXT'             then 'Text'
         when 'TEXTAREA'         then 'Textarea'
         when 'NUMBER'           then 'Number'
         when 'INTEGER'          then 'Integer'
         when 'PAGE ITEM'        then 'Page Item'
         when 'PAGE ITEMS'       then 'Page Items'
         when 'PAGE NUMBER'      then 'Page Number'
         when 'SQL'              then 'SQL Query'
         when 'PLSQL'            then 'PL/SQL Code'
         when 'PLSQL EXPRESSION' then 'PL/SQL Expression'
         else a.attribute_type
       end                  as attribute_type,
       case a.is_required
         when 'Y' then 'Yes'
         else          'No'
       end                  as is_required,
       case a.is_common
         when 'Y' then 'Yes'
         else          'No'
       end                  as is_common,
       case a.show_in_wizard
         when 'Y' then 'Yes'
         else          'No'
       end                  as show_in_wizard,
       a.default_value,
       a.display_length,
       a.max_length,
       a.min_value,
       a.max_value,
       a.unit,
       a.sql_min_column_count,
       a.sql_max_column_count,
       a.column_data_types,
       a.supported_ui_types,
       a.supported_component_types,
       a.is_translatable,
       a.depending_on_attribute_id,
       a.depending_on_comp_prop_id,
       case a.depending_on_has_to_exist
         when 'Y' then 'Yes'
         when 'N' then 'No'
       end as depending_on_always_eval,
       case a.depending_on_condition_type
         when 'EQUALS'       then 'equal to'
         when 'NOT_EQUALS'   then 'not equal to'
         when 'NULL'         then 'is null'
         when 'NOT_NULL'     then 'is not null'
         when 'IN_LIST'      then 'in list'
         when 'NOT_IN_LIST'  then 'not in list'
         else a.depending_on_condition_type
       end depending_on_condition_type,
       a.depending_on_expression,
       a.reference_scope,
       a.examples,
       a.help_text,
       a.attribute_comment  as component_comment,
       a.created_by,
       a.created_on,
       a.last_updated_by,
       a.last_updated_on
  from wwv_flow_authorized_and_4411 f,
       wwv_flow_plugins p,
       wwv_flow_plugin_attributes a
 where p.security_group_id = f.workspace_id
   and p.flow_id           = f.application_id
   and a.security_group_id = p.security_group_id
   and a.plugin_id         = p.id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugin_attributes',
    p_table_comment     => 'Stores the meta data for the dynamic attributes of a plug-in.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_attribute_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'plugin_id' , 'Id of the plug-in this plugin attribute is part of',
      'plugin_name' , 'Name of the plug-in this plugin attribute is part of',
      'attribute_scope' , 'Is the attribute set for an application or a component.',
      'attribute_sequence' , 'Sequence which correlates with the attribute_xx columns for example in apex_application_page_items.',
      'display_sequence' , 'Sequence in which the attributes are displayed in the Builder.',
      'prompt' , 'Prompt which is displayed for that attribute in the Builder.',
      'help_text' , 'Help text which is displayed for that attribute in the Builder.',
      'attribute_type' , 'Type of control which is used to render the attribute field in the Builder.',
      'is_required' , 'Should the Builder enforce this attribute to be filled out?',
      'is_common' , 'Should the Builder always show that attribute?',
      'show_in_wizard' , 'Should the Builder show that attribute in a Create Wizard?',
      'default_value' , 'Default value which is used in the Builder when the item type is selected.',
      'display_length' , 'Display length of the attribute used when displayed in the Builder.',
      'max_length' , 'Maximum length of the attribute used when displayed in the Builder.',
      'min_value' , 'Minimum allowed value of the attribute.',
      'max_value' , 'Maximum allowed value of the attribute.',
      'unit' , 'Unit of the attribute used when displayed in the Builder.',
      'sql_min_column_count' , 'Minimum number of columns the SQL query of the attribute has to have.',
      'sql_max_column_count' , 'Maximum number of columns the SQL query of the attribute can have.',
      'column_data_types' , 'Data types which are supported by the attribute of type "Region SQL Statement Column".',
      'supported_ui_types' , 'User Interface types for which the attribute should be displayed on the UI.',
      'supported_component_types' , 'Component types for which the attribute should be displayed on the UI.',
      'is_translatable' , 'Is the attribute value exported into the XLIFF file for translation.',
      'depending_on_attribute_id' , 'Attribute which defines if the current attribute is displayed or not.',
      'depending_on_comp_prop_id' , 'Property which defines if the current attribute is displayed or not.',
      'depending_on_always_eval' , 'Is the dependency check only evaluated if the depending attribute is displayed?',
      'depending_on_condition_type' , 'Identifies the type of condition used to define when the attribute gets displayed.',
      'depending_on_expression' , 'Identifies the condition expression used to define when the attribute gets displayed.',
      'reference_scope' , 'Can the attribute reference region columns?',
      'examples' , 'Examples displayed in APEX Builder for this attribute.',
      'help_text' , 'Help text displayed in APEX Builder for this attribute.',
      'component_comment' , 'Developer Comment',
      'created_by' , 'APEX developer who created the plugin attribute',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/
create or replace view apex_appl_plugin_attr_values
as
select v.id                 as plugin_attribute_value_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       v.plugin_attribute_id,
       a.prompt             as plugin_attribute_prompt,
       v.display_sequence,
       v.display_value,
       v.return_value,
       v.is_quick_pick,
       v.help_text,
       v.created_by,
       v.created_on,
       v.last_updated_by,
       v.last_updated_on
  from wwv_flow_authorized_and_4411 f,
       wwv_flow_plugin_attr_values v,
       wwv_flow_plugin_attributes a
 where v.security_group_id = f.workspace_id
   and v.flow_id           = f.application_id
   and a.security_group_id = v.security_group_id
   and a.id                = v.plugin_attribute_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugin_attr_values',
    p_table_comment     => 'Stores the possible values of a plug-in attribute if it''s of type selectlist.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_attribute_value_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'plugin_attribute_id' , 'Id of the attribute this value is part of',
      'plugin_attribute_prompt' , 'Label of the attribute this value is part of',
      'display_sequence' , 'Order sequence in which the values are displayed.',
      'display_value' , 'Value displayed to end users',
      'return_value' , 'Value stored in attribute_xx column.',
      'is_quick_pick' , 'Should the Builder show the value as a quick pick option?',
      'help_text' , 'Help text displayed in APEX Builder for this value.',
      'created_by' , 'APEX developer who created the plugin attribute value',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/

create or replace view apex_appl_plugin_files
as
select a.id                 as plugin_file_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       a.plugin_id,
       p.name               as plugin_name,
       a.file_name,
       a.mime_type,
       a.file_charset,
       a.file_content,
       a.created_by,
       a.created_on,
       a.last_updated_by,
       a.last_updated_on
  from wwv_flow_authorized_and_4411 f,
       wwv_flow_plugins p,
       wwv_flow_plugin_files a
 where p.security_group_id = f.workspace_id
   and p.flow_id   = f.application_id
   and a.security_group_id = p.security_group_id
   and a.plugin_id = p.id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugin_files',
    p_table_comment     => 'Stores the files like CSS, images, javascript files, ... of a plug-in.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_file_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'plugin_id' , 'Id of the plug-in this plug-in file is part of',
      'plugin_name' , 'Name of the plug-in this plug-in file is part of',
      'file_name' , 'Name of the file.',
      'mime_type' , 'Mime type of the file.',
      'file_charset' , 'IANA charset used for text files.',
      'file_content' , 'Blob content of the file.',
      'created_by' , 'APEX developer who created the plug-in file',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/

create or replace view apex_appl_plugin_events
as
select e.id                 as plugin_event_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       e.plugin_id,
       p.name               as plugin_name,
       e.name,
       e.display_name,
       e.created_by,
       e.created_on,
       e.last_updated_by,
       e.last_updated_on
  from wwv_flow_authorized_and_4411 f,
       wwv_flow_plugins p,
       wwv_flow_plugin_events e
 where p.security_group_id = f.workspace_id
   and p.flow_id   = f.application_id
   and e.security_group_id = p.security_group_id
   and e.plugin_id = p.id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugin_events',
    p_table_comment     => 'Stores which events can be triggered by this plug-in. This events are used for binding dynamic actions.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_event_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'plugin_id' , 'Id of the plug-in this plug-in event is part of',
      'plugin_name' , 'Name of the plug-in this plug-in event is part of',
      'name' , 'Internal name of the event which is used to reference it from apex_application_page_da and to bind the Javascript event.',
      'display_name' , 'Contains the name of the event which is displayed on the UI.',
      'created_by' , 'APEX developer who created the plug-in event',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/
/* $$$ pawolf09082016: didn't make it into 5.1
create or replace view apex_appl_plugin_item_filters
as
select f.id                 as plugin_filter_id,
       a.workspace,
       a.workspace_display_name,
       a.application_id,
       a.application_name,
       f.plugin_id,
       p.name               as plugin_name,
       f.name,
       f.display_name,
       f.created_by,
       f.created_on,
       f.last_updated_by,
       f.last_updated_on
  from wwv_flow_authorized_and_4411 a,
       wwv_flow_plugins p,
       wwv_flow_plugin_item_filters f
 where p.flow_id   = a.application_id
   and f.plugin_id = p.id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugin_item_filters',
    p_table_comment     => 'Stores which filter operators can be used by this plug-in.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_filter_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'plugin_id' , 'Id of the plug-in this plug-in event is part of',
      'plugin_name' , 'Name of the plug-in this plug-in event is part of',
      'name' , 'Internal name of the filter which is used to reference it in the plug-in and in Page Designer.',
      'display_name' , 'Contains the name of the filter operator which is displayed in Page Designer.',
      'created_by' , 'APEX developer who created the plug-in event',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/
*/

create or replace view apex_appl_plugin_settings
as
select p.id                        as plugin_setting_id,
       f.workspace,
       f.workspace_display_name,
       f.application_id,
       f.application_name,
       p.plugin_type,
       case substr(p.plugin, 1, 7)
         when 'NATIVE_' then
             ( select np.display_name from wwv_flow_plugins np where np.security_group_id = 10 and np.flow_id = 4411 and np.plugin_type = p.plugin_type and np.name = substr(p.plugin, 8) )
         when 'PLUGIN_' then
             ( select fp.display_name from wwv_flow_plugins fp where fp.security_group_id = p.security_group_id and fp.flow_id = p.flow_id and fp.plugin_type = p.plugin_type and fp.name = substr(p.plugin, 8) )
         else p.plugin
       end            as plugin,
       p.plugin       as plugin_code,
       p.attribute_01,
       p.attribute_02,
       p.attribute_03,
       p.attribute_04,
       p.attribute_05,
       p.attribute_06,
       p.attribute_07,
       p.attribute_08,
       p.attribute_09,
       p.attribute_10,
       p.attribute_11,
       p.attribute_12,
       p.attribute_13,
       p.attribute_14,
       p.attribute_15,
       p.created_by,
       p.created_on,
       p.last_updated_by,
       p.last_updated_on
  from wwv_flow_authorized f,
       wwv_flow_plugin_settings p
 where p.security_group_id = f.workspace_id
   and p.flow_id           = f.application_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_appl_plugin_settings',
    p_table_comment     => 'Stores the values of custom attribute of scope "application" of a plug-in.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'plugin_setting_id' , 'Identifies the primary key of this component',
      'workspace' , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id' , 'Application Primary Key, Unique over all workspaces',
      'application_name' , 'Identifies the application',
      'plugin_type' , 'Type of the plug-in.',
      'plugin' , 'Identifies the referenced plug-in',
      'plugin_code' , 'Internal code of PLUGIN',
      'attribute_01' , 'Dynamic attribute to store additional data',
      'attribute_02' , 'Dynamic attribute to store additional data',
      'attribute_03' , 'Dynamic attribute to store additional data',
      'attribute_04' , 'Dynamic attribute to store additional data',
      'attribute_05' , 'Dynamic attribute to store additional data',
      'attribute_06' , 'Dynamic attribute to store additional data',
      'attribute_07' , 'Dynamic attribute to store additional data',
      'attribute_08' , 'Dynamic attribute to store additional data',
      'attribute_09' , 'Dynamic attribute to store additional data',
      'attribute_10' , 'Dynamic attribute to store additional data',
      'attribute_11' , 'Dynamic attribute to store additional data',
      'attribute_12' , 'Dynamic attribute to store additional data',
      'attribute_13' , 'Dynamic attribute to store additional data',
      'attribute_14' , 'Dynamic attribute to store additional data',
      'attribute_15' , 'Dynamic attribute to store additional data',
      'created_by' , 'APEX developer who created the plug-in setting',
      'created_on' , 'Date of creation',
      'last_updated_by' , 'APEX developer who made last update',
      'last_updated_on' , 'Date of last update'
    )
  );
end;
/

--------------------------------------------------------------------------------
-- A P P L I C A T I O N   J E T   C H A R T S
--
prompt ...apex_application_page_charts

create or replace view apex_application_page_charts
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    p.id                             page_id,
    p.name                           page_name,
    c.region_id                      region_id,
    (select pl.plug_name
       from wwv_flow_page_plugs pl
      where pl.id                = c.region_id
        and pl.security_group_id = c.security_group_id) region_name,
    c.id                             chart_id,
    --
    decode(c.chart_type,
    'area', 'Area',
    'bar', 'Bar',
    'bubble', 'Bubble',
    'combo', 'Combination',
    'dial', 'Status Meter Gauge',
    'funnel', 'Funnel',
    'line', 'Line',
    'lineWithArea', 'Line with Area',
    'pie', 'Pie',
    'scatter', 'Scatter',
    'spark', 'Spark',
    'donut','Donut',
    'range','Range',
    'polar','Polar',
    'pyramid','Pyramid',
    'radar','Radar',
    'stock','Stock',
    'gantt','Gantt',
    c.chart_type)                    chart_type,
    c.title                          chart_title,
    c.width                          chart_width,
    c.height                         chart_height,
    --
    decode(c.animation_on_display,
    'alphaFade', 'Alpha Fade',
    'auto', 'Automatic',
    'none','None',
    'zoom','Zoom',
    c.animation_on_display)          animation_on_display,
    decode(c.animation_on_data_change,
    'auto', 'Automatic',
    'slideToLeft','Slide to Left',
    'slideToRight','Slide to Right',
    'none','None',
    animation_on_data_change)        animation_on_data_change,
    decode(c.orientation,
    'vertical','Vertical',
    'horizontal','Horizontal',
    c.orientation)                   orientation,
    decode(c.data_cursor,
    'auto','Automatic',
    'on','On',
    'off','Off',
    c.data_cursor)                   data_cursor,
    decode(c.data_cursor_behavior,
    'auto','Automatic',
    'smooth','Smooth',
    'snap','Snap',
    c.data_cursor_behavior)          data_cursor_behavior,
    decode(c.hide_and_show_behavior,
    'none','None',
    'withRescale','With Rescale',
    'withoutRescale', 'Without Rescale',
    c.hide_and_show_behavior)        hide_and_show_behavior,
    decode(c.hover_behavior,
    'dim','Dim',
    'none','None',
    c.hover_behavior)                hover_behavior,
    decode(c.stack,
    'on','On',
    'off','Off',
    c.stack)                         stack,
    decode(c.stack_label,
    'on','On',
    'off','Off',
    c.stack_label)                   stack_label,
    c.stack_font_family              stack_font_family,
    decode (c.stack_font_style,
    'normal','Normal',
    'italic','Italic',
    'oblique','Oblique',
    c.stack_font_style)              stack_font_style,
    c.stack_font_size                stack_font_size,
    c.stack_font_color               stack_font_color,
    decode(c.spark_chart,
    'Y','Yes',
    'N','No',
    c.spark_chart)                   spark_chart,
    decode(c.connect_nulls,
    'Y','Yes',
    'N','No',
    c.connect_nulls)                 connect_nulls,
    decode(c.stock_render_as,
    'area', 'Area',
    'bar', 'Bar',
    'candlestick','Candlestick',
    'line', 'Line',
    'lineWithArea', 'Line with Area',
    c.stock_render_as)               stock_render_as,
    --
    c.value_min                      value_min,
    decode(c.value_text_type,
    'number','Number',
    'percent','Percent',
    c.value_text_type)               value_text_type,
    decode(c.value_position,
      'auto', 'Automatic',
      'center','Center',
      'insideIndicatorEdge', 'Inside Indicator Edge',
      'outsideIndicatorEdge', 'Outside Indicator Edge',
      'outsidePlotArea','Outside Plot Area',
      'withLabel','With Label',
    c.value_position)                value_position,
    c.value_format_type              value_format_type,
    c.value_decimal_places           value_decimal_places,
    c.value_currency                 value_currency,
    c.value_numeric_pattern          value_numeric_pattern,
    decode(c.value_format_scaling,
    'auto','Automatic',
    'none','None',
    'thousand','Thousand',
    'million','Million',
    'billion','Billion',
    'trillion','Trillion',
    'quadrillion','Quadrillion',
    c.value_format_scaling)          value_format_scaling,
    --
    decode(c.sorting,
       'value-asc','Value - Ascending',
       'value-desc','Value - Descending',
       'label-asc','Label - Ascending',
       'label-asc','Label - Descending',
       c.sorting)                    chart_sort_order,
    decode(c.fill_multi_series_gaps,
    'Y','Yes',
    'N','No',
    c.fill_multi_series_gaps)        fill_multi_series_gaps,
    --
    decode(c.zoom_and_scroll,
    'delayed','Delayed',
    'delayedScrollOnly','Delayed Scroll Only',
    'live','Live',
    'liveScrollOnly','Live Scroll Only',
    'off','Off',
    c.zoom_and_scroll)               zoom_and_scroll,
    --
    decode(c.zoom_direction,
    'auto','Both X and Y axes',
    'x','X-Axis',
    'y','Y-Axis',
    c.zoom_direction)                zoom_direction,
    --
    decode(c.initial_zooming,
    'none','None',
    'first','First',
    'last','Last',
    c.initial_zooming)               initial_zooming,
    --
    decode(c.tooltip_rendered,
    'Y','Yes',
    'N','No',
    c.tooltip_rendered)              tooltip_rendered,
    decode(c.show_series_name,
    'Y','Yes',
    'N','No',
    c.show_series_name)              show_series_name,
    decode(c.show_group_name,
    'Y','Yes',
    'N','No',
    c.show_group_name)               show_group_name,
    decode(c.show_value,
    'Y','Yes',
    'N','No',
    c.show_value)                    show_value,
    decode(c.show_label,
    'Y','Yes',
    'N','No',
    c.show_label)                    show_label,
    decode(c.show_row,
    'Y','Yes',
    'N','No',
    c.show_row)                      show_row,
    decode(c.show_start,
    'Y','Yes',
    'N','No',
    c.show_start)                    show_start,
    decode(c.show_end,
    'Y','Yes',
    'N','No',
    c.show_end)                      show_end,
    decode(c.show_progress,
    'Y','Yes',
    'N','No',
    c.show_progress)                 show_progress,
    decode(c.show_baseline,
    'Y','Yes',
    'N','No',
    c.show_baseline)                 show_baseline,
    c.custom_tooltip                 custom_tooltip,
    --
    decode(c.legend_rendered,
    'on','On',
    'off','Off',
    c.legend_rendered)               legend_rendered,
    c.legend_title                   legend_title,
    decode(c.legend_position,
    'start','Start',
    'end','End',
    'bottom','Bottom',
    'top','Top',
    'auto','Automatic',
    c.legend_position)               legend_position,
    c.legend_font_family             legend_font_family,
    decode (c.legend_font_style,
    'normal','Normal',
    'italic','Italic',
    'oblique','Oblique',
    c.legend_font_style)             legend_font_style,
    c.legend_font_size               legend_font_size,
    c.legend_font_color              legend_font_color,
    --
    decode(c.overview_rendered,
    'on','On',
    'off','Off',
    c.overview_rendered)             overview_rendered,
    c.overview_height                overview_height,
    c.pie_other_threshold            pie_other_threshold,
    decode(c.pie_selection_effect,
    'explode','Explode',
    'highlightAndExplode','Highlight and Explode',
    'highlight','Highlight',
    c.pie_selection_effect)          pie_selection_effect,
    decode(c.time_axis_type,
    'enabled','Enabled',
    'mixedFrequency','Mixed Frequency',
    'disabled','Disabled',
    'auto','Automatic',
    c.time_axis_type)                time_axis_type,
    --
    decode(c.horizontal_grid,
      'auto','Automatic',
      'hidden','Hidden',
      'visible','Visible',
      c.horizontal_grid)             horizontal_grid,
    decode(c.vertical_grid,
      'auto','Automatic',
      'hidden','Hidden',
      'visible','Visible',
      c.vertical_grid)               vertical_grid,
    decode(c.row_axis_rendered,
      'on','On',
      'off','Off',
      c.row_axis_rendered )          row_axis_rendered,
    decode (c.gantt_axis_position,
       'top','Top',
       'bottom','Bottom',
       c.gantt_axis_position )       gantt_axis_position,
    --
    decode(c.gauge_orientation,
      'circular','Circular',
      'horizontal','Horizontal',
      'vertical','Vertical',
      c.gauge_orientation)           gauge_orientation,
    c.gauge_indicator_size           gauge_indicator_size,
    c.gauge_inner_radius             gauge_inner_radius,
    decode(c.gauge_plot_area,
      'on','On',
      'off','Off',
      c.gauge_plot_area)             gauge_plot_area,
    c.gauge_start_angle              gauge_start_angle,
    c.gauge_angle_extent             gauge_angle_extent,
    decode(c.show_gauge_value,
      'Y','Yes',
      'N','No',
      c.show_gauge_value)            show_gauge_value,
    --
    c.no_data_found_message,
    c.javascript_code                javascript_code,
    --
    (select count(*) from wwv_flow_jet_chart_axes ca where ca.chart_id = c.id and ca.security_group_id = c.security_group_id) number_of_axes,
    (select count(*) from wwv_flow_jet_chart_series cs where cs.chart_id = c.id and cs.security_group_id = c.security_group_id) number_of_series,
    --
    c.updated_by                     last_updated_by,
    c.updated_on                     last_updated_on,
    --
    decode(c.chart_type,
    'area', 'Area',
    'bar', 'Bar',
    'bubble', 'Bubble',
    'combo', 'Combination',
    'dial', 'Status Meter Gauge',
    'funnel', 'Funnel',
    'line', 'Line',
    'lineWithArea', 'Line with Area',
    'pie', 'Pie',
    'scatter', 'Scatter',
    'spark', 'Spark',
    'donut','Donut',
    'range','Range',
    'polar','Polar',
    'pyramid','Pyramid',
    'radar','Radar',
    'stock','Stock',
    'gantt','Gantt',
    c.chart_type)
    ||' title='||substr(c.title,1,20)||'.'||length(c.title)
    ||' javascript='||sys.dbms_lob.substr(c.javascript_code,30,1)||'.'||sys.dbms_lob.getlength(c.javascript_code)  component_signature
from wwv_flow_jet_charts c,
     wwv_flow_steps      p,
     wwv_flow_authorized f
where f.workspace_id      = p.security_group_id
  and f.application_id    = p.flow_id
  and p.security_group_id = c.security_group_id
  and p.flow_id           = c.flow_id
  and p.id                = c.page_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_charts',
    p_table_comment     => 'Identifies a chart associated with a Page and Region.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                , 'A work area mapped to one or more database schemas.',
      'workspace_display_name'   , 'Display name for the workspace.',
      'application_id'           , 'Application Primary Key, Unique over all workspaces.',
      'application_name'         , 'Identifies the application.',
      'page_id'                  , 'ID of the application page.',
      'page_name'                , 'Name of the application page.',
      'region_name'              , 'Identifies the region name in which this chart is displayed.',
      'region_id'                , 'Identifies the Page Region foreign key to the apex_application_page_regions view.',
      'chart_id'                 , 'Primary Key of the chart.',
      'chart_type'               , 'Chart type to indicate the style in which the chart will render.',
      'chart_title'              , 'A title to display at the top of the chart.',
      'chart_width'              , 'Width of the chart.',
      'chart_height'             , 'Height of the chart.',
      'animation_on_display'     , 'An animation to control the initial appearance of the chart.',
      'animation_on_data_change' , 'An animation to control the appearance of the chart following a data change.',
      'orientation'              , 'Identifies the orientation of the chart.  Only applies to bar, line, area, combo, and funnel charts.',
      'data_cursor'              , 'Identifies whether a cursor will be rendered on the chart.',
      'data_cursor_behavior'     , 'Identifies the behavior of the cursor when moving between data items on the chart.',
      'hide_and_show_behavior'   , 'Defines the hide and show behavior that is performed when clicking on a legend item.',
      'hover_behavior'           , 'Defines the behavior applied when hovering over data items.',
      'stack'                    , 'Defines whether the data items are stacked. Only applies to bar, line, area, and combo charts. Does not apply to range series.',
      'stack_label'              , 'Defines whether the total values of stacked data items should be displayed. Only applies to bar charts.',
      'stack_font_family'        , 'Defines the font family of the stack label.',
      'stack_font_style'         , 'Defines the font style of the stack label.',
      'stack_font_size'          , 'Defines the font size of the stack label.',
      'stack_font_color'         , 'Defines the color of the stack label.',
      'spark_chart'              , 'Identifies whether the chart is to be rendered as a spark chart.',
      'connect_nulls'            , 'Specifies the handling of connecting null data points on a multi-series line chart.',
      'stock_render_as'          , 'Identifies the type of series to be used to render the stock chart.',
      'value_min'                , 'The minimum value of the gauge chart.',
      'value_text_type'          , 'Specifies whether the tick labels on a dial gauge chart label are displayed as a number or as a percentage of the total value.',
      'value_position'           , 'Specifies the position of the value on a Status Meter Gauge chart.',
      'value_format_type'        , 'The format type to be used to represent the values.',
      'value_decimal_places'     , 'Number of decimal places to use in values.',
      'value_currency'           , 'The ISO 4217 alphabetic currency code used to represent monetary values.',
      'value_numeric_pattern'    , 'The numeric pattern used to represent numeric values.',
      'value_format_scaling'     , 'The scaling behavior of the values.',
      'chart_sort_order'         , 'Identifies the current chart sorting method to be applied to a multi-series chart.',
      'fill_multi_series_gaps'   , 'Identifies whether gaps in a multi-series chart should be automatically filled.',
      'zoom_and_scroll'          , 'Specifies the zoom and scroll behavior of the chart.',
      'zoom_direction'           , 'Specifies the zoom direction on a bubble or scatter chart.',
      'initial_zooming'          , 'Specifies the initial zooming position on the chart.',
      'tooltip_rendered'         , 'Specifies whether a tooltip is rendered on the chart.',
      'show_series_name'         , 'Specifies whether the series name should be included in the tooltip.',
      'show_group_name'          , 'Specifies whether the group name should be included in the tooltip.',
      'show_value'               , 'Specifies whether the value should be included in the tooltip.',
      'show_label'               , 'Specifies whether the label should be included in the tooltip.',
      'show_row'                 , 'Specifies whether the row information should be included in the tooltip.',
      'show_start'               , 'Specifies whether the start date should be included in the tooltip.',
      'show_end'                 , 'Specifies whether the end date should be included in the tooltip.',
      'show_progress'            , 'Specifies whether the progress information should be included in the tooltip.',
      'show_baseline'            , 'Specifies whether the baseline information should be included in the tooltip.',
      'custom_tooltip'           , 'Specifies the custom tooltip to be used on the chart.',
      'legend_rendered'          , 'Specifies whether a legend is rendered on the chart.',
      'legend_title'             , 'A title to display at the top of the legend.',
      'legend_position'          , 'Specifies the position of the legend.',
      'legend_font_family'       , 'Defines the font family of the legend entry.',
      'legend_font_style'        , 'Defines the font style of the legend entry.',
      'legend_font_size'         , 'Defines the font size of the legend entry.',
      'legend_font_color'        , 'Defines the color of the legend entry.',
      'overview_rendered'        , 'Specifies whether an overview scrollbar is visible on the chart. Only applies if "Zoom and Scroll" is not off. Currently only supported for vertical bar, line, area, stock, and combo charts.',
      'overview_height'          , 'Specifies the height of the overview scrollbar in pixels (e.g. "50px") or percent (e.g. "15%").',
      'pie_other_threshold'      , 'Specifies the fraction of the whole pie under which a slice would be aggregated into an "Other" slice. Valid values range from 0 (default) to 1.',
      'time_axis_type'           , 'The type of time axis to display in the chart.',
      'no_data_found_message'    , 'Message to display when no rows are returned by the chart',
      'horizontal_grid'          , 'Specifies the visibility of the gantt chart horizontal grid',
      'vertical_grid'            , 'Specifies the visibility of the gantt chart vertical grid',
      'row_axis_rendered'        , 'Specifies whether the gantt chart row axis is rendered',
      'gantt_axis_position'      , 'Specifies the position of the gantt chart axis',
      'gauge_orientation'        , 'Identifies the orientation of the Status Meter Gauge chart.',
      'gauge_indicator_size'     , 'Identifies the indicator size, which defines the ratio of relative thickness of the indicator to the plot area of a Status Meter Gauge chart.',
      'gauge_inner_radius'       , 'Specifies the inner radius of a Status Meter gauge with circular orientation, defined by the distance from the center of the gauge to the innermost edge of the indicator and plot area. Valid values are a percent or ratio from 0 to 1.',
      'gauge_plot_area'          , 'Defines if the plot area of a Status Meter Gauge chart is to be rendered.',
      'gauge_start_angle'        , 'Specifies the start angle of a Status Meter Gauge chart with circular orientation. The value is provided in degrees.',
      'gauge_angle_extent'       , 'Specifies the angle extent of a Status Meter Gauge chart with circular orientation. The value is provided in degrees.',
      'show_gauge_value'         , 'Specifies whether the value is rendered on the Status Meter Gauge chart.',
      'javascript_code'          , 'User defined custom JavaScript code for the chart.',
      'number_of_axes'           , 'Specifies the number of axes associated with the chart.',
      'number_of_series'         , 'Specifies the number of series associated with the chart.',
      'last_updated_by'          , 'APEX developer who made last update.',
      'last_updated_on'          , 'Date of last update.',
      'component_signature'      , 'Identifies attributes defined at a given component level to facilitate application comparisons.',
      'pie_selection_effect'     , 'Effect applied when a pie segment is selected.'
    )
  );
end;
/

prompt ...apex_application_page_chart_s

create or replace view apex_application_page_chart_s
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    c.page_id                            page_id ,
    (select st.name
       from wwv_flow_steps st
      where st.id                = c.page_id
        and st.flow_id           = c.flow_id
        and st.security_group_id = c.security_group_id) page_name,
    c.region_id                          region_id,
    (select pp.plug_name
       from wwv_flow_page_plugs pp
      where pp.id                = c.region_id
        and pp.flow_id           = c.flow_id
        and pp.security_group_id = c.security_group_id) region_name,
    --
    cs.chart_id                          chart_id,
    cs.id                                series_id,
    cs.seq                               series_seq,
    cs.name                              series_name,
    --
    decode(cs.data_source_type,
      'SQL_QUERY','SQL Query',
      'FUNCTION_RETURNING_SQL_QUERY','Function Body returning SQL Query',
      'REGION_SOURCE', 'Region Source',
      'TABLE',       'Table',
    cs.data_source_type)                 data_source_type,
    cs.function_body_language            function_body_language,
    cs.data_source                       data_source,
    cs.max_row_count                     max_row_count,
    cs.ajax_items_to_submit              items_to_submit,
    --
    cs.location                          location_code,
    decode( cs.location,
       'LOCAL',       'Local Database',
       'REMOTE',      'Remote Database',
       'WEB_SOURCE',  'Web Source' )     location,
    cs.query_owner                       table_owner,
    cs.query_table                       table_name,
    cs.query_where                       where_clause,
    cs.query_order_by_type               order_by_type_code,
    case cs.query_order_by_type
      when 'STATIC' then 'Static Value'
      when 'ITEM'   then 'Item'
    end                                  order_by_type,
    json_value(
        cs.query_order_by,
        '$.itemName' )                   order_by_item,
    cs.query_order_by                    order_by_clause,
    decode( cs.source_post_processing,
       'SQL', 'SQL Query',
       'WHERE_ORDER_BY_CLAUSE', 'Where/Order By Clause',
       'FUNC_BODY_RETURNING_SQL', 'Function Body returning SQL Query' ) source_post_processing,
    decode(cs.include_rowid_column,
    'Y','Yes',
    'N','No',
    cs.include_rowid_column)             include_rowid_column,
    cs.optimizer_hint,
    --
    (select rs.name
       from wwv_remote_servers rs
      where rs.id                = cs.remote_server_id
        and rs.security_group_id = cs.security_group_id)    remote_database_name,
    cs.remote_server_id                  remote_database_id,
    (select wm.name
       from wwv_flow_web_src_modules wm
      where wm.id                = cs.web_src_module_id
        and wm.security_group_id = cs.security_group_id)   web_source_module_name,
    cs.web_src_module_id                 web_source_module_id,
    case cs.use_local_sync_table
        when 'Y' then 'Yes'
        when 'N' then 'No' end           use_local_sync_table,
    --
    cs.external_filter_expr,
    cs.external_order_by_type            external_order_by_type_code,
    case cs.external_order_by_type
      when 'STATIC' then 'Static Value'
      when 'ITEM'   then 'Item'
    end                                  external_order_by_type,
    json_value(
        cs.external_order_by_expr,
        '$.itemName' )                   external_order_by_item,
    cs.external_order_by_expr,
    --
    decode(cs.series_type,
      'area','Area',
      'auto','Automatic',
      'areaRange','Area Range',
      'bar','Bar',
      'barRange','Bar Range',
      'line','Line',
      'lineWithArea','Line with Area',
      'candlestick','Candlestick',
      'boxPlot', 'Box Plot',
      'gantt','Gantt',
    cs.series_type)                       series_type,
    cs.series_name_column_name            series_name_column_name,
    cs.items_value_column_name            items_value_column_name,
    cs.items_label_column_name            items_label_column_name,
    cs.items_short_desc_column_name       items_short_desc_column_name,
    cs.items_low_column_name              items_low_column_name,
    cs.items_high_column_name             items_high_column_name,
    cs.items_open_column_name             items_open_column_name,
    cs.items_close_column_name            items_close_column_name,
    cs.items_volume_column_name           items_volume_column_name,
    cs.items_x_column_name                items_x_column_name,
    cs.items_y_column_name                items_y_column_name,
    cs.items_z_column_name                items_z_column_name,
    cs.items_target_value                 items_target_value,
    cs.items_min_value                    items_min_value,
    cs.items_max_value                    items_max_value,
    cs.group_name_column_name             group_name_column_name,
    cs.group_short_desc_column_name       group_short_desc_column_name,
    cs.custom_column_name                 custom_column_name,
    --
    cs.aggregate_function                 aggregate_function,
    --
    cs.color                              color,
    cs.q2_color                           q2_color,
    cs.q3_color                           q3_color,
    decode(cs.line_style,
      'dashed','Dashed',
      'dotted','Dotted',
      'solid','Solid',
    cs.line_style)                        line_style,
    cs.line_width                         line_width,
    decode(cs.line_type,
      'auto','Automatic',
      'centeredStepped','Centered Stepped',
      'centeredSegmented','Centered Segmented',
      'curved','Curved',
      'segmented','Segmented',
      'stepped','Stepped',
      'straight','Straight',
      'none','None',
    cs.line_type)                         line_type,
    --
    decode(cs.marker_rendered,
    'auto','Automatic',
    'on','On',
    'off','Off',
    cs.marker_rendered)                   marker_rendered,
    decode(cs.marker_shape,
      'auto','Automatic',
      'circle','Circle',
      'diamond','Diamond',
      'plus','Plus',
      'triangleDown','Triangle Down',
      'triangleUp','Triangle Up',
      'human','Human',
      'square','Square',
    cs.marker_shape)                      marker_shape,
    cs.assigned_to_y2                     assigned_to_y2,
    cs.stack_category                     stack_category,
    --
    decode(cs.items_label_rendered,
    'Y','Yes',
    'N','No',
    cs.items_label_rendered)              items_label_rendered,
    decode(cs.items_label_position,
      'center','Center',
      'aboveMarker','Above Marker',
      'belowMarker','Below Marker',
      'beforeMarker','Before Marker',
      'afterMarker','After Marker',
      'insideBarEdge','Inside Bar Edge',
      'outsideBarEdge','Outside Bar Edge',
      'auto','Automatic',
    cs.items_label_position)              items_label_position,
    decode(cs.items_label_display_as,
           'LABEL', 'Label',
           'VALUE', 'Value',
           'PERCENT','Percentage Value',
           'ALL','All',
           'COMBO', 'Combination',
           'LBL_VAL','Label - Value',
           cs.items_label_display_as)     items_label_display_as,
    cs.items_label_css_classes            items_label_css_classes,
    cs.items_label_font_family            items_label_font_family,
    decode (cs.items_label_font_style,
    'normal','Normal',
    'italic','Italic',
    'oblique','Oblique',
    cs.items_label_font_style)            items_label_font_style,
    cs.items_label_font_size              items_label_font_size,
    cs.items_label_font_color             items_label_font_color,
    --
    cs.gantt_start_date_source,
    cs.gantt_start_date_column,
    cs.gantt_start_date_item,
    cs.gantt_end_date_source,
    cs.gantt_end_date_column,
    cs.gantt_end_date_item,
    cs.gantt_row_id,
    cs.gantt_row_name,
    cs.gantt_task_id,
    cs.gantt_task_name,
    cs.gantt_task_start_date,
    cs.gantt_task_end_date,
    cs.gantt_task_css_style,
    cs.gantt_task_css_class,
    cs.gantt_predecessor_task_id,
    cs.gantt_successor_task_id,
    cs.gantt_baseline_start_column,
    cs.gantt_baseline_end_column,
    cs.gantt_baseline_css_class,
    cs.gantt_progress_column,
    cs.gantt_progress_css_class,
    cs.gantt_viewport_start_source,
    cs.gantt_viewport_start_column,
    cs.gantt_viewport_start_item,
    cs.gantt_viewport_end_source,
    cs.gantt_viewport_end_column,
    cs.gantt_viewport_end_item,
    decode (cs.task_label_position,
       'end','End',
       'innerCenter', 'Inner Center',
       'innerEnd','Inner End',
       'innerStart','Inner Start',
       'none','None',
       'start','Start',
       cs.task_label_position)            task_label_position,
    --
    cs.gauge_plot_area_color              gauge_plot_area_color,
    cs.threshold_values                   threshold_values,
    cs.threshold_colors                   threshold_colors,
    decode (cs.threshold_display,
      'all', 'All thresholds displayed in plot area',
      'currentOnly', 'Current threshold displayed in plot area',
      'onIndicator','Current threshold displayed on gauge indicator',
    cs.threshold_display)                 threshold_display,
    cs.reference_line_values              reference_line_values,
    cs.reference_line_colors              reference_line_colors,
    --
    cs.link_target                        link_target,
    decode(cs.link_target_type,
      'REDIRECT_URL','URL',
      'REDIRECT_PAGE','Page in this application',
      'REDIRECT_APP','Page in a different application',
      'DEFINED_BY_DA','Defined by a Dynamic Action',
    cs.link_target_type)                  link_target_type,
    --
    decode(substr(cs.security_scheme,1,1),'!','Not ')||
      nvl((select ss.name
             from wwv_flow_security_schemes ss
            where to_char(ss.id)       = ltrim(cs.security_scheme,'!')
              and ss.flow_id           = f.application_id
              and ss.security_group_id = cs.security_group_id),
    cs.security_scheme)                   authorization_scheme,
    cs.security_scheme                    authorization_scheme_id,
    (select case
              when cs.required_patch > 0
                then pa.PATCH_NAME
              else '{Not '||pa.PATCH_NAME||'}'
            end PATCH_NAME
       from wwv_flow_patches pa
      where pa.id                = abs(cs.REQUIRED_PATCH)
        and pa.security_group_id = cs.security_group_id) build_option,
    cs.required_patch                     build_option_id,
    --
    cs.series_comment                     series_comment,
    nvl((select r
           from apex_standard_conditions
          where d = cs.DISPLAY_WHEN_COND_TYPE),
    cs.DISPLAY_WHEN_COND_TYPE)            condition_type,
    cs.display_when_cond_type             condition_type_code,
    cs.DISPLAY_WHEN_CONDITION             condition_expression1,
    cs.DISPLAY_WHEN_CONDITION2            condition_expression2,
    cs.updated_on                         last_updated_on,
    cs.updated_by                         last_updated_by,
    --
    cs.name
    ||' seq='||lpad(cs.seq,5,'00000')
    ||' q='||sys.dbms_lob.substr(cs.data_source,20,1)||'.'||sys.dbms_lob.getlength(cs.data_source)
    ||' max='||cs.max_row_count
    ||' link='||substr(cs.link_target,1,20) component_signature
from wwv_flow_jet_chart_series cs,
     wwv_flow_jet_charts       c,
     wwv_flow_authorized       f
where f.workspace_id      = c.security_group_id
  and f.application_id    = c.flow_id
  and c.security_group_id = cs.security_group_id
  and c.flow_id           = cs.flow_id
  and c.id                = cs.chart_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_chart_s',
    p_table_comment     => 'Identifies a chart series associated with a chart on a Page and Region.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas.',
      'workspace_display_name'        , 'Display name for the workspace.',
      'application_id'                , 'Application Primary Key, Unique over all workspaces.',
      'application_name'              , 'Identifies the application.',
      'page_id'                       , 'ID of the application page.',
      'page_name'                     , 'Name of the application page.',
      'region_id'                     , 'Identifies the Page Region foreign key to the apex_application_page_regions view.',
      'region_name'                   , 'Identifies the region name in which this chart series is displayed.',
      'chart_id'                      , 'Foreign Key of the chart.',
      'series_id'                     , 'Primary Key of the chart series',
      'series_seq'                    , 'Identifies the series sequence to determine the order of evaluation',
      'series_name'                   , 'Chart series name',
      'data_source'                   , 'The data source that will return the data to display the chart series',
      'data_source_type'              , 'The data source type',
      'function_body_language'        , 'Identifies the programming language of DATA_SOURCE.',
      'max_row_count'                 , 'Defines the maximum number of rows to query.',
      'items_to_submit'               , 'Identifies the page items which should be submitted when an Ajax call is performed for the chart series.',
      'location_code'                 , 'Identifies the internal code of LOCATION',
      'location'                      , 'Identifies whether the data source of the region is the local database, a REST Enabled SQL service or a REST Data Source',
      'table_owner'                   , 'Denotes the table owner for TABLE query types',
      'table_name'                    , 'Denotes the table name for TABLE query types',
      'where_clause'                  , 'Identifies the WHERE clause provided by the developer and to be applied to TABLE or REST Data Source.',
      'order_by_type_code'            , 'Identifies the internal code of ORDER_BY_TYPE',
      'order_by_type'                 , 'Identifies whether the data source is sorted by a static order by clause or derived from a page item.',
      'order_by_item'                 , 'Identifies the page item which is used to derive the current order by clause.',
      'order_by_clause'               , 'Identifies the ORDER BY clause provided by the developer and to be applied to TABLE or REST Data Source',
      'source_post_processing'        , 'Identifies type of SQL post processing for data from external data sources',
      'include_rowid_column'          , 'Identifies if the pseudo column ROWID should be selected',
      'optimizer_hint'                , 'Optimizer hint which is added to the region source SQL statement when it is executed',
      'remote_database_id'            , 'Identifies the primary key of the REST Enabled SQL definition',
      'remote_database_name'          , 'Identifies the name of the REST Enabled SQL service',
      'web_source_module_id'          , 'Identifies the primary key of the REST Data Source',
      'web_source_module_name'        , 'Identifies the REST Data Source name',
      'use_local_sync_table'          , 'Whether to use the local synchronization table instead of the REST service',
      'external_filter_expr'          , 'Filter expression to be sent to the external data source.',
      'external_order_by_type_code'   , 'Identifies the internal code of EXTERNAL_ORDER_BY_TYPE',
      'external_order_by_type'        , 'Identifies whether the external data source is sorted by a static order by clause or derived from a page item.',
      'external_order_by_item'        , 'Identifies the page item which is used to derive the current external order by clause.',
      'external_order_by_expr'        , 'Order By expression to be sent to the external data source.',
      'series_type'                   , 'Chart series type: Bar, Area, Line, Line with Area, Candlestick, Bar Range, Area Range, or Automatic, which will automatically set the series type based upon the chosen chart type',
      'series_name_column_name'       , 'Identifies the "Series Name" column from a multi-series query',
      'items_value_column_name'       , 'Depending on the chosen chart type, identifies the "Value" column for the chart.  Only for Bar, Line, Area, Combination, Pie, Funnel,Pyramid, and Dial Gauge charts.',
      'items_label_column_name'       , 'Identifies the "Label" column for the chart, for those chart types that do not use the "Groups" attribute in Oracle JET Charts.',
      'items_short_desc_column_name'  , 'Identifies the column used to display a custom tooltip on the data point(s) of the chart.',
      'items_low_column_name'         , 'Identifies the column used to display the low value on a Bar Range, Area Range, or Candlestick chart.',
      'items_high_column_name'        , 'Identifies the column used to display the high value on a Bar Range, Area Range, or Candlestick chart.',
      'items_open_column_name'        , 'Identifies the column used to display the opening value on a Stock chart.',
      'items_close_column_name'       , 'Identifies the column used to display the closing value on a Stock chart.',
      'items_volume_column_name'      , 'Identifies the column used to display the volume value on a Stock chart.',
      'items_x_column_name'           , 'Identifies the column used to display the X value on a Bubble or Scatter chart.',
      'items_y_column_name'           , 'Identifies the column used to display the Y value on a Bubble or Scatter chart.',
      'items_z_column_name'           , 'Identifies the column used to display the Z value on a Bubble or Scatter chart.',
      'items_target_value'            , 'Identifies the column used to display the target value on a Funnel chart.',
      'items_min_value'               , 'Identifies the column used to display the minimum value on a Dial Gauge chart.',
      'items_max_value'               , 'Identifies the column used to display the maximum value on a Dial Gauge chart.',
      'group_name_column_name'        , 'Identifies the column used to display a tooltip on the series name(s) displayed on the chart legend.',
      'group_short_desc_column_name'  , 'Identifies the column used to display a tooltip on the x-axis label(s) of the chart, for accessibility purposes.',
      'custom_column_name'            , 'Identifies the custom column used to display a more detailed tooltip on the chart or custom colors based on a database value.',
      'aggregate_function'            , 'Identifies the aggregate function to apply to the value column before rendering the chart.',
      'color'                         , 'Identifies the color to be used to represent the items of the chart series.',
      'q2_color'                      , 'Identifies the color to be used to represent the q2 segment of a Box Plot chart series.',
      'q3_color'                      , 'Identifies the color to be used to represent the q3 segment of a Box Plot chart series.',
      'line_style'                    , 'Identifies the style of the line to be rendered on the chart.',
      'line_width'                    , 'Identifies the width of the line to be rendered on the chart.',
      'line_type'                     , 'Identifies the type of line to be rendered on the chart.',
      'marker_rendered'               , 'Identifies whether the marker is to be rendered on the chart.',
      'marker_shape'                  , 'Identifies the shape of the marker displayed on the chart.',
      'assigned_to_y2'                , 'Identifies whether the series is assigned to the extra Y axis on the chart.',
      'stack_category'                , 'Identifies the stack category of the series, used to group stacked elements of the chart.',
      'items_label_rendered'          , 'Identifies whether the label is rendered on a chart item.',
      'items_label_position'          , 'Identifies the position of the label on a chart item.',
      'items_label_display_as'        , 'Identifies the format of the label on the pie chart item',
      'items_label_css_classes'       , 'Identifies the CSS classes used to display the label on a chart item.',
      'items_label_font_family'       , 'Defines the font family of the label on a chart item.',
      'items_label_font_style'        , 'Defines the font style of the label on a chart item.',
      'items_label_font_size'         , 'Defines the font size of the label on a chart item.',
      'items_label_font_color'        , 'Defines the color of the label on a chart item.',
      'gantt_start_date_source'       , 'Identifies the data source for the Gantt chart start date.',
      'gantt_start_date_column'       , 'Identifies the column used to display the Gantt chart start date.',
      'gantt_start_date_item'         , 'Identifies the page item used to define the Gantt chart start date.',
      'gantt_end_date_source'         , 'Identifies the data source for the Gantt chart end date.',
      'gantt_end_date_column'         , 'Identifies the column used to display the Gantt chart end date.',
      'gantt_end_date_item'           , 'Identifies the page item used to define the Gantt chart end date.',
      'gantt_row_id'                  , 'Identifies the column used to display the row ID on the gantt chart.',
      'gantt_row_name'                , 'Identifies the column used to display the row name on the gantt chart.',
      'gantt_task_id'                 , 'Identifies the column used to display the task ID on the gantt chart.',
      'gantt_task_name'               , 'Identifies the column used to display the task name on the gantt chart.',
      'gantt_task_start_date'         , 'Identifies the column used to display the task start date on the gantt chart.',
      'gantt_task_end_date'           , 'Identifies the column used to display the task end date on the gantt chart.',
      'gantt_task_css_style'          , 'Identifies the column used to display the task css style on the gantt chart.',
      'gantt_task_css_class'          , 'Identifies the CSS class used to style the task bar on the gantt chart.',
      'gantt_predecessor_task_id'     , 'Identifies the column used to display the predecessor task ID on the gantt chart.',
      'gantt_successor_task_id'       , 'Identifies the column used to display the successor task ID on the gantt chart.',
      'gantt_baseline_start_column'   , 'Identifies the column used to display the baseline start date on the gantt chart.',
      'gantt_baseline_end_column'     , 'Identifies the column used to display the baseline end date on the gantt chart.',
      'gantt_baseline_css_class'      , 'Identifies the CSS class used to style the baseline bar on the gantt chart.',
      'gantt_progress_column'         , 'Identifies the column used to display the progress value on the gantt chart.',
      'gantt_progress_css_class'      , 'Identifies the CSS class used to style the progress bar on the gantt chart.',
      'gantt_viewport_start_source'   , 'Identifies the data source for the Gantt chart viewport start date.',
      'gantt_viewport_start_column'   , 'Identifies the column used to display the viewport start date on the gantt chart.',
      'gantt_viewport_start_item'     , 'Identifies the page item used to define the Gantt chart viewport start date.',
      'gantt_viewport_end_source'     , 'Identifies the data source for the Gantt chart viewport end date.',
      'gantt_viewport_end_column'     , 'Identifies the column used to display the viewport end date on the gantt chart.',
      'gantt_viewport_end_item'       , 'Identifies the page item used to define the Gantt chart viewport end date.',
      'task_label_position'           , 'Specifies the position of the task label on the gantt chart.',
      'gauge_plot_area_color'         , 'Identifies the color to be used to represent the plot area of a Status Meter Gauge chart.',
      'threshold_values'              , 'Identifies the values to be used to define thresholds on a Status Meter Gauge chart.',
      'threshold_colors'              , 'Identifies the colors to be used to represent thresholds on a Status Meter Gauge chart.',
      'threshold_display'             , 'Identifies the location on a Status Meter Gauge chart where threshold colors are to be rendered.',
      'reference_line_values'         , 'Identifies the values to be used to define reference lines on a Status Meter Gauge chart.',
      'reference_line_colors'         , 'Identifies the colors to be used to represent reference lines on a Status Meter Gauge chart.',
      'link_target'                   , 'Action link for the chart series, used when chart item selected.',
      'link_target_type'              , 'Defines the type of Action link for the chart series is to be used.',
      'authorization_scheme'          , 'Identifies an Authorization Scheme that will be applied to the chart series query',
      'authorization_scheme_id'       , 'Identifies an Authorization Scheme foreign key',
      'build_option'                  , 'Chart Series query will be processed if the Build Option is enabled',
      'build_option_id'               , 'Identifies the primary key of this component',
      'series_comment'                , 'Identifies the series comment.',
      'condition_type'                , 'Identifies the condition type that must be met in order for the chart series to be rendered or processed.',
      'condition_type_code'           , 'Identifies the condition type code that must be met in order for the chart series to be rendered or processed.',
      'condition_expression1'         , 'Identifies the attribute that allows you to conditionally control whether or not a component is rendered or processed.',
      'condition_expression2'         , 'Identifies the second attribute that allows you to conditionally control whether or not a component is rendered or processed.',
      'last_updated_by'               , 'APEX developer who made last update.',
      'last_updated_on'               , 'Date of last update.',
      'component_signature'           , 'Identifies attributes defined at a given component level to facilitate application comparisons.'
    )
  );
end;
/

prompt ...apex_application_page_chart_a

create or replace view apex_application_page_chart_a
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    c.page_id                            page_id,
    (select st.name
       from wwv_flow_steps st
      where st.id                = c.page_id
        and st.flow_id           = c.flow_id
        and st.security_group_id = c.security_group_id) page_name,
    --
    ca.chart_id                          chart_id,
    ca.static_id                         static_id,
    ca.id                                axis_id,
    decode(ca.axis,
    'x','X Axis',
    'y','Y Axis',
    'y2','Extra Y Axis',
    'major','Major Axis',
    'minor','Minor Axis',
    ca.axis)                             axis,
    ca.title                             axis_title,
    ca.title_font_family                 axis_title_font_family,
    decode (ca.title_font_style,
    'normal','Normal',
    'italic','Italic',
    'oblique','Oblique',
    ca.title_font_style)                 axis_title_font_style,
    ca.title_font_size                   axis_title_font_size,
    ca.title_font_color                  axis_title_font_color,
    decode(ca.is_rendered,
    'on','On',
    'off','Off',
    ca.is_rendered)                      is_rendered,
    --
    ca.min                               axis_minimum,
    ca.max                               axis_maximum,
    decode(ca.format_type,
    'date-short', 'Short Date format',
    'date-medium', 'Medium Date format',
    'date-long', 'Long Date format',
    'date-full', 'Full Date format',
    'time-short', 'Short Time format',
    'time-medium', 'Medium Time format',
    'time-long', 'Long Time format',
    'time-full', 'Full Time format',
    'datetime-short', 'Short Datetime format',
    'datetime-medium', 'Medium Datetime format',
    'datetime-long','Long Datetime format',
    'datetime-full','Full Datetime format',
    'decimal','Decimal',
    'currency','Currency',
    'percent','Percent',
    ca.format_type)                     axis_format_type,
    ca.decimal_places                   axis_decimal_places,
    ca.currency                         axis_currency,
    ca.numeric_pattern                  axis_numeric_pattern,
    decode(ca.format_scaling,
    'auto','Automatic',
    'none','None',
    'thousand','Thousand',
    'million','Million',
    'billion','Billion',
    'trillion','Trillion',
    'quadrillion','Quadrillion',
    ca.format_scaling)                  axis_format_scaling,
    decode(ca.scaling,
    'linear','Linear',
    'log','Log',
    ca.scaling)                         axis_scaling,
    decode(ca.baseline_scaling,
    'min','Minimum',
    'zero','Zero',
    ca.baseline_scaling)                axis_baseline_scaling,
    ca.step                             axis_step,
    decode(ca.position,
    'start','Start',
    'end','End',
    'top','Top',
    'bottom','Bottom',
    'auto','Automatic',
    ca.position)                        axis_position,
    decode(ca.major_tick_rendered,
    'auto','Automatic',
    'on','On',
    'off','Off',
    ca.major_tick_rendered)             axis_major_tick_rendered,
    ca.min_step                         axis_minimum_step,
    decode(ca.minor_tick_rendered,
    'auto','Automatic',
    'on','On',
    'off','Off',
    ca.minor_tick_rendered)             axis_minor_tick_rendered,
    ca.minor_step                       axis_minor_step,
    decode(ca.tick_label_rendered,
    'auto','Automatic',
    'on','On',
    'off','Off',
    ca.tick_label_rendered)             axis_tick_label_rendered,
    decode(ca.tick_label_rotation,
    'auto','Automatic',
    'none','None',
    ca.tick_label_rotation)             axis_tick_label_rotation,
    decode(ca.tick_label_position,
    'inside','Inside',
    'outside','Outside',
    ca.tick_label_position)             axis_tick_label_position,
    ca.tick_label_font_family           axis_tick_label_font_family,
    decode (ca.tick_label_font_style,
    'normal','Normal',
    'italic','Italic',
    'oblique','Oblique',
    ca.tick_label_font_style)           axis_tick_label_font_style,
    ca.tick_label_font_size             axis_tick_label_font_size,
    ca.tick_label_font_color            axis_tick_label_font_color,
    decode(ca.split_dual_y,
    'auto','Automatic',
    'on','On',
    'off','Off',
    ca.split_dual_y)                    y2_axis_split_dual,
    ca.splitter_position                y2_axis_splitter_position,
    --
    decode (ca.axis_scale,
      'seconds','Seconds',
      'minutes','Minutes',
      'hours','Hours',
      'days','Days',
      'weeks','Weeks',
      'months','Months',
      'quarters','Quarters',
      'years','Years',
      ca.axis_scale)                    axis_scale,
    decode(ca.zoom_order_seconds,
      'Y','Yes',
      'N','No',
      ca.zoom_order_seconds)            zoom_order_seconds,
    decode(ca.zoom_order_minutes,
      'Y','Yes',
      'N','No',
      ca.zoom_order_minutes)            zoom_order_minutes,
    decode(ca.zoom_order_hours,
      'Y','Yes',
      'N','No',
      ca.zoom_order_hours)              zoom_order_hours,
    decode(ca.zoom_order_days,
      'Y','Yes',
      'N','No',
      ca.zoom_order_days)               zoom_order_days,
    decode(ca.zoom_order_weeks,
      'Y','Yes',
      'N','No',
      ca.zoom_order_weeks)              zoom_order_weeks,
    decode(ca.zoom_order_months,
      'Y','Yes',
      'N','No',
      ca.zoom_order_months)             zoom_order_months,
    decode(ca.zoom_order_quarters,
      'Y','Yes',
      'N','No',
      ca.zoom_order_quarters)           zoom_order_quarters,
    decode(ca.zoom_order_years,
      'Y','Yes',
      'N','No',
      ca.zoom_order_years)              zoom_order_years,
    --
    ca.updated_on                       last_updated_on,
    ca.updated_by                       last_updated_by,
    --
    ca.title
    ||' axis='||ca.axis
    ||' max='||ca.max
    ||' min='||ca.min
    ||' position='||ca.position
    ||' format_type='||ca.format_type component_signature
from wwv_flow_jet_chart_axes ca,
     wwv_flow_jet_charts     c,
     wwv_flow_authorized     f
where f.workspace_id      = c.security_group_id
  and f.application_id    = c.flow_id
  and c.security_group_id = ca.security_group_id
  and c.flow_id           = ca.flow_id
  and c.id                = ca.chart_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_chart_a',
    p_table_comment     => 'Identifies a chart axis associated with a chart on a Page and Region.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas.',
      'workspace_display_name'        , 'Display name for the workspace.',
      'application_id'                , 'Application Primary Key, Unique over all workspaces.',
      'application_name'              , 'Identifies the application.',
      'page_id'                       , 'ID of the application page.',
      'page_name'                     , 'Name of the application page.',
      'chart_id'                      , 'Foreign Key of the chart.',
      'axis_id'                       , 'Primary Key of the chart axis.',
      'axis'                          , 'The chart axis, whether it is the X, Y, Extra Y, Major or Minor axis of the chart. The major and minor axes only apply to a Gantt chart.',
      'axis_title'                    , 'The title of the chart axis.',
      'axis_title_font_family'        , 'Defines the font family of the axis title.',
      'axis_title_font_style'         , 'Defines the font style of the axis title.',
      'axis_title_font_size'          , 'Defines the font size of the axis title.',
      'axis_title_font_color'         , 'Defines the color of the axis title.',
      'is_rendered'                   , 'Identifies whether the chart axis is rendered on the chart.',
      'axis_minimum'                  , 'Identifies the minimum value of the chart axis.',
      'axis_maximum'                  , 'Identifies the maximum value of the chart axis.',
      'axis_format_type'              , 'The format type to be used to represent the axis values.',
      'axis_decimal_places'           , 'Number of decimal places to use in axis values.',
      'axis_currency'                 , 'The ISO 4217 alphabetic currency code used to represent monetary axis values.',
      'axis_numeric_pattern'          , 'The numeric pattern used to represent numeric axis values.',
      'axis_format_scaling'           , 'The scaling behavior of the values on the axis.',
      'axis_scaling'                  , 'The scaling used to represent numeric axis values.',
      'axis_baseline_scaling'         , 'Defines whether the axis baseline starts at the minimum value of the data or at zero. Only applies to numerical data axes.',
      'axis_step'                     , 'The increment between major tick marks. Defaults to null for automatic calculation based on the data. Only applies to time and numerical axes.',
      'axis_position'                 , 'The position of the axis relative to its content. For vertical charts, only start and end apply. For horizontal charts, only top and bottom apply.',
      'axis_major_tick_rendered'      , 'Defines whether the major tick marks are rendered.',
      'axis_minimum_step'             , 'The minimum increment between major tick marks. This is typically used to prevent fractional axis values for discrete measures.',
      'axis_minor_tick_rendered'      , 'Defines whether the minor tick marks are rendered.',
      'axis_minor_step'               , 'The increment between minor tick marks. Defaults to null for automatic calculation based on the data.',
      'axis_tick_label_rendered'      , 'Defines whether the tick labels are rendered.',
      'axis_tick_label_rotation'      , 'Defines whether the chart will automatically rotate the labels by 90 degrees in order to fit more labels on the axis. The rotation will only be applied to categorical labels for a horizontal axis.',
      'axis_tick_label_position'      , 'Defines the position of the tick labels relative to the plot area.',
      'axis_tick_label_font_family'   , 'Defines the font family of the tick labels.',
      'axis_tick_label_font_style'    , 'Defines the font style of the tick labels.',
      'axis_tick_label_font_size'     , 'Defines the font size of the tick labels.',
      'axis_tick_label_font_color'    , 'Defines the color of the tick labels.',
      'y2_axis_split_dual'            , 'Defines whether the plot area is split into two sections, so that sets of data assigned to the different Y-axes appear in different parts of the plot area. Stock charts do not support "off".',
      'y2_axis_splitter_position'     , 'In a split dual-Y chart, specifies the fraction of the space that is given to the Y-axis subchart. Valid values are numbers from 0 to 1.',
      'axis_scale'                    , 'Identifies the type of scale to be applied to the gantt chart',
      'zoom_order_seconds'            , 'Identifies whether the gantt chart can zoom to seconds time interval',
      'zoom_order_minutes'            , 'Identifies whether the gantt chart can zoom to minutes time interval',
      'zoom_order_hours'              , 'Identifies whether the gantt chart can zoom to hours time interval',
      'zoom_order_days'               , 'Identifies whether the gantt chart can zoom to days time interval',
      'zoom_order_weeks'              , 'Identifies whether the gantt chart can zoom to weeks time interval',
      'zoom_order_months'             , 'Identifies whether the gantt chart can zoom to months time interval',
      'zoom_order_quarters'           , 'Identifies whether the gantt chart can zoom to quarters time interval',
      'zoom_order_years'              , 'Identifies whether the gantt chart can zoom to years time interval',
      'last_updated_by'               , 'APEX developer who made last update.',
      'last_updated_on'               , 'Date of last update.',
      'component_signature'           , 'Identifies attributes defined at a given component level to facilitate application comparisons.',
      'STATIC_ID'                     , 'Static identifier of the chart axis. Useful when developing custom JavaScript behavior.'
    )
  );
end;
/

--
-- Interactive Report Views
--
prompt ...apex_application_page_ir
create or replace view apex_application_page_ir
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
ir.page_id                   page_id,
ir.id                        interactive_report_id,
ir.region_id                 region_id,
r.plug_name                  region_name,
--
(select count(*) from wwv_flow_worksheet_columns wc where wc.worksheet_id = ir.id and wc.security_group_id = ir.security_group_id) number_of_columns,
(select count(*) from wwv_flow_worksheet_col_groups cg where cg.worksheet_id = ir.id and cg.security_group_id = ir.security_group_id) number_of_column_groups,
(select count(*) from wwv_flow_worksheet_rpts ws where ws.worksheet_id = ir.id and ws.session_id is null and nvl(ws.is_default,'N')='Y' and ws.application_user='APXWS_ALTERNATIVE' and ws.security_group_id = ir.security_group_id) number_of_alt_default_reportS,
(select count(*) from wwv_flow_worksheet_rpts ws where ws.worksheet_id = ir.id and ws.session_id is null and nvl(ws.is_default,'N')='N' and ws.status='PUBLIC' and ws.security_group_id = ir.security_group_id) number_of_public_reports,
(select count(*) from wwv_flow_worksheet_rpts ws where ws.worksheet_id = ir.id and ws.session_id is null and nvl(ws.is_default,'N')='N' and ws.status='PRIVATE' and ws.security_group_id = ir.security_group_id) number_of_private_reports,
ir.max_row_count             ,
ir.max_row_count_message     ,
ir.no_data_found_message     ,
ir.max_rows_per_page         ,
ir.search_button_label       ,
r.ajax_items_to_submit       page_items_to_submit, /* Only for compatibility */
ir.base_table_or_view        ,
r.plug_source                sql_query,
--
ir.show_nulls_as             ,
decode(ir.pagination_type,
   'ROWS_X_TO_Y_OF_Z','Row Ranges X to Y of Z (no pagination)',
   'ROWS_X_TO_Y','Row Ranges X to Y (no pagination)',
   'SEARCH_ENGINE','Search Engine 1,2,3,4 (set based pagination)',
   'COMPUTED_BUT_NOT_DISPLAYED','Use Externally Created Pagination Buttons',
   'ROW_RANGES','Row Ranges 1-15 16-30 (with set pagination)',
   'ROW_RANGES_IN_SELECT_LIST','Row Ranges 1-15 16-30 in select list (with pagination)',
   'ROW_RANGES_WITH_LINKS','Row Ranges X to Y of Z (with pagination)',
   'NEXT_PREVIOUS_LINKS','Row Ranges X to Y (with next and previous links)',
   ir.pagination_type)       pagination_scheme,
decode(ir.pagination_display_position,
  'BOTTOM_LEFT','Bottom - Left',
  'BOTTOM_RIGHT','Bottom - Right',
  'TOP_LEFT','Top - Left',
  'TOP_RIGHT','Top - Right',
  'TOP_AND_BOTTOM_LEFT','Top and Bottom - Left',
  'TOP_AND_BOTTOM_RIGHT','Top and Bottom - Right',
  ir.pagination_display_position)       pagination_display_position,
  cast(null as varchar2(10))            button_template,
--
decode(ir.show_finder_drop_down ,'Y','Yes','N','No',ir.show_finder_drop_down ) show_finder_drop_down ,
decode(ir.report_list_mode,'TABS', 'Yes','No') show_reports_select_list,
decode(ir.show_display_row_count,'Y','Yes','N','No',ir.show_display_row_count) show_display_row_count,
decode(ir.show_search_bar       ,'Y','Yes','N','No',ir.show_search_bar       ) show_search_bar       ,
decode(ir.show_search_textbox   ,'Y','Yes','N','No',ir.show_search_textbox   ) show_search_textbox   ,
decode(ir.show_actions_menu     ,'Y','Yes','N','No',ir.show_actions_menu     ) show_actions_menu     ,
cast(null as varchar2(10))  actions_menu_icon,
cast(null as varchar2(10))  finder_icon,
decode(ir.show_select_columns,'Y','Yes','N','No',ir.show_select_columns) show_select_columns,
decode(ir.show_rows_per_page ,'Y','Yes','N','No',ir.show_rows_per_page)  show_rows_per_page,
decode(ir.show_filter        ,'Y','Yes','N','No',ir.show_filter        ) show_filter        ,
decode(ir.show_sort          ,'Y','Yes','N','No',ir.show_sort          ) show_sort          ,
decode(ir.show_control_break ,'Y','Yes','N','No',ir.show_control_break ) show_control_break ,
decode(ir.show_highlight     ,'Y','Yes','N','No',ir.show_highlight     ) show_highlight     ,
decode(ir.show_computation   ,'Y','Yes','N','No',ir.show_computation   ) show_compute       ,
decode(ir.show_aggregate     ,'Y','Yes','N','No',ir.show_aggregate     ) show_aggregate     ,
decode(ir.show_chart         ,'Y','Yes','N','No',ir.show_chart         ) show_chart         ,
decode(ir.show_group_by      ,'Y','Yes','N','No',ir.show_group_by      ) show_group_by      ,
decode(ir.show_pivot         ,'Y','Yes','N','No',ir.show_pivot         ) show_pivot         ,
decode(ir.show_notify        ,'Y','Yes','N','No',ir.show_notify        ) show_notify        ,
decode(ir.show_flashback     ,'Y','Yes','N','No',ir.show_flashback     ) show_flashback     ,
decode(ir.allow_report_saving,'Y','Yes','N','No',ir.allow_report_saving) show_save          ,
decode(ir.allow_save_rpt_public,'Y','Yes','N','No',ir.allow_save_rpt_public) show_save_public,
decode(substr(ir.save_rpt_public_auth_scheme,1,1),'!','Not ')||
nvl((select ss.name
  from wwv_flow_security_schemes ss
 where to_char(ss.id)       = ltrim(ir.save_rpt_public_auth_scheme,'!')
   and ss.flow_id           = f.application_id
   and ss.security_group_id = ir.security_group_id),
 ir.save_rpt_public_auth_scheme)  save_public_auth_scheme,
ir.save_rpt_public_auth_scheme    save_public_auth_scheme_id,
decode(ir.show_reset         ,'Y','Yes','N','No',ir.show_reset         ) show_reset         ,
decode(ir.show_download      ,'Y','Yes','N','No',ir.show_download      ) show_download      ,
decode(ir.show_help          ,'Y','Yes','N','No',ir.show_help          ) show_help          ,
--
case when substr( ir.download_auth_scheme, 1, 1) = '!' then 'Not ' end ||
coalesce (
    ( select ss.name
        from wwv_flow_security_schemes ss
       where to_char( ss.id )     = ltrim( ir.download_auth_scheme, '!' )
         and ss.flow_id           = f.application_id
         and ss.security_group_id = ir.security_group_id
    ),
    ir.download_auth_scheme ) as download_auth_scheme,
ir.download_auth_scheme       as download_auth_scheme_id,
ir.download_formats          ,
ir.download_filename         filename,
decode(ir.enable_mail_download ,'Y','Yes','N','No',ir.enable_mail_download ) send_as_email,
ir.csv_output_separator      separator,
ir.csv_output_enclosed_by    enclosed_by,
ir.supplemental_text         supplemental_text,
--
decode(ir.show_detail_link,
       'Y', 'Single Row View',
       'C', 'Custom Link target',
       'N', 'No Link Column') detail_link_type,
ir.detail_link                detail_link_target,
ir.detail_link_text           detail_link_text,
ir.detail_link_attr           detail_link_attributes,
ir.detail_link_checksum_type  detail_link_checksum_type,
nvl((select r from apex_standard_conditions where d = ir.detail_link_condition_type),ir.detail_link_condition_type)
                                     detail_link_condition_type,
ir.detail_link_condition_type detail_link_cond_type_code,
ir.detail_link_cond           detail_link_cond_expression,
ir.detail_link_cond2          detail_link_cond_expression2,
decode(substr(ir.detail_link_auth_SCHEME,1,1),'!','Not ')||
nvl((select ss.name
  from wwv_flow_security_schemes ss
 where to_char(ss.id)       = ltrim(ir.detail_link_auth_scheme,'!')
   and ss.flow_id           = f.application_id
   and ss.security_group_id = ir.security_group_id),
 ir.detail_link_auth_scheme)  detail_link_auth_scheme,
ir.detail_link_auth_scheme    detail_link_auth_scheme_id,
--
ir.alias                      alias,
ir.report_id_item             ,
ir.max_query_cost             ,
ir.email_from                 ,
ir.oracle_text_index_column   ,
decode(ir.fixed_header,
    'NONE',     'None',
    'PAGE',     'Page',
    'REGION',   'Region',
    ir.fixed_header) fixed_header,
ir.fixed_header_max_height,
--
decode(ir.lazy_loading,'Y','Yes','N','No','No') lazy_loading,
--
    decode(icon_view_enabled_yn,'Y','Yes','N','No','No') icon_view_enabled_yn,
    decode(icon_view_use_custom,'Y','Yes','N','No','No') icon_view_use_custom,
    icon_view_custom_link     ,
    icon_view_link_column     ,
    icon_view_img_src_column  ,
    icon_view_label_column    ,
    icon_view_img_attr_text   ,
    icon_view_alt_text        ,
    icon_view_title_text      ,
    icon_view_columns_per_row ,
    --
    decode(detail_view_enabled_yn,'Y','Yes','N','No','No') detail_view_enabled_yn,
    detail_view_before_rows,
    detail_view_for_each_row,
    detail_view_after_rows,
--
ir.internal_uid,
--
ir.created_on,
ir.created_by,
ir.updated_on,
ir.updated_by,
--
'Interactive Report'||
' rc='||ir.max_row_count||
' '||length(ir.max_row_count_message)||
length(ir.no_data_found_message)||
ir.max_rows_per_page||
substr(ir.search_button_label,1,30)||
length(r.ajax_items_to_submit)||
substr(ir.show_nulls_as,1,30)||
' p='||ir.pagination_type||
' '||ir.pagination_display_position||
' opt='||ir.show_finder_drop_down||
ir.show_display_row_count||
ir.show_search_bar||
ir.show_search_textbox||
ir.show_actions_menu||
ir.show_select_columns||
ir.show_rows_per_page||
ir.show_filter||
ir.show_sort||
ir.show_control_break||
ir.show_highlight||
ir.show_computation||
ir.show_aggregate||
ir.show_chart||
ir.show_group_by||
ir.show_pivot||
ir.show_notify||
ir.show_flashback||
ir.allow_report_saving||
ir.allow_save_rpt_public||
ir.save_rpt_public_auth_scheme||
ir.show_reset||
ir.show_download||
ir.download_auth_scheme||
ir.show_help||
' rpts='||ir.report_list_mode||
' dl='||substr(ir.download_formats,1,40)||
' fn='||substr(ir.download_filename,1,30)||
ir.csv_output_separator||
ir.csv_output_enclosed_by||
ir.show_detail_link||
' l='||substr(ir.detail_link,1,30)||
' lt='||substr(ir.detail_link_text,1,30)||
substr(ir.detail_link_attr,1,30)||
ir.detail_link_checksum_type||
' lc='||ir.detail_link_condition_type||
length(ir.detail_link_cond)||
length(ir.detail_link_cond2)||
nvl((select ss.name n from wwv_flow_security_schemes ss where to_char(ss.id) = ltrim(ir.detail_link_auth_SCHEME,'!') and ss.flow_id = f.application_id and ss.security_group_id = ir.security_group_id),'Not found')||
' a='||substr(ir.alias,1,30)||
ir.report_id_item||
ir.max_query_cost||
ir.email_from
component_signature
--
from wwv_flow_worksheets ir,
     wwv_flow_page_plugs r,
     wwv_flow_authorized f
where f.workspace_id       = ir.security_group_id
  and f.application_id     = ir.flow_id
  and ir.security_group_id = r.security_group_id(+)
  and ir.flow_id           = r.flow_id(+)
  and ir.region_id         = r.id(+)
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir',
    p_table_comment     => 'Identifies attributes of an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'            , 'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'application_id'       , 'Application Primary Key, Unique over all workspaces',
      'application_name'     , 'Identifies the application',
      'page_id'              , 'Identifies page number',
      'interactive_report_id'        , 'ID of the interactive report',
      'region_id'                    , 'ID of the interactive report region',
      'region_name'                  , 'Name of the interactive report region',
      'number_of_columns'            , 'Total number of columns returned by the query for this report region',
      'number_of_column_groups'      , 'Number of defined column groups for this report region',
      'number_of_alt_default_reports' , 'Number of developer-defined alternative default reports for this report region',
      'number_of_public_reports'      , 'Number of user-defined public shared reports for this report region',
      'number_of_private_reports'     , 'Number of user-defined private reports for this report region',

      'max_row_count'                , 'Maximum number of rows to query before sorting',
      'max_row_count_message'        , 'Message to display if the maximum number of rows is exceeded',
      'no_data_found_message'        , 'Message to display when no rows are returned by the report query',
      'max_rows_per_page'            , 'Maximum number of rows to display on a single HTML page.  The Rows select list will not display any values higher than this amount.',
      'search_button_label'          , 'Text to use for the search button label',
      'page_items_to_submit'         , 'List of page items values which are submitted when the search button is clicked',
      'base_table_or_view'           , 'Table or View used as the base for this interactive report region.',
      'sql_query'                    , 'Query used as the base for this interactive report region.  The original query will be preserved as a subquery in the actual SQL used for the report',
      'show_nulls_as'                , 'Identifies the text to display for null columns. The default value is "-".',
      'pagination_scheme'            , 'Identifies the pagination style to use for this report',
      'pagination_display_position'  , 'Identifies the position of pagination links relative to this report',
      'button_template'              , 'Obsolete.',
      'show_finder_drop_down'        , 'Determines whether to display the drop down to the left of the search field',
      'show_reports_select_list'     , 'Determines whether saved reports are displayed as select list in the Search Bar.',
      'show_display_row_count'       , 'Determines whether to display the row count selector in the Search Bar',
      'show_search_bar'              , 'Determines whether to display the Search Bar for the interactive report',
      'show_search_textbox'          , 'Determines whether to allow searching from the Search Bar',
      'show_actions_menu'            , 'Determines whether to display the Actions Menu and Actions Menu icon in the Search Bar',
      'actions_menu_icon'            , 'Obsolete.',
      'finder_icon'                  , 'Obsolete.',
      'show_select_columns'          , 'Determines whether to show the Select Columns option in the Actions Menu',
      'show_rows_per_page'           , 'Determines whether to show the Rows Per Page option in the Actions Menu',
      'show_filter'                  , 'Determines whether to show the Filter option in the Actions Menu',
      'show_sort'                    , 'Determines whether to show the Sort option in the Actions Menu',
      'show_control_break'           , 'Determines whether to show the Control option in the Actions Menu',
      'show_highlight'               , 'Determines whether to show the Highlight option in the Actions Menu',
      'show_compute'                 , 'Determines whether to show the Compute option in the Actions Menu',
      'show_aggregate'               , 'Determines whether to show the Aggregate option in the Actions Menu',
      'show_chart'                   , 'Determines whether to show the Chart option in the Actions Menu',
      'show_group_by'                , 'Determines whether to show the Group By option in the Actions Menu',
      'show_pivot'                   , 'Determines whether to show the Pivot option in the Actions Menu',
      'show_notify'                  , 'Determines whether to show the Notify option in the Actions Menu',
      'show_flashback'               , 'Determines whether to show the Flashback option in the Actions Menu',
      'show_save'                    , 'Determines whether to show the Save Report option in the Actions Menu',
      'show_save_public'             , 'Determines whether to show the Public option in the Save Report dialog',
      'save_public_auth_scheme'      , 'This authorization scheme must evaluate to TRUE in order for Public option to display in Save Report dialog',
      'save_public_auth_scheme_id'   , 'Foreign Key',
      'show_reset'                   , 'Determines whether to show the Reset option in the Actions Menu',
      'show_download'                , 'Determines whether to show the Download option in the Actions Menu',
      'show_help'                    , 'Determines whether to show the Help option in the Actions Menu',
      'download_auth_scheme'         , 'Authorization scheme to restrict who can download',
      'download_auth_scheme_id'      , 'Identifies the internal code of DOWNLOAD_AUTH_SCHEME',
      'download_formats'             , 'Identifies the download formats supported for this interactive report',
      'filename'                     , 'Identifies the filename prefix used when this report data is downloaded',
      'send_as_email'                , 'Identifies whether end users can send email with one of the supported download file as an attachment',
      'separator'                    , 'Identifies a column separator. If no value is entered, a comma or semicolon is used depending on your current NLS settings.',
      'enclosed_by'                  , 'Identifies a delimiter character. This character is used to delineate the starting and ending boundary of a data value. Default delimiter is double quotes.',
      'supplemental_text'            , 'Identifies text at the top of all download formats.',
      'detail_link_type'             , 'Identifies the type of link to display in the Link Column on the left of the interactive report',
      'detail_link_target'           , 'Target of the Link Column, if a custom target is defined',
      'detail_link_text'             , 'Text or HTML to display in the Link Column',
      'detail_link_attributes'       , 'Link attributes for the Link Column.  Displayed within the HTML "A" tag',
      'detail_link_checksum_type'    , 'An appropriate checksum when linking to protected pages',
      'detail_link_condition_type'   , 'For conditionally displayed Link Column; identifies the condition type to evaluate',
      'detail_link_cond_type_code'   , 'Internal code of DETAIL_LINK_CONDITION_TYPE.',
      'detail_link_cond_expression'  , 'Specifies an expression based on the specific condition type selected.',
      'detail_link_cond_expression2' , 'Specifies an expression based on the specific condition type selected.',
      'detail_link_auth_scheme'      , 'This authorization scheme must evaluate to TRUE in order for this page to be displayed',
      'detail_link_auth_scheme_id'   , 'Foreign Key',
      'alias'                        , 'Alias for this interactive report, may be used for API references',
      'report_id_item'               , 'Item which holds the saved report ID to display',
      'max_query_cost'               , 'Prevent execution of queries that are estimated to take longer than specified amount of time.',
      'email_from'                   , 'From address to use when sending email from email download or subscription',
      'oracle_text_index_column'     , 'A column which is indexed by an Oracle Text index',
      'fixed_header'                 , 'Identifies if the report has a fixed header, and if so where it is fixed to (Page or Region)',
      'fixed_header_max_height'      , 'For reports with a header fixed to Region, this identifies the maximum height',
      'lazy_loading'                 , 'Identifies whether to defer getting initial set of data from the server until needed',
      'icon_view_enabled_yn'         , 'Identifies that an icon view is enabled and selectable from the interactive report',
      'icon_view_use_custom'         , 'Identifies if icon view uses custom link',
      'icon_view_custom_link'        , 'If use custom link set to Yes, use defined custom link for the icon view',
      'icon_view_link_column'        , 'Identifies the report column that returns the link target of the icon',
      'icon_view_img_src_column'     , 'Identifies the report column that returns the image source',
      'icon_view_label_column'       , 'Identifies the report column that returns the image label',
      'icon_view_img_attr_text'      , 'Identifies the image attributes used for the HTML IMG tag.',
      'icon_view_alt_text'           , 'Identifies the HTML ALT text for the image.',
      'icon_view_title_text'         , 'Identifies the HTML TITLE tag for the image.',
      'icon_view_columns_per_row'    , 'Identifies the number of icons to be displayed per row',
      'detail_view_enabled_yn'       , 'Identifies that detail view is enabled and selectable from the interactive report',
      'detail_view_before_rows'      , 'Identifies the HTML to be displayed before report rows',
      'detail_view_for_each_row'     , 'Identifies the HTML to be displayed for each report row',
      'detail_view_after_rows'       , 'Identifies the HTML to be displayed after report rows',
      'internal_uid'                 , 'Identifies unique Interactive report ID used by APEX internally',
      'created_on'                   , 'Auditing; date the record was created.',
      'created_by'                   , 'Auditing; user that created the record.',
      'updated_on'                   , 'Auditing; date the record was last modified.',
      'updated_by'                   , 'Auditing; user that last modified the record.',
      'component_signature'          , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/

create or replace view apex_application_page_ir_col
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
ir.page_id                   page_id,
ir.id                       interactive_report_id,
ir.region_id                region_id,
r.plug_name                 region_name,
c.id                        column_id,
c.db_column_name            column_alias,
c.display_order             display_order,
(select cg.name from wwv_flow_worksheet_col_groups cg where cg.id = c.group_id and cg.security_group_id = ir.security_group_id) column_group,
c.group_id                  column_group_id,
c.column_identifier         ,
c.report_label              report_label,
c.column_label              form_label,
c.column_html_expression    html_expression,
c.column_link               ,
c.column_linktext           ,
c.column_link_attr          ,
c.column_link_checksum_type ,
--
decode(c.allow_sorting     ,'Y','Yes','N','No',c.allow_sorting     ) allow_sorting     ,
decode(c.allow_filtering   ,'Y','Yes','N','No',c.allow_filtering   ) allow_filtering   ,
decode(c.allow_highlighting   ,'Y','Yes','N','No',c.allow_highlighting) allow_highlighting,
decode(c.allow_ctrl_breaks ,'Y','Yes','N','No',c.allow_ctrl_breaks ) allow_ctrl_breaks ,
decode(c.allow_aggregations,'Y','Yes','N','No',c.allow_aggregations) allow_aggregations,
decode(c.allow_computations,'Y','Yes','N','No',c.allow_computations) allow_computations,
decode(c.allow_charting    ,'Y','Yes','N','No',c.allow_charting    ) allow_charting    ,
decode(c.allow_group_by    ,'Y','Yes','N','No',c.allow_group_by    ) allow_group_by    ,
decode(c.allow_pivot       ,'Y','Yes','N','No',c.allow_pivot       ) allow_pivot       ,
decode(c.allow_hide        ,'Y','Yes','N','No',c.allow_hide        ) allow_hide        ,
decode(c.sync_form_label   ,'Y','Yes','N','No',c.sync_form_label   ) synchronize_form_label,
--
c.column_type               ,
c.display_text_as           ,
c.heading_alignment         ,
c.column_alignment          ,
c.format_mask               ,
c.tz_dependent              ,
c.static_id                 ,
c.css_classes               ,
case c.use_as_row_header
  when 'Y' then 'Yes'
  when 'N' then 'No'
end as use_as_row_header,
--
c.rich_text_format,
--
decode(c.rpt_show_filter_lov,
       'D','Default Based on Column Type',
       'S','Use Defined List of Values to Filter Exact Match',
       'C','Use Defined List of Values to Filter Word Contains',
       '1','Use Named List of Values to Filter Exact Match',
       '2','Use Named List of Values to Filter Word Contains',
       'N','None',
       c.rpt_show_filter_lov)
                            filter_lov_source,
(select lv.lov_name
   from wwv_flow_lists_of_values$ lv
  where lv.id                = c.rpt_named_lov
    and lv.security_group_id = ir.security_group_id
)                           named_lov,
c.rpt_lov                   ,
c.rpt_filter_date_ranges    filter_date_ranges,
--
nvl((select r from apex_standard_conditions where d = c.display_condition_type), c.display_condition_type)
                            display_condition_type,
c.display_condition_type    display_condition_type_code,
c.display_condition         ,
c.display_condition2        ,
--
c.help_text                 ,
--
decode(substr(c.SECURITY_SCHEME,1,1),'!','Not ')||
nvl((select ss.name
  from wwv_flow_security_schemes ss
 where to_char(ss.id)       = ltrim(c.SECURITY_SCHEME,'!')
   and ss.flow_id           = f.application_id
   and ss.security_group_id = ir.security_group_id),
 c.SECURITY_SCHEME)             authorization_scheme,
c.security_scheme               authorization_scheme_id,
c.column_expr,
--
(select case when c.required_patch > 0 then pa.PATCH_NAME else '{Not '||pa.PATCH_NAME||'}' end PATCH_NAME
  from wwv_flow_patches pa
 where pa.id                = abs(c.REQUIRED_PATCH)
   and pa.security_group_id = ir.security_group_id)    build_option,
c.required_patch                     build_option_id,
--
c.column_comment                component_comment,
--
c.created_on,
c.created_by,
c.updated_on,
c.updated_by,
--
'Interactive Report Column-'||
c.db_column_name||
' s='||c.display_order||
' g='||(select cg.name from wwv_flow_worksheet_col_groups cg where cg.id = c.group_id and cg.security_group_id = ir.security_group_id)||
' l='||c.report_label||
substr(c.column_label,1,30)||
substr(c.column_link,1,30)||
substr(c.column_linktext,1,30)||
substr(c.column_link_attr,1,30)||
c.column_link_checksum_type||
c.allow_sorting||
c.allow_filtering||
c.allow_highlighting||
c.allow_ctrl_breaks||
c.allow_aggregations||
c.allow_computations||
c.allow_charting||
c.allow_group_by||
c.allow_pivot||
c.allow_hide||
c.column_type||
c.display_text_as||
' ha='||c.heading_alignment||
' ca='||c.column_alignment||
' f='||c.format_mask||
c.rpt_show_filter_lov||
' lov='||length(c.rpt_lov)||
' d='||c.rpt_filter_date_ranges||
' cond='||c.display_condition_type||
length(c.display_condition)||
length(c.display_condition2)||
' h='||length(c.help_text)||
' as='||decode(substr(c.SECURITY_SCHEME,1,1),'!','Not ')||
nvl((select ss.name
  from wwv_flow_security_schemes ss
 where to_char(ss.id)       = ltrim(c.SECURITY_SCHEME,'!')
   and ss.flow_id           = f.application_id
   and ss.security_group_id = ir.security_group_id),
 'Not found')||
 ' bo='||(select pa.PATCH_NAME from wwv_flow_patches pa where pa.id = abs( c.REQUIRED_PATCH) and pa.security_group_id = ir.security_group_id)
component_signature
from wwv_flow_worksheet_columns c,
     wwv_flow_worksheets        ir,
     wwv_flow_page_plugs        r,
     wwv_flow_authorized        f
where f.workspace_id       = ir.security_group_id
  and f.application_id     = ir.flow_id
  and f.workspace_id       = c.security_group_id
  and f.application_id     = c.flow_id
  and ir.security_group_id = r.security_group_id (+)
  and ir.flow_id           = r.flow_id (+)
  and ir.region_id         = r.id (+)
  and ir.security_group_id = c.security_group_id
  and ir.id                = c.worksheet_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_col',
    p_table_comment     => 'Report column definitions for interactive report columns',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                   , 'A work area mapped to one or more database schemas',
      'workspace_display_name'      , 'Display name for the workspace',
      'application_id'              , 'Application Primary Key, Unique over all workspaces',
      'application_name'            , 'Identifies the application',
      'page_id'                     , 'Identifies the page',
      'interactive_report_id'       , 'ID of the interactive report',
      'region_id'                   , 'ID of the interactive report region',
      'region_name'                 , 'Name of the interactive report region',
      'column_id'                   , 'ID of this column',
      'column_alias'                , 'Database column name or expression to use in SQL query when displaying this worksheet column',
      'display_order'               , 'Default display order of this column',
      'column_group'                , 'Name of the column group for this column',
      'column_group_id'             , 'ID of the column group for this column',
      'column_identifier'           , 'Unique column identifier within an interactive report used to build computation or row filter expression',
      'report_label'                , 'Report heading label to use for this column',
      'form_label'                  , 'Single row view label to use for this column',
      'html_expression'             , 'HTML column template used to display this column',
      'column_link'                 , 'Optional link target for this column',
      'column_linktext'             , 'Text do display if a link is defined',
      'column_link_attr'            , 'Link attributes for the column link.  Displayed within the HTML "A" tag',
      'column_link_checksum_type'   , 'An appropriate checksum when linking to protected pages',
      'allow_sorting'               , 'Determines whether to allow sorting for this column.',
      'allow_filtering'             , 'Determines whether to allow filtering for this column.',
      'allow_highlighting'          , 'Determines whether to allow highlighting for this column.',
      'allow_ctrl_breaks'           , 'Determines whether to allow control breaks for this column.',
      'allow_aggregations'          , 'Determines whether to allow aggregations for this column.',
      'allow_computations'          , 'Determines whether to allow computations for this column.',
      'allow_charting'              , 'Determines whether to allow charting for this column.',
      'allow_group_by'              , 'Determines whether to allow group by for this column.',
      'allow_pivot'                 , 'Determines whether to allow pivot for this column.',
      'allow_hide'                  , 'Determines whether to allow hiding this column.',
      'synchronize_form_label'      , 'Determines whether the form label should be synchronized or not.',
      'column_type'                 , 'Type of data in this column',
      'display_text_as'             , 'Format to display this column',
      'heading_alignment'           , 'Horizontal alignment of this column''s report heading',
      'column_alignment'            , 'Horizontal alignment of this column''s report data',
      'format_mask'                 , 'Format mask for this column',
      'tz_dependent'                , 'Column is time zone dependent',
      'static_id'                   , 'Identifies the Static ID of the column',
      'css_classes'                 , 'Identifies CSS classes to be applied to the column',
      'use_as_row_header'           , 'Value Identifies Row: Identifies if the current column value should be used as the row header',
      'rich_text_format'            , 'Identifies the format of a Rich Text column.',
      'named_lov'                   , 'Identifies the Shared List of Values to be used to display and filter this report column',
      'rpt_lov'                     , 'LOV query to display in filter dropdown',
      'filter_lov_source'           , 'Query used to retrieve a list of values for the interactive report.  Displayed in the column header dropdowns, and during filter and highlight creation.',
      'filter_date_ranges'          , 'Determines the range of dates to display for filters on this column',
      'display_condition_type'      , 'For conditionally displayed this column; identifies the condition type to evaluate',
      'display_condition_type_code' , 'Internal code of DISPLAY_CONDITION_TYPE',
      'display_condition'           , 'Specifies an expression based on the specific condition type selected.',
      'display_condition2'          , 'Specifies an expression based on the specific condition type selected.',
      'help_text'                   , 'Descriptive help text for this column, displayed when a user clicks on the column information icon',
      'authorization_scheme'        , 'Optional authorization scheme determining whether this column and data is available to a user',
      'authorization_scheme_id'     , 'ID of the authorization scheme',
      'column_expr'                 , 'Attribute for internal use only',
      'build_option'                , 'Column will be displayed if the Build Option is enabled',
      'build_option_id'             , 'Foreign Key',
      'created_on'                  , 'Auditing; date the record was created.',
      'created_by'                  , 'Auditing; user that created the record.',
      'updated_on'                  , 'Auditing; date the record was last modified.',
      'updated_by'                  , 'Auditing; user that last modified the record.',
      'component_comment'           , 'Developer Comment',
      'component_signature'         , 'Identifies attributes defined at a given component level to facilitate application comparisons'
    )
  );
end;
/
create or replace view apex_application_page_ir_cgrps
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
cg.worksheet_id             interactive_report_id,
cg.name                     column_group_name,
cg.description              column_group_description,
cg.display_sequence         display_sequence,
--
(select count(*) from wwv_flow_worksheet_columns wc where wc.group_id = cg.id and wc.security_group_id = cg.security_group_id) columns,
--
cg.created_on                ,
cg.created_by                ,
cg.updated_on                ,
cg.updated_by
from wwv_flow_worksheet_col_groups cg,
     wwv_flow_worksheets           ws,
     wwv_flow_authorized           f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = cg.security_group_id
  and ws.flow_id           = cg.flow_id
  and ws.id                = cg.worksheet_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_cgrps',
    p_table_comment     => 'Column group definitions for interactive report columns',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                , 'A work area mapped to one or more database schemas',
      'workspace_display_name'   , 'Display name for the workspace',
      'application_id'           , 'Application Primary Key, Unique over all workspaces',
      'application_name'         , 'Identifies the application',
      'interactive_report_id'    , 'ID of the interactive report',
      'column_group_name'        , 'Name of this column group, displayed on the single row view of this report and in the Select Columns dialog box',
      'column_group_description' , 'Description of this column group',
      'display_sequence'         , 'Display sequence of this column group',
      'columns'                  , 'Number of columns in this column group',
      'created_on'               , 'Auditing; date the record was created.',
      'created_by'               , 'Auditing; user that created the record.',
      'updated_on'               , 'Auditing; date the record was last modified.',
      'updated_by'               , 'Auditing; user that last modified the record.'
    )
  );
end;
/
create or replace view apex_application_page_ir_rpt
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id                    page_id,
ws.region_id                  region_id,
r.worksheet_id               interactive_report_id,
r.id                         report_id,
r.application_user           application_user,
r.name                       report_name,
r.report_alias               ,
r.session_id                 ,
r.tenant_id                  app_tenant_id,
r.base_report_id             ,
r.description                report_description,
r.report_seq                 display_sequence,
(case when r.view_mode is null or r.view_mode = 'REPORT' then r.report_type
 else r.view_mode end)       report_view_mode,
r.status                     ,
r.category_id                ,
(case when r.is_default = 'Y' and r.application_user='APXWS_DEFAULT' then 'PRIMARY_DEFAULT'
      when r.is_default = 'Y' and r.application_user='APXWS_ALTERNATIVE' then 'ALTERNATIVE_DEFAULT'
      when r.session_id is null and r.status='PUBLIC' then 'PUBLIC'
      when r.session_id is null and r.status='PRIVATE' then 'PRIVATE'
 else 'SESSION' end)         report_type,
(case when r.report_alias is not null then
   'f?p=' || chr( 38 ) || 'APP_ID.:' || r.page_id || ':' || chr( 38 ) || 'APP_SESSION.:IR' ||
   case when p.region_name is not null then
     '[' || p.region_name || ']'
   end|| '_' || r.report_alias
end)                         report_link_example,
r.display_rows               ,
r.report_columns             ,
--
r.sort_column_1              ,
r.sort_direction_1           ,
r.sort_column_2              ,
r.sort_direction_2           ,
r.sort_column_3              ,
r.sort_direction_3           ,
r.sort_column_4              ,
r.sort_direction_4           ,
r.sort_column_5              ,
r.sort_direction_5           ,
r.sort_column_6              ,
r.sort_direction_6           ,
--
r.break_on                   ,
r.break_enabled_on           ,
--
r.sum_columns_on_break       ,
r.avg_columns_on_break       ,
r.max_columns_on_break       ,
r.min_columns_on_break       ,
r.median_columns_on_break    ,
r.count_columns_on_break     ,
r.count_distnt_col_on_break  ,
--
r.flashback_mins_ago         flashback_minutes,
decode(r.flashback_enabled,'Y','Yes','N','No',r.flashback_enabled) flashback_enabled,
--
r.chart_type                 ,
r.chart_label_column         ,
r.chart_label_title          ,
r.chart_value_column         ,
r.chart_aggregate            ,
r.chart_value_title          ,
decode(r.chart_sorting,
       'DEFAULT','Default',
       'VALUE_DESC','Value - Descending',
       'VALUE_ASC','Value - Ascending',
       'LABEL_DESC','Label - Descending',
       'LAVEL_ASC','Label - Ascending',
       r.chart_sorting)  chart_sort_order,
decode(r.chart_orientation,
       'vertical', 'Vertical',
       'horizontal','Horizontal',
       r.chart_orientation) chart_orientation,
--
r.created_on        ,
r.created_by        ,
r.updated_on        last_updated_on,
r.updated_by        last_updated_by
--
from wwv_flow_worksheet_rpts r,
     wwv_flow_worksheets     ws,
     wwv_flow_page_plugs     p,
     wwv_flow_authorized     f
where f.workspace_id       = p.security_group_id
  and f.application_id     = p.flow_id
  and p.security_group_id  = ws.security_group_id
  and p.flow_id            = ws.flow_id
  and p.id                 = ws.region_id
  and ws.security_group_id = r.security_group_id
  and ws.id                = r.worksheet_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_rpt',
    p_table_comment     => 'Identifies user-level report settings for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'               , 'A work area mapped to one or more database schemas',
      'workspace_display_name'  , 'Display name for the workspace',
      'application_id'          , 'Application Primary Key, Unique over all workspaces',
      'application_name'        , 'Identifies the application',
      'page_id'                 , 'Identifies page number',
      'region_id'               , 'Identifies region number',
      'interactive_report_id'   , 'ID of the interactive report',
      'report_id'               , 'ID of the report',
      'application_user'        , 'The user these report settings are used by',
      'report_name'             , 'The name of these report settings',
      'report_alias'            , 'An alternate alphanumeric report identifier to use in report link',
      'session_id'              , 'For session-level report settings, the session ID associated with this record',
      'app_tenant_id'           , 'For non-default report settings, application tenant ID associated with this report',
      'base_report_id'          , 'For session-level report settings, the base report settings this record was derived from',
      'report_description'      , 'The description given to the report',
      'display_sequence'        , 'The order the report will appear in a list',
      'report_view_mode'        , 'Identifies the current view settings of the report',
      'status'                  , 'The shared status of these settings',
      'category_id'             , 'The category_id of the report',
      'report_type'             , 'Identifies whether this is a PRIMARY_DEFAULT,ALTERNATIVE_DEFAULT,PUBLIC,PRIVATE or SESSION based set of report settings',
      'report_link_example'     , 'An example to link to specific saved reports',
      'display_rows'            , 'Number of rows to display in the report',
      'report_columns'          , 'List of columns to display in the report',
      'sort_column_1'           , 'First column to sort by',
      'sort_direction_1'        , 'Direction to use for first column sort',
      'sort_column_2'           , 'First column to sort by',
      'sort_direction_2'        , 'Direction to use for first column sort',
      'sort_column_3'           , 'Second column to sort by',
      'sort_direction_3'        , 'Direction to use for first column sort',
      'sort_column_4'           , 'Third column to sort by',
      'sort_direction_4'        , 'Direction to use for first column sort',
      'sort_column_5'           , 'Fourth column to sort by',
      'sort_direction_5'        , 'Direction to use for first column sort',
      'sort_column_6'           , 'Fifth column to sort by',
      'sort_direction_6'        , 'Direction to use for first column sort',
      'break_on'                , 'Identifies a set of columns to break on',
      'break_enabled_on'        , 'Identifies which control breaks are enabled',
      'sum_columns_on_break'    , 'Identifies which columns to aggregate with a sum',
      'avg_columns_on_break'    , 'Identifies which columns to aggregate with a avg',
      'max_columns_on_break'    , 'Identifies which columns to aggregate with a max',
      'min_columns_on_break'    , 'Identifies which columns to aggregate with a min',
      'median_columns_on_break' , 'Identifies which columns to aggregate with a median',
      'count_columns_on_break'  , 'Identifies which columns to aggregate with a count',
      'count_distnt_col_on_break' , 'Identifies which columns to aggregate with a count distinct',
      'flashback_minutes'       , 'Identifies the number of minutes to flashback in the query',
      'flashback_enabled'       , 'Identifies whether flashback is enabled',
      'chart_type'              , 'Identifies the current chart type',
      'chart_label_column'      , 'Identifies the alias of the column to use as labels in a chart',
      'chart_label_title'       , 'Text to use as the label axis title for a chart',
      'chart_value_column'      , 'Identifies the alias of the column to use as values in a chart',
      'chart_aggregate'         , 'Identifies an aggregation function to use on the values in the chart',
      'chart_value_title'       , 'Text to use as the value axis title for a chart',
      'chart_sort_order'        , 'Identifies the current chart sorting method',
      'chart_orientation'       , 'Identifies the current orientation of the chart',
      'created_on'              , 'Auditing; date the record was created.',
      'created_by'              , 'Auditing; user that created the record.',
      'last_updated_on'         , 'Auditing; date the record was last modified.',
      'last_updated_by'         , 'Auditing; user that last modified the record.'
    )
  );
end;
/
create or replace view apex_application_page_ir_comp
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id             page_id,
r.worksheet_id        interactive_report_id,
r.id                  report_id,
r.application_user    application_user,
r.name                report_name,
c.id                  computation_id,
c.db_column_name      computation_column_alias,
c.column_identifier   computation_column_identifier,
c.computation_expr    computation_expression,
c.format_mask         computation_format_mask,
c.column_type         computation_column_type,
c.report_label        computation_report_label,
c.column_label        computation_form_label,
-- audit
--
c.created_on        ,
c.created_by        ,
c.updated_on        last_updated_on,
c.updated_by        last_updated_by
--
from wwv_flow_worksheet_computation c,
     wwv_flow_worksheet_rpts r,
     wwv_flow_worksheets ws,
     wwv_flow_authorized    f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = r.security_group_id
  and ws.flow_id           = r.flow_id
  and ws.id                = r.worksheet_id
  and r.security_group_id  = c.security_group_id
  and r.flow_id            = c.flow_id
  and r.id                 = c.report_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_comp',
    p_table_comment     => 'Identifies computations defined in user-level report settings for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                     , 'A work area mapped to one or more database schemas',
      'workspace_display_name'        , 'Display name for the workspace',
      'application_id'                , 'Application Primary Key, Unique over all workspaces',
      'application_name'              , 'Identifies the application',
      'page_id'                       , 'Identifies page number',
      'interactive_report_id'         , 'ID of the interactive report',
      'report_id'                     , 'ID of the report',
      'application_user'              , 'The user these report settings are used by',
      'report_name'                   , 'The name of these report settings',
      'computation_id'                , 'ID of this computation',
      'computation_column_alias'      , 'Database column name or expression to use in SQL query when displaying this computation column',
      'computation_column_identifier' , 'Computation column identifier to use in finding computation column alias',
      'computation_expression'        , 'Computation expression',
      'computation_format_mask'       , 'Format mask for this computation',
      'computation_column_type'       , 'Type of data in this computation',
      'computation_report_label'      , 'Report heading label to use for this computation',
      'computation_form_label'        , 'Single row view label to use for this computation',
      'created_on'                    , 'Auditing; date the record was created.',
      'created_by'                    , 'Auditing; user that created the record.',
      'last_updated_on'               , 'Auditing; date the record was last modified.',
      'last_updated_by'               , 'Auditing; user that last modified the record.'
    )
  );
end;
/
create or replace view apex_application_page_ir_cond
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id             page_id,
r.worksheet_id        interactive_report_id,
r.id                  report_id,
r.application_user    application_user,
r.name                report_name,
c.id                  condition_id,
c.name                condition_name,
decode(c.condition_type,'FILTER','Filter','HIGHLIGHT','Highlight','SEARCH','Search',c.condition_type)
                      condition_type,
decode(c.allow_delete,'Y','Yes','N','No',c.allow_delete)
                      condition_allow_delete,
c.time_zone           time_zone,
-- filter expression
c.column_name         condition_column_name,
c.operator            condition_operator,
c.expr_type           condition_expr_type,
c.expr                condition_expression,
c.expr2               condition_expression2,
c.condition_sql       condition_sql,
c.condition_display   condition_display,
-- enabled?
decode(c.enabled,'Y','Yes','N','No',c.enabled)
                      condition_enabled,
-- highlighting settings
c.highlight_sequence  ,
c.row_bg_color        highlight_row_color,
c.row_font_color      highlight_row_font_color,
c.column_bg_color     highlight_cell_color,
c.column_font_color   highlight_cell_font_color,
-- audit
--
c.created_on        ,
c.created_by        ,
c.updated_on        last_updated_on,
c.updated_by        last_updated_by
--
from wwv_flow_worksheet_conditions c,
     wwv_flow_worksheet_rpts       r,
     wwv_flow_worksheets           ws,
     wwv_flow_authorized           f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = r.security_group_id
  and ws.flow_id           = r.flow_id
  and ws.id                = r.worksheet_id
  and r.security_group_id  = c.security_group_id
  and r.flow_id            = c.flow_id
  and r.id                 = c.report_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_cond',
    p_table_comment     => 'Identifies filters and highlights defined in user-level report settings for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                 , 'A work area mapped to one or more database schemas',
      'workspace_display_name'    , 'Display name for the workspace',
      'application_id'            , 'Application Primary Key, Unique over all workspaces',
      'application_name'          , 'Identifies the application',
      'page_id'                   , 'Identifies page number',
      'interactive_report_id'     , 'ID of the interactive report',
      'report_id'                 , 'ID of the report',
      'application_user'          , 'The user these report settings are used by',
      'report_name'               , 'The name of these report settings',
      'condition_id'              , 'ID of this filter or highlight',
      'condition_name'            , 'Name of this filter or highlight',
      'condition_type'            , 'Identifies this as a filter or highlight condtiion',
      'condition_allow_delete'    , 'Identifies whether this condition can be deleted',
      'condition_column_name'     , 'Alias of the column this condition is based on',
      'condition_operator'        , 'Operator used in this condition',
      'condition_expr_type'       , 'If expression is date/time, expression type is Value',
      'condition_expression'      , 'Expression used for this condition, depending on condition type.  This will be included in the report SQL as a bind variable',
      'condition_expression2'     , 'Expression used for this condition, depending on condition type.  This will be included in the report SQL as a bind variable',
      'time_zone'                 , 'Time zone of user''s database session when condition was saved',
      'condition_sql'             , 'Stored SQL expression to be used for this condition',
      'condition_display'         , 'Stored text expression representing this condition',
      'condition_enabled'         , 'Identifies whether this condition is enabled in the current report settings',
      'highlight_sequence'        , 'Determines the order a highlight rule is applied',
      'highlight_row_color'       , 'Determines the row background color for a row-level highlight rule',
      'highlight_row_font_color'  , 'Determines the row font color for a row-level highlight rule',
      'highlight_cell_color'      , 'Determines the cell background color for a cell-level highlight rule',
      'highlight_cell_font_color' , 'Determines the cell font color for a cell-level highlight rule',
      'created_on'                , 'Auditing; date the record was created.',
      'created_by'                , 'Auditing; user that created the record.',
      'last_updated_on'           , 'Auditing; date the record was last modified.',
      'last_updated_by'           , 'Auditing; user that last modified the record.'
    )
  );
end;
/
create or replace view apex_application_page_ir_cat
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
c.worksheet_id       interactive_report_id,
c.id                 category_id,
c.name               category_name,
c.application_user   application_user,
c.display_sequence   display_sequence
from wwv_flow_worksheet_categories c,
     wwv_flow_worksheets           ws,
     wwv_flow_authorized           f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = c.security_group_id
  and ws.flow_id           = c.flow_id
  and ws.id                = c.worksheet_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_cat',
    p_table_comment     => 'Report column category definitions for interactive report columns',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'             , 'A work area mapped to one or more database schemas',
      'workspace_display_name', 'Display name for the workspace',
      'application_id'        , 'Application Primary Key, Unique over all workspaces',
      'application_name'      , 'Identifies the application',
      'interactive_report_id' , 'ID of the interactive report',
      'category_id'           , 'ID of the category',
      'category_name'         , 'The name of the column category',
      'display_sequence'      , 'The order the category will appear in a list',
      'application_user'      , 'The user of these column category definitions'
    )
  );
end;
/
create or replace view apex_application_page_ir_sub
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id             page_id,
r.worksheet_id        interactive_report_id,
null                  websheet_id,
r.id                  report_id,
r.name                report_name,
n.id                  notify_id,
n.session_id,
n.owner               owner,
n.language            language,
decode(n.download_format,
    'XLSX','Excel',
    n.download_format) download_format,
n.strip_rich_text,
n.data_only,
n.pdf_page_size,
n.pdf_page_orientation,
n.pdf_accessible,
n.skip_no_data_found,
n.email_from          email_from,
n.email_address       email_address,
n.email_subject       email_subject,
n.start_date          start_date,
n.end_date            end_date,
n.end_day             end_day,
n.end_day_unit        end_day_unit,
n.offset_date         offset_date,
decode(n.notify_interval,'D','Daily','W','Weekly','M','Monthly',n.notify_interval)
                      notify_interval,
n.status              ,
n.notify_error        error_message,
n.created_on          ,
n.created_by          ,
n.updated_on          last_updated_on,
n.updated_by          last_updated_by
--
from wwv_flow_worksheet_notify n,
     wwv_flow_worksheet_rpts   r,
     wwv_flow_worksheets       ws,
     wwv_flow_authorized       f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = r.security_group_id
  and ws.flow_id           = r.flow_id
  and ws.id                = r.worksheet_id
  and r.security_group_id  = n.security_group_id
  and r.id                 = n.report_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_sub',
    p_table_comment     => 'Identifies subscriptions scheduled in saved reports for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                 , 'A work area mapped to one or more database schemas',
      'workspace_display_name'    , 'Display name for the workspace',
      'application_id'            , 'Application Primary Key, Unique over all workspaces',
      'application_name'          , 'Identifies the application',
      'page_id'                   , 'Identifies page number',
      'interactive_report_id'     , 'ID of the interactive report',
      'websheet_id'               , 'Obsolete',
      'report_id'                 , 'ID of the report',
      'owner'                     , 'The user the subscription is used by',
      'report_name'               , 'The name of report to send subscription',
      'notify_id'                 , 'ID of this subscription',
      'session_id'                , 'The cloned session ID of the subscription to reference session state',
      'start_date'                , 'Date the subscription starts',
      'end_date'                  , 'Date the subscription ends',
      'end_day'                   , 'Number of day the subscription ends',
      'end_day_unit'              , 'The day unit subscription ends (Day, Week, Month, Year)',
      'offset_date'               , 'Date the next subscription needs to send',
      'notify_interval'           , 'The interval between the subscriptions',
      'status'                    , 'The subscription status',
      'error_message'             , 'The error message if subscription fails',
      'language'                  , 'The language code to use when sending translated text if translation exits',
      'download_format'           , 'Identifies which download format is used for the subscription',
      'strip_rich_text'           , 'Specifies whether or not to remove rich text markup tags from the original column values for report data exported as CSV or Excel file.',
      'data_only'                 , 'Whether to include column groups, control breaks, aggregates, and highlights.',
      'pdf_page_size'             , 'Page size of the PDF report.',
      'pdf_page_orientation'      , 'Page orientation of the PDF report.',
      'pdf_accessible'            , 'Whether to include accessibility tags in the PDF.',
      'skip_no_data_found'        , 'Whether to skip the subscription if no data is found.',
      'email_from'                , 'The from email address when sending subscription',
      'email_address'             , 'The email address to send subscription',
      'email_subject'             , 'The email subject to send subscription',
      'created_on'                , 'Auditing; date the record was created.',
      'created_by'                , 'Auditing; user that created the record.',
      'last_updated_on'           , 'Auditing; date the record was last modified.',
      'last_updated_by'           , 'Auditing; user that last modified the record.'
    )
  );
end;
/
create or replace view apex_application_page_ir_grpby
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id             page_id,
r.worksheet_id        interactive_report_id,
null                  websheet_id,
r.id                  report_id,
r.application_user    application_user,
r.name                report_name,
g.id                  group_by_id,
group_by_columns      ,
g.created_on          ,
g.created_by          ,
g.updated_on          last_updated_on,
g.updated_by          last_updated_by,
g.function_01         ,
g.function_column_01  ,
g.function_db_column_name_01 ,
g.function_label_01          ,
g.function_format_mask_01    ,
g.function_sum_01            ,
g.function_02                ,
g.function_column_02         ,
g.function_db_column_name_02 ,
g.function_label_02          ,
g.function_format_mask_02    ,
g.function_sum_02            ,
g.function_03                ,
g.function_column_03         ,
g.function_db_column_name_03 ,
g.function_label_03          ,
g.function_format_mask_03    ,
g.function_sum_03            ,
g.function_04                ,
g.function_column_04         ,
g.function_db_column_name_04 ,
g.function_label_04          ,
g.function_format_mask_04    ,
g.function_sum_04            ,
g.function_05                ,
g.function_column_05         ,
g.function_db_column_name_05 ,
g.function_label_05          ,
g.function_format_mask_05    ,
g.function_sum_05            ,
g.function_06                ,
g.function_column_06         ,
g.function_db_column_name_06 ,
g.function_label_06          ,
g.function_format_mask_06    ,
g.function_sum_06            ,
g.function_07                ,
g.function_column_07         ,
g.function_db_column_name_07 ,
g.function_label_07          ,
g.function_format_mask_07    ,
g.function_sum_07            ,
g.function_08                ,
g.function_column_08         ,
g.function_db_column_name_08 ,
g.function_label_08          ,
g.function_format_mask_08    ,
g.function_sum_08            ,
--
g.sort_column_01             ,
g.sort_direction_01          ,
g.sort_column_02             ,
g.sort_direction_02          ,
g.sort_column_03             ,
g.sort_direction_03          ,
g.sort_column_04             ,
g.sort_direction_04          ,
g.sort_column_05             ,
g.sort_direction_05          ,
g.sort_column_06             ,
g.sort_direction_06
--
from wwv_flow_worksheet_group_by g,
     wwv_flow_worksheet_rpts     r,
     wwv_flow_worksheets         ws,
     wwv_flow_authorized         f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = r.security_group_id
  and ws.id                = r.worksheet_id
  and r.security_group_id = g.security_group_id
  and r.id                = g.report_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_grpby',
    p_table_comment     => 'Identifies group by view defined in user-level report settings for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_id'                    , 'Identifies page number',
      'interactive_report_id'      , 'ID of the interactive report',
      'websheet_id'                , 'Obsolete',
      'report_id'                  , 'ID of the report',
      'application_user'           , 'The user these group by settings are used by',
      'report_name'                , 'The name of these report settings',
      'group_by_id'                , 'ID of this group by settings',
      'group_by_columns'           , 'Columns to group by',
      'created_on'                 , 'Auditing; date the record was created.',
      'created_by'                 , 'Auditing; user that created the record.',
      'last_updated_on'            , 'Auditing; date the record was last modified.',
      'last_updated_by'            , 'Auditing; user that last modified the record.',
--
      'function_01'                , 'Identifies the function to use in group by view',
      'function_column_01'         , 'Identifies which column to use in the function',
      'function_db_column_name_01' , 'Identifies the database column name the function used on',
      'function_label_01'          , 'Identifies the function column label',
      'function_format_mask_01'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_01'            , 'Determines whether to show the sum of the function column',
      'function_02'                , 'Identifies the function to use in group by view',
      'function_column_02'         , 'Identifies which column to use in the function',
      'function_db_column_name_02' , 'Identifies the database column name the function used on',
      'function_label_02'          , 'Identifies the function column label',
      'function_format_mask_02'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_02'            , 'Determines whether to show the sum of the function column',
      'function_03'                , 'Identifies the function to use in group by view',
      'function_column_03'         , 'Identifies which column to use in the function',
      'function_db_column_name_03' , 'Identifies the database column name the function used on',
      'function_label_03'          , 'Identifies the function column label',
      'function_format_mask_03'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_03'            , 'Determines whether to show the sum of the function column',
      'function_04'                , 'Identifies the function to use in group by view',
      'function_column_04'         , 'Identifies which column to use in the function',
      'function_db_column_name_04' , 'Identifies the database column name the function used on',
      'function_label_04'          , 'Identifies the function column label',
      'function_format_mask_04'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_04'            , 'Determines whether to show the sum of the function column',
      'function_05'                , 'Identifies the function to use in group by view',
      'function_column_05'         , 'Identifies which column to use in the function',
      'function_db_column_name_05' , 'Identifies the database column name the function used on',
      'function_label_05'          , 'Identifies the function column label',
      'function_format_mask_05'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_05'            , 'Determines whether to show the sum of the function column',
      'function_06'                , 'Identifies the function to use in group by view',
      'function_column_06'         , 'Identifies which column to use in the function',
      'function_db_column_name_06' , 'Identifies the database column name the function used on',
      'function_label_06'          , 'Identifies the function column label',
      'function_format_mask_06'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_06'            , 'Determines whether to show the sum of the function column',
      'function_07'                , 'Identifies the function to use in group by view',
      'function_column_07'         , 'Identifies which column to use in the function',
      'function_db_column_name_07' , 'Identifies the database column name the function used on',
      'function_label_07'          , 'Identifies the function column label',
      'function_format_mask_07'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_07'            , 'Determines whether to show the sum of the function column',
      'function_08'                , 'Identifies the function to use in group by view',
      'function_column_08'         , 'Identifies which column to use in the function',
      'function_db_column_name_08' , 'Identifies the database column name the function used on',
      'function_label_08'          , 'Identifies the function column label',
      'function_format_mask_08'    , 'Identifies the format mask which can be used to format the function column',
      'function_sum_08'            , 'Determines whether to show the sum of the function column',
--
      'sort_column_01'             , 'First column to sort by',
      'sort_direction_01'          , 'Direction to use for first column sort',
      'sort_column_02'             , 'Second column to sort by',
      'sort_direction_02'          , 'Direction to use for second column sort',
      'sort_column_03'             , 'Third column to sort by',
      'sort_direction_03'          , 'Direction to use for third column sort',
      'sort_column_04'             , 'Fourth column to sort by',
      'sort_direction_04'          , 'Direction to use for fourth column sort',
      'sort_column_05'             , 'Fourth column to sort by',
      'sort_direction_05'          , 'Direction to use for fifth column sort',
      'sort_column_06'             , 'Fourth column to sort by',
      'sort_direction_06'          , 'Direction to use for sixth column sort'
    )
  );
end;
/
create or replace view apex_application_page_ir_pivot
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id             page_id,
r.worksheet_id        interactive_report_id,
null                  websheet_id,
r.id                  report_id,
r.application_user    application_user,
r.name                report_name,
p.id                  pivot_id,
p.pivot_columns       ,
p.row_columns         ,
p.created_on          ,
p.created_by          ,
p.updated_on          last_updated_on,
p.updated_by          last_updated_by
--
from wwv_flow_worksheet_pivot p,
     wwv_flow_worksheet_rpts r,
     wwv_flow_worksheets ws,
     wwv_flow_authorized    f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = r.security_group_id
  and ws.id                = r.worksheet_id
  and r.security_group_id  = p.security_group_id
  and r.id                 = p.report_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_pivot',
    p_table_comment     => 'Identifies pivot view defined in user-level report settings for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_id'                    , 'Identifies page number',
      'interactive_report_id'      , 'ID of the interactive report',
      'websheet_id'                , 'Obsolete',
      'report_id'                  , 'ID of the report',
      'application_user'           , 'The owner of these pivot settings',
      'report_name'                , 'The name of these report settings',
      'pivot_id'                   , 'ID of this pivot settings',
      'pivot_columns'              , 'Columns to pivot',
      'row_columns'                , 'Columns to display in report row',
      'created_on'                 , 'Auditing; date the record was created.',
      'created_by'                 , 'Auditing; user that created the record.',
      'last_updated_on'            , 'Auditing; date the record was last modified.',
      'last_updated_by'            , 'Auditing; user that last modified the record.'
    )
  );
end;
/
create or replace view apex_application_page_ir_pvagg
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id             page_id,
r.worksheet_id        interactive_report_id,
null                  websheet_id,
r.id                  report_id,
r.application_user    application_user,
r.name                report_name,
p.id                  pivot_id,
a.display_seq         ,
a.function_name       ,
a.column_name         ,
a.db_column_name      ,
a.column_label        ,
a.format_mask         ,
decode(a.display_sum,'Y','Yes','N','No',a.display_sum)
                      display_sum,
a.created_on          ,
a.created_by          ,
a.updated_on          last_updated_on,
a.updated_by          last_updated_by
--
from wwv_flow_worksheet_pivot p,
     wwv_flow_worksheet_pivot_agg a,
     wwv_flow_worksheet_rpts r,
     wwv_flow_worksheets ws,
     wwv_flow_authorized    f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = r.security_group_id
  and ws.id                = r.worksheet_id
  and r.security_group_id = p.security_group_id
  and r.id                = p.report_id
  and p.security_group_id = a.security_group_id
  and p.id                = a.pivot_id
/

begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_pvagg',
    p_table_comment     => 'Identifies aggregates defined for a pivot view in user-level report settings for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_id'                    , 'Identifies page number',
      'interactive_report_id'      , 'ID of the interactive report',
      'websheet_id'                , 'Obsolete',
      'report_id'                  , 'ID of the report',
      'application_user'           , 'The owner of these pivot settings',
      'report_name'                , 'The name of these report settings',
      'pivot_id'                   , 'ID of this pivot settings',
      'display_seq'                , 'Used to display aggregate information in sequence within a pivot view',
      'function_name'              , 'Identifies the aggregate function to use in pivot view',
      'column_name'                , 'Identifies which column to use in the aggregate',
      'db_column_name'             , 'Identifies the alias of the aggregated column',
      'column_label'               , 'Identifies the aggregate column label',
      'format_mask'                , 'Identifies the format mask which can be used to format the aggregate column',
      'display_sum'                , 'Determines whether to show the sum of the aggregate column',
      'created_on'                 , 'Auditing; date the record was created.',
      'created_by'                 , 'Auditing; user that created the record.',
      'last_updated_on'            , 'Auditing; date the record was last modified.',
      'last_updated_by'            , 'Auditing; user that last modified the record.'
    )
  );
end;
/
create or replace view apex_application_page_ir_pvsrt
as
select
f.workspace,
f.workspace_display_name,
f.application_id,
f.application_name,
r.page_id             page_id,
r.worksheet_id        interactive_report_id,
null                  websheet_id,
r.id                  report_id,
r.application_user    application_user,
r.name                report_name,
p.id                  pivot_id,
ps.sort_seq            ,
ps.sort_column_name    ,
ps.sort_direction      ,
ps.created_on          ,
ps.created_by          ,
ps.updated_on          last_updated_on,
ps.updated_by          last_updated_by
--
from wwv_flow_worksheet_pivot      p,
     wwv_flow_worksheet_pivot_sort ps,
     wwv_flow_worksheet_rpts       r,
     wwv_flow_worksheets           ws,
     wwv_flow_authorized           f
where f.workspace_id       = ws.security_group_id
  and f.application_id     = ws.flow_id
  and ws.security_group_id = r.security_group_id
  and ws.id                = r.worksheet_id
  and r.security_group_id  = p.security_group_id
  and r.id                 = p.report_id
  and p.security_group_id  = ps.security_group_id
  and p.id                 = ps.pivot_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_application_page_ir_pvsrt',
    p_table_comment     => 'Identifies sorts defined for a pivot view in user-level report settings for an interactive report',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace'                  , 'A work area mapped to one or more database schemas',
      'workspace_display_name'     , 'Display name for the workspace',
      'application_id'             , 'Application Primary Key, Unique over all workspaces',
      'application_name'           , 'Identifies the application',
      'page_id'                    , 'Identifies page number',
      'interactive_report_id'      , 'ID of the interactive report',
      'websheet_id'                , 'Obsolete',
      'report_id'                  , 'ID of the report',
      'application_user'           , 'The owner of these pivot settings',
      'report_name'                , 'The name of these report settings',
      'pivot_id'                   , 'ID of this pivot settings',
      'sort_seq'                   , 'The sequence of sort to apply within a pivot view',
      'sort_column_name'           , 'Identifies the column to sort by',
      'sort_direction'             , 'Identifies the direction to use for column sort',
      'created_on'                 , 'Auditing; date the record was created.',
      'created_by'                 , 'Auditing; user that created the record.',
      'last_updated_on'            , 'Auditing; date the record was last modified.',
      'last_updated_by'            , 'Auditing; user that last modified the record.'
    )
  );
end;
/

----------------------------------
-- C R O S S   V I E W   V I E W S
--
prompt ...apex_application_all_auth
create or replace view apex_application_all_auth
as
          select workspace, workspace_display_name, application_id, application_name, null page_id, 'Application'            component_type, application_name                component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATIONS                where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, null page_id, 'Breadcrumb Entry'       component_type, ENTRY_LABEL                     component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_BC_ENTRIES      where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, null page_id, 'Application Computation' component_type, COMPUTATION_ITEM                component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_COMPUTATIONS    where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, null page_id, 'List Entry'             component_type, ENTRY_TEXT                      component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_LIST_ENTRIES    where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, null page_id, 'Navigation Bar'         component_type, ICON_SUBTEXT                    component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_NAV_BAR         where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page'                   component_type, page_name                       component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGES           where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Branch'            component_type, nvl(branch_name, substr(BRANCH_ACTION,1,30))      component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_BRANCHES   where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Button'            component_type, BUTTON_NAME                     component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_BUTTONS    where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Computation'       component_type, ITEM_NAME                       component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_COMP       where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Dynamic Action'    component_type, dynamic_action_name             component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_DA         where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Interactive Report Column' component_type, region_name||'. '||column_alias component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status from APEX_APPLICATION_PAGE_IR_COL where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Item'              component_type, item_name                       component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_ITEMS      where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Process'           component_type, process_name                    component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_PROC       where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Region'            component_type, region_name                     component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_REGIONS    where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Region Column'     component_type, region_name||'. '||name         component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_REG_COLS   where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Report Column'     component_type, region_name||'. '||column_alias component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_RPT_COLS   where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, page_id,      'Page Validation'        component_type, VALIDATION_NAME                 component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PAGE_VAL        where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, null page_id, 'Parent Tab'             component_type, tab_name                        component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PARENT_TABS     where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, null page_id, 'Application Process'    component_type, process_name                    component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_PROCESSES       where authorization_scheme is not null
union all select workspace, workspace_display_name, application_id, application_name, null page_id, 'Tab'                    component_type, tab_name                        component_name, authorization_scheme, decode(replace(translate(authorization_scheme,'1234567890!','00000000000'),'0',null),null,'Invalid','Valid') status  from APEX_APPLICATION_TABS            where authorization_scheme is not null
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'APEX_APPLICATION_ALL_AUTH',
    p_table_comment     => 'All authorization schemes for all components by Application',
    p_column_comments   => wwv_flow_t_varchar2 (
      'WORKSPACE'             , 'Authorization Scheme Workspace',
      'workspace_display_name' , 'Display name for the workspace',
      'APPLICATION_ID'        , 'Authorization Scheme Application ID',
      'APPLICATION_NAME'      , 'Authorization Scheme Application Name',
      'PAGE_ID'               , 'Authorization Scheme Page ID',
      'COMPONENT_TYPE'        , 'Authorization Scheme APEX component type',
      'COMPONENT_NAME'        , 'Authorization Scheme APEX component name',
      'AUTHORIZATION_SCHEME'  , 'Authorization Scheme name',
      'STATUS'                , 'Authorization Scheme status'
    )
  );
end;
/

-- for application builder

prompt ...apex_developer_activity_log
create or replace view apex_developer_activity_log
as
select
    f.workspace,
    f.workspace_display_name,
    f.application_id,
    f.application_name,
    a.flow_user            developer,
    a.audit_date           audit_date,
    decode(a.audit_action,
       'I','Insert',
       'U','Update',
       'D','Delete',
       a.audit_action)     audit_action,
    a.page_id              page_id,
    (select st.name
       from wwv_flow_steps st
      where st.flow_id = a.flow_id
        and st.id = a.page_id
        and st.security_group_id = a.security_group_id
         ) page_name,
    --
    decode(a.flow_table,
       'WWV_FLOW_COMPANY_STATIC_FILES' , 'Workspace Files',
       'WWV_FLOWS'                     , 'Application Attributes',
       'WWV_FLOW_USER_INTERFACES'      , 'Application User Interfaces',
       'WWV_FLOW_STATIC_FILES'         , 'Application Files',
       'WWV_FLOW_COMPUTATIONS'         , 'Application Computation',
       'WWV_FLOW_APP_SETTINGS'         , 'Application Settings',
       'WWV_FLOW_DEVELOPERS'           , 'Application Developers (Administration)',
       'WWV_FLOW_EFFECTIVE_USERID_MAP' , 'Effective User ID (Administration)',
       'WWV_FLOW_ENTRY_POINTS'         , 'Entry Points',
       'WWV_FLOW_ENTRY_POINT_ARGS'     , 'Entry Point Arguments',
       'WWV_FLOW_ICON_BAR'             , 'Navigation Bar',
       'WWV_FLOW_LISTS'                , 'Lists',
       'WWV_FLOW_ITEMS'                , 'Application Items',
       'WWV_FLOW_LISTS_OF_VALUES$'     , 'List of Values',
       'WWV_FLOW_LISTS_OF_VALUES_DATA' , 'List of Values Entry',
       'WWV_FLOW_LIST_OF_VALUES_COLS'  , 'List of Values Columns',
       'WWV_FLOW_LIST_ITEMS'           , 'List Item',
       'WWV_FLOW_LIST_TEMPLATES'       , 'List Template',
       'WWV_FLOW_MENU_OPTIONS'         , 'Breadcrumb Entry',
       'WWV_FLOW_MESSAGES$'            , 'Messages',
       'WWV_FLOW_PAGE_DA_ACTIONS'      , 'Dynamic Action - Actions',
       'WWV_FLOW_PAGE_DA_EVENTS'       , 'Dynamic Action - Events',
       'WWV_FLOW_PAGE_PLUGS'           , 'Region Attributes',
       'WWV_FLOW_REGION_COLUMNS'       , 'Region Columns',
       'WWV_FLOW_PAGE_PLUG_TEMPLATES'  , 'Region Template',
       'WWV_FLOW_PATCHES'              , 'Build Options',
       'WWV_FLOW_PLUGINS'              , 'Plug-ins',
       'WWV_FLOW_PLUGIN_ATTRIBUTES'    , 'Plug-in Attributes',
       'WWV_FLOW_PLUGIN_STD_ATTRIBUTES', 'Plug-in Standard Attributes',
       'WWV_FLOW_PLUGIN_ATTR_VALUES'   , 'Plug-in Attribute Values',
       'WWV_FLOW_PLUGIN_FILES'         , 'Plug-in Files',
       'WWV_FLOW_PLUGIN_EVENTS'        , 'Plug-in Events',
       'WWV_FLOW_PLUGIN_ITEM_FILTERS'  , 'Plug-in Filter Operators',
       'WWV_FLOW_PLUGIN_SETTINGS'      , 'Plug-in Settings',
       'WWV_FLOW_PROCESSING'           , 'Application Process',
       'WWV_FLOW_ROW_TEMPLATES'        , 'Row Template',
       'WWV_FLOW_SHORTCUTS'            , 'Shortcuts',
       'WWV_FLOW_STEPS'                , 'Page Attributes',
       'WWV_FLOW_STEP_BRANCHES'        , 'Page Branch',
       'WWV_FLOW_STEP_BRANCH_ARGS'     , 'Page Branch Arguments',
       'WWV_FLOW_STEP_BUTTONS'         , 'Page Button',
       'WWV_FLOW_STEP_COMPUTATIONS'    , 'Page Computation',
       'WWV_FLOW_STEP_ITEMS'           , 'Page Item',
       'WWV_FLOW_STEP_ITEM_GROUPS'     , 'Filter Groups',
       'WWV_FLOW_STEP_ITEM_HELP'       , 'Page Item Help Text',
       'WWV_FLOW_STEP_PROCESSING'      , 'Page Process',
       'WWV_FLOW_STEP_VALIDATIONS'     , 'Page Validation',
       'WWV_FLOW_TABS'                 , 'Tabs',
       'WWV_FLOW_TEMPLATES'            , 'Page Template',
       'WWV_FLOW_THEMES'               , 'Theme',
       'WWV_FLOW_THEME_STYLES'         , 'Theme Style',
       'WWV_FLOW_THEME_FILES'          , 'Theme Files',
       'WWV_FLOW_PAGE_TMPL_DISP_POINTS' ,'Page Template Display Points',
       'WWV_FLOW_PLUG_TMPL_DISP_POINTS' ,'Region Template Display Points',
       'WWV_FLOW_TOPLEVEL_TABS'        , 'Parent Tab',
       'WWV_FLOW_SHARED_WEB_SERVICES'  , 'Web Service',
       'WWV_FLOW_AUTHENTICATIONS'      , 'Authentication Scheme',
       'WWV_FLOW_SECURITY_SCHEMES'     , 'Authorization Scheme',
       'WWV_FLOW_TREES'                , 'Tree',
       'WWV_FLOW_SHARED_QUERIES'       , 'Report Query',
       'WWV_FLOW_REPORT_LAYOUTS'       , 'Report Layout',
       'WWV_FLOW_LOAD_TABLES'          , 'Data Load Tables',
       'WWV_FLOW_LOAD_TABLE_LOOKUPS'   , 'Data Load Table Lookups',
       'WWV_FLOW_LOAD_TABLE_RULES'     , 'Data Load Table Transformation Rules',
       'WWV_FLOW_UI_TYPES'             , 'User Interface Types',
       'WWV_REMOTE_SERVERS'            , 'Remote Servers',
       'WWV_CREDENTIALS'               , 'Credentials for REST and Remote SQL',
       'WWV_FLOW_DATA_PROFILES'        , 'REST Source Data Profiles',
       'WWV_FLOW_DATA_PROFILE_COLS'    , 'REST Source Data Profile Columns',
       'WWV_FLOW_WEB_SRC_MODULES'      , 'REST Data Sources',
       'WWV_FLOW_WEB_SRC_OPERATIONS'   , 'REST Source Operations',
       'WWV_FLOW_WEB_SRC_SYNC_STEPS'   , 'REST Source Synchronization Steps',
       'WWV_FLOW_TASK_DEFS'            , 'Workflow Task Definitions',
       'WWV_FLOW_TASK_DEF_PARAMETERS'  , 'Workflow Task Definition Parameters',
       'WWV_FLOW_TASK_DEF_PARTICIPANTS', 'Workflow Task Participants',
       'WWV_FLOW_TASK_DEF_ACTIONS'     , 'Workflow Task Actions',
       'WWV_FLOW_TASK_DEF_COMP_PARAMS' , 'Workflow Task Definition Comp Parameters',
       a.flow_table) component,
       decode(FLOW_TABLE,
'WWV_FLOW_COMPANY_STATIC_FILES',(select sf.file_name from wwv_flow_company_static_files sf where sf.id=a.flow_table_pk and sf.security_group_id = a.security_group_id),
'WWV_FLOW_PAGE_PLUGS',(select pp.plug_name from wwv_flow_page_plugs pp where pp.id = a.flow_table_pk and pp.security_group_id = a.security_group_id),
'WWV_FLOW_ITEMS',(select fi.name from wwv_flow_items fi where fi.id = a.flow_table_pk and fi.security_group_id = a.security_group_id),
'WWV_FLOW_STEP_BRANCHES',(select nvl(sb.branch_name, substr(sb.BRANCH_ACTION,1,50)) from wwv_flow_step_branches sb where sb.id = a.flow_table_pk and sb.security_group_id = a.security_group_id),
'WWV_FLOW_PROCESSING',(select fp.process_name from wwv_flow_processing fp where fp.id = a.flow_table_pk and fp.security_group_id = a.security_group_id),
'WWV_FLOW_TOPLEVEL_TABS',(select tt.TAB_NAME from WWV_FLOW_TOPLEVEL_TABS tt where tt.id=a.flow_table_pk and tt.security_group_id = a.security_group_id),
'WWV_FLOW_LISTS_OF_VALUES_DATA',(select vd.lov_return_value from WWV_FLOW_LIST_OF_VALUES_DATA vd where vd.id=a.flow_table_pk and vd.security_group_id = a.security_group_id),
'WWV_FLOW_STEP_ITEM_HELP',(select si.name from wwv_flow_step_items si where (si.id) = (select ih.flow_item_id from WWV_FLOW_STEP_ITEM_HELP ih where ih.id=a.flow_table_pk and ih.security_group_id = a.security_group_id) and si.security_group_id = a.security_group_id),
'WWV_FLOW_DEVELOPERS',(select de.USERID from WWV_FLOW_DEVELOPERS de where de.id=a.flow_table_pk and de.security_group_id = a.security_group_id),
'WWV_FLOW_LISTS_OF_VALUES$',(select lv.LOV_NAME from WWV_FLOW_LISTS_OF_VALUES$ lv where lv.id=a.flow_table_pk and lv.security_group_id = a.security_group_id),
'WWV_FLOW_LIST_OF_VALUES_COLS',(select lc.query_column_name from wwv_flow_list_of_values_cols lc where lc.id = a.flow_table_pk and lc.security_group_id = a.security_group_id),
'WWV_FLOW_PAGE_DA_ACTIONS',(select da.action from wwv_flow_page_da_actions da where da.id=a.flow_table_pk and da.security_group_id = a.security_group_id),
'WWV_FLOW_PAGE_DA_EVENTS',(select de.name from wwv_flow_page_da_events de where de.id=a.flow_table_pk and de.security_group_id = a.security_group_id),
'WWV_FLOW_PAGE_PLUG_TEMPLATES',(select pt.PAGE_PLUG_TEMPLATE_NAME from WWV_FLOW_PAGE_PLUG_TEMPLATES pt where pt.id=a.flow_table_pk and pt.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGINS',(select pl.display_name from wwv_flow_plugins pl where pl.id=a.flow_table_pk and pl.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGIN_ATTRIBUTES',(select pa.prompt from wwv_flow_plugin_attributes pa where pa.id=a.flow_table_pk and pa.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGIN_STD_ATTRIBUTES',(select ps.name from wwv_flow_plugin_std_attributes ps where ps.id=a.flow_table_pk and ps.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGIN_ATTR_VALUES',(select pav.display_value from wwv_flow_plugin_attr_values pav where pav.id=a.flow_table_pk and pav.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGIN_FILES',(select pf.file_name from wwv_flow_plugin_files pf where pf.id=a.flow_table_pk and pf.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGIN_EVENTS',(select pe.display_name from wwv_flow_plugin_events pe where pe.id=a.flow_table_pk and pe.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGIN_ITEM_FILTERS',(select if.display_name from wwv_flow_plugin_item_filters if where if.id=a.flow_table_pk and if.security_group_id = a.security_group_id),
'WWV_FLOWS',flow_id,
'WWV_FLOW_STEP_COMPUTATIONS',(select co.COMPUTATION_ITEM from WWV_FLOW_STEP_COMPUTATIONS co where co.id=a.flow_table_pk and co.security_group_id = a.security_group_id),
'WWV_FLOW_ICON_BAR',(select ib.ICON_SEQUENCE||'. '||ib.ICON_SUBTEXT n from WWV_FLOW_ICON_BAR ib where ib.id=a.flow_table_pk and ib.security_group_id = a.security_group_id),
'WWV_FLOW_STEP_BUTTONS',(select sb.BUTTON_NAME||' "'||sb.BUTTON_IMAGE_ALT||'"' n from wwv_flow_step_buttons sb where sb.id=a.flow_table_pk and sb.security_group_id = a.security_group_id),
'WWV_FLOW_STEPS',flow_table_pk,
'WWV_FLOW_COMPUTATIONS',(select co.COMPUTATION_SEQUENCE||'. '||co.COMPUTATION_ITEM n from WWV_FLOW_COMPUTATIONS co where co.id=a.flow_table_pk and co.security_group_id = a.security_group_id),
'WWV_FLOW_APP_SETTINGS',(select se.name from wwv_flow_app_settings se where se.id = a.flow_table_pk and se.security_group_id = a.security_group_id),
'WWV_FLOW_TEMPLATES',(select te.NAME from WWV_FLOW_TEMPLATES te where te.id=a.flow_table_pk and te.security_group_id = a.security_group_id),
'WWV_FLOW_TABS',(select ta.TAB_NAME||' "'||ta.tab_text||'"' n from WWV_FLOW_TABS ta where ta.id=a.flow_table_pk and ta.security_group_id = a.security_group_id),
'WWV_FLOW_STEP_ITEMS',(select si.name from wwv_flow_step_items si where si.id = a.flow_table_pk and si.security_group_id = a.security_group_id),
'WWV_FLOW_STEP_ITEM_GROUPS',(select ig.label from wwv_flow_step_item_groups ig where ig.id = a.flow_table_pk and ig.security_group_id = a.security_group_id),
'WWV_FLOW_MESSAGES$',(select me.NAME from WWV_FLOW_MESSAGES$ me where me.id = a.flow_table_pk and me.security_group_id = a.security_group_id),
'WWV_FLOW_LIST_ITEMS',(select lt.LIST_ITEM_LINK_TEXT from wwv_flow_list_items lt where lt.id=a.flow_table_pk and lt.security_group_id = a.security_group_id),
'WWV_FLOW_STEP_VALIDATIONS',(select sv.VALIDATION_SEQUENCE||'. '||sv.VALIDATION_NAME n from WWV_FLOW_STEP_VALIDATIONS sv where sv.id = a.flow_table_pk and sv.security_group_id = a.security_group_id),
'WWV_FLOW_STEP_PROCESSING',(select pr.process_name from wwv_flow_step_processing pr where pr.id=a.flow_table_pk and pr.security_group_id = a.security_group_id),
'WWV_FLOW_ROW_TEMPLATES',(select rt.ROW_TEMPLATE_NAME from WWV_FLOW_ROW_TEMPLATES rt where rt.id=a.flow_table_pk and rt.security_group_id = a.security_group_id),
'WWV_FLOW_LISTS',(select li.NAME from WWV_FLOW_LISTS li where li.id=a.flow_table_pk and li.security_group_id = a.security_group_id),
'WWV_FLOW_THEMES',(select th.theme_id||'. '||th.THEME_NAME n from WWV_FLOW_THEMES th where th.id=a.flow_table_pk and th.security_group_id = a.security_group_id),
'WWV_FLOW_THEME_STYLES',(select ts.name from wwv_flow_theme_styles ts where ts.id=a.flow_table_pk and ts.security_group_id = a.security_group_id),
'WWV_FLOW_TEMPLATE_OPT_GROUPS',(select og.name from wwv_flow_template_opt_groups og where og.id=a.flow_table_pk and og.security_group_id = a.security_group_id),
'WWV_FLOW_TEMPLATE_OPTIONS',(select op.name from wwv_flow_template_options op where op.id=a.flow_table_pk and op.security_group_id = a.security_group_id),
'WWV_FLOW_THEME_FILES',(select tf.file_name from wwv_flow_theme_files tf where tf.id=a.flow_table_pk and tf.security_group_id = a.security_group_id),
'WWV_FLOW_PAGE_TMPL_DISP_POINTS',(select dp.name from wwv_flow_page_tmpl_disp_points dp where dp.id=a.flow_table_pk and dp.security_group_id = a.security_group_id),
'WWV_FLOW_PLUG_TMPL_DISP_POINTS',(select dp.name from wwv_flow_plug_tmpl_disp_points dp where dp.id=a.flow_table_pk and dp.security_group_id = a.security_group_id),
'WWV_FLOW_SHORTCUTS',(select sc.SHORTCUT_NAME from WWV_FLOW_SHORTCUTS sc where sc.id=a.flow_table_pk and sc.security_group_id = a.security_group_id),
'WWV_FLOW_MENU_OPTIONS',(select mo.short_name from wwv_flow_menu_options mo where mo.id = a.flow_table_pk and mo.security_group_id = a.security_group_id),
'WWV_FLOW_LIST_TEMPLATES',(select lt.LIST_TEMPLATE_NAME from WWV_FLOW_LIST_TEMPLATES lt where lt.id=a.flow_table_pk and lt.security_group_id = a.security_group_id),
'WWV_FLOW_SHARED_WEB_SERVICES',(select ws.NAME from WWV_FLOW_SHARED_WEB_SERVICES ws where ws.id = a.flow_table_pk and ws.security_group_id = a.security_group_id),
'WWV_FLOW_AUTHENTICATIONS',(select ah.NAME from WWV_FLOW_AUTHENTICATIONS ah where ah.id=a.flow_table_pk and ah.security_group_id = a.security_group_id),
'WWV_FLOW_SECURITY_SCHEMES',(select ss.NAME from WWV_FLOW_SECURITY_SCHEMES ss where ss.id = a.flow_table_pk and ss.security_group_id = a.security_group_id),
'WWV_FLOW_TREES',(select tr.TREE_NAME from WWV_FLOW_TREES tr where tr.id = a.flow_table_pk and tr.security_group_id = a.security_group_id),
'WWV_FLOW_SHARED_QUERIES',(select sq.NAME from WWV_FLOW_SHARED_QUERIES sq where sq.id = a.flow_table_pk and sq.security_group_id = a.security_group_id),
'WWV_FLOW_REPORT_LAYOUTS',(select rl.REPORT_LAYOUT_NAME from WWV_FLOW_REPORT_LAYOUTS rl where rl.id = a.flow_table_pk and rl.security_group_id = a.security_group_id),
'WWV_FLOW_LOAD_TABLES',(select lt.TABLE_NAME from WWV_FLOW_LOAD_TABLES lt where lt.id = a.flow_table_pk and lt.security_group_id = a.security_group_id),
'WWV_FLOW_LOAD_TABLE_LOOKUPS',(select tl.LOAD_COLUMN_NAME||' . '||tl.LOOKUP_TABLE_NAME n from WWV_FLOW_LOAD_TABLE_LOOKUPS tl where tl.id=a.flow_table_pk and tl.security_group_id = a.security_group_id),
'WWV_FLOW_LOAD_TABLE_RULES',(select tr.LOAD_COLUMN_NAME||' . '||tr.RULE_NAME||' . '||tr.RULE_TYPE n from WWV_FLOW_LOAD_TABLE_RULES tr where tr.id=a.flow_table_pk and tr.security_group_id = a.security_group_id),
'WWV_FLOW_UI_TYPES', (select ut.name from wwv_flow_ui_types ut where ut.id = a.flow_table_pk and ut.security_group_id = a.security_group_id),
'WWV_FLOW_USER_INTERFACES', (select ui.display_name from wwv_flow_user_interfaces ui where ui.id = a.flow_table_pk and ui.security_group_id = a.security_group_id),
'WWV_FLOW_STATIC_FILES',(select sf.file_name from wwv_flow_static_files sf where sf.id=a.flow_table_pk and sf.security_group_id = a.security_group_id),
'WWV_FLOW_PLUGIN_SETTINGS', (select ps.plugin from wwv_flow_plugin_settings ps where ps.id = a.flow_table_pk and ps.security_group_id = a.security_group_id),
'WWV_FLOW_REGION_COLUMNS', (select rc.name from wwv_flow_region_columns rc where rc.id = a.flow_table_pk and rc.security_group_id = a.security_group_id),
'WWV_REMOTE_SERVERS', (select rs.name from wwv_remote_servers rs where rs.id = a.flow_table_pk and rs.security_group_id = a.security_group_id),
'WWV_CREDENTIALS', (select cr.name from wwv_credentials cr where cr.id = a.flow_table_pk and cr.security_group_id = a.security_group_id),
'WWV_FLOW_DATA_PROFILES', (select dp.name from wwv_flow_data_profiles dp where dp.id = a.flow_table_pk and dp.security_group_id = a.security_group_id),
'WWV_FLOW_DATA_PROFILE_COLS', (select dpc.name from wwv_flow_data_profile_cols dpc where dpc.id = a.flow_table_pk and dpc.security_group_id = a.security_group_id),
'WWV_FLOW_WEB_SRC_MODULES', (select wm.name from wwv_flow_web_src_modules wm where wm.id = a.flow_table_pk and wm.security_group_id = a.security_group_id),
'WWV_FLOW_WEB_SRC_OPERATIONS', (select so.operation from wwv_flow_web_src_operations so where so.id = a.flow_table_pk and so.security_group_id = a.security_group_id),
'WWV_FLOW_WEB_SRC_SYNC_STEPS', (select ss.static_id from wwv_flow_web_src_sync_steps ss where ss.id = a.flow_table_pk and ss.security_group_id = a.security_group_id),
'WWV_FLOW_TASK_DEFS', (select td.id from wwv_flow_task_defs td where td.id = a.flow_table_pk and td.security_group_id = a.security_group_id),
'WWV_FLOW_TASK_DEF_PARAMS', (select tp.id from wwv_flow_task_def_params tp where tp.id = a.flow_table_pk and tp.security_group_id = a.security_group_id),
'WWV_FLOW_TASK_DEF_PARTICIPANTS', (select tps.id from wwv_flow_task_def_participants tps where tps.id = a.flow_table_pk and tps.security_group_id = a.security_group_id),
'WWV_FLOW_TASK_DEF_ACTIONS', (select ta.id from wwv_flow_task_def_actions ta where ta.id = a.flow_table_pk and ta.security_group_id = a.security_group_id),
'WWV_FLOW_TASK_DEF_COMP_PARAMS', (select tcp.id from wwv_flow_task_def_comp_params tcp where tcp.id = a.flow_table_pk and tcp.security_group_id = a.security_group_id),
FLOW_TABLE_PK) component_name,
    a.object_name,
    a.security_group_id   workspace_id,
    a.flow_table_pk       component_id
from
     wwv_flow_builder_audit_trail a,
     wwv_flow_authorized    f
where f.workspace_id       = a.security_group_id
  and f.application_id     = a.flow_id
  /* keep this not exists */
  and not exists ( select 1
                     from wwv_flow_language_map m
                    where m.translation_flow_id = f.application_id
                      and m.security_group_id   = f.workspace_id
                      and f.authentication_id   <> trunc(f.authentication_id) )
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_developer_activity_log',
    p_table_comment     => 'Identifies developer changes to applications',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'        , 'Primary key that identifies the workspace',
      'workspace_display_name' , 'Display name for the workspace',
      'workspace'           , 'A work area mapped to one or more database schemas',
      'application_id'      , 'Application Primary Key, Unique over all workspaces',
      'application_name'    , 'Identifies the application',
      'DEVELOPER'           , 'Developer who made the change',
      'AUDIT_DATE'          , 'Date of application component change',
      'AUDIT_ACTION'        , 'Insert, Update or Delete',
      'PAGE_ID'             , 'Identifies page number if component corresponds to a specific page',
      'PAGE_NAME'           , 'Identifies corresponding page name',
      'COMPONENT'           , 'The type of component changed',
      'COMPONENT_NAME'      , 'The name of the affected component',
      'COMPONENT_ID'        , 'The unique ID of the affected component',
      'OBJECT_NAME'         , 'The name of the affected component (stored at time of action)'
    )
  );
end;
/

prompt ...apex_release
create or replace view apex_release
as
select (select wwv_flows_release from sys.dual) version_no,
       (select wwv_flows_version from sys.dual) api_compatibility,
       case
       when apex_patch_status is null or apex_patch_status = 'APPLIED'
       then wwv_patch_applied
       else apex_patch_status
       end patch_applied
  from (select min(value) apex_patch_status
          from wwv_flow_platform_prefs
         where name = 'APEX_PATCH_STATUS')
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_release',
    p_table_comment     => 'Identifies the currently installed Oracle APEX release',
    p_column_comments   => wwv_flow_t_varchar2 (
      'version_no'                         , 'The specific version number of this Oracle APEX instance',
      'api_compatibility'                  , 'The version of the API that this release is compatible with for importing applications or components',
      'patch_applied'                      , 'Reserved for internal use during Oracle APEX upgrade'
    )
  );
end;
/
--------------------------------------------
-- UI DEFAULTS
--
prompt ...apex_ui_defaults_tables
create or replace view apex_ui_defaults_tables as
select i.schema,
       i.table_name,
       i.form_region_title,
       i.report_region_title,
       i.created_by,
       i.created_on,
       i.last_updated_by,
       i.last_updated_on
  from wwv_flow_current_sgid sgid,
       wwv_flow_hnt_table_info i
 where (    sgid.has_apex_admin_read_role_yn = 'Y'
         or (    sgid.cu = i.schema
             and (   sgid.nls_sort='BINARY'
                  or nlssort(i.schema,'NLS_SORT=BINARY')=nlssort(sgid.cu,'NLS_SORT=BINARY')))
         or exists ( select null
                       from wwv_flow_company_schemas s
                      where s.security_group_id = sgid.security_group_id
                        and s.schema = i.schema
                        and (   sgid.nls_sort='BINARY'
                             or nlssort(s.schema,'NLS_SORT=BINARY')=nlssort(i.schema,'NLS_SORT=BINARY'))))
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_ui_defaults_tables',
    p_table_comment     => 'The User Interface Defaults for the tables within this schema.  Used by the wizards when generating applications.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'schema' ,              'Schema owning table.',
      'table_name' ,          'Name of table in the schema.',
      'form_region_title' ,   'When creating a form based upon this table, this title will be used as the resulting region title.',
      'report_region_title' , 'When creating a report or tabular form based upon this table, this title will be used as the resulting region title.',
      'created_by' ,          'Auditing; user that created the record.',
      'created_on' ,          'Auditing; date the record was created.',
      'last_updated_by' ,     'Auditing; user that last modified the record.',
      'last_updated_on' ,     'Auditing; date the record was last modified.'
    )
  );
end;
/

prompt ...apex_ui_defaults_groups
create or replace view apex_ui_defaults_groups as
select t.schema,
       t.table_name,
       g.group_name,
       g.description,
       g.display_sequence,
       g.created_by,
       g.created_on,
       g.last_updated_by,
       g.last_updated_on
  from wwv_flow_current_sgid sgid,
       wwv_flow_hnt_groups g,
       wwv_flow_hnt_table_info t
 where g.table_id = t.table_id
   and (    sgid.has_apex_admin_read_role_yn = 'Y'
         or (    sgid.cu = t.schema
             and (   sgid.nls_sort='BINARY'
                  or nlssort(t.schema,'NLS_SORT=BINARY')=nlssort(sgid.cu,'NLS_SORT=BINARY')))
         or exists ( select null
                       from wwv_flow_company_schemas s
                      where s.security_group_id = sgid.security_group_id
                        and s.schema = t.schema
                        and (   sgid.nls_sort='BINARY'
                             or nlssort(s.schema,'NLS_SORT=BINARY')=nlssort(t.schema,'NLS_SORT=BINARY'))))
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_ui_defaults_groups',
    p_table_comment     => 'The User Interface Defaults for the groups within the tables in this schema.  Used by the wizards when generating applications.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'schema' ,           'Schema owning table.',
      'table_name' ,       'Name of table in the schema.',
      'group_name' ,       'When creating a form based upon this table, this will be used to group columns together into regions and this will be used as the region title.  When creating an interactive report against this table, this will be used to group the columns together and this will be used as the group name.',
      'description' ,      'Description of the group.  Not used during generation.',
      'display_sequence' , 'Used to provide sequence of regions within a form and groups within an interactive report.',
      'created_by' ,       'Auditing; user that created the record.',
      'created_on' ,       'Auditing; date the record was created.',
      'last_updated_by' ,  'Auditing; user that last modified the record.',
      'last_updated_on' ,  'Auditing; date the record was last modified.'
    )
  );
end;
/

prompt ...apex_ui_defaults_columns
create or replace view apex_ui_defaults_columns as
select t.schema,
       t.table_name,
       c.column_name,
       c.label,
       c.help_text,
       g.group_name,
       c.mask_form,
       c.display_seq_form,
       c.display_in_form,
       c.display_as_form,
       c.form_attribute_01,
       c.form_attribute_02,
       c.form_attribute_03,
       c.form_attribute_04,
       c.form_attribute_05,
       c.form_attribute_06,
       c.form_attribute_07,
       c.form_attribute_08,
       c.form_attribute_09,
       c.form_attribute_10,
       c.form_attribute_11,
       c.form_attribute_12,
       c.form_attribute_13,
       c.form_attribute_14,
       c.form_attribute_15,
       c.display_as_tab_form,
       c.display_seq_report,
       c.display_in_report,
       c.display_as_report,
       c.mask_report,
       c.aggregate_by,
       c.default_value,
       c.required,
       c.alignment,
       c.display_width,
       c.max_width,
       c.height,
       c.group_by,
       c.searchable,
       c.lov_query,
       c.created_by,
       c.created_on,
       c.last_updated_by,
       c.last_updated_on
  from wwv_flow_current_sgid sgid,
       wwv_flow_hnt_column_info c,
       wwv_flow_hnt_table_info t,
       wwv_flow_hnt_groups g
 where c.table_id = t.table_id
   and c.group_id = g.group_id (+)
   and (    sgid.has_apex_admin_read_role_yn = 'Y'
         or (    sgid.cu = t.schema
             and (   sgid.nls_sort='BINARY'
                  or nlssort(t.schema,'NLS_SORT=BINARY')=nlssort(sgid.cu,'NLS_SORT=BINARY')))
         or exists ( select null
                       from wwv_flow_company_schemas s
                      where s.security_group_id = sgid.security_group_id
                        and s.schema = t.schema
                        and (   sgid.nls_sort='BINARY'
                             or nlssort(s.schema,'NLS_SORT=BINARY')=nlssort(t.schema,'NLS_SORT=BINARY'))))
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_ui_defaults_columns',
    p_table_comment     => 'The User Interface Defaults for the columns within this schema.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'schema' ,              'Schema owning table.',
      'table_name' ,          'The table associated with the user interface defaults.',
      'column_name' ,         'The column associated with the user interface defaults.',
      'group_name' ,          'The group associated with this column.  Will be used for creating regions in forms and groups in interactive reports.',
      'label' ,               'When creating a form against this table or view, this will be used as the label for the item if this column is included. When creating a report or tabular form, this will be used as the column heading if this column is included.',
      'help_text' ,           'When creating a form against this table or view, this becomes the help text for the resulting item.',
      'mask_form' ,           'When creating a form against this table or view, this specifies the mask that will be applied to the item, such as 999-99-9999. This is not used for character based items.',
      'display_seq_form' ,    'When creating a form against this table or view, this determines the sequence in which the columns will be displayed in the resulting form page.',
      'display_in_form' ,     'When creating a form against this table or view, this determines whether this column will be displayed in the resulting form page.',
      'display_as_form' ,     'When creating a form against this table or view, this determines the way the column will be displayed, such as text area or text field.',
      'form_attribute_01' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_02' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_03' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_04' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_05' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_06' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_07' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_08' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_09' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_10' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_11' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_12' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_13' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_14' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'form_attribute_15' ,   'When creating a form against this table or view, this dynamic attribute stores additional information for item type specified in display_as_form.',
      'display_as_tab_form' , 'When creating a tabular form against this table or view, this determines the way the column will be displayed, such as select list or popup LOV.',
      'display_seq_report' ,  'When creating a report against this table or view, this determines the sequence in which the columns will be displayed in the resulting report.',
      'display_in_report' ,   'When creating a report against this table or view, this determines whether this column will be displayed in the resulting report.',
      'display_as_report' ,   'When creating a report against this table or view, this determines the way the column will be displayed, such as Standard Report Column or Display as Text (based on LOV).',
      'mask_report' ,         'When creating a report against this table or view, this specifies the mask that will be applied against the data, such as 999-99-9999. This is not used for character based items.',
      'aggregate_by' ,        'Obsolete.',
      'default_value' ,       'When creating a form against this table or view, this specifies the default value for the item resulting from this column.',
      'required' ,            'When creating a form against this table or view, this specifies to generate a validation in which the resulting item must be NOT NULL.',
      'alignment' ,           'When creating a report against this table or view, this determines the alignment for the resulting report column (left, center, or right).',
      'display_width' ,       'When creating a form against this table or view, this specifies the display width of the item resulting from this column.',
      'max_width' ,           'When creating a form against this table or view, this specifies the maximum string length that a user is allowed to enter in the item resulting from this column.',
      'height' ,              'When creating a form against this table or view, this specifies the display height of the item resulting from this column.',
      'group_by' ,            'Obsolete.',
      'searchable' ,          'Obsolete.',
      'lov_query' ,           'A query to be turned into a Named List of Values if this column is included in a Form, Report or Tabular Form and the column is displayed as a type that uses a List of Values (such as Radio Group or Select List).',
      'created_by' ,          'Auditing; user that created the record.',
      'created_on' ,          'Auditing; date the record was created.',
      'last_updated_by' ,     'Auditing; user that last modified the record.',
      'last_updated_on' ,     'Auditing; date the record was last modified.'
    )
  );
end;
/

prompt ...apex_ui_defaults_lov_data
create or replace view apex_ui_defaults_lov_data as
select t.schema,
       t.table_name,
       c.column_name,
       l.lov_disp_sequence,
       l.lov_disp_value,
       l.lov_return_value,
       l.last_updated_by,
       l.last_updated_on
  from wwv_flow_current_sgid sgid,
       wwv_flow_hnt_lov_data l,
       wwv_flow_hnt_column_info c,
       wwv_flow_hnt_table_info t
 where l.column_id = c.column_id
   and c.table_id  = t.table_id
   and (    sgid.has_apex_admin_read_role_yn = 'Y'
         or (    sgid.cu = t.schema
             and (   sgid.nls_sort='BINARY'
                  or nlssort(t.schema,'NLS_SORT=BINARY')=nlssort(sgid.cu,'NLS_SORT=BINARY')))
         or exists ( select null
                       from wwv_flow_company_schemas s
                      where s.security_group_id = sgid.security_group_id
                        and s.schema = t.schema
                        and (   sgid.nls_sort='BINARY'
                             or nlssort(s.schema,'NLS_SORT=BINARY')=nlssort(t.schema,'NLS_SORT=BINARY'))))
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_ui_defaults_lov_data',
    p_table_comment     => 'If you create a form, report, or tabular form that includes this column and if the appropriate Display As Type is set to use a list of values (Radio Group or Select List) then a Named List of Values will be created within the application and will be referenced by the resulting item or report column.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'schema' , 'Schema owning table.',
      'table_name' , 'The table associated with the user interface defaults.',
      'column_name' , 'The column associated with the user interface defaults.',
      'lov_disp_sequence' , 'The display sequence of the static list of values record.',
      'lov_disp_value' , 'The display value of the static list of values record.',
      'lov_return_value' , 'The return value of the static list of values record.',
      'last_updated_by' , 'Auditing; user that last modified the record.',
      'last_updated_on' , 'Auditing; date the record was last modified.'
    )
  );
end;
/

prompt ...apex_ui_defaults_attr_dict
create or replace view apex_ui_defaults_attr_dict as
select
    w.workspace,
    w.workspace_display_name,
    --
    syn.syn_name column_name,
    case when c.column_name != syn.syn_name
         then c.column_name
         end synonym_of,
    c.label,
    c.help_text,
    c.format_mask,
    c.default_value,
    c.form_format_mask,
    c.form_display_width,
    c.form_display_height,
    c.form_data_type,
    c.report_format_mask,
    c.report_col_alignment,
    c.created_by,
    c.created_on,
    c.last_updated_by,
    c.last_updated_on
from wwv_flow_hnt_column_dict c,
     wwv_flow_hnt_col_dict_syn syn,
     wwv_flow_current_workspaces w
where w.workspace_id = syn.security_group_id
  and syn.security_group_id = c.security_group_id
  and syn.column_id = c.column_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_ui_defaults_attr_dict',
    p_table_comment     => 'The Attribute Dictionary is specific to a workspace.  It is part of User Interface Defaults and can be used in report and form creation.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace' ,            'A work area mapped to one or more database schemas',
      'workspace_display_name' , 'Display name for the workspace',
      'column_name' ,          'Name of column or synonym.  Used to match against columns when UI Defaults are used by create wizards.',
      'synonym_of' ,           'If a synonym, the name of the base column that it is a synonym of.',
      'label' ,                'Used for item label and report column heading.',
      'help_text' ,            'Used for help text for items and interactive report columns.',
      'format_mask' ,          'Used as the format mask for items and report columns.  Can be overwritten by report for form specific format masks.',
      'default_value' ,        'Used as the default value for items.',
      'form_format_mask' ,     'If provided, used as the format mask for items, overriding any value for the general format mask.',
      'form_display_width' ,   'Used as the width of any items using this Attribute Definition.',
      'form_display_height' ,  'Used as the height of any items using this Attribute Definition (only used by item types such as textares and shuttles).',
      'form_data_type' ,       'Used as the data type for items (specifies whether this item contains VARCHAR or NUMBER data which results in an automatic validation).',
      'report_format_mask' ,   'If provided, used as the format mask for report columns, overriding any value for the general format mask.',
      'report_col_alignment' , 'Used as the alignment for report column data (e.g. number are usually right justified).',
      'created_by' ,           'User that created this column.',
      'created_on' ,           'Date this column was created.',
      'last_updated_by' ,     'APEX developer who made last update',
      'last_updated_on' ,     'Date of last update'
    )
  );
end;
/

create or replace view apex_themes as
select x.theme_number,
       x.theme_name,
       x.theme_type,
       x.workspace,
       x.workspace_display_name,
       x.workspace_id
from (-- repository theme
      select 'Theme - '||i theme_name,
             i theme_number,
             'BUILTIN' theme_type,
             null workspace,
             null workspace_display_name,
             null workspace_id
      from wwv_flow_dual100
      where i <=20
      union all
      -- public theme
      select distinct wwv_flow_escape.html(t.theme_name)||' - '||t.theme_id theme_name,
             theme_id theme_number,
             'PUBLIC' theme_type,
             null workspace,
             null workspace_display_name,
             null workspace_id
        from wwv_flow_themes t
       where t.security_group_id = 11
         and t.theme_security_group_id is null
      union all
      -- workspace theme
      select distinct wwv_flow_escape.html(t.theme_name)||' - '||t.theme_id theme_name,
             theme_id theme_number,
             'WORKSPACE' theme_type,
             w.workspace,
             w.workspace_display_name,
             w.workspace_id
      from wwv_flow_themes t,
           wwv_flow_current_workspaces w
     where t.theme_security_group_id = w.workspace_id
       and t.security_group_id       = 11
      ) x
order by 1
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_themes',
    p_table_comment     => 'List of APEX built-in, public and workspace themes',
    p_column_comments   => wwv_flow_t_varchar2 (
      'theme_number'             , 'Identifies the theme number associated with all templates within the theme',
      'theme_name'               , 'Identifies the name of the theme',
      'theme_type'               , 'Identifies the theme type as APEX built-in, public or workspace themes',
      'workspace'                , 'Identifies a workspace where the theme exists',
      'workspace_display_name'   , 'Display name for the workspace',
      'workspace_id'             , 'Identifies the primary key of the workspace where the theme exists'
    )
  );
end;
/

---------------------------------
-- Team Development views
--  only feedback is within apex_views, rest are defined within team_tab.sql
--

prompt ...apex_team_feedback
create or replace view apex_team_feedback
as
select
    w.workspace_id,
    w.workspace_name,
    w.workspace_display_name,
    --
    f.ID                        feedback_id,
    f.deployment_system||
    f.feedback_id               feedback_number,
    f.FEEDBACK_COMMENT          feedback,
    f.FEEDBACK_TYPE             feedback_type,
    f.FEEDBACK_STATUS           feedback_status,
    f.feedback_rating           feedback_rating,
    f.deployment_system         deployment_system,
    --
    f.APPLICATION_ID            application_id,
    f.APPLICATION_NAME          application_name,
    f.APPLICATION_VERSION       application_version,
    f.PAGE_ID                   page_id,
    f.PAGE_NAME                 page_name,
    f.PAGE_LAST_UPDATED_BY      page_last_updated_by,
    f.PAGE_LAST_UPDATED_ON      page_last_updated,
    f.SESSION_ID                logging_session_id,
    f.APEX_USER                 logging_apex_user,
    f.user_email                logging_user_email,
    --
    f.SESSION_STATE,
    f.PARSING_SCHEMA,
    f.screen_width,
    f.screen_height,
    f.HTTP_USER_AGENT,
    f.REMOTE_ADDR,
    f.REMOTE_USER,
    f.HTTP_HOST,
    f.SERVER_NAME,
    f.SERVER_PORT,
    --
    f.LOGGED_BY_WORKSPACE_NAME   logging_workspace_name,
    f.LOGGING_SECURITY_GROUP_ID  logging_workspace_id,
    f.LOGGING_EMAIL,
    --
    f.SESSION_INFO               session_info,
    f.tags                       tags,
    f.DEVELOPER_COMMENT          developer_comment,
    f.public_response            public_response,
    f.BUG_ID                     logged_as_bug_id,
    f.FEATURE_ID                 logged_as_feature_id,
    f.TASK_ID                    logged_as_todo_id,
    f.ISSUE_ID   				 logged_as_issue_id,
    --
    f.LABEL_01,
    f.LABEL_02,
    f.LABEL_03,
    f.LABEL_04,
    f.LABEL_05,
    f.LABEL_06,
    f.LABEL_07,
    f.LABEL_08,
    --
    f.ATTRIBUTE_01,
    f.ATTRIBUTE_02,
    f.ATTRIBUTE_03,
    f.ATTRIBUTE_04,
    f.ATTRIBUTE_05,
    f.ATTRIBUTE_06,
    f.ATTRIBUTE_07,
    f.ATTRIBUTE_08,
    --
    f.CREATED_BY,
    f.CREATED_ON,
    f.UPDATED_BY,
    f.UPDATED_ON
from
    wwv_flow_feedback           f,
    wwv_flow_current_workspaces w
where
    f.security_group_id = w.workspace_id
/


begin
  wwv_flow_upgrade.add_table_comments (
    p_table_name        => 'apex_team_feedback',
    p_table_comment     => 'Identifies user feedback.',
    p_column_comments   => wwv_flow_t_varchar2 (
      'workspace_id'        , 'Primary key that identifies the workspace.',
      'workspace_name'      , 'Name of the current workspace.',
      'workspace_display_name' , 'Display name for the workspace',
      'feedback_id'         , 'Primary key of the feedback entry.',
      'feedback_number'     , 'External identifier for the feedback entry.',
      'feedback'            , 'Comment provided by application user',
      'tags'                , 'Tags associated with the feedback.',
      'DEVELOPER_COMMENT'   , 'Comments or notes for internal use, not typically viewable by end-users.',
      'public_response'     , 'The public response to the feedback.',
      'deployment_system'   , 'Identifies the system where the feedback entry has been created.',
      'feedback_type'       , 'The type of feedback (identified by the user).',
      'feedback_status'     , 'The status of the Feedback as assigned by the reviewing developers.',
      'feedback_rating'     , 'The rating associated with the feedback.  Typically 1-3 (low to high).',

      'application_id'        , 'Unique identifier of the application the feedback was provided on.',
      'application_name'      , 'Name of the application the Feedback was provided on.',
      'application_version'   , 'Version of the application the feedback was provided on.',
      'page_id'               , 'Unique identifier of the page within the application that the feedback was provided on.',
      'page_name'             , 'Name of the page within the application that the feedback was provided on.',
      'page_last_updated_by'  , 'The developer that last updated the referenced page.',
      'page_last_updated'     , 'The date on which the referenced page was last updated.',
      'logging_session_id'    , 'The Session ID when the feedback was created.',
      'logging_apex_user'     , 'Oracle APEX user providing the feedback.',
      'logging_user_email'    , 'Email address of Oracle APEX user providing the feedback.',

      'session_state'           , 'The session state when the feedback was created.',
      'parsing_schema'          , 'The parsing schema of the application when the feedback was created.',
      'screen_width'            , 'Screen width when feedback was created.',
      'screen_height'           , 'Screen height when feedback was created.',
      'http_user_agent'         , 'HTTP User Agent where the feedback was provided (can be different that where the feedback is being reviewed).',
      'remote_addr'             , 'Remote Address where the feed