Rem  Copyright (c) 2006, 2022, Oracle and/or its affiliates.
Rem


Rem    NAME
Rem      apex_install_pe_data.sql
Rem


Rem    DESCRIPTION
Rem      Insert Property Editor metadata into wwv_pe_* tables during APEX installation
Rem


Rem    MODIFIED     (MM/DD/YYYY)
Rem    arayner       05/13/2013 - Created

set define '^'
set sqlprefix off
set serveroutput on

prompt Enter YES (or hit enter) to load system messages and NO to skip it (NO should only be used during installation)
define LOAD_SYSTEM_MESSAGES = '^1'

prompt
prompt Property Editor - Metadata
prompt

declare
/* Useful SQL queries:

Find existing properties that could be re-used:

select p.name,
       wwv_flow_lang.system_message(p.prompt) as property_prompt,
       p.property_type,
       p.db_column,
       wwv_flow_lang.system_message(dg.title) as group_title,
       p.*
  from wwv_pe_properties p,
       wwv_pe_display_groups dg
 where p.name like '%FEEDB%'
   and dg.id = p.display_group_id
 order by p.name;

Show all properties of a component type (useful to verify the order and groups):

select wwv_flow_lang.system_message(dg.title) as group_title,
       wwv_flow_lang.system_message(p.prompt) as property_prompt,
       p.name,
       coalesce( cp.db_column, p.db_column )  as db_column
  from wwv_pe_component_types ct,
       wwv_pe_component_props cp,
       wwv_pe_properties p,
       wwv_pe_display_groups dg
 where ct.name              = 'USER_INTERFACE'
   and cp.component_type_id = ct.id
   and p.id                 = cp.property_id
   and dg.id                = p.display_group_id
  order by cp.display_sequence;

Show columns that are missing in the current configuration:

select ct.name, ct.table_name, c.column_name
  from wwv_pe_component_types ct,
       user_tab_cols c
 where c.table_name = ct.table_name
   and c.table_name not in ( 'WWV_FLOW_FILTERS_DEV',
                             'WWV_FLOW_FND_USER_GROUPS'
                           )
   and ct.name      not in ( 'DATA_LOAD_TABLE',
                             'WEB_SRC_SYNC_STEP_WS_PARAM',
                             'AUTOMATION_WS_PARAM',
                             'LOV_WS_PARAM',
                             'CALENDAR_TEMPLATE', -- legacy calendar
                             'PAGE',
                             'TREE', -- legacy tree
                             'USER_INTERFACE'
                           )
   and ct.is_query_only  = 'N'
   and ct.page_id_column is null -- no page meta data, they are valid
   and not exists ( select 1
                      from wwv_pe_component_props cp,
                           wwv_pe_properties p
                     where cp.component_type_id = ct.id
                       and p.id                 = cp.property_id
                       and coalesce( cp.db_column, p.db_column ) = c.column_name
                  )
   and c.column_name not in ( 'ID',
                              'FLOW_ID',
                              'PAGE_ID',
                              'SECURITY_GROUP_ID',
                              'CREATED_BY',
                              'CREATED_ON',
                              'APPLICATION_TYPE',
                              'EMAIL_TEMPLATE_ID',
                              'TASK_DEF_ID',
                              'WEB_SRC_MODULE_ID',
                              'WEB_SRC_OPERATION_ID',
                              'FILE_CONTENT',
                              'LAST_REFRESH',
                              'LAST_REFRESHED'
                            )
   and not (   ct.table_name = 'WWV_FLOWS'
           and c.column_name like 'DEFAULT_%' -- not used anymore
           )
   and c.column_name not like 'ATTRIBUTE___'
   and c.column_name not like '%_COMPILED'
   and c.column_name not like 'SYS_NC%'
   and c.column_name not like '%_RUN_TIMESTAMP'
   and c.column_name not like 'ATTRIBUTE_' -- build_option
   and c.column_name <> ct.parent_fk_column
 order by ct.name, c.column_name;

*/
    type t_parent_component_types is table of number index by binary_integer;
    type t_used_system_messages is table of boolean index by wwv_flow_messages$.name%type;

    c_load_system_messages          constant boolean := ( nvl( upper( '^LOAD_SYSTEM_MESSAGES' ), 'YES' ) = 'YES' );
    ------------------------------------------------------------------------
    -- Component Type Constants
    ------------------------------------------------------------------------
    -- Workspace component types
    c_comp_type_credential          constant wwv_pe_component_types.id%type :=   10;
    c_comp_type_remote_database     constant wwv_pe_component_types.id%type :=   20;
    c_comp_type_remote_oracle_db    constant wwv_pe_component_types.id%type :=   22;
    c_comp_type_auth_server         constant wwv_pe_component_types.id%type :=   30;
    c_comp_type_print_server        constant wwv_pe_component_types.id%type :=   40;
    c_comp_type_rest_server         constant wwv_pe_component_types.id%type :=   50;
    c_comp_type_workspace_file      constant wwv_pe_component_types.id%type :=   60;
    c_comp_type_app_group           constant wwv_pe_component_types.id%type :=   70;
    --
    -- Application component types
    c_comp_type_application         constant wwv_pe_component_types.id%type := 1000;
    c_comp_type_app_substitution    constant wwv_pe_component_types.id%type := 1005;
    --
    -- Theme related component types
    c_comp_type_theme               constant wwv_pe_component_types.id%type := 2000;
    --
    -- Templates
    c_comp_type_page_template       constant wwv_pe_component_types.id%type := 2510;
    c_comp_type_field_template      constant wwv_pe_component_types.id%type := 2520;
    c_comp_type_button_template     constant wwv_pe_component_types.id%type := 2530;
    c_comp_type_region_template     constant wwv_pe_component_types.id%type := 2540;
    c_comp_type_list_template       constant wwv_pe_component_types.id%type := 2550;
    c_comp_type_bc_template         constant wwv_pe_component_types.id%type := 2560;
    c_comp_type_calendar_template   constant wwv_pe_component_types.id%type := 2570;
    c_comp_type_report_template     constant wwv_pe_component_types.id%type := 2580;
    --
    -- Shared component related component types
    c_comp_type_app_item            constant wwv_pe_component_types.id%type := 3010;
    c_comp_type_app_computation     constant wwv_pe_component_types.id%type := 3020;
    c_comp_type_app_process         constant wwv_pe_component_types.id%type := 3030;
    c_comp_type_build_option        constant wwv_pe_component_types.id%type := 3040;
    c_comp_type_authentication      constant wwv_pe_component_types.id%type := 3050;
    c_comp_type_authorization       constant wwv_pe_component_types.id%type := 3060;
    c_comp_type_acl_role            constant wwv_pe_component_types.id%type := 3070;
    --
    c_comp_type_web_src_module      constant wwv_pe_component_types.id%type := 3080;
    c_comp_type_web_src_mod_param   constant wwv_pe_component_types.id%type := 3082;
    c_comp_type_web_src_mod_dp      constant wwv_pe_component_types.id%type := 3084;
    c_comp_type_web_src_mod_dp_col  constant wwv_pe_component_types.id%type := 3086;
    c_comp_type_web_src_operation   constant wwv_pe_component_types.id%type := 3088;
    c_comp_type_web_src_oper_param  constant wwv_pe_component_types.id%type := 3090;
    c_comp_type_web_src_oper_dp     constant wwv_pe_component_types.id%type := 3092;
    c_comp_type_web_src_oper_dpcol  constant wwv_pe_component_types.id%type := 3094;
    c_comp_type_web_src_sync_step   constant wwv_pe_component_types.id%type := 3096;
    c_comp_type_web_src_sync_param  constant wwv_pe_component_types.id%type := 3098;  -- web source sync step parameter
    c_comp_type_web_src_query       constant wwv_pe_component_types.id%type := 3100;
    c_comp_type_web_src_param       constant wwv_pe_component_types.id%type := 3102;
    --
    c_comp_type_shortcut            constant wwv_pe_component_types.id%type := 3105;
    --
    c_comp_type_plugin              constant wwv_pe_component_types.id%type := 3110;
    --
    c_comp_type_application_file    constant wwv_pe_component_types.id%type := 3150;
    c_comp_type_text_message        constant wwv_pe_component_types.id%type := 3170;
    c_comp_type_app_setting         constant wwv_pe_component_types.id%type := 3180;
    --
    c_comp_type_classic_nav_bar     constant wwv_pe_component_types.id%type := 3505;
    c_comp_type_breadcrumb          constant wwv_pe_component_types.id%type := 3510;
    c_comp_type_breadcrumb_entry    constant wwv_pe_component_types.id%type := 3515;
    c_comp_type_list                constant wwv_pe_component_types.id%type := 3520;
    c_comp_type_list_entry          constant wwv_pe_component_types.id%type := 3525;
    c_comp_type_lov                 constant wwv_pe_component_types.id%type := 3530;
    c_comp_type_lov_ws_param        constant wwv_pe_component_types.id%type := 3532;  -- web source lov parameter
    c_comp_type_lov_entry           constant wwv_pe_component_types.id%type := 3534;
    c_comp_type_lov_column          constant wwv_pe_component_types.id%type := 3536;
    c_comp_type_ws_ref              constant wwv_pe_component_types.id%type := 3540;
    c_comp_type_ws_ref_operations   constant wwv_pe_component_types.id%type := 3541;
    c_comp_type_ws_ref_oper_p_i     constant wwv_pe_component_types.id%type := 3545;  -- web service input parameters
    c_comp_type_ws_ref_oper_p_o     constant wwv_pe_component_types.id%type := 3546;  -- web service output parameters
    c_comp_type_ws_ref_oper_p_a     constant wwv_pe_component_types.id%type := 3547;  -- web service basic auth parameters
    c_comp_type_ws_ref_oper_p_h     constant wwv_pe_component_types.id%type := 3548;  -- web service http header parameters
    c_comp_type_legacy_data_load    constant wwv_pe_component_types.id%type := 3550;
    -- placeholder: child tables of data load
    c_comp_type_tab_set             constant wwv_pe_component_types.id%type := 3560;
    c_comp_type_report_layout       constant wwv_pe_component_types.id%type := 3570;
    c_comp_type_report_query        constant wwv_pe_component_types.id%type := 3580;
    c_comp_type_report_query_stmt   constant wwv_pe_component_types.id%type := 3585;
    -- placeholder: report layout details
    c_comp_type_automation          constant wwv_pe_component_types.id%type := 3610;
    c_comp_type_autom_ws_param      constant wwv_pe_component_types.id%type := 3612;
    c_comp_type_automation_action   constant wwv_pe_component_types.id%type := 3620;
    c_comp_type_email_template      constant wwv_pe_component_types.id%type := 3630;
    c_comp_type_data_load_def       constant wwv_pe_component_types.id%type := 3640;
    c_comp_type_data_load_dp        constant wwv_pe_component_types.id%type := 3645;
    c_comp_type_data_load_dp_col    constant wwv_pe_component_types.id%type := 3647;
    -- Task definition component type
    c_comp_type_task_def            constant wwv_pe_component_types.id%type := 3700;
    c_comp_type_task_def_param      constant wwv_pe_component_types.id%type := 3705;
    c_comp_type_task_def_action     constant wwv_pe_component_types.id%type := 3710;
    c_comp_type_task_def_particip   constant wwv_pe_component_types.id%type := 3715;
    --
    c_comp_type_language_map        constant wwv_pe_component_types.id%type := 3890;
    c_comp_type_dyn_translation     constant wwv_pe_component_types.id%type := 3891;
    --
    c_comp_type_user_interface      constant wwv_pe_component_types.id%type := 3900;
    c_comp_type_concatenated_file   constant wwv_pe_component_types.id%type := 3905;
    --
    -- Misc component types
    c_comp_type_page_group          constant wwv_pe_component_types.id%type := 4010;
    c_comp_type_tree                constant wwv_pe_component_types.id%type := 4015;
    --
    -- Supporting Objects
    c_comp_type_supobj              constant wwv_pe_component_types.id%type := 4950;
    c_comp_type_supobj_substitut    constant wwv_pe_component_types.id%type := 4955;
    c_comp_type_supobj_build_opt    constant wwv_pe_component_types.id%type := 4960;
    c_comp_type_supobj_validation   constant wwv_pe_component_types.id%type := 4965;
    c_comp_type_supobj_install      constant wwv_pe_component_types.id%type := 4970;
    c_comp_type_supobj_upgrade      constant wwv_pe_component_types.id%type := 4975;
    c_comp_type_supobj_deinstall    constant wwv_pe_component_types.id%type := 4980;
    --
    -- Page related component types
    --
    c_comp_type_page                constant wwv_pe_component_types.id%type := 5000;
    -- Page rendering
    c_comp_type_region              constant wwv_pe_component_types.id%type := 5110;
    c_comp_type_page_item           constant wwv_pe_component_types.id%type := 5120;
    c_comp_type_button              constant wwv_pe_component_types.id%type := 5130;
    c_comp_type_da_event            constant wwv_pe_component_types.id%type := 5140;
    c_comp_type_da_action           constant wwv_pe_component_types.id%type := 5150;
    -- Page processing
    c_comp_type_validation          constant wwv_pe_component_types.id%type := 5510;
    c_comp_type_page_computation    constant wwv_pe_component_types.id%type := 5520;
    c_comp_type_page_process        constant wwv_pe_component_types.id%type := 5530;
    c_comp_type_branch              constant wwv_pe_component_types.id%type := 5540;
    --
    -- Region type specific component types ( use a new 7xxx hundred for each region type )
    c_comp_type_region_plugin_attr  constant wwv_pe_component_types.id%type := 7000;
    --
    c_comp_type_ir_attributes       constant wwv_pe_component_types.id%type := 7010;
    c_comp_type_ir_saved_report     constant wwv_pe_component_types.id%type := 7020;
    c_comp_type_ir_column_group     constant wwv_pe_component_types.id%type := 7030;
    c_comp_type_ir_column           constant wwv_pe_component_types.id%type := 7040;
    c_comp_type_ir_print            constant wwv_pe_component_types.id%type := 7050;
    --
    c_comp_type_page_proc_ws_p_i    constant wwv_pe_component_types.id%type := 7110;  -- web service input parameters
    c_comp_type_page_proc_ws_p_o    constant wwv_pe_component_types.id%type := 7120;  -- web service output parameters
    c_comp_type_page_proc_ws_p_a    constant wwv_pe_component_types.id%type := 7130;  -- web service basic auth parameters
    c_comp_type_page_proc_ws_p      constant wwv_pe_component_types.id%type := 7150;  -- web source process parameter
    c_comp_type_region_ws_param     constant wwv_pe_component_types.id%type := 7160;  -- web source region parameter
    --
    c_comp_type_classic_report      constant wwv_pe_component_types.id%type := 7310;
    c_comp_type_classic_rpt_column  constant wwv_pe_component_types.id%type := 7320;
    c_comp_type_classic_rpt_print   constant wwv_pe_component_types.id%type := 7330;
    --
    c_comp_type_tab_form            constant wwv_pe_component_types.id%type := 7410;
    c_comp_type_tab_form_column     constant wwv_pe_component_types.id%type := 7420;
    c_comp_type_tab_form_print      constant wwv_pe_component_types.id%type := 7430;
    --
    c_comp_type_classic_calendar    constant wwv_pe_component_types.id%type := 7610;
    --
    c_comp_type_region_column       constant wwv_pe_component_types.id%type := 7710;
    -- JET Charts
    c_comp_type_jet_chart           constant wwv_pe_component_types.id%type := 7810;
    c_comp_type_jet_chart_series    constant wwv_pe_component_types.id%type := 7820;
    c_comp_type_jet_chart_axes      constant wwv_pe_component_types.id%type := 7830;
    c_comp_type_jet_chart_ws_param  constant wwv_pe_component_types.id%type := 7840;  -- web source jet chart series parameter
    -- interactive grid
    c_comp_type_ig_attributes       constant wwv_pe_component_types.id%type := 7910;
    c_comp_type_ig_saved_report     constant wwv_pe_component_types.id%type := 7920;
    c_comp_type_ig_column_group     constant wwv_pe_component_types.id%type := 7930;
    c_comp_type_ig_column           constant wwv_pe_component_types.id%type := 7940;
    c_comp_type_ig_print            constant wwv_pe_component_types.id%type := 7950;
    -- faceted search
    c_comp_type_facet               constant wwv_pe_component_types.id%type := 8010;
    c_comp_type_facet_group         constant wwv_pe_component_types.id%type := 8020;
    c_comp_type_facet_group_item    constant wwv_pe_component_types.id%type := 8030;
    -- smart filters
    c_comp_type_sfilter             constant wwv_pe_component_types.id%type := 8050;
    c_comp_type_sfilter_group       constant wwv_pe_component_types.id%type := 8051;
    c_comp_type_sfilter_group_item  constant wwv_pe_component_types.id%type := 8052;
    -- card
    c_comp_type_card                constant wwv_pe_component_types.id%type := 8110;
    c_comp_type_card_action         constant wwv_pe_component_types.id%type := 8120;
    --
    c_comp_type_map_region          constant wwv_pe_component_types.id%type := 8210;
    c_comp_type_map_layer           constant wwv_pe_component_types.id%type := 8220;
    c_comp_type_map_layer_ws_param  constant wwv_pe_component_types.id%type := 8230;  -- web source map layer parameter
    --
    c_comp_type_pg_pr_task_param    constant wwv_pe_component_types.id%type := 8400;  -- task page process parameter

    ------------------------------------------------------------------------
    -- Unit constants
    ------------------------------------------------------------------------
    c_unit_pixels         constant varchar2( 20 ) := 'pixels';
    c_unit_chars          constant varchar2( 20 ) := 'characters';
    c_unit_lines          constant varchar2( 20 ) := 'lines';
    c_unit_degrees        constant varchar2( 20 ) := 'degrees';
    c_unit_seconds        constant varchar2( 20 ) := 'seconds';
    c_unit_days           constant varchar2( 20 ) := 'days';
    c_unit_kb             constant varchar2( 20 ) := 'KB';

    ------------------------------------------------------------------------
    -- JET Chart constants
    ------------------------------------------------------------------------
    c_jet_area                constant varchar2( 1000 ) := 'area';
    c_jet_bar                 constant varchar2( 1000 ) := 'bar';
    c_jet_box_plot            constant varchar2( 1000 ) := 'boxPlot';
    c_jet_bubble              constant varchar2( 1000 ) := 'bubble';
    c_jet_combo               constant varchar2( 1000 ) := 'combo';
    c_jet_dial                constant varchar2( 1000 ) := 'dial';
    c_jet_donut               constant varchar2( 1000 ) := 'donut';
    c_jet_funnel              constant varchar2( 1000 ) := 'funnel';
    c_jet_gantt               constant varchar2( 1000 ) := 'gantt';

    c_jet_line                constant varchar2( 1000 ) := 'line';
    c_jet_line_with_area      constant varchar2( 1000 ) := 'lineWithArea';
    c_jet_pie                 constant varchar2( 1000 ) := 'pie';
    c_jet_polar               constant varchar2( 1000 ) := 'polar';
    c_jet_pyramid             constant varchar2( 1000 ) := 'pyramid';
    c_jet_radar               constant varchar2( 1000 ) := 'radar';
    c_jet_range               constant varchar2( 1000 ) := 'range';
    c_jet_scatter             constant varchar2( 1000 ) := 'scatter';
    c_jet_stock               constant varchar2( 1000 ) := 'stock';
    c_jet_pie_donut           constant varchar2 (1000 ) := c_jet_pie||','|| c_jet_donut;
    c_jet_single              constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar||','|| c_jet_line||','|| c_jet_line_with_area||','|| c_jet_bubble||','|| c_jet_funnel||','|| c_jet_pyramid||','|| c_jet_pie_donut||','|| c_jet_scatter||','|| c_jet_stock||','|| c_jet_dial;
    --
    c_jet_stack               constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar || ',' || c_jet_combo || ',' || c_jet_line || ',' || c_jet_line_with_area;
    --c_jet_range               constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar;

    c_jet_spark               constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar || ',' || c_jet_line || ',' || c_jet_line_with_area;
    c_jet_time_axis_type      constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar|| ',' || c_jet_combo || ',' || c_jet_line || ',' || c_jet_line_with_area || ',' || c_jet_stock  || ',' ||c_jet_box_plot;
    c_jet_overview            constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar|| ',' || c_jet_combo || ',' || c_jet_line || ',' || c_jet_stock || ',' || c_jet_box_plot || ',' || c_jet_line_with_area;
    c_jet_value_column        constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_combo ||','|| c_jet_dial ||','|| c_jet_funnel ||','|| c_jet_pyramid ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_pie_donut || ',' ||c_jet_box_plot;
    c_jet_series_column       constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_bubble ||','|| c_jet_combo ||','|| c_jet_funnel ||','|| c_jet_pyramid ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter;
    --c_jet_group_column        constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_bubble ||','|| c_jet_combo ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter ||','|| c_jet_stock;
    c_jet_group_column        constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_bubble ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter ||','|| c_jet_stock || ',' ||c_jet_box_plot;
    c_jet_common_series       constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar || ',' || c_jet_combo || ',' ||  c_jet_line || ',' || c_jet_line_with_area;
    c_jet_line_charts         constant varchar2( 1000 ) := c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_combo ||','||c_jet_area || ',' || c_jet_polar || ',' || c_jet_radar || ',' || c_jet_scatter || ',' ||c_jet_box_plot;
    c_jet_line_style_charts   constant varchar2( 1000 ) := c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_combo ||','|| c_jet_bubble ||','|| c_jet_scatter;
    c_jet_marker_charts       constant varchar2( 1000 ) := c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter ||','|| c_jet_area ||','|| c_jet_bubble || ','|| c_jet_combo || ',' ||c_jet_box_plot;
    c_jet_no_axes             constant varchar2( 1000 ) := c_jet_dial ||','|| c_jet_funnel ||','|| c_jet_pyramid ||','|| c_jet_pie_donut;
    c_jet_no_sorting          constant varchar2( 1000 ) := c_jet_stock ||','|| c_jet_gantt ||','|| c_jet_dial  ||','|| c_jet_funnel ||','|| c_jet_pyramid  ||','|| c_jet_scatter ||','|| c_jet_pie||','|| c_jet_donut;
    --
    c_data_profile_lookup_example constant varchar2( 32767 ) := q'~
<pre>
ENAME  EMPNO  SAL    DNAME
------ ------ ------ ------------
  7839 KING     5000 ACCOUNTING
  7788 SCOTT    3000 RESEARCH
     : :           : :
</pre>
<ul>
<li>Configure the <strong>DNAME_LOADED</strong> column, as a <strong>Data</strong> column with the selector of <strong>DNAME</strong>, or <strong>4</strong> (based on the selector type).</li>
<li>Use <strong>DEPTNO</strong> as the data profile column name and <strong>Lookup</strong> as the column type.</li>
<li>Use <strong>DEPT</strong> as the <strong>Table Name</strong></li>
<li>Use <strong>DEPTNO</strong> as the <strong>Return Column</strong></li>
<li>Use <strong>DNAME</strong> is the <strong>Table Column 1</strong></li>
<li><strong>DNAME_LOADED</strong> is the <strong>Data Column 1</strong>. Note that this references the data profile column which contains the actually uploaded data.</li>
</ul>
<p>
As a result, &PRODUCT_NAME. will look up <strong>DEPTNO</strong> values from the <strong>DEPT</strong> table, using the uploaded <strong>DNAME</strong> values. These will be uploaded as the <strong>DEPTNO</strong> column of the target table.
</p>~';

    ------------------------------------------------------------------------
    -- Global Variables
    ------------------------------------------------------------------------
    g_display_group_id            number  := 1;
    g_component_property_id       number  := 1;
    g_property_lov_value_id       number  := 1;
    g_component_property_sequence number;
    g_property_lov_value_sequence number;
    --
    g_null_parent_display_seq     number  := 0;
    g_parent_component_types      t_parent_component_types;
    --
    l_message_name                varchar2( 255 );
    g_used_system_messages        t_used_system_messages;

    g_cached_component_type_id    number;
    g_cached_is_page_comp_type    boolean;

    ------------------------------------------------------------------------
    -- Variables used in the generated script
    ------------------------------------------------------------------------
    l_property_id       number;
    l_component_type_id number;

    ------------------------------------------------------------------------
    -- Verifies if the entered configuration matches with our database constraints
    ------------------------------------------------------------------------
    procedure check_configuration
    is
        l_data_length number;
        l_is_nullable varchar2( 1 );

        procedure emit_message (
            p_severity      in varchar2,
            p_type_name     in varchar2,
            p_property_name in varchar2,
            p_db_column     in varchar2,
            p_error         in varchar2 )
        is
        begin
            sys.dbms_output.put_line( upper( p_severity ) || ': Component Type "' || p_type_name || '" - Property: "' || p_property_name || '" - DB Column "' || p_db_column || '" - ' || p_error );
        end emit_message;
    begin
        for l_type in ( select id,
                               name,
                               table_name
                          from wwv_pe_component_types
                         order by id )
        loop
            for l_comp_prop in ( select p.name as property_name,
                                        nvl( cp.db_column, p.db_column ) as db_column,
                                        p.max_length,
                                        cp.is_required,
                                        coalesce( to_char( cp.depending_on_comp_prop_id ), to_char( cp.depending_on_comp_prop_id2 ), cp.plugin_std_attributes, case when p.supported_ui_types = 'DESKTOP:JQM_SMARTPHONE' then null else 'X' end, p.deprecated_values) as conditional,
                                        depending_on_condition_type,
                                        depending_on_expression
                                   from wwv_pe_component_props cp,
                                        wwv_pe_properties p
                                  where cp.component_type_id = l_type.id
                                    and p. id = cp.property_id
                                  order by 1 )
            loop
                begin
                    select case
                             when data_type = 'CLOB' then 32767
                             else data_length
                           end as data_length,
                           nullable
                      into l_data_length,
                           l_is_nullable
                      from user_tab_columns
                     where table_name  = l_type.table_name
                       and column_name = l_comp_prop.db_column;

                    if l_comp_prop.max_length > l_data_length then
                        emit_message (
                            p_severity      => 'error',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Constraint defines ' || l_data_length || ' chars, config uses ' || l_comp_prop.max_length );
                    elsif l_comp_prop.max_length <> l_data_length then
                        emit_message (
                            p_severity      => 'warning',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Constraint defines ' || l_data_length || ' chars, config uses ' || l_comp_prop.max_length );
                    end if;

                    if l_comp_prop.is_required = 'N' and l_is_nullable = 'N' then
                        emit_message (
                            p_severity      => 'error',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Has not null constraint, config defines it nullable' );
                    end if;

                    if l_is_nullable = 'N' and l_comp_prop.conditional is not null and l_comp_prop.property_name not in ( 'FIRE_ON_PAGE_LOAD' )
                    then
                        emit_message (
                            p_severity      => 'error',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Has not null constraint, property is conditional.' );
                    end if;

                    if    l_comp_prop.is_required = 'Y'
                      and l_is_nullable = 'Y'
                      and l_comp_prop.conditional   is null
                      and l_comp_prop.property_name not like 'GRID_NEW%'
                    then
                        emit_message (
                            p_severity      => 'warning',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Is nullable, config always requires value - Add DB NN constraint.' );
                    end if;

                    if    l_comp_prop.depending_on_condition_type in ( 'EQUALS', 'NOT_EQUALS' )
                      and l_comp_prop.depending_on_expression like '%,%'
                    then
                        emit_message (
                            p_severity      => 'error',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Depending-on expression contains comma, but ' || l_comp_prop.depending_on_condition_type || ' is used!' );
                    end if;

                exception when no_data_found then
                        emit_message (
                            p_severity      => 'error',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Column not found in ' || l_type.table_name );
                end;
            end loop;
        end loop;

    end check_configuration;

    ------------------------------------------------------------------------
    -- Get the Display Group ID, by name
    ------------------------------------------------------------------------
    function get_display_group_id (
        p_name in wwv_pe_display_groups.name%type )
        return wwv_pe_display_groups.id%type
    is
        l_id wwv_pe_display_groups.id%type;
    begin
        select id
          into l_id
          from wwv_pe_display_groups
         where name = p_name;

        return l_id;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Display Group "' || p_name || '" not found!' );
        -- too_many_rows will never be raised, as name is uniquely constrained
    end get_display_group_id;

    ------------------------------------------------------------------------
    -- Get the Property ID, by name
    ------------------------------------------------------------------------
    function get_property_id (
        p_name in wwv_pe_properties.name%type )
        return wwv_pe_properties.id%type
    is
        l_id wwv_pe_properties.id%type;
    begin
        select id
          into l_id
          from wwv_pe_properties
         where name = p_name;

        return l_id;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Property "' || p_name || '" not found!' );
        -- too_many_rows will never be raised, as name is uniquely constrained
    end get_property_id;

    ------------------------------------------------------------------------
    -- Get the Property Name, by ID
    ------------------------------------------------------------------------
    function get_property_name (
        p_id    in wwv_pe_properties.id%type )
        return wwv_pe_properties.name%type
    is
        l_name wwv_pe_properties.name%type;
    begin
        select name
          into l_name
          from wwv_pe_properties
         where id = p_id;

        return l_name;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Property "' || p_id || '" not found!' );
        -- too_many_rows will never be raised, as ID is unique
    end get_property_name;

    ------------------------------------------------------------------------
    -- Get the Component Property ID, by Component Type ID and Property Name
    ------------------------------------------------------------------------
    function get_component_prop_id (
        p_component_type_id in wwv_pe_component_types.id%type,
        p_property_name     in wwv_pe_properties.name%type )
        return wwv_pe_component_props.id%type
    is
        l_id wwv_pe_component_props.id%type;
    begin
        select cp.id
          into l_id
          from wwv_pe_component_props cp,
               wwv_pe_properties p
         where p.name               = p_property_name
           and cp.component_type_id = p_component_type_id
           and cp.property_id       = p.id;

        return l_id;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Property "' || p_property_name || '" for component type "' || p_component_type_id || '" not found!' );
        -- too_many_rows will never be raised, unique constraint on property_id
        -- and component_type_id
    end get_component_prop_id;

    ------------------------------------------------------------------------
    -- Get the Component Property ID, by Component Type ID and Property Name
    ------------------------------------------------------------------------
    function get_component_type_name (
        p_component_type_id in wwv_pe_component_types.id%type )
        return wwv_pe_component_types.name%type
    is
        l_name wwv_pe_component_types.name%type;
    begin
        select name
          into l_name
          from wwv_pe_component_types
         where id = p_component_type_id;

        return l_name;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Component Type "' || p_component_type_id || '" not found!' );
        -- too_many_rows will never be raised as ID is unique
    end get_component_type_name;

    ------------------------------------------------------------------------
    -- System Message
    ------------------------------------------------------------------------
    procedure store_msg (
        p_name         in out varchar2,
        p_message_text in     varchar2 )
    is
    begin
        p_name := upper( 'PE.' || p_name );

        if p_message_text is not null and c_load_system_messages then
            ------------------------------------------------------------------------
            -- Try an update first
            ------------------------------------------------------------------------
            update wwv_flow_messages$
               set message_text = p_message_text
             where flow_id           = 4411
               and security_group_id = 10
               and name              = p_name
               and message_language  = 'en';
            ------------------------------------------------------------------------
            -- Do an insert if nothing has been updated
            ------------------------------------------------------------------------
            if sql%rowcount = 0 then
                insert into wwv_flow_messages$ (
                    flow_id,
                    security_group_id,
                    name,
                    message_text,
                    message_language )
                values (
                    4411,
                    10,
                    p_name,
                    p_message_text,
                    'en' );
            end if;
            g_used_system_messages( p_name ) := true;
        end if;
    end store_msg;
    --
    procedure remove_unused_messages
    is
    begin
        if c_load_system_messages then
            for l_msg in ( select rowid,
                                  name
                             from wwv_flow_messages$
                            where flow_id           = 4411
                              and security_group_id = 10
                              and name              like 'PE.%'
                              and message_language  = 'en' )
            loop
                if not g_used_system_messages.exists ( l_msg.name ) then
                    delete wwv_flow_messages$
                     where rowid = l_msg.rowid;
                end if;
            end loop;
        end if;
    end remove_unused_messages;

    ------------------------------------------------------------------------
    -- Component Types
    ------------------------------------------------------------------------
    procedure store_component_type (
        p_id                         in number,
        p_name                       in varchar2,
        p_singular_title             in varchar2,
        p_plural_title               in varchar2,
        p_table_name                 in varchar2,
        p_view_name                  in varchar2 default null,
        p_pk_column                  in varchar2 default 'ID',
        p_parent_component_type_id   in number   default null,
        p_parent_fk_column           in varchar2 default null,
        p_page_id_column             in varchar2 default null,
        p_display_column             in varchar2,
        p_where_clause               in varchar2 default null,
        p_validation_plsql_ret_error in varchar2 default null,
        p_plugin_type                in varchar2 default null,
        p_plugin_column              in varchar2 default null,
        p_create_url                 in varchar2 default null,
        p_edit_url                   in varchar2 default null,
        p_copy_url                   in varchar2 default null,
        p_file_path                  in varchar2 default null,
        p_is_page_designer_shared    in boolean  default false,
        p_is_query_only              in boolean  default false )
    is
        l_is_page_designer_shared varchar2( 1 ) := case when p_is_page_designer_shared then 'Y' else 'N' end;
        l_is_query_only           varchar2( 1 ) := case when p_is_query_only           then 'Y' else 'N' end;
        l_msg_plural_title        wwv_flow_messages$.name%type;
        l_msg_singular_title      wwv_flow_messages$.name%type;
        l_display_sequence        number;
    begin
        ------------------------------------------------------------------------
        -- Store system message
        ------------------------------------------------------------------------
        l_msg_plural_title := 'COMPONEN.TYPE.' || p_name || '.PLURAL';
        store_msg (
            p_name         => l_msg_plural_title,
            p_message_text => p_plural_title );

        l_msg_singular_title := 'COMPONEN.TYPE.' || p_name || '.SINGULAR';
        store_msg (
            p_name         => l_msg_singular_title,
            p_message_text => p_singular_title );

        ------------------------------------------------------------------------
        -- Calculate Display Sequence
        ------------------------------------------------------------------------
        if p_parent_component_type_id is not null then
            if not g_parent_component_types.exists( p_parent_component_type_id ) then
                g_parent_component_types( p_parent_component_type_id ) := 0;
            end if;
            --
            g_parent_component_types( p_parent_component_type_id ) := g_parent_component_types( p_parent_component_type_id ) + 1;
            l_display_sequence := g_parent_component_types( p_parent_component_type_id );
        else
            l_display_sequence := g_null_parent_display_seq + 1;
        end if;

        insert into wwv_pe_component_types (
            id,
            name,
            table_name,
            view_name,
            plural_title,
            singular_title,
            display_sequence,
            pk_column,
            parent_component_type_id,
            parent_fk_column,
            page_id_column,
            display_column,
            where_clause,
            validation_plsql_ret_error,
            plugin_type,
            plugin_column,
            create_url,
            edit_url,
            copy_url,
            file_path,
            is_page_designer_shared,
            is_query_only )
        values (
            p_id,
            p_name,
            p_table_name,
            p_view_name,
            l_msg_plural_title, -- Store msg reference
            l_msg_singular_title, -- Store msg reference
            l_display_sequence,
            p_pk_column,
            p_parent_component_type_id,
            p_parent_fk_column,
            p_page_id_column,
            p_display_column,
            p_where_clause,
            p_validation_plsql_ret_error,
            p_plugin_type,
            p_plugin_column,
            p_create_url,
            p_edit_url,
            p_copy_url,
            p_file_path,
            l_is_page_designer_shared,
            l_is_query_only )
        ;

    end store_component_type;

    ------------------------------------------------------------------------
    -- Display Groups
    ------------------------------------------------------------------------
    procedure store_display_group (
        p_name  in varchar2,
        p_title in varchar2 )
    is
        l_msg_name_title wwv_flow_messages$.name%type;
    begin
        ------------------------------------------------------------------------
        -- Store system message
        ------------------------------------------------------------------------
        l_msg_name_title := 'DISPLAY_GROUP.' || p_name;
        store_msg (
            p_name         => l_msg_name_title,
            p_message_text => p_title );

        insert into wwv_pe_display_groups (
            id,
            name,
            title )
        values (
            g_display_group_id,
            p_name,
            l_msg_name_title ) -- Store msg reference
        ;

        -- Always increment global display group id
        g_display_group_id := g_display_group_id + 1;
    end store_display_group;

    ------------------------------------------------------------------------
    -- Properties
    ------------------------------------------------------------------------
    function is_new_shared_comp_property (
        p_id in number )
        return boolean
    is
    begin
        return (  p_id between 1250 and 1280
               or p_id between 1283 and 1363
               or p_id between 1370 and 1500
               or p_id between 1606 and 1658
               or p_id between 5700 and 5977
               );
    end is_new_shared_comp_property;
    --
    --
    procedure store_property (
        p_id                           in number,
        p_name                         in varchar2,
        p_prompt                       in varchar2,
        p_property_type                in varchar2,
        p_prop_type_yesno_override_y   in varchar2     default null,
        p_prop_type_yesno_override_n   in varchar2     default null,
        p_multi_value_delimiter        in varchar2     default null,
        p_db_column                    in varchar2,
        p_is_query_only                in boolean      default false,
        p_display_length               in number       default null,
        p_max_length                   in number       default null,
        p_min_value                    in number       default null,
        p_max_value                    in number       default null,
        p_unit                         in varchar2     default null,
        p_validation_plsql_ret_error   in varchar2     default null,
        p_text_case                    in varchar2     default null,
        p_restricted_characters        in varchar2     default null,
        p_sql_min_column_count         in number       default null,
        p_sql_max_column_count         in number       default null,
        p_column_data_types            in varchar2     default null,
        p_examples                     in varchar2     default null,
        p_supported_ui_types           in varchar2     default 'DESKTOP:JQM_SMARTPHONE',
        p_display_group_name           in varchar2,
        p_lov_type                     in varchar2     default null,
        p_lov_component_type_id        in number       default null,
        p_lov_component_type_scope     in varchar2     default null,
        p_is_translatable              in boolean      default false,
        p_is_searchable                in boolean      default true,
        p_deprecated_values            in varchar2     default null,
        p_is_internal                  in boolean      default false,
        p_help_text                    in varchar2     default null,
        p_comment_text                 in varchar2     default null )
    is
        l_display_group_id      wwv_pe_display_groups.id%type;
        l_msg_name_prop_prompt  wwv_flow_messages$.name%type;
        l_msg_name_prop_help    wwv_flow_messages$.name%type;
        l_msg_name_unit         wwv_flow_messages$.name%type;
        l_msg_name_examples     wwv_flow_messages$.name%type;
        l_text_case             wwv_pe_properties.text_case%type             := p_text_case;
        l_display_length        wwv_pe_properties.display_length%type        := p_display_length;
        l_max_length            wwv_pe_properties.max_length%type            := p_max_length;
        l_multi_value_delimiter wwv_pe_properties.multi_value_delimiter%type := p_multi_value_delimiter;
        l_help_text             wwv_flow_messages$.message_text%type         := p_help_text;
        l_examples              wwv_flow_messages$.message_text%type         := p_examples;
        l_is_translatable       varchar2( 1 ) := case when p_is_translatable then 'Y' else 'N' end;
        l_is_searchable         varchar2( 1 ) := case when p_is_searchable   then 'Y' else 'N' end;
        l_is_query_only         varchar2( 1 ) := case when p_is_query_only   then 'Y' else 'N' end;
        l_is_internal           varchar2( 1 ) := case when p_is_internal     then 'Y' else 'N' end;
    begin
        ------------------------------------------------------------------------
        -- Get Display Group ID, by Display Group Name
        ------------------------------------------------------------------------
        l_display_group_id := get_display_group_id( p_display_group_name );

        ------------------------------------------------------------------------
        -- Reset Property LOV Values sequence ready for  Property LOV values
        -- for this Property.
        ------------------------------------------------------------------------
        g_property_lov_value_sequence := 10;

        ------------------------------------------------------------------------
        -- Store system messages
        ------------------------------------------------------------------------
        l_msg_name_prop_prompt := 'PROPERTY.' || p_name || '.PROMPT';
        l_msg_name_prop_help   := 'PROPERTY.' || p_name || '.HELP';
        store_msg (
            p_name         => l_msg_name_prop_prompt,
            p_message_text => p_prompt );
        --
        if p_property_type = 'COLOR' then
            l_help_text := l_help_text ||
q'~
<p>The color is specified using hexadecimal (hex) notation, RGB color codes, or HTML color names. Hex notation is defined as #xxyyzz (a # followed by 3 double digit hex numbers), where xx, yy, and zz represent the two character hex values for red, green and blue respectively.
RGB codes are defined as RGB(xxx,xxx,xxx), where xxx is a number between 0 and 255 each representing the single 8-bit byte value for red, green, and blue respectively.</p>
<p>Search the internet for 'HTML Color Codes' to find the equivalent hex code or RGB code for the color you wish to utilize.</p>~';
        end if;
        --
        -- Don't save the help text yet for the Shared Components properties. The help text
        -- has to be reworked and we don't want to translate them multiple times.
        if is_new_shared_comp_property( p_id ) then
            l_msg_name_prop_help := null;
        else
            store_msg (
                p_name         => l_msg_name_prop_help,
                p_message_text => l_help_text );
        end if;
        --
        if p_unit is not null then
            l_msg_name_unit := 'PROPERTY.' || p_name || '.UNIT';
            store_msg (
                p_name         => l_msg_name_unit,
                p_message_text => p_unit );
        end if;
        --
        if p_property_type = 'COLOR' and l_examples is null then
            l_examples :=
'Examples of major HTML color codes include -
<ul>
  <li><strong><span style="background-color: lightgray; color: black;">&nbsp;&nbsp;Black&nbsp;&nbsp;</span></strong> <code>#000000 RGB(0,0,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: red;">&nbsp;&nbsp;Red&nbsp;&nbsp;</span></strong> <code>#FF0000 RGB(255,0,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: yellow;">&nbsp;&nbsp;Yellow&nbsp;&nbsp;</span></strong> <code>#FFFF00 RGB(255,255,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: lime;">&nbsp;&nbsp;Lime&nbsp;&nbsp;</span></strong> <code>#00FF00 RGB(0,255,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: green;">&nbsp;&nbsp;Green&nbsp;&nbsp;</span></strong> <code>#008000 RGB(0,128,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: aqua;">&nbsp;&nbsp;Aqua&nbsp;&nbsp;</span></strong> <code>#00FFFF RGB(0,255,255)</code></li>
  <li><strong><span style="background-color: lightgray; color: blue;">&nbsp;&nbsp;Blue&nbsp;&nbsp;</span></strong> <code>#0000FF RGB(0,255,255)</code></li>
  <li><strong><span style="background-color: lightgray; color: navy;">&nbsp;&nbsp;Navy&nbsp;&nbsp;</span></strong> <code>#000080 RGB(0,0,128)</code></li>
  <li><strong><span style="background-color: lightgray; color: white;">&nbsp;&nbsp;White&nbsp;&nbsp;</span></strong> <code>#FFFFFF RGB(255,255,255)</code></li>
</ul>';
        end if;
        --
        if l_examples is not null then
            -- Don't save the examples yet for the Shared Components properties. The examples
            -- have to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( l_property_id ) then
                l_msg_name_examples := 'PROPERTY.' || p_name || '.EXAMPLES';
                store_msg (
                    p_name         => l_msg_name_examples,
                    p_message_text => l_examples );
            end if;
        end if;
        -- Search doesn't make sense for a few property types
        if   ( p_lov_type is not null and p_lov_type not like '%FORMAT_MASK%' and p_lov_type <> 'DISTINCT' )
          or p_property_type in ( 'YES NO', 'SUBSCRIPTION', 'HIDDEN' )
        then
            l_is_searchable := 'N';
        end if;

        -- Numbers can have up to 22 digits
        if p_property_type = 'NUMBER' and l_max_length is null then
            l_max_length := 22;
        elsif p_property_type = 'COLOR' and ( l_display_length is null or l_max_length is null ) then
            l_display_length := nvl( l_display_length, 7 );
            l_max_length     := nvl( l_max_length, 255 );
        end if;

        -- Items are always in uppercase
        if p_property_type in ( 'ITEM', 'FACET', 'SFILTER' ) then
            l_text_case := 'UPPER';
        end if;
        
        -- Checkboxes should always use a multi-value delimiter
        if p_property_type = 'CHECKBOXES' and l_multi_value_delimiter is null then
            l_multi_value_delimiter := ':';
        end if;

        insert into wwv_pe_properties (
            id,
            name,
            prompt,
            property_type,
            prop_type_yesno_override_y,
            prop_type_yesno_override_n,
            multi_value_delimiter,
            db_column,
            is_query_only,
            display_length,
            max_length,
            min_value,
            max_value,
            unit,
            validation_plsql_ret_error,
            text_case,
            restricted_characters,
            sql_min_column_count,
            sql_max_column_count,
            column_data_types,
            examples,
            supported_ui_types,
            display_group_id,
            lov_type,
            lov_component_type_id,
            lov_component_type_scope,
            is_translatable,
            is_searchable,
            deprecated_values,
            is_internal,
            help_text,
            comment_text )
        values (
            p_id,
            p_name,
            l_msg_name_prop_prompt,
            p_property_type,
            p_prop_type_yesno_override_y,
            p_prop_type_yesno_override_n,
            l_multi_value_delimiter,
            p_db_column,
            l_is_query_only,
            l_display_length,
            l_max_length,
            p_min_value,
            p_max_value,
            l_msg_name_unit,
            p_validation_plsql_ret_error,
            l_text_case,
            p_restricted_characters,
            p_sql_min_column_count,
            p_sql_max_column_count,
            p_column_data_types,
            l_msg_name_examples,
            p_supported_ui_types,
            l_display_group_id,
            p_lov_type,
            p_lov_component_type_id,
            p_lov_component_type_scope,
            l_is_translatable,
            l_is_searchable,
            p_deprecated_values,
            l_is_internal,
            l_msg_name_prop_help,
            p_comment_text )
        ;

    exception when others then
    for l_i in ( select name from wwv_pe_properties where id = p_id ) loop
    raise_application_error(-20111, 'exists: '||l_i.name||' new: '||p_id||' name: '||p_name);
    end loop;
        raise_application_error( -20111, 'Property Name: ' || p_name || ' sqlerrm: ' || sqlerrm );
    end store_property;

    ------------------------------------------------------------------------
    -- Property LOV Values
    ------------------------------------------------------------------------
    procedure store_property_lov_values (
        p_property_id   in number,
        p_display_value in varchar2,
        p_return_value  in varchar2,
        p_message_code  in varchar2 default null,
        p_icon          in varchar2 default null,
        p_is_quick_pick in varchar2 default 'N',
        p_is_internal   in varchar2 default 'N',
        p_is_deprecated in varchar2 default 'N',
        p_help_text     in varchar2 default null,
        p_comment_text  in varchar2 default null )
    is
        l_msg_name_prop_lov_val wwv_flow_messages$.name%type;
        l_msg_name_prop_help    wwv_flow_messages$.name%type;
    begin
        ------------------------------------------------------------------------
        -- Store system messages
        ------------------------------------------------------------------------
        l_msg_name_prop_lov_val := 'PROPERTY.' || get_property_name( p_property_id ) || '.LOV.' || upper( coalesce( p_message_code, p_return_value )) || '.D';
        store_msg (
            p_name         => l_msg_name_prop_lov_val,
            p_message_text => p_display_value );
        --
        if p_help_text is not null then
            --
            -- Don't save the help text yet for the Shared Components properties. The help text
            -- has to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( p_property_id ) then
                l_msg_name_prop_help := 'PROPERTY.' || get_property_name( p_property_id ) || '.LOV.' || upper( coalesce( p_message_code, p_return_value )) || '.HELP';
                store_msg (
                    p_name         => l_msg_name_prop_help,
                    p_message_text => p_help_text );
            end if;
        end if;

        insert into wwv_pe_property_lov_values (
            id,
            property_id,
            display_sequence,
            display_value,
            return_value,
            icon,
            is_quick_pick,
            is_internal,
            is_deprecated,
            help_text,
            comment_text )
        values (
            g_property_lov_value_id,
            p_property_id,
            g_property_lov_value_sequence,
            l_msg_name_prop_lov_val,
            p_return_value,
            p_icon,
            p_is_quick_pick,
            p_is_internal,
            p_is_deprecated,
            l_msg_name_prop_help,
            p_comment_text )
        ;
        --
        g_property_lov_value_id       := g_property_lov_value_id + 1;
        g_property_lov_value_sequence := g_property_lov_value_sequence + 10;

    end store_property_lov_values;

    ------------------------------------------------------------------------
    -- Component Properties
    ------------------------------------------------------------------------
    procedure store_component_prop (
        p_component_type_id             in number,
        p_property_name                 in varchar2,
        p_db_column                     in varchar2     default null,
        p_view_column                   in varchar2     default null,
        p_is_required                   in boolean,
        p_check_uniqueness              in varchar2     default null,
        p_is_common                     in boolean,
        p_is_read_only                  in boolean      default false,
        p_reference_on_delete           in varchar2     default null,
        p_reference_scope               in varchar2     default 'COMPONENT',
        p_supports_substitution         in boolean      default true,
        p_supports_template_directive   in boolean      default false,
        p_default_value                 in varchar2     default null,
        p_null_text                     in varchar2     default null,
        p_null_help_text                in varchar2     default null,
        p_plugin_std_attributes         in varchar2     default null,
        p_depending_on_comp_prop_name   in varchar2     default null,
        p_depending_on_has_to_exist     in boolean      default true,
        p_depending_on_condition_type   in varchar2     default null,
        p_depending_on_expression       in varchar2     default null,
        p_depending_on_comp_prop_name2  in varchar2     default null,
        p_depending_on_has_to_exist2    in boolean      default true,
        p_depending_on_condition_type2  in varchar2     default null,
        p_depending_on_expression2      in varchar2     default null,
        p_parent_comp_type_id           in number       default null,
        p_parent_comp_prop_name         in varchar2     default null,
        p_examples                      in varchar2     default null,
        p_help_text                     in varchar2     default null,
        p_comment_text                  in varchar2     default null )
    is
        l_property_id                 wwv_pe_properties.id%type;
        l_property_type               wwv_pe_properties.property_type%type;
        l_depending_on_com_prop_id    wwv_pe_component_props.depending_on_comp_prop_id%type;
        l_depending_on_has_to_exist   varchar2( 1 );
        l_depending_on_com_prop_id2   wwv_pe_component_props.depending_on_comp_prop_id2%type;
        l_depending_on_has_to_exist2  varchar2( 1 );
        l_parent_comp_prop_id         wwv_pe_component_props.parent_comp_prop_id%type;
        l_msg_name_comp_prop_help     wwv_flow_messages$.name%type;
        l_msg_name_null_text          wwv_flow_messages$.name%type;
        l_msg_name_null_help_text     wwv_flow_messages$.name%type;
        l_msg_name_examples           wwv_flow_messages$.message_text%type;
        l_is_required                 varchar2( 1 ) := case p_is_required
                                                         when true  then 'Y'
                                                         when false then 'N'
                                                         else null
                                                       end;
        l_is_common                   varchar2( 1 ) := case when p_is_common    then 'Y' else 'N' end;
        l_is_read_only                varchar2( 1 ) := case when p_is_read_only then 'Y' else 'N' end;
        l_supports_substitution       varchar2( 1 ) := case when p_supports_substitution       then 'Y' else 'N' end;
        l_supports_template_directive varchar2( 1 ) := case when p_supports_template_directive then 'Y' else 'N' end;
        l_reference_on_delete         wwv_pe_component_props.reference_on_delete%type := p_reference_on_delete;
        l_is_page_component_type      varchar2( 1 );
        
        procedure check_depending_on (
            p_condition_type in varchar2,
            p_expression     in varchar2 )
        is
        begin
            if p_expression like '%,%' and p_condition_type not in ( 'IN_LIST', 'NOT_IN_LIST', 'IN_LIST_LIKE', 'NOT_IN_LIST_LIKE' ) then
                raise_application_error(
                    -20999,
                    'Property "' || p_property_name || '" for component type "' || p_component_type_id ||
                    '" contains list values but does not use a list operator! - ' || p_condition_type );
            end if;

            if p_expression is not null and p_condition_type in ( 'NULL', 'NOT_NULL' ) then
                raise_application_error(
                    -20999,
                    'Property "' || p_property_name || '" for component type "' || p_component_type_id ||
                    '" contains list values but does not use a list operator! - ' || p_condition_type );
            end if;
        end check_depending_on;
    begin
        if g_cached_component_type_id is null or g_cached_component_type_id <> p_component_type_id then
            select id,
                   case
                     when page_id_column is not null or id = c_comp_type_page then 'Y'
                     else
                   'N' end
              into g_cached_component_type_id,
                   l_is_page_component_type
              from wwv_pe_component_types
             where id = p_component_type_id;
            --
            g_cached_is_page_comp_type := ( l_is_page_component_type = 'Y' );
        end if;
        ------------------------------------------------------------------------
        -- Get Property ID and type
        ------------------------------------------------------------------------
        l_property_id := get_property_id ( p_property_name );
        select property_type
          into l_property_type
          from wwv_pe_properties
         where id = l_property_id;
        ------------------------------------------------------------------------
        -- Store system messages (Only if help text is passed)
        ------------------------------------------------------------------------
        if p_null_text is not null then
            l_msg_name_null_text := 'PROPERTY.' || p_property_name || '.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.NULL_TEXT';
            store_msg (
                p_name         => l_msg_name_null_text,
                p_message_text => p_null_text );
        end if;

        if p_null_text is not null and p_null_help_text is not null then
            -- Don't save the help text yet for the Shared Components properties. The help text
            -- has to be reworked and we don't want to translate them multiple times.
            if g_cached_is_page_comp_type then
                l_msg_name_null_help_text := 'PROPERTY.' || p_property_name || '.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.NULL_TEXT.HELP';
                store_msg (
                    p_name         => l_msg_name_null_help_text,
                    p_message_text => p_null_help_text );
            end if;
        end if;

        if p_examples is not null then
            -- Don't save the examples yet for the Shared Components properties. The examples
            -- have to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( l_property_id ) and g_cached_is_page_comp_type then
                l_msg_name_examples := 'PROPERTY.'||p_property_name||'.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.EXAMPLES';
                store_msg (
                    p_name         => l_msg_name_examples,
                    p_message_text => p_examples );
            end if;
        end if;

        if p_help_text is not null then
            -- Don't save the help text yet for the Shared Components properties. The help text
            -- has to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( l_property_id ) and g_cached_is_page_comp_type then
                l_msg_name_comp_prop_help := 'PROPERTY.'||p_property_name||'.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.HELP';
                store_msg (
                    p_name         => l_msg_name_comp_prop_help,
                    p_message_text => p_help_text );
            end if;
        end if;

        if p_depending_on_comp_prop_name is not null then
            l_depending_on_com_prop_id  := get_component_prop_id (
                                               p_component_type_id => p_component_type_id,
                                               p_property_name     => p_depending_on_comp_prop_name );
            l_depending_on_has_to_exist := case when p_depending_on_has_to_exist then 'Y' else 'N' end;
        end if;

        if p_depending_on_comp_prop_name2 is not null then
            l_depending_on_com_prop_id2  := get_component_prop_id (
                                                p_component_type_id => p_component_type_id,
                                                p_property_name     => p_depending_on_comp_prop_name2 );
            l_depending_on_has_to_exist2 := case when p_depending_on_has_to_exist2 then 'Y' else 'N' end;
        end if;

        if p_parent_comp_prop_name is not null then
            l_parent_comp_prop_id := get_component_prop_id (
                                         p_component_type_id => nvl( p_parent_comp_type_id, p_component_type_id ),
                                         p_property_name     => p_parent_comp_prop_name );
        elsif l_property_type = 'REGION COLUMN' then
            l_parent_comp_prop_id := get_component_prop_id (
                                         p_component_type_id => c_comp_type_region,
                                         p_property_name     => 'REGION_SQL' );
        elsif l_property_type = 'COLUMN' then
            raise_application_error( -20999, 'Column property requires a parent property' );
        end if;

        if l_property_type not in ( 'COMBOBOX', 'CSS', 'JAVASCRIPT', 'HTML', 'LINK', 'LINK EXTENDED', 'TEXT', 'TEXT EDITOR', 'TEXTAREA' ) then
            l_supports_substitution       := 'N';
            l_supports_template_directive := 'N';
        end if;

        if l_property_type = 'YES NO' then
            l_is_required := 'Y';
        end if;

        if l_property_type in ( 'ITEM', 'FACET', 'SFILTER', 'ORDER BY ITEM' ) and l_reference_on_delete is null then
            l_reference_on_delete := 'WEAK_NULL';
        end if;
        
        -- Check if we do have a valid depending-on config
        check_depending_on (
            p_condition_type => p_depending_on_condition_type,
            p_expression     => p_depending_on_expression );
            
        check_depending_on (
            p_condition_type => p_depending_on_condition_type2,
            p_expression     => p_depending_on_expression2 );
        
        insert into wwv_pe_component_props (
            id,
            component_type_id,
            property_id,
            db_column,
            view_column,
            display_sequence,
            is_required,
            check_uniqueness,
            is_common,
            is_read_only,
            reference_on_delete,
            reference_scope,
            supports_substitution,
            supports_template_directive,
            default_value,
            null_text,
            null_help_text,
            plugin_std_attributes,
            depending_on_comp_prop_id,
            depending_on_has_to_exist,
            depending_on_condition_type,
            depending_on_expression,
            depending_on_comp_prop_id2,
            depending_on_has_to_exist2,
            depending_on_condition_type2,
            depending_on_expression2,
            parent_comp_prop_id,
            examples,
            help_text,
            comment_text )
        values (
            g_component_property_id,
            p_component_type_id,
            l_property_id,
            p_db_column,
            p_view_column,
            g_component_property_sequence,
            l_is_required,
            p_check_uniqueness,
            l_is_common,
            l_is_read_only,
            l_reference_on_delete,
            p_reference_scope,
            l_supports_substitution,
            l_supports_template_directive,
            p_default_value,
            l_msg_name_null_text,
            l_msg_name_null_help_text,
            p_plugin_std_attributes,
            l_depending_on_com_prop_id,
            l_depending_on_has_to_exist,
            p_depending_on_condition_type,
            p_depending_on_expression,
            l_depending_on_com_prop_id2,
            l_depending_on_has_to_exist2,
            p_depending_on_condition_type2,
            p_depending_on_expression2,
            l_parent_comp_prop_id,
            l_msg_name_examples,
            l_msg_name_comp_prop_help,
            p_comment_text )
        ;
        --
        g_component_property_id       := g_component_property_id + 1;
        g_component_property_sequence := g_component_property_sequence + 10;
        --
    exception when others then
        raise_application_error( -20999, 'Type: ' || p_component_type_id || ', Name: ' || p_property_name || ', SQLERRM: ' || sqlerrm );
    end store_component_prop;
    --
    --
    procedure store_plugin_attr_seq_offset (
        p_component_type_id in number )
    is
    begin
        update wwv_pe_component_types
           set plugin_attributes_seq_offset = g_component_property_sequence
         where id = p_component_type_id;
        --
        g_component_property_sequence := g_component_property_sequence + 2500; -- 25 custom attributes
    end store_plugin_attr_seq_offset;
    --
    --
    procedure store_comp_type_data_profile (
        p_profile_type_id  in number,
        p_column_type_id   in number,
        p_name_prefix      in varchar2,
        p_parent_type_id   in number,
        p_parent_fk_column in varchar2 )
    is
    begin
        store_component_type (
            p_id                       => p_profile_type_id,
            p_name                     => p_name_prefix || '_DATA_PROFILE',
            p_singular_title           => 'Data Profile',
            p_plural_title             => 'Data Profiles',
            p_table_name               => 'WWV_FLOW_DATA_PROFILES_DEV',
            p_parent_component_type_id => p_parent_type_id,
            p_parent_fk_column         => p_parent_fk_column,
            p_display_column           => 'NAME',
            p_create_url               => null,
            p_edit_url                 => 'f?p=4000:1925:%session%::NO:1925:P1925_DATA_PROFILE_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
        --
        store_component_type (
            p_id                       => p_column_type_id,
            p_name                     => p_name_prefix || '_DATA_PROFILE_COLUMN',
            p_singular_title           => 'Column',
            p_plural_title             => 'Columns',
            p_table_name               => 'WWV_FLOW_DATA_PROFILE_COLS_DEV',
            p_parent_component_type_id => p_profile_type_id,
            p_parent_fk_column         => 'DATA_PROFILE_ID',
            p_display_column           => 'NAME',
            p_where_clause             => p_parent_fk_column || ' is not null',
            p_create_url               => null,
            p_edit_url                 => 'f?p=4000:1926:%session%::NO:1926:P1926_DATA_PROFILE_COLUMN_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
        --
    end store_comp_type_data_profile;
    --
    --
    procedure store_order_by_clause_property (
        p_name               in varchar2,
        p_display_group_name in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => 'Order By Clause',
            p_property_type      => 'ORDER BY CLAUSE',
            p_db_column          => 'QUERY_ORDER_BY',
            p_display_length     => 40,
            p_max_length         => 255,
            p_display_group_name => p_display_group_name,
            p_examples           => q'~
    <dl><dt>To show the newest hires first and those which have not been hired yet last. If the date is equal, sort them by department number.</dt>
    <dd><pre>
    HIREDATE DESC NULLS LAST, DEPTNO
    </pre></dd>
    </dl>~',
            p_help_text          => q'~<p>Enter one or more comma separated column names, including directives, to define the order in which records are retrieved.</p>
            <p>Directives include:
              <ul>
                <li>ASC - Return column in ascending order</li>
                <li>DESC - Return column in descending order</li>
                <li>NULLS FIRST - Return column with no value before those with values</li>
                <li>NULLS LAST - Return column with values before those with no value</li>
              </ul>
            </p>~' );
    end store_order_by_clause_property;
    --
    --
    procedure store_ext_order_by_property (
        p_name               in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => 'Order By',
            p_property_type      => 'TEXT EDITOR',
            p_db_column          => 'EXTERNAL_ORDER_BY_EXPR',
            p_display_length     => 40,
            p_max_length         => 255,
            p_display_group_name => 'EXTERNAL_FILTER_ORDER_BY',
            p_examples             => q'~
    <dl><dt>For an ORDS REST Service, order by the "job" attribute ascending, then by "ename", descending.</dt>
    <dd><pre>
    {"job": "asc", "ename": "desc"}
    </pre></dd>
    </dl>~',
            p_help_text            => q'~Provide an order by expression to be passed to the external REST service. The expression will be passed to the remote server as-is. The syntax depends on the type of the REST Data Source. For instance, Oracle REST Data Services (ORDS) expects JSON syntax.~' );
    end store_ext_order_by_property;
    --
    --
    procedure store_lov_type_property (
        p_name in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'LOV_TYPE',
            p_display_group_name => 'LIST_OF_VALUES',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the list of values type.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Shared Component',
            p_return_value  => 'SHARED',
            p_help_text     => q'~The list of values is based on a predefined list of values, defined in Shared Components.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'SQL Query',
            p_return_value  => 'SQL_QUERY',
            p_help_text     => q'~The Dynamic list of values is based on the SQL Query you enter.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Static Values',
            p_return_value  => 'STATIC',
            p_help_text     => q'~The Static list of values is based on the text value you enter.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body returning SQL Query', -- @@@
            p_return_value  => 'FUNCTION_BODY',
            p_help_text     => q'~The Dynamic list of values is based on the SQL Query returned by the entered Function Body you enter.~' );
        --
        if p_name = 'FACET_LOV_TYPE' then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Distinct Values',
                p_return_value  => 'DISTINCT',
                p_help_text     => q'~The list of values is based on an automatically generated query that selects the distinct column values.~' );
        end if;
        --
    end store_lov_type_property;
    --
    --
    procedure store_process_point_property (
        p_property_id in number,
        p_name        in varchar2 )
    is
    begin
        store_property (
            p_id                 => p_property_id,
            p_name               => p_name,
            p_prompt             => 'Point',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'PROCESS_POINT',
            p_display_group_name => 'EXECUTION_OPTIONS',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select at what point in page rendering or processing this process is executed.~' );
        --
        if p_name = 'APP_PROCESS_POINT' then
            store_property_lov_values (
                p_property_id   => p_property_id,
                p_display_value => 'After Authentication',
                p_return_value  => 'AFTER_LOGIN',
                p_help_text     => q'~Executes after successful authentication.~' );
        end if;
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'New Session',
            p_return_value  => 'ON_NEW_INSTANCE',
            p_help_text     => q'~Executes when a new session is generated.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Before Header',
            p_return_value  => 'BEFORE_HEADER',
            p_help_text     => q'~Rendering: Executes before the page renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Header',
            p_return_value  => 'AFTER_HEADER',
            p_help_text     => q'~Rendering: Executes after the header renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Before Regions',
            p_return_value  => 'BEFORE_BOX_BODY',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Rendering: Executes before the regions render.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Regions',
            p_return_value  => 'AFTER_BOX_BODY',
            p_help_text     => q'~Rendering: Executes after the regions render.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Before Footer',
            p_return_value  => 'BEFORE_FOOTER',
            p_help_text     => q'~Rendering: Executes before the footer renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Footer',
            p_return_value  => 'AFTER_FOOTER',
            p_help_text     => q'~Rendering: Executes after the page renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Submit',
            p_return_value  => 'ON_SUBMIT_BEFORE_COMPUTATION',
            p_help_text     => q'~Processing: Executes after the page is submitted, but before computations, validations, and page processes performed.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Processing',
            p_return_value  => 'AFTER_SUBMIT',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Processing: Executes after validations and computations performed, when page processing performed.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Ajax Callback',
            p_return_value  => 'ON_DEMAND',
            p_help_text     => q'~Executes when called by apex.server JavaScipt APIs.~' );
    end store_process_point_property;
    --
    --
    procedure store_default_svg_shapes_lov(
        p_property_id  in number )
    is
        c_shape_names constant wwv_flow_t_varchar2 :=
            wwv_flow_t_varchar2(
                'Default',
                'Camera',
                'Circle',
                'Circle Alternative',
                'Face Frown',
                'Face Meh',
                'Face Smile',
                'Square',
                'Square Alternative',
                'Pin Circle',
                'Pin Square',
                'Heart',
                'Pentagon',
                'Triangle',
                'Check',
                'Home',
                'Slash',
                'Shine',
                'Street View',
                'Flag Pennant',
                'Flag Swallowtail');

        c_shape_svgs constant wwv_flow_t_varchar2 :=
            wwv_flow_t_varchar2(
                -- default
                'M10 2C6.7 2 4 4.7 4 8c0 4.6 5.4 9.7 5.7 9.9.2.2.5.2.7 0 .2-.2 5.6-5.3 5.6-9.9 0-3.3-2.7-6-6-6zm0 7.9c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Camera
                'M15 4h-1.2l-.9-1.2c-.4-.5-1-.8-1.6-.8H8.8c-.7 0-1.3.3-1.6.8L6.2 4H5c-1.1 0-2 .9-2 2v5c0 1.1.9 2 2 2h2.2l2.4 4.7c.1.2.4.3.7.2.1 0 .2-.1.2-.2l2.4-4.7H15c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm-5 7c-1.4 0-2.5-1.1-2.5-2.5S8.6 6 10 6s2.5 1.1 2.5 2.5S11.4 11 10 11z',
                -- Circle
                'M10 2C6.7 2 4 4.7 4 8c0 2.5 1.6 4.7 3.9 5.6l1.6 4.1c.1.3.4.4.7.3l.3-.3 1.6-4.1c3.1-1.2 4.7-4.6 3.5-7.7C14.7 3.6 12.5 2 10 2zm0 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Circle Alternative
                'M10 2C6.7 2 4 4.7 4 8v9.5c0 .2.1.4.3.5h.2c.1 0 .3-.1.4-.2l3.7-4c3.2.8 6.5-1.2 7.3-4.4.8-3.2-1.2-6.5-4.4-7.3-.5 0-1-.1-1.5-.1zm0 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Face Frown
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zM7.2 7c0-.4.3-.8.8-.8.4 0 .8.3.8.8s-.4.8-.8.8-.8-.4-.8-.8zm5 3.8c-.2.1-.4.1-.5-.1-.7-.9-2-1.1-3-.4-.2.1-.3.3-.4.4-.1.2-.4.2-.5.1-.2-.1-.2-.4-.1-.5 1-1.3 2.8-1.5 4-.6l.6.6c.1.1.1.4-.1.5zm-.2-3c-.4 0-.8-.3-.8-.8s.3-.8.8-.8.8.3.8.8-.4.8-.8.8z',
                -- Face Meh
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zM7.2 7c0-.4.3-.8.8-.8.4 0 .8.3.8.8s-.4.8-.8.8-.8-.4-.8-.8zm4.3 3.9h-3c-.2 0-.4-.2-.4-.4s.2-.4.4-.4h3c.2 0 .4.2.4.4s-.2.4-.4.4zm.5-3.1c-.4 0-.8-.3-.8-.8s.3-.8.8-.8.8.3.8.8-.4.8-.8.8z',
                -- Face Smile
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zM7.2 7c0-.4.3-.8.8-.8.4 0 .8.3.8.8s-.4.8-.8.8-.8-.4-.8-.8zm5.1 2.7c-1 1.3-2.8 1.5-4 .6l-.6-.6c-.1-.1-.1-.4.1-.5.2-.1.4-.1.5.1.7.9 2 1.1 3 .4.2-.1.3-.3.4-.4.1-.2.4-.2.5-.1s.2.4.1.5zM12 7.8c-.4 0-.8-.3-.8-.8s.3-.8.8-.8.8.3.8.8-.4.8-.8.8z',
                -- Square
                'M14 2H6c-1.1 0-2 .9-2 2v8c0 1.1.9 2 2 2h2.1l1.5 3.7c.1.3.4.4.7.3l.3-.3 1.5-3.7H14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm-4 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Square Alternative
                'M14 2H6c-1.1 0-2 .9-2 2v13.5c0 .2.1.4.3.5h.2c.1 0 .3-.1.4-.2L8.4 14H14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm-4 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Pin Circle
                'M10 2C7.2 2 5 4.2 5 7c0 2.6 1.9 4.7 4.5 5v5.6c0 .3.2.5.5.5s.5-.2.5-.5V12c2.7-.3 4.7-2.7 4.5-5.4-.3-2.7-2.4-4.6-5-4.6z',
                -- Pin Square
                'M13 2H7c-1.1 0-2 .9-2 2v6c0 1.1.9 2 2 2h2.5v5.5c0 .3.2.5.5.5s.5-.2.5-.5V12H13c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2z',
                -- Heart
                'M16 5c0-.8-.3-1.6-.9-2.1-1.2-1.2-3.1-1.2-4.3 0l-.8.8-.8-.8C8 1.7 6.1 1.7 4.9 2.9c-1.2 1.1-1.2 3 0 4.2l4.6 4.6v5.8c0 .3.2.5.5.5s.5-.2.5-.5v-5.8l4.6-4.6c.6-.5.9-1.3.9-2.1z',
                -- Pentagon
                'M15 5.5l-4.7-3.4c-.2-.1-.4-.1-.6 0L5 5.5c-.2.2-.3.4-.2.6l1.8 5.6c.1.2.3.3.5.3h2.4v5.5c0 .3.2.5.5.5s.5-.2.5-.5V12h2.4c.2 0 .4-.1.5-.3l1.8-5.6c.1-.2 0-.4-.2-.6z',
                -- Triangle
                'M15.9 11.2l-5.5-9c-.1-.2-.5-.3-.7-.1-.1 0-.1.1-.1.1l-5.5 9c-.1.2-.1.5.2.7 0 .1.1.1.2.1h5v5.5c0 .3.2.5.5.5s.5-.2.5-.5V12h5c.3 0 .5-.2.5-.5 0-.1 0-.2-.1-.3z',
                -- Check
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zm2.9 4.9l-3 3c-.2.1-.6.1-.8 0L7.6 8.4c-.2-.2-.2-.5 0-.7s.5-.2.7 0l1.1 1.1L12 6.2c.2-.2.5-.2.7 0s.3.5.2.7z',
                -- Home
                '<path d="M17.8 7.6l-7.5-5.5c-.2-.1-.4-.1-.6 0L2.2 7.5c-.2.2-.3.5-.1.7.2.2.5.3.7.1L4 7.5v6c0 .3.2.5.5.5h3.2l1.9 3.7c.1.2.4.3.7.2.1 0 .2-.1.2-.2l1.9-3.7h3.2c.3 0 .5-.2.5-.5v-6l1.2.9c.2.2.5.1.7-.1.1-.3 0-.6-.2-.7z"/><path d="M8.5 8.5v2c0 .3.2.5.5.5h2c.3 0 .5-.2.5-.5v-2c0-.3-.2-.5-.5-.5H9c-.3 0-.5.2-.5.5z" fill="#fff"/>',
                -- Slash
                'M17.9 2.1c-.2-.2-.5-.2-.7 0l-2.3 2.3C13 1.7 9.2 1.1 6.5 3.1 4.9 4.3 4 6.1 4 8c.1 1.9.8 3.8 1.9 5.4l-3.8 3.8c-.2.2-.2.5 0 .7s.5.2.7 0l15-15c.2-.2.2-.6.1-.8zM9.5 9.8c-1.1-.3-1.7-1.4-1.4-2.4S9.5 5.7 10.5 6c.7.2 1.2.7 1.4 1.4L9.5 9.8zM7.1 15c.8 1 1.6 2 2.6 2.8.2.2.5.2.7 0 .2-.1 5.6-5.2 5.6-9.8 0-.6-.1-1.1-.2-1.6L7.1 15z',
                -- Shine
                'M10 7c-2.2 0-4 1.8-4 4 0 2.3 3.2 6.4 3.6 6.8.2.2.5.2.7.1l.1-.1c.4-.5 3.6-4.5 3.6-6.8 0-2.2-1.8-4-4-4zm0 5c-.6 0-1-.4-1-1s.4-1 1-1 1 .4 1 1-.4 1-1 1zM3.6 4.9c-.2-.1-.5-.1-.7.1-.2.3-.1.6.1.7l1.9 1.4c0 .1.1.1.3.1.3 0 .5-.2.5-.5 0-.2-.1-.3-.2-.4L3.6 4.9zm4 .1c.1.2.3.3.5.3h.2c.3-.1.4-.4.3-.6l-.9-2.4c0-.2-.3-.4-.6-.3-.3.1-.4.4-.3.6L7.6 5zm9.5 0c-.2-.2-.5-.3-.7-.1l-1.9 1.4c-.2.2-.3.5-.1.7.1.1.2.2.4.2.1 0 .2 0 .3-.1L17 5.7c.2-.1.3-.4.1-.7zm-5.3.3h.2c.2 0 .4-.1.5-.3l.8-2.3c.1-.3 0-.5-.3-.6-.3-.1-.5 0-.6.3l-.8 2.3c-.2.2-.1.5.2.6z',
                -- Street View
                '<path d="M14.7 11.6c-.3-.1-.5.1-.6.3s.1.5.3.6c1.6.5 2.6 1.2 2.6 1.9 0 1.2-2.9 2.5-7 2.5s-7-1.3-7-2.5c0-.7 1-1.4 2.6-1.9.3 0 .4-.3.3-.5-.1-.3-.4-.4-.6-.3-2.1.6-3.3 1.6-3.3 2.8 0 2 3.4 3.5 8 3.5s8-1.5 8-3.5c0-1.2-1.2-2.2-3.3-2.9z"/><path d="M7.7 10.9c.2.2.3.5.3.7V14c0 .6.4 1 1 1h2c.6 0 1-.4 1-1v-2.4c0-.3.1-.5.3-.7l.1-.1c.4-.4.6-.9.6-1.4V7.8c0-.7-.3-1.3-.9-1.6-1.2 1.2-3.1 1.2-4.2 0-.6.3-.9.9-.9 1.6v1.6c0 .5.2 1 .6 1.4l.1.1z"/><circle cx="10" cy="4" r="2"/>',
                -- Flag Pennant
                'M16.7 6.5L3.7 2c-.3-.1-.6.1-.7.3v15.2c0 .3.2.5.5.5s.5-.2.5-.5v-5.6l12.7-4.4c.3-.1.4-.4.3-.6-.1-.2-.2-.3-.3-.4z',
                -- Flag Swallowtail
                'M13.2 7l3.7-4.2c.2-.2.2-.5 0-.7-.2-.1-.3-.1-.4-.1h-13c-.3 0-.5.2-.5.5v15c0 .3.2.5.5.5s.5-.2.5-.5V12h12.5c.3 0 .5-.2.5-.5 0-.1 0-.2-.1-.3L13.2 7z' );

    begin
        for i in 1 .. c_shape_names.count loop
            store_property_lov_values (
                p_property_id   => p_property_id,
                p_display_value => c_shape_names(i),
                p_return_value  => c_shape_names(i),
                p_help_text     =>    '<svg xmlns="http://www.w3.org/2000/svg" width="20px" height="20px" paint-order="stroke" viewBox="0 0 20 20">'
                                   || case when substr( c_shape_svgs( i ), 1, 5 ) != '<path' then '<path d="' end
                                   || c_shape_svgs( i )
                                   || case when substr( c_shape_svgs( i ), 1, 5 ) != '<path' then '"/>' end
                                   || '</svg>' );

        end loop;
    end store_default_svg_shapes_lov;
    --
    --
    procedure store_color_spectrum_property (
        p_start_property_id  in number,
        p_name_prefix        in varchar2,
        p_db_column_prefix   in varchar2,
        p_display_group_name in varchar2 )
    is
        l_property_id number := p_start_property_id;

        ------------------------------------------------------------------------
        function generate_color_html(
            p_color_list in wwv_flow_t_varchar2 )
            return varchar2
        is
            l_html  varchar2(32767);
            l_width varchar2(4) := '30px';
        begin
            l_html := '<div aria-hidden="true">';
            for i in 1 .. p_color_list.count loop
                l_html := l_html || '<div style="display:inline-block;background-color:' || p_color_list( i ) || '; width:' || l_width || ';">&nbsp;</div>';
            end loop;
            l_html := l_html || '</div>';
            return l_html;
        end generate_color_html;

    begin
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'TYPE',
            p_prompt                      => 'Color Scheme',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'TYPE',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose the color scheme type.~' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Sequential',
            p_return_value  => 'SEQUENTIAL',
            p_help_text     => 'This color scheme is suitable to visualize sequential values, going from a low to a high value.' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Diverging',
            p_return_value  => 'DIVERGING',
            p_help_text     => 'This color scheme is suitable to visualize diverging values, going from one extreme to another.' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Qualitative',
            p_return_value  => 'QUALITATIVE',
            p_help_text     => 'This color scheme is suitable to visualize distinct values.' );
        --
        l_property_id := l_property_id + 1;
        --
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'SEQ_NAME',
            p_prompt                      => 'Scheme Name',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'NAME',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose one of the predefined sequential color schemes. These are suitable to visualize sequential values, going from a low to a high value. Color schemes from cartocolors (<a rel="noreferrer" href="https://github.com/CartoDB/cartocolor" target="_blank">CARTOcolor on GitHub)</a>.~' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Burgundy',
            p_return_value  => 'Burg',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#ffc6c4','#f4a3a8','#e38191','#cc607d','#ad466c','#8b3058','#672044' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Burgundy to Yellow',
            p_return_value  => 'BurgYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fbe6c5','#f5ba98','#ee8a82','#dc7176','#c8586c','#9c3f5d','#70284a' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Red to Orange',
            p_return_value  => 'RedOr',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f6d2a9','#f5b78e','#f19c7c','#ea8171','#dd686c','#ca5268','#b13f64' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Orange to Yellow',
            p_return_value  => 'OrYel',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#ecda9a','#efc47e','#f3ad6a','#f7945d','#f97b57','#f66356','#ee4d5a' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Peach',
            p_return_value  => 'Peach',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fde0c5','#facba6','#f8b58b','#f59e72','#f2855d','#ef6a4c','#eb4a40' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Pink to Yellow',
            p_return_value  => 'PinkYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fef6b5','#ffdd9a','#ffc285','#ffa679','#fa8a76','#f16d7a','#e15383' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Mint',
            p_return_value  => 'Mint',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#e4f1e1','#b4d9cc','#89c0b6','#63a6a0','#448c8a','#287274','#0d585f' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Blue to Green',
            p_return_value  => 'BluGrn',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#c4e6c3','#96d2a4','#6dbc90','#4da284','#36877a','#266b6e','#1d4f60' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Dark Mint',
            p_return_value  => 'DarkMint',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#d2fbd4','#a5dbc2','#7bbcb0','#559c9e','#3a7c89','#235d72','#123f5a' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Emerald',
            p_return_value  => 'Emrld',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#d3f2a3','#97e196','#6cc08b','#4c9b82','#217a79','#105965','#074050' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Blue to Yellow',
            p_return_value  => 'BluYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f7feae','#b7e6a5','#7ccba2','#46aea0','#089099','#00718b','#045275' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Teal',
            p_return_value  => 'Teal',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#d1eeea','#a8dbd9','#85c4c9','#68abb8','#4f90a6','#3b738f','#2a5674' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Teal to Green',
            p_return_value  => 'TealGrn',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#b0f2bc','#89e8ac','#67dba5','#4cc8a3','#38b2a3','#2c98a0','#257d98' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Purple',
            p_return_value  => 'Purp',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f3e0f7','#e4c7f1','#d1afe8','#b998dd','#9f82ce','#826dba','#63589f' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Purple to Orange',
            p_return_value  => 'PurpOr',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f9ddda','#f2b9c4','#e597b9','#ce78b3','#ad5fad','#834ba0','#573b88' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Sunset',
            p_return_value  => 'Sunset',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f3e79b','#fac484','#f8a07e','#eb7f86','#ce6693','#a059a0','#5c53a5' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Magenta',
            p_return_value  => 'Magenta',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f3cbd3','#eaa9bd','#dd88ac','#ca699d','#b14d8e','#91357d','#6c2167' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Dark Sunset',
            p_return_value  => 'SunsetDark',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fcde9c','#faa476','#f0746e','#e34f6f','#dc3977','#b9257a','#7c1d6f' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Brown to Yellow',
            p_return_value  => 'BrwnYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#ede5cf','#e0c2a2','#d39c83','#c1766f','#a65461','#813753','#541f3f' ) ) );

        --
        l_property_id := l_property_id + 1;
        --
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'DIV_NAME',
            p_prompt                      => 'Scheme Name',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'NAME',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose one of the predefined diverging color schemes. These are suitable to visualize diverging values, going from one extreme to another. Color schemes from cartocolors (<a rel="noreferrer" href="https://github.com/CartoDB/cartocolor" target="_blank">CARTOcolor on GitHub)</a>.~' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => '"Army" to Rose',
            p_return_value  => 'ArmyRose',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#798234','#a3ad62','#d0d3a2','#fdfbe4','#f0c6c3','#df91a3','#d46780' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Fall Colors',
            p_return_value  => 'Fall',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#3d5941','#778868','#b5b991','#f6edbd','#edbb8a','#de8a5a','#ca562c' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Geyser',
            p_return_value  => 'Geyser',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#008080','#70a494','#b4c8a8','#f6edbd','#edbb8a','#de8a5a','#ca562c' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Temps',
            p_return_value  => 'Temps',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#009392','#39b185','#9ccb86','#e9e29c','#eeb479','#e88471','#cf597e' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Teal to Rose',
            p_return_value  => 'TealRose',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#009392','#72aaa1','#b1c7b3','#f1eac8','#e5b9ad','#d98994','#d0587e' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Tropic Colors',
            p_return_value  => 'Tropic',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#009B9E','#42B7B9','#A7D3D4','#F1F1F1','#E4C1D9','#D691C1','#C75DAB' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Earth Colors',
            p_return_value  => 'Earth',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#A16928','#bd925a','#d6bd8d','#edeac2','#b5c8b8','#79a7ac','#2887a1' ) ) );

        --
        l_property_id := l_property_id + 1;
        --
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'QUA_NAME',
            p_prompt                      => 'Scheme Name',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'NAME',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose one of the predefined qualitative color schemes. These are suitable to visualize distinct values. Color schemes from cartocolors (<a rel="noreferrer" href="https://github.com/CartoDB/cartocolor" target="_blank">CARTOcolor on GitHub)</a>.~' );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Antique',
            p_return_value  => 'Antique',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#855C75','#D9AF6B','#AF6458','#736F4C','#526A83','#625377','#7C7C7C' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Bold',
            p_return_value  => 'Bold',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#7F3C8D','#11A579','#3969AC','#F2B701','#E73F74','#80BA5A','#A5AA99' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Pastel',
            p_return_value  => 'Pastel',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#66C5CC','#F6CF71','#F89C74','#DCB0F2','#87C55F','#9EB9F3','#B3B3B3' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Prism',
            p_return_value  => 'Prism',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#5F4690','#1D6996','#38A6A5','#0F8554','#73AF48','#EDAD08','#666666' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Safe',
            p_return_value  => 'Safe',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#88CCEE','#CC6677','#DDCC77','#117733','#332288','#AA4499','#888888' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Vivid',
            p_return_value  => 'Vivid',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#E58606','#5D69B1','#52BCA3','#99C945','#CC61B0','#24796C','#A5AA99' ) ) );
    end store_color_spectrum_property;
    --
    --
    procedure store_code_language_properties (
        p_property_id        in number,
        p_name               in varchar2,
        p_db_column          in varchar2,
        p_display_group_name in varchar2,
        p_include_sql        in boolean,
        p_help_text          in varchar2 )
    is
    begin
        store_property (
            p_id                 => p_property_id,
            p_name               => p_name,
            p_prompt             => 'Language',
            p_property_type      => 'CODE LANGUAGE',
            p_db_column          => p_db_column,
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => p_help_text );
        --
        if p_include_sql then
            store_property_lov_values (
                p_property_id   => p_property_id,
                p_display_value => 'SQL',
                p_return_value  => 'SQL' );
        end if;
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'PL/SQL',
            p_return_value  => 'PLSQL',
            p_is_quick_pick => 'Y' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'JavaScript (MLE)',
            p_return_value  => 'JAVASCRIPT',
            p_is_quick_pick => 'Y' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Java (MLE)',
            p_return_value  => 'JAVA' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Python (MLE)',
            p_return_value  => 'PYTHON' );
    end store_code_language_properties;
    --
    --
    procedure store_expression_properties (
        p_start_property_id  in number,
        p_name_prefix        in varchar2,
        p_return_type        in varchar2,
        p_db_column_lang     in varchar2,
        p_db_column_expr     in varchar2,
        p_display_group_name in varchar2,
        p_help_text          in varchar2,
        p_examples_sql       in varchar2 default null,
        p_examples_plsql     in varchar2 default null,
        p_examples_js        in varchar2 default null,
        p_examples_java      in varchar2 default null,
        p_examples_python    in varchar2 default null )
    is
        c_postfix constant varchar2( 30 ) := '_' || p_return_type;
        --
        l_property_id number := p_start_property_id;
    begin
        store_code_language_properties (
            p_property_id        => l_property_id,
            p_name               => p_name_prefix || 'EXPRESSION_LANG',
            p_db_column          => p_db_column_lang,
            p_display_group_name => p_display_group_name,
            p_include_sql        => true,
            p_help_text          => q'~Select a language that is used to evaluate the entered expression.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'SQL_EXPRESSION' || c_postfix,
            p_prompt             => 'SQL Expression',
            p_property_type      => 'SQL EXPRESSION' || case when p_return_type = 'BOOLEAN' then ' BOOLEAN' end,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'SQL' ),
            p_examples           => coalesce(
                                        p_examples_sql,
                                        case p_return_type
                                        when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is a manager and their department is located in Chicago:</dt>
<dd><pre>
:P2_JOB = 'MANAGER' and
exists ( select 1
           from dept
          where deptno = :P2_DEPTNO
            and loc    = 'CHICAGO' )
</pre></dd>
</dl>~'
                                        when 'VARCHAR2' then
q'~
<dl><dt>To return YES when the employee is a manager and their department is located in Chicago:</dt>
<dd><pre>
case
when :P2_JOB = 'MANAGER' and
     exists ( select 1
                from dept
               where deptno = :P2_DEPTNO
                 and loc    = 'CHICAGO' ) then 'YES'
else 'NO'
end
</pre></dd>
</dl>~'
                                        end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'PLSQL_EXPRESSION' || c_postfix,
            p_prompt             => 'PL/SQL Expression',
            p_property_type      => 'PLSQL EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples           => coalesce(
                                        p_examples_plsql,
                                        case p_return_type
                                        when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( :P2_DEPTNO = 30 or :P2_JOB = 'MANAGER' )
</pre></dd>
</dl>~'
                                        when 'VARCHAR2' then
q'~
<dl><dt>To return YES when the employee is in department 30 or is a manager:</dt>
<dd><pre>
case when ( :P2_DEPTNO = 30 or :P2_JOB = 'MANAGER' ) then 'YES' else 'NO' end
</pre></dd>
</dl>~'
                                        end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'JS_EXPRESSION' || c_postfix,
            p_prompt             => 'JavaScript Expression',
            p_property_type      => 'MLE JAVASCRIPT EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples           => coalesce(
                                        p_examples_js,
                                        case p_return_type
                                        when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( apex.env.P2_DEPTNO === "30" || apex.env.P2_JOB === "MANAGER" )
</pre></dd>
</dl>~'
                                        when 'VARCHAR2' then
q'~
<dl><dt>To return YES when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( apex.env.P2_DEPTNO === "30" || apex.env.P2_JOB === "MANAGER" ? "YES" : "NO" )
</pre></dd>
</dl>~'
                                        end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'JAVA_EXPRESSION' || c_postfix,
            p_prompt             => 'Java Expression',
            p_property_type      => 'MLE JAVA EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'Java' ),
            p_examples           => p_examples_java );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'PYTHON_EXPRESSION' || c_postfix,
            p_prompt             => 'Python Expression',
            p_property_type      => 'MLE PYTHON EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'Python' ),
            p_examples           => p_examples_python );
        --
    end store_expression_properties;
    --
    --
    procedure store_expression_comp_props (
        p_component_type_id in number,
        p_name_prefix       in varchar2,
        p_return_type       in varchar2,
        p_db_column_lang    in varchar2 default null,
        p_db_column_expr    in varchar2 default null,
        p_reference_scope   in varchar2 default 'COMPONENT',
        p_type_property     in varchar2 )
    is
        c_postfix constant varchar2( 30 ) := '_' || p_return_type;
    begin
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'EXPRESSION_LANG',
            p_db_column                   => p_db_column_lang,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'PLSQL',
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_type_property,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'EXPRESSION' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'SQL_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_name_prefix || 'EXPRESSION_LANG',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PLSQL_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_name_prefix || 'EXPRESSION_LANG',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLSQL' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JS_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_name_prefix || 'EXPRESSION_LANG',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'JAVASCRIPT' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JAVA_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_name_prefix || 'EXPRESSION_LANG',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'JAVA' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PYTHON_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_name_prefix || 'EXPRESSION_LANG',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PYTHON' );
        --
    end store_expression_comp_props;
    --
    --
    procedure store_func_body_properties (
        p_start_property_id    in number,
        p_name_prefix          in varchar2,
        p_postfix              in varchar2 default null,
        p_return_type          in varchar2,
        p_prompt               in varchar2 default null,
        p_db_column_lang       in varchar2,
        p_db_column_func_body  in varchar2,
        p_display_group_name   in varchar2,
        p_max_length           in varchar2 default null,
        p_sql_min_column_count in number   default null,
        p_sql_max_column_count in number   default null,
        p_help_text            in varchar2,
        p_examples_plsql       in varchar2 default null,
        p_examples_js          in varchar2 default null,
        p_examples_java        in varchar2 default null,
        p_examples_python      in varchar2 default null )
    is
        c_postfix        constant varchar2( 30 )  := '_' || coalesce( p_postfix, p_return_type );
        c_func_body_lang constant varchar2( 255 ) := p_name_prefix || 'FUNC_BODY_LANG' || case when p_postfix is not null then '_' || p_postfix end;
        c_prompt         constant varchar2( 255 ) := coalesce (
                                                         p_prompt,
                                                         case p_return_type
                                                           when 'SQL' then '%language% Function Body returning SQL Query'
                                                         end,
                                                         '%language% Function Body' );
        c_max_length     constant number          := coalesce (
                                                         p_max_length,
                                                         case when p_return_type = 'SQL' then 32767 else 4000 end );
        --
        l_property_id number := p_start_property_id;
    begin
        store_code_language_properties (
            p_property_id        => l_property_id,
            p_name               => c_func_body_lang,
            p_db_column          => p_db_column_lang,
            p_display_group_name => p_display_group_name,
            p_include_sql        => false,
            p_help_text          => q'~Select a language that is used to execute the entered function body.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PLSQL_FUNC_BODY' || c_postfix,
            p_prompt               => replace( c_prompt, '%language%', 'PL/SQL' ),
            p_property_type        => 'PLSQL FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples             => coalesce(
                                          p_examples_plsql,
                                          case p_return_type
                                          when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30:</dt>
<dd><pre>
declare
    l_deptno number;
begin
    select deptno
      into l_deptno
      from emp
     where empno = :P2_EMPNO;

    if l_deptno = 30 then
        return true;
    else
        return false;
    end if;
end;
</pre></dd>
</dl>~'
                                          when 'VARCHAR2' then
q'~
<dl><dt>To return YES when the employee is in department 30:</dt>
<dd><pre>
declare
    l_deptno number;
begin
    select deptno
      into l_deptno
      from emp
     where empno = :P2_EMPNO;

    if l_deptno = 30 then
        return 'YES';
    else
        return 'NO';
    end if;
end;
</pre></dd>
</dl>~'
                                          end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JS_FUNC_BODY' || c_postfix,
            p_prompt               => replace( c_prompt, '%language%', 'JavaScript' ),
            p_property_type        => 'MLE JAVASCRIPT FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples             => coalesce(
                                          p_examples_js,
                                          case p_return_type
                                          when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30:</dt>
<dd><pre>
let deptno,
    rows;

rows = apex.conn.execute( "select deptno from emp empno = :1", [ apex.env.P2_EMPNO ] ).rows;
for (var row of rows) {
  deptno = row.DEPTNO;
}

return ( deptno === 30 );
</pre></dd>
</dl>~'
                                          when 'VARCHAR2' then
q'~
<dl><dt>To return YES when the employee is in department 30:</dt>
<dd><pre>
let deptno,
    rows;

rows = apex.conn.execute( "select deptno from emp empno = :1", [ apex.env.P2_EMPNO ] ).rows;
for (var row of rows) {
  deptno = row.DEPTNO;
}

return ( deptno === 30 ? "YES", "NO" );
</pre></dd>
</dl>~'
                                          end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JAVA_FUNC_BODY' || c_postfix,
            p_prompt               => replace( c_prompt, '%language%', 'Java' ),
            p_property_type        => 'MLE JAVA FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Java' ),
            p_examples             => p_examples_java );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PYTHON_FUNC_BODY' || c_postfix,
            p_prompt               => replace( c_prompt, '%language%', 'Python' ),
            p_property_type        => 'MLE PYTHON FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Python' ),
            p_examples             => p_examples_python );
        --
    end store_func_body_properties;
    --
    --
    procedure store_func_body_comp_props (
        p_component_type_id     in number,
        p_name_prefix           in varchar2,
        p_postfix               in varchar2 default null,
        p_return_type           in varchar2,
        p_db_column_lang        in varchar2 default null,
        p_db_column_func_body   in varchar2 default null,
        p_reference_scope       in varchar2 default 'COMPONENT',
        p_type_property         in varchar2,
        p_type_operator         in varchar2 default 'EQUALS',
        p_type_value            in varchar2 default 'FUNCTION_BODY',
        p_parent_comp_prop_name in varchar2 default null )
    is
        c_postfix        constant varchar2( 30 )  := '_' || coalesce( p_postfix, p_return_type );
        c_func_body_lang constant varchar2( 255 ) := p_name_prefix || 'FUNC_BODY_LANG' || case when p_postfix is not null then '_' || p_postfix end;
    begin
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => c_func_body_lang,
            p_db_column                   => p_db_column_lang,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'PLSQL',
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_type_property,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => p_type_value );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PLSQL_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PLSQL',
            p_parent_comp_prop_name       => p_parent_comp_prop_name );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JS_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'JAVASCRIPT',
            p_parent_comp_prop_name       => p_parent_comp_prop_name );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JAVA_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'JAVA',
            p_parent_comp_prop_name       => p_parent_comp_prop_name );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PYTHON_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PYTHON',
            p_parent_comp_prop_name       => p_parent_comp_prop_name );
        --
    end store_func_body_comp_props;
    --
    --
    procedure store_snippet_properties (
        p_start_property_id    in number,
        p_name_prefix          in varchar2,
        p_prompt               in varchar2 default null,
        p_db_column_lang       in varchar2,
        p_db_column_snippet    in varchar2,
        p_display_group_name   in varchar2,
        p_max_length           in varchar2 default 32767,
        p_help_text            in varchar2,
        p_examples_plsql       in varchar2 default null,
        p_examples_js          in varchar2 default null,
        p_examples_java        in varchar2 default null,
        p_examples_python      in varchar2 default null )
    is
        c_snippet_lang constant varchar2( 255 ) := p_name_prefix || 'SNIPPET_LANG';
        c_prompt       constant varchar2( 255 ) := coalesce( p_prompt, '%language% Code' );
        --
        l_property_id number := p_start_property_id;
    begin
        store_code_language_properties (
            p_property_id        => l_property_id,
            p_name               => c_snippet_lang,
            p_db_column          => p_db_column_lang,
            p_display_group_name => p_display_group_name,
            p_include_sql        => false,
            p_help_text          => q'~Select a language that is used to execute the entered code.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PLSQL_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'PL/SQL' ),
            p_property_type        => 'PLSQL',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples             => coalesce(
                                          p_examples_plsql,
q'~
<dl><dt>Increases the salary by 3% for those employees which are qualified for a salary raise:</dt>
<dd><pre>
begin
    for l_emp ( select empno
                  from emp
                 where deptno = :P2_DEPTNO )
    loop
        if needs_salary_raise( l_empno.empno ) then
            update emp
               set sal = sal * 1.03
             where empno = l_emp.empno;
        end if;
    end loop;
end;
</pre></dd>
</dl>~'
            ));
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JS_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'JavaScript' ),
            p_property_type        => 'MLE JAVASCRIPT',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples             => coalesce(
                                          p_examples_js,
q'~
<dl><dt>Increases the salary by 3% for those employees which are qualified for a salary raise:</dt>
<dd><pre>
var rows;

function needsSalaryRaise( empno ) {
    // placeholder for some complicated logic to calculate if employee needs a raise
    return true;
}

rows = apex.conn.execute( "select empno from emp where deptno = :deptno", { deptno: apex.env.P2_DEPTNO } ).rows;
for ( var row of rows ) {
    if ( needsSalaryRaise( row.EMPNO )) {
      apex.conn.execute( "update emp set sal = sal * 1.03 where empno = :empno", { empno: row.EMPNO } );
    }
}
</pre></dd>
</dl>~'
            ));
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JAVA_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'Java' ),
            p_property_type        => 'MLE JAVA',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Java' ),
            p_examples             => p_examples_java );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PYTHON_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'Python' ),
            p_property_type        => 'MLE PYTHON',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Python' ),
            p_examples             => p_examples_python );
        --
    end store_snippet_properties;
    --
    --
    procedure store_snippet_comp_props (
        p_component_type_id     in number,
        p_name_prefix           in varchar2,
        p_db_column_lang        in varchar2 default null,
        p_db_column_snippet     in varchar2 default null,
        p_reference_scope       in varchar2 default 'COMPONENT',
        p_type_property         in varchar2,
        p_type_operator         in varchar2 default 'EQUALS',
        p_type_value            in varchar2,
        p_parent_comp_prop_name in varchar2 default null,
        p_help_text             in varchar2 default null,
        p_examples_plsql        in varchar2 default null,
        p_examples_js           in varchar2 default null,
        p_examples_java         in varchar2 default null,
        p_examples_python       in varchar2 default null )
    is
        c_snippet_lang constant varchar2( 255 ) := p_name_prefix || 'SNIPPET_LANG';
    begin
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => c_snippet_lang,
            p_db_column                   => p_db_column_lang,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'PLSQL',
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_type_property,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => p_type_value );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PLSQL_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PLSQL',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples                    => p_examples_plsql );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JS_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'JAVASCRIPT',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples                    => p_examples_js );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JAVA_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PYTHON',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'Java' ),
            p_examples                    => p_examples_java );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PYTHON_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PYTHON',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'Python' ),
            p_examples                    => p_examples_python );
        --
    end store_snippet_comp_props;
    --
    --
    procedure store_cond_type_properties (
        p_start_property_id  in number,
        p_name_prefix        in varchar2,
        p_prompt_type        in varchar2 default 'Type',
        p_help_text_type     in varchar2,
        p_db_column_type     in varchar2,
        p_db_column_expr1    in varchar2,
        p_db_column_expr2    in varchar2,
        p_display_group_name in varchar2 )
    is
        c_name_prefix constant varchar2( 30) := case when p_name_prefix is not null then p_name_prefix || '_' end;
        l_property_id number := p_start_property_id;
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'CONDITION_TYPE',
            p_prompt             => p_prompt_type,
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column_type,
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => p_help_text_type );
        --
        if p_name_prefix = 'READ_ONLY' then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Always',
                p_return_value  => 'ALWAYS',
                p_is_deprecated => 'N',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The component will always be rendered as read only.~' );
        else
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Always',
                p_return_value  => 'ALWAYS',
                p_is_deprecated => 'Y',
                p_help_text     => q'~The component will always be rendered or executed.~' );
        end if;
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Rows returned',
            p_return_value  => 'EXISTS',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~The SQL query returns one or more rows.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'No Rows returned',
            p_return_value  => 'NOT_EXISTS',
            p_help_text     => q'~The SQL Query returns no rows~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Expression',
            p_return_value  => 'EXPRESSION',
            p_help_text     => q'~The expression evaluates to TRUE.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body',
            p_return_value  => 'FUNCTION_BODY',
            p_help_text     => q'~The function body returns TRUE.~' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request = Value',
                p_return_value  => 'REQUEST_EQUALS_CONDITION',
                p_help_text     => q'~The page request is equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request != Value',
                p_return_value  => 'REQUEST_NOT_EQUAL_CONDITION',
                p_help_text     => q'~The page request is not equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request is contained in Value',
                p_return_value  => 'REQUEST_IN_CONDITION',
                p_help_text     => q'~The page request is contained in the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request is NOT contained in Value',
                p_return_value  => 'REQUEST_NOT_IN_CONDITION',
                p_help_text     => q'~The page request is not contained in the text you enter into the <em>Value</em> attribute.~' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item = Value',
                p_return_value  => 'VAL_OF_ITEM_IN_COND_EQ_COND2',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Item</em> is equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item != Value',
                p_return_value  => 'VAL_OF_ITEM_IN_COND_NOT_EQ_COND2',
                p_help_text     => q'~The value of the selected <em>Item</em> is not equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NULL',
                p_return_value  => 'ITEM_IS_NULL',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Item</em> is empty.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT NULL',
                p_return_value  => 'ITEM_IS_NOT_NULL',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Item</em> is not empty.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is zero',
                p_return_value  => 'ITEM_IS_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT zero',
                p_return_value  => 'ITEM_IS_NOT_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is not the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NULL or zero',
                p_return_value  => 'ITEM_IS_NULL_OR_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is empty or the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT NULL and NOT zero',
                p_return_value  => 'ITEM_NOT_NULL_OR_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is not empty and not the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item contains no spaces',
                p_return_value  => 'ITEM_CONTAINS_NO_SPACES',
                p_help_text     => q'~The value of the selected <em>Item</em> has no spaces.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is numeric',
                p_return_value  => 'ITEM_IS_NUMERIC',
                p_help_text     => q'~The value of the selected <em>Item</em> is numeric.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT numeric',
                p_return_value  => 'ITEM_IS_NOT_NUMERIC',
                p_help_text     => q'~The value of the selected <em>Item</em> is not numeric.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is alphanumeric',
                p_return_value  => 'ITEM_IS_ALPHANUMERIC',
                p_help_text     => q'~The value of the selected <em>Item</em> is alphanumeric, containing only letters or numbers and no special characters.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is in colon delimited list',
                p_return_value  => 'VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST',
                p_help_text     => q'~The value of the selected <em>Item</em> is completely contained in the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT in colon delimited list',
                p_return_value  => 'VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST',
                p_help_text     => q'~The value of the selected <em>Item</em> is not completely contained in the text you entered into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is contained in Item',
                p_return_value  => 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text = Value',
                p_return_value  => 'DISPLAY_COND_EQUAL_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text != Value',
                p_return_value  => 'DISP_COND_NOT_EQUAL_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is contained in Value',
                p_return_value  => 'DISPLAY_COND_IN_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is NOT contained in Value',
                p_return_value  => 'DISPLAY_COND_NOT_IN_COND_TEXT',
                p_is_deprecated => 'Y' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User Preference = Value',
                p_return_value  => 'USER_PREF_IN_COND_EQ_COND2',
                p_help_text     => q'~The value of the <em>Preference</em> entered is equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User Preference != Value',
                p_return_value  => 'USER_PREF_IN_COND_NOT_EQ_COND2',
                p_help_text     => q'~The value of the <em>Preference</em> entered is not equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page = Page',
                p_return_value  => 'CURRENT_PAGE_EQUALS_CONDITION',
                p_help_text     => q'~The current page is equal to the value you enter into <em>Page</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page != Page',
                p_return_value  => 'CURRENT_PAGE_NOT_EQUAL_CONDITION',
                p_help_text     => q'~The current page is not equal to the value you enter into <em>Page</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page is in comma delimited list',
                p_return_value  => 'CURRENT_PAGE_IN_CONDITION',
                p_help_text     => q'~The current page is in the comma separated list you enter into <em>Pages</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page is NOT in comma delimited list',
                p_return_value  => 'CURRENT_PAGE_NOT_IN_CONDITION',
                p_help_text     => q'~The current page is not in the comma separated list you enter into <em>Pages</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page = Page Submitted (posted)',
                p_return_value  => 'WHEN_THIS_PAGE_SUBMITTED',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page != Page Submitted (posted)',
                p_return_value  => 'WHEN_THIS_PAGE_NOT_SUBMITTED',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page is in Printer Friendly Mode',
                p_return_value  => 'PAGE_IS_IN_PRINTER_FRIENDLY_MODE',
                p_help_text     => q'~The current page has been toggled to 'Printer Friendly Mode' by the end user.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current page is NOT in Printer Friendly Mode',
                p_return_value  => 'PAGE_IS_NOT_IN_PRINTER_FRIENDLY_MODE',
                p_help_text     => q'~The current page has not been toggled to 'Printer Friendly Mode' by the end user.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Page/Region is Read Only',
                p_return_value  => 'IS_READ_ONLY',
                p_help_text     => q'~Either the page or parent region, if appropriate, 'Read Only' condition evaluates to TRUE.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Page/Region is NOT Read Only',
                p_return_value  => 'IS_NOT_READ_ONLY',
                p_help_text     => q'~The page and parent region, if appropriate, are displayed normally.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User is Authenticated (not public)',
                p_return_value  => 'USER_IS_NOT_PUBLIC_USER',
                p_help_text     => q'~The end user is logged into the application.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User is the Public User (user has not authenticated)',
                p_return_value  => 'USER_IS_PUBLIC_USER',
                p_help_text     => q'~The end user is not logged into the application.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Inline Validation Errors displayed',
                p_return_value  => 'DISPLAYING_INLINE_VALIDATION_ERRORS',
                p_help_text     => q'~An inline validation error is displayed on the page, following a page submission.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Inline Validation Errors NOT displayed',
                p_return_value  => 'NOT_DISPLAYING_INLINE_VALIDATION_ERRORS',
                p_help_text     => q'~An inline validation error is not displayed on the page.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'SQL Reports (OK to show the forward button)',
                p_return_value  => 'MAX_ROWS_LT_ROWS_FETCHED',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'SQL Reports (OK to show the back button)',
                p_return_value  => 'MIN_ROW_GT_THAN_ONE',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language = Value',
                p_return_value  => 'CURRENT_LANG_EQ_COND1',
                p_help_text     => q'~The end user is currently running the application in the language you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language != Value',
                p_return_value  => 'CURRENT_LANG_NOT_EQ_COND1',
                p_help_text     => q'~The end user is not currently running the application in the language you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language is contained in Value',
                p_return_value  => 'CURRENT_LANG_IN_COND1',
                p_help_text     => q'~The end user is currently running the application in one of the languages you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language is NOT contained in Value',
                p_return_value  => 'CURRENT_LANG_NOT_IN_COND1',
                p_help_text     => q'~The end user is not currently running the application in one of the languages you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is Mozilla, Netscape 6.x/7x or higher',
                p_return_value  => 'BROWSER_IS_NSCP',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is Microsoft Internet Explorer 7.0 or higher',
                p_return_value  => 'BROWSER_IS_MSIE',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is XHTML / CSS capable browser',
                p_return_value  => 'BROWSER_IS_MSIE_OR_NSCP',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is Other browsers (or older version)',
                p_return_value  => 'BROWSER_IS_OTHER',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: DAD_NAME = Value',
                p_return_value  => 'DAD_NAME_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: DAD_NAME != Value',
                p_return_value  => 'DAD_NAME_NOT_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: SERVER_NAME = Value',
                p_return_value  => 'SERVER_NAME_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: SERVER_NAME != Value',
                p_return_value  => 'SERVER_NAME_NOT_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: HTTP_HOST = Value',
                p_return_value  => 'HTTP_HOST_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: HTTP_HOST != Value',
                p_return_value  => 'HTTP_HOST_NOT_EQ_CONDITION',
                p_is_deprecated => 'Y' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) in ( 'CONDITION', 'LINK', 'READ_ONLY', 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Never',
                p_return_value  => 'NEVER',
                p_is_quick_pick => case when nvl( p_name_prefix, 'CONDITION' ) in ( 'CONDITION', 'READ_ONLY', 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND' ) then 'Y' else 'N' end,
                p_help_text     => case
                                     when p_name_prefix = 'READ_ONLY' then
                                         q'~The component is rendered normally, irrespective of the read only condition for parent component(s).~'
                                     else
                                         q'~<p>The component is never rendered or executed.</p>
<p>Note: Changing a condition to <strong>Never</strong> will remove any existing condition. For temporarily disabling a component, Oracle recommends using the <em>Build Option</em> attribute instead.</p>~'
                                   end );
        else
            -- Remove Never for caching, ...
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Never',
                p_return_value  => 'NEVER',
                p_is_quick_pick => 'N',
                p_is_deprecated => 'Y',
                p_help_text     => q'~~' );
        end if;
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => c_name_prefix || 'CONDITION_SQL_STATEMENT',
            p_prompt               => 'SQL Query',
            p_property_type        => 'SQL',
            p_db_column            => p_db_column_expr1,
            p_display_length       => 80,
            p_max_length           => 4000,
            p_display_group_name   => p_display_group_name,
            p_sql_min_column_count => 1,
            p_help_text            => q'~Enter a SQL query for this condition.~',
            p_examples             => q'~
<dl><dt>To return successfully as one or more rows exist when checking if the employee is in department 30:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno = 30
</pre></dd>
<dt>To return successfully as no rows exist when checking if the employee is in department 40:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno = 40
</pre></dd>
</dl>~' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND' ) then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_VALUE1',
                p_prompt             => 'Value',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the value to compare against the condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_ITEM1',
                p_prompt             => 'Item',
                p_property_type      => 'ITEM',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the page or application item used in this condition. You can type in the name or pick from the list of available items.~' );
        end if;
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'CONDITION_LIST',
            p_prompt             => 'List',
            p_property_type      => 'TEXT',
            p_db_column          => p_db_column_expr2,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~Enter a colon(:) separated list of values for this condition.~' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND' ) then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_PREFERENCE',
                p_prompt             => 'Preference',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_text_case          => 'UPPER',
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter a User Preference that is compared against the specified value for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_PAGE',
                p_prompt             => 'Page',
                p_property_type      => 'PAGE',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the Page Number for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                    => l_property_id,
                p_name                  => c_name_prefix || 'CONDITION_PAGES',
                p_prompt                => 'Pages',
                p_property_type         => 'PAGE',
                p_multi_value_delimiter => ',',
                p_db_column             => p_db_column_expr1,
                p_display_length        => 80,
                p_max_length            => 4000,
                p_display_group_name    => p_display_group_name,
                p_help_text             => q'~Enter a comma separated list of Page Numbers for this condition.~' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND' ) then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_TEXT',
                p_prompt             => 'Text',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the text value for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_VALUE2',
                p_prompt             => 'Value',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr2,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the value, including <code>&ITEM.</code> substitution strings, for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_ITEM2',
                p_prompt             => 'Item',
                p_property_type      => 'ITEM',
                p_db_column          => p_db_column_expr2,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the page or application item used in this condition. You can type in the name or pick from the list of available items.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_TEXT2',
                p_prompt             => 'Value',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr2,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the value, including <code>&ITEM.</code> substitution strings, for this condition.~' );
        end if;
        --
        -- Note: the calls to store_expression_properties and store_func_body_properties have to be last, because if we have to add
        --       new properties in those procedures, it would change the ID of all properties which are created after
        --
        l_property_id := l_property_id + 1;
        store_expression_properties (
            p_start_property_id  => l_property_id,
            p_name_prefix        => c_name_prefix || 'CONDITION_',
            p_return_type        => 'BOOLEAN',
            p_db_column_lang     => p_db_column_expr2,
            p_db_column_expr     => p_db_column_expr1,
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~Enter a %language% Expression for this condition.~' );
        --
        l_property_id := l_property_id + 20;
        store_func_body_properties (
            p_start_property_id   => l_property_id,
            p_name_prefix         => c_name_prefix || 'CONDITION_',
            p_return_type         => 'BOOLEAN',
            p_db_column_lang      => p_db_column_expr2,
            p_db_column_func_body => p_db_column_expr1,
            p_display_group_name  => p_display_group_name,
            p_help_text           => q'~Enter a %language% Function Body that returns a boolean value for this condition.~' );
    end store_cond_type_properties;
    --
    --
    procedure store_condition_type_comp_prop (
        p_component_type_id           in number,
        p_name_prefix                 in varchar2,
        p_help_text_type              in varchar2,
        p_db_column_type              in varchar2,
        p_view_column_type            in varchar2 default null,
        p_db_column_expr1             in varchar2,
        p_db_column_expr2             in varchar2,
        p_reference_scope             in varchar2 default 'COMPONENT',
        p_is_common                   in boolean  default true,
        p_plugin_std_attributes       in varchar2 default null,
        p_parent_comp_prop_name       in varchar2 default null,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null,
        p_has_execute_condition       in boolean  default false,
        p_execute_cond_dep_prop_name  in varchar2 default null,
        p_execute_condition_help_text in varchar2 default null )
    is
        c_name_prefix constant varchar2( 30 ) := case when p_name_prefix is not null then p_name_prefix || '_' end;
    begin
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => c_name_prefix || 'CONDITION_TYPE',
            p_db_column                   => p_db_column_type,
            p_view_column                 => p_view_column_type,
            p_is_required                 => false,
            p_is_common                   => p_is_common,
            p_reference_scope             => p_reference_scope,
            p_plugin_std_attributes       => p_plugin_std_attributes,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_help_text                   => p_help_text_type );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => c_name_prefix || 'CONDITION_SQL_STATEMENT',
            p_db_column                   => p_db_column_expr1,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'EXISTS,NOT_EXISTS' );
        --
        store_expression_comp_props (
            p_component_type_id => p_component_type_id,
            p_name_prefix       => c_name_prefix || 'CONDITION_',
            p_return_type       => 'BOOLEAN',
            p_db_column_lang    => p_db_column_expr2,
            p_db_column_expr    => p_db_column_expr1,
            p_reference_scope   => p_reference_scope,
            p_type_property     => c_name_prefix || 'CONDITION_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id   => p_component_type_id,
            p_name_prefix         => c_name_prefix || 'CONDITION_',
            p_return_type         => 'BOOLEAN',
            p_db_column_lang      => p_db_column_expr2,
            p_db_column_func_body => p_db_column_expr1,
            p_reference_scope     => p_reference_scope,
            p_type_property       => c_name_prefix || 'CONDITION_TYPE' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND' ) then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_VALUE1',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'REQUEST_EQUALS_CONDITION,REQUEST_NOT_EQUAL_CONDITION,REQUEST_IN_CONDITION,REQUEST_NOT_IN_CONDITION,CURRENT_LANG_IN_COND1,CURRENT_LANG_NOT_IN_COND1,CURRENT_LANG_NOT_EQ_COND1,CURRENT_LANG_EQ_COND1,DAD_NAME_EQ_CONDITION,DAD_NAME_NOT_EQ_CONDITION,SERVER_NAME_EQ_CONDITION,SERVER_NAME_NOT_EQ_CONDITION,HTTP_HOST_EQ_CONDITION,HTTP_HOST_NOT_EQ_CONDITION' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_ITEM1',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_parent_comp_prop_name       => p_parent_comp_prop_name,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'VAL_OF_ITEM_IN_COND_EQ_COND2,VAL_OF_ITEM_IN_COND_NOT_EQ_COND2,VALUE_OF_ITEM_IN_COND_1_EQ_COND_2,VALUE_OF_ITEM_IN_COND_1_NOT_EQ_COND_2,ITEM_IS_NULL,ITEM_IS_NOT_NULL,ITEM_IS_ZERO,ITEM_IS_NOT_ZERO,ITEM_IS_NULL_OR_ZERO,ITEM_NOT_NULL_OR_ZERO,ITEM_CONTAINS_NO_SPACES,ITEM_IS_NUMERIC,ITEM_IS_NOT_NUMERIC,ITEM_IS_ALPHANUMERIC,VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST,VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND' ) then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_PREFERENCE',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'USER_PREF_IN_COND_EQ_COND2,USER_PREF_IN_COND_NOT_EQ_COND2' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_PAGE',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'CURRENT_PAGE_EQUALS_CONDITION,CURRENT_PAGE_NOT_EQUAL_CONDITION' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_PAGES',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'CURRENT_PAGE_IN_CONDITION,CURRENT_PAGE_NOT_IN_CONDITION' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND' ) then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_TEXT',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2,DISPLAY_COND_IN_COND_TEXT,DISPLAY_COND_NOT_IN_COND_TEXT,DISPLAY_COND_EQUAL_COND_TEXT,DISP_COND_NOT_EQUAL_COND_TEXT' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_VALUE2',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'VAL_OF_ITEM_IN_COND_EQ_COND2,VAL_OF_ITEM_IN_COND_NOT_EQ_COND2,VALUE_OF_ITEM_IN_COND_1_EQ_COND_2,VALUE_OF_ITEM_IN_COND_1_NOT_EQ_COND_2,USER_PREF_IN_COND_EQ_COND2,USER_PREF_IN_COND_NOT_EQ_COND2' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_LIST',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST,VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_ITEM2',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_parent_comp_prop_name       => p_parent_comp_prop_name,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_TEXT2',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'DISPLAY_COND_IN_COND_TEXT,DISPLAY_COND_NOT_IN_COND_TEXT,DISPLAY_COND_EQUAL_COND_TEXT,DISP_COND_NOT_EQUAL_COND_TEXT' );
        end if;
        --
        if p_has_execute_condition then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXECUTE_CONDITION',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'Y',
                p_depending_on_comp_prop_name  => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type  => 'NOT_IN_LIST',
                p_depending_on_expression      => 'NEVER,$NULL$',
                p_depending_on_comp_prop_name2 => p_execute_cond_dep_prop_name,
                p_depending_on_condition_type2 => case when p_execute_cond_dep_prop_name is not null then 'NOT_NULL' end,
                p_help_text                    => p_execute_condition_help_text );
        end if;
    end store_condition_type_comp_prop;
    --
    --
    procedure store_val_type_properties (
        p_start_property_id in number,
        p_name_prefix       in varchar2 )
    is
        c_name_prefix constant   varchar2( 30 )          := case when p_name_prefix is not null then p_name_prefix || '_' end;
        c_item_or_column_lower   constant varchar2( 10 ) := case when p_name_prefix is null then 'item' else 'column' end;
        c_item_or_column_initcap constant varchar2( 10 ) := initcap( c_item_or_column_lower );
        l_property_id number := p_start_property_id;
    begin
        --
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VALIDATION_TYPE',
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'VALIDATION_TYPE',
            p_display_group_name => 'VALIDATION',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~<p>Select the type of equality to be tested for this validation. Depending on your selection, one or more additional attributes are required to fully define this validation.</p>
<p>If the validation passes the equality test, or evaluates to TRUE, then the validation error message does not display.
Validation error messages display when the validation fails the equality test, or evaluates to FALSE, or a non-empty text string is returned.</p>
<p>A validation failure does not stop subsequent validations from being evaluated, therefore, each subsequent validation may raise an additional validation error message. If you want to stop subsequent validations from executing then you need to add additional logic into the first failed validation, such as setting a page item to a specific value, and then enhancing the conditional logic on subsequent validations as needed. Subsequent processes and branches are not executed if one or more validations fail evaluation.</p>~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Rows returned',
            p_return_value  => 'EXISTS',
            p_help_text     => q'~Passes if the SQL query returns one or more rows.<br>
Fails if the SQL query returns no rows.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'No Rows returned',
            p_return_value  => 'NOT_EXISTS',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Passes if the SQL query returns no rows.<br>
Fails if the SQL query returns one or more rows.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Expression',
            p_return_value  => 'EXPRESSION',
            p_help_text     => q'~Passes if the expression evaluates to TRUE.<br>
Fails if the expression evaluates to FALSE.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'PL/SQL Error',
            p_return_value  => 'PLSQL_ERROR',
            p_help_text     => q'~Passes if the PL/SQL code executes without generating an error.<br>
Fails if the PL/SQL expression generates an error when it executes.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body (returning Boolean)',
            p_return_value  => 'FUNC_BODY_RETURNING_BOOLEAN',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Passes if the Function body returns TRUE.<br>
Fails if the Function body returns FALSE.<br>
Note: The result returned from the function must be a boolean.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body (returning Error Text)',
            p_return_value  => 'FUNC_BODY_RETURNING_ERR_TEXT',
            p_help_text     => q'~Passes if the Function body returns an empty string.<br>
Fails if the Function body returns a non-empty string. The returned string displays as the error message for the validation.<br>
Note: The result returned from the function must be a text string.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT NULL',
            p_return_value  => 'ITEM_NOT_NULL',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not empty.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is empty.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT NULL or zero',
            p_return_value  => 'ITEM_NOT_NULL_OR_ZERO',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not empty and is not the number zero.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is empty or is the number zero.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT zero',
            p_return_value  => 'ITEM_NOT_ZERO',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not the number zero.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is the number zero.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' contains no spaces',
            p_return_value  => 'ITEM_CONTAINS_NO_SPACES',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value has no spaces.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value contains spaces.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is alphanumeric',
            p_return_value  => 'ITEM_IS_ALPHANUMERIC',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is alphanumeric, containing only letters and numbers and no special characters.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value includes special characters.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is numeric',
            p_return_value  => 'ITEM_IS_NUMERIC',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is numeric.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value contains non-numeric characters, including spaces, other than leading spaces.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is a valid date',
            p_return_value  => 'ITEM_IS_DATE',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is a valid date.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not a valid date.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is a valid timestamp',
            p_return_value  => 'ITEM_IS_TIMESTAMP',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is a valid timestamp.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not a valid timestamp.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' = Value',
            p_return_value  => 'ITEM_IN_VALIDATION_EQ_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is equal to the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not equal to the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' != Value',
            p_return_value  => 'ITEM_IN_VALIDATION_NOT_EQ_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not equal to the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is equal to the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is contained in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is completely contained in the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not completely contained in the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT contained in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_NOT_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not completely contained in the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is completely contained in the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' contains only characters specified in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_CONTAINS_ONLY_CHAR_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value contains only characters found in the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value contains any characters that are not in the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' contains at least one of the characters in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_CONTAINS_AT_LEAST_ONE_CHAR_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value includes at least one character from the the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value does not include any characters from the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' does NOT contain any of the characters in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_CONTAINS_NO_CHAR_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value does not include any characters from the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value includes at least one character from the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' matches Regular Expression',
            p_return_value  => 'REGULAR_EXPRESSION',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value matches the regular expression you enter into the <em>Value</em> attribute.
Fails if the ~' || c_item_or_column_initcap || q'~ value does not match the regular expression you enter into the <em>Value</em> attribute.<br>
Note: Regular expressions enable you to search for patterns in string data by using standardized syntax conventions, rather than just a straight character comparisons.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => c_name_prefix || 'VAL_SQL_STATEMENT',
            p_prompt               => 'SQL Query',
            p_property_type        => 'SQL',
            p_db_column            => 'VALIDATION',
            p_display_length       => 80,
            p_max_length           => 4000,
            p_display_group_name   => 'VALIDATION',
            p_sql_min_column_count => 1,
            p_help_text            => q'~Enter a SQL query to define this validation.~',
            p_examples             => q'~
<dl><dt>For <strong>Rows Returned</strong>, to pass the validation, and not raise an error message, when the employee is in department 30:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno = 30
</pre></dd>
<dt>For <strong>No Rows Returned</strong, to pass the validation, and not raise an error message, when the employee is in department 40:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno <> 40
</pre></dd>
</dl>~' );
        --
        if p_name_prefix is null then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                       => l_property_id,
                p_name                     => 'VAL_ITEM',
                p_prompt                   => 'Item',
                p_property_type            => 'ITEM',
                p_db_column                => 'VALIDATION',
                p_display_length           => 80,
                p_max_length               => 255,
                p_display_group_name       => 'VALIDATION',
                p_lov_component_type_scope => 'APPLICATION',
                p_help_text                => q'~Enter a page or application item used in this validation. You can type in the name or pick from the list of available items.~' );
        end if;
        --
        if p_name_prefix = 'REGION' then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'VAL_COLUMN',
                p_prompt             => 'Column',
                p_property_type      => 'REGION COLUMN',
                p_db_column          => 'VALIDATION',
                p_display_length     => 80,
                p_max_length         => 128,
                p_display_group_name => 'VALIDATION',
                p_help_text          => q'~Enter a case-sensitive column name for this validation. You can type in the name or pick from the list of available columns.~' );
        end if;
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VAL_VALUE',
            p_prompt             => 'Value',
            p_property_type      => 'TEXT EDITOR',
            p_db_column          => 'VALIDATION2',
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter the value used to compare against the value of the ~' || c_item_or_column_lower || q'~ column specified for this validation.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VAL_REGULAR_EXPRESSION',
            p_prompt             => 'Regular Expression',
            p_property_type      => 'TEXT',
            p_db_column          => 'VALIDATION2',
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter the regular expression used to match the ~' || c_item_or_column_lower || q'~ specified for this validation.~' );
        --
        -- Note: the calls to store_expression_properties and store_func_body_properties have to be last, because if we have to add
        --       new properties in those procedures, it would change the ID of all properties which are created after
        --
        --
        l_property_id := l_property_id + 1;
        store_func_body_properties (
            p_start_property_id   => l_property_id,
            p_name_prefix         => c_name_prefix || 'VAL_',
            p_postfix             => 'ERROR',
            p_return_type         => 'VARCHAR2',
            p_prompt              => '%language% Function Body Returning Error Text',
            p_db_column_lang      => 'VALIDATION2',
            p_db_column_func_body => 'VALIDATION',
            p_display_group_name  => 'VALIDATION',
            p_help_text           => q'~Enter a %language% function body that returns the error message text as the basis for this validation.~',
            p_examples_plsql      => q'~
<dl><dt>To pass the validation, and not raise the sepcified error message, when the employee is in department 30:</dt>
<dd><pre>
begin
    if :P2_DEPTNO <> 30 then
        return 'Employee is in Department ' || :P2_DEPTNO || ' and not Department 30';
    end if;
    return null;
end;
</pre></dd>
</dl>~',
            p_examples_js         => q'~
<dl><dt>To pass the validation, and not raise the sepcified error message, when the employee is in department 30:</dt>
<dd><pre>
if ( apex.env.P2_DEPTNO !== "30" ) {
    return "Employee is in Department " + apex.env.P2_DEPTNO + " and not Department 30";
}
return "";
</pre></dd>
</dl>~' );
        --
        l_property_id := l_property_id + 20;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VAL_PLSQL',
            p_prompt             => 'PL/SQL Code raising Error',
            p_property_type      => 'PLSQL',
            p_db_column          => 'VALIDATION',
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter a PL/SQL expression that raises an error as the basis for this validation.~',
            p_examples           => q'~
<dl><dt>To pass the validation, and not raise the sepcified error, when the employee is in department 30:</dt>
<dd><pre>
begin
    if :P2_DEPTNO <> 30 then
        raise_application_error( -20000, 'Employee is in Department ' || :P2_DEPTNO || ' and not Department 30' );
    end if;
end;
</pre></dd>
</dl>~' );
        l_property_id := l_property_id + 1;
        store_expression_properties (
            p_start_property_id  => l_property_id,
            p_name_prefix        => c_name_prefix || 'VAL_',
            p_return_type        => 'BOOLEAN',
            p_db_column_lang     => 'VALIDATION2',
            p_db_column_expr     => 'VALIDATION',
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter a %language% expression to define the validation.~',
            p_examples_sql       => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is a manager or the department is located in Chicago:</dt>
<dd><pre>
:P2_JOB = 'MANAGER' or
exists ( select 1
           from dept
          where deptno = :P2_DEPTNO
            and loc    = 'CHICAGO' )
</pre></dd>
</dl>~',
            p_examples_plsql     => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( :P2_DEPTNO = 30 or :P2_JOB = 'MANAGER' )
</pre></dd>
</dl>~',
            p_examples_js        => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( apex.env.P2_DEPTNO === "30" || apex.env.P2_JOB == "MANAGER" )
</pre></dd>
</dl>~' );
        --
        l_property_id := l_property_id + 20;
        store_func_body_properties (
            p_start_property_id   => l_property_id,
            p_name_prefix         => c_name_prefix || 'VAL_',
            p_return_type         => 'BOOLEAN',
            p_db_column_lang      => 'VALIDATION2',
            p_db_column_func_body => 'VALIDATION',
            p_display_group_name  => 'VALIDATION',
            p_help_text           => q'~Enter a %language% function body that returns a BOOLEAN value as the basis for this validation.~',
            p_examples_plsql      => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30:</dt>
<dd><pre>
begin
    if :P2_DEPTNO = 30 then
        return true;
    else
        return false;
    end if;
end;
</pre></dd>
</dl>~',
            p_examples_js         => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30:</dt>
<dd><pre>
if ( apex.env.P2_DEPTNO === "30" ) {
    return true;
} else {
    return false;
}
</pre></dd>
</dl>~' );
        --
    end store_val_type_properties;
    --
    --
    procedure store_val_type_comp_props (
        p_name_prefix in varchar2 )
    is
        c_name_prefix     constant varchar2( 30 ) := case when p_name_prefix is not null then p_name_prefix || '_' end;
        c_type_property   constant varchar2( 30 ) := c_name_prefix || 'VALIDATION_TYPE';
        c_reference_scope constant varchar2( 30 ) := case when p_name_prefix is not null then 'ROW' else 'COMPONENT' end;
    begin
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_type_property,
            p_db_column                   => null,
            p_view_column                 => 'VALIDATION_TYPE',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'EXPRESSION',
            p_depending_on_comp_prop_name => 'VALIDATION_REGION',
            p_depending_on_condition_type => case when p_name_prefix is not null then 'NOT_NULL' else 'NULL' end );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_name_prefix || 'VAL_SQL_STATEMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => c_reference_scope,
            p_depending_on_comp_prop_name => c_type_property,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'EXISTS,NOT_EXISTS' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => c_name_prefix || 'VAL_',
            p_return_type       => 'BOOLEAN',
            p_reference_scope   => c_reference_scope,
            p_type_property     => c_type_property );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => c_name_prefix || 'VAL_',
            p_return_type       => 'BOOLEAN',
            p_reference_scope   => c_reference_scope,
            p_type_property     => c_type_property,
            p_type_value        => 'FUNC_BODY_RETURNING_BOOLEAN' );
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => c_name_prefix || 'VAL_',
            p_postfix             => 'ERROR',
            p_return_type         => 'VARCHAR2',
            p_db_column_lang      => null,
            p_db_column_func_body => null,
            p_reference_scope     => c_reference_scope,
            p_type_property       => c_type_property,
            p_type_value          => 'FUNC_BODY_RETURNING_ERR_TEXT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_name_prefix || 'VAL_PLSQL',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => c_reference_scope,
            p_depending_on_comp_prop_name => c_type_property,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLSQL_ERROR' );
        --
        if p_name_prefix is null then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => c_name_prefix || 'VAL_ITEM',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_reference_on_delete          => 'WEAK_CASCADE',
                p_depending_on_comp_prop_name  => c_type_property,
                p_depending_on_condition_type  => 'IN_LIST_LIKE',
                p_depending_on_expression      => 'ITEM,REGULAR_EXPRESSION' );
        end if;
        --
        if p_name_prefix = 'REGION' then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => c_name_prefix || 'VAL_COLUMN',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_parent_comp_prop_name        => 'VALIDATION_REGION',
                p_depending_on_comp_prop_name  => c_type_property,
                p_depending_on_condition_type  => 'IN_LIST_LIKE',
                p_depending_on_expression      => 'ITEM,REGULAR_EXPRESSION' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => c_name_prefix || 'VAL_REGULAR_EXPRESSION',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => c_type_property,
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'REGULAR_EXPRESSION' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => c_name_prefix || 'VAL_VALUE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => c_reference_scope,
            p_depending_on_comp_prop_name  => c_type_property,
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM_IN_VALIDATION_EQ_STRING2,ITEM_IN_VALIDATION_NOT_EQ_STRING2,ITEM_IN_VALIDATION_IN_STRING2,ITEM_IN_VALIDATION_NOT_IN_STRING2,ITEM_IN_VALIDATION_CONTAINS_AT_LEAST_ONE_CHAR_IN_STRING2,ITEM_IN_VALIDATION_CONTAINS_ONLY_CHAR_IN_STRING2,ITEM_IN_VALIDATION_CONTAINS_NO_CHAR_IN_STRING2' );
        --
    end store_val_type_comp_props;
    --
    --
    procedure store_audit_info_comp_prop (
        p_component_type_id in number,
        p_db_column_prefix  in varchar2 default 'LAST_' )
    is
    begin
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'CHANGED_BY',
            p_db_column             => p_db_column_prefix || 'UPDATED_BY',
            p_is_required           => false,
            p_is_common             => false,
            p_is_read_only          => true,
            p_supports_substitution => false );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'CHANGED_ON',
            p_db_column             => p_db_column_prefix || 'UPDATED_ON',
            p_is_required           => false,
            p_is_common             => false,
            p_is_read_only          => true,
            p_supports_substitution => false );
    end store_audit_info_comp_prop;
    --
    --
    procedure store_standard_attr_comp_prop (
        p_component_type_id        in number   default l_component_type_id,
        p_has_condition            in boolean  default false,
        p_cond_type_name_prefix    in varchar2 default null,
        p_cond_type_help_text      in varchar2 default null,
        p_cond_type_db_column      in varchar2 default null,
        p_cond_type_view_db_column in varchar2 default null,
        p_cond_exp1_db_column      in varchar2 default null,
        p_cond_exp2_db_column      in varchar2 default null,
        p_has_authorization_scheme in boolean  default false,
        p_authorization_db_column  in varchar2 default null,
        p_has_build_option         in boolean  default false,
        p_build_option_db_column   in varchar2 default null,
        p_has_comments             in boolean  default false,
        p_comment_db_column        in varchar2 default null,
        p_audit_db_column_prefix   in varchar2 default 'LAST_' )
    is
    begin
        if p_has_condition then
            store_condition_type_comp_prop (
                p_component_type_id => p_component_type_id,
                p_name_prefix       => p_cond_type_name_prefix,
                p_help_text_type    => p_cond_type_help_text,
                p_db_column_type    => p_cond_type_db_column,
                p_view_column_type  => p_cond_type_view_db_column,
                p_db_column_expr1   => p_cond_exp1_db_column,
                p_db_column_expr2   => p_cond_exp2_db_column );
        end if;
        --
        if p_has_authorization_scheme or p_authorization_db_column is not null then
            store_component_prop (
                p_component_type_id   => p_component_type_id,
                p_property_name       => 'AUTHORIZATION_SCHEME',
                p_db_column           => p_authorization_db_column,
                p_is_required         => false,
                p_is_common           => false,
                p_reference_on_delete => 'WEAK_NULL' );
        end if;
        --
        if p_has_build_option or p_build_option_db_column is not null then
            store_component_prop (
                p_component_type_id   => p_component_type_id,
                p_property_name       => 'BUILD_OPTION',
                p_db_column           => p_build_option_db_column,
                p_is_required         => false,
                p_is_common           => false,
                p_reference_on_delete => 'WEAK_NULL' );
        end if;
        --
        if p_has_comments or p_comment_db_column is not null then
            store_component_prop (
                p_component_type_id     => p_component_type_id,
                p_property_name         => 'COMMENTS',
                p_db_column             => p_comment_db_column,
                p_is_required           => false,
                p_is_common             => false,
                p_supports_substitution => false );
        end if;
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id,
            p_db_column_prefix  => p_audit_db_column_prefix );
        --
    end store_standard_attr_comp_prop;
    --
    --
    procedure store_process_type_comp_prop (
        p_name_db_column           in varchar2,
        p_name_help_text           in varchar2,
        p_db_column_prefix         in varchar2,
        p_execution_seq_db_column  in varchar2,
        p_error_message_db_column  in varchar2,
        p_error_message_help_text  in varchar2 default null,
        p_stop_execution_help_text in varchar2 default null,
        p_has_authorization_scheme in boolean,
        p_build_option_db_column   in varchar2,
        p_comment_db_column        in varchar2 )
    is
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => p_name_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => case l_component_type_id
                                         when c_comp_type_app_process  then 'APPLICATION'
                                         when c_comp_type_page_process then 'PAGE'
                                         else 'PARENT'
                                       end,
            p_supports_substitution => false,
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => p_name_help_text );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PROCESS_TYPE',
            p_db_column         => case when p_db_column_prefix is not null then p_db_column_prefix || 'TYPE' end,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'NATIVE_PLSQL' );
        --
        if l_component_type_id = c_comp_type_page_process then
            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'PROCESS_DATA_SOURCE_REGION',
                p_db_column             => null,
                p_is_required           => null, -- defined by plug-in standard attributes
                p_is_common             => true,
                p_reference_on_delete   => 'WEAK_CASCADE',
                p_plugin_std_attributes => 'FORM' );
            --
            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'PROCESS_REGION',
                p_db_column             => null,
                p_is_required           => null, -- defined by plug-in standard attributes
                p_is_common             => true,
                p_reference_on_delete   => 'WEAK_CASCADE',
                p_plugin_std_attributes => 'REGION' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PLSQL_PROCESS_LOCATION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'LOCAL',
            p_depending_on_comp_prop_name => 'PROCESS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NATIVE_PLSQL' );
        --
        store_snippet_comp_props (
            p_component_type_id           => l_component_type_id,
            p_name_prefix                 => 'SOURCE_',
            p_db_column_lang              => case when p_db_column_prefix is not null then p_db_column_prefix || 'CLOB_LANGUAGE' end,
            p_db_column_snippet           => case when p_db_column_prefix is not null then p_db_column_prefix || 'CLOB' end,
            p_reference_scope             => case l_component_type_id
                                               when c_comp_type_page_process then 'DEPENDS'
                                               else 'COMPONENT'
                                             end,
            p_type_property               => 'PLSQL_PROCESS_LOCATION',
            p_type_value                  => 'LOCAL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_REMOTE_ORACLE_DATABASE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'PLSQL_PROCESS_LOCATION',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REMOTE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PLSQL_CODE',
            p_db_column                   => case
                                               when p_db_column_prefix is not null then p_db_column_prefix || 'CLOB'
                                               else 'PROCESS_SQL_CLOB'
                                             end,
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => case l_component_type_id
                                               when c_comp_type_page_process then 'DEPENDS'
                                               else 'COMPONENT'
                                             end,
            p_depending_on_comp_prop_name => 'PLSQL_PROCESS_LOCATION',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REMOTE',
            p_help_text                   => q'~Enter the PL/SQL code to be executed on the remote database.~' );
        --
        store_plugin_attr_seq_offset (
            p_component_type_id => l_component_type_id );
        --
        if l_component_type_id = c_comp_type_task_def_action then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'ON_EVENT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'COMPLETE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'OUTCOME',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'ON_EVENT',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'COMPLETE' );
        end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'EXECUTION_SEQUENCE',
            p_db_column         => p_execution_seq_db_column,
            p_is_required       => true,
            p_is_common         => true );
        --
        if l_component_type_id = c_comp_type_app_process then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'APP_PROCESS_POINT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'BEFORE_HEADER' );
            --
        elsif l_component_type_id = c_comp_type_page_process then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PROCESS_POINT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'AFTER_SUBMIT' );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'RUN_PROCESS',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => false,
                p_default_value     => 'N' );
            --
            -- $$$ todo do we have to expose SUCCESS_MESSAGE for task def actions?
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'SUCCESS_MESSAGE',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => true,
                p_help_text         => q'~<p>Enter the success message for this process.
If the process runs and does not generate an error, then this process success message displays in the notification section of the resulting page displayed.
If you are branching to another page via a URL redirect, you may need to check the <em>preserve success message</em> attribute.</p>',
<p>For multi row processes, the following substitution string can be used to get the number of records processed:</p>
<ul>
  <li>Insert: <code>#MRI_COUNT#</code></li>
  <li>Update: <code>#MRU_COUNT#</code></li>
  <li>Delete: <code>#MRD_COUNT#</code></li>
</ul>
<p>For the Send E-Mail process type the substitution string <code>#TO#</code> can be used to get the addressees of the e-mail.</p>
<p>Plug-ins can have other substitution strings as well. See Plug-in documentation for details.</p>~' );
            --
        end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ERROR_MESSAGE',
            p_db_column         => p_error_message_db_column,
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => coalesce( p_error_message_help_text, q'~
<p>Enter the error message for this process.
This message displays if an unhandled exception is raised. After any error processing stops, a rollback is issued and an error message displays.</p>
<p>Note: the SQL error message (sqlerrm) displays by default if <strong>On Error Page</strong> is defined as the error display location, there is no need to add <code>#SQLERRM#</code> to your error text.</p>
<p>Error messages can include the following substitution strings:</p>
<dl>
  <dt><code>#SQLERRM_TEXT#</code></dt>
  <dd>Text of error message without the error number.</dd>
  <dt><code>#SQLERRM#</code></dt>
  <dd>Complete error message.</dd>
</dl>~' ) );
        --
        if l_component_type_id in ( c_comp_type_app_process, c_comp_type_page_process ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PROCESS_ERROR_DISPLAY_LOCATION',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'INLINE_IN_NOTIFICATION',
                p_depending_on_comp_prop_name => case l_component_type_id
                                                   when c_comp_type_app_process  then 'APP_PROCESS_POINT'
                                                   when c_comp_type_page_process then 'PROCESS_POINT'
                                                 end,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'ON_SUBMIT_BEFORE_COMPUTATION,AFTER_SUBMIT' );
        else
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'ERROR_STOP_EXECUTION_ON_ERROR',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'Y',
                p_help_text         => p_stop_execution_help_text );
        end if;
        --
        if l_component_type_id = c_comp_type_app_process then
            --
            store_condition_type_comp_prop (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => null,
                p_help_text_type    => null,
                p_db_column_type    => 'PROCESS_WHEN_TYPE',
                p_view_column_type  => 'CONDITION_TYPE',
                p_db_column_expr1   => 'PROCESS_WHEN',
                p_db_column_expr2   => 'PROCESS_WHEN2' );
            --
        elsif l_component_type_id = c_comp_type_page_process then
            --
            store_component_prop (
                p_component_type_id   => l_component_type_id,
                p_property_name       => 'WHEN_BUTTON_PRESSED',
                p_db_column           => 'PROCESS_WHEN_BUTTON_ID',
                p_is_required         => false,
                p_is_common           => true,
                p_reference_on_delete => 'WEAK_NULL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'EXECUTION_SCOPE',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'Y',
                p_depending_on_comp_prop_name => 'PROCESS_REGION',
                p_depending_on_condition_type => 'NOT_NULL' );
            --
            store_condition_type_comp_prop (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => null,
                p_help_text_type              => null,
                p_reference_scope             => 'DEPENDS',
                p_db_column_type              => 'PROCESS_WHEN_TYPE',
                p_view_column_type            => 'CONDITION_TYPE',
                p_db_column_expr1             => 'PROCESS_WHEN',
                p_db_column_expr2             => 'PROCESS_WHEN2',
                p_has_execute_condition       => true,
                p_execute_cond_dep_prop_name  => 'PROCESS_REGION' );
        else
            -- Automation and Task Def Actions
            store_condition_type_comp_prop (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => 'ACTION',
                p_help_text_type              => null,
                p_db_column_type              => 'CONDITION_TYPE',
                p_view_column_type            => 'CONDITION_TYPE',
                p_db_column_expr1             => 'CONDITION_EXPR1',
                p_db_column_expr2             => 'CONDITION_EXPR2',
                p_has_execute_condition       => ( l_component_type_id = c_comp_type_automation_action ),
                p_execute_condition_help_text => q'~Specify whether the condition is to be executed for each row or only once.~' );
        end if;
        --
        store_standard_attr_comp_prop (
            p_component_type_id        => l_component_type_id,
            p_has_authorization_scheme => p_has_authorization_scheme,
            p_authorization_db_column  => case when p_has_authorization_scheme then 'SECURITY_SCHEME' end,
            p_has_build_option         => true,
            p_build_option_db_column   => p_build_option_db_column,
            p_has_comments             => true,
            p_comment_db_column        => p_comment_db_column );
        --
    end store_process_type_comp_prop;
    --
    --
    procedure store_computation_comp_prop (
        p_component_type_id in number )
    is
    begin
        --
        store_component_prop (
            p_component_type_id   => p_component_type_id,
            p_property_name       => 'COMPUTATION_ITEM_NAME',
            p_db_column           => null,
            p_is_required         => true,
            p_is_common           => true,
            p_reference_on_delete => 'WEAK_CASCADE' );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'EXECUTION_SEQUENCE',
            p_db_column         => 'COMPUTATION_SEQUENCE',
            p_is_required       => true,
            p_is_common         => false );
        --
        -- page and application computations have different default values
        -- for computation point and computation type
        if p_component_type_id = c_comp_type_page_computation then
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_POINT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'BEFORE_BOX_BODY' );
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_TYPE',
                p_db_column         => null,
                p_view_column       => 'COMPUTATION_TYPE',
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'QUERY' );
        else
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_POINT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'ON_NEW_INSTANCE' );
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_TYPE',
                p_db_column         => null,
                p_view_column       => 'COMPUTATION_TYPE',
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'STATIC' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_STATIC_VALUE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC_ASSIGNMENT' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_STATEMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'QUERY' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_COLON',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'QUERY_COLON' );
        --
        store_expression_comp_props (
            p_component_type_id => p_component_type_id,
            p_name_prefix       => 'COMPUTATION_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'COMPUTATION_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => p_component_type_id,
            p_name_prefix       => 'COMPUTATION_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'COMPUTATION_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_ITEM_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM_VALUE' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_PREFERENCE_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SET_ITEM_EQUAL_THIS_PREFERENCE' );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'ERROR_MESSAGE',
            p_db_column         => 'COMPUTATION_ERROR_MESSAGE',
            p_is_required       => false,
            p_is_common         => false,
        p_help_text         => q'~<p>Enter an error message that displays if this computation fails.
    Use <code>#SQLERRM#</code> as a substitution string for the SQL error message resulting from a failed computation.</p>
    <p>Computations are designed to always succeed, and only fail due to unanticipated errors.
    It is not advisable to implement a computation that regularly fails and acts as a psuedo-validation utilizing this error message.~' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id        => p_component_type_id,
            p_has_condition            => true,
            p_cond_type_db_column      => 'COMPUTE_WHEN_TYPE',
            p_cond_type_view_db_column => 'CONDITION_TYPE',
            p_cond_exp1_db_column      => 'COMPUTE_WHEN',
            p_cond_exp2_db_column      => 'COMPUTE_WHEN_TEXT',
            p_has_authorization_scheme => true,
            p_authorization_db_column  => 'SECURITY_SCHEME',
            p_has_build_option         => true,
            p_has_comments             => true,
            p_comment_db_column        => 'COMPUTATION_COMMENT' );
        --
    end store_computation_comp_prop;
    --
    procedure store_grid_comp_prop (
        p_component_type_id     in number,
        p_new_grid_db_column    in varchar2 default 'GRID_NEW_GRID',           -- Pass if component's DB column is not GRID_NEW_GRID
        p_new_row_db_column     in varchar2 default 'GRID_NEW_ROW',            -- Pass if component's DB column is not GRID_NEW_ROW
        p_row_css_db_column     in varchar2 default 'GRID_ROW_CSS_CLASSES',    -- Pass if component's DB column is not GRID_ROW_CSS_CLASSES
        p_column_db_column      in varchar2 default 'GRID_COLUMN',             -- Pass if component's DB column is not GRID_COLUMN
        p_new_column_db_column  in varchar2 default 'GRID_NEW_COLUMN',         -- Pass if component's DB column is not GRID_NEW_COLUMN
        p_column_span_db_column in varchar2 default 'GRID_COLUMN_SPAN',        -- Pass if component's DB column is not GRID_COLUMN_SPAN
        p_row_span_db_column    in varchar2 default 'GRID_ROW_SPAN',           -- Pass if component's DB column is not GRID_ROW_SPAN
        p_column_css_db_column  in varchar2 default 'GRID_COLUMN_CSS_CLASSES', -- Pass if component's DB column is not GRID_COLUMN_CSS_CLASSES
        p_column_attr_db_column in varchar2 default 'GRID_COLUMN_ATTRIBUTES',  -- Pass if component's DB column is not GRID_COLUMN_ATTRIBUTES
        p_has_label_column_span in boolean  default false,
        p_plugin_std_attributes in varchar2 default null )
    is
    begin
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'GRID_NEW_GRID',
            p_db_column             => p_new_grid_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'N',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'GRID_NEW_ROW',
            p_db_column             => p_new_row_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'Y',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_ROW_CSS_CLASSES',
            p_db_column             => p_row_css_db_column,
            p_is_required           => false,
            p_is_common             => false,
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'GRID_COLUMN',
            p_db_column             => p_column_db_column,
            p_is_required           => false,
            p_is_common             => true,
            p_null_text             => 'Automatic',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_NEW_COLUMN',
            p_db_column             => p_new_column_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'Y',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_COLUMN_SPAN',
            p_db_column             => p_column_span_db_column,
            p_is_required           => false,
            p_is_common             => true,
            p_null_text             => 'Automatic',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        if p_row_span_db_column is not null then
            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'GRID_ROW_SPAN',
                p_db_column             => p_row_span_db_column,
                p_is_required           => false,
                p_is_common             => false,
                p_plugin_std_attributes => p_plugin_std_attributes );
        end if;
        --
        if p_has_label_column_span then
            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'GRID_LABEL_COLUMN_SPAN',
                p_db_column             => null,
                p_is_required           => false,
                p_is_common             => false,
                p_null_text             => 'Page Template Default',
                p_plugin_std_attributes => p_plugin_std_attributes );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_COLUMN_CSS_CLASSES',
            p_db_column             => p_column_css_db_column,
            p_is_required           => false,
            p_is_common             => false,
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_COLUMN_ATTRIBUTES',
            p_db_column             => p_column_attr_db_column,
            p_is_required           => false,
            p_is_common             => false,
            p_plugin_std_attributes => p_plugin_std_attributes );

    end store_grid_comp_prop;
    --
    procedure store_report_attributes
    is
    begin
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'NUMBER_OF_ROWS_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NUMBER_OF_ROWS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '15',
            p_depending_on_comp_prop_name => 'NUMBER_OF_ROWS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NUMBER_OF_ROWS_ITEM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'NUMBER_OF_ROWS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'TEMPLATE_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'THEME' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'REPORT_TEMPLATE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '#DEFAULT_REPORT_TEMPLATE#',
            p_depending_on_comp_prop_name => 'TEMPLATE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'THEME' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'REPORT_PREDEFINED_TEMPLATE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '1',
            p_depending_on_comp_prop_name => 'TEMPLATE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PREDEFINED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TEMPLATE_OPTIONS',
            p_db_column                   => 'COMPONENT_TEMPLATE_OPTIONS',
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => '#DEFAULT##PRESET_TEMPLATE_OPTIONS#',
            p_parent_comp_prop_name       => 'REPORT_TEMPLATE',
            p_depending_on_comp_prop_name => 'REPORT_TEMPLATE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'APPEARANCE_CSS_CLASSES',
            p_db_column                   => 'REGION_SUB_CSS_CLASSES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'TEMPLATE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'THEME',
            p_help_text                   => q'~Enter additional space delimited CSS classes you want to add to your report. Your report template must include the <code>#COMPONENT_CSS_CLASSES#</code> substitution string.~',
            p_comment_text                => '$$$ we should only show this property if the template supports the #COMPONENT_CSS_CLASSES# placeholder' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'SHOW_NULL_VALUES_AS',
            p_db_column         => 'PLUG_QUERY_SHOW_NULLS_AS',
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => null );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'CLASSIC_REPORT_PAGINATION_TYPE',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true,
            p_default_value     => 'NEXT_PREVIOUS_LINKS',
            p_null_text         => 'No Pagination (Show All Rows)' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PAGINATION_DISPLAY_POSITION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'BOTTOM_RIGHT',
            p_depending_on_comp_prop_name => 'CLASSIC_REPORT_PAGINATION_TYPE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PARTIAL_PAGE_REFRESH',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LAZY_LOADING',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => 'PARTIAL_PAGE_REFRESH',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'MAXIMUM_ROW_COUNT',
            p_db_column         => 'PLUG_QUERY_ROW_COUNT_MAX',
            p_is_required       => false,
            p_is_common         => false,
            p_comment_text      => q'~$$$ should be moved to region attributes, I think that is needed by all plug-ins~' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'WHEN_NO_DATA_FOUND_MESSAGE',
            p_db_column         => 'PLUG_QUERY_NO_DATA_FOUND',
            p_is_required       => false,
            p_is_common         => false );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'WHEN_MORE_DATA_FOUND_MESSAGE',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'REPORT_SUM_LABEL',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'BREAK_COLUMNS',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_null_text         => 'No Break' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_COLUMN_STYLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'DEFAULT_BREAK_FORMATTING',
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_REPEAT_HEADING_FORMAT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMN_STYLE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REPEAT_HEADINGS_ON_BREAK_1' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_COLUMN_TEXT_FORMAT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_BEFORE_ROW',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_GENERIC_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_AFTER_ROW',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'CUSTOM_ATTRIBUTES',
            p_db_column         => 'REPORT_ATTRIBUTES_SUBSTITUTION',
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => q'~The report template substitution string <code>#REPORT_ATTRIBUTES#</code> is replaced with this value.  The <code>#REPORT_ATTRIBUTES#</code> substitution string is only replaced in the before rows attribute of the report template.~' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'STRIP_HTML',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'SORT_NULLS',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'L' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'HEADINGS_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'COLON_DELMITED_LIST' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HEADING_TYPE_PLSQL_FUNCTION_BODY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'HEADINGS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'FUNCTION_BODY_RETURNING_COLON_DELIMITED_LIST' );
        --
        /* Add back when we have fixed header support for classic reports / tabular forms
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FIXED_HEADER',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'PAGE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FIXED_HEADER_MAX_HEIGHT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'FIXED_HEADER',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REGION' );
        */
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'CSV_EXPORT_ENABLED',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CSV_SEPARATOR',
            p_db_column                   => 'PLUG_QUERY_EXP_SEPARATOR',
            p_is_required                 => false,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CSV_ENCLOSED_BY',
            p_db_column                   => 'PLUG_QUERY_EXP_ENCLOSED_BY',
            p_is_required                 => false,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DOWNLOAD_LINK_TEXT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Download',
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DOWNLOAD_FILENAME',
            p_db_column                   => 'PLUG_QUERY_EXP_FILENAME',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y',
            p_help_text                   => q'~Specify a name for the CSV download file. If no name is specified, the region name is used followed by the extension .csv.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SUPPLEMENTAL_TEXT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => null );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ENABLE_PRINTING',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'EXTERNAL_PROCESSING_ENABLED',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'EXTERNAL_PROCESSING_URL',
            p_db_column                   => null,
            p_view_column                 => 'URL',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'EXTERNAL_PROCESSING_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'EXTERNAL_PROCESSING_LINK_TEXT',
            p_db_column                   => null,
            p_view_column                 => 'LINK_LABEL',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'EXTERNAL_PROCESSING_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
    end store_report_attributes;
    --
    procedure store_report_columns (
        p_type_property_name in varchar2 )
    is
    begin
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'COLUMN_NAME',
            p_db_column         => 'COLUMN_ALIAS',
            p_is_required       => true,
            p_is_common         => true,
            p_is_read_only      => true,
            p_check_uniqueness  => 'PARENT',
            p_comment_text      => q'~$$$ the old edit page maps CHECK$, LINK$ and DERIVED$ to system messages. See wwv_flow_edit_report.get_alias_display_name~' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'QUERY_COLUMN_ID',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DERIVED_COLUMN',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => p_type_property_name,
            p_db_column         => null,
            p_view_column       => 'DISPLAY_AS',
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'PLAIN' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_HEADING',
            p_db_column                   => 'COLUMN_HEADING',
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HEADING_ALIGNMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'CENTER',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DISPLAY_SEQUENCE',
            p_db_column         => 'COLUMN_DISPLAY_SEQUENCE',
            p_is_required       => true,
            p_is_common         => true,
            p_comment_text      => q'~$$$ make this read only for now~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_ALIGNMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'LEFT',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'USE_AS_ROW_HEADER',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD,PCT_GRAPH,ROW_SELECTOR,RICH_TEXT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FORMAT_MASK',
            p_db_column                   => 'COLUMN_FORMAT',
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,DISPLAY_AND_SAVE,TEXT,DATE_PICKER',
            p_help_text                   => q'~<p>Enter the format mask to apply to this column.
    You can type in the format mask or pick from the predefined list, based on a sample representation of how that format mask is displayed.</p>
    <p>It is important that number format masks are only applied to columns that contain numbers and date format masks are only applied to columns that contain dates.
    Otherwise, a runtime error is raised when any record contains a value that can not be converted using the specified format mask.</p>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATE_PICKER_FORMAT_MASK',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DATE_POPUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'RICH_TEXT_FORMAT',
            p_db_column                   => 'ATTRIBUTE_01',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'MARKDOWN',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'RICH_TEXT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PCT_GRAPH_BACKGROUD_COLOR',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PCT_GRAPH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PCT_GRAPH_FOREGROUND_COLOR',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PCT_GRAPH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PCT_GRAPH_BAR_WIDTH',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PCT_GRAPH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_TABLE_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_comment_text                => q'~$$$ Note: A owner column is not yet supported by the format mask~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_TABLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE_OWNER' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_CONTENT_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_PK_COLUMN1',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_PK_COLUMN2',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_MIME_TYPE_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_FILENAME_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_LAST_UPDATED_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_CHARSET_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN_LOV,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY',
            p_comment_text                => q'~$$$ for POPUP and POPUPKEY we should NOT show STATIC~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NAMED_LOV',
            p_db_column                   => 'NAMED_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SHARED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_SQL',
            p_db_column                   => 'INLINE_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_STATIC_VALUES',
            p_db_column                   => 'INLINE_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'STATIC:Display1;Return1,Display2;Return2',
            p_depending_on_comp_prop_name => 'LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => 'LOV_',
            p_return_type         => 'SQL',
            p_db_column_func_body => 'INLINE_LOV',
            p_type_property       => 'LOV_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SIMPLE_CHECKBOX_VALUES',
            p_db_column                   => 'INLINE_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SIMPLE_CHECKBOX' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISPLAY_EXTRA_VALUES',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'SELECT_LIST,RADIOGROUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISPLAY_NULL_VALUE',
            p_db_column                   => 'LOV_SHOW_NULLS',
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NULL_DISPLAY_VALUE',
            p_db_column                   => 'LOV_NULL_TEXT',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DISPLAY_NULL_VALUE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NULL_RETURN_VALUE',
            p_db_column                   => 'LOV_NULL_VALUE',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DISPLAY_NULL_VALUE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_TARGET',
            p_db_column                   => 'COLUMN_LINK',
            p_is_required                 => false,
            p_is_common                   => true,
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'LINK,PCT_GRAPH',
            p_comment_text                => q'~$$$ IMAGE raises the runtime error "Numeric value error" if a value is set~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_TEXT',
            p_db_column                   => 'COLUMN_LINKTEXT',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '##CURRENT_COLUMN_NAME##',
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => 'LINK_TARGET',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_ATTRIBUTES',
            p_db_column                   => 'COLUMN_LINK_ATTR',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => 'LINK_TARGET',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DOWNLOAD_TEXT',
            p_db_column                   => 'BLOB_DOWNLOAD_TEXT',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CONTENT_DISPOSITION',
            p_db_column                   => 'BLOB_CONTENT_DISPOSITION',
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'attachment',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DOWNLOAD' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ELEMENT_WIDTH',
            p_db_column                   => 'COLUMN_WIDTH',
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,POPUP,POPUPKEY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ELEMENT_HEIGHT',
            p_db_column                   => 'COLUMN_HEIGHT',
            p_is_required                 => false,
            p_is_common                   => true,
            p_default_value               => '3',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TEXTAREA' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'RADIO_COLUMNS',
            p_db_column                   => 'ATTRIBUTE_01',
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => '1',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'RADIOGROUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HTML_EXPRESSION',
            p_db_column                   => 'COLUMN_HTML_EXPRESSION',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLAIN' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FORMATTING_CSS_CLASSES',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,PCT_GRAPH',
            p_comment_text                => q'~$$$ IMAGE or DOWNLOAD do raise the runtime error "Numeric value error" if a value is set~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CSS_STYLE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,PCT_GRAPH',
            p_comment_text                => q'~$$$ IMAGE or DOWNLOAD do raise the runtime error "Numeric value error" if a value is set~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HIGHLIGHT_WORDS',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLAIN' );
        --
        if p_type_property_name = 'TAB_FORM_COLUMN_TYPE' then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'COLUMN_DEFAULT_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_type_property_name,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'DISPLAY_AND_SAVE,DATE_POPUP,DATE_PICKER,TEXT,TEXTAREA,SELECT_LIST,RADIOGROUP,HIDDEN_FIELD,POPUP,POPUPKEY,SIMPLE_CHECKBOX,ROW_SELECTOR',
                p_comment_text                => '$$$ Property should only be visible if the current column name NOT matches with the primary key column(s) in the tabular form DML process' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DEFAULT_VALUE_ITEM',
                p_db_column                   => 'COLUMN_DEFAULT',
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'COLUMN_DEFAULT_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'ITEM' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DEFAULT_VALUE_PLSQL_EXPRESSION_VARCHAR2',
                p_db_column                   => 'COLUMN_DEFAULT',
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'COLUMN_DEFAULT_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'FUNCTION' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PK_SOURCE_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_type_property_name,
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'ROW_SELECTOR',
                p_comment_text                => '$$$ Property should only be visible if the current column name matches with the primary key column(s) in the tabular form DML process' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PK_SOURCE_SEQUENCE_NAME',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'PK_SOURCE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'S' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PK_SOURCE_PLSQL_EXPRESSION',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'PK_SOURCE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'F' );
            --
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_SORT_SEQUENCE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'ROW_SELECTOR' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_SORT_DIRECTION',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_null_text                   => 'Ascending',
            p_depending_on_comp_prop_name => 'COLUMN_SORT_SEQUENCE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISABLE_SORT_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,ROW_SELECTOR' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'INCLUDE_IN_EXPORT_PRINT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,ROW_SELECTOR',
            p_comment_text                => q'~$$$ should we hide it for LINK as well? Is Hidden supported?~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PRINT_COLUMN_WIDTH',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'INCLUDE_IN_EXPORT_PRINT',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ADVANCED_CSS_CLASSES',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,HIDDEN_FIELD,HIDDEN_PROTECTED,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY,SIMPLE_CHECKBOX' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CUSTOM_ATTRIBUTES',
            p_db_column                   => 'CATTRIBUTES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,HIDDEN_FIELD,HIDDEN_PROTECTED,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY,SIMPLE_CHECKBOX' );
        --
    /*
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ELEMENT_OPTION_HTML_ATTRIBUTES', -- $$$ is this property really used by reports3.plb, can't find it in the rendering code
            p_db_column                   => 'CATTRIBUTES_ELEMENT',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name, -- $$$ would it be better to restricted to specific display types?
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'HIDDEN_COLUMN' );
    */
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_WIDTH',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COMPUTE_SUM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD,ROW_SELECTOR,DATE_POPUP,DATE_PICKER,RICH_TEXT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FIELD_TEMPLATE',
            p_db_column                   => 'COLUMN_FIELD_TEMPLATE',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_on_delete         => 'WEAK_NULL',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DISPLAY_AND_SAVE,TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,HIDDEN_FIELD,HIDDEN_PROTECTED,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY,SIMPLE_CHECKBOX',
            p_help_text                   => q'~Choose the field template to be used when rendering the tabular form field. When no template is selected, the field is rendered using the default accessibility compliant label.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'UI_DEFAULT_REF_TABLE_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DERIVED_COLUMN',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'N' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'UI_DEFAULT_REF_TABLE',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => false,
            p_parent_comp_prop_name => 'UI_DEFAULT_REF_TABLE_OWNER' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'UI_DEFAULT_REF_COLUMN',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => false,
            p_parent_comp_prop_name => 'UI_DEFAULT_REF_TABLE' );
        --
        store_condition_type_comp_prop (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => null,
            p_help_text_type    => null,
            p_db_column_type    => 'DISPLAY_WHEN_COND_TYPE',
            p_view_column_type  => 'CONDITION_TYPE',
            p_db_column_expr1   => 'DISPLAY_WHEN_CONDITION',
            p_db_column_expr2   => 'DISPLAY_WHEN_CONDITION2' );
        --
        store_component_prop (
            p_component_type_id   => l_component_type_id,
            p_property_name       => 'AUTHORIZATION_SCHEME',
            p_db_column           => 'REPORT_COLUMN_REQUIRED_ROLE',
            p_is_required         => false,
            p_is_common           => false,
            p_reference_on_delete => 'WEAK_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ESCAPE_SPECIAL_CHARACTERS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,HIDDEN_COLUMN,PLAIN_LOV' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_build_option  => true,
            p_has_comments      => true,
            p_comment_db_column => 'COLUMN_COMMENT' );
    end store_report_columns;
    --
    procedure store_comp_props_web_src_param
    is
    begin
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'NAME',
            p_db_column         => null,
            p_check_uniqueness  => 'PARENT',
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => '#PLEASE_CHANGE#' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'WEB_SRC_PARAM_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'HEADER' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PARAM_DIRECTION',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'IS_STATIC',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'PARAM_DIRECTION',
            p_depending_on_condition_type  => 'NOT_EQUALS',
            p_depending_on_expression      => 'OUT' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'PARAMETER_VALUE',
            p_db_column                    => null,
            p_is_required                  => false, -- $$$ should be required depending on IS_STATIC = Y
            p_is_common                    => true );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'PARAMETER_REQUIRED',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'IS_STATIC',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'N' );
        --
        if l_component_type_id = c_comp_type_web_src_mod_param then
            -- $$$ todo only one parameter can have USE_FOR_ROW_SEARCH = Y
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'USE_FOR_ROW_SEARCH',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'PARAM_DIRECTION',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'OUT' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'OMIT_WHEN_NULL',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'WEB_SRC_PARAM_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'HEADER,COOKIE,QUERY_STRING' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'MULTIPLE_VALUES',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'WEB_SRC_PARAM_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'QUERY_STRING',
            p_depending_on_comp_prop_name2 => 'OMIT_WHEN_NULL',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'Y' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'MULTI_VALUE_SEPARATOR',
            p_db_column                    => 'ARRAY_SEPARATOR_CHAR',
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'MULTIPLE_VALUES',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'Y',
            p_help_text                    => q'~
    Enter the separator character to use for splitting up the value. If empty, values will be separated by
    the colon character (":").~' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_comment_db_column => 'PARAM_COMMENT' );
    end store_comp_props_web_src_param;
    --
    procedure store_comp_props_ws_param_comp
    is
    begin
        store_component_prop (
            p_component_type_id   => l_component_type_id,
            p_property_name       => 'WEB_SRC_PARAM',
            p_db_column           => null,
            p_is_required         => true,
            p_is_common           => true,
            p_is_read_only        => true,
            p_reference_on_delete => 'DB_CASCADE' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'VALUE_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_STATIC_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
    /*
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY_COLON',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY_COLON' );

        --
    */
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_ITEM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_PREFERENCE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PREFERENCE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_COLLECTION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'COLLECTION' );
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id );
        --
    end store_comp_props_ws_param_comp;
    --
    procedure store_comp_props_task_param
    is
    begin
        store_component_prop (
            p_component_type_id   => l_component_type_id,
            p_property_name       => 'TASK_DEF_PARAM',
            p_db_column           => null,
            p_is_required         => true,
            p_is_common           => true,
            p_is_read_only        => true,
            p_reference_on_delete => 'DB_CASCADE' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'TASK_VALUE_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_STATIC_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_ITEM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'TASK_VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'TASK_VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_PREFERENCE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PREFERENCE' );
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id );
        --
    end store_comp_props_task_param;
    --
    procedure store_property_series_source (
        p_prefix               in varchar2,
        p_property_id_type     in number,
        p_property_id_sql      in number,
        p_property_id_plsql    in number,
        p_sql_min_column_count in number,
        p_sql_max_column_count in number,
        p_examples_sql         in varchar2,
        p_examples_plsql       in varchar2 )
    is
    begin
        l_property_id := p_property_id_type;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_prefix || '_SERIES_SOURCE_TYPE',
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'SERIES_QUERY_TYPE',
            p_display_group_name => 'SOURCE',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the query source type for the chart series.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'SQL Query',
            p_return_value  => 'SQL_QUERY' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body returning SQL Query',
            p_return_value  => 'FUNCTION_RETURNING_SQL_QUERY' );
        --
        l_property_id := p_property_id_sql;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_prefix || '_SOURCE_QUERY',
            p_prompt               => 'SQL Query',
            p_property_type        => 'SQL',
            p_db_column            => 'SERIES_QUERY',
            p_display_length       => 80,
            p_max_length           => 32767,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_examples             => p_examples_sql,
            p_display_group_name   => 'SOURCE',
            p_help_text            => q'~Enter the SQL Query for the chart series.~',
            p_comment_text         => '$$$ how can we check the data types of the SQL query?' );
        --
        l_property_id := p_property_id_plsql;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_prefix || '_SOURCE_FUNC_RETURNING_SQL',
            p_prompt               => 'PL/SQL Function Body returning SQL Query',
            p_property_type        => 'PLSQL FUNCTION BODY SQL',
            p_db_column            => 'SERIES_QUERY',
            p_display_length       => 80,
            p_max_length           => 32767,
            p_display_group_name   => 'SOURCE',
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_examples             => p_examples_plsql,
            p_help_text            => q'~Enter a PL/SQL function body that returns an SQL Query for the chart series.~',
            p_comment_text         => '$$$ how can we check the data types of the SQL query?' );
    end store_property_series_source;

    --
    procedure store_chart_font_properties (
        p_font_group             in varchar2,
        p_prompt_prefix          in varchar2 default null,
        p_display_group_name     in varchar2,
        p_property_id_font       in number,
        p_property_id_font_size  in number,
        p_property_id_font_color in number )
    is
        c_prompt_prefix varchar2( 30 ) := p_prompt_prefix || case when p_prompt_prefix is not null then ' ' end;
    begin
        --
        l_property_id := p_property_id_font;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FACE',
            p_prompt             => c_prompt_prefix || 'Font Face',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_FACE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select a font face.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arial',
            p_return_value  => 'Arial' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Helvetica',
            p_return_value  => 'Helvetica' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Geneva',
            p_return_value  => 'Geneva' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Sans-serif',
            p_return_value  => 'sans-serif' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Verdana',
            p_return_value  => 'Verdana' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Tahoma',
            p_return_value  => 'Tahoma' );
        --
        l_property_id := p_property_id_font_size;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_SIZE',
            p_prompt             => c_prompt_prefix || 'Font Size',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_SIZE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font size.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '2',
            p_return_value  => '2' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '4',
            p_return_value  => '4' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '6',
            p_return_value  => '6' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '8',
            p_return_value  => '8' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '10',
            p_return_value  => '10' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '12',
            p_return_value  => '12' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '14',
            p_return_value  => '14' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '16',
            p_return_value  => '16' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '18',
            p_return_value  => '18' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '20',
            p_return_value  => '20' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '22',
            p_return_value  => '22' );
        --
        l_property_id := p_property_id_font_color;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_COLOR',
            p_prompt             => c_prompt_prefix || 'Font Color',
            p_property_type      => 'COLOR',
            p_db_column          => p_font_group || '_COLOR',
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~<p>Enter the color used to display the font for the specified attribute.</p>~' );

    end store_chart_font_properties;
    --
    procedure store_chart_font_comp_props (
        p_font_group                  in varchar2,
        p_default_font_size           in number   default 10,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FACE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Tahoma',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_SIZE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => p_default_font_size,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_COLOR',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => '#000000',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
    end store_chart_font_comp_props;
    --
    procedure store_jet_font_properties (
        p_font_group              in varchar2,
        p_display_group_name      in varchar2,
        p_property_id_font_family in number,
        p_property_id_font_style  in number,
        p_property_id_font_size   in number,
        p_property_id_font_color  in number )
    is
    begin
        --
        l_property_id := p_property_id_font_family;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_FAMILY',
            p_prompt             => 'Font Family',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_FAMILY',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font family.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arial',
            p_return_value  => 'Arial');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arial Black',
            p_return_value  => 'Arial Black');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bookman',
            p_return_value  => 'Bookman');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Comic Sans MS',
            p_return_value  => 'Comic Sans MS');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Courier',
            p_return_value  => 'Courier');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Courier New',
            p_return_value  => 'Courier New');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Garamond',
            p_return_value  => 'Garamond');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Georgia',
            p_return_value  => 'Georgia');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Helvetica',
            p_return_value  => 'Helvetica');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Impact',
            p_return_value  => 'Impact');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Palatino',
            p_return_value  => 'Palatino');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Times',
            p_return_value  => 'Times');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Times New Roman',
            p_return_value  => 'Times New Roman');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Trebuchet MS',
            p_return_value  => 'Trebuchet MS');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Verdana',
            p_return_value  => 'Verdana');
        --
        l_property_id := p_property_id_font_style;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_STYLE',
            p_prompt             => 'Font Style',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_STYLE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font style.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Normal',
            p_return_value  => 'normal');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Italic',
            p_return_value  => 'italic');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Oblique',
            p_return_value  => 'oblique');
        --
        l_property_id := p_property_id_font_size;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_SIZE',
            p_prompt             => 'Font Size',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_SIZE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font size.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '8',
            p_return_value  => '8' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '10',
            p_return_value  => '10' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '12',
            p_return_value  => '12' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '14',
            p_return_value  => '14' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '16',
            p_return_value  => '16' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '18',
            p_return_value  => '18' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '20',
            p_return_value  => '20' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '22',
            p_return_value  => '22' );
        --
        l_property_id := p_property_id_font_color;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_COLOR',
            p_prompt             => 'Font Color',
            p_property_type      => 'COLOR',
            p_db_column          => p_font_group || '_COLOR',
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~<p>Enter the color used to display the font for the specified attribute.</p>~' );

    end store_jet_font_properties;
    --
    procedure store_jet_font_comp_props (
        p_font_group                   in varchar2,
        p_depending_on_comp_prop_name  in varchar2 default null,
        p_depending_on_condition_type  in varchar2 default null,
        p_depending_on_expression      in varchar2 default null,
        p_depending_on_comp_prop_name2 in varchar2 default null,
        p_depending_on_condition_type2 in varchar2 default null,
        p_depending_on_expression2     in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'JET_'|| p_font_group || '_FAMILY',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => p_depending_on_comp_prop_name,
            p_depending_on_condition_type  => p_depending_on_condition_type,
            p_depending_on_expression      => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2);
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'JET_'|| p_font_group || '_STYLE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2 );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'JET_'|| p_font_group || '_SIZE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2 );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'JET_'|| p_font_group || '_COLOR',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2 );
        --
    end store_jet_font_comp_props;
    --
    --
    procedure store_cal_weekday_property (
        p_property_id        in number,
        p_name               in varchar2,
        p_prompt             in varchar2,
        p_db_column          in varchar2,
        p_display_group_name in varchar2,
        p_help_text          in varchar2 )
    is
    begin
        l_property_id := p_property_id;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => p_prompt,
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column,
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => p_help_text );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Sunday',
            p_return_value  => '19721105' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Monday',
            p_return_value  => '19721106' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Tuesday',
            p_return_value  => '19721107' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Wednesday',
            p_return_value  => '19721108' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Thursday',
            p_return_value  => '19721109' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Friday',
            p_return_value  => '19721110' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Saturday',
            p_return_value  => '19721111' );
        --
    end store_cal_weekday_property;
    --
    --
    procedure store_print_font_properties (
        p_font_group                in varchar2,
        p_display_group_name        in varchar2,
        p_db_column_prefix          in varchar2,
        p_property_id_font          in number,
        p_property_id_font_weight   in number,
        p_property_id_font_size     in number,
        p_property_id_font_color    in number,
        p_property_id_font_bg_color in number default null )
    is
    begin
        --
        l_property_id := p_property_id_font;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_FAMILY',
            p_prompt             => 'Font',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column_prefix || '_FONT_FAMILY',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select a font face for this component.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Helvetica',
            p_return_value  => 'Helvetica' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Times',
            p_return_value  => 'Times' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Courier',
            p_return_value  => 'Courier' );
        --
        l_property_id := p_property_id_font_weight;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_WEIGHT',
            p_prompt             => 'Font Weight',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column_prefix || '_FONT_WEIGHT',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select a font weight for this component.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Normal',
            p_return_value  => 'normal' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bold',
            p_return_value  => 'bold' );
        --
        l_property_id := p_property_id_font_size;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_SIZE',
            p_prompt             => 'Font Size',
            p_property_type      => 'INTEGER',
            p_db_column          => p_db_column_prefix || '_FONT_SIZE',
            p_display_length     => 4,
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~Enter a font size for this component.~' );
        --
        l_property_id := p_property_id_font_color;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_COLOR',
            p_prompt             => 'Font Color',
            p_property_type      => 'COLOR',
            p_db_column          => p_db_column_prefix || '_FONT_COLOR',
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~<p>Enter the color used to display this component.</p>~' );
        --
        if p_property_id_font_bg_color is not null then
            l_property_id := p_property_id_font_bg_color;
            store_property (
                p_id                 => l_property_id,
                p_name               => p_font_group || '_BG_COLOR',
                p_prompt             => 'Background Color',
                p_property_type      => 'COLOR',
                p_db_column          => p_db_column_prefix || '_BG_COLOR',
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~<p>Enter the color used to display the background for this component.</p>~' );
        end if;
        --
    end store_print_font_properties;
    --
    procedure store_print_font_comp_props (
        p_font_group                  in varchar2,
        p_default_size                in varchar2,
        p_default_weight              in varchar2 default 'normal',
        p_has_bg_color                in boolean  default false,
        p_default_bg_color            in varchar2 default null,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_FAMILY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Helvetica',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_WEIGHT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => p_default_weight,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_SIZE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => p_default_size,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_COLOR',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => '#000000',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        if p_has_bg_color then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_font_group || '_BG_COLOR',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => p_default_bg_color,
                p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
                p_depending_on_condition_type => p_depending_on_condition_type,
                p_depending_on_expression     => p_depending_on_expression );
        end if;
        --
    end store_print_font_comp_props;
    --
    --
    procedure store_print_comp_props (
        p_component_type_id in number )
    is
    begin
        --
        if p_component_type_id in ( c_comp_type_classic_rpt_print, c_comp_type_tab_form_print ) then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_LINK_TEXT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false,
                p_default_value     => '#PRINT_LINK_TEXT#' );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_OUTPUT_FORMAT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => true,
                p_default_value     => 'PDF',
                p_null_text         => 'Derive from Item');
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PRINT_OUTPUT_FORMAT_ITEM',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'PRINT_OUTPUT_FORMAT',
                p_depending_on_condition_type => 'NULL' );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'REPORT_LAYOUT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false,
                p_null_text         => 'Default Report Layout' );
        end if;
        --
        if p_component_type_id != c_comp_type_ig_print then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PRINT_VIEW_FILE_AS',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'ATTACHMENT' );
        end if;
        --
        if p_component_type_id in ( c_comp_type_classic_rpt_print, c_comp_type_tab_form_print ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PRINT_FILE_NAME',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'LINK_EXAMPLE',
                p_db_column         => 'PRINT_EXAMPLE_URL',
                p_is_required       => false,
                p_is_common         => false,
                p_help_text         => q'~<p>Report regions can be printed using the standard print link, shown below the report region, or using a button. If you want to use a button, remove the link label and create a button on your page, using the URL shown in Print URL as the button target.</p>
<p>Using buttons to print report regions enables you to set session state. This is useful if you are planning to support printing to several different output formats (Word, Excel, PDF, and so on) and want to display a button, link, or image for each option. For this implementation, use the output format derived from item option and include the item name and value in your Print URL.</p>~' );
        end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_SIZE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'LETTER',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_ORIENTATION',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'HORIZONTAL' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_UNITS',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'INCHES',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_WIDTH',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => '11',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_HEIGHT',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => '8.5',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_BORDER_WIDTH',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => '.5' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_BORDER_COLOR',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => '#666666' );
        --
        store_print_font_comp_props (
            p_font_group   => 'PRINT_PAGE_HEADER',
            p_default_size => 12 );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_HEADER_ALIGNMENT',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'CENTER' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_HEADER_TEXT',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true );
        --
        store_print_font_comp_props (
            p_font_group       => 'PRINT_COLUMN_HEADING',
            p_default_size     => 10,
            p_default_weight   => 'bold',
            p_has_bg_color     => true,
            p_default_bg_color => '#EEEEEE' );
        --
        store_print_font_comp_props (
            p_font_group       => 'PRINT_COLUMN',
            p_default_size     => 10,
            p_has_bg_color     => true,
            p_default_bg_color => '#FFFFFF' );
        --
        if p_component_type_id in ( c_comp_type_classic_rpt_print, c_comp_type_tab_form_print ) then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_COLUMN_WIDTH_UNITS',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => false,
                p_default_value     => 'PERCENTAGE' );
        end if;
        --
        store_print_font_comp_props (
            p_font_group   => 'PRINT_PAGE_FOOTER',
            p_default_size => 12 );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_FOOTER_ALIGNMENT',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'CENTER' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_FOOTER_TEXT',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false );
        --
    end store_print_comp_props;
    --
    --
    procedure store_lov_type_component_props (
        p_name                in varchar2,
        p_db_column_named_lov in varchar2 default null,
        p_db_column_source    in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => p_name,
            p_db_column             => null,
            p_is_required           => null, -- defined by plug-in standard attributes
            p_is_common             => true,
            p_plugin_std_attributes => 'LOV' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NAMED_LOV',
            p_db_column                   => p_db_column_named_lov,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SHARED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_SQL',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY',
            p_help_text                   => q'~<p>Enter the SQL query definition to populate this list of values. Generally list of value queries are of the form:</p>
<pre>
select [displayValue],
       [returnValue]
  from ...
 where ...
 order by ...
</pre>
<p>Each column selected must have a unique name or alias. Oracle recommends using an alias on any column that includes an SQL expression.</p>
<p>Note: When defining a Popup LOV item type, if you would like to display multiple columns in the popup, you must instead define your List of Values in Shared Components,
with the required additional metadata. Inline list of values can only be used to display single columns for Popup LOVs.</p>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_STATIC_VALUES',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'STATIC:Display1;Return1,Display2;Return2',
            p_depending_on_comp_prop_name => p_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => 'LOV_',
            p_return_type         => 'SQL',
            p_db_column_func_body => p_db_column_source,
            p_type_property       => p_name );
        --
    end store_lov_type_component_props;
    --
    --
    procedure store_comp_prop_fc_attributes
    is
        c_is_facet   constant boolean := ( l_component_type_id in ( c_comp_type_facet,   c_comp_type_facet_group_item ));
        c_is_sfilter constant boolean := ( l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group_item ));
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_COMPUTE_COUNTS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_plugin_std_attributes       => case when c_is_facet or c_is_sfilter then 'FC_HAS_FEEDBACK' end );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_SHOW_COUNTS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => 'FC_COMPUTE_COUNTS',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_ZERO_COUNT_ENTRIES',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'H',
            p_depending_on_comp_prop_name => 'FC_COMPUTE_COUNTS',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'FC_SORT_BY_TOP_COUNTS',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => false,
            p_default_value                => 'Y',
            p_depending_on_comp_prop_name  => 'FC_SHOW_COUNTS',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'Y',
            p_depending_on_comp_prop_name2 => case when c_is_facet or c_is_sfilter then 'ITEM_TYPE'                         end,
            p_depending_on_condition_type2 => case when c_is_facet or c_is_sfilter then 'IN_LIST'                           end,
            p_depending_on_expression2     => case when c_is_facet or c_is_sfilter then 'NATIVE_CHECKBOX,NATIVE_RADIOGROUP' end );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_SHOW_SELECTED_FIRST',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_plugin_std_attributes       => case when c_is_facet or c_is_sfilter then 'FC_SHOW_SELECTED_FIRST' end,
            p_depending_on_comp_prop_name => case when c_is_facet or c_is_sfilter then 'FACET_LOV_TYPE'         end,
            p_depending_on_condition_type => case when c_is_facet or c_is_sfilter then 'NOT_NULL'               end );
        --
        if c_is_facet or l_component_type_id = c_comp_type_facet_group then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_SHOW_MORE_COUNT',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => '7',
                p_plugin_std_attributes       => case when c_is_facet then 'FC_SHOW_MORE_COUNT' end,
                p_depending_on_comp_prop_name => case when c_is_facet then 'FACET_LOV_TYPE'     end,
                p_depending_on_condition_type => case when c_is_facet then 'NOT_NULL'           end );
        end if;
        --
        if l_component_type_id = c_comp_type_sfilter then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_FILTER_VALUES',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_plugin_std_attributes       => 'FC_FILTER_VALUES',
                p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
                p_depending_on_condition_type => 'NOT_NULL' );
            --        
        elsif l_component_type_id = c_comp_type_facet then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_FILTER_VALUES_INITIALLY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_plugin_std_attributes       => 'FC_FILTER_VALUES',
                p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
                p_depending_on_condition_type => 'NOT_NULL' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'FC_ACTIONS_FILTER',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'Y',
                p_plugin_std_attributes        => 'FC_FILTER_VALUES',
                p_depending_on_comp_prop_name  => 'FACET_LOV_TYPE',
                p_depending_on_condition_type  => 'NOT_NULL' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'FC_ACTIONS_CHART',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'Y',
                p_depending_on_comp_prop_name  => 'FC_COMPUTE_COUNTS',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'Y',
                p_depending_on_comp_prop_name2 => 'FACET_LOV_TYPE',
                p_depending_on_condition_type2 => 'NOT_NULL' );
        --
        elsif l_component_type_id = c_comp_type_facet_group then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'FC_ACTIONS_CHART',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'Y',
                p_depending_on_comp_prop_name  => 'FC_COMPUTE_COUNTS',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'Y' );
        end if;
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_facet_group ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_INITIAL_CHART',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'FC_ACTIONS_CHART',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
        end if;
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_facet_group, c_comp_type_facet_group_item ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_TOGGLEABLE',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => case when c_is_facet then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_facet then 'NOT_EQUALS'    end,
                p_depending_on_expression     => case when c_is_facet then 'NATIVE_SEARCH' end );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_INITIAL_TOGGLED',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'FC_TOGGLEABLE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_COLLAPSIBLE',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => case when c_is_facet then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_facet then 'NOT_EQUALS'    end,
                p_depending_on_expression     => case when c_is_facet then 'NATIVE_SEARCH' end );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_INITIAL_COLLAPSED',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'FC_COLLAPSIBLE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
        end if;
        --
        if l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group ) then
            -- Suggestion Chips
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SUGGESTIONS_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_default_value               => 'DYNAMIC',
                p_null_text                   => 'None',
                p_depending_on_comp_prop_name => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_sfilter then 'NOT_EQUALS'    end,
                p_depending_on_expression     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
            --
            /* $$$ Phase 2
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SEARCH_SUGGESTIONS_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true );
                p_depending_on_comp_prop_name => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_sfilter then 'EQUALS'        end,
                p_depending_on_expression     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
            --
            -- For the depending properties we would use
            --
            -- p_depending_on_comp_prop_name  => 'SUGGESTIONS_TYPE',
            -- p_depending_on_has_to_exist    => 'N',
            -- ...
            -- p_depending_on_comp_prop_name2 => 'SEARCH_SUGGESTIONS_TYPE',
            -- p_depending_on_has_to_exist2   => 'N',
            -- ...
            */
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SUGGESTIONS_STATIC_VALUES',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'SUGGESTIONS_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'STATIC' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SUGGESTIONS_SQL_QUERY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'SUGGESTIONS_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SQL_QUERY' );
            --
            /* $$$ phase 2
            store_func_body_comp_props (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => 'SUGGESTIONS_',
                p_return_type       => 'ARRAY',
                p_type_property     => 'SUGGESTIONS_TYPE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MAXIMUM_SUGGESTIONS',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => '1',
                p_depending_on_comp_prop_name => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_sfilter then 'NOT_EQUALS'    end,
                p_depending_on_expression     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
            --
            if l_component_type_id = c_comp_type_sfilter then
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'MAXIMUM_SEARCH_SUGGESTIONS',
                    p_db_column                   => null,
                    p_is_required                 => true,
                    p_is_common                   => false,
                    p_default_value               => '5',
                    p_depending_on_comp_prop_name => 'ITEM_TYPE',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => 'NATIVE_SEARCH' );
            end if;
            */
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SHOW_SUGGESTIONS_LABEL',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => case l_component_type_id
                                                    when c_comp_type_sfilter       then 'Y'
                                                    when c_comp_type_sfilter_group then 'N'
                                                  end,
                p_depending_on_comp_prop_name  => 'SUGGESTIONS_TYPE',
                p_depending_on_condition_type  => 'NOT_NULL',
                p_depending_on_comp_prop_name2 => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type2 => case when c_is_sfilter then 'NOT_EQUALS'    end,
                p_depending_on_expression2     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
        end if;
        --
    end store_comp_prop_fc_attributes;
    --
    --
    function replace_facet_sfil_help_text (
        p_help_text varchar2 )
        return varchar2
    is
    begin
        return replace (
                   p_help_text,
                   '%component%',
                   case
                     when l_component_type_id in ( c_comp_type_facet,   c_comp_type_facet_group,   c_comp_type_facet_group_item )   then 'facet'
                     when l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group, c_comp_type_sfilter_group_item ) then 'filter'
                   end );
    end replace_facet_sfil_help_text;
    --
    --
    procedure store_comp_prop_facet_sfilter (
        p_default_item_type          in varchar2,
        p_depending_on_property_name in varchar2 )
    is
        c_cascading_lov_parent_prop constant varchar2(30) := case
                                                               when l_component_type_id in ( c_comp_type_facet,   c_comp_type_facet_group_item )   then 'CASCADING_LOV_PARENT_FACET'
                                                               when l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group_item ) then 'CASCADING_LOV_PARENT_FILTER'
                                                             end;
    begin
        if l_component_type_id in ( c_comp_type_facet_group_item, c_comp_type_sfilter_group_item ) then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'HIDDEN_PARENT_REGION',
                p_is_required       => false,
                p_is_common         => false );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ITEM_NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'APPLICATION',
            p_supports_substitution => false,
            p_default_value         => 'P#PAGE_ID#_NEW' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ITEM_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => p_default_item_type );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'LABEL',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => true,
            p_default_value         => '#PLEASE_CHANGE#',
            p_plugin_std_attributes => 'VISIBLE',
            p_help_text             => replace_facet_sfil_help_text( q'~<p>Enter the label for the %component%.
    The label displays on the page only if the %component% displays. The label for type <strong>Search</strong> is visually hidden, but available to assistive technology.</p>~' ));
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_facet_group_item ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SHOW_CURRENT_FACET_LABEL',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_default_value               => 'Y',
                p_plugin_std_attributes       => 'VISIBLE',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
        end if;
        --
        store_plugin_attr_seq_offset (
            p_component_type_id => l_component_type_id );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DISPLAY_SEQUENCE',
            p_db_column         => 'ITEM_SEQUENCE',
            p_is_required       => true,
            p_is_common         => true );
        --
        if l_component_type_id = c_comp_type_facet_group_item then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'FACET_GROUP',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true );
        elsif l_component_type_id = c_comp_type_sfilter_group_item then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'SFILTER_GROUP',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true );
        end if;
        --
        if l_component_type_id = c_comp_type_facet then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_MAX_HEIGHT',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_IN_LIST',
                p_depending_on_expression     => 'NATIVE_SEARCH,NATIVE_INPUT' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ELEMENT_VALUE_PLACEHOLDER',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => true,
            p_plugin_std_attributes => 'PLACEHOLDER' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ICON_CSS_CLASSES',
            p_db_column                   => 'ITEM_ICON_CSS_CLASSES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH',
            p_help_text                   => replace_facet_sfil_help_text (
q'~<p>Enter the classes for the icon you want to use to identify this %component%. You may add multiple classes by separating them with spaces.</p>
            <p>If your theme uses Font APEX then review the <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a> to view available icons and modifiers.</p>~' ));
        --
        store_lov_type_component_props (
            p_name => 'FACET_LOV_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'INCLUDE_NULL_OPTION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_plugin_std_attributes       => 'FC_LOV_DISPLAY_NULL',
            p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NULL_DISPLAY_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'INCLUDE_NULL_OPTION',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y',
            p_help_text                   => q'~Enter the text to be displayed within the list NULL option at the top of this list.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_cascading_lov_parent_prop,
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_on_delete         => 'WEAK_NULL',
            p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SHARED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CASCADING_LOV_PARENT_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
       --     p_plugin_std_attributes       => 'CASCADING_LOV',
            p_depending_on_comp_prop_name => c_cascading_lov_parent_prop,
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CASCADING_LOV_PARENT_REQUIRED',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => c_cascading_lov_parent_prop,
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_sfilter ) then
            store_comp_prop_fc_attributes;
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'ADVANCED_CSS_CLASSES',
                p_db_column                   => 'ITEM_CSS_CLASSES',
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH',
                p_help_text                   => replace_facet_sfil_help_text( q'~Enter classes to add to this %component%. You may add multiple classes by separating them with spaces.~' ));
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_DB_COLUMN',
            p_db_column                   => 'SOURCE',
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH',
            p_help_text                   => replace_facet_sfil_help_text( q'~<p>Enter the case sensitive database column name used as the source for this %component%.</p>~' ));
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_sfilter ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'TEXT_SEARCH_DB_COLUMNS',
                p_db_column                   => 'SOURCE',
                p_is_required                 => false,
                p_is_common                   => true,
                p_supports_substitution       => false,
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_TYPE',
            p_db_column                   => 'SOURCE_DATA_TYPE',
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_default_value               => 'VARCHAR2',
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH' );
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_sfilter ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MULTI_VALUE_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_null_text                   => 'No',
                p_null_help_text              => 'Column does not contain multiple values.',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_CHECKBOX,NATIVE_RADIOGROUP,NATIVE_SELECT_LIST' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MULTI_VALUE_SEPARATOR',
                p_db_column                   => null,
                p_default_value               => ':',
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'MULTI_VALUE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SEPARATED' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MULTI_VALUE_TRIM_SPACE',
                p_db_column                   => null,
                p_default_value               => 'N',
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'MULTI_VALUE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SEPARATED' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_MULTI_VALUE_COMBINATION',
                p_is_required                 => true,
                p_default_value               => 'OR',
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'MULTI_VALUE_TYPE',
                p_depending_on_condition_type => 'NOT_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'DEFAULT_VALUE_TYPE',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => true,
            p_plugin_std_attributes => 'SOURCE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_STATIC_TEXT',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC_TEXT_WITH_SUBSTITUTIONS' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_ITEM',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_SQL_QUERY',
            p_db_column                   => 'ITEM_DEFAULT',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_SQL_QUERY_COLON',
            p_db_column                   => 'ITEM_DEFAULT',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY_COLON' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'DEFAULT_VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'DEFAULT_VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'DEFAULT_VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'DEFAULT_VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_SEQUENCE',
            p_db_column                   => 'ITEM_DEFAULT',
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SEQUENCE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_depending_on_property_name,
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_on_delete         => 'DB_NULL',
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH' );
        --
        store_component_prop (
            p_component_type_id             => l_component_type_id,
            p_property_name                 => 'DEPENDING_ON_CONDITION',
            p_db_column                     => null,
            p_is_required                   => true,
            p_is_common                     => false,
            p_default_value                 => 'NOT_NULL',
            p_depending_on_comp_prop_name   => p_depending_on_property_name,
            p_depending_on_condition_type   => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_COND_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEPENDING_ON_CONDITION',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'EQUALS,NOT_EQUALS' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_COND_LIST',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEPENDING_ON_CONDITION',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IN_LIST,NOT_IN_LIST' );
        --
        if l_component_type_id = c_comp_type_sfilter then
            store_condition_type_comp_prop (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => null,
                p_help_text_type              => null,
                p_db_column_type              => 'DISPLAY_WHEN_TYPE',
                p_view_column_type            => 'CONDITION_TYPE',
                p_db_column_expr1             => 'DISPLAY_WHEN',
                p_db_column_expr2             => 'DISPLAY_WHEN2',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'AUTHORIZATION_SCHEME',
                p_db_column                   => 'SECURITY_SCHEME',
                p_is_required                 => false,
                p_is_common                   => false,
                p_reference_on_delete         => 'WEAK_NULL',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
        else
            store_condition_type_comp_prop (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => null,
                p_help_text_type    => null,
                p_db_column_type    => 'DISPLAY_WHEN_TYPE',
                p_view_column_type  => 'CONDITION_TYPE',
                p_db_column_expr1   => 'DISPLAY_WHEN',
                p_db_column_expr2   => 'DISPLAY_WHEN2' );
            --
            store_component_prop (
                p_component_type_id   => l_component_type_id,
                p_property_name       => 'AUTHORIZATION_SCHEME',
                p_db_column           => 'SECURITY_SCHEME',
                p_is_required         => false,
                p_is_common           => false,
                p_reference_on_delete => 'WEAK_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ENCRYPT_SESSION_STATE',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => false,
            p_default_value         => 'Y',
            p_plugin_std_attributes => 'ENCRYPT' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ESCAPE_SPECIAL_CHARACTERS',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => false,
            p_default_value         => 'Y',
            p_plugin_std_attributes => 'ESCAPE_OUTPUT' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id     => l_component_type_id,
            p_has_build_option      => true,
            p_has_comments          => true,
            p_comment_db_column     => 'ITEM_COMMENT' );
        --
    end store_comp_prop_facet_sfilter;
    --
    --
    procedure store_comp_prop_facet_sfil_grp
    is
    begin
        store_component_prop (
            p_component_type_id      => l_component_type_id,
            p_property_name          => 'LABEL',
            p_db_column              => 'LABEL',
            p_is_required            => true,
            p_is_common              => true,
            p_check_uniqueness       => 'PARENT',
            p_supports_substitution  => true,
            p_default_value          => '#PLEASE_CHANGE#',
            p_help_text              => replace_facet_sfil_help_text( '~Enter the label of the %component% group.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DISPLAY_SEQUENCE',
            p_db_column         => 'DISPLAY_SEQUENCE',
            p_is_required       => true,
            p_is_common         => true );
        --
        if l_component_type_id = c_comp_type_facet_group then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'FC_MAX_HEIGHT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false );
        end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ICON_CSS_CLASSES',
            p_db_column         => 'ICON_CSS_CLASSES',
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => replace_facet_sfil_help_text(
q'~<p>Enter the classes for the icon you want to use to identify this %component%. You may add multiple classes by separating them with spaces.</p>
            <p>If your theme uses Font APEX then review the <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a> to view available icons and modifiers.</p>~' ));
        --
        store_comp_prop_fc_attributes;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ADVANCED_CSS_CLASSES',
            p_db_column         => 'CSS_CLASSES',
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => replace_facet_sfil_help_text( q'~Enter classes to add to this %component% group. You may add multiple classes by separating them with spaces.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'FC_FILTER_COMBINATION',
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'OR' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id      => l_component_type_id,
            p_has_condition            => true,
            p_cond_type_db_column      => 'CONDITION_TYPE',
            p_cond_exp1_db_column      => 'CONDITION_EXPR1',
            p_cond_exp2_db_column      => 'CONDITION_EXPR2',
            p_has_authorization_scheme => true,
            p_authorization_db_column  => 'AUTHORIZATION_SCHEME',
            p_has_build_option         => true,
            p_build_option_db_column   => 'BUILD_OPTION_ID',
            p_has_comments             => true,
            p_comment_db_column        => 'GROUP_COMMENT' );
        --
    end store_comp_prop_facet_sfil_grp;
    --
    --
    procedure store_comp_prop_js_condition (
        p_db_column_type         in varchar2,
        p_db_column_element_type in varchar2,
        p_db_column_element      in varchar2,
        p_db_column_expression   in varchar2,
        p_help_text_type         in varchar2 default null )
    is
        procedure store_comp_props (
            p_is_ig in boolean )
        is
            c_postfix      constant varchar2(30) := case when p_is_ig then '_WHEN_IG' end;
            c_js_condition constant varchar2(30) := 'JS_CONDITION' || c_postfix;
        begin
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => c_js_condition,
                p_db_column                   => p_db_column_type,
                p_is_required                 => false,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'IG_REGION',
                p_depending_on_condition_type => case when p_is_ig then 'NOT_NULL' else 'NULL' end,
                p_help_text                   => p_help_text_type );
            --
            if p_is_ig then
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_ELEMENT_TYPE',
                    p_db_column                   => p_db_column_element_type,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_default_value               => 'COLUMN',
                    p_depending_on_comp_prop_name => 'JS_CONDITION_WHEN_IG',
                    p_depending_on_condition_type => 'IN_LIST',
                    p_depending_on_expression     => 'EQUALS,NOT_EQUALS,GREATER_THAN,GREATER_THAN_OR_EQUAL,LESS_THAN,LESS_THAN_OR_EQUAL,NULL,NOT_NULL,IN_LIST,NOT_IN_LIST' );
                --
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_ITEM_WHEN_IG',
                    p_db_column                   => p_db_column_element,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_depending_on_comp_prop_name => 'JS_CONDITION_ELEMENT_TYPE',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => 'ITEM' );
                --
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_COLUMN_WHEN_IG',
                    p_db_column                   => p_db_column_element,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_parent_comp_prop_name       => 'IG_REGION',
                    p_reference_on_delete         => 'WEAK_CASCADE',
                    p_depending_on_comp_prop_name => 'JS_CONDITION_ELEMENT_TYPE',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => 'COLUMN' );
            else
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_ITEM' || c_postfix,
                    p_db_column                   => p_db_column_element,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_depending_on_comp_prop_name => c_js_condition,
                    p_depending_on_condition_type => 'IN_LIST',
                    p_depending_on_expression     => 'EQUALS,NOT_EQUALS,GREATER_THAN,GREATER_THAN_OR_EQUAL,LESS_THAN,LESS_THAN_OR_EQUAL,NULL,NOT_NULL,IN_LIST,NOT_IN_LIST',
                    p_help_text                   => q'~<p>Enter the page item used in this condition. You can type in the name or pick from the list of available items.</p>~' );
            end if;
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'JS_COND_VALUE' || c_postfix,
                p_db_column                   => p_db_column_expression,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => c_js_condition,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'EQUALS,NOT_EQUALS,GREATER_THAN,GREATER_THAN_OR_EQUAL,LESS_THAN,LESS_THAN_OR_EQUAL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'JS_COND_LIST' || c_postfix,
                p_db_column                   => p_db_column_expression,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => c_js_condition,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'IN_LIST,NOT_IN_LIST' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'JS_COND_JS_EXPRESSION' || c_postfix,
                p_db_column                   => p_db_column_expression,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => c_js_condition,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'JAVASCRIPT_EXPRESSION' );
        end store_comp_props;
    begin
        -- for page items
        store_comp_props (
            p_is_ig => false );
        -- for IG columns
        store_comp_props (
            p_is_ig => true );
    end store_comp_prop_js_condition;
    --
    --
    procedure store_comp_prop_remote_server (
        p_server_type in varchar2 )
    is
        c_server_type_desc constant varchar2( 255 ) := case p_server_type
                                                       when 'REMOTE_SQL'     then 'REST Enabled SQL Service'
                                                       when 'AUTHENTICATION' then 'Authentication Server'
                                                       when 'PRINT_SERVER'   then 'Print Server'
                                                       when 'WEB_SOURCE'     then 'REST Data Source Server'
                                                       end;
        function get_help (
            p_help_text in varchar2 )
            return varchar2
        is
        begin
            return replace( p_help_text, '%type%', c_server_type_desc );
        end get_help;
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'WORKSPACE',
            p_supports_substitution => false,
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => get_help( q'~Enter a descriptive name for this %type%.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'REMOTE_SERVER_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => p_server_type );
        --
        if p_server_type = 'PRINT_SERVER' then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_SERVER_TYPE',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'BIP' );
            end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ENDPOINT_URL',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_help_text         => case p_server_type
                                   when 'REMOTE_SQL' then q'~Enter the endpoint URL for this REST Enabled SQL Service. Include the ORDS context root and schema URL prefix, e.g. <em>https://{host}:{port}/ords/{schema}</em>. Do not append /_/sql; Application Express will take care of that.~' 
                                   else get_help( q'~Enter the endpoint URL for this %type%.~' )
                                   end
            );
        --
        if p_server_type = 'REMOTE_SQL' then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'CREDENTIALS',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'DB_SESSION_INIT',
                p_db_column         => 'ORDS_INIT_CODE',
                p_is_required       => false,
                p_is_common         => false,
                p_help_text         => q'~Enter code to be executed immediately after connecting to the REST Enabled SQL Service and before the component SQL is being executed.~' );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'DB_SESSION_CLEANUP',
                p_db_column         => 'ORDS_CLEANUP_CODE',
                p_is_required       => false,
                p_is_common         => false,
                p_help_text         => q'~Enter code to be executed immediately after the component SQL is being executed.~' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'STATIC_ID',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'WORKSPACE',
            p_supports_substitution => false,
            p_help_text             => get_help( q'~Use the Static ID to reference the %type% in API Calls. Static IDs are also used to identify an existing %type% when the application is being exported and imported to another workspace.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PROMPT_ON_INSTALL',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N',
            p_help_text         => get_help( q'~Choose whether prompts for this %type% should be displayed when the application is being imported on another Application Express instance.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'HTTPS_HOST_NAME',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => q'~The host name to be matched against the common name (CN) of the remote server's certificate for an HTTPS request. It can also be a domain name like *.example.com. If NULL, the host name in the given URL will be used.~' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'SERVER_TIMEOUT',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => '300' );
        --
        if p_server_type = 'REMOTE_SQL' then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'SERVER_TIME_ZONE',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'REMOTE_SQL_DATABASE_TYPE',
                p_db_column         => null,
                p_is_required       => false,
                p_is_read_only      => true,
                p_is_common         => false );
        end if;
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_comments      => true,
            p_comment_db_column => 'SERVER_COMMENT' );
        --
    end store_comp_prop_remote_server;
    --
    --
    procedure store_comp_prop_static_file
    is
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'FILE_NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => case l_component_type_id
                                         when c_comp_type_workspace_file   then 'WORKSPACE'
                                         when c_comp_type_application_file then 'APPLICATION'
                                       end,
            p_supports_substitution => false,
            p_help_text             => case l_component_type_id
                                         when c_comp_type_workspace_file   then null -- use property level help
                                         when c_comp_type_application_file then
                                             q'~This is the name of the file in the application file repository. Use <strong><code>&#35;APP_IMAGES#filename</code></strong> in your application to reference a file.~'
                                       end
            );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'MIME_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'FILE_CHARACTER_SET',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true ); -- $$$ depending on upper(:P315_MIME_TYPE) like 'TEXT/%' or upper(:P315_MIME_TYPE) like '%JAVASCRIPT%'
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id );
    end store_comp_prop_static_file;
    --
    --
    procedure store_comp_prop_location (
        p_location_property_name      in varchar2,
        p_query_type_property_name    in varchar2,
        p_db_column_query_type        in varchar2,
        p_db_column_source_lang       in varchar2,
        p_db_column_source            in varchar2,
        p_has_items_to_submit         in boolean,
        p_db_column_items_to_submit   in varchar2,
        p_has_include_rowid           in boolean,
        p_has_order_by                in boolean,
        p_has_external_order_by       in boolean default null,
        p_has_order_by_item           in boolean,
        p_has_web_src_has_local_table in boolean,
        p_has_remote_caching          in boolean,
        p_examples_sql_query          in varchar2,
        p_help_text_items_to_submit   in varchar2,
        p_is_region                   in boolean  default false,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
        c_is_automation constant boolean := ( l_component_type_id = c_comp_type_automation );
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_location_property_name,
            p_db_column                   => null,
            p_is_required                 => case when p_is_region then null else true end, -- defined by plug-in standard attributes
            p_is_common                   => false,
            p_default_value               => 'LOCAL',
            p_plugin_std_attributes       => case when p_is_region then 'SOURCE_LOCATION' end,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        if c_is_automation then
            -- Initiated = Function Body Returning Boolean
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PLSQL_PROCESS_LOCATION',
                p_db_column                   => 'LOCATION',
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'LOCAL',
                p_depending_on_comp_prop_name => 'ACTIONS_INITIATED_ON',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'FUNC_BODY_RETURNING_BOOLEAN' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_REMOTE_DATABASE', -- todo $$$ should be SOURCE_REMOTE_ORACLE_DATABASE, but only for pl/sql. Has to be checked again.
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_depending_on_comp_prop_name  => p_location_property_name,
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_has_to_exist    => false,
                p_depending_on_expression      => 'REMOTE',
                p_depending_on_comp_prop_name2 => 'PLSQL_PROCESS_LOCATION',
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_has_to_exist2   => false,
                p_depending_on_expression2     => 'REMOTE' );
            --
            store_func_body_comp_props (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => 'AUTOM_',
                p_return_type                 => 'BOOLEAN',
                p_db_column_lang              => 'FUNCTION_BODY_LANGUAGE',
                p_db_column_func_body         => 'QUERY_SOURCE',
                p_type_property               => 'ACTIONS_INITIATED_ON',
                p_type_value                  => 'FUNC_BODY_RETURNING_BOOLEAN' );
            --
        else
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_REMOTE_DATABASE',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'REMOTE' );
        end if;
        --
        if p_has_include_rowid then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_REMOTE_DATABASE_TYPE',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => p_location_property_name,
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'REMOTE',
                p_depending_on_comp_prop_name2 => 'SOURCE_REMOTE_DATABASE',
                p_depending_on_condition_type2 => 'NOT_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => p_query_type_property_name,
            p_db_column                    => p_db_column_query_type,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'TABLE',
            p_depending_on_comp_prop_name  => p_location_property_name,
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'LOCAL,REMOTE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_QUERY_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_QUERY_TABLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TABLE',
            p_parent_comp_prop_name       => 'SOURCE_QUERY_OWNER' );
        --
        if p_has_include_rowid then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'INCLUDE_ROWID_COLUMN',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => p_query_type_property_name,
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'TABLE',
                p_depending_on_comp_prop_name2 => 'SOURCE_REMOTE_DATABASE_TYPE',
                p_depending_on_condition_type2 => 'IN_LIST',
                p_depending_on_expression2     => 'ORACLE,$NULL$',
                p_depending_on_has_to_exist2   => false );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_QUERY_WHERE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TABLE',
            p_parent_comp_prop_name       => 'SOURCE_QUERY_TABLE' );
        --
        if p_has_order_by then
            if p_is_region then
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'SOURCE_QUERY_STATIC_ORDER_BY',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => p_query_type_property_name,
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'TABLE',
                    p_depending_on_comp_prop_name2 => 'REGION_TYPE',
                    p_depending_on_condition_type2 => 'IN_LIST',
                    p_depending_on_expression2     => 'NATIVE_IG,NATIVE_IR,NATIVE_MAP_REGION',
                    p_parent_comp_prop_name        => 'SOURCE_QUERY_TABLE' );
            elsif not p_has_order_by_item then
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'SOURCE_QUERY_STATIC_ORDER_BY',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => p_query_type_property_name,
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'TABLE' );
            end if;
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_SQL_QUERY',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL',
            p_examples                    => p_examples_sql_query );
        --
        if p_query_type_property_name = 'LOV_QUERY_TYPE' then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'LOV_LEGACY_SQL_QUERY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => p_query_type_property_name,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'LEGACY_SQL' );
        end if;
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => 'SOURCE_QUERY_',
            p_return_type         => 'SQL',
            p_db_column_lang      => p_db_column_source_lang,
            p_db_column_func_body => p_db_column_source,
            p_type_property       => p_query_type_property_name,
            p_type_value          => 'FUNC_BODY_RETURNING_SQL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_WEB_SRC_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_location_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'WEB_SOURCE' );
        --
        if p_is_region then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_LIST',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'REGION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_LIST' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_BREADCRUMB',
                p_db_column                   => 'MENU_ID',
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'REGION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_BREADCRUMB' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_FILTERED_REGION',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_reference_on_delete         => 'WEAK_CASCADE',
                p_depending_on_comp_prop_name => 'REGION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_FACETED_SEARCH,NATIVE_SMART_FILTERS' );
        end if;
        --
        if p_has_web_src_has_local_table then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'WEB_SRC_HAS_LOCAL_TABLE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'SOURCE_WEB_SRC_QUERY',
                p_depending_on_condition_type => 'NOT_NULL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'WEB_SRC_USE_LOCAL_TABLE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'WEB_SRC_HAS_LOCAL_TABLE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
        end if;
        --
        if p_is_region then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_PAGE_ITEMS_TO_SUBMIT',
                p_db_column                   => p_db_column_items_to_submit,
                p_is_required                 => false,
                p_is_common                   => true,
        /* $$$ has to be investigated, doesn't work in wwv_flow_property_dev.clear_property_values the reason might be that SOURCE_LOCATION has a higher property id and
               that's why the order in which the properties are processed causes the issue

                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_has_to_exist   => false,
                p_depending_on_condition_type => 'NOT_NULL',
        */
                p_plugin_std_attributes       => case when p_is_region then 'AJAX_ITEMS_TO_SUBMIT' end,
                p_help_text                   => p_help_text_items_to_submit );
        elsif p_has_items_to_submit then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_PAGE_ITEMS_TO_SUBMIT',
                p_db_column                   => p_db_column_items_to_submit,
                p_is_required                 => false,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'REGION_SOURCE',
                p_help_text                   => p_help_text_items_to_submit );
        end if;
        --
        if p_has_order_by_item then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_null_text                   => 'None',
                p_plugin_std_attributes       => 'ORDER_BY',
                p_depending_on_comp_prop_name => 'SOURCE_QUERY_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'TABLE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_QUERY_ORDER_BY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'STATIC',
                p_parent_comp_prop_name       => 'SOURCE_QUERY_TABLE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_QUERY_ORDER_BY_ITEM',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'ITEM',
                p_parent_comp_prop_name       => 'SOURCE_QUERY_TABLE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_QUERY_ORDER_BY_ITEM2',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_plugin_std_attributes       => 'ORDER_BY',
                p_depending_on_comp_prop_name => 'SOURCE_QUERY_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'SQL,FUNC_BODY_RETURNING_SQL',
                p_parent_comp_prop_name       => 'SOURCE_QUERY_TABLE' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'OPTIMIZER_HINT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'TABLE' );
        --
        if p_has_web_src_has_local_table then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'USE_EXTERNAL_EXPR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => 'SOURCE_WEB_SRC_QUERY',
                p_depending_on_condition_type  => 'NOT_NULL',
                --
                p_depending_on_comp_prop_name2 => 'WEB_SRC_USE_LOCAL_TABLE',
                p_depending_on_has_to_exist2   => false,
                p_depending_on_condition_type2 => 'IN_LIST',
                p_depending_on_expression2     => 'N,$NULL$' );
        else
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'USE_EXTERNAL_EXPR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => 'SOURCE_WEB_SRC_QUERY',
                p_depending_on_condition_type  => 'NOT_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'EXTERNAL_FILTER_EXPR',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_default_value                => null,
            p_depending_on_comp_prop_name  => 'USE_EXTERNAL_EXPR',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'Y' );
        --
        if p_has_order_by_item then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'EXTERNAL_ORDER_BY_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_null_text                   => 'None',
                p_plugin_std_attributes       => 'ORDER_BY',
                p_depending_on_comp_prop_name => 'USE_EXTERNAL_EXPR',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXTERNAL_ORDER_BY_EXPR',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => null,
                p_depending_on_comp_prop_name  => 'EXTERNAL_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'STATIC' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXTERNAL_ORDER_BY_ITEM',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => null,
                p_depending_on_comp_prop_name  => 'EXTERNAL_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'ITEM' );
            --
            if p_is_region then
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'EXTERNAL_STATIC_ORDER_BY_EXPR',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => 'USE_EXTERNAL_EXPR',
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'Y',
                    p_depending_on_comp_prop_name2 => 'REGION_TYPE',
                    p_depending_on_condition_type2 => 'IN_LIST',
                    p_depending_on_expression2     => 'NATIVE_IG,NATIVE_IR,NATIVE_MAP_REGION' );
            end if;
        elsif p_has_order_by or p_has_external_order_by then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXTERNAL_STATIC_ORDER_BY_EXPR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => 'USE_EXTERNAL_EXPR',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'Y' );
        end if;
        --
        if p_is_region then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'HAS_GENERIC_COLUMNS',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'DERIVED_REPORT_COLUMNS',
                p_depending_on_comp_prop_name  => 'REGION_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'NATIVE_SQL_REPORT',
                p_depending_on_comp_prop_name2 => p_query_type_property_name,
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_expression2     => 'FUNC_BODY_RETURNING_SQL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'GENERIC_COLUMN_COUNT',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'HAS_GENERIC_COLUMNS',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'GENERIC_REPORT_COLUMNS' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_POST_PROCESSING',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => null,
            p_depending_on_comp_prop_name => 'SOURCE_WEB_SRC_QUERY',
            p_depending_on_condition_type => 'NOT_NULL',
            p_parent_comp_prop_name       => 'SOURCE_WEB_SRC_QUERY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'POST_PROC_WHERE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'SOURCE_POST_PROCESSING',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'WHERE_ORDER_BY_CLAUSE',
            p_parent_comp_prop_name       => 'SOURCE_WEB_SRC_QUERY' );
            --
        if p_has_order_by and not p_has_order_by_item then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'POST_PROC_ORDER_BY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'SOURCE_POST_PROCESSING',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'WHERE_ORDER_BY_CLAUSE',
                p_parent_comp_prop_name       => 'SOURCE_WEB_SRC_QUERY' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'POST_PROC_SQL_QUERY',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'SOURCE_POST_PROCESSING',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL',
            p_parent_comp_prop_name       => 'SOURCE_WEB_SRC_QUERY' );
        --
        store_func_body_comp_props (
            p_component_type_id     => l_component_type_id,
            p_name_prefix           => 'POST_PROC_',
            p_db_column_lang        => p_db_column_source_lang,
            p_db_column_func_body   => p_db_column_source,
            p_return_type           => 'SQL',
            p_type_property         => 'SOURCE_POST_PROCESSING',
            p_type_value            => 'FUNC_BODY_RETURNING_SQL',
            p_parent_comp_prop_name => 'SOURCE_WEB_SRC_QUERY' );
        --
        if p_has_order_by_item then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'POST_PROC_ORDER_BY_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_null_text                   => 'None',
                p_depending_on_comp_prop_name => 'SOURCE_POST_PROCESSING',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'WHERE_ORDER_BY_CLAUSE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'POST_PROC_ORDER_BY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'POST_PROC_ORDER_BY_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'STATIC',
                p_parent_comp_prop_name       => 'SOURCE_WEB_SRC_QUERY' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'POST_PROC_ORDER_BY_ITEM',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'POST_PROC_ORDER_BY_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'ITEM',
                p_parent_comp_prop_name       => 'SOURCE_WEB_SRC_QUERY' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'POST_PROC_ORDER_BY_ITEM2',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'SOURCE_POST_PROCESSING',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'SQL,FUNC_BODY_RETURNING_SQL',
                p_parent_comp_prop_name       => 'POST_PROC_SQL_QUERY' );
        end if;
        --
        if p_has_remote_caching then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_CACHING',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_null_text                   => 'Disabled',
                p_null_help_text              => q'~Result is not cached. Data is fetched from the remote database for each request.~',
                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'REMOTE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_INVALIDATE_WHEN',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'REMOTE_CACHING',
                p_depending_on_condition_type => 'NOT_NULL' );
        end if;
        --
    end store_comp_prop_location;
    --
    --
    procedure store_order_by_item_property (
        p_name               in varchar2,
        p_prompt             in varchar2,
        p_display_group_name in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => p_prompt,
            p_property_type      => 'ORDER BY ITEM',
            p_db_column          => 'QUERY_ORDER_BY',
            p_display_group_name => p_display_group_name,
            p_examples           => q'~
<dl><dt>To show the newest hires first and those which have not been hired yet last. If the date is equal, sort them by department number.</dt>
<dd><pre>
HIREDATE DESC NULLS LAST, DEPTNO
</pre></dd>
</dl>~',
            p_help_text          => q'~<p>Enter multiple Order By Clauses that can be picked by end-users. Each can contain one or more comma-separated column names, including directives, to define the order in which records are retrieved.</p>
            <p>Directives include:
              <ul>
                <li>ASC - Return column in ascending order</li>
                <li>DESC - Return column in descending order</li>
                <li>NULLS FIRST - Return column with no value before those with values</li>
                <li>NULLS LAST - Return column with values before those with no value</li>
              </ul>
            </p>~' );
    end store_order_by_item_property;
    --
    --
    procedure store_i18n_lang_prop_lov_value
    is
    begin
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Afrikaans (af)',
            p_return_value  => 'af' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Albanian',
            p_return_value  => 'sq' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Algeria) (ar-dz)',
            p_return_value  => 'ar-dz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Bahrain) (ar-bh)',
            p_return_value  => 'ar-bh' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Egypt) (ar-eg)',
            p_return_value  => 'ar-eg' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Iraq) (ar-iq)',
            p_return_value  => 'ar-iq' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Jordan) (ar-jo)',
            p_return_value  => 'ar-jo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Kuwait) (ar-kw)',
            p_return_value  => 'ar-kw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Lebanon) (ar-lb)',
            p_return_value  => 'ar-lb' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Libya) (ar-ly)',
            p_return_value  => 'ar-ly' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Morocco) (ar-ma)',
            p_return_value  => 'ar-ma' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Oman) (ar-om)',
            p_return_value  => 'ar-om' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Qatar) (ar-qa)',
            p_return_value  => 'ar-qa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Saudi Arabia) (ar-sa)',
            p_return_value  => 'ar-sa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Syria) (ar-sy)',
            p_return_value  => 'ar-sy' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Tunisia) (ar-tn)',
            p_return_value  => 'ar-tn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (U.A.E.) (ar-ae)',
            p_return_value  => 'ar-ae' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Yemen) (ar-ye)',
            p_return_value  => 'ar-ye' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (ar)',
            p_return_value  => 'ar' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Armenian',
            p_return_value  => 'hy' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Assamese (as)',
            p_return_value  => 'as' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Azerbaijani (az)',
            p_return_value  => 'az' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Basque (eu)',
            p_return_value  => 'eu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Belarusian (be)',
            p_return_value  => 'be' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bengali (bn)',
            p_return_value  => 'bn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bulgarian (bg)',
            p_return_value  => 'bg' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Catalan (ca)',
            p_return_value  => 'ca' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (China) (zh-cn)',
            p_return_value  => 'zh-cn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Hong Kong SAR) (zh-hk)',
            p_return_value  => 'zh-hk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Macau SAR) (zh-mo)',
            p_return_value  => 'zh-mo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Singapore) (zh-sg)',
            p_return_value  => 'zh-sg' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Taiwan) (zh-tw)',
            p_return_value  => 'zh-tw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (zh)',
            p_return_value  => 'zh' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Croatian (hr)',
            p_return_value  => 'hr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Czech (cs)',
            p_return_value  => 'cs' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Danish (da)',
            p_return_value  => 'da' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Dutch (Belgium) (nl-be)',
            p_return_value  => 'nl-be' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Dutch (Netherlands) (nl)',
            p_return_value  => 'nl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Australia) (en-au)',
            p_return_value  => 'en-au' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Belize) (en-bz)',
            p_return_value  => 'en-bz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Canada) (en-ca)',
            p_return_value  => 'en-ca' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Ireland) (en-ie)',
            p_return_value  => 'en-ie' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Jamaica) (en-jm)',
            p_return_value  => 'en-jm' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (New Zealand) (en-nz)',
            p_return_value  => 'en-nz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Philippines) (en-ph)',
            p_return_value  => 'en-ph' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (South Africa) (en-za)',
            p_return_value  => 'en-za' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Trinidad) (en-tt)',
            p_return_value  => 'en-tt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (United Kingdom) (en-gb)',
            p_return_value  => 'en-gb' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (United States) (en-us)',
            p_return_value  => 'en-us' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Zimbabwe) (en-zw)',
            p_return_value  => 'en-zw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (en)',
            p_return_value  => 'en' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Estonian (et)',
            p_return_value  => 'et' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'FYRO Macedonian (mk)',
            p_return_value  => 'mk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Faeroese (fo)',
            p_return_value  => 'fo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Farsi (fa)',
            p_return_value  => 'fa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Finnish (fi)',
            p_return_value  => 'fi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Belgium) (fr-be)',
            p_return_value  => 'fr-be' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Canada) (fr-ca)',
            p_return_value  => 'fr-ca' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (France) (fr)',
            p_return_value  => 'fr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Luxembourg) (fr-lu)',
            p_return_value  => 'fr-lu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Monaco) (fr-mc)',
            p_return_value  => 'fr-mc' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Switzerland) (fr-ch)',
            p_return_value  => 'fr-ch' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Gaelic (gd)',
            p_return_value  => 'gd' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Galician (gl)',
            p_return_value  => 'gl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Austria) (de-at)',
            p_return_value  => 'de-at' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Germany) (de)',
            p_return_value  => 'de' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Liechtenstein) (de-li)',
            p_return_value  => 'de-li' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Luxemgourg) (de-lu)',
            p_return_value  => 'de-lu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Switzerland) (de-ch)',
            p_return_value  => 'de-ch' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Greek (el)',
            p_return_value  => 'el' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Gujarati (gu)',
            p_return_value  => 'gu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Hebrew (he)',
            p_return_value  => 'he' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Hindi (hi)',
            p_return_value  => 'hi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Hungarian (hu)',
            p_return_value  => 'hu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Icelandic (is)',
            p_return_value  => 'is' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Indonesian (id)',
            p_return_value  => 'id' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Irish (ga)',
            p_return_value  => 'ga' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Italian (Italy) (it)',
            p_return_value  => 'it' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Italian (Switzerland) (it-ch)',
            p_return_value  => 'it-ch' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Japanese (ja)',
            p_return_value  => 'ja' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Kannada (kn)',
            p_return_value  => 'kn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Kazakh (kk)',
            p_return_value  => 'kk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Konkani (kok)',
            p_return_value  => 'kok' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Korean (ko)',
            p_return_value  => 'ko' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Kyrgyz (kz)',
            p_return_value  => 'kz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Latvian (lv)',
            p_return_value  => 'lv' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Lithuanian (lt)',
            p_return_value  => 'lt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Malay (Malaysia) (ms)',
            p_return_value  => 'ms' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Malayalam (ml)',
            p_return_value  => 'ml' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Maltese (mt)',
            p_return_value  => 'mt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Marathi (mr)',
            p_return_value  => 'mr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Nepali (India) (ne)',
            p_return_value  => 'ne' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Norwegian (Bokmal) (nb-no)',
            p_return_value  => 'nb-no' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Norwegian (Bokmal) (no)',
            p_return_value  => 'no' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Norwegian (Nynorsk) (nn-no)',
            p_return_value  => 'nn-no' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Oriya (or)',
            p_return_value  => 'or' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Polish (pl)',
            p_return_value  => 'pl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Portuguese (Brazil) (pt-br)',
            p_return_value  => 'pt-br' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Portuguese (Portugal) (pt)',
            p_return_value  => 'pt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Punjabi (pa)',
            p_return_value  => 'pa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Romanian (ro)',
            p_return_value  => 'ro' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Russian (Moldova) (ru-md)',
            p_return_value  => 'ru-md' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Russian (ru)',
            p_return_value  => 'ru' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Serbian (sr)',
            p_return_value  => 'sr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Slovak (sk)',
            p_return_value  => 'sk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Slovenian (sl)',
            p_return_value  => 'sl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Argentina) (es-ar)',
            p_return_value  => 'es-ar' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Bolivia) (es-bo)',
            p_return_value  => 'es-bo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Chile) (es-cl)',
            p_return_value  => 'es-cl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Colombia) (es-co)',
            p_return_value  => 'es-co' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Costa Rica) (es-cr)',
            p_return_value  => 'es-cr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Dominican Republic) (es-do)',
            p_return_value  => 'es-do' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Ecuador) (es-ec)',
            p_return_value  => 'es-ec' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (El Salvador) (es-sv)',
            p_return_value  => 'es-sv' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Guatemala) (es-gt)',
            p_return_value  => 'es-gt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Honduras) (es-hn)',
            p_return_value  => 'es-hn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Mexico) (es-mx)',
            p_return_value  => 'es-mx' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Nicaragua) (es-ni)',
            p_return_value  => 'es-ni' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Panama) (es-pa)',
            p_return_value  => 'es-pa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Paraguay) (es-py)',
            p_return_value  => 'es-py' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Peru) (es-pe)',
            p_return_value  => 'es-pe' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Puerto Rico) (es-pr)',
            p_return_value  => 'es-pr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Traditional Sort) (es)',
            p_return_value  => 'es' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (United States) (es-us)',
            p_return_value  => 'es-us' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Uruguay) (es-uy)',
            p_return_value  => 'es-uy' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Venezuela) (es-ve)',
            p_return_value  => 'es-ve' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Swahili (sw)',
            p_return_value  => 'sw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Swedish (Finland) (sv-fi)',
            p_return_value  => 'sv-fi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Swedish (sv)',
            p_return_value  => 'sv' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Tamil (ta)',
            p_return_value  => 'ta' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Telugu (te)',
            p_return_value  => 'te' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Thai (th)',
            p_return_value  => 'th' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Turkish (tr)',
            p_return_value  => 'tr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Ukrainian (uk)',
            p_return_value  => 'uk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Urdu (ur)',
            p_return_value  => 'ur' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Uzbek (uz)',
            p_return_value  => 'uz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Vietnamese (vi)',
            p_return_value  => 'vi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Welsh (cy)',
            p_return_value  => 'cy' );
        --
    end store_i18n_lang_prop_lov_value;
    --
    --
    procedure store_comp_prop_data_profile (
        p_profile_type_id in number,
        p_column_type_id  in number )
    is
    begin
        --
        -- Component Properties for XZY_DATA_PROFILE
        --
        l_component_type_id           := p_profile_type_id;
        -- Reset Component Property sequence for this component type
        g_component_property_sequence := 10;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_check_uniqueness      => 'APPLICATION',
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => q'~Enter a name for the data profile.~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'DATA_PROFILE_FORMAT',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'JSON' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_ROW_SELECTOR',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'JSON,XML' );
        --
        if l_component_type_id in ( c_comp_type_web_src_mod_dp, c_comp_type_web_src_oper_dp ) then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'DATA_PROFILE_IS_SINGLE_ROW',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
                p_depending_on_condition_type  => 'IN_LIST',
                p_depending_on_expression      => 'JSON,XML' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'DATA_PROFILE_SINGLE_ROW_SELECTOR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => true,
                p_supports_substitution        => false,
                p_depending_on_comp_prop_name  => 'DATA_PROFILE_IS_SINGLE_ROW',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'N',
                p_depending_on_comp_prop_name2 => 'DATA_PROFILE_FORMAT',
                p_depending_on_condition_type2 => 'NOT_EQUALS',
                p_depending_on_expression2     => 'CSV' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'I18N_IANA_CHARSET_ENCODING',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_null_text                    => 'Database Character Set',
            p_null_help_text               => q'~Treat the data as encoded in the <em>database character set</em>.~' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_NUMERIC_CHARACTERS',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'XML_NAMESPACES',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'XML' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_CSV_SEPARATOR',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'CSV' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_CSV_ENCLOSED_BY',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'CSV' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DEFAULT_XLSX_SHEET_NAME',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'XLSX' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_SKIP_ROWS',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'CSV,XLSX' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_HAS_HEADER_ROW',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => false,
            p_default_value                => 'Y',
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'CSV,XLSX' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_comments      => true,
            p_comment_db_column => 'PROFILE_COMMENT' );
        --
        --
        -- Component Properties for XYZ_DATA_PROFILE_COLUMN
        --
        l_component_type_id           := p_column_type_id;
        -- Reset Component Property sequence for this component type
        g_component_property_sequence := 10;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'COLUMN_NAME',
            p_db_column             => 'NAME',
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_check_uniqueness      => 'PARENT',
            p_help_text             => q'~<p>Name under which the column is exposed to &PRODUCT_NAME. components.</p>
    <p>If the data profile is part of a <em>Data Load Definition</em>, this name is used to map data profile columns to the target table. For instance, the <strong>ENAME</strong> data profile column will be loaded to the <strong>ENAME</strong> column of the target table. If the profile column does <em>not exist</em> in the target table, it will be ignored during data load.</p>~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'DATA_PROFILE_FORMAT_HIDDEN',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => false );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DATA_PROFILE_COLUMN_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'DATA' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'VARCHAR2' ); -- $$$ todo: Page 1926 has some advanced defaulting logic which has to go into pe.callbacks.js
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'MAX_LENGTH',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '4000',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'VARCHAR2' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'IS_PRIMARY_KEY',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'N',
            p_help_text         => q'~Determines whether this is a primary key column. A primary key can consist of multiple columns.~' );
        --
        -- Parsing
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_SELECTOR_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'NAME',
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'DATA',
            p_depending_on_comp_prop_name2 => 'DATA_PROFILE_FORMAT_HIDDEN',
            p_depending_on_condition_type2 => 'IN_LIST',
            p_depending_on_expression2     => 'CSV,XLSX' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_COLUMN_SELECTOR',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_SELECTOR_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'NAME,REGEXP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_COLUMN_SELECTOR_NO',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_SELECTOR_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SEQ' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_COLUMN_PATH',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_FORMAT_HIDDEN',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'JSON,XML' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PARSING_FORMAT_MASK',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'NUMBER,DATE,TIMESTAMP,TIMESTAMP WITH TIME ZONE,TIMESTAMP WITH LOCAL TIME ZONE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HAS_TIME_ZONE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DATE,TIMESTAMP,TIMESTAMP WITH TIME ZONE,TIMESTAMP WITH LOCAL TIME ZONE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_DECIMAL_CHARACTER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NUMBER' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_GROUP_CHARACTER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NUMBER' );
        --
        -- Computation
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_EXPRESSION_VARCHAR2',
            p_db_column                   => 'EXPRESSION1',
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_EXPRESSION',
            p_help_text                   => q'~Enter the SQL Expression to compute the value of this column. The SQL expression can reference data profile columns of the <strong>Data</strong> column type.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_STATEMENT',
            p_db_column                   => 'EXPRESSION1',
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY',
            p_help_text                   => q'~Enter the SQL Query to compute the value of this column. The SQL query must return a single value and can reference data profile columns of the <strong>Data</strong> column type.~' );
        --
        -- Lookup
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_OWNER',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_RETURN_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_WHERE_CLAUSE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE_COLUMN1',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_DATA_COLUMN1',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE_COLUMN2',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN1',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_DATA_COLUMN2',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN2',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE_COLUMN3',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN2',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_DATA_COLUMN3',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN3',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        -- Transformation
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_TYPE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DATA,LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_TRIM_CHARS',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'LTRIM,RTRIM,TRIM' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_FIND',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REPLACE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_REGEXP',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'REPLACE,REGEXP_NULLIF' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_REPLACE_WITH',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'REPLACE,REGEXP_REPLACE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_NULLIF',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NULLIF' );
        --
        -- Remote REST Data Source
        if l_component_type_id in ( c_comp_type_web_src_mod_dp_col, c_comp_type_web_src_oper_dpcol ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_IS_FILTERABLE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false,
                p_default_value               => 'Y' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_ATTRIBUTE_NAME',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_DATA_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false );
        end if;
        --
        -- Advanced
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ADVANCED_SEQUENCE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'IS_HIDDEN',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_comments      => true,
            p_comment_db_column => 'COLUMN_COMMENT' );
        --
    end store_comp_prop_data_profile;
    --
    --
    procedure store_comp_prop_sup_obj_script
    is
    begin
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'PARENT',
            p_supports_substitution => false,
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => q'~The name of this installation script.~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'EXECUTION_SEQUENCE',
            p_db_column             => 'SEQUENCE',
            p_is_required           => true,
            p_is_common             => true,
            p_help_text             => q'~
    The sequence number of this installation script. Scripts run in order of ascending sequence number.~' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id     => l_component_type_id,
            p_has_condition         => true,
            p_cond_type_name_prefix => 'SO_COND' );
        --
    end store_comp_prop_sup_obj_script;
    --
    --
begin
    sys.dbms_output.enable( 1000000 );

    ------------------------------------------------------------------------
    -- Delete all configuration first
    ------------------------------------------------------------------------
    delete from wwv_pe_properties;
    delete from wwv_pe_component_types;
    delete from wwv_pe_display_groups;


    ------------------------------------------------------------------------
    -- Component Types
    ------------------------------------------------------------------------

    -- Workspace component types
    store_component_type (
        p_id                        => c_comp_type_credential,
        p_name                      => 'CREDENTIAL',
        p_singular_title            => 'Web Credential',
        p_plural_title              => 'Web Credentials',
        p_table_name                => 'WWV_CREDENTIALS',
        p_display_column            => 'NAME',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1621:%session%::NO:RP,1621:P1621_ID,P1621_BACK_TO_PAGE:%pk_value%,1621',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/credentials#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_remote_database,
        p_name                      => 'REMOTE_DATABASE',
        p_singular_title            => 'REST Enabled SQL',
        p_plural_title              => 'Remote Databases',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'REMOTE_SQL' and remote_sql_database_type is not null ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1601:%session%::NO:RP,1601:P1601_ID:%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/remote_databases#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_remote_oracle_db,
        p_name                      => 'REMOTE_ORACLE_DATABASE',
        p_singular_title            => 'REST Enabled SQL',
        p_plural_title              => 'Remote Databases',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'REMOTE_SQL' and remote_sql_database_type = 'ORACLE' ~',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true,
        p_is_query_only             => true );
    --
    store_component_type (
        p_id                        => c_comp_type_auth_server,
        p_name                      => 'AUTHENTICATION_SERVER',
        p_singular_title            => 'Authentication Server',
        p_plural_title              => 'Authentication Servers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'AUTHENTICATION' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1932:%session%::NO:RP,1932:P1932_ID,P1932_BACK_TO_PAGE:%pk_value%,1932',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/authentication_servers#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_print_server,
        p_name                      => 'PRINT_SERVER',
        p_singular_title            => 'Print Server',
        p_plural_title              => 'Print Servers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'PRINT_SERVER' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1932:%session%::NO:RP,1932:P1932_ID,P1932_BACK_TO_PAGE:%pk_value%,1932',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/print_servers#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_rest_server,
        p_name                      => 'REST_SERVER',
        p_singular_title            => 'REST Data Source Server',
        p_plural_title              => 'REST Data Source Servers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'WEB_SERVICE' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1932:%session%::NO:RP,1932:P1932_ID,P1932_BACK_TO_PAGE:%pk_value%,1932',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/rest_servers#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_workspace_file,
        p_name                      => 'STATIC_WORKSPACE_FILE',
        p_singular_title            => 'Static Workspace File',
        p_plural_title              => 'Static Workspace Files',
        p_table_name                => 'WWV_FLOW_COMPANY_STATIC_FILES',
        p_display_column            => 'FILE_NAME',
        p_create_url                => 'f?p=4000:314:%session%::NO:RP,314',
        p_edit_url                  => 'f?p=4000:315:%session%::NO:RP,315:P315_ID:%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/static_files#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_app_group,
        p_name                      => 'APP_GROUP',
        p_singular_title            => 'Application Group',
        p_plural_title              => 'Application Groups',
        p_table_name                => 'WWV_FLOW_APPLICATION_GROUPS',
        p_display_column            => 'GROUP_NAME',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_file_path                 => 'workspace/app_groups#ext#' );

    -- Application related component types
    store_component_type (
        p_id                        => c_comp_type_application,
        p_name                      => 'APPLICATION',
        p_singular_title            => 'Application',
        p_plural_title              => 'Applications',
        p_table_name                => 'WWV_FLOWS',
        p_display_column            => 'NAME',
        p_file_path                 => 'application/',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_app_substitution,
        p_name                      => 'APP_SUBSTITUTION',
        p_singular_title            => 'Substitution',
        p_plural_title              => 'Substitutions',
        p_table_name                => 'WWV_FLOW_SUBSTITUTIONS_DEV',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'SUBSTITUTION_STRING' );

    -- Theme related component types
    store_component_type (
        p_id                        => c_comp_type_theme,
        p_name                      => 'THEME',
        p_singular_title            => 'Theme',
        p_plural_title              => 'Themes',
        p_table_name                => 'WWV_FLOW_THEMES',
        p_pk_column                 => 'THEME_ID',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'THEME_NAME',
        p_file_path                 => 'shared_components/theme_#id#/',
        p_is_page_designer_shared   => true );

    -- Templates
    store_component_type (
        p_id                        => c_comp_type_page_template,
        p_name                      => 'PAGE_TEMPLATE',
        p_singular_title            => 'Page Template',
        p_plural_title              => 'Page Templates',
        p_table_name                => 'WWV_FLOW_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4307:%session%::NO:4307:F4000_P4307_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:568:%session%::NO:568:FB_FLOW_ID,P568_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'page_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_region_template,
        p_name                      => 'REGION_TEMPLATE',
        p_singular_title            => 'Region Template',
        p_plural_title              => 'Region Templates',
        p_table_name                => 'WWV_FLOW_PAGE_PLUG_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'PAGE_PLUG_TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4653:%session%::NO:4653:F4000_P4653_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:538:%session%::NO:538:FB_FLOW_ID,P538_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'region_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_field_template,
        p_name                      => 'FIELD_TEMPLATE',
        p_singular_title            => 'Field Template',
        p_plural_title              => 'Field Templates',
        p_table_name                => 'WWV_FLOW_FIELD_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:85:%session%::NO:85:F4000_P85_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:574:%session%::NO:574:FB_FLOW_ID,P574_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'field_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_button_template,
        p_name                      => 'BUTTON_TEMPLATE',
        p_singular_title            => 'Button Template',
        p_plural_title              => 'Button Templates',
        p_table_name                => 'WWV_FLOW_BUTTON_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:204:%session%::NO:204:F4000_P204_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:571:%session%::NO:571:FB_FLOW_ID,P571_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'button_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_list_template,
        p_name                      => 'LIST_TEMPLATE',
        p_singular_title            => 'List Template',
        p_plural_title              => 'List Templates',
        p_table_name                => 'WWV_FLOW_LIST_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'LIST_TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4655:%session%::NO:4655:F4000_P4655_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:573:%session%::NO:573:FB_FLOW_ID,P573_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'list_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_bc_template,
        p_name                      => 'BREADCRUMB_TEMPLATE',
        p_singular_title            => 'Breadcrumb Template',
        p_plural_title              => 'Breadcrumb Templates',
        p_table_name                => 'WWV_FLOW_MENU_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:289:%session%::NO:289:F4000_P289_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:636:%session%::NO:636,596:FB_FLOW_ID,P636_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'breadcrumb_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_calendar_template,
        p_name                      => 'CALENDAR_TEMPLATE',
        p_singular_title            => 'Legacy Calendar Template',
        p_plural_title              => 'Legacy Calendar Templates',
        p_table_name                => 'WWV_FLOW_CAL_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:697:%session%::NO:697:F4000_P697_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:564:%session%::NO:564:FB_FLOW_ID,P564_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'legacy_calendar_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_report_template,
        p_name                      => 'REPORT_TEMPLATE',
        p_singular_title            => 'Classic Report Template',
        p_plural_title              => 'Classic Report Templates',
        p_table_name                => 'WWV_FLOW_ROW_TEMPLATES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'ROW_TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%::NO:162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:258:%session%::NO:258:F4000_P258_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:569:%session%::NO:569:FB_FLOW_ID,P569_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'classic_report_templates#ext#',
        p_is_page_designer_shared   => true );

    -- Shared Component related component types
    store_component_type (
        p_id                        => c_comp_type_app_item,
        p_name                      => 'APP_ITEM',
        p_singular_title            => 'Application Item',
        p_plural_title              => 'Application Items',
        p_table_name                => 'WWV_FLOW_ITEMS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:4303:%session%::NO:4303:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4303:%session%::NO:4303:F4000_P4303_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/app_items#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_app_computation,
        p_name                      => 'APP_COMPUTATION',
        p_singular_title            => 'Application Computation',
        p_plural_title              => 'Application Computations',
        p_table_name                => 'WWV_FLOW_COMPUTATIONS',
        p_view_name                 => 'APEX_APPLICATION_COMPUTATIONS',
        p_parent_component_type_id  => C_COMP_TYPE_APPLICATION,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'COMPUTATION_ITEM',
        p_edit_url                  => 'f?p=4000:4304:%session%::::F4000_P4304_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/app_computations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_app_process,
        p_name                      => 'APP_PROCESS',
        p_singular_title            => 'Application Process',
        p_plural_title              => 'Application Processes',
        p_table_name                => 'WWV_FLOW_PROCESSING',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'PROCESS_NAME',
        p_create_url                => 'f?p=4000:147:%session%::NO:147,148,150:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4309:%session%::::F4000_P4309_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/app_processes#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_build_option,
        p_name                      => 'BUILD_OPTION',
        p_singular_title            => 'Build Option',
        p_plural_title              => 'Build Options',
        p_table_name                => 'WWV_FLOW_PATCHES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'PATCH_NAME',
        p_create_url                => 'f?p=4000:4911:%session%::NO:4911:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4911:%session%::NO:4911:F4000_P4911_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/build_options#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_authentication,
        p_name                      => 'AUTHENTICATION',
        p_singular_title            => 'Authentication',
        p_plural_title              => 'Authentications',
        p_table_name                => 'WWV_FLOW_AUTHENTICATIONS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'AUTHENTICATION TYPE',
        p_plugin_column             => 'SCHEME_TYPE',
        p_create_url                => 'f?p=4000:815:%session%::NO:815,4495:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4495:%session%::::P4495_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/authentications#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_authorization,
        p_name                      => 'AUTHORIZATION',
        p_singular_title            => 'Authorization',
        p_plural_title              => 'Authorizations',
        p_table_name                => 'WWV_FLOW_SECURITY_SCHEMES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'AUTHORIZATION TYPE',
        p_plugin_column             => 'SCHEME_TYPE',
        p_create_url                => 'f?p=4000:184:%session%::::FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4008:%session%::NO::F4000_P4008_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:493:%session%::NO:493:FB_FLOW_ID,P493_ID:%application_id%,%pk_value%',
        p_file_path                 => 'shared_components/authorizations#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_acl_role,
        p_name                      => 'ACL_ROLE',
        p_singular_title            => 'Role',
        p_plural_title              => 'Roles',
        p_table_name                => 'WWV_FLOW_FND_USER_GROUPS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'GROUP_NAME',
        p_create_url                => 'f?p=4000:2320:%session%:::RP,2320:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:2320:%session%:::RP,2320:F4000_P2320_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/acl_roles#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_module,
        p_name                      => 'WEB_SRC_MODULE',
        p_singular_title            => 'REST Data Source',
        p_plural_title              => 'REST Data Sources',
        p_table_name                => 'WWV_FLOW_WEB_SRC_MODULES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'WEB SOURCE TYPE',
        p_plugin_column             => 'WEB_SOURCE_TYPE',
        p_where_clause              => null,
        p_create_url                => 'f?p=4000:1905:%session%:::1941,1942,1943,1944,1946,1905:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:1921:%session%:::RP,1921:P1921_WEB_SRC_MODULE_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/rest_data_sources#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_mod_param,
        p_name                      => 'WEB_SRC_MODULE_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_PARAMS',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'web_src_operation_id is null',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_web_src_mod_dp,
        p_column_type_id            => c_comp_type_web_src_mod_dp_col,
        p_name_prefix               => 'WEB_SRC_MODULE',
        p_parent_type_id            => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_operation,
        p_name                      => 'WEB_SRC_OPERATION',
        p_singular_title            => 'Operation',
        p_plural_title              => 'Operations',
        p_table_name                => 'WWV_FLOW_WEB_SRC_OPERATIONS',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'OPERATION',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_oper_param,
        p_name                      => 'WEB_SRC_OPER_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_PARAMS',
        p_parent_component_type_id  => c_comp_type_web_src_operation,
        p_parent_fk_column          => 'WEB_SRC_OPERATION_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'web_src_operation_id is not null',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_web_src_oper_dp,
        p_column_type_id            => c_comp_type_web_src_oper_dpcol,
        p_name_prefix               => 'WEB_SRC_OPERATION',
        p_parent_type_id            => c_comp_type_web_src_operation,
        p_parent_fk_column          => 'WEB_SRC_OPERATION_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_sync_step,
        p_name                      => 'WEB_SRC_SYNC_STEP',
        p_singular_title            => 'Synchronization Step',
        p_plural_title              => 'Synchronization Steps',
        p_table_name                => 'WWV_FLOW_WEB_SRC_SYNC_STEPS',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'STATIC_ID',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_sync_param,
        p_name                      => 'WEB_SRC_SYNC_STEP_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_web_src_sync_step,
        p_parent_fk_column          => 'MODULE_SYNC_STEP_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_query,
        p_name                      => 'WEB_SRC_QUERY',
        p_singular_title            => 'REST Source Query',
        p_plural_title              => 'REST Source Query',
        p_table_name                => 'WWV_FLOW_WEB_SRC_MODULES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~exists( select 1 from wwv_flow_web_src_operations where web_src_module_id=wwv_flow_web_src_modules.id and database_operation = 'FETCH_COLLECTION') ~',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true,
        p_is_query_only             => true );
    --
    -- Helper component type to get all parameters of a web src module (incl. operation parameters)
    -- This is needed for wwv_flow_web_src_comp_params
    store_component_type (
        p_id                        => c_comp_type_web_src_param,
        p_name                      => 'WEB_SRC_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_PARAMS',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true,
        p_is_query_only             => true );
    --
    store_component_type (
        p_id                        => c_comp_type_application_file,
        p_name                      => 'STATIC_APPLICATION_FILE',
        p_singular_title            => 'Static Application File',
        p_plural_title              => 'Static Application Files',
        p_table_name                => 'WWV_FLOW_STATIC_FILES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'FILE_NAME',
        p_create_url                => 'f?p=4000:271:%session%::NO:RP,271:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:275:%session%::NO:RP,275:P275_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/app_static_files#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_text_message,
        p_name                      => 'TEXT_MESSAGE',
        p_singular_title            => 'Text Message',
        p_plural_title              => 'Text Messages',
        p_table_name                => 'WWV_FLOW_MESSAGES$',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4009:%session%::NO:4009:P4009_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/messages#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_app_setting,
        p_name                      => 'APP_SETTING',
        p_singular_title            => 'Application Setting',
        p_plural_title              => 'Application Settings',
        p_table_name                => 'WWV_FLOW_APP_SETTINGS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4851:%session%::NO:4851:F4000_P4851_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_classic_nav_bar,
        p_name                      => 'CLASSIC_NAV_BAR',
        p_singular_title            => 'Classic Navigation Bar Entrie',
        p_plural_title              => 'Classic Navigation Bar Entries',
        p_table_name                => 'WWV_FLOW_ICON_BAR',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'ICON_SUBTEXT',
        p_file_path                 => 'shared_components/classic_navigation_bar_entries#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_breadcrumb,
        p_name                      => 'BREADCRUMB',
        p_singular_title            => 'Breadcrumb',
        p_plural_title              => 'Breadcrumbs',
        p_table_name                => 'WWV_FLOW_MENUS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:288:%session%::NO:RP,288:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:288:%session%::NO:RP,288:F4000_P288_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/breadcrumbs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_breadcrumb_entry,
        p_name                      => 'BREADCRUMB_ENTRY',
        p_singular_title            => 'Entry',
        p_plural_title              => 'Entries',
        p_table_name                => 'WWV_FLOW_MENU_OPTIONS',
        p_parent_component_type_id  => c_comp_type_breadcrumb,
        p_parent_fk_column          => 'MENU_ID',
        p_display_column            => 'SHORT_NAME',
        p_create_url                => 'f?p=4000:290:%session%::NO:RP,290:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:290:%session%::NO:RP,290:F4000_P290_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_list,
        p_name                      => 'LIST',
        p_singular_title            => 'List',
        p_plural_title              => 'Lists',
        p_table_name                => 'WWV_FLOW_LISTS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:427:%session%::NO:427,440,464,529,734:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4050:%session%::NO:RP,4050:F4000_P4050_LIST_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:663:%session%::NO:663:FB_FLOW_ID,P663_ID:%application_id%,%pk_value%',
        p_file_path                 => 'shared_components/lists#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_list_entry,
        p_name                      => 'LIST_ENTRY',
        p_singular_title            => 'Entry',
        p_plural_title              => 'Entries',
        p_table_name                => 'WWV_FLOW_LIST_ITEMS',
        p_parent_component_type_id  => c_comp_type_list,
        p_parent_fk_column          => 'LIST_ID',
        p_display_column            => 'LIST_ITEM_LINK_TEXT',
        p_create_url                => 'f?p=4000:4052:%session%::NO:RP,4052,4050:F4000_P4050_LIST_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_edit_url                  => 'f?p=4000:4052:%session%::NO:RP,4052,4050:F4000_P4052_ID,F4000_P4050_LIST_ID,FB_FLOW_ID:%pk_value%,%parent_pk_value%,%application_id%',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_lov,
        p_name                      => 'LOV',
        p_singular_title            => 'List of Values',
        p_plural_title              => 'Lists of Values',
        p_table_name                => 'WWV_FLOW_LOVS_DEV',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'LOV_NAME',
        p_create_url                => 'f?p=4000:137:%session%::NO:4004,4007,4014,137,138,139:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4111:%session%::NO:4111:F4000_P4111_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:603:%session%::NO:603:FB_FLOW_ID,P603_ID:%application_id%,%pk_value%',
        p_file_path                 => 'shared_components/lovs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_lov_entry,
        p_name                      => 'LOV_ENTRY',
        p_singular_title            => 'Entry',
        p_plural_title              => 'Entries',
        p_table_name                => 'WWV_FLOW_LIST_OF_VALUES_DATA',
        p_parent_component_type_id  => c_comp_type_lov,
        p_parent_fk_column          => 'LOV_ID',
        p_display_column            => 'LOV_DISP_VALUE',
        p_create_url                => 'f?p=4000:4113:%session%::NO:4113,4111:F4000_P4111_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_edit_url                  => 'f?p=4000:4113:%session%::NO:4113,4111:F4000_P4113_ID,F4000_P4111_ID,FB_FLOW_ID:%pk_value%,%parent_pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_lov_column,
        p_name                      => 'LOV_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_LIST_OF_VALUES_COLS',
        p_parent_component_type_id  => c_comp_type_lov,
        p_parent_fk_column          => 'LOV_ID',
        p_display_column            => 'QUERY_COLUMN_NAME',
        p_create_url                => 'f?p=4000:4111:%session%::NO:4111:F4000_P4111_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_edit_url                  => 'f?p=4000:4111:%session%::NO:4111:F4000_P4111_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_lov_ws_param,
        p_name                      => 'LOV_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_lov,
        p_parent_fk_column          => 'SHARED_LOV_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref,
        p_name                      => 'WEB_SERVICE_REFERENCE',
        p_singular_title            => 'Web Service Reference',
        p_plural_title              => 'Web Service References',
        p_table_name                => 'WWV_FLOW_SHARED_WEB_SERVICES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:870:%session%::NO:550,555,560,565,570,575,580,585,595,870,871,872,873,874,875,REST_INPUT_PARMS,REST_OUTPUT_PARMS,REST_HTTP_HEADERS:FB_FLOW_ID:%application_id%',
        p_copy_url                  => '',
        p_edit_url                  => '',  /* web services do have different edit pages */
        p_file_path                 => 'shared_components/legacy_web_service_references#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_operations,
        p_name                      => 'WEB_SERVICE_REFERENCE_OPERATION',
        p_singular_title            => 'Operation',
        p_plural_title              => 'Operations',
        p_table_name                => 'WWV_FLOW_WS_OPERATIONS_DEV',
        p_parent_component_type_id  => c_comp_type_ws_ref,
        p_parent_fk_column          => 'WS_ID',
        p_display_column            => 'FULL_NAME',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_i,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_I',
        p_singular_title            => 'In Parameter',
        p_plural_title              => 'In Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'I' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_o,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_O',
        p_singular_title            => 'Out Parameter',
        p_plural_title              => 'Out Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'O' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_a,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_A',
        p_singular_title            => 'Basic Authentication Parameter',
        p_plural_title              => 'Basic Authentication Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'A' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_h,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_H',
        p_singular_title            => 'HTTP Header Parameter',
        p_plural_title              => 'HTTP Header Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'H' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_data_load_def,
        p_name                      => 'DATA_LOAD_DEFINITION',
        p_singular_title            => 'Data Load Definition',
        p_plural_title              => 'Data Load Definitions',
        p_table_name                => 'WWV_FLOW_LOAD_TABLES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~target_type is not null ~',
        p_edit_url                  => 'f?p=4000:1820:%session%::NO:1820:P1820_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/data_load_definitions#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_data_load_dp,
        p_column_type_id            => c_comp_type_data_load_dp_col,
        p_name_prefix               => 'DATA_LOAD_DEF',
        p_parent_type_id            => c_comp_type_data_load_def,
        p_parent_fk_column          => 'LOAD_TABLE_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_legacy_data_load,
        p_name                      => 'DATA_LOAD_TABLE',
        p_singular_title            => 'Legacy Data Load Definition',
        p_plural_title              => 'Legacy Data Load Definitions',
        p_table_name                => 'WWV_FLOW_LOAD_TABLES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~target_type is null ~',
        p_edit_url                  => 'f?p=4000:1801:%session%::NO:1801:P1801_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/legacy_data_load_definitions#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_tab_set,
        p_name                      => 'TAB SET',
        p_singular_title            => 'Standard Tab Set',
        p_plural_title              => 'Standard Tab Sets',
        p_table_name                => 'WWV_FLOW_TABSET_VIEW',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'D',
        p_edit_url                  => 'f?p=4000:643:%session%::NO:643:FB_FLOW_ID:%application_id%',
        p_file_path                 => 'shared_components/legacy_tabs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_report_layout,
        p_name                      => 'REPORT LAYOUT',
        p_singular_title            => 'Report Layout',
        p_plural_title              => 'Report Layouts',
        p_table_name                => 'WWV_FLOW_REPORT_LAYOUTS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'REPORT_LAYOUT_NAME',
        p_edit_url                  => 'f?p=4000:4111:%session%::NO:4111:F4000_P4111_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/report_layouts#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_report_query,
        p_name                      => 'REPORT_QUERY',
        p_singular_title            => 'Report Query',
        p_plural_title              => 'Report Queries',
        p_table_name                => 'WWV_FLOW_SHARED_QUERIES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:1116:%session%::NO:1116,1117,1112:P1116_ID,P1116_INIT,FB_FLOW_ID:%pk_value%,YES,%application_id%',
        p_file_path                 => 'shared_components/report_queries#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_report_query_stmt,
        p_name                      => 'REPORT_QUERY_STATEMENT',
        p_singular_title            => 'Source Query',
        p_plural_title              => 'Source Queries',
        p_table_name                => 'WWV_FLOW_SHARED_QRY_SQL_STMTS',
        p_parent_component_type_id  => c_comp_type_report_query,
        p_parent_fk_column          => 'SHARED_QUERY_ID',
        p_display_column            => 'SQL_STATEMENT',
        p_edit_url                  => 'f?p=4000:1116:%session%::NO:1116,1117,1112:P1116_ID,P1116_INIT,FB_FLOW_ID:%parent_pk_value%,YES,%application_id%' ); -- intentionally pointing to parent page because collections have to be populated
    --
    store_component_type (
        p_id                        => c_comp_type_automation,
        p_name                      => 'AUTOMATION',
        p_singular_title            => 'Automation',
        p_plural_title              => 'Automations',
        p_table_name                => 'WWV_FLOW_AUTOMATIONS_DEV',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:7021:%session%::NO:RP,7021:P7021_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/automations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_autom_ws_param,
        p_name                      => 'AUTOMATION_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_automation,
        p_parent_fk_column          => 'AUTOMATION_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_automation_action,
        p_name                      => 'AUTOMATION_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_AUTOMATION_ACTIONS',
        p_parent_component_type_id  => c_comp_type_automation,
        p_parent_fk_column          => 'AUTOMATION_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'PROCESS TYPE',
        p_plugin_column             => 'ACTION_TYPE',
        p_edit_url                  => 'f?p=4000:7022:%session%::NO:RP,7022:P7022_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_email_template,
        p_name                      => 'EMAIL_TEMPLATE',
        p_singular_title            => 'Email Template',
        p_plural_title              => 'Email Templates',
        p_table_name                => 'WWV_FLOW_EMAIL_TEMPLATES_DEV',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:4861:%session%::NO:RP,4861:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4861:%session%::NO:RP,4861:F4000_P4861_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/email_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_language_map,
        p_name                      => 'LANGUAGE_MAP',
        p_singular_title            => 'Language Mapping',
        p_plural_title              => 'Language Mappings',
        p_table_name                => 'WWV_FLOW_LANGUAGE_MAP',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'PRIMARY_LANGUAGE_FLOW_ID',
        p_display_column            => 'TRANSLATION_FLOW_LANGUAGE_CODE',
        p_edit_url                  => 'f?p=4000:4754:%session%::NO:4754:P4754_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_dyn_translation,
        p_name                      => 'DYNAMIC_TRANSLATION',
        p_singular_title            => 'Dynamic Translation',
        p_plural_title              => 'Dynamic Translations',
        p_table_name                => 'WWV_FLOW_DYNAMIC_TRANSLATIONS$',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'TRANSLATE_FROM_TEXT',
        p_edit_url                  => 'f?p=4000:4757:%session%::NO:4757:F4000_P4757_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/dynamic_translations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_user_interface,
        p_name                      => 'USER_INTERFACE',
        p_singular_title            => 'User Interface',
        p_plural_title              => 'User Interfaces',
        p_table_name                => 'WWV_FLOW_USER_INTERFACES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_edit_url                  => 'f?p=4000:677:%session%::NO:677:P677_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_concatenated_file,
        p_name                      => 'CONCATENATED_FILE',
        p_singular_title            => 'Concatenated File',
        p_plural_title              => 'Concatenated Files',
        p_table_name                => 'WWV_FLOW_COMBINED_FILES',
        p_parent_component_type_id  => c_comp_type_user_interface,
        p_parent_fk_column          => 'USER_INTERFACE_ID',
        p_display_column            => 'COMBINED_FILE_URL',
        p_file_path                 => 'shared_components/concatenated_files#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def,
        p_name                      => 'TASK_DEF',
        p_singular_title            => 'Task Definition',
        p_plural_title              => 'Task Definitions',
        p_table_name                => 'WWV_FLOW_TASK_DEFS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:9502:%session%::NO:9502:P9502_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/task_definitions#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def_param,
        p_name                      => 'TASK_DEF_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_TASK_DEF_PARAMS',
        p_parent_component_type_id  => c_comp_type_task_def,
        p_parent_fk_column          => 'TASK_DEF_ID',
        p_display_column            => 'LABEL',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def_action,
        p_name                      => 'TASK_DEF_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_TASK_DEF_ACTIONS',
        p_parent_component_type_id  => c_comp_type_task_def,
        p_parent_fk_column          => 'TASK_DEF_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'PROCESS TYPE',
        p_plugin_column             => 'ACTION_TYPE' );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def_particip,
        p_name                      => 'TASK_DEF_PARTICIPANT',
        p_singular_title            => 'Participant',
        p_plural_title              => 'Participants',
        p_table_name                => 'WWV_FLOW_TASK_DEF_PARTICIPANTS',
        p_parent_component_type_id  => c_comp_type_task_def,
        p_parent_fk_column          => 'TASK_DEF_ID',
        p_display_column            => 'PARTICIPANT_TYPE' );
    --
    -- Misc component types
    store_component_type (
        p_id                        => c_comp_type_page_group,
        p_name                      => 'PAGE_GROUP',
        p_singular_title            => 'Page Group',
        p_plural_title              => 'Page Groups',
        p_table_name                => 'WWV_FLOW_PAGE_GROUPS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'GROUP_NAME',
        p_file_path                 => 'page_groups#ext#',
        p_is_page_designer_shared   => true );
    --
    -- Page related component types
    --
    store_component_type (
        p_id                        => c_comp_type_page,
        p_name                      => 'PAGE',
        p_singular_title            => 'Page',
        p_plural_title              => 'Pages',
        p_table_name                => 'WWV_FLOW_STEPS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:2500:%session%::NO:2500,2501:FB_FLOW_ID:%application_id%',
        p_copy_url                  => 'f?p=4000:618:%session%::NO:618,301,302,305,308,309:FB_FLOW_ID:%application_id%',
        p_file_path                 => 'pages/' );     
      
    -- Page Rendering
    store_component_type (
        p_id                        => c_comp_type_region,
        p_name                      => 'REGION',
        p_singular_title            => 'Region',
        p_plural_title              => 'Regions',
        p_table_name                => 'WWV_FLOW_PAGE_PLUGS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_REGIONS',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'PAGE_ID',
        p_page_id_column            => 'PAGE_ID',
        p_plugin_type               => 'REGION TYPE',
        p_plugin_column             => 'PLUG_SOURCE_TYPE',
        p_display_column            => 'PLUG_NAME',
        p_copy_url                  => 'f?p=4000:194:%session%::NO:194:FB_FLOW_ID,F4000_P194_PLUG_ID_FROM,F4000_P194_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_item,
        p_name                      => 'PAGE_ITEM',
        p_singular_title            => 'Page Item',
        p_plural_title              => 'Page Items',
        p_table_name                => 'WWV_FLOW_STEP_ITEMS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_ITEMS',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => 'f?p=4000:171:%session%::NO:171,172:FB_FLOW_ID,F4000_P171_COPY_FROM_ID,F4000_P171_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_button,
        p_name                      => 'BUTTON',
        p_singular_title            => 'Button',
        p_plural_title              => 'Buttons',
        p_table_name                => 'WWV_FLOW_STEP_BUTTONS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_BUTTONS',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'BUTTON_NAME',
        p_copy_url                  => 'f?p=4000:231:%session%::NO:231,191:FB_FLOW_ID,F4000_P231_COPY_FROM_ID,F4000_P231_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_da_event,
        p_name                      => 'DYNAMIC_ACTION_EVENT',
        p_singular_title            => 'Dynamic Action',
        p_plural_title              => 'Dynamic Actions',
        p_table_name                => 'WWV_FLOW_PAGE_DA_EVENTS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_DA',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'PAGE_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_copy_url                  => 'f?p=4000:796:%session%::NO:128,796,797:FB_FLOW_ID,F4000_P796_COPY_FROM_ID,F4000_P796_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_da_action,
        p_name                      => 'DYNAMIC_ACTION_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_PAGE_DA_ACTIONS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_DA_ACTS',
        p_parent_component_type_id  => c_comp_type_da_event,
        p_parent_fk_column          => 'EVENT_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'ACTION',
        p_plugin_type               => 'DYNAMIC ACTION',
        p_plugin_column             => 'ACTION' );

    -- Page Processing
    store_component_type (
        p_id                        => c_comp_type_validation,
        p_name                      => 'VALIDATION',
        p_singular_title            => 'Validation',
        p_plural_title              => 'Validations',
        p_table_name                => 'WWV_FLOW_STEP_VALIDATIONS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_VAL',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'VALIDATION_NAME',
        p_copy_url                  => 'f?p=4000:583:%session%::NO:583,596:FB_FLOW_ID,P583_ID:%application_id%,%pk_value%' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_computation,
        p_name                      => 'PAGE_COMPUTATION',
        p_singular_title            => 'Computation',
        p_plural_title              => 'Computations',
        p_table_name                => 'WWV_FLOW_STEP_COMPUTATIONS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_COMP',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'COMPUTATION_ITEM',
        p_copy_url                  => 'f?p=4000:592:%session%::NO:592,597:FB_FLOW_ID,P592_ID:%application_id%,%pk_value%' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_process,
        p_name                      => 'PAGE_PROCESS',
        p_singular_title            => 'Process',
        p_plural_title              => 'Processes',
        p_table_name                => 'WWV_FLOW_STEP_PROCESSING',
        p_view_name                 => 'APEX_APPLICATION_PAGE_PROC',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'PROCESS_NAME',
        p_plugin_type               => 'PROCESS TYPE',
        p_plugin_column             => 'PROCESS_TYPE',
        p_copy_url                  => 'f?p=4000:590:%session%::NO:590,645:FB_FLOW_ID,P590_ID:%application_id%,%pk_value%' );
    --
    store_component_type (
        p_id                        => c_comp_type_branch,
        p_name                      => 'BRANCH',
        p_singular_title            => 'Branch',
        p_plural_title              => 'Branches',
        p_table_name                => 'WWV_FLOW_STEP_BRANCHES_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_BRANCHES',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'BRANCH_NAME',
        p_copy_url                  => 'f?p=4000:587:%session%::NO:587,685:FB_FLOW_ID,P587_ID:%application_id%,%pk_value%' );
    --
    -- Region type specific component types
    --
    -- Region Plug-in Attributes
    store_component_type (
        p_id                       => c_comp_type_region_plugin_attr,
        p_name                     => 'REGION_PLUGIN_ATTRIBUTES',
        p_singular_title           => 'Attributes',
        p_plural_title             => 'Attributes',
        p_table_name               => 'WWV_FLOW_REGION_PLUGIN_DEV',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_plugin_type              => 'REGION TYPE',
        p_plugin_column            => 'PLUG_SOURCE_TYPE',
        p_display_column           => null );

    -- Interactive Report
    store_component_type (
        p_id                        => c_comp_type_ir_attributes,
        p_name                      => 'IR_ATTRIBUTES',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_WORKSHEETS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IR',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_saved_report,
        p_name                      => 'IR_SAVED_REPORT',
        p_singular_title            => 'Saved Report',
        p_plural_title              => 'Saved Reports',
        p_table_name                => 'WWV_FLOW_WORKSHEET_RPTS_DEV',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'visibility <> ''PRIVATE''' );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_column_group,
        p_name                      => 'IR_COLUMN_GROUP',
        p_singular_title            => 'Column Group',
        p_plural_title              => 'Column Groups',
        p_table_name                => 'WWV_FLOW_WORKSHEET_COL_GRP_DEV',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_column,
        p_name                      => 'IR_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_WORKSHEET_COL_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IR_COL',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'DB_COLUMN_NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_print,
        p_name                      => 'IR_PRINT_ATTR',
        p_singular_title            => 'Printing',
        p_plural_title              => 'Printing',
        p_table_name                => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                 => 'WORKSHEET_ID',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null,
        p_where_clause              => q'~plug_source_type = 'NATIVE_IR' and ( download_formats like '%XLSX%' or download_formats like '%PDF%' ) ~' );

    -- Interactive grid
    store_component_type (
        p_id                        => c_comp_type_ig_attributes,
        p_name                      => 'IG_ATTRIBUTES',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_INTERACTIVE_GRIDS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IG',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_column_group,
        p_name                      => 'IG_COLUMN_GROUP',
        p_singular_title            => 'Column Group',
        p_plural_title              => 'Column Groups',
        p_table_name                => 'WWV_FLOW_REGION_COLUMN_GROUPS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'HEADING' );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_column,
        p_name                      => 'IG_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_IG_COLUMNS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IG_COL',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'ITEM_TYPE',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_saved_report,
        p_name                      => 'IG_SAVED_REPORT',
        p_singular_title            => 'Saved Report',
        p_plural_title              => 'Saved Reports',
        p_table_name                => 'WWV_FLOW_IG_REPORTS_DEV',
        p_parent_component_type_id  => c_comp_type_ig_attributes,
        p_parent_fk_column          => 'INTERACTIVE_GRID_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'visibility in (''PRIMARY'',''ALTERNATIVE'')' );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_print,
        p_name                      => 'IG_PRINT_ATTR',
        p_singular_title            => 'Printing',
        p_plural_title              => 'Printing',
        p_table_name                => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                 => 'REGION_ID',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null,
        p_where_clause              => q'~plug_source_type = 'NATIVE_IG' and ( download_formats like '%XLSX%' or download_formats like '%PDF%' ) ~' );

    -- Web Service
    store_component_type (
        p_id                        => c_comp_type_page_proc_ws_p_i,
        p_name                      => 'PAGE_PROCESS_WS_PARAM_IN',
        p_singular_title            => 'In Parameter',
        p_plural_title              => 'In Parameters',
        p_table_name                => 'WWV_FLOW_PAGE_PROC_WS_PARM_DEV',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'PARAMETER_NAME',
        p_where_clause              => q'~parameter_type in ( 'I', 'H' ) ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_proc_ws_p_o,
        p_name                      => 'PAGE_PROCESS_WS_PARAM_OUT',
        p_singular_title            => 'Out Parameter',
        p_plural_title              => 'Out Parameters',
        p_table_name                => 'WWV_FLOW_PAGE_PROC_WS_PARM_DEV',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'PARAMETER_NAME',
        p_where_clause              => q'~parameter_type = 'O' ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_proc_ws_p_a,
        p_name                      => 'PAGE_PROCESS_WS_PARAM_AUTH',
        p_singular_title            => 'Basic Authentication Parameter',
        p_plural_title              => 'Basic Authentication Parameters',
        p_table_name                => 'WWV_FLOW_PAGE_PROC_WS_PARM_DEV',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'PARAMETER_NAME',
        p_where_clause              => q'~parameter_type = 'A' ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_region_ws_param,
        p_name                      => 'REGION_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PAGE_PLUG_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    -- Classic Report
    store_component_type (
        p_id                       => c_comp_type_classic_report,
        p_name                     => 'CLASSIC_REPORT',
        p_singular_title           => 'Attributes',
        p_plural_title             => 'Attributes',
        p_table_name               => 'WWV_FLOW_REGION_RPT_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_REGIONS',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_SQL_REPORT' ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_classic_rpt_column,
        p_name                     => 'CLASSIC_REPORT_COLUMN',
        p_singular_title           => 'Column',
        p_plural_title             => 'Columns',
        p_table_name               => 'WWV_FLOW_REGION_RPT_COL_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_RPT_COLS',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'FLOW_STEP_ID',
        p_display_column           => 'COLUMN_ALIAS',
        p_where_clause             => q'~exists ( select 1 from wwv_flow_page_plugs a where a.id = region_id and a.plug_source_type = 'NATIVE_SQL_REPORT' ) ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_classic_rpt_print,
        p_name                     => 'CLASSIC_REPORT_PRINT_ATTR',
        p_singular_title           => 'Printing',
        p_plural_title             => 'Printing',
        p_table_name               => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_classic_report,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_SQL_REPORT' and prn_output = 'Y' ~' );

    -- Tabular Form
    store_component_type (
        p_id                       => c_comp_type_tab_form,
        p_name                     => 'TABULAR_FORM',
        p_singular_title           => 'Attributes',
        p_plural_title             => 'Attributes',
        p_table_name               => 'WWV_FLOW_REGION_RPT_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_REGIONS',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_TABFORM' ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_tab_form_column,
        p_name                     => 'TABULAR_FORM_COLUMN',
        p_singular_title           => 'Column',
        p_plural_title             => 'Columns',
        p_table_name               => 'WWV_FLOW_REGION_RPT_COL_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_RPT_COLS',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'FLOW_STEP_ID',
        p_display_column           => 'COLUMN_ALIAS',
        p_where_clause             => q'~exists ( select 1 from wwv_flow_page_plugs a where a.id = region_id and a.plug_source_type = 'NATIVE_TABFORM' ) ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_tab_form_print,
        p_name                     => 'TABULAR_FORM_PRINT_ATTR',
        p_singular_title           => 'Printing',
        p_plural_title             => 'Printing',
        p_table_name               => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_tab_form,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_TABFORM' and prn_output = 'Y' ~' );
    -- Classic Calendar
    store_component_type (
        p_id                        => c_comp_type_classic_calendar,
        p_name                      => 'CLASSIC_CALENDAR',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_CALS_DEV',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PLUG_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_region_column,
        p_name                      => 'REGION_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_REGION_COLUMNS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_REG_COLS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_plugin_type               => 'REGION TYPE',
        p_plugin_column             => 'REGION_TYPE',
        p_display_column            => 'NAME' );

    -- JET Chart
    store_component_type (
        p_id                        => c_comp_type_jet_chart,
        p_name                      => 'JET_CHART',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_JET_CHARTS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_jet_chart_series,
        p_name                      => 'JET_CHART_SERIES',
        p_singular_title            => 'Series',
        p_plural_title              => 'Series',
        p_table_name                => 'WWV_FLOW_JET_SERIES_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_CHART_S',
        p_parent_component_type_id  => c_comp_type_jet_chart,
        p_parent_fk_column          => 'CHART_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_jet_chart_axes,
        p_name                      => 'JET_CHART_AXES',
        p_singular_title            => 'Axis',
        p_plural_title              => 'Axes',
        p_table_name                => 'WWV_FLOW_JET_AXES_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_CHART_A',
        p_parent_component_type_id  => c_comp_type_jet_chart,
        p_parent_fk_column          => 'CHART_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'AXIS' );
    --
    store_component_type (
        p_id                        => c_comp_type_jet_chart_ws_param,
        p_name                      => 'JET_CHART_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_jet_chart_series,
        p_parent_fk_column          => 'JET_CHART_SERIES_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_map_region,
        p_name                      => 'MAP_REGION',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_MAP_REGIONS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_MAP_REGIONS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_map_layer,
        p_name                      => 'MAP_LAYER',
        p_singular_title            => 'Layer',
        p_plural_title              => 'Layers',
        p_table_name                => 'WWV_FLOW_MAP_LAYERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_MAP_LAYERS',
        p_parent_component_type_id  => c_comp_type_map_region,
        p_parent_fk_column          => 'MAP_REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_map_layer_ws_param,
        p_name                      => 'MAP_LAYER_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_map_layer,
        p_parent_fk_column          => 'MAP_REGION_LAYER_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_facet,
        p_name                      => 'FACET',
        p_singular_title            => 'Facet',
        p_plural_title              => 'Facets',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'ITEM_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is null' );
    --
    store_component_type (
        p_id                        => c_comp_type_facet_group,
        p_name                      => 'FACET_GROUP',
        p_singular_title            => 'Facet Group',
        p_plural_title              => 'Facet Groups',
        p_table_name                => 'WWV_FLOW_STEP_ITEM_GROUPS',
        p_view_name                 => 'APEX_APPL_PAGE_FILTER_GROUPS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PAGE_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'LABEL',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_facet_group_item,
        p_name                      => 'FACET_GROUP_ITEM',
        p_singular_title            => 'Checkbox',
        p_plural_title              => 'Checkboxes',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_parent_component_type_id  => c_comp_type_facet_group,
        p_parent_fk_column          => 'ITEM_GROUP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is not null' );
    --
    store_component_type (
        p_id                        => c_comp_type_sfilter,
        p_name                      => 'SFILTER',
        p_singular_title            => 'Filter',
        p_plural_title              => 'Filters',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'ITEM_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is null' );
    --
    store_component_type (
        p_id                        => c_comp_type_sfilter_group,
        p_name                      => 'SFILTER_GROUP',
        p_singular_title            => 'Filter Group',
        p_plural_title              => 'Filter Groups',
        p_table_name                => 'WWV_FLOW_STEP_ITEM_GROUPS',
        p_view_name                 => 'APEX_APPL_PAGE_FILTER_GROUPS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PAGE_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'LABEL',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_sfilter_group_item,
        p_name                      => 'SFILTER_GROUP_ITEM',
        p_singular_title            => 'Checkbox',
        p_plural_title              => 'Checkboxes',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_parent_component_type_id  => c_comp_type_sfilter_group,
        p_parent_fk_column          => 'ITEM_GROUP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is not null' );
    --
    store_component_type (
        p_id                        => c_comp_type_card,
        p_name                      => 'CARD',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_CARDS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_CARDS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_card_action,
        p_name                      => 'CARD_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_CARD_ACTIONS',
        p_view_name                 => 'APEX_APPL_PAGE_CARD_ACTIONS',
        p_parent_component_type_id  => c_comp_type_card,
        p_parent_fk_column          => 'CARD_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'ACTION_TYPE' );
    --
    store_component_type (
        p_id                        => c_comp_type_pg_pr_task_param,
        p_name                      => 'PAGE_PROCESS_TASK_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_TASK_DEF_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PAGE_PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'TASK_DEF_PARAM_ID' );
    --    
    store_component_type (
        p_id                        => c_comp_type_shortcut,
        p_name                      => 'SHORTCUT',
        p_singular_title            => 'Shortcut',
        p_plural_title              => 'Shortcuts',
        p_table_name                => 'WWV_FLOW_SHORTCUTS',
        p_view_name                 => 'APEX_APPLICATION_SHORTCUTS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'SHORTCUT_NAME',
        p_create_url                => 'f?p=4000:131:%session%::NO:RP,131:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4048:%session%::NO:RP,4048:F4000_P4048_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/shortcuts#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_tree,
        p_name                      => 'TREE',
        p_singular_title            => 'Tree',
        p_plural_title              => 'Trees',
        p_table_name                => 'WWV_FLOW_TREES',
        p_view_name                 => 'APEX_APPLICATION_TREES',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'TREE_NAME',
        p_file_path                 => 'shared_components/legacy_trees#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin,
        p_name                      => 'PLUGIN',
        p_singular_title            => 'Plugin',
        p_plural_title              => 'Plugins',
        p_table_name                => 'WWV_FLOW_PLUGINS',
        p_view_name                 => 'APEX_APPL_PLUGINS',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_file_path                 => 'shared_components/plugins#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj,
        p_name                      => 'SUPPORT_OBJECTS',
        p_singular_title            => 'Supporting Object',
        p_plural_title              => 'Supporting Object',
        p_table_name                => 'WWV_FLOW_INSTALL',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'FLOW_ID',
        p_file_path                 => 'supporting_objects/supporting_objects#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_substitut,
        p_name                      => 'SUPPORT_OBJECTS_SUBSTITUTION',
        p_singular_title            => 'Substitution',
        p_plural_title              => 'Substitutions',
        p_table_name                => 'WWV_FLOW_INSTALL_SUBST_DEV',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'SUBSTITUTION_STRING',
        p_file_path                 => 'supporting_objects/substitutions#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_build_opt,
        p_name                      => 'SUPPORT_OBJECTS_BUILD_OPTION',
        p_singular_title            => 'Build Option',
        p_plural_title              => 'Build Options',
        p_table_name                => 'WWV_FLOW_INSTALL_BUILD_OPT',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'BUILD_OPT_ID',
        p_file_path                 => 'supporting_objects/build_options#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_validation,
        p_name                      => 'SUPPORT_OBJECTS_VALIDATION',
        p_singular_title            => 'Validation',
        p_plural_title              => 'Validations',
        p_table_name                => 'WWV_FLOW_INSTALL_CHECKS',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'NAME',
        p_file_path                 => 'supporting_objects/validations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_install,
        p_name                      => 'SUPPORT_OBJECTS_INSTALL',
        p_singular_title            => 'Install Script',
        p_plural_title              => 'Install Scripts',
        p_table_name                => 'WWV_FLOW_INSTALL_SCRIPTS',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~coalesce( script_type, 'INSTALL' ) = 'INSTALL'~', -- $$$ todo we should do a data migration
        p_file_path                 => 'supporting_objects/install_scripts#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_upgrade,
        p_name                      => 'SUPPORT_OBJECTS_UPGRADE',
        p_singular_title            => 'Upgrade Script',
        p_plural_title              => 'Upgrade Scripts',
        p_table_name                => 'WWV_FLOW_INSTALL_SCRIPTS',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~script_type = 'UPGRADE'~',
        p_file_path                 => 'supporting_objects/upgrade_scripts#ext#' );
    --
    --*******************
    -- END COMPONENT TYPE
    --*******************
    --
    ------------------------------------------------------------------------
    -- Display Groups
    ------------------------------------------------------------------------
    store_display_group (
        p_name  => 'IDENTIFICATION',
        p_title => 'Identification' );
    --
    store_display_group (
        p_name  => 'SETTINGS',
        p_title => 'Settings' );
    --
    store_display_group (
        p_name  => 'ATTRIBUTES',
        p_title => 'Attributes' );
    --
    store_display_group (
        p_name  => 'COMMENTS',
        p_title => 'Comments' );
    --
    store_display_group (
        p_name  => 'ERROR',
        p_title => 'Error' );
    --
    store_display_group (
        p_name  => 'CONFIGURATION',
        p_title => 'Configuration' );
    --
    store_display_group (
        p_name  => 'CSS',
        p_title => 'CSS' );
    --
    store_display_group (
        p_name  => 'DIALOG',
        p_title => 'Dialog' );
    --
    store_display_group (
        p_name  => 'ERROR_HANDLING',
        p_title => 'Error Handling' );
    --
    store_display_group (
        p_name  => 'HEADER_AND_FOOTER',
        p_title => 'Header and Footer' );
    --
    store_display_group (
        p_name  => 'HELP',
        p_title => 'Help' );
    --
    store_display_group (
        p_name  => 'HTML_HEADER',
        p_title => 'HTML Header' );
    --
    store_display_group (
        p_name  => 'JAVASCRIPT',
        p_title => 'JavaScript' );
    --
    store_display_group (
        p_name  => 'READ_ONLY',
        p_title => 'Read Only' );
    --
    store_display_group (
        p_name  => 'SECURITY',
        p_title => 'Security' );
    --
    store_display_group (
        p_name  => 'SERVER_CACHE',
        p_title => 'Server Cache' );
    --
    store_display_group (
        p_name  => 'SERVER_SIDE_CONDITION',
        p_title => 'Server-side Condition' );
    store_display_group (
        p_name  => 'CLIENT_SIDE_CONDITION',
        p_title => 'Client-side Condition' );
    --
    store_display_group (
        p_name  => 'EXECUTION_OPTIONS',
        p_title => 'Execution Options' );
    --
    store_display_group (
        p_name  => 'COMPUTATION',
        p_title => 'Computation' );
    --
    store_display_group (
        p_name  => 'ADVANCED',
        p_title => 'Advanced' );
    --
    store_display_group (
        p_name  => 'AUTHORIZATION',
        p_title => 'Authorization' );
    --
    store_display_group (
        p_name  => 'WHEN',
        p_title => 'When' );
    --
    store_display_group (
        p_name  => 'AFFECTED_ELEMENTS',
        p_title => 'Affected Elements' );
    --
    store_display_group (
        p_name  => 'DEFAULT',
        p_title => 'Default' );
    --
    store_display_group (
        p_name  => 'ELEMENT',
        p_title => 'Element' );
    --
    store_display_group (
        p_name  => 'HELP_TEXT',
        p_title => 'Help Text' );
    --
    store_display_group (
        p_name  => 'LABEL',
        p_title => 'Label' );
    --
    store_display_group (
        p_name  => 'LIST_OF_VALUES',
        p_title => 'List of Values' );
    --
    store_display_group (
        p_name  => 'QUICK_PICKS',
        p_title => 'Quick Picks' );
    --
    store_display_group (
        p_name  => 'SOURCE',
        p_title => 'Source' );
    --
    store_display_group (
        p_name  => 'VALIDATION',
        p_title => 'Validation' );
    --
    store_display_group (
        p_name  => 'BEHAVIOR',
        p_title => 'Behavior' );
    --
    store_display_group (
        p_name  => 'APPEARANCE',
        p_title => 'Appearance' );
    --
    store_display_group (
        p_name  => 'REGION_IMAGE',
        p_title => 'Image' );
    --
    store_display_group (
        p_name  => 'THRESHOLDS',
        p_title => 'Thresholds' );
    --
    store_display_group (
        p_name  => 'LAYOUT',
        p_title => 'Layout' );
    --
    store_display_group (
        p_name  => 'LOGIN_PROCESSING',
        p_title => 'Login Processing' );
    --
    store_display_group (
        p_name  => 'POST_LOGOUT_URL',
        p_title => 'Post-Logout URL' );
    --
    store_display_group (
        p_name  => 'REAL_APPLICATION_SECURITY',
        p_title => 'Real Application Security' );
    --
    store_display_group (
        p_name  => 'REFERENCE_LINES',
        p_title => 'Reference Lines' );
    --
    store_display_group (
        p_name  => 'SESSION_SHARING',
        p_title => 'Session Sharing' );
    --
    store_display_group (
        p_name  => 'SESSION_NOT_VALID',
        p_title => 'Session Not Valid' );
    --
    store_display_group (
        p_name  => 'SUBSCRIPTION',
        p_title => 'Subscription' );
    --
    store_display_group (
        p_name  => 'CUSTOMIZATION',
        p_title => 'Customization' );
    --
    store_display_group (
        p_name  => 'NAVIGATION',
        p_title => 'Navigation' );
    --
    store_display_group (
        p_name  => 'NAVIGATION_MENU',
        p_title => 'Navigation Menu' );
    --
    store_display_group (
        p_name  => 'TABS',
        p_title => 'Tabs' );
    --
    store_display_group (
        p_name  => 'SEARCH_BAR',
        p_title => 'Search Bar' );
    --
    store_display_group (
        p_name  => 'ACTIONS_MENU',
        p_title => 'Actions Menu' );
    --
    store_display_group (
        p_name  => 'COLUMN_FORMATTING',
        p_title => 'Column Formatting' );
    --
    store_display_group (
        p_name  => 'ENABLE_USER_TO',
        p_title => 'Enable Users To' );
    --
    store_display_group (
        p_name  => 'DESCRIPTION',
        p_title => 'Description' );
    --
    store_display_group (
        p_name  => 'DETAIL_VIEW',
        p_title => 'Detail View' );
    --
    store_display_group (
        p_name  => 'DOWNLOAD',
        p_title => 'Download' );
    --
    store_display_group (
        p_name  => 'ICON_VIEW',
        p_title => 'Icon View' );
    --
    store_display_group (
        p_name  => 'PAGINATION',
        p_title => 'Pagination' );
    --
    store_display_group (
        p_name  => 'PERFORMANCE',
        p_title => 'Performance' );        
    --
    store_display_group (
        p_name  => 'SINGLE_ROW_VIEW',
        p_title => 'Single Row View' );
    --
    store_display_group (
        p_name  => 'LINK',
        p_title => 'Link' );
    --
    store_display_group (
        p_name  => 'SUCCESS_MESSAGE',
        p_title => 'Success Message' );
    --
    store_display_group (
        p_name  => 'PARAMETER',
        p_title => 'Parameter' );
    --
    store_display_group (
        p_name  => 'REFRESH',
        p_title => 'Refresh' );
    --
    store_display_group (
        p_name  => 'AUDIT_INFORMATION',
        p_title => 'Audit Information' );
    --
    store_display_group (
        p_name  => 'SORTING',
        p_title => 'Sorting' );
    --
    store_display_group (
        p_name  => 'EXPORT_PRINTING',
        p_title => 'Export / Printing' );
    --
    store_display_group (
        p_name  => 'BLOB',
        p_title => 'BLOB Attributes' );
    --
    store_display_group (
        p_name  => 'BREAK_FORMATTING',
        p_title => 'Break Formatting' );
    --
    store_display_group (
        p_name  => 'EXTERNAL_PROCESSING',
        p_title => 'External Processing' );
    --
    store_display_group (
        p_name  => 'MESSAGES',
        p_title => 'Messages' );
    --
    store_display_group (
        p_name  => 'HEADING',
        p_title => 'Heading' );
    --
    store_display_group (
        p_name  => 'UI_DEFAULTS_REFERENCE',
        p_title => 'UI Defaults Reference' );
    --
    store_display_group (
        p_name  => 'PRIMARY_KEY_SOURCE',
        p_title => 'Primary Key Source' );
    --
    store_display_group (
        p_name  => 'AUTOMATIC_REFRESH',
        p_title => 'Automatic Refresh' );
    --
    store_display_group (
        p_name  => 'LEGEND',
        p_title => 'Legend' );
    --
    store_display_group (
        p_name  => 'MARGIN',
        p_title => 'Margin' );
    --
    store_display_group (
        p_name  => 'SERIES_COLOR',
        p_title => 'Series Color' );
    --
    store_display_group (
        p_name  => 'X_AXIS',
        p_title => 'X Axis' );
    --
    store_display_group (
        p_name  => 'Y_AXIS',
        p_title => 'Y Axis' );
    --
    store_display_group (
        p_name  => 'GAUGE',
        p_title => 'Gauge' );
    --
    store_display_group (
        p_name  => 'VALUE',
        p_title => 'Value' );
    --
    store_display_group (
        p_name  => 'TOOLTIP',
        p_title => 'Tooltip' );
    --
    store_display_group (
        p_name  => 'TITLE',
        p_title => 'Title' );
    --
    store_display_group (
        p_name  => 'GANTT_TIMELINE',
        p_title => 'Timeline' );
    --
    store_display_group (
        p_name  => 'CHART',
        p_title => 'Chart' );
    --
    store_display_group (
        p_name  => 'DATAGRID',
        p_title => 'Datagrid' );
    --
    store_display_group (
        p_name  => 'MONTHLY_CALENDAR',
        p_title => 'Monthly Calendar' );
    --
    store_display_group (
        p_name  => 'WEEKLY_CALENDAR',
        p_title => 'Weekly Calendar' );
    --
    store_display_group (
        p_name  => 'DAILY_WEEKLY_CALENDAR',
        p_title => 'Daily & Weekly Calendar' );
    --
    store_display_group (
        p_name  => 'LIST_VIEW_CALENDAR',
        p_title => 'Agenda View Calendar' );
    --
    store_display_group (
        p_name  => 'COLORS',
        p_title => 'Colors' );
    --
    store_display_group (
        p_name  => 'DRAG_AND_DROP',
        p_title => 'Drag and Drop' );
    --
    store_display_group (
        p_name  => 'CALENDAR_STATE_ITEMS',
        p_title => 'Calendar State Items' );
    --
    store_display_group (
        p_name  => 'ENTRY',
        p_title => 'Entry' );
    --
    store_display_group (
        p_name  => 'CONTROLS',
        p_title => 'Controls' );
    --
    store_display_group (
        p_name  => 'COLUMN_FILTER',
        p_title => 'Column Filter' );
    --
    store_display_group (
        p_name  => 'PRINTING',
        p_title => 'Printing' );
    --
    store_display_group (
        p_name  => 'PAGE',
        p_title => 'Page' );
    --
    store_display_group (
        p_name  => 'COLUMN_HEADINGS',
        p_title => 'Column Headings' );
    --
    store_display_group (
        p_name  => 'COLUMNS',
        p_title => 'Columns' );
    --
    store_display_group (
        p_name  => 'PAGE_HEADER',
        p_title => 'Page Header' );
    --
    store_display_group (
        p_name  => 'PAGE_FOOTER',
        p_title => 'Page Footer' );
    --
    store_display_group (
        p_name  => 'OUTPUT',
        p_title => 'Output' );
    --
    store_display_group (
        p_name  => 'EDIT',
        p_title => 'Edit' );
    --
    store_display_group (
        p_name  => 'EDIT_AUTH',
        p_title => 'Edit Authorization' );
    --
    store_display_group (
        p_name  => 'TOOLBAR',
        p_title => 'Toolbar' );
    --
    store_display_group (
        p_name  => 'OVERVIEW',
        p_title => 'Overview' );
    --
    store_display_group (
        p_name  => 'DATA_CURSOR',
        p_title => 'Data Cursor' );
    --
    store_display_group (
        p_name  => 'COLUMN_MAPPING',
        p_title => 'Column Mapping' );
    --
    store_display_group (
        p_name  => 'LINE',
        p_title => 'Line' );
    --
    store_display_group (
        p_name  => 'MARKER',
        p_title => 'Marker' );
    --
    store_display_group (
        p_name  => 'MAJOR_TICKS',
        p_title => 'Major Ticks' );
    --
    store_display_group (
        p_name  => 'MINOR_TICKS',
        p_title => 'Minor Ticks' );
    --
    store_display_group (
        p_name  => 'TICK_LABEL',
        p_title => 'Tick Label' );
    --
    store_display_group (
        p_name  => 'DUAL_Y_AXES',
        p_title => 'Dual Y Axes' );
    --
    store_display_group (
        p_name  => 'ANIMATION',
        p_title => 'Animation' );
    --
    store_display_group (
        p_name  => 'MASTER_DETAIL',
        p_title => 'Master Detail' );
    --
    store_display_group (
        p_name  => 'REMOTE_CACHE',
        p_title => 'Remote Cache' );
    --
    store_display_group (
        p_name  => 'ZOOM_SCALE',
        p_title => 'Zoom Scale' );
    --
    store_display_group (
        p_name  => 'TIME_SCALE',
        p_title => 'Time Scale' );
    --
    store_display_group (
        p_name  => 'GANTT_START_DATE',
        p_title => 'Start Date' );
    --
    store_display_group (
        p_name  => 'GANTT_VIEWPORT',
        p_title => 'Viewport' );
    --
    store_display_group (
        p_name  => 'LOCAL_POST_PROCESSING',
        p_title => 'Local Post Processing' );
    --
    store_display_group (
        p_name  => 'MULTI_SERIES',
        p_title => 'Multi-Series Chart Data' );
    --
    store_display_group (
        p_name  => 'LIST_ENTRIES',
        p_title => 'List Entries' );
    --
    store_display_group (
        p_name  => 'DEPENDING_ON',
        p_title => 'Depending On' );
    --
    store_display_group (
        p_name  => 'CASCADING_LOV',
        p_title => 'Cascading List of Values' );
    --
    store_display_group (
        p_name  => 'QUERY',
        p_title => 'Query' );
    --
    store_display_group (
        p_name  => 'CARD',
        p_title => 'Card' );
    --
    store_display_group (
        p_name  => 'MAP_REGION',
        p_title => 'Map' );
    --
    store_display_group (
        p_name  => 'MAP_LAYER_ZOOM',
        p_title => 'Zoom Levels' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_INIT',
        p_title => 'Initial Position and Zoom' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_POINT_FEATURE_STYLE',
        p_title => 'Point Objects' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_POINT_CLUSTER',
        p_title => 'Point Clustering' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_ROW_SELECTION',
        p_title => 'Row Assignment' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_FEATURE_INFO_WINDOW',
        p_title => 'Info Window' );
    --
    store_display_group (
        p_name  => 'SUB_TITLE',
        p_title => 'Subtitle' );
    --
    store_display_group (
        p_name  => 'BODY',
        p_title => 'Body' );
    --
    store_display_group (
        p_name  => 'SECONDARY_BODY',
        p_title => 'Secondary Body' );
    --
    store_display_group (
        p_name  => 'ICON_AND_BADGE',
        p_title => 'Icon and Badge' );
    --
    store_display_group (
        p_name  => 'MEDIA',
        p_title => 'Media' );
    --
    store_display_group (
        p_name  => 'AUTHORIZATION_SCHEME',
        p_title => 'Authorization Scheme' );
    --
    store_display_group (
        p_name  => 'DATABASE_SESSION',
        p_title => 'Database Session' );
    --
    store_display_group (
        p_name  => 'NAME',
        p_title => 'Name' );
    --
    store_display_group (
        p_name  => 'SESSION',
        p_title => 'Session' );
    --
    store_display_group (
        p_name  => 'ADDITIONAL_CODE',
        p_title => 'Additional Code Execution' );
    --
    store_display_group (
        p_name  => 'CODE',
        p_title => 'Code' );
    --
    store_display_group (
        p_name  => 'AUTHENTICATION',
        p_title => 'Authentication' );
    --
    store_display_group (
        p_name  => 'ENDPOINT_URL',
        p_title => 'Endpoint URL' );
    --
    store_display_group (
        p_name  => 'HTML_FORMAT',
        p_title => 'HTML Format' );
    --
    store_display_group (
        p_name  => 'PLAIN_TEXT_FORMAT',
        p_title => 'Plain Text Format' );
    --
    store_display_group (
        p_name  => 'MESSAGE',
        p_title => 'Message' );
    --
    store_display_group (
        p_name  => 'SESSION_STATE',
        p_title => 'Session State' );
    --
    store_display_group (
        p_name  => 'STATUS',
        p_title => 'Status' );
    --
    store_display_group (
        p_name  => 'ICON',
        p_title => 'Icon' );
    --
    store_display_group (
        p_name  => 'IS_CURRENT',
        p_title => 'Is Current' );
    --
    store_display_group (
        p_name  => 'CLICK_COUNTING',
        p_title => 'Click Counting' );
    --
    store_display_group (
        p_name  => 'USER_DEFINED_ATTRIBUTES',
        p_title => 'User Defined Attributes' );
    --
    store_display_group (
        p_name  => 'GLOBALIZATION',
        p_title => 'Globalization' );
    --
    store_display_group (
        p_name  => 'ACTION_EXECUTION',
        p_title => 'Action Execution' );
    --
    store_display_group (
        p_name  => 'MULTIPLE_VALUES',
        p_title => 'Multiple Values' );
    --
    store_display_group (
        p_name  => 'CONFIRMATION',
        p_title => 'Confirmation' );
    --
    store_display_group (
        p_name  => 'SUGGESTIONS',
        p_title => 'Suggestions' );
  --
    store_display_group (
        p_name  => 'ACCESSIBILITY',
        p_title => 'Accessibility' );
    --
    store_display_group (
        p_name  => 'EXTERNAL_FILTER_ORDER_BY',
        p_title => 'External Filter and Order By' );
    --
    store_display_group (
        p_name  => 'SOURCE_ORDER_BY',
        p_title => 'Order By' );
    --
    store_display_group (
        p_name  => 'DATA_PROFILE',
        p_title => 'Data Profile' );
    --
    store_display_group (
        p_name  => 'PARSING',
        p_title => 'Parsing' );
    --
    store_display_group (
        p_name  => 'LOOKUP',
        p_title => 'Lookup' );
    --
    store_display_group (
        p_name  => 'TRANSFORMATION',
        p_title => 'Transformation' );
    --
    store_display_group (
        p_name  => 'REMOTE_REST',
        p_title => 'Remote REST Data Source Server' );
    --
    store_display_group (
        p_name  => 'TARGET',
        p_title => 'Target' );
    --
    store_display_group (
        p_name  => 'SYNCHRONIZATION',
        p_title => 'Synchronization' );
    --
    store_display_group (
        p_name  => 'SYNCHRONIZATION_RATE_LIMIT',
        p_title => 'Synchronization Rate Limit' );
    --
    store_display_group (
        p_name  => 'OPERATION',
        p_title => 'Operation' );
    --
    store_display_group (
        p_name  => 'ROW_SEARCH',
        p_title => 'Row Search' );
    --
    store_display_group (
        p_name  => 'AVAILABILITY',
        p_title => 'Availability' );
    --
    store_display_group (
        p_name  => 'REPORT_PRINTING',
        p_title => 'Report Printing' );
    --
    store_display_group (
        p_name  => 'SESSION_MANAGEMENT',
        p_title => 'Session Management' );
    --
    store_display_group (
        p_name  => 'SESSION_STATE_PROTECTION',
        p_title => 'Session State Protection' );
    --
    store_display_group (
        p_name  => 'APP_FORMAT_MASKS',
        p_title => 'Application Format Masks' );
    --
    store_display_group (
        p_name  => 'CHARACTER_VALUE_COMPARISON',
        p_title => 'Character Value Comparison' );
    --
    store_display_group (
        p_name  => 'LOGO',
        p_title => 'Logo' );
    --
    store_display_group (
        p_name  => 'NAVIGATION_BAR',
        p_title => 'Navigation Bar' );
    --
    store_display_group (
        p_name  => 'PWA',
        p_title => 'Progressive Web App' );
    --
    store_display_group (
        p_name  => 'INSTALLATION',
        p_title => 'Installation' );
    --
    store_display_group (
        p_name  => 'PREREQUISITES',
        p_title => 'Prerequisites' );
    --
    store_display_group (
        p_name  => 'INSTALL_MESSAGES',
        p_title => 'Installation Messages' );
    --
    store_display_group (
        p_name  => 'UPGRADE_MESSAGES',
        p_title => 'Upgrade Messages' );
    --
    store_display_group (
        p_name  => 'DEINSTALL_MESSAGES',
        p_title => 'Deinstallation Messages' );
    --
    store_display_group (
        p_name  => 'REST_SOURCE_CATALOG',
        p_title => 'REST Source Catalog' );
    --
    store_display_group (
        p_name  => 'TEMPLATES',
        p_title => 'Templates' );
    --
    --******************
    -- END DISPLAY GROUP
    --******************
    --
    ------------------------------------------------------------------------
    -- Error Messages
    ------------------------------------------------------------------------
    l_message_name := 'ERROR.APP_ID_EXISTS';
    store_msg (
        p_name         => l_message_name,
        p_message_text => 'Application with this ID already exists.' );
    --
    l_message_name := 'ERROR.TRANS_APP_ID_NO_ZERO';
    store_msg (
        p_name         => l_message_name,
        p_message_text => 'Translation Application must not end in zero.' );
    --
    --*******************
    -- END ERROR MESSAGES
    --*******************
    --
    ------------------------------------------------------------------------
    -- Properties
    ------------------------------------------------------------------------
    --
    l_property_id := 1;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'NAME',
        p_prompt             => 'Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'NAME',
        p_display_length     => 32,
        p_max_length         => 255,
        p_display_group_name => 'IDENTIFICATION',
        p_is_translatable    => true,
        p_help_text          => q'~Enter a descriptive name for this component.~',
        p_comment_text       => q'~$$$ Note: Only true because name used by PAGE is defined as translatable in old metadata! Check.~' );
    --
    l_property_id := 2;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUILD_OPTION_STATUS',
        p_prompt             => 'Status',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PATCH_STATUS',
        p_display_group_name => 'STATUS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select a build option status for this component.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Include',
        p_return_value  => 'INCLUDE',
        p_help_text     => q'~Use all components flagged with this build option.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Exclude',
        p_return_value  => 'EXCLUDE',
        p_help_text     => q'~Do not use any component flagged with this build option.~' );
    --
    l_property_id := 3;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DEFAULT_ON_EXPORT',
        p_prompt             => 'Default on Export',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DEFAULT_ON_EXPORT',
        p_display_group_name => 'STATUS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the default build option applied to this component when exporting. This option can be over ridden by export options.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Exclude',
        p_return_value  => 'EXCLUDE',
        p_help_text     => q'~Disabled and not a part of the application.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Include',
        p_return_value  => 'INCLUDE',
        p_help_text     => q'~Enabled and part of the application.~' );
    --
    l_property_id := 4;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMMENTS',
        p_prompt             => 'Comments',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'COMMENTS',
        p_display_length     => 50,
        p_max_length         => 4000,
        p_display_group_name => 'COMMENTS',
        p_help_text          => q'~Enter any developer comments or notes here. These comments only display within App Builder and never display when running the application.~' );
    --
    l_property_id := 5;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'AUTHORIZATION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'SCHEME_TYPE',
        p_display_group_name => 'AUTHORIZATION_SCHEME',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Specify how this authorization scheme will be applied. The following types are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may also have additional authorization type plug-ins available here.
These display as <em>&lt;My Plug-in&gt; [Plug-in]</em>.
The following plug-ins install into this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 6;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ERROR_MESSAGE',
        p_prompt             => 'Error Message',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'ERROR_MESSAGE',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_examples           => '#LABEL# must have some value.',
        p_display_group_name => 'ERROR',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Enter the text to be displayed in the event that the validation does not pass.</p>
<p>You can use <code>#LABEL#</code> as placeholder for the label of the associated page item or <code>#COLUMN_HEADER#</code> for the heading of the associated editable region column. The placeholders are not supported if error display location is set to On Error Page.</p>~' );
    --
    l_property_id := 7;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EVALUATION_POINT',
        p_prompt             => 'Evaluation Point',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'CACHING',
        p_display_group_name => 'SERVER_CACHE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Authorization schemes are evaluated on first use in a session.
Select the appropraite evaluation point to determine whether subsequent use of the given authorization obtains the result from memory or re-evaluates the authorization scheme.</p>
#LOV_VALUES#
<p>The default value <em>Once per session</em> is the most efficient.
You should choose another value if the authorization check depends on changing session state or other factors that are not consistent over an entire session.
The options <em>Once per component</em> and <em>Always (No Caching)</em> provide additional parameters that can be defined for use by the authorization scheme.
In authorization plugins, you can use the <em>component.type</em>, <em>component.id</em> and <em>component.name</em> attributes of the <em>t_authorization</em> parameter.
For the built-in <em>Exists SQL Query</em> and <em>PL/SQL Function Body Returning Boolean</em> types, equivalent bind variables <em>:APP_COMPONENT_TYPE</em>, <em>:APP_COMPONENT_ID</em> and <em>:APP_COMPONENT_NAME</em> are available.
</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once per Session',
        p_return_value  => 'BY_USER_BY_SESSION',
        p_help_text     => q'~Evaluates only once and then uses the result in memory.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once per Page View',
        p_return_value  => 'BY_USER_BY_PAGE_VIEW',
        p_help_text     => q'~Evaluates once for each request that is processed. Use the result in memory if the authorization scheme is referenced in more than one component on the page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once per Component',
        p_return_value  => 'BY_COMPONENT',
        p_help_text     => q'~Evaluates once for each component that references the authorization scheme, but saves the result in the session. Further requests to render or process the scheme use the result in memory.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Always (No Caching)',
        p_return_value  => 'NOCACHE',
        p_help_text     => q'~The authorization scheme is always evaluated.~' );
    --
    l_property_id := 8;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'ALIAS',
        p_prompt                => 'Alias',
        p_property_type         => 'TEXT',
        p_db_column             => 'ALIAS',
        p_display_length        => 30,
        p_max_length            => 30,
        p_text_case             => 'LOWER',
        p_restricted_characters => 'SAFE_COMPONENT_NAME_NO_NUMBER',
        p_display_group_name    => 'IDENTIFICATION',
        p_help_text             => q'~Enter an alphanumeric alias for this page. You may use this alias to reference the page instead of the page number.
For example, if you were working on page 1 of application 100, you could create an alias called <code>home</code>.
You could then access this page from other pages using the following f?p syntax:
<pre>f?p=100:home</pre>~' );
    --
    l_property_id := 9;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'PAGE_GROUP',
        p_prompt                => 'Page Group',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'GROUP_ID',
        p_display_group_name    => 'IDENTIFICATION',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_page_group,
        p_help_text             => q'~Specify the page group to associate with this page.
Page groups do not affect functionality, but help developers manage the pages within an application.~' );
    --
    l_property_id := 10;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_MODE',
        p_prompt             => 'Page Mode',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PAGE_MODE',
        p_display_group_name => 'APPEARANCE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select the page mode for this page.</p>
#LOV_VALUES#
<p>Note that options <em>Modal</em> and <em>Non-Modal Dialog</em> will only be available for selection where the current application theme contains a default <em>Dialog Page Template</em>.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Normal',
        p_return_value  => 'NORMAL',
        p_help_text     => q'~The page is presented as a normal &PRODUCT_NAME. application page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Modal Dialog',
        p_return_value  => 'MODAL',
        p_help_text     => q'~The page is presented as a modal dialog.<br>
A modal dialog is an overlay window positioned within the viewport, which remains active and focused until the end user dismisses (closes) it.
The underlying page is grayed out and the end user is prevented from interacting with the rest of the page until the dialog is closed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Non-Modal Dialog',
        p_return_value  => 'NON_MODAL',
        p_help_text     => q'~The page is presented as a non-modal dialog.<br>
A non-modal dialog is an overlay window positioned within the viewport, where the end user can interact with the dialog, and content on the page that launched the dialog.
A non-modal dialog can be used when the requested information is not essential to continue, work can continue elsewhere while the window is still open.~' );
    --
    l_property_id := 11;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'PAGE_TEMPLATE',
        p_prompt                   => 'Page Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'STEP_TEMPLATE',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_page_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~<p>Select a page template to define the appearance and layout of this page.</p>
        <p>Page templates are defined in the application theme.
        When a page is created, the template is automatically set to <em>Theme Default</em> which will render the page using the default page template defined in the current theme.
        Making a selection here overrides the default page template.</p>
        <p>Many themes contain page templates with specific purposes, such as for displaying content with side columns, login page, dialog pages, and so forth.<br>
        Universal Theme provides the following page templates:
          <dl>
            <dt>Standard</dt>
              <dd>The default page template. Ideal for home pages, dashboards, large reports, and other layouts where maximum screen real-estate is desired.</dd>
            <dt>Left Side Column</dt>
              <dd>Provides a left display position that is useful for search and filter pages.</dd>
            <dt>Right Side Column</dt>
              <dd>Provides a collapsible right-side display position that is useful for displaying action-oriented controls such as buttons or lists.</dd>
            <dt>Left and Right Side Columns</dt>
              <dd>Provides both the left side column and collapsible right-side column that is well suited for very complex pages.</dd>
            <dt>Login</dt>
              <dd>Used exclusively for the application login page.</dd>
            <dt>Marquee</dt>
              <dd>Provides a collapsible right side column and a title-bar area which contains primary information.
              This page template is well suited for displaying a single record which has several related reports or data.
              Marquee is commonly used for displaying Master-Detail relationships where a single master record is first selected from a report. </dd>
            <dt>Minimal (No Navigation)</dt>
              <dd>Useful for very simple pages, disclaimers, single page applications, or where navigation is not necessary.</dd>
          </dl>
          See Also: <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a>
        </p>~' );
    --
    l_property_id := 12;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'STANDARD_TAB_SET',
        p_prompt                => 'Standard Tab Set',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'TAB_SET',
        p_display_group_name    => 'TABS',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_tab_set,
        p_help_text             => q'~Select a standard tab set used for this page. A standard tab set is associated with a specific page number. You can use standard tabs to link users to a specific page.~' );
    --
    l_property_id := 13;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'TITLE',
        p_prompt             => 'Title',
        p_property_type      => 'TEXT',
        p_db_column          => 'TITLE',
        p_display_length     => 70,
        p_max_length         => 255,
        p_display_group_name => 'IDENTIFICATION',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the page title. The page title is displayed in the title bar of the browser window.~' );
    --
    l_property_id := 14;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CURSOR_FOCUS',
        p_prompt             => 'Cursor Focus',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'FIRST_ITEM',
        p_supported_ui_types => 'DESKTOP',
        p_display_group_name => 'NAVIGATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify whether the cursor is focussed when first entering the page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'First item on page',
        p_return_value  => 'AUTO_FIRST_ITEM',
        p_help_text     => q'~The cursor focus is placed on the first field on the page using JavaScript, generated by the &PRODUCT_NAME. engine.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Do not focus cursor',
        p_return_value  => 'NO_FIRST_ITEM',
        p_help_text     => q'~The cursor is not focussed and no JavaScript is generated.<br>
Select this option if the generated JavaScript conflicts with custom JavaScript.~' );
    --
    l_property_id := 15;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_FILE_URLS',
        p_prompt             => 'File URLs',
        p_property_type      => 'FILE URLS JAVASCRIPT',
        p_db_column          => 'JAVASCRIPT_FILE_URLS',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>Standard file reference</dt>
<dd><pre>/myjs/main.js</pre></dd>
<dt>Standard file reference which loads the minified file <code>main.min.js</code> from <code>/myjs/minified/</code> for regular page views and <code>main.js</code> from <code>/myjs/</code> in debug mode or when running the application from App Builder.</dt>
<dd><pre>/myjs/#MIN_DIRECTORY#main#MIN#.js</pre></dd>
<dt>Use this syntax if the library is also used from a RequireJS require call. The define name must match what is used in the requirejs.config paths and the value must be the global symbol exposed by the library. Only applies when using RequireJS. This avoids RequireJS  loading that library a second time.</dt>
<dd><pre>[define hammerjs=Hammer]/myjs/hammer.js</pre></dd>
<dt>Use this syntax if the file has a dependency on RequireJS. This will automatically load RequireJS before this file.</dt>
<dd><pre>[require requirejs]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file has a dependency on Oracle JET. This will automatically load RequireJS and the Oracle JET RequireJS configuration before this file.</dt>
<dd><pre>[require jet]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file is a JavaScript ES Module. It will add <code>type="module"</code> to the script tag.</dt>
<dd><pre>[module]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file should be loaded asynchronously. It will add attribute <code>async</code> to the script tag.</dt>
<dd><pre>[async]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file should only be executed after the document was been parsed. It will add attribute <code>defer</code> to the script tag. Note that modules are deferred by default.</dt>
<dd><pre>[defer]/myjs/main.js</pre></dd>
<dt>Multiple File URL directives can be provided but must be comma separated.</dt>
<dd><pre>[module, async]/myjs/main.js</pre></dd>
<dt>File reference using a query string in the URL referencing the application version, such that when the application is upgraded (and the version is incremented), the browser is instructed to always load the new file and ignore any previously cached files.</dt>
<dd><pre>/myjs/main.js?version=#APP_VERSION#</pre></dd>
</dl>~',
        p_display_group_name => 'JAVASCRIPT',
        p_help_text          => q'~<p>Enter JavaScript file URLs for code loaded on this page. Each URL must be written on a new line.
If you provide a minified version of your file you can use the substitution string <code>#MIN#</code> to include <code>.min</code> or <code>#MIN_DIRECTORY#</code> to include <code>minified/</code> in your file URL for a regular page view and an empty string if the page is viewed in debug mode or when running the application from App Builder.
You also have access to the substitution string <code>#APP_VERSION#</code> if you want to include the application's version in the file URL.</p>
<p>JavaScript file URLs you enter here replaces the <code>#PAGE_JAVASCRIPT#</code> substitution string in the page template.</p>
<p>Each file URL can be prefixed with a pair of brackets (<code>[...]</code>) containing one or more of the following directives. See the examples for details.</p>
<p><ul>
    <li>module</li>
    <li>defer</li>
    <li>async</li>
    <li>define</li>
    <li>require requirejs</li>
    <li>require jet</li>
</ul></p>
<p>Note: You do not need to include opening or closing <code>script</code> tags. Just specify the URL.</p>
<p>To comment out a File URL, prefix the line with two dashes: <code>--</code>. To comment out multiple lines, use the typical code block syntax: <code>/* */</code>.</p>~' );
    --
    l_property_id := 16;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_FUNCTION_AND_GLOBAL_VARIABLE_DECLARATION',
        p_prompt             => 'Function and Global Variable Declaration',
        p_property_type      => 'JAVASCRIPT',
        p_db_column          => 'JAVASCRIPT_CODE',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'JAVASCRIPT',
        p_help_text          => q'~<p>Enter JavaScript code such as functions or global variable declarations referenced on this page.
To reference a shortcut, use <code>"SHORTCUTNAME"</code>.
If the same JavaScript is required on multiple pages, consider putting the code into an external JavaScript file to avoid duplication and to leverage browser caching of static files.</p>
<p>Code you enter here replaces the <code>#PAGE_JAVASCRIPT#</code> substitution string in the page template.</p>
<p>Note: You do not need to include opening or closing <code>script</code> tags. Just include the JavaScript code.</p>~' );
    --
    l_property_id := 17;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_EXECUTE_WHEN_PAGE_LOADS',
        p_prompt             => 'Execute when Page Loads',
        p_property_type      => 'JAVASCRIPT',
        p_db_column          => 'JAVASCRIPT_CODE_ONLOAD',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'JAVASCRIPT',
        p_help_text          => q'~<p>Enter JavaScript code to execute when the page loads.
The code is executed after the JavaScript code generated by &PRODUCT_NAME..</p>
<p>Code you enter here replaces the <code>#GENERATED_JAVASCRIPT#</code> substitution string in the page template.</p>
<p>Note: You do not need to add any opening or closing <code>script</code> tags, just the JavaScript code.</p>~' );
    --
    l_property_id := 18;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INCLUDE_STANDARD_JAVASCRIPT_AND_CSS',
        p_prompt             => 'Include Standard JavaScript and CSS',
        p_property_type      => 'YES NO',
        p_db_column          => 'INCLUDE_APEX_CSS_JS_YN',
        p_display_group_name => 'JAVASCRIPT',
        p_deprecated_values  => 'N',
        p_help_text          => q'~<p>&PRODUCT_NAME. typically requires specific JavaScript and CSS files.
For some applications, such as mobile applications, you may want to suppress the inclusion of these files on the page.</p>
<p>To Include the standard &PRODUCT_NAME. and JavaScript files select <strong>On</strong>. Selecting <strong>Off</strong> suppresses the inclusion of these files in the HTML Header.</p>
<p>Suppressing inclusion of these files can have adverse effects on an application and may cause it to break.
Therefore, disabling this attribute is only recommended for advanced developers.</p>~' );
    --
    l_property_id := 19;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HTML_HEADER',
        p_prompt             => 'HTML Header',
        p_property_type      => 'HTML',
        p_db_column          => 'HTML_PAGE_HEADER',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'HTML_HEADER',
        p_help_text          => q'~<p>Enter text to replace the <code>#HEAD#</code> substitution string in the page template header.
The values entered here are inserted after the HTML <code>&lt;head&gt;</code> tag. Use this page attribute to add page specific meta tags.
</p>
<p>Note: JavaScript code or Cascading Style Sheets should be added to the dedicated JavaScript and CSS attributes.</p>~' );
    --
    l_property_id := 20;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_HTML_BODY_ATTRIBUTE',
        p_prompt             => 'Page HTML Body Attribute',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'HTML_PAGE_ONLOAD',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HTML_HEADER',
        p_help_text          => q'~Enter text substituted in for your page template's <code>#ONLOAD#</code> substitution string.
To use this feature, your page template must include the <code>#ONLOAD#</code> substitution string.~' );
    --
    l_property_id := 21;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HEADER_TEXT',
        p_prompt             => 'Header Text',
        p_property_type      => 'HTML',
        p_db_column          => 'WELCOME_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HEADER_AND_FOOTER',
        p_is_translatable    => true,
        p_help_text          => q'~Enter text or HTML you want to appear immediately following the page header and before the body content.~' );
    --
    l_property_id := 22;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BODY_HEADER',
        p_prompt             => 'Body Header',
        p_property_type      => 'HTML',
        p_db_column          => 'BOX_WELCOME_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HEADER_AND_FOOTER',
        p_is_translatable    => true,
        p_deprecated_values  => '$NOT_NULL$',
        p_help_text          => q'~Enter text or HTML you want to appear before regions that display on the running page.
Text you enter here appears before the page template <code>#BODY#</code> substitution string.~' );
    --
    l_property_id := 23;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FOOTER_TEXT',
        p_prompt             => 'Footer Text',
        p_property_type      => 'HTML',
        p_db_column          => 'FOOTER_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HEADER_AND_FOOTER',
        p_is_translatable    => true,
        p_help_text          => q'~Enter text or HTML that appears before the page template footer.~' );
    --
    l_property_id := 24;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'AUTHORIZATION_SCHEME',
        p_prompt                => 'Authorization Scheme',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'REQUIRED_ROLE',
        p_display_group_name    => 'SECURITY',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_authorization,
        p_help_text             => q'~Select an authorization scheme to control this component. The authorization must evaluate to TRUE in order for this component to be rendered or executed.~' );
    --
    l_property_id := 25;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_AUTHENTICATION',
        p_prompt             => 'Authentication',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PAGE_IS_PUBLIC_Y_N',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify whether this page is public or requires authentication.
If a page is identified as public, the page can be viewed by any user who accesses the page.
The application's page sentry function may access this page attribute to identify pages that do not require prior authentication to view.
The implementation of the authentication scheme's page sentry function determines whether this attribute has any effect.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page Requires Authentication',
        p_return_value  => 'N' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page Is Public',
        p_return_value  => 'Y' );
    --
    l_property_id := 26;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DEEP_LINKING',
        p_prompt             => 'Deep Linking',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DEEP_LINKING',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how external URL requests to a page are handled.
Page requests from within this application to this page, are not considered deep linking, and are not governed by this attribute's setting.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Enabled',
        p_return_value  => 'Y',
        p_help_text     => q'~If this page is specified as the target page in a URL, it ultimately redirects to this page.<br>
If the user is not already authenticated and the page is not public, then the user is prompted to login.
After successfully logging into the application, the user is redirected to this page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Disabled',
        p_return_value  => 'N',
        p_help_text     => q'~If the URL does not contain a valid session ID, &PRODUCT_NAME. starts a new session and redirects the user to the application home page, rather than to this page.~' );
    --
    l_property_id := 27;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_ACCESS_PROTECTION',
        p_prompt             => 'Page Access Protection',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PROTECTION_LEVEL',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select what type of session state protection is implemented for this page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Unrestricted',
        p_return_value  => 'N',
        p_help_text     => q'~The page may be requested using a URL, with or without session state arguments, and without having to have a checksum.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Arguments Must Have Checksum',
        p_return_value  => 'C',
        p_help_text     => q'~If Request, Clear Cache, or Name/Value Pair arguments appear in the URL, a checksum must also be provided.<br>
The checksum type must be compatible with the most stringent <em>Session State Protection</em> attribute of all the items passed as arguments.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'No Arguments Supported',
        p_return_value  => 'U',
        p_help_text     => q'~A URL may be used to request the page, but the URL can not contain Request, Clear Cache, or Name/Value Pair arguments.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'No URL Access',
        p_return_value  => 'D',
        p_help_text     => q'~The page may not be accessed using a URL.<br>
However, the page may be the target of a <em>Branch to Page</em> branch type, as this does not perform a URL redirect.~' );
    --
    l_property_id := 28;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_DUPLICATE_PAGE_SUBMISSIONS',
        p_prompt             => 'Enable duplicate page submissions',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'ALLOW_DUPLICATE_SUBMISSIONS',
        p_display_group_name => 'ADVANCED',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select whether the end user may submit the same page more than once.
Use this feature to avoid duplicate inserts into the same table.
This can happen when an end user clicks the browser's back button and then submits the page again, or if the end user clicks the browser's reload button.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'No - Prevent page from being re-posted',
        p_return_value  => 'N' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Yes - Enable page to be re-posted',
        p_return_value  => 'Y' );
    --
    l_property_id := 29;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ON_DUPLICATE_PAGE_SUBMISSIONS_GO_TO_THIS_URL',
        p_prompt             => 'On duplicate page submissions go to this URL',
        p_property_type      => 'TEXT',
        p_db_column          => 'ON_DUP_SUBMISSION_GOTO_URL',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~Enter a URL target for when an end user attempts to perform a second page submission. &PRODUCT_NAME. displays an error message and a link to this URL to enable the user to continue.~' );
    --
    l_property_id := 30;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FORM_AUTO_COMPLETE',
        p_prompt             => 'Form Auto Complete',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'AUTOCOMPLETE_ON_OFF',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify whether auto-complete is disabled for this page. A value of <strong>On</strong> has no effect on this page. A value of <strong>Off</strong> generates the following HTML in the FORM tag:
<pre>
autocomplete="off"
</pre>
<p>Note: Different browsers may have different rules around when <code>autocomplete="off"</code> is respected or ignored. Therefore please bear this in mind specifically when setting this to <strong>Off</strong>, and test in your target browsers.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'On',
        p_return_value  => 'ON' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Off',
        p_return_value  => 'OFF' );
    --
    l_property_id := 32;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CACHE_TIMEOUT',
        p_prompt             => 'Cache Timeout',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'CACHE_TIMEOUT_SECONDS',
        p_display_group_name => 'SERVER_CACHE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how long a cached page is valid. Subsequent page requests do not utilize the cached page if the timeout expires.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '10 seconds',
        p_return_value  => '10' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 minute',
        p_return_value  => '60',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '10 minutes',
        p_return_value  => '600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '30 minutes',
        p_return_value  => '1800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 hour',
        p_return_value  => '3600',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2 hours',
        p_return_value  => '7200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '3 hours',
        p_return_value  => '10800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4 hours',
        p_return_value  => '14400' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '6 hours',
        p_return_value  => '21600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '12 hours',
        p_return_value  => '43200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 day',
        p_return_value  => '86400',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2 days',
        p_return_value  => '172800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '3 days',
        p_return_value  => '259200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4 days',
        p_return_value  => '345600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 week',
        p_return_value  => '604800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2 weeks',
        p_return_value  => '1209600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4 Weeks',
        p_return_value  => '2419200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '10 Weeks',
        p_return_value  => '6048000' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 Year',
        p_return_value  => '31536000' );
    --
    l_property_id := 33;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BROWSER_CACHE',
        p_prompt             => 'Browser Cache',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'BROWSER_CACHE',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>If the cache is enabled, the browser can save the contents of your application's pages in its cache, both in memory and on disk.
If a user clicks the browser back button, the page typically loads from the cache, not from the server.
If the cache is disabled, the browser is instructed to not save page content and requests it anew from the server.</p>
<p>Oracle recommends, from a security perspective, that the cache is disabled, so the browser does not store sensitive data and always requests pages if the URL changes.
Otherwise, it may even be possible to go back in the browser history after a log out and see cached content of the former session.
Disabling the browser cache prevents subtle back button issues with pages that use partial page refresh, such as interactive reports.</p>
<p>If this attribute is set to <strong>Disabled</strong>, &PRODUCT_NAME. sends the HTTP header <em>cache-control: no-store</em> which instructs the Browser to not cache the page contents on disk or in memory.
If it is set to <strong>Application Default</strong>, the value at runtime is taken from the application's <em>Browser Cache</em> attribute.</p>
<p>Note: This feature requires modern browsers that support the HTTP header response variable <code>cache-control</code>.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Enabled',
        p_return_value  => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Disabled',
        p_return_value  => 'N' );
    --
    l_property_id := 38;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'BUILD_OPTION',
        p_prompt                => 'Build Option',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'REQUIRED_PATCH',
        p_display_group_name    => 'CONFIGURATION',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_build_option,
        p_help_text             => q'~<p>Select a build option for this component.</p>
                                     <p>Build Options allow application components and functionality to be conditionally enabled or disabled when the application is installed, or at runtime via API (APEX_UTIL.SET_BUILD_OPTION_STATUS).<br>
                                       Build Options can set this component to be Included or Excluded in the runtime application:</p>
                                     <p><strong>Included</strong> - Enabled and are part of the application.</p>
                                     <p><strong>Excluded</strong> - Disabled and are not a part of the application.</p>~',
        p_examples              => q'~<p>1. You are developing an application that will have many installations.
                                        Additionally, each installation may choose to enable or disable certain application functionality.
                                        With Build Options, you can develop one application and when installed, build options can be configured to include or exclude associated application functionality.
                                        You may also use the API to expose this feature within the application.
                                        This allows privileged end-users to selectively enable or disable application functionality.</p>
                                      <p>2. You are developing functionality which is not yet complete, so you do not wish to expose to end-users.
                                        Using Build Options enables you to disable this functionality from the application when it is built into the test and production environments.
                                        The benefit of this approach is that the application can still be deployed without including incomplete functionality.</p>
                                      <p>3. You are removing functionality you think is not needed by the application, but are unsure this is a safe operation.
                                        In this case, you can use Build Options to essentially "comment out" specific functionality.
                                        If on deployment, the removed functionality was deemed to be necessary, it can be re-instated.
                                        If this functionality was deemed to be unnecessary, then it can be safely removed from the application.</p>~' );
    --
    l_property_id := 39;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INLINE_ERROR_NOTIFICATION_TEXT',
        p_prompt             => 'In-line Error Notification Text',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'ERROR_NOTIFICATION_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'ERROR_HANDLING',
        p_help_text          => q'~Enter the error message displayed when an error occurs on this page. The error message is substituted for <code>#NOTIFICATION_MESSAGE#</code> in the page template.~' );
    --
    l_property_id := 40;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HELP_TEXT',
        p_prompt             => 'Help Text',
        p_property_type      => 'HTML',
        p_db_column          => 'HELP_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HELP',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Use this attribute to enter help text for the current page. Page level help supports shortcuts using the following syntax:</p>
<pre>"SHORTCUT_NAME"</pre>
<p>Page help text is not displayed by default. However, you can readily develop a help system that exposes this text.</p>~' );
    --
    l_property_id := 41;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ERROR_HANDLING_FUNCTION',
        p_prompt             => 'Error Handling Function',
        p_property_type      => 'TEXT', -- $$$ for Advisor we should have a new property_type FUNCTION_NAME/PROCEDURE_NAME
        p_db_column          => 'ERROR_HANDLING_FUNCTION',
        p_display_length     => 78,
        p_max_length         => 255,
        p_display_group_name => 'ERROR_HANDLING',
        p_help_text          => q'~<p>Enter the name of the PL/SQL function called when an error occurs within the application.
You can use this function to act on various error codes including those emanating from the Oracle database, and log such errors in a custom table you define if necessary.
The function can include logic to modify the error message and display location, transforming Oracle database errors into user friendly messages.
For example, you can capture a unique key violation on <code>DEMO_CUSTOMERS</code> and replace the <code>ORA-00001</code> message with this text: Customer Name must be unique.
Installing the <strong>Poll</strong> Starter App, from the Gallery, creates the <code>EBA_QPOLL_FW</code> package, which includes the <code>apex_error_handling</code> function.
This error handling function includes extensive comments to make it easy for you to utilize as the basis for defining your own function.</p>
<p>The function name entered can reference a package function or stand alone function in the database. For example:
<pre>
log_apex_error
</pre>
</p>
<p>When referencing a database PL/SQL package or stand alone function, you can use the <code>#OWNER#</code> substitution string to reference the parsing schema of the current application. For example:
<pre>#OWNER#.log_apex_error</pre>
</p>
<p>If an error handling function is specified on a page then it is utilized, instead of this application level setting, for that page.</p>
<p>Error Handling Functions must implement the following interface:
<pre>
function &lt;name of function> (
    p_error in apex_error.t_error )
    return apex_error.t_error_result
</pre>
</p>
<p>See the API Reference manual for a description of the package <code>apex_error</code>.</p>~' );
    --
    l_property_id := 42;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_TRANSITION',
        p_prompt             => 'Page Transition',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PAGE_TRANSITION',
        p_supported_ui_types => 'JQM_SMARTPHONE',
        p_display_group_name => 'NAVIGATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Specify the transition used when navigating to a new page or when the page is getting submitted.</p>
<p>Older browsers, that do not support 3D transforms, are not able to utilize most of the transitions.
By default, devices that lack 3D support (such as Android 2.x) fallback to fade for all transition types.
See jQuery Mobile documentation for additional details.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'None',
        p_return_value  => 'NONE',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Fade',
        p_return_value  => 'FADE',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Flip',
        p_return_value  => 'FLIP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Flow',
        p_return_value  => 'FLOW' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Pop',
        p_return_value  => 'POP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide',
        p_return_value  => 'SLIDE',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide Down',
        p_return_value  => 'SLIDEDOWN' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide Fade',
        p_return_value  => 'SLIDEFADE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide Up',
        p_return_value  => 'SLIDEUP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Turn',
        p_return_value  => 'TURN' );
    --
    l_property_id := 43;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'POPUP_TRANSITION',
        p_prompt             => 'Popup Transition',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'POPUP_TRANSITION',
        p_supported_ui_types => 'JQM_SMARTPHONE',
        p_display_group_name => 'NAVIGATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Specify the transition used when a popup is opened.</p>
<p>Older browsers, that do not support 3D transforms, are not able to utilize most of the transitions.
By default, devices that lack 3D support (such as Android 2.x) fallback to fade for all transition types.
See jQuery Mobile documentation for additional details.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'None',
        p_return_value  => 'NONE',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Fade',
        p_return_value  => 'FADE',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Flip',
        p_return_value  => 'FLIP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Flow',
        p_return_value  => 'FLOW' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Pop',
        p_return_value  => 'POP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide',
        p_return_value  => 'SLIDE',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide Down',
        p_return_value  => 'SLIDEDOWN' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide Fade',
        p_return_value  => 'SLIDEFADE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Slide Up',
        p_return_value  => 'SLIDEUP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Turn',
        p_return_value  => 'TURN' );
    --
    l_property_id := 44;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'MEDIA_TYPE',
        p_prompt             => 'Media Type',
        p_property_type      => 'TEXT',
        p_db_column          => 'MEDIA_TYPE',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'APPEARANCE',
        p_help_text          => q'~<p>Enter the Internet media type. An Internet media type is a two part identifier for file formats on the Internet.
A media type is composed of at least two parts: a type, a subtype, and one or more optional parameters.
This media type is used in the Content-Type HTTP header when rendering the page.</p>
<p>The page-level Media type overrides the application-level media type.
If both the page-level and application-level values for media type are <code>NULL</code>, the media type <code>text/html</code> is used.</p>~',
        p_comment_text       => q'~$$$ Should this property be deprecated???~' );
    --
    l_property_id := 45;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CSS_FILE_URLS',
        p_prompt             => 'File URLs',
        p_property_type      => 'FILE URLS CSS',
        p_db_column          => 'CSS_FILE_URLS',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>Standard file reference</dt>
<dd><pre>/mycss/main.css</pre></dd>
<dt>Standard file reference which loads the minified file main.min.css from /mycss/minified/ for regular page views and main.css from /mycss/ in debug mode or when running the application from App Builder.</dt>
<dd><pre>/mycss/#MIN_DIRECTORY#main#MIN#.css</pre></dd>
<dt>Conditional media query</dt>
<dd><pre>[media="only screen and (max-device-width: 480px)"]/mycss/smartphone.css</pre></dd>
<dt>File reference using a query string in the URL referencing the application version, such that when the application is upgraded (and the version is incremented), the browser is instructed to always load the new file and ignore any previously cached files.</dt>
<dd><pre>/mycss/main.css?version=#APP_VERSION#</pre></dd>
</dl>~',
        p_display_group_name => 'CSS',
        p_help_text          => q'~<p>Enter Cascading Style Sheet (CSS) file URLs loaded on this page. Each URL must be written on a new line.
If you provide a minified version of your file you can use the substitution string <code>#MIN#</code> to include <code>.min</code> or <code>#MIN_DIRECTORY#</code> to include <code>minified/</code> in your file URL for a regular page view and an empty string if the page is viewed in debug mode or when running the application from App Builder.
You also have access to the substitution string <code>#APP_VERSION#</code> if you want to include the application's version in the file URL.</p>
<p>File URLs you enter here replace the <code>#PAGE_CSS#</code> substitution string in the page template.</p>
<p>Each file URL can optionally be prefixed with <code>[media ...]</code> for conditional loading. See the examples for details.</p>
<p>Note: You do not need to include opening or closing <code>link</code> tags. Just include the file URL.</p>
<p>To comment out a File URL, prefix the line with two dashes: <code>--</code>. To comment out multiple lines, use the typical code block syntax: <code>/* */</code>.</p>~' );
    --
    l_property_id := 46;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INLINE_CSS',
        p_prompt             => 'Inline',
        p_property_type      => 'CSS',
        p_db_column          => 'INLINE_CSS',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_examples           => q'~<dl>
<dt>Sets the CSS class <code>display_only</code> to bold</dt>
<dd><pre>
.display_only {
  font-weight: bold;
}
</pre></dd>
</dl>~',
        p_display_group_name => 'CSS',
        p_help_text          => q'~<p>Enter Cascading Style Sheet (CSS) code used on this page.
If the same CSS is required on multiple pages, consider putting the code into an external file to avoid duplication and to leverage browser caching of static files.</p>
<p>Code you enter here replaces the <code>#PAGE_CSS#</code> substitution string in the page template.</p>
<p>Note: You do not need to include opening or closing <code>style</code> tags. Just include the cascading style sheet code.</p>~' );
    --
    l_property_id := 47;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DIALOG_WIDTH',
        p_prompt             => 'Width',
        p_property_type      => 'TEXT',
        p_db_column          => 'DIALOG_WIDTH',
        p_display_length     => 10,
        p_max_length         => 20,
        p_display_group_name => 'DIALOG',
        p_help_text          => q'~<p>Enter the width of the dialog, in pixels or a percentage.
The substitution string <code>#DIALOG_WIDTH#</code> in the dialog initialization code of the page template is replaced with the value entered in this item.</p>
<p>For a page using a Desktop user interface, the value is set in pixels, such as <strong><em>500</em></strong>.
However, for a page using the jQuery Mobile Smartphone user interface, the value is set in percent, such as <strong><em>80%</em></strong>.</p>~' );
    --
    l_property_id := 48;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DIALOG_MAX_WIDTH',
        p_prompt             => 'Maximum Width',
        p_property_type      => 'TEXT',
        p_db_column          => 'DIALOG_MAX_WIDTH',
        p_display_length     => 10,
        p_max_length         => 20,
        p_display_group_name => 'DIALOG',
        p_help_text          => q'~<p>Enter the maximum width of the dialog in pixels.
The substitution string <code>#DIALOG_MAX_WIDTH#</code> in the dialog initialization code of the page template is replaced with the value entered in this item.</p>
<p>For a page using the jQuery Mobile Smartphone user interface, the maximum width is <strong><em>500</em></strong>.</p>~' );
    --
    l_property_id := 49;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'DIALOG_ATTRIBUTES',
        p_prompt                => 'Attributes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ',',
        p_db_column             => 'DIALOG_ATTRIBUTES',
        p_display_length        => 100,
        p_max_length            => 255,
        p_display_group_name    => 'DIALOG',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~<p>Dialog support in &PRODUCT_NAME. varies depending on the user interface of the page.
A subset of dialog attributes are declaratively supported: <em>height, width, and maximum width</em>.
To define any additional attributes for the dialog, enter the name/value pairings in a comma separated list.</p>
<p>The substitution string <code>#DIALOG_ATTRIBUTES#</code> in the dialog initialization code defined at page template level, is replaced by any value(s) entered in this item.</p>
<p>For pages using Desktop user interface, the jQuery UI dialog widget is utilized.
For pages using jQuery Mobile Smartphone user interface, the jQuery Mobile dialog widget is utilized.</p>
<p>For example, a dialog page using a Desktop user interface could have the following attributes defined: <code>resizable:true, draggable:true</code></p>
<p>For further information on the attributes available, refer to the <a rel="noreferrer" target="_blank" href="http://api.jqueryui.com/dialog/">jQuery UI dialog widget documentation</a> and <a rel="noreferrer" target="_blank" href="http://api.jqueryui.com/dialog/">jQuery Mobile dialog widget documentation</a>.</p>~' );
    --
    l_property_id := 50;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DESCRIPTION',
        p_prompt             => 'Description',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'DESCRIPTION',
        p_display_length     => 60,
        p_max_length         => 4000,
        p_display_group_name => 'DESCRIPTION',
        p_help_text          => q'~Enter a description.~' );
    --
    l_property_id := 51;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'COMPUTATION_ITEM_NAME',
        p_prompt                   => 'Item Name',
        p_property_type            => 'ITEM',
        p_db_column                => 'COMPUTATION_ITEM',
        p_display_length           => 50,
        p_max_length               => 255,
        p_display_group_name       => 'IDENTIFICATION',
        p_lov_component_type_scope => 'APPLICATION',
        p_help_text                => q'~Enter the page or application item populated by this computation. You can type in the name or pick from the list of available items.~' );
    --
    l_property_id := 52;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'COMPUTATION_TYPE',
        p_display_group_name => 'COMPUTATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select a computation type from the following list.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static Value',
        p_return_value  => 'STATIC_ASSIGNMENT',
        p_help_text     => q'~Set to the text entered into <em>Static Value</em>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item',
        p_return_value  => 'ITEM_VALUE',
        p_help_text     => q'~Set to the value held in session state for the <em>Item</em> selected.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return single value)',
        p_return_value  => 'QUERY',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the first value returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return colon separated value)',
        p_return_value  => 'QUERY_COLON',
        p_help_text     => q'~Set to a single colon separated value, which concatenates each row value, returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Expression',
        p_return_value  => 'EXPRESSION',
        p_help_text     => q'~Set to the value that results from executing the <em>Expression</em> entered.~'  );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the value returned from the <em>Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Preference',
        p_return_value  => 'SET_ITEM_EQUAL_THIS_PREFERENCE',
        p_help_text     => q'~Set to the value of the <em>Preference</em> entered.~' );
    --
    l_property_id := 53;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_STATIC_VALUE',
        p_prompt             => 'Static Value',
        p_property_type      => 'TEXT',
        p_db_column          => 'COMPUTATION',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'COMPUTATION',
        p_help_text          => q'~Enter the text that is used to populate the computation item.~' );
    --
    l_property_id := 54;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'COMPUTATION_SQL_STATEMENT',
        p_prompt               => 'SQL Query',
        p_property_type        => 'SQL',
        p_db_column            => 'COMPUTATION',
        p_display_length       => 80,
        p_max_length           => 4000,
        p_sql_min_column_count => 1,
        p_sql_max_column_count => 1,
        p_display_group_name   => 'COMPUTATION',
        p_help_text            => q'~Enter a SQL query that provides a value used to populate the computation item. If the result contains multiple rows then only the value from the first row is returned.~' );
    --
    l_property_id := 55;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PRINT_SERVER_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PRINT_SERVER_TYPE',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the type of print server.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Oracle BI Publisher',
        p_return_value  => 'BIP',
        p_help_text     => q'~Select this option if you are using a licensed version of Oracle BI Publisher.
This option enables you to take report query results and convert them from XML to RTF format using Oracle BI Publisher.
Select this option if you want to upload your own customized RTF or XSL-FO templates for printing reports within &PRODUCT_NAME..~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Apache FOP',
        p_return_value  => 'FOP',
        p_help_text     => q'~Select this option if you are using Apache FOP on an external J2EE server.
This option enables you to use the basic printing functionality, which includes creating report queries and
printing report regions using the default templates provided in &PRODUCT_NAME. and using your own customized XSL-FO templates.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'APEX Office Print',
        p_return_value  => 'AOP',
        p_help_text     => q'~Select this option if you are using APEX Office Print.~' );
    --
    l_property_id := 56;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CREDENTIAL_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'AUTHENTICATION_TYPE',
        p_display_group_name => 'AUTHENTICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Specify the Authentication type here. &PRODUCT_NAME. supports the following authentication types:</p>
<ul>
    <li><strong>Basic Authentication</strong> - Sends username and password in Base64-encoded form as the <
strong>Authorization</strong> request header.
    </li>
    <li><strong>OAuth2 Client Credentials</strong> - &PRODUCT_NAME. exchanges the client ID and client secret for an <strong>Access Token</strong> using a token server URL. The access token is t
hen used to perform the actual request. If the access token is expired, &PRODUCT_NAME. will transparently request a new one.</li>
    <li><strong>OCI Native Authentication</strong> - &PRODUCT_NAME. signs requests to the <em>Oracle Cloud Infrastructur
e (OCI)</em> REST API, as described in the <a href="https://docs.cloud.oracle.com/en-us/iaas/Content/API/Concepts/signingrequests.htm">OCI Documentation</a>. This type is only available in Oracle Database 18c and higher.</li>
    <li><strong>HTTP Hea
der</strong> - The credential is added to the REST request as an HTTP Header. The name of the credential is used as the HTTP Header name, and the Secret of the credential is used as the HTTP Header value. &PRODUCT_NAME. does not add these secrets to
the Debug log or any other logs.</li>
    <li><strong>URL Query String</strong> -  The credential is added to the URL of the REST request as a <em>Query String</em> Parameter (for example: <strong>?name=value</strong>). This option can be useful with
 API keys. &PRODUCT_NAME. does not add these secrets to the Debug log or any other logs. However, web proxy or other intermediate servers may record these credentials in their log files.</li>
</ul>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Basic Authentication',
        p_return_value  => 'BASIC' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'OAuth2 Client Credentials Flow',
        p_return_value  => 'OAUTH2_CLIENT_CREDENTIALS' );
    --
    /* $$$ currently commented out
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'JSON Web Token',
        p_return_value  => 'JWT' );
    */
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Oracle Cloud Infrastructure (OCI)',
        p_return_value  => 'OCI',
        p_comment_text  => '$$$ Condition of type sys.dbms_db_version.version >= 18' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Certificate/Private Key Pair',
        p_return_value  => 'CERTIFICATE_PAIR',
        p_comment_text  => q'~$$$ Condition of type wwv_flow_platform.get_preference('SAML_ENABLED')='Y'~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'HTTP Header',
        p_return_value  => 'HTTP_HEADER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'URL Query String',
        p_return_value  => 'HTTP_QUERY_STRING' );
    --
    l_property_id := 57;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'OAUTH_SCOPE',
        p_prompt             => 'OAuth Scope',
        p_property_type      => 'TEXT',
        p_db_column          => 'SCOPE',
        p_display_length     => 50,
        p_max_length         => 255,
        p_display_group_name => 'AUTHENTICATION',
        p_help_text          => q'~
<p>
Permissions represented by the Access Token in OAuth 2.0 terms are known as scopes. The scope parameter allows the application to express the desired scope of the access request. 
</p>
<p>
If your authentication server requires a scope to be specified for the access token request, provide it here. The OAuth2 acess token will then be requested with the following request body:
<pre>
grant_type=client_credentials&amp;scope=<strong>{scope}</strong>
</pre>
</p>~' );
    --
    l_property_id := 58;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'COMPUTATION_ITEM_VALUE',
        p_prompt                   => 'Item Name',
        p_property_type            => 'ITEM',
        p_db_column                => 'COMPUTATION',
        p_display_length           => 80,
        p_max_length               => 4000,
        p_display_group_name       => 'COMPUTATION',
        p_lov_component_type_scope => 'APPLICATION',
        p_help_text                => q'~Enter the item whose value is used to populate the computation item. You can type in the name or pick from the list of available items.~' );
    --
    l_property_id := 59;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_PREFERENCE_VALUE',
        p_prompt             => 'Preference',
        p_property_type      => 'TEXT',
        p_db_column          => 'COMPUTATION',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'COMPUTATION',
        p_help_text          => q'~Enter a User Preference that has a value used to populate the computation item.~' );
    --
    l_property_id := 60;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EXECUTION_SEQUENCE',
        p_prompt             => 'Sequence',
        p_property_type      => 'NUMBER',
        p_db_column          => 'SEQUENCE',
        p_display_length     => 10,
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_help_text          => q'~<p>Specify the sequence for this component. The sequence determines the order of execution.</p>
<p>Note: If two components have the same sequence value, then they may be executed in a different order when the application is exported and imported into another environment.</p>~' );
    --
    l_property_id := 61;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_POINT',
        p_prompt             => 'Point',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'COMPUTATION_POINT',
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select at what point in page rendering or processing this computation is executed.
The most typical computation point is <strong>After Submit</strong>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'New Session',
        p_return_value  => 'ON_NEW_INSTANCE',
        p_help_text     => q'~Executes when a new session is generated.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Before Header',
        p_return_value  => 'BEFORE_HEADER',
        p_help_text     => q'~Rendering: Executes before the page renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Header',
        p_return_value  => 'AFTER_HEADER',
        p_help_text     => q'~Rendering: Executes after the header renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Before Regions',
        p_return_value  => 'BEFORE_BOX_BODY',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Rendering: Executes before regions render.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Regions',
        p_return_value  => 'AFTER_BOX_BODY',
        p_help_text     => q'~Rendering: Executes after regions render.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Before Footer',
        p_return_value  => 'BEFORE_FOOTER',
        p_help_text     => q'~Rendering: Executes before footer renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Footer',
        p_return_value  => 'AFTER_FOOTER',
        p_help_text     => q'~Rendering: Executes after footer renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Submit',
        p_return_value  => 'AFTER_SUBMIT',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Processing: Executes after the page is submitted, but before page processing.~' );
    --
    l_property_id := 62;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EVENT',
        p_prompt             => 'Event',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'BIND_EVENT_TYPE',
        p_display_group_name => 'WHEN',
        p_lov_type           => 'EVENTS',
        p_help_text          => q'~<p>Specify the event that causes the dynamic action to fire.
For information about specific events, please see the relevant section in the <em>App Builder User's Guide</em>.</p>
Events are categrorized into the following types:
<dl>
  <dt>Browser Events</dt>
    <dd>Standard browser events.</dd>
  <dt>Framework Events</dt>
    <dd>Framework events are triggered from within the Oracle APEX framework.</dd>
  <dt>Component Events</dt>
    <dd>Only available when there is a component available to your application that triggers a custom event. These events are triggered from either native components or plug-in components you have installed in your application.</dd>
  <dt>Custom Event</dt>
    <dd>Any custom event triggered from your application.</dd>
</dl>
<p>Note: The events displayed differ based on page's current User Interface type.
If you wish to select an event that corresponds to a different type, then you have the option of selecting <strong>Show unsupported</strong>,
which displays all events including those that do not correspond to the current type.</p>~' );
    --
    l_property_id := 63;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CUSTOM_EVENT',
        p_prompt             => 'Custom Event',
        p_property_type      => 'TEXT',
        p_db_column          => 'BIND_EVENT_TYPE_CUSTOM',
        p_display_length     => 50,
        p_max_length         => 130,
        p_display_group_name => 'WHEN',
        p_help_text          => q'~Specify a custom JavaScript event name if the declarative events provided are not sufficient.~' );
    --
    l_property_id := 64;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WHEN_TYPE',
        p_prompt             => 'Selection Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'TRIGGERING_ELEMENT_TYPE',
        p_display_group_name => 'WHEN',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the type of page element or construct to be used to trigger the event. For examples of using a specific selection type, see Help on the associated attribute.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item(s)',
        p_return_value  => 'ITEM',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by one or more page items.~');
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Button',
        p_return_value  => 'BUTTON',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by a button.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Region',
        p_return_value  => 'REGION',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by a region.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Column(s)',
        p_return_value  => 'COLUMN',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by one or more columns.' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DOM Object',
        p_return_value  => 'DOM_OBJECT',
        p_is_deprecated => 'Y',
        p_help_text     => q'~<p>Triggered by the DOM object (<code>document</code>, <code>window</code>) or the <em>id</em> of a DOM object (<code>myElementID</code>).</p>
        <p>Note: <em>DOM Object</em> is deprecated and should be replaced by <em>JavaScript Expression</em> or <em>jQuery Selector</em>.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'jQuery Selector',
        p_return_value  => 'JQUERY_SELECTOR',
        p_help_text     => q'~Triggered by one or more page elements returned by the jQuery Selector.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'JavaScript Expression',
        p_return_value  => 'JAVASCRIPT_EXPRESSION',
        p_help_text     => q'~Triggered by the page elements returned by the JavaScript expression.~' );
    --
    l_property_id := 65;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_REGION',
        p_prompt                   => 'Region',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'TRIGGERING_REGION_ID',
        p_display_group_name       => 'WHEN',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~<p>Select the name of the region that triggers the dynamic action.
For example this could be an 'Interactive Report' region, used in conjunction with an 'After Refresh' event, to define a dynamic action that fire whenever the report refreshes.</p>
<p>Having region trigger a dynamic action can be useful when you want to define similar behavior for multiple page items in a region. However, this is only possible for certain events that are raised by items within that region.
For a list of events that support this type of definition and an example of how they could be utilized, see item Help for the <em>Affected Elements</em> &gt; <em>Selection Type</em> of Event Scope on the Action page.</p>~' );
    --
    l_property_id := 66;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_BUTTON',
        p_prompt                   => 'Button',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'TRIGGERING_BUTTON_ID',
        p_display_group_name       => 'WHEN',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_button,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Select the button to trigger the dynamic action.~' );
    --
    l_property_id := 67;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_ITEMS',
        p_prompt                   => 'Item(s)',
        p_property_type            => 'ITEM',
        p_multi_value_delimiter    => ',',
        p_db_column                => 'TRIGGERING_ELEMENT',
        p_display_length           => 60,
        p_max_length               => 4000,
        p_display_group_name       => 'WHEN',
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Enter page or application items to trigger the dynamic action.
For multiple items, separate each item name with a comma.
You can type in the name or pick from the list of available items.
If you pick from the list and there is already text entered, a comma is placed at the end of the existing text, followed by the item name returned from the list.~' );
    --
    l_property_id := 68;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WHEN_DOM_OBJECT',
        p_prompt             => 'DOM Object',
        p_property_type      => 'TEXT',
        p_db_column          => 'TRIGGERING_ELEMENT',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'WHEN',
        p_help_text          => q'~<p>Enter either a DOM object (<code>document</code>, <code>window</code>) or the <em>id</em> of a DOM object (<code>myElementID</code>) as the element to trigger the dynamic action.</p>
<p>Note: <em>DOM Object</em> is deprecated and should be replaced by <em>JavaScript Expression</em> or <em>jQuery Selector</em>.</p>~' );
    --
    l_property_id := 69;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WHEN_JQUERY_SELECTOR',
        p_prompt             => 'jQuery Selector',
        p_property_type      => 'TEXT',
        p_db_column          => 'TRIGGERING_ELEMENT',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>To select a page element with an DOM id of <code>my_id</code></dt>
<dd><pre>#my_id</pre></dd>
<dt>To select all the page elements with a class of <code>my_class</code></dt>
<dd><pre>.my_class</pre></dd>
<dt>To select all the page elements that are inputs</dt>
<dd><pre>input</pre></dd>
</dl>~',
        p_display_group_name => 'WHEN',
        p_help_text          => q'~<p>Enter jQuery Selector syntax to return one or more page element to trigger the dynamic action. The jQuery library offers a very powerful selector syntax which provides a way to easily access one or more elements from a page.</p>
<p>Please see the official documentation available from the <a rel="noreferrer" target="_blank" href="http://docs.jquery.com">jQuery homepage</a> for further information about the selector syntax.</p>~' );
    --
    l_property_id := 70;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_CONDITION',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'TRIGGERING_CONDITION_TYPE',
        p_display_group_name => 'CLIENT_SIDE_CONDITION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify a client-side condition to control whether True or False actions fire. If the condition
                                  evaluates to True, then True actions fire, if the condition evaluates to False, then False actions fire.
                                  If no condition is specified, only True actions fire.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item = Value',
        p_return_value  => 'EQUALS',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item != Value',
        p_return_value  => 'NOT_EQUALS',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is not equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item > Value',
        p_return_value  => 'GREATER_THAN',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is greater than the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item >= Value',
        p_return_value  => 'GREATER_THAN_OR_EQUAL',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is greater than or equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item < Value',
        p_return_value  => 'LESS_THAN',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is less than the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item <= Value',
        p_return_value  => 'LESS_THAN_OR_EQUAL',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is less than or equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is null',
        p_return_value  => 'NULL',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Checks if the selected <em>Item</em> is empty.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is not null',
        p_return_value  => 'NOT_NULL',
        p_help_text     => q'~Checks if the selected <em>Item</em> is not empty.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is in list',
        p_return_value  => 'IN_LIST',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is in the <em>List</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is not in list',
        p_return_value  => 'NOT_IN_LIST',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is not in the <em>List</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'JavaScript expression',
        p_return_value  => 'JAVASCRIPT_EXPRESSION',
        p_help_text     => q'~Evaluates the JavaScript Expression specified.~' );
    --
    l_property_id := 80;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EVENT_SCOPE',
        p_prompt             => 'Event Scope',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'BIND_TYPE',
        p_display_group_name => 'ADVANCED',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the scope of the event, which determines when the event is evaluated a second or successive time.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static',
        p_return_value  => 'bind',
        p_help_text     => q'~Binds the event handler to the triggering element(s) for the lifetime of the current page, but is no longer bound if a triggering element is updated via Partial Page Refresh (PPR).~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Dynamic',
        p_return_value  => 'live',
        p_help_text     => q'~Binds the event handler to the triggering element(s) for the lifetime of the current page, irrespective of any triggering elements being recreated via Partial Page Refresh (PPR).~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once',
        p_return_value  => 'one',
        p_help_text     => q'~Binds the event handler to the triggering element(s) only once. The dynamic action is not triggered again until after the page has been fully refeshed.~' );
    --
    l_property_id := 81;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'STATIC_CONTAINER_JQUERY',
        p_prompt             => 'Static Container (jQuery Selector)',
        p_property_type      => 'TEXT',
        p_db_column          => 'BIND_DELEGATE_TO_SELECTOR',
        p_display_length     => 50,
        p_max_length         => 255,
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~<p>Enter the jQuery Selector, of a static container, for use by this dynamic action.</p>
<p>You can improve the performance of dynamic actions that have an event scope of <strong>Dynamic</strong>.
This must be an element on the page that itself does not get recreated, but contains the triggering elements that are recreated using Partial Page Refresh (PPR).
This element reference needs to be defined as a jQuery selector.</p>~' );
    --
    l_property_id := 82;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DA_ACTION_TYPE',
        p_prompt             => 'Action',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'ACTION',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Specify which action you want to perform. The following actions are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may also have additional dynamic action plug-ins available here.
These are displayed as <em>&lt;My Plug-in&gt; [Plug-in]</em>.
The following plug-ins are installed within this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 83;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FIRE_WHEN_EVENT_RESULT_IS',
        p_prompt             => 'Fire When Event Result Is',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'EVENT_RESULT',
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Specify whether this action fires when the triggering element condition is met by selecting <strong>True</strong> (True Action), or when it is not met by selecting <strong>False</strong> (False Action).</p>
<p>Note: if there is no condition specified, only <strong>True</strong> actions fire.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'True',
        p_return_value  => 'TRUE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'False',
        p_return_value  => 'FALSE' );
    --
    l_property_id := 84;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FIRE_ON_PAGE_LOAD',
        p_prompt             => 'Fire on Initialization',
        p_property_type      => 'YES NO',
        p_db_column          => 'EXECUTE_ON_PAGE_INIT',
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_help_text          => q'~<p>Specify if the action fires on initialization.</p>
                                    <p>Initialization has a slightly different meaning depending on how the Dynamic Action is defined. For Dynamic
                                    Actions defined to fire on Interactive Grid columns, this specifies if the action fires when the Interactive
                                    Grid row is activated for editing. For all other Dynamic Actions, this specifies if the action fires when the
                                    page loads.</p>~' );
    --
    l_property_id := 85;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'AFFECTED_TYPE',
        p_prompt             => 'Selection Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'AFFECTED_ELEMENTS_TYPE',
        p_display_group_name => 'AFFECTED_ELEMENTS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how to define the page components to be affected when this action is executed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item(s)',
        p_return_value  => 'ITEM',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~The action affects the one or more page items you enter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Button',
        p_return_value  => 'BUTTON',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~The action affects the button you select.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Region',
        p_return_value  => 'REGION',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~The action affects the region you select.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Column(s)',
        p_return_value  => 'COLUMN',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~The action affects the one or more columns you enter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DOM Object',
        p_return_value  => 'DOM_OBJECT',
        p_is_deprecated => 'Y',
        p_help_text     => q'~The action affects the page element idenitifed by the DOM Object or DOM ID you enter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'jQuery Selector',
        p_return_value  => 'JQUERY_SELECTOR',
        p_help_text     => q'~The action affects the page element(s) returned using the jQuery selector syntax you enter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'JavaScript Expression',
        p_return_value  => 'JAVASCRIPT_EXPRESSION',
        p_help_text     => q'~The action affects the page element(s) returned using the JavaScript Expression you enter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Triggering Element',
        p_return_value  => 'TRIGGERING_ELEMENT',
        p_help_text     => q'~The action affects the page element or component that triggered the dynamic action.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Event Source',
        p_return_value  => 'EVENT_SOURCE',
        p_help_text     => q'~<p>The action affects the page element where the event originated, which could be different to the "Triggering Element" in some cases.</p>
<p>In this advanced configuration, the triggering element specified is generally a container element (such as a region) which handles events raised by all of it's sub-elements (such as page items in that region).
This selection type can then be used to act upon the sub-element that triggered the event, rather than the container element.
For example, if you have a dynamic action that defines a <strong>'Change'</strong> event on a region on your page, the action affects the actual item in the region that has changed.</p>
<p>Utilizing <em>'Event Source'</em> can be a more efficient way of defining common behaviors that affect many page elements.
The ability of container elements to catch events raised by sub-elements is determined by the event type.
The following event types support this:</p>
<ul>
  <li><strong>After Refresh</strong> (apexafterrefresh)</li>
  <li><strong>Before Refresh</strong> (apexbeforerefresh)</li>
  <li><strong>Change</strong> (change)</li>
  <li><strong>Click</strong> (click)</li>
  <li><strong>Double Click</strong> (dblclick)</li>
  <li><strong>Get Focus</strong> (focus)</li>
  <li><strong>Key Down</strong> (keydown)</li>
  <li><strong>Key Press</strong> (keypress)</li>
  <li><strong>Key Release</strong> (keyup)</li>
  <li><strong>Lose Focus</strong> (blur)</li>
  <li><strong>Mouse Button Press</strong> (mousedown)</li>
  <li><strong>Mouse Button Release</strong> (mouseup)</li>
  <li><strong>Mouse Move</strong> (mousemove)</li>
</ul>~' );
    --
    l_property_id := 86;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'AFFECTED_REGION',
        p_prompt                   => 'Region',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'AFFECTED_REGION_ID',
        p_display_group_name       => 'AFFECTED_ELEMENTS',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Select the region to be affected by this action.~' );
    --
    l_property_id := 87;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'AFFECTED_BUTTON',
        p_prompt                   => 'Button',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'AFFECTED_BUTTON_ID',
        p_display_group_name       => 'AFFECTED_ELEMENTS',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_button,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Select the button to be affected by this action.~' );
    --
    l_property_id := 88;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'AFFECTED_ITEMS',
        p_prompt                   => 'Item(s)',
        p_property_type            => 'ITEM',
        p_multi_value_delimiter    => ',',
        p_db_column                => 'AFFECTED_ELEMENTS',
        p_display_length           => 100,
        p_max_length               => 4000,
        p_display_group_name       => 'AFFECTED_ELEMENTS',
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Enter one or more page items to be affected by this action.
For multiple items, separate each item name with a comma.
<p>You can type in the name or pick from the list of available items.
If you pick from the list and there is already text entered, then a comma is placed at the end of the existing text, followed by the item name returned from the list.~' );
    --
    l_property_id := 89;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'AFFECTED_DOM_OBJECT',
        p_prompt             => 'DOM Object',
        p_property_type      => 'TEXT',
        p_db_column          => 'AFFECTED_ELEMENTS',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'AFFECTED_ELEMENTS',
        p_help_text          => q'~<p>Enter either a DOM object (<code>document</code>, <code>window</code>) or the <em>id</em> of a DOM object (<code>myElementID</code>) as the element to be affected by this action.</p>
<p>Note: <em>DOM Object</em> is deprecated and should be replaced by <em>JavaScript Expression</em> or <em>jQuery Selector</em>.</p>~' );
    --
    l_property_id := 90;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'AFFECTED_JQUERY_SELECTOR',
        p_prompt             => 'jQuery Selector',
        p_property_type      => 'TEXT',
        p_db_column          => 'AFFECTED_ELEMENTS',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>To select a page element with an DOM id of <code>my_id</code></dt>
<dd><pre>#my_id</pre></dd>
<dt>To select all the page elements with a class of <code>my_class</code></dt>
<dd><pre>.my_class</pre></dd>
<dt>To select all the page elements that are inputs</dt>
<dd><pre>input</pre></dd>
</dl>~',
        p_display_group_name => 'AFFECTED_ELEMENTS',
        p_help_text          => q'~<p>Enter jQuery Selector syntax to return one or more page elements to be affected by this action. The jQuery library offers a very powerful selector syntax which provides a way to easily access one or more elements from a page.</p>
<p>Please see the official documentation available from the <a rel="noreferrer" target="_blank" href="http://docs.jquery.com">jQuery homepage</a> for further information about the selector syntax.</p>~' );
    --
    l_property_id := 91;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EXEC_STOP_EXECUTION_ON_ERROR',
        p_prompt             => 'Stop Execution On Error',
        p_property_type      => 'YES NO',
        p_db_column          => 'STOP_EXECUTION_ON_ERROR',
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_help_text          => q'~<p>Specify whether processing of all other pending actions is stopped when an error occurs while executing the current action.</p>
<p>Note: The order of execution of specific actions is determined by the sequence specified for each action. To ensure that actions are run in the same order, Oracle recommends that the sequence numbers be unique for all True Actions and all False Actions.</p>~' );
    --
    l_property_id := 92;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WAIT_FOR_RESULT',
        p_prompt             => 'Wait For Result',
        p_property_type      => 'YES NO',
        p_db_column          => 'WAIT_FOR_RESULT',
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_help_text          => q'~<p>Specify whether all pending actions wait for the result of the current action.
Set this attribute to <strong>On</strong> if pending actions are dependent on the result of this action,
or to <strong>Off</strong> if there is no dependency and they can execute asynchronously, without waiting.</p>
<p>Note: The order of execution of specific actions is determined by the sequence specified for each action.
To ensure that actions are run in the same order, Oracle recommends that the sequence numbers be unique for all True Actions and all False Actions.</p>~' );
    --
    l_property_id := 93;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ITEM_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'DISPLAY_AS',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Specify how this item is rendered. The following item types are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may also have additional item type plug-ins available here.
These display as <em>&lt;My Plug-in&gt; [Plug-in]</em>.
The following plug-ins install into this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 94;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'REGION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'PLUG_SOURCE_TYPE',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Specify how this region is rendered. The following region types are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may also have additional region type plug-ins available here.
These display as <em>&lt;My Plug-in&gt; [Plug-in]</em>.
The following plug-ins are installed within this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 95;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PROCESS_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'PROCESS_TYPE',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Specify which process type is executed. The following process types are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may also have additional process type plug-ins available here.
These display as <em>&lt;My Plug-in&gt; [Plug-in]</em>.
The following plug-ins are installed within this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 96;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'ITEM_NAME',
        p_prompt                => 'Name',
        p_property_type         => 'TEXT',
        p_db_column             => 'NAME',
        p_display_length        => 50,
        p_max_length            => 255,
        p_text_case             => 'UPPER',
        p_restricted_characters => 'SAFE_COMPONENT_NAME',
        p_display_group_name    => 'IDENTIFICATION',
        p_help_text             => q'~Enter the name of this item. Item names must be valid Oracle identifiers. It is recommended that item names be no longer then 30 bytes so they can be used as bind variables in SQL Queries.~' );
    --
    l_property_id := 97;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DISPLAY_SEQUENCE',
        p_prompt             => 'Sequence',
        p_property_type      => 'NUMBER',
        p_db_column          => 'SEQUENCE',
        p_display_length     => 6,
        p_display_group_name => 'LAYOUT',
        p_help_text          => q'~<p>Enter the display sequence for this item. The sequence and other layout settings determine where this item is displayed in relation to other items within the region.</p>
<p>Note: If two items, within the same region, have the same sequence value then they may be displayed in a different order when the application is exported and imported into another environment, such as a test or production environment. To ensure consistency, Oracle recommends you specify unique sequence numbers for every item, or at least for those within the same region.</p>~' );
    --
    l_property_id := 98;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'REGION',
        p_prompt                   => 'Region',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'REGION_ID',
        p_display_group_name       => 'LAYOUT',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Select the region in which this component is associated. If the component is displayed then it is displayed within the specified region and relative to other components within this region.~' );
    --
    l_property_id := 99;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LABEL',
        p_prompt             => 'Label',
        p_property_type      => 'TEXT',
        p_db_column          => 'PROMPT',
        p_display_length     => 60,
        p_max_length         => 4000,
        p_display_group_name => 'LABEL',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Enter the field label for the page item.
The label displays on the page only if the item displays.
Depending on the label alignment and the display, the label displays either in its own HTML table cell or in the same cell.</p>
<p>Item labels can include shortcuts using <code>"SHORTCUT_NAME"</code> syntax.</p>~' );
    --
    l_property_id := 100;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LABEL_ALIGNMENT',
        p_prompt             => 'Alignment',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'LABEL_ALIGNMENT',
        p_display_group_name => 'LABEL',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the label alignment. Labels can display above, below, or to the left of the item.
When the label displays next to a form element, it occupies its own HTML table cell.
By changing this label alignment, you can control the label's position within that table cell.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Above ',
        p_return_value  => 'ABOVE',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Below',
        p_return_value  => 'BELOW' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center',
        p_return_value  => 'CENTER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center bottom',
        p_return_value  => 'CENTER-BOTTOM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center center',
        p_return_value  => 'CENTER-CENTER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center top',
        p_return_value  => 'CENTER-TOP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left',
        p_return_value  => 'LEFT',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left bottom',
        p_return_value  => 'LEFT-BOTTOM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left center',
        p_return_value  => 'LEFT-CENTER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left top',
        p_return_value  => 'LEFT-TOP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right',
        p_return_value  => 'RIGHT',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right bottom',
        p_return_value  => 'RIGHT-BOTTOM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right center',
        p_return_value  => 'RIGHT-CENTER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right top',
        p_return_value  => 'RIGHT-TOP' );
    --
    l_property_id := 101;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'FIELD_TEMPLATE',
        p_prompt                   => 'Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'ITEM_FIELD_TEMPLATE',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_field_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~<p>Select a label template to define the appearance and layout of this item label.</p>
        <p>Label templates are defined in the application theme.
        When an item is created, the template is automatically set to the default item template defined in the current theme.</p>
        <p>Many themes contain label templates for displaying required, optional, and hidden labels.<br>
        Universal Theme provides the following label templates:
          <dl>
            <dt>Hidden</dt>
              <dd>Use this when the label should not be displayed on screen. Note that Assistive Technology will still be able to read the label.
              Hidden is ignored for Checkbox items because a label must be shown after the checkbox.</dd>
            <dt>Optional</dt>
              <dd>Use this for form items that are optional. The label will be displayed to the side of the item.</dd>
            <dt>Optional - Above</dt>
              <dd>Use this for form items that are optional. The label will be displayed above the item.</dd>
            <dt>Optional - Floating</dt>
              <dd>Use this for form items that are optional. The label will be displayed inside of the input item, and will automatically shrink once the input field has a value.</dd>
            <dt>Required</dt>
              <dd>This label template displays a red asterisk next to the label, and is used for form items where user input is required.  The label will be displayed to the side of the item.</dd>
            <dt>Required - Above</dt>
              <dd>This label template displays a red asterisk next to the label, and is used for form items where user input is required.  The label will be displayed above the item. Item templates enable you to define the user interface attributes as part of a theme and share that definition among many page items.</dd>
            <dt>Required - Floating</dt>
              <dd>This label template displays a red asterisk next to the label, and is used for form items where user input is required.  The label will be displayed inside of the input item, and will automatically shrink once the input field has a value. Item templates enable you to define the user interface attributes as part of a theme and share that definition among many page items.</dd>
          </dl>
          See Also: <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a>
        </p>~' );
    --
    l_property_id := 102;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'FIELD_TABLE_CELL_ATTRIBUTES',
        p_prompt                => 'Table Cell Attributes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ' ',
        p_db_column             => 'CATTRIBUTES',
        p_display_length        => 60,
        p_max_length            => 255,
        p_display_group_name    => 'LABEL',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~Enter any additional attributes used in the HTML table cell that contains this label.~' );
    --
    l_property_id := 103;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'VALUE_REQUIRED',
        p_prompt             => 'Value Required',
        p_property_type      => 'YES NO',
        p_db_column          => 'IS_REQUIRED',
        p_display_group_name => 'VALIDATION',
        p_help_text          => q'~<p>If set to <strong>On</strong> and the page item is visible, &PRODUCT_NAME. automatically performs a NOT NULL validation when the page is submitted.</p>
<p>By defining a message called <strong>APEX.PAGE_ITEM_IS_REQUIRED</strong> in Shared Components > Text Messages, the predefined error text can be replaced by an application specific error message.</p>
<p>The display location of the message is defined by the application level setting <strong>Default error display location</strong>.</p>~' );
    --
    l_property_id := 104;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GRID_NEW_GRID',
        p_prompt             => 'Start New Layout',
        p_property_type      => 'YES NO',
        p_db_column          => 'GRID_NEW_GRID',
        p_display_group_name => 'LAYOUT',
        p_help_text          => q'~Specify whether this component displays in a new layout, or in the same layout as the previous component.~' );
    --
    l_property_id := 105;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GRID_NEW_ROW',
        p_prompt             => 'Start New Row',
        p_property_type      => 'YES NO',
        p_db_column          => 'GRID_NEW_ROW',
        p_display_group_name => 'LAYOUT',
        p_help_text          => q'~Specify whether this component displays on the next row after the previous component, or on the same row.~' );
    --
    l_property_id := 106;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GRID_COLUMN',
        p_prompt             => 'Column',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'GRID_COLUMN',
        p_display_group_name => 'LAYOUT',
        p_lov_type           => 'GRID_COLUMNS',
        p_help_text          => q'~Select a specific layout column. Use <strong>Automatic</strong> to automatically determine the next available layout column where the component displays.~' );
    --
    l_property_id := 107;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GRID_NEW_COLUMN',
        p_prompt             => 'New Column',
        p_property_type      => 'YES NO',
        p_db_column          => 'GRID_NEW_COLUMN',
        p_display_group_name => 'LAYOUT',
        p_help_text          => q'~Specify whether this component displays in the next column or in the same column as the previous component.~' );
    --
    l_property_id := 108;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GRID_COLUMN_SPAN',
        p_prompt             => 'Column Span',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'GRID_COLUMN_SPAN',
        p_display_group_name => 'LAYOUT',
        p_lov_type           => 'GRID_COLUMNS',
        p_help_text          => q'~Select how many layout columns this component occupies. Use <strong>Automatic</strong> to balance the available layout columns across all components in the same row.~' );
    --
    l_property_id := 109;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GRID_ROW_SPAN',
        p_prompt             => 'Row Span',
        p_property_type      => 'NUMBER',
        p_db_column          => 'GRID_ROW_SPAN',
        p_display_length     => 2,
        p_display_group_name => 'LAYOUT',
        p_help_text          => q'~Enter how many rows within the HTML table this component occupies. Components are laid out in HTML tables. This attribute determines the value to be used for the <em>ROWSPAN</em> attribute in the table cell for this component.~' );
    --
    l_property_id := 110;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'GRID_COLUMN_ATTRIBUTES',
        p_prompt                => 'Column Attributes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ' ',
        p_db_column             => 'GRID_COLUMN_ATTRIBUTES',
        p_display_length        => 40,
        p_max_length            => 255,
        p_display_group_name    => 'LAYOUT',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~Enter additional attributes set for the layout column. This value replaces the <code>#ATTRIBUTES#</code> substitution placeholder in the layout column template.~' );
    --
    l_property_id := 111;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FORMAT_MASK',
        p_prompt             => 'Format Mask',
        p_property_type      => 'COMBOBOX',
        p_db_column          => 'FORMAT_MASK',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'APPEARANCE',
        p_lov_type           => 'ALL_FORMAT_MASKS',
        p_help_text          => q'~<p>Enter a format mask to apply to this component.
   You can type in the format mask or pick from the list of predefined format masks, based on a sample representation of how that format mask is displayed.</p>
   <p>It is important that number format masks are only applied to components that contain numbers and date format masks are only applied to components that contain dates.
    Otherwise, a runtime error is raised when the component contains a value that can not be converted using the specified format mask.</p>~' );
    --
    l_property_id := 112;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FORMAT_MASK_DATE',
        p_prompt             => 'Format Mask',
        p_property_type      => 'COMBOBOX',
        p_db_column          => 'FORMAT_MASK',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'APPEARANCE',
        p_lov_type           => 'DATE_FORMAT_MASKS',
        p_help_text          => q'~Enter a date format mask to apply to this component.
You can type in the format mask or pick from the list of predefined format masks, based on a sample representation of how that format mask displays.
You can also use substitution syntax to reference a dynamic format mask from an application or page item.~' );
    --
    l_property_id := 113;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FORMAT_MASK_NUMBER',
        p_prompt             => 'Format Mask',
        p_property_type      => 'COMBOBOX',
        p_db_column          => 'FORMAT_MASK',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'APPEARANCE',
        p_lov_type           => 'NUMBER_FORMAT_MASKS',
        p_help_text          => q'~Enter a number format mask to apply to this component.
You can type in the format mask or pick from the list of predefined format masks, based on a sample representation of how that format mask displays.
You can also use substitution syntax to reference a format mask from an application or page item.~' );
    --
    l_property_id := 114;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'NAMED_LOV',
        p_prompt                => 'List of Values',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'NAMED_LOV_ID',
        p_display_group_name    => 'LIST_OF_VALUES',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_lov,
        p_help_text             => q'~Select the name of an existing list of values.~' );
    --
    l_property_id := 115;
    store_property (
        p_id                         => l_property_id,
        p_name                       => 'DISPLAY_EXTRA_VALUES',
        p_prompt                     => 'Display Extra Values',
        p_property_type              => 'YES NO',
        p_db_column                  => 'LOV_DISPLAY_EXTRA',
        p_display_group_name         => 'LIST_OF_VALUES',
        p_help_text                  => q'~An item may have a session state value which does not occur in the given list of values definition.
Select whether this list of values displays this extra session state value.
If you choose to not display this extra session state value and there is no matching value in the list of values definition, the first value in the list of values is the selected and displayed value.~' );
    --
    l_property_id := 116;
    store_property (
        p_id                         => l_property_id,
        p_name                       => 'DISPLAY_NULL_VALUE',
        p_prompt                     => 'Display Null Value',
        p_property_type              => 'YES NO',
        p_db_column                  => 'LOV_DISPLAY_NULL',
        p_display_group_name         => 'LIST_OF_VALUES',
        p_help_text                  => q'~Specify whether the list of values used for this item displays a NULL value at the top of the list.~' );
    --
    l_property_id := 117;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'NULL_DISPLAY_VALUE',
        p_prompt             => 'Null Display Value',
        p_property_type      => 'TEXT',
        p_db_column          => 'LOV_NULL_TEXT',
        p_display_length     => 20,
        p_max_length         => 255,
        p_display_group_name => 'LIST_OF_VALUES',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the text to be displayed within the list NULL option at the top of this list. This value also displays when rendering the item if the value is null. If you do not enter any text, it is displayed as a blank line in the list and a blank item when rendered.~' );
    --
    l_property_id := 118;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'NULL_RETURN_VALUE',
        p_prompt             => 'Null Return Value',
        p_property_type      => 'TEXT',
        p_db_column          => 'LOV_NULL_VALUE',
        p_display_length     => 20,
        p_max_length         => 255,
        p_display_group_name => 'LIST_OF_VALUES',
        p_help_text          => q'~Enter the value to be returned if the NULL value is selected in this list of values. If you do not provide a value, the item value is left as null when the NULL option is selected from this list.~' );
    --
    l_property_id := 119;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'CASCADING_LOV_PARENT_ITEMS',
        p_prompt                   => 'Parent Item(s)',
        p_property_type            => 'ITEM',
        p_multi_value_delimiter    => ',',
        p_db_column                => 'LOV_CASCADE_PARENT_ITEMS',
        p_display_length           => 50,
        p_max_length               => 255,
        p_display_group_name       => 'CASCADING_LOV',
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~<p>Enter page or application items that trigger the refresh of this list of values.
For multiple items, separate each item name with a comma.
You can type in the name or pick from the list of available items.
If you pick from the list, and there is already text entered, then a comma is placed at the end of the existing text, followed by the item name returned from the list.</p>
<p>This list of values is refreshed whenever the value of any of the specified items are changed on this page.</p>
<p>You should use the page item(s) specified in this attribute within the <em>where clause</em> of this list of values SQL Query to restrict the values returned.</p>
<p>For example, if you have page items for Car Make and Model, you could specify Car Make as the Cascading LOV Parent Item within the List of Values definition for Model, and use Car Make within the where clause for that list of values.
Then whenever Car Make is changed, the LOV for Model is refreshed, and the list only displays values for the currently selected Car Make.</p>~' );
    --
    l_property_id := 120;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'LOV_PAGE_ITEMS_TO_SUBMIT',
        p_prompt                   => 'Items to Submit',
        p_property_type            => 'ITEM',
        p_multi_value_delimiter    => ',',
        p_db_column                => 'AJAX_ITEMS_TO_SUBMIT',
        p_display_length           => 50,
        p_max_length               => 4000,
        p_display_group_name       => 'CASCADING_LOV',
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~<p>Enter column, page or application items to set into session state when the value of any of the specified Cascading LOV Parent Item(s) change on this page.
For multiple items, separate each item name with a comma.
You can type in the name or pick from the list of available items.
If you pick from the list and there is already text entered, then a comma is placed at the end of the existing text, followed by the item name returned from the list.</p>
<p>You should use the page item(s) specified in this attribute within the <em>where clause</em> of this list of values SQL Query to restrict the values returned.</p>
<p>Note: The page items defined in <em>Cascading LOV Parent Item(s)</em> are automatically transmitted and do not need to be entered into this attribute.</p>~' );
    --
    l_property_id := 121;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CASCADING_LOV_PARENT_REQUIRED',
        p_prompt             => 'Parent Required',
        p_property_type      => 'YES NO',
        p_db_column          => 'AJAX_OPTIMIZE_REFRESH',
        p_display_group_name => 'CASCADING_LOV',
        p_help_text          => q'~<p>Specify under what conditions this list of values is refreshed.</p>
<p>If set to <strong>On</strong> the refresh call to get new values performs only when all <em>Cascading LOV Parent Item(s)</em> are not null.
If one item is null, it is assumed that the LOV Query does not return any rows.
In such cases, the select list entries are simply cleared in the browser, without doing a refresh.</p>
<p>If set to <strong>Off</strong>, the refresh is always performed.
This option is only used if you specifically handle nullable <em>Cascading LOV Parent Item(s)</em> in the <em>where clause</em> of this list of values SQL Query.
If nullable page items are not properly handled, unnecessary refresh calls are executed that never return any rows.</p>~' );
    --
    l_property_id := 122;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'LOV_SQL',
        p_prompt               => 'SQL Query',
        p_property_type        => 'SQL',
        p_db_column            => 'LOV',
        p_display_length       => 80,
        p_max_length           => 4000,
        p_sql_min_column_count => 2,
        p_sql_max_column_count => 2,
        p_display_group_name   => 'LIST_OF_VALUES',
        p_examples             => q'~<dl>
<dt>Display department name, set department number</dt>
<dd><pre>
select dname  as d,
       deptno as r
  from dept
 order by 1
</pre></dd>
</dl>~',
        p_help_text            => q'~<p>Enter the SQL query definition to populate this list of values. Generally list of value queries are of the form:</p>
<pre>
select [displayValue],
       [returnValue]
  from ...
 where ...
 order by ...
</pre>
<p>Each column selected must have a unique name or alias. Oracle recommends using an alias on any column that includes an SQL expression.</p>~' );
    --
    l_property_id := 123;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ELEMENT_FIELD_ALIGNMENT',
        p_prompt             => 'Alignment',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'FIELD_ALIGNMENT',
        p_display_group_name => 'LAYOUT',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the alignment of the item within the HTML table cell that surrounds it.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center',
        p_return_value  => 'CENTER',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center bottom',
        p_return_value  => 'CENTER-BOTTOM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center center',
        p_return_value  => 'CENTER-CENTER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Center top',
        p_return_value  => 'CENTER-TOP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left',
        p_return_value  => 'LEFT',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left bottom',
        p_return_value  => 'LEFT-BOTTOM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left center',
        p_return_value  => 'LEFT-CENTER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left top',
        p_return_value  => 'LEFT-TOP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right',
        p_return_value  => 'RIGHT',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right bottom',
        p_return_value  => 'RIGHT-BOTTOM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right center',
        p_return_value  => 'RIGHT-CENTER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right top',
        p_return_value  => 'RIGHT-TOP' );
    --
    l_property_id := 124;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ELEMENT_WIDTH',
        p_prompt             => 'Width',
        p_property_type      => 'NUMBER',
        p_db_column          => 'CSIZE',
        p_display_length     => 3,
        p_unit               => c_unit_chars,
        p_display_group_name => 'APPEARANCE',
        p_help_text          => q'~Enter the width of the form element to be used to display this item.~' );
    --
    l_property_id := 125;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ELEMENT_MAX_CHARACTERS',
        p_prompt             => 'Maximum Length',
        p_property_type      => 'NUMBER',
        p_db_column          => 'CMAXLENGTH',
        p_display_length     => 4,
        p_unit               => c_unit_chars,
        p_display_group_name => 'VALIDATION',
        p_help_text          => q'~Enter the maximum number of characters users can type into this item.~' );
    --
    l_property_id := 126;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ELEMENT_HEIGHT',
        p_prompt             => 'Height',
        p_property_type      => 'NUMBER',
        p_db_column          => 'CHEIGHT',
        p_display_length     => 3,
        p_unit               => c_unit_lines,
        p_display_group_name => 'APPEARANCE',
        p_help_text          => q'~Enter the height (approximate number of rows) for textareas, shuttles, multi select lists, and any plug-in Item types that support the Has Height Attribute attribute.~' );
    --
    l_property_id := 127;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ELEMENT_VALUE_PLACEHOLDER',
        p_prompt             => 'Value Placeholder',
        p_property_type      => 'TEXT',
        p_db_column          => 'PLACEHOLDER',
        p_display_length     => 50,
        p_max_length         => 255,
        p_display_group_name => 'APPEARANCE',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Enter hint text displayed in this item when no value for that item is entered. Once any text is entered into the item, the hint is removed.</p>
<p>This placeholder attribute specifies a short hint that can be used to describe the expected value of a page item.
Such hints are useful for providing a sample value, or an example of the expected format, or instructions such as: Search on title, description, and tag.
For accessibility reasons, this attribute should not be used as an alternative to a well defined, meaningful label.</p>
<p><strong>Attention:</strong> This is a HTML5 feature and only works in modern browsers! Older browsers do not display the text entered into this attribute.</p>~' );
    --
    l_property_id := 128;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'ADVANCED_CSS_CLASSES',
        p_prompt                => 'CSS Classes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ' ',
        p_db_column             => 'CSS_CLASSES',
        p_display_length        => 50,
        p_max_length            => 255,
        p_examples              => q'~<dl>
<dt>Add the CSS classes <code>articlenumber</code> and <code>highlight</code> to the HTML tag of the component</dt>
<dd><pre>articlenumber highlight</pre></dd>
</dl>~',
        p_display_group_name    => 'ADVANCED',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~Enter additional space delimited CSS classes to be included in the component.~' );
    --
    l_property_id := 129;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'CUSTOM_ATTRIBUTES',
        p_prompt                => 'Custom Attributes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ' ',
        p_db_column             => 'HTML_ATTRIBUTES',
        p_display_length        => 50,
        p_max_length            => 2000,
        p_display_group_name    => 'ADVANCED',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~Enter additional space delimited attributes to be included in the item tag.
For the HTML CLASS attribute, Oracle recommends using the dedicated CSS Classes attribute.~' );
    --
    l_property_id := 130;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'ELEMENT_OPTION_HTML_ATTRIBUTES',
        p_prompt                => 'Option HTML Attributes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ' ',
        p_db_column             => 'TAG_ATTRIBUTES2',
        p_display_length        => 50,
        p_max_length            => 255,
        p_display_group_name    => 'ADVANCED',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~Enter additional space-delimited attributes for the HTML table cell used to display each individual option in a radio group or set of check boxes.~' );
    --
    l_property_id := 131;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ELEMENT_PRE_TEXT',
        p_prompt             => 'Pre Text',
        p_property_type      => 'HTML',
        p_db_column          => 'PRE_ELEMENT_TEXT',
        p_display_length     => 50,
        p_max_length         => 4000,
        p_display_group_name => 'ADVANCED',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Enter additional attributes displayed before the HTML table cell that displays each individual option in a radio group or set of check boxes.
You can include HTML, JavaScript, and shortcuts.</p>
<p>You can reference the following substitution strings:</p>
<dl>
  <dt><code>#CURRENT_FORM_ELEMENT#</code></dt>
  <dd>Obtains the name of the HTML form element that this page item is associated with.</dd>
  <dt><code>#CURRENT_ITEM_NAME#</code></dt>
  <dd>Obtains the name of the current page item.</dd>
</dl>~' );
    --
    l_property_id := 132;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ELEMENT_POST_TEXT',
        p_prompt             => 'Post Text ',
        p_property_type      => 'HTML',
        p_db_column          => 'POST_ELEMENT_TEXT',
        p_display_length     => 50,
        p_max_length         => 4000,
        p_display_group_name => 'ADVANCED',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Enter additional attributes displayed after the HTML table cell that displays each individual option in a radio group or set of check boxes.
You can include HTML, JavaScript, and shortcuts.</p>
<p>You can reference the following substitution strings:</p>
<dl>
  <dt><code>#CURRENT_FORM_ELEMENT#</code></dt>
  <dd>Obtains the name of the HTML form element that this page item is associated with.</dd>
  <dt><code>#CURRENT_ITEM_NAME#</code></dt>
  <dd>Obtains the name of the current page item.</dd>
</dl>~' );
    --
    l_property_id := 133;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SOURCE_USED',
        p_prompt             => 'Used',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'USE_CACHE_BEFORE_DEFAULT',
        p_display_group_name => 'SOURCE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select which value takes precedence, the value from session state or the item source value.</p>
<p>For items with the source type <strong>Database Column</strong>, select <strong>Always, replacing any existing value in session state</strong> unless you do not want the associated row fetch operation to populate the item if the page branches back to itself.
For other source types you must decide, based on how the page is used, whether the previously sourced value is displayed, or if the source type is applied anew for each page view.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Always, replacing any existing value in session state',
        p_return_value  => 'NO' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Only when current value in session state is null',
        p_return_value  => 'YES' );
    --
    l_property_id := 134;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SOURCE_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'SOURCE_TYPE',
        p_display_group_name => 'SOURCE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select the source type the value of this item is based on.</p>
<p>Note: If <strong>Source: Used</strong> is set to <em>Only when current value in session state is null</em>, the source will not be utilized if a value has been set into session state for this item.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static Value',
        p_return_value  => 'STATIC',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the text entered into <em>Static Value</em>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Database Column',
        p_return_value  => 'DB_COLUMN',
        p_is_quick_pick => 'Y',
        p_is_deprecated => 'N', -- $$$ this should actually be legacy!
        p_help_text     => q'~Set to the column value entered in <em>Database Column</em>, that is retrieved from a built-in fetch row process.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item',
        p_return_value  => 'ITEM',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the value held in session state for the <em>Item</em> selected.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return single value)',
        p_return_value  => 'QUERY',
        p_help_text     => q'~Set to the first value returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return colon separated value)',
        p_return_value  => 'QUERY_COLON',
        p_help_text     => q'~Set to a single colon separated value, which concatenates each row value, returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Expression',
        p_return_value  => 'EXPRESSION',
        p_help_text     => q'~Set to the value that results from executing the <em>Expression</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_help_text     => q'~Set to the value returned from the <em>Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Preference',
        p_return_value  => 'PREFERENCE',
        p_help_text     => q'~Set to the value of the <em>Preference</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Null',
        p_return_value  => 'ALWAYS_NULL',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to null as no value specified.~' );
    --
    l_property_id := 135;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SOURCE_STATIC_VALUE',
        p_prompt             => 'Static Value',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'SOURCE',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'SOURCE',
        p_help_text          => q'~Enter the static value used as the source for this item value.~' );
    --
    l_property_id := 136;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'SOURCE_QUERY',
        p_prompt               => 'SQL Query',
        p_property_type        => 'SQL',
        p_db_column            => 'SOURCE',
        p_display_length       => 80,
        p_max_length           => 4000,
        p_sql_min_column_count => 1,
        p_sql_max_column_count => 1,
        p_display_group_name   => 'SOURCE',
        p_help_text            => q'~Enter the SQL Query used as the source for this item value. If more than one row is returned then the value of the first row is used.~' );
    --
    l_property_id := 137;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'SOURCE_QUERY_COLON',
        p_prompt               => 'SQL Query',
        p_property_type        => 'SQL',
        p_db_column            => 'SOURCE',
        p_display_length       => 80,
        p_max_length           => 4000,
        p_sql_min_column_count => 1,
        p_sql_max_column_count => 1,
        p_display_group_name   => 'SOURCE',
        p_help_text            => q'~Enter the SQL Query that returns one or more rows to be used as the source for this item value. If the result contains multiple rows then the value from each row is formed into a single colon delimited value.~' );
    --
    l_property_id := 138;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'SOURCE_ITEM',
        p_prompt                   => 'Item',
        p_property_type            => 'ITEM',
        p_db_column                => 'SOURCE',
        p_display_length           => 80,
        p_max_length               => 255,
        p_display_group_name       => 'SOURCE',
        p_lov_component_type_scope => 'APPLICATION',
        p_help_text                => q'~Enter the page or application item used as the source for this item value.
You can type in the name or pick from the list of available items.~' );
    --
    l_property_id := 139;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CLIENT_ID_OR_USERNAME',
        p_prompt             => 'Client ID or Username',
        p_property_type      => 'TEXT',
        p_db_column          => 'CLIENT_ID',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'AUTHENTICATION',
        p_help_text          => q'~Enter the Username for Basic Authentication and the Client ID for the OAuth2 Client Credentials flow. &PRODUCT_NAME. does not store this information encrypted.~' );
    --
    l_property_id := 140;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'TASK_VALUE_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'VALUE_TYPE',
        p_display_group_name => 'VALUE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how this parameter is populated.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static Value',
        p_return_value  => 'STATIC',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the text entered into <em>Static Value</em>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item',
        p_return_value  => 'ITEM',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the value held in session state for the <em>Item</em> selected.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return single value)',
        p_return_value  => 'SQL_QUERY',
        p_help_text     => q'~Set to the first value returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Expression',
        p_return_value  => 'EXPRESSION',
        p_message_code  => 'PLSQL_EXPRESSION',
        p_help_text     => q'~Set to the value that results from executing the <em>Expression</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_message_code  => 'PLSQL_FUNCTION_BODY',
        p_help_text     => q'~Set to the value returned from the <em>Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Preference',
        p_return_value  => 'PREFERENCE',
        p_help_text     => q'~Set to the value of the <em>Preference</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Null',
        p_return_value  => 'NULL',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to null as no value specified.~' );
    --
    l_property_id := 141;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SOURCE_DB_COLUMN',
        p_prompt             => 'Database Column',
        p_property_type      => 'TEXT',
        p_db_column          => 'SOURCE',
        p_display_length     => 80,
        p_max_length         => 128,
        p_display_group_name => 'SOURCE',
        p_help_text          => q'~<p>Enter the case sensitive database column name used as the source for this item value.</p>
<p>This column must exist in the table specified within the built-in fetch row process defined on this page.
An error is raised, and the page is not loaded, if the database column name does not exist in the referenced table.
Similarly, if there is no built-in fetch row process defined and there is an item with a source type of <strong>Database Column</strong>, then an error is raised, and the page is not loaded when trying to render the page.</p>~' );
    --
    l_property_id := 142;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SOURCE_PREFERENCE',
        p_prompt             => 'Preference',
        p_property_type      => 'TEXT',
        p_db_column          => 'SOURCE',
        p_display_length     => 80,
        p_max_length         => 255,
        p_display_group_name => 'SOURCE',
        p_help_text          => q'~Enter the User Preference to be used as the source for this item value.~' );
    --
    l_property_id := 143;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'POST_CALCULATION_COMPUTATION',
        p_prompt             => 'Post Calculation Computation',
        p_property_type      => 'PLSQL EXPRESSION VARCHAR2', -- no need to make this property available for other languages, because it's deprecated
        p_db_column          => 'SOURCE_POST_COMPUTATION',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'SOURCE',
        p_deprecated_values  => '$NOT_NULL$',
        p_help_text          => q'~<p>Enter a PL/SQL expression to refine the value of the item.</p>
<p>For example, if this item is named <code>P1_X</code>, you could use post calculation expressions such as <code>UPPER(:P1_X)</code>, <code>INITCAP(:P2_Y)</code>, <code>MIN(:P1_X, :P2_Y - 200)</code>, <code>NVL(:P1_X, :P2_Y)</code>, or a function call such as <code>my_func(:P1_X)</code> to set the final value of the item.
Note that these examples demonstrate that the expressions can reference the current item, or they may reference any other item in session state.</p>~' );
    --
    l_property_id := 144;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'MAINTAIN_SESSION_STATE',
        p_prompt             => 'Maintain Session State',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'IS_PERSISTENT',
        p_display_group_name => 'SOURCE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how session state is maintained.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Per Request (Memory Only)',
        p_return_value  => 'N',
        p_help_text     => q'~Do not save state in the database. State is only available when processing the current request. When AJAX requests need to use an item, make sure to pass the item name via "Page Items To Submit".~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Per Session (Disk)',
        p_return_value  => 'Y',
        p_help_text     => q'~Maintain for each session by storing the value in the database, to access it across requests.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Per User (Disk)',
        p_return_value  => 'U',
        p_help_text     => q'~Maintain for a user by storing the value in the database, to access it across sessions.~' );
    --
    l_property_id := 145;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DEFAULT_VALUE_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'ITEM_DEFAULT_TYPE',
        p_display_group_name => 'DEFAULT',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select a default value type for this item.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static',
        p_return_value  => 'STATIC_TEXT_WITH_SUBSTITUTIONS',
        p_help_text     => q'~Set to the value entered in <em>Static</em> value.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item',
        p_return_value  => 'ITEM',
        p_help_text     => q'~Set to the value returned from the session state for the <em>Item</em> selected.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query',
        p_return_value  => 'SQL_QUERY',
        p_help_text     => q'~Set to the value returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query returning Colon Delimited List',
        p_return_value  => 'SQL_QUERY_COLON',
        p_help_text     => q'~<p>Set to the colon delimited list of values returned from the <em>SQL Query</em> entered.</p>
        <p>Note - colon delimited lists are required when the column is displayed as a checkbox, radio group, or shuttle.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Expression',
        p_return_value  => 'EXPRESSION',
        p_help_text     => q'~Set to the value returned from the <em>Expression</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_help_text     => q'~Set to the value returned from the <em>Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Sequence',
        p_return_value  => 'SEQUENCE',
        p_help_text     => q'~<p>Set to the value returned from the <em>Sequence</em> entered. This is a "special" default type: It is executed as part of the Form Region DML process and <em>not</em> during page load. Use this type for auto-generated primary key values, when <strong>Return Primary Key(s) after Insert</strong> cannot be used, for instance, when working with views and INSTEAD OF triggers or with database links. The sequence name is case-sensitive.</p>~' );
    --
    l_property_id := 146;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DEFAULT_VALUE_STATIC_TEXT',
        p_prompt             => 'Static Value',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'ITEM_DEFAULT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'DEFAULT',
        p_help_text          => q'~Specify a default value for this item. The default value is used when the item's value is not derived from session state and when the source value is null.~' );
    --
    l_property_id := 147;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'OCI_USER_ID',
        p_prompt             => 'OCI User ID',
        p_property_type      => 'TEXT',
        p_db_column          => 'CLIENT_ID',
        p_display_length     => 80,
        p_max_length         => null,
        p_display_group_name => 'AUTHENTICATION',
        p_help_text          => q'~<p>Enter the Oracle Cloud Infrastructure <em>User OCID</em>. &PRODUCT_NAME. does not store this information encrypted.</p> <p><a target="_blank" href="https://docs.cloud.oracle.com/iaas/Content/API/Concepts/apisigning
key.htm">Oracle Cloud Infrastructure Documentation</a></p>~' );
    --
    l_property_id := 148;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'OCI_PRIVATE_KEY',
        p_prompt             => 'OCI Private Key',
        p_property_type      => 'PASSWORD',
        p_db_column          => 'CLIENT_SECRET',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'AUTHENTICATION',
        p_help_text          => q'~<p>Enter the private key in PEM format for OCI authentication. &PRODUCT_NAME. stores this information encrypted and secure, so it cannot be retrieved back in clear text.</p><p><a target="_blank" href="https://docs.clo
ud.oracle.com/iaas/Content/API/Concepts/apisigningkey.htm">Oracle Cloud Infrastructure Documentation</a></p>~' );
    --
    l_property_id := 149;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_QUICK_PICKS',
        p_prompt             => 'Show Quick Picks',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_QUICK_PICKS',
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~<p>Specify whether quick picks display for this page item.
Quick picks enable users to select predefined values with just one click, rather than typing in text or selecting from a list of available options.</p>
<p>You should specify quick picks when there is a relatively small, discrete set of <em>common</em> input values.
There is a maximum of ten quick picks that can be defined for an item.
Such user aids are especially useful when defined for an item that has a lengthy select list of available values.</p>~' );
    --
    l_property_id := 150;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LINK_ATTRIBUTES',
        p_prompt             => 'Link Attributes',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LINK_ATTR',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter HTML attributes that are associated with the display of each <strong>Quick Pick</strong> link.~' );
    --
    l_property_id := 151;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_01',
        p_prompt             => 'Label 1',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_01',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the first <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 152;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_01',
        p_prompt             => 'Value 1',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_01',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the first <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 153;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_02',
        p_prompt             => 'Label 2',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_02',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the second <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 154;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_02',
        p_prompt             => 'Value 2',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_02',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the second <strong>Quick Pick</strong>. This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 155;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_03',
        p_prompt             => 'Label 3',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_03',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the third <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 156;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_03',
        p_prompt             => 'Value 3',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_03',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the third <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 157;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_04',
        p_prompt             => 'Label 4',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_04',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the forth <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 158;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_04',
        p_prompt             => 'Value 4',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_04',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the forth <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 159;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_05',
        p_prompt             => 'Label 5',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_05',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the fifth <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 160;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_05',
        p_prompt             => 'Value 5',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_05',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the fifth <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 161;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_06',
        p_prompt             => 'Label 6',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_06',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the sixth <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 162;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_06',
        p_prompt             => 'Value 6',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_06',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the sixth <strong>Quick Pick</strong>. This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 163;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_07',
        p_prompt             => 'Label 7',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_07',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the seventh <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 164;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_07',
        p_prompt             => 'Value 7',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_07',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the seventh <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 165;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_08',
        p_prompt             => 'Label 8',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_08',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the eighth <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 166;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_08',
        p_prompt             => 'Value 8',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_08',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the eighth <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 167;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_09',
        p_prompt             => 'Label 9',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_09',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the ninth <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 168;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_09',
        p_prompt             => 'Value 9',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_09',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the ninth <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 169;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_LABEL_10',
        p_prompt             => 'Label 10',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_LABEL_10',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the tenth <strong>Quick Pick</strong>. This label is the text the end user of the application sees.~' );
    --
    l_property_id := 170;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'QUICK_PICK_VALUE_10',
        p_prompt             => 'Value 10',
        p_property_type      => 'TEXT',
        p_db_column          => 'QUICK_PICK_VALUE_10',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'QUICK_PICKS',
        p_help_text          => q'~Enter the value associated with the tenth <strong>Quick Pick</strong>.  This value replaces the existing item value when the user clicks this quick pick.~' );
    --
    l_property_id := 171;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'READ_ONLY_HTML_ATTRIBUTES',
        p_prompt             => 'Read Only HTML Attributes',
        p_property_type      => 'TEXT',
        p_db_column          => 'READ_ONLY_DISP_ATTR',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<pre>bgcolor="#FF0000"</pre>~',
        p_display_group_name => 'READ_ONLY',
        p_help_text          => q'~Enter attributes included in the item HTML table cell tag, <code>&lt;td&gt;</code>, when displayed in read only mode.~' );
    --
    l_property_id := 172;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SESSION_STATE_PROTECTION',
        p_prompt             => 'Session State Protection',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PROTECTION_LEVEL',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select a Session State Protection level.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Unrestricted',
        p_return_value  => 'N',
        p_help_text     => q'~The item can be set by passing the item in a URL or in a form. No checksum is required in the URL.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Checksum Required - Application Level',
        p_return_value  => 'B',
        p_help_text     => q'~The item can be set by passing the item in a URL that includes a checksum specific to the the workspace and application.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Checksum Required - User Level',
        p_return_value  => 'P',
        p_help_text     => q'~The item can be set by passing the item in a URL that includes a checksum specific to the the workspace, application and user.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Checksum Required - Session Level',
        p_return_value  => 'S',
        p_help_text     => q'~The item can be set by passing the item in a URL that includes a checksum specific to the session.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Restricted - May not be set from browser',
        p_return_value  => 'I',
        p_help_text     => q'~The item can not be altered using the URL or POSTDATA.
Select this option to restrict what can set the item value to internal processes, computations, and so on.
This attribute only applies to items that are not used as data entry items and is always observed, even if Session State Protection is disabled.
Use this attribute for page or application items that have the following <strong>Display As</strong> types:
<ul>
  <li>Display Only (Save State=No)</li>
  <li>Text Field (Disabled, does not save state)</li>
</ul>~' );
    --
    l_property_id := 173;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ENCRYPT_SESSION_STATE',
        p_prompt             => 'Store value encrypted in session state',
        p_property_type      => 'YES NO',
        p_db_column          => 'ENCRYPT_SESSION_STATE_YN',
        p_display_group_name => 'SECURITY',
        p_help_text          => q'~<p>Specify whether to encrypt this item when stored in session state.
If the contents of an item contain sensitive data, then you should encrypt the value when it is stored in the &PRODUCT_NAME. session state management tables.
Otherwise, anyone with rights to read the &PRODUCT_NAME. meta data tables could potentially write a query to extract this sensitive data.</p>~' );
    --
    l_property_id := 174;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ESCAPE_SPECIAL_CHARACTERS',
        p_prompt             => 'Escape special characters',
        p_property_type      => 'YES NO',
        p_db_column          => 'ESCAPE_ON_HTTP_OUTPUT',
        p_display_group_name => 'SECURITY',
        p_help_text          => q'~To prevent Cross-Site Scripting (XSS) attacks, always set this attribute to <strong>On</strong>.
If you need to render HTML tags stored in the page item or in the entries of a list of values, you can set this flag to <strong>Off</strong>.
In such cases, you should take additional precautions to ensure any user input to such fields are properly escaped when entered and before saving.~' );
    --
    l_property_id := 175;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'RESTRICTED_CHARACTERS',
        p_prompt             => 'Restricted Characters',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'RESTRICTED_CHARACTERS',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select how to restrict the characters that can be saved in session state.
&PRODUCT_NAME. displays an error message if a user tries to save data that does not conform to the selected character restriction.</p>
<p>Restricting the characters hardens application security and is a possible counter measure to cross-site scripting (XSS) attacks.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Allowlist for a-Z, 0-9 and space',
        p_return_value  => 'US_ONLY',
        p_help_text     => q'~Only characters a-z, A-Z, 0-9 and space can be saved.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Blocklist HTML command characters (<>")',
        p_return_value  => 'WEB_SAFE',
        p_help_text     => q'~Reserved HTML characters can NOT be saved.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Blocklist &<>"/;,*|=% and --',
        p_return_value  => 'NO_SPECIAL_CHAR',
        p_help_text     => q'~The following characters: &amp;, &lt;, &gt;, ", /, ;, ",", *, |, =, % and "--" (PL/SQL comment) can NOT be saved.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Blocklist &<>"/;,*|=% or -- and new line',
        p_return_value  => 'NO_SPECIAL_CHAR_NL',
        p_help_text     => q'~The following characters: &amp;, &lt;, &gt;, ", /, ;, ",", *, |, =, %, "--", and new line characters can NOT be saved.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Workspace Schema',
        p_return_value  => 'WORKSPACE_SCHEMA',
        p_is_internal   => 'Y' );
    --
    l_property_id := 176;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'BUTTON_NAME',
        p_prompt                => 'Button Name',
        p_property_type         => 'TEXT',
        p_db_column             => 'BUTTON_NAME',
        p_display_length        => 60,
        p_max_length            => 255,
        p_restricted_characters => 'SAFE_COMPONENT_NAME',
        p_display_group_name    => 'IDENTIFICATION',
        p_help_text             => q'~<p>Enter the name used to identify this button within susbsequent page processing.
When the page is submitted, by pressing this button, the REQUEST value is set to the name entered.</p>
<p>Button name does not have to be unique, for buttons on the same page or in the same region.
You can define multiple buttons with the same button name, which sets the same REQUEST value when pressed.
This is useful for defining buttons with different <strong>Button Labels</strong> or conditions, that need to execute the same page processing, by simply evaluating the common REQUEST value.</p>
<p>If you name the button <code>FLOW_RESET_BUTTON</code> then the button resets the current page without doing a page submit.</p>~' );
    --
    l_property_id := 177;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUTTON_LABEL',
        p_prompt             => 'Label',
        p_property_type      => 'TEXT',
        p_db_column          => 'BUTTON_IMAGE_ALT',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'IDENTIFICATION',
        p_is_translatable    => true,
        p_help_text          => q'~If the button is text, enter the text to be displayed on the button. If the button is an image, then this text becomes the ALT text for the image.~' );
    --
    l_property_id := 179;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HORIZONTAL_ALIGNMENT',
        p_prompt             => 'Horizontal Alignment',
        p_property_type      => 'PILL BUTTONS',
        p_db_column          => 'HORIZONTAL_ALIGNMENT',
        p_display_group_name => 'LAYOUT',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select the horizontal alignment for this button within the specified template position.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Left',
        p_return_value  => 'LEFT',
        p_icon          => 'icon-align-start' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Right',
        p_return_value  => 'RIGHT',
        p_icon          => 'icon-align-end' );
    --
    l_property_id := 180;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'STATIC_ID',
        p_prompt                => 'Static ID',
        p_property_type         => 'TEXT',
        p_db_column             => 'STATIC_ID',
        p_restricted_characters => 'DOM_STATIC_ID',
        p_display_length        => 60,
        p_max_length            => 255,
        p_display_group_name    => 'ADVANCED',
        p_help_text             => q'~Enter the static ID for this component. If defined, this ID can be useful if developing custom JavaScript behavior for the component.</p>~' );
    --
    l_property_id := 182;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'BUTTON_TEMPLATE',
        p_prompt                   => 'Button Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'BUTTON_TEMPLATE_ID',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_button_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~Select a button template to define the appearance of this button.~' );
    --
    l_property_id := 183;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUTTON_IS_HOT',
        p_prompt             => 'Hot',
        p_property_type      => 'YES NO',
        p_db_column          => 'BUTTON_IS_HOT',
        p_display_group_name => 'APPEARANCE',
        p_help_text          => q'~Specify whether to use the <strong>Normal</strong> or <strong>Hot</strong> button template when rendering this button.~' );
    --
    l_property_id := 187;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUTTON_TARGET',
        p_prompt             => 'Target',
        p_property_type      => 'LINK',
        p_db_column          => 'BUTTON_REDIRECT_URL',
        p_display_group_name => 'BEHAVIOR',
        p_help_text          => q'~Click the Target to invoke a modal dialog. You can enter the target to be called when this button is clicked. Only enter a target for buttons that should not invoke page processing when clicked (for example, a Cancel button).~' );
    --
    l_property_id := 188;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUTTON_EXECUTE_VALIDATIONS',
        p_prompt             => 'Execute Validations',
        p_property_type      => 'YES NO',
        p_db_column          => 'BUTTON_EXECUTE_VALIDATIONS',
        p_display_group_name => 'BEHAVIOR',
        p_help_text          => q'~<p>If the page is submitted and this attribute is set to <strong>On</strong>, all validations defined for the page and also the built-in validations are executed (such as required, valid number or valid date) on the server-side.
In addition, if <em>Compatibility Mode</em> of the application is set to <strong>5.1</strong> or above, &PRODUCT_NAME. will also perform client-side validations, such as item required. Please see "Compatibility Mode" in the Release Notes for further details.
The page will not be submitted until all issues are fixed.</p>
<p>If set to <strong>Off</strong>, none of the defined validations nor the built-in validations are executed.
If you want a specific validation to execute independent of this setting, modify the validation and set the validation attribute <strong>Always Execute</strong> to <strong>On</strong>.</p>
<p>This attribute is particularly useful for buttons such as Delete or Cancel where the majority of validations should not be executed.
Users should generally not be stopped from performing such actions if an item is null or of the wrong type.
However, if actions, such as Delete, should only be performed by authorized users it is important to set the corresponding validation attribute <strong>Always Execute</strong> to <strong>On</strong>, or ensure there is an appropriate condition defined on the button itself so that the button is only displayed to authorized users.</p>~' );
    --
    l_property_id := 189;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUTTON_DATABASE_ACTION',
        p_prompt             => 'Database Action',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DATABASE_ACTION',
        p_display_group_name => 'BEHAVIOR',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the type of SQL operation that clicking this button performs.
This selection is used to trigger the specified database action within built-in Automatic Row Processing (DML).
If you do not select a type, database processing executes based on the REQUEST value, which is set based on the <strong>Button Name</strong> defined for this button.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL INSERT action',
        p_return_value  => 'INSERT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL UPDATE action',
        p_return_value  => 'UPDATE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL DELETE action',
        p_return_value  => 'DELETE' );
    --
    l_property_id := 190;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'PARENT_REGION',
        p_prompt                   => 'Parent Region',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'PARENT_REGION_ID',
        p_display_group_name       => 'LAYOUT',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE',
        p_help_text                => q'~Select the parent region in which the region will be displayed.~' );
    --
    l_property_id := 191;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DISPLAY_POINT',
        p_prompt             => 'Position',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DISPLAY_POINT',
        p_display_group_name => 'LAYOUT',
        p_lov_type           => 'DISPLAY_POINTS',
        p_help_text          => q'~Select the position used to display this component. The positions are derived from the appropriate page or region template display points.~' );
    --
    l_property_id := 192;
    store_process_point_property (
        p_property_id => l_property_id,
        p_name        => 'PROCESS_POINT' ); -- $$$ should be renamed into PAGE_PROCESS_POINT
    --
    l_property_id := 193;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'REGION_SQL',
        p_prompt               => 'SQL Query',
        p_property_type        => 'SQL',
        p_db_column            => 'PLUG_SOURCE',
        p_display_length       => 80,
        p_max_length           => 32767,
        p_sql_min_column_count => 1,
        p_display_group_name   => 'SOURCE',
        p_help_text            => q'~Enter the SQL source for this component.~' );
    --
    l_property_id := 194;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'REGION_PLAIN',
        p_prompt             => 'Text',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'PLUG_SOURCE',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'SOURCE',
        p_help_text          => q'~Enter the text source for this component.~' );
    --
    l_property_id := 195;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DIALOG_HEIGHT',
        p_prompt             => 'Height',
        p_property_type      => 'TEXT',
        p_db_column          => 'DIALOG_HEIGHT',
        p_display_length     => 10,
        p_max_length         => 20,
        p_display_group_name => 'DIALOG',
        p_help_text          => q'~Enter the height of the dialog, in pixels, for example, <strong><em>500</em></strong>.
The substitution string <code>#DIALOG_HEIGHT#</code> in the dialog initialization code of the page template is replaced with the value entered.~' );
    --
    l_property_id := 196;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'AUTHENTICATION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'SCHEME_TYPE',
        p_display_group_name => 'SETTINGS',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Select which authentication scheme is applied. The following authentication types are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may have additional authentication type plug-ins available here.
These are displayed as <em>&lt;My Plugin&gt; [Plug-in]</em>.
The following plug-ins are installed in this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 197;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'AUTHENTICATION_SUBSCRIBED_FROM',
        p_prompt                   => 'Master',
        p_property_type            => 'SUBSCRIPTION',
        p_db_column                => 'REFERENCE_ID',
        p_display_group_name       => 'SUBSCRIPTION',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_authentication,
        p_lov_component_type_scope => 'WORKSPACE',
        p_help_text                => q'~Use this field to base this authentication scheme on an existing authentication scheme in this application or another application in your workspace.~' );
    --
    l_property_id := 198;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PLSQL_CODE',
        p_prompt             => 'PL/SQL Code',
        p_property_type      => 'PLSQL',
        p_db_column          => 'PLSQL_CODE',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'SOURCE',
        p_help_text          => q'~Enter the PL/SQL source utilized within this authentication scheme.~' );
    --
    l_property_id := 199;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'VERIFY_FUNCTION_NAME',
        p_prompt             => 'Verify Function Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'VERIFICATION_FUNCTION',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'SESSION_NOT_VALID',
        p_examples           => q'~<dl>
<dt>Restrict the use of an application to specific business hours</dt>
<dd><pre>
function check_business_hours return boolean
is
begin
    return to_char(to_char(sysdate, 'hh24:mi')) between '08:00' and '17:00';
end check_business_hours;
</pre></dd>
</dl>~',
        p_help_text          => q'~Enter the name of a function that is called after the session sentry has returned successfully.
The function can either be defined in the <strong>PL/SQL code</strong> attribute or in the database.
The result must be boolean, with TRUE representing the user is verified, and FALSE representing the user has failed verification.~' );
    --
    l_property_id := 200;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INVALID_SESSION_TYPE',
        p_prompt             => 'Redirect to',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'INVALID_SESSION_TYPE',
        p_display_group_name => 'SESSION_NOT_VALID',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select where &PRODUCT_NAME. redirects to if the session is not valid.
This happens if the <strong>Sentry Function</strong> or the <strong>Verify Function</strong> returns FALSE on normal page requests, but not on Ajax requests.</p>
<p>Note: If using the <strong>Custom</strong> Authentication Scheme Type, the <strong>Sentry Function</strong> attribute in the Authentication Scheme can also be utilized.
For plug-in Authentication Scheme Types this is defined by the plug-in developer in the <strong>Session Sentry Function Name</strong> callback.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Login Page',
        p_return_value  => 'LOGIN',
        p_help_text     => q'~Redirect to the Login URL defined in the application attributes.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Built-In Login Page',
        p_return_value  => 'BUILTIN_LOGIN',
        p_help_text     => q'~Redirect to the URL defined in <code>wwv_flow_custom_auth_std.login_page</code>, provided by &PRODUCT_NAME..~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'URL',
        p_return_value  => 'URL',
        p_help_text     => q'~Redirect to the URL you enter, such as <code>f?p=&amp;APP_ID.:101:&amp;SESSION</code>.~' );
    --
    l_property_id := 201;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INVALID_SESSION_URL',
        p_prompt             => 'Invalid Session URL',
        p_property_type      => 'LINK',
        p_db_column          => 'INVALID_SESSION_URL',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'SESSION_NOT_VALID',
        p_help_text          => q'~Enter a URL that &PRODUCT_NAME. redirects to if the session sentry function returns FALSE. This would normally be a URL to a login page or single sign-on portal. For example, f?p=&amp;APP_ID.:101:&amp;SESSION~' );
    --
    l_property_id := 202;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PRE_AUTHENTICATION_PROCEDURE_NAME',
        p_prompt             => 'Pre-Authentication Procedure Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'PRE_AUTH_PROCESS',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'LOGIN_PROCESSING',
        p_help_text          => q'~<p>Enter the name of a procedure to be executed after the login page is submitted and just before credentials verification is performed.
The procedure can be defined in the <strong>PL/SQL Code</strong> attribute or within the database.</p>
<p>Authentication schemes, where user credential checking is done outside of &PRODUCT_NAME., typically do not execute the Pre-Authentiation procedure.
Examples include <em>HTTP Header Variable</em>, <em>Oracle Application Server Single Sign-On</em> and custom authentication schemes that use <code>apex_authentication.post_login</code> instead of <code>apex_authentication.login</code>.</p>~' );
    --
    l_property_id := 203;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'POST_AUTHENTICATION_PROCEDURE_NAME',
        p_prompt             => 'Post-Authentication Procedure Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'POST_AUTH_PROCESS',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'LOGIN_PROCESSING',
        p_examples           => q'~<dl>
<dt>Remove the domain from the authenticated username, i.e. transforms <code>JOHNDOE@EXAMPLE.COM</code> to <code>JOHNDOE</code></dt>
<dd><pre>
apex_custom_auth.set_user (
    p_user => regexp_replace(:APP_USER, '@.*', null) );
</pre></dd>
</dl>~',
        p_help_text          => q'~Enter the name of a procedure to be executed by the &PRODUCT_NAME. login procedure (login API) after the authentication step (login credentials verification).
The login procedure executes this code after performing normal login procedure duties, which include setting a cookie and registering the session, but before it redirects to the desired application page.
The procedure can be defined in the <strong>PL/SQL Code</strong> attribute or in the database.~' );
    --
    l_property_id := 204;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LOGOUT_TYPE',
        p_prompt             => 'Logout Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'LOGOUT_TYPE',
        p_is_query_only      => true,
        p_display_group_name => 'POST_LOGOUT_URL',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Enter where &PRODUCT_NAME. redirects to after logout.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Home Page',
        p_return_value  => 'HOME',
        p_help_text     => q'~Redirect to the Home Link value specified in the application attributes.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'URL',
        p_return_value  => 'URL',
        p_help_text     => q'~Redirect to the URL you enter, such as <code>f?p=&amp;APP_ID.:101:&amp;SESSION</code>.~' );
    --
    l_property_id := 205;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'POST_LOGOUT_URL',
        p_prompt             => 'Post-Logout URL',
        p_property_type      => 'TEXT',
        p_db_column          => 'LOGOUT_URL',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'POST_LOGOUT_URL',
        p_help_text          => q'~Enter the URL where your application is redirected after &PRODUCT_NAME. has cleaned up the user session.~' );
    --
    l_property_id := 206;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COOKIE_NAME',
        p_prompt             => 'Cookie Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'COOKIE_NAME',
        p_display_length     => 32,
        p_max_length         => 255,
        p_display_group_name => 'SESSION_SHARING',
        p_help_text          => q'~<p>Enter the name for the session cookie, which is required to identify a &PRODUCT_NAME. session, together with the session id in the URL.
If you do not specify a session cookie name, &PRODUCT_NAME. utilizes a default value.</p>
<p>By entering the same cookie name in multiple applications, each of the applications can utilize the same session cookie.
Therefore, a URL call from a &PRODUCT_NAME. application to another &PRODUCT_NAME. application, with the same cookie name, does not require the user to re-authenticate, because the new application can utilize the existing session cookie.</p>~' );
    --
    l_property_id := 207;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COOKIE_PATH',
        p_prompt             => 'Cookie Path',
        p_property_type      => 'TEXT',
        p_db_column          => 'COOKIE_PATH',
        p_display_length     => 32,
        p_max_length         => 255,
        p_display_group_name => 'SESSION_SHARING',
        p_help_text          => q'~<p>Enter the URL path attribute for the session cookie.
If the start of the request URL does not match the path specified, the cookie is not sent to the browser.
The default value is the path of the request URL.</p>
<p>Use this attribute to specify a less restrictive path, when you wish to share session cookies between applications that have initially conforming URL paths.</p>~' );
    --
    l_property_id := 208;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COOKIE_DOMAIN',
        p_prompt             => 'Cookie Domain',
        p_property_type      => 'TEXT',
        p_db_column          => 'COOKIE_DOMAIN',
        p_display_length     => 32,
        p_max_length         => 255,
        p_display_group_name => 'SESSION_SHARING',
        p_help_text          => q'~<p>Enter the URL domain attribute for the session cookie.
If the tail comparison of the request URL does not match the domain specified, the cookie is not sent to the browser.
The default value is the host name of the request URL.</p>
<p>Use this attribute to specify a less restrictive domain, when you wish to share session cookies between applications that have common domains.
For example, <code>my.yahoo.com</code>, <code>sport.yahoo.com</code>, and <code>finance.yahoo.com</code> could all access the same cookie if a domain of <code>yahoo.com</code> is entered.</p>~' );
    --
    l_property_id := 209;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SECURE_COOKIE',
        p_prompt             => 'Secure',
        p_property_type      => 'YES NO',
        p_db_column          => 'USE_SECURE_COOKIE_YN',
        p_display_group_name => 'SESSION_SHARING',
        p_help_text          => q'~<p>Specify <strong>On</strong> to enable the session management cookie to be sent from the browser only when the protocol is HTTPS.
Select <strong>Off</strong> to enable the session management cookie to be sent from the browser when the protocol is HTTP or HTTPS.</p>
<p>If SSL is configured on your web server, you should select <strong>On</strong> to strengthen your application's security by preventing unencrypted HTTP traffic.</p>~' );
    --
    l_property_id := 210;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'RAS_MODE',
        p_prompt             => 'RAS Mode',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'RAS_MODE',
        p_display_group_name => 'REAL_APPLICATION_SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select how Oracle Real Application Security (RAS) is used in this application.</p>
<p>If you enable RAS Mode, make sure that the users have privileges to access the application's schema objects.
For example, with <strong>External Users</strong> you can grant database privileges to a <em>RAS Dynamic Application Role</em> and configure it in this authentication scheme as a <em>Dynamic Role</em>.
You can also enable roles via a call to <em>apex_authorization.enable_dynamic_groups</em>, such as within a Post-Authentication procedure.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Disabled',
        p_return_value  => '0',
        p_help_text     => q'~Real Application Security does not get used in the application~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'External Users',
        p_return_value  => '1',
        p_help_text     => q'~For light weight external users, &PRODUCT_NAME. passes <em>true</em> via the <em>is_external</em> parameter to <em>dbms_xs_sessions.assign_user</em>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Internal Users',
        p_return_value  => '2',
        p_help_text     => q'~&PRODUCT_NAME. assumes that all users are internal and passes <em>false</em> via the <em>is_external</em> parameter to <em>dbms_xs_sessions.assign_user</em>.~' );
    --
    l_property_id := 211;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'RAS_DYNAMIC_ROLES',
        p_prompt             => 'Dynamic Roles',
        p_property_type      => 'TEXT',
        p_db_column          => 'RAS_DYNAMIC_ROLES',
        p_display_length     => 60,
        p_max_length         => 4000,
        p_display_group_name => 'REAL_APPLICATION_SECURITY',
        p_help_text          => q'~<p>Enter a colon separated list of Real Application Security Dynamic Roles, such as:
<pre>
DB_EMP:DB_DEPT
</pre></p>
<p>&PRODUCT_NAME. enables these roles on login, by passing them via the <em>enable_dynamic_roles</em> parameter of <em>sys.dbms_xs_sessions.assign_user</em>.</p>~' );
    --
    l_property_id := 212;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'RAS_NAMESPACES',
        p_prompt             => 'Namespaces',
        p_property_type      => 'TEXT',
        p_db_column          => 'RAS_NAMESPACES',
        p_display_length     => 60,
        p_max_length         => 4000,
        p_display_group_name => 'REAL_APPLICATION_SECURITY',
        p_help_text          => q'~<p>Enter a colon separated list of Real Application Security Namespace Templates, such as:
<pre>
NS1:NS2
</pre></p>
<p>&PRODUCT_NAME. enables these namespaces on login, by passing them via the <em>namespaces</em> parameter of <em>sys.dbms_xs_sessions.assign_user</em>.</p>~' );
    --
    l_property_id := 213;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EXCLUDE_TITLE_FROM_TRANSLATION',
        p_prompt             => 'Exclude Title from Translation',
        p_property_type      => 'YES NO',
        p_db_column          => 'EXCLUDE_TITLE_FROM_TRANSLATION',
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~Specify whether you want to exclude this title from translation. This attribute has no effect, except when translating the application into another language.~' );
    --
    l_property_id := 214;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'REGION_HTML_CODE',
        p_prompt             => 'HTML Code',
        p_property_type      => 'HTML',
        p_db_column          => 'PLUG_SOURCE',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'SOURCE',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the region source or content, depending on the <em>Region Type</em> selected.~' );
    --
    l_property_id := 215;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'SOURCE_PAGE_ITEMS_TO_SUBMIT',
        p_prompt                   => 'Page Items to Submit',
        p_property_type            => 'ITEM',
        p_multi_value_delimiter    => ',',
        p_db_column                => 'AJAX_ITEMS_TO_SUBMIT',
        p_display_length           => 50,
        p_max_length               => 4000,
        p_display_group_name       => 'SOURCE',
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~<p>Enter page or application items to be set into session state when the region is refreshed by a partial page refresh.
For multiple items, separate each item name with a comma.
You can type in the name or pick from the list of available items.
If you pick from the list and there is already text entered, then a comma is placed at the end of the existing text, followed by the item name returned from the list.</p>
<p>The region can be refreshed with partial page refresh either during pagination, if <em>Enable Partial Page Refresh</em> is set to <strong>On</strong>), or with the Refresh action within a Dynamic Action.</p>
These page items can then be used in the <em>where condition</em> of the Region Source SQL Query to restrict the query result.</p>~' );
    --
    l_property_id := 216;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SCHEMA_OVERRIDE_ITEM',
        p_prompt             => 'Schema Override Item',
        p_property_type      => 'ITEM',
        p_db_column          => 'PLUG_QUERY_PARSE_OVERRIDE',
        p_display_length     => 45,
        p_max_length         => 4000,
        p_display_group_name => 'ADVANCED',
        p_is_internal        => true,
        p_help_text          => q'~Enter the page or application item to be used as the overriding parsing schema.  This is for internal applications only.~' );
    --
    l_property_id := 217;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'REGION_TEMPLATE',
        p_prompt                   => 'Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'PLUG_TEMPLATE',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~<p>Select a region template to define the appearance and layout of this region.</p>
        <p>Region templates are defined in the application theme.
        When a region is created, the template is automatically set to the default region template defined in the current theme.</p>
        <p>Many themes contain region templates for displaying reports, forms, alerts, buttons, collapsible (hide / show) content, login forms, and so forth.<br>
        Universal Theme provides the following region templates:
          <dl>
            <dt>Alert</dt>
            <dt>Blank with Attributes</dt>
            <dt>Buttons Container</dt>
            <dt>Carousel Container</dt>
            <dt>Collapsible</dt>
            <dt>Content Block</dt>
            <dt>Hero</dt>
            <dt>Inline Dialog</dt>
            <dt>Inline Popup</dt>
            <dt>Interactive Report</dt>
            <dt>Login</dt>
            <dt>Standard</dt>
            <dt>Tabs Container</dt>
            <dt>Title Bar</dt>
            <dt>Wizard Container</dt>
          </dl>
          See Also: <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a>
        </p>~' );
    --
    l_property_id := 218;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ITEM_DISPLAY_POSITION',
        p_prompt             => 'Render Components',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PLUG_ITEM_DISPLAY_POINT',
        p_display_group_name => 'APPEARANCE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select if the region components (sub regions, items, buttons) display above or below the main region content.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Above Content',
        p_return_value  => 'ABOVE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Below Content',
        p_return_value  => 'BELOW' );
    --
    l_property_id := 219;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'REGION_DISPLAY_SELECTOR',
        p_prompt             => 'Region Display Selector',
        p_property_type      => 'YES NO',
        p_db_column          => 'INCLUDE_IN_REG_DISP_SEL_YN',
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~<p>Specify <strong>On</strong> if you want this region included in the Region Display Selector.
This attribute is only utilized if a region type of <em>Region Display Selector</em> is defined and viewable on the page.</p>
<p>When a user clicks on a region name, included in the Region Display Selector, only the selected region shows and all other listed regions are hidden.
The first entry within the Region Display Selector is <em>Show All</em>, which enables the user to show all of the available regions rather than only the previously selected region.</p>
<p>If a region is not displayable, based on the region conditions, then it is not included in the lists of regions within the Region Display Selector.</p>~' );
    --
    l_property_id := 220;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'REGION_IMAGE',
        p_prompt             => 'Region Image',
        p_property_type      => 'TEXT',
        p_db_column          => 'REGION_IMAGE',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'ADVANCED',
        p_deprecated_values  => '$NOT_NULL$',
        p_help_text          => q'~Enter a reference to an image that displays in the upper left of the region. There is no control over the table tag used to display this image.~' );
    
    l_property_id := 221;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'IMAGE_TAG_ATTRIBUTES',
        p_prompt             => 'Image Tag Attributes',
        p_property_type      => 'TEXT',
        p_db_column          => 'REGION_IMAGE_ATTR',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'ADVANCED',
        p_deprecated_values  => '$NOT_NULL$',
        p_help_text          => q'~<p>Enter custom region image attributes to be included in the image HTML. This can be used for attributes such as height or width.</p>~' );
    --
    l_property_id := 222;
    --
    -- region and page caching
    --
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CACHE_MODE',
        p_prompt             => 'Caching',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'CACHE_MODE',
        p_display_group_name => 'SERVER_CACHE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how the &PRODUCT_NAME. engine can cache HTML text on the server side. If caching can be used, it emits the cached result instead of computing it again. This cuts down on expensive computation time on the server. This feature is unrelated to the browser's caching capabilities.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Disabled',
        p_return_value  => 'NOCACHE',
        p_help_text     => q'~Content is not cached and computed for each request.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Enabled',
        p_return_value  => 'CACHE',
        p_help_text     => q'~Content is cached and utilized by all users.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Cache By User',
        p_return_value  => 'USER',
        p_help_text     => q'~Content is cached specifically for each user.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Cache By Session',
        p_return_value  => 'SESSION',
        p_help_text     => q'~Content is cached specifically for each session.~' );
    --
    l_property_id := 223;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CUSTOMIZABLE',
        p_prompt             => 'Customizable',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PLUG_CUSTOMIZED',
        p_display_group_name => 'CUSTOMIZATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select whether display attributes in this region can be customized by the user.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Not Customizable By End Users',
        p_return_value  => '0' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Customizable and Shown By Default',
        p_return_value  => '1' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Customizable and Not Shown By Default',
        p_return_value  => '2' );
    --
    l_property_id := 224;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CUSTOMIZED_OPTION_NAME',
        p_prompt             => 'Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'PLUG_CUSTOMIZED_NAME',
        p_display_length     => 70,
        p_max_length         => 255,
        p_display_group_name => 'CUSTOMIZATION',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the label for the link that displays under this region, which provides access to the region customization page.
This label only displays for regions that are customizable.~' );
    --
    l_property_id := 225;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'REGION_FUNCTION_RETURNING_SQL',
        p_prompt               => 'PL/SQL Function Body returning SQL Query',
        p_property_type        => 'PLSQL FUNCTION BODY SQL',
        p_db_column            => 'PLUG_SOURCE',
        p_display_length       => 80,
        p_max_length           => 32767,
        p_sql_min_column_count => 1,
        p_display_group_name   => 'SOURCE',
        p_help_text            => q'~Enter a PL/SQL Function that returns a SQL query that is used as the basis for this region.~' );
    --
    l_property_id := 226;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BRANCH_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'BRANCH_TYPE',
        p_display_group_name => 'BEHAVIOR',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select the type of page branch to perform.</p>
<dl>
<dt>Redirect branches</dt>
<dd>&PRODUCT_NAME. returns the new page or URL to the browser, which performs a new request before showing the resultant page.
The major advantage is that the end users can refresh the page without losing context, as the browser address contains the full page URL.
The disadvantage is that the browser performs an additional request to show the page.</dd>
<dt>Show Only branches</dt>
<dd>&PRODUCT_NAME. immediately renders the new page as part of the current page processing, without performing another request.
This has the advantage that the browser does not have to perform an additional request.
The major disadvantage is that the browser address will be <code>wwv_flow.accept</code>.
If the end users refreshes the page the browser will ask to resend the data, as the browser address is not a full page URL.</dd>
</dl>
<p>Oracle recommends that you select one of the redirect branch types to provide a better end user experience.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page or URL (Redirect)',
        p_return_value  => 'REDIRECT_URL',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'URL Identified by Item (Redirect)',
        p_return_value  => 'BRANCH_TO_URL_IDENT_BY_ITEM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Returning a URL (Redirect)',
        p_return_value  => 'BRANCH_TO_FUNCTION_RETURNING_URL' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page (Show only)',
        p_return_value  => 'BRANCH_TO_STEP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page Identified by Item (Show only)',
        p_return_value  => 'BRANCH_TO_PAGE_IDENT_BY_ITEM' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Returning a Page (Show only)',
        p_return_value  => 'BRANCH_TO_FUNCTION_RETURNING_PAGE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'PL/SQL procedure [Legacy]',
        p_return_value  => 'PLSQL' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page Processing {not common}',
        p_return_value  => 'BRANCH_TO_PAGE_ACCEPT' );
    --
    l_property_id := 227;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BRANCH_POINT',
        p_prompt             => 'Point',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'BRANCH_POINT',
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select at what point in page rendering or processing this branch is executed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Before Header',
        p_return_value  => 'BEFORE_HEADER',
        p_help_text     => q'~Rendering: Branches before the page renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Submit',
        p_return_value  => 'BEFORE_COMPUTATION',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Processing: Branches after submitting the page and before computation, validations and processes are executed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Validating',
        p_return_value  => 'BEFORE_VALIDATION',
        p_help_text     => q'~Processing: Branches before validations performed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Processing',
        p_return_value  => 'BEFORE_PROCESSING',
        p_help_text     => q'~Processing: Branches before processing performed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Processing',
        p_return_value  => 'AFTER_PROCESSING',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Processing: Branches after processing performed.~' );
    --
    l_property_id := 228;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'TARGET',
        p_prompt             => 'Target',
        p_property_type      => 'LINK',
        p_db_column          => 'TARGET',
        p_display_group_name => 'BEHAVIOR',
        p_help_text          => q'~Click the Target to invoke a modal dialog. You can enter the target that is called when this branch is invoked.~',
        p_comment_text       => q'~$$$ can't we merge that with BUTTON_TARGET?~' );
    --
    l_property_id := 229;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BRANCH_PAGE_NUMBER',
        p_prompt             => 'Page Number',
        p_property_type      => 'PAGE',
        p_db_column          => 'BRANCH_PAGE_NUMBER',
        p_display_length     => 5,
        p_max_length         => 20,
        p_display_group_name => 'BEHAVIOR',
        p_help_text          => q'~Enter the page to which this branch is redirected.~' );
    --
    l_property_id := 230;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'BRANCH_ITEM',
        p_prompt                   => 'Item',
        p_property_type            => 'ITEM',
        p_db_column                => 'BRANCH_ITEM',
        p_display_length           => 30,
        p_max_length               => 255,
        p_display_group_name       => 'BEHAVIOR',
        p_lov_component_type_scope => 'APPLICATION',
        p_help_text                => q'~Enter the page or application item containing a page number or URL to redirect to.
This item can be set by the calling page, or by a process when the page first loads.
Specifying an item provides an easy way to dynamically set the branch for pages that are called from many other pages.~' );
    --
    l_property_id := 231;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'VALID_FOR_URLS',
        p_prompt             => 'Valid for URLs',
        p_property_type      => 'TEXTAREA',
        p_db_column          => 'VALID_FOR_URLS',
        p_display_length     => 64,
        p_max_length         => null,
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~<p>&PRODUCT_NAME. will check whether the URL, for which the credential is being used at run time, matches one of the URLs within this list. If the URL does not match, &PRODUCT_NAME. will raise an error.</p>
<p>Enter each URL on a new line. The URL endpoint being used must start with one of the URLs provided here.</p>
<p>Example:</p>
<pre>
https://www.example.com/restapis/my-protected-api
https://www.example.com/allowed-urls
</pre>
<p>If the credential is used to
invoke a REST service at <strong>https://www.example.com/restapis/my-protected-api/restendpoint</strong> or <strong>https://www.example.com/allowed-urls/page1</strong>, the credential can be used and the call will succeed.</p>
<p>If the credential is
 used to invoke a REST service at <strong>https://localhost/myapis/test</strong> or <strong>https://dev.example.com/myapis/test</strong>, the credential <em>cannot</em> be used and &PRODUCT_NAME. will raise an error.</p>
<p>Note that changing this va
lue requires to re-enter the client secret.</p>~' );
    --
    l_property_id := 232;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BRANCH_PLSQL',
        p_prompt             => 'PL/SQL Code',
        p_property_type      => 'PLSQL',
        p_db_column          => 'BRANCH_PLSQL',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'BEHAVIOR',
        p_help_text          => q'~Enter the PL/SQL code executed by the branch. This PL/SQL procedure is responsible for invoking the necessary URL.~' );
    --
    l_property_id := 233;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SAVE_STATE_BEFORE_BRANCH',
        p_prompt             => 'Save State before Branching',
        p_property_type      => 'YES NO',
        p_db_column          => 'SAVE_STATE_BEFORE_BRANCH_YN',
        p_display_group_name => 'BEHAVIOR',
        p_deprecated_values  => 'Y',
        p_help_text          => q'~Specify whether values in session state are saved to the database before the branch executes.
It is important to set this attribute to <strong>On</strong> if you are navigating to a different application that needs to access the session state values from this application.~' );
    --
    l_property_id := 234;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_BUTTON_PRESSED',
        p_prompt                   => 'When Button Pressed',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'WHEN_BUTTON_ID',
        p_display_group_name       => 'SERVER_SIDE_CONDITION',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_button,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~If you want this page processing component to execute only when the specified button is clicked, select a button from the list. You can incorporate this button condition with other conditions to further refine when this page processing component executes.~' );
    --
    l_property_id := 245;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALWAYS_EXECUTE',
        p_prompt             => 'Always Execute',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALWAYS_EXECUTE',
        p_display_group_name => 'VALIDATION',
        p_help_text          => q'~<p>Specify whether this validation is always executed.
If set to <strong>On</strong>, this validation is always evaluated, irrespective of the <em>Execute Validations</em> setting defined against the button that submitted the page.
If set to <strong>Off</strong>, this validation is only evaluated if the triggering button has an <em>Execute Validations</em> setting of <strong>On</strong>.</p>
<p>Note: This validation must also comply with the conditions specified before being evaluated.</p>~' );
    --
    l_property_id := 246;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ERROR_MESSAGE_DISPLAY_LOCATION',
        p_prompt             => 'Display Location',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'ERROR_DISPLAY_LOCATION',
        p_display_group_name => 'ERROR',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select where the error message displays for this validation.
Validation error messages display on a separate error page, or inline with the existing page.
Inline error messages display underneath the <em>Associated Item</em> label and/or in a Notification area, defined as part of the page template.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Inline with Field and in Notification',
        p_return_value  => 'INLINE_WITH_FIELD_AND_NOTIFICATION' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Inline with Field',
        p_return_value  => 'INLINE_WITH_FIELD' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Inline in Notification',
        p_return_value  => 'INLINE_IN_NOTIFICATION' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'On Error Page',
        p_return_value  => 'ON_ERROR_PAGE' );
    --
    l_property_id := 247;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'ASSOCIATED_ITEM',
        p_prompt                   => 'Associated Item',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'ASSOCIATED_ITEM',
        p_display_group_name       => 'ERROR',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_page_item,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Select the item where this validation error message displays.
If you select an item and the <em>Error message display location</em> includes "Inline with Field", the error message displays underneath the label of the specified item.
If the <em>Error message display location</em> does not include "Inline with Field", this attribute has no impact on the display of the error message.~' );
    --
    l_property_id := 248;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ASSOCIATED_COLUMN',
        p_prompt             => 'Associated Column',
        p_property_type      => 'REGION COLUMN',
        p_db_column          => 'ASSOCIATED_COLUMN',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'ERROR',
        p_help_text          => q'~Select the column associated with this page processing component.~' );
    --
    l_property_id := 249;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EXECUTION_SCOPE',
        p_prompt             => 'Execution Scope',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'ONLY_FOR_CHANGED_ROWS',
        p_display_group_name => 'SERVER_SIDE_CONDITION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the execution scope for this editable region validation.
Select <strong>For Created and Modified Rows</strong>, if you want this validation to be evaluated for new and updated rows in your editable region.
If you want your validation to be evaluated for every submitted row, select <strong>All Submitted Rows</strong>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'For Created and Modified Rows',
        p_return_value  => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'All Submitted Rows',
        p_return_value  => 'N' );
    --
    l_property_id := 250;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EXECUTE_CONDITION',
        p_prompt             => 'Execute Condition',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'EXEC_COND_FOR_EACH_ROW',
        p_display_group_name => 'SERVER_SIDE_CONDITION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how many times this editable region validation is executed.
If you refer to <em>Columns</em> in your condition, you must specify <strong>For Each Row</strong> in order for the condition to be evaluated for each row of the editable region.
Otherwise, set to <strong>Once</strong> to evaluate the validations one time for the region.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'For Each Row',
        p_return_value  => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once',
        p_return_value  => 'N' );
    --
    l_property_id := 251;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'VALIDATION_REGION',
        p_prompt                   => 'Editable Region',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'TABULAR_FORM_REGION_ID',
        p_display_group_name       => 'VALIDATION',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE',
        p_help_text                => q'~Select the associated region for this validation.~' );
    --
    l_property_id := 252;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUTTON_ACTION',
        p_prompt             => 'Action',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'BUTTON_ACTION',
        p_display_group_name => 'BEHAVIOR',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the action to be performed when this button is clicked.~',
        p_comment_text       => q'~$$$ Missing ability to create "Download Printable Report Query" button, which would create a submit button with corresponding branch to invoke a report query.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Submit Page',
        p_return_value  => 'SUBMIT',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Submit the current page with a REQUEST value equal to the Button Name.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Redirect to Page in this Application',
        p_return_value  => 'REDIRECT_PAGE',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Redirect to a page in the current application, with optional additional attributes for resetting pagination, setting the request value, clearing cache and setting item values, on the target page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Redirect to Page in a different Application',
        p_return_value  => 'REDIRECT_APP',
        p_help_text     => q'~<p>Redirect to a page in a different application, with optional additional attributes for resetting pagination, setting the request value, clearing cache and setting item values, on the target page.</p>
<p>Note: Redirecting to a page in different application relies on deep linking being enabled in the target application. To enable deep linking, go to the application's <em>Security</em> attributes and enable <em>Deep Linking</em>.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Redirect to URL',
        p_return_value  => 'REDIRECT_URL',
        p_help_text     => q'~Redirect to the target URL you specify.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Defined by Dynamic Action',
        p_return_value  => 'DEFINED_BY_DA',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Does not submit the page or redirect to another page. Instead the button's behavior is defined by a Dynamic Action.~' );
    --
    /*
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Download Printable Report Query',
        p_return_value  => 'REPORT',
        p_comment_text  => q'~Download the report query.~' );
    */
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Reset Page',
        p_return_value  => 'RESET',
        p_is_deprecated => 'Y',
        p_help_text     => q'~Reset the page by performing a complete page refresh, not a Partial Page Refresh (PPR).~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Next Page',
        p_return_value  => 'NEXT_PAGE',
        p_is_deprecated => 'Y',
        p_help_text     => q'~Redirect to the next page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Previous Page',
        p_return_value  => 'PREVIOUS_PAGE',
        p_is_deprecated => 'Y',
        p_help_text     => q'~Redirect to the previous page.~' );
    --
    l_property_id := 253;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'USER_INTERFACE',
        p_prompt                => 'User Interface',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'USER_INTERFACE_ID',
        p_display_group_name    => 'APPEARANCE',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_user_interface,
        p_help_text             => q'~Shows the user interface for this page.~' );
    --
    l_property_id := 254;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'DIALOG_TEMPLATE',
        p_prompt                   => 'Dialog Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'STEP_TEMPLATE',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_page_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~<p>Select a dialog template to define the appearance and layout of this page.</p>
        <p>Dialog templates are defined in the application theme.
        When a dialog page is created, the template is automatically set to <em>Theme Default</em> which will render the page using the default page template defined in the current theme.
        Making a selection here overrides the default dialog template.</p>
        <p>Universal Theme provides the following dialog templates:
          <dl>
            <dt>Modal Dialog</dt>
              <dd>Useful for most dialog use cases.</dd>
            <dt>Wizard Modal Dialog</dt>
              <dd>Provides a streamlined user interface and is well suited for modal dialog wizards, alerts, notifications, or small forms.</dd>
          </dl>
          See Also: <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a>
        </p>~' );
    --
    l_property_id := 255;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'OVERWRITE_NAVIGATION_LIST',
        p_prompt             => 'Override User Interface Level',
        p_property_type      => 'YES NO',
        p_db_column          => 'OVERWRITE_NAVIGATION_LIST',
        p_display_group_name => 'NAVIGATION_MENU',
        p_help_text      => q'~Specify whether the default User Interface level navigation is overwritten. This can be used to either have no navigation for the current page, or choose alternative navigation for the current page (for example to provide different navigation around a sub-section of your app).~' );
    --
    l_property_id := 256;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'NAVIGATION_LIST',
        p_prompt                => 'List',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'NAVIGATION_LIST_ID',
        p_display_group_name    => 'NAVIGATION_MENU',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_list,
        p_help_text             => q'~Select the navigation menu list utilized for navigation on this page.~' );
    --
    l_property_id := 257;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'NAVIGATION_LIST_TEMPLATE',
        p_prompt                   => 'List Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'NAVIGATION_LIST_TEMPLATE_ID',
        p_display_group_name       => 'NAVIGATION_MENU',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_list_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~Select the navigation menu list template utilized with the navigation menu list on this page.~' );
    --
    l_property_id := 258;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'USER_INTERFACE_TYPE',
        p_prompt             => 'User Interface Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'UI_TYPE_ID',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select a user interface type for this page. Select <strong>Desktop</strong> for pages primarily designed for desktop use. For other interfaces, such as smartphones or tablets, select the corresponding user interface.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Desktop',
        p_return_value  => 'DESKTOP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'jQuery Mobile Smartphone',
        p_return_value  => 'JQM_SMARTPHONE' );
    --
    l_property_id := 259;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'AUTO_DETECT',
        p_prompt             => 'Auto Detect',
        p_property_type      => 'YES NO',
        p_db_column          => 'USE_AUTO_DETECT',
        p_display_group_name => 'ATTRIBUTES',
        p_help_text          => q'~Specify whether the appropriate user interface is automatically detected.
If auto-detection is <strong>On</strong>, the user is automatically redirected to the most appropriate login page or home page, based on the size of the device in use.
Media Queries are utilized to determine the corresponding user interface to display.~' );
    --
    l_property_id := 260;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DEFAULT',
        p_prompt             => 'Default',
        p_property_type      => 'YES NO',
        p_db_column          => 'IS_DEFAULT',
        p_display_group_name => 'ATTRIBUTES',
        p_help_text          => q'~Specify whether or not the user interface is the default application interface.~' );
    --
    l_property_id := 261;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HOME_URL',
        p_prompt             => 'Home URL',
        p_property_type      => 'LINK',
        p_db_column          => 'HOME_URL',
        p_display_length     => 60,
        p_max_length         => 2000,
        p_display_group_name => 'NAVIGATION',
        p_help_text          => q'~Specify the application home page for the user interface.~' );
    --
    l_property_id := 262;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LOGIN_URL',
        p_prompt             => 'Login URL',
        p_property_type      => 'LINK',
        p_db_column          => 'LOGIN_URL',
        p_display_length     => 60,
        p_max_length         => 2000,
        p_display_group_name => 'NAVIGATION',
        p_help_text          => q'~Specify the application login page for the user interface.~' );
    --
    l_property_id := 263;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'THEME',
        p_prompt                => 'Theme',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'THEME_ID',
        p_display_group_name    => 'ATTRIBUTES',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_theme,
        p_help_text             => q'~Select the theme to associate with the current user interface.~' );
    --
    l_property_id := 264;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GLOBAL_PAGE',
        p_prompt             => 'Global Page',
        p_property_type      => 'PAGE',
        p_db_column          => 'GLOBAL_PAGE_ID',
        p_display_group_name => 'ATTRIBUTES',
        p_help_text          => q'~Select the global page for the current user interface, if one is not defined.~' );
    --
    l_property_id := 265;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PLUGIN_CATEGORY',
        p_prompt             => 'Category',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'CATEGORY',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select under which category the plug-in is displayed in the user interface.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Component',
        p_return_value  => 'COMPONENT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Effect',
        p_return_value  => 'EFFECT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Execute',
        p_return_value  => 'EXECUTE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Initialize',
        p_return_value  => 'INIT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Miscellaneous',
        p_return_value  => 'MISC' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Navigation',
        p_return_value  => 'NAVIGATION' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Notification',
        p_return_value  => 'NOTIFICATION' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Style',
        p_return_value  => 'STYLE' );
    --
    l_property_id := 266;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INCLUDE_SEARCH_BAR',
        p_prompt             => 'Include Search Bar',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_SEARCH_BAR',
        p_display_group_name => 'SEARCH_BAR',
        p_help_text          => q'~<p>A search bar is displayed above the report.  If you include the search bar, you can also fully customize which functions to display.</p>
The search bar can include the following components:
<dl>
  <dt>Search Field</dt>
    <dd>Displays a text field in the search bar used to enter search criteria.</dd>
  <dt>Finder Drop Down</dt>
    <dd>Embedded within the <em>Search Field</em>, displays a drop down list of displayed columns. If a column is selected then the search is only performed against the values in the selected column.</dd>
  <dt>Reports Select List</dt>
    <dd>Displays a list of all available reports for the user, including primary, alternate, public and private reports. This list is only displayed if a report, other than the primary report, is accessible to the user.</dd>
  <dt>Rows Per Page Selector</dt>
    <dd>Displays a select list used to select the number of rows to display per page.</dd>
  <dt>Actions Menu</dt>
    <dd>Displays the Actions menu button, which generally includes a comprehensive set of options. These options enables the end user to customize the report based upon the functions you make available.</dd>
</dl>~' );
    --
    l_property_id := 267;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COLUMN_NAME',
        p_prompt             => 'Column Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'DB_COLUMN_NAME',
        p_display_length     => 128,
        p_max_length         => 128,
        p_display_group_name => 'IDENTIFICATION',
        p_help_text          => q'~Enter the column name for this report column.~' );
    --
    l_property_id := 268;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COLUMN_TYPE',
        p_prompt             => 'Data Type',
        p_property_type      => 'HIDDEN',
        p_db_column          => 'COLUMN_TYPE',
        p_display_length     => 32,
        p_max_length         => 255,
        p_display_group_name => 'IDENTIFICATION',
        p_help_text          => q'~Enter the data type for the report column.~' );
    --
    l_property_id := 269;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'IR_COLUMN_GROUP',
        p_prompt                   => 'Group',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'GROUP_ID',
        p_display_group_name       => 'LAYOUT',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_ir_column_group,
        p_lov_component_type_scope => 'PARENT',
        p_help_text                => q'~Select the group for this column. Groups are used to display columns in logical groups in the Single Row View.~' );
    --
    l_property_id := 270;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_BY_DEFAULT',
        p_prompt             => 'Show by Default',
        p_property_type      => 'YES NO',
        p_db_column          => 'DISPLAY_IN_DEFAULT_RPT',
        p_display_group_name => 'LAYOUT',
        p_help_text          => q'~Specify whether or not this column displays in the report by default.~' );
    --
    l_property_id := 271;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'IR_COLUMN_DISPLAY_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DISPLAY_TEXT_AS',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the relevant column display type to define how column values are displayed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Plain Text',
        p_return_value  => 'PLAIN',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Displays the column value.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Plain Text (based on List of Values)',
        p_return_value  => 'PLAIN_LOV',
        p_help_text     => q'~Displays the value returned from the list of values specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Rich Text',
        p_return_value  => 'RICH_TEXT',
        p_help_text     => q'~Renders the column using HTML or Markdown.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Link',
        p_return_value  => 'LINK',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Displays the column as a link.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Display Image',
        p_return_value  => 'IMAGE',
        p_help_text     => q'~Displays the column as an image.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Download BLOB',
        p_return_value  => 'DOWNLOAD',
        p_help_text     => q'~Displays the column as a link to download the file.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Percent Graph',
        p_return_value  => 'PCT_GRAPH',
        p_help_text     => q'~Displays the column as a percentage graph. The value retrieved must be between 0 and 100.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Remove HTML',
        p_return_value  => 'STRIP_HTML',
        p_help_text     => q'~Displayes the column value, after removing HTML tags and escaping special characters.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Hidden Column',
        p_return_value  => 'HIDDEN_COLUMN',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Does not display the column, but it is referenceable.~' );
    --
    l_property_id := 272;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COLUMN_HEADING',
        p_prompt             => 'Heading',
        p_property_type      => 'TEXT',
        p_db_column          => 'REPORT_LABEL',
        p_display_length     => 50,
        p_max_length         => 4000,
        p_display_group_name => 'HEADING',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the report column heading text.~' );
    --
    l_property_id := 273;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SYNC_FORM_LABEL',
        p_prompt             => 'Use Column Heading',
        p_property_type      => 'YES NO',
        p_db_column          => 'SYNC_FORM_LABEL',
        p_display_group_name => 'SINGLE_ROW_VIEW',
        p_help_text          => q'~Specify this attribute if you want to use the column heading text as the label in the Detail View.~' );
    --
    l_property_id := 274;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SINGLE_ROW_VIEW_LABEL',
        p_prompt             => 'Label',
        p_property_type      => 'TEXT',
        p_db_column          => 'COLUMN_LABEL',
        p_display_length     => 50,
        p_max_length         => 4000,
        p_display_group_name => 'SINGLE_ROW_VIEW',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the column label to use in the Single Row View.~' );
    --
    l_property_id := 275;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HEADING_ALIGNMENT',
        p_prompt             => 'Alignment',
        p_property_type      => 'PILL BUTTONS',
        p_db_column          => 'HEADING_ALIGNMENT',
        p_display_group_name => 'HEADING',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the heading alignment. Note that your alignment selection must be supported by the selected report template.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'start',
        p_return_value  => 'LEFT',
        p_icon          => 'icon-align-start' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'center',
        p_return_value  => 'CENTER',
        p_icon          => 'icon-align-center' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'end',
        p_return_value  => 'RIGHT',
        p_icon          => 'icon-align-end' );
    --
    l_property_id := 276;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COLUMN_ALIGNMENT',
        p_prompt             => 'Column Alignment',
        p_property_type      => 'PILL BUTTONS',
        p_db_column          => 'COLUMN_ALIGNMENT',
        p_display_group_name => 'LAYOUT',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the column display alignment. Note that your alignment selection must be supported by the selected report template.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'start',
        p_return_value  => 'LEFT',
        p_icon          => 'icon-align-start' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'center',
        p_return_value  => 'CENTER',
        p_icon          => 'icon-align-center' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'end',
        p_return_value  => 'RIGHT',
        p_icon          => 'icon-align-end' );
    --
    l_property_id := 277;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_HIDE',
        p_prompt             => 'Hide',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_HIDE',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can hide this column.~' );
    --
    l_property_id := 278;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_SORT',
        p_prompt             => 'Sort',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_SORTING',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can sort this column.~' );
    --
    l_property_id := 279;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_FILTER',
        p_prompt             => 'Filter',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_FILTERING',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can filter this column.~' );
    --
    l_property_id := 280;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_HIGHLIGHT',
        p_prompt             => 'Highlight',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_HIGHLIGHTING',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can highlight this column.~' );
    --
    l_property_id := 281;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_CONTROL_BREAK',
        p_prompt             => 'Control Break',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_CTRL_BREAKS',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can utilize this column in a control break.~' );
    --
    l_property_id := 282;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_AGGREGATE',
        p_prompt             => 'Aggregate',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_AGGREGATIONS',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can aggregate this column.~' );
    --
    l_property_id := 283;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_COMPUTE',
        p_prompt             => 'Compute',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_COMPUTATIONS',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can utilize this column in a computational column they define.~' );
    --
    l_property_id := 284;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_CHART',
        p_prompt             => 'Chart',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_CHARTING',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can use this column within a chart.~' );
    --
    l_property_id := 285;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_GROUP_BY',
        p_prompt             => 'Group By',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_GROUP_BY',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end users can group by this column.~' );
    --
    l_property_id := 286;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'IR_COLUMN_FILTER_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'RPT_SHOW_FILTER_LOV',
        p_display_group_name => 'COLUMN_FILTER',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how to derive the filters provided in the report column drop down list.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'None',
        p_return_value  => 'N',
        p_help_text     => q'~There is no column filter provided for this report.~',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Default Based on Column Type',
        p_return_value  => 'D',
        p_help_text     => q'~STRING and NUMBER columns derive the list of values from distinct column values. DATE columns derive the list of values from predefined date ranges.~',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Use Defined List of Values to Filter Exact Match',
        p_return_value  => 'S',
        p_help_text     => q'~Derive the list of values from a defined list of values SQL query to create an equal match filter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Use Defined List of Values to Filter Word Contains',
        p_return_value  => 'C',
        p_help_text     => q'~Derive the list of values from a defined list of values SQL query to create a contains filter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Use Named List of Values to Filter Exact Match',
        p_return_value  => '1',
        p_help_text     => q'~Derive the list of values from the named list of values to create an equal filter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Use Named List of Values to Filter Word Contains',
        p_return_value  => '2',
        p_help_text     => q'~Derive the list of values from the named list of values to create a contains filter.~' );
    --
    l_property_id := 288;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'REPORT_FILTER_DATE_RANGES',
        p_prompt             => 'Date Ranges',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'RPT_FILTER_DATE_RANGES',
        p_display_group_name => 'COLUMN_FILTER',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the list of values date restrictions.
For date columns, the list of values can display past time, future time, or both in the filter for the report column drop down list.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'All',
        p_return_value  => 'ALL' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Past',
        p_return_value  => 'PAST' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Future',
        p_return_value  => 'FUTURE' );
    --
    l_property_id := 289;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HTML_EXPRESSION',
        p_prompt             => 'HTML Expression',
        p_property_type      => 'HTML',
        p_db_column          => 'COLUMN_HTML_EXPRESSION',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
  <dt>Change the background and text colors</dt>
    <dd><pre>&lt;span style="background-color:yellow; color:red;">#EMPLOYEE_STATUS#&lt;/span></pre></dd>
  <dt>Use Heading 2</dt>
    <dd><pre>&lt;h2>#EMPLOYEE_NAME#&lt;/h2></pre></dd>
  <dt>Use a dynamic class from the SQL query</dt>
    <dd><pre>&lt;span class="#COLUMN_CLASS#">#EMPLOYEE_STATUS#&lt;/span></pre><br>
    The <code>#COLUMN_CLASS#</code> and <code>#EMPLOYEE_STATUS#</code> substitutions in this example are columns selected in the report SQL query.</dd>
  <dt>Show an image, and a tooltip on hover</dt>
    <dd><pre>&lt;img src="#IMAGE_NAME#" title="#EMPLOYEE_NAME#"&gt;</pre></dd>
</dl>~',
        p_display_group_name => 'COLUMN_FORMATTING',
        p_is_translatable    => true,
        p_help_text          => q'~Enter HTML expressions to be shown in this column. Use <code>#COLUMN#</code> syntax to show column values in HTML.~' );
    --
    l_property_id := 290;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LINK_TEXT',
        p_prompt             => 'Link Text',
        p_property_type      => 'TEXT',
        p_db_column          => 'COLUMN_LINKTEXT',
        p_display_length     => 60,
        p_max_length         => 4000,
        p_display_group_name => 'LINK',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the HTML text to be shown as a link. Use an image tag to display images, or pick from the list of default images.~' );
    --
    l_property_id := 291;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LINK_ATTRIBUTES',
        p_prompt             => 'Link Attributes',
        p_property_type      => 'TEXT',
        p_db_column          => 'COLUMN_LINK_ATTR',
        p_display_length     => 62,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>Open the link in a new window</dt>
<dd><pre>target="_blank"</pre></dd>
<dt>Show a tooltip on hover</dt>
<dd><pre>title="Click to edit"</pre></dd>
</dl>~',
        p_display_group_name => 'LINK',
        p_help_text          => q'~Enter additional column link attributes to be included in the <code>&lt;a href=...&gt;</code> tag, including link target, classes or styles.~' );
    --
    l_property_id := 292;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'IR_PAGINATION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PAGINATION_TYPE',
        p_display_group_name => 'PAGINATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select the pagination type for this report. Often only a certain number of rows of a report display on a page.
In order to include additional rows, the application end user must to navigate to the next page of the report.</p>
<p>Pagination provides the end user with information about the number of rows and the current position within the result set.</p>
<p>Pagination can be shown above or below the report (or both) and can be positioned on the left or right.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Row Ranges X to Y of Z',
        p_return_value  => 'ROWS_X_TO_Y_OF_Z' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Row Ranges X to Y',
        p_return_value  => 'ROWS_X_TO_Y' );
    --
    l_property_id := 293;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGINATION_DISPLAY_POSITION',
        p_prompt             => 'Display Position',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PAGINATION_DISPLAY_POSITION',
        p_display_group_name => 'PAGINATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select where the report pagination displays.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Bottom - Left',
        p_return_value  => 'BOTTOM_LEFT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Bottom - Right',
        p_return_value  => 'BOTTOM_RIGHT',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Top - Left',
        p_return_value  => 'TOP_LEFT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Top - Right',
        p_return_value  => 'TOP_RIGHT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Top and Bottom - Left',
        p_return_value  => 'TOP_AND_BOTTOM_LEFT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Top and Bottom - Right',
        p_return_value  => 'TOP_AND_BOTTOM_RIGHT',
        p_is_quick_pick => 'Y' );
    --
    l_property_id := 294;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_NULL_VALUES_AS',
        p_prompt             => 'Show Null Values as',
        p_property_type      => 'TEXT',
        p_db_column          => 'SHOW_NULLS_AS',
        p_display_length     => 16,
        p_max_length         => 255,
        p_display_group_name => 'APPEARANCE',
        p_help_text          => q'~Enter the text you want to display for null columns.~' );
    --
    l_property_id := 295;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'MAXIMUM_ROW_COUNT',
        p_prompt             => 'Maximum Rows to Process',
        p_property_type      => 'INTEGER',
        p_db_column          => 'MAX_ROW_COUNT',
        p_min_value          => 1,
        p_display_group_name => 'PERFORMANCE',
        p_help_text          => q'~<p>Enter the maximum number of rows which can be processed by the query. If the maximum number of rows to be processed exceedes the limit, then an error message will be displayed instead of data.
A custom message can be defined by specifying a message in <strong>More Data Found</stron>.</p>
<p>This attribute is to safeguard against end users performing actions, such as sorts or aggregations, on all the records in very large tables or views, which might not be performant.
Instead, the number of records being processed must first be reduced by adding additional filters.</p>~' );
    --
    l_property_id := 297;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WHEN_NO_DATA_FOUND_MESSAGE',
        p_prompt             => 'When No Data Found',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'NO_DATA_FOUND_MESSAGE',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'MESSAGES',
        p_is_translatable    => true,
        p_help_text          => q'~Enter a text message to display if the query does not return any rows.~' );
    --
    l_property_id := 298;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'NO_DATA_FOUND_ICON_CLASSES',
        p_prompt             => 'No Data Found Icon',
        p_property_type      => 'ICON',
        p_db_column          => 'NO_DATA_FOUND_ICON_CLASSES',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>Show an arrow pointing right</dt>
<dd><pre>fa-chevron-right</pre></dd>
</dl>~',
        p_display_group_name => 'MESSAGES',
        p_lov_type           => 'DISTINCT',
        p_help_text          => q'~<p>Enter the classes for the icon you want to add to the component. You may add multiple classes by separating them with spaces.</p>
        <p>If your theme uses Font APEX then review the <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a> to view available icons and modifiers.</p>~',
        p_comment_text       => '$$$ CSS class names are depending on the theme. Different icon classes are supported. For new universal theme it is http://fontawesome.io/icons/ we should mention something in the help.'  );
    --
    l_property_id := 302;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'SAVE_PUBLIC_REPORT_AUTHORIZATION',
        p_prompt                => 'Save Public Report Authorization',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'SAVE_RPT_PUBLIC_AUTH_SCHEME',
        p_display_group_name    => 'ACTIONS_MENU',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_authorization,
        p_help_text             => q'~Select an authorization scheme to restrict who can save public report definitions.
To enable an end user to save a public report, this authorization scheme must evaluate to TRUE.
If no authorization scheme is provided then reports may be saved by any user.~',
        p_comment_text          => q'~$$$ should we put it into display group SECURITY?~' );
    --
    l_property_id := 306;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SEARCH_BUTTON_LABEL',
        p_prompt             => 'Search Button Label',
        p_property_type      => 'TEXT',
        p_db_column          => 'SEARCH_BUTTON_LABEL',
        p_display_length     => 30,
        p_max_length         => 4000,
        p_display_group_name => 'SEARCH_BAR',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the text for the search button label.~' );
    --
    l_property_id := 307;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'MAXIMUM_ROWS_PER_PAGE',
        p_prompt             => 'Maximum Rows Per Page',
        p_property_type      => 'TEXT',
        p_db_column          => 'MAX_ROWS_PER_PAGE',
        p_display_length     => 7,
        p_max_length         => 4000,
        p_display_group_name => 'SEARCH_BAR',
        p_help_text          => q'~Enter the maximum number to display in the Rows Per Page selector.~',
        p_comment_text       => q'~$$$ should be number, but is a string in the data model~' );
    --
    l_property_id := 308;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DOWNLOAD_FORMATS',
        p_prompt             => 'Formats',
        p_property_type      => 'CHECKBOXES',
        p_db_column          => 'DOWNLOAD_FORMATS',
        p_display_group_name => 'DOWNLOAD',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Check which download formats users can utilize to download the currently displayed columns.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'CSV',
        p_return_value  => 'CSV' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'HTML',
        p_return_value  => 'HTML' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Excel',
        p_return_value  => 'XLSX' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'PDF',
        p_return_value  => 'PDF' );
    --
    --
    l_property_id := 309;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CSV_SEPARATOR',
        p_prompt             => 'CSV Separator',
        p_property_type      => 'TEXT',
        p_db_column          => 'CSV_OUTPUT_SEPARATOR',
        p_display_length     => 1,
        p_max_length         => 1,
        p_display_group_name => 'DOWNLOAD',
        p_help_text          => q'~Enter a column separator. If no value is entered, a comma or semicolon is used depending on your current NLS settings.~' );
    --
    l_property_id := 310;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CSV_ENCLOSED_BY',
        p_prompt             => 'CSV Enclosed By',
        p_property_type      => 'TEXT',
        p_db_column          => 'CSV_OUTPUT_ENCLOSED_BY',
        p_display_length     => 1,
        p_max_length         => 1,
        p_display_group_name => 'DOWNLOAD',
        p_help_text          => q'~Enter a delimiter character to delineate the starting and ending boundary of a data value. The default delimiter is double quotation marks.</p>~' );
    --
    l_property_id := 311;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DOWNLOAD_FILENAME',
        p_prompt             => 'Filename',
        p_property_type      => 'TEXT',
        p_db_column          => 'DOWNLOAD_FILENAME',
        p_display_length     => 60,
        p_max_length         => 255,
        p_display_group_name => 'DOWNLOAD',
        p_help_text          => q'~Enter the filename for the document when downloaded. Do not specify a file extension. The extension is added automatically. If you leave this attribute blank, the region name is used as the file name.~' );
    --
    l_property_id := 312;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'IR_LINK_COLUMN',
        p_prompt             => 'Link Column',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'SHOW_DETAIL_LINK',
        p_display_group_name => 'LINK',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the link column type for this report.
If you select the link column, it always displays as the first column in the report and cannot be hidden by the user.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Link to Single Row View',
        p_return_value  => 'Y',
        p_help_text     => q'~The end user is directed to a page which shows the column values for the specified row. What and how columns display is governed by various report and column attributes.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Link to Custom Target',
        p_return_value  => 'C',
        p_help_text     => q'~The end user is directed to another page based on the target URL you enter.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Exclude Link Column',
        p_return_value  => 'N',
        p_help_text     => q'~The link column is not displayed.~' );
    --
    l_property_id := 313;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_EXCLUDE_NULL_VALUES',
        p_prompt             => 'Exclude Null Values',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_EXCLUDE_NULL_VALUES',
        p_display_group_name => 'SINGLE_ROW_VIEW',
        p_help_text          => q'~Specify whether null columns are hidden on the Single Record View.~' );
    --
    l_property_id := 314;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_HIDE_EXTRA_COLUMNS',
        p_prompt             => 'Only Displayed Columns',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_HIDE_EXTRA_COLUMNS',
        p_display_group_name => 'SINGLE_ROW_VIEW',
        p_help_text          => q'~Specify whether only those columns currently displayed are displayed in the Single Record View. If you specify <strong>Off</strong> then non displayed report columns may also be shown on the Single Record View.~' );
    --
    l_property_id := 315;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'UNIQUELY_IDENTIFY_ROWS_BY',
        p_prompt             => 'Uniquely Identify Rows by',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'UNIQUELY_IDENTIFY_ROWS_BY',
        p_is_query_only      => true,
        p_display_group_name => 'LINK',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the type of column used to identify the current row when utilizing the link column.
By default, <strong>ROWID</strong> is used to uniquely identify rows in a report. However, you can also utilize another column by selecting <strong>Unique Column</strong>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'ROWID',
        p_return_value  => 'ROWID' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Unique Column',
        p_return_value  => 'COLUMN' );
    --
    l_property_id := 316;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'UNIQUE_COLUMN',
        p_prompt             => 'Unique Column',
        p_property_type      => 'COLUMN',
        p_db_column          => 'BASE_PK1',
        p_column_data_types  => 'VARCHAR2:ROWID',
        p_display_group_name => 'LINK',
        p_help_text          => q'~<p>Select the first column used to uniquely identify rows for this component.</p>
<p><strong>ROWID</strong> is also included in the list of available columns for backward compatability.</p>
<p>Note: For an interactive report with the <em>Link Column</em> attribute set to <strong>Link to Single Row View</strong>, only a single unique column can be selected. Other components support the selection of up to three columns.</p>~' );
    --
    l_property_id := 317;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'UNIQUE_COLUMN_2',
        p_prompt             => 'Unique Column 2',
        p_property_type      => 'COLUMN',
        p_db_column          => 'BASE_PK2',
        p_column_data_types  => 'VARCHAR2',
        p_display_group_name => 'LINK',
        p_help_text          => q'~<p>Select the second column to be used to uniquely identify rows for this component. Use this attribute when your table has a two-part primary key.</p>
<p><strong>ROWID</strong> is also included in the list of available columns for backward compatability.</p>~' );
    --
    l_property_id := 318;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'UNIQUE_COLUMN_3',
        p_prompt             => 'Unique Column 3',
        p_property_type      => 'COLUMN',
        p_db_column          => 'BASE_PK3',
        p_column_data_types  => 'VARCHAR2',
        p_display_group_name => 'LINK',
        p_help_text          => q'~<p>Select the third column to be used to uniquely identify rows for this component. Use this attribute when your table has a three-part primary key.</p>
<p><strong>ROWID</strong> is also included in the list of available columns for backward compatability.</p>
<p>Note: If your table's primary key has more than three columns, then you need to uniquely identify rows by <strong>ROWID</strong>.~' );
    --
    l_property_id := 319;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LINK_ICON',
        p_prompt             => 'Link Icon',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'DETAIL_LINK_TEXT',
        p_display_length     => 60,
        p_max_length         => 4000,
        p_display_group_name => 'LINK',
        p_help_text          => q'~Enter the HTML to be used for the link. Use an image tag to display images, or pick from the list of default images.  If not defined, <code>#APEX_FILES#ws/small_page.gif</code> is used.~' );
    --
    l_property_id := 320;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'LINK_AUTHORIZATION_SCHEME',
        p_prompt                => 'Authorization Scheme',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'DETAIL_LINK_AUTH_SCHEME',
        p_display_group_name    => 'LINK',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_authorization,
        p_help_text             => q'~Select an authorization scheme to associate with the Link Column. This authorization scheme must evaluate to TRUE in order for the Link Column to be rendered in the report.~' );
    --
    l_property_id := 321;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'REPORT_ALIAS',
        p_prompt             => 'Region Alias',
        p_property_type      => 'TEXT',
        p_db_column          => 'ALIAS',
        p_display_length     => 32,
        p_max_length         => 255,
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~Enter an alphanumeric alias for this interactive report region.
The alias must be unique within the application.
You can use an alias to reference an interactive report region in your custom API call.
For example, you can write a function to return the <code>INTERACTIVE_REPORT_ID</code> by querying the <code>ALIAS</code> column in the <code>APEX_APPLICATION_PAGE_IR</code> view.
Since the <code>INTERACTIVE_REPORT_ID</code> value can change when you export or import the application to different instances, referencing interactive report regions using an alias may be preferable.~' );
    --
    l_property_id := 322;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'REPORT_ID_ITEM',
        p_prompt                   => 'Report ID Item',
        p_property_type            => 'ITEM',
        p_db_column                => 'REPORT_ID_ITEM',
        p_display_length           => 32,
        p_max_length               => 255,
        p_display_group_name       => 'ADVANCED',
        p_lov_component_type_scope => 'APPLICATION',
        p_help_text                => q'~Select a page or application item whose value stores the saved report ID.
Interactive report uses this item value to link to a saved report.
For example, you can define list of saved reports with links with defined item name and saved report ID as its value to link to a saved default reports, public or private reports.
The saved report IDs can be retrieved from the <code>APEX_APPLICATION_PAGE_IR_RPT</code> view.~' );
    --
    l_property_id := 323;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EMAIL_FROM_ADDRESS',
        p_prompt             => 'Email From Address',
        p_property_type      => 'TEXT',
        p_db_column          => 'EMAIL_FROM',
        p_display_length     => 30,
        p_max_length         => 255,
        p_examples           => q'~<ul>
<li><code>john.doe@abc.com</code></li>
<li><code>&amp;MY_APP_EMAIL_FROM.</code></li>
</ul>~',
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~<p>Enter a valid email address to use as the <em>From</em> address when sending email from this report. Emails are sent from this report for Downloads of type Email and for Subscriptions.</p>
<p>The value can be a literal string containing a valid email address or a static substitution reference defined in the application using substitution syntax.
Using a page or application item substitution is NOT recommended because it only works in email download and causes any subscription emails to fail since these items can not be accessed outside of a valid session.
Subscriptions are sent directly from the &PRODUCT_NAME. engine and not from a user session.</p>~',
        p_comment_text       => q'~$$$ should we have a type EMAIL? -> used in a few places, otherwise we have to use a PL/SQL validation or do not verify it at all~' );
    --
    l_property_id := 324;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_ENABLED',
        p_prompt             => 'Show',
        p_property_type      => 'YES NO',
        p_db_column          => 'ICON_VIEW_ENABLED_YN',
        p_display_group_name => 'ICON_VIEW',
        p_help_text          => q'~Specify whether the Icon View is enabled. Selecting <strong>On</strong> adds a View Icons icon to the Search bar for interactive reports, or to the interactive grid View button, enabling users to access this view.~' );
    --
    l_property_id := 325;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'USE_CUSTOM_ICON_VIEW',
        p_prompt             => 'Custom',
        p_property_type      => 'YES NO',
        p_db_column          => 'ICON_VIEW_USE_CUSTOM',
        p_display_group_name => 'ICON_VIEW',
        p_help_text          => q'~Specify whether you wish to define a custom link for the Icon View. This option provides greater flexibility in defining the content of the Icon View, including the ability to utilize more than one column.~' );
    --
    l_property_id := 326;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COLUMNS_PER_ROW',
        p_prompt             => 'Columns Per Row',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'ICON_VIEW_COLUMNS_PER_ROW',
        p_display_group_name => 'ICON_VIEW',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the number of icons to display per row in Icon View. A value of 5 displays 5 icons per row of the report.~',
        p_comment_text       => q'~$$$ should this really be a LOV, why not a one digit INTEGER?~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1',
        p_return_value  => '1' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2',
        p_return_value  => '2' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '3',
        p_return_value  => '3' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4',
        p_return_value  => '4' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '5',
        p_return_value  => '5' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '6',
        p_return_value  => '6' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '7',
        p_return_value  => '7' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '8',
        p_return_value  => '8' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '9',
        p_return_value  => '9' );
    --
    l_property_id := 327;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_LINK_COLUMN',
        p_prompt             => 'Link Column',
        p_property_type      => 'COLUMN',
        p_db_column          => 'ICON_VIEW_LINK_COLUMN',
        p_column_data_types  => 'VARCHAR2',
        p_examples           => q'~<dl>
<dt>Examine the <code>LINK</code> column</dt>
<dd><pre>
select ename,
       job,
       'F?P=&APP_ID.:2:' || :APP_SESSION || '::::P2_ID:' || empno as link,
       '#IMAGE_PREFIX#menu/user_red_64.gif' as image
  from emp
</pre></dd>
</dl>~',
        p_display_group_name => 'ICON_VIEW',
        p_help_text          => q'~Select the report column that returns the link target of the icon.~' );
    --
    l_property_id := 328;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_IMAGE_SOURCE_COLUMN',
        p_prompt             => 'Image Source Column',
        p_property_type      => 'COLUMN',
        p_db_column          => 'ICON_VIEW_IMG_SRC_COLUMN',
        p_column_data_types  => 'VARCHAR2',
        p_examples           => q'~<dl>
<dt>Examine the <code>IMAGE</code> column</dt>
<dd><pre>
select ename,
       job,
       'F?P=&APP_ID.:2:' || :APP_SESSION || '::::P2_ID:' || empno as link,
       '#IMAGE_PREFIX#menu/user_red_64.gif' as image
  from emp
</pre></dd>
</dl>~',
        p_display_group_name => 'ICON_VIEW',
        p_help_text          => q'~Select the report column that returns the image source.~' );
    --
    l_property_id := 329;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_LABEL_COLUMN',
        p_prompt             => 'Label Column',
        p_property_type      => 'COLUMN',
        p_db_column          => 'ICON_VIEW_LABEL_COLUMN',
        p_column_data_types  => 'VARCHAR2',
        p_examples           => q'~<dl>
<dt>Examine the <code>ENAME</code> column</dt>
<dd><pre>
select ename,
       job,
       'F?P=&APP_ID.:2:' || :APP_SESSION || '::::P2_ID:' || empno as link,
       '#IMAGE_PREFIX#menu/user_red_64.gif' as image
  from emp
</pre></dd>
</dl>~',
        p_display_group_name => 'ICON_VIEW',
        p_help_text          => q'~Select the report column that returns the image label.~' );
    --
    l_property_id := 330;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_IMAGE_ATTRIBUTES',
        p_prompt             => 'Image Attributes',
        p_property_type      => 'TEXT',
        p_db_column          => 'ICON_VIEW_IMG_ATTR_TEXT',
        p_display_length     => 30,
        p_max_length         => 512,
        p_display_group_name => 'ICON_VIEW',
        p_help_text          => q'~Enter the image attributes used for the HTML IMG tag.~' );
    --
    l_property_id := 331;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_HTML_ALT_TEXT',
        p_prompt             => 'HTML ALT Text',
        p_property_type      => 'TEXT',
        p_db_column          => 'ICON_VIEW_ALT_TEXT',
        p_display_length     => 30,
        p_max_length         => 512,
        p_display_group_name => 'ICON_VIEW',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the HTML ALT text for the image.~' );
    --
    l_property_id := 332;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_HTML_TITLE_TEXT',
        p_prompt             => 'HTML TITLE Text',
        p_property_type      => 'TEXT',
        p_db_column          => 'ICON_VIEW_TITLE_TEXT',
        p_display_length     => 30,
        p_max_length         => 512,
        p_display_group_name => 'ICON_VIEW',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the HTML TITLE tag for the image.~' );
    --
    l_property_id := 333;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ICON_VIEW_CUSTOM_LINK',
        p_prompt             => 'Custom Link',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'ICON_VIEW_CUSTOM_LINK',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<pre>
&lt;a href="#EMP_LINK#"&gt;&lt;img src="#EMP_IMG#" title="#ENAME#" alt="#ENAME#"&gt;&lt;br&gt;#ENAME#&lt;/a&gt;
</pre>~',
        p_display_group_name => 'ICON_VIEW',
        p_help_text          => q'~Enter the column link for the Icon View. Use <code>#COLUMN_NAME#</code> substitution strings for column names as required.~' );
    --
    l_property_id := 334;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DETAIL_VIEW_ENABLED',
        p_prompt             => 'Show',
        p_property_type      => 'YES NO',
        p_db_column          => 'DETAIL_VIEW_ENABLED_YN',
        p_display_group_name => 'DETAIL_VIEW',
        p_help_text          => q'~Specify whether the Detail View is enabled. Selecting <strong>On</strong> adds a View Details icon to the Search bar for interactive reports, or to the interactive grid View button, enabling users to access this view.~' );
    --
    l_property_id := 335;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DETAIL_VIEW_BEFORE_ROWS',
        p_prompt             => 'Before Rows',
        p_property_type      => 'HTML',
        p_db_column          => 'DETAIL_VIEW_BEFORE_ROWS',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<pre>
&lt;table&gt;
</pre>~',
        p_display_group_name => 'DETAIL_VIEW',
        p_help_text          => q'~Enter the HTML to be displayed before report rows.~' );
    --
    l_property_id := 336;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DETAIL_VIEW_FOR_EACH_ROW',
        p_prompt             => 'For Each Row',
        p_property_type      => 'HTML',
        p_db_column          => 'DETAIL_VIEW_FOR_EACH_ROW',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<pre>
&lt;tr&gt;&lt;td align="right"&gt;#ENAME_LABEL#:&lt;/td&gt;&lt;td&gt;#ENAME#&lt;/td&gt;&lt;/tr&gt;
&lt;tr&gt;&lt;td align="right"&gt;#JOB_LABEL#:&lt;/td&gt;&lt;td&gt;#JOB#&lt;/td&gt;&lt;/tr&gt;
</pre>~',
        p_display_group_name => 'DETAIL_VIEW',
        p_help_text          => q'~Enter the body for report rows in the Detail View. Use <code>#COLUMN_NAME#</code> substitution strings for column names and column labels.~' );
    --
    l_property_id := 337;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DETAIL_VIEW_AFTER_ROWS',
        p_prompt             => 'After Rows',
        p_property_type      => 'HTML',
        p_db_column          => 'DETAIL_VIEW_AFTER_ROWS',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<pre>
&lt;/table&gt;
</pre>~',
        p_display_group_name => 'DETAIL_VIEW',
        p_help_text          => q'~Enter the HTML to be displayed after report rows.~' );
    --
    l_property_id := 338;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_FILTER',
        p_prompt             => 'Filter',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_FILTER',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define filters on this report.~' );
    --
    l_property_id := 339;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_SELECT_COLUMNS',
        p_prompt             => 'Select Columns',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_SELECT_COLUMNS',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can use <strong>Select Columns</strong> in the Actions menu to change the columns displayed in this report. User can select (or show), deselect (or hide) and reorder multiple columns using this option.~' );
    --
    l_property_id := 340;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_ROWS_PER_PAGE',
        p_prompt             => 'Rows Per Page',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_ROWS_PER_PAGE',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can change the number of rows displayed for this report.~' );
    --
    l_property_id := 341;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_SORT',
        p_prompt             => 'Sort',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_SORT',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define sort columns for this report.~' );
    --
    l_property_id := 342;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_CONTROL_BREAK',
        p_prompt             => 'Control Break',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_CONTROL_BREAK',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define control breaks for this report.~' );
    --
    l_property_id := 343;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_HIGHLIGHT',
        p_prompt             => 'Highlight',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_HIGHLIGHT',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define highlights for this report.~' );
    --
    l_property_id := 344;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_COMPUTATION',
        p_prompt             => 'Compute',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_COMPUTATION',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define computational columns for this report.~' );
    --
    l_property_id := 345;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_AGGREGATE',
        p_prompt             => 'Aggregate',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_AGGREGATE',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define aggregates for this report.~' );
    --
    l_property_id := 346;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_CHART',
        p_prompt             => 'Chart',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_CHART',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define charts for this report.~' );
    --
    l_property_id := 347;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_NOTIFY',
        p_prompt             => 'Subscription',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_NOTIFY',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define subscription settings for this report.~' );
    --
    l_property_id := 348;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_GROUP_BY',
        p_prompt             => 'Group By',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_GROUP_BY',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end,users can define group by break-downs for this report.~' );
    --
    l_property_id := 349;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_PIVOT',
        p_prompt             => 'Pivot',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_PIVOT',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can define pivot break-downs for this report.~' );
    --
    l_property_id := 350;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_FLASHBACK',
        p_prompt             => 'Flashback',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_FLASHBACK',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~<p>Specify whether end users can utilize flashback queries for this report.</p>
<p>Flashback queries utilize the Oracle Flashback Technology and Database Point-in-Time Recovery to view data as it existed at a previous point in time. Users can only query this data, but can not rewind any data changes. How far back in time they can successfully perform a flashback query is determined by when the redo log files were last switched within the Oracle database, as this technology utilizes the RMAN backup.~' );
    --
    l_property_id := 351;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_RESET',
        p_prompt             => 'Reset',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_RESET',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~<p>Specify whether end users can reset their reports from the Actions menu.</p>
<p>Reseting the report reverts all customizations made by the user, or specified by the calling URL, and displays the Primary Report.</p>~' );
    --
    l_property_id := 352;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_DOWNLOAD',
        p_prompt             => 'Download',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_DOWNLOAD',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can download the report to one of the download formats specified.~' );
    --
    l_property_id := 353;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_HELP',
        p_prompt             => 'Help',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_HELP',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~<p>Specify whether end users can invoke the built-in help modal dialog for interactive reports.</p>
<p>The help dialog only outlines those options available to the current user.</p>~' );
    --
    l_property_id := 354;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_SAVE_REPORTS',
        p_prompt             => 'Save Report',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_REPORT_SAVING',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~<p>Specify whether end users can save their current report definition as a <em>Private</em> report.</p>
<p>Private reports are displayed in a select list of available reports in the Search Bar, and are only displayed to the user who saved the report, providing they are authenticated.
The option on the Actions menu is only available to authenticated users.</p>~' );
    --
    l_property_id := 355;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_SAVE_PUBLIC_REPORT',
        p_prompt             => 'Save Public Report',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_SAVE_RPT_PUBLIC',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~<p>Specify whether end users can save their current report definition as a <em>Public</em> report.</p>
<p>Public reports are displayed in a select list of available reports in the Search Bar, and are displayed to everyone.
The option on the Actions menu is only available to authenticated users and is further restricted if you also populate the <strong>Save Public Report Authorization</strong>.</p>~' );
    --
    l_property_id := 356;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_SAVE_REPORT_CATEGORIES',
        p_prompt             => 'Save Report Categories',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_REPORT_CATEGORIES',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether end users can save report categories.~',
        p_comment_text       => q'~$$$ what does this property do?~' );
    --
    l_property_id := 357;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_FINDER_DROP_DOWN',
        p_prompt             => 'Finder Drop Down',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_FINDER_DROP_DOWN',
        p_display_group_name => 'SEARCH_BAR',
        p_help_text          => q'~Specify whether the Finder drop down is included at the beginning of the Search field within the search bar.
The Finder drop down is a list of columns that enables users to limit their search to a specific column.~' );
    --
    l_property_id := 358;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_DISPLAY_ROW_COUNT',
        p_prompt             => 'Rows Per Page Selector',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_DISPLAY_ROW_COUNT',
        p_display_group_name => 'SEARCH_BAR',
        p_help_text          => q'~Specify whether a separate drop down row selector is included within the search bar.
You should not select both this attribute and <strong>Rows Per Page</strong>.~' );
    --
    l_property_id := 360;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_SEARCH_TEXTBOX',
        p_prompt             => 'Search Field',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_SEARCH_TEXTBOX',
        p_display_group_name => 'SEARCH_BAR',
        p_help_text          => q'~Specify whether the search field is included within the search bar.~' );
    --
    l_property_id := 361;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SHOW_ACTIONS_MENU',
        p_prompt             => 'Include Actions Menu',
        p_property_type      => 'YES NO',
        p_db_column          => 'SHOW_ACTIONS_MENU',
        p_display_group_name => 'ACTIONS_MENU',
        p_help_text          => q'~Specify whether the <em>Actions</em> menu is included within the Search Bar.~' );
    --
    l_property_id := 362;
    store_property (
        p_id                         => l_property_id,
        p_name                       => 'SHOW_REPORTS_SELECT_LIST',
        p_prompt                     => 'Reports Select List',
        p_property_type              => 'YES NO',
        p_prop_type_yesno_override_y => 'TABS',
        p_prop_type_yesno_override_n => 'NONE',
        p_db_column                  => 'REPORT_LIST_MODE',
        p_display_group_name         => 'SEARCH_BAR',
        p_help_text                  => q'~<p>Specify whether the Reports select list displays in the search bar.</p>
<p>The Reports select list only displays if a report, other than the primary report, is accessible to the user.</p>~' );
    --
    l_property_id := 363;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LINK_TARGET',
        p_prompt             => 'Target',
        p_property_type      => 'LINK',
        p_db_column          => 'DETAIL_LINK',
        p_display_group_name => 'LINK',
        p_help_text          => q'~<p>Click the Target to invoke a modal dialog. You can enter a custom target to be called when the Link Column is clicked.</p>
        <p>Select the appropriate type in Link Builder - Target based on your requirements:
        <dl>
        <dt>Page in this application</dt>
        <dd>Enter the following:
        <ul>
        <li>Page - enter the page number or page alias to navigate to. You can also select the page number from the select list.</li>
        <li>Set Items: Name - enter or select page items (<code>P3_ID</code>), application items (<code>LAST_PAGE</code>) or interactive report filters (<code>IR_EMPNO</code>) to be set into session state.</li>
        <li>Set Items: Value - enter or select a report column (<code>#ID#</code>), enter an application or page item (<code>&P2_EMPNO.</code>), or enter a static value (<code>2</code>).</li>
        <li>Clear Cache - enter a comma-delimited list of page numbers (<code>2,3</code>), items (<code>P3_EMPNO,LAST_PAGE</code>), and report settings to be cleared. To reset pagination for a report enter <code>RP</code>. To clear or reset interactive report settings enter <code>CIR</code> or <code>RIR</code>.</li>
        <li>Reset Pagination - select Yes to reset pagination, back to the first set of records, for reports on the target page. This has the same affect as passing <code>RP</code> in Clear Cache.</li>
        <li>Advanced: Request - enter the request to be used.</li>
        </ul>
        </dd>
        <dt>Page in a different application</dt>
        <dd>Enter the same attributes as listed for Page in this application, with the addition that you need to enter the application ID or application alias for Application.</dd>
        <dt>URL</dt>
        <dd>Enter the URL</dd>
        </dl></p>~' );
    --
    l_property_id := 364;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'IR_SAVED_VISIBILITY',
        p_prompt             => 'Visibility',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'VISIBILITY',
        p_is_query_only      => true,
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~View the saved report category, which dictates where this report displays within the Reports select List. This attribute is view only and cannot be updated.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Primary Default',
        p_return_value  => 'PRIMARY_DEFAULT',
        p_help_text     => q'~Report saved by developers, that is displayed by default.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Alternative Default',
        p_return_value  => 'ALTERNATIVE_DEFAULT',
        p_help_text     => q'~Subsequent report layouts saved by developers, that end users can select.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Public',
        p_return_value  => 'PUBLIC',
        p_help_text     => q'~Report layouts saved by end users, that all other end users can select.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Private',
        p_return_value  => 'PRIVATE',
        p_help_text     => q'~Report layouts saved by end users, that only the end user who saved it can select.~' );
    --
    l_property_id := 365;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'IR_SAVED_REPORT_ALIAS',
        p_prompt                => 'Alias',
        p_property_type         => 'TEXT',
        p_db_column             => 'REPORT_ALIAS',
        p_restricted_characters => 'US_ONLY_UNDERSCORE_NO_SPACE',
        p_display_group_name    => 'IDENTIFICATION',
        p_help_text             => q'~Enter an alias for this saved report. Use this alias to define a more meaningful URL link to this specific saved report. This link can be used to link directly from elsewhere in the application.~' );
    --
    l_property_id := 366;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LINK_EXAMPLE',
        p_prompt             => 'Link Example',
        p_property_type      => 'TEXTAREA',
        p_db_column          => 'LINK_EXAMPLE',
        p_is_query_only      => true,
        p_display_group_name => 'LINK',
        p_help_text          => q'~<p>View the URL link for this saved report. This attribute is view only and cannot be updated.</p>
<p>This attribute enables you to quickly copy the displayed URL. This URL can be used to link directly to this particular saved report from anywhere else in the application.</p>~' );
    --
    l_property_id := 367;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'RUN_PROCESS',
        p_prompt             => 'Run Process',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PROCESS_IS_STATEFUL_Y_N',
        p_display_group_name => 'EXECUTION_OPTIONS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the frequency that this process is executed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once Per Session or When Reset',
        p_return_value  => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once Per Page Visit (default)',
        p_return_value  => 'N' );
    --
    l_property_id := 368;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SUCCESS_MESSAGE',
        p_prompt             => 'Success Message',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'PROCESS_SUCCESS_MESSAGE',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'SUCCESS_MESSAGE',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the text to be displayed in the "Notification" area (defined as part of the page template), when this process completes successfully.~' );
    --
    l_property_id := 369;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PROCESS_ERROR_DISPLAY_LOCATION',
        p_prompt             => 'Display Location',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'ERROR_DISPLAY_LOCATION',
        p_display_group_name => 'ERROR',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select where this process error message displays, when the process fails to complete successfully.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Inline in Notification',
        p_return_value  => 'INLINE_IN_NOTIFICATION',
        p_help_text     => q'~The process error message is displayed in the "Notification" area (defined as part of the page template)~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'On Error Page',
        p_return_value  => 'ON_ERROR_PAGE',
        p_help_text     => q'~The Process error message is displayed on the error page.~' );
    --
    l_property_id := 370;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'PROCESS_REGION',
        p_prompt                   => 'Editable Region',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'REGION_ID',
        p_display_group_name       => 'IDENTIFICATION',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE',
        p_help_text                => q'~Select the region to associate with this process.~',
        p_comment_text             => q'~$$$ could this be merged with VALIDATION_REGION by putting the property into the display group Execution Options?
p_lov_component_type_scope = PAGE -> should only include tabular forms~' );
    --
    l_property_id := 371;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'DA_EVENT',
        p_prompt                   => 'Event',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'EVENT_ID',
        p_display_group_name       => 'EXECUTION_OPTIONS',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_da_event,
        p_lov_component_type_scope => 'PAGE',
        p_help_text                => q'~Select the appropriate dynamic action event from the available list.~' );
    --
    l_property_id := 372;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'IR_COLUMN_FILTER_LOV_SQL',
        p_prompt               => 'SQL Query',
        p_property_type        => 'SQL',
        p_db_column            => 'RPT_LOV',
        p_display_length       => 80,
        p_max_length           => 4000,
        p_sql_min_column_count => 1,
        p_sql_max_column_count => 1,
        p_display_group_name   => 'COLUMN_FILTER',
        p_examples             => q'~<dl>
<dt>To return the department name:</dt>
<dd><pre>select dname from dept order by 1</pre></dd>
</dl>~',
        p_help_text            => q'~<p>Enter a SQL query which needs to return one column. If the query returns multiple columns, only the first column is used in the list of values.</p>
<p>Entering a SQL query can improve performance in some reports, or be used to limit the values displayed.</p>~' );
    --
    l_property_id := 373;
    store_lov_type_property (
        p_name => 'LOV_TYPE' );
    --
    l_property_id := 374;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'LOV_STATIC_VALUES',
        p_prompt             => 'Static Values',
        p_property_type      => 'STATIC LOV',
        p_db_column          => 'LOV',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'LIST_OF_VALUES',
        p_help_text          => q'~Enter the values for this static list of values.~' );
    --
    l_property_id := 375;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'TZ_DEPENDENT',
        p_prompt             => 'Timezone Aware',
        p_property_type      => 'HIDDEN',
        p_db_column          => 'TZ_DEPENDENT',
        p_display_group_name => 'IDENTIFICATION',
        p_help_text          => q'~Indicates whether or not this column is timezone aware.~' );
    --
    l_property_id := 376;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_PROC_WS_PARM_ITEM',
        p_prompt             => 'Item',
        p_property_type      => 'ITEM',
        p_db_column          => 'PARAMETER_VALUE',
        p_display_group_name => 'PARAMETER',
        p_help_text          => q'~Enter the page or application item used as the source for this parameter. You can type in the name or pick from the list of available items.~' );
    --
    l_property_id := 377;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_PROC_WS_PARM_SOURCE',
        p_prompt             => 'Source',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'MAP_TYPE',
        p_display_group_name => 'PARAMETER',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the source for the parameter from the available list.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item',
        p_return_value  => 'ITEM',
        p_help_text     => q'~Set to the value held in session state for the <em>Item</em> selected.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static Value',
        p_return_value  => 'STATIC',
        p_help_text     => q'~Set to the text entered into <em>Static Value</em>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'PL/SQL Function Body',
        p_return_value  => 'FUNCTION',
        p_help_text     => q'~Set to the value returned from the <em>PL/SQL Function Body</em> entered.~' );
    --
    l_property_id := 379;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_PROC_WS_PARM_VALUE',
        p_prompt             => 'Static Value',
        p_property_type      => 'TEXT',
        p_db_column          => 'PARAMETER_VALUE',
        p_display_group_name => 'PARAMETER',
        p_help_text          => q'~Enter the value for this parameter.~' );
    --
    l_property_id := 380;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CHART_TITLE',
        p_prompt             => 'Title',
        p_property_type      => 'TEXT',
        p_db_column          => 'CHART_TITLE',
        p_display_length     => 65,
        p_max_length         => 255,
        p_display_group_name => 'TITLE',
        p_is_translatable    => true,
        p_help_text          => q'~Enter a title to display at the top of the chart.  For gantt charts, this value is only used to define the accessible chart title.~' );
    --
    l_property_id := 381;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CHANGED_BY',
        p_prompt             => 'Changed By',
        p_property_type      => 'TEXT',
        p_db_column          => 'LAST_UPDATED_BY',
        p_is_query_only      => true,
        p_display_length     => 30,
        p_display_group_name => 'AUDIT_INFORMATION',
        p_is_searchable      => false,
        p_help_text          => q'~Displays the user credentials of the developer who last updated this attribute.~' );
    --
    l_property_id := 382;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CHANGED_ON',
        p_prompt             => 'Changed On',
        p_property_type      => 'TEXT',
        p_db_column          => 'LAST_UPDATED_ON',
        p_is_query_only      => true,
        p_display_length     => 30,
        p_display_group_name => 'AUDIT_INFORMATION',
        p_is_searchable      => false,
        p_help_text          => q'~Displays when this attribute was last updated.~' );
    --
    l_property_id := 383;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_USERS_TO_PIVOT',
        p_prompt             => 'Pivot',
        p_property_type      => 'YES NO',
        p_db_column          => 'ALLOW_PIVOT',
        p_display_group_name => 'ENABLE_USER_TO',
        p_help_text          => q'~Specify whether end user can define a pivot break-down within this report.~' );
    --
    l_property_id := 384;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'SOURCE_LIST',
        p_prompt                => 'List',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'LIST_ID',
        p_display_group_name    => 'SOURCE',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_list,
        p_help_text             => q'~Select the source for this list.~' );
    --
    l_property_id := 385;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'SOURCE_BREADCRUMB',
        p_prompt                => 'Breadcrumb',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'BREADCRUMB_ID',
        p_display_group_name    => 'SOURCE',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_breadcrumb,
        p_help_text             => q'~Select the breadcrumb list associated with this page.~' );
    --
    l_property_id := 386;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'LIST_TEMPLATE',
        p_prompt                   => 'List Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'LIST_TEMPLATE_ID',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_list_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~<p>Select a template to define the appearance and layout of this list.</p>
        <p>List templates are defined in the application theme.
        When a list region is created, the template is automatically set to the default list template defined in the current theme.</p>
        <p>Universal Theme provides the following list templates:
          <dl>
            <dt>Badge List</dt>
            <dt>Cards</dt>
            <dt>Links List</dt>
            <dt>Media List</dt>
            <dt>Menu Bar</dt>
            <dt>Menu Popup</dt>
            <dt>Navigation Bar</dt>
            <dt>Side Navigation Menu</dt>
            <dt>Tabs</dt>
            <dt>Top Navigation Menu</dt>
            <dt>Wizard Progress</dt>
          </dl>
          See Also: <a rel="noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a>
        </p>~' );
    --
    l_property_id := 387;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'BREADCRUMB_TEMPLATE',
        p_prompt                   => 'Breadcrumb Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'MENU_TEMPLATE_ID',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_bc_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~Select the breadcrumb template utilized with this breadcrumb.~' );
    --
    l_property_id := 388;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'APP_ITEM_SCOPE',
        p_prompt             => 'Scope',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'SCOPE',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select the scope of this application item. Select <strong>Global</strong> if the &PRODUCT_NAME. session is shared by more than one application and the item's value should be the same for all applications.
Otherwise, select <strong>Application</strong>, which is the default scope.</p>
<p>Applications can share the same session, if their authentications have the same <em>Session Cookie Attributes</em>.
The scope attribute, of application items, has to be the same across these applications as well.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Application',
        p_return_value  => 'APP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Global',
        p_return_value  => 'GLOBAL' );
    --
    l_property_id := 389;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'LOV_SUBSCRIBED_FROM',
        p_prompt                   => 'Master',
        p_property_type            => 'SUBSCRIPTION',
        p_db_column                => 'REFERENCE_ID',
        p_display_group_name       => 'SUBSCRIPTION',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_lov,
        p_lov_component_type_scope => 'WORKSPACE',
        p_help_text                => q'~Use this field to base this list of values on an existing list of values in this application or another application in your workspace.~' );
    --
    l_property_id := 390;
    store_property (
        p_id                         => l_property_id,
        p_name                       => 'HAS_GENERIC_COLUMNS',
        p_prompt                     => 'Use Generic Column Names',
        p_property_type              => 'YES NO',
        p_prop_type_yesno_override_y => 'GENERIC_REPORT_COLUMNS',
        p_prop_type_yesno_override_n => 'DERIVED_REPORT_COLUMNS',
        p_db_column                  => 'PLUG_QUERY_OPTIONS',
        p_display_group_name         => 'SOURCE',
        p_help_text                  => q'~Specify whether column definitions are derived from the SQL query.
Doing this requires &PRODUCT_NAME. to parse the query.
If you do not want the query parsed at this time, use generic column names, which produces placeholder column definitions.~' );
    --
    l_property_id := 391;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'GENERIC_COLUMN_COUNT',
        p_prompt             => 'Generic Column Count',
        p_property_type      => 'INTEGER',
        p_db_column          => 'PLUG_QUERY_MAX_COLUMNS',
        p_display_length     => 3,
        p_max_length         => 3,
        p_min_value          => 1,
        p_max_value          => 999,
        p_display_group_name => 'SOURCE',
        p_help_text          => q'~<p>Enter the maximum number of generic report columns.
Reports that use generic column names are parsed at runtime only, so the number of columns is not known when editing the report.</p>
<p>The maximum number of report columns should match the expected number of report columns.
If the actual number is smaller, the report renders with the actual number of columns.
However, if the actual number is higher than the maximum, then an error message shows at runtime.</p>~' );
    --
    l_property_id := 392;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_PROC_WS_PARM_FUNCTION',
        p_prompt             => 'PL/SQL Function Body',
        p_property_type      => 'PLSQL FUNCTION BODY VARCHAR2',
        p_db_column          => 'PARAMETER_VALUE',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'PARAMETER',
        p_help_text          => q'~Enter a PL/SQL function body that returns a value used as the source for this parameter.~' );
    --
    l_property_id := 394;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'PAGE_PROC_WS_PARM_ITEMS',
        p_prompt                => 'Item(s)',
        p_property_type         => 'ITEM',
        p_multi_value_delimiter => ',',
        p_db_column             => 'PARAMETER_VALUE',
        p_display_group_name    => 'PARAMETER',
        p_help_text             => q'~<p>Enter one or more page or application item used as the destination for this parameter. For multiple items, separate each item name with a comma.</p>
<p>You can type in the name or pick from the list of available items.
If you pick from the list and there is already text entered, then a comma is placed at the end of the existing text, followed by the item name returned from the list.~' );
    --
    l_property_id := 395;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTE_SUM',
        p_prompt             => 'Compute Sum',
        p_property_type      => 'YES NO',
        p_db_column          => 'SUM_COLUMN',
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~For this column, specify whether the sum is computed and displays at the end of the report.
When used with break columns, the total sum and the sum on each break displays.~' );
    --
    l_property_id := 396;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COLUMN_WIDTH',
        p_prompt             => 'Cell Width',
        p_property_type      => 'INTEGER',
        p_db_column          => 'REPORT_COLUMN_WIDTH',
        p_display_length     => 3,
        p_min_value          => 1,
        p_unit               => c_unit_pixels,
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~Enter the column width in pixels.
Use this attribute when displaying reports in a browser.
Note that setting the column width must be supported by the selected report template.~' );
    --
    l_property_id := 397;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'USE_AS_ROW_HEADER',
        p_prompt             => 'Value Identifies Row',
        p_property_type      => 'YES NO',
        p_db_column          => 'USE_AS_ROW_HEADER',
        p_display_group_name => 'ACCESSIBILITY',
        p_help_text          => q'~<p>Define the column value, or combination of column values that help to identify the row of data. For example for person data this could be the first and last name, for customer data this could be the customer name, and so on.</p>
<p>For regions that render tabular report data (like Classic or Interactive Reports), defining this attribute is very important for the usability of the report for users of assistive technologies like screen readers. When a screen reader user navigates to different rows of the report, the column, or columns defined as identifying the row will be announced as the row header, as soon as they move to the new row. If this is not defined, it is much harder for these users to orientate themselves when moving around the report.</p>~' );
    --
    l_property_id := 398;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CLASSIC_REPORT_COLUMN_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DISPLAY_AS',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the basis for this report column.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Plain Text',
        p_return_value  => 'PLAIN',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Displays the column value.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Plain Text (based on List of Values)',
        p_return_value  => 'PLAIN_LOV',
        p_help_text     => q'~Displays the value returned from the list of values specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Rich Text',
        p_return_value  => 'RICH_TEXT',
        p_help_text     => q'~Renders the column using HTML or Markdown.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Link',
        p_return_value  => 'LINK',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Displays the column as a link.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Display Image',
        p_return_value  => 'IMAGE',
        p_help_text     => q'~Displays the column as an image.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Download BLOB',
        p_return_value  => 'DOWNLOAD',
        p_help_text     => q'~Displays the column as a link to download the file.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Percent Graph',
        p_return_value  => 'PCT_GRAPH',
        p_help_text     => q'~Displays the column as a percentage graph. The value retrieved must be between 0 and 100.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Hidden Column',
        p_return_value  => 'HIDDEN_COLUMN',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Does not display the column, but it is referenceable.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Display as Text (saves state)',
        p_return_value  => 'DISPLAY_AND_SAVE',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Date Picker (Classic)',
        p_return_value  => 'DATE_POPUP',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Date Picker',
        p_return_value  => 'DATE_PICKER',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Text Field',
        p_return_value  => 'TEXT',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Text Area',
        p_return_value  => 'TEXTAREA',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Select List',
        p_return_value  => 'SELECT_LIST',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Radio Group',
        p_return_value  => 'RADIOGROUP',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Hidden Field',
        p_return_value  => 'HIDDEN_FIELD',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Popup LOV',
        p_return_value  => 'POPUP',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Popup Key LOV',
        p_return_value  => 'POPUPKEY',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Simple Checkbox',
        p_return_value  => 'SIMPLE_CHECKBOX',
        p_is_deprecated => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '* Row Selector',
        p_return_value  => 'ROW_SELECTOR',
        p_is_deprecated => 'Y' );
    --
    l_property_id := 399;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DATE_PICKER_FORMAT_MASK',
        p_prompt             => 'Format Mask',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'COLUMN_FORMAT',
        p_display_group_name => 'APPEARANCE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the format mask to apply to the date picker for this column.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Use Application Date Format',
        p_return_value  => 'PICK_DATE_USING_APP_DATE_FORMAT',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Use Application Format Mask',
        p_return_value  => 'PICK_DATE_USING_APP_FORMAT_MASK',
        p_comment_text  => '$$$ ???? DJP: What is the difference b/n this and Application Date Format Mask' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Use Item Format Mask',
        p_return_value  => 'PICK_DATE_USING_FORMAT_MASK' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MM-YYYY',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_DASH' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD.MM.YYYY',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_DOT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD/MM/YYYY',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MM-YYYY HH:MI',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_HH_MI_DASH' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD.MM.YYYY HH:MI',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_HH_MI_DOT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD/MM/YYYY HH:MI',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_HH_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MM-YYYY HH24:MI',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_HH24_MI_DASH' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD.MM.YYYY HH24:MI',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_HH24_MI_DOT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD/MM/YYYY HH24:MI',
        p_return_value  => 'PICK_DATE_DD_MM_YYYY_HH24_MI',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MON-RR',
        p_return_value  => 'PICK_DATE_DD_MON_RR' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MON-RR HH:MI',
        p_return_value  => 'PICK_DATE_DD_MON_RR_HH_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MON-RR HH24:MI',
        p_return_value  => 'PICK_DATE_DD_MON_RR_HH24_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MON-YYYY',
        p_return_value  => 'PICK_DATE_DD_MON_YYYY',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MON-YYYY HH:MI',
        p_return_value  => 'PICK_DATE_DD_MON_YYYY_HH_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DD-MON-YYYY HH24:MI',
        p_return_value  => 'PICK_DATE_DD_MON_YYYY_HH24_MI',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'MM/DD/YYYY',
        p_return_value  => 'PICK_DATE_MM_DD_YYYY',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'MM/DD/YYYY HH:MI',
        p_return_value  => 'PICK_DATE_MM_DD_YYYY_HH_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'MM/DD/YYYY HH24:MI',
        p_return_value  => 'PICK_DATE_MM_DD_YYYY_HH24_MI',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'RR-MON-DD',
        p_return_value  => 'PICK_DATE_RR_MON_DD' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'RR-MON-DD HH:MI',
        p_return_value  => 'PICK_DATE_RR_MON_DD_HH_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'RR-MON-DD HH24:MI',
        p_return_value  => 'PICK_DATE_RR_MON_DD_HH24_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY.DD.MM',
        p_return_value  => 'PICK_DATE_YYYY_DD_MM_DOT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY.DD.MM HH:MI',
        p_return_value  => 'PICK_DATE_YYYY_DD_MM_HH_MI_DOT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY.DD.MM HH24:MI',
        p_return_value  => 'PICK_DATE_YYYY_DD_MM_HH24_MI_DOT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY-MM-DD',
        p_return_value  => 'PICK_DATE_YYYY_MM_DD' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY/MM/DD',
        p_return_value  => 'PICK_DATE_YYYY_MM_DD_SLASH' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY-MM-DD HH:MI',
        p_return_value  => 'PICK_DATE_YYYY_MM_DD_HH_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY/MM/DD HH:MI',
        p_return_value  => 'PICK_DATE_YYYY_MM_DD_HH_MI_SLASH' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY-MM-DD HH24:MI',
        p_return_value  => 'PICK_DATE_YYYY_MM_DD_HH24_MI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'YYYY/MM/DD HH24:MI',
        p_return_value  => 'PICK_DATE_YYYY_MM_DD_HH24_MI_SLASH' );
    --
    l_property_id := 400;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'RADIO_COLUMNS',
        p_prompt             => 'Number of Columns',
        p_property_type      => 'INTEGER',
        p_db_column          => 'ATTRIBUTE_01',
        p_display_length     => 2,
        p_min_value          => 1,
        p_display_group_name => 'ELEMENT',
        p_help_text          => q'~Enter the number of columns to display for this radio group column.~' );
    --
    l_property_id := 402;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'SIMPLE_CHECKBOX_VALUES',
        p_prompt             => 'Checkbox Values',
        p_property_type      => 'TEXT',
        p_db_column          => 'INLINE_LOV',
        p_display_length     => 60,
        p_display_group_name => 'LIST_OF_VALUES',
        p_help_text          => q'~For the Simple Checkbox display type, enter either one value or two comma separated values.
The first value is submitted if the checkbox is selected. The second value, if defined, is submitted if the checkbox is deselected.',
        p_examples           => q'~<dl>
<dt>Submit Yes when selected and No when deselected</dt>
<dd><pre>
Yes,No
</pre></dd>
</dl>~' );
    --
    l_property_id := 403;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'FORMATTING_CSS_CLASSES',
        p_prompt                => 'CSS Classes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ' ',
        p_db_column             => 'COLUMN_CSS_CLASS',
        p_display_length        => 50,
        p_max_length            => 255,
        p_examples              => q'~<dl>
<dt>Setting this attribute to <code>fielddatabold</code> results in the following HTML being generated</dt>
<dd><pre>
&lt;span class="fielddatabold"&gt;Sample Data&lt;/span&gt;
</pre></dd>
</dl>~',
        p_display_group_name    => 'COLUMN_FORMATTING',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~<p>Enter CSS classes to apply to this column's value. You can type in the CSS class or pick from the list of available items.</p>
<p>This class is included by default on all &PRODUCT_NAME. pages and has the definition of <code>'.fielddatabold{font-size: 10pt; font-weight: bold; color: #000000;}'</code>.</p>~' );
    --
    l_property_id := 404;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'CSS_STYLE',
        p_prompt                => 'CSS Style',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ' ',
        p_db_column             => 'COLUMN_CSS_STYLE',
        p_display_length        => 50,
        p_max_length            => 255,
        p_examples              => q'~<dl>
<dt>Setting this attribute to <code>color:#FF0000;</code> changes the text color of the column to red and generates the following HTML</dt>
<dd><pre>
&lt;span style="color:#FF0000"&gt;Sample Data&lt;/span&gt;
</pre></dd>
</dl>~',
        p_display_group_name    => 'COLUMN_FORMATTING',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~Enter CSS styles to apply to this column's value. You can type in the CSS style or pick from the list of available items.~' );
    --
    l_property_id := 405;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'HIGHLIGHT_WORDS',
        p_prompt                => 'Highlight Words',
        p_property_type         => 'TEXT',
        p_db_column             => 'COLUMN_HIT_HIGHLIGHT',
        p_display_length        => 84,
        p_max_length            => 255,
        p_restricted_characters => 'WEB_SAFE',
        p_display_group_name    => 'COLUMN_FORMATTING',
        p_help_text             => q'~Enter text strings to be highlighted in a report column.
For multiple highlighted words, use a comma delimited list.
You can also reference page or application items using <code>&amp;ITEM.</code> syntax.
For example, you can highlight strings entered into a search field.~' );
    --
    l_property_id := 406;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COLUMN_SORT_SEQUENCE',
        p_prompt             => 'Default Sequence',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DEFAULT_SORT_COLUMN_SEQUENCE',
        p_display_group_name => 'SORTING',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the sort sequence for this column.
Each column should have a unique sequence to correctly order the report output.~',
        p_comment_text       => q'~$$$ should this really be a LOV, why not a one digit INTEGER?~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1',
        p_return_value  => '1' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2',
        p_return_value  => '2' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '3',
        p_return_value  => '3' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4',
        p_return_value  => '4' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '5',
        p_return_value  => '5' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '6',
        p_return_value  => '6' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '7',
        p_return_value  => '7' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '8',
        p_return_value  => '8' );
    --
    l_property_id := 407;
    store_property (
        p_id                 => l_property_id,
        p_name               =>