/*
 * Copyright (c) 2003, 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package sun.awt.X11;

import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.peer.RobotPeer;

import sun.awt.AWTAccessor;
import sun.awt.SunToolkit;
import sun.awt.UNIXToolkit;
import sun.awt.X11GraphicsConfig;
import sun.awt.X11GraphicsDevice;
import sun.awt.screencast.ScreencastHelper;
import sun.awt.screencast.XdgDesktopPortal;

final class XRobotPeer implements RobotPeer {

    private static final boolean tryGtk;

    static {
        loadNativeLibraries();

        tryGtk = Boolean.parseBoolean(
                            System.getProperty("awt.robot.gtk", "true")
                 );
    }

    private static volatile boolean useGtk;
    private final X11GraphicsConfig xgc;

    XRobotPeer(X11GraphicsDevice gd) {
        xgc = (X11GraphicsConfig) gd.getDefaultConfiguration();
        SunToolkit tk = (SunToolkit)Toolkit.getDefaultToolkit();
        setup(tk.getNumberOfButtons(),
                AWTAccessor.getInputEventAccessor().getButtonDownMasks());

        boolean isGtkSupported = false;
        if (tryGtk) {
            if (tk instanceof UNIXToolkit && ((UNIXToolkit) tk).loadGTK()) {
                isGtkSupported = true;
            }
        }

        useGtk = (tryGtk && isGtkSupported);
    }

    @Override
    public void mouseMove(int x, int y) {
        mouseMoveImpl(xgc, xgc.scaleUp(x), xgc.scaleUp(y));
        if (XdgDesktopPortal.isRemoteDesktop() && ScreencastHelper.isAvailable()) {
            // We still call mouseMoveImpl on purpose to change the mouse position
            // within the XWayland server so that we can retrieve it later.
            ScreencastHelper.remoteDesktopMouseMove(xgc.scaleUp(x), xgc.scaleUp(y));
        }
    }

    @Override
    public void mousePress(int buttons) {
        if (XdgDesktopPortal.isRemoteDesktop() && ScreencastHelper.isAvailable()) {
            ScreencastHelper.remoteDesktopMouseButton(true, buttons);
        } else {
            mousePressImpl(buttons);
        }
    }

    @Override
    public void mouseRelease(int buttons) {
        if (XdgDesktopPortal.isRemoteDesktop() && ScreencastHelper.isAvailable()) {
            ScreencastHelper.remoteDesktopMouseButton(false, buttons);
        } else {
            mouseReleaseImpl(buttons);
        }
    }

    @Override
    public void mouseWheel(int wheelAmt) {
        if (XdgDesktopPortal.isRemoteDesktop() && ScreencastHelper.isAvailable()) {
            ScreencastHelper.remoteDesktopMouseWheel(wheelAmt);
        } else {
            mouseWheelImpl(wheelAmt);
        }
    }

    @Override
    public void keyPress(int keycode) {
        if (XdgDesktopPortal.isRemoteDesktop() && ScreencastHelper.isAvailable()) {
            ScreencastHelper.remoteDesktopKey(true, keycode);
        } else {
            keyPressImpl(keycode);
        }
    }

    @Override
    public void keyRelease(int keycode) {
        if (XdgDesktopPortal.isRemoteDesktop() && ScreencastHelper.isAvailable()) {
            ScreencastHelper.remoteDesktopKey(false, keycode);
        } else {
            keyReleaseImpl(keycode);
        }
    }

    @Override
    public int getRGBPixel(int x, int y) {
        int[] pixelArray = new int[1];
        if ((XdgDesktopPortal.isScreencast()
                || XdgDesktopPortal.isRemoteDesktop()) && ScreencastHelper.isAvailable()) {
            ScreencastHelper.getRGBPixels(x, y, 1, 1, pixelArray);
        } else {
            getRGBPixelsImpl(xgc, x, y, 1, 1, pixelArray, useGtk);
        }
        return pixelArray[0];
    }

    @Override
    public int[] getRGBPixels(Rectangle bounds) {
        int[] pixelArray = new int[bounds.width * bounds.height];
        if ((XdgDesktopPortal.isScreencast()
                || XdgDesktopPortal.isRemoteDesktop()) && ScreencastHelper.isAvailable()) {

            ScreencastHelper.getRGBPixels(bounds.x, bounds.y,
                                          bounds.width, bounds.height,
                                          pixelArray);
        } else {
            getRGBPixelsImpl(xgc,
                             bounds.x, bounds.y,
                             bounds.width, bounds.height,
                             pixelArray, useGtk);
        }
        return pixelArray;
    }

    private static synchronized native void setup(int numberOfButtons, int[] buttonDownMasks);
    private static native void loadNativeLibraries();

    private static synchronized native void mouseMoveImpl(X11GraphicsConfig xgc, int x, int y);
    private static synchronized native void mousePressImpl(int buttons);
    private static synchronized native void mouseReleaseImpl(int buttons);
    private static synchronized native void mouseWheelImpl(int wheelAmt);

    private static synchronized native void keyPressImpl(int keycode);
    private static synchronized native void keyReleaseImpl(int keycode);

    private static synchronized native void getRGBPixelsImpl(X11GraphicsConfig xgc,
            int x, int y, int width, int height, int[] pixelArray, boolean isGtkSupported);
}
