/*
 * Copyright (c) 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.jpackage.internal;

import static jdk.jpackage.internal.util.function.ThrowingConsumer.toConsumer;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Stream;


public final class Codesign {

    public static final class CodesignException extends Exception {

        CodesignException(String[] output) {
            this.output = output;
        }

        String[] getOutput() {
            return output;
        }

        private final String[] output;

        private static final long serialVersionUID = 1L;
    }

    public static final class Builder {

        private Builder(Supplier<List<String>> args) {
            this.args = Objects.requireNonNull(args);
        }

        Codesign create() {
            List<String> cmdline = new ArrayList<>();
            cmdline.add("/usr/bin/codesign");
            cmdline.addAll(args.get());
            if (force) {
                cmdline.add("--force");
            }

            return new Codesign(cmdline, quiet ? exec -> {
                exec.setQuiet(true);
            } : null);
        }

        public Builder force(boolean v) {
            force = v;
            return this;
        }

        public Builder quiet(boolean v) {
            quiet = v;
            return this;
        }

        private final Supplier<List<String>> args;
        private boolean force;
        private boolean quiet;
    }

    public static Builder build(Supplier<List<String>> args) {
        return new Builder(args);
    }

    public void applyTo(Path path) throws IOException, CodesignException {

        var exec = Executor.of(Stream.concat(
                cmdline.stream(),
                Stream.of(path.toString())).toArray(String[]::new)
        ).saveOutput(true);
        configureExecutor.ifPresent(configure -> configure.accept(exec));

        if (exec.execute() != 0) {
            throw new CodesignException(exec.getOutput().toArray(String[]::new));
        }
    }

    public Consumer<Path> asConsumer() {
        return toConsumer(this::applyTo);
    }

    private Codesign(List<String> cmdline, Consumer<Executor> configureExecutor) {
        this.cmdline = Objects.requireNonNull(cmdline);
        this.configureExecutor = Optional.ofNullable(configureExecutor);
    }

    private final List<String> cmdline;
    private final Optional<Consumer<Executor>> configureExecutor;
}
