/*
 * Copyright (c) 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.jpackage.internal.model;

import java.nio.file.Path;
import jdk.jpackage.internal.util.CompositeProxy;

/**
 * Linux package.
 * <p>
 * Use {@link #create} method to create objects implementing this interface.
 */
public interface LinuxPackage extends Package, LinuxPackageMixin {

    LinuxApplication app();

    @Override
    AppImageLayout packageLayout();

    @Override
    default String packageFileName() {
        String packageFileNameTemlate = asStandardPackageType().map(stdType -> {
            switch (stdType) {
                case LINUX_DEB -> {
                    return "%s_%s_%s";
                }
                case LINUX_RPM -> {
                    return "%s-%s.%s";
                }
                default -> {
                    throw new IllegalStateException();
                }
            }
        }).orElseThrow(UnsupportedOperationException::new);

        return String.format(packageFileNameTemlate, packageName(), versionWithRelease(), arch());
    }

    /**
     * Gets the version with the release component of this Linux package.
     *
     * @return the version with the release component of this Linux package
     */
    default String versionWithRelease() {
        return String.format("%s%s", version(), release().map(r -> "-" + r).orElse(""));
    }

    /**
     * Returns <code>true</code> in this Linux package installs in "/usr" tree.
     *
     * @return <code>true</code> in this Linux package installs in "/usr" tree
     */
    default boolean isInstallDirInUsrTree() {
        return !relativeInstallDir().getFileName().equals(Path.of(packageName()));
    }

    /**
     * Constructs {@link LinuxPackage} instance from the given
     * {@link Package} and {@link LinuxPackageMixin} instances.
     *
     * @param pkg the generic package
     * @param mixin Linux-specific details supplementing the Linux package
     * @return the proxy dispatching calls to the given objects
     */
    public static LinuxPackage create(Package pkg, LinuxPackageMixin mixin) {
        return CompositeProxy.create(LinuxPackage.class, pkg, mixin);
    }
}
