/*
 * Copyright (c) 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.jpackage.internal;

import java.io.IOException;
import static jdk.jpackage.internal.util.function.ThrowingSupplier.toSupplier;
import jdk.jpackage.internal.model.StandardPackageType;

final class LinuxPackageArch {

    static String getValue(StandardPackageType pkgType) {
        switch (pkgType) {
            case LINUX_RPM -> {
                return RpmPackageArch.VALUE;
            }
            case LINUX_DEB -> {
                return DebPackageArch.VALUE;
            }
            default -> {
                throw new IllegalArgumentException();
            }
        }
    }

    private static class DebPackageArch {

        static final String VALUE = toSupplier(DebPackageArch::getValue).get();

        private static String getValue() throws IOException {
            return Executor.of("dpkg", "--print-architecture").saveOutput(true)
                    .executeExpectSuccess().getOutput().get(0);
        }
    }

    private static class RpmPackageArch {

        /*
         * Various ways to get rpm arch. Needed to address JDK-8233143. rpmbuild is mandatory for
         * rpm packaging, try it first. rpm is optional and may not be available, use as the last
         * resort.
         */
        private static enum RpmArchReader {
            Rpmbuild("rpmbuild", "--eval=%{_target_cpu}"),
            Rpm("rpm", "--eval=%{_target_cpu}");

            RpmArchReader(String... cmdline) {
                this.cmdline = cmdline;
            }

            String getRpmArch() throws IOException {
                Executor exec = Executor.of(cmdline).saveOutput(true);
                switch (this) {
                    case Rpm -> {
                        exec.executeExpectSuccess();
                    }
                    case Rpmbuild -> {
                        if (exec.execute() != 0) {
                            return null;
                        }
                    }
                    default -> {
                        throw new UnsupportedOperationException();
                    }
                }
                return exec.getOutput().get(0);
            }

            private final String[] cmdline;
        }

        static final String VALUE = toSupplier(RpmPackageArch::getValue).get();

        private static String getValue() throws IOException {
            for (var rpmArchReader : RpmArchReader.values()) {
                var rpmArchStr = rpmArchReader.getRpmArch();
                if (rpmArchStr != null) {
                    return rpmArchStr;
                }
            }
            throw new RuntimeException("error.rpm-arch-not-detected");
        }
    }
}
