/*
 * Copyright (c) 2003, 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package sun.print;

import java.io.Serial;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Objects;

import javax.print.attribute.EnumSyntax;
import javax.print.attribute.standard.Media;
import javax.print.attribute.standard.MediaSize;
import javax.print.attribute.standard.MediaSizeName;
import javax.print.attribute.Size2DSyntax;

class CustomMediaSizeName extends MediaSizeName {
    private static ArrayList<String> customStringTable = new ArrayList<>();
    private static ArrayList<MediaSizeName> customEnumTable = new ArrayList<>();
    private static Map<SizeNameChoiceItem, CustomMediaSizeName> customMap = new HashMap<>();
    private String choiceName;
    private MediaSizeName mediaName;

    private CustomMediaSizeName(int x) {
        super(x);

    }

    private static synchronized int nextValue(String name) {
      customStringTable.add(name);

      return (customStringTable.size()-1);
    }

    public CustomMediaSizeName(String name) {
        super(nextValue(name));
        customEnumTable.add(this);
        choiceName = null;
        mediaName = null;
    }

    public CustomMediaSizeName(String name, String choice,
                               float width, float length) {
        super(nextValue(name));
        choiceName = choice;
        customEnumTable.add(this);
        mediaName = null;
        try {
            mediaName = MediaSize.findMedia(width, length,
                                            MediaSize.INCH);
        } catch (IllegalArgumentException iae) {
        }
        // The public API method finds a closest match even if it not
        // all that close. Here we want to be sure its *really* close.
        if (mediaName != null) {
            MediaSize sz = MediaSize.getMediaSizeForName(mediaName);
            if (sz == null) {
                mediaName = null;
            } else {
                float w = sz.getX(MediaSize.INCH);
                float h = sz.getY(MediaSize.INCH);
                float dw = Math.abs(w - width);
                float dh = Math.abs(h - length);
                if (dw > 0.1 || dh > 0.1) {
                    mediaName = null;
                }
            }
        }
    }

    /**
     * Use serialVersionUID from JDK 1.5 for interoperability.
     */
    @Serial
    private static final long serialVersionUID = 7412807582228043717L;

    /**
     * Returns the command string for this media.
     */
    public String getChoiceName() {
        return choiceName;
    }


    /**
     * Returns matching standard MediaSizeName.
     */
    public MediaSizeName getStandardMedia() {
        return mediaName;
    }


    // moved from RasterPrinterJob
    /**
     * Returns closest matching MediaSizeName among given array of Media
     */
    public static MediaSizeName findMedia(Media[] media, float x, float y,
                                          int units) {


        if (x <= 0.0f || y <= 0.0f || units < 1) {
            throw new IllegalArgumentException("args must be +ve values");
        }

        if (media == null || media.length == 0) {
            throw new IllegalArgumentException("args must have valid array of media");
        }

        int size =0;
        MediaSizeName[] msn = new MediaSizeName[media.length];
        for (int i=0; i<media.length; i++) {
            if (media[i] instanceof MediaSizeName) {
                msn[size++] = (MediaSizeName)media[i];
            }
        }

        if (size == 0) {
            return null;
        }

        int match = 0;

        double ls = x * x + y * y;
        double tmp_ls;
        float []dim;
        float diffx = x;
        float diffy = y;

        for (int i=0; i < size ; i++) {
            MediaSize mediaSize = MediaSize.getMediaSizeForName(msn[i]);
            if (mediaSize == null) {
                continue;
            }
            dim = mediaSize.getSize(units);
            if (x == dim[0] && y == dim[1]) {
                match = i;
                break;
            } else {
                diffx = x - dim[0];
                diffy = y - dim[1];
                tmp_ls = diffx * diffx + diffy * diffy;
                if (tmp_ls < ls) {
                    ls = tmp_ls;
                    match = i;
                }
            }
        }

        return msn[match];
    }

    /**
     * Returns the string table for super class MediaSizeName.
     */
    public  Media[] getSuperEnumTable() {
        return (Media[])super.getEnumValueTable();
    }


    /**
     * Returns the string table for class CustomMediaSizeName.
     */
    protected String[] getStringTable() {
      String[] nameTable = new String[customStringTable.size()];
      return customStringTable.toArray(nameTable);
    }

    /**
     * Returns the enumeration value table for class CustomMediaSizeName.
     */
    protected EnumSyntax[] getEnumValueTable() {
      MediaSizeName[] enumTable = new MediaSizeName[customEnumTable.size()];
      return customEnumTable.toArray(enumTable);
    }

    public static CustomMediaSizeName create(String name, String choice,
                                             float width, float length) {
        SizeNameChoiceItem key = new SizeNameChoiceItem(name, choice, width, length);
        CustomMediaSizeName value = customMap.get(key);
        if (value == null) {
            value = new CustomMediaSizeName(name, choice, width, length);
            customMap.put(key, value);
            if (value.getStandardMedia() == null) {
                // add this new custom media size name to MediaSize array
                if ((width > 0.0) && (length > 0.0)) {
                    try {
                        new MediaSize(width, length, Size2DSyntax.INCH, value);
                    } catch (IllegalArgumentException e) {
                        /* PDF printer in Linux for Ledger paper causes
                        "IllegalArgumentException: X dimension > Y dimension".
                        We rotate based on IPP spec. */
                        new MediaSize(length, width, Size2DSyntax.INCH, value);
                    }
                }
            }
        }
        return value;
    }

    private static class SizeNameChoiceItem {

        private final String name;
        private final String choice;
        private final float width;
        private final float length;

        public SizeNameChoiceItem(String name, String choice, float width, float length) {
            this.name = name;
            this.choice = choice;
            this.width = width;
            this.length = length;
        }

        public boolean equals(Object object) {
            if (this == object) return true;
            if (object == null || getClass() != object.getClass()) return false;
            SizeNameChoiceItem that = (SizeNameChoiceItem) object;
            return Objects.equals(this.name, that.name)
                    && Objects.equals(this.choice, that.choice) &&
                    Float.compare(this.width, that.width) == 0 &&
                    Float.compare(this.length, that.length) == 0;
        }

        public int hashCode() {
            return Objects.hash(name, choice, width, length);
        }
    }
}
