/*
 * Copyright (c) 1998, 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package javax.swing.text.html.parser;

import java.util.Map;

/**
 * An entity is described in a DTD using the ENTITY construct.
 * It defines the type and value of the entity.
 *
 * @see DTD
 * @author Arthur van Hoff
 */
public final class Entity implements DTDConstants {
    /**
     * The name of the entity.
     */
    public String name;

    /**
     * The type of the entity.
     */
    public int type;

    /**
     * The char array of data.
     */
    public char[] data;

    /**
     * Creates an entity.
     * @param name the name of the entity
     * @param type the type of the entity
     * @param data the char array of data
     */
    public Entity(String name, int type, char[] data) {
        this.name = name;
        this.type = type;
        this.data = data;
    }

    /**
     * Gets the name of the entity.
     * @return the name of the entity, as a <code>String</code>
     */
    public String getName() {
        return name;
    }

    /**
     * Gets the type of the entity.
     * @return the type of the entity
     */
    public int getType() {
        return type & 0xFFFF;
    }

    /**
     * Returns <code>true</code> if it is a parameter entity.
     * @return <code>true</code> if it is a parameter entity
     */
    public boolean isParameter() {
        return (type & PARAMETER) != 0;
    }

    /**
     * Returns <code>true</code> if it is a general entity.
     * @return <code>true</code> if it is a general entity
     */
    public boolean isGeneral() {
        return (type & GENERAL) != 0;
    }

    /**
     * Returns the <code>data</code>.
     * @return the <code>data</code>
     */
    public char[] getData() {
        return data;
    }

    /**
     * Returns the data as a <code>String</code>.
     * @return the data as a <code>String</code>
     */
    public String getString() {
        return new String(data);
    }

    private static final Map<String, Integer> entityTypes = Map.of(
        "PUBLIC", PUBLIC,
        "CDATA", CDATA,
        "SDATA", SDATA,
        "PI", PI,
        "STARTTAG", STARTTAG,
        "ENDTAG", ENDTAG,
        "MS", MS,
        "MD", MD,
        "SYSTEM", SYSTEM
    );

    /**
     * Converts <code>nm</code> string to the corresponding
     * entity type.  If the string does not have a corresponding
     * entity type, returns the type corresponding to "CDATA".
     * Valid entity types are: "PUBLIC", "CDATA", "SDATA", "PI",
     * "STARTTAG", "ENDTAG", "MS", "MD", "SYSTEM".
     *
     * @param nm the string to be converted
     * @return the corresponding entity type, or the type corresponding
     *   to "CDATA", if none exists
     */
    public static int name2type(String nm) {
        return entityTypes.getOrDefault(nm, CDATA);
    }
}
