/*
 * Copyright (c) 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.internal.invoke;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.invoke.VarHandle;

/**
 * Static factories for certain VarHandle/MethodHandle variants.
 * <p>
 * The methods will throw an {@link InternalError} if the lookup fails.
 * <p>
 * Here is an example of how one of these methods could be used:
 * {@snippet lang=java
 * static MethodHandle BAR_HANDLE =
 *         MhUtil.findVirtual(MethodHandles.lookup(),
 *                 Foo.class,"bar",MethodType.methodType(int.class));
 * }
 */
public final class MhUtil {

    private MhUtil() {}

    public static VarHandle findVarHandle(MethodHandles.Lookup lookup,
                                          String name,
                                          Class<?> type) {
        return findVarHandle(lookup, lookup.lookupClass(), name, type);
    }

    public static VarHandle findVarHandle(MethodHandles.Lookup lookup,
                                          Class<?> recv,
                                          String name,
                                          Class<?> type) {
        try {
            return lookup.findVarHandle(recv, name, type);
        } catch (ReflectiveOperationException e) {
            throw new InternalError(e);
        }
    }


    public static MethodHandle findVirtual(MethodHandles.Lookup lookup,
                                           Class<?> refc,
                                           String name,
                                           MethodType type) {
        try {
            return lookup.findVirtual(refc, name, type);
        } catch (ReflectiveOperationException e) {
            throw new InternalError(e);
        }
    }

}
