/*
 * Copyright (c) 2008, 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package java.lang.invoke;

import sun.invoke.util.Wrapper;

import java.lang.ref.SoftReference;

import static java.lang.invoke.MethodHandleStatics.newIllegalArgumentException;
import static java.lang.invoke.MethodHandleNatives.USE_SOFT_CACHE;

/**
 * Shared information for a group of method types, which differ
 * only by reference types, and therefore share a common erasure
 * and wrapping.
 * <p>
 * For an empirical discussion of the structure of method types,
 * see <a href="http://groups.google.com/group/jvm-languages/browse_thread/thread/ac9308ae74da9b7e/">
 * the thread "Avoiding Boxing" on jvm-languages</a>.
 * There are approximately 2000 distinct erased method types in the JDK.
 * There are a little over 10 times that number of unerased types.
 * No more than half of these are likely to be loaded at once.
 * @author John Rose
 */
final class MethodTypeForm {
    final short parameterSlotCount;
    final short primitiveCount;
    final MethodType erasedType;        // the canonical erasure
    final MethodType basicType;         // the canonical erasure, with primitives simplified

    // Cached adapter information:
    private final Object[] methodHandles;

    // Indexes into methodHandles:
    static final int
            MH_BASIC_INV      =  0,  // cached instance of MH.invokeBasic
            MH_NF_INV         =  1,  // cached helper for LF.NamedFunction
            MH_UNINIT_CS      =  2,  // uninitialized call site
            MH_LIMIT          =  3;

    // Cached lambda form information, for basic types only:
    private final Object[] lambdaForms;

    // Indexes into lambdaForms:
    static final int
            LF_INVVIRTUAL              =  0,  // DMH invokeVirtual
            LF_INVSTATIC               =  1,
            LF_INVSPECIAL              =  2,
            LF_NEWINVSPECIAL           =  3,
            LF_INVINTERFACE            =  4,
            LF_INVSTATIC_INIT          =  5,  // DMH invokeStatic with <clinit> barrier
            LF_INTERPRET               =  6,  // LF interpreter
            LF_REBIND                  =  7,  // BoundMethodHandle
            LF_DELEGATE                =  8,  // DelegatingMethodHandle
            LF_DELEGATE_BLOCK_INLINING =  9,  // Counting DelegatingMethodHandle w/ @DontInline
            LF_EX_LINKER               = 10,  // invokeExact_MT (for invokehandle)
            LF_EX_INVOKER              = 11,  // MHs.invokeExact
            LF_GEN_LINKER              = 12,  // generic invoke_MT (for invokehandle)
            LF_GEN_INVOKER             = 13,  // generic MHs.invoke
            LF_CS_LINKER               = 14,  // linkToCallSite_CS
            LF_MH_LINKER               = 15,  // linkToCallSite_MH
            LF_GWC                     = 16,  // guardWithCatch (catchException)
            LF_GWT                     = 17,  // guardWithTest
            LF_TF                      = 18,  // tryFinally
            LF_LOOP                    = 19,  // loop
            LF_INVSPECIAL_IFC          = 20,  // DMH invokeSpecial of (private) interface method
            LF_INVNATIVE               = 21,  // NMH invokeNative
            LF_VH_EX_INVOKER           = 22,  // VarHandle exact invoker
            LF_VH_GEN_INVOKER          = 23,  // VarHandle generic invoker
            LF_VH_GEN_LINKER           = 24,  // VarHandle generic linker
            LF_COLLECTOR               = 25,  // collector handle
            LF_LIMIT                   = 26;

    /** Return the type corresponding uniquely (1-1) to this MT-form.
     *  It might have any primitive returns or arguments, but will have no references except Object.
     */
    public MethodType erasedType() {
        return erasedType;
    }

    /** Return the basic type derived from the erased type of this MT-form.
     *  A basic type is erased (all references Object) and also has all primitive
     *  types (except int, long, float, double, void) normalized to int.
     *  Such basic types correspond to low-level JVM calling sequences.
     */
    public MethodType basicType() {
        return basicType;
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    public MethodHandle cachedMethodHandle(int which) {
        Object entry = methodHandles[which];
        if (entry == null) {
            return null;
        } else if (entry instanceof MethodHandle mh) {
            return mh;
        } else {
            return ((SoftReference<MethodHandle>)entry).get();
        }
    }

    public synchronized MethodHandle setCachedMethodHandle(int which, MethodHandle mh) {
        // Simulate a CAS, to avoid racy duplication of results.
        MethodHandle prev = cachedMethodHandle(which);
        if (prev != null) {
            return prev;
        }
        if (USE_SOFT_CACHE) {
            methodHandles[which] = new SoftReference<>(mh);
        } else {
            methodHandles[which] = mh;
        }
        return mh;
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    public LambdaForm cachedLambdaForm(int which) {
        Object entry = lambdaForms[which];
        if (entry == null) {
            return null;
        } else if (entry instanceof LambdaForm lf) {
            return lf;
        } else {
            return ((SoftReference<LambdaForm>)entry).get();
        }
    }

    public synchronized LambdaForm setCachedLambdaForm(int which, LambdaForm form) {
        // Simulate a CAS, to avoid racy duplication of results.
        LambdaForm prev = cachedLambdaForm(which);
        if (prev != null) {
            return prev;
        }
        if (USE_SOFT_CACHE) {
            lambdaForms[which] = new SoftReference<>(form);
        } else {
            lambdaForms[which] = form;
        }
        return form;
    }

    /**
     * Build an MTF for a given type, which must have all references erased to Object.
     * This MTF will stand for that type and all un-erased variations.
     * Eagerly compute some basic properties of the type, common to all variations.
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    protected MethodTypeForm(MethodType erasedType) {
        this.erasedType = erasedType;

        Class<?>[] ptypes = erasedType.ptypes();
        int pslotCount = ptypes.length;

        // Walk the argument types, looking for primitives.
        short primitiveCount = 0, longArgCount = 0;
        Class<?>[] erasedPtypes = ptypes;
        Class<?>[] basicPtypes = erasedPtypes;
        for (int i = 0; i < erasedPtypes.length; i++) {
            Class<?> ptype = erasedPtypes[i];
            if (ptype != Object.class) {
                ++primitiveCount;
                Wrapper w = Wrapper.forPrimitiveType(ptype);
                if (w.isDoubleWord())  ++longArgCount;
                if (w.isSubwordOrInt() && ptype != int.class) {
                    if (basicPtypes == erasedPtypes)
                        basicPtypes = basicPtypes.clone();
                    basicPtypes[i] = int.class;
                }
            }
        }
        pslotCount += longArgCount;                  // #slots = #args + #longs
        Class<?> returnType = erasedType.returnType();
        Class<?> basicReturnType = returnType;
        if (returnType != Object.class) {
            ++primitiveCount; // even void.class counts as a prim here
            Wrapper w = Wrapper.forPrimitiveType(returnType);
            if (w.isSubwordOrInt() && returnType != int.class)
                basicReturnType = int.class;
        }
        if (erasedPtypes == basicPtypes && basicReturnType == returnType) {
            // Basic type
            this.basicType = erasedType;

            if (pslotCount >= 256)  throw newIllegalArgumentException("too many arguments");

            this.primitiveCount = primitiveCount;
            this.parameterSlotCount = (short)pslotCount;
            this.lambdaForms   = new Object[LF_LIMIT];
            this.methodHandles = new Object[MH_LIMIT];
        } else {
            this.basicType = MethodType.methodType(basicReturnType, basicPtypes, true);
            // fill in rest of data from the basic type:
            MethodTypeForm that = this.basicType.form();
            assert(this != that);

            this.parameterSlotCount = that.parameterSlotCount;
            this.primitiveCount = that.primitiveCount;
            this.methodHandles = null;
            this.lambdaForms = null;
        }
    }

    public int parameterCount() {
        return erasedType.parameterCount();
    }
    public int parameterSlotCount() {
        return parameterSlotCount;
    }
    public boolean hasPrimitives() {
        return primitiveCount != 0;
    }

    static MethodTypeForm findForm(MethodType mt) {
        MethodType erased = canonicalize(mt, ERASE);
        if (erased == null) {
            // It is already erased.  Make a new MethodTypeForm.
            return new MethodTypeForm(mt);
        } else {
            // Share the MethodTypeForm with the erased version.
            return erased.form();
        }
    }

    /** Codes for {@link #canonicalize(java.lang.Class, int)}.
     * ERASE means change every reference to {@code Object}.
     * WRAP means convert primitives (including {@code void} to their
     * corresponding wrapper types.  UNWRAP means the reverse of WRAP.
     */
    public static final int ERASE = 1, WRAP = 2, UNWRAP = 3;

    /** Canonicalize the types in the given method type.
     * If any types change, intern the new type, and return it.
     * Otherwise return null.
     */
    public static MethodType canonicalize(MethodType mt, int how) {
        Class<?>[] ptypes = mt.ptypes();
        Class<?>[] ptypesCanonical = canonicalizeAll(ptypes, how);
        Class<?> rtype = mt.returnType();
        Class<?> rtypeCanonical = canonicalize(rtype, how);
        if (ptypesCanonical == null && rtypeCanonical == null) {
            // It is already canonical.
            return null;
        }
        // Find the erased version of the method type:
        if (rtypeCanonical == null)  rtypeCanonical = rtype;
        if (ptypesCanonical == null)  ptypesCanonical = ptypes;
        return MethodType.methodType(rtypeCanonical, ptypesCanonical, true);
    }

    /** Canonicalize the given return or param type.
     *  Return null if the type is already canonicalized.
     */
    static Class<?> canonicalize(Class<?> t, int how) {
        if (t == Object.class) {
            // no change, ever
        } else if (!t.isPrimitive()) {
            switch (how) {
                case UNWRAP:
                    Class<?> ct = Wrapper.asPrimitiveType(t);
                    if (ct != t)  return ct;
                    break;
                case ERASE:
                    return Object.class;
            }
        } else if (how == WRAP) {
            return Wrapper.asWrapperType(t);
        }
        // no change; return null to signify
        return null;
    }

    /** Canonicalize each param type in the given array.
     *  Return null if all types are already canonicalized.
     */
    static Class<?>[] canonicalizeAll(Class<?>[] ts, int how) {
        Class<?>[] cs = null;
        for (int imax = ts.length, i = 0; i < imax; i++) {
            Class<?> c = canonicalize(ts[i], how);
            // Void parameters may be unwrapped to void; ignore those
            if (c != null && c != void.class) {
                if (cs == null)
                    cs = ts.clone();
                cs[i] = c;
            }
        }
        return cs;
    }

    @Override
    public String toString() {
        return "Form"+erasedType;
    }
}
