/*
 * Copyright (c) 2022, 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package java.lang.classfile.instruction;

import java.lang.classfile.CodeElement;
import java.lang.classfile.CodeModel;
import java.lang.classfile.Instruction;
import java.lang.classfile.Opcode;
import java.lang.classfile.TypeKind;

import jdk.internal.classfile.impl.AbstractInstruction;
import jdk.internal.classfile.impl.Util;

/**
 * Models an arithmetic operator instruction in the {@code code} array of a
 * {@code Code} attribute.  Corresponding opcodes have a {@linkplain Opcode#kind() kind} of
 * {@link Opcode.Kind#OPERATOR}.  Delivered as a {@link CodeElement} when
 * traversing the elements of a {@link CodeModel}.
 * <p>
 * An operator instruction is composite:
 * {@snippet lang=text :
 * // @link substring="OperatorInstruction" target="#of" :
 * OperatorInstruction(Opcode opcode) // @link substring="opcode" target="#opcode()"
 * }
 *
 * @see Opcode.Kind#OPERATOR
 * @since 24
 */
public sealed interface OperatorInstruction extends Instruction
        permits AbstractInstruction.UnboundOperatorInstruction {
    /**
     * {@return the operand type of the instruction}
     * This is derived from the {@link #opcode opcode}.
     */
    TypeKind typeKind();

    /**
     * {@return an operator instruction}
     *
     * @param op the opcode for the specific type of operator instruction,
     *           which must be of kind {@link Opcode.Kind#OPERATOR}
     * @throws IllegalArgumentException if the opcode kind is not
     *         {@link Opcode.Kind#OPERATOR}.
     */
    static OperatorInstruction of(Opcode op) {
        Util.checkKind(op, Opcode.Kind.OPERATOR);
        return new AbstractInstruction.UnboundOperatorInstruction(op);
    }
}
